import { __extends } from "tslib";
import { Hierarchy } from "./Hierarchy";
import { Circle } from "../../core/render/Circle";
import { Template } from "../../core/util/Template";
import { ListTemplate } from "../../core/util/List";
import * as $array from "../../core/util/Array";
import * as d3hierarchy from "d3-hierarchy";
import * as $utils from "../../core/util/Utils";
;
/**
 * Builds a pack diagram.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/hierarchy/sunburst/} for more info
 * @important
 */
var Pack = /** @class */ (function (_super) {
    __extends(Pack, _super);
    function Pack() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "_tag", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: "pack"
        });
        Object.defineProperty(_this, "_packLayout", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: d3hierarchy.pack()
        });
        Object.defineProperty(_this, "_packData", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        /**
         * A list of node circle elements in a [[Pack]] chart.
         *
         * @default new ListTemplate<Circle>
         */
        Object.defineProperty(_this, "circles", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new ListTemplate(Template.new({}), function () { return Circle._new(_this._root, {
                themeTags: $utils.mergeTags(_this.circles.template.get("themeTags", []), [_this._tag, "shape"])
            }, [_this.circles.template]); })
        });
        return _this;
    }
    Object.defineProperty(Pack.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._afterNew.call(this);
            this.setPrivate("scaleR", 1);
        }
    });
    Object.defineProperty(Pack.prototype, "_prepareChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._prepareChildren.call(this);
            if (this.isPrivateDirty("scaleR")) {
                if (this._rootNode) {
                    this._updateNodesScale(this._rootNode);
                }
            }
        }
    });
    Object.defineProperty(Pack.prototype, "_updateVisuals", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (this._rootNode) {
                var packLayout = this._packLayout;
                packLayout.size([this.innerWidth(), this.innerHeight()]);
                packLayout(this._rootNode);
                this._updateNodes(this._rootNode);
            }
        }
    });
    Object.defineProperty(Pack.prototype, "_updateNode", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            _super.prototype._updateNode.call(this, dataItem);
            var node = dataItem.get("node");
            var circle = dataItem.get("circle");
            var hierarchyNode = dataItem.get("d3HierarchyNode");
            var scaleR = this.getPrivate("scaleR", 1);
            var x = hierarchyNode.x * scaleR;
            var y = hierarchyNode.y * scaleR;
            var radius = hierarchyNode.r * scaleR;
            var duration = this.get("animationDuration", 0);
            var easing = this.get("animationEasing");
            node.animate({ key: "x", to: x, duration: duration, easing: easing });
            node.animate({ key: "y", to: y, duration: duration, easing: easing });
            if (circle) {
                var fill = dataItem.get("fill");
                circle.animate({ key: "radius", to: radius, duration: duration, easing: easing });
                circle._setDefault("fill", fill);
                circle._setDefault("stroke", fill);
            }
        }
    });
    Object.defineProperty(Pack.prototype, "_updateNodesScale", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (hierarchyNode) {
            var _this = this;
            var dataItem = hierarchyNode.data.dataItem;
            if (dataItem) {
                var node = dataItem.get("node");
                var circle = dataItem.get("circle");
                var scaleR = this.getPrivate("scaleR", 1);
                var x = hierarchyNode.x * scaleR;
                var y = hierarchyNode.y * scaleR;
                var radius = hierarchyNode.r * scaleR;
                node.setAll({ x: x, y: y });
                circle.set("radius", radius);
                var hierarchyChildren = hierarchyNode.children;
                if (hierarchyChildren) {
                    $array.each(hierarchyChildren, function (hierarchyChild) {
                        _this._updateNodesScale(hierarchyChild);
                    });
                }
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Pack.prototype, "makeNode", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var _this = this;
            var node = _super.prototype.makeNode.call(this, dataItem);
            var circle = node.children.moveValue(this.circles.make(), 0);
            node.setPrivate("tooltipTarget", circle);
            this.circles.push(circle);
            dataItem.setRaw("circle", circle);
            var label = dataItem.get("label");
            circle.on("radius", function () {
                var d = circle.get("radius", _this.width()) * 2;
                label.setAll({ maxWidth: d, maxHeight: d });
            });
            return node;
        }
    });
    Object.defineProperty(Pack.prototype, "_zoom", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var hierarchyNode = dataItem.get("d3HierarchyNode");
            var x = hierarchyNode.x;
            var y = hierarchyNode.y;
            var r = hierarchyNode.r;
            var scaleR = Math.min(this.innerWidth(), this.innerHeight()) / (r * 2);
            var easing = this.get("animationEasing");
            var duration = this.get("animationDuration", 0);
            if (!this.inited) {
                duration = 0;
            }
            this.animatePrivate({ key: "scaleR", to: scaleR, duration: duration, easing: easing });
            var nodesContainer = this.nodesContainer;
            nodesContainer.animate({ key: "x", from: nodesContainer.x(), to: this.width() / 2 - x * scaleR, duration: duration, easing: easing });
            nodesContainer.animate({ key: "y", from: nodesContainer.y(), to: this.height() / 2 - y * scaleR, duration: duration, easing: easing });
        }
    });
    Object.defineProperty(Pack, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "Pack"
    });
    Object.defineProperty(Pack, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: Hierarchy.classNames.concat([Pack.className])
    });
    return Pack;
}(Hierarchy));
export { Pack };
//# sourceMappingURL=Pack.js.map