import { __extends } from "tslib";
import { Hierarchy } from "./Hierarchy";
import { Template } from "../../core/util/Template";
import { ListTemplate } from "../../core/util/List";
import { RoundedRectangle } from "../../core/render/RoundedRectangle";
import * as $array from "../../core/util/Array";
import * as $type from "../../core/util/Type";
import * as $utils from "../../core/util/Utils";
import * as d3hierarchy from "d3-hierarchy";
;
/**
 * Treemap series.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/hierarchy/treemap/} for more info
 */
var Treemap = /** @class */ (function (_super) {
    __extends(Treemap, _super);
    function Treemap() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "_tag", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: "treemap"
        });
        Object.defineProperty(_this, "rectangleTemplate", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: Template.new({})
        });
        Object.defineProperty(_this, "_treemapLayout", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: d3hierarchy.treemap().tile(d3hierarchy.treemapSquarify)
        });
        /**
         * A list of node rectangle elements in a [[Treemap]] chart.
         *
         * @default new ListTemplate<RoundedRectangle>
         */
        Object.defineProperty(_this, "rectangles", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new ListTemplate(Template.new({}), function () { return RoundedRectangle._new(_this._root, {
                themeTags: $utils.mergeTags(_this.rectangles.template.get("themeTags", []), [_this._tag, "shape"])
            }, [_this.rectangles.template]); })
        });
        return _this;
    }
    Object.defineProperty(Treemap.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._afterNew.call(this);
            this.setPrivate("scaleX", 1);
            this.setPrivate("scaleY", 1);
        }
    });
    Object.defineProperty(Treemap.prototype, "_prepareChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._prepareChildren.call(this);
            if (this.isDirty("layoutAlgorithm")) {
                var algorithm = void 0;
                switch (this.get("layoutAlgorithm")) {
                    case "squarify":
                        algorithm = d3hierarchy.treemapSquarify;
                        break;
                    case "binary":
                        algorithm = d3hierarchy.treemapBinary;
                        break;
                    case "slice":
                        algorithm = d3hierarchy.treemapSlice;
                        break;
                    case "dice":
                        algorithm = d3hierarchy.treemapDice;
                        break;
                    case "sliceDice":
                        algorithm = d3hierarchy.treemapSliceDice;
                        break;
                }
                if (algorithm) {
                    this._treemapLayout = d3hierarchy.treemap().tile(algorithm);
                }
                if (this._rootNode) {
                    this._updateNodes(this._rootNode);
                }
            }
            if (this.isDirty("nodePaddingRight") || this.isDirty("nodePaddingLeft") || this.isDirty("nodePaddingTop") || this.isDirty("nodePaddingBottom") || this.isDirty("nodePaddingOuter") || this.isDirty("nodePaddingInner")) {
                if (this._rootNode) {
                    this._updateNodes(this._rootNode);
                }
            }
            if (this.isPrivateDirty("scaleX") || this.isPrivateDirty("scaleY")) {
                if (this._rootNode) {
                    this._updateNodesScale(this._rootNode);
                }
            }
        }
    });
    Object.defineProperty(Treemap.prototype, "_updateVisuals", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (this._rootNode) {
                var treemapLayout = this._treemapLayout;
                treemapLayout.size([this.innerWidth(), this.innerHeight()]);
                var paddingLeft = this.get("nodePaddingLeft");
                var paddingRight = this.get("nodePaddingRight");
                var paddingTop = this.get("nodePaddingTop");
                var paddingBottom = this.get("nodePaddingBottom");
                var paddingInner = this.get("nodePaddingInner");
                var paddingOuter = this.get("nodePaddingOuter");
                if ($type.isNumber(paddingLeft)) {
                    treemapLayout.paddingLeft(paddingLeft);
                }
                if ($type.isNumber(paddingRight)) {
                    treemapLayout.paddingRight(paddingRight);
                }
                if ($type.isNumber(paddingTop)) {
                    treemapLayout.paddingTop(paddingTop);
                }
                if ($type.isNumber(paddingBottom)) {
                    treemapLayout.paddingBottom(paddingBottom);
                }
                if ($type.isNumber(paddingInner)) {
                    treemapLayout.paddingInner(paddingInner);
                }
                if ($type.isNumber(paddingOuter)) {
                    treemapLayout.paddingOuter(paddingOuter);
                }
                treemapLayout(this._rootNode);
                this._updateNodes(this._rootNode);
            }
        }
    });
    Object.defineProperty(Treemap.prototype, "_updateNode", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            _super.prototype._updateNode.call(this, dataItem);
            var node = dataItem.get("node");
            var rectangle = dataItem.get("rectangle");
            var hierarchyNode = dataItem.get("d3HierarchyNode");
            var scaleX = this.getPrivate("scaleX", 1);
            var scaleY = this.getPrivate("scaleY", 1);
            var x0 = hierarchyNode.x0 * scaleX;
            var x1 = hierarchyNode.x1 * scaleX;
            var y0 = hierarchyNode.y0 * scaleY;
            var y1 = hierarchyNode.y1 * scaleY;
            var w = x1 - x0;
            var h = y1 - y0;
            var duration = this.get("animationDuration", 0);
            var easing = this.get("animationEasing");
            node.animate({ key: "x", to: x0, duration: duration, easing: easing });
            node.animate({ key: "y", to: y0, duration: duration, easing: easing });
            node.animate({ key: "width", to: w, duration: duration, easing: easing });
            node.animate({ key: "height", to: h, duration: duration, easing: easing });
            if (rectangle) {
                var fill = dataItem.get("fill");
                rectangle.animate({ key: "width", to: w, duration: duration, easing: easing });
                rectangle.animate({ key: "height", to: h, duration: duration, easing: easing });
                rectangle._setDefault("fill", fill);
                rectangle._setDefault("stroke", fill);
            }
        }
    });
    Object.defineProperty(Treemap.prototype, "_updateNodesScale", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (hierarchyNode) {
            var _this = this;
            var dataItem = hierarchyNode.data.dataItem;
            if (dataItem) {
                var node = dataItem.get("node");
                var rectangle = dataItem.get("rectangle");
                var scaleX = this.getPrivate("scaleX", 1);
                var scaleY = this.getPrivate("scaleY", 1);
                var x0 = hierarchyNode.x0 * scaleX;
                var x1 = hierarchyNode.x1 * scaleX;
                var y0 = hierarchyNode.y0 * scaleY;
                var y1 = hierarchyNode.y1 * scaleY;
                var w = x1 - x0;
                var h = y1 - y0;
                node.setAll({ x: x0, y: y0, width: w, height: h });
                rectangle.setAll({ width: w, height: h });
                var hierarchyChildren = hierarchyNode.children;
                if (hierarchyChildren) {
                    $array.each(hierarchyChildren, function (hierarchyChild) {
                        _this._updateNodesScale(hierarchyChild);
                    });
                }
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Treemap.prototype, "makeNode", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var node = _super.prototype.makeNode.call(this, dataItem);
            var rectangle = node.children.moveValue(this.rectangles.make(), 0);
            node.setPrivate("tooltipTarget", rectangle);
            dataItem.setRaw("rectangle", rectangle);
            var label = dataItem.get("label");
            rectangle.on("width", function () {
                label.set("maxWidth", rectangle.width());
            });
            rectangle.on("height", function () {
                label.set("maxHeight", rectangle.height());
            });
            return node;
        }
    });
    Object.defineProperty(Treemap.prototype, "_zoom", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var hierarchyNode = dataItem.get("d3HierarchyNode");
            var nodePaddingOuter = this.get("nodePaddingOuter", 0);
            var x0 = hierarchyNode.x0 + nodePaddingOuter;
            var x1 = hierarchyNode.x1 - nodePaddingOuter;
            var y0 = hierarchyNode.y0 + nodePaddingOuter;
            var y1 = hierarchyNode.y1 - nodePaddingOuter;
            var scaleX = (this.innerWidth() - nodePaddingOuter * 2) / (x1 - x0);
            var scaleY = (this.innerHeight() - nodePaddingOuter * 2) / (y1 - y0);
            var easing = this.get("animationEasing");
            var duration = this.get("animationDuration", 0);
            this.animatePrivate({ key: "scaleX", to: scaleX, duration: duration, easing: easing });
            this.animatePrivate({ key: "scaleY", to: scaleY, duration: duration, easing: easing });
            this.nodesContainer.animate({ key: "x", to: nodePaddingOuter - x0 * scaleX, duration: duration, easing: easing });
            this.nodesContainer.animate({ key: "y", to: nodePaddingOuter - y0 * scaleY, duration: duration, easing: easing });
        }
    });
    Object.defineProperty(Treemap, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "Treemap"
    });
    Object.defineProperty(Treemap, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: Hierarchy.classNames.concat([Treemap.className])
    });
    return Treemap;
}(Hierarchy));
export { Treemap };
//# sourceMappingURL=Treemap.js.map