import { __extends } from "tslib";
import { MapSeries } from "./MapSeries";
import * as $array from "../../core/util/Array";
import * as $type from "../../core/util/Type";
import * as $math from "../../core/util/Math";
;
/**
 * Creates a map series for displaying markers on the map.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/map-chart/map-point-series/} for more info
 * @important
 */
var MapPointSeries = /** @class */ (function (_super) {
    __extends(MapPointSeries, _super);
    function MapPointSeries() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "_types", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: ["Point", "MultiPoint"]
        });
        return _this;
    }
    Object.defineProperty(MapPointSeries.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this.fields.push("polygonId", "lineId", "longitude", "latitude");
            _super.prototype._afterNew.call(this);
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(MapPointSeries.prototype, "markDirtyProjection", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this.markDirty();
        }
    });
    /**
     * Forces a repaint of the element which relies on data.
     *
     * @since 5.0.21
     */
    Object.defineProperty(MapPointSeries.prototype, "markDirtyValues", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            _super.prototype.markDirtyValues.call(this);
            if (dataItem) {
                this._positionBullets(dataItem);
            }
        }
    });
    Object.defineProperty(MapPointSeries.prototype, "processDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            _super.prototype.processDataItem.call(this, dataItem);
            var geometry = dataItem.get("geometry");
            if (!geometry) {
                dataItem.set("geometry", { type: "Point", coordinates: [dataItem.get("longitude", 0), dataItem.get("latitude", 0)] });
            }
            else {
                if (geometry.type == "Point") {
                    var coordinates = geometry.coordinates;
                    if (coordinates) {
                        dataItem.set("longitude", coordinates[0]);
                        dataItem.set("latitude", coordinates[1]);
                    }
                }
                else if (geometry.type == "MultiPoint") {
                    var coordinates = geometry.coordinates;
                    if (coordinates && coordinates[0]) {
                        dataItem.set("longitude", coordinates[0][0]);
                        dataItem.set("latitude", coordinates[0][1]);
                    }
                }
            }
        }
    });
    Object.defineProperty(MapPointSeries.prototype, "_makeBullets", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var _this = this;
            dataItem.bullets = [];
            this.bullets.each(function (bulletFunction) {
                var geometry = dataItem.get("geometry");
                if (geometry) {
                    if (geometry.type == "Point") {
                        _this._makeBullet(dataItem, bulletFunction);
                    }
                    else if (geometry.type = "MultiPoint") {
                        var i_1 = 0;
                        $array.each(geometry.coordinates, function () {
                            _this._makeBullet(dataItem, bulletFunction, i_1);
                            i_1++;
                        });
                    }
                }
            });
        }
    });
    Object.defineProperty(MapPointSeries.prototype, "_positionBullet", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (bullet) {
            var sprite = bullet.get("sprite");
            if (sprite) {
                var dataItem_1 = sprite.dataItem;
                var latitude = dataItem_1.get("latitude");
                var longitude = dataItem_1.get("longitude");
                var lineDataItem = dataItem_1.get("lineDataItem");
                var chart = this.chart;
                var line_1;
                if (lineDataItem) {
                    line_1 = lineDataItem.get("mapLine");
                }
                else {
                    var lineId_1 = dataItem_1.get("lineId");
                    if (lineId_1 && chart) {
                        chart.series.each(function (series) {
                            if (series.isType("MapLineSeries")) {
                                var lineDI = series.getDataItemById(lineId_1);
                                if (lineDI) {
                                    dataItem_1.set("lineDataItem", lineDI);
                                    line_1 = lineDI.get("mapLine");
                                }
                            }
                        });
                    }
                }
                var polygonDataItem = dataItem_1.get("polygonDataItem");
                var polygon_1;
                if (polygonDataItem) {
                    polygon_1 = polygonDataItem.get("mapPolygon");
                }
                else {
                    var polygonId_1 = dataItem_1.get("polygonId");
                    if (polygonId_1 && chart) {
                        chart.series.each(function (series) {
                            if (series.isType("MapPolygonSeries")) {
                                var polygonDI = series.getDataItemById(polygonId_1);
                                if (polygonDI) {
                                    dataItem_1.set("polygonDataItem", polygonDI);
                                    polygon_1 = polygonDI.get("mapPolygon");
                                }
                            }
                        });
                    }
                }
                var positionOnLine = dataItem_1.get("positionOnLine");
                var coordinates = void 0;
                var angle = void 0;
                if (polygon_1) {
                    var geoPoint = polygon_1.visualCentroid();
                    coordinates = [geoPoint.longitude, geoPoint.latitude];
                }
                else if (line_1 && $type.isNumber(positionOnLine)) {
                    var geoPoint = line_1.positionToGeoPoint(positionOnLine);
                    coordinates = [geoPoint.longitude, geoPoint.latitude];
                    if (dataItem_1.get("autoRotate", bullet.get("autoRotate")) && chart) {
                        var geoPoint0 = line_1.positionToGeoPoint(positionOnLine - 0.002);
                        var geoPoint1 = line_1.positionToGeoPoint(positionOnLine + 0.002);
                        var point0 = chart.convert(geoPoint0);
                        var point1 = chart.convert(geoPoint1);
                        //dataItem.set("autoRotateAngle", $math.getAngle(point0, point1));
                        angle = $math.getAngle(point0, point1);
                    }
                }
                else if ($type.isNumber(longitude) && $type.isNumber(latitude)) {
                    coordinates = [longitude, latitude];
                }
                else {
                    var geometry = dataItem_1.get("geometry");
                    if (geometry) {
                        if (geometry.type == "Point") {
                            this._positionBulletReal(bullet, geometry, geometry.coordinates, angle);
                        }
                        else if (geometry.type == "MultiPoint") {
                            var index = bullet._index || 0;
                            coordinates = geometry.coordinates[index];
                        }
                    }
                }
                if (coordinates) {
                    this._positionBulletReal(bullet, { type: "Point", coordinates: coordinates }, coordinates, angle);
                }
            }
        }
    });
    Object.defineProperty(MapPointSeries.prototype, "_positionBulletReal", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (bullet, geometry, coordinates, angle) {
            var sprite = bullet.get("sprite");
            var chart = this.chart;
            if (chart) {
                var projection = chart.get("projection");
                var geoPath = chart.getPrivate("geoPath");
                var dataItem = sprite.dataItem;
                var xy = projection(coordinates);
                if (xy) {
                    sprite.setAll({ x: xy[0], y: xy[1] });
                }
                var visible = true;
                if (geoPath(geometry)) {
                    if (this.get("clipFront")) {
                        visible = false;
                    }
                }
                else {
                    if (this.get("clipBack")) {
                        visible = false;
                    }
                }
                sprite.setPrivate("visible", visible);
                if (dataItem && angle != null && dataItem.get("autoRotate", bullet.get("autoRotate"))) {
                    sprite.set("rotation", angle + dataItem.get("autoRotateAngle", bullet.get("autoRotateAngle", 0)));
                }
            }
        }
    });
    /**
     * Centers the map to specific series' data item and zooms to the level
     * specified in the parameters.
     *
     * @param  dataItem   Map point
     * @param  zoomLevel  Zoom level
     */
    Object.defineProperty(MapPointSeries.prototype, "zoomToDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, zoomLevel) {
            var chart = this.chart;
            if (chart) {
                return chart.zoomToGeoPoint({ longitude: dataItem.get("longitude", 0), latitude: dataItem.get("latitude", 0) }, zoomLevel, true);
            }
        }
    });
    Object.defineProperty(MapPointSeries, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "MapPointSeries"
    });
    Object.defineProperty(MapPointSeries, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: MapSeries.classNames.concat([MapPointSeries.className])
    });
    return MapPointSeries;
}(MapSeries));
export { MapPointSeries };
//# sourceMappingURL=MapPointSeries.js.map