import { __extends } from "tslib";
import { Graphics } from "../../core/render/Graphics";
import * as $mapUtils from "./MapUtils";
import $polylabel from "polylabel";
import { geoArea } from "d3-geo";
/**
 * A polygon in a [[MapPolygonSeries]].
 */
var MapPolygon = /** @class */ (function (_super) {
    __extends(MapPolygon, _super);
    function MapPolygon() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "_projectionDirty", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        /**
         * A [[MapPolygonSeries]] polygon belongs to.
         */
        Object.defineProperty(_this, "series", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        return _this;
    }
    Object.defineProperty(MapPolygon.prototype, "_beforeChanged", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            _super.prototype._beforeChanged.call(this);
            if (this._projectionDirty || this.isDirty("geometry") || this.isDirty("precision")) {
                var geometry_1 = this.get("geometry");
                if (geometry_1) {
                    var series = this.series;
                    if (series) {
                        var projection = series.projection();
                        if (projection) {
                            projection.precision(this.get("precision", 0.5));
                        }
                        var geoPath_1 = series.geoPath();
                        if (geoPath_1) {
                            this._clear = true;
                            this.set("draw", function (_display) {
                                geoPath_1.context(_this._display);
                                geoPath_1(geometry_1);
                                geoPath_1.context(null);
                            });
                            if (this.isHover()) {
                                this.showTooltip();
                            }
                        }
                    }
                }
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(MapPolygon.prototype, "markDirtyProjection", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this.markDirty();
            this._projectionDirty = true;
        }
    });
    Object.defineProperty(MapPolygon.prototype, "_clearDirty", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._clearDirty.call(this);
            this._projectionDirty = false;
        }
    });
    /**
     * Returns latitude/longitude of the geometrical center of the polygon.
     *
     * @return Center
     */
    Object.defineProperty(MapPolygon.prototype, "geoCentroid", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var geometry = this.get("geometry");
            if (geometry) {
                return $mapUtils.getGeoCentroid(geometry);
            }
            else {
                return { latitude: 0, longitude: 0 };
            }
        }
    });
    /**
     * Returns latitude/longitude of the visual center of the polygon.
     *
     * @return Center
     */
    Object.defineProperty(MapPolygon.prototype, "visualCentroid", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var biggestArea = 0;
            var coordinates = [];
            var geometry = this.get("geometry");
            if (geometry) {
                if (geometry.type == "Polygon") {
                    coordinates = geometry.coordinates;
                }
                else if (geometry.type == "MultiPolygon") {
                    if (geometry.coordinates.length > 1) {
                        for (var i = 0; i < geometry.coordinates.length; i++) {
                            var coords = geometry.coordinates[i];
                            var area = geoArea({ type: "Polygon", coordinates: coords });
                            if (area > biggestArea) {
                                coordinates = coords;
                                biggestArea = area;
                            }
                        }
                    }
                }
                var center = $polylabel(coordinates);
                return { longitude: center[0], latitude: center[1] };
            }
            return { longitude: 0, latitude: 0 };
        }
    });
    Object.defineProperty(MapPolygon.prototype, "_getTooltipPoint", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var series = this.series;
            if (series) {
                var projection = series.projection();
                if (projection) {
                    var geoPoint = this.visualCentroid();
                    var xy = projection([geoPoint.longitude, geoPoint.latitude]);
                    if (xy) {
                        return { x: xy[0], y: xy[1] };
                    }
                }
            }
            return { x: 0, y: 0 };
        }
    });
    Object.defineProperty(MapPolygon, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "MapPolygon"
    });
    Object.defineProperty(MapPolygon, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: Graphics.classNames.concat([MapPolygon.className])
    });
    return MapPolygon;
}(Graphics));
export { MapPolygon };
//# sourceMappingURL=MapPolygon.js.map