import { __awaiter, __extends, __generator } from "tslib";
import { Series } from "../../core/render/Series";
import { Container } from "../../core/render/Container";
import { visualSettings } from "../../core/render/Graphics";
import * as $array from "../../core/util/Array";
import * as $type from "../../core/util/Type";
/**
 * A base class for any percent chart series.
 */
var PercentSeries = /** @class */ (function (_super) {
    __extends(PercentSeries, _super);
    function PercentSeries() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "slicesContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.children.push(Container.new(_this._root, { position: "absolute", isMeasured: false }))
        });
        Object.defineProperty(_this, "labelsContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.children.push(Container.new(_this._root, { position: "absolute", isMeasured: false }))
        });
        Object.defineProperty(_this, "ticksContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.children.push(Container.new(_this._root, { position: "absolute", isMeasured: false }))
        });
        Object.defineProperty(_this, "_lLabels", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        Object.defineProperty(_this, "_rLabels", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        Object.defineProperty(_this, "_hLabels", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        /**
         * A [[ListTemplate]] of all slices in series.
         *
         * `slices.template` can also be used to configure slices.
         */
        Object.defineProperty(_this, "slices", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this._makeSlices()
        });
        /**
         * A [[ListTemplate]] of all slice labels in series.
         *
         * `labels.template` can also be used to configure slice labels.
         */
        Object.defineProperty(_this, "labels", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this._makeLabels()
        });
        /**
         * A [[ListTemplate]] of all slice ticks in series.
         *
         * `ticks.template` can also be used to configure slice ticks.
         */
        Object.defineProperty(_this, "ticks", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this._makeTicks()
        });
        return _this;
    }
    /**
     * @ignore
     */
    Object.defineProperty(PercentSeries.prototype, "makeSlice", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var _this = this;
            var slice = this.slicesContainer.children.push(this.slices.make());
            slice.on("fill", function () {
                _this.updateLegendMarker(dataItem);
            });
            slice.on("stroke", function () {
                _this.updateLegendMarker(dataItem);
            });
            slice._setDataItem(dataItem);
            dataItem.set("slice", slice);
            this.slices.push(slice);
            return slice;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(PercentSeries.prototype, "makeLabel", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var label = this.labelsContainer.children.push(this.labels.make());
            label._setDataItem(dataItem);
            dataItem.set("label", label);
            this.labels.push(label);
            return label;
        }
    });
    Object.defineProperty(PercentSeries.prototype, "_shouldMakeBullet", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            if (dataItem.get("value") != null) {
                return true;
            }
            return false;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(PercentSeries.prototype, "makeTick", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var tick = this.ticksContainer.children.push(this.ticks.make());
            tick._setDataItem(dataItem);
            dataItem.set("tick", tick);
            this.ticks.push(tick);
            return tick;
        }
    });
    Object.defineProperty(PercentSeries.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this.fields.push("category", "fill");
            _super.prototype._afterNew.call(this);
        }
    });
    Object.defineProperty(PercentSeries.prototype, "_onDataClear", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var colors = this.get("colors");
            if (colors) {
                colors.reset();
            }
        }
    });
    Object.defineProperty(PercentSeries.prototype, "_prepareChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._prepareChildren.call(this);
            this._lLabels = [];
            this._rLabels = [];
            this._hLabels = [];
            if (this._valuesDirty) {
                var sum_1 = 0;
                var absSum_1 = 0;
                var valueHigh_1 = 0;
                var valueLow_1 = Infinity;
                var count_1 = 0;
                $array.each(this._dataItems, function (dataItem) {
                    var valueWorking = dataItem.get("valueWorking", 0);
                    sum_1 += valueWorking;
                    absSum_1 += Math.abs(valueWorking);
                });
                $array.each(this._dataItems, function (dataItem) {
                    var value = dataItem.get("valueWorking", 0);
                    if (value >= absSum_1) {
                        sum_1 = dataItem.get("value", 0);
                    }
                    if (value > valueHigh_1) {
                        valueHigh_1 = value;
                    }
                    if (value < valueLow_1) {
                        valueLow_1 = value;
                    }
                    count_1++;
                    var percentTotal = value / absSum_1;
                    if (absSum_1 == 0) {
                        percentTotal = 0;
                    }
                    dataItem.setRaw("valuePercentTotal", percentTotal * 100);
                });
                this.setPrivateRaw("valueLow", valueLow_1);
                this.setPrivateRaw("valueHigh", valueHigh_1);
                this.setPrivateRaw("valueSum", sum_1);
                this.setPrivateRaw("valueAverage", sum_1 / count_1);
                this.setPrivateRaw("valueAbsoluteSum", absSum_1);
            }
        }
    });
    /**
     * Shows hidden series.
     *
     * @param   duration  Animation duration in milliseconds
     * @return            Animation promise
     */
    Object.defineProperty(PercentSeries.prototype, "show", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (duration) {
            return __awaiter(this, void 0, void 0, function () {
                var promises;
                return __generator(this, function (_a) {
                    switch (_a.label) {
                        case 0:
                            promises = [];
                            promises.push(_super.prototype.show.call(this, duration));
                            promises.push(this._sequencedShowHide(true, duration));
                            return [4 /*yield*/, Promise.all(promises)];
                        case 1:
                            _a.sent();
                            return [2 /*return*/];
                    }
                });
            });
        }
    });
    /**
     * Hide whole series.
     *
     * @param   duration  Animation duration in milliseconds
     * @return            Animation promise
     */
    Object.defineProperty(PercentSeries.prototype, "hide", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (duration) {
            return __awaiter(this, void 0, void 0, function () {
                var promises;
                return __generator(this, function (_a) {
                    switch (_a.label) {
                        case 0:
                            promises = [];
                            promises.push(_super.prototype.hide.call(this, duration));
                            promises.push(this._sequencedShowHide(false, duration));
                            return [4 /*yield*/, Promise.all(promises)];
                        case 1:
                            _a.sent();
                            return [2 /*return*/];
                    }
                });
            });
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(PercentSeries.prototype, "_updateChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._updateChildren.call(this);
            if (this._valuesDirty) {
                $array.each(this._dataItems, function (dataItem) {
                    dataItem.get("label").text.markDirtyText();
                });
            }
            this._arrange();
        }
    });
    Object.defineProperty(PercentSeries.prototype, "_arrange", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            this._arrangeDown(this._lLabels);
            this._arrangeUp(this._lLabels);
            this._arrangeDown(this._rLabels);
            this._arrangeUp(this._rLabels);
            this._arrangeLeft(this._hLabels);
            this._arrangeRight(this._hLabels);
            $array.each(this.dataItems, function (dataItem) {
                _this._updateTick(dataItem);
            });
        }
    });
    Object.defineProperty(PercentSeries.prototype, "_afterChanged", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._afterChanged.call(this);
            this._arrange();
        }
    });
    Object.defineProperty(PercentSeries.prototype, "processDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            _super.prototype.processDataItem.call(this, dataItem);
            if (dataItem.get("fill") == null) {
                var colors = this.get("colors");
                if (colors) {
                    dataItem.setRaw("fill", colors.next());
                }
            }
        }
    });
    /**
     * Shows series's data item.
     *
     * @param   dataItem  Data item
     * @param   duration  Animation duration in milliseconds
     * @return            Promise
     */
    Object.defineProperty(PercentSeries.prototype, "showDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, duration) {
            return __awaiter(this, void 0, void 0, function () {
                var promises, easing, value, animation, tick, label, slice;
                return __generator(this, function (_a) {
                    switch (_a.label) {
                        case 0:
                            promises = [_super.prototype.showDataItem.call(this, dataItem, duration)];
                            if (!$type.isNumber(duration)) {
                                duration = this.get("stateAnimationDuration", 0);
                            }
                            easing = this.get("stateAnimationEasing");
                            value = dataItem.get("value");
                            animation = dataItem.animate({ key: "valueWorking", to: value, duration: duration, easing: easing });
                            if (animation) {
                                promises.push(animation.waitForStop());
                            }
                            tick = dataItem.get("tick");
                            if (tick) {
                                promises.push(tick.show(duration));
                            }
                            label = dataItem.get("label");
                            if (label) {
                                promises.push(label.show(duration));
                            }
                            slice = dataItem.get("slice");
                            if (slice) {
                                promises.push(slice.show(duration));
                            }
                            return [4 /*yield*/, Promise.all(promises)];
                        case 1:
                            _a.sent();
                            return [2 /*return*/];
                    }
                });
            });
        }
    });
    /**
     * Hides series's data item.
     *
     * @param   dataItem  Data item
     * @param   duration  Animation duration in milliseconds
     * @return            Promise
     */
    Object.defineProperty(PercentSeries.prototype, "hideDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, duration) {
            return __awaiter(this, void 0, void 0, function () {
                var promises, hiddenState, easing, animation, tick, label, slice;
                return __generator(this, function (_a) {
                    switch (_a.label) {
                        case 0:
                            promises = [_super.prototype.hideDataItem.call(this, dataItem, duration)];
                            hiddenState = this.states.create("hidden", {});
                            if (!$type.isNumber(duration)) {
                                duration = hiddenState.get("stateAnimationDuration", this.get("stateAnimationDuration", 0));
                            }
                            easing = hiddenState.get("stateAnimationEasing", this.get("stateAnimationEasing"));
                            animation = dataItem.animate({ key: "valueWorking", to: 0, duration: duration, easing: easing });
                            if (animation) {
                                promises.push(animation.waitForStop());
                            }
                            tick = dataItem.get("tick");
                            if (tick) {
                                promises.push(tick.hide(duration));
                            }
                            label = dataItem.get("label");
                            if (label) {
                                promises.push(label.hide(duration));
                            }
                            slice = dataItem.get("slice");
                            slice.hideTooltip();
                            if (slice) {
                                promises.push(slice.hide(duration));
                            }
                            return [4 /*yield*/, Promise.all(promises)];
                        case 1:
                            _a.sent();
                            return [2 /*return*/];
                    }
                });
            });
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(PercentSeries.prototype, "disposeDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            _super.prototype.disposeDataItem.call(this, dataItem);
            var label = dataItem.get("label");
            if (label) {
                this.labels.removeValue(label);
                label.dispose();
            }
            var tick = dataItem.get("tick");
            if (tick) {
                this.ticks.removeValue(tick);
                tick.dispose();
            }
            var slice = dataItem.get("slice");
            if (slice) {
                this.slices.removeValue(slice);
                slice.dispose();
            }
        }
    });
    /**
     * Triggers hover on a series data item.
     *
     * @since 5.0.7
     * @param  dataItem  Target data item
     */
    Object.defineProperty(PercentSeries.prototype, "hoverDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var slice = dataItem.get("slice");
            if (slice && !slice.isHidden()) {
                slice.hover();
            }
        }
    });
    /**
     * Triggers un-hover on a series data item.
     *
     * @since 5.0.7
     * @param  dataItem  Target data item
     */
    Object.defineProperty(PercentSeries.prototype, "unhoverDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var slice = dataItem.get("slice");
            if (slice) {
                slice.unhover();
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(PercentSeries.prototype, "updateLegendMarker", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var slice = dataItem.get("slice");
            if (slice) {
                var legendDataItem = dataItem.get("legendDataItem");
                if (legendDataItem) {
                    var markerRectangle_1 = legendDataItem.get("markerRectangle");
                    $array.each(visualSettings, function (setting) {
                        if (slice.get(setting) != null) {
                            markerRectangle_1.set(setting, slice.get(setting));
                        }
                    });
                }
            }
        }
    });
    Object.defineProperty(PercentSeries.prototype, "_arrangeDown", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (labels) {
            if (labels) {
                var next_1 = this._getNextDown();
                labels.sort(function (a, b) {
                    if (a.y > b.y) {
                        return 1;
                    }
                    else if (a.y < b.y) {
                        return -1;
                    }
                    else {
                        return 0;
                    }
                });
                $array.each(labels, function (l) {
                    var bounds = l.label.adjustedLocalBounds();
                    var labelTop = bounds.top;
                    if (l.y + labelTop < next_1) {
                        l.y = next_1 - labelTop;
                    }
                    l.label.set("y", l.y);
                    next_1 = l.y + bounds.bottom;
                });
            }
        }
    });
    Object.defineProperty(PercentSeries.prototype, "_getNextUp", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this.labelsContainer.maxHeight();
        }
    });
    Object.defineProperty(PercentSeries.prototype, "_getNextDown", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return 0;
        }
    });
    Object.defineProperty(PercentSeries.prototype, "_arrangeUp", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (labels) {
            if (labels) {
                var next_2 = this._getNextUp();
                labels.sort(function (a, b) {
                    if (a.y < b.y) {
                        return 1;
                    }
                    else if (a.y > b.y) {
                        return -1;
                    }
                    else {
                        return 0;
                    }
                });
                $array.each(labels, function (l) {
                    var bounds = l.label.adjustedLocalBounds();
                    var labelBottom = bounds.bottom;
                    if (l.y + labelBottom > next_2) {
                        l.y = next_2 - labelBottom;
                    }
                    l.label.set("y", l.y);
                    next_2 = l.y + bounds.top;
                });
            }
        }
    });
    Object.defineProperty(PercentSeries.prototype, "_arrangeRight", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (labels) {
            if (labels) {
                var next_3 = 0;
                labels.sort(function (a, b) {
                    if (a.y > b.y) {
                        return 1;
                    }
                    else if (a.y < b.y) {
                        return -1;
                    }
                    else {
                        return 0;
                    }
                });
                $array.each(labels, function (l) {
                    var bounds = l.label.adjustedLocalBounds();
                    var labelLeft = bounds.left;
                    if (l.y + labelLeft < next_3) {
                        l.y = next_3 - labelLeft;
                    }
                    l.label.set("x", l.y);
                    next_3 = l.y + bounds.right;
                });
            }
        }
    });
    Object.defineProperty(PercentSeries.prototype, "_arrangeLeft", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (labels) {
            if (labels) {
                var next_4 = this.labelsContainer.maxWidth();
                labels.sort(function (a, b) {
                    if (a.y < b.y) {
                        return 1;
                    }
                    else if (a.y > b.y) {
                        return -1;
                    }
                    else {
                        return 0;
                    }
                });
                $array.each(labels, function (l) {
                    var bounds = l.label.adjustedLocalBounds();
                    var labelRight = bounds.right;
                    if (l.y + labelRight > next_4) {
                        l.y = next_4 - labelRight;
                    }
                    l.label.set("x", l.y);
                    next_4 = l.y + bounds.left;
                });
            }
        }
    });
    Object.defineProperty(PercentSeries.prototype, "_updateSize", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._updateSize.call(this);
            this.markDirty();
        }
    });
    Object.defineProperty(PercentSeries.prototype, "_updateTick", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_dataItem) {
        }
    });
    Object.defineProperty(PercentSeries.prototype, "_dispose", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._dispose.call(this);
            var chart = this.chart;
            if (chart) {
                chart.series.removeValue(this);
            }
        }
    });
    Object.defineProperty(PercentSeries, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "PercentSeries"
    });
    Object.defineProperty(PercentSeries, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: Series.classNames.concat([PercentSeries.className])
    });
    return PercentSeries;
}(Series));
export { PercentSeries };
//# sourceMappingURL=PercentSeries.js.map