import { __extends, __read } from "tslib";
import { RadarDefaultTheme } from "./RadarDefaultTheme";
import { XYChart } from "../xy/XYChart";
import { Percent, p50, percent } from "../../core/util/Percent";
import { Container } from "../../core/render/Container";
import { Graphics } from "../../core/render/Graphics";
import { arc } from "d3-shape";
import * as $utils from "../../core/util/Utils";
import * as $math from "../../core/util/Math";
/**
 * Radar chart.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/radar-chart/} for more info
 * @important
 */
var RadarChart = /** @class */ (function (_super) {
    __extends(RadarChart, _super);
    function RadarChart() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        /**
         * [[Container]] where radar-related elements go.
         *
         * @default Container.new()
         */
        Object.defineProperty(_this, "radarContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.plotContainer.children.push(Container.new(_this._root, { x: p50, y: p50 }))
        });
        Object.defineProperty(_this, "_arcGenerator", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: arc()
        });
        Object.defineProperty(_this, "_maxRadius", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 1
        });
        return _this;
    }
    Object.defineProperty(RadarChart.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            this._defaultThemes.push(RadarDefaultTheme.new(this._root));
            _super.prototype._afterNew.call(this);
            var radarContainer = this.radarContainer;
            var gridContainer = this.gridContainer;
            var topGridContainer = this.topGridContainer;
            var seriesContainer = this.seriesContainer;
            var bulletsContainer = this.bulletsContainer;
            radarContainer.children.pushAll([gridContainer, seriesContainer, topGridContainer, bulletsContainer]);
            seriesContainer.set("mask", Graphics.new(this._root, {}));
            gridContainer.set("mask", Graphics.new(this._root, {}));
            this._disposers.push(this.plotContainer.events.on("boundschanged", function () {
                _this._updateRadius();
            }));
        }
    });
    Object.defineProperty(RadarChart.prototype, "_prepareChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._prepareChildren.call(this);
            if (this._sizeDirty || this.isDirty("radius") || this.isDirty("innerRadius") || this.isDirty("startAngle") || this.isDirty("endAngle")) {
                var chartContainer = this.chartContainer;
                var w = chartContainer.innerWidth();
                var h = chartContainer.innerHeight();
                var startAngle = this.get("startAngle", 0);
                var endAngle = this.get("endAngle", 0);
                var innerRadius = this.get("innerRadius");
                var bounds = $math.getArcBounds(0, 0, startAngle, endAngle, 1);
                var wr = w / (bounds.right - bounds.left);
                var hr = h / (bounds.bottom - bounds.top);
                var innerBounds = { left: 0, right: 0, top: 0, bottom: 0 };
                if (innerRadius instanceof Percent) {
                    var value = innerRadius.value;
                    var mr = Math.min(wr, hr);
                    value = Math.max(mr * value, mr - Math.min(h, w)) / mr;
                    innerBounds = $math.getArcBounds(0, 0, startAngle, endAngle, value);
                    this.setPrivateRaw("irModifyer", value / innerRadius.value);
                }
                bounds = $math.mergeBounds([bounds, innerBounds]);
                this._maxRadius = Math.max(0, Math.min(wr, hr));
                var radius = $utils.relativeToValue(this.get("radius", 0), this._maxRadius);
                this.radarContainer.setAll({
                    dy: -radius * (bounds.bottom + bounds.top) / 2, dx: -radius * (bounds.right + bounds.left) / 2
                });
                this._updateRadius();
            }
        }
    });
    Object.defineProperty(RadarChart.prototype, "_addCursor", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (cursor) {
            this.radarContainer.children.push(cursor);
        }
    });
    // do not delete
    Object.defineProperty(RadarChart.prototype, "_updateRadius", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            var radius = $utils.relativeToValue(this.get("radius", percent(80)), this._maxRadius);
            this.setPrivateRaw("radius", radius);
            var innerRadius = $utils.relativeToValue(this.get("innerRadius", 0), radius);
            if (innerRadius < 0) {
                innerRadius = radius + innerRadius;
            }
            this.setPrivateRaw("innerRadius", innerRadius);
            this.xAxes.each(function (axis) {
                var renderer = axis.get("renderer");
                renderer.updateLayout();
            });
            this.yAxes.each(function (axis) {
                var renderer = axis.get("renderer");
                renderer.updateLayout();
            });
            this._updateMask(this.seriesContainer, innerRadius, radius);
            this._updateMask(this.gridContainer, innerRadius, radius);
            this.series.each(function (series) {
                if (series.get("maskBullets")) {
                    _this._updateMask(series.bulletsContainer, innerRadius, radius);
                }
                else {
                    series.bulletsContainer.remove("mask");
                }
            });
            var cursor = this.get("cursor");
            if (cursor) {
                cursor.updateLayout();
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(RadarChart.prototype, "_updateMask", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (container, innerRadius, radius) {
            var _this = this;
            var mask = container.get("mask");
            if (mask) {
                mask.set("draw", function (display) {
                    _this._arcGenerator.context(display);
                    _this._arcGenerator({ innerRadius: innerRadius, outerRadius: radius, startAngle: (_this.get("startAngle", 0) + 90) * $math.RADIANS, endAngle: (_this.get("endAngle", 0) + 90) * $math.RADIANS });
                });
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(RadarChart.prototype, "processAxis", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (axis) {
            this.radarContainer.children.push(axis);
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(RadarChart.prototype, "inPlot", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (point, radius, innerRadius) {
            var _a;
            var r = Math.hypot(point.x, point.y);
            var angle = $math.normalizeAngle(Math.atan2(point.y, point.x) * $math.DEGREES);
            var startAngle = $math.normalizeAngle(this.get("startAngle", 0));
            var endAngle = $math.normalizeAngle(this.get("endAngle", 0));
            var inArc = false;
            if (startAngle < endAngle) {
                if (startAngle < angle && angle < endAngle) {
                    inArc = true;
                }
            }
            if (startAngle > endAngle) {
                if (angle > startAngle) {
                    inArc = true;
                }
                if (angle < endAngle) {
                    inArc = true;
                }
            }
            if (startAngle == endAngle) {
                inArc = true;
            }
            if (!inArc) {
                return false;
            }
            if (radius == null) {
                radius = this.getPrivate("radius", 0);
            }
            if (innerRadius == null) {
                innerRadius = this.getPrivate("innerRadius", 0);
            }
            if (innerRadius > radius) {
                _a = __read([radius, innerRadius], 2), innerRadius = _a[0], radius = _a[1];
            }
            if (r <= radius + .5 && r >= innerRadius - .5) {
                return true;
            }
            return false;
        }
    });
    Object.defineProperty(RadarChart.prototype, "_tooltipToLocal", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (point) {
            return this.radarContainer._display.toLocal(point);
        }
    });
    Object.defineProperty(RadarChart, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "RadarChart"
    });
    Object.defineProperty(RadarChart, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: XYChart.classNames.concat([RadarChart.className])
    });
    return RadarChart;
}(XYChart));
export { RadarChart };
//# sourceMappingURL=RadarChart.js.map