import { __awaiter, __extends, __generator } from "tslib";
import { VennDefaultTheme } from "./VennDefaultTheme";
import { Series } from "../../core/render/Series";
import { Template } from "../../core/util/Template";
import { Graphics, visualSettings } from "../../core/render/Graphics";
import { Container } from "../../core/render/Container";
import { Label } from "../../core/render/Label";
import { ListTemplate } from "../../core/util/List";
import * as $utils from "../../core/util/Utils";
import * as $array from "../../core/util/Array";
import * as $type from "../../core/util/Type";
import * as venn from "./vennjs/index.js";
/**
 * Creates a Venn diagram.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/venn/} for more info
 * @important
 */
var Venn = /** @class */ (function (_super) {
    __extends(Venn, _super);
    function Venn() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "_sets", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: ""
        });
        /**
         * A [[Container]] that holds all slices (circles and intersections).
         *
         * @default Container.new()
         */
        Object.defineProperty(_this, "slicesContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.children.push(Container.new(_this._root, {}))
        });
        /**
         * A [[Container]] that holds all labels.
         *
         * @default Container.new()
         */
        Object.defineProperty(_this, "labelsContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.children.push(Container.new(_this._root, {}))
        });
        /**
         * A [[Graphics]] element that is used to show the shape of the hovered slice
         * or intersection.
         *
         * @default Graphics.new()
         */
        Object.defineProperty(_this, "hoverGraphics", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.slicesContainer.children.push(Graphics.new(_this._root, { position: "absolute", isMeasured: false }))
        });
        Object.defineProperty(_this, "_hovered", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        /**
         * A [[ListTemplate]] of all slices in series.
         *
         * `slices.template` can also be used to configure slices.
         */
        Object.defineProperty(_this, "slices", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this._makeSlices()
        });
        /**
         * A [[ListTemplate]] of all slice labels in series.
         *
         * `labels.template` can also be used to configure slice labels.
         */
        Object.defineProperty(_this, "labels", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this._makeLabels()
        });
        return _this;
    }
    Object.defineProperty(Venn.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._defaultThemes.push(VennDefaultTheme.new(this._root));
            this.fields.push("intersections", "category", "fill");
            _super.prototype._afterNew.call(this);
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Venn.prototype, "makeSlice", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var _this = this;
            var slice = this.slicesContainer.children.push(this.slices.make());
            slice.events.on("pointerover", function (e) {
                _this._hovered = e.target;
                _this._updateHover();
            });
            slice.events.on("pointerout", function () {
                _this._hovered = undefined;
                _this.hoverGraphics.hide();
            });
            slice.on("fill", function () {
                _this.updateLegendMarker(dataItem);
            });
            slice.on("stroke", function () {
                _this.updateLegendMarker(dataItem);
            });
            slice._setDataItem(dataItem);
            dataItem.set("slice", slice);
            this.slices.push(slice);
            return slice;
        }
    });
    Object.defineProperty(Venn.prototype, "_updateHover", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (this._hovered) {
                var hoverGraphics = this.hoverGraphics;
                hoverGraphics.set("svgPath", this._hovered.get("svgPath"));
                hoverGraphics.show();
                hoverGraphics.toFront();
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Venn.prototype, "makeLabel", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var label = this.labelsContainer.children.push(this.labels.make());
            label._setDataItem(dataItem);
            dataItem.set("label", label);
            this.labels.push(label);
            return label;
        }
    });
    Object.defineProperty(Venn.prototype, "_makeSlices", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            return new ListTemplate(Template.new({}), function () { return Graphics._new(_this._root, {
                themeTags: $utils.mergeTags(_this.slices.template.get("themeTags", []), ["venn", "series"])
            }, [_this.slices.template]); });
        }
    });
    Object.defineProperty(Venn.prototype, "_makeLabels", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            return new ListTemplate(Template.new({}), function () { return Label._new(_this._root, {
                themeTags: $utils.mergeTags(_this.labels.template.get("themeTags", []), ["venn", "series"])
            }, [_this.labels.template]); });
        }
    });
    Object.defineProperty(Venn.prototype, "processDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            _super.prototype.processDataItem.call(this, dataItem);
            if (dataItem.get("fill") == null) {
                var colors = this.get("colors");
                if (colors) {
                    dataItem.setRaw("fill", colors.next());
                }
            }
            this.makeSlice(dataItem);
            this.makeLabel(dataItem);
        }
    });
    Object.defineProperty(Venn.prototype, "_prepareChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            _super.prototype._prepareChildren.call(this);
            if (this._valuesDirty || this._sizeDirty) {
                var sets_1 = [];
                // prepare data for venn
                $array.each(this.dataItems, function (dataItem) {
                    var set = {};
                    var intersections = dataItem.get("intersections");
                    if (intersections) {
                        set.sets = intersections;
                    }
                    else {
                        set.sets = [dataItem.get("category")];
                    }
                    set.size = dataItem.get("valueWorking");
                    if (set.size > 0) {
                        sets_1.push(set);
                    }
                });
                var newSets = sets_1.toString();
                this._sets = newSets;
                if (sets_1.length > 0) {
                    var vennData = venn.venn(sets_1);
                    vennData = venn.normalizeSolution(vennData, null, null);
                    vennData = venn.scaleSolution(vennData, this.innerWidth(), this.innerHeight(), 0);
                    var circles_1 = {};
                    for (var name_1 in vennData) {
                        var item = vennData[name_1];
                        var r = item.radius;
                        var dataItem = this.getDataItemByCategory(name_1);
                        if (dataItem) {
                            var slice = dataItem.get("slice");
                            var color = dataItem.get("fill");
                            slice._setDefault("fill", color);
                            slice._setDefault("stroke", color);
                            this.updateLegendMarker(dataItem);
                            slice.set("svgPath", "M" + item.x + "," + item.y + " m -" + r + ", 0 a " + r + "," + r + " 0 1,1 " + r * 2 + ",0 a " + r + "," + r + " 0 1,1 -" + r * 2 + ",0");
                            circles_1[name_1] = item;
                        }
                    }
                    var centers_1 = venn.computeTextCentres(circles_1, sets_1);
                    $array.each(this.dataItems, function (dataItem) {
                        var name = dataItem.get("category");
                        var center = centers_1[name];
                        var intersections = dataItem.get("intersections");
                        if (intersections) {
                            name = intersections.toString();
                            center = centers_1[name];
                            if (center) {
                                var set = intersections;
                                var cc = [];
                                for (var s = 0; s < set.length; s++) {
                                    cc.push(circles_1[set[s]]);
                                }
                                var intersectionPath = venn.intersectionAreaPath(cc);
                                var slice = dataItem.get("slice");
                                var color = dataItem.get("fill");
                                slice._setDefault("fill", color);
                                slice._setDefault("stroke", color);
                                slice.setAll({ svgPath: intersectionPath });
                            }
                        }
                        if (center) {
                            var label = dataItem.get("label");
                            label.setAll({ x: center.x, y: center.y });
                        }
                        _this.updateLegendValue(dataItem);
                    });
                }
                this._updateHover();
            }
        }
    });
    /**
     * Looks up and returns a data item by its category.
     *
     * @param   category  Category
     * @return      Data item
     */
    Object.defineProperty(Venn.prototype, "getDataItemByCategory", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (id) {
            return $array.find(this.dataItems, function (dataItem) {
                return dataItem.get("category") == id;
            });
        }
    });
    /**
     * Shows series's data item.
     *
     * @param   dataItem  Data item
     * @param   duration  Animation duration in milliseconds
     * @return            Promise
     */
    Object.defineProperty(Venn.prototype, "showDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, duration) {
            return __awaiter(this, void 0, void 0, function () {
                var promises, easing, value, animation, label, slice, intersections, category_1;
                var _this = this;
                return __generator(this, function (_a) {
                    switch (_a.label) {
                        case 0:
                            promises = [_super.prototype.showDataItem.call(this, dataItem, duration)];
                            if (!$type.isNumber(duration)) {
                                duration = this.get("stateAnimationDuration", 0);
                            }
                            easing = this.get("stateAnimationEasing");
                            value = dataItem.get("value");
                            animation = dataItem.animate({ key: "valueWorking", to: value, duration: duration, easing: easing });
                            if (animation) {
                                promises.push(animation.waitForStop());
                            }
                            label = dataItem.get("label");
                            if (label) {
                                promises.push(label.show(duration));
                            }
                            slice = dataItem.get("slice");
                            if (slice) {
                                promises.push(slice.show(duration));
                            }
                            intersections = dataItem.get("intersections");
                            if (intersections) {
                                $array.each(intersections, function (cat) {
                                    var di = _this.getDataItemByCategory(cat);
                                    if (di && di.isHidden()) {
                                        _this.showDataItem(di, duration);
                                    }
                                });
                            }
                            if (!intersections) {
                                category_1 = dataItem.get("category");
                                $array.each(this.dataItems, function (di) {
                                    var intersections = di.get("intersections");
                                    if (di != dataItem && intersections) {
                                        var allVisible_1 = true;
                                        $array.each(intersections, function (cat) {
                                            var dii = _this.getDataItemByCategory(cat);
                                            if (dii && dii.isHidden()) {
                                                allVisible_1 = false;
                                            }
                                        });
                                        if (allVisible_1 && intersections.indexOf(category_1) != -1) {
                                            if (di.isHidden()) {
                                                _this.showDataItem(di, duration);
                                            }
                                        }
                                    }
                                });
                            }
                            return [4 /*yield*/, Promise.all(promises)];
                        case 1:
                            _a.sent();
                            return [2 /*return*/];
                    }
                });
            });
        }
    });
    /**
     * Hides series's data item.
     *
     * @param   dataItem  Data item
     * @param   duration  Animation duration in milliseconds
     * @return            Promise
     */
    Object.defineProperty(Venn.prototype, "hideDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, duration) {
            return __awaiter(this, void 0, void 0, function () {
                var promises, hiddenState, easing, animation, label, slice;
                var _this = this;
                return __generator(this, function (_a) {
                    switch (_a.label) {
                        case 0:
                            promises = [_super.prototype.hideDataItem.call(this, dataItem, duration)];
                            hiddenState = this.states.create("hidden", {});
                            if (!$type.isNumber(duration)) {
                                duration = hiddenState.get("stateAnimationDuration", this.get("stateAnimationDuration", 0));
                            }
                            easing = hiddenState.get("stateAnimationEasing", this.get("stateAnimationEasing"));
                            animation = dataItem.animate({ key: "valueWorking", to: 0, duration: duration, easing: easing });
                            if (animation) {
                                promises.push(animation.waitForStop());
                            }
                            label = dataItem.get("label");
                            if (label) {
                                promises.push(label.hide(duration));
                            }
                            slice = dataItem.get("slice");
                            if (slice) {
                                promises.push(slice.hide(duration));
                                slice.hideTooltip();
                            }
                            if (!dataItem.get("intersections")) {
                                $array.each(this.dataItems, function (di) {
                                    var intersections = di.get("intersections");
                                    if (di != dataItem && intersections) {
                                        if (intersections.indexOf(dataItem.get("category")) != -1) {
                                            _this.hideDataItem(di, duration);
                                        }
                                    }
                                });
                            }
                            return [4 /*yield*/, Promise.all(promises)];
                        case 1:
                            _a.sent();
                            return [2 /*return*/];
                    }
                });
            });
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Venn.prototype, "disposeDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            _super.prototype.disposeDataItem.call(this, dataItem);
            var label = dataItem.get("label");
            if (label) {
                this.labels.removeValue(label);
                label.dispose();
            }
            var slice = dataItem.get("slice");
            if (slice) {
                this.slices.removeValue(slice);
                slice.dispose();
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Venn.prototype, "updateLegendMarker", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var slice = dataItem.get("slice");
            if (slice) {
                var legendDataItem = dataItem.get("legendDataItem");
                if (legendDataItem) {
                    var markerRectangle_1 = legendDataItem.get("markerRectangle");
                    $array.each(visualSettings, function (setting) {
                        markerRectangle_1.set(setting, slice.get(setting));
                    });
                }
            }
        }
    });
    /**
     * Triggers hover on a series data item.
     *
     * @since 5.0.7
     * @param  dataItem  Target data item
     */
    Object.defineProperty(Venn.prototype, "hoverDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var slice = dataItem.get("slice");
            if (slice && !slice.isHidden()) {
                slice.hover();
            }
        }
    });
    /**
     * Triggers un-hover on a series data item.
     *
     * @since 5.0.7
     * @param  dataItem  Target data item
     */
    Object.defineProperty(Venn.prototype, "unhoverDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var slice = dataItem.get("slice");
            if (slice) {
                slice.unhover();
            }
        }
    });
    Object.defineProperty(Venn, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "Venn"
    });
    Object.defineProperty(Venn, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: Series.classNames.concat([Venn.className])
    });
    return Venn;
}(Series));
export { Venn };
//# sourceMappingURL=Venn.js.map