import { __extends } from "tslib";
import { Graphics } from "../../../core/render/Graphics";
import { Template } from "../../../core/util/Template";
import { ListTemplate } from "../../../core/util/List";
import { AxisTick } from "./AxisTick";
import { Grid } from "./Grid";
import { AxisLabel } from "./AxisLabel";
import * as $utils from "../../../core/util/Utils";
/**
 * Base class for an axis renderer.
 *
 * Should not be used on its own.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/xy-chart/#Axis_renderer} for more info
 */
var AxisRenderer = /** @class */ (function (_super) {
    __extends(AxisRenderer, _super);
    function AxisRenderer() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        // save for quick access
        Object.defineProperty(_this, "_axisLength", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 100
        });
        Object.defineProperty(_this, "_start", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "_end", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 1
        });
        Object.defineProperty(_this, "_inversed", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(_this, "_minSize", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        /**
         * Chart the renderer is used in.
         */
        Object.defineProperty(_this, "chart", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_lc", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 1
        });
        Object.defineProperty(_this, "_ls", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "_thumbDownPoint", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_downStart", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_downEnd", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        /**
         * A list of ticks in the axis.
         *
         * `ticks.template` can be used to configure ticks.
         *
         * @default new ListTemplate<AxisTick>
         */
        Object.defineProperty(_this, "ticks", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new ListTemplate(Template.new({}), function () { return AxisTick._new(_this._root, {
                themeTags: $utils.mergeTags(_this.ticks.template.get("themeTags", []), _this.get("themeTags", []))
            }, [_this.ticks.template]); })
        });
        /**
         * A list of grid elements in the axis.
         *
         * `grid.template` can be used to configure grid.
         *
         * @default new ListTemplate<Grid>
         */
        Object.defineProperty(_this, "grid", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new ListTemplate(Template.new({}), function () { return Grid._new(_this._root, {
                themeTags: $utils.mergeTags(_this.grid.template.get("themeTags", []), _this.get("themeTags", []))
            }, [_this.grid.template]); })
        });
        /**
         * A list of fills in the axis.
         *
         * `axisFills.template` can be used to configure axis fills.
         *
         * @default new ListTemplate<Graphics>
         */
        Object.defineProperty(_this, "axisFills", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new ListTemplate(Template.new({}), function () { return Graphics._new(_this._root, {
                themeTags: $utils.mergeTags(_this.axisFills.template.get("themeTags", ["axis", "fill"]), _this.get("themeTags", []))
            }, [_this.axisFills.template]); })
        });
        /**
         * A list of labels in the axis.
         *
         * `labels.template` can be used to configure axis labels.
         *
         * @default new ListTemplate<AxisLabel>
         */
        Object.defineProperty(_this, "labels", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new ListTemplate(Template.new({}), function () { return AxisLabel._new(_this._root, {
                themeTags: $utils.mergeTags(_this.labels.template.get("themeTags", []), _this.get("themeTags", []))
            }, [_this.labels.template]); })
        });
        /**
         * An [[Axis]] renderer is for.
         */
        Object.defineProperty(_this, "axis", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "thumb", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        return _this;
    }
    /**
     * @ignore
     */
    Object.defineProperty(AxisRenderer.prototype, "makeTick", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, themeTags) {
            var tick = this.ticks.make();
            tick._setDataItem(dataItem);
            dataItem.setRaw("tick", tick);
            tick.set("themeTags", $utils.mergeTags(tick.get("themeTags"), themeTags));
            this.axis.labelsContainer.children.push(tick);
            this.ticks.push(tick);
            return tick;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRenderer.prototype, "makeGrid", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, themeTags) {
            var grid = this.grid.make();
            grid._setDataItem(dataItem);
            dataItem.setRaw("grid", grid);
            grid.set("themeTags", $utils.mergeTags(grid.get("themeTags"), themeTags));
            this.axis.gridContainer.children.push(grid);
            this.grid.push(grid);
            return grid;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRenderer.prototype, "makeAxisFill", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, themeTags) {
            var axisFill = this.axisFills.make();
            axisFill._setDataItem(dataItem);
            axisFill.set("themeTags", $utils.mergeTags(axisFill.get("themeTags"), themeTags));
            this.axis.gridContainer.children.push(axisFill);
            dataItem.setRaw("axisFill", axisFill);
            this.axisFills.push(axisFill);
            return axisFill;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRenderer.prototype, "makeLabel", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, themeTags) {
            var label = this.labels.make();
            if (this.get("opposite")) {
                themeTags.push("opposite");
            }
            if (this.get("inside")) {
                themeTags.push("inside");
            }
            label.set("themeTags", $utils.mergeTags(label.get("themeTags"), themeTags));
            this.axis.labelsContainer.children.moveValue(label, 0);
            label._setDataItem(dataItem);
            dataItem.setRaw("label", label);
            this.labels.push(label);
            return label;
        }
    });
    Object.defineProperty(AxisRenderer.prototype, "axisLength", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return 0;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRenderer.prototype, "gridCount", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this.axisLength() / this.get("minGridDistance", 50);
        }
    });
    Object.defineProperty(AxisRenderer.prototype, "_updatePositions", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
        }
    });
    Object.defineProperty(AxisRenderer.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            _super.prototype._afterNew.call(this);
            this.set("isMeasured", false);
            var thumb = this.thumb;
            if (thumb) {
                this._disposers.push(thumb.events.on("pointerdown", function (event) {
                    _this._handleThumbDown(event.originalEvent);
                }));
                this._disposers.push(thumb.events.on("globalpointerup", function (event) {
                    _this._handleThumbUp(event.originalEvent);
                }));
                this._disposers.push(thumb.events.on("globalpointermove", function (event) {
                    _this._handleThumbMove(event.originalEvent);
                }));
            }
        }
    });
    Object.defineProperty(AxisRenderer.prototype, "_changed", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._changed.call(this);
            if (this.isDirty("pan")) {
                var thumb = this.thumb;
                if (thumb) {
                    var labelsContainer = this.axis.labelsContainer;
                    var pan = this.get("pan");
                    if (pan == "zoom") {
                        labelsContainer.children.push(thumb);
                    }
                    else if (pan == "none") {
                        labelsContainer.children.removeValue(thumb);
                    }
                }
            }
        }
    });
    Object.defineProperty(AxisRenderer.prototype, "_handleThumbDown", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (event) {
            this._thumbDownPoint = this.toLocal(this._root.documentPointToRoot({ x: event.clientX, y: event.clientY }));
            var axis = this.axis;
            this._downStart = axis.get("start");
            this._downEnd = axis.get("end");
        }
    });
    Object.defineProperty(AxisRenderer.prototype, "_handleThumbUp", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_event) {
            this._thumbDownPoint = undefined;
        }
    });
    Object.defineProperty(AxisRenderer.prototype, "_handleThumbMove", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (event) {
            var downPoint = this._thumbDownPoint;
            if (downPoint) {
                var point = this.toLocal(this._root.documentPointToRoot({ x: event.clientX, y: event.clientY }));
                var downStart = this._downStart;
                var downEnd = this._downEnd;
                var extra = this._getPan(point, downPoint) * Math.min(1, (downEnd - downStart)) / 2;
                this.axis.setAll({ start: downStart - extra, end: downEnd + extra });
            }
        }
    });
    Object.defineProperty(AxisRenderer.prototype, "_getPan", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_point1, _point2) {
            return 0;
        }
    });
    /**
     * Converts relative position (0-1) on axis to a pixel coordinate.
     *
     * @param position  Position (0-1)
     * @return Coordinate (px)
     */
    Object.defineProperty(AxisRenderer.prototype, "positionToCoordinate", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (position) {
            if (this._inversed) {
                return (this._end - position) * this._axisLength;
            }
            else {
                return (position - this._start) * this._axisLength;
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRenderer.prototype, "updateTooltipBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_tooltip) { }
    });
    Object.defineProperty(AxisRenderer.prototype, "_updateSize", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this.markDirty();
            this._clear = true;
        }
    });
    Object.defineProperty(AxisRenderer.prototype, "toAxisPosition", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (position) {
            var start = this._start || 0;
            var end = this._end || 1;
            position = position * (end - start);
            if (!this.get("inversed")) {
                position = start + position;
            }
            else {
                position = end - position;
            }
            return position;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRenderer.prototype, "fixPosition", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (position) {
            if (this.get("inversed")) {
                return 1 - position;
            }
            return position;
        }
    });
    Object.defineProperty(AxisRenderer.prototype, "_updateLC", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
        }
    });
    Object.defineProperty(AxisRenderer.prototype, "toggleVisibility", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (sprite, position, minPosition, maxPosition) {
            var axis = this.axis;
            var start = axis.get("start", 0);
            var end = axis.get("end", 1);
            var updatedStart = start + (end - start) * (minPosition - 0.0001);
            var updatedEnd = start + (end - start) * (maxPosition + 0.0001);
            if (position < updatedStart || position > updatedEnd) {
                sprite.setPrivate("visible", false);
            }
            else {
                sprite.setPrivate("visible", true);
            }
        }
    });
    Object.defineProperty(AxisRenderer.prototype, "_positionTooltip", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (tooltip, point) {
            var chart = this.chart;
            if (chart) {
                if (chart.inPlot(point)) {
                    tooltip.set("pointTo", this._display.toGlobal(point));
                }
                else {
                    tooltip.hide();
                }
            }
        }
    });
    Object.defineProperty(AxisRenderer.prototype, "processAxis", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () { }
    });
    Object.defineProperty(AxisRenderer, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "AxisRenderer"
    });
    Object.defineProperty(AxisRenderer, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: Graphics.classNames.concat([AxisRenderer.className])
    });
    return AxisRenderer;
}(Graphics));
export { AxisRenderer };
//# sourceMappingURL=AxisRenderer.js.map