import { __extends, __read } from "tslib";
import { AxisRenderer } from "./AxisRenderer";
import { p100 } from "../../../core/util/Percent";
import * as $type from "../../../core/util/Type";
import * as $utils from "../../../core/util/Utils";
import { Rectangle } from "../../../core/render/Rectangle";
/**
 * Used to render horizontal axis.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/xy-chart/#Axis_renderer} for more info
 * @important
 */
var AxisRendererX = /** @class */ (function (_super) {
    __extends(AxisRendererX, _super);
    function AxisRendererX() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "thumb", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: Rectangle.new(_this._root, { width: p100, themeTags: ["axis", "x", "thumb"] })
        });
        return _this;
    }
    Object.defineProperty(AxisRendererX.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._settings.themeTags = $utils.mergeTags(this._settings.themeTags, ["renderer", "x"]);
            _super.prototype._afterNew.call(this);
            this.setPrivateRaw("letter", "X");
            var gridTemplate = this.grid.template;
            gridTemplate.set("height", p100);
            gridTemplate.set("width", 0);
            gridTemplate.set("draw", function (display, graphics) {
                display.moveTo(0, 0);
                display.lineTo(0, graphics.height());
            });
            this.set("draw", function (display, graphics) {
                display.moveTo(0, 0);
                display.lineTo(graphics.width(), 0);
            });
        }
    });
    Object.defineProperty(AxisRendererX.prototype, "_changed", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._changed.call(this);
            var axis = this.axis;
            if (this.isDirty("inside")) {
                axis.markDirtySize();
            }
            var opposite = "opposite";
            if (this.isDirty(opposite)) {
                var chart = this.chart;
                if (chart) {
                    var axisChildren = axis.children;
                    if (this.get(opposite)) {
                        var children = chart.topAxesContainer.children;
                        if (children.indexOf(axis) == -1) {
                            children.insertIndex(0, axis);
                        }
                        axisChildren.moveValue(this);
                        axis.addTag(opposite);
                    }
                    else {
                        var children = chart.bottomAxesContainer.children;
                        if (children.indexOf(axis) == -1) {
                            children.moveValue(axis);
                        }
                        axisChildren.moveValue(this, 0);
                        axis.removeTag(opposite);
                    }
                    axis.markDirtySize();
                }
                axis.ghostLabel._applyThemes();
            }
            this.thumb.setPrivate("height", axis.labelsContainer.height());
        }
    });
    Object.defineProperty(AxisRendererX.prototype, "_getPan", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (point1, point2) {
            return (point2.x - point1.x) / this.width();
        }
    });
    Object.defineProperty(AxisRendererX.prototype, "toAxisPosition", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (position) {
            var start = this._start || 0;
            var end = this._end || 1;
            position -= this._ls;
            position = position * (end - start) / this._lc;
            if (!this.get("inversed")) {
                position = start + position;
            }
            else {
                position = end - position;
            }
            return position;
        }
    });
    Object.defineProperty(AxisRendererX.prototype, "_updateLC", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var axis = this.axis;
            var parent = axis.parent;
            if (parent) {
                var w = parent.innerWidth();
                this._lc = this.axisLength() / w;
                this._ls = (axis.x() - parent.get("paddingLeft", 0)) / w;
            }
        }
    });
    Object.defineProperty(AxisRendererX.prototype, "_updatePositions", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var axis = this.axis;
            axis.gridContainer.set("x", axis.x() - $utils.relativeToValue(axis.get("centerX", 0), axis.width()) - axis.parent.get("paddingLeft", 0));
            axis.bulletsContainer.set("y", this.y());
            var chart = axis.chart;
            if (chart) {
                var plotContainer = chart.plotContainer;
                var axisHeader = axis.axisHeader;
                var width = axis.get("marginLeft", 0);
                var x = axis.x() - width;
                var parent_1 = axis.parent;
                if (parent_1) {
                    x -= parent_1.get("paddingLeft", 0);
                }
                if (axisHeader.children.length > 0) {
                    width = axis.axisHeader.width();
                    axis.set("marginLeft", width);
                }
                else {
                    axisHeader.set("width", width);
                }
                axisHeader.setAll({ x: x, y: -1, height: plotContainer.height() + 2 });
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererX.prototype, "processAxis", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype.processAxis.call(this);
            var axis = this.axis;
            axis.set("width", p100);
            var verticalLayout = this._root.verticalLayout;
            axis.set("layout", verticalLayout);
            axis.labelsContainer.set("width", p100);
            axis.axisHeader.setAll({ layout: verticalLayout });
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererX.prototype, "axisLength", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this.axis.width();
        }
    });
    /**
     * Converts axis relative position to actual coordinate in pixels.
     *
     * @param   position  Position
     * @return            Point
     */
    Object.defineProperty(AxisRendererX.prototype, "positionToPoint", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (position) {
            return { x: this.positionToCoordinate(position), y: 0 };
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererX.prototype, "updateTick", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (tick, position, endPosition, count) {
            if (tick) {
                if (!$type.isNumber(position)) {
                    position = 0;
                }
                var location_1 = 0.5;
                if ($type.isNumber(count) && count > 1) {
                    location_1 = tick.get("multiLocation", location_1);
                }
                else {
                    location_1 = tick.get("location", location_1);
                }
                if ($type.isNumber(endPosition) && endPosition != position) {
                    position = position + (endPosition - position) * location_1;
                }
                tick.set("x", this.positionToCoordinate(position));
                var length_1 = tick.get("length", 0);
                var inside = tick.get("inside", this.get("inside", false));
                if (this.get("opposite")) {
                    tick.set("y", p100);
                    if (!inside) {
                        length_1 *= -1;
                    }
                }
                else {
                    tick.set("y", 0);
                    if (inside) {
                        length_1 *= -1;
                    }
                }
                tick.set("draw", function (display) {
                    display.moveTo(0, 0);
                    display.lineTo(0, length_1);
                });
                this.toggleVisibility(tick, position, tick.get("minPosition", 0), tick.get("maxPosition", 1));
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererX.prototype, "updateLabel", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (label, position, endPosition, count) {
            if (label) {
                var location_2 = 0.5;
                if ($type.isNumber(count) && count > 1) {
                    location_2 = label.get("multiLocation", location_2);
                }
                else {
                    location_2 = label.get("location", location_2);
                }
                if (!$type.isNumber(position)) {
                    position = 0;
                }
                var inside = label.get("inside", this.get("inside", false));
                var opposite = this.get("opposite");
                if (opposite) {
                    if (!inside) {
                        label.set("position", "relative");
                        label.set("y", p100);
                    }
                    else {
                        label.set("position", "absolute");
                        label.set("y", 0);
                    }
                }
                else {
                    if (!inside) {
                        label.set("y", undefined);
                        label.set("position", "relative");
                    }
                    else {
                        label.set("y", 0);
                        label.set("position", "absolute");
                    }
                }
                if ($type.isNumber(endPosition) && endPosition != position) {
                    position = position + (endPosition - position) * location_2;
                }
                label.set("x", this.positionToCoordinate(position));
                this.toggleVisibility(label, position, label.get("minPosition", 0), label.get("maxPosition", 1));
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererX.prototype, "updateGrid", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (grid, position, endPosition) {
            if (grid) {
                if (!$type.isNumber(position)) {
                    position = 0;
                }
                var location_3 = grid.get("location", 0.5);
                if ($type.isNumber(endPosition) && endPosition != position) {
                    position = position + (endPosition - position) * location_3;
                }
                grid.set("x", Math.round(this.positionToCoordinate(position)));
                this.toggleVisibility(grid, position, 0, 1);
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererX.prototype, "updateBullet", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (bullet, position, endPosition) {
            if (bullet) {
                var sprite = bullet.get("sprite");
                if (sprite) {
                    if (!$type.isNumber(position)) {
                        position = 0;
                    }
                    var location_4 = bullet.get("location", 0.5);
                    if ($type.isNumber(endPosition) && endPosition != position) {
                        position = position + (endPosition - position) * location_4;
                    }
                    sprite.set("x", this.positionToCoordinate(position));
                    this.toggleVisibility(sprite, position, 0, 1);
                }
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererX.prototype, "updateFill", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (fill, position, endPosition) {
            if (fill) {
                if (!$type.isNumber(position)) {
                    position = 0;
                }
                if (!$type.isNumber(endPosition)) {
                    endPosition = 1;
                }
                var x0 = this.positionToCoordinate(position);
                var x1 = this.positionToCoordinate(endPosition);
                this.fillDrawMethod(fill, x0, x1);
            }
        }
    });
    Object.defineProperty(AxisRendererX.prototype, "fillDrawMethod", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (fill, x0, x1) {
            var _this = this;
            fill.set("draw", function (display) {
                var _a;
                //display.drawRect(x0, 0, x1 - x0, this.axis!.gridContainer.height());
                // using for holes, so can not be rectangle
                var h = _this.axis.gridContainer.height();
                var w = _this.width();
                if (x1 < x0) {
                    _a = __read([x0, x1], 2), x1 = _a[0], x0 = _a[1];
                }
                if (x0 > w || x1 < 0) {
                    return;
                }
                x0 = Math.max(0, x0);
                x1 = Math.min(w, x1);
                display.moveTo(x0, 0);
                display.lineTo(x1, 0);
                display.lineTo(x1, h);
                display.lineTo(x0, h);
                display.lineTo(x0, 0);
            });
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererX.prototype, "positionTooltip", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (tooltip, position) {
            this._positionTooltip(tooltip, { x: this.positionToCoordinate(position), y: 0 });
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererX.prototype, "updateTooltipBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (tooltip) {
            var inside = this.get("inside");
            var num = 100000;
            var global = this._display.toGlobal({ x: 0, y: 0 });
            var x = global.x;
            var y = 0;
            var w = this.axisLength();
            var h = num;
            var pointerOrientation = "up";
            if (this.get("opposite")) {
                if (inside) {
                    pointerOrientation = "up";
                    y = global.y;
                    h = num;
                }
                else {
                    pointerOrientation = "down";
                    y = global.y - num;
                    h = num;
                }
            }
            else {
                if (inside) {
                    pointerOrientation = "down";
                    y = global.y - num;
                    h = num;
                }
                else {
                    pointerOrientation = "up";
                    y = global.y;
                    h = num;
                }
            }
            var bounds = { left: x, right: x + w, top: y, bottom: y + h };
            var oldBounds = tooltip.get("bounds");
            if (!$utils.sameBounds(bounds, oldBounds)) {
                tooltip.set("bounds", bounds);
                tooltip.set("pointerOrientation", pointerOrientation);
            }
        }
    });
    Object.defineProperty(AxisRendererX, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "AxisRendererX"
    });
    Object.defineProperty(AxisRendererX, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: AxisRenderer.classNames.concat([AxisRendererX.className])
    });
    return AxisRendererX;
}(AxisRenderer));
export { AxisRendererX };
//# sourceMappingURL=AxisRendererX.js.map