import { __extends } from "tslib";
import { Axis } from "./Axis";
import * as $array from "../../../core/util/Array";
import * as $type from "../../../core/util/Type";
import * as $math from "../../../core/util/Math";
import * as $utils from "../../../core/util/Utils";
import { populateString } from "../../../core/util/PopulateString";
/**
 * Creates a category axis.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/xy-chart/axes/category-axis/} for more info
 * @important
 */
var CategoryAxis = /** @class */ (function (_super) {
    __extends(CategoryAxis, _super);
    function CategoryAxis() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "_frequency", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 1
        });
        Object.defineProperty(_this, "_itemMap", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        return _this;
    }
    Object.defineProperty(CategoryAxis.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._settings.themeTags = $utils.mergeTags(this._settings.themeTags, ["axis"]);
            this.fields.push("category");
            this.setPrivateRaw("name", "category");
            this.addTag("category");
            _super.prototype._afterNew.call(this);
        }
    });
    Object.defineProperty(CategoryAxis.prototype, "_prepareChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            _super.prototype._prepareChildren.call(this);
            var len = this.dataItems.length;
            var i = 0;
            if (this._valuesDirty) {
                $array.each(this.dataItems, function (dataItem) {
                    dataItem.setRaw("index", i);
                    _this._itemMap[dataItem.get("category")] = dataItem;
                    i++;
                });
                this.setPrivateRaw("maxZoomFactor", len);
            }
            this.setPrivateRaw("startIndex", Math.max(Math.round(this.get("start", 0) * len), 0));
            this.setPrivateRaw("endIndex", Math.min(Math.round(this.get("end", 1) * len), len));
            if (this._sizeDirty || this._valuesDirty || (this.isDirty("start") || this.isDirty("end") || this.isPrivateDirty("endIndex") || this.isPrivateDirty("startIndex") || this.isPrivateDirty("width") || this.isPrivateDirty("height"))) {
                if (this.dataItems.length > 0) {
                    this._handleRangeChange();
                    this._prepareAxisItems();
                    this._updateAxisRanges();
                }
            }
        }
    });
    Object.defineProperty(CategoryAxis.prototype, "_handleRangeChange", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            $array.each(this.series, function (series) {
                var startCategory = _this.dataItems[_this.startIndex()].get("category");
                var endCategory = _this.dataItems[_this.endIndex() - 1].get("category");
                var baseAxis = series.get("baseAxis");
                var xAxis = series.get("xAxis");
                var yAxis = series.get("yAxis");
                if (xAxis instanceof CategoryAxis && yAxis instanceof CategoryAxis) {
                    series._markDirtyAxes();
                }
                else if (baseAxis === _this) {
                    var key = void 0;
                    var openKey = void 0;
                    var otherAxis = yAxis;
                    if (xAxis === baseAxis) {
                        if (series.get("categoryXField")) {
                            key = "categoryX";
                        }
                        if (series.get("openCategoryXField")) {
                            openKey = "openCategoryX";
                        }
                    }
                    else if (yAxis === baseAxis) {
                        if (series.get("categoryYField")) {
                            key = "categoryY";
                        }
                        if (series.get("openCategoryYField")) {
                            openKey = "openCategoryY";
                        }
                        otherAxis = xAxis;
                    }
                    if (otherAxis.className == "ValueAxis") {
                        if (key || openKey) {
                            var startDataItem = void 0;
                            var endDataItem = void 0;
                            for (var i = 0, len = series.dataItems.length; i < len; i++) {
                                var dataItem = series.dataItems[i];
                                if (key) {
                                    if (dataItem.get(key) === startCategory) {
                                        startDataItem = dataItem;
                                        break;
                                    }
                                }
                                if (openKey) {
                                    if (dataItem.get(openKey) === startCategory) {
                                        startDataItem = dataItem;
                                        break;
                                    }
                                }
                            }
                            for (var i = series.dataItems.length - 1; i >= 0; i--) {
                                var dataItem = series.dataItems[i];
                                if (key) {
                                    if (dataItem.get(key) === endCategory) {
                                        endDataItem = dataItem;
                                        break;
                                    }
                                }
                                if (openKey) {
                                    if (dataItem.get(openKey) === endCategory) {
                                        endDataItem = dataItem;
                                        break;
                                    }
                                }
                            }
                            if (startDataItem) {
                                series.setPrivate("startIndex", series.dataItems.indexOf(startDataItem));
                            }
                            if (endDataItem) {
                                series.setPrivate("endIndex", series.dataItems.indexOf(endDataItem) + 1);
                            }
                        }
                    }
                    series._markDirtyAxes(); // must be outside
                }
            });
        }
    });
    Object.defineProperty(CategoryAxis.prototype, "_prepareAxisItems", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var renderer = this.get("renderer");
            var len = this.dataItems.length;
            var startIndex = this.startIndex();
            if (startIndex > 0) {
                startIndex--;
            }
            var endIndex = this.endIndex();
            if (endIndex < len) {
                endIndex++;
            }
            var maxCount = renderer.axisLength() / Math.max(renderer.get("minGridDistance"), 1 / Number.MAX_SAFE_INTEGER);
            var frequency = Math.max(1, Math.min(len, Math.ceil((endIndex - startIndex) / maxCount)));
            startIndex = Math.floor(startIndex / frequency) * frequency;
            this._frequency = frequency;
            for (var j = 0; j < len; j++) {
                this.dataItems[j].hide();
            }
            var f = this.dataItems[startIndex].get("index", 0);
            for (var i = startIndex; i < endIndex; i = i + frequency) {
                var dataItem = this.dataItems[i];
                this._createAssets(dataItem, []);
                if (dataItem.isHidden()) {
                    dataItem.show();
                }
                this._prepareDataItem(dataItem, f, frequency);
                f++;
            }
            this._updateGhost();
        }
    });
    Object.defineProperty(CategoryAxis.prototype, "_prepareDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, fillIndex, count) {
            var renderer = this.get("renderer");
            var categoryLocation = dataItem.get("categoryLocation", 0);
            var endCategoryLocation = dataItem.get("endCategoryLocation", 1);
            var index = dataItem.get("index");
            if (!$type.isNumber(index)) {
                index = this.categoryToIndex(dataItem.get("category"));
            }
            var position = this.indexToPosition(index, categoryLocation);
            var endCategory = dataItem.get("endCategory");
            var endIndex;
            if (endCategory) {
                endIndex = this.categoryToIndex(endCategory);
                if (!$type.isNumber(endIndex)) {
                    endIndex = index;
                }
            }
            else {
                endIndex = index;
            }
            var endPosition = this.indexToPosition(endIndex, endCategoryLocation);
            var fillEndIndex;
            var fillEndPosition;
            if (dataItem.get("isRange")) {
                fillEndIndex = endIndex;
            }
            else {
                fillEndIndex = index + this._frequency - 1;
            }
            fillEndPosition = this.indexToPosition(fillEndIndex, endCategoryLocation);
            renderer.updateLabel(dataItem.get("label"), position, endPosition, count);
            renderer.updateGrid(dataItem.get("grid"), position, endPosition);
            renderer.updateTick(dataItem.get("tick"), position, endPosition, count);
            renderer.updateFill(dataItem.get("axisFill"), position, fillEndPosition);
            this._processBullet(dataItem);
            renderer.updateBullet(dataItem.get("bullet"), position, endPosition);
            var fillRule = this.get("fillRule");
            if (fillRule) {
                fillRule(dataItem, fillIndex);
            }
        }
    });
    Object.defineProperty(CategoryAxis.prototype, "startIndex", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var len = this.dataItems.length;
            return Math.min(Math.max(this.getPrivate("startIndex", 0), 0), len - 1);
        }
    });
    Object.defineProperty(CategoryAxis.prototype, "endIndex", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var len = this.dataItems.length;
            return Math.max(1, Math.min(this.getPrivate("endIndex", len), len));
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(CategoryAxis.prototype, "baseValue", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(CategoryAxis.prototype, "basePosition", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return 0;
        }
    });
    /**
     * Returns X coordinate in pixels corresponding to specific category index.
     *
     * @param   value  Index
     * @return         X coordinate
     */
    Object.defineProperty(CategoryAxis.prototype, "getX", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (value) {
            var axisDataItem = this._itemMap[value];
            if (axisDataItem) {
                return this._settings.renderer.positionToCoordinate(this.indexToPosition(axisDataItem.get("index", 0)));
            }
            return NaN;
        }
    });
    /**
     * Returns Y coordinate in pixels corresponding to specific category index.
     *
     * @param   value  Index
     * @return         Y coordinate
     */
    Object.defineProperty(CategoryAxis.prototype, "getY", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (value) {
            var axisDataItem = this._itemMap[value];
            if (axisDataItem) {
                return this._settings.renderer.positionToCoordinate(this.indexToPosition(axisDataItem.get("index", 0)));
            }
            return NaN;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(CategoryAxis.prototype, "getDataItemPositionX", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, field, cellLocation, _axisLocation) {
            var category = dataItem.get(field);
            var axisDataItem = this._itemMap[category];
            if (axisDataItem) {
                return this.indexToPosition(axisDataItem.get("index", 0), cellLocation);
            }
            return NaN;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(CategoryAxis.prototype, "getDataItemCoordinateX", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, field, cellLocation, _axisLocation) {
            return this._settings.renderer.positionToCoordinate(this.getDataItemPositionX(dataItem, field, cellLocation, _axisLocation));
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(CategoryAxis.prototype, "getDataItemPositionY", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, field, cellLocation, _axisLocation) {
            var category = dataItem.get(field);
            var axisDataItem = this._itemMap[category];
            if (axisDataItem) {
                return this.indexToPosition(axisDataItem.get("index", 0), cellLocation);
            }
            return NaN;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(CategoryAxis.prototype, "getDataItemCoordinateY", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, field, cellLocation, _axisLocation) {
            return this._settings.renderer.positionToCoordinate(this.getDataItemPositionY(dataItem, field, cellLocation, _axisLocation));
        }
    });
    /**
     * Converts category index to a relative position.
     *
     * `location` indicates relative position within category: 0 - start, 1 - end.
     *
     * If not set, will use middle (0.5) of the category.
     *
     * @param   index     Index
     * @param   location  Location
     * @return            Index
     */
    Object.defineProperty(CategoryAxis.prototype, "indexToPosition", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (index, location) {
            if (!$type.isNumber(location)) {
                location = 0.5;
            }
            var len = this.dataItems.length;
            var startLocation = this.get("startLocation", 0);
            var endLocation = this.get("endLocation", 1);
            len -= startLocation;
            len -= (1 - endLocation);
            var position = (index + location - startLocation) / len;
            var dataItem = this.dataItems[index];
            if (dataItem) {
                position += dataItem.get("deltaPosition", 0);
            }
            return position;
        }
    });
    /**
     * Returns an index of a category.
     *
     * @param   category  Category to look up
     * @return            Index
     */
    Object.defineProperty(CategoryAxis.prototype, "categoryToIndex", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (category) {
            var dataItem = this._itemMap[category];
            if (dataItem) {
                return dataItem.get("index");
            }
            return NaN;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(CategoryAxis.prototype, "dataItemToPosition", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            return this.indexToPosition(dataItem.get("index"));
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(CategoryAxis.prototype, "roundAxisPosition", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (position, location) {
            return this.indexToPosition(this.axisPositionToIndex(position), location);
        }
    });
    /**
     * Returns an index of the category that corresponds to specific pixel
     * position within axis.
     *
     * @param position  Position (px)
     * @return Category index
     */
    Object.defineProperty(CategoryAxis.prototype, "axisPositionToIndex", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (position) {
            var len = this.dataItems.length;
            return $math.fitToRange(Math.floor(position * len), 0, len - 1); //$math.fitToRange(Math.floor((end - start) * len * position + len * start), 0, len - 1);
        }
    });
    /**
     * Returns text to be used in an axis tooltip for specific relative position.
     *
     * @param   position  Position
     * @return            Tooltip text
     */
    Object.defineProperty(CategoryAxis.prototype, "getTooltipText", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (position) {
            //@todo number formatter + tag
            var dataItem = this.dataItems[this.axisPositionToIndex(position)];
            if (dataItem) {
                var label = dataItem.get("label");
                if (label) {
                    return populateString(label, this.get("tooltipText", ""));
                }
            }
        }
    });
    Object.defineProperty(CategoryAxis.prototype, "_updateTooltipText", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (tooltip, position) {
            tooltip._setDataItem(this.dataItems[this.axisPositionToIndex(position)]);
            tooltip.label.text.markDirtyText();
        }
    });
    /**
     * Returns a data item from series that is closest to the `position`.
     *
     * @param   series    Series
     * @param   position  Relative position
     * @return            Data item
     */
    Object.defineProperty(CategoryAxis.prototype, "getSeriesItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (series, position) {
            if (this.dataItems.length > 0) {
                var fieldName = (this.getPrivate("name") + this.get("renderer").getPrivate("letter"));
                var index = this.axisPositionToIndex(position);
                // try simple first
                var seriesDataItem = series.dataItems[index];
                var axisDataItem = this.dataItems[index];
                var category = axisDataItem.get("category");
                if (seriesDataItem && axisDataItem) {
                    if (seriesDataItem.get(fieldName) === category) {
                        return seriesDataItem;
                    }
                }
                // if not found, try looking
                for (var i = 0, len = series.dataItems.length; i < len; i++) {
                    var dataItem = series.dataItems[i];
                    if (dataItem.get(fieldName) === category) {
                        return dataItem;
                    }
                }
            }
        }
    });
    /**
     * Zooms the axis to specific `start` and `end` indexes.
     *
     * Optional `duration` specifies duration of zoom animation in milliseconds.
     *
     * @param  start     Start index
     * @param  end       End index
     * @param  duration  Duration in milliseconds
     */
    Object.defineProperty(CategoryAxis.prototype, "zoomToIndexes", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (start, end, duration) {
            var len = this.dataItems.length;
            this.zoom(start / len, end / len, duration);
        }
    });
    Object.defineProperty(CategoryAxis.prototype, "zoomToCategories", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (startCategory, endCategory, duration) {
            this.zoomToIndexes(this.categoryToIndex(startCategory), this.categoryToIndex(endCategory) + 1, duration);
        }
    });
    Object.defineProperty(CategoryAxis, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "CategoryAxis"
    });
    Object.defineProperty(CategoryAxis, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: Axis.classNames.concat([CategoryAxis.className])
    });
    return CategoryAxis;
}(Axis));
export { CategoryAxis };
//# sourceMappingURL=CategoryAxis.js.map