import { __assign, __extends } from "tslib";
import { DateAxis } from "./DateAxis";
import { DataItem } from "../../../core/render/Component";
import * as $array from "../../../core/util/Array";
import * as $order from "../../../core/util/Order";
import * as $time from "../../../core/util/Time";
import * as $type from "../../../core/util/Type";
/**
 * A version of a [[DateAxis]] which removes intervals that don't have any data
 * items in them.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/xy-chart/axes/gapless-date-axis/} for more info
 * @important
 */
var GaplessDateAxis = /** @class */ (function (_super) {
    __extends(GaplessDateAxis, _super);
    function GaplessDateAxis() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "_frequency", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 1
        });
        Object.defineProperty(_this, "_dates", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        return _this;
    }
    Object.defineProperty(GaplessDateAxis.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this.valueFields.push("date");
            _super.prototype._afterNew.call(this);
        }
    });
    Object.defineProperty(GaplessDateAxis.prototype, "_updateDates", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (date) {
            var dates = this._dates;
            var result = $array.getSortedIndex(dates, function (x) { return $order.compare(x, date); });
            if (!result.found) {
                $array.insertIndex(dates, result.index, date);
            }
        }
    });
    Object.defineProperty(GaplessDateAxis.prototype, "_updateAllDates", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            this._dates.length = 0;
            $array.each(this.series, function (series) {
                var field = "valueX";
                if (series.get("yAxis") == _this) {
                    field = "valueY";
                }
                $array.each(series.dataItems, function (dataItem) {
                    var value = dataItem.get(field);
                    if ($type.isNumber(value)) {
                        if (dataItem.open) {
                            _this._updateDates(dataItem.open[field]);
                        }
                    }
                });
            });
        }
    });
    /**
     * Convers value to a relative position on axis.
     *
     * @param   value  Value
     * @return         Relative position
     */
    Object.defineProperty(GaplessDateAxis.prototype, "valueToPosition", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (value) {
            var dates = this._dates;
            var len = dates.length;
            var result = $array.getSortedIndex(dates, function (x) { return $order.compare(x, value); });
            var index = result.index;
            if (result.found) {
                return index / len;
            }
            else {
                if (index > 0) {
                    index -= 1;
                }
                var itemValue = dates[index];
                var d = 0;
                if (itemValue > value) {
                    d = itemValue - value;
                }
                else {
                    d = value - itemValue;
                }
                return index / len + d / this.baseDuration() / len;
            }
        }
    });
    /**
     * Converts numeric value from axis scale to index.
     *
     * @param  value  Value
     * @return        Index
     */
    Object.defineProperty(GaplessDateAxis.prototype, "valueToIndex", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (value) {
            var dates = this._dates;
            var result = $array.getSortedIndex(dates, function (x) { return $order.compare(x, value); });
            var index = result.index;
            if (result.found) {
                return index;
            }
            else {
                if (index > 0) {
                    index -= 1;
                }
                return index;
            }
        }
    });
    /**
     * Converts a relative position to a corresponding numeric value from axis
     * scale.
     *
     * @param   position  Relative position
     * @return            Value
     */
    Object.defineProperty(GaplessDateAxis.prototype, "positionToValue", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (position) {
            var len = this._dates.length;
            var index = position * len;
            var findex = Math.floor(index);
            if (findex < 0) {
                findex = 0;
            }
            if (findex > len - 1) {
                findex = len - 1;
            }
            return this._dates[findex] + (index - findex) * this.baseDuration();
        }
    });
    Object.defineProperty(GaplessDateAxis.prototype, "_prepareAxisItems", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            var startTime = this.getPrivate("selectionMin", 0);
            var endTime = this.getPrivate("selectionMax", 0);
            if ($type.isNumber(startTime) && $type.isNumber(endTime)) {
                if (this._seriesValuesDirty) {
                    this._seriesValuesDirty = false;
                    this._updateAllDates();
                }
                var dates_1 = this._dates;
                var renderer = this.get("renderer");
                var len = dates_1.length;
                var startIndex_1 = this.valueToIndex(startTime);
                if (startIndex_1 > 0) {
                    startIndex_1--;
                }
                var endIndex_1 = this.valueToIndex(endTime);
                if (endIndex_1 < len - 1) {
                    endIndex_1++;
                }
                var maxCount = renderer.axisLength() / Math.max(renderer.get("minGridDistance"), 1 / Number.MAX_SAFE_INTEGER);
                var frequency = Math.min(len, Math.ceil((endIndex_1 - startIndex_1) / maxCount));
                startIndex_1 = Math.floor(startIndex_1 / frequency) * frequency;
                this._frequency = frequency;
                for (var j = 0, length_1 = this.dataItems.length; j < length_1; j++) {
                    this.dataItems[j].hide();
                }
                var realDuration = (endTime - startTime);
                if (endIndex_1 - startIndex_1 < maxCount) {
                    realDuration = (endTime - startTime) - ((endTime - startTime) / this.baseDuration() - (endIndex_1 - startIndex_1)) * this.baseDuration();
                }
                // if all items are on axis
                var gridInterval_1 = $time.chooseInterval(0, realDuration, maxCount, this.get("gridIntervals"));
                var baseInterval = this.getPrivate("baseInterval");
                var intervalDuration_1 = $time.getIntervalDuration(gridInterval_1);
                if (intervalDuration_1 < this.baseDuration()) {
                    gridInterval_1 = __assign({}, baseInterval);
                    intervalDuration_1 = $time.getIntervalDuration(gridInterval_1);
                }
                this._intervalDuration = intervalDuration_1;
                var formats_1 = this.get("dateFormats");
                var previousIndex = -Infinity;
                var previousUnitValue = -Infinity;
                var selectedItems_1 = [];
                var changed = false;
                // 0, not a mistake, starting from start index is not good
                for (var i = 0; i < len; i++) {
                    var index = i;
                    var skip = false;
                    var value = dates_1[i];
                    var date = new Date(value);
                    var unitValue = $time.getUnitValue(date, gridInterval_1.timeUnit);
                    var shouldAdd = false;
                    if (gridInterval_1.timeUnit === "day" || gridInterval_1.timeUnit === "week") {
                        if (index - previousIndex >= frequency) {
                            shouldAdd = true;
                        }
                    }
                    else {
                        if (unitValue % gridInterval_1.count === 0) {
                            if (unitValue != previousUnitValue) {
                                shouldAdd = true;
                            }
                        }
                    }
                    if (shouldAdd) {
                        if (index - frequency * 0.7 < previousIndex) {
                            if (changed) {
                                skip = true;
                            }
                        }
                        if (!skip) {
                            selectedItems_1.push(i);
                            previousIndex = index;
                            previousUnitValue = unitValue;
                        }
                        changed = false;
                    }
                }
                if (selectedItems_1.length > 0) {
                    var i_1 = 0;
                    var previousValue_1 = -Infinity;
                    var nextGridUnit_1 = $time.getNextUnit(gridInterval_1.timeUnit);
                    $array.each(selectedItems_1, function (index) {
                        var dataItem;
                        if (_this.dataItems.length < i_1 + 1) {
                            dataItem = new DataItem(_this, undefined, {});
                            _this._dataItems.push(dataItem);
                            _this.processDataItem(dataItem);
                        }
                        else {
                            dataItem = _this.dataItems[i_1];
                        }
                        var value = dates_1[index];
                        var date = new Date(value);
                        var endValue = value;
                        if (i_1 < selectedItems_1.length - 1) {
                            endValue = dates_1[selectedItems_1[i_1 + 1]];
                        }
                        else {
                            endValue += intervalDuration_1;
                        }
                        dataItem.setRaw("value", value);
                        dataItem.setRaw("endValue", endValue);
                        dataItem.setRaw("index", i_1);
                        if (index > startIndex_1 - 100 && index < endIndex_1 + 100) {
                            var format = formats_1[gridInterval_1.timeUnit];
                            format = formats_1[gridInterval_1.timeUnit];
                            if (nextGridUnit_1 && _this.get("markUnitChange") && $type.isNumber(previousValue_1)) {
                                if (gridInterval_1.timeUnit != "year") {
                                    if ($time.checkChange(value, previousValue_1, nextGridUnit_1, _this._root.utc, _this._root.timezone)) {
                                        format = _this.get("periodChangeDateFormats")[gridInterval_1.timeUnit];
                                    }
                                }
                            }
                            _this._createAssets(dataItem, []);
                            var label = dataItem.get("label");
                            if (label) {
                                label.set("text", _this._root.dateFormatter.format(date, format));
                            }
                            if (dataItem.isHidden()) {
                                dataItem.show();
                            }
                            _this._prepareDataItem(dataItem, gridInterval_1.count);
                        }
                        i_1++;
                        previousValue_1 = value;
                    });
                }
                $array.each(this.series, function (series) {
                    if (series.inited) {
                        series._markDirtyAxes();
                    }
                });
            }
        }
    });
    Object.defineProperty(GaplessDateAxis, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "GaplessDateAxis"
    });
    Object.defineProperty(GaplessDateAxis, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: DateAxis.classNames.concat([GaplessDateAxis.className])
    });
    return GaplessDateAxis;
}(DateAxis));
export { GaplessDateAxis };
//# sourceMappingURL=GaplessDateAxis.js.map