import { __extends } from "tslib";
import { XYSeries } from "./XYSeries";
import { Graphics } from "../../../core/render/Graphics";
import * as $type from "../../../core/util/Type";
import * as $array from "../../../core/util/Array";
import { line, area } from "d3-shape";
import { Template } from "../../../core/util/Template";
import { ListTemplate } from "../../../core/util/List";
import { DataItem } from "../../../core/render/Component";
import * as $utils from "../../../core/util/Utils";
/**
 * Used to plot line and/or area series.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/xy-chart/series/line-series/} for more info
 * @important
 */
var LineSeries = /** @class */ (function (_super) {
    __extends(LineSeries, _super);
    function LineSeries() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "_endIndex", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_strokeGenerator", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: line()
        });
        Object.defineProperty(_this, "_fillGenerator", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: area()
        });
        Object.defineProperty(_this, "_legendStroke", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_legendFill", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        /**
         * A [[TemplateList]] of all line segments in series.
         *
         * `strokes.template` can be used to set default settings for all line
         * segments, or to change on existing ones.
         *
         * @default new ListTemplate<Graphics>
         */
        Object.defineProperty(_this, "strokes", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new ListTemplate(Template.new({}), function () { return Graphics._new(_this._root, {
                themeTags: $utils.mergeTags(_this.strokes.template.get("themeTags", []), ["line", "series", "stroke"])
            }, [_this.strokes.template]); })
        });
        /**
         * A [[TemplateList]] of all segment fills in series.
         *
         * `fills.template` can be used to set default settings for all segment
         * fills, or to change on existing ones.
         *
         * @default new ListTemplate<Graphics>
         */
        Object.defineProperty(_this, "fills", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new ListTemplate(Template.new({}), function () { return Graphics._new(_this._root, {
                themeTags: $utils.mergeTags(_this.strokes.template.get("themeTags", []), ["line", "series", "fill"])
            }, [_this.fills.template]); })
        });
        // custom set from data
        Object.defineProperty(_this, "_fillTemplate", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_strokeTemplate", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        return _this;
    }
    Object.defineProperty(LineSeries.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._fillGenerator.y0(function (p) {
                return p[3];
            });
            this._fillGenerator.x0(function (p) {
                return p[2];
            });
            this._fillGenerator.y1(function (p) {
                return p[1];
            });
            this._fillGenerator.x1(function (p) {
                return p[0];
            });
            _super.prototype._afterNew.call(this);
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(LineSeries.prototype, "makeStroke", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (strokes) {
            var stroke = this.mainContainer.children.push(strokes.make());
            strokes.push(stroke);
            return stroke;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(LineSeries.prototype, "makeFill", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (fills) {
            var fill = this.mainContainer.children.push(fills.make());
            fills.push(fill);
            return fill;
        }
    });
    Object.defineProperty(LineSeries.prototype, "_updateChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._strokeTemplate = undefined;
            this._fillTemplate = undefined;
            var xAxis = this.get("xAxis");
            var yAxis = this.get("yAxis");
            if (this.isDirty("stroke")) {
                var stroke = this.get("stroke");
                this.strokes.template.set("stroke", stroke);
                var legendStroke = this._legendStroke;
                if (legendStroke) {
                    legendStroke.states.lookup("default").set("stroke", stroke);
                }
            }
            if (this.isDirty("fill")) {
                var fill = this.get("fill");
                this.fills.template.set("fill", fill);
                var legendFill = this._legendFill;
                if (legendFill) {
                    legendFill.states.lookup("default").set("fill", fill);
                }
            }
            if (this.isDirty("curveFactory")) {
                var curveFactory = this.get("curveFactory");
                if (curveFactory) {
                    this._strokeGenerator.curve(curveFactory);
                    this._fillGenerator.curve(curveFactory);
                }
            }
            if (xAxis.inited && yAxis.inited) {
                if (this._axesDirty || this._valuesDirty || this._stackDirty || this.isDirty("vcx") || this.isDirty("vcy") || this._sizeDirty || this.isDirty("connect") || this.isDirty("curveFactory")) {
                    this.fills.each(function (fill) {
                        fill.setPrivate("visible", false);
                    });
                    this.strokes.each(function (fill) {
                        fill.setPrivate("visible", false);
                    });
                    this.axisRanges.each(function (axisRange) {
                        var fills = axisRange.fills;
                        if (fills) {
                            fills.each(function (fill) {
                                fill.setPrivate("visible", false);
                            });
                        }
                        var strokes = axisRange.strokes;
                        if (strokes) {
                            strokes.each(function (stroke) {
                                stroke.setPrivate("visible", false);
                            });
                        }
                    });
                    var startIndex = this.startIndex();
                    var strokeTemplateField = this.strokes.template.get("templateField");
                    var fillTemplateField = this.fills.template.get("templateField");
                    var strokeTemplateFound = true;
                    var fillTemplateFound = true;
                    if (strokeTemplateField) {
                        strokeTemplateFound = false;
                    }
                    if (fillTemplateField) {
                        fillTemplateFound = false;
                    }
                    var _loop_1 = function (i) {
                        var dataItem = this_1.dataItems[i];
                        var hasValues = true;
                        var dataContext = dataItem.dataContext;
                        if (strokeTemplateField) {
                            if (dataContext[strokeTemplateField]) {
                                strokeTemplateFound = true;
                            }
                        }
                        if (fillTemplateField) {
                            if (dataContext[fillTemplateField]) {
                                fillTemplateFound = true;
                            }
                        }
                        $array.each(this_1._valueFields, function (field) {
                            if (!$type.isNumber(dataItem.get(field))) {
                                hasValues = false;
                            }
                        });
                        if (hasValues && strokeTemplateFound && fillTemplateFound) {
                            startIndex = i;
                            return "break";
                        }
                    };
                    var this_1 = this;
                    for (var i = startIndex - 1; i >= 0; i--) {
                        var state_1 = _loop_1(i);
                        if (state_1 === "break")
                            break;
                    }
                    var len = this.dataItems.length;
                    var endIndex = this.endIndex();
                    if (endIndex < len) {
                        endIndex++;
                        var _loop_2 = function (i) {
                            var dataItem = this_2.dataItems[i];
                            var hasValues = true;
                            $array.each(this_2._valueFields, function (field) {
                                if (!$type.isNumber(dataItem.get(field))) {
                                    hasValues = false;
                                }
                            });
                            if (hasValues) {
                                endIndex = i + 1;
                                return "break";
                            }
                        };
                        var this_2 = this;
                        for (var i = endIndex; i < len; i++) {
                            var state_2 = _loop_2(i);
                            if (state_2 === "break")
                                break;
                        }
                    }
                    this._endIndex = endIndex;
                    this._clearGraphics();
                    this._startSegment(0, startIndex);
                }
            }
            else {
                this._skipped = true;
            }
            _super.prototype._updateChildren.call(this);
        }
    });
    Object.defineProperty(LineSeries.prototype, "_clearGraphics", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this.strokes.clear();
            this.fills.clear();
        }
    });
    Object.defineProperty(LineSeries.prototype, "_startSegment", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (segmentIndex, dataItemIndex) {
            var _this = this;
            var endIndex = this._endIndex;
            var currentEndIndex = endIndex;
            var autoGapCount = this.get("autoGapCount");
            var connect = this.get("connect");
            var fill = this.makeFill(this.fills);
            var fillTemplate = this._fillTemplate;
            var originalTemplate = this.fills.template;
            if (fillTemplate && fillTemplate != originalTemplate) {
                fill.template = fillTemplate;
            }
            fill.setPrivate("visible", true);
            var stroke = this.makeStroke(this.strokes);
            var strokeTemplate = this._strokeTemplate;
            if (strokeTemplate && strokeTemplate != this.strokes.template) {
                stroke.template = strokeTemplate;
            }
            stroke.setPrivate("visible", true);
            var xAxis = this.get("xAxis");
            var yAxis = this.get("yAxis");
            var baseAxis = this.get("baseAxis");
            var vcx = this.get("vcx", 1);
            var vcy = this.get("vcy", 1);
            var xField = this._xField;
            var yField = this._yField;
            var xOpenField = this._xOpenField;
            var yOpenField = this._yOpenField;
            var xOpenFieldValue = this.get("openValueXField");
            var yOpenFieldValue = this.get("openValueYField");
            if (!xOpenFieldValue) {
                xOpenField = this._xField;
            }
            if (!yOpenFieldValue) {
                yOpenField = this._yField;
            }
            var stacked = this.get("stacked");
            var basePosX = xAxis.basePosition();
            var basePosY = yAxis.basePosition();
            var baseField;
            if (baseAxis === yAxis) {
                baseField = this._yField;
            }
            else {
                baseField = this._xField;
            }
            var segments = [];
            var points = [];
            segments.push(points);
            var strokeTemplateField = this.strokes.template.get("templateField");
            var fillTemplateField = this.fills.template.get("templateField");
            var locationX = this.get("locationX", 0.5);
            var locationY = this.get("locationY", 0.5);
            var openLocationX = this.get("openLocationX", locationX);
            var openLocationY = this.get("openLocationY", locationY);
            var i;
            var fillVisible = this.fills.template.get("visible");
            if (this.axisRanges.length > 0) {
                fillVisible = true;
            }
            var getOpen = false;
            if (stacked || xOpenFieldValue || yOpenFieldValue) {
                getOpen = true;
            }
            var o = {
                points: points,
                segments: segments,
                stacked: stacked,
                getOpen: getOpen,
                basePosX: basePosX,
                basePosY: basePosY,
                fillVisible: fillVisible,
                xField: xField,
                yField: yField,
                xOpenField: xOpenField,
                yOpenField: yOpenField,
                vcx: vcx,
                vcy: vcy,
                baseAxis: baseAxis,
                xAxis: xAxis,
                yAxis: yAxis,
                locationX: locationX,
                locationY: locationY,
                openLocationX: openLocationX,
                openLocationY: openLocationY
            };
            for (i = dataItemIndex; i < currentEndIndex; i++) {
                var dataItem = this._dataItems[i];
                var valueX = dataItem.get(xField);
                var valueY = dataItem.get(yField);
                if (valueX == null || valueY == null) {
                    if (!connect) {
                        points = [];
                        segments.push(points);
                        o.points = points;
                    }
                }
                else {
                    this._getPoints(dataItem, o);
                }
                if (strokeTemplateField) {
                    var strokeTemplate_1 = dataItem.dataContext[strokeTemplateField];
                    if (strokeTemplate_1) {
                        if (!(strokeTemplate_1 instanceof Template)) {
                            strokeTemplate_1 = Template.new(strokeTemplate_1);
                        }
                        this._strokeTemplate = strokeTemplate_1;
                        if (i > dataItemIndex) {
                            currentEndIndex = i;
                            break;
                        }
                        else {
                            stroke.template = strokeTemplate_1;
                        }
                    }
                }
                if (fillTemplateField) {
                    var fillTemplate_1 = dataItem.dataContext[fillTemplateField];
                    if (fillTemplate_1) {
                        if (!(fillTemplate_1 instanceof Template)) {
                            fillTemplate_1 = Template.new(fillTemplate_1);
                        }
                        this._fillTemplate = fillTemplate_1;
                        if (i > dataItemIndex) {
                            currentEndIndex = i;
                            break;
                        }
                        else {
                            fill.template = fillTemplate_1;
                        }
                    }
                }
                if (!connect) {
                    var nextItem = this.dataItems[i + 1];
                    if (nextItem) {
                        if (baseAxis.shouldGap(dataItem, nextItem, autoGapCount, baseField)) {
                            points = [];
                            segments.push(points);
                            o.points = points;
                        }
                    }
                }
            }
            fill.setRaw("userData", [dataItemIndex, i]);
            stroke.setRaw("userData", [dataItemIndex, i]);
            if (i === endIndex) {
                this._endLine(points, segments[0][0]);
            }
            if (stroke) {
                this._drawStroke(stroke, segments);
            }
            if (fill) {
                this._drawFill(fill, segments);
            }
            this.axisRanges.each(function (axisRange) {
                var container = axisRange.container;
                var fills = axisRange.fills;
                var fill = _this.makeFill(fills);
                if (container) {
                    container.children.push(fill);
                }
                fill.setPrivate("visible", true);
                _this._drawFill(fill, segments);
                var strokes = axisRange.strokes;
                var stroke = _this.makeStroke(strokes);
                if (container) {
                    container.children.push(stroke);
                }
                stroke.setPrivate("visible", true);
                _this._drawStroke(stroke, segments);
                fill.setRaw("userData", [dataItemIndex, i]);
                stroke.setRaw("userData", [dataItemIndex, i]);
            });
            if (currentEndIndex < endIndex) {
                this._startSegment(segmentIndex + 1, currentEndIndex);
            }
        }
    });
    Object.defineProperty(LineSeries.prototype, "_getPoints", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, o) {
            var points = o.points;
            var itemLocationX = dataItem.get("locationX", o.locationX);
            var itemLocationY = dataItem.get("locationY", o.locationY);
            var xPos = o.xAxis.getDataItemPositionX(dataItem, o.xField, itemLocationX, o.vcx);
            var yPos = o.yAxis.getDataItemPositionY(dataItem, o.yField, itemLocationY, o.vcy);
            if (this._shouldInclude(xPos)) {
                var iPoint = this.getPoint(xPos, yPos);
                var point = [iPoint.x, iPoint.y];
                iPoint.x += this._x;
                iPoint.y += this._y;
                dataItem.set("point", iPoint);
                if (o.fillVisible) {
                    var xPos0 = xPos;
                    var yPos0 = yPos;
                    if (o.baseAxis === o.xAxis) {
                        yPos0 = o.basePosY;
                    }
                    else if (o.baseAxis === o.yAxis) {
                        xPos0 = o.basePosX;
                    }
                    if (o.getOpen) {
                        var valueX = dataItem.get(o.xOpenField);
                        var valueY = dataItem.get(o.yOpenField);
                        if (valueX != null && valueY != null) {
                            var itemLocationX_1 = dataItem.get("openLocationX", o.openLocationX);
                            var itemLocationY_1 = dataItem.get("openLocationY", o.openLocationY);
                            if (o.stacked) {
                                var stackToItemX = dataItem.get("stackToItemX");
                                var stackToItemY = dataItem.get("stackToItemY");
                                if (stackToItemX) {
                                    xPos0 = o.xAxis.getDataItemPositionX(stackToItemX, o.xField, itemLocationX_1, stackToItemX.component.get("vcx"));
                                    if ($type.isNaN(xPos0)) {
                                        xPos0 = o.basePosX;
                                    }
                                }
                                else {
                                    if (o.yAxis === o.baseAxis) {
                                        xPos0 = o.basePosX;
                                    }
                                    else {
                                        xPos0 = o.xAxis.getDataItemPositionX(dataItem, o.xOpenField, itemLocationX_1, o.vcx);
                                    }
                                }
                                if (stackToItemY) {
                                    yPos0 = o.yAxis.getDataItemPositionY(stackToItemY, o.yField, itemLocationY_1, stackToItemY.component.get("vcy"));
                                    if ($type.isNaN(yPos0)) {
                                        yPos0 = o.basePosY;
                                    }
                                }
                                else {
                                    if (o.xAxis === o.baseAxis) {
                                        yPos0 = o.basePosY;
                                    }
                                    else {
                                        yPos0 = o.yAxis.getDataItemPositionY(dataItem, o.yOpenField, itemLocationY_1, o.vcy);
                                    }
                                }
                            }
                            else {
                                xPos0 = o.xAxis.getDataItemPositionX(dataItem, o.xOpenField, itemLocationX_1, o.vcx);
                                yPos0 = o.yAxis.getDataItemPositionY(dataItem, o.yOpenField, itemLocationY_1, o.vcy);
                            }
                        }
                    }
                    var closeIPoint = this.getPoint(xPos0, yPos0);
                    point[2] = closeIPoint.x;
                    point[3] = closeIPoint.y;
                }
                points.push(point);
            }
        }
    });
    Object.defineProperty(LineSeries.prototype, "_endLine", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_points, _firstPoint) {
        }
    });
    Object.defineProperty(LineSeries.prototype, "_drawStroke", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (graphics, segments) {
            var _this = this;
            if (graphics.get("visible")) {
                graphics.set("draw", function (display) {
                    $array.each(segments, function (segment) {
                        _this._strokeGenerator.context(display);
                        _this._strokeGenerator(segment);
                    });
                });
            }
        }
    });
    Object.defineProperty(LineSeries.prototype, "_drawFill", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (graphics, segments) {
            var _this = this;
            if (graphics.get("visible")) {
                graphics.set("draw", function (display) {
                    $array.each(segments, function (segment) {
                        _this._fillGenerator.context(display);
                        _this._fillGenerator(segment);
                    });
                });
            }
        }
    });
    Object.defineProperty(LineSeries.prototype, "_processAxisRange", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (axisRange) {
            var _this = this;
            _super.prototype._processAxisRange.call(this, axisRange);
            axisRange.fills = new ListTemplate(Template.new({}), function () { return Graphics._new(_this._root, {
                themeTags: $utils.mergeTags(axisRange.fills.template.get("themeTags", []), ["line", "series", "fill"]),
            }, [_this.fills.template, axisRange.fills.template]); });
            axisRange.strokes = new ListTemplate(Template.new({}), function () { return Graphics._new(_this._root, {
                themeTags: $utils.mergeTags(axisRange.strokes.template.get("themeTags", []), ["line", "series", "stroke"]),
            }, [_this.strokes.template, axisRange.strokes.template]); });
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(LineSeries.prototype, "createLegendMarker", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_dataItem) {
            var legendDataItem = this.get("legendDataItem");
            if (legendDataItem) {
                var marker = legendDataItem.get("marker");
                var markerRectangle = legendDataItem.get("markerRectangle");
                if (markerRectangle) {
                    markerRectangle.setPrivate("visible", false);
                }
                var legendStroke = marker.children.push(Graphics._new(marker._root, {
                    themeTags: ["line", "series", "legend", "marker", "stroke"]
                }, [this.strokes.template]));
                this._legendStroke = legendStroke;
                var legendFill = marker.children.push(Graphics._new(marker._root, {
                    themeTags: ["line", "series", "legend", "marker", "fill"]
                }, [this.fills.template]));
                this._legendFill = legendFill;
                var disabledColor = this._root.interfaceColors.get("disabled");
                legendStroke.states.create("disabled", { fill: disabledColor, stroke: disabledColor });
                legendFill.states.create("disabled", { fill: disabledColor, stroke: disabledColor });
                if (this.bullets.length > 0) {
                    var bulletFunction = this.bullets.getIndex(0);
                    if (bulletFunction) {
                        var bullet = bulletFunction(marker._root, this, new DataItem(this, {}, {}));
                        if (bullet) {
                            var sprite = bullet.get("sprite");
                            if (sprite instanceof Graphics) {
                                sprite.states.create("disabled", { fill: disabledColor, stroke: disabledColor });
                            }
                            if (sprite) {
                                sprite.set("tooltipText", undefined);
                                marker.children.push(sprite);
                                sprite.setAll({ x: marker.width() / 2, y: marker.height() / 2 });
                            }
                        }
                    }
                }
            }
        }
    });
    Object.defineProperty(LineSeries, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "LineSeries"
    });
    Object.defineProperty(LineSeries, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: XYSeries.classNames.concat([LineSeries.className])
    });
    return LineSeries;
}(XYSeries));
export { LineSeries };
//# sourceMappingURL=LineSeries.js.map