import { Template } from "./util/Template";
import * as $order from "./util/Order";
import * as $array from "./util/Array";
/**
 * A base class for an amCharts theme.
 *
 * @see {@link https://www.amcharts.com/docs/v5/concepts/themes/} for more info
 * @important
 */
var Theme = /** @class */ (function () {
    function Theme(root, isReal) {
        Object.defineProperty(this, "_root", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_rules", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        this._root = root;
        if (!isReal) {
            throw new Error("You cannot use `new Class()`, instead use `Class.new()`");
        }
    }
    /**
     * Use this method to create an instance of this class.
     *
     * @see {@link https://www.amcharts.com/docs/v5/getting-started/#New_element_syntax} for more info
     * @param   root      Root element
     * @param   settings  Settings
     * @param   template  Template
     * @return            Instantiated object
     */
    Object.defineProperty(Theme, "new", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (root) {
            var x = (new this(root, true));
            x.setupDefaultRules();
            return x;
        }
    });
    Object.defineProperty(Theme.prototype, "setupDefaultRules", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () { }
    });
    /**
     * Looks up the rules for a specific theme class.
     *
     * @see {@link https://www.amcharts.com/docs/v5/themes/} for more info
     * @param   themeClass Theme class
     * @return             Array<IRule<A>>
     */
    Object.defineProperty(Theme.prototype, "_lookupRules", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (themeClass) {
            return this._rules[themeClass];
        }
    });
    /**
     * Creates a [[Template]] for specific theme class and tags.
     *
     * NOTE: the difference from `rule()` is that `ruleRaw()` does not do any
     * type checks.
     *
     * @see {@link https://www.amcharts.com/docs/v5/themes/} for more info
     * @param   themeClass Theme class
     * @param   themeTags  Theme tags
     * @return             Template
     */
    Object.defineProperty(Theme.prototype, "ruleRaw", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (themeClass, themeTags) {
            if (themeTags === void 0) { themeTags = []; }
            var rules = this._rules[themeClass];
            if (!rules) {
                rules = this._rules[themeClass] = [];
            }
            themeTags.sort($order.compare);
            var _a = $array.getSortedIndex(rules, function (x) {
                var order = $order.compare(x.tags.length, themeTags.length);
                if (order === 0) {
                    return $order.compareArray(x.tags, themeTags, $order.compare);
                }
                else {
                    return order;
                }
            }), index = _a.index, found = _a.found;
            if (found) {
                return rules[index].template;
            }
            else {
                var template = Template.new({});
                rules.splice(index, 0, {
                    tags: themeTags,
                    template: template,
                });
                return template;
            }
        }
    });
    /**
     * Creates a [[Template]] for specific theme class and tags.
     *
     * @see {@link https://www.amcharts.com/docs/v5/themes/} for more info
     * @param   themeClass Theme class
     * @param   themeTags  Theme tags
     * @return             Template
     */
    Object.defineProperty(Theme.prototype, "rule", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (themeClass, themeTags) {
            if (themeTags === void 0) { themeTags = []; }
            return this.ruleRaw(themeClass, themeTags);
        }
    });
    return Theme;
}());
export { Theme };
//# sourceMappingURL=Theme.js.map