import { __extends } from "tslib";
import { Children } from "../util/Children";
import { Percent } from "../util/Percent";
import { Sprite } from "./Sprite";
import { Rectangle } from "./Rectangle";
import { HorizontalLayout } from "./HorizontalLayout";
import { VerticalLayout } from "./VerticalLayout";
import { GridLayout } from "./GridLayout";
import * as $array from "../util/Array";
import * as $type from "../util/Type";
/**
 * A basic element that can have child elements, maintain their layout, and
 * have a background.
 *
 * It can have any [[Sprite]] element as a child, from very basic shapes, to
 * full-fledged charts.
 *
 * @see {@link https://www.amcharts.com/docs/v5/concepts/common-elements/containers/} for more info
 * @important
 */
var Container = /** @class */ (function (_super) {
    __extends(Container, _super);
    function Container() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "_display", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this._root._renderer.makeContainer()
        });
        Object.defineProperty(_this, "_childrenDisplay", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this._root._renderer.makeContainer()
        });
        /**
         * List of Container's child elements.
         */
        Object.defineProperty(_this, "children", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Children(_this)
        });
        Object.defineProperty(_this, "_percentageSizeChildren", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        Object.defineProperty(_this, "_percentagePositionChildren", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        Object.defineProperty(_this, "_prevWidth", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "_prevHeight", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "_contentWidth", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "_contentHeight", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "_contentMask", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        return _this;
    }
    Object.defineProperty(Container.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._afterNew.call(this);
            this._display.addChild(this._childrenDisplay);
        }
    });
    Object.defineProperty(Container.prototype, "_dispose", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            $array.eachReverse(this.allChildren(), function (child) {
                child.dispose();
            });
            _super.prototype._dispose.call(this);
        }
    });
    Object.defineProperty(Container.prototype, "_changed", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._changed.call(this);
            if (this.isDirty("interactiveChildren")) {
                this._display.interactiveChildren = this.get("interactiveChildren", false);
            }
            if (this.isDirty("layout")) {
                this._prevWidth = 0;
                this._prevHeight = 0;
                this.markDirtyBounds();
                if (this._prevSettings.layout) {
                    this.children.each(function (child) {
                        child.removePrivate("x");
                        child.removePrivate("y");
                    });
                }
            }
            if (this.isDirty("paddingTop") || this.isDirty("paddingBottom") || this.isDirty("paddingLeft") || this.isDirty("paddingRight")) {
                this.children.each(function (child) {
                    child.markDirtyPosition();
                });
            }
            if (this.isDirty("maskContent")) {
                var childrenDisplay = this._childrenDisplay;
                var contentMask = this._contentMask;
                if (this.get("maskContent")) {
                    if (!contentMask) {
                        contentMask = Rectangle.new(this._root, {
                            width: this.width(),
                            height: this.height()
                        });
                        this._contentMask = contentMask;
                        childrenDisplay.addChildAt(contentMask._display, 0);
                        childrenDisplay.mask = contentMask._display;
                    }
                }
                else {
                    if (contentMask) {
                        childrenDisplay.removeChild(contentMask._display);
                        childrenDisplay.mask = null;
                        contentMask.dispose();
                    }
                }
            }
        }
    });
    Object.defineProperty(Container.prototype, "_updateSize", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._updateSize.call(this);
            $array.each(this._percentageSizeChildren, function (child) {
                child._updateSize();
            });
            $array.each(this._percentagePositionChildren, function (child) {
                child.markDirtyPosition();
                child._updateSize();
            });
            this.updateBackground();
        }
    });
    Object.defineProperty(Container.prototype, "updateBackground", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var background = this.get("background");
            var bounds = this._localBounds;
            if (bounds && !this.isHidden()) {
                var x = bounds.left;
                var y = bounds.top;
                var w = bounds.right - x;
                var h = bounds.bottom - y;
                var width = this.width();
                var height = this.height();
                if (background) {
                    background.setAll({ width: w, height: h, x: x, y: y });
                    if (this._display.interactive) {
                        background._display.interactive = true;
                    }
                }
                var contentMask = this._contentMask;
                if (contentMask) {
                    contentMask.setAll({ width: width, height: height });
                }
                var verticalScrollbar = this.get("verticalScrollbar");
                if (verticalScrollbar) {
                    verticalScrollbar.set("height", height);
                    verticalScrollbar.set("x", width - verticalScrollbar.width() - verticalScrollbar.get("marginRight", 0));
                    verticalScrollbar.set("end", verticalScrollbar.get("start", 0) + height / this._contentHeight);
                    var bg = verticalScrollbar.get("background");
                    if (bg) {
                        bg.setAll({ width: verticalScrollbar.width(), height: height });
                    }
                    var visible = true;
                    if (this._contentHeight <= height) {
                        visible = false;
                    }
                    verticalScrollbar.setPrivate("visible", visible);
                }
            }
        }
    });
    Object.defineProperty(Container.prototype, "_applyThemes", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (_super.prototype._applyThemes.call(this)) {
                this.eachChildren(function (child) {
                    child._applyThemes();
                });
                return true;
            }
            else {
                return false;
            }
        }
    });
    Object.defineProperty(Container.prototype, "_applyState", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (name) {
            _super.prototype._applyState.call(this, name);
            if (this.get("setStateOnChildren")) {
                this.eachChildren(function (child) {
                    child.states.apply(name);
                });
            }
        }
    });
    Object.defineProperty(Container.prototype, "_applyStateAnimated", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (name, duration) {
            _super.prototype._applyStateAnimated.call(this, name, duration);
            if (this.get("setStateOnChildren")) {
                this.eachChildren(function (child) {
                    child.states.applyAnimate(name, duration);
                });
            }
        }
    });
    /**
     * Returns container's inner width (width without padding) in pixels.
     *
     * @return Inner width (px)
     */
    Object.defineProperty(Container.prototype, "innerWidth", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this.width() - this.get("paddingRight", 0) - this.get("paddingLeft", 0);
        }
    });
    /**
     * Returns container's inner height (height without padding) in pixels.
     *
     * @return Inner height (px)
     */
    Object.defineProperty(Container.prototype, "innerHeight", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this.height() - this.get("paddingTop", 0) - this.get("paddingBottom", 0);
        }
    });
    Object.defineProperty(Container.prototype, "_getBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var width = this.get("width");
            var height = this.get("height");
            var pWidth = this.getPrivate("width");
            var pHeight = this.getPrivate("height");
            var bounds = {
                left: 0,
                top: 0,
                right: this.width(),
                bottom: this.height(),
            };
            var layout = this.get("layout");
            var horizontal = false;
            var vertical = false;
            if (layout instanceof HorizontalLayout || layout instanceof GridLayout) {
                horizontal = true;
            }
            if (layout instanceof VerticalLayout) {
                vertical = true;
            }
            if ((width != null || pWidth != null) && (height != null || pHeight != null) && !this.get("verticalScrollbar")) {
                // void
            }
            else {
                var m = Number.MAX_VALUE;
                var l_1 = m;
                var r_1 = -m;
                var t_1 = m;
                var b_1 = -m;
                var paddingLeft = this.get("paddingLeft", 0);
                var paddingTop = this.get("paddingTop", 0);
                var paddingRight = this.get("paddingRight", 0);
                var paddingBottom = this.get("paddingBottom", 0);
                this.children.each(function (child) {
                    if (child.get("position") != "absolute" && child.get("isMeasured")) {
                        var childBounds = child.adjustedLocalBounds();
                        var childX = child.x();
                        var childY = child.y();
                        var cl = childX + childBounds.left;
                        var cr = childX + childBounds.right;
                        var ct = childY + childBounds.top;
                        var cb = childY + childBounds.bottom;
                        if (horizontal) {
                            cl -= child.get("marginLeft", 0);
                            cr += child.get("marginRight", 0);
                        }
                        if (vertical) {
                            ct -= child.get("marginTop", 0);
                            cb += child.get("marginBottom", 0);
                        }
                        if (cl < l_1) {
                            l_1 = cl;
                        }
                        if (cr > r_1) {
                            r_1 = cr;
                        }
                        if (ct < t_1) {
                            t_1 = ct;
                        }
                        if (cb > b_1) {
                            b_1 = cb;
                        }
                    }
                });
                if (l_1 == m) {
                    l_1 = 0;
                }
                if (r_1 == -m) {
                    r_1 = 0;
                }
                if (t_1 == m) {
                    t_1 = 0;
                }
                if (b_1 == -m) {
                    b_1 = 0;
                }
                bounds.left = l_1 - paddingLeft;
                bounds.top = t_1 - paddingTop;
                bounds.right = r_1 + paddingRight;
                bounds.bottom = b_1 + paddingBottom;
            }
            this._contentWidth = bounds.right - bounds.left;
            this._contentHeight = bounds.bottom - bounds.top;
            if ($type.isNumber(width)) {
                bounds.left = 0;
                bounds.right = width;
            }
            if ($type.isNumber(pWidth)) {
                bounds.left = 0;
                bounds.right = pWidth;
            }
            if ($type.isNumber(height)) {
                bounds.top = 0;
                bounds.bottom = height;
            }
            if ($type.isNumber(pHeight)) {
                bounds.top = 0;
                bounds.bottom = pHeight;
            }
            this._localBounds = bounds;
        }
    });
    Object.defineProperty(Container.prototype, "_updateBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var layout = this.get("layout");
            if (layout) {
                layout.updateContainer(this);
            }
            _super.prototype._updateBounds.call(this);
            this.updateBackground();
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Container.prototype, "markDirty", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype.markDirty.call(this);
            this._root._addDirtyParent(this);
        }
    });
    Object.defineProperty(Container.prototype, "_prepareChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var innerWidth = this.innerWidth();
            var innerHeight = this.innerHeight();
            if (innerWidth != this._prevWidth || innerHeight != this._prevHeight) {
                var layout = this.get("layout");
                var horizontal_1 = false;
                var vertical_1 = false;
                if (layout) {
                    if (layout instanceof HorizontalLayout || layout instanceof GridLayout) {
                        horizontal_1 = true;
                    }
                    if (layout instanceof VerticalLayout) {
                        vertical_1 = true;
                    }
                }
                $array.each(this._percentageSizeChildren, function (child) {
                    if (!horizontal_1) {
                        var width = child.get("width");
                        if (width instanceof Percent) {
                            child.setPrivate("width", width.value * innerWidth);
                        }
                    }
                    if (!vertical_1) {
                        var height = child.get("height");
                        if (height instanceof Percent) {
                            child.setPrivate("height", height.value * innerHeight);
                        }
                    }
                });
                $array.each(this._percentagePositionChildren, function (child) {
                    child.markDirtyPosition();
                    child.markDirtyBounds();
                });
                this._prevWidth = innerWidth;
                this._prevHeight = innerHeight;
                this._sizeDirty = true;
                this.updateBackground();
            }
            this._handleStates();
        }
    });
    Object.defineProperty(Container.prototype, "_updateChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            if (this.isDirty("verticalScrollbar")) {
                var verticalScrollbar_1 = this.get("verticalScrollbar");
                if (verticalScrollbar_1) {
                    verticalScrollbar_1._setParent(this);
                    verticalScrollbar_1.children.removeValue(verticalScrollbar_1.startGrip);
                    verticalScrollbar_1.children.removeValue(verticalScrollbar_1.endGrip);
                    this.set("maskContent", true);
                    this.set("paddingRight", verticalScrollbar_1.width() + verticalScrollbar_1.get("marginRight", 0) + verticalScrollbar_1.get("marginLeft", 0));
                    var background = this.get("background");
                    if (!background) {
                        background = this.set("background", Rectangle.new(this._root, {
                            themeTags: ["background"],
                            fillOpacity: 0,
                            fill: this._root.interfaceColors.get("alternativeBackground")
                        }));
                    }
                    this._disposers.push(this.events.on("wheel", function (event) {
                        var wheelEvent = event.originalEvent;
                        wheelEvent.preventDefault();
                        var shiftY = wheelEvent.deltaY / 5000;
                        var start = verticalScrollbar_1.get("start", 0);
                        var end = verticalScrollbar_1.get("end", 1);
                        if (start + shiftY > 0 && end + shiftY < 1) {
                            verticalScrollbar_1.set("start", start + shiftY);
                            verticalScrollbar_1.set("end", end + shiftY);
                        }
                    }));
                    this._disposers.push(verticalScrollbar_1.events.on("rangechanged", function () {
                        var h = _this._contentHeight;
                        var childrenDisplay = _this._childrenDisplay;
                        var contentMask = _this._contentMask;
                        childrenDisplay.y = -verticalScrollbar_1.get("start") * h;
                        childrenDisplay.markDirtyLayer();
                        if (contentMask) {
                            contentMask._display.y = -childrenDisplay.y;
                            childrenDisplay.mask = contentMask._display;
                        }
                    }));
                    this._display.addChild(verticalScrollbar_1._display);
                }
            }
            if (this.isDirty("background")) {
                // TODO maybe this should dispose ?
                var previous = this._prevSettings["background"];
                if (previous) {
                    this._display.removeChild(previous._display);
                }
                var background = this.get("background");
                if (background instanceof Sprite) {
                    background.set("isMeasured", false);
                    background._setParent(this);
                    this._display.addChildAt(background._display, 0);
                }
            }
            if (this.isDirty("mask")) {
                var mask = this.get("mask");
                var previous = this._prevSettings["mask"];
                if (previous) {
                    this._display.removeChild(previous._display);
                    if (previous != mask) {
                        previous.dispose();
                    }
                }
                if (mask) {
                    var parent = mask.parent;
                    if (parent) {
                        parent.children.removeValue(mask);
                    }
                    mask._setParent(this);
                    this._display.addChildAt(mask._display, 0);
                    this._childrenDisplay.mask = mask._display;
                }
            }
        }
    });
    Object.defineProperty(Container.prototype, "_processTemplateField", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._processTemplateField.call(this);
            this.children.each(function (child) {
                child._processTemplateField();
            });
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Container.prototype, "walkChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (f) {
            this.children.each(function (child) {
                if (child instanceof Container) {
                    child.walkChildren(f);
                }
                f(child);
            });
        }
    });
    Object.defineProperty(Container.prototype, "eachChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (f) {
            var background = this.get("background");
            if (background) {
                f(background);
            }
            var verticalScrollbar = this.get("verticalScrollbar");
            if (verticalScrollbar) {
                f(verticalScrollbar);
            }
            var mask = this.get("mask");
            if (mask) {
                f(mask);
            }
            this.children.values.forEach(function (child) {
                f(child);
            });
        }
    });
    Object.defineProperty(Container.prototype, "allChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var output = [];
            this.eachChildren(function (x) {
                output.push(x);
            });
            return output;
        }
    });
    Object.defineProperty(Container, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "Container"
    });
    Object.defineProperty(Container, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: Sprite.classNames.concat([Container.className])
    });
    return Container;
}(Sprite));
export { Container };
//# sourceMappingURL=Container.js.map