import { __extends } from "tslib";
import { Sprite } from "./Sprite";
import { BlendMode } from "./backend/Renderer";
import * as $type from "../util/Type";
import * as $array from "../util/Array";
export var visualSettings = ["fill", "fillOpacity", "stroke", "strokeWidth", "strokeOpacity", "fillPattern", "strokePattern", "fillGradient", "strokeGradient", "strokeDasharray", "strokeDashoffset"];
/**
 * Base class used for drawing shapes.
 *
 * @see {@link https://www.amcharts.com/docs/v5/concepts/common-elements/graphics/} for more info
 * @important
 */
var Graphics = /** @class */ (function (_super) {
    __extends(Graphics, _super);
    function Graphics() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "_display", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this._root._renderer.makeGraphics()
        });
        Object.defineProperty(_this, "_clear", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        return _this;
    }
    Object.defineProperty(Graphics.prototype, "_beforeChanged", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            _super.prototype._beforeChanged.call(this);
            if (this.isDirty("draw") || this.isDirty("svgPath")) {
                this.markDirtyBounds();
            }
            if (this.isDirty("fill") || this.isDirty("stroke") || this.isDirty("fillGradient") || this.isDirty("strokeGradient") || this.isDirty("fillPattern") || this.isDirty("strokePattern") || this.isDirty("fillOpacity") || this.isDirty("strokeOpacity") || this.isDirty("strokeWidth") || this.isDirty("draw") || this.isDirty("blendMode") || this.isDirty("strokeDasharray") || this.isDirty("strokeDashoffset") || this.isDirty("svgPath") || this.isDirty("shadowColor") || this.isDirty("shadowBlur") || this.isDirty("shadowOffsetX") || this.isDirty("shadowOffsetY")) {
                this._clear = true;
            }
            if (this.isDirty("fillGradient")) {
                var gradient = this.get("fillGradient");
                if (gradient) {
                    this._display.isMeasured = true;
                    var gradientTarget = gradient.get("target");
                    if (gradientTarget) {
                        this._disposers.push(gradientTarget.events.on("boundschanged", function () {
                            _this._markDirtyKey("fill");
                        }));
                        this._disposers.push(gradientTarget.events.on("positionchanged", function () {
                            _this._markDirtyKey("fill");
                        }));
                    }
                }
            }
            if (this.isDirty("strokeGradient")) {
                var gradient = this.get("strokeGradient");
                if (gradient) {
                    this._display.isMeasured = true;
                    var gradientTarget = gradient.get("target");
                    if (gradientTarget) {
                        this._disposers.push(gradientTarget.events.on("boundschanged", function () {
                            _this._markDirtyKey("stroke");
                        }));
                        this._disposers.push(gradientTarget.events.on("positionchanged", function () {
                            _this._markDirtyKey("stroke");
                        }));
                    }
                }
            }
        }
    });
    Object.defineProperty(Graphics.prototype, "_changed", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._changed.call(this);
            if (this._clear) {
                this.markDirtyLayer();
                this._display.clear();
                var strokeDasharray = this.get("strokeDasharray");
                if ($type.isNumber(strokeDasharray)) {
                    if (strokeDasharray < 0.5) {
                        strokeDasharray = [0];
                    }
                    else {
                        strokeDasharray = [strokeDasharray];
                    }
                }
                this._display.setLineDash(strokeDasharray);
                var strokeDashoffset = this.get("strokeDashoffset");
                if (strokeDashoffset) {
                    this._display.setLineDashOffset(strokeDashoffset);
                }
                var blendMode = this.get("blendMode", BlendMode.NORMAL);
                this._display.blendMode = blendMode;
                var draw = this.get("draw");
                if (draw) {
                    draw(this._display, this);
                }
                var svgPath = this.get("svgPath");
                if (svgPath != null) {
                    this._display.svgPath(svgPath);
                }
            }
        }
    });
    Object.defineProperty(Graphics.prototype, "_afterChanged", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._afterChanged.call(this);
            if (this._clear) {
                var fill_1 = this.get("fill");
                var fillGradient = this.get("fillGradient");
                var fillPattern = this.get("fillPattern");
                var fillOpacity_1 = this.get("fillOpacity");
                var stroke_1 = this.get("stroke");
                var strokeGradient = this.get("strokeGradient");
                var strokePattern = this.get("strokePattern");
                var shadowColor = this.get("shadowColor");
                var shadowBlur = this.get("shadowBlur");
                var shadowOffsetX = this.get("shadowOffsetX");
                var shadowOffsetY = this.get("shadowOffsetY");
                var shadowOpacity = this.get("shadowOpacity");
                //const bounds = this._display.getLocalBounds();
                if (shadowColor && (shadowBlur || shadowOffsetX || shadowOffsetY)) {
                    this._display.shadow(shadowColor, shadowBlur, shadowOffsetX, shadowOffsetY, shadowOpacity);
                }
                if (fillPattern) {
                    var changed = false;
                    if (fill_1 && (!fillPattern.get("fill") || fillPattern.get("fillInherited"))) {
                        fillPattern.set("fill", fill_1);
                        fillPattern.set("fillInherited", true);
                        changed = true;
                    }
                    if (stroke_1 && (!fillPattern.get("color") || fillPattern.get("colorInherited"))) {
                        fillPattern.set("color", stroke_1);
                        fillPattern.set("colorInherited", true);
                        changed = true;
                    }
                    if (changed) {
                        // @todo: is this OK?
                        fillPattern._changed();
                    }
                    var pattern = fillPattern.pattern;
                    if (pattern) {
                        this._display.beginFill(pattern, fillOpacity_1);
                        this._display.endFill();
                    }
                }
                else if (fillGradient) {
                    if (fill_1) {
                        var stops = fillGradient.get("stops", []);
                        if (stops.length) {
                            $array.each(stops, function (stop) {
                                if ((!stop.color || stop.colorInherited) && fill_1) {
                                    stop.color = fill_1;
                                    stop.colorInherited = true;
                                }
                                if (stop.opacity == null || stop.opacityInherited) {
                                    stop.opacity = fillOpacity_1;
                                    stop.opacityInherited = true;
                                }
                            });
                        }
                    }
                    var gradient = fillGradient.getFill(this);
                    if (gradient) {
                        this._display.beginFill(gradient, fillOpacity_1);
                        this._display.endFill();
                    }
                }
                else if (fill_1) {
                    this._display.beginFill(fill_1, fillOpacity_1);
                    this._display.endFill();
                }
                if (stroke_1 || strokeGradient || strokePattern) {
                    var strokeOpacity_1 = this.get("strokeOpacity");
                    var strokeWidth = this.get("strokeWidth", 1);
                    if (strokePattern) {
                        var changed = false;
                        if (stroke_1 && (!strokePattern.get("color") || strokePattern.get("colorInherited"))) {
                            strokePattern.set("color", stroke_1);
                            strokePattern.set("colorInherited", true);
                            changed = true;
                        }
                        if (changed) {
                            // @todo: is this OK?
                            strokePattern._changed();
                        }
                        var pattern = strokePattern.pattern;
                        if (pattern) {
                            this._display.lineStyle(strokeWidth, pattern, strokeOpacity_1);
                            this._display.endStroke();
                        }
                    }
                    else if (strokeGradient) {
                        var stops = strokeGradient.get("stops", []);
                        if (stops.length) {
                            $array.each(stops, function (stop) {
                                if ((!stop.color || stop.colorInherited) && stroke_1) {
                                    stop.color = stroke_1;
                                    stop.colorInherited = true;
                                }
                                if (stop.opacity == null || stop.opacityInherited) {
                                    stop.opacity = strokeOpacity_1;
                                    stop.opacityInherited = true;
                                }
                            });
                        }
                        var gradient = strokeGradient.getFill(this);
                        if (gradient) {
                            this._display.lineStyle(strokeWidth, gradient, strokeOpacity_1);
                            this._display.endStroke();
                        }
                    }
                    else if (stroke_1) {
                        this._display.lineStyle(strokeWidth, stroke_1, strokeOpacity_1);
                        this._display.endStroke();
                    }
                }
            }
            this._clear = false;
        }
    });
    Object.defineProperty(Graphics, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "Graphics"
    });
    Object.defineProperty(Graphics, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: Sprite.classNames.concat([Graphics.className])
    });
    return Graphics;
}(Sprite));
export { Graphics };
//# sourceMappingURL=Graphics.js.map