import { __extends } from "tslib";
import { Container } from "./Container";
import { Label } from "../../core/render/Label";
import { p100 } from "../../core/util/Percent";
import { RoundedRectangle } from "../../core/render/RoundedRectangle";
import { Template } from "../../core/util/Template";
import { ListTemplate } from "../../core/util/List";
import { Color } from "../../core/util/Color";
import { Tooltip } from "../../core/render/Tooltip";
import { LinearGradient } from "../../core/render/gradients/LinearGradient";
import * as $utils from "../../core/util/Utils";
import * as $type from "../../core/util/Type";
/**
 * Heat legend.
 *
 * @see {@link https://www.amcharts.com/docs/v5/concepts/legend/heat-legend/} for more info
 */
var HeatLegend = /** @class */ (function (_super) {
    __extends(HeatLegend, _super);
    function HeatLegend() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        /**
         * A [[Container]] that all labels are placed in.
         *
         * @default Container.new()
         */
        Object.defineProperty(_this, "labelContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.children.push(Container.new(_this._root, {}))
        });
        /**
         * A [[Container]] that all markers are placed in.
         *
         * @default Container.new()
         */
        Object.defineProperty(_this, "markerContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.children.push(Container.new(_this._root, {}))
        });
        /**
         * A start [[Label]].
         *
         * @default Label.new()
         */
        Object.defineProperty(_this, "startLabel", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.labelContainer.children.push(Label.new(_this._root, { themeTags: ["start"] }))
        });
        /**
         * An end [[Label]].
         *
         * @default Label.new()
         */
        Object.defineProperty(_this, "endLabel", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.labelContainer.children.push(Label.new(_this._root, { themeTags: ["end"] }))
        });
        /**
         * List of rectangle elements used for default legend markers.
         *
         * @default new ListTemplate<RoundedRectangle>
         */
        Object.defineProperty(_this, "markers", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new ListTemplate(Template.new({}), function () { return RoundedRectangle._new(_this._root, {
                themeTags: $utils.mergeTags(_this.markers.template.get("themeTags", []), [_this.get("orientation"), "heatlegend", "marker"])
            }, [_this.markers.template]); })
        });
        return _this;
    }
    Object.defineProperty(HeatLegend.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._settings.themeTags = $utils.mergeTags(this._settings.themeTags, ["heatlegend", this._settings.orientation]);
            _super.prototype._afterNew.call(this);
            this.set("tooltip", Tooltip.new(this._root, {
                themeTags: ["heatlegend"]
            }));
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(HeatLegend.prototype, "makeMarker", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var marker = this.markers.make();
            marker.states.create("disabled", {});
            return marker;
        }
    });
    /**
     * Moves and shows tooltip at specific value.
     *
     * Can also specify optional text to show in tooltip, as well as the color.
     *
     * @param  value  Value
     * @param  text   Text
     * @param  color  Color
     */
    Object.defineProperty(HeatLegend.prototype, "showValue", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (value, text, color) {
            var tooltip = this.getTooltip();
            if (tooltip && $type.isNumber(value)) {
                var startValue = this.get("startValue", 0);
                var endValue = this.get("endValue", 1);
                var c = (value - startValue) / (endValue - startValue);
                var startColor = this.get("startColor");
                var endColor = this.get("endColor");
                if (!text) {
                    text = this.getNumberFormatter().format(value);
                }
                if (!color) {
                    color = Color.interpolate(c, startColor, endColor);
                }
                tooltip.label.set("text", text);
                var p = void 0;
                if (this.get("orientation") == "vertical") {
                    p = this.markerContainer.toGlobal({ x: 0, y: this.innerHeight() * (1 - c) });
                }
                else {
                    p = this.markerContainer.toGlobal({ x: this.innerWidth() * c, y: 0 });
                }
                var background = tooltip.get("background");
                if (background) {
                    background.set("fill", Color.interpolate(c, startColor, endColor));
                }
                tooltip.set("pointTo", p);
                tooltip.show();
            }
        }
    });
    Object.defineProperty(HeatLegend.prototype, "_prepareChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._prepareChildren.call(this);
            var labelContainer = this.labelContainer;
            var orientation = this.get("orientation");
            var startLabel = this.startLabel;
            var endLabel = this.endLabel;
            var tooltip = this.getTooltip();
            if (this.isDirty("orientation")) {
                if (orientation == "vertical") {
                    this.markerContainer.setAll({ layout: this._root.verticalLayout, height: p100 });
                    this.set("layout", this._root.horizontalLayout);
                    startLabel.setAll({ y: p100, x: undefined, centerY: p100, centerX: p100 });
                    endLabel.setAll({ y: 0, x: undefined, centerY: 0, centerX: p100 });
                    labelContainer.setAll({ height: p100, width: undefined });
                    if (tooltip) {
                        tooltip.set("pointerOrientation", "horizontal");
                    }
                }
                else {
                    this.markerContainer.setAll({ layout: this._root.horizontalLayout, width: p100 });
                    this.set("layout", this._root.verticalLayout);
                    startLabel.setAll({ x: 0, y: undefined, centerX: 0, centerY: 0 });
                    endLabel.setAll({ x: p100, y: undefined, centerX: p100, centerY: 0 });
                    labelContainer.setAll({ width: p100, height: undefined });
                    if (tooltip) {
                        tooltip.set("pointerOrientation", "vertical");
                    }
                }
            }
            if (this.isDirty("stepCount")) {
                var stepCount = this.get("stepCount", 1);
                var startColor = this.get("startColor");
                var endColor = this.get("endColor");
                this.markerContainer.children.clear();
                if (stepCount > 1) {
                    for (var i = 0; i < stepCount; i++) {
                        var marker = this.makeMarker();
                        if (orientation == "vertical") {
                            this.markerContainer.children.moveValue(marker, 0);
                        }
                        else {
                            this.markerContainer.children.push(marker);
                        }
                        if (startColor && endColor) {
                            marker.set("fill", Color.interpolate(i / stepCount, startColor, endColor));
                        }
                    }
                }
                else if (stepCount == 1) {
                    var marker = this.makeMarker();
                    this.markerContainer.children.push(marker);
                    var gradient = LinearGradient.new(this._root, { stops: [{ color: startColor }, { color: endColor }] });
                    if (orientation == "vertical") {
                        gradient.set("rotation", 90);
                        var stops = gradient.get("stops");
                        if (stops) {
                            stops.reverse();
                        }
                    }
                    else {
                        gradient.set("rotation", 0);
                    }
                    if (startColor && endColor) {
                        marker.set("fillGradient", gradient);
                    }
                }
            }
            if (this.isDirty("startText") || this.isDirty("startValue")) {
                startLabel.set("text", this.get("startText", this.getNumberFormatter().format(this.get("startValue", 0))));
            }
            if (this.isDirty("endText") || this.isDirty("endValue")) {
                endLabel.set("text", this.get("endText", this.getNumberFormatter().format(this.get("endValue", 1))));
            }
        }
    });
    Object.defineProperty(HeatLegend, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "HeatLegend"
    });
    Object.defineProperty(HeatLegend, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: Container.classNames.concat([HeatLegend.className])
    });
    return HeatLegend;
}(Container));
export { HeatLegend };
//# sourceMappingURL=HeatLegend.js.map