import { __extends } from "tslib";
import { Label } from "../render/Label";
import { PointedRectangle } from "../render/PointedRectangle";
import { Container } from "./Container";
import { Percent } from "../util/Percent";
import { Color } from "../util/Color";
import * as $math from "../util/Math";
import * as $array from "../util/Array";
import * as $utils from "../util/Utils";
/**
 * Creates a tooltip.
 *
 * @see {@link https://www.amcharts.com/docs/v5/concepts/common-elements/tooltips/} for more info
 * @important
 */
var Tooltip = /** @class */ (function (_super) {
    __extends(Tooltip, _super);
    function Tooltip() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "_arrangeDisposer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_fx", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "_fy", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "_label", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_fillDp", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_strokeDp", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_labelDp", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_w", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "_h", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        return _this;
    }
    Object.defineProperty(Tooltip.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            this._settings.themeTags = $utils.mergeTags(this._settings.themeTags, ["tooltip"]);
            _super.prototype._afterNew.call(this);
            this.set("background", PointedRectangle.new(this._root, {
                themeTags: ["tooltip", "background"]
            }));
            this._label = this.children.push(Label.new(this._root, {}));
            this._disposers.push(this._label.events.on("boundschanged", function () {
                _this._updateBackground();
            }));
            this.on("bounds", function () {
                _this._updateBackground();
            });
            this._updateTextColor();
            this._root.tooltipContainer.children.push(this);
            this.hide(0);
            this._root._tooltips.push(this);
        }
    });
    Object.defineProperty(Tooltip.prototype, "label", {
        /**
         * A [[Label]] element for the tooltip.
         *
         * @readonly
         * @return Label
         */
        get: function () {
            return this._label;
        },
        enumerable: false,
        configurable: true
    });
    /**
     * Permanently disposes the tooltip.
     */
    Object.defineProperty(Tooltip.prototype, "dispose", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype.dispose.call(this);
            $array.remove(this._root._tooltips, this);
        }
    });
    Object.defineProperty(Tooltip.prototype, "_updateChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._updateChildren.call(this);
            var labelText = this.get("labelText");
            if (labelText != null) {
                this.label.set("text", this.get("labelText"));
            }
        }
    });
    Object.defineProperty(Tooltip.prototype, "_changed", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._changed.call(this);
            if (this.isDirty("pointTo")) {
                // can't compare to previous, as sometimes pointTo is set twice (when pointer moves, so the position won't be udpated)
                this._updateBackground();
            }
            if (this.isDirty("tooltipTarget")) {
                this.updateBackgroundColor();
            }
        }
    });
    Object.defineProperty(Tooltip.prototype, "_onShow", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._onShow.call(this);
            this.updateBackgroundColor();
        }
    });
    Object.defineProperty(Tooltip.prototype, "updateBackgroundColor", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            var tooltipTarget = this.get("tooltipTarget");
            var background = this.get("background");
            var fill;
            var stroke;
            if (tooltipTarget && background) {
                fill = tooltipTarget.get("fill");
                stroke = tooltipTarget.get("stroke");
                if (fill == null) {
                    fill = stroke;
                }
                if (this.get("getFillFromSprite")) {
                    if (this._fillDp) {
                        this._fillDp.dispose();
                    }
                    if (fill != null) {
                        background.set("fill", fill);
                    }
                    this._fillDp = tooltipTarget.on("fill", function (fill) {
                        if (fill != null) {
                            background.set("fill", fill);
                            _this._updateTextColor(fill);
                        }
                    });
                }
                if (this.get("getStrokeFromSprite")) {
                    if (this._strokeDp) {
                        this._strokeDp.dispose();
                    }
                    if (fill != null) {
                        background.set("stroke", fill);
                    }
                    this._strokeDp = tooltipTarget.on("fill", function (fill) {
                        if (fill != null) {
                            background.set("stroke", fill);
                        }
                    });
                }
                if (this.get("getLabelFillFromSprite")) {
                    if (this._labelDp) {
                        this._labelDp.dispose();
                    }
                    if (fill != null) {
                        this.label.set("fill", fill);
                    }
                    this._labelDp = tooltipTarget.on("fill", function (fill) {
                        if (fill != null) {
                            _this.label.set("fill", fill);
                        }
                    });
                }
            }
            this._updateTextColor(fill);
        }
    });
    Object.defineProperty(Tooltip.prototype, "_updateTextColor", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (fill) {
            if (this.get("autoTextColor")) {
                if (fill == null) {
                    fill = this.get("background").get("fill");
                }
                if (fill == null) {
                    fill = this._root.interfaceColors.get("background");
                }
                if (fill instanceof Color) {
                    this.label.set("fill", Color.alternative(fill, this._root.interfaceColors.get("alternativeText"), this._root.interfaceColors.get("text")));
                }
            }
        }
    });
    Object.defineProperty(Tooltip.prototype, "_setDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            _super.prototype._setDataItem.call(this, dataItem);
            this.label._setDataItem(dataItem);
        }
    });
    Object.defineProperty(Tooltip.prototype, "_updateBackground", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype.updateBackground.call(this);
            var parent = this._root.container;
            if (parent) {
                var cw = 0.5;
                var ch = 0.5;
                var centerX = this.get("centerX");
                if (centerX instanceof Percent) {
                    cw = centerX.value;
                }
                var centerY = this.get("centerY");
                if (centerY instanceof Percent) {
                    ch = centerY.value;
                }
                var parentW = parent.width();
                var parentH = parent.height();
                var bounds = this.get("bounds", { left: 0, top: 0, right: parentW, bottom: parentH });
                this._updateBounds();
                var w = this.width();
                var h = this.height();
                // use old w and h,as when tooltip is hidden, these are 0 and unneeded animation happens
                if (w === 0) {
                    w = this._w;
                }
                if (h === 0) {
                    h = this._h;
                }
                var pointTo = this.get("pointTo", { x: parentW / 2, y: parentH / 2 });
                var x = pointTo.x;
                var y = pointTo.y;
                var pointerOrientation = this.get("pointerOrientation");
                var background = this.get("background");
                var pointerLength = 0;
                if (background instanceof PointedRectangle) {
                    pointerLength = background.get("pointerLength", 0);
                }
                var pointerX = 0;
                var pointerY = 0;
                var boundsW = bounds.right - bounds.left;
                var boundsH = bounds.bottom - bounds.top;
                // horizontal
                if (pointerOrientation == "horizontal" || pointerOrientation == "left" || pointerOrientation == "right") {
                    if (pointerOrientation == "horizontal") {
                        if (x > bounds.left + boundsW / 2) {
                            x -= (w * (1 - cw) + pointerLength);
                        }
                        else {
                            x += (w * cw + pointerLength);
                        }
                    }
                    else if (pointerOrientation == "left") {
                        x += (w * (1 - cw) + pointerLength);
                    }
                    else {
                        x -= (w * cw + pointerLength);
                    }
                }
                // vertical pointer
                else {
                    if (pointerOrientation == "vertical") {
                        if (y > bounds.top + h / 2 + pointerLength) {
                            y -= (h * (1 - ch) + pointerLength);
                        }
                        else {
                            y += (h * ch + pointerLength);
                        }
                    }
                    else if (pointerOrientation == "down") {
                        y -= (h * (1 - ch) + pointerLength);
                    }
                    else {
                        y += (h * ch + pointerLength);
                    }
                }
                x = $math.fitToRange(x, bounds.left + w * cw, bounds.left + boundsW - w * (1 - cw));
                y = $math.fitToRange(y, bounds.top + h * ch, bounds.top + boundsH - h * (1 - ch));
                pointerX = pointTo.x - x + w * cw;
                pointerY = pointTo.y - y + h * ch;
                this._fx = x;
                this._fy = y;
                var animationDuration = this.get("animationDuration", 0);
                if (animationDuration > 0 && this.get("visible") && this.get("opacity") > 0.1) {
                    var animationEasing = this.get("animationEasing");
                    this.animate({ key: "x", to: x, duration: animationDuration, easing: animationEasing });
                    this.animate({ key: "y", to: y, duration: animationDuration, easing: animationEasing });
                }
                else {
                    this.set("x", x);
                    this.set("y", y);
                }
                if (background instanceof PointedRectangle) {
                    background.set("pointerX", pointerX);
                    background.set("pointerY", pointerY);
                }
                if (w > 0) {
                    this._w = w;
                }
                if (h > 0) {
                    this._h = h;
                }
            }
        }
    });
    Object.defineProperty(Tooltip, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "Tooltip"
    });
    Object.defineProperty(Tooltip, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: Container.classNames.concat([Tooltip.className])
    });
    return Tooltip;
}(Container));
export { Tooltip };
//# sourceMappingURL=Tooltip.js.map