import { Time } from "./Animation";
import * as $utils from "./Utils";
/**
 * @ignore
 */
export declare function rgba2hex(color: string): number;
/**
 * Returns a new [[Color]] object base on input.
 *
 * Accepts parameters in CSS hex or rgb/rtba strings, or hex numbers.
 *
 * * `"#f00"`
 * * `"#ff0000"`
 * * `"rgb(255, 0, 0)"`
 * * `"rgba(255, 0, 0, 1)"`
 * * `0xff0000`
 *
 * @param   input  Input color
 * @return         Color
 */
export declare function color(input: number | string): Color;
/**
 * Wherever color needs to be specified in amCharts 5, `Color` object needs to
 * be used.
 *
 * @see {@link https://www.amcharts.com/docs/v5/concepts/colors-gradients-and-patterns/} for more info
 * @important
 */
export declare class Color {
    protected _hex: number;
    private constructor();
    /**
     * Color numeric value.
     */
    get hex(): number;
    /**
     * Value of color's R channel.
     * @return R value
     */
    get r(): number;
    /**
     * Value of color's G channel.
     * @return G value
     */
    get g(): number;
    /**
     * Value of color's B channel.
     * @return B value
     */
    get b(): number;
    /**
     * Returns color CSS representation in form of `rgba(r, g, b, a)` string.
     *
     * @param   alpha  Opacity
     * @return         CSS string
     */
    toCSS(alpha?: number): string;
    /**
     * Returns color CSS representation in form of `#rgb` string.
     *
     * @return         CSS string
     */
    toCSSHex(): string;
    /**
     * Returns color's HSL info.
     * @param   alpha Opacity
     * @return        HSL info
     */
    toHSL(alpha?: number): $utils.iHSL;
    /**
     * Converts HSL values into a new [[Color]] object.
     *
     * @param   h H value
     * @param   s S value
     * @param   l L value
     * @return    Color object
     */
    static fromHSL(h: number, s: number, l: number): Color;
    toString(): string;
    /**
     * Converts hex number into a new [[Color]] object.
     *
     * ```TypeScript
     * Color.fromHex(0xff0000) // red
     * ```
     * ```JavaScript
     * Color.fromHex(0xff0000) // red
     * ```
     *
     * @param   hex  Hex color
     * @return       Color
     */
    static fromHex(hex: number): Color;
    /**
     * Converts RGB values to a new [[Color]] object.
     *
     * @param   r  R value
     * @param   g  G value
     * @param   b  B value
     * @return     Color
     */
    static fromRGB(r: number, g: number, b: number): Color;
    /**
     * Converts RGB string to a new [[Color]] object.
     *
     * ```TypeScript
     * Color.fromString("#ff0000") // red
     * ```
     * ```JavaScript
     * Color.fromString("#ff0000") // red
     * ```
     *
     * @param   s  RGB string
     * @return     Color
     */
    static fromString(s: string): Color;
    /**
     * Converts CSS rgba() syntax to a new [[Color]] object.
     *
     * ```TypeScript
     * Color.fromCSS("rgba(255, 0, 0, 1)") // red
     * ```
     * ```JavaScript
     * Color.fromCSS("rgba(255, 0, 0, 1)") // red
     * ```
     *
     * @param  {string} s [description]
     * @return {Color}    [description]
     */
    static fromCSS(s: string): Color;
    /**
     * Convert to color from virtually anything.
     *
     * Will throw an exception if unable to resolve the color.
     *
     * @param   s  Source
     * @return     Color
     */
    static fromAny(s: string | number | Color): Color;
    /**
     * Returns a new [[Color]] object based on either `lightAlternative` or
     * `darkAlternative` depending on which one is more contrasting with
     * the `color`.
     *
     * @param   color             Reference color
     * @param   lightAlternative  Light color
     * @param   darkAlternative   Dark color
     * @return                    Alternative color
     */
    static alternative(color: Color, lightAlternative?: Color, darkAlternative?: Color): Color;
    /**
     * Returns an intermediate Color between two reference colors depending on
     * the progress (`diff`) between the two.
     *
     * @param   diff  Progress
     * @param   from  Source color
     * @param   to    Target color
     * @param   mode  Interpolation mode
     * @return        Color
     */
    static interpolate(diff: Time, from: Color, to: Color, mode?: "rgb" | "hsl"): Color;
    /**
     * Returns a new [[Color]] lightened by `percent` value.
     *
     * Use negative value to darken the color.
     *
     * @param   color    Source color
     * @param   percent  Percent
     * @return           New color
     */
    static lighten(color: Color, percent: number): Color;
    /**
     * Returns a new [[Color]] brightened by `percent` value.
     *
     * Use negative value to dim the color.
     *
     * @param   color    Source color
     * @param   percent  Percent
     * @return           New color
     */
    static brighten(color: Color, percent: number): Color;
    /**
     * Returns a new [[Color]] saturated by `percent` value.
     *
     * Use negative value to desaturate the color.
     *
     * @param   color    Source color
     * @param   percent  Percent
     * @return           New color
     */
    static saturate(color: Color, percent: number): Color;
}
//# sourceMappingURL=Color.d.ts.map