import { __extends, __read, __spreadArray } from "tslib";
import { Disposer } from "./Disposer";
import { EventDispatcher } from "./EventDispatcher";
import { getInterpolate } from "./Animation";
import { States } from "./States";
import { registry } from "../Registry";
import * as $object from "./Object";
import * as $ease from "./Ease";
import * as $array from "./Array";
import * as $order from "./Order";
/**
 * Allows to dynamically modify setting value of its target element.
 *
 * @see {@link https://www.amcharts.com/docs/v5/concepts/settings/adapters/} for more info
 */
var Adapters = /** @class */ (function () {
    function Adapters(entity) {
        Object.defineProperty(this, "_entity", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_callbacks", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        Object.defineProperty(this, "_disabled", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        this._entity = entity;
    }
    /**
     * Add a function (`callback`) that will modify value for setting `key`.
     *
     * @see {@link https://www.amcharts.com/docs/v5/concepts/settings/adapters/} for more info
     */
    Object.defineProperty(Adapters.prototype, "add", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, callback) {
            var _this = this;
            var callbacks = this._callbacks[key];
            if (callbacks === undefined) {
                callbacks = this._callbacks[key] = [];
            }
            callbacks.push(callback);
            this._entity._markDirtyKey(key);
            return new Disposer(function () {
                if ($array.removeFirst(callbacks, callback)) {
                    _this._entity._markDirtyKey(key);
                }
            });
        }
    });
    /**
     * Removes all adapters for the specific key.
     *
     * @since 5.1.0
     */
    Object.defineProperty(Adapters.prototype, "remove", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key) {
            var callbacks = this._callbacks[key];
            if (callbacks !== undefined) {
                delete this._callbacks[key];
                if (callbacks.length !== 0) {
                    this._entity._markDirtyKey(key);
                }
            }
        }
    });
    /**
     * Enables (previously disabled) adapters for specific key.
     *
     * @since 5.1.0
     */
    Object.defineProperty(Adapters.prototype, "enable", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key) {
            if (this._disabled[key]) {
                delete this._disabled[key];
                this._entity._markDirtyKey(key);
            }
        }
    });
    /**
     * Disables all adapters for specific key.
     *
     * @since 5.1.0
     */
    Object.defineProperty(Adapters.prototype, "disable", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key) {
            if (!this._disabled[key]) {
                this._disabled[key] = true;
                this._entity._markDirtyKey(key);
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Adapters.prototype, "fold", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, value) {
            if (!this._disabled[key]) {
                var callbacks = this._callbacks[key];
                if (callbacks !== undefined) {
                    for (var i = 0, len = callbacks.length; i < len; ++i) {
                        value = callbacks[i](value, this._entity, key);
                    }
                }
            }
            return value;
        }
    });
    return Adapters;
}());
export { Adapters };
/**
 * Animation object.
 *
 * @see {@link https://www.amcharts.com/docs/v5/concepts/animations/} for more info
 */
var Animation = /** @class */ (function () {
    function Animation(from, to, duration, easing, loops, startingTime) {
        Object.defineProperty(this, "_from", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_to", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_duration", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_easing", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_loops", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_interpolate", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_oldTime", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_time", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(this, "_stopped", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(this, "_playing", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: true
        });
        Object.defineProperty(this, "events", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new EventDispatcher()
        });
        this._from = from;
        this._to = to;
        this._duration = duration;
        this._easing = easing;
        this._loops = loops;
        this._interpolate = getInterpolate(from, to);
        this._oldTime = startingTime;
    }
    Object.defineProperty(Animation.prototype, "to", {
        get: function () {
            return this._to;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(Animation.prototype, "from", {
        get: function () {
            return this._from;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(Animation.prototype, "playing", {
        get: function () {
            return this._playing;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(Animation.prototype, "stopped", {
        get: function () {
            return this._stopped;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(Animation.prototype, "stop", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (!this._stopped) {
                this._stopped = true;
                this._playing = false;
                if (this.events.isEnabled("stopped")) {
                    this.events.dispatch("stopped", {
                        type: "stopped",
                        target: this,
                    });
                }
            }
        }
    });
    Object.defineProperty(Animation.prototype, "pause", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._playing = false;
            this._oldTime = null;
        }
    });
    Object.defineProperty(Animation.prototype, "play", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (!this._stopped) {
                this._playing = true;
            }
        }
    });
    Object.defineProperty(Animation.prototype, "percentage", {
        get: function () {
            return this._time / this._duration;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(Animation.prototype, "waitForStop", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            return new Promise(function (resolve, _reject) {
                if (_this._stopped) {
                    resolve();
                }
                else {
                    var listener = function () {
                        stopped_1.dispose();
                        resolve();
                    };
                    var stopped_1 = _this.events.on("stopped", listener);
                }
            });
        }
    });
    Object.defineProperty(Animation.prototype, "_checkEnded", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (this._loops > 1) {
                --this._loops;
                return false;
            }
            else {
                return true;
            }
        }
    });
    Object.defineProperty(Animation.prototype, "_run", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (currentTime) {
            if (this._oldTime !== null) {
                this._time += currentTime - this._oldTime;
                if (this._time > this._duration) {
                    this._time = this._duration;
                }
            }
            this._oldTime = currentTime;
        }
    });
    Object.defineProperty(Animation.prototype, "_reset", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (currentTime) {
            this._oldTime = currentTime;
            this._time = 0;
        }
    });
    Object.defineProperty(Animation.prototype, "_value", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (diff) {
            return this._interpolate(this._easing(diff), this._from, this._to);
        }
    });
    return Animation;
}());
export { Animation };
/**
 * @ignore
 */
var counter = 0;
/**
 * Base class for [[Entity]] objects that support Settings.
 *
 * @see {@link https://www.amcharts.com/docs/v5/concepts/settings/} for more info
 */
var Settings = /** @class */ (function () {
    function Settings(settings) {
        /**
         * Unique ID.
         */
        Object.defineProperty(this, "uid", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: ++counter
        });
        Object.defineProperty(this, "_settings", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_privateSettings", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        Object.defineProperty(this, "_settingEvents", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        Object.defineProperty(this, "_privateSettingEvents", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        Object.defineProperty(this, "_prevSettings", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        Object.defineProperty(this, "_prevPrivateSettings", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        Object.defineProperty(this, "_animatingSettings", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        Object.defineProperty(this, "_animatingPrivateSettings", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        Object.defineProperty(this, "_animatingCount", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(this, "_disposed", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        // TODO move this into Entity
        Object.defineProperty(this, "_userProperties", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        this._settings = settings;
    }
    Object.defineProperty(Settings.prototype, "_checkDirty", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            $object.keys(this._settings).forEach(function (key) {
                _this._userProperties[key] = true;
                _this._markDirtyKey(key);
            });
        }
    });
    Object.defineProperty(Settings.prototype, "_runAnimation", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (currentTime) {
            var _this = this;
            if (!this.isDisposed()) {
                $object.each(this._animatingSettings, function (key, animation) {
                    if (animation._stopped) {
                        _this._stopAnimation(key);
                    }
                    else if (animation._playing) {
                        animation._run(currentTime);
                        var diff = animation.percentage;
                        if (diff >= 1) {
                            if (animation._checkEnded()) {
                                _this.set(key, animation._value(1));
                            }
                            else {
                                animation._reset(currentTime);
                                _this._set(key, animation._value(1));
                            }
                        }
                        else {
                            _this._set(key, animation._value(diff));
                        }
                    }
                });
                $object.each(this._animatingPrivateSettings, function (key, animation) {
                    if (animation._stopped) {
                        _this._stopAnimationPrivate(key);
                    }
                    else if (animation._playing) {
                        animation._run(currentTime);
                        var diff = animation.percentage;
                        if (diff >= 1) {
                            if (animation._checkEnded()) {
                                _this.setPrivate(key, animation._value(1));
                            }
                            else {
                                animation._reset(currentTime);
                                _this._setPrivate(key, animation._value(1));
                            }
                        }
                        else {
                            _this._setPrivate(key, animation._value(diff));
                        }
                    }
                });
                if (this._animatingCount < 0) {
                    throw new Error("Invalid animation count");
                }
                return this._animatingCount === 0;
            }
            else {
                return true;
            }
        }
    });
    Object.defineProperty(Settings.prototype, "_markDirtyKey", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_key) {
            this.markDirty();
        }
    });
    Object.defineProperty(Settings.prototype, "_markDirtyPrivateKey", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_key) {
            this.markDirty();
        }
    });
    /**
     * Sets a callback function to invoke when specific key of settings changes
     * or is set.
     *
     * @see {@link https://www.amcharts.com/docs/v5/concepts/events/#Settings_value_change} for more info
     * @param   key       Settings key
     * @param   callback  Callback
     * @return            Disposer for event
     */
    Object.defineProperty(Settings.prototype, "on", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, callback) {
            var _this = this;
            var events = this._settingEvents[key];
            if (events === undefined) {
                events = this._settingEvents[key] = [];
            }
            events.push(callback);
            return new Disposer(function () {
                $array.removeFirst(events, callback);
                if (events.length === 0) {
                    delete _this._settingEvents[key];
                }
            });
        }
    });
    /**
     * Sets a callback function to invoke when specific key of private settings
     * changes or is set.
     *
     * @see {@link https://www.amcharts.com/docs/v5/concepts/events/#Settings_value_change} for more info
     * @ignore
     * @param   key       Private settings key
     * @param   callback  Callback
     * @return            Disposer for event
     */
    Object.defineProperty(Settings.prototype, "onPrivate", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, callback) {
            var _this = this;
            var events = this._privateSettingEvents[key];
            if (events === undefined) {
                events = this._privateSettingEvents[key] = [];
            }
            events.push(callback);
            return new Disposer(function () {
                $array.removeFirst(events, callback);
                if (events.length === 0) {
                    delete _this._privateSettingEvents[key];
                }
            });
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Settings.prototype, "getRaw", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, fallback) {
            var value = this._settings[key];
            if (value !== undefined) {
                return value;
            }
            else {
                return fallback;
            }
        }
    });
    Object.defineProperty(Settings.prototype, "get", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, fallback) {
            return this.getRaw(key, fallback);
        }
    });
    Object.defineProperty(Settings.prototype, "_sendKeyEvent", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, value) {
            var _this = this;
            var events = this._settingEvents[key];
            if (events !== undefined) {
                $array.each(events, function (callback) {
                    callback(value, _this, key);
                });
            }
        }
    });
    Object.defineProperty(Settings.prototype, "_sendPrivateKeyEvent", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, value) {
            var _this = this;
            var events = this._privateSettingEvents[key];
            if (events !== undefined) {
                $array.each(events, function (callback) {
                    callback(value, _this, key);
                });
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Settings.prototype, "_setRaw", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, old, value) {
            this._prevSettings[key] = old;
            this._sendKeyEvent(key, value);
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Settings.prototype, "setRaw", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, value) {
            var old = this._settings[key];
            this._settings[key] = value;
            if (old !== value) {
                this._setRaw(key, old, value);
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Settings.prototype, "_set", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, value) {
            var old = this._settings[key];
            this._settings[key] = value;
            if (old !== value) {
                this._setRaw(key, old, value);
                this._markDirtyKey(key);
            }
        }
    });
    Object.defineProperty(Settings.prototype, "_stopAnimation", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key) {
            var animation = this._animatingSettings[key];
            if (animation) {
                delete this._animatingSettings[key];
                --this._animatingCount;
                animation.stop();
            }
        }
    });
    /**
     * Sets a setting `value` for the specified `key`, and returns the same `value`.
     *
     * @see {@link https://www.amcharts.com/docs/v5/concepts/settings/} for more info
     * @param   key       Setting key
     * @param   value     Setting value
     * @return            Setting value
     */
    Object.defineProperty(Settings.prototype, "set", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, value) {
            this._set(key, value);
            this._stopAnimation(key);
            return value;
        }
    });
    /**
     * Removes a setting value for the specified `key`;
     *
     * @see {@link https://www.amcharts.com/docs/v5/concepts/settings/} for more info
     * @param   key       Setting key
     */
    Object.defineProperty(Settings.prototype, "remove", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key) {
            if (key in this._settings) {
                this._prevSettings[key] = this._settings[key];
                delete this._settings[key];
                this._sendKeyEvent(key, undefined);
                this._markDirtyKey(key);
            }
            this._stopAnimation(key);
        }
    });
    /**
     * Removes all keys;
     *
     * @see {@link https://www.amcharts.com/docs/v5/concepts/settings/} for more info
     */
    Object.defineProperty(Settings.prototype, "removeAll", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            $array.each($object.keys(this._settings), function (key) {
                _this.remove(key);
            });
        }
    });
    /**
     * Returns a value of a private setting.
     *
     * @see {@link https://www.amcharts.com/docs/v5/concepts/settings/#Private_settings} for more info
     */
    Object.defineProperty(Settings.prototype, "getPrivate", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, fallback) {
            var value = this._privateSettings[key];
            if (value !== undefined) {
                return value;
            }
            else {
                return fallback;
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Settings.prototype, "_setPrivateRaw", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, old, value) {
            this._prevPrivateSettings[key] = old;
            this._sendPrivateKeyEvent(key, value);
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Settings.prototype, "setPrivateRaw", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, value) {
            var old = this._privateSettings[key];
            this._privateSettings[key] = value;
            if (old !== value) {
                this._setPrivateRaw(key, old, value);
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Settings.prototype, "_setPrivate", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, value) {
            var old = this._privateSettings[key];
            this._privateSettings[key] = value;
            if (old !== value) {
                this._setPrivateRaw(key, old, value);
                this._markDirtyPrivateKey(key);
            }
        }
    });
    Object.defineProperty(Settings.prototype, "_stopAnimationPrivate", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key) {
            var animation = this._animatingPrivateSettings[key];
            if (animation) {
                animation.stop();
                delete this._animatingPrivateSettings[key];
                --this._animatingCount;
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Settings.prototype, "setPrivate", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, value) {
            this._setPrivate(key, value);
            this._stopAnimationPrivate(key);
            return value;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Settings.prototype, "removePrivate", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key) {
            if (key in this._privateSettings) {
                this._prevPrivateSettings[key] = this._privateSettings[key];
                delete this._privateSettings[key];
                this._markDirtyPrivateKey(key);
            }
            this._stopAnimationPrivate(key);
        }
    });
    /**
     * Sets multiple settings at once.
     *
     * `settings` must be an object with key: value pairs.
     *
     * @see {@link https://www.amcharts.com/docs/v5/concepts/settings/} for more info
     * @param settings Settings
     */
    Object.defineProperty(Settings.prototype, "setAll", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (settings) {
            var _this = this;
            $object.each(settings, function (key, value) {
                _this.set(key, value);
            });
        }
    });
    /**
     * Animates setting values from current/start values to new ones.
     *
     * @see {@link https://www.amcharts.com/docs/v5/concepts/animations/#Animating_settings} for more info
     * @param   options  Animation options
     * @return           Animation object
     */
    Object.defineProperty(Settings.prototype, "animate", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (options) {
            var key = options.key;
            var to = options.to;
            var duration = options.duration || 0;
            var loops = options.loops || 1;
            var from = (options.from === undefined ? this.get(key) : options.from);
            var easing = (options.easing === undefined ? $ease.linear : options.easing);
            if (duration === 0) {
                this.set(key, to);
            }
            else {
                if (from === undefined || from === to) {
                    this.set(key, to);
                }
                else {
                    ++this._animatingCount;
                    this.set(key, from);
                    var animation_1 = this._animatingSettings[key] = new Animation(from, to, duration, easing, loops, this._animationTime());
                    this._startAnimation();
                    return animation_1;
                }
            }
            var animation = new Animation(from, to, duration, easing, loops, null);
            animation.stop();
            return animation;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Settings.prototype, "animatePrivate", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (options) {
            var key = options.key;
            var to = options.to;
            var duration = options.duration || 0;
            var loops = options.loops || 1;
            var from = (options.from === undefined ? this.getPrivate(key) : options.from);
            var easing = (options.easing === undefined ? $ease.linear : options.easing);
            if (duration === 0) {
                this.setPrivate(key, to);
            }
            else {
                if (from === undefined || from === to) {
                    this.setPrivate(key, to);
                }
                else {
                    ++this._animatingCount;
                    this.setPrivate(key, from);
                    var animation_2 = this._animatingPrivateSettings[key] = new Animation(from, to, duration, easing, loops, this._animationTime());
                    this._startAnimation();
                    return animation_2;
                }
            }
            var animation = new Animation(from, to, duration, easing, loops, null);
            animation.stop();
            return animation;
        }
    });
    Object.defineProperty(Settings.prototype, "_dispose", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () { }
    });
    /**
     * Returns `true` if this element is disposed.
     *
     * @return Disposed
     */
    Object.defineProperty(Settings.prototype, "isDisposed", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this._disposed;
        }
    });
    /**
     * Disposes this object.
     */
    Object.defineProperty(Settings.prototype, "dispose", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (!this._disposed) {
                this._disposed = true;
                this._dispose();
            }
        }
    });
    return Settings;
}());
export { Settings };
/**
 * Base class.
 *
 * @important
 */
var Entity = /** @class */ (function (_super) {
    __extends(Entity, _super);
    /**
     * IMPORTANT! Do not instantiate this class via `new Class()` syntax.
     *
     * Use static method `Class.new()` instead.
     *
     * @see {@link https://www.amcharts.com/docs/v5/getting-started/#New_element_syntax} for more info
     * @ignore
     */
    function Entity(root, settings, isReal, templates) {
        if (templates === void 0) { templates = []; }
        var _this = _super.call(this, settings) || this;
        Object.defineProperty(_this, "_root", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_user_id", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        }); // for testing purposes
        Object.defineProperty(_this, "states", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new States(_this)
        });
        Object.defineProperty(_this, "adapters", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Adapters(_this)
        });
        Object.defineProperty(_this, "events", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this._createEvents()
        });
        Object.defineProperty(_this, "_userPrivateProperties", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        Object.defineProperty(_this, "_dirty", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        Object.defineProperty(_this, "_dirtyPrivate", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        Object.defineProperty(_this, "_template", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        // Templates for the themes
        Object.defineProperty(_this, "_templates", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        // Internal templates which can be overridden by the user's templates
        Object.defineProperty(_this, "_internalTemplates", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        // Default themes which can be overridden by the user's themes
        Object.defineProperty(_this, "_defaultThemes", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        // Disposers for all of the templates
        Object.defineProperty(_this, "_templateDisposers", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        Object.defineProperty(_this, "_disposers", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        // Whether the template setup function should be run
        Object.defineProperty(_this, "_runSetup", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: true
        });
        Object.defineProperty(_this, "_disposerProperties", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        if (!isReal) {
            throw new Error("You cannot use `new Class()`, instead use `Class.new()`");
        }
        _this._root = root;
        _this._internalTemplates = templates;
        return _this;
    }
    /**
     * Use this method to create an instance of this class.
     *
     * @see {@link https://www.amcharts.com/docs/v5/getting-started/#New_element_syntax} for more info
     * @param   root      Root element
     * @param   settings  Settings
     * @param   template  Template
     * @return            Instantiated object
     */
    Object.defineProperty(Entity, "new", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (root, settings, template) {
            var x = (new this(root, settings, true));
            x._template = template;
            x._afterNew();
            return x;
        }
    });
    Object.defineProperty(Entity, "_new", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (root, settings, templates) {
            if (templates === void 0) { templates = []; }
            var x = (new this(root, settings, true, templates));
            x._afterNew();
            return x;
        }
    });
    Object.defineProperty(Entity.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            this._checkDirty();
            var shouldApply = false;
            var template = this._template;
            if (template) {
                shouldApply = true;
                template._setObjectTemplate(this);
            }
            $array.each(this._internalTemplates, function (template) {
                shouldApply = true;
                template._setObjectTemplate(_this);
            });
            if (shouldApply) {
                this._applyTemplates(false);
            }
            this.states.create("default", {});
            this._setDefaults();
        }
    });
    // This is the same as _afterNew, except it also applies the themes.
    // This should only be used for classes which don't have a parent (because they extend from Entity and not Sprite).
    Object.defineProperty(Entity.prototype, "_afterNewApplyThemes", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            this._checkDirty();
            var template = this._template;
            if (template) {
                template._setObjectTemplate(this);
            }
            $array.each(this._internalTemplates, function (template) {
                template._setObjectTemplate(_this);
            });
            this.states.create("default", {});
            this._setDefaults();
            this._applyThemes();
        }
    });
    Object.defineProperty(Entity.prototype, "_createEvents", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return new EventDispatcher();
        }
    });
    Object.defineProperty(Entity.prototype, "classNames", {
        /**
         * @ignore
         */
        get: function () {
            return this.constructor.classNames;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(Entity.prototype, "className", {
        /**
         * @ignore
         */
        get: function () {
            return this.constructor.className;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(Entity.prototype, "_setDefaults", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
        }
    });
    Object.defineProperty(Entity.prototype, "_setDefault", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, value) {
            if (!(key in this._settings)) {
                _super.prototype.set.call(this, key, value);
            }
        }
    });
    Object.defineProperty(Entity.prototype, "_setRawDefault", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, value) {
            if (!(key in this._settings)) {
                _super.prototype.setRaw.call(this, key, value);
            }
        }
    });
    Object.defineProperty(Entity.prototype, "_clearDirty", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            $object.keys(this._dirty).forEach(function (key) {
                _this._dirty[key] = false;
            });
            $object.keys(this._dirtyPrivate).forEach(function (key) {
                _this._dirtyPrivate[key] = false;
            });
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Entity.prototype, "isDirty", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key) {
            return !!this._dirty[key];
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Entity.prototype, "isPrivateDirty", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key) {
            return !!this._dirtyPrivate[key];
        }
    });
    Object.defineProperty(Entity.prototype, "_markDirtyKey", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key) {
            this._dirty[key] = true;
            _super.prototype._markDirtyKey.call(this, key);
        }
    });
    Object.defineProperty(Entity.prototype, "_markDirtyPrivateKey", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key) {
            this._dirtyPrivate[key] = true;
            _super.prototype._markDirtyKey.call(this, key);
        }
    });
    /**
     * Checks if element is of certain class (or inherits one).
     *
     * @param   type  Class name to check
     * @return {boolean} Is of class?
     */
    Object.defineProperty(Entity.prototype, "isType", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (type) {
            return this.classNames.indexOf(type) !== -1;
        }
    });
    Object.defineProperty(Entity.prototype, "_pushPropertyDisposer", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, disposer) {
            var disposers = this._disposerProperties[key];
            if (disposers === undefined) {
                disposers = this._disposerProperties[key] = [];
            }
            disposers.push(disposer);
            return disposer;
        }
    });
    Object.defineProperty(Entity.prototype, "_disposeProperty", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key) {
            var disposers = this._disposerProperties[key];
            if (disposers !== undefined) {
                $array.each(disposers, function (disposer) {
                    disposer.dispose();
                });
                delete this._disposerProperties[key];
            }
        }
    });
    Object.defineProperty(Entity.prototype, "template", {
        get: function () {
            return this._template;
        },
        /**
         * @todo needs description
         * @param  value  Template
         */
        set: function (value) {
            var template = this._template;
            if (template !== value) {
                this._template = value;
                if (template) {
                    template._removeObjectTemplate(this);
                }
                if (value) {
                    value._setObjectTemplate(this);
                }
                this._applyTemplates();
            }
        },
        enumerable: false,
        configurable: true
    });
    /**
     * @ignore
     */
    Object.defineProperty(Entity.prototype, "markDirty", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._root._addDirtyEntity(this);
        }
    });
    Object.defineProperty(Entity.prototype, "_startAnimation", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._root._addAnimation(this);
        }
    });
    Object.defineProperty(Entity.prototype, "_animationTime", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this._root.animationTime;
        }
    });
    Object.defineProperty(Entity.prototype, "_applyState", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_name) { }
    });
    Object.defineProperty(Entity.prototype, "_applyStateAnimated", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_name, _duration) { }
    });
    Object.defineProperty(Entity.prototype, "get", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, fallback) {
            var value = this.adapters.fold(key, this._settings[key]);
            if (value !== undefined) {
                return value;
            }
            else {
                return fallback;
            }
        }
    });
    /**
     * Sets a setting `value` for the specified `key`, and returns the same `value`.
     *
     * @see {@link https://www.amcharts.com/docs/v5/concepts/settings/} for more info
     * @param   key       Setting key
     * @param   value     Setting value
     * @return            Setting value
     */
    Object.defineProperty(Entity.prototype, "set", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, value) {
            this._userProperties[key] = true;
            return _super.prototype.set.call(this, key, value);
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Entity.prototype, "setRaw", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, value) {
            this._userProperties[key] = true;
            _super.prototype.setRaw.call(this, key, value);
        }
    });
    /**
     * Sets a setting `value` for the specified `key` only if the value for this key was not set previously using set method, and returns the same `value`.
     *
     * @see {@link https://www.amcharts.com/docs/v5/concepts/settings/} for more info
     * @param   key       Setting key
     * @param   value     Setting value
     * @return            Setting value
     */
    Object.defineProperty(Entity.prototype, "_setSoft", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, value) {
            if (!this._userProperties[key]) {
                return _super.prototype.set.call(this, key, value);
            }
            return value;
        }
    });
    /**
     * Removes a setting value for the specified `key`.
     *
     * @see {@link https://www.amcharts.com/docs/v5/concepts/settings/} for more info
     * @param   key       Setting key
     */
    Object.defineProperty(Entity.prototype, "remove", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key) {
            delete this._userProperties[key];
            this._removeTemplateProperty(key);
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Entity.prototype, "setPrivate", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, value) {
            this._userPrivateProperties[key] = true;
            return _super.prototype.setPrivate.call(this, key, value);
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Entity.prototype, "setPrivateRaw", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, value) {
            this._userPrivateProperties[key] = true;
            _super.prototype.setPrivateRaw.call(this, key, value);
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Entity.prototype, "removePrivate", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key) {
            delete this._userPrivateProperties[key];
            this._removeTemplatePrivateProperty(key);
        }
    });
    Object.defineProperty(Entity.prototype, "_setTemplateProperty", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (template, key, value) {
            if (!this._userProperties[key]) {
                var match = this._findTemplateByKey(key);
                if (template === match) {
                    _super.prototype.set.call(this, key, value);
                }
            }
        }
    });
    Object.defineProperty(Entity.prototype, "_setTemplatePrivateProperty", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (template, key, value) {
            if (!this._userPrivateProperties[key]) {
                var match = this._findTemplateByPrivateKey(key);
                if (template === match) {
                    _super.prototype.setPrivate.call(this, key, value);
                }
            }
        }
    });
    Object.defineProperty(Entity.prototype, "_removeTemplateProperty", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key) {
            if (!this._userProperties[key]) {
                var match = this._findTemplateByKey(key);
                if (match) {
                    // TODO don't stop the animation if the property didn't change
                    _super.prototype.set.call(this, key, match._settings[key]);
                }
                else {
                    _super.prototype.remove.call(this, key);
                }
            }
        }
    });
    Object.defineProperty(Entity.prototype, "_removeTemplatePrivateProperty", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key) {
            if (!this._userPrivateProperties[key]) {
                var match = this._findTemplateByPrivateKey(key);
                if (match) {
                    // TODO don't stop the animation if the property didn't change
                    _super.prototype.setPrivate.call(this, key, match._privateSettings[key]);
                }
                else {
                    _super.prototype.removePrivate.call(this, key);
                }
            }
        }
    });
    Object.defineProperty(Entity.prototype, "_walkParents", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (f) {
            f(this._root._rootContainer);
            f(this);
        }
    });
    // TODO faster version of this method which is specialized to just 1 key
    Object.defineProperty(Entity.prototype, "_applyStateByKey", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (name) {
            var other = this.states.create(name, {});
            var seen = {};
            this._eachTemplate(function (template) {
                var state = template.states.lookup(name);
                if (state) {
                    state._apply(other, seen);
                }
            });
            $object.each(other._settings, function (key) {
                if (!seen[key] && !other._userSettings[key]) {
                    other.remove(key);
                }
            });
        }
    });
    Object.defineProperty(Entity.prototype, "_applyTemplate", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (template, state) {
            var _this = this;
            this._templateDisposers.push(template._apply(this, state));
            $object.each(template._settings, function (key, value) {
                if (!state.settings[key] && !_this._userProperties[key]) {
                    state.settings[key] = true;
                    _super.prototype.set.call(_this, key, value);
                }
            });
            $object.each(template._privateSettings, function (key, value) {
                if (!state.privateSettings[key] && !_this._userPrivateProperties[key]) {
                    state.privateSettings[key] = true;
                    _super.prototype.setPrivate.call(_this, key, value);
                }
            });
            if (this._runSetup && template.setup) {
                this._runSetup = false;
                template.setup(this);
            }
        }
    });
    /**
     * Calls the closure with each template and returns the first template which is true
     */
    Object.defineProperty(Entity.prototype, "_findStaticTemplate", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (f) {
            if (this._template) {
                if (f(this._template)) {
                    return this._template;
                }
            }
        }
    });
    Object.defineProperty(Entity.prototype, "_eachTemplate", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (f) {
            this._findStaticTemplate(function (template) {
                f(template);
                return false;
            });
            // _internalTemplates is sorted with most specific to the right
            $array.eachReverse(this._internalTemplates, f);
            // _templates is sorted with most specific to the left
            $array.each(this._templates, f);
        }
    });
    Object.defineProperty(Entity.prototype, "_applyTemplates", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (remove) {
            var _this = this;
            if (remove === void 0) { remove = true; }
            if (remove) {
                this._disposeTemplates();
            }
            var state = {
                settings: {},
                privateSettings: {},
                states: {},
            };
            this._eachTemplate(function (template) {
                _this._applyTemplate(template, state);
            });
            if (remove) {
                $object.each(this._settings, function (key) {
                    if (!_this._userProperties[key] && !state.settings[key]) {
                        _super.prototype.remove.call(_this, key);
                    }
                });
                $object.each(this._privateSettings, function (key) {
                    if (!_this._userPrivateProperties[key] && !state.privateSettings[key]) {
                        _super.prototype.removePrivate.call(_this, key);
                    }
                });
            }
        }
    });
    Object.defineProperty(Entity.prototype, "_findTemplate", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (f) {
            var value = this._findStaticTemplate(f);
            if (value === undefined) {
                // _internalTemplates is sorted with most specific to the right
                var value_1 = $array.findReverse(this._internalTemplates, f);
                if (value_1 === undefined) {
                    // _templates is sorted with most specific to the left
                    return $array.find(this._templates, f);
                }
                else {
                    return value_1;
                }
            }
            else {
                return value;
            }
        }
    });
    Object.defineProperty(Entity.prototype, "_findTemplateByKey", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key) {
            return this._findTemplate(function (template) {
                return key in template._settings;
            });
        }
    });
    Object.defineProperty(Entity.prototype, "_findTemplateByPrivateKey", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key) {
            return this._findTemplate(function (template) {
                return key in template._privateSettings;
            });
        }
    });
    Object.defineProperty(Entity.prototype, "_disposeTemplates", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            $array.each(this._templateDisposers, function (disposer) {
                disposer.dispose();
            });
            this._templateDisposers.length = 0;
        }
    });
    Object.defineProperty(Entity.prototype, "_removeTemplates", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            $array.each(this._templates, function (template) {
                template._removeObjectTemplate(_this);
            });
            this._templates.length = 0;
        }
    });
    Object.defineProperty(Entity.prototype, "_applyThemes", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            var isConnected = false;
            var defaults = [];
            var themes = [];
            var themeTags = new Set();
            var tags = this.get("themeTagsSelf");
            if (tags) {
                $array.each(tags, function (tag) {
                    themeTags.add(tag);
                });
            }
            this._walkParents(function (entity) {
                if (entity === _this._root._rootContainer) {
                    isConnected = true;
                }
                if (entity._defaultThemes.length > 0) {
                    defaults.push(entity._defaultThemes);
                }
                var theme = entity.get("themes");
                if (theme) {
                    themes.push(theme);
                }
                var tags = entity.get("themeTags");
                if (tags) {
                    $array.each(tags, function (tag) {
                        themeTags.add(tag);
                    });
                }
            });
            themes = defaults.concat(themes);
            this._removeTemplates();
            if (isConnected) {
                $array.eachReverse(this.classNames, function (name) {
                    var allRules = [];
                    $array.each(themes, function (themes) {
                        $array.each(themes, function (theme) {
                            var rules = theme._lookupRules(name);
                            if (rules) {
                                $array.eachReverse(rules, function (rule) {
                                    var matches = rule.tags.every(function (tag) {
                                        return themeTags.has(tag);
                                    });
                                    if (matches) {
                                        var result = $array.getFirstSortedIndex(allRules, function (x) {
                                            var order = $order.compare(rule.tags.length, x.tags.length);
                                            if (order === 0) {
                                                return $order.compareArray(rule.tags, x.tags, $order.compare);
                                            }
                                            else {
                                                return order;
                                            }
                                        });
                                        allRules.splice(result.index, 0, rule);
                                    }
                                });
                            }
                        });
                    });
                    $array.each(allRules, function (rule) {
                        _this._templates.push(rule.template);
                        rule.template._setObjectTemplate(_this);
                    });
                });
            }
            this._applyTemplates();
            if (isConnected) {
                // This causes it to only run the setup function the first time that the themes are applied
                this._runSetup = false;
            }
            return isConnected;
        }
    });
    Object.defineProperty(Entity.prototype, "_changed", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () { }
    });
    Object.defineProperty(Entity.prototype, "_beforeChanged", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (this.isDirty("id")) {
                var id = this.get("id");
                if (id) {
                    if (registry.entitiesById[id]) {
                        throw new Error("An entity with id \"" + id + "\" already exists.");
                    }
                    registry.entitiesById[id] = this;
                }
                var prevId = this._prevSettings.id;
                if (prevId) {
                    delete registry.entitiesById[prevId];
                }
            }
        }
    });
    Object.defineProperty(Entity.prototype, "_afterChanged", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () { }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Entity.prototype, "addDisposer", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (disposer) {
            this._disposers.push(disposer);
            return disposer;
        }
    });
    Object.defineProperty(Entity.prototype, "_dispose", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            _super.prototype._dispose.call(this);
            var template = this._template;
            if (template) {
                template._removeObjectTemplate(this);
            }
            $array.each(this._internalTemplates, function (template) {
                template._removeObjectTemplate(_this);
            });
            this._removeTemplates();
            this._disposeTemplates();
            this.events.dispose();
            this._disposers.forEach(function (x) {
                x.dispose();
            });
            $object.each(this._disposerProperties, function (_, disposers) {
                $array.each(disposers, function (disposer) {
                    disposer.dispose();
                });
            });
            var id = this.get("id");
            if (id) {
                delete registry.entitiesById[id];
            }
        }
    });
    /**
     * Creates and returns a "disposable" timeout.
     *
     * @param   fn     Callback
     * @param   delay  Delay in milliseconds
     * @return         Timeout disposer
     */
    Object.defineProperty(Entity.prototype, "setTimeout", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (fn, delay) {
            var _this = this;
            var id = setTimeout(function () {
                _this.removeDispose(disposer);
                fn();
            }, delay);
            var disposer = new Disposer(function () {
                clearTimeout(id);
            });
            this._disposers.push(disposer);
            return disposer;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Entity.prototype, "removeDispose", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (target) {
            if (!this.isDisposed()) {
                var index = $array.indexOf(this._disposers, target);
                if (index > -1) {
                    this._disposers.splice(index, 1);
                }
            }
            target.dispose();
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Entity.prototype, "hasTag", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (tag) {
            return $array.indexOf(this.get("themeTags", []), tag) !== -1;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Entity.prototype, "addTag", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (tag) {
            if (!this.hasTag(tag)) {
                var tags = this.get("themeTags", []);
                tags.push(tag);
                this.set("themeTags", tags);
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Entity.prototype, "removeTag", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (tag) {
            if (this.hasTag(tag)) {
                var tags = this.get("themeTags", []);
                $array.remove(tags, tag);
                this.set("themeTags", tags);
            }
        }
    });
    Object.defineProperty(Entity.prototype, "_t", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (text, locale) {
            var _a;
            var rest = [];
            for (var _i = 2; _i < arguments.length; _i++) {
                rest[_i - 2] = arguments[_i];
            }
            return (_a = this._root.language).translate.apply(_a, __spreadArray([text, locale], __read(rest)));
        }
    });
    Object.defineProperty(Entity.prototype, "root", {
        /**
         * An instance of [[Root]] object.
         *
         * @readonly
         * @since 5.0.6
         * @return Root object
         */
        get: function () {
            return this._root;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(Entity, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "Entity"
    });
    Object.defineProperty(Entity, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: ["Entity"]
    });
    return Entity;
}(Settings));
export { Entity };
//# sourceMappingURL=Entity.js.map