/**
 * Event Dispatcher module is used for registering listeners and dispatching
 * events across amCharts system.
 */
import { __extends } from "tslib";
/**
 * ============================================================================
 * IMPORTS
 * ============================================================================
 * @hidden
 */
import { Disposer, MultiDisposer } from "./Disposer";
import * as $array from "./Array";
import * as $type from "./Type";
/**
 * Universal Event Dispatcher.
 *
 * @see {@link https://www.amcharts.com/docs/v5/concepts/events/} for more info
 */
var EventDispatcher = /** @class */ (function () {
    /**
     * Constructor
     */
    function EventDispatcher() {
        Object.defineProperty(this, "_listeners", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_killed", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_disabled", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_iterating", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_enabled", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_disposed", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        this._listeners = [];
        this._killed = [];
        this._disabled = {};
        this._iterating = 0;
        this._enabled = true;
        this._disposed = false;
    }
    /**
     * Returns if this object has been already disposed.
     *
     * @return Disposed?
     */
    Object.defineProperty(EventDispatcher.prototype, "isDisposed", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this._disposed;
        }
    });
    /**
     * Dispose (destroy) this object.
     */
    Object.defineProperty(EventDispatcher.prototype, "dispose", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (!this._disposed) {
                this._disposed = true;
                var a = this._listeners;
                this._iterating = 1;
                this._listeners = null;
                this._disabled = null;
                try {
                    $array.each(a, function (x) {
                        x.disposer.dispose();
                    });
                }
                finally {
                    this._killed = null;
                    this._iterating = null;
                }
            }
        }
    });
    /**
     * Checks if this particular event dispatcher has any listeners set.
     *
     * @return Has listeners?
     */
    Object.defineProperty(EventDispatcher.prototype, "hasListeners", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this._listeners.length !== 0;
        }
    });
    /**
     * Checks if this particular event dispatcher has any particular listeners set.
     *
     * @return Has particular event listeners?
     */
    Object.defineProperty(EventDispatcher.prototype, "hasListenersByType", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (type) {
            return $array.any(this._listeners, function (x) { return (x.type === null || x.type === type) && !x.killed; });
        }
    });
    /**
     * Enable dispatching of events if they were previously disabled by
     * `disable()`.
     */
    Object.defineProperty(EventDispatcher.prototype, "enable", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._enabled = true;
        }
    });
    /**
     * Disable dispatching of events until re-enabled by `enable()`.
     */
    Object.defineProperty(EventDispatcher.prototype, "disable", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._enabled = false;
        }
    });
    /**
     * Enable dispatching particular event, if it was disabled before by
     * `disableType()`.
     *
     * @param type Event type
     */
    Object.defineProperty(EventDispatcher.prototype, "enableType", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (type) {
            delete this._disabled[type];
        }
    });
    /**
     * Disable dispatching of events for a certain event type.
     *
     * Optionally, can set how many dispatches to skip before automatically
     * re-enabling the dispatching.
     *
     * @param type    Event type
     * @param amount  Number of event dispatches to skip
     */
    Object.defineProperty(EventDispatcher.prototype, "disableType", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (type, amount) {
            if (amount === void 0) { amount = Infinity; }
            this._disabled[type] = amount;
        }
    });
    /**
     * Removes listener from dispatcher.
     *
     * Will throw an exception if such listener does not exists.
     *
     * @param listener Listener to remove
     */
    Object.defineProperty(EventDispatcher.prototype, "_removeListener", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (listener) {
            if (this._iterating === 0) {
                var index = this._listeners.indexOf(listener);
                if (index === -1) {
                    throw new Error("Invalid state: could not remove listener");
                }
                this._listeners.splice(index, 1);
            }
            else {
                this._killed.push(listener);
            }
        }
    });
    /**
     * Removes existing listener by certain parameters.
     *
     * @param once         Listener's once setting
     * @param type         Listener's type
     * @param callback     Callback function
     * @param context      Callback context
     */
    Object.defineProperty(EventDispatcher.prototype, "_removeExistingListener", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (once, type, callback, context) {
            if (this._disposed) {
                throw new Error("EventDispatcher is disposed");
            }
            this._eachListener(function (info) {
                if (info.once === once && // TODO is this correct ?
                    info.type === type &&
                    (callback === undefined || info.callback === callback) &&
                    info.context === context) {
                    info.disposer.dispose();
                }
            });
        }
    });
    /**
     * Checks if dispatching for particular event type is enabled.
     *
     * @param type  Event type
     * @return Enabled?
     */
    Object.defineProperty(EventDispatcher.prototype, "isEnabled", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (type) {
            if (this._disposed) {
                throw new Error("EventDispatcher is disposed");
            }
            // TODO is this check correct ?
            return this._enabled && this._listeners.length > 0 && this.hasListenersByType(type) && this._disabled[type] === undefined;
        }
    });
    /**
     * Checks if there's already a listener with specific parameters.
     *
     * @param type      Listener's type
     * @param callback  Callback function
     * @param context   Callback context
     * @return Has listener?
     */
    Object.defineProperty(EventDispatcher.prototype, "has", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (type, callback, context) {
            var index = $array.findIndex(this._listeners, function (info) {
                return info.once !== true && // Ignoring "once" listeners
                    info.type === type &&
                    (callback === undefined || info.callback === callback) &&
                    info.context === context;
            });
            return index !== -1;
        }
    });
    /**
     * Checks whether event of the particular type should be dispatched.
     *
     * @param type  Event type
     * @return Dispatch?
     */
    Object.defineProperty(EventDispatcher.prototype, "_shouldDispatch", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (type) {
            if (this._disposed) {
                throw new Error("EventDispatcher is disposed");
            }
            var count = this._disabled[type];
            if (!$type.isNumber(count)) {
                return this._enabled;
            }
            else {
                if (count <= 1) {
                    delete this._disabled[type];
                }
                else {
                    --this._disabled[type];
                }
                return false;
            }
        }
    });
    /**
     * [_eachListener description]
     *
     * All of this extra code is needed when a listener is removed while iterating
     *
     * @todo Description
     * @param fn [description]
     */
    Object.defineProperty(EventDispatcher.prototype, "_eachListener", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (fn) {
            var _this = this;
            ++this._iterating;
            try {
                $array.each(this._listeners, fn);
            }
            finally {
                --this._iterating;
                // TODO should this be inside or outside the finally ?
                if (this._iterating === 0 && this._killed.length !== 0) {
                    // Remove killed listeners
                    $array.each(this._killed, function (killed) {
                        _this._removeListener(killed);
                    });
                    this._killed.length = 0;
                }
            }
        }
    });
    /**
     * Dispatches an event immediately without waiting for next cycle.
     *
     * @param type   Event type
     * @param event  Event object
     * @todo automatically add in type and target properties if they are missing
     */
    Object.defineProperty(EventDispatcher.prototype, "dispatch", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (type, event) {
            if (this._shouldDispatch(type)) {
                // TODO check if it's faster to use an object of listeners rather than a single big array
                // TODO if the function throws, maybe it should keep going ?
                this._eachListener(function (listener) {
                    if (!listener.killed && (listener.type === null || listener.type === type)) {
                        listener.dispatch(type, event);
                    }
                });
            }
        }
    });
    /**
     * Shelves the event to be dispatched within next update cycle.
     *
     * @param type   Event type
     * @param event  Event object
     * @todo automatically add in type and target properties if they are missing
     */
    /*public dispatchLater<Key extends keyof T>(type: Key, event: T[Key]): void {
        if (this._shouldDispatch(type)) {
            this._eachListener((listener) => {
                // TODO check if it's faster to use an object of listeners rather than a single big array
                if (!listener.killed && (listener.type === null || listener.type === type)) {
                    // TODO if the function throws, maybe it should keep going ?
                    // TODO dispatch during the update cycle, rather than using whenIdle
                    $async.whenIdle(() => {
                        if (!listener.killed) {
                            listener.dispatch(type, event);
                        }
                    });
                }
            });
        }
    }*/
    /**
     * Creates, catalogs and returns an [[EventListener]].
     *
     * Event listener can be disposed.
     *
     * @param once         Listener's once setting
     * @param type         Listener's type
     * @param callback     Callback function
     * @param context      Callback context
     * @param shouldClone  Whether the listener should be copied when the EventDispatcher is copied
     * @param dispatch
     * @returns An event listener
     */
    Object.defineProperty(EventDispatcher.prototype, "_on", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (once, type, callback, context, shouldClone, dispatch) {
            var _this = this;
            if (this._disposed) {
                throw new Error("EventDispatcher is disposed");
            }
            this._removeExistingListener(once, type, callback, context);
            var info = {
                type: type,
                callback: callback,
                context: context,
                shouldClone: shouldClone,
                dispatch: dispatch,
                killed: false,
                once: once,
                disposer: new Disposer(function () {
                    info.killed = true;
                    _this._removeListener(info);
                })
            };
            this._listeners.push(info);
            return info;
        }
    });
    /**
     * Creates an event listener to be invoked on **any** event.
     *
     * @param callback     Callback function
     * @param context      Callback context
     * @param shouldClone  Whether the listener should be copied when the EventDispatcher is copied
     * @returns A disposable event listener
     */
    Object.defineProperty(EventDispatcher.prototype, "onAll", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (callback, context, shouldClone) {
            if (shouldClone === void 0) { shouldClone = true; }
            return this._on(false, null, callback, context, shouldClone, function (_type, event) { return callback.call(context, event); }).disposer;
        }
    });
    /**
     * Creates an event listener to be invoked on a specific event type.
     *
     * ```TypeScript
     * button.events.once("click", (ev) => {
     *   console.log("Button clicked");
     * }, this);
     * ```
     * ```JavaScript
     * button.events.once("click", (ev) => {
     *   console.log("Button clicked");
     * }, this);
     * ```
     *
     * The above will invoke our custom event handler whenever series we put
     * event on is hidden.
     *
     * @param type         Listener's type
     * @param callback     Callback function
     * @param context      Callback context
     * @param shouldClone  Whether the listener should be copied when the EventDispatcher is copied
     * @returns A disposable event listener
     */
    Object.defineProperty(EventDispatcher.prototype, "on", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (type, callback, context, shouldClone) {
            if (shouldClone === void 0) { shouldClone = true; }
            return this._on(false, type, callback, context, shouldClone, function (_type, event) { return callback.call(context, event); }).disposer;
        }
    });
    /**
     * Creates an event listener to be invoked on a specific event type once.
     *
     * Once the event listener is invoked, it is automatically disposed.
     *
     * ```TypeScript
     * button.events.once("click", (ev) => {
     *   console.log("Button clicked");
     * }, this);
     * ```
     * ```JavaScript
     * button.events.once("click", (ev) => {
     *   console.log("Button clicked");
     * }, this);
     * ```
     *
     * The above will invoke our custom event handler the first time series we
     * put event on is hidden.
     *
     * @param type         Listener's type
     * @param callback     Callback function
     * @param context      Callback context
     * @param shouldClone  Whether the listener should be copied when the EventDispatcher is copied
     * @returns A disposable event listener
     */
    Object.defineProperty(EventDispatcher.prototype, "once", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (type, callback, context, shouldClone) {
            if (shouldClone === void 0) { shouldClone = true; }
            var x = this._on(true, type, callback, context, shouldClone, function (_type, event) {
                x.disposer.dispose();
                callback.call(context, event);
            });
            // TODO maybe this should return a different Disposer ?
            return x.disposer;
        }
    });
    /**
     * Removes the event listener with specific parameters.
     *
     * @param type         Listener's type
     * @param callback     Callback function
     * @param context      Callback context
     */
    Object.defineProperty(EventDispatcher.prototype, "off", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (type, callback, context) {
            this._removeExistingListener(false, type, callback, context);
        }
    });
    /**
     * Copies all dispatcher parameters, including listeners, from another event
     * dispatcher.
     *
     * @param source Source event dispatcher
     * @ignore
     */
    Object.defineProperty(EventDispatcher.prototype, "copyFrom", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (source) {
            var _this = this;
            if (this._disposed) {
                throw new Error("EventDispatcher is disposed");
            }
            if (source === this) {
                throw new Error("Cannot copyFrom the same TargetedEventDispatcher");
            }
            var disposers = [];
            $array.each(source._listeners, function (x) {
                // TODO is this correct ?
                if (!x.killed && x.shouldClone) {
                    if (x.type === null) {
                        disposers.push(_this.onAll(x.callback, x.context));
                    }
                    else if (x.once) {
                        disposers.push(_this.once(x.type, x.callback, x.context));
                    }
                    else {
                        disposers.push(_this.on(x.type, x.callback, x.context));
                    }
                }
            });
            return new MultiDisposer(disposers);
        }
    });
    return EventDispatcher;
}());
export { EventDispatcher };
/**
 * A version of the [[EventDispatcher]] that dispatches events for a specific
 * target object.
 *
 * @ignore
 */
var TargetedEventDispatcher = /** @class */ (function (_super) {
    __extends(TargetedEventDispatcher, _super);
    /**
     * Constructor
     *
     * @param target Event dispatcher target
     */
    function TargetedEventDispatcher(target) {
        var _this = _super.call(this) || this;
        /**
         * A target object which is originating events using this dispatcher.
         */
        Object.defineProperty(_this, "target", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        _this.target = target;
        return _this;
    }
    /**
     * Copies all dispatcher parameters, including listeners, from another event
     * dispatcher.
     *
     * @param source Source event dispatcher
     * @ignore
     */
    Object.defineProperty(TargetedEventDispatcher.prototype, "copyFrom", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (source) {
            var _this = this;
            if (this._disposed) {
                throw new Error("EventDispatcher is disposed");
            }
            if (source === this) {
                throw new Error("Cannot copyFrom the same TargetedEventDispatcher");
            }
            var disposers = [];
            $array.each(source._listeners, function (x) {
                // TODO very hacky
                if (x.context === source.target) {
                    return;
                }
                // TODO is this correct ?
                if (!x.killed && x.shouldClone) {
                    if (x.type === null) {
                        disposers.push(_this.onAll(x.callback, x.context));
                    }
                    else if (x.once) {
                        disposers.push(_this.once(x.type, x.callback, x.context));
                    }
                    else {
                        disposers.push(_this.on(x.type, x.callback, x.context));
                    }
                }
            });
            return new MultiDisposer(disposers);
        }
    });
    return TargetedEventDispatcher;
}(EventDispatcher));
export { TargetedEventDispatcher };
//# sourceMappingURL=EventDispatcher.js.map