import { __extends, __generator } from "tslib";
import { EventDispatcher } from "./EventDispatcher";
import * as $array from "./Array";
/**
 * Checks if specific index fits into length.
 *
 * @param index  Index
 * @param len    Length
 * @ignore
 */
function checkBounds(index, len) {
    if (!(index >= 0 && index < len)) {
        throw new Error("Index out of bounds: " + index);
    }
}
/**
 * A List class is used to hold a number of indexed items of the same type.
 */
var List = /** @class */ (function () {
    /**
     * Constructor
     *
     * @param initial  Inital list of values to add to list
     */
    function List(initial) {
        if (initial === void 0) { initial = []; }
        /**
         * List values.
         */
        Object.defineProperty(this, "_values", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "events", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new EventDispatcher()
        });
        this._values = initial;
    }
    Object.defineProperty(List.prototype, "values", {
        /**
         * An array of values in the list.
         *
         * Do not use this property to add values. Rather use dedicated methods, like
         * `push()`, `removeIndex()`, etc.
         *
         * @readonly
         * @return List values
         */
        get: function () {
            return this._values;
        },
        enumerable: false,
        configurable: true
    });
    /**
     * Checks if list contains specific item reference.
     *
     * @param item  Item to search for
     * @return `true` if found, `false` if not found
     */
    Object.defineProperty(List.prototype, "contains", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (value) {
            return this._values.indexOf(value) !== -1;
        }
    });
    /**
     * Removes specific item from the list.
     *
     * @param item An item to remove
     */
    Object.defineProperty(List.prototype, "removeValue", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (value) {
            var i = 0;
            var length = this._values.length;
            while (i < length) {
                // TODO handle NaN
                if (this._values[i] === value) {
                    this.removeIndex(i);
                    --length;
                }
                else {
                    ++i;
                }
            }
        }
    });
    /**
     * Searches the list for specific item and returns its index.
     *
     * @param item  An item to search for
     * @return Index or -1 if not found
     */
    Object.defineProperty(List.prototype, "indexOf", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (value) {
            return $array.indexOf(this._values, value);
        }
    });
    Object.defineProperty(List.prototype, "length", {
        /**
         * Number of items in list.
         *
         * @readonly
         * @return Number of items
         */
        get: function () {
            return this._values.length;
        },
        enumerable: false,
        configurable: true
    });
    /**
     * Checks if there's a value at specific index.
     *
     * @param index  Index
     * @return Value exists?
     */
    Object.defineProperty(List.prototype, "hasIndex", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (index) {
            return index >= 0 && index < this._values.length;
        }
    });
    /**
     * Returns an item at specified index.
     *
     * @param index  Index
     * @return List item
     */
    Object.defineProperty(List.prototype, "getIndex", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (index) {
            return this._values[index];
        }
    });
    Object.defineProperty(List.prototype, "_onPush", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (newValue) {
            if (this.events.isEnabled("push")) {
                this.events.dispatch("push", {
                    type: "push",
                    target: this,
                    newValue: newValue
                });
            }
        }
    });
    Object.defineProperty(List.prototype, "_onInsertIndex", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (index, newValue) {
            if (this.events.isEnabled("insertIndex")) {
                this.events.dispatch("insertIndex", {
                    type: "insertIndex",
                    target: this,
                    index: index,
                    newValue: newValue
                });
            }
        }
    });
    Object.defineProperty(List.prototype, "_onSetIndex", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (index, oldValue, newValue) {
            if (this.events.isEnabled("setIndex")) {
                this.events.dispatch("setIndex", {
                    type: "setIndex",
                    target: this,
                    index: index,
                    oldValue: oldValue,
                    newValue: newValue
                });
            }
        }
    });
    Object.defineProperty(List.prototype, "_onRemoveIndex", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (index, oldValue) {
            if (this.events.isEnabled("removeIndex")) {
                this.events.dispatch("removeIndex", {
                    type: "removeIndex",
                    target: this,
                    index: index,
                    oldValue: oldValue
                });
            }
        }
    });
    Object.defineProperty(List.prototype, "_onMoveIndex", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (oldIndex, newIndex, value) {
            if (this.events.isEnabled("moveIndex")) {
                this.events.dispatch("moveIndex", {
                    type: "moveIndex",
                    target: this,
                    oldIndex: oldIndex,
                    newIndex: newIndex,
                    value: value,
                });
            }
        }
    });
    Object.defineProperty(List.prototype, "_onClear", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (oldValues) {
            if (this.events.isEnabled("clear")) {
                this.events.dispatch("clear", {
                    type: "clear",
                    target: this,
                    oldValues: oldValues
                });
            }
        }
    });
    /**
     * Sets value at specific index.
     *
     * If there's already a value at the index, it is overwritten.
     *
     * @param index  Index
     * @param value  New value
     * @return New value
     */
    Object.defineProperty(List.prototype, "setIndex", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (index, value) {
            checkBounds(index, this._values.length);
            var oldValue = this._values[index];
            // Do nothing if the old value and the new value are the same
            if (oldValue !== value) {
                this._values[index] = value;
                this._onSetIndex(index, oldValue, value);
            }
            return oldValue;
        }
    });
    /**
     * Adds an item to the list at a specific index, which pushes all the other
     * items further down the list.
     *
     * @param index Index
     * @param item  An item to add
     */
    Object.defineProperty(List.prototype, "insertIndex", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (index, value) {
            checkBounds(index, this._values.length + 1);
            $array.insertIndex(this._values, index, value);
            this._onInsertIndex(index, value);
            return value;
        }
    });
    /**
     * Swaps indexes of two items in the list.
     *
     * @param a  Item 1
     * @param b  Item 2
     */
    Object.defineProperty(List.prototype, "swap", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (a, b) {
            var len = this._values.length;
            checkBounds(a, len);
            checkBounds(b, len);
            if (a !== b) {
                var value_a = this._values[a];
                var value_b = this._values[b];
                this._values[a] = value_b;
                this._onSetIndex(a, value_a, value_b);
                this._values[b] = value_a;
                this._onSetIndex(b, value_b, value_a);
            }
        }
    });
    /**
     * Removes a value at specific index.
     *
     * @param index  Index of value to remove
     * @return Removed value
     */
    Object.defineProperty(List.prototype, "removeIndex", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (index) {
            checkBounds(index, this._values.length);
            var oldValue = this._values[index];
            $array.removeIndex(this._values, index);
            this._onRemoveIndex(index, oldValue);
            return oldValue;
        }
    });
    /**
     * Moves an item to a specific index within the list.
     *
     * If the index is not specified it will move the item to the end of the
     * list.
     *
     * @param value  Item to move
     * @param index  Index to place item at
     */
    Object.defineProperty(List.prototype, "moveValue", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (value, toIndex) {
            // TODO don't do anything if the desired index is the same as the current index
            var index = this.indexOf(value);
            // TODO remove all old values rather than only the first ?
            if (index !== -1) {
                $array.removeIndex(this._values, index);
                if (toIndex == null) {
                    var toIndex_1 = this._values.length;
                    this._values.push(value);
                    this._onMoveIndex(index, toIndex_1, value);
                }
                else {
                    $array.insertIndex(this._values, toIndex, value);
                    this._onMoveIndex(index, toIndex, value);
                }
            }
            else if (toIndex == null) {
                this._values.push(value);
                this._onPush(value);
            }
            else {
                $array.insertIndex(this._values, toIndex, value);
                this._onInsertIndex(toIndex, value);
            }
            return value;
        }
    });
    /**
     * Adds an item to the end of the list.
     *
     * @param item  An item to add
     */
    Object.defineProperty(List.prototype, "push", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (value) {
            this._values.push(value);
            this._onPush(value);
            return value;
        }
    });
    /**
     * Adds an item as a first item in the list.
     *
     * @param item  An item to add
     */
    Object.defineProperty(List.prototype, "unshift", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (value) {
            this.insertIndex(0, value);
            return value;
        }
    });
    /**
     * Adds multiple items to the list.
     *
     * @param items  An Array of items to add
     */
    Object.defineProperty(List.prototype, "pushAll", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (values) {
            var _this = this;
            $array.each(values, function (value) {
                _this.push(value);
            });
        }
    });
    /**
     * Copies and adds items from abother list.
     *
     * @param source  A list top copy items from
     */
    Object.defineProperty(List.prototype, "copyFrom", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (source) {
            this.pushAll(source._values);
        }
    });
    /**
     * Returns the last item from the list, and removes it.
     *
     * @return Item
     */
    Object.defineProperty(List.prototype, "pop", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var index = this._values.length - 1;
            return index < 0 ? undefined : this.removeIndex(this._values.length - 1);
        }
    });
    /**
     * Returns the first item from the list, and removes it.
     *
     * @return Item
     */
    Object.defineProperty(List.prototype, "shift", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this._values.length ? this.removeIndex(0) : undefined;
        }
    });
    /**
     * Sets multiple items to the list.
     *
     * All current items are removed.
     *
     * @param newArray  New items
     */
    Object.defineProperty(List.prototype, "setAll", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (newArray) {
            var _this = this;
            var old = this._values;
            this._values = [];
            this._onClear(old);
            $array.each(newArray, function (value) {
                _this._values.push(value);
                _this._onPush(value);
            });
        }
    });
    /**
     * Removes all items from the list.
     */
    Object.defineProperty(List.prototype, "clear", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this.setAll([]);
        }
    });
    /**
     * Returns an ES6 iterator for the list.
     */
    Object.defineProperty(List.prototype, Symbol.iterator, {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var length, i;
            return __generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        length = this._values.length;
                        i = 0;
                        _a.label = 1;
                    case 1:
                        if (!(i < length)) return [3 /*break*/, 4];
                        return [4 /*yield*/, this._values[i]];
                    case 2:
                        _a.sent();
                        _a.label = 3;
                    case 3:
                        ++i;
                        return [3 /*break*/, 1];
                    case 4: return [2 /*return*/];
                }
            });
        }
    });
    /**
     * Calls `f` for each element in the list.
     *
     * `f` should have at least one parameter defined which will get a current
     * item, with optional second argument - index.
     */
    Object.defineProperty(List.prototype, "each", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (f) {
            $array.each(this._values, f);
        }
    });
    /**
     * Calls `f` for each element in the list, from right to left.
     *
     * `f` should have at least one parameter defined which will get a current
     * item, with optional second argument - index.
     */
    Object.defineProperty(List.prototype, "eachReverse", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (f) {
            $array.eachReverse(this._values, f);
        }
    });
    return List;
}());
export { List };
/**
 * A version of a [[List]] where the elements are disposed automatically when
 * removed from the list, unless `autoDispose` is set to `false`.
 */
var ListAutoDispose = /** @class */ (function (_super) {
    __extends(ListAutoDispose, _super);
    function ListAutoDispose() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        /**
         * Automatically disposes elements that are removed from the list.
         *
         * @default true
         */
        Object.defineProperty(_this, "autoDispose", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: true
        });
        Object.defineProperty(_this, "_disposed", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        return _this;
    }
    Object.defineProperty(ListAutoDispose.prototype, "_onSetIndex", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (index, oldValue, newValue) {
            if (this.autoDispose) {
                oldValue.dispose();
            }
            _super.prototype._onSetIndex.call(this, index, oldValue, newValue);
        }
    });
    Object.defineProperty(ListAutoDispose.prototype, "_onRemoveIndex", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (index, oldValue) {
            if (this.autoDispose) {
                oldValue.dispose();
            }
            _super.prototype._onRemoveIndex.call(this, index, oldValue);
        }
    });
    Object.defineProperty(ListAutoDispose.prototype, "_onClear", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (oldValues) {
            if (this.autoDispose) {
                $array.each(oldValues, function (x) {
                    x.dispose();
                });
            }
            _super.prototype._onClear.call(this, oldValues);
        }
    });
    Object.defineProperty(ListAutoDispose.prototype, "isDisposed", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this._disposed;
        }
    });
    Object.defineProperty(ListAutoDispose.prototype, "dispose", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (!this._disposed) {
                this._disposed = true;
                if (this.autoDispose) {
                    $array.each(this._values, function (x) {
                        x.dispose();
                    });
                }
            }
        }
    });
    return ListAutoDispose;
}(List));
export { ListAutoDispose };
/**
 * A version of a [[List]] that is able to create new elements as well as
 * apply additional settings to newly created items.
 *
 * @see {@link https://www.amcharts.com/docs/v5/concepts/settings/list-templates/} for more info
 */
var ListTemplate = /** @class */ (function (_super) {
    __extends(ListTemplate, _super);
    function ListTemplate(template, make) {
        var _this = _super.call(this) || this;
        Object.defineProperty(_this, "template", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "make", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        _this.template = template;
        _this.make = make;
        return _this;
    }
    return ListTemplate;
}(ListAutoDispose));
export { ListTemplate };
//# sourceMappingURL=List.js.map