import * as $object from "./Object";
import * as $ease from "./Ease";
/**
 * An object representing a collection of setting values to apply as required.
 *
 * @see {@link https://www.amcharts.com/docs/v5/concepts/settings/states/} for more info
 */
var State = /** @class */ (function () {
    function State(entity, settings) {
        Object.defineProperty(this, "_entity", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_settings", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_userSettings", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        this._entity = entity;
        this._settings = settings;
    }
    Object.defineProperty(State.prototype, "get", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, fallback) {
            var value = this._settings[key];
            if (value !== undefined) {
                return value;
            }
            else {
                return fallback;
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(State.prototype, "setRaw", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, value) {
            this._settings[key] = value;
        }
    });
    /**
     * Sets a setting `value` for the specified `key` to be set when the state
     * is applied.
     *
     * @param   key       Setting key
     * @param   value     Setting value
     * @return            Setting value
     */
    Object.defineProperty(State.prototype, "set", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, value) {
            this._userSettings[key] = true;
            this.setRaw(key, value);
        }
    });
    /**
     * Removes a setting value for the specified `key`.
     *
     * @see {@link https://www.amcharts.com/docs/v5/concepts/settings/} for more info
     * @param   key       Setting key
     */
    Object.defineProperty(State.prototype, "remove", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key) {
            delete this._userSettings[key];
            delete this._settings[key];
        }
    });
    /**
     * Sets multiple settings at once.
     *
     * `settings` must be an object with key: value pairs.
     *
     * @see {@link https://www.amcharts.com/docs/v5/concepts/settings/} for more info
     * @param settings Settings
     */
    Object.defineProperty(State.prototype, "setAll", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (settings) {
            var _this = this;
            $object.keys(settings).forEach(function (key) {
                _this.set(key, settings[key]);
            });
        }
    });
    Object.defineProperty(State.prototype, "_eachSetting", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (f) {
            $object.each(this._settings, f);
        }
    });
    /**
     * Applies the state to the target element.
     *
     * All setting values are set immediately.
     */
    Object.defineProperty(State.prototype, "apply", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            var seen = {};
            seen["stateAnimationEasing"] = true;
            seen["stateAnimationDuration"] = true;
            var defaultState = this._entity.states.lookup("default");
            this._eachSetting(function (key, value) {
                if (!seen[key]) {
                    seen[key] = true;
                    // save values to default state
                    if (_this !== defaultState) {
                        if (!(key in defaultState._settings)) {
                            defaultState._settings[key] = _this._entity.get(key);
                        }
                    }
                    _this._entity.set(key, value);
                }
            });
        }
    });
    /**
     * Applies the state to the target element.
     *
     * Returns an object representing all [[Animation]] objects created for
     * each setting key transition.
     *
     * @return           Animations
     */
    Object.defineProperty(State.prototype, "applyAnimate", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (duration) {
            var _this = this;
            if (duration == null) {
                duration = this._settings.stateAnimationDuration;
            }
            if (duration == null) {
                duration = this.get("stateAnimationDuration", this._entity.get("stateAnimationDuration", 0));
            }
            var easing = this._settings.stateAnimationEasing;
            if (easing == null) {
                easing = this.get("stateAnimationEasing", this._entity.get("stateAnimationEasing", $ease.cubic));
            }
            var defaultState = this._entity.states.lookup("default");
            var seen = {};
            seen["stateAnimationEasing"] = true;
            seen["stateAnimationDuration"] = true;
            var animations = {};
            this._eachSetting(function (key, value) {
                if (!seen[key]) {
                    seen[key] = true;
                    // save values to default state
                    if (_this != defaultState) {
                        if (!(key in defaultState._settings)) {
                            defaultState._settings[key] = _this._entity.get(key);
                        }
                    }
                    var animation = _this._entity.animate({
                        key: key,
                        to: value,
                        duration: duration,
                        easing: easing
                    });
                    if (animation) {
                        animations[key] = animation;
                    }
                }
            });
            return animations;
        }
    });
    return State;
}());
export { State };
/**
 * Collection of [[State]] objects for an element.
 *
 * @see {@link https://www.amcharts.com/docs/v5/concepts/settings/states/} for more info
 */
var States = /** @class */ (function () {
    function States(entity) {
        Object.defineProperty(this, "_states", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        Object.defineProperty(this, "_entity", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        this._entity = entity;
    }
    /**
     * Checks if a state by `name` exists. Returns it there is one.
     *
     * @param  name  State name
     * @return       State
     */
    Object.defineProperty(States.prototype, "lookup", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (name) {
            return this._states[name];
        }
    });
    /**
     * Sets supplied `settings` on a state by the `name`.
     *
     * If such state does not yet exists, it is created.
     *
     * @param   name      State name
     * @param   settings  Settings
     * @return            New State
     */
    Object.defineProperty(States.prototype, "create", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (name, settings) {
            var state = this._states[name];
            if (state) {
                state.setAll(settings);
                return state;
            }
            else {
                var state_1 = new State(this._entity, settings);
                this._states[name] = state_1;
                return state_1;
            }
        }
    });
    /**
     * Removes the state called `name`.
     *
     * @param   name      State name
     */
    Object.defineProperty(States.prototype, "remove", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (name) {
            delete this._states[name];
        }
    });
    /**
     * Applies a named state to the target element.
     *
     * @param  newState  State name
     */
    Object.defineProperty(States.prototype, "apply", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (newState) {
            var state = this._states[newState];
            if (state) {
                state.apply();
            }
            this._entity._applyState(newState);
        }
    });
    /**
     * Applies a named state to the element.
     *
     * Returns an object representing all [[Animation]] objects created for
     * each setting key transition.
     *
     * @param   newState  State name
     * @return            Animations
     */
    Object.defineProperty(States.prototype, "applyAnimate", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (newState, duration) {
            var animations;
            var state = this._states[newState];
            if (state) {
                animations = state.applyAnimate(duration);
            }
            this._entity._applyStateAnimated(newState, duration);
            return animations;
        }
    });
    return States;
}());
export { States };
//# sourceMappingURL=States.js.map