import { __awaiter, __extends, __generator } from "tslib";
import { Entity } from "../../core/util/Entity";
import { Color } from "../../core/util/Color";
import { Root } from "../../core/Root";
import { StyleRule } from "../../core/util/Utils";
import * as $array from "../../core/util/Array";
import * as $object from "../../core/util/Object";
import * as $type from "../../core/util/Type";
import * as $utils from "../../core/util/Utils";
/**
 * This is used to cache the pdfmake loading.
 *
 * @ignore
 */
var pdfmakePromise;
/**
 * Loads pdfmake dynamic module
 *
 * This is an asynchronous function. Check the description of `getImage()`
 * for description and example usage.
 *
 * @ignore
 */
function _pdfmake() {
    return __awaiter(this, void 0, void 0, function () {
        var a, pdfmake, vfs_fonts, global;
        return __generator(this, function (_a) {
            switch (_a.label) {
                case 0: return [4 /*yield*/, Promise.all([
                        import(/* webpackChunkName: "pdfmake" */ "pdfmake/build/pdfmake.js"),
                        import(/* webpackChunkName: "pdfmake" */ "./pdfmake/vfs_fonts")
                    ])];
                case 1:
                    a = _a.sent();
                    pdfmake = a[0];
                    vfs_fonts = a[1];
                    global = window;
                    global.pdfMake = global.pdfMake || {};
                    global.pdfMake.vfs = vfs_fonts.default;
                    pdfmake.vfs = vfs_fonts.default;
                    return [2 /*return*/, pdfmake];
            }
        });
    });
}
/**
 * A plugin that can be used to export chart snapshots and data.
 *
 * @see {@link https://www.amcharts.com/docs/v5/concepts/exporting/} for more info
 */
var Exporting = /** @class */ (function (_super) {
    __extends(Exporting, _super);
    function Exporting() {
        return _super !== null && _super.apply(this, arguments) || this;
    }
    //public extraImages: Array<Root | IExportingImageSource> = [];
    //public dataSources: any[] = [];
    Object.defineProperty(Exporting.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._afterNew.call(this);
            this._setRawDefault("filePrefix", "chart");
            this._setRawDefault("charset", "utf-8");
            this._setRawDefault("numericFields", []);
            this._setRawDefault("dateFields", []);
            this._setRawDefault("durationFields", []);
            this._setRawDefault("extraImages", []);
            this._setRawDefault("pngOptions", { quality: 1, maintainPixelRatio: false });
            this._setRawDefault("jpgOptions", { quality: 0.8, maintainPixelRatio: false });
            this._setRawDefault("printOptions", { quality: 1, maintainPixelRatio: false, delay: 500, printMethod: "iframe", imageFormat: "png" });
            this._setRawDefault("jsonOptions", { indent: 2, renameFields: true });
            this._setRawDefault("csvOptions", { separator: ",", addColumnNames: true, emptyAs: "", addBOM: true });
            this._setRawDefault("htmlOptions", { emptyAs: "-", addColumnNames: true });
            this._setRawDefault("xlsxOptions", { emptyAs: "", addColumnNames: true });
            this._setRawDefault("pdfOptions", { fontSize: 14, imageFormat: "png", align: "left", addURL: true });
            this._setRawDefault("pdfdataOptions", { emptyAs: "", addColumnNames: true });
            this._root.addDisposer(this);
        }
    });
    Object.defineProperty(Exporting.prototype, "_beforeChanged", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._beforeChanged.call(this);
            if (this.isDirty("menu")) {
                var menu = this.get("menu");
                if (menu) {
                    menu.set("exporting", this);
                }
            }
        }
    });
    Object.defineProperty(Exporting.prototype, "_getFormatOptions", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (format, options) {
            var newOptions = $object.copy(this.get(format + "Options", {}));
            if (options) {
                $object.each(options, function (key, value) {
                    newOptions[key] = value;
                });
            }
            return newOptions;
        }
    });
    /**
     * Triggers a download of the chart/data in specific format.
     *
     * @param  format         Format
     * @param  customOptions  Format options
     */
    Object.defineProperty(Exporting.prototype, "download", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (format, customOptions) {
            return __awaiter(this, void 0, void 0, function () {
                var ext, fileName, options, uri;
                return __generator(this, function (_a) {
                    switch (_a.label) {
                        case 0:
                            ext = format == "pdfdata" ? "pdf" : format;
                            fileName = this.get("filePrefix", "chart") + "." + ext;
                            options = this._getFormatOptions(format, customOptions);
                            this.events.dispatch("downloadstarted", {
                                type: "downloadstarted",
                                format: format,
                                options: options,
                                fileName: fileName,
                                target: this
                            });
                            return [4 /*yield*/, this.export(format, options)];
                        case 1:
                            uri = _a.sent();
                            this.streamFile(uri, fileName, (options && options.addBOM));
                            return [2 /*return*/];
                    }
                });
            });
        }
    });
    /**
     * Triggers print of the chart.
     *
     * @param  customOptions  Print options
     */
    Object.defineProperty(Exporting.prototype, "print", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (customOptions) {
            return __awaiter(this, void 0, void 0, function () {
                var options, uri;
                return __generator(this, function (_a) {
                    switch (_a.label) {
                        case 0:
                            options = this._getFormatOptions("print", customOptions);
                            this.events.dispatch("printstarted", {
                                type: "printstarted",
                                format: "print",
                                options: options,
                                target: this
                            });
                            return [4 /*yield*/, this.export(options.imageFormat || "png", options)];
                        case 1:
                            uri = _a.sent();
                            this.initiatePrint(uri, options, this.get("title"));
                            return [2 /*return*/];
                    }
                });
            });
        }
    });
    /**
     * Returns data uri of the chart/data in specific format.
     *
     * @param          format  Format
     * @param   customOptions  Format options
     * @return                 Promise
     */
    Object.defineProperty(Exporting.prototype, "export", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (format, customOptions) {
            return __awaiter(this, void 0, void 0, function () {
                var options;
                return __generator(this, function (_a) {
                    options = this._getFormatOptions(format, customOptions);
                    this.events.dispatch("exportstarted", {
                        type: "exportstarted",
                        format: format,
                        options: options,
                        target: this
                    });
                    switch (format) {
                        case "png":
                        case "jpg":
                            return [2 /*return*/, this.exportImage(format, options)];
                            break;
                        case "json":
                            return [2 /*return*/, this.exportJSON(options)];
                            break;
                        case "csv":
                            return [2 /*return*/, this.exportCSV(options)];
                            break;
                        case "html":
                            return [2 /*return*/, this.exportHTML(options)];
                            break;
                        case "xlsx":
                            return [2 /*return*/, this.exportXLSX(options)];
                            break;
                        case "pdf":
                            return [2 /*return*/, this.exportPDF(options)];
                            break;
                        case "pdfdata":
                            return [2 /*return*/, this.exportPDFData(options)];
                            break;
                    }
                    this.events.dispatch("exportfinished", {
                        type: "exportfinished",
                        format: format,
                        options: options,
                        target: this
                    });
                    return [2 /*return*/, ""];
                });
            });
        }
    });
    /**
     * ==========================================================================
     * Images
     * ==========================================================================
     */
    /**
     * Returns chart image as a data:uri.
     *
     * @param   format         Image format
     * @param   customOptions  Format options
     * @return                 Promise
     */
    Object.defineProperty(Exporting.prototype, "exportImage", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (format, customOptions) {
            return __awaiter(this, void 0, void 0, function () {
                var options;
                return __generator(this, function (_a) {
                    switch (_a.label) {
                        case 0:
                            options = this._getFormatOptions(format, customOptions);
                            return [4 /*yield*/, this.getCanvas(options)];
                        case 1: return [2 /*return*/, (_a.sent()).toDataURL(this.getContentType(format), options.quality || 1)];
                    }
                });
            });
        }
    });
    /**
     * Returns canvas data.
     *
     * @param   customOptions  Image options
     * @return                 Promise
     */
    Object.defineProperty(Exporting.prototype, "exportCanvas", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (customOptions) {
            return __awaiter(this, void 0, void 0, function () {
                var options;
                return __generator(this, function (_a) {
                    switch (_a.label) {
                        case 0:
                            options = this._getFormatOptions("canvas", customOptions);
                            return [4 /*yield*/, this.getCanvas(options)];
                        case 1: return [2 /*return*/, (_a.sent()).toDataURL(this.getContentType("canvas"), options.quality || 1)];
                    }
                });
            });
        }
    });
    /**
     * Returns a `<canvas>` element with snapshot of the chart.
     *
     * @param   options  Image options
     * @return           Promise
     */
    Object.defineProperty(Exporting.prototype, "getCanvas", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (options) {
            return __awaiter(this, void 0, void 0, function () {
                var mainCanvas, extraImages, middleLeft, middleTop, middleWidth, middleHeight, extraRight, extraBottom, extras, newCanvas, ctx, background, left, top, right, bottom;
                return __generator(this, function (_a) {
                    mainCanvas = this._root._renderer.getCanvas(this._root._rootContainer._display, options);
                    extraImages = this.get("extraImages", []);
                    middleLeft = 0;
                    middleTop = 0;
                    middleWidth = mainCanvas.width;
                    middleHeight = mainCanvas.height;
                    extraRight = 0;
                    extraBottom = 0;
                    extras = [];
                    $array.each(extraImages, function (extraRoot) {
                        // Get that extra
                        var extra;
                        if (extraRoot instanceof Root) {
                            extra = {
                                source: extraRoot,
                                position: "bottom"
                            };
                        }
                        else {
                            extra = extraRoot;
                        }
                        // Set defaults
                        extra.position = extra.position || "bottom";
                        extra.marginTop = extra.marginTop || 0;
                        extra.marginRight = extra.marginRight || 0;
                        extra.marginBottom = extra.marginBottom || 0;
                        extra.marginLeft = extra.marginLeft || 0;
                        var extraCanvas = extra.source._renderer.getCanvas(extra.source._rootContainer._display, options);
                        var extraWidth = extraCanvas.width + extra.marginLeft + extra.marginRight;
                        var extraHeight = extraCanvas.height + extra.marginTop + extra.marginBottom;
                        if (extra.position == "top") {
                            middleWidth = extra.crop ? middleHeight : Math.max(middleWidth, extraWidth);
                            middleTop += extraHeight;
                        }
                        else if (extra.position == "right") {
                            middleHeight = extra.crop ? middleHeight : Math.max(middleHeight, extraHeight);
                            extraRight += extraWidth;
                        }
                        else if (extra.position == "left") {
                            middleHeight = extra.crop ? middleHeight : Math.max(middleHeight, extraHeight);
                            middleLeft += extraWidth;
                        }
                        else if (extra.position === "bottom") {
                            middleWidth = extra.crop ? middleHeight : Math.max(middleWidth, extraWidth);
                            extraBottom += extraHeight;
                        }
                        extras.push({
                            canvas: extraCanvas,
                            position: extra.position,
                            left: extra.marginLeft,
                            top: extra.marginTop,
                            width: extraWidth,
                            height: extraHeight
                        });
                    });
                    newCanvas = this.getDisposableCanvas();
                    newCanvas.width = middleLeft + middleWidth + extraRight;
                    newCanvas.height = middleTop + middleHeight + extraBottom;
                    ctx = newCanvas.getContext("2d");
                    background = this.get("backgroundColor", this.findBackgroundColor(this._root.dom));
                    if (background) {
                        ctx.fillStyle = background.toCSS();
                        ctx.fillRect(0, 0, newCanvas.width, newCanvas.height);
                    }
                    left = middleLeft;
                    top = middleTop;
                    right = left + middleWidth;
                    bottom = top + middleHeight;
                    // Radiates outwards from center
                    $array.each(extras, function (extra) {
                        if (extra.position == "top") {
                            top -= extra.height;
                            ctx.drawImage(extra.canvas, middleLeft + extra.left, top + extra.top);
                        }
                        else if (extra.position == "right") {
                            ctx.drawImage(extra.canvas, right + extra.left, middleTop + extra.top);
                            right += extra.width;
                        }
                        else if (extra.position == "left") {
                            left -= extra.width;
                            ctx.drawImage(extra.canvas, left + extra.left, middleTop + extra.top);
                        }
                        else if (extra.position === "bottom") {
                            ctx.drawImage(extra.canvas, middleLeft + extra.left, bottom + extra.top);
                            bottom += extra.height;
                        }
                        //this.disposeCanvas(extra.canvas);
                    });
                    ctx.drawImage(mainCanvas, middleLeft, middleTop);
                    return [2 /*return*/, newCanvas];
                });
            });
        }
    });
    /**
     * ==========================================================================
     * JSON
     * ==========================================================================
     */
    /**
     * Returns a data:uri representation of a JSON file with chart data.
     *
     * @param   customOptions  Format options
     * @return                 Promise
     */
    Object.defineProperty(Exporting.prototype, "exportJSON", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (customOptions) {
            return __awaiter(this, void 0, void 0, function () {
                var _a, _b;
                return __generator(this, function (_c) {
                    switch (_c.label) {
                        case 0:
                            _a = "data:" + this.getContentType("json") + ";" + this.get("charset", "utf-8") + ",";
                            _b = encodeURIComponent;
                            return [4 /*yield*/, this.getJSON(customOptions)];
                        case 1: return [2 /*return*/, _a + _b.apply(void 0, [_c.sent()])];
                    }
                });
            });
        }
    });
    /**
     * Returns data in JSON format.
     *
     * @param   customOptions  Format options
     * @return                 Promise
     */
    Object.defineProperty(Exporting.prototype, "getJSON", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (customOptions) {
            return __awaiter(this, void 0, void 0, function () {
                var options;
                var _this = this;
                return __generator(this, function (_a) {
                    options = this._getFormatOptions("json", customOptions);
                    return [2 /*return*/, JSON.stringify(this.getData("json", customOptions, options.renameFields), function (_key, value) {
                            if ($type.isObject(value)) {
                                $object.each(value, function (field, item) {
                                    value[field] = _this.convertToSpecialFormat(field, item, options);
                                });
                            }
                            return value;
                        }, options.indent)];
                });
            });
        }
    });
    /**
     * ==========================================================================
     * CSV
     * ==========================================================================
     */
    /**
     * Returns a data:uri representation of a CSV file with chart data.
     *
     * @param   customOptions  Format options
     * @return                 Promise
     */
    Object.defineProperty(Exporting.prototype, "exportCSV", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (customOptions) {
            return __awaiter(this, void 0, void 0, function () {
                var _a, _b;
                return __generator(this, function (_c) {
                    switch (_c.label) {
                        case 0:
                            _a = "data:" + this.getContentType("csv") + ";" + this.get("charset", "utf-8") + ",";
                            _b = encodeURIComponent;
                            return [4 /*yield*/, this.getCSV(customOptions)];
                        case 1: return [2 /*return*/, _a + _b.apply(void 0, [_c.sent()])];
                    }
                });
            });
        }
    });
    /**
     * Returns a CSV with export data.
     *
     * @param   customOptions  CSV options
     * @return                 Promise
     */
    Object.defineProperty(Exporting.prototype, "getCSV", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (customOptions) {
            return __awaiter(this, void 0, void 0, function () {
                var options, csv, br, data, dataFields, dataFieldsOrder_1, len, i, row;
                var _this = this;
                return __generator(this, function (_a) {
                    options = this._getFormatOptions("csv", customOptions);
                    csv = "";
                    br = "";
                    data = this.getData("csv", options);
                    dataFields = this.getDataFields(data);
                    // Vertical or horizontal (default) layout
                    if (options.pivot) {
                        dataFieldsOrder_1 = this.get("dataFieldsOrder", []);
                        $object.eachOrdered(dataFields, function (key, val) {
                            var dataRow = [];
                            if (options.addColumnNames) {
                                dataRow.push(val);
                            }
                            for (var len = data.length, i = 0; i < len; i++) {
                                var dataValue = data[i][key];
                                dataRow.push(_this.convertToSpecialFormat(key, dataValue, options, true));
                            }
                            csv += br + _this.getCSVRow(dataRow, options, undefined, true);
                            br = "\n";
                        }, function (a, b) {
                            var ai = dataFieldsOrder_1.indexOf(a);
                            var bi = dataFieldsOrder_1.indexOf(b);
                            if (ai > bi) {
                                return -1;
                            }
                            else if (ai < bi) {
                                return 1;
                            }
                            return 0;
                        });
                    }
                    else {
                        for (len = data.length, i = 0; i < len; i++) {
                            row = this.getCSVRow(data[i], options, dataFields);
                            if (options.reverse) {
                                csv = row + br + csv;
                            }
                            else {
                                csv += br + row;
                            }
                            br = "\n";
                        }
                        // Add column names?
                        if (options.addColumnNames) {
                            csv = this.getCSVRow(dataFields, options, undefined, true) + br + csv;
                        }
                    }
                    return [2 /*return*/, csv];
                });
            });
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Exporting.prototype, "getCSVRow", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (row, options, dataFields, asIs) {
            var _this = this;
            if (asIs === void 0) { asIs = false; }
            // Init
            var separator = options.separator || ",";
            var items = [];
            // Data fields
            if (!dataFields) {
                dataFields = {};
                $object.each(row, function (key, value) {
                    dataFields[key] = value;
                });
            }
            // Data fields order
            var dataFieldsOrder = this.get("dataFieldsOrder", []);
            // Process each row item
            $object.eachOrdered(dataFields, function (key, _name) {
                // Get value
                var value = _this.convertEmptyValue(key, row[key], options);
                // Check if we need to skip
                // This is no longer required because we are iterating via dataFields anyway
                /*if (this.dataFields != null && this.dataFields[key] == null) {
                    return;
                }*/
                // Convert dates
                var item = asIs ? value : _this.convertToSpecialFormat(key, value, options);
                // Cast and escape doublequotes
                item = "" + item;
                item = item.replace(/"/g, '""');
                // Enclose into double quotes
                if (options.forceQuotes || (item.search(new RegExp("\"|\n|" + separator, "g")) >= 0)) {
                    item = "\"" + item + "\"";
                }
                // Add to item
                items.push(item);
            }, function (a, b) {
                var ai = dataFieldsOrder.indexOf(a);
                var bi = dataFieldsOrder.indexOf(b);
                if (ai > bi) {
                    return 1;
                }
                else if (ai < bi) {
                    return -1;
                }
                return 0;
            });
            return items.join(separator);
        }
    });
    /**
     * ==========================================================================
     * HTML
     * ==========================================================================
     */
    /**
     * Returns a data:uri representation of an HTML file with chart data.
     *
     * @param   customOptions  Format options
     * @return                 Promise
     */
    Object.defineProperty(Exporting.prototype, "exportHTML", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (customOptions) {
            return __awaiter(this, void 0, void 0, function () {
                var _a, _b;
                return __generator(this, function (_c) {
                    switch (_c.label) {
                        case 0:
                            _a = "data:" + this.getContentType("html") + ";" + this.get("charset", "utf-8") + ",";
                            _b = encodeURIComponent;
                            return [4 /*yield*/, this.getHTML(customOptions)];
                        case 1: return [2 /*return*/, _a + _b.apply(void 0, [_c.sent()])];
                    }
                });
            });
        }
    });
    /**
     * Returns an HTML with a table with export data.
     *
     * @param   customOptions  HTML options
     * @return                 Promise
     */
    Object.defineProperty(Exporting.prototype, "getHTML", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (customOptions) {
            return __awaiter(this, void 0, void 0, function () {
                var options, html, data, dataFields, dataFieldsOrder_2, len, i;
                var _this = this;
                return __generator(this, function (_a) {
                    options = this._getFormatOptions("html", customOptions);
                    html = "<table>";
                    if (options.tableClass) {
                        html = "<table class=\"" + options.tableClass + "\">";
                    }
                    data = this.getData("html", options);
                    dataFields = this.getDataFields(data);
                    // Vertical or horizontal (default) layout
                    if (options.pivot) {
                        dataFieldsOrder_2 = this.get("dataFieldsOrder", []);
                        html += "\n<tbody>";
                        $object.eachOrdered(dataFields, function (key, val) {
                            var dataRow = [];
                            if (options.addColumnNames) {
                                dataRow.push(val);
                            }
                            for (var len = data.length, i = 0; i < len; i++) {
                                var dataValue = data[i][key];
                                dataRow.push(_this.convertToSpecialFormat(key, dataValue, options, true));
                            }
                            html += "\n" + _this.getHTMLRow(dataRow, options, undefined, true);
                        }, function (a, b) {
                            var ai = dataFieldsOrder_2.indexOf(a);
                            var bi = dataFieldsOrder_2.indexOf(b);
                            if (ai > bi) {
                                return -1;
                            }
                            else if (ai < bi) {
                                return 1;
                            }
                            return 0;
                        });
                        html += "\n</tbody>";
                    }
                    else {
                        // Add column names?
                        if (options.addColumnNames) {
                            html += "\n<thead>\n" + this.getHTMLRow(dataFields, options, undefined, true, true) + "\n</thead>";
                        }
                        html += "\n<tbody>";
                        for (len = data.length, i = 0; i < len; i++) {
                            html += "\n" + this.getHTMLRow(data[i], options, dataFields);
                        }
                        html += "\n</tbody>";
                    }
                    html += "\n</table>";
                    return [2 /*return*/, html];
                });
            });
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Exporting.prototype, "getHTMLRow", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (row, options, dataFields, asIs, headerRow) {
            var _this = this;
            if (asIs === void 0) { asIs = false; }
            if (headerRow === void 0) { headerRow = false; }
            // Init output
            var html = "\t<tr>";
            if (options.rowClass) {
                html = "\t<tr class=\"" + options.rowClass + "\">";
            }
            // Data fields
            if (!dataFields) {
                dataFields = row;
            }
            // Data fields order
            var dataFieldsOrder = this.get("dataFieldsOrder", []);
            // th or dh?
            var tag = headerRow ? "th" : "td";
            // Process each row item
            var first = true;
            $object.eachOrdered(dataFields, function (key, _name) {
                // Get value
                var value = _this.convertEmptyValue(key, row[key], options);
                // Convert dates
                var item = asIs ? value : _this.convertToSpecialFormat(key, value, options);
                // Escape HTML entities
                item = "" + item;
                item = item.replace(/[\u00A0-\u9999<>\&]/gim, function (i) {
                    return "&#" + i.charCodeAt(0) + ";";
                });
                // Which tag to use
                var useTag = tag;
                if (options.pivot && first) {
                    useTag = "th";
                }
                // Add cell
                if (options.cellClass) {
                    html += "\n\t\t<" + useTag + " class=\"" + options.cellClass + "\">" + item + "</" + useTag + ">";
                }
                else {
                    html += "\n\t\t<" + useTag + ">" + item + "</" + useTag + ">";
                }
                first = false;
            }, function (a, b) {
                var ai = dataFieldsOrder.indexOf(a);
                var bi = dataFieldsOrder.indexOf(b);
                if (ai > bi) {
                    return 1;
                }
                else if (ai < bi) {
                    return -1;
                }
                return 0;
            });
            html += "\n\t</tr>";
            return html;
        }
    });
    /**
     * ==========================================================================
     * XLSX
     * ==========================================================================
     */
    /**
     * Returns a data:uri representation of an XLSX file with chart data.
     *
     * @param   customOptions  Format options
     * @return                 Promise
     */
    Object.defineProperty(Exporting.prototype, "exportXLSX", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (customOptions) {
            return __awaiter(this, void 0, void 0, function () {
                var _a, _b;
                return __generator(this, function (_c) {
                    switch (_c.label) {
                        case 0:
                            _a = "data:" + this.getContentType("xlsx") + ";" + this.get("charset", "utf-8") + ",";
                            _b = encodeURIComponent;
                            return [4 /*yield*/, this.getXLSX(customOptions)];
                        case 1: return [2 /*return*/, _a + _b.apply(void 0, [_c.sent()])];
                    }
                });
            });
        }
    });
    /**
     * Returns a data:uri of XLSX data.
     *
     * @param  customOptions  Format options
     * @return                Promise
     */
    Object.defineProperty(Exporting.prototype, "getXLSX", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (customOptions) {
            return __awaiter(this, void 0, void 0, function () {
                var options, XLSX, wbOptions, sheetName, wb, wsData, data, dataFields, dataFieldsOrder_3, len, i;
                var _this = this;
                return __generator(this, function (_a) {
                    switch (_a.label) {
                        case 0:
                            options = this._getFormatOptions("xlsx", customOptions);
                            return [4 /*yield*/, this.getXLSXLib()];
                        case 1:
                            XLSX = _a.sent();
                            wbOptions = {
                                bookType: "xlsx",
                                bookSST: false,
                                type: "base64",
                                //dateNF: 'yyyy-mm-dd'
                            };
                            sheetName = this._normalizeExcelSheetName(this.get("title", this._t("Data")));
                            wb = {
                                SheetNames: [sheetName],
                                Sheets: {}
                            };
                            wsData = [];
                            data = this.getData("html", options);
                            dataFields = this.getDataFields(data);
                            // Vertical or horizontal (default) layout
                            if (options.pivot) {
                                dataFieldsOrder_3 = this.get("dataFieldsOrder", []);
                                $object.eachOrdered(dataFields, function (key, val) {
                                    var dataRow = [];
                                    if (options.addColumnNames) {
                                        dataRow.push(val);
                                    }
                                    for (var len = data.length, i = 0; i < len; i++) {
                                        var dataValue = data[i][key];
                                        dataRow.push(_this.convertToSpecialFormat(key, dataValue, options, true));
                                    }
                                    wsData.push(_this.getXLSXRow(dataRow, options, undefined, true));
                                }, function (a, b) {
                                    //console.log(a, b)
                                    var ai = dataFieldsOrder_3.indexOf(a);
                                    var bi = dataFieldsOrder_3.indexOf(b);
                                    if (ai > bi) {
                                        return 1;
                                    }
                                    else if (ai < bi) {
                                        return -1;
                                    }
                                    return 0;
                                });
                            }
                            else {
                                // Add column names?
                                if (options.addColumnNames) {
                                    wsData.push(this.getXLSXRow(dataFields, options, undefined, true));
                                }
                                // Add lines
                                for (len = data.length, i = 0; i < len; i++) {
                                    wsData.push(this.getXLSXRow(data[i], options, dataFields));
                                }
                            }
                            // Create sheet and add data
                            wb.Sheets[sheetName] = XLSX.utils.aoa_to_sheet(wsData);
                            this.events.dispatch("workbookready", {
                                type: "workbookready",
                                format: "xlsx",
                                options: options,
                                workbook: wb,
                                workbookOptions: wbOptions,
                                xlsx: XLSX,
                                target: this
                            });
                            return [2 /*return*/, XLSX.write(wb, wbOptions)];
                    }
                });
            });
        }
    });
    Object.defineProperty(Exporting.prototype, "_normalizeExcelSheetName", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (name) {
            name = name.replace(/([:\\\/?*\[\]]+)/g, " ");
            return name.length > 30 ? name.substr(0, 30) + "..." : name;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Exporting.prototype, "getXLSXRow", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (row, options, dataFields, asIs) {
            var _this = this;
            if (asIs === void 0) { asIs = false; }
            // Init
            var items = [];
            // Data fields
            if (!dataFields) {
                dataFields = row;
            }
            // Data fields order
            var dataFieldsOrder = this.get("dataFieldsOrder", []);
            // Process each row item
            $object.eachOrdered(dataFields, function (key, _name) {
                // Get value
                var value = _this.convertEmptyValue(key, row[key], options);
                // Convert dates
                var item = asIs ? value : _this.convertToSpecialFormat(key, value, options, true);
                items.push(item);
            }, function (a, b) {
                //console.log(a, b)
                var ai = dataFieldsOrder.indexOf(a);
                var bi = dataFieldsOrder.indexOf(b);
                if (ai > bi) {
                    return 1;
                }
                else if (ai < bi) {
                    return -1;
                }
                return 0;
            });
            return items;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Exporting.prototype, "_xlsx", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return __awaiter(this, void 0, void 0, function () {
                return __generator(this, function (_a) {
                    switch (_a.label) {
                        case 0: return [4 /*yield*/, import(/* webpackChunkName: "xlsx" */ "xlsx/dist/xlsx.core.min.js")];
                        case 1: return [2 /*return*/, _a.sent()];
                    }
                });
            });
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Exporting.prototype, "getXLSXLib", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this._xlsx();
        }
    });
    /**
     * ==========================================================================
     * PDF
     * ==========================================================================
     */
    /**
     * Returns a data:uri representation of a PDF file with chart image.
     *
     * @param   customOptions  Format options
     * @return                 Promise
     */
    Object.defineProperty(Exporting.prototype, "exportPDF", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (customOptions) {
            return __awaiter(this, void 0, void 0, function () {
                var _a, _b;
                return __generator(this, function (_c) {
                    switch (_c.label) {
                        case 0:
                            _a = "data:" + this.getContentType("pdf") + ";" + this.get("charset", "utf-8") + ",";
                            _b = encodeURIComponent;
                            return [4 /*yield*/, this.getPDF(customOptions, true)];
                        case 1: return [2 /*return*/, _a + _b.apply(void 0, [_c.sent()])];
                    }
                });
            });
        }
    });
    /**
     * Returns a data:uri representation of a PDF file with chart data.
     *
     * @param   customOptions  Format options
     * @return                 Promise
     */
    Object.defineProperty(Exporting.prototype, "exportPDFData", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (customOptions) {
            return __awaiter(this, void 0, void 0, function () {
                var _a, _b;
                return __generator(this, function (_c) {
                    switch (_c.label) {
                        case 0:
                            _a = "data:" + this.getContentType("pdf") + ";" + this.get("charset", "utf-8") + ",";
                            _b = encodeURIComponent;
                            return [4 /*yield*/, this.getPDF(customOptions, false, true)];
                        case 1: return [2 /*return*/, _a + _b.apply(void 0, [_c.sent()])];
                    }
                });
            });
        }
    });
    /**
     * Returns Base64-encoded binary data for a PDF file.
     * @param   customOptions  PDF options
     * @param   includeImage   Include chart snapshot
     * @param   includeData    Include data
     * @return                 Promise
     */
    Object.defineProperty(Exporting.prototype, "getPDF", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (customOptions, includeImage, includeData) {
            if (includeImage === void 0) { includeImage = true; }
            if (includeData === void 0) { includeData = false; }
            return __awaiter(this, void 0, void 0, function () {
                function addFont(font) {
                    var paths = {};
                    paths.normal = font.normal.path;
                    vfs[font.normal.path] = font.normal.bytes;
                    if (font.bold) {
                        paths.bold = font.bold.path;
                        vfs[font.bold.path] = font.bold.bytes;
                    }
                    else {
                        paths.bold = font.normal.path;
                    }
                    if (font.italics) {
                        paths.italics = font.italics.path;
                        vfs[font.italics.path] = font.italics.bytes;
                    }
                    else {
                        paths.italics = font.normal.path;
                    }
                    if (font.bolditalics) {
                        paths.bolditalics = font.bolditalics.path;
                        vfs[font.bolditalics.path] = font.bolditalics.bytes;
                    }
                    else {
                        paths.bolditalics = font.normal.path;
                    }
                    fonts[font.name] = paths;
                }
                var options, dataOptions, image, pdfmake, defaultMargins, doc, title, extraMargin, _a, _b, fonts, vfs;
                var _c;
                return __generator(this, function (_d) {
                    switch (_d.label) {
                        case 0:
                            options = this._getFormatOptions("pdf", customOptions);
                            dataOptions = this._getFormatOptions("pdfdata", customOptions);
                            if (!includeImage) return [3 /*break*/, 2];
                            return [4 /*yield*/, this.exportImage(options.imageFormat || "png", options)];
                        case 1:
                            image = _d.sent();
                            _d.label = 2;
                        case 2: return [4 /*yield*/, this.getPdfmake()];
                        case 3:
                            pdfmake = _d.sent();
                            defaultMargins = [30, 30, 30, 30];
                            doc = {
                                pageSize: options.pageSize || "A4",
                                pageOrientation: options.pageOrientation || "portrait",
                                pageMargins: options.pageMargins || defaultMargins,
                                defaultStyle: {
                                    font: options.font ? options.font.name : undefined
                                },
                                //header: <any>[],
                                content: []
                            };
                            title = this.get("title");
                            extraMargin = 0;
                            if (title) {
                                doc.content.push({
                                    text: title,
                                    fontSize: options.fontSize || 14,
                                    bold: true,
                                    margin: [0, 0, 0, 15]
                                });
                                // Add some leftover margin for title
                                extraMargin += 50;
                            }
                            // Add page URL?
                            if (options.addURL) {
                                doc.content.push({
                                    text: this._t("Saved from") + ": " + document.location.href,
                                    fontSize: options.fontSize,
                                    margin: [0, 0, 0, 15]
                                });
                                // Add some leftover margin for URL
                                extraMargin += 50;
                            }
                            // Add image
                            if (includeImage && image) {
                                doc.content.push({
                                    image: image,
                                    alignment: options.align || "left",
                                    fit: this.getPageSizeFit(doc.pageSize, doc.pageMargins, extraMargin)
                                });
                            }
                            if (!((includeData || options.includeData) && this.hasData())) return [3 /*break*/, 5];
                            _b = (_a = doc.content).push;
                            _c = {};
                            return [4 /*yield*/, this.getPDFData(dataOptions)];
                        case 4:
                            _b.apply(_a, [(_c.table = _d.sent(),
                                    _c.fontSize = options.fontSize || 14,
                                    _c)]);
                            _d.label = 5;
                        case 5:
                            fonts = null;
                            vfs = null;
                            if (options.font) {
                                fonts = {};
                                vfs = {};
                                addFont(options.font);
                                if (options.extraFonts) {
                                    $array.each(options.extraFonts, addFont);
                                }
                            }
                            this.events.dispatch("pdfdocready", {
                                type: "pdfdocready",
                                format: "pdf",
                                options: options,
                                doc: doc,
                                target: this
                            });
                            // Create PDF
                            return [2 /*return*/, new Promise(function (success, _error) {
                                    pdfmake.createPdf(doc, null, fonts, vfs).getBase64(function (uri) {
                                        success(uri);
                                    });
                                })];
                    }
                });
            });
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Exporting.prototype, "getPDFData", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (customOptions) {
            return __awaiter(this, void 0, void 0, function () {
                var options, content, data, dataFields, dataFieldsOrder_4, len, i;
                var _this = this;
                return __generator(this, function (_a) {
                    options = this._getFormatOptions("pdfdata", customOptions);
                    content = {
                        "body": []
                    };
                    data = this.getData("html", options);
                    dataFields = this.getDataFields(data);
                    // Vertical or horizontal (default) layout
                    if (options.pivot) {
                        dataFieldsOrder_4 = this.get("dataFieldsOrder", []);
                        $object.eachOrdered(dataFields, function (key, val) {
                            var dataRow = [];
                            if (options.addColumnNames) {
                                dataRow.push(val);
                            }
                            for (var len = data.length, i = 0; i < len; i++) {
                                var dataValue = data[i][key];
                                dataRow.push(_this.convertToSpecialFormat(key, dataValue, options, true));
                            }
                            content.body.push(_this.getPDFDataRow(dataRow, options, undefined, true));
                        }, function (a, b) {
                            //console.log(a, b)
                            var ai = dataFieldsOrder_4.indexOf(a);
                            var bi = dataFieldsOrder_4.indexOf(b);
                            if (ai > bi) {
                                return 1;
                            }
                            else if (ai < bi) {
                                return -1;
                            }
                            return 0;
                        });
                    }
                    else {
                        // Add column names?
                        if (options.addColumnNames) {
                            content.body.push(this.getPDFDataRow(dataFields, options, undefined, true));
                            content.headerRows = 1;
                        }
                        for (len = data.length, i = 0; i < len; i++) {
                            content.body.push(this.getPDFDataRow(data[i], options, dataFields));
                        }
                    }
                    return [2 /*return*/, content];
                });
            });
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Exporting.prototype, "getPDFDataRow", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (row, options, dataFields, asIs) {
            var _this = this;
            if (asIs === void 0) { asIs = false; }
            // Init
            var items = [];
            // Data fields
            if (!dataFields) {
                dataFields = row;
            }
            // Data fields order
            var dataFieldsOrder = this.get("dataFieldsOrder", []);
            // Process each row item
            $object.eachOrdered(dataFields, function (key, _name) {
                // Get value
                var value = _this.convertEmptyValue(key, row[key], options);
                // Convert dates
                var item = asIs ? value : _this.convertToSpecialFormat(key, value, options);
                item = "" + item;
                // Add to item
                items.push(item);
            }, function (a, b) {
                //console.log(a, b)
                var ai = dataFieldsOrder.indexOf(a);
                var bi = dataFieldsOrder.indexOf(b);
                if (ai > bi) {
                    return 1;
                }
                else if (ai < bi) {
                    return -1;
                }
                return 0;
            });
            return items;
        }
    });
    /**
     * Returns pdfmake instance.
     *
     * @ignore
     * @return Instance of pdfmake
     */
    Object.defineProperty(Exporting.prototype, "getPdfmake", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (pdfmakePromise === undefined) {
                pdfmakePromise = _pdfmake();
            }
            return pdfmakePromise;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Exporting.prototype, "getPageSizeFit", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (pageSize, margins, extraMargin) {
            if (extraMargin === void 0) { extraMargin = 0; }
            // Check margins
            var newMargins = [0, 0, 0, 0];
            if ($type.isNumber(margins)) {
                newMargins = [margins, margins, margins, margins];
            }
            else if (margins.length == 2) {
                newMargins = [margins[0], margins[1], margins[0], margins[1]];
            }
            else if (margins.length == 4) {
                newMargins = margins;
            }
            // Define available page sizes
            var sizes = {
                "4A0": [4767.87, 6740.79],
                "2A0": [3370.39, 4767.87],
                A0: [2383.94, 3370.39],
                A1: [1683.78, 2383.94],
                A2: [1190.55, 1683.78],
                A3: [841.89, 1190.55],
                A4: [595.28, 841.89],
                A5: [419.53, 595.28],
                A6: [297.64, 419.53],
                A7: [209.76, 297.64],
                A8: [147.40, 209.76],
                A9: [104.88, 147.40],
                A10: [73.70, 104.88],
                B0: [2834.65, 4008.19],
                B1: [2004.09, 2834.65],
                B2: [1417.32, 2004.09],
                B3: [1000.63, 1417.32],
                B4: [708.66, 1000.63],
                B5: [498.90, 708.66],
                B6: [354.33, 498.90],
                B7: [249.45, 354.33],
                B8: [175.75, 249.45],
                B9: [124.72, 175.75],
                B10: [87.87, 124.72],
                C0: [2599.37, 3676.54],
                C1: [1836.85, 2599.37],
                C2: [1298.27, 1836.85],
                C3: [918.43, 1298.27],
                C4: [649.13, 918.43],
                C5: [459.21, 649.13],
                C6: [323.15, 459.21],
                C7: [229.61, 323.15],
                C8: [161.57, 229.61],
                C9: [113.39, 161.57],
                C10: [79.37, 113.39],
                RA0: [2437.80, 3458.27],
                RA1: [1729.13, 2437.80],
                RA2: [1218.90, 1729.13],
                RA3: [864.57, 1218.90],
                RA4: [609.45, 864.57],
                SRA0: [2551.18, 3628.35],
                SRA1: [1814.17, 2551.18],
                SRA2: [1275.59, 1814.17],
                SRA3: [907.09, 1275.59],
                SRA4: [637.80, 907.09],
                EXECUTIVE: [521.86, 756.00],
                FOLIO: [612.00, 936.00],
                LEGAL: [612.00, 1008.00],
                LETTER: [612.00, 792.00],
                TABLOID: [792.00, 1224.00]
            };
            // Calculate size
            var fitSize = sizes[pageSize];
            fitSize[0] -= newMargins[0] + newMargins[2];
            fitSize[1] -= newMargins[1] + newMargins[3] + extraMargin;
            return fitSize;
        }
    });
    /**
     * ==========================================================================
     * Data
     * ==========================================================================
     */
    /**
        * Returns `true` if `dataSource` is set, and the contents are proper
        * data (array).
        *
        * @return Has data?
        */
    Object.defineProperty(Exporting.prototype, "hasData", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var dataSource = this.get("dataSource");
            return $type.isArray(dataSource) && dataSource.length ? true : false;
        }
    });
    /**
     * Returns processed data according to format options.
     *
     * @param   format         Format
     * @param   customOptions  Format options
     * @param   renameFields   Should fields be renamed?
     * @return                 Processed data
     */
    Object.defineProperty(Exporting.prototype, "getData", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (format, customOptions, renameFields) {
            var _this = this;
            if (renameFields === void 0) { renameFields = false; }
            var options = this._getFormatOptions(format, customOptions);
            var dataSource = this.get("dataSource", []);
            var data = dataSource;
            // Re-generate the data based on data fields if set
            var dataFields = this.get("dataFields");
            if (dataFields && $type.isArray(dataSource)) {
                data = [];
                $array.each(dataSource, function (row) {
                    if ($type.isObject(row)) {
                        var newRow_1 = {};
                        $object.each(dataFields, function (field, value) {
                            if (dataFields[field] != null) {
                                newRow_1[renameFields ? value : field] = _this.convertToSpecialFormat(field, row[field], options);
                            }
                        });
                        data.push(newRow_1);
                    }
                });
            }
            var event = {
                type: "dataprocessed",
                format: format,
                options: options,
                data: data,
                target: this
            };
            this.events.dispatch("dataprocessed", event);
            return event.data;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Exporting.prototype, "getDataFields", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (data) {
            var dataFields = this.get("dataFields");
            if (!dataFields) {
                dataFields = {};
                if ($type.isArray(data) && data.length) {
                    $array.each(data, function (row) {
                        $object.each(row, function (key, _value) {
                            if (dataFields[key] == null) {
                                dataFields[key] = key;
                            }
                        });
                    });
                }
            }
            return dataFields;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Exporting.prototype, "convertEmptyValue", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_field, value, options) {
            return value != null ? value : options.emptyAs;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Exporting.prototype, "convertToSpecialFormat", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (field, value, options, keepOriginal) {
            // Is this a timestamp or duration?
            if (typeof value == "number") {
                if (this.isDateField(field)) {
                    value = new Date(value);
                }
                else if (this.isNumericField(field) && this.get("numberFormat")) {
                    return this._root.numberFormatter.format(value, this.get("numberFormat"));
                }
                else if (this.isDurationField(field)) {
                    return this._root.durationFormatter.format(value, this.get("durationFormat"), this.get("durationUnit"));
                }
            }
            if (value instanceof Date) {
                if (options.useTimestamps) {
                    value = value.getTime();
                }
                else if (options.useLocale) {
                    if (!keepOriginal) {
                        value = value.toLocaleString();
                    }
                }
                else {
                    value = this._root.dateFormatter.format(value, this.get("dateFormat"));
                }
            }
            return value;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Exporting.prototype, "isDateField", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (field) {
            return this.get("dateFields").indexOf(field) !== -1;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Exporting.prototype, "isNumericField", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (field) {
            return this.get("numericFields").indexOf(field) !== -1;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Exporting.prototype, "isDurationField", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (field) {
            return this.get("durationFields").indexOf(field) !== -1;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Exporting.prototype, "getContentType", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (type) {
            var contentType = "";
            switch (type) {
                case "png":
                    contentType = "image/" + type;
                    break;
                case "jpg":
                    contentType = "image/jpeg";
                    break;
                case "csv":
                    contentType = "text/csv";
                    break;
                case "json":
                    contentType = "application/json";
                    break;
                case "html":
                    contentType = "text/html";
                    break;
                case "pdf":
                case "pdfdata":
                    contentType = "application/pdf";
                    break;
                case "xlsx":
                    contentType = "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet";
                    break;
                default:
                    contentType = "application/octet-stream";
            }
            return contentType;
        }
    });
    Object.defineProperty(Exporting.prototype, "getDisposableCanvas", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var canvas = document.createElement("canvas");
            canvas.style.position = "fixed";
            canvas.style.top = "-10000px";
            document.body.appendChild(canvas);
            return canvas;
        }
    });
    Object.defineProperty(Exporting.prototype, "disposeCanvas", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (canvas) {
            document.body.removeChild(canvas);
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Exporting.prototype, "findBackgroundColor", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (element) {
            // Check if element has styles set
            var opacity = 1;
            var currentColor = getComputedStyle(element, "background-color").getPropertyValue("background-color");
            // Check opacity
            if (currentColor.match(/[^,]*,[^,]*,[^,]*,[ ]?0/) || currentColor == "transparent") {
                opacity = 0;
            }
            if (opacity == 0) {
                var parent_1 = element.parentElement; // || <Element>element.parentNode;
                // Completely transparent. Look for a parent
                if (parent_1) {
                    return this.findBackgroundColor(parent_1);
                }
                else {
                    return Color.fromHex(0xffffff);
                }
            }
            else {
                return Color.fromCSS(currentColor);
            }
        }
    });
    /**
     * Triggers download of the file.
     *
     * @param   uri       data:uri with file content
     * @param   fileName  File name
     * @param   addBOM    Should download include byte order mark?
     * @return            Promise
     */
    Object.defineProperty(Exporting.prototype, "streamFile", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (uri, fileName, addBOM) {
            if (addBOM === void 0) { addBOM = false; }
            if (this.msBlobDownloadSupport()) {
                /**
                 * For all IEs 10 and up we use native method `msSaveBlob`
                 */
                // Extract content type and get pure data without headers
                var parts = uri.split(";");
                var contentType = parts.shift().replace(/data:/, "");
                uri = decodeURIComponent(parts.join(";").replace(/^[^,]*,/, ""));
                // Check if we need to Base64-decode
                if (["image/svg+xml", "application/json", "text/csv"].indexOf(contentType) == -1) {
                    try {
                        var decoded = atob(uri);
                        uri = decoded;
                    }
                    catch (e) {
                        // Error occurred, meaning string was not Base64-encoded. Do nothing.
                        return false;
                    }
                }
                else {
                    var blob_1 = new Blob([uri], { type: contentType });
                    window.navigator.msSaveBlob(blob_1, fileName);
                    return true;
                }
                // Dissect uri into array
                var chars = new Array(uri.length);
                for (var i = 0; i < uri.length; ++i) {
                    var charCode = uri.charCodeAt(i);
                    chars[i] = charCode;
                }
                // Prep Blob and force the download
                var blob = new Blob([new Uint8Array(chars)], { type: contentType });
                window.navigator.msSaveBlob(blob, fileName);
            }
            else if (this.blobDownloadSupport()) {
                /**
                 * Supports Blob object.
                 * Use it.
                 */
                var link_1 = document.createElement("a");
                link_1.download = fileName;
                document.body.appendChild(link_1);
                // Extract content type and get pure data without headers
                var parts = uri.split(";");
                var contentType = parts.shift().replace(/data:/, "");
                uri = decodeURIComponent(parts.join(";").replace(/^[^,]*,/, ""));
                if (["image/svg+xml", "application/json", "text/csv", "text/html"].indexOf(contentType) == -1) {
                    try {
                        var decoded = atob(uri);
                        uri = decoded;
                    }
                    catch (e) {
                        // Error occurred, meaning string was not Base64-encoded. Do nothing.
                        return false;
                    }
                }
                else {
                    if (addBOM) {
                        uri = "\ufeff" + uri;
                    }
                    var blob_2 = new Blob([uri], { type: contentType });
                    var url_1 = window.URL.createObjectURL(blob_2);
                    link_1.href = url_1;
                    link_1.download = fileName;
                    link_1.click();
                    setTimeout(function () {
                        document.body.removeChild(link_1);
                        window.URL.revokeObjectURL(url_1);
                    }, 100);
                    return true;
                }
                // Dissect uri into array
                var chars = new Array(uri.length);
                for (var i = 0; i < uri.length; ++i) {
                    var charCode = uri.charCodeAt(i);
                    chars[i] = charCode;
                }
                if (addBOM) {
                    chars = [0xEF, 0xBB, 0xBF].concat(chars);
                }
                var blob = new Blob([new Uint8Array(chars)], { type: contentType });
                var url_2 = window.URL.createObjectURL(blob);
                link_1.href = url_2;
                link_1.download = fileName;
                document.body.appendChild(link_1);
                link_1.click();
                document.body.removeChild(link_1);
                setTimeout(function () {
                    window.URL.revokeObjectURL(url_2);
                }, 100);
            }
            else if (this.linkDownloadSupport()) {
                /**
                 * For regular browsers, we create a link then simulate a click on it
                 */
                var link = document.createElement("a");
                link.download = fileName;
                link.href = uri;
                document.body.appendChild(link);
                link.click();
                document.body.removeChild(link);
            }
            else {
                /**
                 * Something else - perhaps a mobile.
                 * Let's just display it in the same page.
                 * (hey we don't like it either)
                 */
                window.location.href = uri;
            }
            return true;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Exporting.prototype, "downloadSupport", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this.linkDownloadSupport() || this.msBlobDownloadSupport();
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Exporting.prototype, "linkDownloadSupport", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var a = document.createElement("a");
            var res = typeof a.download !== "undefined";
            return res;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Exporting.prototype, "blobDownloadSupport", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return window.Blob != null;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Exporting.prototype, "msBlobDownloadSupport", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return window.navigator.msSaveOrOpenBlob != null;
        }
    });
    /**
     * ==========================================================================
     * Print
     * ==========================================================================
     */
    /**
     * Initiates print of the chart.
     *
     * @param   data     data:uri for the image
     * @param   options  Options
     * @param   title    Optional title to use (uses window's title by default)
     * @return           Promise
     */
    Object.defineProperty(Exporting.prototype, "initiatePrint", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (data, customOptions, title) {
            var options = this._getFormatOptions("print", customOptions);
            if (options.printMethod == "css") {
                this._printViaCSS(data, options, title);
            }
            else {
                this._printViaIframe(data, options, title);
            }
        }
    });
    Object.defineProperty(Exporting.prototype, "_printViaCSS", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (data, customOptions, title) {
            var options = this._getFormatOptions("print", customOptions);
            var delay = options.delay || 500;
            //Save current scroll position
            var scroll = document.documentElement.scrollTop || document.body.scrollTop;
            // Hide all document nodes by applyin custom CSS
            var rule = new StyleRule($utils.getShadowRoot(this._root.dom), "body > *", {
                "display": "none",
                "position": "fixed",
                "visibility": "hidden",
                "opacity": "0",
                "clipPath": "polygon(0px 0px,0px 0px,0px 0px,0px 0px);"
            }, this._root.nonce);
            // Replace title?
            var originalTitle;
            if (title && document && document.title) {
                originalTitle = document.title;
                document.title = title;
            }
            // Create and add exported image
            var img = new Image();
            img.src = data;
            img.style.maxWidth = "100%";
            img.style.display = "block";
            img.style.position = "relative";
            img.style.visibility = "visible";
            img.style.opacity = "1";
            img.style.clipPath = "none";
            document.body.appendChild(img);
            // Print
            this.setTimeout(function () {
                window.print();
            }, 50);
            // Delay needs to be at least a second for iOS devices
            var isIOS = /iPad|iPhone|iPod/.test(navigator.userAgent) && !window.MSStream;
            if (isIOS && (delay < 1000)) {
                delay = 1000;
            }
            else if (delay < 100) {
                delay = 100;
            }
            // Delay function that resets back the document the way ot was before
            this.setTimeout(function () {
                // Remove image
                document.body.removeChild(img);
                // Reset back all elements
                /*for (let len = items.length, i = 0; i < len; i++) {
                    let item = <HTMLElement>items[i];
                    if ($dom.isElement(item)) {
                        item.style.display = states[i];
                    }
                }*/
                rule.dispose();
                // Restore title
                if (originalTitle) {
                    document.title = document.title;
                }
                // Scroll back the document the way it was before
                document.documentElement.scrollTop = document.body.scrollTop = scroll;
            }, delay || 500);
        }
    });
    Object.defineProperty(Exporting.prototype, "_printViaIframe", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (data, customOptions, title) {
            var options = this._getFormatOptions("print", customOptions);
            var delay = options.delay || 500;
            // Create an iframe
            var iframe = document.createElement("iframe");
            iframe.style.visibility = "hidden";
            document.body.appendChild(iframe);
            // This is needed for FireFox
            iframe.contentWindow.document.open();
            iframe.contentWindow.document.close();
            // Create and add exported image
            var img = new Image();
            img.src = data;
            img.style.maxWidth = "100%";
            img.style.height = "auto";
            if (title) {
                iframe.contentWindow.document.title = title;
            }
            iframe.contentWindow.document.body.appendChild(img);
            iframe.load = function () {
                iframe.contentWindow.document.body.appendChild(img);
            };
            // Print
            this.setTimeout(function () {
                try {
                    if (!iframe.contentWindow.document.execCommand("print", false, null)) {
                        iframe.contentWindow.print();
                    }
                }
                catch (e) {
                    iframe.contentWindow.print();
                }
            }, delay || 50);
            // Delay needs to be at least a second for iOS devices
            var isIOS = /iPad|iPhone|iPod/.test(navigator.userAgent) && !window.MSStream;
            if (isIOS && (delay < 1000)) {
                delay = 1000;
            }
            else if (delay < 100) {
                delay = 100;
            }
            // Delay function that resets back the document the way ot was before
            this.setTimeout(function () {
                // Remove image
                document.body.removeChild(iframe);
            }, delay + 50 || 100);
        }
    });
    /**
     * Returns a list of formats that can be exported in current browser.
     *
     * @return Formats
     */
    Object.defineProperty(Exporting.prototype, "supportedFormats", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            var res = [];
            var hasData = this.hasData();
            var downloadSupport = this.downloadSupport();
            $array.each(["png", "jpg", "canvas", "pdf", "xlsx", "csv", "json", "html", "pdfdata", "print"], function (format) {
                var options = _this._getFormatOptions(format);
                if (options.disabled !== true) {
                    if (["xlsx", "csv", "json", "html", "pdfdata"].indexOf(format) == -1 || (hasData && downloadSupport)) {
                        res.push(format);
                    }
                }
            });
            return res;
        }
    });
    /**
     * Returns a list of supported export types: image or print.
     *
     * @return Supported types
     */
    Object.defineProperty(Exporting.prototype, "supportedExportTypes", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var res = ["image", "print"];
            if (this.downloadSupport() && this.hasData()) {
                res.push("data");
            }
            return res;
        }
    });
    Object.defineProperty(Exporting, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "Exporting"
    });
    Object.defineProperty(Exporting, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: Entity.classNames.concat([Exporting.className])
    });
    return Exporting;
}(Entity));
export { Exporting };
//# sourceMappingURL=Exporting.js.map