import { __extends } from "tslib";
import { Button } from "../../core/render/Button";
import { Graphics } from "../../core/render/Graphics";
import { Entity } from "../../core/util/Entity";
import * as $array from "../../core/util/Array";
/**
 * A plugin that can be used to automatically group small slices on percent
 * charts into a single slice.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/percent-charts/grouping-slices/} for more info
 */
var SliceGrouper = /** @class */ (function (_super) {
    __extends(SliceGrouper, _super);
    function SliceGrouper() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        /**
         * A button that is shown when chart small buttons are visible.
         */
        Object.defineProperty(_this, "zoomOutButton", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        return _this;
    }
    Object.defineProperty(SliceGrouper.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._afterNew.call(this);
            this._setRawDefault("threshold", 5);
            this._setRawDefault("groupName", "Other");
            this._setRawDefault("clickBehavior", "none");
            this.initZoomButton();
            this._root.addDisposer(this);
        }
    });
    Object.defineProperty(SliceGrouper.prototype, "initZoomButton", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            var clickBehavior = this.get("clickBehavior");
            if (clickBehavior !== "none") {
                var container = this.root.tooltipContainer;
                this.zoomOutButton = container.children.push(Button.new(this._root, {
                    themeTags: ["zoom"],
                    icon: Graphics.new(this._root, {
                        themeTags: ["button", "icon"]
                    })
                }));
                this.zoomOutButton.hide();
                this.zoomOutButton.events.on("click", function () {
                    _this.zoomOut();
                });
            }
        }
    });
    Object.defineProperty(SliceGrouper.prototype, "handleData", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            var series = this.get("series");
            if (series) {
                // Create group data item if not yet available
                var groupDataItem_1 = this.getPrivate("groupDataItem");
                if (!groupDataItem_1) {
                    var legend = this.get("legend");
                    var categoryField = series.get("categoryField", "category");
                    var valueField = series.get("valueField", "value");
                    // Add slice
                    var groupSliceData = {};
                    groupSliceData[categoryField] = this.get("groupName", "");
                    groupSliceData[valueField] = 0;
                    series.data.push(groupSliceData);
                    groupDataItem_1 = series.dataItems[series.dataItems.length - 1];
                    groupDataItem_1.get("slice").events.on("click", function () {
                        _this.handleClick();
                    });
                    this.setPrivate("groupDataItem", groupDataItem_1);
                    // Add to legend
                    if (legend) {
                        legend.data.push(groupDataItem_1);
                        //const legendDataItem = groupDataItem.get("legendDataItem");
                        groupDataItem_1.on("visible", function (visible) {
                            if (visible) {
                                _this.zoomOut();
                            }
                        });
                    }
                }
                // Recalculate group value and decorate small slices as necessary
                var threshold_1 = this.get("threshold", 0);
                var limit_1 = this.get("limit", 1000);
                var normalDataItems_1 = [];
                var smallDataItems_1 = [];
                var groupValue_1 = 0;
                if (threshold_1 || limit_1) {
                    $array.each(series.dataItems, function (item, index) {
                        var legendDataItem = item.get("legendDataItem");
                        if (((item.get("valuePercentTotal") <= threshold_1) || (index > (limit_1 - 1))) && groupDataItem_1 !== item) {
                            groupValue_1 += item.get("value", 0);
                            smallDataItems_1.push(item);
                            item.hide(0);
                            if (legendDataItem) {
                                legendDataItem.get("itemContainer").hide(0);
                            }
                        }
                        else {
                            normalDataItems_1.push(item);
                            if (legendDataItem) {
                                legendDataItem.get("itemContainer").show(0);
                            }
                        }
                    });
                    this.setPrivate("normalDataItems", normalDataItems_1);
                    this.setPrivate("smallDataItems", smallDataItems_1);
                    this.updateGroupDataItem(groupValue_1);
                }
            }
        }
    });
    /**
     * Resets slice setup to original grouping state.
     */
    Object.defineProperty(SliceGrouper.prototype, "zoomOut", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var groupDataItem = this.getPrivate("groupDataItem");
            if (groupDataItem) {
                groupDataItem.show();
            }
            var clickBehavior = this.get("clickBehavior");
            if (clickBehavior == "zoom") {
                var normalDataItems = this.getPrivate("normalDataItems", []);
                $array.each(normalDataItems, function (item, _index) {
                    item.show();
                });
            }
            var smallDataItems = this.getPrivate("smallDataItems", []);
            $array.each(smallDataItems, function (item, _index) {
                item.hide();
            });
            if (this.zoomOutButton) {
                this.zoomOutButton.hide();
            }
        }
    });
    Object.defineProperty(SliceGrouper.prototype, "updateGroupDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (groupValue) {
            var series = this.get("series");
            if (series) {
                var groupSliceData = {};
                var categoryField = series.get("categoryField", "category");
                var valueField = series.get("valueField", "value");
                groupSliceData[categoryField] = this.get("groupName", "");
                groupSliceData[valueField] = groupValue;
                series.data.setIndex(series.data.length - 1, groupSliceData);
                var groupDataItem = this.getPrivate("groupDataItem");
                if (groupValue == 0) {
                    groupDataItem.hide(0);
                }
                else if (groupDataItem.isHidden()) {
                    groupDataItem.show();
                }
                var clickBehavior = this.get("clickBehavior");
                if (clickBehavior != "none") {
                    groupDataItem.get("slice").set("toggleKey", "none");
                }
            }
        }
    });
    Object.defineProperty(SliceGrouper.prototype, "handleClick", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var clickBehavior = this.get("clickBehavior");
            var smallDataItems = this.getPrivate("smallDataItems");
            if (clickBehavior == "none" || (smallDataItems && smallDataItems.length == 0)) {
                return;
            }
            var series = this.get("series");
            var groupDataItem = this.getPrivate("groupDataItem");
            // Hide group slice
            groupDataItem.hide();
            // Reveal small slices
            $array.each(series.dataItems, function (item) {
                if (smallDataItems.indexOf(item) !== -1) {
                    item.show();
                }
                else if (clickBehavior == "zoom") {
                    item.hide();
                }
            });
            this.zoomOutButton.show();
        }
    });
    Object.defineProperty(SliceGrouper.prototype, "_beforeChanged", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            _super.prototype._beforeChanged.call(this);
            if (this.isDirty("series")) {
                var series = this.get("series");
                if (series) {
                    series.events.on("datavalidated", function (_ev) {
                        _this.handleData();
                    });
                }
            }
        }
    });
    Object.defineProperty(SliceGrouper, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "SliceGrouper"
    });
    Object.defineProperty(SliceGrouper, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: Entity.classNames.concat([SliceGrouper.className])
    });
    return SliceGrouper;
}(Entity));
export { SliceGrouper };
//# sourceMappingURL=SliceGrouper.js.map