import { __extends } from "tslib";
import { Flow } from "./Flow";
import { chord, ribbon } from "d3-chord";
import { ascending, descending } from "d3";
import { Template } from "../../core/util/Template";
import { ListTemplate } from "../../core/util/List";
import { ChordNodes } from "./ChordNodes";
import { ChordLink } from "./ChordLink";
import { p100, p50 } from "../../core/util/Percent";
import * as $array from "../../core/util/Array";
import * as $utils from "../../core/util/Utils";
import * as $math from "../../core/util/Math";
/**
 * Regular chord series.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/flow-charts/} for more information
 * @important
 */
var Chord = /** @class */ (function (_super) {
    __extends(Chord, _super);
    function Chord() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        /**
         * List of link elements.
         *
         * @default new ListTemplate<ChordLink>
         */
        Object.defineProperty(_this, "links", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new ListTemplate(Template.new({}), function () { return ChordLink._new(_this._root, { themeTags: ["link", "shape"] }, [_this.links.template]); })
        });
        /**
         * A series for all chord nodes.
         *
         * @default ChordNodes.new()
         */
        Object.defineProperty(_this, "nodes", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.children.push(ChordNodes.new(_this._root, {}))
        });
        Object.defineProperty(_this, "_d3chord", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: chord()
        });
        Object.defineProperty(_this, "_chordLayout", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        Object.defineProperty(_this, "_ribbon", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: ribbon()
        });
        return _this;
    }
    Object.defineProperty(Chord.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._settings.themeTags = $utils.mergeTags(this._settings.themeTags, ["chord"]);
            this.linksContainer.setAll({ x: p50, y: p50 });
            this.bulletsContainer.setAll({ x: p50, y: p50 });
            _super.prototype._afterNew.call(this);
        }
    });
    Object.defineProperty(Chord.prototype, "_fixRibbon", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (ribbon) {
            var _this = this;
            ribbon.startAngle(function (d) {
                return d.startAngle + _this.get("startAngle", 0) * $math.RADIANS + Math.PI / 2;
            });
            ribbon.endAngle(function (d) {
                return d.endAngle + _this.get("startAngle", 0) * $math.RADIANS + Math.PI / 2;
            });
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Chord.prototype, "makeLink", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var link = this.linksContainer.children.push(this.links.make());
            this.links.push(link);
            link._setDataItem(dataItem);
            link.set("source", dataItem.get("source"));
            link.set("target", dataItem.get("target"));
            link.series = this;
            return link;
        }
    });
    Object.defineProperty(Chord.prototype, "_makeMatrix", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            var matrix = [];
            $array.each(this.nodes.dataItems, function (sourceDataItem) {
                var group = [];
                matrix.push(group);
                var outgoing = sourceDataItem.get("outgoingLinks");
                $array.each(_this.nodes.dataItems, function (targetDataItem) {
                    var value = 0;
                    if (outgoing) {
                        $array.each(outgoing, function (outgoingLink) {
                            if (outgoingLink.get("target") === targetDataItem) {
                                value = outgoingLink.get("valueWorking");
                            }
                        });
                    }
                    group.push(value);
                });
            });
            return matrix;
        }
    });
    Object.defineProperty(Chord.prototype, "_prepareChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            _super.prototype._prepareChildren.call(this);
            this._fixRibbon(this._ribbon);
            var chordChanged = false;
            if (this._valuesDirty || this._sizeDirty || this.isDirty("padAngle") || this.isDirty("startAngle")) {
                var matrix = this._makeMatrix();
                this._d3chord.padAngle(this.get("padAngle", 0) * $math.RADIANS);
                var sort = this.get("sort");
                if (sort === "ascending") {
                    this._d3chord.sortGroups(ascending);
                }
                else if (sort === "descending") {
                    this._d3chord.sortGroups(descending);
                }
                /*
                            this._d3chord.sortSubgroups((a, b)=>{
                                if (a != Math.round(a)) return false
                                if (b != Math.round(b)) return true
                                return b < a ? -1 : b > a ? 1 : 0;
                            })
                */
                this._chordLayout = this._d3chord(matrix);
                chordChanged = true;
            }
            if (chordChanged || this.isDirty("radius") || this.isDirty("nodeWidth")) {
                var radius_1 = $utils.relativeToValue(this.get("radius", 0), Math.min(this.innerWidth(), this.innerHeight())) / 2;
                var i_1 = 0;
                var chordStartAngle_1 = this.get("startAngle", 0);
                var nodeWidth_1 = this.get("nodeWidth", 0);
                $array.each(this.nodes.dataItems, function (dataItem) {
                    var slice = dataItem.get("slice");
                    var chordGroup = _this._chordLayout.groups[i_1];
                    var startAngle = chordGroup.startAngle * $math.DEGREES + chordStartAngle_1;
                    var endAngle = chordGroup.endAngle * $math.DEGREES + chordStartAngle_1;
                    slice.setAll({ radius: radius_1, innerRadius: radius_1 - nodeWidth_1, startAngle: startAngle, arc: endAngle - startAngle });
                    var label = dataItem.get("label");
                    label.setAll({ labelAngle: startAngle + (endAngle - startAngle) / 2 });
                    label.setPrivate("radius", radius_1);
                    label.setPrivate("innerRadius", 0.1);
                    i_1++;
                });
                var linkRadius_1 = radius_1 - this.get("nodeWidth", 0);
                $array.each(this._chordLayout, function (chord) {
                    var dataItem = _this._linksByIndex[chord.source.index + "_" + chord.target.index];
                    if (!dataItem) {
                        dataItem = _this._linksByIndex[chord.target.index + "_" + chord.source.index];
                    }
                    if (dataItem) {
                        var link = dataItem.get("link");
                        _this._getLinkPoints(link, linkRadius_1, chord);
                        _this._updateLink(_this._ribbon, link, linkRadius_1, chord);
                    }
                });
            }
        }
    });
    Object.defineProperty(Chord.prototype, "_getLinkPoints", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (link, linkRadius, chordLayoutItem) {
            var source = chordLayoutItem.source;
            var target = chordLayoutItem.target;
            var chordStartAngle = this.get("startAngle", 0) * $math.RADIANS;
            if (source && target) {
                var startAngle0 = source.startAngle;
                var endAngle0 = source.endAngle;
                var angle0 = startAngle0 + (endAngle0 - startAngle0) / 2 + chordStartAngle;
                var startAngle1 = target.startAngle;
                var endAngle1 = target.endAngle;
                var angle1 = startAngle1 + (endAngle1 - startAngle1) / 2 + chordStartAngle;
                link._p0 = { x: linkRadius * Math.cos(angle0), y: linkRadius * Math.sin(angle0) };
                link._p1 = { x: linkRadius * Math.cos(angle1), y: linkRadius * Math.sin(angle1) };
            }
        }
    });
    Object.defineProperty(Chord.prototype, "_updateLink", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (ribbon, link, linkRadius, chordLayoutItem) {
            if (chordLayoutItem) {
                ribbon.sourceRadius($utils.relativeToValue(link.get("sourceRadius", p100), linkRadius));
                ribbon.targetRadius($utils.relativeToValue(link.get("targetRadius", p100), linkRadius));
                link.set("draw", function (display) {
                    ribbon.context(display);
                    ribbon(chordLayoutItem);
                });
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Chord.prototype, "disposeDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            _super.prototype.disposeDataItem.call(this, dataItem);
            var label = dataItem.get("label");
            if (label) {
                label.dispose();
            }
        }
    });
    Object.defineProperty(Chord, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "Chord"
    });
    Object.defineProperty(Chord, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: Flow.classNames.concat([Chord.className])
    });
    return Chord;
}(Flow));
export { Chord };
//# sourceMappingURL=Chord.js.map