import { __awaiter, __extends, __generator } from "tslib";
import { HierarchyDefaultTheme } from "./HierarchyDefaultTheme";
import { Series } from "../../core/render/Series";
import { DataItem } from "../../core/render/Component";
import { HierarchyNode } from "./HierarchyNode";
import { Container } from "../../core/render/Container";
import { Label } from "../../core/render/Label";
import { Template } from "../../core/util/Template";
import { ListTemplate } from "../../core/util/List";
import * as $array from "../../core/util/Array";
import * as $type from "../../core/util/Type";
import * as $utils from "../../core/util/Utils";
import * as d3hierarchy from "d3-hierarchy";
;
/**
 * A base class for all hierarchy charts.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/hierarchy/} for more info
 */
var Hierarchy = /** @class */ (function (_super) {
    __extends(Hierarchy, _super);
    function Hierarchy() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        /**
         * A [[Container]] that nodes are placed in.
         *
         * @default Container.new()
         */
        Object.defineProperty(_this, "nodesContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.children.push(Container.new(_this._root, { isMeasured: false }))
        });
        Object.defineProperty(_this, "_rootNode", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_treeData", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_index", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "_tag", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: "hierarchy"
        });
        /**
         * A list of nodes in a [[Hierarchy]] chart.
         *
         * @default new ListTemplate<HierarchyNode>
         */
        Object.defineProperty(_this, "nodes", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new ListTemplate(Template.new({}), function () { return HierarchyNode.new(_this._root, {
                themeTags: $utils.mergeTags(_this.nodes.template.get("themeTags", []), [_this._tag, "hierarchy", "node"])
            }, _this.nodes.template); })
        });
        /**
         * A list of label elements in a [[Hierarchy]] chart.
         *
         * @default new ListTemplate<Label>
         */
        Object.defineProperty(_this, "labels", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new ListTemplate(Template.new({}), function () { return Label.new(_this._root, {
                themeTags: $utils.mergeTags(_this.labels.template.get("themeTags", []), [_this._tag])
            }, _this.labels.template); })
        });
        Object.defineProperty(_this, "_currentDownDepth", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        return _this;
    }
    /**
     * @ignore
     */
    Object.defineProperty(Hierarchy.prototype, "makeNode", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var childData = dataItem.get("childData");
            var node = this.nodes.make();
            node.series = this;
            node._setDataItem(dataItem);
            this.nodes.push(node);
            dataItem.setRaw("node", node);
            var label = this.labels.make();
            label._setDataItem(dataItem);
            dataItem.setRaw("label", label);
            this.labels.push(label);
            if (!childData || childData.length == 0) {
                node.addTag("last");
            }
            var depth = dataItem.get("depth");
            node.addTag("depth" + depth);
            this.nodesContainer.children.push(node);
            node.children.push(label);
            return node;
        }
    });
    Object.defineProperty(Hierarchy.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._defaultThemes.push(HierarchyDefaultTheme.new(this._root));
            this.fields.push("category", "childData", "disabled", "fill");
            this.children.push(this.bulletsContainer);
            _super.prototype._afterNew.call(this);
        }
    });
    Object.defineProperty(Hierarchy.prototype, "_prepareChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._prepareChildren.call(this);
            if (this._valuesDirty) {
                this._treeData = {};
                var first = this.dataItems[0];
                if (first) {
                    this._makeHierarchyData(this._treeData, first);
                    this._index = 0;
                    this._rootNode = d3hierarchy.hierarchy(this._treeData);
                    if (this._rootNode) {
                        this._rootNode.sum(function (d) {
                            return d.value;
                        });
                        var sort = this.get("sort");
                        if (sort == "descending") {
                            this._rootNode.sort(function (a, b) { return b.value - a.value; });
                        }
                        else if (sort == "ascending") {
                            this._rootNode.sort(function (a, b) { return a.value - b.value; });
                        }
                        this.setPrivateRaw("valueLow", Infinity);
                        this.setPrivateRaw("valueHigh", -Infinity);
                        this.setPrivateRaw("maxDepth", 0);
                        this._updateValues(this._rootNode);
                    }
                }
            }
            if (this._valuesDirty || this._sizeDirty) {
                this._updateVisuals();
            }
            if (this._sizeDirty) {
                this._selectDataItem(this.get("selectedDataItem"), this._currentDownDepth);
            }
        }
    });
    Object.defineProperty(Hierarchy.prototype, "_changed", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._changed.call(this);
            if (this.isDirty("selectedDataItem")) {
                this._selectDataItem(this.get("selectedDataItem"));
            }
        }
    });
    Object.defineProperty(Hierarchy.prototype, "_updateVisuals", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (this._rootNode) {
                this._updateNodes(this._rootNode);
            }
        }
    });
    Object.defineProperty(Hierarchy.prototype, "_updateNodes", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (hierarchyNode) {
            var _this = this;
            var dataItem = hierarchyNode.data.dataItem;
            if (dataItem) {
                this._updateNode(dataItem);
                if (this.bullets.length > 0 && !dataItem.bullets) {
                    this._makeBullets(dataItem);
                }
                var hierarchyChildren = hierarchyNode.children;
                if (hierarchyChildren) {
                    $array.each(hierarchyChildren, function (hierarchyChild) {
                        _this._updateNodes(hierarchyChild);
                    });
                }
            }
        }
    });
    Object.defineProperty(Hierarchy.prototype, "_updateNode", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_dataItem) {
        }
    });
    Object.defineProperty(Hierarchy.prototype, "_getDataItemById", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItems, id) {
            var _this = this;
            var di;
            $array.each(dataItems, function (dataItem) {
                if (dataItem.get("id") == id) {
                    di = dataItem;
                }
                var children = dataItem.get("children");
                if (children) {
                    var childDataItem = _this._getDataItemById(children, id);
                    if (childDataItem) {
                        di = childDataItem;
                    }
                }
            });
            return di;
        }
    });
    Object.defineProperty(Hierarchy.prototype, "_handleBullets", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItems) {
            var _this = this;
            $array.each(dataItems, function (dataItem) {
                var bullets = dataItem.bullets;
                if (bullets) {
                    $array.each(bullets, function (bullet) {
                        bullet.dispose();
                    });
                    dataItem.bullets = undefined;
                }
                var children = dataItem.get("children");
                if (children) {
                    _this._handleBullets(children);
                }
            });
            this._updateVisuals();
        }
    });
    Object.defineProperty(Hierarchy.prototype, "_onDataClear", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._onDataClear.call(this);
            var colors = this.get("colors");
            if (colors) {
                colors.reset();
            }
        }
    });
    Object.defineProperty(Hierarchy.prototype, "processDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var _this = this;
            _super.prototype.processDataItem.call(this, dataItem);
            var childData = dataItem.get("childData");
            var colors = this.get("colors");
            var topDepth = this.get("topDepth", 0);
            if (!dataItem.get("parent")) {
                dataItem.setRaw("depth", 0);
                if (colors && topDepth == 0 && dataItem.get("fill") == null) {
                    dataItem.setRaw("fill", colors.next());
                }
            }
            var depth = dataItem.get("depth");
            var initialDepth = this.get("initialDepth", 1);
            this.makeNode(dataItem);
            this._processDataItem(dataItem);
            if (childData) {
                var children_1 = [];
                dataItem.setRaw("children", children_1);
                $array.each(childData, function (child) {
                    var childDataItem = new DataItem(_this, child, _this._makeDataItem(child));
                    children_1.push(childDataItem);
                    childDataItem.setRaw("parent", dataItem);
                    childDataItem.setRaw("depth", depth + 1);
                    if (_this.dataItems.length == 1 && depth == 0) {
                        if (colors && childDataItem.get("fill") == null) {
                            childDataItem.setRaw("fill", colors.next());
                        }
                    }
                    else {
                        childDataItem.setRaw("fill", dataItem.get("fill"));
                    }
                    _this.processDataItem(childDataItem);
                });
            }
            var children = dataItem.get("children");
            if (!children || children.length == 0) {
                var node = dataItem.get("node");
                node.setAll({ toggleKey: undefined });
            }
            if (dataItem.get("disabled") == null) {
                if (depth >= topDepth + initialDepth) {
                    this.disableDataItem(dataItem, 0);
                }
            }
        }
    });
    Object.defineProperty(Hierarchy.prototype, "_processDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_dataItem) {
        }
    });
    Object.defineProperty(Hierarchy.prototype, "_updateValues", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (d3HierarchyNode) {
            var _this = this;
            var dataItem = d3HierarchyNode.data.dataItem;
            if (d3HierarchyNode.depth > this.getPrivate("maxDepth")) {
                this.setPrivateRaw("maxDepth", d3HierarchyNode.depth);
            }
            if (dataItem) {
                dataItem.setRaw("d3HierarchyNode", d3HierarchyNode);
                d3HierarchyNode.index = this._index;
                this._index++;
                dataItem.get("node").set("disabled", dataItem.get("disabled"));
                var dataValue = d3HierarchyNode.data.value;
                var value = d3HierarchyNode.value;
                if (dataValue != null) {
                    value = dataValue;
                    d3HierarchyNode["value"] = value;
                }
                if ($type.isNumber(value)) {
                    dataItem.setRaw("sum", value);
                    if (this.getPrivate("valueLow") > value) {
                        this.setPrivateRaw("valueLow", value);
                    }
                    if (this.getPrivate("valueHigh") < value) {
                        this.setPrivateRaw("valueHigh", value);
                    }
                }
                this.updateLegendValue(dataItem);
            }
            var hierarchyChildren = d3HierarchyNode.children;
            if (hierarchyChildren) {
                $array.each(hierarchyChildren, function (d3HierarchyChild) {
                    _this._updateValues(d3HierarchyChild);
                });
            }
        }
    });
    Object.defineProperty(Hierarchy.prototype, "_makeHierarchyData", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (data, dataItem) {
            var _this = this;
            data.dataItem = dataItem;
            var children = dataItem.get("children");
            if (children) {
                var childrenDataArray_1 = [];
                data.children = childrenDataArray_1;
                $array.each(children, function (childDataItem) {
                    var childData = {};
                    childrenDataArray_1.push(childData);
                    _this._makeHierarchyData(childData, childDataItem);
                });
                var value = dataItem.get("valueWorking");
                if ($type.isNumber(value)) {
                    data.value = value;
                }
            }
            else {
                var value = dataItem.get("valueWorking");
                if ($type.isNumber(value)) {
                    data.value = value;
                }
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Hierarchy.prototype, "disposeDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var _this = this;
            _super.prototype.disposeDataItem.call(this, dataItem);
            var node = dataItem.get("node");
            if (node) {
                this.nodes.removeValue(node);
                node.dispose();
            }
            var label = dataItem.get("label");
            if (label) {
                this.labels.removeValue(label);
                label.dispose();
            }
            var children = dataItem.get("children");
            if (children) {
                $array.each(children, function (child) {
                    _this.disposeDataItem(child);
                });
            }
        }
    });
    /**
     * Hides hierarchy's data item.
     *
     * @param   dataItem  Data item
     * @param   duration  Animation duration in milliseconds
     * @return            Promise
     */
    Object.defineProperty(Hierarchy.prototype, "hideDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, duration) {
            return __awaiter(this, void 0, void 0, function () {
                var promises, hiddenState, stateAnimationDuration, stateAnimationEasing, easing, children, node;
                var _this = this;
                return __generator(this, function (_a) {
                    switch (_a.label) {
                        case 0:
                            promises = [_super.prototype.hideDataItem.call(this, dataItem, duration)];
                            hiddenState = this.states.create("hidden", {});
                            if (!$type.isNumber(duration)) {
                                stateAnimationDuration = "stateAnimationDuration";
                                duration = hiddenState.get(stateAnimationDuration, this.get(stateAnimationDuration, 0));
                            }
                            stateAnimationEasing = "stateAnimationEasing";
                            easing = hiddenState.get(stateAnimationEasing, this.get(stateAnimationEasing));
                            children = dataItem.get("children");
                            if ((!children || children.length == 0) && $type.isNumber(dataItem.get("value"))) {
                                promises.push(dataItem.animate({ key: "valueWorking", to: 0, duration: duration, easing: easing }).waitForStop());
                            }
                            node = dataItem.get("node");
                            node.hide();
                            node.hideTooltip();
                            if (children) {
                                $array.each(children, function (childDataItem) {
                                    promises.push(_this.hideDataItem(childDataItem));
                                });
                            }
                            return [4 /*yield*/, Promise.all(promises)];
                        case 1:
                            _a.sent();
                            return [2 /*return*/];
                    }
                });
            });
        }
    });
    /**
     * Shows hierarchy's data item.
     *
     * @param   dataItem  Data item
     * @param   duration  Animation duration in milliseconds
     * @return            Promise
     */
    Object.defineProperty(Hierarchy.prototype, "showDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, duration) {
            return __awaiter(this, void 0, void 0, function () {
                var promises, easing, children, node;
                var _this = this;
                return __generator(this, function (_a) {
                    switch (_a.label) {
                        case 0:
                            promises = [_super.prototype.showDataItem.call(this, dataItem, duration)];
                            if (!$type.isNumber(duration)) {
                                duration = this.get("stateAnimationDuration", 0);
                            }
                            easing = this.get("stateAnimationEasing");
                            children = dataItem.get("children");
                            if ((!children || children.length == 0) && $type.isNumber(dataItem.get("value"))) {
                                promises.push(dataItem.animate({ key: "valueWorking", to: dataItem.get("value"), duration: duration, easing: easing }).waitForStop());
                            }
                            node = dataItem.get("node");
                            node.show();
                            if (children) {
                                $array.each(children, function (childDataItem) {
                                    promises.push(_this.showDataItem(childDataItem));
                                });
                            }
                            return [4 /*yield*/, Promise.all(promises)];
                        case 1:
                            _a.sent();
                            return [2 /*return*/];
                    }
                });
            });
        }
    });
    /**
     * Enables a disabled data item.
     *
     * @param  dataItem  Target data item
     * @param  duration  Animation duration in milliseconds
     */
    Object.defineProperty(Hierarchy.prototype, "enableDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, maxDepth, depth, duration) {
            var _this = this;
            if (depth == null) {
                depth = 0;
            }
            if (maxDepth == null) {
                maxDepth = 1;
            }
            dataItem.set("disabled", false);
            dataItem.get("node").set("disabled", false);
            if (!dataItem.isHidden()) {
                dataItem.get("node").show(duration);
            }
            if (depth == 0) {
                var upDepth = this.get("upDepth", Infinity);
                var parent_1 = dataItem;
                var count = 0;
                while (parent_1 !== undefined) {
                    if (count > upDepth) {
                        parent_1.get("node").hide();
                    }
                    parent_1 = parent_1.get("parent");
                    count++;
                }
            }
            var children = dataItem.get("children");
            if (children) {
                if (depth < maxDepth - 1) {
                    $array.each(children, function (child) {
                        _this.enableDataItem(child, maxDepth, depth + 1, duration);
                    });
                }
                else {
                    $array.each(children, function (child) {
                        if (!child.isHidden()) {
                            child.get("node").show(duration);
                            child.get("node").states.applyAnimate("disabled");
                            child.set("disabled", true);
                            _this.disableDataItem(child);
                        }
                    });
                }
            }
        }
    });
    /**
     * Disables a data item.
     *
     * @param  dataItem  Target data item
     * @param  duration  Animation duration in milliseconds
     */
    Object.defineProperty(Hierarchy.prototype, "disableDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, duration) {
            var _this = this;
            dataItem.set("disabled", true);
            var children = dataItem.get("children");
            if (children) {
                $array.each(children, function (child) {
                    _this.disableDataItem(child, duration);
                    child.get("node").hide(duration);
                });
            }
        }
    });
    Object.defineProperty(Hierarchy.prototype, "_selectDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, downDepth) {
            if (dataItem) {
                var type = "dataitemselected";
                this.events.dispatch(type, { type: type, target: this, dataItem: dataItem });
                var maxDepth = this.getPrivate("maxDepth", 1);
                var topDepth = this.get("topDepth", 0);
                if (downDepth == null) {
                    downDepth = Math.min(this.get("downDepth", 1), maxDepth - dataItem.get("depth"));
                }
                if (!this.inited) {
                    downDepth = Math.min(this.get("initialDepth", 1), maxDepth - topDepth);
                }
                this._currentDownDepth = downDepth;
                var hierarchyNode = dataItem.get("d3HierarchyNode");
                var currentDepth = hierarchyNode.depth;
                if (currentDepth + downDepth > maxDepth) {
                    downDepth = maxDepth - currentDepth;
                }
                if (currentDepth < topDepth) {
                    downDepth += topDepth - currentDepth;
                    currentDepth = topDepth;
                }
                var children = dataItem.get("children");
                if (children && children.length > 0) {
                    if (downDepth > 0) {
                        this.enableDataItem(dataItem, downDepth);
                    }
                    else {
                        dataItem.get("node").show();
                        $array.each(children, function (child) {
                            child.get("node").hide();
                        });
                    }
                    if (hierarchyNode.depth < topDepth) {
                        dataItem.get("node").hide(0);
                    }
                    if (this.get("singleBranchOnly")) {
                        this._handleSingle(dataItem);
                    }
                }
                else {
                    this.enableDataItem(this.dataItems[0], downDepth, 0);
                }
                this._zoom(dataItem);
            }
        }
    });
    Object.defineProperty(Hierarchy.prototype, "_zoom", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_dataItem) {
        }
    });
    Object.defineProperty(Hierarchy.prototype, "_handleSingle", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var _this = this;
            var parent = dataItem.get("parent");
            if (parent) {
                var children = parent.get("children");
                if (children) {
                    $array.each(children, function (child) {
                        if (child != dataItem) {
                            _this.disableDataItem(child);
                        }
                    });
                }
            }
        }
    });
    /**
     * Selects specific data item.
     *
     * @param  dataItem  Target data item
     */
    Object.defineProperty(Hierarchy.prototype, "selectDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var parent = dataItem.get("parent");
            var maxDepth = this.getPrivate("maxDepth", 1);
            if (this.get("selectedDataItem") == dataItem) {
                if (parent) {
                    this.set("selectedDataItem", parent);
                }
                else {
                    var depth = Math.min(this.get("downDepth", 1), maxDepth - dataItem.get("depth"));
                    if (this._currentDownDepth == depth) {
                        depth = Math.min(this.get("initialDepth", 1), maxDepth - this.get("topDepth", 0));
                    }
                    this._selectDataItem(dataItem, depth);
                }
            }
            else {
                this.set("selectedDataItem", dataItem);
            }
        }
    });
    Object.defineProperty(Hierarchy.prototype, "_makeBullet", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, bulletFunction, index) {
            var _this = this;
            var bullet = _super.prototype._makeBullet.call(this, dataItem, bulletFunction, index);
            if (bullet) {
                var sprite = bullet.get("sprite");
                var node = dataItem.get("node");
                if (sprite) {
                    node.children.push(sprite);
                    node.on("width", function () {
                        _this._positionBullet(bullet);
                    });
                    node.on("height", function () {
                        _this._positionBullet(bullet);
                    });
                }
            }
            return bullet;
        }
    });
    Object.defineProperty(Hierarchy.prototype, "_positionBullet", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (bullet) {
            var sprite = bullet.get("sprite");
            if (sprite) {
                var dataItem = sprite.dataItem;
                var locationX = bullet.get("locationX", 0.5);
                var locationY = bullet.get("locationY", 0.5);
                var node = dataItem.get("node");
                sprite.set("x", node.width() * locationX);
                sprite.set("y", node.height() * locationY);
            }
        }
    });
    /**
     * Triggers hover on a series data item.
     *
     * @since 5.0.7
     * @param  dataItem  Target data item
     */
    Object.defineProperty(Hierarchy.prototype, "hoverDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var node = dataItem.get("node");
            if (node && !node.isHidden()) {
                node.hover();
            }
        }
    });
    /**
     * Triggers un-hover on a series data item.
     *
     * @since 5.0.7
     * @param  dataItem  Target data item
     */
    Object.defineProperty(Hierarchy.prototype, "unhoverDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var node = dataItem.get("node");
            if (node) {
                node.unhover();
            }
        }
    });
    Object.defineProperty(Hierarchy, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "Hierarchy"
    });
    Object.defineProperty(Hierarchy, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: Series.classNames.concat([Hierarchy.className])
    });
    return Hierarchy;
}(Series));
export { Hierarchy };
//# sourceMappingURL=Hierarchy.js.map