import { __extends } from "tslib";
import { Partition } from "./Partition";
import { Template } from "../../core/util/Template";
import { ListTemplate } from "../../core/util/List";
import { Slice } from "../../core/render/Slice";
import { RadialLabel } from "../../core/render/RadialLabel";
import { p100, p50 } from "../../core/util/Percent";
import * as $array from "../../core/util/Array";
import * as d3hierarchy from "d3-hierarchy";
import * as $utils from "../../core/util/Utils";
import * as $type from "../../core/util/Type";
import * as $math from "../../core/util/Math";
;
/**
 * Builds a sunburst diagram.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/hierarchy/sunburst/} for more info
 * @important
 */
var Sunburst = /** @class */ (function (_super) {
    __extends(Sunburst, _super);
    function Sunburst() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "_tag", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: "sunburst"
        });
        Object.defineProperty(_this, "_partitionLayout", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: d3hierarchy.partition()
        });
        /**
         * A list of node slice elements in a [[Sunburst]] chart.
         *
         * @default new ListTemplate<Slice>
         */
        Object.defineProperty(_this, "slices", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new ListTemplate(Template.new({}), function () { return Slice._new(_this._root, {
                themeTags: $utils.mergeTags(_this.slices.template.get("themeTags", []), [_this._tag, "hierarchy", "node", "shape"])
            }, [_this.slices.template]); })
        });
        /**
         * A list of label elements in a [[Hierarchy]] chart.
         *
         * @default new ListTemplate<RadialLabel>
         */
        Object.defineProperty(_this, "labels", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new ListTemplate(Template.new({}), function () { return RadialLabel._new(_this._root, {
                themeTags: $utils.mergeTags(_this.labels.template.get("themeTags", []), [_this._tag])
            }, [_this.labels.template]); })
        });
        return _this;
    }
    Object.defineProperty(Sunburst.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._afterNew.call(this);
            this.nodesContainer.setAll({ x: p50, y: p50 });
            this.setPrivateRaw("dx", 0);
            this.setPrivateRaw("dr", 0);
        }
    });
    Object.defineProperty(Sunburst.prototype, "_prepareChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._prepareChildren.call(this);
            if (this.isPrivateDirty("dr") || this.isPrivateDirty("dx")) {
                if (this._rootNode) {
                    this._updateNodesScale(this._rootNode);
                }
            }
        }
    });
    Object.defineProperty(Sunburst.prototype, "_updateVisuals", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (this._rootNode) {
                var partitionLayout = this._partitionLayout;
                var bounds = $math.getArcBounds(0, 0, this.get("startAngle", 0), this.get("endAngle", 360), 1);
                var w = this.innerWidth();
                var h = this.innerHeight();
                var wr = w / (bounds.right - bounds.left);
                var hr = h / (bounds.bottom - bounds.top);
                var s = Math.min(wr, hr);
                var r = $utils.relativeToValue(this.get("radius", p100), s);
                var ir = $utils.relativeToValue(this.get("innerRadius", 0), r);
                if (ir < 0) {
                    ir = r + ir;
                }
                s = r - ir;
                this.setPrivateRaw("innerRadius", ir);
                this.setPrivateRaw("hierarchySize", s);
                partitionLayout.size([s, s]);
                this.nodesContainer.setAll({
                    dy: -r * (bounds.bottom + bounds.top) / 2, dx: -r * (bounds.right + bounds.left) / 2
                });
                var nodePadding = this.get("nodePadding");
                if ($type.isNumber(nodePadding)) {
                    partitionLayout.padding(nodePadding);
                }
                partitionLayout(this._rootNode);
                this._updateNodes(this._rootNode);
            }
        }
    });
    Object.defineProperty(Sunburst.prototype, "_updateNode", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            _super.prototype._updateNode.call(this, dataItem);
            var hierarchyNode = dataItem.get("d3HierarchyNode");
            var node = dataItem.get("node");
            node.setAll({ x: 0, y: 0 });
            var duration = this.get("animationDuration", 0);
            var easing = this.get("animationEasing");
            var scaleX = this.getPrivate("scaleX", 1);
            var scaleY = this.getPrivate("scaleY", 1);
            var dr = this.getPrivate("dr", 0);
            var dx = this.getPrivate("dx", 0);
            var x0 = hierarchyNode.x0 * scaleX + dx;
            var x1 = hierarchyNode.x1 * scaleX + dx;
            var y0 = hierarchyNode.y0 * scaleY;
            var y1 = hierarchyNode.y1 * scaleY;
            var ir = this.getPrivate("innerRadius");
            var hs = this.getPrivate("hierarchySize", 0);
            var slice = dataItem.get("slice");
            if (slice) {
                var startAngle = this.get("startAngle", -90);
                var endAngle = this.get("endAngle", 270);
                var sliceStartAngle = startAngle + x0 / hs * (endAngle - startAngle);
                var arc = startAngle + x1 / hs * (endAngle - startAngle) - sliceStartAngle;
                var sliceInnerRadius = ir + y0;
                var sliceRadius = ir + y1;
                sliceInnerRadius -= dr;
                sliceRadius -= dr;
                sliceRadius = Math.max(0, sliceRadius);
                sliceInnerRadius = Math.max(0, sliceInnerRadius);
                slice.animate({ key: "radius", to: sliceRadius, duration: duration, easing: easing });
                slice.animate({ key: "innerRadius", to: sliceInnerRadius, duration: duration, easing: easing });
                slice.animate({ key: "startAngle", to: sliceStartAngle, duration: duration, easing: easing });
                slice.animate({ key: "arc", to: arc, duration: duration, easing: easing });
                var fill = dataItem.get("fill");
                slice._setDefault("fill", fill);
                slice._setDefault("stroke", fill);
            }
        }
    });
    Object.defineProperty(Sunburst.prototype, "_updateNodesScale", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (hierarchyNode) {
            var _this = this;
            var dataItem = hierarchyNode.data.dataItem;
            if (dataItem) {
                var scaleX = this.getPrivate("scaleX", 1);
                var scaleY = this.getPrivate("scaleY", 1);
                var dr = this.getPrivate("dr", 0);
                var dx = this.getPrivate("dx", 0);
                var x0 = hierarchyNode.x0 * scaleX + dx;
                var x1 = hierarchyNode.x1 * scaleX + dx;
                var y0 = hierarchyNode.y0 * scaleY;
                var y1 = hierarchyNode.y1 * scaleY;
                var ir = this.getPrivate("innerRadius");
                var hs = this.getPrivate("hierarchySize", 0);
                var slice = dataItem.get("slice");
                if (slice) {
                    var startAngle = this.get("startAngle", -90);
                    var endAngle = this.get("endAngle", 270);
                    var sliceStartAngle = startAngle + x0 / hs * (endAngle - startAngle);
                    var arc = startAngle + x1 / hs * (endAngle - startAngle) - sliceStartAngle;
                    var sliceInnerRadius = ir + y0;
                    var sliceRadius = ir + y1;
                    sliceInnerRadius -= dr;
                    sliceRadius -= dr;
                    sliceRadius = Math.max(0, sliceRadius);
                    sliceInnerRadius = Math.max(0, sliceInnerRadius);
                    slice.setAll({ radius: sliceRadius, innerRadius: sliceInnerRadius, startAngle: sliceStartAngle, arc: arc });
                }
                var hierarchyChildren = hierarchyNode.children;
                if (hierarchyChildren) {
                    $array.each(hierarchyChildren, function (hierarchyChild) {
                        _this._updateNodesScale(hierarchyChild);
                    });
                }
            }
        }
    });
    Object.defineProperty(Sunburst.prototype, "_makeNode", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, node) {
            var _this = this;
            var slice = node.children.moveValue(this.slices.make(), 0);
            node.setPrivate("tooltipTarget", slice);
            dataItem.setRaw("slice", slice);
            slice._setDataItem(dataItem);
            slice.on("arc", function () {
                _this._updateLabel(dataItem);
            });
            slice.on("innerRadius", function () {
                _this._updateLabel(dataItem);
            });
            slice.on("radius", function () {
                _this._updateLabel(dataItem);
            });
        }
    });
    Object.defineProperty(Sunburst.prototype, "_updateLabel", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var slice = dataItem.get("slice");
            var label = dataItem.get("label");
            if (slice && label) {
                var innerRadius = slice.get("innerRadius", 0);
                var radius = slice.get("radius", 0);
                var angle = slice.get("startAngle", 0);
                var arc = Math.abs(slice.get("arc", 0));
                var labelAngle = angle + arc / 2;
                var maxWidth = radius - innerRadius;
                var maxHeight = radius * arc * $math.RADIANS;
                if (innerRadius == 0 && arc >= 360) {
                    radius = 1;
                    labelAngle = 0;
                }
                label.set("labelAngle", labelAngle);
                label.setPrivate("radius", radius);
                label.setPrivate("innerRadius", innerRadius);
                if (arc >= 360) {
                    maxWidth *= 2;
                    maxHeight = maxWidth;
                }
                label.setAll({
                    maxHeight: maxHeight,
                    maxWidth: maxWidth
                });
            }
        }
    });
    Object.defineProperty(Sunburst.prototype, "_zoom", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var x0 = 0;
            var x1 = 0;
            var hs = this.getPrivate("hierarchySize", 0);
            var hierarchyNode = dataItem.get("d3HierarchyNode");
            var upDepth = this.get("upDepth", 0);
            var topDepth = this.get("topDepth", 0);
            var currentDepth = hierarchyNode.depth;
            var maxDepth = this.getPrivate("maxDepth", 1);
            var downDepth = this._currentDownDepth;
            if (downDepth == null) {
                downDepth = this.get("downDepth", 1);
            }
            var levelHeight = hs / (maxDepth + 1);
            if (currentDepth < topDepth) {
                currentDepth = topDepth;
            }
            if (currentDepth - upDepth < 0) {
                upDepth = currentDepth;
            }
            x0 = hierarchyNode.x0;
            x1 = hierarchyNode.x1;
            var scaleDepth = (downDepth + upDepth + 1);
            if (scaleDepth > maxDepth - topDepth + 1) {
                scaleDepth = maxDepth - topDepth + 1;
            }
            var scaleX = hs / (x1 - x0);
            var scaleY = hs / (levelHeight * scaleDepth);
            var dr = Math.max(currentDepth - upDepth, topDepth) * levelHeight * scaleY;
            var easing = this.get("animationEasing");
            var duration = this.get("animationDuration", 0);
            if (!this.inited) {
                duration = 0;
            }
            var dx = -x0 * scaleX;
            this.animatePrivate({ key: "scaleX", to: scaleX, duration: duration, easing: easing });
            this.animatePrivate({ key: "scaleY", to: scaleY, duration: duration, easing: easing });
            this.animatePrivate({ key: "dr", to: dr, duration: duration, easing: easing });
            this.animatePrivate({ key: "dx", to: dx, duration: duration, easing: easing });
        }
    });
    Object.defineProperty(Sunburst.prototype, "_handleSingle", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var _this = this;
            var parent = dataItem.get("parent");
            if (parent) {
                var children = parent.get("children");
                if (children) {
                    $array.each(children, function (child) {
                        if (child != dataItem) {
                            _this.disableDataItem(child);
                            child.get("node").hide();
                        }
                    });
                }
                this._handleSingle(parent);
            }
        }
    });
    Object.defineProperty(Sunburst.prototype, "_positionBullet", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (bullet) {
            var sprite = bullet.get("sprite");
            if (sprite) {
                var dataItem = sprite.dataItem;
                var locationX = bullet.get("locationX", 0.5);
                var locationY = bullet.get("locationY", 0.5);
                var slice = dataItem.get("slice");
                var arc = slice.get("arc", 0);
                var angle = slice.get("startAngle", 0) + slice.get("arc", 0) * locationX;
                var innerRadius = slice.get("innerRadius", 0);
                var radius = innerRadius + (slice.get("radius", 0) - innerRadius) * locationY;
                var x = $math.cos(angle) * radius;
                var y = $math.sin(angle) * radius;
                if ($math.round(arc, 5) === 360 && $math.round(innerRadius, 2) === 0) {
                    x = 0;
                    y = 0;
                }
                sprite.set("x", x);
                sprite.set("y", y);
            }
        }
    });
    Object.defineProperty(Sunburst.prototype, "_makeBullet", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, bulletFunction, index) {
            var _this = this;
            var bullet = _super.prototype._makeBullet.call(this, dataItem, bulletFunction, index);
            if (bullet) {
                var sprite = bullet.get("sprite");
                var slice = dataItem.get("slice");
                if (sprite && slice) {
                    slice.on("arc", function () {
                        _this._positionBullet(bullet);
                    });
                    slice.on("radius", function () {
                        _this._positionBullet(bullet);
                    });
                }
                return bullet;
            }
        }
    });
    Object.defineProperty(Sunburst, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "Sunburst"
    });
    Object.defineProperty(Sunburst, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: Partition.classNames.concat([Sunburst.className])
    });
    return Sunburst;
}(Partition));
export { Sunburst };
//# sourceMappingURL=Sunburst.js.map