import { __extends } from "tslib";
import { MapChartDefaultTheme } from "./MapChartDefaultTheme";
import { SerialChart } from "../../core/render/SerialChart";
import { Rectangle } from "../../core/render/Rectangle";
import { geoPath } from "d3-geo";
import { Color } from "../../core/util/Color";
import { registry } from "../../core/Registry";
import * as $math from "../../core/util/Math";
import * as $array from "../../core/util/Array";
import * as $type from "../../core/util/Type";
import * as $mapUtils from "./MapUtils";
import * as $object from "../../core/util/Object";
import * as $utils from "../../core/util/Utils";
var MapChart = /** @class */ (function (_super) {
    __extends(MapChart, _super);
    function MapChart() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "_downTranslateX", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_downTranslateY", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_downRotationX", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_downRotationY", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_downRotationZ", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_pLat", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "_pLon", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "_movePoints", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        Object.defineProperty(_this, "_downZoomLevel", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 1
        });
        Object.defineProperty(_this, "_doubleDownDistance", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "_dirtyGeometries", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(_this, "_geometryColection", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: { type: "GeometryCollection", geometries: [] }
        });
        Object.defineProperty(_this, "_centerLocation", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: null
        });
        Object.defineProperty(_this, "_za", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_rxa", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_rya", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_txa", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_tya", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_mapBounds", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: [[0, 0], [0, 0]]
        });
        Object.defineProperty(_this, "_geoCentroid", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: { longitude: 0, latitude: 0 }
        });
        Object.defineProperty(_this, "_geoBounds", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: { left: 0, right: 0, top: 0, bottom: 0 }
        });
        Object.defineProperty(_this, "_prevGeoBounds", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: { left: 0, right: 0, top: 0, bottom: 0 }
        });
        Object.defineProperty(_this, "_dispatchBounds", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(_this, "_wheelDp", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_pw", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_ph", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_mapFitted", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        return _this;
    }
    Object.defineProperty(MapChart.prototype, "_makeGeoPath", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var projection = this.get("projection");
            var path = geoPath();
            path.projection(projection);
            this.setPrivateRaw("geoPath", path);
        }
    });
    /**
     * Returns coordinates to geographical center of the map.
     */
    Object.defineProperty(MapChart.prototype, "geoCentroid", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this._geoCentroid;
        }
    });
    /**
     * Returns geographical bounds of the map.
     */
    Object.defineProperty(MapChart.prototype, "geoBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this._geoBounds;
        }
    });
    Object.defineProperty(MapChart.prototype, "_handleSetWheel", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            var wheelX = this.get("wheelX");
            var wheelY = this.get("wheelY");
            var chartContainer = this.chartContainer;
            if (wheelX != "none" || wheelY != "none") {
                this._wheelDp = chartContainer.events.on("wheel", function (event) {
                    var wheelEasing = _this.get("wheelEasing");
                    var wheelSensitivity = _this.get("wheelSensitivity", 1);
                    var wheelDuration = _this.get("wheelDuration", 0);
                    var wheelEvent = event.originalEvent;
                    wheelEvent.preventDefault();
                    var chartContainer = _this.chartContainer;
                    var point = chartContainer._display.toLocal(event.point);
                    if ((wheelY == "zoom")) {
                        _this._handleWheelZoom(wheelEvent.deltaY, point);
                    }
                    else if (wheelY == "rotateY") {
                        _this._handleWheelRotateY(wheelEvent.deltaY / 5 * wheelSensitivity, wheelDuration, wheelEasing);
                    }
                    else if (wheelY == "rotateX") {
                        _this._handleWheelRotateX(wheelEvent.deltaY / 5 * wheelSensitivity, wheelDuration, wheelEasing);
                    }
                    if ((wheelX == "zoom")) {
                        _this._handleWheelZoom(wheelEvent.deltaX, point);
                    }
                    else if (wheelX == "rotateY") {
                        _this._handleWheelRotateY(wheelEvent.deltaX / 5 * wheelSensitivity, wheelDuration, wheelEasing);
                    }
                    else if (wheelX == "rotateX") {
                        _this._handleWheelRotateX(wheelEvent.deltaX / 5 * wheelSensitivity, wheelDuration, wheelEasing);
                    }
                });
                this._disposers.push(this._wheelDp);
            }
            else {
                if (this._wheelDp) {
                    this._wheelDp.dispose();
                }
            }
        }
    });
    Object.defineProperty(MapChart.prototype, "_prepareChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            _super.prototype._prepareChildren.call(this);
            var projection = this.get("projection");
            var w = this.innerWidth();
            var h = this.innerHeight();
            if (this.isDirty("projection")) {
                this._makeGeoPath();
                this.markDirtyProjection();
                this._fitMap();
                projection.scale(this.getPrivate("mapScale") * this.get("zoomLevel", 1));
                if (projection.rotate) {
                    projection.rotate([this.get("rotationX", 0), this.get("rotationY", 0), this.get("rotationZ", 0)]);
                }
                var prev = this._prevSettings.projection;
                if (prev && prev != projection) {
                    var hw = w / 2;
                    var hh = h / 2;
                    if (prev.invert) {
                        var centerLocation = prev.invert([hw, hh]);
                        if (centerLocation) {
                            var xy = projection(centerLocation);
                            if (xy) {
                                var translate = projection.translate();
                                var xx = hw - ((xy[0] - translate[0]));
                                var yy = hh - ((xy[1] - translate[1]));
                                projection.translate([xx, yy]);
                                this.setRaw("translateX", xx);
                                this.setRaw("translateY", yy);
                            }
                        }
                    }
                }
            }
            if (this.isDirty("wheelX") || this.isDirty("wheelY")) {
                this._handleSetWheel();
            }
            var previousGeometries = this._geometryColection.geometries;
            if (this._dirtyGeometries) {
                this._geometryColection.geometries = [];
                this.series.each(function (series) {
                    $array.pushAll(_this._geometryColection.geometries, series._geometries);
                });
                this._fitMap();
            }
            if (previousGeometries.length != 0 && (w != this._pw || h != this._ph || this._dirtyGeometries)) {
                if (w > 0 && h > 0) {
                    var hw = w / 2;
                    var hh = h / 2;
                    projection.fitSize([w, h], this._geometryColection);
                    var newScale = projection.scale();
                    this.setPrivateRaw("mapScale", newScale);
                    projection.scale(newScale * this.get("zoomLevel", 1));
                    if (this._centerLocation) {
                        var xy = projection(this._centerLocation);
                        if (xy) {
                            var translate = projection.translate();
                            var xx = hw - ((xy[0] - translate[0]));
                            var yy = hh - ((xy[1] - translate[1]));
                            projection.translate([xx, yy]);
                            this.setRaw("translateX", xx);
                            this.setRaw("translateY", yy);
                        }
                    }
                    this.markDirtyProjection();
                }
            }
            this._pw = w;
            this._ph = h;
            if (this.isDirty("zoomControl")) {
                var previous = this._prevSettings.zoomControl;
                var zoomControl = this.get("zoomControl");
                if (zoomControl !== previous) {
                    this._disposeProperty("zoomControl");
                    if (previous) {
                        previous.dispose();
                    }
                    if (zoomControl) {
                        zoomControl.setPrivate("chart", this);
                        this.children.push(zoomControl);
                    }
                    this.setRaw("zoomControl", zoomControl);
                }
            }
            if (this.isDirty("zoomLevel")) {
                projection.scale(this.getPrivate("mapScale") * this.get("zoomLevel", 1));
                this.markDirtyProjection();
            }
            if (this.isDirty("translateX") || this.isDirty("translateY")) {
                projection.translate([this.get("translateX", this.width() / 2), this.get("translateY", this.height() / 2)]);
                this.markDirtyProjection();
            }
            if (projection.rotate) {
                if (this.isDirty("rotationX") || this.isDirty("rotationY") || this.isDirty("rotationZ")) {
                    projection.rotate([this.get("rotationX", 0), this.get("rotationY", 0), this.get("rotationZ", 0)]);
                    this.markDirtyProjection();
                }
            }
        }
    });
    Object.defineProperty(MapChart.prototype, "_fitMap", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var projection = this.get("projection");
            var w = this.innerWidth();
            var h = this.innerHeight();
            if (w > 0 && h > 0) {
                projection.fitSize([w, h], this._geometryColection);
                this.setPrivateRaw("mapScale", projection.scale());
                var translate = projection.translate();
                this.setRaw("translateX", translate[0]);
                this.setRaw("translateY", translate[1]);
                var geoPath_1 = this.getPrivate("geoPath");
                this._mapBounds = geoPath_1.bounds(this._geometryColection);
                this._geoCentroid = $mapUtils.getGeoCentroid(this._geometryColection);
                var bounds = $mapUtils.getGeoBounds(this._geometryColection);
                this._geoBounds = bounds;
                if (this._geometryColection.geometries.length > 0) {
                    bounds.left = $math.round(this._geoBounds.left, 3);
                    bounds.right = $math.round(this._geoBounds.right, 3);
                    bounds.top = $math.round(this._geoBounds.top, 3);
                    bounds.bottom = $math.round(this._geoBounds.bottom, 3);
                    var prevGeoBounds = this._prevGeoBounds;
                    if (prevGeoBounds && !$utils.sameBounds(bounds, prevGeoBounds)) {
                        this._dispatchBounds = true;
                        this._prevGeoBounds = bounds;
                    }
                }
                this._mapFitted = true;
            }
        }
    });
    /**
     * Repositions the map to the "home" zoom level and center coordinates.
     *
     * @see {@link https://www.amcharts.com/docs/v5/charts/map-chart/map-pan-zoom/#Resetting_position_level} for more info
     * @param  duration  Animation duration in milliseconds
     */
    Object.defineProperty(MapChart.prototype, "goHome", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (duration) {
            var homeGeoPoint = this.get("homeGeoPoint");
            var homeZoomLevel = this.get("homeZoomLevel", 1);
            if (!homeGeoPoint) {
                var geoPath_2 = this.getPrivate("geoPath");
                var bounds = geoPath_2.bounds(this._geometryColection);
                var left = bounds[0][0];
                var top_1 = bounds[0][1];
                var right = bounds[1][0];
                var bottom = bounds[1][1];
                homeGeoPoint = this.invert({ x: left + (right - left) / 2, y: top_1 + (bottom - top_1) / 2 });
            }
            this.zoomToGeoPoint(homeGeoPoint, homeZoomLevel, true, duration);
        }
    });
    Object.defineProperty(MapChart.prototype, "_updateChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var projection = this.get("projection");
            if (projection.invert) {
                var w = this.innerWidth();
                var h = this.innerHeight();
                if (w > 0 && h > 0) {
                    this._centerLocation = projection.invert([this.innerWidth() / 2, this.innerHeight() / 2]);
                }
            }
        }
    });
    Object.defineProperty(MapChart.prototype, "_afterChanged", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._afterChanged.call(this);
            if (this._dispatchBounds) {
                this._dispatchBounds = false;
                var type = "geoboundschanged";
                if (this.events.isEnabled(type)) {
                    this.events.dispatch(type, { type: type, target: this });
                }
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(MapChart.prototype, "markDirtyGeometries", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._dirtyGeometries = true;
            this.markDirty();
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(MapChart.prototype, "markDirtyProjection", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this.series.each(function (series) {
                series.markDirtyProjection();
            });
        }
    });
    Object.defineProperty(MapChart.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            this._defaultThemes.push(MapChartDefaultTheme.new(this._root));
            this._settings.themeTags = $utils.mergeTags(this._settings.themeTags, ["map"]);
            _super.prototype._afterNew.call(this);
            this._makeGeoPath();
            this.chartContainer.children.push(this.seriesContainer);
            if (this.get("translateX") == null) {
                this.set("translateX", this.width() / 2);
            }
            if (this.get("translateY") == null) {
                this.set("translateY", this.height() / 2);
            }
            // Setting trasnparent background so that full body of the plot container
            // is interactive
            this.chartContainer.set("interactive", true);
            this.chartContainer.set("interactiveChildren", false);
            this.chartContainer.set("background", Rectangle.new(this._root, {
                themeTags: ["map", "background"],
                fill: Color.fromHex(0x000000),
                fillOpacity: 0
            }));
            this._disposers.push(this.chartContainer.events.on("pointerdown", function (event) {
                _this._handleChartDown(event);
            }));
            this._disposers.push(this.chartContainer.events.on("globalpointerup", function (event) {
                _this._handleChartUp(event);
            }));
            this._disposers.push(this.chartContainer.events.on("globalpointermove", function (event) {
                _this._handleChartMove(event);
            }));
            var license = false;
            for (var i = 0; i < registry.licenses.length; i++) {
                if (registry.licenses[i].match(/^AM5M.{5,}/i)) {
                    license = true;
                }
            }
            if (!license) {
                this._root._showBranding();
            }
        }
    });
    Object.defineProperty(MapChart.prototype, "_handleChartDown", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (event) {
            this._downZoomLevel = this.get("zoomLevel", 1);
            var count = $object.keys(this.chartContainer._downPoints).length;
            if (count > 0) {
                this._downTranslateX = this.get("translateX");
                this._downTranslateY = this.get("translateY");
                this._downRotationX = this.get("rotationX");
                this._downRotationY = this.get("rotationY");
                this._downRotationZ = this.get("rotationZ");
                var downId = this.chartContainer._getDownPointId();
                if (downId) {
                    var movePoint = this._movePoints[downId];
                    if (movePoint) {
                        this.chartContainer._downPoints[downId] = movePoint;
                    }
                }
            }
            else if (count == 0) {
                var bg = this.chartContainer.get("background");
                if (bg) {
                    bg.events.enableType("click");
                }
                if (this.get("panX") || this.get("panY")) {
                    if (this._za) {
                        this._za.stop();
                    }
                    if (this._txa) {
                        this._txa.stop();
                    }
                    if (this._tya) {
                        this._tya.stop();
                    }
                    if (this._rxa) {
                        this._rxa.stop();
                    }
                    if (this._rya) {
                        this._rya.stop();
                    }
                    var downPoint = this.chartContainer._display.toLocal(event.point);
                    this._downTranslateX = this.get("translateX");
                    this._downTranslateY = this.get("translateY");
                    this._downRotationX = this.get("rotationX");
                    this._downRotationY = this.get("rotationY");
                    this._downRotationZ = this.get("rotationZ");
                    var projection = this.get("projection");
                    if (projection.invert) {
                        var l0 = projection.invert([downPoint.x, downPoint.y]);
                        var l1 = projection.invert([downPoint.x + 1, downPoint.y + 1]);
                        if (l0 && l1) {
                            this._pLon = Math.abs(l1[0] - l0[0]);
                            this._pLat = Math.abs(l1[1] - l0[1]);
                        }
                    }
                }
            }
        }
    });
    /**
     * Converts screen coordinates (X and Y) within chart to latitude and
     * longitude.
     *
     * @param  point  Screen coordinates
     * @return        Geographical coordinates
     */
    Object.defineProperty(MapChart.prototype, "invert", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (point) {
            var projection = this.get("projection");
            if (projection.invert) {
                var ll = projection.invert([point.x, point.y]);
                if (ll) {
                    return { longitude: ll[0], latitude: ll[1] };
                }
            }
            return { longitude: 0, latitude: 0 };
        }
    });
    /**
     * Converts latitude/longitude to screen coordinates (X and Y).
     *
     * @param  point  Geographical coordinates
     * @return        Screen coordinates
     */
    Object.defineProperty(MapChart.prototype, "convert", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (point) {
            var projection = this.get("projection");
            var xy = projection([point.longitude, point.latitude]);
            if (xy) {
                return { x: xy[0], y: xy[1] };
            }
            return { x: 0, y: 0 };
        }
    });
    Object.defineProperty(MapChart.prototype, "_handleChartUp", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_event) {
            this.chartContainer._downPoints = {};
        }
    });
    Object.defineProperty(MapChart.prototype, "_handlePinch", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            var chartContainer = this.chartContainer;
            var i = 0;
            var downPoints = [];
            var movePoints = [];
            $object.each(chartContainer._downPoints, function (k, point) {
                downPoints[i] = point;
                var movePoint = _this._movePoints[k];
                if (movePoint) {
                    movePoints[i] = _this._movePoints[k];
                }
                i++;
            });
            if (downPoints.length > 1 && movePoints.length > 1) {
                var display = chartContainer._display;
                var downPoint0 = downPoints[0];
                var downPoint1 = downPoints[1];
                var movePoint0 = movePoints[0];
                var movePoint1 = movePoints[1];
                if (downPoint0 && downPoint1 && movePoint0 && movePoint1) {
                    downPoint0 = display.toLocal(downPoint0);
                    downPoint1 = display.toLocal(downPoint1);
                    movePoint0 = display.toLocal(movePoint0);
                    movePoint1 = display.toLocal(movePoint1);
                    var initialDistance = Math.hypot(downPoint1.x - downPoint0.x, downPoint1.y - downPoint0.y);
                    var currentDistance = Math.hypot(movePoint1.x - movePoint0.x, movePoint1.y - movePoint0.y);
                    var level = currentDistance / initialDistance * this._downZoomLevel;
                    var moveCenter = { x: movePoint0.x + (movePoint1.x - movePoint0.x) / 2, y: movePoint0.y + (movePoint1.y - movePoint0.y) / 2 };
                    var downCenter = { x: downPoint0.x + (downPoint1.x - downPoint0.x) / 2, y: downPoint0.y + (downPoint1.y - downPoint0.y) / 2 };
                    var tx = this._downTranslateX || 0;
                    var ty = this._downTranslateY || 0;
                    var zoomLevel = this._downZoomLevel;
                    var xx = moveCenter.x - (moveCenter.x - tx - moveCenter.x + downCenter.x) / zoomLevel * level;
                    var yy = moveCenter.y - (moveCenter.y - ty - moveCenter.y + downCenter.y) / zoomLevel * level;
                    this.set("zoomLevel", level);
                    this.set("translateX", xx);
                    this.set("translateY", yy);
                }
            }
        }
    });
    Object.defineProperty(MapChart.prototype, "_handleChartMove", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (event) {
            var chartContainer = this.chartContainer;
            var downPoint = chartContainer._getDownPoint();
            var downPointId = chartContainer._getDownPointId();
            var originalEvent = event.originalEvent;
            var pointerId = originalEvent.pointerId;
            if (this.get("pinchZoom")) {
                if (pointerId) {
                    this._movePoints[pointerId] = event.point;
                    if ($object.keys(chartContainer._downPoints).length > 1) {
                        this._handlePinch();
                        return;
                    }
                }
            }
            if (downPointId && pointerId && pointerId != downPointId) {
                return;
            }
            else {
                if (downPoint) {
                    var panX = this.get("panX");
                    var panY = this.get("panY");
                    if (panX != "none" || panY != "none") {
                        var display = chartContainer._display;
                        var local = display.toLocal(event.point);
                        downPoint = display.toLocal(downPoint);
                        var x = this._downTranslateX;
                        var y = this._downTranslateY;
                        if (Math.hypot(downPoint.x - local.x, downPoint.y - local.y) > 5) {
                            var bg = chartContainer.get("background");
                            if (bg) {
                                bg.events.disableType("click");
                            }
                            if ($type.isNumber(x) && $type.isNumber(y)) {
                                var projection = this.get("projection");
                                if (panX == "translateX") {
                                    x += local.x - downPoint.x;
                                }
                                if (panY == "translateY") {
                                    y += local.y - downPoint.y;
                                }
                                this.set("translateX", x);
                                this.set("translateY", y);
                                if (projection.invert) {
                                    var downLocation = projection.invert([downPoint.x, downPoint.y]);
                                    if (location && downLocation) {
                                        if (panX == "rotateX") {
                                            this.set("rotationX", this._downRotationX - (downPoint.x - local.x) * this._pLon);
                                        }
                                        if (panY == "rotateY") {
                                            this.set("rotationY", this._downRotationY + (downPoint.y - local.y) * this._pLat);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    });
    Object.defineProperty(MapChart.prototype, "_handleWheelRotateY", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (delta, duration, easing) {
            this._rya = this.animate({ key: "rotationY", to: this.get("rotationY", 0) - delta, duration: duration, easing: easing });
        }
    });
    Object.defineProperty(MapChart.prototype, "_handleWheelRotateX", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (delta, duration, easing) {
            this._rxa = this.animate({ key: "rotationX", to: this.get("rotationX", 0) - delta, duration: duration, easing: easing });
        }
    });
    Object.defineProperty(MapChart.prototype, "_handleWheelZoom", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (delta, point) {
            var step = this.get("zoomStep", 2);
            var zoomLevel = this.get("zoomLevel", 1);
            var newZoomLevel = zoomLevel;
            if (delta > 0) {
                newZoomLevel = zoomLevel / step;
            }
            else if (delta < 0) {
                newZoomLevel = zoomLevel * step;
            }
            if (newZoomLevel != zoomLevel) {
                this.zoomToPoint(point, newZoomLevel);
            }
        }
    });
    /**
     * Zoom the map to geographical bounds.
     *
     * @param  geoBounds  Bounds
     * @param  duration   Animation duration in milliseconds
     */
    Object.defineProperty(MapChart.prototype, "zoomToGeoBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (geoBounds, duration) {
            if (geoBounds.right < geoBounds.left) {
                geoBounds.right = 180;
                geoBounds.left = -180;
            }
            var geoPath = this.getPrivate("geoPath");
            var mapBounds = geoPath.bounds(this._geometryColection);
            var p0 = this.convert({ longitude: geoBounds.left, latitude: geoBounds.top });
            var p1 = this.convert({ longitude: geoBounds.right, latitude: geoBounds.bottom });
            if (p0.y < mapBounds[0][1]) {
                p0.y = mapBounds[0][1];
            }
            if (p1.y > mapBounds[1][1]) {
                p1.y = mapBounds[1][1];
            }
            var zl = this.get("zoomLevel", 1);
            var bounds = { left: p0.x, right: p1.x, top: p0.y, bottom: p1.y };
            var seriesContainer = this.seriesContainer;
            var zoomLevel = .9 * Math.min(seriesContainer.innerWidth() / (bounds.right - bounds.left) * zl, seriesContainer.innerHeight() / (bounds.bottom - bounds.top) * zl);
            var x = bounds.left + (bounds.right - bounds.left) / 2;
            var y = bounds.top + (bounds.bottom - bounds.top) / 2;
            var geoPoint = this.invert({ x: x, y: y });
            return this.zoomToGeoPoint(geoPoint, zoomLevel, true, duration);
        }
    });
    /**
     * Zooms the map to specific screen point.
     *
     * @param  point    Point
     * @param  level    Zoom level
     * @param  center   Center the map
     * @param  duration Duration of the animation in milliseconds
     */
    Object.defineProperty(MapChart.prototype, "zoomToPoint", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (point, level, center, duration) {
            if (level) {
                level = $math.fitToRange(level, this.get("minZoomLevel", 1), this.get("maxZoomLevel", 32));
            }
            if (!$type.isNumber(duration)) {
                duration = this.get("animationDuration", 0);
            }
            var easing = this.get("animationEasing");
            var zoomLevel = this.get("zoomLevel", 1);
            var x = point.x;
            var y = point.y;
            var tx = this.get("translateX", 0);
            var ty = this.get("translateY", 0);
            var cx = x;
            var cy = y;
            if (center) {
                cx = this.width() / 2;
                cy = this.height() / 2;
            }
            var xx = cx - ((x - tx) / zoomLevel * level);
            var yy = cy - ((y - ty) / zoomLevel * level);
            this._txa = this.animate({ key: "translateX", to: xx, duration: duration, easing: easing });
            this._tya = this.animate({ key: "translateY", to: yy, duration: duration, easing: easing });
            this._za = this.animate({ key: "zoomLevel", to: level, duration: duration, easing: easing });
            if (zoomLevel != level) {
                this._root.readerAlert(this._t("Zoom level changed to %1", this._root.locale, $type.numberToString(level)));
            }
            return this._za;
        }
    });
    /**
     * Zooms the map to specific geographical point.
     *
     * @param  geoPoint  Point
     * @param  level     Zoom level
     * @param  center    Center the map
     * @param  duration  Duration of the animation in milliseconds
     */
    Object.defineProperty(MapChart.prototype, "zoomToGeoPoint", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (geoPoint, level, center, duration) {
            var xy = this.convert(geoPoint);
            if (xy) {
                return this.zoomToPoint(xy, level, center, duration);
            }
        }
    });
    /**
     * Zooms the map in.
     */
    Object.defineProperty(MapChart.prototype, "zoomIn", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this.zoomToPoint({ x: this.width() / 2, y: this.height() / 2 }, this.get("zoomLevel", 1) * this.get("zoomStep", 2));
        }
    });
    /**
     * Zooms the map out.
     */
    Object.defineProperty(MapChart.prototype, "zoomOut", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this.zoomToPoint({ x: this.width() / 2, y: this.height() / 2 }, this.get("zoomLevel", 1) / this.get("zoomStep", 2));
        }
    });
    Object.defineProperty(MapChart.prototype, "_clearDirty", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._clearDirty.call(this);
            this._dirtyGeometries = false;
            this._mapFitted = false;
        }
    });
    Object.defineProperty(MapChart, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "MapChart"
    });
    Object.defineProperty(MapChart, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: SerialChart.classNames.concat([MapChart.className])
    });
    return MapChart;
}(SerialChart));
export { MapChart };
//# sourceMappingURL=MapChart.js.map