import { __extends, __read } from "tslib";
import { Slice } from "../../core/render/Slice";
import { AxisRenderer } from "../xy/axes/AxisRenderer";
import { AxisLabelRadial } from "../xy/axes/AxisLabelRadial";
import { p100 } from "../../core/util/Percent";
import { ListTemplate } from "../../core/util/List";
import { Template } from "../../core/util/Template";
import { arc } from "d3-shape";
import * as $utils from "../../core/util/Utils";
import * as $math from "../../core/util/Math";
/**
 * Renderer for circular axes.
 */
var AxisRendererCircular = /** @class */ (function (_super) {
    __extends(AxisRendererCircular, _super);
    function AxisRendererCircular() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        /**
         * A list of labels in the axis.
         *
         * `labels.template` can be used to configure labels.
         *
         * @default new ListTemplate<AxisLabelRadial>
         */
        Object.defineProperty(_this, "labels", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new ListTemplate(Template.new({}), function () { return AxisLabelRadial._new(_this._root, {
                themeTags: $utils.mergeTags(_this.labels.template.get("themeTags", []), _this.get("themeTags", []))
            }, [_this.labels.template]); })
        });
        /**
         * A list of fills in the axis.
         *
         * `axisFills.template` can be used to configure axis fills.
         *
         * @default new ListTemplate<Slice>
         */
        Object.defineProperty(_this, "axisFills", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new ListTemplate(Template.new({}), function () { return Slice._new(_this._root, {
                themeTags: $utils.mergeTags(_this.axisFills.template.get("themeTags", ["fill"]), _this.get("themeTags", []))
            }, [_this.axisFills.template]); })
        });
        Object.defineProperty(_this, "_fillGenerator", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: arc()
        });
        return _this;
    }
    Object.defineProperty(AxisRendererCircular.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._settings.themeTags = $utils.mergeTags(this._settings.themeTags, ["renderer", "circular"]);
            _super.prototype._afterNew.call(this);
            this.setPrivateRaw("letter", "X");
            this.setRaw("position", "absolute");
        }
    });
    Object.defineProperty(AxisRendererCircular.prototype, "_changed", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._changed.call(this);
            if (this.isDirty("radius") || this.isDirty("innerRadius") || this.isDirty("startAngle") || this.isDirty("endAngle")) {
                this.updateLayout();
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererCircular.prototype, "updateLayout", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            var chart = this.chart;
            if (chart) {
                var radius = chart.getPrivate("radius", 0);
                var r_1 = $utils.relativeToValue(this.get("radius", p100), radius);
                if (r_1 < 0) {
                    r_1 = radius + r_1;
                }
                this.setPrivate("radius", r_1);
                var ir = $utils.relativeToValue(this.get("innerRadius", chart.getPrivate("innerRadius", 0)), radius) * chart.getPrivate("irModifyer", 1);
                if (ir < 0) {
                    ir = r_1 + ir;
                }
                this.setPrivate("innerRadius", ir);
                var startAngle_1 = this.get("startAngle", chart.get("startAngle", -90));
                var endAngle_1 = this.get("endAngle", chart.get("endAngle", 270));
                this.setPrivate("startAngle", startAngle_1);
                this.setPrivate("endAngle", endAngle_1);
                this.set("draw", function (display) {
                    var _a;
                    var p0 = _this.positionToPoint(0);
                    display.moveTo(p0.x, p0.y);
                    if (startAngle_1 > endAngle_1) {
                        _a = __read([endAngle_1, startAngle_1], 2), startAngle_1 = _a[0], endAngle_1 = _a[1];
                    }
                    display.arc(0, 0, r_1, startAngle_1 * $math.RADIANS, endAngle_1 * $math.RADIANS);
                });
                this.axis.markDirtySize();
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererCircular.prototype, "updateGrid", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (grid, position, endPosition) {
            if (grid) {
                if (position == null) {
                    position = 0;
                }
                var location_1 = grid.get("location", 0.5);
                if (endPosition != null && endPosition != position) {
                    position = position + (endPosition - position) * location_1;
                }
                var radius_1 = this.getPrivate("radius", 0);
                var innerRadius_1 = this.getPrivate("innerRadius", 0);
                var angle_1 = this.positionToAngle(position);
                this.toggleVisibility(grid, position, 0, 1);
                if (radius_1 != null) {
                    grid.set("draw", function (display) {
                        display.moveTo(innerRadius_1 * $math.cos(angle_1), innerRadius_1 * $math.sin(angle_1));
                        display.lineTo(radius_1 * $math.cos(angle_1), radius_1 * $math.sin(angle_1));
                    });
                }
            }
        }
    });
    /**
     * Converts relative position to angle.
     *
     * @param   position  Position
     * @return            Angle
     */
    Object.defineProperty(AxisRendererCircular.prototype, "positionToAngle", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (position) {
            var axis = this.axis;
            var startAngle = this.getPrivate("startAngle", 0);
            var endAngle = this.getPrivate("endAngle", 360);
            var start = axis.get("start", 0);
            var end = axis.get("end", 1);
            var arc = (endAngle - startAngle) / (end - start);
            var angle;
            if (this.get("inversed")) {
                angle = startAngle + (end - position) * arc;
            }
            else {
                angle = startAngle + (position - start) * arc;
            }
            return angle;
        }
    });
    // do not delete
    Object.defineProperty(AxisRendererCircular.prototype, "_handleOpposite", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () { }
    });
    /**
     * Converts relative position to an X/Y coordinate.
     *
     * @param   position  Position
     * @return            Point
     */
    Object.defineProperty(AxisRendererCircular.prototype, "positionToPoint", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (position) {
            var radius = this.getPrivate("radius", 0);
            var angle = this.positionToAngle(position);
            return { x: radius * $math.cos(angle), y: radius * $math.sin(angle) };
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererCircular.prototype, "updateLabel", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (label, position, endPosition, count) {
            if (label) {
                if (position == null) {
                    position = 0;
                }
                var location_2 = 0.5;
                if (count != null && count > 1) {
                    location_2 = label.get("multiLocation", location_2);
                }
                else {
                    location_2 = label.get("location", location_2);
                }
                if (endPosition != null && endPosition != position) {
                    position = position + (endPosition - position) * location_2;
                }
                var radius = this.getPrivate("radius", 0);
                var innerRadius = this.getPrivate("innerRadius", 0);
                var angle = this.positionToAngle(position);
                label.setPrivate("radius", radius);
                label.setPrivate("innerRadius", innerRadius);
                label.set("labelAngle", angle);
                this.toggleVisibility(label, position, label.get("minPosition", 0), label.get("maxPosition", 1));
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererCircular.prototype, "fillDrawMethod", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (fill, startAngle, endAngle) {
            var _this = this;
            fill.set("draw", function (display) {
                if (startAngle == null) {
                    startAngle = _this.getPrivate("startAngle", 0);
                }
                if (endAngle == null) {
                    endAngle = _this.getPrivate("endAngle", 0);
                }
                var y0 = _this.getPrivate("innerRadius", 0);
                var y1 = _this.getPrivate("radius", 0);
                _this._fillGenerator.context(display);
                _this._fillGenerator({ innerRadius: y0, outerRadius: y1, startAngle: (startAngle + 90) * $math.RADIANS, endAngle: (endAngle + 90) * $math.RADIANS });
            });
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererCircular.prototype, "updateTick", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (tick, position, endPosition, count) {
            if (tick) {
                if (position == null) {
                    position = 0;
                }
                var location_3 = 0.5;
                if (count != null && count > 1) {
                    location_3 = tick.get("multiLocation", location_3);
                }
                else {
                    location_3 = tick.get("location", location_3);
                }
                if (endPosition != null && endPosition != position) {
                    position = position + (endPosition - position) * location_3;
                }
                var length_1 = tick.get("length", 0);
                var inside = tick.get("inside");
                if (inside) {
                    length_1 *= -1;
                }
                var radius_2 = this.getPrivate("radius", 0);
                var angle_2 = this.positionToAngle(position);
                this.toggleVisibility(tick, position, tick.get("minPosition", 0), tick.get("maxPosition", 1));
                if (radius_2 != null) {
                    tick.set("draw", function (display) {
                        display.moveTo(radius_2 * $math.cos(angle_2), radius_2 * $math.sin(angle_2));
                        radius_2 += length_1;
                        display.lineTo(radius_2 * $math.cos(angle_2), radius_2 * $math.sin(angle_2));
                    });
                }
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererCircular.prototype, "updateBullet", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (bullet, position, endPosition) {
            if (bullet) {
                var sprite = bullet.get("sprite");
                if (sprite) {
                    if (position == null) {
                        position = 0;
                    }
                    var location_4 = bullet.get("location", 0.5);
                    if (endPosition != null && endPosition != position) {
                        position = position + (endPosition - position) * location_4;
                    }
                    var radius = this.getPrivate("radius", 0);
                    var angle = this.positionToAngle(position);
                    this.toggleVisibility(sprite, position, 0, 1);
                    sprite.setAll({ rotation: angle, x: radius * $math.cos(angle), y: radius * $math.sin(angle) });
                }
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererCircular.prototype, "updateFill", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (fill, position, endPosition) {
            if (fill) {
                if (position == null) {
                    position = 0;
                }
                if (endPosition == null) {
                    endPosition = 1;
                }
                var startAngle = this.fitAngle(this.positionToAngle(position));
                var endAngle = this.fitAngle(this.positionToAngle(endPosition));
                fill.setAll({ startAngle: startAngle, arc: endAngle - startAngle });
                fill._setSoft("innerRadius", this.getPrivate("innerRadius"));
                fill._setSoft("radius", this.getPrivate("radius"));
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererCircular.prototype, "fitAngle", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (angle) {
            var startAngle = this.getPrivate("startAngle", 0);
            var endAngle = this.getPrivate("endAngle", 0);
            var minAngle = Math.min(startAngle, endAngle);
            var maxAngle = Math.max(startAngle, endAngle);
            if (angle < minAngle) {
                angle = minAngle;
            }
            if (angle > maxAngle) {
                angle = maxAngle;
            }
            return angle;
        }
    });
    /**
     * Returns axis length in pixels.
     *
     * @return Length
     */
    Object.defineProperty(AxisRendererCircular.prototype, "axisLength", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return Math.abs(this.getPrivate("radius", 0) * Math.PI * 2 * (this.getPrivate("endAngle", 360) - this.getPrivate("startAngle", 0)) / 360);
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererCircular.prototype, "positionTooltip", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (tooltip, position) {
            var radius = this.getPrivate("radius", 0);
            var angle = this.positionToAngle(position);
            //return tooltip.set("pointTo", this.axis._display.toGlobal({ x: radius * $math.cos(angle), y: radius * $math.sin(angle) }));
            this._positionTooltip(tooltip, { x: radius * $math.cos(angle), y: radius * $math.sin(angle) });
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererCircular.prototype, "updateTooltipBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_tooltip) {
        }
    });
    Object.defineProperty(AxisRendererCircular, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "AxisRendererCircular"
    });
    Object.defineProperty(AxisRendererCircular, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: AxisRenderer.classNames.concat([AxisRendererCircular.className])
    });
    return AxisRendererCircular;
}(AxisRenderer));
export { AxisRendererCircular };
//# sourceMappingURL=AxisRendererCircular.js.map