import { __extends, __read } from "tslib";
import { XYCursor } from "../xy/XYCursor";
import { p100 } from "../../core/util/Percent";
import { arc } from "d3-shape";
import * as $math from "../../core/util/Math";
import * as $utils from "../../core/util/Utils";
/**
 * Creates a cursor for a [[RadarChart]].
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/radar-chart/#Cursor} for more info
 */
var RadarCursor = /** @class */ (function (_super) {
    __extends(RadarCursor, _super);
    function RadarCursor() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "_fillGenerator", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: arc()
        });
        return _this;
    }
    Object.defineProperty(RadarCursor.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._settings.themeTags = $utils.mergeTags(this._settings.themeTags, ["radar", "cursor"]);
            _super.prototype._afterNew.call(this);
        }
    });
    Object.defineProperty(RadarCursor.prototype, "_handleXLine", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
        }
    });
    Object.defineProperty(RadarCursor.prototype, "_handleYLine", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
        }
    });
    Object.defineProperty(RadarCursor.prototype, "_getPosition", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (point) {
            var radius = Math.hypot(point.x, point.y);
            var angle = $math.normalizeAngle(Math.atan2(point.y, point.x) * $math.DEGREES);
            var innerRadius = this.getPrivate("innerRadius");
            var startAngle = $math.normalizeAngle(this.getPrivate("startAngle"));
            var endAngle = $math.normalizeAngle(this.getPrivate("endAngle"));
            if (endAngle < startAngle || endAngle == startAngle) {
                if (angle < startAngle) {
                    angle += 360;
                }
                endAngle = endAngle + 360;
            }
            var xPos = (angle - startAngle) / (endAngle - startAngle);
            if (xPos < 0) {
                xPos = 1 + xPos;
            }
            return { x: xPos, y: (radius - innerRadius) / (this.getPrivate("radius") - innerRadius) };
        }
    });
    Object.defineProperty(RadarCursor.prototype, "_getPoint", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (positionX, positionY) {
            var innerRadius = this.getPrivate("innerRadius");
            var startAngle = this.getPrivate("startAngle");
            var endAngle = this.getPrivate("endAngle");
            var radius = this.getPrivate("radius");
            var angle = startAngle + positionX * (endAngle - startAngle);
            var r = innerRadius + (radius - innerRadius) * positionY;
            return { x: r * $math.cos(angle), y: r * $math.sin(angle) };
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(RadarCursor.prototype, "updateLayout", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var chart = this.chart;
            if (chart) {
                var radius = chart.getPrivate("radius", 0);
                this.setPrivate("radius", $utils.relativeToValue(this.get("radius", p100), radius));
                var innerRadius = $utils.relativeToValue(this.get("innerRadius", chart.getPrivate("innerRadius", 0)), radius);
                if (innerRadius < 0) {
                    innerRadius = radius + innerRadius;
                }
                this.setPrivate("innerRadius", innerRadius);
                var startAngle = this.get("startAngle", chart.get("startAngle", -90));
                var endAngle = this.get("endAngle", chart.get("endAngle", 270));
                this.setPrivate("startAngle", startAngle);
                this.setPrivate("endAngle", endAngle);
            }
        }
    });
    Object.defineProperty(RadarCursor.prototype, "_updateLines", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (x, y) {
            if (!this._tooltipX) {
                this._drawXLine(x, y);
            }
            if (!this._tooltipY) {
                this._drawYLine(x, y);
            }
        }
    });
    Object.defineProperty(RadarCursor.prototype, "_drawXLine", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (x, y) {
            var innerRadius = this.getPrivate("innerRadius");
            var radius = this.getPrivate("radius");
            var angle = Math.atan2(y, x);
            this.lineX.set("draw", function (display) {
                display.moveTo(innerRadius * Math.cos(angle), innerRadius * Math.sin(angle));
                display.lineTo(radius * Math.cos(angle), radius * Math.sin(angle));
            });
        }
    });
    Object.defineProperty(RadarCursor.prototype, "_drawYLine", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (x, y) {
            var _this = this;
            var positionRadius = Math.hypot(x, y);
            this.lineY.set("draw", function (display) {
                display.arc(0, 0, positionRadius, _this.getPrivate("startAngle", 0) * $math.RADIANS, _this.getPrivate("endAngle", 0) * $math.RADIANS);
            });
        }
    });
    Object.defineProperty(RadarCursor.prototype, "_updateXLine", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (tooltip) {
            var point = tooltip.get("pointTo");
            if (point) {
                point = this._display.toLocal(point);
                this._drawXLine(point.x, point.y);
            }
        }
    });
    Object.defineProperty(RadarCursor.prototype, "_updateYLine", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (tooltip) {
            var point = tooltip.get("pointTo");
            if (point) {
                point = this._display.toLocal(point);
                this._drawYLine(point.x, point.y);
            }
        }
    });
    Object.defineProperty(RadarCursor.prototype, "_inPlot", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (point) {
            var chart = this.chart;
            if (chart) {
                return chart.inPlot(point, this.getPrivate("radius"), this.getPrivate("innerRadius"));
            }
            return false;
        }
    });
    Object.defineProperty(RadarCursor.prototype, "_updateSelection", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (point) {
            var _this = this;
            this.selection.set("draw", function (display) {
                var _a;
                var behavior = _this.get("behavior");
                var downPoint = _this._downPoint;
                var cursorStartAngle = _this.getPrivate("startAngle");
                var cursorEndAngle = _this.getPrivate("endAngle");
                var cursorRadius = _this.getPrivate("radius");
                var cursorInnerRadius = _this.getPrivate("innerRadius");
                if (cursorRadius < cursorInnerRadius) {
                    _a = __read([cursorInnerRadius, cursorRadius], 2), cursorRadius = _a[0], cursorInnerRadius = _a[1];
                }
                var startAngle = cursorStartAngle;
                var endAngle = cursorEndAngle;
                var radius = cursorRadius;
                var innerRadius = cursorInnerRadius;
                if (downPoint) {
                    if (behavior == "zoomXY" || behavior == "selectXY") {
                        startAngle = Math.atan2(downPoint.y, downPoint.x) * $math.DEGREES;
                        endAngle = Math.atan2(point.y, point.x) * $math.DEGREES;
                        innerRadius = Math.hypot(downPoint.x, downPoint.y);
                        radius = Math.hypot(point.x, point.y);
                    }
                    else if (behavior == "zoomX" || behavior == "selectX") {
                        startAngle = Math.atan2(downPoint.y, downPoint.x) * $math.DEGREES;
                        endAngle = Math.atan2(point.y, point.x) * $math.DEGREES;
                    }
                    else if (behavior == "zoomY" || behavior == "selectY") {
                        innerRadius = Math.hypot(downPoint.x, downPoint.y);
                        radius = Math.hypot(point.x, point.y);
                    }
                }
                innerRadius = $math.fitToRange(innerRadius, cursorInnerRadius, cursorRadius);
                radius = $math.fitToRange(radius, cursorInnerRadius, cursorRadius);
                startAngle = $math.fitAngleToRange(startAngle, cursorStartAngle, cursorEndAngle);
                endAngle = $math.fitAngleToRange(endAngle, cursorStartAngle, cursorEndAngle);
                if (startAngle == endAngle) {
                    endAngle = startAngle + 360;
                }
                startAngle *= $math.RADIANS;
                endAngle *= $math.RADIANS;
                _this._fillGenerator.context(display);
                _this._fillGenerator({ innerRadius: innerRadius, outerRadius: radius, startAngle: startAngle + Math.PI / 2, endAngle: endAngle + Math.PI / 2 });
            });
        }
    });
    Object.defineProperty(RadarCursor, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "RadarCursor"
    });
    Object.defineProperty(RadarCursor, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: XYCursor.classNames.concat([RadarCursor.className])
    });
    return RadarCursor;
}(XYCursor));
export { RadarCursor };
//# sourceMappingURL=RadarCursor.js.map