import { __extends, __read } from "tslib";
import { XYChartDefaultTheme } from "./XYChartDefaultTheme";
import { Container } from "../../core/render/Container";
import { Rectangle } from "../../core/render/Rectangle";
import { SerialChart } from "../../core/render/SerialChart";
import { ListAutoDispose } from "../../core/util/List";
import { p100 } from "../../core/util/Percent";
import { Color } from "../../core/util/Color";
import { Button } from "../../core/render/Button";
import { Graphics } from "../../core/render/Graphics";
import { Percent } from "../../core/util/Percent";
import * as $array from "../../core/util/Array";
import * as $order from "../../core/util/Order";
import * as $type from "../../core/util/Type";
/**
 * Creates an XY chart.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/xy-chart/} for more info
 * @important
 */
var XYChart = /** @class */ (function (_super) {
    __extends(XYChart, _super);
    function XYChart() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        /**
         * A list of horizontal axes.
         */
        Object.defineProperty(_this, "xAxes", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new ListAutoDispose()
        });
        /**
         * A list of vertical axes.
         */
        Object.defineProperty(_this, "yAxes", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new ListAutoDispose()
        });
        /**
         * A [[Container]] located on top of the chart, used to store top horizontal
         * axes.
         *
         * @see {@link https://www.amcharts.com/docs/v5/charts/xy-chart/xy-chart-containers/} for more info
         * @default Container.new()
         */
        Object.defineProperty(_this, "topAxesContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.chartContainer.children.push(Container.new(_this._root, { width: p100, layout: _this._root.verticalLayout }))
        });
        /**
         * A [[Container]] located in the middle the chart, used to store vertical axes
         * and plot area container.
         *
         * @see {@link https://www.amcharts.com/docs/v5/charts/xy-chart/xy-chart-containers/} for more info
         * @default Container.new()
         */
        Object.defineProperty(_this, "yAxesAndPlotContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.chartContainer.children.push(Container.new(_this._root, { width: p100, height: p100, layout: _this._root.horizontalLayout }))
        });
        /**
         * A [[Container]] located on bottom of the chart, used to store bottom
         * horizontal axes.
         *
         * @see {@link https://www.amcharts.com/docs/v5/charts/xy-chart/xy-chart-containers/} for more info
         * @default Container.new()
         */
        Object.defineProperty(_this, "bottomAxesContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.chartContainer.children.push(Container.new(_this._root, { width: p100, layout: _this._root.verticalLayout }))
        });
        /**
         * A [[Container]] located on left of the chart, used to store left-hand
         * vertical axes.
         *
         * @see {@link https://www.amcharts.com/docs/v5/charts/xy-chart/xy-chart-containers/} for more info
         * @default Container.new()
         */
        Object.defineProperty(_this, "leftAxesContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.yAxesAndPlotContainer.children.push(Container.new(_this._root, { height: p100, layout: _this._root.horizontalLayout }))
        });
        /**
         * A [[Container]] located in the middle of the chart, used to store actual
         * plots (series).
         *
         * NOTE: `plotContainer` will automatically have its `background` preset. If
         * you need to modify background or outline for chart's plot area, you can
         * use `plotContainer.get("background")` for that.
         *
         * @see {@link https://www.amcharts.com/docs/v5/charts/xy-chart/xy-chart-containers/} for more info
         * @default Container.new()
         */
        Object.defineProperty(_this, "plotContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.yAxesAndPlotContainer.children.push(Container.new(_this._root, { width: p100, height: p100, maskContent: false }))
        });
        /**
         * A [[Container]] axis grid elements are stored in.
         *
         * @see {@link https://www.amcharts.com/docs/v5/charts/xy-chart/xy-chart-containers/} for more info
         * @default Container.new()
         */
        Object.defineProperty(_this, "gridContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.plotContainer.children.push(Container.new(_this._root, { width: p100, height: p100, isMeasured: false }))
        });
        /**
         * A [[Container]] axis background grid elements are stored in.
         *
         * @see {@link https://www.amcharts.com/docs/v5/charts/xy-chart/xy-chart-containers/} for more info
         * @default Container.new()
         */
        Object.defineProperty(_this, "topGridContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: Container.new(_this._root, { width: p100, height: p100, isMeasured: false })
        });
        /**
         * A [[Container]] located on right of the chart, used to store right-hand
         * vertical axes.
         *
         * @see {@link https://www.amcharts.com/docs/v5/charts/xy-chart/xy-chart-containers/} for more info
         * @default Container.new()
         */
        Object.defineProperty(_this, "rightAxesContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.yAxesAndPlotContainer.children.push(Container.new(_this._root, { height: p100, layout: _this._root.horizontalLayout }))
        });
        /**
         * A [[Container]] axis headers are stored in.
         *
         * @see {@link https://www.amcharts.com/docs/v5/charts/xy-chart/axes/axis-headers/} for more info
         * @default Container.new()
         */
        Object.defineProperty(_this, "axisHeadersContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.plotContainer.children.push(Container.new(_this._root, {}))
        });
        /**
         * A button that is shown when chart is not fully zoomed out.
         *
         * @see {@link https://www.amcharts.com/docs/v5/charts/xy-chart/zoom-and-pan/#Zoom_out_button} for more info
         * @default Button.new()
         */
        Object.defineProperty(_this, "zoomOutButton", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.plotContainer.children.push(Button.new(_this._root, {
                themeTags: ["zoom"],
                icon: Graphics.new(_this._root, {
                    themeTags: ["button", "icon"]
                })
            }))
        });
        Object.defineProperty(_this, "_movePoint", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: { x: 0, y: 0 }
        });
        Object.defineProperty(_this, "_wheelDp", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_otherCharts", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        return _this;
    }
    Object.defineProperty(XYChart.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            this._defaultThemes.push(XYChartDefaultTheme.new(this._root));
            _super.prototype._afterNew.call(this);
            this._disposers.push(this.xAxes);
            this._disposers.push(this.yAxes);
            var root = this._root;
            var verticalLayout = this._root.verticalLayout;
            var zoomOutButton = this.zoomOutButton;
            zoomOutButton.events.on("click", function () {
                _this.zoomOut();
            });
            zoomOutButton.set("opacity", 0);
            zoomOutButton.states.lookup("default").set("opacity", 1);
            this.chartContainer.set("layout", verticalLayout);
            var plotContainer = this.plotContainer;
            plotContainer.children.push(this.seriesContainer);
            this._disposers.push(this._processAxis(this.xAxes, this.bottomAxesContainer));
            this._disposers.push(this._processAxis(this.yAxes, this.leftAxesContainer));
            plotContainer.children.push(this.topGridContainer);
            plotContainer.children.push(this.bulletsContainer);
            plotContainer.children.moveValue(zoomOutButton);
            // Setting trasnparent background so that full body of the plot container
            // is interactive
            plotContainer.set("interactive", true);
            plotContainer.set("interactiveChildren", false);
            plotContainer.set("background", Rectangle.new(root, {
                themeTags: ["xy", "background"],
                fill: Color.fromHex(0x000000),
                fillOpacity: 0
            }));
            this._disposers.push(plotContainer.events.on("pointerdown", function (event) {
                _this._handlePlotDown(event.originalEvent);
            }));
            this._disposers.push(plotContainer.events.on("globalpointerup", function (event) {
                _this._handlePlotUp(event.originalEvent);
            }));
            this._disposers.push(plotContainer.events.on("globalpointermove", function (event) {
                _this._handlePlotMove(event.originalEvent);
            }));
        }
    });
    Object.defineProperty(XYChart.prototype, "_removeSeries", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (series) {
            var xAxis = series.get("xAxis");
            if (xAxis) {
                $array.remove(xAxis.series, series);
            }
            var yAxis = series.get("yAxis");
            if (yAxis) {
                $array.remove(yAxis.series, series);
            }
            _super.prototype._removeSeries.call(this, series);
        }
    });
    Object.defineProperty(XYChart.prototype, "_handleSetWheel", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            var wheelX = this.get("wheelX");
            var wheelY = this.get("wheelY");
            var plotContainer = this.plotContainer;
            if (wheelX !== "none" || wheelY !== "none") {
                this._wheelDp = plotContainer.events.on("wheel", function (event) {
                    var wheelEvent = event.originalEvent;
                    wheelEvent.preventDefault();
                    var plotPoint = plotContainer.toLocal(_this._root.documentPointToRoot({ x: wheelEvent.clientX, y: wheelEvent.clientY }));
                    var wheelStep = _this.get("wheelStep", 0.2);
                    var shiftY = wheelEvent.deltaY / 100;
                    var shiftX = wheelEvent.deltaX / 100;
                    if ((wheelX === "zoomX" || wheelX === "zoomXY") && shiftX != 0) {
                        _this.xAxes.each(function (axis) {
                            if (axis.get("zoomX")) {
                                var start = axis.get("start");
                                var end = axis.get("end");
                                var position = axis.fixPosition(plotPoint.x / plotContainer.width());
                                var newStart = start - wheelStep * (end - start) * shiftX * position;
                                var newEnd = end + wheelStep * (end - start) * shiftX * (1 - position);
                                if (1 / (newEnd - newStart) < axis.get("maxZoomFactor", Infinity)) {
                                    _this._handleWheelAnimation(axis.zoom(newStart, newEnd));
                                }
                            }
                        });
                    }
                    if ((wheelY === "zoomX" || wheelY === "zoomXY") && shiftY != 0) {
                        _this.xAxes.each(function (axis) {
                            if (axis.get("zoomX")) {
                                var start = axis.get("start");
                                var end = axis.get("end");
                                var position = axis.fixPosition(plotPoint.x / plotContainer.width());
                                var newStart = start - wheelStep * (end - start) * shiftY * position;
                                var newEnd = end + wheelStep * (end - start) * shiftY * (1 - position);
                                if (1 / (newEnd - newStart) < axis.get("maxZoomFactor", Infinity)) {
                                    _this._handleWheelAnimation(axis.zoom(newStart, newEnd));
                                }
                            }
                        });
                    }
                    if ((wheelX === "zoomY" || wheelX === "zoomXY") && shiftX != 0) {
                        _this.yAxes.each(function (axis) {
                            if (axis.get("zoomY")) {
                                var start = axis.get("start");
                                var end = axis.get("end");
                                var position = axis.fixPosition(plotPoint.y / plotContainer.height());
                                var newStart = start - wheelStep * (end - start) * shiftX * position;
                                var newEnd = end + wheelStep * (end - start) * shiftX * (1 - position);
                                if (1 / (newEnd - newStart) < axis.get("maxZoomFactor", Infinity)) {
                                    _this._handleWheelAnimation(axis.zoom(newStart, newEnd));
                                }
                            }
                        });
                    }
                    if ((wheelY === "zoomY" || wheelY === "zoomXY") && shiftY != 0) {
                        _this.yAxes.each(function (axis) {
                            if (axis.get("zoomY")) {
                                var start = axis.get("start");
                                var end = axis.get("end");
                                var position = axis.fixPosition(plotPoint.y / plotContainer.height());
                                var newStart = start - wheelStep * (end - start) * shiftY * position;
                                var newEnd = end + wheelStep * (end - start) * shiftY * (1 - position);
                                if (1 / (newEnd - newStart) < axis.get("maxZoomFactor", Infinity)) {
                                    _this._handleWheelAnimation(axis.zoom(newStart, newEnd));
                                }
                            }
                        });
                    }
                    if ((wheelX === "panX" || wheelX === "panXY") && shiftX != 0) {
                        _this.xAxes.each(function (axis) {
                            if (axis.get("panX")) {
                                var start = axis.get("start");
                                var end = axis.get("end");
                                var position = axis.fixPosition(plotPoint.x / plotContainer.width());
                                var newStart = start + wheelStep * (end - start) * shiftX * position;
                                var newEnd = end + wheelStep * (end - start) * shiftX * (1 - position);
                                _this._handleWheelAnimation(axis.zoom(newStart, newEnd));
                            }
                        });
                    }
                    if ((wheelY === "panX" || wheelY === "panXY") && shiftY != 0) {
                        _this.xAxes.each(function (axis) {
                            if (axis.get("panX")) {
                                var start = axis.get("start");
                                var end = axis.get("end");
                                var position = axis.fixPosition(plotPoint.x / plotContainer.width());
                                var newStart = start + wheelStep * (end - start) * shiftY * position;
                                var newEnd = end + wheelStep * (end - start) * shiftY * (1 - position);
                                _this._handleWheelAnimation(axis.zoom(newStart, newEnd));
                            }
                        });
                    }
                    if ((wheelX === "panY" || wheelX === "panXY") && shiftX != 0) {
                        _this.yAxes.each(function (axis) {
                            if (axis.get("panY")) {
                                var start = axis.get("start");
                                var end = axis.get("end");
                                var position = axis.fixPosition(plotPoint.y / plotContainer.height());
                                var newStart = start + wheelStep * (end - start) * shiftX * position;
                                var newEnd = end + wheelStep * (end - start) * shiftX * (1 - position);
                                _this._handleWheelAnimation(axis.zoom(newStart, newEnd));
                            }
                        });
                    }
                    if ((wheelY === "panY" || wheelY === "panXY") && shiftY != 0) {
                        _this.yAxes.each(function (axis) {
                            if (axis.get("panY")) {
                                var start = axis.get("start");
                                var end = axis.get("end");
                                var position = axis.fixPosition(plotPoint.y / plotContainer.height());
                                var newStart = start + wheelStep * (end - start) * shiftY * position;
                                var newEnd = end + wheelStep * (end - start) * shiftY * (1 - position);
                                _this._handleWheelAnimation(axis.zoom(newStart, newEnd));
                            }
                        });
                    }
                });
                this._disposers.push(this._wheelDp);
            }
            else {
                if (this._wheelDp) {
                    this._wheelDp.dispose();
                }
            }
        }
    });
    Object.defineProperty(XYChart.prototype, "_handlePlotDown", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (event) {
            // TODO: handle multitouch
            if (this.get("panX") || this.get("panY")) {
                var plotContainer = this.plotContainer;
                var local = plotContainer.toLocal(this._root.documentPointToRoot({ x: event.clientX, y: event.clientY }));
                if (local.x >= 0 && local.y >= 0 && local.x <= plotContainer.width() && local.y <= this.height()) {
                    this._downPoint = local;
                    var panX = this.get("panX");
                    var panY = this.get("panY");
                    if (panX) {
                        this.xAxes.each(function (axis) {
                            axis._panStart = axis.get("start");
                            axis._panEnd = axis.get("end");
                        });
                    }
                    if (panY) {
                        this.yAxes.each(function (axis) {
                            axis._panStart = axis.get("start");
                            axis._panEnd = axis.get("end");
                        });
                    }
                    var eventType = "panstarted";
                    if (this.events.isEnabled(eventType)) {
                        this.events.dispatch(eventType, { type: eventType, target: this });
                    }
                }
            }
        }
    });
    Object.defineProperty(XYChart.prototype, "_handleWheelAnimation", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (animation) {
            var _this = this;
            if (animation) {
                animation.events.on("stopped", function () {
                    _this._dispatchWheelAnimation();
                });
            }
            else {
                this._dispatchWheelAnimation();
            }
        }
    });
    Object.defineProperty(XYChart.prototype, "_dispatchWheelAnimation", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var eventType = "wheelended";
            if (this.events.isEnabled(eventType)) {
                this.events.dispatch(eventType, { type: eventType, target: this });
            }
        }
    });
    Object.defineProperty(XYChart.prototype, "_handlePlotUp", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_event) {
            if (this._downPoint) {
                if (this.get("panX") || this.get("panY")) {
                    var eventType = "panended";
                    if (this.events.isEnabled(eventType)) {
                        this.events.dispatch(eventType, { type: eventType, target: this });
                    }
                }
            }
            // TODO: handle multitouch
            this._downPoint = undefined;
            this.xAxes.each(function (xAxis) {
                xAxis._isPanning = false;
            });
            this.yAxes.each(function (yAxis) {
                yAxis._isPanning = false;
            });
        }
    });
    Object.defineProperty(XYChart.prototype, "_handlePlotMove", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (event) {
            // TODO: handle multitouch
            var downPoint = this._downPoint;
            if (downPoint) {
                var plotContainer_1 = this.plotContainer;
                var local_1 = plotContainer_1.toLocal(this._root.documentPointToRoot({ x: event.clientX, y: event.clientY }));
                var panX = this.get("panX");
                var panY = this.get("panY");
                if (panX) {
                    var scrollbarX = this.get("scrollbarX");
                    if (scrollbarX) {
                        scrollbarX.events.disableType("rangechanged");
                    }
                    this.xAxes.each(function (axis) {
                        if (axis.get("panX")) {
                            axis._isPanning = true;
                            //const maxDeviation = axis.get("maxDeviation", 0);
                            var panStart = axis._panStart;
                            var panEnd = axis._panEnd;
                            var difference = (panEnd - panStart);
                            var deltaX = difference * (downPoint.x - local_1.x) / plotContainer_1.width();
                            if (axis.get("renderer").get("inversed")) {
                                deltaX *= -1;
                            }
                            var start = panStart + deltaX;
                            var end = panEnd + deltaX;
                            if (end - start < 1 + axis.get("maxDeviation", 1) * 2) {
                                axis.set("start", start);
                                axis.set("end", end);
                            }
                        }
                    });
                    if (scrollbarX) {
                        scrollbarX.events.enableType("rangechanged");
                    }
                }
                if (panY) {
                    var scrollbarY = this.get("scrollbarY");
                    if (scrollbarY) {
                        scrollbarY.events.disableType("rangechanged");
                    }
                    this.yAxes.each(function (axis) {
                        if (axis.get("panY")) {
                            axis._isPanning = true;
                            //const maxDeviation = axis.get("maxDeviation", 0);
                            var panStart = axis._panStart;
                            var panEnd = axis._panEnd;
                            var difference = (panEnd - panStart);
                            var deltaY = difference * (downPoint.y - local_1.y) / plotContainer_1.height();
                            if (axis.get("renderer").get("inversed")) {
                                deltaY *= -1;
                            }
                            var start = panStart - deltaY;
                            var end = panEnd - deltaY;
                            if (end - start < 1 + axis.get("maxDeviation", 1) * 2) {
                                axis.set("start", start);
                                axis.set("end", end);
                            }
                        }
                    });
                    if (scrollbarY) {
                        scrollbarY.events.enableType("rangechanged");
                    }
                }
            }
        }
    });
    Object.defineProperty(XYChart.prototype, "_handleCursorPosition", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var cursor = this.get("cursor");
            if (cursor) {
                var cursorPoint_1 = cursor.getPrivate("point");
                var snapToSeries = cursor.get("snapToSeries");
                if (snapToSeries && cursorPoint_1) {
                    var snapToSeriesBy_1 = cursor.get("snapToSeriesBy");
                    var dataItems_1 = [];
                    $array.each(snapToSeries, function (series) {
                        if (!series.isHidden() && !series.isHiding()) {
                            if (snapToSeriesBy_1 != "x!" && snapToSeriesBy_1 != "y!") {
                                var startIndex = series.startIndex();
                                var endIndex = series.endIndex();
                                for (var i = startIndex; i < endIndex; i++) {
                                    var dataItem = series.dataItems[i];
                                    if (dataItem && !dataItem.isHidden()) {
                                        dataItems_1.push(dataItem);
                                    }
                                }
                            }
                            else {
                                var tooltipDataItem = series.get("tooltipDataItem");
                                if (tooltipDataItem) {
                                    dataItems_1.push(tooltipDataItem);
                                }
                            }
                        }
                    });
                    var minDistance_1 = Infinity;
                    var closestItem_1;
                    $array.each(dataItems_1, function (dataItem) {
                        var point = dataItem.get("point");
                        if (point) {
                            var distance = 0;
                            if (snapToSeriesBy_1 == "x" || snapToSeriesBy_1 == "x!") {
                                distance = Math.abs(cursorPoint_1.x - point.x);
                            }
                            else if (snapToSeriesBy_1 == "y" || snapToSeriesBy_1 == "y!") {
                                distance = Math.abs(cursorPoint_1.y - point.y);
                            }
                            else {
                                distance = Math.hypot(cursorPoint_1.x - point.x, cursorPoint_1.y - point.y);
                            }
                            if (distance < minDistance_1) {
                                minDistance_1 = distance;
                                closestItem_1 = dataItem;
                            }
                        }
                    });
                    $array.each(snapToSeries, function (series) {
                        var tooltip = series.get("tooltip");
                        if (tooltip) {
                            tooltip._setDataItem(undefined);
                        }
                    });
                    if (closestItem_1) {
                        var series = closestItem_1.component;
                        series.showDataItemTooltip(closestItem_1);
                        var point = closestItem_1.get("point");
                        if (point) {
                            cursor.handleMove(series.toGlobal(point), true);
                        }
                    }
                }
            }
        }
    });
    Object.defineProperty(XYChart.prototype, "_updateCursor", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var cursor = this.get("cursor");
            if (cursor) {
                cursor.handleMove();
            }
        }
    });
    Object.defineProperty(XYChart.prototype, "_addCursor", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (cursor) {
            this.plotContainer.children.push(cursor);
        }
    });
    Object.defineProperty(XYChart.prototype, "_prepareChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            _super.prototype._prepareChildren.call(this);
            this.series.each(function (series) {
                _this._colorize(series);
            });
            if (this.isDirty("wheelX") || this.isDirty("wheelY")) {
                this._handleSetWheel();
            }
            if (this.isDirty("cursor")) {
                var previous = this._prevSettings.cursor;
                var cursor = this.get("cursor");
                if (cursor !== previous) {
                    this._disposeProperty("cursor");
                    if (previous) {
                        previous.dispose();
                    }
                    if (cursor) {
                        cursor._setChart(this);
                        this._addCursor(cursor);
                        this._pushPropertyDisposer("cursor", cursor.events.on("selectended", function () {
                            _this._handleCursorSelectEnd();
                        }));
                    }
                    //this.setRaw("cursor", cursor) // to reset previous value
                    this._prevSettings.cursor = cursor;
                }
            }
            if (this.isDirty("scrollbarX")) {
                var previous = this._prevSettings.scrollbarX;
                var scrollbarX = this.get("scrollbarX");
                if (scrollbarX !== previous) {
                    this._disposeProperty("scrollbarX");
                    if (previous) {
                        previous.dispose();
                    }
                    if (scrollbarX) {
                        if (!scrollbarX.parent) {
                            this.topAxesContainer.children.push(scrollbarX);
                        }
                        this._pushPropertyDisposer("scrollbarX", scrollbarX.events.on("rangechanged", function (e) {
                            _this._handleScrollbar(_this.xAxes, e.start, e.end);
                        }));
                        // Used to populate `ariaLabel` with meaningful values
                        scrollbarX.setPrivate("positionTextFunction", function (position) {
                            var axis = _this.xAxes.getIndex(0);
                            return axis ? axis.getTooltipText(position) || "" : "";
                        });
                    }
                    this._prevSettings.scrollbarX = scrollbarX;
                }
            }
            if (this.isDirty("scrollbarY")) {
                var previous = this._prevSettings.scrollbarY;
                var scrollbarY = this.get("scrollbarY");
                if (scrollbarY !== previous) {
                    this._disposeProperty("scrollbarY");
                    if (previous) {
                        previous.dispose();
                    }
                    if (scrollbarY) {
                        if (!scrollbarY.parent) {
                            this.rightAxesContainer.children.push(scrollbarY);
                        }
                        this._pushPropertyDisposer("scrollbarY", scrollbarY.events.on("rangechanged", function (e) {
                            _this._handleScrollbar(_this.yAxes, e.start, e.end);
                        }));
                        // Used to populate `ariaLabel` with meaningful values
                        scrollbarY.setPrivate("positionTextFunction", function (position) {
                            var axis = _this.yAxes.getIndex(0);
                            return axis ? axis.getTooltipText(position) || "" : "";
                        });
                    }
                    this._prevSettings.scrollbarY = scrollbarY;
                }
            }
            this._handleZoomOut();
        }
    });
    Object.defineProperty(XYChart.prototype, "_processSeries", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (series) {
            _super.prototype._processSeries.call(this, series);
            this._colorize(series);
        }
    });
    Object.defineProperty(XYChart.prototype, "_colorize", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (series) {
            var colorSet = this.get("colors");
            if (colorSet) {
                if (series.get("fill") == null) {
                    var color = colorSet.next();
                    series._setSoft("stroke", color);
                    series._setSoft("fill", color);
                }
            }
        }
    });
    Object.defineProperty(XYChart.prototype, "_handleCursorSelectEnd", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var cursor = this.get("cursor");
            var behavior = cursor.get("behavior");
            var downPositionX = cursor.getPrivate("downPositionX", 0);
            var downPositionY = cursor.getPrivate("downPositionY", 0);
            var positionX = cursor.getPrivate("positionX", 0.5);
            var positionY = cursor.getPrivate("positionY", 0.5);
            this.xAxes.each(function (axis) {
                if (behavior === "zoomX" || behavior === "zoomXY") {
                    var position0 = axis.toAxisPosition(downPositionX);
                    var position1 = axis.toAxisPosition(positionX);
                    axis.zoom(position0, position1);
                }
                axis.setPrivate("updateScrollbar", true);
            });
            this.yAxes.each(function (axis) {
                if (behavior === "zoomY" || behavior === "zoomXY") {
                    var position0 = axis.toAxisPosition(downPositionY);
                    var position1 = axis.toAxisPosition(positionY);
                    axis.zoom(position0, position1);
                }
                axis.setPrivate("updateScrollbar", true);
            });
        }
    });
    Object.defineProperty(XYChart.prototype, "_handleScrollbar", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (axes, start, end) {
            axes.each(function (axis) {
                var axisStart = axis.fixPosition(start);
                var axisEnd = axis.fixPosition(end);
                var zoomAnimation = axis.zoom(axisStart, axisEnd);
                var updateScrollbar = "updateScrollbar";
                axis.setPrivateRaw(updateScrollbar, false);
                if (zoomAnimation) {
                    zoomAnimation.events.on("stopped", function () {
                        axis.setPrivateRaw(updateScrollbar, true);
                    });
                }
                else {
                    axis.setPrivateRaw(updateScrollbar, true);
                }
            });
        }
    });
    Object.defineProperty(XYChart.prototype, "_processAxis", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (axes, container) {
            var _this = this;
            return axes.events.onAll(function (change) {
                if (change.type === "clear") {
                    $array.each(change.oldValues, function (axis) {
                        _this._removeAxis(axis);
                    });
                }
                else if (change.type === "push") {
                    container.children.push(change.newValue);
                    change.newValue.processChart(_this);
                }
                else if (change.type === "setIndex") {
                    container.children.setIndex(change.index, change.newValue);
                    change.newValue.processChart(_this);
                }
                else if (change.type === "insertIndex") {
                    container.children.insertIndex(change.index, change.newValue);
                    change.newValue.processChart(_this);
                }
                else if (change.type === "removeIndex") {
                    _this._removeAxis(change.oldValue);
                }
                else if (change.type === "moveIndex") {
                    container.children.moveValue(change.value, change.newIndex);
                    change.value.processChart(_this);
                }
                else {
                    throw new Error("Unknown IListEvent type");
                }
            });
        }
    });
    Object.defineProperty(XYChart.prototype, "_removeAxis", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (axis) {
            if (!axis.isDisposed()) {
                var axisParent = axis.parent;
                if (axisParent) {
                    axisParent.children.removeValue(axis);
                }
                var gridContainer = axis.gridContainer;
                var gridParent = gridContainer.parent;
                if (gridParent) {
                    gridParent.children.removeValue(gridContainer);
                }
                var topGridContainer = axis.topGridContainer;
                var topGridParent = topGridContainer.parent;
                if (topGridParent) {
                    topGridParent.children.removeValue(topGridContainer);
                }
            }
        }
    });
    Object.defineProperty(XYChart.prototype, "_updateChartLayout", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var left = this.leftAxesContainer.width();
            var right = this.rightAxesContainer.width();
            var bottomAxesContainer = this.bottomAxesContainer;
            bottomAxesContainer.set("paddingLeft", left);
            bottomAxesContainer.set("paddingRight", right);
            var topAxesContainer = this.topAxesContainer;
            topAxesContainer.set("paddingLeft", left);
            topAxesContainer.set("paddingRight", right);
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(XYChart.prototype, "processAxis", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_axis) { }
    });
    ;
    Object.defineProperty(XYChart.prototype, "_handleAxisSelection", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (axis) {
            var _a;
            var start = axis.fixPosition(axis.get("start", 0));
            var end = axis.fixPosition(axis.get("end", 1));
            if (start > end) {
                _a = __read([end, start], 2), start = _a[0], end = _a[1];
            }
            if (this.xAxes.indexOf(axis) != -1) {
                if (axis.getPrivate("updateScrollbar")) {
                    var scrollbarX = this.get("scrollbarX");
                    if (scrollbarX && !scrollbarX.getPrivate("isBusy")) {
                        scrollbarX.setRaw("start", start);
                        scrollbarX.setRaw("end", end);
                        scrollbarX.updateGrips();
                    }
                }
            }
            else if (this.yAxes.indexOf(axis) != -1) {
                if (axis.getPrivate("updateScrollbar")) {
                    var scrollbarY = this.get("scrollbarY");
                    if (scrollbarY && !scrollbarY.getPrivate("isBusy")) {
                        scrollbarY.setRaw("start", start);
                        scrollbarY.setRaw("end", end);
                        scrollbarY.updateGrips();
                    }
                }
            }
            this._handleZoomOut();
        }
    });
    Object.defineProperty(XYChart.prototype, "_handleZoomOut", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var zoomOutButton = this.zoomOutButton;
            if (zoomOutButton && zoomOutButton.parent) {
                var visible_1 = false;
                this.xAxes.each(function (axis) {
                    if (axis.get("start") != 0 || axis.get("end") != 1) {
                        visible_1 = true;
                    }
                });
                this.yAxes.each(function (axis) {
                    if (axis.get("start") != 0 || axis.get("end") != 1) {
                        visible_1 = true;
                    }
                });
                if (visible_1) {
                    if (zoomOutButton.isHidden()) {
                        zoomOutButton.show();
                    }
                }
                else {
                    zoomOutButton.hide();
                }
            }
        }
    });
    /**
     * Checks if point is within plot area.
     *
     * @param   point  Reference point
     * @return         Is within plot area?
     */
    Object.defineProperty(XYChart.prototype, "inPlot", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (point) {
            var plotContainer = this.plotContainer;
            var otherCharts = this._otherCharts;
            var global = plotContainer.toGlobal(point);
            if (point.x >= -0.1 && point.y >= -0.1 && point.x <= plotContainer.width() + 0.1 && point.y <= plotContainer.height() + 0.1) {
                return true;
            }
            if (otherCharts) {
                for (var i = otherCharts.length - 1; i >= 0; i--) {
                    var chart = otherCharts[i];
                    if (chart != this) {
                        var chartPlotContainer = chart.plotContainer;
                        var documentPoint = this._root.rootPointToDocument(global);
                        var chartRoot = chart._root.documentPointToRoot(documentPoint);
                        var local = chartPlotContainer.toLocal(chartRoot);
                        if (local.x >= -0.1 && local.y >= -0.1 && local.x <= chartPlotContainer.width() + 0.1 && local.y <= chartPlotContainer.height() + 0.1) {
                            return true;
                        }
                    }
                }
            }
            return false;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(XYChart.prototype, "arrangeTooltips", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            var plotContainer = this.plotContainer;
            var w = plotContainer.width();
            var h = plotContainer.height();
            var plotT = plotContainer._display.toGlobal({ x: 0, y: 0 });
            var plotB = plotContainer._display.toGlobal({ x: w, y: h });
            var tooltips = [];
            var sum = 0;
            var minDistance = Infinity;
            var movePoint = this._movePoint;
            var maxTooltipDistance = this.get("maxTooltipDistance");
            var closest;
            var closestPoint;
            if ($type.isNumber(maxTooltipDistance)) {
                this.series.each(function (series) {
                    var tooltip = series.get("tooltip");
                    if (tooltip) {
                        var point = tooltip.get("pointTo");
                        if (point) {
                            var distance = Math.hypot(movePoint.x - point.x, movePoint.y - point.y);
                            if (distance < minDistance) {
                                minDistance = distance;
                                closest = series;
                                closestPoint = point;
                            }
                        }
                    }
                });
            }
            var tooltipSeries = [];
            this.series.each(function (series) {
                var tooltip = series.get("tooltip");
                if (tooltip) {
                    var hidden = false;
                    var point = tooltip.get("pointTo");
                    if (point) {
                        if (maxTooltipDistance >= 0) {
                            var point_1 = tooltip.get("pointTo");
                            if (point_1) {
                                if (series != closest) {
                                    var distance = Math.hypot(closestPoint.x - point_1.x, closestPoint.y - point_1.y);
                                    if (distance > maxTooltipDistance) {
                                        hidden = true;
                                    }
                                }
                            }
                        }
                        else if (maxTooltipDistance == -1) {
                            if (series != closest) {
                                hidden = true;
                            }
                        }
                        if (!_this.inPlot(_this._tooltipToLocal(point)) || !tooltip.dataItem) {
                            hidden = true;
                        }
                        else {
                            if (!hidden) {
                                sum += point.y;
                            }
                        }
                        if (hidden || series.isHidden() || series.isHiding()) {
                            tooltip.hide(0);
                        }
                        else {
                            tooltip.show();
                            tooltips.push(tooltip);
                            tooltipSeries.push(series);
                        }
                    }
                }
            });
            this.setPrivate("tooltipSeries", tooltipSeries);
            if (this.get("arrangeTooltips")) {
                var tooltipContainer_1 = this._root.tooltipContainer;
                tooltips.sort(function (a, b) { return $order.compareNumber(a.get("pointTo").y, b.get("pointTo").y); });
                var count = tooltips.length;
                var average = sum / count;
                if (average > h / 2 + plotT.y) {
                    tooltips.reverse();
                    var prevY_1 = plotB.y;
                    $array.each(tooltips, function (tooltip) {
                        var height = tooltip.height();
                        var centerY = tooltip.get("centerY");
                        if (centerY instanceof Percent) {
                            height *= centerY.value;
                        }
                        height += tooltip.get("marginBottom", 0);
                        tooltip.set("bounds", { left: plotT.x, top: plotT.y, right: plotB.x, bottom: prevY_1 });
                        prevY_1 = Math.min(prevY_1 - height, tooltip._fy - height);
                        tooltipContainer_1.children.moveValue(tooltip, 0);
                    });
                }
                else {
                    var prevY_2 = 0;
                    $array.each(tooltips, function (tooltip) {
                        var height = tooltip.height();
                        var centerY = tooltip.get("centerY");
                        if (centerY instanceof Percent) {
                            height *= centerY.value;
                        }
                        height += tooltip.get("marginBottom", 0);
                        tooltip.set("bounds", { left: plotT.x, top: prevY_2, right: plotB.x, bottom: Math.max(plotT.y + h, prevY_2 + height) });
                        tooltipContainer_1.children.moveValue(tooltip, 0);
                        prevY_2 = Math.max(prevY_2 + height, tooltip._fy + height);
                    });
                }
            }
        }
    });
    Object.defineProperty(XYChart.prototype, "_tooltipToLocal", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (point) {
            return this.plotContainer.toLocal(point);
        }
    });
    /**
     * Fully zooms out the chart.
     */
    Object.defineProperty(XYChart.prototype, "zoomOut", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this.xAxes.each(function (axis) {
                axis.setPrivate("updateScrollbar", true);
                axis.zoom(0, 1);
            });
            this.yAxes.each(function (axis) {
                axis.setPrivate("updateScrollbar", true);
                axis.zoom(0, 1);
            });
        }
    });
    Object.defineProperty(XYChart, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "XYChart"
    });
    Object.defineProperty(XYChart, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: SerialChart.classNames.concat([XYChart.className])
    });
    return XYChart;
}(SerialChart));
export { XYChart };
//# sourceMappingURL=XYChart.js.map