import { __extends } from "tslib";
import { Container } from "../../core/render/Container";
import { p100 } from "../../core/util/Percent";
import { Graphics } from "../../core/render/Graphics";
import { Grid } from "./axes/Grid";
//import { Animations } from "../core/util/Animation";
import * as $type from "../../core/util/Type";
import * as $utils from "../../core/util/Utils";
import * as $math from "../../core/util/Math";
import * as $array from "../../core/util/Array";
import * as $object from "../../core/util/Object";
/**
 * Creates a chart cursor for an [[XYChart]].
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/xy-chart/cursor/} for more info
 * @important
 */
var XYCursor = /** @class */ (function (_super) {
    __extends(XYCursor, _super);
    function XYCursor() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        /**
         * A [[Grid]] elment that used for horizontal line of the cursor crosshair.
         *
         * @default Grid.new()
         */
        Object.defineProperty(_this, "lineX", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.children.push(Grid.new(_this._root, {
                themeTags: ["x"]
            }))
        });
        /**
         * A [[Grid]] elment that used for horizontal line of the cursor crosshair.
         *
         * @default Grid.new()
         */
        Object.defineProperty(_this, "lineY", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.children.push(Grid.new(_this._root, {
                themeTags: ["y"]
            }))
        });
        /**
         * An element that represents current selection.
         *
         * @default Graphics.new()
         */
        Object.defineProperty(_this, "selection", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.children.push(Graphics.new(_this._root, {
                themeTags: ["selection", "cursor"], layer: 30
            }))
        });
        Object.defineProperty(_this, "_movePoint", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_lastPoint", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: { x: 0, y: 0 }
        });
        Object.defineProperty(_this, "_tooltipX", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(_this, "_tooltipY", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        /**
         * A chart cursor is attached to.
         */
        Object.defineProperty(_this, "chart", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_toX", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_toY", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        return _this;
    }
    Object.defineProperty(XYCursor.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            this._settings.themeTags = $utils.mergeTags(this._settings.themeTags, ["xy", "cursor"]);
            _super.prototype._afterNew.call(this);
            this.setAll({ "width": p100, height: p100, isMeasured: true, position: "absolute" });
            this.states.create("hidden", { visible: true, opacity: 0 });
            this._drawLines();
            this.setPrivateRaw("visible", false);
            this._disposers.push(this.setTimeout(function () {
                _this.setPrivate("visible", true);
            }, 500));
            this.lineX.events.on("positionchanged", function () {
                _this._handleXLine();
            });
            this.lineY.events.on("positionchanged", function () {
                _this._handleYLine();
            });
        }
    });
    Object.defineProperty(XYCursor.prototype, "_handleXLine", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var x = this.lineX.x();
            var visible = true;
            if (x < 0 || x > this.width()) {
                visible = false;
            }
            this.lineX.setPrivate("visible", visible);
        }
    });
    Object.defineProperty(XYCursor.prototype, "_handleYLine", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var y = this.lineY.y();
            var visible = true;
            if (y < 0 || y > this.height()) {
                visible = false;
            }
            this.lineY.setPrivate("visible", visible);
        }
    });
    Object.defineProperty(XYCursor.prototype, "_prepareChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            _super.prototype._prepareChildren.call(this);
            if (this.isDirty("xAxis")) {
                this._tooltipX = false;
                var xAxis = this.get("xAxis");
                if (xAxis) {
                    var tooltip_1 = xAxis.get("tooltip");
                    if (tooltip_1) {
                        this._tooltipX = true;
                        tooltip_1.on("pointTo", function () {
                            _this._updateXLine(tooltip_1);
                        });
                    }
                }
            }
            if (this.isDirty("yAxis")) {
                this._tooltipY = false;
                var yAxis = this.get("yAxis");
                if (yAxis) {
                    var tooltip_2 = yAxis.get("tooltip");
                    if (tooltip_2) {
                        this._tooltipY = true;
                        tooltip_2.on("pointTo", function () {
                            _this._updateYLine(tooltip_2);
                        });
                    }
                }
            }
        }
    });
    Object.defineProperty(XYCursor.prototype, "_handleSyncWith", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var chart = this.chart;
            if (chart) {
                var syncWith = this.get("syncWith");
                var otherCharts_1 = [];
                if (syncWith) {
                    $array.each(syncWith, function (cursor) {
                        var chart = cursor.chart;
                        if (chart) {
                            otherCharts_1.push(chart);
                        }
                    });
                }
                chart._otherCharts = otherCharts_1;
            }
        }
    });
    Object.defineProperty(XYCursor.prototype, "_updateChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._updateChildren.call(this);
            this._handleSyncWith();
            if (this.isDirty("positionX") || this.isDirty("positionY")) {
                var positionX = this.get("positionX");
                var positionY = this.get("positionY");
                if (positionX == null && positionY == null) {
                    this.hide(0);
                }
                else {
                    this._movePoint = this.toGlobal(this._getPoint(this.get("positionX", 0), this.get("positionY", 0)));
                    this.handleMove();
                }
            }
        }
    });
    Object.defineProperty(XYCursor.prototype, "_updateXLine", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (tooltip) {
            var x = $math.round(this._display.toLocal(tooltip.get("pointTo", { x: 0, y: 0 })).x, 2);
            if (this._toX != x) {
                this.lineX.animate({ key: "x", to: x, duration: tooltip.get("animationDuration", 0), easing: tooltip.get("animationEasing") });
                this._toX = x;
            }
        }
    });
    Object.defineProperty(XYCursor.prototype, "_updateYLine", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (tooltip) {
            var y = $math.round(this._display.toLocal(tooltip.get("pointTo", { x: 0, y: 0 })).y, 2);
            if (this._toY != y) {
                this.lineY.animate({ key: "y", to: y, duration: tooltip.get("animationDuration", 0), easing: tooltip.get("animationEasing") });
                this._toY = y;
            }
        }
    });
    Object.defineProperty(XYCursor.prototype, "_drawLines", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            this.lineX.set("draw", function (display) {
                display.moveTo(0, 0);
                display.lineTo(0, _this.height());
            });
            this.lineY.set("draw", function (display) {
                display.moveTo(0, 0);
                display.lineTo(_this.width(), 0);
            });
        }
    });
    Object.defineProperty(XYCursor.prototype, "_setChart", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (chart) {
            var _this = this;
            this.chart = chart;
            this._handleSyncWith();
            var plotContainer = chart.plotContainer;
            this.events.on("boundschanged", function () {
                _this._disposers.push(_this.setTimeout(function () {
                    if (_this.get("alwaysShow")) {
                        _this._movePoint = _this.toGlobal(_this._getPoint(_this.get("positionX", 0), _this.get("positionY", 0)));
                        _this.handleMove();
                    }
                }, 50));
            });
            //this._display.interactive = true;
            if ($utils.supports("touchevents")) {
                this._disposers.push(plotContainer.events.on("click", function (event) {
                    if ($utils.isTouchEvent(event.originalEvent)) {
                        _this._handleMove(event.originalEvent);
                    }
                }));
            }
            this._disposers.push(plotContainer.events.on("pointerdown", function (event) {
                _this._handleCursorDown(event.originalEvent);
            }));
            this._disposers.push(plotContainer.events.on("globalpointerup", function (event) {
                _this._handleCursorUp(event.originalEvent);
                if (!event.native && !_this.isHidden()) {
                    _this._handleMove(event.originalEvent);
                }
            }));
            this._disposers.push(plotContainer.events.on("globalpointermove", function (event) {
                if (!_this.get("syncWith")) {
                    if ($object.keys(plotContainer._downPoints).length == 0 && !event.native && _this.isHidden()) {
                        // Ignore mouse movement if it originates on outside element and
                        // we're not dragging.
                        return;
                    }
                }
                _this._handleMove(event.originalEvent);
            }));
            var parent = this.parent;
            if (parent) {
                parent.children.moveValue(this.selection);
            }
        }
    });
    Object.defineProperty(XYCursor.prototype, "_inPlot", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (point) {
            var chart = this.chart;
            if (chart) {
                return chart.inPlot(point);
            }
            return false;
        }
    });
    Object.defineProperty(XYCursor.prototype, "_handleCursorDown", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (event) {
            // TODO: handle multitouch
            var rootPoint = this._root.documentPointToRoot({ x: event.clientX, y: event.clientY });
            var local = this._display.toLocal(rootPoint);
            var chart = this.chart;
            this.selection.set("draw", function () { });
            if (chart && this._inPlot(local)) {
                this._downPoint = local;
                if (this.get("behavior") != "none") {
                    this.selection.show();
                    var type = "selectstarted";
                    if (this.events.isEnabled(type)) {
                        this.events.dispatch(type, { type: type, target: this });
                    }
                }
                var positionX = this._getPosition(local).x;
                var positionY = this._getPosition(local).y;
                this.setPrivate("downPositionX", positionX);
                this.setPrivate("downPositionY", positionY);
            }
        }
    });
    Object.defineProperty(XYCursor.prototype, "_handleCursorUp", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_event) {
            // TODO: handle multitouch
            if (this._downPoint) {
                var behavior = this.get("behavior", "none");
                if (behavior != "none") {
                    if (behavior.charAt(0) === "z") {
                        this.selection.hide();
                    }
                    var userPositionX = this.get("positionX");
                    var positionX = this.getPrivate("positionX", 0);
                    if ($type.isNumber(userPositionX)) {
                        positionX = userPositionX;
                    }
                    var userPositionY = this.get("positionY");
                    var positionY = this.getPrivate("positionY", 0);
                    if ($type.isNumber(userPositionY)) {
                        positionY = userPositionY;
                    }
                    var dispatch = false;
                    if (behavior === "zoomX" || behavior === "zoomXY" || behavior === "selectX" || behavior === "selectXY") {
                        if (Math.abs(positionX - this.getPrivate("downPositionX", 0)) > 0.003) {
                            dispatch = true;
                        }
                    }
                    if (behavior === "zoomY" || behavior === "zoomXY" || behavior === "selectY" || behavior === "selectXY") {
                        if (Math.abs(positionY - this.getPrivate("downPositionY", 0)) > 0.003) {
                            dispatch = true;
                        }
                    }
                    if (dispatch) {
                        var type = "selectended";
                        if (this.events.isEnabled(type)) {
                            this.events.dispatch(type, { type: type, target: this });
                        }
                    }
                }
            }
            this._downPoint = undefined;
        }
    });
    Object.defineProperty(XYCursor.prototype, "_handleMove", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (event) {
            if (this.getPrivate("visible")) {
                // TODO: handle multitouch
                var rootPoint = this._root.documentPointToRoot({ x: event.clientX, y: event.clientY });
                var lastPoint = this._lastPoint;
                if (Math.round(lastPoint.x) === Math.round(rootPoint.x) && Math.round(lastPoint.y) === Math.round(rootPoint.y)) {
                    return;
                }
                this._lastPoint = rootPoint;
                this.handleMove({ x: rootPoint.x, y: rootPoint.y });
            }
        }
    });
    Object.defineProperty(XYCursor.prototype, "_getPosition", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (point) {
            return { x: point.x / this.width(), y: point.y / this.height() };
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(XYCursor.prototype, "handleMove", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (point, skipEvent) {
            if (!point) {
                point = this._movePoint;
            }
            var alwaysShow = this.get("alwaysShow");
            if (!point) {
                this.hide(0);
                return;
            }
            this._movePoint = point;
            var local = this._display.toLocal(point);
            var chart = this.chart;
            if (chart && this._inPlot(local)) {
                chart._movePoint = point;
                if (this.isHidden()) {
                    this.show();
                    var behavior = this.get("behavior", "");
                    if (behavior.charAt(0) == "z") {
                        this.selection.set("draw", function () { });
                    }
                }
                var x = local.x;
                var y = local.y;
                var xyPos = this._getPosition(local);
                this.setPrivate("point", local);
                var snapToSeries_1 = this.get("snapToSeries");
                var userPositionX = this.get("positionX");
                var positionX_1 = xyPos.x;
                if ($type.isNumber(userPositionX)) {
                    positionX_1 = userPositionX;
                }
                var userPositionY = this.get("positionY");
                var positionY_1 = xyPos.y;
                if ($type.isNumber(userPositionY)) {
                    positionY_1 = userPositionY;
                }
                this.setPrivate("positionX", positionX_1);
                this.setPrivate("positionY", positionY_1);
                var xy = this._getPoint(positionX_1, positionY_1);
                x = xy.x;
                y = xy.y;
                chart.xAxes.each(function (axis) {
                    axis._handleCursorPosition(positionX_1, snapToSeries_1);
                    if (alwaysShow) {
                        axis.handleCursorShow();
                    }
                });
                chart.yAxes.each(function (axis) {
                    axis._handleCursorPosition(positionY_1, snapToSeries_1);
                    if (alwaysShow) {
                        axis.handleCursorShow();
                    }
                });
                if (!skipEvent) {
                    chart._handleCursorPosition();
                    var type = "cursormoved";
                    if (this.events.isEnabled(type)) {
                        this.events.dispatch(type, { type: type, target: this });
                    }
                }
                this._updateLines(x, y);
                chart.arrangeTooltips();
            }
            else if (!this._downPoint) {
                if (!alwaysShow) {
                    this.hide(0);
                    var type = "cursorhidden";
                    if (this.events.isEnabled(type)) {
                        this.events.dispatch(type, { type: type, target: this });
                    }
                }
            }
            if (this._downPoint && this.get("behavior") != "none") {
                this._updateSelection(local);
            }
        }
    });
    Object.defineProperty(XYCursor.prototype, "_getPoint", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (positionX, positionY) {
            return { x: this.width() * positionX, y: this.height() * positionY };
        }
    });
    Object.defineProperty(XYCursor.prototype, "_updateLines", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (x, y) {
            if (!this._tooltipX) {
                this.lineX.set("x", x);
            }
            if (!this._tooltipY) {
                this.lineY.set("y", y);
            }
            this._drawLines();
        }
    });
    Object.defineProperty(XYCursor.prototype, "_updateSelection", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (point) {
            var _this = this;
            var selection = this.selection;
            var behavior = this.get("behavior");
            var w = this.width();
            var h = this.height();
            if (point.x < 0) {
                point.x = 0;
            }
            if (point.x > w) {
                point.x = w;
            }
            if (point.y < 0) {
                point.y = 0;
            }
            if (point.y > h) {
                point.y = h;
            }
            selection.set("draw", function (display) {
                var downPoint = _this._downPoint;
                if (downPoint) {
                    if (behavior === "zoomXY" || behavior === "selectXY") {
                        display.moveTo(downPoint.x, downPoint.y);
                        display.lineTo(downPoint.x, point.y);
                        display.lineTo(point.x, point.y);
                        display.lineTo(point.x, downPoint.y);
                        display.lineTo(downPoint.x, downPoint.y);
                    }
                    else if (behavior === "zoomX" || behavior === "selectX") {
                        display.moveTo(downPoint.x, 0);
                        display.lineTo(downPoint.x, h);
                        display.lineTo(point.x, h);
                        display.lineTo(point.x, 0);
                        display.lineTo(downPoint.x, 0);
                    }
                    else if (behavior === "zoomY" || behavior === "selectY") {
                        display.moveTo(0, downPoint.y);
                        display.lineTo(w, downPoint.y);
                        display.lineTo(w, point.y);
                        display.lineTo(0, point.y);
                        display.lineTo(0, downPoint.y);
                    }
                }
            });
        }
    });
    Object.defineProperty(XYCursor.prototype, "_onHide", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (this.isHidden()) {
                var chart = this.chart;
                if (chart) {
                    chart.xAxes.each(function (axis) {
                        axis.handleCursorHide();
                    });
                    chart.yAxes.each(function (axis) {
                        axis.handleCursorHide();
                    });
                    chart.series.each(function (series) {
                        series.handleCursorHide();
                    });
                }
            }
            _super.prototype._onHide.call(this);
        }
    });
    Object.defineProperty(XYCursor.prototype, "_onShow", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (!this.isHidden()) {
                var chart = this.chart;
                if (chart) {
                    chart.xAxes.each(function (axis) {
                        axis.handleCursorShow();
                    });
                    chart.yAxes.each(function (axis) {
                        axis.handleCursorShow();
                    });
                }
            }
            _super.prototype._onShow.call(this);
        }
    });
    Object.defineProperty(XYCursor.prototype, "_dispose", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._dispose.call(this);
            this.selection.dispose();
        }
    });
    Object.defineProperty(XYCursor, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "XYCursor"
    });
    Object.defineProperty(XYCursor, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: Container.classNames.concat([XYCursor.className])
    });
    return XYCursor;
}(Container));
export { XYCursor };
//# sourceMappingURL=XYCursor.js.map