import { __extends, __read } from "tslib";
import { DataItem } from "../../../core/render/Component";
import { Component } from "../../../core/render/Component";
import { Container } from "../../../core/render/Container";
import { p100 } from "../../../core/util/Percent";
import { List } from "../../../core/util/List";
import { Rectangle } from "../../../core/render/Rectangle";
import * as $array from "../../../core/util/Array";
import * as $type from "../../../core/util/Type";
import * as $utils from "../../../core/util/Utils";
/**
 * A base class for all axes.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/xy-chart/#Adding_axes} for more info
 */
var Axis = /** @class */ (function (_super) {
    __extends(Axis, _super);
    function Axis() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "_series", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        Object.defineProperty(_this, "_isPanning", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        /**
         * A [[Container]] that holds all the axis label elements.
         *
         * @default Container.new()
         */
        Object.defineProperty(_this, "labelsContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.children.push(Container.new(_this._root, {}))
        });
        /**
         * A [[Container]] that holds all the axis grid and fill elements.
         *
         * @default Container.new()
         */
        Object.defineProperty(_this, "gridContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: Container.new(_this._root, { width: p100, height: p100 })
        });
        /**
         * A [[Container]] that holds axis grid elements which goes above the series.
         *
         * @default Container.new()
         */
        Object.defineProperty(_this, "topGridContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: Container.new(_this._root, { width: p100, height: p100 })
        });
        /**
         * A [[Container]] that holds all the axis bullet elements.
         *
         * @default new Container
         */
        Object.defineProperty(_this, "bulletsContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.children.push(Container.new(_this._root, { isMeasured: false, width: p100, height: p100, position: "absolute" }))
        });
        /**
         * A referenece to the the chart the axis belongs to.
         */
        Object.defineProperty(_this, "chart", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_rangesDirty", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(_this, "_panStart", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "_panEnd", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 1
        });
        Object.defineProperty(_this, "_sAnimation", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_eAnimation", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        /**
         * A list of axis ranges.
         *
         * @see {@link https://www.amcharts.com/docs/v5/charts/xy-chart/axes/axis-ranges/} for more info
         * @default new List()
         */
        Object.defineProperty(_this, "axisRanges", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new List()
        });
        Object.defineProperty(_this, "_seriesAxisRanges", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        /**
         * A control label that is invisible but is used to keep width the width of
         * the axis constant.
         */
        Object.defineProperty(_this, "ghostLabel", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_cursorPosition", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: -1
        });
        Object.defineProperty(_this, "_snapToSeries", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_seriesValuesDirty", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        /**
         * A container above the axis that can be used to add additional stuff into
         * it. For example a legend, label, or an icon.
         *
         * @see {@link https://www.amcharts.com/docs/v5/charts/xy-chart/axes/axis-headers/} for more info
         * @default new Container
         */
        Object.defineProperty(_this, "axisHeader", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.children.push(Container.new(_this._root, {
                themeTags: ["axis", "header"],
                position: "absolute",
                background: Rectangle.new(_this._root, {
                    themeTags: ["header", "background"],
                    fill: _this._root.interfaceColors.get("background")
                })
            }))
        });
        return _this;
    }
    Object.defineProperty(Axis.prototype, "_dispose", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            // these could be in other parents, so disposing just in case
            this.gridContainer.dispose();
            this.topGridContainer.dispose();
            this.bulletsContainer.dispose();
            this.labelsContainer.dispose();
            this.axisHeader.dispose();
            _super.prototype._dispose.call(this);
        }
    });
    Object.defineProperty(Axis.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            _super.prototype._afterNew.call(this);
            this.setPrivate("updateScrollbar", true);
            this._disposers.push(this.axisRanges.events.onAll(function (change) {
                if (change.type === "clear") {
                    $array.each(change.oldValues, function (dataItem) {
                        _this.disposeDataItem(dataItem);
                    });
                }
                else if (change.type === "push") {
                    _this._processAxisRange(change.newValue, ["range"]);
                }
                else if (change.type === "setIndex") {
                    _this._processAxisRange(change.newValue, ["range"]);
                }
                else if (change.type === "insertIndex") {
                    _this._processAxisRange(change.newValue, ["range"]);
                }
                else if (change.type === "removeIndex") {
                    _this.disposeDataItem(change.oldValue);
                }
                else if (change.type === "moveIndex") {
                    _this._processAxisRange(change.value, ["range"]);
                }
                else {
                    throw new Error("Unknown IStreamEvent type");
                }
            }));
            var renderer = this.get("renderer");
            if (renderer) {
                renderer.axis = this;
                renderer.processAxis();
            }
            this.children.push(renderer);
            this.ghostLabel = renderer.makeLabel(new DataItem(this, undefined, {}), []);
            this.ghostLabel.set("opacity", 0);
        }
    });
    /**
     * Zooms the axis to relative locations.
     *
     * Both `start` and `end` are relative: 0 means start of the axis, 1 - end.
     *
     * @param   start     Relative start
     * @param   end       Relative end
     * @param   duration  Duration of the zoom animation in milliseconds
     * @return            Zoom animation
     */
    Object.defineProperty(Axis.prototype, "zoom", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (start, end, duration) {
            var _a;
            if (this.get("start") !== start || this.get("end") != end) {
                var sAnimation = this._sAnimation;
                var eAnimation = this._eAnimation;
                var maxDeviation = this.get("maxDeviation", 0.5) * Math.min(1, (end - start));
                if (start < -maxDeviation) {
                    start = -maxDeviation;
                }
                if (end > 1 + maxDeviation) {
                    end = 1 + maxDeviation;
                }
                if (start > end) {
                    _a = __read([end, start], 2), start = _a[0], end = _a[1];
                }
                if (!$type.isNumber(duration)) {
                    duration = this.get("interpolationDuration", 0);
                }
                var priority = "end";
                var maxZoomFactor = this.getPrivate("maxZoomFactor", this.get("maxZoomFactor", 100));
                if (end === 1 && start !== 0) {
                    if (start < this.get("start")) {
                        priority = "start";
                    }
                    else {
                        priority = "end";
                    }
                }
                if (start === 0 && end !== 1) {
                    if (end > this.get("end")) {
                        priority = "end";
                    }
                    else {
                        priority = "start";
                    }
                }
                var minZoomCount = this.get("minZoomCount");
                var maxZoomCount = this.get("maxZoomCount");
                if ($type.isNumber(minZoomCount)) {
                    maxZoomFactor = maxZoomFactor / minZoomCount;
                }
                var minZoomFactor = 1;
                if ($type.isNumber(maxZoomCount)) {
                    minZoomFactor = maxZoomFactor / maxZoomCount;
                }
                // most likely we are dragging left scrollbar grip here, so we tend to modify end
                if (priority === "start") {
                    if (maxZoomCount > 0) {
                        // add to the end
                        if (1 / (end - start) < minZoomFactor) {
                            end = start + 1 / minZoomFactor;
                        }
                    }
                    // add to the end
                    if (1 / (end - start) > maxZoomFactor) {
                        end = start + 1 / maxZoomFactor;
                    }
                    //unless end is > 0
                    if (end > 1 && end - start < 1 / maxZoomFactor) {
                        //end = 1;
                        start = end - 1 / maxZoomFactor;
                    }
                }
                // most likely we are dragging right, so we modify left
                else {
                    if (maxZoomCount > 0) {
                        // add to the end
                        if (1 / (end - start) < minZoomFactor) {
                            start = end - 1 / minZoomFactor;
                        }
                    }
                    // remove from start
                    if (1 / (end - start) > maxZoomFactor) {
                        start = end - 1 / maxZoomFactor;
                    }
                    if (start < 0 && end - start < 1 / maxZoomFactor) {
                        //start = 0;
                        end = start + 1 / maxZoomFactor;
                    }
                }
                if (1 / (end - start) > maxZoomFactor) {
                    end = start + 1 / maxZoomFactor;
                }
                if (1 / (end - start) > maxZoomFactor) {
                    start = end - 1 / maxZoomFactor;
                }
                if (((sAnimation && sAnimation.playing && sAnimation.to == start) || this.get("start") == start) && ((eAnimation && eAnimation.playing && eAnimation.to == end) || this.get("end") == end)) {
                    return;
                }
                if (duration > 0) {
                    var easing = this.get("interpolationEasing");
                    var sAnimation_1 = this.animate({ key: "start", to: start, duration: duration, easing: easing });
                    var eAnimation_1 = this.animate({ key: "end", to: end, duration: duration, easing: easing });
                    this._sAnimation = sAnimation_1;
                    this._eAnimation = eAnimation_1;
                    if (sAnimation_1) {
                        return sAnimation_1;
                    }
                    else if (eAnimation_1) {
                        return eAnimation_1;
                    }
                }
                else {
                    this.set("start", start);
                    this.set("end", end);
                }
            }
        }
    });
    Object.defineProperty(Axis.prototype, "series", {
        /**
         * A list of series using this axis.
         *
         * @return Series
         */
        get: function () {
            return this._series;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(Axis.prototype, "_processAxisRange", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, themeTags) {
            dataItem.setRaw("isRange", true);
            this._createAssets(dataItem, themeTags);
            this._rangesDirty = true;
            this._prepareDataItem(dataItem);
            var above = dataItem.get("above");
            var container = this.topGridContainer;
            var grid = dataItem.get("grid");
            if (above && grid) {
                container.children.moveValue(grid);
            }
            var fill = dataItem.get("axisFill");
            if (above && fill) {
                container.children.moveValue(fill);
            }
        }
    });
    Object.defineProperty(Axis.prototype, "_prepareDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_dataItem, _index) { }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Axis.prototype, "markDirtyExtremes", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Axis.prototype, "markDirtySelectionExtremes", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
        }
    });
    Object.defineProperty(Axis.prototype, "_calculateTotals", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
        }
    });
    Object.defineProperty(Axis.prototype, "_updateAxisRanges", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            this.axisRanges.each(function (axisRange) {
                _this._prepareDataItem(axisRange);
            });
            $array.each(this._seriesAxisRanges, function (axisRange) {
                _this._prepareDataItem(axisRange);
            });
        }
    });
    Object.defineProperty(Axis.prototype, "_prepareChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._prepareChildren.call(this);
            if (this.get("fixAxisSize")) {
                this.ghostLabel.set("visible", true);
            }
            else {
                this.ghostLabel.set("visible", false);
            }
            if (this.isDirty("start") || this.isDirty("end")) {
                this.chart._updateCursor();
                var start = this.get("start", 0);
                var end = this.get("end", 1);
                var maxDeviation = this.get("maxDeviation", 0.5) * Math.min(1, (end - start));
                if (start < -maxDeviation) {
                    var delta = start + maxDeviation;
                    start = -maxDeviation;
                    this.setRaw("start", start);
                    if (this.isDirty("end")) {
                        this.setRaw("end", end - delta);
                    }
                }
                if (end > 1 + maxDeviation) {
                    var delta = end - 1 - maxDeviation;
                    end = 1 + maxDeviation;
                    this.setRaw("end", end);
                    if (this.isDirty("start")) {
                        this.setRaw("start", start - delta);
                    }
                }
            }
            var renderer = this.get("renderer");
            renderer._start = this.get("start");
            renderer._end = this.get("end");
            renderer._inversed = renderer.get("inversed", false);
            renderer._axisLength = renderer.axisLength() / (renderer._end - renderer._start);
            renderer._updateLC();
            if (this.isDirty("tooltip")) {
                var tooltip = this.get("tooltip");
                if (tooltip) {
                    var rendererTags = renderer.get("themeTags");
                    tooltip.addTag("axis");
                    tooltip.addTag(this.className.toLowerCase());
                    tooltip._applyThemes();
                    if (rendererTags) {
                        tooltip.set("themeTags", $utils.mergeTags(tooltip.get("themeTags"), rendererTags));
                        tooltip.label._applyThemes();
                    }
                }
            }
        }
    });
    Object.defineProperty(Axis.prototype, "_updateTooltipBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var tooltip = this.get("tooltip");
            if (tooltip) {
                this.get("renderer").updateTooltipBounds(tooltip);
            }
        }
    });
    Object.defineProperty(Axis.prototype, "_updateBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._updateBounds.call(this);
            this._updateTooltipBounds();
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Axis.prototype, "processChart", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (chart) {
            var _this = this;
            this.chart = chart;
            var renderer = this.get("renderer");
            renderer.chart = chart;
            chart.gridContainer.children.push(this.gridContainer);
            chart.topGridContainer.children.push(this.topGridContainer);
            chart.axisHeadersContainer.children.push(this.axisHeader);
            this.on("start", function () {
                chart._handleAxisSelection(_this);
            });
            this.on("end", function () {
                chart._handleAxisSelection(_this);
            });
            chart.plotContainer.onPrivate("width", function () {
                _this.markDirtySize();
            });
            chart.plotContainer.onPrivate("height", function () {
                _this.markDirtySize();
            });
            chart.processAxis(this);
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Axis.prototype, "hideDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            this._toggleDataItem(dataItem, false);
            return _super.prototype.hideDataItem.call(this, dataItem);
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Axis.prototype, "showDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            this._toggleDataItem(dataItem, true);
            return _super.prototype.showDataItem.call(this, dataItem);
        }
    });
    Object.defineProperty(Axis.prototype, "_toggleDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, visible) {
            var label = dataItem.get("label");
            if (label) {
                label.setPrivate("visible", visible);
            }
            var grid = dataItem.get("grid");
            if (grid) {
                grid.setPrivate("visible", visible);
            }
            var tick = dataItem.get("tick");
            if (tick) {
                tick.setPrivate("visible", visible);
            }
            var axisFill = dataItem.get("axisFill");
            if (axisFill) {
                axisFill.setPrivate("visible", visible);
            }
            var bullet = dataItem.get("bullet");
            if (bullet) {
                var sprite = bullet.get("sprite");
                if (sprite) {
                    sprite.setPrivate("visible", visible);
                }
            }
        }
    });
    Object.defineProperty(Axis.prototype, "_createAssets", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, tags) {
            var renderer = this.get("renderer");
            if (!dataItem.get("label")) {
                renderer.makeLabel(dataItem, tags);
            }
            if (!dataItem.get("grid")) {
                renderer.makeGrid(dataItem, tags);
            }
            if (!dataItem.get("tick")) {
                renderer.makeTick(dataItem, tags);
            }
            if (!dataItem.get("axisFill")) {
                renderer.makeAxisFill(dataItem, tags);
            }
            this._processBullet(dataItem);
        }
    });
    Object.defineProperty(Axis.prototype, "_processBullet", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var bullet = dataItem.get("bullet");
            var axisBullet = this.get("bullet");
            if (!bullet && axisBullet && !dataItem.get("isRange")) {
                bullet = axisBullet(this._root, this, dataItem);
            }
            if (bullet) {
                bullet.axis = this;
                var sprite = bullet.get("sprite");
                if (sprite) {
                    sprite._setDataItem(dataItem);
                    dataItem.setRaw("bullet", bullet);
                    if (!sprite.parent) {
                        this.bulletsContainer.children.push(sprite);
                    }
                }
            }
        }
    });
    Object.defineProperty(Axis.prototype, "_afterChanged", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._afterChanged.call(this);
            var chart = this.chart;
            if (chart) {
                chart._updateChartLayout();
                chart.axisHeadersContainer.markDirtySize();
            }
            this.get("renderer")._updatePositions();
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Axis.prototype, "disposeDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            _super.prototype.disposeDataItem.call(this, dataItem);
            var renderer = this.get("renderer");
            var label = dataItem.get("label");
            if (label) {
                renderer.labels.removeValue(label);
                label.dispose();
            }
            var tick = dataItem.get("tick");
            if (tick) {
                renderer.ticks.removeValue(tick);
                tick.dispose();
            }
            var grid = dataItem.get("grid");
            if (grid) {
                renderer.grid.removeValue(grid);
                grid.dispose();
            }
            var axisFill = dataItem.get("axisFill");
            if (axisFill) {
                renderer.axisFills.removeValue(axisFill);
                axisFill.dispose();
            }
        }
    });
    Object.defineProperty(Axis.prototype, "_updateGhost", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var ghostLabel = this.ghostLabel;
            if (!ghostLabel.isHidden()) {
                var bounds = ghostLabel.localBounds();
                var gWidth_1 = bounds.right - bounds.left;
                $array.each(this.dataItems, function (dataItem) {
                    var label = dataItem.get("label");
                    if (label && !label.isHidden()) {
                        var bounds_1 = label.localBounds();
                        var w = bounds_1.right - bounds_1.left;
                        if (w > gWidth_1) {
                            ghostLabel.set("text", label.text._getText());
                        }
                    }
                });
            }
            var start = this.get("start", 0);
            var end = this.get("end", 1);
            this.get("renderer").updateLabel(ghostLabel, start + (end - start) * 0.5);
            ghostLabel.setPrivate("visible", true);
        }
    });
    Object.defineProperty(Axis.prototype, "_handleCursorPosition", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (position, snapToSeries) {
            var renderer = this.get("renderer");
            position = renderer.toAxisPosition(position);
            this._cursorPosition = position;
            this._snapToSeries = snapToSeries;
            this.updateTooltip();
        }
    });
    /**
     * Can be called when axis zoom changes and you need to update tooltip
     * position.
     */
    Object.defineProperty(Axis.prototype, "updateTooltip", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            var snapToSeries = this._snapToSeries;
            var position = this._cursorPosition;
            var tooltip = this.get("tooltip");
            var renderer = this.get("renderer");
            if ($type.isNumber(position)) {
                $array.each(this.series, function (series) {
                    if (series.get("baseAxis") === _this) {
                        var dataItem = _this.getSeriesItem(series, position);
                        series.setRaw("tooltipDataItem", dataItem);
                        if (snapToSeries && snapToSeries.indexOf(series) != -1) {
                            series.updateLegendMarker(dataItem);
                            series.updateLegendValue(dataItem);
                        }
                        else {
                            series.showDataItemTooltip(dataItem);
                        }
                    }
                });
                if (tooltip) {
                    renderer.updateTooltipBounds(tooltip);
                    if (this.get("snapTooltip")) {
                        position = this.roundAxisPosition(position, this.get("tooltipLocation", 0.5));
                    }
                    if (!$type.isNaN(position)) {
                        this._updateTooltipText(tooltip, position);
                        renderer.positionTooltip(tooltip, position);
                        if (position < this.get("start") || position > this.get("end")) {
                            tooltip.hide(0);
                        }
                        else {
                            tooltip.show(0);
                        }
                    }
                    else {
                        tooltip.hide(0);
                    }
                }
            }
        }
    });
    Object.defineProperty(Axis.prototype, "_updateTooltipText", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (tooltip, position) {
            tooltip.label.set("text", this.getTooltipText(position));
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Axis.prototype, "roundAxisPosition", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (position, _location) {
            return position;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Axis.prototype, "handleCursorShow", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var tooltip = this.get("tooltip");
            if (tooltip) {
                tooltip.show();
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Axis.prototype, "handleCursorHide", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var tooltip = this.get("tooltip");
            if (tooltip) {
                tooltip.hide();
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Axis.prototype, "processSeriesDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_dataItem, _fields) {
        }
    });
    Object.defineProperty(Axis.prototype, "_clearDirty", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._clearDirty.call(this);
            this._sizeDirty = false;
            this._rangesDirty = false;
        }
    });
    /**
     * Converts pixel coordinate to a relative position on axis.
     *
     * @param   coordinate  Coordinate
     * @return              Relative position
     */
    Object.defineProperty(Axis.prototype, "coordinateToPosition", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (coordinate) {
            var renderer = this.get("renderer");
            return renderer.toAxisPosition(coordinate / renderer.axisLength());
        }
    });
    /**
     * Converts relative position of the plot area to relative position of the
     * axis with zoom taken into account.
     *
     * @param position Position
     * @return Relative position
     */
    Object.defineProperty(Axis.prototype, "toAxisPosition", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (position) {
            return this.get("renderer").toAxisPosition(position);
        }
    });
    /**
     * Adjusts position with inversed taken into account.
     *
     * @ignore
     */
    Object.defineProperty(Axis.prototype, "fixPosition", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (position) {
            return this.get("renderer").fixPosition(position);
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Axis.prototype, "shouldGap", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_dataItem, _nextItem, _autoGapCount, _fieldName) {
            return false;
        }
    });
    /**
     * Creates and returns an axis range object.
     *
     * @see {@link https://www.amcharts.com/docs/v5/charts/xy-chart/axes/axis-ranges/} for more info
     * @param   axisDataItem  Axis data item
     * @return                Axis range
     */
    Object.defineProperty(Axis.prototype, "createAxisRange", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (axisDataItem) {
            return this.axisRanges.push(axisDataItem);
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Axis.prototype, "_groupSeriesData", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_series) { }
    });
    Object.defineProperty(Axis, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "Axis"
    });
    Object.defineProperty(Axis, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: Component.classNames.concat([Axis.className])
    });
    return Axis;
}(Component));
export { Axis };
//# sourceMappingURL=Axis.js.map