import { __extends, __read } from "tslib";
import { AxisRenderer } from "./AxisRenderer";
import { p100 } from "../../../core/util/Percent";
import * as $type from "../../../core/util/Type";
import * as $utils from "../../../core/util/Utils";
import { Rectangle } from "../../../core/render/Rectangle";
/**
 * Used to render vertical axis.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/xy-chart/#Axis_renderer} for more info
 * @important
 */
var AxisRendererY = /** @class */ (function (_super) {
    __extends(AxisRendererY, _super);
    function AxisRendererY() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "_downY", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "thumb", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: Rectangle.new(_this._root, { height: p100, themeTags: ["axis", "y", "thumb"] })
        });
        return _this;
    }
    Object.defineProperty(AxisRendererY.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._settings.themeTags = $utils.mergeTags(this._settings.themeTags, ["renderer", "y"]);
            if (this._settings.opposite) {
                this._settings.themeTags.push("opposite");
            }
            _super.prototype._afterNew.call(this);
            this.setPrivateRaw("letter", "Y");
            var gridTemplate = this.grid.template;
            gridTemplate.set("width", p100);
            gridTemplate.set("height", 0);
            gridTemplate.set("draw", function (display, graphics) {
                display.moveTo(0, 0);
                display.lineTo(graphics.width(), 0);
            });
            this.set("draw", function (display, renderer) {
                display.moveTo(0, 0);
                display.lineTo(0, renderer.height());
            });
        }
    });
    Object.defineProperty(AxisRendererY.prototype, "_getPan", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (point1, point2) {
            return (point1.y - point2.y) / this.height();
        }
    });
    Object.defineProperty(AxisRendererY.prototype, "_changed", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._changed.call(this);
            var axis = this.axis;
            if (this.isDirty("inside")) {
                axis.markDirtySize();
            }
            var thumb = this.thumb;
            var opposite = "opposite";
            if (this.isDirty(opposite)) {
                var chart = this.chart;
                var axisChildren = axis.children;
                if (chart) {
                    if (this.get(opposite)) {
                        var children = chart.rightAxesContainer.children;
                        if (children.indexOf(axis) == -1) {
                            children.moveValue(axis, 0);
                        }
                        axis.addTag(opposite);
                        axisChildren.moveValue(this, 0);
                        thumb.set("centerX", 0);
                    }
                    else {
                        var children = chart.leftAxesContainer.children;
                        if (children.indexOf(axis) == -1) {
                            children.moveValue(axis);
                        }
                        axis.removeTag("opposite");
                        axisChildren.moveValue(this);
                        thumb.set("centerX", p100);
                    }
                    axis.markDirtySize();
                }
                axis.ghostLabel._applyThemes();
            }
            thumb.setPrivate("width", axis.labelsContainer.width());
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererY.prototype, "processAxis", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype.processAxis.call(this);
            var axis = this.axis;
            if (axis.get("height") == null) {
                axis.set("height", p100);
            }
            var horizontalLayout = this._root.horizontalLayout;
            axis.set("layout", horizontalLayout);
            axis.labelsContainer.set("height", p100);
            axis.axisHeader.set("layout", horizontalLayout);
        }
    });
    Object.defineProperty(AxisRendererY.prototype, "_updatePositions", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var axis = this.axis;
            axis.gridContainer.set("y", axis.y() - $utils.relativeToValue(axis.get("centerY", 0), axis.height()));
            axis.bulletsContainer.set("x", this.x());
            var chart = axis.chart;
            if (chart) {
                var plotContainer = chart.plotContainer;
                var axisHeader = axis.axisHeader;
                var height = axis.get("marginTop", 0);
                if (axisHeader.children.length > 0) {
                    height = axis.axisHeader.height();
                    axis.set("marginTop", height);
                }
                else {
                    axisHeader.set("height", height);
                }
                axisHeader.setAll({ y: axis.y() - height, x: -1, width: plotContainer.width() + 2 });
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererY.prototype, "axisLength", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this.axis.innerHeight();
        }
    });
    /**
     * Converts axis relative position to actual coordinate in pixels.
     *
     * @param   position  Position
     * @return            Point
     */
    Object.defineProperty(AxisRendererY.prototype, "positionToPoint", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (position) {
            return { x: 0, y: this.positionToCoordinate(position) };
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererY.prototype, "updateLabel", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (label, position, endPosition, count) {
            if (label) {
                if (!$type.isNumber(position)) {
                    position = 0;
                }
                var location_1 = 0.5;
                if ($type.isNumber(count) && count > 1) {
                    location_1 = label.get("multiLocation", location_1);
                }
                else {
                    location_1 = label.get("location", location_1);
                }
                var opposite = this.get("opposite");
                var inside = label.get("inside", this.get("inside", false));
                if (opposite) {
                    label.set("x", 0);
                    if (inside) {
                        label.set("position", "absolute");
                    }
                    else {
                        label.set("position", "relative");
                    }
                }
                else {
                    if (inside) {
                        label.set("x", 0);
                        label.set("position", "absolute");
                    }
                    else {
                        label.set("x", undefined);
                        label.set("position", "relative");
                    }
                }
                if ($type.isNumber(endPosition) && endPosition != position) {
                    position = position + (endPosition - position) * location_1;
                }
                label.set("y", this.positionToCoordinate(position));
                this.toggleVisibility(label, position, label.get("minPosition", 0), label.get("maxPosition", 1));
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererY.prototype, "updateGrid", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (grid, position, endPosition) {
            if (grid) {
                if (!$type.isNumber(position)) {
                    position = 0;
                }
                var location_2 = grid.get("location", 0.5);
                if ($type.isNumber(endPosition) && endPosition != position) {
                    position = position + (endPosition - position) * location_2;
                }
                var y = this.positionToCoordinate(position);
                grid.set("y", y);
                this.toggleVisibility(grid, position, 0, 1);
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererY.prototype, "updateTick", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (tick, position, endPosition, count) {
            if (tick) {
                if (!$type.isNumber(position)) {
                    position = 0;
                }
                var location_3 = 0.5;
                if ($type.isNumber(count) && count > 1) {
                    location_3 = tick.get("multiLocation", location_3);
                }
                else {
                    location_3 = tick.get("location", location_3);
                }
                if ($type.isNumber(endPosition) && endPosition != position) {
                    position = position + (endPosition - position) * location_3;
                }
                tick.set("y", this.positionToCoordinate(position));
                var length_1 = tick.get("length", 0);
                var inside = tick.get("inside", this.get("inside", false));
                if (this.get("opposite")) {
                    tick.set("x", 0);
                    if (inside) {
                        length_1 *= -1;
                    }
                }
                else {
                    if (!inside) {
                        length_1 *= -1;
                    }
                }
                tick.set("draw", function (display) {
                    display.moveTo(0, 0);
                    display.lineTo(length_1, 0);
                });
                this.toggleVisibility(tick, position, tick.get("minPosition", 0), tick.get("maxPosition", 1));
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererY.prototype, "updateBullet", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (bullet, position, endPosition) {
            if (bullet) {
                var sprite = bullet.get("sprite");
                if (sprite) {
                    if (!$type.isNumber(position)) {
                        position = 0;
                    }
                    var location_4 = bullet.get("location", 0.5);
                    if ($type.isNumber(endPosition) && endPosition != position) {
                        position = position + (endPosition - position) * location_4;
                    }
                    sprite.set("y", this.positionToCoordinate(position));
                    this.toggleVisibility(sprite, position, 0, 1);
                }
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererY.prototype, "updateFill", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (fill, position, endPosition) {
            if (fill) {
                if (!$type.isNumber(position)) {
                    position = 0;
                }
                if (!$type.isNumber(endPosition)) {
                    endPosition = 1;
                }
                var y0 = this.positionToCoordinate(position);
                var y1 = this.positionToCoordinate(endPosition);
                this.fillDrawMethod(fill, y0, y1);
            }
        }
    });
    Object.defineProperty(AxisRendererY.prototype, "fillDrawMethod", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (fill, y0, y1) {
            var _this = this;
            fill.set("draw", function (display) {
                var _a;
                // using for holes, so can not be rectangle
                var w = _this.axis.gridContainer.width();
                var h = _this.height();
                if (y1 < y0) {
                    _a = __read([y0, y1], 2), y1 = _a[0], y0 = _a[1];
                }
                if (y0 > h || y1 < 0) {
                    return;
                }
                y0 = Math.max(0, y0);
                y1 = Math.min(h, y1);
                display.moveTo(0, y0);
                display.lineTo(w, y0);
                display.lineTo(w, y1);
                display.lineTo(0, y1);
                display.lineTo(0, y0);
            });
        }
    });
    /**
     * Converts relative position (0-1) on axis to a pixel coordinate.
     *
     * @param position  Position (0-1)
     * @return Coordinate (px)
     */
    Object.defineProperty(AxisRendererY.prototype, "positionToCoordinate", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (position) {
            if (!this._inversed) {
                return (this._end - position) * this._axisLength;
            }
            else {
                return (position - this._start) * this._axisLength;
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererY.prototype, "positionTooltip", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (tooltip, position) {
            this._positionTooltip(tooltip, { x: 0, y: this.positionToCoordinate(position) });
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererY.prototype, "updateTooltipBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (tooltip) {
            var inside = this.get("inside");
            var num = 100000;
            var global = this._display.toGlobal({ x: 0, y: 0 });
            var y = global.y;
            var x = 0;
            var h = this.axisLength();
            var w = num;
            var pointerOrientation = "right";
            if (this.get("opposite")) {
                if (inside) {
                    pointerOrientation = "right";
                    x = global.x - num;
                    w = num;
                }
                else {
                    pointerOrientation = "left";
                    x = global.x;
                    w = num;
                }
            }
            else {
                if (inside) {
                    pointerOrientation = "left";
                    x = global.x;
                    w = num;
                }
                else {
                    pointerOrientation = "right";
                    x = global.x - num;
                    w = num;
                }
            }
            var bounds = { left: x, right: x + w, top: y, bottom: y + h };
            var oldBounds = tooltip.get("bounds");
            if (!$utils.sameBounds(bounds, oldBounds)) {
                tooltip.set("bounds", bounds);
                tooltip.set("pointerOrientation", pointerOrientation);
            }
        }
    });
    Object.defineProperty(AxisRendererY.prototype, "_updateLC", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var axis = this.axis;
            var parent = axis.parent;
            if (parent) {
                var h = parent.innerHeight();
                this._lc = this.axisLength() / h;
                this._ls = axis.y() / h;
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererY.prototype, "toAxisPosition", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (position) {
            var start = this._start || 0;
            var end = this._end || 1;
            position -= this._ls;
            position = position * (end - start) / this._lc;
            if (this.get("inversed")) {
                position = start + position;
            }
            else {
                position = end - position;
            }
            return position;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(AxisRendererY.prototype, "fixPosition", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (position) {
            if (!this.get("inversed")) {
                return 1 - position;
            }
            return position;
        }
    });
    Object.defineProperty(AxisRendererY, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "AxisRendererY"
    });
    Object.defineProperty(AxisRendererY, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: AxisRenderer.classNames.concat([AxisRendererY.className])
    });
    return AxisRendererY;
}(AxisRenderer));
export { AxisRendererY };
//# sourceMappingURL=AxisRendererY.js.map