import { __extends, __read } from "tslib";
import { DataItem } from "../../../core/render/Component";
import { Axis } from "./Axis";
import * as $type from "../../../core/util/Type";
import * as $array from "../../../core/util/Array";
import * as $math from "../../../core/util/Math";
import * as $utils from "../../../core/util/Utils";
import { MultiDisposer } from "../../../core/util/Disposer";
/**
 * Creates a value axis.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/xy-chart/axes/value-axis/} for more info
 * @important
 */
var ValueAxis = /** @class */ (function (_super) {
    __extends(ValueAxis, _super);
    function ValueAxis() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "_dirtyExtremes", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(_this, "_dirtySelectionExtremes", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(_this, "_deltaMinMax", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 1
        });
        Object.defineProperty(_this, "_minReal", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_maxReal", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_baseValue", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "_syncDp", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        return _this;
    }
    /**
     * @ignore
     */
    Object.defineProperty(ValueAxis.prototype, "markDirtyExtremes", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._dirtyExtremes = true;
            this.markDirty();
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(ValueAxis.prototype, "markDirtySelectionExtremes", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._dirtySelectionExtremes = true;
            this.markDirty();
        }
    });
    Object.defineProperty(ValueAxis.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._settings.themeTags = $utils.mergeTags(this._settings.themeTags, ["axis"]);
            this.setPrivateRaw("name", "value");
            this.addTag("value");
            _super.prototype._afterNew.call(this);
        }
    });
    Object.defineProperty(ValueAxis.prototype, "_prepareChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            _super.prototype._prepareChildren.call(this);
            if (this.isDirty("syncWithAxis")) {
                var previousValue = this._prevSettings.syncWithAxis;
                if (previousValue) {
                    if (this._syncDp) {
                        this._syncDp.dispose();
                    }
                }
                var syncWithAxis = this.get("syncWithAxis");
                if (syncWithAxis) {
                    this._syncDp = new MultiDisposer([
                        syncWithAxis.onPrivate("selectionMinFinal", function () {
                            _this._dirtySelectionExtremes = true;
                        }),
                        syncWithAxis.onPrivate("selectionMaxFinal", function () {
                            _this._dirtySelectionExtremes = true;
                        })
                    ]);
                }
            }
            //if (this._dirtyExtremes || this.isPrivateDirty("width") || this.isPrivateDirty("height") || this.isDirty("min") || this.isDirty("max") || this.isDirty("extraMin") || this.isDirty("extraMax") || this.isDirty("logarithmic") || this.isDirty("treatZeroAs") || this.isDirty("baseValue") || this.isDirty("strictMinMax") || this.isDirty("maxPrecision")) {
            if (this._sizeDirty || this._dirtyExtremes || this._valuesDirty || this.isPrivateDirty("width") || this.isPrivateDirty("height") || this.isDirty("min") || this.isDirty("max") || this.isDirty("extraMin") || this.isDirty("extraMax") || this.isDirty("logarithmic") || this.isDirty("treatZeroAs") || this.isDirty("baseValue") || this.isDirty("strictMinMax") || this.isDirty("maxPrecision") || this.isDirty("numberFormat")) {
                this._getMinMax();
                this._dirtyExtremes = false;
            }
            if (this._dirtySelectionExtremes && !this._isPanning) {
                this._getSelectionMinMax();
                this._dirtySelectionExtremes = false;
            }
            this._groupData();
            if (this._sizeDirty || this._valuesDirty || this.isDirty("start") || this.isDirty("end") || this.isPrivateDirty("min") || this.isPrivateDirty("selectionMax") || this.isPrivateDirty("selectionMin") || this.isPrivateDirty("max") || this.isPrivateDirty("step") || this.isPrivateDirty("width") || this.isPrivateDirty("height") || this.isDirty("logarithmic")) {
                this._handleRangeChange();
                this._prepareAxisItems();
                this._updateAxisRanges();
            }
            this._baseValue = this.baseValue();
        }
    });
    Object.defineProperty(ValueAxis.prototype, "_groupData", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
        }
    });
    Object.defineProperty(ValueAxis.prototype, "_formatText", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (value) {
            var numberFormat = this.get("numberFormat");
            var formatter = this.getNumberFormatter();
            var text = "";
            if (numberFormat) {
                text = formatter.format(value, numberFormat);
            }
            else {
                text = formatter.format(value, undefined, this.getPrivate("stepDecimalPlaces"));
            }
            return text;
        }
    });
    Object.defineProperty(ValueAxis.prototype, "_prepareAxisItems", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var min = this.getPrivate("min");
            var max = this.getPrivate("max");
            if ($type.isNumber(min) && $type.isNumber(max)) {
                var logarithmic = this.get("logarithmic");
                var step = this.getPrivate("step");
                var selectionMin = this.getPrivate("selectionMin");
                var selectionMax = this.getPrivate("selectionMax") + step;
                var value = selectionMin - step;
                var i = 0;
                if (logarithmic) {
                    value = selectionMin;
                }
                while (value < selectionMax) {
                    var dataItem = void 0;
                    if (this.dataItems.length < i + 1) {
                        dataItem = new DataItem(this, undefined, {});
                        this._dataItems.push(dataItem);
                        this.processDataItem(dataItem);
                    }
                    else {
                        dataItem = this.dataItems[i];
                    }
                    this._createAssets(dataItem, []);
                    if (dataItem.isHidden()) {
                        dataItem.show();
                    }
                    dataItem.setRaw("value", value);
                    var label = dataItem.get("label");
                    if (label) {
                        label.set("text", this._formatText(value));
                    }
                    this._prepareDataItem(dataItem);
                    if (!logarithmic) {
                        value += step;
                    }
                    else {
                        var differencePower = Math.log(max) * Math.LOG10E - Math.log(min) * Math.LOG10E;
                        if (differencePower > 1) {
                            value = Math.pow(10, Math.log(min) * Math.LOG10E + i);
                        }
                        else {
                            value += step;
                        }
                    }
                    var stepPower = Math.pow(10, Math.floor(Math.log(Math.abs(step)) * Math.LOG10E));
                    if (stepPower < 1) {
                        // exponent is less then 1 too. Count decimals of exponent
                        var decCount = Math.round(Math.abs(Math.log(Math.abs(stepPower)) * Math.LOG10E)) + 2;
                        // round value to avoid floating point issues
                        value = $math.round(value, decCount);
                    }
                    i++;
                }
                for (var j = i; j < this.dataItems.length; j++) {
                    this.dataItems[j].hide();
                }
                $array.each(this.series, function (series) {
                    if (series.inited) {
                        series._markDirtyAxes();
                    }
                });
                this._updateGhost();
            }
        }
    });
    Object.defineProperty(ValueAxis.prototype, "_prepareDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, count) {
            var renderer = this.get("renderer");
            var value = dataItem.get("value");
            var endValue = dataItem.get("endValue");
            var position = this.valueToPosition(value);
            var endPosition = position;
            var fillEndPosition = this.valueToPosition(value + this.getPrivate("step"));
            if ($type.isNumber(endValue)) {
                endPosition = this.valueToPosition(endValue);
                fillEndPosition = endPosition;
            }
            renderer.updateLabel(dataItem.get("label"), position, endPosition, count);
            var grid = dataItem.get("grid");
            renderer.updateGrid(grid, position, endPosition);
            if (grid) {
                if (value == this.get("baseValue", 0)) {
                    grid.addTag("base");
                    grid._applyThemes();
                }
                else if (grid.hasTag("base")) {
                    grid.removeTag("base");
                    grid._applyThemes();
                }
            }
            renderer.updateTick(dataItem.get("tick"), position, endPosition, count);
            renderer.updateFill(dataItem.get("axisFill"), position, fillEndPosition);
            this._processBullet(dataItem);
            renderer.updateBullet(dataItem.get("bullet"), position, endPosition);
            if (!dataItem.get("isRange")) {
                var fillRule = this.get("fillRule");
                if (fillRule) {
                    fillRule(dataItem);
                }
            }
        }
    });
    Object.defineProperty(ValueAxis.prototype, "_handleRangeChange", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var selectionMin = this.positionToValue(this.get("start", 0));
            var selectionMax = this.positionToValue(this.get("end", 1));
            var gridCount = this.get("renderer").gridCount();
            var minMaxStep = this._adjustMinMax(selectionMin, selectionMax, gridCount, true);
            var stepDecimalPlaces = $utils.decimalPlaces(minMaxStep.step);
            this.setPrivateRaw("stepDecimalPlaces", stepDecimalPlaces);
            selectionMin = $math.round(selectionMin, stepDecimalPlaces);
            selectionMax = $math.round(selectionMax, stepDecimalPlaces);
            minMaxStep = this._adjustMinMax(selectionMin, selectionMax, gridCount, true);
            var step = minMaxStep.step;
            selectionMin = minMaxStep.min;
            selectionMax = minMaxStep.max;
            if (this.getPrivate("selectionMin") !== selectionMin || this.getPrivate("selectionMax") !== selectionMax || this.getPrivate("step") !== step) {
                this.setPrivateRaw("selectionMin", selectionMin);
                this.setPrivateRaw("selectionMax", selectionMax);
                this.setPrivateRaw("step", step);
            }
        }
    });
    /**
     * Converts a relative position to a corresponding numeric value from axis
     * scale.
     *
     * @param   position  Relative position
     * @return            Value
     */
    Object.defineProperty(ValueAxis.prototype, "positionToValue", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (position) {
            var min = this.getPrivate("min");
            var max = this.getPrivate("max");
            if (!this.get("logarithmic")) {
                return position * (max - min) + min;
            }
            else {
                return Math.pow(Math.E, (position * ((Math.log(max) * Math.LOG10E - Math.log(min) * Math.LOG10E)) + Math.log(min) * Math.LOG10E) / Math.LOG10E);
            }
        }
    });
    /**
     * Convers value to a relative position on axis.
     *
     * @param   value  Value
     * @return         Relative position
     */
    Object.defineProperty(ValueAxis.prototype, "valueToPosition", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (value) {
            var min = this.getPrivate("min");
            var max = this.getPrivate("max");
            if (!this.get("logarithmic")) {
                return (value - min) / (max - min);
            }
            else {
                if (value <= 0) {
                    var treatZeroAs = this.get("treatZeroAs");
                    if ($type.isNumber(treatZeroAs)) {
                        value = treatZeroAs;
                    }
                }
                return (Math.log(value) * Math.LOG10E - Math.log(min) * Math.LOG10E) / ((Math.log(max) * Math.LOG10E - Math.log(min) * Math.LOG10E));
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(ValueAxis.prototype, "valueToFinalPosition", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (value) {
            var min = this.getPrivate("minFinal");
            var max = this.getPrivate("maxFinal");
            if (!this.get("logarithmic")) {
                return (value - min) / (max - min);
            }
            else {
                if (value <= 0) {
                    var treatZeroAs = this.get("treatZeroAs");
                    if ($type.isNumber(treatZeroAs)) {
                        value = treatZeroAs;
                    }
                }
                return (Math.log(value) * Math.LOG10E - Math.log(min) * Math.LOG10E) / ((Math.log(max) * Math.LOG10E - Math.log(min) * Math.LOG10E));
            }
        }
    });
    /**
     * Returns X coordinate in pixels corresponding to specific value.
     *
     * @param   value     Numeric value
     * @param   location  Location
     * @param   baseValue Base value
     * @return            X coordinate
     */
    Object.defineProperty(ValueAxis.prototype, "getX", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (value, location, baseValue) {
            value = baseValue + (value - baseValue) * location;
            var position = this.valueToPosition(value);
            return this._settings.renderer.positionToCoordinate(position);
        }
    });
    /**
     * Returns X coordinate in pixels corresponding to specific value.
     *
     * @param   value     Numeric value
     * @param   location  Location
     * @param   baseValue Base value
     * @return            X coordinate
     */
    Object.defineProperty(ValueAxis.prototype, "getY", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (value, location, baseValue) {
            value = baseValue + (value - baseValue) * location;
            var position = this.valueToPosition(value);
            return this._settings.renderer.positionToCoordinate(position);
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(ValueAxis.prototype, "getDataItemCoordinateX", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, field, _cellLocation, axisLocation) {
            return this._settings.renderer.positionToCoordinate(this.getDataItemPositionX(dataItem, field, _cellLocation, axisLocation));
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(ValueAxis.prototype, "getDataItemPositionX", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, field, _cellLocation, axisLocation) {
            var value = dataItem.get(field);
            var stackToItem = dataItem.get("stackToItemX");
            if (stackToItem) {
                var series = dataItem.component;
                value = value * axisLocation + series.getStackedXValueWorking(dataItem, field);
            }
            else {
                value = this._baseValue + (value - this._baseValue) * axisLocation;
            }
            return this.valueToPosition(value);
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(ValueAxis.prototype, "getDataItemCoordinateY", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, field, _cellLocation, axisLocation) {
            return this._settings.renderer.positionToCoordinate(this.getDataItemPositionY(dataItem, field, _cellLocation, axisLocation));
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(ValueAxis.prototype, "getDataItemPositionY", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem, field, _cellLocation, axisLocation) {
            var value = dataItem.get(field);
            var stackToItem = dataItem.get("stackToItemY");
            if (stackToItem) {
                var series = dataItem.component;
                value = value * axisLocation + series.getStackedYValueWorking(dataItem, field);
            }
            else {
                value = this._baseValue + (value - this._baseValue) * axisLocation;
            }
            return this.valueToPosition(value);
        }
    });
    /**
     * Returns relative position of axis' `baseValue`.
     *
     * @return  Base value position
     */
    Object.defineProperty(ValueAxis.prototype, "basePosition", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this.valueToPosition(this.baseValue());
        }
    });
    /**
     * Base value of the [[ValueAxis]], which determines positive and negative
     * values.
     *
     * @return Base value
     */
    Object.defineProperty(ValueAxis.prototype, "baseValue", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var min = Math.min(this.getPrivate("minFinal", -Infinity), this.getPrivate("selectionMin", -Infinity));
            var max = Math.max(this.getPrivate("maxFinal", Infinity), this.getPrivate("selectionMax", Infinity));
            var baseValue = this.get("baseValue", 0);
            if (baseValue < min) {
                baseValue = min;
            }
            if (baseValue > max) {
                baseValue = max;
            }
            return baseValue;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(ValueAxis.prototype, "cellEndValue", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (value) {
            return value;
        }
    });
    Object.defineProperty(ValueAxis.prototype, "fixSmallStep", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (step) {
            // happens because of floating point error
            if (1 + step === 1) {
                step *= 2;
                return this.fixSmallStep(step);
            }
            return step;
        }
    });
    Object.defineProperty(ValueAxis.prototype, "_fixMin", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (min) {
            return min;
        }
    });
    Object.defineProperty(ValueAxis.prototype, "_fixMax", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (max) {
            return max;
        }
    });
    Object.defineProperty(ValueAxis.prototype, "_calculateTotals", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (this.get("calculateTotals")) {
                var series = this.series[0];
                if (series) {
                    var startIndex = series.startIndex();
                    if (series.dataItems.length > 0) {
                        if (startIndex > 0) {
                            startIndex--;
                        }
                        var endIndex = series.endIndex();
                        if (endIndex < series.dataItems.length) {
                            endIndex++;
                        }
                        var field_1;
                        var vc_1;
                        if (series.get("yAxis") == this) {
                            field_1 = "valueY";
                            vc_1 = "vcy";
                        }
                        else if (series.get("xAxis") == this) {
                            field_1 = "valueX";
                            vc_1 = "vcx";
                        }
                        var fieldWorking_1 = field_1 + "Working";
                        if (field_1) {
                            var _loop_1 = function (i) {
                                var sum = 0;
                                var total = 0;
                                $array.each(this_1.series, function (series) {
                                    if (!series.get("excludeFromTotal")) {
                                        var dataItem = series.dataItems[i];
                                        if (dataItem) {
                                            var value = dataItem.get(fieldWorking_1) * series.get(vc_1);
                                            if (!$type.isNaN(value)) {
                                                sum += value;
                                                total += Math.abs(value);
                                            }
                                        }
                                    }
                                });
                                $array.each(this_1.series, function (series) {
                                    if (!series.get("excludeFromTotal")) {
                                        var dataItem = series.dataItems[i];
                                        if (dataItem) {
                                            var value = dataItem.get(fieldWorking_1) * series.get(vc_1);
                                            if (!$type.isNaN(value)) {
                                                dataItem.set((field_1 + "Total"), total);
                                                dataItem.set((field_1 + "Sum"), sum);
                                                dataItem.set((field_1 + "TotalPercent"), value / total * 100);
                                            }
                                        }
                                    }
                                });
                            };
                            var this_1 = this;
                            for (var i = startIndex; i < endIndex; i++) {
                                _loop_1(i);
                            }
                        }
                    }
                }
            }
        }
    });
    Object.defineProperty(ValueAxis.prototype, "_getSelectionMinMax", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _a;
            var _this = this;
            var min = this.getPrivate("minFinal");
            var max = this.getPrivate("maxFinal");
            var minDefined = this.get("min");
            var maxDefined = this.get("max");
            var extraMin = this.get("extraMin", 0);
            var extraMax = this.get("extraMax", 0);
            var gridCount = this.get("renderer").gridCount();
            var strictMinMax = this.get("strictMinMax", false);
            if ($type.isNumber(min) && $type.isNumber(max)) {
                var selectionMin_1 = max;
                var selectionMax_1 = min;
                $array.each(this.series, function (series) {
                    if (!series.get("ignoreMinMax")) {
                        var seriesMin = void 0;
                        var seriesMax = void 0;
                        if (series.get("xAxis") === _this) {
                            seriesMin = series.getPrivate("selectionMinX", series.getPrivate("minX"));
                            seriesMax = series.getPrivate("selectionMaxX", series.getPrivate("maxX"));
                        }
                        else if (series.get("yAxis") === _this) {
                            seriesMin = series.getPrivate("selectionMinY", series.getPrivate("minY"));
                            seriesMax = series.getPrivate("selectionMaxY", series.getPrivate("maxY"));
                        }
                        if (!series.isHidden() && !series.isShowing()) {
                            if ($type.isNumber(seriesMin)) {
                                selectionMin_1 = Math.min(selectionMin_1, seriesMin);
                            }
                            if ($type.isNumber(seriesMax)) {
                                selectionMax_1 = Math.max(selectionMax_1, seriesMax);
                            }
                        }
                    }
                });
                this.axisRanges.each(function (range) {
                    if (range.get("affectsMinMax")) {
                        var value = range.get("value");
                        if (value != null) {
                            selectionMin_1 = Math.min(selectionMin_1, value);
                            selectionMax_1 = Math.max(selectionMax_1, value);
                        }
                        value = range.get("endValue");
                        if (value != null) {
                            selectionMin_1 = Math.min(selectionMin_1, value);
                            selectionMax_1 = Math.max(selectionMax_1, value);
                        }
                    }
                });
                if (selectionMin_1 > selectionMax_1) {
                    _a = __read([selectionMax_1, selectionMin_1], 2), selectionMin_1 = _a[0], selectionMax_1 = _a[1];
                }
                if ($type.isNumber(minDefined)) {
                    if (strictMinMax) {
                        selectionMin_1 = minDefined;
                    }
                    else {
                        selectionMin_1 = min;
                    }
                }
                else if (strictMinMax) {
                    if ($type.isNumber(this._minReal)) {
                        selectionMin_1 = this._minReal;
                    }
                }
                if ($type.isNumber(maxDefined)) {
                    if (strictMinMax) {
                        selectionMax_1 = maxDefined;
                    }
                    else {
                        selectionMax_1 = max;
                    }
                }
                else if (strictMinMax) {
                    if ($type.isNumber(this._maxReal)) {
                        selectionMax_1 = this._maxReal;
                    }
                }
                if (selectionMin_1 === selectionMax_1) {
                    selectionMin_1 -= this._deltaMinMax;
                    selectionMax_1 += this._deltaMinMax;
                    var minMaxStep2 = this._adjustMinMax(selectionMin_1, selectionMax_1, gridCount, strictMinMax);
                    selectionMin_1 = minMaxStep2.min;
                    selectionMax_1 = minMaxStep2.max;
                }
                var minMaxStep = this._adjustMinMax(selectionMin_1, selectionMax_1, gridCount);
                selectionMin_1 = minMaxStep.min;
                selectionMax_1 = minMaxStep.max;
                selectionMin_1 -= (selectionMax_1 - selectionMin_1) * extraMin;
                selectionMax_1 += (selectionMax_1 - selectionMin_1) * extraMax;
                selectionMin_1 = $math.fitToRange(selectionMin_1, min, max);
                selectionMax_1 = $math.fitToRange(selectionMax_1, min, max);
                // do it for the second time !important			
                minMaxStep = this._adjustMinMax(selectionMin_1, selectionMax_1, gridCount, true);
                if (!strictMinMax) {
                    selectionMin_1 = minMaxStep.min;
                    selectionMax_1 = minMaxStep.max;
                }
                var syncWithAxis = this.get("syncWithAxis");
                if (syncWithAxis) {
                    minMaxStep = this._syncAxes(selectionMin_1, selectionMax_1, minMaxStep.step, syncWithAxis.getPrivate("selectionMinFinal", syncWithAxis.getPrivate("minFinal", 0)), syncWithAxis.getPrivate("selectionMaxFinal", syncWithAxis.getPrivate("maxFinal", 1)), syncWithAxis.getPrivate("selectionStepFinal", syncWithAxis.getPrivate("step", 1)));
                    selectionMin_1 = minMaxStep.min;
                    selectionMax_1 = minMaxStep.max;
                }
                if (strictMinMax) {
                    if ($type.isNumber(minDefined)) {
                        selectionMin_1 = Math.max(selectionMin_1, minDefined);
                    }
                    if ($type.isNumber(maxDefined)) {
                        selectionMax_1 = Math.min(selectionMax_1, maxDefined);
                    }
                }
                var start = this.valueToFinalPosition(selectionMin_1);
                var end = this.valueToFinalPosition(selectionMax_1);
                this.setPrivateRaw("selectionMinFinal", selectionMin_1);
                this.setPrivateRaw("selectionMaxFinal", selectionMax_1);
                this.setPrivateRaw("selectionStepFinal", minMaxStep.step);
                this.zoom(start, end);
            }
        }
    });
    Object.defineProperty(ValueAxis.prototype, "_getMinMax", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            var minDefined = this.get("min");
            var maxDefined = this.get("max");
            var min = Infinity;
            var max = -Infinity;
            var extraMin = this.get("extraMin", 0);
            var extraMax = this.get("extraMax", 0);
            var minDiff = Infinity;
            $array.each(this.series, function (series) {
                if (!series.get("ignoreMinMax")) {
                    var seriesMin = void 0;
                    var seriesMax = void 0;
                    if (series.get("xAxis") === _this) {
                        seriesMin = series.getPrivate("minX");
                        seriesMax = series.getPrivate("maxX");
                    }
                    else if (series.get("yAxis") === _this) {
                        seriesMin = series.getPrivate("minY");
                        seriesMax = series.getPrivate("maxY");
                    }
                    if ($type.isNumber(seriesMin) && $type.isNumber(seriesMax)) {
                        min = Math.min(min, seriesMin);
                        max = Math.max(max, seriesMax);
                        var diff = seriesMax - seriesMin;
                        if (diff <= 0) {
                            diff = Math.abs(seriesMax / 100);
                        }
                        if (diff < minDiff) {
                            minDiff = diff;
                        }
                    }
                }
            });
            this.axisRanges.each(function (range) {
                if (range.get("affectsMinMax")) {
                    var value = range.get("value");
                    if (value != null) {
                        min = Math.min(min, value);
                        max = Math.max(max, value);
                    }
                    value = range.get("endValue");
                    if (value != null) {
                        min = Math.min(min, value);
                        max = Math.max(max, value);
                    }
                }
            });
            if (this.get("logarithmic")) {
                var treatZeroAs = this.get("treatZeroAs");
                if ($type.isNumber(treatZeroAs)) {
                    if (min <= 0) {
                        min = treatZeroAs;
                    }
                }
                if (min <= 0) {
                    new Error("Logarithmic value axis can not have values <= 0.");
                }
            }
            if (min === 0 && max === 0) {
                max = 0.9;
                min = -0.9;
            }
            if ($type.isNumber(minDefined)) {
                min = minDefined;
            }
            if ($type.isNumber(maxDefined)) {
                max = maxDefined;
            }
            // meaning no min/max found on series/ranges and no min/max was defined
            if (min === Infinity || max === -Infinity) {
                return;
            }
            // adapter
            var minAdapted = this.adapters.fold("min", min);
            var maxAdapted = this.adapters.fold("max", max);
            if ($type.isNumber(minAdapted)) {
                min = minAdapted;
            }
            if ($type.isNumber(maxAdapted)) {
                max = maxAdapted;
            }
            // DateAxis does some magic here
            min = this._fixMin(min);
            max = this._fixMax(max);
            // this happens if starLocation and endLocation are 0.5 and DateAxis has only one date
            if (max - min <= 1 / Math.pow(10, 15)) {
                if (max - min !== 0) {
                    this._deltaMinMax = (max - min) / 2;
                }
                else {
                    // the number by which we need to raise 10 to get difference
                    var exponent = Math.log(Math.abs(max)) * Math.LOG10E;
                    // here we find a number which is power of 10 and has the same count of numbers as difference has
                    var power = Math.pow(10, Math.floor(exponent));
                    // reduce this number by 10 times
                    power = power / 10;
                    this._deltaMinMax = power;
                }
                min -= this._deltaMinMax;
                max += this._deltaMinMax;
            }
            // add extras
            min -= (max - min) * extraMin;
            max += (max - min) * extraMax;
            this._minReal = min;
            this._maxReal = max;
            var strict = this.get("strictMinMax");
            if ($type.isNumber(maxDefined)) {
                strict = true;
            }
            var gridCount = this.get("renderer").gridCount();
            var minMaxStep = this._adjustMinMax(min, max, gridCount, strict);
            min = minMaxStep.min;
            max = minMaxStep.max;
            // do it for the second time with strict true (importat!)
            minMaxStep = this._adjustMinMax(min, max, gridCount, true);
            min = minMaxStep.min;
            max = minMaxStep.max;
            // return min max if strict
            if (this.get("strictMinMax")) {
                if ($type.isNumber(minDefined)) {
                    min = minDefined;
                }
                else {
                    min = this._minReal;
                }
                if ($type.isNumber(maxDefined)) {
                    max = maxDefined;
                }
                else {
                    max = this._maxReal;
                }
                if (max - min <= 0.00000001) {
                    min -= this._deltaMinMax;
                    max += this._deltaMinMax;
                }
                min -= (max - min) * extraMin;
                max += (max - min) * extraMax;
            }
            minAdapted = this.adapters.fold("min", min);
            maxAdapted = this.adapters.fold("max", max);
            if ($type.isNumber(minAdapted)) {
                min = minAdapted;
            }
            if ($type.isNumber(maxAdapted)) {
                max = maxAdapted;
            }
            if (minDiff == Infinity) {
                minDiff = (max - min);
            }
            var syncWithAxis = this.get("syncWithAxis");
            if (syncWithAxis) {
                minMaxStep = this._syncAxes(min, max, minMaxStep.step, syncWithAxis.getPrivate("minFinal", syncWithAxis.getPrivate("min", 0)), syncWithAxis.getPrivate("maxFinal", syncWithAxis.getPrivate("max", 1)), syncWithAxis.getPrivate("step", 1));
                min = minMaxStep.min;
                max = minMaxStep.max;
            }
            this.setPrivateRaw("maxZoomFactor", (max - min) / minDiff * this.get("maxZoomFactor", 100));
            if ($type.isNumber(min) && $type.isNumber(max)) {
                if (this.getPrivate("minFinal") !== min || this.getPrivate("maxFinal") !== max) {
                    this.setPrivate("minFinal", min);
                    this.setPrivate("maxFinal", max);
                    var duration = this.get("interpolationDuration", 0);
                    var easing = this.get("interpolationEasing");
                    this.animatePrivate({ key: "min", to: min, duration: duration, easing: easing });
                    this.animatePrivate({ key: "max", to: max, duration: duration, easing: easing });
                }
            }
        }
    });
    Object.defineProperty(ValueAxis.prototype, "_adjustMinMax", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (min, max, gridCount, strictMode) {
            var logarithmic = this.get("logarithmic");
            // will fail if 0
            if (gridCount <= 1) {
                gridCount = 1;
            }
            gridCount = Math.round(gridCount);
            var initialMin = min;
            var initialMax = max;
            var difference = max - min;
            // in case min and max is the same, use max
            if (difference === 0) {
                difference = Math.abs(max);
            }
            // the number by which we need to raise 10 to get difference
            var exponent = Math.log(Math.abs(difference)) * Math.LOG10E;
            // here we find a number which is power of 10 and has the same count of numbers as difference has
            var power = Math.pow(10, Math.floor(exponent));
            // reduce this number by 10 times
            power = power / 10;
            var extra = power;
            if (strictMode) {
                extra = 0;
            }
            if (!logarithmic) {
                // round down min
                if (strictMode) {
                    min = Math.floor(min / power) * power;
                    // round up max
                    max = Math.ceil(max / power) * power;
                }
                else {
                    min = Math.ceil(min / power) * power - extra;
                    // round up max
                    max = Math.floor(max / power) * power + extra;
                }
                // don't let min go below 0 if real min is >= 0
                if (min < 0 && initialMin >= 0) {
                    min = 0;
                }
                // don't let max go above 0 if real max is <= 0
                if (max > 0 && initialMax <= 0) {
                    max = 0;
                }
            }
            // logarithmic
            else {
                if (min <= 0) {
                    //throw Error("Logarithmic value axis can not have values <= 0.");
                    min = this.get("baseValue", 0);
                }
                if (min === Infinity) {
                    min = 1;
                }
                if (max === -Infinity) {
                    max = 10;
                }
                min = Math.pow(10, Math.floor(Math.log(Math.abs(min)) * Math.LOG10E));
                max = Math.pow(10, Math.ceil(Math.log(Math.abs(max)) * Math.LOG10E));
                if (this.get("strictMinMax")) {
                    var minDefined = this.get("min");
                    var maxDefined = this.get("max");
                    if ($type.isNumber(minDefined) && minDefined > 0) {
                        min = minDefined;
                    }
                    if ($type.isNumber(maxDefined) && maxDefined > 0) {
                        max = maxDefined;
                    }
                }
            }
            exponent = Math.log(Math.abs(difference)) * Math.LOG10E;
            power = Math.pow(10, Math.floor(exponent));
            power = power / 100; // used to be 10 in v4, but this caused issue that there could be limited number of grids with even very small minGridDistance
            // approximate difference between two grid lines
            var step = Math.ceil((difference / gridCount) / power) * power;
            var stepPower = Math.pow(10, Math.floor(Math.log(Math.abs(step)) * Math.LOG10E));
            // the step should divide by  2, 5, and 10.
            var stepDivisor = Math.ceil(step / stepPower); // number 0 - 10
            if (stepDivisor > 5) {
                stepDivisor = 10;
            }
            else if (stepDivisor <= 5 && stepDivisor > 2) {
                stepDivisor = 5;
            }
            // now get real step
            step = Math.ceil(step / (stepPower * stepDivisor)) * stepPower * stepDivisor;
            var maxPrecision = this.get("maxPrecision");
            if ($type.isNumber(maxPrecision)) {
                var ceiledStep = $math.ceil(step, maxPrecision);
                if (maxPrecision < Number.MAX_VALUE && step !== ceiledStep) {
                    step = ceiledStep;
                }
            }
            var decCount = 0;
            // in case numbers are smaller than 1
            if (stepPower < 1) {
                // exponent is less then 1 too. Count decimals of exponent
                decCount = Math.round(Math.abs(Math.log(Math.abs(stepPower)) * Math.LOG10E)) + 1;
                // round step
                step = $math.round(step, decCount);
            }
            if (!logarithmic) {
                // final min and max
                var minCount = Math.floor(min / step);
                min = $math.round(step * minCount, decCount);
                var maxCount = void 0;
                if (!strictMode) {
                    maxCount = Math.ceil(max / step);
                }
                else {
                    maxCount = Math.floor(max / step);
                }
                if (maxCount === minCount) {
                    maxCount++;
                }
                max = $math.round(step * maxCount, decCount);
                if (max < initialMax) {
                    max = max + step;
                }
                if (min > initialMin) {
                    min = min - step;
                }
            }
            step = this.fixSmallStep(step);
            return { min: min, max: max, step: step };
        }
    });
    /**
     * Returns text to be used in an axis tooltip for specific relative position.
     *
     * @param   position  Position
     * @return            Tooltip text
     */
    Object.defineProperty(ValueAxis.prototype, "getTooltipText", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (position) {
            var numberFormat = this.get("tooltipNumberFormat", this.get("numberFormat"));
            var formatter = this.getNumberFormatter();
            var extraDecimals = this.get("extraTooltipPrecision", 0);
            var decimals = this.getPrivate("stepDecimalPlaces", 0) + extraDecimals;
            var value = $math.round(this.positionToValue(position), decimals);
            if (numberFormat) {
                return formatter.format(value, numberFormat);
            }
            else {
                return formatter.format(value, undefined, decimals);
                //label.set("text", this.getNumberFormatter().format(value, undefined, this.getPrivate("stepDecimalPlaces")));
            }
            // //@todo number formatter + tag
            // return $math.round(this.positionToValue(position), this.getPrivate("stepDecimalPlaces")).toString();
        }
    });
    /**
     * Returns a data item from series that is closest to the `position`.
     *
     * @param   series    Series
     * @param   position  Relative position
     * @return            Data item
     */
    Object.defineProperty(ValueAxis.prototype, "getSeriesItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (series, position) {
            var fieldName = (this.getPrivate("name") + this.get("renderer").getPrivate("letter"));
            var value = this.positionToValue(position);
            var index = undefined;
            var oldDiff;
            $array.each(series.dataItems, function (dataItem, i) {
                var diff = Math.abs(dataItem.get(fieldName) - value);
                if (index === undefined || diff < oldDiff) {
                    index = i;
                    oldDiff = diff;
                }
            });
            if (index != null) {
                return series.dataItems[index];
            }
        }
    });
    /**
     * Zooms the axis to specific `start` and `end` values.
     *
     * Optional `duration` specifies duration of zoom animation in milliseconds.
     *
     * @param  start     Start value
     * @param  end       End value
     * @param  duration  Duration in milliseconds
     */
    Object.defineProperty(ValueAxis.prototype, "zoomToValues", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (start, end, duration) {
            var min = this.getPrivate("minFinal", 0);
            var max = this.getPrivate("maxFinal", 0);
            if (this.getPrivate("min") != null && this.getPrivate("max") != null) {
                this.zoom((start - min) / (max - min), (end - min) / (max - min), duration);
            }
        }
    });
    /**
     * Syncs with a target axis.
     *
     * @param  min  Min
     * @param  max  Max
     * @param  step Step
     */
    Object.defineProperty(ValueAxis.prototype, "_syncAxes", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (min, max, step, syncMin, syncMax, syncStep) {
            var axis = this.get("syncWithAxis");
            if (axis) {
                var count = Math.round(syncMax - syncMin) / syncStep;
                var currentCount = Math.round((max - min) / step);
                var gridCount = this.get("renderer").gridCount();
                if ($type.isNumber(count) && $type.isNumber(currentCount)) {
                    var synced = false;
                    var c = 0;
                    var diff = (max - min) * 0.01;
                    var omin = min;
                    var omax = max;
                    var ostep = step;
                    while (synced != true) {
                        synced = this._checkSync(omin, omax, ostep, count);
                        c++;
                        if (c > 500) {
                            synced = true;
                        }
                        if (!synced) {
                            if (c / 3 == Math.round(c / 3)) {
                                omin = min - diff * c;
                                if (min >= 0 && omin < 0) {
                                    omin = 0;
                                }
                            }
                            else {
                                omax = max + diff * c;
                                if (omax <= 0 && omax > 0) {
                                    omax = 0;
                                }
                            }
                            var minMaxStep = this._adjustMinMax(omin, omax, gridCount, true);
                            omin = minMaxStep.min;
                            omax = minMaxStep.max;
                            ostep = minMaxStep.step;
                        }
                        else {
                            min = omin;
                            max = omax;
                            step = ostep;
                        }
                    }
                }
            }
            return { min: min, max: max, step: step };
        }
    });
    /**
     * Returns `true` if axis needs to be resunced with some other axis.
     */
    Object.defineProperty(ValueAxis.prototype, "_checkSync", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (min, max, step, count) {
            var currentCount = (max - min) / step;
            for (var i = 1; i < count; i++) {
                if ($math.round(currentCount / i, 1) == count || currentCount * i == count) {
                    return true;
                }
            }
            return false;
        }
    });
    Object.defineProperty(ValueAxis, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "ValueAxis"
    });
    Object.defineProperty(ValueAxis, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: Axis.classNames.concat([ValueAxis.className])
    });
    return ValueAxis;
}(Axis));
export { ValueAxis };
//# sourceMappingURL=ValueAxis.js.map