import { Container } from "./render/Container";
import { HorizontalLayout } from "./render/HorizontalLayout";
import { VerticalLayout } from "./render/VerticalLayout";
import { GridLayout } from "./render/GridLayout";
import { Disposer } from "./util/Disposer";
import { ResizeSensor } from "./util/ResizeSensor";
import { InterfaceColors } from "./util/InterfaceColors";
import { Graphics } from "./render/Graphics";
import { Rectangle } from "./render/Rectangle";
import { Tooltip } from "./render/Tooltip";
import { NumberFormatter } from "./util/NumberFormatter";
import { DateFormatter } from "./util/DateFormatter";
import { DurationFormatter } from "./util/DurationFormatter";
import { Language } from "./util/Language";
import { EventDispatcher } from "./util/EventDispatcher";
import { DefaultTheme } from "../themes/DefaultTheme";
import { CanvasRenderer } from "./render/backend/CanvasRenderer";
import { p100, percent } from "./util/Percent";
import { color } from "./util/Color";
import { populateString } from "./util/PopulateString";
import { registry } from "./Registry";
import * as $order from "./util/Order";
import * as $array from "./util/Array";
import * as $object from "./util/Object";
import * as $utils from "./util/Utils";
import en from "../../locales/en";
function rAF(fps, callback) {
    if (fps == null) {
        requestAnimationFrame(callback);
    }
    else {
        setTimeout(function () {
            requestAnimationFrame(callback);
        }, 1000 / fps);
    }
}
// TODO implement Disposer
/**
 * Root element of the chart.
 *
 * @see {@link https://www.amcharts.com/docs/v5/getting-started/#Root_element} for more info
 */
var Root = /** @class */ (function () {
    function Root(id, settings, isReal) {
        if (settings === void 0) { settings = {}; }
        /**
         * A reference to original chart container (div element).
         */
        Object.defineProperty(this, "dom", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_inner", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_isDirty", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(this, "_isDirtyParents", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(this, "_dirty", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        Object.defineProperty(this, "_dirtyParents", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        Object.defineProperty(this, "_dirtyBounds", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        Object.defineProperty(this, "_dirtyPositions", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        Object.defineProperty(this, "_ticker", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: null
        });
        Object.defineProperty(this, "_tickers", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        /**
         * Root's event dispatcher.
         *
         * @see {@link https://www.amcharts.com/docs/v5/concepts/events/} for more info
         */
        Object.defineProperty(this, "events", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new EventDispatcher()
        });
        /**
         * @ignore
         * @todo needs description
         */
        Object.defineProperty(this, "animationTime", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: null
        });
        Object.defineProperty(this, "_animations", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        Object.defineProperty(this, "_renderer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_rootContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        /**
         * Main content container.
         */
        Object.defineProperty(this, "container", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        /**
         * A [[Container]] used to display tooltips in.
         */
        Object.defineProperty(this, "tooltipContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_tooltip", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        // Locale-related
        /**
         * @ignore
         */
        Object.defineProperty(this, "language", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: Language.new(this, {})
        });
        /**
         * Locale used by the chart.
         *
         * @see {@link https://www.amcharts.com/docs/v5/concepts/locales/}
         */
        Object.defineProperty(this, "locale", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: en
        });
        // Date-time related
        /**
         * Use UTC when formatting date/time.
         *
         * @see {@link https://www.amcharts.com/docs/v5/concepts/formatters/formatting-dates/#UTC_and_time_zones} for more info
         */
        Object.defineProperty(this, "utc", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        /**
         * If set, will format date/time in specific time zone.
         *
         * The value should be named time zone, e.g.:
         * `"America/Vancouver"`, `"Australia/Sydney"`, `"UTC"`.
         *
         * @see {@link https://www.amcharts.com/docs/v5/getting-started/root-element/#Time_zone} for more info
         * @since 5.1.0
         */
        Object.defineProperty(this, "timezone", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        /**
         * The maximum FPS that the Root will run at.
         *
         * If `undefined` it will run at the highest FPS.
         *
         * @see {@link https://www.amcharts.com/docs/v5/getting-started/root-element/#Performance} for more info
         */
        Object.defineProperty(this, "fps", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        /**
         * Number formatter.
         *
         * @see {@link https://www.amcharts.com/docs/v5/concepts/formatters/formatting-numbers/} for more info
         */
        Object.defineProperty(this, "numberFormatter", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: NumberFormatter.new(this, {})
        });
        /**
         * Date/time formatter.
         *
         * @see {@link https://www.amcharts.com/docs/v5/concepts/formatters/formatting-dates/} for more info
         */
        Object.defineProperty(this, "dateFormatter", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: DateFormatter.new(this, {})
        });
        /**
         * Duration formatter.
         *
         * @see {@link https://www.amcharts.com/docs/v5/concepts/formatters/formatting-dates/} for more info
         */
        Object.defineProperty(this, "durationFormatter", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: DurationFormatter.new(this, {})
        });
        // Accessibility
        /**
         * Global tab index for using for the whole chart
         *
         * @see {@link https://www.amcharts.com/docs/v5/concepts/accessibility/} for more info
         */
        Object.defineProperty(this, "tabindex", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        //@todo maybe make this better
        Object.defineProperty(this, "_tabindexes", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        Object.defineProperty(this, "_focusElementDirty", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(this, "_focusElementContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_focusedSprite", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_keyboardDragPoint", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_tooltipElementContainer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_readerAlertElement", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_logo", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        /**
         * Used for dynamically-created CSS and JavaScript with strict source policies.
         */
        Object.defineProperty(this, "nonce", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        /**
         * Special color set to be used for various controls.
         *
         * @see {@link https://www.amcharts.com/docs/v5/concepts/colors-gradients-and-patterns/#Interface_colors} for more info
         */
        Object.defineProperty(this, "interfaceColors", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        /**
         * An instance of vertical layout object that can be used to set `layout` setting
         * of a [[Container]].
         *
         * @default VerticalLayout.new()
         */
        Object.defineProperty(this, "verticalLayout", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: VerticalLayout.new(this, {})
        });
        /**
         * An instance of horizontal layout object that can be used to set `layout` setting
         * of a [[Container]].
         *
         * @default HorizontalLayout.new()
         */
        Object.defineProperty(this, "horizontalLayout", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: HorizontalLayout.new(this, {})
        });
        /**
         * An instance of grid layout object that can be used to set `layout` setting
         * of a [[Container]].
         *
         * @default VerticalLayout.new()
         */
        Object.defineProperty(this, "gridLayout", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: GridLayout.new(this, {})
        });
        /**
         * Indicates whether chart should resized automatically when parent container
         * width and/or height changes.
         *
         * If disabled (`autoResize = false`) you can make the chart resize manually
         * by calling root element's `resize()` method.
         */
        Object.defineProperty(this, "autoResize", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: true
        });
        Object.defineProperty(this, "_isDisposed", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(this, "_disposers", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        Object.defineProperty(this, "_resizeSensorDisposer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "_tooltips", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        if (!isReal) {
            throw new Error("You cannot use `new Class()`, instead use `Class.new()`");
        }
        if (settings.useSafeResolution == null) {
            settings.useSafeResolution = true;
        }
        var resolution;
        if (settings.useSafeResolution) {
            resolution = $utils.getSafeResolution();
        }
        this._renderer = new CanvasRenderer(resolution);
        var dom;
        if (id instanceof HTMLElement) {
            dom = id;
        }
        else {
            dom = document.getElementById(id);
        }
        $array.each(registry.rootElements, function (root) {
            if (root.dom === dom) {
                throw new Error("You cannot have multiple Roots on the same DOM node");
            }
        });
        this.interfaceColors = InterfaceColors.new(this, {});
        if (dom === null) {
            throw new Error("Could not find HTML element with id `" + id + "`");
        }
        this.dom = dom;
        var inner = document.createElement("div");
        inner.style.position = "relative";
        dom.appendChild(inner);
        this._inner = inner;
        registry.rootElements.push(this);
    }
    Object.defineProperty(Root, "new", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (id, settings) {
            var root = new Root(id, settings, true);
            root._init();
            return root;
        }
    });
    Object.defineProperty(Root.prototype, "moveDOM", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (id) {
            var dom;
            if (id instanceof HTMLElement) {
                dom = id;
            }
            else {
                dom = document.getElementById(id);
            }
            if (dom) {
                while (this.dom.childNodes.length > 0) {
                    dom.appendChild(this.dom.childNodes[0]);
                }
                this.dom = dom;
                this._initResizeSensor();
                this.resize();
            }
        }
    });
    Object.defineProperty(Root.prototype, "_handleLogo", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (this._logo) {
                var w = this.dom.offsetWidth;
                var h = this.dom.offsetHeight;
                if ((w <= 150) || (h <= 60)) {
                    this._logo.hide();
                }
                else {
                    this._logo.show();
                }
            }
        }
    });
    Object.defineProperty(Root.prototype, "_showBranding", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (!this._logo) {
                var logo = this.tooltipContainer.children.push(Container.new(this, {
                    interactive: true,
                    interactiveChildren: false,
                    position: "absolute",
                    setStateOnChildren: true,
                    paddingTop: 9,
                    paddingRight: 9,
                    paddingBottom: 9,
                    paddingLeft: 9,
                    scale: .6,
                    y: percent(100),
                    centerY: p100,
                    tooltipText: "Created using amCharts 5",
                    tooltipX: p100,
                    cursorOverStyle: "pointer",
                    background: Rectangle.new(this, {
                        fill: color(0x474758),
                        fillOpacity: 0,
                        tooltipY: 5
                    })
                }));
                var tooltip = Tooltip.new(this, {
                    pointerOrientation: "horizontal",
                    paddingTop: 4,
                    paddingRight: 7,
                    paddingBottom: 4,
                    paddingLeft: 7
                });
                tooltip.label.setAll({
                    fontSize: 12
                });
                tooltip.get("background").setAll({
                    fill: this.interfaceColors.get("background"),
                    stroke: this.interfaceColors.get("grid"),
                    strokeOpacity: 0.3
                });
                logo.set("tooltip", tooltip);
                logo.events.on("click", function () {
                    window.open("https://www.amcharts.com/", "_blank");
                });
                logo.states.create("hover", {});
                var m = logo.children.push(Graphics.new(this, {
                    stroke: color(0xcccccc),
                    strokeWidth: 3,
                    svgPath: "M5 25 L13 25h13.6c3.4 0 6 0 10.3-4.3s5.2-12 8.6-12c3.4 0 4.3 8.6 7.7 8.6M83.4 25H79.8c-3.4 0-6 0-10.3-4.3s-5.2-12-8.6-12-4.3 8.6-7.7 8.6"
                }));
                m.states.create("hover", { stroke: color(0x3CABFF) });
                var a = logo.children.push(Graphics.new(this, {
                    stroke: color(0x888888),
                    strokeWidth: 3,
                    svgPath: "M83.4 25h-31C37 25 39.5 4.4 28.4 4.4S18.9 24.2 4.3 25H0"
                }));
                a.states.create("hover", { stroke: color(0x474758) });
                //logo.set("tooltip", this._tooltip);
                //logo.setPrivate("tooltipTarget", logo.get("background"));
                this._logo = logo;
                this._handleLogo();
            }
        }
    });
    Object.defineProperty(Root.prototype, "_init", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            var renderer = this._renderer;
            var rootContainer = Container.new(this, { visible: true, width: this.dom.clientWidth, height: this.dom.clientHeight });
            this._rootContainer = rootContainer;
            this._rootContainer._defaultThemes.push(DefaultTheme.new(this));
            var container = rootContainer.children.push(Container.new(this, { visible: true, width: p100, height: p100 }));
            this.container = container;
            renderer.resize(this.dom.clientWidth, this.dom.clientHeight);
            //@todo: better appendChild - refer
            this._inner.appendChild(renderer.view);
            // TODO: TMP TMP TMP for testing only, remove
            //document.body.appendChild((<any>renderer)._ghostView);
            this._initResizeSensor();
            // Create element which is used to make announcements to screen reader
            var readerAlertElement = document.createElement("div");
            readerAlertElement.setAttribute("role", "alert");
            readerAlertElement.style.zIndex = "-100000";
            readerAlertElement.style.opacity = "0";
            readerAlertElement.style.position = "absolute";
            readerAlertElement.style.top = "0";
            this._readerAlertElement = readerAlertElement;
            this._inner.appendChild(this._readerAlertElement);
            var focusElementContainer = document.createElement("div");
            focusElementContainer.style.position = "absolute";
            focusElementContainer.style.pointerEvents = "none";
            focusElementContainer.style.top = "0px";
            focusElementContainer.style.left = "0px";
            focusElementContainer.style.overflow = "hidden";
            focusElementContainer.style.width = this.dom.clientWidth + "px";
            focusElementContainer.style.height = this.dom.clientHeight + "px";
            focusElementContainer.setAttribute("role", "application");
            $utils.setInteractive(focusElementContainer, false);
            this._focusElementContainer = focusElementContainer;
            this._inner.appendChild(this._focusElementContainer);
            this._tooltipElementContainer = document.createElement("div");
            this._inner.appendChild(this._tooltipElementContainer);
            // Add keyboard events for accessibility, e.g. simulating drag with arrow
            // keys and click with ENTER
            if ($utils.supports("keyboardevents")) {
                this._disposers.push($utils.addEventListener(focusElementContainer, "keydown", function (ev) {
                    var focusedSprite = _this._focusedSprite;
                    if (focusedSprite) {
                        if (ev.keyCode == 27) {
                            // ESC pressed - lose current focus
                            $utils.blur();
                            _this._focusedSprite = undefined;
                        }
                        var dragOffsetX = 0;
                        var dragOffsetY = 0;
                        // TODO: figure out if using bogus MouseEvent is fine, or it will
                        // fail on some platforms
                        switch (ev.keyCode) {
                            case 13:
                                ev.preventDefault();
                                var downEvent = renderer.getEvent(new MouseEvent("click"));
                                focusedSprite.events.dispatch("click", {
                                    type: "click",
                                    originalEvent: downEvent.event,
                                    point: downEvent.point,
                                    simulated: true,
                                    target: focusedSprite
                                });
                                return;
                            case 37:
                                dragOffsetX = -6;
                                break;
                            case 39:
                                dragOffsetX = 6;
                                break;
                            case 38:
                                dragOffsetY = -6;
                                break;
                            case 40:
                                dragOffsetY = 6;
                                break;
                            default:
                                return;
                        }
                        if (dragOffsetX != 0 || dragOffsetY != 0) {
                            ev.preventDefault();
                            if (!focusedSprite.isDragging()) {
                                // Start dragging
                                _this._keyboardDragPoint = {
                                    x: 0,
                                    y: 0
                                };
                                var downEvent = renderer.getEvent(new MouseEvent("mousedown", {
                                    clientX: 0,
                                    clientY: 0
                                }));
                                if (focusedSprite.events.isEnabled("pointerdown")) {
                                    focusedSprite.events.dispatch("pointerdown", {
                                        type: "pointerdown",
                                        originalEvent: downEvent.event,
                                        point: downEvent.point,
                                        simulated: true,
                                        target: focusedSprite
                                    });
                                }
                            }
                            else {
                                // Move focus marker
                                //this._positionFocusElement(focusedSprite);
                            }
                            // Move incrementally
                            var dragPoint = _this._keyboardDragPoint;
                            dragPoint.x += dragOffsetX;
                            dragPoint.y += dragOffsetY;
                            var moveEvent = renderer.getEvent(new MouseEvent("mousemove", {
                                clientX: dragPoint.x,
                                clientY: dragPoint.y
                            }), false);
                            if (focusedSprite.events.isEnabled("globalpointermove")) {
                                focusedSprite.events.dispatch("globalpointermove", {
                                    type: "globalpointermove",
                                    originalEvent: moveEvent.event,
                                    point: moveEvent.point,
                                    simulated: true,
                                    target: focusedSprite
                                });
                            }
                        }
                    }
                }));
                this._disposers.push($utils.addEventListener(focusElementContainer, "keyup", function (ev) {
                    if (_this._focusedSprite) {
                        var focusedSprite = _this._focusedSprite;
                        var keyCode = ev.keyCode;
                        switch (keyCode) {
                            case 37:
                            case 39:
                            case 38:
                            case 40:
                                if (focusedSprite.isDragging()) {
                                    // Simulate drag stop
                                    var dragPoint = _this._keyboardDragPoint;
                                    var upEvent = renderer.getEvent(new MouseEvent("mouseup", {
                                        clientX: dragPoint.x,
                                        clientY: dragPoint.y
                                    }));
                                    if (focusedSprite.events.isEnabled("globalpointerup")) {
                                        focusedSprite.events.dispatch("globalpointerup", {
                                            type: "globalpointerup",
                                            originalEvent: upEvent.event,
                                            point: upEvent.point,
                                            simulated: true,
                                            target: focusedSprite
                                        });
                                    }
                                    //this._positionFocusElement(focusedSprite);
                                    _this._keyboardDragPoint = undefined;
                                    // @todo dispatch mouseup event instead of calling dragStop?
                                    // this._dispatchEvent("globalpointerup", target, upEvent);
                                    return;
                                }
                                else if (focusedSprite.get("focusableGroup")) {
                                    // Find next item in focusable group
                                    var group_1 = focusedSprite.get("focusableGroup");
                                    var items = _this._tabindexes.filter(function (item) { return item.get("focusableGroup") == group_1; });
                                    var index = items.indexOf(focusedSprite);
                                    var lastIndex = items.length - 1;
                                    index += (keyCode == 39 || keyCode == 40) ? 1 : -1;
                                    if (index < 0) {
                                        index = lastIndex;
                                    }
                                    else if (index > lastIndex) {
                                        index = 0;
                                    }
                                    $utils.focus(items[index].getPrivate("focusElement").dom);
                                }
                                break;
                        }
                    }
                }));
            }
            this._startTicker();
            this.setThemes([]);
            this._addTooltip();
            if (!this._hasLicense()) {
                this._showBranding();
            }
        }
    });
    Object.defineProperty(Root.prototype, "_initResizeSensor", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            if (this._resizeSensorDisposer) {
                this._resizeSensorDisposer.dispose();
            }
            this._resizeSensorDisposer = new ResizeSensor(this.dom, function () {
                if (_this.autoResize) {
                    _this.resize();
                }
            });
            this._disposers.push(this._resizeSensorDisposer);
        }
    });
    /**
     * If automatic resizing of char is disabled (`root.autoResize = false`), it
     * can be resized manually by calling this method.
     */
    Object.defineProperty(Root.prototype, "resize", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var dom = this.dom;
            var w = dom.clientWidth;
            var h = dom.clientHeight;
            if (w > 0 && h > 0) {
                var focusElementContainer = this._focusElementContainer;
                focusElementContainer.style.width = w + "px";
                focusElementContainer.style.height = h + "px";
                this._renderer.resize(w, h);
                var rootContainer = this._rootContainer;
                rootContainer.setPrivate("width", w);
                rootContainer.setPrivate("height", h);
                this._render();
                this._handleLogo();
            }
        }
    });
    Object.defineProperty(Root.prototype, "_render", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._renderer.render(this._rootContainer._display);
            if (this._focusElementDirty) {
                this._updateCurrentFocus();
                this._focusElementDirty = false;
            }
        }
    });
    Object.defineProperty(Root.prototype, "_runTickers", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (currentTime) {
            $array.each(this._tickers, function (f) {
                f(currentTime);
            });
        }
    });
    Object.defineProperty(Root.prototype, "_runAnimations", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (currentTime) {
            $array.keepIf(this._animations, function (animation) {
                return !animation._runAnimation(currentTime);
            });
        }
    });
    Object.defineProperty(Root.prototype, "_runDirties", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            //console.log("tick **************************************************************");
            var allParents = {};
            while (this._isDirtyParents) {
                // This must be before calling _prepareChildren
                this._isDirtyParents = false;
                $object.keys(this._dirtyParents).forEach(function (key) {
                    var parent = _this._dirtyParents[key];
                    delete _this._dirtyParents[key];
                    if (!parent.isDisposed()) {
                        allParents[parent.uid] = parent;
                        parent._prepareChildren();
                    }
                });
            }
            $object.keys(allParents).forEach(function (key) {
                allParents[key]._updateChildren();
            });
            var objects = [];
            //		console.log("_beforeChanged")
            $object.keys(this._dirty).forEach(function (key) {
                var entity = _this._dirty[key];
                if (entity.isDisposed()) {
                    delete _this._dirty[entity.uid];
                }
                else {
                    objects.push(entity);
                    entity._beforeChanged();
                }
            });
            //		console.log("_changed")
            objects.forEach(function (entity) {
                entity._changed();
                delete _this._dirty[entity.uid];
                entity._clearDirty();
            });
            this._isDirty = false;
            var depths = {};
            var bounds = [];
            $object.keys(this._dirtyBounds).forEach(function (key) {
                var entity = _this._dirtyBounds[key];
                delete _this._dirtyBounds[key];
                if (!entity.isDisposed()) {
                    depths[entity.uid] = entity.depth();
                    bounds.push(entity);
                }
            });
            // High depth -> low depth
            bounds.sort(function (x, y) {
                return $order.compare(depths[y.uid], depths[x.uid]);
            });
            //		console.log("_updateBounds")
            bounds.forEach(function (entity) {
                entity._updateBounds();
            });
            //		console.log("_updatePosition")
            var dirtyPositions = this._dirtyPositions;
            $object.keys(dirtyPositions).forEach(function (key) {
                var sprite = dirtyPositions[key];
                delete dirtyPositions[key];
                if (!sprite.isDisposed()) {
                    sprite._updatePosition();
                }
            });
            //		console.log("_afterChanged")
            objects.forEach(function (entity) {
                entity._afterChanged();
            });
        }
    });
    Object.defineProperty(Root.prototype, "_runTicker", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (currentTime) {
            if (!this.isDisposed()) {
                this.animationTime = currentTime;
                if (this.events.isEnabled("framestarted")) {
                    this.events.dispatch("framestarted", {
                        type: "framestarted",
                        target: this,
                        timestamp: currentTime,
                    });
                }
                this._runTickers(currentTime);
                this._runAnimations(currentTime);
                this._runDirties();
                this._render();
                if (this.events.isEnabled("frameended")) {
                    this.events.dispatch("frameended", {
                        type: "frameended",
                        target: this,
                        timestamp: currentTime,
                    });
                }
                // No more work to do
                if (this._tickers.length === 0 &&
                    this._animations.length === 0 &&
                    !this._isDirty) {
                    this._ticker = null;
                    this.animationTime = null;
                }
                else {
                    rAF(this.fps, this._ticker);
                }
            }
        }
    });
    Object.defineProperty(Root.prototype, "_startTicker", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            if (this._ticker === null) {
                this.animationTime = null;
                this._ticker = function (currentTime) {
                    _this._runTicker(currentTime);
                };
                rAF(this.fps, this._ticker);
            }
        }
    });
    Object.defineProperty(Root.prototype, "_addDirtyEntity", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (entity) {
            if (this._dirty[entity.uid] === undefined) {
                this._isDirty = true;
                this._dirty[entity.uid] = entity;
                this._startTicker();
            }
        }
    });
    Object.defineProperty(Root.prototype, "_addDirtyParent", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (parent) {
            if (this._dirtyParents[parent.uid] === undefined) {
                this._isDirty = true;
                this._isDirtyParents = true;
                this._dirtyParents[parent.uid] = parent;
                this._startTicker();
            }
        }
    });
    Object.defineProperty(Root.prototype, "_addDirtyBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (entity) {
            if (this._dirtyBounds[entity.uid] === undefined) {
                this._isDirty = true;
                this._dirtyBounds[entity.uid] = entity;
                this._startTicker();
            }
        }
    });
    Object.defineProperty(Root.prototype, "_addDirtyPosition", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (sprite) {
            if (this._dirtyPositions[sprite.uid] === undefined) {
                this._isDirty = true;
                this._dirtyPositions[sprite.uid] = sprite;
                this._startTicker();
            }
        }
    });
    Object.defineProperty(Root.prototype, "_addAnimation", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (animation) {
            // TODO use numeric id instead
            if (this._animations.indexOf(animation) === -1) {
                this._animations.push(animation);
                this._startTicker();
            }
        }
    });
    Object.defineProperty(Root.prototype, "eachFrame", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (f) {
            var _this = this;
            this._tickers.push(f);
            this._startTicker();
            return new Disposer(function () {
                $array.removeFirst(_this._tickers, f);
            });
        }
    });
    /**
     * Returns width of the target container, in pixels.
     *
     * @return Width
     */
    Object.defineProperty(Root.prototype, "width", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this.dom.clientWidth;
        }
    });
    /**
     * Returns height of the target container, in pixels.
     *
     * @return Height
     */
    Object.defineProperty(Root.prototype, "height", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this.dom.clientHeight;
        }
    });
    /**
     * Disposes root and all the content in it.
     */
    Object.defineProperty(Root.prototype, "dispose", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (!this._isDisposed) {
                this._isDisposed = true;
                this._rootContainer.dispose();
                this._renderer.dispose();
                this.horizontalLayout.dispose();
                this.verticalLayout.dispose();
                this.interfaceColors.dispose();
                $array.each(this._disposers, function (x) {
                    x.dispose();
                });
                if (this._inner) {
                    $utils.removeElement(this._inner);
                }
                $array.remove(registry.rootElements, this);
            }
        }
    });
    /**
     * Returns `true` if root element is disposed.
     *
     * @return Disposed?
     */
    Object.defineProperty(Root.prototype, "isDisposed", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this._isDisposed;
        }
    });
    /**
     * Triggers screen reader read out a message.
     *
     * @see {@link https://www.amcharts.com/docs/v5/concepts/accessibility/} for more info
     * @param  text  Alert text
     */
    Object.defineProperty(Root.prototype, "readerAlert", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (text) {
            this._readerAlertElement.innerHTML = text;
        }
    });
    /**
     * Sets themes to be used for the chart.
     *
     * @see {@link https://www.amcharts.com/docs/v5/concepts/themes/} for more info
     * @param  themes  A list of themes
     */
    Object.defineProperty(Root.prototype, "setThemes", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (themes) {
            this._rootContainer.set("themes", themes);
            // otherwise new themes are not applied
            var tooltipContainer = this.tooltipContainer;
            if (tooltipContainer) {
                tooltipContainer._applyThemes();
            }
            // @todo review this
            var interfaceColors = this.interfaceColors;
            if (interfaceColors) {
                interfaceColors._applyThemes();
            }
        }
    });
    Object.defineProperty(Root.prototype, "_addTooltip", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (!this.tooltipContainer) {
                var tooltipContainer = this._rootContainer.children.push(Container.new(this, { position: "absolute", isMeasured: false, width: p100, height: p100, layer: 30 }));
                this.tooltipContainer = tooltipContainer;
                var tooltip = Tooltip.new(this, {});
                this.container.set("tooltip", tooltip);
                tooltip.hide(0);
                this._tooltip = tooltip;
            }
        }
    });
    /**
     * Accesibility
     */
    Object.defineProperty(Root.prototype, "_registerTabindexOrder", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (target) {
            if (target.get("focusable")) {
                $array.pushOne(this._tabindexes, target);
            }
            else {
                $array.remove(this._tabindexes, target);
            }
            this._invalidateTabindexes();
        }
    });
    Object.defineProperty(Root.prototype, "_unregisterTabindexOrder", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (target) {
            $array.remove(this._tabindexes, target);
            this._invalidateTabindexes();
        }
    });
    Object.defineProperty(Root.prototype, "_invalidateTabindexes", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            this._tabindexes.sort(function (a, b) {
                var aindex = a.get("tabindexOrder", 0);
                var bindex = b.get("tabindexOrder", 0);
                if (aindex == bindex) {
                    return 0;
                }
                else if (aindex > bindex) {
                    return 1;
                }
                else {
                    return -1;
                }
            });
            var groups = [];
            $array.each(this._tabindexes, function (item, index) {
                if (!item.getPrivate("focusElement")) {
                    _this._makeFocusElement(index, item);
                }
                else {
                    _this._moveFocusElement(index, item);
                }
                var group = item.get("focusableGroup");
                if (group) {
                    if (groups.indexOf(group) !== -1) {
                        // Non-first element in the group, make it not directly focusable
                        item.getPrivate("focusElement").dom.setAttribute("tabindex", "-1");
                    }
                    else {
                        groups.push(group);
                    }
                }
            });
        }
    });
    Object.defineProperty(Root.prototype, "_updateCurrentFocus", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (this._focusedSprite) {
                this._decorateFocusElement(this._focusedSprite);
                this._positionFocusElement(this._focusedSprite);
            }
        }
    });
    Object.defineProperty(Root.prototype, "_decorateFocusElement", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (target, focusElement) {
            // Decorate with proper accessibility attributes
            if (!focusElement) {
                focusElement = target.getPrivate("focusElement").dom;
            }
            if (!focusElement) {
                return;
            }
            if (target.get("visible") && target.get("role") != "tooltip" && !target.isHidden()) {
                if (focusElement.getAttribute("tabindex") != "-1") {
                    focusElement.setAttribute("tabindex", "" + this.tabindex);
                }
            }
            else {
                focusElement.removeAttribute("tabindex");
            }
            var role = target.get("role");
            if (role) {
                focusElement.setAttribute("role", role);
            }
            else {
                focusElement.removeAttribute("role");
            }
            var ariaLabel = target.get("ariaLabel");
            if (ariaLabel) {
                var label = populateString(target, ariaLabel);
                focusElement.setAttribute("aria-label", label);
            }
            else {
                focusElement.removeAttribute("aria-label");
            }
            var ariaLive = target.get("ariaLive");
            if (ariaLive) {
                focusElement.setAttribute("aria-live", ariaLive);
            }
            else {
                focusElement.removeAttribute("aria-live");
            }
            var ariaChecked = target.get("ariaChecked");
            if (ariaChecked != null) {
                focusElement.setAttribute("aria-checked", ariaChecked ? "true" : "false");
            }
            else {
                focusElement.removeAttribute("aria-checked");
            }
            if (target.get("ariaHidden")) {
                focusElement.setAttribute("aria-hidden", "hidden");
            }
            else {
                focusElement.removeAttribute("aria-hidden");
            }
            var ariaOrientation = target.get("ariaOrientation");
            if (ariaOrientation) {
                focusElement.setAttribute("aria-orientation", ariaOrientation);
            }
            else {
                focusElement.removeAttribute("aria-orientation");
            }
            var ariaValueNow = target.get("ariaValueNow");
            if (ariaValueNow) {
                focusElement.setAttribute("aria-valuenow", ariaValueNow);
            }
            else {
                focusElement.removeAttribute("aria-valuenow");
            }
            var ariaValueMin = target.get("ariaValueMin");
            if (ariaValueMin) {
                focusElement.setAttribute("aria-valuemin", ariaValueMin);
            }
            else {
                focusElement.removeAttribute("aria-valuemin");
            }
            var ariaValueMax = target.get("ariaValueMax");
            if (ariaValueMax) {
                focusElement.setAttribute("aria-valuemax", ariaValueMax);
            }
            else {
                focusElement.removeAttribute("aria-valuemax");
            }
            var ariaValueText = target.get("ariaValueText");
            if (ariaValueText) {
                focusElement.setAttribute("aria-valuetext", ariaValueText);
            }
            else {
                focusElement.removeAttribute("aria-valuetext");
            }
            var ariaControls = target.get("ariaControls");
            if (ariaControls) {
                focusElement.setAttribute("aria-controls", ariaControls);
            }
            else {
                focusElement.removeAttribute("aria-controls");
            }
        }
    });
    Object.defineProperty(Root.prototype, "_makeFocusElement", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (index, target) {
            var _this = this;
            if (target.getPrivate("focusElement")) {
                return;
            }
            // Init
            var focusElement = document.createElement("div");
            if (target.get("role") != "tooltip") {
                focusElement.tabIndex = this.tabindex;
            }
            focusElement.style.position = "absolute";
            $utils.setInteractive(focusElement, false);
            var disposers = [];
            target.setPrivate("focusElement", {
                dom: focusElement,
                disposers: disposers,
            });
            this._decorateFocusElement(target);
            disposers.push($utils.addEventListener(focusElement, "focus", function (ev) {
                _this._handleFocus(ev, index);
            }));
            disposers.push($utils.addEventListener(focusElement, "blur", function (ev) {
                _this._handleBlur(ev, index);
            }));
            this._moveFocusElement(index, target);
        }
    });
    Object.defineProperty(Root.prototype, "_removeFocusElement", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (target) {
            // Init
            var container = this._focusElementContainer;
            var focusElement = target.getPrivate("focusElement");
            container.removeChild(focusElement.dom);
            $array.each(focusElement.disposers, function (x) {
                x.dispose();
            });
        }
    });
    Object.defineProperty(Root.prototype, "_moveFocusElement", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (index, target) {
            // Get container
            var container = this._focusElementContainer;
            var focusElement = target.getPrivate("focusElement").dom;
            if (focusElement === this._focusElementContainer.children[index]) {
                // Nothing to do
                return;
            }
            var next = this._focusElementContainer.children[index + 1];
            if (next) {
                container.insertBefore(focusElement, next);
            }
            else {
                container.append(focusElement);
            }
        }
    });
    Object.defineProperty(Root.prototype, "_positionFocusElement", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (target) {
            var bounds = target.globalBounds();
            var width = bounds.right == bounds.left ? target.width() : bounds.right - bounds.left;
            var height = bounds.top == bounds.bottom ? target.height() : bounds.bottom - bounds.top;
            var focusElement = target.getPrivate("focusElement").dom;
            focusElement.style.top = (bounds.top - 2) + "px";
            focusElement.style.left = (bounds.left - 2) + "px";
            focusElement.style.width = (width + 4) + "px";
            focusElement.style.height = (height + 4) + "px";
        }
    });
    Object.defineProperty(Root.prototype, "_handleFocus", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (ev, index) {
            // Get element
            var focused = this._tabindexes[index];
            // Size and position
            this._positionFocusElement(focused);
            //this._decorateFocusElement(focused);
            this._focusedSprite = focused;
            if (focused.events.isEnabled("focus")) {
                focused.events.dispatch("focus", {
                    type: "focus",
                    originalEvent: ev,
                    target: focused
                });
            }
        }
    });
    Object.defineProperty(Root.prototype, "_handleBlur", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (ev, _index) {
            var focused = this._focusedSprite;
            if (focused && focused.events.isEnabled("blur")) {
                focused.events.dispatch("blur", {
                    type: "blur",
                    originalEvent: ev,
                    target: focused
                });
            }
            this._focusedSprite = undefined;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Root.prototype, "updateTooltip", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (target) {
            var text = target._getText();
            var tooltipElement = target.getPrivate("tooltipElement");
            if (target.get("role") == "tooltip" && text != "") {
                if (!tooltipElement) {
                    tooltipElement = this._makeTooltipElement(target);
                }
                if (tooltipElement.innerHTML != text) {
                    tooltipElement.innerHTML = text;
                }
            }
            else if (tooltipElement) {
                tooltipElement.remove();
                target.removePrivate("tooltipElement");
            }
        }
    });
    Object.defineProperty(Root.prototype, "_makeTooltipElement", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (target) {
            var container = this._tooltipElementContainer;
            var tooltipElement = document.createElement("div");
            tooltipElement.style.position = "absolute";
            tooltipElement.style.opacity = "0.0000001";
            $utils.setInteractive(tooltipElement, false);
            this._decorateFocusElement(target, tooltipElement);
            container.append(tooltipElement);
            target.setPrivate("tooltipElement", tooltipElement);
            return tooltipElement;
        }
    });
    Object.defineProperty(Root.prototype, "_invalidateAccessibility", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (target) {
            this._focusElementDirty = true;
            var focusElement = target.getPrivate("focusElement");
            if (target.get("focusable")) {
                if (focusElement) {
                    this._decorateFocusElement(target);
                    this._positionFocusElement(target);
                }
                // else {
                // 	this._renderer._makeFocusElement(0, this);
                // }
            }
            else if (focusElement) {
                this._removeFocusElement(target);
            }
            //this.updateCurrentFocus();
        }
    });
    /**
     * Returns `true` if `target` is currently focused.
     *
     * @param   target  Target
     * @return          Focused?
     */
    Object.defineProperty(Root.prototype, "focused", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (target) {
            return this._focusedSprite === target;
        }
    });
    /**
     * Converts document coordinates to coordinates withing root element.
     *
     * @param   point  Document point
     * @return         Root point
     */
    Object.defineProperty(Root.prototype, "documentPointToRoot", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (point) {
            var bbox = this.dom.getBoundingClientRect();
            return {
                x: point.x - bbox.left,
                y: point.y - bbox.top
            };
        }
    });
    /**
     * Converts root coordinates to document
     *
     * @param   point  Document point
     * @return         Root point
     */
    Object.defineProperty(Root.prototype, "rootPointToDocument", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (point) {
            var bbox = this.dom.getBoundingClientRect();
            return {
                x: point.x + bbox.left,
                y: point.y + bbox.top
            };
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Root.prototype, "addDisposer", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (disposer) {
            this._disposers.push(disposer);
            return disposer;
        }
    });
    /**
 * To all the clever heads out there. Yes, we did not make any attempts to
 * scramble this.
 *
 * This is a part of a tool meant for our users to manage their commercial
 * licenses for removal of amCharts branding from charts.
 *
 * The only legit way to do so is to purchase a commercial license for amCharts:
 * https://www.amcharts.com/online-store/
 *
 * Removing or altering this code, or disabling amCharts branding in any other
 * way is against the license and thus illegal.
 */
    Object.defineProperty(Root.prototype, "_hasLicense", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            for (var i = 0; i < registry.licenses.length; i++) {
                if (registry.licenses[i].match(/^AM5C.{5,}/i)) {
                    return true;
                }
            }
            return false;
        }
    });
    return Root;
}());
export { Root };
//# sourceMappingURL=Root.js.map