import { __extends } from "tslib";
import { Layout, eachChildren } from "./Layout";
import * as $array from "../util/Array";
import * as $math from "../util/Math";
/**
 * A grid children layout for [[Container]].
 *
 * @see {@link https://www.amcharts.com/docs/v5/concepts/common-elements/containers/#Layout} for more info
 */
var GridLayout = /** @class */ (function (_super) {
    __extends(GridLayout, _super);
    function GridLayout() {
        return _super !== null && _super.apply(this, arguments) || this;
    }
    Object.defineProperty(GridLayout.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._setRawDefault("maxColumns", Number.MAX_VALUE);
            _super.prototype._afterNew.call(this);
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(GridLayout.prototype, "updateContainer", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (container) {
            var paddingLeft = container.get("paddingLeft", 0);
            var paddingRight = container.get("paddingRight", 0);
            var paddingTop = container.get("paddingTop", 0);
            var availableWidth = container.maxWidth() - paddingLeft - paddingRight;
            var minCellWidth = availableWidth;
            var maxCellWidth = 1;
            eachChildren(container, function (child) {
                if (child.get("position") != "absolute") {
                    var childWidth = child.width();
                    if (childWidth < minCellWidth) {
                        minCellWidth = childWidth;
                    }
                    if (childWidth > maxCellWidth) {
                        maxCellWidth = childWidth;
                    }
                }
            });
            minCellWidth = $math.fitToRange(minCellWidth, 1, availableWidth);
            maxCellWidth = $math.fitToRange(maxCellWidth, 1, availableWidth);
            var columnCount = 1;
            if (this.get("fixedWidthGrid")) {
                columnCount = availableWidth / maxCellWidth;
            }
            else {
                columnCount = availableWidth / minCellWidth;
            }
            columnCount = Math.max(1, Math.floor(columnCount));
            columnCount = Math.min(this.get("maxColumns", Number.MAX_VALUE), columnCount);
            var columnWidths = this.getColumnWidths(container, columnCount, maxCellWidth, availableWidth);
            var prevY = paddingTop;
            var column = 0;
            var maxColumnHeight = 0;
            columnCount = columnWidths.length;
            var prevX = paddingLeft;
            eachChildren(container, function (child) {
                if (child.get("position") == "relative" && !child.isHidden()) {
                    var marginTop = child.get("marginTop", 0);
                    var marginBottom = child.get("marginBottom", 0);
                    var bounds = child.adjustedLocalBounds();
                    var marginLeft = child.get("marginLeft", 0);
                    var marginRight = child.get("marginRight", 0);
                    var x = prevX + marginLeft - bounds.left;
                    var y = prevY + marginTop - bounds.top;
                    child.setPrivate("x", x);
                    child.setPrivate("y", y);
                    prevX += columnWidths[column] + marginRight;
                    maxColumnHeight = Math.max(maxColumnHeight, child.height() + marginTop + marginBottom);
                    column++;
                    if (column >= columnCount) {
                        column = 0;
                        prevX = paddingLeft;
                        prevY += maxColumnHeight;
                    }
                }
            });
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(GridLayout.prototype, "getColumnWidths", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (container, columnCount, maxCellWidth, availableWidth) {
            var _this = this;
            var totalWidth = 0;
            var columnWidths = [];
            var column = 0;
            eachChildren(container, function (child) {
                var bounds = child.adjustedLocalBounds();
                if (child.get("position") != "absolute" && !child.isHidden()) {
                    if (_this.get("fixedWidthGrid")) {
                        columnWidths[column] = maxCellWidth;
                    }
                    else {
                        columnWidths[column] = Math.max(columnWidths[column] | 0, bounds.right - bounds.left + child.get("marginLeft", 0) + child.get("marginRight", 0));
                    }
                    if (column < container.children.length - 1) {
                        column++;
                        if (column == columnCount) {
                            column = 0;
                        }
                    }
                }
            });
            $array.each(columnWidths, function (w) {
                totalWidth += w;
            });
            if (totalWidth > availableWidth) {
                if (columnCount > 2) {
                    columnCount -= 1;
                    return this.getColumnWidths(container, columnCount, maxCellWidth, availableWidth);
                }
                else {
                    return [availableWidth];
                }
            }
            return columnWidths;
        }
    });
    Object.defineProperty(GridLayout, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "GridLayout"
    });
    Object.defineProperty(GridLayout, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: Layout.classNames.concat([GridLayout.className])
    });
    return GridLayout;
}(Layout));
export { GridLayout };
//# sourceMappingURL=GridLayout.js.map