import { __extends } from "tslib";
import { RoundedRectangle } from "../render/RoundedRectangle";
import { Container } from "./Container";
import * as $type from "../util/Type";
import { Graphics } from "./Graphics";
import { Button } from "./Button";
import * as $utils from "../util/Utils";
/**
 * A control that allows zooming chart's axes, or other uses requiring range
 * selection.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/xy-chart/scrollbars/} for more info
 */
var Scrollbar = /** @class */ (function (_super) {
    __extends(Scrollbar, _super);
    function Scrollbar() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        /**
         * A thumb elment - a draggable square between the grips, used for panning
         * the selection.
         */
        Object.defineProperty(_this, "thumb", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this._makeThumb()
        });
        /**
         * Start grip button.
         */
        Object.defineProperty(_this, "startGrip", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this._makeButton()
        });
        /**
         * End grip button.
         */
        Object.defineProperty(_this, "endGrip", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this._makeButton()
        });
        Object.defineProperty(_this, "_thumbBusy", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(_this, "_startDown", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(_this, "_endDown", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(_this, "_thumbDown", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        return _this;
    }
    Object.defineProperty(Scrollbar.prototype, "_addOrientationClass", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._settings.themeTags = $utils.mergeTags(this._settings.themeTags, ["scrollbar", this._settings.orientation]);
            if (!this._settings.background) {
                this._settings.background = RoundedRectangle.new(this._root, {
                    themeTags: $utils.mergeTags(this._settings.themeTags, ["main", "background"])
                });
            }
        }
    });
    Object.defineProperty(Scrollbar.prototype, "_makeButton", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this.children.push(Button.new(this._root, {
                themeTags: ["resize", "button", this.get("orientation")], icon: Graphics.new(this._root, {
                    themeTags: ["icon"]
                })
            }));
        }
    });
    Object.defineProperty(Scrollbar.prototype, "_makeThumb", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this.children.push(RoundedRectangle.new(this._root, {
                themeTags: ["thumb", this.get("orientation")]
            }));
        }
    });
    Object.defineProperty(Scrollbar.prototype, "_handleAnimation", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (animation) {
            var _this = this;
            if (animation) {
                this._disposers.push(animation.events.on("stopped", function () {
                    _this.setPrivateRaw("isBusy", false);
                    _this._thumbBusy = false;
                }));
            }
        }
    });
    Object.defineProperty(Scrollbar.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            this._addOrientationClass();
            _super.prototype._afterNew.call(this);
            var startGrip = this.startGrip;
            var endGrip = this.endGrip;
            var thumb = this.thumb;
            var background = this.get("background");
            if (background) {
                this._disposers.push(background.events.on("click", function (event) {
                    _this.setPrivateRaw("isBusy", true);
                    var point = _this._display.toLocal(event.point);
                    var w = _this.width();
                    var h = _this.height();
                    var orientation = _this.get("orientation");
                    var newMiddle;
                    if (orientation == "vertical") {
                        newMiddle = (point.y - thumb.height() / 2) / h;
                    }
                    else {
                        newMiddle = (point.x - thumb.width() / 2) / w;
                    }
                    var newCoordinate;
                    var key;
                    if (orientation == "vertical") {
                        newCoordinate = newMiddle * h;
                        key = "y";
                    }
                    else {
                        newCoordinate = newMiddle * w;
                        key = "x";
                    }
                    var duration = _this.get("animationDuration", 0);
                    if (duration > 0) {
                        _this._thumbBusy = true;
                        _this._handleAnimation(_this.thumb.animate({ key: key, to: newCoordinate, duration: duration, easing: _this.get("animationEasing") }));
                    }
                    else {
                        _this.thumb.set(key, newCoordinate);
                    }
                }));
            }
            this._disposers.push(thumb.events.on("dblclick", function () {
                var duration = _this.get("animationDuration", 0);
                var easing = _this.get("animationEasing");
                _this.animate({ key: "start", to: 0, duration: duration, easing: easing });
                _this.animate({ key: "end", to: 1, duration: duration, easing: easing });
            }));
            this._disposers.push(startGrip.events.on("pointerdown", function () {
                _this.setPrivateRaw("isBusy", true);
                _this._startDown = true;
            }));
            this._disposers.push(endGrip.events.on("pointerdown", function () {
                _this.setPrivateRaw("isBusy", true);
                _this._endDown = true;
            }));
            this._disposers.push(thumb.events.on("pointerdown", function () {
                _this.setPrivateRaw("isBusy", true);
                _this._thumbDown = true;
            }));
            this._disposers.push(startGrip.events.on("globalpointerup", function () {
                if (_this._startDown) {
                    _this.setPrivateRaw("isBusy", false);
                }
                _this._startDown = false;
            }));
            this._disposers.push(endGrip.events.on("globalpointerup", function () {
                if (_this._endDown) {
                    _this.setPrivateRaw("isBusy", false);
                }
                _this._endDown = false;
            }));
            this._disposers.push(thumb.events.on("globalpointerup", function () {
                if (_this._thumbDown) {
                    _this.setPrivateRaw("isBusy", false);
                }
                _this._thumbDown = false;
            }));
            this._disposers.push(startGrip.on("x", function () {
                _this._updateThumb();
            }));
            this._disposers.push(endGrip.on("x", function () {
                _this._updateThumb();
            }));
            this._disposers.push(startGrip.on("y", function () {
                _this._updateThumb();
            }));
            this._disposers.push(endGrip.on("y", function () {
                _this._updateThumb();
            }));
            this._disposers.push(thumb.events.on("positionchanged", function () {
                _this._updateGripsByThumb();
            }));
            if (this.get("orientation") == "vertical") {
                startGrip.set("x", 0);
                endGrip.set("x", 0);
                this._disposers.push(thumb.adapters.add("y", function (value) {
                    return Math.max(Math.min(Number(value), _this.height() - thumb.height()), 0);
                }));
                this._disposers.push(thumb.adapters.add("x", function (_value) {
                    return _this.width() / 2;
                }));
                this._disposers.push(startGrip.adapters.add("x", function (_value) {
                    return _this.width() / 2;
                }));
                this._disposers.push(endGrip.adapters.add("x", function (_value) {
                    return _this.width() / 2;
                }));
                this._disposers.push(startGrip.adapters.add("y", function (value) {
                    return Math.max(Math.min(Number(value), _this.height()), 0);
                }));
                this._disposers.push(endGrip.adapters.add("y", function (value) {
                    return Math.max(Math.min(Number(value), _this.height()), 0);
                }));
            }
            else {
                startGrip.set("y", 0);
                endGrip.set("y", 0);
                this._disposers.push(thumb.adapters.add("x", function (value) {
                    return Math.max(Math.min(Number(value), _this.width() - thumb.width()), 0);
                }));
                this._disposers.push(thumb.adapters.add("y", function (_value) {
                    return _this.height() / 2;
                }));
                this._disposers.push(startGrip.adapters.add("y", function (_value) {
                    return _this.height() / 2;
                }));
                this._disposers.push(endGrip.adapters.add("y", function (_value) {
                    return _this.height() / 2;
                }));
                this._disposers.push(startGrip.adapters.add("x", function (value) {
                    return Math.max(Math.min(Number(value), _this.width()), 0);
                }));
                this._disposers.push(endGrip.adapters.add("x", function (value) {
                    return Math.max(Math.min(Number(value), _this.width()), 0);
                }));
            }
        }
    });
    Object.defineProperty(Scrollbar.prototype, "_updateChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._updateChildren.call(this);
            if (this.isDirty("end") || this.isDirty("start") || this._sizeDirty) {
                this.updateGrips();
            }
        }
    });
    Object.defineProperty(Scrollbar.prototype, "_changed", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._changed.call(this);
            if (this.isDirty("start") || this.isDirty("end")) {
                var eventType = "rangechanged";
                if (this.events.isEnabled(eventType)) {
                    this.events.dispatch(eventType, { type: eventType, target: this, start: this.get("start", 0), end: this.get("end", 1) });
                }
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Scrollbar.prototype, "updateGrips", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var startGrip = this.startGrip;
            var endGrip = this.endGrip;
            var orientation = this.get("orientation");
            var height = this.height();
            var width = this.width();
            if (orientation == "vertical") {
                startGrip.set("y", height * this.get("start", 0));
                endGrip.set("y", height * this.get("end", 1));
            }
            else {
                startGrip.set("x", width * this.get("start", 0));
                endGrip.set("x", width * this.get("end", 1));
            }
            var valueFunction = this.getPrivate("positionTextFunction");
            var from = Math.round(this.get("start", 0) * 100);
            var to = Math.round(this.get("end", 0) * 100);
            var fromValue;
            var toValue;
            if (valueFunction) {
                fromValue = valueFunction.call(this, this.get("start", 0));
                toValue = valueFunction.call(this, this.get("end", 0));
            }
            else {
                fromValue = from + "%";
                toValue = to + "%";
            }
            startGrip.set("ariaLabel", this._t("From %1", undefined, fromValue));
            startGrip.set("ariaValueNow", "" + from);
            startGrip.set("ariaValueText", from + "%");
            startGrip.set("ariaValueMin", "0");
            startGrip.set("ariaValueMax", "100");
            endGrip.set("ariaLabel", this._t("To %1", undefined, toValue));
            endGrip.set("ariaValueNow", "" + to);
            endGrip.set("ariaValueText", to + "%");
            endGrip.set("ariaValueMin", "0");
            endGrip.set("ariaValueMax", "100");
        }
    });
    Object.defineProperty(Scrollbar.prototype, "_updateThumb", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var thumb = this.thumb;
            var startGrip = this.startGrip;
            var endGrip = this.endGrip;
            var height = this.height();
            var width = this.width();
            var x0 = startGrip.x();
            var x1 = endGrip.x();
            var y0 = startGrip.y();
            var y1 = endGrip.y();
            var start = 0;
            var end = 1;
            if (this.get("orientation") == "vertical") {
                if ($type.isNumber(y0) && $type.isNumber(y1)) {
                    if (!this._thumbBusy && !thumb.isDragging()) {
                        thumb.set("height", y1 - y0);
                        thumb.set("y", y0);
                    }
                    start = y0 / height;
                    end = y1 / height;
                }
            }
            else {
                if ($type.isNumber(x0) && $type.isNumber(x1)) {
                    if (!this._thumbBusy && !thumb.isDragging()) {
                        thumb.set("width", x1 - x0);
                        thumb.set("x", x0);
                    }
                    start = x0 / width;
                    end = x1 / width;
                }
            }
            if (this.getPrivate("isBusy") && (this.get("start") != start || this.get("end") != end)) {
                this.set("start", start);
                this.set("end", end);
            }
            var valueFunction = this.getPrivate("positionTextFunction");
            var from = Math.round(this.get("start", 0) * 100);
            var to = Math.round(this.get("end", 0) * 100);
            var fromValue;
            var toValue;
            if (valueFunction) {
                fromValue = valueFunction.call(this, this.get("start", 0));
                toValue = valueFunction.call(this, this.get("end", 0));
            }
            else {
                fromValue = from + "%";
                toValue = to + "%";
            }
            thumb.set("ariaLabel", this._t("From %1 to %2", undefined, fromValue, toValue));
            thumb.set("ariaValueNow", "" + from);
            thumb.set("ariaValueText", from + "%");
        }
    });
    Object.defineProperty(Scrollbar.prototype, "_updateGripsByThumb", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var thumb = this.thumb;
            var startGrip = this.startGrip;
            var endGrip = this.endGrip;
            if (this.get("orientation") == "vertical") {
                var thumbSize = thumb.height();
                startGrip.set("y", thumb.y());
                endGrip.set("y", thumb.y() + thumbSize);
            }
            else {
                var thumbSize = thumb.width();
                startGrip.set("x", thumb.x());
                endGrip.set("x", thumb.x() + thumbSize);
            }
        }
    });
    Object.defineProperty(Scrollbar, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "Scrollbar"
    });
    Object.defineProperty(Scrollbar, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: Container.classNames.concat([Scrollbar.className])
    });
    return Scrollbar;
}(Container));
export { Scrollbar };
//# sourceMappingURL=Scrollbar.js.map