import { __awaiter, __extends, __generator } from "tslib";
import { Entity } from "../util/Entity";
import { Template } from "../util/Template";
import { Percent } from "../util/Percent";
import { EventDispatcher } from "../util/EventDispatcher";
import { MultiDisposer, CounterDisposer } from "../util/Disposer";
import { waitForAnimations } from "../util/Animation";
import * as $utils from "../util/Utils";
import * as $array from "../util/Array";
import * as $type from "../util/Type";
import * as $object from "../util/Object";
import * as $math from "../util/Math";
//import { populateString } from "../util/PopulateString";
/**
 * An [[EventDispatcher]] for [[Sprite]].
 *
 * @see {@link https://www.amcharts.com/docs/v5/concepts/events/} for more info
 */
var SpriteEventDispatcher = /** @class */ (function (_super) {
    __extends(SpriteEventDispatcher, _super);
    function SpriteEventDispatcher(sprite) {
        var _this = _super.call(this) || this;
        Object.defineProperty(_this, "_sprite", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_rendererDisposers", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        Object.defineProperty(_this, "_dispatchParents", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: true
        });
        _this._sprite = sprite;
        return _this;
    }
    Object.defineProperty(SpriteEventDispatcher.prototype, "_makePointerEvent", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, event) {
            return {
                type: key,
                originalEvent: event.event,
                point: event.point,
                simulated: event.simulated,
                native: event.native,
                target: this._sprite
            };
        }
    });
    Object.defineProperty(SpriteEventDispatcher.prototype, "_onRenderer", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, dispatch) {
            var _this = this;
            // TODO: is this OK? it'd be good not to require to set this on each individual element
            this._sprite.set("interactive", true);
            this._sprite._display.interactive = true;
            var events = this._rendererDisposers[key];
            if (events === undefined) {
                var disposer_1 = this._sprite._display.on(key, function (e) {
                    dispatch.call(_this, e);
                });
                events = this._rendererDisposers[key] = new CounterDisposer(function () {
                    delete _this._rendererDisposers[key];
                    disposer_1.dispose();
                });
            }
            return events.increment();
        }
    });
    Object.defineProperty(SpriteEventDispatcher.prototype, "_on", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (once, type, callback, context, shouldClone, dispatch) {
            var info = _super.prototype._on.call(this, once, type, callback, context, shouldClone, dispatch);
            var rendererEvent = SpriteEventDispatcher.RENDERER_EVENTS[type];
            if (rendererEvent !== undefined) {
                info.disposer = new MultiDisposer([
                    info.disposer,
                    this._onRenderer(type, rendererEvent),
                ]);
            }
            return info;
        }
    });
    /**
     * Will stop any bubbling up of the event to element's parents.
     *
     * Should be called in an event handler, e.g.:
     *
     * ```TypeScript
     * element.events.on("pointerdown", function(ev) {
     *   // Do something here and prevent from "pointerdown" bubbling up
     *   // ...
     *   ev.target.events.stopParentDispatch();
     * });
     * ```
     * ```JavaScript
     * element.events.on("pointerdown", function(ev) {
     *   // Do something here and prevent from "pointerdown" bubbling up
     *   // ...
     *   ev.target.events.stopParentDispatch();
     * });
     * ```
     */
    Object.defineProperty(SpriteEventDispatcher.prototype, "stopParentDispatch", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._dispatchParents = false;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(SpriteEventDispatcher.prototype, "dispatchParents", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (type, event) {
            var old = this._dispatchParents;
            this._dispatchParents = true;
            try {
                this.dispatch(type, event);
                if (this._dispatchParents && this._sprite.parent) {
                    this._sprite.parent.events.dispatchParents(type, event);
                }
            }
            finally {
                this._dispatchParents = old;
            }
        }
    });
    Object.defineProperty(SpriteEventDispatcher, "RENDERER_EVENTS", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: {
            "click": function (event) {
                if (this.isEnabled("click") && !this._sprite.isDragging() && this._sprite._hasDown() && !this._sprite._hasMoved(this._makePointerEvent("click", event))) {
                    this.dispatch("click", this._makePointerEvent("click", event));
                }
            },
            "rightclick": function (event) {
                if (this.isEnabled("rightclick")) {
                    this.dispatch("rightclick", this._makePointerEvent("rightclick", event));
                }
            },
            "middleclick": function (event) {
                if (this.isEnabled("middleclick")) {
                    this.dispatch("middleclick", this._makePointerEvent("middleclick", event));
                }
            },
            "dblclick": function (event) {
                this.dispatchParents("dblclick", this._makePointerEvent("dblclick", event));
            },
            "pointerover": function (event) {
                if (this.isEnabled("pointerover")) {
                    this.dispatch("pointerover", this._makePointerEvent("pointerover", event));
                }
            },
            "pointerout": function (event) {
                if (this.isEnabled("pointerout")) {
                    this.dispatch("pointerout", this._makePointerEvent("pointerout", event));
                }
            },
            "pointerdown": function (event) {
                this.dispatchParents("pointerdown", this._makePointerEvent("pointerdown", event));
            },
            "pointerup": function (event) {
                if (this.isEnabled("pointerup")) {
                    this.dispatch("pointerup", this._makePointerEvent("pointerup", event));
                }
            },
            "globalpointerup": function (event) {
                if (this.isEnabled("globalpointerup")) {
                    this.dispatch("globalpointerup", this._makePointerEvent("globalpointerup", event));
                }
            },
            "globalpointermove": function (event) {
                if (this.isEnabled("globalpointermove")) {
                    this.dispatch("globalpointermove", this._makePointerEvent("globalpointermove", event));
                }
            },
            "wheel": function (event) {
                if (this.isEnabled("wheel")) {
                    this.dispatch("wheel", {
                        type: "wheel",
                        target: this._sprite,
                        originalEvent: event.event,
                        point: event.point,
                    });
                }
            },
        }
    });
    return SpriteEventDispatcher;
}(EventDispatcher));
/**
 * A base class for all visual elements.
 *
 * @important
 */
var Sprite = /** @class */ (function (_super) {
    __extends(Sprite, _super);
    function Sprite() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "_adjustedLocalBounds", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: { left: 0, right: 0, top: 0, bottom: 0 }
        });
        Object.defineProperty(_this, "_localBounds", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: { left: 0, right: 0, top: 0, bottom: 0 }
        });
        Object.defineProperty(_this, "_parent", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_dataItem", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_templateField", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_sizeDirty", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        // Will be true only when dragging
        Object.defineProperty(_this, "_isDragging", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        // The event when the dragging starts
        Object.defineProperty(_this, "_dragEvent", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        // The position when dragging starts
        Object.defineProperty(_this, "_dragPoint", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_isHidden", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(_this, "_isShowing", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(_this, "_isHiding", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(_this, "_isDown", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(_this, "_downPoint", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_downPoints", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        Object.defineProperty(_this, "_toggleDp", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_dragDp", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_tooltipDp", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_hoverDp", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_focusDp", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_tooltipMoveDp", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_tooltipPointerDp", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_statesHandled", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        return _this;
    }
    Object.defineProperty(Sprite.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this.setPrivateRaw("visible", true);
            _super.prototype._afterNew.call(this);
        }
    });
    Object.defineProperty(Sprite.prototype, "_markDirtyKey", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key) {
            _super.prototype._markDirtyKey.call(this, key);
            if (key == "x" || key == "y" || key == "dx" || key == "dy") {
                this.markDirtyBounds();
                this._addPercentagePositionChildren();
                this.markDirtyPosition();
            }
        }
    });
    Object.defineProperty(Sprite.prototype, "_markDirtyPrivateKey", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key) {
            _super.prototype._markDirtyPrivateKey.call(this, key);
            if (key == "x" || key == "y") {
                this.markDirtyPosition();
            }
        }
    });
    Object.defineProperty(Sprite.prototype, "_removeTemplateField", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (this._templateField) {
                this._templateField._removeObjectTemplate(this);
            }
        }
    });
    Object.defineProperty(Sprite.prototype, "_createEvents", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return new SpriteEventDispatcher(this);
        }
    });
    Object.defineProperty(Sprite.prototype, "_processTemplateField", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var template;
            var field = this.get("templateField");
            if (field) {
                var dataItem = this.dataItem;
                if (dataItem) {
                    var context = dataItem.dataContext;
                    if (context) {
                        template = context[field];
                        if (!(template instanceof Template) && template) {
                            template = Template.new(template);
                        }
                    }
                }
            }
            if (this._templateField !== template) {
                this._removeTemplateField();
                this._templateField = template;
                if (template) {
                    template._setObjectTemplate(this);
                }
                this._applyTemplates();
            }
        }
    });
    // TODO change this to run before the element is added to the parent, so that way
    //      it doesn't need to apply the themes twice
    Object.defineProperty(Sprite.prototype, "_setDataItem", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dataItem) {
            var oldDataItem = this._dataItem;
            this._dataItem = dataItem;
            this._processTemplateField();
            var eventType = "dataitemchanged";
            if (this.events.isEnabled(eventType)) {
                this.events.dispatch(eventType, {
                    type: eventType,
                    target: this,
                    oldDataItem: oldDataItem,
                    newDataItem: dataItem
                });
            }
        }
    });
    Object.defineProperty(Sprite.prototype, "dataItem", {
        /**
         * @return DataItem
         */
        get: function () {
            if (this._dataItem) {
                return this._dataItem;
            }
            else {
                var parent_1 = this._parent;
                while (parent_1) {
                    if (parent_1._dataItem) {
                        return parent_1._dataItem;
                    }
                    else {
                        parent_1 = parent_1._parent;
                    }
                }
            }
        },
        /**
         * A [[DataItem]] used for this element.
         *
         * NOTE: data item is being assigned automatically in most cases where it
         * matters. Use this accessor to set data item only if you know what you're
         * doing.
         *
         * @param  value  Data item
         */
        set: function (value) {
            this._setDataItem(value);
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(Sprite.prototype, "_addPercentageSizeChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var parent = this.parent;
            if (parent) {
                if (this.get("width") instanceof Percent || this.get("height") instanceof Percent) {
                    $array.pushOne(parent._percentageSizeChildren, this);
                }
                else {
                    $array.removeFirst(parent._percentageSizeChildren, this);
                }
            }
        }
    });
    Object.defineProperty(Sprite.prototype, "_addPercentagePositionChildren", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var parent = this.parent;
            if (parent) {
                if (this.get("x") instanceof Percent || this.get("y") instanceof Percent) {
                    $array.pushOne(parent._percentagePositionChildren, this);
                }
                else {
                    $array.removeFirst(parent._percentagePositionChildren, this);
                }
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Sprite.prototype, "markDirtyPosition", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._root._addDirtyPosition(this);
        }
    });
    Object.defineProperty(Sprite.prototype, "updatePivotPoint", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var bounds = this._localBounds;
            if (bounds) {
                var centerX = this.get("centerX");
                if (centerX != null) {
                    this._display.pivot.x = bounds.left + $utils.relativeToValue(centerX, bounds.right - bounds.left);
                }
                var centerY = this.get("centerY");
                if (centerY != null) {
                    this._display.pivot.y = bounds.top + $utils.relativeToValue(centerY, bounds.bottom - bounds.top);
                }
            }
        }
    });
    Object.defineProperty(Sprite.prototype, "_beforeChanged", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            _super.prototype._beforeChanged.call(this);
            if (this.isDirty("tooltip")) {
                var previous = this._prevSettings.tooltip;
                if (previous) {
                    previous.dispose();
                }
            }
            if (this.isDirty("layer")) {
                this._display.setLayer(this.get("layer"));
                this.markDirtyLayer();
            }
            if (this.isDirty("tooltipPosition")) {
                var tooltipMoveDp = this._tooltipMoveDp;
                if (tooltipMoveDp) {
                    tooltipMoveDp.dispose();
                    this._tooltipMoveDp = undefined;
                }
                var tooltipPointerDp = this._tooltipPointerDp;
                if (tooltipPointerDp) {
                    tooltipPointerDp.dispose();
                    this._tooltipPointerDp = undefined;
                }
                if (this.get("tooltipPosition") == "pointer") {
                    this._tooltipPointerDp = new MultiDisposer([
                        this.events.on("pointerover", function () {
                            _this._tooltipMoveDp = _this.events.on("globalpointermove", function (e) {
                                _this.showTooltip(e.point);
                            });
                        }),
                        this.events.on("pointerout", function () {
                            var tooltipMoveDp = _this._tooltipMoveDp;
                            if (tooltipMoveDp) {
                                tooltipMoveDp.dispose();
                                _this._tooltipMoveDp = undefined;
                            }
                        })
                    ]);
                }
            }
        }
    });
    Object.defineProperty(Sprite.prototype, "_handleStates", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (!this._statesHandled) {
                if (this.isDirty("active")) {
                    if (this.get("active")) {
                        this.states.applyAnimate("active");
                        this.set("ariaChecked", true);
                    }
                    else {
                        if (!this.isHidden()) {
                            this.states.applyAnimate("default");
                        }
                        this.set("ariaChecked", false);
                    }
                    this.markDirtyAccessibility();
                }
                if (this.isDirty("disabled")) {
                    if (this.get("disabled")) {
                        this.states.applyAnimate("disabled");
                        this.set("ariaChecked", false);
                    }
                    else {
                        if (!this.isHidden()) {
                            this.states.applyAnimate("default");
                        }
                        this.set("ariaChecked", true);
                    }
                    this.markDirtyAccessibility();
                }
                this._statesHandled = true;
            }
        }
    });
    Object.defineProperty(Sprite.prototype, "_changed", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            _super.prototype._changed.call(this);
            var display = this._display;
            var events = this.events;
            if (this.isDirty("draggable")) {
                var draggable = this.get("draggable");
                if (draggable) {
                    this.set("interactive", true);
                    this._dragDp = new MultiDisposer([
                        events.on("pointerdown", function (ev) {
                            _this.dragStart(ev);
                        }),
                        events.on("globalpointermove", function (ev) {
                            _this.dragMove(ev);
                        }),
                        events.on("globalpointerup", function (ev) {
                            _this.dragStop(ev);
                        })
                    ]);
                }
                else {
                    if (this._dragDp) {
                        this._dragDp.dispose();
                        this._dragDp = undefined;
                    }
                }
            }
            if (this.isDirty("tooltipText")) {
                var tooltipText = this.get("tooltipText");
                if (tooltipText) {
                    this._tooltipDp = new MultiDisposer([
                        events.on("pointerover", function () {
                            _this.showTooltip();
                        }),
                        events.on("pointerout", function () {
                            if (_this.get("showTooltipOn") != "always") {
                                _this.hideTooltip();
                            }
                        })
                    ]);
                }
                else {
                    if (this._tooltipDp) {
                        this._tooltipDp.dispose();
                        this._tooltipDp = undefined;
                    }
                }
            }
            if (this.isDirty("toggleKey")) {
                var toggleKey_1 = this.get("toggleKey");
                if (toggleKey_1 && toggleKey_1 != "none") {
                    this._toggleDp = events.on("click", function () {
                        if (!_this._isDragging) {
                            _this.set(toggleKey_1, !_this.get(toggleKey_1));
                        }
                    });
                }
                else {
                    if (this._toggleDp) {
                        this._toggleDp.dispose();
                        this._toggleDp = undefined;
                    }
                }
            }
            this._handleStates();
            if (this.isDirty("opacity")) {
                display.alpha = Math.max(0, this.get("opacity", 1));
            }
            if (this.isDirty("rotation")) {
                this.markDirtyBounds();
                display.angle = this.get("rotation", 0);
            }
            if (this.isDirty("scale")) {
                this.markDirtyBounds();
                display.scale = this.get("scale", 0);
            }
            if (this.isDirty("centerX") || this.isDirty("centerY")) {
                this.markDirtyBounds();
                this.updatePivotPoint();
            }
            if (this.isDirty("visible") || this.isPrivateDirty("visible") || this.isDirty("forceHidden")) {
                if (!this.get("visible") || !this.getPrivate("visible") || this.get("forceHidden")) {
                    display.visible = false;
                    this.hideTooltip();
                }
                else {
                    display.visible = true;
                }
                this.markDirtyBounds();
                if (this.get("focusable")) {
                    this.markDirtyAccessibility();
                }
            }
            if (this.isDirty("width") || this.isDirty("height")) {
                this.markDirtyBounds();
                this._addPercentageSizeChildren();
                var parent_2 = this.parent;
                if (parent_2) {
                    if ((this.isDirty("width") && this.get("width") instanceof Percent) || (this.isDirty("height") && this.get("height") instanceof Percent)) {
                        parent_2.markDirty();
                        parent_2._prevWidth = 0;
                    }
                }
                this._sizeDirty = true;
            }
            if (this.isDirty("maxWidth") || this.isDirty("maxHeight") || this.isPrivateDirty("width") || this.isPrivateDirty("height")) {
                this.markDirtyBounds();
                this._sizeDirty = true;
            }
            if (this._sizeDirty) {
                this._updateSize();
            }
            if (this.isDirty("wheelable")) {
                var wheelable = this.get("wheelable");
                if (wheelable) {
                    this.set("interactive", true);
                }
                display.wheelable = wheelable ? true : false;
            }
            // Accessibility
            if (this.isDirty("tabindexOrder") || this.isDirty("focusableGroup")) {
                if (this.get("focusable")) {
                    this._root._registerTabindexOrder(this);
                }
                else {
                    this._root._unregisterTabindexOrder(this);
                }
            }
            if (this.isDirty("filter")) {
                //this.markDirtyBounds();
                display.filter = this.get("filter");
            }
            if (this.isDirty("cursorOverStyle")) {
                display.cursorOverStyle = this.get("cursorOverStyle");
            }
            if (this.isDirty("hoverOnFocus")) {
                if (this.get("hoverOnFocus")) {
                    this._focusDp = new MultiDisposer([
                        events.on("focus", function () {
                            // TODO: proper hover, not just tooltip
                            _this.showTooltip();
                        }),
                        events.on("blur", function () {
                            // TODO: proper hover, not just tooltip
                            _this.hideTooltip();
                        })
                    ]);
                }
                else {
                    if (this._focusDp) {
                        this._focusDp.dispose();
                        this._focusDp = undefined;
                    }
                }
            }
            if (this.isDirty("focusable")) {
                if (this.get("focusable")) {
                    this._root._registerTabindexOrder(this);
                }
                else {
                    this._root._unregisterTabindexOrder(this);
                }
                this.markDirtyAccessibility();
            }
            if (this.isDirty("role") || this.isDirty("ariaLive") || this.isDirty("ariaChecked") || this.isDirty("ariaHidden") || this.isDirty("ariaOrientation") || this.isDirty("ariaValueNow") || this.isDirty("ariaValueMin") || this.isDirty("ariaValueMax") || this.isDirty("ariaValueText") || this.isDirty("ariaLabel") || this.isDirty("ariaControls")) {
                // display.accessibility.ariaLabel = populateString(this, this.get("ariaLabel", ""));
                // @todo make sure ariaLabel gets populated in Root
                this.markDirtyAccessibility();
            }
            if (this.isDirty("exportable")) {
                display.exportable = this.get("exportable");
            }
            if (this.isDirty("interactive")) {
                var events_1 = this.events;
                if (this.get("interactive")) {
                    this._hoverDp = new MultiDisposer([
                        events_1.on("click", function (ev) {
                            if ($utils.isTouchEvent(ev.originalEvent)) {
                                if (!_this.getPrivate("touchHovering")) {
                                    _this.setTimeout(function () {
                                        _this._handleOver();
                                        if (_this.get("tooltipText")) {
                                            _this.showTooltip();
                                        }
                                        _this.setPrivateRaw("touchHovering", true);
                                    }, 10);
                                }
                            }
                        }),
                        events_1.on("globalpointerup", function (ev) {
                            if ($utils.isTouchEvent(ev.originalEvent)) {
                                if (_this.getPrivate("touchHovering")) {
                                    _this._handleOut();
                                    if (_this.get("tooltipText")) {
                                        _this.hideTooltip();
                                    }
                                }
                                _this.setPrivateRaw("touchHovering", false);
                            }
                            if (_this._isDown) {
                                _this._handleUp(ev);
                            }
                            //this._isDown = false;
                        }),
                        events_1.on("pointerover", function () {
                            _this._handleOver();
                        }),
                        events_1.on("pointerout", function () {
                            _this._handleOut();
                        }),
                        events_1.on("pointerdown", function (e) {
                            _this._handleDown(e);
                        })
                    ]);
                }
                else {
                    this._display.interactive = false;
                    if (this._hoverDp) {
                        this._hoverDp.dispose();
                        this._hoverDp = undefined;
                    }
                }
            }
            if (this.isDirty("forceInactive")) {
                this._display.inactive = this.get("forceInactive", false);
            }
            if (this.get("showTooltipOn") == "always" && this._display.visible) {
                this.showTooltip();
            }
        }
    });
    /**
     * @ignore
     * @todo should this be user-accessible?
     */
    Object.defineProperty(Sprite.prototype, "dragStart", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (e) {
            this._dragEvent = e;
            this.events.stopParentDispatch();
        }
    });
    /**
     * @ignore
     * @todo should this be user-accessible?
     */
    Object.defineProperty(Sprite.prototype, "dragStop", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (e) {
            this._dragEvent = undefined;
            this._dragPoint = undefined;
            this.events.stopParentDispatch();
            if (this._isDragging) {
                this._isDragging = false;
                var type = "dragstop";
                if (this.events.isEnabled(type)) {
                    this.events.dispatch(type, {
                        type: type,
                        target: this,
                        originalEvent: e.originalEvent,
                        point: e.point,
                        simulated: e.simulated,
                    });
                }
            }
        }
    });
    Object.defineProperty(Sprite.prototype, "_handleOver", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (!this.isHidden()) {
                if (this.get("active") && this.states.lookup("hoverActive")) {
                    this.states.applyAnimate("hoverActive");
                }
                else if (this.get("disabled") && this.states.lookup("hoverDisabled")) {
                    this.states.applyAnimate("hoverDisabled");
                }
                else {
                    this.states.applyAnimate("hover");
                }
            }
        }
    });
    Object.defineProperty(Sprite.prototype, "_handleOut", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (!this.isHidden()) {
                if (this.get("active") && this.states.lookup("active")) {
                    this.states.applyAnimate("active");
                }
                else if (this.get("disabled") && this.states.lookup("disabled")) {
                    this.states.applyAnimate("disabled");
                }
                else {
                    if (this.states.lookup("hover") || this.states.lookup("hoverActive")) {
                        this.states.applyAnimate("default");
                    }
                }
            }
        }
    });
    Object.defineProperty(Sprite.prototype, "_handleUp", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (e) {
            if (!this.isHidden()) {
                if (this.get("active") && this.states.lookup("active")) {
                    this.states.applyAnimate("active");
                }
                else if (this.get("disabled") && this.states.lookup("disabled")) {
                    this.states.applyAnimate("disabled");
                }
                else if (this.states.lookup("down")) {
                    if (this.isHover()) {
                        this.states.applyAnimate("hover");
                    }
                    else {
                        this.states.applyAnimate("default");
                    }
                }
                // @todo remove this once migrated to _downPoints
                this._downPoint = undefined;
                var pointerId = $utils.getPointerId(e.originalEvent);
                delete this._downPoints[pointerId];
                if ($object.keys(this._downPoints).length == 0) {
                    this._isDown = false;
                }
            }
        }
    });
    Object.defineProperty(Sprite.prototype, "_hasMoved", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (e) {
            // @todo remove this once migrated to _downPoints
            // if (this._downPoint) {
            // 	const x = Math.abs(this._downPoint.x - e.point.x);
            // 	const y = Math.abs(this._downPoint.y - e.point.y);
            // 	return (x > 5) || (y > 5);
            // }
            var pointerId = $utils.getPointerId(e.originalEvent);
            var downPoint = this._downPoints[pointerId];
            if (downPoint) {
                var x = Math.abs(downPoint.x - e.point.x);
                var y = Math.abs(downPoint.y - e.point.y);
                return (x > 5) || (y > 5);
            }
            return false;
        }
    });
    Object.defineProperty(Sprite.prototype, "_hasDown", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return $object.keys(this._downPoints).length > 0;
        }
    });
    Object.defineProperty(Sprite.prototype, "_handleDown", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (e) {
            var parent = this.parent;
            if (parent && !this.get("draggable")) {
                parent._handleDown(e);
            }
            if (this.get("interactive") && !this.isHidden()) {
                if (this.states.lookup("down")) {
                    this.states.applyAnimate("down");
                }
                this._downPoint = {
                    x: e.point.x,
                    y: e.point.y
                };
                // @todo remove this once migrated to _downPoints
                this._isDown = true;
                var pointerId = $utils.getPointerId(e.originalEvent);
                this._downPoints[pointerId] = {
                    x: e.point.x,
                    y: e.point.y
                };
            }
        }
    });
    /**
     * @ignore
     * @todo should this be user-accessible?
     */
    Object.defineProperty(Sprite.prototype, "dragMove", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (e) {
            var dragEvent = this._dragEvent;
            if (dragEvent) {
                var angle = 0;
                var parent_3 = this.parent;
                while (parent_3 != null) {
                    angle += parent_3.get("rotation", 0);
                    parent_3 = parent_3.parent;
                }
                var x = e.point.x - dragEvent.point.x;
                var y = e.point.y - dragEvent.point.y;
                var events = this.events;
                if (dragEvent.simulated && !this._isDragging) {
                    this._isDragging = true;
                    this._dragEvent = e;
                    this._dragPoint = {
                        x: this.x(),
                        y: this.y()
                    };
                    var type = "dragstart";
                    if (events.isEnabled(type)) {
                        events.dispatch(type, {
                            type: type,
                            target: this,
                            originalEvent: e.originalEvent,
                            point: e.point,
                            simulated: e.simulated,
                        });
                    }
                }
                if (this._isDragging) {
                    var dragPoint = this._dragPoint;
                    this.set("x", dragPoint.x + x * $math.cos(angle) + y * $math.sin(angle));
                    this.set("y", dragPoint.y + y * $math.cos(angle) - x * $math.sin(angle));
                    var type = "dragged";
                    if (events.isEnabled(type)) {
                        events.dispatch(type, {
                            type: type,
                            target: this,
                            originalEvent: e.originalEvent,
                            point: e.point,
                            simulated: e.simulated,
                        });
                    }
                }
                else {
                    if (Math.hypot(x, y) > 5) {
                        this._isDragging = true;
                        this._dragEvent = e;
                        this._dragPoint = {
                            x: this.x(),
                            y: this.y()
                        };
                        var type = "dragstart";
                        if (events.isEnabled(type)) {
                            events.dispatch(type, {
                                type: type,
                                target: this,
                                originalEvent: e.originalEvent,
                                point: e.point,
                                simulated: e.simulated
                            });
                        }
                    }
                }
            }
        }
    });
    Object.defineProperty(Sprite.prototype, "_updateSize", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
        }
    });
    Object.defineProperty(Sprite.prototype, "_getBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._localBounds = this._display.getLocalBounds();
        }
    });
    /**
     * Returns depth (how deep in the hierachy of the content tree) of this
     * element.
     *
     * @return Depth
     */
    Object.defineProperty(Sprite.prototype, "depth", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var self = this.parent;
            var depth = 0;
            while (true) {
                if (self) {
                    ++depth;
                    self = self.parent;
                }
                else {
                    return depth;
                }
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Sprite.prototype, "markDirtySize", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._sizeDirty = true;
            this.markDirty();
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Sprite.prototype, "markDirtyBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var display = this._display;
            if (this.get("isMeasured")) {
                this._root._addDirtyBounds(this);
                display.isMeasured = true;
                display.invalidateBounds();
                var parent_4 = this.parent;
                if (parent_4 && this.get("position") != "absolute") {
                    if (parent_4.get("width") == null || parent_4.get("height") == null || parent_4.get("layout")) {
                        parent_4.markDirtyBounds();
                    }
                }
                if (this.get("focusable") && this.isFocus()) {
                    this.markDirtyAccessibility();
                }
            }
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Sprite.prototype, "markDirtyAccessibility", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            //if (this._root.focused(this)) {
            this._root._invalidateAccessibility(this);
            //}
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Sprite.prototype, "markDirtyLayer", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            //this._display.markDirtyLayer(this.isDirty("opacity") || this.isDirty("visible")); https://codepen.io/team/amcharts/pen/gOWZPmP <- problems
            this._display.markDirtyLayer(true);
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Sprite.prototype, "markDirty", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype.markDirty.call(this);
            this.markDirtyLayer();
        }
    });
    Object.defineProperty(Sprite.prototype, "_updateBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var oldBounds = this._adjustedLocalBounds;
            var newBounds;
            // if display.visible == false, it still returns bounds
            if (!this.get("visible") || !this.getPrivate("visible") || this.get("forceHidden")) {
                newBounds = {
                    left: 0,
                    right: 0,
                    top: 0,
                    bottom: 0
                };
                this._localBounds = newBounds;
                this._adjustedLocalBounds = newBounds;
            }
            else {
                this._getBounds();
                this._fixMinBounds(this._localBounds);
                this.updatePivotPoint();
                this._adjustedLocalBounds = this._display.getAdjustedBounds(this._localBounds);
                newBounds = this._adjustedLocalBounds;
            }
            if (!oldBounds || (oldBounds.left !== newBounds.left || oldBounds.top !== newBounds.top || oldBounds.right !== newBounds.right || oldBounds.bottom !== newBounds.bottom)) {
                var eventType = "boundschanged";
                if (this.events.isEnabled(eventType)) {
                    this.events.dispatch(eventType, { type: eventType, target: this });
                }
                if (this.parent) {
                    this.parent.markDirty();
                    this.parent.markDirtyBounds();
                }
            }
        }
    });
    Object.defineProperty(Sprite.prototype, "_fixMinBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (bounds) {
            var minWidth = this.get("minWidth");
            var minHeight = this.get("minHeight");
            if ($type.isNumber(minWidth)) {
                if (bounds.right - bounds.left < minWidth) {
                    bounds.right = bounds.left + minWidth;
                }
            }
            if ($type.isNumber(minHeight)) {
                if (bounds.bottom - bounds.top < minHeight) {
                    bounds.bottom = bounds.top + minHeight;
                }
            }
            var privateWidth = this.getPrivate("width");
            var privateHeight = this.getPrivate("height");
            if ($type.isNumber(privateWidth)) {
                bounds.right = bounds.left + privateWidth;
            }
            if ($type.isNumber(privateHeight)) {
                bounds.bottom = bounds.top + privateHeight;
            }
        }
    });
    Object.defineProperty(Sprite.prototype, "_removeParent", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (parent) {
            if (parent) {
                parent.children.removeValue(this);
                $array.removeFirst(parent._percentageSizeChildren, this);
                $array.removeFirst(parent._percentagePositionChildren, this);
            }
        }
    });
    Object.defineProperty(Sprite.prototype, "_clearDirty", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._clearDirty.call(this);
            this._sizeDirty = false;
            this._statesHandled = false;
        }
    });
    /**
     * Simulate hover over element.
     */
    Object.defineProperty(Sprite.prototype, "hover", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this.showTooltip();
            this._handleOver();
        }
    });
    /**
     * Simulate unhover over element.
     */
    Object.defineProperty(Sprite.prototype, "unhover", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this.hideTooltip();
            this._handleOut();
        }
    });
    /**
     * Shows element's [[Tooltip]].
     */
    Object.defineProperty(Sprite.prototype, "showTooltip", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (point) {
            var tooltip = this.getTooltip();
            var tooltipText = this.get("tooltipText");
            if (tooltipText && tooltip) {
                var tooltipPosition = this.get("tooltipPosition");
                var tooltipTarget = this.getPrivate("tooltipTarget", this);
                if (tooltipPosition == "fixed" || !point) {
                    this._display._setMatrix();
                    point = this.toGlobal(tooltipTarget._getTooltipPoint());
                }
                tooltip.set("pointTo", point);
                tooltip.set("tooltipTarget", tooltipTarget);
                if (!tooltip.get("x")) {
                    tooltip.set("x", point.x);
                }
                if (!tooltip.get("y")) {
                    tooltip.set("y", point.y);
                }
                tooltip.label.set("text", tooltipText);
                var dataItem = this.dataItem;
                if (dataItem) {
                    tooltip.label._setDataItem(dataItem);
                }
                if (this.get("showTooltipOn") == "always" && (point.x < 0 || point.x > this._root.width() || point.y < 0 || point.y > this._root.height())) {
                    this.hideTooltip();
                    return;
                }
                tooltip.label.text.markDirtyText();
                var promise = tooltip.show();
                this.setPrivateRaw("showingTooltip", true);
                return promise;
            }
        }
    });
    /**
     * Hides element's [[Tooltip]].
     */
    Object.defineProperty(Sprite.prototype, "hideTooltip", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var tooltip = this.getTooltip();
            if (tooltip) {
                var promise = tooltip.hide();
                this.setPrivateRaw("showingTooltip", false);
                return promise;
            }
        }
    });
    Object.defineProperty(Sprite.prototype, "_getTooltipPoint", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var bounds = this._localBounds;
            if (bounds) {
                var x = 0;
                var y = 0;
                if (!this.get("isMeasured")) {
                    x = $utils.relativeToValue(this.get("tooltipX", 0), this.width());
                    y = $utils.relativeToValue(this.get("tooltipY", 0), this.height());
                }
                else {
                    x = bounds.left + $utils.relativeToValue(this.get("tooltipX", 0), bounds.right - bounds.left);
                    y = bounds.top + $utils.relativeToValue(this.get("tooltipY", 0), bounds.bottom - bounds.top);
                }
                return { x: x, y: y };
            }
            return { x: 0, y: 0 };
        }
    });
    /**
     * Returns [[Tooltip]] used for this element.
     *
     * @return Tooltip
     */
    Object.defineProperty(Sprite.prototype, "getTooltip", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var tooltip = this.get("tooltip");
            if (!tooltip) {
                var parent_5 = this.parent;
                if (parent_5) {
                    return parent_5.getTooltip();
                }
            }
            else {
                return tooltip;
            }
        }
    });
    Object.defineProperty(Sprite.prototype, "_updatePosition", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var parent = this.parent;
            var dx = this.get("dx", 0);
            var dy = this.get("dy", 0);
            var x = this.get("x");
            var _x = this.getPrivate("x");
            var xx = 0;
            var yy = 0;
            var position = this.get("position");
            if (x instanceof Percent) {
                if (parent) {
                    x = parent.innerWidth() * x.value + parent.get("paddingLeft", 0);
                }
                else {
                    x = 0;
                }
            }
            if ($type.isNumber(x)) {
                xx = x + dx;
            }
            else {
                if (_x != null) {
                    xx = _x;
                }
                else if (parent) {
                    if (position == "relative") {
                        xx = parent.get("paddingLeft", 0) + dx;
                    }
                }
            }
            var y = this.get("y");
            var _y = this.getPrivate("y");
            if (y instanceof Percent) {
                if (parent) {
                    y = parent.innerHeight() * y.value + parent.get("paddingTop", 0);
                }
                else {
                    y = 0;
                }
            }
            if ($type.isNumber(y)) {
                yy = y + dy;
            }
            else {
                if (_y != null) {
                    yy = _y;
                }
                else if (parent) {
                    if (position == "relative") {
                        yy = parent.get("paddingTop", 0) + dy;
                    }
                }
            }
            var display = this._display;
            if (display.x != xx || display.y != yy) {
                display.invalidateBounds();
                display.x = xx;
                display.y = yy;
                var eventType = "positionchanged";
                if (this.events.isEnabled(eventType)) {
                    this.events.dispatch(eventType, { type: eventType, target: this });
                }
            }
            // Update tooltip position together with the Sprite
            if (this.getPrivate("showingTooltip")) {
                this.showTooltip();
            }
        }
    });
    /**
     * Returns element's actual X position in pixels.
     *
     * @return X (px)
     */
    Object.defineProperty(Sprite.prototype, "x", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var x = this.get("x");
            var _x = this.getPrivate("x");
            var parent = this.parent;
            if (parent) {
                if (x instanceof Percent) {
                    return $utils.relativeToValue(x, parent.innerWidth()) + parent.get("paddingLeft", 0);
                }
                else {
                    if (!$type.isNumber(x)) {
                        if (_x != null) {
                            return _x;
                        }
                        else {
                            return parent.get("paddingLeft", this._display.x);
                        }
                    }
                    else {
                        return x;
                    }
                }
            }
            return this._display.x;
        }
    });
    /**
     * Returns element's actual Y position in pixels.
     *
     * @return Y (px)
     */
    Object.defineProperty(Sprite.prototype, "y", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _y = this.getPrivate("y");
            if (_y != null) {
                return _y;
            }
            var y = this.get("y");
            var parent = this.parent;
            if (parent) {
                if (y instanceof Percent) {
                    return $utils.relativeToValue(y, parent.innerHeight()) + parent.get("paddingTop", 0);
                }
                else {
                    if (!$type.isNumber(y)) {
                        if (_y != null) {
                            return _y;
                        }
                        else {
                            return parent.get("paddingTop", this._display.y);
                        }
                    }
                    else {
                        return y;
                    }
                }
            }
            return this._display.y;
        }
    });
    Object.defineProperty(Sprite.prototype, "_dispose", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._dispose.call(this);
            this._display.dispose();
            this._removeTemplateField();
            this._removeParent(this.parent);
            var focusElement = this.getPrivate("focusElement");
            if (focusElement) {
                $array.each(focusElement.disposers, function (x) {
                    x.dispose();
                });
            }
            var tooltip = this.get("tooltip");
            if (tooltip) {
                tooltip.dispose();
            }
            this.markDirty();
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(Sprite.prototype, "adjustedLocalBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._fixMinBounds(this._adjustedLocalBounds);
            return this._adjustedLocalBounds;
        }
    });
    /**
     * Returns local coordinates of the element's bounds.
     *
     * @ignore
     * @return Global bounds
     */
    Object.defineProperty(Sprite.prototype, "localBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this._localBounds;
        }
    });
    /**
     * Returns adjusted local coordinates of the element's bounds.
     *
     * @ignore
     * @return Global bounds
     */
    Object.defineProperty(Sprite.prototype, "bounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var bounds = this._adjustedLocalBounds;
            var x = this.x();
            var y = this.y();
            return { left: bounds.left + x, right: bounds.right + x, top: bounds.top + y, bottom: bounds.bottom + y };
        }
    });
    /**
     * Returns global coordinates of the element's bounds.
     *
     * @ignore
     * @return Global bounds
     */
    Object.defineProperty(Sprite.prototype, "globalBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var bounds = this.localBounds();
            var p0 = this.toGlobal({ x: bounds.left, y: bounds.top });
            var p1 = this.toGlobal({ x: bounds.right, y: bounds.top });
            var p2 = this.toGlobal({ x: bounds.right, y: bounds.bottom });
            var p3 = this.toGlobal({ x: bounds.left, y: bounds.bottom });
            return {
                left: Math.min(p0.x, p1.x, p2.x, p3.x),
                top: Math.min(p0.y, p1.y, p2.y, p3.y),
                right: Math.max(p0.x, p1.x, p2.x, p3.x),
                bottom: Math.max(p0.y, p1.y, p2.y, p3.y)
            };
        }
    });
    Object.defineProperty(Sprite.prototype, "_onShow", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_duration) {
        }
    });
    Object.defineProperty(Sprite.prototype, "_onHide", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_duration) {
        }
    });
    /**
     * Plays initial reveal animation regardless if element is currently hidden
     * or visible.
     *
     * @param   duration  Duration of the animation in milliseconds
     * @param   delay     Delay showing of the element by X milliseconds
     * @return            Promise
     */
    Object.defineProperty(Sprite.prototype, "appear", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (duration, delay) {
            return __awaiter(this, void 0, void 0, function () {
                var _this = this;
                return __generator(this, function (_a) {
                    switch (_a.label) {
                        case 0: return [4 /*yield*/, this.hide(0)];
                        case 1:
                            _a.sent();
                            if (delay) {
                                return [2 /*return*/, new Promise(function (success, _error) {
                                        _this.setTimeout(function () {
                                            success(_this.show(duration));
                                        }, delay);
                                    })];
                            }
                            else {
                                return [2 /*return*/, this.show(duration)];
                            }
                            return [2 /*return*/];
                    }
                });
            });
        }
    });
    /**
     * Shows currently hidden element and returns a `Promise` which completes
     * when all showing animations are finished.
     *
     * ```TypeScript
     * series.show().then(function(ev) {
     *   console.log("Series is now fully visible");
     * })
     * ```
     * ```JavaScript
     * series.show().then(function(ev) {
     *   console.log("Series is now fully visible");
     * })
     * ```
     *
     * @return Promise
     */
    Object.defineProperty(Sprite.prototype, "show", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (duration) {
            return __awaiter(this, void 0, void 0, function () {
                var animations;
                return __generator(this, function (_a) {
                    switch (_a.label) {
                        case 0:
                            if (!!this._isShowing) return [3 /*break*/, 2];
                            this._isHidden = false;
                            this._isShowing = true;
                            this._isHiding = false;
                            if (this.states.lookup("default").get("visible")) {
                                this.set("visible", true);
                            }
                            this._onShow(duration);
                            animations = this.states.applyAnimate("default", duration);
                            return [4 /*yield*/, waitForAnimations(animations)];
                        case 1:
                            _a.sent();
                            this._isShowing = false;
                            _a.label = 2;
                        case 2: return [2 /*return*/];
                    }
                });
            });
        }
    });
    /**
     * Hides the element and returns a `Promise` which completes when all hiding
     * animations are finished.
     *
     * ```TypeScript
     * series.hide().then(function(ev) {
     *   console.log("Series finished hiding");
     * })
     * ```
     * ```JavaScript
     * series.hide().then(function(ev) {
     *   console.log("Series finished hiding");
     * })
     * ```
     *
     * @return Promise
     */
    Object.defineProperty(Sprite.prototype, "hide", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (duration) {
            return __awaiter(this, void 0, void 0, function () {
                var state, animations;
                return __generator(this, function (_a) {
                    switch (_a.label) {
                        case 0:
                            if (!(!this._isHiding && !this._isHidden)) return [3 /*break*/, 2];
                            this._isHiding = true;
                            this._isShowing = false;
                            state = this.states.lookup("hidden");
                            if (!state) {
                                state = this.states.create("hidden", {
                                    "opacity": 0,
                                    "visible": false
                                });
                            }
                            this._isHidden = true;
                            this._onHide(duration);
                            animations = this.states.applyAnimate("hidden", duration);
                            return [4 /*yield*/, waitForAnimations(animations)];
                        case 1:
                            _a.sent();
                            this._isHiding = false;
                            _a.label = 2;
                        case 2: return [2 /*return*/];
                    }
                });
            });
        }
    });
    /**
     * Returns `true` if this element is currently hidden.
     *
     * @return Is hidden?
     */
    Object.defineProperty(Sprite.prototype, "isHidden", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this._isHidden;
        }
    });
    /**
     * Returns `true` if this element is currently animating to a default state.
     *
     * @return Is showing?
     */
    Object.defineProperty(Sprite.prototype, "isShowing", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this._isShowing;
        }
    });
    /**
     * Returns `true` if this element is currently animating to a hidden state.
     *
     * @return Is hiding?
     */
    Object.defineProperty(Sprite.prototype, "isHiding", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this._isHiding;
        }
    });
    /**
     * Returns `true` if this element is currently hovered by a pointer.
     *
     * @return Is hovered?
     */
    Object.defineProperty(Sprite.prototype, "isHover", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this._display.hovering();
        }
    });
    /**
     * Returns `true` if this element does currently have focus.
     *
     * @return Is focused?
     */
    Object.defineProperty(Sprite.prototype, "isFocus", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this._root.focused(this);
        }
    });
    /**
     * Returns `true` if this element is currently being dragged.
     *
     * @return Is dragged?
     */
    Object.defineProperty(Sprite.prototype, "isDragging", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this._isDragging;
        }
    });
    /**
     * Returns width of this element in pixels.
     *
     * @return Width (px)
     */
    Object.defineProperty(Sprite.prototype, "width", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var width = this.get("width");
            var maxWidth = this.get("maxWidth");
            var minWidth = this.get("minWidth");
            var privateWidth = this.getPrivate("width");
            var w = 0;
            if ($type.isNumber(privateWidth)) {
                w = privateWidth;
            }
            else {
                if (width == null) {
                    if (this._adjustedLocalBounds) {
                        w = this._adjustedLocalBounds.right - this._adjustedLocalBounds.left;
                    }
                }
                else {
                    if (width instanceof Percent) {
                        var parent_6 = this.parent;
                        if (parent_6) {
                            w = parent_6.innerWidth() * width.value;
                        }
                        else {
                            w = this._root.width() * width.value;
                        }
                    }
                    else if ($type.isNumber(width)) {
                        w = width;
                    }
                }
            }
            if ($type.isNumber(minWidth)) {
                w = Math.max(minWidth, w);
            }
            if ($type.isNumber(maxWidth)) {
                w = Math.min(maxWidth, w);
            }
            return w;
        }
    });
    /**
     * Returns maximum allowed width of this element in pixels.
     *
     * @return Maximum width (px)
     */
    Object.defineProperty(Sprite.prototype, "maxWidth", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var maxWidth = this.get("maxWidth");
            if ($type.isNumber(maxWidth)) {
                return maxWidth;
            }
            else {
                var width = this.get("width");
                if ($type.isNumber(width)) {
                    return width;
                }
            }
            var parent = this.parent;
            if (parent) {
                return parent.innerWidth();
            }
            return this._root.width();
        }
    });
    /**
     * Returns maximum allowed height of this element in pixels.
     *
     * @return Maximum height (px)
     */
    Object.defineProperty(Sprite.prototype, "maxHeight", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var maxHeight = this.get("maxHeight");
            if ($type.isNumber(maxHeight)) {
                return maxHeight;
            }
            else {
                var height = this.get("height");
                if ($type.isNumber(height)) {
                    return height;
                }
            }
            var parent = this.parent;
            if (parent) {
                return parent.innerHeight();
            }
            return this._root.height();
        }
    });
    /**
     * Returns height of this element in pixels.
     *
     * @return Height (px)
     */
    Object.defineProperty(Sprite.prototype, "height", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var height = this.get("height");
            var maxHeight = this.get("maxHeight");
            var minHeight = this.get("minHeight");
            var privateHeight = this.getPrivate("height");
            var h = 0;
            if ($type.isNumber(privateHeight)) {
                h = privateHeight;
            }
            else {
                if (height == null) {
                    if (this._adjustedLocalBounds) {
                        h = this._adjustedLocalBounds.bottom - this._adjustedLocalBounds.top;
                    }
                }
                else {
                    if (height instanceof Percent) {
                        var parent_7 = this.parent;
                        if (parent_7) {
                            h = parent_7.innerHeight() * height.value;
                        }
                        else {
                            h = this._root.height() * height.value;
                        }
                    }
                    else if ($type.isNumber(height)) {
                        h = height;
                    }
                }
            }
            if ($type.isNumber(minHeight)) {
                h = Math.max(minHeight, h);
            }
            if ($type.isNumber(maxHeight)) {
                h = Math.min(maxHeight, h);
            }
            return h;
        }
    });
    Object.defineProperty(Sprite.prototype, "_findStaticTemplate", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (f) {
            // templateField overrides template
            if (this._templateField && f(this._templateField)) {
                return this._templateField;
            }
            return _super.prototype._findStaticTemplate.call(this, f);
        }
    });
    Object.defineProperty(Sprite.prototype, "_walkParents", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (f) {
            if (this._parent) {
                this._walkParent(f);
            }
        }
    });
    Object.defineProperty(Sprite.prototype, "_walkParent", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (f) {
            if (this._parent) {
                this._parent._walkParent(f);
            }
            f(this);
        }
    });
    Object.defineProperty(Sprite.prototype, "parent", {
        /**
         * Parent [[Container]] of this element.
         *
         * @return Parent container
         */
        get: function () {
            return this._parent;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(Sprite.prototype, "_setParent", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (parent, updateChildren) {
            if (updateChildren === void 0) { updateChildren = false; }
            var prevParent = this._parent;
            if (parent !== prevParent) {
                this.markDirtyBounds();
                parent.markDirty();
                this._parent = parent;
                if (updateChildren) {
                    this._removeParent(prevParent);
                    if (parent) {
                        this._addPercentageSizeChildren();
                        this._addPercentagePositionChildren();
                    }
                }
                this.markDirtyPosition();
                this._applyThemes();
            }
        }
    });
    /**
     * Returns an instance of [[NumberFormatter]] used in this element.
     *
     * If this element does not have it set, global one form [[Root]] is used.
     *
     * @see {@link https://www.amcharts.com/docs/v5/concepts/using-formatters/} for more info
     * @return NumberFormatter instace
     */
    Object.defineProperty(Sprite.prototype, "getNumberFormatter", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this.get("numberFormatter", this._root.numberFormatter);
        }
    });
    /**
     * Returns an instance of [[DateFormatter]] used in this element.
     *
     * If this element does not have it set, global one form [[Root]] is used.
     *
     * @see {@link https://www.amcharts.com/docs/v5/concepts/using-formatters/} for more info
     * @return DateFormatter instace
     */
    Object.defineProperty(Sprite.prototype, "getDateFormatter", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this.get("dateFormatter", this._root.dateFormatter);
        }
    });
    /**
     * Returns an instance of [[DurationFormatter]] used in this element.
     *
     * If this element does not have it set, global one form [[Root]] is used.
     *
     * @see {@link https://www.amcharts.com/docs/v5/concepts/using-formatters/} for more info
     * @return DurationFormatter instace
     */
    Object.defineProperty(Sprite.prototype, "getDurationFormatter", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this.get("durationFormatter", this._root.durationFormatter);
        }
    });
    /**
     * Converts X/Y coordinate within this element to a global coordinate.
     *
     * @param  point  Local coordinate
     * @return        Global coordinate
     */
    Object.defineProperty(Sprite.prototype, "toGlobal", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (point) {
            return this._display.toGlobal(point);
        }
    });
    /**
     * Converts global X/Y coordinate to a coordinate within this element.
     *
     * @param  point  Global coordinate
     * @return        Local coordinate
     */
    Object.defineProperty(Sprite.prototype, "toLocal", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (point) {
            return this._display.toLocal(point);
        }
    });
    Object.defineProperty(Sprite.prototype, "_getDownPoint", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var id = this._getDownPointId();
            if (id) {
                return this._downPoints[id];
            }
        }
    });
    Object.defineProperty(Sprite.prototype, "_getDownPointId", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (this._downPoints) {
                return $object.keysOrdered(this._downPoints, function (a, b) {
                    if (a > b) {
                        return 1;
                    }
                    if (a < b) {
                        return -1;
                    }
                    return 0;
                })[0];
            }
        }
    });
    /**
     * Moves sprite to the end of the parent's children array.
     *
     * Depending on `layout` setting of the parten container, it may effect the
     * positioning or overlapping order of the elements.
     */
    Object.defineProperty(Sprite.prototype, "toFront", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var parent = this.parent;
            if (parent) {
                parent.children.moveValue(this, parent.children.length - 1);
            }
        }
    });
    /**
     * Moves sprite to the beginning of the parent's children array.
     *
     * Depending on `layout` setting of the parten container, it may effect the
     * positioning or overlapping order of the elements.
     */
    Object.defineProperty(Sprite.prototype, "toBack", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var parent = this.parent;
            if (parent) {
                parent.children.moveValue(this, 0);
            }
        }
    });
    Object.defineProperty(Sprite, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "Sprite"
    });
    Object.defineProperty(Sprite, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: Entity.classNames.concat([Sprite.className])
    });
    return Sprite;
}(Entity));
export { Sprite };
//# sourceMappingURL=Sprite.js.map