/** @ignore */ /** */
import { __extends } from "tslib";
import { BlendMode } from "./Renderer";
import { Color } from "../../util/Color";
import { Matrix } from "../../util/Matrix";
import { Percent, percent } from "../../util/Percent";
import { Throttler } from "../../util/Throttler";
import { ArrayDisposer, Disposer, DisposerClass, CounterDisposer, MultiDisposer } from "../../util/Disposer";
import { TextFormatter } from "../../util/TextFormatter";
import * as $utils from "../../util/Utils";
import * as $array from "../../util/Array";
import * as $object from "../../util/Object";
import * as $type from "../../util/Type";
import * as $math from "../../util/Math";
import arcToBezier from 'svg-arc-to-cubic-bezier';
/**
 * @ignore
 */
function checkArgs(name, actual, expected) {
    if (actual !== expected) {
        throw new Error("Required " + expected + " arguments for " + name + " but got " + actual);
    }
}
/**
 * @ignore
 */
function checkMinArgs(name, actual, expected) {
    if (actual < expected) {
        throw new Error("Required at least " + expected + " arguments for " + name + " but got " + actual);
    }
}
/**
 * @ignore
 */
function checkEvenArgs(name, actual, expected) {
    checkMinArgs(name, actual, expected);
    if ((actual % expected) !== 0) {
        throw new Error("Arguments for " + name + " must be in pairs of " + expected);
    }
}
/**
 * @ignore
 */
function assertBinary(value) {
    if (value === 0 || value === 1) {
        return value;
    }
    else {
        throw new Error("Flag must be 0 or 1");
    }
}
//  1 -> 0xffffff * (2 / 2)
//  2 -> 0xffffff * (1 / 2)
//
//  3 -> 0xffffff * (3 / 4)
//  4 -> 0xffffff * (1 / 4)
//
//  5 -> 0xffffff * (7 / 8)
//  6 -> 0xffffff * (5 / 8)
//  7 -> 0xffffff * (3 / 8)
//  8 -> 0xffffff * (1 / 8)
//
//  9 -> 0xffffff * (15 / 16)
// 10 -> 0xffffff * (13 / 16)
// 11 -> 0xffffff * (11 / 16)
// 12 -> 0xffffff *  (9 / 16)
// 13 -> 0xffffff *  (7 / 16)
// 14 -> 0xffffff *  (5 / 16)
// 15 -> 0xffffff *  (3 / 16)
// 16 -> 0xffffff *  (1 / 16)
// @todo remove this old color distribution algo if the new one pans out
// function distributeIdBAK(id: number): number {
// 	if (id === 1) {
// 		return 0x000001;
// 	} else {
// 		// Finds the closest power of 2
// 		const base = Math.pow(2, Math.ceil(Math.log(id) / Math.log(2)));
// 		// Translates the id into an odd fraction index
// 		const index = ((base - id) * 2) + 1;
// 		// TODO is Math.round correct ?
// 		return Math.round(0xffffff * (index / base));
// 	}
// }
/**
 * Function by smeans:
 * https://lowcode.life/generating-unique-contrasting-colors-in-javascript/
 * @ignore
 */
function distributeId(id) {
    var rgb = [0, 0, 0];
    for (var i = 0; i < 24; i++) {
        rgb[i % 3] <<= 1;
        rgb[i % 3] |= id & 0x01;
        id >>= 1;
    }
    return (rgb[2] | 0) + (rgb[1] << 8) + (rgb[0] << 16);
}
/**
 * @ignore
 */
function eachTargets(hitTarget, f) {
    for (;;) {
        if (hitTarget.interactive) {
            if (!f(hitTarget)) {
                break;
            }
        }
        if (hitTarget._parent) {
            hitTarget = hitTarget._parent;
        }
        else {
            break;
        }
    }
}
// TODO feature detection for mouse/touch/pointer
/**
 * @ignore
 */
function onPointerEvent(element, name, f) {
    return $utils.addEventListener(element, $utils.getRendererEvent(name), function (event) {
        var touches = event.touches;
        if (touches) {
            if (touches.length == 0) {
                touches = event.changedTouches;
            }
            $array.each(touches, function (touch) {
                f(touch);
            });
        }
        else {
            f(event);
        }
    });
}
/**
 * @ignore
 */
function isTainted(image) {
    var canvas = document.createElement("canvas");
    canvas.width = 1;
    canvas.height = 1;
    var context = canvas.getContext("2d");
    context.drawImage(image, 0, 0, 1, 1);
    try {
        context.getImageData(0, 0, 1, 1);
        return false;
    }
    catch (err) {
        console.warn("Image \"" + image.src + "\" is loaded from different host and is not covered by CORS policy. For more information about the implications read here: https://www.amcharts.com/docs/v5/concepts/cors");
        return true;
    }
}
/**
 * This is needed to workaround a bug in iOS which causes it to not GC canvas elements.
 *
 * @ignore
 */
function clearCanvas(view) {
    view.width = 0;
    view.height = 0;
    view.style.width = "0px";
    view.style.height = "0px";
}
/**
 * @ignore
 */
var CanvasPivot = /** @class */ (function () {
    function CanvasPivot() {
        Object.defineProperty(this, "_x", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(this, "_y", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
    }
    Object.defineProperty(CanvasPivot.prototype, "x", {
        get: function () {
            return this._x;
        },
        set: function (value) {
            this._x = value;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(CanvasPivot.prototype, "y", {
        get: function () {
            return this._y;
        },
        set: function (value) {
            this._y = value;
        },
        enumerable: false,
        configurable: true
    });
    return CanvasPivot;
}());
export { CanvasPivot };
/**
 * @ignore
 */
var CanvasDisplayObject = /** @class */ (function (_super) {
    __extends(CanvasDisplayObject, _super);
    function CanvasDisplayObject(renderer) {
        var _this = _super.call(this) || this;
        Object.defineProperty(_this, "_layer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "mask", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: null
        });
        Object.defineProperty(_this, "visible", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: true
        });
        Object.defineProperty(_this, "exportable", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: true
        });
        Object.defineProperty(_this, "interactive", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(_this, "inactive", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(_this, "wheelable", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(_this, "isMeasured", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(_this, "buttonMode", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(_this, "alpha", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 1
        });
        Object.defineProperty(_this, "compoundAlpha", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 1
        });
        Object.defineProperty(_this, "angle", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "scale", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 1
        });
        Object.defineProperty(_this, "x", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "y", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "pivot", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new CanvasPivot()
        });
        Object.defineProperty(_this, "filter", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "cursorOverStyle", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_replacedCursorStyle", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_localMatrix", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Matrix()
        });
        Object.defineProperty(_this, "_matrix", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Matrix()
        });
        // TODO can this be replaced with _localMatrix ?
        Object.defineProperty(_this, "_uMatrix", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Matrix()
        });
        Object.defineProperty(_this, "_renderer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_parent", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_localBounds", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_bounds", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_colorId", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        _this._renderer = renderer;
        return _this;
    }
    Object.defineProperty(CanvasDisplayObject.prototype, "_dispose", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._renderer._removeObject(this);
        }
    });
    Object.defineProperty(CanvasDisplayObject.prototype, "getCanvas", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this.getLayer().view;
        }
    });
    Object.defineProperty(CanvasDisplayObject.prototype, "getLayer", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var self = this;
            for (;;) {
                if (self._layer) {
                    return self._layer;
                }
                else if (self._parent) {
                    self = self._parent;
                }
                else {
                    return this._renderer.defaultLayer;
                }
            }
        }
    });
    Object.defineProperty(CanvasDisplayObject.prototype, "setLayer", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (order, visible) {
            if (visible === void 0) { visible = true; }
            if (order == null) {
                this._layer = undefined;
            }
            else {
                this._layer = this._renderer.getLayer(order, visible);
                this._layer.visible = visible;
                if (this._parent) {
                    this._parent.registerChildLayer(this._layer);
                }
            }
        }
    });
    Object.defineProperty(CanvasDisplayObject.prototype, "markDirtyLayer", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this.getLayer().dirty = true;
        }
    });
    Object.defineProperty(CanvasDisplayObject.prototype, "clear", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this.invalidateBounds();
        }
    });
    Object.defineProperty(CanvasDisplayObject.prototype, "invalidateBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._localBounds = undefined;
        }
    });
    Object.defineProperty(CanvasDisplayObject.prototype, "_addBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_bounds) { }
    });
    Object.defineProperty(CanvasDisplayObject.prototype, "_getColorId", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (this._colorId === undefined) {
                this._colorId = this._renderer.paintId(this);
            }
            return this._colorId;
        }
    });
    Object.defineProperty(CanvasDisplayObject.prototype, "_isInteractive", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this.inactive == false && (this.interactive || this._renderer._forceInteractive > 0);
        }
    });
    Object.defineProperty(CanvasDisplayObject.prototype, "_isInteractiveMask", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this._isInteractive();
        }
    });
    Object.defineProperty(CanvasDisplayObject.prototype, "contains", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (child) {
            for (;;) {
                if (child === this) {
                    return true;
                }
                else if (child._parent) {
                    child = child._parent;
                }
                else {
                    return false;
                }
            }
        }
    });
    Object.defineProperty(CanvasDisplayObject.prototype, "toGlobal", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (point) {
            return this._matrix.apply(point);
        }
    });
    Object.defineProperty(CanvasDisplayObject.prototype, "toLocal", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (point) {
            return this._matrix.applyInverse(point);
        }
    });
    Object.defineProperty(CanvasDisplayObject.prototype, "getLocalMatrix", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._uMatrix.setTransform(0, 0, this.pivot.x, this.pivot.y, this.angle * Math.PI / 180, this.scale);
            return this._uMatrix;
        }
    });
    Object.defineProperty(CanvasDisplayObject.prototype, "getLocalBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (!this._localBounds) {
                var bn = 10000000;
                this._localBounds = {
                    left: bn,
                    top: bn,
                    right: -bn,
                    bottom: -bn
                };
                this._addBounds(this._localBounds);
            }
            return this._localBounds;
        }
    });
    Object.defineProperty(CanvasDisplayObject.prototype, "getAdjustedBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (bounds) {
            this._setMatrix();
            var matrix = this.getLocalMatrix();
            var p0 = matrix.apply({ x: bounds.left, y: bounds.top });
            var p1 = matrix.apply({ x: bounds.right, y: bounds.top });
            var p2 = matrix.apply({ x: bounds.right, y: bounds.bottom });
            var p3 = matrix.apply({ x: bounds.left, y: bounds.bottom });
            return {
                left: Math.min(p0.x, p1.x, p2.x, p3.x),
                top: Math.min(p0.y, p1.y, p2.y, p3.y),
                right: Math.max(p0.x, p1.x, p2.x, p3.x),
                bottom: Math.max(p0.y, p1.y, p2.y, p3.y)
            };
        }
    });
    Object.defineProperty(CanvasDisplayObject.prototype, "on", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, callback, context) {
            if (this.interactive) {
                return this._renderer._addEvent(this, key, callback, context);
            }
            else {
                return new Disposer(function () { });
            }
        }
    });
    Object.defineProperty(CanvasDisplayObject.prototype, "_setMatrix", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            // TODO only calculate this if it has actually changed
            this._localMatrix.setTransform(this.x, this.y, this.pivot.x, this.pivot.y, 
            // Converts degrees to radians
            this.angle * Math.PI / 180, this.scale);
            this._matrix.copyFrom(this._localMatrix);
            if (this._parent) {
                // TODO only calculate this if it has actually changed
                this._matrix.prepend(this._parent._matrix);
            }
        }
    });
    Object.defineProperty(CanvasDisplayObject.prototype, "_transform", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (context, resolution) {
            var m = this._matrix;
            context.setTransform(m.a * resolution, m.b * resolution, m.c * resolution, m.d * resolution, m.tx * resolution, m.ty * resolution);
        }
    });
    Object.defineProperty(CanvasDisplayObject.prototype, "render", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (parentLayer) {
            var _this = this;
            if (this.visible && (this.exportable !== false || !this._renderer._omitTainted)) {
                this._setMatrix();
                var resolution_1 = this._renderer.resolution;
                var layers = this._renderer.layers;
                var ghostContext = this._renderer._ghostContext;
                var mask_1 = this.mask;
                if (mask_1) {
                    mask_1._setMatrix();
                }
                // TODO improve this
                $array.each(layers, function (layer) {
                    if (layer) {
                        var context = layer.context;
                        context.save();
                        // We must apply the mask before we transform the element
                        if (mask_1) {
                            mask_1._transform(context, layer.scale || resolution_1);
                            mask_1._runPath(context);
                            context.clip();
                        }
                        context.globalAlpha = _this.compoundAlpha * _this.alpha;
                        _this._transform(context, layer.scale || resolution_1);
                        if (_this.filter) {
                            context.filter = _this.filter;
                        }
                    }
                });
                ghostContext.save();
                // We must apply the mask before we transform the element
                if (mask_1 && this._isInteractiveMask()) {
                    mask_1._transform(ghostContext, resolution_1);
                    mask_1._runPath(ghostContext);
                    ghostContext.clip();
                }
                this._transform(ghostContext, resolution_1);
                this._render(parentLayer);
                ghostContext.restore();
                $array.each(layers, function (layer) {
                    if (layer) {
                        layer.context.restore();
                    }
                });
            }
        }
    });
    Object.defineProperty(CanvasDisplayObject.prototype, "_render", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (parentLayer) {
            if (this.exportable === false) {
                var layer = this._layer || parentLayer;
                layer.tainted = true;
            }
        }
    });
    Object.defineProperty(CanvasDisplayObject.prototype, "hovering", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this._renderer._hovering.has(this);
        }
    });
    Object.defineProperty(CanvasDisplayObject.prototype, "dragging", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            return this._renderer._dragging.some(function (x) { return x.value === _this; });
        }
    });
    Object.defineProperty(CanvasDisplayObject.prototype, "dispose", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this.getLayer().dirty = true;
        }
    });
    return CanvasDisplayObject;
}(DisposerClass));
export { CanvasDisplayObject };
/**
 * @ignore
 */
var CanvasContainer = /** @class */ (function (_super) {
    __extends(CanvasContainer, _super);
    function CanvasContainer() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "interactiveChildren", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: true
        });
        Object.defineProperty(_this, "_childLayers", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_children", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        return _this;
    }
    Object.defineProperty(CanvasContainer.prototype, "_isInteractiveMask", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return this.interactiveChildren || _super.prototype._isInteractiveMask.call(this);
        }
    });
    Object.defineProperty(CanvasContainer.prototype, "addChild", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (child) {
            child._parent = this;
            this._children.push(child);
            if (child._layer) {
                this.registerChildLayer(child._layer);
            }
        }
    });
    Object.defineProperty(CanvasContainer.prototype, "addChildAt", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (child, index) {
            child._parent = this;
            this._children.splice(index, 0, child);
            if (child._layer) {
                this.registerChildLayer(child._layer);
            }
        }
    });
    Object.defineProperty(CanvasContainer.prototype, "removeChild", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (child) {
            child._parent = undefined;
            $array.removeFirst(this._children, child);
        }
    });
    Object.defineProperty(CanvasContainer.prototype, "_render", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (parentLayer) {
            var _this = this;
            _super.prototype._render.call(this, parentLayer);
            var renderer = this._renderer;
            if (this.interactive && this.interactiveChildren) {
                ++renderer._forceInteractive;
            }
            var layer = this._layer || parentLayer;
            $array.each(this._children, function (child) {
                child.compoundAlpha = _this.compoundAlpha * _this.alpha;
                child.render(layer);
            });
            if (this.interactive && this.interactiveChildren) {
                --renderer._forceInteractive;
            }
        }
    });
    Object.defineProperty(CanvasContainer.prototype, "registerChildLayer", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (layer) {
            if (!this._childLayers) {
                this._childLayers = [];
            }
            $array.pushOne(this._childLayers, layer);
            if (this._parent) {
                this._parent.registerChildLayer(layer);
            }
        }
    });
    Object.defineProperty(CanvasContainer.prototype, "markDirtyLayer", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (deep) {
            if (deep === void 0) { deep = false; }
            _super.prototype.markDirtyLayer.call(this);
            if (deep && this._childLayers) {
                $array.each(this._childLayers, function (layer) { return layer.dirty = true; });
            }
        }
    });
    Object.defineProperty(CanvasContainer.prototype, "dispose", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype.dispose.call(this);
            if (this._childLayers) {
                $array.each(this._childLayers, function (layer) {
                    layer.dirty = true;
                });
            }
        }
    });
    return CanvasContainer;
}(CanvasDisplayObject));
export { CanvasContainer };
/**
 * @ignore
 */
function setPoint(bounds, point) {
    bounds.left = Math.min(bounds.left, point.x);
    bounds.top = Math.min(bounds.top, point.y);
    bounds.right = Math.max(bounds.right, point.x);
    bounds.bottom = Math.max(bounds.bottom, point.y);
}
/**
 * @ignore
 */
var Op = /** @class */ (function () {
    function Op() {
    }
    Object.defineProperty(Op.prototype, "colorize", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_context, _forceColor) { }
    });
    Object.defineProperty(Op.prototype, "path", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_context) { }
    });
    Object.defineProperty(Op.prototype, "addBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_bounds) { }
    });
    return Op;
}());
/**
 * @ignore
 */
var BeginFill = /** @class */ (function (_super) {
    __extends(BeginFill, _super);
    function BeginFill(color) {
        var _this = _super.call(this) || this;
        Object.defineProperty(_this, "color", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: color
        });
        return _this;
    }
    Object.defineProperty(BeginFill.prototype, "colorize", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (context, forceColor) {
            if (forceColor !== undefined) {
                context.fillStyle = forceColor;
            }
            else {
                context.fillStyle = this.color;
            }
        }
    });
    return BeginFill;
}(Op));
/**
 * @ignore
 */
var EndFill = /** @class */ (function (_super) {
    __extends(EndFill, _super);
    function EndFill(clearShadow) {
        var _this = _super.call(this) || this;
        Object.defineProperty(_this, "clearShadow", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: clearShadow
        });
        return _this;
    }
    Object.defineProperty(EndFill.prototype, "colorize", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (context, _forceColor) {
            context.fill();
            if (this.clearShadow) {
                context.shadowColor = "";
                context.shadowBlur = 0;
                context.shadowOffsetX = 0;
                context.shadowOffsetY = 0;
            }
        }
    });
    return EndFill;
}(Op));
/**
 * @ignore
 */
var EndStroke = /** @class */ (function (_super) {
    __extends(EndStroke, _super);
    function EndStroke() {
        return _super !== null && _super.apply(this, arguments) || this;
    }
    Object.defineProperty(EndStroke.prototype, "colorize", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (context, _forceColor) {
            context.stroke();
        }
    });
    return EndStroke;
}(Op));
/**
 * @ignore
 */
var LineStyle = /** @class */ (function (_super) {
    __extends(LineStyle, _super);
    function LineStyle(width, color) {
        var _this = _super.call(this) || this;
        Object.defineProperty(_this, "width", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: width
        });
        Object.defineProperty(_this, "color", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: color
        });
        return _this;
    }
    Object.defineProperty(LineStyle.prototype, "colorize", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (context, forceColor) {
            if (forceColor !== undefined) {
                context.strokeStyle = forceColor;
            }
            else {
                context.strokeStyle = this.color;
            }
            context.lineWidth = this.width;
        }
    });
    return LineStyle;
}(Op));
/**
 * @ignore
 */
var LineDash = /** @class */ (function (_super) {
    __extends(LineDash, _super);
    function LineDash(dash) {
        var _this = _super.call(this) || this;
        Object.defineProperty(_this, "dash", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: dash
        });
        return _this;
    }
    Object.defineProperty(LineDash.prototype, "colorize", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (context, _forceColor) {
            context.setLineDash(this.dash);
        }
    });
    return LineDash;
}(Op));
/**
 * @ignore
 */
var LineDashOffset = /** @class */ (function (_super) {
    __extends(LineDashOffset, _super);
    function LineDashOffset(dashOffset) {
        var _this = _super.call(this) || this;
        Object.defineProperty(_this, "dashOffset", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: dashOffset
        });
        return _this;
    }
    Object.defineProperty(LineDashOffset.prototype, "colorize", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (context, _forceColor) {
            context.lineDashOffset = this.dashOffset;
        }
    });
    return LineDashOffset;
}(Op));
/**
 * @ignore
 */
var DrawRect = /** @class */ (function (_super) {
    __extends(DrawRect, _super);
    function DrawRect(x, y, width, height) {
        var _this = _super.call(this) || this;
        Object.defineProperty(_this, "x", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: x
        });
        Object.defineProperty(_this, "y", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: y
        });
        Object.defineProperty(_this, "width", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: width
        });
        Object.defineProperty(_this, "height", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: height
        });
        return _this;
    }
    Object.defineProperty(DrawRect.prototype, "path", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (context) {
            context.rect(this.x, this.y, this.width, this.height);
        }
    });
    Object.defineProperty(DrawRect.prototype, "addBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (bounds) {
            var l = this.x;
            var t = this.y;
            var r = l + this.width;
            var b = t + this.height;
            setPoint(bounds, { x: l, y: t });
            setPoint(bounds, { x: r, y: t });
            setPoint(bounds, { x: l, y: b });
            setPoint(bounds, { x: r, y: b });
        }
    });
    return DrawRect;
}(Op));
/**
 * @ignore
 */
var DrawCircle = /** @class */ (function (_super) {
    __extends(DrawCircle, _super);
    function DrawCircle(x, y, radius) {
        var _this = _super.call(this) || this;
        Object.defineProperty(_this, "x", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: x
        });
        Object.defineProperty(_this, "y", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: y
        });
        Object.defineProperty(_this, "radius", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: radius
        });
        return _this;
    }
    Object.defineProperty(DrawCircle.prototype, "path", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (context) {
            context.moveTo(this.x + this.radius, this.y);
            context.arc(this.x, this.y, this.radius, 0, 2 * Math.PI);
        }
    });
    // TODO handle skewing and rotation
    Object.defineProperty(DrawCircle.prototype, "addBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (bounds) {
            setPoint(bounds, { x: this.x - this.radius, y: this.y - this.radius });
            setPoint(bounds, { x: this.x + this.radius, y: this.y + this.radius });
        }
    });
    return DrawCircle;
}(Op));
/**
 * @ignore
 */
var DrawEllipse = /** @class */ (function (_super) {
    __extends(DrawEllipse, _super);
    function DrawEllipse(x, y, radiusX, radiusY) {
        var _this = _super.call(this) || this;
        Object.defineProperty(_this, "x", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: x
        });
        Object.defineProperty(_this, "y", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: y
        });
        Object.defineProperty(_this, "radiusX", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: radiusX
        });
        Object.defineProperty(_this, "radiusY", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: radiusY
        });
        return _this;
    }
    Object.defineProperty(DrawEllipse.prototype, "path", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (context) {
            context.ellipse(0, 0, this.radiusX, this.radiusY, 0, 0, Math.PI * 2);
        }
    });
    // TODO handle skewing and rotation
    Object.defineProperty(DrawEllipse.prototype, "addBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (bounds) {
            setPoint(bounds, { x: this.x - this.radiusX, y: this.y - this.radiusY });
            setPoint(bounds, { x: this.x + this.radiusX, y: this.y + this.radiusY });
        }
    });
    return DrawEllipse;
}(Op));
/**
 * @ignore
 */
var Arc = /** @class */ (function (_super) {
    __extends(Arc, _super);
    function Arc(cx, cy, radius, startAngle, endAngle, anticlockwise) {
        var _this = _super.call(this) || this;
        Object.defineProperty(_this, "cx", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: cx
        });
        Object.defineProperty(_this, "cy", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: cy
        });
        Object.defineProperty(_this, "radius", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: radius
        });
        Object.defineProperty(_this, "startAngle", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: startAngle
        });
        Object.defineProperty(_this, "endAngle", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: endAngle
        });
        Object.defineProperty(_this, "anticlockwise", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: anticlockwise
        });
        return _this;
    }
    Object.defineProperty(Arc.prototype, "path", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (context) {
            if (this.radius > 0) {
                context.arc(this.cx, this.cy, this.radius, this.startAngle, this.endAngle, this.anticlockwise);
            }
        }
    });
    Object.defineProperty(Arc.prototype, "addBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (bounds) {
            var arcBounds = $math.getArcBounds(this.cx, this.cy, this.startAngle * $math.DEGREES, this.endAngle * $math.DEGREES, this.radius);
            setPoint(bounds, { x: arcBounds.left, y: arcBounds.top });
            setPoint(bounds, { x: arcBounds.right, y: arcBounds.bottom });
        }
    });
    return Arc;
}(Op));
/**
 * @ignore
 */
var ArcTo = /** @class */ (function (_super) {
    __extends(ArcTo, _super);
    function ArcTo(x1, y1, x2, y2, radius) {
        var _this = _super.call(this) || this;
        Object.defineProperty(_this, "x1", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: x1
        });
        Object.defineProperty(_this, "y1", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: y1
        });
        Object.defineProperty(_this, "x2", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: x2
        });
        Object.defineProperty(_this, "y2", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: y2
        });
        Object.defineProperty(_this, "radius", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: radius
        });
        return _this;
    }
    Object.defineProperty(ArcTo.prototype, "path", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (context) {
            if (this.radius > 0) {
                context.arcTo(this.x1, this.y1, this.x2, this.y2, this.radius);
            }
        }
    });
    // TODO: add points
    Object.defineProperty(ArcTo.prototype, "addBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (_bounds) {
            /*
            // not finished
            https://math.stackexchange.com/questions/1781438/finding-the-center-of-a-circle-given-two-points-and-a-radius-algebraically
    
            if (prevPoint) {
                let x1 = prevPoint.x;
                let y1 = prevPoint.y;
                let x2 = this.x2;
                let y2 = this.y2;
                let r = this.radius;
    
                let xa = (x2 - x1) / 2;
                let ya = (y2 - y1) / 2;
    
                let x0 = x1 + xa;
                let y0 = y1 + ya;
    
                let a = Math.hypot(xa, ya);
                let b = Math.sqrt(r * r - a * a);
    
                let cx = x0 + b * ya / a;
                let cy = y0 - b * xa / a;
    
                console.log(cx, cy);
            }*/
        }
    });
    return ArcTo;
}(Op));
/**
 * @ignore
 */
var LineTo = /** @class */ (function (_super) {
    __extends(LineTo, _super);
    function LineTo(x, y) {
        var _this = _super.call(this) || this;
        Object.defineProperty(_this, "x", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: x
        });
        Object.defineProperty(_this, "y", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: y
        });
        return _this;
    }
    Object.defineProperty(LineTo.prototype, "path", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (context) {
            context.lineTo(this.x, this.y);
        }
    });
    Object.defineProperty(LineTo.prototype, "addBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (bounds) {
            setPoint(bounds, { x: this.x, y: this.y });
        }
    });
    return LineTo;
}(Op));
/**
 * @ignore
 */
var MoveTo = /** @class */ (function (_super) {
    __extends(MoveTo, _super);
    function MoveTo(x, y) {
        var _this = _super.call(this) || this;
        Object.defineProperty(_this, "x", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: x
        });
        Object.defineProperty(_this, "y", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: y
        });
        return _this;
    }
    Object.defineProperty(MoveTo.prototype, "path", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (context) {
            context.moveTo(this.x, this.y);
        }
    });
    Object.defineProperty(MoveTo.prototype, "addBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (bounds) {
            setPoint(bounds, { x: this.x, y: this.y });
        }
    });
    return MoveTo;
}(Op));
/**
 * @ignore
 */
var ClosePath = /** @class */ (function (_super) {
    __extends(ClosePath, _super);
    function ClosePath() {
        return _super !== null && _super.apply(this, arguments) || this;
    }
    Object.defineProperty(ClosePath.prototype, "path", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (context) {
            context.closePath();
        }
    });
    return ClosePath;
}(Op));
/**
 * @ignore
 */
var BezierCurveTo = /** @class */ (function (_super) {
    __extends(BezierCurveTo, _super);
    function BezierCurveTo(cpX, cpY, cpX2, cpY2, toX, toY) {
        var _this = _super.call(this) || this;
        Object.defineProperty(_this, "cpX", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: cpX
        });
        Object.defineProperty(_this, "cpY", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: cpY
        });
        Object.defineProperty(_this, "cpX2", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: cpX2
        });
        Object.defineProperty(_this, "cpY2", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: cpY2
        });
        Object.defineProperty(_this, "toX", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: toX
        });
        Object.defineProperty(_this, "toY", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: toY
        });
        return _this;
    }
    Object.defineProperty(BezierCurveTo.prototype, "path", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (context) {
            context.bezierCurveTo(this.cpX, this.cpY, this.cpX2, this.cpY2, this.toX, this.toY);
        }
    });
    // TODO: OK?
    Object.defineProperty(BezierCurveTo.prototype, "addBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (bounds) {
            setPoint(bounds, { x: this.cpX, y: this.cpY });
            setPoint(bounds, { x: this.cpX2, y: this.cpY2 });
            setPoint(bounds, { x: this.toX, y: this.toY });
        }
    });
    return BezierCurveTo;
}(Op));
/**
 * @ignore
 */
var QuadraticCurveTo = /** @class */ (function (_super) {
    __extends(QuadraticCurveTo, _super);
    function QuadraticCurveTo(cpX, cpY, toX, toY) {
        var _this = _super.call(this) || this;
        Object.defineProperty(_this, "cpX", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: cpX
        });
        Object.defineProperty(_this, "cpY", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: cpY
        });
        Object.defineProperty(_this, "toX", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: toX
        });
        Object.defineProperty(_this, "toY", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: toY
        });
        return _this;
    }
    Object.defineProperty(QuadraticCurveTo.prototype, "path", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (context) {
            context.quadraticCurveTo(this.cpX, this.cpY, this.toX, this.toY);
        }
    });
    // TODO: OK?
    Object.defineProperty(QuadraticCurveTo.prototype, "addBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (bounds) {
            setPoint(bounds, { x: this.cpX, y: this.cpY });
            setPoint(bounds, { x: this.toX, y: this.toY });
        }
    });
    return QuadraticCurveTo;
}(Op));
/**
 * @ignore
 */
var Shadow = /** @class */ (function (_super) {
    __extends(Shadow, _super);
    function Shadow(color, blur, offsetX, offsetY, opacity) {
        var _this = _super.call(this) || this;
        Object.defineProperty(_this, "color", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: color
        });
        Object.defineProperty(_this, "blur", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: blur
        });
        Object.defineProperty(_this, "offsetX", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: offsetX
        });
        Object.defineProperty(_this, "offsetY", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: offsetY
        });
        Object.defineProperty(_this, "opacity", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: opacity
        });
        return _this;
    }
    Object.defineProperty(Shadow.prototype, "colorize", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (context, _forceColor) {
            if (this.opacity) {
                context.fillStyle = this.color;
            }
            context.shadowColor = this.color;
            if (this.blur) {
                context.shadowBlur = this.blur;
            }
            if (this.offsetX) {
                context.shadowOffsetX = this.offsetX;
            }
            if (this.offsetY) {
                context.shadowOffsetY = this.offsetY;
            }
        }
    });
    return Shadow;
}(Op));
/**
 * @ignore
 */
var CanvasGraphics = /** @class */ (function (_super) {
    __extends(CanvasGraphics, _super);
    function CanvasGraphics() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "_operations", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        Object.defineProperty(_this, "blendMode", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: BlendMode.NORMAL
        });
        Object.defineProperty(_this, "_hasShadows", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(_this, "_fillAlpha", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_strokeAlpha", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        return _this;
    }
    Object.defineProperty(CanvasGraphics.prototype, "clear", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype.clear.call(this);
            this._operations.length = 0;
        }
    });
    Object.defineProperty(CanvasGraphics.prototype, "_pushOp", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (op) {
            this._operations.push(op);
        }
    });
    Object.defineProperty(CanvasGraphics.prototype, "beginFill", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (color, alpha) {
            if (alpha === void 0) { alpha = 1; }
            this._fillAlpha = alpha;
            if (color) {
                if (color instanceof Color) {
                    this._pushOp(new BeginFill(color.toCSS(alpha)));
                }
                else {
                    this.isMeasured = true;
                    this._pushOp(new BeginFill(color));
                }
            }
            else {
                this._pushOp(new BeginFill("rgba(0, 0, 0, " + alpha + ")"));
            }
        }
    });
    Object.defineProperty(CanvasGraphics.prototype, "endFill", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._pushOp(new EndFill(this._hasShadows));
        }
    });
    Object.defineProperty(CanvasGraphics.prototype, "endStroke", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._pushOp(new EndStroke());
        }
    });
    Object.defineProperty(CanvasGraphics.prototype, "lineStyle", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (width, color, alpha) {
            if (width === void 0) { width = 0; }
            if (alpha === void 0) { alpha = 1; }
            this._strokeAlpha = alpha;
            if (color) {
                if (color instanceof Color) {
                    this._pushOp(new LineStyle(width, color.toCSS(alpha)));
                }
                else {
                    this._pushOp(new LineStyle(width, color));
                }
            }
            else {
                this._pushOp(new LineStyle(width, "rgba(0, 0, 0, " + alpha + ")"));
            }
        }
    });
    Object.defineProperty(CanvasGraphics.prototype, "setLineDash", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dash) {
            this._pushOp(new LineDash(dash ? dash : []));
        }
    });
    Object.defineProperty(CanvasGraphics.prototype, "setLineDashOffset", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (dashOffset) {
            this._pushOp(new LineDashOffset(dashOffset || 0));
        }
    });
    Object.defineProperty(CanvasGraphics.prototype, "drawRect", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (x, y, width, height) {
            this._pushOp(new DrawRect(x, y, width, height));
        }
    });
    Object.defineProperty(CanvasGraphics.prototype, "drawCircle", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (x, y, radius) {
            this._pushOp(new DrawCircle(x, y, radius));
        }
    });
    Object.defineProperty(CanvasGraphics.prototype, "drawEllipse", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (x, y, radiusX, radiusY) {
            this._pushOp(new DrawEllipse(x, y, radiusX, radiusY));
        }
    });
    Object.defineProperty(CanvasGraphics.prototype, "arc", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (cx, cy, radius, startAngle, endAngle, anticlockwise) {
            if (anticlockwise === void 0) { anticlockwise = false; }
            this._pushOp(new Arc(cx, cy, radius, startAngle, endAngle, anticlockwise));
        }
    });
    Object.defineProperty(CanvasGraphics.prototype, "arcTo", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (x1, y1, x2, y2, radius) {
            this._pushOp(new ArcTo(x1, y1, x2, y2, radius));
        }
    });
    Object.defineProperty(CanvasGraphics.prototype, "lineTo", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (x, y) {
            this._pushOp(new LineTo(x, y));
        }
    });
    Object.defineProperty(CanvasGraphics.prototype, "moveTo", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (x, y) {
            this._pushOp(new MoveTo(x, y));
        }
    });
    Object.defineProperty(CanvasGraphics.prototype, "bezierCurveTo", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (cpX, cpY, cpX2, cpY2, toX, toY) {
            this._pushOp(new BezierCurveTo(cpX, cpY, cpX2, cpY2, toX, toY));
        }
    });
    Object.defineProperty(CanvasGraphics.prototype, "quadraticCurveTo", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (cpX, cpY, toX, toY) {
            this._pushOp(new QuadraticCurveTo(cpX, cpY, toX, toY));
        }
    });
    Object.defineProperty(CanvasGraphics.prototype, "closePath", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this._pushOp(new ClosePath());
        }
    });
    Object.defineProperty(CanvasGraphics.prototype, "shadow", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (color, blur, offsetX, offsetY, opacity) {
            this._hasShadows = true;
            this._pushOp(new Shadow(opacity ? color.toCSS(opacity) : color.toCSS(this._fillAlpha || this._strokeAlpha), blur, offsetX, offsetY));
        }
    });
    // https://svgwg.org/svg2-draft/paths.html#DProperty
    // TODO better error checking
    Object.defineProperty(CanvasGraphics.prototype, "svgPath", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (path) {
            var _this = this;
            var x = 0;
            var y = 0;
            var cpx = null;
            var cpy = null;
            var qcpx = null;
            var qcpy = null;
            var SEGMENTS_REGEXP = /([MmZzLlHhVvCcSsQqTtAa])([^MmZzLlHhVvCcSsQqTtAa]*)/g;
            var ARGS_REGEXP = /[\u0009\u0020\u000A\u000C\u000D]*([\+\-]?[0-9]*\.?[0-9]+(?:[eE][\+\-]?[0-9]+)?)[\u0009\u0020\u000A\u000C\u000D]*,?/g;
            var match;
            while ((match = SEGMENTS_REGEXP.exec(path)) !== null) {
                var name_1 = match[1];
                var rest = match[2];
                var args = [];
                while ((match = ARGS_REGEXP.exec(rest)) !== null) {
                    args.push(+match[1]);
                }
                // Reset control point
                if (name_1 !== "S" && name_1 !== "s" && name_1 !== "C" && name_1 !== "c") {
                    cpx = null;
                    cpy = null;
                }
                // Reset control point
                if (name_1 !== "Q" && name_1 !== "q" && name_1 !== "T" && name_1 !== "t") {
                    qcpx = null;
                    qcpy = null;
                }
                switch (name_1) {
                    case "M":
                        checkEvenArgs(name_1, args.length, 2);
                        x = args[0];
                        y = args[1];
                        this.moveTo(x, y);
                        for (var i = 2; i < args.length; i += 2) {
                            x = args[i];
                            y = args[i + 1];
                            this.lineTo(x, y);
                        }
                        break;
                    case "m":
                        checkEvenArgs(name_1, args.length, 2);
                        x += args[0];
                        y += args[1];
                        this.moveTo(x, y);
                        for (var i = 2; i < args.length; i += 2) {
                            x += args[i];
                            y += args[i + 1];
                            this.lineTo(x, y);
                        }
                        break;
                    case "L":
                        checkEvenArgs(name_1, args.length, 2);
                        for (var i = 0; i < args.length; i += 2) {
                            x = args[i];
                            y = args[i + 1];
                            this.lineTo(x, y);
                        }
                        break;
                    case "l":
                        checkEvenArgs(name_1, args.length, 2);
                        for (var i = 0; i < args.length; i += 2) {
                            x += args[i];
                            y += args[i + 1];
                            this.lineTo(x, y);
                        }
                        break;
                    case "H":
                        checkMinArgs(name_1, args.length, 1);
                        for (var i = 0; i < args.length; ++i) {
                            x = args[i];
                            this.lineTo(x, y);
                        }
                        break;
                    case "h":
                        checkMinArgs(name_1, args.length, 1);
                        for (var i = 0; i < args.length; ++i) {
                            x += args[i];
                            this.lineTo(x, y);
                        }
                        break;
                    case "V":
                        checkMinArgs(name_1, args.length, 1);
                        for (var i = 0; i < args.length; ++i) {
                            y = args[i];
                            this.lineTo(x, y);
                        }
                        break;
                    case "v":
                        checkMinArgs(name_1, args.length, 1);
                        for (var i = 0; i < args.length; ++i) {
                            y += args[i];
                            this.lineTo(x, y);
                        }
                        break;
                    case "C":
                        checkEvenArgs(name_1, args.length, 6);
                        for (var i = 0; i < args.length; i += 6) {
                            var x1 = args[i];
                            var y1 = args[i + 1];
                            cpx = args[i + 2];
                            cpy = args[i + 3];
                            x = args[i + 4];
                            y = args[i + 5];
                            this.bezierCurveTo(x1, y1, cpx, cpy, x, y);
                        }
                        break;
                    case "c":
                        checkEvenArgs(name_1, args.length, 6);
                        for (var i = 0; i < args.length; i += 6) {
                            var x1 = args[i] + x;
                            var y1 = args[i + 1] + y;
                            cpx = args[i + 2] + x;
                            cpy = args[i + 3] + y;
                            x += args[i + 4];
                            y += args[i + 5];
                            this.bezierCurveTo(x1, y1, cpx, cpy, x, y);
                        }
                        break;
                    case "S":
                        checkEvenArgs(name_1, args.length, 4);
                        if (cpx === null || cpy === null) {
                            cpx = x;
                            cpy = y;
                        }
                        for (var i = 0; i < args.length; i += 4) {
                            var x1 = 2 * x - cpx;
                            var y1 = 2 * y - cpy;
                            cpx = args[i];
                            cpy = args[i + 1];
                            x = args[i + 2];
                            y = args[i + 3];
                            this.bezierCurveTo(x1, y1, cpx, cpy, x, y);
                        }
                        break;
                    case "s":
                        checkEvenArgs(name_1, args.length, 4);
                        if (cpx === null || cpy === null) {
                            cpx = x;
                            cpy = y;
                        }
                        for (var i = 0; i < args.length; i += 4) {
                            var x1 = 2 * x - cpx;
                            var y1 = 2 * y - cpy;
                            cpx = args[i] + x;
                            cpy = args[i + 1] + y;
                            x += args[i + 2];
                            y += args[i + 3];
                            this.bezierCurveTo(x1, y1, cpx, cpy, x, y);
                        }
                        break;
                    case "Q":
                        checkEvenArgs(name_1, args.length, 4);
                        for (var i = 0; i < args.length; i += 4) {
                            qcpx = args[i];
                            qcpy = args[i + 1];
                            x = args[i + 2];
                            y = args[i + 3];
                            this.quadraticCurveTo(qcpx, qcpy, x, y);
                        }
                        break;
                    case "q":
                        checkEvenArgs(name_1, args.length, 4);
                        for (var i = 0; i < args.length; i += 4) {
                            qcpx = args[i] + x;
                            qcpy = args[i + 1] + y;
                            x += args[i + 2];
                            y += args[i + 3];
                            this.quadraticCurveTo(qcpx, qcpy, x, y);
                        }
                        break;
                    case "T":
                        checkEvenArgs(name_1, args.length, 2);
                        if (qcpx === null || qcpy === null) {
                            qcpx = x;
                            qcpy = y;
                        }
                        for (var i = 0; i < args.length; i += 2) {
                            qcpx = 2 * x - qcpx;
                            qcpy = 2 * y - qcpy;
                            x = args[i];
                            y = args[i + 1];
                            this.quadraticCurveTo(qcpx, qcpy, x, y);
                        }
                        break;
                    case "t":
                        checkEvenArgs(name_1, args.length, 2);
                        if (qcpx === null || qcpy === null) {
                            qcpx = x;
                            qcpy = y;
                        }
                        for (var i = 0; i < args.length; i += 2) {
                            qcpx = 2 * x - qcpx;
                            qcpy = 2 * y - qcpy;
                            x += args[i];
                            y += args[i + 1];
                            this.quadraticCurveTo(qcpx, qcpy, x, y);
                        }
                        break;
                    case "A":
                    case "a":
                        var relative = (name_1 === "a");
                        checkEvenArgs(name_1, args.length, 7);
                        for (var i = 0; i < args.length; i += 7) {
                            var cx = args[i + 5];
                            var cy = args[i + 6];
                            if (relative) {
                                cx += x;
                                cy += y;
                            }
                            var bs = arcToBezier({
                                px: x,
                                py: y,
                                rx: args[i],
                                ry: args[i + 1],
                                xAxisRotation: args[i + 2],
                                largeArcFlag: assertBinary(args[i + 3]),
                                sweepFlag: assertBinary(args[i + 4]),
                                cx: cx,
                                cy: cy,
                            });
                            $array.each(bs, function (b) {
                                _this.bezierCurveTo(b.x1, b.y1, b.x2, b.y2, b.x, b.y);
                                x = b.x;
                                y = b.y;
                            });
                        }
                        break;
                    case "Z":
                    case "z":
                        checkArgs(name_1, args.length, 0);
                        this.closePath();
                        break;
                }
            }
        }
    });
    Object.defineProperty(CanvasGraphics.prototype, "_runPath", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (context) {
            context.beginPath();
            $array.each(this._operations, function (op) {
                op.path(context);
            });
        }
    });
    Object.defineProperty(CanvasGraphics.prototype, "_render", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (parentLayer) {
            _super.prototype._render.call(this, parentLayer);
            var layer = this._layer || parentLayer;
            var layerDirty = layer.dirty;
            var interactive = this._isInteractive();
            if (layerDirty || interactive) {
                var context_1 = layer.context;
                var ghostContext_1 = this._renderer._ghostContext;
                if (layerDirty) {
                    context_1.globalCompositeOperation = this.blendMode;
                    context_1.beginPath();
                }
                var color_1;
                if (interactive) {
                    ghostContext_1.beginPath();
                    color_1 = this._getColorId();
                }
                $array.each(this._operations, function (op) {
                    if (layerDirty) {
                        op.path(context_1);
                        op.colorize(context_1, undefined);
                    }
                    if (interactive) {
                        op.path(ghostContext_1);
                        op.colorize(ghostContext_1, color_1);
                    }
                });
            }
        }
    });
    Object.defineProperty(CanvasGraphics.prototype, "renderDetached", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (context) {
            if (this.visible) {
                this._setMatrix();
                context.save();
                // We must apply the mask before we transform the element
                var mask = this.mask;
                if (mask) {
                    mask._setMatrix();
                    mask._transform(context, 1);
                    mask._runPath(context);
                    context.clip();
                }
                // TODO handle compoundAlpha somehow ?
                context.globalAlpha = this.compoundAlpha * this.alpha;
                this._transform(context, 1);
                if (this.filter) {
                    context.filter = this.filter;
                }
                context.globalCompositeOperation = this.blendMode;
                context.beginPath();
                $array.each(this._operations, function (op) {
                    op.path(context);
                    op.colorize(context, undefined);
                });
                context.restore();
            }
        }
    });
    Object.defineProperty(CanvasGraphics.prototype, "_addBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (bounds) {
            if (this.visible && this.isMeasured) {
                $array.each(this._operations, function (op) {
                    op.addBounds(bounds);
                });
            }
        }
    });
    return CanvasGraphics;
}(CanvasDisplayObject));
export { CanvasGraphics };
/**
 * @ignore
 */
var CanvasText = /** @class */ (function (_super) {
    __extends(CanvasText, _super);
    function CanvasText(renderer, text, style) {
        var _this = _super.call(this, renderer) || this;
        Object.defineProperty(_this, "text", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "style", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "resolution", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 1
        });
        Object.defineProperty(_this, "_textInfo", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_textVisible", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: true
        });
        Object.defineProperty(_this, "_originalScale", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 1
        });
        _this.text = text;
        _this.style = style;
        return _this;
    }
    Object.defineProperty(CanvasText.prototype, "invalidateBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype.invalidateBounds.call(this);
            this._textInfo = undefined;
        }
    });
    Object.defineProperty(CanvasText.prototype, "_shared", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (context) {
            if (this.style.textAlign) {
                context.textAlign = this.style.textAlign;
            }
            if (this.style.direction) {
                context.direction = this.style.direction;
            }
            if (this.style.textBaseline) {
                context.textBaseline = this.style.textBaseline;
            }
        }
    });
    Object.defineProperty(CanvasText.prototype, "_prerender", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (layer, ignoreGhost, ignoreFontWeight) {
            if (ignoreGhost === void 0) { ignoreGhost = false; }
            if (ignoreFontWeight === void 0) { ignoreFontWeight = false; }
            _super.prototype._render.call(this, layer);
            var context = layer.context;
            var ghostContext = this._renderer._ghostContext;
            // Font style
            var style = this.style;
            var fontStyle = this._getFontStyle(undefined, ignoreFontWeight);
            context.font = fontStyle;
            if (this._isInteractive() && !ignoreGhost) {
                ghostContext.font = fontStyle;
            }
            // Other parameters
            if (style.fill) {
                if (style.fill instanceof Color) {
                    context.fillStyle = style.fill.toCSS();
                }
                else {
                    context.fillStyle = style.fill;
                }
            }
            if (style.shadowColor) {
                layer.context.shadowColor = style.shadowColor.toCSS(style.shadowOpacity || 1);
            }
            if (style.shadowBlur) {
                layer.context.shadowBlur = style.shadowBlur;
            }
            if (style.shadowOffsetX) {
                layer.context.shadowOffsetX = style.shadowOffsetX;
            }
            if (style.shadowOffsetY) {
                layer.context.shadowOffsetY = style.shadowOffsetY;
            }
            this._shared(context);
            if (this._isInteractive() && !ignoreGhost) {
                ghostContext.fillStyle = this._getColorId();
                this._shared(ghostContext);
            }
        }
    });
    Object.defineProperty(CanvasText.prototype, "_getFontStyle", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (style2, ignoreFontWeight) {
            if (ignoreFontWeight === void 0) { ignoreFontWeight = false; }
            // Process defaults
            var style = this.style;
            var fontStyle = [];
            if (style2 && style2.fontVariant) {
                fontStyle.push(style2.fontVariant);
            }
            else if (style.fontVariant) {
                fontStyle.push(style.fontVariant);
            }
            if (!ignoreFontWeight) {
                if (style2 && style2.fontWeight) {
                    fontStyle.push(style2.fontWeight);
                }
                else if (style.fontWeight) {
                    fontStyle.push(style.fontWeight);
                }
            }
            if (style2 && style2.fontStyle) {
                fontStyle.push(style2.fontStyle);
            }
            else if (style.fontStyle) {
                fontStyle.push(style.fontStyle);
            }
            if (style2 && style2.fontSize) {
                if ($type.isNumber(style2.fontSize)) {
                    style2.fontSize = style2.fontSize + "px";
                }
                fontStyle.push(style2.fontSize);
            }
            else if (style.fontSize) {
                if ($type.isNumber(style.fontSize)) {
                    style.fontSize = style.fontSize + "px";
                }
                fontStyle.push(style.fontSize);
            }
            if (style2 && style2.fontFamily) {
                fontStyle.push(style2.fontFamily);
            }
            else if (style.fontFamily) {
                fontStyle.push(style.fontFamily);
            }
            else if (fontStyle.length) {
                fontStyle.push("Arial");
            }
            return fontStyle.join(" ");
        }
    });
    Object.defineProperty(CanvasText.prototype, "_render", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (parentLayer) {
            var _this = this;
            var layer = this._layer || parentLayer;
            // We need measurements in order to properly position text for alignment
            if (!this._textInfo) {
                this._measure(layer);
            }
            if (this._textVisible) {
                var interactive_1 = this._isInteractive();
                var context_2 = layer.context;
                var layerDirty_1 = layer.dirty;
                var ghostContext_2 = this._renderer._ghostContext;
                context_2.save();
                ghostContext_2.save();
                this._prerender(layer);
                // const lines = this.text.toString().replace(/\r/g, "").split(/\n/);
                // const x = this._localBounds && (this._localBounds.left < 0) ? Math.abs(this._localBounds.left) : 0;
                // Process text info produced by _measure()
                $array.each(this._textInfo, function (line, _index) {
                    $array.each(line.textChunks, function (chunk, _index) {
                        // Set style
                        if (chunk.style) {
                            context_2.save();
                            ghostContext_2.save();
                            context_2.font = chunk.style;
                            if (_this._isInteractive()) {
                                ghostContext_2.font = chunk.style;
                            }
                        }
                        if (chunk.fill) {
                            context_2.save();
                            context_2.fillStyle = chunk.fill.toCSS();
                            // Color does not affect ghostContext so we not set it
                        }
                        // Draw text
                        if (layerDirty_1) {
                            context_2.fillText(chunk.text, chunk.offsetX, line.offsetY + chunk.offsetY);
                        }
                        // Draw underline
                        if (chunk.textDecoration == "underline") {
                            var thickness = 1;
                            var offset = 1;
                            var fontSize = chunk.height;
                            var offsetX = chunk.offsetX;
                            switch (_this.style.textAlign) {
                                case "right":
                                case "end":
                                    offsetX -= chunk.width;
                                    break;
                                case "center":
                                    offsetX -= chunk.width / 2;
                                    break;
                            }
                            if (chunk.style) {
                                var format = TextFormatter.getTextStyle(chunk.style);
                                switch (format.fontWeight) {
                                    case "bolder":
                                    case "bold":
                                    case "700":
                                    case "800":
                                    case "900":
                                        thickness = 2;
                                        break;
                                }
                            }
                            if (fontSize) {
                                offset = fontSize / 20;
                            }
                            var y = thickness + offset * 1.5 + line.offsetY + chunk.offsetY;
                            context_2.save();
                            context_2.beginPath();
                            if (chunk.fill) {
                                context_2.strokeStyle = chunk.fill.toCSS();
                            }
                            else if (_this.style.fill && _this.style.fill instanceof Color) {
                                context_2.strokeStyle = _this.style.fill.toCSS();
                            }
                            context_2.lineWidth = thickness * offset;
                            context_2.moveTo(offsetX, y);
                            context_2.lineTo(offsetX + chunk.width, y);
                            context_2.stroke();
                            context_2.restore();
                        }
                        if (interactive_1 && _this.interactive) {
                            // Draw text in ghost canvas ONLY if it is set as interactive
                            // explicitly. This way we avoid hit test anomalies caused by anti
                            // aliasing of text.
                            ghostContext_2.fillText(chunk.text, chunk.offsetX, line.offsetY + chunk.offsetY);
                        }
                        if (chunk.fill) {
                            context_2.restore();
                            // Color does not affect ghostContext so we not set it
                        }
                        // Reset style
                        if (chunk.style) {
                            context_2.restore();
                            ghostContext_2.restore();
                        }
                    });
                });
                context_2.restore();
                ghostContext_2.restore();
            }
        }
    });
    Object.defineProperty(CanvasText.prototype, "_addBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (bounds) {
            if (this.visible && this.isMeasured) {
                //if (this._textVisible) {
                var x = this._measure(this.getLayer());
                setPoint(bounds, { x: x.left, y: x.top });
                setPoint(bounds, { x: x.right, y: x.bottom });
                //}
            }
        }
    });
    Object.defineProperty(CanvasText.prototype, "_measure", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (layer) {
            var _this = this;
            var context = layer.context;
            var ghostContext = this._renderer._ghostContext;
            var rtl = this.style.direction == "rtl";
            // Reset text info
            this._textInfo = [];
            // Init
            var oversizedBehavior = this.style.oversizedBehavior;
            var maxWidth = this.style.maxWidth;
            var truncate = $type.isNumber(maxWidth) && oversizedBehavior == "truncate";
            var wrap = $type.isNumber(maxWidth) && oversizedBehavior == "wrap";
            // Pre-render
            context.save();
            ghostContext.save();
            this._prerender(layer, true, true);
            // Get default font metrix
            var refText = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789 ";
            // Split up text into lines
            var lines = this.text.toString().replace(/\r/g, "").split(/\n/);
            var styleRestored = true;
            var minX = 0;
            var maxX = 0;
            // Iterate through the lines
            var offsetY = 0;
            var currentStyle;
            $array.each(lines, function (line, _index) {
                // Split up line into format/value chunks
                var chunks;
                if (line == "") {
                    chunks = [{
                            type: "value",
                            text: ""
                        }];
                }
                else {
                    chunks = TextFormatter.chunk(line, false, _this.style.ignoreFormatting);
                }
                var _loop_1 = function () {
                    // Init line object
                    var lineInfo = {
                        offsetY: offsetY,
                        ascent: 0,
                        width: 0,
                        height: 0,
                        left: 0,
                        right: 0,
                        textChunks: []
                    };
                    // Measure reference text
                    var metrics = _this._measureText(refText, context);
                    var height = metrics.actualBoundingBoxAscent + metrics.actualBoundingBoxDescent;
                    lineInfo.height = height;
                    lineInfo.ascent = metrics.actualBoundingBoxAscent;
                    var currentFormat;
                    var currentDecoration = _this.style.textDecoration;
                    var currentFill;
                    var currentChunkWidth;
                    var skipFurtherText = false;
                    var firstTextChunk = true;
                    var leftoverChunks = [];
                    //let offsetX = 0;
                    //let chunk;
                    //while(chunk = chunks.shift()) {
                    $array.eachContinue(chunks, function (chunk, index) {
                        // Format chunk
                        if (chunk.type == "format") {
                            if (chunk.text == "[/]") {
                                if (!styleRestored) {
                                    context.restore();
                                    ghostContext.restore();
                                    styleRestored = true;
                                }
                                currentFill = undefined;
                                currentStyle = undefined;
                                currentChunkWidth = undefined;
                                currentDecoration = _this.style.textDecoration;
                                currentFormat = chunk.text;
                            }
                            else {
                                if (!styleRestored) {
                                    context.restore();
                                    ghostContext.restore();
                                }
                                var format = TextFormatter.getTextStyle(chunk.text);
                                var fontStyle = _this._getFontStyle(format);
                                context.save();
                                ghostContext.save();
                                context.font = fontStyle;
                                currentStyle = fontStyle;
                                currentFormat = chunk.text;
                                if (format.textDecoration) {
                                    currentDecoration = format.textDecoration;
                                }
                                if (format.fill) {
                                    currentFill = format.fill;
                                }
                                if (format.width) {
                                    currentChunkWidth = $type.toNumber(format.width);
                                }
                                styleRestored = false;
                                // Measure reference text after change of format
                                var metrics_1 = _this._measureText(refText, context);
                                var height_1 = metrics_1.actualBoundingBoxAscent + metrics_1.actualBoundingBoxDescent;
                                if (height_1 > lineInfo.height) {
                                    lineInfo.height = height_1;
                                }
                                if (metrics_1.actualBoundingBoxAscent > lineInfo.ascent) {
                                    lineInfo.ascent = metrics_1.actualBoundingBoxAscent;
                                }
                            }
                        }
                        // Text chunk
                        else if (chunk.type == "value" && !skipFurtherText) {
                            // Measure
                            var metrics_2 = _this._measureText(chunk.text, context);
                            var chunkWidth = metrics_2.actualBoundingBoxLeft + metrics_2.actualBoundingBoxRight;
                            // Check for fit
                            if (truncate) {
                                // Break words?
                                var breakWords = firstTextChunk || _this.style.breakWords || false;
                                // Measure ellipsis and check if it fits
                                var ellipsis = _this.style.ellipsis || "";
                                var ellipsisMetrics = _this._measureText(ellipsis, context);
                                var ellipsisWidth = ellipsisMetrics.actualBoundingBoxLeft + ellipsisMetrics.actualBoundingBoxRight;
                                // Check fit
                                if ((lineInfo.width + chunkWidth) > maxWidth) {
                                    var excessWidth = maxWidth - lineInfo.width - ellipsisWidth;
                                    chunk.text = _this._truncateText(context, chunk.text, excessWidth, breakWords);
                                    chunk.text += ellipsis;
                                    skipFurtherText = true;
                                }
                            }
                            else if (wrap) {
                                // Check fit
                                if ((lineInfo.width + chunkWidth) > maxWidth) {
                                    var excessWidth = maxWidth - lineInfo.width;
                                    var tmpText = _this._truncateText(context, chunk.text, excessWidth, false, firstTextChunk);
                                    if (tmpText == "") {
                                        // Unable to fit a single letter - hide the whole label
                                        _this._textVisible = true;
                                        return false;
                                    }
                                    //skipFurtherText = true;
                                    //Add remaining chunks for the next line
                                    leftoverChunks = chunks.slice(index + 1);
                                    //Add remaining text of current chunk if it was forced-cut
                                    if ($utils.trim(tmpText) != $utils.trim(chunk.text)) {
                                        leftoverChunks.unshift({
                                            type: "value",
                                            text: chunk.text.substr(tmpText.length)
                                        });
                                        if (currentFormat) {
                                            leftoverChunks.unshift({
                                                type: "format",
                                                text: currentFormat
                                            });
                                        }
                                    }
                                    // Set current chunk (truncated)
                                    chunk.text = $utils.trim(tmpText);
                                    chunks = [];
                                    skipFurtherText = true;
                                }
                            }
                            // Chunk width?
                            var leftBoundMod = 1;
                            var rightBoundMod = 1;
                            if (currentStyle && currentChunkWidth && (currentChunkWidth > chunkWidth)) {
                                // increase horizontal bounding boxes accordingly
                                var boundsMod = chunkWidth / currentChunkWidth;
                                switch (_this.style.textAlign) {
                                    case "right":
                                    case "end":
                                        leftBoundMod = boundsMod;
                                        break;
                                    case "center":
                                        leftBoundMod = boundsMod;
                                        rightBoundMod = boundsMod;
                                        break;
                                    default:
                                        rightBoundMod = boundsMod;
                                }
                                chunkWidth = currentChunkWidth;
                            }
                            var chunkHeight = metrics_2.actualBoundingBoxAscent + metrics_2.actualBoundingBoxDescent;
                            if (chunkHeight > lineInfo.height) {
                                lineInfo.height = chunkHeight;
                            }
                            if (metrics_2.actualBoundingBoxAscent > lineInfo.ascent) {
                                lineInfo.ascent = metrics_2.actualBoundingBoxAscent;
                            }
                            lineInfo.width += chunkWidth;
                            lineInfo.left += metrics_2.actualBoundingBoxLeft / leftBoundMod;
                            lineInfo.right += metrics_2.actualBoundingBoxRight / rightBoundMod;
                            lineInfo.textChunks.push({
                                style: currentStyle,
                                fill: currentFill,
                                text: chunk.text,
                                width: chunkWidth,
                                height: chunkHeight,
                                left: metrics_2.actualBoundingBoxLeft,
                                right: metrics_2.actualBoundingBoxRight,
                                ascent: metrics_2.actualBoundingBoxAscent,
                                offsetX: 0,
                                offsetY: 0,
                                textDecoration: currentDecoration
                            });
                            //offsetX += chunkWidth;
                            firstTextChunk = false;
                        }
                        if (leftoverChunks) {
                            //return false;
                        }
                        return true;
                        //}
                    });
                    if (_this.style.lineHeight instanceof Percent) {
                        lineInfo.height *= _this.style.lineHeight.value;
                        lineInfo.ascent *= _this.style.lineHeight.value;
                    }
                    else {
                        lineInfo.height *= _this.style.lineHeight || 1.2;
                        lineInfo.ascent *= _this.style.lineHeight || 1.2;
                    }
                    if (minX < lineInfo.left) {
                        minX = lineInfo.left;
                    }
                    if (maxX < lineInfo.right) {
                        maxX = lineInfo.right;
                    }
                    _this._textInfo.push(lineInfo);
                    //lineInfo.offsetY += lineInfo.ascent;
                    offsetY += lineInfo.height;
                    // Reset chunks so that it can proceed to the next line
                    chunks = leftoverChunks || [];
                };
                while (chunks.length > 0) {
                    _loop_1();
                }
            });
            if (!styleRestored) {
                context.restore();
                ghostContext.restore();
            }
            // Adjust chunk internal offsets
            $array.each(this._textInfo, function (lineInfo, _index) {
                var currentChunkOffset = 0;
                $array.each(lineInfo.textChunks, function (chunk) {
                    chunk.offsetX = currentChunkOffset + chunk.left - lineInfo.left;
                    chunk.offsetY += lineInfo.height - lineInfo.height * (_this.style.baselineRatio || 0.19);
                    currentChunkOffset += chunk.width;
                });
            });
            var bounds = {
                left: rtl ? -maxX : -minX,
                top: 0,
                right: rtl ? minX : maxX,
                bottom: offsetY,
            };
            // We need to fit?
            if (oversizedBehavior !== "none") {
                var ratio = this._fitRatio(bounds);
                if (ratio < 1) {
                    if (oversizedBehavior == "fit") {
                        if ($type.isNumber(this.style.minScale) && (ratio < this.style.minScale)) {
                            this._textVisible = false;
                        }
                        else {
                            if (!this._originalScale || this._originalScale == 1) {
                                this._originalScale = this.scale;
                            }
                            this.scale = ratio;
                            this._textVisible = true;
                        }
                    }
                    else if (oversizedBehavior == "hide") {
                        this._textVisible = false;
                    }
                    else {
                        switch (this.style.textAlign) {
                            case "right":
                            case "end":
                                bounds.left = -maxWidth;
                                bounds.right = 0;
                                break;
                            case "center":
                                bounds.left = -maxWidth / 2;
                                bounds.right = maxWidth / 2;
                                break;
                            default:
                                bounds.left = 0;
                                bounds.right = maxWidth;
                        }
                        this.scale = this._originalScale || 1;
                        this._originalScale = undefined;
                        this._textVisible = true;
                    }
                }
                else {
                    this.scale = this._originalScale || 1;
                    this._originalScale = undefined;
                    this._textVisible = true;
                }
            }
            context.restore();
            ghostContext.restore();
            return bounds;
        }
    });
    Object.defineProperty(CanvasText.prototype, "_fitRatio", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (bounds) {
            var maxW = this.style.maxWidth;
            var maxH = this.style.maxHeight;
            if (!$type.isNumber(maxW) && !$type.isNumber(maxH)) {
                return 1;
            }
            var w = bounds.right - bounds.left;
            var h = bounds.bottom - bounds.top;
            return Math.min(maxW / w || 1, maxH / h || 1);
        }
    });
    Object.defineProperty(CanvasText.prototype, "_truncateText", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (context, text, maxWidth, breakWords, fallbackBreakWords) {
            if (breakWords === void 0) { breakWords = false; }
            if (fallbackBreakWords === void 0) { fallbackBreakWords = true; }
            var width;
            do {
                if (breakWords) {
                    text = text.slice(0, -1);
                }
                else {
                    var tmp = text.replace(/[^,;:!?\\\/\s]+[,;:!?\\\/\s]*$/g, "");
                    if (tmp == "" && fallbackBreakWords) {
                        breakWords = true;
                    }
                    else {
                        text = tmp;
                    }
                }
                var metrics = this._measureText(text, context);
                width = metrics.actualBoundingBoxLeft + metrics.actualBoundingBoxRight;
            } while ((width > maxWidth) && text != "");
            return text;
        }
    });
    Object.defineProperty(CanvasText.prototype, "_measureText", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (text, context) {
            var metrics = context.measureText(text);
            var fakeMetrics = {};
            if (metrics.actualBoundingBoxAscent == null) {
                var div = document.createElement("div");
                div.innerText = text;
                div.style.visibility = "hidden";
                div.style.position = "absolute";
                div.style.top = "-1000000px;";
                div.style.fontFamily = this.style.fontFamily || "";
                div.style.fontSize = this.style.fontSize + "";
                document.body.appendChild(div);
                var bbox = div.getBoundingClientRect();
                document.body.removeChild(div);
                var h = bbox.height;
                var w_1 = metrics.width;
                var left = 0;
                var right = w_1;
                fakeMetrics = {
                    actualBoundingBoxAscent: h,
                    actualBoundingBoxDescent: 0,
                    actualBoundingBoxLeft: left,
                    actualBoundingBoxRight: right,
                    fontBoundingBoxAscent: h,
                    fontBoundingBoxDescent: 0,
                    width: w_1
                };
                //return fake;
            }
            else {
                fakeMetrics = {
                    actualBoundingBoxAscent: metrics.actualBoundingBoxAscent,
                    actualBoundingBoxDescent: metrics.actualBoundingBoxDescent,
                    actualBoundingBoxLeft: metrics.actualBoundingBoxLeft,
                    actualBoundingBoxRight: metrics.actualBoundingBoxRight,
                    fontBoundingBoxAscent: metrics.actualBoundingBoxAscent,
                    fontBoundingBoxDescent: metrics.actualBoundingBoxDescent,
                    width: metrics.width
                };
            }
            var w = metrics.width;
            switch (this.style.textAlign) {
                case "right":
                case "end":
                    fakeMetrics.actualBoundingBoxLeft = w;
                    fakeMetrics.actualBoundingBoxRight = 0;
                    break;
                case "center":
                    fakeMetrics.actualBoundingBoxLeft = w / 2;
                    fakeMetrics.actualBoundingBoxRight = w / 2;
                    break;
                default:
                    fakeMetrics.actualBoundingBoxLeft = 0;
                    fakeMetrics.actualBoundingBoxRight = w;
            }
            return fakeMetrics;
        }
    });
    return CanvasText;
}(CanvasDisplayObject));
export { CanvasText };
/**
 * @ignore
 */
var CanvasTextStyle = /** @class */ (function () {
    function CanvasTextStyle() {
        //public wordWrapWidth: number = 100;
        Object.defineProperty(this, "fill", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "textAlign", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "fontFamily", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "fontSize", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "fontWeight", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "fontStyle", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "fontVariant", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "textDecoration", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "shadowColor", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "shadowBlur", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "shadowOffsetX", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "shadowOffsetY", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "shadowOpacity", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        // leading?: number;
        // letterSpacing?: number;
        Object.defineProperty(this, "lineHeight", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: percent(120)
        });
        Object.defineProperty(this, "baselineRatio", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0.19
        });
        // padding?: number;
        // stroke?: number;
        // strokeThickness?: number;
        // trim?: number;
        // wordWrap?: boolean;
        Object.defineProperty(this, "direction", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "textBaseline", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "oversizedBehavior", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: "none"
        });
        Object.defineProperty(this, "breakWords", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(this, "ellipsis", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: "…"
        });
        Object.defineProperty(this, "maxWidth", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "maxHeight", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "minScale", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "ignoreFormatting", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
    }
    return CanvasTextStyle;
}());
export { CanvasTextStyle };
/**
 * @ignore
 */
var CanvasRadialText = /** @class */ (function (_super) {
    __extends(CanvasRadialText, _super);
    function CanvasRadialText() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "textType", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: "circular"
        });
        Object.defineProperty(_this, "radius", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "startAngle", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "inside", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(_this, "orientation", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: "auto"
        });
        Object.defineProperty(_this, "kerning", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "_textReversed", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        return _this;
    }
    Object.defineProperty(CanvasRadialText.prototype, "_render", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (parentLayer) {
            switch (this.textType) {
                case "circular":
                    this._renderCircular(parentLayer);
                    break;
                default:
                    _super.prototype._render.call(this, parentLayer);
                    break;
            }
        }
    });
    Object.defineProperty(CanvasRadialText.prototype, "_renderCircular", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (parentLayer) {
            var layer = this._layer || parentLayer;
            this._prerender(layer);
            var interactive = this._isInteractive();
            var context = layer.context;
            var layerDirty = layer.dirty;
            var ghostContext = this._renderer._ghostContext;
            // Savepoint
            context.save();
            if (interactive) {
                ghostContext.save();
            }
            // Init
            var radius = (this.radius || 0);
            var startAngle = (this.startAngle || 0);
            var deltaAngle = 0;
            var orientation = this.orientation;
            var inward = orientation == "auto" ? "auto" : orientation == "inward";
            var inside = this.inside;
            var align = this.style.textAlign || "left";
            var kerning = this.kerning || 0;
            var clockwise = align == "left" ? 1 : -1;
            var shouldReverse = !this._textReversed;
            // We need measurements in order to properly position text for alignment
            if (!this._textInfo) {
                this._measure(layer);
            }
            // Check if we need to invert the whole stuff
            if (inward == "auto") {
                // Calc max angle so we know whether we need to flip it
                var maxAngle_1 = 0;
                var midAngle = 0;
                $array.each(this._textInfo, function (line, _index) {
                    var deltaAngle = startAngle + (line.width / (radius - line.height)) / 2 * -clockwise;
                    if (deltaAngle > maxAngle_1) {
                        maxAngle_1 = deltaAngle;
                    }
                });
                if (align == "left") {
                    midAngle = (maxAngle_1 + deltaAngle / 2) * $math.DEGREES;
                }
                else if (align == "right") {
                    midAngle = (maxAngle_1 - deltaAngle / 2) * $math.DEGREES;
                }
                else {
                    midAngle = startAngle * $math.DEGREES;
                }
                midAngle = $math.normalizeAngle(midAngle);
                inward = (midAngle >= 270) || (midAngle <= 90);
            }
            if (inward == true && shouldReverse) {
                this._textInfo.reverse();
                this._textReversed = true;
            }
            // if ((inward == false && align == "left") || (inward == true && align == "right")) {
            // 	clockwise *= -1;
            // }
            // Process text info produced by _measure()
            $array.each(this._textInfo, function (line, _index) {
                var textHeight = line.height;
                // Adjust radius (for `inside = false`)
                // Radius adjustment for `inside = false` is below the line calculation
                if (!inside) {
                    radius += textHeight;
                }
                // Reverse letters if we're painting them counter-clockwise
                if (((clockwise == -1 && inward) || (clockwise == 1 && !inward)) && shouldReverse) {
                    line.textChunks.reverse();
                }
                // Init angles
                var lineStartAngle = startAngle;
                deltaAngle = 0;
                // Adjust for center-align
                if (align == "center") {
                    lineStartAngle += (line.width / (radius - textHeight)) / 2 * -clockwise;
                    deltaAngle = lineStartAngle - startAngle;
                }
                // if (inward == "auto") {
                // 	let midAngle;
                // 	if (align == "left") {
                // 		midAngle = (lineStartAngle + deltaAngle / 2) * $math.DEGREES;
                // 	}
                // 	else if () {
                // 		midAngle = (lineStartAngle - deltaAngle / 2) * $math.DEGREES;
                // 	}
                // 	inward = (midAngle >= 270) || (midAngle <= 90);
                // }
                // Rotate letters if they are facing outward
                lineStartAngle += (Math.PI * (inward ? 0 : 1)); // Rotate 180 if outward
                // Savepoint
                context.save();
                if (interactive) {
                    ghostContext.save();
                }
                // Assume starting angle
                context.rotate(lineStartAngle);
                if (interactive) {
                    ghostContext.rotate(lineStartAngle);
                }
                var angleShift = 0;
                $array.each(line.textChunks, function (chunk, _index) {
                    // Draw the letter
                    var char = chunk.text;
                    var charWidth = chunk.width;
                    // Rotate half a letter
                    angleShift = (charWidth / 2) / (radius - textHeight) * clockwise;
                    context.rotate(angleShift);
                    if (interactive) {
                        ghostContext.rotate(angleShift);
                    }
                    // Set style
                    if (chunk.style) {
                        context.save();
                        ghostContext.save();
                        context.font = chunk.style;
                        if (interactive) {
                            ghostContext.font = chunk.style;
                        }
                    }
                    if (chunk.fill) {
                        context.save();
                        context.fillStyle = chunk.fill.toCSS();
                        // Color does not affect ghostContext so we not set it
                    }
                    // Center letters
                    context.textBaseline = "middle";
                    context.textAlign = "center";
                    if (interactive) {
                        ghostContext.textBaseline = "middle";
                        ghostContext.textAlign = "center";
                    }
                    // Plop the letter
                    if (layerDirty) {
                        context.fillText(char, 0, (inward ? 1 : -1) * (0 - radius + textHeight / 2));
                    }
                    if (interactive) {
                        ghostContext.fillText(char, 0, (inward ? 1 : -1) * (0 - radius + textHeight / 2));
                    }
                    if (chunk.fill) {
                        context.restore();
                        // Color does not affect ghostContext so we not set it
                    }
                    // Reset style
                    if (chunk.style) {
                        context.restore();
                        ghostContext.restore();
                    }
                    // Rotate half a letter and add spacing
                    angleShift = (charWidth / 2 + kerning) / (radius - textHeight) * clockwise;
                    context.rotate(angleShift);
                    if (interactive) {
                        ghostContext.rotate(angleShift);
                    }
                });
                // Restore angle
                context.restore();
                if (interactive) {
                    ghostContext.restore();
                }
                // Adjust radius (for `inside = true`)
                if (inside) {
                    radius -= textHeight;
                }
            });
            // Restore
            context.restore();
            if (interactive) {
                ghostContext.restore();
            }
        }
    });
    Object.defineProperty(CanvasRadialText.prototype, "_measure", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (layer) {
            switch (this.textType) {
                case "circular":
                    return this._measureCircular(layer);
                default:
                    return _super.prototype._measure.call(this, layer);
            }
        }
    });
    Object.defineProperty(CanvasRadialText.prototype, "_measureCircular", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (layer) {
            var _this = this;
            var context = layer.context;
            var ghostContext = this._renderer._ghostContext;
            var rtl = this.style.direction == "rtl";
            // Reset text info
            this._textInfo = [];
            this._textReversed = false;
            // Pre-render
            context.save();
            ghostContext.save();
            this._prerender(layer, true);
            // Split up text into lines
            var lines = this.text.toString().replace(/\r/g, "").split(/\n/);
            var styleRestored = true;
            // Iterate through the lines
            var offsetY = 0;
            $array.each(lines, function (line, _index) {
                // Split up line into format/value chunks
                var chunks = TextFormatter.chunk(line, false, _this.style.ignoreFormatting);
                // Init line object
                var lineInfo = {
                    offsetY: offsetY,
                    ascent: 0,
                    width: 0,
                    height: 0,
                    left: 0,
                    right: 0,
                    textChunks: []
                };
                var currentStyle;
                var currentFill;
                var currentChunkWidth;
                //while(chunk = chunks.shift()) {
                $array.each(chunks, function (chunk, _index) {
                    // Format chunk
                    if (chunk.type == "format") {
                        if (chunk.text == "[/]") {
                            if (!styleRestored) {
                                context.restore();
                                ghostContext.restore();
                                styleRestored = true;
                            }
                            currentFill = undefined;
                            currentStyle = undefined;
                            currentChunkWidth = undefined;
                        }
                        else {
                            var format = TextFormatter.getTextStyle(chunk.text);
                            var fontStyle = _this._getFontStyle(format);
                            context.save();
                            ghostContext.save();
                            context.font = fontStyle;
                            currentStyle = fontStyle;
                            if (format.fill) {
                                currentFill = format.fill;
                            }
                            if (format.width) {
                                currentChunkWidth = $type.toNumber(format.width);
                            }
                            styleRestored = false;
                        }
                    }
                    // Text format
                    else if (chunk.type == "value") {
                        // Measure each letter
                        var chars = chunk.text.match(/./ug) || [];
                        if (rtl) {
                            chars.reverse();
                        }
                        for (var i = 0; i < chars.length; i++) {
                            var char = chars[i];
                            // Measure
                            var metrics = _this._measureText(char, context);
                            var chunkWidth = metrics.width;
                            // Chunk width?
                            if (currentStyle && currentChunkWidth && (currentChunkWidth > chunkWidth)) {
                                chunkWidth = currentChunkWidth;
                            }
                            var chunkHeight = metrics.actualBoundingBoxAscent + metrics.actualBoundingBoxDescent;
                            if (chunkHeight > lineInfo.height) {
                                lineInfo.height = chunkHeight;
                            }
                            if (metrics.actualBoundingBoxAscent > lineInfo.ascent) {
                                lineInfo.ascent = metrics.actualBoundingBoxAscent;
                            }
                            lineInfo.width += chunkWidth;
                            lineInfo.left += metrics.actualBoundingBoxLeft;
                            lineInfo.right += metrics.actualBoundingBoxRight;
                            lineInfo.textChunks.push({
                                style: currentStyle,
                                fill: currentFill,
                                text: char,
                                width: chunkWidth,
                                height: chunkHeight + metrics.actualBoundingBoxDescent,
                                left: metrics.actualBoundingBoxLeft,
                                right: metrics.actualBoundingBoxRight,
                                ascent: metrics.actualBoundingBoxAscent,
                                offsetX: 0,
                                offsetY: chunkHeight,
                                textDecoration: undefined
                            });
                            if (rtl) {
                                break;
                            }
                        }
                    }
                });
                if (_this.style.lineHeight instanceof Percent) {
                    lineInfo.height *= _this.style.lineHeight.value;
                }
                else {
                    lineInfo.height *= _this.style.lineHeight || 1.2;
                }
                _this._textInfo.push(lineInfo);
                //lineInfo.offsetY += lineInfo.ascent;
                offsetY += lineInfo.height;
            });
            if (!styleRestored) {
                context.restore();
                ghostContext.restore();
            }
            // Adjust chunk internal offsets
            $array.each(this._textInfo, function (lineInfo) {
                $array.each(lineInfo.textChunks, function (chunk) {
                    chunk.offsetY += Math.round((lineInfo.height - chunk.height + (lineInfo.ascent - chunk.ascent)) / 2);
                });
            });
            context.restore();
            ghostContext.restore();
            return {
                left: 0,
                top: 0,
                right: 0,
                bottom: 0,
            };
        }
    });
    return CanvasRadialText;
}(CanvasText));
export { CanvasRadialText };
/**
 * @ignore
 */
var CanvasImage = /** @class */ (function (_super) {
    __extends(CanvasImage, _super);
    function CanvasImage(renderer, image) {
        var _this = _super.call(this, renderer) || this;
        Object.defineProperty(_this, "width", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "height", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "image", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "tainted", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "shadowColor", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "shadowBlur", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "shadowOffsetX", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "shadowOffsetY", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "shadowOpacity", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_imageMask", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        _this.image = image;
        return _this;
    }
    Object.defineProperty(CanvasImage.prototype, "_dispose", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._dispose.call(this);
            if (this._imageMask) {
                clearCanvas(this._imageMask);
            }
        }
    });
    Object.defineProperty(CanvasImage.prototype, "getLocalBounds", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (!this._localBounds) {
                var w = 0;
                var h = 0;
                if (this.width) {
                    w = this.width;
                }
                if (this.height) {
                    h = this.height;
                }
                this._localBounds = {
                    left: 0,
                    top: 0,
                    right: w,
                    bottom: h
                };
                this._addBounds(this._localBounds);
            }
            return this._localBounds;
        }
    });
    Object.defineProperty(CanvasImage.prototype, "_render", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (parentLayer) {
            _super.prototype._render.call(this, parentLayer);
            if (this.image) {
                var layer = this._layer || parentLayer;
                if (this.tainted === undefined) {
                    this.tainted = isTainted(this.image);
                    layer.tainted = true;
                }
                if (this.tainted && this._renderer._omitTainted) {
                    return;
                }
                if (layer.dirty) {
                    if (this.shadowColor) {
                        layer.context.shadowColor = this.shadowColor.toCSS(this.shadowOpacity || 1);
                    }
                    if (this.shadowBlur) {
                        layer.context.shadowBlur = this.shadowBlur;
                    }
                    if (this.shadowOffsetX) {
                        layer.context.shadowOffsetX = this.shadowOffsetX;
                    }
                    if (this.shadowOffsetY) {
                        layer.context.shadowOffsetY = this.shadowOffsetY;
                    }
                    var width = this.width || this.image.naturalWidth;
                    var height = this.height || this.image.naturalHeight;
                    layer.context.drawImage(this.image, 0, 0, width, height);
                }
                if (this.interactive && this._isInteractive()) {
                    var mask = this._getMask(this.image);
                    this._renderer._ghostContext.drawImage(mask, 0, 0);
                }
            }
        }
    });
    Object.defineProperty(CanvasImage.prototype, "clear", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype.clear.call(this);
            this.image = undefined;
            this._imageMask = undefined;
        }
    });
    Object.defineProperty(CanvasImage.prototype, "_getMask", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (image) {
            if (this._imageMask === undefined) {
                var width = this.width || image.naturalWidth;
                var height = this.height || image.naturalHeight;
                // We need to create a second canvas because destination-in clears out the entire canvas
                var canvas = document.createElement("canvas");
                canvas.width = width;
                canvas.height = height;
                var context = canvas.getContext("2d");
                context.imageSmoothingEnabled = false;
                context.fillStyle = this._getColorId();
                context.fillRect(0, 0, width, height);
                if (!isTainted(image)) {
                    context.globalCompositeOperation = "destination-in";
                    context.drawImage(image, 0, 0, width, height);
                }
                this._imageMask = canvas;
            }
            return this._imageMask;
        }
    });
    return CanvasImage;
}(CanvasDisplayObject));
export { CanvasImage };
/**
 * @ignore
 */
var CanvasRendererEvent = /** @class */ (function () {
    function CanvasRendererEvent(event, point, bbox) {
        Object.defineProperty(this, "event", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: event
        });
        Object.defineProperty(this, "point", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: point
        });
        Object.defineProperty(this, "bbox", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: bbox
        });
        Object.defineProperty(this, "id", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "simulated", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(this, "native", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: true
        });
        if ($utils.supports("touchevents") && event instanceof Touch) {
            this.id = event.identifier;
        }
        else {
            this.id = null;
        }
    }
    return CanvasRendererEvent;
}());
export { CanvasRendererEvent };
/**
 * @ignore
 */
var CanvasRenderer = /** @class */ (function (_super) {
    __extends(CanvasRenderer, _super);
    function CanvasRenderer(resolution) {
        var _this = _super.call(this) || this;
        Object.defineProperty(_this, "view", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: document.createElement("div")
        });
        Object.defineProperty(_this, "_layerDom", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: document.createElement("div")
        });
        Object.defineProperty(_this, "layers", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        Object.defineProperty(_this, "_dirtyLayers", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        Object.defineProperty(_this, "defaultLayer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this.getLayer(0)
        });
        Object.defineProperty(_this, "_ghostView", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_ghostContext", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_patternCanvas", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: document.createElement("canvas")
        });
        Object.defineProperty(_this, "_patternContext", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: _this._patternCanvas.getContext("2d")
        });
        Object.defineProperty(_this, "_width", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "_height", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "_clientWidth", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "_clientHeight", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "resolution", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "interactionsEnabled", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: true
        });
        Object.defineProperty(_this, "_listeners", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        Object.defineProperty(_this, "_events", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        Object.defineProperty(_this, "_colorId", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "_colorMap", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {}
        });
        Object.defineProperty(_this, "_forceInteractive", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: 0
        });
        Object.defineProperty(_this, "_omitTainted", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        // TODO this should store the Id as well
        Object.defineProperty(_this, "_hovering", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Set()
        });
        Object.defineProperty(_this, "_dragging", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        Object.defineProperty(_this, "_mousedown", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        Object.defineProperty(_this, "_lastPointerMoveEvent", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_mouseMoveThrottler", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Throttler(function () {
                _this._dispatchGlobalMousemove(_this._lastPointerMoveEvent.event, _this._lastPointerMoveEvent.native);
            })
        });
        if (resolution == null) {
            _this.resolution = window.devicePixelRatio;
        }
        else {
            _this.resolution = resolution;
        }
        _this.view.appendChild(_this._layerDom);
        _this._disposers.push(new Disposer(function () {
            $object.each(_this._events, function (_key, events) {
                events.disposer.dispose();
            });
            $array.each(_this.layers, function (layer) {
                clearCanvas(layer.view);
                if (layer.exportableView) {
                    clearCanvas(layer.exportableView);
                }
            });
            clearCanvas(_this._ghostView);
            clearCanvas(_this._patternCanvas);
        }));
        // @todo : do the same for ghost
        _this._ghostView = document.createElement("canvas");
        _this._ghostContext = _this._ghostView.getContext("2d", { alpha: false });
        _this._ghostContext.imageSmoothingEnabled = false;
        // Monitor for possible pixel ratio changes (when page is zoomed)
        _this._disposers.push($utils.addEventListener(window, "resize", function (_ev) {
            if (resolution == null) {
                _this.resolution = window.devicePixelRatio;
            }
        }));
        // We need this in order top prevent default touch gestures when dragging
        // draggable elements
        if ($utils.supports("touchevents")) {
            var listener = function (ev) {
                if (_this._dragging.length !== 0) {
                    ev.preventDefault();
                }
            };
            _this._disposers.push($utils.addEventListener(window, "touchstart", listener, { passive: false }));
            _this._disposers.push($utils.addEventListener(_this.view, "touchstart", listener, { passive: false }));
        }
        // Prevent scrolling of the window when hovering on "wheelable" object
        if ($utils.supports("wheelevents")) {
            _this._disposers.push($utils.addEventListener(_this.view, "wheel", function (ev) {
                var prevent = false;
                _this._hovering.forEach(function (obj) {
                    if (obj.wheelable) {
                        prevent = true;
                        return false;
                    }
                });
                if (prevent) {
                    ev.preventDefault();
                }
            }, { passive: false }));
        }
        return _this;
    }
    Object.defineProperty(CanvasRenderer.prototype, "createLinearGradient", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (x1, y1, x2, y2) {
            return this.defaultLayer.context.createLinearGradient(x1, y1, x2, y2);
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "createRadialGradient", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (x1, y1, radius1, x2, y2, radius2) {
            return this.defaultLayer.context.createRadialGradient(x1, y1, radius1, x2, y2, radius2);
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "createPattern", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (graphics, background, repetition, width, height) {
            // const patternCanvas = document.createElement("canvas");
            // const patternContext = patternCanvas.getContext("2d")!;
            // patternCanvas.width = width;
            // patternCanvas.height = height;
            // if (fill) {
            // 	patternContext.fillStyle = fill.toCSS();
            // 	patternContext.fillRect(0, 0, patternCanvas.width, patternCanvas.height);
            // }
            // const layer = {
            // 	view: patternCanvas,
            // 	context: patternContext,
            // 	visible: true,
            // 	order: 0,
            // 	width: width,
            // 	height: height,
            // 	dirty: true
            // };
            // // patternContext.arc(0, 0, 50, 0, .5 * Math.PI);
            // // patternContext.stroke();
            // image.targetLayer = layer;
            // image.render(layer);
            //this._layerDom.appendChild(patternCanvas);
            this._patternCanvas.width = width;
            this._patternCanvas.height = height;
            this._patternContext.clearRect(0, 0, width, height);
            // patternCanvas.style.width = width * this.resolution + "px";
            // patternCanvas.style.height = height * this.resolution + "px";
            background.renderDetached(this._patternContext);
            graphics.renderDetached(this._patternContext);
            return this._patternContext.createPattern(this._patternCanvas, repetition);
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "makeContainer", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return new CanvasContainer(this);
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "makeGraphics", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return new CanvasGraphics(this);
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "makeText", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (text, style) {
            return new CanvasText(this, text, style);
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "makeTextStyle", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            return new CanvasTextStyle();
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "makeRadialText", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (text, style) {
            return new CanvasRadialText(this, text, style);
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "makePicture", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (image) {
            return new CanvasImage(this, image);
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "resize", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (width, height) {
            var _this = this;
            this._clientWidth = width;
            this._clientHeight = height;
            this._width = Math.floor(width * this.resolution);
            this._height = Math.floor(height * this.resolution);
            $array.each(this.layers, function (layer) {
                if (layer) {
                    layer.dirty = true;
                    if (layer.width != null) {
                        layer.view.width = layer.width;
                        layer.view.style.width = layer.width + "px";
                    }
                    else {
                        layer.view.width = _this._width;
                        layer.view.style.width = width + "px";
                    }
                    if (layer.height != null) {
                        layer.view.height = layer.height;
                        layer.view.style.height = layer.height + "px";
                    }
                    else {
                        layer.view.height = _this._height;
                        layer.view.style.height = height + "px";
                    }
                }
            });
            // @todo: do the same for ghost canvases
            this._ghostView.width = this._width;
            this._ghostView.height = this._height;
            this._ghostView.style.width = width + "px";
            this._ghostView.style.height = height + "px";
            this.view.style.width = width + "px";
            this.view.style.height = height + "px";
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "createDetachedLayer", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var view = document.createElement("canvas");
            var context = view.getContext("2d");
            var layer = {
                view: view,
                context: context,
                order: 0,
                visible: true,
                width: undefined,
                height: undefined,
                dirty: true,
                tainted: false
            };
            view.style.position = "absolute";
            view.style.top = "0px";
            view.style.left = "0px";
            return layer;
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "getLayerByOrder", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (order) {
            var layers = this.layers;
            var length = layers.length;
            for (var i = 0; i < length; i++) {
                var layer = layers[i];
                if (layer.order == order) {
                    return layer;
                }
            }
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "getLayer", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (order, visible) {
            if (visible === void 0) { visible = true; }
            var layers = this.layers;
            var existingLayer = this.getLayerByOrder(order);
            if (existingLayer) {
                return existingLayer;
            }
            var layer = this.createDetachedLayer();
            layer.order = order;
            layer.visible = visible;
            if (layer.visible && this._width) {
                layer.view.width = this._width;
                layer.view.style.width = this._clientWidth + "px";
                layer.view.height = this._height;
                layer.view.style.height = this._clientHeight + "px";
            }
            layers.push(layer);
            layers.sort(function (a, b) {
                if (a.order > b.order) {
                    return 1;
                }
                else if (a.order < b.order) {
                    return -1;
                }
                else {
                    return 0;
                }
            });
            var length = layers.length;
            var layerIndex = $array.indexOf(layers, layer);
            var next;
            for (var i = layerIndex + 1; i < length; i++) {
                if (layers[i].visible) {
                    next = layers[i];
                    break;
                }
            }
            if (visible) {
                if (next === undefined) {
                    this._layerDom.appendChild(layer.view);
                }
                else {
                    this._layerDom.insertBefore(layer.view, next.view);
                }
            }
            return layer;
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "render", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (root) {
            var _this = this;
            this._dirtyLayers.length = 0;
            $array.each(this.layers, function (layer) {
                if (layer) {
                    if (layer.dirty && layer.visible) {
                        var context = layer.context;
                        _this._dirtyLayers.push(layer);
                        context.save();
                        context.clearRect(0, 0, _this._width, _this._height);
                    }
                }
            });
            this._ghostContext.save();
            //this._ghostContext.clearRect(0, 0, this._width, this._height);
            //this._ghostContext.beginPath();
            this._ghostContext.fillStyle = '#000';
            this._ghostContext.fillRect(0, 0, this._width, this._height);
            root.render(this.defaultLayer);
            this._ghostContext.restore();
            //setTimeout(() => {
            // Remove this after the Chrome bug is fixed:
            // https://bugs.chromium.org/p/chromium/issues/detail?id=1279394
            $array.each(this.layers, function (layer) {
                if (layer) {
                    var context = layer.context;
                    context.beginPath();
                    context.moveTo(0, 0);
                    context.stroke();
                }
            });
            $array.each(this._dirtyLayers, function (layer) {
                layer.context.restore();
                layer.dirty = false;
            });
            //}, 100)
            if (this._hovering.size && this._lastPointerMoveEvent) {
                this._mouseMoveThrottler.run();
            }
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "paintId", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (obj) {
            var id = distributeId(++this._colorId);
            var color = Color.fromHex(id).toCSS();
            this._colorMap[color] = obj;
            return color;
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "_removeObject", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (obj) {
            if (obj._colorId !== undefined) {
                delete this._colorMap[obj._colorId];
            }
        }
    });
    // protected _identifyObjectByColor(colorId: number): CanvasDisplayObject | undefined {
    // 	return this._colorMap[colorId];
    // }
    Object.defineProperty(CanvasRenderer.prototype, "getEvent", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (originalEvent, adjustPoint) {
            if (adjustPoint === void 0) { adjustPoint = true; }
            var bbox = adjustPoint ? this.view.getBoundingClientRect() : new DOMRect(0, 0, 0, 0);
            return new CanvasRendererEvent(originalEvent, (originalEvent.clientX || originalEvent.clientY ? {
                x: originalEvent.clientX - (originalEvent.clientX ? bbox.left : 0),
                y: originalEvent.clientY - (originalEvent.clientY ? bbox.top : 0),
            } : {
                x: 0,
                y: 0
            }), bbox);
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "_getHitTarget", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (point, bbox) {
            if (point.x < 0 || point.x > bbox.width || point.y < 0 || point.y > bbox.height) {
                return;
            }
            else {
                var pixel = this._ghostContext.getImageData(
                // TODO should this round ?
                Math.round((point.x / bbox.width) * this._width), Math.round((point.y / bbox.height) * this._height), 1, 1);
                if (pixel.data[0] === 0 && pixel.data[1] === 0 && pixel.data[2] === 0) {
                    return false;
                }
                var colorId = Color.fromRGB(pixel.data[0], pixel.data[1], pixel.data[2]).toCSS();
                var hit = this._colorMap[colorId];
                return hit;
            }
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "_withEvents", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, f) {
            var events = this._events[key];
            if (events !== undefined) {
                events.dispatching = true;
                try {
                    f(events);
                }
                finally {
                    events.dispatching = false;
                    if (events.cleanup) {
                        events.cleanup = false;
                        $array.keepIf(events.callbacks, function (callback) {
                            return !callback.disposed;
                        });
                        if (events.callbacks.length === 0) {
                            events.disposer.dispose();
                            delete this._events[key];
                        }
                    }
                }
            }
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "_dispatchEventAll", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, event) {
            if (!this.interactionsEnabled) {
                return;
            }
            this._withEvents(key, function (events) {
                $array.each(events.callbacks, function (callback) {
                    if (!callback.disposed) {
                        callback.callback.call(callback.context, event);
                    }
                });
            });
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "_dispatchEvent", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, target, event) {
            if (!this.interactionsEnabled) {
                return false;
            }
            var dispatched = false;
            this._withEvents(key, function (events) {
                $array.each(events.callbacks, function (callback) {
                    if (!callback.disposed && callback.object === target) {
                        callback.callback.call(callback.context, event);
                        dispatched = true;
                    }
                });
            });
            return dispatched;
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "_dispatchMousedown", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (originalEvent) {
            var _this = this;
            var button = originalEvent.button;
            if (button != 0 && button != 2 && button != 1 && button !== undefined) {
                // Ignore non-primary mouse buttons
                return;
            }
            var event = this.getEvent(originalEvent);
            var target = this._getHitTarget(event.point, event.bbox);
            if (target) {
                var id_1 = event.id;
                var dragged_1 = false;
                eachTargets(target, function (obj) {
                    var info = { id: id_1, value: obj };
                    _this._mousedown.push(info);
                    if (!dragged_1 && _this._dispatchEvent("pointerdown", obj, event)) {
                        // Only dispatch the first element which matches
                        dragged_1 = true;
                        var has = _this._dragging.some(function (x) {
                            return x.value === obj && x.id === id_1;
                        });
                        if (!has) {
                            _this._dragging.push(info);
                        }
                    }
                    return true;
                });
            }
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "_dispatchGlobalMousemove", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (originalEvent, native) {
            var _this = this;
            var event = this.getEvent(originalEvent);
            var target = this._getHitTarget(event.point, event.bbox);
            event.native = native;
            if (target) {
                this._hovering.forEach(function (obj) {
                    if (!obj.contains(target)) {
                        _this._hovering.delete(obj);
                        if (obj.cursorOverStyle) {
                            $utils.setStyle(document.body, "cursor", obj._replacedCursorStyle);
                        }
                        _this._dispatchEvent("pointerout", obj, event);
                    }
                });
                if (event.native) {
                    eachTargets(target, function (obj) {
                        if (!_this._hovering.has(obj)) {
                            _this._hovering.add(obj);
                            if (obj.cursorOverStyle) {
                                obj._replacedCursorStyle = $utils.getStyle(document.body, "cursor");
                                $utils.setStyle(document.body, "cursor", obj.cursorOverStyle);
                            }
                            _this._dispatchEvent("pointerover", obj, event);
                        }
                        return true;
                    });
                }
                //} else if (target === false) {
            }
            else {
                this._hovering.forEach(function (obj) {
                    if (obj.cursorOverStyle) {
                        $utils.setStyle(document.body, "cursor", obj._replacedCursorStyle);
                    }
                    _this._dispatchEvent("pointerout", obj, event);
                });
                this._hovering.clear();
            }
            this._dispatchEventAll("globalpointermove", event);
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "_dispatchGlobalMouseup", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (originalEvent, native) {
            var event = this.getEvent(originalEvent);
            event.native = native;
            //const target = this._getHitTarget(event.point);
            this._dispatchEventAll("globalpointerup", event);
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "_dispatchDragMove", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (originalEvent) {
            var _this = this;
            if (this._dragging.length !== 0) {
                var event_1 = this.getEvent(originalEvent);
                var id_2 = event_1.id;
                this._dragging.forEach(function (obj) {
                    if (obj.id === id_2) {
                        _this._dispatchEvent("pointermove", obj.value, event_1);
                    }
                });
            }
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "_dispatchDragEnd", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (originalEvent) {
            var _this = this;
            var button = originalEvent.button;
            var clickevent;
            if (button == 0 || button === undefined) {
                clickevent = "click";
            }
            else if (button == 2) {
                clickevent = "rightclick";
            }
            else if (button == 1) {
                clickevent = "middleclick";
            }
            else {
                // Ignore non-primary mouse buttons
                return;
            }
            var event = this.getEvent(originalEvent);
            var id = event.id;
            if (this._mousedown.length !== 0) {
                var target_1 = this._getHitTarget(event.point, event.bbox);
                if (target_1) {
                    this._mousedown.forEach(function (obj) {
                        if (obj.id === id && obj.value.contains(target_1)) {
                            _this._dispatchEvent(clickevent, obj.value, event);
                        }
                    });
                }
                this._mousedown.length = 0;
            }
            if (this._dragging.length !== 0) {
                this._dragging.forEach(function (obj) {
                    if (obj.id === id) {
                        _this._dispatchEvent("pointerup", obj.value, event);
                    }
                });
                this._dragging.length = 0;
            }
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "_dispatchDoubleClick", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (originalEvent) {
            var _this = this;
            var event = this.getEvent(originalEvent);
            var target = this._getHitTarget(event.point, event.bbox);
            if (target) {
                eachTargets(target, function (obj) {
                    if (_this._dispatchEvent("dblclick", obj, event)) {
                        return false;
                    }
                    else {
                        return true;
                    }
                });
            }
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "_dispatchWheel", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (originalEvent) {
            var _this = this;
            var event = this.getEvent(originalEvent);
            this._hovering.forEach(function (obj) {
                _this._dispatchEvent("wheel", obj, event);
            });
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "_makeSharedEvent", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key, f) {
            var _this = this;
            if (this._listeners[key] === undefined) {
                var listener_1 = f();
                this._listeners[key] = new CounterDisposer(function () {
                    delete _this._listeners[key];
                    listener_1.dispose();
                });
            }
            return this._listeners[key].increment();
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "_onPointerEvent", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (name, f) {
            var native = false;
            var timer = null;
            function clear() {
                timer = null;
                native = false;
            }
            return new MultiDisposer([
                new Disposer(function () {
                    if (timer !== null) {
                        clearTimeout(timer);
                    }
                    clear();
                }),
                $utils.addEventListener(this.view, $utils.getRendererEvent(name), function (_) {
                    native = true;
                    if (timer !== null) {
                        clearTimeout(timer);
                    }
                    timer = window.setTimeout(clear, 0);
                }),
                onPointerEvent(window, name, function (ev) {
                    if (timer !== null) {
                        clearTimeout(timer);
                        timer = null;
                    }
                    f(ev, native);
                    native = false;
                }),
            ]);
        }
    });
    // This ensures that only a single DOM event is added (e.g. only a single mousemove event listener)
    Object.defineProperty(CanvasRenderer.prototype, "_initEvent", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (key) {
            var _this = this;
            switch (key) {
                case "globalpointermove":
                case "pointerover":
                case "pointerout":
                    return this._makeSharedEvent("pointermove", function () {
                        //const throttler = new Throttler();
                        // TODO handle throttling properly for multitouch
                        return _this._onPointerEvent("pointermove", function (event, native) {
                            _this._lastPointerMoveEvent = { event: event, native: native };
                            _this._mouseMoveThrottler.run();
                            //throttler.throttle(() => {
                            //});
                        });
                    });
                case "globalpointerup":
                    return this._makeSharedEvent("pointerup", function () {
                        //const throttler = new Throttler();
                        // TODO handle throttling properly for multitouch
                        return _this._onPointerEvent("pointerup", function (event, native) {
                            //throttler.throttle(() => {
                            _this._dispatchGlobalMouseup(event, native);
                            _this._lastPointerMoveEvent = { event: event, native: native };
                            //});
                        });
                    });
                case "click":
                case "rightclick":
                case "middleclick":
                case "pointerdown":
                case "pointermove":
                case "pointerup":
                    return this._makeSharedEvent("pointerdown", function () {
                        //const throttler = new Throttler();
                        var mousedown = onPointerEvent(_this.view, "pointerdown", function (ev) {
                            _this._dispatchMousedown(ev);
                        });
                        // TODO handle throttling properly for multitouch
                        var mousemove = _this._onPointerEvent("pointermove", function (ev) {
                            //throttler.throttle(() => {
                            _this._dispatchDragMove(ev);
                            //});
                        });
                        var mouseup = _this._onPointerEvent("pointerup", function (ev) {
                            _this._dispatchDragEnd(ev);
                        });
                        return new Disposer(function () {
                            mousedown.dispose();
                            mousemove.dispose();
                            mouseup.dispose();
                        });
                    });
                case "dblclick":
                    return this._makeSharedEvent("dblclick", function () {
                        return _this._onPointerEvent("dblclick", function (ev) {
                            _this._dispatchDoubleClick(ev);
                        });
                    });
                case "wheel":
                    return this._makeSharedEvent("wheel", function () {
                        return $utils.addEventListener(window, $utils.getRendererEvent("wheel"), function (event) {
                            _this._dispatchWheel(event);
                        }, { passive: false });
                    });
            }
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "_addEvent", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (object, key, callback, context) {
            var _this = this;
            var events = this._events[key];
            if (events === undefined) {
                events = this._events[key] = {
                    disposer: this._initEvent(key),
                    callbacks: [],
                    dispatching: false,
                    cleanup: false,
                };
            }
            var listener = { object: object, context: context, callback: callback, disposed: false };
            events.callbacks.push(listener);
            return new Disposer(function () {
                listener.disposed = true;
                if (events.dispatching) {
                    events.cleanup = true;
                }
                else {
                    $array.removeFirst(events.callbacks, listener);
                    if (events.callbacks.length === 0) {
                        events.disposer.dispose();
                        delete _this._events[key];
                    }
                }
            });
        }
    });
    Object.defineProperty(CanvasRenderer.prototype, "getCanvas", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (root, options) {
            // Make sure everything is rendered
            this.render(root);
            if (!options) {
                options = {};
            }
            var scale = this.resolution;
            // Check if we need to scale
            if (options.minWidth && (options.minWidth > this._width)) {
                var minScale = options.minWidth / this._width;
                if (minScale > scale) {
                    scale = minScale * this.resolution;
                }
            }
            if (options.minHeight && (options.minHeight > this._height)) {
                var minScale = options.minHeight / this._height;
                if (minScale > scale) {
                    scale = minScale * this.resolution;
                }
            }
            if (options.maxWidth && (options.maxWidth < this._width)) {
                var maxScale = options.maxWidth / this._width;
                if (maxScale < scale) {
                    scale = maxScale * this.resolution;
                }
            }
            if (options.maxHeight && (options.maxHeight > this._height)) {
                var maxScale = options.maxHeight / this._height;
                if (maxScale < scale) {
                    scale = maxScale * this.resolution;
                }
            }
            // Check if we need to compensate for pixel ratio
            if (options.maintainPixelRatio) {
                scale /= this.resolution;
            }
            // Set up new canvas for export
            var forceRender = false;
            var canvasWidth = this._width;
            var canvasHeight = this._height;
            var canvas = document.createElement("canvas");
            if (scale != this.resolution) {
                forceRender = true;
                canvasWidth = this._width * scale / this.resolution;
                canvasHeight = this._height * scale / this.resolution;
            }
            canvas.width = canvasWidth;
            canvas.height = canvasHeight;
            // Context
            var context = canvas.getContext("2d");
            var width = 0;
            var height = 0;
            var needRerender = false;
            $array.each(this.layers, function (layer) {
                if (layer && layer.visible) {
                    if (layer.tainted || forceRender) {
                        needRerender = true;
                        layer.exportableView = layer.view;
                        layer.exportableContext = layer.context;
                        layer.view = document.createElement("canvas");
                        layer.view.width = canvasWidth;
                        layer.view.height = canvasHeight;
                        layer.context = layer.view.getContext("2d");
                        layer.dirty = true;
                        layer.scale = scale;
                    }
                }
            });
            if (needRerender) {
                this._omitTainted = true;
                this.render(root);
                this._omitTainted = false;
            }
            $array.each(this.layers, function (layer) {
                if (layer && layer.visible) {
                    // Layer is fine. Just plop it into our target canvas
                    context.drawImage(layer.view, 0, 0);
                    // Restore layer original canvas
                    if (layer.exportableView) {
                        layer.view = layer.exportableView;
                        layer.exportableView = undefined;
                    }
                    if (layer.exportableContext) {
                        layer.context = layer.exportableContext;
                        layer.exportableContext = undefined;
                    }
                    if (width < layer.view.clientWidth) {
                        width = layer.view.clientWidth;
                    }
                    if (height < layer.view.clientHeight) {
                        height = layer.view.clientHeight;
                    }
                    layer.scale = undefined;
                }
            });
            canvas.style.width = width + "px";
            canvas.style.height = height + "px";
            return canvas;
        }
    });
    return CanvasRenderer;
}(ArrayDisposer));
export { CanvasRenderer };
//# sourceMappingURL=CanvasRenderer.js.map