import * as $type from "./Type";
import * as $object from "./Object";
/**
 * Tool to parse JSON string into structured data.
 *
 * @see {@link https://www.amcharts.com/docs/v5/concepts/data/#Parsing} for more info
 * @important
 */
var JSONParser = /** @class */ (function () {
    function JSONParser() {
    }
    /**
     * Parses JSON string.
     *
     * @param   input    JSON
     * @param   options  Options
     * @return           Data
     */
    Object.defineProperty(JSONParser, "parse", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (input, options) {
            options = this._applyDefaults(options);
            try {
                if ($type.isString(input)) {
                    var data = JSON.parse(input);
                    if (options.reverse && $type.isArray(data)) {
                        data.reverse();
                    }
                    return data;
                }
                else if ($type.isArray(input) || $type.isObject(input)) {
                    return input;
                }
                else {
                    throw ("Unable to parse JSON data");
                }
            }
            catch (e) {
                return undefined;
            }
        }
    });
    Object.defineProperty(JSONParser, "_applyDefaults", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (options) {
            var normalized = {};
            var defaults = {
                reverse: false
            };
            if (!options) {
                options = {};
            }
            $object.each(defaults, function (key, val) {
                normalized[key] = options[key] || val;
            });
            return normalized;
        }
    });
    return JSONParser;
}());
export { JSONParser };
/**
 * Tool to parse JSON string into structured data.
 *
 * @see {@link https://www.amcharts.com/docs/v5/concepts/data/#Parsing} for more info
 * @important
 */
var CSVParser = /** @class */ (function () {
    function CSVParser() {
    }
    /**
     * Parses CSV string.
     *
     * @param   input    CSV
     * @param   options  Options
     * @return           Data
     */
    Object.defineProperty(CSVParser, "parse", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (input, options) {
            options = this._applyDefaults(options);
            // Get CSV data as array
            var data = this.CSVToArray(input, options.delimiter);
            // Init resuling array
            var res = [], cols = [], col, i;
            // Skip rows
            for (i = 0; i < options.skipRows; i++) {
                data.shift();
            }
            // First row holds column names?
            if (options.useColumnNames) {
                cols = data.shift();
                // Normalize column names
                for (var x = 0; x < cols.length; x++) {
                    // trim
                    col = cols[x] != null ? cols[x].replace(/^\s+|\s+$/gm, "") : "";
                    // Check for empty
                    if ("" === col) {
                        col = "col" + x;
                    }
                    cols[x] = col;
                }
            }
            // Iterate through the result set
            var row;
            while (true) {
                row = options.reverse ? data.pop() : data.shift();
                if (!row) {
                    break;
                }
                if (options.skipEmpty && row.length === 1 && row[0] === "") {
                    continue;
                }
                var dataPoint = {};
                for (i = 0; i < row.length; i++) {
                    col = undefined === cols[i] ? "col" + i : cols[i];
                    dataPoint[col] = row[i];
                }
                res.push(dataPoint);
            }
            return res;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(CSVParser, "CSVToArray", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (data, delimiter) {
            // Check to see if the delimiter is defined. If not,
            // then default to comma.
            delimiter = (delimiter || ',');
            // Create a regular expression to parse the CSV values.
            var objPattern = new RegExp((
            // Delimiters.
            "(\\" + delimiter + "|\\r?\\n|\\r|^)" +
                // Quoted fields.
                "(?:\"([^\"]*(?:\"\"[^\"]*)*)\"|" +
                // Standard fields.
                "([^\"\\" + delimiter + "\\r\\n]*))"), "gi");
            // Create an array to hold our data. Give the array
            // a default empty first row.
            var arrData = [
                []
            ];
            // Create an array to hold our individual pattern
            // matching groups.
            var arrMatches = null;
            // Keep looping over the regular expression matches
            // until we can no longer find a match.
            while (true) {
                arrMatches = objPattern.exec(data);
                if (!arrMatches) {
                    break;
                }
                // Get the delimiter that was found.
                var strMatchedDelimiter = arrMatches[1];
                // Check to see if the given delimiter has a length
                // (is not the start of string) and if it matches
                // field delimiter. If id does not, then we know
                // that this delimiter is a row delimiter.
                if (strMatchedDelimiter.length &&
                    (strMatchedDelimiter !== delimiter)) {
                    // Since we have reached a new row of data,
                    // add an empty row to our data array.
                    arrData.push([]);
                }
                // Now that we have our delimiter out of the way,
                // let's check to see which kind of value we
                // captured (quoted or unquoted).
                var strMatchedValue = void 0;
                if (arrMatches[2]) {
                    // We found a quoted value. When we capture
                    // this value, unescape any double quotes.
                    strMatchedValue = arrMatches[2].replace(new RegExp("\"\"", "g"), "\"");
                }
                else {
                    // We found a non-quoted value.
                    strMatchedValue = arrMatches[3];
                }
                // Now that we have our value string, let's add
                // it to the data array.
                arrData[arrData.length - 1].push(strMatchedValue);
            }
            // Return the parsed data.
            return (arrData);
        }
    });
    Object.defineProperty(CSVParser, "_applyDefaults", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (options) {
            var normalized = {};
            var defaults = {
                delimiter: ",",
                reverse: false,
                skipRows: 0,
                skipEmpty: true,
                useColumnNames: false
            };
            if (!options) {
                options = {};
            }
            $object.each(defaults, function (key, val) {
                normalized[key] = options[key] || val;
            });
            return normalized;
        }
    });
    return CSVParser;
}());
export { CSVParser };
//# sourceMappingURL=DataParser.js.map