import { __extends } from "tslib";
import { Entity } from "./Entity";
import { Color } from "./Color";
import * as $type from "./Type";
import * as $object from "./Object";
import * as $array from "./Array";
/**
 * A tool that can process the data before it is being used in charts.
 *
 * @see {@link https://www.amcharts.com/docs/v5/concepts/data/#Pre_processing_data} for more info
 * @important
 */
var DataProcessor = /** @class */ (function (_super) {
    __extends(DataProcessor, _super);
    function DataProcessor() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "_checkDates", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(_this, "_checkNumbers", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(_this, "_checkColors", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(_this, "_checkEmpty", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(_this, "_checkDeep", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        return _this;
    }
    Object.defineProperty(DataProcessor.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            _super.prototype._afterNew.call(this);
            this._checkFeatures();
            this.on("dateFields", function () { return _this._checkFeatures(); });
            this.on("dateFormat", function () { return _this._checkFeatures(); });
            this.on("numericFields", function () { return _this._checkFeatures(); });
            this.on("colorFields", function () { return _this._checkFeatures(); });
            this.on("emptyAs", function () { return _this._checkFeatures(); });
        }
    });
    Object.defineProperty(DataProcessor.prototype, "_checkFeatures", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (this.isDirty("dateFields") || this.isDirty("dateFormat")) {
                this._checkDates = this.get("dateFields") && (this.get("dateFields").length > 0);
            }
            if (this.isDirty("numericFields")) {
                this._checkNumbers = this.get("numericFields") && (this.get("numericFields").length > 0);
            }
            if (this.isDirty("colorFields")) {
                this._checkColors = this.get("colorFields") && (this.get("colorFields").length > 0);
            }
            if (this.isDirty("emptyAs")) {
                this._checkEmpty = this.get("emptyAs") != null;
            }
            this._checkDeepFeatures();
        }
    });
    Object.defineProperty(DataProcessor.prototype, "_checkDeepFeatures", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            var deepFields = [];
            $array.each(["dateFields", "numericFields", "colorFields"], function (where) {
                $array.each(_this.get(where, []), function (field) {
                    var steps = field.split(".");
                    steps.pop();
                    while (steps.length > 0) {
                        deepFields.push(steps.join("."));
                        steps.pop();
                    }
                });
            });
            this._checkDeep = deepFields.length > 0;
            this.setPrivate("deepFields", deepFields);
        }
    });
    /**
     * Processess entire array of data.
     *
     * NOTE: calling this will modify original array!
     */
    Object.defineProperty(DataProcessor.prototype, "processMany", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (data) {
            var _this = this;
            if ($type.isArray(data) && (this._checkDates || this._checkNumbers || this._checkColors || this._checkEmpty)) {
                $array.each(data, function (row) {
                    _this.processRow(row);
                });
            }
        }
    });
    /**
     * Processes a row (object) of data.
     *
     * NOTE: calling this will modify values of the original object!
     */
    Object.defineProperty(DataProcessor.prototype, "processRow", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (row, prefix) {
            var _this = this;
            if (prefix === void 0) { prefix = ""; }
            $object.each(row, function (key, _value) {
                var lookupKey = prefix + key;
                if (_this._checkEmpty) {
                    row[key] = _this._maybeToEmpty(row[key]);
                }
                if (_this._checkNumbers) {
                    row[key] = _this._maybeToNumber(lookupKey, row[key]);
                }
                if (_this._checkDates) {
                    row[key] = _this._maybeToDate(lookupKey, row[key]);
                }
                if (_this._checkColors) {
                    row[key] = _this._maybeToColor(lookupKey, row[key]);
                }
                if (_this._checkDeep && _this.getPrivate("deepFields", []).indexOf(lookupKey) !== -1 && $type.isObject(row[key])) {
                    _this.processRow(row[key], lookupKey + ".");
                }
            });
        }
    });
    Object.defineProperty(DataProcessor.prototype, "_maybeToNumber", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (field, value) {
            if (this.get("numericFields").indexOf(field) !== -1) {
                return $type.toNumber(value);
            }
            return value;
        }
    });
    Object.defineProperty(DataProcessor.prototype, "_maybeToDate", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (field, value) {
            if (this.get("dateFields").indexOf(field) !== -1) {
                return this._root.dateFormatter.parse(value, this.get("dateFormat", "")).getTime();
            }
            return value;
        }
    });
    Object.defineProperty(DataProcessor.prototype, "_maybeToEmpty", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (value) {
            if ((value == null || value == "") && this.get("emptyAs") != null) {
                return this.get("emptyAs");
            }
            return value;
        }
    });
    Object.defineProperty(DataProcessor.prototype, "_maybeToColor", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (field, value) {
            if (this.get("colorFields").indexOf(field) !== -1) {
                return Color.fromAny(value);
            }
            return value;
        }
    });
    return DataProcessor;
}(Entity));
export { DataProcessor };
//# sourceMappingURL=DataProcessor.js.map