import { __extends } from "tslib";
import { Entity } from "../../core/util/Entity";
import { Disposer } from "../../core/util/Disposer";
import exportingCSS from "./ExportingCSS";
import * as $array from "../../core/util/Array";
import * as $utils from "../../core/util/Utils";
/**
 * Displays a menu for [[Exporting]].
 *
 * @see {@link https://www.amcharts.com/docs/v5/concepts/exporting/} for more info
 */
var ExportingMenu = /** @class */ (function (_super) {
    __extends(ExportingMenu, _super);
    function ExportingMenu() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        Object.defineProperty(_this, "_menuElement", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_iconElement", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_listElement", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_itemElements", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        Object.defineProperty(_this, "_itemDisposers", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        Object.defineProperty(_this, "_cssDisposer", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "_activeItem", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(_this, "isOpen", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(_this, "_isOver", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        return _this;
    }
    Object.defineProperty(ExportingMenu.prototype, "_afterNew", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            _super.prototype._afterNew.call(this);
            this._setRawDefault("container", this._root._inner);
            this._setRawDefault("align", "right");
            this._setRawDefault("valign", "top");
            this._setRawDefault("useDefaultCSS", true);
            this._setRawDefault("autoClose", true);
            this._setRawDefault("deactivateRoot", true);
            this._setRawDefault("items", [{
                    type: "separator",
                    label: this._t("Export")
                }, {
                    type: "format",
                    format: "png",
                    exportType: "image",
                    label: this._t("PNG"),
                    sublabel: this._t("Image")
                }, {
                    type: "format",
                    format: "jpg",
                    exportType: "image",
                    label: this._t("JPG"),
                    sublabel: this._t("Image")
                }, {
                    type: "format",
                    format: "pdf",
                    exportType: "image",
                    label: this._t("PDF"),
                    sublabel: this._t("Image")
                }, {
                    type: "separator",
                    exportType: "data",
                    //label: this._t("Data")
                }, {
                    type: "format",
                    format: "json",
                    exportType: "data",
                    label: this._t("JSON"),
                    sublabel: this._t("Data")
                }, {
                    type: "format",
                    format: "csv",
                    exportType: "data",
                    label: this._t("CSV"),
                    sublabel: this._t("Data")
                }, {
                    type: "format",
                    format: "xlsx",
                    exportType: "data",
                    label: this._t("XLSX"),
                    sublabel: this._t("Data")
                }, {
                    type: "format",
                    format: "pdfdata",
                    exportType: "data",
                    label: this._t("PDF"),
                    sublabel: this._t("Data")
                }, {
                    type: "format",
                    format: "html",
                    exportType: "data",
                    label: this._t("HTML"),
                    sublabel: this._t("Data")
                }, {
                    type: "separator"
                }, {
                    type: "format",
                    format: "print",
                    exportType: "print",
                    label: this._t("Print")
                }]);
            var menuElement = document.createElement("div");
            this._menuElement = menuElement;
            var iconElement = document.createElement("a");
            this._iconElement = iconElement;
            this._listElement = document.createElement("ul");
            this._listElement.setAttribute("role", "menu");
            this._applyClassNames();
            iconElement.innerHTML = '<svg fill="none" height="20" width="20" xmlns="http://www.w3.org/2000/svg"><path d="M6 10a2 2 0 11-4 0 2 2 0 014 0zM12 10a2 2 0 11-4 0 2 2 0 014 0zM16 12a2 2 0 100-4 2 2 0 000 4z"/></svg>';
            iconElement.setAttribute("tabindex", this._root.tabindex.toString());
            iconElement.setAttribute("aria-label", this._t("Export"));
            iconElement.setAttribute("aria-description", this._t("Press ENTER to open"));
            if ($utils.supports("keyboardevents")) {
                this._disposers.push($utils.addEventListener(document, "keydown", function (ev) {
                    if (document.activeElement == _this._iconElement || _this.isOpen) {
                        if (ev.keyCode == 27) {
                            // ESC
                            _this.close();
                        }
                        else if (ev.keyCode == 13) {
                            // ENTER
                            if (_this._activeItem) {
                                _this._handleClick(_this._activeItem);
                            }
                            else {
                                _this.toggle();
                            }
                        }
                        else if (ev.keyCode == 38 || ev.keyCode == 40) {
                            var items = _this.get("items", []);
                            var currentIndex = items.indexOf(_this._activeItem);
                            if (_this.get("valign") == "top" && currentIndex == -1) {
                                currentIndex = items.length;
                            }
                            var dir = ev.keyCode == 38 ? -1 : 1;
                            var newIndex = currentIndex + dir;
                            var newItem = void 0;
                            do {
                                if (newIndex < 0) {
                                    newIndex = items.length - 1;
                                }
                                else if (newIndex > (items.length - 1)) {
                                    newIndex = 0;
                                }
                                if (items[newIndex].type == "separator") {
                                    newIndex += dir;
                                }
                                else {
                                    newItem = items[newIndex];
                                }
                            } while (!newItem);
                            if (newItem) {
                                _this._handleItemFocus(newItem);
                            }
                        }
                    }
                }));
            }
            this._disposers.push($utils.addEventListener(iconElement, "click", function (ev) {
                ev.stopImmediatePropagation();
                _this.toggle();
            }));
            menuElement.appendChild(this._iconElement);
            menuElement.appendChild(this._listElement);
            this._root._inner.appendChild(this._menuElement);
            this._disposers.push($utils.addEventListener(menuElement, $utils.getRendererEvent("pointerover"), function (_ev) {
                _this._isOver = true;
                if (_this.get("deactivateRoot")) {
                    _this._root._renderer.interactionsEnabled = false;
                }
            }));
            this._disposers.push($utils.addEventListener(menuElement, $utils.getRendererEvent("pointerout"), function (_ev) {
                _this._isOver = false;
                if (_this.get("deactivateRoot")) {
                    _this._root._renderer.interactionsEnabled = true;
                }
            }));
            this._disposers.push(new Disposer(function () {
                if (_this._menuElement) {
                    _this._root._inner.removeChild(_this._menuElement);
                }
            }));
            this._disposers.push($utils.addEventListener(document, "click", function (_ev) {
                if (_this.isOpen && !_this._isOver) {
                    _this.close();
                }
            }));
            this.loadDefaultCSS();
            this._root.addDisposer(this);
            this.events.dispatch("menucreated", {
                type: "menucreated",
                target: this
            });
        }
    });
    Object.defineProperty(ExportingMenu.prototype, "_afterChanged", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._afterChanged.call(this);
            if (this._itemElements.length == 0) {
                this.createItems();
            }
            if (this.isDirty("useDefaultCSS")) {
                if (this.get("useDefaultCSS")) {
                    this.loadDefaultCSS();
                }
                else if (this._cssDisposer) {
                    this._cssDisposer.dispose();
                }
            }
            if (this.isDirty("exporting")) {
                var exporting = this.get("exporting");
                if (exporting) {
                    this.createItems();
                }
            }
            if (this.isDirty("align") || this.isDirty("valign")) {
                this._applyClassNames();
            }
            if (this.isDirty("container")) {
                var container = this.get("container");
                if (container) {
                    container.appendChild(this._menuElement);
                }
            }
        }
    });
    Object.defineProperty(ExportingMenu.prototype, "_dispose", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            _super.prototype._dispose.call(this);
            $array.each(this._itemDisposers, function (x) {
                x.dispose();
            });
        }
    });
    Object.defineProperty(ExportingMenu.prototype, "_applyClassNames", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var align = this.get("align", "right");
            var valign = this.get("valign", "top");
            var status = this.isOpen ? "am5exporting-menu-open" : "am5exporting-menu-closed";
            this._menuElement.className = "am5exporting am5exporting-menu am5exporting-align-" + align + " am5exporting-valign-" + valign + " " + status;
            this._iconElement.className = "am5exporting am5exporting-icon am5exporting-align-" + align + " am5exporting-valign-" + valign;
            this._listElement.className = "am5exporting am5exporting-list am5exporting-align-" + align + " am5exporting-valign-" + valign;
        }
    });
    /**
     * @ignore
     */
    Object.defineProperty(ExportingMenu.prototype, "createItems", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var _this = this;
            var exporting = this.get("exporting");
            if (!exporting) {
                return;
            }
            this._listElement.innerHTML = "";
            this._itemElements = [];
            var items = this.get("items", []);
            var supportedFormats = exporting.supportedFormats();
            var supportedExportTypes = exporting.supportedExportTypes();
            $array.each(this._itemDisposers, function (x) {
                x.dispose();
            });
            this._itemDisposers.length = 0;
            $array.each(items, function (item) {
                if (item.format && supportedFormats.indexOf(item.format) == -1) {
                    return;
                }
                if (item.exportType && supportedExportTypes.indexOf(item.exportType) == -1) {
                    return;
                }
                var li = document.createElement("li");
                li.setAttribute("role", "menuitem");
                li.className = "am5exporting am5exporting-item am5exporting-type-" + item.type;
                if (item.format) {
                    li.className += " am5exporting-format-" + item.format;
                }
                var a = document.createElement("a");
                var ariaLabel = _this._t("Export");
                if (item.label) {
                    a.innerHTML = item.label;
                    ariaLabel += " " + item.label;
                }
                if (item.sublabel) {
                    a.innerHTML += " <span class=\"am5exporting-label-alt\">" + item.sublabel + "</span>";
                    ariaLabel += " (" + item.sublabel + ")";
                }
                if (item.callback) {
                    _this._itemDisposers.push($utils.addEventListener(a, "click", function (_ev) {
                        item.callback.call(item.callbackTarget || _this);
                    }));
                    a.setAttribute("tabindex", _this._root.tabindex.toString());
                }
                else if (item.format && exporting) {
                    _this._itemDisposers.push($utils.addEventListener(a, "click", function (_ev) {
                        _this._handleClick(item);
                    }));
                    _this._itemDisposers.push($utils.addEventListener(a, "focus", function (_ev) {
                        _this._handleItemFocus(item);
                    }));
                    _this._itemDisposers.push($utils.addEventListener(a, "blur", function (_ev) {
                        _this._handleItemBlur(item);
                    }));
                    a.setAttribute("tabindex", _this._root.tabindex.toString());
                    a.setAttribute("aria-label", ariaLabel);
                }
                item.element = a;
                li.appendChild(a);
                _this._listElement.appendChild(li);
                _this._itemElements.push(li);
            });
        }
    });
    Object.defineProperty(ExportingMenu.prototype, "_handleClick", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (item) {
            var exporting = this.get("exporting");
            if (this.get("autoClose")) {
                this.close();
            }
            if (item.format == "print") {
                exporting.print();
            }
            else {
                exporting.download(item.format);
            }
        }
    });
    Object.defineProperty(ExportingMenu.prototype, "_handleItemFocus", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (item) {
            if (item != this._activeItem) {
                if (this._activeItem) {
                    this._activeItem.element.className = "";
                }
                this._activeItem = item;
                item.element.className = "am5exporting-item-active";
                item.element.focus();
            }
        }
    });
    Object.defineProperty(ExportingMenu.prototype, "_handleItemBlur", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function (item) {
            var _this = this;
            item.element.className = "";
            if (item == this._activeItem) {
                this._activeItem = undefined;
            }
            this.setTimeout(function () {
                if (!document.activeElement || !$utils.contains(_this.get("container"), document.activeElement)) {
                    _this.close();
                }
            }, 10);
        }
    });
    /**
     * Loads the default CSS.
     *
     * @ignore Exclude from docs
     */
    Object.defineProperty(ExportingMenu.prototype, "loadDefaultCSS", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            var disposer = exportingCSS($utils.getShadowRoot(this._root.dom), this._root);
            this._disposers.push(disposer);
            this._cssDisposer = disposer;
            // if (this._element) {
            // 	this._element.style.display = "";
            // }
        }
    });
    /**
     * Opens menu.
     */
    Object.defineProperty(ExportingMenu.prototype, "open", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this.isOpen = true;
            this._applyClassNames();
            this.events.dispatch("menuopened", {
                type: "menuopened",
                target: this
            });
        }
    });
    /**
     * Closes menu.
     */
    Object.defineProperty(ExportingMenu.prototype, "close", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            this.isOpen = false;
            $utils.blur();
            this._applyClassNames();
            this.events.dispatch("menuclosed", {
                type: "menuclosed",
                target: this
            });
        }
    });
    /**
     * Toggles menu open and close.
     */
    Object.defineProperty(ExportingMenu.prototype, "toggle", {
        enumerable: false,
        configurable: true,
        writable: true,
        value: function () {
            if (this.isOpen) {
                this.close();
            }
            else {
                this.open();
            }
        }
    });
    Object.defineProperty(ExportingMenu, "className", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: "ExportingMenu"
    });
    Object.defineProperty(ExportingMenu, "classNames", {
        enumerable: true,
        configurable: true,
        writable: true,
        value: Entity.classNames.concat([ExportingMenu.className])
    });
    return ExportingMenu;
}(Entity));
export { ExportingMenu };
//# sourceMappingURL=ExportingMenu.js.map