/*! Atlassian UI and the Atlassian Design Guidelines are created by Atlassian. See https://developer.atlassian.com/display/AUI/ for API documentation and https://developer.atlassian.com/design/ for license details. */

/**
 * A collection of Atlassian JavaScript UI components.
 *
 * AUI components/functions should be assumed Private unless included in the API documentation at http://developer.atlassian.com/display/AUI
 *
 * @module AJS
 * @requires jQuery
 */
(function () {

    'use strict';

    if (!window.jQuery && !window.Zepto) {
        throw new Error('either jQuery or Zepto is required for AJS to function.');
    }

    /**
     * AJS contains utility methods, used by various components. It also provides the namespacing for all AUI components.
     *
     * @class AJS
     * @requires jQuery
     */
    window.AJS = (function () {
        var included = [];
        var uniqueID;
        var uniqueIDstring;
        var uniqueIDcounter = 0;

        function escapeHtmlReplacement(str) {
            var special = {
                '<': '&lt;',
                '>': '&gt;',
                '&': '&amp;',
                '\'': '&#39;',
                '`': '&#96;'
            };

            if (typeof special[str] === 'string') {
                return special[str];
            }

            return '&quot;';
        }

        var ESCAPE_HTML_SPECIAL_CHARS = /[&"'<>`]/g;
        var res = {
            /**
             * Version number to allow for rough backwards compatibility via conditionals
             * NOTE: Don't change. Generated by the Maven at build.
             * @property version
             */
            version: '${project.version}',

            /**
             * Parameters are loaded from the DOM on page load.
             * @property params
             */
            params: {},

            /**
             * Returns an HTMLElement reference.
             * @method $
             * @param {String | HTMLElement |Array} el Accepts a string to use as an ID for getting a DOM reference, an actual DOM reference, or an Array of IDs and/or HTMLElements.
             * @return {HTMLElement | Array} A DOM reference to an HTML element or an array of HTMLElements.
             */
            $: window.jQuery || window.Zepto,

            /**
             * Calls e.preventDefault. This is designed for event handlers that only need to prevent the default browser
             * action, eg:
             * AJS.$(".my-class").click(AJS.preventDefault)
             * @param e jQuery event
             */
            preventDefault: function (e) {
                e.preventDefault();
            },

            /**
             * Prevent further handling of an event. Returns false, which you should use as the return value of your event handler:
             * return AJS.stopEvent(e);
             * @param e jQuery event
             * @deprecated use AJS.preventDefault() instead
             */
            stopEvent: function (e) {
                e.stopPropagation();
                return false; // required for JWebUnit pop-up links to work properly
            },

            include: function (url) {
                if (!this.contains(included, url)) {
                    included.push(url);
                    var s = document.createElement('script');
                    s.src = url;
                    this.$('body').append(s);
                }
            },

            /**
             * Shortcut function to toggle class name of an element.
             * @method toggleClassName
             * @param {String | HTMLElement} element The HTMLElement or an ID to toggle class name on.
             * @param {String} className The class name to remove or add.
             */
            toggleClassName: function (element, className) {
                if (!(element = this.$(element))) {
                    return;
                }

                element.toggleClass(className);
            },

            /**
             * Shortcut function adds or removes 'hidden' classname to an element based on a passed boolean.
             * @method setVisible
             * @param {String | HTMLElement} element The HTMLElement or an ID to show or hide.
             * @param {boolean} show true to show, false to hide
             */
            setVisible: function (element, show) {
                if (!(element = this.$(element))) {
                    return;
                }
                // aliased for use inside function below
                var $ = this.$;

                $(element).each(function () {
                    var isHidden = $(this).hasClass('hidden');

                    if (isHidden && show) {
                        $(this).removeClass('hidden');
                    } else if (!isHidden && !show) {
                        $(this).addClass('hidden');
                    }
                });
            },

            /**
             * Shortcut function adds or removes 'current' classname to an element based on a passed boolean.
             * @param {String | HTMLElement} element The HTMLElement or an ID to show or hide.
             * @param {boolean} show true to add 'current' class, false to remove
             */
            setCurrent: function (element, current) {
                if (!(element = this.$(element))) {
                    return;
                }

                if (current) {
                    element.addClass('current');
                }
                else {
                    element.removeClass('current');
                }
            },

            /**
             * Shortcut function to see if passed element is currently visible on screen.
             * @method isVisible
             * @param {String | HTMLElement} element The HTMLElement or an jQuery selector to check.
             */
            isVisible: function (element) {
                return !this.$(element).hasClass('hidden');
            },

            /**
             * Shortcut function to see if passed element is truncated/clipped, eg. with text-overflow: ellipsis
             * @method isClipped
             * @param {String | HTMLElement} element The HTMLElement or an jQuery selector to check.
             */
            isClipped: function (el) {
                el = AJS.$(el);
                return (el.prop('scrollWidth') > el.prop('clientWidth'));
            },

            /**
             * Find parameters in the DOM and store them in the provided object, or the ajs.params object if parameter is not present.
             */
            populateParameters: function (parameters) {
                if (!parameters) {
                    parameters = this.params;
                }

                var ajs = this;

                this.$('.parameters input').each(function () {
                    var value = this.value,
                        id = this.title || this.id;
                    if (ajs.$(this).hasClass('list')) {
                        if (parameters[id]) {
                            parameters[id].push(value);
                        } else {
                            parameters[id] = [value];
                        }
                    } else {
                        parameters[id] = (value.match(/^(tru|fals)e$/i) ? value.toLowerCase() === 'true' : value);
                    }
                });
            },

            /**
             * Adds functions to the list of methods to be run on initialisation. Wraps
             * error handling around the provided function so its failure won't prevent
             * other init functions running.
             * @method toInit
             * @param {Function} func Function to be call on initialisation.
             * @return AJS object.
             */
            toInit: function (func) {
                var ajs = this;

                this.$(function () {
                    try {
                        func.apply(this, arguments);
                    } catch (ex) {
                        ajs.log('Failed to run init function: ' + ex + '\n' + func.toString());
                    }
                });

                return this;
            },

            /**
             * Finds the index of an element in the array.
             * @method indexOf
             * @param item Array element which will be searched.
             * @param fromIndex (optional) the index from which the item will be searched. Negative values will search from the
             * end of the array.
             * @return a zero based index of the element.
             */
            indexOf: function (array, item, fromIndex) {
                var length = array.length;

                if (!fromIndex) {
                    fromIndex = 0;
                } else if (fromIndex < 0) {
                    fromIndex = Math.max(0, length + fromIndex);
                }

                for (var i = fromIndex; i < length; i++) {
                    if (array[i] === item) {
                        return i;
                    }
                }

                return -1;
            },

            /**
             * Looks for an element inside the array.
             * @method contains
             * @param item Array element which will be searched.
             * @return {Boolean} Is element in array.
             */
            contains: function (array, item) {
                return this.indexOf(array, item) > -1;
            },

            /**
             * Includes firebug lite for debugging in IE. Especially in IE.
             * @method firebug
             * @usage Type in addressbar "javascript:alert(AJS.firebug());"
             * @deprecated
             */
            firebug: function () {
                // Deprecated in 5.1
                var script = this.$(document.createElement('script'));
                script.attr('src', 'https://getfirebug.com/releases/lite/1.2/firebug-lite-compressed.js');
                this.$('head').append(script);
                (function () {
                    if (window.firebug) {
                        firebug.init();
                    } else {
                        setTimeout(AJS.firebug, 0);
                    }
                })();
            },

            /**
             * Clones the element specified by the selector and removes the id attribute
             * @param selector a jQuery selector
             */
            clone : function (selector) {
                return AJS.$(selector).clone().removeAttr('id');
            },

            /**
             * Compare two strings in alphanumeric way
             * @method alphanum
             * @param {String} a first string to compare
             * @param {String} b second string to compare
             * @return {Number(-1|0|1)} -1 if a < b, 0 if a = b, 1 if a > b
             * @usage a.sort(AJS.alphanum)
             */
            alphanum: function (a, b) {
                a = (a + '').toLowerCase();
                b = (b + '').toLowerCase();

                var chunks = /(\d+|\D+)/g;
                var am = a.match(chunks);
                var bm = b.match(chunks);
                var len = Math.max(am.length, bm.length);

                for (var i = 0; i < len; i++) {
                    if (i === am.length) {
                        return -1;
                    }

                    if (i === bm.length) {
                        return 1;
                    }

                    var ad = parseInt(am[i], 10) + '';
                    var bd = parseInt(bm[i], 10) + '';

                    if (ad === am[i] && bd === bm[i] && ad !== bd) {
                        return (ad - bd) / Math.abs(ad - bd);
                    }

                    if ((ad !== am[i] || bd !== bm[i]) && am[i] !== bm[i]) {
                        return am[i] < bm[i] ? -1 : 1;
                    }
                }
                return 0;
            },

            onTextResize: function (f) {
                if (typeof f === 'function') {
                    if (AJS.onTextResize['on-text-resize']) {
                        AJS.onTextResize['on-text-resize'].push(function (emsize) {
                            f(emsize);
                        });
                    } else {
                        var em = AJS('div');

                        em.css({
                            width: '1em',
                            height: '1em',
                            position: 'absolute',
                            top: '-9999em',
                            left: '-9999em'
                        });

                        this.$('body').append(em);
                        em.size = em.width();

                        setInterval(function () {
                            if (em.size !== em.width()) {
                                em.size = em.width();

                                for (var i = 0, ii = AJS.onTextResize['on-text-resize'].length; i < ii; i++) {
                                    AJS.onTextResize['on-text-resize'][i](em.size);
                                }
                            }
                        }, 0);
                        AJS.onTextResize.em = em;
                        AJS.onTextResize['on-text-resize'] = [function (emsize) {
                            f(emsize);
                        }];
                    }
                }
            },

            unbindTextResize: function (f) {
                for (var i = 0, ii = AJS.onTextResize['on-text-resize'].length; i < ii; i++) {
                    if (AJS.onTextResize['on-text-resize'][i] === f) {
                        return AJS.onTextResize['on-text-resize'].splice(i, 1);
                    }
                }
            },

            /**
             * Similar to Javascript's in-built escape() function, but where the built-in escape()
             * might encode unicode charaters as %uHHHH, this function will leave them as-is.
             *
             * NOTE: this function does not do html-escaping, see AJS.escapeHtml()
             */
            escape: function (string) {
                return escape(string).replace(/%u\w{4}/gi, function (w) {
                    return unescape(w);
                });
            },

            /**
             * Sanitise a string for use with innerHTML or as an attribute.
             *
             * @param {String} str
             */
            escapeHtml: function (str) {
                return str.replace(ESCAPE_HTML_SPECIAL_CHARS, escapeHtmlReplacement);
            },

            /**
             * Filters a list of entries by a passed search term.
             *
             * Options :
             *   - "keywordsField" - name of entry field containing keywords, default "keywords"
             *   - "ignoreForCamelCase" - ignore search case for camel case, e.g. CB matches Code Block *and* Code block
             *   - "matchBoundary" - match words only at boundary, e.g. link matches "linking" but not "hyperlinks"
             *   - "splitRegex" - regex to split search words, instead of on whitespace
             *
             * @param entries an index array of objects with a "keywords" property
             * @param search one or more words to search on, which may include camel-casing.
             * @param options - optional - specifiy to override default behaviour
             */
            filterBySearch : function (entries, search, options) {
                // search for nothing, get nothing - up to calling code to handle.
                if (!search) {
                    return [];
                }

                var $ = this.$;
                var keywordsField = (options && options.keywordsField) || 'keywords';
                var camelCaseFlags = (options && options.ignoreForCamelCase) ? 'i' : '';
                var boundaryFlag  = (options && options.matchBoundary) ? '\\b' : '';
                var splitRegex = (options && options.splitRegex) || (/\s+/);

                // each word in the input is considered a distinct filter that has to match a keyword in the record
                var filterWords = search.split(splitRegex);
                var filters = [];

                filterWords.forEach(function(word) {
                    // anchor on word boundaries
                    var subfilters = [new RegExp(boundaryFlag + word, 'i')];

                    // split camel-case into separate words
                    if (/^([A-Z][a-z]*) {2,}$/.test(this)) {
                        var camelRegexStr = this.replace(/([A-Z][a-z]*)/g, '\\b$1[^,]*');

                        subfilters.push(new RegExp(camelRegexStr, camelCaseFlags));
                    }

                    filters.push(subfilters);
                });

                var result = [];

                entries.forEach(function(entry) {
                    for (var i = 0; i < filters.length; i++) {
                        var somethingMatches = false;

                        for (var j = 0; j < filters[i].length; j++) {
                            if (filters[i][j].test(entry[keywordsField])) {
                                somethingMatches = true;
                                break;
                            }
                        }

                        if (!somethingMatches) {
                            return;
                        }
                    }

                    result.push(entry);
                });

                return result;
            },

            /**
             * Draws an AUI logo with SVG.
             * @deprecated
             */
            drawLogo : function (options) {
                // Deprecated in 5.1
                var scale = options.scaleFactor || 1;
                var fill = options.fill || '#fff';
                var stroke = options.stroke || '#000';
                var width = 400 * scale;
                var height = 40 * scale;
                var strokeWidth = options.strokeWidth || 1;
                var containerID = options.containerID || '.aui-logo';

                if (!AJS.$('.aui-logo').length) {
                    AJS.$('body').append('<div id="aui-logo" class="aui-logo"><div>');
                }

                var logoCanvas = Raphael(containerID, width + 50 * scale, height + 100 * scale);
                var logo = logoCanvas.path('M 0,0 c 3.5433333,-4.7243333 7.0866667,-9.4486667 10.63,-14.173 -14.173,0 -28.346,0 -42.519,0 C -35.432667,-9.4486667 -38.976333,-4.7243333 -42.52,0 -28.346667,0 -14.173333,0 0,0 z m 277.031,28.346 c -14.17367,0 -28.34733,0 -42.521,0 C 245.14,14.173 255.77,0 266.4,-14.173 c -14.17267,0 -28.34533,0 -42.518,0 C 213.25167,0 202.62133,14.173 191.991,28.346 c -14.17333,0 -28.34667,0 -42.52,0 14.17333,-18.8976667 28.34667,-37.7953333 42.52,-56.693 -7.08667,-9.448667 -14.17333,-18.897333 -21.26,-28.346 -14.173,0 -28.346,0 -42.519,0 7.08667,9.448667 14.17333,18.897333 21.26,28.346 -14.17333,18.8976667 -28.34667,37.7953333 -42.52,56.693 -14.173333,0 -28.346667,0 -42.52,0 10.63,-14.173 21.26,-28.346 31.89,-42.519 -14.390333,0 -28.780667,0 -43.171,0 C 42.520733,1.330715e-4 31.889933,14.174867 21.26,28.347 c -42.520624,6.24e-4 -85.039187,-8.13e-4 -127.559,-0.001 11.220667,-14.961 22.441333,-29.922 33.662,-44.883 -6.496,-8.661 -12.992,-17.322 -19.488,-25.983 5.905333,0 11.810667,0 17.716,0 -10.63,-14.173333 -21.26,-28.346667 -31.89,-42.52 14.173333,0 28.346667,0 42.52,0 10.63,14.173333 21.26,28.346667 31.89,42.52 14.173333,0 28.3466667,0 42.52,0 -10.63,-14.173333 -21.26,-28.346667 -31.89,-42.52 14.1733333,0 28.3466667,0 42.52,0 10.63,14.173333 21.26,28.346667 31.89,42.52 14.390333,0 28.780667,0 43.171,0 -10.63,-14.173333 -21.26,-28.346667 -31.89,-42.52 42.51967,0 85.03933,0 127.559,0 10.63033,14.173333 21.26067,28.346667 31.891,42.52 14.17267,0 28.34533,0 42.518,0 -10.63,-14.173333 -21.26,-28.346667 -31.89,-42.52 14.17367,0 28.34733,0 42.521,0 14.17333,18.897667 28.34667,37.795333 42.52,56.693 -14.17333,18.8976667 -28.34667,37.7953333 -42.52,56.693 z');

                logo.scale(scale, -scale, 0, 0);
                logo.translate(120 * scale, height);
                logo.attr('fill', fill);
                logo.attr('stroke', stroke);
                logo.attr('stroke-width', strokeWidth);
            },

            /**
             * Debounce a function to avoid performance problems.
             */
            debounce: function (func, wait) {
                var timeout;
                var result;
                return function () {
                    var args = arguments;
                    var context = this;
                    var debounced = function () {
                        result = func.apply(context, args);
                    };
                    clearTimeout(timeout);
                    timeout = setTimeout(debounced, wait);
                    return result;
                };
            },

            /**
             * Generate a unique ID string, checking the ID is not present in the DOM before returning.
             * Note uniqueID, uniqueIDstring, uniqueIDcounter = 0; set at top of file.
             * @param {string} prefix Optional. String to prepend to ID instead of default AUI prefix.
             */
            id: function (prefix) {
                uniqueID = uniqueIDcounter++ + '';
                uniqueIDstring = prefix ? prefix + uniqueID : 'aui-uid-' + uniqueID;

                if (!document.getElementById(uniqueIDstring)) {
                    return uniqueIDstring;
                } else {
                    uniqueIDstring = uniqueIDstring + '-' + new Date().getTime();

                    if (!document.getElementById(uniqueIDstring)) {
                        return uniqueIDstring;
                    } else {
                        // if we still have a clash, something is deeply weird and needs attention.
                        throw new Error('ERROR: timestamped fallback ID ' + uniqueIDstring + ' exists. AJS.id stopped.');
                    }
                }
            },

            /**
             * Apply a unique ID to the element. Preserves ID if the element already has one.
             * @private
             * @param {HTMLElement} el Selector to find target element.
             * @param {string} prefix Optional. String to prepend to ID instead of default AUI prefix.
             */
            _addID: function (el, prefix) {
                var element = AJS.$(el);
                var addprefix = prefix || false;

                element.each(function () {
                    var $el = AJS.$(this);

                    if (!$el.attr('id')) {
                        $el.attr('id', AJS.id(addprefix));
                    }
                });
            },

            /**
             * Enables or disables any matching elements.
             */
            enable: function (el, b) {
                var $el = AJS.$(el);

                if (typeof b === 'undefined') {
                    b = true;
                }

                return $el.each(function () {
                    this.disabled = !b;
                });
            }
        };

        if (typeof AJS !== 'undefined') {
            for (var i in AJS) {
                res[i] = AJS[i];
            }
        }

        /**
         * Creates DOM object
         * @method AJS
         * @param {String} element tag name
         * @return {jQuery object}
         * @usage var a = AJS("div");
         */
        var result = function () {
            var res = null;

            if (arguments.length && typeof arguments[0] === 'string') {
                res = AJS.$(document.createElement(arguments[0]));

                if (arguments.length === 2) {
                    res.html(arguments[1]);
                }
            }

            return res;
        };

        for (var j in res) {
            result[j] = res[j];
        }

        return result;
    })();

    AJS.$(function () {
        //add version data to the body
        var $body = AJS.$('body');

        if (!$body.data('auiVersion')) {
            $body.attr('data-aui-version', AJS.version);
        }

        AJS.populateParameters();
    });

    // Setting Traditional to handle our default param serialisation
    // See http://api.jquery.com/jQuery.param/ for more
    AJS.$.ajaxSettings.traditional = true;

    AJS.deprecate.prop(AJS, 'firebug');
    AJS.deprecate.prop(AJS, 'stopEvent', {alternativeName: 'AJS.preventDefault()'});
    AJS.deprecate.prop(AJS, 'drawLogo');
    AJS.deprecate.prop(AJS, 'toggleClassName');

})();
