/* jshint expr:true */
define([
    '../helpers/all.js',
    'aui/internal/skate',
    'aui/inline-dialog2',
    'soy/inline-dialog2',
    'soy/trigger',
    'aui/trigger',
    'dialog2',
    'soy/dialog2',
    'layer'
], function (
    helpers,
    skate
) {
    'use strict';

    describe('Component layering -', function () {
        var baseInlineDialog;
        var basePersistentInlineDialog;
        var baseDialogEl;
        var baseDialog;
        var baseModalDialogEl;
        var baseModalDialog;

        function pressEsc () {
            helpers.pressKey(AJS.keyCode.ESCAPE);
        }

        function clickBlanket () {
            helpers.click($('.aui-blanket'));
        }

        function clickDocument () {
            helpers.click(document);
        }

        function clickDialogCloseTrigger (element) {
            helpers.dispatch('click', element.querySelector('.aui-dialog2-header-close'));
        }

        function clickInlineDialogCloseTrigger (element) {
            helpers.dispatch('click', element.querySelector('.aui-inline-dialog2-header-close'));
        }

        function createInlineDialog (persistent, id) {
            var $inlineDialog = $('<aui-inline-dialog2 id="inline-dialog2-' + (id || 1) + '" responds-to="toggle" persistent="' + (persistent === true) + '"><h3>I was generated by soy :)</h3></aui-inline-dialog2>').appendTo('#test-fixture');

            skate.init($inlineDialog);

            if (persistent !== true) {
                $inlineDialog.append($('<div></div>').addClass('aui-inline-dialog2-header-close'));
                $inlineDialog.find('.aui-inline-dialog2-header-close').on('click', function (e) {
                    $(e.target).closest('aui-inline-dialog2').get(0).hide();
                });
            }

            return $inlineDialog[0];
        }

        function createDialogEl (modal, id) {
            return $(aui.dialog.dialog2({
                id: 'dialog2-' + (id || 1),
                content: 'Hello world',
                modal: modal === true
            })).appendTo('#test-fixture').get(0);
        }

        /**
         * Use this to open the base, assert on base visible, then open the inside and assert both base and inside are visible
         * Expects there to exist a trigger with aria-controls pointing to the inside.
         * @param base should have isVisible and() show()
         * @param inside should have isVisible() and show()
         * @param $insideTrigger should be the used to open the inside component.
         */
        function openBaseAndInside (base, inside, $insideTrigger) {
            base.show();
            expect(base.isVisible()).to.be.true;
            expect(inside.isVisible()).to.be.false;
            $insideTrigger.click();
            expect(base.isVisible()).to.be.true;
            expect(inside.isVisible()).to.be.true;
        }

        /**
         * Use this to open the base, assert it is visible and the beside hidden. Then open the beside, assert it is visible.
         * If the base is persistent, assert's it is still visible, otherwise assert's it is now hidden
         * @param base should have show() and isVisible()
         * @param beside should have show() and isVisible()
         * @param {boolean} isBasePersistent if true, base not hidden on beside show, otherwise base should be hidden
         */
        function openBaseAndBeside (base, beside, isBasePersistent) {
            base.show();
            expect(base.isVisible()).to.be.true;
            expect(beside.isVisible()).to.be.false;
            beside.show();
            expect(base.isVisible()).to.equal(isBasePersistent);
            expect(beside.isVisible()).to.be.true;
        }

        /**
         * Creates a trigger with an on click handler that shows the component
         * @param component the element to be shown when trigger is clicked, should have show()
         * @param id the id of the component. If not passed in, this assumes it can get it from the component i.e. component.id
         * @returns {jQuery}
         */
        function createInsideShowTrigger (component, id) {
            var $insidePersistentInlineDialogTrigger = $('<div></div>').attr('aria-controls', id ? id : component.id);

            $insidePersistentInlineDialogTrigger.on('click', function () {
                component.show();
            });

            return $insidePersistentInlineDialogTrigger;
        }

        beforeEach(function () {
            baseInlineDialog = createInlineDialog();
            basePersistentInlineDialog = createInlineDialog(true);
            baseDialogEl = createDialogEl();
            baseDialog = AJS.dialog2(baseDialogEl);
            baseModalDialogEl = createDialogEl(true);
            baseModalDialog = AJS.dialog2(baseModalDialogEl);

            skate.init(baseInlineDialog);
            skate.init(basePersistentInlineDialog);
        });

        afterEach(function () {
            $(baseInlineDialog).remove();
            baseDialog.remove();
            baseModalDialog.remove();
            helpers.removeLayers();
        });

        describe('Above Empty -', function () {
            describe('Dialog closes on', function () {
                it('blanket click', function () {
                    baseDialog.show();
                    expect(baseDialog.isVisible()).to.be.true;
                    clickBlanket();
                    expect(baseDialog.isVisible()).to.be.false;
                });

                it('close trigger', function () {
                    baseDialog.show();
                    expect(baseDialog.isVisible()).to.be.true;
                    clickDialogCloseTrigger(baseDialogEl);
                    expect(baseDialog.isVisible()).to.be.false;
                });

                it('escape press', function () {
                    baseDialog.show();
                    expect(baseDialog.isVisible()).to.be.true;
                    pressEsc();
                    expect(baseDialog.isVisible()).to.be.false;
                });
            });

            it('Modal dialog closes programmatically, but not on blanket click or escape', function () {
                baseModalDialog.show();
                expect(baseModalDialog.isVisible()).to.be.true;
                baseModalDialog.hide();
                expect(baseModalDialog.isVisible()).to.be.false;

                baseModalDialog.show();
                expect(baseModalDialog.isVisible()).to.be.true;
                clickBlanket();
                expect(baseModalDialog.isVisible()).to.be.true;

                pressEsc();
                expect(baseModalDialog.isVisible()).to.be.true;
            });

            describe('Inline dialog closes on', function () {
                it('outside click', function () {
                    baseInlineDialog.show();
                    expect(baseInlineDialog.isVisible()).to.be.true;
                    clickDocument();
                    expect(baseInlineDialog.isVisible()).to.be.false;
                });

                it('trigger click', function () {
                    baseInlineDialog.show();
                    expect(baseInlineDialog.isVisible()).to.be.true;
                    clickInlineDialogCloseTrigger(baseInlineDialog);
                    expect(baseInlineDialog.isVisible()).to.be.false;
                });

                it('escape press', function () {
                    baseInlineDialog.show();
                    expect(baseInlineDialog.isVisible()).to.be.true;
                    pressEsc();
                    expect(baseInlineDialog.isVisible()).to.be.false;
                });
            });

            it('Persistent inline dialog closes programmatically, not outside click or escape', function () {
                basePersistentInlineDialog.show();
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
                basePersistentInlineDialog.hide();
                expect(basePersistentInlineDialog.isVisible()).to.be.false;

                basePersistentInlineDialog.show();
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
                clickDocument();
                expect(basePersistentInlineDialog.isVisible()).to.be.true;

                pressEsc();
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
            });

            it('Dropdown closes on outside click, close trigger click and escape press', function () {
                // TODO - requires new dropdown2
                expect(0);
            });

            it('Autocomplete closes on outside click, close trigger click and escape press', function () {
                // TODO - requires autocomplete
                expect(0);
            });
        });

        describe('Below Dialog -', function () {
            it('Dialog inside should close on blanket click, close trigger click and escape press', function () {
                var insideDialogEl = createDialogEl(false, 2);
                var insideDialog = AJS.dialog2(insideDialogEl);

                var $insideDialogTrigger = createInsideShowTrigger(insideDialog, insideDialogEl.id).appendTo(baseDialogEl);

                openBaseAndInside(baseDialog, insideDialog, $insideDialogTrigger);

                clickBlanket();
                expect(baseDialog.isVisible()).to.be.true;
                expect(insideDialog.isVisible()).to.be.false;

                $insideDialogTrigger.click();
                expect(baseDialog.isVisible()).to.be.true;
                expect(insideDialog.isVisible()).to.be.true;
                expect(insideDialog.isVisible()).to.be.true;
                pressEsc();
                expect(baseDialog.isVisible()).to.be.true;
                expect(insideDialog.isVisible()).to.be.false;

                $insideDialogTrigger.click();
                expect(baseDialog.isVisible()).to.be.true;
                expect(insideDialog.isVisible()).to.be.true;
                clickDialogCloseTrigger(insideDialogEl);
                expect(baseDialog.isVisible()).to.be.true;
                expect(insideDialog.isVisible()).to.be.false;
            });

            it('Modal dialog inside should close programatically, not on blanket click or escape press', function () {
                var insideModalDialogEl = createDialogEl(true, 2);
                var insideModalDialog = AJS.dialog2(insideModalDialogEl);

                var $insideModalDialogTrigger = createInsideShowTrigger(insideModalDialog, insideModalDialogEl.id).appendTo(baseDialogEl);

                // Opens ontop
                openBaseAndInside(baseDialog, insideModalDialog, $insideModalDialogTrigger);

                clickBlanket();
                expect(baseDialog.isVisible()).to.be.true;
                expect(insideModalDialog.isVisible()).to.be.true;

                pressEsc();
                expect(baseDialog.isVisible()).to.be.true;
                expect(insideModalDialog.isVisible()).to.be.true;

                insideModalDialog.hide();
                expect(baseDialog.isVisible()).to.be.true;
                expect(insideModalDialog.isVisible()).to.be.false;
            });

            it('Inline dialog inside should close on close trigger click and escape press', function () {
                var insideInlineDialog = createInlineDialog(false, 2);
                var $insideInlineDialogTrigger = createInsideShowTrigger(insideInlineDialog).appendTo(baseDialogEl);

                openBaseAndInside(baseDialog, insideInlineDialog, $insideInlineDialogTrigger);

                clickInlineDialogCloseTrigger(insideInlineDialog);
                expect(baseDialog.isVisible()).to.be.true;
                expect(insideInlineDialog.isVisible()).to.be.false;

                $insideInlineDialogTrigger.click();
                expect(insideInlineDialog.isVisible()).to.be.true;
                pressEsc();
                expect(baseDialog.isVisible()).to.be.true;
                expect(insideInlineDialog.isVisible()).to.be.false;
            });

            it('Inline dialog inside should close when dialog below is clicked', function () {
                var insideInlineDialog = createInlineDialog(false, 2);
                var $insideInlineDialogTrigger = createInsideShowTrigger(insideInlineDialog).appendTo(baseDialogEl);

                openBaseAndInside(baseDialog, insideInlineDialog, $insideInlineDialogTrigger);

                $(baseDialogEl).click();
                expect(baseDialog.isVisible()).to.be.true;
                expect(insideInlineDialog.isVisible()).to.be.false;
            });

            it('Inline dialog inside and dialog below should close when belows blanket and close trigger is clicked', function () {
                var insideInlineDialog = createInlineDialog(false, 2);
                var $insideInlineDialogTrigger = createInsideShowTrigger(insideInlineDialog).appendTo(baseDialogEl);

                openBaseAndInside(baseDialog, insideInlineDialog, $insideInlineDialogTrigger);

                clickDialogCloseTrigger(baseDialogEl);
                expect(baseDialog.isVisible()).to.be.false;
                expect(insideInlineDialog.isVisible()).to.be.false;

                openBaseAndInside(baseDialog, insideInlineDialog, $insideInlineDialogTrigger);

                clickBlanket();
                expect(baseDialog.isVisible()).to.be.false;
                expect(insideInlineDialog.isVisible()).to.be.false;
            });

            it('Persistent inline dialog inside closes on programmatic close, not when dialog below is clicked or escape is pressed', function () {
                var insidePersistentInlineDialog = createInlineDialog(true, 2);
                var $insideInlineDialogTrigger = createInsideShowTrigger(insidePersistentInlineDialog).appendTo(baseDialogEl);

                openBaseAndInside(baseDialog, insidePersistentInlineDialog, $insideInlineDialogTrigger);

                $(baseDialogEl).click();
                expect(baseDialog.isVisible()).to.be.true;
                expect(insidePersistentInlineDialog.isVisible()).to.be.true;

                insidePersistentInlineDialog.hide();
                expect(baseDialog.isVisible()).to.be.true;
                expect(insidePersistentInlineDialog.isVisible()).to.be.false;
            });

            it('Persistent inline dialog and dialog below both close on blanket click, close trigger click and escape press', function () {
                var insidePersistentInlineDialog = createInlineDialog(true, 2);
                var $insideInlineDialogTrigger = createInsideShowTrigger(insidePersistentInlineDialog).appendTo(baseDialogEl);

                openBaseAndInside(baseDialog, insidePersistentInlineDialog, $insideInlineDialogTrigger);

                clickBlanket();
                expect(baseDialog.isVisible()).to.be.false;
                expect(insidePersistentInlineDialog.isVisible()).to.be.false;

                openBaseAndInside(baseDialog, insidePersistentInlineDialog, $insideInlineDialogTrigger);

                pressEsc();
                expect(baseDialog.isVisible()).to.be.false;
                expect(insidePersistentInlineDialog.isVisible()).to.be.false;

                openBaseAndInside(baseDialog, insidePersistentInlineDialog, $insideInlineDialogTrigger);

                clickDialogCloseTrigger(baseDialogEl);
                expect(baseDialog.isVisible()).to.be.false;
                expect(insidePersistentInlineDialog.isVisible()).to.be.false;
            });

            it('Dropdown inside closes when dialog below is clicked, close trigger is clicked and escape is pressed', function () {
                // TODO - requires new dropdown2
                expect(0);
            });

            it('Dropdown inside closes when an item is clicked', function () {
                // TODO - requires new dropdown2
                expect(0);
            });

            it('Dropdown inside and dialog below both close when blanket is clicked or close trigger is clicked', function () {
                // TODO - requires new dropdown2
                expect(0);
            });

            it('Autocomplete inside closes when dialog below is clicked, close trigger is clicked and escape is pressed', function () {
                // TODO - requires autocomplete
                expect(0);
            });

            it('Autocomplete inside closes when an item is clicked', function () {
                // TODO - requires autocomplete
                expect(0);
            });

            it('Autocomplete inside and dialog below both close when blanket is clicked or close trigger is clicked', function () {
                // TODO - requires autocomplete
                expect(0);
            });
        });

        describe('Above Modal Dialog -', function () {
            it('Dialog inside closes on blanket click, close trigger click and escape press', function () {
                var insideDialogEl = createDialogEl(false, 2);
                var insideDialog = AJS.dialog2(insideDialogEl);

                var $insideDialogTrigger = createInsideShowTrigger(insideDialog, insideDialogEl.id).appendTo(baseModalDialogEl);

                openBaseAndInside(baseModalDialog, insideDialog, $insideDialogTrigger);

                clickBlanket();
                expect(baseModalDialog.isVisible()).to.be.true;
                expect(insideDialog.isVisible()).to.be.false;

                $insideDialogTrigger.click();
                expect(baseModalDialog.isVisible()).to.be.true;
                expect(insideDialog.isVisible()).to.be.true;
                pressEsc();
                expect(baseModalDialog.isVisible()).to.be.true;
                expect(insideDialog.isVisible()).to.be.false;

                $insideDialogTrigger.click();
                expect(baseModalDialog.isVisible()).to.be.true;
                expect(insideDialog.isVisible()).to.be.true;
                clickDialogCloseTrigger(insideDialogEl);
                expect(baseModalDialog.isVisible()).to.be.true;
                expect(insideDialog.isVisible()).to.be.false;
            });

            it('Modal dialog inside closes on close trigger click, not blanket click or escape press', function () {
                var insideModalDialogEl = createDialogEl(true, 2);
                var insideModalDialog = AJS.dialog2(insideModalDialogEl);

                var $insideModalDialogTrigger = createInsideShowTrigger(insideModalDialog, insideModalDialogEl.id).appendTo(baseModalDialogEl);

                openBaseAndInside(baseModalDialog, insideModalDialog, $insideModalDialogTrigger);

                clickBlanket();
                expect(baseModalDialog.isVisible()).to.be.true;
                expect(insideModalDialog.isVisible()).to.be.true;

                pressEsc();
                expect(baseModalDialog.isVisible()).to.be.true;
                expect(insideModalDialog.isVisible()).to.be.true;

                insideModalDialog.hide();
                expect(baseModalDialog.isVisible()).to.be.true;
                expect(insideModalDialog.isVisible()).to.be.false;
            });

            it('Inline dialog inside closes on blanket click, close trigger click and escape press', function () {
                var insideInlineDialog = createInlineDialog(false, 2);
                var $insideInlineDialogTrigger = createInsideShowTrigger(insideInlineDialog).appendTo(baseModalDialogEl);

                openBaseAndInside(baseModalDialog, insideInlineDialog, $insideInlineDialogTrigger);

                clickInlineDialogCloseTrigger(insideInlineDialog);
                expect(baseModalDialog.isVisible()).to.be.true;
                expect(insideInlineDialog.isVisible()).to.be.false;

                $insideInlineDialogTrigger.click();
                expect(insideInlineDialog.isVisible()).to.be.true;
                pressEsc();
                expect(baseModalDialog.isVisible()).to.be.true;
                expect(insideInlineDialog.isVisible()).to.be.false;

                $insideInlineDialogTrigger.click();
                expect(insideInlineDialog.isVisible()).to.be.true;
                clickBlanket();
                expect(baseModalDialog.isVisible()).to.be.true;
                expect(insideInlineDialog.isVisible()).to.be.false;
            });

            it('Inline dialog inside should close when modal dialog below is clicked', function () {
                var insideInlineDialog = createInlineDialog(false, 2);
                var $insideInlineDialogTrigger = createInsideShowTrigger(insideInlineDialog).appendTo(baseModalDialogEl);

                openBaseAndInside(baseModalDialog, insideInlineDialog, $insideInlineDialogTrigger);

                $(baseModalDialogEl).click();
                expect(baseModalDialog.isVisible()).to.be.true;
                expect(insideInlineDialog.isVisible()).to.be.false;
            });

            it('Persistent inline dialog inside closes on programatic close, not blanket click or escape press', function () {
                var insidePersistentInlineDialog = createInlineDialog(true, 2);
                var $insideInlineDialogTrigger = createInsideShowTrigger(insidePersistentInlineDialog).appendTo(baseModalDialogEl);

                openBaseAndInside(baseModalDialog, insidePersistentInlineDialog, $insideInlineDialogTrigger);

                clickBlanket();
                expect(baseModalDialog.isVisible()).to.be.true;
                expect(insidePersistentInlineDialog.isVisible()).to.be.true;

                pressEsc();
                expect(baseModalDialog.isVisible()).to.be.true;
                expect(insidePersistentInlineDialog.isVisible()).to.be.true;

                insidePersistentInlineDialog.hide();
                expect(baseModalDialog.isVisible()).to.be.true;
                expect(insidePersistentInlineDialog.isVisible()).to.be.false;
            });

            it('Persistent inline dialog inside should not close when modal dialog below is clicked', function () {
                var insidePersistentInlineDialog = createInlineDialog(true, 2);
                var $insidePersistentInlineDialogTrigger = createInsideShowTrigger(insidePersistentInlineDialog).appendTo(baseModalDialogEl);

                openBaseAndInside(baseModalDialog, insidePersistentInlineDialog, $insidePersistentInlineDialogTrigger);

                $(baseModalDialogEl).click();
                expect(baseModalDialog.isVisible()).to.be.true;
                expect(insidePersistentInlineDialog.isVisible()).to.be.true;
            });

            it('Persistent inline dialog inside and modal dialog below should close when belows close trigger is clicked', function () {
                var insidePersistentInlineDialog = createInlineDialog(true, 2);

                var $insideInlineDialogTrigger = createInsideShowTrigger(insidePersistentInlineDialog).appendTo(baseModalDialogEl);
                openBaseAndInside(baseModalDialog, insidePersistentInlineDialog, $insideInlineDialogTrigger);

                baseModalDialog.hide();
                expect(baseModalDialog.isVisible()).to.be.false;
                expect(insidePersistentInlineDialog.isVisible()).to.be.false;
            });

            it('Dropdown inside should close on outside click, close trigger click and escape press', function () {
                // TODO - requires new dropdown2
                expect(0);
            });

            it('Autocomplete inside should close on outside click, close trigger click and escape press', function () {
                // TODO - requires autocomplete
                expect(0);
            });
        });

        describe('Above Inline Dialog -', function () {
            it('Opening an inline dialog closes the non-persistent one that is open', function () {
                var newInlineDialog = createInlineDialog(false, 2);

                baseInlineDialog.show();
                expect(baseInlineDialog.isVisible()).to.be.true;
                expect(newInlineDialog.isVisible()).to.be.false;

                newInlineDialog.show();
                expect(baseInlineDialog.isVisible()).to.be.false;
                expect(newInlineDialog.isVisible()).to.be.true;
            });

            it('Opening a persistent inline dialog closes the non-persistent one that is open', function () {
                var newInlineDialog = createInlineDialog(true, 2);

                baseInlineDialog.show();
                expect(baseInlineDialog.isVisible()).to.be.true;
                expect(newInlineDialog.isVisible()).to.be.false;

                newInlineDialog.show();
                expect(baseInlineDialog.isVisible()).to.be.false;
                expect(newInlineDialog.isVisible()).to.be.true;
            });

            it('Dialog beside closes the non-persistent inline dialog below', function () {
                var besideDialogEl = createDialogEl(false, 2);

                var besideDialog = AJS.dialog2(besideDialogEl);
                openBaseAndBeside(baseInlineDialog, besideDialog, false);
            });

            it('Dialog beside closes on blanket click, close trigger click and escape press', function () {
                var besideDialogEl = createDialogEl(false, 2);

                var besideDialog = AJS.dialog2(besideDialogEl);
                openBaseAndBeside(baseInlineDialog, besideDialog, false);

                clickBlanket();
                expect(baseInlineDialog.isVisible()).to.be.false;
                expect(besideDialog.isVisible()).to.be.false;

                besideDialog.show();
                expect(baseInlineDialog.isVisible()).to.be.false;
                expect(besideDialog.isVisible()).to.be.true;
                pressEsc();
                expect(baseInlineDialog.isVisible()).to.be.false;
                expect(besideDialog.isVisible()).to.be.false;

                besideDialog.show();
                expect(baseInlineDialog.isVisible()).to.be.false;
                expect(besideDialog.isVisible()).to.be.true;
                clickDialogCloseTrigger(besideDialogEl);
                expect(baseInlineDialog.isVisible()).to.be.false;
                expect(besideDialog.isVisible()).to.be.false;
            });

            it('Dialog inside closes on blanket click, close trigger click and escape press', function () {
                var insideDialogEl = createDialogEl(false, 2);
                var insideDialog = AJS.dialog2(insideDialogEl);

                var $insideDialogTrigger = createInsideShowTrigger(insideDialog, insideDialogEl.id).appendTo(baseInlineDialog);

                openBaseAndInside(baseInlineDialog, insideDialog, $insideDialogTrigger);

                clickBlanket();
                expect(baseInlineDialog.isVisible()).to.be.true;
                expect(insideDialog.isVisible()).to.be.false;

                $insideDialogTrigger.click();
                expect(baseInlineDialog.isVisible()).to.be.true;
                expect(insideDialog.isVisible()).to.be.true;
                pressEsc();
                expect(baseInlineDialog.isVisible()).to.be.true;
                expect(insideDialog.isVisible()).to.be.false;

                $insideDialogTrigger.click();
                expect(baseInlineDialog.isVisible()).to.be.true;
                expect(insideDialog.isVisible()).to.be.true;
                clickDialogCloseTrigger(insideDialogEl);
                expect(baseInlineDialog.isVisible()).to.be.true;
                expect(insideDialog.isVisible()).to.be.false;
            });

            it('Modal dialog inside closes on close trigger click, not blanket click or escape press', function () {
                var insideModalDialogEl = createDialogEl(true, 2);
                var insideModalDialog = AJS.dialog2(insideModalDialogEl);

                var $insideModalDialogTrigger = createInsideShowTrigger(insideModalDialog, insideModalDialogEl.id).appendTo(baseInlineDialog);

                openBaseAndInside(baseInlineDialog, insideModalDialog, $insideModalDialogTrigger);

                clickBlanket();
                expect(baseInlineDialog.isVisible()).to.be.true;
                expect(insideModalDialog.isVisible()).to.be.true;

                pressEsc();
                expect(baseInlineDialog.isVisible()).to.be.true;
                expect(insideModalDialog.isVisible()).to.be.true;

                insideModalDialog.hide();
                expect(baseInlineDialog.isVisible()).to.be.true;
                expect(insideModalDialog.isVisible()).to.be.false;
            });

            it('Modal dialog beside closes the non-persistent inline dialog below', function () {
                var besideModalDialogEl = createDialogEl(true, 2);

                var besideModalDialog = AJS.dialog2(besideModalDialogEl);
                openBaseAndBeside(baseInlineDialog, besideModalDialog, false);
            });

            it('Modal dialog beside closes on close trigger click, not blanket click or escape press', function () {
                var besideModalDialogEl = createDialogEl(true, 2);
                var besideModalDialog = AJS.dialog2(besideModalDialogEl);
                openBaseAndBeside(baseInlineDialog, besideModalDialog, false);

                clickBlanket();
                expect(baseInlineDialog.isVisible()).to.be.false;
                expect(besideModalDialog.isVisible()).to.be.true;

                pressEsc();
                expect(baseInlineDialog.isVisible()).to.be.false;
                expect(besideModalDialog.isVisible()).to.be.true;

                besideModalDialog.hide();
                expect(baseInlineDialog.isVisible()).to.be.false;
                expect(besideModalDialog.isVisible()).to.be.false;
            });

            it('Dropdown inside closes, escape press and select item', function () {
                // TODO - requires new dropdown2
                expect(0);
            });

            it('Dropdown inside closes when inline dialog below is clicked', function () {
                // TODO - requires new dropdown2
                expect(0);
            });

            it('Dropdown inside and inline dialog below both close on outside click and close trigger click', function () {
                // TODO - requires new dropdown2
                expect(0);
            });

            it('Dropdown beside should close inline dialog below', function () {
                // TODO - requires new dropdown2
                expect(0);
            });

            it('Autocomplete inside closes, escape press and select item', function () {
                // TODO - requires autocomplete
                expect(0);
            });

            it('Autocomplete inside closes when inline dialog below is clicked', function () {
                // TODO - requires autocomplete
                expect(0);
            });

            it('Autocomplete inside and inline dialog below both close on outside click and close trigger click', function () {
                // TODO - requires autocomplete
                expect(0);
            });
        });

        describe('Above Persistent Inline Dialog -', function () {
            it('Dialog inside should close on blanket click, close trigger click and escape press', function () {
                var insideDialogEl = createDialogEl(false, 2);
                var insideDialog = AJS.dialog2(insideDialogEl);

                var $insideDialogTrigger = createInsideShowTrigger(insideDialog, insideDialogEl.id).appendTo(basePersistentInlineDialog);

                openBaseAndInside(basePersistentInlineDialog, insideDialog, $insideDialogTrigger);

                clickBlanket();
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
                expect(insideDialog.isVisible()).to.be.false;

                $insideDialogTrigger.click();
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
                expect(insideDialog.isVisible()).to.be.true;
                pressEsc();
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
                expect(insideDialog.isVisible()).to.be.false;

                $insideDialogTrigger.click();
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
                expect(insideDialog.isVisible()).to.be.true;
                clickDialogCloseTrigger(insideDialogEl);
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
                expect(insideDialog.isVisible()).to.be.false;
            });

            it('Dialog beside should not close persistent inline dialog below', function () {
                var besideDialogEl = createDialogEl(false, 2);
                var besideDialog = AJS.dialog2(besideDialogEl);

                openBaseAndBeside(basePersistentInlineDialog, besideDialog, true);
            });

            it('Dialog beside should close on blanket click, close trigger click and escape press', function () {
                var besideDialogEl = createDialogEl(false, 2);
                var besideDialog = AJS.dialog2(besideDialogEl);

                openBaseAndBeside(basePersistentInlineDialog, besideDialog, true);

                clickBlanket();
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
                expect(besideDialog.isVisible()).to.be.false;

                besideDialog.show();
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
                expect(besideDialog.isVisible()).to.be.true;
                pressEsc();
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
                expect(besideDialog.isVisible()).to.be.false;

                besideDialog.show();
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
                expect(besideDialog.isVisible()).to.be.true;
                clickDialogCloseTrigger(besideDialogEl);
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
                expect(besideDialog.isVisible()).to.be.false;
            });

            it('Modal dialog inside should close on close trigger click, not blanket click or escape press', function () {
                var insideModalDialogEl = createDialogEl(true, 2);
                var insideModalDialog = AJS.dialog2(insideModalDialogEl);

                var $insideModalDialogTrigger = createInsideShowTrigger(insideModalDialog, insideModalDialogEl.id).appendTo(basePersistentInlineDialog);

                // Opens ontop
                openBaseAndInside(basePersistentInlineDialog, insideModalDialog, $insideModalDialogTrigger);

                clickBlanket();
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
                expect(insideModalDialog.isVisible()).to.be.true;

                pressEsc();
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
                expect(insideModalDialog.isVisible()).to.be.true;

                insideModalDialog.hide();
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
                expect(insideModalDialog.isVisible()).to.be.false;
            });

            it('Modal dialog beside should not close persistent inline dialog below', function () {
                var besideModalDialogEl = createDialogEl(true, 2);
                var besideModalDialog = AJS.dialog2(besideModalDialogEl);

                openBaseAndBeside(basePersistentInlineDialog, besideModalDialog, true);
            });

            it('Modal dialog beside should close on close trigger click, not blanket click or escape press', function () {
                var besideModalDialogEl = createDialogEl(true, 2);
                var besideModalDialog = AJS.dialog2(besideModalDialogEl);

                openBaseAndBeside(basePersistentInlineDialog, besideModalDialog, true);

                clickBlanket();
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
                expect(besideModalDialog.isVisible()).to.be.true;

                pressEsc();
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
                expect(besideModalDialog.isVisible()).to.be.true;

                besideModalDialog.hide();
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
                expect(besideModalDialog.isVisible()).to.be.false;
            });

            it('Inline dialog beside should not close persistent inline dialog below', function () {
                var besideInlineDialog = createInlineDialog(false, 2);
                openBaseAndBeside(basePersistentInlineDialog, besideInlineDialog, true);
            });

            it('Inline dialog beside should close on outside click, close trigger click and escape press', function () {
                var besideInlineDialog = createInlineDialog(false, 2);
                openBaseAndBeside(basePersistentInlineDialog, besideInlineDialog, true);

                clickDocument();
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
                expect(besideInlineDialog.isVisible()).to.be.false;

                besideInlineDialog.show();
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
                expect(besideInlineDialog.isVisible()).to.be.true;
                pressEsc();
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
                expect(besideInlineDialog.isVisible()).to.be.false;

                besideInlineDialog.show();
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
                expect(besideInlineDialog.isVisible()).to.be.true;
                clickInlineDialogCloseTrigger(besideInlineDialog);
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
                expect(besideInlineDialog.isVisible()).to.be.false;
            });

            it('Persistent inline dialog beside should not close persistent inline dialog below', function () {
                var newPersistentInlineDialog = createInlineDialog(true, 2);
                openBaseAndBeside(basePersistentInlineDialog, newPersistentInlineDialog, true);
            });

            it('Persistent inline dialog beside closes programatically, not outside click or escape press', function () {
                var besidePersistentInlineDialog = createInlineDialog(true, 2);
                openBaseAndBeside(basePersistentInlineDialog, besidePersistentInlineDialog, true);

                clickDocument();
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
                expect(besidePersistentInlineDialog.isVisible()).to.be.true;

                pressEsc();
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
                expect(besidePersistentInlineDialog.isVisible()).to.be.true;

                besidePersistentInlineDialog.hide();
                expect(basePersistentInlineDialog.isVisible()).to.be.true;
                expect(besidePersistentInlineDialog.isVisible()).to.be.false;
            });

            it('Persistent inline dialog beside does not close when below is closed programatically', function () {
                var besidePersistentInlineDialog = createInlineDialog(true, 2);
                openBaseAndBeside(basePersistentInlineDialog, besidePersistentInlineDialog, true);

                basePersistentInlineDialog.hide();
                expect(basePersistentInlineDialog.isVisible()).to.be.false;
                expect(besidePersistentInlineDialog.isVisible()).to.be.true;
            });

            it('Dropdown inside closes on outside click, close trigger click, escape press and item select', function () {
                // TODO - requires new dropdown2
                expect(0);
            });

            it('Dropdown inside and inline dialog below both close if inline dialogs close trigger is clicked', function () {
                // TODO - requires new dropdown2
                expect(0);
            });

            it('Dropdown beside does not close persistent inline dialog below', function () {
                // TODO - requires new dropdown2
                expect(0);
            });

            it('Autocomplete inside closes on outside click, close trigger click, escape press and item select', function () {
                // TODO - requires autocomplete
                expect(0);
            });

            it('Autocomplete inside and inline dialog below both close if inline dialogs close trigger is clicked', function () {
                // TODO - requires autocomplete
                expect(0);
            });

            it('Autocomplete beside does not close persistent inline dialog below', function () {
                // TODO - requires new dropdown2
                expect(0);
            });
        });

        describe('Above Dropdown -', function () {
            it('Dropdown inside closes on trigger and escape', function () {
                // TODO - requires new dropdown2
                expect(0);
            });

            it('Dropdown inside and below close on outside click', function () {
                // TODO - requires new dropdown2
                expect(0);
            });

            it('Opening a dropdown outside closes the open dropdown', function () {
                // TODO - requires new dropdown2
                expect(0);
            });

            it('Inline dialog beside should close dropdown', function () {
                // TODO - requires new dropdown2
                expect(0);
            });

            it('Dialog beside should close dropdown', function () {
                // TODO - requires new dropdown2
                expect(0);
            });
        });
    });
});
