import Popper from 'popper.js';
import startCase from 'lodash.startcase';

/**
 * Register and event to listen on specified element once.
 * @param {Element} element to listen on
 * @param {String} event to listen for
 * @param {Function} callback when event fires
 */

// Production steps of ECMA-262, Edition 6, 22.1.2.1
// es6-ified by @alexsasharegan
if (!Array.from) {
  Array.from = function () {
    var toStr = Object.prototype.toString;
    var isCallable = function isCallable(fn) {
      return typeof fn === 'function' || toStr.call(fn) === '[object Function]';
    };
    var toInteger = function toInteger(value) {
      var number = Number(value);
      if (isNaN(number)) {
        return 0;
      }
      if (number === 0 || !isFinite(number)) {
        return number;
      }
      return (number > 0 ? 1 : -1) * Math.floor(Math.abs(number));
    };
    var maxSafeInteger = Math.pow(2, 53) - 1;
    var toLength = function toLength(value) {
      return Math.min(Math.max(toInteger(value), 0), maxSafeInteger);
    };

    // The length property of the from method is 1.
    return function from(arrayLike /*, mapFn, thisArg */) {
      // 1. Let C be the this value.
      var C = this;

      // 2. Let items be ToObject(arrayLike).
      var items = Object(arrayLike);

      // 3. ReturnIfAbrupt(items).
      if (arrayLike == null) {
        throw new TypeError('Array.from requires an array-like object - not null or undefined');
      }

      // 4. If mapfn is undefined, then let mapping be false.
      var mapFn = arguments.length > 1 ? arguments[1] : void undefined;
      var T = void 0;

      if (typeof mapFn !== 'undefined') {
        // 5. else
        // 5. a If IsCallable(mapfn) is false, throw a TypeError exception.
        if (!isCallable(mapFn)) {
          throw new TypeError('Array.from: when provided, the second argument must be a function');
        }

        // 5. b. If thisArg was supplied, let T be thisArg; else let T be undefined.
        if (arguments.length > 2) {
          T = arguments[2];
        }
      }

      // 10. Let lenValue be Get(items, "length").
      // 11. Let len be ToLength(lenValue).
      var len = toLength(items.length);

      // 13. If IsConstructor(C) is true, then
      // 13. a. Let A be the result of calling the [[Construct]] internal method
      // of C with an argument list containing the single item len.
      // 14. a. Else, Let A be ArrayCreate(len).
      var A = isCallable(C) ? Object(new C(len)) : new Array(len);

      // 16. Let k be 0.
      var k = 0;
      // 17. Repeat, while k < len… (also steps a - h)
      var kValue = void 0;
      while (k < len) {
        kValue = items[k];
        if (mapFn) {
          A[k] = typeof T === 'undefined' ? mapFn(kValue, k) : mapFn.call(T, kValue, k);
        } else {
          A[k] = kValue;
        }
        k += 1;
      }
      // 18. Let putStatus be Put(A, "length", len, true).
      A.length = len;
      // 20. Return A.
      return A;
    };
  }();
}

// https://tc39.github.io/ecma262/#sec-array.prototype.find
// Needed for IE support
if (!Array.prototype.find) {
  // eslint-disable-next-line no-extend-native
  Object.defineProperty(Array.prototype, 'find', {
    value: function value(predicate) {
      // 1. Let O be ? ToObject(this value).
      if (this == null) {
        throw new TypeError('"this" is null or not defined');
      }

      var o = Object(this);

      // 2. Let len be ? ToLength(? Get(O, "length")).
      var len = o.length >>> 0;

      // 3. If IsCallable(predicate) is false, throw a TypeError exception.
      if (typeof predicate !== 'function') {
        throw new TypeError('predicate must be a function');
      }

      // 4. If thisArg was supplied, let T be thisArg; else let T be undefined.
      var thisArg = arguments[1];

      // 5. Let k be 0.
      var k = 0;

      // 6. Repeat, while k < len
      while (k < len) {
        // a. Let Pk be ! ToString(k).
        // b. Let kValue be ? Get(O, Pk).
        // c. Let testResult be ToBoolean(? Call(predicate, T, « kValue, k, O »)).
        // d. If testResult is true, return kValue.
        var kValue = o[k];
        if (predicate.call(thisArg, kValue, k, o)) {
          return kValue;
        }
        // e. Increase k by 1.
        k++;
      }

      // 7. Return undefined.
      return undefined;
    }
  });
}

if (!Array.isArray) {
  Array.isArray = function (arg) {
    return Object.prototype.toString.call(arg) === '[object Array]';
  };
}

// Static
var from = Array.from;
var isArray = Array.isArray;

// Instance
var arrayIncludes = function arrayIncludes(array, value) {
  return array.indexOf(value) !== -1;
};

function concat() {
  return Array.prototype.concat.apply([], arguments);
}

/**
 * Aliasing Object[method] allows the minifier to shorten methods to a single character variable,
 * as well as giving BV a chance to inject polyfills.
 * As long as we avoid
 * - import * as Object from "utils/object"
 * all unused exports should be removed by tree-shaking.
 */

// @link https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Object/assign
if (typeof Object.assign !== 'function') {
  Object.assign = function (target, varArgs) {
    // .length of function is 2

    if (target == null) {
      // TypeError if undefined or null
      throw new TypeError('Cannot convert undefined or null to object');
    }

    var to = Object(target);

    for (var index = 1; index < arguments.length; index++) {
      var nextSource = arguments[index];

      if (nextSource != null) {
        // Skip over if undefined or null
        for (var nextKey in nextSource) {
          // Avoid bugs when hasOwnProperty is shadowed
          if (Object.prototype.hasOwnProperty.call(nextSource, nextKey)) {
            to[nextKey] = nextSource[nextKey];
          }
        }
      }
    }
    return to;
  };
}

// @link https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Object/is#Polyfill
if (!Object.is) {
  Object.is = function (x, y) {
    // SameValue algorithm
    if (x === y) {
      // Steps 1-5, 7-10
      // Steps 6.b-6.e: +0 != -0
      return x !== 0 || 1 / x === 1 / y;
    } else {
      // Step 6.a: NaN == NaN
      // eslint-disable-next-line no-self-compare
      return x !== x && y !== y;
    }
  };
}

var assign = Object.assign;

var keys = Object.keys;
var defineProperties = Object.defineProperties;
var defineProperty = Object.defineProperty;




var create = Object.create;



function readonlyDescriptor() {
  return { enumerable: true, configurable: false, writable: false };
}

// Determine if an element is an HTML Element
var isElement = function isElement(el) {
  return el && el.nodeType === Node.ELEMENT_NODE;
};

// Determine if an HTML element is visible - Faster than CSS check
var isVisible = function isVisible(el) {
  return isElement(el) && document.body.contains(el) && el.getBoundingClientRect().height > 0 && el.getBoundingClientRect().width > 0;
};

// Determine if an element is disabled
var isDisabled = function isDisabled(el) {
  return !isElement(el) || el.disabled || el.classList.contains('disabled') || Boolean(el.getAttribute('disabled'));
};

// Cause/wait-for an element to reflow it's content (adjusting it's height/width)
var reflow = function reflow(el) {
  // requsting an elements offsetHight will trigger a reflow of the element content
  return isElement(el) && el.offsetHeight;
};

// Select all elements matching selector. Returns [] if none found
var selectAll = function selectAll(selector, root) {
  if (!isElement(root)) {
    root = document;
  }
  return from(root.querySelectorAll(selector));
};

// Select a single element, returns null if not found
var select = function select(selector, root) {
  if (!isElement(root)) {
    root = document;
  }
  return root.querySelector(selector) || null;
};

// Determine if an element matches a selector
var matches = function matches(el, selector) {
  if (!isElement(el)) {
    return false;
  }

  // https://developer.mozilla.org/en-US/docs/Web/API/Element/matches#Polyfill
  // Prefer native implementations over polyfill function
  var proto = Element.prototype;
  var Matches = proto.matches || proto.matchesSelector || proto.mozMatchesSelector || proto.msMatchesSelector || proto.oMatchesSelector || proto.webkitMatchesSelector || function (sel) {
    var element = this;
    var m = selectAll(sel, element.document || element.ownerDocument);
    var i = m.length;
    // eslint-disable-next-line no-empty
    while (--i >= 0 && m.item(i) !== element) {}
    return i > -1;
  };

  return Matches.call(el, selector);
};

// Finds closest element matching selector. Returns null if not found
var closest = function closest(selector, root) {
  if (!isElement(root)) {
    return null;
  }

  // https://developer.mozilla.org/en-US/docs/Web/API/Element/closest
  // Since we dont support IE < 10, we can use the "Matches" version of the polyfill for speed
  // Prefer native implementation over polyfill function
  var Closest = Element.prototype.closest || function (sel) {
    var element = this;
    if (!document.documentElement.contains(element)) {
      return null;
    }
    do {
      // Use our "patched" matches function
      if (matches(element, sel)) {
        return element;
      }
      element = element.parentElement;
    } while (element !== null);
    return null;
  };

  var el = Closest.call(root, selector);
  // Emulate jQuery closest and return null if match is the passed in element (root)
  return el === root ? null : el;
};

// Get an element given an ID
var getById = function getById(id) {
  return document.getElementById(/^#/.test(id) ? id.slice(1) : id) || null;
};

// Add a class to an element
var addClass = function addClass(el, className) {
  if (className && isElement(el)) {
    el.classList.add(className);
  }
};

// Remove a class from an element
var removeClass = function removeClass(el, className) {
  if (className && isElement(el)) {
    el.classList.remove(className);
  }
};

// Test if an element has a class
var hasClass = function hasClass(el, className) {
  if (className && isElement(el)) {
    return el.classList.contains(className);
  }
  return false;
};

// Set an attribute on an element
var setAttr = function setAttr(el, attr, value) {
  if (attr && isElement(el)) {
    el.setAttribute(attr, value);
  }
};

// Remove an attribute from an element
var removeAttr = function removeAttr(el, attr) {
  if (attr && isElement(el)) {
    el.removeAttribute(attr);
  }
};

// Get an attribute value from an element (returns null if not found)
var getAttr = function getAttr(el, attr) {
  if (attr && isElement(el)) {
    return el.getAttribute(attr);
  }
  return null;
};

// Determine if an attribute exists on an element (returns true or false, or null if element not found)
var hasAttr = function hasAttr(el, attr) {
  if (attr && isElement(el)) {
    return el.hasAttribute(attr);
  }
  return null;
};

// Return the Bounding Client Rec of an element. Retruns null if not an element
var getBCR = function getBCR(el) {
  return isElement(el) ? el.getBoundingClientRect() : null;
};

// Get computed style object for an element
var getCS = function getCS(el) {
  return isElement(el) ? window.getComputedStyle(el) : {};
};

// Return an element's offset wrt document element
// https://j11y.io/jquery/#v=git&fn=jQuery.fn.offset
var offset = function offset(el) {
  if (isElement(el)) {
    if (!el.getClientRects().length) {
      return { top: 0, left: 0 };
    }
    var bcr = getBCR(el);
    var win = el.ownerDocument.defaultView;
    return {
      top: bcr.top + win.pageYOffset,
      left: bcr.left + win.pageXOffset
    };
  }
};

// Return an element's offset wrt to it's offsetParent
// https://j11y.io/jquery/#v=git&fn=jQuery.fn.position
var position = function position(el) {
  if (!isElement(el)) {
    return;
  }
  var parentOffset = { top: 0, left: 0 };
  var offsetSelf = void 0;
  var offsetParent = void 0;
  if (getCS(el).position === 'fixed') {
    offsetSelf = getBCR(el);
  } else {
    offsetSelf = offset(el);
    var doc = el.ownerDocument;
    offsetParent = el.offsetParent || doc.documentElement;
    while (offsetParent && (offsetParent === doc.body || offsetParent === doc.documentElement) && getCS(offsetParent).position === 'static') {
      offsetParent = offsetParent.parentNode;
    }
    if (offsetParent && offsetParent !== el && offsetParent.nodeType === Node.ELEMENT_NODE) {
      parentOffset = offset(offsetParent);
      parentOffset.top += parseFloat(getCS(offsetParent).borderTopWidth);
      parentOffset.left += parseFloat(getCS(offsetParent).borderLeftWidth);
    }
  }
  return {
    top: offsetSelf.top - parentOffset.top - parseFloat(getCS(el).marginTop),
    left: offsetSelf.left - parentOffset.left - parseFloat(getCS(el).marginLeft)
  };
};

// Attach an event listener to an element
var eventOn = function eventOn(el, evtName, handler) {
  if (el && el.addEventListener) {
    el.addEventListener(evtName, handler);
  }
};

// Remove an event listener from an element
var eventOff = function eventOff(el, evtName, handler) {
  if (el && el.removeEventListener) {
    el.removeEventListener(evtName, handler);
  }
};

function identity(x) {
  return x;
}

var _typeof = typeof Symbol === "function" && typeof Symbol.iterator === "symbol" ? function (obj) {
  return typeof obj;
} : function (obj) {
  return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj;
};











var classCallCheck = function (instance, Constructor) {
  if (!(instance instanceof Constructor)) {
    throw new TypeError("Cannot call a class as a function");
  }
};

var createClass = function () {
  function defineProperties(target, props) {
    for (var i = 0; i < props.length; i++) {
      var descriptor = props[i];
      descriptor.enumerable = descriptor.enumerable || false;
      descriptor.configurable = true;
      if ("value" in descriptor) descriptor.writable = true;
      Object.defineProperty(target, descriptor.key, descriptor);
    }
  }

  return function (Constructor, protoProps, staticProps) {
    if (protoProps) defineProperties(Constructor.prototype, protoProps);
    if (staticProps) defineProperties(Constructor, staticProps);
    return Constructor;
  };
}();





var defineProperty$1 = function (obj, key, value) {
  if (key in obj) {
    Object.defineProperty(obj, key, {
      value: value,
      enumerable: true,
      configurable: true,
      writable: true
    });
  } else {
    obj[key] = value;
  }

  return obj;
};



var inherits = function (subClass, superClass) {
  if (typeof superClass !== "function" && superClass !== null) {
    throw new TypeError("Super expression must either be null or a function, not " + typeof superClass);
  }

  subClass.prototype = Object.create(superClass && superClass.prototype, {
    constructor: {
      value: subClass,
      enumerable: false,
      writable: true,
      configurable: true
    }
  });
  if (superClass) Object.setPrototypeOf ? Object.setPrototypeOf(subClass, superClass) : subClass.__proto__ = superClass;
};











var possibleConstructorReturn = function (self, call) {
  if (!self) {
    throw new ReferenceError("this hasn't been initialised - super() hasn't been called");
  }

  return call && (typeof call === "object" || typeof call === "function") ? call : self;
};



















var toConsumableArray = function (arr) {
  if (Array.isArray(arr)) {
    for (var i = 0, arr2 = Array(arr.length); i < arr.length; i++) arr2[i] = arr[i];

    return arr2;
  } else {
    return Array.from(arr);
  }
};

/**
 * @param {[]|{}} props
 * @param {Function} transformFn
 */
function copyProps(props) {
  var transformFn = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : identity;

  if (isArray(props)) {
    return props.map(transformFn);
  }
  // Props as an object.
  var copied = {};

  for (var prop in props) {
    if (props.hasOwnProperty(prop)) {
      if ((typeof prop === 'undefined' ? 'undefined' : _typeof(prop)) === 'object') {
        copied[transformFn(prop)] = assign({}, props[prop]);
      } else {
        copied[transformFn(prop)] = props[prop];
      }
    }
  }

  return copied;
}

/*
 * Key Codes (events)
 */

var KeyCodes = {
  SPACE: 32,
  ENTER: 13,
  ESC: 27,
  LEFT: 37,
  UP: 38,
  RIGHT: 39,
  DOWN: 40,
  PAGEUP: 33,
  PAGEDOWN: 34,
  HOME: 36,
  END: 35
};

/**
 * @param {string} str
 */
function lowerFirst(str) {
  if (typeof str !== 'string') {
    str = String(str);
  }
  return str.charAt(0).toLowerCase() + str.slice(1);
}

/**
 * Quick object check - this is primarily used to tell
 * Objects from primitive values when we know the value
 * is a JSON-compliant type.
 */
function isObject(obj) {
  return obj !== null && (typeof obj === 'undefined' ? 'undefined' : _typeof(obj)) === 'object';
}

/**
 * Check if two values are loosely equal - that is,
 * if they are plain objects, do they have the same shape?
 * Returns boolean true or false
 */
function looseEqual(a, b) {
  if (a === b) return true;
  var isObjectA = isObject(a);
  var isObjectB = isObject(b);
  if (isObjectA && isObjectB) {
    try {
      var isArrayA = isArray(a);
      var isArrayB = isArray(b);
      if (isArrayA && isArrayB) {
        return a.length === b.length && a.every(function (e, i) {
          return looseEqual(e, b[i]);
        });
      } else if (!isArrayA && !isArrayB) {
        var keysA = keys(a);
        var keysB = keys(b);
        return keysA.length === keysB.length && keysA.every(function (key) {
          return looseEqual(a[key], b[key]);
        });
      } else {
        return false;
      }
    } catch (e) {
      return false;
    }
  } else if (!isObjectA && !isObjectB) {
    return String(a) === String(b);
  } else {
    return false;
  }
}

function concat$1() {
  return Array.prototype.concat.apply([], arguments);
}function mergeData() {
  for (var e = __assign({}, arguments[0]), a = 1; a < arguments.length; a++) {
    for (var s = 0, t = keys$1(arguments[a]); s < t.length; s++) {
      var c = t[s];if (void 0 !== e[c]) switch (c) {case "class":case "style":case "directives":
          e[c] = concat$1(e[c], arguments[a][c]);break;case "staticClass":
          e[c] && (e[c] = e[c].trim() + " "), e[c] += arguments[a][c].trim();break;case "on":case "nativeOn":
          for (var r = 0, o = keys$1(arguments[a][c]); r < o.length; r++) {
            var n = o[r];e[c][n] ? e[c][n] = concat$1(arguments[a][c][n], e[c][n]) : e[c][n] = arguments[a][c][n];
          }break;case "attrs":case "props":case "domProps":case "scopedSlots":case "staticStyle":case "hook":case "transition":
          e[c] = __assign({}, e[c], arguments[a][c]);break;case "slot":case "key":case "ref":case "tag":case "show":case "keepAlive":default:
          e[c] = arguments[a][c];} else e[c] = arguments[a][c];
    }
  }return e;
}var __assign = Object.assign || function (e) {
  for (var a, s = 1, t = arguments.length; s < t; s++) {
    a = arguments[s];for (var c in a) {
      Object.prototype.hasOwnProperty.call(a, c) && (e[c] = a[c]);
    }
  }return e;
};
var keys$1 = Object.keys;var lib_common = mergeData;

function memoize(fn) {
  var cache = create(null);

  return function memoizedFn() {
    var args = JSON.stringify(arguments);
    return cache[args] = cache[args] || fn.apply(null, arguments);
  };
}

/**
 * Observe a DOM element changes, falls back to eventListener mode
 * @param {Element} el The DOM element to observe
 * @param {Function} callback callback to be called on change
 * @param {object} [opts={childList: true, subtree: true}] observe options
 * @see http://stackoverflow.com/questions/3219758
 */
function observeDOM(el, callback, opts) {
  var MutationObserver = window.MutationObserver || window.WebKitMutationObserver || window.MozMutationObserver;
  var eventListenerSupported = window.addEventListener;

  // Handle case where we might be passed a vue instance
  el = el ? el.$el || el : null;
  if (!isElement(el)) {
    // We can't observe somthing that isn't an element
    return null;
  }

  var obs = null;

  if (MutationObserver) {
    // Define a new observer
    obs = new MutationObserver(function (mutations) {
      var changed = false;
      // A Mutation can contain several change records, so we loop through them to see what has changed.
      // We break out of the loop early if any "significant" change has been detected
      for (var i = 0; i < mutations.length && !changed; i++) {
        // The muttion record
        var mutation = mutations[i];
        // Mutation Type
        var type = mutation.type;
        // DOM Node (could be any DOM Node type - HTMLElement, Text, comment, etc)
        var target = mutation.target;
        if (type === 'characterData' && target.nodeType === Node.TEXT_NODE) {
          // We ignore nodes that are not TEXt (i.e. comments, etc) as they don't change layout
          changed = true;
        } else if (type === 'attributes') {
          changed = true;
        } else if (type === 'childList' && (mutation.addedNodes.length > 0 || mutation.removedNodes.length > 0)) {
          // This includes HTMLElement and Text Nodes being added/removed/re-arranged
          changed = true;
        }
      }
      if (changed) {
        // We only call the callback if a change that could affect layout/size truely happened.
        callback();
      }
    });

    // Have the observer observe foo for changes in children, etc
    obs.observe(el, assign({ childList: true, subtree: true }, opts));
  } else if (eventListenerSupported) {
    // Legacy interface. most likely not used in modern browsers
    el.addEventListener('DOMNodeInserted', callback, false);
    el.addEventListener('DOMNodeRemoved', callback, false);
  }

  // We return a reference to the observer so that obs.disconnect() can be called if necessary
  // To reduce overhead when the root element is hiiden
  return obs;
}

/**
 * Given an array of properties or an object of property keys,
 * plucks all the values off the target object.
 * @param {{}|string[]} keysToPluck
 * @param {{}} objToPluck
 * @param {Function} transformFn
 * @return {{}}
 */
function pluckProps(keysToPluck, objToPluck) {
  var transformFn = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : identity;

  return (isArray(keysToPluck) ? keysToPluck.slice() : keys(keysToPluck)).reduce(function (memo, prop) {
    // eslint-disable-next-line no-sequences
    return memo[transformFn(prop)] = objToPluck[prop], memo;
  }, {});
}

/**
 * @param {string} str
 */
function upperFirst(str) {
  if (typeof str !== 'string') {
    str = String(str);
  }
  return str.charAt(0).toUpperCase() + str.slice(1);
}

/**
 * @param {string} prefix
 * @param {string} value
 */
function prefixPropName(prefix, value) {
  return prefix + upperFirst(value);
}

/**
 * Register a component plugin as being loaded. returns true if compoent plugin already registered
 * @param {object} Vue
 * @param {string} Component name
 * @param {object} Component definition
 */
function registerComponent(Vue, name, def) {
  Vue._bootstrap_vue_components_ = Vue._bootstrap_vue_components_ || {};
  var loaded = Vue._bootstrap_vue_components_[name];
  if (!loaded && def && name) {
    Vue._bootstrap_vue_components_[name] = true;
    Vue.component(name, def);
  }
  return loaded;
}

/**
 * Register a group of components as being loaded.
 * @param {object} Vue
 * @param {object} Object of component definitions
 */
function registerComponents(Vue, components) {
  for (var component in components) {
    registerComponent(Vue, component, components[component]);
  }
}

/**
 * Register a directive as being loaded. returns true if directive plugin already registered
 * @param {object} Vue
 * @param {string} Directive name
 * @param {object} Directive definition
 */
function registerDirective(Vue, name, def) {
  Vue._bootstrap_vue_directives_ = Vue._bootstrap_vue_directives_ || {};
  var loaded = Vue._bootstrap_vue_directives_[name];
  if (!loaded && def && name) {
    Vue._bootstrap_vue_directives_[name] = true;
    Vue.directive(name, def);
  }
  return loaded;
}

/**
 * Register a group of directives as being loaded.
 * @param {object} Vue
 * @param {object} Object of directive definitions
 */
function registerDirectives(Vue, directives) {
  for (var directive in directives) {
    registerDirective(Vue, directive, directives[directive]);
  }
}

/**
 * Install plugin if window.Vue available
 * @param {object} Plugin definition
 */
function vueUse(VuePlugin) {
  if (typeof window !== 'undefined' && window.Vue) {
    window.Vue.use(VuePlugin);
  }
}

/*
 * Consitant and stable sort function across JavsaScript platforms
 *
 * Inconsistant sorts can cause SSR problems between client and server
 * such as in <b-table> if sortBy is applied to the data on server side render.
 * Chrome and V8 native sorts are inconsistant/unstable
 *
 * This function uses native sort with fallback to index compare when the a and b
 * compare returns 0
 *
 * Algorithm bsaed on:
 * https://stackoverflow.com/questions/1427608/fast-stable-sorting-algorithm-implementation-in-javascript/45422645#45422645
 *
 * @param {array} array to sort
 * @param {function} sortcompare function
 * @return {array}
 */

function stableSort(array, compareFn) {
  // Using `.bind(compareFn)` on the wrapped anonymous function improves
  // performance by avoiding the function call setup. We don't use an arrow
  // function here as it binds `this` to the `stableSort` context rather than
  // the `compareFn` context, which wouldn't give us the performance increase.
  return array.map(function (a, index) {
    return [index, a];
  }).sort(function (a, b) {
    return this(a[1], b[1]) || a[0] - b[0];
  }.bind(compareFn)).map(function (e) {
    return e[1];
  });
}

/**
 * Suffix can be a falsey value so nothing is appended to string.
 * (helps when looping over props & some shouldn't change)
 * Use data last parameters to allow for currying.
 * @param {string} suffix
 * @param {string} str
 */
function suffixPropName(suffix, str) {
  return str + (suffix ? upperFirst(suffix) : '');
}

/**
 * @param {string} prefix
 * @param {string} value
 */
function unPrefixPropName(prefix, value) {
  return lowerFirst(value.replace(prefix, ''));
}

/**
 * Log a warning message to the console with bootstrap-vue formatting sugar.
 * @param {string} message
 */
function warn(message) {
  console.warn("[Bootstrap-Vue warn]: " + message);
}

var props = {
  disabled: {
    type: Boolean,
    default: false
  },
  ariaLabel: {
    type: String,
    default: 'Close'
  },
  textVariant: {
    type: String,
    default: null
  }
};

var bBtnClose = {
  functional: true,
  props: props,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        listeners = _ref.listeners,
        slots = _ref.slots;

    var componentData = {
      staticClass: 'close',
      class: defineProperty$1({}, 'text-' + props.textVariant, props.textVariant),
      attrs: {
        type: 'button',
        disabled: props.disabled,
        'aria-label': props.ariaLabel ? String(props.ariaLabel) : null
      },
      on: {
        click: function click(e) {
          // Ensure click on button HTML content is also disabled
          if (props.disabled && e instanceof Event) {
            e.stopPropagation();
            e.preventDefault();
          }
        }
      }
      // Careful not to override the slot with innerHTML
    };if (!slots().default) {
      componentData.domProps = { innerHTML: '&times;' };
    }
    return h('button', lib_common(data, componentData), slots().default);
  }
};

var bAlert = {
  components: { bBtnClose: bBtnClose },
  render: function render(h) {
    if (!this.localShow) {
      // If not showing, render placeholder
      return h(false);
    }
    var dismissBtn = h(false);
    if (this.dismissible) {
      // Add dismiss button
      dismissBtn = h('b-button-close', { attrs: { 'aria-label': this.dismissLabel }, on: { 'click': this.dismiss } }, [this.$slots.dismiss]);
    }
    return h('div', { class: this.classObject, attrs: { role: 'alert', 'aria-live': 'polite', 'aria-atomic': true } }, [dismissBtn, this.$slots.default]);
  },

  model: {
    prop: 'show',
    event: 'input'
  },
  data: function data() {
    return {
      countDownTimerId: null,
      dismissed: false
    };
  },

  computed: {
    classObject: function classObject() {
      return ['alert', this.alertVariant, this.dismissible ? 'alert-dismissible' : ''];
    },
    alertVariant: function alertVariant() {
      var variant = this.variant;
      return 'alert-' + variant;
    },
    localShow: function localShow() {
      return !this.dismissed && (this.countDownTimerId || this.show);
    }
  },
  props: {
    variant: {
      type: String,
      default: 'info'
    },
    dismissible: {
      type: Boolean,
      default: false
    },
    dismissLabel: {
      type: String,
      default: 'Close'
    },
    show: {
      type: [Boolean, Number],
      default: false
    }
  },
  watch: {
    show: function show() {
      this.showChanged();
    }
  },
  mounted: function mounted() {
    this.showChanged();
  },
  destroyed: function destroyed() {
    this.clearCounter();
  },

  methods: {
    dismiss: function dismiss() {
      this.clearCounter();
      this.dismissed = true;
      this.$emit('dismissed');
      this.$emit('input', false);
      if (typeof this.show === 'number') {
        this.$emit('dismiss-count-down', 0);
        this.$emit('input', 0);
      } else {
        this.$emit('input', false);
      }
    },
    clearCounter: function clearCounter() {
      if (this.countDownTimerId) {
        clearInterval(this.countDownTimerId);
        this.countDownTimerId = null;
      }
    },
    showChanged: function showChanged() {
      var _this = this;

      // Reset counter status
      this.clearCounter();
      // Reset dismiss status
      this.dismissed = false;
      // No timer for boolean values
      if (this.show === true || this.show === false || this.show === null || this.show === 0) {
        return;
      }
      // Start counter
      var dismissCountDown = this.show;
      this.countDownTimerId = setInterval(function () {
        if (dismissCountDown < 1) {
          _this.dismiss();
          return;
        }
        dismissCountDown--;
        _this.$emit('dismiss-count-down', dismissCountDown);
        _this.$emit('input', dismissCountDown);
      }, 1000);
    }
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$1 = {
  bAlert: bAlert
};

var VuePlugin$2 = {
  install: function install(Vue) {
    registerComponents(Vue, components$1);
  }
};

vueUse(VuePlugin$2);

/**
 * The Link component is used in many other BV components.
 * As such, sharing its props makes supporting all its features easier.
 * However, some components need to modify the defaults for their own purpose.
 * Prefer sharing a fresh copy of the props to ensure mutations
 * do not affect other component references to the props.
 *
 * https://github.com/vuejs/vue-router/blob/dev/src/components/link.js
 * @return {{}}
 */
function propsFactory() {
  return {
    href: {
      type: String,
      default: null
    },
    rel: {
      type: String,
      default: null
    },
    target: {
      type: String,
      default: '_self'
    },
    active: {
      type: Boolean,
      default: false
    },
    activeClass: {
      type: String,
      default: 'active'
    },
    append: {
      type: Boolean,
      default: false
    },
    disabled: {
      type: Boolean,
      default: false
    },
    event: {
      type: [String, Array],
      default: 'click'
    },
    exact: {
      type: Boolean,
      default: false
    },
    exactActiveClass: {
      type: String,
      default: 'active'
    },
    replace: {
      type: Boolean,
      default: false
    },
    routerTag: {
      type: String,
      default: 'a'
    },
    to: {
      type: [String, Object],
      default: null
    }
  };
}



function pickLinkProps(propsToPick) {
  var freshLinkProps = propsFactory();
  // Normalize everything to array.
  propsToPick = concat(propsToPick);

  return keys(freshLinkProps).reduce(function (memo, prop) {
    if (arrayIncludes(propsToPick, prop)) {
      memo[prop] = freshLinkProps[prop];
    }

    return memo;
  }, {});
}





function computeTag(props, parent) {
  return Boolean(parent.$router) && props.to && !props.disabled ? 'router-link' : 'a';
}

function computeHref(_ref, tag) {
  var disabled = _ref.disabled,
      href = _ref.href,
      to = _ref.to;

  // We've already checked the parent.$router in computeTag,
  // so router-link means live router.
  // When deferring to Vue Router's router-link,
  // don't use the href attr at all.
  // Must return undefined for router-link to populate href.
  if (tag === 'router-link') return void 0;
  // If href explicitly provided
  if (href) return href;
  // Reconstruct href when `to` used, but no router
  if (to) {
    // Fallback to `to` prop (if `to` is a string)
    if (typeof to === 'string') return to;
    // Fallback to `to.path` prop (if `to` is an object)
    if ((typeof to === 'undefined' ? 'undefined' : _typeof(to)) === 'object' && typeof to.path === 'string') return to.path;
  }
  // If nothing is provided use '#'
  return '#';
}

function computeRel(_ref2) {
  var target = _ref2.target,
      rel = _ref2.rel;

  if (target === '_blank' && rel === null) {
    return 'noopener';
  }
  return rel || null;
}

function clickHandlerFactory(_ref3) {
  var disabled = _ref3.disabled,
      tag = _ref3.tag,
      href = _ref3.href,
      suppliedHandler = _ref3.suppliedHandler,
      parent = _ref3.parent;

  var isRouterLink = tag === 'router-link';

  return function onClick(e) {
    if (disabled && e instanceof Event) {
      // Stop event from bubbling up.
      e.stopPropagation();
      // Kill the event loop attached to this specific EventTarget.
      e.stopImmediatePropagation();
    } else {
      parent.$root.$emit('clicked::link', e);

      if (isRouterLink && e.target.__vue__) {
        e.target.__vue__.$emit('click', e);
      }
      if (typeof suppliedHandler === 'function') {
        suppliedHandler.apply(undefined, arguments);
      }
    }

    if (!isRouterLink && href === '#' || disabled) {
      // Stop scroll-to-top behavior or navigation.
      e.preventDefault();
    }
  };
}

var Link = {
  functional: true,
  props: propsFactory(),
  render: function render(h, _ref4) {
    var props = _ref4.props,
        data = _ref4.data,
        parent = _ref4.parent,
        children = _ref4.children;

    var tag = computeTag(props, parent);
    var rel = computeRel(props);
    var href = computeHref(props, tag);
    var eventType = tag === 'router-link' ? 'nativeOn' : 'on';
    var suppliedHandler = (data[eventType] || {}).click;
    var handlers = { click: clickHandlerFactory({ tag: tag, href: href, disabled: props.disabled, suppliedHandler: suppliedHandler, parent: parent }) };

    var componentData = lib_common(data, {
      class: [props.active ? props.exact ? props.exactActiveClass : props.activeClass : null, { disabled: props.disabled }],
      attrs: {
        rel: rel,
        href: href,
        target: props.target,
        tabindex: props.disabled ? '-1' : data.attrs ? data.attrs.tabindex : null,
        'aria-disabled': tag === 'a' && props.disabled ? 'true' : null
      },
      props: assign(props, { tag: props.routerTag })
    });

    // If href prop exists on router-link (even undefined or null) it fails working on SSR
    if (!componentData.attrs.href) {
      delete componentData.attrs.href;
    }

    // We want to overwrite any click handler since our callback
    // will invoke the supplied handler if !props.disabled
    componentData[eventType] = assign(componentData[eventType] || {}, handlers);

    return h(tag, componentData, children);
  }
};

var linkProps = propsFactory();
delete linkProps.href.default;
delete linkProps.to.default;

var props$1 = assign(linkProps, {
  tag: {
    type: String,
    default: 'span'
  },
  variant: {
    type: String,
    default: 'secondary'
  },
  pill: {
    type: Boolean,
    default: false
  }
});

var bBadge = {
  functional: true,
  props: props$1,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    var tag = !props.href && !props.to ? props.tag : Link;

    var componentData = {
      staticClass: 'badge',
      class: [!props.variant ? 'badge-secondary' : 'badge-' + props.variant, {
        'badge-pill': Boolean(props.pill),
        active: props.active,
        disabled: props.disabled
      }],
      props: pluckProps(linkProps, props)
    };

    return h(tag, lib_common(data, componentData), children);
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$2 = {
  bBadge: bBadge
};

var VuePlugin$4 = {
  install: function install(Vue) {
    registerComponents(Vue, components$2);
  }
};

vueUse(VuePlugin$4);

var props$5 = assign(propsFactory(), {
  text: {
    type: String,
    default: null
  },
  active: {
    type: Boolean,
    default: false
  },
  href: {
    type: String,
    default: '#'
  },
  ariaCurrent: {
    type: String,
    default: 'location'
  }
});

var BreadcrumbLink = {
  functional: true,
  props: props$5,
  render: function render(h, _ref) {
    var suppliedProps = _ref.props,
        data = _ref.data,
        children = _ref.children;

    var tag = suppliedProps.active ? 'span' : Link;

    var componentData = {
      props: pluckProps(props$5, suppliedProps),
      domProps: { innerHTML: suppliedProps.text }
    };

    if (suppliedProps.active) {
      componentData.attrs = { 'aria-current': suppliedProps.ariaCurrent };
    } else {
      componentData.attrs = { href: suppliedProps.href };
    }

    return h(tag, lib_common(data, componentData), children);
  }
};

var props$4 = assign({}, props$5, {
  text: {
    type: String,
    default: null
  },
  href: {
    type: String,
    default: null
  }
});

var BreadcrumbItem = {
  functional: true,
  props: props$4,
  render: function render(h, _ref) {
    var props$$1 = _ref.props,
        data = _ref.data,
        children = _ref.children;

    return h('li', lib_common(data, {
      staticClass: 'breadcrumb-item',
      class: { active: props$$1.active },
      attrs: { role: 'presentation' }
    }), [h(BreadcrumbLink, { props: props$$1 }, children)]);
  }
};

var props$3 = {
  items: {
    type: Array,
    default: null
  }
};

var bBreadcrumb = {
  functional: true,
  props: props$3,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    var childNodes = children;
    // Build child nodes from items if given.
    if (isArray(props.items)) {
      var activeDefined = false;
      childNodes = props.items.map(function (item, idx) {
        if ((typeof item === 'undefined' ? 'undefined' : _typeof(item)) !== 'object') {
          item = { text: item };
        }
        // Copy the value here so we can normalize it.
        var active = item.active;
        if (active) {
          activeDefined = true;
        }
        if (!active && !activeDefined) {
          // Auto-detect active by position in list.
          active = idx + 1 === props.items.length;
        }

        return h(BreadcrumbItem, { props: assign({}, item, { active: active }) });
      });
    }

    return h('ol', lib_common(data, { staticClass: 'breadcrumb' }), childNodes);
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$3 = {
  bBreadcrumb: bBreadcrumb,
  bBreadcrumbItem: BreadcrumbItem,
  bBreadcrumbLink: BreadcrumbLink
};

var VuePlugin$6 = {
  install: function install(Vue) {
    registerComponents(Vue, components$3);
  }
};

vueUse(VuePlugin$6);

var btnProps = {
  block: {
    type: Boolean,
    default: false
  },
  disabled: {
    type: Boolean,
    default: false
  },
  size: {
    type: String,
    default: null
  },
  variant: {
    type: String,
    default: null
  },
  type: {
    type: String,
    default: 'button'
  },
  pressed: {
    // tri-state prop: true, false or null
    // => on, off, not a toggle
    type: Boolean,
    default: null
  }
};

var linkProps$1 = propsFactory();
delete linkProps$1.href.default;
delete linkProps$1.to.default;
var linkPropKeys = keys(linkProps$1);

var props$6 = assign(linkProps$1, btnProps);

function handleFocus(evt) {
  if (evt.type === 'focusin') {
    addClass(evt.target, 'focus');
  } else if (evt.type === 'focusout') {
    removeClass(evt.target, 'focus');
  }
}

var bBtn = {
  functional: true,
  props: props$6,
  render: function render(h, _ref) {
    var _ref2;

    var props = _ref.props,
        data = _ref.data,
        listeners = _ref.listeners,
        children = _ref.children;

    var isLink = Boolean(props.href || props.to);
    var isToggle = typeof props.pressed === 'boolean';
    var on = {
      click: function click(e) {
        if (props.disabled && e instanceof Event) {
          e.stopPropagation();
          e.preventDefault();
        } else if (isToggle) {
          // Concat will normalize the value to an array
          // without double wrapping an array value in an array.
          concat(listeners['update:pressed']).forEach(function (fn) {
            if (typeof fn === 'function') {
              fn(!props.pressed);
            }
          });
        }
      }
    };

    if (isToggle) {
      on.focusin = handleFocus;
      on.focusout = handleFocus;
    }

    var componentData = {
      staticClass: 'btn',
      class: [props.variant ? 'btn-' + props.variant : 'btn-secondary', (_ref2 = {}, defineProperty$1(_ref2, 'btn-' + props.size, Boolean(props.size)), defineProperty$1(_ref2, 'btn-block', props.block), defineProperty$1(_ref2, 'disabled', props.disabled), defineProperty$1(_ref2, 'active', props.pressed), _ref2)],
      props: isLink ? pluckProps(linkPropKeys, props) : null,
      attrs: {
        type: isLink ? null : props.type,
        disabled: isLink ? null : props.disabled,
        // Data attribute not used for js logic,
        // but only for BS4 style selectors.
        'data-toggle': isToggle ? 'button' : null,
        'aria-pressed': isToggle ? String(props.pressed) : null,
        // Tab index is used when the component becomes a link.
        // Links are tabable, but don't allow disabled,
        // so we mimic that functionality by disabling tabbing.
        tabindex: props.disabled && isLink ? '-1' : data.attrs ? data.attrs['tabindex'] : null
      },
      on: on
    };

    return h(isLink ? Link : 'button', lib_common(data, componentData), children);
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$4 = {
  bButton: bBtn,
  bBtn: bBtn,
  bButtonClose: bBtnClose,
  bBtnClose: bBtnClose
};

var VuePlugin$8 = {
  install: function install(Vue) {
    registerComponents(Vue, components$4);
  }
};

vueUse(VuePlugin$8);

var props$7 = {
  vertical: {
    type: Boolean,
    default: false
  },
  size: {
    type: String,
    default: null,
    validator: function validator(size) {
      return arrayIncludes(['sm', '', 'lg'], size);
    }
  },
  tag: {
    type: String,
    default: 'div'
  },
  ariaRole: {
    type: String,
    default: 'group'
  }
};

var bButtonGroup = {
  functional: true,
  props: props$7,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    return h(props.tag, lib_common(data, {
      class: defineProperty$1({
        'btn-group': !props.vertical,
        'btn-group-vertical': props.vertical
      }, 'btn-group-' + props.size, Boolean(props.size)),
      attrs: { 'role': props.ariaRole }
    }), children);
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$5 = {
  bButtonGroup: bButtonGroup,
  bBtnGroup: bButtonGroup
};

var VuePlugin$10 = {
  install: function install(Vue) {
    registerComponents(Vue, components$5);
  }
};

vueUse(VuePlugin$10);

var ITEM_SELECTOR = ['.btn:not(.disabled):not([disabled]):not(.dropdown-item)', '.form-control:not(.disabled):not([disabled])', 'select:not(.disabled):not([disabled])', 'input[type="checkbox"]:not(.disabled)', 'input[type="radio"]:not(.disabled)'].join(',');

var bButtonToolbar = {
  render: function render(h) {
    var t = this;
    return h('div', {
      class: t.classObject,
      attrs: {
        role: 'toolbar',
        tabindex: t.keyNav ? '0' : null
      },
      on: {
        focusin: t.onFocusin,
        keydown: t.onKeydown
      }
    }, [t.$slots.default]);
  },

  computed: {
    classObject: function classObject() {
      return ['btn-toolbar', this.justify && !this.vertical ? 'justify-content-between' : ''];
    }
  },
  props: {
    justify: {
      type: Boolean,
      default: false
    },
    keyNav: {
      type: Boolean,
      default: false
    }
  },
  methods: {
    onFocusin: function onFocusin(evt) {
      if (evt.target === this.$el) {
        evt.preventDefault();
        evt.stopPropagation();
        this.focusFirst(evt);
      }
    },
    onKeydown: function onKeydown(evt) {
      if (!this.keyNav) {
        return;
      }
      var key = evt.keyCode;
      var shift = evt.shiftKey;
      if (key === KeyCodes.UP || key === KeyCodes.LEFT) {
        evt.preventDefault();
        evt.stopPropagation();
        if (shift) {
          this.focusFirst(evt);
        } else {
          this.focusNext(evt, true);
        }
      } else if (key === KeyCodes.DOWN || key === KeyCodes.RIGHT) {
        evt.preventDefault();
        evt.stopPropagation();
        if (shift) {
          this.focusLast(evt);
        } else {
          this.focusNext(evt, false);
        }
      }
    },
    setItemFocus: function setItemFocus(item) {
      this.$nextTick(function () {
        item.focus();
      });
    },
    focusNext: function focusNext(evt, prev) {
      var items = this.getItems();
      if (items.length < 1) {
        return;
      }
      var index = items.indexOf(evt.target);
      if (prev && index > 0) {
        index--;
      } else if (!prev && index < items.length - 1) {
        index++;
      }
      if (index < 0) {
        index = 0;
      }
      this.setItemFocus(items[index]);
    },
    focusFirst: function focusFirst(evt) {
      var items = this.getItems();
      if (items.length > 0) {
        this.setItemFocus(items[0]);
      }
    },
    focusLast: function focusLast(evt) {
      var items = this.getItems();
      if (items.length > 0) {
        this.setItemFocus([items.length - 1]);
      }
    },
    getItems: function getItems() {
      var items = selectAll(ITEM_SELECTOR, this.$el);
      items.forEach(function (item) {
        // Ensure tabfocus is -1 on any new elements
        item.tabIndex = -1;
      });
      return items.filter(function (el) {
        return isVisible(el);
      });
    }
  },
  mounted: function mounted() {
    if (this.keyNav) {
      // Pre-set the tabindexes if the markup does not include tabindex="-1" on the toolbar items
      this.getItems();
    }
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$6 = {
  bButtonToolbar: bButtonToolbar,
  bBtnToolbar: bButtonToolbar
};

var VuePlugin$12 = {
  install: function install(Vue) {
    registerComponents(Vue, components$6);
  }
};

vueUse(VuePlugin$12);

var props$9 = {
  id: {
    type: String,
    default: null
  },
  tag: {
    type: String,
    default: 'div'
  }
};

var InputGroupAddon = {
  functional: true,
  props: props$9,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    return h(props.tag, lib_common(data, {
      staticClass: 'input-group-addon',
      attrs: { id: props.id }
    }), children);
  }
};

var props$8 = {
  id: {
    type: String,
    default: null
  },
  size: {
    type: String,
    default: null
  },
  left: {
    type: String,
    default: null
  },
  right: {
    type: String,
    default: null
  },
  tag: {
    type: String,
    default: 'div'
  }
};

var bInputGroup = {
  functional: true,
  props: props$8,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        slots = _ref.slots;

    var childNodes = [];

    // Left Slot / Prop
    if (slots().left) {
      childNodes.push(slots().left);
    } else if (props.left) {
      childNodes.push(h(InputGroupAddon, { domProps: { innerHTML: props.left } }));
    }

    // Default slot
    childNodes.push(slots().default);

    // Right slot / prop
    if (slots().right) {
      childNodes.push(slots().right);
    } else if (props.right) {
      childNodes.push(h(InputGroupAddon, { domProps: { innerHTML: props.right } }));
    }

    return h(props.tag, lib_common(data, {
      staticClass: 'input-group',
      class: defineProperty$1({}, 'input-group-' + props.size, Boolean(props.size)),
      attrs: {
        id: props.id || null,
        role: 'group'
      }
    }), childNodes);
  }
};

var props$10 = {
  id: {
    type: String,
    default: null
  },
  tag: {
    type: String,
    default: 'div'
  }
};

var bInputGroupButton = {
  functional: true,
  props: props$10,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    return h(props.tag, lib_common(data, {
      staticClass: 'input-group-btn',
      attrs: {
        id: props.id
      }
    }), children);
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$7 = {
  bInputGroup: bInputGroup,
  bInputGroupAddon: InputGroupAddon,
  bInputGroupButton: bInputGroupButton,
  bInputGroupBtn: bInputGroupButton
};

var VuePlugin$14 = {
  install: function install(Vue) {
    registerComponents(Vue, components$7);
  }
};

vueUse(VuePlugin$14);

var cardMixin = {
  props: {
    tag: {
      type: String,
      default: 'div'
    },
    bgVariant: {
      type: String,
      default: null
    },
    borderVariant: {
      type: String,
      default: null
    },
    textVariant: {
      type: String,
      default: null
    }
  }
};

var clickoutMixin = {
  mounted: function mounted() {
    if (typeof document !== 'undefined') {
      document.documentElement.addEventListener('click', this._clickOutListener);
    }
  },
  beforeDestroy: function beforeDestroy() {
    if (typeof document !== 'undefined') {
      document.removeEventListener('click', this._clickOutListener);
    }
  },

  methods: {
    _clickOutListener: function _clickOutListener(e) {
      if (!this.$el.contains(e.target)) {
        if (this.clickOutListener) {
          this.clickOutListener();
        }
      }
    }
  }
};

/**
 * Issue #569: collapse::toggle::state triggered too many times
 * @link https://github.com/bootstrap-vue/bootstrap-vue/issues/569
 */

var BVRL = '__BV_root_listeners__';

var listenOnRootMixin = {
  methods: {
    /**
         * Safely register event listeners on the root Vue node.
         * While Vue automatically removes listeners for individual components,
         * when a component registers a listener on root and is destroyed,
         * this orphans a callback because the node is gone,
         * but the root does not clear the callback.
         *
         * This adds a non-reactive prop to a vm on the fly
         * in order to avoid object observation and its performance costs
         * to something that needs no reactivity.
         * It should be highly unlikely there are any naming collisions.
         * @param {string} event
         * @param {function} callback
         * @chainable
         */
    listenOnRoot: function listenOnRoot(event, callback) {
      if (!this[BVRL] || !isArray(this[BVRL])) {
        this[BVRL] = [];
      }
      this[BVRL].push({ event: event, callback: callback });
      this.$root.$on(event, callback);
      return this;
    },


    /**
         * Convenience method for calling vm.$emit on vm.$root.
         * @param {string} event
         * @param {*} args
         * @chainable
         */
    emitOnRoot: function emitOnRoot(event) {
      var _$root;

      for (var _len = arguments.length, args = Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
        args[_key - 1] = arguments[_key];
      }

      (_$root = this.$root).$emit.apply(_$root, [event].concat(toConsumableArray(args)));
      return this;
    }
  },

  beforeDestroy: function beforeDestroy() {
    if (this[BVRL] && isArray(this[BVRL])) {
      while (this[BVRL].length > 0) {
        // shift to process in order
        var _BVRL$shift = this[BVRL].shift(),
            event = _BVRL$shift.event,
            callback = _BVRL$shift.callback;

        this.$root.$off(event, callback);
      }
    }
  }
};

// Return an Array of visible items
function filterVisible(els) {
  return (els || []).filter(isVisible);
}

// Dropdown item CSS selectors
// TODO: .dropdown-form handling
var ITEM_SELECTOR$1 = '.dropdown-item:not(.disabled):not([disabled])';

// Popper attachment positions
var AttachmentMap = {
  // DropUp Left Align
  TOP: 'top-start',
  // DropUp Right Align
  TOPEND: 'top-end',
  // Dropdown left Align
  BOTTOM: 'bottom-start',
  // Dropdown Right Align
  BOTTOMEND: 'bottom-end'
};

var dropdownMixin = {
  mixins: [clickoutMixin, listenOnRootMixin],
  props: {
    disabled: {
      type: Boolean,
      default: false
    },
    text: {
      // Button label
      type: String,
      default: ''
    },
    dropup: {
      // place on top if possible
      type: Boolean,
      default: false
    },
    right: {
      // Right align menu (default is left align)
      type: Boolean,
      default: false
    },
    offset: {
      // Number of pixels to offset menu, or a CSS unit value (i.e. 1px, 1rem, etc)
      type: [Number, String],
      default: 0
    },
    noFlip: {
      // Disable auto-flipping of menu from bottom<=>top
      type: Boolean,
      default: false
    },
    popperOpts: {
      type: Object,
      default: function _default() {}
    }
  },
  data: function data() {
    return {
      visible: false,
      _popper: null,
      inNavbar: null
    };
  },
  mounted: function mounted() {
    var _this = this;

    var listener = function listener(vm) {
      if (vm !== _this) {
        _this.visible = false;
      }
    };

    // To keep one dropdown opened on page
    this.listenOnRoot('bv::dropdown::shown', listener);

    // Hide when clicked on links
    this.listenOnRoot('clicked::link', listener);
    // Use new namespaced events
    this.listenOnRoot('bv::link::clicked', listener);
  },
  beforeDestroy: function beforeDestroy() {
    if (this._popper) {
      // Ensure popper event listeners are removed cleanly
      this._popper.destroy();
    }
    this._popper = null;
    this.setTouchStart(false);
  },

  watch: {
    visible: function visible(state, old) {
      if (state === old) {
        // Avoid duplicated emits
        return;
      }
      if (state) {
        this.showMenu();
      } else {
        this.hideMenu();
      }
    },
    disabled: function disabled(state, old) {
      if (state !== old && state && this.visible) {
        // Hide dropdown if disabled changes to true
        this.visible = false;
      }
    }
  },
  computed: {
    toggler: function toggler() {
      return this.$refs.toggle.$el || this.$refs.toggle;
    }
  },
  methods: {
    showMenu: function showMenu() {
      if (this.disabled) {
        return;
      }
      // TODO: move emit show to visible watcher, to allow cancelling of show
      this.$emit('show');
      // Ensure other menus are closed
      this.emitOnRoot('bv::dropdown::shown', this);

      // Are we in a navbar ?
      if (this.inNavbar === null && this.isNav) {
        this.inNavbar = Boolean(closest('.navbar', this.$el));
      }

      // Disable totally Popper.js for Dropdown in Navbar
      if (!this.inNavbar) {
        if (typeof Popper === 'undefined') {
          warn('b-dropdown: Popper.js not found. Falling back to CSS positioning.');
        } else {
          // for dropup with alignment we use the parent element as popper container
          var element = this.dropup && this.right || this.split ? this.$el : this.$refs.toggle;
          // Make sure we have a reference to an element, not a component!
          element = element.$el || element;
          // Instantiate popper.js
          this._popper = new Popper(element, this.$refs.menu, this.getPopperConfig());
        }
      }

      this.setTouchStart(true);
      this.$emit('shown');

      // Focus on the first item on show
      this.$nextTick(this.focusFirstItem);
    },
    hideMenu: function hideMenu() {
      // TODO: move emit hide to visible watcher, to allow cancelling of hide
      this.$emit('hide');
      if (this._popper) {
        // Ensure popper event listeners are removed cleanly
        this._popper.destroy();
      }
      this._popper = null;
      this.setTouchStart(false);
      this.emitOnRoot('bv::dropdown::hidden', this);
      this.$emit('hidden');
    },
    getPopperConfig: function getPopperConfig() {
      var placement = AttachmentMap.BOTTOM;
      if (this.dropup && this.right) {
        // dropup + right
        placement = AttachmentMap.TOPEND;
      } else if (this.dropup) {
        // dropup + left
        placement = AttachmentMap.TOP;
      } else if (this.right) {
        // dropdown + right
        placement = AttachmentMap.BOTTOMEND;
      }
      var popperConfig = {
        placement: placement,
        modifiers: {
          offset: {
            offset: this.offset || 0
          },
          flip: {
            enabled: !this.noFlip
          }
        }
      };
      return assign(popperConfig, this.popperOpts || {});
    },
    setTouchStart: function setTouchStart(on) {
      var _this2 = this;

      /*
       * If this is a touch-enabled device we add extra
       * empty mouseover listeners to the body's immediate children;
       * only needed because of broken event delegation on iOS
       * https://www.quirksmode.org/blog/archives/2014/02/mouse_event_bub.html
       */
      if ('ontouchstart' in document.documentElement) {
        var children = from(document.body.children);
        children.forEach(function (el) {
          if (on) {
            eventOn('mouseover', _this2._noop);
          } else {
            eventOff('mouseover', _this2._noop);
          }
        });
      }
    },
    _noop: function _noop() {
      // Do nothing event handler (used in touchstart event handler)
    },
    clickOutListener: function clickOutListener() {
      this.visible = false;
    },
    show: function show() {
      // Public method to show dropdown
      if (this.disabled) {
        return;
      }
      this.visible = true;
    },
    hide: function hide() {
      // Public method to hide dropdown
      if (this.disabled) {
        return;
      }
      this.visible = false;
    },
    toggle: function toggle(evt) {
      // Called only by a button that toggles the menu
      evt = evt || {};
      var type = evt.type;
      var key = evt.keyCode;
      if (type !== 'click' && !(type === 'keydown' && (key === KeyCodes.ENTER || key === KeyCodes.SPACE || key === KeyCodes.DOWN))) {
        // We only toggle on Click, Enter, Space, and Arrow Down
        return;
      }
      evt.preventDefault();
      evt.stopPropagation();
      if (this.disabled) {
        this.visible = false;
        return;
      }
      // Toggle visibility
      this.visible = !this.visible;
    },
    click: function click(evt) {
      // Calle only in split button mode, for the split button
      if (this.disabled) {
        this.visible = false;
        return;
      }
      this.$emit('click', evt);
    },
    onKeydown: function onKeydown(evt) {
      // Called from dropdown menu context
      var key = evt.keyCode;
      if (key === KeyCodes.ESC) {
        // Close on ESC
        this.onEsc(evt);
      } else if (key === KeyCodes.TAB) {
        // Close on tab out
        this.onTab(evt);
      } else if (key === KeyCodes.DOWN) {
        // Down Arrow
        this.focusNext(evt, false);
      } else if (key === KeyCodes.UP) {
        // Up Arrow
        this.focusNext(evt, true);
      }
    },
    onEsc: function onEsc(evt) {
      if (this.visible) {
        this.visible = false;
        evt.preventDefault();
        evt.stopPropagation();
        // Return focus to original trigger button
        this.$nextTick(this.focusToggler);
      }
    },
    onTab: function onTab(evt) {
      if (this.visible) {
        // TODO: Need special handler for dealing with form inputs
        // Tab, if in a text-like input, we should just focus next item in the dropdown
        // Note: Inputs are in a special .dropdown-form container
        this.visible = false;
      }
    },
    onFocusOut: function onFocusOut(evt) {
      if (this.$refs.menu.contains(evt.relatedTarget)) {
        return;
      }
      this.visible = false;
    },
    onMouseOver: function onMouseOver(evt) {
      // Focus the item on hover
      // TODO: Special handling for inputs? Inputs are in a special .dropdown-form container
      var item = evt.target;
      if (item.classList.contains('dropdown-item') && !item.disabled && !item.classList.contains('disabled') && item.focus) {
        item.focus();
      }
    },
    focusNext: function focusNext(evt, up) {
      var _this3 = this;

      if (!this.visible) {
        return;
      }
      evt.preventDefault();
      evt.stopPropagation();
      this.$nextTick(function () {
        var items = _this3.getItems();
        if (items.length < 1) {
          return;
        }
        var index = items.indexOf(evt.target);
        if (up && index > 0) {
          index--;
        } else if (!up && index < items.length - 1) {
          index++;
        }
        if (index < 0) {
          index = 0;
        }
        _this3.focusItem(index, items);
      });
    },
    focusItem: function focusItem(idx, items) {
      var el = items.find(function (el, i) {
        return i === idx;
      });
      if (el && getAttr(el, 'tabindex') !== '-1') {
        el.focus();
      }
    },
    getItems: function getItems() {
      // Get all items
      return filterVisible(selectAll(ITEM_SELECTOR$1, this.$refs.menu));
    },
    getFirstItem: function getFirstItem() {
      // Get the first non-disabled item
      var item = this.getItems()[0];
      return item || null;
    },
    focusFirstItem: function focusFirstItem() {
      var item = this.getFirstItem();
      if (item) {
        this.focusItem(0, [item]);
      }
    },
    focusToggler: function focusToggler() {
      var toggler = this.toggler;
      if (toggler && toggler.focus) {
        toggler.focus();
      }
    }
  }
};

var formMixin = {
  props: {
    name: {
      type: String
    },
    id: {
      type: String
    },
    disabled: {
      type: Boolean
    },
    required: {
      type: Boolean,
      default: false
    }
  }
};

var formCustomMixin = {
  computed: {
    custom: function custom() {
      return !this.plain;
    }
  },
  props: {
    plain: {
      type: Boolean,
      default: false
    }
  }
};

function isObject$1(obj) {
  return obj && {}.toString.call(obj) === '[object Object]';
}

var formOptionsMixin = {
  props: {
    options: {
      type: [Array, Object],
      default: function _default() {
        return [];
      }
    },
    valueField: {
      type: String,
      default: 'value'
    },
    textField: {
      type: String,
      default: 'text'
    },
    disabledField: {
      type: String,
      default: 'disabled'
    }
  },
  computed: {
    formOptions: function formOptions() {
      var options = this.options || [];

      var valueField = this.valueField || 'value';
      var textField = this.textField || 'text';
      var disabledField = this.disabledField || 'disabled';

      if (isArray(options)) {
        // Normalize flat-ish arrays to Array of Objects
        return options.map(function (option) {
          if (isObject$1(option)) {
            return {
              value: option[valueField],
              text: String(option[textField]),
              disabled: option[disabledField] || false
            };
          }
          return {
            text: String(option),
            value: option,
            disabled: false
          };
        });
      } else if (isObject$1(options)) {
        // Normalize Objects to Array of Objects
        return keys(options).map(function (key) {
          var option = options[key] || {};
          if (isObject$1(option)) {
            var value = option[valueField];
            var text = option[textField];
            return {
              text: typeof text === 'undefined' ? key : String(text),
              value: typeof value === 'undefined' ? key : value,
              disabled: option[disabledField] || false
            };
          }
          return {
            text: String(option),
            value: key,
            disabled: false
          };
        });
      }
      // Option unsupported type
      return [];
    }
  }
};

/*
 * form-radio & form-check mixin
 *
 */

var formRadioCheckMixin = {
  data: function data() {
    return {
      localChecked: this.checked,
      hasFocus: false
    };
  },

  model: {
    prop: 'checked',
    event: 'input'
  },
  props: {
    value: {},
    checked: {
      // This is the model, except when in group mode
    },
    buttonVariant: {
      // Only applicable when rendered with button style
      type: String,
      default: null
    }
  },
  computed: {
    computedLocalChecked: {
      get: function get() {
        if (this.is_Child) {
          return this.$parent.localChecked;
        } else {
          return this.localChecked;
        }
      },
      set: function set(val) {
        if (this.is_Child) {
          this.$parent.localChecked = val;
        } else {
          this.localChecked = val;
        }
      }
    },
    is_Child: function is_Child() {
      return Boolean(this.$parent && this.$parent.is_RadioCheckGroup);
    },
    is_Disabled: function is_Disabled() {
      // Child can be disabled while parent isn't
      return Boolean(this.is_Child ? this.$parent.disabled || this.disabled : this.disabled);
    },
    is_Required: function is_Required() {
      return Boolean(this.is_Child ? this.$parent.required : this.required);
    },
    is_Plain: function is_Plain() {
      return Boolean(this.is_Child ? this.$parent.plain : this.plain);
    },
    is_Custom: function is_Custom() {
      return !this.is_Plain;
    },
    get_Size: function get_Size() {
      return this.is_Child ? this.$parent.size : this.size;
    },
    get_State: function get_State() {
      // This is a tri-state prop (true, false, null)
      if (this.is_Child && typeof this.$parent.get_State === 'boolean') {
        return this.$parent.get_State;
      }
      return this.computedState;
    },
    get_StateClass: function get_StateClass() {
      // This is a tri-state prop (true, false, null)
      return typeof this.get_State === 'boolean' ? this.get_State ? 'is-valid' : 'is-invalid' : '';
    },
    is_Stacked: function is_Stacked() {
      return Boolean(this.is_Child && this.$parent.stacked);
    },
    is_Inline: function is_Inline() {
      return !this.is_Stacked;
    },
    is_ButtonMode: function is_ButtonMode() {
      return Boolean(this.is_Child && this.$parent.buttons);
    },
    get_ButtonVariant: function get_ButtonVariant() {
      // Local variant trumps parent variant
      return this.buttonVariant || (this.is_Child ? this.$parent.buttonVariant : null) || 'secondary';
    },
    get_Name: function get_Name() {
      return (this.is_Child ? this.$parent.name || this.$parent.safeId() : this.name) || null;
    },
    buttonClasses: function buttonClasses() {
      // Same for radio & check
      return ['btn', 'btn-' + this.get_ButtonVariant, this.get_Size ? 'btn-' + this.get_Size : '',
      // 'disabled' class makes "button" look disabled
      this.is_Disabled ? 'disabled' : '',
      // 'active' class makes "button" look pressed
      this.is_Checked ? 'active' : '',
      // Focus class makes button look focused
      this.hasFocus ? 'focus' : '',
      // Fix for issue in BS V4.beta.2 CSS (remove spacing between stacked buttons)
      this.is_Stacked ? 'mb-0' : ''];
    }
  },
  methods: {
    handleFocus: function handleFocus(evt) {
      // When in buttons mode, we need to add 'focus' class to label when radio focused
      if (this.is_ButtonMode && evt.target) {
        if (evt.type === 'focus') {
          this.hasFocus = true;
        } else if (evt.type === 'blur') {
          this.hasFocus = false;
        }
      }
    }
  }
};

var formSizeMixin = {
  props: {
    size: {
      type: String,
      default: null
    }
  },
  computed: {
    sizeFormClass: function sizeFormClass() {
      return [this.size ? "form-control-" + this.size : null];
    },
    sizeBtnClass: function sizeBtnClass() {
      return [this.size ? "btn-" + this.size : null];
    }
  }
};

/* Form control contextual state class computation
 *
 * Returned class is either 'is-valid' or 'is-invalid' based on the 'state' prop
 * state can be one of five values:
 *  - true or 'valid' for is-valid
 *  - false or 'invalid' for is-invalid
 *  - null (or empty string) for no contextual state
 */

var formStateMixin = {
  props: {
    state: {
      // true/'valid', false/'invalid', '',null
      type: [Boolean, String],
      default: null
    }
  },
  computed: {
    computedState: function computedState() {
      var state = this.state;
      if (state === true || state === 'valid') {
        return true;
      } else if (state === false || state === 'invalid') {
        return false;
      }
      return null;
    },
    stateClass: function stateClass() {
      var state = this.computedState;
      if (state === true) {
        return 'is-valid';
      } else if (state === false) {
        return 'is-invalid';
      }
      return null;
    }
  }
};

/*
 * SSR Safe Client Side ID attribute generation
 *
 */

var idMixin = {
  props: {
    id: {
      type: String,
      default: null
    }
  },
  data: function data() {
    return {
      localId_: null
    };
  },
  mounted: function mounted() {
    if (!this.$isServer && !this.id && this._uid) {
      this.localId_ = '__BVID__' + this._uid + '_';
    }
  },

  methods: {
    safeId: function safeId() {
      var suffix = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : '';

      var id = this.id || this.localId_ || null;
      if (!id) {
        return null;
      }
      suffix = String(suffix).replace(/\s+/g, '_');
      return suffix ? id + '_' + suffix : id;
    }
  }
};

/**
 * @param {number} length
 * @return {Array}
 */
var range = (function (length) {
  return Array.apply(null, { length: length });
});

/*
 * Comon props, computed, data, render function, and methods for b-pagination and b-pagination-nav
 */

// Make an array of N to N+X
function makePageArray(startNum, numPages) {
  return range(numPages).map(function (value, index) {
    return { number: index + startNum, className: null };
  });
}

// Threshold of limit size when we start/stop showing ellipsis
var ELLIPSIS_THRESHOLD = 3;

// Props object
var props$12 = {
  disabled: {
    type: Boolean,
    default: false
  },
  value: {
    type: Number,
    default: 1
  },
  limit: {
    type: Number,
    default: 5
  },
  size: {
    type: String,
    default: 'md'
  },
  align: {
    type: String,
    default: 'left'
  },
  hideGotoEndButtons: {
    type: Boolean,
    default: false
  },
  ariaLabel: {
    type: String,
    default: 'Pagination'
  },
  labelFirstPage: {
    type: String,
    default: 'Goto first page'
  },
  firstText: {
    type: String,
    default: '&laquo;'
  },
  labelPrevPage: {
    type: String,
    default: 'Goto previous page'
  },
  prevText: {
    type: String,
    default: '&lsaquo;'
  },
  labelNextPage: {
    type: String,
    default: 'Goto next page'
  },
  nextText: {
    type: String,
    default: '&rsaquo;'
  },
  labelLastPage: {
    type: String,
    default: 'Goto last page'
  },
  lastText: {
    type: String,
    default: '&raquo;'
  },
  labelPage: {
    type: String,
    default: 'Goto page'
  },
  hideEllipsis: {
    type: Boolean,
    default: false
  },
  ellipsisText: {
    type: String,
    default: '&hellip;'
  }
};

var paginationMixin = {
  components: { bLink: Link },
  data: function data() {
    return {
      showFirstDots: false,
      showLastDots: false,
      currentPage: this.value
    };
  },

  props: props$12,
  render: function render(h) {
    var t = this;
    var buttons = [];

    // Factory function for prev/next/first/last buttons
    var makeEndBtns = function makeEndBtns(linkTo, ariaLabel, btnText, pageTest) {
      var button = void 0;
      pageTest = pageTest || linkTo; // Page # to test against to disable
      if (t.disabled || t.isActive(pageTest)) {
        button = h('li', {
          class: ['page-item', 'disabled'],
          attrs: { role: 'none presentation', 'aria-hidden': 'true' }
        }, [h('span', { class: ['page-link'], domProps: { innerHTML: btnText } })]);
      } else {
        button = h('li', {
          class: ['page-item'],
          attrs: { role: 'none presentation' }
        }, [h('b-link', {
          class: ['page-link'],
          props: t.linkProps(linkTo),
          attrs: {
            role: 'menuitem',
            tabindex: '-1',
            'aria-label': ariaLabel,
            'aria-controls': t.ariaControls || null
          },
          on: {
            click: function click(evt) {
              t.onClick(linkTo, evt);
            },
            keydown: function keydown(evt) {
              // Links don't normally respond to SPACE, so we add that functionality
              if (evt.keyCode === KeyCodes.SPACE) {
                evt.preventDefault();
                t.onClick(linkTo, evt);
              }
            }
          }
        }, [h('span', { attrs: { 'aria-hidden': 'true' }, domProps: { innerHTML: btnText } })])]);
      }
      return button;
    };

    // Ellipsis factory
    var makeEllipsis = function makeEllipsis() {
      return h('li', {
        class: ['page-item', 'disabled', 'd-none', 'd-sm-flex'],
        attrs: { role: 'separator' }
      }, [h('span', { class: ['page-link'], domProps: { innerHTML: t.ellipsisText } })]);
    };

    // Goto First Page button
    buttons.push(t.hideGotoEndButtons ? h(false) : makeEndBtns(1, t.labelFirstPage, t.firstText));

    // Goto Previous page button
    buttons.push(makeEndBtns(t.currentPage - 1, t.labelPrevPage, t.prevText, 1));

    // First Ellipsis Bookend
    buttons.push(t.showFirstDots ? makeEllipsis() : h(false));

    // Individual Page links
    t.pageList.forEach(function (page) {
      var inner = void 0;
      var pageNum = t.makePage(page.number);
      if (t.disabled) {
        inner = h('span', { class: ['page-link'], domProps: { innerHTML: pageNum } });
      } else {
        var active = t.isActive(page.number);
        inner = h('b-link', {
          class: t.pageLinkClasses(page),
          props: t.linkProps(page.number),
          attrs: {
            role: 'menuitemradio',
            tabindex: active ? '0' : '-1',
            'aria-controls': t.ariaControls || null,
            'aria-label': t.labelPage + ' ' + page.number,
            'aria-checked': active ? 'true' : 'false',
            'aria-posinset': page.number,
            'aria-setsize': t.numberOfPages
          },
          domProps: { innerHTML: pageNum },
          on: {
            click: function click(evt) {
              t.onClick(page.number, evt);
            },
            keydown: function keydown(evt) {
              if (evt.keyCode === KeyCodes.SPACE) {
                evt.preventDefault();
                t.onClick(page.number, evt);
              }
            }
          }
        });
      }
      buttons.push(h('li', {
        key: page.number,
        class: t.pageItemClasses(page),
        attrs: { role: 'none presentation' }
      }, [inner]));
    });

    // Last Ellipsis Bookend
    buttons.push(t.showLastDots ? makeEllipsis() : h(false));

    // Goto Next page button
    buttons.push(makeEndBtns(t.currentPage + 1, t.labelNextPage, t.nextText, t.numberOfPages));

    // Goto Last Page button
    buttons.push(t.hideGotoEndButtons ? h(false) : makeEndBtns(t.numberOfPages, t.labelLastPage, t.lastText));

    // Assemble the paginatiom buttons
    var pagination = h('ul', {
      ref: 'ul',
      class: ['pagination', 'b-pagination', t.btnSize, t.alignment],
      attrs: {
        role: 'menubar',
        'aria-disabled': t.disabled ? 'true' : 'false',
        'aria-label': t.ariaLabel || null
      },
      on: {
        keydown: function keydown(evt) {
          var keyCode = evt.keyCode;
          var shift = evt.shiftKey;
          if (keyCode === KeyCodes.LEFT) {
            evt.preventDefault();
            shift ? t.focusFirst() : t.focusPrev();
          } else if (keyCode === KeyCodes.RIGHT) {
            evt.preventDefault();
            shift ? t.focusLast() : t.focusNext();
          }
        }
      }
    }, buttons);

    // if we are pagination-nav, wrap in '<nav>' wrapper
    return t.isNav ? h('nav', {}, [pagination]) : pagination;
  },

  watch: {
    currentPage: function currentPage(newPage, oldPage) {
      if (newPage !== oldPage) {
        this.$emit('input', newPage);
      }
    },
    value: function value(newValue, oldValue) {
      if (newValue !== oldValue) {
        this.currentPage = newValue;
      }
    }
  },
  computed: {
    btnSize: function btnSize() {
      return this.size ? 'pagination-' + this.size : '';
    },
    alignment: function alignment() {
      if (this.align === 'center') {
        return 'justify-content-center';
      } else if (this.align === 'end' || this.align === 'right') {
        return 'justify-content-end';
      }
      return '';
    },
    pageList: function pageList() {
      // Sanity checks
      if (this.currentPage > this.numberOfPages) {
        this.currentPage = this.numberOfPages;
      } else if (this.currentPage < 1) {
        this.currentPage = 1;
      }
      // - Hide first ellipsis marker
      this.showFirstDots = false;
      // - Hide last ellipsis marker
      this.showLastDots = false;
      var numLinks = this.limit;
      var startNum = 1;
      if (this.numberOfPages <= this.limit) {
        // Special Case: Less pages available than the limit of displayed pages
        numLinks = this.numberOfPages;
      } else if (this.currentPage < this.limit - 1 && this.limit > ELLIPSIS_THRESHOLD) {
        // We are near the beginning of the page list
        if (!this.hideEllipsis) {
          numLinks = this.limit - 1;
          this.showLastDots = true;
        }
      } else if (this.numberOfPages - this.currentPage + 2 < this.limit && this.limit > ELLIPSIS_THRESHOLD) {
        // We are near the end of the list
        if (!this.hideEllipsis) {
          this.showFirstDots = true;
          numLinks = this.limit - 1;
        }
        startNum = this.numberOfPages - numLinks + 1;
      } else {
        // We are somewhere in the middle of the page list
        if (this.limit > ELLIPSIS_THRESHOLD && !this.hideEllipsis) {
          this.showFirstDots = true;
          this.showLastDots = true;
          numLinks = this.limit - 2;
        }
        startNum = this.currentPage - Math.floor(numLinks / 2);
      }
      // Sanity checks
      if (startNum < 1) {
        startNum = 1;
      } else if (startNum > this.numberOfPages - numLinks) {
        startNum = this.numberOfPages - numLinks + 1;
      }
      // Generate list of page numbers
      var pages = makePageArray(startNum, numLinks);
      // We limit to a total of 3 page buttons on small screens
      // Ellipsis will also be hidden on small screens
      if (pages.length > 3) {
        var idx = this.currentPage - startNum;
        if (idx === 0) {
          // Keep leftmost 3 buttons visible
          for (var i = 3; i < pages.length; i++) {
            pages[i].className = 'd-none d-sm-flex';
          }
        } else if (idx === pages.length - 1) {
          // Keep rightmost 3 buttons visible
          for (var _i = 0; _i < pages.length - 3; _i++) {
            pages[_i].className = 'd-none d-sm-flex';
          }
        } else {
          // hide left button(s)
          for (var _i2 = 0; _i2 < idx - 1; _i2++) {
            pages[_i2].className = 'd-none d-sm-flex';
          }
          // hide right button(s)
          for (var _i3 = pages.length - 1; _i3 > idx + 1; _i3--) {
            pages[_i3].className = 'd-none d-sm-flex';
          }
        }
      }
      return pages;
    }
  },
  methods: {
    isActive: function isActive(pagenum) {
      return pagenum === this.currentPage;
    },
    pageItemClasses: function pageItemClasses(page) {
      return ['page-item', this.disabled ? 'disabled' : '', this.isActive(page.number) ? 'active' : '', page.className];
    },
    pageLinkClasses: function pageLinkClasses(page) {
      return ['page-link', this.disabled ? 'disabled' : '',
      // Interim workaround to get better focus styling of active button
      // See https://github.com/twbs/bootstrap/issues/24838
      this.isActive(page.number) ? 'btn-primary' : ''];
    },
    getButtons: function getButtons() {
      // Return only buttons that are visible
      return selectAll('a.page-link', this.$el).filter(function (btn) {
        return isVisible(btn);
      });
    },
    setBtnFocus: function setBtnFocus(btn) {
      this.$nextTick(function () {
        btn.focus();
      });
    },
    focusCurrent: function focusCurrent() {
      var _this = this;

      var btn = this.getButtons().find(function (el) {
        return parseInt(getAttr(el, 'aria-posinset'), 10) === _this.currentPage;
      });
      if (btn && btn.focus) {
        this.setBtnFocus(btn);
      } else {
        // Fallback if current page is not in button list
        this.focusFirst();
      }
    },
    focusFirst: function focusFirst() {
      var btn = this.getButtons().find(function (el) {
        return !isDisabled(el);
      });
      if (btn && btn.focus && btn !== document.activeElement) {
        this.setBtnFocus(btn);
      }
    },
    focusLast: function focusLast() {
      var btn = this.getButtons().reverse().find(function (el) {
        return !isDisabled(el);
      });
      if (btn && btn.focus && btn !== document.activeElement) {
        this.setBtnFocus(btn);
      }
    },
    focusPrev: function focusPrev() {
      var buttons = this.getButtons();
      var idx = buttons.indexOf(document.activeElement);
      if (idx > 0 && !isDisabled(buttons[idx - 1]) && buttons[idx - 1].focus) {
        this.setBtnFocus(buttons[idx - 1]);
      }
    },
    focusNext: function focusNext() {
      var buttons = this.getButtons();
      var idx = buttons.indexOf(document.activeElement);
      var cnt = buttons.length - 1;
      if (idx < cnt && !isDisabled(buttons[idx + 1]) && buttons[idx + 1].focus) {
        this.setBtnFocus(buttons[idx + 1]);
      }
    }
  }
};

/*
 * Tooltip/Popover component mixin
 * Common props
 */
var PLACEMENTS = {
  top: 'top',
  topleft: 'topleft',
  topright: 'topright',
  right: 'right',
  righttop: 'righttop',
  rightbottom: 'rightbottom',
  bottom: 'bottom',
  bottomleft: 'bottomleft',
  bottomright: 'bottomright',
  left: 'left',
  lefttop: 'lefttop',
  leftbottom: 'leftbottom',
  auto: 'auto'
};

var OBSERVER_CONFIG = {
  subtree: true,
  childList: true,
  characterData: true,
  attributes: true,
  attributeFilter: ['class', 'style']
};

var toolpopMixin = {
  props: {
    target: {
      // String ID of element, or element/component reference
      type: [String, Object]
    },
    delay: {
      type: [Number, Object, String],
      default: 0
    },
    offset: {
      type: [Number, String],
      default: 0
    },
    noFade: {
      type: Boolean,
      default: false
    },
    container: {
      // String ID of container, if null body is used (default)
      type: String,
      default: null
    },
    show: {
      type: Boolean,
      default: false
    },
    disabled: {
      type: Boolean,
      default: false
    }
  },
  watch: {
    show: function show(_show, old) {
      if (_show === old) {
        return;
      }
      _show ? this.onOpen() : this.onClose();
    },
    disabled: function disabled(_disabled, old) {
      if (_disabled === old) {
        return;
      }
      _disabled ? this.onDisable() : this.onEnable();
    }
  },
  created: function created() {
    // Create non-reactive property
    this._toolpop = null;
    this._obs_title = null;
    this._obs_content = null;
  },
  mounted: function mounted() {
    var _this = this;

    // We do this in a next tick to ensure DOM has rendered first
    this.$nextTick(function () {
      // Instantiate ToolTip/PopOver on target
      // The createToolpop method must exist in main component
      if (_this.createToolpop()) {
        if (_this.disabled) {
          // Initially disabled
          _this.onDisable();
        }
        // Listen to open signals from others
        _this.$on('open', _this.onOpen);
        // Listen to close signals from others
        _this.$on('close', _this.onClose);
        // Listen to disable signals from others
        _this.$on('disable', _this.onDisable);
        // Listen to disable signals from others
        _this.$on('enable', _this.onEnable);
        // Observe content Child changes so we can notify popper of possible size change
        _this.setObservers(true);
        // Set intially open state
        if (_this.show) {
          _this.onOpen();
        }
      }
    });
  },
  updated: function updated() {
    // If content/props changes, etc
    if (this._toolpop) {
      this._toolpop.updateConfig(this.getConfig());
    }
  },
  activated: function activated() {
    // Called when component is inside a <keep-alive> and component brought offline
    this.setObservers(true);
  },
  deactivated: function deactivated() {
    // Called when component is inside a <keep-alive> and component taken offline
    if (this._toolpop) {
      this.setObservers(false);
      this._toolpop.hide();
    }
  },
  beforeDestroy: function beforeDestroy() {
    // Shutdown our local event listeners
    this.$off('open', this.onOpen);
    this.$off('close', this.onClose);
    this.$off('disable', this.onDisable);
    this.$off('enable', this.onEnable);
    this.setObservers(false);
    // bring our content back if needed
    this.bringItBack();
    if (this._toolpop) {
      this._toolpop.destroy();
      this._toolpop = null;
    }
  },

  computed: {
    baseConfig: function baseConfig() {
      var cont = this.container;
      var delay = _typeof(this.delay) === 'object' ? this.delay : parseInt(this.delay, 10) || 0;
      return {
        // Title prop
        title: (this.title || '').trim() || '',
        // Contnt prop (if popover)
        content: (this.content || '').trim() || '',
        // Tooltip/Popover placement
        placement: PLACEMENTS[this.placement] || 'auto',
        // Container curently needs to be an ID with '#' prepended, if null then body is used
        container: cont ? /^#/.test(cont) ? cont : '#' + cont : false,
        // Show/Hide delay
        delay: delay || 0,
        // Offset can be css distance. if no units, pixels are assumed
        offset: this.offset || 0,
        // Disable fade Animation?
        animation: !this.noFade,
        // Open/Close Trigger(s)
        trigger: isArray(this.triggers) ? this.triggers.join(' ') : this.triggers,
        // Callbacks so we can trigger events on component
        callbacks: {
          show: this.onShow,
          shown: this.onShown,
          hide: this.onHide,
          hidden: this.onHidden,
          enabled: this.onEnabled,
          disabled: this.onDisabled
        }
      };
    }
  },
  methods: {
    getConfig: function getConfig() {
      var cfg = assign({}, this.baseConfig);
      if (this.$refs.title && this.$refs.title.innerHTML.trim()) {
        // If slot has content, it overrides 'title' prop
        // We use the DOM node as content to allow components!
        cfg.title = this.$refs.title;
        cfg.html = true;
      }
      if (this.$refs.content && this.$refs.content.innerHTML.trim()) {
        // If slot has content, it overrides 'content' prop
        // We use the DOM node as content to allow components!
        cfg.content = this.$refs.content;
        cfg.html = true;
      }
      return cfg;
    },
    onOpen: function onOpen() {
      if (this._toolpop) {
        this._toolpop.show();
      }
    },
    onClose: function onClose(callback) {
      if (this._toolpop) {
        this._toolpop.hide(callback);
      } else if (typeof callback === 'function') {
        callback();
      }
    },
    onDisable: function onDisable() {
      if (this._toolpop) {
        this._toolpop.disable();
      }
    },
    onEnable: function onEnable() {
      if (this._toolpop) {
        this._toolpop.enable();
      }
    },
    updatePosition: function updatePosition() {
      if (this._toolpop) {
        // Instruct popper to reposition popover if necessary
        this._toolpop.update();
      }
    },
    getTarget: function getTarget() {
      var target = this.target;
      if (typeof target === 'string') {
        // Assume ID of element
        return getById(target);
      } else if ((typeof target === 'undefined' ? 'undefined' : _typeof(target)) === 'object' && isElement(target.$el)) {
        // Component reference
        return target.$el;
      } else if ((typeof target === 'undefined' ? 'undefined' : _typeof(target)) === 'object' && isElement(target)) {
        // Element reference
        return target;
      }
      return null;
    },
    onShow: function onShow(evt) {
      this.$emit('show', evt);
    },
    onShown: function onShown(evt) {
      this.setObservers(true);
      this.$emit('update:show', true);
      this.$emit('shown', evt);
    },
    onHide: function onHide(evt) {
      this.$emit('hide', evt);
    },
    onHidden: function onHidden(evt) {
      this.setObservers(false);
      // bring our content back if needed to keep Vue happy
      // Tooltip class will move it back to tip when shown again
      this.bringItBack();
      this.$emit('update:show', false);
      this.$emit('hidden', evt);
    },
    onEnabled: function onEnabled(evt) {
      if (!evt || evt.type !== 'enabled') {
        // Prevent possible endless loop if user mistakienly fires enabled instead of enable
        return;
      }
      this.$emit('update:disabled', false);
      this.$emit('disabled');
    },
    onDisabled: function onDisabled(evt) {
      if (!evt || evt.type !== 'disabled') {
        // Prevent possible endless loop if user mistakienly fires disabled instead of disable
        return;
      }
      this.$emit('update:disabled', true);
      this.$emit('enabled');
    },
    bringItBack: function bringItBack() {
      // bring our content back if needed to keep Vue happy
      if (this.$el && this.$refs.title) {
        this.$el.appendChild(this.$refs.title);
      }
      if (this.$el && this.$refs.content) {
        this.$el.appendChild(this.$refs.content);
      }
    },
    setObservers: function setObservers(on) {
      if (on) {
        if (this.$refs.title) {
          this._obs_title = observeDOM(this.$refs.title, this.updatePosition.bind(this), OBSERVER_CONFIG);
        }
        if (this.$refs.content) {
          this._obs_content = observeDOM(this.$refs.content, this.updatePosition.bind(this), OBSERVER_CONFIG);
        }
      } else {
        if (this._obs_title) {
          this._obs_title.disconnect();
          this._obs_title = null;
        }
        if (this._obs_content) {
          this._obs_content.disconnect();
          this._obs_content = null;
        }
      }
    }
  }
};

var props$13 = assign({}, copyProps(cardMixin.props, prefixPropName.bind(null, 'body')), {
  bodyClass: {
    type: [String, Object, Array],
    default: null
  },
  title: {
    type: String,
    default: null
  },
  titleTag: {
    type: String,
    default: 'h4'
  },
  subTitle: {
    type: String,
    default: null
  },
  subTitleTag: {
    type: String,
    default: 'h6'
  },
  overlay: {
    type: Boolean,
    default: false
  }
});

var CardBody = {
  functional: true,
  props: props$13,
  render: function render(h, _ref) {
    var _ref2;

    var props = _ref.props,
        data = _ref.data,
        slots = _ref.slots;

    var cardBodyChildren = [];
    if (props.title) {
      cardBodyChildren.push(h(props.titleTag, {
        staticClass: 'card-title',
        domProps: { innerHTML: props.title }
      }));
    }
    if (props.subTitle) {
      cardBodyChildren.push(h(props.subTitleTag, {
        staticClass: 'card-subtitle mb-2 text-muted',
        domProps: { innerHTML: props.subTitle }
      }));
    }
    cardBodyChildren.push(slots().default);

    return h(props.bodyTag, lib_common(data, {
      staticClass: 'card-body',
      class: [(_ref2 = {
        'card-img-overlay': props.overlay
      }, defineProperty$1(_ref2, 'bg-' + props.bodyBgVariant, Boolean(props.bodyBgVariant)), defineProperty$1(_ref2, 'border-' + props.bodyBorderVariant, Boolean(props.bodyBorderVariant)), defineProperty$1(_ref2, 'text-' + props.bodyTextVariant, Boolean(props.bodyTextVariant)), _ref2), props.bodyClass || {}]
    }), cardBodyChildren);
  }
};

var props$14 = assign({}, copyProps(cardMixin.props, prefixPropName.bind(null, 'header')), {
  header: {
    type: String,
    default: null
  },
  headerClass: {
    type: [String, Object, Array],
    default: null
  }
});

var CardHeader = {
  functional: true,
  props: props$14,
  render: function render(h, _ref) {
    var _ref2;

    var props = _ref.props,
        data = _ref.data,
        slots = _ref.slots;

    return h(props.headerTag, lib_common(data, {
      staticClass: 'card-header',
      class: [props.headerClass, (_ref2 = {}, defineProperty$1(_ref2, 'bg-' + props.headerBgVariant, Boolean(props.headerBgVariant)), defineProperty$1(_ref2, 'border-' + props.headerBorderVariant, Boolean(props.headerBorderVariant)), defineProperty$1(_ref2, 'text-' + props.headerTextVariant, Boolean(props.headerTextVariant)), _ref2)]
    }), slots().default || [h('div', { domProps: { innerHTML: props.header } })]);
  }
};

var props$15 = assign({}, copyProps(cardMixin.props, prefixPropName.bind(null, 'footer')), {
  footer: {
    type: String,
    default: null
  },
  footerClass: {
    type: [String, Object, Array],
    default: null
  }
});

var CardFooter = {
  functional: true,
  props: props$15,
  render: function render(h, _ref) {
    var _ref2;

    var props = _ref.props,
        data = _ref.data,
        slots = _ref.slots;

    return h(props.footerTag, lib_common(data, {
      staticClass: 'card-footer',
      class: [props.footerClass, (_ref2 = {}, defineProperty$1(_ref2, 'bg-' + props.footerBgVariant, Boolean(props.footerBgVariant)), defineProperty$1(_ref2, 'border-' + props.footerBorderVariant, Boolean(props.footerBorderVariant)), defineProperty$1(_ref2, 'text-' + props.footerTextVariant, Boolean(props.footerTextVariant)), _ref2)]
    }), slots().default || [h('div', { domProps: { innerHTML: props.footer } })]);
  }
};

var props$16 = {
  src: {
    type: String,
    default: null,
    required: true
  },
  alt: {
    type: String,
    default: null
  },
  top: {
    type: Boolean,
    default: false
  },
  bottom: {
    type: Boolean,
    default: false
  },
  fluid: {
    type: Boolean,
    default: false
  }
};

var CardImg = {
  functional: true,
  props: props$16,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        slots = _ref.slots;

    var staticClass = 'card-img';
    if (props.top) {
      staticClass += '-top';
    } else if (props.bottom) {
      staticClass += '-bottom';
    }

    return h('img', lib_common(data, {
      staticClass: staticClass,
      class: { 'img-fluid': props.fluid },
      attrs: { src: props.src, alt: props.alt }
    }));
  }
};

var cardImgProps = copyProps(props$16, prefixPropName.bind(null, 'img'));
cardImgProps.imgSrc.required = false;

var props$11 = assign({}, props$13, props$14, props$15, cardImgProps, copyProps(cardMixin.props), {
  align: {
    type: String,
    default: null
  },
  noBody: {
    type: Boolean,
    default: false
  }
});

var bCard = {
  functional: true,
  props: props$11,
  render: function render(h, _ref) {
    var _class;

    var props$$1 = _ref.props,
        data = _ref.data,
        slots = _ref.slots;

    // The order of the conditionals matter.
    // We are building the component markup in order.
    var childNodes = [];
    var img = props$$1.imgSrc ? h(CardImg, { props: pluckProps(cardImgProps, props$$1, unPrefixPropName.bind(null, 'img')) }) : null;

    if (img) {
      // Above the header placement.
      if (props$$1.imgTop || !props$$1.imgBottom) {
        childNodes.push(img);
      }
    }
    if (props$$1.header || slots().header) {
      childNodes.push(h(CardHeader, { props: pluckProps(props$14, props$$1) }, slots().header));
    }
    if (props$$1.noBody) {
      childNodes.push(slots().default);
    } else {
      childNodes.push(h(CardBody, { props: pluckProps(props$13, props$$1) }, slots().default));
    }
    if (props$$1.footer || slots().footer) {
      childNodes.push(h(CardFooter, { props: pluckProps(props$15, props$$1) }, slots().footer));
    }
    if (img && props$$1.imgBottom) {
      // Below the footer placement.
      childNodes.push(img);
    }

    return h(props$$1.tag, lib_common(data, {
      staticClass: 'card',
      class: (_class = {}, defineProperty$1(_class, 'text-' + props$$1.align, Boolean(props$$1.align)), defineProperty$1(_class, 'bg-' + props$$1.bgVariant, Boolean(props$$1.bgVariant)), defineProperty$1(_class, 'border-' + props$$1.borderVariant, Boolean(props$$1.borderVariant)), defineProperty$1(_class, 'text-' + props$$1.textVariant, Boolean(props$$1.textVariant)), _class)
    }), childNodes);
  }
};

var props$17 = {
  tag: {
    type: String,
    default: 'div'
  },
  deck: {
    type: Boolean,
    default: false
  },
  columns: {
    type: Boolean,
    default: false
  }
};

var bCardGroup = {
  functional: true,
  props: props$17,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    var staticClass = 'card-group';
    if (props.columns) {
      staticClass = 'card-columns';
    }
    if (props.deck) {
      staticClass = 'card-deck';
    }

    return h(props.tag, lib_common(data, { staticClass: staticClass }), children);
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$8 = {
  bCard: bCard,
  bCardHeader: CardHeader,
  bCardBody: CardBody,
  bCardFooter: CardFooter,
  bCardImg: CardImg,
  bCardGroup: bCardGroup
};

var VuePlugin$16 = {
  install: function install(Vue) {
    registerComponents(Vue, components$8);
  }
};

vueUse(VuePlugin$16);

// Slide directional classes
var DIRECTION = {
  next: {
    dirClass: 'carousel-item-left',
    overlayClass: 'carousel-item-next'
  },
  prev: {
    dirClass: 'carousel-item-right',
    overlayClass: 'carousel-item-prev'
  }

  // Fallback Transition duration (with a little buffer) in ms
};var TRANS_DURATION = 600 + 50;

// Transition Event names
var TransitionEndEvents = {
  WebkitTransition: 'webkitTransitionEnd',
  MozTransition: 'transitionend',
  OTransition: 'otransitionend oTransitionEnd',
  transition: 'transitionend'

  // Return the browser specific transitionend event name
};function getTransisionEndEvent(el) {
  for (var name in TransitionEndEvents) {
    if (el.style[name] !== undefined) {
      return TransitionEndEvents[name];
    }
  }
  // fallback
  return null;
}

var bCarousel = {
  mixins: [idMixin],
  render: function render(h) {
    var t = this;

    // Wrapper for slides
    var inner = h('div', {
      ref: 'inner',
      class: ['carousel-inner'],
      attrs: {
        id: t.safeId('__BV_inner_'),
        role: 'list'
      }
    }, [t.$slots.default]);

    // Prev and Next Controls
    var controls = h(false);
    if (t.controls) {
      controls = [h('a', {
        class: ['carousel-control-prev'],
        attrs: { href: '#', role: 'button', 'aria-controls': t.safeId('__BV_inner_') },
        on: {
          click: function click(evt) {
            evt.preventDefault();
            evt.stopPropagation();
            t.prev();
          },
          keydown: function keydown(evt) {
            var keyCode = evt.keyCode;
            if (keyCode === KeyCodes.SPACE || keyCode === KeyCodes.ENTER) {
              evt.preventDefault();
              evt.stopPropagation();
              t.prev();
            }
          }
        }
      }, [h('span', { class: ['carousel-control-prev-icon'], attrs: { 'aria-hidden': 'true' } }), h('span', { class: ['sr-only'] }, [t.labelPrev])]), h('a', {
        class: ['carousel-control-next'],
        attrs: { href: '#', role: 'button', 'aria-controls': t.safeId('__BV_inner_') },
        on: {
          click: function click(evt) {
            evt.preventDefault();
            evt.stopPropagation();
            t.next();
          },
          keydown: function keydown(evt) {
            var keyCode = evt.keyCode;
            if (keyCode === KeyCodes.SPACE || keyCode === KeyCodes.ENTER) {
              evt.preventDefault();
              evt.stopPropagation();
              t.next();
            }
          }
        }
      }, [h('span', { class: ['carousel-control-next-icon'], attrs: { 'aria-hidden': 'true' } }), h('span', { class: ['sr-only'] }, [t.labelNext])])];
    }

    // Indicators
    var indicators = h('ol', {
      class: ['carousel-indicators'],
      directives: [{ name: 'show', rawName: 'v-show', value: t.indicators, expression: 'indicators' }],
      attrs: {
        id: t.safeId('__BV_indicators_'),
        'aria-hidden': t.indicators ? 'false' : 'true',
        'aria-label': t.labelIndicators,
        'aria-owns': t.safeId('__BV_inner_')
      }
    }, t.slides.map(function (slide, n) {
      return h('li', {
        key: 'slide_' + n,
        class: { active: n === t.index },
        attrs: {
          role: 'button',
          id: t.safeId('__BV_indicator_' + (n + 1) + '_'),
          tabindex: t.indicators ? '0' : '-1',
          'aria-current': n === t.index ? 'true' : 'false',
          'aria-label': t.labelGotoSlide + ' ' + (n + 1),
          'aria-describedby': t.slides[n].id || null,
          'aria-controls': t.safeId('__BV_inner_')
        },
        on: {
          click: function click(evt) {
            t.setSlide(n);
          },
          keydown: function keydown(evt) {
            var keyCode = evt.keyCode;
            if (keyCode === KeyCodes.SPACE || keyCode === KeyCodes.ENTER) {
              evt.preventDefault();
              evt.stopPropagation();
              t.setSlide(n);
            }
          }
        }
      });
    }));

    // Return the carousel
    return h('div', {
      class: ['carousel', 'slide'],
      style: { background: t.background },
      attrs: {
        role: 'region',
        id: t.safeId(),
        'aria-busy': t.isSliding ? 'true' : 'false'
      },
      on: {
        mouseenter: t.pause,
        mouseleave: t.restart,
        focusin: t.pause,
        focusout: t.restart,
        keydown: function keydown(evt) {
          var keyCode = evt.keyCode;
          if (keyCode === KeyCodes.LEFT || keyCode === KeyCodes.RIGHT) {
            evt.preventDefault();
            evt.stopPropagation();
            t[keyCode === KeyCodes.LEFT ? 'prev' : 'next']();
          }
        }
      }
    }, [inner, controls, indicators]);
  },
  data: function data() {
    return {
      index: this.value || 0,
      isSliding: false,
      intervalId: null,
      transitionEndEvent: null,
      slides: []
    };
  },

  props: {
    labelPrev: {
      type: String,
      default: 'Previous Slide'
    },
    labelNext: {
      type: String,
      default: 'Next Slide'
    },
    labelGotoSlide: {
      type: String,
      default: 'Goto Slide'
    },
    labelIndicators: {
      type: String,
      default: 'Select a slide to display'
    },
    interval: {
      type: Number,
      default: 5000
    },
    indicators: {
      type: Boolean,
      default: false
    },
    controls: {
      type: Boolean,
      default: false
    },
    imgWidth: {
      // Sniffed by carousel-slide
      type: [Number, String]
    },
    imgHeight: {
      // Sniffed by carousel-slide
      type: [Number, String]
    },
    background: {
      type: String
    },
    value: {
      type: Number,
      default: 0
    }
  },
  computed: {
    isCycling: function isCycling() {
      return Boolean(this.intervalId);
    }
  },
  methods: {
    // Set slide
    setSlide: function setSlide(slide) {
      var _this = this;

      // Don't animate when page is not visible
      if (typeof document !== 'undefined' && document.visibilityState && document.hidden) {
        return;
      }
      var len = this.slides.length;
      // Don't do anything if nothing to slide to
      if (len === 0) {
        return;
      }
      // Don't change slide while transitioning, wait until transition is done
      if (this.isSliding) {
        // Schedule slide after sliding complete
        this.$once('sliding-end', function () {
          return _this.setSlide(slide);
        });
        return;
      }
      // Make sure we have an integer (you never know!)
      slide = Math.floor(slide);
      // Set new slide index. Wrap around if necessary
      this.index = slide >= len ? 0 : slide >= 0 ? slide : len - 1;
    },

    // Previous slide
    prev: function prev() {
      this.setSlide(this.index - 1);
    },

    // Next slide
    next: function next() {
      this.setSlide(this.index + 1);
    },

    // Pause auto rotation
    pause: function pause() {
      if (this.isCycling) {
        clearInterval(this.intervalId);
        this.intervalId = null;
        if (this.slides[this.index]) {
          // Make current slide focusable for screen readers
          this.slides[this.index].tabIndex = 0;
        }
      }
    },

    // Start auto rotate slides
    start: function start() {
      var _this2 = this;

      // Don't start if no intetrval, or if we are already running
      if (!this.interval || this.isCycling) {
        return;
      }
      this.slides.forEach(function (slide) {
        slide.tabIndex = -1;
      });
      this.intervalId = setInterval(function () {
        _this2.next();
      }, Math.max(1000, this.interval));
    },

    // Re-Start auto rotate slides when focus/hover leaves the carousel
    restart: function restart(evt) {
      if (!this.$el.contains(document.activeElement)) {
        this.start();
      }
    },

    // Update slide list
    updateSlides: function updateSlides() {
      var _this3 = this;

      this.pause();
      // Get all slides as DOM elements
      this.slides = selectAll('.carousel-item', this.$refs.inner);
      var numSlides = this.slides.length;
      // Keep slide number in range
      var index = Math.max(0, Math.min(Math.floor(this.index), numSlides - 1));
      this.slides.forEach(function (slide, idx) {
        var n = idx + 1;
        var id = _this3.safeId('__BV_indicator_' + n + '_');
        if (idx === index) {
          addClass(slide, 'active');
        } else {
          removeClass(slide, 'active');
        }
        setAttr(slide, 'aria-current', idx === index ? 'true' : 'false');
        setAttr(slide, 'aria-posinset', String(n));
        setAttr(slide, 'aria-setsize', String(numSlides));
        slide.tabIndex = -1;
        if (id) {
          setAttr(slide, 'aria-controlledby', id);
        }
      });
      // Set slide as active
      this.setSlide(index);
      this.start();
    }
  },
  watch: {
    value: function value(newVal, oldVal) {
      if (newVal !== oldVal) {
        this.setSlide(newVal);
      }
    },
    interval: function interval(newVal, oldVal) {
      if (newVal === oldVal) {
        return;
      }
      if (!newVal) {
        // Pausing slide show
        this.pause();
      } else {
        // Restarting or Changing interval
        this.pause();
        this.start();
      }
    },
    index: function index(val, oldVal) {
      var _this4 = this;

      if (val === oldVal || this.isSliding) {
        return;
      }
      // Determine sliding direction
      var direction = val > oldVal ? DIRECTION.next : DIRECTION.prev;
      // Rotates
      if (oldVal === 0 && val === this.slides.length - 1) {
        direction = DIRECTION.prev;
      } else if (oldVal === this.slides.length - 1 && val === 0) {
        direction = DIRECTION.next;
      }
      // Determine current and next slides
      var currentSlide = this.slides[oldVal];
      var nextSlide = this.slides[val];
      // Don't do anything if there aren't any slides to slide to
      if (!currentSlide || !nextSlide) {
        return;
      }
      // Start animating
      this.isSliding = true;
      this.$emit('sliding-start', val);
      // Update v-model
      this.$emit('input', this.index);
      nextSlide.classList.add(direction.overlayClass);
      // Trigger a reflow of next slide
      reflow(nextSlide);
      addClass(currentSlide, direction.dirClass);
      addClass(nextSlide, direction.dirClass);
      // Transition End handler
      var called = false;
      var onceTransEnd = function onceTransEnd(evt) {
        if (called) {
          return;
        }
        called = true;
        if (_this4.transitionEndEvent) {
          var events = _this4.transitionEndEvent.split(/\s+/);
          events.forEach(function (event) {
            eventOff(currentSlide, event, onceTransEnd);
          });
        }
        _this4._animationTimeout = null;
        removeClass(nextSlide, direction.dirClass);
        removeClass(nextSlide, direction.overlayClass);
        addClass(nextSlide, 'active');
        removeClass(currentSlide, 'active');
        removeClass(currentSlide, direction.dirClass);
        removeClass(currentSlide, direction.overlayClass);
        setAttr(currentSlide, 'aria-current', 'false');
        setAttr(nextSlide, 'aria-current', 'true');
        setAttr(currentSlide, 'aria-hidden', 'true');
        setAttr(nextSlide, 'aria-hidden', 'false');
        currentSlide.tabIndex = -1;
        nextSlide.tabIndex = -1;
        if (!_this4.isCycling) {
          // Focus the next slide for screen readers if not in play mode
          nextSlide.tabIndex = 0;
          _this4.$nextTick(function () {
            nextSlide.focus();
          });
        }
        _this4.isSliding = false;
        // Notify ourselves that we're done sliding (slid)
        _this4.$nextTick(function () {
          return _this4.$emit('sliding-end', val);
        });
      };
      // Clear transition classes after transition ends
      if (this.transitionEndEvent) {
        var events = this.transitionEndEvent.split(/\s+/);
        events.forEach(function (event) {
          eventOn(currentSlide, event, onceTransEnd);
        });
      }
      // Fallback to setTimeout
      this._animationTimeout = setTimeout(onceTransEnd, TRANS_DURATION);
    }
  },
  created: function created() {
    // Create private non-reactive props
    this._animationTimeout = null;
  },
  mounted: function mounted() {
    // Cache current browser transitionend event name
    this.transitionEndEvent = getTransisionEndEvent(this.$el) || null;
    // Get all slides
    this.updateSlides();
    // Observe child changes so we can update slide list
    observeDOM(this.$refs.inner, this.updateSlides.bind(this), {
      subtree: false,
      childList: true,
      attributes: true,
      attributeFilter: ['id']
    });
  },
  beforeDestroy: function beforeDestroy() {
    clearInterval(this.intervalId);
    clearTimeout(this._animationTimeout);
    this.intervalId = null;
    this._animationTimeout = null;
  }
};

// Blank image with fill template
var BLANK_TEMPLATE = '<svg width="%{w}" height="%{h}" ' + 'xmlns="http://www.w3.org/2000/svg" ' + 'viewBox="0 0 %{w} %{h}" preserveAspectRatio="none">' + '<rect width="100%" height="100%" style="fill:%{f};"></rect>' + '</svg>';

function makeBlankImgSrc(width, height, color) {
  var src = encodeURIComponent(BLANK_TEMPLATE.replace('%{w}', String(width)).replace('%{h}', String(height)).replace('%{f}', color));
  return 'data:image/svg+xml;charset=UTF-8,' + src;
}

var props$18 = {
  src: {
    type: String,
    default: null
  },
  alt: {
    type: String,
    default: null
  },
  width: {
    type: [Number, String],
    default: null
  },
  height: {
    type: [Number, String],
    default: null
  },
  block: {
    type: Boolean,
    default: false
  },
  fluid: {
    type: Boolean,
    default: false
  },
  fluidGrow: {
    // Gives fluid images class `w-100` to make them grow to fit container
    type: Boolean,
    default: false
  },
  rounded: {
    // rounded can be:
    //   false: no rounding of corners
    //   true: slightly rounded corners
    //   'top': top corners rounded
    //   'right': right corners rounded
    //   'bottom': bottom corners rounded
    //   'left': left corners rounded
    //   'circle': circle/oval
    //   '0': force rounding off
    type: [Boolean, String],
    default: false
  },
  thumbnail: {
    type: Boolean,
    default: false
  },
  left: {
    type: Boolean,
    default: false
  },
  right: {
    type: Boolean,
    default: false
  },
  center: {
    type: Boolean,
    default: false
  },
  blank: {
    type: Boolean,
    default: false
  },
  blankColor: {
    type: String,
    default: 'transparent'
  }
};

var bImg = {
  functional: true,
  props: props$18,
  render: function render(h, _ref) {
    var _class;

    var props = _ref.props,
        data = _ref.data;

    var src = props.src;
    var width = parseInt(props.width, 10) ? parseInt(props.width, 10) : null;
    var height = parseInt(props.height, 10) ? parseInt(props.height, 10) : null;
    var align = null;
    var block = props.block;
    if (props.blank) {
      if (!height && Boolean(width)) {
        height = width;
      } else if (!width && Boolean(height)) {
        width = height;
      }
      if (!width && !height) {
        width = 1;
        height = 1;
      }
      // Make a blank SVG image
      src = makeBlankImgSrc(width, height, props.blankColor || 'transparent');
    }
    if (props.left) {
      align = 'float-left';
    } else if (props.right) {
      align = 'float-right';
    } else if (props.center) {
      align = 'mx-auto';
      block = true;
    }
    return h('img', lib_common(data, {
      attrs: {
        'src': src,
        'alt': props.alt,
        'width': width ? String(width) : null,
        'height': height ? String(height) : null
      },
      class: (_class = {
        'img-thumbnail': props.thumbnail,
        'img-fluid': props.fluid || props.fluidGrow,
        'w-100': props.fluidGrow,
        'rounded': props.rounded === '' || props.rounded === true
      }, defineProperty$1(_class, 'rounded-' + props.rounded, typeof props.rounded === 'string' && props.rounded !== ''), defineProperty$1(_class, align, Boolean(align)), defineProperty$1(_class, 'd-block', block), _class)
    }));
  }
};

var bCarouselSlide = {
  components: { bImg: bImg },
  mixins: [idMixin],
  render: function render(h) {
    var t = this;
    var $slots = t.$slots;

    var img = $slots.img;
    if (!img && (t.imgSrc || t.imgBlank)) {
      img = h('b-img', {
        props: {
          fluidGrow: true,
          block: true,
          src: t.imgSrc,
          blank: t.imgBlank,
          blankColor: t.imgBlankColor,
          width: t.computedWidth,
          height: t.computedHeight,
          alt: t.imgAlt
        }
      });
    }

    var content = h(t.contentTag, { class: t.contentClasses }, [t.caption ? h(t.captionTag, { domProps: { innerHTML: t.caption } }) : h(false), t.text ? h(t.textTag, { domProps: { innerHTML: t.text } }) : h(false), $slots.default]);

    return h('div', {
      class: ['carousel-item'],
      style: { background: t.background },
      attrs: { id: t.safeId(), role: 'listitem' }
    }, [img, content]);
  },

  props: {
    imgSrc: {
      type: String,
      default: function _default() {
        if (this && this.src) {
          // Deprecate src
          warn("b-carousel-slide: prop 'src' has been deprecated. Use 'img-src' instead");
          return this.src;
        }
        return null;
      }
    },
    src: {
      // Deprecated: use img-src instead
      type: String
    },
    imgAlt: {
      type: String
    },
    imgWidth: {
      type: [Number, String]
    },
    imgHeight: {
      type: [Number, String]
    },
    imgBlank: {
      type: Boolean,
      default: false
    },
    imgBlankColor: {
      type: String,
      default: 'transparent'
    },
    contentVisibleUp: {
      type: String
    },
    contentTag: {
      type: String,
      default: 'div'
    },
    caption: {
      type: String
    },
    captionTag: {
      type: String,
      default: 'h3'
    },
    text: {
      type: String
    },
    textTag: {
      type: String,
      default: 'p'
    },
    background: {
      type: String
    }
  },
  computed: {
    contentClasses: function contentClasses() {
      return ['carousel-caption', this.contentVisibleUp ? 'd-none' : '', this.contentVisibleUp ? 'd-' + this.contentVisibleUp + '-block' : ''];
    },
    computedWidth: function computedWidth() {
      // Use local width, or try parent width
      return this.imgWidth || this.$parent.imgWidth;
    },
    computedHeight: function computedHeight() {
      // Use local height, or try parent height
      return this.imgHeight || this.$parent.imgHeight;
    }
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$9 = {
  bCarousel: bCarousel,
  bCarouselSlide: bCarouselSlide
};

var VuePlugin$18 = {
  install: function install(Vue) {
    registerComponents(Vue, components$9);
  }
};

vueUse(VuePlugin$18);

var props$19 = {
  tag: {
    type: String,
    default: 'div'
  },
  fluid: {
    type: Boolean,
    default: false
  }
};

var Container = {
  functional: true,
  props: props$19,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    return h(props.tag, lib_common(data, {
      class: {
        'container': !props.fluid,
        'container-fluid': props.fluid
      }
    }), children);
  }
};

var COMMON_ALIGNMENT = ['start', 'end', 'center'];

var props$20 = {
  tag: {
    type: String,
    default: 'div'
  },
  noGutters: {
    type: Boolean,
    default: false
  },
  alignV: {
    type: String,
    default: null,
    validator: function validator(str) {
      return arrayIncludes(COMMON_ALIGNMENT.concat(['baseline', 'stretch']), str);
    }
  },
  alignH: {
    type: String,
    default: null,
    validator: function validator(str) {
      return arrayIncludes(COMMON_ALIGNMENT.concat(['between', 'around']), str);
    }
  },
  alignContent: {
    type: String,
    default: null,
    validator: function validator(str) {
      return arrayIncludes(COMMON_ALIGNMENT.concat(['between', 'around', 'stretch']), str);
    }
  }
};

var bRow = {
  functional: true,
  props: props$20,
  render: function render(h, _ref) {
    var _class;

    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    return h(props.tag, lib_common(data, {
      staticClass: 'row',
      class: (_class = {
        'no-gutters': props.noGutters
      }, defineProperty$1(_class, 'align-items-' + props.alignV, props.alignV), defineProperty$1(_class, 'justify-content-' + props.alignH, props.alignH), defineProperty$1(_class, 'align-content-' + props.alignContent, props.alignContent), _class)
    }), children);
  }
};

/**
 * Generates a prop object with a type of
 * [Boolean, String, Number]
 */
function boolStrNum() {
  return {
    type: [Boolean, String, Number],
    default: false
  };
}

/**
 * Generates a prop object with a type of
 * [String, Number]
 */
function strNum() {
  return {
    type: [String, Number],
    default: null
  };
}

var computeBkPtClass = memoize(function computeBkPt(type, breakpoint, val) {
  var className = type;
  if (val === false || val === null || val === undefined) {
    return undefined;
  }
  if (breakpoint) {
    className += '-' + breakpoint;
  }
  // Handling the boolean style prop when accepting [Boolean, String, Number]
  // means Vue will not convert <b-col sm /> to sm: true for us.
  // Since the default is false, an empty string indicates the prop's presence.
  if (type === 'col' && (val === '' || val === true)) {
    // .col-md
    return className.toLowerCase();
  }
  // .order-md-6
  className += '-' + val;
  return className.toLowerCase();
});

var BREAKPOINTS = ['sm', 'md', 'lg', 'xl'];
// Supports classes like: .col-sm, .col-md-6, .col-lg-auto
var breakpointCol = BREAKPOINTS.reduce(
// eslint-disable-next-line no-sequences
function (propMap, breakpoint) {
  return propMap[breakpoint] = boolStrNum(), propMap;
}, create(null));
// Supports classes like: .offset-md-1, .offset-lg-12
var breakpointOffset = BREAKPOINTS.reduce(
// eslint-disable-next-line no-sequences
function (propMap, breakpoint) {
  return propMap[suffixPropName(breakpoint, 'offset')] = strNum(), propMap;
}, create(null));
// Supports classes like: .order-md-1, .order-lg-12
var breakpointOrder = BREAKPOINTS.reduce(
// eslint-disable-next-line no-sequences
function (propMap, breakpoint) {
  return propMap[suffixPropName(breakpoint, 'order')] = strNum(), propMap;
}, create(null));

// For loop doesn't need to check hasOwnProperty
// when using an object created from null
var breakpointPropMap = assign(create(null), {
  col: keys(breakpointCol),
  offset: keys(breakpointOffset),
  order: keys(breakpointOrder)
});

var props$21 = assign({}, breakpointCol, breakpointOffset, breakpointOrder, {
  tag: {
    type: String,
    default: 'div'
  },
  // Generic flexbox .col
  col: {
    type: Boolean,
    default: false
  },
  // .col-[1-12]|auto
  cols: strNum(),
  // .offset-[1-12]
  offset: strNum(),
  // Flex ordering utility .order-[1-12]
  order: strNum(),
  alignSelf: {
    type: String,
    default: null,
    validator: function validator(str) {
      return arrayIncludes(['auto', 'start', 'end', 'center', 'baseline', 'stretch'], str);
    }
  }
});

/**
 * We need ".col" to default in when no other props are passed,
 * but always render when col=true.
 */
var bCol = {
  functional: true,
  props: props$21,
  render: function render(h, _ref) {
    var _classList$push;

    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    var classList = [];
    // Loop through `col`, `offset`, `order` breakpoint props
    for (var type in breakpointPropMap) {
      // Returns colSm, offset, offsetSm, orderMd, etc.
      var _keys = breakpointPropMap[type];
      for (var i = 0; i < _keys.length; i++) {
        // computeBkPt(col, colSm => Sm, value=[String, Number, Boolean])
        var c = computeBkPtClass(type, _keys[i].replace(type, ''), props[_keys[i]]);
        // If a class is returned, push it onto the array.
        if (c) {
          classList.push(c);
        }
      }
    }

    classList.push((_classList$push = {
      // Default to .col if no other classes generated nor `cols` specified.
      col: props.col || classList.length === 0 && !props.cols
    }, defineProperty$1(_classList$push, 'col-' + props.cols, props.cols), defineProperty$1(_classList$push, 'offset-' + props.offset, props.offset), defineProperty$1(_classList$push, 'order-' + props.order, props.order), defineProperty$1(_classList$push, 'align-self-' + props.alignSelf, props.alignSelf), _classList$push));

    return h(props.tag, lib_common(data, { class: classList }), children);
  }
};

var props$22 = {
  tag: {
    type: String,
    default: 'div'
  }
};

var bFormRow = {
  functional: true,
  props: props$22,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    return h(props.tag, lib_common(data, {
      staticClass: 'form-row'
    }), children);
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$10 = {
  bContainer: Container,
  bRow: bRow,
  bCol: bCol,
  bFormRow: bFormRow
};

var VuePlugin$20 = {
  install: function install(Vue) {
    registerComponents(Vue, components$10);
  }
};

vueUse(VuePlugin$20);

// Events we emit on $root
var EVENT_STATE = 'bv::collapse::state';
var EVENT_ACCORDION = 'bv::collapse::accordion';
// Events we listen to on $root
var EVENT_TOGGLE = 'bv::toggle::collapse';

var bCollapse = {
  mixins: [listenOnRootMixin],
  render: function render(h) {
    var t = this;
    var content = h(t.tag, {
      class: t.classObject,
      directives: [{ name: 'show', value: t.show }],
      attrs: { id: t.id || null },
      on: { click: t.clickHandler }
    }, [t.$slots.default]);
    return h('transition', {
      props: {
        enterClass: '',
        enterActiveClass: 'collapsing',
        enterToClass: '',
        leaveClass: '',
        leaveActiveClass: 'collapsing',
        leaveToClass: ''
      },
      on: {
        enter: t.onEnter,
        afterEnter: t.onAfterEnter,
        leave: t.onLeave,
        afterLeave: t.onAfterLeave
      }
    }, [content]);
  },
  data: function data() {
    return {
      show: this.visible,
      transitioning: false
    };
  },

  model: {
    prop: 'visible',
    event: 'input'
  },
  props: {
    id: {
      type: String,
      required: true
    },
    isNav: {
      type: Boolean,
      default: false
    },
    accordion: {
      type: String,
      default: null
    },
    visible: {
      type: Boolean,
      default: false
    },
    tag: {
      type: String,
      default: 'div'
    }
  },
  watch: {
    visible: function visible(newVal) {
      if (newVal !== this.show) {
        this.show = newVal;
      }
    },
    show: function show(newVal, oldVal) {
      if (newVal !== oldVal) {
        this.emitState();
      }
    }
  },
  computed: {
    classObject: function classObject() {
      return {
        'navbar-collapse': this.isNav,
        'collapse': !this.transitioning,
        'show': this.show && !this.transitioning
      };
    }
  },
  methods: {
    toggle: function toggle() {
      this.show = !this.show;
    },
    onEnter: function onEnter(el) {
      el.style.height = 0;
      reflow(el);
      el.style.height = el.scrollHeight + 'px';
      this.transitioning = true;
      // This should be moved out so we can add cancellable events
      this.$emit('show');
    },
    onAfterEnter: function onAfterEnter(el) {
      el.style.height = null;
      this.transitioning = false;
      this.$emit('shown');
    },
    onLeave: function onLeave(el) {
      el.style.height = 'auto';
      el.style.display = 'block';
      el.style.height = el.getBoundingClientRect().height + 'px';
      reflow(el);
      this.transitioning = true;
      el.style.height = 0;
      // This should be moved out so we can add cancellable events
      this.$emit('hide');
    },
    onAfterLeave: function onAfterLeave(el) {
      el.style.height = null;
      this.transitioning = false;
      this.$emit('hidden');
    },
    emitState: function emitState() {
      this.$emit('input', this.show);
      // Let v-b-toggle know the state of this collapse
      this.$root.$emit(EVENT_STATE, this.id, this.show);
      if (this.accordion && this.show) {
        // Tell the other collapses in this accordion to close
        this.$root.$emit(EVENT_ACCORDION, this.id, this.accordion);
      }
    },
    clickHandler: function clickHandler(evt) {
      // If we are in a nav/navbar, close the collapse when non-disabled link clicked
      var el = evt.target;
      if (!this.isNav || !el || getComputedStyle(this.$el).display !== 'block') {
        return;
      }
      if (hasClass(el, 'nav-link') || hasClass(el, 'dropdown-item')) {
        this.show = false;
      }
    },
    handleToggleEvt: function handleToggleEvt(target) {
      if (target !== this.id) {
        return;
      }
      this.toggle();
    },
    handleAccordionEvt: function handleAccordionEvt(openedId, accordion) {
      if (!this.accordion || accordion !== this.accordion) {
        return;
      }
      if (openedId === this.id) {
        // Open this collapse if not shown
        if (!this.show) {
          this.toggle();
        }
      } else {
        // Close this collapse if shown
        if (this.show) {
          this.toggle();
        }
      }
    },
    handleResize: function handleResize() {
      // Handler for orientation/resize to set collapsed state in nav/navbar
      this.show = getComputedStyle(this.$el).display === 'block';
    }
  },
  created: function created() {
    // Listen for toggle events to open/close us
    this.listenOnRoot(EVENT_TOGGLE, this.handleToggleEvt);
    // Listen to otehr collapses for accordion events
    this.listenOnRoot(EVENT_ACCORDION, this.handleAccordionEvt);
  },
  mounted: function mounted() {
    if (this.isNav && typeof document !== 'undefined') {
      // Set up handlers
      window.addEventListener('resize', this.handleResize, false);
      window.addEventListener('orientationchange', this.handleResize, false);
      this.handleResize();
    }
    this.emitState();
  },
  beforeDestroy: function beforeDestroy() {
    if (this.isNav && typeof document !== 'undefined') {
      window.removeEventListener('resize', this.handleResize, false);
      window.removeEventListener('orientationchange', this.handleResize, false);
    }
  }
};

var allListenTypes = { hover: true, click: true, focus: true };

function targets(vnode, binding, listenTypes, fn) {
  var targets = keys(binding.modifiers || {}).filter(function (t) {
    return !allListenTypes[t];
  });

  if (binding.value) {
    targets.push(binding.value);
  }

  var listener = function listener() {
    fn({ targets: targets, vnode: vnode });
  };

  keys(allListenTypes).forEach(function (type) {
    if (listenTypes[type] || binding.modifiers[type]) {
      vnode.elm.addEventListener(type, listener);
    }
  });

  // Return the list of targets
  return targets;
}

// Are we client side?
var inBrowser = typeof window !== 'undefined';

// target listen types
var listenTypes = { click: true

  // Property key for handler storage
};var BVT = '__BV_toggle__';

// Emitted Control Event for collapse (emitted to collapse)
var EVENT_TOGGLE$1 = 'bv::toggle::collapse';

// Listen to Event for toggle state update (Emited by collapse)
var EVENT_STATE$1 = 'bv::collapse::state';

var bToggle = {
  bind: function bind(el, binding, vnode) {
    var targets$$1 = targets(vnode, binding, listenTypes, function (_ref) {
      var targets$$1 = _ref.targets,
          vnode = _ref.vnode;

      targets$$1.forEach(function (target$$1) {
        vnode.context.$root.$emit(EVENT_TOGGLE$1, target$$1);
      });
    });

    if (inBrowser && vnode.context && targets$$1.length > 0) {
      // Add aria attributes to element
      setAttr(el, 'aria-controls', targets$$1.join(' '));
      setAttr(el, 'aria-expanded', 'false');

      // Toggle state hadnler, stored on element
      el[BVT] = function toggleDirectiveHandler(id, state) {
        if (targets$$1.indexOf(id) !== -1) {
          // Set aria-expanded state
          setAttr(el, 'aria-expanded', state ? 'true' : 'false');
          // Set/Clear 'collapsed' class state
          if (state) {
            removeClass(el, 'collapsed');
          } else {
            addClass(el, 'collapsed');
          }
        }
      };

      // Listen for toggle state changes
      vnode.context.$root.$on(EVENT_STATE$1, el[BVT]);
    }
  },
  unbind: function unbind(el, binding, vnode) {
    if (el[BVT]) {
      // Remove our $root listener
      vnode.context.$root.$off(EVENT_STATE$1, el[BVT]);
      el[BVT] = null;
    }
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var directives = {
  bToggle: bToggle
};

var VuePlugin$23 = {
  install: function install(Vue) {
    registerDirectives(Vue, directives);
  }
};

vueUse(VuePlugin$23);

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$11 = {
  bCollapse: bCollapse
};

var VuePlugin$22 = {
  install: function install(Vue) {
    registerComponents(Vue, components$11);
    Vue.use(VuePlugin$23);
  }
};

vueUse(VuePlugin$22);

var bDropdown = {
  mixins: [idMixin, dropdownMixin],
  components: { bButton: bBtn },
  render: function render(h) {
    var t = this;
    var split = h(false);
    if (t.split) {
      split = h('b-button', {
        ref: 'button',
        props: {
          disabled: t.disabled,
          variant: t.variant,
          size: t.size
        },
        attrs: {
          id: t.safeId('_BV_button_')
        },
        on: {
          click: t.click
        }
      }, [t.$slots['button-content'] || t.$slots.text || t.text]);
    }
    var toggle = h('b-button', {
      ref: 'toggle',
      class: {
        'dropdown-toggle': !t.noCaret || t.split,
        'dropdown-toggle-split': t.split
      },
      props: {
        variant: t.variant,
        size: t.size,
        disabled: t.disabled
      },
      attrs: {
        id: t.safeId('_BV_toggle_'),
        'aria-haspopup': 'true',
        'aria-expanded': t.visible ? 'true' : 'false'
      },
      on: {
        click: t.toggle, // click
        keydown: t.toggle // enter, space, down
      }
    }, [t.split ? h('span', { class: ['sr-only'] }, [t.toggleText]) : t.$slots['button-content'] || t.$slots.text || t.text]);
    var menu = h('div', {
      ref: 'menu',
      class: t.menuClasses,
      attrs: {
        role: t.role,
        'aria-labelledby': t.safeId(split ? '_BV_toggle_' : '_BV_button_')
      },
      on: {
        mouseover: t.onMouseOver,
        keydown: t.onKeydown // tab, up, down, esc
      }
    }, [this.$slots.default]);
    return h('div', { attrs: { id: t.safeId() }, class: t.dropdownClasses }, [split, toggle, menu]);
  },

  props: {
    split: {
      type: Boolean,
      default: false
    },
    toggleText: {
      type: String,
      default: 'Toggle Dropdown'
    },
    size: {
      type: String,
      default: null
    },
    variant: {
      type: String,
      default: null
    },
    noCaret: {
      type: Boolean,
      default: false
    },
    role: {
      type: String,
      default: 'menu'
    }
  },
  computed: {
    dropdownClasses: function dropdownClasses() {
      return ['btn-group', 'b-dropdown', 'dropdown', this.dropup ? 'dropup' : '', this.visible ? 'show' : ''];
    },
    menuClasses: function menuClasses() {
      return ['dropdown-menu', this.right ? 'dropdown-menu-right' : '', this.visible ? 'show' : ''];
    }
  }
};

var props$23 = propsFactory();

var bDropdownItem = {
  functional: true,
  props: props$23,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    return h(Link, lib_common(data, {
      props: props,
      staticClass: 'dropdown-item',
      attrs: { role: 'menuitem' }
    }), children);
  }
};

var props$24 = {
  disabled: {
    type: Boolean,
    default: false
  }
};

var bDropdownItemButton = {
  functional: true,
  props: props$24,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        parent = _ref.parent,
        children = _ref.children;

    return h('button', lib_common(data, {
      props: props,
      staticClass: 'dropdown-item',
      attrs: { role: 'menuitem', type: 'button', disabled: props.disabled },
      on: {
        click: function click(e) {
          parent.$root.$emit('clicked::link', e);
        }
      }
    }), children);
  }
};

var props$25 = {
  id: {
    type: String,
    default: null
  },
  tag: {
    type: String,
    default: 'h6'
  }
};

var bDropdownHeader = {
  functional: true,
  props: props$25,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    return h(props.tag, lib_common(data, {
      staticClass: 'dropdown-header',
      attrs: { id: props.id || null }
    }), children);
  }
};

var props$26 = {
  tag: {
    type: String,
    default: 'div'
  }
};

var bDropdownDivider = {
  functional: true,
  props: props$26,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data;

    return h(props.tag, lib_common(data, {
      staticClass: 'dropdown-divider',
      attrs: { role: 'separator' }
    }));
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$12 = {
  bDropdown: bDropdown,
  bDd: bDropdown,
  bDropdownItem: bDropdownItem,
  bDdItem: bDropdownItem,
  bDropdownItemButton: bDropdownItemButton,
  bDropdownItemBtn: bDropdownItemButton,
  bDdItemButton: bDropdownItemButton,
  bDdItemBtn: bDropdownItemButton,
  bDropdownHeader: bDropdownHeader,
  bDdHeader: bDropdownHeader,
  bDropdownDivider: bDropdownDivider,
  bDdDivider: bDropdownDivider
};

var VuePlugin$24 = {
  install: function install(Vue) {
    registerComponents(Vue, components$12);
  }
};

vueUse(VuePlugin$24);

var props$27 = {
  type: {
    type: String,
    default: 'iframe',
    validator: function validator(str) {
      return arrayIncludes(['iframe', 'embed', 'video', 'object', 'img', 'b-img', 'b-img-lazy'], str);
    }
  },
  tag: {
    type: String,
    default: 'div'
  },
  aspect: {
    type: String,
    default: '16by9'
  }
};

var bEmbed = {
  functional: true,
  props: props$27,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    return h(props.tag, {
      ref: data.ref,
      staticClass: 'embed-responsive',
      class: defineProperty$1({}, 'embed-responsive-' + props.aspect, Boolean(props.aspect))
    }, [h(props.type, lib_common(data, { ref: '', staticClass: 'embed-responsive-item' }), children)]);
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$13 = {
  bEmbed: bEmbed
};

var VuePlugin$25 = {
  install: function install(Vue) {
    registerComponents(Vue, components$13);
  }
};

vueUse(VuePlugin$25);

var props$28 = {
  id: {
    type: String,
    default: null
  },
  inline: {
    type: Boolean,
    default: false
  },
  novalidate: {
    type: Boolean,
    default: false
  },
  validated: {
    type: Boolean,
    default: false
  }
};

var Form = {
  functional: true,
  props: props$28,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    return h('form', lib_common(data, {
      class: {
        'form-inline': props.inline,
        'was-validated': props.validated
      },
      attrs: {
        id: props.id,
        novalidate: props.novalidate
      }
    }), children);
  }
};

var props$29 = {
  id: {
    type: String,
    default: null
  },
  tag: {
    type: String,
    default: 'small'
  },
  textVariant: {
    type: String,
    default: 'muted'
  },
  inline: {
    type: Boolean,
    default: false
  }
};

var bFormText = {
  functional: true,
  props: props$29,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    return h(props.tag, lib_common(data, {
      class: defineProperty$1({
        'form-text': !props.inline
      }, 'text-' + props.textVariant, Boolean(props.textVariant)),
      attrs: {
        id: props.id
      }
    }), children);
  }
};

var props$30 = {
  id: {
    type: String,
    default: null
  },
  tag: {
    type: String,
    default: 'div'
  }
};

var bFormInvalidFeedback = {
  functional: true,
  props: props$30,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    return h(props.tag, lib_common(data, {
      staticClass: 'invalid-feedback',
      attrs: {
        id: props.id
      }
    }), children);
  }
};

var props$31 = {
  id: {
    type: String,
    default: null
  },
  tag: {
    type: String,
    default: 'div'
  }
};

var bFormValidFeedback = {
  functional: true,
  props: props$31,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    return h(props.tag, lib_common(data, {
      staticClass: 'valid-feedback',
      attrs: {
        id: props.id
      }
    }), children);
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$14 = {
  bForm: Form,
  bFormRow: bFormRow,
  bFormText: bFormText,
  bFormInvalidFeedback: bFormInvalidFeedback,
  bFormFeedback: bFormInvalidFeedback,
  bFormValidFeedback: bFormValidFeedback
};

var VuePlugin$27 = {
  install: function install(Vue) {
    registerComponents(Vue, components$14);
  }
};

vueUse(VuePlugin$27);

var bFormGroup$2 = {
  mixins: [idMixin, formStateMixin],
  components: { bFormRow: bFormRow, bFormText: bFormText, bFormInvalidFeedback: bFormInvalidFeedback, bFormValidFeedback: bFormValidFeedback },
  render: function render(h) {
    var t = this;
    var $slots = t.$slots;
    // Label
    var legend = h(false);
    if (t.label || $slots.label || t.horizontal) {
      legend = h('legend', { class: t.labelClasses, attrs: { id: t.labelId } }, [$slots.label || h('span', { domProps: { innerHTML: t.label || '' } })]);
    }
    // Invalid feeback text
    var invalidFeedback = h(false);
    if (t.invalidFeedback || t.feedback || $slots['invalid-feedback'] || $slots['feedback']) {
      invalidFeedback = h('b-form-invalid-feedback', {
        directives: [{
          name: 'show',
          rawName: 'v-show',
          value: Boolean(t.invalidFeedback || t.feedback || $slots['invalid-feedback'] || $slots['feedback']),
          expression: "Boolean(t.invalidFeedback || t.feedback || $slots['invalid-feedback'] || $slots['feedback'])"
        }],
        attrs: {
          id: t.invalidFeedbackId,
          role: 'alert',
          'aria-live': 'assertive',
          'aria-atomic': 'true'
        }
      }, [t.computedState === false ? $slots['invalid-feedback'] || $slots['feedback'] || h('span', { domProps: { innerHTML: t.invalidFeedback || t.feedback || '' } }) : h(false)]);
    }
    // Valid feeback text
    var validFeedback = h(false);
    if (t.validFeedback || $slots['valid-feedback']) {
      validFeedback = h('b-form-valid-feedback', {
        directives: [{
          name: 'show',
          rawName: 'v-show',
          value: Boolean(t.validFeedback || $slots['valid-feedback']),
          expression: "Boolean(t.validFeedback || $slots['valid-feedback'])"
        }],
        attrs: {
          id: t.validFeedbackId,
          role: 'alert',
          'aria-live': 'assertive',
          'aria-atomic': 'true'
        }
      }, [t.computedState === true ? $slots['valid-feedback'] || h('span', { domProps: { innerHTML: t.validFeedback || '' } }) : h(false)]);
    }
    // Form help text (description)
    var description = h(false);
    if (t.description || $slots['description']) {
      description = h('b-form-text', { attrs: { id: t.descriptionId } }, [$slots['description'] || h('span', { domProps: { innerHTML: t.description || '' } })]);
    }
    // Build layout
    var content = h('div', { ref: 'content', class: t.inputLayoutClasses }, [$slots.default, invalidFeedback, validFeedback, description]);
    // Generate fieldset wrapper
    return h('fieldset', {
      class: t.groupClasses,
      attrs: { id: t.safeId(), 'aria-describedby': t.describedByIds }
    }, [h('b-form-row', {}, [legend, content])]);
  },

  props: {
    horizontal: {
      type: Boolean,
      default: false
    },
    labelCols: {
      type: Number,
      default: 3,
      validator: function validator(value) {
        if (value >= 1 && value <= 11) {
          return true;
        }
        warn('b-form-group: label-cols must be a value between 1 and 11');
        return false;
      }
    },
    breakpoint: {
      type: String,
      default: 'sm'
    },
    labelTextAlign: {
      type: String,
      default: null
    },
    label: {
      type: String,
      default: null
    },
    labelSrOnly: {
      type: Boolean,
      default: false
    },
    labelClass: {
      type: [String, Array],
      default: null
    },
    description: {
      type: String,
      default: null
    },
    invalidFeedback: {
      type: String,
      default: null
    },
    feedback: {
      // Deprecated in favor of invalid-feedback
      type: String,
      default: null
    },
    validFeedback: {
      type: String,
      default: null
    },
    validated: {
      type: Boolean,
      value: false
    }
  },
  computed: {
    inputState: function inputState() {
      return this.stateClass;
    },
    groupClasses: function groupClasses() {
      return ['b-form-group', 'form-group', this.validated ? 'was-validated' : null, this.inputState];
    },
    labelClasses: function labelClasses() {
      return [this.labelSrOnly ? 'sr-only' : 'col-form-legend', this.labelLayout, this.labelAlignClass, this.labelClass];
    },
    labelLayout: function labelLayout() {
      if (this.labelSrOnly) {
        return null;
      }
      return this.horizontal ? 'col-' + this.breakpoint + '-' + this.labelCols : 'col-12';
    },
    labelAlignClass: function labelAlignClass() {
      if (this.labelSrOnly) {
        return null;
      }
      return this.labelTextAlign ? 'text-' + this.labelTextAlign : null;
    },
    inputLayoutClasses: function inputLayoutClasses() {
      return [this.horizontal ? 'col-' + this.breakpoint + '-' + (12 - this.labelCols) : 'col-12'];
    },
    labelId: function labelId() {
      return this.label || this.$slots['label'] ? this.safeId('_BV_label_') : null;
    },
    descriptionId: function descriptionId() {
      if (this.description || this.$slots['description']) {
        return this.safeId('_BV_description_');
      }
      return null;
    },
    invalidFeedbackId: function invalidFeedbackId() {
      if (this.invalidFeedback || this.feedback || this.$slots['invalid-feedback'] || this.$slots['feedback']) {
        return this.safeId('_BV_feedback_invalid_');
      }
      return null;
    },
    validFeedbackId: function validFeedbackId() {
      if (this.validFeedback || this.$slots['valid-feedback']) {
        return this.safeId('_BV_feedback_valid_');
      }
      return null;
    },
    describedByIds: function describedByIds() {
      return [this.labelId, this.descriptionId, this.computedState === false ? this.invalidFeedbackId : null, this.computedState === true ? this.validFeedbackId : null].filter(function (i) {
        return i;
      }).join(' ') || null;
    }
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$15 = {
  bFormGroup: bFormGroup$2,
  bFormFieldset: bFormGroup$2
};

var VuePlugin$29 = {
  install: function install(Vue) {
    registerComponents(Vue, components$15);
  }
};

vueUse(VuePlugin$29);

var bFormCheckbox = {
  mixins: [idMixin, formRadioCheckMixin, formMixin, formSizeMixin, formStateMixin, formCustomMixin],
  render: function render(h) {
    var t = this;

    var input = h('input', {
      ref: 'check',
      class: [t.is_ButtonMode ? '' : t.is_Plain ? 'form-check-input' : 'custom-control-input', t.get_StateClass],
      directives: [{ name: 'model', rawName: 'v-model', value: t.computedLocalChecked, expression: 'computedLocalChecked' }],
      attrs: {
        id: t.safeId(),
        type: 'checkbox',
        name: t.get_Name,
        disabled: t.is_Disabled,
        required: t.is_Required,
        autocomplete: 'off',
        'true-value': t.value,
        'false-value': t.uncheckedValue,
        'aria-required': t.is_Required ? 'true' : null
      },
      domProps: { value: t.value, checked: t.is_Checked },
      on: {
        focus: t.handleFocus,
        blur: t.handleFocus,
        change: t.emitChange,
        __c: function __c(evt) {
          var $$a = t.computedLocalChecked;
          var $$el = evt.target;
          if (isArray($$a)) {
            // Multiple checkbox
            var $$v = t.value;
            var $$i = t._i($$a, $$v); // Vue's 'loose' Array.indexOf
            if ($$el.checked) {
              // Append value to array
              $$i < 0 && (t.computedLocalChecked = $$a.concat([$$v]));
            } else {
              // Remove value from array
              $$i > -1 && (t.computedLocalChecked = $$a.slice(0, $$i).concat($$a.slice($$i + 1)));
            }
          } else {
            // Single checkbox
            t.computedLocalChecked = $$el.checked ? t.value : t.uncheckedValue;
          }
        }
      }
    });

    var indicator = h(false);
    if (!t.is_ButtonMode && !t.is_Plain) {
      indicator = h('span', { class: 'custom-control-indicator', attrs: { 'aria-hidden': 'true' } });
    }

    var description = h('span', { class: t.is_ButtonMode ? null : t.is_Plain ? 'form-check-description' : 'custom-control-description' }, [t.$slots.default]);

    var label = h('label', { class: [t.is_ButtonMode ? t.buttonClasses : t.is_Plain ? 'form-check-label' : t.labelClasses] }, [input, indicator, description]);

    if (t.is_Plain && !t.is_ButtonMode) {
      return h('div', { class: ['form-check', { 'form-check-inline': !t.is_Stacked }] }, [label]);
    } else {
      return label;
    }
  },

  props: {
    value: {
      default: true
    },
    uncheckedValue: {
      // Not applicable in multi-check mode
      default: false
    },
    indeterminate: {
      // Not applicable in multi-check mode
      type: Boolean,
      default: false
    }
  },
  computed: {
    labelClasses: function labelClasses() {
      return ['custom-control', 'custom-checkbox', this.get_Size ? 'form-control-' + this.get_Size : '', this.get_StateClass];
    },
    is_Checked: function is_Checked() {
      var checked = this.computedLocalChecked;
      if (isArray(checked)) {
        for (var i = 0; i < checked.length; i++) {
          if (looseEqual(checked[i], this.value)) {
            return true;
          }
        }
        return false;
      } else {
        return looseEqual(checked, this.value);
      }
    }
  },
  watch: {
    computedLocalChecked: function computedLocalChecked(newVal, oldVal) {
      if (looseEqual(newVal, oldVal)) {
        return;
      }
      this.$emit('input', newVal);
      this.$emit('update:indeterminate', this.$refs.check.indeterminate);
    },
    checked: function checked(newVal, oldVal) {
      if (this.is_Child || looseEqual(newVal, oldVal)) {
        return;
      }
      this.computedLocalChecked = newVal;
    },
    indeterminate: function indeterminate(newVal, oldVal) {
      this.setIndeterminate(newVal);
    }
  },
  methods: {
    emitChange: function emitChange(_ref) {
      var checked = _ref.target.checked;

      // Change event is only fired via user interaction
      // And we only emit the value of this checkbox
      if (this.is_Child || isArray(this.computedLocalChecked)) {
        this.$emit('change', checked ? this.value : null);
        if (this.is_Child) {
          // If we are a child of form-checkbbox-group, emit change on parent
          this.$parent.$emit('change', this.computedLocalChecked);
        }
      } else {
        // Single radio mode supports unchecked value
        this.$emit('change', checked ? this.value : this.uncheckedValue);
      }
      this.$emit('update:indeterminate', this.$refs.check.indeterminate);
    },
    setIndeterminate: function setIndeterminate(state) {
      // Indeterminate only supported in single checkbox mode
      if (this.is_Child || isArray(this.computedLocalChecked)) {
        return;
      }
      this.$refs.check.indeterminate = state;
      // Emit update event to prop
      this.$emit('update:indeterminate', this.$refs.check.indeterminate);
    }
  },
  mounted: function mounted() {
    // Set initial indeterminate state
    this.setIndeterminate(this.indeterminate);
  }
};

var bFormCheckboxGroup = {
  mixins: [idMixin, formMixin, formSizeMixin, formStateMixin, formCustomMixin, formOptionsMixin],
  components: { bFormCheckbox: bFormCheckbox },
  render: function render(h) {
    var t = this;
    var $slots = t.$slots;

    var checks = t.formOptions.map(function (option, idx) {
      return h('b-form-checkbox', {
        key: 'radio_' + idx + '_opt',
        props: {
          id: t.safeId('_BV_radio_' + idx + '_opt_'),
          name: t.name,
          value: option.value,
          required: t.name && t.required,
          disabled: option.disabled
        }
      }, [h('span', { domProps: { innerHTML: option.text } })]);
    });
    return h('div', {
      class: t.groupClasses,
      attrs: {
        id: t.safeId(),
        role: 'group',
        tabindex: '-1',
        'data-toggle': t.buttons ? 'buttons' : null, // Needed for styling only!
        'aria-required': t.required ? 'true' : null,
        'aria-invalid': t.computedAriaInvalid
      }
    }, [$slots.first, checks, $slots.default]);
  },
  data: function data() {
    return {
      localChecked: this.checked || [],
      // Flag for children
      is_RadioCheckGroup: true
    };
  },

  model: {
    prop: 'checked',
    event: 'input'
  },
  props: {
    checked: {
      type: [String, Number, Object, Array, Boolean],
      default: null
    },
    validated: {
      type: Boolean,
      default: false
    },
    ariaInvalid: {
      type: [Boolean, String],
      default: false
    },
    stacked: {
      type: Boolean,
      default: false
    },
    buttons: {
      // Render as button style
      type: Boolean,
      default: false
    },
    buttonVariant: {
      // Only applicable when rendered with button style
      type: String,
      default: 'secondary'
    }
  },
  watch: {
    checked: function checked(newVal, oldVal) {
      this.localChecked = this.checked;
    },
    localChecked: function localChecked(newVal, oldVal) {
      this.$emit('input', newVal);
    }
  },
  computed: {
    groupClasses: function groupClasses() {
      var t = this;
      if (t.buttons) {
        return [t.stacked ? 'btn-group-vertical' : 'btn-group', t.size ? 'btn-group-' + this.size : '', t.validated ? 'was-validated' : ''];
      }
      return [t.sizeFormClass, t.stacked && t.custom ? 'custom-controls-stacked' : '', t.validated ? 'was-validated' : ''];
    },
    computedAriaInvalid: function computedAriaInvalid() {
      var t = this;
      if (t.ariaInvalid === true || t.ariaInvalid === 'true' || t.ariaInvalid === '') {
        return 'true';
      }
      return t.get_State === false ? 'true' : null;
    },
    get_State: function get_State() {
      // Child radios sniff this value
      return this.computedState;
    }
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$16 = {
  bFormCheckbox: bFormCheckbox,
  bCheckbox: bFormCheckbox,
  bCheck: bFormCheckbox,
  bFormCheckboxGroup: bFormCheckboxGroup,
  bCheckboxGroup: bFormCheckboxGroup,
  bCheckGroup: bFormCheckboxGroup
};

var VuePlugin$31 = {
  install: function install(Vue) {
    registerComponents(Vue, components$16);
  }
};

vueUse(VuePlugin$31);

var bFormRadio = {
  mixins: [idMixin, formRadioCheckMixin, formMixin, formStateMixin],
  render: function render(h) {
    var t = this;

    var input = h('input', {
      ref: 'radio',
      class: [t.is_ButtonMode ? '' : t.is_Plain ? 'form-check-input' : 'custom-control-input', t.get_StateClass],
      directives: [{ name: 'model', rawName: 'v-model', value: t.computedLocalChecked, expression: 'computedLocalChecked' }],
      attrs: {
        id: t.safeId(),
        type: 'radio',
        name: t.get_Name,
        required: t.get_Name && t.is_Required,
        disabled: t.is_Disabled,
        autocomplete: 'off'
      },
      domProps: { value: t.value, checked: looseEqual(t.computedLocalChecked, t.value) },
      on: {
        focus: t.handleFocus,
        blur: t.handleFocus,
        change: t.emitChange,
        __c: function __c(evt) {
          t.computedLocalChecked = t.value;
        }
      }
    });

    var indicator = h(false);
    if (!t.is_ButtonMode && !t.is_Plain) {
      indicator = h('span', { class: 'custom-control-indicator', attrs: { 'aria-hidden': 'true' } });
    }

    var description = h('span', { class: t.is_ButtonMode ? null : t.is_Plain ? 'form-check-description' : 'custom-control-description' }, [t.$slots.default]);

    var label = h('label', { class: [t.is_ButtonMode ? t.buttonClasses : t.is_Plain ? 'form-check-label' : t.labelClasses] }, [input, indicator, description]);

    if (t.is_Plain && !t.is_ButtonMode) {
      return h('div', { class: ['form-check', { 'form-check-inline': !t.is_Stacked }] }, [label]);
    } else {
      return label;
    }
  },

  watch: {
    // Radio Groups can only have a single value, so our watchers are simple
    checked: function checked(newVal, oldVal) {
      this.computedLocalChecked = newVal;
    },
    computedLocalChceked: function computedLocalChceked(newVal, oldVal) {
      this.$emit('input', this.computedLocalChceked);
    }
  },
  computed: {
    is_Checked: function is_Checked() {
      return looseEqual(this.value, this.computedLocalChecked);
    },
    labelClasses: function labelClasses() {
      // Specific to radio
      return [this.get_Size ? 'form-control-' + this.get_Size : '', 'custom-control', 'custom-radio', this.get_StateClass];
    }
  },
  methods: {
    emitChange: function emitChange(_ref) {
      var checked = _ref.target.checked;

      // Change is only emitted on user interaction
      this.$emit('change', checked ? this.value : null);
      // If this is a child of form-radio-group, we emit a change event on it as well
      if (this.is_Child) {
        this.$parent.$emit('change', this.computedLocalChecked);
      }
    }
  }
};

var bFormRadioGroup = {
  mixins: [idMixin, formMixin, formSizeMixin, formStateMixin, formCustomMixin, formOptionsMixin],
  components: { bFormRadio: bFormRadio },
  render: function render(h) {
    var t = this;
    var $slots = t.$slots;

    var radios = t.formOptions.map(function (option, idx) {
      return h('b-form-radio', {
        key: 'radio_' + idx + '_opt',
        props: {
          id: t.safeId('_BV_radio_' + idx + '_opt_'),
          name: t.name,
          value: option.value,
          required: Boolean(t.name && t.required),
          disabled: option.disabled
        }
      }, [h('span', { domProps: { innerHTML: option.text } })]);
    });
    return h('div', {
      class: t.groupClasses,
      attrs: {
        id: t.safeId(),
        role: 'radiogroup',
        tabindex: '-1',
        'data-toggle': t.buttons ? 'buttons' : null, // Needed for styling only!
        'aria-required': t.required ? 'true' : null,
        'aria-invalid': t.computedAriaInvalid
      }
    }, [$slots.first, radios, $slots.default]);
  },
  data: function data() {
    return {
      localChecked: this.checked,
      // Flag for children
      is_RadioCheckGroup: true
    };
  },

  model: {
    prop: 'checked',
    event: 'input'
  },
  props: {
    checked: {
      type: [String, Object, Number, Boolean],
      default: null
    },
    validated: {
      // Used for applying hte `was-validated` class to the group
      type: Boolean,
      default: false
    },
    ariaInvalid: {
      type: [Boolean, String],
      default: false
    },
    stacked: {
      type: Boolean,
      default: false
    },
    buttons: {
      // Render as button style
      type: Boolean,
      default: false
    },
    buttonVariant: {
      // Only applicable when rendered with button style
      type: String,
      default: 'secondary'
    }
  },
  watch: {
    checked: function checked(newVal, oldVal) {
      this.localChecked = this.checked;
    },
    localChecked: function localChecked(newVal, oldVal) {
      this.$emit('input', newVal);
    }
  },
  computed: {
    groupClasses: function groupClasses() {
      if (this.buttons) {
        return [this.stacked ? 'btn-group-vertical' : 'btn-group', this.size ? 'btn-group-' + this.size : '', this.validated ? 'was-validated' : ''];
      }
      return [this.sizeFormClass, this.stacked && this.custom ? 'custom-controls-stacked' : '', this.validated ? 'was-validated' : ''];
    },
    computedAriaInvalid: function computedAriaInvalid() {
      if (this.ariaInvalid === true || this.ariaInvalid === 'true' || this.ariaInvalid === '') {
        return 'true';
      }
      return this.get_State === false ? 'true' : null;
    },
    get_State: function get_State() {
      // Required by child radios
      return this.computedState;
    }
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$17 = {
  bFormRadio: bFormRadio,
  bRadio: bFormRadio,
  bFormRadioGroup: bFormRadioGroup,
  bRadioGroup: bFormRadioGroup
};

var VuePlugin$33 = {
  install: function install(Vue) {
    registerComponents(Vue, components$17);
  }
};

vueUse(VuePlugin$33);

// Valid supported input types
var TYPES = ['text', 'password', 'email', 'number', 'url', 'tel', 'search', 'range', 'color', 'date', 'time', 'datetime', 'datetime-local', 'month', 'week'];

var bFormInput$2 = {
  mixins: [idMixin, formMixin, formSizeMixin, formStateMixin],
  render: function render(h) {
    var t = this;
    return h('input', {
      ref: 'input',
      class: t.inputClass,
      domProps: { value: t.localValue },
      attrs: {
        id: t.safeId(),
        name: t.name,
        type: t.localType,
        disabled: t.disabled,
        required: t.required,
        readonly: t.readonly || t.plaintext,
        placeholder: t.placeholder,
        autocomplete: t.autocomplete || null,
        'aria-required': t.required ? 'true' : null,
        'aria-invalid': t.computedAriaInvalid
      },
      on: {
        input: t.onInput,
        change: t.onChange
      }
    });
  },
  data: function data() {
    return {
      localValue: this.value
    };
  },

  props: {
    value: {
      default: null
    },
    type: {
      type: String,
      default: 'text',
      validator: function validator(type) {
        return arrayIncludes(TYPES, type);
      }
    },
    ariaInvalid: {
      type: [Boolean, String],
      default: false
    },
    readonly: {
      type: Boolean,
      default: false
    },
    plaintext: {
      type: Boolean,
      default: false
    },
    autocomplete: {
      type: String,
      default: null
    },
    placeholder: {
      type: String,
      default: null
    },
    formatter: {
      type: Function
    },
    lazyFormatter: {
      type: Boolean,
      default: false
    }
  },
  computed: {
    localType: function localType() {
      // We only allow certain types
      return arrayIncludes(TYPES, this.type) ? this.type : 'text';
    },
    inputClass: function inputClass() {
      return [this.plaintext ? 'form-control-plaintext' : 'form-control',
      // Fix missing width:100% in Bootstrap V4.beta.2
      this.plaintext ? 'w-100' : '', this.sizeFormClass, this.stateClass];
    },
    computedAriaInvalid: function computedAriaInvalid() {
      if (!this.ariaInvalid || this.ariaInvalid === 'false') {
        // this.ariaInvalid is null or false or 'false'
        return this.computedState === false ? 'true' : null;
      }
      if (this.ariaInvalid === true) {
        // User wants explicit aria-invalid=true
        return 'true';
      }
      // Most likely a string value (which could be 'true')
      return this.ariaInvalid;
    }
  },
  watch: {
    value: function value(newVal, oldVal) {
      if (newVal !== oldVal) {
        this.localValue = newVal;
      }
    },
    localValue: function localValue(newVal, oldVal) {
      if (newVal !== oldVal) {
        this.$emit('input', newVal);
      }
    }
  },
  methods: {
    format: function format(value, e) {
      if (this.formatter) {
        var formattedValue = this.formatter(value, e);
        if (formattedValue !== value) {
          return formattedValue;
        }
      }
      return value;
    },
    onInput: function onInput(evt) {
      var value = evt.target.value;
      if (this.lazyFormatter) {
        // Update the model with the current unformated value
        this.localValue = value;
      } else {
        this.localValue = this.format(value, evt);
      }
    },
    onChange: function onChange(evt) {
      this.localValue = this.format(evt.target.value, evt);
      this.$emit('change', this.localValue);
    },
    focus: function focus() {
      if (!this.disabled) {
        this.$el.focus();
      }
    }
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$18 = {
  bFormInput: bFormInput$2,
  bInput: bFormInput$2
};

var VuePlugin$35 = {
  install: function install(Vue) {
    registerComponents(Vue, components$18);
  }
};

vueUse(VuePlugin$35);

var bFormTextarea = {
  mixins: [idMixin, formMixin, formSizeMixin, formStateMixin],
  render: function render(h) {
    var t = this;
    return h('textarea', {
      ref: 'input',
      class: t.inputClass,
      style: t.inputStyle,
      directives: [{ name: 'model', rawName: 'v-model', value: t.localValue, expression: 'localValue' }],
      domProps: { value: t.value },
      attrs: {
        id: t.safeId(),
        name: t.name,
        disabled: t.disabled,
        placeholder: t.placeholder,
        required: t.required,
        autocomplete: t.autocomplete || null,
        readonly: t.readonly || t.plaintext,
        rows: t.rowsCount,
        wrap: t.wrap || null,
        'aria-required': t.required ? 'true' : null,
        'aria-invalid': t.computedAriaInvalid
      },
      on: {
        input: function input(evt) {
          t.localValue = evt.target.value;
        }
      }
    });
  },
  data: function data() {
    return {
      localValue: this.value
    };
  },

  props: {
    value: {
      type: String,
      default: ''
    },
    ariaInvalid: {
      type: [Boolean, String],
      default: false
    },
    readonly: {
      type: Boolean,
      default: false
    },
    plaintext: {
      type: Boolean,
      default: false
    },
    autocomplete: {
      type: String,
      default: null
    },
    placeholder: {
      type: String,
      default: null
    },
    rows: {
      type: [Number, String],
      default: null
    },
    maxRows: {
      type: [Number, String],
      default: null
    },
    wrap: {
      // 'soft', 'hard' or 'off'. Browser default is 'soft'
      type: String,
      default: 'soft'
    },
    noResize: {
      type: Boolean,
      default: false
    }
  },
  computed: {
    rowsCount: function rowsCount() {
      // A better option could be based on https://codepen.io/vsync/pen/frudD
      // As linebreaks aren't added until the input is submitted
      var rows = parseInt(this.rows, 10) || 1;
      var maxRows = parseInt(this.maxRows, 10) || 0;
      var lines = (this.localValue || '').toString().split('\n').length;
      return maxRows ? Math.min(maxRows, Math.max(rows, lines)) : Math.max(rows, lines);
    },
    inputClass: function inputClass() {
      return [this.plaintext ? 'form-control-plaintext' : 'form-control',
      // Interim fix until BS V4.beta.3 is released
      this.plaintext ? 'w-100' : '', this.sizeFormClass, this.stateClass];
    },
    inputStyle: function inputStyle() {
      // We set width 100% in plaintext mode to get around a shortcoming in bootstrap CSS
      // setting noResize to true will disable the ability for the user to resize the textarea
      return {
        width: this.plaintext ? '100%' : null,
        resize: this.noResize ? 'none' : null
      };
    },
    computedAriaInvalid: function computedAriaInvalid() {
      if (!this.ariaInvalid || this.ariaInvalid === 'false') {
        // this.ariaInvalid is null or false or 'false'
        return this.computedState === false ? 'true' : null;
      }
      if (this.ariaInvalid === true) {
        // User wants explicit aria-invalid=true
        return 'true';
      }
      // Most likely a string value (which could be the string 'true')
      return this.ariaInvalid;
    }
  },
  watch: {
    value: function value(newVal, oldVal) {
      // Update our localValue
      if (newVal !== oldVal) {
        this.localValue = newVal;
      }
    },
    localValue: function localValue(newVal, oldVal) {
      // update Parent value
      if (newVal !== oldVal) {
        this.$emit('input', newVal);
      }
    }
  },
  methods: {
    focus: function focus() {
      // For external handler that may want a focus method
      if (!this.disabled) {
        this.$el.focus();
      }
    }
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$19 = {
  bFormTextarea: bFormTextarea,
  bTextarea: bFormTextarea
};

var VuePlugin$37 = {
  install: function install(Vue) {
    registerComponents(Vue, components$19);
  }
};

vueUse(VuePlugin$37);

var bFormFile$2 = {
  mixins: [idMixin, formMixin, formStateMixin, formCustomMixin],
  render: function render(h) {
    var t = this;
    // Form Input
    var input = h('input', {
      ref: 'input',
      class: t.inputClasses,
      attrs: {
        type: 'file',
        id: t.safeId(),
        name: t.name,
        disabled: t.disabled,
        required: t.required,
        capture: t.capture || null,
        'aria-required': t.required ? 'true' : null,
        accept: t.accept || null,
        multiple: t.multiple,
        webkitdirectory: t.directory,
        'aria-describedby': t.plain ? null : t.safeId('_BV_file_control_')
      },
      on: {
        change: t.onFileChange,
        focusin: t.focusHandler,
        focusout: t.focusHandler
      }
    });
    if (t.plain) {
      return input;
    }

    // 'Drop Here' target
    var droptarget = h(false);
    if (t.dragging) {
      droptarget = h('span', {
        class: ['drop-here'],
        attrs: { 'data-drop': t.dropLabel },
        on: {
          dragover: t.dragover,
          drop: t.drop,
          dragleave: t.dragleave
        }
      });
    }
    // Overlay Labels
    var labels = h('span', {
      class: ['custom-file-control', t.dragging ? 'dragging' : null],
      attrs: {
        id: t.safeId('_BV_file_control_'),
        'data-choose': t.computedChooseLabel,
        'data-selected': t.selectedLabel
      }
    });

    // Return rendered custom file input
    return h('label', {
      class: ['custom-file', 'b-form-file', t.stateClass, 'w-100', 'd-block'],
      attrs: { id: t.safeId('_BV_file_outer_') },
      on: { dragover: t.dragover }
    }, [droptarget, input, labels]);
  },
  data: function data() {
    return {
      selectedFile: null,
      dragging: false,
      hasFocus: false
    };
  },

  props: {
    accept: {
      type: String,
      default: ''
    },
    capture: {
      // Instruct input to capture from camera
      type: Boolean,
      default: false
    },
    placeholder: {
      type: String,
      default: null
    },
    chooseLabel: {
      type: String,
      default: null
    },
    multiple: {
      type: Boolean,
      default: false
    },
    directory: {
      type: Boolean,
      default: false
    },
    noTraverse: {
      type: Boolean,
      default: false
    },
    selectedFormat: {
      type: String,
      default: ':count Files'
    },
    noDrop: {
      type: Boolean,
      default: false
    },
    dropLabel: {
      type: String,
      default: 'Drop files here'
    }
  },
  computed: {
    inputClasses: function inputClasses() {
      return [{
        'form-control-file': this.plain,
        'custom-file-input': this.custom,
        'w-100': true, // BS4 beta missing this
        'focus': this.custom && this.hasFocus
      }, this.stateClass];
    },
    selectedLabel: function selectedLabel() {
      if (!this.selectedFile || this.selectedFile.length === 0) {
        return this.placeholder || 'No file chosen';
      }
      if (this.multiple) {
        if (this.selectedFile.length === 1) {
          return this.selectedFile[0].name;
        }
        return this.selectedFormat.replace(':names', this.selectedFile.map(function (file) {
          return file.name;
        }).join(',')).replace(':count', this.selectedFile.length);
      }
      return this.selectedFile.name;
    },
    computedChooseLabel: function computedChooseLabel() {
      return this.chooseLabel || (this.multiple ? 'Choose Files' : 'Choose File');
    }
  },
  watch: {
    selectedFile: function selectedFile(newVal, oldVal) {
      if (newVal === oldVal) {
        return;
      }
      if (!newVal && this.multiple) {
        this.$emit('input', []);
      } else {
        this.$emit('input', newVal);
      }
    }
  },
  methods: {
    focusHandler: function focusHandler(evt) {
      // Boostrap v4.beta doesn't have focus styling for custom file input
      // Firefox has a borked '[type=file]:focus ~ sibling' selector issue,
      // So we add a 'focus' class to get around these "bugs"
      if (this.plain || evt.type === 'focusout') {
        this.hasFocus = false;
      } else {
        // Add focus styling for custom file input
        this.hasFocus = true;
      }
    },
    reset: function reset() {
      try {
        // Wrapped in try in case IE < 11 craps out
        this.$refs.input.value = '';
      } catch (e) {}
      // IE < 11 doesn't support setting input.value to '' or null
      // So we use this little extra hack to reset the value, just in case
      // This also appears to work on modern browsers as well.
      this.$refs.input.type = '';
      this.$refs.input.type = 'file';
      this.selectedFile = this.multiple ? [] : null;
    },
    onFileChange: function onFileChange(evt) {
      var _this = this;

      // Always emit original event
      this.$emit('change', evt);
      // Check if special `items` prop is available on event (drop mode)
      // Can be disabled by setting no-traverse
      var items = evt.dataTransfer && evt.dataTransfer.items;
      if (items && !this.noTraverse) {
        var queue = [];
        for (var i = 0; i < items.length; i++) {
          var item = items[i].webkitGetAsEntry();
          if (item) {
            queue.push(this.traverseFileTree(item));
          }
        }
        Promise.all(queue).then(function (filesArr) {
          _this.setFiles(from(filesArr));
        });
        return;
      }
      // Normal handling
      this.setFiles(evt.target.files || evt.dataTransfer.files);
    },
    setFiles: function setFiles(files) {
      if (!files) {
        this.selectedFile = null;
        return;
      }
      if (!this.multiple) {
        this.selectedFile = files[0];
        return;
      }
      // Convert files to array
      var filesArray = [];
      for (var i = 0; i < files.length; i++) {
        if (files[i].type.match(this.accept)) {
          filesArray.push(files[i]);
        }
      }
      this.selectedFile = filesArray;
    },
    dragover: function dragover(evt) {
      evt.preventDefault();
      evt.stopPropagation();
      if (this.noDrop || !this.custom) {
        return;
      }
      this.dragging = true;
      evt.dataTransfer.dropEffect = 'copy';
    },
    dragleave: function dragleave(evt) {
      evt.preventDefault();
      evt.stopPropagation();
      this.dragging = false;
    },
    drop: function drop(evt) {
      evt.preventDefault();
      evt.stopPropagation();
      if (this.noDrop) {
        return;
      }
      this.dragging = false;
      if (evt.dataTransfer.files && evt.dataTransfer.files.length > 0) {
        this.onFileChange(evt);
      }
    },
    traverseFileTree: function traverseFileTree(item, path) {
      var _this2 = this;

      // Based on http://stackoverflow.com/questions/3590058
      return new Promise(function (resolve) {
        path = path || '';
        if (item.isFile) {
          // Get file
          item.file(function (file) {
            file.$path = path; // Inject $path to file obj
            resolve(file);
          });
        } else if (item.isDirectory) {
          // Get folder contents
          item.createReader().readEntries(function (entries) {
            var queue = [];
            for (var i = 0; i < entries.length; i++) {
              queue.push(_this2.traverseFileTree(entries[i], path + item.name + '/'));
            }
            Promise.all(queue).then(function (filesArr) {
              resolve(from(filesArr));
            });
          });
        }
      });
    }
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$20 = {
  bFormFile: bFormFile$2,
  bFile: bFormFile$2
};

var VuePlugin$39 = {
  install: function install(Vue) {
    registerComponents(Vue, components$20);
  }
};

vueUse(VuePlugin$39);

var bFormSelect = {
  mixins: [idMixin, formMixin, formSizeMixin, formStateMixin, formCustomMixin, formOptionsMixin],
  render: function render(h) {
    var t = this;
    var $slots = t.$slots;
    var options = t.formOptions.map(function (option, index) {
      return h('option', {
        key: 'option_' + index + '_opt',
        attrs: { disabled: Boolean(option.disabled) },
        domProps: { innerHTML: option.text, value: option.value }
      });
    });
    return h('select', {
      ref: 'input',
      class: t.inputClass,
      directives: [{ name: 'model', rawName: 'v-model', value: t.localValue, expression: 'localValue' }],
      attrs: {
        id: t.safeId(),
        name: t.name,
        multiple: t.multiple || null,
        size: t.multiple || t.selectSize > 1 ? t.selectSize : null,
        disabled: t.disabled,
        required: t.required,
        'aria-required': t.required ? 'true' : null,
        'aria-invalid': t.computedAriaInvalid
      },
      on: {
        change: function change(evt) {
          var target = evt.target;
          var selectedVal = from(target.options).filter(function (o) {
            return o.selected;
          }).map(function (o) {
            return '_value' in o ? o._value : o.value;
          });
          t.localValue = target.multiple ? selectedVal : selectedVal[0];
          t.$emit('change', t.localValue);
        }
      }
    }, [$slots.first, options, $slots.default]);
  },
  data: function data() {
    return {
      localValue: this.value
    };
  },

  watch: {
    value: function value(newVal, oldVal) {
      this.localValue = newVal;
    },
    localValue: function localValue(newVal, oldVal) {
      this.$emit('input', this.localValue);
    }
  },
  props: {
    value: {},
    multiple: {
      type: Boolean,
      default: false
    },
    selectSize: {
      // Browsers default size to 0, which shows 4 rows in most browsers in multiple mode
      // Size of 1 can bork out firefox
      type: Number,
      default: 0
    },
    ariaInvalid: {
      type: [Boolean, String],
      default: false
    }
  },
  computed: {
    inputClass: function inputClass() {
      return ['form-control', this.stateClass, this.sizeFormClass, this.plain || !this.multiple && this.selectSize > 1 ? null : 'custom-select'];
    },
    computedAriaInvalid: function computedAriaInvalid() {
      if (this.ariaInvalid === true || this.ariaInvalid === 'true') {
        return 'true';
      }
      return this.stateClass === 'is-invalid' ? 'true' : null;
    }
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$21 = {
  bFormSelect: bFormSelect,
  bSelect: bFormSelect
};

var VuePlugin$41 = {
  install: function install(Vue) {
    registerComponents(Vue, components$21);
  }
};

vueUse(VuePlugin$41);

var THROTTLE = 100;

var bImgLazy = {
  components: { bImg: bImg },
  render: function render(h) {
    var t = this;
    return h('b-img', {
      props: {
        src: t.computedSrc,
        alt: t.alt,
        blank: t.computedBlank,
        blankColor: t.blankColor,
        width: t.computedWidth,
        height: t.computedHeight,
        fluid: t.fluid,
        fluidGrow: t.fluidGrow,
        block: t.block,
        thumbnail: t.thumbnail,
        rounded: t.rounded,
        left: t.left,
        right: t.right,
        center: t.center
      }
    });
  },
  data: function data() {
    return {
      isShown: false,
      scrollTimeout: null
    };
  },

  props: {
    src: {
      type: String,
      default: null,
      rqeuired: true
    },
    alt: {
      type: String,
      default: null
    },
    width: {
      type: [Number, String],
      default: null
    },
    height: {
      type: [Number, String],
      default: null
    },
    blankSrc: {
      // If null, a blank image is generated
      type: String,
      default: null
    },
    blankColor: {
      type: String,
      default: 'transparent'
    },
    blankWidth: {
      type: [Number, String],
      default: null
    },
    blankHeight: {
      type: [Number, String],
      default: null
    },
    fluid: {
      type: Boolean,
      default: false
    },
    fluidGrow: {
      type: Boolean,
      default: false
    },
    block: {
      type: Boolean,
      default: false
    },
    thumbnail: {
      type: Boolean,
      default: false
    },
    rounded: {
      type: [Boolean, String],
      default: false
    },
    left: {
      type: Boolean,
      default: false
    },
    right: {
      type: Boolean,
      default: false
    },
    center: {
      type: Boolean,
      default: false
    },
    offset: {
      type: [Number, String],
      default: 360
    },
    throttle: {
      type: [Number, String],
      default: THROTTLE
    }
  },
  computed: {
    computedSrc: function computedSrc() {
      return !this.blankSrc || this.isShown ? this.src : this.blankSrc;
    },
    computedBlank: function computedBlank() {
      return !(this.isShown || this.blankSrc);
    },
    computedWidth: function computedWidth() {
      return this.isShown ? this.width : this.blankWidth || this.width;
    },
    computedHeight: function computedHeight() {
      return this.isShown ? this.height : this.blankHeight || this.height;
    }
  },
  mounted: function mounted() {
    this.setListeners(true);
    this.checkView();
  },
  activated: function activated() {
    this.setListeners(true);
    this.checkView();
  },
  deactivated: function deactivated() {
    this.setListeners(false);
  },
  beforeDdestroy: function beforeDdestroy() {
    this.setListeners(false);
  },

  methods: {
    setListeners: function setListeners(on) {
      clearTimeout(this.scrollTimer);
      this.scrollTimout = null;
      var root = window;
      if (on) {
        eventOn(root, 'scroll', this.onScroll);
        eventOn(root, 'resize', this.onScroll);
        eventOn(root, 'orientationchange', this.onScroll);
      } else {
        eventOff(root, 'scroll', this.onScroll);
        eventOff(root, 'resize', this.onScroll);
        eventOff(root, 'orientationchange', this.onScroll);
      }
    },
    checkView: function checkView() {
      // check bounding box + offset to see if we should show
      if (!isVisible(this.$el)) {
        // Element is hidden, so skip for now
        return;
      }
      var offset$$1 = parseInt(this.offset, 10) || 0;
      var docElement = document.documentElement;
      var view = {
        l: 0 - offset$$1,
        t: 0 - offset$$1,
        b: docElement.clientHeight + offset$$1,
        r: docElement.clientWidth + offset$$1
      };
      var box = getBCR(this.$el);
      if (box.right >= view.l && box.bottom >= view.t && box.left <= view.r && box.top <= view.b) {
        // image is in view (or about to be in view)
        this.isShown = true;
        this.setListeners(false);
      }
    },
    onScroll: function onScroll() {
      if (this.isShown) {
        this.setListeners(false);
      } else {
        clearTimeout(this.scrollTimeout);
        this.scrollTimeout = setTimeout(this.checkView, parseInt(this.throttle, 10) || THROTTLE);
      }
    }
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$22 = {
  bImg: bImg,
  bImgLazy: bImgLazy
};

var VuePlugin$43 = {
  install: function install(Vue) {
    registerComponents(Vue, components$22);
  }
};

vueUse(VuePlugin$43);

var props$32 = {
  fluid: {
    type: Boolean,
    default: false
  },
  containerFluid: {
    type: Boolean,
    default: false
  },
  header: {
    type: String,
    default: null
  },
  headerTag: {
    type: String,
    default: 'h1'
  },
  headerLevel: {
    type: [Number, String],
    default: '3'
  },
  lead: {
    type: String,
    default: null
  },
  leadTag: {
    type: String,
    default: 'p'
  },
  tag: {
    type: String,
    default: 'div'
  },
  bgVariant: {
    type: String,
    default: null
  },
  borderVariant: {
    type: String,
    default: null
  },
  textVariant: {
    type: String,
    default: null
  }
};

var bJumbotron = {
  functional: true,
  props: props$32,
  render: function render(h, _ref) {
    var _class2;

    var props = _ref.props,
        data = _ref.data,
        slots = _ref.slots;

    // The order of the conditionals matter.
    // We are building the component markup in order.
    var childNodes = [];

    // Header
    if (props.header || slots().header) {
      childNodes.push(h(props.headerTag, {
        class: defineProperty$1({}, 'display-' + props.headerLevel, Boolean(props.headerLevel))
      }, slots().header || props.header));
    }

    // Lead
    if (props.lead || slots().lead) {
      childNodes.push(h(props.leadTag, { staticClass: 'lead' }, slots().lead || props.lead));
    }

    // Default slot
    if (slots().default) {
      childNodes.push(slots().default);
    }

    // If fluid, wrap content in a container/container-fluid
    if (props.fluid) {
      // Children become a child of a container
      childNodes = [h(Container, { props: { 'fluid': props.containerFluid } }, childNodes)];
    }
    // Return the jumbotron
    return h(props.tag, lib_common(data, {
      staticClass: 'jumbotron',
      class: (_class2 = {
        'jumbotron-fluid': props.fluid
      }, defineProperty$1(_class2, 'text-' + props.textVariant, Boolean(props.textVariant)), defineProperty$1(_class2, 'bg-' + props.bgVariant, Boolean(props.bgVariant)), defineProperty$1(_class2, 'border-' + props.borderVariant, Boolean(props.borderVariant)), defineProperty$1(_class2, 'border', Boolean(props.borderVariant)), _class2)
    }), childNodes);
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$23 = {
  bJumbotron: bJumbotron
};

var VuePlugin$45 = {
  install: function install(Vue) {
    registerComponents(Vue, components$23);
  }
};

vueUse(VuePlugin$45);

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$24 = {
  bLink: Link
};

var VuePlugin$47 = {
  install: function install(Vue) {
    registerComponents(Vue, components$24);
  }
};

vueUse(VuePlugin$47);

var props$33 = {
  tag: {
    type: String,
    default: 'div'
  },
  flush: {
    type: Boolean,
    default: false
  }
};

var bListGroup = {
  functional: true,
  props: props$33,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    var componentData = {
      staticClass: 'list-group',
      class: { 'list-group-flush': props.flush }
    };

    return h(props.tag, lib_common(data, componentData), children);
  }
};

var actionTags = ['a', 'router-link', 'button', 'b-link'];
var linkProps$2 = propsFactory();
delete linkProps$2.href.default;
delete linkProps$2.to.default;

var props$34 = assign(linkProps$2, {
  tag: {
    type: String,
    default: 'div'
  },
  action: {
    type: Boolean,
    default: null
  },
  variant: {
    type: String,
    default: null
  }
});

var bListGroupItem = {
  functional: true,
  props: props$34,
  render: function render(h, _ref) {
    var _class;

    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    var tag = !props.href && !props.to ? props.tag : Link;

    var componentData = {
      staticClass: 'list-group-item',
      class: (_class = {
        'list-group-flush': props.flush
      }, defineProperty$1(_class, 'list-group-item-' + props.variant, Boolean(props.variant)), defineProperty$1(_class, 'active', props.active), defineProperty$1(_class, 'disabled', props.disabled), defineProperty$1(_class, 'list-group-item-action', Boolean(props.href || props.to || props.action || arrayIncludes(actionTags, props.tag))), _class),
      props: pluckProps(linkProps$2, props)
    };

    return h(tag, lib_common(data, componentData), children);
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$25 = {
  bListGroup: bListGroup,
  bListGroupItem: bListGroupItem
};

var VuePlugin$49 = {
  install: function install(Vue) {
    registerComponents(Vue, components$25);
  }
};

vueUse(VuePlugin$49);

var props$36 = {
  tag: {
    type: String,
    default: 'div'
  }
};

var MediaBody = {
  functional: true,
  props: props$36,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    return h(props.tag, lib_common(data, {
      staticClass: 'media-body'
    }), children);
  }
};

var props$37 = {
  tag: {
    type: String,
    default: 'div'
  },
  verticalAlign: {
    type: String,
    default: 'top'
  }
};

var MediaAside = {
  functional: true,
  props: props$37,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    return h(props.tag, lib_common(data, {
      staticClass: 'd-flex',
      class: defineProperty$1({}, 'align-self-' + props.verticalAlign, props.verticalAlign)
    }), children);
  }
};

var props$35 = {
  tag: {
    type: String,
    default: 'div'
  },
  rightAlign: {
    type: Boolean,
    default: false
  },
  verticalAlign: {
    type: String,
    default: 'top'
  },
  noBody: {
    type: Boolean,
    default: false
  }
};

var bMedia = {
  functional: true,
  props: props$35,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        slots = _ref.slots,
        children = _ref.children;

    var childNodes = props.noBody ? children : [];

    if (!props.noBody) {
      if (slots().aside && !props.rightAlign) {
        childNodes.push(h(MediaAside, { staticClass: 'mr-3', props: { verticalAlign: props.verticalAlign } }, slots().aside));
      }

      childNodes.push(h(MediaBody, slots().default));

      if (slots().aside && props.rightAlign) {
        childNodes.push(h(MediaAside, { staticClass: 'ml-3', props: { verticalAlign: props.verticalAlign } }, slots().aside));
      }
    }

    return h(props.tag, lib_common(data, { staticClass: 'media' }), childNodes);
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$26 = {
  bMedia: bMedia,
  bMediaAside: MediaAside,
  bMediaBody: MediaBody
};

var VuePlugin$51 = {
  install: function install(Vue) {
    registerComponents(Vue, components$26);
  }
};

vueUse(VuePlugin$51);

var BvEvent = function () {
  function BvEvent(type) {
    var eventInit = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    classCallCheck(this, BvEvent);

    // Start by emulating native Event constructor.
    if (!type) {
      throw new TypeError('Failed to construct \'' + this.constructor.name + '\'. 1 argument required, ' + arguments.length + ' given.');
    }
    // Assign defaults first, the eventInit,
    // and the type last so it can't be overwritten.
    assign(this, BvEvent.defaults(), eventInit, { type: type });
    // Freeze some props as readonly, but leave them enumerable.
    defineProperties(this, {
      type: readonlyDescriptor(),
      cancelable: readonlyDescriptor(),
      nativeEvent: readonlyDescriptor(),
      target: readonlyDescriptor(),
      relatedTarget: readonlyDescriptor(),
      vueTarget: readonlyDescriptor()
    });
    // Create a private variable using closure scoping.
    var defaultPrevented = false;
    // Recreate preventDefault method. One way setter.
    this.preventDefault = function preventDefault() {
      if (this.cancelable) {
        defaultPrevented = true;
      }
    };
    // Create 'defaultPrevented' publicly accessible prop
    // that can only be altered by the preventDefault method.
    defineProperty(this, 'defaultPrevented', {
      enumerable: true,
      get: function get$$1() {
        return defaultPrevented;
      }
    });
  }

  createClass(BvEvent, null, [{
    key: 'defaults',
    value: function defaults$$1() {
      return {
        type: '',
        cancelable: true,
        nativeEvent: null,
        target: null,
        relatedTarget: null,
        vueTarget: null
      };
    }
  }]);
  return BvEvent;
}();

// Selectors for padding/margin adjustments
var Selector = {
  FIXED_CONTENT: '.fixed-top, .fixed-bottom, .is-fixed, .sticky-top',
  STICKY_CONTENT: '.sticky-top',
  NAVBAR_TOGGLER: '.navbar-toggler'

  // ObserveDom config
};var OBSERVER_CONFIG$1 = {
  subtree: true,
  childList: true,
  characterData: true,
  attributes: true,
  attributeFilter: ['style', 'class']
};

var bModal$2 = {
  mixins: [idMixin, listenOnRootMixin],
  components: { bBtn: bBtn, bBtnClose: bBtnClose },
  render: function render(h) {
    var t = this;
    var $slots = t.$slots;
    // Modal Header
    var header = h(false);
    if (!t.hideHeader) {
      var modalHeader = $slots['modal-header'];
      if (!modalHeader) {
        var closeButton = h(false);
        if (!t.hideHeaderClose) {
          closeButton = h('b-btn-close', {
            props: {
              disabled: t.is_transitioning,
              ariaLabel: t.headerCloseLabel,
              textVariant: t.headerTextVariant
            },
            on: { click: function click(evt) {
                t.hide('header-close');
              } }
          }, [$slots['modal-header-close']]);
        }
        modalHeader = [h(t.titleTag, { class: ['modal-title'] }, [$slots['modal-title'] || t.title]), closeButton];
      }
      header = h('header', {
        ref: 'header',
        class: t.headerClasses,
        attrs: { id: t.safeId('__BV_modal_header_') }
      }, [modalHeader]);
    }
    // Modal Body
    var body = h('div', {
      ref: 'body',
      class: t.bodyClasses,
      attrs: { id: t.safeId('__BV_modal_body_') }
    }, [$slots.default]);
    // Modal Footer
    var footer = h(false);
    if (!t.hideFooter) {
      var modalFooter = $slots['modal-footer'];
      if (!modalFooter) {
        var okButton = h(false);
        if (!t.okOnly) {
          okButton = h('b-btn', {
            props: {
              variant: t.cancelVariant,
              size: t.buttonSize,
              disabled: t.cancelDisabled || t.busy || t.is_transitioning
            },
            on: { click: function click(evt) {
                t.hide('cancel');
              } }
          }, [$slots['modal-cancel'] || t.cancelTitle]);
        }
        var cancelButton = h('b-btn', {
          props: {
            variant: t.okVariant,
            size: t.buttonSize,
            disabled: t.okDisabled || t.busy || t.is_transitioning
          },
          on: { click: function click(evt) {
              t.hide('ok');
            } }
        }, [$slots['modal-ok'] || t.okTitle]);
        modalFooter = [cancelButton, okButton];
      }
      footer = h('footer', {
        ref: 'footer',
        class: t.footerClasses,
        attrs: { id: t.safeId('__BV_modal_footer_') }
      }, [modalFooter]);
    }
    // Assemble Modal Content
    var modalContent = h('div', {
      ref: 'content',
      class: ['modal-content'],
      attrs: {
        tabindex: '-1',
        role: 'document',
        'aria-labelledby': t.hideHeader ? null : t.safeId('__BV_modal_header_'),
        'aria-describedby': t.safeId('__BV_modal_body_')
      },
      on: {
        focusout: t.onFocusout,
        click: function click(evt) {
          evt.stopPropagation();
        }
      }
    }, [header, body, footer]);
    // Modal Dialog wrapper
    var modalDialog = h('div', { class: t.dialogClasses }, [modalContent]);
    // Modal
    var modal = h('div', {
      ref: 'modal',
      class: t.modalClasses,
      directives: [{ name: 'show', rawName: 'v-show', value: t.is_visible, expression: 'is_visible' }],
      attrs: {
        id: t.safeId(),
        role: 'dialog',
        'aria-hidden': t.is_visible ? null : 'true'
      },
      on: {
        click: t.onClickOut,
        keydown: t.onEsc
      }
    }, [modalDialog]);
    // Wrap modal in transition
    modal = h('transition', {
      props: {
        enterClass: '',
        enterToClass: '',
        enterActiveClass: '',
        leaveClass: '',
        leaveActiveClass: '',
        leaveToClass: ''
      },
      on: {
        'before-enter': t.onBeforeEnter,
        'enter': t.onEnter,
        'after-enter': t.onAfterEnter,
        'before-leave': t.onBeforeLeave,
        'leave': t.onLeave,
        'after-leave': t.onAfterLeave
      }
    }, [modal]);
    // Modal Backdrop
    var backdrop = h(false);
    if (!t.hideBackdrop && (t.is_visible || t.is_transitioning)) {
      backdrop = h('div', { class: t.backdropClasses, attrs: { id: t.safeId('__BV_modal_backdrop_') } });
    }
    // Assemble modal and backdrop
    var outer = h(false);
    if (!t.is_hidden) {
      outer = h('div', { attrs: { id: t.safeId('__BV_modal_outer_') } }, [modal, backdrop]);
    }
    // Wrap in DIV to maintain thi.$el reference for hide/show method aceess
    return h('div', {}, [outer]);
  },
  data: function data() {
    return {
      is_hidden: this.lazy || false,
      is_visible: false,
      is_transitioning: false,
      is_show: false,
      is_block: false,
      scrollbarWidth: 0,
      isBodyOverflowing: false,
      return_focus: this.returnFocus || null
    };
  },

  model: {
    prop: 'visible',
    event: 'change'
  },
  props: {
    title: {
      type: String,
      default: ''
    },
    titleTag: {
      type: String,
      default: 'h5'
    },
    size: {
      type: String,
      default: 'md'
    },
    centered: {
      type: Boolean,
      default: false
    },
    buttonSize: {
      type: String,
      default: ''
    },
    noFade: {
      type: Boolean,
      default: false
    },
    noCloseOnBackdrop: {
      type: Boolean,
      default: false
    },
    noCloseOnEsc: {
      type: Boolean,
      default: false
    },
    noEnforceFocus: {
      type: Boolean,
      default: false
    },
    headerBgVariant: {
      type: String,
      default: null
    },
    headerBorderVariant: {
      type: String,
      default: null
    },
    headerTextVariant: {
      type: String,
      default: null
    },
    bodyBgVariant: {
      type: String,
      default: null
    },
    bodyTextVariant: {
      type: String,
      default: null
    },
    footerBgVariant: {
      type: String,
      default: null
    },
    footerBorderVariant: {
      type: String,
      default: null
    },
    footerTextVariant: {
      type: String,
      default: null
    },
    hideHeader: {
      type: Boolean,
      default: false
    },
    hideFooter: {
      type: Boolean,
      default: false
    },
    hideHeaderClose: {
      type: Boolean,
      default: false
    },
    hideBackdrop: {
      type: Boolean,
      default: false
    },
    okOnly: {
      type: Boolean,
      default: false
    },
    okDisabled: {
      type: Boolean,
      default: false
    },
    cancelDisabled: {
      type: Boolean,
      default: false
    },
    visible: {
      type: Boolean,
      default: false
    },
    returnFocus: {
      default: null
    },
    headerCloseLabel: {
      type: String,
      default: 'Close'
    },
    cancelTitle: {
      type: String,
      default: 'Cancel'
    },
    okTitle: {
      type: String,
      default: 'OK'
    },
    cancelVariant: {
      type: String,
      default: 'secondary'
    },
    okVariant: {
      type: String,
      default: 'primary'
    },
    lazy: {
      type: Boolean,
      default: false
    },
    busy: {
      type: Boolean,
      default: false
    }
  },
  computed: {
    modalClasses: function modalClasses() {
      return ['modal', {
        fade: !this.noFade,
        show: this.is_show,
        'd-block': this.is_block
      }];
    },
    dialogClasses: function dialogClasses() {
      var _ref;

      return ['modal-dialog', (_ref = {}, defineProperty$1(_ref, 'modal-' + this.size, Boolean(this.size)), defineProperty$1(_ref, 'modal-dialog-centered', this.centered), _ref)];
    },
    backdropClasses: function backdropClasses() {
      return ['modal-backdrop', {
        fade: !this.noFade,
        show: this.is_show || this.noFade
      }];
    },
    headerClasses: function headerClasses() {
      var _ref2;

      return ['modal-header', (_ref2 = {
        'rounded-top': Boolean(this.headerBgVariant)
      }, defineProperty$1(_ref2, 'bg-' + this.headerBgVariant, Boolean(this.headerBgVariant)), defineProperty$1(_ref2, 'text-' + this.headerTextVariant, Boolean(this.headerTextVariant)), defineProperty$1(_ref2, 'border-' + this.headerBorderVariant, Boolean(this.headerBorderVariant)), _ref2)];
    },
    bodyClasses: function bodyClasses() {
      var _ref3;

      return ['modal-body', (_ref3 = {}, defineProperty$1(_ref3, 'bg-' + this.bodyBgVariant, Boolean(this.bodyBgVariant)), defineProperty$1(_ref3, 'text-' + this.bodyTextVariant, Boolean(this.bodyTextVariant)), _ref3)];
    },
    footerClasses: function footerClasses() {
      var _ref4;

      return ['modal-footer', (_ref4 = {
        'rounded-bottom': Boolean(this.footerBgVariant)
      }, defineProperty$1(_ref4, 'bg-' + this.footerBgVariant, Boolean(this.footerBgVariant)), defineProperty$1(_ref4, 'text-' + this.footerTextVariant, Boolean(this.footerTextVariant)), defineProperty$1(_ref4, 'border-' + this.footerBorderVariant, Boolean(this.footerBorderVariant)), _ref4)];
    }
  },
  watch: {
    visible: function visible(newVal, oldVal) {
      if (newVal === oldVal) {
        return;
      }
      this[newVal ? 'show' : 'hide']();
    }
  },
  methods: {
    // Public Methods
    show: function show() {
      if (this.is_visible) {
        return;
      }
      var showEvt = new BvEvent('show', {
        cancelable: true,
        vueTarget: this,
        target: this.$refs.modal,
        relatedTarget: null
      });
      this.emitEvent(showEvt);
      if (showEvt.defaultPrevented || this.is_visible) {
        // Don't show if canceled
        return;
      }
      if (hasClass(document.body, 'modal-open')) {
        // If another modal is already open, wait for it to close
        this.$root.$once('bv::modal::hidden', this.doShow);
      } else {
        // Show the modal
        this.doShow();
      }
    },
    hide: function hide(trigger) {
      if (!this.is_visible) {
        return;
      }
      var hideEvt = new BvEvent('hide', {
        cancelable: true,
        vueTarget: this,
        target: this.$refs.modal,
        // this could be the trigger element/component reference
        relatedTarget: null,
        isOK: trigger || null,
        trigger: trigger || null,
        cancel: function cancel() {
          // Backwards compatibility
          warn('b-modal: evt.cancel() is deprecated. Please use evt.preventDefault().');
          this.preventDefault();
        }
      });
      if (trigger === 'ok') {
        this.$emit('ok', hideEvt);
      } else if (trigger === 'cancel') {
        this.$emit('cancel', hideEvt);
      }
      this.emitEvent(hideEvt);
      // Hide if not canceled
      if (hideEvt.defaultPrevented || !this.is_visible) {
        return;
      }
      // stop observing for content changes
      if (this._observer) {
        this._observer.disconnect();
        this._observer = null;
      }
      this.is_visible = false;
      this.$emit('change', false);
    },

    // Private method to finish showing modal
    doShow: function doShow() {
      var _this = this;

      // Plce modal in DOM if lazy
      this.is_hidden = false;
      this.$nextTick(function () {
        // We do this in nextTick to ensure the modal is in DOM first before we show it
        _this.is_visible = true;
        _this.$emit('change', true);
        // Observe changes in modal content and adjust if necessary
        _this._observer = observeDOM(_this.$refs.content, _this.adjustDialog.bind(_this), OBSERVER_CONFIG$1);
      });
    },

    // Transition Handlers
    onBeforeEnter: function onBeforeEnter() {
      this.is_transitioning = true;
      this.checkScrollbar();
      this.setScrollbar();
      this.adjustDialog();
      addClass(document.body, 'modal-open');
      this.setResizeEvent(true);
    },
    onEnter: function onEnter() {
      this.is_block = true;
      this.$refs.modal.scrollTop = 0;
    },
    onAfterEnter: function onAfterEnter() {
      var _this2 = this;

      this.is_show = true;
      this.is_transitioning = false;
      this.$nextTick(function () {
        _this2.focusFirst();
        var shownEvt = new BvEvent('shown', {
          cancelable: false,
          vueTarget: _this2,
          target: _this2.$refs.modal,
          relatedTarget: null
        });
        _this2.emitEvent(shownEvt);
      });
    },
    onBeforeLeave: function onBeforeLeave() {
      this.is_transitioning = true;
      this.setResizeEvent(false);
    },
    onLeave: function onLeave() {
      // Remove the 'show' class
      this.is_show = false;
    },
    onAfterLeave: function onAfterLeave() {
      var _this3 = this;

      this.is_block = false;
      this.resetAdjustments();
      this.resetScrollbar();
      this.is_transitioning = false;
      removeClass(document.body, 'modal-open');
      this.$nextTick(function () {
        _this3.is_hidden = _this3.lazy || false;
        _this3.returnFocusTo();
        var hiddenEvt = new BvEvent('hidden', {
          cancelable: false,
          vueTarget: _this3,
          target: _this3.lazy ? null : _this3.$refs.modal,
          relatedTarget: null
        });
        _this3.emitEvent(hiddenEvt);
      });
    },

    // Event emitter
    emitEvent: function emitEvent(bvEvt) {
      var type = bvEvt.type;
      this.$emit(type, bvEvt);
      this.$root.$emit('bv::modal::' + type, bvEvt);
    },

    // UI Event Handlers
    onClickOut: function onClickOut(evt) {
      // If backdrop clicked, hide modal
      if (this.is_visible && !this.noCloseOnBackdrop) {
        this.hide('backdrop');
      }
    },
    onEsc: function onEsc(evt) {
      // If ESC pressed, hide modal
      if (evt.keyCode === KeyCodes.ESC && this.is_visible && !this.noCloseOnEsc) {
        this.hide('esc');
      }
    },
    onFocusout: function onFocusout(evt) {
      // If focus leaves modal, bring it back
      // 'focusout' Event Listener bound on content
      var content = this.$refs.content;
      if (!this.noEnforceFocus && this.is_visible && content && !content.contains(evt.relatedTarget)) {
        content.focus();
      }
    },

    // Resize Listener
    setResizeEvent: function setResizeEvent(on) {
      var _this4 = this;

      ['resize', 'orientationchange'].forEach(function (evtName) {
        if (on) {
          eventOn(window, evtName, _this4.adjustDialog);
        } else {
          eventOff(window, evtName, _this4.adjustDialog);
        }
      });
    },

    // Root Listener handlers
    showHandler: function showHandler(id, triggerEl) {
      if (id === this.id) {
        this.return_focus = triggerEl || null;
        this.show();
      }
    },
    hideHandler: function hideHandler(id) {
      if (id === this.id) {
        this.hide();
      }
    },
    modalListener: function modalListener(bvEvt) {
      // If another modal opens, close this one
      if (bvEvt.vueTarget !== this) {
        this.hide();
      }
    },

    // Focus control handlers
    focusFirst: function focusFirst() {
      // Don't try and focus if we are SSR
      if (typeof document === 'undefined') {
        return;
      }
      var content = this.$refs.content;
      var modal = this.$refs.modal;
      var activeElement = document.activeElement;
      if (activeElement && content && content.contains(activeElement)) {
        // If activeElement is child of content, no need to change focus
      } else if (content) {
        if (modal) {
          modal.scrollTop = 0;
        }
        // Focus the modal content wrapper
        content.focus();
      }
    },
    returnFocusTo: function returnFocusTo() {
      // Prefer returnFocus prop over event specified return_focus value
      var el = this.returnFocus || this.return_focus || null;
      if (typeof el === 'string') {
        // CSS Selector
        el = select(el);
      }
      if (el) {
        el = el.$el || el;
        if (isVisible(el)) {
          el.focus();
        }
      }
    },

    // Utility methods
    getScrollbarWidth: function getScrollbarWidth() {
      var scrollDiv = document.createElement('div');
      scrollDiv.className = 'modal-scrollbar-measure';
      document.body.appendChild(scrollDiv);
      this.scrollbarWidth = scrollDiv.getBoundingClientRect().width - scrollDiv.clientWidth;
      document.body.removeChild(scrollDiv);
    },
    adjustDialog: function adjustDialog() {
      if (!this.is_visible) {
        return;
      }
      var modal = this.$refs.modal;
      var isModalOverflowing = modal.scrollHeight > document.documentElement.clientHeight;
      if (!this.isBodyOverflowing && isModalOverflowing) {
        modal.style.paddingLeft = this.scrollbarWidth + 'px';
      }
      if (this.isBodyOverflowing && !isModalOverflowing) {
        modal.style.paddingRight = this.scrollbarWidth + 'px';
      }
    },
    resetAdjustments: function resetAdjustments() {
      var modal = this.$refs.modal;
      if (modal) {
        modal.style.paddingLeft = '';
        modal.style.paddingRight = '';
      }
    },
    checkScrollbar: function checkScrollbar() {
      var rect = getBCR(document.body);
      this.isBodyOverflowing = rect.left + rect.right < window.innerWidth;
    },
    setScrollbar: function setScrollbar() {
      if (this.isBodyOverflowing) {
        // Note: DOMNode.style.paddingRight returns the actual value or '' if not set
        //   while $(DOMNode).css('padding-right') returns the calculated value or 0 if not set
        var computedStyle = window.getComputedStyle;
        var body = document.body;
        var scrollbarWidth = this.scrollbarWidth;
        // Adjust fixed content padding
        selectAll(Selector.FIXED_CONTENT).forEach(function (el) {
          var actualPadding = el.style.paddingRight;
          var calculatedPadding = computedStyle(el).paddingRight || 0;
          setAttr(el, 'data-padding-right', actualPadding);
          el.style.paddingRight = parseFloat(calculatedPadding) + scrollbarWidth + 'px';
        });
        // Adjust sticky content margin
        selectAll(Selector.STICKY_CONTENT).forEach(function (el) {
          var actualMargin = el.style.marginRight;
          var calculatedMargin = computedStyle(el).marginRight || 0;
          setAttr(el, 'data-margin-right', actualMargin);
          el.style.marginRight = parseFloat(calculatedMargin) - scrollbarWidth + 'px';
        });
        // Adjust navbar-toggler margin
        selectAll(Selector.NAVBAR_TOGGLER).forEach(function (el) {
          var actualMargin = el.style.marginRight;
          var calculatedMargin = computedStyle(el).marginRight || 0;
          setAttr(el, 'data-margin-right', actualMargin);
          el.style.marginRight = parseFloat(calculatedMargin) + scrollbarWidth + 'px';
        });
        // Adjust body padding
        var actualPadding = body.style.paddingRight;
        var calculatedPadding = computedStyle(body).paddingRight;
        setAttr(body, 'data-padding-right', actualPadding);
        body.style.paddingRight = parseFloat(calculatedPadding) + scrollbarWidth + 'px';
      }
    },
    resetScrollbar: function resetScrollbar() {
      // Restore fixed content padding
      selectAll(Selector.FIXED_CONTENT).forEach(function (el) {
        if (hasAttr(el, 'data-padding-right')) {
          el.style.paddingRight = getAttr(el, 'data-padding-right') || '';
          removeAttr(el, 'data-padding-right');
        }
      });
      // Restore sticky content and navbar-toggler margin
      selectAll(Selector.STICKY_CONTENT + ', ' + Selector.NAVBAR_TOGGLER).forEach(function (el) {
        if (hasAttr(el, 'data-margin-right')) {
          el.style.marginRight = getAttr(el, 'data-margin-right') || '';
          removeAttr(el, 'data-margin-right');
        }
      });
      // Restore body padding
      var body = document.body;
      if (hasAttr(body, 'data-padding-right')) {
        body.style.paddingRight = getAttr(body, 'data-padding-right') || '';
        removeAttr(body, 'data-padding-right');
      }
    }
  },
  created: function created() {
    // create non-reactive property
    this._observer = null;
  },
  mounted: function mounted() {
    // Measure scrollbar
    this.getScrollbarWidth();
    // Listen for events from others to either open or close ourselves
    this.listenOnRoot('bv::show::modal', this.showHandler);
    this.listenOnRoot('bv::hide::modal', this.hideHandler);
    // Listen for bv:modal::show events, and close ourselves if the opening modal not us
    this.listenOnRoot('bv::modal::show', this.modalListener);
    // Initially show modal?
    if (this.visible === true) {
      this.show();
    }
  },
  beforeDestroy: function beforeDestroy() {
    // Ensure everything is back to normal
    if (this._observer) {
      this._observer.disconnect();
      this._observer = null;
    }
    this.setResizeEvent(false);
    // Re-adjust body/navbar/fixed padding/margins (if needed)
    removeClass(document.body, 'modal-open');
    this.resetAdjustments();
    this.resetScrollbar();
  }
};

var listenTypes$1 = { click: true };

var bModal$3 = {
  // eslint-disable-next-line no-shadow-restricted-names
  bind: function bind(undefined, binding, vnode) {
    targets(vnode, binding, listenTypes$1, function (_ref) {
      var targets$$1 = _ref.targets,
          vnode = _ref.vnode;

      targets$$1.forEach(function (target) {
        vnode.context.$root.$emit('bv::show::modal', target, vnode.elm);
      });
    });
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var directives$1 = {
  bModal: bModal$3
};

var VuePlugin$55 = {
  install: function install(Vue) {
    registerDirectives(Vue, directives$1);
  }
};

vueUse(VuePlugin$55);

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$27 = {
  bModal: bModal$2
};

var VuePlugin$53 = {
  install: function install(Vue) {
    registerComponents(Vue, components$27);
    Vue.use(VuePlugin$55);
  }
};

vueUse(VuePlugin$53);

var props$38 = {
  tag: {
    type: String,
    default: 'ul'
  },
  fill: {
    type: Boolean,
    default: false
  },
  justified: {
    type: Boolean,
    default: false
  },
  tabs: {
    type: Boolean,
    default: false
  },
  pills: {
    type: Boolean,
    default: false
  },
  vertical: {
    type: Boolean,
    default: false
  },
  isNavBar: {
    type: Boolean,
    default: false
  }
};

var bNav = {
  functional: true,
  props: props$38,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    if (props.isNavBar) {
      warn("b-nav: Prop 'is-nav-bar' is deprecated. Please use component '<b-navbar-nav>' instead.");
    }
    return h(props.tag, lib_common(data, {
      class: {
        'nav': !props.isNavBar,
        'navbar-nav': props.isNavBar,
        'nav-tabs': props.tabs && !props.isNavBar,
        'nav-pills': props.pills && !props.isNavBar,
        'flex-column': props.vertical && !props.isNavBar,
        'nav-fill': props.fill,
        'nav-justified': props.justified
      }
    }), children);
  }
};

var props$39 = propsFactory();

var bNavItem = {
  functional: true,
  props: props$39,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    return h('li', lib_common(data, {
      staticClass: 'nav-item'
    }), [h(Link, { staticClass: 'nav-link', props: props }, children)]);
  }
};

var props$40 = {
  tag: {
    type: String,
    default: 'span'
  }
};

var bNavText = {
  functional: true,
  props: props$40,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    return h(props.tag, lib_common(data, { staticClass: 'navbar-text' }), children);
  }
};

var bNavForm = {
  functional: true,
  props: {
    id: {
      type: String,
      default: null
    }
  },
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    return h(Form, lib_common(data, { attrs: { id: props.id }, props: { inline: true } }), children);
  }
};

var bNavItemDropdown = {
  mixins: [idMixin, dropdownMixin],
  render: function render(h) {
    var t = this;
    var button = h('a', {
      class: t.toggleClasses,
      ref: 'toggle',
      attrs: {
        href: '#',
        id: t.safeId('_BV_button_'),
        disabled: t.disabled,
        'aria-haspopup': 'true',
        'aria-expanded': t.visible ? 'true' : 'false'
      },
      on: {
        click: t.toggle,
        keydown: t.toggle // space, enter, down
      }
    }, [t.$slots['button-content'] || t.$slots.text || h('span', { domProps: { innerHTML: t.text } })]);
    var menu = h('div', {
      class: t.menuClasses,
      ref: 'menu',
      attrs: { 'aria-labelledby': t.safeId('_BV_button_') },
      on: {
        mouseover: t.onMouseOver,
        keydown: t.onKeydown // tab, up, down, esc
      }
    }, [this.$slots.default]);
    return h('li', { attrs: { id: t.safeId() }, class: t.dropdownClasses }, [button, menu]);
  },

  computed: {
    isNav: function isNav() {
      // Signal to dropdown mixin that we are in a navbar
      return true;
    },
    dropdownClasses: function dropdownClasses() {
      return ['nav-item', 'b-nav-dropdown', 'dropdown', this.dropup ? 'dropup' : '', this.visible ? 'show' : ''];
    },
    toggleClasses: function toggleClasses() {
      return ['nav-link', this.noCaret ? '' : 'dropdown-toggle', this.disabled ? 'disabled' : ''];
    },
    menuClasses: function menuClasses() {
      return ['dropdown-menu', this.right ? 'dropdown-menu-right' : 'dropdown-menu-left', this.visible ? 'show' : ''];
    }
  },
  props: {
    noCaret: {
      type: Boolean,
      default: false
    },
    role: {
      type: String,
      default: 'menu'
    }
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$28 = {
  bNav: bNav,
  bNavItem: bNavItem,
  bNavText: bNavText,
  bNavForm: bNavForm,
  bNavItemDropdown: bNavItemDropdown,
  bNavItemDd: bNavItemDropdown,
  bNavDropdown: bNavItemDropdown,
  bNavDd: bNavItemDropdown
};

var VuePlugin$56 = {
  install: function install(Vue) {
    registerComponents(Vue, components$28);
    Vue.use(VuePlugin$24);
  }
};

vueUse(VuePlugin$56);

var props$41 = {
  tag: {
    type: String,
    default: 'nav'
  },
  type: {
    type: String,
    default: 'light'
  },
  variant: {
    type: String
  },
  toggleable: {
    type: [Boolean, String],
    default: false
  },
  toggleBreakpoint: {
    // Deprecated.  Set toggleable to a string breakpoint
    type: String,
    default: null
  },
  fixed: {
    type: String
  },
  sticky: {
    type: Boolean,
    default: false
  }
};

var bNavbar = {
  functional: true,
  props: props$41,
  render: function render(h, _ref) {
    var _class;

    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    var breakpoint = props.toggleBreakpoint || (props.toggleable === true ? 'sm' : props.toggleable) || 'sm';
    return h(props.tag, lib_common(data, {
      staticClass: 'navbar',
      class: (_class = {}, defineProperty$1(_class, 'navbar-' + props.type, Boolean(props.type)), defineProperty$1(_class, 'bg-' + props.variant, Boolean(props.variant)), defineProperty$1(_class, 'fixed-' + props.fixed, Boolean(props.fixed)), defineProperty$1(_class, 'sticky-top', props.sticky), defineProperty$1(_class, 'navbar-expand-' + breakpoint, props.toggleable !== false), _class)
    }), children);
  }
};

var props$42 = {
  tag: {
    type: String,
    default: 'ul'
  },
  fill: {
    type: Boolean,
    default: false
  },
  justified: {
    type: Boolean,
    default: false
  }
};

var bNavbarNav = {
  functional: true,
  props: props$42,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    return h(props.tag, lib_common(data, {
      staticClass: 'navbar-nav',
      class: {
        'nav-fill': props.fill,
        'nav-justified': props.justified
      }
    }), children);
  }
};

var linkProps$3 = propsFactory();
linkProps$3.href.default = undefined;
linkProps$3.to.default = undefined;

var props$43 = assign(linkProps$3, {
  tag: {
    type: String,
    default: 'div'
  }
});

var bNavbarBrand = {
  functional: true,
  props: props$43,
  render: function render(h, _ref) {
    var props = _ref.props,
        data = _ref.data,
        children = _ref.children;

    var isLink = Boolean(props.to || props.href);
    var tag = isLink ? Link : props.tag;

    return h(tag, lib_common(data, {
      staticClass: 'navbar-brand',
      props: isLink ? pluckProps(linkProps$3, props) : {}
    }), children);
  }
};

var bNavbarToggle = {
  mixins: [listenOnRootMixin],
  render: function render(h) {
    var t = this;
    return h('button', {
      class: ['navbar-toggler'],
      attrs: {
        type: 'button',
        'aria-label': t.label,
        'aria-controls': t.target,
        'aria-expanded': t.toggleState ? 'true' : 'false'
      },
      on: { click: t.onClick }
    }, [t.$slots.default || h('span', { class: ['navbar-toggler-icon'] })]);
  },
  data: function data() {
    return {
      toggleState: false
    };
  },

  props: {
    label: {
      type: String,
      default: 'Toggle navigation'
    },
    target: {
      type: String,
      required: true
    }
  },
  methods: {
    onClick: function onClick() {
      this.$root.$emit('bv::toggle::collapse', this.target);
    },
    handleStateEvt: function handleStateEvt(id, state) {
      if (id === this.target) {
        this.toggleState = state;
      }
    }
  },
  created: function created() {
    this.listenOnRoot('bv::collapse::state', this.handleStateEvt);
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$29 = {
  bNavbar: bNavbar,
  bNavbarNav: bNavbarNav,
  bNavbarBrand: bNavbarBrand,
  bNavbarToggle: bNavbarToggle,
  bNavToggle: bNavbarToggle
};

var VuePlugin$57 = {
  install: function install(Vue) {
    registerComponents(Vue, components$29);
    Vue.use(VuePlugin$56);
    Vue.use(VuePlugin$22);
    Vue.use(VuePlugin$24);
  }
};

vueUse(VuePlugin$57);

var props$44 = {
  perPage: {
    type: Number,
    default: 20
  },
  totalRows: {
    type: Number,
    default: 20
  },
  ariaControls: {
    type: String,
    default: null
  }

  // Our render function is brought in from the pagination mixin
};var bPagination = {
  mixins: [paginationMixin],
  props: props$44,
  computed: {
    numberOfPages: function numberOfPages() {
      var result = Math.ceil(this.totalRows / this.perPage);
      return result < 1 ? 1 : result;
    }
  },
  methods: {
    // These methods are used by the render function
    onClick: function onClick(num, evt) {
      var _this = this;

      // Handle edge cases where number of pages has changed (i.e. if perPage changes)
      if (num > this.numberOfPages) {
        num = this.numberOfPages;
      } else if (num < 1) {
        num = 1;
      }
      this.currentPage = num;
      this.$nextTick(function () {
        // Keep the current button focused if possible
        var target = evt.target;
        if (isVisible(target) && _this.$el.contains(target) && target.focus) {
          target.focus();
        } else {
          _this.focusCurrent();
        }
      });
      this.$emit('change', this.currentPage);
    },
    makePage: function makePage(pagenum) {
      return pagenum;
    },
    linkProps: function linkProps(pagenum) {
      return { href: '#' };
    }
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$30 = {
  bPagination: bPagination
};

var VuePlugin$59 = {
  install: function install(Vue) {
    registerComponents(Vue, components$30);
  }
};

vueUse(VuePlugin$59);

// Props needed for router links
var routerProps = pickLinkProps('activeClass', 'exactActiveClass', 'append', 'exact', 'replace', 'target', 'rel');

// Props object
var props$45 = assign(
// pagination-nav specific props
{
  numberOfPages: {
    type: Number,
    default: 1
  },
  baseUrl: {
    type: String,
    default: '/'
  },
  useRouter: {
    type: Boolean,
    default: false
  },
  linkGen: {
    type: Function,
    default: null
  },
  pageGen: {
    type: Function,
    default: null
  }
},
// Router specific props
routerProps);
// Our render function is brought in via the pagination mixin
var bPaginationNav = {
  mixins: [paginationMixin],
  props: props$45,
  computed: {
    // Used by render function to trigger wraping in '<nav>' element
    isNav: function isNav() {
      return true;
    }
  },
  methods: {
    onClick: function onClick(pageNum, evt) {
      this.currentPage = pageNum;
    },
    makePage: function makePage(pagenum) {
      if (this.pageGen && typeof this.pageGen === 'function') {
        return this.pageGen(pagenum);
      }
      return pagenum;
    },
    makeLink: function makeLink(pagenum) {
      if (this.linkGen && typeof this.linkGen === 'function') {
        return this.linkGen(pagenum);
      }
      var link = '' + this.baseUrl + pagenum;
      return this.useRouter ? { path: link } : link;
    },
    linkProps: function linkProps(pagenum) {
      var link = this.makeLink(pagenum);
      var props = {
        href: typeof link === 'string' ? link : void 0,
        target: this.target || null,
        rel: this.rel || null,
        disabled: this.disabled
      };
      if (this.useRouter || (typeof link === 'undefined' ? 'undefined' : _typeof(link)) === 'object') {
        props = assign(props, {
          to: link,
          exact: this.exact,
          activeClass: this.activeClass,
          exactActiveClass: this.exactActiveClass,
          append: this.append,
          replace: this.replace
        });
      }
      return props;
    }
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$31 = {
  bPaginationNav: bPaginationNav
};

var VuePlugin$61 = {
  install: function install(Vue) {
    registerComponents(Vue, components$31);
  }
};

vueUse(VuePlugin$61);

var NAME$1 = 'tooltip';
var CLASS_PREFIX$1 = 'bs-tooltip';
var BSCLS_PREFIX_REGEX$1 = new RegExp('\\b' + CLASS_PREFIX$1 + '\\S+', 'g');

var TRANSITION_DURATION = 150;

// Modal $root event (prepare for future evnt name change)
var MODAL_CLOSE_EVENT = 'bv::modal::hidden';
var MODAL_CLASS = '.modal';

var AttachmentMap$1 = {
  AUTO: 'auto',
  TOP: 'top',
  RIGHT: 'right',
  BOTTOM: 'bottom',
  LEFT: 'left',
  TOPLEFT: 'top',
  TOPRIGHT: 'top',
  RIGHTTOP: 'right',
  RIGHTBOTTOM: 'right',
  BOTTOMLEFT: 'bottom',
  BOTTOMRIGHT: 'bottom',
  LEFTTOP: 'left',
  LEFTBOTTOM: 'left'
};

var OffsetMap = {
  AUTO: 0,
  TOPLEFT: -1,
  TOP: 0,
  TOPRIGHT: +1,
  RIGHTTOP: -1,
  RIGHT: 0,
  RIGHTBOTTOM: +1,
  BOTTOMLEFT: -1,
  BOTTOM: 0,
  BOTTOMRIGHT: +1,
  LEFTTOP: -1,
  LEFT: 0,
  LEFTBOTTOM: +1
};

var HoverState = {
  SHOW: 'show',
  OUT: 'out'
};

var ClassName$1 = {
  FADE: 'fade',
  SHOW: 'show'
};

var Selector$2 = {
  TOOLTIP: '.tooltip',
  TOOLTIP_INNER: '.tooltip-inner',
  ARROW: '.arrow'

  // ESLINT: Not used
  // const Trigger = {
  //   HOVER: 'hover',
  //   FOCUS: 'focus',
  //   CLICK: 'click',
  //   BLUR: 'blur',
  //   MANUAL: 'manual'
  // }

};var Defaults$1 = {
  animation: true,
  template: '<div class="tooltip" role="tooltip">' + '<div class="arrow"></div>' + '<div class="tooltip-inner"></div>' + '</div>',
  trigger: 'hover focus',
  title: '',
  delay: 0,
  html: false,
  placement: 'top',
  offset: 0,
  arrowPadding: 6,
  container: false,
  fallbackPlacement: 'flip',
  callbacks: {}

  // Transition Event names
};var TransitionEndEvents$1 = {
  WebkitTransition: ['webkitTransitionEnd'],
  MozTransition: ['transitionend'],
  OTransition: ['otransitionend', 'oTransitionEnd'],
  transition: ['transitionend']

  // Client Side Tip ID counter for aria-describedby attribute
  // Could use Alex's uid generator util
  // Each tooltip requires a unique client side ID
};var NEXTID = 1;
function generateId(name) {
  return '__BV_' + name + '_' + NEXTID++ + '__';
}

/*
 * ToolTip Class definition
 */

var ToolTip = function () {
  // Main constructor
  function ToolTip(element, config, $root) {
    classCallCheck(this, ToolTip);

    // New tooltip object
    this.$isEnabled = true;
    this.$fadeTimeout = null;
    this.$hoverTimeout = null;
    this.$visibleInterval = null;
    this.$hoverState = '';
    this.$activeTrigger = {};
    this.$popper = null;
    this.$element = element;
    this.$tip = null;
    this.$id = generateId(this.constructor.NAME);
    this.$root = $root || null;
    this.$routeWatcher = null;
    // We use a bound version of the following handlers for root/modal listeners to maintain the 'this' context
    this.$forceHide = this.forceHide.bind(this);
    this.$doHide = this.doHide.bind(this);
    this.$doShow = this.doShow.bind(this);
    this.$doDisable = this.doDisable.bind(this);
    this.$doEnable = this.doEnable.bind(this);
    // Set the configuration
    this.updateConfig(config);
  }

  // NOTE: Overridden by PopOver class


  createClass(ToolTip, [{
    key: 'updateConfig',


    // Update config
    value: function updateConfig(config) {
      // Merge config into defaults. We use "this" here because PopOver overrides Default
      var updatedConfig = assign({}, this.constructor.Default, config);

      // Sanitize delay
      if (config.delay && typeof config.delay === 'number') {
        updatedConfig.delay = {
          show: config.delay,
          hide: config.delay
        };
      }

      // Title for tooltip and popover
      if (config.title && typeof config.title === 'number') {
        updatedConfig.title = config.title.toString();
      }

      // Content only for popover
      if (config.content && typeof config.content === 'number') {
        updatedConfig.content = config.content.toString();
      }

      // Hide element original title if needed
      this.fixTitle();
      // Update the config
      this.$config = updatedConfig;
      // Stop/Restart listening
      this.unListen();
      this.listen();
    }

    // Destroy this instance

  }, {
    key: 'destroy',
    value: function destroy() {
      // Stop listening to trigger events
      this.unListen();
      // Disable while open listeners/watchers
      this.setWhileOpenListeners(false);
      // Clear any timouts
      clearTimeout(this.$hoverTimeout);
      this.$hoverTimeout = null;
      clearTimeout(this.$fadeTimeout);
      this.$fadeTimeout = null;
      // Remove popper
      if (this.$popper) {
        this.$popper.destroy();
      }
      this.$popper = null;
      // Remove tip from document
      if (this.$tip && this.$tip.parentElement) {
        this.$tip.parentElement.removeChild(this.$tip);
      }
      this.$tip = null;
      // Null out other properties
      this.$id = null;
      this.$isEnabled = null;
      this.$root = null;
      this.$element = null;
      this.$config = null;
      this.$hoverState = null;
      this.$activeTrigger = null;
      this.$forceHide = null;
      this.$doHide = null;
      this.$doShow = null;
      this.$doDisable = null;
      this.$doEnable = null;
    }
  }, {
    key: 'enable',
    value: function enable() {
      // Create a non-cancelable BvEvent
      var enabledEvt = new BvEvent('enabled', {
        cancelable: false,
        target: this.$element,
        relatedTarget: null
      });
      this.$isEnabled = true;
      this.emitEvent(enabledEvt);
    }
  }, {
    key: 'disable',
    value: function disable() {
      // Create a non-cancelable BvEvent
      var disabledEvt = new BvEvent('disabled', {
        cancelable: false,
        target: this.$element,
        relatedTarget: null
      });
      this.$isEnabled = false;
      this.emitEvent(disabledEvt);
    }

    // Click toggler

  }, {
    key: 'toggle',
    value: function toggle(event) {
      if (!this.$isEnabled) {
        return;
      }
      if (event) {
        this.$activeTrigger.click = !this.$activeTrigger.click;

        if (this.isWithActiveTrigger()) {
          this.enter(null);
        } else {
          this.leave(null);
        }
      } else {
        if (hasClass(this.getTipElement(), ClassName$1.SHOW)) {
          this.leave(null);
        } else {
          this.enter(null);
        }
      }
    }

    // Show tooltip

  }, {
    key: 'show',
    value: function show() {
      var _this = this;

      if (!document.body.contains(this.$element) || !isVisible(this.$element)) {
        // If trigger element isn't in the DOM or is not visible
        return;
      }
      // Build tooltip element (also sets this.$tip)
      var tip = this.getTipElement();
      this.fixTitle();
      this.setContent(tip);
      if (!this.isWithContent(tip)) {
        // if No content, don't bother showing
        this.$tip = null;
        return;
      }

      // Set ID on tip and aria-describedby on element
      setAttr(tip, 'id', this.$id);
      this.addAriaDescribedby();

      // Set animation on or off
      if (this.$config.animation) {
        addClass(tip, ClassName$1.FADE);
      } else {
        removeClass(tip, ClassName$1.FADE);
      }

      var placement = this.getPlacement();
      var attachment = this.constructor.getAttachment(placement);
      this.addAttachmentClass(attachment);

      // Create a cancelable BvEvent
      var showEvt = new BvEvent('show', {
        cancelable: true,
        target: this.$element,
        relatedTarget: tip
      });
      this.emitEvent(showEvt);
      if (showEvt.defaultPrevented) {
        // Don't show if event cancelled
        this.$tip = null;
        return;
      }

      // Insert tooltip if needed
      var container = this.getContainer();
      if (!document.body.contains(tip)) {
        container.appendChild(tip);
      }

      // Refresh popper
      this.removePopper();
      this.$popper = new Popper(this.$element, tip, this.getPopperConfig(placement, tip));

      // Transitionend Callback
      var complete = function complete() {
        if (_this.$config.animation) {
          _this.fixTransition(tip);
        }
        var prevHoverState = _this.$hoverState;
        _this.$hoverState = null;
        if (prevHoverState === HoverState.OUT) {
          _this.leave(null);
        }
        // Create a non-cancelable BvEvent
        var shownEvt = new BvEvent('shown', {
          cancelable: false,
          target: _this.$element,
          relatedTarget: tip
        });
        _this.emitEvent(shownEvt);
      };

      // Enable while open listeners/watchers
      this.setWhileOpenListeners(true);

      // Show tip
      addClass(tip, ClassName$1.SHOW);

      // Start the transition/animation
      this.transitionOnce(tip, complete);
    }

    // handler for periodic visibility check

  }, {
    key: 'visibleCheck',
    value: function visibleCheck(on) {
      var _this2 = this;

      clearInterval(this.$visibleInterval);
      this.$visibleInterval = null;
      if (on) {
        this.$visibleInterval = setInterval(function () {
          var tip = _this2.getTipElement();
          if (tip && !isVisible(_this2.$element) && hasClass(tip, ClassName$1.SHOW)) {
            // Element is no longer visible, so force-hide the tooltip
            _this2.forceHide();
          }
        }, 100);
      }
    }
  }, {
    key: 'setWhileOpenListeners',
    value: function setWhileOpenListeners(on) {
      // Modal close events
      this.setModalListener(on);
      // Periodic $element visibility check
      // For handling when tip is in <keepalive>, tabs, carousel, etc
      this.visibleCheck(on);
      // Route change events
      this.setRouteWatcher(on);
      // Ontouch start listeners
      this.setOnTouchStartListener(on);
      if (on && /(focus|blur)/.test(this.$config.trigger)) {
        // If focus moves between trigger element and tip container, dont close
        eventOn(this.$tip, 'focusout', this);
      } else {
        eventOff(this.$tip, 'focusout', this);
      }
    }

    // force hide of tip (internal method)

  }, {
    key: 'forceHide',
    value: function forceHide() {
      if (!this.$tip || !hasClass(this.$tip, ClassName$1.SHOW)) {
        return;
      }
      // Disable while open listeners/watchers
      this.setWhileOpenListeners(false);
      // Clear any hover enter/leave event
      clearTimeout(this.$hoverTimeout);
      this.$hoverTimeout = null;
      this.$hoverState = '';
      // Hide the tip
      this.hide(null, true);
    }

    // Hide tooltip

  }, {
    key: 'hide',
    value: function hide(callback, force) {
      var _this3 = this;

      var tip = this.$tip;
      if (!tip) {
        return;
      }

      // Create a canelable BvEvent
      var hideEvt = new BvEvent('hide', {
        // We disable cancelling if force is true
        cancelable: !force,
        target: this.$element,
        relatedTarget: tip
      });
      this.emitEvent(hideEvt);
      if (hideEvt.defaultPrevented) {
        // Don't hide if event cancelled
        return;
      }

      // Transitionend Callback
      var complete = function complete() {
        if (_this3.$hoverState !== HoverState.SHOW && tip.parentNode) {
          // Remove tip from dom, and force recompile on next show
          tip.parentNode.removeChild(tip);
          _this3.removeAriaDescribedby();
          _this3.removePopper();
          _this3.$tip = null;
        }
        if (callback) {
          callback();
        }
        // Create a non-cancelable BvEvent
        var hiddenEvt = new BvEvent('hidden', {
          cancelable: false,
          target: _this3.$element,
          relatedTarget: null
        });
        _this3.emitEvent(hiddenEvt);
      };

      // Disable while open listeners/watchers
      this.setWhileOpenListeners(false);

      // If forced close, disable animation
      if (force) {
        removeClass(tip, ClassName$1.FADE);
      }
      // Hide tip
      removeClass(tip, ClassName$1.SHOW);

      this.$activeTrigger.click = false;
      this.$activeTrigger.focus = false;
      this.$activeTrigger.hover = false;

      // Start the hide transition
      this.transitionOnce(tip, complete);

      this.$hoverState = '';
    }
  }, {
    key: 'emitEvent',
    value: function emitEvent(evt) {
      var evtName = evt.type;
      if (this.$root && this.$root.$emit) {
        // Emit an event on $root
        this.$root.$emit('bv::' + this.constructor.NAME + '::' + evtName, evt);
      }
      var callbacks = this.$config.callbacks || {};
      if (typeof callbacks[evtName] === 'function') {
        callbacks[evtName](evt);
      }
    }
  }, {
    key: 'getContainer',
    value: function getContainer() {
      var container = this.$config.container;
      var body = document.body;
      // If we are in a modal, we append to the modal instead of body, unless a container is specified
      return container === false ? closest(MODAL_CLASS, this.$element) || body : select(container, body) || body;
    }

    // Will be overritten by popover if needed

  }, {
    key: 'addAriaDescribedby',
    value: function addAriaDescribedby() {
      // Add aria-describedby on trigger element, without removing any other IDs
      var desc = getAttr(this.$element, 'aria-describedby') || '';
      desc = desc.split(/\s+/).concat(this.$id).join(' ').trim();
      setAttr(this.$element, 'aria-describedby', desc);
    }

    // Will be overritten by popover if needed

  }, {
    key: 'removeAriaDescribedby',
    value: function removeAriaDescribedby() {
      var _this4 = this;

      var desc = getAttr(this.$element, 'aria-describedby') || '';
      desc = desc.split(/\s+/).filter(function (d) {
        return d !== _this4.$id;
      }).join(' ').trim();
      if (desc) {
        setAttr(this.$element, 'aria-describedby', desc);
      } else {
        removeAttr(this.$element, 'aria-describedby');
      }
    }
  }, {
    key: 'removePopper',
    value: function removePopper() {
      if (this.$popper) {
        this.$popper.destroy();
      }
      this.$popper = null;
    }
  }, {
    key: 'transitionOnce',
    value: function transitionOnce(tip, complete) {
      var _this5 = this;

      var transEvents = this.getTransitionEndEvents();
      var called = false;
      clearTimeout(this.$fadeTimeout);
      this.$fadeTimeout = null;
      var fnOnce = function fnOnce() {
        if (called) {
          return;
        }
        called = true;
        clearTimeout(_this5.$fadeTimeout);
        _this5.$fadeTimeout = null;
        transEvents.forEach(function (evtName) {
          eventOff(tip, evtName, fnOnce);
        });
        // Call complete callback
        complete();
      };
      if (hasClass(tip, ClassName$1.FADE)) {
        transEvents.forEach(function (evtName) {
          eventOn(tip, evtName, fnOnce);
        });
        // Fallback to setTimeout
        this.$fadeTimeout = setTimeout(fnOnce, TRANSITION_DURATION);
      } else {
        fnOnce();
      }
    }

    // What transitionend event(s) to use? (returns array of event names)

  }, {
    key: 'getTransitionEndEvents',
    value: function getTransitionEndEvents() {
      for (var name in TransitionEndEvents$1) {
        if (this.$element.style[name] !== undefined) {
          return TransitionEndEvents$1[name];
        }
      }
      // fallback
      return [];
    }
  }, {
    key: 'update',
    value: function update() {
      if (this.$popper !== null) {
        this.$popper.scheduleUpdate();
      }
    }

    // NOTE: Overridden by PopOver class

  }, {
    key: 'isWithContent',
    value: function isWithContent(tip) {
      tip = tip || this.$tip;
      if (!tip) {
        return false;
      }
      return Boolean((select(Selector$2.TOOLTIP_INNER, tip) || {}).innerHTML);
    }

    // NOTE: Overridden by PopOver class

  }, {
    key: 'addAttachmentClass',
    value: function addAttachmentClass(attachment) {
      addClass(this.getTipElement(), CLASS_PREFIX$1 + '-' + attachment);
    }
  }, {
    key: 'getTipElement',
    value: function getTipElement() {
      if (!this.$tip) {
        // Try and compile user supplied template, or fallback to default template
        this.$tip = this.compileTemplate(this.$config.template) || this.compileTemplate(this.constructor.Default.template);
      }
      // Add tab index so tip can be focused, and to allow it to be set as relatedTargt in focusin/out events
      this.$tip.tabIndex = -1;
      return this.$tip;
    }
  }, {
    key: 'compileTemplate',
    value: function compileTemplate(html) {
      if (!html || typeof html !== 'string') {
        return null;
      }
      var div = document.createElement('div');
      div.innerHTML = html.trim();
      var node = div.firstElementChild ? div.removeChild(div.firstElementChild) : null;
      div = null;
      return node;
    }

    // NOTE: Overridden by PopOver class

  }, {
    key: 'setContent',
    value: function setContent(tip) {
      this.setElementContent(select(Selector$2.TOOLTIP_INNER, tip), this.getTitle());
      removeClass(tip, ClassName$1.FADE);
      removeClass(tip, ClassName$1.SHOW);
    }
  }, {
    key: 'setElementContent',
    value: function setElementContent(container, content) {
      if (!container) {
        // If container element doesn't exist, just return
        return;
      }
      var allowHtml = this.$config.html;
      if ((typeof content === 'undefined' ? 'undefined' : _typeof(content)) === 'object' && content.nodeType) {
        // content is a DOM node
        if (allowHtml) {
          if (content.parentElement !== container) {
            container.innerHtml = '';
            container.appendChild(content);
          }
        } else {
          container.innerText = content.innerText;
        }
      } else {
        // We have a plain HTML string or Text
        container[allowHtml ? 'innerHTML' : 'innerText'] = content;
      }
    }

    // NOTE: Overridden by PopOver class

  }, {
    key: 'getTitle',
    value: function getTitle() {
      var title = this.$config.title || '';
      if (typeof title === 'function') {
        // Call the function to get the title value
        title = title(this.$element);
      }
      if ((typeof title === 'undefined' ? 'undefined' : _typeof(title)) === 'object' && title.nodeType && !title.innerHTML.trim()) {
        // We have a DOM node, but without inner content, so just return empty string
        title = '';
      }
      if (typeof title === 'string') {
        title = title.trim();
      }
      if (!title) {
        // If an explicit title is not given, try element's title atributes
        title = getAttr(this.$element, 'title') || getAttr(this.$element, 'data-original-title') || '';
        title = title.trim();
      }

      return title;
    }
  }, {
    key: 'listen',
    value: function listen() {
      var _this6 = this;

      var triggers = this.$config.trigger.trim().split(/\s+/);
      var el = this.$element;

      // Listen for global show/hide events
      this.setRootListener(true);

      // Using 'this' as the handler will get automagically directed to this.handleEvent
      // And maintain our binding to 'this'
      triggers.forEach(function (trigger) {
        if (trigger === 'click') {
          eventOn(el, 'click', _this6);
        } else if (trigger === 'focus') {
          eventOn(el, 'focusin', _this6);
          eventOn(el, 'focusout', _this6);
        } else if (trigger === 'blur') {
          // Used to close $tip when element looses focus
          eventOn(el, 'focusout', _this6);
        } else if (trigger === 'hover') {
          eventOn(el, 'mouseenter', _this6);
          eventOn(el, 'mouseleave', _this6);
        }
      }, this);
    }
  }, {
    key: 'unListen',
    value: function unListen() {
      var _this7 = this;

      var events = ['click', 'focusin', 'focusout', 'mouseenter', 'mouseleave'];
      // Using "this" as the handler will get automagically directed to this.handleEvent
      events.forEach(function (evt) {
        eventOff(_this7.$element, evt, _this7);
      }, this);

      // Stop listening for global show/hide/enable/disable events
      this.setRootListener(false);
    }
  }, {
    key: 'handleEvent',
    value: function handleEvent(e) {
      // This special method allows us to use "this" as the event handlers
      if (isDisabled(this.$element)) {
        // If disabled, don't do anything. Note: if tip is shown before element gets
        // disabled, then tip not close until no longer disabled or forcefully closed.
        return;
      }
      if (!this.$isEnabled) {
        // If not enable
        return;
      }
      var type = e.type;
      var target = e.target;
      var relatedTarget = e.relatedTarget;
      var $element = this.$element;
      var $tip = this.$tip;
      if (type === 'click') {
        this.toggle(e);
      } else if (type === 'focusin' || type === 'mouseenter') {
        this.enter(e);
      } else if (type === 'focusout') {
        // target is the element which is loosing focus
        // And relatedTarget is the element gaining focus
        if ($tip && $element && $element.contains(target) && $tip.contains(relatedTarget)) {
          // If focus moves from $element to $tip, don't trigger a leave
          return;
        }
        if ($tip && $element && $tip.contains(target) && $element.contains(relatedTarget)) {
          // If focus moves from $tip to $element, don't trigger a leave
          return;
        }
        if ($tip && $tip.contains(target) && $tip.contains(relatedTarget)) {
          // If focus moves within $tip, don't trigger a leave
          return;
        }
        if ($element && $element.contains(target) && $element.contains(relatedTarget)) {
          // If focus moves within $element, don't trigger a leave
          return;
        }
        // Otherwise trigger a leave
        this.leave(e);
      } else if (type === 'mouseleave') {
        this.leave(e);
      }
    }
  }, {
    key: 'setRouteWatcher',
    value: function setRouteWatcher(on) {
      var _this8 = this;

      if (on) {
        this.setRouteWatcher(false);
        if (this.$root && Boolean(this.$root.$route)) {
          this.$routeWatcher = this.$root.$watch('$route', function (newVal, oldVal) {
            if (newVal === oldVal) {
              return;
            }
            // If route has changed, we force hide the tooltip/popover
            _this8.forceHide();
          });
        }
      } else {
        if (this.$routeWatcher) {
          // cancel the route watcher by calling hte stored reference
          this.$routeWatcher();
          this.$routeWatcher = null;
        }
      }
    }
  }, {
    key: 'setModalListener',
    value: function setModalListener(on) {
      var modal = closest(MODAL_CLASS, this.$element);
      if (!modal) {
        // If we are not in a modal, don't worry. be happy
        return;
      }
      // We can listen for modal hidden events on $root
      if (this.$root) {
        this.$root[on ? '$on' : '$off'](MODAL_CLOSE_EVENT, this.$forceHide);
      }
    }
  }, {
    key: 'setRootListener',
    value: function setRootListener(on) {
      // Listen for global 'bv::{hide|show}::{tooltip|popover}' hide request event
      if (this.$root) {
        this.$root[on ? '$on' : '$off']('bv::hide::' + this.constructor.NAME, this.$doHide);
        this.$root[on ? '$on' : '$off']('bv::show::' + this.constructor.NAME, this.$doShow);
        this.$root[on ? '$on' : '$off']('bv::disable::' + this.constructor.NAME, this.$doDisable);
        this.$root[on ? '$on' : '$off']('bv::enable::' + this.constructor.NAME, this.$doEnable);
      }
    }
  }, {
    key: 'doHide',
    value: function doHide(id) {
      // Programmatically hide tooltip or popover
      if (!id) {
        // Close all tooltips or popovers
        this.forceHide();
      } else if (this.$element && this.$element.id && this.$element.id === id) {
        // Close this specific tooltip or popover
        this.hide();
      }
    }
  }, {
    key: 'doShow',
    value: function doShow(id) {
      // Programmatically show tooltip or popover
      if (!id) {
        // Open all tooltips or popovers
        this.show();
      } else if (id && this.$element && this.$element.id && this.$element.id === id) {
        // Show this specific tooltip or popover
        this.show();
      }
    }
  }, {
    key: 'doDisable',
    value: function doDisable(id) {
      // Programmatically disable tooltip or popover
      if (!id) {
        // Disable all tooltips or popovers
        this.disable();
      } else if (this.$element && this.$element.id && this.$element.id === id) {
        // Disable this specific tooltip or popover
        this.disable();
      }
    }
  }, {
    key: 'doEnable',
    value: function doEnable(id) {
      // Programmatically enable tooltip or popover
      if (!id) {
        // Enable all tooltips or popovers
        this.enable();
      } else if (this.$element && this.$element.id && this.$element.id === id) {
        // Enable this specific tooltip or popover
        this.enable();
      }
    }
  }, {
    key: 'setOnTouchStartListener',
    value: function setOnTouchStartListener(on) {
      var _this9 = this;

      // if this is a touch-enabled device we add extra
      // empty mouseover listeners to the body's immediate children;
      // only needed because of broken event delegation on iOS
      // https://www.quirksmode.org/blog/archives/2014/02/mouse_event_bub.html
      if ('ontouchstart' in document.documentElement) {
        from(document.body.children).forEach(function (el) {
          if (on) {
            eventOn(el, 'mouseover', _this9._noop);
          } else {
            eventOff(el, 'mouseover', _this9._noop);
          }
        });
      }
    }
  }, {
    key: '_noop',
    value: function _noop() {
      // Empty noop handler for ontouchstart devices
    }
  }, {
    key: 'fixTitle',
    value: function fixTitle() {
      var el = this.$element;
      var titleType = _typeof(getAttr(el, 'data-original-title'));
      if (getAttr(el, 'title') || titleType !== 'string') {
        setAttr(el, 'data-original-title', getAttr(el, 'title') || '');
        setAttr(el, 'title', '');
      }
    }

    // Enter handler

  }, {
    key: 'enter',
    value: function enter(e) {
      var _this10 = this;

      if (e) {
        this.$activeTrigger[e.type === 'focusin' ? 'focus' : 'hover'] = true;
      }
      if (hasClass(this.getTipElement(), ClassName$1.SHOW) || this.$hoverState === HoverState.SHOW) {
        this.$hoverState = HoverState.SHOW;
        return;
      }
      clearTimeout(this.$hoverTimeout);
      this.$hoverState = HoverState.SHOW;
      if (!this.$config.delay || !this.$config.delay.show) {
        this.show();
        return;
      }
      this.$hoverTimeout = setTimeout(function () {
        if (_this10.$hoverState === HoverState.SHOW) {
          _this10.show();
        }
      }, this.$config.delay.show);
    }

    // Leave handler

  }, {
    key: 'leave',
    value: function leave(e) {
      var _this11 = this;

      if (e) {
        this.$activeTrigger[e.type === 'focusout' ? 'focus' : 'hover'] = false;
        if (e.type === 'focusout' && /blur/.test(this.$config.trigger)) {
          // Special case for `blur`: we clear out the other triggers
          this.$activeTrigger.click = false;
          this.$activeTrigger.hover = false;
        }
      }
      if (this.isWithActiveTrigger()) {
        return;
      }
      clearTimeout(this.$hoverTimeout);
      this.$hoverState = HoverState.OUT;
      if (!this.$config.delay || !this.$config.delay.hide) {
        this.hide();
        return;
      }
      this.$hoverTimeout = setTimeout(function () {
        if (_this11.$hoverState === HoverState.OUT) {
          _this11.hide();
        }
      }, this.$config.delay.hide);
    }
  }, {
    key: 'getPopperConfig',
    value: function getPopperConfig(placement, tip) {
      var _this12 = this;

      return {
        placement: this.constructor.getAttachment(placement),
        modifiers: {
          offset: { offset: this.getOffset(placement, tip) },
          flip: { behavior: this.$config.fallbackPlacement },
          arrow: { element: '.arrow' }
        },
        onCreate: function onCreate(data) {
          // Handle flipping arrow classes
          if (data.originalPlacement !== data.placement) {
            _this12.handlePopperPlacementChange(data);
          }
        },
        onUpdate: function onUpdate(data) {
          // Handle flipping arrow classes
          _this12.handlePopperPlacementChange(data);
        }
      };
    }
  }, {
    key: 'getOffset',
    value: function getOffset(placement, tip) {
      if (!this.$config.offset) {
        var arrow = select(Selector$2.ARROW, tip);
        var arrowOffset = parseFloat(getCS(arrow).width) + parseFloat(this.$config.arrowPadding);
        switch (OffsetMap[placement.toUpperCase()]) {
          case +1:
            return '+50%p - ' + arrowOffset + 'px';
          case -1:
            return '-50%p + ' + arrowOffset + 'px';
          default:
            return 0;
        }
      }
      return parseFloat(this.$config.offset);
    }
  }, {
    key: 'getPlacement',
    value: function getPlacement() {
      var placement = this.$config.placement;
      if (typeof placement === 'function') {
        return placement.call(this, this.$tip, this.$element);
      }
      return placement;
    }
  }, {
    key: 'isWithActiveTrigger',
    value: function isWithActiveTrigger() {
      for (var trigger in this.$activeTrigger) {
        if (this.$activeTrigger[trigger]) {
          return true;
        }
      }
      return false;
    }

    // NOTE: Overridden by PopOver class

  }, {
    key: 'cleanTipClass',
    value: function cleanTipClass() {
      var tip = this.getTipElement();
      var tabClass = tip.className.match(BSCLS_PREFIX_REGEX$1);
      if (tabClass !== null && tabClass.length > 0) {
        tabClass.forEach(function (cls) {
          removeClass(tip, cls);
        });
      }
    }
  }, {
    key: 'handlePopperPlacementChange',
    value: function handlePopperPlacementChange(data) {
      this.cleanTipClass();
      this.addAttachmentClass(this.constructor.getAttachment(data.placement));
    }
  }, {
    key: 'fixTransition',
    value: function fixTransition(tip) {
      var initConfigAnimation = this.$config.animation || false;
      if (getAttr(tip, 'x-placement') !== null) {
        return;
      }
      removeClass(tip, ClassName$1.FADE);
      this.$config.animation = false;
      this.hide();
      this.show();
      this.$config.animation = initConfigAnimation;
    }
  }], [{
    key: 'getAttachment',
    value: function getAttachment(placement) {
      return AttachmentMap$1[placement.toUpperCase()];
    }
  }, {
    key: 'Default',
    get: function get$$1() {
      return Defaults$1;
    }

    // NOTE: Overridden by PopOver class

  }, {
    key: 'NAME',
    get: function get$$1() {
      return NAME$1;
    }
  }]);
  return ToolTip;
}();

var NAME = 'popover';
var CLASS_PREFIX = 'bs-popover';
var BSCLS_PREFIX_REGEX = new RegExp('\\b' + CLASS_PREFIX + '\\S+', 'g');

var Defaults = assign({}, ToolTip.Default, {
  placement: 'right',
  trigger: 'click',
  content: '',
  template: '<div class="popover" role="tooltip">' + '<div class="arrow"></div>' + '<h3 class="popover-header"></h3>' + '<div class="popover-body"></div></div>'
});

var ClassName = {
  FADE: 'fade',
  SHOW: 'show'
};

var Selector$1 = {
  TITLE: '.popover-header',
  CONTENT: '.popover-body'
};

var PopOver = function (_ToolTip) {
  inherits(PopOver, _ToolTip);

  function PopOver() {
    classCallCheck(this, PopOver);
    return possibleConstructorReturn(this, (PopOver.__proto__ || Object.getPrototypeOf(PopOver)).apply(this, arguments));
  }

  createClass(PopOver, [{
    key: 'isWithContent',


    // Method overrides

    value: function isWithContent(tip) {
      tip = tip || this.$tip;
      if (!tip) {
        return false;
      }
      var hasTitle = Boolean((select(Selector$1.TITLE, tip) || {}).innerHTML);
      var hasContent = Boolean((select(Selector$1.CONTENT, tip) || {}).innerHTML);
      return hasTitle || hasContent;
    }
  }, {
    key: 'addAttachmentClass',
    value: function addAttachmentClass(attachment) {
      addClass(this.getTipElement(), CLASS_PREFIX + '-' + attachment);
    }
  }, {
    key: 'setContent',
    value: function setContent(tip) {
      // we use append for html objects to maintain js events/components
      this.setElementContent(select(Selector$1.TITLE, tip), this.getTitle());
      this.setElementContent(select(Selector$1.CONTENT, tip), this.getContent());

      removeClass(tip, ClassName.FADE);
      removeClass(tip, ClassName.SHOW);
    }

    // This method may look identical to ToolTip version, but it uses a different RegEx defined above

  }, {
    key: 'cleanTipClass',
    value: function cleanTipClass() {
      var tip = this.getTipElement();
      var tabClass = tip.className.match(BSCLS_PREFIX_REGEX);
      if (tabClass !== null && tabClass.length > 0) {
        tabClass.forEach(function (cls) {
          removeClass(tip, cls);
        });
      }
    }
  }, {
    key: 'getTitle',
    value: function getTitle() {
      var title = this.$config.title || '';
      if (typeof title === 'function') {
        title = title(this.$element);
      }
      if ((typeof title === 'undefined' ? 'undefined' : _typeof(title)) === 'object' && title.nodeType && !title.innerHTML.trim()) {
        // We have a dom node, but without inner content, so just return an empty string
        title = '';
      }
      if (typeof title === 'string') {
        title = title.trim();
      }
      if (!title) {
        // Try and grab element's title attribute
        title = getAttr(this.$element, 'title') || getAttr(this.$element, 'data-original-title') || '';
        title = title.trim();
      }
      return title;
    }

    // New methods

  }, {
    key: 'getContent',
    value: function getContent() {
      var content = this.$config.content || '';
      if (typeof content === 'function') {
        content = content(this.$element);
      }
      if ((typeof content === 'undefined' ? 'undefined' : _typeof(content)) === 'object' && content.nodeType && !content.innerHTML.trim()) {
        // We have a dom node, but without inner content, so just return an empty string
        content = '';
      }
      if (typeof content === 'string') {
        content = content.trim();
      }
      return content;
    }
  }], [{
    key: 'Default',

    // Getter overrides

    get: function get$$1() {
      return Defaults;
    }
  }, {
    key: 'NAME',
    get: function get$$1() {
      return NAME;
    }
  }]);
  return PopOver;
}(ToolTip);

var bPopover = {
  mixins: [toolpopMixin],
  render: function render(h) {
    return h('div', { class: ['d-none'], style: { display: 'none' }, attrs: { 'aria-hidden': true } }, [h('div', { ref: 'title' }, this.$slots.title), h('div', { ref: 'content' }, this.$slots.default)]);
  },
  data: function data() {
    return {};
  },

  props: {
    title: {
      type: String,
      default: ''
    },
    content: {
      type: String,
      default: ''
    },
    triggers: {
      type: [String, Array],
      default: 'click'
    },
    placement: {
      type: String,
      default: 'right'
    }
  },
  methods: {
    createToolpop: function createToolpop() {
      // getTarget is in toolpop mixin
      var target = this.getTarget();
      if (target) {
        this._toolpop = new PopOver(target, this.getConfig(), this.$root);
      } else {
        this._toolpop = null;
        warn("b-popover: 'target' element not found!");
      }
      return this._toolpop;
    }
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$32 = {
  bPopover: bPopover
};

var VuePlugin$63 = {
  install: function install(Vue) {
    registerComponents(Vue, components$32);
  }
};

vueUse(VuePlugin$63);

var bProgressBar = {
  render: function render(h) {
    var t = this;
    var childNodes = h(false);
    if (t.$slots.default) {
      childNodes = t.$slots.default;
    } else if (t.label) {
      childNodes = h('span', { domProps: { innerHTML: t.label } });
    } else if (t.computedShowProgress) {
      childNodes = t.progress.toFixed(t.computedPrecision);
    } else if (t.computedShowValue) {
      childNodes = t.value.toFixed(t.computedPrecision);
    }
    return h('div', {
      class: t.progressBarClasses,
      style: t.progressBarStyles,
      attrs: {
        role: 'progressbar',
        'aria-valuemin': '0',
        'aria-valuemax': t.computedMax.toString(),
        'aria-valuenow': t.value.toFixed(t.computedPrecision)
      }
    }, [childNodes]);
  },

  computed: {
    progressBarClasses: function progressBarClasses() {
      return ['progress-bar', this.computedVariant ? 'bg-' + this.computedVariant : '', this.computedStriped || this.computedAnimated ? 'progress-bar-striped' : '', this.computedAnimated ? 'progress-bar-animated' : ''];
    },
    progressBarStyles: function progressBarStyles() {
      return {
        width: 100 * (this.value / this.computedMax) + '%'
      };
    },
    progress: function progress() {
      var p = Math.pow(10, this.computedPrecision);
      return Math.round(100 * p * this.value / this.computedMax) / p;
    },
    computedMax: function computedMax() {
      // Prefer our max over parent setting
      return typeof this.max === 'number' ? this.max : this.$parent.max || 100;
    },
    computedVariant: function computedVariant() {
      // Prefer our variant over parent setting
      return this.variant || this.$parent.variant;
    },
    computedPrecision: function computedPrecision() {
      // Prefer our precision over parent setting
      return typeof this.precision === 'number' ? this.precision : this.$parent.precision || 0;
    },
    computedStriped: function computedStriped() {
      // Prefer our striped over parent setting
      return typeof this.striped === 'boolean' ? this.striped : this.$parent.striped || false;
    },
    computedAnimated: function computedAnimated() {
      // Prefer our animated over parent setting
      return typeof this.animated === 'boolean' ? this.animated : this.$parent.animated || false;
    },
    computedShowProgress: function computedShowProgress() {
      // Prefer our showProgress over parent setting
      return typeof this.showProgress === 'boolean' ? this.showProgress : this.$parent.showProgress || false;
    },
    computedShowValue: function computedShowValue() {
      // Prefer our showValue over parent setting
      return typeof this.showValue === 'boolean' ? this.showValue : this.$parent.showValue || false;
    }
  },
  props: {
    value: {
      type: Number,
      default: 0
    },
    label: {
      type: String,
      value: null
    },
    // $parent prop values take precedence over the following props
    // Which is why they are defaulted to null
    max: {
      type: Number,
      default: null
    },
    precision: {
      type: Number,
      default: null
    },
    variant: {
      type: String,
      default: null
    },
    striped: {
      type: Boolean,
      default: null
    },
    animated: {
      type: Boolean,
      default: null
    },
    showProgress: {
      type: Boolean,
      default: null
    },
    showValue: {
      type: Boolean,
      default: null
    }
  }
};

var bProgress$2 = {
  components: { bProgressBar: bProgressBar },
  render: function render(h) {
    var t = this;
    var childNodes = t.$slots.default;
    if (!childNodes) {
      childNodes = h('b-progress-bar', {
        props: {
          value: t.value,
          max: t.max,
          precision: t.precision,
          variant: t.variant,
          animated: t.animated,
          striped: t.striped,
          showProgress: t.showProgress,
          showValue: t.showValue
        }
      });
    }
    return h('div', { class: ['progress'], style: t.progressHeight }, [childNodes]);
  },

  props: {
    // These props can be inherited via the child b-progress-bar(s)
    variant: {
      type: String,
      default: null
    },
    striped: {
      type: Boolean,
      default: false
    },
    animated: {
      type: Boolean,
      default: false
    },
    height: {
      type: String,
      default: null
    },
    precision: {
      type: Number,
      default: 0
    },
    showProgress: {
      type: Boolean,
      default: false
    },
    showValue: {
      type: Boolean,
      default: false
    },
    max: {
      type: Number,
      default: 100
    },
    // This prop is not inherited by child b-progress-bar(s)
    value: {
      type: Number,
      default: 0
    }
  },
  computed: {
    progressHeight: function progressHeight() {
      return { height: this.height || null };
    }
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$33 = {
  bProgress: bProgress$2,
  bProgressBar: bProgressBar
};

var VuePlugin$65 = {
  install: function install(Vue) {
    registerComponents(Vue, components$33);
  }
};

vueUse(VuePlugin$65);

function toString(v) {
  if (!v) {
    return '';
  }
  if (v instanceof Object) {
    return keys(v).map(function (k) {
      return toString(v[k]);
    }).join(' ');
  }
  return String(v);
}

function recToString(obj) {
  if (!(obj instanceof Object)) {
    return '';
  }
  return toString(keys(obj).reduce(function (o, k) {
    // Ignore fields that start with _
    if (!/^_/.test(k)) {
      o[k] = obj[k];
    }
    return o;
  }, {}));
}

function defaultSortCompare(a, b, sortBy) {
  if (typeof a[sortBy] === 'number' && typeof b[sortBy] === 'number') {
    return a[sortBy] < b[sortBy] && -1 || a[sortBy] > b[sortBy] && 1 || 0;
  }
  return toString(a[sortBy]).localeCompare(toString(b[sortBy]), undefined, {
    numeric: true
  });
}

function processField(key, value) {
  var field = null;
  if (typeof value === 'string') {
    // Label shortcut
    field = { key: key, label: value };
  } else if (typeof value === 'function') {
    // Formatter shortcut
    field = { key: key, formatter: value };
  } else if ((typeof value === 'undefined' ? 'undefined' : _typeof(value)) === 'object') {
    field = assign({}, value);
    field.key = field.key || key;
  } else if (value !== false) {
    // Fallback to just key
    field = { key: key };
  }
  return field;
}

var bTable$2 = {
  mixins: [idMixin, listenOnRootMixin],
  render: function render(h) {
    var t = this;
    var $slots = t.$slots;
    var $scoped = t.$scopedSlots;
    var fields = t.computedFields;
    var items = t.computedItems;

    // Build the caption
    var caption = h(false);
    if (t.caption || $slots['table-caption']) {
      var data = { style: t.captionStyles };
      if (!$slots['table-caption']) {
        data.domProps = { innerHTML: t.caption };
      }
      caption = h('caption', data, $slots['table-caption']);
    }

    // Build the colgroup
    var colgroup = $slots['table-colgroup'] ? h('colgroup', {}, $slots['table-colgroup']) : h(false);

    // factory function for thead and tfoot cells (th's)
    var makeHeadCells = function makeHeadCells() {
      var isFoot = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : false;

      return fields.map(function (field, colIndex) {
        var data = {
          key: field.key,
          class: t.fieldClasses(field),
          style: field.thStyle || {},
          attrs: {
            'tabindex': field.sortable ? '0' : null,
            'abbr': field.headerAbbr || null,
            'title': field.headerTitle || null,
            'aria-colindex': String(colIndex + 1),
            'aria-label': field.sortable ? t.localSortDesc && t.localSortBy === field.key ? t.labelSortAsc : t.labelSortDesc : null,
            'aria-sort': field.sortable && t.localSortBy === field.key ? t.localSortDesc ? 'descending' : 'ascending' : null
          },
          on: {
            click: function click(evt) {
              evt.stopPropagation();
              evt.preventDefault();
              t.headClicked(evt, field);
            },
            keydown: function keydown(evt) {
              var keyCode = evt.keyCode;
              if (keyCode === KeyCodes.ENTER || keyCode === KeyCodes.SPACE) {
                evt.stopPropagation();
                evt.preventDefault();
                t.headClicked(evt, field);
              }
            }
          }
        };
        var slot = isFoot && $scoped['FOOT_' + field.key] ? $scoped['FOOT_' + field.key] : $scoped['HEAD_' + field.key];
        if (slot) {
          slot = [slot({ label: field.label, column: field.key, field: field })];
        } else {
          data.domProps = { innerHTML: field.label };
        }
        return h('th', data, slot);
      });
    };

    // Build the thead
    var thead = h(false);
    if (t.isStacked !== true) {
      // If in always stacked mode (t.isStacked === true), then we don't bother rendering the thead
      thead = h('thead', { class: t.headClasses }, [h('tr', { class: t.theadTrClass }, makeHeadCells(false))]);
    }

    // Build the tfoot
    var tfoot = h(false);
    if (t.footClone && t.isStacked !== true) {
      // If in always stacked mode (t.isStacked === true), then we don't bother rendering the tfoot
      tfoot = h('tfoot', { class: t.footClasses }, [h('tr', { class: t.tfootTrClass }, makeHeadCells(true))]);
    }

    // Prepare the tbody rows
    var rows = [];

    // Add static Top Row slot (hidden in visibly stacked mode as we can't control the data-label)
    // If in always stacked mode, we don't bother rendering the row
    if ($scoped['top-row'] && t.isStacked !== true) {
      rows.push(h('tr', { key: 'top-row', class: ['b-table-top-row', t.tbodyTrClass] }, [$scoped['top-row']({ columns: fields.length, fields: fields })]));
    } else {
      rows.push(h(false));
    }

    // Add the item data rows
    items.forEach(function (item, rowIndex) {
      var detailsSlot = $scoped['row-details'];
      var rowShowDetails = Boolean(item._showDetails && detailsSlot);
      var detailsId = rowShowDetails ? t.safeId('_details_' + rowIndex + '_') : null;
      var toggleDetailsFn = function toggleDetailsFn() {
        if (detailsSlot) {
          t.$set(item, '_showDetails', !item._showDetails);
        }
      };
      // For each item data field in row
      var tds = fields.map(function (field, colIndex) {
        var data = {
          key: 'row-' + rowIndex + '-cell-' + colIndex,
          class: t.tdClasses(field, item),
          attrs: field.tdAttr || {},
          domProps: {}
        };
        data.attrs['aria-colindex'] = String(colIndex + 1);
        var childNodes = void 0;
        if ($scoped[field.key]) {
          childNodes = [$scoped[field.key]({
            item: item,
            index: rowIndex,
            unformatted: item[field.key],
            value: t.getFormattedValue(item, field),
            toggleDetails: toggleDetailsFn,
            detailsShowing: Boolean(item._showDetails)
          })];
          if (t.isStacked) {
            // We wrap in a DIV to ensure rendered as a single cell when visually stacked!
            childNodes = [h('div', {}, [childNodes])];
          }
        } else {
          var formatted = t.getFormattedValue(item, field);
          if (t.isStacked) {
            // We innerHTML a DIV to ensure rendered as a single cell when visually stacked!
            childNodes = [h('div', { domProps: { innerHTML: formatted } })];
          } else {
            // Non stcaked, so we just innerHTML the td
            data.domProps['innerHTML'] = formatted;
          }
        }
        if (t.isStacked) {
          // Generate the "header cell" label content in stacked mode
          data.attrs['data-label'] = field.label;
          if (field.isRowHeader) {
            data.attrs['role'] = 'rowheader';
          } else {
            data.attrs['role'] = 'cell';
          }
        }
        // Render either a td or th cell
        return h(field.isRowHeader ? 'th' : 'td', data, childNodes);
      });
      // Calculate the row number in the dataset (indexed from 1)
      var ariaRowIndex = null;
      if (t.currentPage && t.perPage && t.perPage > 0) {
        ariaRowIndex = (t.currentPage - 1) * t.perPage + rowIndex + 1;
      }
      // Assemble and add the row
      rows.push(h('tr', {
        key: 'row-' + rowIndex,
        class: [t.rowClasses(item), { 'b-table-has-details': rowShowDetails }],
        attrs: {
          'aria-describedby': detailsId,
          'aria-rowindex': ariaRowIndex,
          role: t.isStacked ? 'row' : null
        },
        on: {
          click: function click(evt) {
            t.rowClicked(evt, item, rowIndex);
          },
          dblclick: function dblclick(evt) {
            t.rowDblClicked(evt, item, rowIndex);
          },
          mouseenter: function mouseenter(evt) {
            t.rowHovered(evt, item, rowIndex);
          }
        }
      }, tds));
      // Row Details slot
      if (rowShowDetails) {
        var tdAttrs = { colspan: String(fields.length) };
        var trAttrs = { id: detailsId };
        if (t.isStacked) {
          tdAttrs['role'] = 'cell';
          trAttrs['role'] = 'row';
        }
        var details = h('td', { attrs: tdAttrs }, [detailsSlot({ item: item, index: rowIndex, fields: fields, toggleDetails: toggleDetailsFn })]);
        rows.push(h('tr', { key: 'details-' + rowIndex, class: ['b-table-details', t.tbodyTrClass], attrs: trAttrs }, [details]));
      } else if (detailsSlot) {
        // Only add the placeholder if a the table has a row-details slot defined (but not shown)
        rows.push(h(false));
      }
    });

    // Empty Items / Empty Filtered Row slot
    if (t.showEmpty && (!items || items.length === 0)) {
      var empty = t.filter ? $slots['emptyfiltered'] : $slots['empty'];
      if (!empty) {
        empty = h('div', {
          class: ['text-center', 'my-2'],
          domProps: { innerHTML: t.filter ? t.emptyFilteredText : t.emptyText }
        });
      }
      empty = h('td', { attrs: { colspan: String(fields.length), role: t.isStacked ? 'cell' : null } }, [h('div', { attrs: { role: 'alert', 'aria-live': 'polite' } }, [empty])]);
      rows.push(h('tr', {
        key: 'empty-row',
        class: ['b-table-empty-row', t.tbodyTrClass],
        attrs: t.isStacked ? { role: 'row' } : {}
      }, [empty]));
    } else {
      rows.push(h(false));
    }

    // Static bottom row slot (hidden in visibly stacked mode as we can't control the data-label)
    // If in always stacked mode, we don't bother rendering the row
    if ($scoped['bottom-row'] && t.isStacked !== true) {
      rows.push(h('tr', { key: 'bottom-row', class: ['b-table-bottom-row', t.tbodyTrClass] }, [$scoped['bottom-row']({ columns: fields.length, fields: fields })]));
    } else {
      rows.push(h(false));
    }

    // Assemble the rows into the tbody
    var tbody = h('tbody', { class: t.bodyClasses, attrs: t.isStacked ? { role: 'rowgroup' } : {} }, rows);

    // Return the assembled table
    return h('table', {
      class: t.tableClasses,
      attrs: {
        id: t.safeId(),
        role: t.isStacked ? 'table' : null,
        'aria-busy': t.computedBusy ? 'true' : 'false',
        'aria-colcount': String(fields.length),
        'aria-rowcount': t.$attrs['aria-rowcount'] || t.perPage && t.perPage > 0 ? '-1' : null
      }
    }, [caption, colgroup, thead, tfoot, tbody]);
  },
  data: function data() {
    return {
      localSortBy: this.sortBy || '',
      localSortDesc: this.sortDesc || false,
      localItems: [],
      // Note: filteredItems only used to determine if # of items changed
      filteredItems: [],
      localBusy: false
    };
  },

  props: {
    items: {
      type: [Array, Function],
      default: function _default() {
        return [];
      }
    },
    fields: {
      type: [Object, Array],
      default: null
    },
    sortBy: {
      type: String,
      default: null
    },
    sortDesc: {
      type: Boolean,
      default: false
    },
    caption: {
      type: String,
      default: null
    },
    captionTop: {
      type: Boolean,
      default: false
    },
    striped: {
      type: Boolean,
      default: false
    },
    bordered: {
      type: Boolean,
      default: false
    },
    outlined: {
      type: Boolean,
      default: false
    },
    dark: {
      type: Boolean,
      default: function _default() {
        if (this && typeof this.inverse === 'boolean') {
          // Deprecate inverse
          warn("b-table: prop 'inverse' has been deprecated. Use 'dark' instead");
          return this.dark;
        }
        return false;
      }
    },
    inverse: {
      // Deprecated in v1.0.0 in favor of `dark`
      type: Boolean,
      default: null
    },
    hover: {
      type: Boolean,
      default: false
    },
    small: {
      type: Boolean,
      default: false
    },
    fixed: {
      type: Boolean,
      default: false
    },
    footClone: {
      type: Boolean,
      default: false
    },
    responsive: {
      type: [Boolean, String],
      default: false
    },
    stacked: {
      type: [Boolean, String],
      default: false
    },
    headVariant: {
      type: String,
      default: ''
    },
    footVariant: {
      type: String,
      default: ''
    },
    theadClass: {
      type: [String, Array],
      default: null
    },
    theadTrClass: {
      type: [String, Array],
      default: null
    },
    tbodyClass: {
      type: [String, Array],
      default: null
    },
    tbodyTrClass: {
      type: [String, Array],
      default: null
    },
    tfootClass: {
      type: [String, Array],
      default: null
    },
    tfootTrClass: {
      type: [String, Array],
      default: null
    },
    perPage: {
      type: Number,
      default: 0
    },
    currentPage: {
      type: Number,
      default: 1
    },
    filter: {
      type: [String, RegExp, Function],
      default: null
    },
    sortCompare: {
      type: Function,
      default: null
    },
    noLocalSorting: {
      type: Boolean,
      default: false
    },
    noProviderPaging: {
      type: Boolean,
      default: false
    },
    noProviderSorting: {
      type: Boolean,
      default: false
    },
    noProviderFiltering: {
      type: Boolean,
      default: false
    },
    busy: {
      type: Boolean,
      default: false
    },
    value: {
      type: Array,
      default: function _default() {
        return [];
      }
    },
    labelSortAsc: {
      type: String,
      default: 'Click to sort Ascending'
    },
    labelSortDesc: {
      type: String,
      default: 'Click to sort Descending'
    },
    showEmpty: {
      type: Boolean,
      default: false
    },
    emptyText: {
      type: String,
      default: 'There are no records to show'
    },
    emptyFilteredText: {
      type: String,
      default: 'There are no records matching your request'
    },
    apiUrl: {
      // Passthrough prop. Passed to the context object. Not used by b-table directly
      type: String,
      default: ''
    }
  },
  watch: {
    items: function items(newVal, oldVal) {
      if (oldVal !== newVal) {
        this._providerUpdate();
      }
    },
    context: function context(newVal, oldVal) {
      if (!looseEqual(newVal, oldVal)) {
        this.$emit('context-changed', newVal);
      }
    },
    filteredItems: function filteredItems(newVal, oldVal) {
      if (this.localFiltering && newVal.length !== oldVal.length) {
        // Emit a filtered notification event, as number of filtered items has changed
        this.$emit('filtered', newVal);
      }
    },
    sortDesc: function sortDesc(newVal, oldVal) {
      if (newVal === this.localSortDesc) {
        return;
      }
      this.localSortDesc = newVal || false;
    },
    localSortDesc: function localSortDesc(newVal, oldVal) {
      // Emit update to sort-desc.sync
      if (newVal !== oldVal) {
        this.$emit('update:sortDesc', newVal);
        if (!this.noProviderSorting) {
          this._providerUpdate();
        }
      }
    },
    sortBy: function sortBy(newVal, oldVal) {
      if (newVal === this.localSortBy) {
        return;
      }
      this.localSortBy = newVal || null;
    },
    localSortBy: function localSortBy(newVal, oldVal) {
      if (newVal !== oldVal) {
        this.$emit('update:sortBy', newVal);
        if (!this.noProviderSorting) {
          this._providerUpdate();
        }
      }
    },
    perPage: function perPage(newVal, oldVal) {
      if (oldVal !== newVal && !this.noProviderPaging) {
        this._providerUpdate();
      }
    },
    currentPage: function currentPage(newVal, oldVal) {
      if (oldVal !== newVal && !this.noProviderPaging) {
        this._providerUpdate();
      }
    },
    filter: function filter(newVal, oldVal) {
      if (oldVal !== newVal && !this.noProviderFiltering) {
        this._providerUpdate();
      }
    },
    localBusy: function localBusy(newVal, oldVal) {
      if (newVal !== oldVal) {
        this.$emit('update:busy', newVal);
      }
    }
  },
  mounted: function mounted() {
    var _this = this;

    this.localSortBy = this.sortBy;
    this.localSortDesc = this.sortDesc;
    if (this.hasProvider) {
      this._providerUpdate();
    }
    this.listenOnRoot('bv::refresh::table', function (id) {
      if (id === _this.id || id === _this) {
        _this._providerUpdate();
      }
    });
  },

  computed: {
    isStacked: function isStacked() {
      return this.stacked === '' ? true : this.stacked;
    },
    isResponsive: function isResponsive() {
      var responsive = this.responsive === '' ? true : this.responsive;
      return this.isStacked ? false : responsive;
    },
    tableClasses: function tableClasses() {
      return ['table', 'b-table', this.striped ? 'table-striped' : '', this.hover ? 'table-hover' : '', this.dark ? 'table-dark' : '', this.bordered ? 'table-bordered' : '', this.small ? 'table-sm' : '', this.outlined ? 'border' : '', this.fixed ? 'b-table-fixed' : '', this.isResponsive === true ? 'table-responsive' : this.isResponsive ? 'table-responsive-' + this.responsive : '', this.isStacked === true ? 'b-table-stacked' : this.isStacked ? 'b-table-stacked-' + this.stacked : ''];
    },
    headClasses: function headClasses() {
      return [this.headVariant ? 'thead-' + this.headVariant : '', this.theadClass];
    },
    bodyClasses: function bodyClasses() {
      return [this.tbodyClass];
    },
    footClasses: function footClasses() {
      var variant = this.footVariant || this.headVariant || null;
      return [variant ? 'thead-' + variant : '', this.tfootClass];
    },
    captionStyles: function captionStyles() {
      // Move caption to top
      return this.captionTop ? { captionSide: 'top' } : {};
    },
    hasProvider: function hasProvider() {
      return this.items instanceof Function;
    },
    localFiltering: function localFiltering() {
      return this.hasProvider ? this.noProviderFiltering : true;
    },
    localSorting: function localSorting() {
      return this.hasProvider ? this.noProviderSorting : !this.noLocalSorting;
    },
    localPaging: function localPaging() {
      return this.hasProvider ? this.noProviderPaging : true;
    },
    context: function context() {
      return {
        perPage: this.perPage,
        currentPage: this.currentPage,
        filter: this.filter,
        sortBy: this.localSortBy,
        sortDesc: this.localSortDesc,
        apiUrl: this.apiUrl
      };
    },
    computedFields: function computedFields() {
      var _this2 = this;

      // We normalize fields into an array of objects
      // [ { key:..., label:..., ...}, {...}, ..., {..}]
      var fields = [];
      if (isArray(this.fields)) {
        // Normalize array Form
        this.fields.filter(function (f) {
          return f;
        }).forEach(function (f) {
          if (typeof f === 'string') {
            fields.push({ key: f, label: startCase(f) });
          } else if ((typeof f === 'undefined' ? 'undefined' : _typeof(f)) === 'object' && f.key && typeof f.key === 'string') {
            // Full object definition. We use assign so that we don't mutate the original
            fields.push(assign({}, f));
          } else if ((typeof f === 'undefined' ? 'undefined' : _typeof(f)) === 'object' && keys(f).length === 1) {
            // Shortcut object (i.e. { 'foo_bar': 'This is Foo Bar' }
            var key = keys(f)[0];
            var field = processField(key, f[key]);
            if (field) {
              fields.push(field);
            }
          }
        });
      } else if (this.fields && _typeof(this.fields) === 'object' && keys(this.fields).length > 0) {
        // Normalize object Form
        keys(this.fields).forEach(function (key) {
          var field = processField(key, _this2.fields[key]);
          if (field) {
            fields.push(field);
          }
        });
      }
      // If no field provided, take a sample from first record (if exits)
      if (fields.length === 0 && this.computedItems.length > 0) {
        var sample = this.computedItems[0];
        keys(sample).forEach(function (k) {
          fields.push({ key: k, label: startCase(k) });
        });
      }
      // Ensure we have a unique array of fields and that they have labels
      var memo = {};
      return fields.filter(function (f) {
        if (!memo[f.key]) {
          memo[f.key] = true;
          f.label = f.label || startCase(f.key);
          return true;
        }
        return false;
      });
    },
    computedItems: function computedItems() {
      // Grab some props/data to ensure reactivity
      var perPage = this.perPage;
      var currentPage = this.currentPage;
      var filter = this.filter;
      var sortBy = this.localSortBy;
      var sortDesc = this.localSortDesc;
      var sortCompare = this.sortCompare;
      var localFiltering = this.localFiltering;
      var localSorting = this.localSorting;
      var localPaging = this.localPaging;
      var items = this.hasProvider ? this.localItems : this.items;
      if (!items) {
        this.$nextTick(this._providerUpdate);
        return [];
      }
      // Array copy for sorting, filtering, etc.
      items = items.slice();
      // Apply local filter
      if (filter && localFiltering) {
        if (filter instanceof Function) {
          items = items.filter(filter);
        } else {
          var regex = void 0;
          if (filter instanceof RegExp) {
            regex = filter;
          } else {
            regex = new RegExp('.*' + filter + '.*', 'ig');
          }
          items = items.filter(function (item) {
            var test = regex.test(recToString(item));
            regex.lastIndex = 0;
            return test;
          });
        }
      }
      if (localFiltering) {
        // Make a local copy of filtered items to trigger filtered event
        this.filteredItems = items.slice();
      }
      // Apply local Sort
      if (sortBy && localSorting) {
        items = stableSort(items, function sortItemsFn(a, b) {
          var ret = null;
          if (typeof sortCompare === 'function') {
            // Call user provided sortCompare routine
            ret = sortCompare(a, b, sortBy);
          }
          if (ret === null || ret === undefined) {
            // Fallback to defaultSortCompare if sortCompare not defined or returns null
            ret = defaultSortCompare(a, b, sortBy);
          }
          // Handle sorting direction
          return (ret || 0) * (sortDesc ? -1 : 1);
        });
      }
      // Apply local pagination
      if (Boolean(perPage) && localPaging) {
        // Grab the current page of data (which may be past filtered items)
        items = items.slice((currentPage - 1) * perPage, currentPage * perPage);
      }
      // Update the value model with the filtered/sorted/paginated data set
      this.$emit('input', items);
      return items;
    },
    computedBusy: function computedBusy() {
      return this.busy || this.localBusy;
    }
  },
  methods: {
    keys: keys,
    fieldClasses: function fieldClasses(field) {
      return [field.sortable ? 'sorting' : '', field.sortable && this.localSortBy === field.key ? 'sorting_' + (this.localSortDesc ? 'desc' : 'asc') : '', field.variant ? 'table-' + field.variant : '', field.class ? field.class : '', field.thClass ? field.thClass : ''];
    },
    tdClasses: function tdClasses(field, item) {
      var cellVariant = '';
      if (item._cellVariants && item._cellVariants[field.key]) {
        cellVariant = (this.dark ? 'bg' : 'table') + '-' + item._cellVariants[field.key];
      }
      return [field.variant && !cellVariant ? (this.dark ? 'bg' : 'table') + '-' + field.variant : '', cellVariant, field.class ? field.class : '', field.tdClass ? field.tdClass : ''];
    },
    rowClasses: function rowClasses(item) {
      return [item._rowVariant ? (this.dark ? 'bg' : 'table') + '-' + item._rowVariant : '', this.tbodyTrClass];
    },
    rowClicked: function rowClicked(e, item, index) {
      if (this.stopIfBusy(e)) {
        // If table is busy (via provider) then don't propagate
        return;
      }
      this.$emit('row-clicked', item, index, e);
    },
    rowDblClicked: function rowDblClicked(e, item, index) {
      if (this.stopIfBusy(e)) {
        // If table is busy (via provider) then don't propagate
        return;
      }
      this.$emit('row-dblclicked', item, index, e);
    },
    rowHovered: function rowHovered(e, item, index) {
      if (this.stopIfBusy(e)) {
        // If table is busy (via provider) then don't propagate
        return;
      }
      this.$emit('row-hovered', item, index, e);
    },
    headClicked: function headClicked(e, field) {
      if (this.stopIfBusy(e)) {
        // If table is busy (via provider) then don't propagate
        return;
      }
      var sortChanged = false;
      if (field.sortable) {
        if (field.key === this.localSortBy) {
          // Change sorting direction on current column
          this.localSortDesc = !this.localSortDesc;
        } else {
          // Start sorting this column ascending
          this.localSortBy = field.key;
          this.localSortDesc = false;
        }
        sortChanged = true;
      } else if (this.localSortBy) {
        this.localSortBy = null;
        this.localSortDesc = false;
        sortChanged = true;
      }
      this.$emit('head-clicked', field.key, field, e);
      if (sortChanged) {
        // Sorting parameters changed
        this.$emit('sort-changed', this.context);
      }
    },
    stopIfBusy: function stopIfBusy(evt) {
      if (this.computedBusy) {
        // If table is busy (via provider) then don't propagate
        evt.preventDefault();
        evt.stopPropagation();
        return true;
      }
      return false;
    },
    refresh: function refresh() {
      // Expose refresh method
      if (this.hasProvider) {
        this._providerUpdate();
      }
    },
    _providerSetLocal: function _providerSetLocal(items) {
      this.localItems = items && items.length > 0 ? items.slice() : [];
      this.localBusy = false;
      this.$emit('refreshed');
      // Deprecated root emit
      this.emitOnRoot('table::refreshed', this.id);
      // New root emit
      if (this.id) {
        this.emitOnRoot('bv::table::refreshed', this.id);
      }
    },
    _providerUpdate: function _providerUpdate() {
      var _this3 = this;

      // Refresh the provider items
      if (this.computedBusy || !this.hasProvider) {
        // Don't refresh remote data if we are 'busy' or if no provider
        return;
      }
      // Set internal busy state
      this.localBusy = true;
      // Call provider function with context and optional callback
      var data = this.items(this.context, this._providerSetLocal);
      if (data && data.then && typeof data.then === 'function') {
        // Provider returned Promise
        data.then(function (items) {
          _this3._providerSetLocal(items);
        });
      } else {
        // Provider returned Array data
        this._providerSetLocal(data);
      }
    },
    getFormattedValue: function getFormattedValue(item, field) {
      var key = field.key;
      var formatter = field.formatter;
      var parent = this.$parent;
      var value = item[key];
      if (formatter) {
        if (typeof formatter === 'function') {
          value = formatter(value, key, item);
        } else if (typeof formatter === 'string' && typeof parent[formatter] === 'function') {
          value = parent[formatter](value, key, item);
        }
      }
      return value;
    }
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$34 = {
  bTable: bTable$2
};

var VuePlugin$67 = {
  install: function install(Vue) {
    registerComponents(Vue, components$34);
  }
};

vueUse(VuePlugin$67);

// Helper component
var bTabButtonHelper = {
  name: 'bTabButtonHelper',
  props: {
    content: { type: String, default: '' },
    href: { type: String, default: '#' },
    posInSet: { type: Number, default: null },
    setSize: { type: Number, default: null },
    controls: { type: String, default: null },
    id: { type: String, default: null },
    active: { type: Boolean, default: false },
    disabled: { type: Boolean, default: false },
    linkClass: { default: null },
    itemClass: { default: null }
  },
  render: function render(h) {
    var t = this;
    var link = h('a', {
      class: ['nav-link', { active: t.active, disabled: t.disabled }, t.linkClass],
      attrs: {
        role: 'tab',
        tabindex: '-1',
        href: t.href,
        id: t.id,
        disabled: t.disabled,
        'aria-selected': t.active ? 'true' : 'false',
        'aria-setsize': t.setSize,
        'aria-posinset': t.posInSet,
        'aria-controls': t.controls
      },
      domProps: { innerHTML: t.content },
      on: {
        click: t.handleClick,
        keydown: t.handleClick
      }
    });
    return h('li', { class: ['nav-item', t.itemClass], attrs: { role: 'presentation' } }, [link]);
  },

  methods: {
    handleClick: function handleClick(evt) {
      function stop() {
        evt.preventDefault();
        evt.stopPropagation();
      }
      if (this.disabled) {
        stop();
        return;
      }
      if (evt.type === 'click' || evt.keyCode === KeyCodes.ENTER || evt.keyCode === KeyCodes.SPACE) {
        stop();
        this.$emit('click', evt);
      }
    }
  }
};

var bTabs = {
  mixins: [idMixin],
  render: function render(h) {
    var _ref;

    var t = this;
    var tabs = this.tabs;

    // Navigation 'buttons'
    var buttons = tabs.map(function (tab, index) {
      return h(bTabButtonHelper, {
        key: index,
        props: {
          content: tab.headHtml || tab.title,
          href: tab.href,
          id: tab.controlledBy || t.safeId('_BV_tab_' + (index + 1) + '_'),
          active: tab.localActive,
          disabled: tab.disabled,
          setSize: tabs.length,
          posInSet: index + 1,
          controls: t.safeId('_BV_tab_container_'),
          linkClass: tab.titleLinkClass,
          itemClass: tab.titleItemClass
        },
        on: { click: function click(evt) {
            t.setTab(index);
          } }
      });
    });

    // Nav 'button' wrapper
    var navs = h('ul', {
      class: ['nav', 'nav-' + t.navStyle, (_ref = {}, defineProperty$1(_ref, 'card-header-' + t.navStyle, t.card && !t.vertical), defineProperty$1(_ref, 'card-header', t.card && t.vertical), defineProperty$1(_ref, 'h-100', t.card && t.vertical), defineProperty$1(_ref, 'flex-column', t.vertical), defineProperty$1(_ref, 'border-bottom-0', t.vertical), defineProperty$1(_ref, 'rounded-0', t.vertical), defineProperty$1(_ref, 'small', t.small), _ref), t.navClass],
      attrs: { role: 'tablist', tabindex: '0', id: t.safeId('_BV_tab_controls_') },
      on: { keydown: t.onKeynav }
    }, [buttons, t.$slots.tabs]);
    navs = h('div', {
      class: [{
        'card-header': t.card && !t.vertical && !(t.end || t.bottom),
        'card-footer': t.card && !t.vertical && (t.end || t.bottom),
        'col-auto': t.vertical
      }, t.navWrapperClass]
    }, [navs]);

    var empty = void 0;
    if (tabs && tabs.length) {
      empty = h(false);
    } else {
      empty = h('div', { class: ['tab-pane', 'active', { 'card-body': t.card }] }, t.$slots.empty);
    }

    // Main content section
    var content = h('div', {
      ref: 'tabsContainer',
      class: ['tab-content', { 'col': t.vertical }, t.contentClass],
      attrs: { id: t.safeId('_BV_tab_container_') }
    }, [t.$slots.default, empty]);

    // Render final output
    return h(t.tag, {
      class: ['tabs', { 'row': t.vertical, 'no-gutters': t.vertical && t.card }],
      attrs: { id: t.safeId() }
    }, [t.end || t.bottom ? content : h(false), [navs], t.end || t.bottom ? h(false) : content]);
  },
  data: function data() {
    return {
      currentTab: this.value,
      tabs: []
    };
  },

  props: {
    tag: {
      type: String,
      default: 'div'
    },
    card: {
      type: Boolean,
      default: false
    },
    small: {
      type: Boolean,
      default: false
    },
    value: {
      type: Number,
      default: null
    },
    pills: {
      type: Boolean,
      default: false
    },
    vertical: {
      type: Boolean,
      default: false
    },
    bottom: {
      type: Boolean,
      default: false
    },
    end: {
      // Synonym for 'bottom'
      type: Boolean,
      default: false
    },
    noFade: {
      type: Boolean,
      default: false
    },
    lazy: {
      // This prop is sniffed by the tab child
      type: Boolean,
      default: false
    },
    contentClass: {
      type: [String, Array, Object],
      default: null
    },
    navClass: {
      type: [String, Array, Object],
      default: null
    },
    navWrapperClass: {
      type: [String, Array, Object],
      default: null
    }
  },
  watch: {
    currentTab: function currentTab(val, old) {
      if (val === old) {
        return;
      }
      this.$root.$emit('changed::tab', this, val, this.tabs[val]);
      this.$emit('input', val);
      this.tabs[val].$emit('click');
    },
    value: function value(val, old) {
      if (val === old) {
        return;
      }
      if (typeof old !== 'number') {
        old = 0;
      }
      // Moving left or right?
      var direction = val < old ? -1 : 1;
      this.setTab(val, false, direction);
    }
  },
  computed: {
    fade: function fade() {
      // This computed prop is sniffed by the tab child
      return !this.noFade;
    },
    navStyle: function navStyle() {
      return this.pills ? 'pills' : 'tabs';
    }
  },
  methods: {
    /**
         * Util: Return the sign of a number (as -1, 0, or 1)
         */
    sign: function sign(x) {
      return x === 0 ? 0 : x > 0 ? 1 : -1;
    },

    /*
         * handle keyboard navigation
         */
    onKeynav: function onKeynav(evt) {
      var key = evt.keyCode;
      var shift = evt.shiftKey;
      function stop() {
        evt.preventDefault();
        evt.stopPropagation();
      }
      if (key === KeyCodes.UP || key === KeyCodes.LEFT) {
        stop();
        if (shift) {
          this.setTab(0, false, 1);
        } else {
          this.previousTab();
        }
      } else if (key === KeyCodes.DWON || key === KeyCodes.RIGHT) {
        stop();
        if (shift) {
          this.setTab(this.tabs.length - 1, false, -1);
        } else {
          this.nextTab();
        }
      }
    },

    /**
         * Move to next tab
         */
    nextTab: function nextTab() {
      this.setTab(this.currentTab + 1, false, 1);
    },

    /**
         * Move to previous tab
         */
    previousTab: function previousTab() {
      this.setTab(this.currentTab - 1, false, -1);
    },

    /**
         * Set active tab on the tabs collection and the child 'tab' component
         * Index is the tab we want to activate. Direction is the direction we are moving
         * so if the tab we requested is disabled, we can skip over it.
         * Force is used by updateTabs to ensure we have cleared any previous active tabs.
         */
    setTab: function setTab(index, force, direction) {
      var _this = this;

      direction = this.sign(direction || 0);
      index = index || 0;
      // Prevent setting same tab and infinite loops!
      if (!force && index === this.currentTab) {
        return;
      }
      var tab = this.tabs[index];
      // Don't go beyond indexes!
      if (!tab) {
        // Reset the v-model to the current Tab
        this.$emit('input', this.currentTab);
        return;
      }
      // Ignore or Skip disabled
      if (tab.disabled) {
        if (direction) {
          // Skip to next non disabled tab in specified direction (recursive)
          this.setTab(index + direction, force, direction);
        }
        return;
      }
      // Activate requested current tab, and deactivte any old tabs
      this.tabs.forEach(function (t) {
        if (t === tab) {
          // Set new tab as active
          _this.$set(t, 'localActive', true);
        } else {
          // Ensure non current tabs are not active
          _this.$set(t, 'localActive', false);
        }
      });
      // Update currentTab
      this.currentTab = index;
    },

    /**
         * Dynamically update tabs list
         */
    updateTabs: function updateTabs() {
      // Probe tabs
      this.tabs = this.$children.filter(function (child) {
        return child._isTab;
      });
      // Set initial active tab
      var tabIndex = null;
      // Find *last* active non-dsabled tab in current tabs
      // We trust tab state over currentTab
      this.tabs.forEach(function (tab, index) {
        if (tab.localActive && !tab.disabled) {
          tabIndex = index;
        }
      });
      // Else try setting to currentTab
      if (tabIndex === null) {
        if (this.currentTab >= this.tabs.length) {
          // Handle last tab being removed
          this.setTab(this.tabs.length - 1, true, -1);
          return;
        } else if (this.tabs[this.currentTab] && !this.tabs[this.currentTab].disabled) {
          tabIndex = this.currentTab;
        }
      }
      // Else find *first* non-disabled tab in current tabs
      if (tabIndex === null) {
        this.tabs.forEach(function (tab, index) {
          if (!tab.disabled && tabIndex === null) {
            tabIndex = index;
          }
        });
      }
      this.setTab(tabIndex || 0, true, 0);
    }
  },
  mounted: function mounted() {
    this.updateTabs();
    // Observe Child changes so we can notify tabs change
    observeDOM(this.$refs.tabsContainer, this.updateTabs.bind(this), { subtree: false });
  }
};

var bTab = {
  mixins: [idMixin],
  render: function render(h) {
    var t = this;
    var content = h(false);
    if (t.localActive || !this.computedLazy) {
      content = h(t.tag, {
        ref: 'panel',
        class: t.tabClasses,
        directives: [{ name: 'show', value: t.localActive }],
        attrs: {
          role: 'tabpanel',
          id: t.safeId(),
          'aria-hidden': t.localActive ? 'false' : 'true',
          'aria-expanded': t.localActive ? 'true' : 'false',
          'aria-lablelledby': t.controlledBy || null
        }
      }, [t.$slots.default]);
    }
    return h('transition', {
      props: { mode: 'out-in' },
      on: {
        beforeEnter: t.beforeEnter,
        afterEnter: t.afterEnter,
        afterLeave: t.afterLeave
      }
    }, [content]);
  },

  methods: {
    beforeEnter: function beforeEnter() {
      this.show = false;
    },
    afterEnter: function afterEnter() {
      this.show = true;
    },
    afterLeave: function afterLeave() {
      this.show = false;
    }
  },
  data: function data() {
    return {
      localActive: this.active && !this.disabled,
      show: false
    };
  },
  mounted: function mounted() {
    this.show = this.localActive;
  },

  computed: {
    tabClasses: function tabClasses() {
      return ['tab-pane', this.$parent && this.$parent.card && !this.noBody ? 'card-body' : '', this.show ? 'show' : '', this.computedFade ? 'fade' : '', this.disabled ? 'disabled' : '', this.localActive ? 'active' : ''];
    },
    controlledBy: function controlledBy() {
      return this.buttonId || this.safeId('__BV_tab_button__');
    },
    computedFade: function computedFade() {
      return this.$parent.fade;
    },
    computedLazy: function computedLazy() {
      return this.$parent.lazy;
    },
    _isTab: function _isTab() {
      // For parent sniffing of child
      return true;
    }
  },
  props: {
    active: {
      type: Boolean,
      default: false
    },
    tag: {
      type: String,
      default: 'div'
    },
    buttonId: {
      type: String,
      default: ''
    },
    title: {
      type: String,
      default: ''
    },
    titleItemClass: {
      // Sniffed by tabs.vue and added to nav 'li.nav-item'
      type: [String, Array, Object],
      default: null
    },
    titleLinkClass: {
      // Sniffed by tabs.vue and added to nav 'a.nav-link'
      type: [String, Array, Object],
      default: null
    },
    headHtml: {
      // Is this actually ever used?
      type: String,
      default: null
    },
    disabled: {
      type: Boolean,
      default: false
    },
    noBody: {
      type: Boolean,
      default: false
    },
    href: {
      type: String,
      default: '#'
    }
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$35 = {
  bTabs: bTabs,
  bTab: bTab
};

var VuePlugin$69 = {
  install: function install(Vue) {
    registerComponents(Vue, components$35);
  }
};

vueUse(VuePlugin$69);

var bTooltip = {
  mixins: [toolpopMixin],
  render: function render(h) {
    return h('div', { class: ['d-none'], style: { display: 'none' }, attrs: { 'aria-hidden': true } }, [h('div', { ref: 'title' }, this.$slots.default)]);
  },
  data: function data() {
    return {};
  },

  props: {
    title: {
      type: String,
      default: ''
    },
    triggers: {
      type: [String, Array],
      default: 'hover focus'
    },
    placement: {
      type: String,
      default: 'top'
    }
  },
  methods: {
    createToolpop: function createToolpop() {
      // getTarget is in toolpop mixin
      var target = this.getTarget();
      if (target) {
        this._toolpop = new ToolTip(target, this.getConfig(), this.$root);
      } else {
        this._toolpop = null;
        warn("b-tooltip: 'target' element not found!");
      }
      return this._toolpop;
    }
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var components$36 = {
  bTooltip: bTooltip
};

var VuePlugin$71 = {
  install: function install(Vue) {
    registerComponents(Vue, components$36);
  }
};

vueUse(VuePlugin$71);



var components = Object.freeze({
	Alert: VuePlugin$2,
	Badge: VuePlugin$4,
	Breadcrumb: VuePlugin$6,
	Button: VuePlugin$8,
	ButtonToolbar: VuePlugin$12,
	ButtonGroup: VuePlugin$10,
	Card: VuePlugin$16,
	Carousel: VuePlugin$18,
	Collapse: VuePlugin$22,
	Dropdown: VuePlugin$24,
	Embed: VuePlugin$25,
	Form: VuePlugin$27,
	FormGroup: VuePlugin$29,
	FormInput: VuePlugin$35,
	FormTextarea: VuePlugin$37,
	FormFile: VuePlugin$39,
	FormCheckbox: VuePlugin$31,
	FormRadio: VuePlugin$33,
	FormSelect: VuePlugin$41,
	Image: VuePlugin$43,
	InputGroup: VuePlugin$14,
	Jumbotron: VuePlugin$45,
	Layout: VuePlugin$20,
	Link: VuePlugin$47,
	ListGroup: VuePlugin$49,
	Media: VuePlugin$51,
	Modal: VuePlugin$53,
	Nav: VuePlugin$56,
	Navbar: VuePlugin$57,
	Pagination: VuePlugin$59,
	PaginationNav: VuePlugin$61,
	Popover: VuePlugin$63,
	Progress: VuePlugin$65,
	Table: VuePlugin$67,
	Tabs: VuePlugin$69,
	Tooltip: VuePlugin$71
});

/*
 * ScrollSpy class definition
 */

/*
 * Constants / Defaults
 */

var NAME$2 = 'v-b-scrollspy';
var ACTIVATE_EVENT = 'bv::scrollspy::activate';

var Default = {
  element: 'body',
  offset: 10,
  method: 'auto',
  throttle: 75
};

var DefaultType = {
  element: '(string|element|component)',
  offset: 'number',
  method: 'string',
  throttle: 'number'
};

var ClassName$2 = {
  DROPDOWN_ITEM: 'dropdown-item',
  ACTIVE: 'active'
};

var Selector$3 = {
  ACTIVE: '.active',
  NAV_LIST_GROUP: '.nav, .list-group',
  NAV_LINKS: '.nav-link',
  NAV_ITEMS: '.nav-item',
  LIST_ITEMS: '.list-group-item',
  DROPDOWN: '.dropdown, .dropup',
  DROPDOWN_ITEMS: '.dropdown-item',
  DROPDOWN_TOGGLE: '.dropdown-toggle'
};

var OffsetMethod = {
  OFFSET: 'offset',
  POSITION: 'position'

  // HREFs must start with # but can be === '#', or start with '#/' or '#!' (which can be router links)
};var HREF_REGEX = /^#[^/!]+/;

// Transition Events
var TransitionEndEvents$2 = ['webkitTransitionEnd', 'transitionend', 'otransitionend', 'oTransitionEnd'];

/*
 * Utility Methods
 */

// Better var type detection
function toType(obj) {
  return {}.toString.call(obj).match(/\s([a-zA-Z]+)/)[1].toLowerCase();
}

// Check config properties for expected types
function typeCheckConfig(componentName, config, configTypes) {
  for (var property in configTypes) {
    if (Object.prototype.hasOwnProperty.call(configTypes, property)) {
      var expectedTypes = configTypes[property];
      var value = config[property];
      var valueType = value && isElement(value) ? 'element' : toType(value);
      // handle Vue instances
      valueType = value && value._isVue ? 'component' : valueType;

      if (!new RegExp(expectedTypes).test(valueType)) {
        warn(componentName + ': Option "' + property + '" provided type "' + valueType + '", but expected type "' + expectedTypes + '"');
      }
    }
  }
}

/*
 * ------------------------------------------------------------------------
 * Class Definition
 * ------------------------------------------------------------------------
 */

var ScrollSpy = function () {
  function ScrollSpy(element, config, $root) {
    classCallCheck(this, ScrollSpy);

    // The element we activate links in
    this.$el = element;
    this.$scroller = null;
    this.$selector = [Selector$3.NAV_LINKS, Selector$3.LIST_ITEMS, Selector$3.DROPDOWN_ITEMS].join(',');
    this.$offsets = [];
    this.$targets = [];
    this.$activeTarget = null;
    this.$scrollHeight = 0;
    this.$resizeTimeout = null;
    this.$obs_scroller = null;
    this.$obs_targets = null;
    this.$root = $root || null;
    this.$config = null;

    this.updateConfig(config);
  }

  createClass(ScrollSpy, [{
    key: 'updateConfig',
    value: function updateConfig(config, $root) {
      if (this.$scroller) {
        // Just in case out scroll element has changed
        this.unlisten();
        this.$scroller = null;
      }
      var cfg = assign({}, this.constructor.Default, config);
      if ($root) {
        this.$root = $root;
      }
      typeCheckConfig(this.constructor.Name, cfg, this.constructor.DefaultType);
      this.$config = cfg;

      if (this.$root) {
        var self = this;
        this.$root.$nextTick(function () {
          self.listen();
        });
      } else {
        this.listen();
      }
    }
  }, {
    key: 'dispose',
    value: function dispose() {
      this.unlisten();
      clearTimeout(this.$resizeTimeout);
      this.$resizeTimeout = null;
      this.$el = null;
      this.$config = null;
      this.$scroller = null;
      this.$selector = null;
      this.$offsets = null;
      this.$targets = null;
      this.$activeTarget = null;
      this.$scrollHeight = null;
    }
  }, {
    key: 'listen',
    value: function listen() {
      var _this = this;

      var scroller = this.getScroller();
      if (scroller && scroller.tagName !== 'BODY') {
        eventOn(scroller, 'scroll', this);
      }
      eventOn(window, 'scroll', this);
      eventOn(window, 'resize', this);
      eventOn(window, 'orientationchange', this);
      TransitionEndEvents$2.forEach(function (evtName) {
        eventOn(window, evtName, _this);
      });
      this.setObservers(true);
      // Scedule a refresh
      this.handleEvent('refresh');
    }
  }, {
    key: 'unlisten',
    value: function unlisten() {
      var _this2 = this;

      var scroller = this.getScroller();
      this.setObservers(false);
      if (scroller && scroller.tagName !== 'BODY') {
        eventOff(scroller, 'scroll', this);
      }
      eventOff(window, 'scroll', this);
      eventOff(window, 'resize', this);
      eventOff(window, 'orientationchange', this);
      TransitionEndEvents$2.forEach(function (evtName) {
        eventOff(window, evtName, _this2);
      });
    }
  }, {
    key: 'setObservers',
    value: function setObservers(on) {
      var _this3 = this;

      // We observe both the scroller for content changes, and the target links
      if (this.$obs_scroller) {
        this.$obs_scroller.disconnect();
        this.$obs_scroller = null;
      }
      if (this.$obs_targets) {
        this.$obs_targets.disconnect();
        this.$obs_targets = null;
      }
      if (on) {
        this.$obs_targets = observeDOM(this.$el, function () {
          _this3.handleEvent('mutation');
        }, {
          subtree: true,
          childList: true,
          attributes: true,
          attributeFilter: ['href']
        });
        this.$obs_scroller = observeDOM(this.getScroller(), function () {
          _this3.handleEvent('mutation');
        }, {
          subtree: true,
          childList: true,
          characterData: true,
          attributes: true,
          attributeFilter: ['id', 'style', 'class']
        });
      }
    }

    // general event handler

  }, {
    key: 'handleEvent',
    value: function handleEvent(evt) {
      var type = typeof evt === 'string' ? evt : evt.type;

      var self = this;
      function resizeThrottle() {
        if (!self.$resizeTimeout) {
          self.$resizeTimeout = setTimeout(function () {
            self.refresh();
            self.process();
            self.$resizeTimeout = null;
          }, self.$config.throttle);
        }
      }

      if (type === 'scroll') {
        if (!this.$obs_scroller) {
          // Just in case we are added to the DOM before the scroll target is
          // We re-instantiate our listeners, just in case
          this.listen();
        }
        this.process();
      } else if (/(resize|orientationchange|mutation|refresh)/.test(type)) {
        // Postpone these events by throttle time
        resizeThrottle();
      }
    }

    // Refresh the list of target links on the element we are applied to

  }, {
    key: 'refresh',
    value: function refresh() {
      var _this4 = this;

      var scroller = this.getScroller();
      if (!scroller) {
        return;
      }
      var autoMethod = scroller !== scroller.window ? OffsetMethod.POSITION : OffsetMethod.OFFSET;
      var method = this.$config.method === 'auto' ? autoMethod : this.$config.method;
      var methodFn = method === OffsetMethod.POSITION ? position : offset;
      var offsetBase = method === OffsetMethod.POSITION ? this.getScrollTop() : 0;

      this.$offsets = [];
      this.$targets = [];

      this.$scrollHeight = this.getScrollHeight();

      // Find all the unique link href's
      selectAll(this.$selector, this.$el).map(function (link) {
        return getAttr(link, 'href');
      }).filter(function (href) {
        return HREF_REGEX.test(href || '');
      }).map(function (href) {
        var el = select(href, scroller);
        if (isVisible(el)) {
          return {
            offset: parseInt(methodFn(el).top, 10) + offsetBase,
            target: href
          };
        }
        return null;
      }).filter(function (item) {
        return item;
      }).sort(function (a, b) {
        return a.offset - b.offset;
      }).reduce(function (memo, item) {
        // record only unique targets/offfsets
        if (!memo[item.target]) {
          _this4.$offsets.push(item.offset);
          _this4.$targets.push(item.target);
          memo[item.target] = true;
        }
        return memo;
      }, {});

      return this;
    }

    // Handle activating/clearing

  }, {
    key: 'process',
    value: function process() {
      var scrollTop = this.getScrollTop() + this.$config.offset;
      var scrollHeight = this.getScrollHeight();
      var maxScroll = this.$config.offset + scrollHeight - this.getOffsetHeight();

      if (this.$scrollHeight !== scrollHeight) {
        this.refresh();
      }

      if (scrollTop >= maxScroll) {
        var target = this.$targets[this.$targets.length - 1];
        if (this.$activeTarget !== target) {
          this.activate(target);
        }
        return;
      }

      if (this.$activeTarget && scrollTop < this.$offsets[0] && this.$offsets[0] > 0) {
        this.$activeTarget = null;
        this.clear();
        return;
      }

      for (var i = this.$offsets.length; i--;) {
        var isActiveTarget = this.$activeTarget !== this.$targets[i] && scrollTop >= this.$offsets[i] && (typeof this.$offsets[i + 1] === 'undefined' || scrollTop < this.$offsets[i + 1]);

        if (isActiveTarget) {
          this.activate(this.$targets[i]);
        }
      }
    }
  }, {
    key: 'getScroller',
    value: function getScroller() {
      if (this.$scroller) {
        return this.$scroller;
      }
      var scroller = this.$config.element;
      if (!scroller) {
        return null;
      } else if (isElement(scroller.$el)) {
        scroller = scroller.$el;
      } else if (typeof scroller === 'string') {
        scroller = select(scroller);
      }
      if (!scroller) {
        return null;
      }
      this.$scroller = scroller.tagName === 'BODY' ? window : scroller;
      return this.$scroller;
    }
  }, {
    key: 'getScrollTop',
    value: function getScrollTop() {
      var scroller = this.getScroller();
      return scroller === window ? scroller.pageYOffset : scroller.scrollTop;
    }
  }, {
    key: 'getScrollHeight',
    value: function getScrollHeight() {
      return this.getScroller().scrollHeight || Math.max(document.body.scrollHeight, document.documentElement.scrollHeight);
    }
  }, {
    key: 'getOffsetHeight',
    value: function getOffsetHeight() {
      var scroller = this.getScroller();
      return scroller === window ? window.innerHeight : getBCR(scroller).height;
    }
  }, {
    key: 'activate',
    value: function activate(target) {
      var _this5 = this;

      this.$activeTarget = target;
      this.clear();

      // Grab the list of target links (<a href="{$target}">)
      var links = selectAll(this.$selector.split(',').map(function (selector) {
        return selector + '[href="' + target + '"]';
      }).join(','), this.$el);

      links.forEach(function (link) {
        if (hasClass(link, ClassName$2.DROPDOWN_ITEM)) {
          // This is a dropdown item, so find the .dropdown-toggle and set it's state
          var dropdown = closest(Selector$3.DROPDOWN, link);
          if (dropdown) {
            _this5.setActiveState(select(Selector$3.DROPDOWN_TOGGLE, dropdown), true);
          }
          // Also set this link's state
          _this5.setActiveState(link, true);
        } else {
          // Set triggered link as active
          _this5.setActiveState(link, true);
          if (matches(link.parentElement, Selector$3.NAV_ITEMS)) {
            // Handle nav-link inside nav-item, and set nav-item active
            _this5.setActiveState(link.parentElement, true);
          }
          // Set triggered links parents as active
          // With both <ul> and <nav> markup a parent is the previous sibling of any nav ancestor
          var el = link;
          while (el) {
            el = closest(Selector$3.NAV_LIST_GROUP, el);
            var sibling = el ? el.previousElementSibling : null;
            if (matches(sibling, Selector$3.NAV_LINKS + ', ' + Selector$3.LIST_ITEMS)) {
              _this5.setActiveState(sibling, true);
            }
            // Handle special case where nav-link is inside a nav-item
            if (matches(sibling, Selector$3.NAV_ITEMS)) {
              _this5.setActiveState(select(Selector$3.NAV_LINKS, sibling), true);
              // Add active state to nav-item as well
              _this5.setActiveState(sibling, true);
            }
          }
        }
      });

      // Signal event to via $root, passing ID of activaed target and reference to array of links
      if (links && links.length > 0 && this.$root) {
        this.$root.$emit(ACTIVATE_EVENT, target, links);
      }
    }
  }, {
    key: 'clear',
    value: function clear() {
      var _this6 = this;

      selectAll(this.$selector + ', ' + Selector$3.NAV_ITEMS, this.$el).filter(function (el) {
        return hasClass(el, ClassName$2.ACTIVE);
      }).forEach(function (el) {
        return _this6.setActiveState(el, false);
      });
    }
  }, {
    key: 'setActiveState',
    value: function setActiveState(el, active) {
      if (!el) {
        return;
      }
      if (active) {
        addClass(el, ClassName$2.ACTIVE);
      } else {
        removeClass(el, ClassName$2.ACTIVE);
      }
    }
  }], [{
    key: 'Name',
    get: function get$$1() {
      return NAME$2;
    }
  }, {
    key: 'Default',
    get: function get$$1() {
      return Default;
    }
  }, {
    key: 'DefaultType',
    get: function get$$1() {
      return DefaultType;
    }
  }]);
  return ScrollSpy;
}();

/*
 * ScrollSpy directive v-b-scrollspy
 */

var inBrowser$1 = typeof window !== 'undefined';
var isServer = !inBrowser$1;

// Key we use to store our Instance
var BVSS = '__BV_ScrollSpy__';

// Generate config from bindings
function makeConfig(binding) {
  var config = {};

  // If Argument, assume element ID
  if (binding.arg) {
    // Element ID specified as arg. We must pre-pend #
    config.element = '#' + binding.arg;
  }

  // Process modifiers
  keys(binding.modifiers).forEach(function (mod) {
    if (/^\d+$/.test(mod)) {
      // Offest value
      config.offset = parseInt(mod, 10);
    } else if (/^(auto|position|offset)$/.test(mod)) {
      // Offset method
      config.method = mod;
    }
  });

  // Process value
  if (typeof binding.value === 'string') {
    // Value is a CSS ID or selector
    config.element = binding.value;
  } else if (typeof binding.value === 'number') {
    // Value is offset
    config.offset = Math.round(binding.value);
  } else if (_typeof(binding.value) === 'object') {
    // Value is config object
    // Filter the object based on our supported config options
    keys(binding.value).filter(function (k) {
      return Boolean(ScrollSpy.DefaultType[k]);
    }).forEach(function (k) {
      config[k] = binding.value[k];
    });
  }

  return config;
}

function addBVSS(el, binding, vnode) {
  if (isServer) {
    return;
  }
  var cfg = makeConfig(binding);
  if (!el[BVSS]) {
    el[BVSS] = new ScrollSpy(el, cfg, vnode.context.$root);
  } else {
    el[BVSS].updateConfig(cfg, vnode.context.$root);
  }
  return el[BVSS];
}

function removeBVSS(el) {
  if (el[BVSS]) {
    el[BVSS].dispose();
    el[BVSS] = null;
  }
}

/*
 * Export our directive
 */

var bScrollspy = {
  bind: function bind(el, binding, vnode) {
    addBVSS(el, binding, vnode);
  },
  inserted: function inserted(el, binding, vnode) {
    addBVSS(el, binding, vnode);
  },
  update: function update(el, binding, vnode) {
    addBVSS(el, binding, vnode);
  },
  componentUpdated: function componentUpdated(el, binding, vnode) {
    addBVSS(el, binding, vnode);
  },
  unbind: function unbind(el) {
    if (isServer) {
      return;
    }
    // Remove scroll event listener on scrollElId
    removeBVSS(el);
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var directives$3 = {
  bScrollspy: bScrollspy
};

var VuePlugin$73 = {
  install: function install(Vue) {
    registerDirectives(Vue, directives$3);
  }
};

vueUse(VuePlugin$73);

var inBrowser$2 = typeof window !== 'undefined' && typeof document !== 'undefined';

// Key which we use to store tooltip object on element
var BVTT = '__BV_ToolTip__';

// Valid event triggers
var validTriggers = {
  'focus': true,
  'hover': true,
  'click': true,
  'blur': true

  // Build a ToolTip config based on bindings (if any)
  // Arguments and modifiers take precedence over passed value config object
};function parseBindings(bindings) {
  // We start out with a blank config
  var config = {};

  // Process bindings.value
  if (typeof bindings.value === 'string') {
    // Value is tooltip content (html optionally supported)
    config.title = bindings.value;
  } else if (typeof bindings.value === 'function') {
    // Title generator function
    config.title = bindings.value;
  } else if (_typeof(bindings.value) === 'object') {
    // Value is config object, so merge
    config = assign(bindings.value);
  }

  // If Argument, assume element ID of container element
  if (bindings.arg) {
    // Element ID specified as arg. We must prepend '#' to become a CSS selector
    config.container = '#' + bindings.arg;
  }

  // Process modifiers
  keys(bindings.modifiers).forEach(function (mod) {
    if (/^html$/.test(mod)) {
      // Title allows HTML
      config.html = true;
    } else if (/^nofade$/.test(mod)) {
      // no animation
      config.animation = false;
    } else if (/^(auto|top(left|right)?|bottom(left|right)?|left(top|bottom)?|right(top|bottom)?)$/.test(mod)) {
      // placement of tooltip
      config.placement = mod;
    } else if (/^d\d+$/.test(mod)) {
      // delay value
      var delay = parseInt(mod.slice(1), 10) || 0;
      if (delay) {
        config.delay = delay;
      }
    } else if (/^o-?\d+$/.test(mod)) {
      // offset value. Negative allowed
      var offset = parseInt(mod.slice(1), 10) || 0;
      if (offset) {
        config.offset = offset;
      }
    }
  });

  // Special handling of event trigger modifiers Trigger is a space separated list
  var selectedTriggers = {};

  // parse current config object trigger
  var triggers = typeof config.trigger === 'string' ? config.trigger.trim().split(/\s+/) : [];
  triggers.forEach(function (trigger) {
    if (validTriggers[trigger]) {
      selectedTriggers[trigger] = true;
    }
  });

  // Parse Modifiers for triggers
  keys(validTriggers).forEach(function (trigger) {
    if (bindings.modifiers[trigger]) {
      selectedTriggers[trigger] = true;
    }
  });

  // Sanitize triggers
  config.trigger = keys(selectedTriggers).join(' ');
  if (config.trigger === 'blur') {
    // Blur by itself is useless, so convert it to 'focus'
    config.trigger = 'focus';
  }
  if (!config.trigger) {
    // remove trigger config
    delete config.trigger;
  }

  return config;
}

//
// Add or Update tooltip on our element
//
function applyBVTT(el, bindings, vnode) {
  if (!inBrowser$2) {
    return;
  }
  if (!Popper) {
    // Popper is required for tooltips to work
    warn('v-b-tooltip: Popper.js is required for tooltips to work');
    return;
  }
  if (el[BVTT]) {
    el[BVTT].updateConfig(parseBindings(bindings));
  } else {
    el[BVTT] = new ToolTip(el, parseBindings(bindings), vnode.context.$root);
  }
}

//
// Remove tooltip on our element
//
function removeBVTT(el) {
  if (!inBrowser$2) {
    return;
  }
  if (el[BVTT]) {
    el[BVTT].destroy();
    el[BVTT] = null;
    delete el[BVTT];
  }
}

/*
 * Export our directive
 */
var bTooltip$1 = {
  bind: function bind(el, bindings, vnode) {
    applyBVTT(el, bindings, vnode);
  },
  inserted: function inserted(el, bindings, vnode) {
    applyBVTT(el, bindings, vnode);
  },
  update: function update(el, bindings, vnode) {
    if (bindings.value !== bindings.oldValue) {
      applyBVTT(el, bindings, vnode);
    }
  },
  componentUpdated: function componentUpdated(el, bindings, vnode) {
    if (bindings.value !== bindings.oldValue) {
      applyBVTT(el, bindings, vnode);
    }
  },
  unbind: function unbind(el) {
    removeBVTT(el);
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var directives$4 = {
  bTooltip: bTooltip$1
};

var VuePlugin$75 = {
  install: function install(Vue) {
    registerDirectives(Vue, directives$4);
  }
};

vueUse(VuePlugin$75);

var inBrowser$3 = typeof window !== 'undefined' && typeof document !== 'undefined';

// Key which we use to store tooltip object on element
var BVPO = '__BV_PopOver__';

// Vlid event triggers
var validTriggers$1 = {
  'focus': true,
  'hover': true,
  'click': true,
  'blur': true

  // Build a PopOver config based on bindings (if any)
  // Arguments and modifiers take precedence over pased value config object
};function parseBindings$1(bindings) {
  // We start out with a blank config
  var config = {};

  // Process bindings.value
  if (typeof bindings.value === 'string') {
    // Value is popover content (html optionally supported)
    config.content = bindings.value;
  } else if (typeof bindings.value === 'function') {
    // Content generator function
    config.content = bindings.value;
  } else if (_typeof(bindings.value) === 'object') {
    // Value is config object, so merge
    config = assign(bindings.value);
  }

  // If Argument, assume element ID of container element
  if (bindings.arg) {
    // Element ID specified as arg. We must prepend '#' to become a CSS selector
    config.container = '#' + bindings.arg;
  }

  // Process modifiers
  keys(bindings.modifiers).forEach(function (mod) {
    if (/^html$/.test(mod)) {
      // Title allows HTML
      config.html = true;
    } else if (/^nofade$/.test(mod)) {
      // no animation
      config.animation = false;
    } else if (/^(auto|top(left|right)?|bottom(left|right)?|left(top|bottom)?|right(top|bottom)?)$/.test(mod)) {
      // placement of popover
      config.placement = mod;
    } else if (/^d\d+$/.test(mod)) {
      // delay value
      var delay = parseInt(mod.slice(1), 10) || 0;
      if (delay) {
        config.delay = delay;
      }
    } else if (/^o-?\d+$/.test(mod)) {
      // offset value (negative allowed)
      var offset = parseInt(mod.slice(1), 10) || 0;
      if (offset) {
        config.offset = offset;
      }
    }
  });

  // Special handling of event trigger modifiers Trigger is a space separated list
  var selectedTriggers = {};

  // parse current config object trigger
  var triggers = typeof config.trigger === 'string' ? config.trigger.trim().split(/\s+/) : [];
  triggers.forEach(function (trigger) {
    if (validTriggers$1[trigger]) {
      selectedTriggers[trigger] = true;
    }
  });

  // Parse Modifiers for triggers
  keys(validTriggers$1).forEach(function (trigger) {
    if (bindings.modifiers[trigger]) {
      selectedTriggers[trigger] = true;
    }
  });

  // Sanitize triggers
  config.trigger = keys(selectedTriggers).join(' ');
  if (config.trigger === 'blur') {
    // Blur by itself is useless, so convert it to focus
    config.trigger = 'focus';
  }
  if (!config.trigger) {
    // remove trigger config
    delete config.trigger;
  }

  return config;
}

//
// Add or Update popover on our element
//
function applyBVPO(el, bindings, vnode) {
  if (!inBrowser$3) {
    return;
  }
  if (!Popper) {
    // Popper is required for tooltips to work
    warn('v-b-popover: Popper.js is required for popovers to work');
    return;
  }
  if (el[BVPO]) {
    el[BVPO].updateConfig(parseBindings$1(bindings));
  } else {
    el[BVPO] = new PopOver(el, parseBindings$1(bindings), vnode.context.$root);
  }
}

//
// Remove popover on our element
//
function removeBVPO(el) {
  if (!inBrowser$3) {
    return;
  }
  if (el[BVPO]) {
    el[BVPO].destroy();
    el[BVPO] = null;
    delete el[BVPO];
  }
}

/*
 * Export our directive
 */
var bPopover$1 = {
  bind: function bind(el, bindings, vnode) {
    applyBVPO(el, bindings, vnode);
  },
  inserted: function inserted(el, bindings, vnode) {
    applyBVPO(el, bindings, vnode);
  },
  update: function update(el, bindings, vnode) {
    if (bindings.value !== bindings.oldValue) {
      applyBVPO(el, bindings, vnode);
    }
  },
  componentUpdated: function componentUpdated(el, bindings, vnode) {
    if (bindings.value !== bindings.oldValue) {
      applyBVPO(el, bindings, vnode);
    }
  },
  unbind: function unbind(el) {
    removeBVPO(el);
  }
};

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var directives$5 = {
  bPopover: bPopover$1
};

var VuePlugin$77 = {
  install: function install(Vue) {
    registerDirectives(Vue, directives$5);
  }
};

vueUse(VuePlugin$77);



var directives$2 = Object.freeze({
	Toggle: VuePlugin$23,
	Modal: VuePlugin$55,
	Scrollspy: VuePlugin$73,
	Tooltip: VuePlugin$75,
	Popover: VuePlugin$77
});

/* eslint-disable no-var, no-undef, guard-for-in, object-shorthand */

var VuePlugin = {
  install: function install(Vue) {
    if (Vue._bootstrap_vue_installed) {
      return;
    }

    Vue._bootstrap_vue_installed = true;

    // Register component plugins
    for (var plugin in components) {
      Vue.use(components[plugin]);
    }

    // Register directive plugins
    for (var _plugin in directives$2) {
      Vue.use(directives$2[_plugin]);
    }
  }
};

vueUse(VuePlugin);

export default VuePlugin;
//# sourceMappingURL=bootstrap-vue.esm.js.map
