/**
 * marked - a markdown parser
 * Copyright (c) 2011-2014, Christopher Jeffrey. (MIT Licensed)
 * https://github.com/chjj/marked
 */

;(function() {

/**
 * Block-Level Grammar
 */

var block = {
  newline: /^\n+/,
  code: /^( {4}[^\n]+\n*)+/,
  fences: noop,
  hr: /^( *[-*_]){3,} *(?:\n+|$)/,
  heading: /^ *(#{1,6}) *([^\n]+?) *#* *(?:\n+|$)/,
  nptable: noop,
  lheading: /^([^\n]+)\n *(=|-){2,} *(?:\n+|$)/,
  blockquote: /^( *>[^\n]+(\n(?!def)[^\n]+)*\n*)+/,
  list: /^( *)(bull) [\s\S]+?(?:hr|def|\n{2,}(?! )(?!\1bull )\n*|\s*$)/,
  html: /^ *(?:comment *(?:\n|\s*$)|closed *(?:\n{2,}|\s*$)|closing *(?:\n{2,}|\s*$))/,
  def: /^ *\[([^\]]+)\]: *<?([^\s>]+)>?(?: +["(]([^\n]+)[")])? *(?:\n+|$)/,
  table: noop,
  paragraph: /^((?:[^\n]+\n?(?!hr|heading|lheading|blockquote|tag|def))+)\n*/,
  text: /^[^\n]+/
};

block.bullet = /(?:[*+-]|\d+\.)/;
block.item = /^( *)(bull) [^\n]*(?:\n(?!\1bull )[^\n]*)*/;
block.item = replace(block.item, 'gm')
  (/bull/g, block.bullet)
  ();

block.list = replace(block.list)
  (/bull/g, block.bullet)
  ('hr', '\\n+(?=\\1?(?:[-*_] *){3,}(?:\\n+|$))')
  ('def', '\\n+(?=' + block.def.source + ')')
  ();

block.blockquote = replace(block.blockquote)
  ('def', block.def)
  ();

block._tag = '(?!(?:'
  + 'a|em|strong|small|s|cite|q|dfn|abbr|data|time|code'
  + '|var|samp|kbd|sub|sup|i|b|u|mark|ruby|rt|rp|bdi|bdo'
  + '|span|br|wbr|ins|del|img)\\b)\\w+(?!:/|[^\\w\\s@]*@)\\b';

block.html = replace(block.html)
  ('comment', /<!--[\s\S]*?-->/)
  ('closed', /<(tag)[\s\S]+?<\/\1>/)
  ('closing', /<tag(?:"[^"]*"|'[^']*'|[^'">])*?>/)
  (/tag/g, block._tag)
  ();

block.paragraph = replace(block.paragraph)
  ('hr', block.hr)
  ('heading', block.heading)
  ('lheading', block.lheading)
  ('blockquote', block.blockquote)
  ('tag', '<' + block._tag)
  ('def', block.def)
  ();

/**
 * Normal Block Grammar
 */

block.normal = merge({}, block);

/**
 * GFM Block Grammar
 */

block.gfm = merge({}, block.normal, {
  fences: /^ *(`{3,}|~{3,})[ \.]*(\S+)? *\n([\s\S]*?)\s*\1 *(?:\n+|$)/,
  paragraph: /^/,
  heading: /^ *(#{1,6}) +([^\n]+?) *#* *(?:\n+|$)/
});

block.gfm.paragraph = replace(block.paragraph)
  ('(?!', '(?!'
    + block.gfm.fences.source.replace('\\1', '\\2') + '|'
    + block.list.source.replace('\\1', '\\3') + '|')
  ();

/**
 * GFM + Tables Block Grammar
 */

block.tables = merge({}, block.gfm, {
  nptable: /^ *(\S.*\|.*)\n *([-:]+ *\|[-| :]*)\n((?:.*\|.*(?:\n|$))*)\n*/,
  table: /^ *\|(.+)\n *\|( *[-:]+[-| :]*)\n((?: *\|.*(?:\n|$))*)\n*/
});

/**
 * Block Lexer
 */

function Lexer(options) {
  this.tokens = [];
  this.tokens.links = {};
  this.options = options || marked.defaults;
  this.rules = block.normal;

  if (this.options.gfm) {
    if (this.options.tables) {
      this.rules = block.tables;
    } else {
      this.rules = block.gfm;
    }
  }
}

/**
 * Expose Block Rules
 */

Lexer.rules = block;

/**
 * Static Lex Method
 */

Lexer.lex = function(src, options) {
  var lexer = new Lexer(options);
  return lexer.lex(src);
};

/**
 * Preprocessing
 */

Lexer.prototype.lex = function(src) {
  src = src
    .replace(/\r\n|\r/g, '\n')
    .replace(/\t/g, '    ')
    .replace(/\u00a0/g, ' ')
    .replace(/\u2424/g, '\n');

  return this.token(src, true);
};

/**
 * Lexing
 */

Lexer.prototype.token = function(src, top, bq) {
  var src = src.replace(/^ +$/gm, '')
    , next
    , loose
    , cap
    , bull
    , b
    , item
    , space
    , i
    , l;

  while (src) {
    // newline
    if (cap = this.rules.newline.exec(src)) {
      src = src.substring(cap[0].length);
      if (cap[0].length > 1) {
        this.tokens.push({
          type: 'space'
        });
      }
    }

    // code
    if (cap = this.rules.code.exec(src)) {
      src = src.substring(cap[0].length);
      cap = cap[0].replace(/^ {4}/gm, '');
      this.tokens.push({
        type: 'code',
        text: !this.options.pedantic
          ? cap.replace(/\n+$/, '')
          : cap
      });
      continue;
    }

    // fences (gfm)
    if (cap = this.rules.fences.exec(src)) {
      src = src.substring(cap[0].length);
      this.tokens.push({
        type: 'code',
        lang: cap[2],
        text: cap[3] || ''
      });
      continue;
    }

    // heading
    if (cap = this.rules.heading.exec(src)) {
      src = src.substring(cap[0].length);
      this.tokens.push({
        type: 'heading',
        depth: cap[1].length,
        text: cap[2]
      });
      continue;
    }

    // table no leading pipe (gfm)
    if (top && (cap = this.rules.nptable.exec(src))) {
      src = src.substring(cap[0].length);

      item = {
        type: 'table',
        header: cap[1].replace(/^ *| *\| *$/g, '').split(/ *\| */),
        align: cap[2].replace(/^ *|\| *$/g, '').split(/ *\| */),
        cells: cap[3].replace(/\n$/, '').split('\n')
      };

      for (i = 0; i < item.align.length; i++) {
        if (/^ *-+: *$/.test(item.align[i])) {
          item.align[i] = 'right';
        } else if (/^ *:-+: *$/.test(item.align[i])) {
          item.align[i] = 'center';
        } else if (/^ *:-+ *$/.test(item.align[i])) {
          item.align[i] = 'left';
        } else {
          item.align[i] = null;
        }
      }

      for (i = 0; i < item.cells.length; i++) {
        item.cells[i] = item.cells[i].split(/ *\| */);
      }

      this.tokens.push(item);

      continue;
    }

    // lheading
    if (cap = this.rules.lheading.exec(src)) {
      src = src.substring(cap[0].length);
      this.tokens.push({
        type: 'heading',
        depth: cap[2] === '=' ? 1 : 2,
        text: cap[1]
      });
      continue;
    }

    // hr
    if (cap = this.rules.hr.exec(src)) {
      src = src.substring(cap[0].length);
      this.tokens.push({
        type: 'hr'
      });
      continue;
    }

    // blockquote
    if (cap = this.rules.blockquote.exec(src)) {
      src = src.substring(cap[0].length);

      this.tokens.push({
        type: 'blockquote_start'
      });

      cap = cap[0].replace(/^ *> ?/gm, '');

      // Pass `top` to keep the current
      // "toplevel" state. This is exactly
      // how markdown.pl works.
      this.token(cap, top, true);

      this.tokens.push({
        type: 'blockquote_end'
      });

      continue;
    }

    // list
    if (cap = this.rules.list.exec(src)) {
      src = src.substring(cap[0].length);
      bull = cap[2];

      this.tokens.push({
        type: 'list_start',
        ordered: bull.length > 1
      });

      // Get each top-level item.
      cap = cap[0].match(this.rules.item);

      next = false;
      l = cap.length;
      i = 0;

      for (; i < l; i++) {
        item = cap[i];

        // Remove the list item's bullet
        // so it is seen as the next token.
        space = item.length;
        item = item.replace(/^ *([*+-]|\d+\.) +/, '');

        // Outdent whatever the
        // list item contains. Hacky.
        if (~item.indexOf('\n ')) {
          space -= item.length;
          item = !this.options.pedantic
            ? item.replace(new RegExp('^ {1,' + space + '}', 'gm'), '')
            : item.replace(/^ {1,4}/gm, '');
        }

        // Determine whether the next list item belongs here.
        // Backpedal if it does not belong in this list.
        if (this.options.smartLists && i !== l - 1) {
          b = block.bullet.exec(cap[i + 1])[0];
          if (bull !== b && !(bull.length > 1 && b.length > 1)) {
            src = cap.slice(i + 1).join('\n') + src;
            i = l - 1;
          }
        }

        // Determine whether item is loose or not.
        // Use: /(^|\n)(?! )[^\n]+\n\n(?!\s*$)/
        // for discount behavior.
        loose = next || /\n\n(?!\s*$)/.test(item);
        if (i !== l - 1) {
          next = item.charAt(item.length - 1) === '\n';
          if (!loose) loose = next;
        }

        this.tokens.push({
          type: loose
            ? 'loose_item_start'
            : 'list_item_start'
        });

        // Recurse.
        this.token(item, false, bq);

        this.tokens.push({
          type: 'list_item_end'
        });
      }

      this.tokens.push({
        type: 'list_end'
      });

      continue;
    }

    // html
    if (cap = this.rules.html.exec(src)) {
      src = src.substring(cap[0].length);
      this.tokens.push({
        type: this.options.sanitize
          ? 'paragraph'
          : 'html',
        pre: !this.options.sanitizer
          && (cap[1] === 'pre' || cap[1] === 'script' || cap[1] === 'style'),
        text: cap[0]
      });
      continue;
    }

    // def
    if ((!bq && top) && (cap = this.rules.def.exec(src))) {
      src = src.substring(cap[0].length);
      this.tokens.links[cap[1].toLowerCase()] = {
        href: cap[2],
        title: cap[3]
      };
      continue;
    }

    // table (gfm)
    if (top && (cap = this.rules.table.exec(src))) {
      src = src.substring(cap[0].length);

      item = {
        type: 'table',
        header: cap[1].replace(/^ *| *\| *$/g, '').split(/ *\| */),
        align: cap[2].replace(/^ *|\| *$/g, '').split(/ *\| */),
        cells: cap[3].replace(/(?: *\| *)?\n$/, '').split('\n')
      };

      for (i = 0; i < item.align.length; i++) {
        if (/^ *-+: *$/.test(item.align[i])) {
          item.align[i] = 'right';
        } else if (/^ *:-+: *$/.test(item.align[i])) {
          item.align[i] = 'center';
        } else if (/^ *:-+ *$/.test(item.align[i])) {
          item.align[i] = 'left';
        } else {
          item.align[i] = null;
        }
      }

      for (i = 0; i < item.cells.length; i++) {
        item.cells[i] = item.cells[i]
          .replace(/^ *\| *| *\| *$/g, '')
          .split(/ *\| */);
      }

      this.tokens.push(item);

      continue;
    }

    // top-level paragraph
    if (top && (cap = this.rules.paragraph.exec(src))) {
      src = src.substring(cap[0].length);
      this.tokens.push({
        type: 'paragraph',
        text: cap[1].charAt(cap[1].length - 1) === '\n'
          ? cap[1].slice(0, -1)
          : cap[1]
      });
      continue;
    }

    // text
    if (cap = this.rules.text.exec(src)) {
      // Top-level should never reach here.
      src = src.substring(cap[0].length);
      this.tokens.push({
        type: 'text',
        text: cap[0]
      });
      continue;
    }

    if (src) {
      throw new
        Error('Infinite loop on byte: ' + src.charCodeAt(0));
    }
  }

  return this.tokens;
};

/**
 * Inline-Level Grammar
 */

var inline = {
  escape: /^\\([\\`*{}\[\]()#+\-.!_>])/,
  autolink: /^<([^ >]+(@|:\/)[^ >]+)>/,
  url: noop,
  tag: /^<!--[\s\S]*?-->|^<\/?\w+(?:"[^"]*"|'[^']*'|[^'">])*?>/,
  link: /^!?\[(inside)\]\(href\)/,
  reflink: /^!?\[(inside)\]\s*\[([^\]]*)\]/,
  nolink: /^!?\[((?:\[[^\]]*\]|[^\[\]])*)\]/,
  strong: /^__([\s\S]+?)__(?!_)|^\*\*([\s\S]+?)\*\*(?!\*)/,
  em: /^\b_((?:[^_]|__)+?)_\b|^\*((?:\*\*|[\s\S])+?)\*(?!\*)/,
  code: /^(`+)\s*([\s\S]*?[^`])\s*\1(?!`)/,
  br: /^ {2,}\n(?!\s*$)/,
  del: noop,
  text: /^[\s\S]+?(?=[\\<!\[_*`]| {2,}\n|$)/
};

inline._inside = /(?:\[[^\]]*\]|[^\[\]]|\](?=[^\[]*\]))*/;
inline._href = /\s*<?([\s\S]*?)>?(?:\s+['"]([\s\S]*?)['"])?\s*/;

inline.link = replace(inline.link)
  ('inside', inline._inside)
  ('href', inline._href)
  ();

inline.reflink = replace(inline.reflink)
  ('inside', inline._inside)
  ();

/**
 * Normal Inline Grammar
 */

inline.normal = merge({}, inline);

/**
 * Pedantic Inline Grammar
 */

inline.pedantic = merge({}, inline.normal, {
  strong: /^__(?=\S)([\s\S]*?\S)__(?!_)|^\*\*(?=\S)([\s\S]*?\S)\*\*(?!\*)/,
  em: /^_(?=\S)([\s\S]*?\S)_(?!_)|^\*(?=\S)([\s\S]*?\S)\*(?!\*)/
});

/**
 * GFM Inline Grammar
 */

inline.gfm = merge({}, inline.normal, {
  escape: replace(inline.escape)('])', '~|])')(),
  url: /^(https?:\/\/[^\s<]+[^<.,:;"')\]\s])/,
  del: /^~~(?=\S)([\s\S]*?\S)~~/,
  text: replace(inline.text)
    (']|', '~]|')
    ('|', '|https?://|')
    ()
});

/**
 * GFM + Line Breaks Inline Grammar
 */

inline.breaks = merge({}, inline.gfm, {
  br: replace(inline.br)('{2,}', '*')(),
  text: replace(inline.gfm.text)('{2,}', '*')()
});

/**
 * Inline Lexer & Compiler
 */

function InlineLexer(links, options) {
  this.options = options || marked.defaults;
  this.links = links;
  this.rules = inline.normal;
  this.renderer = this.options.renderer || new Renderer;
  this.renderer.options = this.options;

  if (!this.links) {
    throw new
      Error('Tokens array requires a `links` property.');
  }

  if (this.options.gfm) {
    if (this.options.breaks) {
      this.rules = inline.breaks;
    } else {
      this.rules = inline.gfm;
    }
  } else if (this.options.pedantic) {
    this.rules = inline.pedantic;
  }
}

/**
 * Expose Inline Rules
 */

InlineLexer.rules = inline;

/**
 * Static Lexing/Compiling Method
 */

InlineLexer.output = function(src, links, options) {
  var inline = new InlineLexer(links, options);
  return inline.output(src);
};

/**
 * Lexing/Compiling
 */

InlineLexer.prototype.output = function(src) {
  var out = ''
    , link
    , text
    , href
    , cap;

  while (src) {
    // escape
    if (cap = this.rules.escape.exec(src)) {
      src = src.substring(cap[0].length);
      out += cap[1];
      continue;
    }

    // autolink
    if (cap = this.rules.autolink.exec(src)) {
      src = src.substring(cap[0].length);
      if (cap[2] === '@') {
        text = cap[1].charAt(6) === ':'
          ? this.mangle(cap[1].substring(7))
          : this.mangle(cap[1]);
        href = this.mangle('mailto:') + text;
      } else {
        text = escape(cap[1]);
        href = text;
      }
      out += this.renderer.link(href, null, text);
      continue;
    }

    // url (gfm)
    if (!this.inLink && (cap = this.rules.url.exec(src))) {
      src = src.substring(cap[0].length);
      text = escape(cap[1]);
      href = text;
      out += this.renderer.link(href, null, text);
      continue;
    }

    // tag
    if (cap = this.rules.tag.exec(src)) {
      if (!this.inLink && /^<a /i.test(cap[0])) {
        this.inLink = true;
      } else if (this.inLink && /^<\/a>/i.test(cap[0])) {
        this.inLink = false;
      }
      src = src.substring(cap[0].length);
      out += this.options.sanitize
        ? this.options.sanitizer
          ? this.options.sanitizer(cap[0])
          : escape(cap[0])
        : cap[0]
      continue;
    }

    // link
    if (cap = this.rules.link.exec(src)) {
      src = src.substring(cap[0].length);
      this.inLink = true;
      out += this.outputLink(cap, {
        href: cap[2],
        title: cap[3]
      });
      this.inLink = false;
      continue;
    }

    // reflink, nolink
    if ((cap = this.rules.reflink.exec(src))
        || (cap = this.rules.nolink.exec(src))) {
      src = src.substring(cap[0].length);
      link = (cap[2] || cap[1]).replace(/\s+/g, ' ');
      link = this.links[link.toLowerCase()];
      if (!link || !link.href) {
        out += cap[0].charAt(0);
        src = cap[0].substring(1) + src;
        continue;
      }
      this.inLink = true;
      out += this.outputLink(cap, link);
      this.inLink = false;
      continue;
    }

    // strong
    if (cap = this.rules.strong.exec(src)) {
      src = src.substring(cap[0].length);
      out += this.renderer.strong(this.output(cap[2] || cap[1]));
      continue;
    }

    // em
    if (cap = this.rules.em.exec(src)) {
      src = src.substring(cap[0].length);
      out += this.renderer.em(this.output(cap[2] || cap[1]));
      continue;
    }

    // code
    if (cap = this.rules.code.exec(src)) {
      src = src.substring(cap[0].length);
      out += this.renderer.codespan(escape(cap[2], true));
      continue;
    }

    // br
    if (cap = this.rules.br.exec(src)) {
      src = src.substring(cap[0].length);
      out += this.renderer.br();
      continue;
    }

    // del (gfm)
    if (cap = this.rules.del.exec(src)) {
      src = src.substring(cap[0].length);
      out += this.renderer.del(this.output(cap[1]));
      continue;
    }

    // text
    if (cap = this.rules.text.exec(src)) {
      src = src.substring(cap[0].length);
      out += this.renderer.text(escape(this.smartypants(cap[0])));
      continue;
    }

    if (src) {
      throw new
        Error('Infinite loop on byte: ' + src.charCodeAt(0));
    }
  }

  return out;
};

/**
 * Compile Link
 */

InlineLexer.prototype.outputLink = function(cap, link) {
  var href = escape(link.href)
    , title = link.title ? escape(link.title) : null;

  return cap[0].charAt(0) !== '!'
    ? this.renderer.link(href, title, this.output(cap[1]))
    : this.renderer.image(href, title, escape(cap[1]));
};

/**
 * Smartypants Transformations
 */

InlineLexer.prototype.smartypants = function(text) {
  if (!this.options.smartypants) return text;
  return text
    // em-dashes
    .replace(/---/g, '\u2014')
    // en-dashes
    .replace(/--/g, '\u2013')
    // opening singles
    .replace(/(^|[-\u2014/(\[{"\s])'/g, '$1\u2018')
    // closing singles & apostrophes
    .replace(/'/g, '\u2019')
    // opening doubles
    .replace(/(^|[-\u2014/(\[{\u2018\s])"/g, '$1\u201c')
    // closing doubles
    .replace(/"/g, '\u201d')
    // ellipses
    .replace(/\.{3}/g, '\u2026');
};

/**
 * Mangle Links
 */

InlineLexer.prototype.mangle = function(text) {
  if (!this.options.mangle) return text;
  var out = ''
    , l = text.length
    , i = 0
    , ch;

  for (; i < l; i++) {
    ch = text.charCodeAt(i);
    if (Math.random() > 0.5) {
      ch = 'x' + ch.toString(16);
    }
    out += '&#' + ch + ';';
  }

  return out;
};

/**
 * Renderer
 */

function Renderer(options) {
  this.options = options || {};
}

Renderer.prototype.code = function(code, lang, escaped) {
  if (this.options.highlight) {
    var out = this.options.highlight(code, lang);
    if (out != null && out !== code) {
      escaped = true;
      code = out;
    }
  }

  if (!lang) {
    return '<pre><code>'
      + (escaped ? code : escape(code, true))
      + '\n</code></pre>';
  }

  return '<pre><code class="'
    + this.options.langPrefix
    + escape(lang, true)
    + '">'
    + (escaped ? code : escape(code, true))
    + '\n</code></pre>\n';
};

Renderer.prototype.blockquote = function(quote) {
  return '<blockquote>\n' + quote + '</blockquote>\n';
};

Renderer.prototype.html = function(html) {
  return html;
};

Renderer.prototype.heading = function(text, level, raw) {
  return '<h'
    + level
    + ' id="'
    + this.options.headerPrefix
    + raw.toLowerCase().replace(/[^\w]+/g, '-')
    + '">'
    + text
    + '</h'
    + level
    + '>\n';
};

Renderer.prototype.hr = function() {
  return this.options.xhtml ? '<hr/>\n' : '<hr>\n';
};

Renderer.prototype.list = function(body, ordered) {
  var type = ordered ? 'ol' : 'ul';
  return '<' + type + '>\n' + body + '</' + type + '>\n';
};

Renderer.prototype.listitem = function(text) {
  return '<li>' + text + '</li>\n';
};

Renderer.prototype.paragraph = function(text) {
  return '<p>' + text + '</p>\n';
};

Renderer.prototype.table = function(header, body) {
  return '<table>\n'
    + '<thead>\n'
    + header
    + '</thead>\n'
    + '<tbody>\n'
    + body
    + '</tbody>\n'
    + '</table>\n';
};

Renderer.prototype.tablerow = function(content) {
  return '<tr>\n' + content + '</tr>\n';
};

Renderer.prototype.tablecell = function(content, flags) {
  var type = flags.header ? 'th' : 'td';
  var tag = flags.align
    ? '<' + type + ' style="text-align:' + flags.align + '">'
    : '<' + type + '>';
  return tag + content + '</' + type + '>\n';
};

// span level renderer
Renderer.prototype.strong = function(text) {
  return '<strong>' + text + '</strong>';
};

Renderer.prototype.em = function(text) {
  return '<em>' + text + '</em>';
};

Renderer.prototype.codespan = function(text) {
  return '<code>' + text + '</code>';
};

Renderer.prototype.br = function() {
  return this.options.xhtml ? '<br/>' : '<br>';
};

Renderer.prototype.del = function(text) {
  return '<del>' + text + '</del>';
};

Renderer.prototype.link = function(href, title, text) {
  if (this.options.sanitize) {
    try {
      var prot = decodeURIComponent(unescape(href))
        .replace(/[^\w:]/g, '')
        .toLowerCase();
    } catch (e) {
      return '';
    }
    if (prot.indexOf('javascript:') === 0 || prot.indexOf('vbscript:') === 0) {
      return '';
    }
  }
  var out = '<a href="' + href + '"';
  if (title) {
    out += ' title="' + title + '"';
  }
  out += '>' + text + '</a>';
  return out;
};

Renderer.prototype.image = function(href, title, text) {
  var out = '<img src="' + href + '" alt="' + text + '"';
  if (title) {
    out += ' title="' + title + '"';
  }
  out += this.options.xhtml ? '/>' : '>';
  return out;
};

Renderer.prototype.text = function(text) {
  return text;
};

/**
 * Parsing & Compiling
 */

function Parser(options) {
  this.tokens = [];
  this.token = null;
  this.options = options || marked.defaults;
  this.options.renderer = this.options.renderer || new Renderer;
  this.renderer = this.options.renderer;
  this.renderer.options = this.options;
}

/**
 * Static Parse Method
 */

Parser.parse = function(src, options, renderer) {
  var parser = new Parser(options, renderer);
  return parser.parse(src);
};

/**
 * Parse Loop
 */

Parser.prototype.parse = function(src) {
  this.inline = new InlineLexer(src.links, this.options, this.renderer);
  this.tokens = src.reverse();

  var out = '';
  while (this.next()) {
    out += this.tok();
  }

  return out;
};

/**
 * Next Token
 */

Parser.prototype.next = function() {
  return this.token = this.tokens.pop();
};

/**
 * Preview Next Token
 */

Parser.prototype.peek = function() {
  return this.tokens[this.tokens.length - 1] || 0;
};

/**
 * Parse Text Tokens
 */

Parser.prototype.parseText = function() {
  var body = this.token.text;

  while (this.peek().type === 'text') {
    body += '\n' + this.next().text;
  }

  return this.inline.output(body);
};

/**
 * Parse Current Token
 */

Parser.prototype.tok = function() {
  switch (this.token.type) {
    case 'space': {
      return '';
    }
    case 'hr': {
      return this.renderer.hr();
    }
    case 'heading': {
      return this.renderer.heading(
        this.inline.output(this.token.text),
        this.token.depth,
        this.token.text);
    }
    case 'code': {
      return this.renderer.code(this.token.text,
        this.token.lang,
        this.token.escaped);
    }
    case 'table': {
      var header = ''
        , body = ''
        , i
        , row
        , cell
        , flags
        , j;

      // header
      cell = '';
      for (i = 0; i < this.token.header.length; i++) {
        flags = { header: true, align: this.token.align[i] };
        cell += this.renderer.tablecell(
          this.inline.output(this.token.header[i]),
          { header: true, align: this.token.align[i] }
        );
      }
      header += this.renderer.tablerow(cell);

      for (i = 0; i < this.token.cells.length; i++) {
        row = this.token.cells[i];

        cell = '';
        for (j = 0; j < row.length; j++) {
          cell += this.renderer.tablecell(
            this.inline.output(row[j]),
            { header: false, align: this.token.align[j] }
          );
        }

        body += this.renderer.tablerow(cell);
      }
      return this.renderer.table(header, body);
    }
    case 'blockquote_start': {
      var body = '';

      while (this.next().type !== 'blockquote_end') {
        body += this.tok();
      }

      return this.renderer.blockquote(body);
    }
    case 'list_start': {
      var body = ''
        , ordered = this.token.ordered;

      while (this.next().type !== 'list_end') {
        body += this.tok();
      }

      return this.renderer.list(body, ordered);
    }
    case 'list_item_start': {
      var body = '';

      while (this.next().type !== 'list_item_end') {
        body += this.token.type === 'text'
          ? this.parseText()
          : this.tok();
      }

      return this.renderer.listitem(body);
    }
    case 'loose_item_start': {
      var body = '';

      while (this.next().type !== 'list_item_end') {
        body += this.tok();
      }

      return this.renderer.listitem(body);
    }
    case 'html': {
      var html = !this.token.pre && !this.options.pedantic
        ? this.inline.output(this.token.text)
        : this.token.text;
      return this.renderer.html(html);
    }
    case 'paragraph': {
      return this.renderer.paragraph(this.inline.output(this.token.text));
    }
    case 'text': {
      return this.renderer.paragraph(this.parseText());
    }
  }
};

/**
 * Helpers
 */

function escape(html, encode) {
  return html
    .replace(!encode ? /&(?!#?\w+;)/g : /&/g, '&amp;')
    .replace(/</g, '&lt;')
    .replace(/>/g, '&gt;')
    .replace(/"/g, '&quot;')
    .replace(/'/g, '&#39;');
}

function unescape(html) {
	// explicitly match decimal, hex, and named HTML entities 
  return html.replace(/&(#(?:\d+)|(?:#x[0-9A-Fa-f]+)|(?:\w+));?/g, function(_, n) {
    n = n.toLowerCase();
    if (n === 'colon') return ':';
    if (n.charAt(0) === '#') {
      return n.charAt(1) === 'x'
        ? String.fromCharCode(parseInt(n.substring(2), 16))
        : String.fromCharCode(+n.substring(1));
    }
    return '';
  });
}

function replace(regex, opt) {
  regex = regex.source;
  opt = opt || '';
  return function self(name, val) {
    if (!name) return new RegExp(regex, opt);
    val = val.source || val;
    val = val.replace(/(^|[^\[])\^/g, '$1');
    regex = regex.replace(name, val);
    return self;
  };
}

function noop() {}
noop.exec = noop;

function merge(obj) {
  var i = 1
    , target
    , key;

  for (; i < arguments.length; i++) {
    target = arguments[i];
    for (key in target) {
      if (Object.prototype.hasOwnProperty.call(target, key)) {
        obj[key] = target[key];
      }
    }
  }

  return obj;
}


/**
 * Marked
 */

function marked(src, opt, callback) {
  if (callback || typeof opt === 'function') {
    if (!callback) {
      callback = opt;
      opt = null;
    }

    opt = merge({}, marked.defaults, opt || {});

    var highlight = opt.highlight
      , tokens
      , pending
      , i = 0;

    try {
      tokens = Lexer.lex(src, opt)
    } catch (e) {
      return callback(e);
    }

    pending = tokens.length;

    var done = function(err) {
      if (err) {
        opt.highlight = highlight;
        return callback(err);
      }

      var out;

      try {
        out = Parser.parse(tokens, opt);
      } catch (e) {
        err = e;
      }

      opt.highlight = highlight;

      return err
        ? callback(err)
        : callback(null, out);
    };

    if (!highlight || highlight.length < 3) {
      return done();
    }

    delete opt.highlight;

    if (!pending) return done();

    for (; i < tokens.length; i++) {
      (function(token) {
        if (token.type !== 'code') {
          return --pending || done();
        }
        return highlight(token.text, token.lang, function(err, code) {
          if (err) return done(err);
          if (code == null || code === token.text) {
            return --pending || done();
          }
          token.text = code;
          token.escaped = true;
          --pending || done();
        });
      })(tokens[i]);
    }

    return;
  }
  try {
    if (opt) opt = merge({}, marked.defaults, opt);
    return Parser.parse(Lexer.lex(src, opt), opt);
  } catch (e) {
    e.message += '\nPlease report this to https://github.com/chjj/marked.';
    if ((opt || marked.defaults).silent) {
      return '<p>An error occured:</p><pre>'
        + escape(e.message + '', true)
        + '</pre>';
    }
    throw e;
  }
}

/**
 * Options
 */

marked.options =
marked.setOptions = function(opt) {
  merge(marked.defaults, opt);
  return marked;
};

marked.defaults = {
  gfm: true,
  tables: true,
  breaks: false,
  pedantic: false,
  sanitize: false,
  sanitizer: null,
  mangle: true,
  smartLists: false,
  silent: false,
  highlight: null,
  langPrefix: 'lang-',
  smartypants: false,
  headerPrefix: '',
  renderer: new Renderer,
  xhtml: false
};

/**
 * Expose
 */

marked.Parser = Parser;
marked.parser = Parser.parse;

marked.Renderer = Renderer;

marked.Lexer = Lexer;
marked.lexer = Lexer.lex;

marked.InlineLexer = InlineLexer;
marked.inlineLexer = InlineLexer.output;

marked.parse = marked;

if (typeof module !== 'undefined' && typeof exports === 'object') {
  module.exports = marked;
} else if (typeof define === 'function' && define.amd) {
  define(function() { return marked; });
} else {
  this.marked = marked;
}

}).call(function() {
  return this || (typeof window !== 'undefined' ? window : global);
}());

!function(e,t){"object"==typeof exports&&"object"==typeof module?module.exports=t(require("typescript"),function(){try{return require("RAML.JsonValidation")}catch(e){}}(),function(){try{return require("RAML.XmlValidation")}catch(e){}}()):"function"==typeof define&&define.amd?define(["typescript","RAML.JsonValidation","RAML.XmlValidation"],t):"object"==typeof exports?exports.Parser=t(require("typescript"),function(){try{return require("RAML.JsonValidation")}catch(e){}}(),function(){try{return require("RAML.XmlValidation")}catch(e){}}()):(e.RAML=e.RAML||{},e.RAML.Parser=t(e.typescript,e["RAML.JsonValidation"],e["RAML.XmlValidation"]))}(this,function(__WEBPACK_EXTERNAL_MODULE_160__,__WEBPACK_EXTERNAL_MODULE_176__,__WEBPACK_EXTERNAL_MODULE_177__){return function(e){function t(r){if(n[r])return n[r].exports;var i=n[r]={exports:{},id:r,loaded:!1};return e[r].call(i.exports,i,i.exports,t),i.loaded=!0,i.exports}var n={};return t.m=e,t.c=n,t.p="",t(0)}([function(e,t,n){"use strict";function r(e,t){return m.load(e,t)}function i(e,t){return m.loadSync(e,t)}function a(e,t,n){return m.loadApi(e,t,n).getOrElse(null)}function o(e,t,n){return m.loadApi(e,t,n).getOrElse(null)}function s(e,t,n){return{fsResolver:{content:function(r){return r===(n||y.resolve("/","#local.raml")).replace(/\\/,"/")?e:t&&t.fsResolver?t.fsResolver.content(r):void 0},contentAsync:function(r){return r===(n||y.resolve("/","#local.raml")).replace(/\\/,"/")?Promise.resolve(e):t&&t.fsResolver?t.fsResolver.contentAsync(r):void 0}},httpResolver:t?t.httpResolver:null,rejectOnErrors:t?t.rejectOnErrors:!1,attributeDefaults:t?t.attributeDefaults:!0}}function u(e,t){var n=null;return t&&t.filePath&&(n=t.filePath),m.loadApi(n||"/#local.raml",[],s(e,t,n)).getOrElse(null)}function l(e,t){var n=null;return t&&t.filePath&&(n=t.filePath),m.loadApiAsync(n||"/#local.raml",[],s(e,t,n))}function p(e,t,n){return m.loadApiAsync(e,t,n)}function c(e,t,n){return m.loadRAMLAsync(e,t,n)}function f(e){return m.getLanguageElementByRuntimeType(e)}function d(e){return t.api10.isFragment(e)}function h(e){return t.api10.asFragment(e)}var m=n(8),y=n(15),v=n(23);t.api10=n(11),t.api08=n(12),t.load=r,t.loadSync=i,t.loadApiSync=a,t.loadRAMLSync=o,t.parseRAMLSync=u,t.parseRAML=l,t.loadApi=p,t.loadRAML=c,t.getLanguageElementByRuntimeType=f,t.isFragment=d,t.asFragment=h,t.hl=n(9),t.ll=n(10),t.search=n(13),t.stubs=n(1),t.utils=n(2),t.project=n(3),t.universeHelpers=n(14),t.ds=n(39),t.schema=n(4),t.universes=t.ds.universesInfo,t.parser=n(5),t.expander=n(6),t.wrapperHelper=n(7),"undefined"==typeof Promise&&"undefined"!=typeof window&&(window.Promise=v)},function(e,t,n){"use strict";function r(e,t,n,r){return void 0===n&&(n=null),S.createStubNode(e,t,n,r)}function i(e,t,n){return S.createStub(e,t,n)}function a(e,t,n){return S.createStub0(e,t,n)}function o(e,t){return S.createResourceStub(e,t)}function s(e,t){return S.createMethodStub(e,t)}function u(e,t){return S.createResponseStub(e,t)}function l(e,t){return S.createBodyStub(e,t)}function p(e,t){return S.createUriParameterStub(e,t)}function c(e,t){return S.createQueryParameterStub(e,t)}function f(e,t,n,r,i){return void 0===i&&(i=!1),new E.ASTPropImpl(e,t,n,r,i)}function d(e,t,n,r){return new E.ASTNodeImpl(e,t,n,r)}function h(e,t,n,r){return new b(e,t,n,r)}function m(e,t,n,r){return new _(e,t,n,r)}function y(e,t){return T.createMapping(e,t)}function v(){return T.createMap([])}function g(e,t){return S.createAttr(e,t)}var A=this&&this.__extends||function(e,t){function n(){this.constructor=e}for(var r in t)t.hasOwnProperty(r)&&(e[r]=t[r]);e.prototype=null===t?Object.create(t):(n.prototype=t.prototype,new n)},E=n(16),T=n(24),S=n(17);t.createStubNode=r,t.createStub=i,t.createStubNoParentPatch=a,t.createResourceStub=o,t.createMethodStub=s,t.createResponseStub=u,t.createBodyStub=l,t.createUriParameterStub=p,t.createQueryParameterStub=c,t.createASTPropImpl=f,t.createASTNodeImpl=d,t.createVirtualASTPropImpl=h,t.createVirtualNodeImpl=m;var b=function(e){function t(){e.apply(this,arguments)}return A(t,e),t.prototype.value=function(){return""},t}(E.ASTPropImpl),_=function(e){function t(){e.apply(this,arguments)}return A(t,e),t.prototype.value=function(){return""},t}(E.ASTNodeImpl);t.createMapping=y,t.createMap=v,t.createAttr=g},function(e,t,n){"use strict";function r(){return l.hasAsyncRequests()}function i(e){l.addLoadCallback(e)}function a(){return d.getTransformNames()}function o(e){return p.getFragmentDefenitionName(e)}function s(e,t,n){if(c.ReferenceType.isInstance(n.range())){var r=(n.range(),f.createNode(e));return new p.StructuredValue(r,t,n)}return e}function u(e){return(new h.UrlParameterNameValidator).parseUrl(e)}var l=n(25),p=n(16),c=n(39),f=n(24),d=n(27),h=n(28),m=n(29),y=n(18);t.hasAsyncRequests=r,t.addLoadCallback=i,t.getTransformerNames=a,t.updateType=function(e){var t=m.doDescrimination(e);null==t&&e.property()&&(t=e.property().range()),t&&e.patchType(t)},t.getFragmentDefenitionName=o,t.genStructuredValue=s,t.parseUrl=u;var v=function(){function e(e,t){this.node=e,this.targetUnitRoot=t}return e}();t.UnitLink=v;var g=function(){function e(e,t){this.errors=e,this.primaryUnitRoot=t}return e.prototype.accept=function(e){this.transformIssue(e),this.errors.push(e)},e.prototype.transformIssue=function(e){var t=this,n=null,r=this.findIssueTail(e);r.node&&(n=r.node.lowLevel().unit());var i=this.primaryUnitRoot.lowLevel().unit();if(n&&i&&n!=i){var a=this.findPathToNodeUnit(this.primaryUnitRoot,r.node);if(a&&a.length>0){var o=a.map(function(n){return t.convertConnectingNodeToError(n,e)});if(o&&o.length>0)for(var s=r,u=o.length-1;u>=0;u--){var l=o[u];s.extras=[],s.extras.push(l),s=l}}}},e.prototype.begin=function(){},e.prototype.end=function(){},e.prototype.acceptUnique=function(e){for(var t=0,n=this.errors;t<n.length;t++){var r=n[t];if(r.start==e.start&&r.message==e.message)return}this.accept(e)},e.prototype.findPathToNodeUnit=function(e,t){return t.lowLevel()&&t.lowLevel().unit()?this.findPathToNodeUnitRecursively(e,t.lowLevel().unit()):[]},e.prototype.findPathToNodeUnitRecursively=function(e,t){for(var n=this.findUnitLinks(e),r=0;r<n.length;r++){var i=n[r];if(i.targetUnitRoot&&i.targetUnitRoot.lowLevel()&&i.targetUnitRoot.lowLevel().unit()==t)return[i.node];var a=this.findPathToNodeUnitRecursively(i.targetUnitRoot,t);if(a)return a.unshift(i.node),a}return null},e.prototype.findUnitLinks=function(e){var t=[];return t=t.concat(this.findMasterLinks(e))},e.prototype.findMasterLinks=function(e){if(!e.getMaster)return[];var t=e.getMaster();if(!t)return[];var n=e.asElement().attr(y.Universe10.Extension.properties["extends"].name);return n&&n.value()?[new v(n,t)]:[]},e.prototype.convertConnectingNodeToError=function(e,t){if(!e)return null;var n=e.lowLevel(),r=n.start(),i=n.end(),a=this.generateLinkMessageByNode(e,t);return{code:t.code,isWarning:t.isWarning,message:a,node:e,start:r,end:i,path:n.unit()?n.unit().path():"",extras:[],unit:n.unit()}},e.prototype.generateLinkMessageByNode=function(e,t){return e.property().nameId()==y.Universe10.Extension.properties["extends"].name?"Error in the master file: "+t.message:t.message},e.prototype.findIssueTail=function(e){return e.extras&&0!=e.extras.length?this.findIssueTail(e.extras[0]):e},e}();t.PointOfViewValidationAcceptorImpl=g},function(e,t,n){"use strict";function r(e,t,n){return new i.Project(e,t,n)}var i=n(24);t.createProject=r},function(e,t,n){"use strict";function r(e,t){var n=n.createProject(m.dirname(e)),r={$ref:t},i=JSON.stringify(r),a=n.setCachedUnitContent(m.basename(e),i),s=o(i,a);return s.jsonSchema.$ref}function i(e,t){return v.createSchema(e,new y.ContentProvider(t))}function a(e){return v.getXMLSchema(e)}function o(e,t){return v.getJSONSchema(e,new y.ContentProvider(t))}function s(e,t,n){return d.completeReference(e,t,n)}function u(e){return d.getIncludePath(e)}function l(e){return d.getIncludeReference(e)}function p(){return new h.SchemaToModelGenerator}function c(){return new h.ModelToSchemaGenerator}var f=n(39),d=n(26),h=n(19),m=n(15),y=n(31),v=f.getSchemaUtils();t.dereference=r,t.createSchema=i,t.getXMLSchema=a,t.getJSONSchema=o,t.completeReference=s,t.getIncludePath=u,t.getIncludeReference=l,t.createSchemaModelGenerator=p,t.createModelToSchemaGenerator=c},function(e,t,n){"use strict";t.modify=n(20)},function(e,t,n){"use strict";function r(e){return i.expandTraitsAndResourceTypes(e)}var i=n(27);t.expandTraitsAndResourceTypes=r},function(e,t,n){"use strict";function r(e){return i.completeRelativeUri(e)}var i=n(34);t.completeRelativeUri=r},function(e,t,n){"use strict";function r(e,t){return t=t||{},c(e).then(function(e){var n;return n=!t.hasOwnProperty("expandLibraries")||t.expandLibraries?R.expandLibrariesHL(e):R.expandTraitsAndResourceTypesHL(e),C.dump(n,{rootNodeDetails:!0,attributeDefaults:!0,serializeMetadata:!0,unfoldTypes:t.unfoldTypes})})}function i(e,t){t=t||{};var n,r=u(e);return n=!t.hasOwnProperty("expandLibraries")||t.expandLibraries?L.isLibraryType(r.definition())?R.expandLibraryHL(r)||r:R.expandLibrariesHL(r)||r:R.expandTraitsAndResourceTypesHL(r)||r,C.dump(n,{rootNodeDetails:!0,attributeDefaults:!0,serializeMetadata:!0,unfoldTypes:t.unfoldTypes})}function a(e,t,n){var r=u(e,t,n);if(!r)return S.empty();var i=r.wrapperNode(),a=Array.isArray(t)?n:t;return A(i,a),new S(i)}function o(e,t,n){var r=u(e,t,n);if(!r)return S.empty();var i=r.wrapperNode(),a=Array.isArray(t)?n:t;return A(i,a),new S(i)}function s(e,t,n){var r=u(e,t,n);return r?new S(r):S.empty()}function u(e,t,n){var r=Array.isArray(t),i=r?t:null,a=r?n:t;a=a||{};var o=h(e,a),s=e.indexOf("://"),u=-1!=s&&6>s?e:T.basename(e),l=o.unit(u);n&&!i&&(i=null);var p;if(l)if(i&&i.length>0){var c=[];i.forEach(function(e){if(!e||0==e.trim().length)throw new Error("Extensions and overlays list should contain legal file paths")}),i.forEach(function(e){c.push(o.unit(e,T.isAbsolute(e)))}),c.forEach(function(e){return m(e,a)}),p=m(w.mergeAPIs(l,c,_.OverlayMergeMode.MERGE),a)}else p=m(l,a),p.setMergeMode(_.OverlayMergeMode.MERGE);if(!l)throw new Error("Can not resolve :"+e);if(a.rejectOnErrors&&p&&p.errors().filter(function(e){return!e.isWarning}).length)throw y(p);return p}function l(e,t,n){var r=p(e,t,n);return r.then(function(e){return e})}function p(e,t,n){return c(e,t,n).then(function(e){if(!e)return null;for(var r=Array.isArray(t),i=r?n:t,a=e;null!=a;){var o=a.wrapperNode();A(o,i);var s=a.getMaster();a=s?s.asElement():null}return e.wrapperNode()})}function c(e,t,n){var r=Array.isArray(t),i=r?t:null,a=r?n:t;a=a||{};var o=h(e,a),s=e.indexOf("://"),u=-1!=s&&6>s?e:T.basename(e);return n&&!i&&(i=null),i&&0!=i.length?(i.forEach(function(e){if(!e||0==e.trim().length)throw new Error("Extensions and overlays list should contain legal file paths")}),d(o,u,a).then(function(e){var t=[];return i.forEach(function(e){t.push(d(o,e,a))}),Promise.all(t).then(function(t){var n=[];t.forEach(function(e){return n.push(e.lowLevel().unit())});var r=w.mergeAPIs(e.lowLevel().unit(),n,_.OverlayMergeMode.MERGE);return r}).then(function(e){return m(e,a)})})):d(o,u,a).then(function(e){return e.setMergeMode(_.OverlayMergeMode.MERGE),e})}function f(e){if(null==e)return null;var t=e.getAdapter(M.RAMLService).getDeclaringNode();return null==t?null:t.wrapperNode()}function d(e,t,n){return N.fetchIncludesAndMasterAsync(e,t).then(function(e){try{var t=m(e,n);return n.rejectOnErrors&&t&&t.errors().filter(function(e){return!e.isWarning}).length?Promise.reject(y(t)):t}catch(r){return Promise.reject(r)}})}function h(e,t){t=t||{};var n,r=t.fsResolver,i=t.httpResolver,a=t.reusedNode;if(a)n=a.lowLevel().unit().project(),n.deleteUnit(T.basename(e)),r&&n.setFSResolver(r),i&&n.setHTTPResolver(i);else{var o=T.dirname(e);n=new b.Project(o,r,i)}return n}function m(e,t,n){if(void 0===n&&(n=!1),t=t||{},!e)return null;var r=null,i=null;e.isRAMLUnit?r=e:(i=e,r=i.lowLevel().unit());var a=r.contents(),o=_.ramlFirstLine(a);if(!o)throw new Error("Invalid first line. A RAML document is expected to start with '#%RAML <version> <?fragment type>'.");var s,u,l=o[1];o[2];if("0.8"==l?u="RAML08":"1.0"==l&&(u="RAML10"),!u)throw new Error("Unknown version of RAML expected to see one of '#%RAML 0.8' or '#%RAML 1.0'");if("RAML08"==u&&n)throw new Error("Extensions and overlays are not supported in RAML 0.8.");var p=O(u);p.type(s);return i||(i=_.fromUnit(r),t.reusedNode&&t.reusedNode.lowLevel().unit().absolutePath()==r.absolutePath()&&g(i,t.reusedNode)&&i.setReusedNode(t.reusedNode)),i}function y(e){var t=new Error("Api contains errors.");return t.parserErrors=_.toParserErrors(e.errors(),e),t}function v(e,t,n){void 0===t&&(t=!1),void 0===n&&(n=!0);var r=O("RAML10"),i=r.type(I.Universe10.Api.name),a=new b.Project(e),o=[];return a.units().forEach(function(e){var r=e.ast();t&&(r=N.toChildCachingNode(r));var a=new E.ApiImpl(new _.ASTNodeImpl(r,null,i,null));n&&(a=w.expandTraitsAndResourceTypes(a)),o.push(a)}),o}function g(e,t){if(!t)return!1;for(var n=e.lowLevel().unit().contents(),r=t.lowLevel().unit().contents(),i=Math.min(n.length,r.length),a=-1,o=0;i>o;o++)if(n.charAt(o)!=r.charAt(o)){a=o;break}for(;a>0&&""==n.charAt(a).replace(/\s/,"");)a--;0>a&&n.length!=r.length&&(a=i);var s=P.deepFindNode(t,a,a+1);if(!s)return!0;if(s.lowLevel().unit().absolutePath()!=e.lowLevel().unit().absolutePath())return!0;var u=s.isElement()?s.asElement():s.parent();if(!u)return!0;var l=u.property();if(!l)return!0;if(L.isAnnotationsProperty(l)&&(u=u.parent()),!u)return!0;for(var p=u;p;){var c=p.definition();if(L.isResourceTypeType(c)||L.isTraitType(c))return!1;var f=p.property();if(!f)return!0;if(L.isTypeDeclarationDescendant(c)&&(L.isTypesProperty(f)||L.isAnnotationTypesProperty(f)))return!1;var d=f.range();if(L.isResourceTypeRefType(d)||L.isTraitRefType(d))return!1;p=p.parent()}return!0}function A(e,t){t=t||{},null!=t.attributeDefaults&&e?e.setAttributeDefaults(t.attributeDefaults):e&&e.setAttributeDefaults(!0)}var E=n(21),T=n(15),S=n(22),b=n(24),_=n(16),N=n(24),w=n(27),R=n(30),I=n(18),M=n(37),C=n(32),L=n(14),P=n(13),O=n(38);t.load=r,t.loadSync=i,t.loadApi=a,t.loadRAML=o,t.loadRAMLHL=s,t.loadApiAsync=l,t.loadRAMLAsync=p,t.loadRAMLAsyncHL=c,t.getLanguageElementByRuntimeType=f,t.toError=y,t.loadApis1=v},function(e,t,n){(function(e){"use strict";function n(e){return e.asElement&&e.getKind&&e.asAttr&&e.lowLevel}function r(){var t=e.ramlValidation;if(t){var n=t.nodeValidators;if(Array.isArray(n))return n}return[]}function i(){var t=e.ramlValidation;if(t){var n=t.astAnnotationValidators;if(Array.isArray(n))return n}return[]}!function(e){e[e.BASIC=0]="BASIC",e[e.NODE=1]="NODE",e[e.ATTRIBUTE=2]="ATTRIBUTE"}(t.NodeKind||(t.NodeKind={}));t.NodeKind;!function(e){e[e.RAML10=0]="RAML10",e[e.RAML08=1]="RAML08"}(t.RAMLVersion||(t.RAMLVersion={}));t.RAMLVersion;t.isParseResult=n,function(e){e[e.UNRESOLVED_REFERENCE=0]="UNRESOLVED_REFERENCE",e[e.YAML_ERROR=1]="YAML_ERROR",e[e.UNKNOWN_NODE=2]="UNKNOWN_NODE",e[e.MISSING_REQUIRED_PROPERTY=3]="MISSING_REQUIRED_PROPERTY",e[e.PROPERTY_EXPECT_TO_HAVE_SINGLE_VALUE=4]="PROPERTY_EXPECT_TO_HAVE_SINGLE_VALUE",e[e.KEY_SHOULD_BE_UNIQUE_INTHISCONTEXT=5]="KEY_SHOULD_BE_UNIQUE_INTHISCONTEXT",e[e.UNABLE_TO_RESOLVE_INCLUDE_FILE=6]="UNABLE_TO_RESOLVE_INCLUDE_FILE",e[e.INVALID_VALUE_SCHEMA=7]="INVALID_VALUE_SCHEMA",e[e.MISSED_CONTEXT_REQUIREMENT=8]="MISSED_CONTEXT_REQUIREMENT",e[e.NODE_HAS_VALUE=9]="NODE_HAS_VALUE",e[e.ONLY_OVERRIDE_ALLOWED=10]="ONLY_OVERRIDE_ALLOWED",e[e.ILLEGAL_PROPERTY_VALUE=11]="ILLEGAL_PROPERTY_VALUE",e[e.ILLEGAL_PROPERTY=12]="ILLEGAL_PROPERTY",e[e.INVALID_PROPERTY=13]="INVALID_PROPERTY"}(t.IssueCode||(t.IssueCode={}));t.IssueCode;t.getNodeValidationPlugins=r,t.getNodeAnnotationValidationPlugins=i}).call(t,function(){return this}())},function(e,t,n){"use strict";function r(e,t){return new E(v.CHANGE_VALUE,e,t,-1)}function i(e,t){return new E(v.CHANGE_VALUE,e,t.lowLevel(),-1)}function a(e,t){return new E(v.CHANGE_KEY,e,t,-1)}function o(e,t){return new E(v.REMOVE_CHILD,e,t,-1)}function s(e,t,n,r){void 0===n&&(n=null),void 0===r&&(r=!1);var i=new E(v.ADD_CHILD,e,t,-1);return i.insertionPoint=n,i.toSeq=r,i}function u(e,t){return new E(v.INIT_RAML_FILE,e,t,-1)}function l(e,t,n){if(d.isAbsolute(e)){var r=d.extname(t);".xsd"!=r&&(e=e.substr(1),t=p(n,d.basename(t)))}return c(e)||d.isAbsolute(e)?e:c(t)||d.isAbsolute(t)?p(d.dirname(t),e):p(d.dirname(p(n,t)),e)}function p(e,t){if(c(t))return t;var n;if(c(e)){var r=m.stringEndsWith(e,"/")?e:e+"/";n=h.resolve(r,t).replace(/\\/g,"/")}else n=d.resolve(e,t).replace(/\\/g,"/");return n}function c(e){return null==e?!1:m.stringStartsWith(e,"http://")||m.stringStartsWith(e,"https://")}function f(e){return e.start&&e.end&&e.unit&&e.key&&e.value&&e.children&&e.includePath}var d=n(15),h=n(40),m=n(33),y=function(){function e(){}return e}();t.ASTDelta=y,function(e){e[e.ADD_CHILD=0]="ADD_CHILD",e[e.REMOVE_CHILD=1]="REMOVE_CHILD",e[e.MOVE_CHILD=2]="MOVE_CHILD",e[e.CHANGE_KEY=3]="CHANGE_KEY",e[e.CHANGE_VALUE=4]="CHANGE_VALUE",e[e.INIT_RAML_FILE=5]="INIT_RAML_FILE"}(t.CommandKind||(t.CommandKind={}));var v=t.CommandKind,g=function(){function e(e,t,n,r,i){void 0===i&&(i=null),this.offset=e,this.replacementLength=t,this.text=n,this.unit=r,this.target=i}return e}();t.TextChangeCommand=g;var A=function(){function e(){this.commands=[]}return e}();t.CompositeCommand=A,function(e){e[e.NONE=0]="NONE",e[e.START=1]="START",e[e.END=2]="END",e[e.POINT=3]="POINT"}(t.InsertionPointType||(t.InsertionPointType={}));var E=(t.InsertionPointType,function(){function e(e,t,n,r){this.toSeq=!1,this.kind=e,this.target=t,this.value=n,this.position=r}return e}());t.ASTChangeCommand=E,t.setAttr=r,t.setAttrStructured=i,t.setKey=a,t.removeNode=o,t.insertNode=s,t.initRamlFile=u;var T=function(){function e(e,t){this.content=e,this.absPath=t}return e.prototype.position=function(e){var t=e;this.initMapping();for(var n=0;n<this.mapping.length;n++){var r=this.mapping[n];if(r>t)return{line:n,column:t,position:e};t-=r}if(0==t)return{line:this.mapping.length-1,column:this.mapping[this.mapping.length-1],position:this.content.length};if(1==t)return{line:this.mapping.length-1,column:this.mapping[this.mapping.length-1]-1,position:e-1};throw new Error("Character position exceeds text length: "+e+" > + "+this.content.length+".\nUnit path: "+this.absPath)},e.prototype.initMapping=function(){if(null==this.mapping){if(null==this.content)throw new Error("Line Mapper has been given null content"+(null!=this.absPath?". Path: "+this.absPath:" and null path."));this.mapping=[];for(var e=0,t=this.content.length,n=0;t>n;n++)"\r"==this.content.charAt(n)?t-1>n&&"\n"==this.content.charAt(n+1)?(this.mapping.push(n-e+2),e=n+2,n++):(this.mapping.push(n-e+1),e=n+1):"\n"==this.content.charAt(n)&&(this.mapping.push(n-e+1),e=n+1);this.mapping.push(t-e)}},e.prototype.toPosition=function(e,t){var n=t;this.initMapping();for(var r=e;r<this.mapping.length;r++){var i=this.mapping[r];if(i>n){for(var a=n,o=0;r>o;o++)a+=this.mapping[o];return{line:r,column:n,position:a}}n-=i}return{line:r,column:n,position:this.content.length}},e}();t.LineMapperImpl=T,t.buildPath=l,t.toAbsolutePath=p,t.isWebPath=c,t.isLowLevelNode=f},function(e,t,n){"use strict";function r(e){return"Api"==e.kind()&&"RAML10"==e.RAMLVersion()}function i(e){return"LibraryBase"==e.kind()&&"RAML10"==e.RAMLVersion()}function a(e){return"Annotable"==e.kind()&&"RAML10"==e.RAMLVersion()}function o(e){return"AnnotationRef"==e.kind()&&"RAML10"==e.RAMLVersion()}function s(e){return"Reference"==e.kind()&&"RAML10"==e.RAMLVersion()}function u(e){return"ValueType"==e.kind()&&"RAML10"==e.RAMLVersion()}function l(e){return"StringType"==e.kind()&&"RAML10"==e.RAMLVersion()}function p(e){return"UriTemplate"==e.kind()&&"RAML10"==e.RAMLVersion()}function c(e){return"RelativeUriString"==e.kind()&&"RAML10"==e.RAMLVersion()}function f(e){return"FullUriTemplateString"==e.kind()&&"RAML10"==e.RAMLVersion()}function d(e){return"StatusCodeString"==e.kind()&&"RAML10"==e.RAMLVersion()}function h(e){return"FixedUriString"==e.kind()&&"RAML10"==e.RAMLVersion()}function m(e){return"ContentType"==e.kind()&&"RAML10"==e.RAMLVersion()}function y(e){return"MarkdownString"==e.kind()&&"RAML10"==e.RAMLVersion()}function v(e){return"SchemaString"==e.kind()&&"RAML10"==e.RAMLVersion()}function g(e){return"MimeType"==e.kind()&&"RAML10"==e.RAMLVersion()}function A(e){return"AnyType"==e.kind()&&"RAML10"==e.RAMLVersion()}function E(e){return"NumberType"==e.kind()&&"RAML10"==e.RAMLVersion()}function T(e){return"IntegerType"==e.kind()&&"RAML10"==e.RAMLVersion()}function S(e){return"NullType"==e.kind()&&"RAML10"==e.RAMLVersion()}function b(e){return"TimeOnlyType"==e.kind()&&"RAML10"==e.RAMLVersion()}function _(e){return"DateOnlyType"==e.kind()&&"RAML10"==e.RAMLVersion()}function N(e){return"DateTimeOnlyType"==e.kind()&&"RAML10"==e.RAMLVersion()}function w(e){return"DateTimeType"==e.kind()&&"RAML10"==e.RAMLVersion()}function R(e){return"FileType"==e.kind()&&"RAML10"==e.RAMLVersion()}function I(e){return"BooleanType"==e.kind()&&"RAML10"==e.RAMLVersion()}function M(e){return"AnnotationTarget"==e.kind()&&"RAML10"==e.RAMLVersion()}function C(e){return"TraitRef"==e.kind()&&"RAML10"==e.RAMLVersion()}function L(e){return"Trait"==e.kind()&&"RAML10"==e.RAMLVersion()}function P(e){return"MethodBase"==e.kind()&&"RAML10"==e.RAMLVersion()}function O(e){return"Operation"==e.kind()&&"RAML10"==e.RAMLVersion()}function D(e){return"TypeDeclaration"==e.kind()&&"RAML10"==e.RAMLVersion()}function x(e){return"ModelLocation"==e.kind()&&"RAML10"==e.RAMLVersion()}function U(e){return"LocationKind"==e.kind()&&"RAML10"==e.RAMLVersion()}function k(e){return"ExampleSpec"==e.kind()&&"RAML10"==e.RAMLVersion()}function F(e){return"UsesDeclaration"==e.kind()&&"RAML10"==e.RAMLVersion()}function B(e){return"XMLFacetInfo"==e.kind()&&"RAML10"==e.RAMLVersion()}function K(e){return"ArrayTypeDeclaration"==e.kind()&&"RAML10"==e.RAMLVersion()}function V(e){return"UnionTypeDeclaration"==e.kind()&&"RAML10"==e.RAMLVersion()}function j(e){return"ObjectTypeDeclaration"==e.kind()&&"RAML10"==e.RAMLVersion()}function W(e){return"StringTypeDeclaration"==e.kind()&&"RAML10"==e.RAMLVersion()}function q(e){return"BooleanTypeDeclaration"==e.kind()&&"RAML10"==e.RAMLVersion()}function Y(e){return"NumberTypeDeclaration"==e.kind()&&"RAML10"==e.RAMLVersion()}function H(e){return"IntegerTypeDeclaration"==e.kind()&&"RAML10"==e.RAMLVersion()}function $(e){return"DateOnlyTypeDeclaration"==e.kind()&&"RAML10"==e.RAMLVersion()}function G(e){return"TimeOnlyTypeDeclaration"==e.kind()&&"RAML10"==e.RAMLVersion()}function X(e){return"DateTimeOnlyTypeDeclaration"==e.kind()&&"RAML10"==e.RAMLVersion()}function z(e){return"DateTimeTypeDeclaration"==e.kind()&&"RAML10"==e.RAMLVersion()}function J(e){return"FileTypeDeclaration"==e.kind()&&"RAML10"==e.RAMLVersion()}function Q(e){return"Response"==e.kind()&&"RAML10"==e.RAMLVersion()}function Z(e){return"SecuritySchemePart"==e.kind()&&"RAML10"==e.RAMLVersion()}function ee(e){return"SecuritySchemeRef"==e.kind()&&"RAML10"==e.RAMLVersion()}function te(e){return"AbstractSecurityScheme"==e.kind()&&"RAML10"==e.RAMLVersion()}function ne(e){return"SecuritySchemeSettings"==e.kind()&&"RAML10"==e.RAMLVersion()}function re(e){return"OAuth1SecuritySchemeSettings"==e.kind()&&"RAML10"==e.RAMLVersion()}function ie(e){return"OAuth2SecuritySchemeSettings"==e.kind()&&"RAML10"==e.RAMLVersion()}function ae(e){return"OAuth2SecurityScheme"==e.kind()&&"RAML10"==e.RAMLVersion()}function oe(e){return"OAuth1SecurityScheme"==e.kind()&&"RAML10"==e.RAMLVersion()}function se(e){return"PassThroughSecurityScheme"==e.kind()&&"RAML10"==e.RAMLVersion()}function ue(e){return"BasicSecurityScheme"==e.kind()&&"RAML10"==e.RAMLVersion()}function le(e){return"DigestSecurityScheme"==e.kind()&&"RAML10"==e.RAMLVersion()}function pe(e){return"CustomSecurityScheme"==e.kind()&&"RAML10"==e.RAMLVersion()}function ce(e){return"Method"==e.kind()&&"RAML10"==e.RAMLVersion()}function fe(e){return"ResourceTypeRef"==e.kind()&&"RAML10"==e.RAMLVersion()}function de(e){return"ResourceType"==e.kind()&&"RAML10"==e.RAMLVersion()}function he(e){return"ResourceBase"==e.kind()&&"RAML10"==e.RAMLVersion()}function me(e){return"Resource"==e.kind()&&"RAML10"==e.RAMLVersion()}function ye(e){return"DocumentationItem"==e.kind()&&"RAML10"==e.RAMLVersion()}function ve(e){return"Library"==e.kind()&&"RAML10"==e.RAMLVersion()}function ge(e){return"Overlay"==e.kind()&&"RAML10"==e.RAMLVersion()}function Ae(e){return"Extension"==e.kind()&&"RAML10"==e.RAMLVersion()}function Ee(e){return null==e.highLevel().parent()}function Te(e){return Ee(e)?e:null}t.isApi=r,t.isLibraryBase=i,t.isAnnotable=a,t.isAnnotationRef=o,t.isReference=s,t.isValueType=u,t.isStringType=l,t.isUriTemplate=p,t.isRelativeUriString=c,t.isFullUriTemplateString=f,t.isStatusCodeString=d,t.isFixedUriString=h,t.isContentType=m,t.isMarkdownString=y,t.isSchemaString=v,t.isMimeType=g,t.isAnyType=A,t.isNumberType=E,t.isIntegerType=T,t.isNullType=S,t.isTimeOnlyType=b,t.isDateOnlyType=_,t.isDateTimeOnlyType=N,t.isDateTimeType=w,t.isFileType=R,t.isBooleanType=I,t.isAnnotationTarget=M,t.isTraitRef=C,t.isTrait=L,t.isMethodBase=P,t.isOperation=O,t.isTypeDeclaration=D,t.isModelLocation=x,t.isLocationKind=U,t.isExampleSpec=k,t.isUsesDeclaration=F,t.isXMLFacetInfo=B,t.isArrayTypeDeclaration=K,t.isUnionTypeDeclaration=V,t.isObjectTypeDeclaration=j,t.isStringTypeDeclaration=W,t.isBooleanTypeDeclaration=q,t.isNumberTypeDeclaration=Y,t.isIntegerTypeDeclaration=H,t.isDateOnlyTypeDeclaration=$,t.isTimeOnlyTypeDeclaration=G,t.isDateTimeOnlyTypeDeclaration=X,t.isDateTimeTypeDeclaration=z,t.isFileTypeDeclaration=J,t.isResponse=Q,t.isSecuritySchemePart=Z,t.isSecuritySchemeRef=ee,t.isAbstractSecurityScheme=te,t.isSecuritySchemeSettings=ne,t.isOAuth1SecuritySchemeSettings=re,t.isOAuth2SecuritySchemeSettings=ie,t.isOAuth2SecurityScheme=ae,t.isOAuth1SecurityScheme=oe,t.isPassThroughSecurityScheme=se,t.isBasicSecurityScheme=ue,t.isDigestSecurityScheme=le,t.isCustomSecurityScheme=pe,t.isMethod=ce,t.isResourceTypeRef=fe,t.isResourceType=de,t.isResourceBase=he,t.isResource=me,t.isDocumentationItem=ye,t.isLibrary=ve,t.isOverlay=ge,t.isExtension=Ae,t.isFragment=Ee,t.asFragment=Te},function(e,t,n){"use strict";function r(e){return"Api"==e.kind()&&"RAML08"==e.RAMLVersion()}function i(e){return"FullUriTemplateString"==e.kind()&&"RAML08"==e.RAMLVersion()}function a(e){return"UriTemplate"==e.kind()&&"RAML08"==e.RAMLVersion()}function o(e){return"StringType"==e.kind()&&"RAML08"==e.RAMLVersion()}function s(e){return"ValueType"==e.kind()&&"RAML08"==e.RAMLVersion()}function u(e){return"AnyType"==e.kind()&&"RAML08"==e.RAMLVersion()}function l(e){return"NumberType"==e.kind()&&"RAML08"==e.RAMLVersion()}function p(e){return"BooleanType"==e.kind()&&"RAML08"==e.RAMLVersion()}function c(e){return"Reference"==e.kind()&&"RAML08"==e.RAMLVersion()}function f(e){return"ResourceTypeRef"==e.kind()&&"RAML08"==e.RAMLVersion()}function d(e){return"ResourceType"==e.kind()&&"RAML08"==e.RAMLVersion()}function h(e){return"Method"==e.kind()&&"RAML08"==e.RAMLVersion()}function m(e){return"MethodBase"==e.kind()&&"RAML08"==e.RAMLVersion()}function y(e){return"Response"==e.kind()&&"RAML08"==e.RAMLVersion()}function v(e){return"StatusCodeString"==e.kind()&&"RAML08"==e.RAMLVersion()}function g(e){return"Parameter"==e.kind()&&"RAML08"==e.RAMLVersion()}function A(e){return"ParameterLocation"==e.kind()&&"RAML08"==e.RAMLVersion()}function E(e){return"MarkdownString"==e.kind()&&"RAML08"==e.RAMLVersion()}function T(e){return"StringTypeDeclaration"==e.kind()&&"RAML08"==e.RAMLVersion()}function S(e){return"BooleanTypeDeclaration"==e.kind()&&"RAML08"==e.RAMLVersion()}function b(e){return"NumberTypeDeclaration"==e.kind()&&"RAML08"==e.RAMLVersion()}function _(e){return"IntegerTypeDeclaration"==e.kind()&&"RAML08"==e.RAMLVersion()}function N(e){return"DateTypeDeclaration"==e.kind()&&"RAML08"==e.RAMLVersion()}function w(e){return"FileTypeDeclaration"==e.kind()&&"RAML08"==e.RAMLVersion()}function R(e){return"BodyLike"==e.kind()&&"RAML08"==e.RAMLVersion()}function I(e){return"SchemaString"==e.kind()&&"RAML08"==e.RAMLVersion()}function M(e){return"JSonSchemaString"==e.kind()&&"RAML08"==e.RAMLVersion()}function C(e){return"XMLSchemaString"==e.kind()&&"RAML08"==e.RAMLVersion()}function L(e){return"ExampleString"==e.kind()&&"RAML08"==e.RAMLVersion()}function P(e){return"JSONExample"==e.kind()&&"RAML08"==e.RAMLVersion()}function O(e){return"XMLExample"==e.kind()&&"RAML08"==e.RAMLVersion()}function D(e){return"XMLBody"==e.kind()&&"RAML08"==e.RAMLVersion()}function x(e){return"JSONBody"==e.kind()&&"RAML08"==e.RAMLVersion()}function U(e){return"SecuritySchemeRef"==e.kind()&&"RAML08"==e.RAMLVersion()}function k(e){return"AbstractSecurityScheme"==e.kind()&&"RAML08"==e.RAMLVersion()}function F(e){return"SecuritySchemePart"==e.kind()&&"RAML08"==e.RAMLVersion()}function B(e){return"TraitRef"==e.kind()&&"RAML08"==e.RAMLVersion()}function K(e){return"Trait"==e.kind()&&"RAML08"==e.RAMLVersion()}function V(e){return"SecuritySchemeSettings"==e.kind()&&"RAML08"==e.RAMLVersion()}function j(e){return"OAuth1SecuritySchemeSettings"==e.kind()&&"RAML08"==e.RAMLVersion()}function W(e){return"FixedUri"==e.kind()&&"RAML08"==e.RAMLVersion()}function q(e){return"OAuth2SecuritySchemeSettings"==e.kind()&&"RAML08"==e.RAMLVersion()}function Y(e){return"OAuth2SecurityScheme"==e.kind()&&"RAML08"==e.RAMLVersion()}function H(e){return"OAuth1SecurityScheme"==e.kind()&&"RAML08"==e.RAMLVersion()}function $(e){return"BasicSecurityScheme"==e.kind()&&"RAML08"==e.RAMLVersion()}function G(e){return"DigestSecurityScheme"==e.kind()&&"RAML08"==e.RAMLVersion()}function X(e){return"CustomSecurityScheme"==e.kind()&&"RAML08"==e.RAMLVersion()}function z(e){return"MimeType"==e.kind()&&"RAML08"==e.RAMLVersion()}function J(e){return"RelativeUriString"==e.kind()&&"RAML08"==e.RAMLVersion()}function Q(e){return"GlobalSchema"==e.kind()&&"RAML08"==e.RAMLVersion()}function Z(e){return"RAMLSimpleElement"==e.kind()&&"RAML08"==e.RAMLVersion()}function ee(e){return"DocumentationItem"==e.kind()&&"RAML08"==e.RAMLVersion()}function te(e){return"Resource"==e.kind()&&"RAML08"==e.RAMLVersion()}t.isApi=r,t.isFullUriTemplateString=i,t.isUriTemplate=a,t.isStringType=o,t.isValueType=s,t.isAnyType=u,t.isNumberType=l,t.isBooleanType=p,t.isReference=c,t.isResourceTypeRef=f,t.isResourceType=d,t.isMethod=h,t.isMethodBase=m,t.isResponse=y,t.isStatusCodeString=v,t.isParameter=g,t.isParameterLocation=A,t.isMarkdownString=E,t.isStringTypeDeclaration=T,t.isBooleanTypeDeclaration=S,t.isNumberTypeDeclaration=b,t.isIntegerTypeDeclaration=_,t.isDateTypeDeclaration=N,t.isFileTypeDeclaration=w,t.isBodyLike=R,t.isSchemaString=I,t.isJSonSchemaString=M,t.isXMLSchemaString=C,t.isExampleString=L,t.isJSONExample=P,t.isXMLExample=O,t.isXMLBody=D,t.isJSONBody=x,t.isSecuritySchemeRef=U,t.isAbstractSecurityScheme=k,t.isSecuritySchemePart=F,t.isTraitRef=B,t.isTrait=K,t.isSecuritySchemeSettings=V,t.isOAuth1SecuritySchemeSettings=j,t.isFixedUri=W,t.isOAuth2SecuritySchemeSettings=q,t.isOAuth2SecurityScheme=Y,t.isOAuth1SecurityScheme=H,t.isBasicSecurityScheme=$,t.isDigestSecurityScheme=G,t.isCustomSecurityScheme=X,t.isMimeType=z,t.isRelativeUriString=J,t.isGlobalSchema=Q,t.isRAMLSimpleElement=Z,t.isDocumentationItem=ee,t.isResource=te},function(e,t,n){"use strict";function r(e,t,n){return _.findDeclaration(e,t,n)}function i(e,t){return _.findUsages(e,t)}function a(e){return _.globalDeclarations(e)}function o(e,t,n){_.refFinder(e,t,n)}function s(e,t){return _.findDeclarationByNode(e,t)}function u(e,t){return _.determineCompletionKind(e,t)}function l(e,t){return _.enumValues(e,t)}function p(e,t){return N.qName(e,t)}function c(e,t){return _.subTypesWithLocals(e,t)}function f(e,t){return _.nodesDeclaringType(e,t)}function d(e){return _.isExampleNodeContent(e)}function h(e){return _.findExampleContentType(e)}function m(e,t){return _.parseDocumentationContent(e,t)}function y(e,t){return _.parseStructuredExample(e,t)}function v(e){return _.isExampleNode(e)}function g(e,t){return _.referenceTargets(e,t)}function A(e){return w.getNominalTypeSource(e)}function E(e,t){
return _.findAllSubTypes(e,t)}function T(e){return _.declRoot(e)}function S(e,t,n,r,i){return void 0===r&&(r=!0),void 0===i&&(i=!0),_.deepFindNode(e,t,n,r,i)}function b(e){return _.allChildren(e)}var _=n(35),N=n(16),w=n(36);t.findDeclaration=r,t.findUsages=i,t.globalDeclarations=a,t.refFinder=o,t.findDeclarationByNode=s,function(e){e[e.VALUE_COMPLETION=0]="VALUE_COMPLETION",e[e.KEY_COMPLETION=1]="KEY_COMPLETION",e[e.PATH_COMPLETION=2]="PATH_COMPLETION",e[e.DIRECTIVE_COMPLETION=3]="DIRECTIVE_COMPLETION",e[e.VERSION_COMPLETION=4]="VERSION_COMPLETION",e[e.ANNOTATION_COMPLETION=5]="ANNOTATION_COMPLETION",e[e.SEQUENCE_KEY_COPLETION=6]="SEQUENCE_KEY_COPLETION",e[e.INCOMMENT=7]="INCOMMENT"}(t.LocationKind||(t.LocationKind={}));t.LocationKind;t.determineCompletionKind=u,t.enumValues=l,t.qName=p,t.subTypesWithLocals=c,t.nodesDeclaringType=f,t.isExampleNodeContent=d,t.findExampleContentType=h,t.parseDocumentationContent=m,t.parseStructuredExample=y,t.isExampleNode=v,t.referenceTargets=g,t.getNominalTypeSource=A,t.findAllSubTypes=E,t.declRoot=T,t.deepFindNode=S,t.allChildren=b},function(e,t,n){"use strict";function r(e){return e?e.nameId()===$e.Universe10.Api.properties.documentation.name||e.nameId()===$e.Universe08.Api.properties.documentation.name:!1}function i(e){return e===$e.Universe10.Trait.properties.usage.name||e===$e.Universe08.Trait.properties.usage.name||e===$e.Universe10.ResourceType.properties.usage.name||e===$e.Universe08.ResourceType.properties.usage.name||e===$e.Universe10.Library.properties.usage.name||e===$e.Universe10.Overlay.properties.usage.name||e===$e.Universe10.Extension.properties.usage.name}function a(e){return e?i(e.nameId()):!1}function o(e){return e?e.nameId()==$e.Universe10.Overlay.properties["extends"].name||e.nameId()==$e.Universe10.Extension.properties["extends"].name:!1}function s(e){return e===$e.Universe10.TypeDeclaration.properties.description.name||"description"===e}function u(e){return e?s(e.nameId()):!1}function l(e){return e===$e.Universe10.TypeDeclaration.properties.required.name||e===$e.Universe08.Parameter.properties.required.name||"required"===e}function p(e){return e===$e.Universe10.TypeDeclaration.properties.displayName.name||"displayName"===e}function c(e){return e?p(e.nameId()):!1}function f(e){return e?l(e.nameId()):!1}function d(e){return e===$e.Universe10.Api.properties.title.name||e===$e.Universe08.Api.properties.title.name||e===$e.Universe10.DocumentationItem.properties.title.name||e===$e.Universe08.DocumentationItem.properties.title.name||e===$e.Universe10.Overlay.properties.title.name||e===$e.Universe10.Extension.properties.title.name}function h(e){return e?d(e.nameId()):!1}function m(e){return e?y(e.nameId()):!1}function y(e){return e===$e.Universe08.MethodBase.properties.headers.name||e===$e.Universe08.Response.properties.headers.name||e===$e.Universe08.SecuritySchemePart.properties.headers.name||e===$e.Universe10.MethodBase.properties.headers.name||e===$e.Universe10.Response.properties.headers.name}function v(e){return e?g(e.nameId()):!1}function g(e){return e===$e.Universe08.BodyLike.properties.formParameters.name}function A(e){return e?E(e.nameId()):!1}function E(e){return e===$e.Universe08.MethodBase.properties.queryParameters.name||e===$e.Universe08.SecuritySchemePart.properties.queryParameters.name||e===$e.Universe10.MethodBase.properties.queryParameters.name}function T(e){return e?e.nameId()===$e.Universe10.Api.properties.annotations.name||e.nameId()===$e.Universe10.TypeDeclaration.properties.annotations.name||e.nameId()===$e.Universe10.Response.properties.annotations.name:!1}function S(e){return e?e.nameId()===$e.Universe10.AnnotationRef.properties.annotation.name:!1}function b(e){return e?e.nameId()===$e.Universe10.MethodBase.properties.is.name||e.nameId()===$e.Universe08.Method.properties.is.name||e.nameId()===$e.Universe10.ResourceBase.properties.is.name||e.nameId()===$e.Universe08.ResourceType.properties.is.name||e.nameId()===$e.Universe08.Resource.properties.is.name:!1}function _(e){return e?e.nameId()===$e.Universe10.Api.properties.securedBy.name||e.nameId()===$e.Universe08.Api.properties.securedBy.name||e.nameId()===$e.Universe10.MethodBase.properties.securedBy.name||e.nameId()===$e.Universe08.MethodBase.properties.securedBy.name||e.nameId()===$e.Universe08.ResourceType.properties.securedBy.name||e.nameId()===$e.Universe08.Resource.properties.securedBy.name||e.nameId()===$e.Universe10.ResourceBase.properties.securedBy.name:!1}function N(e){return e?e.nameId()===$e.Universe10.LibraryBase.properties.securitySchemes.name||e.nameId()===$e.Universe08.Api.properties.securitySchemes.name:!1}function w(e){return e?e.nameId()===$e.Universe10.SecuritySchemeRef.properties.securityScheme.name||e.nameId()===$e.Universe08.SecuritySchemeRef.properties.securityScheme.name:!1}function R(e){return I(e)||k(e)}function I(e){return e?e.nameId()===$e.Universe10.AbstractSecurityScheme.properties.type.name||e.nameId()===$e.Universe08.AbstractSecurityScheme.properties.type.name||e.nameId()===$e.Universe08.ResourceType.properties.type.name||e.nameId()===$e.Universe08.Resource.properties.type.name||e.nameId()===$e.Universe08.Parameter.properties.type.name||e.nameId()===$e.Universe10.ResourceBase.properties.type.name||e.nameId()===$e.Universe10.TypeDeclaration.properties.type.name:!1}function M(e){return e?e.nameId()===$e.Universe10.ArrayTypeDeclaration.properties.items.name:!1}function C(e){return e?e.nameId()===$e.Universe10.ArrayTypeDeclaration.properties.structuredItems.name:!1}function L(e){return e?e.nameId()===$e.Universe10.ObjectTypeDeclaration.properties.properties.name:!1}function P(e){return e?e.nameId()===$e.Universe10.MethodBase.properties.responses.name||e.nameId()===$e.Universe08.MethodBase.properties.responses.name:!1}function O(e){return e?e.nameId()===$e.Universe10.Api.properties.protocols.name||e.nameId()===$e.Universe08.Api.properties.protocols.name||e.nameId()===$e.Universe10.MethodBase.properties.protocols.name:!1}function D(e){return e?e.nameId()===$e.Universe10.TypeDeclaration.properties.name.name||e.nameId()===$e.Universe10.TypeDeclaration.properties.name.name||e.nameId()===$e.Universe08.AbstractSecurityScheme.properties.name.name||e.nameId()===$e.Universe10.AbstractSecurityScheme.properties.name.name||e.nameId()===$e.Universe08.Trait.properties.name.name||e.nameId()===$e.Universe10.Trait.properties.name.name||"name"===e.nameId():!1}function x(e){return e?e.nameId()===$e.Universe10.MethodBase.properties.body.name||e.nameId()===$e.Universe08.MethodBase.properties.body.name||e.nameId()===$e.Universe10.Response.properties.body.name||e.nameId()===$e.Universe08.Response.properties.body.name:!1}function U(e){return e?e.nameId()===$e.Universe10.TypeDeclaration.properties["default"].name||e.nameId()===$e.Universe08.Parameter.properties["default"].name:!1}function k(e){return e?e.nameId()===$e.Universe08.BodyLike.properties.schema.name||e.nameId()===$e.Universe08.XMLBody.properties.schema.name||e.nameId()===$e.Universe08.JSONBody.properties.schema.name||e.nameId()===$e.Universe10.TypeDeclaration.properties.schema.name:!1}function F(e){return e?e.nameId()===$e.Universe08.Api.properties.traits.name||e.nameId()===$e.Universe10.LibraryBase.properties.traits.name:!1}function B(e){return e?e.nameId()===$e.Universe08.TraitRef.properties.trait.name||e.nameId()===$e.Universe10.TraitRef.properties.trait.name:!1}function K(e){return e?e.nameId()===$e.Universe08.Api.properties.resourceTypes.name||e.nameId()===$e.Universe10.LibraryBase.properties.resourceTypes.name:!1}function V(e){return e?e.nameId()===$e.Universe08.ResourceTypeRef.properties.resourceType.name||e.nameId()===$e.Universe10.ResourceTypeRef.properties.resourceType.name:!1}function j(e){return e?e.nameId()===$e.Universe10.TypeDeclaration.properties.facets.name:!1}function W(e){return e?e.nameId()===$e.Universe08.Api.properties.schemas.name||e.nameId()===$e.Universe10.LibraryBase.properties.schemas.name:!1}function q(e){return e?e.nameId()===$e.Universe10.Api.properties.resources.name||e.nameId()===$e.Universe08.Api.properties.resources.name||e.nameId()===$e.Universe10.Resource.properties.resources.name||e.nameId()===$e.Universe08.Resource.properties.resources.name:!1}function Y(e){return e?e.nameId()===$e.Universe10.ResourceBase.properties.methods.name||e.nameId()===$e.Universe08.Resource.properties.methods.name||e.nameId()===$e.Universe08.ResourceType.properties.methods.name:!1}function H(e){return e&&e.nameId()===$e.Universe10.LibraryBase.properties.types.name}function $(e){return e?e.nameId()===$e.Universe10.TypeDeclaration.properties.example.name||"example"===e.nameId():!1}function G(e){return e?e.nameId()===$e.Universe10.StringTypeDeclaration.properties["enum"].name||e.nameId()===$e.Universe10.NumberTypeDeclaration.properties["enum"].name||e.nameId()===$e.Universe08.StringTypeDeclaration.properties["enum"].name:!1}function X(e){return e?e.nameId()===$e.Universe10.TypeDeclaration.properties.example.name||e.nameId()===$e.Universe10.TypeDeclaration.properties.examples.name:!1}function z(e){return e?e.nameId()===$e.Universe08.GlobalSchema.properties.value.name:!1}function J(e){return e?e.nameId()===$e.Universe08.Api.properties.uriParameters.name||e.nameId()===$e.Universe08.ResourceType.properties.uriParameters.name||e.nameId()===$e.Universe08.Resource.properties.uriParameters.name||e.nameId()===$e.Universe10.ResourceBase.properties.uriParameters.name:!1}function Q(e){return e?e.nameId()===$e.Universe08.Resource.properties.baseUriParameters.name||e.nameId()===$e.Universe08.Api.properties.baseUriParameters.name||e.nameId()===$e.Universe10.Api.properties.baseUriParameters.name:!1}function Z(e){return e?e.nameId()===$e.Universe08.Api.properties.RAMLVersion.name||e.nameId()===$e.Universe10.Api.properties.RAMLVersion.name:!1}function ee(e){return e?e.nameId()===$e.Universe10.FragmentDeclaration.properties.uses.name:!1}function te(e){return e?e.nameId()===$e.Universe10.LibraryBase.properties.annotationTypes.name:!1}function ne(e){return e.key()==$e.Universe10.Method||e.key()==$e.Universe08.Method}function re(e){return e.key()==$e.Universe10.Api||e.key()==$e.Universe08.Api}function ie(e){return e.key()==$e.Universe10.BooleanType||e.key()==$e.Universe08.BooleanType}function ae(e){return e.key()==$e.Universe10.MarkdownString||e.key()==$e.Universe08.MarkdownString}function oe(e){return e.key()==$e.Universe10.Resource||e.key()==$e.Universe08.Resource}function se(e){return e.key()==$e.Universe10.Trait||e.key()==$e.Universe08.Trait}function ue(e){return e.key()==$e.Universe10.TraitRef||e.key()==$e.Universe08.TraitRef}function le(e){return e.key()==$e.Universe10.ResourceTypeRef||e.key()==$e.Universe08.ResourceTypeRef}function pe(e){return e.key()==$e.Universe08.GlobalSchema}function ce(e){return e.key()==$e.Universe10.AbstractSecurityScheme||e.key()==$e.Universe08.AbstractSecurityScheme}function fe(e){return e.isAssignableFrom($e.Universe10.AbstractSecurityScheme.name)}function de(e){return e.key()==$e.Universe10.SecuritySchemeRef||e.key()==$e.Universe08.SecuritySchemeRef}function he(e){return e.key()==$e.Universe10.TypeDeclaration}function me(e){return e.key()==$e.Universe10.Response||e.key()==$e.Universe08.Response}function ye(e){return e.key()==$e.Universe08.BodyLike}function ve(e){return e.key()==$e.Universe10.Overlay}function ge(e){return!1}function Ae(e){return e.key()==$e.Universe10.ResourceType||e.key()==$e.Universe08.ResourceType}function Ee(e){return e.key()==$e.Universe10.SchemaString||e.key()==$e.Universe08.SchemaString}function Te(e){return e.key()==$e.Universe10.MethodBase||e.key()==$e.Universe08.MethodBase}function Se(e){return e.key()==$e.Universe10.Library}function be(e){return e.key()==$e.Universe10.StringType||e.key()==$e.Universe08.StringType}function _e(e){return e.key()==$e.Universe10.AnyType||e.key()==$e.Universe08.AnyType}function Ne(e){return e.key()==$e.Universe10.ExampleSpec}function we(e){return e.key()==$e.Universe10.Extension}function Re(e){return e.isAssignableFrom($e.Universe10.TypeDeclaration.name)}function Ie(e){return e.key()==$e.Universe10.DocumentationItem||e.key()==$e.Universe08.DocumentationItem}function Me(e){return e.isAssignableFrom($e.Universe10.AnnotationRef.name)}function Ce(e){return e.isAssignableFrom($e.Universe10.Api.name)||e.isAssignableFrom($e.Universe08.Api.name)}function Le(e){return e.isAssignableFrom($e.Universe10.LibraryBase.name)}function Pe(e){return e.isAssignableFrom($e.Universe10.ResourceBase.name)||e.isAssignableFrom($e.Universe08.Resource.name)}function Oe(e){return e.isAssignableFrom($e.Universe10.ObjectTypeDeclaration.name)}function De(e){return e.isAssignableFrom($e.Universe10.ArrayTypeDeclaration.name)}function xe(e){return e.isAssignableFrom($e.Universe10.TypeDeclaration.name)}function Ue(e){return e.isAssignableFrom($e.Universe10.StringTypeDeclaration.name)}function ke(e){return e.isAssignableFrom($e.Universe10.TypeDeclaration.name)}function Fe(e){return e.isAssignableFrom($e.Universe10.MethodBase.name)||e.isAssignableFrom($e.Universe08.MethodBase.name)}function Be(e){return e.key()==$e.Universe10.SecuritySchemePart||e.key()==$e.Universe08.SecuritySchemePart}function Ke(e){return e.nameId()===$e.Universe08.Api.properties.mediaType.name||e.nameId()===$e.Universe10.Api.properties.mediaType.name}function Ve(e){return"RAML08"==e.universe().version()}function je(e){return"RAML10"==e.universe().version()}function We(e){return Ve(e.definition())}function qe(e){return Ve(e.definition())}function Ye(e){return je(e.definition())}function He(e){return je(e.definition())}var $e=n(18);t.isDocumentationProperty=r,t.isUsagePropertyName=i,t.isUsageProperty=a,t.isMasterRefProperty=o,t.isDescriptionPropertyName=s,t.isDescriptionProperty=u,t.isRequiredPropertyName=l,t.isDisplayNamePropertyName=p,t.isDisplayNameProperty=c,t.isRequiredProperty=f,t.isTitlePropertyName=d,t.isTitleProperty=h,t.isHeadersProperty=m,t.isHeadersPropertyName=y,t.isFormParametersProperty=v,t.isFormParametersPropertyName=g,t.isQueryParametersProperty=A,t.isQueryParametersPropertyName=E,t.isAnnotationsProperty=T,t.isAnnotationProperty=S,t.isIsProperty=b,t.isSecuredByProperty=_,t.isSecuritySchemesProperty=N,t.isSecuritySchemeProperty=w,t.isTypeOrSchemaProperty=R,t.isTypeProperty=I,t.isItemsProperty=M,t.isStructuredItemsProperty=C,t.isPropertiesProperty=L,t.isResponsesProperty=P,t.isProtocolsProperty=O,t.isNameProperty=D,t.isBodyProperty=x,t.isDefaultValue=U,t.isSchemaProperty=k,t.isTraitsProperty=F,t.isTraitProperty=B,t.isResourceTypesProperty=K,t.isResourceTypeProperty=V,t.isFacetsProperty=j,t.isSchemasProperty=W,t.isResourcesProperty=q,t.isMethodsProperty=Y,t.isTypesProperty=H,t.isExampleProperty=$,t.isEnumProperty=G,t.isExamplesProperty=X,t.isValueProperty=z,t.isUriParametersProperty=J,t.isBaseUriParametersProperty=Q,t.isRAMLVersionProperty=Z,t.isUsesProperty=ee,t.isAnnotationTypesProperty=te,t.isMethodType=ne,t.isApiType=re,t.isBooleanTypeType=ie,t.isMarkdownStringType=ae,t.isResourceType=oe,t.isTraitType=se,t.isTraitRefType=ue,t.isResourceTypeRefType=le,t.isGlobalSchemaType=pe,t.isSecuritySchemaType=ce,t.isSecuritySchemaTypeDescendant=fe,t.isSecuritySchemeRefType=de,t.isTypeDeclarationType=he,t.isResponseType=me,t.isBodyLikeType=ye,t.isOverlayType=ve,t.isAnnotationTypeType=ge,t.isResourceTypeType=Ae,t.isSchemaStringType=Ee,t.isMethodBaseType=Te,t.isLibraryType=Se,t.isStringTypeType=be,t.isAnyTypeType=_e,t.isExampleSpecType=Ne,t.isExtensionType=we,t.isTypeDeclarationTypeOrDescendant=Re,t.isDocumentationType=Ie,t.isAnnotationRefTypeOrDescendant=Me,t.isApiSibling=Ce,t.isLibraryBaseSibling=Le,t.isResourceBaseSibling=Pe,t.isObjectTypeDeclarationSibling=Oe,t.isArrayTypeDeclarationSibling=De,t.isTypeDeclarationDescendant=xe,t.isStringTypeDeclarationDescendant=Ue,t.isTypeDeclarationSibling=ke,t.isMethodBaseSibling=Fe,t.isSecuritySchemePartType=Be,t.isMediaTypeProperty=Ke,t.isRAML08Type=Ve,t.isRAML10Type=je,t.isRAML08Node=We,t.isRAML08Attribute=qe,t.isRAML10Node=Ye,t.isRAML10Attribute=He},function(e,t,n){(function(e){function n(e,t){for(var n=0,r=e.length-1;r>=0;r--){var i=e[r];"."===i?e.splice(r,1):".."===i?(e.splice(r,1),n++):n&&(e.splice(r,1),n--)}if(t)for(;n--;n)e.unshift("..");return e}function r(e,t){if(e.filter)return e.filter(t);for(var n=[],r=0;r<e.length;r++)t(e[r],r,e)&&n.push(e[r]);return n}var i=/^(\/?|)([\s\S]*?)((?:\.{1,2}|[^\/]+?|)(\.[^.\/]*|))(?:[\/]*)$/,a=function(e){return i.exec(e).slice(1)};t.resolve=function(){for(var t="",i=!1,a=arguments.length-1;a>=-1&&!i;a--){var o=a>=0?arguments[a]:e.cwd();if("string"!=typeof o)throw new TypeError("Arguments to path.resolve must be strings");o&&(t=o+"/"+t,i="/"===o.charAt(0))}return t=n(r(t.split("/"),function(e){return!!e}),!i).join("/"),(i?"/":"")+t||"."},t.normalize=function(e){var i=t.isAbsolute(e),a="/"===o(e,-1);return e=n(r(e.split("/"),function(e){return!!e}),!i).join("/"),e||i||(e="."),e&&a&&(e+="/"),(i?"/":"")+e},t.isAbsolute=function(e){return"/"===e.charAt(0)},t.join=function(){var e=Array.prototype.slice.call(arguments,0);return t.normalize(r(e,function(e,t){if("string"!=typeof e)throw new TypeError("Arguments to path.join must be strings");return e}).join("/"))},t.relative=function(e,n){function r(e){for(var t=0;t<e.length&&""===e[t];t++);for(var n=e.length-1;n>=0&&""===e[n];n--);return t>n?[]:e.slice(t,n-t+1)}e=t.resolve(e).substr(1),n=t.resolve(n).substr(1);for(var i=r(e.split("/")),a=r(n.split("/")),o=Math.min(i.length,a.length),s=o,u=0;o>u;u++)if(i[u]!==a[u]){s=u;break}for(var l=[],u=s;u<i.length;u++)l.push("..");return l=l.concat(a.slice(s)),l.join("/")},t.sep="/",t.delimiter=":",t.dirname=function(e){var t=a(e),n=t[0],r=t[1];return n||r?(r&&(r=r.substr(0,r.length-1)),n+r):"."},t.basename=function(e,t){var n=a(e)[2];return t&&n.substr(-1*t.length)===t&&(n=n.substr(0,n.length-t.length)),n},t.extname=function(e){return a(e)[3]};var o="b"==="ab".substr(-1)?function(e,t,n){return e.substr(t,n)}:function(e,t,n){return 0>t&&(t=e.length+t),e.substr(t,n)}}).call(t,n(64))},function(e,t,n){"use strict";function r(e,t){var n=e.name();if(S.LowLevelProxyNode.isInstance(t.lowLevel())){if(S.LowLevelProxyNode.isInstance(e.lowLevel()))return n;var r=t.root().lowLevel().unit(),i=r.project().namespaceResolver(),a=e.lowLevel().unit(),o=i.resolveNamespace(t.lowLevel().unit(),a);if(null!=o){var s=o.namespace();if(null!=s)return s+"."+n}}if(e.lowLevel().unit()!=t.lowLevel().unit())for(var u=t;;){if(u.lowLevel().includePath()||null==u.parent()){u.unitMap||(u.unitMap={},u.asElement().elements().forEach(function(e){if(e.definition().key()==C.Universe10.UsesDeclaration){var t=e.attr("value");if(t){var n=e.root().lowLevel().unit().resolve(t.value());if(null!=n){var r=e.attr("key");r&&(u.unitMap[n.absolutePath()]=r.value())}}}}));var l=u.unitMap[e.lowLevel().unit().absolutePath()];if(l)return l+"."+n}if(!u.parent())break;u=u.parent()}return n}function i(e){var t=e;return t&&t.valueName&&t.toHighLevel&&t.toHighLevel2}function a(e){var t=e;return t&&t.isString&&t.isFromKey&&t.isEmbedded}function o(e){var t=D.newMap([D.newMapping(D.newScalar("example"),e.lowLevel().actual())]),n=D.newMapping(D.newScalar("types"),D.newMap([D.newMapping(D.newScalar("__AUX_TYPE__"),t)])),r=D.newMap([n]),i=new L.ASTNode(r,e.lowLevel().unit(),null,null,null),a=K.parseFromAST(new G(i,e)),o=K.toNominal(a.types()[0],function(e){return null});return o}function s(e){return e.match(/^\s*#%RAML\s+(\d\.\d)\s*(\w*)\s*$/m)}function u(e){var t=e.lowLevel()&&e.lowLevel().unit()&&e.lowLevel().unit().contents();return null==t?null:J(t,e.lowLevel()).ptype}function l(e){if(null==e)return null;var t=e.contents(),n=e.ast(),r=J(t,n),i=r.ptype,a=r.localUniverse,o=a.type(i);o||(o=a.type("Api"));var s=new z(n,null,o,null);s.setUniverse(a);var u=o&&o.universe();return u&&"RAML10"==u.version()?o.isAssignableFrom(b.universesInfo.Universe10.LibraryBase.name)||s.children():s.children(),s}function p(e,t){if(t){var n=t.root().lowLevel().unit();return n?new j.PointOfViewValidationAcceptorImpl(e,t.root()):{accept:function(t){e.push(t)},begin:function(){},end:function(){},acceptUnique:function(t){for(var n=0,r=e;n<r.length;n++){var i=r[n];if(i.start==t.start&&i.message==t.message)return}this.accept(t)}}}}function c(e){if(null!=e.parent())return!1;var t=e.lowLevel().unit();if(null==t)return!1;var n=t.contents(),r=s(n);return r.length<3?!1:"AnnotationTypeDeclaration"==r[2]}function f(e,t,n){var r=e.validationIssue;if(!r){var i=e.issueCode||t,a=e.node||n,o=e.message||"The "+t+" plugin reports an error",s=e.isWarning;r=R.createIssue(i,o,a,s)}return r}function d(e){for(var t=[],n=E.getNodeValidationPlugins(),r=0,i=n;r<i.length;r++){var a=i[r],o=a.process(e);o&&o.forEach(function(n){var r=f(n,a.id(),e);t.push(r)})}return t}function h(e){for(var t=new Q(e),n=[],r=E.getNodeAnnotationValidationPlugins(),i=0,a=r;i<a.length;i++){var o=a[i],s=o.process(t);s&&s.forEach(function(t){var r=f(t,o.id(),e);n.push(r)})}return n}function m(e,t){var n=e.map(function(e){return v(e,t)}),r=y(n);return r}function y(e){var t=[],n={};e.map(function(e){n[JSON.stringify(e)]=e});for(var r=Object.keys(n),i=0;i<r.length;i++)t.push(n[r[i]]);return t}function v(e,t){var n=e.node&&e.node.lowLevel()&&e.node.lowLevel().unit().lineMapper()||t.lowLevel().unit().lineMapper(),r=null;try{r=n.position(e.start)}catch(i){console.warn(i)}var a=null;try{a=n.position(e.end)}catch(i){console.warn(i)}var o;o=e.path?e.path:e.node?e.node.lowLevel().unit().path():N.declRoot(t).lowLevel().unit().path();var s={code:e.code,message:e.message,path:o,range:{start:r,end:a},isWarning:e.isWarning};return e.extras&&e.extras.length>0&&(s.trace=e.extras.map(function(e){return v(e,t)})),s}var g=this&&this.__extends||function(e,t){function n(){this.constructor=e}for(var r in t)t.hasOwnProperty(r)&&(e[r]=t[r]);e.prototype=null===t?Object.create(t):(n.prototype=t.prototype,new n)},A=n(39),E=n(9),T=n(70),S=n(44),b=A,_=n(29),N=n(13),w=n(45),R=n(28),I=n(27),M=n(46),C=n(18),L=n(24),P=n(43),O=b,D=n(77),x=n(34),U=n(47),k=n(48),F=n(14),B=n(25),K=A.rt,V=n(31),j=n(2);t.qName=r;var W=function(){function e(e,t){this._node=e,this._parent=t,this._implicit=!1,this.values={},e&&e.setHighLevelParseResult(this)}return e.isInstance=function(t){return null!=t&&t.getClassIdentifier&&"function"==typeof t.getClassIdentifier&&T.contains(t.getClassIdentifier(),e.CLASS_IDENTIFIER)},e.prototype.getClassIdentifier=function(){var t=[];return t.concat(e.CLASS_IDENTIFIER)},e.prototype.getKind=function(){return E.NodeKind.BASIC},e.prototype.asAttr=function(){return null},e.prototype.asElement=function(){return null},e.prototype.hashkey=function(){return this._hashkey||(this._hashkey=this.parent()?this.parent().hashkey()+"/"+this.name():this.name()),this._hashkey},e.prototype.root=function(){return this.parent()?this.parent().root():this},e.prototype.version=function(){return""},e.prototype.getLowLevelStart=function(){return this.lowLevel().kind()===L.Kind.SCALAR?this.lowLevel().start():this.lowLevel().keyStart()},e.prototype.getLowLevelEnd=function(){return this.lowLevel().kind()===L.Kind.SCALAR?this.lowLevel().end():this.lowLevel().keyEnd()},e.prototype.isSameNode=function(e){return e&&e.lowLevel().actual()==this.lowLevel().actual()?!0:!1},e.prototype.checkContextValue=function(e,t,n){var r=this.computedValue(e);return r&&-1!=r.indexOf(t)?!0:t==r||"false"==t},e.prototype.printDetails=function(e){return(e?e:"")+"Unkown\n"},e.prototype.testSerialize=function(e){return(e?e:"")+"Unkown\n"},e.prototype.errors=function(){var e=[],t=p(e,this);return this.validate(t),e},e.prototype.markCh=function(){for(var e=this.lowLevel();S.LowLevelProxyNode.isInstance(e);)e=e.originalNode();return e=e._node?e._node:e,e.markCh?!0:void(e.markCh=1)},e.prototype.unmarkCh=function(){for(var e=this.lowLevel();S.LowLevelProxyNode.isInstance(e);)e=e.originalNode();e=e._node?e._node:e,delete e.markCh},e.prototype.validate=function(e){R.validate(this,e);for(var t=0,n=d(this);t<n.length;t++){var r=n[t];e.accept(r)}for(var i=0,a=h(this);i<a.length;i++){var r=a[i];e.accept(r)}},e.prototype.allowRecursive=function(){return!1},e.prototype.setComputed=function(e,t){this.values[e]=t},e.prototype.computedValue=function(e){var t=this.values[e];return!t&&this.parent()?this.parent().computedValue(e):t},e.prototype.lowLevel=function(){return this._node},e.prototype.name=function(){var e=this.lowLevel().key();return e?e:""},e.prototype.optional=function(){var e=this.lowLevel(),t=e.optional();if(null!=e.key())return t;var n=this.property();if(!n||!n.isMultiValue())return t;for(var r=e.parent();r&&null==r.highLevelNode();){if(r.kind()==D.Kind.MAPPING)return r.optional();r=r.parent()}return t},e.prototype.parent=function(){return this._parent},e.prototype.setParent=function(e){this._parent=e},e.prototype.isElement=function(){return!1},e.prototype.directChildren=function(){return this.children()},e.prototype.children=function(){return[]},e.prototype.isAttached=function(){return null!=this.parent()},e.prototype.isImplicit=function(){return this._implicit},e.prototype.isAttr=function(){return!1},e.prototype.isUnknown=function(){return!0},e.prototype.id=function(){var e=this;if(this.cachedId)return this.cachedId;if(this._parent){var t=this.parent().id();t+="."+this.name();var n=this.parent().directChildren().filter(function(t){return t.name()==e.name()});if(n.length>1){var r=n.indexOf(this);t+="["+r+"]"}return this.cachedId=t,t}return this.cachedId="",this.cachedId},e.prototype.localId=function(){return this.name()},e.prototype.resetIDs=function(){this.cachedId=null,this.cachedFullId=null},e.prototype.fullLocalId=function(){var e=this;if(this.cachedFullId)return this.cachedFullId;if(this._parent){var t=".";t+=null!=this.property()&&F.isAnnotationsProperty(this.property())?this.lowLevel().key():this.name();var n=this.parent().directChildren().filter(function(t){return t.name()==e.name()});if(n.length>1){var r=n.indexOf(this);t+="["+r+"]"}return this.cachedFullId=t,t}return this.cachedFullId=this.localId(),this.cachedFullId},e.prototype.property=function(){return null},e.prototype.reuseMode=function(){return this._reuseMode},e.prototype.setReuseMode=function(e){this._reuseMode=e},e.prototype.isReused=function(){return this._isReused},e.prototype.setReused=function(e){this._isReused=e,this.children().forEach(function(t){return t.setReused(e)})},e.prototype.setJSON=function(e){this._jsonCache=e},e.prototype.getJSON=function(){return this._jsonCache},e.CLASS_IDENTIFIER="highLevelImpl.BasicASTNode",e}();t.BasicASTNode=W;var q=function(){function e(e,t,n,r){void 0===r&&(r=null),this.node=e,this._parent=t,this.kv=r,this._pr=n}return e.isInstance=function(t){return null!=t&&t.getClassIdentifier&&"function"==typeof t.getClassIdentifier&&T.contains(t.getClassIdentifier(),e.CLASS_IDENTIFIER)},e.prototype.getClassIdentifier=function(){var t=[];return t.concat(e.CLASS_IDENTIFIER)},e.prototype.valueName=function(){var e=null;return this.kv&&(e=this.kv),e=this.node.key(),this._pr&&this._pr.isAnnotation()&&e&&"("==e.charAt(0)&&(e=e.substring(1,e.length-1)),e},e.prototype.children=function(){return this.node.children().map(function(t){return new e(t,null,null)})},e.prototype.lowLevel=function(){return this.node},e.prototype.toHighLevel=function(e){if(!e&&this._parent&&(e=this._parent),this._hl)return this._hl;var t=this.valueName(),n=e;if(S.LowLevelProxyNode.isInstance(this.node)){var i=this.node.definingUnitSequence(),a=i&&i[0]&&i[0].highLevel().asElement(),o=a&&a.lowLevel().unit().absolutePath();o==e.lowLevel().unit().absolutePath()?a=e:o==e.root().lowLevel().unit().absolutePath()&&(a=e.root()),n=a||n}var s=N.referenceTargets(this._pr,n).filter(function(e){return r(e,n)==t});if(s&&s[0]){var u=s[0].localType(),l=new z(this.node,e,u,this._pr);return this._pr&&this._pr.childRestrictions().forEach(function(e){l.setComputed(e.name,e.value)}),this._hl=l,l}return null},e.prototype.toHighLevel2=function(e){!e&&this._parent&&(e=this._parent);var t=this.valueName(),n=N.referenceTargets(this._pr,e).filter(function(n){return r(n,e)==t});if(n&&n[0]){var i=n[0].localType(),a=new z(this.node,e,i,this._pr);return this._pr&&this._pr.childRestrictions().forEach(function(e){a.setComputed(e.name,e.value)}),a}if(this._pr.range()){var a=new z(this.node.parent(),e,this._pr.range(),this._pr);return this._pr&&this._pr.childRestrictions().forEach(function(e){a.setComputed(e.name,e.value)}),a}return null},e.prototype.resetHighLevelNode=function(){this._hl=null},e.CLASS_IDENTIFIER="highLevelImpl.StructuredValue",e}();t.StructuredValue=q,t.isStructuredValue=i;var Y=function(e){function t(t,n,r,i,a){void 0===a&&(a=!1),e.call(this,t,n),this._def=r,this._prop=i,this.fromKey=a}return g(t,e),t.isInstance=function(e){return null!=e&&e.getClassIdentifier&&"function"==typeof e.getClassIdentifier&&T.contains(e.getClassIdentifier(),t.CLASS_IDENTIFIER_ASTPropImpl)},t.prototype.getClassIdentifier=function(){var n=e.prototype.getClassIdentifier.call(this);return n.concat(t.CLASS_IDENTIFIER_ASTPropImpl)},t.prototype.definition=function(){return this._def},t.prototype.asAttr=function(){return this},t.prototype.errors=function(){var e=[],t=p(e,this);return this.parent().validate(t),e},t.prototype.isString=function(){return!this._def||this._def.key()!==C.Universe08.StringType&&this._def.key()!=C.Universe10.StringType?!1:!0},t.prototype.isAnnotatedScalar=function(){return this.property().isAnnotation()||this.property().isKey()?!1:this.lowLevel().isAnnotatedScalar()},t.prototype.annotations=function(){var e=this.lowLevel().children(),n=[],r=this.definition().universe().type(C.Universe10.Annotable.name);if(!r)return n;for(var i=r.property("annotations"),a=0;a<e.length;a++){var o=e[a],s=o.key();if(null!=s&&"("==s[0]&&")"==s[s.length-1]){var u=new t(o,this.parent(),i.range(),i);n.push(u)}}return n},t.prototype.getKind=function(){return E.NodeKind.ATTRIBUTE},t.prototype.owningWrapper=function(){return{node:this.parent().wrapperNode(),property:this.name()}},t.prototype.patchType=function(e){this._def=e},t.prototype.findReferenceDeclaration=function(){var e=this,t=N.referenceTargets(this.property(),this.parent()),n=this.value();if(q.isInstance(n))var i=n,a=i.valueName();else var a=""+n;var o=T.find(t,function(t){return r(t,e.parent())==a});return o},t.prototype.findReferencedValue=function(){var e=this.findReferenceDeclaration();if(e){var t=e.attr("value"),n=e.definition().key();if(n===C.Universe08.GlobalSchema){if(t){var r=t.value();if(r){var i=R.isValid(this._def,this.parent(),r,t.property(),t);return i}}return null}}return e},t.prototype.isElement=function(){return!1},t.prototype.property=function(){return this._prop},t.prototype.convertMultivalueToString=function(e){for(var t=0,n=2;" "==e[n];)t++,n++;var r=P.splitOnLines(e);return r=r.map(function(e){return e.substring(t,e.length)}),r.join("")},t.prototype.overrideValue=function(e){this._value=e},t.prototype.value=function(){return this._value?this._value:(this._value=this.calcValue(),this._value)},t.prototype.calcValue=function(){if(this._computed)return this.computedValue(this.property().nameId());if(this.fromKey){var e=this.parent(),t=e.definition();if("RAML08"==t.universe().version())return this._node.key();if(F.isNameProperty(this.property())&&t.isAssignableFrom(C.Universe10.TypeDeclaration.name)){var n=e.attr("required");if(n&&null!=n.value())return this._node.optional()?this._node.key()+"?":this._node.key()}return this._node.key()}if(this.property().isAnnotation()&&this._node.key()&&"annotations"!=this._node.key())return new q(this._node,this.parent(),this._prop);var r=null!=this.property()&&!(null==this.parent()||null==this.parent().definition()||F.isTypeOrSchemaProperty(this.property())&&F.isTypeDeclarationDescendant(this.parent().definition()))&&F.isStringTypeType(this.property().range()),i=this._node.value(r);if(this.property().isSelfNode()&&(!i||L.ASTNode.isInstance(i))&&(i=this._node,0==i.children().length&&(i=null)),L.ASTNode.isInstance(i)||S.LowLevelProxyNode.isInstance(i)){var a=!1;if(this.property().range().hasStructure()||this._node.isAnnotatedScalar()&&this._node.children().forEach(function(e){
"value"===e.key()&&(i=e.value(r),a=!0)}),!a)return this._sval?this._sval:(this._sval=new q(i,this.parent(),this._prop),this._sval)}if("string"==typeof i&&P.isMultiLineValue(i)){var o=this.convertMultivalueToString(i);return o}return null==i&&this._node.children().length>0&&this.property()&&(F.isTypeOrSchemaProperty(this.property())||F.isItemsProperty(this.property()))&&this.parent()&&F.isTypeDeclarationDescendant(this.parent().definition())?new q(this._node,this.parent(),this._prop):i},t.prototype.name=function(){return this._prop.nameId()},t.prototype.printDetails=function(e){var t=this.definition().nameId(),n=this.property().range().nameId(),r=(e?e:"")+(this.name()+" : "+t+"["+n+"]  =  "+this.value())+(this.property().isKey()&&this.optional()?"?":"")+"\n";if(q.isInstance(this.value())){var i=this.value().toHighLevel();i&&i.printDetails&&(r+=i.printDetails(e+"	"))}return r},t.prototype.testSerialize=function(e){var t=this.definition().nameId(),n=(e?e:"")+(this.name()+" : "+t+"  =  "+this.value())+"\n";if(q.isInstance(this.value())){var r=this.value().toHighLevel();if(r&&r.testSerialize)n+=r.testSerialize((e?e:"")+"  ");else{var i=this.value().lowLevel(),a=i.dumpToObject(),o=JSON.stringify(a),s="",u=o.split("\n");u.forEach(function(t){return s+=(e?e:"")+"  "+t+"\n"}),n+=s+"\n"}}return n},t.prototype.isAttr=function(){return!0},t.prototype.isUnknown=function(){return!1},t.prototype.setValue=function(e){w.setValue(this,e),this._value=null},t.prototype.setKey=function(e){w.setKey(this,e),this._value=null},t.prototype.children=function(){return[]},t.prototype.addStringValue=function(e){w.addStringValue(this,e),this._value=null},t.prototype.addStructuredValue=function(e){w.addStructuredValue(this,e),this._value=null},t.prototype.addValue=function(e){if(!this.property().isMultiValue())throw new Error("setValue(string) only apply to multi-values properties");"string"==typeof e?this.addStringValue(e):this.addStructuredValue(e),this._value=null},t.prototype.isEmbedded=function(){var e=this.lowLevel().asMapping().key.value;return this.property().canBeValue()&&e!=this.property().nameId()},t.prototype.remove=function(){w.removeAttr(this)},t.prototype.setValues=function(e){w.setValues(this,e),this._value=null},t.prototype.isEmpty=function(){if(!this.property().isMultiValue())throw new Error("isEmpty() only apply to multi-values attributes");var e=this.parent(),t=(e.lowLevel(),e.attributes(this.name()));if(0==t.length)return!0;if(1==t.length){var n=t[0].lowLevel();return n.isMapping()&&null==n.value()?!0:!1}return!1},t.prototype.isFromKey=function(){return this.fromKey},t.CLASS_IDENTIFIER_ASTPropImpl="highLevelImpl.ASTPropImpl",t}(W);t.ASTPropImpl=Y,t.isASTPropImpl=a;var H=new _.BasicNodeBuilder;!function(e){e[e.MERGE=0]="MERGE",e[e.AGGREGATE=1]="AGGREGATE"}(t.OverlayMergeMode||(t.OverlayMergeMode={}));var $=t.OverlayMergeMode,G=function(e){function t(n,r){e.call(this),this._node=n,this._highLevelRoot=r;var i=r.root(),a=i.getMaster();if(a&&this._node===r.lowLevel()){var o=r.getMasterCounterPart();o&&(this._toMerge=new t(o.lowLevel(),o))}}return g(t,e),t.isInstance=function(e){return null!=e&&e.getClassIdentifier&&"function"==typeof e.getClassIdentifier&&T.contains(e.getClassIdentifier(),t.CLASS_IDENTIFIER_LowLevelWrapperForTypeSystem)},t.prototype.getClassIdentifier=function(){var n=e.prototype.getClassIdentifier.call(this);return n.concat(t.CLASS_IDENTIFIER_LowLevelWrapperForTypeSystem)},t.prototype.contentProvider=function(){var e=this._node&&this._node.includeBaseUnit()&&(this._node.includePath&&this._node.includePath()?this._node.includeBaseUnit().resolve(this._node.includePath()):this._node.includeBaseUnit());return new V.ContentProvider(e)},t.prototype.key=function(){var e=this._node.key();return this._node.optional()&&(e+="?"),e},t.prototype.value=function(){var e=this._node.resolvedValueKind();if(e===D.Kind.SEQ)return this.children().map(function(e){return e.value()});if(e===D.Kind.MAP){var t=this._node.dumpToObject(!1);return t[this.key()]}if(this._node.kind()==D.Kind.MAP){var t=this._node.dumpToObject(!1);return t}var n=this._node.value();return n},t.prototype.children=function(){var e=this;if(this._children)return this._children;"uses"!=this.key()||this._node.parent().parent()?this._children=this._node.children().map(function(n){return new t(n,e._highLevelRoot)}):this._children=this._node.children().map(function(t){return new X(t,e._highLevelRoot)}),this.childByKey={};for(var n=0;n<this._children.length;n++){var r=this._children[n];this.childByKey[r.key()]=r}if(this._toMerge)for(var i=this._toMerge.children(),n=0;n<i.length;n++){var r=i[n],a=this.childByKey[r.key()];a?a._toMerge=r:(this._children.push(r),this.childByKey[r.key()]=r)}return this._children},t.prototype.childWithKey=function(e){return this._children||this.children(),this.childByKey[e]},t.prototype.kind=function(){var e=this._node.valueKind();if(e==D.Kind.MAPPING||null===e)return K.NodeKind.MAP;if(e==D.Kind.MAP)return K.NodeKind.MAP;var t=this._node.kind();return t==D.Kind.MAP?K.NodeKind.MAP:e==D.Kind.SEQ?K.NodeKind.ARRAY:(e==D.Kind.INCLUDE_REF||e==D.Kind.ANCHOR_REF)&&this._node.children().length>0?K.NodeKind.MAP:K.NodeKind.SCALAR},t.prototype.getSource=function(){if(!this._node)return null;var e=this._node.highLevelNode();if(!e){var t=this._node.start(),n=N.deepFindNode(this._highLevelRoot,t,t,!0,!1);return n&&(this._node.setHighLevelParseResult(n),z.isInstance(n)&&this._node.setHighLevelNode(n)),n}return e},t.prototype.node=function(){return this._node},t.CLASS_IDENTIFIER_LowLevelWrapperForTypeSystem="highLevelImpl.LowLevelWrapperForTypeSystem",t}(A.SourceProvider);t.LowLevelWrapperForTypeSystem=G;var X=function(e){function t(){e.apply(this,arguments)}return g(t,e),t.prototype.children=function(){var e=this._node.unit().resolve(this.value());return e&&e.isRAMLUnit()&&e.contents().trim().length>0?new G(e.ast(),this._highLevelRoot).children():[]},t.prototype.anchor=function(){return this._node.actual()},t.prototype.childWithKey=function(e){for(var t=this.children(),n=0;n<t.length;n++)if(t[n].key()==e)return t[n];return null},t}(G);t.UsesNodeWrapperFoTypeSystem=X,t.auxiliaryTypeForExample=o;var z=function(e){function t(t,n,r,i){e.call(this,t,n),this._def=r,this._prop=i,this._expanded=!1,this._allowQuestion=!1,this._subTypesCache=null,this._auxChecked=!1,this.overlayMergeMode=$.MERGE,t&&t.setHighLevelNode(this),S.LowLevelProxyNode.isInstance(t)&&(this._expanded=!0)}return g(t,e),t.isInstance=function(e){return null!=e&&e.getClassIdentifier&&"function"==typeof e.getClassIdentifier&&T.contains(e.getClassIdentifier(),t.CLASS_IDENTIFIER_ASTNodeImpl)},t.prototype.getClassIdentifier=function(){var n=e.prototype.getClassIdentifier.call(this);return n.concat(t.CLASS_IDENTIFIER_ASTNodeImpl)},t.prototype.createIssue=function(e){return R.toIssue(e,this)},t.prototype.validate=function(t){var n=this.definition().key();if((n==C.Universe10.Api||n==C.Universe08.Api||n==C.Universe10.Extension)&&!this.isExpanded()){var r=I.expandTraitsAndResourceTypes(this.wrapperNode()),i=r.highLevel();return i.resetChildren(),i.children(),i._expanded=!0,i.clearTypesCache(),void i.validate(t)}(n==C.Universe10.Overlay||n==C.Universe10.Extension)&&this.clearTypesCache(),e.prototype.validate.call(this,t)},t.prototype.clearTypesCache=function(){if(this._types=null,this.lowLevel()){var e=this.lowLevel().actual();e.types=null}},t.prototype.types=function(){var e=this;if(!this._types){if(this.parent()&&this.definition().key()!==C.Universe10.Library)return this.parent().types();var t=this.lowLevel().actual();if(t.types)return t.types;this._types=K.parseFromAST(new G(this.lowLevel(),this)),this._types.types().forEach(function(t){var n=M.convertType(e,t);n.putExtra(A.USER_DEFINED_EXTRA,!0)}),t.types=this._types}return this._types},t.prototype.setTypes=function(e){this._types=e},t.prototype.parsedType=function(){if(!this._ptype){if(this.property()&&this.property().nameId()==C.Universe10.MethodBase.properties.body.name){var e=this.isParametrizedType();this._ptype=K.parseTypeFromAST(this.name(),new G(this.lowLevel(),this),this.types(),!0,!1,!1,e)}else{var t=this.property()&&this.property().nameId()==C.Universe10.LibraryBase.properties.annotationTypes.name,n=!this.property()||this.property().nameId()==C.Universe10.LibraryBase.properties.types.name||this.property().nameId()==C.Universe10.LibraryBase.properties.schemas.name;if(!t&&null==this.parent()){var r=this.universe();r&&r.getOriginalTopLevelText()&&"AnnotationTypeDeclaration"==r.getOriginalTopLevelText()&&(t=!0)}this._ptype=K.parseTypeFromAST(this.name(),new G(this.lowLevel(),this),this.types(),!1,t,n)}this.property()&&F.isTypesProperty(this.property())&&this.parent()&&F.isApiType(this.parent().definition())&&this._ptype.putExtra&&this._ptype.putExtra(A.DEFINED_IN_TYPES_EXTRA,!0);var i=this._ptype;i.putExtra(A.USER_DEFINED_EXTRA,!0),this._ptype.putExtra(A.SOURCE_EXTRA,this)}return this._ptype},t.prototype.isParametrizedType=function(){var e=!1,t=this.attr(C.Universe10.TypeDeclaration.properties.type.name);if(t){var n=t.value();if("string"==typeof n&&n.indexOf("<<")>=0)for(var r=this.parent();null!=r;){if(F.isResourceTypeType(r.definition())||F.isTraitType(r.definition())){e=!0;break}r=r.parent()}}return e},t.prototype.localType=function(){return M.typeFromNode(this)},t.prototype.patchProp=function(e){this._prop=e},t.prototype.getKind=function(){return E.NodeKind.NODE},t.prototype.wrapperNode=function(){if(!this._wrapperNode){if(F.isExampleSpecType(this.definition())){var e=o(this),t=x.examplesFromNominal(e,this,!0,!1);return t[0]}var n=this.definition()&&this.definition().universe();n&&"RAML10"==n.version()?this.definition()&&this.definition().isAssignableFrom(b.universesInfo.Universe10.LibraryBase.name)||this.children():this.children(),this._wrapperNode=this.buildWrapperNode()}return this._wrapperNode},t.prototype.asElement=function(){return this},t.prototype.buildWrapperNode=function(){var e=this.definition().universe().version();return"RAML10"==e?U.buildWrapperNode(this):"RAML08"==e?k.buildWrapperNode(this):null},t.prototype.propertiesAllowedToUse=function(){var e=this;return this.definition().allProperties().filter(function(t){return e.isAllowedToUse(t)})},t.prototype.isAllowedToUse=function(e){var t=this,n=!0;return e.getAdapter(O.RAMLPropertyService).isSystem()?!1:(e.getContextRequirements().forEach(function(e){if(-1!=e.name.indexOf("("))return!0;var r=t.computedValue(e.name);r?n=n&&r==e.value:e.value&&(n=!1)}),n)},t.prototype.allowRecursive=function(){return this.definition().getAdapter(O.RAMLService).isUserDefined()?!0:!1},t.prototype.setWrapperNode=function(e){this._wrapperNode=e},t.prototype.setAssociatedType=function(e){this._associatedDef=e},t.prototype.associatedType=function(){return this._associatedDef},t.prototype.knownIds=function(){return this.isAuxilary(),this._knownIds?this._knownIds:{}},t.prototype.findById=function(e){var t=this,n=this.root();if(n!=this)return n.findById(e);if(!this._knownIds){this._knownIds={};var r=N.allChildren(this);r.forEach(function(e){return t._knownIds[e.id()]=e})}if(this.isAuxilary()){if(!this._slaveIds){this._slaveIds={};var r=N.allChildren(this);r.forEach(function(e){return t._slaveIds[e.id()]=e})}var i=this._slaveIds[e];if(i)return i}return this._knownIds[e]},t.prototype.isAuxilary=function(){if(this._isAux)return!0;if(this._auxChecked)return!1;this._auxChecked=!0;var e=this.getMaster();return e?(this._isAux=!0,this.initilizeKnownIDs(e),!0):!1},t.prototype.initilizeKnownIDs=function(e){var t=this;this._knownIds={};var n=N.allChildren(e);n.forEach(function(e){return t._knownIds[e.id()]=e}),this._knownIds[""]=e},t.prototype.getMaster=function(){if(this.masterApi)return this.masterApi;var e=this.calculateMasterByRef();return e&&e.setSlave(this),e},t.prototype.overrideMaster=function(e){this.masterApi=e,this.resetAuxilaryState(),e&&e.setSlave(this)},t.prototype.setSlave=function(e){this.slave=e},t.prototype.setMergeMode=function(e){this.overlayMergeMode=e,this.resetAuxilaryState()},t.prototype.getMergeMode=function(){return this.overlayMergeMode},t.prototype.calculateMasterByRef=function(){var e=this.lowLevel().unit();if(!e)return null;var t=e.getMasterReferenceNode();if(!t||!t.value())return null;var n=this.lowLevel();if(n.master)return n.master;var r=t.value(),i=this.lowLevel().unit().project().resolve(this.lowLevel().unit().path(),r);if(!i)return null;var a=i.expandedHighLevel();return a.setMergeMode(this.overlayMergeMode),n.master=a,a},t.prototype.resetAuxilaryState=function(){this._isAux=!1,this._auxChecked=!1,this._knownIds=null,this.clearChildrenCache()},t.prototype.printDetails=function(e){var t="";e||(e="");var n=this.definition().nameId(),r=this.property()?this.property().range().nameId():"",i=this.property()?this.property().nameId():"";return t+=e+n+"["+r+"] <--- "+i+"\n",this.children().forEach(function(n){t+=n.printDetails(e+"	")}),t},t.prototype.testSerialize=function(e){var t="";e||(e="");var n=this.definition().nameId(),r=this.property()?this.property().nameId():"";return t+=e+n+" <-- "+r+"\n",this.children().forEach(function(n){n.testSerialize&&(t+=n.testSerialize(e+"  "))}),t},t.prototype.getExtractedChildren=function(){var e=this.root();if(e.isAuxilary()){if(e._knownIds){var t=e._knownIds[this.id()];if(t){var n=t.children();return n}}return[]}return[]},t.prototype.getMasterCounterPart=function(){var e=this.root();if(e.isAuxilary()){if(e._knownIds){var t=e._knownIds[this.id()];return t}return null}return null},t.prototype.getSlaveCounterPart=function(){var e=this.root();return e.slave?e.slave.findById(this.id()):null},t.prototype.getLastSlaveCounterPart=function(){var e=this.root(),t=e.slave;if(null==t)return null;for(;null!=t.slave;)t=t.slave;return""==this.id()?t:t.findById(this.id())},t.prototype.getExtractedLowLevelChildren=function(e){var t=this.root();if(t.isAuxilary()){if(t._knownLowLevelIds){var n=t._knownLowLevelIds[this.id()];if(n)return n.children()}return[]}return[]},t.prototype.allowsQuestion=function(){return this._allowQuestion||this.definition().getAdapter(O.RAMLService).getAllowQuestion()},t.prototype.findReferences=function(){var e=this,t=[];N.refFinder(this.root(),this,t),t.length>1&&(t=t.filter(function(t){return t!=e&&t.parent()!=e}));var n=[];return t.forEach(function(e){T.find(n,function(t){return t==e})||n.push(e)}),n},t.prototype.setNamePatch=function(e){this._patchedName=e},t.prototype.isNamePatch=function(){return this._patchedName},t.prototype.name=function(){if(this._patchedName)return this._patchedName;var t=T.find(this.directChildren(),function(e){return e.property()&&e.property().getAdapter(O.RAMLPropertyService).isKey()});if(t&&Y.isInstance(t)){var n=null,r=this.definition(),i=r.universe().version();if(r&&"RAML10"==i&&t.isFromKey()){var a=this._node.key();n=this._node.optional()?a+"?":a}else n=t.value();return n}return e.prototype.name.call(this)},t.prototype.findElementAtOffset=function(e){return this._findNode(this,e,e)},t.prototype.isElement=function(){return!0},t.prototype.universe=function(){return this._universe?this._universe:this.definition().universe()},t.prototype.setUniverse=function(e){this._universe=e},t.prototype._findNode=function(e,t,n){var r=this;if(null==e)return null;if(e.lowLevel()&&e.lowLevel().start()<=t&&e.lowLevel().end()>=n){var i=e;return e.elements().forEach(function(a){if(a.lowLevel().unit()==e.lowLevel().unit()){var o=r._findNode(a,t,n);o&&(i=o)}}),i}return null},t.prototype.isStub=function(){return!this.lowLevel().unit()||this.lowLevel().unit().isStubUnit()},t.prototype.add=function(e){w.addToNode(this,e)},t.prototype.remove=function(e){w.removeNodeFrom(this,e)},t.prototype.dump=function(e){return this._node.dump()},t.prototype.patchType=function(e){this._def=e;this._associatedDef;this._associatedDef=null,this._children=null,this._mergedChildren=null},t.prototype.children=function(){var e=this.lowLevel();return e&&e.isValueInclude&&e.isValueInclude()&&B.isWaitingFor(e.includePath())?(this._children=null,[]):this._children?this._mergedChildren?this._mergedChildren:(this._mergedChildren=this.mergeChildren(this._children,this.getExtractedChildren()),this._mergedChildren):this._node?(this._children=H.process(this,this._node.children()),this._children=this._children.filter(function(e){return null!=e}),this.mergeChildren(this._children,this.getExtractedChildren())):[]},t.prototype.mergeChildren=function(e,t){var n=this,r=this.root();if(r.overlayMergeMode==$.AGGREGATE)return e.concat(t);if(r.overlayMergeMode==$.MERGE){var i=[];return e.forEach(function(e){var r=T.find(t,function(t){return t.fullLocalId()==e.fullLocalId()});r?n.mergeChild(i,e,r):i.push(e)}),t.forEach(function(t){var n=T.find(e,function(e){return t.fullLocalId()==e.fullLocalId()});n||i.push(t)}),i}return null},t.prototype.mergeLowLevelChildren=function(e,t){var n=this,r=this.root();if(r.overlayMergeMode==$.AGGREGATE)return e.concat(t);if(r.overlayMergeMode==$.MERGE){var i=[];return e.forEach(function(e){var r=T.find(t,function(t){return t.key()==e.key()});r?n.mergeLowLevelChild(i,e,r):i.push(e)}),t.forEach(function(t){var n=T.find(e,function(e){return t.key()==e.key()});n||i.push(t)}),i}return null},t.prototype.mergeLowLevelChild=function(e,t,n){return t.kind()!=n.kind()?(e.push(t),void e.push(n)):void e.push(t)},t.prototype.mergeChild=function(e,t,n){return t.getKind()!=n.getKind()?(e.push(t),void e.push(n)):t.getKind()==E.NodeKind.NODE?void e.push(t):t.getKind()==E.NodeKind.ATTRIBUTE?void e.push(t):t.getKind()==E.NodeKind.BASIC?(e.push(t),void e.push(n)):void 0},t.prototype.directChildren=function(){return this._children?this._children:this._node?(this._children=H.process(this,this._node.children()),this._mergedChildren=null,this._children):[]},t.prototype.resetChildren=function(){this._children=null,this._mergedChildren=null},t.prototype.isEmptyRamlFile=function(){var e=this.lowLevel().root();return e.isScalar()},t.prototype.initRamlFile=function(){w.initEmptyRAMLFile(this)},t.prototype.createAttr=function(e,t){w.createAttr(this,e,t)},t.prototype.isAttr=function(){return!1},t.prototype.isUnknown=function(){return!1},t.prototype.value=function(){return this._node.value()},t.prototype.valuesOf=function(e){var t=this._def.property(e);return null!=t?this.elements().filter(function(e){return e.property()==t}):[]},t.prototype.attr=function(e){return T.find(this.attrs(),function(t){return t.name()==e})},t.prototype.attrOrCreate=function(e){var t=this.attr(e);return t||this.createAttr(e,""),this.attr(e)},t.prototype.attrValue=function(e){var t=this.attr(e);return t?t.value():null},t.prototype.attributes=function(e){return T.filter(this.attrs(),function(t){return t.name()==e})},t.prototype.attrs=function(){var e=this.children().filter(function(e){return e.isAttr()});if(this._patchedName){var t=T.find(this.definition().allProperties(),function(e){return e.getAdapter(O.RAMLPropertyService).isKey()});if(t){var n=new Y(this.lowLevel(),this,t.range(),t,!0);return n._value=this._patchedName,[n].concat(e)}}return e},t.prototype.elements=function(){return this.children().filter(function(e){return!e.isAttr()&&!e.isUnknown()})},t.prototype.element=function(e){var t=this.elementsOfKind(e);return t.length>0?t[0]:null},t.prototype.elementsOfKind=function(e){var t=this.elements().filter(function(t){return t.property().nameId()==e});return t},t.prototype.definition=function(){return this._def},t.prototype.property=function(){return this._prop},t.prototype.isExpanded=function(){return this._expanded},t.prototype.copy=function(){return new t(this.lowLevel().copy(),this.parent(),this.definition(),this.property())},t.prototype.clearChildrenCache=function(){this._children=null,this._mergedChildren=null},t.prototype.optionalProperties=function(){var e=this.definition();if(null==e)return[];var t=[],n={},r=this.lowLevel().children();r.forEach(function(e){e.optional()&&(n[e.key()]=!0)});var i=e.allProperties();return i.forEach(function(e){var r=e;n[r.nameId()]&&t.push(r.nameId())}),t},t.prototype.setReuseMode=function(e){this._reuseMode=e,this._children&&this.lowLevel().valueKind()!=D.Kind.SEQ&&this._children.forEach(function(t){return t.setReuseMode(e)})},t.prototype.reusedNode=function(){return this._reusedNode},t.prototype.setReusedNode=function(e){this._reusedNode=e},t.prototype.resetRuntimeTypes=function(){delete this._associatedDef,this.elements().forEach(function(e){return e.resetRuntimeTypes()})},t.CLASS_IDENTIFIER_ASTNodeImpl="highLevelImpl.ASTNodeImpl",t}(W);t.ASTNodeImpl=z,t.universeProvider=n(38);var J=function(e,n){var r=s(e),i=r&&r[1]||"",a=r&&r.length>2&&r[2]||"Api",o=r&&r.length>2&&r[2],u="1.0"==i?new b.Universe(null,"RAML10",t.universeProvider("RAML10"),"RAML10"):new b.Universe(null,"RAML08",t.universeProvider("RAML08"));return"API"==a?a="Api":"NamedExample"==a?a="ExampleSpec":"DataType"==a?a="TypeDeclaration":"SecurityScheme"==a?a="AbstractSecurityScheme":"AnnotationTypeDeclaration"==a&&(a="TypeDeclaration"),u.setOriginalTopLevelText(o),u.setTopLevel(a),u.setTypedVersion(i),{ptype:a,localUniverse:u}};t.ramlFirstLine=s,t.getFragmentDefenitionName=u,t.fromUnit=l,t.createBasicValidationAcceptor=p,t.isAnnotationTypeFragment=c;var Q=function(){function e(e){this._node=e}return e.prototype.kind=function(){return"AnnotatedNode"},e.prototype.annotationsMap=function(){var e=this;return this._annotationsMap||(this._annotationsMap={},this.annotations().forEach(function(t){var n=t.name(),r=n.lastIndexOf(".");r>=0&&(n=n.substring(r+1)),e._annotationsMap[n]=t})),this._annotationsMap},e.prototype.annotations=function(){if(!this._annotations){var e=[];this._node.isElement()?e=this._node.asElement().attributes(b.universesInfo.Universe10.Annotable.properties.annotations.name):this._node.isAttr()&&(e=this._node.asAttr().annotations()),this._annotations=e.map(function(e){return new Z(e)})}return this._annotations},e.prototype.value=function(){if(this._node.isElement())return this._node.asElement().wrapperNode().toJSON();if(this._node.isAttr()){var e=this._node.asAttr().value();return q.isInstance(e)?e.lowLevel().dump():e}return this._node.lowLevel().dump()},e.prototype.name=function(){return this._node.name()},e.prototype.entry=function(){return this._node},e}();t.AnnotatedNode=Q;var Z=function(){function e(e){this.attr=e}return e.prototype.name=function(){return this.attr.value().valueName()},e.prototype.value=function(){var e=this.attr.value();if(q.isInstance(e)){var t=e.lowLevel().dumpToObject(),n=Object.keys(t)[0];return t[n]}return e},e.prototype.definition=function(){var e=this.attr.parent(),t=this.name(),n=N.referenceTargets(this.attr.property(),e).filter(function(n){return r(n,e)==t});return 0==n.length?null:n[0].parsedType()},e}();t.AnnotationInstance=Z,t.applyNodeValidationPlugins=d,t.applyNodeAnnotationValidationPlugins=h,t.toParserErrors=m},function(e,t,n){"use strict";function r(e,t,n){var r=e.definition().property(t);if(!r)return null;var i=r.range(),a=d(i,r,n);return a}function i(e,t,n,r){var i=y.newMap(n.map(function(e){return y.newMapping(y.newScalar(e.key),y.newScalar(e.value))})),a=new h.ASTNode(i,r?r.lowLevel().unit():null,r?r.lowLevel():null,null,null);return new m.StructuredValue(a,r,r?r.definition().property(e):null,t)}function a(e,t,n){var r=e.definition().property(t);if(!r)return null;var i=r.range(),a=e.lowLevel().unit().stub(),o=d(i,r,n,a);return o.isInEdit=!0,o.lowLevel()._unit=a,o._parent=e.copy(),o._parent.lowLevel()._unit=a,o}function o(e,t){return a(e,"resources",t)}function s(e,t){return a(e,"methods",t)}function u(e,t){return a(e,"responses",t)}function l(e,t){return a(e,"body",t)}function p(e,t){return a(e,"uriParameters",t)}function c(e,t){return a(e,"queryParameters",t)}function f(e,t){var n=h.createMapping(e.nameId(),t),r=new m.ASTPropImpl(n,null,e.range(),e);return r}function d(e,t,n,r){void 0===n&&(n=null);var i=h.createNode(n?n:"key",null,r),a=new m.ASTNodeImpl(i,null,e,t);return i.unit()||(i._unit=r),a.children(),a}var h=n(24),m=n(16),y=n(77);t.createStub0=r,t.genStructuredValue=i,t.createStub=a,t.createResourceStub=o,t.createMethodStub=s,t.createResponseStub=u,t.createBodyStub=l,t.createUriParameterStub=p,t.createQueryParameterStub=c,t.createAttr=f,t.createStubNode=d},function(e,t,n){"use strict";var r=n(39);e.exports=r.universesInfo},function(e,t,n){"use strict";var r=n(70),i=n(43),a=n(21),o=function(){function e(){}return e.prototype.generateText=function(e){var t=this,n=JSON.parse(e),r=n.items;if(!r)return"";var i="",a=r instanceof Array?r:[r];return i+="types:\n",a.forEach(function(e){i+="  - "+e.title+":\n",i+=t.generateObj(e,3)}),i},e.prototype.generateObj=function(e,t){var n="";if(n+=i.indent(t,"type: "+e.type)+"\n",e.properties){n+=i.indent(t,"properties:\n");for(var r in e.properties){var a=e.properties[r];n+=i.indent(t+1,r+":\n"),n+="object"==a.type?this.generateObj(a,t+2):i.indent(t+2,"type: "+a.type)+"\n"}return n}},e.prototype.generateTo=function(e,t,n){var r=this,i=JSON.parse(t),o=i.items;if(o){var s=o instanceof Array?o:[o],u=[];return s.forEach(function(t){var n=new a.ObjectTypeDeclarationImpl(t.title);r.generateObjTo(n,t),new a.BasicSecuritySchemeImpl(e).addToProp(n,"types"),u.push(t.title)}),u}i.title&&(n=i.title);var l=new a.ObjectTypeDeclarationImpl(n);return this.generateObjTo(l,i),new a.BasicSecuritySchemeImpl(e).addToProp(l,"types"),[n]},e.prototype.generateObjTo=function(e,t){if(e.setType(t.type),t.properties)for(var n in t.properties){var r=t.properties[n],i=this.makeTypeField(n,r);"array"==r.type,e.addToProp(i,"properties")}},e.prototype.makeTypeField=function(e,t){var n=this.makeType(e,t.type);if(t.type&&n.setType(t.type),"number"==t.type){var r=n;void 0!=t.minimum&&r.setMinimum(t.minimum),void 0!=t.maximum&&r.setMaximum(t.maximum)}if("array"==t.type){var i=t.items.type;n.setType(i+"[]");var a=n;void 0!=t.minItems&&a.setMinItems(t.minItems),void 0!=t.maxItems&&a.setMaxItems(t.maxItems),void 0!=t.uniqueItems&&a.setUniqueItems(t.uniqueItems)}return"object"==t.type&&this.generateObjTo(n,t),n},e.prototype.makeType=function(e,t){return"number"==t?new a.NumberTypeDeclarationImpl(e):"string"==t?new a.StringTypeDeclarationImpl(e):"array"==t?new a.ArrayTypeDeclarationImpl(e):new a.ObjectTypeDeclarationImpl(e)},e.prototype.generateItemsTo=function(e,t){var n=t.items;if(n){var r=n instanceof Array?n:[n];r.forEach(function(e){})}},e}();t.SchemaToModelGenerator=o;var s=function(){function e(){}return e.prototype.generateSchema=function(e){var t=this.generateType(e);return t.$schema="http://json-schema.org/draft-04/schema#",t},e.prototype.isSimpleType=function(e){return"string"==e||"number"==e||"boolean"==e||"object"==e},e.prototype.generateType=function(e){var t=this.allTypes(e),n={};n.title=e.attrValue("name");if(e.attrValue("type")){var i=e.attributes("type"),a=!1,o=!1;for(var s in i){var u=i[s].value(),l=i[s].lowLevel();if(n.type="",l.isValueInclude()){var p=JSON.parse(u);n.type="object",n.properties=p.properties}else if(this.isSimpleType(u)){if(n.type=u,o=!0,a)throw new Error("couldn't mix user defined and basic types in inheritance")}else{var c=this.resolveType(e,u);if(c){var f=this.generateTypeExp(u,t);if(n.properties?r.extend(n.properties,f.properties):n.properties=f.properties,n.type="object",a=!0,o)throw new Error("couldn't mix user defined and basic types in inheritance")}else{var f=this.generateTypeExp(u,t);n.type="object",a=!0,f.anyOf&&(n.anyOf=f.anyOf)}}}}else n.type="object";var d=this.generateProperties(e);return n.properties?r.extend(n.properties,d):n.properties=d,n},e.prototype.makeUnion=function(e,t){var n=this,r=[];return e.forEach(function(e){e=e.trim(),t[e]?r.push({type:"object",properties:n.generateType(t[e]).properties}):r.push({type:e})}),r},e.prototype.generateTypeExp=function(e,t){var n={};if(i.endsWith(e,"[]"))n.type="array",n.items={type:e.substring(0,e.length-2)};else if(e.indexOf("|")>0){var r=e.split("|");n.anyOf=this.makeUnion(r,t)}else if(t[e]){var a=this.generateType(t[e]);n.type="object",n.properties=a.properties}else n.type=e;return n},e.prototype.allTypes=function(e){var t=e.root(),n=t.elementsOfKind("types"),r={};return n.forEach(function(e){r[e.name()]=e}),r},e.prototype.resolveType=function(e,t){var n=this.allTypes(e);return n[t]},e.prototype.generateProperty=function(e,t){var n=this,i=this.allTypes(e),a={},o=e.definition().allProperties();o.forEach(function(t){if("name"!=t.nameId()){var o=e.attrValue(t.nameId());if(null!=o&&void 0!=o&&"undefined"!=o)if("type"==t.nameId()){var s=n.generateTypeExp(o,i);r.extend(a,s)}else if("enum"==t.nameId()){var u=e.attributes("enum"),l=u.map(function(e){return e.value()});a["enum"]=l}else a[t.nameId()]=o}}),t&&(a.required=!1);var s=(e.elements(),this.generateProperties(e));return Object.getOwnPropertyNames(s).length>0&&(a.properties=s),a},e.prototype.generateProperties=function(e){var t=this,n={},r=e.elements(),i=!0;return r.forEach(function(e){var r=e.attrValue("name");if("string"==typeof r){r=r.trim();var a=e.optional();n[r]=t.generateProperty(e,a),i=!1}}),n},e}();t.ModelToSchemaGenerator=s},function(e,t,n){"use strict";function r(e){return new u.TypeDeclarationImpl(e)}function i(e){return new u.ObjectTypeDeclarationImpl(e)}function a(e,t){e.setSchema(t)}function o(e,t){var n=(l.getUniverse("RAML10").type(l.universesInfo.Universe10.ExampleSpec.name),l.universesInfo.Universe10.TypeDeclaration.properties.example.name),r=e.highLevel(),i=r.lowLevel(),a=r.children().filter(function(e){return e.lowLevel().key()==n}),o=p.createNode(n);c.setAttr(o,t),a.length>0&&(c.removeNode(i,a[0].lowLevel()),a[0]._node=o,c.insertNode(i,o)),r.createAttr&&r.createAttr(n,t)}function s(e,t){e.add(t)}var u=n(21),l=n(39),p=n(24),c=n(10);t.createTypeDeclaration=r,t.createObjectTypeDeclaration=i,t.setTypeDeclarationSchema=a,t.setTypeDeclarationExample=o,t.addChild=s},function(e,t,n){"use strict";function r(e){var t=J.getUniverse("RAML10"),n=t.type("Api"),r=z.createStubNode(n,null,e);return r}function i(e){var t=J.getUniverse("RAML10"),n=t.type("LibraryBase"),r=z.createStubNode(n,null,e);return r}function a(e){var t=J.getUniverse("RAML10"),n=t.type("FragmentDeclaration"),r=z.createStubNode(n,null,e);return r}function o(e){var t=J.getUniverse("RAML10"),n=t.type("Trait"),r=z.createStubNode(n,null,e);return r}function s(e){var t=J.getUniverse("RAML10"),n=t.type("MethodBase"),r=z.createStubNode(n,null,e);return r}function u(e){var t=J.getUniverse("RAML10"),n=t.type("Operation"),r=z.createStubNode(n,null,e);return r}function l(e){var t=J.getUniverse("RAML10"),n=t.type("TypeDeclaration"),r=z.createStubNode(n,null,e);return r}function p(e){var t=J.getUniverse("RAML10"),n=t.type("UsesDeclaration"),r=z.createStubNode(n,null,e);return r}function c(e){var t=J.getUniverse("RAML10"),n=t.type("XMLFacetInfo"),r=z.createStubNode(n,null,e);return r}function f(e){var t=J.getUniverse("RAML10"),n=t.type("ArrayTypeDeclaration"),r=z.createStubNode(n,null,e);return r}function d(e){var t=J.getUniverse("RAML10"),n=t.type("UnionTypeDeclaration"),r=z.createStubNode(n,null,e);return r}function h(e){var t=J.getUniverse("RAML10"),n=t.type("ObjectTypeDeclaration"),r=z.createStubNode(n,null,e);return r}function m(e){var t=J.getUniverse("RAML10"),n=t.type("StringTypeDeclaration"),r=z.createStubNode(n,null,e);return r}function y(e){var t=J.getUniverse("RAML10"),n=t.type("BooleanTypeDeclaration"),r=z.createStubNode(n,null,e);return r}function v(e){var t=J.getUniverse("RAML10"),n=t.type("NumberTypeDeclaration"),r=z.createStubNode(n,null,e);return r}function g(e){var t=J.getUniverse("RAML10"),n=t.type("IntegerTypeDeclaration"),r=z.createStubNode(n,null,e);return r}function A(e){var t=J.getUniverse("RAML10"),n=t.type("DateOnlyTypeDeclaration"),r=z.createStubNode(n,null,e);return r}function E(e){var t=J.getUniverse("RAML10"),n=t.type("TimeOnlyTypeDeclaration"),r=z.createStubNode(n,null,e);return r}function T(e){var t=J.getUniverse("RAML10"),n=t.type("DateTimeOnlyTypeDeclaration"),r=z.createStubNode(n,null,e);return r}function S(e){var t=J.getUniverse("RAML10"),n=t.type("DateTimeTypeDeclaration"),r=z.createStubNode(n,null,e);
return r}function b(e){var t=J.getUniverse("RAML10"),n=t.type("FileTypeDeclaration"),r=z.createStubNode(n,null,e);return r}function _(e){var t=J.getUniverse("RAML10"),n=t.type("Response"),r=z.createStubNode(n,null,e);return r}function N(e){var t=J.getUniverse("RAML10"),n=t.type("SecuritySchemePart"),r=z.createStubNode(n,null,e);return r}function w(e){var t=J.getUniverse("RAML10"),n=t.type("AbstractSecurityScheme"),r=z.createStubNode(n,null,e);return r}function R(e){var t=J.getUniverse("RAML10"),n=t.type("SecuritySchemeSettings"),r=z.createStubNode(n,null,e);return r}function I(e){var t=J.getUniverse("RAML10"),n=t.type("OAuth1SecuritySchemeSettings"),r=z.createStubNode(n,null,e);return r}function M(e){var t=J.getUniverse("RAML10"),n=t.type("OAuth2SecuritySchemeSettings"),r=z.createStubNode(n,null,e);return r}function C(e){var t=J.getUniverse("RAML10"),n=t.type("OAuth2SecurityScheme"),r=z.createStubNode(n,null,e);return r}function L(e){var t=J.getUniverse("RAML10"),n=t.type("OAuth1SecurityScheme"),r=z.createStubNode(n,null,e);return r}function P(e){var t=J.getUniverse("RAML10"),n=t.type("PassThroughSecurityScheme"),r=z.createStubNode(n,null,e);return r}function O(e){var t=J.getUniverse("RAML10"),n=t.type("BasicSecurityScheme"),r=z.createStubNode(n,null,e);return r}function D(e){var t=J.getUniverse("RAML10"),n=t.type("DigestSecurityScheme"),r=z.createStubNode(n,null,e);return r}function x(e){var t=J.getUniverse("RAML10"),n=t.type("CustomSecurityScheme"),r=z.createStubNode(n,null,e);return r}function U(e){var t=J.getUniverse("RAML10"),n=t.type("Method"),r=z.createStubNode(n,null,e);return r}function k(e){var t=J.getUniverse("RAML10"),n=t.type("ResourceType"),r=z.createStubNode(n,null,e);return r}function F(e){var t=J.getUniverse("RAML10"),n=t.type("ResourceBase"),r=z.createStubNode(n,null,e);return r}function B(e){var t=J.getUniverse("RAML10"),n=t.type("Resource"),r=z.createStubNode(n,null,e);return r}function K(e){var t=J.getUniverse("RAML10"),n=t.type("DocumentationItem"),r=z.createStubNode(n,null,e);return r}function V(e){var t=J.getUniverse("RAML10"),n=t.type("Library"),r=z.createStubNode(n,null,e);return r}function j(e){var t=J.getUniverse("RAML10"),n=t.type("Overlay"),r=z.createStubNode(n,null,e);return r}function W(e){var t=J.getUniverse("RAML10"),n=t.type("Extension"),r=z.createStubNode(n,null,e);return r}function q(e,t,n){return Z.loadApi(e,t,n).getOrElse(null)}function Y(e,t,n){return Z.loadApi(e,t,n).getOrElse(null)}function H(e,t,n){return Z.loadApiAsync(e,t,n)}function $(e,t,n){return Z.loadRAMLAsync(e,t,n)}function G(e){return Z.getLanguageElementByRuntimeType(e)}var X=this&&this.__extends||function(e,t){function n(){this.constructor=e}for(var r in t)t.hasOwnProperty(r)&&(e[r]=t[r]);e.prototype=null===t?Object.create(t):(n.prototype=t.prototype,new n)},z=n(17),J=n(39),Q=n(51),Z=n(8),ee=n(34),te=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.annotations=function(){return e.prototype.attributes.call(this,"annotations",function(e){return new Lt(e)})},t.prototype.wrapperClassName=function(){return"AnnotableImpl"},t.prototype.kind=function(){return"Annotable"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("Annotable")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.AnnotableImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.AnnotableImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(Q.BasicNodeImpl);t.AnnotableImpl=te;var ne=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"ValueTypeImpl"},t.prototype.kind=function(){return"ValueType"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("ValueType")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("undefined.ValueTypeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("undefined.ValueTypeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"undefined"},t.prototype.value=function(){return this.attr.value()},t}(Q.AttributeNodeImpl);t.ValueTypeImpl=ne;var re=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"StringTypeImpl"},t.prototype.kind=function(){return"StringType"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("StringType")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.StringTypeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.StringTypeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.value=function(){return this.attr.value()},t}(ne);t.StringTypeImpl=re;var ie=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"UriTemplateImpl"},t.prototype.kind=function(){return"UriTemplate"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("UriTemplate")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.UriTemplateImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.UriTemplateImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(re);t.UriTemplateImpl=ie;var ae=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"RelativeUriStringImpl"},t.prototype.kind=function(){return"RelativeUriString"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("RelativeUriString")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.RelativeUriStringImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.RelativeUriStringImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(ie);t.RelativeUriStringImpl=ae;var oe=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"FullUriTemplateStringImpl"},t.prototype.kind=function(){return"FullUriTemplateString"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("FullUriTemplateString")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.FullUriTemplateStringImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.FullUriTemplateStringImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(ie);t.FullUriTemplateStringImpl=oe;var se=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"StatusCodeStringImpl"},t.prototype.kind=function(){return"StatusCodeString"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("StatusCodeString")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.StatusCodeStringImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.StatusCodeStringImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(re);t.StatusCodeStringImpl=se;var ue=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"FixedUriStringImpl"},t.prototype.kind=function(){return"FixedUriString"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("FixedUriString")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.FixedUriStringImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.FixedUriStringImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(re);t.FixedUriStringImpl=ue;var le=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"ContentTypeImpl"},t.prototype.kind=function(){return"ContentType"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("ContentType")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.ContentTypeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.ContentTypeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(re);t.ContentTypeImpl=le;var pe=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"MarkdownStringImpl"},t.prototype.kind=function(){return"MarkdownString"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("MarkdownString")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.MarkdownStringImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.MarkdownStringImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(re);t.MarkdownStringImpl=pe;var ce=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"SchemaStringImpl"},t.prototype.kind=function(){return"SchemaString"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("SchemaString")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.SchemaStringImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.SchemaStringImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(re);t.SchemaStringImpl=ce;var fe=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"MimeTypeImpl"},t.prototype.kind=function(){return"MimeType"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("MimeType")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.MimeTypeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.MimeTypeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(re);t.MimeTypeImpl=fe;var de=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"AnyTypeImpl"},t.prototype.kind=function(){return"AnyType"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("AnyType")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.AnyTypeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.AnyTypeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(ne);t.AnyTypeImpl=de;var he=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"NumberTypeImpl"},t.prototype.kind=function(){return"NumberType"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("NumberType")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.NumberTypeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.NumberTypeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.value=function(){return this.attr.value()},t}(ne);t.NumberTypeImpl=he;var me=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"IntegerTypeImpl"},t.prototype.kind=function(){return"IntegerType"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("IntegerType")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.IntegerTypeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.IntegerTypeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(ne);t.IntegerTypeImpl=me;var ye=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"NullTypeImpl"},t.prototype.kind=function(){return"NullType"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("NullType")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.NullTypeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.NullTypeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(ne);t.NullTypeImpl=ye;var ve=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"TimeOnlyTypeImpl"},t.prototype.kind=function(){return"TimeOnlyType"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("TimeOnlyType")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.TimeOnlyTypeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.TimeOnlyTypeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(ne);t.TimeOnlyTypeImpl=ve;var ge=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"DateOnlyTypeImpl"},t.prototype.kind=function(){return"DateOnlyType"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("DateOnlyType")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.DateOnlyTypeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.DateOnlyTypeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(ne);t.DateOnlyTypeImpl=ge;var Ae=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"DateTimeOnlyTypeImpl"},t.prototype.kind=function(){return"DateTimeOnlyType"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("DateTimeOnlyType")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.DateTimeOnlyTypeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.DateTimeOnlyTypeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(ne);t.DateTimeOnlyTypeImpl=Ae;var Ee=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"DateTimeTypeImpl"},t.prototype.kind=function(){return"DateTimeType"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("DateTimeType")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.DateTimeTypeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.DateTimeTypeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(ne);t.DateTimeTypeImpl=Ee;var Te=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"FileTypeImpl"},t.prototype.kind=function(){return"FileType"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("FileType")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.FileTypeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.FileTypeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(ne);t.FileTypeImpl=Te;var Se=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"BooleanTypeImpl"},t.prototype.kind=function(){return"BooleanType"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("BooleanType")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.BooleanTypeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.BooleanTypeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.value=function(){return this.attr.value()},t}(ne);t.BooleanTypeImpl=Se;var be=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"AnnotationTargetImpl"},t.prototype.kind=function(){return"AnnotationTarget"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("AnnotationTarget")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.AnnotationTargetImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.AnnotationTargetImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(ne);t.AnnotationTargetImpl=be;var _e=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"ReferenceImpl"},t.prototype.kind=function(){return"Reference"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("Reference")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.ReferenceImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.ReferenceImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.value=function(){return Q.toStructuredValue(this.attr)},t.prototype.structuredValue=function(){return ee.structuredValue(this)},t.prototype.name=function(){return ee.referenceName(this)},t}(Q.AttributeNodeImpl);t.ReferenceImpl=_e;var Ne=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"TraitRefImpl"},t.prototype.kind=function(){return"TraitRef"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("TraitRef")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.TraitRefImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.TraitRefImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.trait=function(){return ee.referencedTrait(this)},t}(_e);t.TraitRefImpl=Ne;var we=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?u(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.queryParameters=function(){return e.prototype.elements.call(this,"queryParameters")},t.prototype.headers=function(){return e.prototype.elements.call(this,"headers")},t.prototype.queryString=function(){return e.prototype.element.call(this,"queryString")},t.prototype.responses=function(){return e.prototype.elements.call(this,"responses")},t.prototype.wrapperClassName=function(){return"OperationImpl"},t.prototype.kind=function(){return"Operation"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("Operation")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.OperationImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.OperationImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(te);t.OperationImpl=we;var Re=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?l(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.name=function(){return e.prototype.attribute.call(this,"name",this.toString)},t.prototype.setName=function(e){return this.highLevel().attrOrCreate("name").setValue(""+e),this},t.prototype.displayName=function(){return e.prototype.attribute.call(this,"displayName",this.toString)},t.prototype.setDisplayName=function(e){return this.highLevel().attrOrCreate("displayName").setValue(""+e),this},t.prototype.facets=function(){return e.prototype.elements.call(this,"facets")},t.prototype.schema_original=function(){return e.prototype.attributes.call(this,"schema",this.toString)},t.prototype.setSchema=function(e){return this.highLevel().attrOrCreate("schema").setValue(""+e),this},t.prototype.type_original=function(){return e.prototype.attributes.call(this,"type",this.toString)},t.prototype.setType=function(e){return this.highLevel().attrOrCreate("type").setValue(""+e),this},t.prototype.location=function(){return e.prototype.attribute.call(this,"location",function(e){return new Ie(e)})},t.prototype.locationKind=function(){return e.prototype.attribute.call(this,"locationKind",function(e){return new Me(e)})},t.prototype["default"]=function(){return e.prototype.attribute.call(this,"default",this.toAny)},t.prototype.setDefault=function(e){return this.highLevel().attrOrCreate("default").setValue(""+e),this},t.prototype.example_original=function(){return e.prototype.element.call(this,"example")},t.prototype.examples_original=function(){return e.prototype.elements.call(this,"examples")},t.prototype.required=function(){return e.prototype.attribute.call(this,"required",this.toBoolean)},t.prototype.setRequired=function(e){return this.highLevel().attrOrCreate("required").setValue(""+e),this},t.prototype.description=function(){return e.prototype.attribute.call(this,"description",function(e){return new pe(e)})},t.prototype.xml=function(){return e.prototype.element.call(this,"xml")},t.prototype.allowedTargets=function(){return e.prototype.attributes.call(this,"allowedTargets",function(e){return new be(e)})},t.prototype.isAnnotation=function(){return e.prototype.attribute.call(this,"isAnnotation",this.toBoolean)},t.prototype.setIsAnnotation=function(e){return this.highLevel().attrOrCreate("isAnnotation").setValue(""+e),this},t.prototype.annotations=function(){return e.prototype.attributes.call(this,"annotations",function(e){return new Lt(e)})},t.prototype.uses=function(){return e.prototype.elements.call(this,"uses")},t.prototype.wrapperClassName=function(){return"TypeDeclarationImpl"},t.prototype.kind=function(){return"TypeDeclaration"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("TypeDeclaration")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.TypeDeclarationImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.TypeDeclarationImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.runtimeType=function(){return ee.runtimeType(this)},t.prototype.validateInstance=function(e){return ee.validateInstance(this,e)},t.prototype.validateInstanceWithDetailedStatuses=function(e){return ee.validateInstanceWithDetailedStatuses(this,e)},t.prototype.example=function(){return ee.getTypeExample(this)},t.prototype.examples=function(){return ee.getTypeExamples(this)},t.prototype.fixedFacets=function(){return ee.typeFixedFacets(this)},t.prototype.type=function(){return ee.typeValue(this)},t.prototype.schema=function(){return ee.schemaValue(this)},t.prototype.structuredType=function(){return ee.typeStructuredValue(this)},t.prototype.parametrizedProperties=function(){return ee.getTemplateParametrizedProperties(this)},t.prototype.scalarsAnnotations=function(){return new xe(this.highLevel())},t}(te);t.TypeDeclarationImpl=Re;var Ie=function(){function e(e){this.attr=e}return e.prototype.wrapperClassName=function(){return"ModelLocationImpl"},e.prototype.kind=function(){return"ModelLocation"},e.prototype.allKinds=function(){return["ModelLocation"]},e.prototype.allWrapperClassNames=function(){return["RAML10.ModelLocationImpl"]},e.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.ModelLocationImpl"==r)return!0}return!1},e.prototype.RAMLVersion=function(){return"RAML10"},e}();t.ModelLocationImpl=Ie;var Me=function(){function e(e){this.attr=e}return e.prototype.wrapperClassName=function(){return"LocationKindImpl"},e.prototype.kind=function(){return"LocationKind"},e.prototype.allKinds=function(){return["LocationKind"]},e.prototype.allWrapperClassNames=function(){return["RAML10.LocationKindImpl"]},e.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.LocationKindImpl"==r)return!0}return!1},e.prototype.RAMLVersion=function(){return"RAML10"},e}();t.LocationKindImpl=Me;var Ce=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?p(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.key=function(){return e.prototype.attribute.call(this,"key",this.toString)},t.prototype.setKey=function(e){return this.highLevel().attrOrCreate("key").setValue(""+e),this},t.prototype.value=function(){return e.prototype.attribute.call(this,"value",this.toString)},t.prototype.setValue=function(e){return this.highLevel().attrOrCreate("value").setValue(""+e),this},t.prototype.wrapperClassName=function(){return"UsesDeclarationImpl"},t.prototype.kind=function(){return"UsesDeclaration"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("UsesDeclaration")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.UsesDeclarationImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.UsesDeclarationImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.ast=function(){return ee.referencedNode(this)},t.prototype.scalarsAnnotations=function(){return new Le(this.highLevel())},t}(te);t.UsesDeclarationImpl=Ce;var Le=function(){function e(e){this.node=e}return e.prototype.value=function(){var e=this.node.attr("value");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e}();t.UsesDeclarationScalarsAnnotationsImpl=Le;var Pe=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?c(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.attribute=function(){return e.prototype.attribute.call(this,"attribute",this.toBoolean)},t.prototype.setAttribute=function(e){return this.highLevel().attrOrCreate("attribute").setValue(""+e),this},t.prototype.wrapped=function(){return e.prototype.attribute.call(this,"wrapped",this.toBoolean)},t.prototype.setWrapped=function(e){return this.highLevel().attrOrCreate("wrapped").setValue(""+e),this},t.prototype.name=function(){return e.prototype.attribute.call(this,"name",this.toString)},t.prototype.setName=function(e){return this.highLevel().attrOrCreate("name").setValue(""+e),this},t.prototype.namespace=function(){return e.prototype.attribute.call(this,"namespace",this.toString)},t.prototype.setNamespace=function(e){return this.highLevel().attrOrCreate("namespace").setValue(""+e),this},t.prototype.prefix=function(){return e.prototype.attribute.call(this,"prefix",this.toString)},t.prototype.setPrefix=function(e){return this.highLevel().attrOrCreate("prefix").setValue(""+e),this},t.prototype.wrapperClassName=function(){return"XMLFacetInfoImpl"},t.prototype.kind=function(){return"XMLFacetInfo"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("XMLFacetInfo")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.XMLFacetInfoImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.XMLFacetInfoImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.scalarsAnnotations=function(){return new Oe(this.highLevel())},t}(te);t.XMLFacetInfoImpl=Pe;var Oe=function(){function e(e){this.node=e}return e.prototype.attribute=function(){var e=this.node.attr("attribute");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e.prototype.wrapped=function(){var e=this.node.attr("wrapped");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e.prototype.name=function(){var e=this.node.attr("name");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e.prototype.namespace=function(){var e=this.node.attr("namespace");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e.prototype.prefix=function(){var e=this.node.attr("prefix");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e}();t.XMLFacetInfoScalarsAnnotationsImpl=Oe;
var De=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?f(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.uniqueItems=function(){return e.prototype.attribute.call(this,"uniqueItems",this.toBoolean)},t.prototype.setUniqueItems=function(e){return this.highLevel().attrOrCreate("uniqueItems").setValue(""+e),this},t.prototype.items_original=function(){return e.prototype.attributes.call(this,"items",this.toString)},t.prototype.setItems=function(e){return this.highLevel().attrOrCreate("items").setValue(""+e),this},t.prototype.minItems=function(){return e.prototype.attribute.call(this,"minItems",this.toNumber)},t.prototype.setMinItems=function(e){return this.highLevel().attrOrCreate("minItems").setValue(""+e),this},t.prototype.maxItems=function(){return e.prototype.attribute.call(this,"maxItems",this.toNumber)},t.prototype.setMaxItems=function(e){return this.highLevel().attrOrCreate("maxItems").setValue(""+e),this},t.prototype.wrapperClassName=function(){return"ArrayTypeDeclarationImpl"},t.prototype.kind=function(){return"ArrayTypeDeclaration"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("ArrayTypeDeclaration")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.ArrayTypeDeclarationImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.ArrayTypeDeclarationImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.structuredItems=function(){return ee.itemsStructuredValue(this)},t.prototype.items=function(){return ee.getItems(this)},t.prototype.findComponentTypeDeclaration=function(){return ee.findComponentTypeDeclaration(this)},t.prototype.scalarsAnnotations=function(){return new Ue(this.highLevel())},t}(Re);t.ArrayTypeDeclarationImpl=De;var xe=function(){function e(e){this.node=e}return e.prototype.displayName=function(){var e=this.node.attr("displayName");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e.prototype.schema=function(){var e=this.node.attributes("schema");return e.map(function(e){var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n})},e.prototype.type=function(){var e=this.node.attributes("type");return e.map(function(e){var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n})},e.prototype.location=function(){var e=this.node.attr("location");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e.prototype.locationKind=function(){var e=this.node.attr("locationKind");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e.prototype["default"]=function(){var e=this.node.attr("default");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e.prototype.required=function(){var e=this.node.attr("required");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e.prototype.description=function(){var e=this.node.attr("description");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e.prototype.allowedTargets=function(){var e=this.node.attributes("allowedTargets");return e.map(function(e){var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n})},e.prototype.isAnnotation=function(){var e=this.node.attr("isAnnotation");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e}();t.TypeDeclarationScalarsAnnotationsImpl=xe;var Ue=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.uniqueItems=function(){var e=this.node.attr("uniqueItems");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t.prototype.items=function(){var e=this.node.attributes("items");return e.map(function(e){var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n})},t.prototype.minItems=function(){var e=this.node.attr("minItems");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t.prototype.maxItems=function(){var e=this.node.attr("maxItems");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t}(xe);t.ArrayTypeDeclarationScalarsAnnotationsImpl=Ue;var ke=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?d(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.wrapperClassName=function(){return"UnionTypeDeclarationImpl"},t.prototype.kind=function(){return"UnionTypeDeclaration"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("UnionTypeDeclaration")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.UnionTypeDeclarationImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.UnionTypeDeclarationImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(Re);t.UnionTypeDeclarationImpl=ke;var Fe=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?h(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.properties=function(){return e.prototype.elements.call(this,"properties")},t.prototype.minProperties=function(){return e.prototype.attribute.call(this,"minProperties",this.toNumber)},t.prototype.setMinProperties=function(e){return this.highLevel().attrOrCreate("minProperties").setValue(""+e),this},t.prototype.maxProperties=function(){return e.prototype.attribute.call(this,"maxProperties",this.toNumber)},t.prototype.setMaxProperties=function(e){return this.highLevel().attrOrCreate("maxProperties").setValue(""+e),this},t.prototype.additionalProperties=function(){return e.prototype.attribute.call(this,"additionalProperties",this.toBoolean)},t.prototype.setAdditionalProperties=function(e){return this.highLevel().attrOrCreate("additionalProperties").setValue(""+e),this},t.prototype.discriminator=function(){return e.prototype.attribute.call(this,"discriminator",this.toString)},t.prototype.setDiscriminator=function(e){return this.highLevel().attrOrCreate("discriminator").setValue(""+e),this},t.prototype.discriminatorValue=function(){return e.prototype.attribute.call(this,"discriminatorValue",this.toString)},t.prototype.setDiscriminatorValue=function(e){return this.highLevel().attrOrCreate("discriminatorValue").setValue(""+e),this},t.prototype["enum"]=function(){return e.prototype.attributes.call(this,"enum",this.toAny)},t.prototype.setEnum=function(e){return this.highLevel().attrOrCreate("enum").setValue(""+e),this},t.prototype.wrapperClassName=function(){return"ObjectTypeDeclarationImpl"},t.prototype.kind=function(){return"ObjectTypeDeclaration"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("ObjectTypeDeclaration")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.ObjectTypeDeclarationImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.ObjectTypeDeclarationImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.scalarsAnnotations=function(){return new Be(this.highLevel())},t}(Re);t.ObjectTypeDeclarationImpl=Fe;var Be=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.minProperties=function(){var e=this.node.attr("minProperties");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t.prototype.maxProperties=function(){var e=this.node.attr("maxProperties");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t.prototype.additionalProperties=function(){var e=this.node.attr("additionalProperties");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t.prototype.discriminator=function(){var e=this.node.attr("discriminator");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t.prototype.discriminatorValue=function(){var e=this.node.attr("discriminatorValue");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t.prototype["enum"]=function(){var e=this.node.attributes("enum");return e.map(function(e){var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n})},t}(xe);t.ObjectTypeDeclarationScalarsAnnotationsImpl=Be;var Ke=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?m(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.pattern=function(){return e.prototype.attribute.call(this,"pattern",this.toString)},t.prototype.setPattern=function(e){return this.highLevel().attrOrCreate("pattern").setValue(""+e),this},t.prototype.minLength=function(){return e.prototype.attribute.call(this,"minLength",this.toNumber)},t.prototype.setMinLength=function(e){return this.highLevel().attrOrCreate("minLength").setValue(""+e),this},t.prototype.maxLength=function(){return e.prototype.attribute.call(this,"maxLength",this.toNumber)},t.prototype.setMaxLength=function(e){return this.highLevel().attrOrCreate("maxLength").setValue(""+e),this},t.prototype["enum"]=function(){return e.prototype.attributes.call(this,"enum",this.toString)},t.prototype.setEnum=function(e){return this.highLevel().attrOrCreate("enum").setValue(""+e),this},t.prototype.wrapperClassName=function(){return"StringTypeDeclarationImpl"},t.prototype.kind=function(){return"StringTypeDeclaration"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("StringTypeDeclaration")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.StringTypeDeclarationImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.StringTypeDeclarationImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.scalarsAnnotations=function(){return new Ve(this.highLevel())},t}(Re);t.StringTypeDeclarationImpl=Ke;var Ve=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.pattern=function(){var e=this.node.attr("pattern");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t.prototype.minLength=function(){var e=this.node.attr("minLength");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t.prototype.maxLength=function(){var e=this.node.attr("maxLength");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t.prototype["enum"]=function(){var e=this.node.attributes("enum");return e.map(function(e){var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n})},t}(xe);t.StringTypeDeclarationScalarsAnnotationsImpl=Ve;var je=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?y(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype["enum"]=function(){return e.prototype.attributes.call(this,"enum",this.toBoolean)},t.prototype.setEnum=function(e){return this.highLevel().attrOrCreate("enum").setValue(""+e),this},t.prototype.wrapperClassName=function(){return"BooleanTypeDeclarationImpl"},t.prototype.kind=function(){return"BooleanTypeDeclaration"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("BooleanTypeDeclaration")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.BooleanTypeDeclarationImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.BooleanTypeDeclarationImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.scalarsAnnotations=function(){return new We(this.highLevel())},t}(Re);t.BooleanTypeDeclarationImpl=je;var We=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype["enum"]=function(){var e=this.node.attributes("enum");return e.map(function(e){var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n})},t}(xe);t.BooleanTypeDeclarationScalarsAnnotationsImpl=We;var qe=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?v(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.minimum=function(){return e.prototype.attribute.call(this,"minimum",this.toNumber)},t.prototype.setMinimum=function(e){return this.highLevel().attrOrCreate("minimum").setValue(""+e),this},t.prototype.maximum=function(){return e.prototype.attribute.call(this,"maximum",this.toNumber)},t.prototype.setMaximum=function(e){return this.highLevel().attrOrCreate("maximum").setValue(""+e),this},t.prototype["enum"]=function(){return e.prototype.attributes.call(this,"enum",this.toNumber)},t.prototype.setEnum=function(e){return this.highLevel().attrOrCreate("enum").setValue(""+e),this},t.prototype.format=function(){return e.prototype.attribute.call(this,"format",this.toString)},t.prototype.setFormat=function(e){return this.highLevel().attrOrCreate("format").setValue(""+e),this},t.prototype.multipleOf=function(){return e.prototype.attribute.call(this,"multipleOf",this.toNumber)},t.prototype.setMultipleOf=function(e){return this.highLevel().attrOrCreate("multipleOf").setValue(""+e),this},t.prototype.wrapperClassName=function(){return"NumberTypeDeclarationImpl"},t.prototype.kind=function(){return"NumberTypeDeclaration"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("NumberTypeDeclaration")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.NumberTypeDeclarationImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.NumberTypeDeclarationImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.scalarsAnnotations=function(){return new He(this.highLevel())},t}(Re);t.NumberTypeDeclarationImpl=qe;var Ye=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?g(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.format=function(){return e.prototype.attribute.call(this,"format",this.toString)},t.prototype.setFormat=function(e){return this.highLevel().attrOrCreate("format").setValue(""+e),this},t.prototype.wrapperClassName=function(){return"IntegerTypeDeclarationImpl"},t.prototype.kind=function(){return"IntegerTypeDeclaration"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("IntegerTypeDeclaration")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.IntegerTypeDeclarationImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.IntegerTypeDeclarationImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.scalarsAnnotations=function(){return new $e(this.highLevel())},t}(qe);t.IntegerTypeDeclarationImpl=Ye;var He=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.minimum=function(){var e=this.node.attr("minimum");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t.prototype.maximum=function(){var e=this.node.attr("maximum");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t.prototype["enum"]=function(){var e=this.node.attributes("enum");return e.map(function(e){var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n})},t.prototype.format=function(){var e=this.node.attr("format");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t.prototype.multipleOf=function(){var e=this.node.attr("multipleOf");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t}(xe);t.NumberTypeDeclarationScalarsAnnotationsImpl=He;var $e=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.format=function(){var e=this.node.attr("format");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t}(He);t.IntegerTypeDeclarationScalarsAnnotationsImpl=$e;var Ge=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?A(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.wrapperClassName=function(){return"DateOnlyTypeDeclarationImpl"},t.prototype.kind=function(){return"DateOnlyTypeDeclaration"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("DateOnlyTypeDeclaration")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.DateOnlyTypeDeclarationImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.DateOnlyTypeDeclarationImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(Re);t.DateOnlyTypeDeclarationImpl=Ge;var Xe=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?E(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.wrapperClassName=function(){return"TimeOnlyTypeDeclarationImpl"},t.prototype.kind=function(){return"TimeOnlyTypeDeclaration"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("TimeOnlyTypeDeclaration")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.TimeOnlyTypeDeclarationImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.TimeOnlyTypeDeclarationImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(Re);t.TimeOnlyTypeDeclarationImpl=Xe;var ze=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?T(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.wrapperClassName=function(){return"DateTimeOnlyTypeDeclarationImpl"},t.prototype.kind=function(){return"DateTimeOnlyTypeDeclaration"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("DateTimeOnlyTypeDeclaration")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.DateTimeOnlyTypeDeclarationImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.DateTimeOnlyTypeDeclarationImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(Re);t.DateTimeOnlyTypeDeclarationImpl=ze;var Je=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?S(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.format=function(){return e.prototype.attribute.call(this,"format",this.toString)},t.prototype.setFormat=function(e){return this.highLevel().attrOrCreate("format").setValue(""+e),this},t.prototype.wrapperClassName=function(){return"DateTimeTypeDeclarationImpl"},t.prototype.kind=function(){return"DateTimeTypeDeclaration"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("DateTimeTypeDeclaration")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.DateTimeTypeDeclarationImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.DateTimeTypeDeclarationImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.scalarsAnnotations=function(){return new Qe(this.highLevel())},t}(Re);t.DateTimeTypeDeclarationImpl=Je;var Qe=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.format=function(){var e=this.node.attr("format");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t}(xe);t.DateTimeTypeDeclarationScalarsAnnotationsImpl=Qe;var Ze=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?b(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.fileTypes=function(){return e.prototype.attributes.call(this,"fileTypes",function(e){return new le(e)})},t.prototype.minLength=function(){return e.prototype.attribute.call(this,"minLength",this.toNumber)},t.prototype.setMinLength=function(e){return this.highLevel().attrOrCreate("minLength").setValue(""+e),this},t.prototype.maxLength=function(){return e.prototype.attribute.call(this,"maxLength",this.toNumber)},t.prototype.setMaxLength=function(e){return this.highLevel().attrOrCreate("maxLength").setValue(""+e),this},t.prototype.wrapperClassName=function(){return"FileTypeDeclarationImpl"},t.prototype.kind=function(){return"FileTypeDeclaration"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("FileTypeDeclaration")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.FileTypeDeclarationImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.FileTypeDeclarationImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.scalarsAnnotations=function(){return new et(this.highLevel())},t}(Re);t.FileTypeDeclarationImpl=Ze;var et=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.fileTypes=function(){var e=this.node.attributes("fileTypes");return e.map(function(e){var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n})},t.prototype.minLength=function(){var e=this.node.attr("minLength");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t.prototype.maxLength=function(){var e=this.node.attr("maxLength");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t}(xe);t.FileTypeDeclarationScalarsAnnotationsImpl=et;var tt=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?_(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.code=function(){return e.prototype.attribute.call(this,"code",function(e){return new se(e)})},t.prototype.headers=function(){return e.prototype.elements.call(this,"headers")},t.prototype.body=function(){return e.prototype.elements.call(this,"body")},t.prototype.description=function(){return e.prototype.attribute.call(this,"description",function(e){return new pe(e)})},t.prototype.annotations=function(){return e.prototype.attributes.call(this,"annotations",function(e){return new Lt(e)})},t.prototype.wrapperClassName=function(){return"ResponseImpl"},t.prototype.kind=function(){return"Response"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("Response")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.ResponseImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.ResponseImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.isOkRange=function(){return ee.isOkRange(this)},t.prototype.scalarsAnnotations=function(){return new nt(this.highLevel())},t}(te);t.ResponseImpl=tt;var nt=function(){function e(e){this.node=e}return e.prototype.description=function(){var e=this.node.attr("description");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e}();t.ResponseScalarsAnnotationsImpl=nt;var rt=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?N(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.annotations=function(){return e.prototype.attributes.call(this,"annotations",function(e){return new Lt(e)})},t.prototype.wrapperClassName=function(){return"SecuritySchemePartImpl"},t.prototype.kind=function(){return"SecuritySchemePart"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("SecuritySchemePart")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.SecuritySchemePartImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.SecuritySchemePartImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(we);t.SecuritySchemePartImpl=rt;var it=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?s(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.body=function(){return e.prototype.elements.call(this,"body")},t.prototype.protocols=function(){return e.prototype.attributes.call(this,"protocols",this.toString)},t.prototype.setProtocols=function(e){return this.highLevel().attrOrCreate("protocols").setValue(""+e),this},t.prototype.is=function(){return e.prototype.attributes.call(this,"is",function(e){return new Ne(e)})},t.prototype.securedBy=function(){return e.prototype.attributes.call(this,"securedBy",function(e){return new at(e)})},t.prototype.description=function(){return e.prototype.attribute.call(this,"description",function(e){return new pe(e)})},t.prototype.displayName=function(){return e.prototype.attribute.call(this,"displayName",this.toString)},t.prototype.setDisplayName=function(e){return this.highLevel().attrOrCreate("displayName").setValue(""+e),this},t.prototype.wrapperClassName=function(){return"MethodBaseImpl"},t.prototype.kind=function(){return"MethodBase"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("MethodBase")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.MethodBaseImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.MethodBaseImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.scalarsAnnotations=function(){return new Et(this.highLevel())},t}(we);t.MethodBaseImpl=it;var at=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"SecuritySchemeRefImpl"},t.prototype.kind=function(){return"SecuritySchemeRef"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("SecuritySchemeRef")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.SecuritySchemeRefImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.SecuritySchemeRefImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.securitySchemeName=function(){return ee.securitySchemeName(this)},t.prototype.securityScheme=function(){return ee.securityScheme(this)},t}(_e);t.SecuritySchemeRefImpl=at;var ot=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?w(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.name=function(){return e.prototype.attribute.call(this,"name",this.toString)},t.prototype.setName=function(e){return this.highLevel().attrOrCreate("name").setValue(""+e),this},t.prototype.type=function(){return e.prototype.attribute.call(this,"type",this.toString)},t.prototype.setType=function(e){return this.highLevel().attrOrCreate("type").setValue(""+e),this},t.prototype.description=function(){return e.prototype.attribute.call(this,"description",function(e){return new pe(e)})},t.prototype.describedBy=function(){return e.prototype.element.call(this,"describedBy")},t.prototype.displayName=function(){return e.prototype.attribute.call(this,"displayName",this.toString)},t.prototype.setDisplayName=function(e){return this.highLevel().attrOrCreate("displayName").setValue(""+e),this},t.prototype.settings=function(){return e.prototype.element.call(this,"settings")},t.prototype.wrapperClassName=function(){return"AbstractSecuritySchemeImpl"},t.prototype.kind=function(){return"AbstractSecurityScheme"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("AbstractSecurityScheme")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.AbstractSecuritySchemeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.AbstractSecuritySchemeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.scalarsAnnotations=function(){return new gt(this.highLevel())},t}(te);t.AbstractSecuritySchemeImpl=ot;var st=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?R(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.wrapperClassName=function(){return"SecuritySchemeSettingsImpl"},t.prototype.kind=function(){return"SecuritySchemeSettings"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("SecuritySchemeSettings")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.SecuritySchemeSettingsImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.SecuritySchemeSettingsImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(te);t.SecuritySchemeSettingsImpl=st;var ut=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?I(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.requestTokenUri=function(){return e.prototype.attribute.call(this,"requestTokenUri",function(e){return new ue(e)})},t.prototype.authorizationUri=function(){return e.prototype.attribute.call(this,"authorizationUri",function(e){return new ue(e)})},t.prototype.tokenCredentialsUri=function(){return e.prototype.attribute.call(this,"tokenCredentialsUri",function(e){return new ue(e)})},t.prototype.signatures=function(){return e.prototype.attributes.call(this,"signatures",this.toString);
},t.prototype.setSignatures=function(e){return this.highLevel().attrOrCreate("signatures").setValue(""+e),this},t.prototype.wrapperClassName=function(){return"OAuth1SecuritySchemeSettingsImpl"},t.prototype.kind=function(){return"OAuth1SecuritySchemeSettings"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("OAuth1SecuritySchemeSettings")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.OAuth1SecuritySchemeSettingsImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.OAuth1SecuritySchemeSettingsImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.scalarsAnnotations=function(){return new lt(this.highLevel())},t}(st);t.OAuth1SecuritySchemeSettingsImpl=ut;var lt=function(){function e(e){this.node=e}return e.prototype.requestTokenUri=function(){var e=this.node.attr("requestTokenUri");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e.prototype.authorizationUri=function(){var e=this.node.attr("authorizationUri");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e.prototype.tokenCredentialsUri=function(){var e=this.node.attr("tokenCredentialsUri");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e.prototype.signatures=function(){var e=this.node.attributes("signatures");return e.map(function(e){var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n})},e}();t.OAuth1SecuritySchemeSettingsScalarsAnnotationsImpl=lt;var pt=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?M(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.accessTokenUri=function(){return e.prototype.attribute.call(this,"accessTokenUri",function(e){return new ue(e)})},t.prototype.authorizationUri=function(){return e.prototype.attribute.call(this,"authorizationUri",function(e){return new ue(e)})},t.prototype.authorizationGrants=function(){return e.prototype.attributes.call(this,"authorizationGrants",this.toString)},t.prototype.setAuthorizationGrants=function(e){return this.highLevel().attrOrCreate("authorizationGrants").setValue(""+e),this},t.prototype.scopes=function(){return e.prototype.attributes.call(this,"scopes",this.toString)},t.prototype.setScopes=function(e){return this.highLevel().attrOrCreate("scopes").setValue(""+e),this},t.prototype.wrapperClassName=function(){return"OAuth2SecuritySchemeSettingsImpl"},t.prototype.kind=function(){return"OAuth2SecuritySchemeSettings"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("OAuth2SecuritySchemeSettings")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.OAuth2SecuritySchemeSettingsImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.OAuth2SecuritySchemeSettingsImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.scalarsAnnotations=function(){return new ct(this.highLevel())},t}(st);t.OAuth2SecuritySchemeSettingsImpl=pt;var ct=function(){function e(e){this.node=e}return e.prototype.accessTokenUri=function(){var e=this.node.attr("accessTokenUri");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e.prototype.authorizationUri=function(){var e=this.node.attr("authorizationUri");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e.prototype.authorizationGrants=function(){var e=this.node.attributes("authorizationGrants");return e.map(function(e){var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n})},e.prototype.scopes=function(){var e=this.node.attributes("scopes");return e.map(function(e){var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n})},e}();t.OAuth2SecuritySchemeSettingsScalarsAnnotationsImpl=ct;var ft=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?C(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.settings=function(){return e.prototype.element.call(this,"settings")},t.prototype.wrapperClassName=function(){return"OAuth2SecuritySchemeImpl"},t.prototype.kind=function(){return"OAuth2SecurityScheme"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("OAuth2SecurityScheme")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.OAuth2SecuritySchemeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.OAuth2SecuritySchemeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(ot);t.OAuth2SecuritySchemeImpl=ft;var dt=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?L(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.settings=function(){return e.prototype.element.call(this,"settings")},t.prototype.wrapperClassName=function(){return"OAuth1SecuritySchemeImpl"},t.prototype.kind=function(){return"OAuth1SecurityScheme"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("OAuth1SecurityScheme")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.OAuth1SecuritySchemeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.OAuth1SecuritySchemeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(ot);t.OAuth1SecuritySchemeImpl=dt;var ht=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?P(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.settings=function(){return e.prototype.element.call(this,"settings")},t.prototype.wrapperClassName=function(){return"PassThroughSecuritySchemeImpl"},t.prototype.kind=function(){return"PassThroughSecurityScheme"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("PassThroughSecurityScheme")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.PassThroughSecuritySchemeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.PassThroughSecuritySchemeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(ot);t.PassThroughSecuritySchemeImpl=ht;var mt=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?O(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.wrapperClassName=function(){return"BasicSecuritySchemeImpl"},t.prototype.kind=function(){return"BasicSecurityScheme"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("BasicSecurityScheme")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.BasicSecuritySchemeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.BasicSecuritySchemeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(ot);t.BasicSecuritySchemeImpl=mt;var yt=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?D(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.wrapperClassName=function(){return"DigestSecuritySchemeImpl"},t.prototype.kind=function(){return"DigestSecurityScheme"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("DigestSecurityScheme")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.DigestSecuritySchemeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.DigestSecuritySchemeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(ot);t.DigestSecuritySchemeImpl=yt;var vt=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?x(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.wrapperClassName=function(){return"CustomSecuritySchemeImpl"},t.prototype.kind=function(){return"CustomSecurityScheme"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("CustomSecurityScheme")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.CustomSecuritySchemeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.CustomSecuritySchemeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(ot);t.CustomSecuritySchemeImpl=vt;var gt=function(){function e(e){this.node=e}return e.prototype.type=function(){var e=this.node.attr("type");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e.prototype.description=function(){var e=this.node.attr("description");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e.prototype.displayName=function(){var e=this.node.attr("displayName");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e}();t.AbstractSecuritySchemeScalarsAnnotationsImpl=gt;var At=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?U(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.method=function(){return e.prototype.attribute.call(this,"method",this.toString)},t.prototype.setMethod=function(e){return this.highLevel().attrOrCreate("method").setValue(""+e),this},t.prototype.displayName=function(){return e.prototype.attribute.call(this,"displayName",this.toString)},t.prototype.setDisplayName=function(e){return this.highLevel().attrOrCreate("displayName").setValue(""+e),this},t.prototype.wrapperClassName=function(){return"MethodImpl"},t.prototype.kind=function(){return"Method"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("Method")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.MethodImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.MethodImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.parentResource=function(){return ee.parentResource(this)},t.prototype.ownerApi=function(){return ee.ownerApi(this)},t.prototype.methodId=function(){return ee.methodId(this)},t.prototype.allSecuredBy=function(){return ee.allSecuredBy(this)},t.prototype.parametrizedProperties=function(){return ee.getTemplateParametrizedProperties(this)},t.prototype.scalarsAnnotations=function(){return new Tt(this.highLevel())},t}(it);t.MethodImpl=At;var Et=function(){function e(e){this.node=e}return e.prototype.protocols=function(){var e=this.node.attributes("protocols");return e.map(function(e){var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n})},e.prototype.is=function(){var e=this.node.attributes("is");return e.map(function(e){var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n})},e.prototype.securedBy=function(){var e=this.node.attributes("securedBy");return e.map(function(e){var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n})},e.prototype.description=function(){var e=this.node.attr("description");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e.prototype.displayName=function(){var e=this.node.attr("displayName");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e}();t.MethodBaseScalarsAnnotationsImpl=Et;var Tt=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.displayName=function(){var e=this.node.attr("displayName");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t}(Et);t.MethodScalarsAnnotationsImpl=Tt;var St=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?o(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.name=function(){return e.prototype.attribute.call(this,"name",this.toString)},t.prototype.setName=function(e){return this.highLevel().attrOrCreate("name").setValue(""+e),this},t.prototype.usage=function(){return e.prototype.attribute.call(this,"usage",this.toString)},t.prototype.setUsage=function(e){return this.highLevel().attrOrCreate("usage").setValue(""+e),this},t.prototype.displayName=function(){return e.prototype.attribute.call(this,"displayName",this.toString)},t.prototype.setDisplayName=function(e){return this.highLevel().attrOrCreate("displayName").setValue(""+e),this},t.prototype.uses=function(){return e.prototype.elements.call(this,"uses")},t.prototype.wrapperClassName=function(){return"TraitImpl"},t.prototype.kind=function(){return"Trait"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("Trait")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.TraitImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.TraitImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.parametrizedProperties=function(){return ee.getTemplateParametrizedProperties(this)},t.prototype.scalarsAnnotations=function(){return new bt(this.highLevel())},t}(it);t.TraitImpl=St;var bt=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.usage=function(){var e=this.node.attr("usage");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t.prototype.displayName=function(){var e=this.node.attr("displayName");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t}(Et);t.TraitScalarsAnnotationsImpl=bt;var _t=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"ResourceTypeRefImpl"},t.prototype.kind=function(){return"ResourceTypeRef"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("ResourceTypeRef")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.ResourceTypeRefImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.ResourceTypeRefImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.resourceType=function(){return ee.referencedResourceType(this)},t}(_e);t.ResourceTypeRefImpl=_t;var Nt=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?F(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.methods=function(){return e.prototype.elements.call(this,"methods")},t.prototype.is=function(){return e.prototype.attributes.call(this,"is",function(e){return new Ne(e)})},t.prototype.type=function(){return e.prototype.attribute.call(this,"type",function(e){return new _t(e)})},t.prototype.description=function(){return e.prototype.attribute.call(this,"description",function(e){return new pe(e)})},t.prototype.securedBy=function(){return e.prototype.attributes.call(this,"securedBy",function(e){return new at(e)})},t.prototype.uriParameters_original=function(){return e.prototype.elements.call(this,"uriParameters")},t.prototype.wrapperClassName=function(){return"ResourceBaseImpl"},t.prototype.kind=function(){return"ResourceBase"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("ResourceBase")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.ResourceBaseImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.ResourceBaseImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.uriParameters=function(){return ee.uriParametersPrimary(this)},t.prototype.allUriParameters=function(){return ee.uriParameters(this)},t.prototype.allSecuredBy=function(){return ee.allSecuredBy(this)},t.prototype.scalarsAnnotations=function(){return new Rt(this.highLevel())},t}(te);t.ResourceBaseImpl=Nt;var wt=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?B(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.relativeUri=function(){return e.prototype.attribute.call(this,"relativeUri",function(e){return new ae(e)})},t.prototype.displayName=function(){return e.prototype.attribute.call(this,"displayName",this.toString)},t.prototype.setDisplayName=function(e){return this.highLevel().attrOrCreate("displayName").setValue(""+e),this},t.prototype.resources=function(){return e.prototype.elements.call(this,"resources")},t.prototype.description=function(){return e.prototype.attribute.call(this,"description",function(e){return new pe(e)})},t.prototype.annotations=function(){return e.prototype.attributes.call(this,"annotations",function(e){return new Lt(e)})},t.prototype.wrapperClassName=function(){return"ResourceImpl"},t.prototype.kind=function(){return"Resource"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("Resource")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.ResourceImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.ResourceImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.completeRelativeUri=function(){return ee.completeRelativeUri(this)},t.prototype.absoluteUri=function(){return ee.absoluteUri(this)},t.prototype.parentResource=function(){return ee.parent(this)},t.prototype.childResource=function(e){return ee.childResource(this,e)},t.prototype.childMethod=function(e){return ee.childMethod(this,e)},t.prototype.ownerApi=function(){return ee.ownerApi(this)},t.prototype.absoluteUriParameters=function(){return ee.absoluteUriParameters(this)},t.prototype.scalarsAnnotations=function(){return new It(this.highLevel())},t}(Nt);t.ResourceImpl=wt;var Rt=function(){function e(e){this.node=e}return e.prototype.is=function(){var e=this.node.attributes("is");return e.map(function(e){var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n})},e.prototype.type=function(){var e=this.node.attr("type");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e.prototype.description=function(){var e=this.node.attr("description");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e.prototype.securedBy=function(){var e=this.node.attributes("securedBy");return e.map(function(e){var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n})},e}();t.ResourceBaseScalarsAnnotationsImpl=Rt;var It=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.displayName=function(){var e=this.node.attr("displayName");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t.prototype.description=function(){var e=this.node.attr("description");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t}(Rt);t.ResourceScalarsAnnotationsImpl=It;var Mt=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?k(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.displayName=function(){return e.prototype.attribute.call(this,"displayName",this.toString)},t.prototype.setDisplayName=function(e){return this.highLevel().attrOrCreate("displayName").setValue(""+e),this},t.prototype.name=function(){return e.prototype.attribute.call(this,"name",this.toString)},t.prototype.setName=function(e){return this.highLevel().attrOrCreate("name").setValue(""+e),this},t.prototype.usage=function(){return e.prototype.attribute.call(this,"usage",this.toString)},t.prototype.setUsage=function(e){return this.highLevel().attrOrCreate("usage").setValue(""+e),this},t.prototype.uses=function(){return e.prototype.elements.call(this,"uses")},t.prototype.wrapperClassName=function(){return"ResourceTypeImpl"},t.prototype.kind=function(){return"ResourceType"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("ResourceType")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.ResourceTypeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.ResourceTypeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.parametrizedProperties=function(){return ee.getTemplateParametrizedProperties(this)},t.prototype.scalarsAnnotations=function(){return new Ct(this.highLevel())},t}(Nt);t.ResourceTypeImpl=Mt;var Ct=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.displayName=function(){var e=this.node.attr("displayName");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t.prototype.usage=function(){var e=this.node.attr("usage");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t}(Rt);t.ResourceTypeScalarsAnnotationsImpl=Ct;var Lt=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.wrapperClassName=function(){return"AnnotationRefImpl"},t.prototype.kind=function(){return"AnnotationRef"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("AnnotationRef")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.AnnotationRefImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.AnnotationRefImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.annotation=function(){return ee.referencedAnnotation(this)},t}(_e);t.AnnotationRefImpl=Lt;var Pt=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?K(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.title=function(){return e.prototype.attribute.call(this,"title",this.toString)},t.prototype.setTitle=function(e){return this.highLevel().attrOrCreate("title").setValue(""+e),this},t.prototype.content=function(){return e.prototype.attribute.call(this,"content",function(e){return new pe(e)})},t.prototype.uses=function(){return e.prototype.elements.call(this,"uses")},t.prototype.wrapperClassName=function(){return"DocumentationItemImpl"},t.prototype.kind=function(){return"DocumentationItem"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("DocumentationItem")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.DocumentationItemImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.DocumentationItemImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.scalarsAnnotations=function(){return new Ot(this.highLevel())},t}(te);t.DocumentationItemImpl=Pt;var Ot=function(){function e(e){this.node=e}return e.prototype.title=function(){var e=this.node.attr("title");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e.prototype.content=function(){var e=this.node.attr("content");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e}();t.DocumentationItemScalarsAnnotationsImpl=Ot;var Dt=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?a(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.uses=function(){return e.prototype.elements.call(this,"uses")},t.prototype.wrapperClassName=function(){return"FragmentDeclarationImpl"},t.prototype.kind=function(){return"FragmentDeclaration"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("FragmentDeclaration")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.FragmentDeclarationImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.FragmentDeclarationImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t}(te);t.FragmentDeclarationImpl=Dt;var xt=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?i(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.schemas=function(){return e.prototype.elements.call(this,"schemas")},t.prototype.types=function(){return e.prototype.elements.call(this,"types")},t.prototype.traits_original=function(){return e.prototype.elements.call(this,"traits")},t.prototype.resourceTypes_original=function(){return e.prototype.elements.call(this,"resourceTypes")},t.prototype.annotationTypes=function(){return e.prototype.elements.call(this,"annotationTypes")},t.prototype.securitySchemes=function(){return e.prototype.elements.call(this,"securitySchemes")},t.prototype.wrapperClassName=function(){return"LibraryBaseImpl"},t.prototype.kind=function(){return"LibraryBase"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("LibraryBase")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.LibraryBaseImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.LibraryBaseImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.traits=function(){return ee.traitsPrimary(this)},t.prototype.allTraits=function(){return ee.allTraits(this)},t.prototype.resourceTypes=function(){return ee.resourceTypesPrimary(this)},t.prototype.allResourceTypes=function(){return ee.allResourceTypes(this)},t}(Dt);t.LibraryBaseImpl=xt;var Ut=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?V(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.usage=function(){return e.prototype.attribute.call(this,"usage",this.toString)},t.prototype.setUsage=function(e){return this.highLevel().attrOrCreate("usage").setValue(""+e),this},t.prototype.name=function(){return e.prototype.attribute.call(this,"name",this.toString)},t.prototype.setName=function(e){return this.highLevel().attrOrCreate("name").setValue(""+e),this},t.prototype.wrapperClassName=function(){return"LibraryImpl"},t.prototype.kind=function(){return"Library"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("Library")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.LibraryImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.LibraryImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.expand=function(){return ee.expandLibrarySpec(this)},t.prototype.scalarsAnnotations=function(){return new kt(this.highLevel())},t}(xt);t.LibraryImpl=Ut;var kt=function(){function e(e){this.node=e}return e.prototype.usage=function(){var e=this.node.attr("usage");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e}();t.LibraryScalarsAnnotationsImpl=kt;var Ft=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?r(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.title=function(){return e.prototype.attribute.call(this,"title",this.toString)},t.prototype.setTitle=function(e){return this.highLevel().attrOrCreate("title").setValue(""+e),this},t.prototype.description=function(){return e.prototype.attribute.call(this,"description",function(e){return new pe(e)})},t.prototype.version=function(){return e.prototype.attribute.call(this,"version",this.toString)},t.prototype.setVersion=function(e){return this.highLevel().attrOrCreate("version").setValue(""+e),this},t.prototype.baseUri=function(){return e.prototype.attribute.call(this,"baseUri",function(e){return new oe(e)})},t.prototype.baseUriParameters_original=function(){return e.prototype.elements.call(this,"baseUriParameters")},t.prototype.protocols=function(){return e.prototype.attributes.call(this,"protocols",this.toString)},t.prototype.setProtocols=function(e){return this.highLevel().attrOrCreate("protocols").setValue(""+e),this},t.prototype.mediaType=function(){return e.prototype.attributes.call(this,"mediaType",function(e){return new fe(e)})},t.prototype.securedBy=function(){return e.prototype.attributes.call(this,"securedBy",function(e){return new at(e)})},t.prototype.resources=function(){return e.prototype.elements.call(this,"resources")},t.prototype.documentation=function(){return e.prototype.elements.call(this,"documentation")},t.prototype.annotations=function(){return e.prototype.attributes.call(this,"annotations",function(e){return new Lt(e)})},t.prototype.wrapperClassName=function(){return"ApiImpl"},t.prototype.kind=function(){return"Api"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("Api")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.ApiImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.ApiImpl"==r)return!0}return!1},t.prototype.RAMLVersion_original=function(){return"RAML10"},t.prototype.expand=function(e){return void 0===e&&(e=!1),ee.expandSpec(this,e)},t.prototype.childResource=function(e){return ee.childResource(this,e)},t.prototype.allResources=function(){return ee.allResources(this)},t.prototype.baseUriParameters=function(){
return ee.baseUriParametersPrimary(this)},t.prototype.allBaseUriParameters=function(){return ee.baseUriParameters(this)},t.prototype.allProtocols=function(){return ee.allProtocols(this)},t.prototype.RAMLVersion=function(){return ee.RAMLVersion(this)},t.prototype.scalarsAnnotations=function(){return new Kt(this.highLevel())},t}(xt);t.ApiImpl=Ft;var Bt=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?j(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.usage=function(){return e.prototype.attribute.call(this,"usage",this.toString)},t.prototype.setUsage=function(e){return this.highLevel().attrOrCreate("usage").setValue(""+e),this},t.prototype["extends"]=function(){return e.prototype.attribute.call(this,"extends",this.toString)},t.prototype.setExtends=function(e){return this.highLevel().attrOrCreate("extends").setValue(""+e),this},t.prototype.title=function(){return e.prototype.attribute.call(this,"title",this.toString)},t.prototype.setTitle=function(e){return this.highLevel().attrOrCreate("title").setValue(""+e),this},t.prototype.wrapperClassName=function(){return"OverlayImpl"},t.prototype.kind=function(){return"Overlay"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("Overlay")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.OverlayImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.OverlayImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.scalarsAnnotations=function(){return new Vt(this.highLevel())},t}(Ft);t.OverlayImpl=Bt;var Kt=function(){function e(e){this.node=e}return e.prototype.title=function(){var e=this.node.attr("title");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e.prototype.description=function(){var e=this.node.attr("description");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e.prototype.version=function(){var e=this.node.attr("version");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e.prototype.baseUri=function(){var e=this.node.attr("baseUri");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},e.prototype.protocols=function(){var e=this.node.attributes("protocols");return e.map(function(e){var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n})},e.prototype.mediaType=function(){var e=this.node.attributes("mediaType");return e.map(function(e){var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n})},e.prototype.securedBy=function(){var e=this.node.attributes("securedBy");return e.map(function(e){var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n})},e}();t.ApiScalarsAnnotationsImpl=Kt;var Vt=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.usage=function(){var e=this.node.attr("usage");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t.prototype["extends"]=function(){var e=this.node.attr("extends");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t.prototype.title=function(){var e=this.node.attr("title");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t}(Kt);t.OverlayScalarsAnnotationsImpl=Vt;var jt=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?W(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return X(t,e),t.prototype.usage=function(){return e.prototype.attribute.call(this,"usage",this.toString)},t.prototype.setUsage=function(e){return this.highLevel().attrOrCreate("usage").setValue(""+e),this},t.prototype["extends"]=function(){return e.prototype.attribute.call(this,"extends",this.toString)},t.prototype.setExtends=function(e){return this.highLevel().attrOrCreate("extends").setValue(""+e),this},t.prototype.title=function(){return e.prototype.attribute.call(this,"title",this.toString)},t.prototype.setTitle=function(e){return this.highLevel().attrOrCreate("title").setValue(""+e),this},t.prototype.wrapperClassName=function(){return"ExtensionImpl"},t.prototype.kind=function(){return"Extension"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("Extension")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML10.ExtensionImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML10.ExtensionImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML10"},t.prototype.scalarsAnnotations=function(){return new Wt(this.highLevel())},t}(Ft);t.ExtensionImpl=jt;var Wt=function(e){function t(){e.apply(this,arguments)}return X(t,e),t.prototype.usage=function(){var e=this.node.attr("usage");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t.prototype["extends"]=function(){var e=this.node.attr("extends");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t.prototype.title=function(){var e=this.node.attr("title");if(null==e)return[];var t=e.annotations(),n=Q.attributesToValues(t,function(e){return new Lt(e)});return n},t}(Kt);t.ExtensionScalarsAnnotationsImpl=Wt,t.loadApiSync=q,t.loadRAMLSync=Y,t.loadApi=H,t.loadRAML=$,t.getLanguageElementByRuntimeType=G},function(e,t,n){"use strict";var r,i=n(71),a=function(e){return null!=e},o=function(){function e(t){var n=this;this._value=t,this._isDefined=!1,this._isEmpty=!0,this._arr=void 0,this.getOrThrow=function(){return i(n._isDefined,"Opt is empty. Use `Opt#getOrElse` or check `Opt#isDefined`."),n._value},this.value=function(){return n._value},this.isDefined=function(){return n._isDefined},this.isEmpty=function(){return n._isEmpty},this.toArray=function(){return n._arr||(n._arr=n._isDefined?[n._value]:[])},this.getOrElse=function(e){return n._isDefined?n._value:e},this.getOrElseF=function(e){return n._isDefined?n._value:e()},this.map=function(t){return n._isEmpty?e.empty():new e(t(n._value))},this.flatMap=function(t){return n.map(t).getOrElse(e.empty())},this.equals=function(t){return i(t instanceof e,"Expected other to be an `Opt`, but got `%s`",typeof t),n._isDefined===t.isDefined()&&n._value===t.value()},a(this._value)&&(this._isDefined=!0,this._isEmpty=!1)}return e.prototype.forEach=function(e){this.isDefined()&&e(this._value)},e.empty=function(){return r||(r=new e)},e}();e.exports=o},function(e,t,n){!function(t){function n(){}function r(e,t){return function(){e.apply(t,arguments)}}function i(e){if("object"!=typeof this)throw new TypeError("Promises must be constructed via new");if("function"!=typeof e)throw new TypeError("not a function");this._state=0,this._handled=!1,this._value=void 0,this._deferreds=[],p(e,this)}function a(e,t){for(;3===e._state;)e=e._value;return 0===e._state?void e._deferreds.push(t):(e._handled=!0,void f(function(){var n=1===e._state?t.onFulfilled:t.onRejected;if(null===n)return void(1===e._state?o:s)(t.promise,e._value);var r;try{r=n(e._value)}catch(i){return void s(t.promise,i)}o(t.promise,r)}))}function o(e,t){try{if(t===e)throw new TypeError("A promise cannot be resolved with itself.");if(t&&("object"==typeof t||"function"==typeof t)){var n=t.then;if(t instanceof i)return e._state=3,e._value=t,void u(e);if("function"==typeof n)return void p(r(n,t),e)}e._state=1,e._value=t,u(e)}catch(a){s(e,a)}}function s(e,t){e._state=2,e._value=t,u(e)}function u(e){2===e._state&&0===e._deferreds.length&&f(function(){e._handled||d(e._value)});for(var t=0,n=e._deferreds.length;n>t;t++)a(e,e._deferreds[t]);e._deferreds=null}function l(e,t,n){this.onFulfilled="function"==typeof e?e:null,this.onRejected="function"==typeof t?t:null,this.promise=n}function p(e,t){var n=!1;try{e(function(e){n||(n=!0,o(t,e))},function(e){n||(n=!0,s(t,e))})}catch(r){if(n)return;n=!0,s(t,r)}}var c=setTimeout,f="function"==typeof setImmediate&&setImmediate||function(e){c(e,0)},d=function(e){"undefined"!=typeof console&&console&&console.warn("Possible Unhandled Promise Rejection:",e)};i.prototype["catch"]=function(e){return this.then(null,e)},i.prototype.then=function(e,t){var r=new this.constructor(n);return a(this,new l(e,t,r)),r},i.all=function(e){var t=Array.prototype.slice.call(e);return new i(function(e,n){function r(a,o){try{if(o&&("object"==typeof o||"function"==typeof o)){var s=o.then;if("function"==typeof s)return void s.call(o,function(e){r(a,e)},n)}t[a]=o,0===--i&&e(t)}catch(u){n(u)}}if(0===t.length)return e([]);for(var i=t.length,a=0;a<t.length;a++)r(a,t[a])})},i.resolve=function(e){return e&&"object"==typeof e&&e.constructor===i?e:new i(function(t){t(e)})},i.reject=function(e){return new i(function(t,n){n(e)})},i.race=function(e){return new i(function(t,n){for(var r=0,i=e.length;i>r;r++)e[r].then(t,n)})},i._setImmediateFn=function(e){f=e},i._setUnhandledRejectionFn=function(e){d=e},"undefined"!=typeof e&&e.exports?e.exports=i:t.Promise||(t.Promise=i)}(this)},function(e,t,n){"use strict";function r(){return new W}function i(e){if(null==e)return null;switch(e.kind){case w.Kind.SCALAR:return{errors:[],startPosition:e.startPosition,endPosition:e.endPosition,value:e.value,kind:w.Kind.SCALAR,parent:e.parent};case w.Kind.MAPPING:var t=e;return{errors:[],key:i(t.key),value:i(t.value),startPosition:t.startPosition,endPosition:t.endPosition,kind:w.Kind.MAPPING,parent:t.parent};case w.Kind.MAP:var n=e;return{errors:[],startPosition:e.startPosition,endPosition:e.endPosition,mappings:n.mappings.map(function(e){return i(e)}),kind:w.Kind.MAP,parent:n.parent}}return e}function a(e){var t=e.match(/^.*((\r\n|\n|\r)|$)/gm);return t}function o(e,t){for(var n=a(e),r=[],i=0;i<n.length;i++)0==i?r.push(n[0]):r.push(n[i].substring(t.length));return r.join("")}function s(e){for(var t="",n=0;n<e.length;n++){var r=e[n];if("\r"!=r&&"\n"!=r){if(" "!=r&&"	"!=r)break;t+=r}}return t}function u(e,t){var n=e.substr(t.offset,t.replacementLength);return-1!=n.indexOf("\n")?!0:-1!=t.text.indexOf("\n")?!0:void 0}function l(e){if("null"==e)e=null;else if("~"==e)e=null;else if("true"==e)e=!0;else if("false"==e)e=!1;else{var t=parseFloat(e);isNaN(t)||(""+e).match("^[-+]?[0-9]*\\.?[0-9]+$")&&(e=t)}return e}function p(e,t,n){var r=w.newMapping(w.newScalar(e),w.newMap());return new z(r,n,t,null,null)}function c(e){var t=w.newMap(e);return new z(t,null,null,null,null)}function f(e){var t=w.newScalar(e);return new z(t,null,null,null,null)}function d(e,t,n){return new z(e,n,t,null,null)}function h(e){var t=w.newMapping(w.newScalar(e),w.newItems());return new z(t,null,null,null,null)}function m(e){var t=w.newMapping(w.newScalar(e),w.newMap());return new z(t,null,null,null,null)}function y(e,t){var n=w.newMapping(w.newScalar(e),w.newScalar(t));return new z(n,null,null,null,null)}function v(e){if(!z.isInstance(e))return null;var t=e,n=new z(t.yamlNode(),t.unit(),null,null,null,!0);return n._errors=t._errors,n}function g(e){if(!z.isInstance(e))return null;var t=e,n=new z(t.yamlNode(),t.unit(),null,null,null,!1);return n._errors=t._errors,n}function A(e){var t=e.highLevelNode();if(t)return t.definition();var n=e.parent();if(!n)return null;var r=e.key();if(!r)return null;var i=n.nodeDefinition();if(!i)return null;if(!i.property)return null;var a=i.property(r);return a?a.range():null}function E(e,t,n,r){if(e.isOverlayOrExtension()){var i=e.getMasterReferenceNode();if(i){var a=i.value();if(a){var o=b.dirname(e.absolutePath()),s=S.toAbsolutePath(o,a);if(!t[s]){if(n[s])return void i.errors().push(new F(n[a]));var u=r[s];u||(u=[],r[s]=u),u.push(i)}}}}}function T(e,t){var n={},r={},i=[],a=function(t){for(var o=[],s={};t<i.length;){var u=i[t],l=u.absolutePath();E(u,n,r,s);var p=u.getIncludeNodes();p.forEach(function(t){var i=t.includePath();if(i){var a=O.getIncludeReference(i);if(a)return void o.push(e.resolveAsync(u.absolutePath(),i));if(i){var p=S.buildPath(i,l,e.getRootPath());if(!n[p]){if(r[p]&&t.errors)return void t.errors().push(new F(r[i]));var c=s[p];c||(c=[],s[p]=c),c.push(t)}}}}),t++}var c=Object.keys(s);if(0==o.length&&0==c.length)return Promise.resolve();var f=o;return c.forEach(function(t){f.push(e.unitAsync(t,!0).then(function(e){n[e.absolutePath()]||(n[e.absolutePath()]=!0,i.push(e))},function(n){var i=e.pathToUnit[t];s[t].forEach(function(e){i||e.errors&&e.errors().push(new F(n))}),r[t]=n,e.failedUnits[t]=n,i&&(n.inner=!0)}))}),Promise.all(f).then(function(e){return a(t)})};return e.unitAsync(t,b.isAbsolute(t)).then(function(e){return i.push(e),n[e.absolutePath()]=!0,a(0)}).then(function(e){return i.length>0?i[0]:null})}var S=n(10),b=n(15),_=n(40),N=n(41),w=n(77),R=n(70),I=n(43),M=n(16),C=n(37),L=n(25),P=n(33),O=n(26),D=n(49),x=n(18),U=n(27),k=n(50),F=w.YAMLException;t.Kind={SCALAR:w.Kind.SCALAR};var B=function(){function e(e){this.text="",this.indent=e}return e.prototype.isLastNL=function(){return this.text.length>0&&"\n"==this.text[this.text.length-1]},e.prototype.addWithIndent=function(e,t){this.isLastNL()&&(this.text+=I.indent(e),this.text+=this.indent),this.text+=t},e.prototype.addChar=function(e){this.isLastNL()&&(this.text+=this.indent),this.text+=e},e.prototype.append=function(e){for(var t=0;t<e.length;t++)this.addChar(e[t])},e}();t.MarkupIndentingBuffer=B;var K=function(){function e(e,t,n,r,i){this._path=e,this._content=t,this._tl=n,this._project=r,this._apath=i,this._includedByPaths=[],this._path=null!=this._path?this._path.replace(/\\/g,"/"):null}return e.isInstance=function(t){return null!=t&&t.getClassIdentifier&&"function"==typeof t.getClassIdentifier&&R.contains(t.getClassIdentifier(),e.CLASS_IDENTIFIER)},e.prototype.getClassIdentifier=function(){var t=[];return t.concat(e.CLASS_IDENTIFIER)},e.prototype.highLevel=function(){return this._hl?this._hl:(this._hl=M.fromUnit(this),this._hl)},e.prototype.isStubUnit=function(){return this.stu},e.prototype.resolveAsync=function(e){var t=this._project.resolveAsync(this._path,e);return t},e.prototype.getIncludeNodes=function(){var e=this.ast(),t=[];return null==e?[]:(e.gatherIncludes(t),e.children().forEach(function(e){"uses"==e.key()&&e.children().forEach(function(e){t.push({includePath:function(){var t=e.value();return"string"!=typeof t?null:t}})})}),t)},e.prototype.cloneToProject=function(t){var n=new e(this._path,this._content,this._tl,t,this._apath);return n},e.prototype.clone=function(){var t=new e(this._path,this._content,this._tl,this.project(),this._apath);return t},e.prototype.stub=function(){var t=new e(this._path,this._content,this._tl,this.project(),this._apath);return t.stu=!0,t},e.prototype.isDirty=function(){return!1},e.prototype.expandedHighLevel=function(){if(this.expanded)return this.expanded;var e=this.highLevel().asElement(),t=U.expandTraitsAndResourceTypes(e.wrapperNode()),n=t.highLevel();return n._expanded=!0,this.expanded=n,n},e.prototype.absolutePath=function(){return this._apath},e.prototype.isRAMLUnit=function(){var e=this.absolutePath();if(S.isWebPath(e))try{e=_.parse(e).pathname}catch(t){e=this._path}var n=b.extname(e);if(".raml"==n||".yaml"==n||".yml"==n)return!0;if(!n){if(!this._content)return!1;var r=this._content.trim();if(P.stringStartsWith(r,"#%RAML"))return!0;for(var i=r.split("\n"),a=void 0,o=0,s=i;o<s.length;o++){var u=s[o],l=u.trim();if(l&&!P.stringStartsWith(l,"#")){a=u;break}}if(!a)return!0;try{var p=w.load(a,{ignoreDuplicateKeys:!0});return p.kind!=w.Kind.SCALAR}catch(t){return!1}}return!1},e.prototype.contents=function(){return this._content},e.prototype.resolve=function(e){"string"!=typeof e&&(e=""+e);var t=this._project.resolve(this._path,e);return t},e.prototype.path=function(){return this._path},e.prototype.lexerErrors=function(){return null==this.errors&&this.ast(),this.errors},e.prototype.ast=function(){var e=this;if(this._node)return this._node;try{var t=w.load(this._content,{ignoreDuplicateKeys:!0});return this.errors=t.errors,this.errors.forEach(function(t){t.mark&&(t.mark.filePath=e.absolutePath())}),this._node=new z(t,this,null,null,null),this._node._errors=this.errors,this._node}catch(n){return this.errors=[],this.errors.push(new F(n.message)),this._node=null,this._node}},e.prototype.isTopLevel=function(){return this._tl},e.prototype.updateContent=function(e){this._content=e,this.errors=null,this._node=null,this._lineMapper=null},e.prototype.updateContentSafe=function(e){this._content=e,this._lineMapper=null,this._hl=null},e.prototype.project=function(){return this._project},e.prototype.lineMapper=function(){return null==this._lineMapper&&(this._lineMapper=new S.LineMapperImpl(this.contents(),this.absolutePath())),this._lineMapper},e.prototype.isOverlayOrExtension=function(){for(var e=this.contents(),t="",n=null,r=0,i=0,a=0;a<e.length;a++){var o=e.charAt(a);if("\r"==o||"\n"==o){t?n=e.substring(i,a).trim():t=e.substring(0,a).trim();break}" "==o&&(r++,t||2!=r||(t=e.substring(0,a),i=a))}return"Extension"==n||"Overlay"==n},e.prototype.getMasterReferenceNode=function(){return R.find(this.ast().children(),function(e){return e.key()==x.Universe10.Overlay.properties["extends"].name})},e.prototype.addIncludedBy=function(e){this.includedByContains(e)||this._includedByPaths.push(e)},e.prototype.includedByContains=function(e){return null!=R.find(this._includedByPaths,function(t){return t==e})},e.prototype.getIncludedByPaths=function(){return this._includedByPaths},e.CLASS_IDENTIFIER="jsyaml2lowLevel.CompilationUnit",e}();t.CompilationUnit=K;var V=n(63).XMLHttpRequest,j=n(63).XMLHttpRequest,W=function(){function e(){}return e.prototype.open=function(e,t,n){this.xhr=n?new j:new V,this.async=n,this.xhr.open(e,t,n)},e.prototype.setRequestHeader=function(e,t){this.xhr.setRequestHeader(e,t)},e.prototype.getResponseHeader=function(e){return this.xhr.getResponseHeader(e)},e.prototype.getAllResponseHeaders=function(){return this.xhr.getAllResponseHeaders()},e.prototype.send=function(e){var t=this;this.xhr.onload=function(){return t.onResponse(!1)},this.xhr.onerror=function(){return t.onResponse(!0)},this.xhr.send(e)},e.prototype.onResponse=function(e){return this.statusText=this.xhr.statusText,this.responseText=this.xhr.responseText,this.responseType=this.xhr.responseType,this.status=this.xhr.status,e&&this.onerror?void this.onerror():void(this.onload&&this.onload())},e}();t.XMLHttpRequestWrapper=W,t.buildXHR=r;var q=function(){function e(){}return e.prototype.execute=function(e,t){void 0===t&&(t=!0);var n=r(),i=e.url;t&&(i=this.appendParams(e,e.url)),n.open(e.method,i,!1),this.doRequest(e,n);var a=n.status;if(a>300&&400>a){var o=n.getResponseHeader("location");if(o)return e.url=o,this.execute(e,!1)}var s={status:a,statusText:n.statusText,headers:n.getAllResponseHeaders().split("\n").map(function(e){var t=e.indexOf(":");return{name:e.substring(0,t).trim(),value:e.substring(t+1).trim()}}),content:{text:n.responseText,mimeType:n.responseType}};return s},e.prototype.appendParams=function(e,t){var n=e.queryString&&e.queryString.length>0;if(n){t+="?";var r=[];n&&(r=r.concat(e.queryString.map(function(e){return encodeURIComponent(e.name)+"="+encodeURIComponent(e.value)}))),t+=r.join("&")}return t},e.prototype.log=function(e,t){},e.prototype.executeAsync=function(e,t){void 0===t&&(t=!0);var n=r(),i=e.url;t&&(i=this.appendParams(e,e.url));var a=this;return new Promise(function(t,r){n.open(e.method,i,!0),n.onload=function(){var e=n.status,r={status:e,statusText:n.statusText,headers:n.getAllResponseHeaders().split("\n").map(function(e){var t=e.indexOf(":");return{name:e.substring(0,t).trim(),value:e.substring(t+1).trim()}}),content:{text:n.responseText,mimeType:n.responseType}};t(r)},n.onerror=function(){r(new F("Network Error"))},a.doRequest(e,n)})},e.prototype.doRequest=function(e,t){if(e.headers&&e.headers.forEach(function(e){return t.setRequestHeader(e.name,e.value)}),e.postData)if(e.postData.params){var n=e.postData.params.map(function(e){return encodeURIComponent(e.name)+"="+encodeURIComponent(e.value)}).join("&");t.send(n)}else t.send(e.postData.text);else t.send()},e}();t.SimpleExecutor=q;var Y=function(){function e(){this.executor=new q}return e.prototype.getResource=function(e){if("undefined"!=typeof atom&&null!=atom){var t=L.get(e);if(t)return t;L.addNotify(e);try{var n={content:""};return this.getResourceAsync(e).then(function(t){try{t.errorMessage?L.set(e,{content:null,errorMessage:null}):(t.errorMessage=null,L.set(e,t))}finally{n.callback&&n.callback(L.get(e)),L.removeNotity(e)}},function(t){L.set(e,{content:null,errorMessage:null}),n.callback&&n.callback(L.get(e)),L.removeNotity(e)}),n}catch(r){console.log("Error"),console.log(r)}}var i=this.executor.execute({method:"get",url:e});if(!i)throw new F("Unable to execute GET "+e);var a=this.toResponse(i,e);return a},e.prototype.getResourceAsync=function(e){var t=this;return this.executor.executeAsync({method:"get",url:e}).then(function(n){if(!n)return Promise.reject(new F("Unable to execute GET "+e));var r=t.toResponse(n,e);return r},function(t){return Promise.reject(new F("Unable to execute GET "+e))})},e.prototype.toResponse=function(e,t){var n=null;e.status>=400&&(n="GET "+t+"\nreturned error: "+e.status,e.statusText&&(n+=" "+e.statusText));var r=null;e.content&&e.content.text&&(r=e.content.text);var i={content:r,errorMessage:n};return i},e}();t.HTTPResolverImpl=Y;var H=function(){function e(){}return e.prototype.content=function(e){if("string"!=typeof e&&(e=""+e),!N.existsSync(e))return null;try{return N.readFileSync(e).toString()}catch(t){return null}},e.prototype.list=function(e){return N.readdirSync(e)},e.prototype.contentAsync=function(e){return new Promise(function(t,n){N.readFile(e,function(e,r){if(null!=e)return n(e);var i=r.toString();t(i)})})},e.prototype.listAsync=function(e){return new Promise(function(t,n){N.readdir(e,function(e,r){return null!=e?t(e):void n(r)})})},e}();t.FSResolverImpl=H;var $=function(e,t,n){if(t&&(t.startPosition>=e&&(t.startPosition+=n),t.endPosition>e&&(t.endPosition+=n),t.kind==w.Kind.MAPPING)){var r=t;$(e,r.key,n),$(e,r.value,n)}},G=function(e,t){for(var n="",r=e.start()-1;r>0;){var i=t[r];if(" "!=i&&"-"!=i)break;n=" "+n,r--}return n},X=function(){function e(e,t,n){this.rootPath=e,this.resolver=t,this._httpResolver=n,this.listeners=[],this.tlisteners=[],this.pathToUnit={},this.failedUnits={},this._fsEnabled=!0,this._namespaceResolver=new k.NamespaceResolver,null==this.resolver?this.resolver=new H:this._fsEnabled=!1,null==this._httpResolver&&(this._httpResolver=new Y)}return e.prototype.getRootPath=function(){return this.rootPath},e.prototype.setFSResolver=function(e){this.resolver=e},e.prototype.setHTTPResolver=function(e){this._httpResolver=e},e.prototype.fsEnabled=function(){return this._fsEnabled},e.prototype.cloneWithResolver=function(t,n){void 0===n&&(n=null);var r=new e(this.rootPath,t,n?n:this._httpResolver);for(var i in this.pathToUnit)r.pathToUnit[i]=this.pathToUnit[i].cloneToProject(r);return r},e.prototype.setCachedUnitContent=function(e,t,n){void 0===n&&(n=!0);var r=e,i=S.toAbsolutePath(this.rootPath,e),a=new K(r,t,n,this,i);return this.pathToUnit[i]=a,a},e.prototype.resolveAsync=function(e,t){var n=this;if(!t)return Promise.reject(new F("Unit path is null"));var r=O.getIncludeReference(t),i=t;r&&(t=r.getIncludePath());var a=S.buildPath(t,e,this.rootPath);if(r){var o,s=S.toAbsolutePath(b.dirname(S.toAbsolutePath(this.rootPath,e)),r.encodedName());this.pathToUnit[a]?Promise.resolve(o).then(function(e){return n.pathToUnit[s]=new K(r.encodedName(),O.resolveContents(i,n.pathToUnit[a].contents()),!1,n,s),n.pathToUnit[s]}):this.unitAsync(a,!0).then(function(e){return n.pathToUnit[a]=e,n.pathToUnit[s]=new K(r.encodedName(),O.resolveContents(i,n.pathToUnit[a].contents()),!1,n,s),n.pathToUnit[s]})}return this.unitAsync(a,!0)},e.prototype.resolve=function(e,t){if(!t)return null;t=t.replace(/\\/g,"/");var n=O.getIncludeReference(t),r=t;n&&(t=n.getIncludePath());var i=S.buildPath(t,e,this.rootPath);if(n){this.pathToUnit[i]||(this.pathToUnit[i]=this.unit(i,!0));var a=this.pathToUnit[i],o=S.toAbsolutePath(b.dirname(S.toAbsolutePath(this.rootPath,e)),n.encodedName());return this.pathToUnit[o]?this.pathToUnit[o]:(this.pathToUnit[o]=new K(n.encodedName(),O.resolveContents(r,a&&a.contents()),!1,this,o),this.pathToUnit[o])}return this.unit(i,!0)},e.prototype.units=function(){var e=this;if(!this.resolver.list)throw new F("Provided FSResolver is unable to list files. Please, use ExtendedFSResolver.");var t=this.resolver.list(this.rootPath).filter(function(e){return".raml"==b.extname(e)});return t.map(function(t){return e.unit(t)}).filter(function(e){return e.isTopLevel()})},e.prototype.unitsAsync=function(){var e=this;return this.resolver.listAsync?this.resolver.listAsync(this.rootPath).then(function(t){var n=t.filter(function(e){return".raml"==b.extname(e)}).map(function(t){return e.unitAsync(t).then(function(e){return e.isTopLevel()?e:null},function(e){return null})});return Promise.all(n).then(function(e){return e.filter(function(e){return null!=e})})}):Promise.reject(new F("Provided FSResolver is unable to list files. Please, use ExtendedFSResolver."))},e.prototype.lexerErrors=function(){var e=[];return this.units().forEach(function(t){e=e.concat(t.lexerErrors())}),e},e.prototype.deleteUnit=function(e,t){void 0===t&&(t=!1);var n=null;n=S.isWebPath(e)?e:t?e:S.toAbsolutePath(this.rootPath,e),delete this.pathToUnit[n]},e.prototype.unit=function(e,t){if(void 0===t&&(t=!1),t||S.isWebPath(e)){if(null!=this.failedUnits[e]&&!this.failedUnits[e].inner)return null}else{var n=S.toAbsolutePath(this.rootPath,e);if(this.failedUnits[n]&&!this.failedUnits[e].inner)return null}var r,i=null,a=e;if(S.isWebPath(e)){if(this.pathToUnit[a])return this.pathToUnit[a];if(this._httpResolver){if(r=this._httpResolver.getResource(e),r&&r.errorMessage)throw new F(r.errorMessage);i=r?r.content:null}else r=(new Y).getResource(a),i=r?r.content:null}else{"/"!=e.charAt(0)||t||(e=e.substr(1));var a=S.toAbsolutePath(this.rootPath,e);if(this.pathToUnit[a])return this.pathToUnit[a];if(S.isWebPath(a))if(this._httpResolver){var o=this._httpResolver.getResource(a);if(o&&o.errorMessage)throw new F(o.errorMessage);i=o?o.content:null}else{var s=(new Y).getResource(a);i=s?s.content:null}else i=this.resolver.content(a)}if(null==i)return null;var u=P.stringStartsWith(i,"#%RAML"),l=S.isWebPath(this.rootPath)==S.isWebPath(a)?b.relative(this.rootPath,a):a,p=new K(l,i,u,this,a);return this.pathToUnit[a]=p,r&&(r.callback=function(e){p.updateContent(e&&e.content)}),p},e.prototype.unitAsync=function(e,t){var n=this;void 0===t&&(t=!1);var r=null,i=e;if(S.isWebPath(e)){if(this.pathToUnit[i])return Promise.resolve(this.pathToUnit[i]);if(this._httpResolver){var a=this._httpResolver.getResourceAsync(i);r=a.then(function(e){return e.errorMessage?Promise.reject(new F(e.errorMessage)):e.content})}else r=(new Y).getResourceAsync(i)}else{if("/"!=e.charAt(0)||t||(e=e.substr(1)),i=t?e:S.toAbsolutePath(this.rootPath,e),this.pathToUnit[i])return Promise.resolve(this.pathToUnit[i]);if(S.isWebPath(i))if(this._httpResolver){var a=this._httpResolver.getResourceAsync(i);r=a.then(function(e){return e.errorMessage?Promise.reject(new F(e.errorMessage)):e.content})}else r=(new Y).getResourceAsync(i);else r=this.resolver.contentAsync(i)}if(null==r)return Promise.resolve(null);var o=S.isWebPath(this.rootPath)==S.isWebPath(i)?b.relative(this.rootPath,i):i;return r.then(function(e){if(null==e)return Promise.reject(new F("Can note resolve "+i));var t=P.stringStartsWith(e,"#%RAML"),r=new K(o,e,t,n,i);return n.pathToUnit[i]=r,r},function(e){return"object"==typeof e&&e instanceof F?Promise.reject(e):Promise.reject(new F(e.toString()))}).then(function(e){return e.isRAMLUnit()?e:D.isScheme(e.contents())?D.startDownloadingReferencesAsync(e.contents(),e):e})},e.prototype.visualizeNewlines=function(e){for(var t="",n=0;n<e.length;n++){var r=e[n];"\r"==r&&(r="\\r"),"\n"==r&&(r="\\n"),t+=r}return t},e.prototype.indent=function(e){var t=e.unit().contents();if(e==e.root())return"";var n=G(e,t),r=a(e.dump());if(r.length>1&&r[1].trim().length>0){var i=s(r[1]);return n+i}return n+"  "},e.prototype.startIndent=function(e){e.unit().contents();if(e==e.root())return"";var t=a(e.dump());if(t.length>0){console.log("FIRST: "+t[0]);var n=s(t[0]);return n+"  "}return""},e.prototype.canWriteInOneLine=function(e){return!1},e.prototype.isOneLine=function(e){return e.text().indexOf("\n")<0},e.prototype.recalcPositionsUp=function(e){for(var t=e;t;)t.recalcEndPositionFromChilds(),t=t.parent()},e.prototype.add2=function(e,t,n,r,i){void 0===i&&(i=!1);var a=e.unit(),o=(e.root(),null);if(r&&(z.isInstance(r)&&(o=r),Q.isInstance(r)&&(o=r.point)),e.isValueInclude()){var s=e.children();if(0==s.length)throw new F("not implemented: insert into empty include ref");var u=s[0].parent();return void this.add2(u,t,n,o,i)}var l=(new I.TextRange(a.contents(),t.start(),t.end()),new I.TextRange(a.contents(),e.start(),e.end()),e.unit().contents());e.valueKind()==w.Kind.SEQ&&(e=d(e.valueAsSeq(),e,e.unit()));var i=this.isJson(e),p=i?"":this.indent(e.isSeq()?e.parent():e),c=p,f=p.length,h=e.isSeq()||e.isMapping()&&(e.isValueSeq()||e.isValueScalar()||!e.asMapping().value);n=n,n&&(i||h&&(c+="  ",f+=2));var m=new B(c);t.markupNode(m,t._actualNode(),0,i);var y=m.text;if(n){var v=I.trimEnd(y),g=y.length-v.length;if(g>0){var A=y.length;y=y.substring(0,A-g),t.shiftNodes(A-g,-g)}}n&&!i?(t.highLevelNode(),e.isMapping(),y=e.isSeq()||e.isMapping()&&(e.isValueSeq()||e.isValueScalar()||!e.asMapping().value)?p+"- "+y:p+y):y=p+y;var E=e.end();if(o)o!=e?E=o.end():i&&n||(E=e.keyEnd()+1,E=new I.TextRange(l,E,E).extendAnyUntilNewLines().endpos());else if(i&&n){var T=e.asSeq();T&&(E=T.items.length>0?T.items[T.items.length-1].endPosition:T.endPosition-1)}else if(r&&Q.isInstance(r)){var S=r;S.type==J.START&&(E=e.keyEnd()+1,E=new I.TextRange(l,E,E).extendAnyUntilNewLines().endpos())}var b=new I.TextRange(l,0,E);if(E=b.extendToNewlines().reduceSpaces().endpos(),i&&e.isSeq()){var T=e.asSeq();T.items.length>0&&(y=", "+y,f+=2)}else E>0&&"\n"!=l[E-1]&&(y="\n"+y,f++);var _=0;n&&!i&&(y+="\n",_++);var N=l.substring(0,E)+y+l.substring(E,l.length),R=a;if(R.updateContentSafe(N),this.executeReplace(new I.TextRange(l,E,E),y,R),e.root().shiftNodes(E,f+(t.end()-t.start())+_),o){for(var s=e.children(),M=-1,C=0;C<s.length;C++){var L=s[C];if(L.start()==o.start()&&L.end()==o.end()){M=C;break}}M>=0?e.addChild(t,M+1):e.addChild(t)}else e.addChild(t);t.shiftNodes(0,E+f),this.recalcPositionsUp(e),t.setUnit(e.unit()),t.visit(function(t){var n=t;return n.setUnit(e.unit()),!0})},e.prototype.isJsonMap=function(e){if(!e.isMap())return!1;var t=e.text().trim();return t.length>=2&&"{"==t[0]&&"}"==t[t.length-1]},e.prototype.isJsonSeq=function(e){if(!e.isSeq())return!1;var t=e.text().trim();return t.length>=2&&"["==t[0]&&"]"==t[t.length-1]},e.prototype.isJson=function(e){return this.isJsonMap(e)||this.isJsonSeq(e)},e.prototype.remove=function(e,t,n){var r=n.parent();if(n._oldText=n.dump(),this.isOneLine(n)&&n.isMapping()&&n.parent().isMap()){var i=n.parent();if(1==i.asMap().mappings.length&&null!=i.parent())return void this.remove(e,i.parent(),i)}if(this.isOneLine(n)&&n.isScalar()&&n.parent().isSeq()){var a=n.parent(),o=a.asSeq();if(1==o.items.length)return void this.remove(e,a.parent(),a)}if(t.isMapping()&&n.isSeq()){var s=t.parent();return void this.remove(e,s,t)}var u=new I.TextRange(e.contents(),n.start(),n.end()),l=(new I.TextRange(e.contents(),t.start(),t.end()),new I.TextRange(e.contents(),r.start(),r.end()),u.startpos());if(t.isSeq()){var p=n.isSeq()?n:n.parentOfKind(w.Kind.SEQ);u=p&&this.isJson(p)?u.extendSpaces().extendCharIfAny(",").extendSpaces():u.extendToStartOfLine().extendAnyUntilNewLines().extendToNewlines();
}t.isMap()&&(u=u.trimEnd().extendAnyUntilNewLines().extendToNewlines(),u=u.extendToStartOfLine().extendUntilNewlinesBack()),t.kind()==w.Kind.MAPPING&&(this.isJson(n)&&this.isOneLine(n)||(u=u.extendSpacesUntilNewLines(),u=u.extendToNewlines(),u=u.extendToStartOfLine().extendUntilNewlinesBack())),n.isSeq()&&(u=u.reduceSpaces());var c=e;c.updateContentSafe(u.remove()),this.executeReplace(u,"",c),n.parent().removeChild(n);var f=-u.len();t.root().shiftNodes(l,f),this.recalcPositionsUp(t)},e.prototype.changeKey=function(e,t,n){var r=new I.TextRange(t.unit().contents(),t.keyStart(),t.keyEnd());if(t.kind()==w.Kind.MAPPING){var i=t._actualNode().key;i.value=n,i.endPosition=i.startPosition+n.length}var a=e;this.executeReplace(r,n,a);var o=n.length-r.len();t.root().shiftNodes(r.startpos(),o,t),this.recalcPositionsUp(t)},e.prototype.executeReplace=function(e,t,n){var r=new S.TextChangeCommand(e.startpos(),e.endpos()-e.startpos(),t,n);n.project();try{this.tlisteners.forEach(function(e){return e(r)})}catch(i){return!1}var a=e.replace(t);return n.updateContentSafe(a),!0},e.prototype.changeValue=function(e,t,n){var r,i=new I.TextRange(t.unit().contents(),t.start(),t.end()),a=0,o=0,s=null,u=null;if(t.kind()==w.Kind.SCALAR){if("string"!=typeof n)throw new F("not implemented");t.asScalar().value=n,r=n}else if(t.kind()==w.Kind.MAPPING){if(u=t.asMapping(),t.isValueInclude()){var l=t.valueAsInclude(),p=l.value,f=p,d=t.unit().resolve(f);if(null==d)return void console.log("attr.setValue: couldn't resolve: "+f);if(d.isRAMLUnit())return;return void(O.getIncludeReference(p)||d.updateContent(n))}if(i=u.value?i.withStart(t.valueStart()).withEnd(t.valueEnd()):i.withStart(t.keyEnd()+1).withEnd(t.keyEnd()+1),i=i.reduceNewlinesEnd(),null==n)r="",u.value=null;else if("string"==typeof n||null==n){var h=n,m=this.indent(t);if(h&&I.isMultiLine(h)&&(h=""+I.makeMutiLine(h,m.length/2)),r=h,u.value){if(u.value.kind==w.Kind.SEQ){console.log("seq value");u.value.items[0];throw"assign value!!!"}if(u.value.kind==w.Kind.SCALAR){var y=u.value,v=y.value||"";y.value=h,u.value.endPosition=u.value.startPosition+h.length,u.endPosition=u.value.endPosition,o+=h.length-v.length}}else if(console.log("no value"),u.value=w.newScalar(h),u.value.startPosition=t.keyEnd()+1,u.value.endPosition=u.value.startPosition+h.length,u.endPosition=u.value.endPosition,e.contents().length>t.keyEnd()+1){var g=t.keyEnd()+1;":"==e.contents()[g-1]&&(r=" "+r,u.value.startPosition++,u.value.endPosition++,u.endPosition++,o++)}}else{var A=n;if(A.isMapping())n=c([A.asMapping()]),A=n;else if(!A.isMap())throw new F("only MAP/MAPPING nodes allowed as values");var E=new B("");A.markupNode(E,A._actualNode(),0,!0),r=""+E.text,A.shiftNodes(0,i.startpos()+o),s=A}}else console.log("Unsupported change value case: "+t.kindName());var T=e;this.executeReplace(i,r,T);var S=r.length-i.len();t.root().shiftNodes(i.endpos()+a,S,t),s&&(u.value=s._actualNode()),this.recalcPositionsUp(t)},e.prototype.initWithRoot=function(e,t){var n=e.end();t.markup(!1),t._actualNode().startPosition=n,t._actualNode().endPosition=n,t.setUnit(e.unit())},e.prototype.execute=function(e){var t=this;e.commands.forEach(function(e){switch(e.kind){case S.CommandKind.CHANGE_VALUE:var n=e.target,r=n.value();r||(r="");var i=e.value;if(console.log("set value: "+typeof r+" ==> "+typeof i),"string"!=typeof r&&"number"!=typeof r&&"boolean"!=typeof r||"string"!=typeof i)if("string"!=typeof r&&"number"!=typeof r&&"boolean"!=typeof r||"string"==typeof i)if("string"!=typeof r&&"string"==typeof i){var a=e.value;if(r.kind()!=w.Kind.MAPPING)throw new F("unsupported case: attribute value conversion: "+typeof r+" ==> "+typeof i+" not supported");I.isMultiLine(a)?(n.children().forEach(function(e){t.remove(n.unit(),n,e)}),t.changeValue(n.unit(),n,a)):t.changeKey(n.unit(),r,a)}else{if("string"==typeof r||"string"==typeof i)throw new F("shouldn't be this case: attribute value conversion "+typeof r+" ==> "+typeof i+" not supported");var o=i;if(o.isMapping()&&(i=c([o.asMapping()])),r==i)break;var s=i;s.asMap();n.children().forEach(function(e){t.remove(n.unit(),n,e)}),s.children().forEach(function(e){}),t.changeValue(n.unit(),n,i)}else t.changeValue(n.unit(),n,i);else r!=i&&t.changeValue(n.unit(),n,i);return;case S.CommandKind.CHANGE_KEY:var n=e.target;return void t.changeKey(n.unit(),n,e.value);case S.CommandKind.ADD_CHILD:var n=e.target,u=e.value;return void t.add2(n,u,e.toSeq,e.insertionPoint);case S.CommandKind.REMOVE_CHILD:var l=e.target,s=e.value;return void t.remove(l.unit(),l,s);case S.CommandKind.INIT_RAML_FILE:var p=e.target,f=e.value;return void t.initWithRoot(p,f);default:return void console.log("UNSUPPORTED COMMAND: "+S.CommandKind[e.kind])}})},e.prototype.replaceYamlNode=function(e,t,n,r,i){var a=w.load(t,{ignoreDuplicateKeys:!0});this.updatePositions(e.start(),a),e.root().shiftNodes(n,r);var o=e.parent(),s=e._actualNode(),u=s.parent;if(a.parent=u,o&&o.kind()==w.Kind.MAP){var l=o._actualNode();l.mappings=l.mappings.map(function(e){return e!=s?e:a})}e.updateFrom(a),this.recalcPositionsUp(e)},e.prototype.executeTextChange2=function(e){var t=e.unit,n=t.contents(),r=e.target;if(r){var i=n.substring(r.start(),r.end());n=n.substr(0,e.offset)+e.text+n.substr(e.offset+e.replacementLength);var a=i.substr(0,e.offset-r.start())+e.text+i.substr(e.offset-r.start()+e.replacementLength);if(t.updateContentSafe(n),e.offset>r.start())try{var o=e.text.length-e.replacementLength,s=e.offset;r.unit().project().replaceYamlNode(r,a,s,o,e.unit)}catch(u){console.log("New node contents (causes error below): \n"+a),console.log("Reparse error: "+u.stack)}}else n=n.substr(0,e.offset)+e.text+n.substr(e.offset+e.replacementLength);t.updateContent(n),this.listeners.forEach(function(e){e(null)}),this.tlisteners.forEach(function(t){t(e)})},e.prototype.executeTextChange=function(e){(new Date).getTime();try{var t=e.unit.contents(),n=e.target;null==n&&(n=this.findNode(e.unit.ast(),e.offset,e.offset+e.replacementLength));var r=e.unit;if(n){var a=t.substring(n.start(),n.end()),o=t;t=t.substr(0,e.offset)+e.text+t.substr(e.offset+e.replacementLength);var s=a.substr(0,e.offset-n.start())+e.text+a.substr(e.offset-n.start()+e.replacementLength);r.updateContentSafe(t);u(o,e);if(e.offset>n.start())try{var l=w.load(s,{ignoreDuplicateKeys:!0});this.updatePositions(n.start(),l);var p=e.text.length-e.replacementLength;if(e.unit.ast().shiftNodes(e.offset,p),null!=l&&l.kind==w.Kind.MAP){var c=l.mappings[0],f=n._actualNode(),d=f.parent,h=new S.ASTDelta,m=e.unit;if(h.commands=[new S.ASTChangeCommand(S.CommandKind.CHANGE_VALUE,new z(i(f),m,null,null,null),new z(c,m,null,null,null),0)],d&&d.kind==w.Kind.MAP){var y=d;y.mappings=y.mappings.map(function(e){return e!=f?e:c})}return c.parent=d,this.recalcPositionsUp(n),n.updateFrom(c),this.listeners.forEach(function(e){e(h)}),void this.tlisteners.forEach(function(t){t(e)})}}catch(v){console.log("New node contents (causes error below): \n"+s),console.log("Reparse error: "+v.stack)}}else t=t.substr(0,e.offset)+e.text+t.substr(e.offset+e.replacementLength);(new Date).getTime();
//!find node in scope
r.updateContent(t),this.listeners.forEach(function(e){e(null)}),this.tlisteners.forEach(function(t){t(e)})}finally{(new Date).getTime()}},e.prototype.updatePositions=function(e,t){var n=this;if(null!=t)switch(-1==t.startPosition?t.startPosition=e:t.startPosition=e+t.startPosition,t.endPosition=e+t.endPosition,t.kind){case w.Kind.MAP:var r=t;r.mappings.forEach(function(t){return n.updatePositions(e,t)});break;case w.Kind.MAPPING:var i=t;this.updatePositions(e,i.key),this.updatePositions(e,i.value);break;case w.Kind.SCALAR:break;case w.Kind.SEQ:var a=t;a.items.forEach(function(t){return n.updatePositions(e,t)})}},e.prototype.findNode=function(e,t,n){var r=this;if(null==e)return null;var i=e;if(e.start()<=t&&e.end()>=n){var a=e;return i.directChildren().forEach(function(e){var i=r.findNode(e,t,n);i&&(a=i)}),a}return null},e.prototype.addTextChangeListener=function(e){this.tlisteners.push(e)},e.prototype.removeTextChangeListener=function(e){this.tlisteners=this.tlisteners.filter(function(t){return t!=e})},e.prototype.addListener=function(e){this.listeners.push(e)},e.prototype.removeListener=function(e){this.listeners=this.listeners.filter(function(t){return t!=e})},e.prototype.namespaceResolver=function(){return this._namespaceResolver},e}();t.Project=X;var z=function(){function e(e,t,n,r,i,a,o){void 0===a&&(a=!1),void 0===o&&(o=!1),this._node=e,this._unit=t,this._parent=n,this._anchor=r,this._include=i,this.cacheChildren=a,this._includesContents=o,this._errors=[]}return e.isInstance=function(t){return null!=t&&t.getClassIdentifier&&"function"==typeof t.getClassIdentifier&&R.contains(t.getClassIdentifier(),e.CLASS_IDENTIFIER)},e.prototype.getClassIdentifier=function(){var t=[];return t.concat(e.CLASS_IDENTIFIER)},e.prototype.actual=function(){return this._node},e.prototype.yamlNode=function(){return this._node},e.prototype.includesContents=function(){return this._includesContents},e.prototype.setIncludesContents=function(e){this._includesContents=e},e.prototype.gatherIncludes=function(t,n,r,i){var a=this;if(void 0===t&&(t=[]),void 0===n&&(n=null),void 0===r&&(r=null),void 0===i&&(i=!0),null!=this._node){var o=this._node.kind;if(o!=w.Kind.SCALAR)if(o==w.Kind.MAP){var s=this._node;1!=s.mappings.length||i?s.mappings.map(function(t){return new e(t,a._unit,a,r?r:a._anchor,n?n:a._include,a.cacheChildren)}).forEach(function(e){return e.gatherIncludes(t)}):new e(s.mappings[0].value,this._unit,this,n,r,this.cacheChildren).gatherIncludes(t)}else if(o==w.Kind.MAPPING){var u=this._node;null==u.value||new e(u.value,this._unit,this,r?r:this._anchor,n?n:this._include,this.cacheChildren).gatherIncludes(t)}else if(o==w.Kind.SEQ){var l=this._node;l.items.filter(function(e){return null!=e}).map(function(t){return new e(t,a._unit,a,r?r:a._anchor,n?n:a._include,a.cacheChildren)}).forEach(function(e){return e.gatherIncludes(t)})}else o==w.Kind.INCLUDE_REF&&this._unit&&t.push(this);else if(D.isScheme(this._node.value)){var p=D.getReferences(this._node.value,this.unit());p.forEach(function(n){var r=w.newScalar(n);r.kind=w.Kind.INCLUDE_REF;var i=new e(r,a.unit(),null,null,null);t.push(i)})}}},e.prototype.setHighLevelParseResult=function(e){this._highLevelParseResult=e},e.prototype.highLevelParseResult=function(){return this._highLevelParseResult},e.prototype.setHighLevelNode=function(e){this._highLevelNode=e},e.prototype.highLevelNode=function(){return this._highLevelNode},e.prototype.start=function(){return this._node.startPosition},e.prototype.errors=function(){return this._errors},e.prototype.addIncludeError=function(e){if(!R.find(this._errors,function(t){return t.message==e.message})){var t=this._node;t.includeErrors||(t.includeErrors=[]),R.find(t.includeErrors,function(t){return t.message==e.message})||(this._errors.push(e),t.includeErrors.push(e))}},e.prototype.parent=function(){return this._parent},e.prototype.recalcEndPositionFromChilds=function(){var e=(this.children(),this.children()[0]),t=this.children()[this.children().length-1];if(this.isMapping()){var n=this.asMapping();if(n.value)if(n.value.kind==w.Kind.MAP){var r=n.value;r.startPosition<0&&e&&(r.startPosition=e.start()),t&&(this._node.endPosition=t._node.endPosition),this._node.endPosition=Math.max(this._node.endPosition,n.value.endPosition)}else if(n.value.kind==w.Kind.SEQ){var i=n.value;if(i.startPosition<0&&i.items.length>0){var a=i.items[0].startPosition,o=new I.TextRange(this.unit().contents(),a,a);o=o.extendSpacesBack().extendCharIfAnyBack("-"),i.startPosition=o.startpos()}if(i.items.length>0){var s=i.items[i.items.length-1];this._node.endPosition=Math.max(this._node.endPosition,i.endPosition,s.endPosition),i.endPosition=Math.max(this._node.endPosition,i.endPosition,s.endPosition)}}else n.value.kind==w.Kind.SCALAR||t&&(this._node.endPosition=t._node.endPosition)}else t&&(this._node.endPosition=t._node.endPosition)},e.prototype.isValueLocal=function(){if(this._node.kind==w.Kind.MAPPING){var e=this._node.value.kind;return e!=w.Kind.INCLUDE_REF&&e!=w.Kind.ANCHOR_REF}return!0},e.prototype.keyStart=function(){return this._node.kind==w.Kind.MAPPING?this._node.key.startPosition:-1},e.prototype.keyEnd=function(){return this._node.kind==w.Kind.MAPPING?this._node.key.endPosition:-1},e.prototype.valueStart=function(){if(this._node.kind==w.Kind.MAPPING){var e=this.asMapping();return e.value?e.value.startPosition:e.endPosition}return this._node.kind==w.Kind.SCALAR?this.start():-1},e.prototype.valueEnd=function(){if(this._node.kind==w.Kind.MAPPING){var e=this.asMapping();return e.value?e.value.endPosition:e.endPosition}return this._node.kind==w.Kind.SCALAR?this.end():-1},e.prototype.end=function(){return this._node.endPosition},e.prototype.dump=function(){if(this._oldText)return this._oldText;if(this._unit&&this._node.startPosition>0&&this._node.endPosition>0){var e=this._unit.contents().substring(this._node.startPosition,this._node.endPosition);return e=o(e,G(this,this._unit.contents()))}return w.dump(this.dumpToObject(),{})},e.prototype.dumpToObject=function(e){return void 0===e&&(e=!1),this.dumpNode(this._node,e)},e.prototype.dumpNode=function(e,t){var n=this;if(void 0===t&&(t=!1),!e)return null;if(e.kind==w.Kind.INCLUDE_REF){if(this._unit){var r=e,i=r.value,a=null;try{a=this._unit.resolve(i)}catch(o){}if(null==a)return null;if(a.isRAMLUnit()&&this.canInclude(a)){var s=this.unit(),u=a;u.addIncludedBy(s.absolutePath()),s.getIncludedByPaths().forEach(function(e){u.addIncludedBy(e)});var p=a.ast();if(p)return p.dumpToObject(t)}else if(this.canInclude(a))return a.contents()}return null}if(e.kind==w.Kind.SEQ){var c=e,f=[];return c.items.forEach(function(e){return f.push(n.dumpNode(e,t))}),f}if(e.kind==w.Kind.ANCHOR_REF){var d=e;return this.dumpNode(d.value,t)}if(e.kind==w.Kind.MAPPING){var h=e,m={},y=h.value,v=this.dumpNode(y,t);return m[""+this.dumpNode(h.key,t)]=v,m}if(e.kind==w.Kind.SCALAR){var r=e,g=r.value;return r.plainScalar&&(g=l(g)),g}if(e.kind==w.Kind.MAP){var A=e,E={};return A.mappings&&A.mappings.forEach(function(e){var r=n.dumpNode(e.value,t);null==r&&(r=t?"!$$$novalue":r),E[n.dumpNode(e.key,t)+""]=r}),E}},e.prototype.keyKind=function(){return this._node.key?this._node.key.kind:null},e.prototype._actualNode=function(){return this._node},e.prototype.execute=function(e){this.unit()?this.unit().project().execute(e):e.commands.forEach(function(e){switch(e.kind){case S.CommandKind.CHANGE_VALUE:var t=e.target,n=e.value,r=t._actualNode();t.start();return void(r.kind==w.Kind.MAPPING&&(r.value=w.newScalar(""+n)));case S.CommandKind.CHANGE_KEY:var t=e.target,n=e.value,r=t._actualNode();if(r.kind==w.Kind.MAPPING){var i=r.key;i.value=n}return}})},e.prototype.updateFrom=function(e){this._node=e},e.prototype.isAnnotatedScalar=function(){if(!this.unit())return!1;for(var e,t=this;!e&&t;)e=t.highLevelNode(),t=t.parent();if(e){var n=void 0;if(e.isElement())n=e.asElement().definition();else{var r=e.property();n=r&&(r.domain()||r.range())}if(n){var i=n.universe().version();if("RAML08"==i)return!1}}var a;if(this.kind()==w.Kind.MAPPING&&this.valueKind()==w.Kind.MAP?a=this._node.value.mappings:null==this.key()&&this.kind()==w.Kind.MAP&&(a=this._node.mappings),a){var o=a.length>0;return a.forEach(function(e){"value"!==e.key.value&&(e.key.value&&"("==e.key.value.charAt(0)&&")"==e.key.value.charAt(e.key.value.length-1)||(o=!1))}),o}return!1},e.prototype.value=function(t){if(!this._node)return"";if(this._node.kind==w.Kind.SCALAR){if("~"===this._node.value&&null===this._node.valueObject)return t?"null":null;if(!t&&""+this._node.valueObject===this._node.value)return this._node.valueObject;var n=this._node.value;return t||this._node.plainScalar&&(n=l(n)),n}if(this._node.kind==w.Kind.ANCHOR_REF){var r=this._node;return new e(r.value,this._unit,this,null,null).value(t)}if(this._node.kind==w.Kind.MAPPING){var i=this._node;if(null==i.value)return null;if(this.isAnnotatedScalar())for(var a=new e(i.value,this._unit,this,null,null),o=a.children(),s=0;s<o.length;s++)if("value"===o[s].key())return o[s].valueKind()==w.Kind.SEQ?o[s].children().map(function(e){return e.value()}):o[s].value();return new e(i.value,this._unit,this,null,null).value(t)}if(this._node.kind==w.Kind.INCLUDE_REF){var u=this._node.value,p=u,c=null;try{c=this._unit.resolve(p)}catch(f){return"can not resolve "+p+" due to: "+f.message}if(null==c)return"can not resolve "+p;if(c.isRAMLUnit()){var d=c.ast();return d?d.value():"can not resolve "+p+" due to: file is empty"}var h=c.contents();return I.isMultiLineValue(h)&&(h=I.fromMutiLine(h)),h}if(this._node.kind==w.Kind.MAP){var m=this._node;if(1==m.mappings.length)return new e(m.mappings[0],this._unit,this,null,null);if(this.isAnnotatedScalar())for(var o=this.children(),s=0;s<o.length;s++)if("value"===o[s].key())return o[s].valueKind()==w.Kind.SEQ?o[s].children().map(function(e){return e.value()}):o[s].value()}if(this._node.kind==w.Kind.SEQ){var y=this._node;if(1==y.items.length)return new e(y.items[0],this._unit,this,null,null).value(t)}return null},e.prototype.printDetails=function(e){var t="";e||(e="");var n=this.kindName();return this.kind()==w.Kind.SCALAR?t+=e+"["+n+"] "+this.value()+"\n":this.kind()==w.Kind.MAPPING&&this._node.value&&this._node.value.kind==w.Kind.SCALAR?t+=e+"["+n+"] "+this.key()+" = "+this.value()+"\n":this.kind()==w.Kind.MAPPING?(t+=e+"["+n+"] "+this.key()+" = :\n",this.children().forEach(function(n){t+=n.printDetails(e+"	")})):(t+=e+"["+n+"] :\n",this.children().forEach(function(n){t+=n.printDetails(e+"	")})),t},e.prototype.visit=function(e){this.children().forEach(function(t){e(t)&&t.visit(e)})},e.prototype.rawKey=function(){if(!this._node)return"";if(this._node.kind==w.Kind.MAPPING){var e=this._node;if(e.key.kind==w.Kind.SEQ){var t=e.key,n="[";return t.items.forEach(function(e){return n+=e.value}),n+"]"}return e.key.value}if(this._node.kind==w.Kind.INCLUDE_REF){var r=this.children();if(1==r.length)return r[0].key()}return null},e.prototype.key=function(e){void 0===e&&(e=!1);var t=this.rawKey();return e?t:(null!=t&&P.stringEndsWith(t,"?")&&(t=t.substring(0,t.length-1)),t)},e.prototype.optional=function(){var e=this.rawKey();return null!=e&&P.stringEndsWith(e,"?")},e.prototype.addChild=function(e,t){void 0===t&&(t=-1);var n=e;if(n._parent=this,this._oldText=null,this.isMap()){var r=this.asMap();(null==r.mappings||void 0==r.mappings)&&(r.mappings=[]),t>=0?r.mappings.splice(t,0,n.asMapping()):r.mappings.push(n.asMapping())}else if(this.isMapping()){var i=this.asMapping(),a=i.value;if(!i.value&&n.isMap())return void(i.value=n._actualNode());if(i.value&&i.value.kind==w.Kind.SCALAR&&(i.value=null,a=null),a||(a=n.isScalar()||n.highLevelNode()&&n.highLevelNode().property().getAdapter(C.RAMLPropertyParserService).isEmbedMap()?w.newSeq():w.newMap(),i.value=a),a.kind==w.Kind.MAP){var r=a;(null==r.mappings||void 0==r.mappings)&&(r.mappings=[]),n.isScalar(),t>=0?r.mappings.splice(t,0,n.asMapping()):r.mappings.push(n.asMapping())}else{if(a.kind!=w.Kind.SEQ)throw new F("Insert into mapping with "+w.Kind[i.value.kind]+" value not supported");var o=a;t>=0?o.items.splice(t,0,n._actualNode()):o.items.push(n._actualNode())}}else{if(!this.isSeq())throw new F("Insert into "+this.kindName()+" not supported");var o=this.asSeq();t>=0?o.items.splice(t,0,n._actualNode()):o.items.push(n._actualNode())}},e.prototype.removeChild=function(e){this._oldText=null;var t,n,r=e;if(this.kind()==w.Kind.SEQ){var i=this.asSeq();t=r._node,n=i.items.indexOf(t),n>-1&&i.items.splice(n,1)}else if(this.kind()==w.Kind.MAP){var a=this.asMap();t=r.asMapping(),n=a.mappings.indexOf(t),n>-1&&a.mappings.splice(n,1)}else{if(this.kind()!=w.Kind.MAPPING)throw new F("Delete from "+w.Kind[this.kind()]+" unsupported");var o=this.asMapping();if(r._actualNode()==o.value)o.value=null;else{var a=o.value;t=r.asMapping(),a&&a.mappings&&(n=a.mappings.indexOf(t),n>-1&&a.mappings.splice(n,1))}}},e.prototype.hasInnerIncludeError=function(){return this.innerIncludeErrors},e.prototype.includeErrors=function(){if(this._node.kind==w.Kind.MAPPING){var t=this._node;if(null==t.value)return[];var n=new e(t.value,this._unit,this,this._anchor,this._include),r=n.includeErrors();return this.innerIncludeErrors=n.hasInnerIncludeError(),r}var i=[];if(this._node.kind==w.Kind.INCLUDE_REF){var t=this._node;if(null==t.value)return[];var a=this.includePath(),o=null;try{o=this._unit.resolve(a)}catch(s){this.innerIncludeErrors=s.inner;var u="Can not resolve "+a+" due to: "+s.message;return i.push(u),i}var l=this._node.includeErrors;if(l&&l.length>0)return this.innerIncludeErrors=!0,i=l.map(function(e){return"object"==typeof e&&e instanceof s?e.message:e.toString()});if(null==o)return i.push("Can not resolve "+a),i;if(o.isRAMLUnit()){var p=o.ast();if(p)return[];i.push(""+a+" can not be parsed")}}return i},e.prototype.joinMappingsWithFullIncludeAnchor=function(t,n,r){var i=this,a=R.find(t,function(e){return e.key&&e.value&&e.key.kind==w.Kind.SCALAR&&"<<"==e.key.value&&e.value.kind==w.Kind.ANCHOR_REF});if(!a)return t.map(function(t){return new e(t,i._unit,i,r?r:i._anchor,n?n:i._include,i.cacheChildren)});var o=R.filter(t,function(e){return!(e.key.kind==w.Kind.SCALAR&&"<<"==e.key.value&&e.value.kind==w.Kind.ANCHOR_REF)}),s=new e(a.value,this._unit,this,n,r,this.cacheChildren).children(null,null,!0),u=o.map(function(t){return new e(t,i._unit,i,r?r:i._anchor,n?n:i._include,i.cacheChildren)});return u.concat(s)},e.prototype.children=function(t,n,r){var i=this;if(void 0===t&&(t=null),void 0===n&&(n=null),void 0===r&&(r=!0),null==this._node)return[];if(this.cacheChildren&&this._children)return this._children;var a,o=this._node.kind;if(o==w.Kind.SCALAR)a=[];else if(o==w.Kind.MAP){var s=this._node;a=1!=s.mappings.length||r?this.joinMappingsWithFullIncludeAnchor(s.mappings,t,n):new e(s.mappings[0].value,this._unit,this,t,n,this.cacheChildren).children(null,null,!0)}else if(o==w.Kind.MAPPING){var u=this._node;if(null==u.value)a=[];else{var l=new e(u.value,this._unit,this,n?n:this._anchor,t?t:this._include,this.cacheChildren);a=l.children(),l.includesContents()&&this.setIncludesContents(!0)}}else if(o==w.Kind.SEQ){var p=this._node;a=p.items.filter(function(e){return null!=e}).map(function(r){return new e(r,i._unit,i,n?n:i._anchor,t?t:i._include,i.cacheChildren)})}else if(o==w.Kind.INCLUDE_REF){if(this._unit){var c=this.includePath(),f=null;try{f=this._unit.resolve(c)}catch(d){}if(null==f)a=[];else if(f.isRAMLUnit())if(this.canInclude(f)){var h=this.unit(),m=f;m.addIncludedBy(h.absolutePath()),h.getIncludedByPaths().forEach(function(e){m.addIncludedBy(e)});var y=f.ast();y&&(this.cacheChildren&&(y=v(y)),a=f.ast().children(this,null),this.setIncludesContents(!0))}else this.addIncludeError&&this.addIncludeError(new d("Recursive definition"))}a||(a=[])}else{if(o!=w.Kind.ANCHOR_REF)throw new d("Should never happen; kind : "+w.Kind[this._node.kind]);var g=this._node;a=new e(g.value,this._unit,this,null,null,this.cacheChildren).children()}return this.cacheChildren&&(this._children=a),a},e.prototype.canInclude=function(e){for(var t=this.includedFrom();null!=t;){if(t.unit().absolutePath()==e.absolutePath())return!1;t=t.includedFrom()}return!this.unit().includedByContains(e.absolutePath())},e.prototype.directChildren=function(t,n,r){var i=this;if(void 0===t&&(t=null),void 0===n&&(n=null),void 0===r&&(r=!0),this._node){switch(this._node.kind){case w.Kind.SCALAR:return[];case w.Kind.MAP:var a=this._node;return 1!=a.mappings.length||r?a.mappings.map(function(r){return new e(r,i._unit,i,n?n:i._anchor,t?t:i._include)}):new e(a.mappings[0].value,this._unit,this,t,n).directChildren(null,null,!0);case w.Kind.MAPPING:var o=this._node;return null==o.value?[]:new e(o.value,this._unit,this,n?n:this._anchor,t?t:this._include).directChildren();case w.Kind.SEQ:var s=this._node;return s.items.filter(function(e){return null!=e}).map(function(r){return new e(r,i._unit,i,n?n:i._anchor,t?t:i._include)});case w.Kind.INCLUDE_REF:return[];case w.Kind.ANCHOR_REF:return[]}throw new F("Should never happen; kind : "+w.Kind[this._node.kind])}return[]},e.prototype.anchorId=function(){return this._node.anchorId},e.prototype.unit=function(){return this._unit},e.prototype.containingUnit=function(){return this.valueKind()==w.Kind.INCLUDE_REF?this.unit().resolve(this._node.value.value):this.kind()==w.Kind.INCLUDE_REF?this.unit().resolve(this._node.value):this._unit},e.prototype.includeBaseUnit=function(){return this._unit},e.prototype.setUnit=function(e){this._unit=e},e.prototype.includePath=function(){var e=this.getIncludeString();return e?e:null},e.prototype.includeReference=function(){var e=this.getIncludeString();return e?O.getIncludeReference(e):null},e.prototype.getIncludeString=function(){if(!this._node)return null;if(this._node.kind==w.Kind.INCLUDE_REF){var t=this._node.value;return t}if(this._node.kind==w.Kind.MAPPING){var n=this._node;return null==n.value?null:new e(n.value,this._unit,this,null,null).getIncludeString()}if(this._node.kind==w.Kind.ANCHOR_REF){var r=this._node.value;return new e(r,this._unit,this,null,null).getIncludeString()}return null},e.prototype.anchoredFrom=function(){return this._anchor},e.prototype.includedFrom=function(){return this._include},e.prototype.kind=function(){return this._actualNode().kind},e.prototype.valueKind=function(){if(this._node.kind!=w.Kind.MAPPING)return null;var e=this._node;return e.value?e.value.kind:null},e.prototype.anchorValueKind=function(){if(this.valueKind()==w.Kind.ANCHOR_REF){var e=this._node.value;return e&&e.value&&e.value.kind}return null},e.prototype.resolvedValueKind=function(){var e=this.valueKind();if(e==w.Kind.ANCHOR_REF){var t=this.anchorValueKind();if(t==w.Kind.INCLUDE_REF){var n=this._node.value,r=n.value.value;return this.unitKind(r)}return t}if(e==w.Kind.INCLUDE_REF){var r=this.includePath();return this.unitKind(r)}return e},e.prototype.unitKind=function(e){var t;try{t=this._unit.resolve(e)}catch(n){return null}if(null==t)return w.Kind.SCALAR;if(t.isRAMLUnit()){var r=t.ast();if(r)return r.kind()}return w.Kind.SCALAR},e.prototype.valueKindName=function(){var e=this.valueKind();return void 0!=e?w.Kind[e]:null},e.prototype.kindName=function(){return w.Kind[this.kind()]},e.prototype.indent=function(e,t){void 0===t&&(t="");for(var n="",r=0;e>r;r++)n+="  ";return n+t},e.prototype.replaceNewlines=function(e,t){void 0===t&&(t=null);for(var n="",r=0;r<e.length;r++){var i=e[r];"\r"==i&&(i=null==t?"\\r":t),"\n"==i&&(i=null==t?"\\n":t),n+=i}return n},e.prototype.shortText=function(e,t){void 0===t&&(t=50);var n=this.end()-this.start(),r=n,i=this.unit();!e&&i&&(e=i.contents());var a;if(e){var o=e;a=o?o.substring(this.start(),this.end()):"[no-text]"}else a="[no-unit]";return a="["+this.start()+".."+this.end()+"] "+n+" // "+a+" //",n>r&&(a+="..."),a=this.replaceNewlines(a)},e.prototype.nodeShortText=function(e,t,n){void 0===n&&(n=50);var r=e.endPosition-e.startPosition,i=r,a=this.unit();!t&&a&&(t=a.contents());var o;if(t){var s=t;o=s?s.substring(e.startPosition,e.endPosition):"[no-text]"}else o="[no-unit]";return o="["+e.startPosition+".."+e.endPosition+"] "+r+" // "+o+" //",r>i&&(o+="..."),o=this.replaceNewlines(o)},e.prototype.show=function(e,t,n){void 0===e&&(e=null),void 0===t&&(t=0),void 0===n&&(n=null),e&&0==t&&console.log(e);var r=this.children(),i=this.kindName(),a=this._actualNode().value;this.kind()==w.Kind.MAPPING&&(i+="["+this._actualNode().key.value+"]"),i+=a?"/"+w.Kind[a.kind]:"",0==r.length?(console.log(this.indent(t)+i+" // "+this.shortText(n)),this.isMapping()&&this.asMapping().value&&console.log(this.indent(t+1)+"// "+this.valueKindName()+": "+this.nodeShortText(this.asMapping().value,n))):(console.log(this.indent(t)+i+" { // "+this.shortText(n)),this.isMapping()&&this.asMapping().value&&console.log(this.indent(t+1)+"// "+this.valueKindName()+": "+this.nodeShortText(this.asMapping().value,n)),r.forEach(function(e){var r=e;r.show(null,t+1,n)}),console.log(this.indent(t)+"}"))},e.prototype.showParents=function(e,t){void 0===t&&(t=0),e&&0==t&&console.log(e);var n=0;if(this.parent()){var r=this.parent();n=r.showParents(null,t+1)}var i=this.kindName(),a=this._actualNode().value;return i+=a?"/"+w.Kind[a.kind]:"/null",console.log(this.indent(n)+i+" // "+this.shortText(null)),n+1},e.prototype.inlined=function(e){return e==w.Kind.SCALAR||e==w.Kind.INCLUDE_REF},e.prototype.markupNode=function(e,t,n,r){void 0===r&&(r=!1);var i=e.text.length;switch(t.kind){case w.Kind.MAP:r&&e.append("{");for(var o=t.mappings,s=0;s<o.length;s++)r&&s>0&&e.append(", "),this.markupNode(e,o[s],n,r);r&&e.append("}");break;case w.Kind.SEQ:for(var u=t.items,s=0;s<u.length;s++)e.append(this.indent(n,"- ")),this.markupNode(e,u[s],n+1,r);break;case w.Kind.MAPPING:var l=t,p=l.value;if(r){if(!p){e.append("\n");break}if(e.append(l.key.value),e.append(": "),p.kind==w.Kind.SCALAR){var c=p;e.append(c.value)}else{if(p.kind!=w.Kind.MAP)throw new F("markup not implemented: "+w.Kind[p.kind]);this.markupNode(e,l.value,n+1,r)}break}if(e.addWithIndent(n,l.key.value+":"),!p){e.append("\n");break}if(p.kind==w.Kind.SCALAR)var c=p;l.value?(e.append(this.inlined(l.value.kind)?" ":"\n"),this.markupNode(e,l.value,n+1,r)):e.append("\n");break;case w.Kind.SCALAR:var c=t;if(I.isMultiLine(c.value)){e.append("|\n");for(var f=a(c.value),s=0;s<f.length;s++)e.append(this.indent(n,f[s]));e.append("\n")}else e.append(c.value+"\n");break;case w.Kind.INCLUDE_REF:var d=t;e.append("!include "+d.value+"\n");break;default:throw new F("Unknown node kind: "+w.Kind[t.kind])}for(;i<e.text.length&&" "==e.text[i];)i++;t.startPosition=i,t.endPosition=e.text.length},e.prototype.markup=function(e){void 0===e&&(e=!1);var t=new B("");return this.markupNode(t,this._actualNode(),0,e),t.text},e.prototype.root=function(){for(var e=this;e.parent();){var t=e.parent();e=t}return e},e.prototype.parentOfKind=function(e){for(var t=this.parent();t;){if(t.kind()==e)return t;t=t.parent()}return null},e.prototype.find=function(e){var t=null;return this.directChildren().forEach(function(n){n.key()&&n.key()==e&&(t||(t=n))}),t},e.prototype.shiftNodes=function(e,t,n){if(this.directChildren().forEach(function(r){if(n&&n.start()==r.start()&&n.end()==r.end());else{r.shiftNodes(e,t,n)}}),n&&n.start()==this.start()&&n.end()==this.end());else{var r=this._actualNode();r&&$(e,r,t)}return null},e.prototype.isMap=function(){return this.kind()==w.Kind.MAP},e.prototype.isMapping=function(){return this.kind()==w.Kind.MAPPING},e.prototype.isSeq=function(){return this.kind()==w.Kind.SEQ},e.prototype.isScalar=function(){return this.kind()==w.Kind.SCALAR},e.prototype.asMap=function(){if(!this.isMap())throw new F("map expected instead of "+this.kindName());return this._actualNode()},e.prototype.asMapping=function(){if(!this.isMapping())throw new F("maping expected instead of "+this.kindName());return this._actualNode()},e.prototype.asSeq=function(){if(!this.isSeq())throw new F("seq expected instead of "+this.kindName());return this._actualNode()},e.prototype.asScalar=function(){if(!this.isScalar())throw new F("scalar expected instead of "+this.kindName());return this._actualNode()},e.prototype.isValueSeq=function(){return this.valueKind()==w.Kind.SEQ},e.prototype.isValueMap=function(){return this.valueKind()==w.Kind.MAP},e.prototype.isValueInclude=function(){return this.valueKind()==w.Kind.INCLUDE_REF},e.prototype.isValueScalar=function(){return this.valueKind()==w.Kind.SCALAR},e.prototype.valueAsSeq=function(){if(!this.isMapping())throw new F("mapping expected instead of "+this.kindName());if(this.valueKind()!=w.Kind.SEQ)throw new F("mappng/seq expected instead of mapping/"+this.kindName());return this.asMapping().value},e.prototype.valueAsMap=function(){if(!this.isMapping())throw new F("mapping expected instead of "+this.kindName());if(this.valueKind()!=w.Kind.MAP)throw new F("mappng/map expected instead of mapping/"+this.kindName());return this.asMapping().value},e.prototype.valueAsScalar=function(){if(!this.isMapping())throw new F("mapping expected instead of "+this.kindName());if(this.valueKind()!=w.Kind.SCALAR)throw new F("mappng/scalar expected instead of mapping/"+this.kindName());return this.asMapping().value},e.prototype.valueAsInclude=function(){if(!this.isMapping())throw new F("mapping expected instead of "+this.kindName());if(this.valueKind()!=w.Kind.INCLUDE_REF)throw new F("mappng/include expected instead of mapping/"+this.kindName());return this.asMapping().value},e.prototype.text=function(e){if(void 0===e&&(e=null),!e){if(!this.unit())return"[no-text]";e=this.unit().contents()}return e.substring(this.start(),this.end())},e.prototype.copy=function(){var t=i(this._actualNode());return new e(t,this._unit,this._parent,this._anchor,this._include,this._includesContents)},e.prototype.nodeDefinition=function(){return A(this)},e.CLASS_IDENTIFIER="jsyaml2lowLevel.ASTNode",e}();t.ASTNode=z,function(e){e[e.NONE=0]="NONE",e[e.START=1]="START",e[e.END=2]="END",e[e.POINT=3]="POINT"}(t.InsertionPointType||(t.InsertionPointType={}));var J=t.InsertionPointType,Q=function(){function e(e,t){void 0===t&&(t=null),this.type=e,this.point=t}return e.isInstance=function(t){return null!=t&&t.getClassIdentifier&&"function"==typeof t.getClassIdentifier&&R.contains(t.getClassIdentifier(),e.CLASS_IDENTIFIER)},e.prototype.getClassIdentifier=function(){var t=[];return t.concat(e.CLASS_IDENTIFIER)},e.after=function(t){return new e(J.POINT,t)},e.atStart=function(){return new e(J.START)},e.atEnd=function(){return new e(J.END)},e.node=function(){return new e(J.NONE)},e.prototype.show=function(e){e?(console.log(e),console.log("  insertion point type: "+J[this.type])):console.log("insertion point type: "+J[this.type]),this.type==J.POINT&&this.point&&this.point.show()},e.CLASS_IDENTIFIER="jsyaml2lowLevel.InsertionPoint",e}();t.InsertionPoint=Q,t.createNode=p,t.createMap=c,t.createScalar=f,t.createSeq=d,t.createSeqNode=h,t.createMapNode=m,t.createMapping=y,t.toChildCachingNode=v,t.toIncludingNode=g,t.getDefinitionForLowLevelNode=A,t.fetchIncludesAndMasterAsync=T},function(e,t,n){"use strict";function r(){return Object.keys(d).length>0}function i(e){f.push(e)}function a(e){d[e]=!0}function o(e){delete d[e],f.forEach(function(t){return t(e)})}function s(e){return d[e]?!0:!1}function u(e,t){c.set(e,t)}function l(e){return c.get(e)}n(65);n(66),n(72);var p=n(74),c=p(50);t.hasAsyncRequests=r,t.addLoadCallback=i;var f=[],d={};t.addNotify=a,t.removeNotity=o,t.isWaitingFor=s,t.set=u,t.get=l},function(e,t,n){"use strict";function r(e){if(!e)return e;var t=e.indexOf("#");return-1==t?e:e.substring(0,t)}function i(e){if(!e)return null;"string"!=typeof e&&(e=""+e);var t=e.indexOf("#");if(-1==t)return null;var n=t==e.length-1?"":e.substring(t+1,e.length),i=n.split("/");return 0==i.length?null:(""==i[0].trim()&&i.splice(0,1),new h(n,r(e),i))}function a(){return[new m,new v]}function o(e,t){if(!e)return t;var n=i(e);if(!n)return t;var a=r(e);return s(a,n,t).content}function s(e,t,n){var r=c.find(a(),function(t){return t.isApplicable(e,n)});return r?r.resolveReference(n,t):{content:n,validation:[]}}function u(e,t,n){if(!n)return[];var r=c.find(a(),function(t){return t.isApplicable(e,n)});return r?r.completeReference(n,t):[]}function l(e,t){return e&&t?e.lastIndexOf(t)===e.length-t.length:!1}function p(e,t){for(var n=e.getElementsByTagName(t),r=[],i=0;i<n.length;i++){var a=n[i];a.parentNode===e&&r.push(a)}return r}var c=n(70),f=n(75).DOMParser,d=n(67);t.getIncludePath=r,t.getIncludeReference=i,t.getIncludeReferenceResolvers=a,t.resolveContents=o,t.resolve=s,t.completeReference=u;var h=function(){function e(e,t,n){this.includePath=t,this.segments=n,this.originalString=e}return e.prototype.getIncludePath=function(){return this.includePath},e.prototype.getFragments=function(){return this.segments},e.prototype.asString=function(){return this.originalString},e.prototype.encodedName=function(e){return void 0===e&&(e=!0),d(this.includePath+"/"+this.asString())+(e?this.includePath.substring(this.includePath.lastIndexOf(".")):"")},e}(),m=function(){function e(){}return e.prototype.isApplicable=function(e,t){return e&&(l(e.trim(),".js")||l(e.trim(),".json"))},e.prototype.resolveReference=function(e,t){try{var n={};return n.$ref=t.getIncludePath()+"#"+t.getFragments().map(function(e){return"/"+e}).join(""),{content:JSON.stringify(n,null,2),validation:[]}}catch(r){console.log(r)}return{content:e,validation:[]}},e.prototype.completeReference=function(e,t){try{var n=JSON.parse(e),r=t.getFragments();if(!r||0==r.length)return this.getChildren(n);for(var i=n,a=t.asString().lastIndexOf("/")==t.asString().length-1,o=(a?r.length:r.length-1,0);o<r.length-1;o++){var s=r[o];if(i=this.findChild(i,s),!i)return[]}if(a)return this.getChildren(i);var u=r[r.length-1],l=[];return this.getChildren(i).forEach(function(e){0==e.indexOf(u)&&l.push(e)}),l}catch(p){console.log(p)}return[]},e.prototype.findChild=function(e,t){var n=t.replace("~1","/");return n=t.replace("~0","~"),e[n]},e.prototype.getChildren=function(e){return Object.keys(e)},e}(),y={errorHandler:{warning:function(){return null},error:function(){return null},fatalError:function(){return null}}},v=function(){function e(){}return e.prototype.isApplicable=function(e,t){return e&&(l(e.trim(),".xml")||l(e.trim(),".xsd"))},e.prototype.resolveReference=function(e,t){try{var n=new f(y).parseFromString(e),r=t.asString(),i=t.encodedName(!1),a=p(n,"xs:schema")[0],o=p(a,"xs:element"),s=p(a,"xs:complexType"),u=c.find(o,function(e){return e.getAttribute("name")===r}),l=u?c.find(s,function(e){return e.getAttribute("name")===u.getAttribute("type")}):c.find(s,function(e){return e.getAttribute("name")===r}),d=n.createElement("xs:element");d.setAttribute("name",i),l&&d.setAttribute("type",l.getAttribute("name")),u&&d.setAttribute("originalname",u.getAttribute("name")),d.setAttribute("requestedname",r),d.setAttribute("extraelement","true"),a.appendChild(d);var h=n.toString(),m=h,v=h.indexOf("?>");if(v>0&&h.length>v+2){var g=h.charAt(v+2);"\n"!=g&&(m=h.slice(0,v+2)+"\n"+h.slice(v+2))}return{content:m,validation:[]}}catch(A){console.log(A)}return{content:e,validation:[]}},e.prototype.completeReference=function(e,t){try{var n=new f(y).parseFromString(e),r=[],i=p(n,"xs:schema")[0],a=p(i,"xs:element"),o=p(i,"xs:complexType");a.forEach(function(e){return r.push(e.getAttribute("name"))}),o.forEach(function(e){return r.push(e.getAttribute("name"))});var s=0===t.asString().trim().length;return s?r:r.filter(function(e){return 0===e.indexOf(t.asString())})}catch(u){return[]}},e}()},function(e,t,n){"use strict";function r(e){return e instanceof E.ApiImpl||e instanceof T.ApiImpl?(new C).expandTraitsAndResourceTypes(e):null}function i(e){return(new L).expandLibraries(e)}function a(e){return(new L).expandLibrary(e)}function o(e,t,n){var r=y.fromUnit(e);if(!r)throw new Error("couldn't load api from "+e.absolutePath());if(!t||0==t.length)return r;for(var i=[],a=0;a<t.length;a++){var o=t[a],u=y.fromUnit(o);if(!u)throw new Error("couldn't load api from "+o.absolutePath());i.push(u)}var l=s(r,i,n);return l}function s(e,t,n,r,i){void 0===r&&(r=null),void 0===i&&(i=!1);for(var a=e,o=0,s=t;o<s.length;o++){
var u=s[o];i&&A.LowLevelProxyNode.isInstance(a.lowLevel())&&(a=(new C).expandHighLevelNode(a,r,e.wrapperNode()).highLevel()),u.overrideMaster(a),u.setMergeMode(n),a=u}return a}function u(e){for(var t=[],n=0,r=e;n<r.length;n++){var i=r[n];l(t,i,!0)}return t}function l(e,t,n){void 0===n&&(n=!1);var r=n?e:e.concat([]),i=t.lowLevel().unit();if(null==i)return r;if(0==r.length)r.push(i);else{var a=r[r.length-1].absolutePath();i.absolutePath()!=a&&r.push(i)}return r}function p(e){for(var t=[];e;)t.push(e),e=e.parent();return u(t)}function c(){return O.map(function(e){return e.name})}function f(e){for(var t=[],n=e.split("|").slice(1),r=0,i=n;r<i.length;r++)for(var a=i[r],o=0;o<O.length;o++)if(a.match(O[o].regexp)){t.push(O[o].transformer);break}return t}function d(e){return e?"*/*"==e?null:(e.indexOf("/*")==e.length-2&&(e=e.substring(0,e.length-2)+"/xxx"),I.parse(e)):null}var h=this&&this.__extends||function(e,t){function n(){this.constructor=e}for(var r in t)t.hasOwnProperty(r)&&(e[r]=t[r]);e.prototype=null===t?Object.create(t):(n.prototype=t.prototype,new n)},m=n(24),y=n(16),v=n(77),g=n(33),A=n(44),E=n(21),T=n(52),S=n(76),b=n(18),_=n(70),N=n(53),w=n(39),R=n(14),I=n(68),M=n(78);t.expandTraitsAndResourceTypes=r,t.expandLibraries=i,t.expandLibrary=a,t.mergeAPIs=o;var C=function(){function e(){this.hasGlobalTraits=!1,this.hasGlobalResourceTypes=!1}return e.prototype.expandTraitsAndResourceTypes=function(e,t,n){void 0===t&&(t=null),void 0===n&&(n=!1),this.init(e);var r=e.highLevel().lowLevel();if(!r)return e;if(r.actual().libExpanded)return e;var i=r.unit(),a=i.project().namespaceResolver().hasFragments(i),o=i.project().namespaceResolver().hasTemplates(r.unit());if(!o&&!a&&!n)return e;var s=this.createHighLevelNode(e.highLevel(),!1,t);if(e.definition().key()==b.Universe10.Overlay)return s.wrapperNode();var u=this.expandHighLevelNode(s,t,e);return u},e.prototype.init=function(e){var t=e.highLevel();this.ramlVersion=t.definition().universe().version()},e.prototype.expandHighLevelNode=function(e,t,n){var r=this;this.init(n);var i=e.wrapperNode();i.setAttributeDefaults(n.getDefaultsCalculator().isEnabled()),i.highLevel().setMergeMode(n.highLevel().getMergeMode());var a=i.resources();return a.forEach(function(e){return r.processResource(e)}),"RAML10"==this.ramlVersion&&(t=t||new N.ReferencePatcher,t.process(e),e.lowLevel().actual().referencePatcher=t),i},e.prototype.getTemplate=function(e,t,n,r){var i=t.lowLevel().unit().path(),a=n[i];a||(a={},n[i]=a);var o=a[e];return void 0!==o?o:(o=null,o=_.find(r,function(n){return y.qName(n.highLevel(),t)==e}),o||(o=null),a[e]=o,o)},e.prototype.createHighLevelNode=function(e,t,n,r){void 0===t&&(t=!0),void 0===n&&(n=null),void 0===r&&(r=!1);for(var i=e,a=[],o=i;o;){var u,l=o.lowLevel();i.definition().key()!=b.Universe10.Overlay||r?(A.LowLevelCompositeNode.isInstance(l)&&(l=l.originalNode().originalNode()),u=new A.LowLevelCompositeNode(l,null,null,this.ramlVersion)):u=l;var p=o.definition(),c=new y.ASTNodeImpl(u,null,p,null);if(c.setUniverse(o.universe()),a.push(c),!t)break;o=o.getMaster()}var f=a.pop();a=a.reverse();var d=i.getMergeMode();return s(f,a,d,n,r)},e.prototype.processResource=function(e,t){var n=this;void 0===t&&(t=[]);var r=t.concat(e.highLevel()),i=this.collectResourceData(e,e,void 0,void 0,r),a=e.highLevel().lowLevel();a.preserveAnnotations(),a.takeOnlyOriginalChildrenWithKey(w.universesInfo.Universe10.ResourceBase.properties.type.name),a.takeOnlyOriginalChildrenWithKey(w.universesInfo.Universe10.FragmentDeclaration.properties.uses.name),i.filter(function(e){return null!=e.resourceType}).forEach(function(t){var n=t.resourceType.node.highLevel().lowLevel(),r=t.resourceType.transformer;r.owner=e,a.adopt(n,r)});var o=e.methods();o.forEach(function(e){var t=e.highLevel().lowLevel();A.LowLevelCompositeNode.isInstance(t)&&t.takeOnlyOriginalChildrenWithKey(w.universesInfo.Universe10.FragmentDeclaration.properties.uses.name);var n=e.method(),r=[];i.forEach(function(i){var a=i.methodTraits[n];a&&(r=r.concat(a),a.forEach(function(n){var r=n.node.highLevel().lowLevel(),i=n.transformer;i.owner=e,t.adopt(r,i)}));var o=i.traits;o&&(r=r.concat(o),o.forEach(function(n){var r=n.node.highLevel().lowLevel(),i=n.transformer;i.owner=e,t.adopt(r,i)}))})});var s=e.resources();s.forEach(function(e){return n.processResource(e,r)}),e.methods().forEach(function(e){return n.mergeBodiesForMethod(e.highLevel())})},e.prototype.mergeBodiesForMethod=function(e){var t=e.lowLevel();if(A.LowLevelCompositeNode.isInstance(t)){var n=e.computedValue(b.Universe10.Api.properties.mediaType.name);if(null!=n){for(var r,i=[],a=t.children(),o=0,s=a;o<s.length;o++){var u=s[o];if(u.key()==b.Universe10.Method.properties.body.name)r=u;else if(u.key()==b.Universe10.Method.properties.responses.name)for(var l=u.children(),p=0,c=l;p<c.length;p++)for(var f=c[p],d=f.children(),h=0,m=d;h<m.length;h++){var y=m[h];y.key()==b.Universe10.Response.properties.body.name&&i.push(y)}}r&&i.push(r);for(var v=0,g=i;v<g.length;v++){var E=g[v];this.mergeBodies(E,n)}}}},e.prototype.mergeBodies=function(e,t){for(var n,r=[],i=[],a=[],o=[],s=!1,u=0,l=e.children();u<l.length;u++){var p=l[u],c=p.key();if(c==t)n=p,a.push({node:N.toOriginal(p),transformer:p.transformer()});else try{d(c),i.push(p)}catch(f){var h=N.toOriginal(p).parent();o.indexOf(h)<0&&(a.push({node:h,transformer:p.transformer()}),o.push(h)),F[p.key()]&&(s=!0),r.push(p)}}if(0==r.length||null==n&&0==i.length)return!1;if(s){for(var y=0,g=r;y<g.length;y++){var p=g[y];e.removeChild(p)}if(null==n){var A=N.toOriginal(e),E=v.newMapping(v.newScalar(t),v.newMap([])),T=new m.ASTNode(E,A.unit(),A,null,null);n=e.replaceChild(null,T)}return n.patchAdoptedNodes(a),!0}},e.prototype.collectResourceData=function(e,t,n,r,i,a){var o=this;void 0===n&&(n=[]),void 0===i&&(i=[]),void 0===a&&(a={}),i=i.concat([t.highLevel()]);var s=this.extractTraits(t,r,i),l={},p=t.methods();p.forEach(function(e){var t=o.extractTraits(e,r,i);t&&t.length>0&&(l[e.method()]=t)});var c,f=t.type();if(null!=f){var d=u(i);c=this.readGenerictData(e,f,t.highLevel(),"resource type",r,d)}var h={resourceType:c,traits:s,methodTraits:l};if(n.push(h),c){var m=c.node,v=y.qName(m.highLevel(),e.highLevel());a[v]?h.resourceType=null:(a[v]=!0,this.collectResourceData(e,m,n,c.transformer,i,a))}return n},e.prototype.extractTraits=function(e,t,n,r){var i=this;void 0===r&&(r={}),n=n.concat([e.highLevel()]);for(var a=[],o=-1;o<a.length;o++){var s=0>o?null:a[o],p=s?s.node:e,c=s?s.unitsChain:u(n),f=s?s.transformer:t;p.is&&p.is().forEach(function(t){var n=l(c,t.highLevel()),o=i.readGenerictData(e,t,p.highLevel(),"trait",f,n);if(o){var s=o.name;r[s]=!0,a.push(o)}})}return a},e.prototype.readGenerictData=function(e,t,n,r,i,a){void 0===a&&(a=[]);var o=t.value(),s=S.plural(M.camelCase(r));if("string"==typeof o){i&&(o=i.transform(o).value);var u=N.getDeclaration(o,s,a);if(u){var l=u.wrapperNode();return{name:o,transformer:null,parentTransformer:i,node:l,ref:t,unitsChain:a}}}else if(y.StructuredValue.isInstance(o)){var p=o,c=p.valueName();i&&(c=i.transform(c).value);var f={},d={},u=N.getDeclaration(c,s,a);if(u){var l=u.wrapperNode();"RAML08"==this.ramlVersion?p.children().forEach(function(e){return f[e.valueName()]=e.lowLevel().value()}):p.children().forEach(function(e){var t=e.lowLevel();t.resolvedValueKind()==v.Kind.SCALAR?f[e.valueName()]=t.value():d[e.valueName()]=t});var h=new U(e,null,a);Object.keys(f).forEach(function(e){var t=h.transform(f[e]);t&&"object"!=typeof t&&(f[e]=t)});var m=new x(r,c,a,f,d,i),g=new U(null,m,a);return{name:c,transformer:g,parentTransformer:i,node:l,ref:t,unitsChain:a}}}return null},e}();t.TraitsAndResourceTypesExpander=C;var L=function(){function e(){}return e.prototype.expandLibraries=function(e){var t=e;if(null==t)return null;A.LowLevelCompositeNode.isInstance(t.highLevel().lowLevel())&&(t=t.highLevel().lowLevel().unit().highLevel().asElement().wrapperNode());var n=new C,r=new N.ReferencePatcher,i=n.createHighLevelNode(t.highLevel(),!0,r,!0),a=n.expandHighLevelNode(i,r,t);return this.processNode(r,a.highLevel()),a},e.prototype.expandLibrary=function(e){var t=e;if(null==t)return null;A.LowLevelCompositeNode.isInstance(t.highLevel().lowLevel())&&(t=t.highLevel().lowLevel().unit().highLevel().asElement().wrapperNode());var n=new C,r=new N.ReferencePatcher,i=n.createHighLevelNode(t.highLevel(),!0,r,!0);r.process(i),r.expandLibraries(i);var a=i.wrapperNode();return a},e.prototype.processNode=function(e,t){if(null!=t){var n=t.getMaster();this.processNode(e,n),R.isOverlayType(t.definition())&&e.process(t),e.expandLibraries(t)}},e}();t.LibraryExpander=L,t.toUnits=p;var P=function(){function e(t,n){this.name=t,this.regexp=new RegExp(e.leftTransformRegexp.source+t+e.rightTransformRegexp.source),this.transformer=n}return e.leftTransformRegexp=/\s*!\s*/,e.rightTransformRegexp=/\s*$/,e}(),O=[new P("singularize",function(e){return S.singular(e)}),new P("pluralize",function(e){return S.plural(e)}),new P("uppercase",function(e){return e?e.toUpperCase():e}),new P("lowercase",function(e){return e?e.toLowerCase():e}),new P("lowercamelcase",function(e){return e?M.camelCase(e):e}),new P("uppercamelcase",function(e){if(!e)return e;var t=M.camelCase(e);return t[0].toUpperCase()+t.substring(1,t.length)}),new P("lowerunderscorecase",function(e){if(!e)return e;var t=M.snake(e);return t.toLowerCase()}),new P("upperunderscorecase",function(e){if(!e)return e;var t=M.snake(e);return t.toUpperCase()}),new P("lowerhyphencase",function(e){if(!e)return e;var t=M.param(e);return t.toLowerCase()}),new P("upperhyphencase",function(e){if(!e)return e;var t=M.param(e);return t.toUpperCase()})];t.getTransformNames=c,t.getTransformersForOccurence=f;var D=function(){function e(){this.buf=null}return e.prototype.append=function(e){""!==e&&(null!=this.buf?null!=e&&("string"!=typeof this.buf&&(this.buf=""+this.buf),this.buf+=e):""!==e&&(this.buf=e))},e.prototype.value=function(){return null!=this.buf?this.buf:""},e}(),x=function(){function e(e,t,n,r,i,a){this.templateKind=e,this.templateName=t,this.unitsChain=n,this.params=r,this.structuredParams=i,this.vDelegate=a}return e.prototype.transform=function(e,t,n,r){var i={},a=[];if("string"==typeof e){if(this.structuredParams&&g.stringStartsWith(e,"<<")&&g.stringEndsWith(e,">>")){var o=e.substring(2,e.length-2),s=this.structuredParams[o];if(null!=s)return{value:s,errors:a}}for(var u=e,l=new D,p=0,c=u.indexOf("<<");c>=0;c=u.indexOf("<<",p)){l.append(u.substring(p,c));var d=c;if(c+="<<".length,p=this.paramUpperBound(u,c),-1==p)break;var h=u.substring(c,p);p+=">>".length;var m,o,y=u.substring(d,p),v=f(h);if(v.length>0){var A=h.indexOf("|");if(o=h.substring(0,A).trim(),m=this.params[o],m&&"string"==typeof m&&m.indexOf("<<")>=0&&this.vDelegate&&(m=this.vDelegate.transform(m,t,n,r).value),m){N.PatchedReference.isInstance(m)&&(m=m.value());for(var E=0,T=v;E<T.length;E++){var S=T[E];m=S(m+"")}}}else o=h.trim(),m=this.params[o],m&&"string"==typeof m&&m.indexOf("<<")>=0&&this.vDelegate&&(m=this.vDelegate.transform(m,t,n,r).value);(null===m||void 0===m)&&(i[o]=!0,m=y),l.append(m)}return l.append(u.substring(p,u.length)),{value:l.value(),errors:a}}return{value:e,errors:a}},e.prototype.paramUpperBound=function(e,t){for(var n=0,r=t;r<e.length;r++)if(g.stringStartsWith(e,"<<",r))n++,r++;else if(g.stringStartsWith(e,">>",r)){if(0==n)return r;n--,r++}return e.length},e.prototype.children=function(e){var t=this.substitutionNode(e);return t?t.children():null},e.prototype.valueKind=function(e){var t=this.substitutionNode(e);return t?t.valueKind():null},e.prototype.anchorValueKind=function(e){var t=this.substitutionNode(e);return t&&t.valueKind()==v.Kind.ANCHOR_REF?t.anchorValueKind():null},e.prototype.resolvedValueKind=function(e){var t=this.substitutionNode(e);return t&&t.resolvedValueKind()},e.prototype.includePath=function(e){var t=this.substitutionNode(e);return t?t.includePath():null},e.prototype.substitutionNode=function(e){var t=this.paramName(e);return t&&this.structuredParams[t]},e.prototype.paramName=function(e){var t=null;if(e.valueKind()==v.Kind.SCALAR){var n=(""+e.value()).trim();g.stringStartsWith(n,"<<")&&g.stringEndsWith(n,">>")&&(t=n.substring(2,n.length-2))}return t},e.prototype.definingUnitSequence=function(e){if(e.length<2)return null;if("("==e.charAt(0)&&")"==e.charAt(e.length-1)&&(e=e.substring(1,e.length-1)),e.length<4)return null;if("<<"!=e.substring(0,2))return null;if(">>"!=e.substring(e.length-2,e.length))return null;var t=e.substring(2,e.length-2);return t.indexOf("<<")>=0||t.indexOf(">>")>=0?null:this._definingUnitSequence(t)},e.prototype._definingUnitSequence=function(e){return this.params&&this.params[e]?this.unitsChain:this.vDelegate?this.vDelegate._definingUnitSequence(e):null},e}();t.ValueTransformer=x;var U=function(e){function t(t,n,r){e.call(this,null!=n?n.templateKind:"",null!=n?n.templateName:"",r),this.owner=t,this.delegate=n}return h(t,e),t.prototype.transform=function(t,n,r,i){if(null==t||null!=r&&!r())return{value:t,errors:[]};var a={value:t,errors:[]},o=!1;k.forEach(function(e){return o=o||t.toString().indexOf("<<"+e)>=0}),o&&(this.initParams(),a=e.prototype.transform.call(this,t,n,r,i));var s=null!=this.delegate?this.delegate.transform(a.value,n,r,i):a.value;return null!=r&&r()&&null!=i&&(s.value=i(s.value,this)),s},t.prototype.initParams=function(){for(var e,t,n="",r=this.owner.highLevel().lowLevel(),i=r,a=null;i;){var o=i.key();if(null!=o)if(g.stringStartsWith(o,"/")){if(!t)for(var s=o.split("/"),u=s.length-1;u>=0;u--){var l=s[u];if(-1==l.indexOf("{")){t=s[u];break}l.length>0&&(a=l)}n=o+n}else e=o;i=i.parent()}t||a&&(t=""),this.params={resourcePath:n,resourcePathName:t},e&&(this.params.methodName=e)},t.prototype.children=function(e){return null!=this.delegate?this.delegate.children(e):null},t.prototype.valueKind=function(e){return null!=this.delegate?this.delegate.valueKind(e):null},t.prototype.includePath=function(e){return null!=this.delegate?this.delegate.includePath(e):null},t.prototype.anchorValueKind=function(e){return null!=this.delegate?this.delegate.anchorValueKind(e):null},t.prototype.resolvedValueKind=function(e){return null!=this.delegate?this.delegate.resolvedValueKind(e):null},t.prototype._definingUnitSequence=function(e){return this.params&&this.params[e]?this.unitsChain:this.delegate?this.delegate._definingUnitSequence(e):null},t}(x);t.DefaultTransformer=U;var k=["resourcePath","resourcePathName","methodName"],F={};F[w.universesInfo.Universe10.TypeDeclaration.properties.type.name]=!0,F[w.universesInfo.Universe10.TypeDeclaration.properties.example.name]=!0,F[w.universesInfo.Universe08.BodyLike.properties.schema.name]=!0,F[w.universesInfo.Universe10.ObjectTypeDeclaration.properties.properties.name]=!0,t.parseMediaType=d},function(e,t,n){(function(e){"use strict";function r(e,t,n){return new ke.Function(e).call(t,n)}function i(e,t){var n=e.property();return n&&n.getContextRequirements().forEach(function(r){e.checkContextValue(r.name,r.value,r.value)||t.accept(Q(De.CONTEXT_REQUIREMENT,{name:r.name,value:r.value,propName:n.nameId()},e))}),n}function a(e,t){var n;try{n=e.lowLevel().unit().project().fsEnabled()}catch(r){n=!0}if(n&&"undefined"!=typeof ge&&ge){for(var i=["exists","readFile","writeFile","readdir","existsSync","readFileSync","writeFileSync","readdirSync"],a=(Object.keys(ge),0);a<i.length;a++)if(!ge[i[a]])return;var o=e.lowLevel().unit().absolutePath(),s=ve.dirname(o),u=ve.resolve(s,"raml-lint.js");Fe(u,t,e);var l=ve.resolve(s,".raml");if(ge.existsSync(l)){var p=ge.statSync(l);if(p.isDirectory()){var c=ge.readdirSync(l);c.forEach(function(n){-1!=n.indexOf("-lint.js")&&Fe(ve.resolve(l,n),t,e)})}}}}function o(e){return e.nameId()==Ae.Universe10.TypeDeclaration.properties.type.name||e.nameId()==Ae.Universe10.TypeDeclaration.properties.schema.name}function s(e){return l(e)?e.nameId()==Ae.Universe10.TypeDeclaration.properties["default"].name:!1}function u(e){return l(e)?e.nameId()==Ae.Universe10.TypeDeclaration.properties.example.name:!1}function l(e){return e.domain()?e.domain().getAdapter(Se.RAMLService).isUserDefined()?!1:!0:!1}function p(e){return l(e)?e.nameId()==Ae.Universe08.MethodBase.properties.securedBy.name:!1}function c(e){for(var t=e;;){if(t.definition().getAdapter(Se.RAMLService).isInlinedTemplates())return t.definition().nameId();var n=t.parent();if(!n)break;t=n}return null}function f(e){var n=e.parent(),r=null,i=n.definition();if(n&&me.UserDefinedClass.isInstance(i)){var a=n.property();if(Ee.isIsProperty(a)||Ee.isTypeProperty(a)){var o=e.name();r=null!=t.RESERVED_TEMPLATE_PARAMETERS[o]?Q(De.INVALID_PARAMETER_NAME,{paramName:o},e):Q(De.UNUSED_PARAMETER,{paramName:o},e)}}if(!r){var s=e.name(),u=Te("RAML08"),l=i.universe().version()==u.version();if(l){var p=Ae.Universe08.Parameter.name;Ae.Universe10.TypeDeclaration.name;if(l&&i.isAssignableFrom(p)){for(var c=u.type(p).allSubTypes().filter(function(e){return null!=Ae.Universe08[e.nameId()].properties[s]}),f={},d=0;d<c.length;d++){var h=c[d];if(!f[h.nameId()]){var m=h.valueRequirements();if(m&&0!=m.length){var y=m.filter(function(e){return"type"==e.name}).map(function(e){return e.value});if(0!=y.length){var v=y[0];f[h.nameId()]=v,h.allSubTypes().forEach(function(e){return c.push(e)})}}}}var g=Object.keys(f).map(function(e){return f[e]}).sort();if(g.length>0){var A=g.map(function(e,t){var n="'"+e+"'";return t==g.length-1?n:t==g.length-2?n+" or ":n+", "}).join("");r=Q(De.INVALID_PROPERTY_OWNER_TYPE,{propName:s,namesStr:A},e)}}}}return r}function d(e,t){if(e.isElement()){if(e.invalidSequence){var n=e.property().nameId();n=Pe.sentenceCase(Oe.singular(n)),t.acceptUnique(ee(De.SEQUENCE_NOT_ALLOWED_10,{propName:n},e.lowLevel().parent().parent(),e))}var r=e.asElement();if(r.definition().isAssignableFrom(Ae.Universe10.LibraryBase.name)){var i,a=!1,o=!1;r.lowLevel().children().forEach(function(e){"schemas"==e.key()&&(a=!0,i=e),"types"==e.key()&&(o=!0)}),a&&o&&t.accept(gt(i,r,ce.IssueCode.ILLEGAL_PROPERTY_VALUE,!1,"types and schemas are mutually exclusive",!1))}r.definition().requiredProperties()&&r.definition().requiredProperties().length>0;h(e,t),(new rt).validate(r,t),(new mt).validate(r,t)}else h(e,t);(new dt).validate(e,t)}function h(e,t,n){void 0===n&&(n=!1);var r=e.parent(),i=e.lowLevel().value();if(e.lowLevel()){if(e.lowLevel().keyKind()==he.Kind.MAP&&t.accept(Q(De.NODE_KEY_IS_A_MAP,{},e)),e.lowLevel().keyKind()==he.Kind.SEQ&&null==i){var a=!1;e.isElement()&&e.asElement().definition().isAssignableFrom(Ae.Universe10.TypeDeclaration.name)&&(a=!0),a||t.accept(Q(De.NODE_KEY_IS_A_SEQUENCE,{},e))}if(null==r){var o=e.lowLevel().unit().contents().length;e.lowLevel().errors().forEach(function(n){var r=n.mark?n.mark.position:0,i=r>=o?r:r+1;if(n.mark&&n.mark.toLineEnd){var a=n.mark.buffer,s=a.indexOf("\n",r);0>s&&(s=a.length),s<a.length&&"\r"==a.charAt(s)&&s--,i=s}var u={code:"YAML_ERROR",message:n.message,node:null,start:r,end:i,isWarning:!1,path:e.lowLevel().unit()==e.root().lowLevel().unit()?null:e.lowLevel().unit().path(),unit:e.lowLevel().unit()};t.accept(u)})}}if(e.isUnknown()){if(-1!=e.name().indexOf("<<")&&null!=c(r))return(new We).validateName(e,t),!1;if(e.needSequence&&t.accept(Q(De.SEQUENCE_REQUIRED,{name:e.name()},e)),e.needMap)return e.knownProperty?t.accept(Q(De.PROPERTY_MUST_BE_A_MAP_10,{propName:e.knownProperty.nameId()},e)):t.accept(Q(De.MAP_REQUIRED,{},e)),!1;if(e.unresolvedRef&&t.accept(Q(De.UNRESOLVED_REFERENCE,{ref:i},e)),e.knownProperty){if(0==e.lowLevel().includeErrors().length){if(c(r)&&Re.startsWith(i,"<<")&&Re.endsWith(i,">>"))return!1;if("body"==e.name()&&e.computedValue("mediaType"))return!1;"~"!=e.lowLevel().value()&&(ae(e,t,De.SCALAR_PROHIBITED.code,!1)||t.accept(Q(De.SCALAR_PROHIBITED,{propName:e.name()},e)))}}else{var s=f(e);s||(s=Q(De.UNKNOWN_NODE,{name:e.name()},e)),t.accept(s)}}if(e.markCh()&&!e.allowRecursive())return e.property()?(t.accept(Q(De.RECURSIVE_DEFINITION,{name:e.name()},e)),!1):!1;if(e.definition&&e.definition().isAssignableFrom(Ae.Universe10.Operation.name)){var u=y(e.wrapperNode()),l=u.queryStringComesFrom,p=u.queryParamsComesFrom;l&&p&&(t.accept(m(l,e,!1)),t.accept(m(p,e,!0)))}var d=e.definition&&e.definition()&&(e.definition().key()===Ae.Universe10.Overlay||e.definition().key()===Ae.Universe10.Extension);return d&&S(e,t,n),!0}function m(e,t,n){var r=e,i=e,a=n?Ae.Universe10.Operation.properties.queryString.name:Ae.Universe10.Operation.properties.queryParameters.name;return r.unit?ee(De.PROPERTY_ALREADY_SPECIFIED,{propName:a},r,t):Q(De.PROPERTY_ALREADY_SPECIFIED,{propName:a},i)}function y(e){return{queryParamsComesFrom:v(e,!0),queryStringComesFrom:v(e,!1)}}function v(e,t,n,r){if(void 0===n&&(n=!0),void 0===r&&(r={}),!e)return null;if(e.name){var i=e.name();if(r[i])return;r[i]=!0}var a=A(e,t);if(a)return a;var o=e.is&&e.is()||[],s=de.find(o,function(e){return v(e.trait(),t,n,r)});if(s)return s.highLevel();if(n){var u=e.parentResource&&e.parentResource(),l=u&&g(u,t,r);if(l)return l;if(u=e.parent&&e.parent(),u&&u.highLevel().definition().isAssignableFrom(Ae.Universe10.ResourceBase.name))return g(u,t,r)}return null}function g(e,t,n,r){if(void 0===n&&(n={}),void 0===r&&(r={}),e.name){var i=e.name();if(r[i])return;r[i]=!0}var a=e.is(),o=de.find(a,function(e){return v(e.trait(),t,!1,n)});if(o)return o.highLevel();var s=e.type(),u=s&&s.resourceType(),l=u&&g(u,t,n,r);return l?s.highLevel():void 0}function A(e,t){return t?E(e):T(e)}function E(e){var t=e.highLevel();return t.lowLevel&&de.find(t.lowLevel().children(),function(e){return e.key&&e.key()===Ae.Universe10.Operation.properties.queryParameters.name})}function T(e){var t=e.highLevel(),n=t.element(Ae.Universe10.Operation.properties.queryString.name);return n}function S(e,t,n){if(void 0===n&&(n=!1),!e.parent()){var r=e.asElement();if(r&&r.isAuxilary()){var i=r.getMaster();i&&d(i,t)}}}function b(e,t,n){if(void 0===n&&(n=!1),h(e,t,n))try{var r=e.definition&&e.definition()&&(e.definition().key()===Ae.Universe10.Overlay||e.definition().key()===Ae.Universe10.Extension),i=r?e.children():e.directChildren();i.filter(function(e){return!n||e.property&&e.property()&&e.property().isRequired()}).forEach(function(n){if(n&&n.errorMessage){var r=n.errorMessage;return void t.accept(Q(r.entry,r.parameters,n.name()?n:e))}n.validate(t)})}finally{e.unmarkCh()}}function _(e){var t=e.value();if("string"==typeof t&&-1!=t.indexOf("<<"))return!0;for(var n=e.children(),r=0;r<n.length;r++)if(_(n[r]))return!0;return!1}function N(e,t){if(!e.parent())try{R(e,t)}finally{w(e,t)}if(e.isAttr())(new Ye).validate(e,t);else if(e.isElement()){if(e.invalidSequence){var n=e.property().nameId(),r=Pe.sentenceCase(Oe.singular(n));t.acceptUnique(ee(De.SEQUENCE_NOT_ALLOWED_10,{propName:r},e.lowLevel().parent().parent(),e,!1))}var i=e.asElement();if(Ee.isExampleSpecType(i.definition())){var a=i.children();if(0==a.length)return void b(e,t,!0);var o=a.filter(function(e){var t=e.lowLevel().key();return t?"("==t.charAt(0)&&")"==t.charAt(t.length-1)?!1:null==i.definition().property(t):!0});if(o.length>0)return void b(e,t,!0)}if(i.definition().isAnnotationType()||i.property()&&"annotations"==i.property().nameId())return void(new tt).validate(i,t);if(i.definition().isAssignableFrom(Ae.Universe10.UsesDeclaration.name)){var s=i.attr(Ae.Universe10.UsesDeclaration.properties.value.name);if(s&&s.value()){var u=i.lowLevel().unit().resolve(s.value());if(u&&null!==u.contents()){if(!Me.isWaitingFor(s.value())){var l=[];if(0===u.contents().trim().length)return void t.accept(Q(De.EMPTY_FILE,{path:s.value()},i,!1));if(u.highLevel().validate(ye.createBasicValidationAcceptor(l,u.highLevel())),l.length>0){var p=Ke(s,i);l.forEach(function(e){e.unit=null==e.unit?u:e.unit,e.path||(e.path=u.absolutePath())});for(var f=0,d=l;f<d.length;f++){for(var h=d[f],m=h;m.extras&&m.extras.length>0;)m=m.extras[0];m!=p&&(m.extras||(m.extras=[]),m.extras.push(p)),t.accept(h)}}}}else t.accept(Q(De.INVALID_LIBRARY_PATH,{path:s.value()},i,!1))}}if(i.definition().isAssignableFrom(Ae.Universe10.TypeDeclaration.name)){if(c(i)&&_(i.lowLevel()))return;return i.attrs().forEach(function(n){var r=n.property().range().key();if(r==Ae.Universe08.RelativeUriString||r==Ae.Universe10.RelativeUriString)return void(new Ge).validate(n,t);if(r==Ae.Universe08.FullUriTemplateString||r==Ae.Universe10.FullUriTemplateString)return void(new Ge).validate(n,t);if(n.property().getAdapter(Se.RAMLPropertyService).isKey()){var a=e.property()&&e.property().nameId();if(a==Ae.Universe08.Resource.properties.uriParameters.name||a==Ae.Universe08.Resource.properties.baseUriParameters.name)return;if(i.property()&&i.property().nameId()==Ae.Universe10.MethodBase.properties.body.name)return void(new Xe).validate(n,t)}}),(new ut).validate(i,t),(new lt).validate(i,t),(new ot).validate(i,t),void(new nt).validate(i,t)}if(i.definition().isAssignableFrom(Ae.Universe10.LibraryBase.name)){var y,v=!1,g=!1;i.lowLevel().children().forEach(function(e){"schemas"==e.key()&&(v=!0,y=e),"types"==e.key()&&(g=!0)}),v&&g&&t.accept(ee(De.TYPES_AND_SCHEMAS_ARE_EXCLUSIVE,{},y,i))}var A=i.definition().requiredProperties()&&i.definition().requiredProperties().length>0,E=i.definition().getAdapter(Se.RAMLService).getAllowAny();E?A&&b(e,t,!0):b(e,t),(new ht).validate(i,t),(new rt).validate(i,t),(new mt).validate(i,t)}else b(e,t);(new dt).validate(e,t)}function w(e,t){if(e.lowLevel()){var n=e.lowLevel().actual();delete n._inc,e.children().forEach(function(e){return w(e,t)})}}function R(e,t){var n=e.lowLevel();if(n){var r=n.actual();if(!r._inc){if(e.isElement()){var i=e.name();"string"==typeof i&&null!=i&&-1!=i.indexOf(" ")&&t.accept(Q(De.SPACES_IN_KEY,{value:i},e,!0))}if(r._inc=!0,n){n.includeErrors().forEach(function(n){var r=!1;e.lowLevel().hasInnerIncludeError()&&(r=!0);var i=Q(De.INCLUDE_ERROR,{msg:n},e,r);t.accept(i)});var a=n.includePath();if(null!=a&&!ve.isAbsolute(a)&&!fe.isWebPath(a)){var o=n.unit().absolutePath(),s=M(ve.dirname(o),a);if(s>0){var u=Q(De.PATH_EXCEEDS_ROOT,{},e,!0);t.accept(u)}}}e.children().forEach(function(e){return R(e,t)}),0==e.children().length&&null!=n&&n.children().forEach(function(n){return I(n,t,e)})}}}function I(e,t,n){e.includeErrors().forEach(function(r){var i=!1;e.hasInnerIncludeError()&&(i=!0);var a=ee(De.INCLUDE_ERROR,{msg:r},e,n,i);t.accept(a)});var r=e.includePath();if(null!=r&&!ve.isAbsolute(r)&&!fe.isWebPath(r)){var i=e.unit().absolutePath(),a=M(ve.dirname(i),r);if(a>0){var o=ee(De.PATH_EXCEEDS_ROOT,{},e,n,!0);t.accept(o)}}e.children().forEach(function(e){return I(e,t,n)})}function M(e,t){for(var n=Ve(e),r=Ve(t),i=n.length,a=0,o=0,s=r;o<s.length;o++){var u=s[o];".."==u?(i--,0>i&&(a=Math.min(i,a))):i++}return-1*a}function C(e,t,n,r,i){return e.hasArrayInHierarchy()?L(e,t,n,r,i):e.hasValueTypeInHierarchy()?P(e,t,n,r,i):!0}function L(e,t,n,r,i){return e.arrayInHierarchy().componentType()?C(e.arrayInHierarchy().componentType(),t,n,r):!0}function P(e,t,n,r,i){try{if(e.key()==Ae.Universe10.AnnotationRef){var a=Ne.referenceTargets(r,t),o=de.find(a,function(e){return ye.qName(e,t)==n});if(null!=o){var s=o.attributes("allowedTargets");if(s){var u=s.map(function(e){return e.value()});if(u.length>0){var l=!1,p=t.definition().allSuperTypes();p=p.concat([t.definition()]);var c=p.map(function(e){return e.nameId()});if(u.forEach(function(e){"API"==e&&(e="Api"),"NamedExample"==e&&(e="ExampleSpec"),"SecurityScheme"==e&&(e="AbstractSecurityScheme"),"SecuritySchemeSettings"==e&&(e="SecuritySchemeSettings"),de.find(c,function(t){return t==e})?l=!0:("Parameter"==e&&t.computedValue("location")&&(l=!0),"Field"==e&&t.computedValue("field")&&(l=!0))}),!l){var f=u.map(function(e){return"'"+e+"'"}).join(", ");return new He(De.INVALID_ANNOTATION_LOCATION,{aName:n,aValues:f})}}}}return g}if(e.key()==Ae.Universe08.SchemaString||e.key()==Ae.Universe10.SchemaString){var d=!1;if(me.UserDefinedProp.isInstance(r)){var h=r,m=h.node();if(m){var y=m.property();y&&(d=Ee.isTypeProperty(y)||Ee.isSchemaProperty(y))}}if(d)return!1;var v=n&&n.trim().length>0&&("{"==n.trim().charAt(0)||"<"==n.trim().charAt(0)),g=Ce.createSchema(n,Y(t.lowLevel(),i&&i.lowLevel()));if(!g)return g;if(g instanceof Error)g.isWarning=!0,v||(g.canBeRef=!0);else{var A=!1;try{JSON.parse(n),A=!0}catch(E){}if(A)try{g.validateSelf()}catch(E){return E.isWarning=!0,E}}return g}if(e.key()==Ae.Universe08.StatusCodeString||e.key()==Ae.Universe10.StatusCodeString){var T=ne(n);if(null!=T)return T}if(e.key()==Ae.Universe08.BooleanType||e.isAssignableFrom(Ae.Universe10.BooleanType.name)){if("true"!==n&&"false"!==n&&n!==!0&&n!==!1)return new He(De.BOOLEAN_EXPECTED);if(i){var S=i.lowLevel().value(!0);if("true"!==S&&"false"!==S)return new He(De.BOOLEAN_EXPECTED)}}if(e.key()==Ae.Universe08.NumberType||e.isAssignableFrom(Ae.Universe10.NumberType.name)){var b=parseFloat(n);if(isNaN(b))return new He(De.NUMBER_EXPECTED,{propName:r.nameId()})}if((e.key()==Ae.Universe08.StringType||e.isAssignableFrom(Ae.Universe10.StringType.name))&&null===n&&t&&r){var _=t.attr(r.nameId());if(_){var N=_.lowLevel().children();if(N&&N.length>0)return new He(De.STRING_EXPECTED_3,{propName:r.nameId()})}}return!0}catch(E){return E.canBeRef=!0,E}}function O(e){if(!e)return!1;var t=e.toLowerCase(),n=e.toUpperCase();return e!==t&&e!==n?!0:!1}function D(e){if(!e)return null;if(e.isElement()){var n=e,r=n.definition();if(r&&t.typeToName.hasOwnProperty(r.nameId()))return t.typeToName[r.nameId()];if(r.isAssignableFrom(Ae.Universe10.TypeDeclaration.name)||r.isAssignableFrom(Ae.Universe08.Parameter.name)){if(n.property()&&t.parameterPropertyToName.hasOwnProperty(n.property().nameId()))return t.parameterPropertyToName[n.property().nameId()];if(n.property()&&n.parent()&&n.property().nameId()==Ae.Universe10.LibraryBase.properties.types.name&&n.parent().definition()&&n.parent().definition().isAssignableFrom(Ae.Universe10.LibraryBase.name))return"type";if(n.property()&&n.parent()&&n.property().nameId()==Ae.Universe10.LibraryBase.properties.securitySchemes.name&&n.parent().definition()&&n.parent().definition().isAssignableFrom(Ae.Universe10.LibraryBase.name))return"security scheme"}}return null}function x(e,t,n){var r=Ne.declRoot(n);pe.LowLevelProxyNode.isInstance(n.lowLevel())&&(n=r),r._cach||(r._cach={});var i=e.id();if(e.domain()&&(i+=e.domain().nameId()),i){var a=r._cach[i];if(a)return null!=a[t]}var o=Ne.enumValues(e,n),s={};return o.forEach(function(e){return s[e]=1}),e.id()&&(r._cach[i]=s),null!=s[t]}function U(e,n,r,i){if(F(e,n,i),B(e,n,i),r&&("null"!=r||!e.isAllowNull())){var a=e.getAdapter(Se.RAMLPropertyService),o=x(e,r,n.parent());if(o||n.lowLevel().unit().absolutePath()===n.parent().lowLevel().unit().absolutePath()||(o=x(e,r,ye.fromUnit(n.lowLevel().unit()))),!o){if("string"==typeof r&&0==r.indexOf("x-")&&e.nameId()==Ae.Universe10.TypeDeclaration.properties.type.name)return!0;var s=a.isReference&&a.isReference()&&a.referencesTo&&a.referencesTo()&&a.referencesTo().nameId&&a.referencesTo().nameId(),u=t.typeToName[s]||V(n),l={referencedToName:u,ref:r},p=u?De.UNRECOGNIZED_ELEMENT:De.UNRESOLVED_REFERENCE,c=K(p,e,n),f=e.range().key()===Ae.Universe08.SchemaString;return ae(n,i,c.code,f)||i.accept(Q(c,l,n,f)),!0}return k(n)&&Ee.isTraitRefType(n.definition())?(i.accept(Q(De.DUPLICATE_TRAIT_REFERENCE,{refValue:r},n)),!0):!1}}function k(e){var t,n=e.property().domain().universe().version();if(t="RAML10"==n?le(ue.serialize(e.lowLevel())):e.value()&&e.value().valueName&&e.value().valueName(),!t)return!1;var r=e.parent&&e.parent();if(!r)return!1;var i=e.name&&e.name();if(!i)return!1;var a=r.attributes&&r.attributes(i);if(!a)return!1;if(0===a.length)return!1;var o=0;return a.forEach(function(e){var r;"RAML10"==n?t=le(ue.serialize(e.lowLevel())):r=e.value&&e.value()&&e.value().valueName&&e.value().valueName(),r===t&&o++}),o>1}function F(e,t,n){if(Ee.isIsProperty(e)){var r=t.lowLevel();if(null!=r){var i=null,a=r.parent(),o=null!=a?a.parent():null;if(r.kind()==he.Kind.MAPPING&&r.key()&&"is"==r.key()?i=r:null!=a&&a.kind()==he.Kind.MAPPING&&a.key()&&"is"==a.key()?i=a:null!=o&&o.kind()==he.Kind.MAPPING&&o.key()&&"is"==o.key()&&(i=o),null!=i){null==i.value()||i.children()&&0!=i.children().length||n.accept(Q(De.IS_IS_ARRAY,{},t));
var s=!1;i.children().forEach(function(e){e.kind()!=he.Kind.SCALAR&&e.kind()!=he.Kind.MAP&&(s=!0)}),s&&n.accept(Q(De.IS_IS_ARRAY,{},t))}}}}function B(e,t,n){if(Ee.isTypeProperty(e)&&Ee.isResourceTypeRefType(t.definition())){var r=t.lowLevel();null==t.value()&&r&&r.children()&&0==r.children().length?r.kind()==he.Kind.MAPPING&&null!=r.valueKind()&&n.accept(Q(De.RESOURCE_TYPE_NAME,{},t)):null==t.value()&&r&&r.children()&&r.children().length>1&&n.accept(Q(De.MULTIPLE_RESOURCE_TYPES,{},t))}}function K(e,t,n){return"type"==t.nameId()&&"RAML08"==t.domain().universe().version()&&t.domain().isAssignableFrom(Ae.Universe08.Parameter.name)?De.TYPES_VARIETY_RESTRICTION:null!=n.parent()&&Ee.isSecuritySchemaType(n.parent().definition())?De.UNRECOGNIZED_SECURITY_SCHEME:e}function V(e){var t=e&&e.lowLevel()&&e.lowLevel().key();if(t===Ae.Universe10.AbstractSecurityScheme.properties.type.name){var n=e.parent()&&e.parent().definition()&&e.parent().definition().nameId();if(n===Ae.Universe10.AbstractSecurityScheme.name)return"security scheme type"}else if(t===Ae.Universe08.BodyLike.properties.schema.name){var n=e.parent()&&e.parent().definition()&&e.parent().definition().nameId();if(n===Ae.Universe08.BodyLike.name)return"schema"}}function j(e,t){var n=t.getValidationPath(),r=q(e,n),i=t.getInternalPath(),a=!1;if(i){var o=q(r,i);o&&o!=r&&(r=o,a=!0)}return{node:r,internalPathUsed:a}}function W(e,t){var n=e.getExtra(we.SOURCE_EXTRA);return ye.LowLevelWrapperForTypeSystem.isInstance(n)?n.node():null}function q(e,t){if(!t)return e;var n=e.children().filter(function(e){return e.isAttr()&&e.asAttr().isFromKey()?!1:e.name()===t.name});if(e.isElement()&&Ee.isTypeDeclarationDescendant(e.asElement().definition())){var r=e.lowLevel();n=de.uniq(e.directChildren().concat(e.children())).filter(function(e){return e.isAttr()&&e.asAttr().isFromKey()?!1:e.name()===t.name}).sort(function(e,t){for(var n=e.lowLevel().parent();n&&n.kind()!=he.Kind.MAPPING;)n=n.parent();for(var i=t.lowLevel().parent();i&&i.kind()!=he.Kind.MAPPING;)i=i.parent();return n==r?-1:i==r?1:0})}var i=t.child&&"number"==typeof t.child.name?t.child.name:-1;if(i>=0&&n.length>i)return q(n[i],t.child.child);if(n.length>0)return q(n[0],t.child);if(!e.lowLevel())return e;for(var a=e.lowLevel().children(),o=0;o<a.length;o++)if(a[o].key()===t.name){var s=new ye.BasicASTNode(a[o],e);return q(s,t.child)}if(!isNaN(t.name)&&a[t.name]){var u=a[t.name],s=new ye.BasicASTNode(u,e);return q(s,t.child)}return e}function Y(e,t){var n=e&&e.includeBaseUnit()&&(e.includePath&&e.includePath()?e.includeBaseUnit().resolve(e.includePath()):e.includeBaseUnit());if(t){var r=t&&t.includeBaseUnit()&&(t.includePath&&t.includePath()?t.includeBaseUnit().resolve(t.includePath()):t.includeBaseUnit());r!=n&&(n=r)}return new Ie.ContentProvider(n)}function H(e){return-1!=e.indexOf("json")}function $(e){return-1!=e.indexOf("xml")}function G(e){var t=X(e);if("body"==t){var n=e.root().attr("mediaType");return n?n.value():null}return t}function X(e){if(e.parent()){var t=e.parent().definition();if(t.key()==Ae.Universe08.BodyLike)return e.parent().name();if(e.parent().parent()){var n=e.parent().parent().definition().key();if(n==Ae.Universe08.Response||n==Ae.Universe10.Response)return e.parent().property().nameId()==Ae.Universe08.Response.properties.headers.name?null:e.parent().name();if(n==Ae.Universe08.Method||n==Ae.Universe10.Method)return e.parent().property().nameId()==Ae.Universe10.MethodBase.properties.queryParameters.name||e.parent().property().nameId()==Ae.Universe10.MethodBase.properties.headers.name?null:e.parent().name()}}return null}function z(e,t){return Z(e.getCode(),e.getMessage(),j(t,e).node,e.isWarning())}function J(e,t,n){var r=null!=n?n:e.isWarning,i=e.internalPath,a=t,o=!1,s=t.lowLevel().valueKind();if(s==he.Kind.INCLUDE_REF&&(s=t.lowLevel().resolvedValueKind()),i&&s!=he.Kind.SCALAR){var u=me.rt.toValidationPath(i);if(u){var l=q(t,u);l&&l!=t&&(a=l,o=!0)}}var p=o?null:e.internalRange,c=Q(e.messageEntry,e.parameters,a,r,p);if(e.filePath){var f=t.lowLevel().unit().project().unit(e.filePath,!0);c.unit=f,c.path=e.filePath}return c}function Q(e,t,n,r,i){void 0===r&&(r=!1);var a=re(e,t);return Z(e.code,a,n,r,i)}function Z(e,t,n,r,i,a){void 0===r&&(r=!1),void 0===a&&(a=!1);var o=null,s=null;if(pe.LowLevelProxyNode.isInstance(n.lowLevel()))for(var u=n.lowLevel();!u.primaryNode();)o||(o=vt(n,e,r,t,!0,s,null,i,a),i=null),n=n.parent(),u=n.lowLevel();var l=n;if(n&&(s=n.property(),n.lowLevel().unit()!=n.root().lowLevel().unit())){o=vt(n,e,r,t,!0,s,null,i,a),i=null;for(n.lowLevel().unit();n.lowLevel().unit()!=n.root().lowLevel().unit();)s=n.property(),n=n.parent()}if(o){var p=n.lowLevel().unit().project().namespaceResolver();if(p){var c=p.resolveNamespace(n.root().lowLevel().unit(),l.lowLevel().unit());if(c){var f=c.usesNodes.map(function(e){return ee(De.ISSUES_IN_THE_LIBRARY,{value:e.value()},e,e.unit().highLevel(),!0)});f.push(o),f=f.reverse();for(var d=0;d<f.length-1;d++)f[d].extras.push(f[d+1]);return o}}n.property()&&n.property().nameId()==Ae.Universe10.FragmentDeclaration.properties.uses.name&&null!=n.parent()&&(s=n.property(),n=n.parent())}var h=vt(n,e,r,t,!1,s,null,i,a);if(o){if(o.extras.push(h),n.lowLevel().valueKind()==he.Kind.INCLUDE_REF){var m=De.ERROR_IN_INCLUDED_FILE;h.code=m.code,h.message=re(m,{msg:h.message})}h=o}return h}function ee(e,t,n,r,i,a,o){void 0===i&&(i=!1),void 0===a&&(a=!1);var s=re(e,t);return te(e.code,s,n,r,i,a,o)}function te(e,t,n,r,i,a,o){void 0===i&&(i=!1),void 0===a&&(a=!1);var s=null;if(n){var u=r.root().lowLevel().unit();if(r.lowLevel().unit()!=u){s=gt(n,r,e,i,t,!0,o);var l=r.lowLevel().unit();for(l&&(t=t+" "+l.path());r.lowLevel().unit()!=u;)r=r.parent()}}s&&(o=null,r.property()&&r.property().nameId()==Ae.Universe10.FragmentDeclaration.properties.uses.name&&null!=r.parent()&&(r=r.parent()),n=r.lowLevel());var p=gt(n,r,e,i,t,a,o);return s&&(s.extras.push(p),n.valueKind()==he.Kind.INCLUDE_REF&&(p.message="Error in the included file: "+p.message),p=s),p}function ne(e){if(3!=e.length)return new He(De.STATUS_MUST_BE_3NUMBER);for(var t=0;t<e.length;t++){var n=e[t];if(!de.find(["0","1","2","3","4","5","6","7","8","9"],function(e){return e==n}))return new He(De.STATUS_MUST_BE_3NUMBER)}return null}function re(e,t){for(var n="",r=e.message,i=0,a=r.indexOf("{{");a>=0;a=r.indexOf("{{",i)){if(n+=r.substring(i,a),i=r.indexOf("}}",a),0>i){i=a;break}a+="{{".length;var o=r.substring(a,i);i+="}}".length;var s=t[o];if(void 0===s)throw new Error("Message parameter '"+o+"' has no value specified.");n+=s}return n+=r.substring(i,r.length)}function ie(e){if(!e)return!1;if(e=e.trim().toLowerCase(),e.indexOf("\n")>=0||e.indexOf("\r")>=0)return!1;Re.startsWith(e,"http://")?e=e.substring("http://".length):Re.startsWith(e,"https://")?e=e.substring("https://".length):Re.startsWith(e,"./")?e=e.substring("./".length):Re.startsWith(e,"/")&&(e=e.substring("/".length)),e=e.replace(/\.\.\//g,"");var t=e.split("/");if(0==t.length)return!1;for(var n=0,r=t;n<r.length;n++){var i=r[n];if(!/^[-a-z\\d%_.~+]+$/.test(i))return!1}return!0}function ae(e,t,n,r){if(void 0===r&&(r=!1),n!=De.SCALAR_PROHIBITED.code&&n!=De.SCALAR_PROHIBITED_2.code&&"CAN_NOT_PARSE_JSON"!=n&&"TYPE_EXPECTED"!=n&&"CONTENT_DOES_NOT_MATCH_THE_SCHEMA"!=n&&"INHERITING_UNKNOWN_TYPE"!=n&&"SCHEMA_ERROR"!=n&&"UNRECOGNIZED_ELEMENT"!=n)return!1;if(e){var i=e.lowLevel(),a=i.valueKind();if(a==he.Kind.ANCHOR_REF&&(a=i.anchorValueKind()),a==he.Kind.INCLUDE_REF)return!1;var o=e.property();o||(o=e.knownProperty);var s=Ee.isExampleProperty(o)||Ee.isExamplesProperty(o),u=Ee.isTypeOrSchemaProperty(o);if(u||o&&Ee.isGlobalSchemaType(o.domain())&&Ee.isValueProperty(o)&&(u=!0),o&&(u||s||!o.range().isValueType())){var l=e.parent();if(!l)return!1;var p=l.definition(),c=e.lowLevel().value();if("string"==typeof c&&ie(c)){if(c.indexOf(".")<0)return!1;if(s){if(!(Ee.isBodyLikeType(p)||Ee.isObjectTypeDeclarationSibling(p)||Ee.isArrayTypeDeclarationSibling(p)))return!1;if(!(Re.endsWith(c,".raml")||Re.endsWith(c,".yml")||Re.endsWith(c,".yaml")||Re.endsWith(c,".xml")||Re.endsWith(c,".json")))return!1}else if(u){if(!(Re.endsWith(c,".raml")||Re.endsWith(c,".yml")||Re.endsWith(c,".yaml")||Re.endsWith(c,".xml")||Re.endsWith(c,".json")||Re.endsWith(c,".xsd")))return!1}else if(!(Re.endsWith(c,".raml")||Re.endsWith(c,".yml")||Re.endsWith(c,".yaml")))return!1;var f=Q(De.INCLUDE_TAG_MISSING,null,e,r);return t.accept(f),!0}}return!1}}var oe=this&&this.__extends||function(e,t){function n(){this.constructor=e}for(var r in t)t.hasOwnProperty(r)&&(e[r]=t[r]);e.prototype=null===t?Object.create(t):(n.prototype=t.prototype,new n)},se=n(24),ue=n(54),le=n(69),pe=n(44),ce=n(9),fe=n(10),de=n(70),he=n(77),me=n(39),ye=n(16),ve=n(15),ge=n(42),Ae=n(18),Ee=n(14),Te=n(38),Se=me,be=n(55),_e=n(27),Ne=n(13),we=me.rt,Re=n(43),Ie=n(31),Me=n(25),Ce=me.getSchemaUtils(),Le=(n(68),n(56)),Pe=n(78),Oe=n(76),De=n(83),xe=function(){function e(){this.validateNotStrictExamples=!0}return e}(),Ue=new xe,ke=n(84);t.RESERVED_TEMPLATE_PARAMETERS={resourcePathName:'Part of the resource path following the rightmost "/"',methodName:"Method name",resourcePath:"Path of the resource"};var Fe=function(e,t,n){if(ge.existsSync(e))try{var i=ge.readFileSync(e).toString(),a=new Be(t);r(i,a,null),a.visit(n)}catch(o){console.log("Error in custom linter"),console.log(o)}},Be=function(){function e(e){this.acceptor=e,this.nodes={}}return e.prototype.error=function(e,t){this.acceptor.accept(Q(De.INVALID_VALUE_SCHEMA,{iValue:t},e.highLevel()))},e.prototype.errorOnProperty=function(e,t,n){var r=e.highLevel().attr(t);this.acceptor.accept(Q(De.INVALID_VALUE_SCHEMA,{iValue:n},r))},e.prototype.warningOnProperty=function(e,t,n){var r=e.highLevel().attr(t);this.acceptor.accept(Q(De.INVALID_VALUE_SCHEMA,{iValue:n},r,!0))},e.prototype.warning=function(e,t){this.acceptor.accept(Q(De.INVALID_VALUE_SCHEMA,{iValue:t},e.highLevel(),!0))},e.prototype.registerRule=function(e,t){var n=this.nodes[e];n||(n=[],this.nodes[e]=n),n.push(t)},e.prototype.visit=function(e){var t=this,n=e.definition();this.process(n,e),n.allSuperTypes().forEach(function(n){return t.process(n,e)}),e.elements().forEach(function(e){return t.visit(e)})},e.prototype.process=function(e,t){var n=this;if(me.NodeClass.isInstance(e)&&!e.getAdapter(Se.RAMLService).getDeclaringNode()){var r=this.nodes[e.nameId()];r&&r.forEach(function(e){return e(t.wrapperNode(),n)})}},e}();(function(){function e(){}return e.prototype.toString=function(){return this.prev?this.value+"."+this.prev.toString():this.value},e.prototype.last=function(){return this.prev?this.prev.last():this},e})();t.validateBasicFlat=h;(function(){function e(){}return e})();t.validateBasic=b;var Ke=function(e,t){var n=t.lowLevel().start(),r=[];if(0>n){for(var i=t.attr("key").value().split("."),a=[],o=t.parent(),s=0,u=i;s<u.length;s++){var l=u[s],p=de.find(o.elementsOfKind("uses"),function(e){return e.attr("key")&&e.attr("key").value()==l});a.push(p),o=p.lowLevel().unit().resolve(p.attr("value").value()).highLevel().asElement()}var c=a.map(function(e){return Q(De.ISSUES_IN_THE_LIBRARY,{value:e.attr("value").value()},e,!0)});c=c.reverse();for(var f=0;f<c.length-1;f++)c[f].extras.push(c[f+1]);return c[0]}return r.push(t),Q(De.ISSUES_IN_THE_LIBRARY,{value:e.value()},t,!0)};t.validate=N;var Ve=function(t){t=t.replace(/\\/g,"/").trim(),t.length>1&&":"==t.charAt(1)&&/^win/.test(e.platform)&&(t=t.substring(2));var n=t.split("/");return 0==n[0].length&&(n=n.slice(1)),n.length>0&&0==n[n.length-1].length&&(n=n.slice(0,n.length-1)),n},je=function(e,t,n){try{new RegExp(e)}catch(r){t.accept(Q(De.ILLEGAL_PATTERN,{value:e},n))}},We=function(){function e(){}return e.prototype.validateName=function(e,t){var n=e.name();if(n){var r=e.lowLevel().keyStart();this.check(n,r,e,t)}},e.prototype.validateValue=function(e,t){var n=e.value();if("string"==typeof n){var r=e.lowLevel().valueStart();this.check(n,r,e,t)}},e.prototype.hasTraitOrResourceTypeParent=function(e){for(var t=e.parent();null!=t;){if(!t.definition())return!1;if(Ee.isTraitType(t.definition())||Ee.isResourceTypeType(t.definition()))return!0;t=t.parent()}return!1},e.prototype.check=function(e,t,n,r){if(!this.hasTraitOrResourceTypeParent(n))return[];for(var i=[],a=0,o=e.indexOf("<<");o>=0;o=e.indexOf("<<",a)){o+="<<".length,a=e.indexOf(">>",o);var s=e.substring(o,a),u=s.indexOf("|"),l=u>=0?s.substring(0,u):s;if(0==l.trim().length){var p=Q(De.TEMPLATE_PARAMETER_NAME_MUST_CONTAIN_NONWHITESPACE_CHARACTERS,{},n);p.start=t+o,p.end=t+a,r.accept(p)}if(-1!=u){u++;for(var c=s.split("|").slice(1).map(function(e){return e.trim()}),f=_e.getTransformNames(),d=0,h=c;d<h.length;d++){var m=h[d];if(!de.find(f,function(e){return m===e||m==="!"+e})){var p=Q(De.UNKNOWN_FUNCTION,{transformerName:m},n,!1);p.start=t+u,p.end=t+a,r.accept(p)}}}a+=">>".length}return i},e}(),qe=function(){function e(){}return e.prototype.validate=function(t,n){var r=t.parent();if(r&&(r.definition().isAssignableFrom(Ae.Universe08.Method.name)||r.definition().isAssignableFrom(Ae.Universe10.Method.name))){var i=de.find(r.lowLevel()&&r.lowLevel().children()||[],function(e){var t=e.key();return t&&(Ae.Universe08.MethodBase.properties.body.name===t||Ae.Universe10.MethodBase.properties.body.name===t)});i&&de.find(e.methodsWithoutRequestBody,function(e){return r.name()===e})&&n.accept(Q(De.REQUEST_BODY_DISABLED,{methodName:r.name()},r))}},e.methodsWithoutRequestBody=["trace"],e}(),Ye=function(){function e(){}return e.prototype.validate=function(e,t){var n=i(e,t),r=e.value(),a=e.parent().definition().universe().version(),l=null!=c(e.parent());if(!e.property().range().hasStructure()){if(ye.StructuredValue.isInstance(r)&&!e.property().isSelfNode()){if(o(e.property())&&e.property().domain().key()==Ae.Universe08.BodyLike){var f=new ye.ASTNodeImpl(e.lowLevel(),e.parent(),e.parent().definition().universe().type(Ae.Universe08.BodyLike.name),e.property());return void f.validate(t)}if("RAML10"==a&&l)return;t.accept(Q(De.SCALAR_EXPECTED,{},e))}else{var d=e.lowLevel().valueKind();if(e.lowLevel().valueKind()!=he.Kind.INCLUDE_REF&&!e.property().getAdapter(Se.RAMLPropertyService).isKey()&&!e.property().isMultiValue()){var h=e.property().range().key();(h==Ae.Universe08.StringType||h==Ae.Universe08.MarkdownString||h==Ae.Universe08.MimeType)&&(d!=he.Kind.SEQ&&d!=he.Kind.MAPPING&&d!=he.Kind.MAP&&(!e.property().isRequired()&&"mediaType"!=e.property().nameId()||null!=d&&void 0!==d)||e.property().domain().getAdapter(Se.RAMLService).isInlinedTemplates()||t.accept(Q(De.STRING_EXPECTED,{propName:e.name()},e)))}}if(e.isAnnotatedScalar()){var m=new tt;e.annotations().forEach(function(e){var n=e.value(),r=n.toHighLevel();r?m.validate(r,t):t.accept(Q(De.UNKNOWN_ANNOTATION,{aName:n.valueName()},e))})}}var y;if("string"==typeof r?y=r:ye.StructuredValue.isInstance(r)&&(y=r.valueName()),!(y&&-1!=y.indexOf("<<")&&y.indexOf(">>")>y.indexOf("<<")&&((new We).validateValue(e,t),l))){if((new qe).validate(e,t),e.property().range().key()==Ae.Universe08.MimeType||e.property().range().key()==Ae.Universe10.MimeType||e.property().nameId()==Ae.Universe10.TypeDeclaration.properties.name.name&&e.parent().property().nameId()==Ae.Universe10.MethodBase.properties.body.name)return void(new Xe).validate(e,t);if((u(e.property())||s(e.property()))&&(new ct).validate(e,t),p(e.property())){if("RAML08"==a){var v=e.lowLevel().parent(),g=he.Kind.SEQ;pe.LowLevelProxyNode.isInstance(e.lowLevel())?v.valueKind()!=g&&t.accept(Q(De.SECUREDBY_LIST_08,{},e,!1)):v.kind()!=g&&t.accept(Q(De.SECUREDBY_LIST_08,{},e,!1))}if((new ct).validate(e,t),"RAML10"==a&&ye.StructuredValue.isInstance(r)){var A=r,E=A.children().filter(function(e){return"scopes"==e.valueName()});if(E.length>0){var T=e.findReferencedValue();if(T){var S=[];E.forEach(function(e){var t=e.children();if(t.length>0)t.forEach(function(e){var t=e.lowLevel().value();null==t||l&&t.indexOf("<<")>=0||S.push(e)});else{var n=e.lowLevel().value();null==n||l&&n.indexOf("<<")>=0||S.push(e)}});var b={},_=T.element(me.universesInfo.Universe10.AbstractSecurityScheme.properties.settings.name);if(_){var N=_.attributes(me.universesInfo.Universe10.OAuth2SecuritySchemeSettings.properties.scopes.name);N.forEach(function(e){return b[e.value()]=!0})}for(var w=0,R=S;w<R.length;w++){var I=R[w],M=I.lowLevel().value();b.hasOwnProperty(M)||t.accept(ee(De.INVALID_SECURITY_SCHEME_SCOPE,{invalidScope:M,securityScheme:T.name(),allowedScopes:Object.keys(b).map(function(e){return"'"+e+"'"}).join(", ")},I.lowLevel(),e,!1))}}}}}if(e.property().nameId()==Ae.Universe10.TypeDeclaration.properties.name.name){var C=e.parent().property()&&e.parent().property().nameId();if(C==Ae.Universe08.Resource.properties.uriParameters.name||C==Ae.Universe08.Resource.properties.baseUriParameters.name)return}var L=e.property().range().key();if(L==Ae.Universe08.RelativeUriString||L==Ae.Universe10.RelativeUriString)return void(new Ge).validate(e,t);if(L==Ae.Universe08.FullUriTemplateString||L==Ae.Universe10.FullUriTemplateString)return void(new Ge).validate(e,t);if("pattern"==e.name()&&Ae.Universe10.StringType==e.definition().key()&&e.parent().definition().isAssignableFrom("StringTypeDeclaration")&&je(e.value(),t,e),"name"==e.name()&&Ae.Universe10.StringType==e.definition().key()&&"string"==typeof e.value()&&0==e.value().indexOf("[")&&e.value().lastIndexOf("]")==e.value().length-1&&ye.ASTNodeImpl.isInstance(e.parent())&&Ae.Universe10.ObjectTypeDeclaration.properties.properties.name==e.parent().property().nameId()&&ye.ASTNodeImpl.isInstance(e.parent().parent())&&Ae.Universe10.ObjectTypeDeclaration==e.parent().parent().definition().key()){var P=e.value().substr(1,e.value().length-2);je(P,t,e)}n.isReference()||n.isDescriminator()?(new Je).validate(e,t):(new $e).validate(e,t)}},e}();t.isValid=C;var He=function(e){function t(t,n){void 0===n&&(n={}),e.call(this),this.messageEntry=t,this.parameters=n,this.isWarning=!1}return oe(t,e),t.isInstance=function(e){return null!=e&&e.getClassIdentifier&&"function"==typeof e.getClassIdentifier&&de.contains(e.getClassIdentifier(),t.CLASS_IDENTIFIER_ValidationError)},t.prototype.getClassIdentifier=function(){var e=[];return e.concat(t.CLASS_IDENTIFIER_ValidationError)},t.CLASS_IDENTIFIER_ValidationError="linter.ValidationError",t}(Error),$e=function(){function e(){}return e.prototype.validate=function(e,t){var n=e.value(),r=e.property(),i=r.range(),a=i.getAdapter(Se.RAMLService).getDeclaringNode();if(a&&i.isUserDefined()){var o=a.parsedType(),s=e.parent().lowLevel().dumpToObject(),u=s[e.parent().name()],l=r.canBeValue(),p=l||null===u||void 0===u?u:u[r.nameId()],c=o.validate(p,!0);c.isOk()||c.getErrors().forEach(function(n){return t.accept(Z(n.getCode(),n.getMessage(),e,!1))})}var f=t;if(e.lowLevel().keyKind()!=he.Kind.SEQ)var d=C(r.range(),e.parent(),n,r,e);else d=!0;if(d instanceof Error&&!d.canBeRef)return He.isInstance(d)?f.accept(J(d,e)):f.accept(Q(De.SCHEMA_EXCEPTION,{msg:d.message},e,d.isWarning)),void(d=null);if(!d||d instanceof Error)if("value"!=r.nameId()){if(!U(r,e,n,f)){if((r.nameId()==Ae.Universe10.TypeDeclaration.properties.schema.name||Ae.Universe10.TypeDeclaration.properties.type.name)&&n&&n.trim()&&(r.domain().key()==Ae.Universe08.BodyLike||r.domain().key()==Ae.Universe10.TypeDeclaration)){var h=n.trim().charAt(0);if("{"!=h&&"<"!=h)return}var m=e.findReferencedValue();if(m instanceof Error){var y=e.findReferenceDeclaration();if(y){var v=void 0,g=void 0;if(He.isInstance(m)){var A=m;v=J(A,y),g=Q(A.messageEntry,{msg:A.message},e)}else v=Q(De.SCHEMA_EXCEPTION,{msg:m.message},y),g=Q(De.SCHEMA_EXCEPTION,{msg:m.message},e);v.extras.push(g),f.accept(v)}}if(!m){if(n&&r.nameId()==Ae.Universe10.TypeDeclaration.properties.schema.name){var E=n.trim();if("{"!=E.charAt(0)&&"<"!=E.charAt(0)&&(-1!=n.indexOf("|")||-1!=n.indexOf("[]")||-1!=n.indexOf("(")))return}if(d instanceof Error&&n)return He.isInstance(d)?f.accept(J(d,e)):f.accept(Q(De.SCHEMA_EXCEPTION,{msg:d.message},e)),void(d=null);if(e.property().isRequired()&&null==e.value())f.accept(Q(De.EMPTY_VALUE_NOT_ALLOWED,{},e));else{var T=e.lowLevel().valueKind();(T==he.Kind.MAP||T==he.Kind.SEQ||T==he.Kind.MAPPING)&&f.accept(Q(De.EMPTY_VALUE_NOT_ALLOWED,{},e))}}}}else{var S=r.range().key()==Ae.Universe08.SchemaString;if(d instanceof Error){var b=d.message;ae(e,f,De.SCHEMA_ERROR.code,S)||f.accept(Q(De.SCHEMA_ERROR,{msg:b},e,S))}else{var n=e.value();f.accept(Q(De.INVALID_VALUE_SCHEMA,{iValue:n},e,S))}}var _=r.enumOptions();if(_){var N=e.parent()&&e.parent().definition(),w=N&&N.isAssignableFrom(Ae.Universe10.Api.name),R=N&&N.isAssignableFrom(Ae.Universe08.Api.name),I=r.nameId()===Ae.Universe08.Api.properties.protocols.name,M=r.nameId()===Ae.Universe10.Api.properties.protocols.name;if("string"!=typeof n)return;!R&&!w||!I&&!M||O(n)||(n=n.toUpperCase()),"string"==typeof _?_!=n&&(n&&0==n.indexOf("x-")&&r.nameId()==Ae.Universe08.AbstractSecurityScheme.properties.type.name||f.accept(Q(De.INVALID_VALUE,{iValue:n,aValues:"'"+_+"'"},e))):_.length>0&&(de.find(_,function(e){return e==n})||n&&0==n.indexOf("x-")&&r.nameId()==Ae.Universe08.AbstractSecurityScheme.properties.type.name||f.accept(Q(De.INVALID_VALUE,{iValue:n,aValues:_.map(function(e){return"'"+e+"'"}).join(", ")},e)))}},e}(),Ge=function(){function e(){}return e.prototype.validate=function(e,t){try{var n=(new ze).parseUrl(e.value()||"");if(n.some(function(e){return"version"==e})&&"baseUri"==e.property().nameId()){var r=e.root().attr("version");r||t.accept(Q(De.MISSING_VERSION,{},e,!1))}n.some(function(e){return 0==e.length})&&t.accept(Q(De.URI_PARAMETER_NAME_MISSING,{},e,!1))}catch(i){t.accept(Q(De.URI_EXCEPTION,{msg:i.message},e,!1))}},e}(),Xe=function(){function e(){}return e.prototype.validate=function(e,t){try{var n=e.value();if("body"==n&&e.parent().parent()){var r=e.parent().parent().definition().key();(r===Ae.Universe08.Response||r===Ae.Universe10.Response||e.parent().parent().definition().isAssignableFrom(Ae.Universe10.MethodBase.name))&&(n=e.parent().computedValue("mediaType"))}if(e.parent()&&e.parent().parent()&&e.parent().parent().definition().isAssignableFrom(Ae.Universe10.Trait.name)&&n.indexOf("<<")>=0)return;var i=_e.parseMediaType(n);if(!i)return;i.type.match(/[\w\d][\w\d!#\$&\-\^_+\.]*/)||t.accept(Q(De.INVALID_MEDIATYPE,{mediaType:i.type},e))}catch(a){t.accept(Q(De.MEDIATYPE_EXCEPTION,{msg:a.message},e))}(e.value()&&"multipart/form-data"==e.value()||"application/x-www-form-urlencoded"==e.value())&&e.parent()&&e.parent().parent()&&e.parent().parent().property()&&e.parent().parent().property().nameId()==Ae.Universe10.MethodBase.properties.responses.name&&t.accept(Q(De.FORM_IN_RESPONSE,{},e,!0))},e}(),ze=function(){function e(){}return e.prototype.checkBaseUri=function(e,t,n,r){var i=t.root().attr("baseUri");if(i){var a=i.value();try{var o=this.parseUrl(a);de.find(o,function(e){return e==n})||r.accept(Q(De.UNUSED_URL_PARAMETER,{paramName:""},e))}catch(s){}}else r.accept(Q(De.UNUSED_URL_PARAMETER,{paramName:""},e))},e.prototype.parseUrl=function(e){for(var t=[],n="",r=!1,i=0,a=0;a<e.length;a++){var o=e[a];"{"!=o?"}"!=o?r&&(n+=o):(i--,r=!1,t.push(n),n=""):(i++,r=!0)}if(i>0)throw new Error("Invalid resource name: unmatched '{'");if(0>i)throw new Error("Invalid resource name: unmatched '}'");return t},e.prototype.validate=function(e,t){var n=e.value();if(e.parent().property().nameId()==Ae.Universe10.Api.properties.baseUri.name){var r=e.parent().parent();return void this.checkBaseUri(e,r,n,t)}var r=e.parent().parent(),i=r.name();if(r.definition().key()===Ae.Universe10.Api||r.definition().key()===Ae.Universe08.Api)return void this.checkBaseUri(e,r,n,t);if(r.definition().key()!=Ae.Universe10.ResourceType&&r.definition().key()!=Ae.Universe08.ResourceType)try{var a=this.parseUrl(i),o=de.find(a,function(e){return e==n});if(!o){var s=e.root().attr(Ae.Universe10.Api.properties.baseUri.name);if(s&&e.name()===Ae.Universe08.Api.properties.baseUriParameters.name){var u=s.value();if(u&&(a=this.parseUrl(u),a&&a.length>0&&de.find(a,function(e){return e==n})))return}t.accept(Q(De.UNUSED_URL_PARAMETER,{paramName:"'"+n+"'"},e))}}catch(l){}},e}();t.UrlParameterNameValidator=ze,t.typeToName={},t.typeToName[Ae.Universe08.Trait.name]="trait",t.typeToName[Ae.Universe08.ResourceType.name]="resource type",t.typeToName[Ae.Universe10.Trait.name]="trait",t.typeToName[Ae.Universe10.ResourceType.name]="resource type",t.typeToName[Ae.Universe10.AbstractSecurityScheme.name]="security scheme",t.typeToName[Ae.Universe10.Method.name]="method",t.typeToName[Ae.Universe08.Method.name]="method",t.typeToName[Ae.Universe10.Resource.name]="resource",t.typeToName[Ae.Universe08.Resource.name]="resource",t.typeToName[Ae.Universe10.Api.name]="api",t.typeToName[Ae.Universe08.Api.name]="api",t.typeToName[Ae.Universe10.Response.name]="response",t.typeToName[Ae.Universe08.Response.name]="response",t.typeToName[Ae.Universe08.BodyLike.name]="body",t.parameterPropertyToName={},t.parameterPropertyToName[Ae.Universe08.MethodBase.properties.headers.name]="header",t.parameterPropertyToName[Ae.Universe08.MethodBase.properties.queryParameters.name]="query parameter",t.parameterPropertyToName[Ae.Universe08.Api.properties.uriParameters.name]="uri parameter",t.parameterPropertyToName[Ae.Universe08.Api.properties.baseUriParameters.name]="base uri parameter",t.parameterPropertyToName[Ae.Universe08.BodyLike.properties.formParameters.name]="form parameter",t.parameterPropertyToName[Ae.Universe10.MethodBase.properties.headers.name]="header",t.parameterPropertyToName[Ae.Universe10.MethodBase.properties.queryParameters.name]="query parameter",t.parameterPropertyToName[Ae.Universe10.ResourceBase.properties.uriParameters.name]="uri parameter",t.parameterPropertyToName[Ae.Universe10.Api.properties.baseUriParameters.name]="base uri parameter",t.parameterPropertyToName[Ae.Universe10.MethodBase.properties.body.name]="body",t.getHumanReadableNodeName=D;var Je=function(){function e(){}return e.prototype.validate=function(e,t){var n=e.value(),r=n,i=e.property();if("string"==typeof n){if(U(i,e,n,t),me.ReferenceType.isInstance(i.range())){var a=(i.range(),se.createNode(""+n,e.lowLevel().parent(),e.lowLevel().unit()));a._actualNode().startPosition=e.lowLevel().valueStart(),a._actualNode().endPosition=e.lowLevel().valueEnd();var o=new ye.StructuredValue(a,e.parent(),e.property()),s=o.toHighLevel();s&&s.validate(t)}}else if(ye.StructuredValue.isInstance(n)){var u=n;if(u){r=u.valueName();var l=u.valueName();if(!U(i,e,l,t)){var p=u.toHighLevel();p&&p.validate(t)}}else r=null}else"number"==typeof n||"boolean"==typeof n?e.definition().isAssignableFrom(Ae.Universe10.Reference.name)&&U(i,e,n+"",t):e.definition().isAssignableFrom(Ae.Universe10.Reference.name)&&U(i,e,null,t);if(r){var c=C(i.range(),e.parent(),r,i);c instanceof Error&&(He.isInstance(c)?t.accept(J(c,e)):t.accept(Q(De.SCHEMA_EXCEPTION,{msg:c.message},e,c.isWarning)),c=null)}},e}(),Qe=function(){function e(){}return e.prototype.validate=function(e,t){var n=e.universe(),r=n.getTypedVersion();if(r){if("0.8"!==r&&"1.0"!==r){var i=Q(De.UNKNOWN_RAML_VERSION,{},e);t.accept(i)}var a=n.getOriginalTopLevelText();if(a){var o={typeName:a};if(a!=e.definition().nameId()){if("Api"==e.definition().nameId()){var i=Q(De.UNKNOWN_TOPL_LEVEL_TYPE,o,e);t.accept(i)}}else if("Api"==n.getOriginalTopLevelText()){var i=Q(De.REDUNDANT_FRAGMENT_NAME,o,e);t.accept(i)}}}},e}(),Ze=function(){function e(){}return e.prototype.validate=function(e,t){var n=this;e.definition().getAdapter(Se.RAMLService).getContextRequirements().forEach(function(n){if(!e.checkContextValue(n.name,n.value,n.value)){var r={v1:n.name,v2:n.value,v3:e.definition().nameId()},i=De.CONTEXT_REQUIREMENT_VIOLATION;"location"==n.name&&"ParameterLocation.FORM"==n.value&&(i=De.WEB_FORMS),t.accept(Q(i,r,e))}});var r,i=e.definition().getAdapter(Se.RAMLService).isInlinedTemplates();if(i){for(var a={},o=0,s=e.lowLevel().children();o<s.length;o++){var u=s[o];a[u.key()]=u.value(!0)}for(var l=e.definition().isAssignableFrom(Ae.Universe10.Trait.name)?"trait":"resource type",p=_e.toUnits(e),c=new _e.ValueTransformer(l,e.definition().nameId(),p,a),f=e.parent(),d=f?f.definition():e.definition();null!=f&&!Ee.isResourceType(d)&&!Ee.isMethodType(d);)f=f.parent();r=new _e.DefaultTransformer(f,c,p)}e.definition().requiredProperties().forEach(function(a){if(i){var o=a.getAdapter(Se.RAMLPropertyService).meta("templatePaths");if(o){for(var s=e.parent(),u=!1,l=0,p=o;l<p.length;l++){var c=p[l];if(c=c.map(function(e){return r.transform(e).value}),n.checkPathSufficiency(s.lowLevel(),c,s)){u=!0;break}}if(!u)return}}var f=a.range();if(f.hasArrayInHierarchy()&&(f=f.arrayInHierarchy().componentType()),f.hasValueTypeInHierarchy()){var d=e.attr(a.nameId()),h=!1;if(null!=d&&(d.lowLevel().kind()==he.Kind.SCALAR||d.lowLevel().resolvedValueKind()==he.Kind.SCALAR||d.lowLevel().kind()==he.Kind.INCLUDE_REF||d.lowLevel().valueKind()==he.Kind.INCLUDE_REF||null===d.lowLevel().valueKind()&&!i?h=!0:0!=d.lowLevel().children().length&&(h=!0)),!h){var m={propName:a.nameId()},y=De.MISSING_REQUIRED_PROPERTY;i&&(y=De.VALUE_NOT_PROVIDED);var v=Q(y,m,e);t.accept(v)}}else{var g=e.elementsOfKind(a.nameId());if(!g||0==g.length){var v=Q(De.MISSING_REQUIRED_PROPERTY,{propName:a.nameId()},e);t.accept(v)}}})},e.prototype.checkPathSufficiency=function(e,t,n){if(null==n||null==n.definition())return!1;var r=n.definition();if(Ee.isResourceTypeType(r)||Ee.isTraitType(r))return!0;if(0==t.length)return!1;if(null==e)return!1;var i=t[0];if(null==i)return!1;if("/"==i)return this.checkPathSufficiency(e,t.slice(1),n);if(0==i.length)return!0;var a=e.children().filter(function(e){return e.key()==i});0==a.length&&t.indexOf("/")<0;var o=a[0];if(pe.LowLevelCompositeNode.isInstance(o)&&(o=o.primaryNode()),null==o)return t.indexOf("/")<0;if("type"==o.key())return!0;if(1==t.length)return null==o||null==o.value();var s=t.slice(1);return this.checkPathSufficiency(o,s,n)},e}(),et=function(){function e(){}return e.prototype.validate=function(e,t){var n=e.lowLevel().unit();e.lowLevel().visit(function(r){if(r.unit()!=n)return!1;if(r.value()&&r._node&&r._node.value&&r._node.value.doubleQuoted){var i=(r.value()+"").indexOf(":"),a=(r.value()+"").indexOf("\n");if(-1!=i&&-1!=a&&(!r.includePath()||0==r.includePath().length)){var o=Q(De.SUSPICIOUS_DOUBLEQUOTE,{value:r.value()},e,!0);o.start=r._node.value.startPosition,o.end=r._node.value.endPosition,o.start==o.end&&o.end++,t.accept(o)}}return!0})},e}(),tt=function(){function e(){}return e.prototype.validate=function(e,t){var n=e.definition(),r=n.getAdapter(Se.RAMLService).getDeclaringNode();if(r){var i=r.parsedType(),a=e.lowLevel().dumpToObject(!0);a&&(a=a[Object.keys(a)[0]]);var o=i.validate(a,!1,!1);o.isOk()||o.getErrors().forEach(function(n){return t.accept(Z(n.getCode(),n.getMessage(),j(e,n).node,!1))})}},e}(),nt=function(){function e(){this.annotables={API:!0,DocumentationItem:!0,Resource:!0,Method:!0,Response:!0,RequestBody:!0,ResponseBody:!0,TypeDeclaration:!0,Example:!0,ResourceType:!0,Trait:!0,SecurityScheme:!0,SecuritySchemeSettings:!0,AnnotationType:!0,Library:!0,Overlay:!0,Extension:!0}}return e.prototype.validate=function(e,t){var n=(e.definition(),e.parsedType()),r=n.validateType(e.types().getAnnotationTypeRegistry());if(!r.isOk())for(var i=0,a=r.getErrors();i<a.length;i++){var o=a[i],s=W(o,e.lowLevel().unit().project()),u=void 0,l=j(e,o);if(l.node==e&&!l.internalPathUsed){var p=o.getValidationPath();if(p&&p.name==me.universesInfo.Universe10.TypeDeclaration.properties.type.name&&e.attr(me.universesInfo.Universe10.TypeDeclaration.properties.schema.name)){var c=p.name;p.name=me.universesInfo.Universe10.TypeDeclaration.properties.schema.name,l=j(e,o),p.name=c}}var f=l.internalPathUsed?null:o.getInternalRange();if(s){if(u=te(o.getCode(),o.getMessage(),s,l.node,o.isWarning(),!0,f),s.unit().absolutePath()!=e.lowLevel().unit().absolutePath()){var d=Z(o.getCode(),o.getMessage(),e,o.isWarning());u.extras.push(d)}}else{if(o.getFilePath()&&o.getFilePath()!=e.lowLevel().unit().absolutePath()){var h=e.lowLevel().unit().project().unit(o.getFilePath(),!0);if(h){var m=h.highLevel();if(m){u=Z(o.getCode(),o.getMessage(),m,o.isWarning(),f,!0);
var d=Z(o.getCode(),o.getMessage(),l.node,o.isWarning());u.extras.push(d)}}}if(!u){if(ae(l.node,t,o.getCode(),o.isWarning()))continue;u=Z(o.getCode(),o.getMessage(),l.node,o.isWarning(),f)}}var y=o.getFilePath();if(null!=y){var v=e.lowLevel().unit().project().unit(y);v&&(u.unit=v,u.path=y)}t.accept(u)}var g=e.lowLevel()&&de.find(e.lowLevel().children(),function(e){return"examples"==e.key()});if(g&&g.valueKind&&g.valueKind()===he.Kind.SEQ){var u=ee(De.MAP_EXPECTED,{},g,e,!1);t.accept(u)}if(e.property()&&Ee.isAnnotationTypesProperty(e.property())||ye.isAnnotationTypeFragment(e))for(var A=e.attributes(Ae.Universe10.TypeDeclaration.properties.allowedTargets.name),E=0,T=A;E<T.length;E++){var S=T[E];this.checkAnnotationTarget(S,t)}},e.prototype.checkAnnotationTarget=function(e,t){var n=e.value();if(null!=n){if("string"!=typeof n)return void t.accept(Q(De.ANNOTATION_TARGET_MUST_BE_A_STRING,{},e,!1));var r=n;n.replace(/\w|\s/g,"").length>0?t.accept(Q(De.ALLOWED_TARGETS_MUST_BE_ARRAY,{},e,!1)):this.annotables[r]||t.accept(Q(De.UNSUPPORTED_ANNOTATION_TARGET,{aTarget:r},e,!1))}},e}(),rt=function(){function e(){}return e.prototype.validate=function(e,t){if(!e.definition().isAnnotationType()){if(e.lowLevel().keyKind()==he.Kind.SEQ){var n=e.definition().isAssignableFrom(Ae.Universe10.TypeDeclaration.name);n||t.accept(Q(De.NODE_KEY_IS_A_SEQUENCE,{},e))}var r=e.name();if(null==r&&(r=e.lowLevel().key(),null==r&&(r="")),e.definition().key()==Ae.Universe08.GlobalSchema&&e.lowLevel().valueKind()!=he.Kind.SCALAR){var o=!1;if(e.lowLevel().valueKind()==he.Kind.ANCHOR_REF||e.lowLevel().valueKind()==he.Kind.INCLUDE_REF){var s=e.lowLevel().value();"string"==typeof s&&(o=!0)}o||t.accept(Q(De.SCHEMA_NAME_MUST_BE_STRING,{name:r},e))}e.parent()||((new Qe).validate(e,t),(e.definition().key()==Ae.Universe08.Api||e.definition().key()==Ae.Universe10.Api)&&(new be).validateApi(e.wrapperNode(),t),(new et).validate(e,t),a(e,t)),(new st).validate(e,t);var u=e.definition();if(u.key()==Ae.Universe08.BodyLike&&e.lowLevel().children().map(function(e){return e.key()}).some(function(e){return"formParameters"===e}))if(e.parent()&&e.parent().definition().key()==Ae.Universe08.Response){var l=Q(De.FORM_PARAMS_IN_RESPONSE,{},e);t.accept(l)}else if(e.lowLevel().children().map(function(e){return e.key()}).some(function(e){return"schema"===e||"example"===e})){var l=Q(De.FORM_PARAMS_WITH_EXAMPLE,{},e);t.accept(l)}if(u.key()==Ae.Universe10.OAuth2SecuritySchemeSettings){var p=!1;if(e.attributes("authorizationGrants").forEach(function(e){var n=e.value();if("authorization_code"===n||"implicit"===n)p=!0;else if("password"!==n&&"client_credentials"!==n&&n&&"string"==typeof n&&-1==n.indexOf("://")&&-1==n.indexOf(":")){var r=Q(De.AUTHORIZATION_GRANTS_ENUM,{},e);t.accept(r)}}),p&&!e.attr("authorizationUri")){var l=Q(De.AUTHORIZATION_URI_REQUIRED,{},e);t.accept(l)}}if(e.definition().isAssignableFrom(Ae.Universe08.Parameter.name)||e.definition().isAssignableFrom(Ae.Universe10.TypeDeclaration.name)){var c=e.attributes("enum").map(function(e){return e.value()});if(c.length!=de.uniq(c).length){var l=Q(De.REPEATING_COMPONENTS_IN_ENUM,{},e);t.accept(l)}if(e.definition().isAssignableFrom(Ae.Universe08.NumberTypeDeclaration.name)||e.definition().isAssignableFrom(Ae.Universe10.NumberTypeDeclaration.name)){var f=e.definition().isAssignableFrom(Ae.Universe08.IntegerTypeDeclaration.name)||e.definition().isAssignableFrom(Ae.Universe10.IntegerTypeDeclaration.name);e.attributes("enum").forEach(function(e){var n=f?parseInt(e.value()):parseFloat(e.value()),r=f?!isNaN(n)&&-1===e.value().indexOf("."):!isNaN(n);if(!r){var i=Q(f?De.INTEGER_EXPECTED:De.NUMBER_EXPECTED_2,{},e);t.accept(i)}})}}Ee.isResourceTypeType(e.definition())&&null==e.value()&&"null"===e.lowLevel().value(!0)&&t.accept(Q(De.RESOURCE_TYPE_NULL,{},e)),i(e,t);var d=e.value();if(("string"==typeof d||"number"==typeof d||"boolean"==typeof d)&&!e.definition().getAdapter(Se.RAMLService).allowValue()&&e.parent()&&"~"!=d&&!ae(e,t,De.SCALAR_PROHIBITED_2.code)){var l=Q(De.SCALAR_PROHIBITED_2,{name:r},e);t.accept(l)}(new Ze).validate(e,t),(new pt).validate(e,t),(new ot).validate(e,t)}},e}(),it=function(){function e(){}return e.prototype.validate=function(e,t){var n=e.attrValue(Ae.Universe10.TypeDeclaration.properties.name.name);"version"==n&&t.accept(Q(De.VERSION_NOT_ALLOWED,{},e))},e}(),at=function(){function e(e,t,n,r){void 0===r&&(r=!1),this.definitions=e,this.propertyName=t,this.assignableFrom=r,this.validator=n}return e.prototype.validate=function(e,t){var n=e.definition();if(null!=n){var r=!1;if(r=this.assignableFrom?this.definitions.some(function(e){return n.isAssignableFrom(e.name)}):this.definitions.some(function(e){return e===n})){if(null!=this.propertyName){if(null==e.property())return;if(e.property().nameId()!=this.propertyName)return}this.validator.validate(e,t)}}},e}(),ot=function(){function e(){}return e.createRegistry=function(){var t=[];return e.registerValidator(t,[Ae.Universe10.TypeDeclaration,Ae.Universe08.Parameter],Ae.Universe10.Api.properties.baseUriParameters.name,new it,!0),t},e.registerValidator=function(e,t,n,r,i){void 0===i&&(i=!1);var a=new at(t,n,r,i);e.push(a)},e.prototype.validate=function(t,n){e.entries.forEach(function(e){return e.validate(t,n)})},e.entries=e.createRegistry(),e}(),st=function(){function e(){}return e.prototype.allowsAnyChildren=function(e,t){var n=e.property(),r=e.definition();return(Ee.isAnnotationTypeType(r)||Ee.isTypeDeclarationTypeOrDescendant(r))&&Ee.isAnnotationTypesProperty(n)?!0:e.parent()==t&&Ee.isTypesProperty(n)&&Ee.isTypeDeclarationTypeOrDescendant(r)?!0:Ee.isSchemasProperty(n)&&Ee.isTypeDeclarationTypeOrDescendant(r)?!0:e.parent()==t&&Ee.isDocumentationProperty(n)&&Ee.isDocumentationType(r)?!0:Ee.isAnnotationsProperty(n)?!0:Ee.isUsesProperty(n)?!0:Ee.isExamplesProperty(n)?!0:!1},e.prototype.nodeAllowedDueToParent=function(e,t){for(var n=e;n!=t&&null!=n;){if(this.allowsAnyChildren(n,t))return!0;n=n.parent()}return!1},e.prototype.validate=function(e,t){var n=e.root();if(!n.isExpanded()||n.lowLevel().unit().absolutePath()==e.lowLevel().unit().absolutePath()){e.property(),e.definition();if(Ee.isOverlayType(n.definition())){if(e==n)return void this.validateProperties(e,t);if(!this.nodeAllowedDueToParent(e,n)){var r=n.knownIds();if(r){var i=r.hasOwnProperty(e.id());return i?void this.validateProperties(e,t):void t.accept(Q(De.INVALID_OVERLAY_NODE,{nodeId:e.id()},e))}}}}},e.prototype.validateProperties=function(e,t){var n=e.root(),r=n.lowLevel().unit().absolutePath(),i=n.isExpanded();e.attrs().forEach(function(n){i&&r!=n.lowLevel().unit().absolutePath()||n.property().getAdapter(Se.RAMLPropertyService).isKey()||n.parent()==e&&(n.isElement()||Ee.isTitlePropertyName(n.name())||Ee.isDescriptionPropertyName(n.name())||Ee.isDisplayNamePropertyName(n.name())||Ee.isUsagePropertyName(n.name())||Ee.isExampleProperty(n.property())||Ee.isMasterRefProperty(n.property())||Ee.isAnnotationsProperty(n.property())||Ee.isUsesProperty(n.property())||t.accept(Q(De.INVALID_OVERRIDE_IN_OVERLAY,{propName:n.name()},n)))})},e}(),ut=function(){function e(){}return e.prototype.validate=function(e,t){var n=this,r=new st;r.validate(e,t),e.directChildren().forEach(function(e){e.isElement()&&n.validate(e.asElement(),t)})},e}(),lt=function(){function e(){}return e.prototype.val=function(e,t,n){var r=this;if(e.kind()==he.Kind.MAP||e.kind()==he.Kind.MAPPING){var i={};e.children().forEach(function(e){var r=e.key();if(r){if(i.hasOwnProperty(r)){var a=Q(De.KEYS_SHOULD_BE_UNIQUE,{},n,!1);e.unit()==n.lowLevel().unit()&&(a.start=e.keyStart(),a.end=e.keyEnd()),t.accept(a)}i[r]=1}})}e.children().forEach(function(e){r.val(e,t,n)})},e.prototype.validate=function(e,t){this.val(e.lowLevel(),t,e)},e}(),pt=function(){function e(){}return e.prototype.validate=function(e,t){this.validateChildElements(e,t);var n=e.lowLevel().children(),r=de.groupBy(n.filter(function(e){return null!=e.key()}),function(e){return e.key()});this.validateChildAttributes(e,r,t),this.validateUnrecognizedLowLevelChildren(e,r,t)},e.prototype.validateChildElements=function(e,t){var n={},r=e.directChildren().filter(function(e){return e.isElement()});r.forEach(function(e){var t=e;if(!t._computed&&t.name()){var r=t.name()+t.property().nameId();n.hasOwnProperty(r)?t.isNamePatch()||n[r].push(t):n[r]=[t]}}),Object.keys(n).forEach(function(e){var r=n[e];!r||r.length<2||r.forEach(function(e){var n=D(e),r={name:e.name()},i=De.ALREADY_EXISTS_IN_CONTEXT;n&&(r.capitalized=Pe.upperCaseFirst(n),i=De.ALREADY_EXISTS);var a=Q(i,r,e);t.accept(a)})})},e.prototype.validateChildAttributes=function(e,t,n){var r=this.getHighLevelAttributes(e),i=de.groupBy(r,function(e){return e.name()}),a=this.allowsAnyAndHasRequireds(e);Object.keys(i).forEach(function(r){if(!(i[r].length<2)){var o=i[r][0].isUnknown(),s=!o&&i[r][0].property().isMultiValue();s&&(e.definition().isAssignableFrom(Ae.Universe08.SecuritySchemeSettings.name)||e.definition().isAssignableFrom(Ae.Universe10.SecuritySchemeSettings.name))&&(s=t[r]&&1===t[r].length),(o&&a||!s||s&&null!=t[r]&&t[r].length>1)&&i[r].forEach(function(e){var t={propName:e.property()?e.property().nameId():e.name()},r=De.PROPERTY_USED,i=D(e.parent());i&&(t.parent=Pe.upperCaseFirst(i),r=De.PARENT_PROPERTY_USED);var a=Q(r,t,e);n.accept(a)})}})},e.prototype.validateUnrecognizedLowLevelChildren=function(e,t,n){var r=e.directChildren(),i=de.groupBy(r,function(e){return e.name()});Object.keys(t).forEach(function(r){if(r&&t[r].length>1&&!i[r]){if(e.definition().isAssignableFrom(Ae.Universe10.ObjectTypeDeclaration.name))return;var a={propName:r},o=De.PROPERTY_USED,s=D(e);s&&(a.parent=Pe.upperCaseFirst(s),o=De.PARENT_PROPERTY_USED),t[r].forEach(function(t){var r=ee(o,a,t,e);r.start=t.keyStart(),r.end=t.keyEnd(),n.accept(r)})}})},e.prototype.filterMultiValueAnnotations=function(e,t,n){this.getHighLevelAttributes(e);Object.keys(t).forEach(function(e){"("!==e.charAt(0)||t[e].length<2})},e.prototype.getHighLevelAttributes=function(e){var t=this.allowsAnyAndHasRequireds(e);return e.directChildren().filter(function(e){return e.isAttr()||t})},e.prototype.allowsAnyAndHasRequireds=function(e){var t=e.definition().requiredProperties(),n=t&&t.length>0,r=e.definition().getAdapter(Se.RAMLService),i=r&&r.getAllowAny(),a=i&&n;return a},e}(),ct=function(){function e(){}return e.prototype.validate=function(e,t){var n=this.isStrict(e);if(n||Ue.validateNotStrictExamples){var r=this.parseObject(e,t,n);if(r){var i=this.aquireSchema(e);if(i){var a=r;"object"==typeof a&&(a=e.value()),i.validate(a,t,n)}}}},e.prototype.isExampleNode=function(e){return this.isSingleExampleNode(e)||this.isExampleNodeInMultipleDecl(e)},e.prototype.isSingleExampleNode=function(e){return e.name()==Ae.Universe10.TypeDeclaration.properties.example.name},e.prototype.isExampleNodeInMultipleDecl=function(e){var t=e.parent();return t?Ee.isExampleSpecType(t.definition()):!1},e.prototype.findParentSchemaOrTypeAttribute=function(e){var t=e.parent().attr(Ae.Universe10.TypeDeclaration.properties.schema.name);return t?t:(t=e.parent().attr(Ae.Universe10.TypeDeclaration.properties.type.name))?t:e.parent()?(t=e.parent().parent().attr(Ae.Universe10.TypeDeclaration.properties.schema.name))?t:(t=e.parent().parent().attr(Ae.Universe10.TypeDeclaration.properties.type.name),t?t:null):null},e.prototype.aquireSchema=function(e){var t=e.parent().definition().isAssignableFrom(Ae.Universe10.TypeDeclaration.name);if(this.isExampleNode(e)){var n=e;if(this.isExampleNodeInMultipleDecl(e)&&(n=e.parent()),n.parent()&&(n.parent().definition().isAssignableFrom(Ae.Universe10.TypeDeclaration.name)&&null===n.parent().parent()?t=!1:n.parent().property().nameId()==Ae.Universe10.LibraryBase.properties.types.name&&(t=!1),n.parent().parent())){var r=n.parent().parent().definition().key();(r==Ae.Universe08.Method||r==Ae.Universe10.Method)&&(n.parent().property().nameId()==Ae.Universe10.MethodBase.properties.queryParameters.name||(t=!0)),(r==Ae.Universe08.Response||r==Ae.Universe10.Response)&&(t=!0)}}if(e.parent().definition().key()==Ae.Universe08.BodyLike||t){var i=this.findParentSchemaOrTypeAttribute(e);if(i){var a=i.value();if(ye.StructuredValue.isInstance(a))return null;var o=(""+a).trim(),s=null;if("{"==o.charAt(0))try{s=Ce.getJSONSchema(o,Y(i.lowLevel()))}catch(u){return null}if("<"==o.charAt(0))try{s=Ce.getXMLSchema(o)}catch(u){return null}if(s)return{validate:function(t,n,r){try{if(t.__$validated)return;if(s instanceof Error)return void n.accept(Q(De.INVALID_VALUE_SCHEMA,{iValue:s.message},e,!r));s.validate(t)}catch(a){var o="Cannot assign to read only property '__$validated' of ";if(a.message&&0==a.message.indexOf(o)){var u=a.message.substr(o.length,a.message.length-o.length);return void n.accept(Q(De.INVALID_JSON_SCHEMA,{propName:u},i,!r))}if("Object.keys called on non-object"==a.message)return;return He.isInstance(a)?void(ae(e,n,a.messageEntry.code,!r)||n.accept(J(a,e,!r))):void n.accept(Q(De.EXAMPLE_SCHEMA_FAILURE,{msg:a.message},e,!r))}}};if(o.length>0){var l=e.parent(),p=l&&l.parent(),c=l&&l.definition()&&l.definition().isAssignableFrom(Ae.Universe10.ObjectTypeDeclaration.name)&&l;if(c=c||p&&p.definition()&&p.definition().isAssignableFrom(Ae.Universe10.ObjectTypeDeclaration.name)&&p)return this.typeValidator(c,e)}}}return this.getSchemaFromModel(e)},e.prototype.getSchemaFromModel=function(e){var t=e.parent();return this.typeValidator(t,e)},e.prototype.typeValidator=function(e,t){var n={validate:function(n,r,i){var a=e.parsedType();if(a&&!a.isUnknown()){"number"==typeof n&&a.isString()&&(n=""+n),"boolean"==typeof n&&a.isString()&&(n=""+n),a.getExtra("repeat")&&(n=[n]);var o=a.validate(n,!1);o.isOk()||o.getErrors().forEach(function(e){return r.accept(Z(e.getCode(),e.getMessage(),t,!i,e.getInternalRange()))})}}};return n},e.prototype.toObject=function(e,t,n){var r=t.lowLevel().dumpToObject(!0);return this.testDublication(e,t.lowLevel(),n),r.example?r.example:r.content?r.content:void 0},e.prototype.testDublication=function(e,t,n){var r=this,i={};t.children().forEach(function(t){t.key()&&(i[t.key()]&&n.accept(Q(De.KEYS_SHOULD_BE_UNIQUE,{},new ye.BasicASTNode(t,e.parent()))),i[t.key()]=t),r.testDublication(e,t,n)})},e.prototype.parseObject=function(e,t,n){var r=null,i=e.value(),a=G(e);if(ye.StructuredValue.isInstance(i))r=this.toObject(e,i,t);else if(a){if(H(a))try{me.rt.getSchemaUtils().tryParseJSON(i,!0),r=JSON.parse(i)}catch(o){return void(He.isInstance(o)?ae(e,t,o.messageEntry.code,!n)||t.accept(J(o,e,!n)):t.accept(Q(De.CAN_NOT_PARSE_JSON,{msg:o.message},e,!n)))}if($(a))try{r=Le.parseXML(i)}catch(o){return void t.accept(Q(De.CAN_NOT_PARSE_XML,{msg:o.message},e,!n))}}else try{if(!(i&&i.length>0)||"["!=i.trim().charAt(0)&&"{"!=i.trim().charAt(0)&&"<"!=i.trim().charAt(0)){if("true"==i)return!0;if("false"==i)return!1;var s=parseFloat(i);return isNaN(i)?i:s}r=JSON.parse(i)}catch(o){if(0!=i.trim().indexOf("<"))return i;try{r=Le.parseXML(i)}catch(o){return void t.accept(Q(De.CAN_NOT_PARSE_XML,{msg:o.message},e,!n))}}return r},e.prototype.isStrict=function(e){if(Ee.isDefaultValue(e.property()))return!0;if(Ee.isExampleProperty(e.property())&&"RAML08"==e.parent().definition().universe().version())return!1;var t=!1,n=e.parent().attr("strict");return n&&"true"==n.value()&&(t=!0),t},e}();t.ExampleAndDefaultValueValidator=ct;var ft=function(e,t,n){var r=Pe.sentence(e);return t||(r=Pe.ucFirst(r)),n&&(r=Oe.plural(r)),r},dt=function(){function e(){}return e.prototype.validate=function(e,t){if(e.isAttr()){if(!e.optional())return;var n=e,r=n.property();if(r.isMultiValue()||r.range().isArray())return;if(!r.isFromParentKey()){var i=c(n.parent());if(i&&r.isValueProperty()){var a=ft(i,!0,!0),o=Q(De.OPTIONAL_SCLARAR_PROPERTIES_10,{templateNamePlural:a,propName:n.name()},n,!1);t.accept(o)}}}else if(e.isElement()){var s=e,r=s.property(),u=s.allowsQuestion();if(!u){var l=r?ft(r.nameId(),!0,!0):"API root";s.optionalProperties().forEach(function(n){s.children().forEach(function(n){var r={propName:l,oPropName:n.lowLevel().key()},i=Q(De.OPTIONAL_PROPERTIES_10,r,e,!1);t.accept(i)})})}var p=e.asElement().definition();if(e.optional()&&"RAML10"==p.universe().version()){var r=e.property(),f=Ee.isQueryParametersProperty(r)||Ee.isUriParametersProperty(r)||Ee.isHeadersProperty(r);if(!(Ee.isMethodType(p)||Ee.isTypeDeclarationType(p)&&f)){var o=Q(De.ONLY_METHODS_CAN_BE_OPTIONAL,{},e,!1);t.accept(o)}}}},e}(),ht=function(){function e(){}return e.prototype.validate=function(e,t){var n=e.definition(),r=Ae.Universe10.Api.properties.baseUri.name,i=Ae.Universe10.Api.properties.baseUriParameters.name,a=Ae.Universe10.Resource.properties.relativeUri.name,o=Ae.Universe10.ResourceBase.properties.uriParameters.name;if(Ee.isApiSibling(n))this.inspectParameters(e,t,r,i);else if(Ee.isResourceType(n)){var s=e.root();this.inspectParameters(e,t,r,i,s),this.inspectParameters(e,t,a,o)}else if(Ee.isResourceTypeType(n)){var s=e.root();this.inspectParameters(e,t,r,i,s)}},e.prototype.inspectParameters=function(e,t,n,r,i){i=i||e;var a="",o=i.attr(n);o&&(a=o.value(),a&&"string"==typeof a||(a=""));var s=e.elementsOfKind(r);s.forEach(function(n){var i=n.attr(Ae.Universe10.TypeDeclaration.properties.name.name);if(i){var o=i.value();if(null!=o&&a.indexOf("{"+o+"}")<0){if(Ee.isResourceTypeType(e.definition())&&o.indexOf("<<")>=0)return;var s=Pe.upperCaseFirst(Oe.singular(Pe.sentence(r))),u=Q(De.PROPERTY_UNUSED,{propName:s},n,!0);t.accept(u)}}})},e}(),mt=function(){function e(){this.nameProperty=Ae.Universe10.ResourceType.properties.name.name}return e.prototype.validate=function(e,t){var n=e.definition();if(Ee.isLibraryBaseSibling(n)||Ee.isApiType(n)){var r=(Ae.Universe10.LibraryBase.properties.resourceTypes.name,Ae.Universe10.ResourceBase.properties.type.name),i=(Ae.Universe10.LibraryBase.properties.traits.name,Ae.Universe10.MethodBase.properties.is.name),a=Ne.globalDeclarations(e).filter(function(e){return Ee.isResourceTypeType(e.definition())}),o=Ne.globalDeclarations(e).filter(function(e){return Ee.isTraitType(e.definition())});this.checkCycles(a,r,t),this.checkCycles(o,i,t)}},e.prototype.checkCycles=function(e,t,n){var r=this,i={};e.forEach(function(e){var t=r.templateName(e);i[t]=e});var a={};e.forEach(function(e){var o=r.templateName(e);a[o]||r.findCyclesInDefinition(e,t,i).forEach(function(t){t.forEach(function(e){return a[e]=!0}),t=t.reverse();var r=ft(e.definition().nameId()),i=t.join(" -> "),o={typeName:r,cycle:i},s=Q(De.CYCLE_IN_DEFINITION,o,e,!1);n.accept(s)})})},e.prototype.templateName=function(e){var t=e.attr(this.nameProperty);return t?t.value():null},e.prototype.findCyclesInDefinition=function(e,t,n,r){void 0===r&&(r={});var i=this.templateName(e);if(r[i])return[[i]];var a={};Object.keys(r).forEach(function(e){return a[e]=r[e]}),a[i]=!0;for(var o=[],s=e.attributes(t),u=0;u<s.length;u++){var l=s[u],p=l.value();if(p){var c="string"==typeof p||"number"==typeof p||"boolean"==typeof p?p+"":p.valueName(),f=n[c];if(null!=f){var d=this.findCyclesInDefinition(f,t,n,a);d.forEach(function(e){return o.push(e)})}}}return o.forEach(function(e){return e.push(i)}),o},e}();t.isJson=H,t.isXML=$,t.getMediaType=G;var yt=/^[ ]*/,vt=function(e,t,n,r,i,a,o,s,u){void 0===u&&(u=!1);var l=o?o:e.lowLevel(),p=l.unit()&&l.unit().contents(),c=p&&p.length,f=l.start(),d=l.end(),h=l.actual(),m=h&&h.value,y=m&&m.rawValue,v=l.valueKind();v==he.Kind.ANCHOR_REF&&(v=l.anchorValueKind());var g=v;if(v==he.Kind.INCLUDE_REF&&(v=e.lowLevel().resolvedValueKind()),s&&(v==he.Kind.SCALAR||u)){if(g==he.Kind.INCLUDE_REF){var A=l.unit().resolve(l.includePath());if(A){var E=A.lineMapper(),T=E.toPosition(s.start.line,s.start.column),S=E.toPosition(s.end.line,s.end.column),b={code:t,isWarning:n,message:r,node:null,start:T.position,end:S.position,path:A.path(),extras:[],unit:A},_=vt(e,t,n,r,i,a,o);return b.extras.push(_),b}}var N=l.unit().lineMapper(),w=l.valueStart();0>w&&(w=l.start());var R=N.position(w),I=void 0,M=void 0,C=R.line+s.start.line,L=R.line+s.end.line;if(y&&"string"==typeof y&&"|"==y.charAt(0)){var P=y.indexOf("\n")+1,O=y.indexOf("\n",P);0>O&&(O=y.length);var D=yt.exec(y.substring(P,O))[0].length;I=D+s.start.column,M=D+s.end.column,C++,L++}else I=R.column+s.start.column,M=R.column+s.end.column,m&&(m.singleQuoted||m.doubleQuoted)&&(I++,M++);var x=N.toPosition(C,I),U=N.toPosition(L,M);x&&U&&(f=x.position,d=U.position)}else{if(c&&d>c&&(d=c-1),l.key()&&l.keyStart()){var k=l.keyStart();k>0&&(f=k);var F=l.keyEnd();F>0&&(d=F)}if(f>d&&(d=f+1,e.isElement())){var B=e.definition();Ee.isApiType(B)&&(f=0==c?0:c-1,d=f)}if(a&&!a.getAdapter(Se.RAMLPropertyService).isMerged()&&null==e.parent()){var K=de.find(l.children(),function(e){return e.key()==a.nameId()});if(K){var k=K.keyStart(),F=K.keyEnd();k>0&&F>k&&(f=k,d=F)}}}return{code:t,isWarning:n,message:r,node:e,start:f,end:d,path:i?l.unit()?l.unit().path():"":null,extras:[],unit:e?l.unit():null}},gt=function(e,t,n,r,i,a,o){var s=e.unit()&&e.unit().contents(),u=s&&s.length,l=e.start(),p=e.end();if(u&&p>=u&&(p=u-1),e.key()&&e.keyStart()){var c=e.keyStart();c>0&&(l=c);var f=e.keyEnd();f>0&&(p=f)}return{code:n,isWarning:r,message:i,node:t,start:l,end:p,path:a?e.unit()?e.unit().path():"":null,extras:[],unit:e?e.unit():null}};t.toIssue=z,t.createIssue1=Q,t.createIssue=Z,t.createLLIssue=te,t.validateResponseString=ne}).call(t,n(64))},function(e,t,n){"use strict";function r(e,t){if(void 0===t&&(t=0),t>20)return[];try{var n=[],i=e.leftType();i&&n.push(i);var a=e.rightType();if(a)if(a.hasUnionInHierarchy()){var o=r(a.unionInHierarchy(),t+1);n=n.concat(o)}else n.push(a);return n}finally{}}function i(e){var t=e.definition();if(!t||!N.isTypeDeclarationDescendant(t))return!1;var n=e.lowLevel();if(n.valueKind()!==g.Kind.SEQ)return!1;var r=n.children();if(null==r)return!1;for(var i=0,a=r;i<a.length;i++){var o=a[i];if(o.kind()!==g.Kind.SCALAR)return!1}return!0}function a(e,t){if(!t)return[e.definition().universe().type(_.Universe10.StringTypeDeclaration.name)];var n=e.parsedType(),r=[];return n.isString()?r.push(e.definition().universe().type(_.Universe10.StringTypeDeclaration.name)):n.isNumber()?(n.isInteger()&&r.push(e.definition().universe().type(_.Universe10.IntegerTypeDeclaration.name)),r.push(e.definition().universe().type(_.Universe10.NumberTypeDeclaration.name))):n.isBoolean()?r.push(e.definition().universe().type(_.Universe10.BooleanTypeDeclaration.name)):n.isObject()?r.push(e.definition().universe().type(_.Universe10.ObjectTypeDeclaration.name)):n.isArray()?r.push(e.definition().universe().type(_.Universe10.ArrayTypeDeclaration.name)):n.isFile()?r.push(e.definition().universe().type(_.Universe10.FileTypeDeclaration.name)):n.isDateTime()?r.push(e.definition().universe().type(_.Universe10.DateTimeTypeDeclaration.name)):n.isDateTimeOnly()?r.push(e.definition().universe().type(_.Universe10.DateTimeOnlyTypeDeclaration.name)):n.isDateOnly()?r.push(e.definition().universe().type(_.Universe10.DateOnlyTypeDeclaration.name)):n.isTimeOnly()&&r.push(e.definition().universe().type(_.Universe10.TimeOnlyTypeDeclaration.name)),n.isUnion()&&r.push(e.definition().universe().type(_.Universe10.UnionTypeDeclaration.name)),0==r.length&&r.push(e.definition().universe().type(_.Universe10.TypeDeclaration.name)),r}function o(e){return function(t){var n=e.type(t);if(!n){new y.UserDefinedClass("",e,null,"","")}return n}}function s(e,t){var n=e.asElement();if(null==n)return null;var r=n.elementsOfKind(_.Universe10.TypeDeclaration.properties.facets.name);return null==r||0==r.length?null:A.find(r,function(e){return t==e.attrValue(_.Universe10.TypeDeclaration.properties.name.name)})}function u(e){var t=e.owner();if(null==t)return null;var n=e.facetName();if(!n)return null;var r=t.getExtra(R.SOURCE_EXTRA);return null==r?null:E.isSourceProvider(r)||v.isParseResult(r)?{getSource:function(){if(E.isSourceProvider(r)){var e=r.getSource();if(e&&v.isParseResult(e))return s(e,n)}else if(v.isParseResult(r))return s(r,n);return null},getClassIdentifier:function(){return[y.SourceProvider.CLASS_IDENTIFIER]}}:null}function l(e,t,n){for(var r,i=0,a=e;i<a.length;i++){var o=a[i];if(null!=o){var s=t.parsedType();r||(r=new y.NodeClass(t.name(),t.definition().universe(),"",""),r.addAdapter(s)),r._superTypes.push(o),p(r,t,n)}}return r}function p(e,t,n){var r=t.parsedType();r.allFacets().forEach(function(r){if(r.kind()==y.tsInterfaces.MetaInformationKind.FacetDeclaration){var i=u(r),a=null;null!=i?(a=new y.UserDefinedProp(r.facetName(),null),a.setSourceProvider(i)):a=new y.Property(r.facetName(),"");var s=null;s=n?n.definition().universe():t.definition().universe(),a.withRange(s.type("StringType")),a.withDomain(e),a.withGroupName(r.facetName()),a.withRequired(!1),r.value(),R.setPropertyConstructor(function(e){var t=new y.Property(e,"");return t.unmerge(),t}),a.withRange(R.toNominal(r.value(),o(t.definition().universe())))}})}function c(e,t,n){var r=n.attr("type");if(r){var i=a(n,r.value()),o=l(i,n,t);return o&&0==o.superTypes().length&&o._superTypes.push(n.definition().universe().type(_.Universe10.TypeDeclaration.name)),o}var s=_.Universe10.ObjectTypeDeclaration.properties.properties.name;if(e){if("body"==e.nameId()||A.find(n.lowLevel().children(),function(e){return e.key()===s}))return n.definition().universe().type(_.Universe10.ObjectTypeDeclaration.name)}else if(!t&&n.lowLevel()&&A.find(n.lowLevel().children(),function(e){return e.key()===s}))return n.definition().universe().type(_.Universe10.ObjectTypeDeclaration.name);return n.definition().universe().type(_.Universe10.StringTypeDeclaration.name)}function f(e){try{var t=e.definition().nameId(),n=t===_.Universe10.Api.name||t===_.Universe08.Api.name;if(!n&&!e.property()&&!e.parent()&&e.definition().nameId()===T.getFragmentDefenitionName(e)){if(e.property()&&e.property().nameId()===_.Universe10.LibraryBase.properties.annotationTypes.name)return d(null,null,e);var r=null,i=e.definition().allSubTypes();return i.forEach(function(t){!r&&m(t,e,null)&&(r=t)}),r}}catch(a){}return d(e.property(),e.parent(),e)}function d(e,t,n){var r=n.lowLevel();e&&e.nameId()==_.Universe10.FragmentDeclaration.properties.uses.name&&e.range().nameId()==_.Universe10.Library.name;var i=e?e.range().nameId():n.definition().nameId();if(r._node&&r._node.descriminate)return null;r._node&&(r._node.descriminate=1);try{if(i==_.Universe10.TypeDeclaration.name){var a=c(e,t,n);if(e||!e&&!t&&n.lowLevel()){if(e&&null!=a&&(e.nameId()==_.Universe10.MethodBase.properties.body.name||e.nameId()==_.Universe10.ArrayTypeDeclaration.properties.items.name||e.nameId()==_.Universe10.Response.properties.headers.name||e.nameId()==_.Universe10.MethodBase.properties.queryParameters.name)){var o=new y.UserDefinedClass(n.lowLevel().key(),a.universe(),n,n.lowLevel().unit()?n.lowLevel().unit().path():"","");return o._superTypes.push(a),o}if(null!=a&&N.isAnnotationTypeType(n.definition())){var s=h(a),o=new y.UserDefinedClass(n.lowLevel().key(),a.universe(),n,n.lowLevel().unit()?n.lowLevel().unit().path():"","");return o._superTypes.push(s),o._superTypes.push(a),o}}if(a)return a}var u=null;if(e&&t){var l=b.findAllSubTypes(e,t);return l.length>0&&l.forEach(function(e){u||m(e,n,u)&&(u=e)}),u}}finally{r._node&&delete r._node.descriminate}}function h(e){for(var t,n=[e].concat(e.allSuperTypes()),r=0;r<n.length;r++){var i=n[r];if(!i.isUserDefined()){if(i.isUnion()){var a=i,o=a.leftType(),s=(a.leftType(),h(o)),u=h(o);if(s.isAssignableFrom(u.nameId())&&(null==t||s.isAssignableFrom(t.nameId()))){t=s;continue}if(u.isAssignableFrom(s.nameId())&&(null==t||u.isAssignableFrom(t.nameId()))){t=u;continue}t=e.universe().type(_.Universe10.UnionTypeDeclaration.name);break}if(i.isArray()){t=e.universe().type(_.Universe10.ArrayTypeDeclaration.name);break}for(var l=i.subTypes(),p=0;p<l.length;p++){var c=l[p];c.isAssignableFrom(_.Universe10.TypeDeclaration.name)&&(null==t?t=c:c.isAssignableFrom(t.nameId())&&(t=c))}}}return null!=t?t:e.universe().type(_.Universe10.TypeDeclaration.name)}function m(e,t,n){if(t.isAttr()||t.isUnknown())return!1;var r=t,i=A.find(e.superTypes(),function(t){var n=A.find(t.allProperties(),function(e){return e.isDescriminator()});if(n){var i=r.attr(n.nameId());if(i&&i.value()==e.nameId())return!0}return!1});if(i)return!0;if(0==e.valueRequirements().length)return!1;var a=!0;return e.valueRequirements().forEach(function(t){var i=r.attr(t.name);if(i)if(i.value()==t.value);else if(e.getAdapter(w.RAMLService).getConsumesRefs()){var o=i.value(),s=[];e.superTypes().forEach(function(e){return e.allSubTypes().forEach(function(e){s.push(e)})});var u=[];A.unique(s).forEach(function(e){u.push(e.nameId()),e.valueRequirements().forEach(function(e){u.push(e.value)}),e.getAdapter(w.RAMLService).getAliases().forEach(function(e){return u.push(e)})}),A.find(u,function(e){return e==o})&&(a=!1)}else a=!1;else{var l=e.getAdapter(w.RAMLService).getDefining(),p=!1;if(l.forEach(function(e){r.lowLevel().children().forEach(function(t){t.key()==e&&(p=!0)})}),p)return void(a=!0);if(n)a=!1;else{var c=e.property(t.name);c&&c.defaultValue()==t.value||(a=!1)}}}),a}var y=n(39),v=n(9),g=n(77),A=n(70),E=y,T=n(16),S=n(44),b=n(13),_=n(18),N=n(14),w=y,R=y.rt,I=n(83),M=n(68),C=function(){function e(e){this._props=e,this.parentValue=A.find(e,function(e){return e.isFromParentValue()}),this.parentKey=A.find(e,function(e){return e.isFromParentKey()}),this.canBeValue=A.find(e,function(e){return e.canBeValue()})}return e.prototype.add=function(e){this._props.push(e)},e.prototype.match=function(e){var t=this,n=null,r="";return this._props.forEach(function(i){i.getAdapter(w.RAMLPropertyService).isSystem()||i!=t.parentValue&&i!=t.parentKey&&i.matchKey(e)&&(null!=i.keyPrefix()?i.keyPrefix().length>=r.length&&(r=i.keyPrefix(),n=i):(n=i,r=i.nameId()))}),n},e}(),L=0,P=function(){function e(){this.shouldDescriminate=!1}return e.prototype.process=function(e,t){var n=this,a=e.lowLevel(),o=e;o._mergedChildren=null;var s=a._node?a._node:a;try{if(s.currentChildren)return s.currentChildren;if(!e.definition())return;if(null==e.parent()&&!this.shouldDescriminate){this.shouldDescriminate=!0;try{var u=this.process(e,t),l=e;l._children=u,l._mergedChildren=null;var p=f(e);p&&l.patchType(p);var u=this.process(e,t);l._children=u,l._mergedChildren=null}finally{this.shouldDescriminate=!1}}if(e.definition().hasUnionInHierarchy()&&e.parent()&&e.property().nameId()==_.Universe10.LibraryBase.properties.annotations.name){var c=r(e.definition().unionInHierarchy()),d=null,h=null,m=null,y=1e3,v=e;if(c.forEach(function(r){if(!d&&!r.hasUnionInHierarchy()){v.patchType(r);if(0==L){L++;try{for(var i=n.process(e,t),a=0,o=0;o<i.length;o++)i[o].isUnknown()&&a++;0==a&&(d=i),y>a&&(y=a,h=i,m=r)}finally{L--}}}}),d)return v.patchType(m),d;h&&v.patchType(m)}var A=new C(e.definition().allProperties());if(null==e.parent()||e.lowLevel().includePath()){var E=e.definition().universe();"RAML10"==E.version()&&(e.definition().property("uses")||E.type("FragmentDeclaration").allProperties().forEach(function(e){return A.add(e)}))}var S=e,b=S._allowQuestion||e.definition().getAdapter(w.RAMLService).getAllowQuestion(),R=[];if(A.parentKey){if(e.lowLevel().key()){var M=new T.ASTPropImpl(e.lowLevel(),e,A.parentKey.range(),A.parentKey,!0);R.push(M);var P=e.property()&&N.isBodyProperty(e.property())&&e.lowLevel().key()==e.property().nameId();if(P){O(S);M.overrideValue(S._computedKey)}}if(e.lowLevel().valueKind()===g.Kind.SEQ&&!i(e)){var D=new T.BasicASTNode(e.lowLevel(),S);return D.errorMessage={entry:I.DEFINITION_SHOULD_BE_A_MAP,parameters:{typeName:e.definition().nameId()}},R.push(D),R}}if(null!=e.lowLevel().value(!0))if(A.parentValue)R.push(new T.ASTPropImpl(e.lowLevel(),e,A.parentValue.range(),A.parentValue));else if(A.canBeValue){var x=e.lowLevel().value();null==x&&(x=e.lowLevel().value(!0)),"string"==typeof x&&x.trim().length>0&&R.push(new T.ASTPropImpl(e.lowLevel(),e,A.canBeValue.range(),A.canBeValue))}if(S._children=R,S._mergedChildren=null,S.definition().getAdapter(w.RAMLService).isUserDefined())R=this.processChildren(t,S,R,b,A);else if(S.definition().key()==_.Universe08.Api||S.definition().key()==_.Universe10.Api){var U=t.filter(function(e){
return"uses"==e.key()});R=this.processChildren(U,S,R,b,A);var k=t.filter(function(e){return"types"==e.key()});R=this.processChildren(k,S,R,b,A);var F=t.filter(function(e){return"types"!=e.key()&&"uses"!=e.key()});R=this.processChildren(F,S,R,b,A)}else R=this.processChildren(t,S,R,b,A);return S._children=R,S._mergedChildren=null,R}finally{}},e.prototype.isTypeDeclarationShortcut=function(e,t){var n=N.isTypeOrSchemaProperty(t),r=e.definition()&&N.isTypeDeclarationTypeOrDescendant(e.definition());return r&&n&&e.lowLevel()&&e.lowLevel().valueKind()===g.Kind.SEQ?!0:!1},e.prototype.processChildren=function(e,t,n,r,i){var a=this,o=_.Universe10.TypeDeclaration.name,s=_.Universe10.TypeDeclaration.properties.type.name,u=_.Universe10.ArrayTypeDeclaration.properties.items.name;if(t.definition()&&t.definition().isAssignableFrom(o)&&t.lowLevel()&&i.canBeValue&&(i.canBeValue.nameId()===s||i.canBeValue.nameId()===u)&&t.lowLevel()._node&&t.lowLevel()._node.value&&t.lowLevel()._node.value.kind===g.Kind.SEQ)return e.forEach(function(e){var r=new T.ASTPropImpl(e,t,i.canBeValue.range(),i.canBeValue);n.push(r)}),n;var l=t.root().lowLevel().unit();e.forEach(function(e){if(i.canBeValue&&a.isTypeDeclarationShortcut(t,i.canBeValue))return void n.push(new T.ASTPropImpl(e,t,i.canBeValue.range(),i.canBeValue));var o=e.key(),s=null!=o?i.match(o):null;if(null!=s){var u=s.range();if(s.isAnnotation()&&"annotations"!=o){var p=new T.ASTPropImpl(e,t,u,s);return void n.push(p)}var c=!1,f=s.isMultiValue();u.isArray()?(f=!0,u=u.array().componentType(),c=!0):u.hasArrayInHierarchy()&&(f=!0,c=!0);var h,m=!1;if(t.reuseMode()&&e.valueKind()!=g.Kind.SEQ){var v=t.reusedNode();if(v){var E=[e],b=t.lowLevel();!s.isMerged()&&f&&(h=[],E=e.children(),b=e);for(var R=0,C=E;R<C.length;R++){var L=C[R],P=L.containingUnit();if(P&&P.absolutePath()!=l.absolutePath()){var D=A.find(v.children(),function(e){return e.lowLevel().key()==L.key()});if(D&&D.lowLevel().unit().absolutePath()==P.absolutePath()){m=!0,n.push(D),D.setReused(!0),D.setParent(t),D.isElement()&&D.resetRuntimeTypes(),t.isExpanded()&&(b.replaceChild(L,D.lowLevel()),S.LowLevelCompositeNode.isInstance(D.lowLevel())&&D.lowLevel().setTransformer(b.transformer()));continue}}h&&h.push(L)}}}if(!h){if(m)return;h=e.children()}if(u.hasValueTypeInHierarchy()){var x=h,U=e.valueKind()==g.Kind.SEQ;if((U&&x.length>0||x.length>1)&&f)if(x.length>1&&N.isTypeDeclarationDescendant(t.definition())&&(N.isTypeOrSchemaProperty(s)||N.isItemsProperty(s))&&e.valueKind()!=g.Kind.SEQ){var p=new T.ASTPropImpl(e,t,u,s);n.push(p),t.setComputed(s.nameId(),p)}else{var k=[];x.forEach(function(e){var r=new T.ASTPropImpl(e,t,u,s);n.push(r),k.push(e.value())}),s.isInherited()&&t.setComputed(s.nameId(),k)}else{s.isInherited()&&t.setComputed(s.nameId(),e.value());var F=new T.ASTPropImpl(e,t,u,s);if(U||e.valueKind()==g.Kind.MAP){var B=s.range().nameId();s.getAdapter(w.RAMLPropertyService).isExampleProperty()||("StringType"==B&&(B="string"),"NumberType"==B&&(B="number"),"BooleanType"==B&&(B="boolean"),("string"==B||"number"==B||"boolean"==B)&&(e.isAnnotatedScalar()||(F.errorMessage={entry:I.INVALID_PROPERTY_RANGE,parameters:{propName:s.groupName(),range:B}},0==h.length&&"enum"==s.groupName()&&(F.errorMessage={entry:I.ENUM_IS_EMPTY,parameters:{}},e.valueKind()==g.Kind.MAP&&(F.errorMessage={entry:I.ENUM_MUST_BE_AN_ARRAY,parameters:{}})))))}n.push(F)}return}var K=[];if(t._children=n,t._mergedChildren=null,null!=e.value()&&("string"==typeof e.value()||"boolean"==typeof e.value()||"number"==typeof e.value())&&(""+e.value()).trim().length>0){var V=s.range();if(!V.allProperties().some(function(e){var t=e;return t?t.canBeValue()&&t.isFromParentValue():!1})){var j=new T.BasicASTNode(e,t);j.getLowLevelEnd=function(){return-1},j.getLowLevelStart=function(){return-1},j.knownProperty=s,n.push(j)}}if(s.isMerged()){var W=new T.ASTNodeImpl(e,t,u,s);W._allowQuestion=r,K.push(W)}else if(f)if(s.getAdapter(w.RAMLPropertyService).isEmbedMap()){var q=h,Y=[],H=!1;if(q.forEach(function(e){e.kind()==g.Kind.INCLUDE_REF&&"RAML08"==t.universe().version()?e.children().forEach(function(e){var n=new T.ASTNodeImpl(e,t,u,s);n._allowQuestion=r,K.push(n),H=!0}):Y.push(e)}),q=Y,0==q.length){if(s.range().key()==_.Universe08.ResourceType&&!H){var $=new T.BasicASTNode(e,t);$.errorMessage={entry:I.PROPERTY_MUST_BE_A_MAP,parameters:{propName:s.nameId()}},n.push($)}if(e.valueKind()==g.Kind.SCALAR&&s.range().key()==_.Universe08.AbstractSecurityScheme){var $=new T.BasicASTNode(e.parent(),t);$.errorMessage={entry:I.PROPERTY_MUST_BE_A_MAP,parameters:{propName:s.nameId()}},n.push($)}}if(q.forEach(function(e){var i=e.children();if(e.key()||1!=i.length)if("RAML10"==t.universe().version()){var a=new T.ASTNodeImpl(e,t,u,s);a._allowQuestion=r,K.push(a)}else{var o=new T.BasicASTNode(e,t);n.push(o),e.key()&&(o.needSequence=!0)}else if("RAML10"!=t.universe().version()||t.parent()){var a=new T.ASTNodeImpl(i[0],t,u,s);a._allowQuestion=r,K.push(a)}}),"RAML10"==t.universe().version()&&e.valueKind()==g.Kind.SEQ){var j=new T.BasicASTNode(e,t);n.push(j),j.needMap=!0,j.knownProperty=s}}else{var G={},X=[];if(y.NodeClass.isInstance(u)){var z=u;z.getAdapter(w.RAMLService).getCanInherit().length>0&&z.getAdapter(w.RAMLService).getCanInherit().forEach(function(n){for(var i=t.computedValue(n),a=Array.isArray(i)?i:[i],o=0;o<a.length;o++){var l=a[o];if(l&&s.nameId()==_.Universe10.Response.properties.body.name){for(var p=A.find(h,function(e){return e.key()==l}),c=e;c.originalNode;)c=c.originalNode();var f=A.find(c.children(),function(e){try{return M.parse(e.key()),!0}catch(t){return!1}});if(!f){var m=t.parent().definition().key(),y=new T.ASTNodeImpl(e,t,u,s);y._computedKey=l,(m==_.Universe10.MethodBase||m==_.Universe08.MethodBase)&&y.setComputed("form","true");var v=O(t),g=d(s,t,y);g&&(v||g.setName(l),y.patchType(g));var E=y.children();y._children=E.filter(function(e){return!e.isUnknown()}),y._mergedChildren=null,y._allowQuestion=r,p||X.push(y),y.children().forEach(function(e){var t=e.property();if(t&&t.getAdapter(w.RAMLPropertyService).isKey()){var n=e;return void(n._computed=!0)}if(e.isElement()&&(e.property().getAdapter(w.RAMLPropertyService).isMerged()||(G[e.property().nameId()]=!0)),e.property().isAnnotation()){var n=e,r=n.value(),i="";i=T.StructuredValue.isInstance(r)?r.valueName():""+r,G["("+i+")"]=!0}else G[e.name()]=!0});var S=y.definition().allProperties();S.forEach(function(e){e.getAdapter(w.RAMLPropertyService).isKey()||e.getAdapter(w.RAMLPropertyService).isSystem()||y.lowLevel().children().some(function(t){return t.key()==e.nameId()})&&(G[e.nameId()]=!0)}),y._computed=!0}}}})}var J=[];if(0==h.length&&e.valueKind()==g.Kind.SEQ&&s.range().key()==_.Universe08.Parameter){var $=new T.BasicASTNode(e,t);$.errorMessage={entry:I.PROPERTY_MUST_BE_A_MAP,parameters:{propName:s.nameId()}},n.push($)}if("documentation"===s.nameId()&&e.resolvedValueKind()!==g.Kind.SEQ){if(!t.definition().getAdapter(w.RAMLService).isUserDefined()){var $=new T.BasicASTNode(e,t);$.errorMessage={entry:I.PROPERTY_MUST_BE_A_SEQUENCE,parameters:{propName:s.nameId()}},n.push($)}}else h.forEach(function(e){if(!G[e.key()])if(e.valueKind()==g.Kind.SEQ){var i="RAML10"==t.definition().universe().version();if(e.children().forEach(function(n){var a=new T.ASTNodeImpl(n,t,u,s);a._allowQuestion=r,a.setNamePatch(e.key()),i&&(a.invalidSequence=!0),J.push(a)}),0==e.children().length){var a=new T.BasicASTNode(e,t);s.range().key()==_.Universe08.Parameter?a.errorMessage={entry:I.NAMED_PARAMETER_NEEDS_TYPE,parameters:{}}:a.errorMessage={entry:I.NODE_SHOULD_HAVE_VALUE,parameters:{}},n.push(a)}}else{var o=new T.ASTNodeImpl(e,t,u,s),l=s.domain().key();"body"!=s.nameId()||l!=_.Universe08.MethodBase&&l!=_.Universe10.MethodBase||o.setComputed("form","true"),o._allowQuestion=r,J.push(o)}});J.length>0?J.forEach(function(e){return K.push(e)}):X.forEach(function(e){return K.push(e)})}else K.push(new T.ASTNodeImpl(e,t,u,s));t._children=t._children.concat(K),t._mergedChildren=null,n=n.concat(K),K.forEach(function(e){var n=d(s,t,e);n&&n!=e.definition()&&e.patchType(n),e._associatedDef=null,s.childRestrictions().forEach(function(t){e.setComputed(t.name,t.value)});e.definition()})}else S.LowLevelCompositeNode.isInstance(e)&&null==e.primaryNode()||n.push(new T.BasicASTNode(e,t))});var p=t.reusedNode();if(p&&t.lowLevel().valueKind()!=g.Kind.SEQ){var c={};p.elements().forEach(function(e){return c[e.property().nameId()+"_"+e.lowLevel().key()]=e}),p.attrs().forEach(function(e){return c[e.property().nameId()+"_"+e.lowLevel().key()]=e}),n.filter(function(e){return e.isElement()||e.isAttr()}).forEach(function(e){var n=c[e.property().nameId()+"_"+e.lowLevel().key()];n&&n!=e&&e.isElement()&&e.lowLevel().parent().valueKind()!=g.Kind.SEQ&&(e.setReusedNode(n),e.setReuseMode(t.reuseMode()))})}return n},e}();t.BasicNodeBuilder=P,t.doDescrimination=f;var O=function(e){for(var t=!1,n=e;n;){var r=n.definition();if(N.isTraitType(r)||N.isResourceTypeType(r)){t=!0;break}n=n.parent()}return t}},function(e,t,n){"use strict";function r(e){if(!w.BasicNodeImpl.isInstance(e))return null;var t=e,n=i(t.highLevel());if(!n)return null;var r=n.wrapperNode();return r.setAttributeDefaults(t.getDefaultsCalculator().isEnabled()),r}function i(e){if(null==e)return null;var t=e.definition();if(!t||!M.isApiSibling(t))return null;var n=(new P).expandTraitsAndResourceTypes(e);return n}function a(e){if(!e)return null;var t=s(e.highLevel());if(!t)return null;var n=t.wrapperNode();return n.setAttributeDefaults(e.getDefaultsCalculator().isEnabled()),n}function o(e){if(!e)return null;var t=u(e.highLevel());if(!t)return null;var n=t.wrapperNode();return n.setAttributeDefaults(e.getDefaultsCalculator().isEnabled()),n}function s(e){return(new O).expandLibraries(e)}function u(e){return(new O).expandLibrary(e)}function l(e,t,n){var r=A.fromUnit(e);if(!r)throw new Error("couldn't load api from "+e.absolutePath());if(!t||0==t.length)return r.asElement();for(var i=[],a=0;a<t.length;a++){var o=t[a],s=A.fromUnit(o);if(!s)throw new Error("couldn't load api from "+o.absolutePath());i.push(s)}var u=p(r.asElement(),i,n);return u}function p(e,t,n,r,i){void 0===r&&(r=null),void 0===i&&(i=!1);for(var a,o=e,s=0,u=t;s<u.length;s++){var l=u[s];i&&S.LowLevelProxyNode.isInstance(o.lowLevel())&&(a||(a=new P),a.expandHighLevelNode(o,r,e,i)),l.overrideMaster(o),l.setMergeMode(n),o=l}return o}function c(e){for(var t=[],n=0,r=e;n<r.length;n++){var i=r[n];f(t,i,!0)}return t}function f(e,t,n){void 0===n&&(n=!1);var r=n?e:e.concat([]),i=t.lowLevel().unit();if(null==i)return r;if(0==r.length)r.push(i);else{var a=r[r.length-1].absolutePath();i.absolutePath()!=a&&r.push(i)}return r}function d(e){for(var t=[];e;)t.push(e),e=e.parent();return c(t)}function h(){return x.map(function(e){return e.name})}function m(e){for(var t=[],n=e.split("|").slice(1),r=0,i=n;r<i.length;r++)for(var a=i[r],o=0;o<x.length;o++)if(a.match(x[o].regexp)){t.push(x[o].transformer);break}return t}function y(e){return e?"*/*"==e?null:(e.indexOf("/*")==e.length-2&&(e=e.substring(0,e.length-2)+"/xxx"),C.parse(e)):null}var v=this&&this.__extends||function(e,t){function n(){this.constructor=e}for(var r in t)t.hasOwnProperty(r)&&(e[r]=t[r]);e.prototype=null===t?Object.create(t):(n.prototype=t.prototype,new n)},g=n(24),A=n(16),E=n(77),T=n(33),S=n(44),b=n(76),_=n(18),N=n(70),w=n(51),R=n(53),I=n(39),M=n(14),C=n(68),L=n(78);t.expandTraitsAndResourceTypes=r,t.expandTraitsAndResourceTypesHL=i,t.expandLibraries=a,t.expandLibrary=o,t.expandLibrariesHL=s,t.expandLibraryHL=u,t.mergeAPIs=l;var P=function(){function e(){}return e.prototype.expandTraitsAndResourceTypes=function(e,t,n){void 0===t&&(t=null),void 0===n&&(n=!1),this.init(e);var r=e.lowLevel();if(!r)return e;if(r.actual().libExpanded)return e;var i=e.lowLevel().unit(),a=this.createHighLevelNode(e,!1,t);if(e.definition().key()==_.Universe10.Overlay)return a;var o=i.project().namespaceResolver().hasFragments(i),s=this.expandHighLevelNode(a,t,e,o);return s?s:e},e.prototype.init=function(e){this.ramlVersion=e.definition().universe().version()},e.prototype.expandHighLevelNode=function(e,t,n,r){var i=this;void 0===r&&(r=!1),this.init(e),e.setMergeMode(n.getMergeMode());var a=e.elementsOfKind("resources"),o=!1;if(a.forEach(function(e){return o=i.processResource(e)||o}),!o&&!r)return null;if(A.ASTNodeImpl.isInstance(e)){var s=e;null!=s.reusedNode()&&s.setReuseMode(!0)}return"RAML10"==this.ramlVersion&&(t=t||new R.ReferencePatcher,t.process(e),e.lowLevel().actual().referencePatcher=t),e},e.prototype.getTemplate=function(e,t,n,r){var i=t.lowLevel().unit().path(),a=n[i];a||(a={},n[i]=a);var o=a[e];return void 0!==o?o:(o=null,o=N.find(r,function(n){return A.qName(n.highLevel(),t)==e}),o||(o=null),a[e]=o,o)},e.prototype.createHighLevelNode=function(e,t,n,r){void 0===t&&(t=!0),void 0===n&&(n=null),void 0===r&&(r=!1);for(var i=e,a=[],o=i;o;){var s,u=o.lowLevel();i.definition().key()!=_.Universe10.Overlay||r?(S.LowLevelCompositeNode.isInstance(u)&&(u=u.originalNode().originalNode()),s=new S.LowLevelCompositeNode(u,null,null,this.ramlVersion)):s=u;var l=o.definition(),c=new A.ASTNodeImpl(s,null,l,null);if(c.setUniverse(o.universe()),a.push(c),!t)break;o=o.getMaster()}var f=a.pop();a=a.reverse();var d=i.getMergeMode(),h=p(f,a,d,n,r);return h.setReusedNode(i.reusedNode()),h},e.prototype.processResource=function(e,t){var n=this;void 0===t&&(t=[]);var r=!1,i=t.concat(e),a=this.collectResourceData(e,e,void 0,void 0,i),o=e.lowLevel();if(!S.LowLevelProxyNode.isInstance(o))return r;o.preserveAnnotations(),o.takeOnlyOriginalChildrenWithKey(I.universesInfo.Universe10.ResourceBase.properties.type.name),o.takeOnlyOriginalChildrenWithKey(I.universesInfo.Universe10.FragmentDeclaration.properties.uses.name),a.filter(function(e){return null!=e.resourceType}).forEach(function(t){var n=t.resourceType.node.lowLevel(),i=t.resourceType.transformer;i.owner=e,o.adopt(n,i),r=!0});var s=e.elementsOfKind("methods");s.forEach(function(e){var t=e.lowLevel();S.LowLevelCompositeNode.isInstance(t)&&t.takeOnlyOriginalChildrenWithKey(I.universesInfo.Universe10.FragmentDeclaration.properties.uses.name);var n=e.attr("method").value(),i=[];a.forEach(function(a){var o=a.methodTraits[n];o&&(i=i.concat(o),o.forEach(function(n){var i=n.node.lowLevel(),a=n.transformer;a.owner=e,t.adopt(i,a),r=!0}));var s=a.traits;s&&(i=i.concat(s),s.forEach(function(n){var i=n.node.lowLevel(),a=n.transformer;a.owner=e,t.adopt(i,a),r=!0}))})});var u=e.elementsOfKind("resources");return u.forEach(function(e){return r=n.processResource(e,i)||r}),e.elementsOfKind("methods").forEach(function(e){return n.mergeBodiesForMethod(e)}),r},e.prototype.mergeBodiesForMethod=function(e){var t=e.lowLevel();if(S.LowLevelCompositeNode.isInstance(t)){var n=e.computedValue(_.Universe10.Api.properties.mediaType.name);if(null!=n){for(var r,i=[],a=t.children(),o=0,s=a;o<s.length;o++){var u=s[o];if(u.key()==_.Universe10.Method.properties.body.name)r=u;else if(u.key()==_.Universe10.Method.properties.responses.name)for(var l=u.children(),p=0,c=l;p<c.length;p++)for(var f=c[p],d=f.children(),h=0,m=d;h<m.length;h++){var y=m[h];y.key()==_.Universe10.Response.properties.body.name&&i.push(y)}}r&&i.push(r);for(var v=0,g=i;v<g.length;v++){var A=g[v];this.mergeBodies(A,n)}}}},e.prototype.mergeBodies=function(e,t){for(var n,r=[],i=[],a=[],o=[],s=!1,u=0,l=e.children();u<l.length;u++){var p=l[u],c=p.key();if(c==t)n=p,a.push({node:R.toOriginal(p),transformer:p.transformer()});else try{y(c),i.push(p)}catch(f){var d=R.toOriginal(p).parent();o.indexOf(d)<0&&(a.push({node:d,transformer:p.transformer()}),o.push(d)),K[p.key()]&&(s=!0),r.push(p)}}if(0==r.length||null==n&&0==i.length)return!1;if(s){for(var h=0,m=r;h<m.length;h++){var p=m[h];e.removeChild(p)}if(null==n){var v=R.toOriginal(e),A=E.newMapping(E.newScalar(t),E.newMap([])),T=new g.ASTNode(A,v.unit(),v,null,null);n=e.replaceChild(null,T)}return n.patchAdoptedNodes(a),!0}},e.prototype.collectResourceData=function(e,t,n,r,i,a){var o=this;void 0===n&&(n=[]),void 0===i&&(i=[]),void 0===a&&(a={}),i=i.concat([t]);var s=this.extractTraits(t,r,i),u={},l=t.elementsOfKind("methods");l.forEach(function(e){var t=o.extractTraits(e,r,i);t&&t.length>0&&(u[e.attr("method").value()]=t)});var p,f=t.attr("type");if(null!=f){var d=c(i);p=this.readGenerictData(e,f,t,"resource type",r,d)}var h={resourceType:p,traits:s,methodTraits:u};if(n.push(h),p){var m=p.node,y=A.qName(m,e);a[y]?h.resourceType=null:(a[y]=!0,this.collectResourceData(e,m,n,p.transformer,i,a))}return n},e.prototype.extractTraits=function(e,t,n,r){var i=this;void 0===r&&(r={}),n=n.concat([e]);for(var a=[],o=-1;o<a.length;o++){var s=0>o?null:a[o],u=s?s.node:e,l=s?s.unitsChain:c(n),p=s?s.transformer:t;u.attributes("is").forEach(function(t){var n=f(l,t),o=i.readGenerictData(e,t,u,"trait",p,n);if(o){var s=o.name;r[s]=!0,a.push(o)}})}return a},e.prototype.readGenerictData=function(e,t,n,r,i,a){void 0===a&&(a=[]);var o,s,u=t.value(),l=b.plural(L.camelCase(r));if("string"==typeof u)s=u;else{if(!A.StructuredValue.isInstance(u))return null;o=u,s=o.valueName()}i&&(s=i.transform(s).value);var p={},c={},f=R.getDeclaration(s,l,a);if(f){var d=new F(e,null,a);o&&("RAML08"==this.ramlVersion?o.children().forEach(function(e){return p[e.valueName()]=e.lowLevel().value()}):o.children().forEach(function(e){var t=e.lowLevel(),t=e.lowLevel();t.resolvedValueKind()==E.Kind.SCALAR?p[e.valueName()]=t.value():c[e.valueName()]=t}),Object.keys(p).forEach(function(e){var t=d.transform(p[e]);t&&"object"!=typeof t&&(p[e]=t)}));var h=new k(r,s,a,p,c,i),m=new F(null,h,a);return{name:s,transformer:m,parentTransformer:i,node:f,ref:t,unitsChain:a}}return null},e}();t.TraitsAndResourceTypesExpander=P;var O=function(){function e(){}return e.prototype.expandLibraries=function(e){var t=e;if(null==t)return null;S.LowLevelCompositeNode.isInstance(t.lowLevel())&&(t=t.lowLevel().unit().highLevel().asElement());var n=new P,r=new R.ReferencePatcher,i=n.createHighLevelNode(t,!0,r,!0),a=n.expandHighLevelNode(i,r,t,!0);return this.processNode(r,a),a},e.prototype.expandLibrary=function(e){var t=e;if(null==t)return null;S.LowLevelCompositeNode.isInstance(t.lowLevel())&&(t=t.lowLevel().unit().highLevel().asElement());var n=new P,r=new R.ReferencePatcher,i=n.createHighLevelNode(t,!0,r,!0);return r.process(i),r.expandLibraries(i,!0),i},e.prototype.processNode=function(e,t){if(null!=t){var n=t.getMaster();this.processNode(e,n),M.isOverlayType(t.definition())&&e.process(t),e.expandLibraries(t)}},e}();t.LibraryExpander=O,t.toUnits=d;var D=function(){function e(t,n){this.name=t,this.regexp=new RegExp(e.leftTransformRegexp.source+t+e.rightTransformRegexp.source),this.transformer=n}return e.leftTransformRegexp=/\s*!\s*/,e.rightTransformRegexp=/\s*$/,e}(),x=[new D("singularize",function(e){return b.singular(e)}),new D("pluralize",function(e){return b.plural(e)}),new D("uppercase",function(e){return e?e.toUpperCase():e}),new D("lowercase",function(e){return e?e.toLowerCase():e}),new D("lowercamelcase",function(e){return e?L.camelCase(e):e}),new D("uppercamelcase",function(e){if(!e)return e;var t=L.camelCase(e);return t[0].toUpperCase()+t.substring(1,t.length)}),new D("lowerunderscorecase",function(e){if(!e)return e;var t=L.snake(e);return t.toLowerCase()}),new D("upperunderscorecase",function(e){if(!e)return e;var t=L.snake(e);return t.toUpperCase()}),new D("lowerhyphencase",function(e){if(!e)return e;var t=L.param(e);return t.toLowerCase()}),new D("upperhyphencase",function(e){if(!e)return e;var t=L.param(e);return t.toUpperCase()})];t.getTransformNames=h,t.getTransformersForOccurence=m;var U=function(){function e(){this.buf=null}return e.prototype.append=function(e){""!==e&&(null!=this.buf?null!=e&&("string"!=typeof this.buf&&(this.buf=""+this.buf),this.buf+=e):""!==e&&(this.buf=e))},e.prototype.value=function(){return null!=this.buf?this.buf:""},e}(),k=function(){function e(e,t,n,r,i,a){this.templateKind=e,this.templateName=t,this.unitsChain=n,this.params=r,this.structuredParams=i,this.vDelegate=a}return e.prototype.transform=function(e,t,n,r){var i={},a=[];if("string"==typeof e){if(this.structuredParams&&T.stringStartsWith(e,"<<")&&T.stringEndsWith(e,">>")){var o=e.substring(2,e.length-2),s=this.structuredParams[o];if(null!=s)return{value:s,errors:a}}for(var u=e,l=new U,p=0,c=u.indexOf("<<");c>=0;c=u.indexOf("<<",p)){l.append(u.substring(p,c));var f=c;if(c+="<<".length,p=this.paramUpperBound(u,c),-1==p)break;var d=u.substring(c,p);p+=">>".length;var h,o,y=u.substring(f,p),v=m(d);if(v.length>0){var g=d.indexOf("|");if(o=d.substring(0,g).trim(),h=this.params[o],h&&"string"==typeof h&&h.indexOf("<<")>=0&&this.vDelegate&&(h=this.vDelegate.transform(h,t,n,r).value),h){R.PatchedReference.isInstance(h)&&(h=h.value());for(var A=0,E=v;A<E.length;A++){var S=E[A];h=S(h)}}}else o=d.trim(),h=this.params[o],h&&"string"==typeof h&&h.indexOf("<<")>=0&&this.vDelegate&&(h=this.vDelegate.transform(h,t,n,r).value);(null===h||void 0===h)&&(i[o]=!0,h=y),l.append(h)}return l.append(u.substring(p,u.length)),{value:l.value(),errors:a}}return{value:e,errors:a}},e.prototype.paramUpperBound=function(e,t){for(var n=0,r=t;r<e.length;r++)if(T.stringStartsWith(e,"<<",r))n++,r++;else if(T.stringStartsWith(e,">>",r)){if(0==n)return r;n--,r++}return e.length},e.prototype.children=function(e){var t=this.substitutionNode(e);return t?t.children():null},e.prototype.valueKind=function(e){var t=this.substitutionNode(e);return t?t.valueKind():null},e.prototype.anchorValueKind=function(e){var t=this.substitutionNode(e);return t&&t.valueKind()==E.Kind.ANCHOR_REF?t.anchorValueKind():null},e.prototype.resolvedValueKind=function(e){var t=this.substitutionNode(e);return t&&t.resolvedValueKind()},e.prototype.includePath=function(e){var t=this.substitutionNode(e);return t?t.includePath():null},e.prototype.substitutionNode=function(e){var t=this.paramName(e);return t&&this.structuredParams[t]},e.prototype.paramName=function(e){var t=null;if(e.valueKind()==E.Kind.SCALAR){var n=(""+e.value()).trim();T.stringStartsWith(n,"<<")&&T.stringEndsWith(n,">>")&&(t=n.substring(2,n.length-2))}return t},e.prototype.definingUnitSequence=function(e){if(e.length<2)return null;if("("==e.charAt(0)&&")"==e.charAt(e.length-1)&&(e=e.substring(1,e.length-1)),e.length<4)return null;if("<<"!=e.substring(0,2))return null;if(">>"!=e.substring(e.length-2,e.length))return null;var t=e.substring(2,e.length-2);return t.indexOf("<<")>=0||t.indexOf(">>")>=0?null:this._definingUnitSequence(t)},e.prototype._definingUnitSequence=function(e){return this.params&&this.params[e]?this.unitsChain:this.vDelegate?this.vDelegate._definingUnitSequence(e):null},e}();t.ValueTransformer=k;var F=function(e){function t(t,n,r){e.call(this,null!=n?n.templateKind:"",null!=n?n.templateName:"",r),this.owner=t,this.delegate=n}return v(t,e),t.prototype.transform=function(t,n,r,i){if(null==t||null!=r&&!r())return{value:t,errors:[]};var a={value:t,errors:[]},o=!1;B.forEach(function(e){return o=o||t.toString().indexOf("<<"+e)>=0}),o&&(this.initParams(),a=e.prototype.transform.call(this,t,n,r,i));var s=null!=this.delegate?this.delegate.transform(a.value,n,r,i):a.value;return null!=r&&r()&&null!=i&&(s.value=i(s.value,this)),s},t.prototype.initParams=function(){for(var e,t,n="",r=this.owner.lowLevel(),i=r,a=null;i;){var o=i.key();if(null!=o)if(T.stringStartsWith(o,"/")){if(!t)for(var s=o.split("/"),u=s.length-1;u>=0;u--){var l=s[u];if(-1==l.indexOf("{")){t=s[u];break}l.length>0&&(a=l)}n=o+n}else e=o;i=i.parent()}t||a&&(t=""),this.params={resourcePath:n,resourcePathName:t},e&&(this.params.methodName=e)},t.prototype.children=function(e){return null!=this.delegate?this.delegate.children(e):null},t.prototype.valueKind=function(e){return null!=this.delegate?this.delegate.valueKind(e):null},t.prototype.includePath=function(e){return null!=this.delegate?this.delegate.includePath(e):null},t.prototype.anchorValueKind=function(e){return null!=this.delegate?this.delegate.anchorValueKind(e):null},t.prototype.resolvedValueKind=function(e){return null!=this.delegate?this.delegate.resolvedValueKind(e):null},t.prototype._definingUnitSequence=function(e){return this.params&&this.params[e]?this.unitsChain:this.delegate?this.delegate._definingUnitSequence(e):null},t}(k);t.DefaultTransformer=F;var B=["resourcePath","resourcePathName","methodName"],K={};K[I.universesInfo.Universe10.TypeDeclaration.properties.type.name]=!0,K[I.universesInfo.Universe10.TypeDeclaration.properties.example.name]=!0,K[I.universesInfo.Universe08.BodyLike.properties.schema.name]=!0,K[I.universesInfo.Universe10.ObjectTypeDeclaration.properties.properties.name]=!0,t.parseMediaType=y},function(e,t,n){"use strict";function r(e){return a.isWebPath(e)}var i=n(15),a=n(10),o=n(25),s=function(){function e(e){this.unit=e}return e.prototype.contextPath=function(){if(!this.unit)return"";var e=this.unit.absolutePath();return e||""},e.prototype.normalizePath=function(e){if(!e)return e;var t;if(r(e)){var n=0===e.toLowerCase().indexOf("https")?"https://":"http://";t=n+i.normalize(e.substring(n.length)).replace(/\\/g,"/")}else t=i.normalize(e).replace(/\\/g,"/");return t},e.prototype.content=function(e){var t=this.normalizePath(e),n=this.toRelativeIfNeeded(t),r=this.unit.resolve(n);return r?r.contents()||"":""},e.prototype.contentAsync=function(e){var t=this.normalizePath(e),n=this.toRelativeIfNeeded(t),r=this.unit.resolveAsync(n);if(!r)return Promise.resolve("");var i=r.then(function(e){return e&&e.contents()||""});return i},e.prototype.toRelativeIfNeeded=function(e){var t=e;return i.isAbsolute(e)&&!r(e)&&(t=i.relative(i.dirname(this.unit.absolutePath()),e)),t},e.prototype.hasAsyncRequests=function(){return o.hasAsyncRequests()},e.prototype.resolvePath=function(e,t){return a.buildPath(t,e,this.unit.project().getRootPath())},e.prototype.isAbsolutePath=function(e){return e?r(e)?!0:i.isAbsolute(e):!1},e.prototype.promiseResolve=function(e){return Promise.resolve(e)},e}();t.ContentProvider=s},function(e,t,n){"use strict";function r(e,t){return new M(t).dump(e)}function i(e,t,n,r,i){if(!t)return e;var a={};e&&e.arr.forEach(function(e){var t=a[e.name()];t||(t=[],a[e.name()]=t),t.push(e)});for(var o=new C(r),s=0,u={},l=!1,p=t.indexOf("{");p>=0&&(s=t.indexOf("}",++p),!(0>s));p=t.indexOf("{",s)){var f=t.substring(p,s);if(u[f]=!0,a[f])a[f].forEach(function(e){o.registerValue(e),o.registerMeta(null)});else{l=!0;var d=n.definition().universe(),h=d.type(E.Universe10.StringTypeDeclaration.name),m=_.createStubNode(h,null,f,n.lowLevel().unit());if(m.setParent(n),m.attrOrCreate("name").setValue(f),m.patchProp(r),o.registerValue(m),i){o.hasMeta=!0;var y=new c.NodeMetadataImpl;y.setCalculated(),o.registerMeta(y)}}}return l?(Object.keys(a).filter(function(e){return!u[e]}).forEach(function(e){return a[e].forEach(function(e){o.registerValue(e),o.hasMeta&&o.registerMeta(null)})}),o):e}function a(e,t,n,r){if(0==t.length)return null;var i=e.lowLevel().unit().absolutePath(),a=new C(n);return t.forEach(function(e){if(a.registerValue(e),r)if(e.lowLevel().unit().absolutePath()!=i){a.hasMeta=!0;var t=new c.NodeMetadataImpl;t.setCalculated(),a.mArr.push(t)}else a.mArr.push(null)}),a}function o(e,t,n,r){var i;if(e.isElement())i=e.asElement().definition();else if(e.isAttr()){var a=e.asAttr().property();a&&(i=a.range())}if((i instanceof d.UserDefinedClass||i.isUserDefined())&&(i=N.find(i.allSuperTypes(),function(e){return!e.isUserDefined()})),null==i)return n;var o=i.universe().version(),s=r[o];if(!s)return n;var u=s[i.nameId()];if(!u)return n;var l=t?t.nameId():"__$$anyprop__",p=u[l];if(p||(p=u.__$$anyprop__),!p)return n;for(var c=0,f=p;c<f.length;c++){var h=f[c];n=h.transform(n,e)}return n}function s(e,t){for(var n=0,r=e;n<r.length;n++){var i=r[n],a=i.registrationInfo();if(a)for(var o=0,s=Object.keys(a);o<s.length;o++){var u=s[o],l=a[u],p=t[u];null==p&&(p={},t[u]=p);for(var c=0,f=Object.keys(l);c<f.length;c++){var d=f[c],h=l[d],m=p[d];null==m&&(m={},p[d]=m);for(var y=0,v=Object.keys(h);y<v.length;y++){var g=v[y],A=m[g];if(null==A){if(A=[],"__$$anyprop__"!=g){var E=m.__$$anyprop__;E&&(A=A.concat(E))}m[g]=A}if("__$$anyprop__"==g)for(var T=0,S=Object.keys(m);T<S.length;T++){var b=S[T];m[b].push(i)}else A.push(i)}}}}}function u(e){if(0==e.length)return{};for(var t=e[0],n=1;n<e.length;n++){var r=e[n];t=l(t,r)}return t}function l(e,t){for(var n=0,r=Object.keys(t);n<r.length;n++){var i=r[n],a=e[i],o=t[i];null==a?e[i]=o:"object"==typeof a&&"object"==typeof o&&(e[i]=l(a,o))}return e}var p=this&&this.__extends||function(e,t){function n(){this.constructor=e}for(var r in t)t.hasOwnProperty(r)&&(e[r]=t[r]);e.prototype=null===t?Object.create(t):(n.prototype=t.prototype,new n)},c=n(51),f=n(44),d=n(39),h=n(16),m=n(29),y=n(53),v=d.rt,g=v.typeExpressions,A=n(14),E=n(18),T=n(33),S=n(58),b=n(59),_=n(17),N=n(70),w=n(15),R="application/raml+yaml";t.dump=r;var I=function(e){var t,n=e.root();if(n){var r=n.lowLevel();if(r){var i=r.unit();i&&(t=i.absolutePath())}}return t},M=function(){function e(e){this.options=e,this.nodePropertyTransformers=[new z,new J,new Q,new ee,new Z,new te,new Y,new ne,new ie,new re,new ae],this.nodeTransformersMap={},this.nodePropertyTransformersMap={},this.options=this.options||{},null==this.options.serializeMetadata&&(this.options.serializeMetadata=!0),null==this.options.attributeDefaults&&(this.options.attributeDefaults=!0),this.defaultsCalculator=new S.AttributeDefaultsCalculator(!0,!0),this.nodeTransformers=[new V,new j(this.options),new W,new q,new H,new $,new G,new oe,new X(this.options.allUriParameters)],s(this.nodeTransformers,this.nodeTransformersMap),s(this.nodePropertyTransformers,this.nodePropertyTransformersMap)}return e.prototype.init=function(e){this.helpersMap={baseUriParameters:L,uriParameters:P};var t=e.isElement();if(t){var n=e.asElement(),r=n.definition();if("RAML08"==r.universe().version()&&A.isApiType(r)){var i={};n.elementsOfKind(E.Universe08.Api.properties.schemas.name).forEach(function(e){return i[e.name()]=e}),this.helpersMap.schemaContent=new D(i)}A.isApiSibling(r)&&(this.helpersMap.traits=new O(b.allTraits(n,!1)),this.helpersMap.resourceTypes=new O(b.allResourceTypes(n,!1)))}},e.prototype.dispose=function(){delete this.helpersMap},e.prototype.dump=function(e){this.init(e);var t=e.isElement(),n=e.parent(),r=!n&&this.options.rootNodeDetails,i=I(e),a=this.dumpInternal(e,null,i,null,!0);if(r){var o=a;if(a={},a.specification=o,t){var s=e.asElement(),u=s.definition();if(u){var l=u.universe().version();a.ramlVersion=l,a.type=u.nameId()}a.errors=this.dumpErrors(c.errors(s))}}return this.dispose(),a},e.prototype.dumpInternal=function(e,t,n,r,i){var a=this;if(void 0===i&&(i=!1),null==e)return null;if(e.isReused()){var s=e.getJSON();if(null!=s)return s}var u={};if(e.isElement()){var l={},p=e.asElement(),f=p.definition();if(A.isExampleSpecType(f))if(null!=p.parent())u="";else{var y=h.auxiliaryTypeForExample(p),v=b.dumpExpandableExample(y.examples()[0],this.options.dumpXMLRepresentationOfExamples),g=p.elementsOfKind("uses").map(function(e){return a.dumpInternal(e,e.property(),n)});g.length>0&&(v.uses=g),u=v}else{for(var _={},R=p.attrs().concat(p.children().filter(function(e){return!e.isAttr()})),I=0,M=R;I<M.length;I++){var L=M[I],P=L.property();if(null!=P){var O=P.nameId(),D=l[O];null==D&&(D=new C(P),l[O]=D),D.registerValue(L)}else var x=L.lowLevel(),U=x.key()}for(var k={},F=0,B=f.allProperties().concat(f.allCustomProperties());F<B.length;F++){var K=B[F];if(!d.UserDefinedProp.isInstance(K)){var O=K.nameId();if(i||"uses"!=O||!A.isApiSibling(p.root().definition())){var D=l[O];if(A.isTypeProperty(K)){if(l.schema){var V=null==D;!V&&1==D.arr.length&&D.arr[0].isAttr()&&(V=null==D.arr[0].asAttr().value()),V&&(r=r||new c.NodeMetadataImpl,r.registerInsertedAsDefaultValue("type"));continue}if(A.isStringTypeDeclarationDescendant(f)){if(null==D){u.type="string",r=r||new c.NodeMetadataImpl,r.registerInsertedAsDefaultValue("type");continue}if(1==D.arr.length&&D.arr[0].isAttr()){var j=D.arr[0].asAttr().value();if(null==j){u.type="string",r=r||new c.NodeMetadataImpl,r.registerInsertedAsDefaultValue("type");
continue}if("NULL"===j||"Null"===j){u.type="string";continue}}}}D=this.applyHelpers(D,p,K,this.options.serializeMetadata);var W=_[O],q=void 0;if(void 0!==D){if(D.isMultiValue){if(q=D.arr.map(function(e,t){var r=D.hasMeta?D.mArr[t]:null;return a.dumpInternal(e,D.prop,n,r)}),K.isValueProperty()){var Y=[],H=!1;D.arr.filter(function(e){return e.isAttr()}).map(function(e){return e.asAttr()}).filter(function(e){return e.isAnnotatedScalar()}).forEach(function(e){var t=e.annotations().map(function(e){return a.dumpInternal(e,null,n)});H=H||t.length>0,Y.push(t)}),H&&(k[O]=Y)}A.isTypeDeclarationDescendant(f)&&A.isTypeProperty(K)&&D.arr.map(function(e){return e.value()}).filter(function(e){return h.isStructuredValue(e)}).length>0&&(q=q[0])}else if(q=this.dumpInternal(D.val,D.prop,n),K.isValueProperty()){var $=D.val.asAttr();if($.isAnnotatedScalar()){var Y=$.annotations().map(function(e){return a.dumpInternal(e,null,n)});Y.length>0&&(k[O]=Y)}}}else if(void 0!==W)q=W;else if(this.options.attributeDefaults){var G=this.defaultsCalculator.attributeDefaultIfEnabled(p,K);if(null!=G){r=r||new c.NodeMetadataImpl,Array.isArray(G)?G=G.map(function(e){return h.isASTPropImpl(e)?a.dumpInternal(e,K,n):e}):h.isASTPropImpl(G)&&(G=this.dumpInternal(G,K,n)),q=G,null!=q&&K.isMultiValue()&&!Array.isArray(q)&&(q=[q]);var X=this.defaultsCalculator.insertionKind(p,K);X==S.InsertionKind.CALCULATED?r.registerCalculatedValue(O):X==S.InsertionKind.BY_DEFAULT&&r.registerInsertedAsDefaultValue(O)}}if(q=o(p,K,q,this.nodePropertyTransformersMap),null!=q){if(("type"===O||"schema"==O)&&q&&q.forEach&&"string"==typeof q[0]){var z=q[0].trim(),J="{"===z[0]&&"}"===z[z.length-1],Q="<"===z[0]&&">"===z[z.length-1];if(J||Q){var Z=p.lowLevel().includePath&&p.lowLevel().includePath();if(!Z){var ee=p.attr("type");ee||(ee=p.attr("schema")),ee&&(Z=ee.lowLevel().includePath&&ee.lowLevel().includePath())}if(Z){var te=Z.indexOf("#"),ne="";te>=0&&(ne=Z.substring(te),Z=Z.substring(0,te));var re,ie=p.lowLevel().unit().resolve(Z).absolutePath();re=T.stringStartsWith(ie,"http://")||T.stringStartsWith(ie,"https://")?ie:w.relative(p.lowLevel().unit().project().getRootPath(),ie),re=re.replace(/\\/g,"/"),u.schemaPath=re+ne}}}u[O]=q}}}}if(this.options.dumpSchemaContents&&l.schema&&l.schema.prop.range().key()==E.Universe08.SchemaString){var ae=p.root().elementsOfKind("schemas");ae.forEach(function(e){if(e.name()==u.schema){var t=e.attr("value");t&&(u.schema=t.value(),u.schemaContent=t.value())}})}this.options.serializeMetadata&&this.serializeMeta(u,p,r),Object.keys(k).length>0&&(u.scalarsAnnotations=k);var oe=b.getTemplateParametrizedProperties(p);if(oe&&(u.parametrizedProperties=oe),A.isTypeDeclarationDescendant(f)){var se=b.typeFixedFacets(p);se&&(u.fixedFacets=se)}u=o(p,t||p.property(),u,this.nodeTransformersMap)}}else if(e.isAttr()){var ue,le=e.asAttr(),ue=le.value(),P=le.property(),pe=P.range(),ce=pe.isValueType();if(ce&&le.value&&(ue=le.value(),null==ue&&A.isAnyTypeType(pe))){var fe=le.lowLevel();le.isAnnotatedScalar()&&(fe=N.find(fe.children(),function(e){return"value"==e.key()})),fe&&(ue=le.lowLevel().dumpToObject())}if(null==ue||"number"!=typeof ue&&"string"!=typeof ue&&"boolean"!=typeof ue){if(h.isStructuredValue(ue)){var de=ue,x=de.lowLevel();ue=x?x.dumpToObject():null;var he=P.nameId();if(pe.isAssignableFrom("Reference")){var U=Object.keys(ue)[0],me=de.valueName(),ye=ue[U];void 0===ye&&(ye=null),ue={name:me,structuredValue:ye}}else if("type"==he){var x=le.lowLevel(),ve=d.getUniverse("RAML10").type(E.Universe10.TypeDeclaration.name),ge=d.getUniverse("RAML10").type(E.Universe10.LibraryBase.name),Ae=new h.ASTNodeImpl(x,le.parent(),ve,ge.property(E.Universe10.LibraryBase.properties.types.name));Ae.patchType(m.doDescrimination(Ae)),ue=this.dumpInternal(Ae,t||le.property(),n,null,!0)}else if("items"==he&&"object"==typeof ue){var Ee=Array.isArray(ue),Te=!Ee;if(Ee&&(Te=null!=N.find(ue,function(e){return"object"==typeof e})),Te){ue=null;var Se=e.parent().lowLevel();Se.children().forEach(function(e){if("items"==e.key()){var r=d.getUniverse("RAML10").type(E.Universe10.TypeDeclaration.name),i=d.getUniverse("RAML10").type(E.Universe10.LibraryBase.name),o=new h.ASTNodeImpl(e,le.parent(),r,i.property(E.Universe10.LibraryBase.properties.types.name));o.patchType(m.doDescrimination(o)),ue=a.dumpInternal(o,t||le.property(),n,null,!0),he=e.key()}})}}}ue=o(le,t||le.property(),ue,this.nodeTransformersMap),u=ue}else u=ue}else{var x=e.lowLevel();u=x?x.dumpToObject():null}return e.setJSON(u),u},e.prototype.getDefaultsCalculator=function(){return this.defaultsCalculator},e.prototype.canBeFragment=function(e){var t=e.definition(),n=[t].concat(t.allSubTypes()),r=n.filter(function(e){return e.getAdapter(d.RAMLService).possibleInterfaces().filter(function(e){return e.nameId()==d.universesInfo.Universe10.FragmentDeclaration.name}).length>0});return r.length>0},e.prototype.dumpErrors=function(e){var t=this;return e.map(function(e){var n=t.dumpErrorBasic(e);return e.trace&&e.trace.length>0&&(n.trace=t.dumpErrors(e.trace)),n}).sort(function(e,t){return e.path!=t.path?e.path.localeCompare(t.path):e.range.start.position!=t.range.start.position?e.range.start.position-t.range.start.position:e.code-t.code})},e.prototype.dumpErrorBasic=function(e){var t={code:e.code,message:e.message,path:e.path,line:e.line,column:e.column,position:e.start,range:e.range};return e.isWarning===!0&&(t.isWarning=!0),t},e.prototype.serializeMeta=function(e,t,n){if(this.options.serializeMetadata){var r=t.definition(),i=A.isMethodType(r)&&t.optional();if(n||i){var a=n||new c.NodeMetadataImpl(!1,!1);i&&a.setOptional(),e.__METADATA__=a.toJSON()}}},e.prototype.applyHelpers=function(e,t,n,r){var i=n.nameId(),a=this.helpersMap[i];if(!a)return e;var o=a.apply(t,e,n,r);return o?o:e},e}();t.TCKDumper=M;var C=function(){function e(e){this.prop=e,this.arr=[],this.mArr=[],this.isMultiValue=e.isMultiValue()}return e.prototype.registerValue=function(e){this.isMultiValue?this.arr.push(e):this.val=e},e.prototype.registerMeta=function(e){this.isMultiValue&&this.mArr.push(e)},e}(),L={apply:function(e,t,n,r){var a=e.attr(E.Universe10.Api.properties.baseUri.name),o=a?a.value():"";return i(t,o,e,n,r)}},P={apply:function(e,t,n,r){var a=e.attr(E.Universe10.Resource.properties.relativeUri.name);if(!a)return t;var o=a.value();return i(t,o,e,n,r)}},O=function(){function e(e){this.arr=e}return e.prototype.apply=function(e,t,n,r){return a(e,this.arr,n,r)},e}(),D=function(){function e(e){this.schemasCache08=e}return e.prototype.apply=function(e,t,n,r){var i=null,a=b.schemaContent08Internal(e,this.schemasCache08);return a&&(i=new C(n),i.registerValue(a)),i},e}();t.applyTransformersMap=o;var x=function(){function e(){}return e.prototype.match=function(e,t){if(null==e)return!1;var n=this.registrationInfo(),r=e.universe().version();if((e instanceof d.UserDefinedClass||e.isUserDefined())&&(e=N.find(e.allSuperTypes(),function(e){return!e.isUserDefined()}),null==e))return null==t;var i=n[r];if(!i)return!1;var a=i[e.nameId()];if(!a)return!1;var o=null==t||a[t.nameId()]===!0||a.__$$anyprop__===!0;return o},e}(),U=function(e){function t(t,n,r,i){void 0===r&&(r=!1),void 0===i&&(i=["RAML10","RAML08"]),e.call(this),this.typeName=t,this.propName=n,this.applyToDescendatns=r,this.restrictToUniverses=i}return p(t,e),t.prototype.registrationInfo=function(){var e=this;if(this.regInfo)return this.regInfo;for(var t={},n=[],r=0,i=this.restrictToUniverses;r<i.length;r++){var a=i[r],o={};t[a]=o,n.push(o)}for(var s=0,u=Object.keys(t);s<u.length;s++){var a=u[s],l=d.getUniverse(a).type(this.typeName);if(l){var p=t[a],c=[this.typeName];this.applyToDescendatns&&l.allSubTypes().forEach(function(e){return c.push(e.nameId())});for(var f=0,h=c;f<h.length;f++){var m=h[f],y={};null!=this.propName?y[this.propName]=!0:y.__$$anyprop__=!0,p[m]=y}}}return this.regInfo={},Object.keys(t).forEach(function(n){var r=t[n];Object.keys(r).length>0&&(e.regInfo[n]=r)}),this.regInfo},t}(x),k=function(){function e(e){this.matcher=e}return e.prototype.match=function(e,t){var n;if(e.isElement())n=e.asElement().definition();else if(e.isAttr()){var r=e.asAttr().property();r&&(n=r.range())}return n?this.matcher.match(n,t):!1},e.prototype.registrationInfo=function(){return this.matcher.registrationInfo()},e}(),F=function(e){function t(t,n,r,i){void 0===r&&(r=!1),void 0===i&&(i=["RAML10","RAML08"]),e.call(this,new U(t,n,r,i)),this.typeName=t,this.propName=n,this.applyToDescendatns=r,this.restrictToUniverses=i}return p(t,e),t}(k),B=function(e){function t(t){e.call(this),this.matchers=t}return p(t,e),t.prototype.registrationInfo=function(){return this.regInfo?this.regInfo:(this.regInfo=u(this.matchers.map(function(e){return e.registrationInfo()})),this.regInfo)},t}(x),K=function(){function e(e,t){this.matcher=e,this.propName=t}return e.prototype.match=function(e,t){return e.isElement()&&this.matcher.match(e.asElement().definition(),t)},e.prototype.transform=function(e,t){var n=this;if(Array.isArray(e)&&e.length>0&&e[0][this.propName]){var r={};return e.forEach(function(e){var t=e["$$"+n.propName];null!=t?delete e["$$"+n.propName]:t=e[n.propName];var i=r[t];i?Array.isArray(i)?i.push(e):r[t]=[i,e]:r[t]=e}),r}return e},e.prototype.registrationInfo=function(){return this.matcher.registrationInfo()},e}(),V=function(e){function t(){e.call(this,E.Universe10.Resource.name,null,!0)}return p(t,e),t.prototype.transform=function(e,t){if(Array.isArray(e))return e;var n=e[E.Universe10.Resource.properties.relativeUri.name];if(n){for(var r=n.trim().split("/");r.length>0&&0==r[0].length;)r.shift();e.relativeUriPathSegments=r,e.absoluteUri=b.absoluteUri(t.asElement()),e.completeRelativeUri=b.completeRelativeUri(t.asElement()),A.isResourceType(t.parent().definition())?e.parentUri=b.completeRelativeUri(t.parent()):e.parentUri=""}return e},t}(F),j=function(e){function t(t){void 0===t&&(t={}),e.call(this,E.Universe10.TypeDeclaration.name,null,!0),this.options=t}return p(t,e),t.prototype.transform=function(e,t){var n=Array.isArray(e);if(n&&0==e.length)return e;var r=n?e[0]:e,i=b.typeExample(t.asElement(),this.options.dumpXMLRepresentationOfExamples);if(i)r.examples=[i];else{var a=b.typeExamples(t.asElement(),this.options.dumpXMLRepresentationOfExamples);a.length>0&&(r.examples=a)}if(delete r.example,r.hasOwnProperty("schema")){if(r.hasOwnProperty("type")){var o=r.type;Array.isArray(o)||(o=[o],r.type=o);var s=r.schema;Array.isArray(s)?s.forEach(function(e){return o.push(e)}):o.push(s)}else r.type=r.schema;delete r.schema}if(Array.isArray(r.type)||(r.type=[r.type]),r.mediaType=R,t&&t.isElement()){var u=t.asElement(),l=u.localType().isExternal()?u.localType():null;if(!l)for(var p=0,c=u.localType().allSuperTypes();p<c.length;p++){var f=c[p];f.isExternal()&&(l=f)}if(l){var d=l.external().schema().trim();T.stringStartsWith(d,"<")?r.mediaType="application/xml":r.mediaType="application/json"}}var h=t.property();if(h&&!(A.isHeadersProperty(h)||A.isQueryParametersProperty(h)||A.isUriParametersProperty(h)||A.isPropertiesProperty(h)||A.isBaseUriParametersProperty(h))){delete r.required;var m=r.__METADATA__;if(m){var y=m.primitiveValuesMeta;y&&delete y.required}}var o=r.type;if(o.forEach&&"string"==typeof o[0]){var v=t.asElement().localType();if(v&&v.hasExternalInHierarchy()){var g=o[0].trim(),E="{"===g[0]&&"}"===g[g.length-1],S="<"===g[0]&&">"===g[g.length-1];E?r.typePropertyKind="JSON":S&&(r.typePropertyKind="XML")}else r.typePropertyKind="TYPE_EXPRESSION"}else"object"==typeof o&&(r.typePropertyKind="INPLACE");if(this.options.unfoldTypes&&(r.unfolded=this.processExpressions(r)),1==r.type.length){var _=r.type[0];if("string"==typeof _){_=_.trim();var N=T.stringEndsWith(_,"[]");if(N){for(var w=_.substring(0,_.length-"[]".length).trim();w.length>0&&"("==w.charAt(0)&&")"==w.charAt(w.length-1);)w=w.substring(1,w.length-1);r.type[0]="array",r.items=w}}}return e},t.prototype.processExpressions=function(e){var t=T.deepCopy(e);return this.parseExpressions(t),t},t.prototype.parseExpressions=function(e){if(this.parseExpressionsForProperty(e,"type"),this.parseExpressionsForProperty(e,"items"),e.properties)for(var t=0,n=Object.keys(e.properties);t<n.length;t++){var r=n[t],i=e.properties[r];i.unfolded&&(e.properties[r]=i.unfolded)}},t.prototype.parseExpressionsForProperty=function(e,t){var n=e[t];if(n){var r=!1;if(!Array.isArray(n)){if("object"==typeof n)return void(n.unfolded?e.prop=n.unfolded:this.parseExpressions(n));"string"==typeof n&&(r=!0,n=[n])}for(var i=[],a=0;a<n.length;a++){var o=n[a];if("object"==typeof o&&(o.unfolded?o=o.unfolded:this.parseExpressions(o)),i.push(o),"string"==typeof o){var s=o,u=y.checkExpression(s);if(u){var l={status:y.ParametersEscapingStatus.NOT_REQUIRED};if(o.indexOf("<<")>=0)if(l=y.escapeTemplateParameters(o),l.status==y.ParametersEscapingStatus.OK){if(s=l.resultingString,u=y.checkExpression(s),!u)continue}else if(l.status==y.ParametersEscapingStatus.ERROR)continue;var p=void 0;try{p=g.parse(s)}catch(c){continue}if(p){var f=this.expressionToObject(p,l);null!=f&&(i[a]=f)}}}}e[t]=r?i[0]:i}},t.prototype.expressionToObject=function(e,t){var n,r=0;if("name"==e.type){var i=e;if(r=i.arr,n=i.value,t.status==y.ParametersEscapingStatus.OK){var a=y.unescapeTemplateParameters(n,t.substitutions);a.status==y.ParametersEscapingStatus.OK?n=a.resultingString:a.status==y.ParametersEscapingStatus.ERROR&&(n=null)}}else if("union"==e.type){var o=e;n={type:["union"],options:[]};for(var s=this.toOptionsArray(o),u=0,l=s;u<l.length;u++){var p=l[u];if(null==p){n=null;break}var c=this.expressionToObject(p,t);n.options.push(c)}n.options=N.unique(n.options).sort()}else if("parens"==e.type){var f=e;r=f.arr,n=this.expressionToObject(f.expr,t)}if(null!=n)for(;r-->0;)n={type:["array"],items:n};return n},t.prototype.toOptionsArray=function(e){for(var t,n=e.first,r=e.rest;"parens"==n.type&&0==n.arr;)n=n.expr;for(;"parens"==r.type&&0==r.arr;)r=r.expr;return t="union"==n.type?this.toOptionsArray(n):[n],"union"==r.type?t=t.concat(this.toOptionsArray(r)):t.push(r),t},t}(F),W=function(e){function t(){e.call(this,new B([new U(E.Universe10.TypeDeclaration.name,E.Universe10.LibraryBase.properties.annotationTypes.name,!0,["RAML10"]),new U(E.Universe10.TypeDeclaration.name,E.Universe10.LibraryBase.properties.types.name,!0,["RAML10"]),new U(E.Universe10.Trait.name,E.Universe10.LibraryBase.properties.traits.name,!0,["RAML10"]),new U(E.Universe10.AbstractSecurityScheme.name,E.Universe10.LibraryBase.properties.securitySchemes.name,!0,["RAML10"]),new U(E.Universe10.ResourceType.name,E.Universe10.LibraryBase.properties.resourceTypes.name,!0,["RAML10"])]))}return p(t,e),t.prototype.transform=function(e,t){if(!t.parent()||!t.parent().lowLevel().libProcessed)return e;var n=t.lowLevel(),r=n.key();e.$$name=r;for(var i=n;f.LowLevelProxyNode.isInstance(i);)i=i.originalNode();var a=i.key(),o=e;return o.name=a,o.displayName==r&&(o.displayName=a),e},t}(k),q=function(e){function t(){e.call(this,new B([new U(E.Universe10.ResourceType.name,null,!0),new U(E.Universe10.Trait.name,null,!0),new U(E.Universe10.Method.name,null,!0),new U(E.Universe10.TypeDeclaration.name,null,!0)]))}return p(t,e),t.prototype.transform=function(e){if(Array.isArray(e))return e;var t=d.universesInfo.Universe10.Trait.properties.parametrizedProperties.name,n=e[t];return n&&(Object.keys(n).forEach(function(t){e[t]=n[t]}),delete e[t]),e},t}(k),Y=function(e){function t(){e.call(this,new B([new U(E.Universe10.ObjectTypeDeclaration.name,E.Universe10.ObjectTypeDeclaration.properties.properties.name,!0)]),"name")}return p(t,e),t}(K),H=function(e){function t(){e.call(this,E.Universe08.GlobalSchema.name,E.Universe08.Api.properties.schemas.name,!0,["RAML08"])}return p(t,e),t.prototype.transform=function(e){if(Array.isArray(e))return e;var t={};return t[e.key]=e.value,t},t}(F),$=function(e){function t(){e.call(this,new B([new U(E.Universe10.Api.name,E.Universe10.Api.properties.protocols.name,!0),new U(E.Universe10.MethodBase.name,E.Universe10.MethodBase.properties.protocols.name,!0)]))}return p(t,e),t.prototype.transform=function(e){return"string"==typeof e?e.toUpperCase():Array.isArray(e)?e.map(function(e){return e.toUpperCase()}):e},t}(k),G=function(e){function t(){e.call(this,new B([new U(E.Universe10.SecuritySchemeRef.name,E.Universe10.Api.properties.securedBy.name,!0),new U(E.Universe10.TraitRef.name,E.Universe10.MethodBase.properties.is.name,!0),new U(E.Universe10.ResourceTypeRef.name,E.Universe10.ResourceBase.properties.type.name,!0)]))}return p(t,e),t.prototype.transform=function(e){return e?Array.isArray(e)?e:this.toSimpleValue(e):null},t.prototype.toSimpleValue=function(e){if("string"==typeof e)return e;var t=e.name,n=e.structuredValue;if(n){var r={};return r[t]=n,r}return t},t}(k),X=function(e){function t(t){void 0===t&&(t=!1),e.call(this,new B([new U(E.Universe10.Api.name,null,!0)])),this.enabled=t}return p(t,e),t.prototype.match=function(t,n){return this.enabled?e.prototype.match.call(this,t,n):!1},t.prototype.registrationInfo=function(){return this.enabled?e.prototype.registrationInfo.call(this):null},t.prototype.transform=function(e,n,r){var i=this,a=r,o=e[t.uriParamsPropName];if(o&&(a=[].concat(r||[]),Object.keys(o).forEach(function(e){var t=o[e];Array.isArray(t)?t.forEach(function(e){return a.push(e)}):a.push(t)})),a){e.allUriParameters=a;var s=e[t.methodsPropName];s&&Object.keys(s).forEach(function(e){return s[e].allUriParameters=a})}var u=e[t.resourcesPropName];return u&&u.forEach(function(e){return i.transform(e,null,a)}),e},t.uriParamsPropName=E.Universe10.ResourceBase.properties.uriParameters.name,t.methodsPropName=E.Universe10.ResourceBase.properties.methods.name,t.resourcesPropName=E.Universe10.Api.properties.resources.name,t}(k),z=function(e){function t(){e.call(this,new B([new U(E.Universe10.ResourceBase.name,E.Universe10.ResourceBase.properties.methods.name,!0),new U(E.Universe08.Resource.name,E.Universe08.Resource.properties.methods.name,!0),new U(E.Universe08.ResourceType.name,E.Universe08.ResourceType.properties.methods.name,!0)]),"method")}return p(t,e),t}(K),J=function(e){function t(){e.call(this,new B([new U(E.Universe10.LibraryBase.name,E.Universe10.LibraryBase.properties.types.name,!0),new U(E.Universe10.LibraryBase.name,E.Universe10.LibraryBase.properties.schemas.name,!0),new U(E.Universe10.LibraryBase.name,E.Universe10.LibraryBase.properties.annotationTypes.name,!0)]),"name")}return p(t,e),t}(K),Q=function(e){function t(){e.call(this,new B([new U(E.Universe10.LibraryBase.name,E.Universe10.LibraryBase.properties.traits.name,!0),new U(E.Universe08.Api.name,E.Universe08.Api.properties.traits.name,!0)]),"name")}return p(t,e),t}(K),Z=function(e){function t(){e.call(this,new B([new U(E.Universe10.LibraryBase.name,E.Universe10.LibraryBase.properties.resourceTypes.name,!0),new U(E.Universe08.Api.name,E.Universe10.Api.properties.resourceTypes.name,!0,["RAML08"])]),"name")}return p(t,e),t}(K),ee=function(e){function t(){e.call(this,new B([new U(E.Universe10.LibraryBase.name,E.Universe10.LibraryBase.properties.securitySchemes.name,!0),new U(E.Universe08.Api.name,E.Universe08.Api.properties.securitySchemes.name,!0,["RAML08"])]),"name")}return p(t,e),t}(K),te=function(e){function t(){e.call(this,new B([new U(E.Universe10.Api.name,E.Universe10.Api.properties.baseUriParameters.name,!0),new U(E.Universe10.ResourceBase.name,E.Universe10.ResourceBase.properties.uriParameters.name,!0),new U(E.Universe08.Resource.name,E.Universe08.Resource.properties.uriParameters.name,!0,["RAML08"]),new U(E.Universe10.ResourceBase.name,E.Universe10.MethodBase.properties.queryParameters.name,!0),new U(E.Universe10.MethodBase.name,E.Universe10.MethodBase.properties.queryParameters.name,!0),new U(E.Universe10.Operation.name,E.Universe10.MethodBase.properties.queryParameters.name,!0),new U(E.Universe10.Operation.name,E.Universe10.MethodBase.properties.headers.name,!0),new U(E.Universe10.MethodBase.name,E.Universe10.MethodBase.properties.headers.name,!0),new U(E.Universe08.BodyLike.name,E.Universe08.BodyLike.properties.formParameters.name)]),"name")}return p(t,e),t}(K),ne=function(e){function t(){e.call(this,new B([new U(E.Universe10.MethodBase.name,E.Universe10.MethodBase.properties.responses.name,!0)]),"code")}return p(t,e),t}(K),re=function(e){function t(){e.call(this,new B([new U(E.Universe10.Annotable.name,E.Universe10.Annotable.properties.annotations.name,!0)]),"name")}return p(t,e),t}(K),ie=function(e){function t(){e.call(this,new B([new U(E.Universe10.Response.name,E.Universe10.Response.properties.body.name),new U(E.Universe10.MethodBase.name,E.Universe10.MethodBase.properties.body.name,!0)]),"name")}return p(t,e),t}(K),ae=function(e){function t(){e.call(this,new B([new U(E.Universe10.TypeDeclaration.name,E.Universe10.TypeDeclaration.properties.facets.name,!0)]),"name")}return p(t,e),t}(K),oe=function(e){function t(){e.call(this,new B([new U(E.Universe10.LibraryBase.name,null,!0,["RAML10"])]))}return p(t,e),t.prototype.transform=function(e,t){if(!e)return e;if(!e.hasOwnProperty("schemas"))return e;var n=e.schemas;if(e.hasOwnProperty("types")){var r=e.types;Object.keys(n).forEach(function(e){r.hasOwnProperty(e)||(r[e]=n[e])})}else e.types=n;return delete e.schemas,e},t}(k);t.mergeRegInfos=u},function(e,t,n){"use strict";function r(e){var t={};return e.forEach(function(e){return Object.keys(e).forEach(function(n){return t[n]=e[n]})}),t}function i(e,t){return new N(_.find(e||[],t))}function a(e){return Object.keys(e).map(function(t){return[t,e[t]]})}function o(e){var t={};return e.forEach(function(e){return t[e[0]]=e[1]}),t}function s(e,t){return t(e),e}function u(e,t){"object"==typeof e&&Object.keys(e).forEach(function(n){return t(n,e[n])})}function l(e,n,r){void 0===r&&(r=!1);var i={};return e.forEach(function(e){var a=t.shallowCopy(e);r&&delete a[n],i[e[n]]=a}),i}function p(e,t){var n=e.length-t.length;return n>=0&&e.lastIndexOf(t)===n}function c(e,t,n){return void 0===n&&(n=0),e.length-t.length>=n&&e.substring(n,n+t.length)===t}function f(e){return"_"==e[e.length-1]}function d(e,t,n){var r,i=!1;e[t]=function(){return i||(i=!0,r=n.apply(e)),r}}function h(e,n){void 0===n&&(n=f);for(var r in e)n(r)&&t.ifInstanceOf(e[r],Function,function(t){return 0===t.length?d(e,r,t):null})}function m(e,t){void 0!==e&&t(e)}function y(e){return"string"!=typeof e||""==e?!1:p(e,".raml")}function v(e){for(var t,n=[],r=new RegExp("require\\('([^']+)'\\)","gi");t=r.exec(e);)n.push(t[1]);for(var i=new RegExp('require\\("([^"]+)"\\)',"gi");t=i.exec(e);)n.push(t[1]);return n=_.unique(n).filter(function(e){return""!=e}),n.sort(),n}function g(e){return"undefined"!=typeof e&&null!=e}function A(e){return 0==e.length?e:e.charAt(0).toUpperCase()+e.substr(1)}function E(e,t,n){void 0===n&&(n=!1);var r=Object.keys(t);if(n){var i={};r.forEach(function(e){return i[e]=!0}),Object.keys(e).forEach(function(e){return i[e]=!0}),r=Object.keys(i)}r.forEach(function(n){var r=e[n];r instanceof Object?(t[n]||(t[n]={}),E(r,t[n],!0)):void 0!=r&&(t[n]=e[n])})}function T(e,t){return Object.keys(t).forEach(function(n){return e=S(e,n,t[n])}),e}function S(e,t,n){for(var r="",i=0,a=e.indexOf(t);a<e.length&&a>=0;a=e.indexOf(t,i))r+=e.substring(i,a),r+=n,i=a+t.length;return r+=e.substring(i,e.length)}function b(e){if(null==e||"object"!=typeof e)return e;var t;if(Array.isArray(e)){t=[];for(var n=0,r=e;n<r.length;n++){var i=r[n];t.push(b(i))}}else{t={};for(var a=0,o=Object.keys(e);a<o.length;a++){var s=o[a];t[s]=b(e[s])}}return t}var _=n(70),N=n(22);t.defined=function(e){return null!==e&&void 0!==e},t.flattenArrayOfObjects=r,t.find=i,t.isInstance=function(e,t){return e instanceof t?[e]:[]},t.ifInstanceOf=function(e,t,n){return e instanceof t?n(e):null},t.toTuples=a,t.fromTuples=o,t.collectInstancesOf=function(e,n){return s([],function(r){return e.forEach(function(e){return t.ifInstanceOf(e,n,function(e){return r.push(e)})})})},t.collectInstancesOfInMap=function(e,t){return Object.keys(e).map(function(t){return[t,e[t]]}).filter(function(e){return e[1]instanceof t}).map(function(e){return e})},t.asArray=function(e){return t.defined(e)?e instanceof Array?e:[e]:[]},t.shallowCopy=function(e){return s({},function(t){return Object.keys(e).forEach(function(n){return t[n]=e[n]})})},t.flatMap=function(e,n){return t.flatten(e.map(n))},t.flatten=function(e){return Array.prototype.concat.apply([],e)},t.takeWhile=function(e,t){return s([],function(n){for(var r=0;r<e.length&&t(e[r]);r++)n.push(e[r])})},t.tap=s,t.kv=u,t.indexed=l,t.stringEndsWith=p,t.stringStartsWith=c,t.lazypropkeyfilter=f,t.lazyprops=h,t.iff=m,t.isRAMLUrl=y,t.getAllRequiredExternalModulesFromCode=v,t.serial=function(){var e=0;return function(){return e++}}(),t.isEssential=g,t.firstToUpper=A,t.updateObject=E,t.replaceMap=T,t.replace=S,t.deepCopy=b},function(e,t,n){"use strict";function r(e){return e.highLevel().localType()}function i(e){return e.highLevel().localType()}function a(e){var t=new Pe.Project(De.dirname(e)),n=t.unit(De.basename(e));return n&&n.isRAMLUnit()?Ae.fromUnit(n).wrapperNode():null}function o(e){var t="",n=e;do e=n,t=e.relativeUri().value()+t,n=e.parent();while(n.definition().key().name==_e.Universe10.Resource.name);return t}function s(e){var t=null!=e.highLevel().reusedNode()?Ie:Re;return t.expandLibrary(e)}function u(e,t){return void 0===t&&(t=!1),t?p(e):l(e)}function l(e){var t=e.highLevel().lowLevel();if(Me.LowLevelProxyNode.isInstance(t))return e;var n=null!=e.highLevel().reusedNode()?Ie:Re;return n.expandTraitsAndResourceTypes(e)}function p(e){var t=null!=e.highLevel().reusedNode()?Ie:Re;return t.expandLibraries(e)}function c(e){var t="",n=e;do e=n,t=e.relativeUri().value()+t,n=e.parent();while(n.definition().key().name==_e.Universe10.Resource.name);t=t.replace(/\/\//g,"/");var r=n.baseUri(),i=r?r.value():"";return i=i?i:"",we.stringEndsWith(i,"/")&&(t=t.substring(1)),t=i+t}function f(e,t){return e.runtimeType().validate(t).map(function(e){return e.getMessage()})}function d(e,t){return e.runtimeType().validate(t)}function h(e){return m(e)}function m(e){return e.highLevel().lowLevel().actual().libExpanded?e.traits_original():g(e,function(e){return xe.isTraitType(e)})}function y(e){return v(e)}function v(e){return e.highLevel().lowLevel().actual().libExpanded?e.resourceTypes_original():g(e,function(e){return xe.isResourceTypeType(e)})}function g(e,t){var n=Le.globalDeclarations(e.highLevel()).filter(function(e){return t(e.definition())}),r=e.highLevel().lowLevel(),i=r.includePath();i||(i=r.unit().path());var a=Me.LowLevelProxyNode.isInstance(e.highLevel().lowLevel()),o=a?new Re.TraitsAndResourceTypesExpander:null,s=n.map(function(t){var n,r=t.lowLevel().unit().path();return a&&(Me.LowLevelProxyNode.isInstance(t.lowLevel())||(t=o.createHighLevelNode(t,!1)),(new Ce.ReferencePatcher).process(t,e.highLevel(),!0,!0)),r!=i?(n=me.buildWrapperNode(t,!1),n.meta().setCalculated()):n=t.wrapperNode(),n});return s}function A(e){var t=[],n=e;do e=n,t.push(e.relativeUri().value()),n=e.parent();while(n.definition().key().name==_e.Universe10.Resource.name);return t.reverse()}function E(e){return he.ResourceImpl.isInstance(e.parent())?e.parent():null}function T(e){var t=e.parent();return t.definition().key().name==_e.Universe10.Resource.name?t:null}function S(e,t){if(null==e)return null;var n=e.resources();return n?(n=n.filter(function(e){return e.relativeUri().value()==t}),0==n.length?null:n[0]):null}function b(e,t){if(!e)return null;for(var n=null,r=0;r<t.length;r++){if(n=S(e,t[r]),!n)return null;e=n}return n}function _(e,t){return e?e.methods().filter(function(e){return e.method()==t}):null}function N(e,t,n){var r=b(e,t);return r?_(r,n):null}function w(e){return xe.isApiSibling(e.definition())}function R(e){for(var t=e;!w(t);)t=t.parent();return t}function I(e){var t=e.parent();if(he.ResourceImpl.isInstance(t))return o(t)+" "+e.method().toLowerCase();if(he.ResourceTypeImpl.isInstance(t))return t.name()+" "+e.method().toLowerCase();throw new Error("Method is supposed to be owned by Resource or ResourceType.\nHere the method is owned by "+e.definition().key().name)}function M(e){var t=e.code().value(),n=Ee.validateResponseString(t);if(null!=n)return!1;try{if(parseInt(t.charAt(0))<4)return!0}catch(r){}return!1}function C(e){var t=[],n=function(e){t.push(e),e.resources().forEach(function(e){return n(e)})};return e.resources().forEach(function(e){return n(e)}),t}function L(e,t){for(var n={};null!=t;)O(t).forEach(function(e){return n[e.name()]=new Ke(e)}),t=T(t);var r=ve.ramlPathMatch(o(t),n,{})(e);return r?new Ne(Object.keys(r.params).map(function(e){return new Be(e,r.params[e])})):Ne.empty()}function P(e){return O(e)}function O(e){var t=e.uriParameters_original();if(!he.ResourceImpl.isInstance(e))return t;var n=e.relativeUri().value(),r=_e.Universe10.ResourceBase.properties.uriParameters.name;return pe(t,n,e,r)}function D(e){return x(e)}function x(e){var t=e.baseUri()?e.baseUri().value():"",n=e.baseUriParameters_original(),r=_e.Universe10.Api.properties.baseUriParameters.name;return pe(n,t,e,r)}function U(e){var t=[],n=e;do e=n,t=O(e).concat(t),n=e.parent();while(n.definition().key().name==_e.Universe10.Resource.name);var r=n;r.baseUriParameters();return t=x(r).concat(t)}function k(e){return F(e)}function F(e){return e.protocols().map(function(e){return e.toUpperCase()})}function B(e){return K(e)}function K(e){return e.securedBy()}function V(e){var t=e.highLevel();if(!t)return"";var n=t.value();return n?n.toString():""}function j(e){var t=e.highLevel();if(!t)return null;var n=Le.findDeclarationByNode(t,Le.LocationKind.VALUE_COMPLETION);if(!n)return null;if(!n.getKind||n.getKind()!=ge.NodeKind.NODE)return null;var r=n.wrapperNode();return he.AbstractSecuritySchemeImpl.isInstance(r)?r:null}function W(e){return e.highLevel().definition().universe().version()}function q(e){var t=e.value().lowLevel(),n=null,r=t.highLevelParseResult();if(r){var i=null,a=!1;if(r.isAttr()?(a=xe.isAnnotationsProperty(r.property()),i=r.parent().types()):r.isElement()&&(i=r.asElement().types()),i){var o=e.name(),s=o.indexOf(".")>=0;if(s){var u=a?i.getAnnotationTypeRegistry():i.getTypeRegistry();n=u.get(o)}else n=a?i.getAnnotationType(o):i.getType(o)}}return new ye.TypeInstanceImpl(t,n)}function Y(e){var t=e.highLevel().value();return"string"==typeof t||null==t?t:Ae.StructuredValue.isInstance(t)?t.valueName():null}function H(e){return X(e)}function $(e){return X(e)}function G(e){return X(e)}function X(e){var t=e.highLevel(),n=t.parent(),r=e.name(),i=Le.referenceTargets(t.property(),n).filter(function(e){return Ae.qName(e,n)==r});return 0==i.length?null:i[0].wrapperNode()}function z(e,t,n,r){void 0===r&&(r=!0);var i=t.lowLevel(),a=t.definition().property(n?"example":"examples"),o=Se.getUniverse("RAML10"),s=o.type(be.Universe10.ExampleSpec.name),u=e.examples().filter(function(e){return null!=e&&!e.isEmpty()&&e.isSingle()==n});return u.map(function(e){var n=e.asJSON(),o=e.isSingle()?"example":null,u=i.unit(),l=new Oe.AstNode(u,n,i,null,o),p=r?new Ae.ASTNodeImpl(l,t,s,a):t,c=e.annotations(),f=ke(c,l,p,u),d=e.scalarsAnnotations(),h={};Object.keys(d).forEach(function(e){return h[e]=ke(d[e],l,p,u)});var m=new Ve(p,e,f,{description:function(){return h.description||[]},displayName:function(){return h.displayName||[]},strict:function(){return h.strict||[]}});return m})}function J(e,t){void 0===t&&(t=!1);var n=e.runtimeDefinition();if(!n)return[];var r=e.highLevel();return z(n,r,t)}function Q(e){var t=J(e,!0);return t.length>0?t[0]:null}function Z(e){return J(e)}function ee(e){var t=e.runtimeDefinition(),n=t.fixedFacets();if(e.kind()==be.Universe10.UnionTypeDeclaration.name)for(var r=t.allFixedBuiltInFacets(),i=0,a=Object.keys(r);i<a.length;i++){var o=a[i];n[o]=r[o]}else for(var s=Object.keys(n),u=0,l=s;u<l.length;u++){var o=l[u];null==t.facet(o)&&delete n[o]}if(0==Object.keys(n).length)return null;var p=new Oe.AstNode(null,n);return new ye.TypeInstanceImpl(p)}function te(e){var t=e.highLevel().attributes(Se.universesInfo.Universe10.TypeDeclaration.properties.type.name),n=t.filter(function(e){return Ae.StructuredValue.isInstance(e.value())});if(0==n.length)return e.type_original().map(function(e){
return null===e||"NULL"===e||"Null"===e?"string":e});var r=!1,i=t.map(function(e){var t=e.value();return null==t?null:"string"==typeof t?t:(Ae.StructuredValue.isInstance(t)&&(r=!0),t.toString())});return r?null:i}function ne(e){var t=!1,n=e.highLevel().attributes(Se.universesInfo.Universe10.TypeDeclaration.properties.schema.name);if(t)return null;var r=n.filter(function(e){return Ae.StructuredValue.isInstance(e.value())});if(0==r.length)return e.schema_original();var i=n.map(function(e){var n=e.value();return"string"==typeof n?n:(Ae.StructuredValue.isInstance(n)&&(t=!0),n.toString())});return t?null:i}function re(e){var t=e.highLevel().attributes(Se.universesInfo.Universe10.TypeDeclaration.properties.type.name);t=t.concat(e.highLevel().attributes(Se.universesInfo.Universe10.TypeDeclaration.properties.schema.name));for(var n=t.map(function(e){return e.value()}),r=0,i=n;r<i.length;r++){var a=i[r];if(Ae.StructuredValue.isInstance(a)){var o=new ye.TypeInstanceImpl(a.lowLevel());return o}}return null}function ie(e){for(var t=e.highLevel().attributes(Se.universesInfo.Universe10.ArrayTypeDeclaration.properties.items.name),n=t.map(function(e){return e.value()}),r=0,i=n;r<i.length;r++){var a=i[r];if(Ae.StructuredValue.isInstance(a)){var o=new ye.TypeInstanceImpl(a.lowLevel());return o}}return null}function ae(e){var t=e.value(),n=e.highLevel().lowLevel().unit().resolve(t),r=n.highLevel(),i=r.asElement();if(i){var a=i.wrapperNode();return de.isLibrary(a)?(a.setAttributeDefaults(e.attributeDefaults()),a):null}return null}function oe(e){var t=e.highLevel().attributes(Se.universesInfo.Universe10.ArrayTypeDeclaration.properties.items.name),n=t.filter(function(e){return Ae.StructuredValue.isInstance(e.value())});if(0==n.length)return e.items_original().map(function(e){return null===e||"NULL"===e||"Null"===e?"string":e});var r=!1,i=t.map(function(e){var t=e.value();return null==t?null:"string"==typeof t?t:(Ae.StructuredValue.isInstance(t)&&(r=!0),t.toString())});return r?null:i}function se(e){var t=e.highLevel();if(!t)return null;var n=t.attr(_e.Universe10.TypeDeclaration.properties.type.name);if(null==n)return null;var r=n.lowLevel();if(null==r)return null;var i=n.value();if(!i||"string"!=typeof i)return null;var a=r.end()-i.length+1,o=n.lowLevel().unit();if(!o)return null;var s=Le.findDeclaration(o,a);return s&&s.getKind&&s.getKind()==ge.NodeKind.NODE&&xe.isTypeDeclarationSibling(s.definition())?s.wrapperNode():null}function ue(e){var t=e.runtimeType();if(!t)return null;if(!t.isArray()||!t.componentType)return null;var n=t,r=n.componentType();if(!r)return null;var i=Le.getNominalTypeSource(r);if(!i)return null;var a=i.getSource();if(!a)return null;if(!a.isElement())return null;if(!xe.isTypeDeclarationSibling(a.definition()))return null;var o=a.wrapperNode();return o}function le(e){var t=e.items_original();if(t&&"string"!=typeof t&&(!Array.isArray(t)||0!=t.length)){var n=e.highLevel(),r=be.Universe10.ArrayTypeDeclaration.properties.items.name,i=n.attr(r),a=n.definition().universe().type(be.Universe10.TypeDeclaration.name),o=n.definition().universe().type(be.Universe10.ArrayTypeDeclaration.name),s=new Ae.ASTNodeImpl(i.lowLevel(),n,a,o.property(r));s.patchType(Ue.doDescrimination(s))}var u=ue(e);return u?u:se(e)}function pe(e,t,n,r){if("string"!=typeof t)return[];var i=n.highLevel(),a=i.definition(),o=a.property(r);if(!t)return[];var s={};e.forEach(function(e){var t=s[e.name()];t||(t=[],s[e.name()]=t),t.push(e)});for(var u=[],l=0,p={},c=t.indexOf("{");c>=0&&(l=t.indexOf("}",++c),!(0>l));c=t.indexOf("{",l)){var f=t.substring(c,l);if(p[f]=!0,s[f])s[f].forEach(function(e){return u.push(e)});else{var d=a.universe(),h=d.type(be.Universe10.StringTypeDeclaration.name),m=Te.createStubNode(h,null,f,i.lowLevel().unit()),y=me.buildWrapperNode(m),v=y.highLevel();v.setParent(i),y.meta().setCalculated(),y.setName(f),v.patchProp(o),u.push(y)}}return Object.keys(s).filter(function(e){return!p[e]}).forEach(function(e){return s[e].forEach(function(e){return u.push(e)})}),u}function ce(e){if(e.kind()==be.Universe10.Method.name||xe.isTypeDeclarationSibling(e.definition())){for(var t=!1,n=e.highLevel().parent();null!=n;){var r=n.definition();if(xe.isResourceTypeType(r)||xe.isTraitType(r)){t=!0;break}n=n.parent()}if(!t)return null}var i=e.highLevel();if(null==i)return null;var a=i.lowLevel();if(null==a)return null;var o=a.children().filter(function(e){var t=e.key();return t?"("==t.charAt(0)&&")"==t.charAt(t.length-1)?!1:t.indexOf("<<")>=0:!1});if(0==o.length)return null;var s=new ye.TypeInstanceImpl(o);return s}var fe=this&&this.__extends||function(e,t){function n(){this.constructor=e}for(var r in t)t.hasOwnProperty(r)&&(e[r]=t[r]);e.prototype=null===t?Object.create(t):(n.prototype=t.prototype,new n)},de=n(11),he=n(21),me=n(47),ye=n(51),ve=n(57),ge=n(9),Ae=n(16),Ee=n(28),Te=n(17),Se=n(39),be=n(18),_e=n(18),Ne=n(22),we=n(33),Re=n(27),Ie=n(30),Me=n(44),Ce=n(53),Le=n(13),Pe=n(24),Oe=n(54),De=n(15),xe=n(14),Ue=n(29);t.resolveType=r,t.runtimeType=i,t.load=a,t.completeRelativeUri=o,t.expandLibrarySpec=s,t.expandSpec=u,t.expandTraitsAndResourceTypes=l,t.expandLibraries=p,t.absoluteUri=c,t.validateInstance=f,t.validateInstanceWithDetailedStatuses=d,t.traitsPrimary=h,t.allTraits=m,t.resourceTypesPrimary=y,t.allResourceTypes=v,t.relativeUriSegments=A,t.parentResource=E,t.parent=T,t.childResource=S,t.getResource=b,t.childMethod=_,t.getMethod=N,t.ownerApi=R,t.methodId=I,t.isOkRange=M,t.allResources=C,t.matchUri=L;t.uriParametersPrimary=P,t.uriParameters=O,t.baseUriParametersPrimary=D,t.baseUriParameters=x,t.absoluteUriParameters=U,t.protocolsPrimary=k,t.allProtocols=F,t.securedByPrimary=B,t.allSecuredBy=K,t.securitySchemeName=V,t.securityScheme=j,t.RAMLVersion=W,t.structuredValue=q,t.referenceName=Y,t.referencedTrait=H,t.referencedAnnotation=$,t.referencedResourceType=G;var ke=function(e,t,n,r){var i=[];if(e)for(var a=Se.getUniverse("RAML10"),o=a.type("Annotable").property("annotations"),s=0,u=Object.keys(e);s<u.length;s++){var l=u[s],p=e[l],c=new Oe.AstNode(r,p.value(),t,null,"("+l+")"),f=new Ae.ASTPropImpl(c,n,o.range(),o),d=new he.AnnotationRefImpl(f);i.push(d)}return i};t.examplesFromNominal=z,t.getTypeExample=Q,t.getTypeExamples=Z,t.typeFixedFacets=ee,t.typeValue=te,t.schemaValue=ne,t.typeStructuredValue=re,t.itemsStructuredValue=ie,t.referencedNode=ae,t.getItems=oe,t.findComponentTypeDeclaration=le,t.getTemplateParametrizedProperties=ce;var Fe=function(){function e(e,t){this._content=e,this._name=t}return e.prototype.name=function(){return this._name},e.prototype.content=function(){return this._content},e}();t.SchemaDef=Fe;var Be=function(){function e(e,t){this.key=e,this.value=t}return e}();t.ParamValue=Be;var Ke=function(){function e(e){this._param=e,this.description=e.description()?e.description().value():this.description,this.displayName=e.displayName(),this.type=e.type().length>0?e.type()[0]:"string",this.example=e.example(),this.required=e.required(),this["default"]=e["default"]()}return e}(),Ve=function(e){function t(t,n,r,i){e.call(this,t),this.expandable=n,this._annotations=r,this._scalarAnnotations=i}return fe(t,e),t.prototype.value=function(){return this.expandable.asString()},t.prototype.structuredValue=function(){var e;e=this.expandable.isJSONString()||this.expandable.isYAML()?this.expandable.asJSON():this.expandable.original();var t=this._node.lowLevel(),n=this.expandable.isSingle()?"example":null,r=new Oe.AstNode(t.unit(),e,t,null,n);return new ye.TypeInstanceImpl(r)},t.prototype.strict=function(){return this.expandable.strict()},t.prototype.description=function(){var e=this.expandable.description();if(null==e&&null!==e)return null;var t=Te.createAttr(this._node.definition().property(be.Universe10.ExampleSpec.properties.description.name),e),n=new he.MarkdownStringImpl(t);return n},t.prototype.name=function(){return this.expandable.name()},t.prototype.displayName=function(){return this.expandable.displayName()},t.prototype.annotations=function(){return this._annotations},t.prototype.scalarsAnnotations=function(){return this._scalarAnnotations},t.prototype.uses=function(){return e.prototype.elements.call(this,"uses")},t}(ye.BasicNodeImpl);t.ExampleSpecImpl=Ve},function(e,t,n){"use strict";function r(e){for(var t=[],n={};null!=e.parent();)e.lowLevel().includePath()&&(t=t.concat(a(e,n))),e=e.parent();return t=t.concat(a(e,n))}function i(e){var t=e.node();if(t&&x.isParseResult(t))return t;var n=V.getNominalPropertySource2(e);return n?n.getSource():null}function a(e,t,n){if(void 0===t&&(t={}),void 0===n&&(n=[]),!e.lowLevel())return[];var r=e.lowLevel().unit().absolutePath();if(t[r]=!0,!F.ASTNodeImpl.isInstance(e))return n;var i=!1;return e.elements().forEach(function(e){if(e.definition().key()==B.Universe10.UsesDeclaration){if(i)return;var r=e.attr("value");if(r){var o=e.root().lowLevel().unit().resolve(r.value());if(o&&j.isWaitingFor(o.absolutePath()))return void(i=!0);null!=o&&o.isRAMLUnit()&&!t[o.absolutePath()]&&o.highLevel().isElement()&&a(o.highLevel().asElement(),t,n)}}else n.push(e)}),n}function o(e,t){for(var n="",r=e-1;r>=0;r--){var i=t.charAt(r);if(" "==i||"	"==i)n?n+=i:n=i;else if("\r"==i||"\n"==i)return n}}function s(e,t,n,r,i){if(void 0===r&&(r=!0),void 0===i&&(i=!0),null==e)return null;if(e.lowLevel()&&e.lowLevel().start()<=t&&e.lowLevel().end()>=n){if(F.ASTNodeImpl.isInstance(e)){for(var a=e,o=r?a.children():a.directChildren(),u=0;u<o.length;u++)if(r||o[u].lowLevel().unit()==e.lowLevel().unit()){var l=s(o[u],t,n,r);if(l)return!i&&F.ASTPropImpl.isInstance(l)&&(l=l.parent()),l}return e}if(F.ASTPropImpl.isInstance(e)){var p=e;if(!p.property().isKey()){var c=p.value();if(F.StructuredValue.isInstance(c)){var f=c,d=f.toHighLevel2();if(d&&!r&&d.lowLevel().unit()!=e.lowLevel().unit())return null;var l=s(d,t,n,r);if(l)return!i&&F.ASTPropImpl.isInstance(l)&&(l=l.parent()),l}return i?p:p.parent()}return null}return null}return null}function u(e,t){for(var n=-1,r=t-1;r>=0;r--){var i=e.charAt(r);if("\r"==i||"\n"==i||" "==i||"	"==i||'"'==i||"'"==i||":"==i){n=r+1;break}}for(var a=-1,r=t;r<e.length;r++){var i=e.charAt(r);if("\r"==i||"\n"==i||" "==i||"	"==i||'"'==i||"'"==i||":"==i){a=r;break}}return-1!=n&&-1!=a?e.substring(n,a):""}function l(e,t){for(var n="",r=t;r>=0;r--){var i=e[r];if(" "==i||"\r"==i||"\n"==i||"|"==i||"["==i||"]"==i||":"==i||"("==i||")"==i)break;n=i+n}for(var r=t+1;r<e.length;r++){var i=e[r];if(" "==i||"\r"==i||"\n"==i||"|"==i||"["==i||"]"==i||":"==i||"("==i||")"==i||","==i)break;n+=i}return n}function p(e,t){var n=m(e,t);if(n){if(F.ASTNodeImpl.isInstance(n)){var r=n;return{node:r,results:r.findReferences()}}F.ASTPropImpl.isInstance(n)}var i=s(F.fromUnit(e),t,t,!1);if(F.ASTNodeImpl.isInstance(i))return{node:i,results:i.findReferences()};if(F.ASTPropImpl.isInstance(i)){var a=i;if(a.property().canBeValue())return{node:a.parent(),results:a.parent().findReferences()}}return{node:null,results:[]}}function c(e,t){var n=e;if(n.getAdapter(K.RAMLPropertyService).isTypeExpr()){var i=r(t).filter(function(e){var t=e.definition().key();return t===B.Universe08.GlobalSchema?!0:e.definition().isAssignableFrom(B.Universe10.TypeDeclaration.name)});return i}if(n.getAdapter(K.RAMLPropertyService).isDescriminating()){var a=w(n.range(),t);return a}if(n.isReference()){var o=n.referencesTo(),a=w(o,t);return a}if(n.range().hasValueTypeInHierarchy()){var s=n.range().getAdapter(K.RAMLService);if(s.globallyDeclaredBy().length>0){var i=r(t).filter(function(e){return null!=U.find(s.globallyDeclaredBy(),function(t){return t==e.definition()})});return i}}return[]}function f(e,t){if(t){var n=[];if(e.getAdapter(K.RAMLPropertyService).isTypeExpr()){var i=r(t).filter(function(e){var t=e.definition().key();return t===B.Universe08.GlobalSchema?!0:e.definition().isAssignableFrom(B.Universe10.TypeDeclaration.name)});n=i.map(function(e){return F.qName(e,t)});var a=t.definition().universe().type(B.Universe10.TypeDeclaration.name);if(a){var o=a.allSubTypes();n=n.concat(o.map(function(e){return e.getAdapter(K.RAMLService).descriminatorValue()}))}return n}var s=e.range().key();if((s==B.Universe08.SchemaString||s==B.Universe10.SchemaString)&&"RAML10"==e.range().universe().version()&&e.range().hasValueTypeInHierarchy()){var i=r(t).filter(function(e){return e.definition().isAssignableFrom(B.Universe10.TypeDeclaration.name)});n=i.map(function(e){return F.qName(e,t)})}if(e.getAdapter(K.RAMLPropertyService).isDescriminating()){var o=b(e.domain(),t);n=n.concat(o.map(function(e){return e.getAdapter(K.RAMLService).descriminatorValue()}))}else if(e.isReference()){var u=w(e.referencesTo(),t);n=u.map(function(e){return F.qName(e,t)})}else if(e.range().hasValueTypeInHierarchy()){var l=e.range().getAdapter(K.RAMLService);if(l.globallyDeclaredBy().length>0){var i=r(t).filter(function(e){return null!=U.find(l.globallyDeclaredBy(),function(t){return t==e.definition()})});n=n.concat(i.map(function(e){return F.qName(e,t)}))}}return e.isAllowNull()&&n.push("null"),e.enumOptions()&&(n=n.concat(e.enumOptions())),n}return e.enumOptions()&&"string"==typeof e.enumOptions()?[e.enumOptions()+""]:e.enumOptions()}function d(e){return e.isElement()&&e.asElement().definition().key()!=B.Universe10.Library?null:e.asElement().attrValue("name")}function h(e,t){var n=e.lowLevel().unit();if(!n)return null;var r=e.lowLevel().start(),i=e.lowLevel().end();if(null!=t&&t==q.KEY_COMPLETION?(r=e.lowLevel().keyStart(),i=e.lowLevel().keyEnd()):null!=t&&t==q.VALUE_COMPLETION&&(r=e.lowLevel().valueStart(),i=e.lowLevel().valueEnd()),-1==r||-1==i)return null;var a=Math.floor((r+i)/2);return m(n,a,t)}function m(e,t,n){var r=s(F.fromUnit(e),t,t,!1),a=null;if(r.isElement()&&r.asElement().definition().isAssignableFrom(B.Universe10.TypeDeclaration.name)&&r.asElement().directChildren().forEach(function(e){if(e.isUnknown()&&e.getLowLevelStart()<t&&e.getLowLevelEnd()>t){var n=r.asElement().localType();n.allFacets().forEach(function(t){if(t.nameId()==e.lowLevel().key()&&k.UserDefinedProp.isInstance(t)){var n=i(t);a=n}})}}),!r.property())return r;if("example"==r.property().nameId()){r.parent().localType();r.lowLevel().children().forEach(function(e){"example"==e.key()&&e.children().forEach(function(e){if(e.start()<t&&e.end()>t){var n=r.parent().asElement().localType();n.allProperties().forEach(function(t){if(t.nameId()==e.key()&&k.UserDefinedProp.isInstance(t)){var n=i(t);a=n}})}})})}if(a)return a;var o=null!=n?n:T(e.contents(),t);if(o==q.VALUE_COMPLETION){var l=r;if(F.ASTPropImpl.isInstance(r)){var p=r;if(p&&p.value()){if(!F.StructuredValue.isInstance(p.value()))return W(t,e.contents(),p,l);var c=p.value(),f=c.toHighLevel();if(f){var d=U.find(f.attrs(),function(e){return e.lowLevel().start()<t&&e.lowLevel().end()>=t});if(d)return W(t,e.contents(),d,f,p.property())}}}else{var h=l.property();if(h)return W(t,e.contents(),null,l,h)}}if(o==q.KEY_COMPLETION||o==q.SEQUENCE_KEY_COPLETION){var l=r,m=r.property();if(D.UserDefinedProp.isInstance(m)){var g=m;return i(g)}if(F.ASTNodeImpl.isInstance(r)&&D.isUserDefinedClass(l.definition())){var A=l.definition();return A.isAssignableFrom("TypeDeclaration")?r:A.getAdapter(K.RAMLService).getDeclaringNode()}if(F.ASTPropImpl.isInstance(r)){var S=r;if(E(S)){var b=y(S);if(b){var _=v(S,b);if(_){var r=s(_,t,t);if(m=r.property(),D.UserDefinedProp.isInstance(m)){var g=m;return i(g)}if(F.ASTNodeImpl.isInstance(r)&&D.isUserDefinedClass(l.definition())){var A=l.definition();return A.getAdapter(K.RAMLService).getDeclaringNode()}}}}}}if(o==q.PATH_COMPLETION){var N=u(e.contents(),t);if(N){var w=e.resolve(N);return w}}}function y(e){var t=null;if(e.isElement()?t=e:e.isAttr()&&(t=e.parent()),!t.definition().isAssignableFrom(B.Universe10.TypeDeclaration.name)){var n=t.parent();if(!n)return null;if(n.definition().isAssignableFrom(B.Universe10.TypeDeclaration.name))t=n;else{if(n=n.parent(),null==n)return null;if(!n.definition().isAssignableFrom(B.Universe10.TypeDeclaration.name))return null;t=n}}return t.localType()}function v(e,t){return F.StructuredValue.isInstance(e.value())?new F.ASTNodeImpl(e.value().lowLevel(),e.parent(),t,e.property()):null}function g(e,t){return new F.ASTNodeImpl(e.lowLevel(),e,t,e.property())}function A(e){return e.definition().key()==B.Universe10.ExampleSpec}function E(e){var t=B.Universe10.TypeDeclaration.properties.example.name,n=B.Universe10.ObjectTypeDeclaration.name;if(!F.ASTPropImpl.isInstance(e))return!1;var r=e,i=r.parent(),a=i&&i.property();a&&a.nameId();return t===r.name()&&r.isString()&&F.ASTNodeImpl.isInstance(i)&&i.definition().isAssignableFrom(n)?!0:!1}function T(e,t){for(var n=!1,r=!1,i=!1,a=t-1;a>=0;a--){var s=e.charAt(a);if("("==s)i=!0;else{if(i){if("\r"==s||"\n"==s){for(var u=!1,l=t-1;l<e.length;l++){var p=e[l];if(")"==p&&(u=!0),"\r"==p||"\n"==p)break;if(":"==p){i=!1;break}}if(i&&u)return q.ANNOTATION_COMPLETION;break}if(" "==s||"	"==s)continue;break}if("\r"==s||"\n"==s)break;if(":"==s)break}}for(var a=t-1;a>=0;a--){var s=e.charAt(a);if("#"==s){if(0==a)return q.VERSION_COMPLETION;for(var l=a-1;l>=0;l--){var c=e.charAt(l);if("\r"==c||"\n"==c)break;if("!"==c&&e.indexOf("!include",l)==l)return q.PATH_COMPLETION}return q.INCOMMENT}if(":"==s)return n?q.DIRECTIVE_COMPLETION:q.VALUE_COMPLETION;if("\r"==s||"\n"==s){for(var f=!1,d=o(t,e),h=a;h>0;h--){if(s=e.charAt(h),":"==s){if(f){var m=o(h,e);if(m.length<d.length)return q.VALUE_COMPLETION}break}"|"!=s?(("\r"==s||"\n"==s)&&(f=!1)," "!=s&&"	"!=s&&(f=!1)):f=!0}return r?q.SEQUENCE_KEY_COPLETION:q.KEY_COMPLETION}if("-"==s&&(r=!0),"!"==s){if(e.indexOf("!include",a)==a)return q.PATH_COMPLETION;e.indexOf("!i",a)==a&&(n=!0)}}}function S(e,t){if(!t)return null;for(var n=t.split("/"),r=e,i=0;i<n.length;i++)if("#"!=n[i]){if(r=U.find(r.children(),function(e){return e.key()==n[i]}),!r)return null}else r=e.unit().ast();return r}function b(e,n){if(null==e)return[];var i=e.nameId();n=t.declRoot(n);var a=n;if(a._subTypesCache){var o=a._subTypesCache[i];if(o)return o}else a._subTypesCache={};var s=e.allSubTypes();if(e.getAdapter(K.RAMLService).getRuntimeExtenders().length>0&&n){var u=r(n),l=e.getAdapter(K.RAMLService).getRuntimeExtenders();n.root();l.forEach(function(t){var n=u.filter(function(n){var r=n.definition().allSuperTypes();r.push(n.definition());var i=n.definition()==t||null!=U.find(r,function(e){return e==t})||null!=U.find(r,function(t){return t==e});return i});s=s.concat(n.map(function(e){return e.localType()}))})}return s=U.unique(s),a._subTypesCache[i]=s,s}function _(e,n,i){n=t.declRoot(n);var a=r(n),o=U.find(a,function(t){return F.qName(t,n)==e&&t.property()&&t.property().nameId()==B.Universe10.LibraryBase.properties.types.name});return o.localType()}function N(e,n,i){n=t.declRoot(n);var a=r(n),o=U.find(a,function(t){return F.qName(t,n)==e&&t.property()&&t.property().nameId()==B.Universe10.LibraryBase.properties.schemas.name});return o.localType()}function w(e,t){var n=[],i=[e].concat(e.getAdapter(K.RAMLService).getRuntimeExtenders());if(t){var a=t;i.forEach(function(e){var t=r(a),i=t.filter(function(t){return t.definition().isAssignableFrom(e.nameId())});n=n.concat(i)})}var o=!e.hasValueTypeInHierarchy();if(o&&e.getAdapter(K.RAMLService).isInlinedTemplates()&&t){var a=t,s=r(a).filter(function(t){return t.definition()==e});n=n.concat(s)}else{var a=t,u={};e.allSubTypes().forEach(function(e){return u[e.nameId()]=!0}),u[e.nameId()]=!0;var s=r(a).filter(function(e){return u[e.definition().nameId()]});n=n.concat(s)}return n}function R(e,t){var n=e.range();return b(n,t)}function I(e,t){if(t){if(e.isDescriminator()){var n=e.range(),i=n.getAdapter(K.RAMLService).getRuntimeExtenders();if(i.length>0&&t){var a=[];return i.forEach(function(e){var n=r(t).filter(function(t){return t.definition()==e});a=a.concat(n)}),a}return[]}if(e.isReference())return w(e.referencesTo(),t);if(e.range().hasValueTypeInHierarchy()){var o=e.range().getAdapter(K.RAMLService);if(o.globallyDeclaredBy&&o.globallyDeclaredBy().length>0){var s=r(t).filter(function(e){return null!=U.find(o.globallyDeclaredBy(),function(t){return t==e.definition()})});return s}}}return[]}function M(e){var t=[];return C(e,t),t}function C(e,t){e.children().forEach(function(e){t.push(e),C(e,t)})}function L(e,t,n){e.elements().forEach(function(e){L(e,t,n);e.definition()}),e.attrs().forEach(function(r){var i=r.property(),a=r.value();if(D.UserDefinedProp.isInstance(i)){var o=i.node();o==t?n.push(r):o.lowLevel().start()==t.lowLevel().start()&&o.lowLevel().unit()==t.lowLevel().unit()&&n.push(r)}if(E(r)){var s=y(r);if(s){var u=v(r,s);u&&L(u,t,n)}}else if(i.getAdapter(K.RAMLPropertyService).isTypeExpr()&&"string"==typeof a){var l=e.localType();Y(l,r,t,n);var p=d(t);if(p&&-1!=a.indexOf(p)){var c=P(r);c&&c.lowLevel().start()==t.lowLevel().start()&&n.push(r)}}if(i.isReference()||i.isDescriminator()){if("string"==typeof a){var f=I(i,e);U.find(f,function(e){return e.name()==a&&e==t})&&n.push(r);var p=d(t);if(p&&-1!=a.indexOf(p)){var c=P(r);c&&c.lowLevel().start()==t.lowLevel().start()&&n.push(r)}}else if(F.StructuredValue.isInstance(a)){var h=a;if(h){var m=h.valueName(),f=I(i,e);U.find(f,function(e){return e.name()==m&&e==t})&&n.push(r);var g=h.toHighLevel();g&&L(g,t,n);var p=d(t);if(p&&-1!=m.indexOf(p)){var c=P(g);c&&c.lowLevel().start()==t.lowLevel().start()&&n.push(r)}}}}else{var f=I(i,e);U.find(f,function(e){return e.name()==a&&e==t})&&n.push(r)}})}function P(e){if(!e.lowLevel)return null;var t=e.lowLevel();if(!t)return null;if(t.key()){var n=Math.floor((t.keyEnd()+t.keyStart())/2),r=O(t.unit(),n);if(r)return r}if(t.value()){var n=Math.floor((t.valueEnd()+t.valueStart())/2),r=O(t.unit(),n);if(r)return r}return null}function O(e,t){var n=m(e,t);if(n&&n.isElement&&n.isElement())for(var r=n.asElement(),i=r;i;){if(i.definition().key()==B.Universe10.Library)return i;i=i.parent()}return null}var D=n(39),x=n(9),U=n(70),k=n(39),F=n(16),B=n(18),K=k,V=n(36),j=n(25);t.declRoot=function(e){for(var t=e;;){if(t.definition().key()==B.Universe10.Library)break;var n=t.parent();if(!n)break;t=n}return t},t.globalDeclarations=r,t.findDeclarations=a,t.deepFindNode=s,t.extractName=l;var W=function(e,t,n,r,a){void 0===a&&(a=n.property());var o=c(a,r),s=l(t,e),u=U.find(o,function(e){return F.qName(e,r)==s});if(u)return u;if(D.UserDefinedProp.isInstance(a)){var p=a;return i(p)}return null};t.findUsages=p,t.referenceTargets=c,t.enumValues=f,t.findDeclarationByNode=h,t.findDeclaration=m,t.findExampleContentType=y,t.parseDocumentationContent=v,t.parseStructuredExample=g,t.isExampleNode=A,t.isExampleNodeContent=E,t.determineCompletionKind=T,function(e){e[e.VALUE_COMPLETION=0]="VALUE_COMPLETION",e[e.KEY_COMPLETION=1]="KEY_COMPLETION",e[e.PATH_COMPLETION=2]="PATH_COMPLETION",e[e.DIRECTIVE_COMPLETION=3]="DIRECTIVE_COMPLETION",e[e.VERSION_COMPLETION=4]="VERSION_COMPLETION",e[e.ANNOTATION_COMPLETION=5]="ANNOTATION_COMPLETION",e[e.SEQUENCE_KEY_COPLETION=6]="SEQUENCE_KEY_COPLETION",e[e.INCOMMENT=7]="INCOMMENT"}(t.LocationKind||(t.LocationKind={}));var q=t.LocationKind;t.resolveReference=S,t.subTypesWithLocals=b,t.subTypesWithName=_,t.schemasWithName=N,t.nodesDeclaringType=w,t.findAllSubTypes=R,t.allChildren=M;var Y=function(e,t,n,r){var i=e.getAdapter(K.RAMLService).getDeclaringNode();if(i&&n.isSameNode(i))return void r.push(t);if(e.isArray()&&Y(e.array().componentType(),t,n,r),e.isUnion()){var a=e.union();Y(a.leftType(),t,n,r),Y(a.rightType(),t,n,r)}e.superTypes().some(function(e){return e.nameId()==n.name()})&&r.push(t)};t.refFinder=L},function(e,t,n){"use strict";function r(e){var t=e.getExtra(p.SOURCE_EXTRA);return null==t?null:l.isSourceProvider(t)?t:f.isLowLevelNode(t)?{getSource:function(){return t.highLevelNode()}}:c.isParseResult(t)?{getSource:function(){return t}}:null}function i(e){return r(e)}function a(e){var t=e.getAdapters();return t?h.find(t,function(e){return l.rt.isParsedType(e)}):null}function o(e){if(!e)return null;var t=e.getExtra(p.SOURCE_EXTRA);if(t)return r(e);var n=a(e);return n?i(n):null}function s(e,t){var n=o(e);return n?{getSource:function(){var e=n.getSource(),r=e.asElement();if(null==r)return null;var i=r.elementsOfKind(d.Universe10.ObjectTypeDeclaration.properties.properties.name);return null==i||0==i.length?null:h.find(i,function(e){return t==e.attrValue(d.Universe10.TypeDeclaration.properties.name.name)})}}:null}function u(e){return s(e.domain(),e.nameId())}var l=n(39),p=l.rt,c=n(9),f=n(10),d=n(18),h=n(70);t.getExtraProviderSource=r,t.getRTypeSource=i,t.findRTypeByNominal=a,t.getNominalTypeSource=o,t.getNominalPropertySource=s,t.getNominalPropertySource2=u},function(e,t,n){"use strict";var r=n(39);e.exports=r},function(e,t,n){"use strict";var r=n(39);e.exports=r.getUniverse},function(e,t,n){"use strict";function r(){return t.rt.getSchemaUtils()}function i(e){e.isUnion?e.addAdapter(new N(e)):e.range&&e.addAdapter(new _(e))}function a(e){return e.getSource&&"function"==typeof e.getSource}function o(e){var t=e;return t.genuineUserDefinedType&&t.isUserDefined&&t.isUserDefined()}function s(e,t,n,r,i){var a=new E(e,t);return a.withDomain(n,i).withRange(r)}var u=this&&this.__extends||function(e,t){function n(){this.constructor=e}for(var r in t)t.hasOwnProperty(r)&&(e[r]=t[r]);e.prototype=null===t?Object.create(t):(n.prototype=t.prototype,new n)};t.rt=n(86);var l=t.rt.nominalTypes;t.getSchemaUtils=r,t.TOP_LEVEL_EXTRA=t.rt.TOP_LEVEL_EXTRA,t.DEFINED_IN_TYPES_EXTRA=t.rt.DEFINED_IN_TYPES_EXTRA,t.USER_DEFINED_EXTRA=t.rt.USER_DEFINED_EXTRA,t.SOURCE_EXTRA=t.rt.SOURCE_EXTRA,t.tsInterfaces=t.rt.tsInterfaces,t.injector={inject:function(e){i(e)}},l.registerInjector(t.injector);var p=function(e){function t(){e.apply(this,arguments)}return u(t,e),t}(l.AbstractType);t.AbstractType=p;var c=function(e){function t(){e.apply(this,arguments)}return u(t,e),t}(l.ValueType);t.ValueType=c;var f=function(){function e(){}return e.isInstance=function(t){if(null!=t&&t.getClassIdentifier&&"function"==typeof t.getClassIdentifier)for(var n=0,r=t.getClassIdentifier();n<r.length;n++){var i=r[n];if(i==e.CLASS_IDENTIFIER)return!0}return!1},e.prototype.getClassIdentifier=function(){var t=[];return t.concat(e.CLASS_IDENTIFIER)},e.prototype.getSource=function(){return null},e.CLASS_IDENTIFIER="definitionSystem.SourceProvider",e}();t.SourceProvider=f,t.isSourceProvider=a;var d=function(e){function t(){e.apply(this,arguments),this.values=[]}return u(t,e),t.isInstance=function(e){if(null!=e&&e.getClassIdentifier&&"function"==typeof e.getClassIdentifier)for(var n=0,r=e.getClassIdentifier();n<r.length;n++){var i=r[n];if(i==t.CLASS_IDENTIFIER)return!0}return!1},t.prototype.getClassIdentifier=function(){var e=[];return e.concat(t.CLASS_IDENTIFIER)},t.CLASS_IDENTIFIER="definitionSystem.EnumType",t}(c);t.EnumType=d;var h=function(e){function t(t,n,r,i){e.call(this,t,i,n),this.referenceTo=r}return u(t,e),t.isInstance=function(e){if(null!=e&&e.getClassIdentifier&&"function"==typeof e.getClassIdentifier)for(var n=0,r=e.getClassIdentifier();n<r.length;n++){var i=r[n];if(i==t.CLASS_IDENTIFIER)return!0}return!1},t.prototype.getClassIdentifier=function(){var e=[];return e.concat(t.CLASS_IDENTIFIER)},t.prototype.getReferencedType=function(){return this.universe().type(this.referenceTo)},t.prototype.hasStructure=function(){return!0},t.CLASS_IDENTIFIER="definitionSystem.ReferenceType",t}(c);t.ReferenceType=h;var m=function(e){function t(t,n,r,i){void 0===i&&(i=""),e.call(this,t,n,r)}return u(t,e),t.isInstance=function(e){if(null!=e&&e.getClassIdentifier&&"function"==typeof e.getClassIdentifier)for(var n=0,r=e.getClassIdentifier();n<r.length;n++){var i=r[n];if(i==t.CLASS_IDENTIFIER)return!0}return!1},t.prototype.getClassIdentifier=function(){var e=[];return e.concat(t.CLASS_IDENTIFIER)},t.prototype.allProperties=function(t){return void 0===t&&(t={}),e.prototype.allProperties.call(this,t)},t.CLASS_IDENTIFIER="definitionSystem.NodeClass",t}(l.StructuredType);t.NodeClass=m;var y=function(e){function t(t,n,r,i,a){e.call(this,t,n,i,a),this.getAdapter(N).setDeclaringNode(r)}return u(t,e),t.isInstance=function(e){if(null!=e&&e.getClassIdentifier&&"function"==typeof e.getClassIdentifier)for(var n=0,r=e.getClassIdentifier();n<r.length;n++){var i=r[n];if(i==t.CLASS_IDENTIFIER_UserDefinedClass)return!0}return!1},t.prototype.getClassIdentifier=function(){var n=e.prototype.getClassIdentifier.call(this);return n.concat(t.CLASS_IDENTIFIER_UserDefinedClass)},t.prototype.key=function(){return null},t.prototype.isUserDefined=function(){return!0},t.prototype.typeId=function(){var e=this.nameId(),t=this.getAdapter(N).getDeclaringNode();if(t){e+=t.lowLevel().start();var n=t.lowLevel().unit();n&&(e+=n.absolutePath())}return e},t.prototype.hasValueTypeInHierarchy=function(){return this._value?!0:this.isAssignableFrom("ObjectTypeDeclaration")?!1:!1},t.prototype.isGenuineUserDefinedType=function(){return null!=this.genuineUserDefinedType()},t.prototype.genuineUserDefinedType=function(){var e=this.getAdapter(N).getDeclaringNode();return e?this:null},t.CLASS_IDENTIFIER_UserDefinedClass="definitionSystem.UserDefinedClass",t}(m);t.UserDefinedClass=y,t.isUserDefinedClass=o;var v=function(e){function t(){e.apply(this,arguments)}return u(t,e),t.prototype.allProperties=function(e){void 0===e&&(e={});var t=this.superTypes()[0].allProperties();if(0==t.length){var n=this.getAdapter(N).getDeclaringNode(),r=new T("value",n);r.withDomain(this),r.withCanBeValue(),r.withRequired(!1);this.superTypes()[0];t=[],r.withRange(r.node().asElement().definition().universe().type("string")),t.push(r)}return t},t.prototype.isAnnotationType=function(){return!0},t}(y);t.AnnotationType=v;var g=function(e){function t(t,n,r,i){void 0===n&&(n=""),void 0===r&&(r=null),void 0===i&&(i="RAML08"),e.call(this,n),this._parent=r,this._classes=[],this._uversion="RAML08",this.aMap={},this.matchedObjects=t,this._uversion=i}return u(t,e),t.prototype.matched=function(){return this.matchedObjects?this.matchedObjects:this._parent?this._parent.matched():void 0},t.prototype.setTopLevel=function(e){this._topLevel=e},t.prototype.getTopLevel=function(){return this._topLevel},t.prototype.setOriginalTopLevelText=function(e){this._originalTopLevelText=e},t.prototype.getOriginalTopLevelText=function(){return this._originalTopLevelText},t.prototype.setTypedVersion=function(e){this._typedVersion=e},t.prototype.getTypedVersion=function(){return this._typedVersion},t.prototype.version=function(){return this._uversion},t.prototype.setUniverseVersion=function(e){this._uversion=e},t.prototype.types=function(){var e=[].concat(this._classes);return null!=this._parent&&(e=e.concat(this._parent.types())),e},t.prototype.type=function(e){if(this.aMap[e])return this.aMap[e];for(var t,n=0;n<this._classes.length;n++)if(this._classes[n].nameId()==e){t=this._classes[n];break}if(null==t&&this._parent){var t=this._parent.type(e);if(t instanceof p){var r=t;r._universe=this}}return t},t.prototype.register=function(e){return this._classes.push(e),e instanceof m&&this._classes.forEach(function(t){if(t instanceof m){var n=t;n.getAdapter(N).getExtendedType()==e&&e.getAdapter(N).getRuntimeExtenders().push(t)}}),this},t.prototype.registerAlias=function(e,t){this.aMap[e]=t},t.prototype.unregister=function(e){this._classes=this._classes.filter(function(t){return t!=e});var t=e.superTypes();return t.forEach(function(t){var n=t;n._superTypes=n._superTypes.filter(function(t){return t!=e})}),t=e.subTypes(),t.forEach(function(t){var n=t;n._subTypes=n._subTypes.filter(function(t){return t!=e})}),this},t.prototype.registerSuperClass=function(e,t){var n=e,r=t;n._superTypes.push(t),r._subTypes.push(e)},t}(l.Described);t.Universe=g,t.prop=s;var A=function(){function e(e,t){this.name=e,this.value=t}return e}();t.ChildValueConstraint=A;var E=function(e){function t(){e.apply(this,arguments),this._isFromParentValue=!1,this._isFromParentKey=!1,this._key=!1,this._declaresFields=!1,this._describes=null,this._canBeDuplicator=!1,this._allowsNull=!1,this._canBeValue=!1,this._isInherited=!1,this._selfNode=!1,this._noDirectParse=!1,this._contextReq=[],this.determinesChildValues=[]}return u(t,e),t.prototype.isPrimitive=function(){return this.range().isValueType()&&!(this.range()instanceof h);
},t.prototype.withNoDirectParse=function(){this._noDirectParse=!0},t.prototype.isNoDirectParse=function(){return this._noDirectParse},t.prototype.withSelfNode=function(){this._selfNode=!0},t.prototype.isSelfNode=function(){return this._selfNode},t.prototype.matchKey=function(t){return null!=this._groupName?this.isAnnotation()?"("==t.charAt(0)&&")"==t.charAt(t.length-1)?!0:!1:this._groupName==t:e.prototype.matchKey.call(this,t)},t.prototype.valueDocProvider=function(){return this._vprovider},t.prototype.setValueDocProvider=function(e){return this._vprovider=e,this},t.prototype.suggester=function(){return this._suggester},t.prototype.setValueSuggester=function(e){this._suggester=e},t.prototype.enumOptions=function(){return this._enumOptions&&"string"==typeof this._enumOptions?[this._enumOptions+""]:this._enumOptions},t.prototype.getOftenKeys=function(){return this._oftenKeys},t.prototype.withOftenKeys=function(e){return this._oftenKeys=e,this},t.prototype.withCanBeValue=function(){return this._canBeValue=!0,this},t.prototype.withInherited=function(e){this._isInherited=e},t.prototype.isInherited=function(){return this._isInherited},t.prototype.isAllowNull=function(){return this._allowsNull},t.prototype.withAllowNull=function(){this._allowsNull=!0},t.prototype.getCanBeDuplicator=function(){return this._canBeDuplicator},t.prototype.canBeValue=function(){return this._canBeValue},t.prototype.setCanBeDuplicator=function(){return this._canBeDuplicator=!0,!0},t.prototype.inheritedContextValue=function(){return this._inheritsValueFromContext},t.prototype.withInheritedContextValue=function(e){return this._inheritsValueFromContext=e,this},t.prototype.withContextRequirement=function(e,t){this._contextReq.push({name:e,value:t})},t.prototype.getContextRequirements=function(){return this._contextReq},t.prototype.withDescribes=function(e){return this._describes=e,this},t.prototype.describesAnnotation=function(){return null!=this._describes},t.prototype.describedAnnotation=function(){return this._describes},t.prototype.isReference=function(){return this.range()instanceof h},t.prototype.referencesTo=function(){return this.range().getReferencedType()},t.prototype.newInstanceName=function(){return this._newInstanceName?this._newInstanceName:this.range().nameId()},t.prototype.withThisPropertyDeclaresFields=function(e){return void 0===e&&(e=!0),this._declaresFields=e,this},t.prototype.isThisPropertyDeclaresTypeFields=function(){return this._declaresFields},t.prototype.withNewInstanceName=function(e){return this._newInstanceName=e,this},t.prototype.addChildValueConstraint=function(e){this.determinesChildValues.push(e)},t.prototype.getChildValueConstraints=function(){return this.determinesChildValues},t.prototype.childRestrictions=function(){return this.determinesChildValues},t.prototype.id=function(){return this._id?this._id:this._groupName?this.domain().getAdapter(N).getDeclaringNode()?null:(this._id=this._groupName+this.domain().nameId(),this._id):null},t.prototype.isAnnotation=function(){return"annotations"==this._groupName&&this.domain()&&!this.domain().isUserDefined()},t.prototype.withFromParentValue=function(e){return void 0===e&&(e=!0),this._isFromParentValue=e,this},t.prototype.withFromParentKey=function(e){return void 0===e&&(e=!0),this._isFromParentKey=e,this},t.prototype.isFromParentKey=function(){return this._isFromParentKey},t.prototype.isFromParentValue=function(){return this._isFromParentValue},t.prototype.withGroupName=function(e){return this._groupName=e,this},t.prototype.unmerge=function(){return this._groupName=this.nameId(),this},t.prototype.merge=function(){return this._groupName=null,this},t.prototype.withKey=function(e){return this._key=e,this},t.prototype.isKey=function(){return this._key},t.prototype.isMerged=function(){return null==this._groupName},t.prototype.groupName=function(){return this._groupName},t.prototype.key=function(){return null},t}(l.Property);t.Property=E;var T=function(e){function t(t,n){e.call(this,t),this._node=n}return u(t,e),t.isInstance=function(e){if(null!=e&&e.getClassIdentifier&&"function"==typeof e.getClassIdentifier)for(var n=e.getClassIdentifier(),r=0;r<n.length;r++)if(n[r]==t.CLASS_IDENTIFIER)return!0;return!1},t.prototype.getClassIdentifier=function(){var e=[];return e.concat(t.CLASS_IDENTIFIER)},t.prototype.withDisplayName=function(e){this._displayName=e},t.prototype.getDisplayName=function(){return this._displayName},t.prototype.node=function(){return this._node||null==this.sourceProvider||(this._node=this.sourceProvider.getSource()),this._node},t.prototype.setSourceProvider=function(e){this.sourceProvider=e},t.CLASS_IDENTIFIER="definitionSystem.UserDefinedProp",t}(E);t.UserDefinedProp=T;var S=function(){function e(){}return e.prototype.setDocTableName=function(e){this._documentationTableName=e},e.prototype.docTableName=function(){return this._documentationTableName},e.prototype.setHidden=function(e){this._isHidden=e},e.prototype.isHidden=function(){return this._isHidden},e.prototype.setMarkdownDescription=function(e){this._markdownDescription=e},e.prototype.markdownDescription=function(){return this._markdownDescription},e.prototype.setValueDescription=function(e){this._valueDescription=e},e.prototype.valueDescription=function(){return this._valueDescription},e}();t.RAMLPropertyDocumentationService=S;var b=function(e){function t(){e.apply(this,arguments)}return u(t,e),t.prototype.isSystem=function(){return this._isSystemProperty},t.prototype.withSystem=function(e){return this._isSystemProperty=e,this},t.prototype.isEmbedMap=function(){return this._isEmbededMap},t.prototype.withEmbedMap=function(){return this._isEmbededMap=!0,this},t}(S);t.RAMLPropertyParserService=b;var _=function(e){function t(t){if(e.call(this),this._property=t,this._meta={},!t)throw new Error;e.call(this)}return u(t,e),t.prototype.valueDocProvider=function(){return this._property.valueDocProvider()},t.prototype.withPropertyGrammarType=function(e){this._propertyGrammarType=e},t.prototype.getPropertyGrammarType=function(){return this._propertyGrammarType},t.prototype.id=function(){return this._property.nameId()},t.prototype.range=function(){return this._property.range()},t.prototype.domain=function(){return this._property.domain()},t.prototype.isAllowNull=function(){return this._property.isAllowNull()},t.prototype.referencesTo=function(){return this._property.referencesTo()},t.prototype.isReference=function(){return this._property.isReference()},t.prototype.nameId=function(){return this._property.nameId()},t.prototype.priority=function(){return this.isKey()?128:this.isReference()?64:this.isTypeExpr()?32:"example"==this.nameId()?0:-1024},t.prototype.isKey=function(){return this._property instanceof E?this._property.isKey():!1},t.prototype.isMerged=function(){return this._property instanceof E?this._property.isMerged():!1},t.prototype.isTypeExpr=function(){return this.texpr},t.prototype.isExampleProperty=function(){return this.example},t.prototype.setExample=function(e){this.example=e},t.prototype.setTypeExpression=function(e){this.texpr=e},t.prototype.isDescriminating=function(){return this._property.isDescriminator()},t.prototype.putMeta=function(e,t){this._meta[e]=t},t.prototype.meta=function(e){return this._meta[e]},t}(b);t.RAMLPropertyService=_;var N=function(){function e(e){
//!!!
this._allowsOptionalProperties=!1,this._possibleInterfaces=[],
//!!!
this._canInherit=[],this._declaredBy=[],
//!!!
this._aliases=[],
//!!!???
this._defining=[],this._runtimeExtenders=[],
//!!!???
this._declaresType=null,
//!!!
this._isTemplate=!1,
//!!!
this._contextRequirements=[],this._type=e}return e.prototype.withAllowQuestion=function(){this._allowsOptionalProperties=!0},e.prototype.getAllowQuestion=function(){return this._allowsOptionalProperties},e.prototype.withCanInherit=function(e){this._canInherit.push(e)},e.prototype.getReferenceIs=function(){return this._referenceIs},e.prototype.withReferenceIs=function(e){this._referenceIs=e},e.prototype.descriminatorValue=function(){return 0==this.valueRequirements().length?this.nameId():this.valueRequirements()[0].value},e.prototype.getCanInherit=function(){return this._canInherit},e.prototype.withAllowAny=function(){this._allowsAnyChildren=!0},e.prototype.getAllowAny=function(){return this._allowsAnyChildren||this._type.isExternal()},e.prototype.globallyDeclaredBy=function(){return this._declaredBy},e.prototype.setGloballyDeclaredBy=function(e){this._declaredBy.push(e)},e.prototype.setDeclaringNode=function(e){this._node=e},e.prototype.nameId=function(){return this._type.nameId()},e.prototype.universe=function(){return this._type.universe()},e.prototype.isAssignableFrom=function(e){return this._type.isAssignableFrom(e)},e.prototype.setConsumesRefs=function(e){this._consumesRef=e},e.prototype.definingPropertyIsEnough=function(e){this._defining.push(e)},e.prototype.getDefining=function(){return this._defining},e.prototype.getConsumesRefs=function(){return this._consumesRef},e.prototype.addAlias=function(e){this._aliases.push(e)},e.prototype.getAliases=function(){return this._aliases},e.prototype.valueRequirements=function(){return this._type.valueRequirements()},e.prototype.isAnnotation=function(){return this._annotationChecked?this._isAnnotation:(this._annotationChecked=!0,this._isAnnotation=this._type.allSuperTypes().some(function(e){return e.key()&&"AnnotationRef"==e.key().name}),this._isAnnotation)},e.prototype.allowValue=function(){return this._allowsValueSet?this._allowsValue:this._type.allProperties().some(function(e){return e.isFromParentValue()||e.canBeValue()})?(this._allowsValue=!0,this._allowsValueSet=!0,!0):(this._allowsValueSet=!0,!1)},e.prototype.key=function(){return this._type.key()},e.prototype.getRepresentationOf=function(){return this._representationOf},e.prototype.getPath=function(){return""},e.prototype.isDeclaration=function(){return this._isTemplate?!0:this._convertsToGlobal?!0:this._declaresType?!0:this.key()&&"Library"===this.key().name?!0:!1},e.prototype.isGlobalDeclaration=function(){return this._actuallyExports?!0:this._isTemplate?!0:this._declaresType?!0:!1},e.prototype.isTypeSystemMember=function(){return null!=this._declaresType},e.prototype.getExtendedType=function(){return this.universe().type(this._declaresType)},e.prototype.setInlinedTemplates=function(e){return this._isTemplate=e,this},e.prototype.getRuntimeExtenders=function(){return this._runtimeExtenders},e.prototype.isInlinedTemplates=function(){return this._isTemplate},e.prototype.setExtendedTypeName=function(t){this._declaresType=t;var n=this.universe().type(t);if(n instanceof m){var r=n;r.getAdapter(e)._runtimeExtenders.push(this._type)}},e.prototype.getKeyProp=function(){for(var e=this._type.allProperties(),t=0;t<e.length;t++)if(e[t].getAdapter(_).isKey())return e[t];return null},e.prototype.withActuallyExports=function(e){this._actuallyExports=e},e.prototype.withConvertsToGlobal=function(e){this._convertsToGlobal=e},e.prototype.getConvertsToGlobal=function(){return this._convertsToGlobal},e.prototype.getActuallyExports=function(){return this._actuallyExports},e.prototype.isUserDefined=function(){return this._type instanceof m?this._type.isUserDefined():this._type instanceof p?this._type.isUserDefined():!1},e.prototype.withContextRequirement=function(e,t){this._contextRequirements.push({name:e,value:t})},e.prototype.getContextRequirements=function(){return this._contextRequirements},e.prototype.findMembersDeterminer=function(){for(var e=this._type.allProperties(),t=0;t<e.length;t++)if(e[t].isThisPropertyDeclaresTypeFields())return e[t];return null},e.prototype.getDeclaringNode=function(){if(this._node)return this._node;if(this._type){var e=this._type.getExtra(t.SOURCE_EXTRA);return e?e instanceof f||e.getSource&&"function"==typeof e.getSource?e.getSource():e:null}return null},e.prototype.registerSupertypes=function(e){var t=this,n=this._type.universe();e.forEach(function(e){var r=n.type(e);r&&t._type.addSuperType(r)})},e.prototype.registerPossibleInterfaces=function(e){for(var t=this._type.universe(),n=0,r=e;n<r.length;n++){var i=r[n],a=t.type(i);if(!a)return;this._possibleInterfaces.push(a)}},e.prototype.possibleInterfaces=function(){return this._possibleInterfaces},e}();t.RAMLService=N;var w={},R=n(60),I=R.universeDumps,M=n(61),C=n(62);t.universesInfo=C,t.getUniverse=function(){var e=function(e){if(w[e])return w[e];var t=I[e],n=M.toDefSystem(t,"RAML08"==e?C.Universe08:C.Universe10);return n&&(n.setUniverseVersion(e),w[e]=n),n};return e.availableUniverses=function(){return Object.keys(I)},e}()},function(e,t,n){function r(){this.protocol=null,this.slashes=null,this.auth=null,this.host=null,this.port=null,this.hostname=null,this.hash=null,this.search=null,this.query=null,this.pathname=null,this.path=null,this.href=null}function i(e,t,n){if(e&&l(e)&&e instanceof r)return e;var i=new r;return i.parse(e,t,n),i}function a(e){return u(e)&&(e=i(e)),e instanceof r?e.format():r.prototype.format.call(e)}function o(e,t){return i(e,!1,!0).resolve(t)}function s(e,t){return e?i(e,!1,!0).resolveObject(t):t}function u(e){return"string"==typeof e}function l(e){return"object"==typeof e&&null!==e}function p(e){return null===e}function c(e){return null==e}var f=n(87);t.parse=i,t.resolve=o,t.resolveObject=s,t.format=a,t.Url=r;var d=/^([a-z0-9.+-]+:)/i,h=/:[0-9]*$/,m=["<",">",'"',"`"," ","\r","\n","	"],y=["{","}","|","\\","^","`"].concat(m),v=["'"].concat(y),g=["%","/","?",";","#"].concat(v),A=["/","?","#"],E=255,T=/^[a-z0-9A-Z_-]{0,63}$/,S=/^([a-z0-9A-Z_-]{0,63})(.*)$/,b={javascript:!0,"javascript:":!0},_={javascript:!0,"javascript:":!0},N={http:!0,https:!0,ftp:!0,gopher:!0,file:!0,"http:":!0,"https:":!0,"ftp:":!0,"gopher:":!0,"file:":!0},w=n(85);r.prototype.parse=function(e,t,n){if(!u(e))throw new TypeError("Parameter 'url' must be a string, not "+typeof e);var r=e;r=r.trim();var i=d.exec(r);if(i){i=i[0];var a=i.toLowerCase();this.protocol=a,r=r.substr(i.length)}if(n||i||r.match(/^\/\/[^@\/]+@[^@\/]+/)){var o="//"===r.substr(0,2);!o||i&&_[i]||(r=r.substr(2),this.slashes=!0)}if(!_[i]&&(o||i&&!N[i])){for(var s=-1,l=0;l<A.length;l++){var p=r.indexOf(A[l]);-1!==p&&(-1===s||s>p)&&(s=p)}var c,h;h=-1===s?r.lastIndexOf("@"):r.lastIndexOf("@",s),-1!==h&&(c=r.slice(0,h),r=r.slice(h+1),this.auth=decodeURIComponent(c)),s=-1;for(var l=0;l<g.length;l++){var p=r.indexOf(g[l]);-1!==p&&(-1===s||s>p)&&(s=p)}-1===s&&(s=r.length),this.host=r.slice(0,s),r=r.slice(s),this.parseHost(),this.hostname=this.hostname||"";var m="["===this.hostname[0]&&"]"===this.hostname[this.hostname.length-1];if(!m)for(var y=this.hostname.split(/\./),l=0,R=y.length;R>l;l++){var I=y[l];if(I&&!I.match(T)){for(var M="",C=0,L=I.length;L>C;C++)M+=I.charCodeAt(C)>127?"x":I[C];if(!M.match(T)){var P=y.slice(0,l),O=y.slice(l+1),D=I.match(S);D&&(P.push(D[1]),O.unshift(D[2])),O.length&&(r="/"+O.join(".")+r),this.hostname=P.join(".");break}}}if(this.hostname.length>E?this.hostname="":this.hostname=this.hostname.toLowerCase(),!m){for(var x=this.hostname.split("."),U=[],l=0;l<x.length;++l){var k=x[l];U.push(k.match(/[^A-Za-z0-9_-]/)?"xn--"+f.encode(k):k)}this.hostname=U.join(".")}var F=this.port?":"+this.port:"",B=this.hostname||"";this.host=B+F,this.href+=this.host,m&&(this.hostname=this.hostname.substr(1,this.hostname.length-2),"/"!==r[0]&&(r="/"+r))}if(!b[a])for(var l=0,R=v.length;R>l;l++){var K=v[l],V=encodeURIComponent(K);V===K&&(V=escape(K)),r=r.split(K).join(V)}var j=r.indexOf("#");-1!==j&&(this.hash=r.substr(j),r=r.slice(0,j));var W=r.indexOf("?");if(-1!==W?(this.search=r.substr(W),this.query=r.substr(W+1),t&&(this.query=w.parse(this.query)),r=r.slice(0,W)):t&&(this.search="",this.query={}),r&&(this.pathname=r),N[a]&&this.hostname&&!this.pathname&&(this.pathname="/"),this.pathname||this.search){var F=this.pathname||"",k=this.search||"";this.path=F+k}return this.href=this.format(),this},r.prototype.format=function(){var e=this.auth||"";e&&(e=encodeURIComponent(e),e=e.replace(/%3A/i,":"),e+="@");var t=this.protocol||"",n=this.pathname||"",r=this.hash||"",i=!1,a="";this.host?i=e+this.host:this.hostname&&(i=e+(-1===this.hostname.indexOf(":")?this.hostname:"["+this.hostname+"]"),this.port&&(i+=":"+this.port)),this.query&&l(this.query)&&Object.keys(this.query).length&&(a=w.stringify(this.query));var o=this.search||a&&"?"+a||"";return t&&":"!==t.substr(-1)&&(t+=":"),this.slashes||(!t||N[t])&&i!==!1?(i="//"+(i||""),n&&"/"!==n.charAt(0)&&(n="/"+n)):i||(i=""),r&&"#"!==r.charAt(0)&&(r="#"+r),o&&"?"!==o.charAt(0)&&(o="?"+o),n=n.replace(/[?#]/g,function(e){return encodeURIComponent(e)}),o=o.replace("#","%23"),t+i+n+o+r},r.prototype.resolve=function(e){return this.resolveObject(i(e,!1,!0)).format()},r.prototype.resolveObject=function(e){if(u(e)){var t=new r;t.parse(e,!1,!0),e=t}var n=new r;if(Object.keys(this).forEach(function(e){n[e]=this[e]},this),n.hash=e.hash,""===e.href)return n.href=n.format(),n;if(e.slashes&&!e.protocol)return Object.keys(e).forEach(function(t){"protocol"!==t&&(n[t]=e[t])}),N[n.protocol]&&n.hostname&&!n.pathname&&(n.path=n.pathname="/"),n.href=n.format(),n;if(e.protocol&&e.protocol!==n.protocol){if(!N[e.protocol])return Object.keys(e).forEach(function(t){n[t]=e[t]}),n.href=n.format(),n;if(n.protocol=e.protocol,e.host||_[e.protocol])n.pathname=e.pathname;else{for(var i=(e.pathname||"").split("/");i.length&&!(e.host=i.shift()););e.host||(e.host=""),e.hostname||(e.hostname=""),""!==i[0]&&i.unshift(""),i.length<2&&i.unshift(""),n.pathname=i.join("/")}if(n.search=e.search,n.query=e.query,n.host=e.host||"",n.auth=e.auth,n.hostname=e.hostname||e.host,n.port=e.port,n.pathname||n.search){var a=n.pathname||"",o=n.search||"";n.path=a+o}return n.slashes=n.slashes||e.slashes,n.href=n.format(),n}var s=n.pathname&&"/"===n.pathname.charAt(0),l=e.host||e.pathname&&"/"===e.pathname.charAt(0),f=l||s||n.host&&e.pathname,d=f,h=n.pathname&&n.pathname.split("/")||[],i=e.pathname&&e.pathname.split("/")||[],m=n.protocol&&!N[n.protocol];if(m&&(n.hostname="",n.port=null,n.host&&(""===h[0]?h[0]=n.host:h.unshift(n.host)),n.host="",e.protocol&&(e.hostname=null,e.port=null,e.host&&(""===i[0]?i[0]=e.host:i.unshift(e.host)),e.host=null),f=f&&(""===i[0]||""===h[0])),l)n.host=e.host||""===e.host?e.host:n.host,n.hostname=e.hostname||""===e.hostname?e.hostname:n.hostname,n.search=e.search,n.query=e.query,h=i;else if(i.length)h||(h=[]),h.pop(),h=h.concat(i),n.search=e.search,n.query=e.query;else if(!c(e.search)){if(m){n.hostname=n.host=h.shift();var y=n.host&&n.host.indexOf("@")>0?n.host.split("@"):!1;y&&(n.auth=y.shift(),n.host=n.hostname=y.shift())}return n.search=e.search,n.query=e.query,p(n.pathname)&&p(n.search)||(n.path=(n.pathname?n.pathname:"")+(n.search?n.search:"")),n.href=n.format(),n}if(!h.length)return n.pathname=null,n.search?n.path="/"+n.search:n.path=null,n.href=n.format(),n;for(var v=h.slice(-1)[0],g=(n.host||e.host)&&("."===v||".."===v)||""===v,A=0,E=h.length;E>=0;E--)v=h[E],"."==v?h.splice(E,1):".."===v?(h.splice(E,1),A++):A&&(h.splice(E,1),A--);if(!f&&!d)for(;A--;A)h.unshift("..");!f||""===h[0]||h[0]&&"/"===h[0].charAt(0)||h.unshift(""),g&&"/"!==h.join("/").substr(-1)&&h.push("");var T=""===h[0]||h[0]&&"/"===h[0].charAt(0);if(m){n.hostname=n.host=T?"":h.length?h.shift():"";var y=n.host&&n.host.indexOf("@")>0?n.host.split("@"):!1;y&&(n.auth=y.shift(),n.host=n.hostname=y.shift())}return f=f||n.host&&h.length,f&&!T&&h.unshift(""),h.length?n.pathname=h.join("/"):(n.pathname=null,n.path=null),p(n.pathname)&&p(n.search)||(n.path=(n.pathname?n.pathname:"")+(n.search?n.search:"")),n.auth=e.auth||n.auth,n.slashes=n.slashes||e.slashes,n.href=n.format(),n},r.prototype.parseHost=function(){var e=this.host,t=h.exec(e);t&&(t=t[0],":"!==t&&(this.port=t.substr(1)),e=e.substr(0,e.length-t.length)),e&&(this.hostname=e)}},function(e,t,n){},function(e,t,n){},function(e,t,n){"use strict";function r(e){return e&&e.indexOf("\n")>=0}function i(e){return r(e)&&e.length>2&&"|"==e[0]&&("\n"==e[1]||"\r"==e[1]||"\n"==e[2])}function a(e,t){var n="";if(r(e)){n+="|\n";for(var i=d(e),a=0;a<i.length;a++)n+=u(t,i[a])}else n+=e;return n}function o(e){if(!i(e))return e;for(var t=null,n=d(e),r=1;r<n.length;r++){var a=n[r],o=a.substring(2);t?t+=o:t=o}return t}function s(e){if(!e)return e;for(var t=0;t<e.length;){var n=e[t];if("\r"!=n&&"\n"!=n&&" "!=n&&"	"!=n)break;t++}return e.substring(t,e.length)}function u(e,t){void 0===t&&(t="");for(var n="",r=0;e>r;r++)n+="  ";return n+t}function l(e,t){void 0===t&&(t=""),console.log(u(e,t))}function p(e,t){void 0===t&&(t=null);for(var n="",r=0;r<e.length;r++){var i=e[r];"\r"==i&&(i=null==t?"\\r":t),"\n"==i&&(i=null==t?"\\n":t),n+=i}return n}function c(e){for(var t=e.length;t>0;){var n=e[t-1];if(" "!=n&&"	"!=n&&"\r"!=n&&"\n"!=n)break;t--}return e.substring(0,t)}function f(e){return s(c(e))}function d(e){var t=e.match(/^.*((\r\n|\n|\r)|$)/gm);return t}function h(e,t){if(!e||!t||e.length<t.length)return!1;for(var n=0;n<t.length;n++)if(e[n]!=t[n])return!1;return!0}function m(e,t){if(!e||!t||e.length<t.length)return!1;for(var n=0;n<t.length;n++)if(e[e.length-1-n]!=t[t.length-1-n])return!1;return!0}function y(e){return e.charAt(0).toUpperCase()+e.slice(1)}t.isMultiLine=r,t.isMultiLineValue=i,t.makeMutiLine=a,t.fromMutiLine=o,t.trimStart=s,t.indent=u,t.print=l,t.replaceNewlines=p,t.trimEnd=c,t.trim=f,t.splitOnLines=d,t.startsWith=h,t.endsWith=m,t.capitalize=y;var v=function(){function e(e,t,n){this.contents=e,this.start=t,this.end=n}return e.prototype.text=function(){return this.contents.substring(this.start,this.end)},e.prototype.startpos=function(){return this.start},e.prototype.endpos=function(){return this.end},e.prototype.len=function(){return this.end-this.start},e.prototype.unitText=function(){return this.contents},e.prototype.withStart=function(t){return new e(this.contents,t,this.end)},e.prototype.withEnd=function(t){return new e(this.contents,this.start,t)},e.prototype.sub=function(e,t){return this.contents.substring(e,t)},e.prototype.trimStart=function(){for(var t=this.start;t<this.contents.length-1;){var n=this.contents[t];if(" "!=n&&"	"!=n)break;t++}return new e(this.contents,t,this.end)},e.prototype.trimEnd=function(){for(var t=this.end;t>0;){var n=this.contents[t-1];if(" "!=n&&"	"!=n)break;t--}return new e(this.contents,this.start,t)},e.prototype.extendToStartOfLine=function(){for(var t=this.start;t>0;){var n=this.contents[t-1];if("\r"==n||"\n"==n)break;t--}return new e(this.contents,t,this.end)},e.prototype.extendAnyUntilNewLines=function(){var t=this.end;if(t>0){var n=this.contents[t-1];if("\n"==n)return this}for(;t<this.contents.length-1;){var r=this.contents[t];if("\r"==r||"\n"==r)break;t++}return new e(this.contents,this.start,t)},e.prototype.extendSpacesUntilNewLines=function(){var t=this.end;if(t>0){var n=this.contents[t-1];if("\n"==n)return this}for(;t<this.contents.length-1;){var r=this.contents[t];if(" "!=r||"\r"==r||"\n"==r)break;t++}return new e(this.contents,this.start,t)},e.prototype.extendSpaces=function(){for(var t=this.end;t<this.contents.length-1;){var n=this.contents[t];if(" "!=n)break;t++}return new e(this.contents,this.start,t)},e.prototype.extendSpacesBack=function(){for(var t=this.start;t>0;){var n=this.contents[t-1];if(" "!=n)break;t--}return new e(this.contents,t,this.end)},e.prototype.extendCharIfAny=function(t){var n=this.end;return n<this.contents.length-1&&this.contents[n]==t&&n++,new e(this.contents,this.start,n)},e.prototype.extendCharIfAnyBack=function(t){var n=this.start;return n>0&&this.contents[n-1]==t&&n--,new e(this.contents,n,this.end)},e.prototype.extendToNewlines=function(){var t=this.end;if(t>0){var n=this.contents[t-1];if("\n"==n)return this}for(;t<this.contents.length-1;){var r=this.contents[t];if("\r"!=r&&"\n"!=r)break;t++}return new e(this.contents,this.start,t)},e.prototype.extendUntilNewlinesBack=function(){for(var t=this.start;t>0;){var n=this.contents[t-1];if("\r"==n||"\n"==n)break;t--}return new e(this.contents,t,this.end)},e.prototype.reduceNewlinesEnd=function(){for(var t=this.end;t>this.start;){var n=this.contents[t-1];if("\r"!=n&&"\n"!=n)break;t--}return new e(this.contents,this.start,t)},e.prototype.reduceSpaces=function(){for(var t=this.end;t>this.start;){var n=this.contents[t-1];if(" "!=n)break;t--}return new e(this.contents,this.start,t)},e.prototype.replace=function(e){return this.sub(0,this.start)+e+this.sub(this.end,this.unitText().length)},e.prototype.remove=function(){return this.sub(0,this.start)+this.sub(this.end,this.unitText().length)},e}();t.TextRange=v},function(e,t,n){"use strict";function r(e){var t=e;return t.valueName&&t.toHighLevel&&t.toHighLevel2}var i=this&&this.__extends||function(e,t){function n(){this.constructor=e}for(var r in t)t.hasOwnProperty(r)&&(e[r]=t[r]);e.prototype=null===t?Object.create(t):(n.prototype=t.prototype,new n)},a=n(77),o=n(54),s=n(69),u=n(24),l=n(33),p=n(18),c=n(39),f=n(14),d=n(53),h=n(70),m=function(){function e(e,t,n){this._parent=e,this._transformer=t,this.ramlVersion=n}return e.isInstance=function(t){return null!=t&&t.getClassIdentifier&&"function"==typeof t.getClassIdentifier&&h.contains(t.getClassIdentifier(),e.CLASS_IDENTIFIER)},e.prototype.getClassIdentifier=function(){var t=[];return t.concat(e.CLASS_IDENTIFIER)},e.prototype.hasInnerIncludeError=function(){return this._originalNode.hasInnerIncludeError()},e.prototype.keyKind=function(){return this._originalNode.keyKind()},e.prototype.primaryNode=function(){return null},e.prototype.isAnnotatedScalar=function(){return this._originalNode.isAnnotatedScalar()},e.prototype.actual=function(){return this._originalNode?this._originalNode.actual():this},e.prototype.transformer=function(){return this._transformer},e.prototype.setTransformer=function(e){this._transformer=e},e.prototype.originalNode=function(){return this._originalNode},e.prototype.start=function(){return this._originalNode.start()},e.prototype.end=function(){return this._originalNode.end()},e.prototype.value=function(e){throw new Error("The method must be overridden")},e.prototype.includeErrors=function(){return this._originalNode.includeErrors()},e.prototype.includePath=function(){return this._originalNode.includePath()},e.prototype.includeReference=function(){return this._originalNode.includeReference()},e.prototype.setKeyOverride=function(e){this._keyOverride=e},e.prototype.setValueOverride=function(e){this._valueOverride=e},e.prototype.key=function(e){return void 0===e&&(e=!1),this._keyOverride?this._keyOverride:this._originalNode.key(e)},e.prototype.optional=function(){return this.originalNode().optional()},e.prototype.children=function(){throw new Error("The method must be overridden")},e.prototype.parent=function(){return this._parent},e.prototype.unit=function(){return this._originalNode.unit()},e.prototype.containingUnit=function(){return this._originalNode.containingUnit()},e.prototype.includeBaseUnit=function(){return this._originalNode.unit()},e.prototype.anchorId=function(){return this._originalNode.anchorId()},e.prototype.errors=function(){return this._originalNode.errors()},e.prototype.anchoredFrom=function(){return this._originalNode.anchoredFrom()},e.prototype.includedFrom=function(){return this._originalNode.includedFrom()},e.prototype.visit=function(e){e(this)&&this.children().forEach(function(t){return t.visit(e)})},e.prototype.addChild=function(e){},e.prototype.execute=function(e){},e.prototype.dump=function(){return null},e.prototype.dumpToObject=function(e){var t=o.serialize2(this,e);if(this.kind()==a.Kind.MAPPING){var n={};return n[this.key(!0)]=t,n}return t},e.prototype.keyStart=function(){return this._originalNode.keyStart()},e.prototype.keyEnd=function(){return this._originalNode.keyEnd()},e.prototype.valueStart=function(){return this._originalNode.valueStart()},e.prototype.valueEnd=function(){return this._originalNode.valueEnd()},e.prototype.isValueLocal=function(){return this._originalNode.isValueLocal()},e.prototype.kind=function(){return this._originalNode.kind()},e.prototype.valueKind=function(){return this._originalNode.valueKind()},e.prototype.anchorValueKind=function(){return this._originalNode.anchorValueKind()},e.prototype.resolvedValueKind=function(){return this._originalNode.resolvedValueKind()},e.prototype.show=function(e){this._originalNode.show(e)},e.prototype.setHighLevelParseResult=function(e){this._highLevelParseResult=e},e.prototype.highLevelParseResult=function(){return this._highLevelParseResult},e.prototype.setHighLevelNode=function(e){this._highLevelNode=e},e.prototype.highLevelNode=function(){return this._highLevelNode?this._highLevelNode:this._originalNode.highLevelNode()},e.prototype.text=function(e){throw new Error("not implemented")},e.prototype.copy=function(){throw new Error("not implemented")},e.prototype.markup=function(e){throw new Error("not implemented")},e.prototype.nodeDefinition=function(){return u.getDefinitionForLowLevelNode(this)},e.prototype.includesContents=function(){return this._originalNode.includesContents()},e.prototype.root=function(){for(var e=this;e.parent();){var t=e.parent();e=t}return e},e.prototype.find=function(e){var t=null;return this.children().forEach(function(n){n.key()&&n.key()==e&&(t||(t=n))}),t},e.prototype.isMap=function(){return this.kind()==a.Kind.MAP},e.prototype.isMapping=function(){return this.kind()==a.Kind.MAPPING},e.prototype.isSeq=function(){return this.kind()==a.Kind.SEQ},e.prototype.isScalar=function(){return this.kind()==a.Kind.SCALAR},e.prototype.isValueSeq=function(){return this.valueKind()==a.Kind.SEQ},e.prototype.isValueMap=function(){return this.valueKind()==a.Kind.MAP},e.prototype.isValueInclude=function(){return this.valueKind()==a.Kind.INCLUDE_REF},e.prototype.isValueScalar=function(){return this.valueKind()==a.Kind.SCALAR},e.prototype.definingUnitSequence=function(){var e=d.toOriginal(this).key(),t=this.transformer();return t?t.definingUnitSequence(e):null},e.CLASS_IDENTIFIER="LowLevelASTProxy.LowLevelProxyNode",e}();t.LowLevelProxyNode=m;var y=function(e){function t(t,n,r,i,a){void 0===a&&(a=!0),e.call(this,n,r,i),this.isPrimary=a,this._adoptedNodes=[],this._preserveAnnotations=!1,this.nonMergableChildren={};var o=this.parent()?this.parent().originalNode():null;v.isInstance(t)?this._originalNode=t:this._originalNode=new v(t,o,r,this.ramlVersion),this._adoptedNodes.push(this._originalNode)}return i(t,e),t.isInstance=function(e){return null!=e&&e.getClassIdentifier&&"function"==typeof e.getClassIdentifier&&h.contains(e.getClassIdentifier(),t.CLASS_IDENTIFIER_LowLevelCompositeNode)},t.prototype.getClassIdentifier=function(){var n=e.prototype.getClassIdentifier.call(this);return n.concat(t.CLASS_IDENTIFIER_LowLevelCompositeNode)},t.prototype.adoptedNodes=function(){return this._adoptedNodes},t.prototype.primaryNode=function(){return this.isPrimary?this._originalNode:null},t.prototype.parent=function(){return this._parent},t.prototype.adopt=function(e,t){t||(t=this._transformer);var n=this.parent()?this.parent().originalNode():null,r=new v(e,n,t,this.ramlVersion);this._adoptedNodes.push(r),this._children&&this._children.forEach(function(e){return e._parent=null}),this._children=null,this.highLevelNode()&&this.highLevelNode().resetChildren()},t.prototype.value=function(e){if(this._valueOverride)return this._valueOverride;var t,n=this._adoptedNodes.filter(function(e){return null!=e.value()});if(t=n.length>0?n[0].value(e):this._originalNode.value(e),v.isInstance(t)){var r=t.key();if(r)for(var i=this;1==i.children().length&&(i.kind()==a.Kind.MAPPING||i.kind()==a.Kind.MAP||i.kind()==a.Kind.SEQ);)if(i=i.children()[0],i.originalNode().key()==r){t=i;break}this._valueOverride=t}return t},t.prototype.patchAdoptedNodes=function(e){var t=this;this._adoptedNodes=[],e.forEach(function(e){return t.adopt(e.node,e.transformer)}),this._adoptedNodes.length>0&&(this._originalNode=this._adoptedNodes[0]),this.resetChildren()},t.prototype.children=function(){var e=this;if(this._children)return this._children;for(var n=[],r=!1,i=!1,o=0,s=this._adoptedNodes;o<s.length;o++){var u=s[o],l=u.children();if(l&&l.length>0&&(i=!0,l[0].key()&&this.originalNode().valueKind()!=a.Kind.SEQ&&(r=!0)),r&&i)break}if(r)n=this.collectChildrenWithKeys();else if(i){n=this.collectChildrenWithKeys();var p={};this._adoptedNodes.forEach(function(r){return r.children().filter(function(e){return!e.key()}).forEach(function(i){var a=r==e.primaryNode(),o=e.buildKey(i);if(a||!p[o]){p[o]=!0;var s=r.transformer()?r.transformer():e.transformer(),u=v.isInstance(i)?i.originalNode():i;n.push(new t(u,e,s,e.ramlVersion,a))}})})}else n=[];return this._children=n,this._preserveAnnotations&&this._children.forEach(function(e){return e.preserveAnnotations()}),n},t.prototype.buildKey=function(e){var t=o.serialize(e),n=this.nodeDefinition();if(n&&(n.key()==p.Universe08.TraitRef||n.key()==p.Universe08.ResourceTypeRef||n.key()==p.Universe10.TraitRef||n.key()==p.Universe10.ResourceTypeRef)&&t&&"object"==typeof t){var r=Object.keys(t);r.length>0&&(t=r[0])}return null==t?"":s(t)},t.prototype.collectChildrenWithKeys=function(){for(var e=this,n=[],r={},i=0,a=this._adoptedNodes;i<a.length;i++)for(var o=a[i],s=o==this.primaryNode(),u=0,f=o.children();u<f.length;u++){var d=f[u],h=d.originalNode().key();if(!this.nonMergableChildren[h]||o==this._originalNode){if(h&&o.transformer()){var m=null!=h&&this._preserveAnnotations&&l.stringStartsWith(h,"(")&&l.stringEndsWith(h,")");m||(h=o.transformer().transform(h).value)}if(!this.skipKey(h,s)&&h){var y=r.hasOwnProperty(h)&&r[h];y||(y=[],r[h]=y),y.push({node:d.originalNode(),transformer:o.transformer(),isPrimary:s})}}}var v=this.unit().highLevel().root().definition().universe().version(),g=this.key()&&"/"==this.key()[0],A=c.getUniverse("RAML10").type(p.Universe10.Method.name),E=A.property(p.Universe10.Method.properties.method.name).enumOptions();return Object.keys(r).forEach(function(i){var a=r[i],o=!0,s=!1,u=E.indexOf(i)>=0;if(a.forEach(function(e){var t=e.node.optional()&&("RAML10"!=v||g&&u);o=o&&t,s=s||e.isPrimary}),s){var l=[];a.filter(function(e){return e.isPrimary}).forEach(function(n){var r=n.transformer?n.transformer:e.transformer();l.push(new t(n.node,e,r,e.ramlVersion,!0))});var p=l[0];a.filter(function(e){return!e.isPrimary}).forEach(function(e){p.adopt(e.node,e.transformer)}),l.forEach(function(e){return n.push(e)})}else if(!o){for(var c=a[0].transformer?a[0].transformer:e.transformer(),p=new t(a[0].node,e,c,e.ramlVersion,!1),f=1;f<a.length;f++)p.adopt(a[f].node,a[f].transformer);n.push(p)}}),n},t.prototype.isResource=function(){return this.highLevelNode()&&f.isResourceType(this.highLevelNode().definition())},t.prototype.skipKey=function(e,t){if(t)return!1;if("RAML08"!=this.ramlVersion)return!1;p.Universe08.Method,p.Universe08.Resource;return!1},t.prototype.valueKind=function(){if(this._originalNode.kind()!=a.Kind.MAPPING)return null;for(var e=0;e<this._adoptedNodes.length;e++){var t=this._adoptedNodes[e],n=t.originalNode().actual();if(n&&null!=n.value)return t.valueKind()}return null},t.prototype.includePath=function(){for(var e=0;e<this._adoptedNodes.length;e++){var t=this._adoptedNodes[e],n=t.includePath();if(null!=n)return n}return null},t.prototype.includeBaseUnit=function(){for(var t=0;t<this._adoptedNodes.length;t++){var n=this._adoptedNodes[t],r=n.includePath();if(null!=r)return n.unit()}return e.prototype.includeBaseUnit.call(this)},t.prototype.includeReference=function(){for(var e=0;e<this._adoptedNodes.length;e++){var t=this._adoptedNodes[e];if(null!=t.value())return t.includeReference()}return null},t.prototype.optional=function(){return h.all(this._adoptedNodes,function(e){return e.optional()})},t.prototype.replaceChild=function(e,n,r,i){void 0===r&&(r=!1),void 0===i&&(i=null),this._children||(this._children=[]);var a;if(t.isInstance(n)?(a=n,a._parent=this):a=new t(n,this,null,this.ramlVersion),null==e)return this._children.push(a),a;var o=this._children.indexOf(e);return o>=0?this._children[o]=a:this._children.push(a),a},t.prototype.removeChild=function(e){if(this._children&&null!=e){var t=this._children.indexOf(e);if(t>=0){for(var n=t;n<this._children.length-1;n++)this._children[n]=this._children[n+1];this._children.pop()}}},t.prototype.setChildren=function(e){var n=this;return null==e?void(this._children=null):(this._children=e.map(function(e){return t.isInstance(e)?e:new t(e,n,null,n.ramlVersion)}),void(this._preserveAnnotations&&this._children.forEach(function(e){return e.preserveAnnotations()})))},t.prototype.resetChildren=function(){this._children=null},t.prototype.preserveAnnotations=function(){this._preserveAnnotations=!0,this._children&&this._children.forEach(function(e){return e.preserveAnnotations()})},t.prototype.filterChildren=function(){this.children();var e={},t=[];this._children.forEach(function(n){if(null!=n.key())return void t.push(n);var r=JSON.stringify(o.serialize(n));e[r]||(e[r]=!0,t.push(n))}),this._children=t},t.prototype.containingUnit=function(){for(var e={},t=0,n=this.adoptedNodes();t<n.length;t++){var r=n[t],i=r.containingUnit();i&&(e[i.absolutePath()]=!0)}return Object.keys(e).length<=1?this._originalNode.containingUnit():null},t.prototype.takeOnlyOriginalChildrenWithKey=function(e){this.nonMergableChildren[e]=!0},t.CLASS_IDENTIFIER_LowLevelCompositeNode="LowLevelASTProxy.LowLevelCompositeNode",t}(m);t.LowLevelCompositeNode=y;var v=function(e){function t(t,n,r,i){e.call(this,n,r,i),this._originalNode=t}return i(t,e),t.isInstance=function(e){return null!=e&&e.getClassIdentifier&&"function"==typeof e.getClassIdentifier&&h.contains(e.getClassIdentifier(),t.CLASS_IDENTIFIER_LowLevelValueTransformingNode)},t.prototype.getClassIdentifier=function(){var n=e.prototype.getClassIdentifier.call(this);return n.concat(t.CLASS_IDENTIFIER_LowLevelValueTransformingNode)},t.prototype.value=function(e){var n=this.originalNode().value(e),r=this.transformer();if(r){var i=r.transform(n,e);n=i.value}return null==n||"object"!=typeof n||Array.isArray(n)?(null!=n&&e&&(n=""+n),n):new t(n,this._parent,this._transformer,this.ramlVersion)},t.prototype.children=function(){var e=this,n=null,r=this.originalNode();if(null!=this._transformer){var i=this._transformer.children(r);null!=i&&(n=i)}return null==n&&(n=r.children()),n.map(function(n){return new t(n,e,e._transformer,e.ramlVersion)})},t.prototype.valueKind=function(){var t=this._transformer&&this._transformer.valueKind(this.originalNode());return null!=t?t:e.prototype.valueKind.call(this)},t.prototype.anchorValueKind=function(){var t=this._transformer&&this._transformer.anchorValueKind(this.originalNode());return null!=t?t:e.prototype.anchorValueKind.call(this)},t.prototype.resolvedValueKind=function(){var t=this._transformer&&this._transformer.resolvedValueKind(this.originalNode());return null!=t?t:e.prototype.resolvedValueKind.call(this)},t.prototype.includePath=function(){var t=this._transformer&&this._transformer.includePath(this.originalNode());return null!=t?t:e.prototype.includePath.call(this)},t.prototype.parent=function(){return this._parent},t.prototype.key=function(t){void 0===t&&(t=!1);var n=e.prototype.key.call(this,t);if(null!=this.transformer()){var r=this.transformer().transform(n).value;return"object"==typeof r?null:r}return n},t.CLASS_IDENTIFIER_LowLevelValueTransformingNode="LowLevelASTProxy.LowLevelValueTransformingNode",t}(m);t.LowLevelValueTransformingNode=v,t.isLowLevelProxyNode=r},function(e,t,n){"use strict";function r(e){for(var t=-1,n=e.lowLevel().children(),r=0;r<n.length;r++){var i=n[r];if(i.isMapping()){var a=i.asMapping().key.value,o=e.definition().property(a);o&&(o.isFromParentValue()||o.range().hasValueTypeInHierarchy())&&(t=r)}}return t}function i(e,t,n,r){var i=null,a=n&&n.getAdapter(R.RAMLPropertyService).isEmbedMap();
if(a&&S.find(e.lowLevel().children(),function(e){return e.key()==n.nameId()})&&(a=!1),r||a){var o=v(e);i=o?o:A.InsertionPoint.atStart()}return i}function a(e){for(var t=e;t.originalNode&&t.originalNode();)t=t.originalNode();return t.asMapping()}function o(e,t){e.isStub()||e.clearChildrenCache();var n=e.children(),r=t.property()&&t.property().getAdapter(R.RAMLPropertyService).isEmbedMap();r&&S.find(e.lowLevel().children(),function(e){return e.key()==t.property().nameId()})&&(r=!1);if(!t.isAttr()&&!r||w.isMethodType(e.definition())&&t.property&&w.isIsProperty(t.property())){var i=t.property().nameId(),o=e.definition(),s=(o.allProperties(),e.lowLevel().children()),u=t.property();if(w.isTraitsProperty(u)||w.isResourceTypesProperty(u))return S.find(s,function(e){return e.isMapping()?a(e).key.value===t.property().nameId():!1});var l=o.propertyIndex(i);if(0>l)return null;for(var p=0;p<s.length;p++){var c=s[p];if(c.isMapping()){var f=a(c).key.value,d=o.propertyIndex(f);if(d>l){var h=p-1;return 0>h?null:(console.log("insert to node: "+h),s[h])}}}return null}for(var m=null,p=0;p<n.length&&n[p].isAttr();p++)n[p].property&&n[p].property()&&w.isIsProperty(n[p].property())||(m=n[p].lowLevel());return null==m&&(m=e.lowLevel()),m}function s(e,t){if(e.isStub()){if(_.ASTNodeImpl.isInstance(t)){var n=t;if(n.isInEdit)return}if(!e._children)return;return void(e._children=e._children.filter(function(e){return e!=t}))}var r=new T.CompositeCommand;if(_.ASTNodeImpl.isInstance(t)){var i=t;i.property().getAdapter(R.RAMLPropertyService).isMerged()?r.commands.push(T.removeNode(e.lowLevel(),i.lowLevel())):1==e.elementsOfKind(i.property().nameId()).length?r.commands.push(T.removeNode(e.lowLevel(),i.lowLevel().parent().parent())):r.commands.push(T.removeNode(e.lowLevel(),i.lowLevel()))}else r.commands.push(T.removeNode(e.lowLevel(),t.lowLevel()));e.lowLevel().execute(r)}function u(e){var t=(e.lowLevel().root(),new T.CompositeCommand),n=A.createMap([]);t.commands.push(T.initRamlFile(e.lowLevel(),n)),e.lowLevel().execute(t);var r=e.root();r._node=n,r.clearChildrenCache()}function l(e,t){if(t!=e.value()){var n=new T.CompositeCommand;if("string"==typeof t){var r=t;e.property().isFromParentKey()||e.property().isAnnotation()?(e.property().isAnnotation()&&(r="("+r+")"),n.commands.push(T.setKey(e.lowLevel(),r))):r&&0!=r.length||e.isEmbedded()?(r||(r=""),n.commands.push(T.setAttr(e.lowLevel(),r))):(n.commands.push(T.removeNode(e.lowLevel().parent(),e.lowLevel())),e.parent().clearChildrenCache())}else{if(e.property().isFromParentKey())throw new Error("couldn't set structured value to a key: "+e.property().nameId());var i=t;n.commands.push(T.setAttrStructured(e.lowLevel(),i))}e.lowLevel().execute(n)}}function p(e,t){var n=A.createScalar(t),r=e.lowLevel();r.isScalar()?r=r.parent():r.isMapping();var i=new T.CompositeCommand;i.commands.push(T.insertNode(r,n,null,!0)),e.lowLevel().execute(i),e.parent().clearChildrenCache()}function c(e,t){var n=e.lowLevel();if(n.isScalar())n=n.parent();else if(n.isMapping()){e.lowLevel()}var r=new T.CompositeCommand;r.commands.push(T.insertNode(n,t.lowLevel(),null,!0)),e.lowLevel().execute(r),e.parent().clearChildrenCache()}function f(e){var t=e.lowLevel().parent();if(!e.property().isMultiValue()&&e.isEmbedded())e.setValue("");else{var n=new T.CompositeCommand;n.commands.push(T.removeNode(t,e.lowLevel())),e.lowLevel().execute(n),e.parent().clearChildrenCache()}}function d(e,t){if(!e.property().isMultiValue())throw new Error("setValue(string[]) only apply to multi-values properties");var n=e.parent();if(n&&m(n)){var r=new T.CompositeCommand;return r.commands.push(T.setAttr(n.lowLevel(),t[0])),void n.lowLevel().execute(r)}if(e.isEmpty());else{var i=(n.lowLevel(),n.attributes(e.name()));i.forEach(function(e){return e.remove()})}1==t.length&&e.property().isDescriminator()?n.attrOrCreate(e.name()).setValue(t[0]):t.forEach(function(t){return n.attrOrCreate(e.name()).addValue(t)})}function h(e,t){if(t!=e.name()){var n=new T.CompositeCommand;n.commands.push(T.setKey(e.lowLevel(),t)),e.lowLevel().execute(n)}}function m(e){return e.definition()&&e.definition().isAssignableFrom(N.Universe10.TypeDeclaration.name)&&e.lowLevel()&&e.lowLevel().valueKind()===b.Kind.SCALAR?!0:!1}function y(e,t,n){var a=A.createMapping(t,n);if(e.definition()&&e.definition().isAssignableFrom(N.Universe10.TypeDeclaration.name)&&e.lowLevel()&&(e.lowLevel().valueKind()===b.Kind.SCALAR||!e.lowLevel()._node.value)){var o=t,s=A.createMapping(o,e.lowLevel().value()),u=new T.CompositeCommand;u.commands.push(T.setAttr(e.lowLevel(),"")),e.lowLevel().execute(u),u=new T.CompositeCommand;var l=i(e,s,e.definition().property(o),!0);u.commands.push(T.insertNode(e.lowLevel(),s,l)),e.lowLevel().execute(u),e.resetChildren();var p=e.attr(o)&&e.attr(o).lowLevel();if(p)return u=new T.CompositeCommand,u.commands.push(T.setAttr(p,n)),p.execute(u),void e.clearChildrenCache()}if(e.isStub()){var c=r(e);e.lowLevel().addChild(a,c+1)}else{e.isEmptyRamlFile()&&e.initRamlFile();var u=new T.CompositeCommand,l=i(e,a,e.definition().property(t),!0);u.commands.push(T.insertNode(e.lowLevel(),a,l)),e.lowLevel().execute(u)}e.clearChildrenCache()}function v(e){var t=e.lowLevel().children(),n=r(e);return 0>n?null:t[n]}function g(e,t){!e.isStub()&&e.isEmptyRamlFile()&&e.initRamlFile();var n=t.lowLevel();if(e._children||(e._children=[]),!t.property()){var i=t,a=e.definition().allProperties(),s=null;if(a.forEach(function(e){var t=e.range();t==i.definition()&&(s=e);var n=S.find(i.definition().allSuperTypes(),function(e){return e==t});n&&(s=e)}),!s)throw new Error("Unable to find correct child");i.patchProp(s)}var u=o(e,t),l=new T.CompositeCommand,p=null;if(t.property().getAdapter(R.RAMLPropertyService).isMerged()||t.property().range().hasValueTypeInHierarchy())l.commands.push(T.insertNode(e.lowLevel(),t.lowLevel(),u)),p=e.lowLevel();else{var c=t.property().nameId(),f=e.lowLevel(),d=e.lowLevel().find(c);if(p=d,d){var h=null===d.value()&&d.key&&d.key()===N.Universe10.Api.properties.types.name,m=!h&&t.property().getAdapter(R.RAMLPropertyService).isEmbedMap();l.commands.push(T.insertNode(d,t.lowLevel(),u,m))}else{var y=null;if(t.property().getAdapter(R.RAMLPropertyService).isEmbedMap()){var v="RAML10"==e.definition().universe().version();y=n.isValueMap()&&v?A.createMapNode(c):A.createSeqNode(c),y.addChild(t.lowLevel())}else y=A.createNode(c),y.addChild(t.lowLevel());l.commands.push(T.insertNode(f,y,u)),p=f}}if(e.isStub()){var g=r(e);return 0>g?e._children.push(t):e._children.splice(g,0,t),void l.commands.forEach(function(e){return p.addChild(e.value)})}e.lowLevel().execute(l),e._children.push(t),t.setParent(e)}var A=n(24),E=n(39),T=n(10),S=n(70),b=n(77),_=n(16),N=n(18),w=n(14),R=E;t.removeNodeFrom=s,t.initEmptyRAMLFile=u,t.setValue=l,t.addStringValue=p,t.addStructuredValue=c,t.removeAttr=f,t.setValues=d,t.setKey=h,t.createAttr=y,t.addToNode=g},function(e,t,n){"use strict";function r(e,t){var n=e.root().definition().universe(),i=e.lowLevel().key();if(i){var a=y.ASTPropImpl.isInstance(e)?e:null;T(i,t,a,!0,n,!0)}if(e.children().forEach(function(e){return r(e,t)}),y.ASTPropImpl.isInstance(e)){var o=e,s=o.value();if("string"==typeof s){var u=s;T(u,t,o,!1,n)}else e.lowLevel().visit(function(e){if(e.value()){var r=e.value()+"";T(r,t,o,!0,n)}return!0})}else if(y.BasicASTNode.isInstance(e)){var s=e.lowLevel().value();if("string"==typeof s){var u=s;T(u,t,null,!1,n)}else e.lowLevel().visit(function(e){if(e.value()){var r=e.value()+"";T(r,t,null,!0,n)}return!0})}}function i(e,t,n){void 0===n&&(n=[]),e.optional()&&(n=n.concat("/"));var r=e.value();if("string"==typeof r)for(var a=r,o=E(a),s=o.parameterUsages,u=0,l=s;u<l.length;u++){var p=l[u],c=t[p];null==c&&(c=[],t[p]=c),c.push(n)}else for(var f=0,d=e.children();f<d.length;f++){var h=d[f];i(h,t,n.concat(h.key()))}}function a(e,t){var n={},a={};r(t,n),i(t.lowLevel(),a);for(var o=0,s=Object.keys(a);o<s.length;o++){var u=s[o];a[u]=h.unique(a[u])}e.getAdapter(v.RAMLService).setInlinedTemplates(!0),Object.keys(n).forEach(function(r){var i=null;n[r].length>0&&(i=n[r][0].attr);var o=new f.UserDefinedProp(r,i);o.withDomain(e);var s=a[r];o.getAdapter(v.RAMLPropertyService).putMeta("templatePaths",s);var u;if("RAML10"==t.definition().universe().version()){var l=n[r].filter(function(e){return A.isStringTypeType(e.tp)}).length>0;u=l?m.Universe10.StringType.name:m.Universe10.AnyType.name}else u=m.Universe08.StringType.name;var p=h.unique(n[r].map(function(e){return e.tp})).filter(function(e){return e&&e.nameId()!=u});o.withRange(1==p.length?p[0]:t.definition().universe().type(u)),o.withRequired(!0),1==p.length&&"RAML10"==t.definition().universe().version()&&p[0].key()==m.Universe10.SchemaString&&o.getAdapter(v.RAMLPropertyService).setTypeExpression(!0),o.unmerge()});var l=new f.UserDefinedProp("____key",t);return l.withDomain(e),l.withKey(!0),l.withFromParentKey(!0),l.withRange(t.definition().universe().type(m.Universe08.StringType.name)),e}function o(e,t){"RAML08"==t.universe().version()&&e.getAdapter(v.RAMLService).withAllowAny();var n=t.property(t.getAdapter(v.RAMLService).getReferenceIs());return n&&n.range().properties().forEach(function(t){var n=new f.Property(t.nameId());n.unmerge(),n.withDomain(e),n.withRange(t.range()),n.withMultiValue(t.isMultiValue())}),e}function s(e){if(!e)return null;if(e.associatedType())return e.associatedType();var t=e.lowLevel().unit(),n=t?t.path():"";d.setPropertyConstructor(function(t){var n=null,r=e.elementsOfKind("properties").filter(function(e){return e.name()==t});r&&(n=r[0]);var i=new f.UserDefinedProp(t,n);return i.unmerge(),i});try{var r=e.definition();if(e.property()&&e.property().nameId()==m.Universe10.LibraryBase.properties.annotationTypes.name){var i=new S(e.name(),e.definition().universe(),e,n,""),s=u(e);i._superTypes.push(s),0==e.elementsOfKind(m.Universe10.ObjectTypeDeclaration.properties.properties.name).length&&i.getAdapter(v.RAMLService).withAllowAny();var l=r.getAdapter(v.RAMLService).getExtendedType();return l&&i._superTypes.push(l),i}var i=new f.UserDefinedClass(e.name(),e.definition().universe(),e,n,"");if(e.setAssociatedType(i),r.getAdapter(v.RAMLService).isInlinedTemplates())return a(i,e);if(r.getAdapter(v.RAMLService).getReferenceIs())return o(i,r);var p=u(e);p.getAdapter(v.RAMLService).setDeclaringNode(e),e.setAssociatedType(p)}finally{d.setPropertyConstructor(null)}return p}function u(e){return d.toNominal(e.parsedType(),function(t){var n=e.definition().universe().type(t);if(!n){new f.UserDefinedClass("",e.definition().universe(),e,"","")}return n})}function l(e){return function(t){var n=e.type(t);if(!n){new f.UserDefinedClass("",e,null,"","")}return n}}function p(e,t){var n=h.find(e.elementsOfKind("types"),function(e){return e.name()==t.name()});n&&d.setPropertyConstructor(function(e){var t=n.elementsOfKind("properties").filter(function(t){return t.name()==e}),r=null;if(t&&t.length>0)r=t[0];else{var t=n.elementsOfKind("facets").filter(function(t){return t.name()==e});t&&t.length>0&&(r=t[0])}var i=new f.UserDefinedProp(e,r);return i.unmerge(),i});var r=l(e.definition().universe());return d.toNominal(t,r)}var c=this&&this.__extends||function(e,t){function n(){this.constructor=e}for(var r in t)t.hasOwnProperty(r)&&(e[r]=t[r]);e.prototype=null===t?Object.create(t):(n.prototype=t.prototype,new n)},f=n(39),d=f.rt,h=n(70),m=n(18),y=n(16),v=f,g=n(28),A=n(14);d.setPropertyConstructor(function(e){var t=new f.Property(e);return t.unmerge(),t});var E=function(e,t){void 0===t&&(t=!1),t&&"("==e.charAt(0)&&")"==e.charAt(e.length-1)&&(e=e.substring(1,e.length-1));for(var n=[],r=0;;){var i=e.indexOf("<<",r);if(-1==i)break;var a=e.indexOf(">>",i),o=0==i&&a==e.length-2,s=e.substring(i+2,a);r=i+2;var u=s.indexOf("|");-1!=u&&(s=s.substring(0,u)),s=s.trim(),n.push(s)}return{parameterUsages:n,isFull:o}},T=function(e,t,n,r,i,a){void 0===a&&(a=!1);var o=E(e,a),s=o.parameterUsages,u=o.isFull,l=n?n.property().range():null;a&&"("==e.charAt(0)&&")"==e.charAt(e.length-1)?(l=i.type(m.Universe10.SchemaString.name),r=!1):n&&(n.property().nameId()==m.Universe10.TypeDeclaration.properties.type.name||n.property().nameId()==m.Universe10.TypeDeclaration.properties.schema.name)&&n.property().domain().key()==m.Universe10.TypeDeclaration&&(l=i.type(m.Universe10.SchemaString.name)),!u||r?l=i.type(m.Universe10.StringType.name):null==l&&"RAML10"==i.version()&&(l=i.type(m.Universe10.AnyType.name));for(var p=0,c=s;p<c.length;p++){var f=c[p];if(null==g.RESERVED_TEMPLATE_PARAMETERS[f]){var d=t[f];d?d.push({tp:l,attr:n}):t[f]=[{tp:l,attr:n}]}}},S=function(e){function t(){e.apply(this,arguments)}return c(t,e),t.prototype.allProperties=function(e){void 0===e&&(e={});var t=this.superTypes()[0].allProperties();if(1==t.length&&"annotations"==t[0].nameId()){var n=this.getAdapter(f.RAMLService).getDeclaringNode(),r=new f.UserDefinedProp("value",n);r.withDomain(this),r.withCanBeValue(),r.withRequired(!1);this.superTypes()[0];t=[],r.withRange(r.node().asElement().definition().universe().type("string")),t.push(r)}return t},t.prototype.isAnnotationType=function(){return!0},t}(f.UserDefinedClass);t.typeFromNode=s,t.convertType=p},function(e,t,n){"use strict";function r(e){return e.isBuiltIn()?s[e.nameId()]:null}function i(e,t){void 0===t&&(t=!0);var n=e.definition(),i=(n.nameId(),r(n));if(!i){for(var o=a(n),u=n.allSuperTypes().sort(function(e,t){return o[e.nameId()]-o[t.nameId()]}),l=null,p=0;p<u.length;p++){var c=u[p].nameId();if(i=r(u[p]),"DataElement"!=c){if("hl.BasicNode"!=c&&i)break}else l=c}i||(l=c)}return i||(i=s["hl.BasicNode"]),i(e,t)}function a(e,t,n){void 0===t&&(t={}),void 0===n&&(n={});var r=e.nameId();if(!n[r]){n[r]=!0;var i=null!=t[r]&&t[r]+1||0,o=e.superTypes();return o.forEach(function(e){var r=e.nameId(),o=t[r];(null==o||i>o)&&(t[r]=i,a(e,t,n))}),delete n[r],t}}var o=n(21);t.buildWrapperNode=i;var s={AbstractSecurityScheme:function(e,t){return new o.AbstractSecuritySchemeImpl(e,t)},Annotable:function(e,t){return new o.AnnotableImpl(e,t)},AnnotationRef:function(e){return new o.AnnotationRefImpl(e)},AnnotationTarget:function(e){return new o.AnnotationTargetImpl(e)},AnyType:function(e){return new o.AnyTypeImpl(e)},Api:function(e,t){return new o.ApiImpl(e,t)},ArrayTypeDeclaration:function(e,t){return new o.ArrayTypeDeclarationImpl(e,t)},BasicSecurityScheme:function(e,t){return new o.BasicSecuritySchemeImpl(e,t)},BooleanType:function(e){return new o.BooleanTypeImpl(e)},BooleanTypeDeclaration:function(e,t){return new o.BooleanTypeDeclarationImpl(e,t)},ContentType:function(e){return new o.ContentTypeImpl(e)},CustomSecurityScheme:function(e,t){return new o.CustomSecuritySchemeImpl(e,t)},DateOnlyType:function(e){return new o.DateOnlyTypeImpl(e)},DateOnlyTypeDeclaration:function(e,t){return new o.DateOnlyTypeDeclarationImpl(e,t)},DateTimeOnlyType:function(e){return new o.DateTimeOnlyTypeImpl(e)},DateTimeOnlyTypeDeclaration:function(e,t){return new o.DateTimeOnlyTypeDeclarationImpl(e,t)},DateTimeType:function(e){return new o.DateTimeTypeImpl(e)},DateTimeTypeDeclaration:function(e,t){return new o.DateTimeTypeDeclarationImpl(e,t)},DigestSecurityScheme:function(e,t){return new o.DigestSecuritySchemeImpl(e,t)},DocumentationItem:function(e,t){return new o.DocumentationItemImpl(e,t)},Extension:function(e,t){return new o.ExtensionImpl(e,t)},FileType:function(e){return new o.FileTypeImpl(e)},FileTypeDeclaration:function(e,t){return new o.FileTypeDeclarationImpl(e,t)},FixedUriString:function(e){return new o.FixedUriStringImpl(e)},FragmentDeclaration:function(e,t){return new o.FragmentDeclarationImpl(e,t)},FullUriTemplateString:function(e){return new o.FullUriTemplateStringImpl(e)},IntegerType:function(e){return new o.IntegerTypeImpl(e)},IntegerTypeDeclaration:function(e,t){return new o.IntegerTypeDeclarationImpl(e,t)},Library:function(e,t){return new o.LibraryImpl(e,t)},LibraryBase:function(e,t){return new o.LibraryBaseImpl(e,t)},LocationKind:function(e){return new o.LocationKindImpl(e)},MarkdownString:function(e){return new o.MarkdownStringImpl(e)},Method:function(e,t){return new o.MethodImpl(e,t)},MethodBase:function(e,t){return new o.MethodBaseImpl(e,t)},MimeType:function(e){return new o.MimeTypeImpl(e)},ModelLocation:function(e){return new o.ModelLocationImpl(e)},NullType:function(e){return new o.NullTypeImpl(e)},NumberType:function(e){return new o.NumberTypeImpl(e)},NumberTypeDeclaration:function(e,t){return new o.NumberTypeDeclarationImpl(e,t)},OAuth1SecurityScheme:function(e,t){return new o.OAuth1SecuritySchemeImpl(e,t)},OAuth1SecuritySchemeSettings:function(e,t){return new o.OAuth1SecuritySchemeSettingsImpl(e,t)},OAuth2SecurityScheme:function(e,t){return new o.OAuth2SecuritySchemeImpl(e,t)},OAuth2SecuritySchemeSettings:function(e,t){return new o.OAuth2SecuritySchemeSettingsImpl(e,t)},ObjectTypeDeclaration:function(e,t){return new o.ObjectTypeDeclarationImpl(e,t)},Operation:function(e,t){return new o.OperationImpl(e,t)},Overlay:function(e,t){return new o.OverlayImpl(e,t)},PassThroughSecurityScheme:function(e,t){return new o.PassThroughSecuritySchemeImpl(e,t)},Reference:function(e){return new o.ReferenceImpl(e)},RelativeUriString:function(e){return new o.RelativeUriStringImpl(e)},Resource:function(e,t){return new o.ResourceImpl(e,t)},ResourceBase:function(e,t){return new o.ResourceBaseImpl(e,t)},ResourceType:function(e,t){return new o.ResourceTypeImpl(e,t)},ResourceTypeRef:function(e){return new o.ResourceTypeRefImpl(e)},Response:function(e,t){return new o.ResponseImpl(e,t)},SchemaString:function(e){return new o.SchemaStringImpl(e)},SecuritySchemePart:function(e,t){return new o.SecuritySchemePartImpl(e,t)},SecuritySchemeRef:function(e){return new o.SecuritySchemeRefImpl(e)},SecuritySchemeSettings:function(e,t){return new o.SecuritySchemeSettingsImpl(e,t)},StatusCodeString:function(e){return new o.StatusCodeStringImpl(e)},StringType:function(e){return new o.StringTypeImpl(e)},StringTypeDeclaration:function(e,t){return new o.StringTypeDeclarationImpl(e,t)},TimeOnlyType:function(e){return new o.TimeOnlyTypeImpl(e)},TimeOnlyTypeDeclaration:function(e,t){return new o.TimeOnlyTypeDeclarationImpl(e,t)},Trait:function(e,t){return new o.TraitImpl(e,t)},TraitRef:function(e){return new o.TraitRefImpl(e)},TypeDeclaration:function(e,t){return new o.TypeDeclarationImpl(e,t)},UnionTypeDeclaration:function(e,t){return new o.UnionTypeDeclarationImpl(e,t)},UriTemplate:function(e){return new o.UriTemplateImpl(e)},UsesDeclaration:function(e,t){return new o.UsesDeclarationImpl(e,t)},ValueType:function(e){return new o.ValueTypeImpl(e)},XMLFacetInfo:function(e,t){return new o.XMLFacetInfoImpl(e,t)}}},function(e,t,n){"use strict";function r(e){return e.isBuiltIn()?s[e.nameId()]:null}function i(e,t){void 0===t&&(t=!0);var n=e.definition(),i=(n.nameId(),r(n));if(!i){for(var o=a(n),u=n.allSuperTypes().sort(function(e,t){return o[e.nameId()]-o[t.nameId()]}),l=null,p=0;p<u.length;p++){var c=u[p].nameId();if(i=r(u[p]),"DataElement"!=c){if("hl.BasicNode"!=c&&i)break}else l=c}i||(l=c)}return i||(i=s["hl.BasicNode"]),i(e,t)}function a(e,t,n){void 0===t&&(t={}),void 0===n&&(n={});var r=e.nameId();if(!n[r]){n[r]=!0;var i=null!=t[r]&&t[r]+1||0,o=e.superTypes();return o.forEach(function(e){var r=e.nameId(),o=t[r];(null==o||i>o)&&(t[r]=i,a(e,t,n))}),delete n[r],t}}var o=n(52);t.buildWrapperNode=i;var s={AbstractSecurityScheme:function(e,t){return new o.AbstractSecuritySchemeImpl(e,t)},AnyType:function(e){return new o.AnyTypeImpl(e)},Api:function(e,t){return new o.ApiImpl(e,t)},BasicSecurityScheme:function(e,t){return new o.BasicSecuritySchemeImpl(e,t)},BodyLike:function(e,t){return new o.BodyLikeImpl(e,t)},BooleanType:function(e){return new o.BooleanTypeImpl(e)},BooleanTypeDeclaration:function(e,t){return new o.BooleanTypeDeclarationImpl(e,t)},CustomSecurityScheme:function(e,t){return new o.CustomSecuritySchemeImpl(e,t)},DateTypeDeclaration:function(e,t){return new o.DateTypeDeclarationImpl(e,t)},DigestSecurityScheme:function(e,t){return new o.DigestSecuritySchemeImpl(e,t)},DocumentationItem:function(e,t){return new o.DocumentationItemImpl(e,t)},ExampleString:function(e){return new o.ExampleStringImpl(e)},FileTypeDeclaration:function(e,t){return new o.FileTypeDeclarationImpl(e,t)},FixedUri:function(e){return new o.FixedUriImpl(e)},FullUriTemplateString:function(e){return new o.FullUriTemplateStringImpl(e)},GlobalSchema:function(e,t){return new o.GlobalSchemaImpl(e,t)},IntegerTypeDeclaration:function(e,t){return new o.IntegerTypeDeclarationImpl(e,t)},JSONBody:function(e,t){return new o.JSONBodyImpl(e,t)},JSONExample:function(e){return new o.JSONExampleImpl(e)},JSonSchemaString:function(e){return new o.JSonSchemaStringImpl(e)},MarkdownString:function(e){return new o.MarkdownStringImpl(e)},Method:function(e,t){return new o.MethodImpl(e,t)},MethodBase:function(e,t){return new o.MethodBaseImpl(e,t)},MimeType:function(e){return new o.MimeTypeImpl(e)},NumberType:function(e){return new o.NumberTypeImpl(e)},NumberTypeDeclaration:function(e,t){return new o.NumberTypeDeclarationImpl(e,t)},OAuth1SecurityScheme:function(e,t){return new o.OAuth1SecuritySchemeImpl(e,t)},OAuth1SecuritySchemeSettings:function(e,t){return new o.OAuth1SecuritySchemeSettingsImpl(e,t)},OAuth2SecurityScheme:function(e,t){return new o.OAuth2SecuritySchemeImpl(e,t)},OAuth2SecuritySchemeSettings:function(e,t){return new o.OAuth2SecuritySchemeSettingsImpl(e,t)},Parameter:function(e,t){return new o.ParameterImpl(e,t)},ParameterLocation:function(e){return new o.ParameterLocationImpl(e)},RAMLSimpleElement:function(e,t){return new o.RAMLSimpleElementImpl(e,t)},Reference:function(e){return new o.ReferenceImpl(e)},RelativeUriString:function(e){return new o.RelativeUriStringImpl(e)},Resource:function(e,t){return new o.ResourceImpl(e,t)},ResourceType:function(e,t){return new o.ResourceTypeImpl(e,t)},ResourceTypeRef:function(e){return new o.ResourceTypeRefImpl(e)},Response:function(e,t){return new o.ResponseImpl(e,t)},SchemaString:function(e){return new o.SchemaStringImpl(e)},SecuritySchemePart:function(e,t){return new o.SecuritySchemePartImpl(e,t)},SecuritySchemeRef:function(e){return new o.SecuritySchemeRefImpl(e)},SecuritySchemeSettings:function(e,t){return new o.SecuritySchemeSettingsImpl(e,t)},StatusCodeString:function(e){return new o.StatusCodeStringImpl(e)},StringType:function(e){return new o.StringTypeImpl(e)},StringTypeDeclaration:function(e,t){return new o.StringTypeDeclarationImpl(e,t)},Trait:function(e,t){return new o.TraitImpl(e,t)},TraitRef:function(e){return new o.TraitRefImpl(e)},UriTemplate:function(e){return new o.UriTemplateImpl(e)},ValueType:function(e){return new o.ValueTypeImpl(e)},XMLBody:function(e,t){return new o.XMLBodyImpl(e,t)},XMLExample:function(e){return new o.XMLExampleImpl(e)},XMLSchemaString:function(e){return new o.XMLSchemaStringImpl(e)}}},function(e,t,n){"use strict";function r(e){try{var t=JSON.parse(e);return t.$schema}catch(n){return s.isXmlScheme(e)}}function i(e,t){if(s.isXmlScheme(e))return p.getXMLSchema(e,new u.ContentProvider(t)).loadSchemaReferencesAsync().then(function(){return t});var n=p.getJSONSchema(e,new u.ContentProvider(t)),r=n.getMissingReferences([]).map(function(e){return n.contentAsync(e)});if(0===r.length)return Promise.resolve(t);var i=Promise.all(r),a=o(i,n);return a.then(function(){return t})}function a(e,t){var n=p.createSchema(e,new u.ContentProvider(t));return n.getMissingReferences([],!0)}function o(e,t){return e.then(function(e){if(e.length>0){var n=t.getMissingReferences(e);if(0===n.length)return[];var r=[];return n.forEach(function(e){r.push(t.contentAsync(e))}),o(Promise.all(r.concat(e)),t)}return Promise.resolve([])})}var s=n(56),u=n(31),l=n(39),p=l.getSchemaUtils();t.isScheme=r,t.startDownloadingReferencesAsync=i,t.getReferences=a},function(e,t,n){"use strict";function r(e){var t=e.ast(),n=c.find(t.children(),function(e){return e.key()==o.Universe10.Extension.properties["extends"].name});return n&&n.value()}var i=n(10),a=n(15),o=n(18),s=n(77),u=n(25),l=n(16),p=n(39),c=n(70),f=function(){function e(){this.expandedAbsToNsMap={},this._expandedNSMap={},this.byPathMap={},this.byNsMap={},this._hasFragments={},this._unitModels={}}return e.prototype.hasTemplates=function(e){var t=this.unitModel(e);if(!t.traits.isEmpty()||!t.resourceTypes.isEmpty())return!0;var n=this.expandedPathMap(e);if(n)for(var i=0,a=Object.keys(n);i<a.length;i++){var o=a[i],s=n[o].unit,u=this.unitModel(s);if(!u.traits.isEmpty()||!u.resourceTypes.isEmpty())return!0}for(var l=r(e);l;)if(e=e.resolve(l)){if(t=this.unitModel(e),!t.traits.isEmpty()||!t.resourceTypes.isEmpty())return!0;l=r(e)}else l=null;return!1},e.prototype.resolveNamespace=function(e,t){if(null==t)return null;var n=t.absolutePath(),r=this.expandedPathMap(e);if(!r)return null;var i=r[n];return i},e.prototype.expandedNSMap=function(e){var t=e.absolutePath(),n=this._expandedNSMap[t];if(void 0===n){var r=this.expandedPathMap(e);if(r){n={};for(var i=0,a=Object.keys(r);i<a.length;i++){var o=a[i],s=r[o];n[s.namespace()]=s}}else n=null;this._expandedNSMap[t]=n}return n},e.prototype.expandedPathMap=function(e){var t=e.absolutePath(),n=this.expandedAbsToNsMap[t];return void 0===n&&(n=this.calculateExpandedNamespaces(e),0==Object.keys(n).length&&(n=null),this.expandedAbsToNsMap[t]=n),n},e.prototype.calculateExpandedNamespaces=function(e){for(var t=this,n=a.dirname(e.absolutePath()),o={},c=[];e;){c.push(new d([],e,""));var f=e;e=null;var h=f.ast();if(h&&h.kind()!=s.Kind.SCALAR){var m=l.ramlFirstLine(f.contents());if(m&&3==m.length&&(m[2]==p.universesInfo.Universe10.Overlay.name||m[2]==p.universesInfo.Universe10.Extension.name)){var y=r(f);y&&(e=f.resolve(y),e&&u.isWaitingFor(e.absolutePath())&&(e=null))}}}for(var v,g,A,E,h,T,S,b=function(){v={},g={},A=c[_],E=A.unit;var e=E.absolutePath();return h=E.ast(),h&&h.kind()!=s.Kind.SCALAR?(T=A.steps()+1,S=function(r){if(r){var u=r.children(),p=r.unit(),f=p.absolutePath();if(null==r.parent()){if(v[f])return;if(v[f]=!0,f!=e){var h=l.ramlFirstLine(p.contents());h&&3==h.length&&"1.0"==h[1]&&(t._hasFragments[e]=!0)}var m=t.pathMap(p);if(m)for(var y=0,E=Object.keys(m);y<E.length;y++){var b=E[y],_=m[b],N=A.namespaceSegments.concat(_.namespaceSegments),w=A.usesNodes.concat(_.usesNodes),R=o[b];if(R){if(R.steps()<T)continue;if(R.steps()==T&&t.lexLessEq(R.namespaceSegments,N))continue}var I,M=_.includePath,C=_.absolutePath();I=a.isAbsolute(A.includePath)||i.isWebPath(A.includePath)?C:a.isAbsolute(M)||i.isWebPath(M)?C:i.isWebPath(n)!=i.isWebPath(C)?C:n.length>0&&C.length>0&&n.charAt(0).toLowerCase()!=C.charAt(0).toLowerCase()?C:a.relative(n,C),I=I.replace(/\\/g,"/");var L=new d(w,_.unit,I);g[L.absolutePath()]||(o[b]=L,c.push(L),g[L.absolutePath()]=!0)}}u.forEach(function(e){if(e.valueKind()==s.Kind.INCLUDE_REF){var t=p.resolve(e.includePath());if(t){if(!t.isRAMLUnit())return;S(t.ast())}}else S(e)}),null==r.parent()&&(v[f]=!1)}},void S(h)):"continue"},_=0;_<c.length;_++){b()}for(var N={},w=0,R=Object.keys(o);w<R.length;w++){for(var I=R[w],A=o[I],M=A.namespace(),_=0;N[M];)M=A.namespace()+_++;M!=A.namespace()&&(A.namespaceSegments=M.split(".")),N[M]=!0}return o},e.prototype.pathMap=function(e){var t=e.absolutePath(),n=this.byPathMap[t];return void 0===n&&(n=this.calculateNamespaces(e),0==Object.keys(n).length&&(n=null),this.byPathMap[t]=n),n},e.prototype.nsMap=function(e){var t=e.absolutePath(),n=this.byNsMap[t];if(void 0===n){var r=this.pathMap(e);if(null==r)n=null;else{n={};for(var i=0,a=Object.keys(r);i<a.length;i++){var o=a[i],s=r[o];n[s.namespace()]=s}}this.byNsMap[t]=n}return n},e.prototype.calculateNamespaces=function(e){var t=a.dirname(e.absolutePath()),n={},r=e.ast(),u=r.children().filter(function(e){return e.key()==o.Universe10.FragmentDeclaration.properties.uses.name});if(r.actual()&&r.actual().usesNode&&(u=[r.actual().usesNode]),0==u.length)return n;for(var f=[],h=0,m=u;h<m.length;h++){var y=m[h];f=f.concat(y.children())}if(0==f.length)return n;for(var v=0,g=f;v<g.length;v++){var y=g[v],A=y.value(),E=e.resolve(A);if(null!=E){var T,u=[y],S=E.absolutePath();T=a.isAbsolute(A)||i.isWebPath(A)?E.absolutePath():a.relative(t,E.absolutePath()),T=T.replace(/\\/g,"/");var b=new d(u,E,T);n[S]=b}}var _=e.ast();if(_&&_.kind()!=s.Kind.SCALAR){var N=l.ramlFirstLine(e.contents());if(3==N.length&&(N[2]==p.universesInfo.Universe10.Overlay.name||N[2]==p.universesInfo.Universe10.Extension.name)){var w=c.find(_.children(),function(e){return e.key()==o.Universe10.Extension.properties["extends"].name});if(w){var R,I=w.value(!0);try{R=e.resolve(I)}catch(M){}if(R){var C=this.pathMap(R);if(C)for(var L=0,P=Object.keys(C);L<P.length;L++){var O=P[L];n[O]=C[O]}}}}}return n},e.prototype.lexLessEq=function(e,t){if(e.length>t.length)return!1;if(e.length<t.length)return!0;for(var n=0;n<e.length;n++){var r=e[n],i=t[n];if(i>r)return!0;if(r>i)return!1}return!0},e.prototype.hasFragments=function(e){if(!e.isRAMLUnit())return!1;var t=l.ramlFirstLine(e.contents());return!t||t.length<2||"1.0"!=t[1]?!1:(this.expandedPathMap(e),this._hasFragments[e.absolutePath()]?!0:!1)},e.prototype.unitModel=function(e){var t=e.absolutePath(),n=this._unitModels[t];return n||(n=new m(e),this._unitModels[t]=n),n},e}();t.NamespaceResolver=f;var d=function(){function e(e,t,n){this.usesNodes=e,this.unit=t,this.includePath=n,this.namespaceSegments=this.usesNodes.map(function(e){return e.key()})}return e.prototype.steps=function(){return this.namespaceSegments.length},e.prototype.namespace=function(){return this.namespaceSegments.join(".")},e.prototype.absolutePath=function(){return this.unit.absolutePath()},e}();t.UsesInfo=d;var h=function(){function e(e){this.name=e,this.array=[],this.map={}}return e.isInstance=function(t){return null!=t&&t.getClassIdentifier&&"function"==typeof t.getClassIdentifier&&c.contains(t.getClassIdentifier(),e.CLASS_IDENTIFIER)},e.prototype.getClassIdentifier=function(){var t=[];return t.concat(e.CLASS_IDENTIFIER)},e.prototype.addElement=function(e){this.array.push(e),this.map[e.key()]=e},e.prototype.hasElement=function(e){return null!=this.map[e]},e.prototype.getElement=function(e){return this.map[e]},e.prototype.isEmpty=function(){return 0==this.array.length},e.CLASS_IDENTIFIER="namespaceResolver.ElementsCollection",e}();t.ElementsCollection=h;var m=function(){function e(e){this.unit=e,this.initCollections()}return e.prototype.initCollections=function(){this.types=new h(p.universesInfo.Universe10.LibraryBase.properties.types.name),this.annotationTypes=new h(p.universesInfo.Universe10.LibraryBase.properties.annotationTypes.name),this.securitySchemes=new h(p.universesInfo.Universe10.LibraryBase.properties.securitySchemes.name),this.traits=new h(p.universesInfo.Universe10.LibraryBase.properties.traits.name),this.resourceTypes=new h(p.universesInfo.Universe10.LibraryBase.properties.resourceTypes.name);var e=this.unit.ast();if(e&&this.isLibraryBaseDescendant(this.unit))for(var t="0.8"==l.ramlFirstLine(this.unit.contents())[1],n=0,r=e.children();n<r.length;n++){var i=r[n],a=i.key();a==p.universesInfo.Universe10.LibraryBase.properties.types.name?this.contributeCollection(this.types,i.children()):a==p.universesInfo.Universe10.LibraryBase.properties.annotationTypes.name?this.contributeCollection(this.annotationTypes,i.children()):a==p.universesInfo.Universe10.LibraryBase.properties.securitySchemes.name?this.contributeCollection(this.securitySchemes,i.children(),t):a==p.universesInfo.Universe10.LibraryBase.properties.traits.name?this.contributeCollection(this.traits,i.children(),t):a==p.universesInfo.Universe10.LibraryBase.properties.resourceTypes.name&&this.contributeCollection(this.resourceTypes,i.children(),t)}},e.prototype.contributeCollection=function(e,t,n){void 0===n&&(n=!1);var r;n?(r=[],t.forEach(function(e){var t=e.children();t.length>0&&r.push(t[0])})):r=t,r.forEach(function(t){return e.addElement(t)})},e.prototype.isLibraryBaseDescendant=function(e){var t=this,n=l.ramlFirstLine(e.contents());if(!n)return!1;if(n.length<3||!n[2])return!0;var r=n[2];if(!this.libTypeDescendants){this.libTypeDescendants={};var i=p.getUniverse("RAML10"),a=i.type(p.universesInfo.Universe10.LibraryBase.name);
[a].concat(a.allSubTypes()).forEach(function(e){return t.libTypeDescendants[e.nameId()]=!0})}return this.libTypeDescendants[r]},e}();t.UnitModel=m},function(e,t,n){"use strict";function r(e){var t=e.value();if("string"==typeof t||null==t){var n=f.createNode(t,null,e.lowLevel().unit());n._actualNode().startPosition=e.lowLevel().valueStart(),n._actualNode().endPosition=e.lowLevel().valueEnd();var r=new c.StructuredValue(n,e.parent(),e.property());return r}return t}function i(e){var t=e._meta;t.resetPrimitiveValuesMeta();var n=e.highLevel();return n.definition().allProperties().forEach(function(r){var i=r.nameId(),a=n.attributes(i),o=!1,s=!1;if(a.forEach(function(e){o=o||null!=e.value(),s=s||e.optional()}),!o){var u=e.getDefaultsCalculator(),l=u.attributeDefaultIfEnabled(n,r);if(null!=l){var p=u.insertionKind(n,r);p==d.InsertionKind.CALCULATED?t.registerCalculatedValue(i):p==d.InsertionKind.BY_DEFAULT&&t.registerInsertedAsDefaultValue(i)}}}),t}function a(e,t){return t?e.map(function(e){return t(e)}):e.map(function(e){return e.value()})}function o(e){var t=[],n=e.errors();null!=n&&(t=t.concat(n));var r=t.map(function(t){return u(e,t)}),i=s(r);return i}function s(e){var t=[],n={};e.map(function(e){n[JSON.stringify(e)]=e});for(var r=Object.keys(n),i=0;i<r.length;i++)t.push(n[r[i]]);return t}function u(e,t){var n;n=t.unit?t.unit:t.node?t.node.lowLevel().unit():h.declRoot(e).lowLevel().unit();var r=n&&n.lineMapper()||e.lowLevel().unit().lineMapper(),i=null;try{i=r.position(t.start)}catch(a){console.warn(a)}var o=null;try{o=r.position(t.end)}catch(a){console.warn(a)}var s=t.path||n&&n.path(),l={code:t.code,message:t.message,path:s,range:{start:i,end:o},isWarning:t.isWarning};return t.extras&&t.extras.length>0&&(l.trace=t.extras.map(function(t){return u(e,t)})),l}function l(e){if(e.isScalar())return!0;for(var t=e.allSuperTypes().filter(function(e){return e.isUnion()||e.isIntersection()}),n=0,r=t;n<r.length;n++){for(var i=r[n],a=!0,o=0,s=i.allOptions();o<s.length;o++){var u=s[o];if(a=a||l(u),!a)break}if(a)return!0}return!1}var p=this&&this.__extends||function(e,t){function n(){this.constructor=e}for(var r in t)t.hasOwnProperty(r)&&(e[r]=t[r]);e.prototype=null===t?Object.create(t):(n.prototype=t.prototype,new n)},c=n(16),f=n(24),d=(n(39),n(58)),h=n(13),m=n(14),y=n(73),v=n(77),g=function(){function e(e,t){void 0===t&&(t=!0),this._node=e,this._meta=new _(!1,!1,m.isMethodType(this.highLevel().definition())&&this.optional()),t&&e.setWrapperNode(this)}return e.isInstance=function(t){if(null!=t&&t.getClassIdentifier&&"function"==typeof t.getClassIdentifier)for(var n=0,r=t.getClassIdentifier();n<r.length;n++){var i=r[n];if(i==e.CLASS_IDENTIFIER)return!0}return!1},e.prototype.getClassIdentifier=function(){var t=[];return t.concat(e.CLASS_IDENTIFIER)},e.prototype.wrapperClassName=function(){return"BasicNodeImpl"},e.prototype.kind=function(){return"BasicNode"},e.prototype.allKinds=function(){return["BasicNode"]},e.prototype.allWrapperClassNames=function(){return["BasicNodeImpl"]},e.prototype.parent=function(){var e=this._node.parent();return e?e.wrapperNode():null},e.prototype.highLevel=function(){return this._node},e.prototype.patchNode=function(e){this._node=e},e.prototype.attributes=function(e,t){var n=this._node.attributes(e);if(!n||0==n.length){var r=this.getDefaultsCalculator().attributeDefaultIfEnabled(this._node,this._node.definition().property(e));return null==r?[]:Array.isArray(r)?r:[r]}return a(n,t)},e.prototype.attribute=function(e,t){var n=this._node.attr(e);if(t&&t!=this.toString&&t!=this.toBoolean&&t!=this.toNumber&&t!=this.toAny)return null==n?null:t(n);var r=n?n.value():null;return null==r&&(r=this.getDefaultsCalculator().attributeDefaultIfEnabled(this._node,this._node.definition().property(e))),null==r?null:t?t(r):r},e.prototype.elements=function(e){var t=this._node.elementsOfKind(e);return t?t.map(function(e){return e.wrapperNode()}):null},e.prototype.element=function(e){var t=this._node.element(e);return t?t.wrapperNode():null},e.prototype.add=function(e){this.highLevel().add(e.highLevel())},e.prototype.addToProp=function(e,t){var n=e.highLevel(),r=this.highLevel().definition().property(t);n._prop=r,this.highLevel().add(n)},e.prototype.remove=function(e){this.highLevel().remove(e.highLevel())},e.prototype.dump=function(){return this.highLevel().dump("yaml")},e.prototype.toString=function(e){var t;return t=e&&e.lowLevel&&e.property?e.value():e,null!=t?t.toString():t},e.prototype.toAny=function(e){var t;if(e.lowLevel&&e.property){var n=e,r=n.value();t=r?r:new T(n.lowLevel())}else t=e;return c.StructuredValue.isInstance(t)&&(t=new T(t.lowLevel().parent())),t},e.prototype.toBoolean=function(e){var t;return t=e.lowLevel&&e.property?e.value():e,null!=t?"true"==t.toString():t},e.prototype.toNumber=function(e){var t;if(t=e.lowLevel&&e.property?e.value():e,!t)return t;try{var n=parseFloat(t.toString());return n}catch(r){}return Number.MAX_VALUE},e.prototype.errors=function(){return o(this._node)},e.prototype.definition=function(){return this.highLevel().definition()},e.prototype.runtimeDefinition=function(){return m.isTypeDeclarationSibling(this.highLevel().definition())?this.highLevel().localType():null},e.prototype.toJSON=function(e){var t=A;A=this.attributeDefaults();try{return new y.TCKDumper(e).dump(this)}finally{A=t}},e.prototype.optional=function(){var e=this.highLevel();return null!=e?e.optional():!1},e.prototype.optionalProperties=function(){return this.highLevel()?this.highLevel().optionalProperties():[]},e.prototype.getDefaultsCalculator=function(){return this.defaultsCalculator?this.defaultsCalculator:(this.parent()&&(this.defaultsCalculator=this.parent().getDefaultsCalculator()),this.defaultsCalculator||(this.defaultsCalculator=new d.AttributeDefaultsCalculator(A)),this.defaultsCalculator)},e.prototype.setAttributeDefaults=function(e){this.defaultsCalculator=new d.AttributeDefaultsCalculator(e)},e.prototype.attributeDefaults=function(){return this.getDefaultsCalculator()&&this.getDefaultsCalculator().isEnabled()},e.prototype.meta=function(){return i(this)},e.prototype.RAMLVersion=function(){return this.highLevel().definition().universe().version()},e.CLASS_IDENTIFIER="parserCore.BasicNodeImpl",e}();t.BasicNodeImpl=g;var A=!0,E=function(){function e(e){this._meta=new b,this.attr=e}return e.isInstance=function(t){if(null!=t&&t.getClassIdentifier&&"function"==typeof t.getClassIdentifier)for(var n=0,r=t.getClassIdentifier();n<r.length;n++){var i=r[n];if(i==e.CLASS_IDENTIFIER)return!0}return!1},e.prototype.getClassIdentifier=function(){var t=[];return t.concat(e.CLASS_IDENTIFIER)},e.prototype.highLevel=function(){return this.attr},e.prototype.wrapperClassName=function(){return"AttributeNodeImpl"},e.prototype.kind=function(){return"AttributeNode"},e.prototype.allKinds=function(){return["AttributeNode"]},e.prototype.allWrapperClassNames=function(){return["AttributeNodeImpl"]},e.prototype.optional=function(){var e=this.highLevel();return null!=e?e.optional():!1},e.prototype.meta=function(){return this._meta},e.prototype.RAMLVersion=function(){return this.highLevel().definition().universe().version()},e.prototype.parent=function(){var e=this.attr.parent();return e?e.wrapperNode():null},e.prototype.toJSON=function(e){return new y.TCKDumper(e).dump(this)},e.CLASS_IDENTIFIER="parserCore.AttributeNodeImpl",e}();t.AttributeNodeImpl=E,t.toStructuredValue=r;var T=function(){function e(e,t){this.type=t,Array.isArray(e)?this.children=e:this.node=e}return e.isInstance=function(t){if(null!=t&&t.getClassIdentifier&&"function"==typeof t.getClassIdentifier)for(var n=0,r=t.getClassIdentifier();n<r.length;n++){var i=r[n];if(i==e.CLASS_IDENTIFIER)return!0}return!1},e.prototype.getClassIdentifier=function(){var t=[];return t.concat(e.CLASS_IDENTIFIER)},e.prototype.properties=function(){return this.isArray()?[]:this.getChildren().map(function(e){return new S(e)})},e.prototype.getChildren=function(){return this.node&&this.node.children()||this.children},e.prototype.value=function(){return this.node&&this.node.value()},e.prototype.isScalar=function(){return this.node?0!=this.node.children().length?!1:this.type?l(this.type):!0:!1},e.prototype.toJSON=function(){return(new y.TCKDumper).serializeTypeInstance(this)},e.prototype.isArray=function(){return this.node?this.type?this.type.isArray():this.node&&this.node.valueKind()==v.Kind.SEQ:!1},e.prototype.items=function(){return this.isArray()?this.node.children().map(function(t){return new e(t)}):null},e.CLASS_IDENTIFIER="parserCore.TypeInstanceImpl",e}();t.TypeInstanceImpl=T;var S=function(){function e(e){this.node=e}return e.isInstance=function(t){if(null!=t&&t.getClassIdentifier&&"function"==typeof t.getClassIdentifier)for(var n=0,r=t.getClassIdentifier();n<r.length;n++){var i=r[n];if(i==e.CLASS_IDENTIFIER)return!0}return!1},e.prototype.getClassIdentifier=function(){var t=[];return t.concat(e.CLASS_IDENTIFIER)},e.prototype.name=function(){return this.node.optional()?this.node.key()+"?":this.node.key()},e.prototype.value=function(){if(this.isArray()){var e=this.node.children();return e.length>0?new T(e[0]):null}return new T(this.node)},e.prototype.values=function(){return this.isArray()?this.node.children().map(function(e){return new T(e)}):[new T(this.node)]},e.prototype.isArray=function(){var e=this.node.children();if(e.length>0&&null==e[0].key())return!0;var t=this.node.highLevelNode();if(t){var n=t.property();if(n){var r=n.range();if(r)return r.isArray()}}return this.node.valueKind()==v.Kind.SEQ},e.CLASS_IDENTIFIER="parserCore.TypeInstancePropertyImpl",e}();t.TypeInstancePropertyImpl=S;var b=function(){function e(e,t,n){void 0===e&&(e=!1),void 0===t&&(t=!1),void 0===n&&(n=!1),this._insertedAsDefault=e,this._calculated=t,this._optional=n}return e.prototype.calculated=function(){return this._calculated},e.prototype.insertedAsDefault=function(){return this._insertedAsDefault},e.prototype.setCalculated=function(){this._calculated=!0},e.prototype.setInsertedAsDefault=function(){this._insertedAsDefault=!0},e.prototype.setOptional=function(){this._optional=!0},e.prototype.optional=function(){return this._optional},e.prototype.isDefault=function(){return!(this._insertedAsDefault||this._calculated||this._optional)},e.prototype.toJSON=function(){var e={};return this._calculated&&(e.calculated=!0),this._insertedAsDefault&&(e.insertedAsDefault=!0),this._optional&&(e.optional=!0),e},e}();t.ValueMetadataImpl=b;var _=function(e){function t(){e.apply(this,arguments),this.valuesMeta={}}return p(t,e),t.prototype.primitiveValuesMeta=function(){return this.valuesMeta},t.prototype.registerInsertedAsDefaultValue=function(e){var t=this.valuesMeta[e];null==t?this.valuesMeta[e]=new b(!0):t.setInsertedAsDefault()},t.prototype.registerCalculatedValue=function(e){var t=this.valuesMeta[e];null==t?this.valuesMeta[e]=new b(!1,!0):t.setCalculated()},t.prototype.registerOptionalValue=function(e){var t=this.valuesMeta[e];null==t?this.valuesMeta[e]=new b(!1,!1,!0):t.setOptional()},t.prototype.resetPrimitiveValuesMeta=function(){this.valuesMeta={}},t.prototype.isDefault=function(){return e.prototype.isDefault.call(this)?0==Object.keys(this.valuesMeta).length:!1},t.prototype.toJSON=function(){var t=this,n=e.prototype.toJSON.call(this),r={},i=Object.keys(this.valuesMeta);return i.length>0&&(i.forEach(function(e){var n=t.valuesMeta[e].toJSON();Object.keys(n).length>0&&(r[e]=n)}),n.primitiveValuesMeta=r),n},t}(b);t.NodeMetadataImpl=_,t.fillElementMeta=i,t.attributesToValues=a,t.errors=o,t.filterErrors=s,t.basicError=u},function(e,t,n){"use strict";function r(e){var t=C.getUniverse("RAML08"),n=t.type("Method"),r=M.createStubNode(n,null,e);return r}function i(e){var t=C.getUniverse("RAML08"),n=t.type("StringTypeDeclaration"),r=M.createStubNode(n,null,e);return r}function a(e){var t=C.getUniverse("RAML08"),n=t.type("BooleanTypeDeclaration"),r=M.createStubNode(n,null,e);return r}function o(e){var t=C.getUniverse("RAML08"),n=t.type("NumberTypeDeclaration"),r=M.createStubNode(n,null,e);return r}function s(e){var t=C.getUniverse("RAML08"),n=t.type("IntegerTypeDeclaration"),r=M.createStubNode(n,null,e);return r}function u(e){var t=C.getUniverse("RAML08"),n=t.type("DateTypeDeclaration"),r=M.createStubNode(n,null,e);return r}function l(e){var t=C.getUniverse("RAML08"),n=t.type("FileTypeDeclaration"),r=M.createStubNode(n,null,e);return r}function p(e){var t=C.getUniverse("RAML08"),n=t.type("XMLBody"),r=M.createStubNode(n,null,e);return r}function c(e){var t=C.getUniverse("RAML08"),n=t.type("JSONBody"),r=M.createStubNode(n,null,e);return r}function f(e){var t=C.getUniverse("RAML08"),n=t.type("SecuritySchemePart"),r=M.createStubNode(n,null,e);return r}function d(e){var t=C.getUniverse("RAML08"),n=t.type("Trait"),r=M.createStubNode(n,null,e);return r}function h(e){var t=C.getUniverse("RAML08"),n=t.type("OAuth1SecuritySchemeSettings"),r=M.createStubNode(n,null,e);return r}function m(e){var t=C.getUniverse("RAML08"),n=t.type("OAuth2SecuritySchemeSettings"),r=M.createStubNode(n,null,e);return r}function y(e){var t=C.getUniverse("RAML08"),n=t.type("OAuth2SecurityScheme"),r=M.createStubNode(n,null,e);return r}function v(e){var t=C.getUniverse("RAML08"),n=t.type("OAuth1SecurityScheme"),r=M.createStubNode(n,null,e);return r}function g(e){var t=C.getUniverse("RAML08"),n=t.type("BasicSecurityScheme"),r=M.createStubNode(n,null,e);return r}function A(e){var t=C.getUniverse("RAML08"),n=t.type("DigestSecurityScheme"),r=M.createStubNode(n,null,e);return r}function E(e){var t=C.getUniverse("RAML08"),n=t.type("CustomSecurityScheme"),r=M.createStubNode(n,null,e);return r}function T(e){var t=C.getUniverse("RAML08"),n=t.type("GlobalSchema"),r=M.createStubNode(n,null,e);return r}function S(e){var t=C.getUniverse("RAML08"),n=t.type("DocumentationItem"),r=M.createStubNode(n,null,e);return r}function b(e,t,n){return P.loadApi(e,t,n).getOrElse(null)}function _(e,t,n){return P.loadApi(e,t,n).getOrElse(null)}function N(e,t,n){return P.loadApiAsync(e,t,n)}function w(e,t,n){return P.loadRAMLAsync(e,t,n)}function R(e){return P.getLanguageElementByRuntimeType(e)}var I=this&&this.__extends||function(e,t){function n(){this.constructor=e}for(var r in t)t.hasOwnProperty(r)&&(e[r]=t[r]);e.prototype=null===t?Object.create(t):(n.prototype=t.prototype,new n)},M=n(17),C=n(39),L=n(51),P=n(8),O=n(79),D=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.title=function(){return e.prototype.attribute.call(this,"title",this.toString)},t.prototype.setTitle=function(e){return this.highLevel().attrOrCreate("title").setValue(""+e),this},t.prototype.version=function(){return e.prototype.attribute.call(this,"version",this.toString)},t.prototype.setVersion=function(e){return this.highLevel().attrOrCreate("version").setValue(""+e),this},t.prototype.baseUri=function(){return e.prototype.attribute.call(this,"baseUri",function(e){return new Ie(e)})},t.prototype.baseUriParameters_original=function(){return e.prototype.elements.call(this,"baseUriParameters")},t.prototype.uriParameters=function(){return e.prototype.elements.call(this,"uriParameters")},t.prototype.protocols=function(){return e.prototype.attributes.call(this,"protocols",this.toString)},t.prototype.setProtocols=function(e){return this.highLevel().attrOrCreate("protocols").setValue(""+e),this},t.prototype.mediaType=function(){return e.prototype.attribute.call(this,"mediaType",function(e){return new Ne(e)})},t.prototype.schemas=function(){return e.prototype.elements.call(this,"schemas")},t.prototype.traits_original=function(){return e.prototype.elements.call(this,"traits")},t.prototype.securedBy=function(){return e.prototype.attributes.call(this,"securedBy",function(e){return new pe(e)})},t.prototype.securitySchemes=function(){return e.prototype.elements.call(this,"securitySchemes")},t.prototype.resourceTypes_original=function(){return e.prototype.elements.call(this,"resourceTypes")},t.prototype.resources=function(){return e.prototype.elements.call(this,"resources")},t.prototype.documentation=function(){return e.prototype.elements.call(this,"documentation")},t.prototype.wrapperClassName=function(){return"ApiImpl"},t.prototype.kind=function(){return"Api"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("Api")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.ApiImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.ApiImpl"==r)return!0}return!1},t.prototype.RAMLVersion_original=function(){return"RAML08"},t.prototype.expand=function(){return O.expandTraitsAndResourceTypes(this)},t.prototype.traits=function(){return O.traitsPrimary(this)},t.prototype.allTraits=function(){return O.allTraits(this)},t.prototype.resourceTypes=function(){return O.resourceTypesPrimary(this)},t.prototype.allResourceTypes=function(){return O.allResourceTypes(this)},t.prototype.childResource=function(e){return O.childResource(this,e)},t.prototype.allResources=function(){return O.allResources(this)},t.prototype.baseUriParameters=function(){return O.baseUriParametersPrimary(this)},t.prototype.allBaseUriParameters=function(){return O.baseUriParameters(this)},t.prototype.allProtocols=function(){return O.allProtocols(this)},t.prototype.RAMLVersion=function(){return O.RAMLVersion(this)},t}(L.BasicNodeImpl);t.ApiImpl=D;var x=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.wrapperClassName=function(){return"ValueTypeImpl"},t.prototype.kind=function(){return"ValueType"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("ValueType")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("undefined.ValueTypeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("undefined.ValueTypeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"undefined"},t.prototype.value=function(){return this.attr.value()},t}(L.AttributeNodeImpl);t.ValueTypeImpl=x;var U=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.wrapperClassName=function(){return"AnyTypeImpl"},t.prototype.kind=function(){return"AnyType"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("AnyType")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.AnyTypeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.AnyTypeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(x);t.AnyTypeImpl=U;var k=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.wrapperClassName=function(){return"NumberTypeImpl"},t.prototype.kind=function(){return"NumberType"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("NumberType")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.NumberTypeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.NumberTypeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t.prototype.value=function(){return this.attr.value()},t}(x);t.NumberTypeImpl=k;var F=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.wrapperClassName=function(){return"BooleanTypeImpl"},t.prototype.kind=function(){return"BooleanType"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("BooleanType")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.BooleanTypeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.BooleanTypeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t.prototype.value=function(){return this.attr.value()},t}(x);t.BooleanTypeImpl=F;var B=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.wrapperClassName=function(){return"ReferenceImpl"},t.prototype.kind=function(){return"Reference"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("Reference")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.ReferenceImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.ReferenceImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t.prototype.value=function(){return L.toStructuredValue(this.attr)},t.prototype.structuredValue=function(){return O.structuredValue(this)},t.prototype.name=function(){return O.referenceName(this)},t}(L.AttributeNodeImpl);t.ReferenceImpl=B;var K=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.wrapperClassName=function(){return"ResourceTypeRefImpl"},t.prototype.kind=function(){return"ResourceTypeRef"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("ResourceTypeRef")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.ResourceTypeRefImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.ResourceTypeRefImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t.prototype.resourceType=function(){return O.referencedResourceType(this)},t}(B);t.ResourceTypeRefImpl=K;var V=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.name=function(){return e.prototype.attribute.call(this,"name",this.toString)},t.prototype.setName=function(e){return this.highLevel().attrOrCreate("name").setValue(""+e),this},t.prototype.usage=function(){return e.prototype.attribute.call(this,"usage",this.toString)},t.prototype.setUsage=function(e){return this.highLevel().attrOrCreate("usage").setValue(""+e),this},t.prototype.methods=function(){return e.prototype.elements.call(this,"methods")},t.prototype.is=function(){return e.prototype.attributes.call(this,"is",function(e){return new de(e)})},t.prototype.type=function(){return e.prototype.attribute.call(this,"type",function(e){return new K(e)})},t.prototype.securedBy=function(){return e.prototype.attributes.call(this,"securedBy",function(e){return new pe(e)})},t.prototype.uriParameters=function(){return e.prototype.elements.call(this,"uriParameters")},t.prototype.displayName=function(){return e.prototype.attribute.call(this,"displayName",this.toString)},t.prototype.setDisplayName=function(e){return this.highLevel().attrOrCreate("displayName").setValue(""+e),this},t.prototype.baseUriParameters=function(){return e.prototype.elements.call(this,"baseUriParameters")},t.prototype.description=function(){return e.prototype.attribute.call(this,"description",function(e){return new G(e)})},t.prototype.wrapperClassName=function(){return"ResourceTypeImpl"},t.prototype.kind=function(){return"ResourceType"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("ResourceType")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.ResourceTypeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.ResourceTypeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t.prototype.parametrizedProperties=function(){return O.getTemplateParametrizedProperties(this)},t}(L.BasicNodeImpl);t.ResourceTypeImpl=V;var j=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.responses=function(){return e.prototype.elements.call(this,"responses")},t.prototype.body=function(){return e.prototype.elements.call(this,"body")},t.prototype.protocols=function(){return e.prototype.attributes.call(this,"protocols",this.toString)},t.prototype.setProtocols=function(e){return this.highLevel().attrOrCreate("protocols").setValue(""+e),this},t.prototype.securedBy=function(){return e.prototype.attributes.call(this,"securedBy",function(e){return new pe(e)})},t.prototype.baseUriParameters=function(){return e.prototype.elements.call(this,"baseUriParameters")},t.prototype.queryParameters=function(){return e.prototype.elements.call(this,"queryParameters")},t.prototype.headers=function(){return e.prototype.elements.call(this,"headers")},t.prototype.description=function(){return e.prototype.attribute.call(this,"description",function(e){return new G(e)})},t.prototype.wrapperClassName=function(){return"MethodBaseImpl"},t.prototype.kind=function(){return"MethodBase"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("MethodBase")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.MethodBaseImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.MethodBaseImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(L.BasicNodeImpl);t.MethodBaseImpl=j;var W=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.code=function(){return e.prototype.attribute.call(this,"code",function(e){return new Y(e)})},t.prototype.headers=function(){return e.prototype.elements.call(this,"headers")},t.prototype.body=function(){return e.prototype.elements.call(this,"body")},t.prototype.description=function(){return e.prototype.attribute.call(this,"description",function(e){return new G(e)})},t.prototype.wrapperClassName=function(){return"ResponseImpl"},t.prototype.kind=function(){return"Response"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("Response")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.ResponseImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.ResponseImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t.prototype.isOkRange=function(){return O.isOkRange(this)},t}(L.BasicNodeImpl);t.ResponseImpl=W;var q=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.wrapperClassName=function(){return"StringTypeImpl"},t.prototype.kind=function(){return"StringType"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("StringType")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.StringTypeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.StringTypeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t.prototype.value=function(){return this.attr.value()},t}(x);t.StringTypeImpl=q;var Y=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.wrapperClassName=function(){return"StatusCodeStringImpl"},t.prototype.kind=function(){return"StatusCodeString"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("StatusCodeString")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.StatusCodeStringImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.StatusCodeStringImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(q);t.StatusCodeStringImpl=Y;var H=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.name=function(){return e.prototype.attribute.call(this,"name",this.toString)},t.prototype.setName=function(e){return this.highLevel().attrOrCreate("name").setValue(""+e),this},t.prototype.displayName=function(){return e.prototype.attribute.call(this,"displayName",this.toString)},t.prototype.setDisplayName=function(e){return this.highLevel().attrOrCreate("displayName").setValue(""+e),this},t.prototype.type=function(){return e.prototype.attribute.call(this,"type",this.toString)},t.prototype.setType=function(e){return this.highLevel().attrOrCreate("type").setValue(""+e),this},t.prototype.location=function(){return e.prototype.attribute.call(this,"location",function(e){return new $(e)})},t.prototype.required=function(){return e.prototype.attribute.call(this,"required",this.toBoolean)},t.prototype.setRequired=function(e){return this.highLevel().attrOrCreate("required").setValue(""+e),this},t.prototype["default"]=function(){return e.prototype.attribute.call(this,"default",this.toAny)},t.prototype.setDefault=function(e){return this.highLevel().attrOrCreate("default").setValue(""+e),this},t.prototype.example=function(){return e.prototype.attribute.call(this,"example",this.toString)},t.prototype.setExample=function(e){return this.highLevel().attrOrCreate("example").setValue(""+e),this},t.prototype.repeat=function(){return e.prototype.attribute.call(this,"repeat",this.toBoolean)},t.prototype.setRepeat=function(e){return this.highLevel().attrOrCreate("repeat").setValue(""+e),this},t.prototype.description=function(){return e.prototype.attribute.call(this,"description",function(e){return new G(e)})},t.prototype.wrapperClassName=function(){return"ParameterImpl"},t.prototype.kind=function(){return"Parameter"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("Parameter")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.ParameterImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.ParameterImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(L.BasicNodeImpl);t.ParameterImpl=H;var $=function(){function e(e){this.attr=e}return e.prototype.wrapperClassName=function(){return"ParameterLocationImpl"},e.prototype.kind=function(){return"ParameterLocation"},e.prototype.allKinds=function(){return["ParameterLocation"]},e.prototype.allWrapperClassNames=function(){return["RAML08.ParameterLocationImpl"]},e.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.ParameterLocationImpl"==r)return!0}return!1},e.prototype.RAMLVersion=function(){return"RAML08"},e}();t.ParameterLocationImpl=$;var G=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.wrapperClassName=function(){return"MarkdownStringImpl"},t.prototype.kind=function(){return"MarkdownString"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("MarkdownString")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.MarkdownStringImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.MarkdownStringImpl"==r)return!0;
}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(q);t.MarkdownStringImpl=G;var X=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?i(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return I(t,e),t.prototype.pattern=function(){return e.prototype.attribute.call(this,"pattern",this.toString)},t.prototype.setPattern=function(e){return this.highLevel().attrOrCreate("pattern").setValue(""+e),this},t.prototype["enum"]=function(){return e.prototype.attributes.call(this,"enum",this.toString)},t.prototype.setEnum=function(e){return this.highLevel().attrOrCreate("enum").setValue(""+e),this},t.prototype.minLength=function(){return e.prototype.attribute.call(this,"minLength",this.toNumber)},t.prototype.setMinLength=function(e){return this.highLevel().attrOrCreate("minLength").setValue(""+e),this},t.prototype.maxLength=function(){return e.prototype.attribute.call(this,"maxLength",this.toNumber)},t.prototype.setMaxLength=function(e){return this.highLevel().attrOrCreate("maxLength").setValue(""+e),this},t.prototype.wrapperClassName=function(){return"StringTypeDeclarationImpl"},t.prototype.kind=function(){return"StringTypeDeclaration"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("StringTypeDeclaration")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.StringTypeDeclarationImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.StringTypeDeclarationImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(H);t.StringTypeDeclarationImpl=X;var z=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?a(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return I(t,e),t.prototype.wrapperClassName=function(){return"BooleanTypeDeclarationImpl"},t.prototype.kind=function(){return"BooleanTypeDeclaration"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("BooleanTypeDeclaration")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.BooleanTypeDeclarationImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.BooleanTypeDeclarationImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(H);t.BooleanTypeDeclarationImpl=z;var J=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?o(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return I(t,e),t.prototype.minimum=function(){return e.prototype.attribute.call(this,"minimum",this.toNumber)},t.prototype.setMinimum=function(e){return this.highLevel().attrOrCreate("minimum").setValue(""+e),this},t.prototype.maximum=function(){return e.prototype.attribute.call(this,"maximum",this.toNumber)},t.prototype.setMaximum=function(e){return this.highLevel().attrOrCreate("maximum").setValue(""+e),this},t.prototype.wrapperClassName=function(){return"NumberTypeDeclarationImpl"},t.prototype.kind=function(){return"NumberTypeDeclaration"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("NumberTypeDeclaration")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.NumberTypeDeclarationImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.NumberTypeDeclarationImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(H);t.NumberTypeDeclarationImpl=J;var Q=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?s(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return I(t,e),t.prototype.wrapperClassName=function(){return"IntegerTypeDeclarationImpl"},t.prototype.kind=function(){return"IntegerTypeDeclaration"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("IntegerTypeDeclaration")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.IntegerTypeDeclarationImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.IntegerTypeDeclarationImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(J);t.IntegerTypeDeclarationImpl=Q;var Z=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?u(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return I(t,e),t.prototype.wrapperClassName=function(){return"DateTypeDeclarationImpl"},t.prototype.kind=function(){return"DateTypeDeclaration"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("DateTypeDeclaration")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.DateTypeDeclarationImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.DateTypeDeclarationImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(H);t.DateTypeDeclarationImpl=Z;var ee=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?l(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return I(t,e),t.prototype.wrapperClassName=function(){return"FileTypeDeclarationImpl"},t.prototype.kind=function(){return"FileTypeDeclaration"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("FileTypeDeclaration")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.FileTypeDeclarationImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.FileTypeDeclarationImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(H);t.FileTypeDeclarationImpl=ee;var te=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.name=function(){return e.prototype.attribute.call(this,"name",this.toString)},t.prototype.setName=function(e){return this.highLevel().attrOrCreate("name").setValue(""+e),this},t.prototype.schema=function(){return e.prototype.attribute.call(this,"schema",function(e){return new ne(e)})},t.prototype.example=function(){return e.prototype.attribute.call(this,"example",function(e){return new ae(e)})},t.prototype.formParameters=function(){return e.prototype.elements.call(this,"formParameters")},t.prototype.description=function(){return e.prototype.attribute.call(this,"description",function(e){return new G(e)})},t.prototype.wrapperClassName=function(){return"BodyLikeImpl"},t.prototype.kind=function(){return"BodyLike"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("BodyLike")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.BodyLikeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.BodyLikeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t.prototype.schemaContent=function(){return O.schemaContent(this)},t}(L.BasicNodeImpl);t.BodyLikeImpl=te;var ne=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.wrapperClassName=function(){return"SchemaStringImpl"},t.prototype.kind=function(){return"SchemaString"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("SchemaString")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.SchemaStringImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.SchemaStringImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(q);t.SchemaStringImpl=ne;var re=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.wrapperClassName=function(){return"JSonSchemaStringImpl"},t.prototype.kind=function(){return"JSonSchemaString"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("JSonSchemaString")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.JSonSchemaStringImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.JSonSchemaStringImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(ne);t.JSonSchemaStringImpl=re;var ie=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.wrapperClassName=function(){return"XMLSchemaStringImpl"},t.prototype.kind=function(){return"XMLSchemaString"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("XMLSchemaString")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.XMLSchemaStringImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.XMLSchemaStringImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(ne);t.XMLSchemaStringImpl=ie;var ae=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.wrapperClassName=function(){return"ExampleStringImpl"},t.prototype.kind=function(){return"ExampleString"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("ExampleString")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.ExampleStringImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.ExampleStringImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(q);t.ExampleStringImpl=ae;var oe=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.wrapperClassName=function(){return"JSONExampleImpl"},t.prototype.kind=function(){return"JSONExample"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("JSONExample")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.JSONExampleImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.JSONExampleImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(ae);t.JSONExampleImpl=oe;var se=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.wrapperClassName=function(){return"XMLExampleImpl"},t.prototype.kind=function(){return"XMLExample"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("XMLExample")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.XMLExampleImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.XMLExampleImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(ae);t.XMLExampleImpl=se;var ue=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?p(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return I(t,e),t.prototype.schema=function(){return e.prototype.attribute.call(this,"schema",function(e){return new ie(e)})},t.prototype.wrapperClassName=function(){return"XMLBodyImpl"},t.prototype.kind=function(){return"XMLBody"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("XMLBody")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.XMLBodyImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.XMLBodyImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(te);t.XMLBodyImpl=ue;var le=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?c(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return I(t,e),t.prototype.schema=function(){return e.prototype.attribute.call(this,"schema",function(e){return new re(e)})},t.prototype.wrapperClassName=function(){return"JSONBodyImpl"},t.prototype.kind=function(){return"JSONBody"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("JSONBody")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.JSONBodyImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.JSONBodyImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(te);t.JSONBodyImpl=le;var pe=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.wrapperClassName=function(){return"SecuritySchemeRefImpl"},t.prototype.kind=function(){return"SecuritySchemeRef"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("SecuritySchemeRef")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.SecuritySchemeRefImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.SecuritySchemeRefImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t.prototype.securitySchemeName=function(){return O.securitySchemeName(this)},t.prototype.securityScheme=function(){return O.securityScheme(this)},t}(B);t.SecuritySchemeRefImpl=pe;var ce=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.name=function(){return e.prototype.attribute.call(this,"name",this.toString)},t.prototype.setName=function(e){return this.highLevel().attrOrCreate("name").setValue(""+e),this},t.prototype.type=function(){return e.prototype.attribute.call(this,"type",this.toString)},t.prototype.setType=function(e){return this.highLevel().attrOrCreate("type").setValue(""+e),this},t.prototype.description=function(){return e.prototype.attribute.call(this,"description",function(e){return new G(e)})},t.prototype.describedBy=function(){return e.prototype.element.call(this,"describedBy")},t.prototype.settings=function(){return e.prototype.element.call(this,"settings")},t.prototype.wrapperClassName=function(){return"AbstractSecuritySchemeImpl"},t.prototype.kind=function(){return"AbstractSecurityScheme"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("AbstractSecurityScheme")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.AbstractSecuritySchemeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.AbstractSecuritySchemeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(L.BasicNodeImpl);t.AbstractSecuritySchemeImpl=ce;var fe=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?f(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return I(t,e),t.prototype.displayName=function(){return e.prototype.attribute.call(this,"displayName",this.toString)},t.prototype.setDisplayName=function(e){return this.highLevel().attrOrCreate("displayName").setValue(""+e),this},t.prototype.is=function(){return e.prototype.attributes.call(this,"is",function(e){return new de(e)})},t.prototype.headers=function(){return e.prototype.elements.call(this,"headers")},t.prototype.queryParameters=function(){return e.prototype.elements.call(this,"queryParameters")},t.prototype.responses=function(){return e.prototype.elements.call(this,"responses")},t.prototype.description=function(){return e.prototype.attribute.call(this,"description",function(e){return new G(e)})},t.prototype.wrapperClassName=function(){return"SecuritySchemePartImpl"},t.prototype.kind=function(){return"SecuritySchemePart"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("SecuritySchemePart")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.SecuritySchemePartImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.SecuritySchemePartImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(j);t.SecuritySchemePartImpl=fe;var de=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.wrapperClassName=function(){return"TraitRefImpl"},t.prototype.kind=function(){return"TraitRef"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("TraitRef")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.TraitRefImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.TraitRefImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t.prototype.trait=function(){return O.referencedTrait(this)},t}(B);t.TraitRefImpl=de;var he=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?d(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return I(t,e),t.prototype.name=function(){return e.prototype.attribute.call(this,"name",this.toString)},t.prototype.setName=function(e){return this.highLevel().attrOrCreate("name").setValue(""+e),this},t.prototype.usage=function(){return e.prototype.attribute.call(this,"usage",this.toString)},t.prototype.setUsage=function(e){return this.highLevel().attrOrCreate("usage").setValue(""+e),this},t.prototype.displayName=function(){return e.prototype.attribute.call(this,"displayName",this.toString)},t.prototype.setDisplayName=function(e){return this.highLevel().attrOrCreate("displayName").setValue(""+e),this},t.prototype.wrapperClassName=function(){return"TraitImpl"},t.prototype.kind=function(){return"Trait"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("Trait")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.TraitImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.TraitImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t.prototype.parametrizedProperties=function(){return O.getTemplateParametrizedProperties(this)},t}(j);t.TraitImpl=he;var me=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.wrapperClassName=function(){return"SecuritySchemeSettingsImpl"},t.prototype.kind=function(){return"SecuritySchemeSettings"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("SecuritySchemeSettings")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.SecuritySchemeSettingsImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.SecuritySchemeSettingsImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(L.BasicNodeImpl);t.SecuritySchemeSettingsImpl=me;var ye=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?h(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return I(t,e),t.prototype.requestTokenUri=function(){return e.prototype.attribute.call(this,"requestTokenUri",function(e){return new ve(e)})},t.prototype.authorizationUri=function(){return e.prototype.attribute.call(this,"authorizationUri",function(e){return new ve(e)})},t.prototype.tokenCredentialsUri=function(){return e.prototype.attribute.call(this,"tokenCredentialsUri",function(e){return new ve(e)})},t.prototype.wrapperClassName=function(){return"OAuth1SecuritySchemeSettingsImpl"},t.prototype.kind=function(){return"OAuth1SecuritySchemeSettings"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("OAuth1SecuritySchemeSettings")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.OAuth1SecuritySchemeSettingsImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.OAuth1SecuritySchemeSettingsImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(me);t.OAuth1SecuritySchemeSettingsImpl=ye;var ve=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.wrapperClassName=function(){return"FixedUriImpl"},t.prototype.kind=function(){return"FixedUri"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("FixedUri")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.FixedUriImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.FixedUriImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(q);t.FixedUriImpl=ve;var ge=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?m(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return I(t,e),t.prototype.accessTokenUri=function(){return e.prototype.attribute.call(this,"accessTokenUri",function(e){return new ve(e)})},t.prototype.authorizationUri=function(){return e.prototype.attribute.call(this,"authorizationUri",function(e){return new ve(e)})},t.prototype.authorizationGrants=function(){return e.prototype.attributes.call(this,"authorizationGrants",this.toString)},t.prototype.setAuthorizationGrants=function(e){return this.highLevel().attrOrCreate("authorizationGrants").setValue(""+e),this},t.prototype.scopes=function(){return e.prototype.attributes.call(this,"scopes",this.toString)},t.prototype.setScopes=function(e){return this.highLevel().attrOrCreate("scopes").setValue(""+e),this},t.prototype.wrapperClassName=function(){return"OAuth2SecuritySchemeSettingsImpl"},t.prototype.kind=function(){return"OAuth2SecuritySchemeSettings"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("OAuth2SecuritySchemeSettings")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.OAuth2SecuritySchemeSettingsImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.OAuth2SecuritySchemeSettingsImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(me);t.OAuth2SecuritySchemeSettingsImpl=ge;var Ae=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?y(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return I(t,e),t.prototype.settings=function(){return e.prototype.element.call(this,"settings")},t.prototype.wrapperClassName=function(){return"OAuth2SecuritySchemeImpl"},t.prototype.kind=function(){return"OAuth2SecurityScheme"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("OAuth2SecurityScheme")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.OAuth2SecuritySchemeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.OAuth2SecuritySchemeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(ce);t.OAuth2SecuritySchemeImpl=Ae;var Ee=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?v(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return I(t,e),t.prototype.settings=function(){return e.prototype.element.call(this,"settings")},t.prototype.wrapperClassName=function(){return"OAuth1SecuritySchemeImpl"},t.prototype.kind=function(){return"OAuth1SecurityScheme"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("OAuth1SecurityScheme")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.OAuth1SecuritySchemeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.OAuth1SecuritySchemeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(ce);t.OAuth1SecuritySchemeImpl=Ee;var Te=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?g(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return I(t,e),t.prototype.wrapperClassName=function(){return"BasicSecuritySchemeImpl"},t.prototype.kind=function(){return"BasicSecurityScheme"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("BasicSecurityScheme")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.BasicSecuritySchemeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.BasicSecuritySchemeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(ce);t.BasicSecuritySchemeImpl=Te;var Se=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?A(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return I(t,e),t.prototype.wrapperClassName=function(){return"DigestSecuritySchemeImpl"},t.prototype.kind=function(){return"DigestSecurityScheme"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("DigestSecurityScheme")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.DigestSecuritySchemeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.DigestSecuritySchemeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(ce);t.DigestSecuritySchemeImpl=Se;var be=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?E(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return I(t,e),t.prototype.wrapperClassName=function(){return"CustomSecuritySchemeImpl"},t.prototype.kind=function(){return"CustomSecurityScheme"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("CustomSecurityScheme")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.CustomSecuritySchemeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.CustomSecuritySchemeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(ce);t.CustomSecuritySchemeImpl=be;var _e=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?r(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return I(t,e),t.prototype.method=function(){return e.prototype.attribute.call(this,"method",this.toString)},t.prototype.setMethod=function(e){return this.highLevel().attrOrCreate("method").setValue(""+e),this},t.prototype.is=function(){return e.prototype.attributes.call(this,"is",function(e){return new de(e)})},t.prototype.wrapperClassName=function(){return"MethodImpl"},t.prototype.kind=function(){return"Method"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("Method")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.MethodImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.MethodImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t.prototype.parentResource=function(){return O.parentResource(this)},t.prototype.ownerApi=function(){return O.ownerApi(this)},t.prototype.methodId=function(){return O.methodId(this)},t.prototype.allSecuredBy=function(){return O.allSecuredBy(this)},t}(j);t.MethodImpl=_e;var Ne=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.wrapperClassName=function(){return"MimeTypeImpl"},t.prototype.kind=function(){return"MimeType"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("MimeType")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.MimeTypeImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.MimeTypeImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(q);t.MimeTypeImpl=Ne;var we=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.wrapperClassName=function(){return"UriTemplateImpl"},t.prototype.kind=function(){return"UriTemplate"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("UriTemplate")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.UriTemplateImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.UriTemplateImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(q);t.UriTemplateImpl=we;var Re=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.wrapperClassName=function(){return"RelativeUriStringImpl"},t.prototype.kind=function(){return"RelativeUriString"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("RelativeUriString")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.RelativeUriStringImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.RelativeUriStringImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(we);t.RelativeUriStringImpl=Re;var Ie=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.wrapperClassName=function(){
return"FullUriTemplateStringImpl"},t.prototype.kind=function(){return"FullUriTemplateString"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("FullUriTemplateString")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.FullUriTemplateStringImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.FullUriTemplateStringImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(we);t.FullUriTemplateStringImpl=Ie;var Me=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.wrapperClassName=function(){return"RAMLSimpleElementImpl"},t.prototype.kind=function(){return"RAMLSimpleElement"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("RAMLSimpleElement")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.RAMLSimpleElementImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.RAMLSimpleElementImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(L.BasicNodeImpl);t.RAMLSimpleElementImpl=Me;var Ce=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?S(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return I(t,e),t.prototype.title=function(){return e.prototype.attribute.call(this,"title",this.toString)},t.prototype.setTitle=function(e){return this.highLevel().attrOrCreate("title").setValue(""+e),this},t.prototype.content=function(){return e.prototype.attribute.call(this,"content",function(e){return new G(e)})},t.prototype.wrapperClassName=function(){return"DocumentationItemImpl"},t.prototype.kind=function(){return"DocumentationItem"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("DocumentationItem")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.DocumentationItemImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.DocumentationItemImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(Me);t.DocumentationItemImpl=Ce;var Le=function(e){function t(t,n){void 0===n&&(n=!0),e.call(this,"string"==typeof t?T(t):t,n),this.nodeOrKey=t,this.setAsTopLevel=n}return I(t,e),t.prototype.key=function(){return e.prototype.attribute.call(this,"key",this.toString)},t.prototype.setKey=function(e){return this.highLevel().attrOrCreate("key").setValue(""+e),this},t.prototype.value=function(){return e.prototype.attribute.call(this,"value",function(e){return new ne(e)})},t.prototype.wrapperClassName=function(){return"GlobalSchemaImpl"},t.prototype.kind=function(){return"GlobalSchema"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("GlobalSchema")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.GlobalSchemaImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.GlobalSchemaImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t}(Me);t.GlobalSchemaImpl=Le;var Pe=function(e){function t(){e.apply(this,arguments)}return I(t,e),t.prototype.relativeUri=function(){return e.prototype.attribute.call(this,"relativeUri",function(e){return new Re(e)})},t.prototype.type=function(){return e.prototype.attribute.call(this,"type",function(e){return new K(e)})},t.prototype.is=function(){return e.prototype.attributes.call(this,"is",function(e){return new de(e)})},t.prototype.securedBy=function(){return e.prototype.attributes.call(this,"securedBy",function(e){return new pe(e)})},t.prototype.uriParameters_original=function(){return e.prototype.elements.call(this,"uriParameters")},t.prototype.methods=function(){return e.prototype.elements.call(this,"methods")},t.prototype.resources=function(){return e.prototype.elements.call(this,"resources")},t.prototype.displayName=function(){return e.prototype.attribute.call(this,"displayName",this.toString)},t.prototype.setDisplayName=function(e){return this.highLevel().attrOrCreate("displayName").setValue(""+e),this},t.prototype.baseUriParameters=function(){return e.prototype.elements.call(this,"baseUriParameters")},t.prototype.description=function(){return e.prototype.attribute.call(this,"description",function(e){return new G(e)})},t.prototype.wrapperClassName=function(){return"ResourceImpl"},t.prototype.kind=function(){return"Resource"},t.prototype.allKinds=function(){return e.prototype.allKinds.call(this).concat("Resource")},t.prototype.allWrapperClassNames=function(){return e.prototype.allWrapperClassNames.call(this).concat("RAML08.ResourceImpl")},t.isInstance=function(e){if(null!=e&&e.allWrapperClassNames&&"function"==typeof e.allWrapperClassNames)for(var t=0,n=e.allWrapperClassNames();t<n.length;t++){var r=n[t];if("RAML08.ResourceImpl"==r)return!0}return!1},t.prototype.RAMLVersion=function(){return"RAML08"},t.prototype.completeRelativeUri=function(){return O.completeRelativeUri(this)},t.prototype.absoluteUri=function(){return O.absoluteUri(this)},t.prototype.parentResource=function(){return O.parent(this)},t.prototype.childResource=function(e){return O.childResource(this,e)},t.prototype.childMethod=function(e){return O.childMethod(this,e)},t.prototype.ownerApi=function(){return O.ownerApi(this)},t.prototype.uriParameters=function(){return O.uriParametersPrimary(this)},t.prototype.allUriParameters=function(){return O.uriParameters(this)},t.prototype.absoluteUriParameters=function(){return O.absoluteUriParameters(this)},t.prototype.allSecuredBy=function(){return O.allSecuredBy(this)},t}(L.BasicNodeImpl);t.ResourceImpl=Pe,t.loadApiSync=b,t.loadRAMLSync=_,t.loadApi=N,t.loadRAML=w,t.getLanguageElementByRuntimeType=R},function(e,t,n){"use strict";function r(e){if(null==e||"string"!=typeof e)return{status:I.NOT_REQUIRED};for(var t="",n={},r=0,i=e.indexOf("<<");i>=0;i=e.indexOf("<<",r)){if(t+=e.substring(r,i),r=e.indexOf(">>",i),0>r)return{status:I.ERROR};r+=">>".length;var a=e.substring(i,r),o=M+i+M;n[o]=a,t+=o}return 0==t.length?{status:I.NOT_REQUIRED}:(t+=e.substring(r,e.length),{resultingString:t,substitutions:n,status:I.OK})}function i(e,t){if(null==e)return{status:I.NOT_REQUIRED};for(var n="",r=0,i=e.indexOf(M);i>=0;i=e.indexOf(M,r)){if(r=e.indexOf(M,i+1),r+=M.length,0>r)return{status:I.ERROR};var a=e.substring(i,r),o=t[a];if(null==o)return{status:I.ERROR};n+=o}return 0==n.length?{status:I.NOT_REQUIRED}:(n+=e.substring(r,e.length),{resultingString:n,substitutions:t,status:I.OK})}function a(e){for(var t=!1,n=0;n<e.length;n++){var r=e.charAt(n);if("|"==r||"("==r||"["==r){t=!0;break}}return t}function o(e){var t=e.parent&&e.parent();return t=t&&t.parent&&t.parent(),t=t&&t.parent&&t.parent(),t=t&&t.parent&&t.parent(),t&&t.key&&t.key()===E.Universe10.Api.properties.resourceTypes.name}function s(e,t,n){var r=e.lowLevel(),i=e.definition().universe().version(),a=l(r),s=o(a),p=E.Universe10.MethodBase.properties.is.name,c=T.find(r.children(),function(e){return e.key()==p}),f=c;if(null==f){var d=new y.ASTNode(m.newMapping(m.newScalar(p),m.newItems()),a.unit(),a,null,null);f=r.replaceChild(null,d)}var v=s?null:T.find(a.children(),function(e){return e.key()==p}),g=null!=v&&v.unit().absolutePath()==n?v.children():[],b=g.concat(t.map(function(e){var t=u(e.node,f);if(null!=t){var n=new A.LowLevelCompositeNode(t,f,e.transformer,i);return n}return null})).filter(function(e){return null!=e});if(0==b.length&&t.length<=1)return c;f.setChildren(b),f.filterChildren(),b=f.children();var _=e.directChildren(),N=e.definition().property("is"),w=N.range();_&&(_=_.filter(function(e){var t=e.property();return null==t||!S.isIsProperty(t)}),e._children=_.concat(b.map(function(t){return new h.ASTPropImpl(t,e,w,N)})));var R=e.children();return R&&(R=R.filter(function(e){var t=e.property();return null==t||!S.isIsProperty(t)}),e._mergedChildren=R.concat(b.map(function(t){return new h.ASTPropImpl(t,e,w,N)}))),f}function u(e,t){t=l(t),e=l(e);var n=e.actual();if(null==n)return null;if(e.key()==E.Universe10.MethodBase.properties.is.name&&(n=e.yamlNode().value),null==n)return null;if(n.kind==m.Kind.SEQ&&(n=n.items[0]),null==n)return null;var r=new y.ASTNode(n,e.unit(),t,null,null);return r}function l(e){for(var t=0;2>t&&A.LowLevelProxyNode.isInstance(e);t++)e=e.originalNode();return e}function p(e){return e?null!=e.namespace&&"function"==typeof e.namespace&&null!=e.name&&"function"==typeof e.name&&null!=e.collectionName&&"function"==typeof e.collectionName&&null!=e.referencedUnit&&"function"==typeof e.referencedUnit&&null!=e.mode&&"function"==typeof e.mode:!1}function c(e,t,n){var r=n[0].project().namespaceResolver();if(!e)return null;var i="",a=e,o=e.lastIndexOf(".");o>=0&&(i=e.substring(0,o),a=e.substring(o+1));for(var s,u=!1,l=n.length;l>0;l--){var p=n[l-1],c=p.highLevel();if(c.isElement()&&S.isLibraryType(c.asElement().definition())){if(u)break;u=!0}var f=p;if(i){f=null;var d=r.nsMap(p);if(d){var h=d[i];h&&(f=h.unit)}}if(f){var m=f.highLevel();if(m&&m.isElement()&&(s=T.find(m.asElement().elementsOfKind(t),function(e){return e.name()==a})))break}}return s}function f(e){var t=e.indexOf("<<");if(0>t)return!1;if(0!=t)return!0;var n=e.indexOf(">>",t);return n+">>".length!=e.length?!0:!1}var d=n(10),h=n(16),m=n(77),y=n(24),v=n(54),g=n(33),A=n(44),E=n(18),T=n(70),S=n(14),b=n(50),_=n(39),N=_.rt.typeExpressions;!function(e){e[e.DEFAULT=0]="DEFAULT",e[e.PATH=1]="PATH"}(t.PatchMode||(t.PatchMode={}));var w=t.PatchMode,R=function(){function e(e){void 0===e&&(e=w.DEFAULT),this.mode=e,this._outerDependencies={},this._libModels={}}return e.prototype.process=function(e,t,n,r){var i=this;if(void 0===t&&(t=e),void 0===n&&(n=!1),void 0===r&&(r=!1),!e.lowLevel().libProcessed){var a=e.lowLevel().unit().project().namespaceResolver();this.patchReferences(e,t,a),r&&this.patchNodeName(e,t.lowLevel().unit(),a),n?this.removeUses(e):this.patchUses(e,a),e.elements().forEach(function(e){return i.removeUses(e)}),this.resetTypes(e),e.lowLevel().libProcessed=!0}},e.prototype.patchReferences=function(e,t,n,r){if(void 0===t&&(t=e),void 0===n&&(n=new b.NamespaceResolver),void 0===r&&(r=[t.lowLevel().unit()]),!e.isReused()){var i;if(null!=e.definition().property(E.Universe10.TypeDeclaration.properties.annotations.name)){var a=e.lowLevel();if(!A.LowLevelCompositeNode.isInstance(a))return;var o=E.Universe10.MethodBase.properties.is.name,u=e.attributes(o);if(0!=u.length){var l=e.lowLevel().children().filter(function(e){return e.key()==E.Universe10.MethodBase.properties.is.name});1==l.length&&l[0].valueKind()==m.Kind.SEQ&&(i=s(e,u.map(function(e){return e.lowLevel()}).map(function(e){return A.LowLevelProxyNode.isInstance(e)?{node:e,transformer:e.transformer()}:{node:e,transformer:null}}),r[0].absolutePath()))}}for(var p=e.attrs(),c=0,f=p;c<f.length;c++){var d=f[c],h=this.appendUnitIfNeeded(d,r);this.patchReferenceAttr(d,t,n,r),this.popUnitIfNeeded(r,h)}if(S.isTypeDeclarationDescendant(e.definition())){var h=this.appendUnitIfNeeded(e,r);this.patchType(e,t,n,r),this.popUnitIfNeeded(r,h)}for(var y=e.elements(),v=0,g=y;v<g.length;v++){var T=g[v],h=this.appendUnitIfNeeded(T,r);this.patchReferences(T,t,n,r),this.popUnitIfNeeded(r,h)}if(i){i.filterChildren();var _=e.directChildren();_&&(e._children=this.filterTraitReferences(_));var N=e.children();N&&(e._mergedChildren=this.filterTraitReferences(N))}}},e.prototype.filterTraitReferences=function(e){for(var t=[],n={},r=0,i=e;r<i.length;r++){var a=i[r],o=a.property();if(o&&S.isIsProperty(o)){var s=JSON.stringify(v.serialize(a.lowLevel()));n[s]||(n[s]=!0,t.push(a))}else t.push(a)}return t},e.prototype.patchReferenceAttr=function(e,t,n,r,i){void 0===i&&(i=!1);var a=e.property(),o=a.range();if(i||o.isAssignableFrom(E.Universe10.Reference.name)){var s=e.value();if(null!=s){var u=e.lowLevel();if(A.LowLevelProxyNode.isInstance(u)){var p=u.transformer(),c=S.isAnnotationsProperty(a);if(!c&&a&&_.UserDefinedProp.isInstance(a)){var f=a.node(),d=f&&f.property();c=d&&S.isAnnotationsProperty(f.property()),d&&c&&(o=d.range())}if("string"==typeof s){var m=s;if(null!=p){var y=l(u);m=y.value()}var v=c&&"("==m.charAt(0);v&&(m=m.substring(1,m.length-1));var g=this.resolveReferenceValue(m,t.lowLevel().unit(),r,n,p,o);if(null!=g){var T=v?"("+g.value()+")":g.value();e.lowLevel().setValueOverride(T),e.overrideValue(T),this.registerPatchedReference(g)}}else if(h.StructuredValue.isInstance(s)){var b=s,N=b.toHighLevel();if(N)for(var w=0,R=N.attrs();w<R.length;w++){var e=R[w];S.isSchemaStringType(e.definition())&&this.patchReferenceAttr(e,t,n,r,!0)}var I=b.lowLevel().key(),m=I;if(null!=p){var y=l(b.lowLevel());m=y.key()}if(null!=I){c&&(m=m.substring(1,m.length-1));var M=this.resolveReferenceValue(m,t.lowLevel().unit(),r,n,p,o);if(null!=M){var T=c?"("+M.value()+")":M.value();b.lowLevel().setKeyOverride(T),this.registerPatchedReference(M)}}}}}}},e.prototype.patchType=function(e,t,n,o){var s=this,u=e.definition(),p=e.localType();if(p.isAnnotationType()){var c=p.superTypes();c.length>0&&(p=c[0])}var d=O(p);if(!d&&p.isArray()&&(d=O(p.componentType())),!d){var m=t.lowLevel().unit(),y=(m.absolutePath(),e.attributes(E.Universe10.TypeDeclaration.properties.type.name));0==y.length&&(y=e.attributes(E.Universe10.TypeDeclaration.properties.schema.name));var v=e.attributes(E.Universe10.ArrayTypeDeclaration.properties.items.name);y=y.concat(v);for(var g=0,S=y;g<S.length;g++){var b=S[g],_=b.lowLevel();if(A.LowLevelProxyNode.isInstance(_)){var w=b.lowLevel().unit(),R=(w.absolutePath(),b.value());if(null!=R)if("string"==typeof R){var M=a(R),C=_.transformer(),L=R,P={status:I.NOT_REQUIRED},D=C?C.unitsChain:null;if(null!=C||R.indexOf("<<")>=0){var x=l(_),U=x.value();if(P=r(U),P.status==I.OK)L=M?P.resultingString:U;else{if(P.status==I.ERROR)return;C=null}}var k;if(D){k=[];for(var F=0,B=D;F<B.length;F++){var K=B[F];k.push(this.appendUnitIfNeeded(K,o))}}var V,j=this.appendUnitIfNeeded(b,o),W=void 0;try{V=M&&N.parse(L)}catch(q){V=null}if(M&&null!==V){var Y=!1,H=N.parse(L),$=!1;N.visit(H,function(e){if("name"==e.type){var t,r=e,a=r.value,l={status:I.NOT_REQUIRED};if(P.status==I.OK)if(t=P.substitutions[a],null==t){if(l=i(a,P.substitutions),l.status==I.OK)a=l.resultingString;else if(l.status==I.ERROR)return void(Y=!0)}else a=t;if(null==C&&(null!=t||l.status==I.OK))return void(r.value=a);var p=!0;a.indexOf("<<")>=0&&f(a)&&(p=!1);var c=s.resolveReferenceValue(a,m,o,n,C,u,p);null!=c&&(r.value=c.value(),$=!0,s.registerPatchedReference(c))}}),W=$&&!Y?N.serializeToString(H):R}else if(P.status!=I.OK||null!=C){L.indexOf("<<")>=0&&f(L)&&(L=R,C=null);var G=this.resolveReferenceValue(L,m,o,n,C,u);null!=G&&(this.registerPatchedReference(G),W=G.value())}if(null!=W&&(b.lowLevel().setValueOverride(W),b.overrideValue(null)),this.popUnitIfNeeded(o,j),k)for(var X=0,z=k.reverse();X<z.length;X++){var J=z[X];this.popUnitIfNeeded(o,J)}}else{var Q=b.lowLevel();if(Q.key()!=b.property().nameId()&&(Q=T.find(e.lowLevel().children(),function(e){return e.key()==b.property().nameId()})),Q){var Z=e.definition().universe().type(E.Universe10.TypeDeclaration.name),ee=new h.ASTNodeImpl(Q,null,Z,null),te=this.appendUnitIfNeeded(ee,o);this.patchReferences(ee,t,n,o),this.popUnitIfNeeded(o,te)}}}}}},e.prototype.resolveReferenceValue=function(e,t,n,r,i,a,o){var s=this;void 0===o&&(o=!0);var u,l=S.isAnnotationRefTypeOrDescendant(a);if(i&&e&&e.indexOf("<<")>=0){var c=!0,f=t.highLevel().types(),d=i.transform(e,!0,function(){return c},function(e,n){var i=s.resolveReferenceValueBasic(e,t,r,n.unitsChain,a);return null==i&&(i=new C(null,e,s.collectionName(a),t,w.DEFAULT)),c=l?null!=f.getAnnotationType(i.value())?!1:!1:null!=f.getType(i.value())?!1:!1,i});u=d.value}return void 0!==u&&p(u)||(u=this.resolveReferenceValueBasic(e,t,r,n,a)),u},e.prototype.patchNodeName=function(e,t,n){var r=e.lowLevel(),i=r.key(),a=e.definition();if(S.isTypeDeclarationSibling(a)){var o=e.localType();o.isAnnotationType()&&(a=o)}var s=this.resolveReferenceValueBasic(i,t,n,[r.unit()],a);null!=s&&(r.setKeyOverride(s.value()),e.resetIDs())},e.prototype.resolveReferenceValueBasic=function(e,t,n,r,i){if(null==e||"string"!=typeof e)return null;var a=S.isTypeDeclarationDescendant(i),o=a&&g.stringEndsWith(e,"?"),s=o?e.substring(0,e.length-1):e;if(!(s.indexOf("<<")>=0)){var u,l,p=s.lastIndexOf(".");if(p>=0){var c=s.substring(0,p);l=s.substring(p+1);for(var f=r.length;f>0;f--){var h=r[f-1],m=n.nsMap(h);if(null!=m){var y=m[c];if(null!=y&&(u=y.unit,null!=u))break}}}else{if(a&&null!=_.rt.builtInTypes().get(s))return null;l=s,u=r[r.length-1]}var v=this.collectionName(i);if(!u)return null;if(u.absolutePath()==t.absolutePath())return null!=c?new C(null,l,v,u,this.mode):null;var A=n.resolveNamespace(t,u);if(null==A)return null;var E=A.namespace();if(null==E)return null;if(this.mode==w.PATH){var T=u.absolutePath().replace(/\\/g,"/");d.isWebPath(T)||(T="file://"+T),E=T+"#/"+v}return o&&(l+="?"),new C(E,l,v,u,this.mode)}},e.prototype.patchUses=function(e,t){var n=e.lowLevel();if(e.children(),A.LowLevelCompositeNode.isInstance(n)){var r=n.unit(),i=t.expandedPathMap(r);if(null!=i){var a=t.pathMap(r);a||(a={});for(var o=n,s=n.children(),u=E.Universe10.FragmentDeclaration.properties.uses.name,p=s.filter(function(e){return e.key()==u}),c=l(n),f=c;A.LowLevelProxyNode.isInstance(f);)f=f.originalNode();var d,m=Object.keys(a).map(function(e){return i[e]}),v=Object.keys(i).map(function(e){return i[e]}).filter(function(e){return!a[e.absolutePath()]}),g=r.absolutePath(),T={};if(p.length>0)d=p[0],d.children().forEach(function(e){return T[e.key()]=!0});else{var b=y.createMapNode("uses");b._parent=f,b.setUnit(f.unit()),d=o.replaceChild(null,b)}for(var _=e.definition().property(u),N=_.range(),w=e._children.filter(function(e){if(e.lowLevel().unit().absolutePath()==g)return!0;var t=e.property();return!t||!S.isUsesProperty(t)}),R=e._mergedChildren.filter(function(e){if(e.lowLevel().unit().absolutePath()==g)return!0;var t=e.property();return!t||!S.isUsesProperty(t)}),I=0,M=m.concat(v);I<M.length;I++){var C=M[I];C.absolutePath();if(!T[C.namespace()]){var L=C.includePath,P=y.createMapping(C.namespace(),L);P.setUnit(f.unit());var O=new h.ASTNodeImpl(P,e,N,_);w.push(O),R.push(O),d.replaceChild(null,P)}}e._children=w,e._mergedChildren=R}}},e.prototype.removeUses=function(e){var t=e.lowLevel();if(A.LowLevelCompositeNode.isInstance(t)){var n=t,r=t.children(),i=r.filter(function(e){return e.key()==E.Universe10.FragmentDeclaration.properties.uses.name});i.length>0&&n.removeChild(i[0]),e._children=e.directChildren().filter(function(e){var t=e.property();return null==t||!S.isUsesProperty(t)}),e._mergedChildren=e.children().filter(function(e){var t=e.property();return null==t||!S.isUsesProperty(t)})}},e.prototype.resetTypes=function(e){for(var t=0,n=e.elements();t<n.length;t++){var r=n[t];this.resetTypes(r)}for(var i=0,a=e.attrs();i<a.length;i++){var o=a[i],s=o.value();h.StructuredValue.isInstance(s)&&s.resetHighLevelNode()}delete e.lowLevel().actual().types,delete e._ptype,delete e._types,e.setAssociatedType(null)},e.prototype.appendUnitIfNeeded=function(e,t){if(y.CompilationUnit.isInstance(e)){var n=e;return n.absolutePath()!=t[t.length-1].absolutePath()?(t.push(n),!0):!1}var r=l(e.lowLevel()),i=r.unit();if(r.valueKind()==m.Kind.INCLUDE_REF){var a=r.includePath(),o=i.resolve(a);return o?(t.push(o),!0):!1}return i.absolutePath()!=t[t.length-1].absolutePath()?(t.push(i),!0):!1},e.prototype.popUnitIfNeeded=function(e,t){t&&e.pop()},e.prototype.registerPatchedReference=function(e){var t=e.collectionName();if(t){var n=e.referencedUnit().absolutePath(),r=this._outerDependencies[n];null==r&&(r={},this._outerDependencies[n]=r);var i=r[t];null==i&&(i={},r[t]=i),i[e.name()]=e}},e.prototype.collectionName=function(e){var t;return S.isResourceTypeRefType(e)||S.isResourceTypeType(e)?t=_.universesInfo.Universe10.LibraryBase.properties.resourceTypes.name:S.isTraitRefType(e)||S.isTraitType(e)?t=_.universesInfo.Universe10.LibraryBase.properties.traits.name:S.isSecuritySchemeRefType(e)||S.isSecuritySchemaTypeDescendant(e)?t=_.universesInfo.Universe10.LibraryBase.properties.securitySchemes.name:S.isAnnotationRefTypeOrDescendant(e)||e.isAnnotationType()?t=_.universesInfo.Universe10.LibraryBase.properties.annotationTypes.name:(S.isTypeDeclarationDescendant(e)||S.isSchemaStringType(e))&&(t=_.universesInfo.Universe10.LibraryBase.properties.types.name),t},e.prototype.expandLibraries=function(e,t){if(void 0===t&&(t=!1),!e.lowLevel().actual().libExpanded){var n=e.lowLevel(),r=n.unit(),i=(r.absolutePath(),r.project()),a=[],o=n.unit().project().namespaceResolver(),s=o.expandedPathMap(r);if(null!=s){for(var u=Object.keys(s).sort(),l=0,p=u;l<p.length;l++){var c=p[l],f=this._libModels[c];if(null==f){var d=i.unit(c,!0),h=o.resolveNamespace(r,d);d&&null!=h&&null!=h.namespace()&&(f=this.extractLibModel(d))}f&&a.push(f)}for(var m=!1,y=0,v=a;y<v.length;y++)for(var f=v[y],g=0,A=Object.keys(f);g<A.length;g++){var E=A[g],T=f[E];L.isInstance(T)&&(m=this.contributeCollection(e,T)||m)}if(m){var S=!1;do S=this.patchDependencies(e,t);while(S);t||this.removeUnusedDependencies(e)}}this.removeUses(e),e.lowLevel().actual().libExpanded=!0,this.resetTypes(e)}},e.prototype.patchDependencies=function(e,t){for(var n=!1,r=e.lowLevel().unit().absolutePath(),i=0,a=e.children();i<a.length;i++){var o=a[i];if(o.isElement()&&!o.lowLevel().libProcessed){var s=o.asElement();this.removeUses(s);var u=s.property().nameId();if(u==E.Universe10.LibraryBase.properties.types.name||u==E.Universe10.LibraryBase.properties.annotationTypes.name||u==E.Universe10.LibraryBase.properties.resourceTypes.name||u==E.Universe10.LibraryBase.properties.traits.name||u==E.Universe10.LibraryBase.properties.securitySchemes.name){if(!t){var l=o.lowLevel().unit().absolutePath();if(l==r&&null==o.lowLevel().includePath())continue;var p=this._outerDependencies[l];if(null==p)continue;var c=p[u];if(null==c)continue;var f=s.name();if(null==c[f])continue}this.process(s,e,!0,!0),n=!0}}}return n},e.prototype.removeUnusedDependencies=function(e){for(var t=e.lowLevel(),n=t.unit().absolutePath(),r=e._children,i=[],a=0,o=r;a<o.length;a++){var s=o[a],u=s.lowLevel();if(s.isElement()&&u.libProcessed)i.push(s);else{var l=u.unit().absolutePath();l!=n?u.parent().removeChild(u):i.push(s)}}e._children=i,e._mergedChildren=null},e.prototype.contributeCollection=function(e,t){var n=e.lowLevel();if(0==t.array.length)return!1;var r=t.name,i=e.definition().property(r),a=i.range(),o=T.find(n.children(),function(e){return e.key()==r});if(null==o){var s=y.createMapNode(r);o=n.replaceChild(null,s)}for(var u=!1,p=e._children,c=e._mergedChildren,f=0,d=t.array;f<d.length;f++){var m=d[f];if(!o.children().some(function(e){var t=l(e);return t.unit().absolutePath()!=m.lowLevel().unit().absolutePath()?!1:m.lowLevel().key()==t.key()&&m.lowLevel().unit().absolutePath()==t.unit().absolutePath()})){var v=o.replaceChild(null,m.lowLevel()),g=m.isElement()&&m.asElement().definition()||a,A=new h.ASTNodeImpl(v,e,g,i);p.push(A),c&&c.push(A),u=!0}}return u},e.prototype.extractLibModel=function(e){var t=this._libModels[e.absolutePath()];if(null!=t)return t;t=new P(e),this._libModels[e.absolutePath()]=t;var n=e.highLevel();if(n&&n.isElement())for(var r=0,i=["resourceTypes","traits","types","annotationTypes","securitySchemes"];r<i.length;r++){for(var a=i[r],o=new L(a),s=0,u=n.asElement().elementsOfKind(a);s<u.length;s++){var l=u[s];o.array.push(l)}t[a]=o}return t},e}();t.ReferencePatcher=R,function(e){e[e.OK=0]="OK",e[e.NOT_REQUIRED=1]="NOT_REQUIRED",e[e.ERROR=2]="ERROR"}(t.ParametersEscapingStatus||(t.ParametersEscapingStatus={}));var I=t.ParametersEscapingStatus,M="__P_A_R_A_M_E_T_E_R__";t.escapeTemplateParameters=r,t.unescapeTemplateParameters=i,t.checkExpression=a,t.patchMethodIs=s,t.prepareTraitRefNode=u,t.toOriginal=l;var C=function(){function e(e,t,n,r,i){this._namespace=e,this._name=t,this._collectionName=n,this._referencedUnit=r,this._mode=i,this.gotQuestion=!1;var a=this._name.length;"?"==this._name.charAt(a-1)&&(this.gotQuestion=!0,this._name=this._name.substring(0,a-1))}return e.isInstance=function(t){return null!=t&&t.getClassIdentifier&&"function"==typeof t.getClassIdentifier&&T.contains(t.getClassIdentifier(),e.CLASS_IDENTIFIER_PatchedReference)},e.prototype.getClassIdentifier=function(){return[e.CLASS_IDENTIFIER_PatchedReference]},e.prototype.namespace=function(){return this._namespace},e.prototype.name=function(){return this._name},e.prototype.collectionName=function(){return this._collectionName},e.prototype.referencedUnit=function(){return this._referencedUnit},e.prototype.mode=function(){return this._mode},e.prototype.value=function(){if(null==this._namespace)return this._name;var e=this._mode==w.PATH?"/":".";return this._namespace+e+this._name+(this.gotQuestion?"?":"")},e.CLASS_IDENTIFIER_PatchedReference="referencePatcher.PatchedReference",e}();t.PatchedReference=C,t.instanceOfPatchedReference=p;var L=function(){function e(e){this.name=e,this.array=[]}return e.isInstance=function(t){return null!=t&&t.getClassIdentifier&&"function"==typeof t.getClassIdentifier&&T.contains(t.getClassIdentifier(),e.CLASS_IDENTIFIER)},e.prototype.getClassIdentifier=function(){var t=[];return t.concat(e.CLASS_IDENTIFIER)},e.CLASS_IDENTIFIER="referencePatcher.ElementsCollection",e}(),P=function(){function e(e){this.unit=e}return e}();t.getDeclaration=c,t.isCompoundValue=f;var O=function(e){var t=!e||e.isExternal();if(!t)for(var n=0,r=e.superTypes();n<r.length;n++){var i=r[n];if(t=i.isExternal())break}return t}},function(e,t,n){"use strict";function r(e,t){if(void 0===t&&(t=!1),!e)return null;var n=e.kind();if(n==u.Kind.ANCHOR_REF&&(n=e.anchorValueKind()),n==u.Kind.INCLUDE_REF){if(null!=e.unit()){var i=e.includePath(),a=null;try{a=e.unit().resolve(i)}catch(o){}if(null==a)return null;if(a.isRAMLUnit()&&this.canInclude(a)){var s=a.ast();if(s)return r(s,t)}}return null}if(n==u.Kind.SEQ){for(var l=[],p=0,c=e.children();p<c.length;p++){var f=c[p];l.push(r(f,t))}return l}if(n==u.Kind.ANCHOR_REF)return r(e.anchoredFrom(),t);if(n==u.Kind.MAPPING){var d={},h=""+e.key(!0),m=e.valueKind();if(m==u.Kind.ANCHOR_REF&&(m=e.anchorValueKind()),m==u.Kind.INCLUDE_REF){var y=e.children();0==y.length?d[h]=e.value():m=null==y[0].key()?u.Kind.SEQ:u.Kind.MAP}if(m==u.Kind.SCALAR)d[h]=e.value();else if(m==u.Kind.SEQ){var l=[];d[h]=l;for(var v=0,g=e.children();v<g.length;v++){var f=g[v];l.push(r(f,t))}}else if(m==u.Kind.MAP){var A={};d[h]=A;for(var E=0,T=e.children();E<T.length;E++){var f=T[E],S=""+f.key(!0),b=r(f,t);void 0===b&&(b=t?"!$$$novalue":null),A[S]=b}}return d[h]}if(n==u.Kind.SCALAR){var _=e.value(!1);return _}if(n==u.Kind.MAP){for(var A={},N=0,w=e.children();N<w.length;N++){var f=w[N],S=""+f.key(!0),b=r(f,!0);A[S]=void 0===b?null:b}return A}}function i(e,t){if(void 0===t&&(t={}),t=t||{},0==e.children().length)return e.value();if(e.children()[0].key()){var n={};if(e.children().forEach(function(e){n[o(e.key(t.rawKey),t)]=i(e,t)}),t&&t.writeErrors){var r=a(e);null!=r&&r.length>0&&(n.__$errors__=r)}return n}var s=[];return e.children().forEach(function(e){s.push(i(e,t))}),s}function a(e){var t=[].concat(e.errors());return e.children().forEach(function(e){var n=e.children();return 0==n.length?void e.errors().forEach(function(e){return t.push(e)}):void(n[0].key()||n.forEach(function(e){0==e.children().length&&e.errors().forEach(function(e){return t.push(e)})}))}),t}function o(e,t){return t&&e&&t.escapeNumericKeys&&0==e.replace(/\d/g,"").trim().length?"__$EscapedKey$__"+e:e}function s(e,t){return e?(t=t||{},t.escapeNumericKeys&&c.stringStartsWith(e,"__$EscapedKey$__")&&0==e.substring("__$EscapedKey$__".length).replace(/\d/g,"").trim().length?e.substring("__$EscapedKey$__".length):e):e}var u=n(77),l=n(10),p=n(16),c=n(33),f=n(24),d=u.YAMLException,h=function(){function e(e,t,n,r,i,a){void 0===a&&(a={}),this._absolutePath=e,this._path=t,this._content=n,this._project=r,this._isTopoLevel=i,this.serializeOptions=a,this._node=new m(this,JSON.parse(this._content),null,a)}return e.prototype.highLevel=function(){return p.fromUnit(this)},e.prototype.absolutePath=function(){return this._absolutePath},e.prototype.clone=function(){return null},e.prototype.contents=function(){return this._content},e.prototype.lexerErrors=function(){return[]},e.prototype.path=function(){return this._content},e.prototype.isTopLevel=function(){return this._isTopoLevel},e.prototype.ast=function(){return this._node},e.prototype.expandedHighLevel=function(){return this.highLevel()},e.prototype.isDirty=function(){return!0},e.prototype.getIncludeNodes=function(){return[]},e.prototype.resolveAsync=function(e){return null},e.prototype.isRAMLUnit=function(){return!0},e.prototype.project=function(){return this._project},e.prototype.updateContent=function(e){},e.prototype.ramlVersion=function(){throw new d("not implemented")},e.prototype.lineMapper=function(){return new l.LineMapperImpl(this.contents(),this.absolutePath())},e.prototype.resolve=function(e){return null},e.prototype.isOverlayOrExtension=function(){return!1},e.prototype.getMasterReferenceNode=function(){return null},e}();t.CompilationUnit=h;var m=function(){function e(e,t,n,r,i){var a=this;void 0===r&&(r={}),this._unit=e,this._object=t,this._parent=n,this.options=r,this._key=i,this._isOptional=!1,this._object instanceof Object&&Object.keys(this._object).forEach(function(e){var t=s(e,a.options);if(t!=e){var n=a._object[e];delete a._object[e],a._object[t]=n}}),this._key&&c.stringEndsWith(this._key,"?")&&(this._isOptional=!0,this._key=this._key.substring(0,this._key.length-1))}return e.prototype.keyKind=function(){return null},e.prototype.isAnnotatedScalar=function(){return!1},e.prototype.hasInnerIncludeError=function(){return!1},e.prototype.start=function(){return-1},e.prototype.end=function(){return-1},e.prototype.value=function(){return this._object},e.prototype.actual=function(){return this._object},e.prototype.includeErrors=function(){return[]},e.prototype.includePath=function(){return null},e.prototype.includeReference=function(){return null},e.prototype.key=function(){return this._key},e.prototype.optional=function(){return this._isOptional},e.prototype.children=function(){var t=this;return this._object?Array.isArray(this._object)?this._object.map(function(n){return new e(t._unit,n,t,t.options)}):this._object instanceof Object?Object.keys(this._object).map(function(n){return new e(t._unit,t._object[n],t,t.options,n)}):[]:[]},e.prototype.parent=function(){return this._parent},e.prototype.unit=function(){return this._unit},e.prototype.containingUnit=function(){return this._unit},e.prototype.includeBaseUnit=function(){return this._unit},e.prototype.anchorId=function(){return null},e.prototype.errors=function(){return[]},e.prototype.anchoredFrom=function(){return this},e.prototype.includedFrom=function(){return this},e.prototype.visit=function(e){e(this)&&this.children().forEach(function(t){return t.visit(e)})},e.prototype.dumpToObject=function(){return this._object},e.prototype.addChild=function(e){},e.prototype.execute=function(e){},e.prototype.dump=function(){return JSON.stringify(this._object)},e.prototype.keyStart=function(){return-1},e.prototype.keyEnd=function(){return-1},e.prototype.valueStart=function(){return-1},e.prototype.valueEnd=function(){return-1},e.prototype.isValueLocal=function(){return!0},e.prototype.kind=function(){return Array.isArray(this._object)?u.Kind.SEQ:this._object instanceof Object?u.Kind.MAP:u.Kind.SCALAR},e.prototype.valueKind=function(){if(!this._object)return null;var e=typeof this._object;return Array.isArray(this._object)?u.Kind.SEQ:"object"==e?u.Kind.MAP:"string"==e||"number"==e||"boolean"==e?u.Kind.SCALAR:null;
},e.prototype.anchorValueKind=function(){return null},e.prototype.resolvedValueKind=function(){return this.valueKind()},e.prototype.show=function(e){},e.prototype.setHighLevelParseResult=function(e){this._highLevelParseResult=e},e.prototype.highLevelParseResult=function(){return this._highLevelParseResult},e.prototype.setHighLevelNode=function(e){this._highLevelNode=e},e.prototype.highLevelNode=function(){return this._highLevelNode},e.prototype.text=function(e){throw new d("not implemented")},e.prototype.copy=function(){throw new d("not implemented")},e.prototype.markup=function(e){throw new d("not implemented")},e.prototype.nodeDefinition=function(){return f.getDefinitionForLowLevelNode(this)},e.prototype.includesContents=function(){return!1},e}();t.AstNode=m,t.serialize2=r,t.serialize=i},function(e,t,n){"use strict";var r=n(28),i=n(83),a=function(){function e(){this.uriToResources={},this.conflictingUriToResources={}}return e.prototype.validateApi=function(e,t){var n=this,a=e.resources();a.forEach(function(e){n.acceptResource(e);var t=e.resources();t.forEach(function(e){return n.acceptResource(e)})});for(var o in this.conflictingUriToResources){var a=this.conflictingUriToResources[o];if(a.length>1){var s={};a.forEach(function(e){var t=e.highLevel(),n="";null!=t.parent()&&(n+=t.parent().id()+"."),n+=t.localId();var r=s[n];null==r&&(r=[],s[n]=r),r.push(e)});var u=Object.keys(s);u.length>1&&a.forEach(function(e){t.accept(r.createIssue1(i.RESOURCES_SHARE_SAME_URI,{},e.highLevel(),!1))})}}},e.prototype.acceptResource=function(e){var t=e.absoluteUri(),n=this.uriToResources[t];n||(n=[],this.uriToResources[t]=n),n.push(e),n.length>1&&(this.conflictingUriToResources[t]=n)},e}();e.exports=a},function(e,t,n){"use strict";function r(e,t,n){null==n&&(n=i(e)),n.length>0&&(n+=":");for(var r=e.getElementsByTagName(n+t),a=[],o=0;o<r.length;o++){var s=r[o];s.parentNode===e&&a.push(s)}return a}function i(e){var t="";if(e){var n=e;if(e.ownerDocument&&(n=e.ownerDocument),n){var r=n.documentElement;r&&(t=r.prefix)}}return t}function a(e){try{if(e=e.trim(),!c.stringStartsWith(e,"<")&&c.stringEndsWith(e,">"))return!1;var t=new p.DOMParser(f).parseFromString(e),n=r(t,"schema",i(t));return n.length>0}catch(a){return!1}}function o(e){var t={};if(1==e.nodeType){if(e.attributes.length>0)for(var n=0;n<e.attributes.length;n++){var r=e.attributes.item(n);t["@"+r.nodeName]=r.nodeValue}}else 3==e.nodeType&&(t=e.nodeValue);if(e.hasChildNodes())for(var i=0;i<e.childNodes.length;i++){var a=e.childNodes.item(i),s=a.nodeName;if(void 0!=s)if("undefined"==typeof t[s])t[s]=o(a);else{if("undefined"==typeof t[s].push){var u=t[s];t[s]=[],t[s].push(u)}t[s].push(o(a))}}return t}function s(e){if("object"==typeof e)for(var t in e)if("object"==typeof e[t]){for(var n in e[t])if("#text"==n){var r=e[t]["#text"];"string"!=typeof r&&(r=r.join("")),r=r.trim(),0==r.length&&delete e[t]["#text"]}s(e[t])}return e}function u(e){if("object"==typeof e)for(var t in e)if("object"==typeof e[t]){var n=Object.keys(e[t]);1==n.length&&"#text"==n[0]&&(e[t]=e[t]["#text"]),u(e[t])}return e}function l(e){var t=new p.DOMParser(f);if(!e||0==e.trim().indexOf("<<"))return null;var n=t.parseFromString(e);return u(s(o(n)))}var p=n(75),c=n(33),f={errorHandler:{warning:function(){return null},error:function(){return null},fatalError:function(){return null}}};t.isXmlScheme=a,t.parseXML=l},function(e,t,n){"use strict";function r(e,t,n,r){var i=r.end!==!1,a=r.strict,o="";r.sensitive||(o+="i");var s=e.replace(c,function(e,r,i,a){if(a)return"\\"+a;n.push({name:i,prefix:r||"/"}),r=r?"\\"+r:"";var o=t[i],s=o&&l[o.type]||"[^"+(r||"\\/")+"]+",u=o&&o.required===!1;return Array.isArray(o["enum"])&&o["enum"].length&&(s="(?:"+o["enum"].map(function(e){return String(e).replace(p,"\\$1")}).join("|")+")"),r+"("+s+")"+(u?"?":"")}),u="/"===e.charAt(e.length-1);return a||(s=(u?s.slice(0,-2):s)+"(?:\\/(?=$))?"),s+=i?"$":a&&u?"":"(?=\\/|$)",new RegExp("^"+s+(i?"$":""),o)}function i(e){try{return decodeURIComponent(e)}catch(t){var n=new Error('Failed to decode param "'+e+'"');throw n.status=400,n}}function a(e,t,n){if(n=n||{},"/"===e&&n.end===!1)return o;t=t||{};var a=[],l=r(e,t,a,n),p=s.sanitize()(t),c=u.validate()(t);return function(e){var n=l.exec(e);if(!n)return!1;if(t.mediaTypeExtension&&n.length>1&&!n[n.length-1]){var r=n[n.length-2],o=r.lastIndexOf(".");o>=0&&(n[n.length-2]=r.substring(0,o),n[n.length-1]=r.substring(o))}for(var s=n[0],u={},f=1;f<n.length;f++){var d=a[f-1],h=n[f];u[d.name]=null==h?h:i(h)}return u=p(u),c(u).valid?{path:s,params:u}:!1}}function o(e){return{path:"",params:{}}}var s=n(80),u=n(81),l={number:"[-+]?\\d+(?:\\.\\d+)?",integer:"[-+]?\\d+",date:"(?:Mon|Tue|Wed|Thu|Fri|Sat|Sun), \\d{2} (?:Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec) \\d{4} (?:[0-1]\\d|2[0-3]):[0-5]\\d:[0-5]\\d GMT","boolean":"(?:true|false)"},p=/([.*+?=^!:${}()|[\]\/\\])/g,c=new RegExp(["([.\\/])?\\{([^}]+)\\}",p.source].join("|"),"g");t.ramlPathMatch=a},function(e,t,n){"use strict";var r=n(39),i=n(18),a=n(14),o=r,s=n(13),u=function(){function e(e,t){void 0===t&&(t=!1),this.enabled=e,this.toHighLevel=t,this.unconditionalValueCalculators=[new v],this.valueCalculators=[new d,new f,new c,new p,new h(this.toHighLevel),new m,new y]}return e.prototype.attributeDefaultIfEnabled=function(e,t){return this.enabled?this.getAttributeDefault(e,t):this.getUnconditionalAttributeDefault(t,e)},e.prototype.getUnconditionalAttributeDefault=function(e,t){if(!t||!e)return null;for(var n=0;n<this.unconditionalValueCalculators.length;n++){var r=this.unconditionalValueCalculators[n];if(r.matches(e,t)){var i=r.calculate(e,t);if(null!=i)return i}}return null},e.prototype.getAttributeDefault=function(e,t){if(!e||!t)return null;try{return this.getAttributeDefault2(t,e)}catch(n){return console.log(n),null}},e.prototype.getWrapperAttributeDefault=function(e,t){var n=e.highLevel();if(null==n)return null;var r=n.definition().property(t);return null==r?null:this.getAttributeDefault(n,r)},e.prototype.getAttributeDefault2=function(e,t){for(var n=0;n<this.valueCalculators.length;n++){var r=this.valueCalculators[n];if(r.matches(e,t)){var i=r.calculate(e,t);if(null!=i)return i}}return null!=e.defaultValue()?e.defaultValue():null},e.prototype.isEnabled=function(){return this.enabled},e.prototype.insertionKind=function(e,t){for(var n=0;n<this.valueCalculators.length;n++){var r=this.valueCalculators[n];if(r.matches(t,e))return r.kind()}return null!=t.defaultValue()?l.BY_DEFAULT:null},e}();t.AttributeDefaultsCalculator=u,function(e){e[e.CALCULATED=0]="CALCULATED",e[e.BY_DEFAULT=1]="BY_DEFAULT"}(t.InsertionKind||(t.InsertionKind={}));var l=t.InsertionKind,p=function(){function e(){}return e.prototype.calculate=function(e,t){var n=s.declRoot(t);if(n&&a.isApiSibling(n.definition())){var r=n.attr(i.Universe10.Api.properties.mediaType.name);if(r)return r.value()}return null},e.prototype.matches=function(e,t){if(!a.isNameProperty(e))return!1;var n=t.definition();if(!n)return!1;if(!a.isBodyLikeType(n)&&!a.isTypeDeclarationSibling(n))return!1;var r=t.parent();if(null==r)return!1;var i=r.definition();if(null==i)return!1;if(!a.isResponseType(i)&&!a.isMethodBaseSibling(i))return!1;for(var o=r;o;){var s=o.definition();if(a.isTraitType(s))return!1;if(a.isResourceTypeType(s))return!1;o=o.parent()}return!0},e.prototype.kind=function(){return l.CALCULATED},e}(),c=function(){function e(){}return e.prototype.calculate=function(e,t){var n=t.definition();if(null==n)return null;var r=n.getAdapter(o.RAMLService),i=r.getKeyProp();if(null!=i){var a=t.attrValue(i.nameId());return null!=a?a:new u(!0).getAttributeDefault(t,i)}return null},e.prototype.matches=function(e,t){var n=t.definition();return null==n?!1:(a.isTypeDeclarationSibling(n)||n.isAssignableFrom(i.Universe08.Parameter.name)||a.isResourceType(n))&&a.isDisplayNameProperty(e)},e.prototype.kind=function(){return l.CALCULATED},e}(),f=function(){function e(){}return e.prototype.calculate=function(e,t){return"object"},e.prototype.matches=function(e,t){return a.isTypeProperty(e)&&null!=t.definition()&&a.isObjectTypeDeclarationSibling(t.definition())},e.prototype.kind=function(){return l.BY_DEFAULT},e}(),d=function(){function e(){}return e.prototype.calculate=function(e,t){var n=t.definition(),r=t.property();if(null==n)return null;var i=n.getAdapter(o.RAMLService),s=i.getKeyProp();if(null!=s){var u=t.attr(s.nameId());if(null!=u&&u.optional())return!1}if(null!=r){if(a.isHeadersProperty(r)||a.isFormParametersProperty(r)||a.isQueryParametersProperty(r))return"RAML08"==e.domain().universe().version()?!1:!0;if(a.isUriParametersProperty(r)||a.isBaseUriParametersProperty(r))return!0}return null!=e.defaultValue()?e.defaultValue():null},e.prototype.matches=function(e,t){return a.isRequiredProperty(e)},e.prototype.kind=function(){return l.BY_DEFAULT},e}(),h=function(){function e(e){void 0===e&&(e=!1),this.toHighLevel=e}return e.prototype.calculate=function(e,t){if(a.isApiSibling(t.definition()))return null;var n,r=t.definition();if(a.isMethodType(r)){var i=t.parent();if(i){var o=i.getLastSlaveCounterPart();o&&(i=o),n=this.toHighLevel?i.attributes("securedBy"):i.wrapperNode().securedBy()}}if(!n||0==n.length){for(;null!=t&&!a.isApiSibling(t.definition());)t=t.parent();if(t){var s=t.getLastSlaveCounterPart();s&&(t=s),n=this.toHighLevel?t.attributes("securedBy"):t.wrapperNode().securedBy()}}return n&&n.length>0?n:null},e.prototype.matches=function(e,t){var n=t.definition();return null==n?!1:a.isSecuredByProperty(e)},e.prototype.kind=function(){return l.CALCULATED},e}(),m=function(){function e(){}return e.prototype.calculate=function(e,t){for(;null!=t&&!a.isApiSibling(t.definition());)t=t.parent();var n,r=t.attr(i.Universe10.Api.properties.baseUri.name);if(r){var o=r.value();if(o&&"string"==typeof o){var s=o.indexOf("://");s>=0&&(n=[o.substring(0,s).toUpperCase()]),n||(n=["HTTP"])}}return n},e.prototype.matches=function(e,t){if(!a.isProtocolsProperty(e))return!1;var n=t.definition(),r=!1;if(a.isApiSibling(n))r=!0;else if(a.isResourceType(n))r=!0;else if(a.isMethodType(n)){var i=t.parent();r=i&&a.isResourceType(i.definition())}return r},e.prototype.kind=function(){return l.CALCULATED},e}(),y=function(){function e(){}return e.prototype.calculate=function(e,t){for(;null!=t&&!a.isApiSibling(t.definition());)t=t.parent();var n=t.attr(i.Universe10.Api.properties.version.name);if(n){var r=n.value();if(r&&r.trim())return[r]}return null},e.prototype.matches=function(e,t){if(!a.isEnumProperty(e))return!1;var n=t.property();if(!n)return!1;if(!a.isBaseUriParametersProperty(n))return!1;var r=t.attr(i.Universe10.TypeDeclaration.properties.name.name),o=r&&r.value();return"version"!=o?!1:!0},e.prototype.kind=function(){return l.CALCULATED},e}(),v=function(){function e(){}return e.prototype.calculate=function(e,t){var n=t.definition();if(null==n)return null;var r=n.getAdapter(o.RAMLService);if(null==r)return null;var i=r.getKeyProp();if(null==i)return null;var a=t.attr(i.nameId());return null==a?null:a.optional()?!1:null},e.prototype.matches=function(e,t){return a.isRequiredProperty(e)},e.prototype.kind=function(){return l.BY_DEFAULT},e}()},function(e,t,n){"use strict";function r(e,t){void 0===t&&(t=!1);var n=_.Universe10.ResourceBase.properties.uriParameters.name,r=e.elementsOfKind(n);if(!C.isResourceType(e.definition()))return r;var i=e.attr(_.Universe10.Resource.properties.relativeUri.name).value();return a(r,i,e,n,t)}function i(e,t){void 0===t&&(t=!0);var n=e.attr(_.Universe10.Api.properties.baseUri.name),r=n?n.value():"",i=_.Universe10.Api.properties.baseUriParameters.name,o=e.elementsOfKind(i);return a(o,r,e,i,t)}function a(e,t,n,r,i){if("string"!=typeof t)return[];var a=n.definition(),o=a.property(r);if(!t)return[];var s={};e.forEach(function(e){var t=s[e.name()];t||(t=[],s[e.name()]=t),t.push(e)});for(var u=[],l=0,p={},c=t.indexOf("{");c>=0&&(l=t.indexOf("}",++c),!(0>l));c=t.indexOf("{",l)){var f=t.substring(c,l);if(p[f]=!0,s[f])s[f].forEach(function(e){return u.push(e)});else{var d=a.universe(),h=d.type(b.Universe10.StringTypeDeclaration.name),m=T.createStubNode(h,null,f,n.lowLevel().unit());m.setParent(n),i&&m.wrapperNode().meta().setCalculated(),m.attrOrCreate("name").setValue(f),m.patchProp(o),u.push(m)}}return Object.keys(s).filter(function(e){return!p[e]}).forEach(function(e){return s[e].forEach(function(e){return u.push(e)})}),u}function o(e){var t="",n=e;do e=n,t=e.attr(b.Universe10.Resource.properties.relativeUri.name).value()+t,n=e.parent();while(C.isResourceType(n.definition()));return t}function s(e){if(!C.isResourceType(e.definition()))return null;var t="",n=e;do e=n,t=e.attr(b.Universe10.Resource.properties.relativeUri.name).value()+t,n=e.parent();while(C.isResourceType(n.definition()));t=t.replace(/\/\//g,"/");var r="",i=n.attr(b.Universe10.Api.properties.baseUri.name);return i&&(r=i?i.value():""),r=r?r:"",N.stringEndsWith(r,"/")&&(t=t.substring(1)),t=r+t}function u(e,t){void 0===t&&(t=!1);var n=c(e,!0,t);return n.length>0?n[0]:null}function l(e,t){return void 0===t&&(t=!1),c(e,!1,t)}function p(e,t){void 0===t&&(t=!1);var n;n=e.isJSONString()||e.isYAML()?e.asJSON():e.original();var r={value:n,strict:e.strict(),name:e.name()};if(e.hasAnnotations()){var i=e.annotations(),a=f(i);Object.keys(a).length>0&&(r.annotations=a)}if(e.hasScalarAnnotations()){var o=e.scalarsAnnotations(),s={};Object.keys(o).forEach(function(e){var t=f(o[e]);Object.keys(t).length>0&&(s[e]=Object.keys(t).map(function(e){return t[e]}))}),Object.keys(s).length>0&&(r.scalarsAnnotations=s)}var u=e.displayName();u&&(r.displayName=u);var l=e.description();return null!=l&&(r.description=l),t&&(r.asXMLString=e.asXMLString()),r}function c(e,t,n){void 0===n&&(n=!1);var r=e.localType();r.isAnnotationType()&&(r=S.find(r.superTypes(),function(e){return e.nameId()==r.nameId()}));var i=r.examples().filter(function(e){return null!=e&&!e.isEmpty()&&e.isSingle()==t}).map(function(e){return p(e,n)});return i}function f(e){var t={};return e&&Object.keys(e).forEach(function(n){t[n]={structuredValue:e[n].value(),name:n}}),t}function d(e,t){return void 0===t&&(t=!0),e.lowLevel().actual().libExpanded?[]:m(e,function(e){return C.isTraitType(e)},t)}function h(e,t){return void 0===t&&(t=!0),e.lowLevel().actual().libExpanded?[]:m(e,function(e){return C.isResourceTypeType(e)},t)}function m(e,t,n){var r=M.globalDeclarations(e).filter(function(e){return t(e.definition())}),i=e.lowLevel(),a=i.includePath();a||(a=i.unit().path());for(var o="RAML10"==e.definition().universe().version()&&!C.isOverlayType(e.definition()),s=o?new w.TraitsAndResourceTypesExpander:null,u=[],l=new I.ReferencePatcher,p=0,c=r;p<c.length;p++){var f=c[p],d=f.lowLevel().unit().path();o&&(R.LowLevelProxyNode.isInstance(f.lowLevel())||(f=s.createHighLevelNode(f,!1)),l.process(f,e,!0,!0)),n&&d!=a&&f.wrapperNode().meta().setCalculated(),u.push(f)}return u}function y(e){return v(e)}function v(e,t){var n=e.attr(_.Universe08.BodyLike.properties.schema.name);if(!n)return null;var r=n.value();if(!r)return null;if(N.stringStartsWith(r,"{")||N.stringStartsWith(r,"[")||N.stringStartsWith(r,"<"))return n;var i;if(t)i=t[r];else{var a=e.root(),o=a.elementsOfKind(_.Universe08.Api.properties.schemas.name);i=S.find(o,function(e){return e.name()==r})}if(!i)return n;if(!i)return n;if(!i.getKind||i.getKind()!=E.NodeKind.NODE)return n;if(i.definition().key()!=_.Universe08.GlobalSchema)return n;var s=i.attr(_.Universe08.GlobalSchema.properties.value.name);return null==s?null:s}function g(e){var t=e.definition();if(C.isMethodType(t)||C.isTypeDeclarationDescendant(t)){for(var n=!1,r=e.parent();null!=r;){var i=r.definition();if(C.isResourceTypeType(i)||C.isTraitType(i)){n=!0;break}r=r.parent()}if(!n)return null}if(null==e)return null;var a=e.lowLevel();if(null==a)return null;var o=a.children().filter(function(e){var t=e.key();return t?"("==t.charAt(0)&&")"==t.charAt(t.length-1)?!1:t.indexOf("<<")>=0:!1});if(0==o.length)return null;var s={};return o.forEach(function(e){var t=e.dumpToObject();Object.keys(t).forEach(function(e){return s[e]=t[e]})}),s}function A(e){var t=e.localType(),n=t.fixedFacets(),r=Object.keys(n);if(!t.hasUnionInHierarchy())for(var i=0,a=r;i<a.length;i++){var o=a[i];null==t.facet(o)&&delete n[o]}return 0==Object.keys(n).length?null:n}var E=n(9),T=n(17),S=n(70),b=n(18),_=n(18),N=n(33),w=n(30),R=n(44),I=n(53),M=n(13),C=n(14);t.uriParameters=r,t.baseUriParameters=i,t.completeRelativeUri=o,t.absoluteUri=s,t.typeExample=u,t.typeExamples=l,t.dumpExpandableExample=p,t.allTraits=d,t.allResourceTypes=h,t.schemaContent08=y,t.schemaContent08Internal=v,t.getTemplateParametrizedProperties=g,t.typeFixedFacets=A},function(e,t,n){"use strict";function r(e){var t=e[0],n={};return e.forEach(function(e){return n[e.name]=e}),e.forEach(function(e){var t=e.imports;Object.keys(t).forEach(function(e){var r=t[e];t[e]=n[r]})}),t}t.universeDumps={RAML08:r(n(93)),RAML10:r(n(94))}},function(e,t,n){"use strict";function r(e,t){var n=new s.Universe(t,""),r={};return S(e,n,{},r),Object.keys(r).forEach(function(e){A(r[e],n)}),Object.keys(r).forEach(function(e){E(r[e],n)}),Object.keys(r).forEach(function(e){T(r[e],n)}),n.types().forEach(function(e){if(e instanceof s.NodeClass){var t=e;if(t.properties().forEach(function(e){var t=e.range(),n=e;t.hasValueTypeInHierarchy()||t.properties().forEach(function(e){if(e.getAdapter(p.RAMLPropertyService).isKey()){var t=e.keyPrefix();t&&(n.withKeyRestriction(t),n.merge());var r=e.enumOptions();r&&(n.withEnumOptions(r),n.merge())}})}),t.getAdapter(p.RAMLService).isGlobalDeclaration()){if(t.getAdapter(p.RAMLService).getActuallyExports()&&"$self"!=t.getAdapter(p.RAMLService).getActuallyExports()){var r=t.property(t.getAdapter(p.RAMLService).getActuallyExports()).range();if(r.hasValueTypeInHierarchy()){var i=r.getAdapter(p.RAMLService);i.setGloballyDeclaredBy(t)}}if(t.getAdapter(p.RAMLService).getConvertsToGlobal()){var r=n.type(t.getAdapter(p.RAMLService).getConvertsToGlobal());if(r.hasValueTypeInHierarchy()){var i=r.getAdapter(p.RAMLService);i.setGloballyDeclaredBy(t)}}}}}),n}function i(e,t,n,r,i){var a=s.prop(e.name(),"",t,n,r);e.isMultiValue()&&a.withMultiValue(!0),a.unmerge(),n||console.log(e.name()+":"+t.nameId()+" has undefined type"),n.hasValueTypeInHierarchy()||n.properties().forEach(function(e){if(e.getAdapter(p.RAMLPropertyService).isKey()){var t=e.keyPrefix();t&&(a.withKeyRestriction(t),a.merge());var n=e.enumOptions();n&&(a.withEnumOptions(n),a.merge())}}),b(e,a,i)}var a=this&&this.__extends||function(e,t){function n(){this.constructor=e}for(var r in t)t.hasOwnProperty(r)&&(e[r]=t[r]);e.prototype=null===t?Object.create(t):(n.prototype=t.prototype,new n)},o=n(96),s=n(39),u=n(95),l=n(82),p=s,c=function(){function e(e,t){this._field=e,this._clazz=t}return e.prototype.name=function(){return this._field.name},e.prototype.range=function(){return this._clazz.getModule().typeFor(this._field.type,this._clazz)},e.prototype.isMultiValue=function(){return this._field.type.typeKind==o.TypeKind.ARRAY},e.prototype.isKey=function(){return null!=u.find(this._field.annotations,function(e){return"MetaModel.key"==e.name})},e.prototype.isSimpleValue=function(){return null!=u.find(this._field.annotations,function(e){return"MetaModel.value"==e.name})},e.prototype.annotations=function(){return this._field.annotations},e}(),f=function(){function e(e,t){this._field=e,this._clazz=t}return e.prototype.name=function(){return this._field.name},e.prototype.value=function(){return this._field.valueConstraint},e}(),d=function(){function e(e,t){this._clazz=e,this.mw=t}return e.prototype.typeMeta=function(){return this._clazz.annotations},e.prototype.path=function(){return this.mw.path()},e.prototype.getModule=function(){return this.mw},e.prototype.typeArgs=function(){return this._clazz.typeParameters},e.prototype.typConstraints=function(){var e=this;return this._clazz.typeParameterConstraint.map(function(t){return t?e.mw.classForName(t):null})},e.prototype.methods=function(){return this._clazz.methods},e.prototype.name=function(){return this._clazz.name},e.prototype.members=function(){var e=this;return this._clazz.fields.filter(function(e){return null==e.valueConstraint}).map(function(t){return new c(t,e)})},e.prototype.constraints=function(){var e=this;return this._clazz.fields.filter(function(e){return null!=e.valueConstraint}).map(function(t){return new f(t,e)})},e.prototype.isSubTypeOf=function(e){if(this==e)return!0;var t=!1;return this.getAllSuperTypes().forEach(function(n){t||(t=t||n.isSubTypeOf(e))}),t},e.prototype.getExtendsClauses=function(){return this._clazz["extends"]},e.prototype.getSuperTypes=function(){var e=this,t=[];return this._clazz["extends"].forEach(function(n){var r=e.mw.classForName(n.typeName);r&&t.push(r)}),t},e.prototype.getAllSuperTypes=function(){var e=this,t=[];return this._clazz["extends"].forEach(function(n){var r=e.mw.classForName(n.typeName);if(r){var i=r.getAllSuperTypes();t.push(r),t.concat(i)}}),u.unique(t)},e.prototype.annotationOverridings=function(){return this._clazz.annotationOverridings},e}(),h=function(){function e(){}return e.prototype.members=function(){return[]},e.prototype.methods=function(){return[]},e.prototype.isSubTypeOf=function(e){return!1},e.prototype.getSuperTypes=function(){return[]},e.prototype.getAllSuperTypes=function(){return[]},e.prototype.name=function(){return null},e.prototype.constraints=function(){return[]},e.prototype.typeMeta=function(){return[]},e.prototype.getModule=function(){throw new Error("Not implemented")},e.prototype.annotationOverridings=function(){return{}},e}(),m=function(e){function t(t,n){e.call(this),this._clazz=t,this.mw=n}return a(t,e),t.prototype.getModule=function(){return this.mw},t.prototype.values=function(){return this._clazz.members},t.prototype.name=function(){return this._clazz.name},t}(h),y=function(e){function t(t,n){e.call(this),this._clazz=t,this.mw=n}return a(t,e),t.prototype.elements=function(){return this._clazz},t.prototype.name=function(){return this._clazz.map(function(e){return e.name()}).join("|")},t}(h),v=function(){function e(e){var t=this;this._univers=e,this.name2Class={},this.namespaceToMod={},this._classes=[],e.classes.forEach(function(e){var n=new d(e,t);t._classes.push(n),t.name2Class[e.name]=n,e.moduleName&&(t.name2Class[e.moduleName+"."+e.name]=n)}),e.enumDeclarations.forEach(function(e){var n=new m(e,t);t._classes.push(n),t.name2Class[e.name]=n})}return e.prototype.typeFor=function(e,t){var n=this;switch(e.typeKind){case o.TypeKind.BASIC:var r=e,i=r.typeName;"string"==i&&(i="StringType"),"number"==i&&(i="NumberType"),"boolean"==i&&(i="BooleanType"),"any"==i&&(i="AnyType");var a=u.indexOf(t.typeArgs(),i);if(-1!=a){var s=t.typConstraints()[a];return s?s:this.classForName("ValueType")}return this.classForName(i);case o.TypeKind.UNION:var l=e;return new y(l.options.map(function(e){return n.typeFor(e,t)}),this);case o.TypeKind.ARRAY:var p=e;return this.typeFor(p.base,t)}return null},e.prototype.path=function(){return this._univers.name},e.prototype.classForName=function(e,t){var n=this;if(void 0===t&&(t={}),!e)return null;var r=this.name2Class[e];if(!r&&!t[this.path()]){t[this.path()]=this;var i=e.indexOf(".");if(-1!=i){var a=this.namespaceToMod[e.substring(0,i)];if(!a)throw new Error;return a.classForName(e.substring(i+1),t)}Object.keys(this.namespaceToMod).forEach(function(i){if("MetaModel"!=i){var a=n.namespaceToMod[i].classForName(e,t);a&&(r=a)}})}return r},e.prototype.classes=function(){return this._classes},e}(),g=function(e,t){if(e){var n;if(e instanceof y){throw new Error("Union type support was removed from definition system")}return n=t.type(e.name())}},A=function(e,t){var n=e.classForName("ValueType");e.classes().forEach(function(e){if(e instanceof m){var r=new s.EnumType(e.name(),t,e.getModule().path());return r.values=e.values(),void t.register(r)}if(e.isSubTypeOf(n)){var i=e.getAllSuperTypes();i.push(e);var a=null;if(i.forEach(function(e){var t=e.getExtendsClauses();t.forEach(function(e){if(e.typeKind==o.TypeKind.BASIC){var t=e;if("Reference"==t.basicName){var n=t.typeArguments[0];a=n.typeName;var r=a.indexOf(".");-1!=r&&(a=a.substring(r+1))}}})}),a){var u=new s.ReferenceType(e.name(),e.getModule().path(),a,t);u.setBuiltIn(!0),t.register(u)}else{var l=new s.ValueType(e.name(),t,e.getModule().path());l.setBuiltIn(!0),t.register(l)}}else{var p=new s.NodeClass(e.name(),t,e.getModule().path());p.setBuiltIn(!0),t.register(p)}})},E=function(e,t){e.classes().forEach(function(e){e.getSuperTypes().forEach(function(n){var r=t.type(e.name()),i=t.type(n.name());if(!r||!i){var r=t.type(e.name()),i=t.type(n.name());throw new Error}t.registerSuperClass(r,i)})})},T=function(e,t){e.classes().forEach(function(e){var n=t.type(e.name());e.typeMeta().forEach(function(n){var r=g(e,t);l.handleTypeAnnotation(n,r)}),e.members().forEach(function(e){var r=e.range(),a=g(r,t);null==a&&console.log(r+":"+e.name());var o=e.annotations().map(function(e){return e.name}).indexOf("MetaModel.customHandling")>=0;i(e,n,a,o)}),Object.keys(e.annotationOverridings()).forEach(function(r){var a=[].concat(e.annotationOverridings()[r]),o={};a.forEach(function(e){return o[e.name]=!0});for(var s,u=e.getSuperTypes(),l={},p=0;p<u.length;p++){var c=u[p];if(!l[c.name()]){l[c.name()]=!0,c.getSuperTypes().forEach(function(e){return u.push(e)});var f=c.annotationOverridings()[r];if(f)f.filter(function(e){return!o[e.name]}).forEach(function(e){o[e.name]=!0,a.push(e)});else for(var d=c.members(),h=0;h<d.length;h++){var m=d[h];if(m.name()==r){s=m;break}}if(s){var y=s.annotations();y.filter(function(e){return!o[e.name]}).forEach(function(e){o[e.name]=!0,a.push(e)});break}}}if(s){var v=s.range(),A=g(v,t);null==A&&console.log(v+":"+e.name()),i(s,n,A,!1,a)}}),e.methods().forEach(function(e){}),e.constraints().forEach(function(e){if(e.value().isCallConstraint)throw new Error;var t=e.value();n.addRequirement(e.name(),""+t.value)})}),t.types().forEach(function(e){var n=e;n.getAdapter(p.RAMLService).getAliases().forEach(function(e){return t.registerAlias(e,n)})})},S=function(e,t,n,r){if(-1==e.name.indexOf("metamodel.ts")){if(r[e.name])return r[e.name];var i=new v(e);return n[e.name]=i,r[e.name]=i,Object.keys(e.imports).forEach(function(a){var o=e.imports[a];if(n[o.name])return void(i.namespaceToMod[a]=n[o.name]);var s=S(o,t,n,r);i.namespaceToMod[a]=s}),n[e.name]=null,i}};t.toDefSystem=r;var b=function(e,t,n){n||(n=e.annotations()),n.forEach(function(e){var n=e.name.substring(e.name.lastIndexOf(".")+1);l.annotationHandlers[n]||console.log("Can not find handler for:"),l.annotationHandlers[n](e,t)})}},function(e,t,n){"use strict";var r={Universe08:{GlobalSchema:{name:"GlobalSchema",properties:{key:{name:"key",range:"StringType",domain:"GlobalSchema"},value:{name:"value",range:"SchemaString",domain:"GlobalSchema"}}},Api:{name:"Api",properties:{title:{name:"title",range:"StringType",domain:"Api"},version:{name:"version",range:"StringType",domain:"Api"},baseUri:{name:"baseUri",range:"FullUriTemplateString",domain:"Api"},baseUriParameters:{name:"baseUriParameters",range:"Parameter",domain:"Api"},uriParameters:{name:"uriParameters",range:"Parameter",domain:"Api"},protocols:{name:"protocols",range:"StringType",domain:"Api"},mediaType:{name:"mediaType",range:"MimeType",domain:"Api"},schemas:{name:"schemas",range:"GlobalSchema",domain:"Api"},traits:{name:"traits",range:"Trait",domain:"Api"},securedBy:{name:"securedBy",range:"SecuritySchemeRef",domain:"Api"},securitySchemes:{name:"securitySchemes",range:"AbstractSecurityScheme",domain:"Api"},resourceTypes:{name:"resourceTypes",range:"ResourceType",domain:"Api"},resources:{name:"resources",range:"Resource",domain:"Api"},documentation:{name:"documentation",range:"DocumentationItem",domain:"Api"},RAMLVersion:{name:"RAMLVersion"}}},DocumentationItem:{name:"DocumentationItem",properties:{title:{name:"title",range:"StringType",domain:"DocumentationItem"},content:{name:"content",range:"MarkdownString",domain:"DocumentationItem"}}},ValueType:{name:"ValueType",properties:{}},StringType:{name:"StringType",properties:{}},AnyType:{name:"AnyType",properties:{}},NumberType:{name:"NumberType",properties:{}},BooleanType:{name:"BooleanType",properties:{}},Referencable:{name:"Referencable",properties:{}},Reference:{name:"Reference",properties:{structuredValue:{name:"structuredValue"},name:{name:"name"}}},DeclaresDynamicType:{name:"DeclaresDynamicType",properties:{}},UriTemplate:{name:"UriTemplate",properties:{}},RelativeUriString:{name:"RelativeUriString",properties:{}},FullUriTemplateString:{name:"FullUriTemplateString",properties:{}},FixedUri:{name:"FixedUri",properties:{}},MarkdownString:{name:"MarkdownString",properties:{}},SchemaString:{name:"SchemaString",properties:{}},JSonSchemaString:{name:"JSonSchemaString",properties:{}},XMLSchemaString:{name:"XMLSchemaString",properties:{}},ExampleString:{name:"ExampleString",properties:{}},StatusCodeString:{name:"StatusCodeString",properties:{}},JSONExample:{name:"JSONExample",properties:{}},XMLExample:{name:"XMLExample",properties:{}},TypeInstance:{name:"TypeInstance",properties:{properties:{name:"properties",range:"TypeInstanceProperty",domain:"TypeInstance"},isScalar:{name:"isScalar",range:"BooleanType",domain:"TypeInstance"},value:{name:"value",range:"AnyType",domain:"TypeInstance"}}},TypeInstanceProperty:{name:"TypeInstanceProperty",properties:{name:{name:"name",range:"StringType",domain:"TypeInstanceProperty"},value:{name:"value",range:"TypeInstance",domain:"TypeInstanceProperty"},values:{name:"values",range:"TypeInstance",domain:"TypeInstanceProperty"},isArray:{name:"isArray",range:"BooleanType",domain:"TypeInstanceProperty"}}},RAMLSimpleElement:{name:"RAMLSimpleElement",properties:{}},Parameter:{name:"Parameter",properties:{name:{name:"name",range:"StringType",domain:"Parameter"},displayName:{name:"displayName",range:"StringType",domain:"Parameter"},type:{name:"type",range:"StringType",domain:"Parameter"},location:{name:"location",range:"ParameterLocation",domain:"Parameter"},required:{name:"required",range:"BooleanType",domain:"Parameter"},"default":{name:"default",range:"AnyType",domain:"Parameter"},example:{name:"example",range:"StringType",domain:"Parameter"},repeat:{name:"repeat",range:"BooleanType",domain:"Parameter"},description:{name:"description",range:"MarkdownString",domain:"Parameter"}}},StringTypeDeclaration:{name:"StringTypeDeclaration",properties:{name:{name:"name",range:"StringType",domain:"Parameter"},displayName:{name:"displayName",range:"StringType",domain:"Parameter"},type:{name:"type",range:"StringType",domain:"Parameter"},location:{name:"location",range:"ParameterLocation",domain:"Parameter"},required:{name:"required",range:"BooleanType",domain:"Parameter"},"default":{name:"default",range:"AnyType",domain:"Parameter"},example:{name:"example",range:"StringType",domain:"Parameter"},repeat:{name:"repeat",range:"BooleanType",domain:"Parameter"},description:{name:"description",range:"MarkdownString",domain:"Parameter"},pattern:{name:"pattern",range:"StringType",domain:"StringTypeDeclaration"},"enum":{name:"enum",range:"StringType",domain:"StringTypeDeclaration"},minLength:{name:"minLength",range:"NumberType",domain:"StringTypeDeclaration"},maxLength:{name:"maxLength",range:"NumberType",domain:"StringTypeDeclaration"}}},BooleanTypeDeclaration:{name:"BooleanTypeDeclaration",properties:{name:{name:"name",range:"StringType",domain:"Parameter"},displayName:{name:"displayName",range:"StringType",domain:"Parameter"},type:{name:"type",range:"StringType",domain:"Parameter"},location:{name:"location",range:"ParameterLocation",domain:"Parameter"},required:{name:"required",range:"BooleanType",domain:"Parameter"},"default":{name:"default",range:"AnyType",domain:"Parameter"},example:{name:"example",range:"StringType",domain:"Parameter"},repeat:{name:"repeat",range:"BooleanType",domain:"Parameter"},description:{name:"description",range:"MarkdownString",domain:"Parameter"}}},NumberTypeDeclaration:{name:"NumberTypeDeclaration",properties:{name:{name:"name",range:"StringType",domain:"Parameter"
},displayName:{name:"displayName",range:"StringType",domain:"Parameter"},type:{name:"type",range:"StringType",domain:"Parameter"},location:{name:"location",range:"ParameterLocation",domain:"Parameter"},required:{name:"required",range:"BooleanType",domain:"Parameter"},"default":{name:"default",range:"AnyType",domain:"Parameter"},example:{name:"example",range:"StringType",domain:"Parameter"},repeat:{name:"repeat",range:"BooleanType",domain:"Parameter"},description:{name:"description",range:"MarkdownString",domain:"Parameter"},minimum:{name:"minimum",range:"NumberType",domain:"NumberTypeDeclaration"},maximum:{name:"maximum",range:"NumberType",domain:"NumberTypeDeclaration"}}},IntegerTypeDeclaration:{name:"IntegerTypeDeclaration",properties:{name:{name:"name",range:"StringType",domain:"Parameter"},displayName:{name:"displayName",range:"StringType",domain:"Parameter"},type:{name:"type",range:"StringType",domain:"Parameter"},location:{name:"location",range:"ParameterLocation",domain:"Parameter"},required:{name:"required",range:"BooleanType",domain:"Parameter"},"default":{name:"default",range:"AnyType",domain:"Parameter"},example:{name:"example",range:"StringType",domain:"Parameter"},repeat:{name:"repeat",range:"BooleanType",domain:"Parameter"},description:{name:"description",range:"MarkdownString",domain:"Parameter"},minimum:{name:"minimum",range:"NumberType",domain:"NumberTypeDeclaration"},maximum:{name:"maximum",range:"NumberType",domain:"NumberTypeDeclaration"}}},DateTypeDeclaration:{name:"DateTypeDeclaration",properties:{name:{name:"name",range:"StringType",domain:"Parameter"},displayName:{name:"displayName",range:"StringType",domain:"Parameter"},type:{name:"type",range:"StringType",domain:"Parameter"},location:{name:"location",range:"ParameterLocation",domain:"Parameter"},required:{name:"required",range:"BooleanType",domain:"Parameter"},"default":{name:"default",range:"AnyType",domain:"Parameter"},example:{name:"example",range:"StringType",domain:"Parameter"},repeat:{name:"repeat",range:"BooleanType",domain:"Parameter"},description:{name:"description",range:"MarkdownString",domain:"Parameter"}}},FileTypeDeclaration:{name:"FileTypeDeclaration",properties:{name:{name:"name",range:"StringType",domain:"Parameter"},displayName:{name:"displayName",range:"StringType",domain:"Parameter"},type:{name:"type",range:"StringType",domain:"Parameter"},location:{name:"location",range:"ParameterLocation",domain:"Parameter"},required:{name:"required",range:"BooleanType",domain:"Parameter"},"default":{name:"default",range:"AnyType",domain:"Parameter"},example:{name:"example",range:"StringType",domain:"Parameter"},repeat:{name:"repeat",range:"BooleanType",domain:"Parameter"},description:{name:"description",range:"MarkdownString",domain:"Parameter"}}},ParameterLocation:{name:"ParameterLocation",properties:{}},MimeType:{name:"MimeType",properties:{}},BodyLike:{name:"BodyLike",properties:{name:{name:"name",range:"StringType",domain:"BodyLike"},schema:{name:"schema",range:"SchemaString",domain:"BodyLike"},example:{name:"example",range:"ExampleString",domain:"BodyLike"},formParameters:{name:"formParameters",range:"Parameter",domain:"BodyLike"},description:{name:"description",range:"MarkdownString",domain:"BodyLike"},schemaContent:{name:"schemaContent"}}},XMLBody:{name:"XMLBody",properties:{name:{name:"name",range:"StringType",domain:"BodyLike"},schema:{name:"schema",range:"XMLSchemaString",domain:"XMLBody"},example:{name:"example",range:"ExampleString",domain:"BodyLike"},formParameters:{name:"formParameters",range:"Parameter",domain:"BodyLike"},description:{name:"description",range:"MarkdownString",domain:"BodyLike"}}},JSONBody:{name:"JSONBody",properties:{name:{name:"name",range:"StringType",domain:"BodyLike"},schema:{name:"schema",range:"JSonSchemaString",domain:"JSONBody"},example:{name:"example",range:"ExampleString",domain:"BodyLike"},formParameters:{name:"formParameters",range:"Parameter",domain:"BodyLike"},description:{name:"description",range:"MarkdownString",domain:"BodyLike"}}},Response:{name:"Response",properties:{code:{name:"code",range:"StatusCodeString",domain:"Response"},headers:{name:"headers",range:"Parameter",domain:"Response"},body:{name:"body",range:"BodyLike",domain:"Response"},description:{name:"description",range:"MarkdownString",domain:"Response"}}},Resource:{name:"Resource",properties:{relativeUri:{name:"relativeUri",range:"RelativeUriString",domain:"Resource"},type:{name:"type",range:"ResourceTypeRef",domain:"Resource"},is:{name:"is",range:"TraitRef",domain:"Resource"},securedBy:{name:"securedBy",range:"SecuritySchemeRef",domain:"Resource"},uriParameters:{name:"uriParameters",range:"Parameter",domain:"Resource"},methods:{name:"methods",range:"Method",domain:"Resource"},resources:{name:"resources",range:"Resource",domain:"Resource"},displayName:{name:"displayName",range:"StringType",domain:"Resource"},baseUriParameters:{name:"baseUriParameters",range:"Parameter",domain:"Resource"},description:{name:"description",range:"MarkdownString",domain:"Resource"}}},ResourceTypeRef:{name:"ResourceTypeRef",properties:{resourceType:{name:"resourceType"}}},ResourceType:{name:"ResourceType",properties:{name:{name:"name",range:"StringType",domain:"ResourceType"},usage:{name:"usage",range:"StringType",domain:"ResourceType"},methods:{name:"methods",range:"Method",domain:"ResourceType"},is:{name:"is",range:"TraitRef",domain:"ResourceType"},type:{name:"type",range:"ResourceTypeRef",domain:"ResourceType"},securedBy:{name:"securedBy",range:"SecuritySchemeRef",domain:"ResourceType"},uriParameters:{name:"uriParameters",range:"Parameter",domain:"ResourceType"},displayName:{name:"displayName",range:"StringType",domain:"ResourceType"},baseUriParameters:{name:"baseUriParameters",range:"Parameter",domain:"ResourceType"},description:{name:"description",range:"MarkdownString",domain:"ResourceType"},parametrizedProperties:{name:"parametrizedProperties"}}},MethodBase:{name:"MethodBase",properties:{responses:{name:"responses",range:"Response",domain:"MethodBase"},body:{name:"body",range:"BodyLike",domain:"MethodBase"},protocols:{name:"protocols",range:"StringType",domain:"MethodBase"},securedBy:{name:"securedBy",range:"SecuritySchemeRef",domain:"MethodBase"},baseUriParameters:{name:"baseUriParameters",range:"Parameter",domain:"MethodBase"},queryParameters:{name:"queryParameters",range:"Parameter",domain:"MethodBase"},headers:{name:"headers",range:"Parameter",domain:"MethodBase"},description:{name:"description",range:"MarkdownString",domain:"MethodBase"}}},Method:{name:"Method",properties:{responses:{name:"responses",range:"Response",domain:"MethodBase"},body:{name:"body",range:"BodyLike",domain:"MethodBase"},protocols:{name:"protocols",range:"StringType",domain:"MethodBase"},securedBy:{name:"securedBy",range:"SecuritySchemeRef",domain:"MethodBase"},baseUriParameters:{name:"baseUriParameters",range:"Parameter",domain:"MethodBase"},queryParameters:{name:"queryParameters",range:"Parameter",domain:"MethodBase"},headers:{name:"headers",range:"Parameter",domain:"MethodBase"},description:{name:"description",range:"MarkdownString",domain:"MethodBase"},method:{name:"method",range:"StringType",domain:"Method"},is:{name:"is",range:"TraitRef",domain:"Method"}}},Trait:{name:"Trait",properties:{responses:{name:"responses",range:"Response",domain:"MethodBase"},body:{name:"body",range:"BodyLike",domain:"MethodBase"},protocols:{name:"protocols",range:"StringType",domain:"MethodBase"},securedBy:{name:"securedBy",range:"SecuritySchemeRef",domain:"MethodBase"},baseUriParameters:{name:"baseUriParameters",range:"Parameter",domain:"MethodBase"},queryParameters:{name:"queryParameters",range:"Parameter",domain:"MethodBase"},headers:{name:"headers",range:"Parameter",domain:"MethodBase"},description:{name:"description",range:"MarkdownString",domain:"MethodBase"},name:{name:"name",range:"StringType",domain:"Trait"},usage:{name:"usage",range:"StringType",domain:"Trait"},displayName:{name:"displayName",range:"StringType",domain:"Trait"},parametrizedProperties:{name:"parametrizedProperties"}}},TraitRef:{name:"TraitRef",properties:{trait:{name:"trait"}}},SecuritySchemePart:{name:"SecuritySchemePart",properties:{responses:{name:"responses",range:"Response",domain:"SecuritySchemePart"},body:{name:"body",range:"BodyLike",domain:"MethodBase"},protocols:{name:"protocols",range:"StringType",domain:"MethodBase"},securedBy:{name:"securedBy",range:"SecuritySchemeRef",domain:"MethodBase"},baseUriParameters:{name:"baseUriParameters",range:"Parameter",domain:"MethodBase"},queryParameters:{name:"queryParameters",range:"Parameter",domain:"SecuritySchemePart"},headers:{name:"headers",range:"Parameter",domain:"SecuritySchemePart"},description:{name:"description",range:"MarkdownString",domain:"SecuritySchemePart"},displayName:{name:"displayName",range:"StringType",domain:"SecuritySchemePart"},is:{name:"is",range:"TraitRef",domain:"SecuritySchemePart"}}},SecuritySchemeSettings:{name:"SecuritySchemeSettings",properties:{}},AbstractSecurityScheme:{name:"AbstractSecurityScheme",properties:{name:{name:"name",range:"StringType",domain:"AbstractSecurityScheme"},type:{name:"type",range:"StringType",domain:"AbstractSecurityScheme"},description:{name:"description",range:"MarkdownString",domain:"AbstractSecurityScheme"},describedBy:{name:"describedBy",range:"SecuritySchemePart",domain:"AbstractSecurityScheme"},settings:{name:"settings",range:"SecuritySchemeSettings",domain:"AbstractSecurityScheme"}}},SecuritySchemeRef:{name:"SecuritySchemeRef",properties:{securitySchemeName:{name:"securitySchemeName"},securityScheme:{name:"securityScheme"}}},OAuth1SecuritySchemeSettings:{name:"OAuth1SecuritySchemeSettings",properties:{requestTokenUri:{name:"requestTokenUri",range:"FixedUri",domain:"OAuth1SecuritySchemeSettings"},authorizationUri:{name:"authorizationUri",range:"FixedUri",domain:"OAuth1SecuritySchemeSettings"},tokenCredentialsUri:{name:"tokenCredentialsUri",range:"FixedUri",domain:"OAuth1SecuritySchemeSettings"}}},OAuth2SecuritySchemeSettings:{name:"OAuth2SecuritySchemeSettings",properties:{accessTokenUri:{name:"accessTokenUri",range:"FixedUri",domain:"OAuth2SecuritySchemeSettings"},authorizationUri:{name:"authorizationUri",range:"FixedUri",domain:"OAuth2SecuritySchemeSettings"},authorizationGrants:{name:"authorizationGrants",range:"StringType",domain:"OAuth2SecuritySchemeSettings"},scopes:{name:"scopes",range:"StringType",domain:"OAuth2SecuritySchemeSettings"}}},OAuth2SecurityScheme:{name:"OAuth2SecurityScheme",properties:{name:{name:"name",range:"StringType",domain:"AbstractSecurityScheme"},type:{name:"type",range:"StringType",domain:"AbstractSecurityScheme"},description:{name:"description",range:"MarkdownString",domain:"AbstractSecurityScheme"},describedBy:{name:"describedBy",range:"SecuritySchemePart",domain:"AbstractSecurityScheme"},settings:{name:"settings",range:"OAuth2SecuritySchemeSettings",domain:"OAuth2SecurityScheme"}}},OAuth1SecurityScheme:{name:"OAuth1SecurityScheme",properties:{name:{name:"name",range:"StringType",domain:"AbstractSecurityScheme"},type:{name:"type",range:"StringType",domain:"AbstractSecurityScheme"},description:{name:"description",range:"MarkdownString",domain:"AbstractSecurityScheme"},describedBy:{name:"describedBy",range:"SecuritySchemePart",domain:"AbstractSecurityScheme"},settings:{name:"settings",range:"OAuth1SecuritySchemeSettings",domain:"OAuth1SecurityScheme"}}},BasicSecurityScheme:{name:"BasicSecurityScheme",properties:{name:{name:"name",range:"StringType",domain:"AbstractSecurityScheme"},type:{name:"type",range:"StringType",domain:"AbstractSecurityScheme"},description:{name:"description",range:"MarkdownString",domain:"AbstractSecurityScheme"},describedBy:{name:"describedBy",range:"SecuritySchemePart",domain:"AbstractSecurityScheme"},settings:{name:"settings",range:"SecuritySchemeSettings",domain:"AbstractSecurityScheme"}}},DigestSecurityScheme:{name:"DigestSecurityScheme",properties:{name:{name:"name",range:"StringType",domain:"AbstractSecurityScheme"},type:{name:"type",range:"StringType",domain:"AbstractSecurityScheme"},description:{name:"description",range:"MarkdownString",domain:"AbstractSecurityScheme"},describedBy:{name:"describedBy",range:"SecuritySchemePart",domain:"AbstractSecurityScheme"},settings:{name:"settings",range:"SecuritySchemeSettings",domain:"AbstractSecurityScheme"}}},CustomSecurityScheme:{name:"CustomSecurityScheme",properties:{name:{name:"name",range:"StringType",domain:"AbstractSecurityScheme"},type:{name:"type",range:"StringType",domain:"AbstractSecurityScheme"},description:{name:"description",range:"MarkdownString",domain:"AbstractSecurityScheme"},describedBy:{name:"describedBy",range:"SecuritySchemePart",domain:"AbstractSecurityScheme"},settings:{name:"settings",range:"SecuritySchemeSettings",domain:"AbstractSecurityScheme"}}}},Universe10:{Library:{name:"Library",properties:{annotations:{name:"annotations"},uses:{name:"uses"},schemas:{name:"schemas"},types:{name:"types"},traits:{name:"traits"},resourceTypes:{name:"resourceTypes"},annotationTypes:{name:"annotationTypes"},securitySchemes:{name:"securitySchemes"},usage:{name:"usage"},name:{name:"name"}}},LibraryBase:{name:"LibraryBase",properties:{annotations:{name:"annotations"},uses:{name:"uses"},schemas:{name:"schemas"},types:{name:"types"},traits:{name:"traits"},resourceTypes:{name:"resourceTypes"},annotationTypes:{name:"annotationTypes"},securitySchemes:{name:"securitySchemes"}}},Api:{name:"Api",properties:{annotations:{name:"annotations"},uses:{name:"uses"},schemas:{name:"schemas"},types:{name:"types"},traits:{name:"traits"},resourceTypes:{name:"resourceTypes"},annotationTypes:{name:"annotationTypes"},securitySchemes:{name:"securitySchemes"},title:{name:"title"},description:{name:"description"},version:{name:"version"},baseUri:{name:"baseUri"},baseUriParameters:{name:"baseUriParameters"},protocols:{name:"protocols"},mediaType:{name:"mediaType"},securedBy:{name:"securedBy"},resources:{name:"resources"},documentation:{name:"documentation"},RAMLVersion:{name:"RAMLVersion",range:"StringType",domain:"Api"}}},Overlay:{name:"Overlay",properties:{annotations:{name:"annotations"},uses:{name:"uses"},schemas:{name:"schemas"},types:{name:"types"},traits:{name:"traits"},resourceTypes:{name:"resourceTypes"},annotationTypes:{name:"annotationTypes"},securitySchemes:{name:"securitySchemes"},title:{name:"title"},description:{name:"description"},version:{name:"version"},baseUri:{name:"baseUri"},baseUriParameters:{name:"baseUriParameters"},protocols:{name:"protocols"},mediaType:{name:"mediaType"},securedBy:{name:"securedBy"},resources:{name:"resources"},documentation:{name:"documentation"},usage:{name:"usage"},"extends":{name:"extends"}}},Extension:{name:"Extension",properties:{annotations:{name:"annotations"},uses:{name:"uses"},schemas:{name:"schemas"},types:{name:"types"},traits:{name:"traits"},resourceTypes:{name:"resourceTypes"},annotationTypes:{name:"annotationTypes"},securitySchemes:{name:"securitySchemes"},title:{name:"title"},description:{name:"description"},version:{name:"version"},baseUri:{name:"baseUri"},baseUriParameters:{name:"baseUriParameters"},protocols:{name:"protocols"},mediaType:{name:"mediaType"},securedBy:{name:"securedBy"},resources:{name:"resources"},documentation:{name:"documentation"},usage:{name:"usage"},"extends":{name:"extends"}}},UsesDeclaration:{name:"UsesDeclaration",properties:{annotations:{name:"annotations"},key:{name:"key"},value:{name:"value"}}},FragmentDeclaration:{name:"FragmentDeclaration",properties:{annotations:{name:"annotations"},uses:{name:"uses"}}},DocumentationItem:{name:"DocumentationItem",properties:{annotations:{name:"annotations"},title:{name:"title"},content:{name:"content"}}},ValueType:{name:"ValueType",properties:{annotations:{name:"annotations"}}},StringType:{name:"StringType",properties:{annotations:{name:"annotations"}}},AnyType:{name:"AnyType",properties:{annotations:{name:"annotations"}}},NumberType:{name:"NumberType",properties:{annotations:{name:"annotations"}}},IntegerType:{name:"IntegerType",properties:{annotations:{name:"annotations"}}},NullType:{name:"NullType",properties:{annotations:{name:"annotations"}}},TimeOnlyType:{name:"TimeOnlyType",properties:{annotations:{name:"annotations"}}},DateOnlyType:{name:"DateOnlyType",properties:{annotations:{name:"annotations"}}},DateTimeOnlyType:{name:"DateTimeOnlyType",properties:{annotations:{name:"annotations"}}},DateTimeType:{name:"DateTimeType",properties:{annotations:{name:"annotations"}}},FileType:{name:"FileType",properties:{annotations:{name:"annotations"}}},BooleanType:{name:"BooleanType",properties:{annotations:{name:"annotations"}}},Reference:{name:"Reference",properties:{annotations:{name:"annotations"},structuredValue:{name:"structuredValue",range:"TypeInstance",domain:"Reference"},name:{name:"name",range:"StringType",domain:"Reference"}}},UriTemplate:{name:"UriTemplate",properties:{annotations:{name:"annotations"}}},StatusCodeString:{name:"StatusCodeString",properties:{annotations:{name:"annotations"}}},RelativeUriString:{name:"RelativeUriString",properties:{annotations:{name:"annotations"}}},FullUriTemplateString:{name:"FullUriTemplateString",properties:{annotations:{name:"annotations"}}},FixedUriString:{name:"FixedUriString",properties:{annotations:{name:"annotations"}}},ContentType:{name:"ContentType",properties:{annotations:{name:"annotations"}}},MarkdownString:{name:"MarkdownString",properties:{annotations:{name:"annotations"}}},SchemaString:{name:"SchemaString",properties:{annotations:{name:"annotations"}}},ExampleSpec:{name:"ExampleSpec",properties:{annotations:{name:"annotations"},value:{name:"value"},strict:{name:"strict"},name:{name:"name"},displayName:{name:"displayName"},description:{name:"description"},structuredValue:{name:"structuredValue",range:"TypeInstance",domain:"ExampleSpec"}}},TypeDeclaration:{name:"TypeDeclaration",properties:{annotations:{name:"annotations"},name:{name:"name"},displayName:{name:"displayName"},facets:{name:"facets"},schema:{name:"schema"},type:{name:"type"},location:{name:"location"},locationKind:{name:"locationKind"},"default":{name:"default"},example:{name:"example"},examples:{name:"examples"},required:{name:"required"},description:{name:"description"},xml:{name:"xml"},allowedTargets:{name:"allowedTargets"},isAnnotation:{name:"isAnnotation"},fixedFacets:{name:"fixedFacets",range:"TypeInstance",domain:"TypeDeclaration"},structuredType:{name:"structuredType",range:"TypeInstance",domain:"TypeDeclaration"},parametrizedProperties:{name:"parametrizedProperties",range:"TypeInstance",domain:"TypeDeclaration"}}},XMLFacetInfo:{name:"XMLFacetInfo",properties:{annotations:{name:"annotations"},attribute:{name:"attribute"},wrapped:{name:"wrapped"},name:{name:"name"},namespace:{name:"namespace"},prefix:{name:"prefix"}}},ArrayTypeDeclaration:{name:"ArrayTypeDeclaration",properties:{annotations:{name:"annotations"},name:{name:"name"},displayName:{name:"displayName"},facets:{name:"facets"},schema:{name:"schema"},type:{name:"type"},location:{name:"location"},locationKind:{name:"locationKind"},"default":{name:"default"},example:{name:"example"},examples:{name:"examples"},required:{name:"required"},description:{name:"description"},xml:{name:"xml"},allowedTargets:{name:"allowedTargets"},isAnnotation:{name:"isAnnotation"},uniqueItems:{name:"uniqueItems"},items:{name:"items"},minItems:{name:"minItems"},maxItems:{name:"maxItems"},structuredItems:{name:"structuredItems",range:"TypeInstance",domain:"ArrayTypeDeclaration"}}},UnionTypeDeclaration:{name:"UnionTypeDeclaration",properties:{annotations:{name:"annotations"},name:{name:"name"},displayName:{name:"displayName"},facets:{name:"facets"},schema:{name:"schema"},type:{name:"type"},location:{name:"location"},locationKind:{name:"locationKind"},"default":{name:"default"},example:{name:"example"},examples:{name:"examples"},required:{name:"required"},description:{name:"description"},xml:{name:"xml"},allowedTargets:{name:"allowedTargets"},isAnnotation:{name:"isAnnotation"}}},ObjectTypeDeclaration:{name:"ObjectTypeDeclaration",properties:{annotations:{name:"annotations"},name:{name:"name"},displayName:{name:"displayName"},facets:{name:"facets"},schema:{name:"schema"},type:{name:"type"},location:{name:"location"},locationKind:{name:"locationKind"},"default":{name:"default"},example:{name:"example"},examples:{name:"examples"},required:{name:"required"},description:{name:"description"},xml:{name:"xml"},allowedTargets:{name:"allowedTargets"},isAnnotation:{name:"isAnnotation"},properties:{name:"properties"},minProperties:{name:"minProperties"},maxProperties:{name:"maxProperties"},additionalProperties:{name:"additionalProperties"},discriminator:{name:"discriminator"},discriminatorValue:{name:"discriminatorValue"},"enum":{name:"enum"}}},StringTypeDeclaration:{name:"StringTypeDeclaration",properties:{annotations:{name:"annotations"},name:{name:"name"},displayName:{name:"displayName"},facets:{name:"facets"},schema:{name:"schema"},type:{name:"type"},location:{name:"location"},locationKind:{name:"locationKind"},"default":{name:"default"},example:{name:"example"},examples:{name:"examples"},required:{name:"required"},description:{name:"description"},xml:{name:"xml"},allowedTargets:{name:"allowedTargets"},isAnnotation:{name:"isAnnotation"},pattern:{name:"pattern"},minLength:{name:"minLength"},maxLength:{name:"maxLength"},"enum":{name:"enum"}}},BooleanTypeDeclaration:{name:"BooleanTypeDeclaration",properties:{annotations:{name:"annotations"},name:{name:"name"},displayName:{name:"displayName"},facets:{name:"facets"},schema:{name:"schema"},type:{name:"type"},location:{name:"location"},locationKind:{name:"locationKind"},"default":{name:"default"},example:{name:"example"},examples:{name:"examples"},required:{name:"required"},description:{name:"description"},xml:{name:"xml"},allowedTargets:{name:"allowedTargets"},isAnnotation:{name:"isAnnotation"},"enum":{name:"enum"}}},NumberTypeDeclaration:{name:"NumberTypeDeclaration",properties:{annotations:{name:"annotations"},name:{name:"name"},displayName:{name:"displayName"},facets:{name:"facets"},schema:{name:"schema"},type:{name:"type"},location:{name:"location"},locationKind:{name:"locationKind"},"default":{name:"default"},example:{name:"example"},examples:{name:"examples"},required:{name:"required"},description:{name:"description"},xml:{name:"xml"},allowedTargets:{name:"allowedTargets"},isAnnotation:{name:"isAnnotation"},minimum:{name:"minimum"},maximum:{name:"maximum"},"enum":{name:"enum"},format:{name:"format"},multipleOf:{name:"multipleOf"}}},IntegerTypeDeclaration:{name:"IntegerTypeDeclaration",properties:{annotations:{name:"annotations"},name:{name:"name"},displayName:{name:"displayName"},facets:{name:"facets"},schema:{name:"schema"},type:{name:"type"},location:{name:"location"},locationKind:{name:"locationKind"},"default":{name:"default"},example:{name:"example"},examples:{name:"examples"},required:{name:"required"},description:{name:"description"},xml:{name:"xml"},allowedTargets:{name:"allowedTargets"},isAnnotation:{name:"isAnnotation"},minimum:{name:"minimum"},maximum:{name:"maximum"},"enum":{name:"enum"},format:{name:"format"},multipleOf:{name:"multipleOf"}}},DateOnlyTypeDeclaration:{name:"DateOnlyTypeDeclaration",properties:{annotations:{name:"annotations"},name:{name:"name"},displayName:{name:"displayName"},facets:{name:"facets"},schema:{name:"schema"},type:{name:"type"},location:{name:"location"},locationKind:{name:"locationKind"},"default":{name:"default"},example:{name:"example"},examples:{name:"examples"},required:{name:"required"},description:{name:"description"},xml:{name:"xml"},allowedTargets:{name:"allowedTargets"},isAnnotation:{name:"isAnnotation"}}},TimeOnlyTypeDeclaration:{name:"TimeOnlyTypeDeclaration",properties:{annotations:{name:"annotations"},name:{name:"name"},displayName:{name:"displayName"},facets:{name:"facets"},schema:{name:"schema"},type:{name:"type"},location:{name:"location"},locationKind:{name:"locationKind"},"default":{name:"default"},example:{name:"example"},examples:{name:"examples"},required:{name:"required"},description:{name:"description"},xml:{name:"xml"},allowedTargets:{name:"allowedTargets"},isAnnotation:{name:"isAnnotation"}}},DateTimeOnlyTypeDeclaration:{name:"DateTimeOnlyTypeDeclaration",properties:{annotations:{name:"annotations"},name:{name:"name"},displayName:{name:"displayName"},facets:{name:"facets"},schema:{name:"schema"},type:{name:"type"},location:{name:"location"},locationKind:{name:"locationKind"},"default":{name:"default"},example:{name:"example"},examples:{name:"examples"},required:{name:"required"},description:{name:"description"},xml:{name:"xml"},allowedTargets:{name:"allowedTargets"},isAnnotation:{name:"isAnnotation"}}},DateTimeTypeDeclaration:{name:"DateTimeTypeDeclaration",properties:{annotations:{name:"annotations"},name:{name:"name"},displayName:{name:"displayName"},facets:{name:"facets"},schema:{name:"schema"},type:{name:"type"},location:{name:"location"},locationKind:{name:"locationKind"},"default":{name:"default"},example:{name:"example"},examples:{name:"examples"},required:{name:"required"},description:{name:"description"},xml:{name:"xml"},allowedTargets:{name:"allowedTargets"},isAnnotation:{name:"isAnnotation"},format:{name:"format"}}},TypeInstance:{name:"TypeInstance",properties:{properties:{name:"properties"},isScalar:{name:"isScalar"},value:{name:"value"},isArray:{name:"isArray"},items:{name:"items"}}},TypeInstanceProperty:{name:"TypeInstanceProperty",properties:{name:{name:"name"},value:{name:"value"},values:{name:"values"},isArray:{name:"isArray"}}},ModelLocation:{name:"ModelLocation",properties:{}},LocationKind:{name:"LocationKind",properties:{}},MimeType:{name:"MimeType",properties:{annotations:{name:"annotations"}}},Response:{name:"Response",properties:{annotations:{name:"annotations"},code:{name:"code"},headers:{name:"headers"},body:{name:"body"},description:{name:"description"}}},Annotable:{name:"Annotable",properties:{annotations:{name:"annotations"}}},AnnotationRef:{name:"AnnotationRef",properties:{annotations:{name:"annotations"},annotation:{name:"annotation",range:"TypeDeclaration",domain:"AnnotationRef"}}},AnnotationTarget:{name:"AnnotationTarget",properties:{annotations:{name:"annotations"}}},TraitRef:{name:"TraitRef",properties:{annotations:{name:"annotations"},trait:{name:"trait",range:"Trait",domain:"TraitRef"}}},Trait:{name:"Trait",properties:{annotations:{name:"annotations"},queryParameters:{name:"queryParameters"},headers:{name:"headers"},queryString:{name:"queryString"},responses:{name:"responses"},body:{name:"body"},protocols:{name:"protocols"},is:{name:"is"},securedBy:{name:"securedBy"},description:{name:"description"},displayName:{name:"displayName"},name:{name:"name"},usage:{name:"usage"},parametrizedProperties:{name:"parametrizedProperties",range:"TypeInstance",domain:"Trait"}}},MethodBase:{name:"MethodBase",properties:{annotations:{name:"annotations"},queryParameters:{name:"queryParameters"},headers:{name:"headers"},queryString:{name:"queryString"},responses:{name:"responses"},body:{name:"body"},protocols:{name:"protocols"},is:{name:"is"},securedBy:{name:"securedBy"},description:{name:"description"},displayName:{name:"displayName"}}},Method:{name:"Method",properties:{annotations:{name:"annotations"},queryParameters:{name:"queryParameters"},headers:{name:"headers"},queryString:{name:"queryString"},responses:{name:"responses"},body:{name:"body"},protocols:{name:"protocols"},is:{name:"is"},securedBy:{name:"securedBy"},description:{name:"description"},displayName:{name:"displayName"},method:{name:"method"},parametrizedProperties:{name:"parametrizedProperties",range:"TypeInstance",domain:"Method"}}},Operation:{name:"Operation",properties:{annotations:{name:"annotations"},queryParameters:{name:"queryParameters"},headers:{name:"headers"},queryString:{name:"queryString"},responses:{name:"responses"}}},SecuritySchemePart:{name:"SecuritySchemePart",properties:{annotations:{name:"annotations"},queryParameters:{name:"queryParameters"},headers:{name:"headers"},queryString:{name:"queryString"},responses:{name:"responses"}}},SecuritySchemeSettings:{name:"SecuritySchemeSettings",properties:{annotations:{name:"annotations"}}},OAuth1SecuritySchemeSettings:{name:"OAuth1SecuritySchemeSettings",properties:{annotations:{name:"annotations"},requestTokenUri:{name:"requestTokenUri"},authorizationUri:{name:"authorizationUri"},tokenCredentialsUri:{name:"tokenCredentialsUri"},signatures:{name:"signatures"}}},OAuth2SecuritySchemeSettings:{name:"OAuth2SecuritySchemeSettings",properties:{annotations:{name:"annotations"},accessTokenUri:{name:"accessTokenUri"},authorizationUri:{name:"authorizationUri"},authorizationGrants:{name:"authorizationGrants"},scopes:{name:"scopes"}}},SecuritySchemeRef:{name:"SecuritySchemeRef",properties:{annotations:{name:"annotations"},securitySchemeName:{name:"securitySchemeName",range:"StringType",domain:"SecuritySchemeRef"},securityScheme:{name:"securityScheme",range:"AbstractSecurityScheme",domain:"SecuritySchemeRef"}}},AbstractSecurityScheme:{name:"AbstractSecurityScheme",properties:{annotations:{name:"annotations"},name:{name:"name"},type:{name:"type"},description:{name:"description"},describedBy:{name:"describedBy"},displayName:{name:"displayName"},settings:{name:"settings"}}},OAuth2SecurityScheme:{name:"OAuth2SecurityScheme",properties:{annotations:{name:"annotations"},name:{name:"name"},type:{name:"type"},description:{name:"description"},describedBy:{name:"describedBy"},displayName:{name:"displayName"},settings:{name:"settings"}}},OAuth1SecurityScheme:{name:"OAuth1SecurityScheme",properties:{annotations:{name:"annotations"},name:{name:"name"},type:{name:"type"},description:{name:"description"},describedBy:{name:"describedBy"},displayName:{name:"displayName"},settings:{name:"settings"}}},PassThroughSecurityScheme:{name:"PassThroughSecurityScheme",properties:{annotations:{name:"annotations"},name:{name:"name"},type:{name:"type"},description:{name:"description"},describedBy:{name:"describedBy"},displayName:{name:"displayName"},settings:{name:"settings"}}},BasicSecurityScheme:{name:"BasicSecurityScheme",properties:{annotations:{name:"annotations"},name:{name:"name"},type:{name:"type"},description:{name:"description"},describedBy:{name:"describedBy"},displayName:{name:"displayName"},settings:{name:"settings"}}},DigestSecurityScheme:{name:"DigestSecurityScheme",properties:{annotations:{name:"annotations"},name:{name:"name"},type:{name:"type"},description:{name:"description"},describedBy:{name:"describedBy"},displayName:{name:"displayName"},settings:{name:"settings"}}},CustomSecurityScheme:{name:"CustomSecurityScheme",properties:{annotations:{name:"annotations"},name:{name:"name"},type:{name:"type"},description:{name:"description"},describedBy:{name:"describedBy"},displayName:{name:"displayName"},settings:{name:"settings"}}},ResourceTypeRef:{name:"ResourceTypeRef",properties:{annotations:{name:"annotations"},resourceType:{name:"resourceType",range:"ResourceType",domain:"ResourceTypeRef"}}},ResourceType:{name:"ResourceType",properties:{annotations:{name:"annotations"},methods:{name:"methods"},is:{name:"is"},type:{name:"type"},description:{name:"description"},securedBy:{name:"securedBy"},uriParameters:{name:"uriParameters"},displayName:{name:"displayName"},name:{name:"name"},usage:{name:"usage"},parametrizedProperties:{name:"parametrizedProperties",range:"TypeInstance",domain:"ResourceType"}}},ResourceBase:{name:"ResourceBase",properties:{annotations:{name:"annotations"},methods:{name:"methods"},is:{name:"is"},type:{name:"type"},description:{name:"description"},securedBy:{name:"securedBy"},uriParameters:{name:"uriParameters"}}},Resource:{name:"Resource",properties:{annotations:{name:"annotations"},methods:{name:"methods"},is:{name:"is"},type:{name:"type"},description:{name:"description"},securedBy:{name:"securedBy"},uriParameters:{name:"uriParameters"},relativeUri:{name:"relativeUri"
},displayName:{name:"displayName"},resources:{name:"resources"}}},FileTypeDeclaration:{name:"FileTypeDeclaration",properties:{annotations:{name:"annotations"},name:{name:"name"},displayName:{name:"displayName"},facets:{name:"facets"},schema:{name:"schema"},type:{name:"type"},location:{name:"location"},locationKind:{name:"locationKind"},"default":{name:"default"},example:{name:"example"},examples:{name:"examples"},required:{name:"required"},description:{name:"description"},xml:{name:"xml"},allowedTargets:{name:"allowedTargets"},isAnnotation:{name:"isAnnotation"},fileTypes:{name:"fileTypes"},minLength:{name:"minLength"},maxLength:{name:"maxLength"}}}}};e.exports=r},function(e,t,n){t.XMLHttpRequest=XMLHttpRequest},function(e,t,n){function r(){if(!s){s=!0;for(var e,t=o.length;t;){e=o,o=[];for(var n=-1;++n<t;)e[n]();t=o.length}s=!1}}function i(){}var a=e.exports={},o=[],s=!1;a.nextTick=function(e){o.push(e),s||setTimeout(r,0)},a.title="browser",a.browser=!0,a.env={},a.argv=[],a.version="",a.versions={},a.on=i,a.addListener=i,a.once=i,a.off=i,a.removeListener=i,a.removeAllListeners=i,a.emit=i,a.binding=function(e){throw new Error("process.binding is not supported")},a.cwd=function(){return"/"},a.chdir=function(e){throw new Error("process.chdir is not supported")},a.umask=function(){return 0}},function(e,t,n){"use strict";function r(e,t,n,r){if("number"!=typeof e)throw new TypeError("statusCode must be a number but was "+typeof e);if(null===t)throw new TypeError("headers cannot be null");if("object"!=typeof t)throw new TypeError("headers must be an object but was "+typeof t);this.statusCode=e,this.headers={};for(var i in t)this.headers[i.toLowerCase()]=t[i];this.body=n,this.url=r}e.exports=r,r.prototype.getBody=function(e){if(this.statusCode>=300){var t=new Error("Server responded with status code "+this.statusCode+":\n"+this.body.toString());throw t.statusCode=this.statusCode,t.headers=this.headers,t.body=this.body,t.url=this.url,t}return e?this.body.toString(e):this.body}},function(e,t,n){(function(t){function r(e,t){if(!(this instanceof r))return new r(e,t);"function"==typeof e&&(t=e,e={}),e||(e={});var n=e.encoding,i=!1;n?(n=String(n).toLowerCase(),("u8"===n||"uint8"===n)&&(n="uint8array")):i=!0,p.call(this,{objectMode:!0}),this.encoding=n,this.shouldInferEncoding=i,t&&this.on("finish",function(){t(this.getBody())}),this.body=[]}function i(e){return/Array\]$/.test(Object.prototype.toString.call(e))}function a(e){return"string"==typeof e||i(e)||e&&"function"==typeof e.subarray}function o(e){for(var n=[],r=0;r<e.length;r++){var i=e[r];"string"==typeof i?n.push(i):t.isBuffer(i)?n.push(i):a(i)?n.push(new t(i)):n.push(new t(String(i)))}return t.isBuffer(e[0])?(n=t.concat(n),n=n.toString("utf8")):n=n.join(""),n}function s(e){for(var n=[],r=0;r<e.length;r++){var i=e[r];t.isBuffer(i)?n.push(i):a(i)?n.push(new t(i)):n.push(new t(String(i)))}return t.concat(n)}function u(e){for(var t=[],n=0;n<e.length;n++)t.push.apply(t,e[n]);return t}function l(e){for(var n=0,r=0;r<e.length;r++)"string"==typeof e[r]&&(e[r]=new t(e[r])),n+=e[r].length;for(var i=new f(n),r=0,a=0;r<e.length;r++)for(var o=e[r],s=0;s<o.length;s++)i[a++]=o[s];return i}var p=n(114).Writable,c=n(115);if("undefined"==typeof Uint8Array)var f=n(111).Uint8Array;else var f=Uint8Array;e.exports=r,c(r,p),r.prototype._write=function(e,t,n){this.body.push(e),n()},r.prototype.inferEncoding=function(e){var n=void 0===e?this.body[0]:e;return t.isBuffer(n)?"buffer":"undefined"!=typeof Uint8Array&&n instanceof Uint8Array?"uint8array":Array.isArray(n)?"array":"string"==typeof n?"string":"[object Object]"===Object.prototype.toString.call(n)?"object":"buffer"},r.prototype.getBody=function(){return this.encoding||0!==this.body.length?(this.shouldInferEncoding&&(this.encoding=this.inferEncoding()),"array"===this.encoding?u(this.body):"string"===this.encoding?o(this.body):"buffer"===this.encoding?s(this.body):"uint8array"===this.encoding?l(this.body):this.body):[]};Array.isArray||function(e){return"[object Array]"==Object.prototype.toString.call(e)}}).call(t,n(97).Buffer)},function(e,t,n){(function(t){function n(e){return e.replace(/=/g,"").replace(/\+/g,"-").replace(/\//g,"_")}function r(e){t.isBuffer(e)&&(e=e.toString());var n=i(e).replace(/\-/g,"+").replace(/_/g,"/");return n}function i(e){var n=4,r=e.length,i=e.length%n;if(!i)return e;var a=r,o=n-i,s=r+o,u=t(s);for(u.write(e);o--;)u.write("=",a++);return u.toString()}function a(e,n){return t(r(e),"base64").toString(n)}function o(e,r){return n(t(e,r).toString("base64"))}function s(e){return t(r(e),"base64")}o.toBase64=r,o.fromBase64=n,o.decode=a,o.encode=o,o.toBuffer=s,e.exports=o}).call(t,n(97).Buffer)},function(e,t,n){function r(e){if(!e||"object"!=typeof e)throw new TypeError("argument obj is required");var t=e.parameters,n=e.subtype,r=e.suffix,i=e.type;if(!i||!h.test(i))throw new TypeError("invalid type");if(!n||!d.test(n))throw new TypeError("invalid subtype");var a=i+"/"+n;if(r){if(!h.test(r))throw new TypeError("invalid suffix");a+="+"+r}if(t&&"object"==typeof t)for(var s,u=Object.keys(t).sort(),l=0;l<u.length;l++){if(s=u[l],!p.test(s))throw new TypeError("invalid parameter name");a+="; "+s+"="+o(t[s])}return a}function i(e){if(!e)throw new TypeError("argument string is required");if("object"==typeof e&&(e=a(e)),"string"!=typeof e)throw new TypeError("argument string is required to be a string");var t,n,r,i=e.indexOf(";"),o=-1!==i?e.substr(0,i):e,l=s(o),p={};for(u.lastIndex=i;n=u.exec(e);){if(n.index!==i)throw new TypeError("invalid parameter format");i+=n[0].length,t=n[1].toLowerCase(),r=n[2],'"'===r[0]&&(r=r.substr(1,r.length-2).replace(c,"$1")),p[t]=r}if(-1!==i&&i!==e.length)throw new TypeError("invalid parameter format");return l.parameters=p,l}function a(e){return"function"==typeof e.getHeader?e.getHeader("content-type"):"object"==typeof e.headers?e.headers&&e.headers["content-type"]:void 0}function o(e){var t=String(e);if(p.test(t))return t;if(t.length>0&&!l.test(t))throw new TypeError("invalid parameter value");return'"'+t.replace(f,"\\$1")+'"'}function s(e){var t=m.exec(e.toLowerCase());if(!t)throw new TypeError("invalid media type");var n,r=t[1],i=t[2],a=i.lastIndexOf("+");-1!==a&&(n=i.substr(a+1),i=i.substr(0,a));var o={type:r,subtype:i,suffix:n};return o}/*!
	 * media-typer
	 * Copyright(c) 2014 Douglas Christopher Wilson
	 * MIT Licensed
	 */
var u=/; *([!#$%&'\*\+\-\.0-9A-Z\^_`a-z\|~]+) *= *("(?:[ !\u0023-\u005b\u005d-\u007e\u0080-\u00ff]|\\[\u0020-\u007e])*"|[!#$%&'\*\+\-\.0-9A-Z\^_`a-z\|~]+) */g,l=/^[\u0020-\u007e\u0080-\u00ff]+$/,p=/^[!#$%&'\*\+\-\.0-9A-Z\^_`a-z\|~]+$/,c=/\\([\u0000-\u007f])/g,f=/([\\"])/g,d=/^[A-Za-z0-9][A-Za-z0-9!#$&^_.-]{0,126}$/,h=/^[A-Za-z0-9][A-Za-z0-9!#$&^_-]{0,126}$/,m=/^ *([A-Za-z0-9][A-Za-z0-9!#$&^_-]{0,126})\/([A-Za-z0-9][A-Za-z0-9!#$&^_.+-]{0,126}) *$/;t.format=r,t.parse=i},function(e,t,n){var r="undefined"!=typeof JSON?JSON:n(112);e.exports=function(e,t){t||(t={}),"function"==typeof t&&(t={cmp:t});var n=t.space||"";"number"==typeof n&&(n=Array(n+1).join(" "));var o="boolean"==typeof t.cycles?t.cycles:!1,s=t.replacer||function(e,t){return t},u=t.cmp&&function(e){return function(t){return function(n,r){var i={key:n,value:t[n]},a={key:r,value:t[r]};return e(i,a)}}}(t.cmp),l=[];return function p(e,t,c,f){var d=n?"\n"+new Array(f+1).join(n):"",h=n?": ":":";if(c&&c.toJSON&&"function"==typeof c.toJSON&&(c=c.toJSON()),c=s.call(e,t,c),void 0!==c){if("object"!=typeof c||null===c)return r.stringify(c);if(i(c)){for(var m=[],y=0;y<c.length;y++){var v=p(c,y,c[y],f+1)||r.stringify(null);m.push(d+n+v)}return"["+m.join(",")+d+"]"}if(-1!==l.indexOf(c)){if(o)return r.stringify("__cycle__");throw new TypeError("Converting circular structure to JSON")}l.push(c);for(var g=a(c).sort(u&&u(c)),m=[],y=0;y<g.length;y++){var t=g[y],A=p(c,t,c[t],f+1);if(A){var E=r.stringify(t)+h+A;m.push(d+n+E)}}return l.splice(l.indexOf(c),1),"{"+m.join(",")+d+"}"}}({"":e},"",e,0)};var i=Array.isArray||function(e){return"[object Array]"==={}.toString.call(e)},a=Object.keys||function(e){var t=Object.prototype.hasOwnProperty||function(){return!0},n=[];for(var r in e)t.call(e,r)&&n.push(r);return n}},function(e,t,n){var r,i;(function(){function n(e){function t(t,n,r,i,a,o){for(;a>=0&&o>a;a+=e){var s=i?i[a]:a;r=n(r,t[s],s,t)}return r}return function(n,r,i,a){r=b(r,a,4);var o=!C(n)&&S.keys(n),s=(o||n).length,u=e>0?0:s-1;return arguments.length<3&&(i=n[o?o[u]:u],u+=e),t(n,r,i,o,u,s)}}function a(e){return function(t,n,r){n=_(n,r);for(var i=M(t),a=e>0?0:i-1;a>=0&&i>a;a+=e)if(n(t[a],a,t))return a;return-1}}function o(e,t,n){return function(r,i,a){var o=0,s=M(r);if("number"==typeof a)e>0?o=a>=0?a:Math.max(a+s,o):s=a>=0?Math.min(a+1,s):a+s+1;else if(n&&a&&s)return a=n(r,i),r[a]===i?a:-1;if(i!==i)return a=t(h.call(r,o,s),S.isNaN),a>=0?a+o:-1;for(a=e>0?o:s-1;a>=0&&s>a;a+=e)if(r[a]===i)return a;return-1}}function s(e,t){var n=x.length,r=e.constructor,i=S.isFunction(r)&&r.prototype||c,a="constructor";for(S.has(e,a)&&!S.contains(t,a)&&t.push(a);n--;)a=x[n],a in e&&e[a]!==i[a]&&!S.contains(t,a)&&t.push(a)}var u=this,l=u._,p=Array.prototype,c=Object.prototype,f=Function.prototype,d=p.push,h=p.slice,m=c.toString,y=c.hasOwnProperty,v=Array.isArray,g=Object.keys,A=f.bind,E=Object.create,T=function(){},S=function(e){return e instanceof S?e:this instanceof S?void(this._wrapped=e):new S(e)};"undefined"!=typeof e&&e.exports&&(t=e.exports=S),t._=S,S.VERSION="1.8.3";var b=function(e,t,n){if(void 0===t)return e;switch(null==n?3:n){case 1:return function(n){return e.call(t,n)};case 2:return function(n,r){return e.call(t,n,r)};case 3:return function(n,r,i){return e.call(t,n,r,i)};case 4:return function(n,r,i,a){return e.call(t,n,r,i,a)}}return function(){return e.apply(t,arguments)}},_=function(e,t,n){return null==e?S.identity:S.isFunction(e)?b(e,t,n):S.isObject(e)?S.matcher(e):S.property(e)};S.iteratee=function(e,t){return _(e,t,1/0)};var N=function(e,t){return function(n){var r=arguments.length;if(2>r||null==n)return n;for(var i=1;r>i;i++)for(var a=arguments[i],o=e(a),s=o.length,u=0;s>u;u++){var l=o[u];t&&void 0!==n[l]||(n[l]=a[l])}return n}},w=function(e){if(!S.isObject(e))return{};if(E)return E(e);T.prototype=e;var t=new T;return T.prototype=null,t},R=function(e){return function(t){return null==t?void 0:t[e]}},I=Math.pow(2,53)-1,M=R("length"),C=function(e){var t=M(e);return"number"==typeof t&&t>=0&&I>=t};S.each=S.forEach=function(e,t,n){t=b(t,n);var r,i;if(C(e))for(r=0,i=e.length;i>r;r++)t(e[r],r,e);else{var a=S.keys(e);for(r=0,i=a.length;i>r;r++)t(e[a[r]],a[r],e)}return e},S.map=S.collect=function(e,t,n){t=_(t,n);for(var r=!C(e)&&S.keys(e),i=(r||e).length,a=Array(i),o=0;i>o;o++){var s=r?r[o]:o;a[o]=t(e[s],s,e)}return a},S.reduce=S.foldl=S.inject=n(1),S.reduceRight=S.foldr=n(-1),S.find=S.detect=function(e,t,n){var r;return r=C(e)?S.findIndex(e,t,n):S.findKey(e,t,n),void 0!==r&&-1!==r?e[r]:void 0},S.filter=S.select=function(e,t,n){var r=[];return t=_(t,n),S.each(e,function(e,n,i){t(e,n,i)&&r.push(e)}),r},S.reject=function(e,t,n){return S.filter(e,S.negate(_(t)),n)},S.every=S.all=function(e,t,n){t=_(t,n);for(var r=!C(e)&&S.keys(e),i=(r||e).length,a=0;i>a;a++){var o=r?r[a]:a;if(!t(e[o],o,e))return!1}return!0},S.some=S.any=function(e,t,n){t=_(t,n);for(var r=!C(e)&&S.keys(e),i=(r||e).length,a=0;i>a;a++){var o=r?r[a]:a;if(t(e[o],o,e))return!0}return!1},S.contains=S.includes=S.include=function(e,t,n,r){return C(e)||(e=S.values(e)),("number"!=typeof n||r)&&(n=0),S.indexOf(e,t,n)>=0},S.invoke=function(e,t){var n=h.call(arguments,2),r=S.isFunction(t);return S.map(e,function(e){var i=r?t:e[t];return null==i?i:i.apply(e,n)})},S.pluck=function(e,t){return S.map(e,S.property(t))},S.where=function(e,t){return S.filter(e,S.matcher(t))},S.findWhere=function(e,t){return S.find(e,S.matcher(t))},S.max=function(e,t,n){var r,i,a=-(1/0),o=-(1/0);if(null==t&&null!=e){e=C(e)?e:S.values(e);for(var s=0,u=e.length;u>s;s++)r=e[s],r>a&&(a=r)}else t=_(t,n),S.each(e,function(e,n,r){i=t(e,n,r),(i>o||i===-(1/0)&&a===-(1/0))&&(a=e,o=i)});return a},S.min=function(e,t,n){var r,i,a=1/0,o=1/0;if(null==t&&null!=e){e=C(e)?e:S.values(e);for(var s=0,u=e.length;u>s;s++)r=e[s],a>r&&(a=r)}else t=_(t,n),S.each(e,function(e,n,r){i=t(e,n,r),(o>i||i===1/0&&a===1/0)&&(a=e,o=i)});return a},S.shuffle=function(e){for(var t,n=C(e)?e:S.values(e),r=n.length,i=Array(r),a=0;r>a;a++)t=S.random(0,a),t!==a&&(i[a]=i[t]),i[t]=n[a];return i},S.sample=function(e,t,n){return null==t||n?(C(e)||(e=S.values(e)),e[S.random(e.length-1)]):S.shuffle(e).slice(0,Math.max(0,t))},S.sortBy=function(e,t,n){return t=_(t,n),S.pluck(S.map(e,function(e,n,r){return{value:e,index:n,criteria:t(e,n,r)}}).sort(function(e,t){var n=e.criteria,r=t.criteria;if(n!==r){if(n>r||void 0===n)return 1;if(r>n||void 0===r)return-1}return e.index-t.index}),"value")};var L=function(e){return function(t,n,r){var i={};return n=_(n,r),S.each(t,function(r,a){var o=n(r,a,t);e(i,r,o)}),i}};S.groupBy=L(function(e,t,n){S.has(e,n)?e[n].push(t):e[n]=[t]}),S.indexBy=L(function(e,t,n){e[n]=t}),S.countBy=L(function(e,t,n){S.has(e,n)?e[n]++:e[n]=1}),S.toArray=function(e){return e?S.isArray(e)?h.call(e):C(e)?S.map(e,S.identity):S.values(e):[]},S.size=function(e){return null==e?0:C(e)?e.length:S.keys(e).length},S.partition=function(e,t,n){t=_(t,n);var r=[],i=[];return S.each(e,function(e,n,a){(t(e,n,a)?r:i).push(e)}),[r,i]},S.first=S.head=S.take=function(e,t,n){return null==e?void 0:null==t||n?e[0]:S.initial(e,e.length-t)},S.initial=function(e,t,n){return h.call(e,0,Math.max(0,e.length-(null==t||n?1:t)))},S.last=function(e,t,n){return null==e?void 0:null==t||n?e[e.length-1]:S.rest(e,Math.max(0,e.length-t))},S.rest=S.tail=S.drop=function(e,t,n){return h.call(e,null==t||n?1:t)},S.compact=function(e){return S.filter(e,S.identity)};var P=function(e,t,n,r){for(var i=[],a=0,o=r||0,s=M(e);s>o;o++){var u=e[o];if(C(u)&&(S.isArray(u)||S.isArguments(u))){t||(u=P(u,t,n));var l=0,p=u.length;for(i.length+=p;p>l;)i[a++]=u[l++]}else n||(i[a++]=u)}return i};S.flatten=function(e,t){return P(e,t,!1)},S.without=function(e){return S.difference(e,h.call(arguments,1))},S.uniq=S.unique=function(e,t,n,r){S.isBoolean(t)||(r=n,n=t,t=!1),null!=n&&(n=_(n,r));for(var i=[],a=[],o=0,s=M(e);s>o;o++){var u=e[o],l=n?n(u,o,e):u;t?(o&&a===l||i.push(u),a=l):n?S.contains(a,l)||(a.push(l),i.push(u)):S.contains(i,u)||i.push(u)}return i},S.union=function(){return S.uniq(P(arguments,!0,!0))},S.intersection=function(e){for(var t=[],n=arguments.length,r=0,i=M(e);i>r;r++){var a=e[r];if(!S.contains(t,a)){for(var o=1;n>o&&S.contains(arguments[o],a);o++);o===n&&t.push(a)}}return t},S.difference=function(e){var t=P(arguments,!0,!0,1);return S.filter(e,function(e){return!S.contains(t,e)})},S.zip=function(){return S.unzip(arguments)},S.unzip=function(e){for(var t=e&&S.max(e,M).length||0,n=Array(t),r=0;t>r;r++)n[r]=S.pluck(e,r);return n},S.object=function(e,t){for(var n={},r=0,i=M(e);i>r;r++)t?n[e[r]]=t[r]:n[e[r][0]]=e[r][1];return n},S.findIndex=a(1),S.findLastIndex=a(-1),S.sortedIndex=function(e,t,n,r){n=_(n,r,1);for(var i=n(t),a=0,o=M(e);o>a;){var s=Math.floor((a+o)/2);n(e[s])<i?a=s+1:o=s}return a},S.indexOf=o(1,S.findIndex,S.sortedIndex),S.lastIndexOf=o(-1,S.findLastIndex),S.range=function(e,t,n){null==t&&(t=e||0,e=0),n=n||1;for(var r=Math.max(Math.ceil((t-e)/n),0),i=Array(r),a=0;r>a;a++,e+=n)i[a]=e;return i};var O=function(e,t,n,r,i){if(!(r instanceof t))return e.apply(n,i);var a=w(e.prototype),o=e.apply(a,i);return S.isObject(o)?o:a};S.bind=function(e,t){if(A&&e.bind===A)return A.apply(e,h.call(arguments,1));if(!S.isFunction(e))throw new TypeError("Bind must be called on a function");var n=h.call(arguments,2),r=function(){return O(e,r,t,this,n.concat(h.call(arguments)))};return r},S.partial=function(e){var t=h.call(arguments,1),n=function(){for(var r=0,i=t.length,a=Array(i),o=0;i>o;o++)a[o]=t[o]===S?arguments[r++]:t[o];for(;r<arguments.length;)a.push(arguments[r++]);return O(e,n,this,this,a)};return n},S.bindAll=function(e){var t,n,r=arguments.length;if(1>=r)throw new Error("bindAll must be passed function names");for(t=1;r>t;t++)n=arguments[t],e[n]=S.bind(e[n],e);return e},S.memoize=function(e,t){var n=function(r){var i=n.cache,a=""+(t?t.apply(this,arguments):r);return S.has(i,a)||(i[a]=e.apply(this,arguments)),i[a]};return n.cache={},n},S.delay=function(e,t){var n=h.call(arguments,2);return setTimeout(function(){return e.apply(null,n)},t)},S.defer=S.partial(S.delay,S,1),S.throttle=function(e,t,n){var r,i,a,o=null,s=0;n||(n={});var u=function(){s=n.leading===!1?0:S.now(),o=null,a=e.apply(r,i),o||(r=i=null)};return function(){var l=S.now();s||n.leading!==!1||(s=l);var p=t-(l-s);return r=this,i=arguments,0>=p||p>t?(o&&(clearTimeout(o),o=null),s=l,a=e.apply(r,i),o||(r=i=null)):o||n.trailing===!1||(o=setTimeout(u,p)),a}},S.debounce=function(e,t,n){var r,i,a,o,s,u=function(){var l=S.now()-o;t>l&&l>=0?r=setTimeout(u,t-l):(r=null,n||(s=e.apply(a,i),r||(a=i=null)))};return function(){a=this,i=arguments,o=S.now();var l=n&&!r;return r||(r=setTimeout(u,t)),l&&(s=e.apply(a,i),a=i=null),s}},S.wrap=function(e,t){return S.partial(t,e)},S.negate=function(e){return function(){return!e.apply(this,arguments)}},S.compose=function(){var e=arguments,t=e.length-1;return function(){for(var n=t,r=e[t].apply(this,arguments);n--;)r=e[n].call(this,r);return r}},S.after=function(e,t){return function(){return--e<1?t.apply(this,arguments):void 0}},S.before=function(e,t){var n;return function(){return--e>0&&(n=t.apply(this,arguments)),1>=e&&(t=null),n}},S.once=S.partial(S.before,2);var D=!{toString:null}.propertyIsEnumerable("toString"),x=["valueOf","isPrototypeOf","toString","propertyIsEnumerable","hasOwnProperty","toLocaleString"];S.keys=function(e){if(!S.isObject(e))return[];if(g)return g(e);var t=[];for(var n in e)S.has(e,n)&&t.push(n);return D&&s(e,t),t},S.allKeys=function(e){if(!S.isObject(e))return[];var t=[];for(var n in e)t.push(n);return D&&s(e,t),t},S.values=function(e){for(var t=S.keys(e),n=t.length,r=Array(n),i=0;n>i;i++)r[i]=e[t[i]];return r},S.mapObject=function(e,t,n){t=_(t,n);for(var r,i=S.keys(e),a=i.length,o={},s=0;a>s;s++)r=i[s],o[r]=t(e[r],r,e);return o},S.pairs=function(e){for(var t=S.keys(e),n=t.length,r=Array(n),i=0;n>i;i++)r[i]=[t[i],e[t[i]]];return r},S.invert=function(e){for(var t={},n=S.keys(e),r=0,i=n.length;i>r;r++)t[e[n[r]]]=n[r];return t},S.functions=S.methods=function(e){var t=[];for(var n in e)S.isFunction(e[n])&&t.push(n);return t.sort()},S.extend=N(S.allKeys),S.extendOwn=S.assign=N(S.keys),S.findKey=function(e,t,n){t=_(t,n);for(var r,i=S.keys(e),a=0,o=i.length;o>a;a++)if(r=i[a],t(e[r],r,e))return r},S.pick=function(e,t,n){var r,i,a={},o=e;if(null==o)return a;S.isFunction(t)?(i=S.allKeys(o),r=b(t,n)):(i=P(arguments,!1,!1,1),r=function(e,t,n){return t in n},o=Object(o));for(var s=0,u=i.length;u>s;s++){var l=i[s],p=o[l];r(p,l,o)&&(a[l]=p)}return a},S.omit=function(e,t,n){if(S.isFunction(t))t=S.negate(t);else{var r=S.map(P(arguments,!1,!1,1),String);t=function(e,t){return!S.contains(r,t)}}return S.pick(e,t,n)},S.defaults=N(S.allKeys,!0),S.create=function(e,t){var n=w(e);return t&&S.extendOwn(n,t),n},S.clone=function(e){return S.isObject(e)?S.isArray(e)?e.slice():S.extend({},e):e},S.tap=function(e,t){return t(e),e},S.isMatch=function(e,t){var n=S.keys(t),r=n.length;if(null==e)return!r;for(var i=Object(e),a=0;r>a;a++){var o=n[a];if(t[o]!==i[o]||!(o in i))return!1}return!0};var U=function(e,t,n,r){if(e===t)return 0!==e||1/e===1/t;if(null==e||null==t)return e===t;e instanceof S&&(e=e._wrapped),t instanceof S&&(t=t._wrapped);var i=m.call(e);if(i!==m.call(t))return!1;switch(i){case"[object RegExp]":case"[object String]":return""+e==""+t;case"[object Number]":return+e!==+e?+t!==+t:0===+e?1/+e===1/t:+e===+t;case"[object Date]":case"[object Boolean]":return+e===+t}var a="[object Array]"===i;if(!a){if("object"!=typeof e||"object"!=typeof t)return!1;var o=e.constructor,s=t.constructor;if(o!==s&&!(S.isFunction(o)&&o instanceof o&&S.isFunction(s)&&s instanceof s)&&"constructor"in e&&"constructor"in t)return!1}n=n||[],r=r||[];for(var u=n.length;u--;)if(n[u]===e)return r[u]===t;if(n.push(e),r.push(t),a){if(u=e.length,u!==t.length)return!1;for(;u--;)if(!U(e[u],t[u],n,r))return!1}else{var l,p=S.keys(e);if(u=p.length,S.keys(t).length!==u)return!1;for(;u--;)if(l=p[u],!S.has(t,l)||!U(e[l],t[l],n,r))return!1}return n.pop(),r.pop(),!0};S.isEqual=function(e,t){return U(e,t)},S.isEmpty=function(e){return null==e?!0:C(e)&&(S.isArray(e)||S.isString(e)||S.isArguments(e))?0===e.length:0===S.keys(e).length},S.isElement=function(e){return!(!e||1!==e.nodeType)},S.isArray=v||function(e){return"[object Array]"===m.call(e)},S.isObject=function(e){var t=typeof e;return"function"===t||"object"===t&&!!e},S.each(["Arguments","Function","String","Number","Date","RegExp","Error"],function(e){S["is"+e]=function(t){return m.call(t)==="[object "+e+"]"}}),S.isArguments(arguments)||(S.isArguments=function(e){return S.has(e,"callee")}),"function"!=typeof/./&&"object"!=typeof Int8Array&&(S.isFunction=function(e){return"function"==typeof e||!1}),S.isFinite=function(e){return isFinite(e)&&!isNaN(parseFloat(e))},S.isNaN=function(e){return S.isNumber(e)&&e!==+e},S.isBoolean=function(e){return e===!0||e===!1||"[object Boolean]"===m.call(e)},S.isNull=function(e){return null===e},S.isUndefined=function(e){return void 0===e},S.has=function(e,t){return null!=e&&y.call(e,t)},S.noConflict=function(){return u._=l,this},S.identity=function(e){return e},S.constant=function(e){return function(){return e}},S.noop=function(){},S.property=R,S.propertyOf=function(e){return null==e?function(){}:function(t){return e[t]}},S.matcher=S.matches=function(e){return e=S.extendOwn({},e),function(t){return S.isMatch(t,e)}},S.times=function(e,t,n){var r=Array(Math.max(0,e));t=b(t,n,1);for(var i=0;e>i;i++)r[i]=t(i);return r},S.random=function(e,t){return null==t&&(t=e,e=0),e+Math.floor(Math.random()*(t-e+1))},S.now=Date.now||function(){return(new Date).getTime()};var k={"&":"&amp;","<":"&lt;",">":"&gt;",'"':"&quot;","'":"&#x27;","`":"&#x60;"},F=S.invert(k),B=function(e){var t=function(t){return e[t]},n="(?:"+S.keys(e).join("|")+")",r=RegExp(n),i=RegExp(n,"g");return function(e){return e=null==e?"":""+e,r.test(e)?e.replace(i,t):e}};S.escape=B(k),S.unescape=B(F),S.result=function(e,t,n){var r=null==e?void 0:e[t];return void 0===r&&(r=n),S.isFunction(r)?r.call(e):r};var K=0;S.uniqueId=function(e){var t=++K+"";return e?e+t:t},S.templateSettings={evaluate:/<%([\s\S]+?)%>/g,interpolate:/<%=([\s\S]+?)%>/g,escape:/<%-([\s\S]+?)%>/g};var V=/(.)^/,j={"'":"'","\\":"\\","\r":"r","\n":"n","\u2028":"u2028","\u2029":"u2029"},W=/\\|'|\r|\n|\u2028|\u2029/g,q=function(e){return"\\"+j[e]};S.template=function(e,t,n){!t&&n&&(t=n),t=S.defaults({},t,S.templateSettings);var r=RegExp([(t.escape||V).source,(t.interpolate||V).source,(t.evaluate||V).source].join("|")+"|$","g"),i=0,a="__p+='";e.replace(r,function(t,n,r,o,s){return a+=e.slice(i,s).replace(W,q),i=s+t.length,n?a+="'+\n((__t=("+n+"))==null?'':_.escape(__t))+\n'":r?a+="'+\n((__t=("+r+"))==null?'':__t)+\n'":o&&(a+="';\n"+o+"\n__p+='"),t}),a+="';\n",t.variable||(a="with(obj||{}){\n"+a+"}\n"),a="var __t,__p='',__j=Array.prototype.join,print=function(){__p+=__j.call(arguments,'');};\n"+a+"return __p;\n";try{var o=new Function(t.variable||"obj","_",a)}catch(s){throw s.source=a,s}var u=function(e){return o.call(this,e,S)},l=t.variable||"obj";return u.source="function("+l+"){\n"+a+"}",u},S.chain=function(e){var t=S(e);return t._chain=!0,t};var Y=function(e,t){return e._chain?S(t).chain():t};S.mixin=function(e){S.each(S.functions(e),function(t){var n=S[t]=e[t];S.prototype[t]=function(){var e=[this._wrapped];return d.apply(e,arguments),Y(this,n.apply(S,e))}})},S.mixin(S),S.each(["pop","push","reverse","shift","sort","splice","unshift"],function(e){var t=p[e];S.prototype[e]=function(){var n=this._wrapped;return t.apply(n,arguments),"shift"!==e&&"splice"!==e||0!==n.length||delete n[0],Y(this,n)}}),S.each(["concat","join","slice"],function(e){var t=p[e];S.prototype[e]=function(){return Y(this,t.apply(this._wrapped,arguments))}}),S.prototype.value=function(){return this._wrapped},S.prototype.valueOf=S.prototype.toJSON=S.prototype.value,S.prototype.toString=function(){return""+this._wrapped},r=[],i=function(){return S}.apply(t,r),!(void 0!==i&&(e.exports=i))}).call(this)},function(e,t,n){(function(t){"use strict";var n=function(e,n,r,i,a,o,s,u){if("production"!==t.env.NODE_ENV&&void 0===n)throw new Error("invariant requires an error message argument");if(!e){var l;if(void 0===n)l=new Error("Minified exception occurred; use the non-minified dev environment for the full error message and additional helpful warnings.");else{var p=[r,i,a,o,s,u],c=0;l=new Error(n.replace(/%s/g,function(){return p[c++]})),l.name="Invariant Violation"}throw l.framesToPop=1,l}};e.exports=n}).call(t,n(64))},function(e,t,n){"use strict";function r(e,t,n,s){var u=new i(function(i,u){function l(a){r(e,t,{qs:n.qs,headers:n.headers,timeout:n.timeout}).nodeify(function(e,t){var r=e||t.statusCode>=400;if("function"==typeof n.retry&&(r=n.retry(e,t,a+1)),a>=(5|n.maxRetries)&&(r=!1),r){var o=n.retryDelay;"function"==typeof n.retryDelay&&(o=n.retryDelay(e,t,a+1)),o=o||200,setTimeout(function(){l(a+1)},o)}else e?u(e):i(t)})}var p=new window.XMLHttpRequest;if("string"!=typeof e)throw new TypeError("The method must be a string.");if("string"!=typeof t)throw new TypeError("The URL/path must be a string.");if("function"==typeof n&&(s=n,n={}),(null===n||void 0===n)&&(n={}),"object"!=typeof n)throw new TypeError("Options must be an object (or null).");if("function"!=typeof s&&(s=void 0),e=e.toUpperCase(),n.headers=n.headers||{},n.retry&&"GET"===e)return l(0);var c,f=!(!(c=/^([\w-]+:)?\/\/([^\/]+)/.exec(t))||c[2]==window.location.host);if(f||(n.headers["X-Requested-With"]="XMLHttpRequest"),n.qs&&(t=o(t,n.qs)),n.json&&(n.body=JSON.stringify(n.json),n.headers["Content-Type"]="application/json"),n.timeout){p.timeout=n.timeout;var d=Date.now();p.ontimeout=function(){var e=Date.now()-d,t=new Error("Request timed out after "+e+"ms");t.timeout=!0,t.duration=e,u(t)}}p.onreadystatechange=function(){if(4===p.readyState){var e={};p.getAllResponseHeaders().split("\r\n").forEach(function(t){var n=t.split(":");n.length>1&&(e[n[0].toLowerCase()]=n.slice(1).join(":").trim())});var n=new a(p.status,e,p.responseText);n.url=t,i(n)}},p.open(e,t,!0);for(var h in n.headers)p.setRequestHeader(h,n.headers[h]);p.send(n.body?n.body:null)});return u.getBody=function(){return u.then(function(e){return e.getBody()})},u.nodeify(s)}var i=n(113),a=n(65),o=n(92);e.exports=r},function(e,t,n){"use strict";function r(e,t){return void 0===t&&(t=!0),new v({rootNodeDetails:!0,serializeMetadata:t}).dump(e)}var i=this&&this.__extends||function(e,t){function n(){this.constructor=e}for(var r in t)t.hasOwnProperty(r)&&(e[r]=t[r]);e.prototype=null===t?Object.create(t):(n.prototype=t.prototype,new n)},a=n(18),o=n(51),s=n(44),u=n(39),l=n(16),p=n(29),c=n(14),f=n(18),d=n(33),h=n(70),m=n(11),y=n(15);t.dump=r;var v=function(){function e(e){this.options=e,this.nodeTransformers=[new V,new k,new Y,new b,new L,new P,new W,new I,new M,new F,new B,new O,new q,new j],this.nodePropertyTransformers=[new S,new _,new N,new R,new w,new C,new K],this.ignore=new A([new g(c.isResponseType,c.isDisplayNameProperty),new g(c.isApiSibling,c.isDisplayNameProperty),new g(c.isAnnotationRefTypeOrDescendant,c.isAnnotationProperty),new g(c.isSecuritySchemeRefType,c.isSecuritySchemeProperty),new g(c.isTraitRefType,c.isTraitProperty),new g(c.isResourceTypeRefType,c.isResourceTypeProperty),new g(c.isApiSibling,c.isRAMLVersionProperty),new g(c.isTypeDeclarationDescendant,c.isStructuredItemsProperty)]),this.missingProperties=new H,this.options=this.options||{},null==this.options.serializeMetadata&&(this.options.serializeMetadata=!0)}return e.prototype.printMissingProperties=function(){return this.missingProperties.print()},e.prototype.dump=function(e){var t=e.highLevel(),n=t&&t.parent(),r=!n&&this.options.rootNodeDetails,i=this.dumpInternal(e,r);return i},e.prototype.dumpInternal=function(e,t,n){var r=this;if(void 0===t&&(t=!1),null==e)return null;if(o.BasicNodeImpl.isInstance(e)){var i={},s=e,u=s.highLevel().definition();u.allProperties().filter(function(e){return!r.ignore.match(s.definition(),e)}).forEach(function(e){i[e.nameId()]=e}),u.allCustomProperties().filter(function(e){return!r.ignore.match(s.definition(),e)}).forEach(function(e){i[e.nameId()]=e});var p=this.dumpProperties(i,e);if(i.schema&&this.options.dumpSchemaContents&&i.schema.range().key()==f.Universe08.SchemaString){var d=s.highLevel().root().elementsOfKind("schemas");d.forEach(function(e){if(e.name()==p.schema){var t=e.attr("value");t&&(p.schema=t.value(),p.schemaContent=t.value())}})}if(this.serializeScalarsAnnotations(p,s,i),this.serializeMeta(p,s),this.canBeFragment(e)&&m.isFragment(e)){var h=m.asFragment(e),y=h.uses();y.length>0&&(p.uses=y.map(function(e){return e.toJSON()}))}if(c.isTypeDeclarationDescendant(u)){var v=s.highLevel().property();if(v&&!(c.isHeadersProperty(v)||c.isQueryParametersProperty(v)||c.isUriParametersProperty(v)||c.isPropertiesProperty(v)||c.isBaseUriParametersProperty(v))){delete p.required;var g=p.__METADATA__;if(g){var A=g.primitiveValuesMeta;A&&delete A.required}}}this.nodeTransformers.forEach(function(t){t.match(e,n||e.highLevel().property())&&(p=t.transform(p,e))});var E={};if(t){if(u){var T=u.universe().version();E.ramlVersion=T,E.type=u.nameId()}E.specification=p,E.errors=this.dumpErrors(s.errors())}else E=p;return E}if(o.AttributeNodeImpl.isInstance(e)){var i={},S=e,u=S.highLevel().definition(),b=S.highLevel().property().range();b.isArray()&&(b=b.array().componentType()),u.allCustomProperties().filter(function(e){return!r.ignore.match(b,e)}).forEach(function(e){i[e.nameId()]=e});var _=b.isValueType();if(_&&S.value){var N=S.value();if(null==N||"number"==typeof N||"string"==typeof N||"boolean"==typeof N)return N;if(b.isAssignableFrom(a.Universe08.StringType.name)&&l.StructuredValue.isInstance(N)){var w=N,R=w.lowLevel();return N=R?R.dumpToObject():null}}var p=this.dumpProperties(i,e);return this.nodeTransformers.forEach(function(t){t.match(e,e.highLevel().property())&&(p=t.transform(p,e))}),this.serializeScalarsAnnotations(p,e,i),this.serializeMeta(p,S),p}return o.TypeInstanceImpl.isInstance(e)?this.serializeTypeInstance(e):o.TypeInstancePropertyImpl.isInstance(e)?this.serializeTypeInstanceProperty(e):e},e.prototype.canBeFragment=function(e){var t=e.definition(),n=[t].concat(t.allSubTypes()),r=n.filter(function(e){return e.getAdapter(u.RAMLService).possibleInterfaces().filter(function(e){return e.nameId()==u.universesInfo.Universe10.FragmentDeclaration.name}).length>0});return r.length>0},e.prototype.dumpErrors=function(e){var t=this;return e.map(function(e){var n=t.dumpErrorBasic(e);return e.trace&&e.trace.length>0&&(n.trace=t.dumpErrors(e.trace)),n}).sort(function(e,t){return e.path!=t.path?e.path.localeCompare(t.path):e.range.start.position!=t.range.start.position?e.range.start.position-t.range.start.position:e.code-t.code})},e.prototype.dumpErrorBasic=function(e){var t={code:e.code,message:e.message,path:e.path,line:e.line,column:e.column,position:e.start,range:e.range};return e.isWarning===!0&&(t.isWarning=!0),t},e.prototype.stringLooksLikeXML=function(e){return"<"===e[0]&&">"===e[e.length-1]},e.prototype.stringLooksLikeJSON=function(e){return"{"===e[0]&&"}"===e[e.length-1]},e.prototype.dumpProperties=function(e,t){var n,r=this,i={};return t.highLevel().isElement()&&(n=t.highLevel().definition()),Object.keys(e).forEach(function(s){var u;if(n&&(u=n.property(s),u||(u=h.find(n.allCustomProperties(),function(e){return e.nameId()==s}))),!t[s])return void r.missingProperties.addProperty(e[s],t.kind());var f=e[s],d=t[s]();if(d&&"structuredType"==s&&"object"==typeof d){d=null;var m=t.highLevel(),v=m.lowLevel();v.children().forEach(function(e){if("type"==e.key()||"schema"==e.key()){var t=m.definition().universe().type(a.Universe10.TypeDeclaration.name),n=m.definition().universe().type(a.Universe10.LibraryBase.name),i=new l.ASTNodeImpl(e,m,t,n.property(a.Universe10.LibraryBase.properties.types.name));i.patchType(p.doDescrimination(i)),d=r.dumpInternal(i.wrapperNode(),!1,u),s=e.key()}})}else if("items"==s&&"object"==typeof d){var g=Array.isArray(d),A=!g;if(g&&(A=null!=h.find(d,function(e){return"object"==typeof e})),A){d=null;var m=t.highLevel(),v=m.lowLevel();v.children().forEach(function(e){if("items"==e.key()){var t=m.definition().universe().type(a.Universe10.TypeDeclaration.name),n=m.definition().universe().type(a.Universe10.LibraryBase.name),i=new l.ASTNodeImpl(e,m,t,n.property(a.Universe10.LibraryBase.properties.types.name));i.patchType(p.doDescrimination(i)),d=r.dumpInternal(i.wrapperNode(),!1,u),s=e.key()}})}}if(n&&n.isAssignableFrom("TypeDeclaration")&&("type"===s||"schema"==s)&&d)if(d.forEach&&"string"==typeof d[0]){var m=t.highLevel(),E=m.localType();if(E&&E.hasExternalInHierarchy()){var T=d[0].trim(),S="{"===T[0]&&"}"===T[T.length-1],b="<"===T[0]&&">"===T[T.length-1];S?i.typePropertyKind="JSON":b&&(i.typePropertyKind="XML")}else i.typePropertyKind="TYPE_EXPRESSION"}else"object"==typeof d&&(i.typePropertyKind="INPLACE");if(("type"===s||"schema"==s)&&d&&d.forEach&&"string"==typeof d[0]){var T=d[0].trim(),S=r.stringLooksLikeJSON(T),b=r.stringLooksLikeXML(T);if(S||b){var _=t.highLevel().lowLevel().includePath&&t.highLevel().lowLevel().includePath();if(!_&&t.highLevel().isElement()){var N=t.highLevel().asElement().attr("type");N||(N=t.highLevel().asElement().attr("schema")),N&&(_=N.lowLevel().includePath&&N.lowLevel().includePath())}if(_){var w=_.indexOf("#"),R="";w>=0&&(R=_.substring(w),_=_.substring(0,w));var I,M=t.highLevel().lowLevel().unit().resolve(_).absolutePath();I=0===M.indexOf("http://")||0===M.indexOf("https://")?M:y.relative(t.highLevel().lowLevel().unit().project().getRootPath(),M),I=I.replace(/\\/g,"/"),i.schemaPath=I+R}}}if((d||"type"!=s)&&(d||"schema"!=s)&&(d||"items"!=s)){if(t.definition&&c.isTypeDeclarationSibling(t.definition())&&c.isTypeProperty(f)){var C=t[a.Universe10.TypeDeclaration.properties.schema.name]();if(null!=C&&(!Array.isArray(C)||0!=C.length))return;var m=t.highLevel(),v=m.lowLevel(),L=!1;if(v.children().forEach(function(e){return"schema"==e.key()?void(L=!0):void 0}),L)return}if(Array.isArray(d)){for(var P=[],O=0,D=d;O<D.length;O++){var x=D[O],U=r.dumpInternal(x);"examples"===s&&r.options&&r.options.dumpXMLRepresentationOfExamples&&x.expandable&&x.expandable._owner&&(U.asXMLString=x.expandable.asXMLString()),P.push(U)}if(0==P.length&&o.BasicNodeImpl.isInstance(t)&&!r.isDefined(t,s))return;for(var k=0,F=r.nodePropertyTransformers;k<F.length;k++){var B=F[k];B.match(t,f)&&(P=B.transform(P,t))}i[s]=P}else{var x=r.dumpInternal(d);if(null==x&&o.BasicNodeImpl.isInstance(t)&&!r.isDefined(t,s))return;o.BasicNodeImpl.isInstance(t)&&r.nodePropertyTransformers.forEach(function(e){e.match(t,f)&&(x=e.transform(x,t))}),i[s]=x,"example"===s&&r.options&&r.options.dumpXMLRepresentationOfExamples&&d.expandable&&d.expandable._owner&&(x.asXMLString=d.expandable.asXMLString())}}}),i},e.prototype.serializeScalarsAnnotations=function(e,t,n){var r=this;if(t.scalarsAnnotations){for(var i={},a=t.scalarsAnnotations(),o=0,s=Object.keys(n);o<s.length;o++){var u=s[o];if(a[u]){var l=a[u]();if(l.length>0)if(Array.isArray(l[0])){var p=[];l.forEach(function(e,t){p.push(e.map(function(e){return r.dumpInternal(e)}))}),p.filter(function(e){return e.length>0}).length>0&&(i[u]=p)}else i[u]=l.map(function(e){return r.dumpInternal(e)})}}Object.keys(i).length>0&&(e.scalarsAnnotations=i)}},e.prototype.serializeMeta=function(e,t){if(this.options.serializeMetadata){var n=t.meta();n.isDefault()||(e.__METADATA__=n.toJSON())}},e.prototype.serializeTypeInstance=function(e){var t=this;if(e.isScalar())return e.value();if(e.isArray())return e.items().map(function(e){return t.serializeTypeInstance(e)});var n=e.properties();if(0==n.length)return null;var r={};return n.forEach(function(e){return r[e.name()]=t.serializeTypeInstanceProperty(e)}),r},e.prototype.serializeTypeInstanceProperty=function(e){var t=this;if(e.isArray()){var n=e.values(),r=[];return n.forEach(function(e){return r.push(t.serializeTypeInstance(e))}),r}return this.serializeTypeInstance(e.value())},e.prototype.isDefined=function(e,t){var n=e.highLevel();return n.elementsOfKind(t).length>0?!0:n.attributes(t).length>0?!0:!1},e}();t.TCKDumper=v;var g=function(){function e(e,t){this.typeMatcher=e,this.propMatcher=t}return e.prototype.match=function(e,t){return(null==e||this.typeMatcher(e))&&(null==t||this.propMatcher(t))},e}(),A=function(){function e(e){this.matchers=e}return e.prototype.match=function(e,t){for(var n=this.matchers.length,r=0;n>r;r++)if(this.matchers[r].match(e,t))return!0;return!1},e}(),E=function(){function e(e,t){this.matcher=e,this.propName=t}return e.prototype.match=function(e,t){return this.matcher.match(e.definition(),t)},e.prototype.transform=function(e){var t=this;if(Array.isArray(e)&&e.length>0&&e[0][this.propName]){var n={};return e.forEach(function(e){var r=e[t.propName],i=n[r];i?Array.isArray(i)?i.push(e):n[r]=[i,e]:n[r]=e}),n}return e},e}(),T=function(){function e(e,t){this.matcher=e,this.propName=t}return e.prototype.match=function(e,t){return this.matcher.match(e.definition?e.definition():null,t)},e.prototype.transform=function(e){if(Array.isArray(e))return e;var t={};return t[e[this.propName]]=e,t},e}(),S=function(e){function t(){e.call(this,new A([new g(c.isApiSibling,c.isBaseUriParametersProperty),new g(c.isResourceBaseSibling,c.isUriParametersProperty),new g(c.isResourceBaseSibling,c.isQueryParametersProperty),new g(c.isTraitType,c.isQueryParametersProperty),new g(c.isMethodType,c.isQueryParametersProperty),new g(c.isSecuritySchemePartType,c.isQueryParametersProperty),new g(c.isTraitType,c.isHeadersProperty),new g(c.isMethodType,c.isHeadersProperty),new g(c.isSecuritySchemePartType,c.isHeadersProperty),new g(c.isBodyLikeType,c.isFormParametersProperty)]),"name")}return i(t,e),t}(E),b=function(e){function t(){e.call(this,new A([new g(c.isLibraryBaseSibling,c.isTypesProperty),new g(function(e){return c.isLibraryBaseSibling(e)&&c.isRAML10Type(e)},c.isSchemasProperty)]),"name")}return i(t,e),t.prototype.match=function(e,t){var n=null!=e.parent()&&this.matcher.match(e.parent().definition(),t);return n?!0:!1},t}(T),_=(function(e){
function t(){e.call(this,new A([new g(c.isLibraryBaseSibling,c.isUsesProperty)]),"name")}return i(t,e),t}(E),function(e){function t(){e.call(this,new A([new g(c.isObjectTypeDeclarationSibling,c.isPropertiesProperty)]),"name")}return i(t,e),t}(E)),N=function(e){function t(){e.call(this,new A([new g(c.isMethodBaseSibling,c.isResponsesProperty)]),"code")}return i(t,e),t}(E),w=function(e){function t(){e.call(this,new A([new g(function(e){return!0},c.isAnnotationsProperty)]),"name")}return i(t,e),t}(E),R=function(e){function t(){e.call(this,new A([new g(c.isResponseType,c.isBodyProperty),new g(c.isMethodBaseSibling,c.isBodyProperty)]),"name")}return i(t,e),t}(E),I=function(e){function t(){e.call(this,new A([new g(c.isTraitType,c.isTraitsProperty)]),"name")}return i(t,e),t}(T),M=function(e){function t(){e.call(this,new A([new g(c.isResourceTypeType,c.isResourceTypesProperty)]),"name")}return i(t,e),t.prototype.transform=function(t){var n=f.Universe10.ResourceBase.properties.methods.name;if(Array.isArray(t))return t;var r=t[n];return r&&r.forEach(function(e){var n=Object.keys(e);if(n.length>0){var r=n[0];t[r]=e[r]}}),delete t[n],e.prototype.transform.call(this,t)},t}(T),C=function(e){function t(){e.call(this,new A([new g(c.isTypeDeclarationSibling,c.isFacetsProperty)]),"name")}return i(t,e),t}(E),L=function(e){function t(){e.call(this,null,"name")}return i(t,e),t.prototype.match=function(e,t){return null!=t&&c.isSecuritySchemesProperty(t)},t}(T),P=function(e){function t(){e.call(this,new A([new g(c.isLibraryBaseSibling,c.isAnnotationTypesProperty)]),"name")}return i(t,e),t.prototype.match=function(e,t){return null!=e.parent()&&this.matcher.match(e.parent().definition(),t)},t}(T),O=function(e){function t(){e.call(this,null,"method")}return i(t,e),t.prototype.match=function(e,t){return null!=e.parent()&&c.isResourceTypeType(e.parent().definition())&&c.isMethodsProperty(t)},t}(T),D=a.Universe10.ExampleSpec.properties.name.name,x=a.Universe10.ExampleSpec.properties.value.name,U="structuredContent",k=(function(){function e(){}return e.prototype.match=function(e,t){return c.isExampleSpecType(e.definition())},e.prototype.transform=function(e){var t=this;if(Array.isArray(e)&&e.length>0){if(e[0][D]){var n={};return e.forEach(function(e){return n[e[D]]=t.getActualExample(e)}),n}var r=e.map(function(e){return t.getActualExample(e)});return r}return e},e.prototype.getActualExample=function(e){return e[U]?e[U]:e[x]},e}(),function(){function e(){}return e.prototype.match=function(e,t){return e.definition&&c.isTypeDeclarationSibling(e.definition())},e.prototype.transform=function(e){var t=Array.isArray(e),n=t?e:[e];return n.forEach(function(e){var t=e.example;t&&(e.example=t.structuredValue,e.structuredExample=t)}),t?n:n[0]},e}()),F=function(){function e(){this.matcher=new g(function(e){return c.isApiType(e)&&c.isRAML08Type(e)},c.isSchemasProperty)}return e.prototype.match=function(e,t){return null!=e.parent()&&this.matcher.match(e.parent().definition(),t)},e.prototype.transform=function(e){if(Array.isArray(e))return e;var t={};return t[e.key]=e.value,t},e.prototype.getActualExample=function(e){return e[U]?e[U]:e[x]},e}(),B=function(){function e(){}return e.prototype.match=function(e,t){return null!=t&&c.isProtocolsProperty(t)},e.prototype.transform=function(e){return"string"==typeof e?e.toUpperCase():Array.isArray(e)?e.map(function(e){return e.toUpperCase()}):e},e}(),K=function(){function e(){this.usecases=new A([new g(c.isApiSibling,c.isMediaTypeProperty)])}return e.prototype.match=function(e,t){return this.usecases.match(e.definition(),t)},e.prototype.transform=function(e){return Array.isArray(e)&&1==e.length?e[0]:e},e}(),V=function(){function e(){}return e.prototype.match=function(e,t){return null!=t&&c.isResourcesProperty(t)},e.prototype.transform=function(e,t){if(Array.isArray(e))return e;var n=e[f.Universe10.Resource.properties.relativeUri.name];if(n){for(var r=n.trim().split("/");r.length>0&&0==r[0].length;)r.shift();e.relativeUriPathSegments=r,e.absoluteUri=t.absoluteUri()}return e},e}(),j=function(){function e(){}return e.prototype.match=function(e,t){return e.parent()&&e.parent().highLevel().lowLevel().libProcessed?c.isAnnotationTypesProperty(t)||c.isTypesProperty(t)||c.isResourceTypesProperty(t)||c.isTraitsProperty(t)||c.isSecuritySchemesProperty(t):!1},e.prototype.transform=function(e,t){for(var n=t.highLevel().lowLevel(),r=n.key(),i=n;s.LowLevelProxyNode.isInstance(i);)i=i.originalNode();var a=i.key(),o=e[Object.keys(e)[0]];return o.name=a,o.displayName==r&&(o.displayName=a),e},e}(),W=function(){function e(){}return e.prototype.match=function(e,t){var n=e.highLevel();if(!n)return!1;var r=n.definition();return r?c.isResourceTypeType(r)||c.isTraitType(r)||c.isMethodType(r)||c.isTypeDeclarationSibling(r):!1},e.prototype.transform=function(e){if(Array.isArray(e))return e;var t=a.Universe10.Trait.properties.parametrizedProperties.name,n=e[t];return n&&(Object.keys(n).forEach(function(t){e[t]=n[t]}),delete e[t]),e},e}(),q=function(){function e(){}return e.prototype.match=function(e,t){return null!=t&&(c.isSecuredByProperty(t)||c.isIsProperty(t)||null!=e.parent()&&(c.isResourceType(e.parent().highLevel().definition())||c.isResourceTypeType(e.parent().highLevel().definition()))&&c.isTypeProperty(t))},e.prototype.transform=function(e){return e?Array.isArray(e)?e:this.toSimpleValue(e):null},e.prototype.toSimpleValue=function(e){var t=e.name,n=e.structuredValue;if(n){var r={};return r[t]=n,r}return t},e}(),Y=function(){function e(){}return e.prototype.match=function(e,t){if(!o.BasicNodeImpl.isInstance(e))return!1;var n=e.highLevel(),r=n.definition();if(!c.isTypeDeclarationDescendant(r))return!1;var i=n.localType();return i&&i.isArray()?!0:!1},e.prototype.transform=function(e){var t=f.Universe10.TypeDeclaration.properties.type.name,n=f.Universe10.TypeDeclaration.properties.schema.name,r=f.Universe10.ArrayTypeDeclaration.properties.items.name,i=e[t];return i||(i=e[n]),1==i.length&&d.stringEndsWith(i[0],"[]")&&(null==e[r]&&(e[r]=i[0].substring(0,i[0].length-2)),i[0]="array"),e},e}(),H=function(){function e(){this.map={}}return e.prototype.addProperty=function(e,t){var n=this.map[t];n||(n=new $(t),this.map[t]=n),n.addProperty(e)},e.prototype.print=function(){var e=this;return Object.keys(this.map).map(function(t){return e.map[t].print()}).join("\n")+"\n"},e}(),$=function(){function e(e){this.typeName=e,this.map={}}return e.prototype.addProperty=function(e){var t=e.domain().nameId(),n=this.map[t];n||(n=new G(t),this.map[t]=n),n.addProperty(e)},e.prototype.print=function(){var e=this;return this.typeName+":\n"+Object.keys(this.map).map(function(t){return"    "+e.map[t].print()}).join("\n")},e}(),G=function(){function e(e){this.typeName=e,this.map={}}return e.prototype.addProperty=function(e){var t=e.nameId();this.map[t]=e},e.prototype.print=function(){return this.typeName+": "+Object.keys(this.map).sort().join(", ")},e}()},function(e,t,n){var r,i,a;!function(n,o){"use strict";"object"==typeof e&&"object"==typeof e.exports?e.exports=o():(i=[],r=o,a="function"==typeof r?r.apply(t,i):r,!(void 0!==a&&(e.exports=a)))}("object"==typeof window?window:this,function(){"use strict";function e(e){this.capacity=e>0?+e:Number.MAX_VALUE,this.data=Object.create?Object.create(null):{},this.hash=Object.create?Object.create(null):{},this.linkedList={length:0,head:null,end:null}}function t(n){return this instanceof t?void(this._LRUCacheState=new e(n)):new t(n)}function n(e,t){t!==e.head&&(e.end?e.end===t&&(e.end=t.n):e.end=t,r(t.n,t.p),r(t,e.head),e.head=t,e.head.n=null)}function r(e,t){e!==t&&(e&&(e.p=t),t&&(t.n=e))}var i=t.prototype;return i.get=function(e){var t=this._LRUCacheState,r=t.hash[e];return r?(n(t.linkedList,r),t.data[e]):void 0},i.set=function(e,t){var r=this._LRUCacheState,i=r.hash[e];return void 0===t?this:(i||(r.hash[e]={key:e},r.linkedList.length+=1,i=r.hash[e]),n(r.linkedList,i),r.data[e]=t,r.linkedList.length>r.capacity&&this.remove(r.linkedList.end.key),this)},i.update=function(e,t){this._LRUCacheState;if(this.has(e)){var n=this.get(e);this.set(e,t(n))}return this},i.remove=function(e){var t=this._LRUCacheState,n=t.hash[e];return n?(n===t.linkedList.head&&(t.linkedList.head=n.p),n===t.linkedList.end&&(t.linkedList.end=n.n),r(n.n,n.p),delete t.hash[e],delete t.data[e],t.linkedList.length-=1,this):this},i.removeAll=function(){return this._LRUCacheState=new e(this._LRUCacheState.capacity),this},i.info=function(){var e=this._LRUCacheState;return{capacity:e.capacity,length:e.linkedList.length}},i.keys=function(){for(var e=this._LRUCacheState,t=[],n=e.linkedList.head;n;)t.push(n.key),n=n.p;return t},i.has=function(e){return!!this._LRUCacheState.hash[e]},i.staleKey=function(){return this._LRUCacheState.linkedList.end&&this._LRUCacheState.linkedList.end.key},i.popStale=function(){var e=this.staleKey();if(!e)return null;var t=[e,this._LRUCacheState.data[e]];return this.remove(e),t},t.NAME="LRUCache",t.VERSION="v0.2.0",t})},function(e,t,n){function r(e){this.options=e||{locator:{}}}function i(e,t,n){function r(t){var r=e[t];!r&&o&&(r=2==e.length?function(n){e(t,n)}:e),i[t]=r&&function(e){r("[xmldom "+t+"]	"+e+s(n))}||function(){}}if(!e){if(t instanceof a)return t;e=t}var i={},o=e instanceof Function;return n=n||{},r("warning"),r("error"),r("fatalError"),i}function a(){this.cdata=!1}function o(e,t){t.lineNumber=e.lineNumber,t.columnNumber=e.columnNumber}function s(e){return e?"\n@"+(e.systemId||"")+"#[line:"+e.lineNumber+",col:"+e.columnNumber+"]":void 0}function u(e,t,n){return"string"==typeof e?e.substr(t,n):e.length>=t+n||t?new java.lang.String(e,t,n)+"":e}function l(e,t){e.currentElement?e.currentElement.appendChild(t):e.doc.appendChild(t)}r.prototype.parseFromString=function(e,t){var n=this.options,r=new p,o=n.domBuilder||new a,s=n.errorHandler,u=n.locator,l=n.xmlns||{},c={lt:"<",gt:">",amp:"&",quot:'"',apos:"'"};return u&&o.setDocumentLocator(u),r.errorHandler=i(s,o,u),r.domBuilder=n.domBuilder||o,/\/x?html?$/.test(t)&&(c.nbsp=" ",c.copy="©",l[""]="http://www.w3.org/1999/xhtml"),l.xml=l.xml||"http://www.w3.org/XML/1998/namespace",e?r.parse(e,l,c):r.errorHandler.error("invalid doc source"),o.doc},a.prototype={startDocument:function(){this.doc=(new c).createDocument(null,null,null),this.locator&&(this.doc.documentURI=this.locator.systemId)},startElement:function(e,t,n,r){var i=this.doc,a=i.createElementNS(e,n||t),s=r.length;l(this,a),this.currentElement=a,this.locator&&o(this.locator,a);for(var u=0;s>u;u++){var e=r.getURI(u),p=r.getValue(u),n=r.getQName(u),c=i.createAttributeNS(e,n);this.locator&&o(r.getLocator(u),c),c.value=c.nodeValue=p,a.setAttributeNode(c)}},endElement:function(e,t,n){var r=this.currentElement;r.tagName;this.currentElement=r.parentNode},startPrefixMapping:function(e,t){},endPrefixMapping:function(e){},processingInstruction:function(e,t){var n=this.doc.createProcessingInstruction(e,t);this.locator&&o(this.locator,n),l(this,n)},ignorableWhitespace:function(e,t,n){},characters:function(e,t,n){if(e=u.apply(this,arguments)){if(this.cdata)var r=this.doc.createCDATASection(e);else var r=this.doc.createTextNode(e);this.currentElement?this.currentElement.appendChild(r):/^\s*$/.test(e)&&this.doc.appendChild(r),this.locator&&o(this.locator,r)}},skippedEntity:function(e){},endDocument:function(){this.doc.normalize()},setDocumentLocator:function(e){(this.locator=e)&&(e.lineNumber=0)},comment:function(e,t,n){e=u.apply(this,arguments);var r=this.doc.createComment(e);this.locator&&o(this.locator,r),l(this,r)},startCDATA:function(){this.cdata=!0},endCDATA:function(){this.cdata=!1},startDTD:function(e,t,n){var r=this.doc.implementation;if(r&&r.createDocumentType){var i=r.createDocumentType(e,t,n);this.locator&&o(this.locator,i),l(this,i)}},warning:function(e){console.warn("[xmldom warning]	"+e,s(this.locator))},error:function(e){console.error("[xmldom error]	"+e,s(this.locator))},fatalError:function(e){throw console.error("[xmldom fatalError]	"+e,s(this.locator)),e}},"endDTD,startEntity,endEntity,attributeDecl,elementDecl,externalEntityDecl,internalEntityDecl,resolveEntity,getExternalSubset,notationDecl,unparsedEntityDecl".replace(/\w+/g,function(e){a.prototype[e]=function(){return null}});var p=n(88).XMLReader,c=t.DOMImplementation=n(89).DOMImplementation;t.XMLSerializer=n(89).XMLSerializer,t.DOMParser=r},function(e,t,n){!function(t,n){e.exports=n()}(this,function(){function e(e){return e.charAt(0).toUpperCase()+e.substr(1).toLowerCase()}function t(e){return"string"==typeof e?new RegExp("^"+e+"$","i"):e}function n(t,n){return t===t.toUpperCase()?n.toUpperCase():t[0]===t[0].toUpperCase()?e(n):n.toLowerCase()}function r(e,t){return e.replace(/\$(\d{1,2})/g,function(e,n){return t[n]||""})}function i(e,t,i){if(!e.length||l.hasOwnProperty(e))return t;for(var a=i.length;a--;){var o=i[a];if(o[0].test(t))return t.replace(o[0],function(e,t,i){var a=r(o[1],arguments);return""===e?n(i[t-1],a):n(e,a)})}return t}function a(e,t,r){return function(a){var o=a.toLowerCase();return t.hasOwnProperty(o)?n(a,o):e.hasOwnProperty(o)?n(a,e[o]):i(o,a,r)}}function o(e,t,n){var r=1===t?o.singular(e):o.plural(e);return(n?t+" ":"")+r}var s=[],u=[],l={},p={},c={};return o.plural=a(c,p,s),o.singular=a(p,c,u),o.addPluralRule=function(e,n){s.push([t(e),n])},o.addSingularRule=function(e,n){u.push([t(e),n])},o.addUncountableRule=function(e){return"string"==typeof e?void(l[e.toLowerCase()]=!0):(o.addPluralRule(e,"$0"),void o.addSingularRule(e,"$0"))},o.addIrregularRule=function(e,t){t=t.toLowerCase(),e=e.toLowerCase(),c[e]=t,p[t]=e},[["I","we"],["me","us"],["he","they"],["she","they"],["them","them"],["myself","ourselves"],["yourself","yourselves"],["itself","themselves"],["herself","themselves"],["himself","themselves"],["themself","themselves"],["is","are"],["this","these"],["that","those"],["echo","echoes"],["dingo","dingoes"],["volcano","volcanoes"],["tornado","tornadoes"],["torpedo","torpedoes"],["genus","genera"],["viscus","viscera"],["stigma","stigmata"],["stoma","stomata"],["dogma","dogmata"],["lemma","lemmata"],["schema","schemata"],["anathema","anathemata"],["ox","oxen"],["axe","axes"],["die","dice"],["yes","yeses"],["foot","feet"],["eave","eaves"],["goose","geese"],["tooth","teeth"],["quiz","quizzes"],["human","humans"],["proof","proofs"],["carve","carves"],["valve","valves"],["thief","thieves"],["genie","genies"],["groove","grooves"],["pickaxe","pickaxes"],["whiskey","whiskies"]].forEach(function(e){return o.addIrregularRule(e[0],e[1])}),[[/s?$/i,"s"],[/([^aeiou]ese)$/i,"$1"],[/(ax|test)is$/i,"$1es"],[/(alias|[^aou]us|tlas|gas|ris)$/i,"$1es"],[/(e[mn]u)s?$/i,"$1s"],[/([^l]ias|[aeiou]las|[emjzr]as|[iu]am)$/i,"$1"],[/(alumn|syllab|octop|vir|radi|nucle|fung|cact|stimul|termin|bacill|foc|uter|loc|strat)(?:us|i)$/i,"$1i"],[/(alumn|alg|vertebr)(?:a|ae)$/i,"$1ae"],[/(seraph|cherub)(?:im)?$/i,"$1im"],[/(her|at|gr)o$/i,"$1oes"],[/(agend|addend|millenni|dat|extrem|bacteri|desiderat|strat|candelabr|errat|ov|symposi|curricul|automat|quor)(?:a|um)$/i,"$1a"],[/(apheli|hyperbat|periheli|asyndet|noumen|phenomen|criteri|organ|prolegomen|hedr|automat)(?:a|on)$/i,"$1a"],[/sis$/i,"ses"],[/(?:(kni|wi|li)fe|(ar|l|ea|eo|oa|hoo)f)$/i,"$1$2ves"],[/([^aeiouy]|qu)y$/i,"$1ies"],[/([^ch][ieo][ln])ey$/i,"$1ies"],[/(x|ch|ss|sh|zz)$/i,"$1es"],[/(matr|cod|mur|sil|vert|ind|append)(?:ix|ex)$/i,"$1ices"],[/(m|l)(?:ice|ouse)$/i,"$1ice"],[/(pe)(?:rson|ople)$/i,"$1ople"],[/(child)(?:ren)?$/i,"$1ren"],[/eaux$/i,"$0"],[/m[ae]n$/i,"men"],["thou","you"]].forEach(function(e){return o.addPluralRule(e[0],e[1])}),[[/s$/i,""],[/(ss)$/i,"$1"],[/((a)naly|(b)a|(d)iagno|(p)arenthe|(p)rogno|(s)ynop|(t)he)(?:sis|ses)$/i,"$1sis"],[/(^analy)(?:sis|ses)$/i,"$1sis"],[/(wi|kni|(?:after|half|high|low|mid|non|night|[^\w]|^)li)ves$/i,"$1fe"],[/(ar|(?:wo|[ae])l|[eo][ao])ves$/i,"$1f"],[/([^aeiouy]|qu)ies$/i,"$1y"],[/(^[pl]|zomb|^(?:neck)?t|[aeo][lt]|cut)ies$/i,"$1ie"],[/(\b(?:mon|smil))ies$/i,"$1ey"],[/(m|l)ice$/i,"$1ouse"],[/(seraph|cherub)im$/i,"$1"],[/(x|ch|ss|sh|zz|tto|go|cho|alias|[^aou]us|tlas|gas|(?:her|at|gr)o|ris)(?:es)?$/i,"$1"],[/(e[mn]u)s?$/i,"$1"],[/(movie|twelve)s$/i,"$1"],[/(cris|test|diagnos)(?:is|es)$/i,"$1is"],[/(alumn|syllab|octop|vir|radi|nucle|fung|cact|stimul|termin|bacill|foc|uter|loc|strat)(?:us|i)$/i,"$1us"],[/(agend|addend|millenni|dat|extrem|bacteri|desiderat|strat|candelabr|errat|ov|symposi|curricul|quor)a$/i,"$1um"],[/(apheli|hyperbat|periheli|asyndet|noumen|phenomen|criteri|organ|prolegomen|hedr|automat)a$/i,"$1on"],[/(alumn|alg|vertebr)ae$/i,"$1a"],[/(cod|mur|sil|vert|ind)ices$/i,"$1ex"],[/(matr|append)ices$/i,"$1ix"],[/(pe)(rson|ople)$/i,"$1rson"],[/(child)ren$/i,"$1"],[/(eau)x?$/i,"$1"],[/men$/i,"man"]].forEach(function(e){return o.addSingularRule(e[0],e[1])}),["advice","agenda","bison","bream","buffalo","carp","chassis","cod","cooperation","corps","digestion","debris","diabetes","energy","equipment","elk","excretion","expertise","flounder","gallows","garbage","graffiti","headquarters","health","herpes","highjinks","homework","information","jeans","justice","kudos","labour","machinery","mackerel","media","mews","moose","news","pike","plankton","pliers","pollution","premises","rain","rice","salmon","scissors","series","sewage","shambles","shrimp","species","staff","swine","trout","tuna","whiting","wildebeest","wildlife","you",/pox$/i,/ois$/i,/deer$/i,/fish$/i,/sheep$/i,/measles$/i,/[^aeiou]ese$/i].forEach(o.addUncountableRule),o})},function(e,t,n){"use strict";function r(e,t){var n=t?t.endPosition:e.endPosition+1,r={key:e,value:t,startPosition:e.startPosition,endPosition:n,kind:f.MAPPING,parent:null,errors:[]};return r}function i(e,t,n,r){return{errors:[],referencesAnchor:e,value:r,startPosition:t,endPosition:n,kind:f.ANCHOR_REF,parent:null}}function a(e){return void 0===e&&(e=""),{errors:[],startPosition:-1,endPosition:-1,value:e,kind:f.SCALAR,parent:null,doubleQuoted:!1}}function o(){return{errors:[],startPosition:-1,endPosition:-1,items:[],kind:f.SEQ,parent:null}}function s(){return o()}function u(e){return{errors:[],startPosition:-1,endPosition:-1,mappings:e?e:[],kind:f.MAP,parent:null}}var l=n(90),p=n(91),c=function(){function e(e,t){void 0===t&&(t=null),this.name="YAMLException",this.reason=e,this.mark=t,this.message=this.toString(!1)}return e.isInstance=function(t){if(null!=t&&t.getClassIdentifier&&"function"==typeof t.getClassIdentifier)for(var n=0,r=t.getClassIdentifier();n<r.length;n++){var i=r[n];if(i==e.CLASS_IDENTIFIER)return!0}return!1},e.prototype.getClassIdentifier=function(){var t=[];return t.concat(e.CLASS_IDENTIFIER)},e.prototype.toString=function(e){void 0===e&&(e=!1);var t;return t="JS-YAML: "+(this.reason||"(unknown reason)"),!e&&this.mark&&(t+=" "+this.mark.toString()),t},e.CLASS_IDENTIFIER="yaml-ast-parser.YAMLException",e}();t.YAMLException=c,function(e){e[e.SCALAR=0]="SCALAR",e[e.MAPPING=1]="MAPPING",e[e.MAP=2]="MAP",e[e.SEQ=3]="SEQ",e[e.ANCHOR_REF=4]="ANCHOR_REF",e[e.INCLUDE_REF=5]="INCLUDE_REF"}(t.Kind||(t.Kind={}));var f=t.Kind;t.newMapping=r,t.newAnchorRef=i,t.newScalar=a,t.newItems=o,t.newSeq=s,t.newMap=u,t.load=l.load,t.loadAll=l.loadAll,t.safeLoad=l.safeLoad,t.dump=p.dump,t.safeDump=p.safeDump},function(e,t,n){t.dot=t.dotCase=n(123),t.swap=t.swapCase=n(125),t.path=t.pathCase=n(124),t.upper=t.upperCase=n(134),t.lower=t.lowerCase=n(126),t.camel=t.camelCase=n(127),t.snake=t.snakeCase=n(128),t.title=t.titleCase=n(129),t.param=t.paramCase=n(130),t.pascal=t.pascalCase=n(132),t.constant=t.constantCase=n(133),t.sentence=t.sentenceCase=n(131),t.isUpper=t.isUpperCase=n(135),t.isLower=t.isLowerCase=n(136),t.ucFirst=t.upperCaseFirst=n(138),t.lcFirst=t.lowerCaseFirst=n(137)},function(e,t,n){"use strict";function r(e){var t=new ie.Project(ae.dirname(e)),n=t.unit(ae.basename(e));return n&&n.isRAMLUnit()?X.fromUnit(n).wrapperNode():null}function i(e){var t=e.highLevel().lowLevel();if(Z.LowLevelProxyNode.isInstance(t))return e;var n=null!=e.highLevel().reusedNode()?Q:J;return n.expandTraitsAndResourceTypes(e)}function a(e){var t="",n=e;do e=n,t=e.relativeUri().value()+t,n=e.parent();while(n.definition().key().name==z.Universe08.Resource.name);return t}function o(e){var t="",n=e;do e=n,t=e.relativeUri().value()+t,n=e.parent();while(n.definition().key().name==z.Universe08.Resource.name);t=t.replace(/\/\//g,"/");var r=n.baseUri(),i=r?r.value():"";return i=i?i:"",te.stringEndsWith(i,"/")&&(t=t.substring(1)),t=i+t}function s(e){return X.qName(e.highLevel(),e.highLevel().root())}function u(e){return l(e)}function l(e){return f(e,function(e){return re.isTraitType(e)})}function p(e){return c(e)}function c(e){return f(e,function(e){return re.isResourceTypeType(e)})}function f(e,t){var n=ne.globalDeclarations(e.highLevel()).filter(function(e){return t(e.definition())}),r=e.highLevel().lowLevel();if(!r)return[];var i=r.includePath();i||(i=r.unit().path());var a=n.map(function(e){var t,n=e.lowLevel().unit().path();return n!=i?(t=H.buildWrapperNode(e,!1),t.meta().setCalculated()):t=e.wrapperNode(),t});return a}function d(e){var t=[],n=e;do e=n,t.push(e.relativeUri().value()),n=e.parent();while(n.definition().key().name==z.Universe08.Resource.name);return t.reverse()}function h(e){return Y.ResourceImpl.isInstance(e.parent())?e.parent():null}function m(e){var t=e.parent();return t.definition().key().name==z.Universe08.Resource.name,null}function y(e,t){if(null==e)return null;var n=e.resources();return n?(n=n.filter(function(e){return e.relativeUri().value()==t}),0==n.length?null:n[0]):null}function v(e,t){if(!e)return null;for(var n=null,r=0;r<t.length;r++){if(n=y(e,t[r]),!n)return null;e=n}return n}function g(e,t){return e?e.methods().filter(function(e){return e.method()==t}):null}function A(e,t,n){var r=v(e,t);return null==r?null:g(r,n)}function E(e){return e.definition().key().name==z.Universe08.Api.name}function T(e){for(var t=e;!E(t);)t=t.parent();return t}function S(e){var t=e.parent();if(Y.ResourceImpl.isInstance(t))return a(t)+" "+e.method().toLowerCase();if(Y.ResourceTypeImpl.isInstance(t))return t.name()+" "+e.method().toLowerCase();throw new Error("Method is supposed to be owned by Resource or ResourceType.\nHere the method is owned by "+e.definition().key().name)}function b(e){var t=e.code().value(),n=ee.validateResponseString(t);if(null!=n)return!1;try{if(parseInt(t.charAt(0))<4)return!0}catch(r){}return!1}function _(e){var t=[],n=function(e){t.push(e),e.resources().forEach(function(e){return n(e)})};return e.resources().forEach(function(e){return n(e)}),t}function N(e){return w(e)}function w(e){var t=e.relativeUri().value(),n=e.uriParameters_original(),r=z.Universe08.Resource.properties.uriParameters.name;return j(n,t,e,r)}function R(e){return I(e)}function I(e){var t=e.baseUri()?e.baseUri().value():"",n=e.baseUriParameters_original(),r=z.Universe08.Api.properties.baseUriParameters.name;return j(n,t,e,r)}function M(e){var t=[],n=e;do e=n,t=w(e).concat(t),n=e.parent();while(n.definition().key().name==z.Universe10.Resource.name);var r=n;r.baseUriParameters();return t=I(r).concat(t)}function C(e){return L(e)}function L(e){return e.protocols().map(function(e){return e.toUpperCase()})}function P(e){return O(e)}function O(e){return e.securedBy()}function D(e){var t=e.highLevel();if(!t)return"";var n=t.value();return n?n.toString():""}function x(e){var t=e.highLevel();if(!t)return null;var n=ne.findDeclarationByNode(t,ne.LocationKind.VALUE_COMPLETION);if(!n)return null;if(!n.getKind||n.getKind()!=G.NodeKind.NODE)return null;var r=n.wrapperNode();return Y.AbstractSecuritySchemeImpl.isInstance(r)?r:null}function U(e){return e.highLevel().definition().universe().version()}function k(e){var t=e.value().lowLevel();return new $.TypeInstanceImpl(t)}function F(e){var t=e.highLevel().value();return"string"==typeof t||null==t?t:t.valueName()}function B(e){return V(e)}function K(e){return V(e)}function V(e){var t=e.highLevel(),n=t.parent(),r=e.name(),i=ne.referenceTargets(t.property(),n).filter(function(e){return X.qName(e,n)==r});return 0==i.length?null:i[0].wrapperNode()}function j(e,t,n,r){var i=n.highLevel(),a=i.definition().property(r);if(!t)return[];var o={};e.forEach(function(e){var t=o[e.name()];t||(t=[],o[e.name()]=t),t.push(e)});for(var s=[],u=0,l={},p=t.indexOf("{");p>=0&&(u=t.indexOf("}",++p),!(0>u));p=t.indexOf("{",u)){var c=t.substring(p,u);if(l[c]=!0,o[c])o[c].forEach(function(e){return s.push(e)});else{var f=new Y.StringTypeDeclarationImpl(c);f.setName(c);var d=f.highLevel();d.setParent(i),f.meta().setCalculated(),d.patchProp(a),s.push(f)}}return Object.keys(o).filter(function(e){return!l[e]}).forEach(function(e){return o[e].forEach(function(e){return s.push(e)})}),s}function W(e){var t=e.schema();if(null==t)return null;var n=t.value();if(!n)return null;if(te.stringStartsWith(n,"{")||te.stringStartsWith(n,"[")||te.stringStartsWith(n,"<"))return n;var r=e.highLevel(),i=r.root(),a=i.elementsOfKind(z.Universe08.Api.properties.schemas.name),o=oe.find(a,function(e){return e.name()==n});if(!o)return n;if(!o.getKind||o.getKind()!=G.NodeKind.NODE)return n;if(o.definition().key()!=z.Universe08.GlobalSchema)return n;var s=o.attr(z.Universe08.GlobalSchema.properties.value.name);return null==s?null:s.value()}function q(e){var t=e.highLevel();if(null==t)return null;var n=t.lowLevel();if(null==n)return null;var r=n.children().filter(function(e){return e.key().indexOf("<<")>=0});if(0==r.length)return null;var i=new $.TypeInstanceImpl(r);return i}var Y=n(52),H=n(48),$=n(51),G=n(9),X=n(16),z=n(18),J=n(27),Q=n(30),Z=n(44),ee=n(28),te=n(33),ne=n(13),re=n(14),ie=n(24),ae=n(15),oe=n(70);t.load=r,t.expandTraitsAndResourceTypes=i,t.completeRelativeUri=a,t.absoluteUri=o,t.qName=s,t.traitsPrimary=u,t.allTraits=l,t.resourceTypesPrimary=p,t.allResourceTypes=c,t.relativeUriSegments=d,t.parentResource=h,t.parent=m,t.childResource=y,t.getResource=v,t.childMethod=g,t.getMethod=A,t.ownerApi=T,t.methodId=S,t.isOkRange=b,t.allResources=_;t.uriParametersPrimary=N,t.uriParameters=w,t.baseUriParametersPrimary=R,t.baseUriParameters=I,t.absoluteUriParameters=M,t.protocolsPrimary=C,t.allProtocols=L,t.securedByPrimary=P,t.allSecuredBy=O,t.securitySchemeName=D,t.securityScheme=x,t.RAMLVersion=U,t.structuredValue=k,t.referenceName=F,t.referencedTrait=B,t.referencedResourceType=K,t.schemaContent=W,t.getTemplateParametrizedProperties=q},function(e,t,n){"use strict";function r(e){return null==e}function i(e){return r(e)?"":String(e)}function a(e){return-1===[0,!1,"","0","false"].indexOf(e)}function o(e){return isFinite(e)?Number(e):null}function s(e){return e%1===0?Number(e):null}function u(e){return isNaN(Date.parse(e))?null:new Date(e)}function l(e,t,n){var i=Array.isArray(e)?e:[e],a=i.map(function(e){function i(e,t,n){for(var r=0;r<a.length;r++){var i=a[r],e=i(e,t,n);if(null!=e)return e}return null}var a=[],o=n[e.type];return"function"==typeof o&&a.push(o),Object.keys(e).filter(function(e){return"type"!==e&&"repeat"!==e&&"default"!==e}).forEach(function(n){var r=t[n];"function"==typeof r&&a.push(r(e[n],n))}),function(t,n,a){if(r(t))return null!=e["default"]?i(e["default"],n,a):e.repeat&&!e.required?[]:t;if(e.repeat){var o=Array.isArray(t)?t:[t];return o=o.map(function(e){return i(e,n,a)}),o.some(r)?null:t}if(Array.isArray(t)){if(t.length>1)return null;t=t[0]}return i(t,n,a)}});return function(e,t,n){for(var r=0;r<a.length;r++){var i=a[r],o=i(e,t,n);if(null!=o)return o}return e}}function p(){function e(e){return l(e,n,r)}var t,n={},r={string:i,number:o,integer:s,"boolean":a,date:u},p=function(e){if(!e)return function(){return{}};var t={};return Object.keys(e).forEach(function(n){t[n]=p.rule(e[n])}),function(e){e=e||{};var n={};return Object.keys(t).forEach(function(r){var i=e[r],a=t[r];Object.prototype.hasOwnProperty.call(e,r)&&(n[r]=a(i,r,e))}),n}};return t=p,t.rule=e,t.TYPES=r,t.RULES=n,t}t.sanitize=p},function(e,t,n){(function(e){"use strict";function n(e){return"[object Date]"===v.call(e)&&!isNaN(e.getTime())}function r(e){return"boolean"==typeof e}function i(e){return"string"==typeof e}function a(e){return"number"==typeof e&&e%1===0}function o(e){return"number"==typeof e&&isFinite(e)}function s(e){return function(t){return t>=e}}function u(e){return function(t){return e>=t}}function l(t){return function(n){return e.byteLength(n)>=t}}function p(t){return function(n){return e.byteLength(n)<=t}}function c(e){return e&&0!=e.length?function(t){return e.indexOf(t)>-1}:function(e){return!0}}function f(e){var t="string"==typeof e?new RegExp(e):e;return t.test.bind(t)}function d(e,t,n,r,i){return{valid:e,rule:r,attr:i,value:n,key:t}}function h(e,t){var n=[];return Object.keys(e).forEach(function(r){var i=t[r];if(i){var a=e[r];n.push([r,i(a,r),a])}}),function(e,t,r){for(var i=0;i<n.length;i++){var a=n[i],o=a[0],s=a[1],u=a[2],l=s(e,t,r);if(!l)return d(!1,t,e,o,u)}return d(!0,t,e)}}function m(e,t,n){var r=Array.isArray(e)?e:[e],i=!r.length,a=[],o=[];return r.forEach(function(e){var n=[e.type||"string",h(e,t)];e.required||(i=!0),e.repeat?o.push(n):a.push(n)}),function(e,t,r){if(null==e)return d(i,t,e,"required",!i);var s=Array.isArray(e),u=s?e:[e],l=s?o:a;if(!l.length)return d(!1,t,e,"repeat",!s);var p=null,c=e;return l.some(function(e){var i=u.every(function(i){var a=e[0],o=n[a]&&n[a](i,t,r);return o||(p=d(!1,t,c,"type",a)),o});return i?(u.every(function(n){var r=e[1];return p=r(n,t),p.valid}),!0):!1}),p}}function y(){function e(e){return m(e,h,d)}var t,d={date:n,number:o,integer:a,"boolean":r,string:i},h={minimum:s,maximum:u,minLength:l,maxLength:p,"enum":c,pattern:f},y=function(t){if(!t)return function(e){return{valid:!0,errors:[]}};var n={};return Object.keys(t).forEach(function(r){n[r]=e(t[r])}),function(e){e=e||{};var t=Object.keys(n).map(function(t){var r=e[t],i=n[t];return i(r,t,e)}).filter(function(e){return!e.valid});return{valid:0===t.length,errors:t}}};return t=y,t.rule=e,t.TYPES=d,t.RULES=h,t}var v=Object.prototype.toString;t.validate=y}).call(t,n(97).Buffer)},function(e,t,n){"use strict";function r(e,t){"MetaModel.declaresSubTypeOf"==e.name&&t.getAdapter(u.RAMLService).setExtendedTypeName(e.arguments[0]),"MetaModel.nameAtRuntime"==e.name&&t.setNameAtRuntime(e.arguments[0]),"MetaModel.description"==e.name&&t.withDescription(e.arguments[0]),"MetaModel.inlinedTemplates"==e.name&&t.getAdapter(u.RAMLService).setInlinedTemplates(!0),"MetaModel.requireValue"==e.name&&t.getAdapter(u.RAMLService).withContextRequirement(""+e.arguments[0],""+e.arguments[1]),"MetaModel.referenceIs"==e.name&&t.getAdapter(u.RAMLService).withReferenceIs(""+e.arguments[0]),"MetaModel.actuallyExports"==e.name&&t.getAdapter(u.RAMLService).withActuallyExports(""+e.arguments[0]),"MetaModel.convertsToGlobalOfType"==e.name&&t.getAdapter(u.RAMLService).withConvertsToGlobal(""+e.arguments[0]),"MetaModel.allowAny"==e.name&&t.getAdapter(u.RAMLService).withAllowAny(),"MetaModel.allowQuestion"==e.name&&t.getAdapter(u.RAMLService).withAllowQuestion(),"MetaModel.alias"==e.name&&t.getAdapter(u.RAMLService).addAlias(""+e.arguments[0]),"MetaModel.consumesRefs"==e.name&&t.getAdapter(u.RAMLService).setConsumesRefs(!0),"MetaModel.canInherit"==e.name&&t.getAdapter(u.RAMLService).withCanInherit(""+e.arguments[0]),"MetaModel.definingPropertyIsEnough"==e.name&&t.getAdapter(u.RAMLService).definingPropertyIsEnough(""+e.arguments[0]),"MetaModel.customHandling"==e.name&&t.setCustom(!0),"MetaModel.superclasses"==e.name&&t.getAdapter(u.RAMLService).registerSupertypes(e.arguments[0]),"MetaModel.possibleInterfaces"==e.name&&t.getAdapter(u.RAMLService).registerPossibleInterfaces(e.arguments[0])}function i(e,n){t.annotationHandlers[n.name](n,e)}var a=n(39),o=n(139),s=n(95),u=a;t.handleTypeAnnotation=r,t.annotationHandlers={key:function(e,t){t.withFromParentKey(),t.withKey(!0)},value:function(e,t){t.withFromParentValue()},canBeValue:function(e,t){t.withCanBeValue()},unmerged:function(e,t){t.unmerge()},startFrom:function(e,t){t.withKeyRestriction(e.arguments[0]),t.merge()},oneOf:function(e,t){t.withEnumOptions(e.arguments[0])},oftenKeys:function(e,t){t.withOftenKeys(e.arguments[0])},embeddedInMaps:function(e,t){t.getAdapter(u.RAMLPropertyParserService).withEmbedMap();
},system:function(e,t){t.getAdapter(u.RAMLPropertyParserService).withSystem(!0)},required:function(e,t){"false"!=e.arguments[0]&&e.arguments[0]!==!1&&t.withRequired(!0)},noDirectParse:function(e,t){"false"!=e.arguments[0]&&e.arguments[0]!==!1&&t.withNoDirectParse()},setsContextValue:function(e,t){t.addChildValueConstraint(new a.ChildValueConstraint(""+e.arguments[0],""+e.arguments[1]))},defaultValue:function(e,t){t.setDefaultVal(""+e.arguments[0])},defaultIntegerValue:function(e,t){t.setDefaultIntegerVal(e.arguments[0])},defaultBooleanValue:function(e,t){t.setDefaultBooleanVal(e.arguments[0])},facetId:function(e,t){"minItems"==e.arguments[0]&&t.setFacetValidator(function(e,t){if(e instanceof Array){var n=Number.parseInt(""+t.value());if(n>e.length)return"array should contain at least "+t.value()+" items"}return null}),"enum"==e.arguments[0]&&t.setFacetValidator(function(e,t){var n=e+"",r=t;try{if(!r.some(function(e){return e==n}))return"value should be one of :"+r.join(",")}catch(i){return}return null}),"maxItems"==e.arguments[0]&&t.setFacetValidator(function(e,t){if(e instanceof Array){var n=Number.parseInt(""+t);if(n<e.length)return"array should contain not more then "+t.value()+" items"}return null}),"minProperties"==e.arguments[0]&&t.setFacetValidator(function(e,t){if(e instanceof Object){var n=Number.parseInt(""+t);if(n>Object.keys(e).length)return"object should contain at least "+t.value()+" properties"}return null}),"maxProperties"==e.arguments[0]&&t.setFacetValidator(function(e,t){if(e instanceof Object){var n=Number.parseInt(""+t);if(n<Object.keys(e).length)return"object should contain not more then "+t.value()+" properties"}return null}),"uniqueItems"==e.arguments[0]&&t.setFacetValidator(function(e,t){if(t instanceof Array){var n=s.unique(t).length;if(n<t.length)return"array should contain only unique items"}return null}),"minLength"==e.arguments[0]&&t.setFacetValidator(function(e,t){if(("number"==typeof e||"boolean"==typeof e)&&(e=""+e),"string"==typeof e){var n=Number.parseInt(""+t);if(n>e.length)return"string length should be at least "+n}return null}),"maxLength"==e.arguments[0]&&t.setFacetValidator(function(e,t){if(("number"==typeof e||"boolean"==typeof e)&&(e=""+e),"string"==typeof e){var n=Number.parseInt(""+t);if(n<e.length)return"string length should be not more then "+n}return null}),"minimum"==e.arguments[0]&&t.setFacetValidator(function(e,t){if("string"==typeof e&&(e=parseFloat(e)),"number"==typeof e){var n=parseFloat(t);if(n>e)return"value should be not less then "+n}return null}),"maximum"==e.arguments[0]&&t.setFacetValidator(function(e,t){if("string"==typeof e&&(e=parseFloat(e)),"number"==typeof e){var n=parseFloat(t);if(e>n)return"value should be not more then "+n}return null}),"pattern"==e.arguments[0]&&t.setFacetValidator(function(e,t){if(("number"==typeof e||"boolean"==typeof e)&&(e=""+e),"string"==typeof e){var n=new RegExp(t);if(!n.test(e))return"string should match to "+t}return null})},extraMetaKey:function(e,t){"statusCodes"==e.arguments[0]&&(t.withOftenKeys(o.statusCodes.filter(function(e){return e.code.indexOf("x")<0}).map(function(e){return e.code})),t.setValueDocProvider(function(e){var t=s.find(o.statusCodes,function(t){return t.code==e});return t?e+":"+t.description:null})),"headers"==e.arguments[0]&&(t.setValueSuggester(function(e){if(e.property()){var t=e.property().getChildValueConstraints();if(s.find(t,function(e){return"location"==e.name&&"Params.ParameterLocation.HEADERS"==e.value}))return o.headers.map(function(e){return e.header});if(e.property()&&"headers"==e.property().nameId())return o.headers.map(function(e){return e.header})}return null}),t.setValueDocProvider(function(e){var t=s.find(o.headers,function(t){return t.header==e});return t?e+":"+t.description:null})),"methods"==e.arguments[0]&&t.setValueDocProvider(function(e){var t=s.find(o.methods,function(t){return t.method==e.toUpperCase()});return t?e+":"+t.description:null})},requireValue:function(e,t){t.withContextRequirement(""+e.arguments[0],""+e.arguments[1])},allowMultiple:function(e,t){t.withMultiValue(!0)},constraint:function(e,t){},newInstanceName:function(e,t){t.withNewInstanceName(""+e.arguments[0])},declaringFields:function(e,t){t.withThisPropertyDeclaresFields()},describesAnnotation:function(e,t){t.withDescribes(e.arguments[0])},allowNull:function(e,t){t.withAllowNull()},descriminatingProperty:function(e,t){t.withDescriminating(!0)},description:function(e,t){t.withDescription(""+e.arguments[0])},inherited:function(e,t){t.withInherited(!0)},selfNode:function(e,t){t.withSelfNode()},grammarTokenKind:function(e,t){t.getAdapter(u.RAMLPropertyService).withPropertyGrammarType(""+e.arguments[0])},canInherit:function(e,t){t.withInheritedContextValue(""+e.arguments[0])},canBeDuplicator:function(e,t){t.setCanBeDuplicator()},example:function(e,t){t.getAdapter(u.RAMLPropertyService).setExample(!0)},typeExpression:function(e,t){t.getAdapter(u.RAMLPropertyService).setTypeExpression(!0)},hide:function(e,t){0==e.arguments.length?t.getAdapter(u.RAMLPropertyDocumentationService).setHidden(!0):t.getAdapter(u.RAMLPropertyDocumentationService).setHidden(e.arguments[0])},documentationTableLabel:function(e,t){t.getAdapter(u.RAMLPropertyDocumentationService).setDocTableName(""+e.arguments[0])},markdownDescription:function(e,t){t.getAdapter(u.RAMLPropertyDocumentationService).setMarkdownDescription(""+e.arguments[0])},valueDescription:function(e,t){t.getAdapter(u.RAMLPropertyDocumentationService).setValueDescription(null!=e.arguments[0]?""+e.arguments[0]:null)},customHandling:function(e,t){}},t.recordAnnotation=i},function(e,t,n){e.exports={34:{code:"34",message:""},1105:{code:"1105",message:""},1106:{code:"1106",message:""},1107:{code:"1107",message:""},1108:{code:"1108",message:""},1109:{code:"1109",message:""},1110:{code:"1110",message:""},CONTEXT_REQUIREMENT_VIOLATION:{code:"CONTEXT_REQUIREMENT_VIOLATION",message:"{{v1}} should be {{v2}} to use type {{v3}}"},INVALID_PARAMETER_NAME:{code:"INVALID_PARAMETER_NAME",message:"Invalid parameter name: '{{paramName}}' is reserved"},UNUSED_PARAMETER:{code:"UNUSED_PARAMETER",message:"Unused parameter: '{{paramName}}'"},INVALID_PROPERTY_OWNER_TYPE:{code:"INVALID_PROPERTY_OWNER_TYPE",message:"Property '{{propName}}' can only be used if type is {{namesStr}}"},NODE_KEY_IS_A_MAP:{code:"NODE_KEY_IS_A_MAP",message:"Node key can not be a map"},NODE_KEY_IS_A_SEQUENCE:{code:"NODE_KEY_IS_A_SEQUENCE",message:"Node key can not be a sequence"},SEQUENCE_REQUIRED:{code:"SEQUENCE_REQUIRED",message:"Node: '{{name}}' should be wrapped in sequence"},PROPERTY_MUST_BE_A_MAP_10:{code:"PROPERTY_MUST_BE_A_MAP_10",message:"'{{propName}}' should be a map in RAML 1.0"},PROPERTY_MUST_BE_A_MAP:{code:"PROPERTY_MUST_BE_A_MAP",message:"Property '{{propName}}' should be a map"},PROPERTY_MUST_BE_A_SEQUENCE:{code:"PROPERTY_MUST_BE_A_SEQUENCE",message:"Property '{{propName}}' should be a sequence"},INVALID_PROPERTY_RANGE:{code:"INVALID_PROPERTY_RANGE",message:"Property '{{propName}}' must be a {{range}}"},MAP_REQUIRED:{code:"MAP_REQUIRED",message:"should be a map in RAML 1.0"},UNRESOLVED_REFERENCE:{code:"UNRESOLVED_REFERENCE",message:"Reference: '{{ref}}' can not be resolved"},SCALAR_PROHIBITED:{code:"SCALAR_PROHIBITED",message:"Property '{{propName}}' can not have scalar value"},UNKNOWN_NODE:{code:"UNKNOWN_NODE",message:"Unknown node: '{{name}}'"},RECURSIVE_DEFINITION:{code:"RECURSIVE_DEFINITION",message:"Recursive definition: '{{name}}'"},PROPERTY_ALREADY_SPECIFIED:{code:"PROPERTY_ALREADY_SPECIFIED",message:"'{{propName}}' is already specified.`"},ISSUES_IN_THE_LIBRARY:{code:"ISSUES_IN_THE_LIBRARY",message:"Issues in the used library: '{{value}}'"},INVALID_LIBRARY_PATH:{code:"INVALID_LIBRARY_PATH",message:"Can not resolve library from path: '{{path}}'"},EMPTY_FILE:{code:"EMPTY_FILE",message:"Empty file: {{path}}"},SPACES_IN_KEY:{code:"SPACES_IN_KEY",message:"Keys should not have spaces '{{value}}'"},INCLUDE_ERROR:{code:"INCLUDE_ERROR",message:"{{msg}}"},PATH_EXCEEDS_ROOT:{code:"PATH_EXCEEDS_ROOT",message:"Resolved include path exceeds file system root"},ILLEGAL_PATTERN:{code:"ILLEGAL_PATTERN",message:"Illegal pattern: '{{value}}'"},UNKNOWN_FUNCTION:{code:"UNKNOWN_FUNCTION",message:"Unknown function applied to parameter: '{{transformerName}}'"},REQUEST_BODY_DISABLED:{code:"REQUEST_BODY_DISABLED",message:"Request body is disabled for '{{methodName}}' method."},SCALAR_EXPECTED:{code:"SCALAR_EXPECTED",message:"Scalar is expected here"},STRING_EXPECTED:{code:"STRING_EXPECTED",message:"Property '{{propName}}' must be a string"},UNKNOWN_ANNOTATION:{code:"UNKNOWN_ANNOTATION",message:"Unknown annotation: '{{aName}}'"},STRING_EXPECTED_2:{code:"STRING_EXPECTED_2",message:"{{propName}} value should be a string"},SECUREDBY_LIST_08:{code:"SECUREDBY_LIST_08",message:"'securedBy' should be a list in RAML08"},INVALID_ANNOTATION_LOCATION:{code:"INVALID_ANNOTATION_LOCATION",message:"Annotation '{{aName}}' can not be placed at this location, allowed targets are: {{aValues}}"},BOOLEAN_EXPECTED:{code:"BOOLEAN_EXPECTED",message:"'true' or 'false' is expected here"},NUMBER_EXPECTED:{code:"NUMBER_EXPECTED",message:"Value of '{{propName}}' must be a number"},STRING_EXPECTED_3:{code:"STRING_EXPECTED_3",message:"'{{propName}}' must be a string"},STATUS_MUST_BE_3NUMBER:{code:"STATUS_MUST_BE_3NUMBER",message:"Status code should be 3 digits number."},EMPTY_VALUE_NOT_ALLOWED:{code:"EMPTY_VALUE_NOT_ALLOWED",message:"Empty value is not allowed here"},INVALID_VALUE_SCHEMA:{code:"INVALID_VALUE_SCHEMA",message:"Invalid value schema: '{{iValue}}'"},INVALID_VALUE:{code:"INVALID_VALUE",message:"Invalid value: '{{iValue}}'. Allowed values are: {{aValues}}"},SCHEMA_EXCEPTION:{code:"SCHEMA_EXCEPTION",message:"Schema exception: {{msg}}"},SCHEMA_ERROR:{code:"SCHEMA_ERROR",message:"{{msg}}"},MISSING_VERSION:{code:"MISSING_VERSION",message:"Missing 'version'"},URI_PARAMETER_NAME_MISSING:{code:"URI_PARAMETER_NAME_MISSING",message:"URI parameter must have name specified"},URI_EXCEPTION:{code:"URI_EXCEPTION",message:"{{msg}}"},INVALID_MEDIATYPE:{code:"INVALID_MEDIATYPE",message:"Invalid media type '{{mediaType}}'"},MEDIATYPE_EXCEPTION:{code:"MEDIATYPE_EXCEPTION",message:"{{msg}}"},FORM_IN_RESPONSE:{code:"FORM_IN_RESPONSE",message:"Form related media types can not be used in responses"},UNUSED_URL_PARAMETER:{code:"UNUSED_URL_PARAMETER",message:"Unused url parameter {{paramName}}"},UNRECOGNIZED_ELEMENT:{code:"UNRECOGNIZED_ELEMENT",message:"Unrecognized {{referencedToName}}: '{{ref}}'."},TYPES_VARIETY_RESTRICTION:{code:"TYPES_VARIETY_RESTRICTION",message:"Type can be either of: string, number, integer, file, date or boolean"},UNRECOGNIZED_SECURITY_SCHEME:{code:"UNRECOGNIZED_SECURITY_SCHEME",message:"Unrecognized security scheme type: '{{ref}}'. Allowed values are: 'OAuth 1.0', 'OAuth 2.0', 'Basic Authentication', 'DigestSecurityScheme Authentication', 'x-{other}'"},DUPLICATE_TRAIT_REFERENCE:{code:"DUPLICATE_TRAIT_REFERENCE",message:"Duplicate trait reference: '{{refValue}}'."},IS_IS_ARRAY:{code:"IS_IS_ARRAY",message:"Property 'is' must be an array"},RESOURCE_TYPE_NAME:{code:"RESOURCE_TYPE_NAME",message:"Resource type name must be provided"},MULTIPLE_RESOURCE_TYPES:{code:"MULTIPLE_RESOURCE_TYPES",message:"A resource or resourceType can inherit from a single resourceType"},UNKNOWN_RAML_VERSION:{code:"UNKNOWN_RAML_VERSION",message:"Unknown version of RAML expected to see one of '#%RAML 0.8' or '#%RAML 1.0'"},UNKNOWN_TOPL_LEVEL_TYPE:{code:"UNKNOWN_TOPL_LEVEL_TYPE",message:"Unknown top level type: '{{typeName}}'"},REDUNDANT_FRAGMENT_NAME:{code:"REDUNDANT_FRAGMENT_NAME",message:"Redundant fragment name: '{{typeName}}'"},WEB_FORMS:{code:"WEB_FORMS",message:"File type can be only used in web forms"},MISSING_REQUIRED_PROPERTY:{code:"MISSING_REQUIRED_PROPERTY",message:"Missing required property '{{propName}}'"},VALUE_NOT_PROVIDED:{code:"VALUE_NOT_PROVIDED",message:"Value is not provided for parameter: '{{propName}}'"},SUSPICIOUS_DOUBLEQUOTE:{code:"SUSPICIOUS_DOUBLEQUOTE",message:'Suspicious double quoted multiline scalar, it is likely that you forgot closing "{{value}}'},ANNOTATION_TARGET_MUST_BE_A_STRING:{code:"ANNOTATION_TARGET_MUST_BE_A_STRING",message:"Annotation target must be set by a string"},ALLOWED_TARGETS_MUST_BE_ARRAY:{code:"ALLOWED_TARGETS_MUST_BE_ARRAY",message:"'allowedTargets' property value must be an array of type names or a single type name"},UNSUPPORTED_ANNOTATION_TARGET:{code:"UNSUPPORTED_ANNOTATION_TARGET",message:"Unsupported annotation target: '{{aTarget}}'"},SCHEMA_NAME_MUST_BE_STRING:{code:"SCHEMA_NAME_MUST_BE_STRING",message:"Schema '{{name}}' must be a string"},FORM_PARAMS_IN_RESPONSE:{code:"FORM_PARAMS_IN_RESPONSE",message:"Form parameters can not be used in response"},FORM_PARAMS_WITH_EXAMPLE:{code:"FORM_PARAMS_WITH_EXAMPLE",message:"'formParameters' property cannot be used together with the 'example' or 'schema' properties"},AUTHORIZATION_GRANTS_ENUM:{code:"AUTHORIZATION_GRANTS_ENUM",message:"'authorizationGrants' value should be one of 'authorization_code', 'implicit', 'password', 'client_credentials' or to be an abolute URI"},AUTHORIZATION_URI_REQUIRED:{code:"AUTHORIZATION_URI_REQUIRED",message:"'authorizationUri' is required when `authorization_code` or `implicit` grant type are used "},REPEATING_COMPONENTS_IN_ENUM:{code:"REPEATING_COMPONENTS_IN_ENUM",message:"'enum' value contains repeating components"},INTEGER_EXPECTED:{code:"INTEGER_EXPECTED",message:"Integer is expected"},NUMBER_EXPECTED_2:{code:"NUMBER_EXPECTED_2",message:"Number is expected"},RESOURCE_TYPE_NULL:{code:"RESOURCE_TYPE_NULL",message:"Resource type can not be null"},SCALAR_PROHIBITED_2:{code:"SCALAR_PROHIBITED_2",message:"Node '{{name}}' can not be a scalar"},VERSION_NOT_ALLOWED:{code:"VERSION_NOT_ALLOWED",message:"'version' parameter not allowed here",comment:"I dont like the message, but its coming from JS 0.8 parser @Denis"},INVALID_OVERLAY_NODE:{code:"INVALID_OVERLAY_NODE",message:"The '{{nodeId}}' node does not match any node of the master api."},INVALID_OVERRIDE_IN_OVERLAY:{code:"INVALID_OVERRIDE_IN_OVERLAY",message:"Property '{{propName}}' is not allowed to be overriden or added in overlays"},KEYS_SHOULD_BE_UNIQUE:{code:"KEYS_SHOULD_BE_UNIQUE",message:"Keys should be unique"},ALREADY_EXISTS:{code:"ALREADY_EXISTS",message:"{{capitalized}} '{{name}}' already exists"},ALREADY_EXISTS_IN_CONTEXT:{code:"ALREADY_EXISTS_IN_CONTEXT",message:"{{name}}' already exists in this context"},PROPERTY_USED:{code:"PROPERTY_USED",message:"Property already used: '{{propName}}'"},PARENT_PROPERTY_USED:{code:"PARENT_PROPERTY_USED",message:"{{parent}} property already used: '{{propName}}'"},INVALID_JSON_SCHEMA:{code:"INVALID_JSON_SCHEMA",message:"Invalid JSON schema. Unexpected value '{{propName}}'"},EXAMPLE_SCHEMA_FAILURE:{code:"EXAMPLE_SCHEMA_FAILURE",message:"Example does not conform to schema: {{msg}}"},CAN_NOT_PARSE_JSON:{code:"CAN_NOT_PARSE_JSON",message:"Can not parse JSON: {{msg}}"},CAN_NOT_PARSE_XML:{code:"CAN_NOT_PARSE_XML",message:"Can not parse XML: {{msg}}"},OPTIONAL_SCLARAR_PROPERTIES_10:{code:"OPTIONAL_SCLARAR_PROPERTIES_10",message:"Optional scalar properties are not allowed in {{templateNamePlural}} and their descendants: '{{propName}}?'"},OPTIONAL_PROPERTIES_10:{code:"OPTIONAL_PROPERTIES_10",message:"Optional properties are not allowed in '{{propName}}': '{{oPropName}}'"},ONLY_METHODS_CAN_BE_OPTIONAL:{code:"ONLY_METHODS_CAN_BE_OPTIONAL",message:"Only method nodes can be optional"},PROPERTY_UNUSED:{code:"PROPERTY_UNUSED",message:"{{propName}} unused"},CYCLE_IN_DEFINITION:{code:"CYCLE_IN_DEFINITION",message:"{{typeName}} definition contains cycle: {{cycle}}"},SEQUENCE_NOT_ALLOWED_10:{code:"SEQUENCE_NOT_ALLOWED_10",message:"In RAML 1.0 {{propName}} is not allowed to have sequence as definition"},MAP_EXPECTED:{code:"MAP_EXPECTED",message:"Map is expected here"},ERROR_IN_INCLUDED_FILE:{code:"ERROR_IN_INCLUDED_FILE",message:"Error in the included file: {{msg}}"},NODE_SHOULD_HAVE_VALUE:{code:"NODE_SHOULD_HAVE_VALUE",message:"node should have at least one member value"},NAMED_PARAMETER_NEEDS_TYPE:{code:"NAMED_PARAMETER_NEEDS_TYPE",message:"named parameter needs at least one type"},ENUM_IS_EMPTY:{code:"ENUM_IS_EMPTY",message:"enum is empty"},ENUM_MUST_BE_AN_ARRAY:{code:"ENUM_MUST_BE_AN_ARRAY",message:"the value of enum must be an array"},DEFINITION_SHOULD_BE_A_MAP:{code:"DEFINITION_SHOULD_BE_A_MAP",message:"{{typeName}} definition should be a map"},RESOURCES_SHARE_SAME_URI:{code:"RESOURCES_SHARE_SAME_URI",message:"Resources share same URI"},TYPES_AND_SCHEMAS_ARE_EXCLUSIVE:{code:"TYPES_AND_SCHEMAS_ARE_EXCLUSIVE",message:"'types' and 'schemas' are mutually exclusive"},TEMPLATE_PARAMETER_NAME_MUST_CONTAIN_NONWHITESPACE_CHARACTERS:{code:"TEMPLATE_PARAMETER_NAME_MUST_CONTAIN_NONWHITESPACE_CHARACTERS",message:"Trait or resource type parameter name must contain non whitespace characters"},INVALID_SECURITY_SCHEME_SCOPE:{code:"INVALID_SECURITY_SCHEME_SCOPE",message:"The '{{invalidScope}}' scope is not allowed for the '{{securityScheme}}' security scheme. Allowed scopes are: {{allowedScopes}}."},INCLUDE_TAG_MISSING:{code:"INCLUDE_TAG_MISSING",message:"The '!include' tag is missing"}}},function(e,t,n){(function(e){(function(){var r,i=[].slice;r=n(120),t.allowUnsafeEval=function(t){var n;n=e.eval;try{return e.eval=function(e){return r.runInThisContext(e)},t()}finally{e.eval=n}},t.allowUnsafeNewFunction=function(n){var r;r=e.Function;try{return e.Function=t.Function,n()}finally{e.Function=r}},t.Function=function(){var e,t,n,a,o,s,u;for(n=2<=arguments.length?i.call(arguments,0,o=arguments.length-1):(o=0,[]),e=arguments[o++],a=[],s=0,u=n.length;u>s;s++)t=n[s],"string"==typeof t&&(t=t.split(/\s*,\s*/)),a.push.apply(a,t);return r.runInThisContext("(function("+a.join(", ")+") {\n  "+e+"\n})")},t.Function.prototype=e.Function.prototype}).call(this)}).call(t,function(){return this}())},function(e,t,n){"use strict";t.decode=t.parse=n(98),t.encode=t.stringify=n(99)},function(e,t,n){"use strict";function r(){return L}function i(e){return e instanceof b.AbstractType}function a(e,t){return void 0===t&&(t=b.builtInRegistry()),_.parseJSONTypeCollection(e,t)}function o(e){return _.parseJSON(null,e,b.builtInRegistry())}function s(e){return e.types||e.annotationTypes?_.parseJSONTypeCollection(e):_.parseJSON(null,e)}function u(e,t){return _.parseJSON(null,e,t?t.getTypeRegistry():b.builtInRegistry())}function l(e){return _.parseTypeCollection(e,b.builtInRegistry())}function p(e,t,n,r,i,a,o){if(void 0===r&&(r=!1),void 0===i&&(i=!1),void 0===a&&(a=!0),void 0===o&&(o=!1),a){var s;if(s=i?n.getAnnotationType(e):n.getType(e),null!=s)return s}return _.parse(e,t,n?n.getTypeRegistry():b.builtInRegistry(),r,i,a,o)}function c(e){return _.storeAsJSON(e)}function f(e,t,n){void 0===n&&(n=!1),b.autoCloseFlag=n;try{return t.validate(e,n)}finally{b.autoCloseFlag=!1}}function d(e,t){return e.validateType(t.getAnnotationTypeRegistry())}function h(e,t){return t.ac(e)}function m(e){return e.canDoAc()}function y(){return N.getInstance().allPrototypes()}function v(){return b.builtInRegistry()}function g(e){for(var t=[],n=1;n<arguments.length;n++)t[n-1]=arguments[n];return b.derive(e,t)}function A(e){for(var t=[],n=1;n<arguments.length;n++)t[n-1]=arguments[n];return b.union(e,t)}function E(e){C.setPropertyConstructor(e)}function T(e,t){return C.toNominal(e,t)}function S(e){return b.toValidationPath(e)}var b=n(100);t.tsInterfaces=n(101);var _=n(102),N=n(103),w=n(104),R=n(105),I=n(105),M=n(105),C=n(106);t.nominalTypes=n(107),t.nominalInterfaces=n(108),t.typeExpressions=n(109);var L=n(110);t.TOP_LEVEL_EXTRA=t.tsInterfaces.TOP_LEVEL_EXTRA,t.DEFINED_IN_TYPES_EXTRA=t.tsInterfaces.DEFINED_IN_TYPES_EXTRA,t.USER_DEFINED_EXTRA=t.tsInterfaces.USER_DEFINED_EXTRA,t.SOURCE_EXTRA=t.tsInterfaces.SOURCE_EXTRA,t.getSchemaUtils=r,t.isParsedType=i,t.loadTypeCollection=a,t.loadType=o,t.parse=s,t.parseType=u,function(e){e[e.SCALAR=0]="SCALAR",e[e.ARRAY=1]="ARRAY",e[e.MAP=2]="MAP"}(t.NodeKind||(t.NodeKind={}));t.NodeKind;t.parseFromAST=l,t.parseTypeFromAST=p,t.dump=c,t.validate=f,t.validateTypeDefinition=d,t.performAC=h,t.checkACStatus=m,t.builtInFacets=y,t.builtInTypes=v,t.derive=g,t.unify=A;var P=function(){function e(e){this.target=e}return e.prototype.addProperty=function(e,t,n){return this.target.declareProperty(e,t,n),this},e.prototype.closeType=function(){return this.target.addMeta(new w.KnownPropertyRestriction(!1)),this},e.prototype.annotate=function(e,t){return this.target.addMeta(new R.Annotation(e,t)),this},e.prototype.customFacet=function(e,t){return this.target.addMeta(new I.CustomFacet(e,t)),this},e.prototype.customFacetDeclaration=function(e,t,n){return void 0===n&&(n=!0),this.target.addMeta(new M.FacetDeclaration(e,t,n)),this},e.prototype.addSimpleFacet=function(e,t){return this.target.addMeta(N.getInstance().buildFacet(e,t)),this},e.prototype.getResult=function(){return this.target},e}();t.TypeConstructor=P,t.setPropertyConstructor=E,t.toNominal=T,t.toValidationPath=S},function(e,t,n){var r;(function(e,i){!function(a){function o(e){throw RangeError(P[e])}function s(e,t){for(var n=e.length,r=[];n--;)r[n]=t(e[n]);return r}function u(e,t){var n=e.split("@"),r="";n.length>1&&(r=n[0]+"@",e=n[1]),e=e.replace(L,".");var i=e.split("."),a=s(i,t).join(".");return r+a}function l(e){for(var t,n,r=[],i=0,a=e.length;a>i;)t=e.charCodeAt(i++),t>=55296&&56319>=t&&a>i?(n=e.charCodeAt(i++),56320==(64512&n)?r.push(((1023&t)<<10)+(1023&n)+65536):(r.push(t),i--)):r.push(t);return r}function p(e){return s(e,function(e){var t="";return e>65535&&(e-=65536,t+=x(e>>>10&1023|55296),e=56320|1023&e),t+=x(e)}).join("")}function c(e){return 10>e-48?e-22:26>e-65?e-65:26>e-97?e-97:T}function f(e,t){return e+22+75*(26>e)-((0!=t)<<5)}function d(e,t,n){var r=0;for(e=n?D(e/N):e>>1,e+=D(e/t);e>O*b>>1;r+=T)e=D(e/O);return D(r+(O+1)*e/(e+_))}function h(e){var t,n,r,i,a,s,u,l,f,h,m=[],y=e.length,v=0,g=R,A=w;for(n=e.lastIndexOf(I),0>n&&(n=0),r=0;n>r;++r)e.charCodeAt(r)>=128&&o("not-basic"),m.push(e.charCodeAt(r));for(i=n>0?n+1:0;y>i;){for(a=v,s=1,u=T;i>=y&&o("invalid-input"),l=c(e.charCodeAt(i++)),(l>=T||l>D((E-v)/s))&&o("overflow"),v+=l*s,f=A>=u?S:u>=A+b?b:u-A,!(f>l);u+=T)h=T-f,s>D(E/h)&&o("overflow"),s*=h;t=m.length+1,A=d(v-a,t,0==a),D(v/t)>E-g&&o("overflow"),g+=D(v/t),v%=t,m.splice(v++,0,g)}return p(m)}function m(e){var t,n,r,i,a,s,u,p,c,h,m,y,v,g,A,_=[];for(e=l(e),y=e.length,t=R,n=0,a=w,s=0;y>s;++s)m=e[s],128>m&&_.push(x(m));for(r=i=_.length,i&&_.push(I);y>r;){for(u=E,s=0;y>s;++s)m=e[s],m>=t&&u>m&&(u=m);for(v=r+1,u-t>D((E-n)/v)&&o("overflow"),n+=(u-t)*v,t=u,s=0;y>s;++s)if(m=e[s],t>m&&++n>E&&o("overflow"),m==t){for(p=n,c=T;h=a>=c?S:c>=a+b?b:c-a,!(h>p);c+=T)A=p-h,g=T-h,_.push(x(f(h+A%g,0))),p=D(A/g);_.push(x(f(p,0))),a=d(n,v,r==i),n=0,++r}++n,++t}return _.join("")}function y(e){return u(e,function(e){return M.test(e)?h(e.slice(4).toLowerCase()):e})}function v(e){return u(e,function(e){return C.test(e)?"xn--"+m(e):e})}var g=("object"==typeof t&&t&&!t.nodeType&&t,"object"==typeof e&&e&&!e.nodeType&&e,"object"==typeof i&&i);(g.global===g||g.window===g||g.self===g)&&(a=g);var A,E=2147483647,T=36,S=1,b=26,_=38,N=700,w=72,R=128,I="-",M=/^xn--/,C=/[^\x20-\x7E]/,L=/[\x2E\u3002\uFF0E\uFF61]/g,P={overflow:"Overflow: input needs wider integers to process","not-basic":"Illegal input >= 0x80 (not a basic code point)","invalid-input":"Invalid input"},O=T-S,D=Math.floor,x=String.fromCharCode;A={version:"1.3.2",ucs2:{decode:l,encode:p},decode:h,encode:m,toASCII:v,toUnicode:y},r=function(){return A}.call(t,n,t,e),!(void 0!==r&&(e.exports=r))}(this)}).call(t,n(149)(e),function(){return this}())},function(e,t,n){function r(){}function i(e,t,n,r,i){function p(e){if(e>65535){e-=65536;var t=55296+(e>>10),n=56320+(1023&e);return String.fromCharCode(t,n)}return String.fromCharCode(e)}function h(e){var t=e.slice(1,-1);return t in n?n[t]:"#"===t.charAt(0)?p(parseInt(t.substr(1).replace("x","0x"))):(i.error("entity not found:"+e),e)}function m(t){if(t>b){var n=e.substring(b,t).replace(/&#?\w+;/g,h);E&&y(b),r.characters(n,0,t-b),b=t}}function y(t,n){for(;t>=g&&(n=A.exec(e));)v=n.index,g=v+n[0].length,E.lineNumber++;E.columnNumber=t-v+1}for(var v=0,g=0,A=/.*(?:\r\n?|\n)|.*$/g,E=r.locator,T=[{currentNSMap:t}],S={},b=0;;){try{var _=e.indexOf("<",b);if(0>_){if(!e.substr(b).match(/^\s*$/)){var N=r.doc,w=N.createTextNode(e.substr(b));N.appendChild(w),r.currentElement=w}return}switch(_>b&&m(_),e.charAt(_+1)){case"/":var R=e.indexOf(">",_+3),I=e.substring(_+2,R),M=T.pop();0>R?(I=e.substring(_+2).replace(/[\s<].*/,""),i.error("end tag name: "+I+" is not complete:"+M.tagName),R=_+1+I.length):I.match(/\s</)&&(I=I.replace(/[\s<].*/,""),i.error("end tag name: "+I+" maybe not complete"),R=_+1+I.length);var C=M.localNSMap,L=M.tagName==I,P=L||M.tagName&&M.tagName.toLowerCase()==I.toLowerCase();if(P){if(r.endElement(M.uri,M.localName,I),C)for(var O in C)r.endPrefixMapping(O);L||i.fatalError("end tag name: "+I+" is not match the current start tagName:"+M.tagName)}else T.push(M);R++;break;case"?":E&&y(_),R=f(e,_,r);break;case"!":E&&y(_),R=c(e,_,r,i);break;default:E&&y(_);var D=new d,x=T[T.length-1].currentNSMap,R=o(e,_,D,x,h,i),U=D.length;if(!D.closed&&l(e,R,D.tagName,S)&&(D.closed=!0,n.nbsp||i.warning("unclosed xml attribute")),E&&U){for(var k=a(E,{}),F=0;U>F;F++){var B=D[F];y(B.offset),B.locator=a(E,{})}r.locator=k,s(D,r,x)&&T.push(D),r.locator=E}else s(D,r,x)&&T.push(D);"http://www.w3.org/1999/xhtml"!==D.uri||D.closed?R++:R=u(e,R,D.tagName,h,r)}}catch(K){i.error("element parse error: "+K),R=-1}R>b?b=R:m(Math.max(_,b)+1)}}function a(e,t){return t.lineNumber=e.lineNumber,t.columnNumber=e.columnNumber,t}function o(e,t,n,r,i,a){for(var o,s,u=++t,l=A;;){var p=e.charAt(u);switch(p){case"=":if(l===E)o=e.slice(t,u),l=S;else{if(l!==T)throw new Error("attribute equal must after attrName");l=S}break;case"'":case'"':if(l===S||l===E){if(l===E&&(a.warning('attribute value must after "="'),o=e.slice(t,u)),t=u+1,u=e.indexOf(p,t),!(u>0))throw new Error("attribute value no end '"+p+"' match");s=e.slice(t,u).replace(/&#?\w+;/g,i),n.add(o,s,t-1),l=_}else{if(l!=b)throw new Error('attribute value must after "="');s=e.slice(t,u).replace(/&#?\w+;/g,i),n.add(o,s,t),a.warning('attribute "'+o+'" missed start quot('+p+")!!"),t=u+1,l=_}break;case"/":switch(l){case A:n.setTagName(e.slice(t,u));case _:case N:case w:l=w,n.closed=!0;case b:case E:case T:break;default:throw new Error("attribute invalid close char('/')")}break;case"":return a.error("unexpected end of input"),l==A&&n.setTagName(e.slice(t,u)),u;case">":switch(l){case A:n.setTagName(e.slice(t,u));case _:case N:case w:break;case b:case E:s=e.slice(t,u),"/"===s.slice(-1)&&(n.closed=!0,s=s.slice(0,-1));case T:l===T&&(s=o),l==b?(a.warning('attribute "'+s+'" missed quot(")!!'),n.add(o,s.replace(/&#?\w+;/g,i),t)):("http://www.w3.org/1999/xhtml"===r[""]&&s.match(/^(?:disabled|checked|selected)$/i)||a.warning('attribute "'+s+'" missed value!! "'+s+'" instead!!'),n.add(s,s,t));break;case S:throw new Error("attribute value missed!!")}return u;case"":p=" ";default:if(" ">=p)switch(l){case A:n.setTagName(e.slice(t,u)),l=N;break;case E:o=e.slice(t,u),l=T;break;case b:var s=e.slice(t,u).replace(/&#?\w+;/g,i);a.warning('attribute "'+s+'" missed quot(")!!'),n.add(o,s,t);case _:l=N}else switch(l){case T:n.tagName;"http://www.w3.org/1999/xhtml"===r[""]&&o.match(/^(?:disabled|checked|selected)$/i)||a.warning('attribute "'+o+'" missed value!! "'+o+'" instead2!!'),n.add(o,o,t),t=u,l=E;break;case _:a.warning('attribute space is required"'+o+'"!!');case N:l=E,t=u;break;case S:l=b,t=u;break;case w:throw new Error("elements closed character '/' and '>' must be connected to")}}u++}}function s(e,t,n){for(var r=e.tagName,i=null,a=e.length;a--;){var o=e[a],s=o.qName,u=o.value,l=s.indexOf(":");if(l>0)var c=o.prefix=s.slice(0,l),f=s.slice(l+1),d="xmlns"===c&&f;else f=s,c=null,d="xmlns"===s&&"";o.localName=f,d!==!1&&(null==i&&(i={},p(n,n={})),n[d]=i[d]=u,o.uri="http://www.w3.org/2000/xmlns/",t.startPrefixMapping(d,u))}for(var a=e.length;a--;){o=e[a];var c=o.prefix;c&&("xml"===c&&(o.uri="http://www.w3.org/XML/1998/namespace"),"xmlns"!==c&&(o.uri=n[c||""]))}var l=r.indexOf(":");l>0?(c=e.prefix=r.slice(0,l),f=e.localName=r.slice(l+1)):(c=null,f=e.localName=r);var h=e.uri=n[c||""];if(t.startElement(h,f,r,e),!e.closed)return e.currentNSMap=n,e.localNSMap=i,!0;if(t.endElement(h,f,r),i)for(c in i)t.endPrefixMapping(c)}function u(e,t,n,r,i){if(/^(?:script|textarea)$/i.test(n)){var a=e.indexOf("</"+n+">",t),o=e.substring(t+1,a);if(/[&<]/.test(o))return/^script$/i.test(n)?(i.characters(o,0,o.length),a):(o=o.replace(/&#?\w+;/g,r),i.characters(o,0,o.length),a)}return t+1}function l(e,t,n,r){var i=r[n];return null==i&&(i=e.lastIndexOf("</"+n+">"),t>i&&(i=e.lastIndexOf("</"+n)),r[n]=i),t>i}function p(e,t){for(var n in e)t[n]=e[n]}function c(e,t,n,r){var i=e.charAt(t+2);switch(i){case"-":if("-"===e.charAt(t+3)){var a=e.indexOf("-->",t+4);return a>t?(n.comment(e,t+4,a-t-4),a+3):(r.error("Unclosed comment"),-1)}return-1;default:if("CDATA["==e.substr(t+3,6)){var a=e.indexOf("]]>",t+9);return n.startCDATA(),n.characters(e,t+9,a-t-9),n.endCDATA(),a+3}var o=m(e,t),s=o.length;if(s>1&&/!doctype/i.test(o[0][0])){var u=o[1][0],l=s>3&&/^public$/i.test(o[2][0])&&o[3][0],p=s>4&&o[4][0],c=o[s-1];return n.startDTD(u,l&&l.replace(/^(['"])(.*?)\1$/,"$2"),p&&p.replace(/^(['"])(.*?)\1$/,"$2")),n.endDTD(),c.index+c[0].length}}return-1}function f(e,t,n){var r=e.indexOf("?>",t);if(r){var i=e.substring(t,r).match(/^<\?(\S*)\s*([\s\S]*?)\s*$/);if(i){i[0].length;return n.processingInstruction(i[1],i[2]),r+2}return-1}return-1}function d(e){}function h(e,t){return e.__proto__=t,e}function m(e,t){var n,r=[],i=/'[^']+'|"[^"]+"|[^\s<>\/=]+=?|(\/?\s*>|<)/g;for(i.lastIndex=t,i.exec(e);n=i.exec(e);)if(r.push(n),n[1])return r}var y=/[A-Z_a-z\xC0-\xD6\xD8-\xF6\u00F8-\u02FF\u0370-\u037D\u037F-\u1FFF\u200C-\u200D\u2070-\u218F\u2C00-\u2FEF\u3001-\uD7FF\uF900-\uFDCF\uFDF0-\uFFFD]/,v=new RegExp("[\\-\\.0-9"+y.source.slice(1,-1)+"\\u00B7\\u0300-\\u036F\\u203F-\\u2040]"),g=new RegExp("^"+y.source+v.source+"*(?::"+y.source+v.source+"*)?$"),A=0,E=1,T=2,S=3,b=4,_=5,N=6,w=7;r.prototype={parse:function(e,t,n){var r=this.domBuilder;r.startDocument(),p(t,t={}),i(e,t,n,r,this.errorHandler),r.endDocument()}},d.prototype={setTagName:function(e){if(!g.test(e))throw new Error("invalid tagName:"+e);this.tagName=e},add:function(e,t,n){if(!g.test(e))throw new Error("invalid attribute:"+e);this[this.length++]={qName:e,value:t,offset:n}},length:0,getLocalName:function(e){return this[e].localName},getLocator:function(e){return this[e].locator},getQName:function(e){return this[e].qName},getURI:function(e){return this[e].uri},getValue:function(e){return this[e].value}},h({},h.prototype)instanceof h||(h=function(e,t){function n(){}n.prototype=t,n=new n;for(t in e)n[t]=e[t];return n}),t.XMLReader=r},function(e,t,n){function r(e,t){for(var n in e)t[n]=e[n]}function i(e,t){function n(){}var i=e.prototype;if(Object.create){var a=Object.create(t.prototype);i.__proto__=a}i instanceof t||(n.prototype=t.prototype,n=new n,r(i,n),e.prototype=i=n),i.constructor!=e&&("function"!=typeof e&&console.error("unknow Class:"+e),i.constructor=e)}function a(e,t){if(t instanceof Error)var n=t;else n=this,Error.call(this,ae[e]),this.message=ae[e],Error.captureStackTrace&&Error.captureStackTrace(this,a);return n.code=e,t&&(this.message=this.message+": "+t),n}function o(){}function s(e,t){this._node=e,this._refresh=t,u(this)}function u(e){var t=e._node._inc||e._node.ownerDocument._inc;if(e._inc!=t){var n=e._refresh(e._node);j(e,"length",n.length),r(n,e),e._inc=t}}function l(){}function p(e,t){for(var n=e.length;n--;)if(e[n]===t)return n}function c(e,t,n,r){if(r?t[p(t,r)]=n:t[t.length++]=n,e){n.ownerElement=e;var i=e.ownerDocument;i&&(r&&A(i,e,r),g(i,e,n))}}function f(e,t,n){var r=p(t,n);if(!(r>=0))throw a(se,new Error(e.tagName+"@"+n));for(var i=t.length-1;i>r;)t[r]=t[++r];if(t.length=i,e){var o=e.ownerDocument;
o&&(A(o,e,n),n.ownerElement=null)}}function d(e){if(this._features={},e)for(var t in e)this._features=e[t]}function h(){}function m(e){return"<"==e&&"&lt;"||">"==e&&"&gt;"||"&"==e&&"&amp;"||'"'==e&&"&quot;"||"&#"+e.charCodeAt()+";"}function y(e,t){if(t(e))return!0;if(e=e.firstChild)do if(y(e,t))return!0;while(e=e.nextSibling)}function v(){}function g(e,t,n){e&&e._inc++;var r=n.namespaceURI;"http://www.w3.org/2000/xmlns/"==r&&(t._nsMap[n.prefix?n.localName:""]=n.value)}function A(e,t,n,r){e&&e._inc++;var i=n.namespaceURI;"http://www.w3.org/2000/xmlns/"==i&&delete t._nsMap[n.prefix?n.localName:""]}function E(e,t,n){if(e&&e._inc){e._inc++;var r=t.childNodes;if(n)r[r.length++]=n;else{for(var i=t.firstChild,a=0;i;)r[a++]=i,i=i.nextSibling;r.length=a}}}function T(e,t){var n=t.previousSibling,r=t.nextSibling;return n?n.nextSibling=r:e.firstChild=r,r?r.previousSibling=n:e.lastChild=n,E(e.ownerDocument,e),t}function S(e,t,n){var r=t.parentNode;if(r&&r.removeChild(t),t.nodeType===ne){var i=t.firstChild;if(null==i)return t;var a=t.lastChild}else i=a=t;var o=n?n.previousSibling:e.lastChild;i.previousSibling=o,a.nextSibling=n,o?o.nextSibling=i:e.firstChild=i,null==n?e.lastChild=a:n.previousSibling=a;do i.parentNode=e;while(i!==a&&(i=i.nextSibling));return E(e.ownerDocument||e,e),t.nodeType==ne&&(t.firstChild=t.lastChild=null),t}function b(e,t){var n=t.parentNode;if(n){var r=e.lastChild;n.removeChild(t);var r=e.lastChild}var r=e.lastChild;return t.parentNode=e,t.previousSibling=r,t.nextSibling=null,r?r.nextSibling=t:e.firstChild=t,e.lastChild=t,E(e.ownerDocument,e,t),t}function _(){this._nsMap={}}function N(){}function w(){}function R(){}function I(){}function M(){}function C(){}function L(){}function P(){}function O(){}function D(){}function x(){}function U(){}function k(e,t){var n=[],r=9==this.nodeType?this.documentElement:this,i=r.prefix,a=r.namespaceURI;if(a&&null==i){var i=r.lookupPrefix(a);if(null==i)var o=[{namespace:a,prefix:null}]}return B(this,n,e,t,o),n.join("")}function F(e,t,n){var r=e.prefix||"",i=e.namespaceURI;if(!r&&!i)return!1;if("xml"===r&&"http://www.w3.org/XML/1998/namespace"===i||"http://www.w3.org/2000/xmlns/"==i)return!1;for(var a=n.length;a--;){var o=n[a];if(o.prefix==r)return o.namespace!=i}return!0}function B(e,t,n,r,i){if(r){if(e=r(e),!e)return;if("string"==typeof e)return void t.push(e)}switch(e.nodeType){case H:i||(i=[]);var a=(i.length,e.attributes),o=a.length,s=e.firstChild,u=e.tagName;n=q===e.namespaceURI||n,t.push("<",u);for(var l=0;o>l;l++){var p=a.item(l);"xmlns"==p.prefix?i.push({prefix:p.localName,namespace:p.value}):"xmlns"==p.nodeName&&i.push({prefix:"",namespace:p.value})}for(var l=0;o>l;l++){var p=a.item(l);if(F(p,n,i)){var c=p.prefix||"",f=p.namespaceURI,d=c?" xmlns:"+c:" xmlns";t.push(d,'="',f,'"'),i.push({prefix:c,namespace:f})}B(p,t,n,r,i)}if(F(e,n,i)){var c=e.prefix||"",f=e.namespaceURI,d=c?" xmlns:"+c:" xmlns";t.push(d,'="',f,'"'),i.push({prefix:c,namespace:f})}if(s||n&&!/^(?:meta|link|img|br|hr|input)$/i.test(u)){if(t.push(">"),n&&/^script$/i.test(u))for(;s;)s.data?t.push(s.data):B(s,t,n,r,i),s=s.nextSibling;else for(;s;)B(s,t,n,r,i),s=s.nextSibling;t.push("</",u,">")}else t.push("/>");return;case ee:case ne:for(var s=e.firstChild;s;)B(s,t,n,r,i),s=s.nextSibling;return;case $:return t.push(" ",e.name,'="',e.value.replace(/[<&"]/g,m),'"');case G:return t.push(e.data.replace(/[<&]/g,m));case X:return t.push("<![CDATA[",e.data,"]]>");case Z:return t.push("<!--",e.data,"-->");case te:var h=e.publicId,y=e.systemId;if(t.push("<!DOCTYPE ",e.name),h)t.push(' PUBLIC "',h),y&&"."!=y&&t.push('" "',y),t.push('">');else if(y&&"."!=y)t.push(' SYSTEM "',y,'">');else{var v=e.internalSubset;v&&t.push(" [",v,"]"),t.push(">")}return;case Q:return t.push("<?",e.target," ",e.data,"?>");case z:return t.push("&",e.nodeName,";");default:t.push("??",e.nodeName)}}function K(e,t,n){var r;switch(t.nodeType){case H:r=t.cloneNode(!1),r.ownerDocument=e;case ne:break;case $:n=!0}if(r||(r=t.cloneNode(!1)),r.ownerDocument=e,r.parentNode=null,n)for(var i=t.firstChild;i;)r.appendChild(K(e,i,n)),i=i.nextSibling;return r}function V(e,t,n){var r=new t.constructor;for(var i in t){var a=t[i];"object"!=typeof a&&a!=r[i]&&(r[i]=a)}switch(t.childNodes&&(r.childNodes=new o),r.ownerDocument=e,r.nodeType){case H:var s=t.attributes,u=r.attributes=new l,p=s.length;u._ownerElement=r;for(var c=0;p>c;c++)r.setAttributeNode(V(e,s.item(c),!0));break;case $:n=!0}if(n)for(var f=t.firstChild;f;)r.appendChild(V(e,f,n)),f=f.nextSibling;return r}function j(e,t,n){e[t]=n}function W(e){switch(e.nodeType){case H:case ne:var t=[];for(e=e.firstChild;e;)7!==e.nodeType&&8!==e.nodeType&&t.push(W(e)),e=e.nextSibling;return t.join("");default:return e.nodeValue}}var q="http://www.w3.org/1999/xhtml",Y={},H=Y.ELEMENT_NODE=1,$=Y.ATTRIBUTE_NODE=2,G=Y.TEXT_NODE=3,X=Y.CDATA_SECTION_NODE=4,z=Y.ENTITY_REFERENCE_NODE=5,J=Y.ENTITY_NODE=6,Q=Y.PROCESSING_INSTRUCTION_NODE=7,Z=Y.COMMENT_NODE=8,ee=Y.DOCUMENT_NODE=9,te=Y.DOCUMENT_TYPE_NODE=10,ne=Y.DOCUMENT_FRAGMENT_NODE=11,re=Y.NOTATION_NODE=12,ie={},ae={},oe=(ie.INDEX_SIZE_ERR=(ae[1]="Index size error",1),ie.DOMSTRING_SIZE_ERR=(ae[2]="DOMString size error",2),ie.HIERARCHY_REQUEST_ERR=(ae[3]="Hierarchy request error",3)),se=(ie.WRONG_DOCUMENT_ERR=(ae[4]="Wrong document",4),ie.INVALID_CHARACTER_ERR=(ae[5]="Invalid character",5),ie.NO_DATA_ALLOWED_ERR=(ae[6]="No data allowed",6),ie.NO_MODIFICATION_ALLOWED_ERR=(ae[7]="No modification allowed",7),ie.NOT_FOUND_ERR=(ae[8]="Not found",8)),ue=(ie.NOT_SUPPORTED_ERR=(ae[9]="Not supported",9),ie.INUSE_ATTRIBUTE_ERR=(ae[10]="Attribute in use",10));ie.INVALID_STATE_ERR=(ae[11]="Invalid state",11),ie.SYNTAX_ERR=(ae[12]="Syntax error",12),ie.INVALID_MODIFICATION_ERR=(ae[13]="Invalid modification",13),ie.NAMESPACE_ERR=(ae[14]="Invalid namespace",14),ie.INVALID_ACCESS_ERR=(ae[15]="Invalid access",15);a.prototype=Error.prototype,r(ie,a),o.prototype={length:0,item:function(e){return this[e]||null},toString:function(e,t){for(var n=[],r=0;r<this.length;r++)B(this[r],n,e,t);return n.join("")}},s.prototype.item=function(e){return u(this),this[e]},i(s,o),l.prototype={length:0,item:o.prototype.item,getNamedItem:function(e){for(var t=this.length;t--;){var n=this[t];if(n.nodeName==e)return n}},setNamedItem:function(e){var t=e.ownerElement;if(t&&t!=this._ownerElement)throw new a(ue);var n=this.getNamedItem(e.nodeName);return c(this._ownerElement,this,e,n),n},setNamedItemNS:function(e){var t,n=e.ownerElement;if(n&&n!=this._ownerElement)throw new a(ue);return t=this.getNamedItemNS(e.namespaceURI,e.localName),c(this._ownerElement,this,e,t),t},removeNamedItem:function(e){var t=this.getNamedItem(e);return f(this._ownerElement,this,t),t},removeNamedItemNS:function(e,t){var n=this.getNamedItemNS(e,t);return f(this._ownerElement,this,n),n},getNamedItemNS:function(e,t){for(var n=this.length;n--;){var r=this[n];if(r.localName==t&&r.namespaceURI==e)return r}return null}},d.prototype={hasFeature:function(e,t){var n=this._features[e.toLowerCase()];return n&&(!t||t in n)?!0:!1},createDocument:function(e,t,n){var r=new v;if(r.implementation=this,r.childNodes=new o,r.doctype=n,n&&r.appendChild(n),t){var i=r.createElementNS(e,t);r.appendChild(i)}return r},createDocumentType:function(e,t,n){var r=new C;return r.name=e,r.nodeName=e,r.publicId=t,r.systemId=n,r}},h.prototype={firstChild:null,lastChild:null,previousSibling:null,nextSibling:null,attributes:null,parentNode:null,childNodes:null,ownerDocument:null,nodeValue:null,namespaceURI:null,prefix:null,localName:null,insertBefore:function(e,t){return S(this,e,t)},replaceChild:function(e,t){this.insertBefore(e,t),t&&this.removeChild(t)},removeChild:function(e){return T(this,e)},appendChild:function(e){return this.insertBefore(e,null)},hasChildNodes:function(){return null!=this.firstChild},cloneNode:function(e){return V(this.ownerDocument||this,this,e)},normalize:function(){for(var e=this.firstChild;e;){var t=e.nextSibling;t&&t.nodeType==G&&e.nodeType==G?(this.removeChild(t),e.appendData(t.data)):(e.normalize(),e=t)}},isSupported:function(e,t){return this.ownerDocument.implementation.hasFeature(e,t)},hasAttributes:function(){return this.attributes.length>0},lookupPrefix:function(e){for(var t=this;t;){var n=t._nsMap;if(n)for(var r in n)if(n[r]==e)return r;t=t.nodeType==$?t.ownerDocument:t.parentNode}return null},lookupNamespaceURI:function(e){for(var t=this;t;){var n=t._nsMap;if(n&&e in n)return n[e];t=t.nodeType==$?t.ownerDocument:t.parentNode}return null},isDefaultNamespace:function(e){var t=this.lookupPrefix(e);return null==t}},r(Y,h),r(Y,h.prototype),v.prototype={nodeName:"#document",nodeType:ee,doctype:null,documentElement:null,_inc:1,insertBefore:function(e,t){if(e.nodeType==ne){for(var n=e.firstChild;n;){var r=n.nextSibling;this.insertBefore(n,t),n=r}return e}return null==this.documentElement&&e.nodeType==H&&(this.documentElement=e),S(this,e,t),e.ownerDocument=this,e},removeChild:function(e){return this.documentElement==e&&(this.documentElement=null),T(this,e)},importNode:function(e,t){return K(this,e,t)},getElementById:function(e){var t=null;return y(this.documentElement,function(n){return n.nodeType==H&&n.getAttribute("id")==e?(t=n,!0):void 0}),t},createElement:function(e){var t=new _;t.ownerDocument=this,t.nodeName=e,t.tagName=e,t.childNodes=new o;var n=t.attributes=new l;return n._ownerElement=t,t},createDocumentFragment:function(){var e=new D;return e.ownerDocument=this,e.childNodes=new o,e},createTextNode:function(e){var t=new R;return t.ownerDocument=this,t.appendData(e),t},createComment:function(e){var t=new I;return t.ownerDocument=this,t.appendData(e),t},createCDATASection:function(e){var t=new M;return t.ownerDocument=this,t.appendData(e),t},createProcessingInstruction:function(e,t){var n=new x;return n.ownerDocument=this,n.tagName=n.target=e,n.nodeValue=n.data=t,n},createAttribute:function(e){var t=new N;return t.ownerDocument=this,t.name=e,t.nodeName=e,t.localName=e,t.specified=!0,t},createEntityReference:function(e){var t=new O;return t.ownerDocument=this,t.nodeName=e,t},createElementNS:function(e,t){var n=new _,r=t.split(":"),i=n.attributes=new l;return n.childNodes=new o,n.ownerDocument=this,n.nodeName=t,n.tagName=t,n.namespaceURI=e,2==r.length?(n.prefix=r[0],n.localName=r[1]):n.localName=t,i._ownerElement=n,n},createAttributeNS:function(e,t){var n=new N,r=t.split(":");return n.ownerDocument=this,n.nodeName=t,n.name=t,n.namespaceURI=e,n.specified=!0,2==r.length?(n.prefix=r[0],n.localName=r[1]):n.localName=t,n}},i(v,h),_.prototype={nodeType:H,hasAttribute:function(e){return null!=this.getAttributeNode(e)},getAttribute:function(e){var t=this.getAttributeNode(e);return t&&t.value||""},getAttributeNode:function(e){return this.attributes.getNamedItem(e)},setAttribute:function(e,t){var n=this.ownerDocument.createAttribute(e);n.value=n.nodeValue=""+t,this.setAttributeNode(n)},removeAttribute:function(e){var t=this.getAttributeNode(e);t&&this.removeAttributeNode(t)},appendChild:function(e){return e.nodeType===ne?this.insertBefore(e,null):b(this,e)},setAttributeNode:function(e){return this.attributes.setNamedItem(e)},setAttributeNodeNS:function(e){return this.attributes.setNamedItemNS(e)},removeAttributeNode:function(e){return this.attributes.removeNamedItem(e.nodeName)},removeAttributeNS:function(e,t){var n=this.getAttributeNodeNS(e,t);n&&this.removeAttributeNode(n)},hasAttributeNS:function(e,t){return null!=this.getAttributeNodeNS(e,t)},getAttributeNS:function(e,t){var n=this.getAttributeNodeNS(e,t);return n&&n.value||""},setAttributeNS:function(e,t,n){var r=this.ownerDocument.createAttributeNS(e,t);r.value=r.nodeValue=""+n,this.setAttributeNode(r)},getAttributeNodeNS:function(e,t){return this.attributes.getNamedItemNS(e,t)},getElementsByTagName:function(e){return new s(this,function(t){var n=[];return y(t,function(r){r===t||r.nodeType!=H||"*"!==e&&r.tagName!=e||n.push(r)}),n})},getElementsByTagNameNS:function(e,t){return new s(this,function(n){var r=[];return y(n,function(i){i===n||i.nodeType!==H||"*"!==e&&i.namespaceURI!==e||"*"!==t&&i.localName!=t||r.push(i)}),r})}},v.prototype.getElementsByTagName=_.prototype.getElementsByTagName,v.prototype.getElementsByTagNameNS=_.prototype.getElementsByTagNameNS,i(_,h),N.prototype.nodeType=$,i(N,h),w.prototype={data:"",substringData:function(e,t){return this.data.substring(e,e+t)},appendData:function(e){e=this.data+e,this.nodeValue=this.data=e,this.length=e.length},insertData:function(e,t){this.replaceData(e,0,t)},appendChild:function(e){throw new Error(ae[oe])},deleteData:function(e,t){this.replaceData(e,t,"")},replaceData:function(e,t,n){var r=this.data.substring(0,e),i=this.data.substring(e+t);n=r+n+i,this.nodeValue=this.data=n,this.length=n.length}},i(w,h),R.prototype={nodeName:"#text",nodeType:G,splitText:function(e){var t=this.data,n=t.substring(e);t=t.substring(0,e),this.data=this.nodeValue=t,this.length=t.length;var r=this.ownerDocument.createTextNode(n);return this.parentNode&&this.parentNode.insertBefore(r,this.nextSibling),r}},i(R,w),I.prototype={nodeName:"#comment",nodeType:Z},i(I,w),M.prototype={nodeName:"#cdata-section",nodeType:X},i(M,w),C.prototype.nodeType=te,i(C,h),L.prototype.nodeType=re,i(L,h),P.prototype.nodeType=J,i(P,h),O.prototype.nodeType=z,i(O,h),D.prototype.nodeName="#document-fragment",D.prototype.nodeType=ne,i(D,h),x.prototype.nodeType=Q,i(x,h),U.prototype.serializeToString=function(e,t,n){return k.call(e,t,n)},h.prototype.toString=k;try{Object.defineProperty&&(Object.defineProperty(s.prototype,"length",{get:function(){return u(this),this.$$length}}),Object.defineProperty(h.prototype,"textContent",{get:function(){return W(this)},set:function(e){switch(this.nodeType){case H:case ne:for(;this.firstChild;)this.removeChild(this.firstChild);(e||String(e))&&this.appendChild(this.ownerDocument.createTextNode(e));break;default:this.data=e,this.value=e,this.nodeValue=e}}}),j=function(e,t,n){e["$$"+t]=n})}catch(le){}t.DOMImplementation=d,t.XMLSerializer=U},function(e,t,n){"use strict";function r(e){return 10===e||13===e}function i(e){return 9===e||32===e}function a(e){return 9===e||32===e||10===e||13===e}function o(e){return 44===e||91===e||93===e||123===e||125===e}function s(e){var t;return e>=48&&57>=e?e-48:(t=32|e,t>=97&&102>=t?t-97+10:-1)}function u(e){return 120===e?2:117===e?4:85===e?8:0}function l(e){return e>=48&&57>=e?e-48:-1}function p(e){return 48===e?"\x00":97===e?"":98===e?"\b":116===e?"	":9===e?"	":110===e?"\n":118===e?"":102===e?"\f":114===e?"\r":101===e?"":32===e?" ":34===e?'"':47===e?"/":92===e?"\\":78===e?"":95===e?" ":76===e?"\u2028":80===e?"\u2029":""}function c(e){return 65535>=e?String.fromCharCode(e):String.fromCharCode((e-65536>>10)+55296,(e-65536&1023)+56320)}function f(e,t){return new j(t,new W(e.filename,e.input,e.position,e.line-1,e.position-e.lineStart))}function d(e,t,n){var r=A(e,t);if(r){var i=n+t;if(!e.errorMap[i]){var a=new W(e.filename,e.input,t,r.line-1,t-r.start),o=new j(n,a);e.errors.push(o)}}}function h(e,t){var n=f(e,t),r=n.message+n.mark.position;e.errorMap[r]||(e.errors.push(n),e.errorMap[r]=1);for(var i=e.position;;){if(e.position>=e.input.length-1)return;var a=e.input.charAt(e.position);if("\n"==a)return e.position--,void(e.position==i&&(e.position+=1));if("\r"==a)return e.position--,void(e.position==i&&(e.position+=1));e.position++}}function m(e,t){var n=f(e,t);e.onWarning&&e.onWarning.call(null,n)}function y(e,t,n,r){var i,a,o,s,u=e.result;if(-1==u.startPosition&&(u.startPosition=t),n>=t){if(s=e.input.slice(t,n),r)for(i=0,a=s.length;a>i;i+=1)o=s.charCodeAt(i),9===o||o>=32&&1114111>=o||h(e,"expected valid JSON character");u.value+=s,u.endPosition=n}}function v(e,t,n,r,i){if(null!=r){null===t&&(t={startPosition:r.startPosition,endPosition:i.endPosition,parent:null,errors:[],mappings:[],kind:K.Kind.MAP});var a=K.newMapping(r,i);return a.parent=t,r.parent=a,null!=i&&(i.parent=a),!e.ignoreDuplicateKeys&&t.mappings.forEach(function(t){t.key&&t.key.value===(a.key&&a.key.value)&&(d(e,a.key.startPosition,"duplicate key"),d(e,t.key.startPosition,"duplicate key"))}),t.mappings.push(a),t.endPosition=i?i.endPosition:r.endPosition+1,t}}function g(e){var t;t=e.input.charCodeAt(e.position),10===t?e.position++:13===t?(e.position++,10===e.input.charCodeAt(e.position)&&e.position++):h(e,"a line break is expected"),e.line+=1,e.lineStart=e.position,e.lines.push({start:e.lineStart,line:e.line})}function A(e,t){for(var n,r=0;r<e.lines.length&&!(e.lines[r].start>t);r++)n=e.lines[r];return n?n:{start:0,line:0}}function E(e,t,n){for(var a=0,o=e.input.charCodeAt(e.position);0!==o;){for(;i(o);)o=e.input.charCodeAt(++e.position);if(t&&35===o)do o=e.input.charCodeAt(++e.position);while(10!==o&&13!==o&&0!==o);if(!r(o))break;for(g(e),o=e.input.charCodeAt(e.position),a++,e.lineIndent=0;32===o;)e.lineIndent++,o=e.input.charCodeAt(++e.position)}return-1!==n&&0!==a&&e.lineIndent<n&&m(e,"deficient indentation"),a}function T(e){var t,n=e.position;return t=e.input.charCodeAt(n),45!==t&&46!==t||e.input.charCodeAt(n+1)!==t||e.input.charCodeAt(n+2)!==t||(n+=3,t=e.input.charCodeAt(n),0!==t&&!a(t))?!1:!0}function S(e,t,n){1===n?t.value+=" ":n>1&&(t.value+=V.repeat("\n",n-1))}function b(e,t,n){var s,u,l,p,c,f,d,h,m,v=e.kind,g=e.result,A=K.newScalar();if(A.plainScalar=!0,e.result=A,m=e.input.charCodeAt(e.position),a(m)||o(m)||35===m||38===m||42===m||33===m||124===m||62===m||39===m||34===m||37===m||64===m||96===m)return!1;if((63===m||45===m)&&(u=e.input.charCodeAt(e.position+1),a(u)||n&&o(u)))return!1;for(e.kind="scalar",l=p=e.position,c=!1;0!==m;){if(58===m){if(u=e.input.charCodeAt(e.position+1),a(u)||n&&o(u))break}else if(35===m){if(s=e.input.charCodeAt(e.position-1),a(s))break}else{if(e.position===e.lineStart&&T(e)||n&&o(m))break;if(r(m)){if(f=e.line,d=e.lineStart,h=e.lineIndent,E(e,!1,-1),e.lineIndent>=t){c=!0,m=e.input.charCodeAt(e.position);continue}e.position=p,e.line=f,e.lineStart=d,e.lineIndent=h;break}}if(c&&(y(e,l,p,!1),S(e,A,e.line-f),l=p=e.position,c=!1),i(m)||(p=e.position+1),m=e.input.charCodeAt(++e.position),e.position>=e.input.length)return!1}return y(e,l,p,!1),-1!=e.result.startPosition?(A.rawValue=e.input.substring(A.startPosition,A.endPosition),!0):(e.kind=v,e.result=g,!1)}function _(e,t){var n,i,a;if(n=e.input.charCodeAt(e.position),39!==n)return!1;var o=K.newScalar();for(o.singleQuoted=!0,e.kind="scalar",e.result=o,o.startPosition=e.position,e.position++,i=a=e.position;0!==(n=e.input.charCodeAt(e.position));)if(39===n){if(y(e,i,e.position,!0),n=e.input.charCodeAt(++e.position),o.endPosition=e.position,39!==n)return!0;i=a=e.position,e.position++}else r(n)?(y(e,i,a,!0),S(e,o,E(e,!1,t)),i=a=e.position):e.position===e.lineStart&&T(e)?h(e,"unexpected end of the document within a single quoted scalar"):(e.position++,a=e.position,o.endPosition=e.position);h(e,"unexpected end of the stream within a single quoted scalar")}function N(e,t){var n,i,a,o,l,p;if(p=e.input.charCodeAt(e.position),34!==p)return!1;e.kind="scalar";var f=K.newScalar();for(f.doubleQuoted=!0,e.result=f,f.startPosition=e.position,e.position++,n=i=e.position;0!==(p=e.input.charCodeAt(e.position));){if(34===p)return y(e,n,e.position,!0),e.position++,f.endPosition=e.position,f.rawValue=e.input.substring(f.startPosition,f.endPosition),!0;if(92===p){if(y(e,n,e.position,!0),p=e.input.charCodeAt(++e.position),r(p))E(e,!1,t);else if(256>p&&(e.allowAnyEscape?se[p]:ae[p]))f.value+=e.allowAnyEscape?ue[p]:oe[p],e.position++;else if((l=u(p))>0){for(a=l,o=0;a>0;a--)p=e.input.charCodeAt(++e.position),(l=s(p))>=0?o=(o<<4)+l:h(e,"expected hexadecimal character");f.value+=c(o),e.position++}else h(e,"unknown escape sequence");n=i=e.position}else r(p)?(y(e,n,i,!0),S(e,f,E(e,!1,t)),n=i=e.position):e.position===e.lineStart&&T(e)?h(e,"unexpected end of the document within a double quoted scalar"):(e.position++,i=e.position)}h(e,"unexpected end of the stream within a double quoted scalar")}function w(e,t){var n,r,i,o,s,u,l,p,c,f,d,m=!0,y=e.tag,g=e.anchor;if(d=e.input.charCodeAt(e.position),91===d)o=93,l=!1,r=K.newItems(),r.startPosition=e.position;else{if(123!==d)return!1;o=125,l=!0,r=K.newMap(),r.startPosition=e.position}for(null!==e.anchor&&(r.anchorId=e.anchor,e.anchorMap[e.anchor]=r),d=e.input.charCodeAt(++e.position);0!==d;){if(E(e,!0,t),d=e.input.charCodeAt(e.position),d===o)return e.position++,e.tag=y,e.anchor=g,e.kind=l?"mapping":"sequence",e.result=r,r.endPosition=e.position,!0;if(!m){var A=e.position;h(e,"missed comma between flow collection entries"),e.position=A+1}if(c=p=f=null,s=u=!1,63===d&&(i=e.input.charCodeAt(e.position+1),a(i)&&(s=u=!0,e.position++,E(e,!0,t))),n=e.line,O(e,t,$,!1,!0),c=e.tag,p=e.result,E(e,!0,t),d=e.input.charCodeAt(e.position),!u&&e.line!==n||58!==d||(s=!0,d=e.input.charCodeAt(++e.position),E(e,!0,t),O(e,t,$,!1,!0),f=e.result),l)v(e,r,c,p,f);else if(s){var T=v(e,null,c,p,f);T.parent=r,r.items.push(T)}else p.parent=r,r.items.push(p);r.endPosition=e.position+1,E(e,!0,t),d=e.input.charCodeAt(e.position),44===d?(m=!0,d=e.input.charCodeAt(++e.position)):m=!1}h(e,"unexpected end of the stream within a flow collection")}function R(e,t){var n,a,o,s,u=J,p=!1,c=t,f=0,d=!1;if(s=e.input.charCodeAt(e.position),124===s)a=!1;else{if(62!==s)return!1;a=!0}var m=K.newScalar();for(e.kind="scalar",e.result=m,m.startPosition=e.position;0!==s;)if(s=e.input.charCodeAt(++e.position),43===s||45===s)J===u?u=43===s?Z:Q:h(e,"repeat of a chomping mode identifier");else{if(!((o=l(s))>=0))break;0===o?h(e,"bad explicit indentation width of a block scalar; it cannot be less than one"):p?h(e,"repeat of an indentation width identifier"):(c=t+o-1,p=!0)}if(i(s)){do s=e.input.charCodeAt(++e.position);while(i(s));if(35===s)do s=e.input.charCodeAt(++e.position);while(!r(s)&&0!==s)}for(;0!==s;){for(g(e),e.lineIndent=0,s=e.input.charCodeAt(e.position);(!p||e.lineIndent<c)&&32===s;)e.lineIndent++,s=e.input.charCodeAt(++e.position);if(!p&&e.lineIndent>c&&(c=e.lineIndent),r(s))f++;else{if(e.lineIndent<c){u===Z?m.value+=V.repeat("\n",f):u===J&&p&&(m.value+="\n");break}for(a?i(s)?(d=!0,m.value+=V.repeat("\n",f+1)):d?(d=!1,m.value+=V.repeat("\n",f+1)):0===f?p&&(m.value+=" "):m.value+=V.repeat("\n",f):p&&(m.value+=V.repeat("\n",f+1)),p=!0,f=0,n=e.position;!r(s)&&0!==s;)s=e.input.charCodeAt(++e.position);y(e,n,e.position,!1)}}m.endPosition=e.position;for(var v=e.position-1,A=!1;;){var E=e.input[v];if("\r"==E||"\n"==E){A&&v--;break}if(" "!=E&&"	"!=E)break;v--}return m.endPosition=v,m.rawValue=e.input.substring(m.startPosition,m.endPosition),!0}function I(e,t){var n,r,i,o=e.tag,s=e.anchor,u=K.newItems(),l=!1;for(null!==e.anchor&&(u.anchorId=e.anchor,e.anchorMap[e.anchor]=u),u.startPosition=e.position,i=e.input.charCodeAt(e.position);0!==i&&45===i&&(r=e.input.charCodeAt(e.position+1),a(r));)if(l=!0,e.position++,E(e,!0,-1)&&e.lineIndent<=t)u.items.push(null),i=e.input.charCodeAt(e.position);else if(n=e.line,O(e,t,X,!1,!0),e.result.parent=u,u.items.push(e.result),E(e,!0,-1),i=e.input.charCodeAt(e.position),(e.line===n||e.lineIndent>t)&&0!==i)h(e,"bad indentation of a sequence entry");else if(e.lineIndent<t)break;return u.endPosition=e.position,l?(e.tag=o,e.anchor=s,e.kind="sequence",e.result=u,u.endPosition=e.position,!0):!1}function M(e,t,n){var o,s,u,l,p=e.tag,c=e.anchor,f=K.newMap(),d=null,m=null,y=null,g=!1,A=!1;for(f.startPosition=e.position,null!==e.anchor&&(f.anchorId=e.anchor,e.anchorMap[e.anchor]=f),l=e.input.charCodeAt(e.position);0!==l;){if(o=e.input.charCodeAt(e.position+1),u=e.line,63!==l&&58!==l||!a(o)){if(!O(e,n,G,!1,!0))break;if(e.line===u){for(l=e.input.charCodeAt(e.position);i(l);)l=e.input.charCodeAt(++e.position);if(58===l)l=e.input.charCodeAt(++e.position),a(l)||h(e,"a whitespace character is expected after the key-value separator within a block mapping"),g&&(v(e,f,d,m,null),d=m=y=null),A=!0,g=!1,s=!1,d=e.tag,m=e.result;else{if(!A)return e.tag=p,e.anchor=c,!0;h(e,"can not read an implicit mapping pair; a colon is missed")}}else{if(!A)return e.tag=p,e.anchor=c,!0;for(h(e,"can not read a block mapping entry; a multiline key may not be an implicit key");e.position>0;)if(l=e.input.charCodeAt(--e.position),r(l)){e.position++;break}}}else 63===l?(g&&(v(e,f,d,m,null),d=m=y=null),A=!0,g=!0,s=!0):g?(g=!1,s=!0):h(e,"incomplete explicit mapping pair; a key node is missed"),e.position+=1,l=o;if((e.line===u||e.lineIndent>t)&&(O(e,t,z,!0,s)&&(g?m=e.result:y=e.result),g||(v(e,f,d,m,y),d=m=y=null),E(e,!0,-1),l=e.input.charCodeAt(e.position)),e.lineIndent>t&&0!==l)h(e,"bad indentation of a mapping entry");else if(e.lineIndent<t)break}return g&&v(e,f,d,m,null),A&&(e.tag=p,e.anchor=c,e.kind="mapping",e.result=f),A}function C(e){var t,n,r,i,o=!1,s=!1;if(i=e.input.charCodeAt(e.position),33!==i)return!1;if(null!==e.tag&&h(e,"duplication of a tag property"),i=e.input.charCodeAt(++e.position),60===i?(o=!0,i=e.input.charCodeAt(++e.position)):33===i?(s=!0,n="!!",i=e.input.charCodeAt(++e.position)):n="!",t=e.position,o){do i=e.input.charCodeAt(++e.position);while(0!==i&&62!==i);e.position<e.length?(r=e.input.slice(t,e.position),i=e.input.charCodeAt(++e.position)):h(e,"unexpected end of the stream within a verbatim tag")}else{for(;0!==i&&!a(i);)33===i&&(s?h(e,"tag suffix cannot contain exclamation marks"):(n=e.input.slice(t-1,e.position+1),re.test(n)||h(e,"named tag handle cannot contain such characters"),s=!0,t=e.position+1)),i=e.input.charCodeAt(++e.position);r=e.input.slice(t,e.position),ne.test(r)&&h(e,"tag suffix cannot contain flow indicator characters")}return r&&!ie.test(r)&&h(e,"tag name cannot contain such characters: "+r),o?e.tag=r:H.call(e.tagMap,n)?e.tag=e.tagMap[n]+r:"!"===n?e.tag="!"+r:"!!"===n?e.tag="tag:yaml.org,2002:"+r:h(e,'undeclared tag handle "'+n+'"'),!0}function L(e){var t,n;if(n=e.input.charCodeAt(e.position),38!==n)return!1;for(null!==e.anchor&&h(e,"duplication of an anchor property"),n=e.input.charCodeAt(++e.position),t=e.position;0!==n&&!a(n)&&!o(n);)n=e.input.charCodeAt(++e.position);return e.position===t&&h(e,"name of an anchor node must contain at least one character"),e.anchor=e.input.slice(t,e.position),!0}function P(e){var t,n,r;e.length,e.input;if(r=e.input.charCodeAt(e.position),42!==r)return!1;for(r=e.input.charCodeAt(++e.position),t=e.position;0!==r&&!a(r)&&!o(r);)r=e.input.charCodeAt(++e.position);return e.position<=t&&(h(e,"name of an alias node must contain at least one character"),e.position=t+1),n=e.input.slice(t,e.position),e.anchorMap.hasOwnProperty(n)||(h(e,'unidentified alias "'+n+'"'),e.position<=t&&(e.position=t+1)),e.result=K.newAnchorRef(n,t,e.position,e.anchorMap[n]),E(e,!0,-1),!0}function O(e,t,n,r,i){var a,o,s,u,l,p,c,d,m=1,y=!1,v=!1;e.tag=null,e.anchor=null,e.kind=null,e.result=null,a=o=s=z===n||X===n,r&&E(e,!0,-1)&&(y=!0,e.lineIndent>t?m=1:e.lineIndent===t?m=0:e.lineIndent<t&&(m=-1));var g=e.position,A=e.position-e.lineStart;if(1===m)for(;C(e)||L(e);)E(e,!0,-1)?(y=!0,s=a,e.lineIndent>t?m=1:e.lineIndent===t?m=0:e.lineIndent<t&&(m=-1)):s=!1;if(s&&(s=y||i),(1===m||z===n)&&(c=$===n||G===n?t:t+1,d=e.position-e.lineStart,1===m?s&&(I(e,d)||M(e,d,c))||w(e,c)?v=!0:(o&&R(e,c)||_(e,c)||N(e,c)?v=!0:P(e)?(v=!0,(null!==e.tag||null!==e.anchor)&&h(e,"alias node should not have any properties")):b(e,c,$===n)&&(v=!0,null===e.tag&&(e.tag="?")),null!==e.anchor&&(e.anchorMap[e.anchor]=e.result,e.result.anchorId=e.anchor)):0===m&&(v=s&&I(e,d))),null!==e.tag&&"!"!==e.tag)if("!include"==e.tag)e.result||(e.result=K.newScalar(),e.result.startPosition=e.position,e.result.endPosition=e.position,h(e,"!include without value")),e.result.kind=K.Kind.INCLUDE_REF;else if("?"===e.tag)for(u=0,l=e.implicitTypes.length;l>u;u+=1){p=e.implicitTypes[u];var T=e.result.value;if(p.resolve(T)){e.result.valueObject=p.construct(e.result.value),e.tag=p.tag,null!==e.anchor&&(e.result.anchorId=e.anchor,e.anchorMap[e.anchor]=e.result);break}}else if(H.call(e.typeMap,e.tag))p=e.typeMap[e.tag],null!==e.result&&p.kind!==e.kind&&h(e,"unacceptable node kind for !<"+e.tag+'> tag; it should be "'+p.kind+'", not "'+e.kind+'"'),p.resolve(e.result)?(e.result=p.construct(e.result),null!==e.anchor&&(e.result.anchorId=e.anchor,e.anchorMap[e.anchor]=e.result)):h(e,"cannot resolve a node with !<"+e.tag+"> explicit tag");else{var S=f(e,"unknown tag <"+e.tag+">"),O=S.message+S.mark.position;e.errorMap[O]||(e.errors.push(S),e.errorMap[O]=1),S&&(S.mark.position=g,S.mark.column=A,S.mark.toLineEnd=!0)}return null!==e.tag||null!==e.anchor||v}function D(e){var t,n,o,s,u=e.position,l=!1;for(e.version=null,e.checkLineBreaks=e.legacy,e.tagMap={},e.anchorMap={};0!==(s=e.input.charCodeAt(e.position))&&(E(e,!0,-1),s=e.input.charCodeAt(e.position),!(e.lineIndent>0||37!==s));){for(l=!0,s=e.input.charCodeAt(++e.position),t=e.position;0!==s&&!a(s);)s=e.input.charCodeAt(++e.position);for(n=e.input.slice(t,e.position),o=[],n.length<1&&h(e,"directive name must not be less than one character in length");0!==s;){for(;i(s);)s=e.input.charCodeAt(++e.position);if(35===s){do s=e.input.charCodeAt(++e.position);while(0!==s&&!r(s));break}if(r(s))break;for(t=e.position;0!==s&&!a(s);)s=e.input.charCodeAt(++e.position);o.push(e.input.slice(t,e.position))}0!==s&&g(e),H.call(ce,n)?ce[n](e,n,o):(m(e,'unknown document directive "'+n+'"'),e.position++)}return E(e,!0,-1),0===e.lineIndent&&45===e.input.charCodeAt(e.position)&&45===e.input.charCodeAt(e.position+1)&&45===e.input.charCodeAt(e.position+2)?(e.position+=3,E(e,!0,-1)):l&&h(e,"directives end mark is expected"),O(e,e.lineIndent-1,z,!1,!0),E(e,!0,-1),e.checkLineBreaks&&te.test(e.input.slice(u,e.position))&&m(e,"non-ASCII line breaks are interpreted as content"),e.documents.push(e.result),e.position===e.lineStart&&T(e)?void(46===e.input.charCodeAt(e.position)&&(e.position+=3,E(e,!0,-1))):void(e.position<e.length-1&&h(e,"end of the stream or a document separator is expected"))}function x(e,t){e=String(e),t=t||{},0!==e.length&&(10!==e.charCodeAt(e.length-1)&&13!==e.charCodeAt(e.length-1)&&(e+="\n"),65279===e.charCodeAt(0)&&(e=e.slice(1)));var n=new pe(e,t);for(ee.test(n.input)&&h(n,"the stream contains non-printable characters"),n.input+="\x00";32===n.input.charCodeAt(n.position);)n.lineIndent+=1,n.position+=1;for(;n.position<n.length-1;){var r=n.position;if(D(n),n.position<=r)for(;n.position<n.length-1;n.position++){var i=n.input.charAt(n.position);if("\n"==i)break}}return n.documents.forEach(function(e){return e.errors=n.errors}),n.documents}function U(e,t,n){var r,i,a=x(e,n);for(r=0,i=a.length;i>r;r+=1)t(a[r])}function k(e,t){var n=x(e,t);if(0===n.length)return void 0;if(1===n.length){var r=n[0];return r.endPosition=e.length,r.startPosition>r.endPosition&&(r.startPosition=r.endPosition),r}var i=new j("expected a single document in the stream, but found more");return i.mark=new W("","",0,0,0),i.mark.position=n[0].endPosition,n[0].errors.push(i),n[0]}function F(e,t,n){U(e,t,V.extend({schema:q},n))}function B(e,t){return k(e,V.extend({schema:q},t))}for(var K=n(118),V=n(116),j=n(117),W=n(119),q=n(122),Y=n(121),H=Object.prototype.hasOwnProperty,$=1,G=2,X=3,z=4,J=1,Q=2,Z=3,ee=/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F-\x84\x86-\x9F\uD800-\uDFFF\uFFFE\uFFFF]/,te=/[\x85\u2028\u2029]/,ne=/[,\[\]\{\}]/,re=/^(?:!|!!|![a-z\-]+!)$/i,ie=/^(?:!|[^,\[\]\{\}])(?:%[0-9a-f]{2}|[0-9a-z\-#;\/\?:@&=\+\$,_\.!~\*'\(\)\[\]])*$/i,ae=new Array(256),oe=new Array(256),se=new Array(256),ue=new Array(256),le=0;256>le;le++)ue[le]=oe[le]=p(le),ae[le]=oe[le]?1:0,se[le]=1,ae[le]||(ue[le]="\\"+String.fromCharCode(le));var pe=function(){function e(e,t){this.errorMap={},this.errors=[],this.lines=[],this.input=e,this.filename=t.filename||null,this.schema=t.schema||Y,this.onWarning=t.onWarning||null,this.legacy=t.legacy||!1,this.allowAnyEscape=t.allowAnyEscape||!1,this.ignoreDuplicateKeys=t.ignoreDuplicateKeys||!1,this.implicitTypes=this.schema.compiledImplicit,this.typeMap=this.schema.compiledTypeMap,this.length=e.length,this.position=0,this.line=0,this.lineStart=0,this.lineIndent=0,this.documents=[]}return e}(),ce={YAML:function(e,t,n){var r,i,a;null!==e.version&&h(e,"duplication of %YAML directive"),1!==n.length&&h(e,"YAML directive accepts exactly one argument"),
r=/^([0-9]+)\.([0-9]+)$/.exec(n[0]),null===r&&h(e,"ill-formed argument of the YAML directive"),i=parseInt(r[1],10),a=parseInt(r[2],10),1!==i&&h(e,"found incompatible YAML document (version 1.2 is required)"),e.version=n[0],e.checkLineBreaks=2>a,2!==a&&h(e,"found incompatible YAML document (version 1.2 is required)")},TAG:function(e,t,n){var r,i;2!==n.length&&h(e,"TAG directive accepts exactly two arguments"),r=n[0],i=n[1],re.test(r)||h(e,"ill-formed tag handle (first argument) of the TAG directive"),H.call(e.tagMap,r)&&h(e,'there is a previously declared suffix for "'+r+'" tag handle'),ie.test(i)||h(e,"ill-formed tag prefix (second argument) of the TAG directive"),e.tagMap[r]=i}};(function(){function e(){}return e})();t.loadAll=U,t.load=k,t.safeLoadAll=F,t.safeLoad=B,e.exports.loadAll=U,e.exports.load=k,e.exports.safeLoadAll=F,e.exports.safeLoad=B},function(e,t,n){"use strict";function r(e,t){var n,r,i,a,o,s,u;if(null===t)return{};for(n={},r=Object.keys(t),i=0,a=r.length;a>i;i+=1)o=r[i],s=String(t[o]),"!!"===o.slice(0,2)&&(o="tag:yaml.org,2002:"+o.slice(2)),u=e.compiledTypeMap[o],u&&C.call(u.styleAliases,s)&&(s=u.styleAliases[s]),n[o]=s;return n}function i(e){var t,n,r;if(t=e.toString(16).toUpperCase(),255>=e)n="x",r=2;else if(65535>=e)n="u",r=4;else{if(!(4294967295>=e))throw new w("code point within a string may not be greater than 0xFFFFFFFF");n="U",r=8}return"\\"+n+N.repeat("0",r-t.length)+t}function a(e){this.schema=e.schema||R,this.indent=Math.max(1,e.indent||2),this.skipInvalid=e.skipInvalid||!1,this.flowLevel=N.isNothing(e.flowLevel)?-1:e.flowLevel,this.styleMap=r(this.schema,e.styles||null),this.implicitTypes=this.schema.compiledImplicit,this.explicitTypes=this.schema.compiledExplicit,this.tag=null,this.result="",this.duplicates=[],this.usedDuplicates=null}function o(e,t){for(var n,r=N.repeat(" ",t),i=0,a=-1,o="",s=e.length;s>i;)a=e.indexOf("\n",i),-1===a?(n=e.slice(i),i=s):(n=e.slice(i,a+1),i=a+1),n.length&&"\n"!==n&&(o+=r),o+=n;return o}function s(e,t){return"\n"+N.repeat(" ",e.indent*t)}function u(e,t){var n,r,i;for(n=0,r=e.implicitTypes.length;r>n;n+=1)if(i=e.implicitTypes[n],i.resolve(t))return!0;return!1}function l(e){this.source=e,this.result="",this.checkpoint=0}function p(e,t,n){var r,i,a,s,p,f,m,y,v,g,A,E,T,S,b,_,N,w,R,I,M;if(0===t.length)return void(e.dump="''");if(0==t.indexOf("!include"))return void(e.dump=""+t);if(0==t.indexOf("!$$$novalue"))return void(e.dump="");if(-1!==te.indexOf(t))return void(e.dump="'"+t+"'");for(r=!0,i=t.length?t.charCodeAt(0):0,a=D===i||D===t.charCodeAt(t.length-1),(W===i||H===i||$===i||z===i)&&(r=!1),a?(r=!1,s=!1,p=!1):(s=!0,p=!0),f=!0,m=new l(t),y=!1,v=0,g=0,A=e.indent*n,E=80,40>A?E-=A:E=40,S=0;S<t.length;S++){if(T=t.charCodeAt(S),r){if(d(T))continue;r=!1}f&&T===K&&(f=!1),b=ee[T],_=h(T),(b||_)&&(T!==P&&T!==U&&T!==K?(s=!1,p=!1):T===P&&(y=!0,f=!1,S>0&&(N=t.charCodeAt(S-1),N===D&&(p=!1,s=!1)),s&&(w=S-v,v=S,w>g&&(g=w))),T!==U&&(f=!1),m.takeUpTo(S),m.escapeChar())}if(r&&u(e,t)&&(r=!1),R="",(s||p)&&(I=0,t.charCodeAt(t.length-1)===P&&(I+=1,t.charCodeAt(t.length-2)===P&&(I+=1)),0===I?R="-":2===I&&(R="+")),p&&E>g&&(s=!1),y||(p=!1),r)e.dump=t;else if(f)e.dump="'"+t+"'";else if(s)M=c(t,E),e.dump=">"+R+"\n"+o(M,A);else if(p)R||(t=t.replace(/\n$/,"")),e.dump="|"+R+"\n"+o(t,A);else{if(!m)throw new Error("Failed to dump scalar value");m.finish(),e.dump='"'+m.result+'"'}}function c(e,t){var n,r="",i=0,a=e.length,o=/\n+$/.exec(e);for(o&&(a=o.index+1);a>i;)n=e.indexOf("\n",i),n>a||-1===n?(r&&(r+="\n\n"),r+=f(e.slice(i,a),t),i=a):(r&&(r+="\n\n"),r+=f(e.slice(i,n),t),i=n+1);return o&&"\n"!==o[0]&&(r+=o[0]),r}function f(e,t){if(""===e)return e;for(var n,r,i,a=/[^\s] [^\s]/g,o="",s=0,u=0,l=a.exec(e);l;)n=l.index,n-u>t&&(r=s!==u?s:n,o&&(o+="\n"),i=e.slice(u,r),o+=i,u=r+1),s=n+1,l=a.exec(e);return o&&(o+="\n"),o+=u!==s&&e.length-u>t?e.slice(u,s)+"\n"+e.slice(s+1):e.slice(u)}function d(e){return L!==e&&P!==e&&O!==e&&j!==e&&G!==e&&X!==e&&J!==e&&Z!==e&&k!==e&&B!==e&&V!==e&&x!==e&&Q!==e&&Y!==e&&K!==e&&U!==e&&F!==e&&q!==e&&!ee[e]&&!h(e)}function h(e){return!(e>=32&&126>=e||133===e||e>=160&&55295>=e||e>=57344&&65533>=e||e>=65536&&1114111>=e)}function m(e,t,n){var r,i,a="",o=e.tag;for(r=0,i=n.length;i>r;r+=1)E(e,t,n[r],!1,!1)&&(0!==r&&(a+=", "),a+=e.dump);e.tag=o,e.dump="["+a+"]"}function y(e,t,n,r){var i,a,o="",u=e.tag;for(i=0,a=n.length;a>i;i+=1)E(e,t+1,n[i],!0,!0)&&(r&&0===i||(o+=s(e,t)),o+="- "+e.dump);e.tag=u,e.dump=o||"[]"}function v(e,t,n){var r,i,a,o,s,u="",l=e.tag,p=Object.keys(n);for(r=0,i=p.length;i>r;r+=1)s="",0!==r&&(s+=", "),a=p[r],o=n[a],E(e,t,a,!1,!1)&&(e.dump.length>1024&&(s+="? "),s+=e.dump+": ",E(e,t,o,!1,!1)&&(s+=e.dump,u+=s));e.tag=l,e.dump="{"+u+"}"}function g(e,t,n,r){var i,a,o,u,l,p,c="",f=e.tag,d=Object.keys(n);for(i=0,a=d.length;a>i;i+=1)p="",r&&0===i||(p+=s(e,t)),o=d[i],u=n[o],E(e,t+1,o,!0,!0)&&(l=null!==e.tag&&"?"!==e.tag||e.dump&&e.dump.length>1024,l&&(p+=e.dump&&P===e.dump.charCodeAt(0)?"?":"? "),p+=e.dump,l&&(p+=s(e,t)),E(e,t+1,u,!0,l)&&(p+=e.dump&&P===e.dump.charCodeAt(0)?":":": ",p+=e.dump,c+=p));e.tag=f,e.dump=c||"{}"}function A(e,t,n){var r,i,a,o,s,u;for(i=n?e.explicitTypes:e.implicitTypes,a=0,o=i.length;o>a;a+=1)if(s=i[a],(s.instanceOf||s.predicate)&&(!s.instanceOf||"object"==typeof t&&t instanceof s.instanceOf)&&(!s.predicate||s.predicate(t))){if(e.tag=n?s.tag:"?",s.represent){if(u=e.styleMap[s.tag]||s.defaultStyle,"[object Function]"===M.call(s.represent))r=s.represent(t,u);else{if(!C.call(s.represent,u))throw new w("!<"+s.tag+'> tag resolver accepts not "'+u+'" style');r=s.represent[u](t,u)}e.dump=r}return!0}return!1}function E(e,t,n,r,i){e.tag=null,e.dump=n,A(e,n,!1)||A(e,n,!0);var a=M.call(e.dump);r&&(r=0>e.flowLevel||e.flowLevel>t),(null!==e.tag&&"?"!==e.tag||2!==e.indent&&t>0)&&(i=!1);var o,s,u="[object Object]"===a||"[object Array]"===a;if(u&&(o=e.duplicates.indexOf(n),s=-1!==o),s&&e.usedDuplicates[o])e.dump="*ref_"+o;else{if(u&&s&&!e.usedDuplicates[o]&&(e.usedDuplicates[o]=!0),"[object Object]"===a)r&&0!==Object.keys(e.dump).length?(g(e,t,e.dump,i),s&&(e.dump="&ref_"+o+(0===t?"\n":"")+e.dump)):(v(e,t,e.dump),s&&(e.dump="&ref_"+o+" "+e.dump));else if("[object Array]"===a)r&&0!==e.dump.length?(y(e,t,e.dump,i),s&&(e.dump="&ref_"+o+(0===t?"\n":"")+e.dump)):(m(e,t,e.dump),s&&(e.dump="&ref_"+o+" "+e.dump));else{if("[object String]"!==a){if(e.skipInvalid)return!1;throw new w("unacceptable kind of an object to dump "+a)}"?"!==e.tag&&p(e,e.dump,t)}null!==e.tag&&"?"!==e.tag&&(e.dump="!<"+e.tag+"> "+e.dump)}return!0}function T(e,t){var n,r,i=[],a=[];for(S(e,i,a),n=0,r=a.length;r>n;n+=1)t.duplicates.push(i[a[n]]);t.usedDuplicates=new Array(r)}function S(e,t,n){var r,i,a;M.call(e);if(null!==e&&"object"==typeof e)if(i=t.indexOf(e),-1!==i)-1===n.indexOf(i)&&n.push(i);else if(t.push(e),Array.isArray(e))for(i=0,a=e.length;a>i;i+=1)S(e[i],t,n);else for(r=Object.keys(e),i=0,a=r.length;a>i;i+=1)S(e[r[i]],t,n)}function b(e,t){t=t||{};var n=new a(t);return T(e,n),E(n,0,e,!0,!0)?n.dump+"\n":""}function _(e,t){return b(e,N.extend({schema:I},t))}var N=n(116),w=n(117),R=n(121),I=n(122),M=Object.prototype.toString,C=Object.prototype.hasOwnProperty,L=9,P=10,O=13,D=32,x=33,U=34,k=35,F=37,B=38,K=39,V=42,j=44,W=45,q=58,Y=62,H=63,$=64,G=91,X=93,z=96,J=123,Q=124,Z=125,ee={};ee[0]="\\0",ee[7]="\\a",ee[8]="\\b",ee[9]="\\t",ee[10]="\\n",ee[11]="\\v",ee[12]="\\f",ee[13]="\\r",ee[27]="\\e",ee[34]='\\"',ee[92]="\\\\",ee[133]="\\N",ee[160]="\\_",ee[8232]="\\L",ee[8233]="\\P";var te=["y","Y","yes","Yes","YES","on","On","ON","n","N","no","No","NO","off","Off","OFF"];l.prototype.takeUpTo=function(e){var t;if(e<this.checkpoint)throw t=new Error("position should be > checkpoint"),t.position=e,t.checkpoint=this.checkpoint,t;return this.result+=this.source.slice(this.checkpoint,e),this.checkpoint=e,this},l.prototype.escapeChar=function(){var e,t;return e=this.source.charCodeAt(this.checkpoint),t=ee[e]||i(e),this.result+=t,this.checkpoint+=1,this},l.prototype.finish=function(){this.source.length>this.checkpoint&&this.takeUpTo(this.source.length)},t.dump=b,t.safeDump=_},function(e,t,n){"use strict";function r(e,t){e=e.split("?");var n=e[0],r=(e[1]||"").split("#")[0],o=e[1]&&e[1].split("#").length>1?"#"+e[1].split("#")[1]:"",s=i(r);for(var u in t)s[u]=t[u];return r=a(s),""!==r&&(r="?"+r),n+r+o}var i=n(167).parse,a=n(167).stringify;e.exports=r},function(e,t,n){e.exports=[{classes:[{name:"GlobalSchema",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[{typeName:"Referencable",nameSpace:"",basicName:"Referencable",typeKind:0,typeArguments:[{typeName:"Sys.SchemaString",nameSpace:"Sys",basicName:"SchemaString",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/api.ts"}],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/api.ts"}],fields:[{name:"key",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.key",arguments:[]},{name:"MetaModel.description",arguments:["Name of the global schema, used to refer on schema content"]}],valueConstraint:null,optional:!1},{name:"value",type:{typeName:"Sys.SchemaString",nameSpace:"Sys",basicName:"SchemaString",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/api.ts"},annotations:[{name:"MetaModel.description",arguments:["Content of the schema"]},{name:"MetaModel.value",arguments:[]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.actuallyExports",arguments:["value"]},{name:"MetaModel.description",arguments:["Content of the schema"]}],"extends":[{typeName:"RAMLSimpleElement",nameSpace:"",basicName:"RAMLSimpleElement",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/api.ts"}],moduleName:"RAMLSpec",annotationOverridings:{}},{name:"Api",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"title",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.required",arguments:[]},{name:"MetaModel.description",arguments:["The title property is a short plain text description of the RESTful API. The value SHOULD be suitable for use as a title for the contained user documentation."]}],valueConstraint:null,optional:!1},{name:"version",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["If the RAML API definition is targeted to a specific API version, the API definition MUST contain a version property. The version property is OPTIONAL and should not be used if: The API itself is not versioned. The API definition does not change between versions. The API architect can decide whether a change to user documentation elements, but no change to the API's resources, constitutes a version change. The API architect MAY use any versioning scheme so long as version numbers retain the same format. For example, 'v3', 'v3.0', and 'V3' are all allowed, but are not considered to be equal."]}],valueConstraint:null,optional:!1},{name:"baseUri",type:{typeName:"Sys.FullUriTemplateString",nameSpace:"Sys",basicName:"FullUriTemplateString",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/api.ts"},annotations:[{name:"MetaModel.description",arguments:["(Optional during development; Required after implementation) A RESTful API's resources are defined relative to the API's base URI. The use of the baseUri field is OPTIONAL to allow describing APIs that have not yet been implemented. After the API is implemented (even a mock implementation) and can be accessed at a service endpoint, the API definition MUST contain a baseUri property. The baseUri property's value MUST conform to the URI specification RFC2396 or a Level 1 Template URI as defined in RFC6570. The baseUri property SHOULD only be used as a reference value."]}],valueConstraint:null,optional:!1},{name:"baseUriParameters",type:{base:{typeName:"Params.Parameter",nameSpace:"Params",basicName:"Parameter",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/api.ts"},typeKind:1},annotations:[{name:"MetaModel.setsContextValue",arguments:["location","Params.ParameterLocation.BURI"]},{name:"MetaModel.description",arguments:["Base uri parameters are named parameters which described template parameters in the base uri"]}],valueConstraint:null,optional:!1},{name:"uriParameters",type:{base:{typeName:"Params.Parameter",nameSpace:"Params",basicName:"Parameter",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/api.ts"},typeKind:1},annotations:[{name:"MetaModel.setsContextValue",arguments:["location","Params.ParameterLocation.BURI"]},{name:"MetaModel.description",arguments:["URI parameters can be further defined by using the uriParameters property. The use of uriParameters is OPTIONAL. The uriParameters property MUST be a map in which each key MUST be the name of the URI parameter as defined in the baseUri property. The uriParameters CANNOT contain a key named version because it is a reserved URI parameter name. The value of the uriParameters property is itself a map that specifies  the property's attributes as named parameters"]}],valueConstraint:null,optional:!1},{name:"protocols",type:{base:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},typeKind:1},annotations:[{name:"MetaModel.oneOf",arguments:[["HTTP","HTTPS"]]},{name:"MetaModel.description",arguments:["A RESTful API can be reached HTTP, HTTPS, or both. The protocols property MAY be used to specify the protocols that an API supports. If the protocols property is not specified, the protocol specified at the baseUri property is used. The protocols property MUST be an array of strings, of values `HTTP` and/or `HTTPS`."]}],valueConstraint:null,optional:!1},{name:"mediaType",type:{typeName:"Bodies.MimeType",nameSpace:"Bodies",basicName:"MimeType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/api.ts"},annotations:[{name:"MetaModel.oftenKeys",arguments:[["application/json","application/xml","application/x-www-form-urlencoded","multipart/formdata"]]},{name:"MetaModel.description",arguments:["(Optional) The media types returned by API responses, and expected from API requests that accept a body, MAY be defaulted by specifying the mediaType property. This property is specified at the root level of the API definition. The property's value MAY be a single string with a valid media type described in the specification."]},{name:"MetaModel.inherited",arguments:[]}],valueConstraint:null,optional:!1},{name:"schemas",type:{base:{typeName:"GlobalSchema",nameSpace:"",basicName:"GlobalSchema",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/api.ts"},typeKind:1},annotations:[{name:"MetaModel.embeddedInMaps",arguments:[]},{name:"MetaModel.description",arguments:["To better achieve consistency and simplicity, the API definition SHOULD include an OPTIONAL schemas property in the root section. The schemas property specifies collections of schemas that could be used anywhere in the API definition. The value of the schemas property is an array of maps; in each map, the keys are the schema name, and the values are schema definitions. The schema definitions MAY be included inline or by using the RAML !include user-defined data type."]}],valueConstraint:null,optional:!1},{name:"traits",type:{base:{typeName:"Methods.Trait",nameSpace:"Methods",basicName:"Trait",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/api.ts"},typeKind:1},annotations:[{name:"MetaModel.embeddedInMaps",arguments:[]},{name:"MetaModel.description",arguments:["Declarations of traits used in this API"]}],valueConstraint:null,optional:!1},{name:"securedBy",type:{base:{typeName:"Security.SecuritySchemeRef",nameSpace:"Security",basicName:"SecuritySchemeRef",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/api.ts"},typeKind:1},annotations:[{name:"MetaModel.allowNull",arguments:[]},{name:"MetaModel.description",arguments:["A list of the security schemes to apply to all methods, these must be defined in the securitySchemes declaration."]}],valueConstraint:null,optional:!1},{name:"securitySchemes",type:{base:{typeName:"Security.AbstractSecurityScheme",nameSpace:"Security",basicName:"AbstractSecurityScheme",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/api.ts"},typeKind:1},annotations:[{name:"MetaModel.embeddedInMaps",arguments:[]},{name:"MetaModel.description",arguments:["Security schemes that can be applied using securedBy"]}],valueConstraint:null,optional:!1},{name:"resourceTypes",type:{base:{typeName:"Resources.ResourceType",nameSpace:"Resources",basicName:"ResourceType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/api.ts"},typeKind:1},annotations:[{name:"MetaModel.embeddedInMaps",arguments:[]},{name:"MetaModel.description",arguments:["Declaration of resource types used in this API"]}],valueConstraint:null,optional:!1},{name:"resources",type:{base:{typeName:"Resources.Resource",nameSpace:"Resources",basicName:"Resource",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/api.ts"},typeKind:1},annotations:[{name:"MetaModel.newInstanceName",arguments:["New Resource"]},{name:"MetaModel.description",arguments:["Resources are identified by their relative URI, which MUST begin with a slash (/). A resource defined as a root-level property is called a top-level resource. Its property's key is the resource's URI relative to the baseUri. A resource defined as a child property of another resource is called a nested resource, and its property's key is its URI relative to its parent resource's URI. Every property whose key begins with a slash (/), and is either at the root of the API definition or is the child property of a resource property, is a resource property. The key of a resource, i.e. its relative URI, MAY consist of multiple URI path fragments separated by slashes; e.g. `/bom/items` may indicate the collection of items in a bill of materials as a single resource. However, if the individual URI path fragments are themselves resources, the API definition SHOULD use nested resources to describe this structure; e.g. if `/bom` is itself a resource then `/items` should be a nested resource of `/bom`, while `/bom/items` should not be used."]}],valueConstraint:null,optional:!1},{name:"documentation",type:{base:{typeName:"DocumentationItem",nameSpace:"",basicName:"DocumentationItem",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/api.ts"},typeKind:1},annotations:[{name:"MetaModel.description",arguments:["The API definition can include a variety of documents that serve as a user guides and reference documentation for the API. Such documents can clarify how the API works or provide business context. Documentation-generators MUST include all the sections in an API definition's documentation property in the documentation output, and they MUST preserve the order in which the documentation is declared. To add user documentation to the API, include the documentation property at the root of the API definition. The documentation property MUST be an array of documents. Each document MUST contain title and content attributes, both of which are REQUIRED. If the documentation property is specified, it MUST include at least one document. Documentation-generators MUST process the content field as if it was defined using Markdown."]}],valueConstraint:null,optional:!1},{name:"RAMLVersion",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.customHandling",arguments:[]},{name:"MetaModel.description",arguments:["Returns AST node of security scheme, this reference refers to, or null."]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[],"extends":[],moduleName:"RAMLSpec",annotationOverridings:{}},{name:"DocumentationItem",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"title",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["title of documentation section"]},{name:"MetaModel.required",arguments:[]}],valueConstraint:null,optional:!1},{name:"content",type:{typeName:"Sys.MarkdownString",nameSpace:"Sys",basicName:"MarkdownString",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/api.ts"},annotations:[{name:"MetaModel.description",arguments:["Content of documentation section"]},{name:"MetaModel.required",arguments:[]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[],"extends":[{typeName:"RAMLSimpleElement",nameSpace:"",basicName:"RAMLSimpleElement",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/api.ts"}],moduleName:"RAMLSpec",annotationOverridings:{}}],aliases:[],enumDeclarations:[],imports:{MetaModel:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/metamodel.ts",Sys:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts",Params:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/parameters.ts",Common:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/common.ts",Bodies:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/bodies.ts",Resources:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/resources.ts",Methods:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/methods.ts",Security:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"},name:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/api.ts"},{classes:[{name:"SpecPartMetaData",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!0,annotations:[],"extends":[],moduleName:null,annotationOverridings:{}}],aliases:[],enumDeclarations:[],imports:{},name:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/metamodel.ts"},{classes:[{name:"ValueType",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[],"extends":[],moduleName:null,annotationOverridings:{}},{name:"StringType",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.nameAtRuntime",arguments:["string"]}],"extends":[{typeName:"ValueType",nameSpace:"",basicName:"ValueType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"AnyType",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.nameAtRuntime",arguments:["any"]}],"extends":[{typeName:"ValueType",nameSpace:"",basicName:"ValueType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"NumberType",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.nameAtRuntime",arguments:["number"]}],"extends":[{typeName:"ValueType",nameSpace:"",basicName:"ValueType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"BooleanType",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.nameAtRuntime",arguments:["boolean"]}],"extends":[{typeName:"ValueType",nameSpace:"",basicName:"ValueType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"Referencable",methods:[],typeParameters:["T"],typeParameterConstraint:[null],"implements":[],fields:[],isInterface:!0,annotations:[],"extends":[],moduleName:null,annotationOverridings:{}},{name:"Reference",methods:[],typeParameters:["T"],typeParameterConstraint:[null],"implements":[],fields:[{name:"structuredValue",type:{typeName:"TypeInstance",nameSpace:"",basicName:"TypeInstance",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts"},annotations:[{name:"MetaModel.customHandling",arguments:[]},{name:"MetaModel.description",arguments:["Returns a structured object if the reference point to one."]}],valueConstraint:null,optional:!1},{name:"name",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.customHandling",arguments:[]},{name:"MetaModel.description",arguments:["Returns name of referenced object"]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[],"extends":[{typeName:"ValueType",nameSpace:"",basicName:"ValueType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"DeclaresDynamicType",methods:[],typeParameters:["T"],typeParameterConstraint:[null],"implements":[],fields:[],isInterface:!0,annotations:[],"extends":[{typeName:"Referencable",nameSpace:"",basicName:"Referencable",typeKind:0,typeArguments:[{typeName:"T",nameSpace:"",basicName:"T",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts"}],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"UriTemplate",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.description",arguments:["This type currently serves both for absolute and relative urls"]}],"extends":[{typeName:"StringType",nameSpace:"",basicName:"StringType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"RelativeUriString",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.description",arguments:["This  type describes relative uri templates"]}],"extends":[{typeName:"UriTemplate",nameSpace:"",basicName:"UriTemplate",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"FullUriTemplateString",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.description",arguments:["This  type describes absolute uri templates"]}],"extends":[{typeName:"UriTemplate",nameSpace:"",basicName:"UriTemplate",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"FixedUri",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.description",arguments:["This  type describes fixed uris"]}],"extends":[{typeName:"StringType",nameSpace:"",basicName:"StringType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"MarkdownString",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.innerType",arguments:["markdown"]},{name:"MetaModel.description",arguments:["Mardown string is a string which can contain markdown as an extension this markdown should support links with RAML Pointers since 1.0"]}],"extends":[{typeName:"StringType",nameSpace:"",basicName:"StringType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"SchemaString",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.description",arguments:["Schema at this moment only two subtypes are supported (json schema and xsd)"]}],"extends":[{typeName:"StringType",nameSpace:"",basicName:"StringType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"JSonSchemaString",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.functionalDescriminator",arguments:["this.mediaType&&this.mediaType.isJSON()"]},{name:"MetaModel.description",arguments:["JSON schema"]}],"extends":[{typeName:"SchemaString",nameSpace:"",basicName:"SchemaString",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"XMLSchemaString",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.innerType",arguments:["xml"]},{name:"MetaModel.description",arguments:["XSD schema"]}],"extends":[{typeName:"SchemaString",nameSpace:"",basicName:"SchemaString",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"ExampleString",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[],"extends":[{typeName:"StringType",nameSpace:"",basicName:"StringType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"StatusCodeString",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[],"extends":[{typeName:"StringType",nameSpace:"",basicName:"StringType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts"}],moduleName:null,annotationOverridings:{}
},{name:"JSONExample",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.functionalDescriminator",arguments:["this.mediaType.isJSON()"]}],"extends":[{typeName:"ExampleString",nameSpace:"",basicName:"ExampleString",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"XMLExample",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.functionalDescriminator",arguments:["this.mediaType.isXML()"]}],"extends":[{typeName:"ExampleString",nameSpace:"",basicName:"ExampleString",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"TypeInstance",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"properties",type:{base:{typeName:"TypeInstanceProperty",nameSpace:"",basicName:"TypeInstanceProperty",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts"},typeKind:1},annotations:[{name:"MetaModel.description",arguments:["Array of instance properties"]}],valueConstraint:null,optional:!1},{name:"isScalar",type:{typeName:"boolean",nameSpace:"",basicName:"boolean",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["Whether the type is scalar"]}],valueConstraint:null,optional:!1},{name:"value",type:{typeName:"any",nameSpace:"",basicName:"any",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["For instances of scalar types returns scalar value"]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.customHandling",arguments:[]}],"extends":[],moduleName:null,annotationOverridings:{}},{name:"TypeInstanceProperty",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"name",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["Property name"]}],valueConstraint:null,optional:!1},{name:"value",type:{typeName:"TypeInstance",nameSpace:"",basicName:"TypeInstance",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts"},annotations:[{name:"MetaModel.description",arguments:["Property value"]}],valueConstraint:null,optional:!1},{name:"values",type:{base:{typeName:"TypeInstance",nameSpace:"",basicName:"TypeInstance",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts"},typeKind:1},annotations:[{name:"MetaModel.description",arguments:["Array of values if property value is array"]}],valueConstraint:null,optional:!1},{name:"isArray",type:{typeName:"boolean",nameSpace:"",basicName:"boolean",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["Whether property has array as value"]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.customHandling",arguments:[]}],"extends":[],moduleName:null,annotationOverridings:{}}],aliases:[],enumDeclarations:[],imports:{MetaModel:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/metamodel.ts",Common:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/common.ts"},name:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts"},{classes:[{name:"RAMLSimpleElement",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[],"extends":[],moduleName:null,annotationOverridings:{}}],aliases:[],enumDeclarations:[],imports:{MetaModel:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/metamodel.ts",Sys:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts"},name:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/common.ts"},{classes:[{name:"Parameter",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"name",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.key",arguments:[]},{name:"MetaModel.description",arguments:["name of the parameter"]},{name:"MetaModel.extraMetaKey",arguments:["headers"]}],valueConstraint:null,optional:!1},{name:"displayName",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["An alternate, human-friendly name for the parameter"]}],valueConstraint:null,optional:!1},{name:"type",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.defaultValue",arguments:["string"]},{name:"MetaModel.descriminatingProperty",arguments:[]},{name:"MetaModel.description",arguments:["The type attribute specifies the primitive type of the parameter's resolved value. API clients MUST return/throw an error if the parameter's resolved value does not match the specified type. If type is not specified, it defaults to string."]},{name:"MetaModel.canBeDuplicator",arguments:[]}],valueConstraint:null,optional:!1},{name:"location",type:{typeName:"ParameterLocation",nameSpace:"",basicName:"ParameterLocation",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/parameters.ts"},annotations:[{name:"MetaModel.system",arguments:[]},{name:"MetaModel.description",arguments:["Location of the parameter (can not be edited by user)"]}],valueConstraint:null,optional:!1},{name:"required",type:{typeName:"boolean",nameSpace:"",basicName:"boolean",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["Set to true if parameter is required"]},{name:"MetaModel.defaultBooleanValue",arguments:[!0]}],valueConstraint:null,optional:!1},{name:"default",type:{typeName:"any",nameSpace:"",basicName:"any",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["The default attribute specifies the default value to use for the property if the property is omitted or its value is not specified. This SHOULD NOT be interpreted as a requirement for the client to send the default attribute's value if there is no other value to send. Instead, the default attribute's value is the value the server uses if the client does not send a value."]}],valueConstraint:null,optional:!1},{name:"example",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["(Optional) The example attribute shows an example value for the property. This can be used, e.g., by documentation generators to generate sample values for the property."]}],valueConstraint:null,optional:!1},{name:"repeat",type:{typeName:"boolean",nameSpace:"",basicName:"boolean",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["The repeat attribute specifies that the parameter can be repeated. If the parameter can be used multiple times, the repeat parameter value MUST be set to 'true'. Otherwise, the default value is 'false' and the parameter may not be repeated."]},{name:"MetaModel.defaultBooleanValue",arguments:[!1]}],valueConstraint:null,optional:!1},{name:"description",type:{typeName:"Sys.MarkdownString",nameSpace:"Sys",basicName:"MarkdownString",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/parameters.ts"},annotations:[{name:"MetaModel.description",arguments:["The description attribute describes the intended use or meaning of the $self. This value MAY be formatted using Markdown."]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[],"extends":[],moduleName:null,annotationOverridings:{}},{name:"StringTypeDeclaration",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"type",type:null,annotations:[],valueConstraint:{isCallConstraint:!1,value:"string"},optional:!1},{name:"pattern",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["(Optional, applicable only for parameters of type string) The pattern attribute is a regular expression that a parameter of type string MUST match. Regular expressions MUST follow the regular expression specification from ECMA 262/Perl 5. The pattern MAY be enclosed in double quotes for readability and clarity."]}],valueConstraint:null,optional:!1},{name:"enum",type:{base:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},typeKind:1},annotations:[{name:"MetaModel.description",arguments:["(Optional, applicable only for parameters of type string) The enum attribute provides an enumeration of the parameter's valid values. This MUST be an array. If the enum attribute is defined, API clients and servers MUST verify that a parameter's value matches a value in the enum array. If there is no matching value, the clients and servers MUST treat this as an error."]}],valueConstraint:null,optional:!1},{name:"minLength",type:{typeName:"number",nameSpace:"",basicName:"number",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["(Optional, applicable only for parameters of type string) The minLength attribute specifies the parameter value's minimum number of characters."]}],valueConstraint:null,optional:!1},{name:"maxLength",type:{typeName:"number",nameSpace:"",basicName:"number",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["(Optional, applicable only for parameters of type string) The maxLength attribute specifies the parameter value's maximum number of characters."]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.description",arguments:["Value must be a string"]}],"extends":[{typeName:"Parameter",nameSpace:"",basicName:"Parameter",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/parameters.ts"}],moduleName:null,annotationOverridings:{}},{name:"BooleanTypeDeclaration",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"type",type:null,annotations:[],valueConstraint:{isCallConstraint:!1,value:"boolean"},optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.description",arguments:["Value must be a boolean"]}],"extends":[{typeName:"Parameter",nameSpace:"",basicName:"Parameter",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/parameters.ts"}],moduleName:null,annotationOverridings:{}},{name:"NumberTypeDeclaration",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"type",type:null,annotations:[],valueConstraint:{isCallConstraint:!1,value:"number"},optional:!1},{name:"minimum",type:{typeName:"number",nameSpace:"",basicName:"number",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["(Optional, applicable only for parameters of type number or integer) The minimum attribute specifies the parameter's minimum value."]}],valueConstraint:null,optional:!1},{name:"maximum",type:{typeName:"number",nameSpace:"",basicName:"number",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["(Optional, applicable only for parameters of type number or integer) The maximum attribute specifies the parameter's maximum value."]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.description",arguments:["Value MUST be a number. Indicate floating point numbers as defined by YAML."]}],"extends":[{typeName:"Parameter",nameSpace:"",basicName:"Parameter",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/parameters.ts"}],moduleName:null,annotationOverridings:{}},{name:"IntegerTypeDeclaration",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"type",type:null,annotations:[],valueConstraint:{isCallConstraint:!1,value:"integer"},optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.description",arguments:["Value MUST be a integer."]}],"extends":[{typeName:"NumberTypeDeclaration",nameSpace:"",basicName:"NumberTypeDeclaration",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/parameters.ts"}],moduleName:null,annotationOverridings:{}},{name:"DateTypeDeclaration",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"type",type:null,annotations:[],valueConstraint:{isCallConstraint:!1,value:"date"},optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.description",arguments:["Value MUST be a string representation of a date as defined in RFC2616 Section 3.3. "]}],"extends":[{typeName:"Parameter",nameSpace:"",basicName:"Parameter",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/parameters.ts"}],moduleName:null,annotationOverridings:{}},{name:"FileTypeDeclaration",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"type",type:null,annotations:[],valueConstraint:{isCallConstraint:!1,value:"file"},optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.requireValue",arguments:["location","ParameterLocation.FORM"]},{name:"MetaModel.description",arguments:["(Applicable only to Form properties) Value is a file. Client generators SHOULD use this type to handle file uploads correctly."]}],"extends":[{typeName:"Parameter",nameSpace:"",basicName:"Parameter",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/parameters.ts"}],moduleName:null,annotationOverridings:{}}],aliases:[],enumDeclarations:[{name:"ParameterLocation",members:["QUERY","HEADERS","URI","FORM","BURI"]}],imports:{MetaModel:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/metamodel.ts",Sys:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts",Common:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/common.ts"},name:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/parameters.ts"},{classes:[{name:"MimeType",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.description",arguments:["This sub type of the string represents mime types"]}],"extends":[{typeName:"StringType",nameSpace:"",basicName:"StringType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/bodies.ts"}],moduleName:null,annotationOverridings:{}},{name:"BodyLike",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"name",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.key",arguments:[]},{name:"MetaModel.description",arguments:["Mime type of the request or response body"]},{name:"MetaModel.canInherit",arguments:["mediaType"]},{name:"MetaModel.oftenKeys",arguments:[["application/json","application/xml","application/x-www-form-urlencoded","multipart/form-data"]]}],valueConstraint:null,optional:!1},{name:"schema",type:{typeName:"Sys.SchemaString",nameSpace:"Sys",basicName:"SchemaString",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/bodies.ts"},annotations:[{name:"MetaModel.requireValue",arguments:["this.name.isForm()","false"]},{name:"MetaModel.description",arguments:["The structure of a request or response body MAY be further specified by the schema property under the appropriate media type. The schema key CANNOT be specified if a body's media type is application/x-www-form-urlencoded or multipart/form-data. All parsers of RAML MUST be able to interpret JSON Schema and XML Schema. Schema MAY be declared inline or in an external file. However, if the schema is sufficiently large so as to make it difficult for a person to read the API definition, or the schema is reused across multiple APIs or across multiple miles in the same API, the !include user-defined data type SHOULD be used instead of including the content inline. Alternatively, the value of the schema field MAY be the name of a schema specified in the root-level schemas property, or it MAY be declared in an external file and included by using the by using the RAML !include user-defined data type."]}],valueConstraint:null,optional:!1},{name:"example",type:{typeName:"Sys.ExampleString",nameSpace:"Sys",basicName:"ExampleString",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/bodies.ts"},annotations:[{name:"MetaModel.description",arguments:["Documentation generators MUST use body properties' example attributes to generate example invocations."]}],valueConstraint:null,optional:!1},{name:"formParameters",type:{base:{typeName:"Params.Parameter",nameSpace:"Params",basicName:"Parameter",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/bodies.ts"},typeKind:1},annotations:[{name:"MetaModel.setsContextValue",arguments:["location","Params.ParameterLocation.FORM"]},{name:"MetaModel.description",arguments:["Web forms REQUIRE special encoding and custom declaration. If the API's media type is either application/x-www-form-urlencoded or multipart/form-data, the formParameters property MUST specify the name-value pairs that the API is expecting. The formParameters property is a map in which the key is the name of the web form parameter, and the value is itself a map the specifies the web form parameter's attributes."]}],valueConstraint:null,optional:!1},{name:"schemaContent",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.customHandling",arguments:[]},{name:"MetaModel.description",arguments:["Returns schema content for the cases when schema is inlined, when schema is included, and when schema is a reference."]}],valueConstraint:null,optional:!1},{name:"description",type:{typeName:"Sys.MarkdownString",nameSpace:"Sys",basicName:"MarkdownString",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/bodies.ts"},annotations:[{name:"MetaModel.description",arguments:["Human readable description of the body"]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.canInherit",arguments:["mediaType"]}],"extends":[],moduleName:null,annotationOverridings:{}},{name:"XMLBody",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"schema",type:{typeName:"Sys.XMLSchemaString",nameSpace:"Sys",basicName:"XMLSchemaString",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/bodies.ts"},annotations:[{name:"MetaModel.description",arguments:["XSD Schema"]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.functionalDescriminator",arguments:["this.mime.isXML()"]},{name:"MetaModel.description",arguments:["Needed to set connection between xml related mime types and xsd schema"]}],"extends":[{typeName:"BodyLike",nameSpace:"",basicName:"BodyLike",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/bodies.ts"}],moduleName:null,annotationOverridings:{}},{name:"JSONBody",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"schema",type:{typeName:"Sys.JSonSchemaString",nameSpace:"Sys",basicName:"JSonSchemaString",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/bodies.ts"},annotations:[{name:"MetaModel.description",arguments:["JSON Schema"]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.functionalDescriminator",arguments:["this.mime.isJSON()"]},{name:"MetaModel.description",arguments:["Needed to set connection between json related mime types and json schema"]}],"extends":[{typeName:"BodyLike",nameSpace:"",basicName:"BodyLike",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/bodies.ts"}],moduleName:null,annotationOverridings:{}},{name:"Response",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"code",type:{typeName:"Sys.StatusCodeString",nameSpace:"Sys",basicName:"StatusCodeString",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/bodies.ts"},annotations:[{name:"MetaModel.key",arguments:[]},{name:"MetaModel.extraMetaKey",arguments:["statusCodes"]},{name:"MetaModel.description",arguments:["Responses MUST be a map of one or more HTTP status codes, where each status code itself is a map that describes that status code."]}],valueConstraint:null,optional:!1},{name:"headers",type:{base:{typeName:"Params.Parameter",nameSpace:"Params",basicName:"Parameter",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/bodies.ts"},typeKind:1},annotations:[{name:"MetaModel.setsContextValue",arguments:["location","Params.ParameterLocation.HEADERS"]},{name:"MetaModel.newInstanceName",arguments:["New Header"]},{name:"MetaModel.description",arguments:["An API's methods may support custom header values in responses. The custom, non-standard HTTP headers MUST be specified by the headers property. API's may include the the placeholder token {?} in a header name to indicate that any number of headers that conform to the specified format can be sent in responses. This is particularly useful for APIs that allow HTTP headers that conform to some naming convention to send arbitrary, custom data."]}],valueConstraint:null,optional:!1},{name:"body",type:{base:{typeName:"BodyLike",nameSpace:"",basicName:"BodyLike",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/bodies.ts"},typeKind:1},annotations:[{name:"MetaModel.newInstanceName",arguments:["New Body"]},{name:"MetaModel.description",arguments:["Each response MAY contain a body property, which conforms to the same structure as request body properties (see Body). Responses that can return more than one response code MAY therefore have multiple bodies defined. For APIs without a priori knowledge of the response types for their responses, `*/*` MAY be used to indicate that responses that do not matching other defined data types MUST be accepted. Processing applications MUST match the most descriptive media type first if `*/*` is used."]}],valueConstraint:null,optional:!1},{name:"description",type:{typeName:"Sys.MarkdownString",nameSpace:"Sys",basicName:"MarkdownString",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/bodies.ts"},annotations:[{name:"MetaModel.description",arguments:["The description attribute describes the intended use or meaning of the $self. This value MAY be formatted using Markdown."]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[],"extends":[],moduleName:null,annotationOverridings:{}}],aliases:[],enumDeclarations:[],imports:{MetaModel:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/metamodel.ts",Sys:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts",Params:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/parameters.ts",Common:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/common.ts"},name:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/bodies.ts"},{classes:[{name:"Resource",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"relativeUri",type:{typeName:"Sys.RelativeUriString",nameSpace:"Sys",basicName:"RelativeUriString",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/resources.ts"},annotations:[{name:"MetaModel.key",arguments:[]},{name:"MetaModel.startFrom",arguments:["/"]},{name:"MetaModel.description",arguments:["Relative URL of this resource from the parent resource"]}],valueConstraint:null,optional:!1},{name:"type",type:{typeName:"ResourceTypeRef",nameSpace:"",basicName:"ResourceTypeRef",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/resources.ts"},annotations:[{name:"MetaModel.description",arguments:["Instantiation of applyed resource type"]}],valueConstraint:null,optional:!1},{name:"is",type:{base:{typeName:"TraitRef",nameSpace:"",basicName:"TraitRef",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/resources.ts"},typeKind:1},annotations:[{name:"MetaModel.description",arguments:["Instantiation of applyed traits"]}],valueConstraint:null,optional:!1},{name:"securedBy",type:{base:{typeName:"SecuritySchemeRef",nameSpace:"",basicName:"SecuritySchemeRef",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/resources.ts"},typeKind:1},annotations:[{name:"MetaModel.allowNull",arguments:[]},{name:"MetaModel.description",arguments:["securityScheme may also be applied to a resource by using the securedBy key, which is equivalent to applying the securityScheme to all methods that may be declared, explicitly or implicitly, by defining the resourceTypes or traits property for that resource. To indicate that the method may be called without applying any securityScheme, the method may be annotated with the null securityScheme."]}],valueConstraint:null,optional:!1},{name:"uriParameters",type:{base:{typeName:"Params.Parameter",nameSpace:"Params",basicName:"Parameter",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/resources.ts"},typeKind:1},annotations:[{name:"MetaModel.setsContextValue",arguments:["fieldOrParam",!0]},{name:"MetaModel.setsContextValue",arguments:["location","Params.ParameterLocation.URI"]},{name:"MetaModel.description",arguments:["Uri parameters of this resource"]}],valueConstraint:null,optional:!1},{name:"methods",type:{base:{typeName:"Methods.Method",nameSpace:"Methods",basicName:"Method",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/resources.ts"},typeKind:1},annotations:[{name:"MetaModel.newInstanceName",arguments:["New Method"]},{name:"MetaModel.description",arguments:["Methods that can be called on this resource"]}],valueConstraint:null,optional:!1},{name:"resources",type:{base:{typeName:"Resource",nameSpace:"",basicName:"Resource",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/resources.ts"},typeKind:1},annotations:[{name:"MetaModel.newInstanceName",arguments:["New Resource"]},{name:"MetaModel.description",arguments:["Children resources"]}],valueConstraint:null,optional:!1},{name:"displayName",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["An alternate, human-friendly name for the resource"]}],valueConstraint:null,optional:!1},{name:"baseUriParameters",type:{base:{typeName:"Params.Parameter",nameSpace:"Params",basicName:"Parameter",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/resources.ts"},typeKind:1},annotations:[{name:"MetaModel.setsContextValue",arguments:["fieldOrParam",!0]},{name:"MetaModel.setsContextValue",arguments:["location","Params.ParameterLocation.BURI"]},{name:"MetaModel.description",arguments:["A resource or a method can override a base URI template's values. This is useful to restrict or change the default or parameter selection in the base URI. The baseUriParameters property MAY be used to override any or all parameters defined at the root level baseUriParameters property, as well as base URI parameters not specified at the root level."]}],valueConstraint:null,optional:!1},{name:"description",type:{typeName:"Sys.MarkdownString",nameSpace:"Sys",basicName:"MarkdownString",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/resources.ts"},annotations:[{name:"MetaModel.description",arguments:["The description attribute describes the intended use or meaning of the $self. This value MAY be formatted using Markdown."]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[],"extends":[],moduleName:null,annotationOverridings:{}},{name:"ResourceTypeRef",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"resourceType",type:{typeName:"ResourceType",nameSpace:"",basicName:"ResourceType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/resources.ts"},annotations:[{name:"MetaModel.customHandling",arguments:[]},{name:"MetaModel.description",arguments:["Returns referenced resource type"]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[],"extends":[{typeName:"Reference",nameSpace:"",basicName:"Reference",typeKind:0,typeArguments:[{typeName:"ResourceType",nameSpace:"",basicName:"ResourceType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/resources.ts"}],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/resources.ts"}],moduleName:null,annotationOverridings:{}},{name:"ResourceType",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[{typeName:"DeclaresDynamicType",nameSpace:"",basicName:"DeclaresDynamicType",typeKind:0,typeArguments:[{typeName:"ResourceType",nameSpace:"",basicName:"ResourceType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/resources.ts"}],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/resources.ts"}],fields:[{name:"name",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.key",
arguments:[]},{name:"MetaModel.description",arguments:["Name of the resource type"]}],valueConstraint:null,optional:!1},{name:"usage",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["Instructions on how and when the resource type should be used."]}],valueConstraint:null,optional:!1},{name:"methods",type:{base:{typeName:"Methods.Method",nameSpace:"Methods",basicName:"Method",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/resources.ts"},typeKind:1},annotations:[{name:"MetaModel.description",arguments:["Methods that are part of this resource type definition"]}],valueConstraint:null,optional:!1},{name:"is",type:{base:{typeName:"Security.TraitRef",nameSpace:"Security",basicName:"TraitRef",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/resources.ts"},typeKind:1},annotations:[{name:"MetaModel.description",arguments:["Instantiation of applyed traits"]}],valueConstraint:null,optional:!1},{name:"type",type:{typeName:"ResourceTypeRef",nameSpace:"",basicName:"ResourceTypeRef",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/resources.ts"},annotations:[{name:"MetaModel.description",arguments:["Instantiation of applyed resource type"]}],valueConstraint:null,optional:!1},{name:"securedBy",type:{base:{typeName:"Security.SecuritySchemeRef",nameSpace:"Security",basicName:"SecuritySchemeRef",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/resources.ts"},typeKind:1},annotations:[{name:"MetaModel.allowNull",arguments:[]},{name:"MetaModel.description",arguments:["securityScheme may also be applied to a resource by using the securedBy key, which is equivalent to applying the securityScheme to all methods that may be declared, explicitly or implicitly, by defining the resourceTypes or traits property for that resource. To indicate that the method may be called without applying any securityScheme, the method may be annotated with the null securityScheme."]}],valueConstraint:null,optional:!1},{name:"uriParameters",type:{base:{typeName:"Params.Parameter",nameSpace:"Params",basicName:"Parameter",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/resources.ts"},typeKind:1},annotations:[{name:"MetaModel.setsContextValue",arguments:["location","Params.ParameterLocation.URI"]},{name:"MetaModel.description",arguments:["Uri parameters of this resource"]}],valueConstraint:null,optional:!1},{name:"displayName",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["An alternate, human-friendly name for the resource type"]}],valueConstraint:null,optional:!1},{name:"baseUriParameters",type:{base:{typeName:"Params.Parameter",nameSpace:"Params",basicName:"Parameter",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/resources.ts"},typeKind:1},annotations:[{name:"MetaModel.setsContextValue",arguments:["fieldOrParam",!0]},{name:"MetaModel.setsContextValue",arguments:["location","Params.ParameterLocation.BURI"]},{name:"MetaModel.description",arguments:["A resource or a method can override a base URI template's values. This is useful to restrict or change the default or parameter selection in the base URI. The baseUriParameters property MAY be used to override any or all parameters defined at the root level baseUriParameters property, as well as base URI parameters not specified at the root level."]}],valueConstraint:null,optional:!1},{name:"parametrizedProperties",type:{typeName:"Sys.TypeInstance",nameSpace:"Sys",basicName:"TypeInstance",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/resources.ts"},annotations:[{name:"MetaModel.customHandling",arguments:[]},{name:"MetaModel.description",arguments:["Returns object representation of parametrized properties of the resource type"]}],valueConstraint:null,optional:!1},{name:"description",type:{typeName:"Sys.MarkdownString",nameSpace:"Sys",basicName:"MarkdownString",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/resources.ts"},annotations:[{name:"MetaModel.description",arguments:["The description attribute describes the intended use or meaning of the $self. This value MAY be formatted using Markdown."]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.inlinedTemplates",arguments:[]},{name:"MetaModel.allowQuestion",arguments:[]}],"extends":[],moduleName:null,annotationOverridings:{}}],aliases:[],enumDeclarations:[],imports:{MetaModel:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/metamodel.ts",Sys:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts",Params:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/parameters.ts",Bodies:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/bodies.ts",Common:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/common.ts",Methods:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/methods.ts",Security:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"},name:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/resources.ts"},{classes:[{name:"MethodBase",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"responses",type:{base:{typeName:"Bodies.Response",nameSpace:"Bodies",basicName:"Response",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/methods.ts"},typeKind:1},annotations:[{name:"MetaModel.newInstanceName",arguments:["New Response"]},{name:"MetaModel.description",arguments:["Resource methods MAY have one or more responses. Responses MAY be described using the description property, and MAY include example attributes or schema properties."]}],valueConstraint:null,optional:!1},{name:"body",type:{base:{typeName:"Bodies.BodyLike",nameSpace:"Bodies",basicName:"BodyLike",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/methods.ts"},typeKind:1},annotations:[{name:"MetaModel.newInstanceName",arguments:["New Body"]},{name:"MetaModel.description",arguments:["Some method verbs expect the resource to be sent as a request body. For example, to create a resource, the request must include the details of the resource to create. Resources CAN have alternate representations. For example, an API might support both JSON and XML representations. A method's body is defined in the body property as a hashmap, in which the key MUST be a valid media type."]}],valueConstraint:null,optional:!1},{name:"protocols",type:{base:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},typeKind:1},annotations:[{name:"MetaModel.oneOf",arguments:[["HTTP","HTTPS"]]},{name:"MetaModel.description",arguments:["A method can override an API's protocols value for that single method by setting a different value for the fields."]}],valueConstraint:null,optional:!1},{name:"securedBy",type:{base:{typeName:"SecuritySchemeRef",nameSpace:"",basicName:"SecuritySchemeRef",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/methods.ts"},typeKind:1},annotations:[{name:"MetaModel.allowNull",arguments:[]},{name:"MetaModel.description",arguments:["A list of the security schemas to apply, these must be defined in the securitySchemes declaration. To indicate that the method may be called without applying any securityScheme, the method may be annotated with the null securityScheme. Security schemas may also be applied to a resource with securedBy, which is equivalent to applying the security schemas to all methods that may be declared, explicitly or implicitly, by defining the resourceTypes or traits property for that resource."]}],valueConstraint:null,optional:!1},{name:"baseUriParameters",type:{base:{typeName:"Params.Parameter",nameSpace:"Params",basicName:"Parameter",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/methods.ts"},typeKind:1},annotations:[{name:"MetaModel.setsContextValue",arguments:["fieldOrParam",!0]},{name:"MetaModel.setsContextValue",arguments:["location","Params.ParameterLocation.BURI"]},{name:"MetaModel.description",arguments:["A resource or a method can override a base URI template's values. This is useful to restrict or change the default or parameter selection in the base URI. The baseUriParameters property MAY be used to override any or all parameters defined at the root level baseUriParameters property, as well as base URI parameters not specified at the root level."]}],valueConstraint:null,optional:!1},{name:"queryParameters",type:{base:{typeName:"Params.Parameter",nameSpace:"Params",basicName:"Parameter",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/methods.ts"},typeKind:1},annotations:[{name:"MetaModel.setsContextValue",arguments:["location","ParameterLocation.QUERY"]},{name:"MetaModel.newInstanceName",arguments:["New query parameter"]},{name:"MetaModel.description",arguments:["An APIs resources MAY be filtered (to return a subset of results) or altered (such as transforming a response body from JSON to XML format) by the use of query strings. If the resource or its method supports a query string, the query string MUST be defined by the queryParameters property"]}],valueConstraint:null,optional:!1},{name:"headers",type:{base:{typeName:"Params.Parameter",nameSpace:"Params",basicName:"Parameter",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/methods.ts"},typeKind:1},annotations:[{name:"MetaModel.setsContextValue",arguments:["location","ParameterLocation.HEADERS"]},{name:"MetaModel.description",arguments:["Headers that allowed at this position"]},{name:"MetaModel.newInstanceName",arguments:["New Header"]}],valueConstraint:null,optional:!1},{name:"description",type:{typeName:"Sys.MarkdownString",nameSpace:"Sys",basicName:"MarkdownString",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/methods.ts"},annotations:[],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.description",arguments:["Method object allows description of http methods"]}],"extends":[],moduleName:null,annotationOverridings:{}},{name:"Method",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"method",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.key",arguments:[]},{name:"MetaModel.extraMetaKey",arguments:["methods"]},{name:"MetaModel.oneOf",arguments:[["get","put","post","delete","patch","options","head","trace","connect"]]},{name:"MetaModel.description",arguments:["Method that can be called"]}],valueConstraint:null,optional:!1},{name:"is",type:{base:{typeName:"TraitRef",nameSpace:"",basicName:"TraitRef",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/methods.ts"},typeKind:1},annotations:[{name:"MetaModel.description",arguments:["Instantiation of applyed traits"]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[],"extends":[{typeName:"MethodBase",nameSpace:"",basicName:"MethodBase",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/methods.ts"}],moduleName:null,annotationOverridings:{}},{name:"Trait",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[{typeName:"DeclaresDynamicType",nameSpace:"",basicName:"DeclaresDynamicType",typeKind:0,typeArguments:[{typeName:"Trait",nameSpace:"",basicName:"Trait",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/methods.ts"}],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/methods.ts"}],fields:[{name:"name",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.key",arguments:[]},{name:"MetaModel.description",arguments:["Name of the trait"]}],valueConstraint:null,optional:!1},{name:"usage",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["Instructions on how and when the trait should be used."]}],valueConstraint:null,optional:!1},{name:"parametrizedProperties",type:{typeName:"Sys.TypeInstance",nameSpace:"Sys",basicName:"TypeInstance",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/methods.ts"},annotations:[{name:"MetaModel.customHandling",arguments:[]},{name:"MetaModel.description",arguments:["Returns object representation of parametrized properties of the trait"]}],valueConstraint:null,optional:!1},{name:"displayName",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["An alternate, human-friendly name for the trait"]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.inlinedTemplates",arguments:[]},{name:"MetaModel.allowQuestion",arguments:[]}],"extends":[{typeName:"MethodBase",nameSpace:"",basicName:"MethodBase",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/methods.ts"}],moduleName:null,annotationOverridings:{}},{name:"TraitRef",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"trait",type:{typeName:"Trait",nameSpace:"",basicName:"Trait",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/methods.ts"},annotations:[{name:"MetaModel.customHandling",arguments:[]},{name:"MetaModel.description",arguments:["Returns referenced trait"]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[],"extends":[{typeName:"Reference",nameSpace:"",basicName:"Reference",typeKind:0,typeArguments:[{typeName:"Trait",nameSpace:"",basicName:"Trait",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/methods.ts"}],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/methods.ts"}],moduleName:null,annotationOverridings:{}}],aliases:[],enumDeclarations:[],imports:{MetaModel:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/metamodel.ts",Sys:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts",Params:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/parameters.ts",Bodies:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/bodies.ts",Common:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/common.ts",Security:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"},name:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/methods.ts"},{classes:[{name:"SecuritySchemePart",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"displayName",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["An alternate, human-friendly name for the security scheme part"]}],valueConstraint:null,optional:!1},{name:"is",type:{base:{typeName:"TraitRef",nameSpace:"",basicName:"TraitRef",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"},typeKind:1},annotations:[{name:"MetaModel.description",arguments:["Instantiation of applyed traits"]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[],"extends":[{typeName:"MethodBase",nameSpace:"",basicName:"MethodBase",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"}],moduleName:null,annotationOverridings:{headers:[{name:"MetaModel.markdownDescription",arguments:["Optional array of headers, documenting the possible headers that could be accepted."]},{name:"MetaModel.valueDescription",arguments:["Object whose property names are the request header names and whose values describe the values."]}],queryParameters:[{name:"MetaModel.markdownDescription",arguments:["Query parameters, used by the schema in order to authorize the request. Mutually exclusive with queryString."]},{name:"MetaModel.valueDescription",arguments:["Object whose property names are the query parameter names and whose values describe the values."]}],queryString:[{name:"MetaModel.description",arguments:["Specifies the query string, used by the schema in order to authorize the request. Mutually exclusive with queryParameters."]},{name:"MetaModel.valueDescription",arguments:["Type name or type declaration"]}],responses:[{name:"MetaModel.description",arguments:["Optional array of responses, describing the possible responses that could be sent."]}],description:[{name:"MetaModel.description",arguments:["A longer, human-friendly description of the security scheme part"]},{name:"MetaModel.valueDescription",arguments:["Markdown string"]}]}},{name:"SecuritySchemeSettings",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.allowAny",arguments:[]}],"extends":[],moduleName:null,annotationOverridings:{}},{name:"AbstractSecurityScheme",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[{typeName:"Referencable",nameSpace:"",basicName:"Referencable",typeKind:0,typeArguments:[{typeName:"AbstractSecurityScheme",nameSpace:"",basicName:"AbstractSecurityScheme",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"}],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"}],fields:[{name:"name",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.key",arguments:[]},{name:"MetaModel.startFrom",arguments:[""]},{name:"MetaModel.hide",arguments:[]},{name:"MetaModel.description",arguments:["Name of the security scheme"]}],valueConstraint:null,optional:!1},{name:"type",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.required",arguments:[]},{name:"MetaModel.oneOf",arguments:[["OAuth 1.0","OAuth 2.0","Basic Authentication","DigestSecurityScheme Authentication","x-{other}"]]},{name:"MetaModel.descriminatingProperty",arguments:[]},{name:"MetaModel.description",arguments:["The securitySchemes property MUST be used to specify an API's security mechanisms, including the required settings and the authentication methods that the API supports. one authentication method is allowed if the API supports them."]},{name:"MetaModel.valueDescription",arguments:["string<br><br>The value MUST be one of<br>* OAuth 1.0,<br>* OAuth 2.0, <br>* BasicSecurityScheme Authentication<br>* DigestSecurityScheme Authentication<br>* x-&lt;other&gt;"]}],valueConstraint:null,optional:!1},{name:"description",type:{typeName:"Sys.MarkdownString",nameSpace:"Sys",basicName:"MarkdownString",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"},annotations:[{name:"MetaModel.description",arguments:["The description attribute MAY be used to describe a security schemes property."]}],valueConstraint:null,optional:!1},{name:"describedBy",type:{typeName:"SecuritySchemePart",nameSpace:"",basicName:"SecuritySchemePart",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"},annotations:[{name:"MetaModel.description",arguments:["A description of the request components related to Security that are determined by the scheme: the headers, query parameters or responses. As a best practice, even for standard security schemes, API designers SHOULD describe these properties of security schemes. Including the security scheme description completes an API documentation."]}],valueConstraint:null,optional:!1},{name:"settings",type:{typeName:"SecuritySchemeSettings",nameSpace:"",basicName:"SecuritySchemeSettings",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"},annotations:[{name:"MetaModel.description",arguments:["The settings attribute MAY be used to provide security scheme-specific information. The required attributes vary depending on the type of security scheme is being declared. It describes the minimum set of properties which any processing application MUST provide and validate if it chooses to implement the security scheme. Processing applications MAY choose to recognize other properties for things such as token lifetime, preferred cryptographic algorithms, and more."]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.description",arguments:["Declares globally referable security schema definition"]},{name:"MetaModel.actuallyExports",arguments:["$self"]},{name:"MetaModel.referenceIs",arguments:["settings"]}],"extends":[],moduleName:null,annotationOverridings:{}},{name:"SecuritySchemeRef",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"securitySchemeName",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.customHandling",arguments:[]},{name:"MetaModel.description",arguments:["Returns the name of security scheme, this reference refers to."]}],valueConstraint:null,optional:!1},{name:"securityScheme",type:{typeName:"AbstractSecurityScheme",nameSpace:"",basicName:"AbstractSecurityScheme",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"},annotations:[{name:"MetaModel.customHandling",arguments:[]},{name:"MetaModel.description",arguments:["Returns AST node of security scheme, this reference refers to, or null."]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.allowAny",arguments:[]}],"extends":[{typeName:"Reference",nameSpace:"",basicName:"Reference",typeKind:0,typeArguments:[{typeName:"AbstractSecurityScheme",nameSpace:"",basicName:"AbstractSecurityScheme",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"}],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"}],moduleName:null,annotationOverridings:{}},{name:"OAuth1SecuritySchemeSettings",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"requestTokenUri",type:{typeName:"Sys.FixedUri",nameSpace:"Sys",basicName:"FixedUri",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"},annotations:[{name:"MetaModel.required",arguments:[]},{name:"MetaModel.description",arguments:["The URI of the Temporary Credential Request endpoint as defined in RFC5849 Section 2.1"]},{name:"MetaModel.valueDescription",arguments:["FixedUriString"]}],valueConstraint:null,optional:!1},{name:"authorizationUri",type:{typeName:"Sys.FixedUri",nameSpace:"Sys",basicName:"FixedUri",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"},annotations:[{name:"MetaModel.required",arguments:[]},{name:"MetaModel.description",arguments:["The URI of the Resource Owner Authorization endpoint as defined in RFC5849 Section 2.2"]},{name:"MetaModel.valueDescription",arguments:["FixedUriString"]}],valueConstraint:null,optional:!1},{name:"tokenCredentialsUri",type:{typeName:"Sys.FixedUri",nameSpace:"Sys",basicName:"FixedUri",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"},annotations:[{name:"MetaModel.required",arguments:[]},{name:"MetaModel.description",arguments:["The URI of the Token Request endpoint as defined in RFC5849 Section 2.3"]},{name:"MetaModel.valueDescription",arguments:["FixedUriString"]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.allowAny",arguments:[]},{name:"MetaModel.functionalDescriminator",arguments:["$parent.type=='OAuth 1.0'"]}],"extends":[{typeName:"SecuritySchemeSettings",nameSpace:"",basicName:"SecuritySchemeSettings",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"}],moduleName:null,annotationOverridings:{}},{name:"OAuth2SecuritySchemeSettings",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"accessTokenUri",type:{typeName:"Sys.FixedUri",nameSpace:"Sys",basicName:"FixedUri",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"},annotations:[{name:"MetaModel.required",arguments:[]},{name:"MetaModel.description",arguments:["The URI of the Token Endpoint as defined in RFC6749 Section 3.2. Not required forby implicit grant type."]},{name:"MetaModel.valueDescription",arguments:["FixedUriString"]}],valueConstraint:null,optional:!1},{name:"authorizationUri",type:{typeName:"Sys.FixedUri",nameSpace:"Sys",basicName:"FixedUri",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"},annotations:[{name:"MetaModel.required",arguments:[]},{name:"MetaModel.description",arguments:["The URI of the Authorization Endpoint as defined in RFC6749 Section 3.1. Required forby authorization_code and implicit grant types."]},{name:"MetaModel.valueDescription",arguments:["FixedUriString"]}],valueConstraint:null,optional:!1},{name:"authorizationGrants",type:{base:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},typeKind:1},annotations:[{name:"MetaModel.description",arguments:["A list of the Authorization grants supported by the API as defined in RFC6749 Sections 4.1, 4.2, 4.3 and 4.4, can be any of: authorization_code, password, client_credentials, implicit, or refresh_token."]},{name:"MetaModel.markdownDescription",arguments:["A list of the Authorization grants supported by the API as defined in RFC6749 Sections 4.1, 4.2, 4.3 and 4.4, can be any of:<br>* authorization_code<br>* password<br>* client_credentials <br>* implicit<br>* refresh_token."]}],valueConstraint:null,optional:!1},{name:"scopes",type:{base:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},typeKind:1},annotations:[{name:"MetaModel.description",arguments:["A list of scopes supported by the security scheme as defined in RFC6749 Section 3.3"]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.allowAny",arguments:[]}],"extends":[{typeName:"SecuritySchemeSettings",nameSpace:"",basicName:"SecuritySchemeSettings",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"}],moduleName:null,annotationOverridings:{}},{name:"OAuth2SecurityScheme",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"type",type:null,annotations:[],valueConstraint:{isCallConstraint:!1,value:"OAuth 2.0"},optional:!1},{name:"settings",type:{typeName:"OAuth2SecuritySchemeSettings",nameSpace:"",basicName:"OAuth2SecuritySchemeSettings",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"},annotations:[],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.description",arguments:["Declares globally referable security schema definition"]},{name:"MetaModel.actuallyExports",arguments:["$self"]},{name:"MetaModel.referenceIs",arguments:["settings"]}],"extends":[{typeName:"AbstractSecurityScheme",nameSpace:"",basicName:"AbstractSecurityScheme",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"}],moduleName:null,annotationOverridings:{}},{name:"OAuth1SecurityScheme",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"type",type:null,annotations:[],valueConstraint:{isCallConstraint:!1,value:"OAuth 1.0"},optional:!1},{name:"settings",type:{typeName:"OAuth1SecuritySchemeSettings",nameSpace:"",basicName:"OAuth1SecuritySchemeSettings",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"},annotations:[],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.description",arguments:["Declares globally referable security schema definition"]},{name:"MetaModel.actuallyExports",arguments:["$self"]},{name:"MetaModel.referenceIs",arguments:["settings"]}],"extends":[{typeName:"AbstractSecurityScheme",nameSpace:"",basicName:"AbstractSecurityScheme",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"}],moduleName:null,annotationOverridings:{}},{name:"BasicSecurityScheme",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"type",type:null,annotations:[],valueConstraint:{isCallConstraint:!1,value:"Basic Authentication"},optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.description",arguments:["Declares globally referable security schema definition"]},{name:"MetaModel.actuallyExports",arguments:["$self"]},{name:"MetaModel.referenceIs",arguments:["settings"]}],"extends":[{typeName:"AbstractSecurityScheme",nameSpace:"",basicName:"AbstractSecurityScheme",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"}],moduleName:null,annotationOverridings:{}},{name:"DigestSecurityScheme",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"type",type:null,annotations:[],valueConstraint:{isCallConstraint:!1,value:"Digest Authentication"
},optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.description",arguments:["Declares globally referable security schema definition"]},{name:"MetaModel.actuallyExports",arguments:["$self"]},{name:"MetaModel.referenceIs",arguments:["settings"]}],"extends":[{typeName:"AbstractSecurityScheme",nameSpace:"",basicName:"AbstractSecurityScheme",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"}],moduleName:null,annotationOverridings:{}},{name:"CustomSecurityScheme",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"type",type:null,annotations:[],valueConstraint:{isCallConstraint:!1,value:"x-{other}"},optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.description",arguments:["Declares globally referable security schema definition"]},{name:"MetaModel.actuallyExports",arguments:["$self"]},{name:"MetaModel.referenceIs",arguments:["settings"]}],"extends":[{typeName:"AbstractSecurityScheme",nameSpace:"",basicName:"AbstractSecurityScheme",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"}],moduleName:null,annotationOverridings:{}}],aliases:[],enumDeclarations:[],imports:{MetaModel:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/metamodel.ts",Sys:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/systemTypes.ts",Params:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/parameters.ts",Bodies:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/bodies.ts",Common:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/common.ts",Methods:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/methods.ts"},name:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-0.8/security.ts"}]},function(e,t,n){e.exports=[{classes:[{name:"Library",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"usage",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["contains description of why library exist"]}],valueConstraint:null,optional:!1},{name:"name",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.key",arguments:[]},{name:"MetaModel.description",arguments:["Namespace which the library is imported under"]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[],"extends":[{typeName:"LibraryBase",nameSpace:"",basicName:"LibraryBase",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/api.ts"}],moduleName:null,annotationOverridings:{}},{name:"LibraryBase",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"schemas",type:{base:{typeName:"TypeDeclaration",nameSpace:"",basicName:"TypeDeclaration",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/api.ts"},typeKind:1},annotations:[{name:"MetaModel.embeddedInMaps",arguments:[]},{name:"MetaModel.description",arguments:['Alias for the equivalent "types" property, for compatibility with RAML 0.8. Deprecated - API definitions should use the "types" property, as the "schemas" alias for that property name may be removed in a future RAML version. The "types" property allows for XML and JSON schemas.']}],valueConstraint:null,optional:!1},{name:"types",type:{base:{typeName:"DataModel.TypeDeclaration",nameSpace:"DataModel",basicName:"TypeDeclaration",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/api.ts"},typeKind:1},annotations:[{name:"MetaModel.embeddedInMaps",arguments:[]},{name:"MetaModel.setsContextValue",arguments:["locationKind","DataModel.LocationKind.MODELS"]},{name:"MetaModel.description",arguments:["Declarations of (data) types for use within this API"]},{name:"MetaModel.markdownDescription",arguments:["Declarations of (data) types for use within this API."]},{name:"MetaModel.valueDescription",arguments:["An object whose properties map type names to type declarations; or an array of such objects"]}],valueConstraint:null,optional:!1},{name:"traits",type:{base:{typeName:"Methods.Trait",nameSpace:"Methods",basicName:"Trait",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/api.ts"},typeKind:1},annotations:[{name:"MetaModel.embeddedInMaps",arguments:[]},{name:"MetaModel.description",arguments:["Declarations of traits used in this API"]},{name:"MetaModel.description",arguments:["Declarations of traits for use within this API"]},{name:"MetaModel.markdownDescription",arguments:["Declarations of traits for use within this API."]},{name:"MetaModel.valueDescription",arguments:["An object whose properties map trait names to trait declarations; or an array of such objects"]}],valueConstraint:null,optional:!1},{name:"resourceTypes",type:{base:{typeName:"Resources.ResourceType",nameSpace:"Resources",basicName:"ResourceType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/api.ts"},typeKind:1},annotations:[{name:"MetaModel.embeddedInMaps",arguments:[]},{name:"MetaModel.description",arguments:["Declaration of resource types used in this API"]},{name:"MetaModel.description",arguments:["Declarations of resource types for use within this API"]},{name:"MetaModel.markdownDescription",arguments:["Declarations of resource types for use within this API."]},{name:"MetaModel.valueDescription",arguments:["An object whose properties map resource type names to resource type declarations; or an array of such objects"]}],valueConstraint:null,optional:!1},{name:"annotationTypes",type:{base:{typeName:"DataModel.TypeDeclaration",nameSpace:"DataModel",basicName:"TypeDeclaration",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/api.ts"},typeKind:1},annotations:[{name:"MetaModel.setsContextValue",arguments:["decls","true"]},{name:"MetaModel.embeddedInMaps",arguments:[]},{name:"MetaModel.description",arguments:["Declarations of annotation types for use by annotations"]},{name:"MetaModel.markdownDescription",arguments:["Declarations of annotation types for use by annotations."]},{name:"MetaModel.valueDescription",arguments:["An object whose properties map annotation type names to annotation type declarations; or an array of such objects"]}],valueConstraint:null,optional:!1},{name:"securitySchemes",type:{base:{typeName:"Security.AbstractSecurityScheme",nameSpace:"Security",basicName:"AbstractSecurityScheme",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/api.ts"},typeKind:1},annotations:[{name:"MetaModel.embeddedInMaps",arguments:[]},{name:"MetaModel.description",arguments:["Security schemas declarations"]},{name:"MetaModel.description",arguments:["Declarations of security schemes for use within this API."]},{name:"MetaModel.markdownDescription",arguments:["Declarations of security schemes for use within this API."]},{name:"MetaModel.valueDescription",arguments:["An object whose properties map security scheme names to security scheme declarations; or an array of such objects"]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.internalClass",arguments:[]}],"extends":[{typeName:"FragmentDeclaration",nameSpace:"",basicName:"FragmentDeclaration",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/api.ts"}],moduleName:null,annotationOverridings:{}},{name:"Api",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"title",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.required",arguments:[]},{name:"MetaModel.description",arguments:["Short plain-text label for the API"]}],valueConstraint:null,optional:!1},{name:"description",type:{typeName:"Sys.MarkdownString",nameSpace:"Sys",basicName:"MarkdownString",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/api.ts"},annotations:[{name:"MetaModel.description",arguments:["A longer, human-friendly description of the API"]}],valueConstraint:null,optional:!1},{name:"version",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["The version of the API, e.g. 'v1'"]}],valueConstraint:null,optional:!1},{name:"baseUri",type:{typeName:"Sys.FullUriTemplateString",nameSpace:"Sys",basicName:"FullUriTemplateString",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/api.ts"},annotations:[{name:"MetaModel.description",arguments:["A URI that's to be used as the base of all the resources' URIs. Often used as the base of the URL of each resource, containing the location of the API. Can be a template URI."]}],valueConstraint:null,optional:!1},{name:"baseUriParameters",type:{base:{typeName:"DataModel.TypeDeclaration",nameSpace:"DataModel",basicName:"TypeDeclaration",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/api.ts"},typeKind:1},annotations:[{name:"MetaModel.setsContextValue",arguments:["location","DataModel.ModelLocation.BURI"]},{name:"MetaModel.setsContextValue",arguments:["locationKind","DataModel.LocationKind.APISTRUCTURE"]},{name:"MetaModel.description",arguments:["Named parameters used in the baseUri (template)"]}],valueConstraint:null,optional:!1},{name:"protocols",type:{base:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},typeKind:1},annotations:[{name:"MetaModel.oneOf",arguments:[["HTTP","HTTPS"]]},{name:"MetaModel.description",arguments:["The protocols supported by the API"]},{name:"MetaModel.valueDescription",arguments:['Array of strings, with each being "HTTP" or "HTTPS", case-insensitive']}],valueConstraint:null,optional:!1},{name:"mediaType",type:{base:{typeName:"Bodies.MimeType",nameSpace:"Bodies",basicName:"MimeType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/api.ts"},typeKind:1},annotations:[{name:"MetaModel.oftenKeys",arguments:[["application/json","application/xml","application/x-www-form-urlencoded","multipart/form-data"]]},{name:"MetaModel.description",arguments:['The default media type to use for request and response bodies (payloads), e.g. "application/json"']},{name:"MetaModel.inherited",arguments:[]},{name:"MetaModel.valueDescription",arguments:["Media type string"]}],valueConstraint:null,optional:!1},{name:"securedBy",type:{base:{typeName:"Security.SecuritySchemeRef",nameSpace:"Security",basicName:"SecuritySchemeRef",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/api.ts"},typeKind:1},annotations:[{name:"MetaModel.description",arguments:["The security schemes that apply to every resource and method in the API"]}],valueConstraint:null,optional:!1},{name:"resources",type:{base:{typeName:"Resources.Resource",nameSpace:"Resources",basicName:"Resource",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/api.ts"},typeKind:1},annotations:[{name:"MetaModel.documentationTableLabel",arguments:["/&lt;relativeUri&gt;"]},{name:"MetaModel.newInstanceName",arguments:["New Resource"]},{name:"MetaModel.description",arguments:["The resources of the API, identified as relative URIs that begin with a slash (/). Every property whose key begins with a slash (/), and is either at the root of the API definition or is the child property of a resource property, is a resource property, e.g.: /users, /{groupId}, etc"]}],valueConstraint:null,optional:!1},{name:"documentation",type:{base:{typeName:"DocumentationItem",nameSpace:"",basicName:"DocumentationItem",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/api.ts"},typeKind:1},annotations:[{name:"MetaModel.description",arguments:["Additional overall documentation for the API"]}],valueConstraint:null,optional:!1},{name:"RAMLVersion",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.customHandling",arguments:[]},{name:"MetaModel.description",arguments:['Returns RAML version. "RAML10" string is returned for RAML 1.0. "RAML08" string is returned for RAML 0.8.']}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[],"extends":[{typeName:"LibraryBase",nameSpace:"",basicName:"LibraryBase",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/api.ts"}],moduleName:null,annotationOverridings:{annotations:[{name:"MetaModel.markdownDescription",arguments:['Annotations to be applied to this API. Annotations are any property whose key begins with "(" and ends with ")" and whose name (the part between the beginning and ending parentheses) is a declared annotation name.']}]}},{name:"Overlay",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"usage",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["contains description of why overlay exist"]}],valueConstraint:null,optional:!1},{name:"extends",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.required",arguments:[]},{name:"MetaModel.description",arguments:["Location of a valid RAML API definition (or overlay or extension), the overlay is applied to."]}],valueConstraint:null,optional:!1},{name:"title",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["Short plain-text label for the API"]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[],"extends":[{typeName:"Api",nameSpace:"",basicName:"Api",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/api.ts"}],moduleName:null,annotationOverridings:{}},{name:"Extension",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"usage",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["contains description of why extension exist"]}],valueConstraint:null,optional:!1},{name:"extends",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.required",arguments:[]},{name:"MetaModel.description",arguments:["Location of a valid RAML API definition (or overlay or extension), the extension is applied to"]}],valueConstraint:null,optional:!1},{name:"title",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["Short plain-text label for the API"]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[],"extends":[{typeName:"Api",nameSpace:"",basicName:"Api",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/api.ts"}],moduleName:null,annotationOverridings:{}},{name:"UsesDeclaration",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"key",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.key",arguments:[]},{name:"MetaModel.description",arguments:["Name prefix (without dot) used to refer imported declarations"]}],valueConstraint:null,optional:!1},{name:"value",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["Content of the schema"]},{name:"MetaModel.canBeValue",arguments:[]},{name:"MetaModel.value",arguments:[]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[],"extends":[{typeName:"Annotable",nameSpace:"",basicName:"Annotable",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/api.ts"}],moduleName:null,annotationOverridings:{}},{name:"FragmentDeclaration",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"uses",type:{base:{typeName:"UsesDeclaration",nameSpace:"",basicName:"UsesDeclaration",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/api.ts"},typeKind:1},annotations:[{name:"MetaModel.embeddedInMaps",arguments:[]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[],"extends":[{typeName:"Annotable",nameSpace:"",basicName:"Annotable",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/api.ts"}],moduleName:null,annotationOverridings:{}},{name:"DocumentationItem",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"title",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["Title of documentation section"]},{name:"MetaModel.required",arguments:[]}],valueConstraint:null,optional:!1},{name:"content",type:{typeName:"Sys.MarkdownString",nameSpace:"Sys",basicName:"MarkdownString",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/api.ts"},annotations:[{name:"MetaModel.description",arguments:["Content of documentation section"]},{name:"MetaModel.required",arguments:[]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.possibleInterfaces",arguments:[["FragmentDeclaration"]]}],"extends":[{typeName:"Annotable",nameSpace:"",basicName:"Annotable",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/api.ts"}],moduleName:null,annotationOverridings:{}}],aliases:[],enumDeclarations:[],imports:{MetaModel:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/metamodel.ts",Sys:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/systemTypes.ts",Methods:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/methods.ts",Resources:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/resources.ts",Decls:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/declarations.ts",Params:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/parameters.ts",Common:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/common.ts",Bodies:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/bodies.ts",DataModel:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/datamodel.ts",Security:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/security.ts"},name:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/api.ts"},{classes:[{name:"SpecPartMetaData",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!0,annotations:[],"extends":[],moduleName:null,annotationOverridings:{}}],aliases:[],enumDeclarations:[],imports:{},name:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/metamodel.ts"},{classes:[{name:"ValueType",methods:[{returnType:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},name:"value",start:170,end:210,text:"\n\n  value():string {\n    return null\n  }",arguments:[]}],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[],"extends":[{typeName:"Annotable",nameSpace:"",basicName:"Annotable",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"StringType",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.nameAtRuntime",arguments:["string"]},{name:"MetaModel.alias",arguments:["string"]}],"extends":[{typeName:"ValueType",nameSpace:"",basicName:"ValueType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"AnyType",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.nameAtRuntime",arguments:["any"]},{name:"MetaModel.alias",arguments:["any"]}],"extends":[{typeName:"ValueType",nameSpace:"",basicName:"ValueType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"NumberType",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.nameAtRuntime",arguments:["number"]},{name:"MetaModel.alias",arguments:["number"]}],"extends":[{typeName:"ValueType",nameSpace:"",basicName:"ValueType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"IntegerType",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.nameAtRuntime",arguments:["integer"]},{name:"MetaModel.alias",arguments:["integer"]}],"extends":[{typeName:"ValueType",nameSpace:"",basicName:"ValueType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"NullType",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.nameAtRuntime",arguments:["null"]},{name:"MetaModel.alias",arguments:["null"]}],"extends":[{typeName:"ValueType",nameSpace:"",basicName:"ValueType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"TimeOnlyType",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.nameAtRuntime",arguments:["time-only"]},{name:"MetaModel.alias",arguments:["time-only"]}],"extends":[{typeName:"ValueType",nameSpace:"",basicName:"ValueType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"DateOnlyType",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.nameAtRuntime",arguments:["date-only"]},{name:"MetaModel.alias",arguments:["date-only"]}],"extends":[{typeName:"ValueType",nameSpace:"",basicName:"ValueType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"DateTimeOnlyType",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.nameAtRuntime",arguments:["datetime-only"]},{name:"MetaModel.alias",arguments:["datetime-only"]}],"extends":[{typeName:"ValueType",nameSpace:"",basicName:"ValueType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"DateTimeType",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.nameAtRuntime",arguments:["datetime"]},{name:"MetaModel.alias",arguments:["datetime"]}],"extends":[{typeName:"ValueType",nameSpace:"",basicName:"ValueType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"FileType",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.nameAtRuntime",arguments:["file"]},{name:"MetaModel.alias",arguments:["file"]}],"extends":[{typeName:"ValueType",nameSpace:"",basicName:"ValueType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"BooleanType",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.nameAtRuntime",arguments:["boolean"]},{name:"MetaModel.alias",arguments:["boolean"]}],"extends":[{typeName:"ValueType",nameSpace:"",basicName:"ValueType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"Reference",methods:[],typeParameters:["T"],typeParameterConstraint:[null],"implements":[],fields:[{name:"structuredValue",type:{typeName:"DataModel.TypeInstance",nameSpace:"DataModel",basicName:"TypeInstance",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/systemTypes.ts"},annotations:[{name:"MetaModel.customHandling",arguments:[]},{name:"MetaModel.description",arguments:["Returns a structured object if the reference point to one."]}],valueConstraint:null,optional:!1},{name:"name",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.customHandling",arguments:[]},{name:"MetaModel.description",arguments:["Returns name of referenced object"]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[],"extends":[{typeName:"ValueType",nameSpace:"",basicName:"ValueType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"UriTemplate",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.description",arguments:["This type currently serves both for absolute and relative urls"]}],"extends":[{typeName:"StringType",nameSpace:"",basicName:"StringType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"StatusCodeString",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[],"extends":[{typeName:"StringType",nameSpace:"",basicName:"StringType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"RelativeUriString",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.description",arguments:["This  type describes relative uri templates"]}],"extends":[{typeName:"UriTemplate",nameSpace:"",basicName:"UriTemplate",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"FullUriTemplateString",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.description",arguments:["This  type describes absolute uri templates"]}],"extends":[{typeName:"UriTemplate",nameSpace:"",basicName:"UriTemplate",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"FixedUriString",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.description",arguments:["This  type describes fixed uris"]}],"extends":[{typeName:"StringType",nameSpace:"",basicName:"StringType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"ContentType",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[],"extends":[{typeName:"StringType",nameSpace:"",basicName:"StringType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"MarkdownString",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.innerType",arguments:["markdown"]},{name:"MetaModel.description",arguments:["[GitHub Flavored Markdown](https://help.github.com/articles/github-flavored-markdown/)"]}],"extends":[{typeName:"StringType",nameSpace:"",basicName:"StringType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/systemTypes.ts"}],moduleName:null,annotationOverridings:{}},{name:"SchemaString",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[],isInterface:!1,annotations:[{name:"MetaModel.description",arguments:["Schema at this moment only two subtypes are supported (json schema and xsd)"]},{name:"MetaModel.alias",arguments:["schema"]}],"extends":[{typeName:"StringType",nameSpace:"",basicName:"StringType",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/systemTypes.ts"}],moduleName:null,annotationOverridings:{}}],aliases:[],enumDeclarations:[],imports:{MetaModel:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/metamodel.ts",DataModel:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/datamodel.ts",Common:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/common.ts"
},name:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/systemTypes.ts"},{classes:[{name:"ExampleSpec",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"value",type:{typeName:"any",nameSpace:"",basicName:"any",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.example",arguments:[]},{name:"MetaModel.selfNode",arguments:[]},{name:"MetaModel.description",arguments:["String representation of example"]},{name:"MetaModel.required",arguments:[]},{name:"MetaModel.valueDescription",arguments:["* Valid value for this type<br>* String representing the serialized version of a valid value"]}],valueConstraint:null,optional:!1},{name:"structuredValue",type:{typeName:"TypeInstance",nameSpace:"",basicName:"TypeInstance",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/datamodel.ts"},annotations:[{name:"MetaModel.customHandling",arguments:[]},{name:"MetaModel.description",arguments:["Returns object representation of example, if possible"]}],valueConstraint:null,optional:!1},{name:"strict",type:{typeName:"boolean",nameSpace:"",basicName:"boolean",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["By default, examples are validated against any type declaration. Set this to false to allow examples that need not validate."]}],valueConstraint:null,optional:!1},{name:"name",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.key",arguments:[]},{name:"MetaModel.hide",arguments:[]},{name:"MetaModel.description",arguments:["Example identifier, if specified"]}],valueConstraint:null,optional:!1},{name:"displayName",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["An alternate, human-friendly name for the example"]}],valueConstraint:null,optional:!1},{name:"description",type:{typeName:"Sys.MarkdownString",nameSpace:"Sys",basicName:"MarkdownString",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/datamodel.ts"},annotations:[{name:"MetaModel.description",arguments:["A longer, human-friendly description of the example"]},{name:"MetaModel.valueDescription",arguments:["markdown string"]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.customHandling",arguments:[]},{name:"MetaModel.possibleInterfaces",arguments:[["FragmentDeclaration"]]}],"extends":[{typeName:"Annotable",nameSpace:"",basicName:"Annotable",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/datamodel.ts"}],moduleName:null,annotationOverridings:{annotations:[{name:"MetaModel.markdownDescription",arguments:['Annotations to be applied to this example. Annotations are any property whose key begins with "(" and ends with ")" and whose name (the part between the beginning and ending parentheses) is a declared annotation name.']}]}},{name:"TypeDeclaration",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"name",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.key",arguments:[]},{name:"MetaModel.description",arguments:["name of the parameter"]},{name:"MetaModel.extraMetaKey",arguments:["headers"]},{name:"MetaModel.hide",arguments:[]}],valueConstraint:null,optional:!1},{name:"displayName",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["The displayName attribute specifies the type display name. It is a friendly name used only for  display or documentation purposes. If displayName is not specified, it defaults to the element's key (the name of the property itself)."]}],valueConstraint:null,optional:!1},{name:"facets",type:{base:{typeName:"TypeDeclaration",nameSpace:"",basicName:"TypeDeclaration",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/datamodel.ts"},typeKind:1},annotations:[{name:"MetaModel.declaringFields",arguments:[]},{name:"MetaModel.description",arguments:["When extending from a type you can define new facets (which can then be set to concrete values by subtypes)."]},{name:"MetaModel.hide",arguments:[]}],valueConstraint:null,optional:!1},{name:"fixedFacets",type:{typeName:"TypeInstance",nameSpace:"",basicName:"TypeInstance",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/datamodel.ts"},annotations:[{name:"MetaModel.customHandling",arguments:[]},{name:"MetaModel.description",arguments:["Returns facets fixed by the type. Value is an object with properties named after facets fixed. Value of each property is a value of the corresponding facet."]}],valueConstraint:null,optional:!1},{name:"schema",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.typeExpression",arguments:[]},{name:"MetaModel.allowMultiple",arguments:[]},{name:"MetaModel.description",arguments:['Alias for the equivalent "type" property, for compatibility with RAML 0.8. Deprecated - API definitions should use the "type" property, as the "schema" alias for that property name may be removed in a future RAML version. The "type" property allows for XML and JSON schemas.']},{name:"MetaModel.valueDescription",arguments:["Single string denoting the base type or type expression"]}],valueConstraint:null,optional:!1},{name:"type",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.typeExpression",arguments:[]},{name:"MetaModel.allowMultiple",arguments:[]},{name:"MetaModel.canBeValue",arguments:[]},{name:"MetaModel.defaultValue",arguments:["string"]},{name:"MetaModel.descriminatingProperty",arguments:[]},{name:"MetaModel.description",arguments:["A base type which the current type extends, or more generally a type expression."]},{name:"MetaModel.valueDescription",arguments:["string denoting the base type or type expression"]}],valueConstraint:null,optional:!1},{name:"structuredType",type:{typeName:"TypeInstance",nameSpace:"",basicName:"TypeInstance",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/datamodel.ts"},annotations:[{name:"MetaModel.customHandling",arguments:[]},{name:"MetaModel.typeExpression",arguments:[]},{name:"MetaModel.description",arguments:["Inlined supertype definition."]},{name:"MetaModel.valueDescription",arguments:["Inlined supertype definition"]}],valueConstraint:null,optional:!1},{name:"location",type:{typeName:"ModelLocation",nameSpace:"",basicName:"ModelLocation",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/datamodel.ts"},annotations:[{name:"MetaModel.system",arguments:[]},{name:"MetaModel.description",arguments:["Location of the parameter (can not be edited by user)"]},{name:"MetaModel.hide",arguments:[]}],valueConstraint:null,optional:!1},{name:"locationKind",type:{typeName:"LocationKind",nameSpace:"",basicName:"LocationKind",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/datamodel.ts"},annotations:[{name:"MetaModel.system",arguments:[]},{name:"MetaModel.description",arguments:["Kind of location"]},{name:"MetaModel.hide",arguments:[]}],valueConstraint:null,optional:!1},{name:"default",type:{typeName:"any",nameSpace:"",basicName:"any",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["Provides default value for a property"]},{name:"MetaModel.hide",arguments:[]}],valueConstraint:null,optional:!1},{name:"example",type:{typeName:"ExampleSpec",nameSpace:"",basicName:"ExampleSpec",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/datamodel.ts"},annotations:[{name:"MetaModel.example",arguments:[]},{name:"MetaModel.selfNode",arguments:[]},{name:"MetaModel.description",arguments:["An example of this type instance represented as string or yaml map/sequence. This can be used, e.g., by documentation generators to generate sample values for an object of this type. Cannot be present if the examples property is present."]},{name:"MetaModel.valueDescription",arguments:["* Valid value for this type<br>* String representing the serialized version of a valid value"]}],valueConstraint:null,optional:!1},{name:"examples",type:{base:{typeName:"ExampleSpec",nameSpace:"",basicName:"ExampleSpec",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/datamodel.ts"},typeKind:1},annotations:[{name:"MetaModel.example",arguments:[]},{name:"MetaModel.selfNode",arguments:[]},{name:"MetaModel.description",arguments:["An example of this type instance represented as string. This can be used, e.g., by documentation generators to generate sample values for an object of this type. Cannot be present if the example property is present."]},{name:"MetaModel.valueDescription",arguments:["* Valid value for this type<br>* String representing the serialized version of a valid value"]}],valueConstraint:null,optional:!1},{name:"required",type:{typeName:"boolean",nameSpace:"",basicName:"boolean",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.requireValue",arguments:["fieldOrParam",!0]},{name:"MetaModel.description",arguments:["Sets if property is optional or not"]},{name:"MetaModel.describesAnnotation",arguments:["required"]},{name:"MetaModel.hide",arguments:[]},{name:"MetaModel.defaultBooleanValue",arguments:[!0]}],valueConstraint:null,optional:!1},{name:"description",type:{typeName:"Sys.MarkdownString",nameSpace:"Sys",basicName:"MarkdownString",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/datamodel.ts"},annotations:[{name:"MetaModel.description",arguments:["A longer, human-friendly description of the type"]},{name:"MetaModel.valueDescription",arguments:["markdown string"]}],valueConstraint:null,optional:!1},{name:"xml",type:{typeName:"XMLFacetInfo",nameSpace:"",basicName:"XMLFacetInfo",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/datamodel.ts"},annotations:[],valueConstraint:null,optional:!1},{name:"allowedTargets",type:{base:{typeName:"AnnotationTarget",nameSpace:"",basicName:"AnnotationTarget",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/datamodel.ts"},typeKind:1},annotations:[{name:"MetaModel.oneOf",arguments:[["API","DocumentationItem","Resource","Method","Response","RequestBody","ResponseBody","TypeDeclaration","NamedExample","ResourceType","Trait","SecurityScheme","SecuritySchemeSettings","AnnotationTypeDeclaration","Library","Overlay","Extension","Scalar"]]},{name:"MetaModel.description",arguments:["Restrictions on where annotations of this type can be applied. If this property is specified, annotations of this type may only be applied on a property corresponding to one of the target names specified as the value of this property."]},{name:"MetaModel.valueDescription",arguments:["An array, or single, of names allowed target nodes."]}],valueConstraint:null,optional:!1},{name:"isAnnotation",type:{typeName:"boolean",nameSpace:"",basicName:"boolean",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["Whether the type represents annotation"]}],valueConstraint:null,optional:!1},{name:"parametrizedProperties",type:{typeName:"TypeInstance",nameSpace:"",basicName:"TypeInstance",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/datamodel.ts"},annotations:[{name:"MetaModel.customHandling",arguments:[]},{name:"MetaModel.description",arguments:["For types defined in traits or resource types returns object representation of parametrized properties"]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.convertsToGlobalOfType",arguments:["SchemaString"]},{name:"MetaModel.canInherit",arguments:["mediaType"]},{name:"MetaModel.possibleInterfaces",arguments:[["FragmentDeclaration"]]}],"extends":[{typeName:"Annotable",nameSpace:"",basicName:"Annotable",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/datamodel.ts"}],moduleName:null,annotationOverridings:{annotations:[{name:"MetaModel.markdownDescription",arguments:['Annotations to be applied to this type. Annotations are any property whose key begins with "(" and ends with ")" and whose name (the part between the beginning and ending parentheses) is a declared annotation name.']}]}},{name:"XMLFacetInfo",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"attribute",type:{typeName:"boolean",nameSpace:"",basicName:"boolean",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["If attribute is set to true, a type instance should be serialized as an XML attribute. It can only be true for scalar types."]}],valueConstraint:null,optional:!1},{name:"wrapped",type:{typeName:"boolean",nameSpace:"",basicName:"boolean",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["If wrapped is set to true, a type instance should be wrapped in its own XML element. It can not be true for scalar types and it can not be true at the same moment when attribute is true."]}],valueConstraint:null,optional:!1},{name:"name",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["Allows to override the name of the XML element or XML attribute in it's XML representation."]}],valueConstraint:null,optional:!1},{name:"namespace",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["Allows to configure the name of the XML namespace."]}],valueConstraint:null,optional:!1},{name:"prefix",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["Allows to configure the prefix which will be used during serialization to XML."]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[],"extends":[{typeName:"Annotable",nameSpace:"",basicName:"Annotable",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/datamodel.ts"}],moduleName:null,annotationOverridings:{}},{name:"ArrayTypeDeclaration",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"type",type:null,annotations:[],valueConstraint:{isCallConstraint:!1,value:"array"},optional:!1},{name:"uniqueItems",type:{typeName:"boolean",nameSpace:"",basicName:"boolean",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.facetId",arguments:["uniqueItems"]},{name:"MetaModel.description",arguments:["Should items in array be unique"]}],valueConstraint:null,optional:!1},{name:"items",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.typeExpression",arguments:[]},{name:"MetaModel.allowMultiple",arguments:[]},{name:"MetaModel.canBeValue",arguments:[]},{name:"MetaModel.description",arguments:["Array component type."]},{name:"MetaModel.valueDescription",arguments:["Inline type declaration or type name."]}],valueConstraint:null,optional:!1},{name:"structuredItems",type:{typeName:"TypeInstance",nameSpace:"",basicName:"TypeInstance",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/datamodel.ts"},annotations:[{name:"MetaModel.customHandling",arguments:[]},{name:"MetaModel.typeExpression",arguments:[]},{name:"MetaModel.description",arguments:["Inlined component type definition"]},{name:"MetaModel.valueDescription",arguments:["Inlined component type definition"]}],valueConstraint:null,optional:!1},{name:"minItems",type:{typeName:"number",nameSpace:"",basicName:"number",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.facetId",arguments:["minItems"]},{name:"MetaModel.description",arguments:["Minimum amount of items in array"]},{name:"MetaModel.valueDescription",arguments:["integer ( >= 0 ). Defaults to 0"]}],valueConstraint:null,optional:!1},{name:"maxItems",type:{typeName:"number",nameSpace:"",basicName:"number",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.facetId",arguments:["maxItems"]},{name:"MetaModel.description",arguments:["Maximum amount of items in array"]},{name:"MetaModel.valueDescription",arguments:["integer ( >= 0 ). Defaults to undefined."]}],valueConstraint:null,optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.convertsToGlobalOfType",arguments:["SchemaString"]},{name:"MetaModel.alias",arguments:["array"]},{name:"MetaModel.declaresSubTypeOf",arguments:["TypeDeclaration"]}],"extends":[{typeName:"TypeDeclaration",nameSpace:"",basicName:"TypeDeclaration",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/datamodel.ts"}],moduleName:null,annotationOverridings:{}},{name:"UnionTypeDeclaration",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"type",type:null,annotations:[],valueConstraint:{isCallConstraint:!1,value:"union"},optional:!1}],isInterface:!1,annotations:[{name:"MetaModel.convertsToGlobalOfType",arguments:["SchemaString"]},{name:"MetaModel.requireValue",arguments:["locationKind","LocationKind.MODELS"]},{name:"MetaModel.declaresSubTypeOf",arguments:["TypeDeclaration"]}],"extends":[{typeName:"TypeDeclaration",nameSpace:"",basicName:"TypeDeclaration",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/datamodel.ts"}],moduleName:null,annotationOverridings:{}},{name:"ObjectTypeDeclaration",methods:[],typeParameters:[],typeParameterConstraint:[],"implements":[],fields:[{name:"type",type:null,annotations:[{name:"MetaModel.hide",arguments:[]}],valueConstraint:{isCallConstraint:!1,value:"object"},optional:!1},{name:"properties",type:{base:{typeName:"TypeDeclaration",nameSpace:"",basicName:"TypeDeclaration",typeKind:0,typeArguments:[],modulePath:"/Users/munch/work/repositories_official/raml-definition-system/raml-definition/spec-1.0/datamodel.ts"},typeKind:1},annotations:[{name:"MetaModel.setsContextValue",arguments:["fieldOrParam",!0]},{name:"MetaModel.description",arguments:["The properties that instances of this type may or must have."]},{name:"MetaModel.valueDescription",arguments:["An object whose keys are the properties' names and whose values are property declarations."]}],valueConstraint:null,optional:!1},{name:"minProperties",type:{typeName:"number",nameSpace:"",basicName:"number",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.facetId",arguments:["minProperties"]},{name:"MetaModel.description",arguments:["The minimum number of properties allowed for instances of this type."]}],valueConstraint:null,optional:!1},{name:"maxProperties",type:{typeName:"number",nameSpace:"",basicName:"number",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.facetId",arguments:["maxProperties"]},{name:"MetaModel.description",arguments:["The maximum number of properties allowed for instances of this type."]}],valueConstraint:null,optional:!1},{name:"additionalProperties",type:{typeName:"boolean",nameSpace:"",basicName:"boolean",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["A Boolean that indicates if an object instance has additional properties."]}],valueConstraint:null,optional:!1},{name:"discriminator",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["Type property name to be used as discriminator, or boolean"]}],valueConstraint:null,optional:!1},{name:"discriminatorValue",type:{typeName:"string",nameSpace:"",basicName:"string",typeKind:0,typeArguments:[],modulePath:null},annotations:[{name:"MetaModel.description",arguments:["The value of discriminator for the type."]}],valueConstraint:null,optional:!1},{name:"enum",t