var tap = require("tap")
  , test = tap.test
  , semver = require("./semver.js")
  , eq = semver.eq
  , gt = semver.gt
  , lt = semver.lt
  , neq = semver.neq
  , cmp = semver.cmp
  , gte = semver.gte
  , lte = semver.lte
  , satisfies = semver.satisfies
  , validRange = semver.validRange
  , inc = semver.inc
  , replaceStars = semver.replaceStars
  , toComparators = semver.toComparators

tap.plan(8)

test("\ncomparison tests", function (t) {
; [ ["0.0.0", "0.0.0foo"]
  , ["0.0.1", "0.0.0"]
  , ["1.0.0", "0.9.9"]
  , ["0.10.0", "0.9.0"]
  , ["0.99.0", "0.10.0"]
  , ["2.0.0", "1.2.3"]
  , ["v0.0.0", "0.0.0foo"]
  , ["v0.0.1", "0.0.0"]
  , ["v1.0.0", "0.9.9"]
  , ["v0.10.0", "0.9.0"]
  , ["v0.99.0", "0.10.0"]
  , ["v2.0.0", "1.2.3"]
  , ["0.0.0", "v0.0.0foo"]
  , ["0.0.1", "v0.0.0"]
  , ["1.0.0", "v0.9.9"]
  , ["0.10.0", "v0.9.0"]
  , ["0.99.0", "v0.10.0"]
  , ["2.0.0", "v1.2.3"]
  , ["1.2.3", "1.2.3-asdf"]
  , ["1.2.3-4", "1.2.3"]
  , ["1.2.3-4-foo", "1.2.3"]
  , ["1.2.3-5", "1.2.3-5-foo"]
  , ["1.2.3-5", "1.2.3-4"]
  , ["1.2.3-5-foo", "1.2.3-5-Foo"]
  ].forEach(function (v) {
    var v0 = v[0]
      , v1 = v[1]
    t.ok(gt(v0, v1), "gt('"+v0+"', '"+v1+"')")
    t.ok(lt(v1, v0), "lt('"+v1+"', '"+v0+"')")
    t.ok(!gt(v1, v0), "!gt('"+v1+"', '"+v0+"')")
    t.ok(!lt(v0, v1), "!lt('"+v0+"', '"+v1+"')")
    t.ok(eq(v0, v0), "eq('"+v0+"', '"+v0+"')")
    t.ok(eq(v1, v1), "eq('"+v1+"', '"+v1+"')")
    t.ok(neq(v0, v1), "neq('"+v0+"', '"+v1+"')")
    t.ok(cmp(v1, "==", v1), "cmp('"+v1+"' == '"+v1+"')")
    t.ok(cmp(v0, ">=", v1), "cmp('"+v0+"' >= '"+v1+"')")
    t.ok(cmp(v1, "<=", v0), "cmp('"+v1+"' <= '"+v0+"')")
    t.ok(cmp(v0, "!=", v1), "cmp('"+v0+"' != '"+v1+"')")
  })
  t.end()
})

test("\nequality tests", function (t) {
; [ ["1.2.3", "v1.2.3"]
  , ["1.2.3", "=1.2.3"]
  , ["1.2.3", "v 1.2.3"]
  , ["1.2.3", "= 1.2.3"]
  , ["1.2.3", " v1.2.3"]
  , ["1.2.3", " =1.2.3"]
  , ["1.2.3", " v 1.2.3"]
  , ["1.2.3", " = 1.2.3"]
  , ["1.2.3-0", "v1.2.3-0"]
  , ["1.2.3-0", "=1.2.3-0"]
  , ["1.2.3-0", "v 1.2.3-0"]
  , ["1.2.3-0", "= 1.2.3-0"]
  , ["1.2.3-0", " v1.2.3-0"]
  , ["1.2.3-0", " =1.2.3-0"]
  , ["1.2.3-0", " v 1.2.3-0"]
  , ["1.2.3-0", " = 1.2.3-0"]
  , ["1.2.3-01", "v1.2.3-1"]
  , ["1.2.3-01", "=1.2.3-1"]
  , ["1.2.3-01", "v 1.2.3-1"]
  , ["1.2.3-01", "= 1.2.3-1"]
  , ["1.2.3-01", " v1.2.3-1"]
  , ["1.2.3-01", " =1.2.3-1"]
  , ["1.2.3-01", " v 1.2.3-1"]
  , ["1.2.3-01", " = 1.2.3-1"]
  , ["1.2.3beta", "v1.2.3beta"]
  , ["1.2.3beta", "=1.2.3beta"]
  , ["1.2.3beta", "v 1.2.3beta"]
  , ["1.2.3beta", "= 1.2.3beta"]
  , ["1.2.3beta", " v1.2.3beta"]
  , ["1.2.3beta", " =1.2.3beta"]
  , ["1.2.3beta", " v 1.2.3beta"]
  , ["1.2.3beta", " = 1.2.3beta"]
  ].forEach(function (v) {
    var v0 = v[0]
      , v1 = v[1]
    t.ok(eq(v0, v1), "eq('"+v0+"', '"+v1+"')")
    t.ok(!neq(v0, v1), "!neq('"+v0+"', '"+v1+"')")
    t.ok(cmp(v0, "==", v1), "cmp("+v0+"=="+v1+")")
    t.ok(!cmp(v0, "!=", v1), "!cmp("+v0+"!="+v1+")")
    t.ok(!cmp(v0, "===", v1), "!cmp("+v0+"==="+v1+")")
    t.ok(cmp(v0, "!==", v1), "cmp("+v0+"!=="+v1+")")
    t.ok(!gt(v0, v1), "!gt('"+v0+"', '"+v1+"')")
    t.ok(gte(v0, v1), "gte('"+v0+"', '"+v1+"')")
    t.ok(!lt(v0, v1), "!lt('"+v0+"', '"+v1+"')")
    t.ok(lte(v0, v1), "lte('"+v0+"', '"+v1+"')")
  })
  t.end()
})


test("\nrange tests", function (t) {
; [ ["1.0.0 - 2.0.0", "1.2.3"]
  , ["1.0.0", "1.0.0"]
  , [">=*", "0.2.4"]
  , ["", "1.0.0"]
  , ["*", "1.2.3"]
  , ["*", "v1.2.3-foo"]
  , [">=1.0.0", "1.0.0"]
  , [">=1.0.0", "1.0.1"]
  , [">=1.0.0", "1.1.0"]
  , [">1.0.0", "1.0.1"]
  , [">1.0.0", "1.1.0"]
  , ["<=2.0.0", "2.0.0"]
  , ["<=2.0.0", "1.9999.9999"]
  , ["<=2.0.0", "0.2.9"]
  , ["<2.0.0", "1.9999.9999"]
  , ["<2.0.0", "0.2.9"]
  , [">= 1.0.0", "1.0.0"]
  , [">=  1.0.0", "1.0.1"]
  , [">=   1.0.0", "1.1.0"]
  , ["> 1.0.0", "1.0.1"]
  , [">  1.0.0", "1.1.0"]
  , ["<=   2.0.0", "2.0.0"]
  , ["<= 2.0.0", "1.9999.9999"]
  , ["<=  2.0.0", "0.2.9"]
  , ["<    2.0.0", "1.9999.9999"]
  , ["<\t2.0.0", "0.2.9"]
  , [">=0.1.97", "v0.1.97"]
  , [">=0.1.97", "0.1.97"]
  , ["0.1.20 || 1.2.4", "1.2.4"]
  , [">=0.2.3 || <0.0.1", "0.0.0"]
  , [">=0.2.3 || <0.0.1", "0.2.3"]
  , [">=0.2.3 || <0.0.1", "0.2.4"]
  , ["||", "1.3.4"]
  , ["2.x.x", "2.1.3"]
  , ["1.2.x", "1.2.3"]
  , ["1.2.x || 2.x", "2.1.3"]
  , ["1.2.x || 2.x", "1.2.3"]
  , ["x", "1.2.3"]
  , ["2.*.*", "2.1.3"]
  , ["1.2.*", "1.2.3"]
  , ["1.2.* || 2.*", "2.1.3"]
  , ["1.2.* || 2.*", "1.2.3"]
  , ["*", "1.2.3"]
  , ["2", "2.1.2"]
  , ["2.3", "2.3.1"]
  , ["~2.4", "2.4.0"] // >=2.4.0 <2.5.0
  , ["~2.4", "2.4.5"]
  , ["~>3.2.1", "3.2.2"] // >=3.2.1 <3.3.0
  , ["~1", "1.2.3"] // >=1.0.0 <2.0.0
  , ["~>1", "1.2.3"]
  , ["~> 1", "1.2.3"]
  , ["~1.0", "1.0.2"] // >=1.0.0 <1.1.0
  , ["~ 1.0", "1.0.2"]
  , ["<1", "1.0.0beta"]
  , ["< 1", "1.0.0beta"]
  , [">=1", "1.0.0"]
  , [">= 1", "1.0.0"]
  , ["<1.2", "1.1.1"]
  , ["< 1.2", "1.1.1"]
  , ["1", "1.0.0beta"]
  ].forEach(function (v) {
    t.ok(satisfies(v[1], v[0]), v[0]+" satisfied by "+v[1])
  })
  t.end()
})

test("\nnegative range tests", function (t) {
; [ ["1.0.0 - 2.0.0", "2.2.3"]
  , ["1.0.0", "1.0.1"]
  , [">=1.0.0", "0.0.0"]
  , [">=1.0.0", "0.0.1"]
  , [">=1.0.0", "0.1.0"]
  , [">1.0.0", "0.0.1"]
  , [">1.0.0", "0.1.0"]
  , ["<=2.0.0", "3.0.0"]
  , ["<=2.0.0", "2.9999.9999"]
  , ["<=2.0.0", "2.2.9"]
  , ["<2.0.0", "2.9999.9999"]
  , ["<2.0.0", "2.2.9"]
  , [">=0.1.97", "v0.1.93"]
  , [">=0.1.97", "0.1.93"]
  , ["0.1.20 || 1.2.4", "1.2.3"]
  , [">=0.2.3 || <0.0.1", "0.0.3"]
  , [">=0.2.3 || <0.0.1", "0.2.2"]
  , ["2.x.x", "1.1.3"]
  , ["2.x.x", "3.1.3"]
  , ["1.2.x", "1.3.3"]
  , ["1.2.x || 2.x", "3.1.3"]
  , ["1.2.x || 2.x", "1.1.3"]
  , ["2.*.*", "1.1.3"]
  , ["2.*.*", "3.1.3"]
  , ["1.2.*", "1.3.3"]
  , ["1.2.* || 2.*", "3.1.3"]
  , ["1.2.* || 2.*", "1.1.3"]
  , ["2", "1.1.2"]
  , ["2.3", "2.4.1"]
  , ["~2.4", "2.5.0"] // >=2.4.0 <2.5.0
  , ["~2.4", "2.3.9"]
  , ["~>3.2.1", "3.3.2"] // >=3.2.1 <3.3.0
  , ["~>3.2.1", "3.2.0"] // >=3.2.1 <3.3.0
  , ["~1", "0.2.3"] // >=1.0.0 <2.0.0
  , ["~>1", "2.2.3"]
  , ["~1.0", "1.1.0"] // >=1.0.0 <1.1.0
  , ["<1", "1.0.0"]
  , [">=1.2", "1.1.1"]
  , ["1", "2.0.0beta"]
  ].forEach(function (v) {
    t.ok(!satisfies(v[1], v[0]), v[0]+" not satisfied by "+v[1])
  })
  t.end()
})

test("\nincrement versions test", function (t) {
; [ [ "1.2.3",   "major", "2.0.0"   ]
  , [ "1.2.3",   "minor", "1.3.0"   ]
  , [ "1.2.3",   "patch", "1.2.4"   ]
  , [ "1.2.3",   "build", "1.2.3-1" ]
  , [ "1.2.3-4", "build", "1.2.3-5" ]
  , [ "1.2.3tag",    "major", "2.0.0"   ]
  , [ "1.2.3-tag",   "major", "2.0.0"   ]
  , [ "1.2.3tag",    "build", "1.2.3-1" ]
  , [ "1.2.3-tag",   "build", "1.2.3-1" ]
  , [ "1.2.3-4-tag", "build", "1.2.3-5" ]
  , [ "1.2.3-4tag",  "build", "1.2.3-5" ]
  , [ "1.2.3", "fake",  null ]
  , [ "fake",  "major", null ]
  ].forEach(function (v) {
    t.equal(inc(v[0], v[1]), v[2], "inc("+v[0]+", "+v[1]+") === "+v[2])
  })

  t.end()
})

test("\nreplace stars test", function (t) {
; [ [ "", "" ]
  , [ "*", "" ]
  , [ "> *", "" ]
  , [ "<*", "" ]
  , [ " >=  *", "" ]
  , [ "* || 1.2.3", " || 1.2.3" ]
  ].forEach(function (v) {
    t.equal(replaceStars(v[0]), v[1], "replaceStars("+v[0]+") === "+v[1])
  })

  t.end()
})

test("\nvalid range test", function (t) {
; [ ["1.0.0 - 2.0.0", ">=1.0.0 <=2.0.0"]
  , ["1.0.0", "1.0.0"]
  , [">=*", ""]
  , ["", ""]
  , ["*", ""]
  , ["*", ""]
  , [">=1.0.0", ">=1.0.0"]
  , [">=1.0.0", ">=1.0.0"]
  , [">=1.0.0", ">=1.0.0"]
  , [">1.0.0", ">1.0.0"]
  , [">1.0.0", ">1.0.0"]
  , ["<=2.0.0", "<=2.0.0"]
  , ["1", ">=1.0.0- <2.0.0-"]
  , ["<=2.0.0", "<=2.0.0"]
  , ["<=2.0.0", "<=2.0.0"]
  , ["<2.0.0", "<2.0.0"]
  , ["<2.0.0", "<2.0.0"]
  , [">= 1.0.0", ">=1.0.0"]
  , [">=  1.0.0", ">=1.0.0"]
  , [">=   1.0.0", ">=1.0.0"]
  , ["> 1.0.0", ">1.0.0"]
  , [">  1.0.0", ">1.0.0"]
  , ["<=   2.0.0", "<=2.0.0"]
  , ["<= 2.0.0", "<=2.0.0"]
  , ["<=  2.0.0", "<=2.0.0"]
  , ["<    2.0.0", "<2.0.0"]
  , ["<	2.0.0", "<2.0.0"]
  , [">=0.1.97", ">=0.1.97"]
  , [">=0.1.97", ">=0.1.97"]
  , ["0.1.20 || 1.2.4", "0.1.20||1.2.4"]
  , [">=0.2.3 || <0.0.1", ">=0.2.3||<0.0.1"]
  , [">=0.2.3 || <0.0.1", ">=0.2.3||<0.0.1"]
  , [">=0.2.3 || <0.0.1", ">=0.2.3||<0.0.1"]
  , ["||", "||"]
  , ["2.x.x", ">=2.0.0- <3.0.0-"]
  , ["1.2.x", ">=1.2.0- <1.3.0-"]
  , ["1.2.x || 2.x", ">=1.2.0- <1.3.0-||>=2.0.0- <3.0.0-"]
  , ["1.2.x || 2.x", ">=1.2.0- <1.3.0-||>=2.0.0- <3.0.0-"]
  , ["x", ""]
  , ["2.*.*", null]
  , ["1.2.*", null]
  , ["1.2.* || 2.*", null]
  , ["1.2.* || 2.*", null]
  , ["*", ""]
  , ["2", ">=2.0.0- <3.0.0-"]
  , ["2.3", ">=2.3.0- <2.4.0-"]
  , ["~2.4", ">=2.4.0- <2.5.0-"]
  , ["~2.4", ">=2.4.0- <2.5.0-"]
  , ["~>3.2.1", ">=3.2.1- <3.3.0-"]
  , ["~1", ">=1.0.0- <2.0.0-"]
  , ["~>1", ">=1.0.0- <2.0.0-"]
  , ["~> 1", ">=1.0.0- <2.0.0-"]
  , ["~1.0", ">=1.0.0- <1.1.0-"]
  , ["~ 1.0", ">=1.0.0- <1.1.0-"]
  , ["<1", "<1.0.0"]
  , ["< 1", "<1.0.0"]
  , [">=1", ">=1.0.0"]
  , [">= 1", ">=1.0.0"]
  , ["<1.2", "<1.2.0"]
  , ["< 1.2", "<1.2.0"]
  , ["1", ">=1.0.0- <2.0.0-"]
  ].forEach(function (v) {
    t.equal(validRange(v[0]), v[1], "validRange("+v[0]+") === "+v[1])
  })

  t.end()
})

test("\ncomparators test", function (t) {
; [ ["1.0.0 - 2.0.0", [[">=1.0.0", "<=2.0.0"]] ]
  , ["1.0.0", [["1.0.0"]] ]
  , [">=*", [[">=0.0.0"]] ]
  , ["", [[""]]]
  , ["*", [[""]] ]
  , ["*", [[""]] ]
  , [">=1.0.0", [[">=1.0.0"]] ]
  , [">=1.0.0", [[">=1.0.0"]] ]
  , [">=1.0.0", [[">=1.0.0"]] ]
  , [">1.0.0", [[">1.0.0"]] ]
  , [">1.0.0", [[">1.0.0"]] ]
  , ["<=2.0.0", [["<=2.0.0"]] ]
  , ["1", [[">=1.0.0-", "<2.0.0-"]] ]
  , ["<=2.0.0", [["<=2.0.0"]] ]
  , ["<=2.0.0", [["<=2.0.0"]] ]
  , ["<2.0.0", [["<2.0.0"]] ]
  , ["<2.0.0", [["<2.0.0"]] ]
  , [">= 1.0.0", [[">=1.0.0"]] ]
  , [">=  1.0.0", [[">=1.0.0"]] ]
  , [">=   1.0.0", [[">=1.0.0"]] ]
  , ["> 1.0.0", [[">1.0.0"]] ]
  , [">  1.0.0", [[">1.0.0"]] ]
  , ["<=   2.0.0", [["<=2.0.0"]] ]
  , ["<= 2.0.0", [["<=2.0.0"]] ]
  , ["<=  2.0.0", [["<=2.0.0"]] ]
  , ["<    2.0.0", [["<2.0.0"]] ]
  , ["<\t2.0.0", [["<2.0.0"]] ]
  , [">=0.1.97", [[">=0.1.97"]] ]
  , [">=0.1.97", [[">=0.1.97"]] ]
  , ["0.1.20 || 1.2.4", [["0.1.20"], ["1.2.4"]] ]
  , [">=0.2.3 || <0.0.1", [[">=0.2.3"], ["<0.0.1"]] ]
  , [">=0.2.3 || <0.0.1", [[">=0.2.3"], ["<0.0.1"]] ]
  , [">=0.2.3 || <0.0.1", [[">=0.2.3"], ["<0.0.1"]] ]
  , ["||", [[""], [""]] ]
  , ["2.x.x", [[">=2.0.0-", "<3.0.0-"]] ]
  , ["1.2.x", [[">=1.2.0-", "<1.3.0-"]] ]
  , ["1.2.x || 2.x", [[">=1.2.0-", "<1.3.0-"], [">=2.0.0-", "<3.0.0-"]] ]
  , ["1.2.x || 2.x", [[">=1.2.0-", "<1.3.0-"], [">=2.0.0-", "<3.0.0-"]] ]
  , ["x", [[""]] ]
  , ["2.*.*", [[">=2.0.0-", "<3.0.0-"]] ]
  , ["1.2.*", [[">=1.2.0-", "<1.3.0-"]] ]
  , ["1.2.* || 2.*", [[">=1.2.0-", "<1.3.0-"], [">=2.0.0-", "<3.0.0-"]] ]
  , ["1.2.* || 2.*", [[">=1.2.0-", "<1.3.0-"], [">=2.0.0-", "<3.0.0-"]] ]
  , ["*", [[""]] ]
  , ["2", [[">=2.0.0-", "<3.0.0-"]] ]
  , ["2.3", [[">=2.3.0-", "<2.4.0-"]] ]
  , ["~2.4", [[">=2.4.0-", "<2.5.0-"]] ]
  , ["~2.4", [[">=2.4.0-", "<2.5.0-"]] ]
  , ["~>3.2.1", [[">=3.2.1-", "<3.3.0-"]] ]
  , ["~1", [[">=1.0.0-", "<2.0.0-"]] ]
  , ["~>1", [[">=1.0.0-", "<2.0.0-"]] ]
  , ["~> 1", [[">=1.0.0-", "<2.0.0-"]] ]
  , ["~1.0", [[">=1.0.0-", "<1.1.0-"]] ]
  , ["~ 1.0", [[">=1.0.0-", "<1.1.0-"]] ]
  , ["<1", [["<1.0.0"]] ]
  , ["< 1", [["<1.0.0"]] ]
  , [">=1", [[">=1.0.0"]] ]
  , [">= 1", [[">=1.0.0"]] ]
  , ["<1.2", [["<1.2.0"]] ]
  , ["< 1.2", [["<1.2.0"]] ]
  , ["1", [[">=1.0.0-", "<2.0.0-"]] ]
  ].forEach(function (v) {
    t.equivalent(toComparators(v[0]), v[1], "toComparators("+v[0]+") === "+JSON.stringify(v[1]))
  })

  t.end()
})
