(function() {
  var svg;

  //save off default references
  var d3 = window.d3, topojson = window.topojson;

  var defaultOptions = {
    scope: 'world',
    responsive: false,
    aspectRatio: 0.5625,
    setProjection: setProjection,
    projection: 'equirectangular',
    dataType: 'json',
    data: {},
    done: function() {},
    fills: {
      defaultFill: '#ABDDA4'
    },
    filters: {},
    geographyConfig: {
        dataUrl: null,
        hideAntarctica: true,
        hideHawaiiAndAlaska : false,
        borderWidth: 1,
        borderColor: '#FDFDFD',
        popupTemplate: function(geography, data) {
          return '<div class="hoverinfo"><strong>' + geography.properties.name + '</strong></div>';
        },
        popupOnHover: true,
        highlightOnHover: true,
        highlightFillColor: '#FC8D59',
        highlightBorderColor: 'rgba(250, 15, 160, 0.2)',
        highlightBorderWidth: 2
    },
    projectionConfig: {
      rotation: [97, 0]
    },
    bubblesConfig: {
        borderWidth: 2,
        borderColor: '#FFFFFF',
        popupOnHover: true,
        radius: null,
        popupTemplate: function(geography, data) {
          return '<div class="hoverinfo"><strong>' + data.name + '</strong></div>';
        },
        fillOpacity: 0.75,
        animate: true,
        highlightOnHover: true,
        highlightFillColor: '#FC8D59',
        highlightBorderColor: 'rgba(250, 15, 160, 0.2)',
        highlightBorderWidth: 2,
        highlightFillOpacity: 0.85,
        exitDelay: 100,
        key: JSON.stringify
    },
    arcConfig: {
      strokeColor: '#DD1C77',
      strokeWidth: 1,
      arcSharpness: 1,
      animationSpeed: 600
    }
  };

  /*
    Getter for value. If not declared on datumValue, look up the chain into optionsValue
  */
  function val( datumValue, optionsValue, context ) {
    if ( typeof context === 'undefined' ) {
      context = optionsValue;
      optionsValues = undefined;
    }
    var value = typeof datumValue !== 'undefined' ? datumValue : optionsValue;

    if (typeof value === 'undefined') {
      return  null;
    }

    if ( typeof value === 'function' ) {
      var fnContext = [context];
      if ( context.geography ) {
        fnContext = [context.geography, context.data];
      }
      return value.apply(null, fnContext);
    }
    else {
      return value;
    }
  }

  function addContainer( element, height, width ) {
    this.svg = d3.select( element ).append('svg')
      .attr('width', width || element.offsetWidth)
      .attr('data-width', width || element.offsetWidth)
      .attr('class', 'datamap')
      .attr('height', height || element.offsetHeight)
      .style('overflow', 'hidden'); // IE10+ doesn't respect height/width when map is zoomed in

    if (this.options.responsive) {
      d3.select(this.options.element).style({'position': 'relative', 'padding-bottom': (this.options.aspectRatio*100) + '%'});
      d3.select(this.options.element).select('svg').style({'position': 'absolute', 'width': '100%', 'height': '100%'});
      d3.select(this.options.element).select('svg').select('g').selectAll('path').style('vector-effect', 'non-scaling-stroke');

    }

    return this.svg;
  }

  // setProjection takes the svg element and options
  function setProjection( element, options ) {
    var width = options.width || element.offsetWidth;
    var height = options.height || element.offsetHeight;
    var projection, path;
    var svg = this.svg;

    if ( options && typeof options.scope === 'undefined') {
      options.scope = 'world';
    }

    if ( options.scope === 'usa' ) {
      projection = d3.geo.albersUsa()
        .scale(width)
        .translate([width / 2, height / 2]);
    }
    else if ( options.scope === 'world' ) {
      projection = d3.geo[options.projection]()
        .scale((width + 1) / 2 / Math.PI)
        .translate([width / 2, height / (options.projection === "mercator" ? 1.45 : 1.8)]);
    }

    if ( options.projection === 'orthographic' ) {

      svg.append("defs").append("path")
        .datum({type: "Sphere"})
        .attr("id", "sphere")
        .attr("d", path);

      svg.append("use")
          .attr("class", "stroke")
          .attr("xlink:href", "#sphere");

      svg.append("use")
          .attr("class", "fill")
          .attr("xlink:href", "#sphere");
      projection.scale(250).clipAngle(90).rotate(options.projectionConfig.rotation)
    }

    path = d3.geo.path()
      .projection( projection );

    return {path: path, projection: projection};
  }

  function addStyleBlock() {
    if ( d3.select('.datamaps-style-block').empty() ) {
      d3.select('head').append('style').attr('class', 'datamaps-style-block')
      .html('.datamap path.datamaps-graticule { fill: none; stroke: #777; stroke-width: 0.5px; stroke-opacity: .5; pointer-events: none; } .datamap .labels {pointer-events: none;} .datamap path {stroke: #FFFFFF; stroke-width: 1px;} .datamaps-legend dt, .datamaps-legend dd { float: left; margin: 0 3px 0 0;} .datamaps-legend dd {width: 20px; margin-right: 6px; border-radius: 3px;} .datamaps-legend {padding-bottom: 20px; z-index: 1001; position: absolute; left: 4px; font-size: 12px; font-family: "Helvetica Neue", Helvetica, Arial, sans-serif;} .datamaps-hoverover {display: none; font-family: "Helvetica Neue", Helvetica, Arial, sans-serif; } .hoverinfo {padding: 4px; border-radius: 1px; background-color: #FFF; box-shadow: 1px 1px 5px #CCC; font-size: 12px; border: 1px solid #CCC; } .hoverinfo hr {border:1px dotted #CCC; }');
    }
  }

  function drawSubunits( data ) {
    var fillData = this.options.fills,
        colorCodeData = this.options.data || {},
        geoConfig = this.options.geographyConfig;


    var subunits = this.svg.select('g.datamaps-subunits');
    if ( subunits.empty() ) {
      subunits = this.addLayer('datamaps-subunits', null, true);
    }

    var geoData = topojson.feature( data, data.objects[ this.options.scope ] ).features;
    if ( geoConfig.hideAntarctica ) {
      geoData = geoData.filter(function(feature) {
        return feature.id !== "ATA";
      });
    }

    if ( geoConfig.hideHawaiiAndAlaska ) {
      geoData = geoData.filter(function(feature) {
        return feature.id !== "HI" && feature.id !== 'AK';
      });
    }

    var geo = subunits.selectAll('path.datamaps-subunit').data( geoData );

    geo.enter()
      .append('path')
      .attr('d', this.path)
      .attr('class', function(d) {
        return 'datamaps-subunit ' + d.id;
      })
      .attr('data-info', function(d) {
        return JSON.stringify( colorCodeData[d.id]);
      })
      .style('fill', function(d) {
        //if fillKey - use that
        //otherwise check 'fill'
        //otherwise check 'defaultFill'
        var fillColor;

        var datum = colorCodeData[d.id];
        if ( datum && datum.fillKey ) {
          fillColor = fillData[ val(datum.fillKey, {data: colorCodeData[d.id], geography: d}) ];
        }

        if ( typeof fillColor === 'undefined' ) {
          fillColor = val(datum && datum.fillColor, fillData.defaultFill, {data: colorCodeData[d.id], geography: d});
        }

        return fillColor;
      })
      .style('stroke-width', geoConfig.borderWidth)
      .style('stroke', geoConfig.borderColor);
  }

  function handleGeographyConfig () {
    var hoverover;
    var svg = this.svg;
    var self = this;
    var options = this.options.geographyConfig;

    if ( options.highlightOnHover || options.popupOnHover ) {
      svg.selectAll('.datamaps-subunit')
        .on('mouseover', function(d) {
          var $this = d3.select(this);
          var datum = self.options.data[d.id] || {};
          if ( options.highlightOnHover ) {
            var previousAttributes = {
              'fill':  $this.style('fill'),
              'stroke': $this.style('stroke'),
              'stroke-width': $this.style('stroke-width'),
              'fill-opacity': $this.style('fill-opacity')
            };

            $this
              .style('fill', val(datum.highlightFillColor, options.highlightFillColor, datum))
              .style('stroke', val(datum.highlightBorderColor, options.highlightBorderColor, datum))
              .style('stroke-width', val(datum.highlightBorderWidth, options.highlightBorderWidth, datum))
              .style('fill-opacity', val(datum.highlightFillOpacity, options.highlightFillOpacity, datum))
              .attr('data-previousAttributes', JSON.stringify(previousAttributes));

            //as per discussion on https://github.com/markmarkoh/datamaps/issues/19
            if ( ! /((MSIE)|(Trident))/.test(navigator.userAgent) ) {
             moveToFront.call(this);
            }
          }

          if ( options.popupOnHover ) {
            self.updatePopup($this, d, options, svg);
          }
        })
        .on('mouseout', function() {
          var $this = d3.select(this);

          if (options.highlightOnHover) {
            //reapply previous attributes
            var previousAttributes = JSON.parse( $this.attr('data-previousAttributes') );
            for ( var attr in previousAttributes ) {
              $this.style(attr, previousAttributes[attr]);
            }
          }
          $this.on('mousemove', null);
          d3.selectAll('.datamaps-hoverover').style('display', 'none');
        });
    }

    function moveToFront() {
      this.parentNode.appendChild(this);
    }
  }

  //plugin to add a simple map legend
  function addLegend(layer, data, options) {
    data = data || {};
    if ( !this.options.fills ) {
      return;
    }

    var html = '<dl>';
    var label = '';
    if ( data.legendTitle ) {
      html = '<h2>' + data.legendTitle + '</h2>' + html;
    }
    for ( var fillKey in this.options.fills ) {

      if ( fillKey === 'defaultFill') {
        if (! data.defaultFillName ) {
          continue;
        }
        label = data.defaultFillName;
      } else {
        if (data.labels && data.labels[fillKey]) {
          label = data.labels[fillKey];
        } else {
          label= fillKey + ': ';
        }
      }
      html += '<dt>' + label + '</dt>';
      html += '<dd style="background-color:' +  this.options.fills[fillKey] + '">&nbsp;</dd>';
    }
    html += '</dl>';

    var hoverover = d3.select( this.options.element ).append('div')
      .attr('class', 'datamaps-legend')
      .html(html);
  }

    function addGraticule ( layer, options ) {
      var graticule = d3.geo.graticule();
      this.svg.insert("path", '.datamaps-subunits')
        .datum(graticule)
        .attr("class", "datamaps-graticule")
        .attr("d", this.path);
  }

  function handleArcs (layer, data, options) {
    var self = this,
        svg = this.svg;

    if ( !data || (data && !data.slice) ) {
      throw "Datamaps Error - arcs must be an array";
    }

    // For some reason arc options were put in an `options` object instead of the parent arc
    // I don't like this, so to match bubbles and other plugins I'm moving it
    // This is to keep backwards compatability
    for ( var i = 0; i < data.length; i++ ) {
      data[i] = defaults(data[i], data[i].options);
      delete data[i].options;
    }

    if ( typeof options === "undefined" ) {
      options = defaultOptions.arcConfig;
    }

    var arcs = layer.selectAll('path.datamaps-arc').data( data, JSON.stringify );

    var path = d3.geo.path()
        .projection(self.projection);

    arcs
      .enter()
        .append('svg:path')
        .attr('class', 'datamaps-arc')
        .style('stroke-linecap', 'round')
        .style('stroke', function(datum) {
          return val(datum.strokeColor, options.strokeColor, datum);
        })
        .style('fill', 'none')
        .style('stroke-width', function(datum) {
            return val(datum.strokeWidth, options.strokeWidth, datum);
        })
        .attr('d', function(datum) {
            var originXY = self.latLngToXY(val(datum.origin.latitude, datum), val(datum.origin.longitude, datum))
            var destXY = self.latLngToXY(val(datum.destination.latitude, datum), val(datum.destination.longitude, datum));
            var midXY = [ (originXY[0] + destXY[0]) / 2, (originXY[1] + destXY[1]) / 2];
            if (options.greatArc) {
                  // TODO: Move this to inside `if` clause when setting attr `d`
              var greatArc = d3.geo.greatArc()
                  .source(function(d) { return [val(d.origin.longitude, d), val(d.origin.latitude, d)]; })
                  .target(function(d) { return [val(d.destination.longitude, d), val(d.destination.latitude, d)]; });

              return path(greatArc(datum))
            }
            var sharpness = val(datum.arcSharpness, options.arcSharpness, datum);
            return "M" + originXY[0] + ',' + originXY[1] + "S" + (midXY[0] + (50 * sharpness)) + "," + (midXY[1] - (75 * sharpness)) + "," + destXY[0] + "," + destXY[1];
        })
        .transition()
          .delay(100)
          .style('fill', function(datum) {
            /*
              Thank you Jake Archibald, this is awesome.
              Source: http://jakearchibald.com/2013/animated-line-drawing-svg/
            */
            var length = this.getTotalLength();
            this.style.transition = this.style.WebkitTransition = 'none';
            this.style.strokeDasharray = length + ' ' + length;
            this.style.strokeDashoffset = length;
            this.getBoundingClientRect();
            this.style.transition = this.style.WebkitTransition = 'stroke-dashoffset ' + val(datum.animationSpeed, options.animationSpeed, datum) + 'ms ease-out';
            this.style.strokeDashoffset = '0';
            return 'none';
          })

    arcs.exit()
      .transition()
      .style('opacity', 0)
      .remove();
  }

  function handleLabels ( layer, options ) {
    var self = this;
    options = options || {};
    var labelStartCoodinates = this.projection([-67.707617, 42.722131]);
    this.svg.selectAll(".datamaps-subunit")
      .attr("data-foo", function(d) {
        var center = self.path.centroid(d);
        var xOffset = 7.5, yOffset = 5;

        if ( ["FL", "KY", "MI"].indexOf(d.id) > -1 ) xOffset = -2.5;
        if ( d.id === "NY" ) xOffset = -1;
        if ( d.id === "MI" ) yOffset = 18;
        if ( d.id === "LA" ) xOffset = 13;

        var x,y;

        x = center[0] - xOffset;
        y = center[1] + yOffset;

        var smallStateIndex = ["VT", "NH", "MA", "RI", "CT", "NJ", "DE", "MD", "DC"].indexOf(d.id);
        if ( smallStateIndex > -1) {
          var yStart = labelStartCoodinates[1];
          x = labelStartCoodinates[0];
          y = yStart + (smallStateIndex * (2+ (options.fontSize || 12)));
          layer.append("line")
            .attr("x1", x - 3)
            .attr("y1", y - 5)
            .attr("x2", center[0])
            .attr("y2", center[1])
            .style("stroke", options.labelColor || "#000")
            .style("stroke-width", options.lineWidth || 1)
        }

        layer.append("text")
          .attr("x", x)
          .attr("y", y)
          .style("font-size", (options.fontSize || 10) + 'px')
          .style("font-family", options.fontFamily || "Verdana")
          .style("fill", options.labelColor || "#000")
          .text( d.id );
        return "bar";
      });
  }


  function handleBubbles (layer, data, options ) {
    var self = this,
        fillData = this.options.fills,
        filterData = this.options.filters,
        svg = this.svg;

    if ( !data || (data && !data.slice) ) {
      throw "Datamaps Error - bubbles must be an array";
    }

    var bubbles = layer.selectAll('circle.datamaps-bubble').data( data, options.key );

    bubbles
      .enter()
        .append('svg:circle')
        .attr('class', 'datamaps-bubble')
        .attr('cx', function ( datum ) {
          var latLng;
          if ( datumHasCoords(datum) ) {
            latLng = self.latLngToXY(datum.latitude, datum.longitude);
          }
          else if ( datum.centered ) {
            latLng = self.path.centroid(svg.select('path.' + datum.centered).data()[0]);
          }
          if ( latLng ) return latLng[0];
        })
        .attr('cy', function ( datum ) {
          var latLng;
          if ( datumHasCoords(datum) ) {
            latLng = self.latLngToXY(datum.latitude, datum.longitude);
          }
          else if ( datum.centered ) {
            latLng = self.path.centroid(svg.select('path.' + datum.centered).data()[0]);
          }
          if ( latLng ) return latLng[1];
        })
        .attr('r', function(datum) {
          // if animation enabled start with radius 0, otherwise use full size.
          return options.animate ? 0 : val(datum.radius, options.radius, datum);
        })
        .attr('data-info', function(d) {
          return JSON.stringify(d);
        })
        .attr('filter', function (datum) {
          var filterKey = filterData[ val(datum.filterKey, options.filterKey, datum) ];

          if (filterKey) {
            return filterKey;
          }
        })
        .style('stroke', function ( datum ) {
          return val(datum.borderColor, options.borderColor, datum);
        })
        .style('stroke-width', function ( datum ) {
          return val(datum.borderWidth, options.borderWidth, datum);
        })
        .style('fill-opacity', function ( datum ) {
          return val(datum.fillOpacity, options.fillOpacity, datum);
        })
        .style('fill', function ( datum ) {
          var fillColor = fillData[ val(datum.fillKey, options.fillKey, datum) ];
          return fillColor || fillData.defaultFill;
        })
        .on('mouseover', function ( datum ) {
          var $this = d3.select(this);

          if (options.highlightOnHover) {
            //save all previous attributes for mouseout
            var previousAttributes = {
              'fill':  $this.style('fill'),
              'stroke': $this.style('stroke'),
              'stroke-width': $this.style('stroke-width'),
              'fill-opacity': $this.style('fill-opacity')
            };

            $this
              .style('fill', val(datum.highlightFillColor, options.highlightFillColor, datum))
              .style('stroke', val(datum.highlightBorderColor, options.highlightBorderColor, datum))
              .style('stroke-width', val(datum.highlightBorderWidth, options.highlightBorderWidth, datum))
              .style('fill-opacity', val(datum.highlightFillOpacity, options.highlightFillOpacity, datum))
              .attr('data-previousAttributes', JSON.stringify(previousAttributes));
          }

          if (options.popupOnHover) {
            self.updatePopup($this, datum, options, svg);
          }
        })
        .on('mouseout', function ( datum ) {
          var $this = d3.select(this);

          if (options.highlightOnHover) {
            //reapply previous attributes
            var previousAttributes = JSON.parse( $this.attr('data-previousAttributes') );
            for ( var attr in previousAttributes ) {
              $this.style(attr, previousAttributes[attr]);
            }
          }

          d3.selectAll('.datamaps-hoverover').style('display', 'none');
        })

    bubbles.transition()
      .duration(400)
      .attr('r', function ( datum ) {
        return val(datum.radius, options.radius, datum);
      });

    bubbles.exit()
      .transition()
        .delay(options.exitDelay)
        .attr("r", 0)
        .remove();

    function datumHasCoords (datum) {
      return typeof datum !== 'undefined' && typeof datum.latitude !== 'undefined' && typeof datum.longitude !== 'undefined';
    }
  }

  //stolen from underscore.js
  function defaults(obj) {
    Array.prototype.slice.call(arguments, 1).forEach(function(source) {
      if (source) {
        for (var prop in source) {
          if (obj[prop] == null) obj[prop] = source[prop];
        }
      }
    });
    return obj;
  }
  /**************************************
             Public Functions
  ***************************************/

  function Datamap( options ) {

    if ( typeof d3 === 'undefined' || typeof topojson === 'undefined' ) {
      throw new Error('Include d3.js (v3.0.3 or greater) and topojson on this page before creating a new map');
   }
    //set options for global use
    this.options = defaults(options, defaultOptions);
    this.options.geographyConfig = defaults(options.geographyConfig, defaultOptions.geographyConfig);
    this.options.projectionConfig = defaults(options.projectionConfig, defaultOptions.projectionConfig);
    this.options.bubblesConfig = defaults(options.bubblesConfig, defaultOptions.bubblesConfig);
    this.options.arcConfig = defaults(options.arcConfig, defaultOptions.arcConfig);

    //add the SVG container
    if ( d3.select( this.options.element ).select('svg').length > 0 ) {
      addContainer.call(this, this.options.element, this.options.height, this.options.width );
    }

    /* Add core plugins to this instance */
    this.addPlugin('bubbles', handleBubbles);
    this.addPlugin('legend', addLegend);
    this.addPlugin('arc', handleArcs);
    this.addPlugin('labels', handleLabels);
    this.addPlugin('graticule', addGraticule);

    //append style block with basic hoverover styles
    if ( ! this.options.disableDefaultStyles ) {
      addStyleBlock();
    }

    return this.draw();
  }

  // resize map
  Datamap.prototype.resize = function () {

    var self = this;
    var options = self.options;

    if (options.responsive) {
      var newsize = options.element.clientWidth,
          oldsize = d3.select( options.element).select('svg').attr('data-width');

      d3.select(options.element).select('svg').selectAll('g').attr('transform', 'scale(' + (newsize / oldsize) + ')');
    }
  }

  // actually draw the features(states & countries)
  Datamap.prototype.draw = function() {
    //save off in a closure
    var self = this;
    var options = self.options;

    //set projections and paths based on scope
    var pathAndProjection = options.setProjection.apply(self, [options.element, options] );

    this.path = pathAndProjection.path;
    this.projection = pathAndProjection.projection;

    //if custom URL for topojson data, retrieve it and render
    if ( options.geographyConfig.dataUrl ) {
      d3.json( options.geographyConfig.dataUrl, function(error, results) {
        if ( error ) throw new Error(error);
        self.customTopo = results;
        draw( results );
      });
    }
    else {
      draw( this[options.scope + 'Topo'] || options.geographyConfig.dataJson);
    }

    return this;

      function draw (data) {
        // if fetching remote data, draw the map first then call `updateChoropleth`
        if ( self.options.dataUrl ) {
          //allow for csv or json data types
          d3[self.options.dataType](self.options.dataUrl, function(data) {
            //in the case of csv, transform data to object
            if ( self.options.dataType === 'csv' && (data && data.slice) ) {
              var tmpData = {};
              for(var i = 0; i < data.length; i++) {
                tmpData[data[i].id] = data[i];
              }
              data = tmpData;
            }
            Datamaps.prototype.updateChoropleth.call(self, data);
          });
        }
        drawSubunits.call(self, data);
        handleGeographyConfig.call(self);

        if ( self.options.geographyConfig.popupOnHover || self.options.bubblesConfig.popupOnHover) {
          hoverover = d3.select( self.options.element ).append('div')
            .attr('class', 'datamaps-hoverover')
            .style('z-index', 10001)
            .style('position', 'absolute');
        }

        //fire off finished callback
        self.options.done(self);
      }
  };
  /**************************************
                TopoJSON
  ***************************************/
  Datamap.prototype.worldTopo = '__WORLD__';
  Datamap.prototype.abwTopo = '__ABW__';
  Datamap.prototype.afgTopo = '__AFG__';
  Datamap.prototype.agoTopo = '__AGO__';
  Datamap.prototype.aiaTopo = '__AIA__';
  Datamap.prototype.albTopo = '__ALB__';
  Datamap.prototype.aldTopo = '__ALD__';
  Datamap.prototype.andTopo = '__AND__';
  Datamap.prototype.areTopo = '__ARE__';
  Datamap.prototype.argTopo = '__ARG__';
  Datamap.prototype.armTopo = '__ARM__';
  Datamap.prototype.asmTopo = '__ASM__';
  Datamap.prototype.ataTopo = '__ATA__';
  Datamap.prototype.atcTopo = '__ATC__';
  Datamap.prototype.atfTopo = '__ATF__';
  Datamap.prototype.atgTopo = '__ATG__';
  Datamap.prototype.ausTopo = '__AUS__';
  Datamap.prototype.autTopo = '__AUT__';
  Datamap.prototype.azeTopo = '__AZE__';
  Datamap.prototype.bdiTopo = '__BDI__';
  Datamap.prototype.belTopo = '__BEL__';
  Datamap.prototype.benTopo = '__BEN__';
  Datamap.prototype.bfaTopo = '__BFA__';
  Datamap.prototype.bgdTopo = '__BGD__';
  Datamap.prototype.bgrTopo = '__BGR__';
  Datamap.prototype.bhrTopo = '__BHR__';
  Datamap.prototype.bhsTopo = '__BHS__';
  Datamap.prototype.bihTopo = '__BIH__';
  Datamap.prototype.bjnTopo = '__BJN__';
  Datamap.prototype.blmTopo = '__BLM__';
  Datamap.prototype.blrTopo = '__BLR__';
  Datamap.prototype.blzTopo = '__BLZ__';
  Datamap.prototype.bmuTopo = '__BMU__';
  Datamap.prototype.bolTopo = '__BOL__';
  Datamap.prototype.braTopo = '__BRA__';
  Datamap.prototype.brbTopo = '__BRB__';
  Datamap.prototype.brnTopo = '__BRN__';
  Datamap.prototype.btnTopo = '__BTN__';
  Datamap.prototype.norTopo = '__NOR__';
  Datamap.prototype.bwaTopo = '__BWA__';
  Datamap.prototype.cafTopo = '__CAF__';
  Datamap.prototype.canTopo = '__CAN__';
  Datamap.prototype.cheTopo = '__CHE__';
  Datamap.prototype.chlTopo = '__CHL__';
  Datamap.prototype.chnTopo = '__CHN__';
  Datamap.prototype.civTopo = '__CIV__';
  Datamap.prototype.clpTopo = '__CLP__';
  Datamap.prototype.cmrTopo = '__CMR__';
  Datamap.prototype.codTopo = '__COD__';
  Datamap.prototype.cogTopo = '__COG__';
  Datamap.prototype.cokTopo = '__COK__';
  Datamap.prototype.colTopo = '__COL__';
  Datamap.prototype.comTopo = '__COM__';
  Datamap.prototype.cpvTopo = '__CPV__';
  Datamap.prototype.criTopo = '__CRI__';
  Datamap.prototype.csiTopo = '__CSI__';
  Datamap.prototype.cubTopo = '__CUB__';
  Datamap.prototype.cuwTopo = '__CUW__';
  Datamap.prototype.cymTopo = '__CYM__';
  Datamap.prototype.cynTopo = '__CYN__';
  Datamap.prototype.cypTopo = '__CYP__';
  Datamap.prototype.czeTopo = '__CZE__';
  Datamap.prototype.deuTopo = '__DEU__';
  Datamap.prototype.djiTopo = '__DJI__';
  Datamap.prototype.dmaTopo = '__DMA__';
  Datamap.prototype.dnkTopo = '__DNK__';
  Datamap.prototype.domTopo = '__DOM__';
  Datamap.prototype.dzaTopo = '__DZA__';
  Datamap.prototype.ecuTopo = '__ECU__';
  Datamap.prototype.egyTopo = '__EGY__';
  Datamap.prototype.eriTopo = '__ERI__';
  Datamap.prototype.esbTopo = '__ESB__';
  Datamap.prototype.espTopo = '__ESP__';
  Datamap.prototype.estTopo = '__EST__';
  Datamap.prototype.ethTopo = '__ETH__';
  Datamap.prototype.finTopo = '__FIN__';
  Datamap.prototype.fjiTopo = '__FJI__';
  Datamap.prototype.flkTopo = '__FLK__';
  Datamap.prototype.fraTopo = '__FRA__';
  Datamap.prototype.froTopo = '__FRO__';
  Datamap.prototype.fsmTopo = '__FSM__';
  Datamap.prototype.gabTopo = '__GAB__';
  Datamap.prototype.psxTopo = '__PSX__';
  Datamap.prototype.gbrTopo = '__GBR__';
  Datamap.prototype.geoTopo = '__GEO__';
  Datamap.prototype.ggyTopo = '__GGY__';
  Datamap.prototype.ghaTopo = '__GHA__';
  Datamap.prototype.gibTopo = '__GIB__';
  Datamap.prototype.ginTopo = '__GIN__';
  Datamap.prototype.gmbTopo = '__GMB__';
  Datamap.prototype.gnbTopo = '__GNB__';
  Datamap.prototype.gnqTopo = '__GNQ__';
  Datamap.prototype.grcTopo = '__GRC__';
  Datamap.prototype.grdTopo = '__GRD__';
  Datamap.prototype.grlTopo = '__GRL__';
  Datamap.prototype.gtmTopo = '__GTM__';
  Datamap.prototype.gumTopo = '__GUM__';
  Datamap.prototype.guyTopo = '__GUY__';
  Datamap.prototype.hkgTopo = '__HKG__';
  Datamap.prototype.hmdTopo = '__HMD__';
  Datamap.prototype.hndTopo = '__HND__';
  Datamap.prototype.hrvTopo = '__HRV__';
  Datamap.prototype.htiTopo = '__HTI__';
  Datamap.prototype.hunTopo = '__HUN__';
  Datamap.prototype.idnTopo = '__IDN__';
  Datamap.prototype.imnTopo = '__IMN__';
  Datamap.prototype.indTopo = '__IND__';
  Datamap.prototype.ioaTopo = '__IOA__';
  Datamap.prototype.iotTopo = '__IOT__';
  Datamap.prototype.irlTopo = '__IRL__';
  Datamap.prototype.irnTopo = '__IRN__';
  Datamap.prototype.irqTopo = '__IRQ__';
  Datamap.prototype.islTopo = '__ISL__';
  Datamap.prototype.isrTopo = '__ISR__';
  Datamap.prototype.itaTopo = '__ITA__';
  Datamap.prototype.jamTopo = '__JAM__';
  Datamap.prototype.jeyTopo = '__JEY__';
  Datamap.prototype.jorTopo = '__JOR__';
  Datamap.prototype.jpnTopo = '__JPN__';
  Datamap.prototype.kabTopo = '__KAB__';
  Datamap.prototype.kasTopo = '__KAS__';
  Datamap.prototype.kazTopo = '__KAZ__';
  Datamap.prototype.kenTopo = '__KEN__';
  Datamap.prototype.kgzTopo = '__KGZ__';
  Datamap.prototype.khmTopo = '__KHM__';
  Datamap.prototype.kirTopo = '__KIR__';
  Datamap.prototype.knaTopo = '__KNA__';
  Datamap.prototype.korTopo = '__KOR__';
  Datamap.prototype.kosTopo = '__KOS__';
  Datamap.prototype.kwtTopo = '__KWT__';
  Datamap.prototype.laoTopo = '__LAO__';
  Datamap.prototype.lbnTopo = '__LBN__';
  Datamap.prototype.lbrTopo = '__LBR__';
  Datamap.prototype.lbyTopo = '__LBY__';
  Datamap.prototype.lcaTopo = '__LCA__';
  Datamap.prototype.lieTopo = '__LIE__';
  Datamap.prototype.lkaTopo = '__LKA__';
  Datamap.prototype.lsoTopo = '__LSO__';
  Datamap.prototype.ltuTopo = '__LTU__';
  Datamap.prototype.luxTopo = '__LUX__';
  Datamap.prototype.lvaTopo = '__LVA__';
  Datamap.prototype.macTopo = '__MAC__';
  Datamap.prototype.mafTopo = '__MAF__';
  Datamap.prototype.marTopo = '__MAR__';
  Datamap.prototype.mcoTopo = '__MCO__';
  Datamap.prototype.mdaTopo = '__MDA__';
  Datamap.prototype.mdgTopo = '__MDG__';
  Datamap.prototype.mdvTopo = '__MDV__';
  Datamap.prototype.mexTopo = '__MEX__';
  Datamap.prototype.mhlTopo = '__MHL__';
  Datamap.prototype.mkdTopo = '__MKD__';
  Datamap.prototype.mliTopo = '__MLI__';
  Datamap.prototype.mltTopo = '__MLT__';
  Datamap.prototype.mmrTopo = '__MMR__';
  Datamap.prototype.mneTopo = '__MNE__';
  Datamap.prototype.mngTopo = '__MNG__';
  Datamap.prototype.mnpTopo = '__MNP__';
  Datamap.prototype.mozTopo = '__MOZ__';
  Datamap.prototype.mrtTopo = '__MRT__';
  Datamap.prototype.msrTopo = '__MSR__';
  Datamap.prototype.musTopo = '__MUS__';
  Datamap.prototype.mwiTopo = '__MWI__';
  Datamap.prototype.mysTopo = '__MYS__';
  Datamap.prototype.namTopo = '__NAM__';
  Datamap.prototype.nclTopo = '__NCL__';
  Datamap.prototype.nerTopo = '__NER__';
  Datamap.prototype.nfkTopo = '__NFK__';
  Datamap.prototype.ngaTopo = '__NGA__';
  Datamap.prototype.nicTopo = '__NIC__';
  Datamap.prototype.niuTopo = '__NIU__';
  Datamap.prototype.nldTopo = '__NLD__';
  Datamap.prototype.nplTopo = '__NPL__';
  Datamap.prototype.nruTopo = '__NRU__';
  Datamap.prototype.nulTopo = '__NUL__';
  Datamap.prototype.nzlTopo = '__NZL__';
  Datamap.prototype.omnTopo = '__OMN__';
  Datamap.prototype.pakTopo = '__PAK__';
  Datamap.prototype.panTopo = '__PAN__';
  Datamap.prototype.pcnTopo = '__PCN__';
  Datamap.prototype.perTopo = '__PER__';
  Datamap.prototype.pgaTopo = '__PGA__';
  Datamap.prototype.phlTopo = '__PHL__';
  Datamap.prototype.plwTopo = '__PLW__';
  Datamap.prototype.pngTopo = '__PNG__';
  Datamap.prototype.polTopo = '__POL__';
  Datamap.prototype.priTopo = '__PRI__';
  Datamap.prototype.prkTopo = '__PRK__';
  Datamap.prototype.prtTopo = '__PRT__';
  Datamap.prototype.pryTopo = '__PRY__';
  Datamap.prototype.pyfTopo = '__PYF__';
  Datamap.prototype.qatTopo = '__QAT__';
  Datamap.prototype.rouTopo = '__ROU__';
  Datamap.prototype.rusTopo = {"type":"Topology","objects":{"RUS":{"type":"GeometryCollection","geometries":[{"type":"MultiPolygon","properties":{"name":null},"id":"-99","arcs":[[[0]],[[1]],[[2]],[[3]],[[4]],[[5]],[[6]],[[7]],[[8]],[[9]],[[10]],[[11]],[[12]],[[13]],[[14]],[[15]],[[16]],[[17]],[[18]],[[19]],[[20]],[[21]],[[22]],[[23]],[[24]],[[25]],[[26]],[[27]],[[28]],[[29]],[[30]]]},{"type":"Polygon","properties":{"name":"Tomsk"},"id":"RU.TO","arcs":[[31,32,33,34,35,36]]},{"type":"Polygon","properties":{"name":"Adygey"},"id":"RU.AD","arcs":[[37]]},{"type":"Polygon","properties":{"name":"Karachay-Cherkess"},"id":"RU.KC","arcs":[[38,39,40,41]]},{"type":"Polygon","properties":{"name":"Ingush"},"id":"RU.IN","arcs":[[42,43,44]]},{"type":"Polygon","properties":{"name":"Kabardin-Balkar"},"id":"RU.KB","arcs":[[45,46,-39,47]]},{"type":"Polygon","properties":{"name":"North Ossetia"},"id":"RU.NO","arcs":[[48,-45,49,-46,50]]},{"type":"Polygon","properties":{"name":"Stavropol'"},"id":"RU.ST","arcs":[[51,52,-51,-48,-42,53,54,55]]},{"type":"MultiPolygon","properties":{"name":"Chukchi Autonomous Okrug"},"id":"RU.","arcs":[[[56]],[[57]],[[58]],[[59]],[[60]],[[61]],[[62]],[[63]],[[64]],[[65,66,67,68]],[[69]],[[70]],[[71]]]},{"type":"Polygon","properties":{"name":"Kaliningrad"},"id":"RU.KN","arcs":[[72]]},{"type":"MultiPolygon","properties":{"name":"Murmansk"},"id":"RU.MM","arcs":[[[73,74]],[[75]],[[76]],[[77,78]],[[79]]]},{"type":"Polygon","properties":{"name":"Novgorod"},"id":"RU.NG","arcs":[[80,81,82,83]]},{"type":"Polygon","properties":{"name":"Pskov"},"id":"RU.PS","arcs":[[-83,84,85,86,87]]},{"type":"MultiPolygon","properties":{"name":"Leningrad"},"id":"RU.LN","arcs":[[[88]],[[89]],[[90]],[[91]],[[92]],[[93]],[[94]],[[95]],[[96]],[[97]],[[98]],[[99,-84,-88,100,101,102,103]]]},{"type":"MultiPolygon","properties":{"name":"City of St. Petersburg"},"id":"RU.SP","arcs":[[[104]],[[105,-102]]]},{"type":"Polygon","properties":{"name":"Bryansk"},"id":"RU.BR","arcs":[[106,107,108,109,110]]},{"type":"Polygon","properties":{"name":"Smolensk"},"id":"RU.SM","arcs":[[111,112,-110,113,-86,114]]},{"type":"MultiPolygon","properties":{"name":"Karelia"},"id":"RU.KI","arcs":[[[115]],[[116]],[[117]],[[118]],[[119,-74,120,121,122,-104,123,-78]]]},{"type":"MultiPolygon","properties":{"name":"Arkhangel'sk"},"id":"RU.VO","arcs":[[[124]],[[125]],[[126]],[[127]],[[128,129,130,131,-122,132]],[[133]],[[134]],[[135]],[[136]],[[137]],[[138]],[[139]],[[140]],[[141]],[[142]],[[143]],[[144]],[[145]],[[146]],[[147]],[[148]],[[149]],[[150]],[[151]],[[152]],[[153]],[[154]],[[155]],[[156]],[[157]],[[158]],[[159]],[[160]],[[161]],[[162]],[[163]],[[164]],[[165]],[[166]],[[167]],[[168]],[[169]],[[170]],[[171]],[[172]],[[173]],[[174]],[[175]]]},{"type":"Polygon","properties":{"name":"Ivanovo"},"id":"RU.IV","arcs":[[176,177,178,179]]},{"type":"MultiPolygon","properties":{"name":"Kostroma"},"id":"RU.KT","arcs":[[[180]],[[181,182,-180,183,184]]]},{"type":"Polygon","properties":{"name":"Nizhegorod"},"id":"RU.NZ","arcs":[[185,186,187,188,189,190,-177,-183]]},{"type":"Polygon","properties":{"name":"Tver'"},"id":"RU.TV","arcs":[[191,192,-115,-85,-82,193]]},{"type":"Polygon","properties":{"name":"Vologda"},"id":"RU.VO","arcs":[[-132,194,-185,195,-194,-81,-100,-123],[-125],[-181]]},{"type":"Polygon","properties":{"name":"Yaroslavl'"},"id":"RU.YS","arcs":[[-184,-179,196,197,-192,-196]]},{"type":"Polygon","properties":{"name":"Kaluga"},"id":"RU.KG","arcs":[[198,199,200,201,-111,-113,202]]},{"type":"Polygon","properties":{"name":"Kursk"},"id":"RU.KS","arcs":[[203,204,205,206,-108,207]]},{"type":"Polygon","properties":{"name":"Lipetsk"},"id":"RU.LP","arcs":[[208,209,-204,210,211,212]]},{"type":"Polygon","properties":{"name":"Moskovsskaya"},"id":"RU.MS","arcs":[[-198,213,214,215,-200,216,-203,-112,-193],[217],[218]]},{"type":"MultiPolygon","properties":{"name":"Moskva"},"id":"RU.MS","arcs":[[[-219]],[[-199,-217]],[[-218]]]},{"type":"Polygon","properties":{"name":"Orel"},"id":"RU.OL","arcs":[[219,-211,-208,-107,-202]]},{"type":"Polygon","properties":{"name":"Rostov"},"id":"RU.RO","arcs":[[220,-55,221,222,223,224]]},{"type":"Polygon","properties":{"name":"Tula"},"id":"RU.TL","arcs":[[225,-212,-220,-201,-216]]},{"type":"Polygon","properties":{"name":"Volgograd"},"id":"RU.VG","arcs":[[226,227,228,229,-225,230]]},{"type":"Polygon","properties":{"name":"Belgorod"},"id":"RU.BL","arcs":[[231,232,-206]]},{"type":"Polygon","properties":{"name":"Krasnodar"},"id":"RU.KD","arcs":[[-54,-41,233,-222],[-38]]},{"type":"Polygon","properties":{"name":"Mordovia"},"id":"RU.MR","arcs":[[234,235,236,237,-189],[238]]},{"type":"Polygon","properties":{"name":"Penza"},"id":"RU.PZ","arcs":[[239,240,241,242,-237]]},{"type":"MultiPolygon","properties":{"name":"Ryazan'"},"id":"RU.RZ","arcs":[[[-239]],[[-190,-238,-243,243,-213,-226,-215,244]]]},{"type":"Polygon","properties":{"name":"Tambov"},"id":"RU.TB","arcs":[[-242,245,246,-209,-244]]},{"type":"Polygon","properties":{"name":"Vladimir"},"id":"RU.VL","arcs":[[-191,-245,-214,-197,-178]]},{"type":"Polygon","properties":{"name":"Voronezh"},"id":"RU.VR","arcs":[[-247,247,-231,-224,248,-232,-205,-210]]},{"type":"Polygon","properties":{"name":"Bashkortostan"},"id":"RU.BK","arcs":[[249,250,251,252,253,254]]},{"type":"Polygon","properties":{"name":"Chelyabinsk"},"id":"RU.CL","arcs":[[255,256,257,-251,258]]},{"type":"Polygon","properties":{"name":"Kurgan"},"id":"RU.KU","arcs":[[259,260,-256,261]]},{"type":"MultiPolygon","properties":{"name":"Nenets"},"id":"RU.NN","arcs":[[[262]],[[263]],[[264]],[[265]],[[266]],[[267]],[[268]],[[269]],[[270]],[[271]],[[272]],[[273]],[[274]],[[275,276,-129,277]],[[278]]]},{"type":"MultiPolygon","properties":{"name":"Yamal-Nenets"},"id":"RU.YN","arcs":[[[279]],[[280]],[[281]],[[282]],[[283]],[[284]],[[285]],[[286,287,288,-276,289]],[[290]],[[291]],[[292]],[[293]],[[294]],[[295]]]},{"type":"Polygon","properties":{"name":"Komi"},"id":"RU.KO","arcs":[[-289,296,297,298,299,-130,-277]]},{"type":"Polygon","properties":{"name":"Kirov"},"id":"RU.KV","arcs":[[-300,300,301,302,303,-186,-182,-195,-131]]},{"type":"Polygon","properties":{"name":"Mariy-El"},"id":"RU.ME","arcs":[[304,305,-187,-304]]},{"type":"Polygon","properties":{"name":"Sverdlovsk"},"id":"RU.SV","arcs":[[306,-262,-259,-250,307,-298,308]]},{"type":"Polygon","properties":{"name":"Udmurt"},"id":"RU.UD","arcs":[[309,-254,310,-302]]},{"type":"MultiPolygon","properties":{"name":"Astrakhan'"},"id":"RU.AS","arcs":[[[311]],[[312,313,-229]]]},{"type":"Polygon","properties":{"name":"Chuvash"},"id":"RU.CV","arcs":[[314,315,-235,-188,-306]]},{"type":"Polygon","properties":{"name":"Kalmyk"},"id":"RU.KL","arcs":[[-314,316,317,-56,-221,-230]]},{"type":"Polygon","properties":{"name":"Orenburg"},"id":"RU.OB","arcs":[[-252,-258,318,319,320],[321]]},{"type":"MultiPolygon","properties":{"name":"Samara"},"id":"RU.SA","arcs":[[[-322]],[[-320,322,323,324]]]},{"type":"Polygon","properties":{"name":"Saratov"},"id":"RU.SR","arcs":[[-323,325,-227,-248,-246,-241,326]]},{"type":"Polygon","properties":{"name":"Tatarstan"},"id":"RU.TT","arcs":[[-311,-253,-321,-325,327,-315,-305,-303],[328]]},{"type":"MultiPolygon","properties":{"name":"Ul'yanovsk"},"id":"RU.UL","arcs":[[[-324,-327,-240,-236,-316,-328]],[[-329]]]},{"type":"Polygon","properties":{"name":"Khanty-Mansiy"},"id":"RU.KM","arcs":[[329,-37,330,-309,-297,-288]]},{"type":"MultiPolygon","properties":{"name":"Omsk"},"id":"RU.OM","arcs":[[[331]],[[332]],[[333,334,335,-35]]]},{"type":"Polygon","properties":{"name":"Tyumen'"},"id":"RU.TY","arcs":[[-36,-336,336,-260,-307,-331],[-333],[-332]]},{"type":"Polygon","properties":{"name":"Altay"},"id":"RU.AL","arcs":[[337,338,339,340]]},{"type":"Polygon","properties":{"name":"Gorno-Altay"},"id":"RU.GA","arcs":[[341,342,343,344,-338]]},{"type":"Polygon","properties":{"name":"Kemerovo"},"id":"RU.KE","arcs":[[345,-342,-341,346,-33,347]]},{"type":"Polygon","properties":{"name":"Khakass"},"id":"RU.KK","arcs":[[348,-343,-346,349]]},{"type":"Polygon","properties":{"name":"Novosibirsk"},"id":"RU.NS","arcs":[[-347,-340,350,-334,-34]]},{"type":"Polygon","properties":{"name":"Chechnya"},"id":"RU.CN","arcs":[[351,-43,-49,-53,352]]},{"type":"MultiPolygon","properties":{"name":"Dagestan"},"id":"RU.DA","arcs":[[[353]],[[354,-353,-52,-318]]]},{"type":"Polygon","properties":{"name":"Irkutsk"},"id":"RU.IR","arcs":[[355,356,357,358,359]]},{"type":"MultiPolygon","properties":{"name":"Krasnoyarsk"},"id":"RU.KY","arcs":[[[360]],[[361]],[[362]],[[363]],[[364,365]],[[366]],[[367]],[[368]],[[369]],[[370]],[[371]],[[372]],[[373]],[[374]],[[375]],[[376]],[[377]],[[378]],[[379]],[[380]],[[381]],[[382]],[[383]],[[384]],[[385]],[[386]],[[387]],[[388]],[[389]],[[390]],[[391]],[[392]],[[393]],[[394]],[[395]],[[396]],[[397]],[[398]],[[399]],[[400]],[[401]],[[402]],[[403,-359,404,-350,-348,-32,-330,-287,405]],[[406]],[[407]],[[408]],[[409]],[[410]],[[411]],[[412]],[[413]],[[414]],[[415]],[[416]],[[417]],[[418]],[[419]],[[420]],[[421]]]},{"type":"Polygon","properties":{"name":"Tuva"},"id":"RU.TU","arcs":[[422,423,-344,-349,-405,-358]]},{"type":"Polygon","properties":{"name":"Buryat"},"id":"RU.BU","arcs":[[424,425,-423,-357]]},{"type":"Polygon","properties":{"name":"Amur"},"id":"RU.AM","arcs":[[426,427,428,429,430]]},{"type":"Polygon","properties":{"name":"Chita"},"id":"RU.CT","arcs":[[431,-430,432,-425,-356]]},{"type":"MultiPolygon","properties":{"name":"Primor'ye"},"id":"RU.PR","arcs":[[[433]],[[434]],[[435,436]]]},{"type":"MultiPolygon","properties":{"name":"Sakha (Yakutia)"},"id":"RU.CK","arcs":[[[437]],[[438]],[[439]],[[440]],[[441]],[[442]],[[443]],[[444]],[[445]],[[446]],[[447]],[[448]],[[449]],[[-68,450,451,-431,-432,-360,-404,452,-365,453]],[[454]],[[455]],[[456]],[[457]],[[458]],[[459]],[[460]],[[461]],[[462]],[[463]],[[464]],[[465]]]},{"type":"Polygon","properties":{"name":"Yevrey"},"id":"RU.YV","arcs":[[466,-428,467]]},{"type":"MultiPolygon","properties":{"name":"Khabarovsk"},"id":"RU.KH","arcs":[[[468]],[[469]],[[470]],[[471]],[[472]],[[473]],[[474,475,-437,476,-468,-427,-452]]]},{"type":"MultiPolygon","properties":{"name":"Maga Buryatdan"},"id":"RU.MG","arcs":[[[477]],[[478]],[[479]],[[-67,480,481,-475,-451]]]},{"type":"MultiPolygon","properties":{"name":"Sakhalin"},"id":"RU.SL","arcs":[[[482]],[[483]],[[484]],[[485]],[[486]],[[487]],[[488]],[[489]],[[490]],[[491]],[[492]],[[493]],[[494]],[[495]],[[496]],[[497]],[[498]],[[499]],[[500]],[[501]],[[502]],[[503]],[[504]],[[505]]]},{"type":"Polygon","properties":{"name":"Perm'"},"id":"RU.PE","arcs":[[-255,-310,-301,-299,-308]]},{"type":"MultiPolygon","properties":{"name":"Kamchatka"},"id":"RU.KA","arcs":[[[506]],[[507]],[[508]],[[509,-481,-66]]]}]}},"arcs":[[[8636,312],[0,-1],[-1,1],[1,0],[0,3],[0,-1],[0,-1],[0,-1]],[[8651,355],[-1,1],[0,1],[1,2],[0,4],[0,1],[0,-1],[0,-3],[0,-2],[0,-2],[0,-1]],[[8676,385],[0,-4],[0,-3],[-1,-1],[0,6],[0,-1],[0,1],[-1,2],[1,4],[0,1],[1,-2],[0,-3]],[[8657,418],[0,-5],[-1,1],[0,-1],[0,-2],[0,-1],[0,1],[0,1],[0,1],[0,3],[1,2]],[[8658,426],[1,-1],[0,-1],[0,-3],[-1,-1],[0,-1],[0,1],[0,2],[0,2],[0,2]],[[8658,432],[0,-4],[0,2],[0,1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,2],[0,1],[0,1],[0,2],[0,-2],[1,-1],[0,-3],[0,-2],[0,-1],[0,-1],[-1,-1]],[[6320,797],[0,-2],[-1,0],[-1,6],[0,2],[0,3],[0,3],[1,2],[0,-1],[1,-7],[0,-6]],[[6334,983],[-1,-3],[-1,2],[0,7],[0,2],[1,2],[1,-4],[0,-6]],[[6349,1099],[0,-1],[-1,5],[0,9],[0,6],[0,6],[0,2],[1,-2],[0,-7],[0,-6],[0,-5],[0,-7]],[[6021,5613],[-1,8],[1,1],[0,-3],[0,-6]],[[6016,5667],[1,0],[0,-1],[0,-2],[1,-2],[0,-1],[0,1],[1,0],[0,-1],[0,-1],[-1,-1],[-1,-4],[-1,1],[0,1],[0,3],[0,2],[-1,0],[0,1],[0,1],[1,3]],[[5991,5757],[-1,2],[0,3],[0,1],[0,-3],[1,-3]],[[5988,5772],[0,-1],[0,-1],[0,-1],[-1,3],[-1,3],[0,1],[0,2],[1,0],[0,-1],[0,-1],[0,-1],[1,-1],[0,-2]],[[5972,5780],[-1,0],[0,1],[0,1],[1,0],[0,-1],[0,-1]],[[5976,5835],[0,-1],[0,1],[-1,0],[-1,2],[1,3],[0,1],[0,-1],[1,-2],[0,-1],[0,-2]],[[5972,5843],[-1,0],[0,2],[1,1],[0,-2],[0,-1]],[[5990,5843],[0,-1],[-1,2],[0,2],[1,0],[0,-1],[0,-1],[0,-1]],[[5991,5848],[0,-1],[0,-1],[-1,-1],[0,-1],[0,1],[0,1],[0,1],[0,1],[0,-1],[1,1]],[[5967,5852],[0,-1],[0,1],[0,-3],[-1,-1],[0,4],[1,0]],[[5972,6023],[0,-1],[-1,0],[0,2],[1,1],[0,-1],[0,-1]],[[5967,6022],[-1,0],[0,2],[1,1],[0,1],[0,-1],[0,-1],[0,-2]],[[5970,6046],[1,-1],[0,1],[0,-1],[0,-3],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,1],[-1,1],[0,1],[0,2],[0,2],[-1,0],[0,1],[0,1],[0,1],[0,1],[0,1],[1,-1],[0,1],[0,1]],[[5938,6174],[-1,-1],[-2,3],[0,1],[0,1],[0,1],[3,1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2]],[[5938,6188],[1,-1],[0,1],[1,0],[0,-1],[0,-1],[-1,-1],[0,-1],[1,0],[0,-2],[-1,-1],[0,1],[-1,1],[1,1],[0,1],[-1,2],[0,1]],[[6130,6220],[-1,-1],[0,1],[1,0]],[[5904,6352],[1,-3],[-1,-1],[0,-1],[-1,2],[0,2],[0,1],[0,1],[1,0],[0,-1]],[[6125,6550],[0,-2],[0,-1],[1,-1],[0,-2],[-1,-1],[-1,1],[0,2],[0,2],[0,1],[1,-1],[0,2]],[[6068,6679],[-1,2],[0,2],[0,1],[0,1],[0,2],[1,0],[0,-4],[0,-1],[1,-1],[0,-1],[-1,-1]],[[6870,6790],[1,-4],[1,-4],[0,-2],[0,-1],[-1,-1],[-1,2],[0,-1],[0,-1],[-4,10],[0,1],[-1,-4],[-1,0],[0,2],[0,1],[0,1],[3,4],[3,-3]],[[6037,6790],[0,-1],[-1,1],[0,2],[0,2],[1,-1],[0,-3]],[[6512,9856],[2,-3],[-2,0],[-5,12],[0,1],[0,1],[-1,1],[0,1],[0,1],[-1,1],[0,1],[-1,1],[1,-1],[1,-2],[5,-7],[0,-2],[0,-1],[0,-1],[0,-2],[1,-1]],[[7340,4835],[0,-5],[2,-11],[10,-82],[1,-2],[0,-6],[0,-2],[1,-9],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[-1,-5],[0,-2],[0,-3],[-1,-14],[0,-3],[-1,-2],[-1,-5],[0,-1],[0,-3],[0,-5],[0,-28],[-1,-3],[0,-3],[-1,-8],[-1,-3],[0,-1],[0,-2],[0,-1],[1,-2],[2,-5],[1,-7],[1,-3],[22,-3],[13,16],[19,-2],[12,-16],[1,-2],[0,-2],[2,-38],[0,-3],[0,-3],[1,-1],[8,-3],[0,-1],[1,-7],[9,-86],[1,-4],[0,-2],[1,0],[18,8],[1,0],[0,-1],[0,-5],[-1,-10],[1,-5],[0,-4],[6,-40],[0,-1],[0,-2],[-1,-1],[0,-2],[-11,-23],[0,-1],[-1,-1],[-12,-94],[0,-2],[0,-1],[-1,-1],[0,-2],[3,-57],[0,-7],[0,-1],[1,-2],[0,-3],[1,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-7],[1,-6],[0,-1],[0,-1],[1,-1],[0,-1],[15,-17],[1,-2],[3,-13],[0,-2],[0,-1],[0,-1],[1,-1],[13,-2],[0,-1],[1,-7],[0,-10],[-1,-19],[1,-32],[0,-4],[0,-4],[-1,-2],[-2,-7],[-2,1],[-1,3],[-1,-2],[0,-8],[0,-6],[-1,-9],[-1,-1],[0,-2],[-1,0],[0,-1],[-1,-1],[0,-7],[-1,-2],[0,-1],[-2,-1],[0,-1],[0,-3],[-2,-27],[0,-6],[-1,-6],[-1,-7],[0,-3],[0,-1],[0,-1],[0,-1],[-1,-1],[-1,0],[0,-1],[-1,-2],[0,-2],[-3,-17],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[3,-3],[1,0],[1,2],[1,0],[0,-1],[0,-3],[0,-5],[-3,-41],[0,-13]],[[7461,3846],[-2,-12],[0,-1],[-1,2],[-3,10],[0,-1],[-2,-9],[-5,-17],[-3,-12],[-2,-9],[-5,-21],[0,-1],[0,1],[-5,22],[-2,3],[0,1],[-1,-1],[-3,-9],[-1,1],[0,2],[0,2],[-1,2],[-2,5],[-2,3],[-1,0],[0,-2],[0,-2],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[1,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[-1,-6],[0,-2],[-3,-1],[-1,2],[-1,5],[0,1],[-1,-2],[0,-1],[0,-1],[-2,2],[0,1],[0,2],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[-1,3],[0,1],[0,-1],[-1,-1],[0,-1],[-1,1],[-1,5],[-1,1],[-1,-2],[0,-1],[0,-1],[0,-2],[-1,-1],[0,-2],[-1,-2],[-1,-4],[0,-1],[-2,-2],[-2,-3],[0,1],[0,5],[0,3],[-1,5],[-1,3],[0,2],[-1,-1],[-2,0],[0,-2],[0,-2],[0,-4],[-1,-7],[0,-1],[0,-2],[-1,0],[0,-1],[0,-2],[0,-3],[-1,-5],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-3,0],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[-2,-5],[-1,3],[-1,0],[0,-1],[-2,-8],[0,-1],[0,-2],[0,-1],[0,-2],[1,-1],[0,-1],[0,-2],[0,-1],[-1,0],[-5,-9],[0,-2],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[1,-1],[2,-3],[0,-1],[0,-2],[0,-3],[-8,-7],[-2,2],[0,2],[-1,3],[0,1],[-1,0],[-3,-5],[0,-1],[0,-1],[-1,-3],[0,-2],[0,-2],[0,-7],[0,-3],[-2,-3],[-1,-1],[0,3],[0,2],[0,1],[-1,3],[0,2],[-1,0],[-1,-1],[-12,-28],[-1,-3]],[[7344,3652],[-1,-13],[-1,-2],[0,1],[0,1],[0,1],[-1,1],[0,4],[0,1],[0,2],[-1,4],[-1,2],[-1,1],[0,-1],[-3,-5],[-2,-3],[-1,-2],[-1,-1],[0,-1],[-1,-9],[0,-1],[0,-2],[0,-1],[0,-3],[0,-1],[0,-3],[0,-1],[0,-2],[-1,-2],[0,-1],[-1,-3],[-1,-1],[-1,-1],[0,-1],[0,-2],[0,-3],[0,-1],[0,-2],[-1,-4],[0,-1],[0,-6],[-1,-1],[0,-2],[0,-4],[-1,-1],[0,-2],[0,-1],[0,-1],[1,-1],[0,-2],[0,-1],[0,-1],[-1,-2],[0,-2],[0,-3],[-1,-1],[-1,-1],[-2,4],[-1,0],[-1,-4],[0,-2],[0,-1],[-1,1],[0,1],[0,1],[1,6],[0,1],[0,2],[0,1],[-1,1],[-2,-5],[0,-1],[0,-2],[0,-3],[-1,-1],[0,-1],[0,1],[0,1],[0,1],[0,2],[0,2],[-1,3],[-1,1],[0,2],[0,3],[0,1],[0,1],[0,1],[1,4],[0,1],[1,1],[0,1],[0,2],[0,11],[0,1],[0,1],[2,-2],[0,1],[0,2],[0,3],[0,2],[0,1],[1,2],[0,1],[0,1],[0,1],[0,2],[0,1],[-1,8],[-1,2],[0,1],[0,2],[1,0],[0,1],[-1,1],[0,3],[-2,11],[-1,3],[0,2],[0,2],[2,7],[0,1],[0,1],[0,1],[0,1],[-1,3],[0,1],[0,-1],[-1,-3],[-3,7],[0,1],[0,2],[0,1],[1,2],[0,2],[0,1],[0,2],[0,2],[0,1],[0,1],[0,2],[1,0],[0,-1],[1,0],[0,1],[0,1],[0,1],[0,1],[-1,4],[0,1],[0,1],[-1,0],[0,-1],[-1,-4],[0,-1],[-1,1],[0,1],[0,2],[0,1],[0,1],[2,10],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[1,6],[0,2],[0,1],[0,1],[0,1],[0,1],[0,1],[-1,1],[0,1],[0,3],[0,1],[0,1],[2,5],[0,3],[1,1],[0,1],[0,2],[0,2],[0,2],[0,2],[-5,23],[0,1],[-1,-7],[-1,-3],[0,-1],[-1,0],[-1,4],[0,1],[-1,1],[-1,0],[-1,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-1],[1,-6],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[-1,-3],[0,-3],[-1,-1],[-1,4],[-1,0],[-1,0],[-2,-6],[-1,-1],[-1,1],[-1,-1],[-1,0],[0,-1],[0,-1],[-1,-5],[-1,-2],[0,-1],[0,-1],[-2,0],[-4,-5],[-2,3],[-1,0],[-2,-2],[-11,-20],[-1,1],[-3,28],[-6,33],[-1,4],[-1,2],[-18,-19],[-5,-4],[-2,4],[-18,116],[-32,48],[0,9],[-42,7],[-2,2],[-1,3],[-20,10]],[[7113,3948],[0,6],[0,3],[-1,2],[0,3],[-2,19],[-1,5],[-3,-1],[-3,49],[-2,11],[-1,1],[-1,1],[0,2],[0,4],[-1,56],[0,4],[0,3],[0,2],[-13,42],[0,2],[0,2],[0,3],[3,14],[1,3],[0,3],[-1,7],[-3,20],[0,3],[0,3],[6,24],[0,8],[-5,23]],[[7086,4275],[1,10],[0,2],[0,5]],[[7087,4292],[0,7],[1,6],[5,21],[7,48],[1,8],[1,6],[0,6],[-2,37],[0,3],[0,3],[1,2],[3,6],[1,2],[0,4],[1,3],[0,4],[1,17],[0,8],[1,2],[1,-3],[0,-1],[1,0],[0,1],[0,1],[1,1],[4,26],[0,2],[8,0],[5,8],[0,2],[0,6],[0,14],[1,7],[2,9],[0,3],[0,4],[0,44],[0,29],[0,3],[0,1],[0,1],[0,1],[0,1],[0,1],[-1,2],[0,1],[0,2],[0,1],[0,1],[2,10],[0,4],[2,23],[0,1],[0,2],[0,8],[-1,2],[0,1],[0,1],[0,1],[0,1],[-1,1],[-1,1],[0,1],[0,2],[0,5],[1,18],[1,5],[1,3],[1,2],[4,9],[0,1],[0,2],[0,5],[1,2],[0,2],[0,6],[-1,4],[0,2],[0,1],[0,1],[-1,1],[0,1],[-1,2],[0,15],[0,1],[1,2],[0,3],[1,-1],[0,-1],[1,0],[0,3],[0,2],[0,2],[0,2],[1,17],[0,3],[0,1],[0,1],[0,1],[1,0],[3,-6],[2,0],[1,-1],[1,-4],[5,4],[4,-1],[1,-2],[2,-6],[1,-3],[1,-6],[0,-1],[0,-1],[1,0],[0,1],[4,11],[1,2],[2,0],[8,-6],[1,0],[5,12],[3,-11],[0,-1],[3,1],[0,1],[2,8],[0,1],[0,1],[1,-1],[0,-1],[1,1],[1,0],[1,0],[1,3],[1,0],[0,-1],[1,-5],[1,0],[0,-1],[1,0],[0,-1],[0,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-1],[0,-11],[0,-1],[0,-2],[0,-2],[2,-10],[0,-1],[1,1],[3,5],[9,1],[9,-6],[1,0],[0,1],[6,22],[0,1],[1,1],[5,0],[0,1],[2,6],[1,0],[8,-9],[0,-1],[0,-1],[0,-3],[1,-11],[0,-3],[0,-4],[1,-3],[0,-3],[1,-1],[10,-5],[1,4],[8,3],[2,-2],[0,-2],[7,-24],[1,-1],[0,1],[5,17],[1,3],[0,3],[0,4],[0,11],[0,3],[0,3],[0,3],[21,75],[10,5],[0,-1],[0,-1],[3,-14],[8,-27],[1,-8],[1,-3],[1,-1],[7,8]],[[6104,967],[0,1],[1,-2],[0,-2],[1,0],[1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,-1],[2,0],[0,1],[1,1],[2,2],[0,2],[1,1],[1,1],[2,-4],[1,-1],[0,-2],[1,-6],[1,-7],[1,-3],[1,-4],[3,-21],[0,-1],[0,-2],[1,-3],[0,-5],[1,-16],[0,-1],[0,-1],[0,-1],[1,-2],[0,-4],[0,-15],[0,-4],[1,-9],[0,-3],[1,-3],[0,-2],[1,-2],[0,-1],[0,-2],[0,-3],[0,-2],[-1,-3],[1,-6],[0,-17],[0,-2],[0,-1],[-3,-1],[0,1],[0,1],[0,1],[0,1],[1,1],[0,1],[0,2],[0,8],[0,1],[0,2],[0,2],[-1,10],[0,1],[0,1],[0,2],[0,3],[0,1],[0,2],[-1,2],[0,1],[0,1],[0,2],[-1,2],[0,1],[0,2],[0,2],[0,2],[0,2],[0,6],[0,2],[-1,2],[-1,3],[-1,1],[-1,-1],[0,-1],[0,-1],[0,-2],[-1,-1],[0,-10],[0,-8],[0,-3],[-1,-5],[0,-1],[-1,-2],[0,-6],[0,-2],[1,-3],[0,-2],[0,-1],[0,-1],[0,-1],[1,-2],[0,-2],[1,-6],[0,-1],[0,-38],[0,-2],[0,-6],[1,-1],[0,-2],[0,-2],[0,-2],[-1,-2],[0,-2],[0,-1],[0,-3],[-1,-2],[0,-5],[0,-1],[0,-2],[0,-3],[0,-41],[0,-3],[1,-2],[0,-1],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-3],[0,-3],[0,-3],[0,-2],[-1,-1],[0,-2],[0,-2],[0,-3],[-1,-12],[0,-4],[-1,-9],[0,-2],[0,-1],[0,-1],[-1,-3],[0,-1],[-1,-1],[0,1],[-1,2],[-2,9],[-1,6],[0,2],[-2,2],[-1,1],[0,1],[-2,7],[0,5],[-1,1],[0,1],[-1,0],[0,-1],[-1,-2],[0,-1],[-1,0],[0,2],[-1,3],[-1,1],[0,1],[0,4],[0,1],[0,2],[0,2],[0,1],[0,1],[0,1],[1,1],[0,1],[0,1],[-1,2],[0,1],[0,1],[0,1],[0,1],[0,2],[0,2],[0,2],[1,5],[0,1],[0,2],[0,4],[0,1],[0,2],[-1,2],[0,5],[0,1],[0,1],[0,3],[0,1],[0,2],[0,3],[0,4],[0,2],[0,1],[1,1],[1,0],[0,-1],[0,-1],[0,-1],[1,-1],[0,-1],[0,-2],[0,-2],[0,-4],[0,-2],[0,-1],[0,-1],[1,-1],[0,-1],[0,-1],[1,-1],[0,-2],[0,-6],[0,-1],[0,-1],[0,-1],[1,0],[1,1],[0,1],[0,1],[0,3],[0,1],[1,2],[0,3],[0,2],[0,1],[0,1],[0,2],[0,1],[1,1],[0,1],[0,2],[0,7],[0,1],[0,2],[0,4],[1,4],[0,1],[0,2],[0,1],[-1,1],[0,1],[-2,0],[-1,1],[0,1],[0,1],[0,1],[0,2],[-1,11],[0,2],[0,1],[-1,1],[-1,-3],[-1,0],[0,1],[0,1],[0,2],[0,1],[0,2],[0,3],[0,3],[1,8],[0,5],[1,8],[0,1],[0,2],[0,3],[0,15],[0,3],[0,3],[1,1],[0,1],[0,2],[0,6],[0,1],[0,2],[0,3],[0,2],[0,6],[0,2],[0,3],[0,1],[0,1],[1,-1],[0,1],[1,4],[1,0],[0,1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,6],[-1,8],[0,3],[0,1],[0,1],[0,1],[0,1],[0,1],[-1,0],[0,9],[0,2],[0,1],[0,1],[0,1],[0,6],[0,1],[-1,0],[0,1],[0,3],[0,3],[0,3],[0,1],[0,2],[0,1],[0,1],[0,2],[-1,4],[-2,-5],[0,-2],[-1,-11],[-1,0],[0,-1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[-1,1],[0,1],[0,2],[0,1],[0,1],[-1,0],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,0],[0,-1],[0,-1],[1,-1],[0,-1],[-1,-1],[0,1],[0,2],[-1,1],[0,1],[0,1],[1,1],[0,6],[1,0],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[-1,4],[0,-1],[-1,-2],[-1,0],[0,1],[0,1],[0,2],[-1,-1],[0,-1],[0,1],[0,1],[0,-1],[-1,-1],[0,-8],[0,-3],[0,-5],[1,-3],[0,-1],[0,-1],[0,1],[1,-1],[1,-3],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-2,-11],[0,-1],[-1,-2],[-1,-1],[0,1],[-1,0],[-2,1],[-1,2],[-7,-1],[0,1],[-1,2],[-2,5],[-1,2],[-1,6],[-2,8],[-1,4],[0,3],[-1,1],[-1,-1],[-1,2],[0,1],[0,2],[0,2],[0,5],[0,13],[1,-3],[0,-2],[0,-1],[1,1],[0,-1],[1,-2],[1,0],[0,1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,2],[1,1],[0,1],[1,1],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[1,-1],[0,-1],[0,-3],[0,-1],[1,0],[0,-1],[1,-2],[0,-1],[1,-1],[0,1],[1,3],[0,1],[1,0],[0,-2],[1,0],[1,3],[2,3],[1,0],[0,2],[0,2],[0,3],[0,2],[0,1],[1,1],[0,2],[2,7],[1,4],[0,7],[2,7],[0,6],[0,1],[1,0],[1,3],[1,0],[0,-1],[0,-1],[0,-2],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,-3],[0,-1],[0,-1],[0,-1],[1,-1]],[[6185,643],[0,-5],[0,-23],[0,-1],[-1,-2],[0,-3],[0,-2],[-2,-5],[-1,-1],[-2,-3],[0,-2],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-12],[-1,-15],[-1,-10],[0,-5],[1,-12],[0,-4],[-1,-1],[0,-6],[0,-3],[-1,-8],[0,-1]],[[6176,502],[-5,-2],[-1,-2],[-2,-8],[-1,-1],[-1,0],[-1,2],[0,3],[-1,1],[-1,0],[-2,-1],[-1,0],[-2,4],[-1,1],[-2,-2],[-1,0],[0,3],[-2,8],[0,3],[-2,3],[0,2],[-1,3],[0,5],[0,3],[-1,2],[-2,-3],[-1,3],[-1,5],[-1,3],[-1,1],[-3,-3],[0,2],[-4,19],[-1,4],[-3,6],[-1,7]],[[6130,573],[0,5],[0,6],[0,6],[0,1],[0,3],[1,2],[0,2],[0,2],[-1,1],[0,1],[0,2],[0,2],[0,2],[0,1],[0,2],[0,2],[0,2],[0,2],[0,2],[0,1],[0,1],[1,1],[0,1],[0,2],[0,2],[0,2],[0,7],[-1,3],[0,2],[1,1],[0,2],[1,16],[0,2],[0,1],[0,5],[0,2],[0,1],[0,1],[1,0],[1,1],[0,1],[0,4],[0,1],[0,2],[1,0],[0,1],[0,-1],[1,-1],[0,1],[1,4],[0,1],[0,1],[0,1],[0,2],[-1,3],[0,3],[0,8],[0,1],[-2,10],[0,1],[0,1],[0,1],[0,1],[0,1],[1,2],[0,3],[1,0],[0,-1],[2,-5],[0,-1],[1,-1],[1,0],[0,-1],[1,-3],[0,1],[1,1],[1,-2],[0,-2],[0,-1],[0,1],[1,0],[1,-1],[1,-1],[0,-1],[0,-2],[0,-1],[1,-1],[0,-2],[0,-1],[0,-2],[-1,-1],[0,-2],[0,-1],[0,-2],[0,-2],[1,3],[0,2],[1,0],[0,-2],[-1,-4],[0,-1],[0,-1],[0,-2],[1,0],[0,2],[0,3],[1,0],[0,1],[0,-2],[1,-2],[1,0],[0,1],[0,1],[0,2],[0,2],[0,1],[1,2],[0,2],[0,2],[1,1],[0,3],[0,3],[0,2],[0,1],[0,2],[1,1],[1,3],[0,1],[1,0],[0,1],[0,1],[0,7],[1,2],[0,3],[1,10],[0,6],[0,2],[0,4],[0,4],[0,8],[0,2],[-1,3],[0,1],[0,1],[-1,5],[-1,3],[0,1],[0,1],[0,1],[0,1],[1,4],[0,1],[0,1],[1,2],[0,2],[1,2],[0,1],[0,1],[0,1],[0,2],[1,1]],[[6159,803],[1,0],[0,1],[0,1],[1,2],[1,3],[0,1],[1,1],[1,0],[0,-1],[0,-3],[0,-1],[0,-3],[0,-8],[0,-3],[0,-2],[1,-2],[0,-2],[0,-2],[2,-2],[0,-3],[0,-1],[1,0],[1,0],[1,-1],[0,-1],[2,-2],[2,-4],[1,-4],[1,-1],[0,-1],[0,-1],[0,-4],[0,-2],[0,-1],[1,0],[1,2],[1,2],[0,2],[1,1],[2,-3],[0,-7],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[-1,0],[0,-1],[0,-1],[0,-1],[0,-2],[-2,-3],[0,-1],[0,-1],[0,-1],[0,-2],[-1,-1],[0,-3],[-1,-6],[-1,-2],[0,1],[-1,-2],[0,-4],[0,-2],[0,-2],[0,-1],[1,-4],[0,-5],[0,-1],[0,-1],[0,-2],[0,-1],[1,-2],[0,-2],[0,-2],[0,-2],[0,-3],[0,-3],[0,-1],[1,-1],[0,-1],[1,-1],[2,2],[0,1],[-1,6],[0,1],[0,1],[0,2],[0,1],[0,1],[1,1],[0,1],[0,1],[1,1],[1,1],[0,-1],[1,-1],[0,-3],[0,-1],[-1,-2],[0,-3],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[1,0],[1,0],[0,-1],[0,-2],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[1,-2],[0,-1],[0,-2],[1,-2],[0,-4],[-1,-8],[0,-1],[1,-5],[0,-5]],[[6245,588],[1,-4],[0,-1],[1,-2],[1,-2],[3,-13],[0,-7],[0,-4],[0,-3],[0,-12],[1,-6],[0,-3],[0,-7],[1,-21],[0,-5],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[1,-2],[0,-6],[0,-4],[0,-2],[0,-4],[0,-3],[-1,-6],[0,-3],[0,-3],[0,-8],[0,-2],[0,-2],[0,-3],[0,-2],[0,-1],[0,-1],[1,-1],[0,-2],[0,-2],[0,-2],[0,-14],[0,-2],[-1,-7],[0,-4],[1,-3],[0,-3],[0,-2],[0,-1],[0,-2],[-3,-19],[0,-1],[0,-1],[0,-3]],[[6251,369],[0,-1],[0,-1],[-1,1],[0,1],[-3,15],[-2,1],[0,-6],[0,-4],[-1,-11],[-1,-14],[0,-2],[0,3],[-1,14],[0,3],[-1,6],[-1,2],[-1,4]],[[6239,380],[0,1],[0,2],[0,7],[0,3],[0,4],[0,1],[0,18],[2,-2],[0,-1],[2,-8],[1,0],[1,-1],[1,12],[1,3],[0,3],[1,1],[0,2],[0,4],[0,2],[0,2],[0,1],[0,2],[-1,4],[0,1],[0,2],[0,2],[0,6],[0,3],[0,2],[0,1],[0,1],[0,1],[0,1],[0,2],[-2,10],[0,4],[0,3],[-1,5],[0,3],[0,1],[-1,0],[-2,10],[0,1],[-1,7],[0,-2],[-1,0],[0,-1],[0,-1],[1,-1],[0,-3],[0,-2],[-2,4],[0,2],[0,1],[0,25],[0,4],[0,9],[-1,0],[-1,-2],[0,1],[-1,1],[0,1],[0,1],[0,4],[1,36],[0,3],[0,2],[1,2],[0,1],[0,1],[2,2],[0,-2],[0,-1],[0,-9],[0,-1],[0,-1],[0,-1],[2,2],[0,1],[0,2],[0,1],[0,6],[0,2],[0,1],[0,1],[0,1],[2,1],[0,-1],[0,-1],[0,-3],[1,-2],[0,-3],[0,-1],[1,2]],[[6230,646],[-1,-17],[0,-6],[1,-2],[0,-11],[1,-2],[1,2],[0,1],[1,-1],[1,-4],[0,-1],[0,-2],[0,-2],[-1,-15],[0,-9],[0,-6],[-1,-4],[0,-1],[0,-1],[0,-4],[1,-9],[0,-2],[0,-1],[0,-1],[-1,-3],[0,-1],[-1,0],[0,1],[0,1],[0,3],[0,1],[-1,0],[-2,-10],[-1,-3],[0,-5],[-1,0],[0,2],[0,3],[0,10],[0,3],[-1,0],[-1,-8],[-1,-4],[0,-2],[0,-1],[0,-1],[0,-3],[0,-3],[1,-14],[0,-2],[-2,-16],[0,1],[0,1],[0,1],[0,5],[0,1],[0,1],[0,1],[-1,4],[-1,-1],[-1,-3],[-1,2],[0,1],[0,1],[0,3],[0,1],[0,1],[0,1],[-1,4],[0,-3],[-1,-6],[-1,-25],[0,-5],[0,-5],[-1,-6],[-2,-14],[-1,-15],[-1,-5],[0,-1],[0,-7],[-1,0],[0,-1],[0,1],[-2,5],[-1,-2],[-1,-12],[0,-3]],[[6205,417],[-6,9],[0,2],[-1,2],[-3,19],[-1,6],[0,2],[0,4],[0,4],[0,4],[-3,5],[0,4],[-1,5],[-1,3],[-1,2],[-1,0],[-1,-3],[-1,0],[-1,-1],[0,-5],[-1,-1],[0,1],[-2,6],[-1,4],[-1,5],[-1,7],[-2,1]],[[6185,643],[0,2],[1,2],[3,-3],[0,1],[1,0],[1,11],[0,1],[1,5],[1,1],[3,1],[1,0],[0,1],[0,1],[0,2],[1,2],[0,2],[0,2],[0,1],[0,1],[1,1],[0,1],[1,0],[1,-6],[0,-4],[0,-4],[1,-5],[0,-6],[0,-6],[0,-3],[1,-1],[0,-1],[0,1],[1,4],[1,7],[1,3],[1,1],[1,-1],[3,-5],[2,-1],[1,0],[1,1],[0,1],[0,1],[0,1],[0,7],[1,6],[0,2],[0,3],[0,2],[0,1],[2,10],[1,2],[0,1],[2,-2],[0,-2],[0,-1],[1,-1],[0,-1],[0,-1],[3,0],[0,-2],[1,-2],[0,-2],[0,-2],[0,-1],[3,0],[0,1],[0,1],[0,2],[1,1],[0,1],[0,1],[1,3],[1,2],[0,3],[1,3],[0,-1],[0,-9],[0,-2],[0,-1],[0,-1],[-1,0],[0,-1],[-1,-1],[0,-3],[0,-1],[0,-1],[-1,-4],[0,-1],[0,-1],[0,-1],[0,-3],[0,-13]],[[6245,607],[0,-19]],[[6239,380],[-1,3],[-1,1],[-1,-1],[-1,-1],[-2,-8],[-1,-2],[-3,-1],[0,-5],[-1,-7],[-1,-3],[0,-3],[-1,-2],[-3,-1],[-1,-4],[0,-3],[-1,-6],[-1,-2],[0,-1],[-1,2],[0,2],[-1,3],[-1,2],[-1,1],[-1,2],[-1,6],[0,7],[1,5],[1,5],[0,6],[0,7],[0,2],[-1,2],[-3,4],[0,2],[-1,9],[-1,3],[-2,6],[-1,2],[-2,5]],[[6230,646],[0,-1],[1,-1],[1,-1],[0,-2],[0,-1],[1,0],[7,6],[0,2],[0,1],[1,0],[0,-1],[0,-1],[1,-1],[2,-4],[0,-1],[0,-3],[1,-12],[0,-4],[0,-4],[0,-1],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2]],[[6269,930],[-2,-14],[0,-2],[0,-2],[0,-1],[0,-16],[0,-4],[0,-4],[-1,-8],[-5,-37],[0,-1],[-1,2],[0,-1],[-1,-4],[0,-1],[-2,-1],[0,-1],[0,-1],[0,-2],[0,-3],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[-1,-2],[0,-2],[0,-2],[0,-6],[0,-3],[0,-2],[0,-1],[0,-1],[4,-3],[1,-1],[0,-1],[0,-3],[1,-4],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[-1,-15],[-1,-1],[0,-1],[-3,-4],[-2,-5],[-2,-1],[0,-8],[-1,-5],[0,-1],[0,-1],[1,-1],[4,-1],[0,-1],[0,-2],[1,-1],[0,1],[0,1],[1,1],[0,1],[0,1],[4,-3],[1,0],[0,-1],[0,-2],[0,-2],[0,-4],[0,-4],[0,-2],[-1,-6],[1,-22]],[[6264,694],[-2,-4],[0,-1],[0,-2],[0,-2],[0,-1],[0,-15],[0,-5],[0,-1],[0,-1],[0,-1],[-1,-1],[-5,1],[-1,1],[0,1],[0,1],[0,1],[0,3],[0,7],[0,1],[0,1],[0,1],[-3,1],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-4],[0,-3],[0,-2],[0,-1],[1,-1],[0,-1],[0,-22],[0,-1],[-1,-2],[0,-1],[-1,-1],[0,-2],[-1,-10],[0,-3],[0,-5],[-1,-4],[0,-1],[-2,-3],[-1,2]],[[6159,803],[0,7],[-1,1],[0,1],[0,1],[0,1],[-2,1],[0,1],[0,1],[0,1],[0,2],[-1,1],[-1,1],[0,1],[0,2],[0,10],[0,2],[-1,1],[-1,1],[0,2],[0,1],[-1,4],[-1,5],[0,3],[0,4],[0,1],[0,2],[0,2],[0,1],[0,1],[0,1],[1,1],[1,1],[1,1],[0,1],[0,1],[2,13],[0,7],[1,20],[0,2],[1,6],[0,5],[0,2],[0,1],[0,2],[-1,1],[0,1],[-5,2],[0,2],[-1,11],[0,3],[0,2],[-1,3],[-1,8],[0,4],[0,13],[0,9],[-1,3],[-1,5],[0,-1],[0,-1],[0,-1],[0,-1],[-7,-2],[0,4],[0,4],[0,8],[0,1],[0,1],[0,1],[0,2],[0,2],[0,1],[-1,2],[0,2],[0,2],[0,4],[1,4],[0,3],[-1,4],[0,2],[-2,5],[0,2],[0,9],[0,2],[0,4],[0,1],[0,1],[-2,3],[0,1],[0,1],[0,1],[0,2],[0,12],[0,23],[0,2],[1,1],[2,0],[1,0],[1,3],[0,1],[0,-1],[0,-1],[1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[2,1],[1,2],[0,1],[0,2],[0,3],[0,1],[0,2],[0,1],[0,2],[0,1],[0,2],[2,0],[1,1],[0,2],[0,8],[0,2],[0,1],[-1,1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,4],[0,2],[1,3],[-1,1],[-2,2],[0,1],[0,3],[0,3],[0,6],[0,4]],[[6143,1171],[6,-2],[0,1],[0,1],[0,2],[0,3],[0,1],[0,2],[1,1],[0,1],[4,-2],[1,-1],[1,-2],[1,0],[0,1],[1,4]],[[6158,1181],[9,-8],[1,-2],[1,-3],[0,-2],[0,-1],[0,-1],[0,1],[1,3],[0,1],[0,2],[0,1],[0,3],[0,11],[0,2],[0,1],[1,4],[0,1],[0,2],[0,4],[0,4],[0,2],[1,3],[3,4],[0,-1],[0,-2],[0,-2],[0,-11],[0,-5],[0,-15],[0,-1],[0,-1],[4,4],[1,0],[0,1],[1,1],[3,19],[0,2],[1,2],[1,1],[0,1],[1,0],[2,14],[1,3],[0,1],[0,1],[0,1],[0,1],[-1,1],[0,1],[0,2],[0,1],[0,1],[2,8],[1,0],[0,-2],[1,-10],[0,-2],[1,-1],[5,-13],[1,-3],[1,-3],[1,-3],[0,-3],[2,-11],[0,-2],[1,-2],[5,-9],[4,0],[2,-3],[2,-5],[0,-1],[1,-2],[0,-3],[0,-2],[0,-2],[1,-1],[1,-3],[0,-1],[0,-4],[2,-12],[1,-6],[0,-4],[0,-2],[1,-8],[0,-4],[0,-3],[0,-11],[1,-2],[0,-1],[1,-8],[0,-3],[2,-7],[4,-13],[4,-5],[4,0],[1,-3],[3,-10],[14,-52],[7,-21],[0,-1],[0,-2],[0,-7],[3,-25],[0,-4],[0,-5],[0,-1]],[[203,5767],[1,-1],[5,0],[0,-1],[1,-4],[-1,-3],[-1,-4],[-1,-1],[-1,1],[-2,2],[-1,0],[0,2],[0,1],[0,1],[-1,2],[0,4],[1,2],[0,-1]],[[214,5783],[-2,-3],[-4,1],[-1,-5],[-2,0],[0,12],[1,7],[2,7],[1,6],[0,3],[2,3],[2,-2],[2,-3],[4,-7],[2,-9],[-4,-4],[-3,-6]],[[205,5820],[-1,0],[0,1],[1,1],[0,1],[0,1],[0,1],[3,5],[-3,-9],[0,-1]],[[304,6040],[-1,-1],[0,2],[0,4],[0,5],[0,4],[1,3],[2,-3],[-1,-5],[0,-5],[-1,-4]],[[147,6350],[0,-1],[-1,2],[1,9],[0,4],[-1,3],[0,1],[0,2],[1,1],[0,-1],[0,-2],[1,-4],[0,-4],[0,-1],[0,-6],[0,-2],[-1,-1]],[[150,6457],[0,-1],[-1,2],[0,4],[0,1],[1,-1],[0,-2],[0,-3]],[[0,6833],[8,-16],[4,-2],[1,0],[1,3],[-1,3],[0,1],[0,2],[0,4],[0,1],[1,-2],[0,-2],[1,-2],[0,-7],[1,-3],[0,-1],[1,-4],[1,-2],[1,-4],[-1,0],[0,-3],[0,-2],[0,-2],[1,-1],[0,-3],[2,-2],[1,2],[1,-1],[0,-2],[1,-3],[2,0],[4,-4],[2,-5],[5,-18],[2,-8],[2,-10],[1,-3],[-1,-1],[-1,1],[-2,11],[-1,2],[-1,4],[-1,4],[0,-2],[0,-2],[1,-4],[0,-3],[0,-3],[0,-3],[0,-3],[0,-3],[0,-4],[0,-4],[0,-2],[1,-2],[1,-2],[5,-4],[1,-2],[1,-6],[1,-1],[2,1],[1,-1],[1,-5],[4,-6],[1,-3],[0,-11],[1,-5],[0,-3],[1,-1],[-1,-1],[0,-7],[-2,-6],[-1,-3],[1,-2],[0,-1],[2,0],[0,1],[1,3],[-1,0],[0,1],[0,2],[1,2],[0,1],[1,2],[1,0],[0,3],[1,3],[3,0],[1,4],[-7,19],[-7,19],[-1,2],[-2,3],[0,3],[-1,1],[0,3],[-1,1],[0,1],[9,-23],[10,-24],[1,-7],[5,-11],[-3,4],[-1,-1],[0,-1],[0,-3],[0,-1],[-4,-5],[1,-2],[1,-2],[1,-4],[1,-1],[0,-3],[0,-2],[0,-2],[0,-1],[0,1],[5,12],[1,1],[2,-1],[4,-10],[1,-2],[-1,0],[-1,2],[-1,1],[0,-2],[0,-3],[1,-4],[0,-1],[0,-1],[0,-6],[2,-1],[2,4],[0,2],[-1,1],[1,0],[4,-11],[3,-8],[6,-15],[1,-4],[0,-1],[-4,8],[0,-3],[-1,-3],[-1,-3],[1,-2],[-1,-4],[0,-2],[3,-5],[1,-4],[2,-3],[1,-1],[0,-5],[1,-2],[3,2],[0,-1],[4,1],[1,1],[-2,5],[-2,5],[-2,6],[-1,3],[-2,3],[0,3],[4,-9],[3,-9],[6,-11],[3,-6],[1,-2],[0,-1],[-2,2],[-1,-1],[-1,-5],[-1,-6],[-1,-7],[1,-5],[1,-2],[1,-4],[-1,-2],[0,1],[-3,0],[-1,-1],[-2,0],[-1,-4],[0,-3],[1,-1],[1,-6],[1,-6],[3,-4],[3,1],[1,2],[1,0],[1,-3],[0,-4],[0,-2],[0,-4],[1,-1],[2,-1],[1,1],[1,-3],[0,2],[1,2],[0,4],[0,4],[-1,1],[0,4],[-1,3],[-2,0],[-1,-2],[-1,1],[0,4],[1,4],[0,6],[-1,3],[-1,2],[-2,-1],[0,1],[2,5],[2,5],[1,3],[0,5],[0,3],[9,-22],[3,-8],[3,-5],[1,-3],[0,-3],[-1,0],[-1,-2],[1,-4],[0,-4],[1,-4],[0,-3],[0,-4],[1,-3],[2,-12],[0,1],[-2,8],[0,-1],[0,-3],[0,-3],[1,-3],[-1,-3],[1,-2],[0,-4],[0,-3],[0,-2],[0,-1],[-1,-1],[-1,0],[0,-3],[0,-3],[0,-2],[-3,-3],[0,-6],[1,-4],[1,0],[1,0],[1,-1],[1,0],[1,1],[1,1],[1,2],[0,2],[1,3],[0,2],[1,2],[0,3],[0,3],[1,2],[-1,4],[-1,3],[-1,2],[0,5],[3,-9],[1,-5],[0,-3],[0,-3],[0,-5],[1,1],[1,0],[1,-5],[1,-9],[1,-6],[-2,-1],[0,-1],[-1,-1],[0,-2],[0,-2],[0,-3],[0,-1],[0,-1],[-1,-2],[1,-3],[-1,-1],[0,-2],[0,-2],[0,-5],[0,-3],[0,-6],[0,-5],[0,-4],[0,-4],[-1,-4],[1,-5],[0,-3],[-1,-4],[1,-3],[1,-3],[-1,-4],[1,-5],[1,-3],[1,-6],[2,-6],[-1,-1],[0,-1],[-1,-2],[1,-3],[0,-3],[0,-3],[1,-5],[0,-5],[0,-6],[0,-5],[1,-4],[0,-7],[-1,-4],[0,7],[-2,2],[0,-2],[-1,-1],[0,-3],[0,-1],[-1,-4],[0,-2],[-1,-1],[-1,-3],[-1,1],[0,-2],[0,-4],[1,-3],[1,-2],[0,-4],[1,-1],[0,-4],[1,-4],[3,2],[3,-6],[2,-7],[2,-2],[1,-2],[0,-7],[1,-7],[-1,1],[-1,-4],[0,-4],[0,-6],[0,-5],[0,-7],[1,-4],[1,0],[0,-4],[-1,2],[-1,-1],[0,-5],[0,-3],[0,-8],[1,-2],[2,2],[1,6],[0,6],[0,4],[-2,-1],[2,4],[1,4],[1,4],[2,3],[0,4],[0,4],[0,4],[-2,-1],[0,2],[1,5],[1,-4],[1,0],[1,3],[2,1],[1,-2],[1,-5],[-1,-5],[0,-5],[1,-3],[0,-2],[1,-4],[0,-3],[0,-4],[-1,0],[-1,-2],[0,-3],[-1,-3],[0,-5],[0,-4],[0,-4],[0,-4],[0,-4],[1,0],[0,2],[1,4],[1,4],[1,5],[2,3],[0,3],[-1,3],[-1,0],[0,4],[1,2],[1,0],[1,1],[1,2],[1,4],[-1,7],[1,3],[0,3],[0,4],[0,1],[-1,3],[-2,3],[0,5],[-1,4],[-1,2],[-1,1],[-1,0],[-1,3],[-1,0],[-1,0],[-1,2],[-1,2],[-1,-3],[-1,7],[-2,1],[2,6],[3,6],[2,3],[1,5],[2,5],[0,5],[-1,4],[0,5],[0,4],[1,2],[-1,4],[0,4],[0,2],[0,5],[0,4],[0,5],[-1,5],[-1,4],[0,4],[0,3],[0,4],[0,5],[0,3],[-1,4],[-1,0],[-2,1],[-1,-1],[-1,-1],[0,1],[1,4],[1,4],[-1,4],[-1,2],[-1,3],[-4,-5],[-2,-2],[0,1],[4,10],[-1,0],[-5,-7],[-1,2],[0,5],[1,3],[10,1],[8,0],[5,-1],[2,3],[0,7],[2,-2],[1,-4],[2,-3],[1,-1],[0,-1],[-1,-1],[0,-1],[0,-5],[1,-3],[0,1],[1,3],[-1,4],[1,3],[1,1],[2,-3],[4,-4],[2,-2],[0,-1],[0,-1],[-2,0],[-1,1],[-1,3],[-1,0],[0,-1],[0,-2],[0,-2],[0,-2],[1,-1],[1,-1],[1,-2],[0,-2],[0,-3],[0,-1],[-1,2],[-1,0],[-1,0],[-1,-2],[0,-4],[0,-4],[0,-3],[0,-2],[0,-2],[0,-2],[0,-2],[0,-3],[1,-3],[-1,-2],[0,-2],[0,-5],[0,-3],[0,-2],[2,1],[1,0],[1,-1],[2,4],[0,2],[0,1],[-1,2],[-1,1],[0,2],[0,3],[0,3],[0,1],[-2,1],[0,2],[1,1],[1,0],[0,4],[0,4],[0,3],[0,2],[1,1],[1,2],[0,3],[1,-1],[0,-3],[-1,-2],[1,-4],[0,-5],[2,-4],[3,-3],[2,-3],[4,-2],[1,2],[0,4],[0,3],[1,0],[1,-3],[1,-1],[3,3],[1,4],[-4,11],[-4,3],[-5,5],[-3,3],[0,-1],[-1,-2],[-1,1],[-1,2],[1,3],[1,1],[5,-2],[2,-2],[2,-2],[7,-7],[6,-6],[4,-3],[2,3],[1,1],[0,2],[0,1],[1,-2],[0,-2],[2,-3],[1,1],[2,-1],[2,-5],[0,-5],[-1,-3],[0,-4],[0,-7],[1,-5],[2,-5],[4,-9],[1,1],[1,-2],[1,-6],[-1,-3],[-1,-4],[1,-10],[2,-9],[6,-18],[2,-6],[3,-6],[0,-1],[1,-1],[1,-4],[1,-3],[1,-4],[0,-1],[1,-2],[0,-5],[1,-4],[1,-4],[2,-10],[0,-1],[2,-3],[3,-7],[0,-2],[-1,-1],[-2,-2],[-1,-3],[0,-3],[0,-4],[0,-5],[-2,1],[-1,3],[0,-4],[1,-2],[3,0],[0,4],[1,4],[1,4],[1,3],[1,2],[1,0],[1,-1],[2,-4],[1,-2],[0,-4],[1,-6],[1,-8],[-1,-3],[-1,-1],[-1,-1],[-1,3],[0,-1],[-1,0],[-2,1],[4,-9],[2,-2],[1,2],[0,5],[2,0],[1,-6],[1,0],[1,2],[0,3],[0,2],[-1,2],[-2,2],[0,1],[1,-1],[1,-1],[1,-4],[3,-3],[3,-6],[0,-1],[0,-1],[0,-2],[1,-4],[0,-3],[1,-5],[-1,-3],[0,-4],[-1,-4],[0,-4],[-2,0],[-1,0],[-2,4],[0,5],[-3,-1],[-2,-4],[-2,-4],[-3,-12],[-3,-5],[-1,-3],[-1,-4],[-2,-9],[-3,-2],[-1,-3],[1,-1],[1,2],[1,-1],[0,-5],[0,-3],[-1,-14],[1,-7],[0,-3],[0,-4],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[1,-2],[-1,-7],[0,-6],[-2,-4],[-2,-1],[-6,6],[0,2],[0,4],[-4,9],[-3,3],[-1,4],[-1,9],[-1,6],[-4,11],[-2,3],[0,-3],[1,-6],[0,-5],[1,-4],[0,-3],[2,-3],[0,-1],[-1,0],[0,-5],[0,-2],[2,-4],[1,-7],[0,-1],[2,-1],[1,-1],[2,-2],[1,-3],[0,-6],[0,-4],[1,2],[1,-10],[0,-10],[-1,-5],[-1,-5],[-3,3],[-1,4],[-2,6],[-4,0],[-4,-3],[-10,-12],[-4,3],[7,4],[1,3],[-1,3],[-2,1],[-3,0],[-1,5],[-1,3],[-1,2],[0,1],[0,1],[-1,-5],[-1,-2],[-1,-3],[0,3],[-2,2],[0,3],[1,2],[0,4],[0,3],[-1,4],[-2,3],[1,3],[1,3],[-1,5],[-1,-3],[-1,4],[0,1],[-1,1],[-1,0],[-1,2],[-6,4],[-2,-4],[-1,0],[-1,1],[-1,2],[1,-7],[2,-2],[3,2],[0,-2],[-1,-3],[0,-7],[1,-1],[2,-2],[3,-1],[2,-6],[1,-5],[-1,-5],[0,-4],[2,-6],[0,-3],[0,-6],[-1,4],[-1,-1],[-1,-2],[0,-1],[1,-2],[0,-3],[0,-3],[1,-2],[3,-1],[3,6],[1,10],[1,0],[-2,-16],[-1,-17],[0,-6],[0,-7],[0,-8],[0,-6],[-1,0],[-1,-2],[0,5],[-1,3],[-1,2],[-1,0],[-1,0],[-1,-3],[-2,0],[-1,-1],[0,-2],[-1,0],[-1,5],[-1,1],[-1,0],[0,-1],[0,-2],[0,-3],[0,-3],[0,-2],[1,0],[3,-6],[3,4],[2,-3],[2,3],[0,-2],[3,1],[1,-9],[-1,-7],[0,-5],[-1,0],[-1,2],[0,-1],[0,-3],[1,-6],[0,-2],[1,-2],[0,2],[0,2],[1,1],[2,-2],[0,-5],[0,-6],[-3,-8],[-1,-8],[-3,-9],[-4,-7],[-1,-1],[-1,3],[0,-1],[-2,-5],[0,-1],[-1,-2],[-1,-3],[-1,-4],[-1,-6],[-1,2],[-3,3],[-3,-1],[-1,-4],[0,-7],[-1,-7],[-2,-4],[0,-1],[3,2],[1,7],[1,4],[2,2],[1,-1],[2,-4],[1,-2],[0,-8],[0,-7],[-1,-2],[-1,-4],[-1,-2],[-1,-1],[-1,-2],[-2,-3],[-1,-4],[0,-3],[2,2],[2,2],[1,0],[1,-4],[-1,-5],[-1,-8],[-2,-5],[0,-3],[2,1],[1,3],[1,4],[1,4],[0,-1],[1,-1],[1,-1],[2,-1],[0,-1],[1,-3],[4,-5],[1,-2],[0,-4],[0,-4],[1,-7],[5,-20],[-3,1],[-2,-1],[-1,-2],[-1,-3],[-2,-3],[-1,-1],[-2,5],[0,1],[-1,2],[-1,-1],[0,5],[0,6],[0,2],[-1,2],[0,2],[0,3],[-1,2],[0,2],[0,2],[-1,-2],[-1,0],[-2,-3],[-2,-6],[1,-5],[2,2],[2,2],[1,-4],[0,-6],[0,-4],[0,-9],[-1,-6],[0,-4],[0,-4],[-1,-1],[-1,9],[-1,0],[0,-7],[0,-5],[1,-6],[-3,-1],[-3,-5],[-3,7],[-1,5],[-3,-1],[0,7],[0,7],[0,6],[2,6],[1,10],[0,4],[0,1],[-1,0],[0,4],[0,4],[0,4],[1,5],[-1,4],[0,2],[0,1],[-1,0],[0,2],[0,2],[0,5],[0,1],[-1,1],[0,1],[-1,1],[0,4],[0,-5],[0,-8],[1,-8],[1,-6],[0,-7],[-1,-5],[0,-9],[0,-7],[-2,-7],[-2,-6],[-2,-3],[-1,1],[-2,4],[-4,8],[-1,3],[-3,4],[-1,8],[0,7],[0,6],[-1,8],[0,5],[-2,5],[-1,4],[-2,2],[-1,3],[-1,5],[-1,4],[1,2],[1,0],[-1,4],[-1,1],[-2,-1],[-1,1],[-5,13],[-1,4],[1,3],[0,8],[1,8],[0,10],[-1,10],[-1,0],[-1,-7],[0,-9],[0,-7],[-2,-1],[-2,0],[-1,-1],[-1,8],[0,6],[-1,2],[0,-4],[0,-5],[0,-5],[0,-5],[-1,-5],[-2,-1],[-3,1],[-4,2],[-1,3],[-2,-3],[-1,6],[1,2],[2,5],[0,5],[-1,-1],[-1,-2],[-2,2],[0,5],[0,3],[-1,0],[0,2],[-1,2],[0,3],[-1,3],[-1,1],[-2,3],[-2,2],[0,4],[0,4],[-2,4],[0,3],[0,3],[3,6],[-1,6],[-1,3],[0,3],[1,4],[1,5],[0,6],[0,6],[-1,3],[-3,18],[1,7],[0,3],[-1,9],[0,10],[0,7],[-2,5],[0,3],[-2,4],[-1,1],[-3,2],[-5,9],[-7,10],[-10,16],[-3,-4],[-2,-6],[-3,-7],[-1,-12],[-3,-12],[-10,0],[-7,0],[-7,1],[-5,7],[-1,5],[-1,4],[0,3],[0,4],[0,2],[1,4],[0,4],[0,6],[1,4],[0,6],[0,5],[1,4],[0,8],[-2,9],[-1,3],[-2,-8],[-1,0],[-1,10],[1,1],[0,2],[-1,0],[-1,1],[0,1],[0,2],[-1,10],[-1,2],[-3,12],[-1,6],[1,4],[0,10],[-3,18],[1,0],[1,-2],[1,-5],[1,-4],[1,-2],[0,-1],[2,0],[2,5],[0,8],[-1,7],[1,7],[1,1],[2,-1],[1,4],[1,5],[-1,4],[0,4],[0,5],[1,5],[0,2],[-1,1],[0,5],[0,6],[0,1],[0,2],[1,4],[0,3],[0,2],[-1,1],[0,3],[2,8],[-1,4],[0,-1],[-1,-2],[0,-3],[0,-2],[-1,1],[-1,1],[0,-3],[0,-3],[0,-3],[0,-3],[-1,-4],[-1,-9],[-2,-12],[0,-4],[-1,-6],[-3,-5],[-3,3],[-2,10],[0,3],[-1,7],[-1,3],[0,4],[1,5],[1,4],[-1,10],[-1,10],[-1,7],[-1,-4],[1,-7],[0,-7],[0,-7],[0,-6],[-2,0],[-1,2],[-1,6],[-3,5],[1,-12],[0,-1],[1,-2],[1,-2],[0,-6],[1,-5],[2,-12],[-1,-9],[-2,-4],[-3,-2],[-4,-2],[-3,4],[0,2],[0,1],[0,1],[2,0],[1,1],[0,3],[-2,2],[-1,5],[0,-1],[-1,-5],[-2,-15],[0,-7],[0,-10],[0,-3],[0,-3],[-1,-2],[0,-1],[0,-3],[0,-9],[0,-4],[1,-28],[1,-5],[1,-4],[7,-26],[2,-4],[3,-7],[-1,-6],[0,-6],[0,-2],[2,-10],[-2,-8],[-1,-8],[-1,-4],[-1,-3],[-1,-6],[-1,-5],[0,-7],[0,-4],[-1,-5],[1,-5],[-1,-8],[0,-10],[-2,-9],[-1,-8],[-1,-6],[-2,0],[-1,-4],[-1,-5],[0,-4],[-2,-3],[0,-1],[-1,-2],[-2,0],[0,963]],[[9729,7027],[0,-2],[-1,0],[-1,-6],[-1,-7],[-2,-6],[-1,2],[0,4],[-1,1],[-1,1],[-1,2],[1,4],[2,5],[2,2],[4,0]],[[9705,7027],[0,-2],[-3,2],[-1,-1],[0,-2],[0,-2],[1,-2],[-1,-6],[0,-4],[1,-4],[0,-5],[0,-5],[-1,-5],[-1,-8],[-1,-4],[-2,-1],[-2,1],[-5,-2],[-10,16],[-10,15],[-1,5],[-1,6],[-1,6],[-2,3],[-3,6],[-2,2],[-1,6],[1,6],[1,6],[4,19],[6,12],[1,1],[1,3],[10,-9],[2,-4],[2,-2],[2,-6],[11,-8],[1,-3],[3,-2],[1,-3],[1,-6],[0,-4],[0,-2],[0,-1],[0,-4],[-1,-2],[0,-1],[0,-2],[0,-3]],[[9845,5071],[0,1],[0,2],[0,2],[0,1],[0,1],[0,1],[0,1],[0,1],[1,1],[1,1],[0,2],[0,1],[-1,3],[0,1],[0,1],[0,1],[-1,0],[0,1],[0,1],[0,1],[0,6],[0,1],[0,1],[0,4],[0,3],[0,1],[0,1],[0,1],[-1,3],[0,1],[0,4],[-1,1],[-1,3],[0,1],[0,1],[0,1],[0,2],[0,1],[0,3],[0,2],[0,1],[0,1],[-1,1],[0,1],[-1,0],[0,-1],[-1,-5],[0,-1],[0,-1],[-3,2],[-1,1],[0,2],[0,1],[1,3],[0,1],[0,1],[0,1],[0,3],[-1,1],[0,2],[0,1],[0,1],[0,2],[0,1],[0,5],[0,1],[0,1],[0,2],[0,1],[0,2],[-1,1],[0,6],[0,2],[-1,6],[0,1],[0,2],[0,1],[0,2],[0,4],[0,3],[1,1],[0,1],[1,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,2],[0,7],[0,2],[0,2],[0,1],[0,1],[0,1],[0,1],[0,4],[0,1],[0,1],[-1,1],[-3,-2],[-1,1],[0,-1],[-1,0],[0,-1],[0,1],[-1,1],[0,3],[0,1],[0,1],[-1,5],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[-3,5],[-1,0],[0,-1],[-1,-4],[-1,0],[-1,1],[0,1],[-1,2],[-1,-1],[0,-1],[-1,-4],[0,-2],[-1,-2],[-5,-13],[0,-1],[-1,0],[0,-2],[0,-1],[0,-1],[0,-3],[0,-1],[0,-1],[-1,-1],[-2,2],[0,-1],[-1,-4],[0,-1],[-1,-1],[-1,1],[-1,4],[-1,0],[-1,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-3],[0,-1],[0,-1],[0,-1],[-1,0],[-1,1],[0,1],[0,1],[-2,12],[-1,2],[-4,5],[-1,4],[-1,0],[-2,-2],[-2,-7],[-1,0],[-1,1],[0,1],[0,1],[0,2],[0,1],[-1,4],[0,1],[0,1],[-1,1],[-2,1],[-1,1],[-1,1],[0,-1],[-1,-1],[-1,-2],[0,-1],[0,-3],[0,-1],[0,-1],[-1,0],[-1,-1],[-1,-3],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-1],[-4,-2],[-1,-2],[0,-2],[-1,-1],[0,-1],[0,-1],[-2,-2],[0,-1],[-1,-1],[0,-2],[0,-1],[-1,0],[-2,2],[0,1],[-1,1],[0,1],[0,4],[0,2],[0,-1],[-1,-2],[0,-2],[-1,-2],[0,-1],[-1,0],[0,2],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[-1,1],[0,-1],[-1,-1],[0,-1],[0,-3],[-1,-1],[-2,2],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[1,-5],[0,-1],[0,-1],[0,-1],[-4,-1],[-1,-1],[0,-1],[0,-2],[-1,-4],[0,-1],[-1,1],[0,2],[-1,3],[-1,2],[-5,6],[-2,-1],[-2,3],[-1,0],[-1,-2],[0,-1],[-1,-3],[0,-2],[-1,-1],[0,1],[0,1],[-1,4],[-1,8],[-1,1],[0,1],[0,2],[0,1],[0,1],[-1,2],[0,2],[0,2],[0,2],[0,1],[0,4],[0,1],[-1,1],[-1,3],[0,1],[0,1],[-1,2],[0,2],[0,1],[1,0],[0,2],[1,0],[0,1],[0,2],[0,1],[0,1],[0,1],[-2,7],[0,1],[0,1],[-1,0],[0,2],[0,2],[0,2],[0,2],[1,12],[0,1],[0,2],[0,1],[0,2],[0,1],[0,1],[1,1],[0,2],[0,1],[0,1],[0,1],[-1,2],[0,1],[-1,-1],[0,-1],[0,-2],[-1,-1],[-2,2],[0,-1],[0,-1],[-1,-1],[-4,11],[0,1],[-1,1],[-1,4],[-1,5],[0,2],[0,1],[0,1],[0,2],[0,1],[0,1],[-2,9],[-1,4],[-1,1],[-1,1],[0,2],[-1,2],[0,1],[0,2],[0,1],[0,1],[-2,6],[-1,2],[-1,1],[0,1],[-1,1],[-1,6],[0,3],[-1,3],[0,1],[0,1],[-1,0],[-1,0],[-1,-1],[0,-3],[-1,-1],[-1,-1],[0,-2],[-1,0],[0,1],[-1,3],[-2,4],[-3,10],[-1,2],[-1,0],[-3,7],[-1,6],[-1,3],[1,1],[2,8],[0,3],[1,2],[0,3],[0,4],[0,1],[0,2],[1,1],[2,-2],[0,1],[1,0],[0,1],[0,1],[0,2],[-1,2],[0,2],[0,2],[0,2],[0,1],[0,1],[0,1],[1,0],[1,-2],[1,1],[0,2],[0,1],[0,2],[0,2],[0,1],[-1,3],[0,2],[1,1],[0,1],[2,4],[0,1],[1,3],[0,4],[1,3],[0,1],[0,1],[2,4],[2,12],[1,3],[0,1],[0,3],[0,1],[1,3],[1,8],[1,3],[1,1],[0,1],[1,-1],[1,0],[0,1],[1,4],[1,1],[0,1],[2,1],[0,1],[0,1],[1,1],[0,2],[0,1],[0,1],[-1,1],[0,1],[0,1],[0,3],[1,1],[0,2],[0,1],[0,1],[1,1],[1,0],[0,1],[0,1],[0,1],[0,2],[0,1],[-1,6],[0,2],[0,2],[0,4],[0,1],[0,1],[0,1],[0,5],[0,2],[0,1],[0,2],[0,2],[0,1],[0,2],[0,1],[0,1],[0,1],[0,2],[-1,1],[0,2],[-1,5],[0,1],[-1,1],[-1,0],[-1,1],[0,2],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[-1,1],[0,3],[0,1],[0,1],[0,2],[0,2],[0,1],[0,1],[-1,0],[-2,1],[-1,-1],[-1,-1],[-1,1],[-1,2],[-6,15],[0,1],[-1,2],[0,1],[0,1],[0,4],[0,1],[0,1],[0,2],[0,1],[0,1],[0,4],[-1,2],[0,3],[0,1],[0,1],[-1,3],[0,1],[0,1],[0,1],[0,2],[0,1],[0,3],[0,1],[0,1],[0,1],[0,1],[-1,1],[-3,4],[0,1],[0,1],[-1,3],[-1,14],[-1,4],[0,2],[0,1],[-1,2],[-2,0],[-1,-1],[-1,0],[-2,5],[0,1],[-1,0],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-2],[-1,-2],[0,-1],[0,-1],[0,-2],[-1,0],[-1,0],[-2,-1],[-4,8],[-1,3],[0,2],[0,1],[0,1],[0,1],[-1,0],[-2,-2],[0,-1],[0,1],[-1,1],[0,2],[-1,3],[0,1],[0,1],[0,2],[-1,2],[0,1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[-1,1],[-2,7],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[-3,0],[0,1],[0,1],[-1,1],[0,2],[0,1],[0,5],[0,1],[-1,3],[0,3],[0,1],[0,1],[-1,1],[0,1],[0,1],[0,3],[0,1],[0,1],[-1,3],[0,1],[0,2],[0,1],[-3,-1],[-1,-3],[0,-1],[0,-1],[-1,-1],[0,-3],[0,-3],[-1,-1],[-1,-4],[-4,-1],[-1,-3],[-1,0],[-1,3],[0,1],[-1,1],[-3,0],[-6,12],[-1,1],[-1,-1],[0,-1],[-1,-1],[-2,4],[-1,0],[0,-1],[-1,-1],[0,-2],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[-1,-1],[-2,1],[-2,-2],[-1,2],[-1,3],[0,2],[0,2],[2,5],[0,1],[0,1],[-2,4],[-2,11],[0,1],[0,1],[0,3],[0,3],[0,1],[0,1],[0,2],[-2,6],[0,1],[0,1],[-1,0],[-1,-2],[-1,0],[0,1],[-1,3],[0,1],[0,-1],[-1,-4],[0,-1],[0,-1],[-1,-1],[0,1],[-1,1],[0,2],[0,1],[-1,0],[-1,0],[0,1],[0,1],[0,2],[-1,1],[0,1],[1,1],[0,1],[0,2],[0,2],[-2,9],[0,1],[-4,2],[-2,-3],[-1,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-5],[-1,-2],[0,-1],[1,-1],[0,-2],[0,-3],[0,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-3],[0,-1],[0,-1],[0,-1],[-1,-1],[-2,-2],[-1,1],[0,1],[-1,1],[0,1],[0,2],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[-1,3],[0,1],[0,1],[0,1],[0,2],[0,1],[-3,3],[0,1],[0,1],[0,1],[-1,2],[0,1],[-1,2],[-1,0],[0,2],[-2,2],[0,1],[0,1],[-1,1],[0,3],[0,3],[0,1],[-1,0],[-6,12],[-1,2],[0,1],[-1,0],[0,-2],[0,-2],[0,-1],[-1,-3],[0,-1],[0,1],[-1,0],[-1,-1],[0,-1],[0,-1],[0,-3],[-1,-1],[-2,0],[-2,-2],[-1,0],[0,1],[0,1],[-1,3],[0,1],[0,1],[-1,0],[-2,-1],[-1,1],[0,1],[0,1],[-1,1],[0,3],[0,1],[0,1],[-1,0],[0,1],[-2,0],[0,-1],[-1,-2],[0,-2],[0,-2],[0,-2],[0,-2],[0,-1],[-1,-2],[0,-1],[-1,-5],[-1,-7],[-1,-1],[-2,-2],[0,1],[-1,2],[-1,0],[0,-1],[-1,-2],[-1,-2],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[1,-2]],[[9531,5797],[-1,-13],[0,-6],[0,-2],[0,-1],[-1,-2],[-1,-4],[0,-1],[-1,-1],[-8,3],[0,2],[-1,4],[0,1],[-1,1],[0,1],[0,1],[0,1],[0,2],[0,3],[-1,1],[0,1],[-2,3],[-1,2],[-5,-3],[-1,0],[-1,4],[-1,2],[0,1],[-10,11],[-1,3],[-1,1],[0,2],[0,1],[0,1],[0,3],[0,2],[0,1],[0,2],[0,1],[-1,4],[-1,3],[0,1],[0,1],[0,2],[0,1],[0,4],[0,1],[0,2],[0,4],[0,1],[0,1],[-1,2],[-1,5],[-6,14],[0,1],[-1,1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[-1,2],[0,2],[-2,4],[-7,8],[-9,-6],[-5,4],[0,1],[0,2],[0,1],[-1,2],[0,1],[0,2],[-1,3],[0,2],[-1,1],[0,2],[0,2],[0,1],[0,2],[-2,8],[-1,7],[-1,2],[-2,6],[-1,7],[0,2],[0,1],[0,1],[0,1],[0,2],[1,5],[0,1],[0,1],[0,2],[-1,3],[0,1],[-1,3],[-1,4],[-2,7],[-1,2],[0,1],[-2,-1],[-2,2],[-1,3],[-2,5],[-10,23],[-3,3],[-9,18],[-1,2],[0,4],[0,1],[0,2],[0,2],[0,1],[1,4],[0,2],[0,2],[0,3],[0,2],[0,3],[0,4],[0,2],[0,2],[0,2],[0,2],[0,1],[1,4],[0,1],[0,2],[1,1],[0,6],[0,4],[0,1],[0,4],[1,2],[0,2],[0,4],[0,3],[1,5],[0,3],[1,2],[0,2],[1,2],[0,6],[1,4],[0,2],[0,1],[0,2],[0,2],[0,2],[0,3],[0,2],[0,3],[0,2],[0,2],[-1,1],[-4,7],[-3,0],[-3,5],[-1,2],[-3,15],[-1,1],[0,1],[-2,-1],[0,1]],[[9403,6188],[-2,6],[-1,8],[-1,3],[0,1],[-1,2],[0,2],[0,1],[0,2],[1,1],[2,3],[0,1],[0,2],[0,2],[0,4],[0,1],[0,2],[1,4],[0,1],[0,4],[1,2],[0,1],[0,1],[1,2],[1,1],[0,3],[0,2],[1,2],[1,20],[0,5],[1,2],[1,2],[1,2],[3,9],[0,2],[0,2],[0,1],[0,1],[0,1],[0,3],[0,1],[-1,2],[0,1],[0,1],[0,2],[0,2],[0,1],[0,3],[0,1],[0,1],[-1,3],[-1,1],[0,2],[0,1],[0,2],[0,1],[0,2],[0,2],[0,2],[0,4],[0,2],[0,2],[0,1],[-1,7],[-1,2],[0,2],[0,2],[0,2],[0,1],[-1,1],[-6,15],[-1,1],[-2,-1],[-1,1],[-1,6],[0,1],[-2,4],[-1,1],[0,5],[0,1],[-3,6],[-3,10],[-1,1],[-1,0],[0,1],[-1,0],[0,1],[0,2],[-1,7],[-1,5],[0,3],[0,2],[0,1],[0,2],[1,5],[1,3],[0,4],[1,1],[0,2],[0,2],[0,2],[0,2],[-1,1],[0,1],[-3,4],[0,1],[0,2],[0,1],[0,2],[0,2],[0,1],[0,2],[0,1],[-1,2],[-1,2],[0,1],[0,2],[1,7],[0,1],[1,3],[0,1],[0,1],[-1,1],[0,2],[0,1],[0,2],[1,6],[0,1],[0,1],[0,1],[0,1],[1,1],[0,1],[0,1],[0,4],[0,2],[1,1],[3,8],[4,-1],[1,-3],[1,-1],[0,1],[1,2],[2,0],[1,2],[1,1],[0,2],[0,1],[0,1],[0,1],[0,1],[-1,2],[0,2],[0,2],[-1,8],[0,1],[0,1],[-1,2],[-3,0],[-1,1],[0,1],[0,2],[-1,2],[0,5],[0,2],[1,2],[0,1],[0,1],[0,2],[0,1],[0,1],[3,4],[0,1],[0,1],[0,1],[0,2],[1,6],[0,2],[0,2],[1,12],[0,4],[1,2],[4,9],[6,7],[8,1],[1,1],[0,2],[0,2],[1,2],[3,4],[10,8],[1,3],[1,1],[4,0],[1,-1],[0,-5],[0,-1],[1,0],[1,1],[0,1],[0,1],[0,2],[0,5],[0,1],[0,2],[0,1],[1,2],[0,1],[32,5],[1,2],[1,4],[1,3],[4,12],[4,6],[1,1],[6,-6],[2,-3],[0,-1],[13,-4],[0,-1],[1,-1],[0,-2],[5,-7],[1,-1],[0,1],[0,1],[0,2],[0,1],[1,1],[0,1],[1,3],[1,1],[0,1],[0,2],[0,1],[0,2],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[1,9],[0,1],[0,1],[1,2],[-1,1],[0,1],[0,1],[0,3],[0,1],[0,2],[0,2],[0,1],[0,1],[1,4],[0,1],[0,3],[1,1],[0,1],[1,7],[1,2],[0,2],[0,1],[0,1],[0,1],[0,1],[0,2],[-1,6],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[-1,2],[-1,1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,2],[0,1],[0,2],[0,1],[0,2],[0,1],[0,2],[0,5],[0,2],[0,1],[0,1],[0,1],[1,-2],[1,-1],[1,1],[1,1],[0,1],[0,3],[1,2],[0,2],[1,5],[0,2],[1,1],[0,1],[-1,1],[0,1],[-1,2],[0,1],[0,1],[0,2],[0,1],[0,3],[0,1],[-1,2],[-1,3],[-5,12],[0,2],[0,1],[0,2],[0,1],[0,1],[0,2],[0,1],[0,3],[0,1],[0,1],[0,1],[-1,3],[0,1],[0,1],[0,1],[0,3],[0,1],[0,1],[0,2],[1,2],[2,6],[1,0],[1,1],[0,1],[0,2],[1,2],[1,2],[0,2],[1,2],[0,1],[0,1],[0,1],[-4,17],[-2,7],[-1,2],[0,2],[0,1],[0,1],[-1,5],[0,2],[0,2],[0,1],[0,1],[0,1],[0,2],[0,2],[-1,2],[-2,4],[-1,1],[0,1],[1,5],[0,2],[1,1],[0,1],[-1,1],[0,2],[-2,11],[0,2],[0,2],[0,2],[0,2],[0,2],[1,1],[0,2],[1,1],[0,1],[-1,5],[-1,2],[0,1],[1,2],[0,1],[2,3],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[0,2],[0,2]],[[9511,7004],[5,1],[1,-1],[2,-5],[1,-1],[7,8],[2,6],[2,3],[4,-1],[2,-3],[1,1],[2,-5],[0,-1],[3,-1],[5,3],[4,11],[2,6],[1,3],[2,-2],[1,-2],[0,-2],[0,-1],[0,-4],[0,-1],[3,-11],[2,-3],[4,-12],[3,-4],[8,-9],[12,7],[11,0],[2,-3],[1,0],[5,-7],[2,-6],[1,-2],[8,1],[8,2],[1,-5],[1,-2],[-1,-3],[6,0],[1,2],[8,23],[8,24],[1,5],[1,6],[3,8],[1,1],[2,0],[1,-1],[0,-2],[2,-6],[1,-4],[1,-1],[0,-1],[2,1],[1,-2],[-2,-5],[-1,-1],[0,-1],[-1,-7],[1,-5],[4,-7],[3,-9],[0,-2],[1,-4],[0,-3],[0,-2],[0,-2],[-1,-3],[0,-3],[0,-5],[0,-2],[0,-5],[-1,-15],[1,-7],[0,-4],[0,-4],[0,-5],[0,-6],[1,-5],[1,-3],[1,-3],[1,-2],[8,-2],[5,-6],[3,-9],[3,-3],[2,-7],[1,-2],[4,-2],[2,-5],[1,-13],[1,-12],[0,-22],[0,-10],[1,-6],[1,-4],[3,-6],[10,6],[10,6],[1,-2],[0,-2],[1,-1],[2,-2],[2,-7],[0,-2],[1,-3],[1,-1],[-2,11],[-1,5],[1,1],[0,2],[-3,2],[-1,3],[1,5],[0,3],[2,2],[2,6],[1,2],[1,-1],[0,-2],[-1,-3],[0,-2],[1,-2],[1,-5],[0,-2],[0,-3],[0,-2],[0,-1],[0,4],[0,5],[0,4],[0,2],[0,4],[0,2],[0,2],[-1,3],[0,3],[2,3],[4,10],[1,3],[0,4],[1,5],[0,2],[0,1],[1,-1],[0,-4],[1,-2],[3,2],[0,2],[-2,0],[-1,3],[-1,6],[0,6],[0,5],[-1,5],[0,3],[-1,2],[-1,1],[0,4],[0,2],[1,5],[0,2],[0,3],[0,2],[0,2],[0,2],[0,11],[-1,6],[-1,5],[-2,7],[-1,2],[0,1],[0,4],[0,7],[0,2],[0,4],[-1,1],[0,1],[-1,2],[0,3],[0,3],[0,2],[0,3],[0,5],[-1,7],[-1,5],[-1,3],[-2,0],[-1,-1],[-5,-4],[-3,1],[0,7],[0,7],[2,12],[2,5],[1,3],[2,4],[2,0],[2,4],[1,2],[-1,5],[0,2],[0,1],[0,1],[0,1],[-1,4],[0,1],[1,1],[0,4],[0,9],[0,2],[0,2],[-1,7],[0,3],[0,6],[0,6],[0,5],[0,6],[0,2],[1,4],[0,5],[0,4],[-1,3],[-3,2],[0,5],[2,2],[1,0],[2,-6],[12,-9],[8,-2],[6,-8],[11,-8],[2,-4],[9,-2],[9,-3],[1,3],[1,-1],[1,-3],[1,-1],[0,-8],[2,-7],[9,-14],[1,-6],[0,-1],[-1,-2],[0,-2],[0,-1],[0,-3],[1,0],[1,2],[3,0],[1,1],[1,6],[1,2],[0,2],[0,1],[-1,5],[-1,4],[-2,3],[-1,4],[-1,3],[-2,-2],[0,-2],[0,-3],[-1,-2],[0,1],[0,4],[0,2],[0,1],[1,2],[1,3],[1,2],[5,2],[7,-16],[0,-1],[-1,0],[2,-5],[0,-2],[1,0],[0,3],[-1,2],[1,3],[2,-4],[1,-4],[1,-2],[1,2],[1,5],[0,2],[1,1],[1,0],[5,-2],[2,2],[2,-2],[11,-6],[9,-2],[2,1],[7,4],[2,2],[6,2],[0,-3],[-3,-1],[-2,-2],[2,-3],[2,-2],[3,-4],[1,-7],[1,-1],[2,-3],[2,-2],[2,0],[2,4],[0,5],[0,4],[0,5],[-1,4],[-3,0],[-2,0],[-2,1],[-2,3],[0,3],[2,0],[2,-1],[3,1],[1,2],[1,0],[1,-1],[0,-5],[1,-2],[2,-8],[0,-2],[1,-5],[-1,0],[-1,-3],[1,-3],[0,-1],[2,1],[5,-5],[1,-4],[2,-4],[1,-2],[1,-7],[1,-3],[3,-1],[4,-9],[5,-4],[6,0],[8,-12],[1,-4],[-2,2],[0,-1],[0,-3],[2,-3],[1,0],[1,0],[4,-8],[7,-9],[1,-5],[1,0],[2,5],[0,3],[-1,1],[1,2],[2,0],[11,-18],[1,-5],[2,-1],[0,-2],[1,-2],[2,-3],[2,-6],[0,-1],[0,-2],[-4,-1],[-1,0],[0,2],[0,4],[-1,3],[0,2],[-1,1],[-2,1],[-1,-2],[0,1],[-1,2],[0,1],[0,1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-10],[0,-4],[1,-3],[1,-1],[3,4],[3,-4],[1,0],[2,4],[5,-7],[2,-3],[1,-9],[9,-28],[8,-28],[-2,1],[-1,3],[-5,17],[-1,2],[-2,6],[-1,0],[1,-5],[2,-4],[0,-3],[1,-3],[0,-5],[0,-4],[2,-2],[0,-3],[1,-1],[-9993,-2],[0,-963],[9996,-3],[-2,-6],[-1,-3],[-2,-16],[-6,-32],[-1,-1],[-2,1],[-3,-3],[-1,0],[-1,-3],[-1,-1],[-4,-11],[-3,-8],[-4,-19],[-6,-12],[-1,0],[-1,5],[1,3],[5,5],[0,1],[1,3],[0,6],[0,2],[-1,1],[-1,1],[-1,-1],[-2,-4],[-9,-5],[0,1],[1,0],[0,1],[0,4],[-1,1],[-4,0],[-3,-4],[0,1],[-1,2],[0,2],[0,2],[0,1],[-1,1],[-5,0],[-2,3],[-1,0],[-1,1],[1,2],[1,4],[-1,1],[-2,7],[0,1],[-2,1],[1,3],[0,3],[-1,2],[-1,0],[-1,1],[-1,2],[1,3],[1,1],[1,0],[0,3],[-1,4],[0,3],[0,1],[0,2],[0,2],[0,2],[0,1],[0,1],[-1,1],[0,1],[-2,-1],[-1,1],[-1,2],[-1,0],[0,3],[-1,13],[-1,5],[0,1],[-5,6],[-1,5],[0,1],[-2,0],[-2,-1],[-2,-7],[-3,-4],[-1,0],[-2,6],[0,2],[-1,1],[-1,0],[0,1],[0,2],[0,1],[0,1],[-1,0],[-1,0],[1,-6],[-1,-3],[-2,0],[1,-4],[1,-1],[2,2],[1,-1],[0,-6],[1,-1],[0,-1],[1,-1],[10,12],[2,-1],[0,-4],[0,-5],[1,-2],[1,-1],[1,-2],[0,-4],[2,-7],[1,-2],[0,-2],[0,-2],[1,0],[0,-1],[0,-2],[0,-6],[0,-1],[0,-1],[0,-3],[1,-3],[1,-3],[0,-4],[0,-2],[-3,-5],[-3,-3],[-1,-2],[-1,0],[0,1],[0,4],[-2,2],[-1,10],[-2,2],[-6,4],[-1,-3],[0,-4],[-1,-4],[-1,0],[-1,-3],[-1,0],[0,2],[0,2],[0,3],[-1,5],[-4,10],[-1,2],[0,4],[-1,1],[-2,-1],[-1,2],[0,7],[-1,1],[0,-1],[-2,-6],[0,-3],[-1,-1],[0,-1],[0,-3],[0,-1],[-1,-9],[-1,-4],[-2,-5],[-1,-4],[-1,-4],[-4,5],[-2,-2],[-2,-5],[-4,-2],[-1,2],[-1,-1],[0,-2],[-1,-1],[-4,2],[-1,-2],[-4,-13],[-2,-3],[-1,-4],[0,-1],[-8,2],[1,-4],[1,-1],[7,2],[7,18],[2,3],[5,-4],[9,10],[0,-4],[1,-1],[2,1],[3,9],[1,2],[0,3],[1,2],[1,2],[0,3],[1,10],[1,4],[1,2],[1,-1],[1,-3],[3,-8],[1,-12],[1,-2],[0,-6],[0,-3],[1,-3],[0,-3],[1,-2],[0,-2],[1,-3],[0,-1],[0,-2],[1,-2],[0,-1],[-1,-3],[0,-2],[-2,-4],[-4,-5],[-1,-5],[-1,-7],[0,-2],[0,-8],[0,-2],[0,-2],[0,2],[0,2],[0,3],[1,2],[2,10],[1,1],[1,0],[1,0],[1,3],[1,2],[0,3],[1,6],[0,2],[1,1],[2,-2],[1,-2],[3,-12],[0,-2],[1,-10],[1,-2],[0,1],[1,4],[1,2],[0,1],[0,2],[0,1],[0,1],[0,1],[1,5],[1,7],[2,10],[2,2],[2,0],[1,2],[2,3],[2,7],[1,3],[4,-3],[0,-8],[-1,-13],[0,-3],[0,-2],[0,-3],[0,-2],[-1,-4],[0,-2],[0,-6],[-1,-3],[0,-3],[0,-2],[0,-3],[0,-7],[0,-7],[2,-12],[0,-6],[3,-10],[1,-9],[1,-2],[1,-7],[1,-2],[1,-2],[2,-9],[5,-10],[6,-2],[0,2],[-3,6],[0,4],[0,4],[3,6],[0,3],[1,2],[0,3],[1,4],[0,3],[0,3],[0,2],[-1,6],[0,5],[0,2],[-1,-1],[0,-3],[0,2],[0,3],[0,2],[1,-3],[5,-45],[0,-4],[0,-12],[1,-3],[3,-26],[1,-6],[-1,2],[-1,4],[-1,2],[-1,-3],[0,-2],[-1,-1],[0,-2],[-1,-9],[2,-2],[1,-1],[-1,-2],[-2,0],[0,-2],[1,-1],[2,1],[1,0],[3,-6],[1,0],[0,4],[0,3],[-1,3],[0,1],[0,2],[2,-22],[0,-3],[0,-3],[0,-5],[0,-2],[0,-3],[0,-6],[0,-8],[1,-13],[1,-23],[0,-5],[-1,4],[0,3],[-1,1],[-1,1],[0,1],[-1,2],[-1,1],[0,-1],[0,-3],[1,-4],[0,-3],[1,-2],[1,0],[1,-1],[0,-2],[-1,-4],[-1,-1],[-1,0],[0,1],[0,1],[0,4],[-1,1],[0,4],[-1,2],[-1,0],[0,-1],[0,-2],[0,-4],[-1,-4],[0,-1],[-1,1],[0,2],[0,2],[0,1],[0,1],[-1,0],[-1,2],[-1,0],[0,-1],[0,-2],[-1,-2],[0,-2],[-1,-3],[0,-2],[0,-4],[1,-4],[1,-5],[1,-2],[1,2],[0,2],[2,9],[1,1],[4,-3],[1,2],[0,-1],[0,-1],[0,-1],[0,-3],[0,-2],[1,-1],[0,-1],[-1,-3],[1,-2],[0,-2],[0,-2],[0,-2],[-1,-3],[0,-1],[-1,-1],[0,-2],[0,-2],[0,-1],[-1,-2],[-1,-5],[0,-2],[2,-5],[1,-1],[3,8],[1,4],[-1,5],[-1,9],[-1,11],[1,11],[0,3],[0,-2],[0,-3],[0,-3],[0,-4],[1,-8],[0,-5],[5,-42],[-1,4],[-1,9],[-2,7],[-1,-3],[0,-10],[3,-6],[0,-8],[3,-1],[1,-2],[2,-10],[1,-2],[0,1],[1,1],[0,1],[1,-1],[2,-6],[1,-5],[0,-2],[0,-3],[0,-1],[1,-1],[0,-1],[1,-5],[0,-2],[0,-4],[0,-10],[0,-2],[-1,-3],[0,2],[-1,1],[-1,-1],[0,-1],[-1,-4],[-1,-5],[0,-4],[0,-6],[1,-14],[1,-5],[1,-3],[3,-2],[1,-3],[1,-6],[0,-3],[0,-7],[1,-3],[0,-3],[0,-1],[0,-1],[1,-1],[0,-2],[0,-4],[0,-8],[0,-2],[0,-3],[0,-2],[-1,-2],[0,-2],[0,-3],[0,-1],[0,-3],[0,-1],[0,-2],[0,-1],[0,-1],[-3,0],[-1,-1],[0,-5],[0,-6],[0,-6],[-1,-7],[-1,-3],[-1,-2],[-3,1],[-2,-2],[-1,-6],[0,-6],[1,-7],[0,-9],[0,-4],[0,-3],[0,-3],[0,-2],[0,-2],[-1,-7],[0,-2],[-1,0],[0,1],[-1,6],[0,1],[0,1],[-1,7],[-1,3],[-4,4],[-2,5],[-7,9],[-3,8],[-2,1],[-1,3],[-2,4],[-2,6],[-2,3],[-4,5],[-6,4],[-11,1],[0,1],[1,0],[0,1],[-1,4],[-1,2],[1,10],[0,5],[1,1],[0,1],[0,1],[0,2],[0,8],[0,1],[0,1],[0,3],[0,1],[0,3],[0,1],[2,1],[1,0],[0,2],[-1,1],[0,2],[1,1],[0,2],[0,1],[0,1],[0,2],[-2,-2],[-1,-1],[-1,-5],[-1,-7],[0,-7],[0,-1],[0,-4],[-2,0],[-1,2],[-1,7],[1,7],[0,3],[0,2],[-2,6],[-1,3],[0,2],[-1,1],[0,2],[0,1],[0,1],[0,1],[0,1],[-1,0],[0,-2],[0,-5],[0,-6],[1,-4],[0,-2],[0,-2],[1,-2],[0,-2],[0,-3],[-1,-2],[-2,-9],[0,-4],[0,-5],[1,-8],[1,-5],[0,-1],[0,-4],[1,-3],[0,-1],[1,-1],[0,1],[3,3],[2,-2],[-1,-5],[-6,-5],[-4,-7],[-4,-5],[-1,0],[0,3],[0,2],[0,1],[1,1],[0,3],[-1,0],[-1,-1],[0,-3],[0,-6],[-1,-1],[0,-1],[-1,-1],[1,-3],[0,-1],[1,2],[0,-2],[-2,-4],[-1,-2],[-3,-10],[-1,-1],[0,-2],[-2,-14],[-1,-5],[-9,-15],[-9,-16],[-2,-8],[-3,-4],[-3,-15],[-1,-1],[0,2],[1,2],[1,3],[0,4],[-1,-3],[-1,0],[-1,2],[0,9],[-1,4],[0,-2],[0,-17],[0,-7],[0,-5],[-2,-5],[-10,-19],[-1,-7],[0,2],[0,2],[1,2],[-1,4],[-1,1],[-1,-1],[-1,1],[-1,5],[-1,0],[1,-6],[0,-3],[1,-3],[1,-5],[0,-2],[1,-2],[0,-3],[-1,-4],[0,-2],[-6,-11],[-1,0]],[[117,7430],[5,-7],[1,-1],[0,-1],[0,-2],[-1,-2],[0,-1],[-1,0],[0,1],[-1,2],[0,2],[0,2],[-2,0],[0,2],[-2,5],[1,0]],[[0,7327],[9996,-8],[-1,-4],[-1,-7],[-4,-10],[-6,0],[-4,1],[-3,1],[-2,-1],[-8,-17],[-2,-1],[0,2],[0,5],[0,2],[-1,10],[0,3],[-1,9],[0,3],[0,3],[-1,3],[0,3],[0,2],[-1,12],[0,4],[0,2],[0,2],[1,7],[1,4],[3,13],[1,4],[1,7],[-1,1],[0,-1],[2,6],[7,17],[6,19],[1,3],[2,9],[2,5],[4,2],[2,3],[-1,2],[0,1],[1,2],[1,0],[1,0],[0,2],[0,2],[0,2],[4,5],[-9999,-134]],[[0,7461],[4,0],[-2,5],[1,0],[3,3],[1,3],[0,1],[1,-1],[0,-1],[1,-1],[2,3],[6,-4],[2,-3],[1,1],[1,2],[1,3],[4,5],[9,-8],[10,-7],[2,-5],[1,-2],[-1,0],[0,1],[-1,0],[1,-3],[3,-6],[0,-1],[1,2],[1,0],[3,-9],[0,-1],[-1,-4],[1,-2],[0,-1],[-1,-2],[2,-2],[6,-16],[1,-7],[1,0],[1,-1],[0,-2],[2,-2],[1,3],[1,0],[0,-1],[0,-2],[0,-4],[1,-2],[1,-2],[0,-4],[0,-3],[0,-3],[-2,-9],[-2,-7],[-1,-7],[-2,-2],[-2,-4],[-1,-2],[-2,-8],[-2,-3],[-7,-1],[-3,-6],[-8,-4],[-8,-4],[-3,-7],[-10,-11],[0,1],[1,3],[-1,2],[-3,-4],[-1,1],[-5,12],[-5,3],[-1,3],[0,1],[-3,-1],[0,134]],[[5609,3414],[0,1],[1,3],[1,0],[0,-1],[0,-3],[0,-4],[0,-4],[1,-3],[1,-1],[1,2],[0,3],[1,1],[3,4],[5,-3],[3,5],[0,-2],[0,-1],[1,-2],[0,-5],[0,-11],[1,-5],[0,-3],[1,-2],[1,0],[1,-1],[0,-2],[0,-1],[0,-1],[0,-3],[0,-2],[2,-4],[0,-2],[1,-3],[0,-10],[0,-5],[0,-4],[0,-4],[-1,-3],[-1,-3],[0,-3],[0,-2],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[-1,-5],[0,-2],[0,-2],[0,-6],[0,-3],[0,-4],[-1,-12],[0,-19],[0,-9],[1,-9],[2,-15],[-2,-4],[-5,2],[-7,2],[-7,2],[-8,2],[-12,3],[-10,3],[-8,2],[-8,2],[-11,3],[-6,2],[-2,0],[-1,3],[-1,2],[1,1],[0,2],[3,14],[2,18],[2,14],[0,1],[0,1],[0,1],[0,1],[0,3],[0,1],[0,2],[1,2],[0,4],[1,9],[0,4],[0,6],[0,5],[-1,12],[0,3],[0,3],[0,7],[0,4],[1,4],[0,3],[1,1],[1,-4],[0,-1],[1,2],[2,-1],[2,2],[4,-3],[1,1],[2,6],[5,18],[4,31],[1,1],[3,25],[1,-2],[0,-1],[-1,-3],[0,-2],[0,-1],[0,-2],[-1,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[-1,-2],[0,-2],[0,-2],[-1,-6],[0,-2],[-2,-7],[-2,-16],[-3,-17],[-1,0],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[1,-1],[5,2],[1,-2],[2,-7],[3,-3],[2,1],[0,2],[0,1],[1,-1],[0,-2],[0,-1],[0,1],[0,1],[1,1],[0,1],[2,3],[1,3],[1,5],[-1,5],[0,8],[-1,30],[0,2],[0,1],[0,5],[0,1],[0,6],[0,2],[0,1],[0,2],[1,-1],[0,-2],[0,-1],[1,1],[0,2],[0,2],[0,3],[0,3],[0,1],[2,10],[1,0],[1,-4],[2,-15],[1,-4],[3,-1],[1,-3],[2,-7],[0,-4],[1,-2],[2,-2],[0,-4],[1,-2],[1,-1],[1,-2],[0,-1]],[[5921,6236],[-3,4],[-1,1]],[[5917,6241],[2,0],[0,2],[0,3],[0,1],[2,-1],[1,1],[0,1],[0,1],[0,1],[0,1],[0,1],[1,0],[5,-7],[1,-4],[0,-4],[0,-1],[0,-1],[2,-2],[0,-2],[-1,-3],[-9,14],[-1,-2],[1,0],[0,-1],[0,-1],[0,-2]],[[5900,6372],[-1,-2],[0,3],[-1,0],[0,1],[0,2],[0,3],[0,1],[1,-1],[0,-1],[1,-3],[0,-3]],[[5955,6924],[-1,-3],[-9,3],[-2,3],[0,1],[0,3],[1,2],[1,1],[2,3],[3,1],[5,-14]],[[5913,6244],[-1,0],[-7,-1],[-1,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-3],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[-1,0],[0,-1],[0,-1],[-4,-4],[-1,-2],[0,-1],[0,-1],[0,-8],[0,-3],[0,-1],[0,-1],[0,-5],[0,-2],[0,-2],[0,-1],[-1,-1],[-11,-1],[-1,1],[-1,2],[0,1],[0,1],[0,2],[-1,3],[0,2],[0,2],[0,3],[0,1],[0,4],[0,3],[0,2],[0,1],[0,4],[0,1],[0,1],[0,1],[0,1],[-1,0],[-3,9],[-1,5],[-1,5],[0,8],[0,2],[0,4],[0,1],[0,1],[-1,1],[0,2],[-12,5],[-1,2],[-47,-14],[-1,0]],[[5815,6254],[-1,13],[-7,39],[-1,17],[2,20],[9,60],[1,6],[1,1],[0,2],[0,4],[0,3],[11,51],[1,5],[1,11],[0,3],[0,16],[0,3],[0,3],[1,3],[-8,26],[-2,8],[-5,34],[-1,7],[-2,16],[-1,4],[-1,3],[-8,11],[-8,11],[-1,2],[0,6],[-5,63],[-1,7],[0,6],[0,4],[0,3],[7,45],[1,8],[1,19],[-2,7],[-2,2],[-5,-2],[-1,1],[0,3],[0,4],[1,1],[7,16],[3,4],[4,11],[0,2],[1,0],[2,0],[1,3],[2,5],[1,7],[1,5],[0,3],[0,9],[1,3],[0,3],[0,3],[0,2],[0,2],[1,2],[0,2],[0,7],[0,3],[1,6],[2,3],[5,1],[1,2],[2,7],[1,1],[4,0],[1,2],[1,2],[1,5],[3,12],[1,3],[0,2],[0,4],[0,3],[0,4],[0,1],[1,2],[1,2],[0,3],[0,5],[0,5],[0,5],[0,2],[-2,2],[-1,2],[0,2],[1,2],[1,0],[1,-1],[2,-4],[5,-19],[1,-2],[7,-3],[2,1],[2,4],[1,9],[0,4],[0,19],[-1,4],[-1,3],[0,2],[0,2],[0,5],[0,2],[-1,5],[0,9],[5,-7],[3,0],[1,-1],[1,-3],[0,-1],[1,0],[0,2],[1,0],[0,-1],[1,-2],[1,-5],[1,-1],[1,-1],[0,-2],[0,-5],[0,-2],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[-1,-3],[-1,-6],[1,0],[0,3],[3,16],[1,2],[1,-2],[0,1],[1,1],[0,1],[7,-4],[0,5],[-1,2],[-1,3],[-1,2],[0,2],[1,7],[-1,4],[0,5],[0,5],[1,4],[0,2],[3,0],[1,-1],[1,-5],[1,-3],[2,-8],[1,-2],[0,1],[0,1],[0,1],[1,1],[0,1],[0,1],[0,2],[0,2],[-1,6],[-2,6],[0,2],[0,1],[0,4],[0,2],[0,1],[-2,5],[0,2],[0,4],[1,4],[0,1],[1,0],[2,-2],[5,-20],[1,-1],[2,1],[1,-2],[3,-5],[1,-5],[-1,-3],[0,-2],[1,-2],[3,-5],[3,2],[5,-3],[2,-5],[1,-5],[1,0],[0,2],[1,0],[0,-1],[1,-4],[-1,-9],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-4],[-1,-4],[-4,-8],[-11,12],[-2,4],[-1,-2],[-5,3],[-1,0],[0,2],[0,1],[0,2],[0,1],[0,1],[0,3],[0,3],[0,2],[0,3],[0,4],[-1,2],[0,1],[-1,-1],[0,-1],[0,-3],[1,-3],[0,-4],[-1,-8],[-1,-4],[-3,-1],[1,-2],[1,-1],[1,-1],[0,-4],[0,-4],[-1,-3],[0,-3],[0,-4],[1,7],[2,1],[2,-1],[1,-4],[1,-1],[3,3],[1,-1],[0,-2],[0,-2],[0,-1],[-1,-4],[0,-2],[1,0],[2,-5],[0,-3],[-1,0],[-1,0],[0,-2],[-1,-5],[-1,-1],[-1,-2],[-2,-6],[-1,-1],[1,-3],[1,2],[3,8],[1,0],[0,1],[0,1],[1,5],[0,1],[1,-1],[1,0],[0,1],[0,3],[0,1],[1,-1],[1,-5],[0,-1],[1,4],[1,1],[3,-3],[2,0],[1,0],[0,-2],[-2,-6],[-1,-4],[0,-7],[3,12],[3,3],[2,0],[-6,-27],[0,-8],[-1,-3],[0,-4],[0,-1],[0,1],[1,2],[1,0],[0,1],[0,2],[1,6],[1,7],[0,3],[1,1],[8,12],[1,2],[1,-3],[0,-1],[1,1],[1,2],[1,0],[1,-3],[1,-1],[-1,-6],[-1,-3],[-1,-1],[0,-4],[2,2],[-1,-6],[-2,-7],[0,-5],[-1,-3],[-3,-1],[0,-5],[1,1],[0,1],[1,1],[2,2],[2,-1],[0,-4],[-2,-5],[-2,-1],[0,-1],[-1,-3],[2,1],[1,-1],[0,-5],[1,0],[1,3],[1,1],[1,-4],[-1,-6],[-3,-9],[-2,-7],[-1,-2],[-1,-1],[-2,-3],[-2,-1],[-1,-2],[-1,-7],[0,-4],[-1,-8],[0,-8],[1,-5],[0,6],[0,14],[1,5],[1,5],[3,0],[3,5],[2,0],[1,2],[1,6],[0,3],[0,3],[3,7],[0,2],[0,2],[0,4],[0,7],[0,1],[0,2],[0,2],[0,1],[0,1],[0,4],[0,1],[0,3],[1,4],[0,2],[3,2],[1,-1],[0,-2],[0,-2],[-1,-2],[0,-1],[1,1],[1,2],[1,-1],[0,-1],[1,0],[0,1],[0,1],[0,2],[0,4],[8,-4],[8,-3],[-1,-5],[1,-2],[3,-4],[1,1],[1,0],[1,-5],[9,-5],[0,-2],[0,-4],[-1,-3],[1,0],[0,2],[1,3],[0,2],[2,1],[1,-1],[1,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[1,0],[0,2],[1,1],[0,3],[0,1],[1,1],[-1,5],[-1,6],[-1,3],[1,2],[2,-2],[2,1],[13,-19],[2,-1],[1,-3],[1,-1],[1,-3],[1,-3],[1,-4],[0,-2],[1,1],[4,-5],[1,-3],[0,-5],[1,-6],[6,-6],[5,-12],[1,-1],[0,-2],[0,-2],[0,-1],[1,-1],[0,-1],[2,-1],[5,-13],[0,-3],[0,-2],[2,-3],[2,-2],[1,-2],[0,-3],[1,-3],[0,-3],[0,-1],[2,-1],[1,-2],[1,-2],[1,-6],[0,-1],[1,-3],[1,-1],[1,-1],[1,0],[0,-3],[3,-2],[1,-2],[0,-5],[3,-6],[3,-15],[2,-7],[1,-4],[0,-3],[0,-2],[1,-5],[0,-1],[0,-2],[2,-1],[1,-1],[1,-3],[0,-1],[0,-2],[0,-1],[0,-1],[3,-4],[0,-1],[1,-6],[1,-4],[0,-14],[1,-2],[0,1],[0,2],[0,2],[0,1],[1,-1],[0,-2],[0,-1],[1,0],[6,-13],[1,0],[-1,5],[-3,5],[-2,6],[-1,2],[0,3],[1,0],[1,-4],[1,1],[0,4],[0,2],[1,-1],[0,-1],[3,-11],[1,-2],[3,-2],[1,-2],[0,-3],[0,-3],[0,-2],[0,-3],[3,-4],[6,-20],[3,-5],[4,-18],[3,-3],[2,-1],[1,-3],[2,1],[1,-1],[0,-1],[1,1],[0,2],[0,1],[-1,3],[-1,12],[-1,2],[-1,7],[-1,3],[1,1],[1,-1],[1,-1],[2,-9],[1,-1],[0,-3],[0,-3],[1,-4],[0,-4],[1,-10],[1,-2],[0,-1],[1,0],[0,-1],[1,-2],[1,-8],[1,-3],[4,-11],[0,-1],[0,-1],[0,-1],[1,2],[0,-1],[1,0],[1,-7],[0,-3],[0,-3],[-1,-1],[-2,1],[0,-2],[2,-11],[2,-9],[1,-3],[1,1],[1,7],[0,2],[0,1],[0,1],[0,2],[0,1],[1,0],[1,0],[1,1],[0,-1],[0,-4],[1,-3],[0,-3],[1,-2],[2,-8],[1,-1],[2,2],[1,-2],[2,-7],[0,-5],[1,-6],[0,-8],[-1,-5],[-1,-4],[0,-5],[0,-2],[0,-3],[1,-2],[0,-1],[0,-2],[-1,-1],[0,-1],[0,-6],[0,-3],[0,-3],[2,0],[1,-1],[1,-4],[0,-5],[0,-3],[0,-5],[0,-2],[0,-12],[0,-6],[0,-6],[-1,-8],[1,-5],[3,-5],[1,2],[0,2],[1,1],[0,-1],[1,-1],[0,-3],[0,-2],[0,-4],[0,-4],[0,-3],[0,-3],[1,-3],[0,-3],[0,-3],[-1,-2],[0,-2],[0,-2],[0,-10],[0,-4],[0,-2],[0,-2],[0,-3],[-1,-1],[0,1],[-1,0],[0,-3],[0,-2],[0,-2],[1,-3],[0,-3],[-1,-11],[-3,-23],[-1,-5],[-3,-11],[-3,-18],[-2,-5],[0,-5],[-1,-2],[0,-2],[-1,-2],[-1,-2],[0,-4],[-1,-3],[0,-4],[0,-1],[0,-2],[0,-3],[-1,-4],[-1,-2],[-1,-2],[0,-7],[-1,-3],[-1,-4],[-2,-9],[-1,-2],[-3,-3],[-4,-13],[-2,-10],[-7,-11],[-1,-4],[-1,-1],[-2,-4],[-6,-7],[-3,-9],[-1,-2],[-7,-8],[-3,2],[-1,-1],[0,-1],[-11,-12],[-8,4],[-9,4],[-3,6],[-5,3],[-3,5],[-3,12],[-2,3],[-2,6],[-11,14],[-2,-1],[-2,3],[-3,-2],[-3,3],[-3,-1],[-4,7],[-7,4],[-7,5],[-3,7],[-3,0],[-5,7],[-2,10],[-2,3],[-1,3],[-8,29],[-3,4],[-2,-5],[-2,-8],[-2,-5],[-1,-1],[-2,0],[-1,2],[-1,11],[-1,5],[0,5],[1,3],[1,1],[1,-3],[1,1],[0,2],[0,2],[0,1],[-1,0],[1,-2],[-1,-1],[-5,6],[-2,0],[-1,1],[0,2],[0,1],[0,-1],[-1,-3],[-1,-1],[-1,1],[2,6],[0,5],[0,4],[0,2],[0,3],[0,2],[0,2],[0,1],[0,-1],[0,-2],[-1,-3],[0,-5],[-1,-4],[0,-3],[-1,-2],[0,2],[-1,-1],[0,-2],[-2,-1],[0,1],[0,1],[-1,4],[0,1],[-1,0],[0,-1],[0,-2],[0,-2],[-1,1],[0,4],[-1,9],[-1,4],[-1,5],[-1,3],[0,2],[-1,-1],[0,1],[-1,2],[0,4],[-1,-3],[0,-2],[1,-5],[-1,4],[-1,2],[0,4],[0,-7],[0,-5],[2,-8],[0,1],[-2,5],[0,2],[0,-3],[0,-2],[1,-2],[0,-1],[0,-2],[-1,0],[-1,5],[0,2],[0,-4],[1,-4],[1,-2],[-2,-1],[0,1],[-11,32],[-7,29],[-1,3],[0,5],[-1,5],[1,5],[0,3],[1,3],[2,1],[1,3],[-1,2],[-3,-2],[-1,0],[-1,2],[-3,8],[-5,1],[-1,1],[-1,4],[0,1],[-2,0],[0,-1],[-1,-2],[-1,-1],[-1,2],[-4,-3],[-1,1],[-2,5],[-1,1],[-1,-1],[-1,-3],[2,0],[0,-2],[0,-5],[5,-1],[3,4],[1,-1],[1,-2],[1,-6],[1,-4],[0,-3],[0,1],[-1,1],[-1,0],[1,-5],[1,-1],[1,1],[1,-1],[2,-6],[1,-4],[1,-3],[0,-3],[0,-3],[-1,-5],[-1,-2],[-2,2],[-1,-1],[1,-2],[1,0],[1,-3],[0,-1],[0,-5],[-1,-2],[-1,-2],[0,-2],[3,0],[2,-6],[1,-4],[1,-4],[0,-4],[1,-4],[5,-3],[0,-2],[1,-1],[0,-5],[0,-2],[0,-1],[-3,0],[-1,1],[5,-12],[6,0],[4,-9],[0,-3],[0,-4],[-1,1],[-2,3],[-2,2],[-1,0],[0,-5],[0,-5],[-1,-3],[-1,-2],[-1,0],[-1,2],[-1,1],[-2,-1],[1,-3]],[[6022,9577],[-1,-1],[-2,2],[-2,0],[0,1],[-1,2],[2,4],[5,-6],[-1,-2]],[[5964,4404],[0,-9],[1,-2],[1,-3],[1,2],[1,0],[0,-2],[0,-1],[0,-1],[0,-5],[0,-1],[-1,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[3,-5],[0,-1],[1,-4],[0,-1],[1,1],[0,-1],[0,-1],[0,-2],[0,-1],[1,-2],[1,1],[1,4],[1,0],[0,-1],[0,-1],[0,-1],[0,-6],[0,-1],[0,-2],[-1,-9],[0,-4],[0,-1],[0,-3],[0,-1],[1,0],[0,1],[1,2],[0,1],[1,-1],[0,-1],[0,-1],[0,-2],[-1,-1],[1,-2],[0,-3],[0,-1],[1,-4],[0,-1],[1,0],[0,7],[0,3],[1,4],[1,3],[0,2],[1,1],[1,0],[2,-1],[0,-1],[0,-1],[1,0],[1,0],[1,0],[0,-3],[0,-1],[2,0],[0,2],[0,2],[1,1],[0,-1],[0,-1],[2,-8],[0,-1],[0,-1],[0,-1],[0,-1],[0,1],[-1,0],[0,-1],[0,-2],[0,-1],[0,-3],[0,-1],[1,-1],[1,0],[1,2],[2,0],[0,-1],[1,-1],[0,-3],[0,-1],[0,-1],[1,-1],[1,-4],[1,0],[1,-1],[0,-1],[0,-2],[0,-2],[0,-4],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[1,0],[0,-1],[0,1],[2,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-3],[0,-1],[0,-1],[0,-1],[0,-2],[-1,-4],[0,-1],[0,-1],[0,-2]],[[6004,4262],[0,-4],[-1,-1],[0,-1],[0,2],[-1,0],[-3,-8],[-1,-3],[0,-1],[0,-1],[0,-2],[0,-1],[0,-3],[0,-1],[0,-1],[-3,-1],[-1,2],[-1,1],[-5,4],[-1,0],[-1,-4],[-2,-3],[-1,0],[0,3],[-1,-1],[0,-3],[-1,-6],[-1,-1],[-1,0],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-3],[0,-2],[0,-2],[0,-1],[0,-4],[0,-1],[0,-2],[0,-3],[0,-1],[0,-1],[-2,-5],[-1,-4],[-2,-2],[-1,-3],[0,-1],[0,-1],[0,1],[-1,0],[0,1],[0,2],[0,1],[-1,1],[0,-1],[0,1],[0,1],[-1,1],[0,1],[0,1],[-3,-2],[-1,3],[-1,0],[-1,-4],[-1,-2],[0,-2],[0,-3],[0,-1],[1,-5],[0,-1],[-1,-3],[0,-1],[0,-2],[0,-1],[0,-4],[1,-5],[0,-2],[0,-2],[-1,-1],[0,-1],[-1,0],[0,1],[-2,5],[0,2],[-2,0],[0,1],[0,1],[0,2],[0,3],[0,1],[-1,1],[-9,15],[0,-1],[-2,-3],[0,-1],[-1,0],[0,1],[0,1],[0,3],[0,1],[-1,1],[0,1],[0,-1],[-1,-8],[-1,-3],[-1,-5],[-1,-2],[-2,0],[-2,3],[-1,-1],[-1,-2],[0,-4],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[-1,0],[-1,-6],[-1,-5],[0,-2],[1,-2],[0,-1],[1,0],[1,1],[0,-1],[0,-4],[1,-2],[0,-1],[0,-1],[0,-4],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-4],[1,-2],[0,-2],[0,-1],[-1,-2],[0,-2],[0,-5],[-1,-1],[0,-1],[-1,-1],[-1,-2],[0,-2],[0,-1],[0,-9],[0,-2],[0,-2],[0,-2],[-1,-2],[0,-2],[1,-5],[-1,-5],[0,-2],[-1,-6],[0,-2],[-1,0],[-1,0],[-1,-1],[-2,-7],[0,-3],[0,-2],[0,-4],[0,-1],[0,-2],[0,-1],[0,-1],[0,-8],[0,-1],[-1,-1],[-1,0],[0,-1],[0,-1],[0,-1],[0,-1],[0,-9],[0,-1],[0,-3],[1,-10],[0,-2],[0,-2],[-1,-1],[0,1],[-1,3],[0,2],[-3,4],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[-1,0],[-2,3],[0,2],[-1,0],[-1,-1],[-2,2],[0,1],[0,-1],[-1,-5],[-1,-3],[0,-2],[-1,-1],[-1,-3],[0,-2],[0,-2],[0,-1],[0,-2],[0,-2],[0,-5],[0,-2],[-1,-1],[-1,-4],[0,-1],[-1,-1],[-1,-1],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-1],[0,1],[-1,-1],[0,-2],[-1,-7],[-1,-1],[-1,1],[0,1],[-1,2],[0,2],[0,1],[0,1],[0,1],[-1,1],[-1,-1],[0,-1],[0,-1],[0,-1],[1,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-7],[0,-1],[0,-1],[-1,-1],[-1,-2],[-1,1],[-1,-1],[0,-1],[0,-1],[-1,-1],[0,-2],[0,-2],[0,-1],[-1,0],[0,-1],[-2,-6],[-1,-2],[0,-1],[0,-2],[0,-1],[0,-3],[0,-1],[-1,0],[0,1],[0,5],[-1,1],[0,1],[-2,-5],[-2,2],[0,1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-3],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[0,-3],[0,-2],[0,-1],[0,-1],[0,-1],[-1,0],[0,-2],[0,-3],[-1,-2],[-1,-3],[0,-1],[0,-4],[-1,-1],[0,1],[-1,1],[-1,-2],[-1,0],[-1,2],[0,1],[-1,2],[0,2],[-1,1],[-1,-1],[-4,-8],[-1,0],[0,1],[0,3],[0,1],[-2,3]],[[5861,3878],[-3,-1],[-1,-1],[0,-3],[-1,-2],[0,2],[-1,2],[-3,2],[-1,1],[0,1],[0,1],[0,2],[0,3],[0,2],[0,5],[0,4],[0,3],[0,1],[0,2],[0,1],[0,9],[0,3],[0,3],[0,1],[0,2],[0,1],[0,1],[0,1],[0,2],[0,4],[0,2],[0,1],[0,1],[0,1],[0,1],[1,1],[1,1],[0,1],[0,1],[0,1],[0,1],[0,1],[1,1],[0,1],[0,1],[0,4],[-7,15],[-1,5],[0,1],[-1,4],[0,2],[1,3],[0,2],[1,5],[-1,1],[0,6],[0,6],[0,1],[0,3],[0,1],[0,2],[2,11],[2,21],[-3,8],[0,3],[-1,4],[0,3],[0,4],[1,1],[0,1],[0,2],[0,3],[0,1],[0,1],[0,1],[0,1],[-2,4],[-1,0],[0,2],[0,1],[1,1],[2,3],[0,1],[0,1],[0,2],[0,3],[1,2],[0,3],[0,1],[0,2],[0,1],[0,2],[0,1],[0,1],[-1,2],[-10,6],[0,3],[-2,4],[-4,1],[0,1],[0,1],[0,1],[-1,1],[0,1],[0,1],[0,2],[3,10],[0,1],[0,1],[0,1],[0,1],[0,1],[-1,0],[0,-2],[-1,-1],[0,-1],[0,1],[-1,1],[0,2],[-1,4],[0,1],[-1,6],[-1,2],[-1,0],[0,-1],[-1,-3],[0,-1],[-1,0],[0,1],[0,1],[0,5],[0,17],[0,3],[-1,7],[0,4],[0,1],[0,2],[0,1],[1,6],[2,9],[1,2],[0,3],[0,3],[1,3],[2,8],[1,3],[0,2],[0,5],[1,6]],[[5832,4242],[1,1],[1,-1],[1,6],[0,2],[1,1],[0,2],[0,1],[0,1],[0,1],[0,1],[0,2],[0,2],[0,1],[1,1],[0,1],[0,1],[0,1],[0,1],[0,4],[-1,5],[0,5],[0,1],[-1,1],[0,2],[0,1],[0,2],[0,2],[0,2],[0,4],[0,1],[1,1],[0,5],[0,1],[0,1],[0,3],[0,2],[0,2],[0,1],[0,1],[1,6],[0,1],[1,0],[1,-3],[1,1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,-1],[0,-1],[1,-4],[0,-1],[1,0],[1,2],[1,2],[1,2],[1,1],[1,0],[3,2],[0,1],[1,1],[0,1],[0,8],[0,3],[0,1],[0,2],[0,2],[0,1],[1,7],[4,20],[1,9],[1,1],[0,1],[0,3],[0,1],[0,3],[0,1],[0,1],[0,2],[0,8],[1,4],[0,1],[1,1],[1,0],[0,-2],[0,-1],[1,1],[1,0],[0,-1],[0,-1],[0,-2],[0,-1],[1,-8],[2,-10],[1,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,3],[1,3],[1,4],[0,2],[0,1],[1,2],[1,3],[0,2],[0,2],[0,2],[1,5],[0,2],[0,1],[0,2],[0,1],[0,2],[0,3],[0,4],[1,2],[0,1],[0,1],[1,1],[0,1],[0,2],[0,1],[0,2],[0,1],[0,1],[-1,1],[0,2],[0,1],[0,1],[0,2],[0,1],[0,3],[1,1],[0,2],[0,2],[0,1],[0,2],[-1,5],[0,2],[0,1],[0,2],[1,1],[0,1],[1,-1],[0,-1],[2,-4],[1,-1],[1,-2],[0,-4],[2,-1],[0,1],[1,1],[0,3],[0,1],[1,10],[0,1],[1,3],[1,1],[0,-3],[0,-1],[1,-1],[0,-1],[1,0],[1,2],[1,1],[0,2],[1,0],[0,-1],[0,-2],[0,-3],[1,-3],[0,-1],[1,-4],[0,-1],[0,-1],[0,-1],[0,-1],[1,-4],[0,-1],[0,-1],[0,-1],[1,-1],[0,-2],[0,-2],[1,-1],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-3],[0,-1],[0,-2],[1,-1],[0,-1],[0,1],[0,1],[1,1],[2,-1],[0,-1],[1,-2],[1,0],[1,-2],[2,-1],[0,-2],[1,-1],[0,-2],[0,-4],[0,-5],[1,2],[1,2],[1,11],[0,4],[1,2],[0,2],[-1,0],[0,2],[0,1],[0,1],[0,5],[0,1],[0,2],[0,1],[0,1],[1,7],[2,11],[0,1],[1,1],[0,1],[1,5],[0,2],[1,1],[0,3],[0,2],[0,2],[0,1],[0,1],[0,1],[1,-1],[1,-1],[1,-9],[0,-1],[0,-1],[1,0],[2,-3],[1,-6],[1,-1],[0,-3],[0,-1],[2,-1],[1,0],[0,1],[0,2],[0,4],[0,3],[0,2],[0,1],[0,1],[0,1],[5,-2],[0,-1],[1,-3],[1,-3],[0,-4],[1,-3],[0,-1],[0,-1],[1,0],[1,-1],[0,-1],[0,-1],[1,-5],[0,-3],[1,-3],[1,-4],[0,-1],[0,-1],[0,-2],[0,-2],[1,-3],[0,1],[0,1],[1,1],[0,1],[0,-1],[0,-1],[1,-2],[0,-1],[0,-2],[0,-1],[0,-3],[1,-2],[0,-1],[1,-2],[1,-1],[1,1],[1,-1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,2],[1,0],[2,-1],[0,1],[0,5],[1,1],[0,-1],[0,-1],[1,-1],[2,-2],[1,-2],[0,-1],[0,-3],[0,-1],[1,-1],[2,-2],[0,-1],[0,-2],[1,-1],[0,-1],[0,-2],[1,0],[1,2],[3,-3]],[[5861,3878],[-1,-4],[0,-4],[0,-1],[0,-3],[0,-1],[0,-1],[-1,-2],[-1,0],[0,-1],[0,-1],[0,-1],[0,-4],[0,-2],[0,-3],[0,-2],[0,-3],[0,-4],[-1,-2],[0,-1],[0,2],[-1,0],[-1,-2],[0,-2],[0,-2],[-1,-2],[0,-1],[0,-1],[1,-1],[0,-1],[1,-1],[0,-1],[0,-1],[-1,-3],[0,-1],[0,-1],[0,-1],[0,-1],[1,-4],[1,-9],[0,-3],[0,-1],[1,0],[0,1],[0,2],[1,3],[0,1],[0,1],[0,-1],[1,-1],[0,-1],[2,3],[0,-1],[0,-1],[0,-4],[1,-1],[0,-4],[0,-1],[0,-2],[0,-1],[-1,-6],[-1,-20],[0,-2],[0,-4],[0,-1],[0,-1],[0,-1],[0,-1],[1,-1],[2,-4],[1,-1],[1,0],[0,-1],[0,-1],[1,-2],[0,-1],[0,-3],[0,-4],[0,-5],[0,-7],[0,-1],[0,-1],[0,-1],[1,0],[0,2],[1,0],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[1,-2],[0,-1],[4,-2],[0,-2],[0,-1],[0,-3],[0,-17],[0,-3],[0,-2],[0,-11],[0,-3],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-1],[0,-1],[1,-1],[0,-2],[0,-2],[-1,-1],[0,-2],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[-1,-3],[0,-1],[-1,-4],[-1,-1],[0,-1],[0,-3],[1,-4],[1,-11],[0,-12],[0,-3],[0,-6],[1,-12],[1,-7]],[[5874,3574],[-1,-1],[0,-1],[-1,0],[0,1],[0,1],[0,1],[0,1],[-2,-1],[0,-1],[0,-1],[-1,0],[0,2],[0,1],[-1,0],[0,-1],[-1,-1],[-1,-2],[0,-1],[0,1],[0,1],[0,1],[0,1],[0,1],[-4,8],[0,-7],[0,-8],[-1,-2],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-2],[0,-2],[-1,-9],[0,-6]],[[5858,3537],[-1,6],[-1,2],[-1,-2],[-1,-3],[-1,1],[0,8],[-1,4],[0,2],[0,1],[-1,0],[-1,1],[-1,2],[0,3],[0,2],[0,2],[0,2],[0,4],[0,1],[0,1],[-3,6],[0,2],[0,3],[0,2],[0,2],[0,1],[-6,9],[0,2],[-1,4],[-1,1],[0,-2],[-2,-6],[0,-1],[-4,6],[-1,1],[-1,-2],[-1,-3],[-1,-4],[-1,-10],[0,-2],[-2,2],[-1,-1],[-3,-8],[-2,-12],[-1,-2],[0,2],[-2,10],[-1,6],[0,3],[0,5],[0,4],[0,9],[0,2],[1,6],[0,2],[0,2],[0,1],[0,1],[1,1],[0,2],[0,2],[0,2],[-1,4],[0,2],[-1,2],[-4,5],[-1,4],[-1,5],[-2,3],[-1,0],[-2,-3],[-1,-5],[-2,-4],[-1,-4],[0,-5],[0,-1],[-2,3],[-1,3],[0,5],[-1,6],[0,6],[0,2],[-1,7],[-1,4],[0,2],[0,1],[-2,1],[-4,-2],[-1,-3],[-1,-7],[0,-2],[-1,1],[0,3],[-1,6],[-2,11],[-1,4],[1,5],[0,5],[0,6],[1,6],[0,2],[0,2],[0,2],[0,3],[0,3],[-1,17],[0,5],[0,5],[0,9],[0,3],[0,2],[-2,9],[0,2],[0,2],[0,3],[1,3],[0,3],[0,3],[-1,1],[-2,6],[0,2],[-1,4],[0,6],[0,11],[0,4],[-3,9],[-1,4],[1,5],[0,5],[1,6],[0,4],[-1,2],[-1,6],[0,2],[-2,2],[-1,-2],[-2,-6],[-1,1],[0,9],[2,19],[0,3],[0,5],[1,2],[0,3],[0,5],[0,3],[0,6],[-1,3],[0,2],[-1,4],[1,4],[2,6],[1,4],[1,5],[0,8],[0,13],[0,6],[-1,4],[0,2],[-3,8],[0,2],[-2,8],[-3,7],[0,4],[0,4],[0,5],[0,6],[0,9],[-5,-1],[0,3],[0,4],[0,4],[0,3],[0,2],[0,1],[1,1],[0,1],[0,1],[0,5],[0,1],[0,1],[0,3],[0,1],[0,1],[0,3],[0,2],[0,1],[1,2],[2,4],[1,2],[0,4],[-1,7],[0,4],[1,2],[0,1],[1,2],[-1,3],[0,1],[1,1],[4,3],[2,3],[0,2],[1,3],[-1,5],[0,3],[-3,5],[0,2],[0,3],[0,3],[0,3],[0,3],[0,6],[0,2],[-1,1],[0,7],[0,6],[0,7],[-1,10],[-2,11],[0,5],[-1,7],[0,12],[0,12],[2,19],[-2,34],[-1,31],[-1,12],[0,11],[1,16],[1,14],[3,14],[3,11]],[[5769,4373],[1,0],[4,9],[1,-4],[0,-1],[1,1],[0,1],[3,-4],[1,1],[0,1],[1,0],[0,-1],[0,-1],[1,-4],[0,-4],[0,-9],[1,-1],[0,-4],[1,-6],[1,-4],[0,-1],[1,2],[3,0],[3,1],[1,0],[5,-9],[2,0],[3,-2],[1,-2],[0,-1],[0,-1],[0,-4],[0,-1],[0,-1],[1,-1],[1,-1],[0,-1],[0,-2],[1,-3],[1,-2],[0,-2],[3,-6],[1,-1],[0,-2],[0,-5],[1,-2],[0,-1],[0,-3],[0,-1],[0,-1],[0,-1],[-1,0],[0,-1],[0,-1],[0,-1],[0,-1],[5,-6],[1,-1],[0,-1],[0,-2],[0,-4],[0,-2],[0,-2],[1,-1],[0,-1],[1,0],[1,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[1,-1],[0,-1],[1,-3],[1,-2],[1,2],[1,0],[2,-4],[2,2],[3,-1]],[[5755,4584],[-1,1],[0,2],[0,2],[0,2],[1,0],[0,-2],[1,-4],[-1,-1]],[[5774,4623],[-2,-3],[0,2],[0,2],[0,3],[0,2],[0,3],[1,-1],[1,-3],[0,-1],[0,-2],[0,-2]],[[5787,4634],[0,-1],[1,0],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[0,1],[-1,2],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1]],[[5750,4629],[0,-1],[-1,0],[0,1],[0,2],[0,4],[-1,1],[0,3],[0,8],[0,2],[1,-3],[0,-3],[0,-2],[1,-2],[0,-3],[0,-1],[0,-1],[0,-3],[0,-2]],[[5793,4710],[1,-2],[0,-1],[0,-1],[2,-8],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[-1,0],[-1,0],[0,1],[0,1],[-2,5],[0,3],[-1,3],[0,3],[0,2],[0,1],[0,2],[1,-1],[0,1]],[[5792,4701],[-1,0],[0,2],[-1,3],[0,1],[0,1],[-1,1],[0,4],[0,1],[1,1],[0,-1],[1,-1],[0,-2],[0,-1],[0,-2],[1,-1],[0,-1],[0,-2],[0,-3]],[[5791,4729],[0,-7],[1,-4],[0,-1],[0,-1],[0,1],[0,1],[-1,1],[1,-4],[0,-1],[0,-2],[0,-1],[0,1],[-1,0],[0,2],[0,1],[0,1],[0,2],[0,3],[0,2],[0,2],[0,4],[-1,0],[0,1],[0,4],[0,1],[-1,0],[1,1],[0,3],[0,-2],[1,-4],[0,-2],[0,-2]],[[5773,4746],[0,-2],[0,-2],[-1,-2],[0,2],[0,1],[0,2],[0,-1],[0,-1],[-1,2],[0,1],[0,1],[0,1],[1,0],[0,-1],[1,-1]],[[5797,4740],[-1,0],[-2,7],[-1,2],[0,1],[0,3],[0,2],[1,-3],[1,-2],[0,-2],[1,-6],[1,-1],[0,-1]],[[5795,4772],[-1,-6],[-1,1],[0,4],[0,4],[0,4],[0,3],[1,1],[0,-1],[1,-3],[0,-3],[0,-4]],[[5793,4777],[-1,0],[0,-2],[0,1],[0,2],[0,1],[0,2],[-1,2],[1,1],[0,-3],[0,-1],[0,-1],[1,-2]],[[5988,4917],[1,-2],[0,-1],[2,-6],[0,-3],[0,-3],[0,-2],[-1,-3],[0,-3],[-1,-2],[0,-2],[0,-4],[0,-2],[-2,-11],[-1,-7],[-1,0],[0,-3],[0,-3],[0,-2],[1,-6],[0,1],[-1,-13],[-1,-3],[0,-1],[-5,-3],[-1,-4],[0,-1],[0,-1],[-1,-4],[0,-2],[-1,-3],[0,-1],[0,-2],[0,-2],[0,-2],[0,-2],[0,-1],[1,-4],[0,-1],[0,-2],[0,-4],[0,-2],[1,-1],[0,-2],[0,-2],[0,-4],[1,-14],[0,-4],[0,-25],[0,-6],[0,-3],[-1,-16],[0,-7],[0,-3],[0,-2],[0,-2],[-1,-4],[0,-5],[-1,-5],[0,-9],[0,-5],[0,-14],[0,-3],[0,-2],[0,-3],[0,-1],[0,-1],[1,-2],[0,-1],[0,-2],[-1,-2],[0,-1],[0,-2],[0,-1],[-2,-5],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[2,-1],[1,-2],[0,-1],[1,-1],[0,-1],[0,-1],[2,0],[0,-1],[0,-1],[0,-1],[1,0],[2,0],[0,-2],[1,-5],[1,-5],[0,-1],[-1,-1],[-1,-2],[0,-1],[0,-2],[-1,-2],[0,-2],[0,-1],[0,-3],[0,-4],[0,-13],[1,-16],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,-1],[1,-1],[1,1],[1,2],[2,2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-5],[0,-2],[0,-3],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[-1,0],[-1,2],[0,2],[-1,2],[0,1],[0,-1],[-1,0],[0,-3],[1,-4],[0,-1],[-1,-15],[0,-2],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-2],[1,-1],[0,-1],[0,-1],[1,0],[0,-1],[0,-1],[-1,-1],[0,-2],[0,-1],[-1,0],[-3,6],[-1,0],[0,-1],[0,-1],[0,-1],[0,-2],[2,-4],[0,-1],[0,-1],[1,-2],[0,-2],[0,-1],[0,-1],[1,-2],[0,-2],[0,-3],[-1,-7],[0,-3],[0,-1],[-2,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-2],[0,-1],[1,-1],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,0],[-1,1],[-2,-2],[0,-2],[-1,-4],[0,-1],[-2,-3],[-1,-1],[-1,1],[0,-1],[-3,-6],[0,1],[0,1],[0,3],[0,1],[-1,-1],[0,-2],[-1,-7],[0,-7],[-1,-3],[-1,-6],[0,-1],[0,-2],[0,-1],[0,-4],[0,-2],[-1,1],[-1,-1]],[[5769,4373],[0,3],[1,6],[1,10],[0,5],[2,21],[1,19],[0,5],[0,4],[1,4],[1,4],[3,0],[1,3],[-1,0],[-2,4],[1,2],[2,0],[2,5],[0,-2],[0,5],[0,1],[-1,3],[0,2],[0,2],[-1,7],[0,3],[-2,3],[0,5],[1,11],[0,13],[0,11],[-1,9],[-1,5],[0,5],[0,3],[0,7],[0,4],[2,8],[1,2],[1,-2],[1,-4],[0,-4],[1,-6],[0,-6],[0,-5],[4,-5],[1,-1],[0,4],[1,7],[0,6],[0,14],[0,5],[1,6],[0,4],[1,2],[1,0],[1,-2],[0,-4],[1,-2],[1,-2],[1,-1],[0,-1],[0,-1],[0,-3],[1,-1],[0,-1],[1,0],[1,3],[1,-2],[0,-1],[1,1],[1,6],[0,1],[1,-1],[1,3],[0,1],[1,6],[1,8],[0,3],[-2,7],[1,0],[1,3],[1,9],[1,4],[1,2],[3,-1],[7,-7],[5,-5]],[[5824,4612],[0,-20],[1,-4],[2,-7],[1,0],[1,1],[0,-10],[2,-4],[1,-5],[1,-7],[3,-14],[3,-9],[0,-3],[1,-2],[3,-1],[3,4],[1,3],[1,13],[1,4],[2,-2],[3,8],[0,13],[-5,13],[-2,8],[0,17],[0,9],[0,6],[-4,10],[-1,7],[1,8],[1,7],[-2,4],[-2,0],[-2,2],[-15,29],[-2,-1],[-2,-5],[-2,-8],[0,-7],[0,-3]],[[5817,4666],[-5,2],[-4,0],[-2,2],[-1,3],[0,5],[-1,11],[-1,4],[-1,2],[-1,2],[0,5],[-1,6],[0,5],[-1,2],[0,-2],[0,-3],[0,-2],[-1,0],[-1,1],[0,2],[-1,2],[-1,4],[-1,1],[0,2],[-2,25],[-2,12],[-1,5],[1,1],[0,1],[1,-1],[0,-1],[0,-2],[1,-5],[0,-2],[0,-3],[2,-5],[2,-7],[0,-2],[1,2],[0,4],[0,5],[-1,5],[0,4],[0,5],[0,5],[-1,4],[1,5],[0,1],[0,3],[-1,7],[0,3],[0,3],[1,-1],[1,-1],[0,-1],[1,6],[-1,7],[-1,4],[-1,-3],[0,-2],[-1,-4],[-1,-4],[0,-1],[-1,1],[-1,-1],[0,-2],[-1,-1],[-1,1],[-1,-2],[0,-5],[0,-5],[-1,0],[0,1],[-1,-2],[0,-1],[1,-2],[0,-1],[0,-3],[-1,1],[0,1],[0,1],[-1,-1],[0,-1],[0,-1],[-1,-3],[0,-5],[-1,-3],[-1,0],[0,1],[-1,0],[0,-2],[0,-2],[0,-1],[0,1],[-1,1],[-1,0],[0,-3],[-1,0],[-4,0],[0,3],[-1,7],[0,-2],[0,-5],[-1,-3],[-1,1],[1,2],[0,3],[2,13],[3,11],[0,4],[4,23],[3,8],[3,17],[4,18],[1,4],[2,3],[1,7],[1,5],[0,4],[2,8],[1,7],[0,2],[2,4],[2,9],[1,5],[5,11],[2,8],[1,11],[1,3]],[[5813,4945],[2,-4],[0,-1],[1,-1],[0,-2],[0,-2],[0,-1],[3,-5],[0,-1],[0,-2],[1,-6],[0,-2],[0,-2],[0,-2],[1,-2],[8,2],[79,-131],[1,0],[1,2],[2,4],[1,6],[0,3],[0,2],[0,1],[1,1],[4,4],[1,2],[0,2],[1,1],[0,2],[0,2],[0,1],[0,2],[2,3],[0,2],[0,1],[-1,0],[-1,1],[0,1],[0,1],[0,1],[0,3],[1,2],[0,1],[1,3],[0,2],[0,2],[0,1],[0,1],[1,1],[0,1],[4,8],[1,1],[0,1],[0,1],[0,1],[-1,5],[0,1],[0,2],[0,1],[0,2],[0,1],[1,1],[0,1],[1,0],[0,-1],[1,-2],[1,-2],[0,-1],[0,-1],[1,-2],[5,-6],[0,2],[1,6],[1,4],[0,2],[0,4],[0,2],[1,8],[0,1],[0,3],[0,1],[0,2],[0,1],[0,1],[0,1],[-2,5],[-1,2],[-2,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-4],[0,-2],[0,-1],[-2,-1],[0,2],[-1,6],[-1,2],[-1,2],[0,2],[0,1],[0,1],[-1,2],[0,1],[0,1],[1,4],[2,13],[12,3],[7,2],[1,-1],[0,-1],[0,-1],[1,-3],[1,-4],[0,-1],[1,-1],[0,-5],[0,-2],[0,-3],[1,0],[0,-1],[0,-1],[0,1],[1,1],[0,1],[0,1],[1,2],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[-1,1],[0,1],[0,1],[0,2],[1,1],[2,10],[0,1],[3,2],[0,1],[1,2],[1,7],[1,-4],[1,-4],[0,-2],[1,0],[12,-6],[0,1],[1,4],[0,2],[1,1],[0,-1],[0,-2],[1,-2],[0,-2],[-1,-7],[0,-6],[0,-4],[0,-1],[0,-1],[2,-5],[1,0],[4,6],[0,1]],[[5827,4619],[0,-1],[-2,5],[0,2],[0,1],[-1,1],[-1,4],[1,0],[1,-1],[2,-6],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1]],[[5824,4612],[3,-2],[2,-6],[7,-11],[1,-1],[1,3],[1,7],[0,6],[-1,2],[1,4],[0,2],[1,2],[-1,4],[0,3],[-6,2],[-2,5],[0,9],[-1,2],[0,3],[0,3],[0,3],[0,3],[0,2],[0,4],[-1,4],[-5,9],[-4,0],[-2,-6],[-1,-2]],[[5975,2980],[0,-6],[0,-3],[0,-2],[0,-4],[0,-1],[0,-1],[0,-3],[0,-11],[0,-1],[0,-1],[0,-1],[0,-4],[0,-1],[0,-2],[0,-3],[1,-5],[0,-2],[1,-1],[0,1],[0,2],[0,1],[2,-5],[1,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[-1,-3],[0,-2],[-1,-1],[-1,-2],[0,-2],[-1,-1],[0,1],[0,2],[0,1],[0,1],[-1,0],[0,-1],[0,-1],[0,-1],[0,-1],[1,-3],[0,-1],[0,-3],[0,-1],[0,-1],[0,-1],[-1,1],[-1,-1],[-1,0],[0,1],[-1,-1],[0,1],[0,2],[-1,0],[0,1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[-1,1],[-1,0],[0,-1],[0,-4],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[-1,-2],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-2],[1,-7],[1,-9],[0,-1],[1,1],[0,1],[0,1],[1,-2],[0,-1],[1,-1],[0,-3],[0,-1],[1,0],[0,-2],[0,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-1],[1,-2],[1,-3],[-1,-9],[0,-1],[-1,-1],[0,1],[-1,-1],[0,-2],[0,-1],[1,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-13],[0,-8],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-2,0],[0,-1],[0,-1],[-1,-4],[0,-6],[-1,-5],[0,-1],[0,-2],[0,-3],[0,-2],[0,-2],[0,-1],[0,-3],[1,-1],[0,-1],[0,1],[1,-4]],[[5969,2744],[0,-5],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-5],[0,-2],[0,-1],[0,-4],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-7],[0,-2],[0,-1],[-1,-1],[-1,0],[0,1],[0,1],[0,2],[1,1],[-1,1],[0,1],[0,1],[0,-1],[-1,1],[-1,-3],[0,-1],[0,-1],[-1,-2],[0,-6],[0,-1],[0,-1],[1,-1],[0,-1],[0,-1],[1,-1],[0,-2],[0,-1],[0,-3],[0,-2],[0,-2],[-1,-2],[0,-2],[0,-2],[0,-1],[0,-2],[-1,0],[-1,-2],[0,-1],[0,-1],[0,-2],[0,-1],[-1,0],[-1,0],[0,1],[0,1],[0,2],[0,2],[0,1],[0,1],[0,1],[-1,0],[0,-1],[0,-1],[0,-1],[0,-4],[-1,-1],[-2,-7],[0,-1],[0,-3],[-1,-8],[0,-1],[0,-1],[-1,-4],[0,-1],[0,-1],[1,-1],[0,-1],[1,0],[0,-1],[0,-1],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[1,-1],[0,-2],[0,-3],[-1,-4]],[[5954,2615],[0,3],[-1,3],[-1,5],[0,2],[-1,0],[0,-1],[-1,0],[0,1],[0,3],[0,2],[0,1],[0,1],[0,1],[-2,7],[0,2],[-1,0],[0,3],[0,3],[0,4],[-1,6],[0,4],[0,5],[0,8],[-1,4],[0,1],[1,6],[0,3],[0,2],[0,2],[-1,3],[0,4],[-1,3],[-1,7],[-1,2],[0,9],[-1,4],[-1,4],[-1,3],[0,4],[0,3],[-1,3],[0,1],[-2,0],[-1,-1],[-2,-6],[-2,-11],[-1,0],[0,4],[0,9],[-1,4],[0,-1],[0,-2],[-1,-1],[-1,1],[0,3],[-1,0],[-2,0],[-2,3],[-1,0],[0,-2],[-1,-5],[0,-2],[-1,-1],[0,-1],[0,-3],[-1,-2],[0,1],[-1,0],[0,-1],[-3,-14],[0,-1],[-1,0],[-1,4],[0,1],[-1,0],[-2,-3],[-1,0],[-5,16],[-1,-2],[-3,5],[-1,-2],[0,-8],[0,-3],[0,-5],[0,-2],[0,-3],[0,-1],[0,-1],[-1,-2],[0,-4],[0,-15],[-1,-9],[-1,-6],[-4,-10],[-1,0],[-1,3],[-1,-1],[-2,-4],[0,-1],[-1,2],[0,4],[0,5],[0,4],[-1,3],[-1,1],[-1,-1],[0,7],[0,5],[-1,4],[-1,5],[0,1],[-1,3],[0,3],[0,4],[1,3],[0,2],[0,2],[-1,0],[-1,1],[0,2],[-1,3],[-1,7],[1,2],[0,2],[0,3],[0,3],[0,5],[0,3],[0,2],[0,2],[0,7],[0,7],[0,6],[-1,2],[0,3],[0,2],[0,2],[1,1],[-1,3],[0,1],[1,-1],[1,1],[0,2],[0,3],[-1,4],[-1,3],[-1,10],[-1,9],[0,4],[2,6],[0,5],[0,10],[0,5],[0,2],[-1,1],[0,3],[0,4],[0,3],[0,2],[-1,1],[-1,1],[-1,3],[-1,5],[0,5],[-1,12],[-1,5],[-2,9],[1,3],[1,12],[1,1],[1,2],[0,3],[0,2],[0,3],[0,3],[0,12],[1,4],[1,0],[2,-1],[1,3],[1,1],[1,0],[1,-4],[1,0],[1,0],[1,-2],[0,-4],[0,-10],[1,-4],[0,-1],[1,1],[0,-1],[2,-4],[0,-1],[1,0],[1,2],[1,1],[2,-5],[1,0],[0,1],[1,3],[1,1],[0,-1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[2,0],[1,4],[0,5],[1,5],[1,1],[1,2],[0,2],[1,8],[0,5],[1,5],[-1,1],[0,2],[0,3],[1,1],[0,-2],[1,0],[1,1],[1,2],[0,1],[0,2],[-1,1],[2,2],[1,0],[0,-1],[1,-1],[0,3],[0,7],[0,16]],[[5908,3009],[1,2],[0,-1],[0,-1],[1,-1],[0,-1],[0,-2],[1,6],[0,1],[0,-1],[1,-1],[0,-1],[0,2],[0,3],[2,11],[0,2],[-1,1],[0,1],[0,1],[0,1],[0,5],[0,4],[0,1],[0,2],[1,6],[0,1],[1,3],[1,6],[2,10],[2,11],[0,2],[1,0],[1,1],[0,1],[0,3],[0,2],[0,1],[0,1],[0,1],[0,2],[0,3],[0,1],[0,1],[0,1],[1,1],[1,3],[0,1],[0,3],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[-1,3],[0,1],[0,1],[0,1],[0,1],[0,1],[0,3],[0,1],[0,1],[1,2],[0,1],[-1,0],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[1,0],[1,1],[0,4],[0,1],[1,0],[0,1],[1,0],[0,-1],[0,-1],[0,-4],[-1,-1],[1,0],[0,1],[1,0],[0,1],[1,1],[0,1],[0,1],[0,1],[0,1],[0,2],[-1,1],[0,1],[1,1],[1,3]],[[5930,3153],[1,1],[0,1],[1,2],[1,-1],[1,1],[1,-1],[3,-11],[0,-2],[1,-7],[0,-3],[0,-2],[0,-4],[0,-1],[0,-2],[0,-1],[1,-2],[0,-1],[0,-1],[0,-1],[1,0],[1,1],[0,4],[0,1],[0,1],[1,0],[0,-1],[1,-2],[0,-1],[0,-1],[0,-2],[0,-2],[1,0],[0,-1],[2,2],[1,5],[0,1],[1,0],[0,-1],[1,0],[0,-2],[0,-3],[0,-1],[0,-2],[0,-5],[0,-1],[0,-1],[0,-1],[1,0],[1,-1],[0,-1],[1,-7],[0,-5],[1,-6],[1,-6],[0,-4],[0,-1],[1,-1],[1,-4],[0,-3],[1,-5],[0,-4],[0,-4],[1,-4],[0,-1],[-1,-2],[-1,-3],[0,-2],[0,-3],[0,-2],[1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-5],[0,-2],[0,-1],[1,-1],[0,-1],[0,-2],[0,-1],[0,-2],[-1,-3],[0,-4],[2,0],[1,-3],[1,-3],[0,-2],[0,-1],[1,0],[5,1],[1,-2],[1,-2],[1,0],[0,1],[1,3],[0,-2],[0,-1],[0,-5],[0,-3],[0,-3],[0,-7],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,4],[0,1],[0,1],[2,0]],[[5976,3628],[1,-2],[0,-1],[1,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,1],[0,1],[0,1],[1,0],[0,-1],[-1,-3],[0,-2],[0,-1],[1,-3],[0,-1],[0,-1],[0,-3],[1,-1],[0,-3],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[1,-2],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-3],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-2],[0,-1],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-6],[0,-2],[1,-1],[0,-1],[0,-1],[0,-5],[1,-2],[0,-2],[0,-5],[0,-2],[0,-2],[-1,-2],[0,-1],[-1,0],[-1,2],[0,-1],[-1,-2],[0,-1],[1,0],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[1,-5],[-1,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-4],[0,-8],[0,-2],[0,-1],[0,-4],[0,-3],[0,-1],[0,-3],[0,-5],[0,-1],[1,-9],[1,-6]],[[5982,3454],[-1,-8],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[1,-2],[0,-1],[0,-1],[0,-2],[-1,-2],[0,-1],[-1,-9],[0,-1],[0,-2],[-2,-8],[-1,-2],[0,-1],[0,-1],[0,-1],[1,-4],[0,-1],[0,-1],[0,-2],[0,-1],[-1,0],[-1,-1],[-1,-1],[0,-1],[0,-3],[0,-1],[0,-1],[-1,-1],[0,-4],[0,-6],[-1,-1],[-1,1],[0,1],[0,1],[-2,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-2],[0,-2],[0,-1],[-1,-3],[0,-2],[1,-1],[0,-1],[0,-2],[0,-2],[1,-1],[0,-1],[1,-2],[0,-1],[-1,0],[0,-2],[0,-2],[-1,-4],[-1,-2],[0,-4],[0,-1],[-2,-3],[0,-1],[0,-1],[-1,-1],[-1,-1],[0,-4],[-1,0],[0,-1],[-1,0],[0,1],[-1,2],[-2,0],[-1,2],[0,1],[-1,0],[-1,-1],[0,-1],[0,-1],[2,-3],[0,-1],[0,-1],[1,-1],[0,-2],[0,-3],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-1],[0,-1],[0,-4],[-1,-7],[0,-3],[0,-2],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-5],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-3],[-1,-1],[-1,1],[0,1],[-3,18],[-2,9],[-1,2],[-1,-1],[-1,2],[-3,5],[-1,-4],[-1,0],[-1,1],[0,1],[-1,-2],[-1,1],[-1,0],[0,-1],[0,-2],[0,-1],[-1,-9],[0,-2],[-1,-1],[0,-3],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,0],[0,-1],[0,-1],[0,-1],[0,-2],[-1,-5],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-2],[1,-4],[0,-5],[1,-4],[0,-6],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[-1,0],[-1,-1],[0,-1],[0,-1],[1,-1],[0,-1],[0,-1],[0,-6],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-4],[0,-2],[-1,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[0,1],[-1,0],[0,-2],[-1,-5],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[1,1],[0,-1],[1,0],[0,-1],[0,-2],[0,-2]],[[5908,3009],[0,2],[0,6],[-1,0],[0,-2],[-1,1],[0,3],[0,2],[0,2],[0,1],[-1,1],[0,-1],[0,-1],[0,-1],[-1,1],[0,1],[0,5],[-1,2],[-1,4],[-1,3],[-1,3],[0,5],[-1,7],[0,4],[0,2],[1,1],[2,6],[0,2],[0,2],[0,2],[0,1],[-1,4],[0,2],[-1,0],[-1,0],[-1,1],[-1,6],[-1,4],[-3,5],[-2,7],[-6,-8],[-3,5],[-1,0],[0,6],[2,9],[0,6],[0,14],[1,6],[0,2],[0,1],[0,2],[0,2],[-1,13],[0,1],[0,1],[0,1],[-1,-3],[0,1],[0,3],[0,4],[-1,2],[0,2],[0,1],[-1,0],[-1,1],[-2,7],[-2,2],[0,1],[-1,4],[-4,17],[-1,4],[0,7],[0,15],[-1,3],[0,3],[0,2],[-1,3],[0,12],[-1,5],[0,1],[0,1],[-1,3],[-2,3],[-1,3],[1,11],[1,11],[1,10],[-1,5],[-1,6],[-2,2],[0,-1],[-1,5],[0,4],[-4,16],[-1,2],[-1,1],[0,4],[0,7],[1,5],[0,3],[1,4],[0,6],[-1,4],[0,2],[1,3],[2,2],[0,2],[1,4],[0,3],[-1,3],[0,4],[0,3],[1,0],[1,-1],[1,1],[0,4],[-1,6],[0,5],[0,2],[0,2],[0,2],[0,1],[0,3],[0,1],[0,1],[0,1],[0,1],[-1,0],[0,2],[0,2],[0,1],[0,2],[-1,5],[-1,3],[0,9],[0,3],[-1,3],[-1,2],[0,3],[0,5],[0,4],[1,7],[2,6],[0,3],[0,5],[0,3],[-1,4],[0,4],[0,4],[1,3],[0,3],[0,2],[0,8],[0,10],[0,1]],[[5874,3574],[1,1],[1,3],[1,1],[1,-2],[2,-3],[1,-3],[1,0],[0,2],[1,5],[0,1],[0,2],[1,0],[1,1],[0,2],[1,1],[0,-1],[2,-5],[1,-1],[0,-3],[0,-2],[2,-4],[0,-1],[0,-2],[-1,0],[0,-1],[-1,-1],[0,-2],[0,-2],[0,-2],[1,-1],[0,1],[1,2],[0,1],[1,1],[1,1],[1,-2],[0,-1],[1,2],[1,-1],[0,1],[0,1],[1,2],[0,1],[0,3],[1,1],[1,-3],[1,0],[1,-4],[0,1],[0,1],[1,2],[0,1],[1,-9],[1,-2],[0,1],[1,4],[0,-1],[0,-1],[1,0],[1,3],[0,2],[1,0],[1,-1],[0,-1],[0,-2],[0,-2],[2,-1],[1,1],[0,-1],[1,-2],[1,-6],[1,1],[0,1],[0,3],[0,1],[1,0],[1,-2],[1,2],[2,-3],[1,1],[0,1],[0,1],[1,1],[0,2],[0,3],[0,2],[0,1],[0,2],[2,3],[1,-1],[1,-1],[1,0],[0,1],[0,2],[0,1],[0,2],[1,1],[1,4],[1,2],[0,1],[1,4],[0,3],[1,2],[0,2],[0,1],[0,2],[0,1],[0,1],[2,3],[0,3],[1,4],[0,5],[0,3],[0,2],[0,2],[-1,4],[0,1],[0,1],[0,2],[1,0],[0,-1],[1,0],[0,1],[1,0],[1,-5],[1,1],[0,2],[0,2],[0,4],[0,3],[0,3],[1,5],[1,6],[0,2],[0,2],[0,2],[0,2],[0,1],[0,1],[1,0],[1,-5],[3,0],[1,1],[2,4],[1,0],[0,-1],[1,-2],[0,-2],[1,-2],[0,-2],[0,-2],[0,-6],[0,-1],[1,-1],[0,1],[1,2],[0,1],[1,-1],[1,-4],[2,-7],[1,-5],[2,-6],[1,0],[1,0],[0,2],[1,1],[1,7],[0,2],[1,-1],[0,-2],[1,-1],[6,8],[1,-5],[0,-2],[1,-1],[1,-1]],[[6000,5690],[-1,0],[-1,1],[-1,9],[-1,1],[-1,2],[0,4],[1,2],[1,1],[1,-3],[0,-2],[2,-5],[1,-4],[-1,-4],[0,-2]],[[5998,5860],[-1,-1],[0,1],[0,2],[-1,2],[1,1],[0,2],[1,0],[1,1],[0,-1],[1,0],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[-2,0]],[[5994,5892],[1,-8],[1,-11],[0,-1],[-1,-2],[0,1],[-1,5],[0,1],[0,3],[0,2],[0,-4],[-1,-5],[0,-1],[0,-1],[1,-3],[1,-2],[0,-1],[1,-4],[0,-4],[-1,-2],[0,-1],[0,-3],[0,-1],[-1,-2],[-1,-3],[-1,-1],[0,3],[0,3],[0,4],[0,3],[0,3],[-1,1],[0,1],[-1,0],[-2,1],[-1,4],[-1,5],[0,5],[0,7],[0,5],[1,2],[1,1],[2,0],[0,3],[1,1],[1,0],[1,-2],[1,-2]],[[5999,5900],[0,-1],[1,0],[0,-2],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[4,0],[0,1],[1,0],[1,-1],[-1,-2],[-1,-2],[-2,0],[-1,-2],[-1,-1],[-1,1],[-2,4],[0,2],[0,3],[0,3],[0,2],[1,1],[0,-1]],[[5913,6244],[0,-3],[4,0]],[[5921,6236],[-2,-1],[-1,2],[-1,-3],[-2,-1],[-1,-3],[2,-4],[1,-2],[2,1],[2,5],[1,1],[2,-3],[0,-1],[0,-1],[1,-1],[0,-1],[0,-2],[0,-3],[1,-1],[4,-4],[1,1],[1,7],[1,1],[0,-3],[1,-4],[-1,-1],[-1,-2],[1,-2],[3,-2],[1,-5],[-2,-3],[-3,2],[-2,-4],[1,-3],[3,-3],[1,-6],[-1,-3],[-5,2],[-2,-2],[-1,-2],[-1,-4],[4,-1],[4,4],[0,-1],[1,-1],[0,-2],[-1,-1],[7,-11],[1,-2],[0,-2],[0,-1],[0,-2],[0,-2],[1,1],[0,2],[0,2],[1,0],[3,0],[1,-1],[0,-1],[1,-4],[1,-1],[-1,-2],[0,-2],[0,-2],[0,-2],[0,-1],[2,-2],[1,-2],[1,-1],[1,0],[5,-12],[1,-6],[0,3],[0,-2],[1,-7],[1,-2],[1,-6],[0,-1],[0,-2],[2,-6],[0,-2],[0,-3],[-1,-3],[-1,-3],[1,-2],[0,-1],[1,-2],[-1,-1],[0,-2],[0,-3],[0,-1],[1,1],[1,2],[0,1],[1,-1],[0,-2],[0,-1],[1,1],[0,-1],[1,-1],[0,-2],[0,-3],[0,-2],[-1,-7],[0,-4],[1,0],[1,1],[0,-2],[0,-3],[0,-4],[-1,-1],[0,-1],[0,-3],[0,-3],[1,-2],[1,-3],[1,-3],[-1,-3],[0,-2],[0,-2],[-1,1],[-4,17],[-2,6],[0,3],[0,-8],[0,-2],[0,-1],[0,-8],[0,-1],[0,-2],[2,0],[0,-2],[0,-3],[-1,-2],[-1,-2],[0,-2],[1,-2],[0,-1],[1,-2],[0,-4],[0,-7],[0,-4],[0,-1],[-1,-2],[0,-10],[-1,-5],[-1,-4],[-1,-4],[1,-1],[0,-1],[1,-3],[0,-4],[-1,-2],[-3,3],[-1,0],[0,-4],[0,-7],[-1,-1],[-1,2],[0,2],[-1,0],[0,-2],[-1,1],[0,-2],[0,-1],[1,-2],[-1,-2],[-1,1],[-1,-3],[4,-3],[1,-2],[0,-3],[0,-5],[0,-3],[0,-7],[0,-2],[1,-2],[0,1],[1,0],[-1,-4],[1,-2],[2,0],[1,-3],[-1,-3],[1,-1],[1,-6],[0,-4],[-1,-1],[0,-1],[-1,-1],[1,-2],[1,-3],[-1,-6],[1,-3],[2,-2],[0,-2],[0,-3],[0,-5],[0,-3],[1,-1],[1,-2],[-1,-5],[-1,-2],[0,-2],[-1,-2],[-1,-4],[2,-1],[0,-2],[1,-5],[1,-6],[0,-1],[1,0],[0,-1],[0,-1],[0,-3],[0,-2],[0,-2],[1,-2],[0,-2],[1,-2],[0,-4],[-1,-4],[-1,-6],[-1,-2],[-1,-2],[-1,-3],[1,-4],[0,-1],[0,1],[2,0],[2,2],[0,1],[1,0],[1,-6],[-1,-4],[-1,-1],[-2,-2],[-1,-3],[0,-1],[2,-9],[1,-7],[1,-2],[0,-4],[-1,-5],[-2,-2],[-1,-5],[-1,-1],[-1,1],[0,-1],[1,-2],[0,-1],[1,-3],[0,-1],[2,0],[1,-1],[1,-5],[1,-3],[0,-5],[-2,-5],[2,-7],[1,-2],[2,-3],[0,-3],[1,-3],[3,-1],[1,-1],[1,-2],[-1,-5],[1,-3],[0,-4],[0,-4],[2,-3],[2,5],[2,4],[1,0],[1,5],[-1,7],[0,3],[0,3],[-1,3],[1,1],[1,-7],[3,-6],[2,-4],[2,-5],[0,-6],[0,-7],[1,-3],[1,-2],[0,-2],[0,-3],[0,-5],[1,-2],[1,-2],[0,3],[1,4],[1,-3],[1,-3],[1,-6],[1,-4],[1,-5],[0,-2],[0,-3],[-1,-3],[1,-7],[1,-2],[0,-5],[0,-2],[0,-3],[1,-4],[2,-2],[1,-1],[1,-3],[1,-3],[1,0]],[[6014,5601],[0,-1],[0,-4],[0,-1],[-1,-2],[1,-1],[0,-4],[0,-1],[0,-3],[0,-8],[0,-9],[0,-3],[0,-2],[0,-1],[0,-2],[1,-3],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-4],[-1,-11],[0,-4],[-1,-3],[0,-2],[-1,-3],[-1,-2],[-12,-7],[-2,-3],[-1,-2],[1,-1],[0,-2],[0,-3],[3,-21],[1,-3],[1,-4],[4,-8],[0,-1],[1,-2],[0,-2],[0,-2],[0,-8],[0,-5],[0,-5],[1,-2],[0,-1],[0,-3],[0,-5],[0,-11],[0,-3],[1,-1],[2,-1],[0,-1],[1,0],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-4],[-2,-3],[0,-2],[0,-2],[0,-4],[0,-2],[0,-2],[-1,-13],[0,-1],[0,-1],[0,-2],[0,-3],[0,-2],[0,-1],[0,-2],[0,-3],[0,-1],[4,-26],[2,-16],[4,-14],[2,-5],[2,-7],[8,-16],[1,-1],[1,0],[1,0],[0,1],[1,2],[0,2],[0,2],[1,1],[3,0],[1,-1],[1,-3],[2,-5],[0,-2],[1,-3],[0,-7],[0,-6],[0,-1],[0,-4],[0,-7],[1,-3],[0,-3],[0,-3],[0,-2],[0,-2],[1,-2],[0,-2],[0,-4],[-1,-1],[0,-8],[0,-2],[0,-1],[0,-7],[0,-2],[0,-1],[-1,-5],[0,-1],[0,-1],[-1,-4],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-3],[-1,-4],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[1,-1],[0,1],[1,3],[6,-3],[0,-1],[1,-1],[0,-3],[0,-10],[0,-1],[0,-2],[1,-2],[0,-2],[0,-3],[0,-2],[0,-2],[0,-6],[0,-7],[1,-8],[0,-3],[3,-5],[0,-1],[0,-1],[0,-6],[0,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-1],[0,-1],[0,-2],[0,-5],[-1,-3],[0,-3],[0,-11],[-1,-9],[0,-3],[1,-2],[0,-2],[0,-1],[0,-1],[0,-2],[-1,-3],[0,-1],[0,-1],[-2,-7],[0,-1],[0,-1],[-1,-1],[0,-2],[0,-12],[1,-19]],[[6049,4999],[-14,-6],[0,2],[0,5],[0,1],[0,1],[-1,1],[-2,1],[0,1],[-1,2],[-4,9],[-1,-1],[-2,-11],[-1,-1],[-5,4],[-1,-1],[-1,-1],[0,-1],[0,-1],[-1,-3],[-2,-19],[0,-1],[0,-7],[-1,-3],[-1,-2],[-23,-51]],[[5813,4945],[0,3],[1,5],[4,19],[0,5],[1,6],[0,4],[1,2],[2,4],[1,3],[4,20],[0,4],[1,9],[0,4],[4,16],[5,30],[5,35],[4,19],[4,20],[0,8],[1,3],[0,1],[1,3],[3,10],[3,10],[0,2],[1,3],[1,9],[0,2],[4,19],[0,5],[1,3],[0,2],[2,4],[0,3],[1,5],[0,12],[1,6],[1,8],[1,5],[1,24],[1,10],[1,9],[2,20],[-1,3],[0,6],[0,6],[-1,5],[0,1],[-1,2],[-2,12],[-1,3],[-1,6],[-1,4],[-1,3],[0,5],[0,6],[0,4],[0,5],[-1,4],[-1,3],[-5,16],[-4,21],[-10,22],[0,2],[-1,5],[-2,10],[-4,12],[-6,33],[-1,6],[1,2],[1,4],[4,7],[2,9],[2,16],[5,24],[1,7],[1,7],[0,9],[-1,10],[0,5],[0,4],[0,4],[0,5],[0,5],[0,4],[-1,2],[-2,2],[-1,3],[0,4],[-1,5],[-1,4],[-5,6],[-2,5],[-1,4],[0,4],[0,7],[0,6],[1,6],[-1,5],[-2,3],[1,1],[0,1],[-1,1],[0,1],[0,1],[0,3],[0,2],[0,4],[0,1],[0,1],[0,1],[0,1],[5,9],[0,3],[1,4],[0,5],[-1,5],[-1,4],[0,2],[0,3],[0,1],[0,2],[0,3],[0,2],[0,2],[0,2],[0,2],[-2,2],[-5,0],[-2,3],[-3,10],[-1,17],[-2,21],[1,10],[0,7],[2,4],[4,4],[0,3],[1,4],[-1,0],[0,1],[-1,2],[0,2],[1,2],[0,2],[0,2],[0,9],[-2,3],[-4,0],[-1,5],[1,8],[3,15],[0,9],[0,21],[0,10],[0,1],[1,4],[2,9],[0,4],[-1,5],[-2,5],[0,6],[1,4],[7,8],[2,-5],[0,-1],[1,2],[0,6],[0,6],[-1,23],[0,4],[-1,11],[-1,12],[0,4],[-1,10],[-1,5],[0,19],[-1,9],[-1,6],[-2,12],[-4,26],[-2,15],[0,8],[-1,8],[0,5],[0,10],[0,4],[-4,19],[0,2],[0,3],[0,2],[0,2],[-1,5],[0,1]],[[6058,4875],[0,-2],[-2,2],[0,1],[0,2],[0,3],[0,1],[1,0],[0,-2],[1,-3],[0,-1],[0,-1]],[[6112,5780],[1,0],[2,2],[1,0],[1,-2],[1,-3],[0,-2],[0,-6],[1,-5],[1,-4],[2,0],[0,-1],[1,-4],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[-3,2],[-4,0],[-1,4],[-1,6],[0,4],[-2,4],[0,1],[-1,2],[-2,1],[-1,1],[-1,4],[-1,0],[0,-1],[0,2],[0,3],[2,3],[2,2],[1,-1],[0,-1],[1,-3]],[[6110,5787],[-1,0],[0,3],[1,1],[1,-2],[-1,-2]],[[6122,5763],[-1,-1],[-1,3],[-1,3],[-1,5],[0,7],[-2,5],[-1,2],[0,1],[-2,6],[-2,1],[1,2],[2,3],[3,1],[3,-3],[1,-4],[0,-5],[1,-6],[0,-12],[0,-3],[0,-2],[0,-3]],[[6228,6200],[2,1],[1,-1],[4,-9],[0,-1],[3,-15],[7,-17],[2,-2],[1,1],[22,-41],[2,5],[15,-14],[21,-51],[1,-2],[7,5],[7,-7],[2,2],[0,1],[0,1],[1,1],[0,1],[0,2],[0,1],[0,2],[0,1],[0,1],[0,2],[-1,10],[0,1],[0,2],[0,1],[1,4],[4,23],[2,5],[0,1],[2,4],[1,0],[0,1],[0,5],[1,5],[0,1],[1,1],[0,1],[24,-3]],[[6361,6129],[-1,-172],[0,-37],[0,-4],[0,-3],[1,-1],[15,14],[1,-1],[1,-4],[1,-5],[11,-89],[1,-6],[2,-5],[8,-11],[0,-2],[-3,-63],[0,-3],[-1,-3],[-1,0],[-18,19],[-1,0],[-1,-2],[0,-3],[-2,-21],[0,-3],[-1,-2],[-14,7],[0,-2],[-1,-3],[-2,-28],[-1,-3],[-8,2],[0,-1],[0,-2],[-2,-23],[0,-3],[-1,-3],[-4,1],[-2,3],[-1,3],[0,4],[-1,5],[0,4],[0,4],[0,3],[0,3],[-42,-5],[-8,-15],[-11,6],[-3,-1],[-5,-11],[-5,-13],[-1,-4],[1,-4],[3,-19],[1,-3],[5,-14],[3,-11],[2,-3],[4,-1],[9,-15],[11,-34],[0,-2],[4,-25],[0,-2],[1,-9],[1,-9],[0,-4],[0,-3],[0,-1],[-1,-5],[-1,-8],[0,-3],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[0,-4],[-1,-3],[0,-1],[0,-1],[0,-3],[1,-3],[2,-7],[0,-3],[0,-3],[0,-3],[1,-1],[0,-3],[4,-7],[2,-3],[1,-3],[1,-7],[1,-1],[1,-2],[3,-2],[0,1],[0,-3],[0,-4],[0,-18],[-1,-3],[0,-2],[0,-1],[-2,-13],[0,-1],[-1,-3],[-5,-18],[-4,-7],[0,-1],[0,-2],[3,-31],[1,-14],[1,-6],[0,-6],[0,-6],[-1,-51],[0,-6],[2,-3],[3,-5],[0,-8],[0,-16],[0,-4],[0,-3],[2,-2],[3,-1],[1,1],[0,2],[1,4],[0,21],[3,6],[13,-1],[1,0],[1,1],[0,3],[0,4],[0,12],[0,1],[0,1],[-1,4],[0,2],[-1,3],[0,2],[0,2],[0,1],[0,2],[3,57],[0,4],[1,2],[7,-5],[1,0],[0,3],[1,3],[1,21],[4,3],[20,-14],[-6,-153],[0,-2],[0,-1],[-1,0],[-1,0],[-1,1],[0,-1],[0,-3],[0,-1],[0,-2],[0,-1],[0,-1],[-1,0],[0,-1],[-1,-1],[0,-1],[-4,-99],[1,-8],[8,-8],[1,-2],[1,-8],[-1,-32],[0,-3],[-3,-59],[0,-4],[0,-2],[-1,-1],[-2,2],[-3,-2],[-4,-13],[-1,5],[-1,0],[-10,-23],[-1,-2],[0,-1],[0,-6],[-1,-3],[0,-4],[0,-8],[0,-11],[-1,-13]],[[6344,4834],[-10,7],[-2,4],[0,5],[0,4],[0,1],[0,3],[0,5],[0,7],[0,1],[0,1],[-1,0],[-1,-2],[-1,1],[-2,5],[-1,1],[-1,1],[-2,1],[0,2],[0,1],[-1,1],[0,1],[-1,1],[-2,-3],[-7,2],[-1,-1],[-1,-1],[1,-6],[0,-1],[0,-1],[-1,-2],[0,-2],[0,-1],[0,-1],[-2,-6],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[-1,-9],[0,-6],[0,-2],[0,-2],[0,-1],[-1,-1]],[[6306,4834],[-3,1],[0,3],[0,1],[-1,2],[0,1],[-3,3],[-1,1],[0,1],[0,2],[0,14],[0,1],[0,2],[0,1],[-3,1],[-1,1],[-1,1],[0,4],[-1,2],[0,1],[0,4],[0,2],[-1,2],[0,1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-1],[-1,0],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[-2,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-2],[-1,-1],[0,-1],[-11,12],[-2,5],[-1,5],[0,24],[0,2],[0,1],[-2,3],[-9,7],[-1,-1],[0,-3],[-3,-70],[-22,10],[-7,6],[0,3],[-1,3],[0,17],[0,2],[0,2],[0,1],[-1,0],[-2,-1],[0,-1],[0,-1],[0,-2],[-1,-1],[0,-1],[0,-1],[-2,-1],[-1,-3],[0,-2],[0,-1],[-1,0],[0,1],[0,1],[-1,0],[-1,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-10],[0,-6],[0,-3],[-1,-3],[0,-1],[0,-2],[0,-1],[1,-1],[0,-2],[-1,-1],[0,-1],[-2,-2],[-3,-6],[-1,-1],[0,-2],[0,-1],[0,-2],[0,-12],[0,-2],[0,-1],[-8,4],[-1,3],[0,1],[-1,1],[0,1],[0,-1],[0,-1],[-2,-12],[-1,-1],[0,-2],[-1,-1],[-1,1],[0,4],[0,5],[-1,2],[0,2],[0,-2],[-1,-1],[0,1],[0,1],[0,3],[0,2],[0,1],[0,2],[0,1],[-2,8],[-1,0],[-4,2],[-1,-2],[0,-1],[-1,-4],[0,-1],[-1,-1],[0,-2],[0,-1],[1,-2],[0,-1],[0,-2],[0,-1],[-1,-2],[0,-1],[-1,3],[-1,0],[-1,-5],[-1,0],[-10,16],[-1,0],[-1,-1],[0,1],[-1,2],[0,1],[0,1],[0,2],[0,1],[0,1],[1,1],[0,1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[-1,2],[0,1],[0,-1],[0,-1],[-1,-2],[0,-1],[0,2],[-1,1],[0,1],[0,-1],[-1,-2],[0,-1],[-1,-2],[0,-4],[-1,-4],[0,-2],[0,-1],[-1,-1],[-1,1],[0,1],[0,1],[0,1],[0,5],[-1,1],[0,1],[0,1],[-3,3],[0,-1],[-4,-10],[0,-1],[0,-1],[0,-2],[0,-5],[1,-3],[0,-2],[0,-2],[-1,-8],[0,-6],[-1,-3],[0,-1],[0,-6],[0,-3],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[0,-1],[-2,5],[-1,0],[0,-2],[-5,-2],[0,1],[-1,1],[0,2],[-6,11],[0,2],[-1,1],[0,1],[0,1],[0,2],[0,1],[0,2],[0,1],[0,1],[0,2],[0,2],[-1,1],[0,-1],[0,-1],[0,-2],[0,-1],[-1,0],[-1,-1],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-2],[-2,-1],[0,-1],[0,1],[-1,1],[0,5],[-1,2],[-1,5],[0,2],[-1,1],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[-1,-3],[-1,-1],[-2,-2],[-1,-1],[0,-2],[0,-1],[-1,-1],[-2,2],[-1,0],[-1,-1],[0,-3],[0,-2],[-1,-1],[-1,-1],[-1,1],[0,1],[-1,4],[-2,9],[0,4],[-1,2],[0,1],[-1,0],[-1,-1],[-1,0],[0,2],[0,1],[-1,0],[0,-1],[-1,-2],[-4,-6],[-1,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[-1,0],[0,1],[0,1],[0,1],[0,1],[-1,1],[0,1],[-3,2],[-1,-1],[0,-1],[0,-1],[1,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-3],[-1,8],[-1,4],[-1,3],[-1,4],[0,2],[-1,3],[0,3],[0,1],[-1,1],[0,7],[0,2],[-1,-1],[0,-6],[0,-2],[-1,0],[0,1],[-1,0],[0,-3],[-1,-5],[0,-3],[-1,-3],[0,-1],[-4,-1],[0,2],[0,1],[-1,15],[-1,2],[0,1],[0,1],[-4,1],[0,2],[0,3],[0,10],[0,2],[-1,16],[-1,3],[0,1],[0,-1],[0,-1],[0,-3],[-1,-4],[0,-1],[0,-1],[0,1],[-1,1],[-1,-1],[-1,0],[0,1],[0,2],[0,1],[0,1],[-1,5],[0,2],[0,2],[0,1],[0,2],[0,1],[0,3],[0,4],[0,1],[0,2],[-1,7],[0,2],[0,2],[0,2],[0,4],[0,2],[0,1],[1,3],[0,1],[0,1],[0,1],[-2,5],[0,1],[-1,0],[0,-1],[-1,-3],[-2,0],[0,1],[0,1],[0,1],[0,2],[0,8],[0,1],[0,2],[-1,2],[0,2],[0,3],[0,4],[-1,2],[0,1],[-1,3],[0,2],[0,2],[0,2],[0,3],[0,3],[0,2],[0,1],[0,2],[0,2],[0,1],[-1,2],[0,2],[0,5],[-1,4],[0,2],[0,1],[0,2],[1,7],[0,2],[0,2],[0,2],[0,2],[-1,2],[0,6]],[[6014,5601],[1,0],[0,-1],[2,-3],[0,-1],[0,-1],[1,-5],[1,-1],[2,4],[1,-2],[1,-4],[0,-1],[1,1],[1,3],[0,1],[1,-1],[1,-6],[0,-1],[5,-1],[1,-1],[0,-3],[1,-8],[0,-1],[1,-4],[0,-1],[0,-1],[4,-6],[3,2],[2,4],[0,6],[0,4],[0,6],[1,4],[2,2],[3,-1],[1,3],[2,6],[2,2],[0,2],[2,15],[0,4],[0,6],[-1,22],[0,4],[-1,4],[-2,6],[0,4],[-1,5],[1,4],[2,5],[1,4],[0,4],[-1,2],[-1,2],[-1,3],[-1,1],[1,3],[0,1],[1,1],[-4,-1],[-1,5],[0,10],[1,1],[-1,3],[-1,1],[-2,-2],[-1,-5],[0,-6],[0,-2],[-1,0],[-1,2],[-2,0],[-1,-1],[-2,-6],[-1,-4],[-2,2],[-4,11],[-2,12],[-6,37],[-2,11],[-1,10],[-1,7],[-2,3],[-2,-1],[-1,9],[0,4],[0,6],[0,2],[0,6],[-1,2],[2,-2],[0,1],[-2,5],[0,4],[-1,5],[0,7],[0,5],[2,-2],[1,-6],[2,0],[1,6],[1,3],[2,2],[1,3],[0,4],[0,4],[0,7],[0,2],[1,9],[0,2],[0,10],[-1,4],[0,4],[2,1],[2,2],[1,2],[0,1],[2,-1],[2,-3],[2,-4],[3,-6],[2,-3],[3,-8],[2,-5],[1,-2],[2,-1],[1,-5],[0,-5],[0,-5],[0,-4],[2,-10],[7,-16],[3,-3],[3,1],[3,1],[1,-1],[-1,-5],[0,-4],[0,-3],[1,-3],[0,-3],[-1,2],[-1,4],[-1,2],[-1,-5],[0,-1],[-3,2],[-2,-1],[0,-1],[-1,-5],[0,-4],[2,-10],[0,-2],[0,-2],[0,-2],[0,-2],[-2,-3],[0,-3],[1,-1],[0,1],[1,2],[0,1],[1,2],[0,1],[1,3],[0,5],[0,3],[0,2],[1,0],[1,0],[1,2],[1,6],[1,1],[2,-4],[1,0],[1,6],[-1,0],[-1,6],[0,2],[0,4],[1,-2],[1,-5],[3,-5],[3,-5],[3,-2],[1,0],[2,-2],[3,-9],[3,-8],[3,-4],[1,0],[1,-2],[0,-4],[1,-4],[2,-6],[3,-9],[5,2],[1,2],[-1,4],[1,1],[2,1],[0,2],[2,2],[0,5],[-1,-1],[-2,-4],[0,-1],[0,4],[0,11],[1,3],[0,1],[1,-1],[5,-12],[2,-5],[1,-7],[1,-4],[0,-1],[4,2],[1,0],[1,-3],[2,-2],[1,0],[0,4],[0,5],[0,4],[-1,3],[0,3],[0,4],[-1,16],[-1,3],[0,1],[0,1],[0,2],[0,2],[0,3],[0,2],[1,3],[1,0],[0,-1],[1,-3],[0,5],[-1,4],[-1,2],[0,3],[-1,4],[-1,24],[0,4],[-1,3],[-1,6],[0,2],[0,2],[-1,5],[0,2],[-1,4],[-3,6],[0,3],[-7,50],[-3,11],[-1,8],[0,6],[0,6],[-1,11],[0,5],[1,6],[0,8],[0,3],[1,4],[2,13],[4,12],[2,3],[6,15],[6,21],[4,29],[2,10],[3,5],[6,4],[2,5],[7,8],[1,6],[1,1],[0,1],[1,6],[1,1],[0,1],[2,6],[2,3],[0,2],[0,3],[0,1],[0,1],[6,26],[5,28],[2,23],[1,5],[2,2],[4,-6],[8,-14],[0,-2],[0,-5],[-2,-6],[-1,-4],[6,3],[5,6],[5,1],[4,3],[1,-3],[1,-4],[1,-10],[0,-5],[1,-4],[3,0],[1,-2],[3,-9],[1,-5],[0,-7],[-1,-5],[-4,-8],[-1,-4],[-1,-5],[0,-3],[-1,-3],[0,-1],[-1,-2],[-1,-1],[-1,-1],[0,-3],[1,-6],[0,-2],[1,-4],[0,-1],[1,1],[1,-1],[1,-4],[1,-5],[0,-7],[0,11],[0,5],[0,4],[-3,2],[-1,2],[1,7],[2,7],[0,1],[1,1],[1,3],[1,1],[1,0],[1,3],[0,2],[1,3],[1,0],[1,-2],[2,2],[1,-1],[1,-5],[1,-5],[1,-8],[2,-6],[0,-3],[0,-4],[0,-3],[0,-1],[-1,-2],[0,-4],[1,-5],[1,-8],[2,-19],[1,-5],[0,10],[-1,8],[0,6],[-1,7],[0,6],[1,11],[0,7],[0,4],[-2,9],[0,5],[0,11],[0,14],[1,13],[1,10],[0,4],[1,8],[1,1]],[[6185,6264],[-1,-1],[-2,3],[-3,8],[-1,5],[0,6],[1,5],[1,1],[1,-1],[1,-1],[2,-4],[0,-3],[1,-5],[0,-5],[0,-5],[0,-3]],[[6583,7224],[1,-3],[1,-1],[1,1],[2,-5],[1,-3],[-1,-4],[-3,5],[-1,2],[0,1],[0,-4],[2,-3],[1,-5],[-3,2],[-3,7],[-1,7],[0,1],[1,0],[-1,3],[-1,2],[-1,5],[3,-6],[2,-1],[0,-1]],[[6478,7417],[0,-1],[4,-15],[0,-2],[1,-1],[-1,0],[0,-1],[-1,1],[-3,7],[0,2],[-1,4],[0,2],[-1,2],[0,1],[0,1],[1,0],[0,-1],[1,1]],[[6464,7429],[4,-7],[1,-3],[0,-2],[1,-6],[0,-4],[1,-1],[0,-2],[1,-2],[0,-4],[-3,7],[0,3],[-1,5],[0,3],[-2,5],[-1,1],[-1,-1],[1,-1],[2,-6],[1,-3],[-1,-2],[0,-1],[-2,1],[0,-2],[8,-19],[2,-2],[1,3],[-1,2],[-1,0],[0,1],[0,1],[1,2],[1,1],[0,-1],[1,-2],[0,-1],[0,-3],[0,-4],[0,-3],[1,0],[-1,-4],[0,-2],[0,-1],[1,-5],[-1,-4],[-1,-3],[0,-3],[0,-1],[0,-3],[0,-2],[0,-1],[0,-2],[-1,0],[0,-1],[-1,0],[0,1],[-1,0],[0,-4],[0,-10],[0,-4],[2,-2],[0,-4],[0,-4],[-2,-2],[-1,0],[0,6],[0,3],[-1,6],[0,4],[-1,3],[0,2],[-4,7],[0,3],[0,2],[-1,4],[0,1],[0,6],[0,2],[0,3],[-2,2],[-1,2],[0,4],[0,3],[-1,4],[-2,2],[-1,1],[-2,-4],[-2,2],[-3,7],[-1,4],[0,5],[0,6],[0,2],[1,2],[3,4],[1,1],[0,2],[2,-1],[0,1],[1,2],[1,0],[3,-1],[0,-2],[1,-5],[1,-1],[0,1],[-1,4],[0,3],[0,4],[0,2],[1,0]],[[6536,7904],[5,-4],[7,1],[2,-3],[0,-1],[0,-3],[0,-1],[0,-1],[8,-7],[9,-8],[0,-1],[1,-2],[0,-3],[0,-2],[2,-7],[1,-4],[0,-5],[-1,-2],[-10,-7],[-10,-7],[1,-2],[7,1],[8,2],[2,-4],[-1,-7],[-2,-5],[-1,-2],[-3,3],[-8,-5],[-2,-2],[-6,-2],[-1,-2],[1,0],[1,-3],[1,0],[3,1],[4,-4],[4,7],[3,2],[1,-1],[0,-3],[-1,-13],[1,-2],[0,-4],[-1,-7],[0,-2],[0,-4],[0,-1],[0,-1],[-1,-1],[0,-2],[-1,-7],[-2,-1],[-6,6],[-3,-4],[-2,0],[-5,6],[-1,-1],[-1,-5],[5,0],[3,-8],[3,1],[1,-5],[2,-17],[-5,2],[-3,7],[-1,-2],[3,-11],[0,-3],[-1,-3],[-2,-1],[1,-4],[0,-3],[1,-5],[0,-3],[-2,-5],[0,-1],[-2,1],[0,2],[-1,6],[-1,2],[-5,1],[5,-6],[0,-6],[1,0],[1,-3],[0,-1],[0,-2],[0,-4],[0,-2],[0,-2],[0,-2],[-3,0],[-1,-3],[-1,0],[0,1],[-6,0],[-1,-3],[6,-2],[1,-1],[1,-1],[1,2],[1,-2],[0,-2],[0,-3],[0,-3],[1,-2],[0,-3],[-1,-3],[0,-3],[0,-2],[-2,-9],[0,-3],[0,-4],[3,-6],[1,-6],[1,-7],[0,-2],[0,-4],[-1,-3],[-1,-3],[-2,-10],[-1,-1],[0,-2],[-1,-7],[0,-3],[2,0],[1,-2],[-1,-2],[-1,-5],[0,-3],[0,-2],[-1,-3],[0,-2],[0,-3],[0,-1],[1,-2],[0,-2],[0,-3],[-1,0],[-3,3],[0,-3],[0,-2],[-1,-2],[5,-1],[1,-1],[0,-2],[2,-2],[0,-1],[0,-1],[0,-2],[0,-1],[1,-5],[0,-2],[0,-2],[0,-3],[0,-12],[0,-9],[2,-16],[2,-18],[2,-13],[2,-6],[0,-4],[2,-16],[1,-5],[1,-6],[1,-5],[1,-4],[0,-12],[0,-5],[1,-5],[1,-5],[6,-25],[7,-25],[2,-11],[4,-11],[1,-2],[1,-2],[0,-2],[1,-7],[1,-2],[1,-2],[2,-7],[0,-2],[2,-1],[3,-5],[1,-4],[1,-3],[5,2],[0,-7],[4,-14],[-2,-1],[-1,1],[-1,3],[-1,1],[0,-1],[0,-4],[2,-3],[1,-3],[-1,-5],[-1,-4],[0,-5],[-1,-2],[0,-2],[-1,-1],[0,-1],[1,-1],[0,-1],[0,-2],[-5,6],[-5,11],[-1,0],[-1,0],[0,-1],[-1,-2],[0,-1],[-2,2],[-2,8],[-2,3],[-1,-2],[1,-3],[0,-1],[1,-3],[1,-6],[1,-1],[5,-5],[0,1],[2,3],[1,0],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[2,-3],[2,-4],[1,-3],[0,-2],[-1,0],[-8,9],[-8,10],[-2,7],[-2,1],[1,-4],[1,-3],[1,-3],[-1,-1],[-2,3],[-2,6],[0,7],[0,2],[0,6],[0,1],[2,0],[0,2],[0,2],[-1,2],[-2,-1],[0,-1],[-1,-3],[0,-3],[0,-2],[-1,0],[-2,3],[-2,3],[0,1],[-1,0],[0,-2],[0,-1],[1,-3],[1,-1],[1,-3],[-4,2],[1,-4],[1,-2],[1,-1],[0,-3],[-1,-1],[-1,0],[1,-3],[0,-2],[3,0],[0,1],[1,3],[-1,1],[-1,2],[0,2],[3,-2],[1,-3],[2,-5],[-2,0],[0,-1],[-1,-3],[4,-3],[2,-4],[0,-2],[0,-2],[0,-2],[-1,-1],[-2,-5],[-1,-1],[-1,1],[-2,4],[3,-1],[-1,3],[0,2],[0,-2],[1,-1],[1,1],[1,2],[-1,4],[-1,1],[-2,-2],[-2,1],[-2,4],[-4,11],[-2,3],[-1,-6],[3,-4],[1,-5],[-1,-4],[0,1],[-3,6],[-2,10],[-1,2],[-2,3],[-1,3],[1,-11],[4,-8],[1,-8],[-1,-3],[0,2],[-1,3],[-1,2],[0,-1],[-1,-3],[-1,1],[0,2],[0,1],[0,1],[-1,2],[-2,-1],[-1,1],[-3,5],[0,3],[1,-2],[1,-2],[1,-2],[0,1],[0,5],[1,-2],[1,-1],[0,3],[0,6],[-1,6],[0,3],[-2,2],[-1,5],[-1,-1],[0,-2],[0,-2],[1,-1],[1,-3],[1,-5],[1,-8],[-1,7],[-1,4],[-5,5],[-1,3],[-1,5],[0,2],[-1,0],[0,-4],[-1,-4],[0,-4],[1,0],[1,-2],[1,-3],[0,-7],[0,1],[-1,6],[-1,2],[0,-1],[-1,-1],[-2,2],[0,-1],[-1,-2],[0,-3],[0,-4],[1,-7],[0,-4],[0,2],[0,-2],[0,-3],[0,-2],[-2,0],[0,-1],[0,-4],[-1,-3],[-2,5],[-1,6],[-2,5],[0,1],[0,2],[0,1],[-1,1],[-1,0],[0,1],[-1,1],[0,3],[0,1],[-2,1],[-1,2],[-1,2],[0,2],[0,3],[-1,5],[0,2],[0,3],[0,4],[0,2],[-3,3],[-1,6],[-1,2],[0,3],[0,-4],[1,-4],[0,-4],[-2,3],[0,-1],[0,-5],[0,-4],[0,-3],[4,-10],[0,-1],[1,-3],[1,-1],[1,-1],[1,-6],[-4,4],[-2,3],[-1,3],[1,4],[-1,2],[-1,0],[-1,2],[-4,13],[-1,1],[0,-1],[0,-2],[0,-3],[1,-2],[-1,-3],[-1,0],[-1,2],[-1,4],[-1,0],[-1,-3],[-1,1],[-1,3],[-5,15],[0,1],[0,-2],[0,-2],[1,-6],[-1,1],[-1,3],[0,1],[-2,-1],[-1,1],[-1,4],[-1,2],[-1,1],[-1,2],[0,1],[0,2],[-1,-2],[1,-4],[0,-4],[0,-2],[-1,4],[-2,1],[-1,6],[-1,2],[-2,2],[5,2],[1,-1],[0,1],[1,2],[0,1],[1,1],[0,-2],[2,-8],[0,-3],[1,-1],[0,3],[0,3],[-1,4],[0,4],[1,4],[0,5],[1,2],[0,3],[0,5],[-2,9],[-3,10],[-2,7],[0,1],[1,4],[1,-2],[1,-1],[2,-1],[0,4],[0,1],[1,0],[1,-2],[0,-1],[1,0],[1,0],[3,-8],[1,2],[-3,12],[-1,6],[4,2],[2,0],[1,-6],[-2,4],[-1,-1],[-1,-6],[1,-1],[1,1],[3,-1],[4,4],[0,1],[0,2],[0,2],[-1,1],[0,-1],[-1,-2],[-1,-1],[-2,3],[-1,2],[-1,4],[1,0],[0,2],[0,2],[0,2],[1,1],[1,-1],[-1,5],[1,1],[1,-1],[1,1],[-1,0],[-1,5],[-1,1],[-1,-1],[0,-2],[-1,-6],[-1,-3],[0,-3],[-1,0],[-1,3],[0,3],[-1,2],[-1,1],[0,-3],[1,-3],[0,-4],[-1,-3],[0,-1],[-2,4],[-1,1],[1,3],[-1,2],[-1,1],[0,2],[-1,0],[-1,-2],[-1,2],[0,2],[-1,2],[-2,8],[0,2],[4,7],[0,2],[0,5],[2,7],[0,5],[1,7],[1,3],[4,3],[1,3],[1,7],[-1,2],[-1,-2],[-1,-7],[-7,-11],[-1,-3],[0,-4],[0,-11],[-2,-3],[-2,2],[-1,5],[0,1],[-1,6],[0,1],[-1,7],[-1,15],[0,1],[1,2],[0,1],[1,2],[0,1],[0,2],[0,1],[1,5],[2,7],[-2,6],[-1,0],[-4,-15],[-1,-5],[0,-6],[0,-6],[-1,-3],[0,-1],[-1,2],[0,5],[0,3],[-1,5],[0,4],[0,1],[-1,-1],[0,-2],[0,-2],[0,-3],[1,-2],[0,-2],[0,-3],[-1,0],[-1,1],[0,-1],[0,-4],[0,-3],[0,-2],[-1,1],[-1,-4],[-1,0],[0,2],[0,3],[0,3],[0,3],[0,4],[-1,4],[-1,-2],[1,-9],[-2,2],[-1,2],[0,6],[0,5],[-1,0],[-2,-6],[0,1],[0,5],[0,5],[1,4],[0,5],[-1,3],[-1,5],[-1,4],[0,8],[0,2],[-1,-1],[0,-5],[0,-5],[0,-6],[-1,-2],[-2,-2],[-1,-3],[0,-6],[1,-3],[0,-2],[0,-4],[-1,1],[-1,3],[-1,5],[-1,6],[0,9],[0,4],[0,-2],[-2,-6],[0,-2],[-1,-1],[0,-6],[1,-4],[2,-3],[1,-4],[-4,2],[-3,-2],[-2,1],[-2,-3],[-1,-3],[-1,0],[0,1],[0,2],[-1,3],[0,2],[-2,7],[-1,2],[-2,3],[-2,6],[0,1],[0,3],[0,4],[0,2],[0,3],[0,2],[0,2],[0,4],[-2,11],[0,3],[0,1],[0,1],[0,3],[-1,2],[0,1],[0,4],[-1,3],[0,4],[0,6],[0,16],[1,18],[2,15],[2,10],[-1,2],[2,9],[4,8],[2,9],[1,0],[3,-11],[0,-1],[1,-1],[1,0],[0,-5],[1,-3],[0,1],[1,7],[1,4],[1,2],[1,-1],[-1,-4],[0,-2],[0,-2],[0,-3],[1,1],[0,1],[1,0],[1,-4],[1,-1],[1,1],[1,3],[0,2],[-1,3],[1,3],[0,2],[0,2],[-1,2],[0,2],[0,3],[0,2],[1,-1],[2,-3],[0,1],[0,1],[0,3],[-2,12],[1,2],[0,-1],[1,-5],[0,2],[0,4],[0,2],[0,1],[0,3],[0,1],[-1,1],[-1,1],[0,3],[0,2],[1,0],[1,-3],[2,-3],[1,0],[0,1],[0,1],[0,1],[0,3],[2,4],[1,4],[-1,6],[-2,2],[0,5],[2,-5],[2,-2],[4,0],[-1,2],[-2,0],[-1,3],[-1,0],[0,5],[0,3],[0,4],[-1,3],[1,5],[3,0],[1,1],[-2,6],[-1,1],[-1,-2],[0,2],[0,2],[0,2],[1,2],[0,2],[1,0],[0,-2],[2,1],[1,2],[0,5],[0,3],[2,2],[-1,1],[0,3],[1,1],[2,2],[2,2],[0,2],[-5,-1],[-4,-9],[-1,-1],[0,5],[1,4],[0,2],[1,2],[0,1],[0,1],[0,2],[-2,-1],[1,4],[0,2],[6,7],[1,0],[3,-6],[3,-1],[-1,2],[-3,2],[-2,5],[-6,0],[-1,-3],[-5,0],[-1,1],[-1,2],[0,5],[-1,1],[-1,1],[-1,1],[-1,4],[1,5],[1,0],[1,2],[0,2],[0,5],[0,2],[1,3],[1,3],[1,8],[1,2],[11,15],[8,-11],[1,0],[1,2],[-1,2],[-1,3],[0,2],[2,-2],[0,2],[0,1],[1,3],[-1,2],[-4,-1],[-2,3],[0,4],[-1,4],[1,2],[-1,2],[0,2],[1,2],[1,1],[5,-1],[0,1],[-2,2],[0,2],[0,2],[-1,3],[0,2],[0,1],[-2,1],[-1,3],[-1,5],[1,1],[0,1],[0,4],[0,4],[0,1],[0,1],[0,3],[1,2],[4,12],[1,2],[1,3],[1,1],[2,-4],[1,2],[3,9],[2,6],[4,1],[6,-7],[3,0],[3,8],[2,2],[1,2],[1,5],[2,8],[2,3],[3,1],[2,5],[1,0],[2,3],[1,0],[6,-7],[5,-14]],[[6645,8546],[-1,-1],[-1,0],[-2,-3],[-2,0],[-2,-4],[-2,-1],[-2,-3],[-2,-1],[-1,2],[1,3],[7,8],[1,3],[1,2],[2,1],[2,4],[1,0],[0,-1],[1,-5],[-1,-4]],[[6893,8796],[3,-2],[5,2],[1,-3],[0,-3],[-1,-2],[1,-2],[2,0],[1,-1],[1,-5],[1,-5],[0,-1],[2,1],[1,-1],[1,-2],[-1,-3],[-1,-3],[0,-4],[1,-4],[3,-3],[1,-3],[-1,-6],[0,-4],[0,-3],[1,-4],[2,0],[1,-1],[-4,-9],[-1,-5],[0,-2],[0,-4],[0,-2],[-2,-9],[3,-2],[1,-3],[-1,-4],[-4,-10],[-3,-14],[-3,-6],[-1,-1],[-2,0],[0,-2],[-1,-2],[1,-2],[0,-1],[-1,-7],[-1,-5],[-4,-4],[-1,-2],[0,-2],[1,-2],[3,-1],[0,-2],[-1,-3],[-10,-6],[-9,-15],[-2,1],[-4,-8],[-7,-16],[-4,-2],[-5,-8],[-3,-1],[-3,-6],[-2,-3],[-2,-6],[-4,0],[-2,-6],[-1,-3],[-14,-13],[-13,-13],[-3,-9],[-1,-3],[-1,-1],[-4,1],[-8,-10],[-8,-11],[-6,-1],[-1,-1],[-1,-4],[-5,-7],[-3,-7],[-1,-1],[0,3],[0,2],[0,2],[0,3],[0,1],[-1,0],[0,2],[0,4],[-1,4],[-1,-1],[-1,-1],[1,-2],[0,-1],[1,-3],[-1,-2],[-2,-12],[-1,-2],[-1,-1],[-4,-8],[-3,0],[-1,-2],[-3,3],[-1,-1],[-8,-17],[-3,-1],[-10,-18],[-1,-1],[-2,4],[-2,-2],[-1,1],[-1,0],[0,-2],[-1,-3],[-3,-12],[-1,-4],[-1,-2],[-2,6],[-1,-1],[1,-4],[-1,-3],[-1,-5],[1,-3],[0,-1],[-1,-1],[-1,-2],[1,-1],[0,-1],[0,-3],[-1,0],[-1,-1],[-1,-2],[0,-2],[0,-3],[0,-2],[-1,-2],[-1,-1],[-1,0],[-1,3],[-1,5],[0,3],[1,3],[0,2],[0,2],[-1,3],[0,1],[0,2],[0,3],[-3,-2],[1,-6],[0,-6],[-1,-4],[-4,-16],[-2,-2],[-2,-5],[-4,-4],[0,-1],[-1,-7],[0,-1],[-1,-2],[1,-3],[0,-1],[0,-1],[0,-6],[-1,-4],[0,-3],[-1,-2],[-3,1],[-1,1],[0,3],[0,4],[0,4],[1,3],[0,3],[0,3],[-1,1],[-2,-1],[0,-1],[0,-2],[0,-2],[-2,-7],[0,-1],[1,-1],[0,-1],[-1,-2],[1,-3],[-1,-4],[0,-2],[-5,-4],[0,-1],[0,3],[0,1],[-1,3],[-3,-7],[-5,4],[1,-4],[1,-2],[9,-2],[8,-1],[2,-11],[-1,-3],[-2,-9],[0,-2],[-7,-3],[-3,2],[-1,-2],[1,-3],[0,-2],[1,-3],[1,0],[0,-1],[1,-1],[1,-2],[0,-3],[-1,-7],[-1,-4],[-10,-6],[-1,1],[-1,3],[-1,1],[-3,-1],[0,1],[-2,7],[-2,3],[-2,3],[1,-3],[2,-8],[1,-5],[0,-2],[4,0],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[2,-2],[0,-1],[0,-4],[1,-2],[-1,-5],[1,-2],[0,-3],[-1,-1],[-1,-2],[-1,-1],[0,-4],[-1,-2],[-1,-1],[-3,4],[-2,6],[-3,8],[-1,3],[-2,5],[-2,6],[-2,6],[-1,2],[-1,-2],[0,-4],[3,-10],[2,-5],[0,-4],[-1,-1],[-1,2],[-1,3],[0,2],[-1,-1],[-1,-1],[0,-3],[3,-4],[0,-4],[0,-2],[-1,-7],[0,-3],[-1,-4],[0,-2],[0,-4],[-1,-4],[0,-4],[0,-3],[0,-1],[1,-1],[0,-2],[0,-3],[0,-2],[-8,3],[-3,5],[0,1],[0,3],[0,1],[-1,1],[-7,10],[-7,10],[0,-2],[-1,-2],[0,-1],[3,-7],[4,-5],[0,-1],[0,-2],[0,-1],[2,-6],[4,-12],[0,-3],[-2,-6],[2,-6],[1,-6],[0,-6],[0,-1],[1,-1],[1,-2],[-1,-3],[1,-3],[0,-1],[-1,-2],[-1,-2],[1,0],[0,-1],[0,-1],[1,-2],[-6,0],[3,-2],[1,-3],[-4,0],[0,-4],[1,-1],[0,-1],[1,-2],[-1,-2],[-1,-1],[-1,0],[-1,1],[0,2],[-1,5],[-1,0],[-1,-4],[0,1],[0,5],[-1,3],[0,-1],[-1,-3],[-1,-5],[2,-7],[2,-3],[4,0],[-1,-4],[-3,2],[-1,-3],[0,-1],[0,-5],[0,-2],[-1,-2],[0,2],[0,3],[-1,5],[-2,4],[-1,2],[-1,-3],[0,-3],[1,-1],[0,-2],[0,-3],[0,-3],[-2,-8],[2,-1],[1,-2],[0,-4],[0,-3],[0,-2],[0,-1],[-1,-1],[-1,-4],[-1,0],[-2,5],[-2,1],[-4,8],[-1,1],[-1,1],[0,2],[-3,16],[-2,14],[-1,3],[-1,0],[-2,-2],[2,-10],[1,-5],[1,-7],[-2,-5],[-4,4],[-2,-3],[7,-3],[1,-1],[1,-2],[1,-6],[7,-23],[1,-5],[-1,-1],[-1,-1],[-1,-3],[1,-2],[1,-1],[1,-2],[0,-2],[0,-3],[0,-3],[1,0],[-1,-3],[0,-2],[1,-2],[0,-2],[-6,-20],[-1,1],[-2,4],[0,1],[0,5],[-1,2],[-1,5],[-2,6],[-2,6],[-12,5],[-12,5],[3,-6],[8,-2],[4,-6],[3,0],[1,-1],[3,0],[1,-4],[0,-4],[3,-10],[1,-1],[0,-4],[0,-1],[2,-5],[0,-1],[0,-3],[0,-2],[0,-2],[1,-1],[0,-1],[-1,-4],[0,-2],[0,-2],[1,-1],[-1,-2],[0,-2],[0,-2],[-7,-8],[0,-1],[0,-4],[-9,7],[-6,23],[-1,2],[-1,-1],[0,-3],[1,-5],[2,-4],[1,-4],[-1,-2],[-1,0],[-1,0],[1,-3],[2,2],[1,-2],[5,-14],[4,-5],[0,-2],[1,-1],[0,-3],[0,-2],[-1,-2],[0,-2],[1,-3],[-1,0],[-1,-1],[0,-2],[0,-5],[-6,-20],[-2,-7],[-1,1],[0,4],[0,2],[-1,4],[-1,1],[-1,1],[0,-1],[-1,-4],[1,-3],[0,-2],[1,-1],[-2,-7],[0,-3],[0,-2],[0,-2],[1,-1],[0,-2],[-1,-2],[-1,0],[-1,1],[-2,3],[-13,7],[-1,4],[-1,6],[0,10],[-1,10],[-1,7],[-1,2],[-2,-3],[2,-4],[1,-4],[1,-4],[-1,-3],[0,-7],[0,-3],[0,-1],[0,-2],[-1,-2],[-1,1],[-2,6],[-1,1],[-10,-4],[-11,28],[-4,1],[-3,-3],[-3,-7],[-3,0],[-2,-4],[-5,-14],[-2,-1],[-6,11],[1,6],[2,3],[4,1],[1,1],[0,2],[0,1],[0,2],[0,1],[0,1],[1,3],[2,4],[-3,-2],[-1,1],[-3,5],[1,3],[9,26],[2,1],[2,4],[1,3],[2,4],[4,1],[3,9],[2,1],[7,3],[-1,2],[-1,0],[-1,-1],[-8,-1],[-1,-2],[-1,-5],[-1,-1],[-1,2],[1,7],[1,3],[-1,0],[-1,-3],[-1,-1],[-2,0],[-1,-3],[-4,0],[-1,-2],[-1,-4],[-1,-5],[0,-4],[-1,-3],[-2,-2],[-1,-2],[0,-3],[0,-1],[-4,5],[-2,-3],[0,1],[-1,3],[-2,1],[0,1],[-1,3],[1,0],[0,1],[0,2],[-1,1],[-1,1],[0,1],[-1,4],[-2,9],[-1,2],[-2,2],[-1,2],[0,4],[0,2],[0,3],[0,4],[1,4],[0,-1],[2,-6],[1,-1],[1,-1],[2,0],[6,14],[-1,3],[1,4],[0,2],[2,2],[2,9],[2,4],[4,4],[5,1],[3,6],[1,-3],[1,1],[0,4],[-1,3],[-2,2],[-2,-1],[-1,1],[0,3],[0,1],[2,11],[1,2],[0,1],[9,23],[3,4],[1,0],[1,-2],[1,-5],[1,-1],[3,-4],[3,2],[9,-12],[1,0],[0,2],[0,4],[0,2],[-2,4],[-3,1],[-1,3],[-1,6],[-10,14],[-3,-1],[0,1],[-1,1],[0,3],[-1,1],[1,1],[1,0],[0,1],[0,1],[0,1],[0,1],[1,2],[1,2],[0,1],[1,0],[1,-3],[-1,-1],[0,-2],[0,-1],[1,-1],[3,3],[4,-1],[1,2],[2,4],[1,-8],[1,-1],[1,3],[0,4],[-1,2],[-2,6],[0,1],[-1,-1],[0,-1],[-1,0],[-1,5],[-2,2],[-5,0],[-1,1],[0,4],[3,14],[1,3],[3,0],[3,5],[2,-1],[6,0],[1,2],[1,1],[2,-2],[2,3],[4,1],[1,1],[0,2],[0,2],[-1,1],[-1,0],[-1,-2],[-5,-1],[1,2],[0,2],[1,2],[-3,7],[-3,2],[-1,-2],[-3,-6],[-2,-1],[2,6],[0,2],[0,1],[-1,4],[-1,1],[-4,-2],[2,12],[0,2],[-1,0],[0,2],[0,2],[0,2],[0,2],[2,5],[2,1],[2,3],[3,-2],[0,-2],[0,-1],[1,-3],[14,15],[1,0],[0,-4],[3,-5],[12,1],[-1,5],[-1,1],[-2,-1],[-1,-4],[-1,0],[-1,1],[-3,-1],[0,1],[-1,2],[0,5],[-1,3],[0,2],[-1,2],[-1,0],[-1,-1],[-1,1],[1,2],[2,8],[-1,1],[-7,-2],[-8,-2],[-1,1],[-2,3],[0,2],[0,3],[0,3],[2,4],[9,9],[9,10],[1,4],[1,1],[1,0],[1,1],[0,2],[-1,3],[0,1],[-2,-2],[-1,-3],[-1,-1],[-2,2],[-1,-1],[-1,-3],[1,6],[-3,0],[1,1],[2,6],[1,3],[1,2],[0,2],[-1,3],[0,1],[-1,0],[-2,-3],[-4,-1],[-3,-6],[-5,-2],[-1,2],[0,1],[1,5],[0,3],[1,10],[0,3],[-1,0],[-2,0],[-2,-3],[-1,2],[1,1],[0,2],[0,2],[0,1],[0,2],[1,2],[0,1],[0,2],[0,4],[1,1],[0,1],[1,0],[0,1],[1,1],[0,5],[1,2],[3,5],[1,0],[1,-1],[0,-1],[-1,-2],[0,-1],[1,-2],[1,-1],[1,1],[1,2],[0,-4],[-2,-3],[0,-3],[1,-1],[1,0],[2,4],[1,5],[1,0],[0,-1],[1,-2],[1,-14],[1,-8],[-1,-5],[3,3],[2,7],[5,22],[2,6],[1,2],[0,3],[0,4],[0,6],[-2,2],[-2,-1],[-1,3],[1,5],[2,9],[6,10],[1,0],[2,-4],[8,-1],[-1,-3],[1,-2],[1,-1],[2,-4],[5,-2],[0,1],[0,3],[0,2],[0,1],[0,1],[-1,3],[-1,2],[0,7],[0,2],[-1,6],[0,1],[-1,0],[-1,2],[0,3],[-1,3],[0,5],[1,5],[14,15],[2,-1],[1,1],[1,2],[0,4],[-1,4],[0,2],[-6,-1],[-1,1],[1,7],[1,4],[1,2],[7,3],[1,2],[3,2],[1,4],[4,2],[0,2],[-2,4],[1,4],[1,2],[4,2],[0,3],[0,4],[0,3],[5,9],[6,-1],[3,6],[2,0],[0,1],[1,3],[0,1],[0,2],[0,2],[0,1],[1,2],[10,-2],[1,1],[1,4],[2,2],[3,7],[1,2],[8,1],[1,2],[0,3],[-1,1],[-1,2],[-6,1],[0,2],[-1,3],[1,4],[5,10],[8,1],[8,2],[0,-3],[-1,-1],[-1,-1],[-8,-19],[0,-3],[2,2],[0,-2],[2,-4],[2,-1],[1,3],[2,5],[0,2],[0,3],[0,2],[1,1],[2,2],[2,0],[3,4],[1,3],[1,5],[0,3],[0,4],[-1,2],[-5,-3],[0,1],[-1,3],[1,4],[2,7],[0,4],[0,3],[0,4],[0,4],[1,3],[1,1],[3,0],[8,4],[2,-3],[1,1],[1,2],[0,2],[0,2],[2,1],[1,-1],[1,-4],[4,0],[1,1],[1,-9],[3,-2],[8,4],[1,-2],[0,-3],[0,-2],[-1,-3],[0,-2],[0,-6],[1,-2],[2,-1],[1,-2],[0,3],[0,2],[0,3],[1,2],[2,-3],[1,1],[0,5],[0,3],[3,7],[1,0],[1,-2],[1,-4],[-1,-3],[0,-1],[2,0],[0,-3],[1,-3],[6,9],[1,5],[7,6],[1,1],[1,6],[1,2],[2,0],[0,2],[1,2],[1,2],[4,-2],[2,6],[1,0],[0,-2],[1,-2],[1,-6],[0,-2],[1,-1],[1,0],[2,3],[2,8],[2,2],[2,-1],[1,1],[1,2],[0,5],[0,1],[2,1],[3,-6],[1,2],[-1,2],[0,3],[1,3],[0,3],[1,4],[1,1],[2,0],[-1,4],[0,1],[1,4],[5,4],[1,0],[1,-2],[0,-4],[1,-2],[1,0],[1,1],[1,5],[3,5],[1,2],[0,6],[1,3],[2,4],[4,1],[10,-11],[1,0],[1,3],[-1,3],[0,2],[-1,1],[-1,1],[0,1],[0,2],[0,4],[-1,2],[0,6],[-1,4],[-3,5],[0,2],[0,3],[1,2],[1,2],[2,7],[1,3],[5,1],[1,2],[2,8],[1,2],[2,-1],[0,2],[1,3],[0,4],[1,2],[0,1],[1,3],[4,1],[1,3],[0,1],[0,3],[0,1],[1,1],[6,5],[1,6],[1,1],[7,1],[4,8],[2,0],[2,4],[3,3],[12,-4],[4,-8]],[[6632,9522],[1,0],[1,-2],[1,-2],[-2,-4],[-1,-1],[-3,3],[-1,1],[0,3],[-1,4],[-2,2],[-8,0],[0,1],[11,10],[0,-1],[0,-1],[0,-1],[0,-2],[2,-7],[2,-3]],[[6422,9541],[1,-3],[3,-2],[1,-2],[-1,-1],[0,-2],[1,0],[2,-3],[1,-2],[-1,-4],[-1,-1],[-13,5],[-13,5],[-12,5],[0,1],[5,4],[9,-2],[0,2],[2,8],[0,4],[1,2],[5,4],[3,8],[1,0],[1,-3],[1,-4],[0,-3],[5,-1],[0,-2],[-1,-2],[0,-2],[-1,-3],[1,-2],[0,-3],[0,-1]],[[6659,9532],[-6,-5],[-2,2],[-1,0],[0,-3],[-1,-3],[0,-1],[-5,3],[-1,2],[0,1],[1,3],[-2,0],[-4,6],[-6,8],[0,2],[0,4],[1,2],[10,9],[10,10],[2,-3],[5,-4],[1,-1],[0,-2],[1,-3],[0,-2],[0,-2],[0,-2],[0,-2],[0,-1],[-1,-1],[1,-3],[0,-2],[2,-1],[0,-2],[-2,-2],[-3,-7]],[[6672,9577],[-1,-1],[-5,2],[-3,5],[0,2],[0,2],[4,1],[2,-1],[2,-4],[1,-2],[0,-2],[0,-2]],[[6389,9557],[-1,-1],[-3,3],[-3,7],[-1,1],[-3,7],[-2,1],[0,2],[-1,4],[1,4],[8,10],[1,3],[0,2],[10,-2],[2,-8],[1,-3],[-1,-4],[-5,-6],[-2,-8],[1,-2],[-1,0],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[1,-1],[0,-1],[-2,-2]],[[6504,9589],[0,-1],[-1,1],[0,1],[0,1],[0,2],[1,1],[0,1],[0,1],[0,2],[0,2],[0,2],[-1,1],[-1,1],[0,2],[0,3],[0,3],[1,2],[2,3],[1,1],[2,-1],[2,-4],[1,-1],[0,-1],[0,-2],[0,-2],[0,-1],[-1,0],[0,-2],[1,-4],[0,-4],[-1,-2],[-2,-2],[-3,-1],[0,-2],[-1,0]],[[6540,9608],[-1,-2],[-4,-3],[-1,-4],[0,-1],[-5,-2],[-1,1],[-1,3],[0,1],[-1,-1],[0,1],[-1,1],[0,1],[1,0],[-1,1],[0,1],[0,1],[0,1],[1,0],[1,-1],[1,0],[0,1],[3,3],[0,1],[0,1],[-1,0],[-1,2],[0,2],[6,8],[2,0],[1,-1],[0,-1],[3,-2],[0,-1],[1,-2],[-1,-1],[0,-2],[0,-2],[0,-2],[-1,-2]],[[6585,9627],[1,-10],[0,-2],[-1,-6],[0,-4],[0,-4],[0,-5],[1,-2],[1,-1],[0,-2],[0,-3],[-1,-2],[-2,-3],[0,-3],[1,-4],[1,-4],[1,-1],[-1,-5],[-1,-4],[-4,-4],[-9,1],[-9,1],[-10,1],[-5,7],[-1,-1],[8,20],[1,4],[-1,3],[0,4],[0,2],[0,2],[-1,2],[0,2],[0,5],[1,2],[0,3],[0,1],[-1,3],[0,1],[0,3],[1,1],[4,3],[7,-3],[7,-3],[1,4],[0,5],[1,2],[8,-2],[2,-4]],[[6481,9640],[0,-5],[0,-4],[5,-13],[10,-8],[0,-1],[-1,-3],[-1,-5],[-1,-4],[-1,-1],[-1,0],[0,2],[-1,2],[0,2],[-1,1],[-1,-1],[-1,-2],[0,-4],[-1,-2],[0,-5],[0,-3],[-1,-2],[-12,0],[-11,0],[-1,3],[0,4],[-7,2],[-2,4],[-3,13],[1,2],[1,0],[1,2],[8,-1],[0,2],[0,3],[0,1],[3,1],[1,1],[0,2],[-1,1],[0,3],[0,3],[0,2],[2,3],[3,7],[1,1],[1,0],[2,-3],[1,0],[3,3],[4,-1],[1,-2]],[[6535,9646],[0,-1],[0,-2],[-1,-2],[0,-1],[0,-2],[-1,-2],[-2,-2],[-2,1],[1,1],[-1,1],[0,1],[-2,2],[-1,1],[0,1],[1,1],[0,1],[2,1],[1,0],[0,1],[3,4],[1,0],[0,-2],[1,-2]],[[6521,9639],[-7,-5],[-4,2],[0,1],[0,1],[1,0],[0,1],[1,2],[0,2],[1,6],[1,3],[1,4],[1,2],[1,1],[2,0],[0,-1],[1,-1],[2,-5],[0,-2],[1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-2],[-1,-2],[0,-1]],[[6646,9626],[0,-3],[-10,1],[-9,1],[0,-4],[-1,-2],[-4,2],[-1,0],[-2,-4],[-3,-3],[-1,-2],[-1,-1],[-1,-4],[0,-1],[0,-3],[1,-1],[8,-1],[1,-1],[1,-3],[-1,-4],[-1,-5],[1,-2],[-1,-3],[-5,5],[-3,-2],[-1,1],[0,1],[-1,0],[0,1],[0,3],[0,1],[-3,-3],[0,-1],[-1,-2],[0,-3],[0,-2],[1,-2],[1,-2],[0,-3],[0,-2],[0,-2],[1,-2],[-1,-3],[-8,0],[-4,8],[-1,0],[-2,-1],[-1,2],[-1,2],[-2,7],[-1,2],[0,3],[1,4],[0,3],[-2,3],[0,2],[1,3],[0,4],[0,4],[-1,3],[1,5],[0,2],[0,2],[-2,5],[0,3],[2,6],[-1,5],[-1,3],[-7,14],[11,0],[10,-1],[11,-1],[10,-1],[3,-4],[3,-1],[3,-5],[3,0],[9,-12],[0,-1],[1,-4],[1,-4]],[[6528,9655],[-1,-3],[-5,4],[-5,9],[0,3],[0,1],[2,2],[5,2],[5,-7],[0,-1],[0,-2],[0,-2],[0,-2],[-1,-2],[0,-2]],[[6507,9687],[1,-3],[-3,0],[-1,-1],[-1,-4],[1,-2],[7,-2],[-1,-4],[0,-2],[2,-4],[0,-3],[0,-2],[0,-2],[-1,-2],[0,-1],[-1,0],[0,-1],[-1,-2],[0,-3],[0,-2],[-2,-2],[-1,0],[-12,13],[1,3],[0,5],[0,1],[0,1],[1,-1],[3,5],[-1,2],[-3,4],[0,5],[0,7],[1,2],[11,-5]],[[6574,9727],[4,-6],[3,-8],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[-9,-5],[-8,-4],[0,2],[1,1],[-1,4],[-14,-6],[-1,3],[-1,2],[-3,0],[-1,4],[2,5],[1,3],[-6,0],[-2,4],[0,1],[2,2],[1,2],[2,1],[3,-2],[1,2],[0,4],[1,1],[10,3],[11,4],[1,-3],[-1,-3],[1,-2],[3,-1]],[[6329,9742],[5,-2],[6,6],[6,-8],[5,-10],[-1,-3],[-3,1],[-2,-5],[5,-7],[0,-5],[1,0],[3,-5],[0,-3],[0,-3],[0,-2],[0,-2],[-1,-2],[-1,1],[-2,3],[-3,2],[-2,3],[0,1],[-11,3],[-2,4],[1,5],[0,2],[-2,0],[0,1],[-4,13],[-1,1],[-1,0],[-2,-4],[-3,-1],[1,-1],[1,-3],[0,-1],[-2,0],[-1,2],[-1,0],[1,-5],[1,-5],[1,-4],[-2,-2],[-2,3],[-2,1],[-2,-9],[4,-10],[0,-2],[0,-2],[-1,-1],[-6,9],[-1,-3],[-1,-2],[-1,-3],[-1,-2],[0,-2],[-1,-4],[0,-1],[0,-1],[-9,3],[-1,-2],[1,-7],[0,-3],[-1,0],[-1,1],[-2,1],[0,-1],[-1,-3],[1,-6],[0,-2],[-1,-4],[-12,-5],[0,2],[-1,4],[0,6],[0,7],[0,2],[1,5],[0,3],[0,2],[-1,2],[-1,1],[-4,-6],[-2,-1],[0,1],[0,1],[-1,1],[-1,-3],[-1,0],[-1,-4],[-3,-3],[-1,1],[0,2],[-1,4],[-1,2],[-1,4],[-3,4],[-11,4],[-1,3],[1,2],[1,1],[3,1],[2,4],[13,5],[13,6],[14,6],[0,2],[-1,0],[-3,2],[1,1],[0,1],[14,1],[3,3],[2,7],[2,2],[1,3],[-3,3],[-1,2],[2,3],[11,4],[9,-11]],[[6729,9737],[0,-2],[0,-6],[0,-5],[-1,-3],[-2,-3],[-3,-12],[1,-4],[2,-3],[-1,-4],[-1,-2],[-11,-4],[0,-1],[0,-1],[0,-3],[-2,-3],[-1,-2],[1,-7],[-9,-4],[-1,-3],[3,-2],[0,-2],[0,-2],[-1,-5],[-1,-2],[-4,-3],[-2,-6],[-1,-1],[-13,10],[-7,12],[-4,-2],[0,-2],[-1,-3],[0,-1],[-3,-2],[-2,-3],[-8,-4],[-1,1],[-4,13],[-7,11],[0,2],[1,1],[1,1],[1,3],[1,3],[-2,1],[-1,2],[-1,4],[1,5],[-1,6],[-1,2],[1,3],[1,4],[1,2],[-1,3],[1,2],[1,2],[0,1],[0,2],[1,5],[4,4],[1,6],[-2,4],[2,1],[4,7],[2,-2],[1,1],[1,2],[9,3],[1,-1],[1,-5],[-6,-1],[-2,-4],[10,0],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[6,8],[13,5],[14,6],[0,2],[0,1],[0,3],[2,1],[12,-4],[4,-14],[1,-2],[0,-4],[1,-1]],[[6638,9740],[-2,-6],[-1,-2],[-3,2],[-2,-1],[-4,-6],[-4,-1],[-8,8],[-1,2],[-6,0],[-1,2],[-1,3],[1,4],[2,1],[1,2],[2,1],[4,7],[8,4],[8,4],[5,-5],[1,-3],[1,-3],[0,-6],[1,-2],[0,-1],[-1,-1],[0,-3]],[[6537,9754],[9,-7],[9,-7],[-2,-3],[-5,-2],[-12,-8],[-12,-8],[-6,5],[0,4],[0,1],[1,2],[-1,3],[0,2],[-2,0],[-2,-2],[-7,2],[-7,2],[-1,1],[0,4],[0,3],[3,5],[6,4],[3,-4],[8,5],[1,0],[1,-1],[0,-3],[-1,-2],[2,-2],[1,0],[1,3],[2,8],[1,2],[1,2],[2,1],[4,-3],[3,-7]],[[6388,9755],[1,-2],[10,5],[9,4],[3,6],[1,0],[1,-2],[1,-5],[3,-10],[0,-3],[-1,-2],[-6,0],[-3,-4],[-6,-1],[0,-1],[2,-2],[0,-2],[-1,-3],[-5,-3],[-1,-2],[-1,-4],[1,0],[1,-2],[0,-2],[1,-2],[1,-1],[7,13],[1,0],[2,-4],[1,-2],[0,1],[2,6],[1,1],[3,1],[1,0],[0,-3],[1,-3],[1,-1],[2,2],[-1,3],[0,2],[5,3],[0,-2],[0,-1],[0,-2],[2,-2],[2,-5],[0,-1],[6,-2],[1,0],[0,-1],[1,-2],[0,-1],[-1,-1],[0,-2],[0,-1],[-1,-5],[-7,-7],[-7,-7],[-2,-4],[-3,-13],[-1,-3],[-12,-1],[-11,-1],[-1,-3],[-1,-4],[-9,-1],[-2,-4],[1,-2],[2,-2],[1,-3],[0,-6],[0,-1],[1,-2],[0,-1],[-1,-2],[-9,-10],[-9,3],[-9,3],[-1,-5],[0,-1],[0,-3],[-4,-8],[-1,-5],[2,-2],[6,-2],[1,-2],[3,-8],[-1,-2],[-1,-3],[0,-2],[2,-3],[1,-2],[0,-5],[-1,-1],[-1,0],[-1,0],[0,-1],[0,-1],[0,-1],[0,-1],[-7,2],[-4,11],[-2,1],[-1,-4],[2,-5],[0,-3],[-1,-8],[-2,-6],[-2,-2],[-1,2],[-1,3],[-2,5],[-1,1],[-2,0],[-1,-2],[-1,-3],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-1],[-10,-5],[-1,1],[0,1],[0,3],[1,3],[0,4],[2,11],[2,3],[7,3],[1,5],[-5,-1],[-2,2],[1,3],[2,0],[1,3],[-1,3],[-7,0],[-2,-3],[-5,1],[-3,-7],[1,-4],[1,-2],[-5,1],[-3,-5],[-6,5],[0,3],[0,3],[0,3],[0,2],[-7,9],[-1,6],[1,4],[12,14],[1,0],[1,-3],[1,-4],[0,-5],[1,-3],[6,3],[3,-4],[3,3],[10,-2],[-1,3],[4,1],[2,3],[-1,2],[-2,0],[-5,6],[-6,1],[-1,1],[0,2],[-1,5],[8,1],[8,1],[1,3],[-1,2],[-2,0],[-1,3],[1,1],[1,4],[-2,2],[-3,-2],[-4,2],[-11,-8],[-2,3],[1,3],[9,10],[8,9],[4,-3],[1,1],[3,6],[2,2],[1,0],[3,-6],[1,-1],[0,-1],[0,-3],[-1,-2],[2,-3],[3,3],[3,-4],[1,-1],[1,3],[1,4],[1,1],[5,-3],[2,1],[1,4],[-1,3],[-1,2],[1,1],[1,1],[0,1],[1,1],[0,1],[0,1],[2,1],[1,3],[1,5],[-1,0],[-1,0],[-1,3],[-1,4],[4,2],[6,-4],[1,3],[0,1],[-1,3],[1,3],[-1,1],[0,1],[1,8],[-1,3],[-12,-9],[1,5],[1,3],[1,2],[1,0],[-1,2],[-2,0],[-5,-7],[-2,1],[-1,4],[1,3],[3,2],[1,3],[-1,2],[-2,0],[9,9],[9,10],[3,7],[1,3],[12,10],[2,-3],[2,-1],[2,-2],[0,-1],[-9,-4],[-2,-3],[-1,-1],[-1,-2]],[[6626,9777],[-5,-2],[-4,2],[-10,18],[-1,5],[0,2],[1,2],[3,5],[0,3],[1,2],[1,1],[8,-8],[9,-8],[1,-5],[-3,-7],[0,-5],[-1,-5]],[[6702,9782],[-9,-13],[-9,3],[-8,3],[-3,6],[-5,2],[-1,4],[1,3],[4,3],[1,4],[1,3],[6,5],[3,6],[10,1],[10,2],[10,2],[0,-2],[0,-1],[-1,-1],[-1,-2],[0,-3],[0,-1],[0,-2],[0,-3],[0,-3],[-1,-2],[-8,-14]],[[6560,9818],[1,-4],[13,2],[3,-4],[1,-1],[3,-5],[7,-4],[2,2],[11,-10],[0,-1],[0,-4],[1,-1],[2,-7],[7,-6],[7,-5],[-1,-2],[-7,-8],[-3,-7],[-1,-1],[-8,2],[-3,5],[-3,2],[-3,4],[-1,0],[1,3],[0,3],[-1,6],[-1,1],[-4,0],[-1,2],[-1,2],[-1,1],[0,3],[0,3],[0,1],[0,2],[-2,-1],[0,2],[-1,2],[-9,8],[-9,8],[-1,-1],[0,3],[0,1],[0,3],[1,1],[1,0]],[[6542,9790],[1,0],[12,9],[9,-8],[8,-8],[1,-9],[10,-10],[9,-11],[10,-10],[0,-2],[1,-4],[-1,-3],[0,-1],[-2,-1],[-2,-6],[-7,-7],[-7,-7],[-1,2],[-5,15],[-3,5],[-1,1],[0,1],[0,1],[0,2],[-1,2],[-12,6],[-12,5],[0,2],[-1,2],[0,3],[0,2],[-6,2],[-4,8],[0,5],[0,1],[0,2],[0,1],[-2,2],[-3,-5],[-6,-1],[-2,3],[-1,3],[0,2],[0,2],[-1,2],[-8,-1],[-3,0],[-1,3],[0,4],[0,1],[1,5],[1,1],[1,1],[1,2],[-1,2],[0,2],[3,10],[1,1],[1,-2],[4,-8],[13,-9],[6,-10]],[[6410,9801],[0,-2],[-1,-1],[-5,1],[-3,-7],[-1,-1],[-1,2],[0,1],[0,1],[0,2],[1,1],[1,2],[0,2],[0,3],[-3,5],[1,4],[0,1],[0,3],[-1,1],[4,0],[1,1],[0,2],[0,2],[-1,3],[1,2],[7,1],[1,-1],[3,-6],[1,-4],[0,-5],[0,-3],[-1,-1],[-4,0],[-1,-3],[0,-2],[1,-2],[0,-2]],[[6797,9838],[2,-3],[2,0],[10,-14],[1,-3],[3,-15],[1,-4],[1,-2],[0,-4],[-4,-6],[2,-1],[1,-2],[1,-3],[0,-5],[1,-4],[0,-2],[-1,-2],[-8,-17],[-2,-2],[0,-2],[-1,-2],[0,-3],[-1,-1],[-9,-8],[-3,-5],[-2,0],[-2,-4],[-9,-3],[-9,-4],[-3,4],[-7,-6],[-8,-5],[-7,14],[-7,14],[-3,2],[0,1],[0,4],[0,2],[0,2],[7,11],[3,3],[2,3],[2,9],[3,6],[13,2],[12,2],[4,5],[0,3],[0,3],[1,3],[0,2],[0,1],[0,3],[-1,2],[-1,1],[-1,3],[1,3],[0,2],[1,2],[1,2],[1,5],[0,3],[1,1],[4,7],[8,2]],[[6659,9829],[-1,-1],[-9,4],[-10,3],[-1,2],[0,1],[3,2],[9,-3],[9,-2],[0,-2],[0,-1],[0,-2],[0,-1]],[[6602,9822],[-8,-4],[-11,11],[0,3],[2,1],[4,1],[3,4],[2,3],[6,2],[1,2],[4,0],[2,3],[2,0],[2,-3],[2,-4],[-7,-6],[-2,-11],[-2,-2]],[[6679,9849],[-1,-1],[-8,5],[-8,5],[0,2],[0,1],[5,3],[13,-5],[1,-1],[1,-3],[0,-1],[-1,-2],[-2,-3]],[[6577,9883],[-1,-2],[1,-3],[1,1],[2,2],[7,-5],[7,-4],[0,-1],[1,-4],[0,-1],[2,-1],[3,4],[2,0],[6,-8],[-3,-6],[-10,-2],[-11,-3],[-9,-12],[-3,-6],[-5,-1],[-1,1],[0,6],[-2,4],[-2,2],[-12,-10],[-2,3],[-3,0],[-1,1],[-1,2],[0,3],[0,2],[0,2],[0,3],[0,3],[1,2],[0,1],[-5,0],[0,2],[1,1],[1,0],[1,2],[0,2],[-1,2],[-1,3],[3,-2],[4,3],[0,-1],[1,-2],[0,-1],[2,3],[1,0],[1,-1],[0,-1],[0,-2],[2,-2],[3,0],[4,-6],[9,-5],[2,1],[0,2],[-1,0],[-3,5],[-2,7],[0,2],[1,3],[-1,2],[0,1],[-1,1],[0,5],[0,1],[-1,0],[0,2],[11,3],[2,-3]],[[6647,9877],[1,-4],[1,-4],[-1,-6],[-2,-1],[-2,1],[-4,-5],[-11,5],[-2,4],[-1,1],[-1,-1],[-1,0],[-1,3],[0,3],[1,1],[1,0],[4,8],[10,5],[2,-1],[1,-2],[1,-1],[2,-3],[2,-3]],[[6605,9920],[0,-2],[1,-2],[1,-1],[0,-1],[-1,-3],[0,-1],[1,-3],[7,-1],[8,-2],[4,-10],[0,-5],[-1,-2],[-3,-2],[-3,-4],[-7,0],[-7,0],[-4,5],[-2,-1],[0,1],[-1,4],[0,1],[0,1],[-3,3],[-5,-2],[-7,3],[-7,2],[-1,1],[1,3],[1,2],[4,3],[1,2],[1,4],[-1,0],[0,1],[-1,2],[2,4],[13,-1],[1,2],[0,2],[-1,0],[1,3],[7,1],[2,-4],[-1,-3]],[[6727,9925],[-4,-3],[-6,5],[-3,8],[-2,2],[1,1],[6,-1],[3,-1],[2,-3],[1,-1],[1,-2],[1,-3],[0,-2]],[[6629,9934],[-5,-4],[-3,0],[-2,2],[-1,2],[0,1],[1,2],[2,1],[0,3],[1,1],[5,1],[0,1],[1,0],[1,-2],[1,-1],[0,-1],[0,-2],[0,-2],[-1,-2]],[[6743,9962],[12,-2],[13,-1],[4,-9],[-1,-5],[-1,-4],[-1,-2],[-7,-6],[-4,2],[-3,-1],[-7,2],[-7,3],[-2,4],[-3,2],[0,3],[0,2],[-11,3],[0,2],[1,3],[1,1],[1,1],[6,4],[3,-3],[3,4],[4,0],[-1,-2],[0,-1]],[[6638,9970],[-10,-3],[-9,-4],[-9,-5],[-3,4],[1,2],[0,2],[0,1],[0,2],[0,-1],[1,2],[0,2],[0,2],[0,1],[5,4],[-1,2],[-3,0],[-1,4],[1,4],[2,0],[3,3],[10,2],[9,3],[10,2],[3,-4],[2,-1],[1,-5],[-1,-2],[-2,-5],[1,-2],[1,-2],[0,-3],[-1,-2],[-10,-3]],[[6214,4001],[0,-7],[0,-1],[-1,-2],[1,-1],[2,-5],[2,-6],[0,-1],[0,-1],[-1,-2],[0,1],[-1,0],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[-1,-2],[-1,4],[-1,0],[0,-2],[0,-3],[0,-5],[0,-2],[0,-3],[0,-1],[0,-1],[-1,-1],[-2,1],[-1,0],[0,-1],[0,-1],[0,-1],[0,-2],[0,-3],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-2],[0,-3],[0,-1],[0,-1],[0,-2],[0,-4],[0,-3],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[1,0],[2,-2],[0,-1],[0,-1],[0,-3],[1,-1],[0,-2],[1,-1],[0,-1],[0,-2],[0,-1],[0,-2],[-1,-2],[-1,-7],[0,-3],[0,-1],[0,-1],[-1,-1],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-1],[-1,-1],[0,-2],[0,-3],[0,-5],[-1,-2],[-3,-7],[0,-1],[-1,-2],[-2,0],[-1,0],[-1,1],[0,2],[0,1],[-1,0],[0,-2],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[-2,-3],[-1,1],[0,1],[-1,0],[-1,-6],[-1,-1],[-1,1],[-1,-1],[0,-3],[-1,-9],[1,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[-1,-3],[0,-1],[0,-3],[0,-1],[0,-2],[0,-5],[0,-1],[0,-2],[0,-1],[1,-3],[1,0],[0,2],[0,1],[0,-2],[1,0],[0,1],[0,1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-3],[0,-4],[0,-3],[0,-2],[0,-1],[0,-1],[-1,-4],[0,-1],[-1,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[2,-2],[0,-2],[0,-1],[0,-2],[0,-4],[0,-1],[-1,-1],[0,-1],[-2,0],[-1,3],[-2,1],[-1,0],[0,-1],[-1,-1],[0,-2],[1,-1],[0,-3],[0,-2],[-2,-5]],[[6182,3744],[-1,-1],[-2,1],[-3,7],[-2,6],[0,-1],[-1,-2],[0,-1],[0,-2],[-1,-6],[-3,-10],[-1,-2],[0,-2],[0,1],[-1,1],[0,1],[0,1],[-1,0],[-1,1],[0,1],[0,1],[0,4],[0,1],[0,2],[0,3],[0,2],[0,1],[-1,1],[0,3],[0,5],[0,5],[0,3],[0,2],[0,1],[0,1],[0,1],[0,1],[-1,0],[0,-1],[-1,-3],[0,-1],[-1,-1],[-1,-2],[-1,0],[-1,1],[-1,-2],[-2,-4],[0,-1],[-1,0],[0,3],[-1,0],[0,1],[-1,0],[-5,-4],[-4,0],[-2,-3],[-1,1],[-1,1],[0,1],[0,2],[0,2],[0,1],[0,2],[0,1],[-1,-2],[0,1],[0,2],[0,1],[-1,2],[0,-1],[0,-1],[0,-3],[0,-2],[-1,-1],[0,-1],[0,1],[-1,3],[0,1],[0,1],[0,1],[0,1],[0,2],[0,3],[-1,0],[0,1],[0,1],[0,2],[0,3],[-2,2],[-1,2],[0,1],[-1,0],[0,-1],[-2,-1],[-1,-1],[0,-1],[-2,1],[0,1],[0,1],[-1,0],[0,-2],[0,-1],[0,-7],[0,-2],[-1,-1],[0,-1],[0,-1],[-1,0],[0,-1],[0,-1],[0,-1],[-1,1],[0,1],[0,2],[0,-2],[0,-4],[-1,-1],[0,-1],[-1,-1],[0,-1],[0,-2],[-1,0],[0,-2],[0,-1],[-1,-2],[0,-3],[-1,-2],[-1,-4],[0,1],[0,1],[-1,0],[-1,7],[0,1],[0,1],[-2,-1],[-1,1],[-1,0],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,2],[0,2],[0,1],[-1,3],[0,1],[0,1],[0,1],[-2,11],[0,2],[0,2],[0,1],[1,4],[0,2],[0,1],[0,1],[0,2],[0,2],[0,9],[0,1],[0,1],[0,2],[0,6],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[0,2],[-2,2],[-3,8],[-2,1],[-1,-1],[0,-1],[-1,-1],[0,-2],[-1,-2],[0,-5],[0,-1],[0,-1],[-1,-1],[-2,2]],[[6094,3831],[0,3],[0,2],[-1,6],[0,2],[0,1],[0,1],[1,2],[0,3],[0,1],[0,1],[1,1],[0,1],[0,1],[1,1],[0,1],[0,1],[0,1],[-1,3],[0,1],[0,1],[1,1],[0,1],[0,-1],[0,1],[1,2],[0,5],[0,2],[1,0],[1,1],[0,2],[0,1],[-1,3],[0,1],[0,1],[0,2],[-1,1],[0,1],[0,1],[0,2],[0,1],[3,7],[1,6],[1,1],[1,1],[0,1],[1,4],[1,0],[0,-1],[1,2],[0,1],[0,2],[0,1],[0,2],[0,1],[0,1],[0,3],[0,5],[2,1],[1,1],[1,4],[1,3],[1,1],[0,1],[1,-1],[1,-2],[1,-1],[1,2],[0,5],[1,4],[0,2],[0,2],[0,1],[-1,0],[0,1],[0,2],[0,1],[1,0],[1,0],[1,0],[0,2],[1,0],[0,-1],[3,-3]],[[6123,3955],[2,0],[0,1],[1,4],[0,2],[0,3],[1,0],[0,1],[1,0],[0,-1],[1,-2],[0,1],[1,2],[1,1],[4,3],[1,-1],[1,0],[0,1],[2,-1],[0,1],[0,2],[0,1],[1,0],[0,-2],[0,1],[1,6],[1,7],[1,3],[0,1],[0,2],[0,3],[3,-2],[1,0],[0,2],[1,1],[1,0],[0,1],[0,1],[1,0],[0,4],[0,1],[0,1],[1,0],[2,-4],[1,2],[1,10],[0,5],[0,9],[0,4],[0,2],[0,5],[0,4],[0,3],[0,2],[0,1],[-1,6],[0,1],[0,1],[0,6],[0,5],[1,4],[0,2],[1,0],[3,-2],[0,-1],[1,-2],[0,-5],[3,-6],[0,-2],[1,-2],[0,-2],[0,-3],[1,-6],[1,-2],[0,-2],[0,-1],[0,-1],[1,-2],[2,-2],[1,-3],[0,1],[1,0],[0,-1],[0,-1],[0,-1],[0,-1],[4,5],[1,1],[0,2],[1,2],[0,1],[0,5],[0,1],[0,1],[1,1],[1,0],[1,-3],[1,0],[3,3],[0,1],[1,1],[0,6],[0,1],[0,1],[0,2],[1,2],[0,2],[1,1],[0,1],[1,-1],[0,-4],[1,-2],[0,-2],[1,-1],[0,-2],[0,-2],[0,-1],[0,-3],[0,-1],[0,-2],[0,-8],[0,-2],[0,-2],[0,-1],[0,-1],[-1,-3],[0,-2],[0,-1],[0,-3],[0,-1],[0,-3],[0,-2],[1,-1],[0,-1],[0,-3],[-1,-2],[0,-1],[0,-5],[0,-2],[0,-2],[0,-2],[0,-3],[0,-1],[1,-3],[0,-1],[3,-2],[1,1],[1,3],[3,4],[0,2],[0,2],[0,5],[0,2],[1,2],[0,2],[0,1],[0,3],[0,5],[0,4],[0,4],[-1,4],[0,6],[1,3],[1,8],[-1,-10],[0,-5],[0,-5],[0,-1],[0,-1],[0,-2],[1,-3],[0,-8],[0,-2],[-1,-10],[0,-5],[1,-2],[1,2],[1,3],[0,5],[0,11],[1,4],[0,2],[1,1],[1,7],[1,4],[1,8],[0,4],[0,1],[2,2],[1,5],[0,-3],[0,-1],[0,-5],[0,-1],[0,-1],[0,-4],[-1,-4],[0,-2],[0,-1],[2,-5],[0,-1],[0,-1],[0,-2],[1,-2],[0,-1],[0,-1],[1,-3],[0,-1],[1,0],[0,-1],[0,-1],[0,-4],[0,-2]],[[6235,4418],[0,-1],[-1,6],[-1,1],[0,2],[0,2],[1,2],[0,1],[1,-1],[0,-4],[0,-2],[0,-2],[0,-2],[0,-2]],[[6308,4529],[-2,-57],[6,-6],[-1,-27],[1,-6],[3,-17],[1,0],[0,-1],[0,1],[1,3],[0,-5],[0,-3],[1,-2],[0,-1],[0,-2],[2,-8],[1,-4],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-3],[0,-2],[0,-5],[0,-1],[0,-4],[1,-4],[0,-2],[1,-9],[0,-2],[0,-1],[-1,-2],[0,-1],[-1,1],[0,-1],[-2,-2],[0,-1],[0,1],[0,1],[0,2],[0,2],[0,1],[0,1],[0,1],[-1,0],[-1,-3],[-2,2],[-1,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-3],[0,-2],[0,-1],[1,-1],[0,-1],[0,-1],[-1,-3],[0,-2],[0,-2],[-1,-3],[-1,-2],[-1,1],[0,2],[-1,2],[0,1],[-1,0],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-3],[-1,-4],[-1,-4],[0,-3],[0,-2],[0,-10],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[-1,-2],[-1,0],[-1,-1],[-1,0],[-1,-5],[-3,-2],[-1,-2],[0,2],[0,1],[-1,0],[-1,0],[0,-5],[-1,-2],[0,-1],[0,-1],[1,-3],[0,-2],[-1,-1],[0,-3],[0,-1],[0,-2],[0,-1],[-1,-2],[0,-1],[1,-1],[0,-3],[1,-1],[0,-1],[0,-2],[0,-2],[-1,-2],[0,-2],[0,-1],[-1,-1],[0,-2],[0,-2],[0,-1],[-1,-1],[-1,-2],[-1,-3],[0,-2],[0,-1],[0,-1],[0,-1],[1,1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-2],[0,-2],[0,-2],[0,-1],[0,-6],[0,-2],[0,-6],[0,-4],[0,-2],[0,-2],[-1,-1],[0,-3],[-1,-2],[0,-2],[0,-2],[0,-1],[0,-2],[0,-3],[-1,-6],[0,-2],[0,-1],[0,-1],[0,-1],[1,-1],[0,-1],[0,-2],[0,-3]],[[6287,4151],[-5,-2],[-2,0],[-4,-1],[-1,-2],[0,-1],[-1,-1],[-1,1],[-1,0],[0,-1],[0,-1],[0,-1],[-1,-1],[-1,-2],[0,-1],[-1,-2],[0,-1],[0,-5],[0,-3],[0,-1],[0,-2],[-3,-6],[0,-1],[-1,0],[0,3],[0,1],[-1,2],[0,2],[0,1],[0,1],[0,1],[0,2],[-1,2],[0,1],[0,4],[0,3],[-1,1],[-4,2],[-1,0],[0,2],[0,1],[-1,1],[-1,1],[0,1],[0,2],[-1,1],[-4,3],[-1,-1],[0,-1],[0,-2],[-2,1],[-1,-1],[-1,-2],[0,-5],[0,-2],[-1,-3],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-3],[0,-4],[0,-2],[0,-1],[0,-4],[0,-2],[1,-10],[1,-3],[0,-1],[-1,-12],[-1,-9],[-1,-13],[-1,-2],[0,-5],[0,-1],[-2,0],[-1,0],[-1,0],[-1,-2],[-2,-10],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-2],[-1,-1],[0,-2],[0,-1],[0,-2],[-1,-3],[0,-1],[-2,-8],[-1,-4],[-2,-4],[-1,-2],[0,-2],[-2,-3],[-2,-5],[-2,-3],[-4,-2],[-3,8]],[[6123,3955],[0,9],[-1,7],[0,3],[0,2],[1,6],[0,1],[-1,2],[0,1],[0,2],[0,2],[0,1],[1,0],[0,1],[0,2],[0,1],[0,2],[-2,9],[0,1],[0,2],[0,1],[0,2],[1,0],[0,1],[2,-4],[0,1],[0,1],[0,1],[-1,2],[0,1],[0,1],[0,1],[0,2],[0,1],[0,3],[0,1],[0,1],[1,4],[0,1],[1,0],[0,-1],[0,-2],[1,0],[0,1],[0,1],[1,0],[0,-1],[0,-1],[0,-3],[0,-2],[0,-1],[0,-1],[0,-1],[1,0],[0,2],[0,1],[1,4],[0,2],[0,2],[0,1],[0,1],[0,1],[0,1],[0,2],[-1,3],[0,9],[0,5],[0,2],[0,21],[0,15],[0,3],[0,1],[0,2],[-1,9],[0,5],[0,2],[0,2],[1,1],[0,2],[1,5],[1,6],[0,3],[0,1],[0,2],[1,10],[0,3],[1,1],[0,1],[1,1],[0,1],[0,1],[0,2],[1,4],[0,2],[0,1],[1,1],[0,2],[0,1],[0,2],[0,3],[0,1],[0,2],[0,2],[0,3],[0,1],[0,1],[1,1],[1,2],[0,1],[0,1],[0,2],[0,2],[0,2],[0,3],[0,2],[1,1],[0,4],[1,3],[1,2],[1,1],[2,10],[0,2],[0,1],[-2,7],[-1,2],[0,2],[0,2],[0,2],[0,3],[0,5],[0,2],[0,2],[0,1],[0,1],[0,1],[-1,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,2],[-1,0],[0,1],[-1,1],[0,1],[0,2],[-1,4],[0,1]],[[6136,4275],[0,4],[3,6],[0,2],[0,1],[1,5],[3,9],[0,2],[0,1],[0,1],[1,4],[0,2],[0,1],[0,2],[-2,11],[0,2],[0,1],[0,2],[1,2],[0,1],[0,1],[0,1],[0,4],[0,2],[1,1],[1,1],[0,1],[0,2],[0,1],[0,4],[0,1],[0,1],[0,1],[0,1],[-1,-1],[0,-1],[-1,0],[0,1],[0,3],[0,2],[0,2],[0,5],[0,2],[1,1],[0,1],[2,4],[1,-1],[0,-2],[0,-1],[0,-1],[0,-1],[1,-1],[1,6],[2,4],[1,5],[1,5],[3,7],[1,3],[1,1],[0,4],[1,4],[0,1],[1,-4],[1,-4],[1,0],[1,0],[0,1],[0,2],[1,1],[0,2],[1,1],[1,0],[0,1],[1,0],[0,2],[0,1],[0,2],[0,1],[1,3],[0,1],[0,1],[0,2],[0,3],[1,11],[-1,2],[0,1],[-1,1],[-1,2],[-1,3],[0,1],[0,2],[0,1],[0,2],[0,1],[0,1],[0,1],[1,1],[0,1],[0,2],[0,12],[0,2],[0,1],[0,3],[1,7],[0,1],[0,1],[1,-1],[0,-1],[1,-4],[1,0],[1,-1],[0,-1],[0,-1],[0,-2],[0,-3],[0,-3],[0,-2],[0,-2],[0,-2],[0,-4],[0,-2],[1,-2],[1,-4],[0,-3],[1,-2],[0,-1],[1,2],[0,1],[0,1],[0,3],[0,1],[1,1],[4,1],[1,3],[0,1],[1,0],[1,-1],[0,-1],[1,-1],[0,-3],[0,-1],[0,-2],[0,-1],[1,-1],[0,-1],[0,-2],[-1,0],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[1,-2],[0,-1],[1,-1],[1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[1,-3],[0,-4],[1,0],[0,1],[0,1],[1,-1],[0,-3],[1,4],[0,4],[0,1],[1,-1],[1,3],[1,0],[1,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,2],[1,2],[0,2],[0,1],[1,0],[0,1],[0,1],[0,1],[0,1],[0,2],[0,2],[0,-2],[0,-1],[1,-1],[0,-2],[0,-1],[2,-5],[1,-2],[1,-1],[0,-2],[0,-1],[0,-1],[1,0],[1,0],[0,1],[0,1],[0,2],[0,1],[-1,4],[0,1],[0,2],[0,1],[0,1],[1,2],[2,2],[1,1],[0,1],[1,1],[0,-1],[0,-1],[0,-2],[0,-5],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-3],[0,-3],[0,-2],[0,-1],[0,-2],[0,-2],[2,5],[2,13],[1,0],[0,3],[1,1],[1,0],[0,-1],[0,-2],[0,-1],[1,-1],[1,0],[2,-6],[0,-1],[1,-1],[1,0],[0,1],[1,1],[2,4],[1,0],[1,-1],[0,-1],[1,-4],[0,-6],[0,-1],[0,-1],[2,-1],[0,-1],[1,-2],[0,-2],[0,-2],[1,-1],[0,1],[1,2],[0,1],[8,-4],[9,-4],[1,1],[0,3],[1,7],[0,2],[0,1],[0,1],[1,1],[6,-2],[0,1],[1,1],[0,2],[0,1],[1,0],[2,-2],[0,-1],[0,-2],[1,-1],[0,2],[1,1],[0,1],[3,-1],[1,1],[0,2],[0,1],[1,1],[0,-1],[0,-2],[0,-1],[2,-1],[4,0],[1,2],[0,2],[0,4],[0,2],[0,5],[1,5],[0,4],[1,3],[0,2],[0,1],[1,1],[2,0],[0,1],[0,1],[0,2],[0,2],[1,1],[0,1],[4,-1],[1,2],[1,26],[-1,7],[0,2],[0,1],[0,1],[0,7],[1,3],[0,20],[0,3],[10,-6],[1,1],[0,1],[0,1],[1,1],[0,2],[1,0],[7,-5]],[[6287,4151],[0,-2],[0,-1],[1,-2],[0,-3],[0,-1],[0,-2],[1,-1],[0,-1],[0,-1],[1,1],[1,1],[0,2],[1,0],[2,-2],[11,0],[10,0],[0,-1],[1,-1],[0,-3],[1,-2],[0,-1],[0,-2],[2,-1],[2,-3],[1,-2],[0,-1],[1,0],[1,1],[0,-1],[1,-5],[1,-3],[0,-1],[0,-2],[0,-3],[0,-2],[0,-4],[0,-1],[-1,0],[-1,-1],[-1,-4],[-1,-3],[0,-1],[-1,-1],[0,-3],[0,-2],[0,-2],[0,-2],[-1,-6],[1,-2],[0,-2],[0,-1],[0,-1],[0,-2],[-1,-4],[0,-2],[0,-1],[0,-2],[0,-3],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-10],[0,-6],[-1,-1],[0,-1],[0,1],[0,-1],[-1,-2],[0,-11],[-2,1],[-1,3],[0,1],[0,1],[0,1],[-1,0],[-1,2],[-1,1],[0,-1],[-1,-1],[0,-1],[0,-2],[-2,-5],[-3,-8],[0,-1],[-1,0],[-3,5],[0,1],[-1,-2],[-1,0],[0,1],[-1,3],[0,1],[-1,-1],[0,-1],[0,-1],[0,-5],[1,0],[0,-1],[0,-3],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-2],[0,-1],[-1,-1],[0,-1],[0,-2],[-1,-3],[1,-2],[0,1],[1,0],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-1],[-1,-1],[0,-1],[0,-2],[0,-1],[0,-7],[2,-6],[1,-1],[0,-1],[0,-2],[0,-7],[1,-3],[0,-1],[0,-1],[1,-1],[-1,-7],[-2,1],[0,-1],[-1,-2],[1,-5],[0,-3],[0,-1],[0,-1],[0,-2],[0,-3],[0,-14],[0,-3],[0,-5],[0,-2],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[1,-3],[0,-1],[0,-1],[0,-1],[-1,-1]],[[6297,3878],[-1,1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-2],[-1,-2],[0,-1],[-1,1],[0,1],[-1,-2],[0,1],[0,3],[-1,1],[-5,3],[0,-1],[0,-1],[0,-9],[0,-2],[-1,1],[-1,0],[-1,-3],[-3,-2],[-2,-3],[-1,1],[-1,0],[0,-1],[0,-1],[0,-2],[-1,-3],[0,-2],[0,-4],[0,-1],[0,-2],[0,-2],[0,-2],[0,-3],[0,-2],[0,-2],[0,-4],[0,-3],[0,-2],[0,-1],[-1,-3],[0,-2],[0,-4],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-2],[0,-1],[-1,-2],[-1,-1],[-2,0],[0,-1],[-1,-1],[0,-2],[0,-1],[0,-1],[-1,-1],[0,-1],[-1,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-1],[0,-2],[0,-2],[0,-2],[0,-2],[0,-5],[1,-3],[0,-2],[0,-1],[0,-2],[1,0],[0,1],[2,-1],[0,-1],[0,-2],[0,-3],[0,-1],[0,-1],[0,-1],[3,4],[0,-1],[1,0],[0,1],[1,0],[0,-2],[0,-2],[0,-2],[0,-2],[-1,-3],[0,-2],[0,-2],[0,-2],[0,-3],[1,-5],[0,-3],[0,-2],[-1,0],[-1,-2],[0,-1],[0,-1],[0,-2],[0,-7],[0,-2],[0,-2],[1,-4],[2,-10],[0,-1],[1,-2],[0,1],[1,-1],[0,1],[0,-6],[0,-6],[0,-2],[0,-4],[0,-2],[0,-1],[1,0],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,1],[1,0],[0,-3],[0,-2],[2,2]],[[6283,3653],[0,-5],[0,-1],[0,-2],[-1,-2],[0,-3],[0,-3],[-1,-3],[0,-4],[-1,-2],[0,-2],[0,-2],[0,-2],[0,-3],[1,-1],[0,-2],[0,-1],[0,-1],[-1,-5],[0,-1],[0,-2],[0,-6],[0,-3],[0,-3],[-1,-2],[0,-2],[-1,-4],[-1,0],[0,-2],[0,-5],[0,-1],[0,-2],[0,1],[-1,1],[0,-2],[0,-1],[0,-2],[0,-2],[-1,-5],[0,-2],[0,-3],[1,-10],[0,-4],[1,0],[0,1],[0,1],[1,0],[0,-2],[1,0],[0,-2],[0,-6],[1,-3],[0,-2],[0,-3],[0,-2],[0,-1],[0,-3],[1,-1],[2,-2],[1,1],[0,1],[0,-2],[1,-3],[0,-1],[1,1],[0,-1],[0,-4],[1,-3],[0,-1],[1,0],[0,-1],[0,-3],[0,-3],[0,-1],[0,-1],[-1,-5],[0,-1],[0,-1],[-2,-1],[-1,-3],[-1,-7],[0,-9],[0,-7],[0,-3],[1,-2],[-1,-2],[0,-2],[0,-1],[0,-1],[0,-1],[-1,0],[0,-1],[0,-2],[0,-1],[-1,-2],[-1,0],[0,-1],[0,-1],[0,-2],[-1,-14],[0,-2],[0,-4],[0,-5],[0,-2],[0,-2],[0,-2],[0,-2],[0,-1],[0,-6]],[[6279,3405],[-1,-4],[0,-1],[-2,-6],[-1,-1],[-1,1],[0,1],[0,1],[-1,0],[-1,-1],[-2,-5],[0,-1],[0,-1],[-1,0],[0,1],[0,1],[-1,1],[0,1],[0,2],[1,0],[0,1],[0,3],[2,6],[0,2],[0,1],[-1,3],[0,2],[0,2],[0,2],[0,3],[0,1],[0,1],[1,0],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[-1,1],[0,1],[0,1],[0,2],[0,1],[0,2],[0,1],[-1,1],[-2,-2],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-3],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-4],[0,-1],[0,-1],[-1,1],[-1,1],[0,-3],[-1,-8],[0,-4],[0,-1],[-1,-2],[1,-2],[-1,-2],[0,-6],[0,-3],[0,-1],[0,-2],[0,-2],[0,-4],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-3],[0,-2],[-1,2],[-1,-1],[-1,0],[-1,-1],[0,-1],[-1,-1],[0,-3],[0,-9],[-1,-11],[0,-4],[0,-1],[-1,0],[0,1],[0,1],[0,2],[0,3],[0,2],[0,1],[0,1],[-1,0],[0,-1],[-1,-1],[0,-3],[0,-1],[-1,-4],[0,-1],[0,-1],[0,-2],[0,-6],[0,-3],[0,-2],[0,-2],[0,-2],[1,-3],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-1],[1,-1],[1,-1],[0,-1],[0,-3],[1,-3],[0,-3],[0,-2],[0,-2],[0,-2],[-1,-1],[0,-1],[-2,1],[-2,-3],[0,-1],[-1,0],[0,1],[-2,2],[0,1],[0,2],[-1,3],[0,1],[0,1],[-1,0],[-1,-1],[-3,-11],[-1,0],[0,2],[0,1],[0,1],[1,2],[1,5],[0,2],[0,1],[0,1],[0,1],[0,2],[0,1],[0,2],[0,1],[0,2],[0,2],[-1,1],[-1,0],[0,-1],[0,-2],[-1,-7],[0,-1],[0,-1],[0,-1],[-3,2],[-1,2],[0,4],[-1,4],[0,2],[0,1],[-1,2],[-1,2],[-2,1],[-1,3],[-2,3],[0,3],[-1,1],[0,1],[0,2],[0,3],[0,2],[0,2],[0,1],[1,3],[0,2],[0,1],[0,2],[-1,1],[-1,0],[-1,-2],[-2,3],[0,2],[0,1],[0,2],[0,2],[0,3],[1,5],[-1,1],[0,2],[-1,0],[0,1],[0,-1],[-1,-1],[-2,3],[-2,0],[0,1],[-1,2],[0,3],[-1,7],[-1,6],[0,2],[-1,1],[-2,3],[-1,1],[-1,2],[0,2],[0,1],[-1,1],[0,1],[-1,0],[-3,-4],[-1,-1],[-1,-4],[-1,-3],[0,-1],[0,-1],[0,-5],[-1,-11],[0,-3],[0,-5],[-1,-1],[0,-1],[0,-1],[-1,0],[0,2],[-1,3],[-1,0],[-4,-2],[-2,-2],[-1,0],[0,2],[0,2],[0,2],[0,1],[-1,0],[0,-1],[-1,-5],[0,-1],[-1,-1],[-1,1],[-1,3],[0,2],[0,2],[0,5],[-2,8]],[[6179,3363],[0,10],[0,1],[0,2],[0,2],[0,2],[-1,3],[0,1],[0,1],[0,1],[1,2],[0,1],[0,5],[0,1],[0,1],[0,1],[0,1],[-1,0],[-1,-2],[-2,-1],[0,-1],[-1,-1],[-1,-1],[0,-2],[-1,-5],[0,-1],[-1,-1],[0,-1],[-2,4],[-1,2],[-1,3],[-1,1],[0,1],[-1,2],[0,2],[0,1],[1,2],[0,1],[0,1],[-1,3],[0,1],[0,1],[1,3],[0,2],[0,2],[0,1],[0,1],[0,1],[-1,0],[0,-1],[-2,-6],[0,-1],[-1,1],[-1,3],[-1,1],[0,2],[0,7],[0,1]],[[6160,3424],[1,2],[0,3],[1,3],[0,2],[1,1],[0,2],[0,5],[0,2],[0,1],[0,1],[0,1],[-1,5],[0,2],[1,2],[0,2],[1,4],[1,1],[0,3],[0,2],[0,2],[0,1],[-1,4],[0,1],[0,1],[1,1],[1,0],[0,3],[0,6],[1,2],[1,3],[1,2],[0,5],[1,5],[0,4],[-1,1],[0,1],[-1,1],[0,2],[0,2],[0,2],[0,6],[0,2],[0,2],[0,3],[0,2],[-1,3],[0,1],[0,6],[0,3],[1,1],[1,-1],[0,1],[0,4],[0,1],[0,2],[0,5],[0,2],[1,1],[0,2],[1,5],[0,2],[1,1],[0,1],[1,9],[0,2],[1,1],[3,2],[0,1],[1,10],[0,2],[1,3],[1,2],[0,4],[-1,3],[0,4],[1,2],[0,1],[2,15],[0,2],[1,0],[0,-1],[0,-1],[0,-2],[0,-1],[1,-1],[0,1],[0,2],[0,4],[1,4],[0,2],[0,4],[0,2],[0,1],[0,2],[0,2],[0,3],[0,8],[0,4],[0,3],[0,2],[1,2],[0,1],[2,1],[2,0],[1,1],[1,4],[0,4],[0,2],[0,1],[-3,0],[-1,2],[0,1],[-1,3],[0,3],[0,3],[0,8],[0,3],[0,2],[0,2],[-1,9],[0,4],[-1,3],[-2,11],[-1,1]],[[6045,4253],[0,-6],[0,-1],[0,-2],[0,-1],[0,-5],[-1,-3],[-1,-1],[0,-2],[-1,-1],[0,-1],[-1,0],[0,1],[0,1],[0,2],[-1,1],[0,-2],[-3,-14],[2,-4],[0,-2],[0,-3],[0,-2],[0,-1],[-1,-1],[0,-1],[0,-2],[0,-4],[0,-2],[0,-1],[-1,0],[0,-1],[0,-1],[0,-1],[0,-2],[0,-7],[1,-2],[0,-2],[1,-2],[1,0],[0,1],[1,0],[0,-1],[2,-4],[0,-1],[0,-4],[0,-2],[1,-1],[1,-5],[0,-2],[0,-3],[1,-3],[0,-2],[0,-1],[0,-1],[-2,1],[-2,-4],[0,-1],[0,-1],[0,-3],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[-1,-2],[0,-1],[-2,1],[-2,-2],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[1,-1],[1,-1],[1,-4],[1,-2],[0,-2],[0,-2],[0,-1],[0,-1],[-1,0],[-1,1],[-1,2],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-3],[1,-1],[0,1],[0,-1],[1,0],[0,-2],[0,-2],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,0],[0,1],[1,0],[0,-1],[2,-14],[0,-2],[0,-3],[0,-1],[0,-2],[0,-1],[1,-1],[1,-2],[1,-4],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[1,-2],[1,-2],[1,-5],[0,-2],[1,-1],[1,2],[0,-1],[0,-3],[0,-3],[0,-2],[0,-2],[0,-5],[0,-3],[0,-2],[0,-6],[-1,-2],[0,-2],[0,-1],[1,-5],[1,-3],[0,-2],[-1,-1],[0,-1],[0,-1],[-1,-2],[0,-2],[0,-1],[1,0],[1,-4],[4,-14],[0,-1],[1,1],[0,-1],[1,-2],[0,-1],[2,-1],[0,1],[0,2],[0,1],[1,1],[1,0],[0,1],[1,1],[0,1],[1,0],[0,-1],[0,-2],[0,-5],[0,-9],[1,-4],[0,-1],[-1,-1],[0,-1],[-1,0],[0,-1],[0,-2],[0,-4],[-1,-1],[0,-1],[-2,-1],[0,-1],[0,-3],[0,-4],[0,-1],[0,-1],[0,-2],[1,-2],[0,-4],[1,-4],[1,-1],[0,-1],[0,-1],[0,-1],[0,-5],[0,-2],[0,-1],[0,-1],[0,-3],[0,-4],[0,-2],[0,-2],[0,-1],[0,-3],[-1,-2],[0,-2],[0,-1],[0,-2],[-1,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[1,-2],[0,-1],[1,0],[0,-1],[0,-2],[0,-1],[0,-4],[0,-1],[0,-1],[0,-4],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[-1,-1],[-1,0],[0,-1],[0,-1],[0,-1],[1,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[2,-3],[0,-1],[1,-1],[0,-5],[0,-1],[0,-1],[0,-3]],[[6062,3830],[-6,1],[-1,0],[0,1],[0,1],[0,1],[0,2],[0,1],[0,3],[0,1],[0,1],[-1,4],[0,1],[0,1],[0,1],[0,1],[0,1],[-1,5],[0,1],[-1,-1],[0,-1],[-1,1],[0,2],[0,2],[0,1],[0,1],[-1,2],[-2,2],[0,2],[0,7],[-1,0],[-2,-2],[-1,-4],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,0],[0,1],[-1,-2],[0,-1],[0,-1],[1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-6],[0,-3],[-1,-2],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[-2,1],[-1,-1],[-1,-1],[0,-2],[-2,-1],[0,-2],[-1,1],[0,2],[0,1],[0,1],[0,-1],[-2,-6],[-1,1],[0,-5],[0,-2],[-1,-2],[0,-2],[1,-1],[1,-5],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[1,-1],[-1,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-3],[-1,-1],[0,-1],[0,-1],[-1,0],[0,-1],[0,-2],[0,-2],[-1,-1],[0,1],[0,2],[0,1],[-1,0],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[-4,6],[-1,0],[0,-1],[0,-3],[0,-1],[0,-1],[-1,0],[-1,-2],[-2,-4],[-2,-10],[-1,-1],[-1,-2],[0,-1],[0,-3],[0,-6],[0,-1],[-1,-1],[-1,1],[0,-1],[-1,0],[-1,2],[-1,0],[-1,0],[-3,-8],[-1,0],[0,-1],[0,-3],[-1,-2],[0,-1],[0,-1],[0,-1],[-1,-3],[0,-1],[0,-1],[0,-1],[0,-1],[-1,5],[-2,6],[-1,0],[-1,-2],[0,1],[-1,1],[0,1],[0,2],[0,1],[-1,0],[-1,-1],[-1,-2],[0,1],[-1,1],[0,1],[0,1],[0,1],[0,3],[-1,5],[0,2],[0,1],[-1,2],[0,2],[-4,-2],[-1,0],[0,-1],[-1,0],[0,-3],[1,-3],[0,-1],[0,-2],[0,-6],[1,-2],[0,-1],[0,1],[0,2],[0,-1],[1,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,1],[-1,0],[0,2],[0,-2],[-1,-1],[0,-1],[1,-1],[0,-1],[0,-2],[0,-1],[0,-9],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[-1,0],[-1,1],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-2],[0,-1],[0,-2],[0,-1],[0,-2],[-1,-3],[0,-1],[0,-1],[0,-1],[0,-1],[1,-4],[0,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-1],[-1,0],[-1,0],[0,-2],[-1,-3],[0,-2],[-1,-2],[0,-11],[-1,-3],[0,-3]],[[6004,4262],[1,-3],[1,0],[0,1],[0,3],[0,1],[0,1],[1,0],[0,-1],[0,-1],[1,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,0],[0,2],[3,6],[1,0],[0,1],[0,2],[1,4],[0,2],[0,2],[-1,1],[1,2],[0,1],[2,3],[1,10],[1,2],[0,1],[0,7],[0,1],[0,1],[1,0],[0,1],[0,1],[0,1],[0,1],[0,2],[0,2],[0,1],[0,1],[1,-2],[1,-1],[0,1],[1,1],[0,3],[0,2],[0,2],[-1,7],[0,4],[0,1],[0,1],[1,2],[0,3],[1,1],[0,1],[0,1],[1,-1],[1,-1],[1,0],[1,0],[0,1],[2,6],[0,1],[1,0],[1,0],[0,-4],[1,-3],[0,-3],[0,-3],[0,-2],[0,-1],[0,-6],[0,-2],[3,-17],[1,-4],[1,-10],[1,-2],[0,-4],[1,-4],[1,-2],[1,-4],[0,-1],[1,-7],[0,-2],[2,-6],[1,-6]],[[6306,4834],[0,-1],[1,-2],[0,-1],[0,-3],[0,-1],[-1,-18],[-1,-8],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[1,0],[1,-1],[0,-4],[0,-2],[0,-4],[0,-5],[0,-1],[-2,-4],[0,-1],[0,-1],[0,-2],[-1,-1],[-1,-5],[-1,0],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[1,-1],[0,-1],[-1,-2],[0,-3],[0,-6],[0,-3],[-1,-5],[0,-2],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-5],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[-2,-14],[0,-1],[-1,-1],[0,1],[-2,3],[0,1],[-1,-1],[-1,-2],[-1,-1],[-1,1],[-2,2],[0,-1],[-1,-3],[-1,-1],[-1,0],[0,1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[1,-2],[0,-1],[0,-1],[0,-2],[0,-5],[0,-3],[0,-3],[0,-2],[1,-1],[0,-1],[0,-2],[-1,-2],[0,-1],[1,-1],[0,-2],[1,0],[1,1],[1,-2],[3,-8],[0,-1],[1,1],[0,1],[0,1],[1,-1],[0,-1],[0,-1],[0,-1],[1,-1],[0,1],[1,0],[0,2],[0,1],[0,1],[0,-1],[1,-1],[1,0],[0,1],[1,1],[0,1],[0,2],[0,1],[1,2],[0,1],[0,1],[0,2],[0,1],[4,-1],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-25],[-2,-26],[0,-1],[0,-11],[0,-6],[0,-4],[0,-3],[1,-2],[3,-3],[0,-2],[1,-2],[0,-5],[1,-2],[0,-4],[0,-3],[-1,-21]],[[6136,4275],[-1,1],[-2,-1],[-1,2],[-1,0],[-1,0],[-1,0],[0,1],[0,2],[-1,1],[0,1],[0,-1],[-1,-2],[-1,-1],[0,-1],[0,-3],[-1,-2],[-1,-3],[-1,1],[-1,-2],[-1,0],[-1,1],[-1,-1],[0,-2],[-3,2],[-1,1],[0,1],[0,1],[0,2],[0,1],[-1,1],[0,3],[-1,2],[0,1],[0,1],[0,2],[0,10],[0,1],[0,2],[0,1],[-1,0],[-1,1],[0,1],[0,1],[0,1],[0,7],[-1,2],[0,5],[0,1],[0,2],[-1,1],[-2,0],[-1,0],[0,1],[0,2],[0,1],[0,1],[0,3],[0,1],[-1,2],[0,1],[0,2],[0,1],[0,1],[-1,1],[-2,4],[-2,12],[-1,2],[-1,-6],[-1,-1],[0,1],[0,2],[-4,4],[0,1],[-1,3],[-1,1],[0,1],[-1,-1],[-1,-2],[-2,5],[-1,4],[-1,-3],[-1,-1],[-1,0],[-1,0],[0,1],[-1,1],[0,-2],[0,-3],[0,-3],[0,-2],[-1,-2],[0,-2],[0,-1],[1,-1],[0,-1],[1,0],[0,-1],[1,-5],[0,-1],[-3,-12],[0,-1],[0,-2],[0,-1],[0,-5],[0,-2],[0,-2],[0,-1],[0,-1],[-2,0],[0,1],[0,2],[0,1],[-3,0],[-1,-1],[-1,0],[0,1],[0,1],[0,1],[0,1],[-1,3],[-1,0],[-1,0],[0,-1],[0,-2],[0,-1],[0,-5],[-2,-9],[-12,8],[-1,-5],[0,-3],[0,-3],[-1,-6],[-1,-7],[-1,-4],[0,-2],[-1,1],[0,2],[0,1],[-1,1],[-2,-1],[-1,-1],[0,-2],[0,-1],[0,-2],[-1,-16],[0,-5],[-1,-6],[0,-1],[0,-1]],[[6094,3831],[-1,0],[-1,-13],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-2],[0,-3],[0,-1],[0,-2],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-5],[0,-2],[-1,-12],[0,-2],[-1,-2],[-2,5],[0,1],[0,1],[0,1],[0,1],[0,1],[-1,1],[-1,3],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[-1,-3],[-1,-2],[0,-2],[-2,3],[-1,0],[0,3],[0,2],[0,1],[-1,0],[0,2],[0,1],[0,1],[-1,1],[-2,3],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,0],[-3,-1],[0,-1],[0,-3],[0,-1],[-1,-5],[-1,0],[0,1],[0,3],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,3],[0,1],[0,2],[0,1],[0,2],[0,1],[-1,4],[-4,13]],[[6065,3812],[-1,3],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[-1,2],[0,2],[0,3],[-1,1]],[[6025,3451],[0,-1],[1,-6],[0,-1],[0,-4],[0,-1],[0,-1],[1,-2],[1,-8],[0,-4],[0,-1],[1,-1],[0,-1],[0,-1],[1,0],[-1,-8],[0,-4],[1,-1],[0,-2],[1,-11]],[[6031,3393],[0,-9],[0,-2],[0,-1],[0,-1],[0,-1],[1,-2],[1,-5],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[-1,-1],[0,-1],[1,-2],[1,-5]],[[6034,3355],[0,-2],[0,-3],[0,-2],[-1,-4],[0,-4],[0,-3],[0,-5],[-1,-4],[0,-2],[1,-1],[0,-1],[0,-3],[0,-3],[-3,-34],[-1,-3],[0,2],[0,2],[0,1],[-1,2],[0,1],[0,1],[0,1],[-1,2],[-1,1],[-1,-1],[0,1],[-1,1],[0,2],[0,1],[-2,3],[0,-1],[0,-1],[0,-1],[0,-2],[0,-6],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-4],[1,-3],[0,-1],[0,-1],[0,-6],[0,-4],[0,-1],[0,-1],[0,-1],[1,-1],[0,1],[0,1],[1,-1],[0,-3],[1,-8],[0,-2],[0,-1],[0,-2],[0,-2],[0,-1],[-1,-4],[0,-1],[0,-2],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-3],[-1,-4],[0,-1],[-1,-1],[-1,0],[0,2],[-1,0],[-1,-3],[-1,0],[0,2],[0,1],[0,1],[0,1],[0,1],[0,1],[-1,2],[-4,-6],[-3,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[-1,-4],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,0],[0,1],[0,1],[0,1],[-1,0],[-2,-1],[0,1],[0,2],[0,1],[-1,0],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-5],[-1,0],[0,1],[0,1],[-1,2],[0,3],[-1,2],[0,1],[0,1],[-1,0],[0,-1],[-1,-2],[0,-1],[0,-2],[0,-2],[1,-1],[1,-4],[0,-2],[0,-1],[1,0],[0,-1],[0,-1],[-1,-1],[0,-1],[-1,0],[0,-1],[1,-2],[0,-2],[0,-1],[1,2],[1,1],[0,-1],[1,-2],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-2,-3],[0,-1],[0,-3],[-1,-5],[0,-1],[0,-1],[1,-6],[0,-2],[0,-1],[-1,0],[0,-1],[0,-1],[-1,-1],[0,-1],[-1,-2],[-1,-5],[0,-2],[0,-1],[0,-4],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[-1,0],[0,-1],[-1,0],[0,-1],[0,-1],[0,-1],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-3],[0,-6],[0,-1],[0,-4],[0,-1],[0,-2],[0,-1],[1,0],[0,1],[1,0],[0,-1],[1,-4],[0,-1],[0,-1],[0,-2],[0,-2],[-1,0],[0,-1],[0,-3],[-1,-2],[0,-1],[0,-2],[0,-1],[1,-6]],[[6000,3055],[-2,6],[-1,1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[-1,-6],[0,-1],[0,-3],[-1,0],[0,-3],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-2],[0,-1],[-1,1],[0,1],[-1,-1],[0,1],[0,4],[-1,0],[0,1],[-1,-1],[0,-6],[-1,-2],[-1,-5],[-1,-3],[0,-1],[0,-3],[0,-3],[0,-1],[0,-1],[-1,1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-4],[0,-2],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-3],[0,-2],[0,-1],[0,-3],[0,-2],[-1,0],[0,1],[0,1],[-1,3],[0,2],[0,2],[-1,1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[-1,-2],[0,2],[0,1],[0,1],[0,1],[-2,5]],[[5982,3454],[1,-1],[2,-4],[0,1],[1,3],[0,1],[0,3],[0,2],[2,1],[0,1],[-1,4],[0,1],[1,0],[2,-1],[1,0],[0,2],[1,-1],[1,-1],[2,-10],[0,-1],[1,-1],[1,0],[1,0],[1,-5],[1,2],[1,-1],[0,-2],[0,-1],[1,0],[0,-1],[0,-1],[1,-3],[0,-1],[3,3],[0,1],[1,1],[0,1],[0,1],[0,1],[0,1],[1,0],[0,1],[0,1],[0,2],[0,1],[1,0],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[1,2],[1,3],[1,0],[1,-3],[1,0],[0,2],[0,1],[1,0],[0,1],[0,-1],[1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,1],[0,1],[2,-3],[1,-1],[0,-1],[0,-1],[1,-5],[0,-1],[0,-1],[0,-2],[0,-3],[0,-2],[0,-1],[0,-1],[2,1],[1,1],[0,2],[0,1],[0,1],[0,1],[0,1],[0,1],[1,0]],[[6048,2661],[0,-1],[1,-3],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[1,-9],[0,-1],[1,1],[0,1],[0,1],[1,-2],[0,-6],[0,-1],[0,-1],[3,-1],[0,1],[0,1],[1,3],[2,6],[0,1],[1,0],[1,-2],[0,-2],[1,1],[0,2],[0,3],[0,2],[2,3],[0,1]],[[6063,2648],[0,-5],[0,-4],[0,-1],[0,-1],[0,-1],[1,1],[0,2],[1,1],[1,0],[0,-1],[0,-1],[0,-3],[0,-4],[0,-3],[0,-1],[0,-1],[0,-2],[0,-3],[1,-5],[0,-3],[0,-1],[-1,-8],[0,-2],[0,-1],[0,-4],[1,-1],[0,-1],[0,-1],[1,1],[0,-1],[1,-1],[0,-2],[-1,-3],[0,-2],[-1,-3],[0,-2],[-1,0],[-2,0],[-2,2],[-1,2],[0,-3],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-6],[0,-1],[-2,4],[0,-2],[0,-2],[0,-1],[0,-1],[1,-8],[1,-8],[0,-1],[1,-2],[0,-1],[0,-1],[1,-1],[0,-1],[0,-3],[-1,-4],[0,-1],[1,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-2],[0,-9],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,1],[2,2],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-2],[0,-3],[0,-2],[0,-2],[-1,-5]],[[6063,2483],[-2,4],[-1,1],[0,3],[-1,1],[0,1],[-1,1],[-1,4],[0,1],[-1,-1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,3],[0,1],[0,2],[-1,-1],[-2,1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,0],[0,-1],[-1,1],[0,3],[-1,1],[0,-1],[-2,-7],[-1,-2],[0,-1],[0,-1],[-1,-1],[0,-1],[-3,2],[0,1],[0,1],[0,1],[0,1],[-1,1],[-1,-2],[-1,-1],[0,-1],[0,-1],[-2,3],[-1,0],[0,-1],[0,-1],[0,-1],[0,-2],[-3,-8],[-1,-1],[-1,0],[0,1],[0,1],[0,1],[-1,1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-3],[-1,-2],[0,-5],[1,-10],[0,-2],[-1,-1],[0,1],[0,1],[0,1],[0,1],[0,1],[-1,1],[0,-1],[0,-1],[-1,-3],[-1,-8],[-1,-4],[-1,-1],[-1,2],[-1,-1],[-1,-5],[-1,-1],[-2,-8],[-1,-2],[0,-3],[-1,-7],[-1,0],[0,1],[0,1],[-1,2],[-1,3],[-1,4],[0,-1],[0,-3],[0,-4],[0,-1],[-2,4],[0,3],[-1,1],[0,1],[-1,0],[0,-1],[0,-3],[0,-1],[-1,0],[-1,0],[-1,1],[0,3],[-1,1],[-2,0],[-1,1],[0,2],[-1,3],[0,3],[-1,0],[-1,0],[0,-1],[-1,-7],[0,-3],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[1,-2],[0,-2],[0,-2],[0,-1],[0,-1],[-1,0],[0,-1],[-1,-9],[-1,2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-3],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-4],[0,-1],[0,-1],[-1,1],[0,2],[0,1],[0,1],[-1,1],[0,1],[0,1],[0,-1],[-1,-3],[0,-1],[0,-1],[1,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[-1,-1],[0,-1],[-1,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,-1],[-1,-1],[0,-1],[0,-2],[-1,-3],[0,-1],[-1,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[-1,1],[-1,3],[-1,-2],[-1,0],[0,1],[-1,0],[-1,1],[-1,0]],[[5979,2402],[0,5],[0,3],[2,1],[0,2],[0,5],[0,2],[0,2],[-1,1],[-1,0],[-2,-2],[-1,0],[0,5],[-1,8],[-1,22],[0,4],[0,2],[-1,2],[0,-1],[-1,-1],[0,-1],[-1,2],[0,1],[-1,0],[-1,-4],[-2,-6],[-2,-2],[-2,1],[-1,1],[0,2],[-1,4],[0,3],[0,4],[0,2],[-1,1],[-2,-2],[-1,2],[-2,4],[-1,1],[-1,-2],[-1,-5],[0,-1],[-1,1],[-2,5],[2,6],[1,7],[0,8],[0,8],[0,-2],[-1,0],[0,1],[-1,1],[0,1],[-1,3],[0,2],[0,2],[0,2],[1,1],[0,1],[0,2],[0,3],[0,2],[0,2],[0,3],[0,4],[0,3],[0,5],[0,4],[-1,4],[-1,7],[0,2],[0,2],[0,5],[0,2],[-1,2],[-1,1],[0,3],[0,5],[1,4],[5,6],[2,0],[0,1],[1,7],[0,10],[0,9],[-1,3]],[[5969,2744],[3,-5],[0,-1],[0,-1],[1,2],[2,0],[0,1],[1,2],[0,1],[0,1],[0,1],[1,0],[0,2],[0,1],[1,2],[0,2],[0,1],[1,0],[0,-1],[1,-3],[0,-1],[2,-2],[0,1],[0,2],[0,1],[0,3],[0,1],[0,1],[1,4],[0,1],[1,0],[0,-1],[0,-1],[0,-1],[0,-1],[1,0],[0,1],[1,1],[0,2],[0,1],[0,2],[1,1],[0,-1],[1,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-3],[0,-2],[0,-1],[0,-4],[1,-7],[0,-1],[0,-1],[0,-3],[0,-2],[0,-1],[0,-7],[0,-1],[0,-1],[0,-2],[0,-1],[1,-1],[2,9],[1,1],[2,0],[1,-1],[0,-2],[1,0],[1,2],[0,4],[1,1],[0,1],[0,1],[3,0],[1,2],[5,13],[1,-1],[0,-1],[2,-10],[0,-2],[1,-6],[0,-2],[1,-2],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[1,-2],[1,-4],[0,-1],[1,0],[0,-1],[1,-2],[1,-5],[0,-1],[1,0],[1,1],[0,-1],[1,-3],[0,-1],[1,-9],[1,-1],[2,-4],[1,-2],[1,2],[0,1],[1,-1],[0,-2],[1,-1],[2,-3],[0,-1],[1,0],[0,2],[0,-4],[0,-4],[0,-1],[1,-2],[0,-1],[2,-8],[1,-10],[1,-4],[0,-1],[1,-3],[0,1],[2,7],[0,1],[0,3],[0,1],[0,2],[0,1],[0,1],[0,1],[0,2],[1,1],[2,-1],[0,-1],[1,-1],[0,-2],[0,-3],[1,-1],[0,-1],[1,0],[0,1],[1,2],[0,2],[1,6]],[[6115,2987],[-1,-6],[0,-5],[0,-2],[-1,-3],[0,-2],[0,-2],[1,-5],[0,-2],[0,-3],[0,-3],[0,-10],[0,-3],[0,-2],[0,-10],[1,-2],[0,-2],[0,-2],[-1,-2],[0,-4],[-1,-8],[0,-4],[0,-2],[1,0],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[-1,0],[0,-1],[0,-3],[-1,-4],[0,-3],[1,-3],[0,-2],[0,-1],[0,-2],[0,-2],[0,-7],[0,-3],[0,-5],[0,-3],[0,-2],[-1,0],[0,1],[-1,-1],[1,-3],[0,-1],[-1,-1],[-1,-4],[-1,-6],[0,-1],[0,-1],[1,-3],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[-1,-4],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,-4],[0,-1],[0,-2],[1,-1],[0,-1],[1,-1],[0,-2],[0,-2],[0,-1],[0,-3],[0,-2],[0,-2],[3,-8],[1,-1],[0,-1],[0,-2],[1,-7],[1,-1],[0,-5],[0,-1],[0,-3],[1,-1],[0,-3],[1,-6],[0,-1],[0,-1],[1,-1],[3,-4],[0,-1],[2,-7],[0,-2],[0,-5],[0,-1],[0,-1],[1,0],[0,1],[0,2],[1,0],[0,1],[0,-1],[1,-6],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-6],[0,-3],[0,-2],[0,-2],[0,-2],[-1,0],[0,-1],[0,-1],[0,-1],[0,-7],[-1,-2],[0,-4],[0,-2],[0,-2],[0,-3],[0,-2],[0,-1],[0,-1],[0,-3],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[-2,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-4],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[-1,-7]],[[6124,2646],[-3,2],[-2,-1],[-4,2],[0,1],[-1,0],[0,5],[-1,-1],[0,-2],[-1,-2],[0,-1],[0,-2],[0,-1],[-1,0],[-1,1],[0,2],[-1,2],[0,1],[-1,0],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[-1,-3],[0,-1],[-3,1],[-2,4],[-2,2],[-2,6],[-1,3],[1,2],[0,4],[0,1],[0,1],[0,1],[0,1],[0,3],[0,1],[-1,2],[0,1],[0,1],[0,1],[0,2],[-1,1],[0,1],[0,1],[-2,0],[-1,2],[-1,1],[0,2],[-1,0],[-2,-2],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-3],[0,-2],[-1,3],[0,-2],[-1,-5],[1,-6],[-1,-2],[0,-1],[0,-1],[-1,-3],[-1,1],[0,2],[-1,5],[-1,3],[0,7],[-1,3],[-1,5],[0,1],[-1,1],[0,-1],[-1,-1],[0,-2],[0,-2],[-1,-6],[-1,1],[0,1],[0,1],[0,1],[0,1],[0,4],[-1,1],[0,1],[0,1],[0,-1],[0,-1],[-1,-2],[0,-2],[0,-1],[0,-1],[0,-1],[-1,-2],[0,1],[-1,0],[0,1],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[-4,-6],[-3,-9]],[[6048,2661],[0,5],[0,6],[0,1],[1,17],[0,7],[0,3],[1,3],[0,4],[1,2],[0,1],[1,-1],[0,1],[0,1],[0,1],[0,2],[0,1],[2,3],[1,0],[0,1],[0,3],[0,2],[0,1],[0,1],[0,4],[0,1],[0,1],[0,1],[0,1],[1,1],[0,2],[0,1],[0,2],[0,1],[0,1],[0,1],[0,2],[-1,4],[0,2],[0,1],[-1,1],[0,3],[0,3],[-1,1],[0,-1],[-1,0],[0,3],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,-1],[-1,1],[0,1],[-1,3],[0,1],[0,-1],[-1,1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[1,1],[0,1],[0,2],[-1,4],[0,1],[0,-1],[0,-1],[-1,1],[-1,5],[0,4],[1,1],[0,1],[0,2],[0,1],[0,1],[0,1],[1,-1],[1,0],[0,1],[0,1],[0,1],[0,2],[0,1],[-1,0],[0,2],[0,2],[0,5],[0,2],[0,2],[0,1],[0,1],[0,2],[0,1],[1,3],[1,1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,2],[0,2],[-1,1],[0,2],[-1,1],[0,2],[0,1],[0,1],[0,1],[1,2],[0,1],[0,1],[0,2],[0,1],[1,0],[0,1],[-1,1],[-1,6],[0,3],[1,2],[0,1],[0,2],[0,2],[0,2],[0,1],[1,10]],[[6051,2905],[0,3],[0,2],[1,2],[0,7],[0,1],[1,1],[1,-1],[0,-1],[0,-1],[1,0],[0,-1],[1,0],[1,1],[1,0],[2,-10],[0,-1],[1,-1],[1,2],[0,1],[0,-1],[1,0],[0,-7],[0,-4],[1,-1],[0,-1],[0,-1],[1,0],[0,1],[0,1],[0,1],[0,1],[0,3],[0,1],[1,3],[1,2],[1,3],[0,1],[0,2],[0,1],[2,-2],[1,3],[1,1],[0,1],[-1,1],[0,1],[0,2],[1,7],[1,3],[0,1],[0,1],[-1,2],[-1,6],[0,3],[0,1],[0,2],[0,2],[0,2],[0,2],[0,1],[0,1],[1,1],[0,2],[0,2],[0,1],[-1,1],[0,1],[0,1],[0,1],[0,1],[-1,1],[0,4],[0,1],[-1,1],[0,1],[-1,5],[4,8],[1,1],[1,0],[2,6],[0,2],[0,1],[0,1],[0,1],[0,2],[-1,7],[0,3],[0,1],[0,2],[0,1],[0,3],[0,2],[0,1],[0,1],[0,2],[-1,2],[1,3],[0,3],[0,2],[0,1],[3,2],[0,1],[1,1],[0,3],[1,0]],[[6080,3042],[4,-4],[0,-2],[1,-3],[0,-1],[0,-1],[0,-10],[1,-4],[0,-3],[0,-1],[0,-1],[0,-1],[1,-4],[0,-7],[1,-2],[0,1],[0,1],[1,-1],[2,3],[1,0],[2,3],[2,4],[1,4],[0,4],[0,2],[0,2],[0,4],[0,1],[0,2],[0,3],[0,1],[0,2],[1,2],[3,7],[1,5],[1,1],[0,-1],[1,-2],[0,-1],[0,-2],[2,-5],[1,-5],[0,-2],[1,-5],[0,-3],[0,-3],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[4,-9],[2,4],[0,-1],[0,-2],[0,-2],[0,-3],[1,-6]],[[6065,3812],[0,-2],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[1,-1],[0,-2],[0,-1],[-1,-6],[0,-1],[0,-2],[0,-1],[0,-5],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[1,-1],[0,-3],[0,-1],[0,-2],[2,-6],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-7],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-1],[0,-2],[0,-3],[0,-4],[0,-2],[0,-2],[1,-5],[0,-2],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-3],[1,-2],[1,-7],[1,2],[0,1],[0,-1],[0,-1],[0,-8],[0,-2],[0,-3],[0,-2],[0,-7],[0,-2],[1,-1],[0,-5],[1,-4],[0,-5],[0,-2],[0,-4],[0,-2],[0,-2],[0,-1],[0,-2],[0,-2],[0,-4],[0,-5],[1,0],[0,-1],[1,2],[0,-1],[1,-2],[1,0],[0,-1],[2,-8],[1,1],[0,-6],[1,2],[0,3],[1,1],[0,-1],[1,-1],[0,-1],[1,0],[0,1],[1,2],[0,1],[0,-1],[0,-1],[0,-1],[0,-2],[1,-1],[0,-4],[0,-1],[0,-3],[0,-3],[0,-1],[1,-5],[1,-1],[2,2],[3,-2],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-1],[0,-1],[1,-1],[0,-1],[0,-2],[1,-1],[0,1],[1,0],[1,0],[0,1],[1,1],[2,1],[1,1],[0,1],[0,1],[1,1],[1,-3],[0,1],[1,0],[0,2],[0,2],[1,3],[2,4],[1,1],[0,-1],[1,-1],[0,-2],[0,-1],[0,-1],[0,-1],[1,0],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-4],[0,-5],[1,-3],[1,-9],[3,-17],[1,-4],[0,-1],[0,-1],[0,-1],[-1,-10],[0,-4],[-1,-1],[0,-2],[1,-4],[0,-1],[0,-1],[1,3],[1,1],[0,-1],[0,-1],[0,-1],[1,0],[1,-13],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[-1,-1],[0,-3],[0,-4],[2,-20],[0,-1],[0,-3]],[[6116,3471],[-2,-13],[-3,-11],[0,-2],[0,-1],[-3,-3],[-2,-1],[-1,-1],[0,-1],[-1,-6],[0,-1],[0,-2],[-1,-3],[0,-1],[0,-4],[1,-1],[0,-1],[0,1],[1,4],[1,1],[1,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-2,-21],[-1,5],[-1,2],[-1,-1],[-3,-4],[0,-2],[0,-1],[0,-1],[0,-1],[-1,-4],[-1,-1],[0,-4],[-1,-2],[0,-1],[0,-1],[0,-6],[-1,-1],[0,-1],[-1,-6],[0,-1],[-1,-1],[0,-3],[0,-2],[0,-1],[0,-2],[0,-2],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-7],[0,-1],[0,-1],[0,-1],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[-1,0],[-1,0],[-1,-2],[0,-2],[0,-1],[0,-3],[0,-2],[0,-4],[1,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,0],[-1,0],[0,-1],[-3,-1],[-1,-1],[-1,-3],[0,-2],[0,-1],[0,-1],[-1,1],[0,1],[0,2],[0,1],[0,1],[-1,0],[-2,-3],[0,-1],[-1,-3],[0,-5],[0,-4],[0,-2],[0,-6],[0,-3],[0,-1],[0,-1],[0,-5],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[1,0],[2,-5],[0,-4],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,0],[0,1],[0,1],[-1,-2],[0,-2],[-1,-1],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-3],[0,-5],[0,-3],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[-1,-2],[0,-2]],[[6074,3209],[-1,2],[-2,5],[-2,4],[0,6],[0,13],[0,4],[0,2],[0,1],[-1,1],[0,1],[0,1],[-1,9],[-1,5],[0,1],[0,1],[0,1],[0,3],[0,2],[0,1],[-1,2],[0,1],[0,1],[1,0],[0,1],[1,1],[0,1],[1,4],[0,5],[0,2],[1,2],[0,1],[0,2],[0,1],[0,6],[-1,3],[0,1],[0,1],[-2,0],[0,1],[-1,2],[0,1],[0,-1],[0,-1],[-1,0],[-3,2],[0,1],[0,2],[-1,1],[0,2],[0,1],[0,1],[-1,-1],[0,-1],[0,1],[-1,3],[0,1],[-1,0],[-1,1],[0,2],[0,1],[0,2],[0,1],[0,3],[0,1],[0,1],[0,2],[-1,1],[0,1],[-1,0],[0,1],[0,1],[0,2],[0,2],[0,1],[0,2],[-1,4],[0,3],[-2,1],[-1,0],[0,-3],[-1,0],[-1,-1],[0,-3],[0,-1],[0,-1],[0,-2],[1,-1],[0,-1],[0,-1],[0,-1],[-1,-10],[0,-2],[-2,-3],[-3,5],[0,1],[0,1],[0,1],[0,1],[0,1],[-1,1],[-1,4],[0,1],[0,2],[-1,1],[0,2],[0,2],[0,2],[0,1],[0,1],[0,1],[-1,0],[-1,-2],[-4,4]],[[6031,3393],[2,6],[0,17],[2,16],[2,-8],[2,16],[1,24],[-1,10],[1,14],[3,-5],[2,29],[0,23],[2,2],[1,7],[2,11],[1,16],[1,0],[0,5],[-1,6],[-1,11],[-1,8],[-2,8],[-3,8],[-1,-2],[-2,-2],[-1,-1],[-1,14],[-1,-4],[1,-11],[-1,-5],[0,-10],[-1,-9],[0,-9],[2,-21],[-6,-14],[-1,-38],[-4,0],[0,18],[-3,-1],[0,-22],[3,-31],[-3,-18]],[[6034,3631],[2,9],[-1,8],[-2,8],[-1,-17],[2,-8]],[[6028,3560],[2,4],[1,2],[1,9],[-3,3],[-2,-3],[0,-8],[0,-5],[1,-2]],[[6000,3055],[0,-3],[0,-2],[0,-3],[0,-1],[1,0],[0,-1],[0,-1],[1,-2],[0,-1],[0,-1],[1,0],[3,3],[0,1],[0,-3],[0,-1],[1,-2],[0,-6],[1,-2],[0,-2],[0,-6],[1,-4],[1,-4],[0,-4],[0,-1],[1,-1],[1,-1],[1,-5],[1,-2],[0,-1],[1,0],[0,1],[0,1],[1,2],[0,3],[0,1],[1,0],[1,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[1,-9],[0,-1],[0,-1],[2,-9],[0,-1],[1,-2],[3,-2],[1,0],[0,1],[0,1],[0,1],[1,2],[0,-6],[3,-1],[0,-1],[1,-3],[1,-2],[1,-1],[0,1],[1,2],[1,2],[0,3],[1,5],[1,1],[1,0],[1,0],[0,-1],[1,-3],[0,-4],[1,0],[2,5],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-5],[0,-12],[0,-3],[0,-3],[0,-1],[0,-9],[1,-5],[1,0],[1,1],[0,4],[1,1],[0,-1],[1,-5],[1,-6],[1,-4],[0,-1],[0,-8],[1,-7]],[[6214,1545],[1,-7],[1,-11],[0,-8],[0,-2],[0,-5],[0,-6],[0,-5],[0,-2],[0,-1],[0,-1],[4,-3],[0,1],[0,7],[0,16],[0,1],[0,1],[1,3],[0,2],[0,1],[0,2],[1,0],[3,2],[0,1],[0,2],[0,26],[0,1],[0,1],[0,1],[3,7],[0,-1],[0,-13],[0,-1],[1,-1],[0,-1],[1,-3],[1,-1],[0,-1],[0,-1],[0,-1],[0,-6],[0,-12],[0,-72],[-2,-52],[-2,-13],[-1,-3],[-2,1],[0,-1],[0,-1],[-1,-3],[0,-3],[0,-2],[-1,-4],[-2,-14],[-1,-4],[0,-4],[0,-2],[0,-3],[-1,-10],[0,-2],[0,-1],[-1,-3],[0,-2],[0,-2],[-1,-2],[0,-2],[1,-1],[2,-5],[0,-1],[0,-1],[0,-1],[1,-2],[0,-2],[-1,-23],[0,-3],[0,-1],[1,-4],[0,-5],[1,-1],[0,-3],[0,-3],[0,-1],[0,-1],[-2,-4],[0,-3],[-1,-3],[0,-2],[0,1],[0,7],[0,4],[0,3],[-1,4],[0,4],[0,2],[-1,4],[-1,1],[0,-1],[0,-2],[0,-1],[0,-3],[0,-2],[0,-1],[0,-3],[0,-1],[-1,-4],[-1,-16],[-1,-5],[0,-3],[0,-1],[0,-5],[-1,-2],[0,-2],[-2,-7],[0,-2],[-1,0],[-4,2],[-1,1],[0,1],[0,3],[-2,14],[-2,9],[-3,18],[-3,16],[-1,3],[-1,0],[-3,-11],[-1,0],[0,2],[-4,2],[0,1],[0,1],[-1,4],[0,1],[0,1],[-1,2],[0,2],[-1,3],[-1,3],[-1,1],[0,1],[0,1],[0,1],[0,1],[-1,6],[-6,13],[-3,-5],[-1,0],[0,-1],[0,-2],[0,-15],[1,-4],[0,-8],[0,-4],[1,-2],[0,-1],[0,-1],[1,-1],[1,-1],[0,-2],[0,-2],[1,-14],[0,-1],[0,-1],[-2,0],[-4,-3],[-1,-3],[0,-6],[0,-2],[0,-3],[-1,-3],[0,-2],[0,-2],[0,-1],[0,-2],[0,-2],[0,-3],[0,-1],[-1,0],[0,-1],[-1,-5],[-2,2],[0,2],[0,1],[0,3],[0,1],[0,1],[-1,3],[0,1],[-1,1],[0,-1],[0,-1],[0,-1],[0,-2],[-1,-2],[0,-2],[-1,-2],[0,-1],[0,-3],[1,-6],[0,-2],[0,-1],[0,-1],[-1,-2],[1,-2],[0,-1],[1,-10],[0,-3],[0,-2],[0,-2],[0,-2],[0,-2],[0,-3]],[[6143,1171],[0,4],[0,6],[0,1],[0,1],[-1,1],[0,1],[0,2],[0,7],[0,2],[0,3],[0,1],[-1,1],[0,1],[0,2],[-1,5],[0,2],[-1,1],[-1,1],[0,1],[0,1],[0,1],[0,8],[0,1],[0,1],[-1,1],[0,2],[0,1],[0,1],[0,1],[0,1],[0,1],[1,1],[0,5],[0,14],[-1,-3],[-1,0],[0,-2],[0,-6],[0,-2],[0,-1],[-4,-1],[-1,1],[0,2],[0,8],[0,1],[-1,0],[-7,-1],[-3,1],[0,1],[-1,2],[0,6],[0,3],[0,1],[-1,1],[0,1],[-1,0],[0,1],[0,1],[0,2],[0,4],[0,1],[1,2],[0,2],[0,1],[0,1],[0,7],[0,3],[0,2],[-1,0],[0,1],[0,1],[0,3],[0,3],[0,3],[0,1],[0,1],[-1,5],[0,1],[0,1],[-4,2],[0,1],[0,1],[0,3],[0,14],[0,1],[0,2],[0,1],[1,1],[0,1],[0,2],[0,4],[0,2],[0,3],[1,2],[0,7],[0,1],[-2,3],[0,1],[-1,5],[-5,0],[0,1],[0,1],[0,1],[-1,2],[0,1],[-2,0],[0,-2],[-1,-1],[-1,1],[-2,-1],[0,1],[-1,1],[0,1],[0,1],[-2,2],[0,1],[0,1],[-1,0],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[-1,-2],[-1,-2],[-3,1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-5],[0,-1],[1,-2],[0,-2],[-1,-1],[0,-1],[0,-1],[0,-1],[-1,0],[0,-1],[0,-1],[-1,-2],[0,-2],[0,-1],[1,-5],[0,-4],[0,-1],[-1,-1],[-2,-4],[-1,0],[0,1],[-1,3],[-1,1],[-1,2],[-1,1],[-1,-2],[0,-9],[-1,3],[0,2],[0,2],[0,5],[0,2],[0,3],[1,7],[0,2],[0,2],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[1,10],[0,2],[0,2],[0,3],[-1,1],[0,1],[0,1],[1,0],[0,1],[0,1],[0,1],[-1,0],[0,-1],[-1,0],[-4,1],[0,2],[0,8]],[[6074,1394],[1,1],[1,5],[1,2],[1,2],[0,7],[1,3],[0,1],[1,-1],[1,1],[1,2],[0,5],[2,9],[0,2],[1,2],[0,1],[0,-1],[1,-3],[0,-1],[1,1],[0,2],[3,-4],[1,1],[0,4],[0,10],[0,4],[1,3],[1,2],[1,2],[-1,4],[-1,-1],[-1,-3],[0,-6],[-1,2],[0,2],[0,7],[0,1],[0,2],[0,8],[0,3],[1,1],[0,2],[0,8],[-1,6],[-1,2],[-1,0],[-1,3],[0,-2],[-1,1],[0,2],[-1,0],[0,-1],[-1,-3],[0,-1],[-1,-3],[-1,1],[-1,-1],[0,-8],[0,-5],[0,-2],[0,-2],[-1,0],[-1,-2],[0,-2],[-1,-1],[-2,0],[-1,-3],[-1,-1],[-1,-2],[-1,-1],[-2,0],[0,-1],[0,-1],[1,0],[0,-1],[0,-2],[-1,0],[-1,1],[0,2],[-1,4],[1,2],[0,5],[0,1],[1,2],[0,6],[1,3],[0,2],[5,4],[1,2],[0,2],[1,4],[0,2],[-1,1],[0,-2],[0,-5],[-1,-1],[0,1],[-5,-5],[-1,-3],[-1,-4],[0,-7],[0,-2],[-1,0],[0,-1],[0,-2],[0,-3],[-1,-3],[0,-1],[-5,-4],[-1,-2],[0,-2],[0,7],[-1,8],[1,7],[0,6],[1,2],[0,2],[1,2],[0,4],[0,5],[0,2],[0,2],[-1,0],[-1,-1],[-1,3],[0,4],[1,4],[0,7],[1,8],[0,7],[0,17],[1,9],[0,8],[1,7],[1,4],[6,5],[0,2],[1,3],[1,7],[0,2],[1,0],[1,-2],[1,4],[0,11],[0,11],[1,10],[1,6],[1,4],[5,-1],[1,-6],[1,-1],[2,2],[4,2],[0,-1],[1,-3],[1,-2],[1,2],[1,-1],[1,-1],[2,2],[3,-5],[1,2],[1,4],[0,6],[0,7],[0,1],[0,4],[0,2],[0,2],[1,3],[0,2],[0,4],[0,3],[-1,5],[0,3],[0,1],[0,1],[0,3],[0,1],[2,2],[0,-1],[0,1],[0,2],[0,8],[0,5],[0,5],[2,16],[0,3],[1,6],[1,4],[0,4],[0,5],[0,4],[-1,1],[-1,-3],[-1,-1],[-2,8],[2,9],[1,4],[0,3],[0,2],[0,4],[0,6],[-1,11],[-1,2],[0,2],[0,4],[1,0],[0,2],[0,1],[0,1],[0,1],[0,2],[-1,7],[0,4],[-1,2],[-3,3],[-1,-2],[-1,0],[1,6],[1,33],[0,2],[1,2],[0,2],[1,3],[0,3],[0,3],[1,2],[1,0],[3,-7],[1,1],[1,6],[1,4],[0,5],[0,5],[-1,4],[-1,-3],[0,-4],[-1,-2],[0,1],[-1,4],[0,2],[-1,-1],[-3,3],[0,2],[-1,2],[0,4],[0,4],[-1,3],[0,2],[-1,3],[1,13],[0,2],[2,-1],[2,5],[1,0],[0,-1],[1,0],[0,1],[0,3],[0,7],[1,3],[0,6],[1,5],[2,6],[0,4],[1,8],[1,3],[0,2],[0,2],[0,9],[1,8],[0,5],[0,4],[-2,8],[-1,10],[0,8],[-1,7],[1,8],[1,3],[1,4],[0,3],[0,5],[0,2]],[[6114,2064],[0,3],[1,4],[0,1],[0,1],[1,1],[2,-2],[4,-6],[1,0],[1,0],[1,2],[1,3],[1,3],[0,3],[1,2],[1,2],[2,-1],[0,2],[1,5],[1,2],[1,1],[1,-1],[1,3],[3,23],[0,2],[0,4],[0,6],[0,6],[1,6],[0,6],[0,5],[1,9],[1,3],[1,7],[2,4],[1,1],[1,3],[1,7],[0,2],[0,1],[0,2],[0,1],[0,2],[0,2],[-1,3],[-1,6],[0,1],[0,2],[0,4],[2,11]],[[6148,2221],[4,-35],[0,-1],[1,-1],[0,-1],[0,-1],[1,-1],[1,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[-1,-2],[0,-3],[1,-1],[0,-1],[0,-5],[0,-4],[8,-15],[0,-1],[0,-6],[3,-11],[0,-4],[3,-15],[1,-4],[0,-5],[1,-3],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-13],[-1,-11],[0,-4],[0,-4],[0,-2],[0,-1],[0,-1],[0,-1],[-1,0],[0,-1],[0,-1],[0,-1],[0,-1],[0,-13],[0,-5],[0,-6],[0,-3],[0,-5],[0,-3],[1,-2],[0,-2],[-1,-2],[0,-4],[1,-2],[0,-3],[0,-2],[0,-2],[-1,0],[0,-1],[-1,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[1,-7],[0,-1],[0,-3],[0,-2],[0,-2],[0,-6],[0,-2],[0,-1],[0,-1],[1,-1],[2,0],[1,1],[2,4],[1,0],[0,-1],[1,-1],[0,-2],[0,-1],[0,-1],[0,-3],[0,-3],[1,-9],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[2,0],[0,-1],[5,-16],[0,-2],[1,-1],[0,-2],[0,-1],[0,-2],[-1,-8],[0,-1],[0,-2],[1,-2],[0,-2],[1,-9],[0,-3],[1,-4],[0,-3],[0,-2],[0,-2],[0,-1],[-2,-7],[-1,-8],[0,-2],[0,-2],[0,-1],[0,-2],[0,-3],[1,-9],[0,-3],[0,-1],[1,-2],[0,-1],[-1,-2],[0,-2],[0,-2],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[-1,-1],[-1,-1],[0,-2],[0,-1],[-2,-1],[0,-1],[-1,-3],[0,-1],[-1,0],[0,1],[-1,0],[0,-1],[-1,-1],[0,-1],[-2,1],[0,-1],[-3,-8],[-1,-4],[-2,-8],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[1,-13],[1,-8],[0,-2],[0,-3],[0,-1],[0,-1],[-2,-1],[0,-1],[-1,-2],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[1,-1],[0,-1],[0,-3],[1,-2],[0,-1],[0,-2],[0,-4],[0,-3],[0,-1],[0,-1],[-1,-3],[0,-1],[0,-1],[-1,-1],[0,-17],[0,-5],[0,-1],[0,-1],[0,-1],[1,0],[3,2],[1,-1],[1,-1],[0,-2],[1,-1],[0,1],[0,1],[0,2],[1,2],[0,1],[0,1],[1,0],[0,-2],[4,-6],[0,-2],[0,-2],[1,-8],[5,-27],[0,-1],[0,-1],[1,-4],[0,-2],[0,-16],[1,-6],[3,-36],[1,-14],[0,-3],[1,-2],[0,1],[1,2],[0,1],[1,10],[0,2],[1,0],[0,1],[2,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-3],[0,-10],[0,-1],[0,-1],[0,-2],[1,-4],[0,-2],[0,-2],[1,-1],[1,0],[6,8],[3,0],[1,0],[0,-4],[0,-1],[1,-4],[0,-1],[0,1],[1,0],[0,4],[0,1],[1,2]],[[6074,3209],[0,-11],[0,-4],[0,-1],[0,-1],[0,-1],[0,-1],[0,1],[0,2],[0,1],[1,0],[0,1],[1,0],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-4],[0,-1],[0,-2],[-1,-1],[0,-1],[0,-1],[0,-1],[-1,-3],[0,-1],[0,-3],[0,-6],[0,-1],[1,-1],[0,-3],[0,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-1],[0,2],[1,1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-4],[0,-1],[0,-1],[0,-1],[1,-1],[0,-2],[1,-2],[0,-4],[0,-2],[0,-2],[-1,-2],[0,-3],[0,-1],[0,-1],[0,-2],[1,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[1,-3],[0,-3],[0,-1],[-1,-1],[0,-1],[0,-2],[0,-1],[1,-2],[0,-1],[2,0],[0,-2],[0,-1],[1,-4],[0,-2],[0,-1],[-1,-8],[0,-5],[0,-2],[0,-1],[0,-1],[0,-1],[-1,-3],[0,-1],[0,-1],[0,-1],[1,-2],[0,-1],[0,-1],[0,-3],[0,-1],[0,-1],[0,-2],[0,-3]],[[6190,2468],[3,-9],[3,-14],[1,-1],[0,-1],[1,-2],[5,-25],[0,-1],[1,0],[1,1],[1,1],[1,5],[2,4],[1,2],[1,0],[1,0],[0,1],[0,2],[0,1],[0,2],[0,1],[0,1],[0,2],[0,1],[1,0],[3,2],[1,0],[0,-1],[0,-1],[1,0],[0,2],[0,1],[1,2],[0,3],[0,2],[0,1],[0,1],[1,-1],[1,1],[0,2],[0,1],[1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-4],[0,-3],[0,-4],[0,-2],[0,-1],[0,-1],[1,3],[0,2],[1,0],[1,2],[0,-1],[1,-3],[0,-1],[0,1],[1,5],[0,19],[2,-2],[1,-1],[3,2],[0,1],[1,3],[0,1],[0,-1],[1,-1],[0,-2],[1,-3],[0,-1],[0,-2],[1,-5],[0,-2],[1,-1],[0,-1],[1,2],[0,1],[0,2],[0,1],[0,1],[0,2],[0,1],[1,0],[2,-5],[1,-1],[1,1],[2,0],[0,-1],[1,-1],[0,-6],[0,-1],[1,0],[0,1],[1,1],[0,2],[1,0],[1,-2],[0,-1],[0,-1],[0,-5],[0,-3],[1,0],[0,-1],[0,-1],[1,-2],[2,-4],[0,-3],[0,-2],[0,-3],[0,-1],[0,-3],[0,-1],[0,-3],[0,-1],[0,-3],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[1,-1],[1,-7],[1,-4],[0,-1],[0,-3],[0,-3],[0,-4],[0,-3],[0,-4],[0,-3],[0,-6],[-1,-3],[0,-5],[0,-2],[0,-1],[-1,-1],[0,-1],[0,-1],[-1,0],[-1,-1],[0,-2],[0,-1],[0,-1],[0,-1],[1,-1],[1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-5],[0,-2],[0,-3],[-1,-2],[0,-2],[0,-2],[0,-1],[1,-1],[0,-1],[3,0],[1,1],[0,1],[0,3],[1,1],[1,1],[1,-1],[1,-1],[0,-1],[0,-2],[0,-1],[0,-6],[0,-2],[0,-1],[0,1],[1,4],[0,2],[1,1],[2,2],[0,1],[0,3],[0,1],[0,1],[0,3],[0,2],[0,14],[0,2],[0,2],[0,2],[0,2],[0,2],[1,2],[0,1],[1,3],[1,2],[1,1],[1,-1],[1,-3],[3,-15],[0,-1],[1,-2],[0,-1],[0,-1],[1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-27],[0,-1],[0,-1],[0,-1],[0,-1],[1,0],[4,1],[1,0],[0,1],[0,2],[0,2],[0,1],[0,2],[0,1],[1,1],[1,0],[0,-1],[0,-1],[0,-2],[0,-1],[1,0],[2,-1],[0,1],[0,2],[0,1],[0,1],[0,3],[0,1],[0,1],[0,3],[0,2],[1,2],[0,3],[1,1],[0,-1],[1,0],[0,1],[0,2],[0,12],[0,1],[0,1],[0,1],[1,-2],[1,-2],[2,1],[0,-1],[1,-3],[0,-4],[1,-1],[0,-1],[1,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-2],[-1,-1],[0,-2],[0,-1],[1,-2],[0,-2],[1,-1],[0,-1],[0,-1],[0,-2],[0,-4],[1,-3],[1,-4],[0,-3],[0,-1],[1,-1],[1,-1],[1,0],[0,1],[0,1],[0,1],[0,2],[0,1],[1,1],[3,0],[1,-1],[0,-1],[0,-1],[0,-3],[0,-4],[0,-5],[0,-1],[0,-2],[0,-1],[0,-2],[1,-2],[0,-2],[0,-6],[0,-1],[1,-2],[4,1]],[[6317,2258],[-1,-10],[0,-3],[-2,-5],[-1,-2],[0,-5],[0,-3],[1,-2],[0,-4],[0,-4],[-1,-5],[-1,-4],[1,-6],[1,-13],[0,-9],[0,-6],[-2,-13],[-2,-11],[-6,-16],[-1,-6],[-1,-10],[-2,-60],[-1,-60],[1,-3],[4,-18],[2,-11],[0,-12],[0,-14],[-2,-12],[-1,-10],[-1,-3],[0,-3],[-2,-7],[-1,-4],[-1,-5],[-4,-61]],[[6294,1838],[-4,23],[-4,11],[0,2],[0,3],[0,2],[-1,1],[-6,7],[-1,0],[-1,-1],[-1,-3],[0,-2],[0,-1],[0,-1],[0,-1],[-1,-4],[0,-1],[0,-2],[0,-2],[-1,-1],[-1,-12],[-3,-18],[0,-2],[0,-1],[-1,-2],[-1,0],[0,-3],[-1,-2],[0,-6],[0,-1],[0,-1],[-1,0],[-1,-2],[-2,-3],[-1,-1],[0,1],[-1,-1],[0,-3],[-2,-4],[-1,2],[0,2],[-1,1],[0,-1],[-1,0],[0,-1],[0,-2],[-1,-2],[0,-2],[0,-1],[0,-1],[0,-1],[1,-2],[0,-2],[0,-1],[1,0],[0,1],[1,1],[0,-1],[1,-2],[0,-2],[0,-1],[0,-1],[1,0],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-3],[0,-2],[0,-1],[0,-1],[-1,0],[0,1],[-1,5],[0,3],[-1,2],[-1,2],[0,-1],[-1,-1],[0,-6],[0,-1],[0,-2],[0,-1],[0,-1],[-1,0],[0,-1],[-1,-1],[0,-2],[0,-1],[0,-1],[-1,-3],[-1,-2],[0,-1],[0,-3],[0,-1],[0,-2],[0,-2],[-1,-3],[0,-2],[0,-1],[0,-1],[0,-3],[0,-1],[0,-2],[0,-4],[0,-2],[0,-1],[5,-8],[0,-1],[0,-2],[0,-2],[0,-4],[0,-2],[0,-2],[2,-4],[1,-1],[0,-1],[0,-2],[0,-2],[0,-3],[1,-8]],[[6258,1698],[-5,3],[0,1],[0,1],[0,1],[-1,1],[0,3],[-4,17],[0,-1],[0,-3],[0,-2],[0,-1],[0,-2],[0,-3],[0,-1],[0,-1],[0,-2],[0,-2],[-2,-11],[-1,-5],[-1,-2],[-1,-1],[-3,8],[-1,-1],[0,-1],[0,-4],[0,-1],[-1,-2],[-1,1],[0,1],[0,2],[0,1],[0,8],[0,2],[0,5],[0,6],[0,6],[0,3],[0,1],[-1,1],[0,1],[-1,1],[0,1],[-1,1],[0,2],[0,1],[-1,0],[-1,1],[0,1],[0,1],[0,1],[0,1],[-1,1],[-1,0],[0,-2],[0,-1],[0,-1],[0,-2],[0,-5],[0,-29],[0,-4],[1,-4],[0,-1],[0,-1],[3,-1],[0,-1],[1,-2],[0,-2],[0,-10],[0,-1],[-1,-2],[0,-4],[-1,-5],[0,-4],[-1,-4],[0,-10],[-8,10],[-1,-2],[-1,-8],[0,-1],[0,-2],[1,-5],[0,-2],[0,-2],[-1,-7],[-1,-6],[0,-1],[-1,-1],[-1,0],[0,1],[-1,0],[-1,-1],[0,-2],[-1,-4],[-2,-7],[0,-2],[0,-2],[-1,-1],[-1,-19],[0,-6],[0,-2],[-1,-1],[1,-5],[2,-18]],[[6148,2221],[4,32],[0,9],[0,4],[0,4],[-2,14],[0,1],[0,1],[0,3],[0,2],[0,1],[0,2],[0,2],[1,5],[0,1],[0,1],[1,2],[1,1],[0,2],[0,3],[0,2],[0,1],[-1,0],[-2,2],[0,1],[0,1],[-1,1],[0,13],[-1,5],[0,3],[-1,2],[0,2],[-3,9],[-1,3],[0,1],[2,5],[4,8],[0,1],[0,1],[0,4],[0,3],[0,3],[1,8],[1,1],[2,9],[3,7],[2,9],[1,2],[1,10],[0,3],[1,11],[0,7],[0,3],[1,6],[3,4],[0,-1],[2,-4],[1,-5],[0,-1],[2,-5],[3,-2],[1,0],[1,2],[1,2],[0,2],[6,3],[1,3],[1,11],[1,1],[1,2],[0,1],[1,-1],[3,0],[0,-2]],[[6063,2483],[0,-2],[0,-1],[0,-1],[1,-1],[0,-1],[1,0],[1,3],[1,-1],[0,-2],[0,-1],[0,-1],[1,-6],[0,-2],[0,-6],[1,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-2],[0,-1],[-2,0],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,-2],[0,-3],[0,-5],[0,-3],[0,-1],[1,0],[0,1],[1,0],[1,-2],[0,-1],[0,-2],[1,-10],[0,-1],[1,-2],[1,-1],[1,1],[0,1],[0,1],[0,1],[0,4],[1,1],[0,1],[0,2],[0,1],[0,1],[0,1],[1,1],[0,1],[0,1],[0,1],[1,-1],[0,-1],[1,-5],[1,-1],[0,-2],[1,-1],[0,-2],[0,-2],[0,-2],[0,-4],[0,-3],[0,-2],[0,-5],[0,-3],[0,-2],[0,-4],[0,-2],[0,-1],[0,-1],[-1,0],[0,1],[0,1],[0,1],[-1,1],[-2,-3],[0,-1],[0,-2],[1,-3],[0,-1],[0,-1],[-1,0],[0,-1],[0,-1],[0,-1],[1,-3],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-5],[0,-1],[0,-1],[1,0],[1,1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-3],[1,-1],[0,-1],[0,-1],[1,-2],[0,-1],[0,-3],[1,0],[0,1],[0,1],[0,1],[2,-2],[0,-2],[0,-1],[0,-6],[0,-1],[1,-1],[2,-2],[0,-2],[1,-4],[0,-2],[0,-1],[-1,0],[0,1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[0,-8],[0,-2],[-1,-2],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-3],[-1,-2],[0,-1],[0,-1],[1,-2],[0,-1],[0,-2],[0,-1],[1,-7],[0,-2],[0,-1],[1,-2],[0,-3],[0,-1],[0,-1],[0,-2],[0,-2],[-1,-1],[0,-2],[0,-5],[1,-2],[0,1],[0,-1],[1,0],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-8],[0,-1],[0,-3],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[1,-1],[0,-2],[0,-1],[1,-1],[0,-4],[0,-3],[1,-3],[1,-1],[0,-5],[0,-6],[0,-1],[1,1],[0,1],[0,-2],[0,-3],[-1,-10],[0,-2],[0,-1],[0,-2],[-1,-2],[0,-3],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-3],[0,-8],[0,-5]],[[6088,2131],[-1,3],[-1,-2],[0,-3],[-1,-7],[-1,-3],[0,-1],[-3,-1],[-1,4],[0,6],[-1,6],[-3,4],[-2,7],[0,1],[0,3],[0,4],[-1,4],[0,1],[0,-1],[0,-1],[-1,-1],[0,-1],[-1,2],[-2,-2],[-1,2],[-1,3],[0,1],[-2,0],[0,2],[0,3],[-1,14],[-1,2],[0,1],[-1,-1],[-2,1],[0,-1],[0,-7],[0,-11],[0,-11],[-2,-5],[-1,2],[-1,-5],[-1,-2],[0,1],[0,4],[-1,5],[-1,9],[0,6],[-1,5],[-1,4],[-3,6],[-1,4],[-2,23],[-1,6],[0,2],[0,2],[0,3],[0,6],[0,10],[-1,5],[-2,7],[-1,7],[0,8],[-1,6],[0,2],[-3,-2],[-1,-2],[-2,-10],[0,-3],[-4,-7],[-2,2],[-2,-1],[-2,-4],[-5,-15],[0,-2],[0,-2],[-1,-6],[0,-2],[0,-2],[-1,0],[-1,0],[0,2],[-1,2],[0,2],[0,2],[0,6],[0,1],[0,2],[0,1],[-1,1],[-1,3],[-1,1],[0,1],[0,1],[-1,0],[0,-1],[-1,-3],[0,-2],[-1,-2],[-1,2],[-3,25],[-1,2],[0,5],[-1,2],[-4,2],[-3,-1],[-1,-1],[0,-2],[-1,-4],[0,-2],[0,-1],[-1,0],[0,-1],[0,-3],[-1,-3],[0,-3],[-1,-1],[-1,1],[0,2],[-1,2],[-1,4],[0,4],[0,4],[0,6],[0,4],[-1,3],[0,1],[-1,0],[-1,3],[-1,6],[-1,29],[0,4],[1,2],[2,6],[-1,5],[0,5],[0,6],[0,7],[0,2],[-1,2],[0,2],[-1,3],[0,2],[0,3],[0,4],[0,3],[-1,4],[0,3],[0,4],[0,3],[0,3],[0,3],[-1,1],[0,1],[0,2],[-1,2],[0,2]],[[6130,573],[-1,1],[0,3],[-1,-1],[-2,-6],[-1,-1],[-1,0],[-7,17],[-2,0],[-2,-3],[0,-3],[-1,-4],[-2,-32],[0,-4],[-1,1],[0,3],[-1,5],[0,6],[-1,7],[0,4],[-1,4],[-2,8],[-1,4],[0,2],[0,1],[-1,3],[0,5],[0,2],[-1,3],[-1,3],[0,2],[-1,11],[-4,20],[-1,18],[-2,13],[-6,31],[-1,5],[-1,8],[0,3],[-1,1],[0,2],[-1,9],[0,3],[0,2],[-1,1],[-1,2],[-1,1],[0,3],[-3,18],[0,4],[0,1],[-2,3],[-3,10],[-7,8],[-1,3],[0,2],[-2,0],[-1,5],[-1,8],[0,8],[-1,6],[-1,5],[0,2],[0,9],[-1,0],[-1,-2],[0,1],[0,2],[-2,8],[0,2],[0,2],[0,1],[0,1],[0,2],[0,1],[0,1],[-2,13],[0,1],[0,2],[-2,5],[0,2],[0,-6],[1,-5],[0,-4],[0,-5],[-1,-4],[-1,1],[-1,1],[-1,5],[-1,2],[-1,0],[-1,-3],[0,-1],[0,1],[-1,2],[0,2],[-1,0],[0,-1],[-1,2],[0,3],[0,3],[-1,4],[-1,4],[-1,3],[0,8],[-1,16],[-1,6],[0,1],[1,4],[0,5],[-1,4],[-2,16],[-1,2],[-1,0],[0,1],[0,1],[-1,3],[-1,3],[-1,4],[-2,4],[-3,11],[-1,1],[-1,0],[-1,-2],[-1,-1],[0,1],[-2,6],[-1,9],[0,2],[0,1],[-1,2],[0,2],[1,2],[0,1],[3,-1],[1,1],[1,4],[0,2],[1,0],[1,0],[0,3],[1,1],[1,5],[1,2],[0,2],[0,2],[0,3],[0,2],[0,1],[0,1],[0,-1],[-5,-6],[0,2],[0,5],[-1,5],[0,3],[1,3],[0,-4],[0,-6],[1,-3],[0,1],[0,2],[0,3],[0,3],[0,1],[0,1],[1,1],[0,1],[-1,1],[-1,4],[0,1],[-1,-1],[0,-1],[0,-2],[-1,-10],[-1,-2],[0,4],[0,6],[1,6],[4,13],[0,-2],[0,-1],[1,0],[5,-19],[1,-3],[1,0],[0,-1],[0,-3],[-1,-2],[0,-1],[0,-2],[0,-7],[0,-1],[0,-1],[-1,0],[0,-2],[0,-2],[0,-2],[0,-1],[1,0],[2,3],[1,0],[1,1],[0,3],[0,2],[2,-1],[0,4],[-1,1],[-2,2],[-1,0],[0,2],[-1,3],[1,1],[5,1],[0,-2],[1,-3],[0,2],[1,1],[1,2],[0,3],[1,5],[1,2],[0,-2],[-1,-7],[1,1],[0,2],[0,1],[1,-1],[0,-1],[1,-1],[0,-1],[0,-3],[0,-2],[0,-1],[0,-1],[-1,-1],[1,-3],[0,2],[1,4],[0,5],[0,-1],[0,-2],[0,-1],[0,-1],[1,0],[0,-1],[1,-2],[1,1],[0,2],[-1,1],[0,3],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[1,4],[0,2],[0,2],[0,1],[0,2],[0,-1],[-1,-2],[0,6],[0,3],[0,3],[0,5],[0,-1],[1,-2],[0,-1],[0,2],[-1,2],[0,5],[-1,1],[0,3],[1,12],[0,5],[-1,1],[0,1],[0,1],[-1,1],[0,1],[0,5],[0,1],[-1,-2],[0,-4],[1,-3],[1,-2],[0,-1],[-1,-2],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-12],[0,-7],[-1,-2],[0,-1],[0,2],[-1,2],[0,3],[1,8],[0,8],[-1,15],[0,8],[1,8],[0,5],[0,2],[1,1],[0,-1],[0,-3],[-1,-2],[1,-2],[0,-2],[1,-4],[0,5],[1,-1],[0,-1],[0,3],[-1,1],[0,1],[0,2],[0,3],[0,2],[2,13],[1,4],[1,1],[0,2],[1,5],[0,8],[0,9],[1,27],[1,10],[0,7],[1,5],[2,4],[1,1],[1,-1],[-1,-1],[-1,-3],[0,-2],[0,-2],[0,2],[1,0],[0,-2],[0,-3],[-1,-5],[0,-3],[0,-5],[0,-1],[1,-1],[0,-1],[0,-4],[0,-5],[-1,-5],[-1,-8],[0,-6],[0,-4],[1,0],[1,2],[-1,4],[0,3],[1,7],[1,14],[0,3],[0,1],[0,4],[1,5],[0,9],[0,3],[0,2],[0,3],[0,10],[1,5],[0,2],[1,0],[0,2],[1,1],[1,-2],[1,-10],[1,-3],[1,-9],[1,-2],[0,-1],[1,0],[1,0],[0,1],[1,1],[0,-2],[1,0],[0,3],[0,5],[0,6],[0,4],[-1,2],[-7,26],[0,3],[0,6],[-1,4],[0,4],[-1,4],[-3,18],[-1,5],[0,1],[-2,0],[-1,-1],[0,-1],[0,-3],[1,-2],[0,-1],[0,-4],[-1,2],[-2,12],[-1,11],[0,2],[-1,3],[0,7],[-1,9],[0,11],[0,8],[-1,7],[-1,4],[0,2],[1,0],[0,1],[1,-3],[1,-6],[1,-4],[1,2],[1,-4],[1,1],[2,3],[1,4],[1,7],[0,3],[1,1],[1,0],[1,1],[0,2],[1,6],[1,1],[0,-2],[0,-3],[0,-1],[0,-6],[1,-2],[4,-7],[3,2],[0,1],[1,5],[-1,4],[0,4],[-1,1],[-1,4],[-1,3],[-1,0],[-1,-3],[0,-2],[0,-2],[-1,-2],[0,5],[0,4],[0,3],[0,1],[1,4],[0,1],[0,2],[0,6],[0,2],[0,3],[1,3],[1,3],[0,1],[2,-3],[1,0],[1,2],[2,1]],[[6279,3405],[1,-4],[0,-1],[1,-4],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[1,0],[0,1],[0,2],[0,7],[1,4],[0,1],[0,2],[0,1],[0,4],[0,2],[0,1],[0,-1],[1,-4],[1,-4],[0,-2],[0,-2],[0,-2],[0,-2],[0,-4],[0,-2],[0,-2],[0,-2],[0,-3],[1,-1],[1,-8],[1,-2],[1,-1],[0,-2],[0,-2],[0,-4],[0,-2],[0,-2],[-1,-4],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[1,-3],[0,-5]],[[6289,3339],[0,-3],[0,-6],[0,-1],[0,-4],[0,-7],[1,-2],[1,-6],[1,-1],[0,-1],[-1,-3],[0,-1],[0,-1],[-1,-7],[-1,-3],[0,-1],[0,-1],[1,-1],[1,1],[0,2],[1,1],[0,-1],[0,-3],[0,-2],[0,-3],[0,-2],[0,-1],[0,-2],[-1,-8],[0,-1],[0,-1],[2,3],[1,-1],[0,-1],[-1,-1],[0,-1],[0,-2],[0,-1],[-1,-1],[0,-2],[-1,-1],[0,-1],[0,-1],[0,-2],[1,-3],[0,-6],[0,-1],[1,0],[1,1],[1,-1],[2,-3],[1,-1],[0,-5],[0,-3],[0,-2],[-1,-2],[0,-2],[0,-1],[-1,-1],[-2,-4],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,1],[-1,1],[-1,-1],[0,-1],[-3,-13],[-1,-1],[-1,0],[0,-1],[-1,-2],[0,-2],[-4,-4],[-2,1],[-1,-1],[-2,-9],[0,-9],[0,-1],[-1,-3],[0,-1],[-1,-4],[0,-8],[0,-4],[-1,-2],[-1,-2],[-1,-3]],[[6271,3150],[0,-4],[-1,-3],[0,-3],[-1,-5],[0,-4],[0,-1],[0,-1],[-1,1],[0,2],[-1,0],[-2,-5],[-1,-2],[-1,-2],[0,2],[-1,0],[-2,-2],[-1,-2],[-1,1],[0,6],[-4,16],[-1,-5],[0,-1],[-1,0],[-1,1],[-1,0],[-1,2],[0,1],[-3,-1],[-1,-1],[0,-2],[0,-1],[0,-2],[-1,-1],[-1,-1],[0,-1],[0,-2],[0,-2],[0,-1],[-1,0],[-1,1],[-1,-3],[-1,-1],[0,-2],[0,-1],[0,-3],[0,-10],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-4],[1,-2],[0,-2],[0,-1],[0,-1],[1,-4],[0,-1],[0,-1],[1,-4],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[-1,0],[-2,0],[0,-1],[-1,-6],[-1,-1],[-1,1],[0,1],[0,3],[-1,3],[-1,2],[0,3],[-1,2],[0,1],[-1,-1],[-1,0],[0,-1],[0,-1],[0,-3],[0,-1],[-1,1],[0,1],[-1,3],[-2,10],[0,1],[-1,0],[-1,-4],[-1,-3],[-1,1],[-2,4],[-1,8],[0,1],[-2,3],[0,1],[0,2],[0,1],[1,1],[0,1],[0,1],[0,1],[-1,1],[0,2],[1,0],[0,1],[1,1],[0,1],[0,1],[0,2],[0,1],[-1,4],[-1,5],[-1,5],[-2,7],[-5,4],[0,1],[0,2],[0,1],[-1,2],[0,2],[-1,1],[-1,-5],[-1,-1],[0,-1],[-2,2],[-1,1],[0,3],[0,1],[-1,0],[-1,-4],[-1,0],[0,-1],[-1,1],[-1,-1],[0,-2],[0,-1],[0,-1],[-1,-2],[0,-4],[0,-4],[0,-7],[0,-2],[1,1],[0,-1],[0,-1],[0,-1],[1,-13],[0,-2],[-1,-1],[-1,1],[0,1],[0,5],[-1,1],[0,1],[0,-1],[0,-1],[-1,-1],[1,-4],[0,-2],[-1,-1],[0,-1],[-1,1],[-1,0],[0,-1],[-1,-5],[0,-1],[-1,0],[0,1],[-1,0],[0,1],[0,1],[-1,1],[0,-1],[-1,0],[0,1],[0,1],[0,2],[-1,1],[0,-3],[0,-3],[0,-3],[0,-3],[0,-4],[0,-1],[0,-1],[-1,1],[0,3],[0,1],[0,2],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[-1,-1],[-1,1],[0,-1],[-1,-6],[-1,-6],[0,2],[-1,1],[0,2],[0,1],[0,1],[0,1],[0,1],[1,1],[0,1],[0,2],[0,1],[0,2],[0,1],[0,1],[0,-1],[-1,-1],[-1,-4],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[1,-10],[0,-1],[0,-1],[0,-1],[-1,1],[-1,6],[0,2],[-1,7],[0,2],[0,1],[-2,5],[-1,5]],[[6175,3112],[0,6],[1,10],[1,3],[0,2],[1,3],[1,0],[0,1],[0,2],[0,2],[0,1],[0,4],[-1,3],[0,2],[0,1],[0,2],[-1,2],[-1,4],[-2,7],[0,2],[-1,3],[1,7],[0,3],[1,0],[0,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-2],[1,-1],[1,0],[0,2],[1,1],[0,2],[0,-1],[0,-1],[1,-1],[0,-1],[0,-2],[0,-1],[1,-1],[1,0],[0,2],[1,2],[0,1],[0,1],[0,1],[-1,1],[0,3],[0,1],[0,2],[0,1],[0,1],[1,5],[1,2],[0,1],[0,1],[0,1],[0,1],[-1,1],[-2,6],[0,2],[-1,7],[0,1],[-1,0],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-4],[0,-2],[0,-1],[-1,-1],[0,-1],[-1,-4],[-1,-4],[-1,0],[0,2],[-1,1],[0,1],[0,3],[0,3],[0,1],[0,4],[0,2],[-1,5],[2,-5],[0,-1],[1,0],[3,7],[0,1],[0,2],[1,3],[0,1],[0,1],[1,3],[1,11],[1,3],[0,1],[2,0],[0,1],[1,1],[0,1],[0,2],[-1,7],[-1,5],[0,1],[-1,-2],[0,1],[0,1],[0,2],[1,10],[0,2],[0,1],[1,-1],[0,-2],[0,-1],[1,1],[0,2],[0,1],[0,1],[0,1],[0,1],[-1,4],[0,1],[0,2],[0,1],[1,0],[0,1],[1,1],[0,4],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[-1,0],[0,1],[0,1],[0,1],[0,3],[0,1],[0,1],[-1,1],[0,1],[0,1],[0,1],[0,1],[1,3],[0,1],[0,2],[0,2],[0,2],[0,3],[-1,1],[-1,-6],[-1,1],[0,6],[0,1],[-1,0],[0,1],[0,-1],[0,-2],[-1,0],[0,2],[-1,2],[0,1],[0,7],[0,1],[0,1],[-1,2],[0,1],[0,1],[0,2],[0,1],[0,2],[2,12],[0,3]],[[6179,3348],[-1,-1],[0,-2],[0,-1],[0,-3],[1,-2],[0,-2],[0,-1],[1,-1],[0,1],[0,1],[0,1],[0,2],[0,2],[0,1],[0,1],[-1,3],[0,1]],[[6271,3150],[1,-8],[1,-1],[1,3],[1,2],[0,3],[1,1],[1,-1],[1,-4],[2,-3],[0,-1],[0,-1],[0,-2],[0,-1],[1,-1],[1,-5],[1,-1],[0,-1],[0,-1],[0,-1],[0,-3],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-1],[0,-1],[1,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[2,-3],[0,-1],[0,-1],[0,-2],[0,-3],[0,-2],[0,-1],[0,-2],[0,-1],[0,1],[1,1],[1,-1],[0,-1],[0,-9],[1,-6],[0,-2],[0,-2],[0,-1],[2,-4],[0,-1],[1,-1],[0,-5],[0,-1],[0,-1],[0,-1],[-1,0],[0,-1],[0,-1],[0,-1],[0,-10],[0,-2],[1,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-3],[0,-2],[1,-4],[1,-2],[1,-1],[1,2],[1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[1,0],[1,-5],[0,-1],[1,-2],[0,-3],[1,-4],[1,-1],[0,-5],[1,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-6],[0,-2],[1,-1],[0,-2],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[-1,-2],[0,-1],[-1,-2],[0,-2],[0,-4],[0,-6],[1,-2],[0,-1],[1,4],[1,0],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-4],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-6],[0,-2],[0,-1],[0,-22],[0,-6],[0,-9],[0,-5],[-1,-4],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[1,-1],[1,-4],[0,-2],[0,-1],[0,-1],[0,-5],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[-1,0],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-1],[-1,-1],[0,-1],[-1,-2],[0,-5],[0,-2],[0,-2],[0,-9],[0,-2],[0,-3],[0,-1]],[[6300,2809],[-2,1],[-1,2],[0,1],[-1,3],[0,2],[0,2],[-1,0],[-1,0],[0,1],[0,1],[0,2],[0,1],[0,5],[0,2],[0,2],[0,2],[0,1],[-1,0],[0,-1],[0,-2],[0,-2],[-1,-1],[0,-1],[-1,-2],[-2,-2],[-1,-3],[-2,1],[-1,-1],[0,-3],[0,-1],[0,-1],[0,-2],[0,-3],[0,-1],[0,-3],[-1,0],[-1,-2],[0,-1],[0,-1],[-1,1],[-1,-1],[0,-1],[0,-1],[0,-9],[0,-1],[0,-1],[-1,2],[0,1],[-1,-1],[0,-2],[0,-2],[0,-2],[0,-1],[-1,-2],[-1,-5],[-1,-2],[0,-1],[0,-1],[1,-1],[0,-2],[0,-1],[1,-1],[0,-2],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-4],[-3,5],[-2,3],[0,1],[-2,4],[0,1],[-1,1],[0,1],[0,3],[0,2],[1,2],[0,2],[0,1],[0,1],[0,2],[-1,0],[0,-1],[0,-1],[-1,0],[-1,1],[0,-1],[-1,-4],[0,-1],[0,-1],[-2,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-5],[0,-1],[-1,-2],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-2],[-1,-2],[-1,4],[-1,0],[0,-1],[-1,-2],[-1,-2],[0,-1],[-1,0],[0,1],[-2,5],[0,1],[-1,0],[-1,-3],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[1,-3],[0,-1],[0,-1],[0,-1],[0,-2],[-1,-3],[0,-1],[-1,0],[0,2],[0,1],[-1,6],[0,2],[-1,2],[-1,1],[0,1],[0,1],[-1,6],[0,6],[0,2],[-1,2],[0,1],[-2,4],[-2,2],[-1,2],[-1,4],[-1,6],[-1,6],[-1,-5],[0,-1],[-1,-2],[1,-5],[0,-1],[0,-2],[1,-2],[-1,-1],[0,-3],[0,-2],[0,-1],[0,-1],[0,-1],[1,-2],[0,-2],[-1,-1],[-1,1],[0,1],[-1,-2],[0,-1],[-1,0],[0,-2],[0,-2],[-1,-7],[0,-2],[0,-1],[-1,-3],[0,-1],[0,-1],[0,-2],[0,-1],[0,-3],[-3,6],[-5,2],[-3,7],[-1,4],[0,2],[-1,0],[0,1],[-1,0],[0,2],[-2,4],[0,1],[-2,-3],[0,-1],[-1,-1],[-3,3],[-1,-1],[-1,-3],[0,-1],[-2,-2],[0,-1],[-1,0],[-1,0],[0,1],[0,2],[0,4],[0,1],[0,2],[-1,1],[0,2],[-3,4],[0,-1],[-1,-1],[0,-2],[0,-6],[-1,-3],[-1,-2]],[[6196,2756],[-2,20],[0,5],[2,1],[0,2],[1,1],[-1,0],[0,3],[-1,1],[0,1],[0,3],[0,1],[0,1],[3,7],[0,2],[1,8],[0,1],[0,1],[1,1],[0,1],[0,1],[0,2],[0,1],[0,9],[-1,9],[-1,8],[-1,7],[-1,6],[0,3],[-1,2],[0,3],[-1,3],[0,5],[-2,14],[0,5],[-1,2],[-1,4],[-3,6],[0,3],[-2,16],[-1,6],[-1,2],[-1,0],[-1,0],[-1,11],[0,1],[0,2],[1,1],[0,1],[1,2],[0,2],[-1,1],[-1,7],[-3,27],[-1,6],[-1,0],[0,-1],[0,-1],[-1,0],[-1,0],[0,1],[0,1],[0,2],[0,1],[0,2],[0,4],[0,2],[2,4],[0,1],[0,1],[0,1],[-1,3],[0,3],[-1,2],[0,3],[-2,3],[0,2],[0,1],[0,2],[0,2],[-1,0],[0,1],[-1,-2],[0,1],[0,1],[0,2],[0,24],[0,2],[0,1],[0,2],[0,1],[0,1],[0,1],[0,5],[0,2],[0,2],[0,2],[0,2],[0,2],[-1,1],[1,2],[0,1],[0,2],[0,1],[0,1],[1,1],[0,1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,4]],[[6170,3103],[2,2],[3,7]],[[6170,3103],[-1,-2],[-1,1],[-1,3],[-8,1],[-1,-1],[-1,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[-1,0],[-2,5],[-1,-2],[0,-3],[-1,-1],[-2,1],[-1,0],[0,-1],[0,-2],[-1,-2],[0,-1],[0,-2],[0,-3],[0,-20],[0,-1],[0,-3],[0,-12],[0,-6],[0,-3],[0,-2],[0,-2],[-1,-2],[0,-1],[-1,-2],[0,-1],[-1,-2],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-2],[0,-6],[-1,-2],[0,-1],[0,-1],[-1,0],[-2,3],[0,1],[0,2],[0,1],[0,3],[0,2],[-1,2],[0,1],[-3,3],[-1,1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[-1,0],[0,1],[0,1],[-1,3],[0,4],[-1,4],[0,1],[-1,2],[0,2],[0,2],[-1,1],[-1,-5],[-1,-5],[0,-2],[0,-1],[0,-1],[-1,1],[-3,-12],[0,-2],[0,-1],[-1,9],[0,1],[0,1],[-1,0],[0,-2],[0,-6],[0,-4],[0,-13],[0,-2],[-1,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,1],[0,1],[-1,2],[0,3],[0,1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,0],[-1,-1],[0,-2],[0,-1],[-1,2]],[[6116,3471],[1,5],[0,1],[0,1],[2,-1],[1,1],[0,1],[0,-1],[1,-2],[0,-1],[-1,0],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,-1],[1,-1],[1,1],[3,10],[0,-1],[1,-1],[0,-2],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[1,-4],[0,-1],[0,-1],[0,-1],[0,-1],[1,-3],[0,-1],[0,-5],[0,-1],[0,-1],[1,-1],[0,1],[1,1],[1,0],[0,-1],[1,-1],[1,4],[0,1],[1,0],[1,1],[1,2],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[1,0],[1,-1],[1,-1],[0,-1],[0,-5],[1,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-2],[1,-2],[0,-6],[1,-3],[0,11],[0,2],[0,1],[1,1],[1,1],[0,2],[1,0],[2,-2],[1,-2],[0,-2],[0,-2],[1,-3],[1,-2],[0,-2],[0,-5],[1,-1],[0,-1],[1,0],[0,1],[0,1],[0,1],[1,1],[0,1],[0,1],[0,1],[2,-4],[1,-2],[1,-2],[0,-2],[0,-2],[1,0],[0,1],[1,2],[0,1],[2,-2]],[[6196,2756],[-1,-2],[0,-2],[-1,-2],[0,-2],[0,-1],[-1,2],[0,1],[-1,0],[0,-2],[0,-11],[0,-5],[-1,-4],[0,-5],[-1,-7],[-1,-2],[-2,-9],[0,-3],[0,-2],[0,-2],[0,-9],[0,-5],[0,-3],[0,-5],[0,-6],[1,-4],[0,-2],[0,-4],[-1,-2],[-1,-2],[0,-2],[0,-1],[-1,-1],[0,-3],[0,-3],[0,-10],[0,-3],[0,-4],[0,-3],[0,-4],[-1,-7],[0,-4],[0,-2],[-4,-27],[-1,-2],[0,-2],[0,-2],[0,-1],[0,-1],[1,-7],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2]],[[6180,2561],[-1,5],[-1,2],[-1,0],[0,-1],[0,-1],[0,-1],[0,-2],[-1,-2],[-1,-3],[-1,1],[-1,2],[1,1],[0,2],[-1,2],[0,3],[0,2],[0,3],[0,3],[0,5],[0,3],[-1,2],[0,2],[-1,1],[-1,-1],[0,-1],[0,-1],[-1,0],[0,-5],[1,-7],[-1,-4],[0,-1],[0,-1],[-1,1],[0,1],[0,2],[0,1],[0,1],[-1,0],[-2,-2],[-2,2],[-1,4],[0,2],[0,1],[0,1],[0,2],[0,2],[-1,0],[0,1],[-1,-2],[-1,0],[0,1],[-1,4],[0,1],[0,1],[-2,1],[0,1],[-1,4],[-1,1],[-1,1],[0,1],[0,2],[0,2],[-1,2],[0,-1],[-1,-1],[0,-1],[0,-4],[0,-1],[-1,-1],[-1,2],[-2,6],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-3],[0,-1],[0,-2],[0,-1],[-4,-1],[-1,-1],[0,-3],[-1,-1],[0,1],[-1,4],[-1,3],[-1,0],[-1,0],[-1,0],[0,1],[0,1],[0,3],[1,7],[0,3],[0,1],[0,2],[0,3],[0,1],[0,1],[1,5],[0,3],[0,5],[-2,5],[-1,2],[-1,-1],[0,-2],[0,-1],[0,-1],[-1,0],[0,1],[0,1],[0,1],[0,3],[0,1],[-1,2],[0,1],[0,1],[-7,3]],[[6180,2561],[1,-8],[1,-7],[4,-17],[2,-7],[0,-4],[2,-13],[0,-1],[0,-2],[0,-2],[0,-4],[-1,-1],[0,-2],[1,-4],[0,-6],[0,-3],[0,-2],[1,-4],[0,-1],[0,-1],[0,-1],[-1,-3]],[[6114,2064],[0,1],[-3,4],[-1,0],[0,-1],[-2,-10],[-1,-2],[0,-2],[-1,-1],[-1,0],[-1,2],[-1,7],[-1,4],[-1,1],[-1,4],[-1,8],[0,11],[0,5],[-1,3],[-1,1],[-1,1],[-1,4],[-1,-1],[-1,-3],[-1,-2],[-1,2],[-1,3],[-1,5],[-1,23]],[[6594,3722],[0,-3],[0,-1],[0,-3],[0,-1],[0,-6],[0,-5],[0,-2],[0,-2],[1,-8],[1,-3],[0,-3],[0,-2],[0,-2],[0,-2],[-1,-1],[0,-2],[0,-2],[0,-1],[0,-1],[1,-2],[2,-5],[2,0],[1,8],[1,1],[0,1],[1,0],[2,3],[2,-2],[0,-1],[0,-1],[1,-3],[0,-2],[3,0],[1,1],[0,3],[1,3],[4,-9],[1,-1],[0,3],[0,1],[1,-1],[0,-1],[0,-1],[0,-3],[0,-5],[0,-2],[0,-1],[1,-1],[1,1],[0,1],[0,2],[0,3],[0,1],[0,1],[1,2],[0,1],[0,1],[1,8],[0,1],[0,1],[1,2],[2,1],[0,-1],[1,-2],[0,-1],[0,-2],[1,-3],[1,-2],[1,-3],[0,-1],[0,-1],[0,-5],[0,-1],[1,-1],[0,-2],[2,0],[0,-2],[3,2],[1,1],[0,1],[0,1],[1,3],[0,2],[0,2],[0,2],[1,6],[0,3],[0,3],[0,2]],[[6639,3682],[1,-3],[0,-1],[2,0],[1,-1],[0,-1],[0,-3],[0,-1],[1,2],[2,0],[0,-1],[1,-2],[0,-2],[-1,-6],[0,-3],[0,-4],[0,-2],[0,-1],[-2,-7],[-1,-3],[0,-1],[0,-1],[0,-3],[1,-12],[1,-7],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-2],[-1,-3],[0,-2],[0,-1],[1,-2],[0,-3],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-3],[0,-1],[-1,-1],[1,-1],[0,-2],[0,-1],[2,-1],[1,-1],[1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-4],[-1,-4],[0,-3],[0,-3],[0,-1],[0,-2],[2,-10],[0,-1],[0,-1],[-3,-6],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[1,0],[3,8],[0,1],[0,3],[1,1],[0,1],[0,1],[1,-1],[0,-1],[0,-1],[0,-2],[0,-2],[1,-5],[0,-1],[4,2],[0,-2],[0,-3],[0,-1],[0,-2],[0,-2],[-1,-4],[-1,-2],[0,-2],[0,-2],[0,-1],[0,-1],[-2,-4],[0,-1],[0,-2],[-1,-1],[0,-1],[0,1],[-1,1],[0,2],[-1,0],[-1,-2],[-1,-3],[-1,-1],[0,-1],[-2,0],[0,-2],[-1,-2],[0,-5],[0,-6],[0,-2],[0,-3],[0,-2],[0,-8],[-1,-2],[0,-1],[-1,-4],[-3,-14],[0,-1],[-1,1],[0,1],[0,1],[0,2],[0,2],[1,4],[0,2],[0,1],[0,2],[0,1],[-1,0],[0,4],[-1,0],[-1,-1],[-1,0],[0,-2],[0,-2],[0,-2],[0,-1],[2,-1],[0,-1],[0,-2],[0,-1],[-1,-1],[0,-2],[-1,0],[-3,2],[0,-1],[0,-2],[-1,-4],[1,-2],[0,-1],[1,-2],[1,-1],[0,-1],[0,-1],[0,-1],[-2,-10],[0,-2],[0,-1],[0,-1],[-1,0],[0,2],[0,1],[-1,7],[0,1],[0,-1],[-1,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[-1,-1],[0,-2],[0,-1],[0,-2],[1,-1],[0,-1],[0,-1],[2,-2],[2,-1],[0,-1],[0,-1],[1,0],[0,-1],[0,-1],[1,-15],[0,-1],[0,-1],[0,-2],[-5,-18],[-1,-1],[-1,1],[-1,2],[0,1],[0,1],[0,8],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[-1,-1],[-2,-5],[0,-1],[-1,0],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[1,6],[0,1],[1,1],[-1,1],[0,1],[0,1],[-1,3],[0,1],[0,1],[0,1],[0,1],[0,1],[0,5],[0,1],[0,2],[0,2],[0,2],[0,1],[0,1],[0,1],[-2,3],[0,1],[-1,-1],[-3,-13],[-1,-1],[0,-1],[0,-2],[0,-1],[1,-1],[1,-1],[0,-1],[0,-2],[-1,-2],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-2],[-1,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-2],[0,-2],[0,-5],[0,-1],[-1,-5],[-1,-4],[-1,0],[0,1],[0,1],[0,1],[0,1],[-1,0],[0,2],[0,2],[1,3],[0,5],[1,2],[0,3],[0,1],[0,1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,3],[0,1],[0,2],[0,3],[0,1],[0,2],[0,1],[0,5],[-1,2],[0,5],[-1,6],[0,1],[0,1],[1,1],[0,1],[0,1],[1,-1],[0,-1],[0,-1],[0,-1],[1,0],[2,5],[0,1],[0,2],[0,1],[0,1],[0,1],[0,2],[0,2],[-1,3],[-1,2],[0,1],[0,2],[0,1],[0,2],[-1,1],[0,1],[-3,4],[-5,2],[-1,-1],[-1,1],[-2,5],[-1,2],[-1,1],[-1,-1],[-1,-1],[-1,0],[0,-3],[-2,-12],[-1,-2],[-1,1],[0,-1],[0,-1],[0,-3],[0,-2],[0,-2],[0,-1],[0,-1],[-1,0],[0,3],[0,-1],[0,-2],[0,-2],[0,-1],[0,-3],[0,-3],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-2],[-2,-3],[0,-1],[0,-1],[0,-2],[0,-2],[-1,-1],[0,-2],[0,-2],[1,-6],[1,-13],[0,-4],[1,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-1],[-1,-9],[-1,-4],[0,-1],[0,-2],[0,-1],[0,-1],[0,-3],[0,-2],[0,-1],[0,-2],[0,-2],[1,-4],[1,-5],[1,-2],[1,-4],[1,-2],[1,-1],[0,-1],[0,-3],[0,-1],[0,-1],[2,-2],[1,-3],[1,-1],[0,-1],[0,-2],[0,-3],[0,-1],[1,0],[0,1],[1,1],[0,-2],[0,-4],[0,-2],[0,-1],[0,-1],[0,-1],[4,-12],[1,-2],[0,-1],[-1,-12],[0,-1],[0,-2],[0,-1],[0,-1],[1,-1],[1,-1],[0,-1],[0,-1],[0,-1],[1,-2],[0,-2],[0,-1],[0,-1],[1,-1],[0,-1],[0,-1],[0,-5],[0,-1],[0,-1],[1,-1],[0,-1],[1,-2],[0,-1],[1,0],[0,1],[0,2],[1,0],[2,10],[0,1],[0,1],[0,1],[2,6],[1,1],[0,2],[0,1],[0,1],[0,1],[0,1],[0,1],[4,7],[1,1],[2,-5],[2,-2],[0,-1],[0,-1],[2,-8],[1,-2],[0,1],[2,8],[0,1],[1,3],[0,1],[0,2],[0,2],[2,6],[0,1],[2,2],[3,6],[1,2],[1,4],[2,-3],[3,-2],[0,2],[0,2],[1,3],[1,9],[1,2],[1,5],[1,1],[0,2],[0,6],[1,3],[1,7],[0,1],[0,1],[1,3],[0,2],[0,5],[0,1],[0,2],[1,2],[0,1],[1,2],[0,1],[1,3],[0,8],[3,-9],[1,-1],[0,-2],[1,-4],[1,3],[1,1],[0,1],[1,-2],[0,-4],[0,-1],[0,-1],[0,-2],[0,-1],[1,-4],[0,-1],[0,-1],[0,-3],[-1,-9],[0,-3],[-1,-5],[0,-2],[0,-3],[0,-9],[0,-4],[-1,-7],[0,-3],[-2,-9],[0,-2],[-2,-2],[0,-1],[0,-1],[-1,-5],[0,-3],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[2,-5],[0,-1],[0,-2],[0,-12],[-1,-3],[0,-1],[-1,-8],[0,-1],[0,-1],[0,-2],[0,-1],[3,-6],[0,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-3],[1,-2],[0,-2],[-1,-1],[0,-3],[-1,-7],[0,-1],[-1,-1],[0,-2],[0,-2],[1,-4],[0,-3],[0,-2],[-1,-1],[0,-2],[-1,1],[0,2],[-1,2],[0,2],[-3,9],[0,1],[-1,-1],[-2,-4],[-1,0],[-1,2],[-1,-1],[0,-2],[-1,-1],[0,-3],[0,-5],[-2,-24],[-1,-12],[0,-2],[-1,-1],[0,-1],[-2,0],[-1,-3],[0,-1],[-1,0],[0,1],[-1,2],[0,1],[0,-1],[0,-1],[-1,-8],[0,-1],[0,-2],[0,-1],[0,-2],[1,-1],[0,-1],[0,-1],[-1,-3],[0,-2],[0,-3],[0,-1],[0,-2],[0,-3],[0,-9],[0,-2],[0,-2],[-1,-6],[0,-1],[0,-2],[0,-3],[0,-8],[1,-10],[0,-1],[0,-2],[-1,-2],[0,-1],[-1,-8],[0,-1],[-1,-1],[0,-1],[0,-3],[0,-5],[0,-1],[1,-1],[0,1],[1,0],[1,-3],[-1,-11],[0,-6],[-1,-36],[0,-3],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-2],[0,-6],[0,-10],[0,-2],[0,-1],[0,-2],[1,-1],[1,0],[0,-1],[0,-2],[0,-2],[0,-7],[0,-20],[1,-1],[0,-1],[0,-1],[2,0],[0,-1],[0,-1],[1,-7],[-1,0],[-4,-2],[0,-1],[0,-1],[0,-4],[0,-2],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-4],[-1,-4],[-1,-10],[0,-3],[0,-2],[0,-2],[1,-7],[0,-1],[0,-6],[0,-1],[0,-3],[0,-3],[0,-6],[0,-4],[0,-2],[0,-1],[0,-3],[-1,-3],[0,-3],[0,-1],[0,-1],[0,-2],[0,-2],[0,-4],[0,-3],[0,-1],[2,-2],[0,-1],[0,-1],[0,-7],[0,-2],[1,-1],[0,-1],[0,-1],[0,-1],[-2,-2],[0,-1],[0,-2],[0,-2],[0,-1],[0,-13],[1,-3],[0,-1]],[[6633,2764],[0,-3],[0,-10],[0,-1],[0,-2],[0,-2],[0,-1],[1,-1],[1,-1],[1,-5],[0,-1],[0,-1],[0,-1],[1,-3],[0,-1],[0,-2],[-1,-1],[0,-3],[-1,-1],[0,-1],[-1,-1],[-1,0],[-3,1],[-1,-4],[0,-2],[0,-1],[-2,-14],[0,-3],[0,-1],[0,-1],[0,-2],[1,-1],[0,-1],[0,-2],[0,-3],[1,-5],[0,-1],[0,-2],[1,-3],[0,-5],[0,-3],[0,-1],[-1,-1],[-1,-4],[0,-2],[0,-1],[0,-4],[0,-6],[0,-9],[0,-6],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-12],[0,-1],[0,-3],[-1,-3],[0,-1],[0,-1],[0,-1],[0,-2],[-1,-2],[-1,0],[0,1],[0,1],[-1,1],[-1,0],[-1,-1],[0,-1],[0,-2],[0,-1],[-1,1],[0,1],[0,1],[0,1],[-1,0],[0,-1],[0,-3],[0,-1],[-1,0],[0,-2],[-1,-1],[-2,0],[-1,1],[0,1],[0,1],[0,3],[0,1],[0,1],[0,1],[0,1],[-1,4],[-1,0],[0,-2],[-1,-1],[-1,0],[0,1],[0,1],[0,2],[0,1],[0,1],[0,2],[-1,1],[0,1],[0,1],[0,1],[-2,3],[-1,0],[-1,-4],[0,-1],[-1,1],[-1,1],[0,1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[-1,1],[-1,0],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-2],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-3],[0,-1],[0,-2],[0,-1],[-2,-3],[0,-1],[0,-2],[-1,-3],[0,-1],[-1,0],[-1,2],[-1,0],[0,-1],[0,-2],[0,-5],[0,-2],[0,-2],[0,-3],[0,-2],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[-2,-3],[0,-1],[0,-3],[-1,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[-1,1],[0,2],[0,1],[-1,1],[0,10],[0,4],[-1,1],[-1,2],[-1,0],[0,1],[0,1],[0,1],[0,1],[0,8],[0,3],[0,1],[0,1],[-1,1],[-1,1],[-1,-2],[-1,1],[0,1],[-1,4],[-1,-1],[0,-2],[0,-1],[0,-1],[0,-1],[2,-8],[0,-1],[0,-1],[1,-1],[0,-1],[0,-2],[-1,-3],[0,-3],[0,-1],[1,-2],[0,-2],[0,-1],[0,-2],[-1,-1],[0,-2],[0,-1],[-2,-2],[-1,3],[-1,6],[1,3],[0,2],[0,3],[0,1],[0,3],[-1,4],[0,4],[0,3],[0,1],[0,1],[0,1],[1,1],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,2],[-1,1],[0,1],[0,1],[-1,3],[-1,1],[-1,1],[0,1],[0,3],[0,1],[0,7],[0,6],[0,1],[0,1],[0,1],[2,0],[1,1],[0,1],[0,1],[0,1],[0,4],[0,1],[0,1],[-1,-2],[0,1],[-1,2],[0,2],[0,4],[0,1],[0,1],[0,2],[0,5],[1,11],[0,2],[0,2],[-1,1],[0,1],[-1,0],[0,-1],[0,-1],[-1,0],[0,2],[0,1],[1,11],[0,2],[0,1],[-2,3],[0,2],[-1,1],[0,1],[0,1],[-1,-1],[0,-1],[-1,0],[0,1],[0,1],[-1,1],[1,1],[0,1],[0,1],[1,2],[0,1],[0,1],[0,1],[-1,-1],[-1,-2],[0,-3],[-1,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[-1,-1],[-1,-1],[0,1],[-1,3],[0,2],[0,1],[-1,7],[0,2],[0,1],[0,1],[0,1],[1,1],[0,2],[1,3],[1,2],[0,4],[1,6],[0,1],[1,2],[2,3],[0,1],[0,1],[0,1],[0,1],[0,2],[0,2],[0,4],[0,2],[0,8],[0,2],[0,1],[0,2],[0,4],[0,2],[0,1],[0,1],[-1,0],[0,1],[0,1],[0,3],[-1,1],[0,5],[0,3],[0,2],[0,1],[0,2],[0,2],[0,1],[0,1],[0,1],[1,2],[0,3],[0,1],[0,1],[0,1],[0,1],[1,1],[0,2],[0,1],[0,1],[0,2],[0,1],[-1,0],[0,1],[0,-1],[-2,-7],[-1,-5],[-1,1],[0,1],[-2,7],[-1,7],[0,2],[0,2],[0,1],[-1,1],[0,1],[0,1],[-1,-1],[-1,0],[0,-1],[0,-1],[0,-2],[1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-3],[0,-3],[-1,-5],[-1,-3],[0,-1],[0,-1],[-1,-2],[0,-3],[0,-2],[-1,1],[0,1],[0,1],[0,-2],[0,-4],[-1,-11],[0,-5],[0,-4],[0,-1],[0,-2],[0,-2],[-1,-6],[0,-1],[-1,1],[0,-1],[-1,-1],[0,-4],[0,-1],[-1,0],[0,1],[0,1],[0,1],[-1,1],[0,1],[0,2],[0,2],[0,2],[0,1],[-1,0],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[-1,-1],[-3,5],[0,1],[0,1],[0,3],[0,2],[0,2],[0,8],[0,1],[0,1],[0,2],[0,2],[0,1],[0,1],[0,3],[0,1],[0,1],[0,1],[-1,1],[0,1],[0,1],[0,2],[0,1],[1,-1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[1,1],[-1,1],[0,1],[0,1],[0,1],[0,1],[-2,-3],[0,2],[0,2],[0,3],[0,1],[0,1],[0,1],[-1,4],[0,1],[0,2],[-1,0],[-1,-1],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-1],[-1,1],[-1,2],[0,3],[0,3],[1,2],[0,1],[0,1],[0,1],[0,2],[0,3],[0,2],[0,1],[0,2],[0,1],[0,1],[0,1],[1,0],[0,-1],[0,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-1],[1,-1],[0,-1],[1,0],[0,2],[1,1],[0,1],[0,2],[0,1],[0,2],[0,1],[0,2],[-1,1],[-1,1],[0,2],[0,3],[0,1],[0,2],[0,1],[0,1],[1,0],[1,-1],[0,1],[0,1],[0,3],[0,4],[0,2],[-1,1],[0,1],[0,1],[0,1],[-1,3],[-3,1],[0,-3],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-1],[-1,0],[-1,3],[0,2],[0,1],[0,5],[0,8],[0,1],[0,1],[-1,1],[-1,6],[0,2],[0,2],[-1,9],[0,3],[0,4],[0,2],[-2,17],[0,3],[0,8],[0,1],[-1,5],[1,2],[0,2],[0,1],[0,1],[-1,4],[0,1],[0,2],[0,1],[-2,4],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[-1,0],[-2,-5],[-2,-4],[-1,1],[0,4],[-1,1],[-1,0],[0,1],[0,2],[0,2],[0,2],[0,2],[0,8],[0,2],[0,1],[-1,5],[-6,12],[-1,0],[0,-1],[0,-1],[0,-4],[-1,-2],[0,-1],[-1,1],[-3,10],[0,2],[0,2],[-1,6],[0,5],[0,1],[0,2],[0,2],[-1,2],[0,1],[-1,1],[0,1],[-1,9],[-2,25],[-1,7],[-1,2],[-3,9],[-1,3],[0,1],[0,1],[0,3],[0,2],[0,3],[-1,14],[0,4],[-1,6],[-3,35],[0,5]],[[6485,3160],[-1,5],[0,3],[0,4],[0,3],[0,2],[0,2],[0,1],[0,5],[0,3],[0,2],[0,2],[0,3],[0,2],[-1,6],[0,2],[0,3],[1,6],[-1,2],[0,1],[0,3],[-1,7],[0,4],[-1,2],[0,2],[0,8],[0,2],[0,2],[0,1],[1,2],[0,2],[0,2],[0,7],[0,2],[1,1],[0,1],[0,2],[0,1],[0,4],[0,1],[0,1],[0,2],[0,1],[0,4],[0,4],[0,2],[0,1],[1,1],[0,1],[0,2],[2,6],[0,2],[1,12],[0,2],[1,4],[0,2],[0,1],[0,4],[0,2],[0,1],[0,1],[0,1],[0,1],[0,1],[0,2],[0,2],[0,3],[0,2],[0,1],[0,6],[0,1],[0,2],[1,19],[-1,1],[-2,-3],[-1,0],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[-1,3],[0,10],[-1,1],[0,2],[0,1],[0,3],[-1,1],[-1,0],[0,-1],[-1,1],[0,1],[0,2],[-1,-1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[-1,2],[0,4],[-1,1],[-1,3],[-1,1],[0,2],[1,3],[0,1],[0,2],[0,4],[0,1],[3,3],[2,1],[0,1],[0,1],[0,1],[0,2],[0,2],[1,4],[1,2],[0,1],[1,-1],[1,-3],[0,-1],[1,0],[1,0],[1,1],[0,1],[0,1],[0,1],[0,2],[0,4],[0,1],[0,2],[0,1],[2,6],[0,2],[1,1],[0,1],[0,4],[0,2],[0,2],[1,1],[3,7],[1,4],[1,3],[0,4],[1,4],[0,6],[0,2],[0,2],[1,3],[0,2],[0,2],[0,2],[0,2],[0,5],[0,3],[0,1],[0,1],[1,1],[1,3],[1,6],[1,3],[0,1],[0,1],[0,1],[1,0],[0,1],[0,1],[0,2],[1,3],[0,2],[0,1],[1,7],[0,1],[0,1],[-1,1],[-1,2],[0,1],[0,1],[0,2],[0,1],[-1,5],[-1,2],[0,3],[0,1],[-1,-2],[0,-2],[-1,2],[1,3],[0,4],[-1,2],[0,1],[-1,-5],[-1,-1],[0,2],[0,2],[1,6],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[-2,-2],[0,1],[-1,4],[0,2],[0,1],[0,2],[-1,-2],[-1,1],[0,1],[-1,0],[0,-1],[0,-2],[0,-1],[0,-1],[-1,0],[0,3],[-1,1],[-1,5],[-1,1]],[[6489,3618],[1,7],[0,3],[0,1],[0,2],[1,1],[2,2],[1,2],[0,4],[0,1],[0,6],[1,3],[0,1],[1,-2],[2,-3],[2,2],[1,7],[1,16],[0,5],[1,4],[1,7],[1,1],[1,8],[1,2],[1,3],[1,2],[0,5],[1,3],[0,3],[0,2],[-1,1],[0,1],[0,1],[0,1],[-1,1],[0,3],[0,2],[0,1],[1,1],[0,1]],[[6509,3729],[1,0],[1,-1],[0,-3],[0,-1],[0,1],[0,2],[1,7],[0,2],[0,2],[0,2],[0,2],[1,1],[1,24],[0,5],[1,1],[1,-2],[0,-5],[1,-1],[1,-2],[0,-1],[0,-1],[0,-1],[0,-2],[1,-7],[1,-2],[0,-1],[1,-2],[2,-6],[1,0],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[1,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-4],[0,-1],[0,-3],[0,-1],[1,0],[3,5],[1,0],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,-1],[0,1],[0,2],[0,4],[0,1],[1,1],[0,2],[1,-1],[0,-1],[0,-2],[0,-1],[0,-1],[2,-2],[3,1],[1,2],[0,4],[1,7],[1,2],[0,1],[4,5],[1,3],[0,1],[1,1],[0,-1],[1,-1],[1,-1],[1,1],[0,1],[1,4],[1,4],[0,-2],[0,-6],[0,-2],[0,-2],[1,-9],[0,-2],[0,-2],[2,-9],[0,-8],[1,-2],[1,2],[2,0],[0,3],[1,0],[0,-1],[0,-1],[1,-1],[0,1],[1,1],[0,1],[0,2],[0,2],[1,5],[1,1],[0,1],[0,1],[0,1],[0,1],[0,2],[0,1],[1,2],[0,-1],[1,-1],[0,-1],[3,2],[0,-3],[0,-1],[0,-2],[0,-4],[-1,-4],[0,-1],[0,-1],[2,-13],[0,-1],[1,0],[0,-1],[0,-4],[1,-3],[0,-1],[0,-1],[2,1],[1,-1],[0,-1],[0,-7],[0,-3],[0,-1],[1,-2],[1,-3],[0,-1],[0,-1],[0,-6],[1,-1],[0,-2],[0,-1],[1,0],[0,-1],[1,0],[0,1],[2,6],[0,1],[1,2],[0,1],[0,1],[0,1],[3,2],[0,1],[1,2],[1,3],[0,1],[0,2],[-1,-1],[0,1],[0,1],[0,3],[0,1],[1,2],[0,2],[1,1],[0,1],[0,2],[0,3],[0,1],[0,2],[0,1],[0,4],[0,1],[0,1],[0,1],[1,0],[1,1],[2,5]],[[6724,3674],[0,-1],[1,0],[0,-1],[0,-1],[0,-1],[-1,-6],[0,-2],[1,-2],[0,-2],[0,-1],[2,-7],[0,-2],[1,-1],[0,-2],[0,-2],[-1,-3],[0,-1],[0,-2],[0,-3],[0,-3],[0,-2],[0,-2],[1,-2],[1,-1],[0,1],[0,1],[0,2],[0,1],[0,3],[0,1],[0,1],[1,1],[0,1],[0,-2],[1,-4],[0,-7],[0,-3],[0,-3],[-1,-3],[-1,0],[0,-1],[0,-9],[0,-2],[0,-2],[1,-3],[0,2],[0,1],[1,1],[0,1],[0,-1],[1,-1],[0,-2],[0,-1],[1,0],[0,-1],[0,-2],[1,-2],[0,-2],[1,-8],[1,-3],[0,-2],[0,-1],[0,-1],[0,-3],[0,-5],[0,-5],[0,-2],[0,-1],[0,-1],[0,-1],[1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[-1,-3],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[-1,0],[-1,-2],[0,-1],[0,-1],[0,-2],[1,-1],[1,2],[0,-1],[1,-4],[1,-7],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-3],[-1,-1],[-2,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[1,-2],[0,-2],[0,-2],[-1,-1],[-2,3],[0,-1],[-2,-7],[-1,1],[-1,6],[-1,3],[0,-1],[-1,-5],[0,-3],[0,-1],[0,-1],[2,-6],[0,-2],[0,-1],[0,-3],[0,-1],[0,-1],[-1,-3],[-1,0],[-1,-3],[-2,-2],[0,-1],[0,-3],[-1,-3],[0,-2],[1,-1],[3,-7],[0,-2],[0,-1],[0,-2],[1,-1],[1,0],[0,-2],[0,-2],[0,-1],[0,-2],[-1,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[1,-4],[0,-1],[0,1],[1,1],[0,-1],[0,-6],[0,-4],[0,-1],[-1,-2],[-1,-3],[0,-1],[0,-2],[0,-2],[0,-4],[0,-3],[0,-4],[0,-1],[-1,0],[0,1],[-1,10],[0,1],[-1,0],[-1,-3],[-1,1],[-1,0],[0,-3],[0,-1],[0,-3],[0,-2],[0,-16],[0,-5],[1,0],[0,-1],[0,-1],[1,0],[0,-1],[0,-1],[0,-1],[0,-1],[1,-2],[0,-1],[0,-1],[0,-3],[0,-2],[0,-2],[0,-1],[1,-4],[0,-1],[0,-2],[0,-1],[0,-1],[-2,-6],[0,-1],[0,-2],[-1,-4],[0,-3],[0,-3],[0,-1],[0,-2],[1,-1],[1,-1],[0,-1],[0,-1],[0,-2],[0,-6],[0,-3],[2,-2],[0,1],[0,2],[0,3],[0,1],[1,1],[1,-1],[0,-1],[0,-2],[0,-1],[1,-1],[0,-1],[3,5],[0,1],[0,1],[1,7],[0,2],[0,1],[0,1],[1,0],[0,-1],[0,-1],[0,-1],[2,4],[1,0],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[1,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[1,0],[0,1],[0,1],[1,5],[0,1],[0,1],[1,0],[0,-2],[0,-1],[1,-2],[0,-1],[0,-1],[0,-2],[2,4],[2,4],[2,9],[2,-2],[5,1],[1,-1],[0,-2],[0,-6],[1,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[1,0],[0,-2],[1,-2],[0,-7],[0,-3],[-1,-1],[0,-2],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,1],[-1,1],[0,1],[0,2],[0,1],[0,1],[-1,0],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-2],[1,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[-1,0],[0,-1],[0,-1],[0,-1],[0,-2],[0,-3],[0,-5],[0,-2],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[1,-1],[0,-1],[2,2],[0,-2],[0,-2],[1,-12],[0,-1],[1,-1],[0,1],[1,2],[0,1],[0,-1],[1,-1],[0,-1],[0,-2],[0,-2],[-1,-6],[0,-5],[0,-5]],[[6758,3189],[-3,4],[-1,0],[-1,-1],[-1,-4],[1,-5],[0,-3],[0,-4],[-1,-4],[0,-2],[-3,2],[-2,-4],[-1,1],[-1,3],[-1,0],[-1,-3],[-3,-1],[-1,-1],[-1,-5],[-1,-6],[0,-9],[0,-1],[1,-4],[-1,-7],[-1,-3],[0,-2],[0,-2],[1,-8],[0,-1],[-1,-1],[-1,1],[0,2],[0,2],[-1,4],[-1,4],[-1,4],[0,5],[1,3],[1,4],[0,6],[0,2],[-1,1],[-11,-3],[0,-4],[0,-2],[0,-15],[-3,1],[-1,2],[-1,2],[0,2],[0,4],[0,4],[-1,1],[0,1],[-1,0],[0,-4],[-1,0],[0,2],[-3,0],[0,-4],[0,-4],[-1,-2],[0,-1],[-1,1],[0,2],[0,8],[-2,3],[0,2],[-1,9],[-1,2],[-2,2],[-1,-1],[0,-4],[-1,-4],[0,-2],[-2,0],[0,-5],[0,-2],[1,-8],[0,-4],[0,-4],[0,-5],[-1,0],[0,3],[-1,2],[0,3],[0,2],[0,1],[-2,1],[-2,-1],[0,-2],[-1,-4],[0,-3],[0,-2],[0,-3],[0,-3],[1,-1],[2,0],[0,-1],[0,-1],[1,-4],[1,-3],[1,-2],[0,-3],[0,-6],[0,-4],[-1,-4],[-1,-4],[0,-6],[0,-3],[-1,0],[-1,-2],[0,-3],[0,-3],[0,-3],[0,-3],[-1,-4],[-3,5],[-1,-6],[1,-5],[4,-1],[1,-2],[1,-2],[0,-4],[0,-2],[1,-1],[4,-5],[1,0],[1,9],[1,2],[3,0],[0,-1],[0,-3],[1,-16],[-1,-4],[-1,-3],[-1,-8],[-1,-3],[-1,0],[0,4],[-1,5],[-1,5],[0,3],[-1,1],[-1,-1],[0,-3],[0,-7],[0,-3],[-1,-4],[-1,-9],[0,-6],[-1,-6],[1,-6],[0,-6],[1,-6],[2,-1],[2,0],[1,-2],[3,-12],[1,-1],[1,0],[3,8],[0,-1],[1,-9],[2,-8],[2,-5],[6,-6],[1,-3],[1,-6],[0,-6],[0,-6],[-1,-12],[0,-5],[-1,-4],[-3,-5],[-1,1],[-1,8],[-1,2],[-3,-2],[-1,1],[-1,-1],[-1,-3],[0,-3],[-1,2],[-2,9],[-1,5],[-1,2],[0,-1],[-2,-6],[-1,-2],[-1,1],[-1,1],[-1,4],[-1,1],[-1,0],[-1,-5],[-1,-3],[-1,-2],[-2,0],[2,-12],[-4,-24],[-1,-3],[-1,-7],[0,-3],[-1,-3],[-1,-2],[-2,-4],[0,-10],[0,-3],[0,-2],[0,-3],[0,-1],[0,-1],[1,-2],[0,-1],[0,-2],[0,-2],[1,0],[1,1],[0,-2],[1,-4],[0,-5],[0,-11],[0,-4],[0,-3],[3,-4],[0,-2],[0,-2],[0,-4],[1,-10],[0,-5],[0,-3],[2,-13],[-1,-5],[-1,-4],[-1,-2],[-1,-9],[-1,-2],[-2,-6],[-1,-3],[-1,-10],[-1,-6],[-2,-1],[-1,2],[-2,1],[-2,-6],[-2,-10],[-4,-23],[-5,0]],[[6667,2656],[0,2],[-1,8],[0,1],[0,1],[0,1],[-1,1],[-1,2],[0,1],[0,2],[0,3],[0,4],[0,2],[0,1],[0,2],[0,1],[0,1],[1,2],[0,2],[0,4],[0,1],[0,1],[0,2],[0,1],[2,1],[0,1],[0,1],[0,2],[0,3],[1,2],[0,1],[0,1],[1,2],[0,2],[0,1],[0,1],[0,1],[1,1],[0,1],[0,3],[0,1],[0,2],[0,3],[0,3],[0,3],[0,4],[0,3],[0,4],[0,9],[0,3],[0,1],[0,1],[-1,1],[-1,0],[-4,-1],[0,1],[-1,2],[0,2],[0,6],[0,2],[-1,1],[-1,2],[-1,0],[0,-1],[0,-1],[-1,-2],[0,-8],[-1,-2],[0,-1],[0,-1],[-1,0],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[-3,0],[0,-1],[-1,0],[0,-4],[0,-1],[-1,0],[-1,6],[-1,7],[0,2],[-1,0],[-1,-1],[-1,0],[-1,1],[-1,-1],[0,-1],[0,-3],[0,-12],[-1,-24],[0,-3],[0,-2],[0,-5],[-1,-2],[0,2],[-1,4],[-1,7],[-1,4],[-2,4],[0,1],[0,1],[-1,1],[0,2],[0,1],[0,2],[1,2],[0,3],[0,1],[0,1],[0,1],[-1,1],[-2,4],[0,1],[-1,-1],[0,-2],[-1,-2]],[[6639,3682],[0,2],[-1,1],[0,2],[0,5],[0,2],[2,-1],[1,-2],[1,0],[0,2],[0,6],[1,3],[0,2],[0,1],[0,1],[-1,4],[0,4],[-1,1],[0,1],[0,2],[1,1],[0,2],[1,0],[0,1],[0,1],[1,4],[0,1],[1,0],[1,-3],[1,-1],[0,-1],[0,-2],[1,-1],[0,-1],[10,-3],[0,-2],[-1,-5],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[3,-3],[1,0],[1,3],[2,5],[1,-1],[2,-6],[1,-2],[2,-2],[1,-1],[0,-1],[0,-1],[1,0],[1,0],[0,1],[1,2],[0,3],[1,1],[1,0],[2,0],[1,1],[2,-1],[0,-1],[1,-5],[3,6],[1,0],[1,0],[0,-1],[0,-1],[0,-1],[0,-1],[1,-1],[1,-1],[2,-4],[0,-2],[1,0],[0,1],[1,3],[1,-1],[0,1],[0,1],[1,0],[0,-1],[1,-1],[2,4],[0,1],[0,2],[0,6],[0,2],[0,1],[-1,7],[-1,8],[0,3],[2,9],[2,-6],[1,-4],[0,-1],[0,1],[1,0],[0,1],[0,4],[1,1],[0,1],[0,1],[1,1],[0,-1],[1,-2],[0,-1],[1,-4],[0,-1],[0,-1],[0,-1],[1,0],[0,-1],[1,-5],[0,-1],[1,0],[0,1],[1,3],[0,1],[1,1],[1,-1],[2,-9],[0,-1],[0,-2],[0,-2],[0,-3],[1,-4],[1,2],[0,1],[0,-1],[1,-5],[1,-2],[0,-1],[0,-1],[0,-2],[-1,-1],[0,-1],[0,-1],[1,-1],[3,-8],[0,-1],[3,-2]],[[6805,3830],[0,-2],[0,-1],[-1,-3],[0,-2],[0,-3],[1,-6],[0,-3],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[1,-1],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-3],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-3],[0,-3],[0,-1],[0,-1],[-1,0],[0,-1],[0,-1],[0,-1],[0,-5],[0,-1],[0,-1],[1,-2],[2,-9],[0,-2],[0,-1],[0,-1],[0,-1],[1,-2],[0,-1],[0,-2],[0,-3],[1,-1],[0,-2],[0,-1],[0,-1],[1,-1],[1,-1],[0,-2],[0,-1],[-1,0],[0,-1],[-1,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[1,-8],[1,-1],[0,-1],[2,0],[1,3],[0,1],[1,0],[9,-7],[1,-13],[0,-3],[1,-3],[0,-3],[1,-9],[1,-2],[0,-3],[0,-1],[0,-2],[2,-12],[3,-14],[1,-2],[0,-2],[0,-1],[1,1],[1,1],[0,2],[1,6],[1,2],[0,5],[0,1],[0,1],[1,0],[0,1],[2,-1],[1,-1],[0,-1],[0,-2],[0,-1],[1,-2],[0,-1],[1,-6],[0,-1],[1,-5],[0,-2],[0,-4],[1,-2],[1,0],[3,8],[1,5],[1,2],[1,7],[0,1],[0,1],[1,0],[1,-1],[0,-1],[0,-2],[0,-1],[0,-5],[0,-1],[0,-1],[1,-8],[0,-1],[0,-1],[2,0],[1,1],[0,1],[1,3],[1,2],[2,-4],[2,-1],[0,-1],[0,-3],[1,-1],[2,3],[1,0],[1,0],[0,-1],[0,-3],[0,-3],[0,-4],[0,-5],[4,-39],[1,-2],[2,-2],[0,-2],[1,0],[6,2],[1,-1],[0,-4],[1,-4],[0,-3],[1,-2],[1,0],[0,1],[5,10],[1,0],[0,-2],[0,-6],[0,-19],[0,-3],[0,-1],[1,0],[0,-1],[0,-2],[1,-1],[0,-1],[1,1],[1,0],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[0,-3],[0,-6],[1,-1],[0,-1],[0,-2],[2,-2],[0,-1],[0,-2],[0,-1],[1,-6],[0,-4],[0,-1],[0,-1],[0,-1],[1,0],[0,1],[3,4],[0,-1],[1,-6],[0,-1],[0,-3],[2,-2],[1,-5]],[[6909,3481],[-1,1],[0,2],[-1,-5],[0,-4],[0,-4],[0,-5],[0,-4],[-1,-4],[-1,-3],[0,-2],[0,-5],[0,-3],[0,-2],[-1,1],[-1,-1],[-2,-3],[-2,3],[-2,-3],[-1,1],[0,1],[-1,1],[0,-1],[-2,-5],[0,-1],[1,-3],[-1,-7],[2,-6],[2,-7],[0,-5],[-1,-1],[-1,-1],[0,-2],[-1,-4],[0,-3],[0,-2],[0,-2],[-1,-6],[0,-3],[-5,-3],[-2,6],[-1,1],[-1,-2],[-1,-3],[0,-3],[-2,-10],[-1,-8],[-2,-3],[-5,1],[-2,-2],[-2,3],[-1,-2],[0,-4],[-1,-6],[-1,-4],[-1,-1],[-2,0],[-1,-1],[-1,-5],[-1,-1],[-6,-1],[0,-2],[-1,-6],[-1,-2],[-2,-1],[-3,0],[-2,-1],[-10,-18],[-3,-7],[0,4],[-1,17],[0,2],[-2,-4],[-1,-4],[-1,-6],[-1,-7],[0,-4],[-1,-1],[-4,7],[-3,2],[-1,-2],[0,-3],[1,-9],[-1,-5],[0,-1],[-2,2],[-1,0],[-3,-8],[-1,-3],[0,-6],[0,-1],[0,-4],[0,-5],[1,-27],[-1,-8],[-1,2],[-1,-2],[0,-1],[-1,1],[0,2],[-1,2],[-2,15],[-2,4],[-1,-5],[-2,-7],[-2,-5],[-1,1],[-1,2],[-1,8],[-1,2],[-1,-4],[-1,-1],[-1,1],[0,1],[0,-1],[0,-2],[-1,-2],[-3,-2],[0,2],[-1,0],[0,-1],[0,-1],[-1,-2],[-1,-1],[-1,-3],[-1,-1],[-1,2],[-1,-1],[0,-3],[0,-1],[-1,0],[0,1],[-1,3],[0,-1],[0,-2],[0,-1],[0,-2],[0,-4],[1,-6],[-1,-7],[-1,-2],[-1,-1],[0,-2],[-1,-3],[0,1],[-1,1],[-2,9],[-1,2],[-1,1],[-5,-11],[-1,-1],[-1,-2],[-2,-6],[0,-2],[-1,0],[-1,0]],[[6724,3674],[0,5],[0,3],[-1,6],[0,2],[0,1],[0,3],[1,7],[0,1],[0,2],[-1,2],[0,2],[0,2],[1,2],[1,0],[1,7],[1,1],[0,4],[2,6],[0,1],[0,2],[0,5],[0,5],[0,2],[0,1],[1,2],[1,8],[0,1],[0,1],[1,0],[0,2],[1,1],[0,6],[4,16],[1,-3],[0,-1],[0,-1],[0,-5],[0,-1],[0,-2],[0,-4],[0,-1],[0,-1],[1,-1],[0,1],[0,1],[0,1],[0,1],[1,0],[1,-2],[0,1],[1,11],[1,1],[2,-2],[1,0],[0,1],[0,1],[0,2],[0,2],[0,2],[0,1],[1,0],[1,2],[1,1],[3,-7],[1,-1],[0,1],[1,0],[3,1],[1,1],[0,-1],[0,-3],[0,-1],[0,-2],[0,-1],[0,-3],[1,-3],[0,-2],[0,1],[0,3],[0,6],[0,1],[0,1],[1,4],[0,1],[0,2],[0,2],[0,1],[0,1],[0,2],[1,2],[1,-3],[1,1],[1,1],[0,-2],[1,-2],[1,-1],[0,-2],[0,-5],[0,-1],[0,-2],[0,-2],[2,-3],[1,1],[1,0],[0,-1],[0,-2],[0,-1],[1,-4],[2,-5],[1,0],[2,-3],[0,1],[0,1],[1,1],[1,1],[2,-4],[2,-2],[1,1],[0,2],[1,6],[0,2],[0,2],[0,1],[-1,1],[0,6],[0,3],[0,1],[0,5],[0,1],[0,2],[1,1],[1,-1],[1,1],[0,1],[0,2],[0,1],[0,2],[0,1],[0,2],[0,1],[1,1],[0,2],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[-1,0],[0,1],[0,3],[0,1],[0,2],[0,3],[0,1],[0,2],[0,1],[0,1],[5,2],[0,2],[1,4],[0,4],[0,1],[0,1],[3,2],[3,-1],[0,1],[1,3],[1,3],[1,0],[0,-2],[1,-3],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[2,-4],[1,-1]],[[6508,6669],[0,-1],[1,0],[0,-1],[0,-4],[0,-4],[-1,-3],[0,-1],[-3,-3],[0,1],[0,1],[0,2],[0,1],[0,1],[0,2],[0,3],[1,3],[2,6],[1,1],[0,-1],[-1,-1],[0,-2]],[[6498,6664],[-1,-2],[-1,2],[0,3],[-1,3],[0,2],[0,2],[0,2],[1,2],[1,2],[1,-1],[1,-3],[0,-5],[-1,-7]],[[6421,6704],[-2,-8],[-3,0],[-2,-7],[-1,-1],[-1,0],[0,-2],[-1,-1],[-1,0],[1,2],[0,2],[0,2],[1,6],[9,18],[3,-2],[6,-1],[-1,-4],[-8,-4]],[[6364,6756],[0,-2],[-10,7],[0,2],[1,0],[7,-5],[1,0],[1,-2]],[[6376,6787],[-2,-2],[0,1],[2,5],[0,1],[1,1],[0,-1],[0,-2],[-1,-3]],[[6510,6784],[0,-1],[-4,4],[-1,1],[0,2],[0,1],[0,2],[1,0],[1,-1],[3,-6],[0,-2]],[[6382,6807],[-1,0],[0,1],[1,2],[1,3],[1,1],[0,-1],[-2,-5],[0,-1]],[[6528,6817],[-1,-1],[-1,1],[-1,2],[0,1],[0,2],[0,1],[1,0],[1,-1],[1,-2],[0,-3]],[[6540,6820],[1,-4],[-4,3],[-3,-3],[-1,0],[1,3],[3,5],[2,-1],[1,-3]],[[6390,6823],[-2,-3],[0,1],[2,3],[0,2],[1,2],[0,-2],[-1,-3]],[[6517,6827],[-1,-1],[-1,1],[0,1],[1,3],[0,6],[1,2],[0,1],[1,-1],[1,-2],[0,-4],[-1,-3],[0,-2],[-1,-1]],[[6645,6875],[0,-2],[0,2],[-1,2],[0,1],[-1,1],[0,1],[-1,4],[0,2],[-1,8],[-1,3],[-8,21],[0,3],[0,1],[1,-1],[9,-21],[2,-6],[0,-1],[-1,-2],[0,-1],[1,-2],[1,-4],[0,-3],[0,-4],[0,-2]],[[6396,6884],[1,-10],[0,-10],[-2,-13],[-2,-11],[-2,-4],[1,3],[0,1],[0,1],[0,6],[0,6],[2,10],[0,10],[0,2],[0,1],[-1,-1],[-1,-3],[-3,-12],[-1,-2],[-1,-2],[-2,-12],[-1,-4],[1,0],[1,0],[0,-4],[-1,-3],[-2,-4],[-3,-19],[0,-2],[-2,-1],[-1,-3],[-2,-7],[-3,-6],[-2,1],[-1,-1],[0,-2],[-2,-6],[-12,-12],[-1,0],[0,2],[0,1],[0,1],[-1,2],[0,1],[-1,-1],[0,-3],[0,-2],[-3,-1],[-1,3],[-1,8],[-3,9],[-1,1],[0,1],[-1,5],[0,1],[0,1],[-1,4],[1,4],[0,1],[-1,3],[0,1],[-1,-1],[-1,-3],[0,-1],[0,4],[0,3],[0,4],[0,5],[0,4],[1,3],[0,2],[0,3],[-1,0],[0,-1],[0,1],[0,2],[0,1],[0,2],[0,1],[1,6],[-1,2],[0,1],[0,2],[0,1],[0,1],[0,6],[1,6],[0,3],[0,1],[0,8],[0,2],[0,5],[0,7],[1,7],[0,5],[1,6],[2,10],[1,5],[0,3],[2,4],[1,8],[7,18],[3,3],[5,2],[5,-2],[12,-27],[9,-24],[4,-10],[1,-7],[2,-10]],[[6806,6904],[0,-3],[0,-2],[-1,-8],[0,-1],[0,-1],[0,-1],[-1,-3],[-2,-3],[-2,-8],[0,-3],[-1,0],[-1,-1],[0,-1],[0,-1],[-2,-11],[-2,-9],[-2,-3],[0,-1],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-2],[1,-1],[0,-2],[1,-4],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-4],[0,-2],[0,-1],[0,-5],[0,-2],[1,-2],[1,-1],[1,1],[2,-2],[1,0],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[1,1],[0,-1],[2,-3],[0,-1],[1,0],[1,-2],[0,-3],[1,-2],[1,-1],[0,-1],[2,-5],[0,-2],[1,0],[2,3],[1,0],[1,-3],[1,0],[1,-1],[0,-3],[0,-1],[0,-1],[0,-9],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[1,0],[1,2],[0,-1],[0,-1],[1,-4],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[1,-1],[0,-2],[1,-1],[0,-5],[1,-6],[0,-1],[0,-1],[0,-1],[1,-1],[2,-1],[0,-1],[1,-2],[0,-1],[0,-1],[0,-1],[0,-2],[-1,-3],[0,-1],[0,-3],[0,-4],[0,-1],[0,-1],[-1,-2],[-1,-5],[-1,-5],[-1,-3]],[[6818,6697],[-2,0],[-3,-4],[-7,-2],[0,-1],[-1,-1],[0,-1],[0,-1],[-3,4],[-1,0],[-1,-1],[-1,0],[0,1],[-1,0],[-1,-3],[-2,-3],[-2,-5],[-1,-2],[0,-2],[-1,-4],[0,-2],[0,-1],[0,-3],[0,-2],[0,-1],[0,-1],[1,-1],[0,-1],[-1,-4],[0,-1],[0,-2],[1,-1],[0,-1],[0,-5],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,0],[-1,2],[-1,-1],[-2,-7],[-1,-1],[1,-1],[0,-2],[-1,-2],[0,-3],[0,-1],[-1,-1],[0,-1],[-6,-9],[0,-2],[-1,-3],[-3,-6],[-2,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-2],[1,-1],[0,-1],[0,-1],[1,-7],[1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[1,-1],[0,-2],[0,-1],[0,-1],[-1,-6],[-1,-6],[-1,-1],[-1,-2],[-1,0],[-1,-1],[-1,-2],[0,-2],[0,-1],[0,-3],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[-6,2],[-1,-1],[-3,-5],[0,-2],[0,-2],[0,-1],[0,-1],[-1,-1],[-3,-7],[-2,-4],[0,-4],[-1,-1],[-5,-14],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-4],[0,-2],[-1,-4],[0,-3],[0,-2],[0,-4],[0,-5],[0,-1],[0,-1],[0,-2],[0,-2],[0,-3],[0,-6],[0,-2],[-1,-1],[-27,-64],[0,-2],[0,-1],[0,-1],[-3,-7],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[-1,-3],[0,-2],[0,-1],[0,-1],[-1,-2],[-1,-1],[-3,2],[-1,-1],[-1,-1],[0,-1],[0,-2],[-38,-2],[-4,-9],[-1,0],[0,2],[-1,1],[0,1],[1,2],[1,6],[-32,-1],[-139,0],[-42,22],[-3,0],[-2,-8],[-1,-1],[0,-3],[0,-1],[-1,-2],[-1,-2],[-1,0],[0,-3],[-1,0],[-1,-1],[-1,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[-2,1],[0,-1],[0,-2],[0,-1],[-1,-1],[-1,-2],[-1,0],[-2,2],[-1,-1],[0,-1],[-1,-2],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[-1,-4],[0,-1],[0,-2],[1,-1],[0,-1],[0,-1],[1,-3],[0,-1],[-1,-4],[-69,-163]],[[6228,6200],[0,1],[0,9],[0,1],[3,12],[1,2],[0,3],[0,6],[0,14],[1,5],[1,5],[1,3],[2,2],[0,3],[-1,0],[0,-1],[-1,1],[-1,3],[0,3],[0,8],[-2,7],[0,5],[0,6],[3,3],[0,6],[0,12],[0,3],[1,4],[-1,0],[0,2],[0,3],[0,3],[-1,7],[-1,3],[0,4],[-1,5],[-2,6],[0,5],[1,2],[1,2],[0,3],[-2,-4],[-1,2],[-1,3],[-1,4],[-1,2],[0,3],[-2,4],[-1,2],[-2,-1],[-1,0],[-1,-1],[-1,3],[-2,1],[-1,7],[0,4],[0,4],[0,10],[1,16],[2,20],[4,37],[0,10],[0,2],[0,5],[0,3],[1,3],[0,3],[0,1],[1,1],[0,-2],[0,-1],[0,-2],[0,-1],[1,0],[0,1],[1,3],[0,3],[0,3],[-1,1],[-1,0],[0,1],[0,4],[1,6],[0,3],[0,3],[0,3],[0,3],[0,3],[0,2],[-1,2],[0,2],[0,3],[0,4],[0,4],[0,2],[3,-1],[0,1],[1,2],[1,5],[0,1],[-1,3],[-2,-3],[-1,2],[0,10],[1,2],[1,0],[1,2],[-1,1],[-1,-1],[-1,1],[1,4],[5,11],[2,2],[-1,2],[-2,-1],[-1,-3],[-1,-4],[-1,-1],[-1,5],[0,8],[1,6],[0,6],[0,2],[0,5],[0,2],[1,14],[0,6],[0,6],[0,4],[-2,10],[-12,34],[-11,34],[-1,6],[-1,5],[2,2],[1,1],[3,0],[4,-7],[1,-1],[1,-3],[2,-2],[1,-3],[0,-2],[1,-1],[1,-1],[3,-9],[7,-4],[8,5],[12,-1],[10,-2],[8,-7],[8,-14],[2,-3],[0,-5],[0,-2],[0,-5],[0,-2],[1,-1],[0,-3],[1,-2],[0,-2],[0,-6],[0,-2],[0,-1],[1,-2],[3,-10],[1,-8],[2,-6],[0,-1],[2,1],[0,-1],[1,-3],[1,-2],[1,-5],[2,-9],[0,-3],[-1,-1],[-1,-1],[2,0],[0,-1],[0,-2],[0,-2],[0,-1],[0,-3],[0,-8],[0,-3],[1,-4],[0,-4],[1,-3],[0,-1],[1,-3],[0,-2],[0,-2],[0,-2],[0,-2],[1,-5],[1,-6],[0,-2],[0,-3],[0,-7],[0,-7],[-1,-2],[-3,1],[-2,1],[0,1],[-2,0],[-1,0],[-5,-4],[-6,-9],[-10,-5],[-5,-4],[-3,0],[-1,-4],[0,-5],[0,-2],[1,1],[0,-1],[0,-5],[0,-4],[0,-5],[0,-3],[-1,-6],[-1,-2],[-3,0],[-2,-5],[-2,-8],[-2,-10],[-1,-12],[0,-5],[0,-4],[0,-4],[0,-4],[0,-5],[1,-8],[3,-9],[3,-7],[2,-5],[2,-5],[3,-4],[3,-3],[2,-7],[1,-12],[0,-6],[1,-5],[3,-23],[1,-5],[0,-7],[0,-5],[1,-3],[3,-5],[1,-2],[1,-5],[0,-1],[0,-1],[8,2],[5,10],[1,0],[1,-2],[1,-3],[0,-5],[-1,-2],[-1,-1],[-1,-8],[0,-2],[-3,-9],[0,-1],[2,2],[1,4],[1,8],[1,3],[7,7],[3,0],[1,5],[2,-1],[10,17],[2,0],[2,-6],[1,-3],[0,-7],[0,2],[0,1],[1,4],[-1,4],[0,3],[0,3],[1,3],[0,3],[2,7],[0,1],[1,3],[1,7],[0,9],[0,13],[0,3],[-1,3],[-1,7],[0,2],[0,4],[0,2],[2,9],[1,2],[0,3],[0,3],[-1,5],[0,2],[0,3],[0,5],[0,1],[1,5],[3,10],[1,6],[0,5],[0,5],[0,6],[-3,17],[0,5],[0,5],[5,12],[1,1],[1,-3],[0,-3],[0,-3],[0,-3],[0,-3],[1,3],[1,6],[1,6],[1,3],[1,4],[1,2],[2,0],[2,-5],[1,-1],[1,1],[4,11],[1,1],[1,-4],[-1,-1],[0,-1],[6,-1],[2,-6],[-1,-3],[1,-2],[1,-3],[2,0],[0,2],[0,2],[-1,1],[0,1],[0,3],[0,1],[-2,6],[-1,1],[-1,1],[-1,6],[0,1],[-2,-2],[-1,0],[0,4],[0,3],[1,2],[0,3],[0,4],[0,10],[-6,25],[1,3],[1,-3],[4,-13],[1,-4],[4,-3],[4,3],[1,3],[4,6],[6,13],[4,7],[4,13],[4,13],[2,1],[-1,-7],[-3,-6],[-1,-6],[4,11],[2,7],[0,4],[1,5],[3,11],[2,3],[2,7],[2,3],[6,19],[1,8],[1,3],[1,7],[1,2],[0,-1],[2,-6],[1,-2],[2,-1],[3,3],[0,1],[2,5],[0,1],[1,0],[4,5],[3,7],[0,1],[2,-3],[2,4],[0,2],[-1,2],[1,7],[1,2],[3,0],[9,13],[4,10],[1,0],[-1,-6],[0,-4],[-2,-5],[-2,-2],[0,-1],[0,-2],[0,-3],[0,-1],[1,-1],[1,1],[2,4],[3,-1],[1,-1],[1,-5],[-1,0],[-1,0],[0,-3],[0,-2],[0,-2],[0,-2],[0,-3],[-1,-4],[-3,-3],[0,-4],[0,-2],[3,-13],[0,-1],[1,1],[1,3],[0,1],[2,0],[0,2],[1,3],[1,1],[3,12],[1,2],[2,1],[1,5],[0,4],[0,4],[0,4],[-1,3],[-1,2],[-2,3],[0,3],[-1,2],[0,2],[0,2],[0,2],[0,2],[-2,4],[1,3],[0,1],[0,3],[3,1],[0,1],[1,2],[0,3],[-1,2],[0,1],[-2,-3],[-1,-1],[0,-1],[-1,-2],[-1,-7],[0,-3],[-1,-1],[-3,1],[0,1],[0,5],[1,4],[1,4],[2,2],[10,27],[4,6],[4,10],[2,3],[3,8],[2,4],[3,6],[12,14],[6,6],[6,0],[1,3],[5,0],[1,-2],[0,-1],[0,-1],[0,-2],[-1,-1],[0,-1],[0,1],[-1,0],[-2,-1],[-2,-5],[-10,-5],[-10,-6],[-1,-2],[1,-2],[0,-1],[0,-4],[1,-3],[1,-1],[1,1],[0,2],[-1,1],[0,2],[1,3],[6,-5],[1,-2],[1,-6],[-2,-7],[-1,-1],[-1,-1],[0,-2],[-1,-8],[0,-1],[0,-1],[0,-4],[0,-7],[-1,-6],[-1,-4],[-1,-2],[0,-2],[0,-6],[1,-5],[1,-6],[0,-5],[2,-8],[0,-3],[1,-4],[0,-4],[0,-5],[0,-6],[0,-3],[1,-5],[0,-2],[-1,-3],[0,-1],[-1,-1],[-1,-2],[-1,-7],[0,-1],[-1,1],[-1,7],[-2,8],[-2,1],[0,-3],[1,-4],[1,-2],[0,-3],[0,-1],[1,-4],[-1,-3],[-1,1],[-2,6],[-3,3],[-1,-1],[0,-2],[-1,-5],[-1,2],[0,1],[-1,-1],[0,-4],[-2,-9],[-1,-6],[1,-4],[3,-3],[1,0],[3,4],[9,-2],[1,-2],[1,-4],[0,-1],[3,0],[2,1],[1,5],[2,13],[0,5],[1,0],[-1,-9],[0,-5],[0,-7],[1,-6],[1,0],[6,23],[1,2],[1,-1],[1,-5],[2,-23],[2,-4],[1,-2],[2,1],[1,1],[0,2],[1,3],[0,3],[0,3],[1,4],[0,4],[0,7],[-1,3],[0,2],[-1,5],[1,0],[1,1],[-1,4],[1,8],[0,2],[0,3],[2,10],[1,1],[1,0],[0,1],[1,7],[0,2],[2,2],[1,2],[0,1],[0,3],[0,1],[0,1],[5,10],[3,3],[3,-1],[3,1],[2,5],[5,14],[1,2],[1,-2],[3,-7],[2,-1],[8,2],[0,-2],[0,-9],[0,3],[2,10],[0,1],[1,0],[1,-1],[1,-1],[4,-15],[7,-6],[7,4],[1,4],[1,4],[1,12],[1,5],[2,6],[0,3],[1,8],[0,2],[9,2],[2,11],[0,2],[-1,1],[0,2],[0,4],[0,3],[1,2],[5,9],[2,1],[0,-1],[0,-3],[-1,-6],[0,-3],[0,-2],[0,-1],[1,-3],[0,-6],[1,-3],[0,-1],[2,-1],[1,-4],[-1,-5],[1,0],[1,2],[1,5],[-1,1],[-1,3],[0,1],[-2,0],[0,1],[-1,3],[0,2],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[0,3],[0,2],[0,2],[0,2],[1,2],[8,12],[8,11],[9,0],[0,-1],[1,-2],[-1,-2],[-1,2],[-1,-1],[0,-2],[-5,-3],[-1,-3],[-1,-5],[2,-1],[4,-6],[9,-34],[0,-2],[0,-10],[-1,-4],[-1,0],[-2,3],[-1,0],[-1,-2],[-1,-3],[-1,-5],[0,-6],[0,-2],[-1,-1],[-1,-2],[0,-2],[0,-2],[0,-3],[0,-5],[0,-3],[1,-2],[0,-1],[0,-3],[1,-3],[-1,-1],[0,-2],[0,-4],[-1,-5],[0,-6],[0,-5],[1,-5],[0,-3],[8,-9],[8,-8],[5,11],[1,6],[2,15],[0,1],[0,2],[0,10],[-1,5],[-2,11],[-1,6],[-1,3],[0,1],[-1,3],[1,8],[1,5],[3,8],[3,-7],[1,-2],[11,9],[2,6],[4,17],[1,3],[1,9],[0,3],[2,4],[1,3],[3,0],[-1,2],[0,1],[-1,2],[1,5],[0,4],[0,5],[0,5],[0,2],[0,5],[0,2],[0,7],[0,3],[-1,10],[-2,9],[-1,5],[-2,-1],[0,-2],[0,-3],[0,-3],[0,-2],[0,-1],[0,-2],[1,0],[-1,-4],[-2,0],[-1,3],[-1,4],[0,4],[0,4],[0,5],[0,14],[0,3],[0,5],[0,5],[-1,5],[-1,4],[-1,2],[-1,4],[0,2],[0,2],[0,5],[0,2],[0,2],[-3,9],[0,2],[-2,11],[-1,6],[0,2],[0,1],[0,2],[0,1],[0,2],[-1,1],[-1,2],[0,2],[0,6],[0,3],[0,3],[1,6],[1,4],[12,2],[1,1],[0,4],[-1,3],[0,2],[0,3],[0,2],[1,0],[0,-1],[1,0],[0,3],[0,1],[-1,4],[0,5],[0,5],[1,5],[0,4],[1,2],[5,5],[1,-2],[1,-4],[1,0],[0,1],[1,4],[1,0],[0,-1],[0,-4],[1,-2],[0,-1],[4,-2],[1,-2],[1,-4],[1,-2],[10,-7],[11,-4],[13,-1],[8,-7],[8,-8],[6,-3],[4,-8],[7,-11],[9,-10],[7,-17],[4,-8],[5,-16],[5,-12],[3,-8],[0,-3],[-1,-2],[-1,0],[-1,0],[-1,3],[-1,0],[0,-5],[0,-2],[-1,-4],[0,-3],[1,-4],[-1,-5],[-1,-5],[0,-5],[1,-4],[1,0],[0,3],[1,6],[1,2],[1,2],[0,3],[0,2],[0,3],[0,2],[0,1],[0,4],[0,2],[3,1]],[[6648,7170],[2,-16],[1,-3],[1,-1],[1,-2],[0,-2],[0,-9],[0,-3],[1,-2],[2,1],[2,-6],[1,-1],[1,-1],[0,-2],[0,-5],[0,-2],[0,-1],[1,-1],[3,-7],[1,0],[1,2],[1,-2],[1,-4],[0,-2],[1,-2],[6,-23],[0,-3],[2,1],[1,-2],[1,-8],[0,-3],[0,-3],[0,-4],[0,-4],[0,-1],[1,-2],[0,-2],[0,-1],[1,-2],[0,-3],[0,-1],[0,-2],[0,-2],[0,-1],[-1,0],[0,-1],[-1,-3],[-1,-3],[0,-4],[1,-4],[0,-3],[-3,2],[0,-2],[0,-2],[-3,-6],[-1,-1],[-3,7],[-2,2],[-1,-1],[-1,-2],[-1,-6],[0,-1],[0,-3],[-1,2],[0,2],[0,2],[-1,1],[-2,-1],[-6,5],[0,4],[0,3],[-1,2],[0,2],[0,5],[1,1],[1,0],[1,1],[0,4],[-1,3],[-4,12],[-1,6],[-1,1],[-1,0],[-1,-2],[-1,1],[-1,2],[-1,3],[-1,2],[-2,1],[-4,5],[0,-3],[1,-3],[0,-2],[2,-2],[2,-7],[-1,-3],[-1,-1],[-1,1],[0,3],[-1,0],[0,1],[-1,3],[0,1],[0,3],[-2,6],[-5,18],[-4,19],[1,8],[0,2],[-1,1],[0,2],[1,2],[0,1],[0,1],[0,1],[-1,0],[1,1],[0,4],[0,2],[-1,1],[0,-1],[-1,-1],[0,-1],[0,1],[0,2],[0,1],[-1,1],[-1,0],[0,3],[1,3],[0,3],[0,1],[-1,0],[-1,1],[0,5],[0,3],[1,0],[10,-14],[0,3],[0,4],[-1,3],[0,1],[-2,5],[-1,3],[0,2],[-4,8],[-1,0],[1,4],[1,-1],[2,-3],[0,3],[0,2],[1,2],[1,1],[1,1],[0,1],[0,3],[1,3],[-1,0],[0,1],[0,1],[0,2],[0,1],[0,2],[0,1],[1,1],[3,0],[0,1],[3,8],[1,2],[0,-3],[1,-3],[0,-3],[0,-1],[2,-2],[5,-14],[1,-5]],[[6924,6295],[4,3],[1,-2],[1,-11],[0,-4],[2,-7],[1,-2],[1,-2],[0,3],[-2,6],[0,3],[-1,3],[0,2],[1,2],[3,-2],[1,1],[1,-1],[1,-3],[-1,-2],[0,-2],[3,-1],[1,0],[0,-1],[0,-4],[1,0],[1,-1],[1,1],[1,0],[1,-5],[0,-2],[0,-7],[0,-1],[0,-1],[0,-1],[0,-2],[-1,0],[0,-2],[0,-4],[2,-9],[0,-4],[0,1],[-1,1],[-1,0],[0,-3],[1,0],[0,-3],[0,-4],[-1,-4],[0,-1],[-3,0],[-1,-2],[0,-1],[0,-2],[-1,-1],[0,1],[-1,2],[0,1],[-2,9],[-6,15],[0,3],[1,0],[0,2],[0,5],[0,2],[-1,5],[-1,12],[-2,15],[-1,1],[-1,1],[-2,-4],[0,-3],[-1,-1],[-1,3],[-2,4],[0,6],[0,3],[1,0],[3,-4]],[[6865,6941],[1,0],[2,4],[1,0],[0,-2],[-2,-6],[-1,-4],[-1,-3],[0,-2],[-1,-1],[-1,0],[0,2],[-3,4],[-1,6],[0,6],[0,5],[1,0],[0,-1],[0,-1],[0,-1],[0,-1],[1,-1],[4,-4]],[[6867,6944],[-1,0],[-4,4],[0,2],[-1,5],[0,5],[4,15],[1,3],[3,3],[1,-3],[0,-3],[0,-3],[0,-6],[-1,-9],[-1,-5],[0,-4],[0,-3],[-1,-1]],[[6857,7177],[-1,0],[-2,5],[-4,15],[-1,2],[0,5],[1,9],[1,3],[0,-1],[0,-4],[0,-3],[1,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[1,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[1,-1],[1,-8],[0,-1],[1,-1],[1,-3]],[[6849,7207],[-1,-1],[-1,2],[-1,7],[-1,8],[-1,7],[0,8],[-1,9],[0,7],[0,4],[1,6],[0,4],[0,4],[0,5],[1,3],[0,-1],[0,-5],[1,-1],[0,-2],[0,-3],[0,-2],[0,-3],[-1,-3],[0,-4],[0,-5],[1,-4],[0,-3],[0,-2],[1,-2],[0,-2],[0,-1],[0,-3],[1,-1],[0,-1],[0,-5],[0,-10],[1,-7],[0,-3]],[[6846,7285],[-1,0],[0,2],[0,4],[1,13],[1,7],[0,1],[0,1],[0,1],[0,1],[2,15],[1,-2],[0,-3],[-1,-3],[0,-2],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-2],[0,-1],[0,-2],[-1,-2],[0,-2],[0,-5],[-1,-4],[0,-3],[-1,-2]],[[7160,7650],[-6,-5],[-10,1],[-2,-3],[-6,4],[-1,4],[-1,6],[0,1],[1,5],[1,3],[9,22],[0,2],[0,3],[1,9],[1,7],[1,4],[3,4],[3,12],[1,2],[5,-2],[7,-9],[3,-1],[2,-4],[2,-4],[2,-10],[1,-5],[-1,-5],[-3,-4],[-2,-8],[-3,-6],[-2,-2],[-2,-4],[-3,-15],[-1,-2]],[[7180,7669],[0,-1],[-1,-3],[-1,-2],[0,-1],[0,-2],[1,-1],[1,-8],[1,0],[2,-2],[0,-1],[0,-2],[0,-2],[0,-1],[1,-3],[0,-2],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[1,-1],[-1,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[1,-2],[1,-1],[0,-1],[0,-1],[0,-1],[3,1],[0,-1],[0,-1],[0,-1],[3,-6],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[1,-1],[1,2],[1,0],[0,-1],[2,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[1,-1],[1,1],[2,0],[3,-5],[1,1],[1,-1],[0,-1],[0,-2],[1,-1],[1,1],[3,-1],[1,0],[2,4],[4,-8],[1,-1],[0,-1],[0,-3],[0,-1],[0,-1],[1,-2],[1,0],[2,-1],[1,-2],[0,-1],[0,-1],[1,-2],[0,-1],[1,-1],[2,-3],[0,1],[1,0],[0,-1],[0,-1],[0,-3],[1,-2],[0,-1],[1,-1],[0,-1],[1,-3],[0,-1],[0,-2],[0,-1],[-1,-1],[-1,1],[0,-1],[0,-2],[0,-2],[-3,3],[0,-2],[0,-1],[-2,-2],[-1,-1],[0,-1],[0,-2],[-1,-1],[0,-1],[1,-1],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[1,-3],[0,-1],[0,-2],[0,-1],[-1,0],[0,-1],[0,-2],[1,-14],[0,-3],[0,-3],[0,-1],[-1,-2],[0,-1],[-2,0],[0,-2],[0,-1],[0,-1],[-1,-2],[0,-2],[0,-1],[0,-1],[-3,-4],[-9,1],[-1,-1],[-1,0],[-2,3],[-1,0],[-1,-3],[0,-1],[1,-3],[0,-2],[0,-1],[0,-3],[0,-2],[1,-2],[-1,-2],[0,-2],[0,-7],[0,-1],[-1,-2],[0,-1],[0,-1],[-4,-9],[-1,-3],[0,-2],[0,-2],[0,-2],[0,-7],[0,-4],[1,-2],[0,-2],[1,-4],[0,-2],[1,-1],[2,0],[4,-4],[2,-6],[11,-35],[13,-11],[3,-6],[0,-1],[1,-2],[0,-2],[0,-9],[0,-1],[0,-2],[0,-7],[0,-4],[0,-10],[0,-2],[0,-3],[0,-3],[0,-2],[0,-2],[0,-2],[1,-1],[0,-2],[1,-4],[0,-1],[1,-4],[1,-1],[0,-1],[-1,-1],[0,-2],[0,-3],[-1,-4],[-1,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[2,-5],[1,0],[0,-1],[1,-3],[0,-1],[0,-2],[1,-2],[0,-1],[0,-2],[-1,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-2],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-2],[0,-4],[0,-1],[0,-2],[0,-1],[0,-2],[-1,-1],[0,-1],[-1,-1],[0,-2],[1,-1],[0,-1],[1,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[1,-1],[0,-1],[1,0],[1,-1],[1,0],[0,-1],[0,-1],[0,-2],[-1,-1],[-1,1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,-1],[-1,-1],[0,-1],[-1,1],[0,1],[0,3],[0,2],[0,1],[-1,0],[0,-1],[-1,1],[0,2],[-1,4],[0,2],[0,1],[-1,0],[-6,0],[0,-1],[-1,-1],[0,-1],[0,-2],[-1,-2],[0,-1],[-1,0],[-1,2],[-1,-1],[0,-2],[1,-2],[0,-2],[0,-2],[0,-1],[-1,-1],[-1,1],[0,1],[-1,-1],[0,-2],[0,-3],[0,-2],[0,-2],[0,-1],[-4,-10],[0,-1],[-2,-7],[-1,-3],[-1,0],[0,1],[-1,1],[0,1],[-1,1],[0,-1],[-1,-1],[0,-2],[1,-2],[0,-3],[0,-1],[-1,-1],[-1,-1],[-1,-4],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[-1,-3],[-2,-14],[0,-1],[-1,-1],[-1,1],[0,1],[-1,1],[0,1],[0,2],[0,1],[0,2],[0,1],[-1,0],[-1,0],[0,-1],[0,-1],[0,-1],[0,-5],[0,-2],[0,-1],[0,-1],[0,-2],[0,-3],[1,-4],[0,-2],[0,-2],[-1,0],[0,-1],[0,1],[0,1],[-1,1],[0,1],[-2,-2],[-1,0],[0,-1],[0,-2],[0,-1],[0,-2],[0,-3],[0,-1],[-1,1],[-1,-2],[1,-2],[0,-5],[1,-3],[0,-1],[-1,-1],[0,-1],[-1,-1],[0,-2],[0,-1],[-1,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-2],[-1,-4],[-1,-3],[-1,-1],[-1,1],[-1,0],[0,-1],[1,-2],[0,-2],[3,-6],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[1,-2],[1,-3],[1,-2],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[-3,-6],[0,-1],[0,-1],[2,-1],[0,-1],[0,-1],[1,-1],[1,0],[7,-8],[1,0],[0,1],[1,1],[0,-1],[1,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[2,-3],[0,-1],[0,-2],[0,-4],[1,-4],[1,-6],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-2],[1,-3],[0,-2],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[2,-1],[0,-1],[0,-1],[0,-3],[1,-1],[0,-2],[0,-1],[1,-4],[0,-1],[0,-1],[0,-1],[-1,-2],[1,-1],[1,1],[1,-1],[0,-2],[1,0],[0,1],[1,1],[0,1],[2,0],[1,2],[0,-1],[0,-1],[1,0],[2,1],[1,0],[6,-9],[7,-3],[1,-1],[0,-2],[0,-1],[0,-2],[0,-2],[1,-2],[0,-1],[3,-7],[0,-2],[0,-2],[1,-1],[0,-1],[6,7],[2,7],[2,5],[1,0],[0,-1],[3,-7],[2,2],[1,2],[1,4],[0,1],[1,1],[0,1],[0,1],[0,2],[0,1],[0,2],[-1,1],[0,1],[0,2],[1,3],[0,3],[0,1],[0,2],[0,3],[0,2],[0,2],[0,1],[0,1],[0,1],[1,5],[0,1],[0,1],[2,-2],[1,-2],[4,-17],[0,-1],[0,-2],[0,-2],[0,-1],[0,-2],[0,-2],[-1,-1],[-1,-5],[-1,-2],[0,-1],[-1,-6],[-1,-2],[0,-1],[0,-1],[1,-2],[2,-9],[0,-3],[1,-1],[10,-6],[1,0],[2,4],[1,0],[1,-1],[0,-1],[1,-1],[0,-2],[0,-5],[0,-2],[0,-2],[1,-1],[0,-1],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[-3,-2],[0,-1],[-1,0],[0,-2],[0,-2],[1,-5],[0,-2],[1,-1],[1,-1],[0,-1],[1,-1],[0,-2],[1,-2],[0,-3],[1,-12],[0,-4],[0,-9],[0,-2],[0,-6],[0,-2],[0,-1],[1,-2],[0,-2],[4,-6],[1,1],[1,1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[6,-14],[1,-2],[0,-1],[0,-2],[0,-1],[-3,-4],[0,-1],[-1,-2],[-1,-1],[-5,0],[-1,-3],[-1,-3],[-1,0],[-1,2],[0,-3],[0,-2],[-1,-1],[-1,-2],[0,-1],[0,-1],[2,-3],[2,-6],[2,-8],[0,-1],[0,-2],[0,-5],[1,-14],[0,-6],[-1,-4],[0,-2],[0,-1],[0,-2],[0,-2],[0,-3],[0,-3],[1,-4],[0,-1],[0,-2],[-1,-1],[0,-1],[-1,-1],[-1,-3],[-1,-2],[0,-1],[-1,-6],[0,-3],[0,-1],[0,-2],[0,-3],[0,-2],[0,-2],[0,-1],[0,-1],[-1,-1],[0,1],[-1,3],[-1,1],[-1,0],[0,-1],[0,-1],[0,-2],[0,-2],[0,-6],[0,-3],[0,-2],[-1,-2],[0,-2],[0,-2],[0,-2],[0,-1],[1,-3],[0,-1],[0,-2],[0,-2],[0,-1],[0,-2],[0,-2],[0,-6],[0,-1],[0,-1],[0,-1],[0,-5],[-1,-3],[0,-2],[0,-1],[-8,1],[-1,3],[-1,0],[-1,0],[0,-1],[-1,-4],[-1,-2],[-1,1],[-1,2],[-1,1],[-1,0],[0,-2],[0,-3],[-1,-1],[0,-2],[1,-2],[0,-5],[1,-1],[0,-2],[0,-2],[2,-8],[1,-9],[0,-3],[1,-1],[0,-1],[1,-1],[0,-1],[1,-3],[0,-2],[0,-1],[1,1],[0,1],[0,-1],[2,-6],[0,-3],[0,-1],[0,-2],[0,-1],[-1,-1],[0,-1],[1,0],[0,-1],[0,-1],[-1,-3],[0,-1],[-1,-1],[0,-2],[1,-1],[0,-3],[1,-3],[0,-4],[1,-2],[0,-1],[1,-1],[0,-2],[1,-2],[1,-5],[0,-1],[1,0],[1,3],[1,-1],[0,-2],[0,-4],[1,0],[0,-1],[0,-2],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[-1,1],[-1,-5],[0,-2],[0,-2],[0,-3],[-1,-1],[0,-1],[0,-1],[0,-3],[1,-6],[1,-5],[0,-1],[0,-1],[0,-1],[-1,0],[0,-1],[0,-1],[2,-2],[0,-1],[0,-1],[-2,-4],[0,-1],[-1,0],[0,1],[-1,0],[0,-1],[1,-3],[0,-1],[0,-1],[0,-1],[0,-1],[-1,2],[-1,0],[0,-2],[0,-3],[0,-1],[0,-1],[-1,-1],[-1,0],[0,-1],[0,-2],[19,-65],[7,-21],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[1,0],[1,0],[0,-1],[1,-5],[0,-2],[1,-2],[0,-3],[0,-7],[0,-1],[0,-1],[0,-2],[0,-1],[2,-5],[0,-2],[1,-5],[0,-1],[0,-2],[0,-2],[0,-2],[0,-2],[-1,0],[-1,-4],[0,-1],[-1,-1],[0,-2],[-2,-2],[-1,0],[0,-1],[0,-2],[0,-6],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-2],[0,-6],[0,-3],[0,-4],[0,-6],[0,-2],[1,-2],[0,-5],[1,-2],[3,-5],[0,-2],[0,-1],[1,-3],[1,-7],[1,-3],[1,-11],[0,-5],[1,-1],[3,-16],[0,-1],[0,-2],[0,-2],[-1,-1],[0,-2],[-1,-2],[-3,-4],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-1],[-1,-6],[0,-1],[0,-2],[0,-2],[1,-12],[1,-3],[0,-3],[4,-19],[0,-1],[0,-2],[0,-3],[0,-2],[0,-2],[0,-3],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[2,-11],[0,-1],[2,4],[2,3],[7,0],[1,-1],[0,-1],[1,-1],[0,-2],[0,-5],[1,-2],[1,-6],[1,-4],[0,-1],[0,-1],[1,-2],[0,-1],[3,-1],[0,-1],[1,-2],[-1,-1],[0,-5],[0,-1],[0,-5],[0,-1],[0,-1],[0,-1],[-1,-2],[-2,-1],[0,-1],[0,-1],[0,-1],[7,-10],[0,-1],[0,-2],[1,-4],[0,-2],[0,-2],[1,-2],[0,-7],[1,-7],[0,-2],[0,-2],[0,-2],[-1,-2],[0,-2],[0,-1],[0,-4],[0,-1],[0,-1],[1,-2],[0,-2],[0,-2],[0,-3],[0,-2],[1,-2],[0,-2],[0,-3],[0,-2],[0,-1],[-1,-6],[0,-2],[0,-1],[0,-1],[-1,-8],[0,-1],[0,-1],[-1,0],[-1,-1],[-1,-2],[0,-2],[-1,-2],[0,-7],[-1,-6],[-1,-6],[0,-1],[0,-2],[0,-2],[0,-2],[1,-4],[0,-2],[1,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-4],[0,-2],[0,-1],[-2,-4],[0,-2],[0,-1],[0,-2],[-1,-1],[0,-1],[0,-2],[0,-2],[0,-3],[1,-1],[0,-1],[0,-1],[5,-1],[9,10],[3,-1],[1,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-3],[0,-2],[0,-2],[-1,-1],[0,-2],[0,-1],[0,-2],[1,-2],[0,-2],[0,-1],[1,-1],[0,-1],[1,-1],[2,1],[0,-1],[0,-2],[1,-6],[0,-2],[0,-1],[0,-1],[0,-1],[1,0],[0,-1],[1,0],[4,8],[3,3],[1,5],[1,3],[0,1],[1,0],[1,-1],[6,-10],[1,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-4],[-1,-17],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[2,-7],[1,-1],[0,-1],[0,-1],[0,-2],[-1,-3],[-1,-11],[0,-3],[0,-3],[0,-2],[0,-2],[0,-5],[0,-1],[0,-1],[0,-1],[2,-9],[0,-1],[0,-1],[0,-1],[0,-7],[0,-1],[0,-1],[1,-3],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,-4],[1,-3],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-3],[0,-1],[0,-3],[0,-2],[-1,-7],[0,-2],[-1,-2],[0,-2],[0,-2],[0,-2],[0,-4],[-1,-3],[0,-2],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-3],[0,-3],[0,-1],[0,-1],[0,-2],[-8,-15],[0,-2],[-2,-7],[-1,-3],[-1,-2],[0,-1],[0,-2],[-1,-5],[0,-2],[-1,-5],[-1,-2],[0,-2],[0,-4],[-1,-8],[0,-2],[0,-2],[0,-2],[-1,-2],[0,-4],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-3],[0,-2],[-2,-12],[0,-2],[-2,-4],[0,-2],[0,-4],[-1,-1],[0,-1],[-1,1],[-1,2],[-1,0],[0,-1],[-1,-4],[0,-1],[1,-2],[0,-4],[1,-3],[0,-3],[2,1],[1,-2],[0,-4],[2,2],[1,-1],[0,-4],[1,-3],[1,-5],[0,-3],[0,-3],[0,-5],[-1,-2],[1,-2],[0,-3],[0,-4],[0,-2],[1,-2],[0,-2],[0,-1],[1,1],[0,2],[0,2],[1,1],[4,0],[1,-4],[0,-3],[0,-3],[-1,-5],[-2,-10],[-1,-1],[0,-1],[0,-3],[0,-3],[0,-2],[0,-2],[0,-2],[-1,-1],[0,-3],[-1,-12],[0,-2],[0,-1],[1,-1],[0,-3],[0,-1],[1,-4],[0,-1],[0,-1],[1,-3],[1,0],[0,-2],[0,-1],[0,-4],[-1,-2],[0,-3],[-1,-8],[-1,0],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[1,-4],[0,-3],[0,-1],[0,-2],[-1,-4],[-1,-6],[0,-1],[-3,-3],[-1,-1],[-2,-17],[-2,-13],[0,-3],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-2],[0,-1],[-1,-4],[0,-2],[0,-1],[0,-3],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-2],[-1,-1],[-1,-5],[0,-1],[-1,-1],[0,-1],[0,-2],[0,-2],[-1,-2],[0,-2],[0,-2],[-1,-2],[0,-2],[0,-1],[0,-1],[1,-4],[0,-1],[0,-1],[0,-2],[0,-3],[-1,-5],[0,-2],[-2,-6],[-1,-2],[-1,-2],[-8,-52],[0,-5]],[[7345,5160],[-1,3],[0,4],[0,1],[0,2],[0,3],[0,1],[0,1],[-1,1],[-2,7],[0,1],[0,1],[0,2],[0,1],[0,2],[0,5],[-1,1],[0,2],[0,4],[0,1],[0,3],[-1,2],[0,1],[0,1],[-1,0],[-3,-2],[0,1],[-2,5],[-1,2],[0,2],[0,1],[0,9],[-1,2],[0,4],[0,1],[-4,19],[0,1],[-1,0],[0,-1],[-4,-12],[-1,-2],[-1,0],[0,1],[0,3],[0,1],[-4,-8],[-3,-9],[0,1],[-7,17],[0,1],[-1,1],[0,2],[0,1],[0,1],[1,2],[0,1],[0,2],[0,1],[0,1],[-1,1],[0,1],[0,1],[-6,16],[-1,3],[0,2],[0,3],[0,12],[0,1],[0,1],[0,1],[-1,1],[-7,6],[-3,-3],[-1,0],[-1,2],[-2,5],[-1,3],[-1,0],[-1,0],[-1,-2],[-3,-10],[-3,-17],[-1,-1],[-1,0],[-4,12],[0,3],[-1,1],[0,2],[0,1],[0,1],[0,1],[-7,12],[0,1],[0,2],[-1,1],[0,7],[0,3],[0,2],[0,2],[-3,14],[0,1],[-1,4],[-1,15],[0,4],[0,2],[0,7],[0,2],[0,3],[0,1],[-1,1],[-1,1],[-8,-3],[-4,-3],[-1,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[-1,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[-2,-1],[0,-1],[-1,-1],[0,-6],[-1,-2],[0,-2],[-1,-2],[-2,-1],[0,-1],[0,-2],[0,-1],[0,-2],[-1,-1],[-2,-4],[-1,-7],[-2,-7],[0,-1],[-4,-3],[-1,-5],[0,-1],[0,-2],[0,-1],[0,-2],[0,-3],[0,-4],[0,-2],[0,-4],[0,-2],[0,-1],[0,-2],[-1,-3],[-1,-17],[0,-1],[0,-1],[-29,2],[-5,-5],[-3,-6],[-4,0],[-6,-15],[-3,-11],[-1,0],[-1,2],[0,2],[0,1],[0,1],[-1,9],[-1,6],[0,1],[0,1],[-16,38],[-1,8],[-3,12],[-1,2],[0,3],[-1,15],[0,4],[-1,1],[0,2],[0,1],[0,1],[0,1],[0,6],[-1,4],[0,1],[0,1],[0,2],[0,1],[-3,4],[-2,0],[-1,0],[-2,9],[-1,3],[-1,0],[-1,-1],[-8,-18],[-1,-3],[-1,0],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,2],[-1,8],[0,1],[-8,14],[0,1],[-1,2],[0,2],[-1,1],[0,2],[-2,1],[-1,-1],[-1,-3],[-1,-2],[-3,0],[-3,4],[-3,1],[-3,-3],[-1,-4],[0,-1],[-2,-4],[-1,0],[-2,1],[-1,-1],[-2,-7],[-1,0],[-11,3],[-1,1],[0,3],[-3,13],[-3,13],[0,1],[-19,5],[-1,-1],[-1,-1],[0,1],[-1,1],[-1,0],[-2,4],[0,5],[-1,6],[0,5],[-1,6],[0,4],[-2,8],[-1,7],[1,4],[-3,11],[-6,-10],[-1,-8],[0,-3],[-1,-2],[-4,-13],[-1,-2],[-13,5],[-1,-1],[0,-1],[-4,-21],[0,-1],[-1,-2],[-2,4],[-2,2],[-6,-8],[-1,90],[0,4],[0,5],[-2,9],[-2,11],[-1,1],[-5,4],[-2,-1],[-3,-1],[0,1],[-2,-1],[-1,1],[0,1],[0,3],[0,5],[-2,19],[0,2],[0,2],[-1,1],[-1,7],[-1,4],[0,3],[-2,3],[0,2],[-1,1],[0,2],[0,2],[0,3],[0,1],[0,2],[-1,7],[-1,7],[0,1],[1,3],[5,21],[0,3],[1,4],[0,5],[-1,5],[0,5],[-2,9],[0,1],[-1,1],[-5,4],[-1,2],[-4,20],[-1,2],[-1,0],[0,-1],[-2,-3],[-1,-2],[-1,-2],[-2,0],[-5,6],[-1,4],[-1,4],[-1,5],[-1,1],[-8,-3],[-1,1],[0,1],[0,1],[0,3],[-1,2],[-1,5],[-1,6],[-1,1],[0,1],[-2,-4],[-1,0],[-1,1],[0,1],[-1,2],[-1,0],[-3,-7],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[3,-17],[1,-5],[0,-1],[0,-1],[0,-2],[-1,-2],[-2,-9],[0,-1],[0,-1],[0,-1],[0,-5],[0,-2],[0,-2],[-1,-1],[0,-1],[-1,0],[0,1],[-4,9],[-6,10],[-1,0],[-5,-18],[-1,-4],[0,-1],[0,-1],[0,-1],[-1,-3],[0,-1],[0,-1],[-4,-13],[-3,-1],[0,-1],[0,-1],[0,-1],[-1,-4],[0,-3],[-1,-5],[-1,-3],[0,-1],[0,-1],[0,-1],[-1,0],[-2,10],[0,1],[-1,0],[-3,-2],[0,1],[0,1],[-1,2],[-1,1],[0,-1],[0,-1],[0,-2],[0,-1],[-1,0],[0,1],[0,3],[-1,1],[0,1],[0,1],[-1,0],[-2,-3],[-1,-1],[0,2],[0,1],[0,2],[0,1],[0,1],[0,2],[-1,1],[-2,0],[0,1],[0,1],[0,1],[0,2],[0,3],[0,1],[0,1],[-2,3],[-1,-1],[-5,7],[-2,6],[0,4],[0,4],[0,4],[0,2],[0,1],[0,3],[0,1],[0,1],[0,2],[0,2],[0,1],[0,1],[0,2],[1,4],[1,5],[0,1],[0,1],[0,1],[0,2],[1,1],[0,1],[0,2],[1,6],[0,2],[0,1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,2],[0,1],[-1,1],[0,3],[-1,2],[-5,10],[-1,0],[-5,-7],[0,-1],[0,-1],[0,-1],[0,-1],[-1,1],[-8,17],[-4,2],[-1,-1],[-1,-3],[-1,-2],[-1,0],[-2,-1],[0,1],[-1,0],[0,1],[-1,1],[-9,-2],[-1,-1],[-1,-5],[-1,-6],[-3,-17],[0,-1],[0,-1],[-5,5],[-7,-9],[-1,-1],[-3,4],[0,-1],[-1,-2],[-2,-14],[-1,-3],[-1,-1],[-11,1],[-1,-1],[0,-1],[0,-1],[-1,-1],[-1,-5],[-1,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[-1,0],[-2,0],[-5,-3],[-1,1],[0,3],[-1,0],[0,1],[-1,0],[0,1],[-1,0],[-1,8],[0,1],[0,1],[0,1],[2,8],[0,1],[0,1],[0,2],[0,6],[0,2],[-1,5],[0,3],[0,1],[0,1],[0,1],[-1,0],[-1,-1],[0,1],[0,1],[-1,2],[0,3],[-3,9],[-1,0],[-7,-5],[0,-1],[-1,-5],[0,-1],[-1,1],[-1,3],[0,1],[-1,-1],[-3,0],[-1,1],[-1,5],[-2,8],[-1,1],[0,2],[0,1],[1,2],[0,3],[1,3],[0,2],[0,1],[0,2],[0,4],[0,1],[0,1],[4,12],[0,1],[2,7],[0,3],[0,3],[-1,6],[0,3],[0,2],[1,1],[0,1],[0,1],[0,1],[0,3],[0,3],[0,3],[-1,2],[0,1],[-2,3],[0,1],[-1,1],[0,2],[0,1],[1,5],[0,2],[0,1],[0,1],[1,3],[0,2],[1,13],[0,2],[1,10],[0,3],[0,2],[0,3],[0,6],[0,2],[0,1],[0,1],[0,2],[0,4],[0,1],[0,1],[0,3],[0,1],[0,8],[0,1],[0,2],[0,2],[0,2],[0,3],[0,2],[0,3],[0,1],[0,4],[0,1],[0,2],[0,1],[-8,9],[-1,3],[-1,8],[-2,11],[0,3],[0,2],[0,1],[0,2],[1,8],[0,3],[0,1],[0,1],[-2,5],[0,2],[-1,2],[0,1],[0,2],[0,4],[0,5],[0,1],[-1,1],[0,3],[0,1],[-7,23]],[[6722,6030],[2,1],[1,2],[1,2],[10,23],[0,2],[5,6],[1,-1],[3,3],[0,2],[0,1],[0,1],[0,1],[0,2],[0,3],[-1,3],[0,1],[0,1],[0,1],[2,9],[0,2],[0,1],[0,1],[-1,4],[-1,1],[0,1],[0,1],[1,4],[0,2],[0,1],[0,1],[0,1],[0,2],[0,1],[1,2],[0,1],[2,5],[0,2],[0,2],[0,1],[1,3],[0,2],[0,3],[0,1],[0,2],[0,1],[0,1],[0,2],[0,1],[1,3],[0,2],[1,7],[1,4],[2,4],[0,-1],[1,-4],[2,-2],[0,-1],[0,1],[1,0],[0,2],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[-1,2],[0,3],[-1,5],[0,1],[0,2],[1,8],[0,2],[0,1],[0,2],[0,2],[1,2],[0,1],[1,4],[0,1],[2,8],[0,1],[0,1],[0,2],[0,1],[0,2],[0,1],[1,0],[1,-4],[1,0],[2,5],[1,2],[1,5],[0,1],[1,0],[3,6],[0,2],[0,1],[0,2],[1,1],[0,3],[0,1],[1,1],[0,1],[1,4],[0,3],[1,6],[1,1],[0,1],[2,-2],[4,4],[2,-1],[0,1],[1,1],[0,1],[0,2],[1,3],[1,5],[1,3],[0,1],[1,-1],[0,-1],[1,0],[1,2],[0,1],[0,1],[0,1],[0,1],[0,1],[0,2],[0,2],[0,1],[1,1],[0,2],[0,1],[0,2],[0,1],[0,1],[5,5],[2,3],[1,1],[2,6],[0,1],[1,1],[1,-2],[2,4],[1,4],[0,1],[0,2],[0,2],[0,7],[0,4],[0,6],[0,4],[0,1],[0,2],[-1,1],[0,1],[0,1],[0,1],[0,3],[0,1],[0,2],[1,1],[1,1],[1,-1],[0,1],[1,1],[0,2],[0,2],[0,1],[0,1],[0,1],[0,2],[0,2],[0,2],[0,1],[-1,3],[0,2],[0,1],[1,0],[1,1],[2,10],[0,1],[1,3],[1,1],[1,0],[2,10],[1,2],[0,1],[1,5],[0,1],[0,1],[1,1],[1,2],[0,1],[2,6],[0,1],[0,1],[-1,3],[0,2],[0,1],[0,3],[0,1],[0,1],[1,0],[0,1],[2,-2],[4,2],[1,2],[1,4],[2,13],[1,3],[0,2],[0,1],[0,1],[0,2],[-1,3],[0,1],[0,2],[-1,8],[0,1],[0,1],[0,1],[-1,0],[-1,-1],[-4,-4],[-1,1],[0,1],[0,1],[0,3],[0,1],[1,2],[0,2],[1,1],[0,1],[-1,1],[0,1],[0,2],[0,2],[0,3],[0,1],[1,1],[0,2],[2,0],[1,3],[0,1],[1,0],[1,-3],[1,-4],[1,0],[1,2],[1,4],[0,2],[0,3],[0,2],[0,1],[0,5],[0,1],[0,1],[0,1],[-1,7],[-1,0],[0,1],[-1,1],[0,1],[0,1],[-1,1],[0,1],[-1,1],[0,1],[0,1],[0,1],[1,8],[0,1],[0,2],[0,1],[0,3],[0,1],[0,2],[1,4],[0,2],[0,2],[0,3],[0,1],[0,1],[0,1],[-1,0],[-1,-1],[-1,1],[-1,5],[-1,1],[0,-1],[-2,-5],[-1,-3],[-2,-1],[-4,3],[-4,-3],[-1,3],[0,3],[-1,3],[0,1],[0,2],[-1,4],[0,3],[-1,2],[0,1],[0,2],[0,1],[1,15],[0,2],[0,1],[-1,10],[0,1],[0,2],[0,4],[0,4],[0,1],[0,2],[0,3],[-1,3],[0,1],[0,2],[1,1],[0,6],[0,1],[0,1],[1,1],[0,1],[0,2],[0,4],[1,2],[1,9],[0,3],[0,2],[0,2],[0,2],[1,2],[1,6],[0,3],[0,1]],[[6806,6904],[1,-1],[10,-22],[11,-9],[1,-2],[1,-5],[-1,-2],[-3,1],[-1,1],[1,3],[-1,0],[-1,-1],[-1,0],[0,2],[0,-1],[1,-8],[0,-2],[1,-1],[1,0],[3,-4],[2,-1],[3,-4],[1,0],[1,-2],[2,-5],[1,-3],[3,-10],[1,-2],[8,-14],[9,-5],[3,-5],[1,-4],[-2,-3],[0,-2],[0,-6],[0,-3],[2,-7],[1,-4],[0,-4],[1,-4],[1,-1],[2,0],[1,-1],[0,-3],[1,-4],[6,-16],[6,-25],[-3,3],[2,-9],[1,-2],[2,1],[1,-1],[4,-9],[1,-1],[1,-1],[0,-3],[1,-3],[0,-3],[2,-8],[0,-5],[-1,-10],[0,-6],[0,-2],[0,-2],[-1,-3],[2,2],[1,-4],[0,-2],[0,-3],[0,-2],[-1,-1],[1,-3],[1,2],[2,8],[2,2],[1,3],[0,5],[1,8],[1,1],[1,3],[1,0],[0,1],[-1,2],[-1,4],[0,6],[0,3],[1,5],[1,5],[1,5],[0,5],[2,9],[5,33],[0,5],[0,5],[1,4],[1,8],[0,5],[0,5],[0,1],[0,1],[0,1],[0,1],[0,6],[1,2],[1,2],[0,3],[1,5],[0,6],[-3,0],[-1,1],[-1,2],[1,2],[1,3],[2,1],[2,2],[-2,0],[1,4],[2,3],[1,2],[-6,-4],[1,4],[-1,1],[-2,-3],[-4,-9],[-3,0],[-5,13],[-1,3],[-1,5],[-1,11],[0,5],[-7,38],[-2,14],[-1,10],[0,4],[0,6],[1,4],[1,5],[0,6],[0,2],[0,3],[0,3],[0,3],[0,4],[0,3],[0,5],[0,3],[0,2],[0,2],[0,1],[0,1],[-1,0],[-1,0],[0,-3],[0,-4],[0,-2],[0,-2],[0,-4],[-2,0],[-4,5],[-4,12],[-1,8],[-2,2],[-3,6],[-4,1],[-6,16],[-2,2],[-1,-4],[0,-1],[0,-2],[1,-2],[-1,-3],[0,-1],[-1,-2],[-2,0],[-1,-1],[1,-6],[2,-19],[-1,-2],[-1,2],[-2,9],[0,3],[0,5],[-1,4],[0,6],[1,10],[0,6],[-1,4],[0,3],[0,6],[0,5],[1,8],[2,44],[0,4],[1,5],[0,3],[1,2],[0,1],[1,0],[0,-5],[1,-1],[-1,-4],[0,-2],[-1,-3],[1,-6],[1,-2],[1,3],[2,2],[2,3],[1,6],[2,10],[1,11],[-1,0],[0,-1],[-1,-1],[0,2],[0,2],[0,2],[0,2],[0,2],[0,4],[0,5],[0,4],[-1,1],[0,-2],[1,-6],[0,-4],[-1,0],[-2,7],[-1,6],[-1,3],[0,2],[0,6],[1,4],[1,7],[0,4],[1,19],[1,7],[1,1],[-1,5],[1,5],[0,4],[1,1],[-1,1],[-1,7],[-1,4],[1,6],[1,27],[0,5],[2,9],[0,5],[0,6],[-1,6],[-1,6],[0,4],[-4,4],[-1,0],[-1,-6],[-7,-6],[-2,-6],[-1,0],[-1,15],[0,4],[-1,3],[0,6],[0,4],[1,3],[3,9],[0,3],[2,10],[0,3],[0,5],[0,2],[0,1],[0,4],[0,1],[1,2],[0,1],[0,3],[-1,3],[-1,0],[0,-2],[0,-4],[-1,-2],[0,-2],[-1,-2],[1,-6],[0,-1],[-1,-4],[-1,-2],[-1,2],[-1,6],[0,6],[1,6],[4,20],[1,6],[0,5],[0,5],[2,14],[0,3],[3,0],[4,9],[4,4],[5,13],[3,3],[0,4],[0,3],[2,2],[2,4],[3,11],[1,1],[1,1],[1,2],[2,11],[6,28],[2,2],[1,-5],[0,3],[-1,2],[0,2],[0,3],[0,3],[4,22],[0,2],[1,3],[0,12],[0,4],[1,2],[0,1],[1,2],[0,4],[0,9],[0,5],[2,9],[0,6],[0,7],[0,2],[1,2],[0,2],[0,2],[0,3],[0,5],[2,14],[0,10],[0,5],[1,2],[0,4],[0,3],[1,5],[0,3],[0,3],[0,3],[0,3],[0,3],[0,4],[1,8],[0,2],[0,4],[0,2],[0,2],[1,1],[1,1],[-1,5],[-1,3],[0,4],[1,3],[0,2],[1,3],[0,8],[1,6],[0,4],[0,4],[1,1],[1,0],[0,1],[0,2],[-1,1],[-1,3],[0,2],[1,2],[5,18],[1,7],[2,9],[0,10],[0,12],[2,5],[1,4],[2,2],[8,-1],[1,-2],[0,-5],[-1,-2],[-5,-4],[-1,-1],[0,-3],[1,-4],[0,-2],[9,1],[8,4],[13,-2],[10,4],[7,-3],[2,4],[3,-1],[2,2],[1,-2],[0,2],[3,-7],[9,-10],[1,-3],[-1,-1],[-1,-2],[-1,0],[-1,0],[-1,0],[-1,-2],[1,0],[1,-1],[0,-2],[0,-1],[7,8],[5,-9],[5,-1],[5,-12],[5,-8],[-4,-1],[1,-6],[0,-1],[0,-1],[1,0],[1,-1],[0,-3],[-1,-3],[-1,-3],[0,-3],[0,-5],[0,-10],[1,-3],[2,-6],[0,-2],[-1,-3],[0,-5],[-1,-4],[0,-12],[1,-5],[0,-11],[1,-6],[0,-3],[1,-2],[0,-3],[0,-3],[-1,-2],[0,-3],[-1,-4],[-1,-5],[-1,-4],[0,-5],[-1,-7],[0,-6],[0,-2],[0,-3],[-1,-3],[-2,-9],[-1,-5],[0,-5],[0,-11],[0,-6],[-1,-4],[0,-4],[-1,-6],[-3,-5],[-1,-3],[-1,-11],[1,-11],[0,-10],[0,-7],[-2,-3],[-1,-2],[0,-2],[0,-3],[0,-5],[-1,-8],[-1,-4],[-4,-6],[-1,1],[0,1],[-1,-2],[-1,-4],[-2,-6],[-1,-5],[0,-8],[0,-6],[1,-6],[1,-4],[1,-4],[5,-27],[13,-31],[2,-8],[1,-5],[0,-5],[-1,-9],[0,-5],[0,-5],[0,-3],[1,-4],[1,-3],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[1,-1],[1,-6],[1,-3],[0,-5],[1,-6],[-1,-6],[0,-6],[0,-5],[-1,-5],[0,-4],[0,-4],[0,-5],[0,-6],[-1,-4],[-2,-11],[0,-4],[0,-1],[0,-2],[0,-1],[1,-2],[0,-2],[0,-1],[1,-5],[1,-2],[0,-4],[0,-5],[-1,-11],[1,-11],[-1,-4],[-3,-3],[-1,-3],[-4,-20],[-1,-5],[0,-4],[1,-7],[4,-11],[1,-8],[-1,-4],[-1,-8],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-3],[0,-3],[-1,-3],[0,-2],[0,-2],[0,-2],[1,-3],[0,-2],[0,-4],[0,-2],[0,-5],[2,-10],[1,-5],[1,-8],[1,-9],[-1,-9],[0,-4],[0,-4],[-1,-5],[0,-4],[-2,-6],[0,-4],[0,-4],[0,-2],[-1,-3],[-1,-2],[1,-4],[1,-6],[2,-23],[0,-4],[0,-10],[0,-3],[0,-3],[-1,-1],[0,-1],[-3,0],[0,-2],[1,-4],[1,-4],[1,-5],[1,-8],[0,-5],[0,-2],[0,-2],[0,-4],[-1,-3],[-1,-4],[-1,-3],[-1,-4],[0,-5],[0,-5],[-1,-7],[1,-5],[1,-14],[0,-5],[0,-12],[1,-4],[0,-3],[0,-6],[0,-2],[1,-3],[3,-16],[4,-16],[6,-13],[13,-41],[2,-11],[1,-4],[0,-4],[0,-4],[0,-7],[0,-3],[-1,-3],[-1,2],[0,3],[1,3],[0,2],[0,2],[-1,3],[-1,-3],[-2,-9],[-1,-5],[0,-12],[-1,-4],[-2,-8],[-4,-20],[-2,-5],[0,-3],[1,-7],[0,-5],[0,-5],[-1,-2],[-1,-4],[0,-4],[0,-3],[0,-2],[1,-3],[2,-20],[1,-4],[0,-4],[0,-11],[0,-12],[0,-4],[-1,-2],[-2,-2],[-1,-4],[0,-8],[1,-5],[1,-3],[0,-5],[0,-6],[-1,-4],[0,2],[0,3],[-1,3],[0,2],[-1,0],[0,-2],[-2,-4],[-1,-4],[-1,-1],[0,-3],[0,-3],[-1,-6],[-2,-3],[-5,0],[-1,-1],[0,-4],[-1,-3],[0,-3],[0,-3],[0,-3],[1,-4],[-1,-5],[-1,-3],[-2,-6],[-1,-4],[0,-2],[1,-2],[0,-4],[1,-5],[0,-3],[0,-3],[0,-2],[-1,-6],[0,-2],[-1,-5],[-1,-3],[-2,-1],[-1,0],[-1,4],[0,1],[-1,-1],[-1,-2],[0,-1],[-2,-2],[0,-1],[0,-3],[0,-6],[1,-2],[0,-1],[0,-1],[1,-2],[1,-1],[1,-3],[1,-9],[0,-5],[0,-2],[-1,-2],[-1,2],[0,3],[0,6],[-1,2],[0,-1],[0,-3],[0,-3],[0,-4],[-6,-7],[1,-2],[1,0],[1,-3],[-1,-1],[1,-8],[-1,-5],[0,-1],[-1,-2],[-1,-2],[0,-2],[0,-3],[-1,-2],[0,-4],[-2,-2],[0,-2],[-1,-3],[0,-8],[-1,-3],[-2,6],[-1,1],[-1,-1],[-1,-5],[-1,0],[-1,2],[-2,10],[-1,1],[-1,-2],[-1,-3],[3,-6],[1,-6],[-2,0],[1,-2],[3,-4],[-1,-3],[0,-3],[0,-3],[0,-4],[0,-4],[1,-2],[0,-2],[1,-3],[-1,-5],[1,-3],[1,-4],[0,-6],[0,-3],[0,-2],[-1,-2],[0,-4],[1,-1],[-2,-4],[-2,-1],[-2,1],[-1,4],[1,4],[0,1],[0,2],[-1,-2],[-1,-9],[0,-1],[-1,0],[-2,2],[-1,3],[1,-3],[1,-3],[0,-3],[-4,2],[-1,-2],[0,-2],[0,-3],[0,-1],[2,-4],[-1,-3],[-2,0],[-2,3],[1,-7],[0,-2],[0,-1],[0,-2],[-8,-5],[-8,12],[-2,8],[-1,11],[0,3],[1,1],[2,-1],[0,1],[1,5],[1,1],[3,2],[0,1],[1,1],[0,1],[0,3],[0,1],[0,1],[1,-1],[0,-1],[1,1],[1,8],[0,2],[0,1],[-3,1],[-1,-1],[-1,-5],[2,0],[-4,-5],[0,-2],[2,-1],[0,-2],[-6,-5],[-1,0],[-3,7],[-1,2],[0,2],[1,4],[-3,3],[0,1],[-2,10],[-1,5],[0,6],[-1,5],[-1,2],[-1,-3],[1,-2],[0,-3],[-1,-4],[-1,-2],[-2,-1],[-1,1],[-2,5],[-1,1],[-1,-2],[0,-3],[1,-2],[1,-3],[-1,-2],[-6,10],[-1,0],[-1,-3],[-1,0],[-4,3],[-5,-5],[0,-4],[1,-2],[0,-1],[2,-1],[1,-2],[0,-4],[0,-5],[0,-11],[0,-5],[0,-8],[0,-4],[1,-3],[2,-9],[5,-14],[3,-4],[3,0],[2,-3],[1,0],[3,-7],[1,-6],[1,-4],[10,-13],[3,-7],[7,1],[6,5],[9,2],[5,-2],[2,-1],[3,-5],[4,-3],[1,-3],[1,-4],[2,-10],[1,-5],[2,4],[1,-1],[1,-3],[0,-1],[1,1],[0,1],[0,2],[1,0],[0,-1],[0,-1],[1,0],[0,3],[5,11],[1,-1],[1,4],[1,9],[0,10],[0,7],[0,2],[0,2],[0,3],[0,2],[0,3],[0,1],[-1,2],[1,2],[0,2],[0,3],[1,2],[-1,2],[-1,-1],[-2,-3],[1,6],[3,6],[1,4],[0,3],[0,6],[0,2],[1,2],[9,5],[1,2],[1,2],[1,0],[1,3],[0,3],[0,3],[1,1],[0,3],[0,1],[0,2],[0,2],[0,2],[3,3],[2,7],[9,11],[1,4],[0,2],[1,6],[1,7],[0,6],[1,4],[3,3],[1,3],[2,8],[0,1],[0,1],[0,8],[0,4],[1,3],[1,3],[0,2],[-1,8],[1,17],[0,5],[1,5],[-1,6],[-1,10],[6,34],[8,31],[9,31],[1,5],[0,12],[1,11],[0,22],[1,9],[0,2],[0,2],[-1,15],[-1,13],[0,8],[-1,8],[0,8],[-1,7],[-3,20],[-6,22],[-1,8],[0,5],[2,10],[2,15],[0,7],[-1,7],[0,7],[-1,7],[1,3],[0,5],[0,5],[0,5],[1,7],[1,3],[3,10],[9,11],[7,14],[8,6],[4,0],[5,11],[6,3],[3,5],[1,1],[6,3],[6,-5],[0,-2],[1,-5],[0,-7],[1,-4],[1,-7],[0,-4],[-1,-3],[-1,-4],[0,-6],[0,-5],[1,-5],[5,-15],[2,-7],[2,-5],[1,-6],[1,-6],[1,-6],[0,-3],[1,3],[0,2],[-1,1],[1,3],[1,-2],[4,-16],[0,-2],[0,-3],[0,-3],[-1,-2],[0,-2],[-1,-6],[0,-4],[1,-3],[0,-7],[0,-3],[0,-2],[-1,-8],[0,-2],[0,-2],[-1,-1],[-1,-2],[0,-3],[0,-3],[0,-3],[0,-2],[1,-2],[1,-2],[2,-3],[1,-1],[0,-2],[0,-4],[-1,-5],[0,-3],[0,-1],[0,-1],[0,-4],[-1,-3],[0,-1],[-2,-7],[0,-3],[0,-3],[0,-2],[1,-1],[0,-2],[0,-7],[-1,-6],[0,-5],[0,-3],[0,-6],[1,-5],[1,-2],[0,-2],[1,-4],[-1,-5],[0,-6],[-1,-5],[-1,-3],[-1,-1],[0,2],[0,3],[-1,1],[0,-1],[-1,-1],[-1,-3],[0,-2],[0,-8],[2,-5],[8,-22],[3,-4],[4,-12],[1,-4],[0,-3],[-1,-3],[0,-3],[0,-2],[0,-2],[1,1],[1,3],[3,3],[1,-1],[0,-3],[2,2],[0,-3],[1,-1],[1,-1],[0,-1],[1,1],[0,1],[1,1],[1,1],[2,5],[0,1],[1,-1],[1,-1],[1,-2],[1,-3],[4,0],[0,1],[2,6],[1,1],[2,-1],[2,2],[1,-2],[3,-5],[2,-2],[0,1],[2,3],[-1,6],[0,2],[-1,2],[-3,0],[-1,1],[-1,2],[1,2],[1,3],[0,2],[-3,0],[-1,-1],[0,-2],[-1,-5],[-2,-4],[-1,0],[-2,2],[-1,5],[1,2],[2,-2],[1,2],[0,2],[0,3],[-1,3],[-1,0],[-1,-2],[-1,0],[0,3],[0,4],[-1,2],[0,-3],[0,-3],[-1,-3],[0,-4],[0,-3],[0,2],[-1,6],[-1,3],[-1,-1],[-1,-4],[-1,0],[0,1],[-1,1],[-1,-1],[-1,-2],[-3,9],[-1,1],[-1,-1],[-2,-2],[0,1],[0,4],[1,2],[2,4],[-1,-1],[-2,1],[-1,2],[0,3],[-1,4],[0,3],[-1,-1],[0,-9],[-4,-1],[-3,4],[-1,9],[0,6],[0,6],[0,5],[0,6],[0,4],[0,3],[1,2],[0,2],[1,1],[0,1],[1,1],[0,3],[-1,-1],[-1,2],[0,2],[0,3],[-1,5],[1,2],[0,1],[0,5],[0,5],[0,4],[0,5],[1,6],[0,5],[1,9],[3,10],[1,4],[1,2],[4,0],[3,4],[3,0],[1,3],[1,4],[-1,4],[-1,5],[0,6],[0,4],[-1,2],[-1,4],[-1,1],[-1,2],[0,2],[0,3],[-1,5],[0,7],[-1,3],[1,0],[0,1],[1,2],[0,3],[-1,6],[-1,2],[-1,0],[-1,-2],[-1,-1],[0,5],[0,6],[0,3],[-1,2],[-1,4],[0,6],[0,5],[1,5],[0,4],[0,4],[0,5],[0,6],[-1,5],[0,6],[0,4],[-1,4],[0,12],[0,5],[-1,6],[-1,3],[-5,1],[-2,3],[-2,10],[-1,3],[-9,5],[0,1],[-1,2],[-1,6],[-1,9],[0,1],[-2,9],[0,2],[-2,2],[-1,1],[0,1],[0,1],[-1,0],[0,-1],[0,-3],[-1,-1],[0,-1],[-4,4],[-5,11],[-2,2],[-2,8],[-1,2],[-7,-1],[-8,7],[-2,-3],[-4,-13],[-4,-6],[-1,-4],[-1,-6],[2,-2],[0,-1],[-1,-4],[-5,0],[-5,-6],[-1,1],[-6,12],[-5,3],[-2,-1],[-6,-12],[-2,-2],[-2,2],[-2,6],[-1,9],[-1,4],[0,3],[0,4],[1,7],[0,5],[0,1],[1,8],[1,8],[1,24],[0,2],[-1,15],[-3,20],[-3,17],[-2,12],[-1,8],[-1,8],[1,7],[3,13],[0,8],[0,3],[0,5],[1,4],[1,11],[0,3],[0,10],[0,3],[1,10],[0,5],[0,4],[0,3],[-1,4],[-1,3],[1,6],[0,5],[8,30],[1,4],[0,4],[0,4],[1,2],[0,3],[0,2],[3,13],[2,11],[2,10],[0,4],[0,5],[0,6],[0,3],[0,8],[0,8],[0,2],[-1,7],[-1,5],[-8,30],[-1,4],[-1,8],[0,6],[0,5],[0,5],[0,5],[-1,5],[-2,7],[-1,4],[-1,9],[-1,5],[0,4],[0,6],[-1,5],[-2,19],[0,6],[-1,2],[-1,1],[-1,6],[-1,2],[-2,8],[0,1],[-3,5],[-2,6],[-3,3],[-1,3],[0,5],[0,3],[9,35],[1,3],[0,6],[1,2],[1,7],[1,3],[0,5],[1,5],[0,4],[-1,5],[-1,5],[0,2],[0,4],[1,3],[0,3],[1,2],[6,13],[5,10],[9,11],[2,-1],[2,4],[7,16],[8,15],[1,4],[0,6],[3,18],[0,7],[1,26],[0,5],[0,9],[-1,17],[-1,13],[0,5],[0,5],[0,4],[0,3],[0,5],[-1,9],[0,6],[-7,33],[-1,1],[1,2],[0,2],[6,12],[2,1],[1,-2],[1,-2],[1,-6],[1,-2],[3,-4],[4,-11],[2,-1],[0,-2],[0,-5],[0,-1],[-2,-7],[0,1],[-1,2],[0,1],[-1,-5],[-1,-3],[0,-2],[1,0],[1,3],[1,-1],[0,-2],[0,-4],[1,-5],[1,-4],[4,-8],[1,-3],[0,-5],[0,-5],[-1,0],[-1,1],[-1,-1],[-1,-5],[0,-7],[1,-7],[1,-14],[0,-2],[0,-4],[0,-1],[1,-7],[1,-3],[-1,-2],[1,-4],[2,-3],[0,-4],[-1,-2],[0,-4],[0,-2],[0,-2],[-3,-5],[-1,-3],[0,-1],[-2,-14],[0,-4],[0,-2],[0,-4],[0,-2],[0,-2],[0,-4],[0,-1],[0,-4],[-2,-4],[-1,-3],[0,-2],[1,-6],[0,-2],[0,-2],[-2,-5],[-1,-1],[-1,0],[-1,-2],[0,-2],[0,-3],[0,-4],[0,-2],[1,-3],[-1,-3],[-1,-5],[0,-5],[0,-4],[0,-4],[0,-4],[0,-4],[1,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-2],[1,-2],[0,-1],[1,-1],[4,-13],[1,-4],[-1,-3],[-2,-3],[0,-2],[0,-4],[2,-7],[1,-4],[0,-2],[0,-2],[0,-1],[0,-2],[-1,-2],[-1,-2],[0,-3],[-1,-2],[1,-3],[0,-4],[-1,-3],[-4,3],[0,-1],[-1,-1],[0,-3],[0,-5],[0,-2],[0,-1],[0,-1],[1,-10],[2,-8],[11,-17],[8,-7],[2,-8],[1,0],[0,2],[0,2],[0,2],[1,0],[8,-4],[7,-1],[5,-4],[1,-2],[-1,-3],[0,-3],[0,-3],[0,-5],[0,-1],[1,-1],[0,-1],[0,-3],[0,-2],[0,-1],[0,-2],[0,-2],[0,1],[1,2],[0,4],[0,3],[0,3],[-1,9],[2,5],[11,-6],[5,4],[6,-8],[1,-4],[1,-4],[1,-5],[0,-7],[0,-7],[0,-5],[1,-5],[1,-4],[1,-3],[8,-5],[2,-9],[1,-3],[2,0],[1,2],[1,4],[2,2],[4,-3],[3,2],[1,1],[0,3],[1,2],[0,-1],[0,-3],[1,0],[2,2],[0,1],[0,2],[0,3],[0,1],[0,1],[-1,-1],[2,3],[1,3],[0,3],[0,4],[-1,1],[-1,-2],[-2,-7],[-4,6],[-1,-5],[1,0],[0,-1],[2,-3],[-4,-10],[-1,1],[-2,4],[-1,2],[-1,0],[-2,-3],[-1,-7],[0,-2],[-1,0],[0,2],[0,2],[0,3],[0,2],[1,10],[1,5],[-1,3],[0,1],[2,2],[1,2],[-1,3],[0,3],[-1,2],[0,3],[0,1],[1,0],[-1,5],[-2,0],[-2,-3],[0,-6],[3,-4],[1,-4],[-2,0],[-2,3],[-2,6],[-1,7],[-1,3],[1,4],[1,6],[0,4],[0,3],[1,4],[0,5],[-1,8],[-2,3],[-5,2],[-2,-7],[-1,1],[0,6],[0,2],[0,2],[-1,3],[0,1],[1,4],[2,4],[0,5],[0,2],[-1,0],[-1,-3],[-1,-2],[-1,-1],[0,-3],[-2,-3],[-1,-3],[0,-5],[-1,-3],[-1,1],[-2,7],[-3,4],[-2,1],[1,-3],[2,-4],[-1,-1],[-7,15],[-2,6],[-1,6],[-1,3],[-1,2],[-3,0],[-7,8],[0,2],[-1,2],[-1,8],[-1,4],[-2,3],[-6,5],[-1,3],[-1,6],[0,3],[1,5],[0,3],[0,3],[-3,15],[0,5],[-1,9],[-1,22],[-1,5],[-1,4],[-1,2],[1,4],[2,2],[7,17],[8,6],[7,6],[2,-1],[0,-5],[2,-1],[10,-28],[2,-9],[3,-10],[4,-1],[11,14],[1,3],[1,7],[1,4],[0,6],[0,5],[0,4],[-5,23],[-1,4],[-2,2],[-6,-1],[-4,-11],[-2,-2],[-1,2],[-2,6],[0,4],[1,7],[2,9],[0,3],[-1,2],[-1,3],[1,4],[1,0],[3,-8],[1,0],[1,0],[1,4],[1,2],[4,21],[1,6],[4,8],[1,1],[0,2],[0,3],[1,1],[1,-4],[1,-2],[4,6],[2,2],[1,-1],[1,-2]],[[7078,7844],[4,-7],[0,-1],[-3,4],[-2,2],[-1,1],[-1,1],[-1,-4],[0,-6],[0,-7],[-1,-12],[0,-4],[0,-6],[0,-3],[2,-5],[-1,-3],[0,-5],[0,-2],[-1,-1],[-2,4],[0,1],[0,4],[-3,1],[-4,7],[0,2],[0,1],[0,2],[-3,2],[-1,2],[-2,9],[0,3],[-1,4],[2,15],[1,4],[1,2],[7,5],[7,-7],[3,-3]],[[7116,7874],[9,-7],[4,2],[1,-2],[0,-3],[1,-2],[0,-3],[0,-3],[-1,0],[-2,-3],[-9,4],[-4,8],[-1,4],[0,3],[0,2],[2,0]],[[6981,7902],[0,-1],[-1,0],[-1,-3],[0,-1],[-1,-3],[-1,-3],[-1,1],[-1,1],[1,2],[0,3],[0,7],[0,1],[0,4],[2,6],[0,3],[0,5],[2,-1],[1,-3],[0,-4],[0,-4],[0,-1],[0,-1],[0,-3],[0,-1],[0,-4]],[[6970,7948],[8,-13],[1,-5],[-1,-2],[-2,3],[-1,0],[0,-2],[0,-3],[0,-5],[0,-4],[0,-3],[-1,-2],[0,-3],[0,-3],[0,-4],[0,-2],[0,-2],[-1,-4],[-2,-2],[0,-6],[1,0],[1,3],[1,0],[2,-1],[1,0],[1,0],[0,-1],[1,-4],[1,0],[-1,6],[1,5],[1,2],[2,3],[0,2],[0,4],[1,3],[0,-2],[4,-17],[1,-4],[0,-2],[1,-6],[0,-3],[0,-3],[0,-3],[1,-14],[1,-23],[0,-2],[-3,14],[-4,15],[-3,7],[-1,-1],[-1,-2],[0,-4],[-1,-2],[-8,-5],[-9,-5],[-5,-13],[-6,-3],[-2,2],[-1,6],[0,5],[-1,2],[0,1],[-1,-1],[-2,-7],[0,-2],[0,-2],[1,-2],[1,-2],[1,-3],[0,-2],[-3,4],[-1,0],[-1,-2],[-1,0],[0,1],[-1,2],[0,5],[1,7],[0,4],[0,3],[1,4],[2,13],[1,3],[0,5],[1,5],[0,5],[-1,5],[-1,8],[0,5],[0,3],[-1,8],[0,6],[0,2],[14,23],[1,-3],[0,-2],[1,0],[3,5],[8,4]],[[7122,7948],[3,-1],[6,-11],[0,-5],[0,-4],[0,1],[-2,7],[-3,5],[-8,8],[-4,-1],[-2,3],[3,9],[2,-2],[5,-9]],[[7107,7944],[0,-2],[0,-3],[1,-2],[-3,-5],[-9,-1],[1,1],[0,2],[0,2],[0,2],[1,2],[0,3],[-1,-1],[-1,-5],[-2,-3],[-1,-3],[0,-3],[0,-2],[0,-2],[0,-3],[0,-3],[-1,0],[0,1],[-1,2],[0,3],[0,1],[0,1],[1,6],[1,4],[1,7],[6,14],[11,6],[2,-5],[-1,-3],[-4,-5],[-1,-2],[0,-2],[0,-2]],[[6722,6030],[-4,-4],[0,-2],[0,-1],[-1,-3],[0,-2],[-1,-7],[0,-2],[-1,0],[0,-1],[-1,-3],[-1,-7],[0,-2],[0,-1],[-2,-4],[-1,-2],[-1,-5],[-1,-7],[0,-1],[0,-2],[-1,-1],[0,-2],[0,-1],[0,-1],[0,-1],[-1,-9],[0,-1],[-1,-5],[0,-2],[0,-1],[0,-3],[-3,-10],[-1,-5],[0,-1],[0,-1],[0,-1],[2,-3],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[-2,-6],[0,-3],[0,-1],[0,-2],[0,-1],[-1,-5],[0,-2],[-1,-1],[-1,-1],[-1,-2],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-2],[0,-2],[-2,-16],[-1,-3],[0,-1],[-4,1],[-1,-1],[0,-2],[0,-2],[0,-3],[0,-3],[0,-1],[-1,-2],[-1,-7],[0,-3],[-1,-10],[0,-2],[-1,-3],[-1,0],[0,2],[0,2],[0,1],[0,2],[0,1],[0,2],[0,1],[-4,14],[-1,4],[0,2],[0,3],[0,2],[0,1],[0,1],[0,2],[-1,2],[-3,2],[-2,0],[-2,-1],[0,-2],[-1,-3],[-2,-10],[-4,-24],[0,-1],[-4,-11],[-1,-4],[-1,-10],[-1,-5],[0,-1],[0,-1],[0,-2],[0,-6],[0,-2],[0,-2],[0,-1],[0,-2],[1,-4],[0,-3],[1,-4],[-1,-5],[0,-1],[-1,-2],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[-1,-10],[0,-1],[0,-2],[-1,-6],[-1,-2],[0,-4],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[3,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-3],[0,-10],[0,-3],[0,-1],[0,-3],[1,-1],[0,-4],[0,-1],[0,-2],[0,-3],[-1,-1],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-5],[0,-2],[0,-2],[1,-1],[4,-19],[0,-2],[1,0],[1,1],[1,0],[0,-2],[0,-3],[0,-9],[0,-2],[0,-1],[-1,-7],[-1,-2],[0,-2],[0,-4],[0,-5],[-1,-1],[-5,-14],[0,-1],[0,-2],[0,-2],[0,-5],[0,-4],[0,-6],[0,-2],[0,-2],[0,-2],[0,-2],[0,-2],[-1,-3],[0,-4],[0,-1],[0,-3],[-1,-3],[0,-28],[0,-9],[-2,-23],[0,-7],[-3,-21],[0,-1],[0,-1],[0,-2],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-2,-4],[0,-1],[0,-2],[0,-2],[0,-1],[1,-1],[0,-3],[0,-2],[1,-2],[-1,-3],[0,-5],[0,-5],[-1,-2],[0,-3],[0,-2],[0,-1],[0,-2],[0,-1],[0,-8],[1,-4],[-1,-4],[-1,-6],[0,-2],[0,-1],[-1,-2],[0,-1],[0,-3],[0,-4],[1,-2],[0,-5],[1,-11],[0,-2],[0,-1],[1,0],[0,-1],[2,-2],[1,-1],[0,-2],[1,-7],[1,-1],[0,-2],[0,-4],[-1,-2],[0,-5],[0,-1],[0,-3],[0,-1],[0,-2],[0,-2],[0,-4],[0,-3],[0,-2],[0,-1],[-2,-5],[0,-2],[0,-1],[0,-2],[0,-3],[0,-2],[1,-6],[0,-2],[0,-2],[0,-6],[0,-2],[0,-2],[1,-3],[0,-1],[1,-14],[0,-1],[0,-1],[1,-1],[0,-1],[3,-4],[0,-2],[0,-1],[0,-4],[-1,-7],[0,-9],[0,-1],[0,-2],[0,-3],[0,-3],[0,-1],[0,-2],[0,-2],[0,-2],[-1,-8],[0,-1],[-1,-1],[0,-1],[0,-1],[-1,-2],[0,-6],[0,-6],[0,-3],[0,-2],[-2,-23],[-1,-1],[0,-3],[0,-1],[0,-2],[0,-2],[1,-11],[1,-6],[1,-12]],[[6652,5114],[-1,-5],[0,-4],[0,-3],[-1,-2],[-1,-6],[-1,-13],[0,-3],[-1,-5],[0,-1],[1,-2],[0,-1],[0,-1],[0,-2],[0,-1],[1,-3],[0,-2],[0,-1],[0,-2],[0,-2],[0,-2],[-1,-15]],[[6648,5038],[-2,-3],[-2,-6],[-1,-2],[-1,-2],[-1,-3],[-2,-16],[0,-1],[-1,0],[0,1],[-1,1],[-1,0],[0,-2],[0,-1],[-1,-2],[-1,-5],[-1,-2],[-3,-1],[-42,2],[0,-1],[-2,-4],[-1,-1],[-11,9],[-1,-2],[0,-3],[0,-1],[-1,0],[0,-1],[-1,-3],[-1,-12],[0,-1],[-1,0],[-1,1],[-2,-3],[-2,-22],[0,-4],[0,-4],[0,-4],[0,-4],[0,-8],[-1,-3],[0,-3],[-1,-1],[-12,11],[0,-1],[0,-3],[-1,-7],[-1,-23],[-1,-3],[0,-3],[-1,-1],[-10,9],[-1,0],[0,-3],[-1,-2],[0,-7],[0,-2],[-1,-11],[0,-1],[0,-1],[-1,-1],[-3,2],[0,-2],[-1,-13],[-1,-18],[0,-4],[-32,28],[0,-4],[0,-3],[-1,-22],[0,-1],[0,-2],[-13,11],[0,1],[0,1],[0,2],[0,3],[1,23],[0,2],[0,1],[-1,2],[-13,14],[0,-2],[0,-2],[-1,-18],[-1,-11],[0,-1],[0,-1],[-10,7],[0,-2],[-1,-1],[0,-2],[-1,-25],[0,-3],[-2,0],[-11,10],[0,-1],[0,-1],[0,-2],[-1,-6],[0,-1],[-1,-16],[-1,-42],[0,-1],[0,-1],[10,-12],[0,-2],[0,-3],[0,-9],[-1,-2],[1,-2],[0,-2],[3,-2],[1,-2],[1,-4],[0,-9],[0,-3],[0,-8],[-1,-27]],[[6452,4684],[-6,17],[-1,1],[0,-1],[-1,-3],[-2,-9],[-1,-5],[-2,-15],[0,-4],[0,-8],[0,-2],[0,-1],[-1,-1],[-5,-26],[0,-2],[0,-1],[-1,-6],[-1,-5],[0,-1],[0,-1],[-2,-1],[-1,0],[-1,2],[0,4],[0,1],[-1,5],[0,2],[-1,4],[0,2],[0,1],[-4,8],[-1,1],[-2,-1],[0,-1],[-1,-10],[-3,-17],[-2,-7],[0,-2],[0,-2],[0,-4],[0,-2],[-1,-1],[0,-2],[-1,-1],[-1,-1],[-2,3],[-1,-1],[-4,-5],[-2,0],[-1,-1],[-3,-6],[0,-1],[0,-1],[0,-1],[-1,-9],[0,-1],[0,-2],[-1,0],[-5,5],[-1,0],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[3,-10],[0,-2],[0,-1],[-1,-1],[-4,-4],[-1,-1],[0,-2],[0,-1],[0,-3],[-2,-30],[0,-3],[0,-1],[0,-1],[1,-1],[0,-1],[0,-2],[0,-20],[0,-2],[0,-2],[0,-2],[0,-1],[0,-1],[0,-5],[0,-13],[0,-14],[-1,-2],[0,-2],[-1,0],[-1,-2],[-1,0],[-2,2],[-1,3],[-1,1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,2],[0,25],[0,3],[0,4],[-4,5],[-1,1],[0,2],[0,1],[0,2],[-1,2],[0,3],[0,1],[-1,1],[-1,2],[-1,2],[-1,5],[-1,0],[0,-1],[-1,-3],[0,-1],[0,-1],[-1,0],[0,1],[0,1],[0,1],[0,2],[1,36],[0,2],[-1,1],[-14,9],[-1,2],[0,1],[0,1],[0,9],[0,4],[-1,7],[0,3],[0,1],[1,8],[0,2],[-1,16],[1,5],[-1,6],[0,7],[0,23],[-1,8],[0,11],[0,3],[0,2],[0,1],[1,3],[1,5],[5,21],[0,3],[1,2],[1,9],[0,2],[-1,1],[-1,8],[-1,0],[0,-2],[0,-1],[0,1],[-3,3],[0,1],[0,2],[0,1],[0,1],[1,0],[1,1],[0,1],[-3,10],[0,1],[0,1],[0,2],[0,3],[0,2],[0,1],[-1,1],[0,1],[1,5],[0,2],[1,2],[0,3],[0,2],[0,3],[0,5],[-1,1],[0,1],[0,4],[0,2],[-1,11],[0,8],[-1,16],[0,11],[-1,4]],[[6452,4684],[4,-10],[26,-9],[0,1],[1,7],[1,2],[1,-2],[1,-7],[2,-4],[0,-3],[3,-32],[0,-2],[0,-1],[0,-4],[-1,-12],[0,-24],[0,-3],[-1,-17],[-1,-11],[-2,-3],[-2,-2],[-1,-1],[0,-2],[-1,-24],[0,-15],[-1,-4],[-1,-13],[0,-2],[-2,-11],[0,-4],[-1,-4],[1,-11],[0,-5],[3,-28],[0,-2],[1,-4],[1,-3],[0,-1],[1,0],[1,1],[6,-13],[1,0],[-1,6],[0,1],[0,1],[1,2],[1,1],[1,-6],[1,0],[0,-1],[1,-3],[0,-2],[0,-1],[0,-1],[-1,-2],[0,-2],[0,-1],[0,-2],[0,-4],[0,-1],[0,-1],[-2,-1],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-2],[0,-2],[0,-3],[0,-3],[0,-7],[0,-3],[0,-2],[0,-4],[0,-2],[0,-3],[1,-1],[0,-5],[1,-5],[0,-2],[0,-1],[0,-1],[2,-3],[0,-1],[1,-1],[0,-3],[0,-1],[0,-1],[0,-1],[-1,-4],[0,-1],[-2,-5],[-1,-3],[0,-1],[-1,-5],[0,-2],[0,-2],[0,-6],[0,-2],[0,-1],[-1,-8],[0,-1],[0,-3],[2,-22],[0,-2],[0,-4],[0,-10]],[[6493,4241],[-4,-1],[-4,3],[0,-1],[-1,-3],[0,-1],[-1,-1],[0,-1],[0,-4],[-1,-1],[0,-1],[-1,-2],[-1,0],[0,1],[0,2],[0,2],[0,1],[-2,3],[0,1],[-1,3],[0,1],[0,1],[-1,1],[0,1],[0,3],[0,4],[0,1],[-1,2],[0,1],[0,2],[0,3],[0,2],[0,2],[0,1],[-2,0],[0,-3],[-1,-1],[-3,2],[0,-3],[0,-1],[0,-1],[-1,-25],[-12,6],[-3,3],[-1,3],[0,1],[0,2],[-1,1],[-2,2],[0,1],[-1,0],[0,-2],[-1,-2],[-1,0],[-2,0],[-1,-2],[-4,-23],[-1,-2],[0,-1],[0,-2],[0,-3],[0,-9],[0,-3],[0,-2],[-1,-5],[0,-2],[-1,-3],[0,-3],[0,-1],[0,-3],[-1,-6],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[1,-3],[0,-1],[0,-1],[1,2],[1,-1],[0,-2],[1,-7],[0,-4],[0,-2],[-1,-5],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[1,-1],[0,-2],[2,-6],[0,-2],[0,-1],[0,-2],[0,-3],[-1,-5],[0,-3],[0,-1],[-1,-1],[0,-2],[0,-1],[0,-2],[1,-4],[-1,-4],[0,-2],[0,-1],[1,-1],[1,-4],[1,-1],[0,-1],[0,-3],[0,-7],[0,-3],[0,-2],[0,-1],[0,-2],[-1,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-3],[0,-4],[0,-1],[0,-3],[0,-8],[0,-2],[0,-2],[-1,-3],[0,-3],[-1,-4],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-3],[0,-2],[0,-1],[0,-1],[-1,0],[-1,-2],[-1,1],[0,1],[0,1],[0,-1],[-1,-2],[0,-6],[0,-2],[0,-3],[0,-2],[0,-6],[0,-1],[0,-1],[0,-2],[-1,0],[0,1],[-1,5],[0,1],[-4,2],[-1,-1],[-2,-4],[-1,1],[-1,0],[-1,-2],[0,-1],[0,-5],[-1,-13],[-1,-17],[0,-2],[0,-3],[0,-2],[0,-3],[0,-1],[1,-1],[0,-3],[1,-2],[0,-2],[0,-1],[0,-2],[0,-2],[0,-1],[0,-2],[0,-2],[0,-5],[0,-2],[0,-1],[1,-1],[1,-2],[0,-2],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-4],[0,-2],[0,-2],[1,-2],[0,-3],[1,-5],[1,-6],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[1,-5],[5,-11],[0,-4],[-2,-19],[-1,-2],[0,-1],[-1,-1],[0,-2],[0,-4],[0,-2],[0,-4],[0,-4],[0,-3],[0,-1],[0,-3],[-2,-1],[-1,0],[0,-1],[-1,-1],[-1,1],[-1,-1],[0,-2],[0,-3],[0,-21],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,0],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-4],[2,-8],[2,0],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-2],[0,-2],[1,-2],[0,-2],[0,-2],[0,-2],[0,-2],[0,-3],[0,-2],[0,-1],[0,-2],[1,-5],[1,-1],[0,-1],[1,-1],[0,-1],[0,-4],[-1,-2],[0,-1],[0,-3],[-1,-1],[0,-2],[1,-2],[1,2],[0,-1],[1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-2],[0,-10]],[[6428,3676],[-2,-3],[0,-2],[-1,-1],[0,-5],[0,-6],[-1,1],[0,1],[0,2],[0,1],[0,1],[-1,0],[-1,0],[0,1],[0,2],[0,1],[0,4],[0,1],[0,1],[-1,0],[-2,2],[0,-1],[0,-1],[-1,-1],[0,-1],[0,1],[0,1],[0,1],[-1,2],[-1,2],[0,6],[-1,5],[0,1],[-3,6],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[1,0],[0,1],[0,1],[0,1],[0,2],[0,1],[0,2],[0,2],[0,2],[-1,1],[0,2],[0,1],[0,1],[0,1],[0,1],[1,1],[-1,1],[0,2],[0,5],[-1,1],[-1,3],[0,-1],[-1,-1],[0,-1],[1,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-3],[-1,-2],[0,-1],[0,-1],[-1,0],[-3,5],[0,1],[-1,1],[0,1],[1,1],[0,4],[1,1],[0,2],[0,1],[-1,2],[-1,-1],[-1,4],[0,5],[0,4],[-1,4],[0,2],[0,1],[0,3],[0,2],[0,1],[-1,4],[0,4],[0,2],[0,2],[0,1],[0,1],[1,5],[0,3],[0,2],[0,2],[0,1],[-1,0],[0,-1],[0,-1],[0,-4],[-1,-1],[0,1],[0,1],[0,2],[0,2],[0,2],[0,2],[-1,1],[-1,8],[0,1],[-1,1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[-1,0],[0,1],[-1,2],[0,1],[0,1],[-1,0],[0,-1],[-1,0],[0,-1],[0,-7],[0,-1],[-1,-1],[-2,2]],[[6390,3794],[0,5],[0,12],[0,3],[0,3],[1,1],[0,1],[0,3],[0,4],[0,3],[0,2],[0,1],[0,2],[0,2],[0,1],[-1,0],[0,2],[0,2],[0,2],[0,2],[0,2],[0,1],[-1,4],[0,1],[-1,0],[-2,2],[-2,6],[-1,-1],[-1,0],[-1,1],[0,1],[0,1],[0,1],[0,2],[0,3],[0,2],[0,1],[0,4],[0,1],[1,1],[0,1],[-1,2],[0,2],[0,1],[0,1],[0,1],[0,1],[0,1],[-1,3],[0,2],[0,2],[0,1],[0,2],[0,1],[0,3],[0,2],[0,1],[-5,0],[-1,-5],[-2,-1],[0,-2],[0,-1],[0,-1],[0,-2],[-1,1],[0,1],[-3,1],[0,1],[0,2],[-1,3],[0,3],[0,2],[0,1],[0,8],[0,1],[0,2],[-2,2],[0,1],[0,2],[0,1],[-1,1],[0,3],[0,1],[0,1],[1,1],[0,2],[0,1],[0,1],[0,1],[0,1],[0,5],[1,3],[0,2],[0,3],[0,2],[0,1],[0,1],[0,3],[0,10],[-5,-10],[0,-2],[0,-1],[0,-2],[0,-1],[0,-6],[0,-2],[0,-1],[0,-1],[1,-1],[0,-1],[0,-1],[0,-2],[-1,-1],[0,-2],[0,-1],[0,-2],[0,-7],[0,-3],[-1,-3],[0,-5],[0,-3],[-1,-1],[0,-1],[0,-1],[-2,1],[-1,2],[0,1],[0,1],[1,3],[0,1],[0,2],[-1,1],[0,1],[0,2],[0,4],[0,1],[-2,2],[-2,1],[-1,-3],[0,-2],[-1,0],[-1,1],[0,1],[0,1],[0,1],[-2,-2],[-1,-8],[-1,-1],[-1,0],[-1,1],[0,2],[0,2],[0,1],[0,1],[-2,2],[0,-1],[0,-1],[-1,-1],[0,-2],[-1,-3],[-1,-1],[0,-2],[1,-2],[1,-1],[-1,-2],[0,-1],[0,-3],[-1,-1],[-1,-1],[0,-1],[1,-4],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[-1,-1],[0,-2],[-1,-1],[0,-1],[-1,1],[-1,-1],[0,1],[-1,2],[0,1],[-1,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,1],[-1,1],[0,4],[0,1],[0,-1],[0,-2],[-1,-1],[0,-2],[0,2],[-1,1],[-1,1],[0,1],[0,1],[0,1],[0,1],[0,4],[0,1],[0,1],[-1,1],[0,2],[0,2],[-1,1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[1,-5],[0,-1],[0,-1],[0,-1],[0,-1],[-1,1],[-1,0],[0,-1],[-2,-8],[0,-1],[-1,-4],[0,-2],[0,-1],[0,-1],[0,-1],[0,-3],[-1,-2],[-1,-1],[0,-2],[0,-1],[0,-3],[0,-4],[0,-3],[0,-1],[-1,1],[0,5],[-1,0],[0,1],[0,-2],[-1,-2],[0,-1],[0,-1],[-4,-5],[-1,0],[-1,-5],[-1,0],[0,4],[0,1],[0,1],[0,1],[1,1],[-1,5],[0,2],[0,4],[0,2],[0,2],[0,1],[-2,5],[0,-1],[-1,-1],[0,-1],[-2,6],[-1,0],[0,-1],[-3,-2],[-3,1],[0,1],[0,1],[0,1],[0,1],[1,0],[-1,3]],[[6390,3794],[0,-3],[1,-1],[2,-4],[1,0],[0,-2],[-1,-8],[0,-2],[0,-1],[-1,-2],[-2,-4],[0,-2],[0,-2],[-1,-9],[-1,-4],[0,-1],[-1,3],[0,1],[0,1],[0,2],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[-1,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-3],[-1,0],[-3,7],[0,1],[1,2],[1,2],[0,2],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[0,1],[-1,1],[0,1],[-3,-2],[-1,0],[-1,-3],[0,-3],[-1,-1],[0,-1],[0,-3],[0,-2],[0,-2],[0,-2],[0,-2],[0,-3],[0,-1],[0,-1],[-1,-1],[-1,0],[-1,1],[0,1],[0,2],[0,2],[0,5],[0,1],[0,1],[0,1],[-1,0],[0,-1],[0,-2],[0,-2],[0,-13],[0,-5],[0,-3],[0,-1],[-1,-1],[0,1],[-1,-2],[0,-1],[0,-2],[0,-1],[-2,1],[0,1],[0,1],[0,3],[-1,3],[0,1],[0,1],[-1,0],[0,-1],[0,-1],[0,-4],[-1,-1],[0,-1],[-1,0],[0,1],[0,1],[0,4],[0,2],[0,1],[-1,-1],[0,-1],[-1,-8],[-1,-5],[0,-5],[0,-1],[0,-3],[0,-1],[0,-1],[-2,-6],[0,-2],[0,-2],[0,-4],[1,0],[0,-1],[0,1],[1,0],[0,-1],[0,-2],[1,-2],[0,-1],[0,-1],[0,-1],[-1,0],[0,1],[-1,0],[0,-2],[0,-2],[0,-3],[-1,-5],[0,-1],[0,-1],[-2,-4],[0,-1],[0,-1],[0,-3],[-1,-2],[0,-1],[0,-6],[0,-2],[0,-1],[-1,-1],[-1,-2],[-1,0],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-5],[0,-6],[0,-7],[0,-3],[0,-1],[0,-2],[-1,-3],[0,-1],[-1,0],[-1,2],[-1,-1],[-2,-6],[-1,-1],[0,-1],[-1,0],[0,-1],[0,-1],[0,-3],[0,-2],[0,-2],[0,-3],[0,-5],[0,-1],[0,-1]],[[6344,3599],[-1,0],[-2,4],[-3,10],[-5,12],[0,4],[-1,5],[0,7],[-1,7],[0,11],[-1,10],[-1,5],[-1,3],[-8,2],[-1,1],[-1,4],[-1,1],[0,2],[0,2],[0,7],[0,1],[0,2],[-1,1],[0,1],[0,1],[0,1],[0,11],[0,3],[0,3],[0,2],[-1,1],[-2,0],[-1,0],[0,-1],[0,-1],[0,-1],[-1,-1],[-1,0],[-1,-3],[0,-1],[0,-1],[-3,-2],[0,-1],[-1,-7],[-6,-20],[-1,-2],[0,-4],[0,-2],[-1,-3],[0,-1],[0,-1],[-1,-1],[-1,0],[0,1],[0,1],[0,1],[0,1],[-1,1],[-1,-1],[-1,1],[-4,-5],[-2,-7],[-2,-8],[-1,-2]],[[6831,4284],[0,-11],[1,-53],[0,-9],[1,-5],[4,-53],[0,-1],[0,-1],[1,-8],[0,-2],[-1,-2],[-3,-5],[0,-3],[-1,-3],[-1,-3],[-4,-4],[0,-3],[0,-3],[0,-1],[0,-2],[0,-3],[-1,-1],[0,-1],[0,-1],[-1,0],[0,1],[0,2],[-1,1],[0,-1],[-1,-1],[-1,-3],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[-3,-14],[-1,-1],[-1,0],[0,1],[0,1],[-1,4],[0,7],[0,1],[0,1],[-14,-39],[-3,-9],[0,-1],[1,-1],[1,-2],[0,-1],[0,-3],[1,0],[1,3],[0,-1],[0,-1],[1,-3],[0,-1],[0,-2],[0,-3],[-1,-1],[-1,-3],[0,-2],[0,-4],[0,-2],[-1,-2],[0,-7],[0,-2],[-1,-7],[-1,-2],[1,-9],[1,-4],[0,-6],[0,-4],[-1,-4],[-1,-3],[0,-2],[0,-1],[0,-1],[0,-2],[1,-2],[1,2],[1,0],[0,-2],[0,-3],[0,-2],[0,-2],[1,-3],[0,-1],[1,-2],[0,-4],[0,-38],[0,-3],[0,-3],[0,-2],[0,-2],[1,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-5],[0,-1],[0,-2],[1,-3],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[1,-1],[0,1],[1,4],[0,1],[2,3],[0,-1],[0,-8],[0,-2],[0,-1],[0,-3],[0,-2],[-1,-10],[0,-8],[0,-2],[-1,-1],[0,-1],[-4,0]],[[6594,3722],[-1,8],[0,1],[-1,0],[-1,2],[1,1],[0,1],[0,3],[0,1],[0,2],[0,6],[0,2],[0,4],[0,1],[0,3],[0,2],[0,2],[0,1],[0,1],[0,1],[0,1],[0,1],[0,2],[1,9],[0,1],[0,2],[0,2],[0,1],[0,1],[0,1],[0,3],[0,1],[0,1],[0,1],[0,1],[1,5],[0,1],[0,1],[0,2],[1,2],[-1,1],[0,1],[-1,2],[0,1],[-1,2],[0,3],[0,2],[0,1],[0,1],[-1,1],[0,1],[0,1],[0,3],[0,3],[0,1],[0,2],[0,1],[-1,3],[0,7],[0,2],[-1,2],[0,7],[0,3],[1,2],[1,5],[0,3],[1,3],[0,1],[0,1],[1,-1],[0,-1],[1,-1],[0,1],[0,1],[0,2],[1,1],[0,-1],[0,-1],[0,-4],[0,-1],[0,-1],[0,-1],[3,-3],[1,4],[2,3],[0,1],[1,5],[1,-1],[1,0],[0,2],[0,1],[0,3],[0,1],[0,2],[1,-1],[1,0],[0,1],[0,2],[0,1],[0,1],[0,2],[0,1],[0,3],[1,11],[0,2],[0,-1],[0,-1],[1,0],[0,1],[0,1],[1,-1],[0,1],[0,2],[1,2],[0,1],[0,1],[0,-1],[1,0],[0,-1],[0,-2],[-1,-2],[0,-1],[0,-1],[0,-1],[1,0],[0,-1],[0,-1],[0,-1],[0,1],[1,1],[0,2],[0,-1],[0,-1],[1,1],[0,2],[0,1],[0,2],[-1,4],[0,1],[0,1],[-1,0],[0,1],[0,1],[0,2],[0,1],[1,0],[1,-1],[0,1],[0,1],[0,1],[0,1],[-1,1],[0,1],[-1,2],[0,1],[0,1],[1,3],[0,3],[0,1],[-1,1],[0,-1],[-1,1],[0,1],[0,2],[0,2],[0,1],[1,0],[0,1],[0,4],[1,1],[-1,4],[0,5],[0,2],[0,1],[-1,1],[-1,1],[0,1],[0,1],[1,2],[0,1],[0,1],[-1,1],[0,1],[0,1],[0,3],[0,5],[1,28],[0,2],[0,2],[0,1],[0,1],[0,4],[0,2],[0,2],[0,2],[1,6],[0,1],[0,2],[0,2],[1,4],[0,1],[0,-1],[0,-2],[0,-1],[0,-1],[1,-1],[0,-1],[0,-1],[0,-3],[1,-1],[0,1],[0,2],[0,1],[0,2],[0,2],[0,2],[1,6],[0,1],[0,2],[0,1],[0,1],[0,3],[0,1],[0,2],[0,1],[0,2],[0,1],[1,1],[1,0],[0,-1],[1,-3],[0,-1],[1,4],[1,0],[3,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-3],[0,-6],[0,-2],[0,-1],[0,-3],[1,-1],[1,-2],[1,2],[1,3],[2,14],[5,18],[0,6],[0,4],[0,12],[0,3],[0,2],[0,1],[0,1],[-2,-1],[0,1],[-1,1],[0,6],[0,1],[0,1],[0,1],[0,3],[0,1],[0,1],[-2,5],[0,2],[-1,4],[0,7],[-1,5],[0,2],[0,3],[0,1],[1,1],[0,2],[1,2],[0,3],[0,2],[0,2],[0,2],[0,4],[0,2],[0,1],[3,7],[0,3],[1,4],[0,5],[1,4],[1,1],[1,0],[1,-1],[1,0],[0,1],[0,1],[1,7],[0,1],[2,6],[1,4],[2,4],[0,2],[2,15],[0,1],[0,2],[0,5],[0,1],[0,2],[0,1],[0,1],[3,5],[0,2],[0,1],[2,7],[0,2],[0,1],[0,1],[0,1],[0,1],[-1,8],[0,4],[-1,9],[0,3],[1,9],[0,2],[0,5],[0,1],[0,2],[-1,6],[-2,-3],[-1,4],[0,1],[-2,-2],[-1,1],[0,2],[0,6],[0,2],[0,1],[0,1],[-2,-1],[0,1],[0,1],[0,2],[-1,10],[0,8],[0,10],[0,2],[0,1],[0,1],[1,0],[1,0],[0,1],[0,1],[1,8],[0,3],[-2,12],[0,5],[0,4],[1,6],[1,21],[0,1],[0,3],[0,1],[0,1],[-2,1],[-1,2],[0,3],[-1,5],[-1,3],[-4,7],[-2,3],[-3,6],[-1,5],[0,3],[0,5],[0,2],[-1,3],[-1,6],[-1,2],[0,2],[0,4],[0,2],[0,1],[0,1],[-5,3],[-1,1],[0,1],[0,2],[0,2],[1,2],[0,5],[1,8],[0,1],[1,1],[0,1],[1,32],[0,2],[0,2],[0,2],[0,1],[1,0],[2,1],[1,0],[0,2],[0,1],[0,1],[0,4],[0,4],[1,4],[1,17],[0,1],[0,2],[1,1],[2,-2],[1,1],[0,1],[0,1],[0,2],[0,3],[1,9],[0,2],[1,0],[3,-1],[0,1],[-1,11],[0,3],[0,1],[0,2],[0,1],[2,5],[0,2],[1,2],[0,1],[0,2],[0,3],[-1,6],[0,1],[0,2],[0,2],[1,2],[1,6],[0,2],[0,1],[0,2],[0,1],[0,3],[0,5],[1,3],[1,12],[0,2],[0,2],[0,1],[0,3],[0,4],[0,4],[0,2],[0,2],[0,3],[0,3],[0,1],[1,9],[0,3],[1,3],[0,2],[0,2],[0,2],[0,3],[0,1],[0,1],[2,9],[0,2],[2,12],[0,3],[0,2],[0,4],[0,2],[0,1],[0,5],[1,8],[0,9],[0,2],[0,2],[0,2],[0,2],[0,4],[0,2],[0,1],[1,4],[0,5],[0,3],[0,2],[0,3],[0,1],[0,2],[0,2],[0,13],[0,3],[0,4],[0,1],[0,2],[-2,1],[0,1],[0,2],[0,2],[0,1],[0,4],[0,10],[0,2],[0,2],[0,1],[-1,3],[0,1],[0,2],[0,1],[0,2],[0,6],[0,2],[0,3],[0,2],[0,2],[0,1],[-1,4],[-1,4],[0,3],[-1,4],[0,3],[0,2],[0,9],[0,3],[1,3],[0,1],[0,1],[0,2],[0,1],[1,1],[0,2],[0,5],[0,1],[0,2],[0,1],[0,1],[3,10],[0,1],[0,2],[0,1],[0,2],[0,1],[0,1],[-1,3],[0,2],[0,1],[0,2],[0,1],[1,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,2],[0,5],[-1,9],[0,4],[0,3],[-1,5],[0,2],[0,1],[0,2],[0,2],[0,1],[0,3]],[[6652,5114],[0,-2],[0,-2],[1,-3],[0,-1],[1,0],[1,-2],[0,-1],[1,-2],[1,-1],[1,1],[1,3],[0,1],[1,0],[0,-1],[6,-12],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-4],[0,-1],[0,-2],[0,-1],[0,-16],[0,-8],[0,-1],[1,0],[0,-1],[1,-4],[0,-1],[0,-1],[1,0],[10,6],[7,-7],[2,-4],[6,1],[3,-2],[2,-4],[21,-61],[0,-5],[0,-1],[1,-1],[13,-25],[6,-23],[6,-51],[0,-1],[0,-1],[0,-6],[-1,-4],[0,-4],[-1,-5],[0,-3],[0,-7],[0,-1],[0,-2],[0,-1],[2,-27],[1,-16],[0,-2],[0,-3],[0,-1],[0,-5],[0,-4],[6,-41],[1,-11],[1,-8],[1,-20],[1,-6],[-1,-8],[0,-1],[0,-2],[0,-2],[0,-1],[1,0],[1,-2],[3,-11],[1,-2],[5,-1],[0,-1],[0,-1],[1,-3],[0,-2],[0,-4],[0,-5],[0,-3],[0,-5],[0,-2],[0,-2],[2,-9],[0,-2],[1,-22],[1,-28],[1,-3],[0,-4],[-1,-17],[1,-48],[0,-2],[0,-2],[0,-1],[1,-4],[0,-2],[4,-12],[1,-2],[2,-1],[6,5],[19,-18],[0,-2],[1,-1],[0,-4],[0,-2],[1,-10],[0,-1],[0,-2],[1,-3],[0,-2],[0,-1],[2,-1],[0,-2],[1,-1],[0,-5],[0,-2],[0,-2],[4,-62],[1,-7],[7,-48],[0,-8],[6,-9],[2,-3]],[[6493,4241],[0,-8],[0,-1],[1,-3],[2,-14],[0,-4],[-1,-9],[-1,-8],[0,-5],[1,-5],[0,-2],[1,-2],[0,-1],[0,-1],[1,-4],[0,-2],[0,-3],[0,-2],[0,-3],[1,-4],[0,-2],[0,-1],[1,-11],[1,-5],[2,-10],[0,-1],[0,-4],[0,-12],[0,-4],[0,-5],[1,-14],[0,-7],[0,-3],[0,-5],[0,-4],[1,-7],[0,-3],[0,-2],[0,-1],[-1,-1],[-1,0],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-3],[1,-4],[0,-2],[0,-1],[0,-1],[1,-2],[0,-1],[0,-1],[-1,-1],[0,-1],[-1,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[-1,0],[0,-2],[0,-1],[0,-1],[0,-1],[1,0],[0,-1],[1,-1],[0,-3],[-1,-1],[0,-2],[0,-1],[1,-1],[1,-1],[0,1],[0,1],[0,5],[1,1],[0,1],[0,1],[1,0],[0,-1],[0,-1],[1,-3],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-3],[0,-2],[0,-1],[0,-1],[1,0],[0,-1],[1,1],[0,1],[1,0],[0,-1],[0,-1],[0,-2],[0,-3],[0,-2],[0,-1],[0,-2],[-1,-2],[-2,-5],[-1,-4],[0,-16],[1,1],[1,-1],[0,-1],[2,0],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-3],[0,-2],[0,-1],[0,-2],[0,-3],[0,-4],[0,-3],[0,-3],[0,-6],[0,-3],[0,-2],[0,-3],[0,-1],[0,-1],[-1,0],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[1,-1],[1,-1],[0,-1],[1,0],[0,-1],[0,-1],[0,-1],[0,-1],[0,-4],[0,-1],[0,-5],[-1,-1],[-4,-1],[-1,-1],[-1,-7],[0,-1],[-1,-2],[0,-4],[0,-3],[0,-10],[0,-6],[0,-6],[0,-5],[0,-7],[0,-3],[0,-4],[-1,-4],[-2,-4],[-1,1],[-1,9],[0,1],[-1,3],[0,2],[0,1],[-1,1],[-1,-3],[0,-4],[1,-7],[0,-5],[1,-5],[0,-2],[0,-2],[0,-2],[0,-2],[0,-2],[0,-2],[-1,-2],[0,-2],[0,-2],[0,-3],[0,-2],[1,0],[1,2],[1,-1],[0,1],[0,1],[1,10],[0,2],[1,1],[1,-2],[0,-1],[0,-1],[1,-1],[0,-3],[0,-4],[0,-3],[-1,-1],[0,-5],[-1,-1],[1,-1],[0,-1],[1,-1],[0,-5],[0,-2],[0,-1],[1,-3],[0,-1],[1,0],[0,-1],[1,-4],[0,-13],[1,-7],[1,1],[1,5],[0,-1],[0,-2],[1,-21]],[[6489,3618],[-3,0],[0,-1],[-1,-2],[-5,-9],[-1,3],[-2,12],[0,2],[0,2],[1,1],[1,-1],[1,1],[0,1],[0,6],[1,1],[0,1],[0,1],[1,2],[0,2],[1,6],[1,20],[1,4],[0,2],[1,1],[1,2],[0,2],[0,1],[0,5],[0,1],[0,1],[0,1],[0,2],[0,3],[0,4],[0,5],[0,1],[0,1],[-1,1],[-3,6],[0,-1],[-3,-1],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,-6],[0,-3],[0,-4],[1,-8],[0,-1],[0,-2],[0,-4],[0,-2],[0,-4],[0,-3],[0,-1],[0,-1],[-2,1],[0,1],[0,1],[-1,1],[0,2],[0,1],[0,1],[-1,0],[0,1],[-1,1],[0,2],[0,1],[0,5],[0,1],[-1,0],[-1,-2],[0,-1],[0,-1],[0,-2],[-1,0],[0,1],[0,2],[0,1],[0,5],[0,1],[0,2],[0,2],[0,1],[0,1],[0,1],[-1,-3],[-1,0],[0,2],[1,1],[0,1],[0,1],[0,1],[0,1],[0,1],[-1,0],[0,3],[0,2],[-1,1],[0,1],[0,1],[1,1],[0,2],[0,1],[0,2],[0,1],[0,1],[-1,4],[0,1],[0,1],[0,1],[1,3],[0,2],[0,1],[0,1],[0,1],[-1,2],[0,2],[0,1],[0,2],[0,1],[0,1],[0,1],[1,0],[0,1],[0,1],[0,3],[2,15],[0,3],[0,4],[1,0],[0,1],[0,2],[0,2],[0,1],[-1,-1],[-1,4],[-2,1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[-1,-1],[-1,0],[-1,4],[0,-1],[0,-1],[0,-1],[0,-2],[2,-4],[1,-4],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[-1,-7],[-2,-5],[0,-1],[-1,0],[0,2],[-1,0],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-4],[-1,-2],[-1,-11],[-1,-2],[0,-3],[0,-1],[0,-1],[0,1],[0,1],[-1,-1],[0,-1],[0,-1],[-1,-4],[0,-2],[0,-1],[1,-5],[1,0],[1,2],[0,-1],[0,-1],[1,1],[1,-1],[0,-1],[0,-2],[0,-1],[1,0],[1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,-2],[0,-2],[0,-1],[0,-2],[0,-2],[0,-1],[-1,-1],[1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[-1,0],[0,1],[-1,1],[0,2],[0,1],[-1,2],[0,1],[0,1],[0,1],[0,1],[0,-1],[-1,-3],[0,-2],[1,-3],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-3],[-1,-8],[-1,-1],[0,-1],[-1,1],[-1,4],[-1,4],[0,1],[0,1],[-2,3],[-1,-1],[0,-1],[0,-1],[1,-1],[0,-2],[0,-1],[0,-1],[-1,-2],[-2,0],[0,1],[-1,1],[0,1],[0,1],[0,4],[-1,1],[0,1],[-2,3],[-1,-2],[0,-1],[0,-2],[0,-2],[0,-1],[1,0],[0,1],[0,2],[0,1],[1,0],[0,-1],[0,-1],[0,-2],[0,-2],[-1,-2],[0,-1],[0,-1],[-1,-2],[1,-6],[0,-1],[0,-2],[0,-2],[0,-2],[1,-1],[0,-1],[0,-2],[0,-1],[-1,-1],[0,-1],[0,1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-3],[0,-2],[-1,-2],[0,-1],[-1,1],[-1,-1],[-1,1],[0,2],[0,1],[0,1],[1,2],[0,1],[0,1],[-1,2],[0,1],[0,3],[0,1],[-1,1],[0,-1],[-1,0],[0,1],[0,1],[0,1],[0,1],[-1,0],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[-1,-1],[0,-1],[-1,-2],[-1,-2],[0,-2],[-1,-1],[0,-2],[0,-1],[0,-2],[0,1],[0,1],[-1,3],[0,2],[0,1],[0,1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[-1,1],[0,1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-2],[0,1],[-1,2],[0,1],[-5,-4],[0,-2],[-1,-1],[0,4],[-1,5],[0,6],[0,2],[0,1],[1,1],[0,2],[0,2],[1,13],[1,3],[3,6],[1,6],[0,2],[0,1],[0,1],[0,1],[-1,0],[0,-2],[-1,0],[-1,1],[0,-1],[-1,-1],[-2,1]],[[6333,1047],[0,-1],[-1,0],[0,2],[0,5],[-1,6],[0,8],[-1,9],[0,3],[0,4],[0,3],[0,3],[0,3],[1,0],[0,1],[0,2],[1,1],[0,-3],[0,-3],[0,-5],[0,-5],[0,-5],[0,-4],[0,-3],[1,-4],[0,-13],[0,-3],[0,-1]],[[6294,1838],[-4,-63],[7,-15],[7,-14],[3,-5],[1,-2],[0,-3],[0,-3],[0,-4],[-1,-2],[0,-1],[0,-2],[0,-2],[1,-4],[0,-1],[0,-5],[0,-5],[0,-4],[0,-4],[0,-3],[2,-2],[0,-3],[0,-4],[-1,-7],[-2,-8],[-1,-2],[0,-2],[2,-15],[1,-9],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[-1,-4],[0,-4],[0,-2],[1,-4],[1,-6],[5,-21],[0,1],[1,4],[0,12],[0,15],[0,3],[0,2],[1,0],[6,-17],[11,1],[2,-5],[2,-9],[5,-54],[1,-11],[1,-5],[1,-2],[1,0],[0,-2],[1,-4],[0,-6],[4,-66],[9,-82],[-2,-12],[-1,-4],[-4,-5],[-1,2],[-1,9],[0,4],[-1,3],[-1,3],[-2,-2],[-1,-7],[-1,-12],[-1,-5],[1,-1],[0,-1],[1,-4],[1,-3],[0,-3],[0,-4],[0,-2],[0,-5],[1,-3],[2,0],[1,-2],[1,-4],[1,-5],[1,-8],[1,0],[1,1],[1,0],[0,-2],[1,-7],[2,-10],[6,-19],[1,-9],[1,-4],[0,-3],[0,-3],[-1,-1],[0,2],[-1,3],[0,1],[-1,1],[-1,4],[-1,4],[1,5],[0,2],[-1,-2],[0,-6],[-1,-3],[0,-2],[0,-3],[-1,-1],[0,-1],[0,-3],[1,0],[0,-3],[1,-5],[0,-15],[0,-2],[-1,3],[0,2],[-1,5],[0,2],[0,-1],[-1,-2],[0,-2],[0,-1],[-1,-1],[-1,-5],[0,-2],[0,-2],[0,-5],[0,-1],[0,-3],[0,-2],[0,-1],[-1,1],[0,5],[0,5],[0,5],[-1,1],[0,-3],[0,-5],[0,-3],[0,2],[-1,6],[0,2],[0,-2],[0,-4],[1,-6],[0,-6],[0,-1],[-1,4],[0,5],[-1,4],[0,1],[0,-7],[1,-3],[0,-6],[-1,-2],[0,2],[0,3],[-1,5],[-1,5],[0,3],[-1,-4],[0,2],[-1,-3],[0,-2],[0,-1],[0,-2],[-1,0],[1,-2],[0,-2],[1,-2],[1,-8],[0,-2],[0,-1],[0,-3],[-1,-3],[0,-1],[0,-1],[-1,-1],[0,-3],[0,-2],[0,-1],[0,-1],[1,3],[0,1],[1,0],[1,-1],[0,-3],[0,-3],[-1,-5],[0,-2],[1,-2],[0,-1],[0,-4],[0,-7],[1,-4],[-1,2],[0,1],[-1,-3],[0,-1],[0,-1],[0,-5],[-1,4],[0,5],[0,5],[0,3],[0,1],[-1,4],[0,1],[-1,1],[0,4],[0,1],[-1,7],[0,3],[-1,-1],[-1,-3],[0,-2],[0,-2],[-1,0],[0,1],[0,-1],[-1,-2],[0,-4],[-1,-9],[0,-2],[-1,2],[0,-2],[0,-3],[0,-1],[1,0],[0,-2],[0,-2],[-3,1],[0,-1],[0,-2],[0,-3],[0,-4],[-1,-4],[0,-1],[0,-2],[-1,-1],[0,-1],[0,-3],[0,-2],[0,-2],[0,-3],[0,-3],[0,3],[-1,4],[-1,1],[-1,-5],[-1,6],[0,1],[-1,-1],[0,-5],[-1,-1],[0,-1],[0,-2],[0,-3],[0,1],[-1,4],[0,1],[0,1],[0,1],[-1,-1],[0,-1],[-1,-2],[0,-5],[0,1],[0,3],[-1,3],[0,-1],[-1,-3],[0,-7],[0,-5],[0,-2],[1,-2],[-1,-3],[0,-2],[0,-2],[0,-1],[-1,11],[-1,3],[0,4],[0,2],[-1,1],[0,1],[0,2],[0,1],[-1,-1],[0,-1],[0,1],[-1,0],[0,2],[0,1],[0,4],[0,1],[0,3],[-1,2],[0,1],[-1,2],[-1,0],[0,-2],[0,-5],[1,-1],[0,1],[0,-1],[1,-5],[0,-1],[0,-1],[0,-1],[0,-2],[0,-3],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-4],[0,-2],[1,-2],[0,-7],[0,-3],[0,-5],[-1,0],[0,2],[0,10],[-1,4],[-1,10],[0,1],[-2,0],[0,1],[0,3],[-1,2],[0,2],[0,2],[1,2],[1,0],[-1,3],[-1,1],[-1,3],[0,1],[-1,-3],[1,-1],[0,-5],[0,-2],[0,-3],[0,-1],[1,-2],[0,-2],[0,1],[0,-1],[1,-1],[0,-3],[0,-1],[1,-1],[1,-2],[0,-3],[0,-2],[0,-3],[0,-3],[1,-6],[0,-1],[0,-2],[0,-2],[-1,-2],[0,-1],[0,-1],[-1,3],[0,6],[0,5],[0,-2],[0,-3],[0,-2]],[[6319,1075],[-1,2],[0,1],[0,1],[0,3],[0,1],[0,4],[0,4],[0,1],[0,1],[0,1],[-2,4],[0,1],[0,1],[0,3],[-1,1],[0,1],[0,-1],[-1,-1],[0,-2],[0,-4],[-1,-1],[0,-1],[-1,0],[-3,-7],[-6,-10],[-1,0],[0,15],[4,81],[0,2],[0,1],[0,1],[1,0],[0,2],[1,1],[0,1],[0,1],[0,1],[2,17],[0,1],[0,2],[0,1],[0,2],[0,1],[0,2],[-1,0],[0,1],[-2,0],[-6,-10],[-1,0],[0,2],[0,2],[1,17],[0,18],[0,1],[0,1],[-10,7],[0,2],[0,1],[1,13],[2,14],[0,3],[2,6],[1,2],[1,0],[5,-2],[1,1],[0,2],[0,3],[0,14],[-1,7],[0,4],[0,3],[1,4],[0,6],[2,6],[3,8],[1,9],[0,3],[1,4],[0,3],[0,3],[-1,4],[0,3],[-1,8],[0,5],[-1,3],[0,1],[-1,1],[-2,-1],[0,2],[-2,25],[0,3],[-1,1],[-2,3],[0,1],[-1,2],[0,6],[0,1],[0,1],[0,1],[-1,6],[0,1],[0,1],[-1,4],[-1,18],[-1,7],[-1,20],[-1,8],[-1,6],[0,12],[-1,3],[0,2],[0,2],[0,1],[0,2],[0,1],[0,1],[1,1],[0,1],[7,30],[1,5],[-2,8],[-1,2],[-1,2],[0,-1],[-1,-2],[-5,-34],[-2,-8],[-1,-3],[-1,0],[-1,1],[-5,16],[-4,29],[-1,7],[0,16],[-1,1],[-1,1],[-1,2],[0,5],[0,3],[-1,9],[0,2],[0,1],[0,3],[-1,2],[0,3],[-1,11],[0,3],[0,2],[0,6],[0,2],[0,1],[0,2],[1,12],[0,1],[0,1],[-1,5],[0,1],[-1,0],[-3,-9],[-4,23],[-1,2],[-2,0]],[[6344,3599],[0,-6],[-1,-3],[0,-1],[-1,-6],[-3,-12],[0,-2],[0,-2],[-1,-2],[-1,-1],[-1,-4],[0,-1],[0,-2],[0,-1],[0,1],[0,1],[-1,1],[0,3],[0,1],[0,-1],[0,-7],[-1,-3],[-1,-2],[0,-2],[0,-2],[0,-5],[1,-1],[0,2],[0,1],[0,-1],[1,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-3],[1,-1],[0,-2],[-1,-2],[0,-1],[-2,-4],[0,-2],[-1,-2],[-2,-12],[0,-3],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-6],[0,-2],[0,-2],[0,-3],[-1,-3],[0,-1],[0,-1],[-1,-2],[-3,-1],[0,-2],[0,-1],[0,-1],[0,-3],[1,-2],[0,-1],[0,-2],[0,-3],[0,-5],[-1,-3],[0,-3],[1,-1],[1,-1],[1,1],[0,1],[0,3],[0,2],[0,1],[0,1],[0,2],[0,2],[0,1],[0,1],[1,0],[0,-2],[0,-1],[1,-2],[0,-1],[0,-2],[0,-1],[1,0],[0,1],[0,1],[0,1],[0,1],[-1,4],[0,1],[0,1],[0,1],[1,5],[0,1],[0,1],[0,1],[1,-1],[0,-1],[0,-1],[1,-2],[1,-3],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-3],[1,-2],[0,-5],[0,-2],[1,-10],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-1],[0,1],[0,1],[-1,0],[-2,-7],[0,-1],[-1,-2],[0,-3],[1,-1],[0,-1],[0,-3],[0,1],[0,1],[0,1],[1,1],[0,-1],[0,-2],[0,-2],[1,-1],[0,1],[0,-1],[1,0],[0,-1],[-1,-3],[0,-3],[0,-3],[0,-3],[-1,-2],[0,-1],[-1,-1],[-2,-2],[-1,0],[-1,0],[0,-1],[0,-2],[0,-1],[0,-1],[0,-7],[0,-9],[0,-1],[1,-2],[0,-1],[0,-1],[0,-2],[0,-1],[1,-1],[-1,-1],[0,-1],[-1,0],[0,2],[0,1],[0,1],[0,2],[0,3],[0,1],[0,3],[-1,1],[0,1],[0,2],[0,2],[0,2],[0,1],[0,-2],[0,-3],[0,-2],[-1,-2],[0,-2],[0,-1],[1,-2],[0,-1],[0,-2],[0,-1],[-1,0],[-1,4],[0,1],[-1,0],[-1,-2],[0,-1],[-1,1],[-3,4],[0,1],[0,1],[-1,4],[0,1],[0,1],[0,-3],[-1,-4],[0,-1],[0,-2],[0,-3],[-1,-1],[0,-3],[0,-2],[0,-1],[0,-6],[0,-2],[0,-2],[0,-2],[0,-2],[0,-2],[-1,-4],[0,-3],[-2,-9]],[[6312,3316],[-1,-5],[-1,-1],[0,1],[-2,6],[-1,2],[-1,-2],[-1,-1],[-1,0],[0,-3],[-1,-3],[-2,-6],[-1,1],[0,2],[0,2],[0,3],[-1,1],[0,1],[-1,2],[-2,1],[0,1],[-1,1],[-3,13],[-3,7]],[[6319,1075],[0,-2],[1,-8],[0,-4],[-1,-7],[0,-2],[0,-2],[0,-6],[0,-2],[-1,-2],[-1,1],[-1,-1],[0,-3],[0,-1],[1,-1],[1,-1],[0,-2],[0,-2],[-1,-2],[0,-1],[0,-1],[0,-2],[-1,-9],[0,-5],[-1,-4],[0,-20],[0,-4],[-1,1],[0,1],[-1,-2],[0,-5],[0,-11],[-1,-9],[0,-5],[-1,-8],[0,-10],[0,-5],[-1,1],[0,-2],[-1,-4],[0,-5],[0,2],[-1,3],[0,2],[0,-1],[0,-2],[0,-19],[-1,-13],[0,-4],[0,-4],[0,5],[-1,10],[-1,3],[0,11],[0,5],[0,-2],[-1,-5],[1,-6],[0,-5],[0,-5],[0,-3],[-1,-8],[0,-5],[-1,3],[0,-1],[-1,-5]],[[6302,875],[0,1],[0,2],[0,2],[0,1],[0,1],[0,1],[-1,1],[-1,-4],[0,-1],[0,-1],[-1,0],[0,1],[0,2],[0,5],[0,1],[0,2],[0,1],[-5,18],[-5,13],[-7,7],[-2,0],[-1,1],[-2,4],[-2,-1],[-2,4],[-4,-6]],[[6667,2656],[-1,-1],[0,-5],[0,-4],[0,-4],[1,-16],[1,-3],[1,-1],[0,2],[1,4],[0,2],[1,-2],[3,-12],[5,-7],[0,-5],[0,-2],[0,-1],[-1,-3],[0,-1],[0,-5],[0,-1],[0,-3],[-2,-3],[0,-3],[0,-5],[0,-3],[3,-2],[1,-2],[0,-2],[0,-2],[0,-2],[0,-1],[12,-1],[0,-3],[0,-8],[0,-3],[0,-3],[1,-5],[0,-2],[1,-3],[0,-2],[0,-1],[0,-1],[1,-1],[0,-1],[1,1],[1,-1],[1,-2],[0,-1],[2,1],[1,-1],[1,-3],[1,0],[0,1],[1,0],[1,-2],[1,-4],[1,-2],[0,-1],[1,-5],[1,-19],[0,-7],[1,-4],[2,-4],[0,-1],[-1,-2],[-1,-4],[-1,-7],[-1,-11],[-1,-41],[-1,-7],[0,-22],[0,-11],[-1,-7],[-1,-5],[-8,-15],[-7,-14],[-3,-3],[-12,8],[-1,4],[-1,5],[-1,5],[-1,7],[0,9],[0,9],[-2,3],[-1,0],[-2,-3],[0,-2],[0,-1],[-1,-17],[0,-4],[0,-5],[0,-9],[0,-4],[-1,-3],[-1,-6],[0,-3],[-1,-15],[-1,-7],[-2,-2],[-1,1],[-1,-2],[-1,-4],[-1,-5],[-1,-3],[-2,8],[0,4],[0,4],[1,3],[1,-1],[1,2],[0,6],[0,2],[-2,3],[0,4],[-2,4],[-1,0],[-3,-4],[-1,1],[0,2],[0,4],[-1,3],[0,2],[-8,3],[-1,3],[-2,9],[0,3],[-1,3],[0,8],[-1,3],[-1,0],[0,-2],[-1,1],[-1,3],[0,3],[1,9],[-1,3],[-1,-1],[-1,1],[0,3],[0,4],[0,5],[0,3],[0,3],[1,3],[0,3],[0,5],[0,10],[-1,7],[-1,3],[-3,0],[-1,2],[0,3],[0,3],[0,3],[-1,3],[-1,5],[0,2],[-1,-3],[-1,-1],[0,-2],[-1,-5],[0,-5],[0,-4],[-1,-3],[-1,0],[0,2],[-1,3],[0,1],[0,1],[0,2],[0,1],[0,1],[-2,-2],[-1,-1],[-1,-3],[0,1],[0,4],[-1,1],[0,1],[-1,0],[-1,0],[-1,5],[-1,-1],[0,-6],[-1,-5],[0,-9],[0,-17],[0,-8],[0,-7],[-1,-3],[-3,5],[-1,0],[-1,-2],[0,-3],[0,-4],[-1,-3],[0,-2],[-1,-1],[-1,2],[-2,5],[0,3],[0,4],[-1,4],[0,9],[0,4],[-1,4],[0,2],[-1,1],[-1,2],[0,4],[0,10],[-1,4],[0,2],[0,-1],[-2,-6],[-2,-2],[-2,1],[-1,-1],[0,-2],[-1,-1],[-1,0],[0,1],[-2,5],[0,1],[-1,0],[0,-2],[-1,-2],[0,-3],[0,-5],[0,-8],[1,-4],[0,-3],[-1,0],[-2,2],[-1,2],[-2,20],[-1,1],[-1,-4],[0,-2],[0,-3],[0,-4],[0,-7],[0,-3],[-1,-4],[-1,-1],[0,-2],[-1,-9],[0,-4],[0,-3],[-1,-1],[-1,1],[-2,6],[-1,-1],[0,-3],[0,-6],[-1,-30],[-1,-5],[-4,-19],[-5,-15],[-2,-12],[-1,-2],[-1,5],[-1,7],[0,2],[-1,3],[-1,4],[0,4],[0,3],[-1,3],[-1,1],[-2,-2],[-1,1],[0,2],[-7,32],[0,7],[0,10],[-1,5],[-4,7],[-1,5],[-1,6],[0,4],[-2,5],[-1,5],[-1,2],[0,1],[-1,0],[-2,-4],[-1,-2],[0,-3],[1,-9],[-1,-2],[0,-2],[-1,-2],[1,-3],[2,-3],[1,-5],[0,-9],[0,-2],[0,-5],[0,-2],[0,-2],[0,-3],[0,-1],[0,-5],[0,-9],[0,-6],[1,-17],[0,-8],[-1,-7],[-1,-8],[-1,-3],[-2,-2],[0,1],[-1,2],[-1,3],[-1,8],[0,5],[0,5],[0,5],[1,11],[0,5],[0,11],[0,5],[2,11],[0,3],[0,2],[0,2],[0,2],[-1,2],[-1,1],[0,1],[-1,3],[0,1],[-2,3],[-1,2],[0,3],[-1,3],[0,3],[-1,4],[-1,3],[-1,6],[0,4],[0,2],[0,2],[0,4],[0,9],[-1,4],[0,2],[-2,2],[0,2],[-1,8],[0,3],[-1,5],[-1,1],[-2,-1],[-5,8],[0,5],[-1,4],[0,3],[-1,6],[0,4],[0,4],[0,4],[0,3],[1,3],[0,2],[0,4],[-1,2],[-1,5],[-3,6],[0,2],[-1,6],[0,2],[-1,3],[-1,2],[0,-2],[-1,0],[-2,2],[0,-1],[0,-1],[-1,1],[0,1],[0,1],[0,1],[0,1],[-1,0],[0,-1],[0,-1],[0,-5],[-3,1],[-1,-2],[-1,-2],[0,-3],[-1,-1],[0,-1],[-1,3],[-1,0],[1,5],[-1,1],[-1,-1],[-1,1],[0,4],[-1,0],[0,-2],[0,-2],[0,-1],[-1,0],[0,-1],[-1,-1],[0,-2],[0,-3],[-1,-1],[0,2],[-2,1],[-1,-2],[-1,2],[0,3],[-1,4],[0,3],[0,2],[-1,16],[-2,20],[0,7],[0,10],[-1,8],[-1,3],[0,-2],[0,-8],[-1,-2],[-1,-1],[-1,1],[-1,-2],[0,-4],[-1,-8],[-1,-3],[0,-1],[-1,0],[-1,0],[-1,2],[-1,2],[0,2],[-1,1],[-1,-2],[0,-3],[-1,-7],[-2,-10],[0,-7],[1,-8],[0,-9],[-1,-5],[-2,-5],[-1,-1],[-1,8],[0,5],[0,4],[0,2],[-1,2],[-1,-3],[0,-4],[-1,-3],[-1,-5],[-2,-2],[-3,1],[0,6],[-1,3],[0,4],[0,3],[0,2],[2,3],[1,1],[0,3],[1,4],[0,5],[-1,5],[0,2],[-1,0],[-1,2],[0,2],[-1,6],[0,2],[-1,-1],[-1,-2],[0,-1],[-7,3],[-1,1],[0,5],[0,4],[-1,3],[0,3],[-1,1],[-1,2]],[[6410,2598],[3,23],[3,8],[0,1],[1,6],[0,1],[0,1],[2,3],[0,2],[1,5],[6,27],[1,4],[1,4],[0,16],[1,12],[0,2],[-1,2],[0,1],[0,1],[0,1],[-1,3],[1,11],[0,5],[-1,4],[0,2],[2,4],[0,2],[0,1],[0,2],[0,1],[0,2],[0,2],[0,1],[0,1],[0,1],[2,3],[0,1],[1,2],[0,1],[0,3],[-1,2],[0,2],[0,1],[0,2],[-1,1],[0,2],[-1,1],[0,1],[0,1],[0,1],[0,3],[0,7],[0,3],[0,2],[0,1],[1,0],[0,-2],[0,-1],[1,0],[0,2],[0,1],[0,1],[0,3],[-1,4],[0,3],[0,2],[0,1],[1,2],[0,2],[0,2],[0,2],[0,2],[0,1],[1,2],[1,-2],[1,-6],[0,-1],[2,1],[0,1],[0,1],[0,1],[0,1],[1,3],[0,4],[0,3],[0,2],[0,9],[0,6],[0,3],[0,2],[-1,2],[0,1],[0,2],[0,2],[0,2],[0,3],[0,2],[0,2],[0,1],[1,1],[1,-1],[0,1],[1,2],[1,10],[0,1],[0,1],[1,2],[1,0],[0,1],[0,1],[1,5],[0,1],[1,2],[1,0],[0,3],[0,1],[1,2],[0,2],[-1,11],[0,2],[1,2],[0,1],[0,1],[1,1],[1,1],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[-1,1],[-1,1],[0,1],[-1,1],[0,2],[1,1],[0,1],[1,2],[0,1],[0,2],[0,1],[0,2],[0,2],[0,3],[0,2],[0,2],[0,1],[1,2],[0,2],[0,1],[0,2],[0,1],[0,3],[0,4],[0,2],[0,4],[1,3],[0,9],[0,3],[0,2],[0,1],[-1,2],[0,2],[0,2],[0,6],[0,3],[-1,1],[-1,3],[0,7],[0,1],[0,1],[0,2],[0,2],[0,2],[0,4],[1,2],[0,1],[1,-1],[0,1],[1,0],[0,2],[0,2],[0,1],[0,2],[0,5],[1,4],[0,3],[0,2],[0,1],[1,1],[0,2],[0,2],[0,1],[0,2],[0,1],[0,1],[0,1],[1,11],[0,1],[0,2],[1,9],[0,2],[0,2],[1,8],[0,13],[1,11],[0,4],[0,2],[0,1],[0,1],[-1,2],[0,2],[0,1],[0,2],[1,6],[1,3],[1,5],[0,2],[0,1],[0,2],[0,2],[0,1],[-1,3],[0,2],[-1,5],[0,1],[0,1],[0,1],[-1,0],[-2,-4],[-1,1],[0,1],[0,2],[0,2],[0,1],[0,2],[0,1],[1,2],[1,3],[1,3],[0,2],[0,1],[-1,5],[0,1],[0,3],[0,3],[0,1],[0,1],[-3,5],[0,2],[-1,2],[0,2],[1,1],[0,2],[1,2],[0,1],[1,1],[1,-2],[1,2],[1,0],[2,0],[0,-2],[1,0],[0,1],[0,1],[0,2],[0,2],[0,3],[1,5]],[[6459,3242],[1,-5],[1,-3],[0,-1],[0,1],[1,1],[1,1],[2,-1],[0,-1],[3,-6],[2,-3],[0,-1],[0,-1],[0,-1],[0,-1],[-1,0],[-1,-1],[0,-1],[1,-2],[0,-1],[1,-1],[2,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,3],[0,1],[0,1],[0,1],[0,1],[0,2],[1,1],[0,-1],[1,-5],[0,-1],[1,-6],[-1,-3],[0,-2],[1,-1],[0,-2],[0,-1],[-1,-1],[0,-1],[-1,1],[0,-2],[-1,-1],[0,-2],[0,-1],[-1,2],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[2,-9],[0,-4],[0,-1],[0,-1],[1,-1],[0,-1],[1,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[-1,-2],[1,-1],[1,0],[1,2],[1,4],[0,-1],[0,-1],[0,-1],[1,-1],[1,-1],[0,-2],[0,-2],[0,-2],[0,-1],[0,-2],[1,-2],[0,-1],[1,-5],[0,-7],[1,-1],[0,-1],[0,1],[0,1],[0,1],[0,1],[1,4],[0,1],[0,3],[2,7]],[[6451,2987],[1,1],[0,3],[0,2],[0,4],[0,1],[0,1],[-1,1],[0,-1],[0,-1],[-1,-3],[0,-3],[0,-2],[1,-1],[0,-1],[0,-1]],[[6410,2598],[-1,6],[0,3],[-1,1],[0,2],[0,2],[0,2],[0,3],[0,1],[0,3],[0,6],[1,2],[0,3],[0,4],[-1,2],[0,2],[0,1],[-1,1],[0,3],[0,1],[-1,1],[-1,1],[-1,0],[0,1],[-1,2],[0,2],[0,1],[0,1],[-4,1],[0,1],[0,2],[-1,1],[0,1],[0,2],[0,2],[0,3],[0,1],[0,2],[-1,1],[0,-1],[0,-2],[0,-1],[0,-4],[0,-1],[-1,0],[-1,3],[-1,0],[0,-2],[-1,-1],[0,1],[0,1],[0,4],[0,2],[0,1],[0,6],[0,2],[0,1],[0,1],[1,0],[0,1],[0,1],[0,1],[0,2],[0,2],[-2,5],[-2,2],[0,1],[0,2],[0,1],[-1,2],[0,1],[0,3],[-1,1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-5,3],[-2,20],[-1,4],[0,-1],[-1,-5],[0,-1],[0,-1],[0,1],[0,1],[-1,1],[1,3],[0,2],[-1,2],[0,4],[0,2],[0,1],[0,1],[0,2],[0,1],[0,2],[0,3],[-1,6],[0,2],[-1,1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-4],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[-1,0],[0,1],[0,1],[-1,1],[0,1],[0,1],[0,3],[0,1],[0,2],[0,1],[0,2],[0,2],[-1,2],[0,1],[-1,0],[0,-1],[0,-2],[-1,-1],[0,-1],[-1,1],[0,2],[0,1],[0,1],[0,1],[1,1],[0,6],[0,1],[0,2],[0,2],[0,2],[0,1],[-2,6],[-1,3],[-1,0],[0,-2],[0,-1],[-1,-1],[0,-1],[-2,2],[-1,-1],[0,-1],[-2,1],[-1,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-4],[0,-1],[0,-2],[-1,-3],[-1,-4],[0,18],[-1,11],[0,5],[-1,1],[-2,1],[0,1],[0,1],[0,1],[-1,12],[0,5],[0,2],[-2,-1],[-1,1],[-1,7],[-1,2],[-1,1],[-3,-6],[0,1],[-1,1],[1,6],[0,3],[0,3],[1,1],[0,1],[0,4],[0,1],[0,6],[1,2]],[[6345,2849],[3,12],[0,6],[1,3],[0,4],[0,9],[-1,8],[-1,7],[-3,-2],[-1,0],[-1,3],[0,1],[0,1],[0,1],[0,1],[1,13],[0,1],[0,2],[-1,1],[-2,-2],[0,-1],[0,-1],[0,-1],[-1,-12],[0,-1],[0,1],[-1,7],[-1,4],[0,4],[0,2],[0,6],[1,7],[0,5],[0,1],[0,1],[-1,2],[0,1],[-1,2],[0,2],[0,6],[0,2],[-1,2],[0,1],[1,6],[-1,1],[0,1],[0,1],[0,2],[0,2],[-1,2],[0,4],[0,4],[-2,7],[-1,3],[0,4],[0,2],[2,6],[0,1],[0,1],[0,3],[0,2],[0,1],[1,5],[0,4],[1,1],[2,0],[1,-1],[0,-9],[1,-1],[0,-1],[0,1],[0,2],[0,2],[0,1],[0,1],[0,1],[1,1],[1,-1],[0,-2],[2,3],[0,1],[0,14],[0,3],[0,3],[1,1],[0,1],[0,1],[1,3],[0,1],[0,1],[0,1],[-1,0],[-1,1],[0,1],[0,1],[-1,2],[0,1],[0,1],[0,2],[0,2],[0,4],[0,2],[0,1],[0,1],[0,1],[1,0],[0,-1],[1,-3],[0,-1],[0,-4],[1,0],[0,2],[0,2],[0,2],[0,1],[0,1],[3,-3],[1,1],[0,1],[0,2],[0,1],[-1,3],[-1,1],[0,1],[-1,6],[0,5],[0,1],[0,2],[0,2],[0,2],[0,1],[1,-1],[3,-6],[6,-7],[1,0],[1,15],[2,15],[0,1],[1,4],[8,-4],[1,0],[0,2],[0,1],[0,2],[1,2],[1,0],[4,-4],[1,-1],[0,-2],[0,-3],[0,-1],[0,-2],[1,-1],[0,1],[0,1],[0,1],[1,1],[1,2],[1,1],[0,2],[1,1],[0,1],[0,1],[-1,2],[0,1],[1,0],[0,1],[1,2],[1,0],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[1,-1],[3,-3],[1,1],[1,2],[0,2],[0,2],[0,1],[0,1],[0,2],[0,1],[0,1],[-1,2],[0,1],[0,1],[-1,2],[-1,1],[0,1],[0,1],[0,2],[1,10],[0,2],[2,3],[0,1],[0,1],[0,2],[1,3],[2,6],[1,1],[0,4],[0,5],[0,11],[0,5],[0,4],[1,17],[0,14],[0,3],[0,3],[0,4],[0,3],[0,1],[-1,10],[0,2],[0,2],[0,4],[0,2],[0,1],[-2,1],[0,1],[0,2],[0,1],[0,5],[0,3],[0,1],[-1,3],[-1,6]],[[6390,3276],[1,-2],[3,0],[0,-1],[1,-2],[0,-3],[0,-8],[1,-1],[0,-1],[1,1],[0,1],[0,1],[1,1],[0,1],[0,1],[0,1],[0,1],[0,1],[1,1],[0,1],[0,-2],[0,-2],[1,-3],[0,-3],[0,-2],[0,-2],[0,-2],[1,-2],[0,-2],[0,-1],[1,-3],[0,-2],[0,-3],[0,-2],[0,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-1],[1,0],[0,1],[0,3],[1,2],[0,1],[1,1],[0,1],[0,6],[1,2],[0,3],[1,3],[0,1],[1,0],[1,-2],[2,-10],[0,-1],[1,1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-3],[1,-1],[1,2],[0,1],[0,2],[0,1],[0,2],[0,2],[0,2],[1,5],[0,2],[0,3],[0,2],[0,3],[0,2],[0,5],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,9],[0,1],[0,1],[1,1],[1,-1],[1,1],[0,7],[1,4],[0,4],[1,6],[0,2],[1,1],[1,-2],[0,-1],[1,0],[0,1],[3,8],[0,-2],[0,-4],[0,-12],[0,-2],[3,1],[0,1],[0,3],[0,5],[0,1],[1,0],[0,-3],[4,-14],[0,-3],[2,-1],[1,-4],[1,-1],[2,1],[1,0],[0,-1],[0,-1],[0,-2],[0,-3],[0,-9],[0,-9],[0,-1],[1,-1],[0,1],[1,1],[0,1],[0,-1],[0,-1],[1,-2],[0,-3],[0,-1],[0,-1],[0,-4],[0,-1],[0,-3],[1,-1],[0,-1],[0,-2],[0,-2],[0,-1],[1,0],[0,2],[0,1],[0,2],[0,1],[0,1],[0,2],[0,1],[0,2],[0,1],[0,2],[2,2],[0,2],[0,1],[0,1],[0,2],[0,1],[1,1],[2,2],[1,2],[1,0],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[1,1],[0,1],[0,2],[0,4],[0,2],[0,1],[0,1],[0,1],[1,1],[0,1],[0,-1],[1,0],[0,-1],[0,-1],[0,-1],[1,-2],[0,-1],[0,-1],[0,-2],[0,-5],[0,-2],[0,-3],[1,-5]],[[6410,2598],[-1,-4],[0,-8],[1,-21],[0,-7],[-1,-4],[-1,-3],[-1,0],[0,2],[0,5],[1,4],[-1,1],[-1,0],[-1,5],[-1,0],[0,-6],[0,-6],[0,-2],[-1,-1],[0,-2],[0,-4],[0,-14],[0,-6],[0,-3],[-1,0],[0,-1],[-1,-4],[0,-3],[-1,-1],[-1,0],[-1,-1],[0,-1],[0,-2],[1,-1],[0,-1],[-1,-2],[0,-2],[0,-2],[0,-5],[0,-4],[0,-3],[0,-3],[-1,-2],[-3,-9],[-5,-6],[-1,-3],[-4,-20],[-1,-10],[-1,-1],[-7,-1],[-2,4],[0,-1],[-1,-2],[0,-5],[0,-4],[-1,-8],[0,-4],[-1,-6],[0,-7],[1,-6],[1,-7],[0,-8],[0,-8],[-1,-5],[-3,-6],[-4,-5],[-1,-6],[-2,-18],[-5,-22],[-1,-1],[-2,1],[-3,13],[-1,0],[-1,-3],[1,-5],[0,-2],[2,-2],[0,-5],[1,-8],[-1,-17],[1,-35],[1,-17],[1,-14],[1,-9],[-1,-9],[0,-6],[2,-4],[1,-2],[0,-4],[0,-10],[0,-3],[0,-2],[0,-4],[-3,-15],[-2,-9],[-6,-13],[0,-2],[-1,-5],[0,-1],[-1,0],[-1,6],[-2,7],[0,1],[-1,-1],[-1,0],[0,4],[0,5],[-1,6],[-1,5],[0,2],[0,3],[-1,5],[0,2],[0,7],[0,7],[-1,7],[0,6],[-2,7],[0,2],[0,4],[0,3],[0,3],[-2,6],[-1,10],[-1,5],[0,5],[-2,8],[0,1],[-1,2],[0,1],[0,3],[0,3],[-1,3],[-1,4],[-2,13],[0,3],[-2,0],[-1,-4],[-1,-6],[-1,-7],[0,-2]],[[6300,2809],[3,5],[1,-2],[0,-1],[0,-1],[1,-5],[0,-2],[0,-1],[1,-2],[0,-2],[1,0],[0,1],[1,3],[0,2],[1,2],[0,1],[0,-1],[1,-2],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,0],[6,0],[2,4],[1,6],[1,2],[1,0],[1,-2],[1,-2],[0,-1],[0,-3],[1,-1],[0,2],[1,6],[0,2],[1,1],[0,1],[0,2],[0,1],[0,2],[0,1],[-1,2],[0,2],[0,1],[1,2],[0,1],[0,3],[1,3],[0,2],[0,1],[1,1],[1,0],[0,-1],[0,-1],[0,-1],[0,-2],[0,-8],[1,-1],[1,2],[1,0],[1,1],[0,2],[0,1],[1,4],[0,1],[0,1],[0,3],[-1,1],[-1,0],[-1,1],[0,1],[0,1],[1,2],[1,3],[1,2],[0,2],[0,1],[0,2],[1,0],[0,1],[2,-2],[0,1],[1,3],[1,1],[2,-1],[1,-2],[2,-5]],[[6390,3276],[0,1],[-1,0],[-1,-2],[0,-1],[-1,2],[0,3],[0,1],[-2,-1],[-1,2],[-1,5],[-1,2],[0,1],[-1,1],[-1,0],[-2,-3],[-2,2],[0,2],[0,3],[1,1],[0,1],[-1,2],[-1,5],[0,1],[0,2],[1,4],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[-2,4],[0,2],[0,1],[0,1],[1,1],[0,1],[0,1],[-1,1],[0,2],[0,1],[0,8],[0,1],[0,1],[-1,1],[-1,3],[0,1],[0,1],[0,1],[0,4],[0,1],[-1,1],[0,1],[-1,1],[0,1],[-1,9],[-1,1],[0,-1],[0,-2],[0,-4],[-1,-1],[0,-1],[0,-1],[-1,2],[0,-1],[-1,-1],[0,-3],[-1,-2],[0,-2],[0,-3],[0,-1],[0,-1],[-2,0],[-1,-1],[-1,-2],[0,-3],[-2,-14],[-1,-13],[-1,-1],[0,-3],[-1,1],[-5,2],[-3,-2],[-1,-3],[0,-1],[-3,15],[0,1],[-1,0],[0,-1],[0,-2],[-1,-2],[-1,4],[1,7],[0,3],[0,1],[-3,-2],[-2,0],[0,1],[0,1],[0,1],[0,3],[-1,1],[0,1],[0,1],[0,-1],[-1,-2],[0,-2],[0,-1],[0,-1],[-1,-1],[0,1],[0,1],[-1,0],[0,-2],[-1,-4],[0,-2],[0,-3],[-1,-8],[0,-2],[0,-1],[-1,-2],[-1,-1],[0,1],[0,1],[0,1],[0,2],[-1,1],[0,1],[1,1],[0,1],[0,1],[0,2],[0,2],[1,1],[0,2],[-1,1],[0,1],[0,1],[0,1],[0,1],[0,1],[-2,0],[0,-1],[0,-2],[0,-2],[0,-2],[0,-3],[-1,-4],[0,-3],[0,-3],[-1,-2],[-1,-5],[0,-2],[0,-1],[0,-4],[0,-2],[0,-3],[0,-3],[-2,-6],[-1,-1],[0,1],[-1,3],[0,1],[-1,0],[0,-1],[0,-1],[-1,-1],[0,-2],[-1,-9],[0,2],[-1,1],[0,1],[0,3],[0,16],[0,1],[-1,4],[-1,9],[0,2],[-1,5]],[[6334,3385],[0,1],[1,2],[0,1],[1,6],[0,1],[0,1],[0,1],[-1,1],[0,-1],[0,-1],[-1,-12]],[[7345,5160],[3,-44],[0,-1],[0,-1],[0,-1],[-1,-10],[0,-2],[0,-1],[0,-1],[0,-3],[1,-2],[0,-2],[0,-1],[2,-5],[0,-1],[1,-2],[0,-2],[0,-4],[0,-4],[0,-1],[1,-1],[0,-2],[0,-1],[4,-4],[2,2],[1,0],[1,-3],[3,-7],[1,-4],[1,-8],[0,-1],[3,-5],[1,0],[0,1],[1,3],[1,1],[0,-1],[1,-3],[0,-7],[1,-5],[0,-1],[1,-1],[6,-13],[1,-1],[2,3],[1,2],[0,1],[3,-12],[0,-1],[0,-1],[0,-3],[0,-5],[0,-8],[0,-2],[-5,-6],[-1,-1],[-3,-12],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[1,-3],[1,-9],[0,-2],[-1,-3],[-28,-71],[-1,0],[-10,-35]],[[7087,4292],[-2,7],[-1,7],[-1,3],[-1,0],[-2,-5],[-1,6],[-1,4],[-1,1],[-2,2],[-13,-1],[-3,-2],[-1,0],[-1,1],[0,2],[0,1],[0,1],[-2,7],[0,3],[-1,2],[-2,3],[-1,1],[-2,-2],[-1,-1],[-1,1],[-1,4],[0,2],[0,1],[0,1],[-1,1],[-1,0],[-3,-8],[-2,0],[-11,11],[0,1],[0,2],[1,5],[1,12],[0,2],[-1,1],[-3,11],[0,1],[0,3],[0,2],[0,1],[0,2],[0,1],[0,2],[0,2],[0,10],[0,1],[0,2],[0,4],[-1,4],[0,2],[-1,1],[-1,0],[-5,-12],[-1,0],[-3,12],[-2,14],[0,3],[0,2],[0,2],[0,12],[0,1],[-1,2],[-1,6],[-1,3],[-1,11],[-3,12],[-2,11],[0,3],[-1,2],[-1,3],[-1,12],[0,1],[-1,1],[-1,0],[-1,0],[-1,1],[-1,7],[-2,12],[-7,17],[-2,9],[0,1],[-3,7],[-1,3],[-1,9],[-2,17],[0,1],[-1,0],[-1,-1],[0,-1],[0,-2],[-3,-17],[-1,-1],[-5,9],[0,2],[-11,-12],[-3,3],[-1,3],[-2,7],[0,1],[0,4],[0,1],[-1,1],[0,1],[-15,1],[-1,1],[0,2],[0,4],[-1,9],[0,2],[0,2],[-1,1],[-6,0],[0,-1],[-4,-10],[-1,-5],[-1,-4],[0,-1],[0,-2],[0,-1],[6,-18],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[-12,-40],[-6,-12],[-1,1],[-1,0],[0,-3],[-1,-3],[-4,-58],[0,-4],[-1,-4],[-1,-4],[-3,-6],[-9,-8],[-4,1],[-1,-3],[-1,-9],[-3,-18],[-8,-18],[-6,-23],[0,1],[-2,7],[0,1],[0,1],[0,1],[0,1],[-1,0],[-3,-9],[-1,-1],[0,1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-41],[0,-5],[0,-5],[-1,-3],[-10,-20],[-1,2],[-2,5],[-4,-2],[-5,2]],[[6965,3627],[-1,0],[-1,1],[0,2],[0,1],[1,0],[0,1],[1,-1],[0,-1],[0,-1],[0,-2]],[[6965,3640],[0,-1],[-1,2],[0,1],[0,1],[1,3],[0,2],[0,3],[1,1],[0,-3],[0,-5],[0,-3],[0,-1],[-1,0]],[[7113,3948],[0,-8],[0,-1],[2,-42],[1,-7],[0,-8],[0,-3],[0,-3],[0,-1],[0,-2],[1,-4],[0,-2],[0,-2],[0,-5],[0,-3],[0,-2],[1,-7],[0,-21],[0,-5],[0,-1],[0,-2],[0,-3],[-1,-5],[-1,-5],[-1,-2],[0,-1],[0,-6],[0,-1],[-1,0],[-1,-2],[-1,0],[0,-1],[0,-1],[-1,-1],[0,-2],[0,-2],[0,-1],[0,-2],[0,-6],[0,-1],[0,-1],[-1,-1],[0,-1],[-2,-7],[-1,-3],[0,-2],[-1,-1],[0,-7],[0,-1],[0,-2],[1,-1],[1,1],[0,6],[0,1],[1,0],[0,-1],[1,-3],[1,0],[1,4],[0,1],[0,-1],[1,-3],[1,-3],[0,-1],[0,-2],[0,-2],[0,-1],[-1,-2],[0,-1],[0,-2],[0,-1],[0,-5],[0,-13],[0,-2],[1,-1],[0,-1],[1,-3],[1,-1],[0,-2],[0,-5],[1,-2],[1,-3],[1,-3],[0,-2],[0,-1],[0,-8],[0,-2],[-1,0],[-1,-1],[0,-1],[-2,-5],[0,1],[0,1],[0,3],[-1,0],[0,-1],[-1,0],[-2,2],[-1,0],[0,-1],[0,-2],[0,-1],[0,-5],[-1,-2],[0,-1],[-5,3],[-1,-2],[0,-4],[0,-2],[-1,-2],[0,-7],[0,-1],[-4,-4],[0,-1],[0,-2],[-1,-2],[0,-7],[0,-1],[-6,-26],[0,-1],[0,-4],[0,-1],[0,-2],[0,-3],[0,-2],[0,-2],[0,-1],[2,-3],[0,-1],[0,-1],[0,-2],[0,-3],[0,-4],[-1,-1],[0,-1],[-3,-6],[-1,1],[-2,6],[-1,-1],[0,-1],[0,-2],[0,-1],[0,-2],[1,-4],[0,-1],[0,-2],[0,-2],[0,-1],[3,-8],[0,-1],[0,-1],[0,-1],[0,-1],[-3,-11],[0,-1],[0,-1],[2,-7],[0,-1],[2,1],[1,0],[0,-2],[1,-5],[0,-2],[0,-3],[-1,-3],[0,-2],[-3,-9],[0,-2],[-1,-1],[0,-2],[0,-3],[0,-2],[0,-1],[-2,-7],[0,-1],[0,-1],[2,-7],[0,-1],[0,-2],[0,-1],[0,-3],[0,-1],[1,-3],[0,-1],[0,-4],[0,-1],[1,-1],[0,-1],[0,-1],[1,-1],[0,-1],[0,-2],[0,-2],[0,-8],[0,-4],[0,-1],[0,-2],[0,-3],[0,-6],[0,-2],[0,-1],[0,-2],[1,-3],[0,-1],[0,-1],[0,-2],[0,-3],[0,-2],[0,-1],[0,-1],[1,0],[1,0],[0,-2],[0,-2],[1,-5],[0,-3],[-1,-6],[-1,-15],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[1,-1],[2,1],[1,1],[1,-2],[0,-1],[1,-1],[-1,-1],[0,-1],[-1,-5],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[1,-1],[0,-3],[1,-2],[0,-1],[1,0],[1,0],[1,-1],[0,-1],[0,-5],[0,-3],[1,-3],[0,-3],[0,-9],[0,-3],[0,-1],[0,-1],[1,-1],[1,3],[0,2],[0,-1],[1,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[0,-6],[1,-6],[0,-1],[0,-2],[0,-3],[-1,-6],[0,-1],[0,-1],[-3,-3],[0,-2],[0,-1],[0,-12],[0,-91]],[[7102,3177],[-2,-4],[-1,-1],[-2,2],[-1,0],[-1,-3],[-2,-7],[2,-18],[-5,-17],[-3,-13],[-4,-14],[0,-1],[-2,7],[-1,1],[-3,0],[-1,-2],[0,-2],[0,-3],[0,-7],[-1,-4],[0,-1],[-1,-1],[-1,0],[0,-2],[0,-3],[-1,-12],[0,-1],[-5,1],[0,-3],[0,-5],[0,-13],[0,-1],[0,-3],[1,-2],[0,-2],[-2,-22],[0,-4],[-1,1],[-2,6],[-1,6],[0,4],[0,5],[0,1],[0,3],[-1,3],[0,3],[0,1],[-2,0],[0,-1],[-1,-6],[0,-1],[-1,0],[0,2],[0,4],[0,5],[-1,4],[0,1],[-2,1],[-1,1],[-1,0],[-1,-1],[-2,-9],[-1,2],[0,1],[-3,1],[0,-1],[0,-2],[0,-2],[0,-3],[0,-3],[0,-1],[0,-1],[-1,1],[-1,-1],[0,-2],[0,-2],[0,-5],[0,-2],[-1,-2],[0,-4],[-2,-11],[-1,-3],[-1,-1],[-1,7],[1,15],[-5,13],[1,8],[-1,6],[0,3],[0,5],[1,3],[0,2],[2,1],[0,22],[1,3],[2,5],[-1,5],[0,5],[1,4],[1,2],[1,0],[4,-4],[0,1],[1,3],[0,7],[2,38],[-2,-2],[0,1],[0,5],[0,1],[-1,0],[-1,-1],[-1,-2],[1,-2],[0,-2],[0,-3],[-1,-4],[-1,-2],[0,-5],[0,-6],[-1,-3],[0,-1],[-1,-1],[-1,4],[0,2],[-1,1],[0,-1],[0,-3],[0,-3],[-1,-1],[0,-1],[-5,6],[-3,4],[-1,3],[0,7],[0,3],[-2,4],[0,9],[0,5],[0,3],[-1,1],[-1,-1],[-4,2],[-3,2],[-2,-1],[-1,-7],[0,-3],[-1,-5],[0,-4],[0,-2],[3,3],[1,-1],[1,-2],[0,-6],[1,-7],[0,-8],[-1,-2],[-3,6],[-1,-1],[0,-7],[-1,-7],[-1,-2],[-1,-1],[-1,2],[0,2],[0,4],[0,5],[1,9],[0,6],[0,4],[-1,3],[-1,4],[0,2],[0,3],[0,3],[1,1],[0,-2],[1,0],[0,1],[0,3],[0,1],[1,0],[0,-1],[0,1],[1,3],[0,2],[0,1],[-3,6],[-2,5],[1,11],[0,3],[-1,4],[-1,3],[0,2],[-1,8],[-1,7],[-1,6],[0,3],[-3,2],[-1,-2],[0,-4],[0,-2],[1,-3],[1,-3],[0,-2],[0,-3],[-1,-7],[0,-2],[0,-3],[0,-3],[0,-3],[0,-1],[1,-3],[0,-1],[1,-1],[1,-2],[1,-4],[0,-7],[-1,-3],[0,-2],[-1,-1],[-1,2],[-1,4],[-1,16],[-1,3],[0,1],[-1,1],[-5,3],[-1,-1],[0,-4],[-1,-4],[0,-5],[0,-11],[0,-3],[-1,-4],[0,-2],[-1,-1],[0,1],[-1,1],[0,3],[-1,1],[0,-1],[-2,-3],[0,-1],[0,12],[-1,3],[0,3],[0,1],[-1,-1],[-1,2],[-1,2],[-1,1],[0,-3],[-2,-14],[0,-4],[-1,-4],[-1,0],[0,3],[-1,9],[-1,4],[0,2],[-1,3],[0,3],[0,6],[0,6],[0,4],[0,1],[-2,0],[0,9],[0,5],[1,2],[1,0],[0,-1],[1,-3],[1,-1],[2,1],[0,11],[0,3],[0,4],[0,2],[0,1],[-1,1],[0,1],[0,2],[0,8],[0,3],[0,3],[0,1],[0,2],[0,1],[-1,18],[1,7],[1,4],[1,2],[1,2],[0,3],[0,6],[0,5],[-1,8],[-1,2],[0,1],[-2,0],[-1,1],[-1,4],[-1,6],[-1,9],[0,6],[0,4],[-1,6],[0,3],[0,5],[0,4],[1,3],[0,3],[-1,6],[1,5],[0,3],[0,3],[0,7],[0,1],[0,1],[0,3],[0,3],[-1,7],[-2,18],[-2,13],[-1,4],[1,4],[0,4],[1,3],[-2,3],[-1,2],[-2,-1],[-1,-4],[0,-3],[-1,-2],[-1,-1],[-2,4]],[[6957,3465],[0,3],[1,9],[0,3],[0,1],[-1,4],[0,2],[0,2],[0,1],[0,4],[0,2],[0,2],[0,1],[0,3],[2,13],[0,1],[4,3],[0,1],[0,2],[0,1],[1,11],[1,6],[-1,4],[0,3],[-1,3],[-1,0],[-1,1],[0,1],[-2,8],[0,1],[0,1],[0,4],[0,4],[0,2],[0,2],[1,6],[0,4],[0,1],[0,2],[0,3],[0,1],[0,1],[1,3],[0,1],[1,-1],[0,-1],[0,-1],[1,0],[1,3],[1,3],[0,2],[0,4],[-1,0],[-1,0],[-2,-4],[-1,0],[-2,2],[-1,2],[0,5],[0,4],[1,1],[6,1],[1,1],[1,4],[0,1],[1,10],[1,2],[0,3],[0,2],[0,7],[0,5],[1,17],[0,3],[0,2],[0,1],[0,1],[0,1],[-1,1],[0,1],[-1,1],[0,1],[0,2],[-1,2],[0,4],[0,2],[0,2],[1,2],[0,2],[0,1],[0,2],[0,1],[0,2],[0,1],[0,1],[-1,0],[0,1],[-1,5],[0,2],[0,2],[2,7],[0,1],[0,1],[2,2],[1,1],[0,2],[0,2],[0,2],[0,4],[0,4],[0,2],[-1,1],[0,1],[0,1],[0,1],[-1,3],[0,2],[0,2],[-2,-2],[-1,3],[0,3],[1,3],[0,2],[1,2],[1,0],[2,-2],[1,2],[1,6],[1,5],[1,6],[0,11],[0,6],[0,5],[-1,6],[0,3],[0,1],[1,0],[2,-5],[0,-1],[3,2],[2,-3],[1,1],[0,1],[0,1],[0,2],[1,1],[0,1],[-1,3],[0,2],[1,1],[0,2],[1,0],[0,-1],[0,-2],[1,-1],[2,2],[0,2],[1,7],[0,6],[0,3],[0,1],[1,1],[1,5],[0,1],[0,2],[0,5],[0,11],[0,3],[0,2],[-1,7],[-2,7],[-5,18],[0,2],[-4,30],[0,3],[0,1],[-1,1],[-2,1],[0,1],[-1,1],[0,2],[0,1],[0,2],[0,1],[0,1],[1,0],[0,1],[1,1],[0,1],[-1,2],[0,3],[0,1],[0,1],[0,1],[-1,0],[0,2],[0,1],[0,6],[0,3],[0,1],[-2,2],[-1,0],[0,-2],[-1,-1],[-3,-3],[-1,-2],[-1,-9],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[1,-1],[0,-2],[0,-1],[-1,-2],[0,-2],[0,-2],[-1,-1],[0,-1],[-3,2],[0,1],[0,1],[0,2],[0,1],[0,1],[0,2],[1,2],[0,1],[0,1],[0,1],[0,1],[0,1],[-1,1],[-1,0],[-1,1],[-1,7],[-1,1],[-1,17],[0,3],[0,4],[0,1],[0,1],[5,21],[2,4],[0,2],[0,2],[0,1],[0,2],[0,1],[0,2],[0,4],[0,2],[0,2],[0,1],[0,1],[-1,0],[-1,-1],[0,-2],[-1,0],[0,1],[0,24],[0,4],[-1,2],[0,1],[-1,0],[0,1],[0,3],[-1,6],[0,3],[0,1],[-1,2],[0,1],[0,2],[0,3],[0,2],[0,2],[0,4],[0,2],[0,2],[-1,1],[0,2],[0,2],[1,2],[-1,3],[-1,5],[0,4],[2,2],[0,2],[0,3],[0,1],[1,6],[6,78],[5,58],[1,3],[1,0],[0,-1],[0,-1],[1,-1],[0,-2],[1,-1],[0,-1],[1,2],[1,-1],[0,-1],[0,-6],[1,-3],[0,-2],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[1,0],[1,-1],[2,-8],[1,-3],[0,-3],[0,-1],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[-1,-1],[0,-3],[0,-43],[0,-2],[0,-1],[0,-3],[0,-2],[0,-2],[0,-2],[0,-1],[0,-5],[0,-2],[0,-2],[1,0],[4,4],[1,-2],[1,-1],[1,1],[2,3],[10,7],[1,0],[1,-3],[1,-4],[0,-4],[1,-8],[1,-4],[0,-1],[10,1],[1,-4],[9,-1],[1,2],[1,3],[0,3],[1,3],[0,3],[0,4],[0,3],[1,3],[2,8],[1,2],[31,-4],[7,34],[2,3],[2,3],[0,1],[0,1],[0,2],[0,1],[0,1],[1,4],[0,5],[4,15],[1,3],[1,-1],[0,1],[0,1],[0,1],[1,1],[0,2],[0,1],[0,1],[0,1],[0,2],[0,1],[0,3],[1,1],[0,4],[5,18]],[[6957,3465],[0,1],[-1,-1],[-1,-10],[0,-3],[-1,-2],[0,-2],[0,-2],[0,-2],[-2,-4],[-1,-2],[-1,-8],[0,-3],[-1,1],[-4,13],[-1,2],[-2,1],[-1,3],[-2,16],[-1,4],[-3,10],[-6,4],[-4,9],[0,-1],[-3,-12],[-1,-2],[0,5],[-1,8],[-1,5],[-5,9],[0,-2],[-1,-2],[0,-2],[0,-1],[1,-9],[0,-2],[1,-2],[0,-2],[1,-9],[-1,-4],[0,-4],[-1,-2],[0,2],[-1,3],[-1,9],[0,3],[-1,1],[-2,2]],[[7421,2808],[-1,5],[0,4],[-1,4],[0,1],[-4,-9],[-1,0],[-1,1],[-1,1],[0,2],[-1,-2],[0,-4],[-1,-1],[0,-2],[0,-3],[0,-2],[0,-2],[0,-2],[0,-2],[0,-1],[0,-2],[0,-2],[0,-1],[-2,-3],[0,-5],[0,-3],[-3,-27],[0,-1],[0,-2],[0,-1],[1,-3],[0,-1],[0,-1],[1,-2],[4,-13],[0,-1],[0,-1],[0,-2],[0,-2],[-1,-1],[0,-4],[0,-2],[0,-1],[0,-3],[0,-1],[1,-3],[0,-2],[0,-1],[0,-1],[0,-3],[0,-1],[0,-4],[0,-3],[-1,-6],[0,-2],[-1,0],[0,-1],[-1,1],[-2,-1],[-3,3],[-1,0],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[-1,-3],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-1],[0,-1],[-1,-1],[-1,-1],[-2,0],[-2,2],[0,3],[-1,2],[0,2],[0,1],[0,1],[0,2],[0,1],[-1,1],[-6,1],[0,-1],[0,-5],[0,-4],[-2,-8],[0,-5],[0,-3],[0,-3],[1,-7],[0,-2],[-1,-5],[-1,-4],[-2,-3],[0,-3],[1,-3],[0,-3],[-1,-4],[1,-3],[-1,-3],[0,-1],[0,-3],[0,-3],[0,-2],[0,-4],[0,-3],[-1,2],[0,1],[-1,3],[0,-2],[-1,-2],[0,-2],[-1,-5],[0,-1],[-1,0],[-1,-1],[0,-2],[0,-1],[0,-2],[0,-4],[0,-2],[0,-1],[0,-1],[-1,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-5],[0,-1],[-1,-1],[-3,2],[0,-1],[-1,-1],[0,-2],[0,-1],[0,-1],[1,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-1],[-1,-7],[0,-1],[-1,1],[0,1],[0,2],[-1,1],[0,1],[0,1],[-2,1],[-2,-2],[-1,-3],[-1,-7],[0,-5],[-1,-1],[-1,0],[0,1],[0,-1],[-3,-4],[-1,-2],[-1,0],[0,-1],[-1,-9],[-4,-11],[0,-4],[-1,-2],[-2,-6],[-1,0],[0,1],[-1,4],[0,1],[-1,-1],[-1,-1],[0,-1],[0,-1],[-3,-32],[0,-4],[0,-2],[-1,-1],[0,2],[-1,2],[0,1],[0,1],[-2,-1],[0,-1],[0,-1],[0,-12],[0,-2],[0,-1],[0,-1],[7,-15],[0,1],[0,1],[1,0],[0,1],[0,-1],[1,-5],[4,-4],[1,-3],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-6],[0,-2],[0,-1],[1,-1],[1,0],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[1,-2],[0,-1],[0,-2],[-1,-2],[0,-3],[0,-1],[0,-2],[0,-1],[-1,-4],[0,-1],[0,-1],[0,-1],[0,-2],[0,-6],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[-1,0],[-1,0],[0,3],[-1,0],[-2,-7],[-5,-3],[-1,-1],[-1,-3],[-1,-2]],[[7335,2322],[-3,12],[-1,8],[0,15],[-1,5],[-1,3],[0,3],[-1,8],[0,3],[-1,2],[-1,1],[-1,0],[-1,3],[-2,8],[0,2],[-3,3],[-1,2],[0,3],[-1,7],[-3,2],[-3,-2],[-1,-1],[0,-1],[-1,1],[0,1],[-1,0],[-1,-10],[-1,-9],[-1,-6],[-1,-3],[-3,3],[-3,3],[-1,-1],[-1,-5],[0,-17],[0,-5],[-2,-5],[-4,-14],[-1,7],[-1,2],[-1,-2],[0,1],[0,1],[-1,0],[0,-1],[-1,-1],[0,-1],[0,1],[-1,1],[0,-3],[-1,-4],[0,-2],[-1,-2],[-1,0],[-1,2],[-1,2],[-2,8],[0,2],[-1,3],[-2,1],[-2,-2],[-2,4],[0,1],[-1,-3],[0,-7],[-1,-2],[-6,-2],[-1,3],[0,5],[0,6],[1,7],[0,5],[-1,6],[0,7],[-1,9],[0,8],[-3,1],[-3,-6],[-2,0],[-1,1],[0,1],[0,3],[0,3],[1,15],[1,26],[1,5],[0,4],[0,3],[-1,2],[-1,1],[0,-2],[-1,-4],[-1,1],[-1,3],[-1,4],[-1,3],[0,4],[0,9],[0,3],[-1,0],[0,-2],[-1,-2],[0,-2],[-1,1],[-4,10],[0,-1],[-1,-2],[0,-3],[0,-4],[0,-3],[0,-4],[-1,-6],[-1,-2],[-2,2],[-1,0],[0,-2],[-1,-3],[0,-5],[0,-8],[0,-5],[0,-3],[0,-3],[1,-4],[0,-4],[0,-4],[0,-4],[0,-5],[0,-3],[0,-9],[-1,-6],[-1,-3],[-2,-2],[-1,1],[0,-1],[-1,-3],[0,-2],[-1,-2],[0,-2],[-1,-4],[0,-8],[-1,-3],[-1,2],[-1,6],[0,1],[-1,-3],[0,-8],[2,-14],[-1,1],[-1,3],[-1,5],[0,6],[-1,4],[-1,13],[-1,17],[-1,15],[-3,30],[-2,30],[-3,29],[-2,30],[-3,30],[-2,17],[-1,14],[-3,31],[-3,30],[-4,40],[-4,40],[-4,40],[-4,39],[-3,28],[-3,29],[-3,28],[-3,28],[-2,18],[-3,25],[-2,14]],[[7160,2984],[0,4],[1,5],[1,4],[0,1],[0,1],[0,-1],[1,0],[0,-2],[1,0],[0,3],[0,2],[1,1],[0,2],[0,1],[-1,2],[0,1],[0,1],[0,1],[0,1],[0,2],[0,1],[1,1],[2,1],[1,2],[0,1],[0,1],[0,2],[0,1],[-1,1],[0,1],[0,1],[-3,0],[0,1],[0,1],[0,14],[0,2],[1,1],[4,3],[1,0],[0,1],[0,1],[0,2],[0,1],[1,1],[1,0],[0,-1],[0,-1],[0,-2],[0,-3],[0,-15],[0,-2],[1,-2],[0,-1],[0,-1],[0,-1],[3,-2],[1,1],[0,1],[1,3],[1,2],[1,1],[1,2],[2,-1],[0,1],[0,3],[0,3],[0,10],[0,4],[0,2],[0,3],[1,7],[1,-3],[0,-1],[1,0],[4,9],[1,1],[1,-2],[2,-4],[1,1],[0,1],[0,2],[0,3],[0,1],[1,1],[0,-1],[1,-2],[0,-1],[4,0],[0,1],[0,1],[0,1],[0,4],[1,1],[0,1],[2,1],[1,0],[0,-1],[0,-1],[0,-4],[0,-1],[1,0],[0,1],[4,9],[0,4],[1,3],[0,1],[0,2],[0,2],[0,1],[0,2],[0,1],[0,1],[1,7],[1,2],[0,1],[1,0],[0,-1],[1,-2],[0,-1],[1,3],[2,5],[0,1],[0,2],[0,2],[0,2],[0,1],[-1,1],[0,2],[1,2],[0,1],[0,2],[0,1],[0,1],[5,17],[0,3],[0,1],[0,2],[0,2],[0,1],[0,1],[0,1],[1,2],[0,2],[1,4],[0,-1],[1,-1],[1,-6],[1,0],[2,6],[0,1],[0,2],[0,1],[0,1],[1,2],[1,3],[1,4],[0,1],[1,-1],[1,-4],[1,6],[1,1],[0,1],[0,1],[-1,5],[0,1],[0,2],[1,2],[1,4],[0,1],[1,0],[0,-2],[1,1],[3,16],[1,-3],[1,0],[0,1],[0,1],[0,2],[0,1],[0,2],[-1,2],[0,9],[0,5],[3,11],[1,-6],[0,-1],[1,-13],[0,-8],[0,-2],[0,-1],[0,-1],[3,-13],[0,-3],[0,-12],[0,-1],[1,-1],[0,-1],[0,-1],[1,-2],[1,-5],[1,0],[0,-1],[1,0],[1,0],[0,-1],[1,-3],[0,-1],[2,0],[1,-2],[1,-2],[0,-1],[0,-6],[0,-1],[1,-1],[1,-2],[0,-1],[0,-1],[0,-6],[0,-1],[0,-2],[0,-5],[0,-3],[0,-1],[0,-1],[1,7],[3,9],[1,-1],[0,-2],[0,-4],[1,-1],[2,1],[0,-1],[0,-1],[0,-3],[0,-5],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[-3,-10],[-1,-3],[1,-11],[0,-5],[0,-1],[2,-4],[0,-2],[0,-1],[0,-4],[0,-2],[0,-1],[1,-1],[1,0],[0,1],[1,1],[1,0],[1,-3],[2,-6],[1,-4],[0,-1],[1,0],[0,2],[1,1],[1,6],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[-1,-7],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[1,0],[1,0],[0,-1],[2,-10],[1,-4],[1,-3],[1,-2],[0,-1],[0,-2],[1,-2],[0,-5],[0,-1],[0,-1],[1,0],[0,3],[1,3],[0,3],[0,1],[1,0],[1,-1],[0,1],[0,1],[0,1],[-1,1],[-1,7],[0,2],[0,3],[3,16],[0,1],[0,2],[0,4],[0,2],[0,1],[0,3],[0,2],[3,16],[0,1],[-1,5],[0,1],[8,45],[2,8],[3,14],[0,1],[1,0],[0,-2],[0,-1],[0,-1],[1,0],[0,1],[0,1],[0,1],[0,3],[0,1],[1,1],[0,1],[0,-1],[0,-1],[0,-1],[1,-8],[0,-9],[1,-1],[0,-3],[1,-9],[0,-1],[0,-1],[0,-1],[0,1],[1,1],[1,1],[0,2],[0,1],[0,1],[0,1],[0,2],[0,1],[1,2],[0,1],[0,1],[0,2],[0,1],[0,3],[0,7],[0,3],[-1,3],[0,1],[1,1],[1,9],[1,1],[0,1],[3,0],[2,-2],[0,1],[1,0],[0,2],[1,1],[0,-1],[1,0],[0,-1],[0,1],[1,2],[0,-1],[0,-2],[0,-1],[0,-3],[0,-1],[0,-2],[0,-1],[2,6],[1,6],[1,1],[0,1],[3,10],[1,0],[0,-1],[1,0],[0,1],[1,5],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-3],[0,-2],[0,-1],[1,1],[0,1],[1,2],[1,-3],[1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,1],[1,1],[1,6],[1,2],[0,3],[0,2],[1,2],[0,8],[0,1],[0,2],[0,1],[2,10],[1,10],[11,32]],[[7363,3264],[0,1],[1,4],[1,0],[7,-62],[0,-6],[1,-2],[3,7],[1,1],[1,-3],[0,-1],[2,-15],[1,-1],[0,-1],[2,-6],[2,-15],[2,-6],[3,-7],[1,-3],[1,-3],[1,-8],[0,-1],[1,-2],[0,-8],[0,-1],[0,-2],[0,-1],[1,-2],[0,-1],[1,0],[0,-1],[3,-23],[0,-8],[0,-14],[1,-10],[0,-3],[1,-7],[0,-1],[0,-1],[1,0],[0,-1],[1,-4],[0,-7],[1,-2],[0,-3],[1,-5],[2,-5],[1,-3],[2,-2],[2,0],[3,7],[1,2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[-2,-5],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[1,-1],[1,-1],[0,-1],[-1,-2],[-2,-24],[-1,-3],[0,-3],[-1,-3],[-2,-2],[0,-1],[0,-2],[-1,-1],[0,-1],[0,1],[-1,2],[0,-1],[0,-2],[0,-6],[0,-1],[0,-3],[1,-1],[0,-1],[1,-1],[0,-1],[0,-2],[1,-3],[0,-1],[0,-1],[0,-1],[2,-6],[1,-2],[0,-7],[1,-2],[0,-1],[2,2],[2,-2],[0,-1],[0,-3],[0,-7],[0,-1],[0,-1],[0,-1],[-1,-3],[0,-1],[0,-1],[0,-1],[0,-6],[0,-3],[0,-9],[0,-1],[0,-2],[-1,-1],[-1,-7],[-1,-5],[0,-1],[1,-3],[0,-5],[0,-3],[1,-2],[0,-1],[0,-3],[0,-1],[0,-1],[1,-2],[0,-1],[0,-2],[1,-5],[0,-5],[0,-2],[0,-1],[0,1],[1,3],[0,1],[1,0],[0,-1],[1,-1],[0,-3],[0,-18],[0,-1],[0,-1],[0,-3]],[[7421,2808],[0,-2],[0,-1],[-1,-1],[1,-1],[0,-1],[0,-1],[1,0],[3,3],[1,0],[0,-2],[0,-1],[0,-11],[0,-5],[0,-2],[0,-1],[1,0],[2,-1],[3,-4],[0,-1],[1,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[1,-1],[1,1],[1,8],[0,1],[3,4],[0,1],[0,1],[1,5],[0,1],[0,1],[2,1],[0,-1],[0,-1],[1,-5],[0,-1],[1,0],[0,-1],[1,-4],[2,-6],[1,-8],[1,-3],[0,-2],[0,-1],[1,0],[1,2],[1,6],[3,2],[1,-1],[0,-1],[0,-3],[-1,-6],[0,-3],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[1,-2],[0,-1],[-1,-1],[-1,-9],[0,-2],[0,-2],[0,-25],[0,-7]],[[7454,2699],[-1,-9],[0,-2],[-2,-1],[0,-1],[-1,-1],[0,-2],[0,-1],[0,-1],[0,-1],[1,-3],[0,-1],[0,-2],[0,-1],[-1,-2],[-1,0],[0,1],[0,1],[-1,7],[-1,1],[-1,-2],[0,-3],[0,-1],[0,-2],[-1,-3],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-4],[0,-3],[-1,-3],[-1,-2],[0,-1],[0,-7],[0,-1],[0,-1],[0,-2],[-2,-16],[-1,-7],[-1,-2],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[1,-1],[0,1],[1,3],[0,1],[1,-1],[0,-2],[1,-5],[0,-1],[3,6],[0,-1],[1,-1],[0,-1],[0,-1],[0,-2],[0,-2],[-1,-4],[0,-2],[0,-2],[-2,-13],[-1,-3],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[0,-2],[-3,-11],[1,-9],[0,-6],[1,-3],[0,-1],[1,3],[3,-9],[0,-2],[1,-2],[0,-3],[0,-3],[0,-2],[0,-1],[0,-3],[1,-4],[1,-1],[2,-1],[1,-3],[1,-6],[1,-1],[4,-1],[0,1],[1,4],[0,1],[0,2],[1,9],[0,1],[0,1],[1,1],[0,2],[1,3],[0,1],[0,1],[0,1],[0,1],[0,4],[0,5],[0,3],[0,1],[0,3],[0,3],[0,2],[0,1],[0,1],[0,1],[0,1],[2,6]],[[7464,2549],[3,-11],[0,-1],[0,-1],[0,-5],[0,-1],[1,-2],[0,-3],[2,-6],[0,-1],[1,-1],[0,-3],[0,-1],[0,-2],[0,-3],[-1,-4],[0,-4],[0,-3],[0,-1],[0,-3],[0,-10],[0,-2],[0,-3],[0,-5],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[1,-3],[1,-2],[0,-1],[0,-1],[0,-1],[1,-3],[0,-2],[0,-3],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[0,-4],[0,-1],[0,-2],[0,-1],[0,-3],[0,-2],[0,-2],[1,-2],[1,-7],[1,-3],[0,-2],[1,-6],[3,-16],[0,-2],[1,-3],[1,-6],[0,-1],[1,-2],[0,-1],[0,-3],[0,-1],[0,-1],[0,-1],[1,-3],[0,-1],[1,-4],[0,-1],[0,-7],[2,-13],[0,-3],[1,-2],[0,-2],[0,-2],[1,-10],[1,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[1,-1],[1,1],[1,-2],[0,-1],[1,-3],[0,-3],[1,-6],[0,-1],[1,-3],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-3],[-1,-1],[0,-4],[0,-2],[0,-1],[0,-1],[-1,0],[-1,-2],[-4,-8],[0,1],[-1,1],[0,1],[1,3],[0,1],[0,2],[-1,1],[0,1],[-2,9],[0,1],[0,-1],[-1,-2],[0,-2],[0,-4],[0,-1],[0,-3],[0,-1],[0,-1],[0,-2],[0,-1],[1,-2],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-4],[-1,-1],[-1,-1],[0,-1],[0,-1],[-1,-1],[0,-2],[0,-3],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-2],[0,-1],[0,-3],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[1,-4],[1,-1],[0,1],[0,1],[0,2],[1,2],[1,-1],[1,0],[0,-1],[0,-3],[0,-2],[0,-9],[0,-2],[0,-1],[0,-2],[0,-2],[0,-2],[1,0],[0,-1],[1,0],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[-1,-1],[0,-5],[0,-1],[0,-2],[0,-2],[0,-3],[1,-1],[0,-1],[0,-1],[0,-2],[0,-4],[0,-3],[0,-2],[0,-3],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[1,-3]],[[7489,2143],[0,-1],[0,-3],[0,-9],[0,-3],[0,-2],[0,-3],[0,-1],[0,-3],[0,-2],[0,-3],[0,-3],[1,-3],[1,-2],[0,-5],[0,-3],[-1,-3],[-1,-6],[-3,-5],[-1,-5],[-1,0],[-1,-5],[0,-6],[-1,-4],[-1,-1],[-1,4],[-2,8],[-1,2],[0,-3],[-1,-6],[1,-7],[0,-6],[0,-6],[-1,-5],[-4,-10],[-1,-1],[-1,1],[0,3],[0,2],[0,3],[-1,3],[0,4],[0,3],[-1,2],[0,-1],[0,-3],[-1,-4],[0,-9],[1,-8],[0,-1],[-4,2],[-1,1],[-1,3],[0,6],[-1,1],[0,1],[-1,-1],[-1,-4],[0,-2],[-2,0],[-1,0],[-1,3],[-1,0],[-5,-6],[0,-2],[-1,-4],[0,-8],[-1,-2],[0,-8],[0,-7],[1,-7],[0,-7],[-1,-5],[-2,-8],[-1,-6],[-1,-4],[0,-2],[-3,-2],[-1,0],[0,1],[-3,-1],[-5,-10],[-1,0],[0,-4],[-1,-1],[0,-2],[0,-1],[-1,-3],[-1,-1],[-1,1],[-1,3],[-1,7],[0,17],[-1,5],[-1,6],[-1,2],[-1,1],[-2,-1],[-1,0],[0,1],[0,2],[-1,2],[0,4],[0,2],[0,2],[-2,8],[0,4],[-1,3],[1,5],[0,2],[0,3],[-1,3],[-1,4],[-1,3],[0,4],[0,6],[0,5],[0,4],[-1,3],[-1,7],[-1,1],[-1,-2],[-1,0],[-1,3],[0,4],[0,3],[0,4],[0,4],[1,8],[1,3],[1,1],[1,1],[0,2],[1,4],[-1,4],[0,2],[0,3],[0,3],[0,3],[0,2],[0,3],[0,1],[-1,0],[0,-1],[-1,0],[-1,4],[-1,0],[0,-3],[-1,-5],[0,-5],[-1,0],[0,-1],[0,-2],[0,-1],[-2,-4],[0,-3],[1,-3],[0,-2],[-1,-1],[0,-2],[-2,-10],[-2,-3],[-1,-7],[-1,-8],[-1,-13],[-1,-8],[-1,7],[-1,7],[-1,1],[-1,-3],[-1,-5],[0,-3],[-1,-1],[0,1],[-1,3],[0,4],[1,10],[0,2],[-2,1],[-1,-1],[-1,-4],[0,-1],[-1,1],[0,3],[0,3],[-1,2],[-1,-3],[-1,-1],[0,2],[-1,3],[0,7],[-1,2],[0,1],[0,-2],[-1,-4],[-1,-1],[0,-1],[-1,-1],[-1,1],[-1,6],[-1,1],[-1,0],[0,-2],[-1,-2],[0,-2],[-1,-2],[-1,3],[-1,8],[0,19],[-1,7],[-2,12],[0,4],[-1,4],[0,9],[0,5],[-1,3],[-1,2],[-1,4],[0,6],[0,14],[0,1],[2,1],[0,2],[0,4],[0,3],[-1,8],[0,2],[-2,5],[-1,0],[0,-1],[-1,-2],[0,-1],[0,2],[-1,9],[0,2],[-1,2],[-1,-1],[-1,4],[-1,4],[-1,4],[-1,1],[-1,-1],[-1,2],[0,3],[-1,5],[-1,5],[-1,-4],[-1,-4],[-1,0],[0,2],[-1,4],[0,1],[0,2],[-1,1],[0,1],[-1,5],[0,5],[0,6],[0,5],[0,8],[-1,8],[0,8],[0,13],[0,4],[0,4],[-1,6],[0,4],[0,4],[-1,4],[-1,7],[0,2]],[[7461,3505],[-1,-7],[0,-5],[-1,-3],[-1,-11],[0,-1],[-2,-6],[-1,-2],[0,-1],[0,-3],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[2,-10],[0,-2],[0,-1],[0,-5],[1,-2],[0,-2],[0,-3],[0,-2],[0,-2],[0,-2],[0,-2],[1,-4],[0,-2],[0,-1],[0,-2],[0,-6],[0,-2],[0,-2],[0,-1],[0,-2],[0,-2],[0,-1],[1,-1],[0,-2],[0,-3],[1,-2],[0,-1],[1,0],[0,-1],[1,-2],[0,-2],[0,-2],[-1,-2],[0,-5],[0,-2],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[1,0],[0,1],[1,4],[0,1],[0,-1],[0,-1],[1,-14],[0,-4],[0,-3],[-1,-4],[0,-4],[0,-2],[-1,-1],[-1,1],[0,-1],[-1,-1],[0,-2],[0,-2],[0,-2],[0,-2],[0,-2],[0,-1],[0,-4],[0,-2],[0,-18],[0,-1],[0,-2],[-1,-6],[0,-1],[0,-1],[0,-8],[1,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-3],[-1,0],[0,-1],[0,-1],[0,-1],[0,-2],[0,-3],[0,-1],[0,-1],[0,-3],[0,-2],[0,-3],[-1,-1],[0,-1],[0,-1],[-2,-3],[-2,-1],[0,-1],[0,-1],[0,-2],[1,-1],[3,-8],[1,-1],[0,-1],[0,-1],[0,-8],[0,-1],[0,-1],[0,-1],[1,0],[0,3],[1,1],[0,2],[0,1],[0,1],[1,0],[0,1],[1,2],[0,2],[0,1],[0,3],[1,2],[0,1],[1,2],[0,2],[0,1],[1,3],[0,1],[0,1],[0,1],[1,0],[1,-1],[2,-8],[0,-2],[0,-1],[0,-2],[0,-2],[0,-4],[0,-2],[0,-1],[0,-2],[1,-1],[1,-5],[0,1],[1,8],[0,1],[0,1],[2,4],[1,0],[0,-2],[0,-2],[1,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[-1,-2],[0,-1],[0,-2],[1,-1],[0,-1],[0,-2],[1,-1],[0,-1],[0,-2],[-1,-10],[-1,-2],[0,-2],[1,-7],[0,-1],[0,-1],[0,-1],[-4,-7],[-1,-3],[0,-1],[-1,-2],[0,-2],[0,-2],[0,-1],[0,-2],[0,-2],[0,-1],[0,-2],[2,-13],[0,-2],[0,-2],[0,-3],[0,-1],[0,-1],[1,-1],[1,2],[1,0],[1,-3],[0,-1],[1,-1],[0,-2],[-1,-7],[-1,-2],[0,-2],[0,-2],[1,-1],[0,-2],[0,-1],[1,1],[1,0],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[-2,-7],[0,-6],[-1,0],[-2,0],[0,-3],[-1,-1],[-1,-3],[0,-1],[0,-1],[0,-3],[0,-2],[0,-2],[0,-6],[0,-3],[0,-1],[0,-2],[-1,-1],[-2,-9],[0,-3],[-1,-3],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,-5],[0,-2],[0,-2],[0,-10],[0,-5],[0,-1],[-1,-2],[0,-2],[-1,-5],[0,-2],[0,-1],[0,-2],[0,-3],[0,-2],[-1,-5],[0,-3],[0,-1],[0,-2],[0,-1],[1,0],[0,1],[1,4],[1,4],[1,0],[1,-9],[1,-1],[0,-1],[0,-3],[0,-2],[0,-1],[1,-4],[1,-12],[0,-2],[0,-1],[0,-3],[-1,-3],[0,-4],[0,-2],[0,-1],[0,-7],[1,-3],[0,-1],[0,-1],[0,-2],[-1,-3],[0,-2],[0,-1],[0,-5],[0,-1],[-1,-1],[0,-1],[0,-1],[-1,-5],[-1,-4],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-4],[0,-1],[0,-1],[0,-1],[0,-1],[1,-4],[1,-3],[2,1],[0,-1],[2,-7],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-4],[0,-1],[0,-2],[0,-1],[1,2],[1,-1],[1,-14],[0,-1],[0,-2],[-2,-7],[0,-4],[0,-4],[-1,-1],[-2,-4],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-3],[1,-5],[1,-3],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-1],[0,-1],[-1,-1],[0,-2],[0,-7],[-1,-1],[0,-1],[-1,-3],[0,-1],[0,-2],[-1,-2],[0,-3],[0,-1],[0,-1],[-2,-4],[0,-3],[0,-3],[-3,-10],[-1,-1],[0,-2],[0,-2],[0,-2],[0,-2],[-2,-4],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[1,-2],[1,-1],[0,-1],[0,-1],[0,-1],[-1,-5],[0,-7],[0,-1],[-1,-1],[0,-1],[-1,0],[0,-1],[0,-2],[0,-1],[-1,-1],[0,-1],[-1,1],[0,-2],[-1,-3],[-1,-2],[0,-2],[0,-1],[0,-2],[-1,0],[-2,-1]],[[7363,3264],[-4,34],[2,10],[0,2],[1,2],[-1,1],[0,1],[0,9],[0,1],[-1,1],[-1,1],[0,1],[0,2],[0,1],[0,1],[0,1],[1,1],[0,2],[0,1],[1,1],[0,2],[0,3],[0,3],[0,2],[0,2],[1,3],[0,2],[0,2],[0,1],[1,2],[-1,4],[0,1],[0,1],[0,1],[-1,0],[0,1],[0,1],[0,1],[0,3],[-1,2],[0,6],[0,1],[0,1],[-1,1],[-1,1],[0,1],[0,1],[0,1],[-1,3],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[1,2],[0,2],[0,2],[0,3],[0,30],[0,1],[0,2],[0,1],[-2,11],[-1,5],[1,6],[0,3],[1,1],[0,2],[0,1],[0,21],[-1,4],[-1,-1],[0,-1],[-1,1],[0,1],[-1,-2],[0,-1],[-1,0],[-1,2],[-1,1],[0,1],[0,1],[0,1],[3,7],[1,3],[0,1],[0,2],[0,1],[-1,2],[0,1],[0,2],[0,2],[0,1],[0,2],[1,1],[0,2],[0,2],[0,2],[0,7],[0,1],[-1,1],[0,5],[0,7],[-1,9],[0,1],[0,1],[-1,0],[-1,1],[0,3],[-1,2],[0,1],[0,1],[-1,1],[0,4],[0,2],[0,1],[0,1],[2,3],[0,1],[0,1],[0,2],[0,1],[0,2],[0,2],[-1,1],[0,2],[0,1],[0,1],[0,1],[1,1],[0,3],[0,1],[-1,2],[-1,0],[0,1],[0,3],[0,1],[-1,1],[0,2],[0,5],[0,2],[0,2],[0,2],[0,2],[0,1],[0,2],[0,2],[-1,1],[1,1],[1,1],[0,1],[0,4],[0,2],[-1,2],[0,1],[1,2],[0,1],[0,1],[0,1],[0,1],[0,2],[-1,1],[0,1],[-3,1]],[[7461,3846],[1,-2],[0,-1],[0,-2],[0,-2],[0,-4],[0,-10],[0,-2],[0,-1],[2,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-2],[0,-2],[0,-7],[-1,-1],[0,-1],[0,-1],[-4,-2],[0,-1],[0,-1],[0,-1],[0,-4],[0,-2],[0,-2],[0,-4],[0,-3],[0,-1],[1,-1],[2,-2],[0,-2],[0,-3],[0,-7],[0,-3],[0,-2],[-1,-1],[0,-2],[0,-3],[2,-5],[0,-1],[1,-8],[1,-2],[0,-1],[4,-2],[0,-3],[1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[4,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[0,-5],[0,-2],[0,-1],[0,-1],[1,-1],[0,-1],[1,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-2],[0,-2],[0,-1],[1,0],[1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[1,-1],[0,-1],[1,-3],[0,-2],[0,-1],[0,-1],[0,-3],[-1,-2],[1,-4],[0,-8],[0,-4],[1,-5],[0,-2],[0,-6],[0,-2],[0,-1],[0,-1],[1,-2],[0,-2],[1,-1],[1,-2],[1,-2],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[-1,-1],[0,-1],[-1,0],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[0,-1],[0,-3],[0,-1],[0,-1],[0,-1],[-3,-2],[-1,-3],[0,-3],[-1,-4],[-1,-5],[0,-1],[-2,0],[-1,-2],[-5,-17],[0,-4],[-5,-22],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-2],[-1,-6],[0,-1],[0,-1],[0,-2]],[[7512,2702],[-1,-1],[-1,1],[-1,2],[0,-1],[-1,0],[0,-2],[0,-2],[-1,-1],[-1,-1],[0,-1],[0,-1],[0,-2],[-1,-1],[0,-1],[-1,-1],[-1,1],[0,-2],[0,-13],[0,-2],[0,-3],[0,-3],[-1,-2],[0,-1],[-1,-2],[0,-1],[0,-3],[-1,-3],[0,-1],[0,-2],[0,-2],[0,-2],[1,-11],[0,-8],[0,-4],[0,-5],[0,-1],[0,-2],[0,-5],[1,-13],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-2],[0,-1],[-1,0],[0,1],[-1,2],[0,2],[-1,1],[0,1],[0,-3],[-5,-31],[-1,-7],[0,-3],[-2,-4],[-1,-1],[-1,-3],[-1,1],[0,1],[0,1],[-2,7],[0,1],[0,1],[-1,1],[-1,0],[-2,-1],[0,-2],[-1,-1],[0,-7],[0,-1],[-1,-1],[0,-1],[-1,1],[0,1],[0,1],[0,1],[-1,1],[0,1],[0,2],[0,2],[0,2],[1,5],[0,6],[0,1],[0,1],[0,1],[0,1],[-1,0],[-1,-2],[0,-2],[-1,-2],[-2,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-1],[-8,-10]],[[7461,3505],[0,-6],[2,-3],[2,-16],[2,0],[1,-9],[2,-5],[1,-14],[0,-12],[2,-13],[0,-11],[3,-2],[1,-21],[4,2],[1,-7],[1,4],[1,12],[3,2],[0,3],[2,-6],[1,-7],[2,7],[2,4],[3,9],[2,3],[2,-2],[2,4],[0,8],[2,3],[1,-11],[0,-10],[1,-4],[2,0],[2,-8],[1,-16],[1,-6],[-1,-8],[0,-6],[2,4],[4,5],[1,-9],[0,-10],[4,2],[2,-11],[-2,-32],[0,-9],[3,-13],[4,-12],[1,-9],[1,-12],[1,-13],[0,-12],[1,-15],[2,-13],[1,-14],[1,-12],[2,-15],[0,-22],[-1,-18],[-2,-10],[-1,-9],[0,-15],[3,-22],[1,-15],[0,-9],[0,-8],[0,-5],[2,-9],[3,0],[2,-6],[1,-2],[2,-15],[1,-8],[-1,-16],[-1,-13],[-2,-9],[-2,-10],[-2,-9],[-2,-4],[-1,0],[-1,1],[-1,-4],[-1,-4],[1,-13],[0,-19],[0,-19],[-2,-17],[-2,-13],[-3,-12],[-3,-9],[-1,-15],[-3,-6],[-1,-6],[-2,-6],[-2,-17],[-2,-19],[-1,-12],[-2,-18],[-1,-14],[-1,-5]],[[7160,2984],[-1,8],[-4,20],[-4,16],[-5,22],[-5,22],[-5,21],[-5,22],[-4,15],[-2,10],[-1,7],[0,9],[0,3],[-3,26],[0,2],[0,2],[1,1],[2,-1],[3,-7],[2,0],[1,9],[1,2],[0,4],[1,2],[0,2],[0,8],[1,4],[0,4],[1,5],[0,6],[0,6],[-1,3],[-1,2],[-1,3],[1,3],[0,3],[1,2],[2,3],[1,2],[0,6],[-1,2],[-1,-1],[-2,-5],[-1,0],[-1,0],[-1,-1],[0,-3],[-1,-5],[0,-5],[0,-4],[0,-2],[-1,-1],[-1,1],[-1,-2],[-1,-6],[-2,-2],[-5,8],[-1,-2],[0,-4],[-1,-4],[0,-4],[0,-4],[1,-6],[-4,-10],[-5,-12],[-3,-8],[-2,-4]],[[6270,314],[-1,9],[-2,6],[-3,3],[-1,0],[-3,-5],[-1,-1],[-1,3],[0,3],[0,4],[0,3],[-1,4],[-3,21],[0,5],[-1,2],[-1,0],[-1,-2]],[[6264,694],[1,-4],[0,-1],[3,-4],[1,-4],[0,-2],[1,-1],[1,1],[0,2],[0,2],[1,1],[0,1],[2,-2],[1,-3],[3,-6],[1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-10],[0,-2],[0,-1],[1,-1],[0,1],[0,2],[0,1],[0,3],[0,1],[1,1],[0,1],[2,-3],[1,0],[0,1],[0,3],[1,5],[0,3],[0,3],[0,7],[0,1],[0,2],[2,2],[0,2],[0,2],[1,3],[0,1],[0,1],[0,1],[2,0],[0,-1],[1,-4],[0,-2],[0,-2],[0,-1],[0,-1],[0,-3],[0,-9],[0,-4],[0,-2],[0,-1],[1,-3],[1,0],[0,2],[1,1],[0,-1],[1,-4],[0,-3],[0,-1],[0,-4],[1,-2],[1,1],[0,-1],[0,-3],[-1,-5],[0,-1],[-1,-1],[-1,-1],[0,-3],[-1,-13],[-1,-4],[0,-2],[0,-1],[0,-1],[-2,-16],[0,-4],[0,-4],[1,-4],[-1,-5],[0,-10],[-1,-4],[0,-2],[-1,-4],[-1,-2],[0,-1],[0,-3],[0,-1],[0,-2],[0,-3],[1,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[1,-4],[0,-2],[1,-3],[0,-1],[0,-1],[0,-2],[0,-2],[0,-6],[0,-12],[0,-2],[0,-1],[1,-1],[0,-3],[0,-2],[0,-2],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[1,-4],[0,-4],[1,0],[1,4],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-3],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-3],[0,-2],[0,-1],[0,-1],[0,-1],[1,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[-1,-2],[0,-2],[-1,-4],[-1,-4],[0,-2],[-1,-1],[-1,4],[0,2],[0,1],[-1,1],[0,-3],[0,-3],[0,-1],[-1,-1],[-1,0],[0,1],[0,1],[0,2],[-1,3],[0,1],[0,1],[0,-1],[-1,-4],[0,-5],[-1,-4],[0,-1],[0,-1],[0,-1],[0,-2],[0,-3],[0,-2],[0,-2],[0,-1],[1,-2],[0,-1],[0,-1],[0,-1],[0,-4],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[0,-2],[-1,-1],[0,-1],[-1,0],[-1,-1],[-1,-5],[0,-2],[0,-7],[0,-10],[0,-3],[0,-1],[0,-1],[-1,-1],[0,1],[0,1],[0,4],[0,1],[-1,1],[0,-1],[0,-2],[0,-2],[0,-3],[0,-2],[0,-1],[-1,-2],[-2,-12],[0,-4],[-1,-3],[0,-1],[0,-1],[-1,-1],[-1,0],[0,-2]],[[6329,690],[0,-1],[0,-1],[0,-2],[-3,-13],[0,-1],[-1,2],[-1,1],[0,3],[-1,-2],[0,1],[0,2],[-1,2],[1,5],[0,3],[4,0],[1,1],[1,0]],[[6302,875],[-1,-3],[0,-3],[0,-4],[-1,-4],[-1,3],[-1,-6],[0,-7],[-1,0],[0,-4],[0,-17],[0,-3],[0,-2],[0,-1],[-1,0],[0,-1],[0,-3],[0,-5],[1,-10],[0,-5],[0,-4],[1,-4],[0,-3],[1,-2],[1,0],[0,1],[1,0],[4,-7],[1,-2],[0,-4],[0,-4],[1,-11],[0,-5],[0,-3],[1,-4],[0,-3],[1,-1],[0,-1],[1,-3],[0,2],[1,4],[0,1],[1,-5],[0,-10],[1,-19],[1,-8],[0,-5],[1,-2],[0,-3],[1,-15],[0,-8],[1,-4],[0,-6],[0,-2],[1,-1],[0,-3],[0,-2],[-1,-6],[0,-2],[0,1],[1,1],[0,-1],[1,-2],[0,-9],[0,-2],[0,-2],[-1,-1],[0,-1],[-1,-2],[0,-2],[0,-5],[1,-7],[-1,-2],[0,-1],[0,-1],[0,-1],[-1,-8],[0,-8],[-1,3],[0,1],[0,-4],[0,-6],[0,-5],[1,-2],[0,-2],[0,-4],[0,-4],[1,-2],[0,4],[1,8],[0,10],[0,9],[0,4],[0,2],[1,2],[0,3],[0,3],[0,7],[0,3],[0,6],[1,6],[0,7],[0,7],[0,6],[0,2],[0,3],[1,5],[0,1],[1,0],[1,-1],[1,-2],[0,-4],[-1,-11],[-1,-15],[-2,-34],[-1,-27],[0,-14],[-1,-10],[0,-10],[1,-8],[0,-5],[-1,-4],[1,-4],[0,-1],[1,7],[0,1],[1,2],[0,-1],[0,-2],[0,-2],[0,-3],[0,-2],[0,-1],[0,-5],[-1,-2],[-1,-4],[0,-2],[-1,-5],[0,-3],[0,-5],[0,-2],[0,-3],[0,-3],[0,-3],[0,-2],[0,-3],[0,-3],[0,-1],[0,-5],[0,-14],[2,-15],[1,-5],[1,-2],[0,-4],[0,-3],[2,-9],[0,-2],[1,-5],[0,-2],[0,-2],[0,-12],[0,-4],[0,-3],[0,-2],[0,-10],[0,-9],[1,-6],[2,-8],[1,-5],[2,-25],[2,-15],[0,-7],[2,-7],[0,-3],[4,-43],[1,-7],[1,-7],[0,-4],[0,-5],[0,-6],[1,-5],[1,-25],[1,-5],[2,-4],[1,-2],[1,-4],[1,-7],[0,-1],[-2,-14],[-1,-16],[-1,-2],[-1,-21],[0,-7],[-1,-5],[-2,-10],[-2,-8],[-1,-3],[-3,-4],[-1,-4],[0,-7],[-1,-17],[-1,-3],[-1,1],[0,-2],[0,-4],[0,-4],[-1,-4],[0,-1],[0,-3],[0,-5],[0,-5],[0,-3],[0,-2],[-1,-3],[-1,-3],[-2,-1],[-1,1],[-1,3],[-1,5],[-1,0],[0,-1],[-1,-5],[-1,0],[0,3],[-1,7],[-1,2],[-3,1],[0,1],[-1,2],[-2,8],[0,3],[0,6],[0,13],[0,6],[-1,3],[-1,7],[0,2],[0,3],[0,2],[0,2],[-1,2],[0,4],[0,4],[0,4],[-1,4],[0,1],[-1,-1],[0,-1],[-1,-1],[-1,4],[0,7],[-1,8],[-1,8],[0,3],[-1,2],[0,4],[0,2],[0,1],[-1,1],[-1,2],[-1,11],[-1,3],[0,4],[0,9],[0,3],[-1,0],[0,-3],[0,-1],[0,-2],[-2,-1],[0,-2],[-1,-5],[-1,-2],[-1,3],[0,4],[0,10],[0,5],[0,2],[-2,1],[-1,1],[-1,6],[0,2],[-2,1],[-1,2],[-1,10],[-1,4],[0,1],[0,-1],[-1,0],[-1,-2],[-1,0],[-1,2],[0,5],[0,2],[-1,-1],[-1,3],[0,1],[0,-1],[-1,-2],[-1,2],[0,2],[-1,13],[0,3],[-2,3],[-3,14],[-1,3],[-1,4],[0,6],[3,55],[0,5]],[[8308,4187],[-2,3],[-1,0],[-1,-1],[-5,-4],[-1,-2],[-1,-5],[-1,-1],[-1,-1],[0,-1],[-1,1],[0,3],[-1,4],[0,1],[-1,1],[-1,0],[0,1],[-1,1],[0,1],[0,2],[0,1],[0,1],[0,2],[-2,3],[0,1],[-1,4],[0,1],[0,2],[0,3],[0,4],[0,1],[-1,4],[0,1],[0,1],[-4,8],[-1,0],[0,-1],[0,-2],[-1,-3],[0,-1],[0,-1],[0,-1],[0,-1],[-1,0],[0,-1],[0,1],[-1,0],[0,1],[0,2],[0,5],[0,1],[-1,2],[0,2],[-1,1],[-1,0],[-1,-1],[-1,-1],[0,1],[0,1],[0,1],[-1,1],[0,-1],[0,-1],[0,-1],[-1,0],[-3,0],[0,-1],[0,-2],[0,-1],[-1,-2],[0,-1],[-1,-3],[-2,-4],[-1,-4],[0,-1],[0,-1],[-1,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[1,-2],[0,-1],[1,-1],[0,-1],[0,-3],[0,-1],[1,-3],[0,-1],[0,-1],[0,-2],[0,-1],[1,-6],[-1,-1],[0,-1],[0,-7],[0,-2],[0,-1],[-2,0],[0,1],[-1,2],[-1,1],[0,-1],[0,-1],[0,-3],[0,-1],[-1,-2],[0,-1],[0,-1],[1,-3],[0,-1],[0,-2],[-1,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-5],[0,-1],[0,-1],[0,-1],[0,-3],[-1,-1],[0,-1],[0,-1],[0,-1],[1,-4],[0,-1],[0,-2],[0,-1],[0,-3],[-1,-1],[0,-2],[0,-1],[1,-1],[0,-1],[0,-1],[1,-1],[0,-2],[-1,-1],[-1,-3],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-3],[-1,0],[0,1],[-1,0],[-1,-3],[-1,0],[-1,1],[-1,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[1,-3],[0,-1],[0,-1],[1,-3],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-3],[0,-1],[0,-4],[-1,-2],[0,-1],[0,-2],[0,-3],[0,-1],[0,-1],[0,-2],[1,0],[0,1],[3,-5],[0,-1],[0,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-2],[0,-1],[0,-2],[0,-4],[-1,-4],[0,-2],[0,-3],[0,-1],[0,-1],[1,-2],[0,-2],[0,-2],[0,-1],[0,-3],[0,-1],[0,-1],[1,-1],[0,-1],[0,-4],[1,-1],[0,-2],[-1,-2],[0,-1],[1,-2],[0,-2],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[0,-3],[0,-2],[0,-1],[0,-3],[1,0],[0,-1],[0,1],[1,2],[1,-1],[0,-2],[0,-4],[-1,-4],[0,-5],[0,-1],[0,-2],[0,-1],[1,-1],[0,-2],[1,3],[1,2],[0,1],[1,1],[0,11],[1,1],[0,1],[0,-1],[1,-2],[0,-1],[4,-1],[0,-1],[0,-1],[0,-2],[0,-10],[-1,-6],[0,-2],[0,-1],[0,-3],[0,-1],[-1,-1],[-2,-6],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-4],[0,-2],[0,-1],[1,-2],[1,-5],[0,-1],[0,-1],[0,-2],[0,-1],[0,-3],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-3],[0,-2],[0,-2],[0,-2],[0,-1],[0,-1],[-1,-2],[-1,-3],[-1,-3],[0,-1],[0,-2],[0,-5],[0,-3],[0,-5],[0,-3],[0,-2],[-1,4],[0,1],[-1,0],[0,-1],[-1,-3],[0,-2],[0,-2],[0,-2],[0,-2],[0,-1],[-1,0],[0,1],[0,1],[-1,1],[0,1],[0,1],[0,1],[0,1],[1,1],[0,1],[-1,1],[-1,2],[0,1],[0,3],[0,3],[0,4],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[-1,0],[0,-2],[0,-1],[0,-1],[0,-1],[-1,-1],[-1,-2],[0,-3],[-1,-2],[0,-3],[-1,-1],[0,-1],[0,-1],[0,-6],[0,-1],[0,-2],[0,-2],[-1,-3],[0,-1],[-1,1],[0,1],[-2,0],[0,-2],[-1,-1],[0,-1],[1,-3],[0,-2],[0,-1],[-1,1],[0,1],[-1,0],[0,-1],[-4,-13],[-1,0],[0,1],[0,1],[-1,5],[0,1],[0,1],[-1,0],[-1,-3],[-1,0],[-2,2],[0,1],[0,1],[0,1],[-1,4],[0,1],[0,1],[0,1],[-1,1],[-1,2],[0,2],[-1,1],[-1,1],[0,1],[0,1],[0,1],[0,2],[-1,0],[-1,-1],[0,-1],[-1,-1],[0,-3],[0,-1],[-1,0],[-1,0],[-1,-2],[-1,0],[0,1],[-1,2],[-1,6],[0,2],[0,1],[0,1],[-1,-1],[-1,-1],[0,1],[-1,1],[0,1],[-2,14],[0,1],[0,1],[-5,8]],[[8213,3878],[2,12],[1,4],[7,13],[1,4],[1,6],[1,4],[2,2],[0,4],[-1,3],[0,4],[0,2],[0,1],[-1,1],[0,1],[0,1],[0,3],[0,1],[-1,1],[0,1],[-1,0],[-1,-1],[-2,2],[-1,-1],[-1,0],[-1,1],[-1,-1],[0,-4],[-1,-2],[0,-2],[-1,-5],[0,-1],[-1,-4],[-1,-8],[0,-1],[-1,0],[-1,1],[-3,-6],[0,-1],[-1,0],[-1,-2],[0,-1],[-1,1],[0,1],[0,-1],[-1,-1],[0,-1],[-1,1],[-2,4],[-1,0],[-1,1],[-1,0],[-1,0],[-1,1],[0,-2],[0,-3],[-1,-2],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-3],[-1,-2],[0,-2],[1,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-4],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-2],[0,-2],[-1,-1],[0,-1],[0,-2],[0,-1],[-1,-2],[0,-3],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[-1,-1],[0,-2],[-2,-7],[0,-2],[0,-1],[0,-2],[0,-2],[0,-2],[0,-2],[0,-1],[0,-3],[0,-1],[0,-3],[0,-1],[0,-2],[0,-1],[-1,-4],[0,-5],[0,-1],[-1,-1],[0,-1],[0,2],[-1,0],[-2,-3],[-1,-2],[0,-1],[-1,2],[-1,2],[0,1],[-1,1],[0,-1],[0,-3],[0,-2],[0,-1],[0,-1],[-2,-4],[-1,3],[0,1],[0,1],[-3,4],[-2,0],[-2,-3],[-1,-1],[0,-1],[0,-3],[0,-8],[-1,-1],[0,-1],[0,-1],[0,-2],[-2,-4],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[-3,-2],[0,1],[-1,1],[0,2],[0,2],[0,1],[-1,2],[0,2],[0,3],[0,2],[0,2],[-1,1],[-1,2],[-6,15],[-1,2],[0,-1],[-1,-4],[0,-2],[0,-1],[-1,0],[-1,-1],[0,-1],[0,-3],[-1,0],[-3,5],[-1,-1],[-1,0],[-1,2],[0,2],[-1,2],[0,1],[-3,2],[-1,0],[-1,2],[-1,1],[0,1],[0,1],[0,2],[0,2],[0,1],[0,1],[-1,1],[-3,6],[0,1],[0,1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[-1,1],[1,4],[-1,3],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,2],[0,4],[0,1],[0,1],[1,6],[0,2],[0,1],[0,1],[0,1],[0,2],[0,1],[-2,4],[-1,0],[0,-3],[-2,-3],[0,-2],[-1,-3],[0,-1],[-1,-2],[0,-1],[0,-4],[0,-1],[0,-2],[-1,-1],[0,-1],[-6,4],[-1,1],[0,1],[0,2],[0,2],[-1,3],[0,2],[-1,5],[-1,0],[-5,0],[-1,-1],[0,-2],[0,-1],[-1,-4],[0,-1],[0,-1],[-1,1],[0,1],[0,1],[0,1],[-1,4],[0,2],[0,1],[0,1],[-1,1],[0,1],[0,5],[-1,2],[0,2],[-2,2],[0,1],[0,1],[0,1],[0,1],[0,1],[0,4],[0,2],[0,2],[0,1],[0,3],[0,5],[0,1],[0,1],[-1,1],[-3,0],[-1,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-3],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-4],[0,-1],[0,-2],[0,-1],[1,-3],[0,-4],[0,-1],[0,-1],[0,-2],[0,-1],[-2,-3],[-1,-2],[-1,-1],[0,-1],[0,-1],[0,-5],[-1,-1],[-1,-4],[0,-2],[-1,0],[-4,11],[-1,-1],[0,-1],[-1,-3],[0,-1],[-1,0],[-1,2],[-1,5],[-1,3],[-3,-10],[0,-1],[-2,0],[-1,4],[-1,1],[0,1],[0,2],[0,3],[1,1],[-1,1],[0,1],[-1,-2],[-1,4],[0,1],[-1,3],[0,1],[0,1],[-2,4],[0,1],[-1,0],[0,-1],[-1,-1],[0,-1],[0,-1],[-1,0],[0,1],[-1,1],[0,1],[0,2],[0,1],[-1,1],[0,1],[-1,2],[0,1],[-1,1],[0,1],[-1,-2],[-1,0],[-1,0],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-2],[-1,1],[0,-3],[0,-4],[0,-1],[0,-2],[-1,-2],[0,-3],[0,-3],[0,-3],[0,-2],[0,-1],[0,-4],[0,-1],[0,-1],[1,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[-1,-2],[0,-3],[0,-2],[-1,-9],[-1,-1],[0,-1],[0,-1],[-2,-1],[-1,0],[-1,2],[-1,1],[0,2],[0,1],[-1,2],[0,1],[0,1],[-2,-1],[-2,1],[0,-1],[-1,-2],[-1,-13],[0,-1],[-1,0],[-2,-2],[-1,0],[0,2],[-1,2],[-1,-3],[-1,-1],[0,-2],[0,-7],[0,-1],[-1,0],[0,1],[0,1],[0,1],[0,2],[0,1],[-1,1],[0,-1],[-1,-2],[0,-1],[0,-2],[0,-5],[-1,-2],[0,-2],[-3,1],[-1,0],[-1,-3],[-1,1],[0,1],[0,-1],[-1,-1],[0,-2],[0,-1],[0,-1],[1,-3],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-3],[-1,-8],[-1,-3],[0,-1],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[-1,0],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-8],[0,-2],[0,-1],[1,1],[0,-1],[1,-3],[0,-2],[1,-1],[0,1],[1,-1],[0,-2],[0,-1],[1,0],[0,1],[1,2],[1,-1],[0,-3],[1,-8],[1,-2],[0,-2],[0,-3],[0,-1],[0,-1],[0,-1],[1,-2],[0,-2],[0,-1],[0,-1],[0,-3],[0,-2],[-1,-1],[0,-2],[1,-3],[0,-1],[0,-1],[1,-2],[0,-1],[0,-1],[0,-1],[0,-2],[1,-3],[0,-1],[0,-1],[0,-1],[0,-1],[3,-3],[0,-1],[0,-1],[0,-1],[0,-2],[1,0],[0,-2],[0,-1],[0,-1],[1,-1],[0,1],[1,3],[0,1],[1,0],[0,-1],[0,-2],[0,-2],[0,-1],[0,-2],[0,-2],[0,-2],[-1,-3],[0,-2],[-1,-4],[0,-1],[0,1],[-1,1],[-1,1],[-1,0],[-1,1],[0,2],[0,1],[-1,1],[-1,1],[0,-1],[0,-1],[0,-3],[-1,-4],[0,-4],[-1,-1],[0,-1],[-1,-3],[0,-2],[0,-2],[0,-2],[-1,-14],[0,-2],[0,-3],[0,-5],[0,-5],[0,-1],[0,-2],[0,-2],[-1,-10],[0,-12],[-1,-2],[0,-1],[-1,1],[-1,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-2],[0,-2],[0,-4],[1,-2],[0,-2],[-1,-3],[0,-2],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-3],[0,-1],[-1,-3],[0,-1],[1,-2],[0,-3],[0,-2],[1,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-3],[-1,-17],[0,-2],[0,-1],[0,-1],[1,-1],[1,-1],[0,-1],[0,-3],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[1,-1],[1,-2],[0,-1],[0,-1],[0,-3],[0,-3],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-4],[-1,-3],[0,-6],[0,-4],[0,-2],[0,-1],[-1,0],[0,-2],[-1,-2],[0,-3],[0,-5],[0,-7],[1,-3],[0,-3],[0,-3],[0,-1],[0,-1],[0,-2],[1,-6],[0,-1],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-2],[0,-2],[0,-1],[-1,-2],[1,-1],[0,-2],[0,-2],[0,-2],[0,-5],[0,-3],[0,-2],[0,-3],[0,-2],[0,-2],[0,-25],[0,-2],[-1,-2],[0,-2],[0,-2],[-1,0],[0,-2],[0,-1],[0,-3],[0,-6],[0,-3],[0,-1],[-1,-3],[-1,-7],[3,1],[10,-14],[0,-1],[0,-2],[0,-3],[-1,-10],[-16,-158],[-10,-142],[-1,-3],[-51,-185],[0,-1],[-6,-53],[-6,-47],[-8,-41],[-10,-33],[-6,-15],[-7,-12],[0,-6],[0,-28],[0,-5],[0,-2],[0,-3],[0,-2],[0,-1],[0,-2],[0,-2],[1,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-6],[0,-2],[0,-3],[-1,-5],[-1,-13],[-2,6],[-1,1],[0,1],[0,2],[0,1],[-1,-1],[0,-2],[0,-1],[-1,0],[-6,-3],[0,-1],[-1,-2],[-1,-5],[0,-4],[-1,-2],[0,-1],[0,-1],[-1,-1],[-1,1],[-1,0],[0,-1],[0,-3],[-1,-2],[0,-3],[0,-1],[0,-2],[-3,-8],[-1,-1],[0,63],[0,2],[0,1],[0,2],[-1,2],[-3,11],[0,2],[-3,2],[-1,-3],[-1,-2],[-1,0],[-1,-2],[-1,-1],[0,-1],[0,-1],[0,-1],[-1,0],[-1,1],[0,2],[0,3],[-2,14],[0,2],[0,2],[0,2],[0,2],[0,1],[0,1],[0,2],[0,1],[-1,7],[0,1],[-1,1],[0,2],[0,2],[0,2],[1,1],[0,2],[0,2],[0,1],[0,2],[0,1],[0,3],[0,3],[0,2],[0,5],[0,1],[0,2],[0,2],[0,1],[0,1],[0,3],[0,1],[0,2],[0,2],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,2],[-1,2],[-1,3],[-1,1],[0,1],[0,2],[0,2],[0,1],[0,2],[0,1],[-1,1],[-3,6],[0,1],[-1,1],[-1,6],[0,2],[-1,4],[-1,4],[0,1],[-1,0],[0,1],[0,1],[0,4],[-1,7],[0,3],[-1,3],[0,3],[0,2],[0,3],[-1,1],[0,1],[-1,1],[0,2],[0,2],[0,1],[0,2],[-1,1],[0,1],[-1,2],[0,4],[0,8],[0,7],[-3,3],[-1,5],[0,1],[0,-1],[-2,-6],[0,-1],[-1,0],[-2,6],[-1,4],[0,1],[-1,0],[-1,-1],[-1,-2],[0,-1],[0,-2],[0,-2],[-1,-4],[0,-1],[0,-1],[0,-1],[0,-1],[-1,0],[0,1],[-1,2],[-1,1],[0,2],[0,5],[-1,2],[0,4],[0,1],[0,1],[0,2],[0,1],[0,2],[0,1],[0,1],[-3,10],[-2,11],[0,1],[0,1],[-1,-1],[-1,1],[0,2],[0,1],[1,1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[0,2],[0,1],[-1,2],[-3,17],[0,1],[-1,2],[-1,0],[0,2],[0,2],[0,2],[0,1],[0,3],[0,2],[0,1],[0,1],[0,1],[0,1],[-1,3],[-1,1],[-2,1],[-1,1],[0,2],[0,1],[0,1],[0,1],[-1,0],[0,1],[-1,3],[0,1],[0,1],[0,1],[-3,4],[-1,4],[-1,10],[-1,2],[-3,3],[-1,3],[0,1],[0,2],[-1,2],[0,2],[0,4],[0,2],[0,1],[-3,10],[0,1],[-1,1],[-1,6],[-1,1],[0,2],[-1,4],[-1,12],[0,2],[-1,3],[0,1],[0,2],[-1,4],[0,2],[0,1],[-1,2],[0,1],[0,1],[-1,4],[0,2],[-1,5],[0,2],[0,4],[0,1],[0,4],[1,3],[0,4],[0,1],[0,3],[0,2],[0,2],[-1,4],[-1,5],[0,2],[0,1],[0,1],[0,2],[0,2],[0,1],[-1,1],[0,2],[0,1],[0,6],[0,1],[0,2],[0,1],[-1,1],[0,1],[0,1],[-1,0],[0,-1],[-1,-1],[0,-1],[0,-2],[0,-3],[0,-2],[0,-1],[-1,-1],[0,-1],[0,-1],[-1,-2],[0,-1],[-1,-1],[0,-1],[-1,0],[-1,3],[-1,-1],[-2,-2],[-1,-1],[0,-1],[0,-1],[0,-6],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[1,-4],[0,-2],[0,-1],[0,-1],[-2,-3],[-1,-3],[-2,-5],[0,-4],[0,-1],[0,-1],[-1,0],[0,1],[-1,0],[0,-1],[-3,-7],[-3,3],[-1,-1],[0,-1],[-1,-2],[0,-3],[0,-2],[0,-2],[0,-1],[0,-2],[0,-2],[0,-1],[0,-3],[0,-5],[-1,-2],[0,-1],[-1,-2],[0,-3],[-1,-1],[0,-1],[0,-5]],[[7756,2892],[-7,20],[0,1],[0,1],[0,2],[0,1],[0,2],[0,1],[1,0],[0,2],[1,3],[0,1],[0,1],[0,2],[-3,8],[0,2],[-1,1],[-1,0],[0,-2],[0,-3],[0,-1],[-1,0],[-4,7],[-1,0],[0,-1],[-1,-10],[-1,-1],[-1,1],[-5,-2],[0,-2],[-1,0],[0,1],[-1,1],[0,1],[0,1],[-1,4],[0,1],[0,2],[1,4],[0,1],[0,2],[-1,1],[0,2],[0,2],[0,1],[0,1],[0,1],[1,2],[0,1],[0,1],[0,1],[-1,1],[0,1],[-1,0],[-1,0],[0,1],[0,1],[0,2],[-1,0],[-2,2],[-1,0],[-1,0],[0,4],[0,1],[-1,2],[0,2],[0,1],[0,1],[0,1],[0,2],[0,1],[0,7],[0,1],[0,2],[0,1],[-1,0],[-1,-1],[-1,1],[0,2],[-2,0],[-2,3],[-1,0],[-1,-2],[-1,0],[0,1],[-1,2],[0,2],[0,1],[0,7],[0,1],[0,2],[0,1],[0,1],[0,1],[-1,0],[-2,3],[-1,0],[0,1],[0,2],[0,1],[0,2],[0,1],[0,3],[-1,3],[0,1],[0,1],[0,1],[0,1],[-1,4],[0,1],[0,2],[-1,3],[0,1],[0,2],[0,2],[0,1],[0,1],[0,-1],[-1,0],[-2,2],[0,1],[0,2],[0,1],[-1,1],[0,1],[-1,-2],[-1,-1],[0,1],[-1,3],[-1,5],[0,1],[-1,1],[0,1],[0,1],[-1,1],[0,2],[0,1],[0,1],[-1,1],[0,1],[-1,7],[-1,1],[-2,-1],[-1,-3],[-1,-1],[0,-2],[0,-2],[-1,-9],[0,-1],[0,-1],[0,-2]],[[7685,3060],[-1,0],[0,1],[0,1],[-1,3],[0,1],[0,2],[-1,-3],[0,-1],[0,-1],[-1,1],[0,3],[-1,4],[0,2],[0,3],[-1,1],[0,3],[-1,6],[0,2],[0,2],[-1,4],[0,2],[-4,23],[-1,12],[-1,3],[0,2],[-2,6],[0,1],[-1,5],[-1,1],[0,2],[0,3],[1,3],[0,3],[0,1],[0,1],[0,1],[0,2],[0,1],[0,2],[-1,3],[0,1],[-1,-1],[-1,1],[0,3],[-1,2],[0,1],[0,2],[0,1],[0,2],[0,2],[0,2],[-1,2],[0,1],[-1,0],[0,1],[-1,5],[0,1],[0,1],[-4,9],[0,1],[0,1],[0,1],[-1,3],[0,5],[1,14],[0,1],[0,2],[1,2],[0,2],[0,2],[0,2],[0,1],[0,2],[1,1],[0,1],[1,4],[1,0],[0,-1],[0,-1],[1,-2],[0,-1],[0,-2],[1,0],[1,1],[0,1],[0,3],[0,3],[1,1],[0,3],[0,2],[0,1],[0,1],[0,1],[0,2],[0,2],[1,6],[0,1],[0,1],[0,1],[0,1],[0,2],[1,1],[0,1],[0,1],[0,1],[0,2],[0,2],[0,3],[1,1],[0,-1],[1,-3],[2,-3],[1,-1],[0,1],[2,1],[0,1],[1,2],[0,2],[0,-1],[1,-1],[2,-9],[2,-3],[1,0],[1,0],[0,2],[0,1],[0,1],[0,2],[0,3],[-1,4],[0,1],[0,1],[0,2],[0,1],[0,1],[1,2],[0,2],[0,1],[0,1],[0,5],[0,5],[0,1],[0,1],[0,1],[0,2],[0,1],[0,2],[0,1],[0,1],[0,7],[1,7],[0,1],[0,1],[1,2],[0,2],[0,2],[0,3],[1,2],[0,2],[0,1],[0,2],[0,3],[0,1],[0,1],[0,2],[1,3],[-1,3],[0,2],[0,2],[0,1],[1,2],[0,1],[0,1],[0,1],[0,1],[0,1],[-1,6],[0,2],[-2,10],[-1,5],[0,1],[1,2],[0,1],[1,1],[0,2],[0,1],[1,5],[0,1],[0,2],[0,1],[0,1],[-1,1],[0,1],[0,2],[0,1],[1,1],[0,2],[0,5],[0,2],[0,1],[0,4],[0,5],[0,3],[0,9],[0,2],[1,2],[0,1],[2,5],[3,4],[0,1],[0,1],[0,3],[0,2],[0,2],[0,1],[0,1],[0,1],[0,3],[0,2],[0,2],[0,3],[-1,4],[0,1],[0,1],[0,1],[0,3],[-1,1],[0,1],[0,4],[0,2],[1,3],[0,1],[-1,1],[0,2],[0,1],[0,2],[0,5],[0,1],[0,2],[0,2],[0,1],[0,2],[0,1],[0,1],[0,3],[1,1],[0,2],[0,1],[-1,2],[0,4],[-2,5],[0,3],[0,1],[0,1],[1,5],[0,1],[1,3],[0,1],[0,1],[-1,4],[-1,1],[0,1],[0,2],[0,2],[0,3],[0,3],[0,2],[1,1],[0,1],[1,2],[0,1],[0,6],[1,1],[1,3],[0,2],[2,4],[0,3],[0,4],[0,28],[0,2],[0,1],[1,1],[1,1],[0,2],[1,3],[0,8],[1,6],[1,1],[1,0],[2,-4],[0,-1],[0,-1],[0,-1],[1,-1],[1,-3],[0,-2],[0,-1],[1,2],[1,7],[0,3],[1,3],[0,3],[0,2],[0,4],[1,6],[1,2],[0,1],[1,1],[1,2],[0,3],[0,12],[0,4],[0,2],[-1,3],[0,3],[0,4],[0,9],[0,3],[0,3],[1,1],[7,2],[1,1],[0,3],[0,4],[0,25],[0,4],[0,2],[-1,1],[-2,0],[0,1],[0,1],[0,5],[0,2],[0,2],[0,1],[0,1],[0,2],[0,2],[0,1],[0,2],[0,1],[0,3],[0,2],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,3],[0,2],[0,1],[0,2],[0,2],[0,8],[0,2],[0,2],[0,1],[0,1],[-2,4],[-2,0],[0,-1],[-2,-2],[-1,3],[-1,1],[0,1],[0,1],[0,2],[0,11],[0,2],[0,3],[0,1],[0,1],[2,4],[1,2],[1,3],[0,1],[0,2],[0,1],[0,1],[0,1],[-4,1],[0,1],[0,1],[0,1],[-1,7],[0,1],[-1,1],[0,1],[0,1],[0,1],[-1,3],[-1,1],[0,2],[1,3],[1,17],[15,167],[1,3],[16,-8],[2,-3],[1,1],[2,3],[1,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-1],[0,-1],[1,-1],[1,-6],[1,-1],[0,-1],[2,1],[1,2],[0,2],[2,7],[1,1],[1,-1],[1,-1],[0,-1],[1,1],[1,4],[2,3],[0,1],[3,-1],[0,1],[0,1],[0,3],[0,2],[0,1],[1,4],[0,3],[0,1],[1,8],[0,3],[0,1],[0,4],[0,1],[0,2],[1,1],[0,1],[0,1],[0,2],[1,2],[1,9],[1,1],[0,3],[0,2],[0,1],[0,2],[0,2],[2,10],[1,-1],[1,-5],[1,0],[1,2],[4,3],[1,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-11],[0,-6],[0,-3],[0,-1],[0,-1],[0,-1],[0,-2],[1,-5],[0,-2],[0,-1],[0,-3],[0,-4],[0,-2],[1,-2],[0,-4],[2,-5],[0,-2],[1,-3],[1,-1],[0,-2],[0,-1],[0,-5],[-1,-1],[1,-2],[0,-1],[1,0],[1,-5],[0,-1],[0,-2],[0,-4],[-1,-2],[0,-2],[0,-1],[0,-1],[1,-2],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[1,-10],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-3],[0,-2],[0,-2],[0,-1],[1,-1],[6,-16],[0,-2],[0,-1],[1,-2],[0,-5],[1,-4],[0,-2],[0,-1],[0,-1],[1,0],[0,2],[1,2],[0,1],[1,2],[0,4],[0,1],[0,1],[1,1],[0,1],[1,-3],[1,1],[0,1],[0,1],[0,5],[1,1],[0,2],[-1,1],[0,1],[0,1],[0,2],[0,1],[0,3],[0,2],[1,2],[0,1],[-1,2],[0,1],[0,1],[-2,6],[0,4],[-1,3],[0,1],[0,1],[0,2],[0,3],[0,1],[0,2],[-1,1],[0,1],[0,1],[1,4],[0,10],[1,2],[0,1],[0,1],[0,1],[0,2],[1,2],[-1,1],[0,1],[0,1],[0,2],[0,1],[-1,2],[0,5],[1,3],[-1,3],[0,1],[0,2],[0,3],[0,2],[0,1],[0,1],[1,0],[0,-1],[0,-1],[1,-1],[0,-1],[0,-1],[1,0],[1,0],[0,1],[1,2],[0,3],[1,1],[0,2],[0,1],[0,3],[0,1],[0,2],[1,2],[0,1],[1,4],[0,4],[1,2],[0,3],[0,5],[0,5],[0,3],[0,1],[0,1],[0,1],[-1,3],[0,1],[0,1],[0,2],[0,2],[0,3],[0,1],[0,1],[1,2],[1,2],[1,1],[0,3],[0,3],[1,9],[1,3],[0,2],[2,0],[3,-1],[1,0],[0,1],[1,1],[0,1],[0,2],[0,3],[0,2],[0,4],[0,2],[0,1],[1,7],[2,9],[0,2],[0,6],[1,6],[0,2],[0,2],[1,1],[0,2],[1,3],[0,1],[1,0],[3,3],[1,1],[0,1],[1,3],[0,2],[1,0],[4,5],[1,2],[0,1],[1,7],[1,5],[0,1],[0,2],[1,5],[0,3],[0,3],[0,3],[0,1],[1,1],[2,2],[0,1],[1,2],[0,2],[0,3],[0,1],[2,4],[0,2],[0,2],[0,5],[1,9],[0,1],[0,1],[0,2],[0,1],[0,1],[0,6],[0,2],[-1,3],[0,3],[0,1],[0,5],[0,1],[0,1],[0,2],[-1,3],[0,1],[-1,3],[0,1],[0,1],[0,1],[0,3],[0,1],[0,2],[0,1],[0,5],[0,1],[0,1],[0,2],[0,1],[0,1],[0,2],[-1,4],[0,1],[0,1],[0,2],[0,1],[0,1],[0,2],[0,4],[0,5],[1,5],[0,2],[0,2],[1,2],[1,3],[0,3],[2,4],[1,1],[1,-1],[3,0],[1,1],[1,2],[0,1],[0,2],[0,1],[0,1],[0,2],[0,2],[1,3],[0,1],[2,7],[1,5],[1,1],[0,-1],[0,-3],[0,-2],[1,-5],[0,-1],[1,-2],[2,0],[0,1],[1,2],[0,2],[1,0],[2,-6],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,0],[-1,1],[-1,-1],[0,-1],[0,-1],[0,-5],[0,-1],[0,-3],[0,-1],[0,-1],[0,-1],[1,-4],[0,-1],[0,-3],[1,-12],[1,-5],[1,-9],[0,-3],[1,-2],[1,-1],[1,-1],[0,-1],[0,-1],[1,-2],[0,-1],[1,-2],[0,-1],[1,-3],[0,-1],[0,-2],[0,-1],[0,-3],[0,-2],[0,-2],[0,-1],[1,2],[1,1],[1,0],[2,-7],[1,0],[1,0],[1,1],[0,1],[0,-1],[0,-1],[0,-1],[1,-1],[0,-4],[0,-1],[0,-2],[0,-1],[1,-2],[0,-1],[0,-2],[0,-3],[0,-1],[0,-2],[0,-2],[0,-2],[0,-2],[0,-3],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[-1,-2],[-1,-1],[0,-2],[1,0],[0,-1],[1,0],[1,1],[1,0],[1,2],[0,1],[0,2],[0,3],[0,1],[4,-1],[1,-2],[0,-2],[1,-1],[0,-2],[0,-1],[1,-1],[0,-2],[0,-2],[2,1],[0,-1],[1,-2],[0,-2],[0,-8],[0,-5],[1,-5],[1,-4],[0,-1],[0,1],[1,5],[0,1],[1,1],[0,-1],[1,-3],[0,-1],[1,1],[0,3],[0,2],[0,1],[1,1],[4,4],[0,1],[1,4],[1,3],[0,3],[0,2],[0,4],[0,3],[-1,2],[0,1],[0,1],[0,3],[0,2],[0,1],[-1,2],[-1,3],[0,2],[0,1],[0,1],[0,3],[1,7],[1,7],[1,1],[0,1],[1,1],[0,1],[0,1],[1,4],[0,5],[0,2],[1,2],[0,1],[2,2],[1,1],[0,3],[0,1],[2,10],[0,1],[0,1],[1,1],[1,2],[2,1],[0,1],[1,2],[1,4],[0,2],[0,2],[-1,5],[-1,4],[0,1],[0,2],[0,2],[1,5],[0,3],[0,1],[0,1],[-1,3],[0,2],[0,1],[0,1],[0,1],[0,4],[0,1],[0,1],[0,1],[-1,1],[0,1],[0,1],[-2,6],[0,1],[0,2],[0,2],[0,1],[1,1],[0,2],[0,1],[1,2],[0,3],[0,1],[-1,3],[0,1],[0,3],[0,2],[0,2],[0,2],[0,1],[0,1],[-1,6],[0,3],[0,1],[0,1],[-1,0],[0,-1],[-1,-3],[0,-1],[0,1],[-1,1],[0,1],[0,-1],[-1,-1],[-2,1],[0,1],[0,4],[0,2],[-2,9],[-1,3],[0,1],[0,2],[0,1],[0,2],[0,1],[0,2],[0,2],[0,1],[0,2],[-1,8],[-1,2],[0,1],[0,2],[1,4],[0,1],[-1,3],[0,2],[0,2],[0,1],[1,3],[1,2],[1,6],[0,1],[2,4],[1,-1],[0,1],[0,1],[0,1],[1,4],[0,2],[0,1],[0,2],[0,1],[-1,1],[1,1],[0,1],[-1,1],[1,1],[1,0],[0,1],[1,2],[0,1],[2,2],[0,-1],[1,-1],[0,-1],[1,0],[1,1],[0,-1],[0,-1],[0,-1],[1,-1],[0,-2],[1,-1],[2,0],[1,2],[0,1],[0,1],[1,1],[0,2],[0,1],[-1,1],[0,1],[0,2],[-1,2],[0,1],[0,1],[0,2],[0,2],[0,6],[0,5],[0,2],[0,2],[1,4],[0,2],[0,2],[0,1],[0,2],[0,1],[0,2],[0,3],[-1,10],[0,2],[0,3],[0,3],[0,4],[0,2],[0,1],[0,4],[1,1],[0,3],[0,4],[0,3],[0,3],[0,4],[0,2],[0,2],[0,2],[0,1],[0,1],[0,1],[0,1],[-1,0],[0,1],[0,-1],[-1,0],[0,-3],[0,-1],[-1,0],[0,1],[0,1],[0,4],[0,1],[0,2],[-1,1],[0,1],[-1,1],[-1,-1],[0,-1],[-1,-3],[0,-1],[-2,1],[-1,0],[-1,3],[-1,0],[-1,-2],[-3,5],[0,1],[-1,1],[0,3],[0,1],[0,2],[-1,3],[-1,4],[0,3],[-1,2],[0,1],[-2,3],[-1,2],[0,1],[1,1],[0,1],[0,1],[0,1],[0,1],[-1,2],[0,3],[0,7],[0,3],[0,2],[0,1],[-2,5],[0,2],[0,1],[0,2],[0,2],[0,2],[0,3],[0,2],[0,4],[0,1],[0,3],[-1,1],[0,1],[0,1],[0,1],[0,1],[0,2],[1,15],[0,2],[0,1],[1,4],[0,2],[0,1],[0,1],[0,2],[-1,1],[0,1],[-1,2],[-1,1],[0,1],[0,2],[-1,4],[0,3],[0,3],[0,2],[0,1],[0,1],[1,2],[0,1],[2,2],[1,1],[0,2],[0,1],[1,1],[0,2],[0,2],[0,1],[-1,1],[0,2],[-1,3],[0,1],[0,2],[0,1],[0,1],[1,3],[0,1],[0,1],[0,2],[0,1],[0,2],[0,1],[0,1],[0,1],[0,2],[1,3],[0,1],[0,1],[0,1],[-1,2],[0,2],[0,2],[0,1],[0,1],[1,2],[0,1],[1,-1],[0,1],[1,1],[0,1],[0,2],[0,2],[1,2],[0,1],[0,2],[1,1],[0,1],[0,3],[0,1],[1,1],[0,1],[0,2],[0,1],[0,1],[5,-1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,3],[0,2],[-1,3],[0,5],[0,1],[0,1],[0,3],[-1,2],[0,7],[0,2],[0,1],[0,1],[-1,0],[0,1],[-2,1],[0,1],[-1,1],[0,2],[2,2],[1,1],[0,2],[1,4],[0,2],[1,1],[4,5],[2,-1],[2,1],[0,2],[1,1],[0,1],[0,2],[0,2],[-1,1],[0,2],[0,1],[1,2],[2,7],[0,2],[0,3],[0,2],[2,2],[1,4],[2,2],[0,1],[1,2],[0,1],[0,2],[0,2],[1,4],[1,6],[1,2],[1,1],[1,-2],[0,-1],[0,-3],[0,-1],[1,0],[1,1],[0,1],[0,6],[0,2],[1,1],[0,3],[1,1],[1,1],[0,1],[0,1],[0,2],[0,3],[0,7],[0,1],[0,1],[0,2],[-1,2],[0,6],[0,1],[0,1],[-1,1],[0,2],[1,1],[0,2],[0,1],[0,2],[4,15],[0,1],[0,2],[0,3],[0,2],[1,2],[2,3],[0,2],[0,1],[1,7],[0,2],[1,1],[5,7],[0,1],[0,1],[0,3],[-1,1],[0,2],[1,2],[0,1],[0,2],[0,1],[-1,1],[0,1],[0,2],[0,1],[0,2],[1,4],[0,2],[0,2],[0,2],[0,1],[0,2],[1,2],[0,2],[0,1],[0,2],[0,1],[0,2],[0,5],[0,1],[0,1],[-1,0],[-2,3],[0,1],[0,1],[0,1],[0,2],[1,5],[0,2],[1,3],[0,1],[1,0],[1,1],[0,1],[0,1],[0,1],[0,1],[-1,2],[0,1],[0,1],[-1,1],[0,2],[0,1],[0,2],[1,1],[0,1],[0,1],[1,0],[0,-1],[1,-2],[0,-1],[0,1],[1,1],[2,13],[2,10],[0,2],[0,1],[0,2],[0,1],[0,2],[0,2],[-1,3],[0,2],[0,3],[1,6],[0,2],[1,2],[2,2],[0,1],[-6,12],[0,1],[0,1],[-1,2],[0,1],[0,1],[-1,0],[-1,2],[0,2],[1,7],[0,5],[0,2],[0,1],[0,1],[0,1],[0,1],[-1,1],[0,1],[0,1],[0,1],[-1,3],[0,10],[0,1],[0,1],[0,1],[-2,4],[-1,1],[0,3],[-1,1],[-2,3],[0,1],[0,1],[0,1],[0,1],[0,4],[0,1],[0,1],[0,2],[-1,0],[0,3],[-1,0],[0,2],[0,1],[1,3],[0,1],[0,2],[-1,0],[0,2],[0,1],[0,3],[-1,8],[0,1],[0,2],[0,1],[0,1],[5,1],[2,4],[1,4],[0,2],[0,1],[0,1],[0,2],[-1,2],[1,2],[0,4],[0,6],[0,2],[0,2],[-1,3],[0,1],[0,2],[0,4],[0,5],[0,2],[0,1],[1,2],[1,9],[0,1],[1,0],[0,-1],[0,-1],[0,-1],[1,-1],[0,-1],[4,-2],[1,1],[0,1],[0,1],[0,6],[0,2],[0,2],[1,2],[0,2],[0,1],[0,1],[-2,6],[-1,2],[0,4],[0,1],[-1,7],[0,1],[0,1],[-2,5],[0,1],[0,2],[0,2],[0,2],[0,2],[0,1],[0,1],[1,1],[1,1],[0,2],[0,3],[1,2],[1,10],[1,4],[0,2],[0,1],[0,2],[-1,2],[-1,4],[0,1],[0,1],[0,2],[0,1],[0,2],[0,2],[0,1],[1,4],[0,2],[1,8],[0,2],[2,11],[1,4],[0,2],[0,2],[0,1],[-1,2],[-2,11],[0,2],[0,3],[0,1],[0,1],[0,2],[0,4],[0,5],[0,2],[1,1],[0,1],[1,0],[0,-1],[1,-1],[0,-2],[0,-1],[0,-1],[1,1],[1,0],[0,-1],[0,-2],[2,-3],[1,-3],[0,-2],[0,-1],[0,-1],[1,-2],[0,-3],[1,-3],[0,-1],[1,0],[1,3],[0,1],[0,2],[1,7],[0,2],[0,1],[1,1],[2,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-4],[0,-1],[0,-1],[1,-6],[0,-1],[0,-1],[0,-1],[1,-1],[0,-1],[1,-1],[1,2],[1,0],[1,-3],[1,0],[0,1],[1,2],[0,2],[1,7],[0,2],[0,1],[0,1],[0,3],[0,1],[0,2],[0,3],[0,1],[0,2],[1,3],[0,1],[2,2],[1,2],[0,1],[1,-1],[0,-1],[0,-2],[1,-1],[0,-1],[1,1],[0,1],[1,1],[1,1],[1,-3],[1,0],[1,0],[1,-1],[1,0],[0,1],[1,3],[1,0],[1,-2],[1,1],[0,4],[1,3],[0,2],[0,1],[0,2],[0,1],[0,3],[0,1],[0,2],[-1,1],[0,1],[0,1],[0,6],[-1,3],[0,2],[0,1],[0,1],[0,3],[0,1],[0,2],[0,4],[0,1],[-1,5],[0,1],[0,1],[0,2],[-1,2],[0,2],[-1,1],[-3,-4]],[[8000,5663],[0,2],[0,1],[0,1],[2,2],[1,3],[1,-2],[1,0],[2,-1],[1,0],[0,1],[0,1],[0,1],[0,2],[1,1],[0,2],[0,2],[0,2],[0,1],[0,-1],[1,0],[0,-1],[0,-1],[1,0],[1,0],[0,-1],[0,-1],[1,-6],[0,-2],[1,-2],[0,-1],[0,-2],[0,-2],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[-1,-8],[0,-2],[0,-1],[1,-3],[0,-1],[0,-1],[1,-7],[0,-4],[1,-2],[0,-1],[3,-8],[0,-1],[0,-1],[0,-2],[1,-1],[0,-2],[0,-2],[0,-2],[-1,-2],[-1,-8],[0,-1],[0,-2],[1,-1],[1,-5],[0,-1],[0,-1],[1,-2],[-1,-1],[0,-2],[0,-2],[0,-2],[-1,-2],[0,-1],[0,-3],[0,-2],[0,-1],[0,-1],[-1,0],[-1,-3],[0,1],[-1,1],[0,2],[0,1],[-4,-4],[0,1],[0,2],[-1,0],[0,-1],[0,1],[-1,3],[0,1],[-1,1],[0,1],[-2,-4],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-3],[0,-1],[0,-1],[0,-2],[0,-7],[0,-1],[0,-7],[0,-3],[0,-1],[0,-2],[-1,-1],[0,-1],[-1,-4],[-1,-3],[0,-1],[-1,0],[0,-1],[-1,-1],[0,-1],[0,-2],[0,-3],[1,-4],[0,-3],[0,-1],[1,-1],[10,9],[1,0],[1,-1],[0,-4],[1,0],[0,-1],[2,1],[0,-1],[1,-2],[0,-2],[2,-1],[1,-3],[0,-1],[1,0],[1,3],[1,1],[1,-1],[1,1],[0,-1],[1,-3],[0,-3],[1,-2],[0,-1],[1,-1],[0,-1],[0,-3],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[1,-2],[1,-3],[0,-1],[0,-3],[0,-2],[0,-2],[1,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-3],[1,-2],[0,-1],[1,0],[0,4],[1,-1],[0,-1],[0,-4],[1,-2],[0,-2],[0,-1],[-1,-2],[0,-1],[1,-5],[0,-2],[0,-4],[0,-1],[0,-1],[1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[1,-1],[0,-2],[-1,-1],[0,-1],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-2],[1,-2],[0,-2],[0,-1],[1,-1],[-1,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[1,0],[0,-1],[0,-2],[0,-3],[0,-2],[0,-3],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[0,-4],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-1],[1,-1],[1,-4],[2,-4],[0,-1],[0,-1],[0,-2],[0,-3],[1,-4],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[-1,-1],[-1,-2],[-1,-1],[0,-2],[0,-1],[-1,0],[0,-1],[0,-2],[-1,-3],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-7],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[-1,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[-1,-2],[0,-6],[1,-6],[-1,-2],[0,-1],[-3,-1],[0,-1],[0,-1],[0,-3],[0,-2],[0,-2],[0,-2],[0,-1],[1,-1],[0,-3],[0,-2],[0,-2],[0,-2],[1,-2],[0,-1],[2,-2],[1,-1],[0,-1],[0,-2],[1,-8],[1,-1],[1,4],[0,1],[3,-3],[3,-2],[2,2],[1,0],[1,-1],[0,-1],[0,-1],[1,-1],[-1,-2],[0,-9],[0,-1],[0,-1],[1,-2],[0,-1],[1,-2],[0,-3],[0,-2],[0,-2],[1,-2],[0,-1],[0,-1],[0,-2],[-1,-1],[0,-1],[-1,-6],[0,-1],[0,-2],[0,-2],[1,-2],[1,-6],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[-1,-2],[-2,-8],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[-1,-5],[0,-3],[0,-1],[0,-2],[-1,-7],[0,-2],[-1,0],[0,-1],[-1,-4],[0,-1],[0,-1],[-3,-17],[0,-1],[0,-2],[-1,-1],[0,-2],[0,-4],[-1,-5],[0,-8],[1,-6],[0,-9],[0,-4],[0,-3],[0,-2],[0,-1],[1,-2],[1,-5],[0,-1],[1,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[1,-2],[3,-19],[0,-2],[0,-4],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-3],[1,-1],[0,-2],[0,-1],[-1,-1],[0,-3],[0,-2],[0,-1],[-1,-5],[0,-2],[0,-1],[0,-1],[1,-3],[0,-1],[-1,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-1],[2,-4],[1,0],[0,1],[1,1],[0,-1],[1,-2],[3,-4],[0,-1],[0,-2],[1,-4],[0,-2],[0,-2],[0,-4],[0,-2],[0,-2],[1,-2],[0,-4],[0,-1],[0,-1],[1,-2],[0,-1],[1,-1],[1,0],[0,1],[1,0],[1,-1],[1,5],[0,1],[0,1],[1,0],[0,1],[0,-1],[2,-5],[1,-1],[0,-1],[0,-2],[0,-1],[0,-6],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-3],[-1,-1],[0,-3],[0,-2],[0,-2],[0,-3],[0,-3],[-1,-5],[0,-2],[0,-3],[0,-1],[-1,-1],[0,-1],[0,-1],[-1,-4],[0,-1],[-1,-5],[0,-2],[-1,-1],[0,-2],[0,-3],[0,-1],[0,-2],[-1,-3],[0,-3],[-1,-2],[0,-2],[0,-2],[0,-2],[0,-3],[0,-1],[1,-2],[1,-3],[0,-1],[0,-1],[-1,-1],[0,-7],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[-1,-1],[0,-1],[-1,0],[0,1],[0,1],[-1,-1],[0,-1],[-2,0],[-1,0],[0,-1],[0,-2],[0,-1],[1,-1],[0,-1],[0,-1],[1,-3],[0,-1],[0,-2],[0,-2],[0,-4],[0,-2],[0,-2],[0,-1],[-2,-3],[0,-1],[0,-1],[-1,-7],[0,-3],[0,-1],[-1,-2],[0,-1],[-1,-4],[0,-5],[-1,-1],[0,-2],[1,-2],[0,-1],[0,-3],[0,-2],[0,-1],[0,-3],[0,-1],[0,-2],[-1,-2],[-1,-2],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[-1,-2],[0,-1],[-1,-1],[1,-3],[0,-1],[0,-2],[0,-1],[0,-2],[0,-3],[-1,-2],[0,-1],[0,-1],[0,-1],[-1,-2],[-1,-2],[0,-4],[0,-8],[0,-6],[-1,-2],[0,-1],[0,-3],[0,-2],[0,-1],[0,-3],[1,-3],[0,-2],[0,-2],[0,-1],[0,-4],[0,-1],[0,-1],[1,-3],[0,-1],[0,-1],[0,-2],[0,-4],[-1,-3],[0,-3],[0,-1],[-1,-1],[0,-2],[0,-6],[0,-1],[0,-2],[-1,0],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[1,-2],[0,-1],[0,-1],[0,-1],[0,-3],[0,-1],[-1,-1],[0,-3],[-2,-8],[0,-5],[-1,-6],[0,-1],[0,-3],[0,-3],[-1,-3],[0,-3],[0,-2],[0,-2],[1,-3],[0,-2],[0,-1],[0,-1],[1,-1],[0,-2],[0,-2],[0,-2],[-1,-2],[0,-3],[-1,-3],[-1,-5],[0,-1],[-1,-8],[0,-2],[0,-2],[-1,-2],[-1,-1],[0,-2],[0,-2],[0,-2],[0,-2],[0,-1],[-1,-2],[0,-2],[-1,-1],[0,-2],[0,-1],[1,-2],[0,-1],[1,-2],[1,-1],[0,-2],[0,-2],[0,-1],[-1,-1],[0,-2],[-2,-6],[0,-2],[0,-2],[0,-3],[0,-2],[0,-1],[0,-2],[0,-1],[1,-1],[1,-2],[1,-1],[2,-2],[1,1],[1,0],[0,-1],[0,-2],[0,-3],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-3],[1,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-2],[0,-1],[0,-2],[1,-1],[1,-1],[0,-1],[0,-2],[0,-1],[1,-3],[0,-3],[-1,-9],[0,-4],[1,-1],[0,-1],[0,-1],[0,-1],[1,0],[0,3],[0,1],[1,0],[0,-1],[1,-1],[0,-1],[0,-2],[0,-3],[1,-6],[0,-5],[0,-1],[1,-2],[0,-1],[0,1],[1,3],[1,2],[3,-1],[0,-1],[1,-1],[0,-1],[2,-2],[2,-1],[1,1],[1,1],[1,0],[0,3],[0,3],[1,1],[0,3],[0,1],[1,1],[3,0],[0,1],[2,5],[0,1],[0,1],[0,2],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[0,1],[1,3],[0,1],[0,1],[0,1],[0,1],[0,5],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[0,1],[1,2],[0,1],[0,1],[1,3],[0,2],[0,1],[0,1],[1,3],[0,1],[1,0],[3,-1],[1,-4],[1,0],[1,-1],[0,-1],[1,-2],[0,-2],[0,-1],[1,0],[0,1],[0,1],[1,4],[0,1],[1,0],[0,-1],[1,-1],[0,-5],[0,-1],[1,-4],[1,0],[0,2],[1,1],[0,2],[0,1],[0,2],[0,1],[1,2],[0,1],[0,1],[1,2],[0,1],[0,1],[3,2],[2,0],[0,-2],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,-4],[1,-1],[1,-3],[0,1],[1,0],[1,4],[1,8],[0,3],[1,1],[3,1],[1,-2],[1,1],[0,1],[0,2],[0,1],[1,9],[0,2],[0,1],[0,1],[3,12],[0,1],[0,1],[0,3],[1,3],[2,13],[0,6],[1,0],[1,0],[0,-1],[0,-1],[0,-3],[0,-1],[0,-2],[0,-1],[0,-4],[1,-5],[0,-1],[0,-1],[1,-1],[0,-1],[-1,-2],[0,-2],[-1,-4],[0,-3],[0,-1],[0,-1],[0,-2],[1,-2],[0,-1],[0,-1],[0,-1],[0,-2],[-1,-4],[0,-1],[0,-1],[1,0],[1,0],[2,7],[4,-3],[1,0],[1,0],[0,-1],[0,-5],[0,-3],[0,-1],[0,-1],[0,-1],[-1,0],[0,-1],[0,-2],[-1,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-2],[-1,-4],[-1,-10],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[0,-1],[0,-7],[0,-2],[0,-1],[0,-3],[0,-1],[0,-1],[0,-6],[0,-1],[0,-1],[0,-2],[0,-1],[0,-3],[0,-1],[0,-3],[0,-8],[0,-4],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[1,-3],[0,-1],[0,-1],[1,1],[1,9],[1,10],[0,2],[0,2],[0,5],[-1,8],[0,1],[0,2],[0,3],[0,2],[0,2],[0,1],[1,-2],[0,-4],[1,-2],[1,2],[1,4],[0,3],[0,1],[1,2],[1,4],[2,4],[1,0],[0,2],[1,6],[0,1],[0,-1],[1,-3],[1,0],[0,1],[0,1],[1,1],[1,0],[1,0],[1,3],[0,1],[2,-3],[1,1],[1,4],[1,5],[1,4],[1,4],[1,6],[1,3],[0,1],[0,2],[0,2],[0,1],[0,2],[0,1],[0,8],[0,2],[0,1],[-1,5],[0,1],[0,1],[0,1],[0,2],[0,3],[0,1],[0,3],[1,3],[1,2],[0,1],[1,1],[1,11],[1,3],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[-1,2],[0,1],[0,1],[1,1],[1,4],[4,7],[1,7],[0,2],[0,1],[0,1],[0,2],[1,1],[0,2],[0,1],[0,1],[0,1],[0,-1],[1,0],[1,1],[0,1],[3,-2],[0,1],[0,3],[0,8],[0,2],[0,1],[3,1],[1,2],[0,1],[0,4],[1,5],[0,1],[0,2],[1,2],[0,2],[0,1],[0,1],[0,2],[1,1],[0,3],[0,3],[0,1],[0,2],[7,26],[1,3],[0,1],[0,2],[0,1],[0,1],[-1,1],[0,3],[-1,1],[0,2],[0,1],[1,12],[0,2],[0,1],[0,1],[1,3],[1,1],[0,2],[1,5],[1,12],[0,2],[0,1],[1,0],[1,-1],[0,-1],[0,-1],[1,-5],[0,-1],[1,0],[0,1],[2,10],[3,9],[0,2],[0,3],[1,2],[0,4],[0,6],[1,5],[0,3],[0,3],[0,1],[1,2],[3,13],[5,10],[8,10],[0,-1],[1,0],[0,-1],[2,-11],[1,-2],[4,-2],[1,-1],[0,-3],[1,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[4,-2],[1,-4],[0,-1],[1,0],[1,3],[1,0],[7,-7],[0,-1],[1,-3],[0,-3],[2,-16],[1,-1],[0,-1],[2,-3],[1,-1],[0,-1],[0,-6],[1,-2],[0,-3],[2,-6],[1,-4],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-2],[1,-3],[0,-3],[0,-2],[0,-1],[0,-2],[0,-2],[0,-2],[0,-2],[0,-1],[1,-2],[0,-1],[1,-1],[1,1],[0,1],[1,0],[1,-3],[2,0],[0,-1],[1,-1],[0,-2],[0,-2],[0,-2],[-1,-1],[-1,-7],[-1,-1],[-1,2],[-1,1],[-1,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-2],[0,-2],[0,-3],[0,-1],[0,-1],[2,-12],[0,-1],[1,-1],[1,-1],[0,-1],[0,-1],[0,-2],[0,-3],[1,-2],[0,-2],[0,-1],[-1,-2],[0,-1],[0,-3],[0,-3],[0,-3],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-5],[0,-2],[0,-1],[0,-2],[0,-3],[0,-1],[0,-1],[0,-2],[-1,-1],[-1,-8],[-1,-3],[0,-1],[0,-1],[0,-1],[1,-1],[0,-1],[2,-3],[0,-1],[0,-1],[0,-5],[1,-1],[0,-1],[0,-1],[1,-1],[2,-6],[3,0],[1,-1],[1,-4],[1,0],[1,-1],[1,7],[2,7],[1,2],[1,-1],[0,-2],[0,-3],[0,-3],[0,-1],[0,-3],[0,-1],[1,-3],[0,-2],[0,-2],[0,-1],[1,-1],[0,-1],[1,0],[0,1],[1,1],[0,1],[0,1],[0,2],[1,8],[0,3],[0,1],[0,2],[0,2],[1,1],[0,1],[0,2],[1,2],[0,5],[1,2],[0,1],[1,0],[0,1],[1,4],[2,-1],[2,1],[0,-1],[1,-2],[1,-2],[0,-3],[0,-10],[1,-3],[0,-2],[0,-1],[4,-6],[0,-1],[1,-7],[2,-6],[2,-2],[1,-1],[0,-1],[0,-1],[0,-2],[0,-4],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[1,-1],[1,-3],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[-1,-2],[-1,-1],[0,-2],[0,-1],[-1,-2],[0,-5],[0,-2],[0,-1],[0,-9],[0,-5],[1,-15],[0,-9],[1,-1],[1,0],[1,-2],[1,-1],[0,-2],[0,-13],[-1,-2],[0,-1],[-1,0],[0,-1],[0,-3],[0,-5],[0,-2],[0,-2],[0,-2],[1,-2],[1,-6],[0,-2],[1,-1],[0,-2],[0,-2],[0,-3],[0,-1],[-1,-8],[0,-3],[0,-6],[0,-8],[0,-2],[0,-1],[0,-4],[-1,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[-1,0],[-1,-1],[1,-2],[0,-4],[0,-2],[1,-1],[1,-1],[0,-1],[0,-2],[1,-1],[0,-1],[0,-1],[1,0],[0,1],[1,-1],[0,-1],[1,-3],[0,-5],[0,-1],[1,-1],[2,-3],[0,-1],[0,-1],[0,-2],[0,-3],[0,-4],[0,-7],[0,-3],[0,-2],[0,-2],[-1,-4],[0,-1],[0,-3],[0,-3],[0,-4],[0,-2],[1,-5],[0,-2],[0,-2],[0,-15]],[[7310,7176],[-3,-4],[0,1],[1,7],[1,10],[-1,11],[0,11],[1,10],[-1,11],[1,10],[2,9],[0,10],[0,10],[1,8],[1,1],[2,-4],[3,-12],[-1,-12],[-2,-7],[-2,-4],[0,-10],[-1,-14],[0,-9],[1,-8],[0,-9],[-1,-6],[-2,-10]],[[7209,7803],[0,-5],[0,1],[-1,1],[-1,-2],[1,-2],[0,-4],[0,-4],[0,-5],[0,-1],[1,-2],[0,-2],[0,-3],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[1,-7],[-1,-3],[-4,-6],[-11,10],[-3,-1],[-1,2],[-2,7],[-2,3],[-1,4],[-1,6],[0,4],[0,1],[-1,-1],[0,-1],[0,-3],[0,-4],[0,-2],[0,3],[-1,6],[0,5],[1,4],[1,3],[0,1],[7,25],[7,24],[1,2],[0,-1],[1,-3],[3,-10],[1,-3],[0,-2],[1,-2],[0,-1],[0,-2],[0,-2],[0,-2],[1,-6],[3,-9]],[[7409,7969],[-10,-2],[2,6],[1,1],[0,5],[2,3],[3,2],[5,9],[1,1],[0,-2],[0,-1],[-2,-8],[-1,-4],[0,-1],[0,-3],[0,-3],[0,-2],[-1,-1]],[[7342,8071],[0,-1],[0,-4],[2,-7],[-1,-3],[-1,0],[-3,3],[-1,4],[-8,2],[0,1],[-1,2],[0,3],[1,1],[2,1],[0,2],[1,0],[4,-4],[2,2],[4,5],[1,1],[0,-2],[0,-4],[0,-1],[-2,-1]],[[8091,8035],[-1,-3],[-7,-12],[-2,-6],[-2,-4]],[[8079,8010],[-4,2],[-1,2],[-1,0],[-10,-20],[-14,0],[-1,-5],[0,-1],[1,-1],[0,-2],[-2,-1],[-1,4],[-2,7],[-2,12],[0,2],[0,2],[0,6],[0,6],[0,3],[5,5],[3,6],[0,3],[1,6],[0,6],[1,5],[1,6],[1,4],[1,3],[8,1],[8,-6],[4,-9],[2,0],[0,-2],[3,-8],[1,-2],[2,1],[1,4],[1,5],[0,4],[0,2],[-1,2],[1,4],[0,5],[1,3],[1,2],[10,3],[1,-2],[1,-4],[-1,-4],[-1,-1],[-2,0],[-2,-4],[-1,-1],[-2,4],[-1,-2],[-1,-4],[3,-26]],[[7322,8089],[0,-2],[-1,-2],[-2,-2],[1,4],[-1,1],[-3,-3],[0,-3],[1,-3],[1,-1],[-1,-2],[-1,-1],[-9,13],[-7,-2],[0,2],[0,2],[1,6],[2,3],[3,1],[4,5],[9,-8],[1,-1],[1,-1],[0,-3],[1,-3]],[[7296,8083],[-2,-3],[0,1],[-2,4],[-5,2],[-1,1],[0,4],[0,3],[0,3],[0,2],[1,1],[1,3],[2,2],[3,1],[5,-15],[-1,-4],[-1,-5]],[[7359,8185],[-1,-2],[-2,-1],[-5,2],[-2,-2],[-1,-3],[6,-12],[-1,-3],[-8,7],[-1,4],[-1,4],[1,3],[5,7],[8,2],[1,-1],[1,-2],[0,-3]],[[7374,8189],[0,-8],[-5,0],[-2,3],[-1,3],[-1,6],[-1,6],[1,6],[2,3],[8,0],[1,-1],[1,-5],[1,-1],[0,-9],[-4,-3]],[[7388,8209],[2,-6],[3,-2],[1,-3],[0,-5],[-2,-5],[-1,-1],[0,1],[-1,4],[-1,1],[-1,1],[-1,-1],[0,-3],[1,-1],[0,-1],[0,-4],[0,-2],[-2,-2],[-2,-4],[-1,-1],[-3,6],[-1,2],[0,2],[2,5],[0,2],[0,4],[1,10],[1,2],[0,1],[2,0],[1,1],[2,-1]],[[7210,8211],[-1,-4],[-1,-5],[1,-6],[-2,-2],[-1,1],[-7,16],[-1,5],[1,-1],[1,-4],[2,-2],[1,3],[-1,4],[-1,3],[-2,2],[0,6],[2,2],[2,-1],[0,-2],[-1,-1],[0,-1],[0,-1],[1,-3],[6,-3],[1,-2],[1,0],[-1,-4]],[[7375,8264],[1,-1],[1,1],[1,0],[1,-2],[-1,-6],[1,-4],[0,-3],[1,-2],[-1,-2],[-2,3],[-3,-5],[0,1],[-1,4],[-1,2],[-1,2],[-4,-2],[-3,0],[-1,2],[1,3],[6,3],[2,3],[1,6],[1,0],[1,-3]],[[7419,8301],[1,-4],[-1,-1],[-1,-8],[-1,-2],[-1,-2],[-2,0],[0,1],[-2,6],[-4,1],[0,-2],[-1,-3],[0,-2],[2,-4],[1,-2],[1,-3],[3,-3],[0,-5],[-2,0],[-4,8],[-4,-3],[-1,1],[-1,2],[-1,1],[-3,-2],[-1,1],[-1,2],[0,3],[0,3],[-1,3],[-1,2],[1,3],[0,1],[1,-1],[0,-2],[1,-2],[5,-2],[2,1],[1,5],[-1,0],[-2,4],[1,4],[-1,4],[0,3],[1,2],[3,-3],[0,-4],[-1,-2],[3,1],[6,9],[3,0],[2,-9]],[[7281,8339],[0,-3],[-2,6],[-1,2],[0,5],[1,2],[1,1],[1,-4],[0,-3],[0,-3],[0,-3]],[[7281,8425],[0,-4],[1,-1],[-1,-3],[-4,5],[2,-6],[1,-1],[1,1],[2,-1],[0,-2],[0,-2],[0,-3],[1,-3],[0,-3],[1,-4],[0,-3],[-1,-1],[-5,4],[-1,-2],[0,-2],[0,-2],[0,-4],[1,-2],[-1,-2],[0,-3],[0,-3],[0,-3],[0,-1],[-1,-1],[0,-1],[0,-2],[0,-3],[0,-3],[0,-3],[0,-3],[0,-2],[-1,1],[-1,3],[-1,4],[0,4],[0,3],[0,3],[0,4],[-1,12],[0,2],[-1,0],[0,-2],[0,-5],[-2,0],[0,1],[-1,4],[0,-4],[0,-2],[0,-1],[-1,-2],[-1,2],[-2,8],[0,6],[0,1],[-1,0],[-1,1],[0,3],[1,4],[5,18],[1,2],[1,1],[0,-5],[-1,-5],[0,-4],[1,-3],[-1,-3],[1,-1],[4,3],[1,3],[0,4],[-1,3],[-1,2],[-1,1],[1,3],[1,0],[1,2],[0,3],[-1,1],[0,1],[-1,2],[2,4],[6,2],[0,-1],[0,-1],[-1,-1],[0,-2],[0,-10]],[[7267,8542],[0,-1],[1,1],[12,-11],[3,0],[1,-3],[-1,-4],[-13,8],[-2,5],[-2,1],[-1,3],[0,2],[1,0],[1,0],[0,-1]],[[7573,8556],[1,-2],[-1,-1],[0,-2],[0,-2],[0,-1],[-3,0],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[3,1]],[[7304,8553],[8,-4],[1,-1],[0,-4],[-1,-3],[-1,-3],[-1,0],[-3,3],[-1,0],[-3,-3],[-2,0],[-1,2],[1,5],[-1,2],[-3,-2],[-5,3],[-4,-3],[-2,0],[-1,2],[-1,5],[5,-2],[1,2],[2,5],[1,1],[2,0],[1,-2],[8,-3]],[[7695,8560],[-8,-6],[-2,3],[1,7],[1,2],[11,14],[4,2],[1,3],[0,-1],[-1,-5],[-3,-7],[-4,-12]],[[7644,8599],[-2,-3],[-1,1],[-2,2],[-1,3],[1,4],[1,2],[0,4],[1,2],[2,-6],[0,-4],[0,-4],[0,-1],[1,0]],[[7637,8623],[-2,-3],[-1,-3],[0,-2],[1,-2],[0,-2],[-1,-7],[-2,-1],[-5,3],[0,2],[-1,1],[-1,-3],[-1,-1],[-1,1],[-1,2],[0,3],[1,5],[1,1],[8,-7],[1,2],[-1,3],[0,3],[-1,2],[0,1],[0,1],[2,2],[2,3],[0,1],[1,1],[0,1],[1,-3],[0,-1],[1,0],[-1,-3]],[[7674,8635],[9,-11],[1,0],[0,-5],[-1,-3],[0,-3],[-1,-2],[0,-5],[0,-2],[0,-3],[0,-4],[-1,2],[0,2],[-1,4],[1,3],[0,3],[1,4],[-3,9],[-1,1],[0,1],[-1,0],[-1,-3],[0,-2],[0,-3],[1,-5],[1,-2],[-2,-2],[-2,4],[-1,-4],[1,0],[2,-4],[1,-3],[0,-5],[-1,-3],[0,-1],[-1,-1],[0,-3],[1,-2],[-1,-3],[-1,2],[-3,10],[-8,0],[-5,6],[-1,3],[0,2],[-10,1],[-1,4],[1,3],[0,2],[0,2],[0,3],[0,2],[0,2],[0,3],[1,0],[3,0],[1,-2],[1,-6],[0,-2],[1,-2],[1,2],[3,11],[2,2],[1,1],[1,-2],[0,-4],[5,2],[1,-2],[0,-4],[0,-4],[0,-3],[1,-1],[0,1],[1,3],[1,7],[1,7],[2,2]],[[7686,8644],[1,-3],[0,-2],[-1,0],[0,-1],[0,-2],[-1,-2],[0,-2],[0,-1],[-1,1],[-1,4],[0,2],[-5,-1],[-1,4],[1,2],[4,-1],[4,2]],[[7690,8646],[2,-4],[2,1],[1,0],[0,-4],[0,-2],[0,-1],[1,-2],[-1,-4],[-2,-2],[-2,-1],[-1,-2],[-1,-4],[-1,-6],[0,-4],[1,0],[0,3],[2,8],[1,1],[3,-6],[-1,-2],[-2,-1],[-1,-1],[-4,-11],[-1,1],[0,1],[0,4],[0,2],[0,3],[0,4],[0,3],[0,2],[1,3],[1,2],[1,6],[1,5],[0,2],[-1,2],[1,4]],[[8145,8656],[2,-3],[2,1],[1,-5],[0,-2],[-4,1],[-4,1],[-1,2],[1,7],[2,7],[1,3],[2,-5],[-1,-5],[-1,-2]],[[7619,8703],[1,-3],[-13,3],[0,3],[2,4],[6,-5],[4,-2]],[[8125,8710],[-1,-2],[0,-1],[0,-2],[2,-4],[-1,0],[0,-2],[0,-1],[0,-1],[1,-1],[0,-1],[0,-2],[-1,0],[-1,-2],[2,-3],[2,1],[2,-1],[0,-2],[0,-1],[-1,-2],[-2,0],[-1,-2],[1,-1],[0,-3],[0,-3],[0,-3],[0,-3],[-1,0],[-3,2],[-1,3],[-1,4],[-1,6],[-1,7],[-2,3],[-3,1],[-1,1],[0,2],[-1,4],[0,2],[-3,4],[1,1],[1,0],[0,2],[-1,0],[0,2],[-1,1],[5,2],[4,-5],[1,2],[0,1],[0,5],[0,2],[1,1],[1,-1],[0,-1],[1,-4],[2,-2],[1,-1],[0,-2]],[[7679,8732],[0,-3],[-2,-3],[-1,-4],[0,-5],[0,-3],[-1,-2],[-2,-2],[-3,2],[-2,-3],[-2,3],[-1,-1],[0,-2],[-1,-1],[-1,2],[0,2],[0,3],[0,2],[0,1],[1,4],[1,3],[1,1],[2,-3],[3,6],[1,4],[0,1],[2,-2],[1,1],[2,-1],[2,2],[0,-2]],[[7655,8725],[0,-2],[1,2],[1,1],[1,3],[1,1],[2,-1],[0,-4],[-1,-2],[-3,-4],[-4,-1],[-2,1],[-1,4],[0,3],[0,2],[0,4],[2,3],[1,0],[1,-3],[1,-7]],[[7642,8732],[1,0],[1,-5],[2,-4],[1,0],[0,-1],[-1,-6],[-2,0],[-3,2],[-7,-3],[-1,1],[0,3],[1,4],[0,1],[5,3],[0,-1],[1,0],[1,3],[1,6],[0,-1],[0,-2]],[[7709,8703],[-1,-1],[-1,0],[-2,4],[-2,0],[-1,2],[2,4],[1,3],[-1,4],[0,3],[0,1],[0,1],[0,1],[-2,2],[0,2],[1,1],[1,4],[1,1],[1,-1],[1,-4],[0,-6],[0,-6],[0,-5],[1,-3],[2,-3],[-1,-4]],[[7712,8744],[0,-1],[-1,-2],[0,-1],[-1,3],[-1,0],[-1,-2],[-2,-6],[0,1],[0,1],[0,2],[0,1],[0,1],[-8,1],[0,2],[10,3],[0,1],[3,1],[1,-2],[0,-1],[0,-1],[0,-1]],[[7471,8802],[-2,-3],[-3,2],[-3,13],[-1,10],[4,11],[7,4],[2,-3],[-1,-7],[-1,-6],[-2,-9],[1,-6],[-1,-6]],[[7506,8836],[-3,-2],[0,2],[1,3],[2,1],[0,-1],[0,-1],[0,-2]],[[7679,8825],[-1,-4],[-2,-2],[-1,-4],[-2,-2],[0,-2],[1,-2],[0,-1],[0,-4],[-2,-2],[-1,-1],[-2,5],[-1,-1],[-2,-4],[-9,-8],[1,1],[0,3],[1,1],[1,4],[0,1],[4,2],[1,3],[-1,4],[-2,0],[-2,-4],[-7,-3],[-8,-3],[0,4],[0,1],[1,3],[2,3],[2,5],[4,6],[4,-1],[8,10],[8,11],[4,11],[1,1],[2,-1],[0,-3],[1,-9],[0,-3],[-3,-15]],[[7518,8862],[-2,-1],[0,1],[0,2],[1,1],[1,1],[1,-1],[-1,-3]],[[7941,8872],[-1,-5],[-1,0],[-1,3],[0,3],[3,6],[2,0],[1,-4],[-1,-2],[-2,-1]],[[7492,8845],[-7,-6],[-5,2],[-3,12],[-2,5],[1,8],[1,7],[2,5],[1,2],[8,1],[2,-2],[1,-6],[1,-9],[2,-8],[-2,-11]],[[7956,8881],[-2,-2],[-1,1],[0,3],[1,0],[0,2],[1,1],[1,-1],[0,-2],[0,-2]],[[7987,8886],[1,-1],[1,0],[1,2],[1,0],[1,0],[0,-2],[0,-4],[0,-1],[-1,-1],[0,-2],[0,-2],[0,-3],[0,-1],[0,-1],[-2,0],[-8,-8],[-2,1],[-1,0],[0,-1],[-1,0],[0,1],[1,2],[0,2],[5,4],[1,2],[-1,3],[-3,-1],[-1,4],[2,10],[1,4],[2,-2],[0,-1],[2,-1],[1,-3]],[[7967,8921],[1,-2],[1,0],[1,-2],[-1,-1],[-1,-2],[0,-3],[0,-2],[0,-2],[-1,0],[-1,2],[-1,0],[-1,-2],[1,-1],[-2,-7],[-2,-1],[-3,2],[0,4],[1,4],[0,4],[1,2],[1,-1],[1,2],[1,2],[1,3],[3,1]],[[7556,8949],[-2,0],[-2,3],[-1,3],[0,4],[1,3],[1,1],[4,-4],[1,-2],[0,-2],[-1,-2],[-1,-4]],[[8080,7991],[1,-1],[5,-12],[0,-1],[0,-1],[0,-3],[0,-3],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[-2,-3],[-1,-1],[-3,1],[-1,-2],[-1,-1],[0,-7],[-1,-5],[-1,-3],[-1,-1],[0,-4],[-1,-5],[-1,-6],[-2,-4],[0,-2],[-1,-1],[-10,-1],[0,-1],[1,-3],[6,-12],[3,-2],[1,-2],[0,-5],[1,-3],[0,-1],[0,-3],[0,-1],[0,-2],[0,-1],[-5,-11],[0,-1],[0,-1],[0,-2],[0,-1],[1,-1],[9,-2],[0,-1],[0,-1],[1,-2],[0,-1],[0,-2],[0,-2],[0,-4],[1,-4],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[2,-3],[0,-2],[0,-1],[0,-3],[0,-2],[-1,-1],[-3,-3],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[1,-3],[0,-4],[1,0],[0,-1],[0,-1],[-3,-8],[-2,-2],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[0,-1],[1,-2],[1,-2],[0,-2],[0,-2],[0,-2],[0,-1],[-2,-3],[0,-1],[0,-2],[0,-1],[0,-1],[0,-7],[0,-1],[1,-7],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[1,-2],[0,-2],[0,-1],[1,-2],[1,-1],[3,-2],[1,0],[0,-1],[0,-2],[0,-1],[-1,-2],[0,-1],[0,-3],[0,-1],[0,-1],[1,-2],[1,-1],[1,1],[1,0],[5,-7],[1,-3],[1,-2],[1,-4],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[-1,-1],[-2,-6],[0,-1],[-1,-2],[0,-2],[-1,-3],[-2,-3],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[1,-1],[6,-9],[0,-1],[1,-1],[0,-1],[2,-2],[3,1],[1,0],[1,-2],[1,-9],[0,-1],[1,-1],[0,-1],[0,-1],[-1,-3],[0,-2],[0,-4],[0,-2],[0,-1],[2,0],[1,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[1,-1],[0,-1],[1,-2],[5,1],[0,-177],[0,-2],[10,-18],[1,-4],[1,-4],[0,-1],[1,-3],[1,-1],[0,-1],[0,-1],[0,-1],[1,0],[2,5],[1,0],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[1,-1],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-6],[-1,-1],[0,-2],[-3,-6],[-1,-2],[0,-2],[0,-2],[0,-1],[0,-1],[1,-8],[0,-2],[0,-2],[-2,-7],[-1,-1],[-2,0],[-1,-1],[0,-3],[-2,-9],[0,-1],[-1,-1],[-1,-1],[0,1],[-1,1],[0,1],[0,1],[0,3],[0,1],[0,2],[-1,1],[0,1],[0,1],[0,1],[-1,1],[-1,0],[0,-1],[-1,-2],[0,-2],[0,-1],[0,-5],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-2],[-1,-1],[-12,-11],[-1,-3],[-2,-6],[-1,-3],[0,-3],[0,-2],[0,-1],[-1,-2],[0,-1],[-7,-1],[-3,-4],[-2,-6],[-1,-2],[-2,0],[0,1],[-1,1],[0,1],[-1,0],[-1,0],[0,-2],[-1,-1],[-1,0],[-1,1],[-1,-1],[-1,-1],[0,-1],[-1,-2],[0,-2],[0,-2],[0,-1],[0,-2],[0,-4],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-2],[-4,-1],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[-1,-2],[-1,-1],[-2,-8],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-3],[1,-1],[-1,-1],[0,-1],[-1,1],[0,-1],[0,-1],[0,-1],[1,-3],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-1],[0,-1],[3,-5],[1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-1],[-2,1],[-3,-3],[-1,2],[-4,-4],[-1,1],[-4,-5],[-1,-1],[-1,0],[0,-1],[-1,-2],[0,-2],[-1,-2],[0,-1],[1,-3],[0,-1],[1,-1],[0,-3],[0,-1],[-1,-1],[0,-7],[-1,-1],[0,-3],[-5,-8],[-1,-1],[-1,-1],[0,-1],[0,-2],[0,-1],[-1,-4],[0,-1],[0,-1],[1,-1],[0,-2],[0,-4],[1,-4],[1,-5],[0,-1],[0,-1],[-1,-3],[0,-1],[0,-1],[0,-2],[1,-1],[0,-1],[0,-1],[1,-4],[0,-1],[1,-1],[2,0],[0,-1],[0,-2],[0,-1],[0,-1],[0,-4],[0,-1],[-1,-1],[0,-2],[-1,-2],[0,-1],[0,-1],[0,-4],[-1,-4],[0,-3],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[1,-1],[-1,-2],[0,-5],[-1,-3],[0,-1],[0,-1],[0,-2],[-1,-3],[-1,-3],[0,-4],[-1,-3],[0,-2],[-1,-1],[-2,-1],[-3,1],[-3,4],[0,1],[0,1],[-1,9],[0,1],[-1,1],[-7,-1],[-4,4],[-5,1],[-1,-1],[-3,-7],[-2,-11],[-1,-3],[0,-3],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[-1,0],[-2,-6],[-1,-1],[-2,2],[-1,0],[-18,-24],[-5,-12],[-2,-4],[-1,-1],[-7,-1],[0,1],[0,1],[-1,1],[0,1],[0,3],[0,1],[0,1],[-1,3],[-1,1],[-1,1],[0,-2],[-2,-6],[-2,-10],[0,-1],[0,-2],[0,-3],[0,-1],[0,-2],[0,-1],[-1,-1],[-3,-6],[0,-2],[0,-1],[0,-3],[0,-4],[20,-129],[0,-1],[0,-1],[0,-102],[0,-16],[0,-3],[0,-2],[0,-4],[0,-7],[1,-42],[-1,-8],[0,-15],[0,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-1],[0,-21],[0,-3],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-3],[0,-2],[0,-1],[0,-2],[-1,-41],[0,-4],[0,-4],[0,-2],[0,-3],[0,-21],[-1,-3],[0,-1],[0,-1],[0,-1],[0,-2],[0,-3],[0,-9],[0,-13],[0,-2],[0,-5],[0,-3],[0,-1],[0,-2],[0,-1],[0,-2],[-2,-9],[-1,-11],[-1,-1],[0,-1],[-1,1],[-1,5],[0,1],[-2,2],[0,1],[-1,-1],[-2,-5],[-1,-2],[-2,-11],[-2,-6],[-2,-1],[-3,-3],[0,-1],[0,-2],[0,-5],[-1,-3],[0,-3],[0,-6],[0,-1],[-1,-1],[0,-1],[0,-3],[-2,-5],[-10,-5],[0,-2],[0,-4],[2,-9],[1,-3],[1,-2],[1,-1],[1,1],[1,1],[1,0],[0,-2],[1,-4],[1,-3],[1,-1],[3,-1],[0,-1],[1,-2],[0,-4],[0,-2],[0,-2],[-1,-2],[-1,-8],[0,-1],[-1,-1],[0,-1],[1,-1],[0,-2],[1,-1],[0,1],[3,4],[1,-1],[0,-2],[0,-3],[0,-1],[-1,-2],[0,-1],[0,-2],[0,-1],[0,-3],[0,-2],[0,-2],[0,-1],[3,-5],[0,-1],[0,-2],[0,-1],[-1,-3],[0,-2],[0,-2],[0,-3],[0,-8],[0,-2],[0,-3],[1,-2],[1,-3],[0,-3],[0,-2],[0,-4],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[1,-1],[0,-2],[0,-1],[-1,-2],[0,-1],[-2,-5],[0,-1],[-1,-1],[0,-1],[-1,-4],[-1,-2],[0,-1],[0,-4],[0,-1],[-1,-2],[0,-1],[0,-2],[0,-3],[0,-2],[0,-2],[1,-1],[0,-2],[0,-1],[0,-1],[0,-3],[0,-2],[0,-4],[1,-3],[0,-2],[-1,-2],[0,-1],[0,-3],[0,-2],[0,-3],[0,-3],[0,-1],[0,-2],[1,-3],[0,-2],[0,-1],[2,-2],[0,-1],[3,2],[0,-1],[0,-1],[1,-2],[0,-2],[0,-1],[0,-2],[0,-2],[0,-3],[1,-4],[0,-2],[0,-1],[1,-2],[0,-6],[1,-8],[0,-3],[0,-2],[0,-2],[0,-3],[0,-1],[0,-2],[0,-4],[0,-1],[0,-1],[0,-1],[0,-2],[0,-6],[0,-4],[0,-1],[0,-3],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-1],[0,-1],[0,-2],[1,-2],[0,-2],[0,-2],[-1,-3],[0,-1],[0,-1],[0,-3],[1,-4],[0,-2],[-1,-2],[0,-1],[0,-3],[-1,-1],[0,-2],[0,-1],[0,-1],[0,-4],[0,-1],[1,-1],[0,-2],[2,-2],[1,-3],[2,-1],[0,-1],[1,-1],[-1,-1],[0,-2],[0,-1],[0,-1],[2,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[7,-10],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[-2,-10],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-1],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-1],[-1,1],[0,1],[-1,0],[-3,3],[-1,-2],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-3,-9],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[-1,-2],[0,-1],[0,-2],[1,-2],[0,-2],[0,-3],[0,-1],[1,-3],[0,-2],[1,-1],[-1,-1],[-1,3],[-1,0],[0,-1],[-1,-2],[-1,-5],[-1,-6],[0,-1],[0,-3],[0,-1],[-1,-1],[-1,0],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-4],[-1,-2],[0,-1],[0,-1],[-4,-9],[0,-3],[-1,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-3],[3,-14],[1,-2],[0,-2],[0,-3],[0,-3],[0,-1],[1,-1],[1,-3],[0,-2],[1,-2],[0,-1],[0,-1],[0,-1],[-1,-1],[-2,2],[-1,2],[0,3],[-1,4],[0,1],[0,1],[-1,0],[-1,-2],[-1,0],[0,-1],[0,-2],[0,-3],[0,-1],[1,-2],[0,-2],[1,-5],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[-1,0],[-1,2],[0,1],[0,-1],[-1,-1],[0,-2],[0,-2],[-1,-2],[0,-1],[0,-5],[0,-1],[0,-5],[0,-4],[0,-3],[0,-2],[0,-1],[1,-5],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[-1,1],[-1,1],[0,1],[0,2],[-1,1],[0,-1],[0,-1],[0,-2],[0,-1],[1,-6],[0,-1],[0,-1],[0,-1],[-1,-4],[0,-2],[0,-1],[0,-1],[0,-2],[1,-4],[1,-4],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-3],[0,-1],[0,-1],[1,0],[1,2],[1,0],[1,0],[1,1],[1,4],[0,1],[1,0],[0,-1],[1,-2],[1,-5],[1,-2],[0,-1],[0,-1],[0,-2],[-1,-1],[0,-1],[-1,0],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[1,-3],[1,0],[1,0],[1,0],[0,1],[1,3],[0,1],[0,1],[2,0],[0,1],[1,1],[0,1],[1,1],[5,0],[0,1],[0,1],[0,1],[0,3],[0,1],[0,3],[-1,2],[0,1],[0,1],[1,2],[0,1],[1,-1],[0,-1],[1,-2],[0,-2],[1,-1],[0,-3],[0,-1],[0,-1],[0,-3],[0,-3],[1,-1],[0,-2],[0,-1],[0,-1],[0,-1],[1,-1],[3,1],[0,-1],[1,-1],[1,-3],[0,-1],[1,-2],[2,-4],[0,-1],[0,-2],[0,-1],[1,-1],[0,-1],[0,-1],[3,-1],[1,-1],[0,-2],[0,-1],[0,-2],[0,-3],[0,-1],[0,-1],[0,-1],[0,-1],[2,-7],[1,-1],[1,1],[2,4],[2,6],[1,1],[1,-1],[1,-2],[1,-1],[0,-2],[0,-1],[0,-3],[0,-1],[0,-1],[0,-1],[0,-1],[4,-8],[0,-2],[1,-3],[0,-1],[2,-1],[1,-2],[1,2],[1,11]],[[7685,3060],[0,-1],[0,-3],[0,-2],[0,-1],[0,-1],[0,-3],[0,-1],[0,-1],[-1,-6],[0,-1],[0,-1],[0,-1],[-2,-3],[-3,0],[0,-2],[0,-1],[0,-2],[-1,-1],[0,1],[0,2],[0,3],[0,1],[0,1],[-2,3],[0,1],[0,1],[0,1],[0,5],[0,1],[0,1],[0,1],[-1,-1],[-1,-4],[0,1],[-1,0],[0,1],[1,1],[0,4],[0,1],[-1,1],[0,-1],[-1,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-3],[0,-1],[0,-2],[0,-1],[-1,-8],[0,-1],[-2,-4],[0,-1],[0,-2],[0,-4],[0,-1],[0,-1],[0,-1],[0,-1],[-1,0],[-1,1],[-2,5],[-1,2],[0,1],[0,1],[0,1],[-1,0],[-1,-2],[-2,2],[0,1],[0,1],[0,1],[0,1],[0,1],[-1,1],[-1,-3],[-1,-5],[-1,-1],[0,-4],[0,-1],[0,-1],[0,-1],[-1,-2],[-2,-5],[-1,-2],[0,-2],[0,-1],[0,-1],[-2,-4],[-1,-3],[-1,0],[-1,2],[0,3],[-1,1],[0,1],[0,2],[0,1],[0,5],[0,2],[0,1],[0,1],[0,1],[-1,0],[-1,-2],[-1,-1],[0,-2],[0,-3],[0,-3],[-1,-1],[0,-1],[-2,-4],[0,-1],[0,-1],[-1,0],[-1,1],[0,1],[0,4],[-1,1],[0,-1],[-1,-3],[0,-1],[-1,2],[-4,-3],[-1,-3],[-1,-6],[-2,-7],[0,-1],[0,-1],[0,-3],[0,-1],[0,-1],[1,0],[1,-1],[0,-1],[0,-1],[0,-1],[1,-2],[0,-5],[0,-1],[0,-2],[0,-2],[-1,-1],[0,-1],[0,-1],[-1,-2],[0,-3],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-2],[0,-3],[0,-1],[0,-2],[0,-6],[0,-2],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-3],[0,-2],[0,-1],[-1,-6],[0,-3],[0,-2],[0,-1],[0,-1],[0,-2],[-1,-4],[0,-2],[0,-3],[0,-2],[0,-2],[0,-2],[0,-1],[0,-2],[-2,-4],[-1,-1],[0,2],[0,3],[-1,1],[-2,-2],[-1,-1],[-2,-10],[-1,-4],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[1,-1],[-1,-4],[0,-3],[0,-7],[0,-1],[-1,-8],[0,-2],[-1,-12],[0,-1],[0,-1],[-2,-5],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[-4,-17],[0,-1],[0,-6],[-1,-1],[-1,-6],[0,-2],[0,-2],[0,-2],[0,-2],[0,-2],[0,-6],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[-2,-11],[0,-3],[0,-2],[-1,-2],[0,-4],[0,-1],[0,-2],[0,-5],[0,-2],[0,-2],[-2,-7],[-1,-4],[-3,-1],[-1,-2],[-1,-7],[-3,-13],[-1,-7],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[-1,-1],[0,-1],[-1,-2],[0,-3],[0,-2],[0,-2],[-1,-1],[0,-3],[0,-2],[0,-1],[0,-2],[-2,-5],[-1,-2],[-1,0],[0,-1],[-1,-4],[0,-2],[-1,-3],[0,-6],[0,-4],[-1,-1],[0,1],[-1,3],[0,1],[-4,-3],[0,-1],[0,-2],[0,-2],[-1,-1],[0,-1],[-2,-2],[0,-1],[0,-2],[-1,-1],[0,-1],[-1,0],[0,1],[0,1],[-1,2],[0,1],[-6,1],[-3,-6],[-4,3],[0,1],[-1,1],[0,3],[-1,4],[0,2],[0,1],[0,1],[0,1],[0,1],[-1,0],[-1,1],[-1,1],[0,1],[0,1],[-2,-2],[-1,1],[-2,0],[0,3],[-1,3],[0,1],[0,1],[-1,0],[-7,-1],[0,-1],[-1,-2],[0,-1],[-2,2],[-1,3],[-1,10],[-6,-5],[0,-2],[-1,-1],[0,-1],[0,1],[0,1],[-1,3],[0,2],[0,2],[0,2],[0,3],[0,2],[-1,2],[0,3],[-1,3],[0,3],[0,3],[0,1],[0,2],[0,6],[0,1],[-1,2],[-1,3],[-3,6],[0,2],[0,2],[-1,2],[0,6],[-1,5],[-1,0]],[[7180,7669],[0,1],[0,4],[0,1],[7,-4],[7,-3],[6,3],[5,-1],[3,-6],[8,-24],[6,-14],[13,-14],[10,-17],[0,-3],[0,-2],[0,-4],[0,-3],[-1,-2],[-2,5],[0,3],[2,3],[0,3],[-2,0],[-1,-1],[-1,-3],[-1,-5],[0,-3],[9,-21],[8,-22],[0,-1],[1,-2],[0,-2],[0,-1],[0,-1],[-1,-1],[1,-7],[1,-5],[1,-3],[5,-6],[6,-10],[12,2],[3,6],[4,1],[2,5],[10,7],[2,-3],[0,-2],[0,-1],[1,-3],[0,-2],[0,-1],[1,-2],[4,3],[1,-1],[2,-3],[1,-5],[0,-3],[1,-2],[0,-2],[-1,-4],[0,-3],[-1,-5],[-2,-10],[-1,-3],[-3,-3],[0,-3],[-1,-5],[0,-8],[0,-5],[-1,-3],[1,-3],[0,-2],[0,-2],[1,-1],[0,-4],[-2,-1],[-1,-3],[-3,-2],[-7,-13],[-8,-12],[0,-2],[-1,-4],[0,-3],[0,-2],[1,-3],[0,-2],[0,-3],[0,-6],[0,-2],[0,-2],[1,-4],[0,-2],[1,-7],[0,-5],[0,-5],[-1,-4],[-2,-6],[-1,-5],[0,-7],[3,-20],[-1,-2],[1,-3],[0,-3],[0,-3],[0,-4],[0,-2],[1,-1],[0,-4],[1,-7],[1,-3],[-1,-3],[-1,-5],[-3,-12],[-1,-6],[-1,-3],[0,-2],[1,-1],[0,-3],[-1,-3],[0,-2],[-2,-6],[-1,-2],[0,-4],[1,-2],[1,-6],[0,-3],[0,-3],[0,-2],[0,-3],[0,-3],[1,-4],[0,-9],[0,-9],[1,-9],[0,-2],[0,-3],[1,-3],[0,-2],[0,-5],[0,-3],[0,-2],[1,-6],[1,-8],[1,-2],[0,6],[-1,6],[-1,10],[0,1],[-1,20],[0,12],[-1,7],[0,6],[1,4],[-1,6],[-1,8],[-1,5],[1,3],[1,1],[1,1],[1,3],[4,0],[1,2],[0,4],[0,4],[-1,3],[0,2],[0,4],[0,1],[0,1],[0,1],[1,0],[0,1],[0,2],[1,3],[0,3],[0,3],[0,2],[0,2],[1,0],[0,2],[0,2],[-1,2],[0,2],[-1,3],[1,4],[2,8],[0,4],[0,2],[0,3],[0,3],[0,2],[1,2],[1,3],[1,8],[1,1],[3,-4],[1,-2],[0,-3],[0,-3],[1,2],[0,3],[0,3],[-1,9],[0,2],[1,3],[0,-2],[0,-3],[1,-4],[0,-4],[1,-5],[2,-7],[2,-2],[1,-4],[0,-9],[0,-4],[0,-5],[-1,-5],[1,-6],[-1,-3],[-1,-7],[0,-8],[-1,-6],[0,-3],[1,-7],[0,-3],[-1,-4],[0,-3],[-1,-3],[0,-1],[0,-2],[0,-2],[0,-5],[1,-2],[0,-1],[0,-1],[0,-4],[0,-2],[0,-2],[-1,-14],[-1,-5],[-3,-8],[-6,-28],[0,-3],[0,-4],[0,-5],[1,-3],[0,-2],[-1,-2],[1,-4],[3,-6],[3,-13],[1,-2],[2,1],[2,-4],[1,0],[1,2],[1,2],[0,2],[-1,1],[1,3],[0,3],[0,4],[-2,17],[0,2],[-1,1],[0,3],[-1,10],[-1,4],[-1,6],[-1,3],[2,3],[6,-6],[3,0],[4,6],[1,-2],[1,1],[0,1],[0,3],[0,1],[0,1],[1,2],[1,6],[1,2],[3,19],[0,7],[0,5],[0,6],[0,5],[-2,10],[0,3],[0,6],[-1,4],[0,2],[-1,5],[0,3],[-1,4],[1,2],[0,3],[0,4],[0,3],[-1,3],[-1,2],[0,1],[0,1],[-1,11],[0,1],[0,4],[-1,10],[0,4],[-1,4],[-1,6],[-1,4],[1,7],[-1,7],[-1,6],[0,2],[-4,4],[-1,2],[0,2],[-1,4],[4,0],[0,1],[-1,6],[0,12],[-2,10],[0,3],[0,2],[1,1],[1,4],[1,3],[1,5],[2,12],[0,4],[1,3],[-1,6],[0,3],[0,7],[0,4],[1,3],[1,0],[2,2],[1,3],[0,3],[2,12],[0,5],[0,8],[0,5],[-1,4],[0,4],[-1,1],[0,2],[0,3],[0,1],[0,2],[-1,1],[-1,9],[-2,2],[0,1],[0,2],[0,2],[0,1],[0,1],[0,3],[0,3],[0,2],[0,3],[-1,5],[-3,4],[-3,6],[-3,1],[-2,3],[-5,2],[-1,-1],[0,-2],[-1,-3],[0,-1],[-1,0],[0,1],[-1,2],[0,3],[0,3],[-1,7],[0,5],[-4,14],[-1,2],[-1,3],[-2,6],[-1,3],[-3,4],[1,5],[1,2],[1,3],[1,12],[-1,5],[0,4],[0,3],[-1,3],[-1,2],[0,1],[0,2],[0,3],[0,1],[-1,1],[-1,-2],[-5,4],[-3,6],[-7,8],[-11,1],[-1,3],[-7,22],[-2,2],[0,2],[-2,7],[-2,11],[0,3],[2,5],[0,1],[0,2],[0,2],[1,1],[0,1],[1,4],[-1,3],[-3,3],[-1,5],[0,4],[-1,5],[0,3],[0,4],[0,3],[2,4],[2,9],[0,2],[0,1],[0,3],[0,1],[1,6],[0,5],[1,13],[0,4],[-1,8],[0,5],[-1,1],[0,1],[0,2],[0,2],[0,3],[-1,2],[-1,5],[-1,2],[-4,3],[-1,2],[-1,4],[1,2],[1,1],[2,0],[-1,7],[-1,3],[-3,2],[-2,5],[-1,0],[-1,-2],[-1,0],[1,5],[1,2],[4,0],[1,1],[1,3],[1,-1],[0,1],[0,4],[-1,11],[-2,4],[-3,0],[-2,3],[0,5],[0,3],[1,3],[1,4],[0,3],[1,1],[0,-1],[1,-1],[0,-2],[2,12],[0,1],[2,1],[1,2],[0,2],[0,1],[1,2],[0,4],[0,3],[0,2],[1,1],[0,2],[-1,2],[-2,-6],[-1,2],[0,5],[0,9],[0,3],[0,4],[1,0],[8,-3],[2,4],[9,5],[2,4],[6,6],[12,4],[9,-1],[8,0],[14,-1],[11,0],[7,10],[9,2],[9,1],[5,11],[7,3],[3,-8],[3,-6],[2,2],[9,-1],[2,2],[0,4],[0,2],[0,2],[-1,2],[1,2],[1,4],[1,7],[1,1],[0,1],[8,0],[4,11],[10,4],[11,5],[8,-8],[1,-2],[0,-4],[0,-3],[-1,-3],[-1,-3],[0,-4],[0,-3],[-1,-1],[-4,-3],[-1,-4],[-2,-9],[-1,-2],[-1,-2],[-2,-3],[-2,-3],[-4,-12],[-3,-6],[-5,-3],[-4,-13],[-2,-4],[-1,-5],[-1,-1],[-1,-3],[0,-6],[1,-13],[0,-4],[0,-3],[0,-2],[0,-4],[0,-3],[1,-2],[0,-1],[2,-5],[9,-13],[0,-4],[0,-1],[0,-2],[2,-11],[6,-4],[2,-6],[1,-2],[1,-2],[0,-2],[0,-2],[1,-6],[0,-2],[0,-3],[0,-5],[1,-2],[0,-3],[0,-2],[1,-2],[0,-1],[0,11],[0,5],[-1,4],[0,6],[-3,11],[-1,4],[-1,2],[-2,1],[0,1],[-1,3],[-1,8],[-1,4],[-4,8],[-9,11],[-1,4],[-1,6],[0,3],[0,2],[0,5],[0,1],[0,2],[0,2],[0,1],[0,2],[0,1],[1,14],[2,5],[3,3],[4,7],[8,3],[8,3],[2,6],[7,0],[2,1],[1,5],[0,5],[3,8],[0,5],[0,1],[-1,-5],[-1,1],[1,9],[1,6],[6,12],[2,5],[1,10],[0,1],[0,6],[0,3],[-1,2],[-1,1],[-1,0],[0,-2],[0,-4],[0,-7],[-1,-3],[-1,-4],[0,-2],[-1,-1],[-2,1],[-2,4],[-1,4],[-2,7],[0,2],[-1,2],[0,2],[0,3],[0,2],[-1,4],[-1,2],[0,1],[-1,8],[-2,6],[-1,4],[2,4],[7,-3],[3,-7],[2,1],[-2,8],[-2,4],[-11,2],[-1,4],[0,1],[0,2],[0,2],[0,2],[-3,3],[-1,1],[0,3],[2,10],[-1,6],[-1,4],[-1,3],[-5,4],[-13,2],[-2,4],[0,4],[0,1],[0,2],[0,4],[0,2],[-1,1],[0,1],[0,4],[0,3],[7,-2],[2,-6],[1,-5],[1,1],[0,2],[0,3],[0,1],[-1,0],[0,1],[-1,8],[0,2],[0,2],[-1,1],[-3,1],[-1,1],[0,2],[0,2],[1,0],[1,0],[2,2],[1,-3],[1,2],[1,2],[4,5],[3,-3],[1,-2],[0,-2],[0,-2],[0,-2],[1,-1],[-1,-2],[0,-1],[0,-2],[1,0],[2,2],[2,-3],[2,3],[1,0],[1,-2],[1,0],[-1,-6],[-1,-3],[-3,-5],[-2,-1],[0,-1],[0,-4],[0,-1],[1,0],[1,1],[1,-1],[1,1],[1,7],[1,2],[1,0],[0,-1],[1,-8],[0,-2],[0,-3],[-1,-2],[1,-2],[1,1],[0,3],[0,5],[0,4],[-1,6],[0,2],[0,3],[1,0],[2,-3],[4,0],[-1,2],[-3,3],[0,1],[-1,3],[0,3],[0,2],[0,3],[0,3],[0,1],[-1,1],[-7,-5],[0,4],[0,1],[-1,0],[-1,1],[0,1],[0,1],[-1,4],[0,1],[0,1],[-1,0],[0,1],[-1,4],[0,1],[-2,0],[0,2],[1,3],[5,5],[1,4],[-8,1],[0,1],[-1,2],[-1,1],[-1,0],[-1,1],[-1,4],[0,1],[-1,0],[-2,4],[-13,3],[1,4],[1,3],[1,2],[0,4],[0,3],[1,6],[1,1],[3,-4],[1,1],[0,2],[1,3],[1,4],[0,5],[-1,3],[-2,7],[0,1],[1,1],[1,0],[0,-1],[0,-1],[0,-1],[1,-1],[2,1],[0,-1],[2,-6],[1,-1],[1,1],[0,-2],[1,-4],[-5,2],[1,-3],[1,-5],[1,-6],[1,-1],[0,2],[1,1],[0,2],[1,1],[1,-1],[5,-7],[0,-2],[-1,-3],[0,-3],[0,-2],[1,0],[1,0],[0,-1],[0,-2],[0,-5],[1,-1],[2,2],[1,1],[1,0],[0,-2],[2,-3],[1,-1],[0,2],[0,2],[0,2],[-1,1],[-1,5],[0,2],[1,2],[1,3],[-2,2],[-3,-2],[-1,5],[1,1],[0,1],[1,1],[0,2],[1,5],[2,3],[1,-1],[1,-5],[1,-1],[0,-1],[0,-1],[1,1],[0,1],[0,5],[0,1],[0,5],[0,2],[-1,1],[0,3],[1,10],[1,6],[4,12],[2,2],[1,4],[1,2],[-1,3],[-1,1],[-1,-1],[-1,1],[-2,6],[1,4],[1,2],[2,1],[2,-2],[1,0],[0,-1],[1,-2],[0,-2],[0,-2],[-1,-2],[0,-1],[0,-3],[1,-1],[1,-1],[1,-3],[1,-1],[1,2],[-2,6],[1,1],[2,2],[0,1],[1,3],[1,4],[-1,2],[1,0],[1,1],[1,2],[-2,3],[-7,0],[-1,2],[-1,0],[-1,-2],[0,1],[-1,4],[-2,4],[-6,4],[-1,0],[-2,-4],[-1,3],[1,6],[0,4],[2,6],[1,3],[5,1],[9,-7],[1,1],[9,-11],[1,0],[1,2],[3,9],[1,0],[1,-4],[0,5],[2,3],[1,1],[1,1],[0,2],[0,1],[1,2],[0,7],[0,2],[0,2],[0,2],[1,3],[3,6],[1,1],[3,-4],[0,4],[-1,3],[-1,5],[2,7],[4,2],[1,3],[-1,3],[2,6],[2,3],[3,0],[2,-2],[-1,5],[-1,2],[-8,-6],[1,2],[5,6],[2,3],[3,0],[1,1],[0,1],[1,2],[1,8],[0,1],[1,-1],[1,-6],[2,-2],[0,-1],[0,-4],[4,0],[1,-2],[1,2],[1,0],[1,-1],[0,-1],[1,1],[1,2],[-1,2],[-2,4],[0,4],[0,3],[6,2],[6,1],[-1,6],[-1,2],[-1,0],[-1,2],[1,5],[10,5],[8,3],[4,10],[2,1],[1,3],[1,0],[1,-2],[0,-2],[0,-2],[0,-5],[1,-1],[1,1],[1,3],[2,6],[1,1],[1,0],[0,-5],[1,-3],[2,-1],[5,5],[3,-3],[1,2],[1,2],[0,4],[0,4],[-1,3],[1,7],[1,3],[2,0],[1,-4],[1,-1],[2,4],[3,-2],[2,2],[4,5],[4,-3],[4,8],[10,5],[9,8],[10,6],[7,8],[7,8],[2,6],[1,1],[-1,2],[0,-1],[-1,-3],[-1,-2],[-9,-7],[-2,0],[-2,4],[0,5],[0,3],[-1,1],[-1,-1],[-2,-7],[0,-3],[1,-1],[0,-2],[-1,-4],[-1,-2],[0,3],[-1,3],[-2,8],[0,2],[0,1],[-1,1],[0,2],[0,1],[1,4],[-1,2],[-1,-1],[-2,-5],[-4,-3],[-1,-4],[1,0],[2,-4],[1,0],[-1,-6],[-2,0],[-4,2],[0,2],[1,2],[-1,5],[0,5],[-1,3],[0,3],[3,4],[-1,3],[0,3],[0,3],[1,1],[-1,5],[1,3],[1,2],[1,0],[1,-1],[0,-1],[1,-10],[0,-1],[1,0],[2,5],[1,3],[0,4],[-2,4],[-1,2],[4,0],[1,3],[4,0],[0,-2],[-3,-5],[0,-3],[0,-2],[1,-2],[1,-1],[0,-2],[-1,-4],[1,-3],[1,2],[2,7],[1,1],[2,-1],[2,3],[0,1],[1,-1],[0,-3],[1,-2],[0,1],[2,2],[0,1],[3,-2],[1,2],[-1,5],[0,1],[-2,-1],[-1,1],[-1,5],[-1,1],[-2,-2],[-1,1],[0,4],[-1,2],[2,2],[11,-2],[3,-5],[1,2],[1,1],[1,2],[2,-1],[0,-1],[0,-2],[0,-3],[1,0],[0,-1],[1,-2],[0,-2],[-1,-3],[1,-3],[1,0],[1,3],[1,0],[1,0],[0,3],[0,2],[0,2],[0,2],[2,3],[2,1],[-1,6],[-2,2],[-3,0],[1,3],[8,-3],[0,-1],[1,-5],[1,-2],[0,-1],[4,0],[2,5],[3,-2],[3,8],[1,2],[13,-7],[3,-5],[2,-2],[2,1],[1,-1],[1,-3],[0,-2],[0,-2],[-1,-1],[-3,1],[-1,-4],[1,0],[1,-1],[0,-3],[-1,0],[-2,-3],[1,-6],[0,-2],[0,-2],[0,-1],[-1,1],[-1,-1],[-2,-4],[-4,-16],[-1,-2],[-2,0],[-1,-3],[3,-6],[2,-2],[0,4],[1,3],[0,3],[1,5],[1,2],[1,2],[1,4],[2,0],[2,3],[1,0],[1,-2],[1,-1],[3,2],[1,4],[-1,1],[0,2],[-1,2],[1,1],[2,-1],[1,2],[1,2],[1,1],[1,0],[1,-2],[0,-1],[-1,-1],[0,-2],[0,-3],[2,2],[2,0],[0,-5],[0,-1],[-2,-6],[-3,-14],[-1,-1],[1,-1],[3,3],[2,5],[1,1],[3,-4],[1,-1],[0,7],[1,3],[2,2],[0,1],[1,4],[0,3],[1,1],[1,3],[4,12],[2,2],[4,1],[-6,-13],[0,-3],[0,-1],[2,-5],[-1,-2],[-1,-2],[0,-3],[1,0],[2,1],[1,3],[-1,2],[0,3],[2,4],[1,2],[1,-2],[2,2],[3,0],[2,-3],[4,3],[0,1],[0,4],[-1,2],[-3,8],[-1,1],[-2,-2],[-1,2],[2,7],[2,3],[3,0],[2,-5],[0,-2],[1,-2],[1,1],[4,6],[1,0],[1,1],[1,2],[1,1],[-6,0],[-3,5],[2,5],[1,3],[1,0],[1,-3],[1,1],[0,1],[0,4],[1,1],[0,1],[5,0],[2,2],[4,11],[7,9],[0,-2],[0,-2],[-1,-3],[0,-2],[0,-2],[-1,-2],[-1,-2],[1,-2],[3,0],[0,1],[2,5],[0,1],[7,-3],[0,1],[1,2],[1,-1],[1,-1],[0,-3],[-1,-2],[-1,-2],[0,-3],[1,-3],[-4,0],[2,-7],[1,-2],[1,0],[4,5],[4,0],[1,2],[1,-4],[0,-2],[-1,-2],[-3,-4],[1,-3],[1,-2],[0,-2],[1,0],[0,-3],[0,-1],[1,1],[0,1],[1,2],[0,1],[1,0],[0,-1],[2,-7],[0,-4],[0,-6],[0,-3],[-1,-3],[-1,-3],[0,-2],[0,-2],[0,-1],[0,-5],[0,-3],[0,-2],[0,-1],[0,-4],[0,-2],[0,-2],[-2,-10],[-2,-4],[-3,-6],[4,-4],[1,3],[1,5],[1,7],[1,6],[4,9],[1,7],[1,6],[-1,5],[0,5],[-1,3],[-1,4],[1,6],[0,6],[1,6],[-1,3],[-1,3],[0,2],[0,4],[0,2],[-1,1],[0,2],[-3,4],[-1,4],[0,6],[-1,8],[0,4],[-1,2],[-5,0],[0,1],[0,3],[-1,6],[-1,5],[-1,3],[-6,0],[-1,3],[0,6],[0,3],[0,1],[-1,2],[-1,7],[-1,2],[-2,4],[-1,2],[0,1],[0,1],[0,3],[0,1],[1,2],[0,3],[1,1],[9,-7],[8,-6],[8,1],[3,4],[9,0],[1,-4],[1,-3],[11,11],[3,6],[2,2],[1,-2],[1,-5],[2,-2],[1,-4],[0,-2],[-1,-2],[0,-3],[7,3],[1,3],[1,0],[2,-4],[9,-4],[1,-3],[1,-4],[0,-3],[2,-2],[10,-4],[-1,6],[-1,2],[-2,0],[-4,5],[-1,2],[-4,6],[-6,1],[-1,3],[1,1],[0,1],[-5,5],[-9,1],[-3,8],[-2,1],[1,2],[3,3],[1,1],[2,9],[0,1],[-1,2],[1,3],[0,3],[0,3],[0,3],[0,2],[0,2],[1,3],[0,3],[-1,4],[1,0],[1,1],[1,2],[0,5],[-1,2],[-4,1],[-2,2],[-2,6],[0,3],[-1,2],[0,2],[0,1],[0,2],[1,1],[0,1],[0,2],[-1,1],[0,7],[2,13],[0,5],[1,3],[2,2],[3,-3],[2,1],[0,5],[0,3],[0,3],[0,2],[0,1],[0,2],[0,1],[-1,1],[-1,0],[1,6],[1,4],[6,14],[4,16],[6,11],[7,24],[0,4],[1,3],[2,2],[1,0],[0,-1],[1,-1],[0,-2],[0,-1],[0,-1],[1,1],[1,2],[0,3],[1,3],[2,4],[2,9],[2,9],[9,11],[1,2],[0,1],[0,6],[1,1],[7,17],[1,2],[2,-4],[-1,-2],[1,-3],[1,-2],[7,3],[4,9],[1,2],[2,6],[5,8],[1,0],[3,-2],[3,3],[1,-2],[0,-2],[-1,-6],[-1,-3],[1,-3],[1,-1],[10,10],[2,-1],[1,-9],[0,-3],[1,-5],[1,-6],[3,-6],[7,-5],[3,-7],[8,4],[7,3],[0,-2],[0,-2],[-1,-3],[0,-1],[1,-1],[1,-1],[0,-1],[0,-4],[1,-2],[-1,-6],[1,-4],[1,-5],[0,-1],[0,-2],[1,-1],[2,-2],[1,-2],[1,-1],[3,-4],[1,-4],[-2,-3],[-12,4],[-1,-1],[0,-3],[-1,-4],[-3,-9],[-1,-2],[-3,-2],[-1,-3],[2,-1],[1,-1],[-1,-4],[-1,-3],[-4,-2],[-1,-1],[-2,-5],[-1,-2],[-5,-1],[-3,-6],[-3,1],[-5,-3],[-2,-6],[-1,-2],[-4,2],[-5,-7],[-2,-3],[1,-2],[1,0],[1,2],[2,-2],[14,8],[0,-3],[1,-3],[1,-1],[6,-1],[0,1],[2,4],[1,1],[4,-6],[8,7],[5,-1],[1,1],[1,3],[0,1],[1,2],[0,1],[1,0],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-7],[-1,-3],[-3,-2],[-1,-4],[1,-1],[0,-2],[1,-2],[-2,-2],[-7,1],[-1,-9],[0,-4],[2,1],[1,1],[1,0],[1,-2],[1,0],[4,3],[3,0],[1,1],[6,14],[1,2],[3,-1],[2,-4],[3,-1],[0,-1],[1,-3],[1,-2],[2,0],[0,-1],[1,-2],[3,0],[0,1],[0,2],[-1,2],[-2,1],[0,2],[2,-1],[1,1],[1,4],[1,1],[4,-6],[1,-1],[-1,-3],[1,-3],[2,3],[1,1],[1,-1],[3,1],[0,-1],[0,-1],[-1,-1],[1,-3],[0,-3],[2,-2],[0,-2],[0,-3],[1,-2],[3,-3],[0,-2],[0,-3],[-1,-2],[-5,-4],[0,-1],[2,-2],[0,-2],[-1,-2],[-2,-1],[-1,-3],[2,0],[0,-2],[-2,-7],[0,-1],[-1,0],[-2,2],[0,-1],[-1,-2],[0,-3],[0,-6],[0,-2],[-1,-2],[-1,-2],[-1,-4],[-1,-2],[-2,0],[-1,2],[-1,1],[-1,-2],[2,-8],[0,-3],[0,-3],[-2,-8],[0,-2],[-1,-2],[0,-2],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[-1,-4],[0,-2],[-1,-1],[-1,0],[0,2],[0,2],[0,4],[-1,3],[-2,2],[-1,-1],[-1,-7],[0,-5],[1,-5],[-1,-4],[1,-2],[1,-2],[7,-1],[1,-1],[2,-5],[0,-1],[1,1],[1,3],[0,4],[1,2],[3,-1],[7,7],[9,-7],[7,7],[1,3],[0,2],[2,14],[1,3],[0,6],[-1,3],[-2,2],[1,7],[1,4],[-1,1],[1,5],[1,0],[3,-1],[3,1],[3,-3],[0,-2],[-1,-3],[6,5],[5,-2],[10,5],[4,-2],[4,7],[2,0],[0,-1],[0,-1],[0,-2],[1,0],[0,-1],[5,0],[0,-1],[0,-2],[1,-1],[3,-2],[2,1],[1,-1],[0,-2],[3,0],[2,-3],[9,15],[5,2],[2,-1],[0,-5],[1,-2],[1,-1],[4,2],[2,-1],[1,2],[-1,1],[0,1],[0,2],[1,3],[2,0],[2,-1],[1,-4],[-1,-2],[0,-2],[-1,-3],[1,-2],[0,-4],[1,-3],[1,2],[0,3],[1,2],[10,-6],[-1,-1],[-1,-1],[0,-2],[0,-1],[1,-1],[1,-3],[-1,0],[-1,-1],[-1,-1],[0,-3],[1,-4],[5,2],[4,-6],[10,-26],[1,-5],[-9,-27],[-1,-5],[0,3],[3,6],[1,1],[1,0],[1,5],[3,7],[2,5],[0,1],[1,-1],[11,-21],[1,-7],[1,0],[0,-2],[-2,-1],[1,-5],[1,-3],[0,-6],[-2,-3],[0,-2],[0,-1],[0,-2],[1,-1],[-1,-1],[-2,-1],[0,1],[-1,4],[-1,1],[-1,-2],[1,-3],[2,-5],[1,-3],[3,2],[1,-2],[1,-5],[1,-7],[-2,-14],[-1,-3],[-2,-5],[-2,0],[-1,-2],[1,-1],[10,5],[2,4],[0,2],[0,2],[0,5],[0,2],[-1,2],[0,3],[2,5],[2,-1],[3,-4],[1,-7],[0,3],[-1,4],[0,4],[1,2],[-1,2],[-3,6],[-3,2],[-2,3],[0,2],[0,5],[-1,2],[6,-1],[2,3],[1,0],[1,-5],[0,-2],[-2,-3],[1,-3],[0,-2],[2,2],[1,-1],[3,-15],[0,-4],[0,-4],[-1,-3],[0,-4],[0,-6],[1,-5],[1,-2],[1,-1],[0,-2],[-1,-7],[0,-2],[1,-1],[0,-1],[1,-1],[-1,-5],[-1,-1],[-1,0],[-1,-1],[1,-2],[1,0],[1,-2],[0,-2],[0,-2],[0,-2],[1,-2],[-1,-1],[0,-1],[0,-1],[1,-2],[-1,-3],[-1,0],[-1,0],[0,-2],[2,0],[2,2],[1,3],[0,2],[-1,2],[0,8],[1,0],[2,1],[1,-2],[0,-1],[1,6],[1,-1],[0,-3],[1,-3],[-1,-4],[2,-10],[-1,-8],[-1,-6],[0,-5],[0,-1],[0,-4],[-2,-9],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-3],[0,-2],[0,-2],[0,-10],[-1,-6],[-3,-7],[0,-7],[-4,0],[0,5],[3,9],[1,6],[1,4],[0,5],[-10,9],[-5,-2],[1,1],[1,1],[0,5],[0,2],[-8,16],[-8,16],[-6,1],[0,-2],[1,-3],[1,-1],[1,-1],[0,-1],[0,-2],[1,-2],[0,-5],[0,-2],[1,-1],[2,-2],[1,2],[1,-1],[2,-6],[2,-1],[0,-1],[0,-2],[1,-2],[0,-2],[0,-2],[0,-1],[0,-1],[-1,-3],[0,1],[-1,2],[0,1],[-1,-1],[1,0],[0,-1],[1,-2],[0,-4],[1,-3],[1,-2],[0,-5],[-1,-3],[-1,-3],[-1,-4],[1,-7],[2,-3],[3,0],[1,2],[0,3],[-1,5],[0,6],[1,2],[5,7],[2,1],[0,-2],[0,-4],[0,-1],[1,-2],[-1,-1],[0,-3],[0,-3],[-1,-3],[1,-6],[1,-4],[6,-7],[1,0],[2,3],[0,1],[1,-4],[0,-2],[0,-1],[0,-3],[0,-2],[1,-1],[0,-1],[0,-10],[0,-4],[-1,3],[-1,0],[-2,-1],[-1,-2],[0,-4],[1,-2],[2,0],[1,-4],[-1,-5],[-1,-7],[0,-2],[0,-3],[0,-2],[0,-1],[-2,-4],[-5,-21],[-6,-16],[-1,-5],[-1,-3],[-2,-5],[0,-3],[0,-4],[-3,-11],[-10,-18],[-2,0],[-9,-10],[-1,-6],[0,-7],[-1,-9],[-1,-2],[-1,-1],[-1,-2],[0,-3],[-1,-2],[0,-1],[0,-2],[1,-1],[-1,-3],[-1,-5],[-2,-2],[-1,-6],[-1,-1],[-8,8],[-1,-1],[-8,-21],[-1,-1],[-2,0],[-1,-2],[1,-2],[1,-2],[0,-2],[-3,-5],[-4,-3],[-2,-5],[-5,-9],[-1,-1],[-3,5],[-2,0],[-1,-5],[2,-1],[0,-1],[0,-4],[1,0],[-1,-8],[-2,-5],[-2,-7],[-6,1],[-1,-6],[-1,-3],[0,-2],[-3,-1],[0,-1],[-1,-1],[0,-1],[0,1],[0,1],[0,1],[-3,-2],[-1,2],[-2,-3],[2,-6],[2,-1],[6,5],[1,0],[0,-2],[0,-2],[-1,-15],[-1,-4],[0,-4],[-3,-9],[-5,-10],[-4,-10],[-8,-5],[-5,-10],[-1,-5],[-1,-6],[-1,-2],[-2,-2],[-2,-5],[-2,-11],[-1,-5],[-1,-2],[0,-3],[-1,-7],[0,-3],[-1,-1],[-1,0],[-2,4],[1,-3],[1,-6],[0,-3],[-1,-1],[-2,-4],[-9,-29],[0,2],[0,2],[1,1],[0,2],[-2,3],[-2,-1],[-5,-14],[-1,-1],[-2,4],[-8,2],[-1,0],[0,-3],[-1,0],[-4,0],[0,-1],[-1,-3],[-1,-8],[1,0],[0,-1],[1,-2],[0,-2],[-1,-6],[-5,-14],[-1,-4],[-1,-2],[0,-1],[0,-1],[0,-3],[-1,-13],[-1,-5],[-1,-3],[-3,-9],[-2,0],[-2,4],[-1,1],[-2,-3],[-5,2],[-5,-12],[0,-5],[0,-2],[0,-5],[0,-3],[0,-2],[-1,-5],[0,-7],[-4,-10],[0,-5],[0,-5],[0,-5],[-1,-5],[0,-5],[0,-3],[-2,-6],[-2,-10],[-1,-6],[-1,-2],[-2,-5],[-1,-3],[-4,-10],[-2,-7],[-6,-4],[0,-5],[1,3],[10,-2],[1,2],[2,6],[1,2],[1,1],[1,0],[2,11],[1,3],[1,2],[3,2],[0,2],[0,2],[1,6],[1,5],[1,1],[2,-3],[0,-7],[3,-4],[2,-1],[3,-5],[4,-1],[2,-3],[1,-6],[5,-1],[5,5],[2,3],[-1,4],[-2,2],[-3,-1],[-3,2],[-4,2],[-3,8],[-5,7],[-2,3],[-1,2],[0,2],[0,4],[-1,5],[0,4],[0,5],[0,4],[0,2],[0,3],[0,3],[0,4],[0,3],[1,2],[0,1],[1,0],[1,1],[0,3],[0,4],[1,2],[1,2],[11,-14],[2,2],[2,-1],[3,10],[9,-6],[4,3],[6,-1],[4,4],[3,12],[4,3],[1,0],[3,-6],[2,0],[2,3],[0,6],[-1,5],[-3,-1],[-1,1],[1,6],[4,4],[4,-2],[4,7],[7,3],[4,6],[4,3],[1,4],[0,4],[0,2],[-1,2],[0,2],[0,4],[0,2],[4,-8],[0,-4],[1,-1],[1,0],[0,3],[0,3],[0,3],[-1,1],[0,1],[-2,6],[-1,3],[-1,2],[-2,0],[0,1],[0,4],[0,1],[0,3],[0,2],[1,-1],[1,-3],[1,-2],[0,-1],[2,-4],[2,-8],[2,-4],[2,-1],[4,0],[4,5],[4,9],[3,2],[0,2],[1,5],[0,3],[-1,1],[-1,-1],[1,6],[1,1],[3,0],[5,5],[-1,2],[0,6],[2,2],[2,0],[10,9],[0,2]],[[7989,9091],[0,-2],[0,-1],[1,-2],[1,-2],[0,-3],[-2,-11],[-2,-3],[-3,-4],[-1,2],[0,3],[0,3],[1,2],[-2,2],[-5,-4],[-11,5],[-11,6],[1,6],[1,2],[12,4],[0,1],[1,0],[0,-3],[7,0],[1,0],[0,-1],[1,0],[0,1],[0,2],[1,1],[0,1],[1,1],[6,-1],[1,-2],[1,-3]],[[7607,9086],[-2,-1],[-1,1],[-2,2],[-2,1],[-1,4],[0,3],[2,3],[1,1],[1,0],[2,0],[0,-2],[1,-2],[1,-2],[1,-2],[0,-3],[-1,-3]],[[7964,9124],[0,-6],[0,-2],[-1,-1],[0,-1],[-1,1],[0,4],[-1,2],[-5,-7],[1,-1],[0,-1],[1,1],[0,-3],[-9,-12],[-1,0],[-2,3],[-2,1],[-1,2],[1,3],[0,6],[1,3],[1,1],[5,-3],[2,2],[1,5],[0,2],[1,8],[2,3],[3,0],[2,-2],[1,-3],[1,-1],[1,-2],[-1,-2]],[[7573,9319],[2,-4],[-6,1],[-7,8],[-2,4],[1,2],[6,-7],[6,-4]],[[7858,9364],[-3,-12],[1,-2],[0,-4],[-2,-6],[-1,-5],[1,-1],[0,-1],[1,-3],[-1,-5],[-2,-6],[-1,-4],[0,-2],[1,-2],[0,-2],[0,-3],[0,-2],[-2,-4],[0,-1],[0,-3],[0,-1],[0,-1],[0,-1],[-1,-3],[-1,-5],[-3,-17],[-1,-8],[0,-2],[0,-2],[-1,-2],[0,-2],[1,3],[3,3],[4,12],[0,1],[0,3],[1,2],[1,4],[1,5],[2,8],[1,12],[0,2],[1,3],[1,2],[5,-2],[4,7],[1,3],[0,2],[1,3],[3,-1],[2,3],[0,2],[0,1],[0,1],[0,2],[0,1],[0,1],[1,3],[1,1],[1,-1],[0,-1],[0,-2],[-1,-3],[0,-2],[2,0],[4,4],[0,-2],[5,-4],[0,-1],[0,-1],[1,-3],[0,-3],[0,-3],[-1,-2],[-1,-3],[2,-5],[1,-3],[0,-5],[0,-2],[9,-4],[2,-5],[1,-10],[0,-9],[0,-1],[1,0],[-1,-5],[-1,-1],[-2,0],[-1,-1],[8,-4],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[0,-3],[-1,-2],[-1,-2],[1,-1],[3,5],[1,1],[0,3],[0,3],[0,1],[-1,3],[0,2],[1,2],[4,1],[5,-3],[3,-9],[0,-1],[2,-14],[0,-2],[0,-4],[0,-2],[0,-3],[1,-3],[1,-8],[0,-4],[0,-4],[0,-5],[1,-1],[1,-1],[0,-3],[0,-5],[-1,-5],[0,-2],[0,-7],[-2,-4],[-2,-4],[-5,-16],[-7,-18],[-1,-1],[-1,1],[-2,5],[-2,-3],[-6,-1],[-2,-2],[-3,-1],[-1,-4],[-2,-1],[-2,-6],[-8,-7],[-11,-1],[-5,-2],[-3,3],[-2,-1],[-2,-8],[0,-1],[0,-1],[0,-1],[0,-3],[-1,-2],[-5,-5],[0,3],[0,3],[1,2],[-1,4],[0,1],[-2,2],[-10,-2],[-2,-4],[-2,-1],[-1,-3],[-1,0],[0,2],[0,3],[-1,1],[-10,-5],[-11,1],[-2,-3],[-4,-13],[-12,-17],[-3,-7],[-3,-2],[-9,-15],[-5,-3],[-5,4],[-3,-5],[-1,-1],[-2,4],[-2,0],[0,3],[1,2],[0,2],[0,3],[-1,1],[-1,1],[0,3],[-2,4],[0,1],[1,4],[2,2],[1,3],[2,7],[1,10],[0,5],[0,3],[2,4],[0,1],[4,16],[2,2],[4,15],[2,2],[0,1],[0,4],[-1,3],[0,3],[0,5],[0,3],[1,3],[1,2],[1,3],[0,3],[1,3],[0,3],[0,3],[0,3],[1,1],[0,1],[0,2],[0,2],[1,2],[2,9],[0,3],[0,3],[-1,2],[0,3],[0,2],[0,3],[0,2],[0,3],[2,7],[1,4],[0,1],[0,5],[0,2],[1,2],[0,1],[5,10],[1,0],[1,0],[2,-5],[8,-3],[0,-1],[0,-1],[0,-2],[1,-1],[1,2],[2,0],[0,2],[-1,5],[-1,1],[-5,1],[-1,1],[-1,4],[0,6],[0,2],[-1,2],[0,4],[0,1],[0,6],[1,5],[1,4],[0,4],[1,3],[0,2],[0,2],[0,2],[2,5],[0,3],[2,4],[1,1],[7,-7],[7,-8],[1,3],[-1,0],[-6,5],[-4,9],[-7,5],[1,4],[0,3],[1,7],[0,3],[0,1],[4,2],[-1,2],[-1,3],[1,1],[0,1],[0,1],[1,2],[0,3],[0,1],[3,9],[2,1],[4,-1],[1,4],[0,2],[0,2],[0,6],[0,1],[0,1],[-1,-1],[1,3],[0,1],[0,2],[0,1],[0,3],[1,3],[2,3],[4,1],[1,-1],[1,-9],[1,-4],[1,-1],[0,-2],[1,-2],[0,-2],[-1,-4],[0,-2],[1,-1],[2,1],[5,-4],[1,3],[0,1],[-3,7],[0,1],[0,2],[0,4],[0,7],[-1,3],[1,2],[0,1],[0,2],[0,1],[0,1],[0,2],[1,1],[0,2],[1,4],[2,3],[3,-1],[2,-3],[3,0],[4,-7],[4,-2],[1,-2],[0,-6],[1,-2],[5,-4],[1,-4],[-1,-2],[-1,-3],[-1,-2],[-3,0],[-1,-1]],[[7582,9399],[-1,-4],[-1,-1],[-3,0],[1,-2],[-1,-2],[-4,1],[-1,4],[-1,1],[0,1],[0,1],[0,1],[2,0],[4,5],[3,-1],[2,-4]],[[7565,9401],[-2,-3],[-1,0],[0,-2],[1,-2],[0,-2],[-1,-2],[-13,9],[4,1],[3,-2],[1,0],[1,2],[2,6],[1,2],[2,-1],[5,-5],[-1,-1],[-2,0]],[[7537,9415],[5,-8],[-2,2],[-2,0],[-1,2],[0,1],[0,2],[-1,1],[-4,8],[-10,10],[-1,2],[10,-7],[6,-13]],[[7118,9455],[-3,-2],[-1,0],[2,-7],[2,-1],[4,3],[0,5],[2,1],[10,-9],[10,-9],[11,-9],[0,-3],[0,-3],[0,-2],[-1,-1],[-12,-2],[-12,-2],[-2,2],[1,2],[1,4],[0,2],[-1,2],[-1,5],[-1,0],[0,-1],[-1,-2],[-11,8],[-2,9],[-1,6],[0,5],[1,3],[2,0],[2,-1],[1,-3]],[[7779,9465],[4,-2],[1,1],[1,-1],[1,-3],[-4,-2],[-3,-5],[-2,-9],[-2,-3],[-1,4],[1,2],[0,2],[0,2],[0,4],[0,2],[0,2],[0,3],[1,3],[1,3],[1,1],[2,-1],[-1,-3]],[[7615,9538],[-5,-5],[-1,0],[-1,3],[1,6],[0,2],[1,2],[4,1],[3,5],[1,0],[1,-3],[-1,-3],[-1,-1],[-2,-1],[0,-1],[0,-1],[0,-1],[0,-3]],[[7524,9560],[10,-4],[10,-5],[5,7],[7,-7],[11,-1],[6,-12],[13,-6],[14,-6],[3,-2],[1,-2],[1,-5],[-1,-6],[-1,-4],[-2,-6],[-6,-8],[-2,-6],[-3,-2],[-6,-13],[-12,-10],[-10,0],[-9,1],[0,1],[1,1],[0,2],[-1,3],[-1,0],[1,2],[6,5],[2,-1],[2,-5],[0,1],[1,2],[1,3],[-1,5],[-1,4],[-5,9],[-10,4],[-9,3],[-10,3],[0,3],[0,4],[0,1],[3,4],[2,1],[1,4],[-2,6],[-4,2],[-1,4],[3,3],[1,4],[0,4],[0,4],[-1,3],[-1,2],[-1,0],[-2,2],[-3,0],[-1,1],[1,3]],[[7715,9574],[1,0],[0,1],[1,-1],[0,-1],[1,-4],[1,-2],[1,-6],[3,-2],[1,-3],[-1,-6],[-1,-4],[-2,-3],[-2,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[-2,-6],[0,-2],[2,-1],[-1,-5],[-1,-4],[0,-3],[-4,-7],[-1,0],[1,-4],[1,-3],[-1,-2],[-2,1],[-1,-4],[1,-2],[1,-1],[1,-2],[0,-3],[-1,-2],[-3,-3],[-1,-3],[-6,-6],[-1,-3],[1,-2],[1,0],[0,1],[2,4],[10,11],[1,2],[0,6],[0,2],[0,1],[1,1],[2,6],[1,1],[1,1],[0,3],[-1,5],[0,4],[1,2],[5,1],[5,-8],[2,0],[1,1],[1,3],[1,4],[0,1],[0,1],[-1,1],[1,2],[1,2],[0,1],[0,2],[0,2],[2,4],[1,1],[0,2],[-4,4],[0,4],[1,5],[1,4],[1,1],[5,-5],[13,3],[1,-3],[-1,-2],[-1,-2],[0,-1],[1,-1],[1,-1],[2,3],[1,-2],[1,-3],[-1,-3],[1,-2],[5,-3],[-1,-2],[0,-1],[-1,0],[1,-5],[4,0],[2,-3],[2,-4],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,-4],[1,-2],[1,-1],[1,-2],[2,-5],[-1,-5],[1,-3],[1,-2],[-1,-4],[-1,-3],[-2,-2],[-2,0],[-1,-2],[1,-4],[1,-3],[0,-4],[-1,-4],[-2,-5],[0,-3],[0,-3],[0,-2],[-1,-4],[0,-6],[-1,-3],[0,-1],[0,-4],[2,-4],[0,-4],[0,-2],[-2,-6],[-1,-3],[0,-4],[1,-4],[1,-6],[-1,-3],[0,-2],[0,-2],[0,-2],[1,-4],[0,-2],[-2,-9],[0,-2],[-2,-4],[0,-1],[0,-3],[-1,-1],[0,-1],[-2,-1],[-1,0],[0,-3],[0,-2],[-1,4],[-8,11],[-1,0],[-2,-3],[1,-4],[7,-13],[7,-13],[1,-4],[-1,-2],[0,-1],[-2,0],[1,-5],[8,-6],[1,-2],[0,-1],[0,-1],[1,-8],[0,-3],[-1,-4],[1,-6],[2,-7],[-1,-6],[-1,-4],[-6,-8],[-1,-7],[-2,-3],[-3,-7],[-2,-2],[-7,2],[-6,-5],[-9,2],[-1,-3],[1,-2],[0,-1],[0,-1],[0,-1],[-7,0],[-3,6],[-4,0],[1,2],[-1,2],[-4,-4],[-5,0],[0,2],[-1,1],[-2,2],[-6,-3],[-1,3],[1,2],[3,-2],[1,4],[-1,2],[-8,0],[-1,3],[0,1],[-1,2],[-2,2],[-6,14],[-1,-1],[-1,2],[1,1],[0,4],[-2,5],[-7,1],[-7,1],[0,1],[1,0],[1,2],[0,3],[-12,-7],[-2,2],[-5,0],[-1,3],[0,2],[0,2],[0,3],[0,2],[1,4],[0,1],[-1,2],[-1,0],[-1,2],[0,3],[-1,1],[0,2],[-2,1],[-1,-2],[-4,-8],[-2,-2],[-1,-3],[-3,-4],[-1,1],[-1,4],[0,2],[-1,1],[-3,-1],[-1,0],[-5,15],[0,3],[0,3],[0,4],[0,3],[0,6],[-8,8],[0,3],[-1,4],[0,5],[1,2],[0,6],[0,2],[0,3],[-1,3],[0,3],[0,3],[0,2],[0,2],[0,3],[1,6],[0,3],[-1,10],[-2,2],[-5,-3],[0,-4],[1,-3],[-1,-4],[-2,-1],[-7,5],[-1,3],[2,3],[4,0],[2,6],[-1,2],[-3,3],[-1,1],[-1,2],[0,3],[-1,1],[-1,-1],[2,4],[1,2],[1,8],[1,2],[-3,0],[-1,-1],[0,-1],[-1,-3],[0,-2],[0,-3],[-1,-2],[-2,-1],[-1,-1],[-1,-2],[0,-2],[-1,-2],[-1,-2],[0,-4],[0,-1],[3,-2],[-1,0],[-2,-3],[-2,-1],[0,-3],[-2,-3],[-2,-2],[-2,2],[-1,5],[1,4],[3,1],[1,3],[-1,5],[-3,2],[-2,-2],[-1,0],[-1,2],[1,1],[-5,6],[1,4],[2,0],[5,-7],[3,2],[8,14],[2,1],[0,1],[2,5],[2,3],[0,2],[1,7],[1,6],[9,16],[7,5],[8,5],[-1,3],[-1,1],[-2,2],[-1,3],[-3,2],[-2,2],[0,4],[-1,2],[-1,1],[-1,0],[0,4],[2,3],[2,2],[0,3],[-1,2],[-1,3],[1,2],[1,0],[2,-2],[8,10],[2,5],[0,1],[-1,2],[2,1],[1,2],[1,5],[0,6],[1,2],[1,-1],[3,-7],[0,-4],[1,-2],[2,-2],[0,-4],[1,-1],[2,2],[1,1],[0,2],[0,7],[0,2],[1,2],[0,2],[3,4],[9,0],[8,0],[0,-1],[0,-3],[0,-1],[2,0],[0,1],[2,5],[0,1],[1,-1],[1,-3],[1,-1],[4,2],[1,2],[1,4],[0,2],[1,0],[1,-1],[1,-3],[1,-1],[5,4],[7,5],[12,5],[4,-6],[1,-4],[0,-1]],[[7225,9777],[9,-4],[-1,-10],[0,-3],[-1,-3],[-1,-3],[-12,-5],[-11,-4],[-11,-5],[-3,5],[-1,4],[0,5],[3,9],[1,8],[1,2],[1,3],[1,2],[0,-2],[14,7],[2,-1],[9,-5]],[[7536,9808],[-4,-6],[-11,4],[-11,4],[-11,4],[0,1],[-2,5],[0,4],[0,3],[0,2],[-1,2],[2,1],[2,7],[1,0],[0,2],[13,-2],[13,-2],[13,-2],[2,-9],[1,-3],[0,-2],[0,-1],[-6,-7],[0,-1],[-1,-4]],[[7665,9838],[0,-1],[5,1],[1,-3],[-1,-3],[1,-2],[2,-1],[1,-2],[0,-4],[0,-1],[7,-15],[0,-3],[0,-5],[0,-4],[1,-2],[4,-3],[0,-2],[-2,-1],[0,-2],[0,-2],[1,-1],[0,-1],[1,1],[0,-2],[1,-4],[3,-2],[1,-3],[1,-5],[1,-1],[12,-14],[0,-3],[4,1],[3,-3],[3,0],[1,-3],[-1,-4],[-1,-1],[1,0],[1,-2],[2,-5],[-1,-3],[-2,-2],[-1,-2],[2,-2],[3,2],[2,-5],[-1,-5],[-1,-2],[-3,1],[-1,-1],[0,-1],[-1,-4],[0,-2],[-1,1],[-1,1],[-3,1],[-5,-6],[-1,1],[-1,3],[0,1],[-5,-1],[1,-3],[1,-1],[0,-1],[-1,-6],[-2,-5],[-1,-4],[0,-3],[0,-1],[0,-1],[0,-1],[-1,-3],[0,-2],[0,-1],[0,-3],[3,-7],[1,-2],[0,-4],[0,-4],[1,-2],[0,-1],[0,-1],[0,-2],[1,-3],[0,-2],[-1,-3],[0,-2],[0,-5],[0,-4],[1,-2],[1,-1],[2,0],[2,-4],[0,-1],[0,-3],[0,-2],[0,-1],[-2,-2],[-2,-6],[-1,-1],[0,-2],[0,-1],[-1,-4],[-12,-4],[-12,-5],[-12,-5],[-12,-4],[-12,-5],[-5,3],[-1,-2],[-2,-5],[-1,-1],[-5,3],[-1,-1],[0,-3],[0,-2],[-1,-3],[-8,-13],[-9,-5],[-3,-6],[-1,1],[-10,12],[-10,12],[0,1],[0,2],[-13,12],[-3,-2],[-9,5],[0,3],[2,0],[0,1],[1,1],[0,2],[0,1],[0,1],[1,0],[0,-1],[8,-5],[9,-6],[1,3],[0,4],[-3,1],[-2,3],[-7,2],[-2,4],[-2,1],[-1,2],[0,2],[-1,4],[-3,3],[0,3],[8,1],[8,2],[1,-1],[2,0],[1,-1],[13,-6],[0,2],[0,3],[-1,2],[-1,3],[-1,2],[-11,-2],[-2,1],[-11,-2],[-11,-2],[-11,-2],[1,2],[1,2],[1,3],[-4,-3],[-2,1],[1,10],[2,2],[2,0],[2,3],[0,1],[-1,3],[9,4],[2,-3],[1,-1],[1,1],[0,2],[0,1],[12,-6],[2,3],[-7,4],[-8,3],[-2,5],[-2,2],[-1,3],[1,2],[9,10],[13,1],[1,1],[1,3],[0,2],[0,1],[0,2],[0,2],[0,5],[1,2],[-1,1],[0,2],[-1,0],[1,1],[0,2],[0,2],[1,8],[1,7],[10,22],[1,3],[0,4],[-10,-9],[-4,4],[-1,-1],[-1,-4],[-4,-1],[-1,2],[0,4],[0,4],[1,2],[2,2],[0,1],[1,5],[1,2],[0,2],[1,0],[0,3],[-1,3],[0,1],[1,2],[2,1],[2,5],[7,2],[1,3],[0,4],[0,2],[-2,4],[0,4],[-1,1],[0,1],[12,12],[5,-3],[0,1],[0,1],[1,2],[0,1],[10,4],[3,6],[6,5],[5,-3],[9,10],[4,11],[2,2],[2,5],[10,1],[0,2],[-13,3],[-1,2],[2,6],[9,2],[10,2],[1,-3],[2,-6],[3,-3],[0,-1],[-1,-3],[0,-2],[0,-3]],[[7756,2892],[0,-3],[0,-2],[-1,-3],[0,-2],[0,-3],[0,-1],[0,-2],[0,-6],[0,-1],[0,-1],[0,-1],[-1,0],[0,5],[-1,0],[-1,2],[-2,2],[0,1],[0,2],[-1,1],[0,3],[0,1],[0,1],[-1,0],[0,1],[0,-1],[-1,-2],[-2,-18],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[1,-1],[1,2],[1,-1],[0,-1],[-1,-2],[0,-5],[0,-3],[0,-2],[1,-5],[0,-1],[0,-3],[-1,-4],[0,-4],[0,-2],[0,-2],[0,-3],[-1,0],[0,1],[0,1],[0,-2],[-1,-3],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-2],[0,-1],[0,-3],[-1,-4],[1,-2],[0,-1],[0,-3],[0,-1],[0,-1],[0,-1],[0,-1],[0,-4],[0,-1],[-1,-1],[0,-1],[-1,0],[0,1],[0,-1],[-1,-2],[0,-2],[0,-2],[1,-1],[0,-1],[-1,-2],[0,-1],[-1,-6],[0,-1],[-1,-1],[0,-1],[0,-2],[1,-9],[0,-2],[0,-2],[0,-4],[0,-4],[-1,-5],[0,-1],[0,-4],[0,-4],[0,-2],[1,-2],[0,2],[0,1],[0,2],[1,2],[0,1],[0,1],[1,0],[0,-1],[1,-2],[0,-1],[1,0],[0,-1],[0,-3],[0,-1],[0,-2],[0,-3],[0,-7],[0,-1],[0,-1],[0,-2],[0,-1],[1,0],[1,3],[1,1],[0,-1],[0,-1],[0,-1],[0,-1],[1,-2],[0,-6],[0,-2],[0,-1],[-1,-4]],[[7747,2688],[0,1],[-1,0],[-1,-5],[0,-5],[0,-8],[0,-4],[0,-3],[-1,-8],[-1,-4],[0,-12],[-2,-21],[-1,-6],[0,-5],[-1,-5],[-6,-8],[-2,-7],[-2,-16],[0,-4],[-1,-9],[0,-4],[0,-6],[0,-11],[0,-5],[0,-5],[-3,3],[-2,-4],[0,-1],[0,-2],[-1,-6],[-1,-16],[-1,-3],[0,-6],[-1,-3],[0,-5],[0,-3],[0,-9],[1,-5],[-1,-4],[0,-11],[0,-4],[-1,-8],[0,-8],[-2,-21],[0,-8],[1,-7],[1,-6],[1,-5],[1,-2],[0,-3],[0,-3],[1,-4],[0,-3],[-1,-10],[0,-5],[1,-13],[1,-20],[1,-4],[0,-4],[2,-9],[1,-2],[2,-1],[0,-2],[1,-3],[1,-5],[0,-6],[0,-8],[-1,-15],[0,-8],[0,-8],[0,-4],[0,-5],[0,-4],[-5,-54],[-1,-10],[-4,-12],[-1,-5],[-1,-6],[0,-1],[-1,2],[-1,6],[0,1],[-1,-1],[-2,-7],[-3,-3],[0,-2],[1,-3],[0,-4],[0,-8],[-1,-6],[-2,-8],[-2,-10],[-2,-4],[-1,-2],[-1,0],[-2,2],[-1,4],[-1,6],[0,5],[-1,2],[0,1],[-1,2],[0,2],[-1,10],[-1,6],[-8,6],[0,-1],[-2,-9],[0,-2],[-1,-1],[0,1],[-1,4],[0,5],[-1,4],[0,1],[-1,0],[-1,-2],[-1,-5],[-1,-1],[-1,2],[-1,7],[-2,6],[-4,11],[0,1],[-1,0],[0,-1],[-1,-2],[0,-2],[0,-3],[-1,-3],[-1,-4],[0,2],[-1,5],[0,7],[-1,5],[-1,2],[-1,-3],[-1,-4],[-1,-9],[-1,-3],[-3,-2],[0,-2],[-1,-7],[0,-3],[-1,0],[-1,1],[0,1],[0,2],[-1,3],[-1,6],[-1,2],[-8,-1],[-1,5],[0,1],[-1,6],[0,8],[-1,4],[-1,1],[-5,-2],[-3,-5],[0,-1],[-1,1],[0,3],[-1,7],[0,3],[-1,7],[-1,3],[0,3],[0,4],[0,3],[0,2],[-3,6],[0,4],[-1,6],[0,6],[0,13],[-1,25],[0,6],[-1,16],[0,9],[0,4],[0,3],[-1,2],[-10,2],[-10,2],[-3,6],[-6,-2],[-2,0],[-1,0],[-1,3],[-1,1],[0,2],[0,3],[-1,5],[0,9],[0,12],[0,9],[-2,3],[-1,1],[-2,-2],[-1,-4],[0,-4],[0,-8],[-1,-4],[0,-3],[-1,-3],[-1,-1],[0,-1],[-1,4],[-2,13],[-1,5],[-1,2],[-1,4],[-1,6],[0,10],[-1,3],[0,1],[-1,-1],[0,-2],[-1,-3],[0,-4],[0,-4],[0,-5],[-1,-17],[-3,-8],[-2,-1],[-5,5],[-2,-1],[-2,-4],[-2,-7],[-1,-4],[-1,-15],[-1,-4],[-3,-7],[0,-4],[-1,-5],[0,-5],[0,-4],[-1,-2],[-3,3],[-1,0],[-2,-8],[-1,-2],[-4,0],[-1,-3],[0,-4],[-2,-14],[-1,-5],[-3,-3],[-1,-4],[0,-2],[0,-7],[0,-4],[0,-3],[-1,-2],[0,-2],[-1,-1],[-1,0],[-2,3],[-2,-1],[-1,-6],[0,-2],[-2,0],[0,-1],[-4,-15],[-1,-3],[-3,-5],[-1,-3],[-1,-5],[0,-6],[0,-6],[0,-2],[-1,-7],[-1,-4],[-1,-2],[-5,0],[-1,-3],[-1,-5]],[[8213,3878],[0,-2],[-1,-1],[-1,-5],[-1,-2],[0,-5],[0,-4],[0,-5],[0,-6],[0,-9],[0,-5],[0,-5],[0,-4],[-1,-4],[0,-3],[-2,-3],[-1,-2],[0,-3],[0,-12],[1,-5],[4,-1],[1,-3],[0,-1],[1,-1],[0,-3],[-1,-7],[0,-2],[0,-1],[1,-4],[0,-1],[0,-2],[0,-1],[2,-5],[0,-4],[1,-2],[0,-2],[0,-3],[0,-2],[-1,0],[-1,2],[-1,0],[0,-1],[0,-4],[0,-6],[0,-1],[0,-4],[0,-3],[0,-2],[0,-5],[0,-3],[0,-4],[1,-10],[1,-12],[0,-4],[1,-2],[0,-1],[1,-1],[0,-2],[0,-2],[-1,-1],[0,-1],[-1,-2],[0,-4],[0,-5],[1,-3],[-1,-3],[0,-2],[0,-2],[0,-2],[0,-4],[1,-8],[0,-4],[1,-4],[1,-2],[0,-4],[1,-4],[0,-6],[-1,-4],[-1,-7],[0,-5],[0,-6],[1,-11],[0,-4],[-1,-12],[0,-5],[1,-2],[1,-1],[0,-2],[0,-2],[0,-3],[0,-2],[0,-2],[-1,-1],[0,-4],[1,-1],[0,-1],[0,-3],[0,-2],[-1,-5],[0,-3],[0,-5],[0,-5],[1,-4],[1,-2],[0,-2],[0,-2],[1,-2],[-1,-3],[0,-1],[0,-1],[0,-1],[0,-4],[0,-1],[0,-1],[0,-2],[0,-1],[0,-4],[1,-1],[0,1],[2,4],[1,-7],[1,-3],[1,4],[6,4],[0,-1],[1,-4],[0,-4],[0,-4],[1,-2],[0,-1],[0,-3],[0,-2],[0,-1],[0,-2],[0,-2],[0,-2],[0,-6],[0,-3],[1,-1],[1,-2],[1,-8],[1,-1],[0,-3],[0,-2],[1,-1],[0,-2],[1,-2],[1,-5],[1,-1],[1,1],[0,-1],[0,-2],[1,-6],[0,-1],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[1,-4],[0,-3],[0,-3],[-1,-2],[0,-3],[0,-2],[1,0],[1,0],[0,-2],[0,-3],[0,-2],[0,-5],[0,-7],[0,-5],[1,-5],[0,-11],[0,-4],[-1,-3],[0,-2],[-1,-3],[1,-9],[0,-5],[-1,-2],[0,-1],[-1,-6],[-1,-2],[0,-3],[0,-5],[0,-4],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-4],[0,-2],[-1,-3],[1,-2],[0,-2],[-1,-2],[0,-3],[-1,-1],[-2,-1],[-1,-2],[0,1],[-1,0],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-2],[0,-1],[-1,0],[0,1],[0,2],[-1,1],[0,1],[-7,5],[-2,-1],[-1,-3],[0,-1],[-1,0],[-1,-1],[0,-2],[-1,-1],[-1,1],[-1,-1],[-1,-3],[-2,-2],[0,-2],[-5,-7],[0,-1],[-1,-6],[0,-1],[-1,0],[0,-1],[-1,-1],[0,-1],[0,-2],[0,-6],[0,-3],[0,-2],[0,-2],[-1,-2],[-4,-11],[0,-4],[-1,-1],[-1,-1],[0,-1],[0,-1],[0,-4],[-1,-2],[0,-1],[-1,-1],[0,-2],[-1,-7],[0,-5],[0,-3],[-2,-4],[0,-3],[0,-2],[-1,-1],[-1,-1],[0,-2],[0,-2],[1,-3],[-1,-2],[0,-3],[-1,-1],[0,-1],[0,-2],[0,-1],[0,-3],[-1,-1],[0,-1],[-1,-4],[-1,-3],[0,-2],[-1,-1],[-1,0],[0,2],[-1,0],[0,-3],[0,-1],[0,-1],[-1,-1],[0,-1],[-1,-1],[-1,-2],[-1,-2],[0,1],[0,1],[0,-1],[-1,-1],[0,-2],[0,-7],[-2,-11],[0,-2],[-1,-1],[-1,-1],[-1,2],[-1,0],[0,-2],[0,-6],[-1,-4],[0,-2],[-1,-2],[-2,-1],[-1,-2],[-1,5],[0,1],[0,-1],[-1,-4],[0,-2],[-1,-2],[-2,-1],[-1,-2],[0,-3],[0,-4],[1,-3],[-1,-5],[0,-1],[-1,1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,1],[-1,-1],[0,-1],[0,-1],[-1,-2],[0,-3],[0,-1],[0,-2],[1,-2],[0,-2],[1,-10],[1,-2],[0,-2],[0,-2],[0,-5],[0,-4],[-1,-1],[0,-1],[0,-1],[0,-2],[-1,-4],[0,-2],[0,-2],[0,-2],[0,-3],[0,-6],[0,-3],[0,-2],[1,-1],[1,-4],[0,-1],[1,0],[0,-1],[0,-1],[1,-3],[0,-4],[0,-1],[0,-2],[0,-2],[-1,-2],[0,-2],[0,-2],[0,-1],[1,-3],[0,-5],[0,-1],[1,0],[1,4],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[1,-2],[0,-2],[1,-1],[1,1],[1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-5],[0,-4],[-1,-1],[0,-2],[0,-2],[0,-2],[-1,-2],[0,-2],[0,-2],[1,-1],[0,1],[0,1],[0,1],[0,1],[1,-1],[0,-1],[0,-1],[0,1],[1,4],[0,1],[1,0],[0,-1],[0,-1],[0,-2],[2,-4],[0,-4],[-1,-3],[0,-3],[0,-2],[0,-2],[0,-2],[0,-2],[0,-1],[0,-2],[1,-3],[-1,-1],[0,-1],[-1,1],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-4],[1,-1],[0,-2],[-1,-2],[0,-4],[0,-2],[0,-1],[0,-3],[0,-2],[0,-3],[0,-4],[0,-2],[0,-1],[0,-1],[0,-1],[-1,-8],[-2,-12],[0,-6],[-1,-3],[0,-2],[-1,-2],[-1,-2],[0,-1],[-1,-8],[0,-1],[-1,0],[-1,0],[-1,-5],[0,-1],[-1,0],[0,-1],[-1,-3],[0,-1],[0,-1],[-1,0],[0,-2],[-1,-5],[-1,-4],[-1,-2],[0,-1],[0,-4],[0,-1],[0,-1],[-1,-2],[-1,-7],[0,-2],[0,1],[0,1],[-1,5],[0,1],[-1,-3],[-2,-1],[-4,0],[0,1],[-1,0],[0,-1],[-3,-7],[0,-1],[0,-1],[-1,-6],[-4,-16],[-1,-2],[-1,0],[0,2],[-1,0],[-1,-1],[-1,-1],[-1,1],[-1,-3],[-1,1],[-1,5],[-1,0],[0,1],[-1,-2],[-1,0],[0,-1],[0,-3],[-1,-5],[0,-1],[0,-2],[0,-4],[0,-3],[0,-1],[0,-2],[0,-1],[0,-2],[-1,-1],[0,-2],[-1,-1],[0,-5],[-1,1],[-2,8],[-1,1],[-1,-2],[-1,0],[0,1],[0,2],[0,2],[0,3],[0,1],[-1,2],[0,1],[0,2],[-1,0],[0,-1],[0,-3],[0,-1],[-1,1],[-1,-1],[0,-1],[0,-1],[0,-4],[0,-1],[-1,-2],[0,-2],[0,-1],[0,-2],[-1,-1],[0,-1],[-1,-1],[0,-2],[-1,-1],[0,-1],[0,-3],[0,-1],[0,-2],[0,-1],[0,-2],[-1,-2],[0,-2],[0,-1],[0,-1],[0,-1],[-2,-7],[0,-3],[-2,-15],[0,-5],[-1,-7],[-1,-3],[0,-1],[-1,-1],[0,-2],[-1,-5],[0,-1],[0,-1],[-1,-2],[-1,-2],[-1,-4],[0,-2],[-1,-1],[-1,0],[0,-2],[-1,0],[-1,-2],[-4,-9],[-1,1],[0,-1],[-1,-1],[-2,-6],[-1,-3],[0,-3],[0,-1],[0,-3],[0,-1],[0,-5],[0,-1],[-1,-1],[-1,0],[0,-1],[-2,-9],[-1,-3],[0,-1],[0,-1],[0,-4],[1,-2],[0,-1],[0,-3],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-1],[-1,-1],[0,-2],[-1,-1],[0,-2],[0,-1],[0,-2],[0,-3],[-1,-6],[-1,-2],[0,-1],[-1,1],[-1,0],[-2,-3],[0,1],[-1,3],[0,1],[-3,2],[0,1],[0,1],[0,2],[0,6],[0,2],[0,1],[-1,0],[-1,3],[-1,1],[-1,0],[-1,0],[-4,3],[0,1],[-1,1],[0,1],[0,-1],[-1,-1],[0,-1],[-1,0],[-1,-5],[0,-3],[-1,-3],[-2,-5],[0,-3],[-1,-2],[-2,-7],[-1,-3],[-2,-7],[-2,-2],[-3,-5],[0,-3],[-1,-3],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[-1,-2],[0,1],[0,1],[0,1],[0,2],[0,1],[0,2],[-1,1],[0,-1],[-1,-3],[-1,-2],[-1,8],[-1,7],[-1,3],[0,3],[0,1],[0,1],[0,1],[-1,3],[0,1],[0,1],[0,1],[0,1],[0,2],[-1,2],[-1,-1],[0,-1],[0,-4],[0,-4],[-1,-1],[0,-1],[-1,-4],[-1,0],[-2,1],[-1,0],[0,3],[-1,0],[-1,-2],[0,1],[0,1],[-1,1],[0,3],[-1,3],[0,1],[-1,0],[0,1],[0,1],[0,1],[0,1],[-1,1],[0,-1],[0,-1],[-1,-1],[-1,-6],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-2],[0,-7],[0,-3],[-1,-6],[0,-3],[-1,-2],[0,-1],[-1,-4],[-1,-2],[0,-2],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[1,-1],[0,-2],[0,-2],[1,0],[0,-1],[1,-1],[0,-3],[0,-2],[0,-1],[2,-8],[1,-2],[0,-1],[0,-2],[0,-4],[0,-2],[0,-2],[0,-2],[-1,-5],[-1,-4],[0,-1],[0,-1],[0,-1],[-4,0],[0,-2],[-1,-9],[0,-8],[-1,-4],[0,-12],[0,-3],[-1,-3],[0,-11],[0,-1],[0,-1],[0,-1],[-1,-1],[-1,-2],[-1,-5],[-1,-6],[-1,-2],[0,-2],[0,-1],[0,-1],[0,-1],[0,-3],[1,-6],[1,-4],[0,-3],[1,-5],[0,-1],[2,0],[0,-1],[3,-8],[0,-1],[1,-1],[1,0],[0,-2],[0,-1],[1,-1],[0,-1],[1,-5],[1,-6],[0,-3],[0,-1],[1,-1],[0,1],[1,-1],[1,1],[0,1],[1,0],[0,-4],[0,-6],[0,-5],[-1,-2],[0,-1],[-1,-2],[0,1],[-1,2],[-2,-4],[-1,0],[0,1],[0,1],[-1,1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[-1,-1],[-1,-6],[-1,-1],[-1,2],[-1,0],[-1,0],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[-1,-2],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[-1,1],[0,2],[-1,1],[0,1],[0,2],[0,1],[-1,-3],[-1,0],[0,-1],[0,-1],[0,-3],[0,-1],[0,-3],[-1,-5],[-1,-6],[0,-6],[0,-2],[0,-2],[0,-3],[-1,-1],[0,-1],[0,1],[0,2],[-1,1],[-1,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-8],[0,-1],[0,-1],[0,-1],[1,-2],[0,-1],[0,-1],[0,-2],[0,-5],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[-1,-2],[-1,-10],[0,-2]],[[7992,2158],[0,1],[-10,1],[-2,8],[-1,-1],[0,-2],[-1,0],[-1,2],[-1,3],[-1,8],[-1,2],[-1,2],[0,2],[0,2],[0,6],[-1,3],[0,1],[0,4],[-1,2],[0,10],[0,2],[-1,3],[-4,20],[-2,4],[-2,5],[-3,2],[-3,-2],[-6,-9],[-1,2],[-1,5],[-1,2],[-2,2],[0,1],[0,2],[0,2],[0,2],[0,2],[-2,7],[-1,1],[-1,0],[-1,2],[-2,3],[0,1],[-4,-2],[-9,14],[-1,-1],[-1,-4],[-3,-15],[-1,-2],[-1,-1],[-1,0],[-3,2],[-1,-1],[0,-1],[-2,-6],[-1,-2],[-3,0],[0,-1],[-1,-6],[-1,-2],[-1,-1],[-3,1],[-2,-4],[0,-2],[-1,-5],[-1,-4],[-1,-6],[-1,-6],[-1,-9],[-3,-5],[-2,-2],[-2,2],[-3,9],[-2,0],[-3,-13],[-2,0],[-2,5],[-2,7],[0,3],[-1,1],[-3,-2],[-1,2],[-1,4],[0,8],[-1,13],[-2,3],[-4,-4],[-2,2],[-1,3],[-3,15],[-4,6],[-1,4],[-2,21],[-1,6],[-3,2],[-1,3],[-1,4],[-1,7],[0,7],[0,7],[1,7],[0,5],[1,5],[-1,8],[-2,7],[0,5],[-1,7],[0,7],[1,7],[0,7],[0,6],[0,3],[0,4],[0,3],[0,4],[-1,6],[-1,10],[0,9],[0,10],[0,8],[0,4],[0,4],[-1,5],[0,2],[0,3],[2,11],[0,5],[0,4],[-3,10],[0,3],[-1,2],[-2,1],[-1,1],[-1,5],[0,1],[-2,-1],[-1,2],[-2,6],[0,1],[-1,1],[-1,-1],[-2,5],[-2,1],[-1,0],[-2,-6],[-1,0],[-1,3],[-1,8],[-2,5],[-1,2],[-1,0],[-1,1],[-1,7],[-12,34],[-2,3],[-2,5],[-8,0],[-3,3],[-3,-2],[-2,5],[-2,10],[-2,11],[-2,11],[-1,2],[-3,1],[-7,10],[-1,2],[0,4],[-1,5],[0,4],[-1,3],[-5,10],[-1,3],[0,5],[-1,8],[-1,3]],[[8636,3562],[1,-2],[0,-2],[1,-4],[0,-1],[1,-2],[1,-4],[1,-1],[0,1],[2,5],[1,0],[2,-5],[1,0],[1,1],[1,1],[0,1],[0,2],[0,1],[1,0],[0,1],[1,0],[0,-2],[0,-1],[0,-1],[0,-1],[3,-3],[0,1],[0,1],[0,3],[0,2],[1,1],[0,1],[0,1],[1,-1],[3,-5],[0,-1],[1,-1],[0,-1],[0,-1],[3,3],[0,1],[0,1],[0,1],[1,1],[0,2],[0,1],[1,1],[3,-4],[1,0],[0,2],[1,1],[0,1],[0,1],[0,1],[1,2],[0,1],[1,1],[0,1],[0,3],[0,1],[2,-2],[3,2],[2,-6],[1,0],[1,0],[0,-1],[0,-3],[0,-2],[0,-2],[1,-5],[2,-16],[-1,-8],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[1,-2],[2,-9],[0,-2],[0,-4],[1,-2],[0,-1],[0,-1],[2,1],[1,-1],[0,-1],[1,-2],[1,-3],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[-1,0],[0,-1],[-2,0],[-2,-2],[0,-1],[-1,0],[0,1],[0,1],[-1,2],[0,1],[-1,-2],[0,-2],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[1,-5],[0,-1],[0,-1],[0,-2],[0,-1],[-2,-11],[0,-2],[0,-2],[-1,-1],[0,-1],[-2,1],[-1,-1],[-2,2],[-1,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[-1,-2],[0,-2],[0,-1],[1,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[1,-4],[0,-2],[0,-2],[0,-1],[0,-2],[-1,-1],[0,-1],[-1,-2],[-2,-2],[-5,2],[0,-1],[-1,-1],[-1,-6],[-1,-5],[0,-1],[0,-2],[0,-1],[0,-5],[1,-6],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[-1,-1],[0,-1],[0,-1],[-2,-4],[0,-4],[-1,0],[0,-1],[-1,1],[0,-1],[0,-1],[-1,-3],[0,-1],[0,-3],[-1,-2],[0,-2],[0,-1],[0,-1],[-1,-2],[-1,-1],[0,-3],[0,-1],[-1,0],[0,1],[0,1],[-1,2],[0,-1],[-1,-11],[-1,-2],[0,-1],[-1,0],[-2,-2],[-1,1],[0,-1],[0,-1],[-1,-3],[0,-2],[0,-1],[-1,-1],[0,-2],[-1,-4],[0,-4],[-1,-1],[0,-1],[0,-1],[-1,0],[0,-3],[0,-1],[-1,-1],[0,-2],[0,-2],[0,-1],[1,-2],[0,-1],[0,-7],[0,-1],[0,-1],[0,-3],[-1,-25],[0,-5],[-2,-20],[0,-2],[0,-1],[-1,-3],[0,-1],[0,-3],[-1,0],[0,-1],[0,1],[-1,4],[-2,6],[-2,-5],[0,-1],[0,-3],[-1,-1],[0,-1],[-1,2],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-2],[-1,-1],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-3],[0,-1],[0,-2],[-2,-8],[-1,-4],[0,-3],[-2,-7],[0,-2],[0,-2],[0,-6],[0,-3],[0,-1],[0,-6],[0,-6],[0,-6],[0,-1],[0,-1],[-1,-3],[0,-1],[0,-1],[-1,-4],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[1,-5],[0,-1],[1,-1],[1,0],[0,2],[0,2],[1,0],[0,-1],[0,-2],[1,-1],[0,-1],[1,0],[1,2],[0,-1],[1,-1],[2,-6],[1,-3],[1,-1],[0,-1],[1,-1],[0,-4],[0,-2],[0,-2],[0,-1],[0,-1],[1,-3],[1,-2],[0,2],[1,5],[0,2],[1,0],[0,1],[3,1],[2,-4],[1,0],[0,-1],[1,-3],[1,0],[1,0],[1,-3],[0,-1],[2,-3],[0,-2],[0,-1],[0,-3],[0,-1],[0,-2],[0,-3],[0,-5],[0,-1],[0,-2],[1,-1],[0,-1],[0,-2],[0,-1],[-1,-3],[0,-1],[0,-2],[1,0],[0,-3],[0,-2],[0,-2],[1,-1],[0,-5],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-10],[0,-3],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,-14],[0,-1],[0,-3],[0,-3],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[-1,-1],[0,-1],[0,-3],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[-1,-1],[0,-2],[0,-1],[1,-1],[4,0],[2,2],[2,1],[0,1],[0,2],[1,1],[2,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-7],[-1,-3],[0,-1],[0,-1],[0,-1],[1,-6],[1,-1],[0,1],[1,2],[0,1],[1,0],[1,-3],[0,1],[1,1],[0,2],[0,2],[1,10],[0,2],[0,1],[0,1],[0,1],[1,1],[1,-1],[2,1],[1,2],[2,1],[1,1],[0,1],[1,0],[0,-1],[1,-2],[0,-1],[4,1],[1,0],[0,6],[1,0],[1,-1],[0,-2],[1,-1],[2,1],[1,-3],[1,0],[1,1],[0,1],[0,2],[1,7],[0,1],[0,1],[0,1],[1,0],[0,-1],[0,-1],[1,1],[2,5],[1,0],[0,-3],[1,0],[0,2],[0,1],[0,1],[0,2],[0,1],[0,1],[0,2],[0,1],[0,2],[0,1],[0,1],[1,2],[0,2],[0,2],[0,1],[0,2],[0,1],[0,6],[0,1],[1,1],[0,1],[0,3],[2,3],[0,2],[0,2],[1,0],[0,1],[1,0],[1,0],[0,1],[0,2],[0,1],[1,4],[1,0],[0,-1],[0,-1],[0,-1],[0,-1],[1,1],[0,1],[1,4],[0,1],[0,3],[0,1],[0,1],[0,1],[1,0],[1,0],[0,-1],[1,-2],[0,-2],[1,-12],[1,-1],[0,-1],[1,0],[0,1],[1,4],[0,-2],[0,-1],[0,-1],[0,-1],[1,-1],[1,1],[3,-8],[1,0],[1,3],[1,0],[0,1],[1,1],[0,4],[0,2],[1,0],[1,-2],[0,1],[1,2],[0,4],[0,3],[0,1],[0,1],[0,-1],[1,0],[0,2],[1,0],[1,0],[0,1],[0,1],[1,0],[0,-3],[1,-1],[0,1],[1,1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,2],[0,5],[-1,1],[1,2],[0,1],[1,1],[1,0],[0,1],[0,4],[1,1],[1,-3],[1,-3],[0,-1],[1,-4],[0,-3],[2,-6],[0,-1],[0,-1],[0,-2],[0,-5],[0,-1],[0,-1],[0,-2],[0,-2],[1,-1],[0,-1],[0,1],[1,1],[0,-1],[1,-1],[-1,-2],[1,-2],[0,-1],[1,-4],[1,-1],[0,-1],[0,-2],[0,-2],[0,-3],[0,-4],[0,-2],[-1,0],[-1,2],[-1,0],[0,-1],[0,-7],[0,-2],[0,-2],[0,-1],[1,-1],[1,-1],[0,-1],[0,-2],[0,-1],[1,-3],[0,-2],[0,-1],[1,-2],[0,-2],[0,-2],[0,-5],[0,-3],[-1,-2],[-1,-4],[-1,-2],[0,-1],[0,-12],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[0,-3],[1,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[-1,0],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-3],[0,-1],[0,-1],[0,-1],[-1,-2],[0,-5],[0,-1],[-2,-10],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-3],[0,-2],[0,-1],[0,-3],[0,-3],[1,-1],[0,-1],[-1,0],[-1,0],[0,-1],[0,-2],[0,-4],[0,-7],[0,-3],[0,-2],[0,-1],[0,-1],[0,-1],[2,-3],[0,-1],[0,-1],[-1,-4],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[2,1],[0,-1],[0,-1],[0,-3],[0,-3],[1,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-3],[-1,-2],[0,-3],[0,-1],[-1,0],[0,-2],[-1,-3],[0,-3],[-1,-2],[0,-2],[0,-10],[0,-2],[0,-3],[0,-1],[0,-2],[1,-2],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-5],[0,-3],[-1,-2],[-1,1],[-1,0],[0,1],[-1,3],[0,1],[0,1],[-1,1],[-4,2],[0,1],[-2,7],[-1,0],[-2,-2],[-1,1],[0,1],[-1,1],[0,1],[-1,9],[-1,1],[-3,-5],[0,1],[-1,0],[0,1],[0,2],[0,2],[0,1],[0,1],[0,1],[-1,0],[0,-1],[-4,5],[0,1],[-1,5],[-1,2],[-1,4],[-1,1],[0,1],[0,1],[-1,1],[0,1],[0,1],[0,1],[-1,0],[-1,1],[0,1],[0,3],[-1,1],[0,1],[0,1],[0,1],[-1,0],[0,-1],[-1,-1],[-1,-3],[0,-5],[-1,-3],[0,-1],[0,-3],[0,-1],[1,-1],[0,-1],[0,-2],[0,-1],[-1,-2],[0,-3],[0,-1],[0,-2],[0,-1],[1,0],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[1,-1],[0,-1],[0,-2],[-1,-4],[0,-2],[0,-1],[0,-3],[0,-3],[-1,-2],[0,-1],[1,-1],[0,-2],[0,-2],[2,-4],[0,-1],[0,-2],[-1,-7],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[1,-1],[0,-1],[2,-6],[0,-2],[1,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[-1,-1],[0,-1],[-1,-6],[0,-2],[-1,-2],[0,-2],[0,-1],[-3,-4],[-1,-1],[0,-2],[0,-1],[-1,0],[0,2],[-1,0],[0,-1],[0,-1],[-1,-1],[0,1],[-1,1],[0,-1],[-1,-2],[0,-1],[-1,0],[0,1],[-1,1],[0,-1],[-1,0],[-2,5],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-2],[0,-2],[0,-1],[-1,-1],[-3,-8],[0,-2],[0,-1],[0,-1],[-1,-1],[0,1],[-1,0],[0,-1],[0,-1],[0,-3],[-1,-3],[0,-1],[0,-1],[0,-2],[0,-3],[0,-6],[-1,-9],[0,-1],[0,-1],[0,-1],[0,1],[-1,0],[0,2],[0,1],[-1,1],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-8],[0,-1],[0,-1],[1,-6],[0,-2],[0,-1],[0,-1],[0,-1],[-3,-9],[0,-2],[-1,-3],[0,-1],[0,-1],[0,-1],[-1,1],[0,1],[-1,3],[0,1],[-2,-5],[-1,-1],[0,-2],[0,-1],[0,-1],[0,-1],[-1,-1],[0,1],[0,1],[0,1],[0,3],[0,1],[0,1],[-1,1],[-1,-4],[0,-3],[0,-1],[-1,-1],[0,-6],[0,-1],[-1,0],[-1,1],[0,2],[0,1],[-1,0],[0,-2],[0,-1],[-1,-3],[0,-1],[0,-1],[-1,-2],[0,-2],[-1,-5],[-1,-1],[-4,-2],[-4,4],[-1,-2],[1,-2],[0,-2],[0,-1],[0,-2],[0,-2],[0,-3],[-1,-3],[0,-1],[0,-2],[0,-4],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-2],[1,-1],[0,-1],[0,-2],[0,-1],[1,-1],[0,-1],[-1,-4],[0,-1],[1,-1],[0,-1],[0,-2],[1,-2],[0,-6],[0,-16],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[-1,-1],[-1,0],[0,1],[0,1],[-1,6],[-1,1],[-1,0],[-1,-3],[0,-3],[-1,-8],[-1,-12],[-1,-4],[0,-1],[-1,2],[-1,0],[-1,-3],[0,-1],[0,-1],[-1,-1],[0,-2],[0,-4],[0,-5],[0,-1],[0,-2],[0,-1],[-1,-4],[0,-1],[0,-2],[0,-1],[0,-3],[0,-2],[0,-2],[0,-1],[-1,-6],[-1,-10],[0,-2],[0,-2],[0,-1],[-2,-1],[0,-1],[-1,1],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-6],[0,-2],[0,-2],[0,-1],[0,-2],[1,-1],[0,-1],[0,-1],[0,-6],[0,-3],[1,-2],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[-1,-10],[0,-2],[0,-2],[0,-3],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[-1,-4],[0,-2],[0,-1],[0,-2],[0,-2],[1,-2],[0,-1],[0,-2],[-1,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[1,-2],[1,-10],[1,1],[0,1],[0,2],[0,3],[0,4],[0,1],[0,2],[0,1],[0,1],[1,0],[1,-2],[0,-1],[0,1],[1,0],[1,-1],[0,1],[0,2],[0,1],[0,1],[1,0],[0,-1],[0,-1],[1,-4],[1,-16],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[-1,-2],[0,-5],[0,-2],[-2,-4],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-2],[0,-1],[0,-2],[1,-2],[1,-3],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[1,-2],[1,0],[0,2],[0,1],[0,1],[1,2],[0,1],[0,1],[0,1],[0,1],[1,0],[1,0],[0,1],[0,1],[0,1],[1,-1],[0,-1],[0,-1],[0,-7],[0,-3],[0,-1],[0,-1],[0,-7],[0,-1],[1,-1],[1,1],[0,1],[0,1],[0,1],[1,0],[0,1],[1,0],[0,3],[0,-1],[0,-1],[1,-4],[0,-2],[0,-5],[0,-3],[0,-6],[0,-2],[0,-1],[0,-2],[1,-1],[0,-2],[0,-1],[1,-2],[0,-4],[0,-2],[1,-4],[0,-1],[0,-1],[0,-1],[0,-2],[1,-2],[0,-3],[0,-1],[0,-1],[0,-1],[-1,0],[-1,-1],[0,-1],[0,-1],[0,-5],[-1,-5],[0,-2],[0,-2],[-1,-2],[0,-1],[-1,-1],[1,-4],[0,-2],[0,-1],[0,-1],[0,-3],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-2],[0,-2],[1,0],[1,2],[1,0],[0,-1],[0,-1],[0,-1],[0,-1],[1,0],[1,-1],[0,-3],[0,-2],[1,-2],[0,-4],[0,-1],[0,-2],[0,-1],[0,-4],[0,-2],[0,-2],[0,-9],[0,-2],[-1,-5],[0,-2],[0,-2],[0,-3],[0,-4],[1,-1],[0,-2],[-1,-1],[0,-2],[0,-1],[-1,-1],[-1,0],[-1,0],[0,-1],[0,-1],[0,-1],[0,-2],[0,-4],[0,-2],[0,-2],[0,-2],[1,-1],[0,-1],[1,1],[0,-1],[1,-3],[1,-7],[0,-1],[0,-2],[0,-3],[0,-6],[0,-5],[0,-1],[0,-4],[0,-3],[0,-13],[0,-2],[0,-2],[0,-3],[0,-1],[0,-2],[0,-2],[-3,-41]],[[8649,1980],[0,1],[-1,0],[-1,-1],[0,-2],[-1,0],[0,1],[0,1],[0,1],[-1,0],[0,1],[0,1],[0,1],[-1,-2],[-1,-1],[0,-1],[0,-1],[-1,-3],[0,-4],[0,-6],[0,-2],[-1,-2],[-1,-11],[0,-2],[0,-5],[-1,-3],[0,-1],[0,-1],[0,-1],[-1,-2],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[1,-1],[0,-2],[-1,-1],[0,-2],[0,-9],[0,-1],[-1,-1],[0,-1],[-1,1],[-1,1],[0,3],[-1,0],[-1,0],[-3,-12],[-1,-3],[-1,-6],[0,-2],[0,-4]],[[8628,1889],[-2,-5],[-1,0],[-1,0],[0,2],[-1,4],[0,3],[-1,1],[-1,0],[-1,-3],[-2,-3],[-1,-2],[-1,5],[-5,22],[-1,7],[0,3],[-2,3],[0,3],[0,5],[0,6],[-1,4],[-1,2],[0,1],[0,11],[0,3],[-1,1],[-1,1],[-1,2],[0,3],[0,2],[0,8],[-1,9],[-1,1],[-1,-1],[-2,1],[0,3],[-1,3],[0,5],[0,11],[0,4],[0,3],[-1,4],[0,2],[-1,2],[-1,1],[-1,-2],[0,-3],[0,-10],[0,-5],[-1,-2],[-1,0],[-1,8],[-1,3],[0,1],[-1,-1],[-1,-2],[-1,-6],[0,-2],[-1,1],[-1,3],[0,3],[-1,14],[-1,2],[-1,2],[-1,6],[-1,2],[-1,0],[-1,-3],[0,-1],[-1,1],[0,4],[0,4],[0,5],[0,1],[1,0],[0,1],[0,3],[0,3],[0,3],[0,1],[0,1],[-1,1],[-1,1],[-1,5],[-1,2],[-1,-8],[0,-1],[-2,3],[-5,-2],[0,-1],[0,-1],[0,-4],[-1,-2],[-2,3],[0,-2],[-1,-2],[-1,-1],[-3,1],[0,1],[-1,3],[-1,7],[-1,3],[0,1],[-1,0],[0,-2],[0,-2],[0,-1],[-1,-1],[-1,1],[-1,3],[-1,3],[0,4],[-1,3],[-1,3],[-1,3],[0,5],[0,5],[0,4],[0,5],[-1,7],[-3,6],[-1,8],[0,5],[0,6],[0,6],[1,4],[0,3],[-1,5],[0,8],[-1,13],[1,9],[2,17],[0,4],[0,5],[0,4],[0,6],[0,3],[-5,13],[-1,6],[-1,7],[0,3],[0,3],[1,3],[0,2],[0,2],[0,3],[0,6],[0,2],[-2,7],[0,3],[0,3],[1,3],[0,6],[0,3],[1,4],[0,3],[0,6],[0,6],[-1,6],[0,2],[-1,7],[0,3],[0,1],[0,2],[0,1],[0,2],[0,2],[0,1],[0,2],[0,2],[0,5],[-1,3],[0,3],[-1,3],[-2,29],[-1,7],[-2,8],[0,2],[-2,13],[-1,5],[0,4],[0,6],[0,5],[-1,9],[0,5],[0,7],[0,4],[0,3],[1,3],[0,6],[1,6],[0,3],[0,3],[0,4],[0,1],[-1,0],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-2],[0,-3],[-1,-5],[-1,4],[0,6],[0,4],[0,4],[0,4],[1,2],[0,1],[1,1],[0,1],[0,4],[0,4],[0,2],[0,2],[-2,1],[-1,1],[0,5],[0,5],[0,5],[1,7],[0,7],[-1,3],[-2,4],[-1,3],[0,3],[1,6],[0,6],[0,3],[0,4],[0,5],[0,1],[0,2],[0,1],[0,2],[0,2],[0,1],[-1,3],[-1,2],[-5,59],[-1,4],[1,5],[0,3],[1,2],[0,1],[0,1],[1,3],[0,13],[1,4],[-1,6],[-1,2],[-1,1],[-3,6],[-1,1],[0,1],[0,4],[0,3],[2,9],[1,3],[0,3],[-2,4],[-1,3],[1,10],[0,5],[0,4],[-1,6],[-1,4],[0,4],[0,3],[-1,1],[-1,0],[0,5],[0,6],[0,5],[-1,4],[-3,11],[0,1],[0,-2],[0,-3],[0,-1],[-1,-1],[-1,0],[0,5],[0,5],[-1,6],[1,5],[0,3],[1,1],[1,1],[0,1],[0,5],[-1,7],[1,3],[0,1],[1,3],[0,5],[-1,2],[-1,0],[-1,-1],[0,-4],[-4,21],[0,2],[0,2],[0,2],[0,2],[0,2],[0,2],[0,2],[-3,-2],[-1,-4],[-1,-1],[-1,2],[0,4],[1,4],[0,4],[1,1],[0,1],[0,1],[0,2],[1,7],[-2,6],[-1,4],[0,6],[0,3],[-1,0],[-2,-4],[-1,2],[0,8],[-1,2],[-1,0],[-8,28],[-2,1],[-1,0],[-3,-4],[0,-2],[-1,-3],[0,-5],[0,-5],[0,-4],[0,-2],[-1,1],[-1,8],[0,2],[-2,-1],[-1,3],[0,2],[0,2],[0,1],[0,3],[-1,1],[0,3],[-1,0],[-3,0],[0,1],[-2,3],[-1,2],[-1,3],[0,3],[-1,3],[0,8],[0,1],[-1,5],[0,6],[-1,3],[0,3],[-1,1],[-1,0],[0,-1],[-1,-2],[0,-3],[-1,-1],[-1,3],[0,6],[-1,1],[-1,2],[0,1],[0,2],[-1,7],[0,3],[-3,8],[-1,2],[-3,2],[-1,2],[0,2],[-1,8],[-1,3],[-1,-2],[-1,-8],[0,-2],[-1,1],[1,3],[0,1],[0,1],[0,4],[-1,1],[-1,-4],[0,-4],[0,-2],[-1,4],[-1,1],[-1,1],[-1,4],[-1,2],[-1,0],[-1,-2],[-1,-5],[0,-2],[-1,-4],[0,-1],[-4,-1],[-4,-8],[-11,-3],[-1,-3],[-1,0],[0,3],[-1,4],[-1,3],[0,1],[-1,-1],[-2,-4],[-1,-1],[-1,-2],[-1,-6],[-1,-2],[-1,0]],[[8389,3009],[0,18],[0,4],[-1,2],[0,1],[-1,3],[0,1],[0,2],[0,4],[0,1],[0,4],[0,2],[0,1],[0,1],[0,1],[0,1],[0,2],[0,5],[0,4],[0,4],[0,2],[0,1],[0,1],[0,2],[0,1],[0,1],[-1,1],[0,1],[0,1],[0,3],[0,2],[0,2],[0,1],[0,2],[0,1],[0,4],[0,4],[0,1],[0,1],[0,2],[0,1],[0,3],[-3,22],[0,10],[-1,2],[-1,-2],[0,1],[-2,6],[0,2],[-1,3],[0,1],[0,3],[0,2],[0,4],[0,5],[0,2],[0,3],[0,5],[0,2],[0,2],[1,0],[0,1],[2,2],[0,1],[0,1],[0,8],[0,5],[0,3],[0,2],[0,1],[0,3],[0,4],[0,3],[0,1],[0,1],[0,1],[0,1],[0,1],[-1,4],[0,3],[0,2],[0,2],[-1,3],[0,1],[-1,1],[1,3],[1,10],[0,2],[0,2],[1,1],[2,3],[0,-1],[1,-3],[0,-1],[1,-1],[0,-3],[-1,-2],[1,-2],[0,-1],[0,-3],[1,0],[1,2],[1,4],[3,3],[0,1],[1,1],[0,1],[0,1],[0,1],[0,4],[0,2],[0,1],[0,2],[0,1],[-2,8],[-1,1],[0,1],[-1,1],[0,5],[0,2],[-1,3],[-1,3],[0,1],[0,1],[0,2],[0,2],[0,1],[0,3],[0,1],[0,2],[0,1],[0,1],[0,2],[0,1],[-1,2],[0,1],[0,2],[0,2],[0,1],[0,1],[1,3],[0,1],[0,2],[0,1],[0,1],[0,2],[0,4],[0,2],[0,1],[-1,2],[0,1],[0,2],[0,1],[0,1],[0,2],[0,1],[0,2],[0,1],[0,1],[0,-1],[-1,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-2],[-1,-1],[0,-1],[-1,0],[0,-1],[0,-1],[-1,-4],[0,-2],[0,-1],[-1,0],[0,1],[-1,2],[0,1],[0,2],[0,1],[0,2],[1,1],[0,5],[0,1],[0,2],[0,1],[0,3],[0,1],[0,1],[0,1],[0,1],[1,3],[2,10],[0,2],[1,8],[1,2],[0,1],[0,1],[0,1],[0,2],[0,1],[1,3],[1,1],[0,3],[0,2],[1,1],[0,1],[0,2],[0,7],[0,5],[0,6],[0,1],[-1,5],[0,2],[0,2],[0,1],[3,13],[0,2],[-1,1],[-1,0],[0,1],[0,2],[-1,2],[0,1],[0,1],[1,2],[0,3],[0,2],[0,1],[0,2],[0,1],[0,1],[-1,2],[0,1],[0,1],[1,3],[0,1],[0,1],[0,2],[0,3],[0,1],[0,2],[0,3],[0,3],[0,1],[0,2],[0,1],[0,4],[1,3],[0,1],[0,2],[0,1],[0,1],[-1,1],[0,2],[0,1],[0,1],[1,4],[0,2],[0,1],[-1,2],[0,1],[-1,4],[0,1],[0,2],[0,1],[0,2],[0,1],[0,1],[0,3],[-1,2],[0,3],[0,2],[0,1],[0,2],[0,1],[0,2],[0,1],[-1,0],[-1,-1],[0,1],[0,1],[-1,1],[-1,0],[0,-1],[0,-2],[-1,0],[-1,-1],[0,-1],[0,-5],[-1,-5],[0,-5],[-1,-1],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[-1,0],[0,-1],[-2,1],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-1],[0,1],[-1,1],[-1,5],[0,2],[0,2],[0,1],[0,4],[0,1],[0,1],[0,3],[0,2],[0,3],[0,2],[0,1],[0,1],[0,1],[0,6],[-1,1],[0,1],[0,4],[0,1],[0,2],[0,3],[0,5],[0,1],[0,1],[0,2],[0,1],[0,2],[0,1],[0,1],[0,2],[0,7],[1,3],[0,1],[0,2],[0,2],[0,1],[0,1],[-1,2],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[-1,2],[0,1],[0,1],[0,1],[0,4],[0,2],[0,1],[0,5],[0,2],[0,3],[0,1],[0,5],[-1,2],[0,2],[0,5],[-1,3],[0,1],[-3,-4],[-1,-1],[0,1],[-2,5],[-3,-3],[-1,-3],[-1,-1],[0,-4],[0,-1],[-4,-8],[-1,-1],[0,-1],[-3,-4],[0,-2],[-1,-2],[0,-1],[0,-3],[0,-2],[-1,-1],[-1,-1],[-1,0],[0,2],[-1,1],[0,2],[0,1],[0,1],[0,1],[0,1],[0,1],[-1,1],[-1,2],[0,-1],[-1,-1],[0,-2],[0,-1],[-1,-2],[0,-4],[-1,0],[0,5],[-1,2],[0,1],[0,2],[0,3],[0,1],[0,2],[1,3],[0,1],[0,1],[0,1],[0,2],[0,3],[0,6],[-1,2],[-1,8],[0,3],[-1,2],[0,2],[0,2],[0,3],[0,5],[0,3],[0,2],[0,1],[1,0],[0,1],[1,-1],[1,1],[0,1],[0,1],[1,6],[0,1],[0,1],[0,1],[1,2],[1,0],[0,-1],[1,0],[0,1],[0,3],[1,1],[0,1],[1,0],[1,-1],[1,0],[0,1],[1,3],[1,0],[0,1],[0,1],[0,1],[0,1],[0,2],[1,1],[0,2],[0,1],[-1,2],[0,1],[-1,0],[-2,6],[0,1],[-1,1],[0,2],[0,1],[1,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[-1,1],[-1,0],[0,1],[-2,5],[0,2],[0,2],[0,2],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[-1,-1],[-1,0],[0,1],[-3,6],[-3,0],[0,1],[-1,1],[0,2],[0,1],[-1,9],[0,3],[-2,6],[0,1],[-1,1],[0,3],[0,1],[0,2],[1,5],[0,2],[0,1],[1,2],[0,2],[0,2],[0,4],[0,2],[-1,1],[-1,0],[0,1],[-1,1],[0,1],[0,5],[0,1],[0,-1],[-1,0],[-1,2],[0,1],[0,2],[-1,3],[0,1],[0,1],[1,2],[0,2],[0,4],[0,2],[1,1],[4,10],[0,3],[1,6]],[[8330,3860],[0,1],[3,0],[0,1],[5,11],[1,10],[1,4],[1,1],[5,3],[3,-2],[1,1],[0,1],[0,1],[0,1],[1,2],[3,-3],[1,0],[0,3],[1,1],[1,1],[2,-3],[0,2],[1,-1],[1,-1],[0,-2],[0,-5],[1,-3],[0,-1],[1,-1],[0,2],[1,1],[8,-4],[0,-1],[0,-1],[2,-8],[0,-2],[0,-1],[1,-6],[0,-9],[0,-1],[0,-1],[0,-2],[1,-4],[1,-2],[0,-1],[0,-1],[0,-6],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[3,2],[6,-3],[1,1],[1,3],[4,-4],[1,0],[0,-2],[0,-7],[1,-2],[1,-6],[1,-4],[0,-1],[3,-4],[2,-7],[0,-1],[0,-1],[0,-5],[0,-1],[1,-1],[1,-4],[1,-2],[0,-2],[0,-1],[0,-3],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[2,-1],[1,0],[0,1],[0,1],[0,1],[1,1],[1,-2],[1,1],[0,1],[0,1],[0,1],[0,2],[0,3],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[0,2],[0,1],[1,0],[2,3],[0,-2],[0,-1],[0,-4],[0,-3],[1,-1],[0,-1],[2,-6],[1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[3,-9],[4,-8],[3,-2],[1,-2],[1,0],[1,1],[3,-3],[0,-1],[1,0],[0,1],[0,1],[2,-2],[1,1],[0,-1],[0,-1],[1,0],[1,2],[1,-3],[1,-2],[1,-4],[0,-3],[1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[1,-3],[0,-2],[0,-1],[0,-1],[1,-3],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,-1],[1,-4],[1,-1],[0,-3],[0,-1],[1,-3],[0,-3],[1,-3],[0,-1],[0,-1],[0,-3],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-2],[0,-2],[1,-2],[0,-1],[3,-3],[0,-1],[0,-2],[0,-3],[0,-1],[1,-3],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[1,-3],[0,-4],[0,-2],[0,-5],[0,-4],[0,-1],[0,-2],[0,-1],[1,-3],[1,-3],[0,-1],[1,0],[0,1],[1,0],[1,7],[1,1],[0,-1],[1,0],[0,-2],[1,-2],[0,-2],[1,-5],[0,-1],[1,0],[0,1],[1,-1],[0,-2],[0,-1],[1,-1],[0,-1],[1,-1],[1,-1],[1,1],[0,1],[0,1],[1,0],[0,1],[1,-1],[1,1],[0,1],[0,3],[3,5],[1,0],[3,-6],[2,-1],[0,-1],[1,-2],[0,-2],[0,-1],[0,-5],[0,-1],[0,-1],[1,-4],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,-2],[0,-1],[1,0],[0,1],[1,5],[1,2],[3,7],[1,0],[1,0],[0,-1],[3,-6],[1,-1],[1,-8],[1,0],[0,-1],[1,-3],[0,-3],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[1,-1],[0,-1],[1,-2],[0,-2],[0,-1],[0,-1],[-1,0],[0,1],[0,1],[-1,0],[0,-1],[0,-2],[0,-1],[0,-5],[-1,-4],[1,-1],[1,-1],[4,2],[1,0],[1,-2],[1,0],[0,-1],[1,-2],[0,-1],[0,-1],[1,-2],[1,4],[1,1],[0,1],[0,3],[0,1],[1,3],[0,2],[0,2],[0,2],[0,1],[2,5],[1,1],[1,0],[0,-1],[0,-1],[0,-2],[0,-4],[0,-1],[0,-2],[0,-1],[0,-1],[1,-2],[1,0],[0,1],[0,2],[0,1],[1,0],[1,-2],[1,0],[0,1],[1,1],[0,3],[1,4],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[1,0],[2,-3],[2,-1],[1,1],[0,1],[1,2],[0,1],[1,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,2],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-2],[1,-2],[0,-3],[0,-1],[0,-1],[2,-4],[0,-2],[1,-2],[0,-2],[0,-1],[1,1],[0,2],[1,2],[0,1],[1,0],[0,1],[0,1],[0,2],[0,2],[0,1],[1,1],[0,1],[0,2],[0,2],[0,2],[0,1],[0,3],[0,1],[0,2],[1,1],[1,3],[1,0],[1,-1],[0,-1],[0,-1],[1,-2],[0,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-1],[3,-2],[3,4],[1,-1],[1,1],[1,2],[1,0],[1,3],[1,1],[1,-2],[1,-1],[0,-1],[2,-11],[0,-1],[1,-3],[0,-1],[1,-1],[0,-1],[0,-5],[0,-1],[0,-1],[0,-1],[1,-1],[0,-1],[0,-3],[0,-1],[1,0],[3,-1],[1,2],[1,0],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,-1],[0,-1],[1,-1],[0,-1],[0,-1],[1,2],[0,2],[1,-2],[2,-1],[1,-1],[0,-1],[1,1],[2,3],[0,1],[0,2],[0,1],[0,3],[1,3],[0,2],[0,1],[0,1],[0,2],[0,3],[0,2],[0,1],[0,1],[0,2],[0,1],[0,2],[-1,5],[-1,2],[0,2],[0,1],[0,1],[0,1],[0,1],[1,1],[0,1],[5,6],[0,2],[6,3],[0,1],[0,1],[0,1],[0,2],[0,1],[1,1],[1,-1],[1,2],[1,1],[1,0],[0,2],[1,0],[1,-2],[2,0],[0,-1],[1,-4],[0,-1],[0,-1],[1,0],[0,1],[0,1],[0,1],[1,2],[0,1],[0,1],[1,-1],[1,-2],[1,-8],[0,-1],[1,0],[1,1],[0,1],[1,1],[1,-2],[1,-1],[1,2],[0,1],[9,2],[0,1],[0,1],[1,2],[4,0],[1,-1],[3,-8],[0,-1],[2,-1]],[[8308,4187],[0,-4],[1,-11],[0,-3],[0,-3],[0,-1],[0,-8],[0,-5],[0,-2],[0,-2],[0,-4],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-5],[0,-1],[0,-5],[0,-5],[0,-14],[0,-2],[-1,-5],[0,-5],[0,-2],[0,-2],[0,-2],[0,-2],[0,-2],[0,-4],[0,-2],[0,-2],[-1,-2],[0,-2],[0,-4],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[1,-1],[0,-1],[1,-1],[1,-1],[0,-3],[0,-6],[0,-3],[0,-2],[0,-1],[0,-2],[0,-2],[0,-1],[0,-4],[0,-4],[0,-3],[0,-1],[0,-3],[0,-3],[0,-3],[0,-1],[0,-1],[0,-2],[0,-2],[1,-2],[0,-1],[0,1],[0,1],[1,3],[1,7],[0,3],[0,2],[0,2],[0,2],[0,1],[1,2],[0,1],[1,2],[2,1],[1,-1],[1,-3],[0,-1],[0,-1],[0,-1],[1,-2],[-1,-1],[0,-4],[0,-1],[0,-3],[0,-1],[0,-1],[0,-1],[-1,-2],[-1,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-3],[0,-3],[0,-2],[0,-2],[0,-3],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[1,-1],[0,-2],[0,-1],[0,-3],[0,-3],[0,-1],[0,-1],[1,0],[1,0],[0,-1],[1,-5],[0,-2],[0,-2],[1,-6],[0,-7],[0,-5],[0,-1],[0,-2],[0,-1],[1,-3],[1,-10],[1,-1],[0,-2],[0,-1],[0,-2],[-1,-3],[-1,-10],[0,-2],[0,-1],[0,-6],[0,-4],[0,-4],[0,-2],[0,-1],[0,-1],[1,-4],[1,-3],[0,-1],[2,0],[0,-1],[1,-2],[0,-3],[0,-3],[0,-4],[1,-2],[0,-4],[1,-2],[0,-2],[1,0]],[[8389,3009],[-4,0],[-7,-11],[-3,-14],[-2,-2],[-1,-1],[-2,2],[-1,-1],[0,-4],[-2,-4],[-1,-3],[-2,0],[-2,4],[-5,-3],[-1,-2],[-1,-2],[1,-5],[-1,-2],[-2,-7],[-1,-5],[-1,-16],[-1,-6],[-3,-10],[-2,-12],[-2,-12],[0,-5],[-1,-1],[0,-3],[0,-2],[0,-1],[0,-3],[-1,-2],[0,-1],[-1,-3],[0,-2],[0,-2],[0,-1],[-1,-2],[-2,-7],[0,-2],[-3,-6],[-1,-6],[1,-6],[0,-3],[0,-4],[0,-3],[0,-6],[-1,-3],[0,-4],[0,-4],[1,-3],[0,-3],[1,-3],[1,-1],[1,0],[1,1],[1,5],[1,1],[3,1],[0,1],[1,4],[1,0],[1,-5],[1,-3],[2,-8],[2,-1],[1,-4],[1,-2],[-1,-6],[0,-2],[0,-3],[0,-3],[0,-3],[0,-1],[0,-8],[-1,-4],[0,-6],[0,-4],[0,-2],[-1,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-2],[1,-7],[2,-11],[1,-4],[0,-3],[-1,-4],[0,-2],[0,-2],[1,-4],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-2],[0,-2],[0,-2],[-1,-5],[0,-3],[0,-1],[-1,-1],[0,-2],[0,-3],[0,-3],[0,-3],[0,-3],[0,-2],[0,-4],[0,-2],[-1,-5],[0,-8],[-1,-4],[0,-1],[-1,1],[-1,-1],[0,-2],[0,-2],[0,-1],[-1,-2],[-1,0],[0,-2],[0,-2],[0,-3],[-1,-2],[0,-1],[-1,-1],[0,-1],[-1,-6],[-2,-7],[-3,-23],[-1,-3],[0,-1],[-1,0],[-1,-1],[0,-4],[-1,-7],[0,-5],[0,-5],[0,-6],[0,-3],[-2,-9],[0,-4],[0,-5],[0,-3],[0,-3],[0,-5],[-1,-2],[0,-2],[0,-2],[0,-1],[-1,-2],[0,-4],[0,-5],[0,-4],[0,-2],[-1,-1],[0,-7],[-1,-2],[0,-1],[-1,-1],[0,-1],[0,-7],[0,-2],[0,-3],[0,-1],[0,-1],[-1,1],[0,-1],[0,-4],[1,-6],[-1,-3],[0,-1],[0,-4],[0,-3],[0,-3],[0,-4],[0,-4],[-1,-3],[0,-2],[-1,0],[0,-6],[0,-6],[-2,-9],[-1,-2],[0,-6],[-2,-10],[0,-4],[0,-5],[0,-5],[0,-3],[0,-5],[0,-6],[0,-10],[0,-3],[-2,-10],[0,-3],[-1,-2],[0,-1],[-1,-2],[0,-5],[0,-2],[-1,0],[0,-2],[0,-2],[0,-2],[-1,-2],[0,-1],[0,-3],[0,-2],[-1,-15],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-8],[0,-5],[-1,-2],[0,-1],[-1,0],[0,-2],[0,-3],[-1,-3],[1,-1],[1,0],[0,-2],[0,-3],[1,0],[0,-1],[1,-2],[1,2],[1,0],[1,-1],[0,-2],[0,-5],[0,-2],[0,-2],[-1,0],[0,-2],[0,-5],[0,-14],[0,-2],[0,-2],[0,-1],[0,-3],[0,-10],[0,-5],[0,-4],[0,-4],[-1,-3],[-1,-8],[-1,-5],[0,-3],[-1,-2],[-3,-5],[-3,-2],[0,1],[-1,1],[0,1],[-1,0],[0,-2],[0,-1],[-3,-3],[-1,-3],[-1,-1],[-1,3],[-1,-2],[-1,-5],[-1,-3],[-1,-3],[-1,-3],[0,-5],[0,-2],[-1,-2],[0,-4],[0,-2],[0,-2],[-2,-2],[0,-2],[-1,-6],[0,-2],[-1,-1],[-1,-2],[-1,-9],[-1,0],[0,-4],[0,-4],[-1,-4],[0,-2],[-2,-5],[-1,-3],[0,-4],[-1,-2],[-1,-5],[-2,-7],[-1,-3],[0,-2],[-1,-8],[0,-1],[-1,0],[0,1],[-1,1],[0,-1],[0,-1],[-1,1],[-8,25],[-6,2],[-5,12],[-10,32],[-1,5],[0,10],[-1,8],[-2,6],[-7,18],[-3,5],[-2,0],[-2,-4],[-9,-28],[0,-1],[-1,-1],[-3,4],[-2,-1],[-1,0],[-1,1],[-2,0],[0,1],[-5,19],[-4,26],[0,3],[-1,7],[0,3],[0,3],[-7,23],[-3,3],[-5,-4],[-1,2],[-3,8],[-1,1],[-3,-5],[-2,-8],[-2,-10],[0,-2],[-1,-2],[-1,-2],[-3,-20],[-1,-1],[-1,2],[-1,-1],[-5,-15],[-1,-6],[-1,-3],[-1,-6],[-1,-3],[0,-6],[-6,-24],[-1,-5],[-1,-6],[-1,-22],[-1,-5],[0,-4],[0,-8],[0,-6],[-1,-2],[-1,0],[-1,-1],[0,-2],[-1,-2],[-2,-9],[-2,-7],[-1,-2],[-1,-1],[-1,2],[-2,7],[-4,2],[-1,-1],[-8,-22],[-3,-7],[-10,-4],[-1,-2],[-3,-10],[-1,-1],[-1,0],[-1,1],[-1,4],[0,1],[-1,-1],[-1,-2],[0,-6],[-1,-2],[-4,-8],[-4,-20],[-4,-11],[-3,-6],[-1,-1],[-1,0],[-1,2],[-3,10],[-2,3],[0,1],[-2,10],[0,-2],[-1,-3],[-1,-6],[-1,-5],[-1,-3],[-1,-1],[-8,13],[-2,3],[0,1],[-1,5],[-2,4],[-6,-2],[0,1],[-1,2],[0,2],[0,4],[0,2],[-1,1],[-1,-1],[0,2],[-2,3],[0,2],[-1,2],[-4,2],[-2,-4],[-1,-1],[-3,5],[-2,-2],[-3,1],[-4,-5],[-1,1],[-1,0],[-2,7],[-3,20],[-2,11],[0,4],[-1,3],[0,3],[0,3],[-1,2],[-2,1],[-1,2],[-2,9],[0,3],[-1,4],[0,3],[1,6],[-1,2],[-1,-1],[0,1],[-1,1],[0,4],[1,9],[0,3],[0,3],[0,9],[-1,12],[0,5],[0,4],[1,3],[0,4],[0,7],[0,2],[-2,1],[-1,2],[-1,0],[-1,1],[-1,4],[0,1]],[[8678,406],[0,-4],[1,-4],[0,-2],[0,-1],[0,-1],[0,1],[-1,0],[0,2],[0,3],[0,2],[-1,1],[0,-1],[0,1],[0,2],[0,3],[0,2],[0,1],[0,4],[1,4],[0,-1],[0,-4],[0,-4],[0,-4]],[[8662,457],[1,-4],[1,-3],[0,-1],[0,-2],[0,-1],[0,-3],[-1,-2],[0,-6],[-2,0],[-1,3],[-1,4],[1,1],[0,3],[-1,2],[1,2],[0,-1],[0,4],[0,1],[0,1],[0,2],[1,0],[0,-2],[1,-1],[0,1],[-1,5],[1,-1],[0,-2]],[[8861,1524],[0,-2],[-1,-3],[0,-2],[0,-3],[0,-2],[0,-2],[0,-3],[-1,-1],[0,-1],[-1,0],[-1,-1],[0,-1],[-6,-49],[-1,-5],[-1,-5],[-1,-9],[-1,-10],[0,-12],[-1,-11],[0,-5],[0,-2],[0,-1],[0,-1],[-2,-19],[-1,-5],[0,-18],[-1,-3],[0,-3],[0,-6],[0,-11],[0,-11],[-1,-6],[-2,-10],[-1,-7],[-2,-41],[-1,-11],[0,-4],[-2,-18],[0,-4],[-1,-2],[0,-1],[-1,-1],[0,-2],[0,-1],[0,-2],[0,-5],[0,-2],[-1,-3],[-1,-7],[0,-6],[-3,-15],[0,-4],[0,-8],[-2,-11],[0,-9],[-1,-5],[-1,-4],[-3,-22],[-1,-3],[-2,-13],[-2,-14],[-2,-20],[0,-3],[-1,-1],[0,-2],[-2,-22],[-1,-2],[-1,-3],[-4,-24],[-3,-13],[-1,-2],[0,-5],[0,-11],[0,-5],[-1,-5],[-1,-5],[-1,-3],[0,-1],[0,-3],[0,-1],[-1,1],[0,-2],[-2,-19],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,1],[0,-3],[0,-4],[0,-2],[-1,-5],[0,-1],[0,-6],[-1,-1],[0,-1],[0,-1],[-1,-1],[0,-3],[0,-8],[0,-3],[-1,1],[0,3],[-1,1],[0,-1],[0,-2],[0,-3],[-1,-2],[0,-2],[0,-1],[-1,-1],[1,-2],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-1],[0,-1],[-1,-3],[-1,-7],[0,-6],[0,-11],[0,-4],[0,-1],[-1,-1],[0,-5],[0,-5],[0,-1],[-1,-5],[-1,-5],[-2,-6],[-1,2],[0,-1],[0,-1],[0,-2],[-1,-2],[0,-3],[-2,-3],[0,-2],[0,-2],[-1,-3],[0,-2],[-1,-3],[0,-2],[0,-6],[0,-4],[0,-5],[-1,-8],[-1,-2],[0,-2],[-1,-2],[0,-4],[0,-5],[-1,-3],[0,-2],[-1,-2],[0,-3],[0,-3],[0,-3],[0,-2],[0,-2],[0,-3],[0,-4],[0,-5],[0,-4],[-1,-3],[0,-3],[-1,-3],[-1,-3],[0,-4],[0,-5],[0,-1],[0,-1],[-1,-9],[0,2],[0,3],[-1,2],[0,-3],[0,-3],[0,-4],[0,-4],[1,0],[0,-1],[-1,-3],[1,0],[1,3],[0,-2],[-1,-12],[-1,-2],[0,-1],[-1,-10],[0,-2],[0,-2],[0,-1],[-1,-1],[0,-2],[0,-2],[-1,-2],[0,-2],[0,-1],[0,-1],[-1,1],[0,-1],[0,-2],[-1,-3],[0,-1],[-1,1],[0,3],[0,3],[0,1],[0,2],[0,1],[-1,0],[0,-4],[0,-1],[-1,-1],[0,-2],[1,-1],[0,-1],[1,-3],[-1,0],[0,-3],[0,-6],[0,-2],[0,-2],[0,-1],[-1,-8],[0,-2],[0,-4],[0,-2],[-1,-2],[0,-1],[0,-2],[0,-3],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[-1,0],[-1,-3],[-3,-7],[0,-1],[0,-4],[-1,-3],[-1,-3],[0,-2],[-1,-6],[-2,-16],[-1,-3],[0,2],[-1,-2],[-1,-4],[0,-3],[0,-2],[-1,-1],[0,-1],[-1,-2],[0,-4],[0,-1],[-1,1],[-1,-3],[-1,-10],[-2,-6],[0,-1],[-1,-4],[-1,-3],[0,-1],[-1,2],[0,-1],[0,-3],[-2,-4],[0,-1],[-1,-1],[0,-2],[-1,-2],[0,-1],[0,-3],[0,-2],[0,-1],[-1,-1],[0,-4],[0,-3],[-1,-4],[-1,-2],[0,-3],[-4,-22],[0,-1],[-1,1],[0,4],[-1,0],[0,-1],[-1,-3],[0,-2],[-2,-8],[0,-5],[0,-2],[-1,2],[-1,6],[0,1],[-1,0],[0,-1],[-1,0],[0,1],[0,-2],[0,-2],[0,-1],[-1,-1],[0,-2],[-1,2],[0,-1],[0,-2],[0,-1],[-1,-2],[0,-1],[0,-1],[-1,-4],[-1,-2],[-1,-2],[-2,1],[0,-1],[-2,-5],[0,-1],[-1,-8],[0,-3],[-1,-1],[-3,0],[0,1],[-1,1],[0,3],[-1,3],[1,1],[0,3],[1,1],[0,-1],[1,0],[0,3],[-1,3],[0,4],[-2,6],[1,3],[0,1],[-1,2],[-2,-2],[0,-1],[0,-2],[-1,-1],[1,-1],[0,-1],[-1,-8],[0,-2],[0,-2],[-1,0],[0,1],[-1,1],[0,3],[0,6],[0,4],[0,3],[-1,2],[0,1],[0,5],[0,6],[0,5],[0,3],[-1,0],[-1,-7],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[-1,1],[-1,0],[0,-1],[-1,-1],[0,-2],[0,-1],[-1,2],[0,3],[-1,1],[0,2],[0,4],[0,3],[0,3],[0,3],[-1,1],[-1,2],[-1,1],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-1],[0,-2],[-1,-2],[1,-1],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-2],[-1,0],[0,10],[0,1],[0,3],[0,4],[0,8],[0,12],[0,6],[0,5],[0,2],[0,3],[0,3],[0,2],[0,1],[1,2],[0,3],[0,3],[0,1],[0,1],[0,1],[0,1],[0,1],[0,2],[0,5],[0,1],[1,-1],[0,1],[0,4],[0,4],[0,2],[-1,4],[-1,2],[0,2],[1,1],[0,3],[0,1],[1,3],[0,1],[0,1],[-1,3],[0,4],[-1,1],[0,-2],[0,-8],[-1,-2],[0,-1],[0,-5],[-1,-2],[0,-2],[-2,-9],[-1,-1],[-1,-2],[0,-5],[-1,-8],[0,-3],[-1,-5],[-1,-3],[-1,-3],[0,-1],[-1,2],[0,2],[0,3],[0,1],[1,2],[0,3],[-2,-7],[0,-2],[0,1],[0,3],[0,9],[1,11],[0,4],[1,2],[1,6],[0,1],[1,1],[0,2],[0,1],[0,2],[1,7],[0,3],[0,4],[0,2],[-1,-2],[0,-4],[0,-4],[0,-2],[-1,-1],[0,5],[0,4],[-2,3],[-1,-3],[0,-1],[-1,2],[0,2],[0,2],[0,1],[0,1],[-1,-1],[0,-2],[0,-2],[0,-3],[-1,-3],[0,-7],[0,-5],[0,-4],[-1,-4],[0,-2],[1,0],[0,-2],[0,-2],[0,-3],[-1,0],[0,3],[0,3],[-1,1],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-3],[0,-3],[1,0],[0,2],[0,2],[0,-6],[-1,-6],[-2,-18],[0,-3],[0,-4],[0,-4],[0,-3],[-1,0],[0,2],[-1,2],[0,-3],[0,-3],[0,-4],[0,-3],[0,-4],[0,1],[-1,-1],[0,-2],[-1,0],[-1,4],[0,-1],[0,-4],[0,-4],[0,-1],[0,-2],[0,-7],[2,3],[0,-3],[0,-2],[-1,-2],[-1,-4],[0,-2],[-1,-3],[0,-3],[0,-3],[0,-3],[0,-2],[-2,5],[0,-4],[-1,-7],[0,-4],[0,-7],[0,-3],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,0],[0,-2],[0,-4],[0,-2],[0,-3],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[1,-1],[-1,0],[0,-1],[0,-3],[0,-1],[0,-3],[0,-1],[0,2],[-1,0],[0,1],[0,2],[0,1],[0,2],[0,3],[0,2],[-1,-1],[0,3],[0,1],[0,8],[-1,5],[0,-3],[-1,-4],[-1,1],[-1,-1],[-1,-3],[-1,-5],[0,-2],[-1,0],[-1,1],[-1,5],[0,2],[1,3],[0,-1],[1,0],[1,-6],[0,1],[0,3],[1,5],[-1,1],[-1,-1],[0,2],[-1,0],[-1,-2],[-1,0],[0,2],[1,3],[1,5],[-1,1],[-1,-2],[0,-2],[-1,0],[0,3],[-1,2],[0,-4],[-2,-5],[1,-5],[2,-7],[0,-2],[0,-7],[0,4],[0,1],[1,-1],[0,-1],[0,-2],[0,-1],[2,-5],[0,-3],[0,-2],[0,-1],[0,-2],[0,-2],[0,1],[0,-1],[0,1],[0,-1],[0,-2],[0,-1],[0,-3],[0,-2],[0,-3],[-1,-2],[0,-1],[-1,0],[0,-2],[-1,-13],[0,-5],[0,-7],[0,-3],[-1,-11],[-1,4],[0,2],[0,4],[0,6],[-1,6],[0,4],[0,4],[-1,1],[0,3],[-1,3],[0,9],[0,1],[0,1],[-1,1],[0,3],[0,1],[0,5],[1,5],[0,4],[0,8],[0,5],[0,6],[-1,5],[-1,3],[-1,0],[-1,0],[-1,0],[0,5],[1,9],[2,11],[3,9],[1,3],[3,0],[0,-1],[0,3],[0,4],[1,3],[0,1],[1,-1],[1,-4],[4,0],[0,3],[0,3],[0,4],[1,2],[1,0],[1,3],[0,7],[-1,7],[0,7],[0,6],[0,3],[0,3],[1,1],[0,1],[0,2],[0,2],[0,2],[1,2],[0,1],[1,5],[0,3],[0,5],[0,4],[0,3],[0,3],[0,5],[0,4],[0,3],[1,5],[1,4],[0,4],[0,9],[0,6],[0,3],[1,2],[0,4],[0,3],[0,11],[0,3],[-1,3],[0,2],[-2,13],[-1,2],[1,4],[0,7],[0,3],[0,2],[0,4],[0,3],[0,1],[0,5],[0,2],[0,2],[0,5],[0,2],[0,2],[0,6],[0,13],[0,5],[0,15],[1,12],[0,4],[0,8],[0,4],[0,5],[0,3],[0,2],[1,4],[0,2],[0,2],[0,2],[-2,77],[-3,77],[-1,16],[-1,2],[-1,5],[-1,12],[0,4],[1,3],[1,0],[0,1],[1,1],[0,3],[0,2],[0,4],[0,2],[0,1],[2,2],[0,-1],[1,-2],[0,-1],[1,0],[2,5],[0,1],[1,7],[1,2],[0,1],[1,0],[2,-2],[0,1],[0,2],[1,3],[0,2],[0,1],[4,23],[0,4],[0,5],[0,5],[0,5],[0,5],[0,3],[1,2],[1,1],[1,-1],[0,2],[2,22],[1,7],[1,1],[0,-3],[0,-4],[1,-4],[0,-5],[0,-3],[1,-3],[0,-2],[1,-1],[12,-22],[12,-21],[2,-8],[1,-1],[1,1],[0,5],[1,2],[1,6],[1,2],[0,5],[1,3],[0,2],[0,3],[0,3],[0,7],[0,8],[-1,7],[1,5],[-1,6],[1,24],[0,4],[0,4],[0,4],[2,15],[0,2],[0,2],[1,3],[1,1],[1,3],[1,2],[1,5],[0,1],[1,0],[0,2],[0,5],[0,3],[1,2],[1,1],[0,2],[-1,3],[0,1],[1,3],[0,3],[0,3],[0,2],[-1,1],[0,3],[1,4],[0,10],[1,3],[0,2],[-1,2],[0,2],[0,4],[0,2],[0,1],[0,1],[1,6],[0,4],[0,3],[1,0],[0,-1],[0,-1],[0,-1],[1,-1],[0,1],[0,2],[0,1],[0,2],[0,2],[0,3],[1,3],[0,1],[0,1],[1,-3],[0,2],[1,22],[1,10],[0,4],[-1,8],[0,7],[0,2],[1,0],[0,2],[1,5],[0,2],[0,2],[0,3],[1,2],[0,-4],[1,-1],[0,1],[0,5],[0,1],[0,3],[1,2],[0,2],[1,2],[0,4],[-1,4],[0,2],[1,3],[0,2],[-1,1],[0,-1],[0,4],[0,4],[1,4],[1,5],[-1,5],[0,4],[-1,3],[-1,1],[0,3],[0,4],[0,2],[1,12],[0,2],[1,7],[0,2],[0,4],[1,5],[1,4],[0,3]],[[8721,1339],[0,-2],[1,0],[0,-1],[0,1],[1,3],[0,2],[1,1],[3,-3],[0,1],[1,2],[0,2],[0,1],[1,0],[1,-1],[1,2],[1,0],[1,2],[0,2],[1,8],[0,2],[0,2],[0,1],[0,2],[0,1],[0,1],[0,3],[0,2],[0,1],[0,1],[0,1],[0,1],[0,3],[-1,4],[0,3],[0,7],[0,1],[0,2],[0,3],[1,4],[0,2],[0,4],[0,1],[0,2],[0,1],[0,1],[0,1],[1,1],[0,1],[0,-1],[1,-1],[0,1],[1,3],[0,2],[0,2],[0,4],[0,5],[0,1],[0,1],[1,5],[0,1],[1,0],[0,1],[0,1],[0,2],[0,3],[1,2],[0,2],[0,2],[1,3],[1,1],[0,1],[1,-1],[1,1],[0,2],[1,5],[0,-2],[0,-1],[1,-1],[0,-1],[1,1],[0,1],[0,1],[1,0],[0,-1],[0,-1],[0,-1],[0,-2],[1,-1],[0,-1],[0,1],[1,1],[0,2],[0,2],[0,4],[0,1],[0,1],[0,1],[1,2],[0,2],[0,1],[1,1],[0,-1],[1,-3],[2,-6],[0,-1],[0,1],[1,1],[2,-4],[0,-2],[0,-1],[-1,-2],[0,-1],[0,-2],[0,-1],[0,-1],[1,-1],[0,-1],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-4],[0,-1],[1,0],[1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-3],[0,-1],[0,-1],[0,-2],[0,-3],[0,-1],[0,-1],[-1,0],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[2,-1],[0,1],[0,1],[1,1],[0,1],[1,2],[0,2],[0,1],[1,1],[1,3],[0,-1],[1,0],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,1],[1,3],[0,1],[2,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[1,-1],[1,-1],[0,-2],[0,-1],[0,-1],[0,-1],[1,-7],[0,-2],[0,-1],[0,-2],[0,-1],[1,-1],[0,-1],[1,-2],[1,-1],[0,-1],[0,-1],[1,0],[0,1],[1,2],[0,3],[1,2],[0,4],[0,2],[1,1],[0,-1],[1,-3],[0,-2],[1,-2],[0,-3],[0,-1],[0,-1],[0,-1],[1,-1],[1,-3],[1,-2],[0,-1],[3,0],[1,1],[0,2],[0,2],[0,3],[1,1],[0,1],[0,1],[1,0],[0,-3],[1,0],[0,-1],[0,1],[1,0],[1,-4],[0,-1],[0,1],[1,2],[2,9],[1,4],[0,1],[0,2],[0,1],[1,2],[0,2],[1,8],[1,2],[0,5],[0,1],[0,1],[1,-1],[0,-1],[0,1],[1,2],[0,5],[0,1],[1,3],[0,2],[0,2],[1,3],[0,1],[0,1],[0,1],[0,1],[-1,2],[0,2],[0,1],[0,2],[0,1],[0,1],[0,2],[0,1],[0,1],[0,3],[0,1],[1,6],[0,2],[0,1],[0,1],[-1,3],[0,1],[0,1],[1,1],[0,3],[0,2],[1,1],[0,1],[0,8],[0,2],[0,1],[1,12],[0,1],[0,1],[1,1],[1,-1],[0,-2],[1,-1],[0,-3],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,1],[0,1],[1,6],[0,-2],[0,-1],[0,-2],[1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[1,-1],[0,-1],[0,-1],[0,-2],[1,-1],[0,-1],[2,1],[1,1],[0,2],[0,1],[0,2],[0,1],[2,2],[0,3],[0,2],[1,5],[0,2],[0,2],[0,2],[0,1],[1,0],[1,1],[0,1],[0,1],[2,6],[0,-2],[0,-1],[1,-1],[0,1],[0,3],[0,1],[0,1],[1,1],[1,1],[0,3],[0,1],[1,1],[0,4],[1,1],[0,1],[0,4],[0,2],[0,1],[0,1],[0,1],[1,1],[0,1],[0,2],[0,1],[1,0],[0,-1],[0,-1],[0,-1],[1,1],[0,1],[0,5],[0,2],[1,2],[0,2],[0,1],[0,1],[1,1],[0,1],[0,1],[0,2],[0,2],[0,5],[0,2],[-1,3],[0,1],[0,1],[-1,1],[-1,-1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[2,13],[0,1],[0,1],[-1,0],[-1,0],[0,1],[-1,7],[0,1],[-1,2],[0,1],[-1,0],[0,-1],[0,-4],[-1,-1],[0,1],[-1,1],[0,-1],[0,-1],[0,-1],[-2,6],[0,3],[-1,0],[0,1],[0,1],[-1,3],[-1,4],[0,2],[0,1],[0,1],[-1,0],[-1,-2],[-3,0],[0,1],[0,2],[0,1],[0,2],[0,2],[0,2],[0,1],[1,1],[0,1],[0,1],[0,1],[1,2],[0,1],[0,1],[0,1],[1,1],[0,-1],[1,1],[0,1],[0,1],[1,2],[0,2],[-1,3],[0,1],[0,2],[0,1],[0,1],[0,1],[-1,1],[0,2],[0,3],[-1,4],[0,1],[-1,0],[0,1],[-1,0],[-1,1],[0,2],[0,1],[0,3],[0,1],[0,1],[0,2],[0,2],[-1,2],[-1,4],[0,1],[0,1],[0,1],[0,2],[-1,1],[0,3],[0,1],[1,2],[0,6],[1,6],[0,1],[0,1],[1,4],[0,2],[0,1],[1,2],[0,2],[2,0],[0,1],[1,3],[0,2],[1,1],[1,-1],[2,1],[0,-1],[1,-4],[1,2],[1,0],[1,-2],[1,3],[1,4],[1,2],[0,2],[0,2],[0,2],[0,1],[1,6],[1,0],[1,-1],[1,1],[1,4],[1,2],[1,4],[0,2],[0,1],[0,2],[-1,4],[0,1],[0,1],[1,1],[0,3],[1,2],[0,1],[0,2],[0,1],[0,1],[2,4],[1,-2],[0,-1],[1,0],[1,1],[0,-1],[0,-1],[0,-1],[0,-2],[1,-1],[0,-1],[1,1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[0,-1],[1,-2],[0,-2],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-4],[0,-3],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[1,-1],[0,-2],[0,-2],[1,-2],[-1,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[1,-2],[0,-1],[0,-1],[0,-1],[1,0],[1,1],[1,1],[0,1],[0,1],[0,1],[0,1],[1,0],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[0,-4],[1,-1],[-1,-1],[0,-1],[0,-1],[-1,-2],[0,1],[-1,1],[0,-2],[0,-1],[0,-3],[0,-2],[0,-2],[-1,-4],[0,-9],[0,-2],[0,-1],[0,-2],[1,-2],[1,-3],[0,-1],[0,-3],[0,-2],[1,-1],[0,-2],[0,-2],[2,-16],[0,-1],[1,-1],[0,-1],[0,-1],[0,-4],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[-1,1],[-1,-1],[-1,-8],[0,-1],[-1,0],[-2,-2],[0,-3],[-1,-1],[0,-1],[0,-1],[-1,-8],[0,-1],[0,-1],[0,-2],[0,-3],[0,-1],[1,-1],[1,0],[1,-1],[0,-1],[0,-1],[0,-3],[0,-2],[0,-1],[-1,-11],[0,-2],[0,-1],[0,-2],[1,-2],[0,-2],[0,-1],[-2,-4],[0,-1],[0,-2],[0,-1],[0,-1],[1,1],[0,-1],[1,-2],[0,-1],[0,-1],[0,-1],[2,-3],[1,-1],[0,1],[1,0],[0,2],[0,1],[2,4],[1,-2],[0,-2],[2,-9],[0,-2],[0,-3],[1,-2],[1,1]],[[9482,6972],[0,-10],[0,-3],[0,-4],[0,-1],[0,-2],[0,-1],[0,-2],[-2,-4],[0,-2],[-1,-4],[1,-3],[1,-5],[0,-5],[0,-3],[0,-2],[0,-2],[1,-9],[-1,-4],[-1,-9],[1,-4],[1,-6],[0,-3],[0,-2],[0,-5],[0,-10],[0,-5],[1,-5],[2,-13],[1,-12],[0,-11],[-1,-8],[-2,-2],[0,5],[1,10],[1,5],[-1,7],[-1,5],[-1,4],[-2,5],[-5,7],[0,1],[0,1],[0,2],[0,1],[0,1],[0,5],[0,1],[0,4],[0,4],[0,1],[0,11],[0,2],[0,2],[0,1],[0,1],[0,6],[1,9],[0,5],[-1,7],[-1,10],[-1,6],[0,7],[1,5],[1,3],[3,11],[1,1],[0,2],[0,1],[1,-1],[0,1],[1,1],[0,2],[1,-1]],[[9483,6939],[-1,-1],[0,1],[-1,0],[1,3],[1,4],[0,3],[0,3],[0,11],[0,2],[0,6],[-1,9],[0,3],[2,2],[0,2],[0,3],[0,2],[0,2],[1,1],[1,-1],[1,-1],[1,-4],[0,-5],[1,-4],[0,-2],[0,-3],[0,-2],[-1,-3],[0,-1],[0,-5],[0,-4],[1,-4],[-1,-3],[0,-5],[0,-3],[-1,-3],[-4,-3]],[[9494,6973],[-1,-1],[-1,1],[0,3],[-1,6],[-1,3],[0,5],[0,4],[1,0],[1,1],[2,4],[2,-1],[0,-2],[0,-3],[0,-3],[0,-6],[0,-6],[-1,-4],[-1,-1]],[[9511,7249],[2,-5],[-1,0],[-3,-3],[-3,3],[0,2],[1,0],[0,1],[1,-1],[1,0],[0,1],[0,1],[1,2],[0,1],[0,-1],[1,-1]],[[9488,7284],[1,0],[1,-1],[0,-3],[0,-2],[0,-3],[0,-5],[1,-2],[-1,0],[-1,2],[-2,7],[0,1],[-1,0],[-1,0],[0,3],[-1,0],[1,1],[0,2],[0,3],[0,3],[1,0],[1,-1],[0,-4],[1,-1]],[[9463,7304],[0,-3],[0,-2],[1,-10],[0,-3],[1,-5],[-3,-1],[-2,-2],[-2,3],[0,7],[1,5],[0,8],[-2,1],[0,-3],[-1,-1],[-1,10],[0,3],[1,2],[2,0],[1,-1],[3,-4],[1,-4]],[[8829,7472],[1,-5],[0,-1],[0,-1],[2,-10],[-1,-4],[-5,-16],[-1,-2],[-1,-1],[-3,5],[-2,9],[-1,2],[-2,-1],[-2,4],[0,1],[-1,-1],[-1,-1],[1,-3],[2,-3],[1,-3],[0,-1],[-1,-1],[1,-3],[-2,-2],[-8,21],[-2,1],[1,4],[3,4],[3,7],[7,2],[3,-2],[1,1],[1,3],[1,1],[4,-3],[1,-1]],[[8580,7732],[1,-4],[1,-7],[-3,-1],[-1,-2],[-1,-6],[-1,0],[0,-1],[0,-1],[0,-1],[-3,-4],[-1,1],[-2,3],[-2,2],[-3,0],[-1,2],[-6,18],[0,2],[8,4],[7,4],[5,-5],[1,-1],[1,-3]],[[8592,7765],[-4,-3],[-6,3],[-9,5],[-9,5],[0,4],[2,10],[3,2],[6,1],[3,7],[4,5],[7,-1],[5,-15],[0,-13],[-2,-10]],[[8327,7831],[-2,-2],[-1,3],[2,7],[-1,5],[-2,4],[-1,4],[1,1],[4,8],[6,1],[6,-12],[1,-4],[0,-4],[-1,-5],[-1,-3],[1,-3],[-12,0]],[[8546,7954],[8,-12],[2,0],[0,-2],[-8,2],[-2,3],[-2,-1],[-2,4],[-4,0],[0,2],[1,3],[2,2],[5,-1]],[[8951,8034],[6,-5],[3,-5],[8,-28],[1,-2],[1,-1],[2,-8],[0,-2],[4,-4],[7,-25],[0,-5],[2,-10],[1,-9],[-2,-8],[0,-1],[0,-3],[0,-3],[0,-3],[0,-2],[0,-1],[0,-1],[1,-3],[-1,-1],[0,-1],[0,-2],[1,-1],[1,-11],[0,-6],[1,-5],[-1,-6],[-3,-5],[-7,-3],[-11,11],[-6,-2],[-6,6],[-3,1],[-11,7],[-9,7],[-6,8],[-7,10],[-7,10],[-8,-1],[-6,-6],[-4,-9],[-5,-5],[-4,3],[-1,8],[0,12],[3,13],[5,-3],[5,-2],[7,8],[3,19],[1,4],[1,6],[1,7],[0,5],[2,12],[0,4],[1,9],[1,7],[1,5],[1,4],[5,16],[1,0],[0,-1],[1,-1],[1,-1],[4,-2],[6,11],[13,2],[6,-12]],[[8460,8046],[1,-1],[1,-2],[0,-3],[-1,-5],[-3,-7],[-1,2],[-1,1],[-1,-1],[-1,-1],[0,2],[-1,2],[0,1],[-1,0],[0,1],[0,2],[0,4],[0,2],[2,0],[0,3],[-1,0],[0,1],[0,1],[1,4],[6,2],[-3,-3],[-1,-3],[4,-2]],[[9403,6188],[-1,-2],[0,-3],[-1,-6],[0,-2],[0,-1],[-1,-4],[-1,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-3],[0,-2],[0,-1],[0,-3],[1,-4],[0,-6],[1,-10],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-5,2],[-1,1],[-2,7],[-1,1],[-1,0],[-5,-9],[-6,-2],[-1,-2],[-1,-2],[0,-3],[-2,-9],[0,-2],[-5,-9],[0,-1],[-1,-1],[0,-2],[0,-1],[0,-1],[1,-2],[0,-4],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[-7,-2],[-2,1],[-2,5],[-2,6],[0,1],[0,1],[0,2],[0,1],[1,6],[0,2],[0,1],[0,1],[0,2],[0,1],[0,1],[-1,2],[-4,13],[-1,5],[-2,17],[0,1],[-1,1],[0,1],[-1,-1],[0,-1],[-1,-1],[0,-2],[0,-3],[0,-2],[-1,-1],[-1,-3],[0,-2],[-1,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-2],[-1,-3],[-1,-4],[0,-1],[-1,1],[0,1],[0,2],[-1,1],[-9,3],[0,1],[0,1],[0,1],[0,1],[0,1],[-1,2],[-1,1],[-1,1],[0,1],[0,1],[0,1],[-1,2],[1,1],[0,1],[-1,1],[0,1],[0,1],[-1,1],[-2,-3],[-2,-3],[0,-1],[0,-1],[-1,1],[-1,5],[-1,0],[-2,-1],[-3,-5],[-1,0],[0,1],[0,1],[0,2],[0,1],[0,1],[0,2],[0,1],[0,1],[-1,2],[-1,3],[-1,1],[-1,0],[-1,-1],[-1,-4],[-2,-1],[0,-1],[0,-2],[0,-1],[0,-1],[1,-1],[-1,-1],[0,-2],[-4,-4],[0,1],[0,1],[0,2],[-1,3],[0,2],[0,1],[0,1],[-1,1],[0,1],[-5,6],[-1,2],[0,3],[-1,1],[0,1],[-1,-1],[-3,-7],[0,-1],[-1,-2],[0,-1],[0,-5],[0,-3],[0,-3],[-1,-1],[-3,-17],[0,-1],[0,-1],[3,-3],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-3],[0,-2],[0,-2],[0,-3],[0,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-3],[0,-2],[-1,-1],[0,-3],[0,-6],[-1,-1],[0,-1],[0,-1],[-2,-4],[0,-1],[-1,-3],[0,-1],[-1,0],[-1,1],[0,1],[0,1],[0,2],[0,5],[0,3],[0,1],[0,1],[-1,1],[0,1],[0,1],[-1,2],[-1,2],[-2,1],[-3,-3],[-3,3],[0,-1],[-1,-4],[0,-1],[-3,-3],[-1,-2],[0,-2],[0,-2],[-1,-5],[0,-2],[0,-3],[0,-4],[0,-1],[0,-2],[0,-3],[0,-1],[0,-1],[2,-5],[0,-2],[0,-1],[0,-1],[0,-2],[-1,-5],[0,-2],[0,-1],[0,-2],[0,-2],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[-2,-9],[0,-2],[0,-2],[0,-3],[0,-1],[-1,-6],[0,-2],[0,-1],[1,-5],[0,-1],[0,-1],[1,-1],[1,-7],[0,-1],[1,-1],[0,-1],[0,-1],[0,-2],[-1,-4],[-1,-7],[-1,-2],[-2,0],[-3,-8],[-2,-4],[0,-1],[-1,-1],[0,-5],[0,-1],[-1,-1],[-1,2],[-1,0],[0,-1],[-3,-5],[-2,1],[-2,-4],[0,-1],[-1,0],[-1,1],[0,-1],[-1,-1],[0,-2],[0,-2],[0,-3],[0,-1],[0,-1],[0,-2],[-1,-2],[0,-1],[0,-2],[-1,-2],[0,-2],[0,-1],[0,-2],[0,-6],[0,-2],[0,-1],[1,-1],[0,-3],[0,-1],[1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-3],[-1,-2],[0,-1],[0,-2],[0,-1],[0,-2],[-3,-7],[0,-2],[0,-1],[-1,-4],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[1,-2],[0,-3],[0,-1],[1,-2],[0,-3],[0,-2],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[3,-3],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[2,-9],[0,-3],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[-1,-2],[-5,-14],[0,-2],[0,-2],[-2,-6],[-3,-10],[0,-2],[0,-2],[0,-2],[0,-3],[0,-1],[0,-1],[1,-10],[0,-3],[1,-4],[0,-1],[0,-2],[-1,-1],[-4,-6],[0,-2],[-1,-1],[-1,-1],[-1,1],[-1,1],[0,1],[0,1],[0,1],[0,4],[1,6],[0,3],[0,2],[0,1],[0,1],[0,2],[0,1],[-1,5],[0,2],[0,2],[-1,2],[-1,1],[-5,-8],[-10,-9],[-1,-2],[0,-6],[-1,-22],[0,-3],[0,-1],[-1,0],[0,4],[0,1],[-1,1],[0,1],[0,1],[0,2],[0,2],[0,2],[-1,1],[0,1],[-1,-2],[-1,1],[-1,4],[-1,0],[-1,-1],[-4,-10],[-9,1],[-1,-1],[0,-1],[0,-1],[0,-4],[1,-1],[0,-8],[0,-1],[0,-2],[0,-1],[-1,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-6],[0,-1],[0,-3],[0,-1],[0,-1],[0,-1],[-4,3],[-2,5],[-3,3],[-2,0],[0,1],[-1,2],[-1,4],[0,3],[-1,1],[0,2],[0,1],[0,1],[0,2],[0,1],[0,1],[-2,8],[0,2],[0,6],[-1,1],[0,3],[0,1],[1,2],[0,6],[1,6],[0,2],[0,3],[0,3],[0,1],[0,3],[0,2],[0,1],[0,1],[0,1],[-5,13],[-2,3],[-1,0],[-3,-2],[-2,-5],[-1,-3],[-3,-15],[-1,-5],[0,-1],[0,-1],[-1,0],[-2,4],[-1,0],[0,-1],[-1,-1],[0,-1],[0,-3],[0,-1],[0,-2],[0,-1],[0,-1],[-1,0],[-1,1],[-1,3],[-1,3],[0,1],[-5,0],[-3,-4],[-3,2],[-1,-1],[-1,-2],[-1,0],[0,2],[-2,9],[-1,2],[0,2],[0,2],[0,2],[0,1],[-2,6],[-1,3],[-2,4],[-1,-1],[-2,-11],[0,-2],[-1,-1],[0,-1],[0,-2],[0,-3],[0,-1],[0,-2],[-1,-1],[-2,-2],[-1,-1],[0,-2],[0,-2],[0,-1],[0,-2],[1,-2],[0,-1],[0,-1],[0,-2],[-1,-1],[-1,-2],[0,-2],[0,-1],[1,-1],[0,-1],[0,-2],[0,-1],[1,0],[1,-1],[0,-1],[0,-1],[0,-3],[1,-1],[0,-1],[1,0],[0,-1],[0,-1],[0,-3],[0,-1],[0,-1],[1,-2],[1,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-4],[0,-2],[0,-3],[0,-1],[0,-2],[-1,-1],[0,-2],[0,-1],[-1,-3],[0,-3],[-1,-3],[0,-1],[0,-1],[0,-1],[0,-2],[0,-8],[0,-2],[0,-1],[0,-1],[-1,-2],[-2,-10],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[-1,-2],[0,-3],[0,-1],[-1,-1],[0,-1],[-1,0],[0,2],[-1,4],[0,1],[0,1],[-1,-1],[0,-1],[-1,-5],[0,-1],[0,1],[-1,2],[0,1],[-1,1],[0,2],[0,1],[0,1],[0,2],[1,1],[0,2],[0,1],[0,1],[0,1],[-1,1],[0,1],[0,-1],[-1,-1],[-1,0],[-1,3],[0,2],[-1,1],[0,1],[0,1],[0,2],[0,1],[1,3],[0,1],[0,2],[0,2],[0,1],[0,1],[1,2],[0,1],[0,1],[0,2],[0,1],[-1,4],[0,3],[0,1],[0,1],[0,1],[-1,1],[-8,-10],[0,-1],[-1,-1],[0,-1],[0,-2],[1,-1],[0,-1],[0,-2],[1,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-2],[-1,-1],[0,-1],[-1,0],[0,1],[-1,2],[-1,3],[-1,3],[0,2],[-4,12],[-13,15],[-2,1],[-1,0],[0,-1],[-1,-1],[0,-2],[0,-2],[0,-5],[0,-2],[-1,-1],[0,-4],[0,-2],[0,-3],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-2],[0,-2],[0,-1],[-2,-5],[-1,-2],[0,-2],[0,-3],[0,-7],[0,-1],[0,-2],[-2,-7],[-1,-3],[-4,-5],[-1,-3],[-2,-3],[-1,-1],[0,-2],[-1,-3],[-1,-1],[-4,-5],[-1,-1],[0,-1],[0,-1],[0,-2],[1,-3],[1,-3],[0,-1],[0,-1],[0,-1],[0,-3],[0,-1],[0,-2],[0,-6],[0,-1],[0,-2],[0,-1],[0,-1],[1,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[-1,-2],[0,-1],[-1,-2],[0,-1],[0,-1],[-1,-2],[0,-6],[-1,-5],[0,-2],[-1,-10],[-1,-2],[0,-2],[0,-1],[0,-6],[0,-6],[0,-1],[0,-8],[-1,-12],[-1,-4],[0,-2],[0,-2],[0,-2],[0,-3],[0,-2],[0,-2],[-2,-11],[0,-1],[-1,-2],[0,-4],[0,-8],[0,-4],[0,-3],[0,-1],[0,-2],[1,-14],[0,-1],[0,-2],[0,-3],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[-1,-1],[0,-1],[0,-2],[1,-2],[0,-1],[0,-1],[0,-1],[1,0],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-3],[0,-2],[0,-1],[0,-1],[1,-3],[0,-2],[0,-1],[0,-1],[0,-9],[0,-2],[0,-1],[0,-2],[0,-1],[0,-3],[-1,-14],[0,-4],[0,-2],[0,-2],[1,-4],[1,-3],[0,-1],[0,-1],[0,-2],[0,-3],[0,-2],[0,-2],[0,-1],[1,-4],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-1],[1,-2],[0,-1],[0,-1],[1,-5],[0,-2],[0,-1],[0,-1],[-1,-3],[-1,-5],[0,-2],[-1,0],[-1,2],[-1,3],[0,1],[-1,0],[0,-1],[-1,-2],[-1,-6],[0,-1],[-1,0],[0,1],[0,-1],[-1,0],[0,-2],[0,-1],[1,-1],[0,-5],[1,-1],[0,-1],[0,-1],[0,-2],[0,-2],[-1,-6],[0,-3],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[1,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-2],[0,-1],[-1,-1],[0,-1],[1,-1],[0,-2],[1,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[-1,-1],[0,-1],[0,-3],[0,-1],[-1,-3],[-1,-1],[0,1],[-1,2],[-1,0],[0,-1],[0,-1],[1,-2],[0,-1],[0,-1],[0,-3],[0,-2],[0,-1],[0,-2],[0,-2],[0,-2],[-1,-3],[-1,0],[-1,1],[0,-1],[0,-1],[-1,-2],[0,-4],[-1,-2],[0,-1],[-1,0],[0,1],[-1,-1],[0,-3],[-1,-1],[0,-1],[0,-2],[0,-2],[1,-8]],[[9020,5128],[-1,-1],[0,-1],[-1,-1],[0,-5],[0,-3],[-1,-1],[0,-2],[-1,-2],[-1,-1],[0,-8],[0,-4],[-1,-1],[0,-1],[1,-1],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[-2,-5],[0,-1],[-1,0],[-1,-1],[0,-2],[-1,-1],[0,-1],[0,-3],[0,-2],[0,-1],[-2,-1],[0,-1],[0,-1],[-1,-1],[0,-2],[0,-1],[0,-1],[0,-3],[0,-2],[-1,0],[-1,2],[-1,0],[0,-1],[0,-1],[-1,0],[-1,0],[-2,-2],[0,1],[-1,2],[0,2],[-1,2],[0,1],[0,2],[0,1],[0,3],[0,1],[0,1],[0,3],[0,2],[0,1],[0,1],[0,2],[0,1],[1,10],[0,1],[0,1],[0,1],[-1,1],[0,1],[0,1],[0,1],[1,3],[0,1],[0,1],[0,2],[0,1],[0,2],[0,2],[0,2],[0,3],[-1,4],[0,1],[-1,-2],[-1,-6],[0,-1],[-1,0],[-1,0],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-3],[0,-1],[0,-2],[0,-1],[-1,-1],[0,-1],[0,-3],[0,-1],[0,-1],[-1,-1],[-1,0],[0,1],[0,2],[-1,1],[0,2],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[-1,1],[0,1],[0,1],[0,1],[0,1],[1,4],[0,1],[1,1],[-1,1],[0,3],[0,1],[0,1],[0,1],[1,0],[0,1],[0,1],[0,1],[0,1],[-1,2],[0,1],[0,1],[-1,0],[-1,0],[0,-1],[-1,0],[-1,3],[0,1],[-1,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[-1,0],[-3,7],[0,1],[0,1],[0,2],[0,1],[0,4],[0,3],[0,1],[0,1],[-1,1],[0,1],[-1,-2],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-2],[-1,-10],[0,-1],[0,-2],[0,-2],[0,-1],[-2,-9],[0,-2],[-1,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-4],[0,-1],[0,-1],[0,-1],[-1,-2],[-1,1],[0,2],[-1,2],[0,1],[-1,4],[0,1],[0,1],[-1,1],[0,-1],[-1,0],[0,3],[-1,1],[0,-1],[-1,1],[-1,1],[0,-1],[0,-2],[-1,-2],[0,-1],[0,-1],[0,-1],[-2,0],[0,-1],[-1,0],[0,-2],[0,-2],[0,-1],[0,-2],[-1,-1],[0,-1],[0,-1],[-1,-3],[0,-1],[0,-1],[-1,0],[0,2],[-1,2],[0,2],[0,1],[0,2],[0,1],[1,3],[0,1],[0,1],[0,2],[0,1],[-1,1],[0,1],[0,1],[-2,2],[-1,1],[0,2],[0,1],[0,2],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,1],[-2,7],[0,2],[-1,1],[0,2],[0,1],[1,2],[0,1],[-1,1],[0,1],[-1,1],[-1,-1],[0,-1],[0,-1],[0,-4],[-1,-1],[0,-4],[-1,-2],[0,-2],[-1,0],[-1,1],[-1,0],[0,-1],[-1,-1],[0,1],[-1,2],[0,1],[0,2],[0,1],[1,3],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[-1,0],[-1,0],[0,-1],[-1,0],[0,-2],[-1,0],[0,1],[0,2],[0,1],[-1,2],[0,1],[-1,2],[0,2],[0,2],[0,1],[0,1],[0,2],[-1,1],[0,1],[-1,1],[-1,2],[0,1],[0,1],[0,2],[0,2],[0,1],[0,6],[-1,3],[0,4],[0,1],[0,2],[0,1],[-2,6],[-1,1],[0,1],[0,1],[0,6],[0,1],[0,1],[-1,1],[-1,8],[0,3],[0,2],[-1,9],[0,1],[0,1],[-1,-2],[-1,0],[0,1],[-1,1],[-1,-2],[0,-1],[0,-3],[0,-1],[0,-1],[-1,-1],[0,1],[-1,1],[0,1],[0,1],[0,1],[-1,1],[0,1],[0,2],[0,1],[0,5],[0,2],[0,1],[0,2],[0,1],[-1,-2],[-1,0],[0,1],[-1,1],[0,3],[0,1],[-1,0],[0,-1],[-1,0],[0,1],[0,1],[0,1],[0,1],[-1,5],[0,1],[0,1],[0,1],[-2,1],[0,1],[0,1],[-1,3],[-2,-8],[0,-1],[0,-1],[0,-2],[0,-1],[1,-3],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[-1,-4],[0,-1],[0,-5],[0,-2],[-1,-1],[0,-1],[-1,-1],[-1,2],[-1,2],[-1,0],[-2,-2],[0,-2],[-1,-4],[0,-3],[-1,-1],[0,-9],[0,-3],[0,-3],[0,-6],[-1,-3],[0,-2],[0,-2],[0,-2],[0,-1],[1,-3],[0,-2],[0,-1],[0,-2],[-1,-4],[0,-2],[0,-1],[1,-1],[0,-1],[0,-2],[0,-1],[-1,-5],[0,-1],[1,-5],[-1,-1],[0,-2],[-1,-5],[0,-1],[0,-1],[0,-1],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-3],[-1,-3],[0,-2],[-1,-1],[-1,0],[0,-1],[0,-1],[0,-2],[0,-3],[0,-1],[0,-1],[-1,-2],[0,-2],[0,-1],[-1,0],[-1,2],[-1,1],[-1,0],[0,-1],[0,-2],[-1,-6],[0,-2],[0,-3],[0,-3],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-8],[-1,-5],[0,-1],[0,-2],[-1,-3],[0,-1],[-1,-2],[0,-2],[0,-2],[0,-3],[-2,-7],[0,-2],[-1,-2],[0,-1],[-1,-2],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-1],[0,-1],[0,-2],[0,-2],[0,-6],[0,-1],[-1,-3],[0,-2],[0,-1],[0,-3],[0,-2],[0,-2],[-1,-5],[0,-2],[-1,-10],[0,-1],[-1,-1],[0,-1],[0,-1],[-1,-1],[0,1],[-2,4],[-1,-1],[0,-2],[0,-1],[0,-1],[-1,-2],[-1,-4],[-1,-3],[-1,0],[-3,4],[-1,0],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-1],[0,-2],[-1,-1],[0,-2],[-2,-9],[0,-2],[0,-2],[-1,-1],[-1,1],[0,1],[0,1],[0,1],[0,1],[-1,0],[-2,-6],[-1,0],[0,1],[0,2],[0,3],[0,1],[0,1],[0,-1],[-1,-1],[-1,-6],[0,-7],[-1,-4],[0,-2],[0,-1],[0,-1],[0,-2],[0,-3],[0,-3],[1,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-3],[-1,-1],[-2,-8],[0,-1],[-2,0],[-1,-1],[0,-2],[-1,-5],[-1,-1],[-2,3],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[1,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-5],[0,-1],[0,-2],[-1,-2],[0,-1],[0,-1],[-2,0],[0,-1],[0,-1],[-1,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-3],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-1],[0,-3],[0,-1],[1,-2],[0,-1],[0,-2],[0,-3],[0,-1],[0,-1],[0,-1],[1,-4],[0,-1],[0,-1],[0,-2],[0,-3],[0,-1],[0,-1],[1,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,-1],[0,-1],[1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[1,-1],[0,-2],[0,-1],[0,-2],[-1,-7],[0,-3],[0,-2],[0,-1],[0,-2],[-1,-3],[0,-1],[-1,0],[0,-1],[-1,0],[0,-1],[0,-2],[0,-3],[0,-2],[1,-2],[0,-2],[0,-2],[0,-1],[-1,-4],[0,-1],[-2,-2],[0,-1],[0,-1],[0,-2],[0,-1],[1,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[1,-2],[0,-1],[0,-1],[0,-1],[1,-6],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-7],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-4],[0,-1],[-1,-4],[0,-3],[-1,-3],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[1,-3],[0,-1],[0,-2],[0,-1],[0,-1],[0,-3],[-1,-7],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[1,-5],[0,-1],[1,-2],[0,-1],[-1,-2],[0,-5],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-4],[0,-1],[0,-2],[0,-1],[0,-1],[1,-5],[0,-1],[0,-1],[0,-3],[-1,-5],[-1,-5],[0,-3],[0,-2],[0,-5],[-1,-3],[0,-2],[0,-1],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-10],[0,-1],[0,-2],[0,-1],[0,-1],[1,-1],[1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-2],[0,-3],[-1,-5],[-1,0],[-2,1],[-1,0],[-1,0],[0,1],[0,1],[0,1],[0,2],[0,3],[1,3],[0,1],[0,2],[0,1],[0,1],[-1,5],[0,3],[0,2],[0,1],[-1,0],[0,-2],[0,-2],[0,-2],[0,-1],[0,-1],[-1,-2],[-1,-2],[-1,0],[0,1],[-1,2],[0,1],[-6,-6],[-1,-1],[0,-3],[-4,-18],[0,-3],[0,-2],[0,-3],[0,-2],[0,-2],[1,-3],[0,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-1],[0,-1],[-1,0],[-1,2],[0,-1],[-1,-1],[0,-2],[0,-2],[-1,-5],[0,-2],[0,-1],[-2,-5],[0,-3],[-1,-3],[0,-2],[-3,1],[0,-3],[-1,-2],[0,-1],[-2,1],[-4,-8],[0,-1],[-1,-2],[0,-5],[0,-1],[-1,-1],[-3,0],[0,1],[0,1],[0,2],[-1,1],[0,1],[0,1],[-2,-1],[-1,1],[-1,1],[0,2],[-1,0],[-2,-1],[0,1],[0,1],[-1,1],[0,5],[0,1],[0,1],[-1,0],[-1,-1],[0,-3],[-1,-1],[-1,0],[0,1],[0,2],[-1,1],[-2,-1],[0,1],[-1,2],[0,9],[-1,1],[0,2],[-2,0],[0,1],[-1,1],[0,1],[0,3],[0,2],[-1,1],[0,2],[-1,0],[0,1],[-1,0],[-1,-1],[-1,-4],[-1,-1],[-1,-1],[0,-1],[-1,-2],[0,-1],[0,-2],[0,-1],[-1,-3],[0,-1],[0,-1],[0,-1],[-1,-4],[-3,-12],[0,-1],[0,-2],[-1,-8],[0,-2],[-1,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[-1,-3],[0,-3],[0,-1],[0,-1],[0,-1],[-1,-2],[0,-6],[-1,-1],[0,-1],[0,-2],[0,-1],[1,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-3],[0,-1],[-1,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[-1,-1],[-1,0],[-1,0],[0,1],[-1,3],[0,1],[0,1],[-2,0],[-1,3],[-1,0],[0,-1],[-2,-5],[-1,3],[0,1],[-1,1],[-1,0],[0,1],[0,1],[0,1],[0,2],[1,3],[0,1],[0,2],[0,1],[-1,1],[0,2],[-1,0],[-1,-3],[0,-1],[-2,-1],[-1,0],[0,-2],[-1,-2],[0,-1],[0,-3],[0,-1],[0,-1],[-1,1],[-2,3],[0,1],[0,1],[0,1],[0,1],[-1,3],[0,1],[-3,0],[-1,2],[-1,1],[-1,4],[-2,6],[-1,1],[-3,-2],[-2,0],[-2,-5],[-1,-1],[-4,3],[-1,1],[0,1],[0,1],[0,2],[0,2],[0,3],[0,2],[0,1],[-1,1],[0,1],[-2,-4],[-1,0],[-1,-1],[0,-1],[0,-1],[0,-2],[-1,-4],[-1,-1],[0,-3],[-1,-1],[0,-1],[-1,0],[-1,1],[0,1],[0,1],[0,-1],[-1,-2],[0,-2],[0,-2],[0,-1],[-1,0],[-1,2],[-1,-1],[0,-4],[0,-1],[-2,-4],[0,-2],[-2,-15],[0,-1],[-1,0],[-1,4],[-1,0],[0,-1],[0,-2],[0,-1],[0,-2],[1,-5],[0,-1],[0,-1],[0,-2],[-1,-1],[0,-3],[0,-1],[-1,-2],[0,-1],[1,-1],[0,-4],[1,-2],[0,-1],[0,-1],[-1,-8],[0,-2],[0,-1],[-1,-2],[0,-3],[0,-5],[0,-2],[0,-1],[0,-1],[-1,0],[-1,1],[0,2],[-1,1],[-3,0],[-1,-18],[0,-2],[0,-3],[-1,-2],[0,-1],[0,-1],[1,-2],[0,-1],[0,-1],[1,-1],[0,-2],[0,-1],[0,-1],[0,-5],[-1,-13],[0,-3],[0,-1],[0,-1],[1,-2],[-1,-1],[0,-3],[-2,-5],[0,-2],[-1,-2],[-1,-8],[0,-2],[0,-2],[0,-2],[-1,-3],[0,-1],[0,-1],[0,-1],[0,-1],[1,-4],[0,-1],[0,-1],[0,-1],[0,-1],[-1,0],[-2,3],[0,2],[0,1],[-1,0],[-1,-2],[-3,0],[0,-1],[0,-1],[0,-1],[1,-6],[0,-2],[1,-1],[0,-1],[-1,-3],[0,-3],[0,-3],[1,-1],[0,-1],[0,-5],[0,-1],[0,-1],[0,-2],[1,-1],[0,-2],[0,-1],[-1,-3],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-6],[0,-5],[0,-2],[0,-1],[0,-2],[-1,0],[0,1],[0,4],[-1,0],[0,1],[0,-1],[-1,-2],[-1,-8],[0,-1],[0,-1],[-1,-1],[0,-2],[0,-2],[0,-1],[1,-4],[0,-1],[0,-2],[-1,-2],[0,-3],[0,-3],[0,-2],[0,-1],[0,-1],[3,-5],[0,-2],[0,-1],[1,-1],[-1,-2],[0,-2],[-1,-3],[0,-2],[0,-2],[-1,0],[-1,5],[0,1],[-1,1],[0,3],[0,2],[-1,3],[-2,7],[0,4],[-1,1],[-1,-1],[-1,1],[0,1],[0,1],[0,1],[-1,1],[0,1],[0,4],[0,1],[0,1],[-1,0],[-1,0],[-1,-1],[-1,0],[-1,2],[-1,-2],[-1,0],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[1,-2],[0,-2],[1,-2],[0,-3],[0,-3],[0,-1],[0,-1],[0,-1],[2,-3],[1,-1],[0,-1],[0,-1],[0,-1],[0,-4],[0,-1],[0,-2],[0,-1],[0,-1],[-1,0],[0,-1],[0,-2],[0,-1],[0,-1],[3,-16],[1,-1],[1,0],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[-1,-9],[0,-1],[0,-1],[0,-1],[1,0],[1,0],[0,-1],[0,-1],[1,-2],[1,-3],[0,-1],[0,-3],[0,-2],[0,-1],[0,-2],[0,-1],[1,-1],[0,-3],[1,-1],[1,-6],[1,-8],[0,-1],[0,-2],[0,-1],[0,-1],[0,-3],[0,-1],[0,-2],[-1,-3],[0,-1],[-1,0],[0,-1],[0,-2],[0,-1],[0,-2],[0,-3],[0,-1],[0,-1],[0,-1],[-2,3],[-1,0],[0,-1],[-1,-1],[0,-1],[-2,-5],[0,-1],[-1,1],[0,1],[-1,-1],[0,-3],[0,-2],[0,-1],[0,-1],[1,-3],[0,-1],[1,-1],[-1,-1],[0,-2],[-1,-7],[-1,-2],[-1,-3],[0,-4],[-1,-3],[0,-6],[0,-3],[0,-3],[0,-3],[0,-1],[0,-1],[0,-6],[0,-5],[1,-4],[0,-1],[-1,-1],[0,-1],[-1,0],[0,1],[-1,0],[0,-2],[0,-1],[0,-2],[0,-4],[0,-1],[-1,0],[-1,2],[-1,0],[0,-1],[-1,0],[-1,1],[0,2],[0,3],[-1,3],[0,1],[0,2],[0,1],[-1,1],[0,-1],[-1,-2],[0,-3],[0,-7],[-1,-3],[0,-1],[0,-1],[-1,-1],[-1,1],[0,-2],[1,-3],[0,-6],[1,-5],[0,-4],[0,-4],[0,-1],[1,0],[0,1],[0,3],[0,1],[1,2],[0,1],[1,0],[0,-1],[0,-1],[0,-1],[0,-2],[0,-3],[0,-1],[1,-1],[4,-11],[0,-1],[1,-1],[0,-1],[-1,-1],[0,-1],[-1,0],[0,-1],[0,-1],[0,-2],[0,-1],[1,-1],[0,-2],[0,-1],[0,-1],[1,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[1,-1],[0,-2],[0,-4],[0,-1],[0,-1],[0,-1],[1,0],[0,2],[0,-1],[1,-1],[0,-1],[0,-2],[0,-2],[0,-1],[1,-2],[0,-1],[0,-1],[0,-5],[0,-1],[0,-1],[0,-1],[1,-1],[1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[1,0],[0,-1],[0,-1],[0,-2],[0,-3],[0,-2],[0,-1],[0,-1],[1,-1],[0,-1],[0,-2],[0,-4],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-3],[0,-2],[0,-1],[-1,0],[0,-1],[0,-2],[-1,-2],[0,-1],[0,-2],[0,-1],[0,-2],[-1,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-1],[0,-2],[0,-2],[0,-3],[0,-2],[0,-2],[0,-2],[-1,-2],[0,-2],[0,-3],[0,-1],[0,-1],[1,0],[0,-1],[0,-1],[0,-1],[0,-1],[1,-4],[0,-1],[0,-1],[0,-1],[-1,-1],[0,1],[0,1],[0,1],[-1,1],[0,-1],[0,-1],[-1,-3],[0,1],[0,1],[0,2],[0,5],[0,1],[0,1],[0,1],[0,1],[-1,0],[0,1],[-1,1],[0,1],[0,1],[1,2],[-1,1],[0,1],[0,1],[0,1],[-1,1],[-1,-1],[0,-1],[0,-2],[-1,-1],[0,-1],[-1,-10],[0,-1],[0,-2],[0,-2],[1,-1],[0,-1],[0,-1],[0,-2],[0,-2],[-1,-2],[0,-1],[0,-1],[-1,0],[0,1],[0,2],[0,1],[0,1],[0,3],[0,2],[0,1],[0,1],[0,1],[-1,2],[0,1],[0,2],[0,1],[-2,-1],[-1,-1],[-1,-5],[0,-4],[-1,-3],[0,-6],[0,-1],[-1,-4],[0,-9],[0,-2],[-1,-1],[1,-2],[0,-2],[0,-2],[0,-7],[0,-4],[0,-1],[0,-2],[0,-3],[0,-12],[-1,-2],[0,-1],[0,-2],[-2,-7],[0,-2],[-1,-5],[0,-1],[0,-2],[0,-4],[0,-2],[0,-1],[0,-2],[0,-2],[0,-3],[0,-2],[0,-2],[0,-3],[-1,-7],[0,-4],[0,-10],[0,-2],[0,-2],[0,-1],[-1,-1],[0,-1],[-1,-2],[0,-2],[0,-2],[0,-14],[0,-2],[0,-1],[0,-2],[1,-5],[0,-1],[1,-5],[0,-2],[0,-3],[0,-3],[0,-3],[0,-5],[0,-6]],[[8080,7991],[0,3],[0,3],[-1,2],[0,2],[0,3],[1,2],[0,2],[0,1],[-1,1]],[[8091,8035],[4,-14],[10,-17],[9,-9],[7,-1],[3,5],[3,-1],[3,5],[3,1],[3,8],[1,3],[0,4],[0,2],[0,8],[-1,2],[0,1],[0,1],[0,2],[0,1],[0,2],[0,1],[0,1],[-1,5],[0,2],[-1,3],[0,3],[0,1],[-1,2],[0,2],[0,3],[0,3],[0,2],[1,-1],[3,-13],[4,-12],[1,-2],[0,-2],[0,-2],[0,-3],[0,-3],[1,1],[3,-23],[1,-3],[2,-14],[0,-2],[1,-4],[0,-6],[0,-6],[0,-5],[-1,-2],[-1,-2],[-1,0],[-1,-1],[0,-3],[-2,-8],[-1,-10],[0,-2],[-1,-4],[0,-3],[0,-1],[1,-3],[1,-1],[0,-2],[0,-2],[1,-4],[1,-5],[1,-2],[3,-2],[1,-2],[1,-3],[0,-1],[0,-1],[0,-10],[0,-3],[0,-2],[0,-4],[0,-2],[0,-5],[0,-4],[0,-4],[0,-5],[0,-4],[0,-8],[1,-5],[-1,-8],[-1,-10],[0,-9],[1,-7],[-5,-17],[-2,-3],[-2,-2],[-2,-3],[0,-5],[0,-5],[1,-4],[0,-4],[1,-2],[0,-3],[0,-5],[0,-4],[0,-3],[2,-8],[3,-5],[2,-2],[2,-3],[8,-6],[8,-5],[-1,5],[-9,7],[-9,7],[-4,14],[-1,2],[0,8],[0,2],[-1,1],[0,2],[-1,6],[1,4],[0,3],[2,7],[3,3],[4,15],[1,6],[0,10],[0,8],[2,13],[0,6],[-2,14],[0,7],[0,2],[0,4],[0,2],[1,2],[3,10],[0,4],[1,2],[0,1],[4,-2],[2,4],[2,1],[1,3],[-1,3],[-4,-3],[-2,1],[0,1],[-3,8],[-1,2],[-1,3],[-2,8],[-1,4],[0,9],[-1,3],[1,1],[5,3],[6,4],[6,15],[8,-1],[8,0],[1,2],[4,2],[7,16],[8,8],[13,-5],[8,-3],[12,0],[13,-15],[11,-6],[8,-2],[5,3],[7,-1],[7,0],[10,-8],[3,-7],[3,-9],[-2,-7],[-2,-3],[-1,-1],[-2,3],[-1,0],[0,-4],[-1,0],[-1,2],[-1,1],[0,1],[0,4],[0,1],[-1,1],[-2,0],[-1,-2],[-1,-7],[-1,-1],[0,-1],[-1,-2],[0,-3],[0,-4],[0,-4],[1,-2],[0,-1],[1,0],[0,-3],[-2,-6],[0,-2],[0,-4],[0,-2],[0,-3],[0,-3],[0,-1],[1,-10],[0,-1],[1,-1],[0,-2],[1,-2],[0,-1],[12,-22],[5,-3],[5,-8],[3,1],[3,-3],[1,-1],[1,-2],[0,-5],[1,-6],[0,-1],[1,-2],[1,1],[1,-1],[0,-4],[1,-5],[11,-11],[11,-10],[1,3],[0,4],[-10,6],[-10,7],[-1,3],[-1,4],[0,6],[0,2],[13,-8],[5,1],[7,-3],[4,-7],[7,-4],[2,4],[3,1],[4,6],[8,-1],[2,-3],[1,4],[5,-3],[0,-3],[-1,-3],[0,-3],[0,-1],[3,-8],[11,-4],[3,-9],[2,-3],[3,0],[-11,27],[0,1],[0,3],[1,0],[13,-20],[1,0],[3,5],[-8,11],[-7,10],[0,1],[0,1],[1,1],[0,1],[1,0],[-1,4],[0,2],[4,3],[4,-6],[3,-2],[0,-1],[0,-3],[0,-3],[1,-2],[7,-10],[1,1],[1,1],[0,2],[1,2],[0,4],[1,2],[1,3],[1,2],[-1,3],[0,3],[0,3],[0,2],[1,8],[0,3],[1,4],[0,1],[1,-2],[0,-1],[0,-2],[0,-1],[1,0],[0,1],[0,2],[0,3],[0,1],[0,1],[0,2],[-1,0],[-1,-1],[0,1],[0,2],[-1,8],[1,5],[1,3],[3,2],[-1,-5],[1,-4],[1,0],[2,3],[-1,3],[0,2],[1,1],[0,1],[0,1],[0,3],[-1,1],[-2,-1],[0,1],[-1,2],[0,2],[0,1],[0,1],[-1,1],[0,2],[1,0],[0,2],[0,2],[-3,3],[0,2],[-1,3],[-1,9],[0,2],[2,7],[0,3],[-1,1],[-2,-1],[-1,0],[-1,4],[0,2],[2,1],[0,2],[0,2],[0,3],[-1,1],[0,1],[1,2],[1,0],[1,-2],[0,-2],[0,-3],[1,-1],[0,3],[0,5],[-1,5],[-2,5],[0,1],[0,1],[0,1],[-1,4],[0,3],[0,4],[1,1],[2,-2],[0,4],[-1,2],[0,3],[2,-3],[1,0],[-1,4],[0,1],[0,2],[-1,3],[1,2],[0,3],[-1,2],[0,3],[4,6],[1,5],[1,2],[1,-1],[0,-4],[-1,-3],[0,-2],[0,-1],[0,-1],[0,-3],[0,-2],[-1,-1],[2,-1],[1,-1],[0,-2],[-1,-2],[0,-1],[1,-3],[1,0],[3,5],[2,0],[1,3],[1,-2],[1,0],[1,4],[-1,3],[-1,1],[-1,2],[0,5],[0,1],[0,2],[0,2],[0,2],[0,2],[1,5],[-1,1],[0,2],[0,1],[-1,3],[1,0],[1,0],[0,-1],[1,-3],[1,-5],[1,-2],[0,-1],[0,-4],[-1,-2],[-1,-2],[2,-2],[1,1],[1,5],[0,3],[3,1],[0,2],[0,4],[0,2],[0,1],[2,3],[0,2],[1,4],[0,2],[1,0],[1,-1],[0,-2],[-1,-1],[0,-1],[2,0],[1,-2],[0,-1],[1,-2],[0,-2],[1,-1],[2,-4],[1,2],[1,-1],[0,-2],[0,-4],[1,-2],[3,4],[0,-3],[0,-1],[0,-1],[0,-2],[1,-1],[1,0],[0,-1],[0,-4],[0,-1],[1,-1],[1,2],[0,1],[0,-2],[0,-6],[0,-4],[1,1],[1,2],[0,5],[0,6],[0,1],[1,0],[1,0],[0,4],[0,-3],[0,-2],[1,-2],[0,-2],[0,-3],[0,-1],[0,-1],[1,0],[0,-3],[-1,-3],[1,-2],[0,-3],[-1,-3],[0,-3],[0,-5],[1,-2],[0,1],[1,3],[0,-5],[1,-2],[2,4],[1,7],[-1,-2],[1,-1],[0,-1],[0,-2],[0,-1],[1,1],[0,1],[0,1],[0,1],[1,1],[0,-1],[1,-3],[0,-2],[-1,-1],[1,-1],[2,1],[2,-3],[0,-2],[0,-1],[0,-4],[0,-2],[-3,-6],[0,-4],[0,-4],[1,-4],[0,-4],[1,1],[1,2],[0,4],[-1,3],[1,2],[4,3],[0,-2],[1,-2],[2,2],[0,3],[-1,2],[-1,1],[0,2],[2,4],[1,2],[0,2],[0,1],[0,2],[0,1],[1,0],[0,-1],[0,-1],[0,-2],[0,-10],[0,-4],[0,-3],[-1,-5],[1,-2],[0,2],[1,4],[1,6],[0,2],[-1,1],[0,3],[1,2],[0,1],[2,1],[3,9],[2,1],[1,-2],[1,-5],[1,-4],[0,-3],[0,-3],[0,-3],[0,-3],[0,-3],[-1,-2],[-1,-3],[0,3],[-1,2],[0,1],[-1,-2],[0,-1],[-1,3],[-1,1],[-1,-6],[1,0],[1,-2],[1,-2],[0,-4],[0,-3],[-2,-1],[0,-1],[3,1],[3,6],[1,0],[1,-4],[-1,-1],[0,-1],[0,-3],[2,-1],[1,-2],[0,-4],[2,-12],[-1,-3],[0,-4],[0,-4],[0,-3],[0,-2],[1,11],[1,4],[0,2],[0,4],[-2,16],[1,-1],[1,1],[0,1],[1,3],[-1,2],[0,3],[1,2],[0,2],[1,2],[1,-1],[1,-12],[1,-2],[0,1],[1,5],[-1,3],[0,3],[0,2],[1,1],[2,-3],[-1,5],[0,1],[0,2],[1,4],[0,1],[0,2],[0,2],[0,1],[-1,1],[0,1],[-1,1],[-1,0],[1,4],[1,1],[4,3],[4,-3],[7,-12],[2,2],[13,-6],[0,-4],[-1,-2],[0,-1],[-1,-2],[4,-5],[1,-2],[-1,-4],[-2,-7],[0,-4],[6,4],[0,-1],[2,-3],[0,1],[1,3],[1,0],[1,-2],[-1,-6],[-1,-4],[0,-4],[-2,-8],[3,1],[1,-2],[1,-5],[1,-1],[4,7],[1,0],[1,-1],[3,-8],[1,-2],[2,-1],[0,-2],[0,-2],[-1,-3],[1,0],[2,2],[0,-2],[1,-3],[0,-3],[-1,-2],[0,1],[-3,-1],[1,-5],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-1],[-1,-1],[0,-2],[-1,-3],[0,-3],[1,-1],[1,2],[3,10],[1,2],[2,1],[2,-1],[3,-6],[3,-9],[3,-8],[1,-8],[0,-4],[-1,-8],[-3,-2],[-3,5],[-3,3],[-6,2],[-4,-2],[-3,-7],[-10,-9],[-2,-4],[0,-7],[0,-6],[0,-5],[-3,-8],[7,2],[13,-9],[13,-10],[1,-3],[-2,-2],[-1,-2],[-1,-5],[1,-1],[-1,-2],[-9,6],[-2,3],[-2,1],[-1,3],[-1,0],[-4,-1],[0,1],[0,2],[-1,2],[-1,-2],[0,-4],[0,-4],[-1,-3],[-12,-2],[-2,1],[3,-10],[3,-6],[2,-8],[1,-2],[1,-1],[10,-18],[6,-5],[8,4],[4,-4],[0,-4],[-1,-8],[2,-1],[2,-3],[3,-11],[-1,-2],[1,0],[0,-2],[0,-1],[0,-3],[0,-1],[-3,-1],[-1,-2],[0,-5],[0,-2],[0,-3],[1,-2],[2,-1],[1,0],[1,-1],[1,-4],[0,-4],[-1,0],[-1,0],[-1,0],[-3,-10],[0,-4],[-1,-1],[-1,-1],[0,-1],[-1,-3],[1,0],[1,-2],[0,1],[1,2],[3,1],[1,-2],[0,-2],[-1,0],[-3,-9],[-3,-1],[-1,1],[-1,3],[-1,0],[0,-1],[-1,-3],[-1,-2],[-3,-1],[-1,3],[-3,12],[-1,3],[-1,1],[-7,13],[0,4],[1,3],[2,3],[-2,4],[-10,14],[-8,27],[-2,3],[-4,-2],[2,-5],[0,-3],[1,-3],[2,-11],[1,-2],[-1,-3],[-1,0],[-1,1],[1,-5],[3,-3],[4,-7],[0,-3],[2,-5],[0,-1],[2,0],[2,-6],[1,-7],[0,-6],[1,-1],[1,-6],[2,-5],[0,-2],[0,-4],[0,-5],[0,-4],[0,-4],[1,-1],[0,1],[0,-1],[0,-2],[1,-4],[0,-2],[0,-1],[0,-1],[1,-1],[1,-2],[1,-1],[0,-2],[1,-3],[0,-2],[0,-4],[0,-3],[0,-3],[0,-4],[0,-4],[1,-3],[1,-2],[1,-4],[2,-12],[2,-9],[1,-1],[0,1],[1,4],[4,7],[2,6],[0,6],[-1,6],[-1,7],[-1,5],[0,5],[0,5],[-1,5],[-1,7],[0,2],[0,3],[1,1],[0,-2],[1,-2],[2,-18],[3,-12],[1,-4],[0,-3],[0,-3],[0,-2],[0,-3],[0,-4],[1,-2],[1,-2],[5,-16],[-1,-2],[-1,1],[-5,10],[-1,0],[-1,-4],[-2,-3],[-2,-5],[-3,1],[-2,-2],[-1,-3],[-1,-4],[-1,-5],[1,-9],[1,-6],[1,-2],[6,6],[1,0],[0,-2],[1,-5],[0,-2],[1,-4],[0,-3],[1,-3],[-1,-3],[0,-4],[0,-3],[0,-2],[2,-3],[0,-2],[0,-3],[1,-5],[0,-7],[1,-10],[0,-5],[1,-5],[1,-4],[2,-4],[3,-6],[1,-2],[0,-2],[1,-4],[0,-2],[0,-3],[0,-2],[0,-3],[0,-2],[0,-2],[0,-1],[0,-1],[1,-1],[0,-2],[1,-3],[-1,-5],[-1,-4],[0,-2],[-2,-1],[0,-3],[7,6],[2,-4],[-1,-1],[0,-1],[0,-2],[0,-3],[1,1],[2,3],[1,1],[1,-3],[1,-4],[0,-1],[1,0],[-1,-5],[1,-2],[1,-3],[0,-5],[0,-3],[0,-2],[-1,-1],[4,-7],[4,-3],[1,-1],[0,-2],[0,-2],[0,-4],[0,-1],[1,-2],[2,-3],[1,1],[1,1],[1,3],[0,5],[0,3],[0,4],[0,4],[0,3],[1,3],[1,1],[0,-2],[0,-5],[1,-2],[0,-2],[0,-3],[0,-4],[1,-6],[1,-9],[0,-7],[0,-3],[0,-5],[0,-3],[0,-2],[1,-2],[3,-4],[3,0],[2,3],[5,14],[4,19],[2,13],[3,31],[1,9],[0,5],[0,3],[1,2],[1,1],[1,0],[0,1],[0,3],[0,2],[0,2],[-1,1],[1,6],[1,4],[3,1],[5,-3],[0,3],[-1,1],[-5,6],[-1,3],[0,6],[1,5],[2,15],[0,4],[0,2],[0,2],[0,3],[0,3],[1,7],[0,10],[1,4],[2,21],[2,15],[0,5],[8,49],[2,11],[3,5],[2,1],[3,-1],[2,5],[2,1],[4,-3],[2,-9],[2,-9],[0,-13],[-1,12],[-2,7],[-2,7],[-2,4],[-2,0],[-2,-5],[0,-6],[-2,2],[-2,-2],[-1,-5],[0,-3],[1,-4],[0,-4],[0,-4],[-1,-1],[-1,-1],[0,-3],[0,-3],[0,-5],[0,-2],[5,-9],[2,-3],[1,-8],[1,-10],[1,-1],[0,-3],[0,-3],[0,-3],[1,-2],[1,-5],[7,-21],[7,-14],[4,-3],[3,-6],[7,-7],[9,2],[2,6],[1,1],[0,-1],[4,-8],[0,-2],[0,-4],[-1,-4],[-1,-4],[-1,-4],[1,-8],[0,5],[1,6],[0,5],[1,3],[1,6],[-1,4],[-1,4],[0,6],[1,0],[1,1],[0,8],[1,6],[1,2],[1,1],[1,3],[3,8],[1,-1],[1,-5],[1,2],[1,5],[0,1],[1,1],[1,-1],[2,-4],[3,0],[0,4],[-1,2],[-1,2],[1,4],[1,3],[3,3],[2,4],[10,3],[9,-13],[4,-1],[2,-5],[0,-10],[1,0],[0,-2],[0,-3],[0,-5],[1,0],[0,3],[1,3],[0,3],[1,3],[0,1],[1,1],[1,0],[1,-2],[1,-2],[1,-3],[2,-10],[0,-2],[2,-2],[1,-2],[1,-3],[1,-2],[4,-3],[1,-3],[0,-2],[0,-2],[-1,-3],[2,-2],[2,4],[0,-4],[1,-1],[1,-1],[1,-1],[0,-2],[0,-1],[-1,-3],[0,-2],[0,-2],[1,-1],[0,-1],[0,-2],[-2,1],[0,-2],[1,-2],[2,-5],[0,-3],[1,-1],[1,1],[1,1],[2,9],[0,-3],[0,-2],[0,-2],[-1,-1],[1,-2],[1,1],[0,1],[0,-5],[-1,-3],[-1,-1],[-1,0],[0,1],[-1,2],[-1,0],[0,-1],[-1,-1],[1,-2],[0,-2],[2,-1],[3,-7],[1,-1],[0,-6],[0,-4],[1,-2],[1,0],[1,1],[0,1],[0,2],[0,3],[1,1],[1,-1],[1,-1],[-1,-1],[0,-3],[1,-2],[1,1],[1,2],[1,1],[-1,4],[0,3],[-1,2],[0,1],[0,2],[-1,2],[0,1],[-2,0],[0,2],[0,-1],[-2,-4],[0,3],[0,2],[0,1],[-1,1],[-1,1],[0,1],[-1,2],[1,2],[1,3],[1,1],[4,-7],[2,0],[1,1],[0,2],[0,2],[0,1],[2,-1],[1,0],[0,3],[1,5],[-1,2],[-6,2],[0,2],[5,8],[1,-3],[2,-2],[1,0],[2,1],[0,2],[-1,0],[-1,3],[-8,0],[-1,2],[-1,4],[0,1],[0,2],[1,2],[-1,1],[-1,-2],[0,-2],[-1,-1],[0,1],[-1,4],[0,2],[1,1],[1,3],[2,9],[1,7],[0,2],[0,2],[0,6],[0,2],[1,8],[1,6],[4,5],[7,3],[-1,-3],[-1,-2],[0,-3],[0,-4],[0,-1],[1,-6],[3,4],[1,3],[0,4],[0,3],[-1,2],[0,1],[3,9],[1,1],[1,-1],[4,-10],[1,-2],[2,-2],[1,-4],[1,-4],[0,-10],[1,-4],[0,-1],[0,-4],[1,-4],[1,-3],[-2,-5],[1,0],[2,-3],[3,0],[1,2],[0,1],[0,2],[1,4],[0,2],[1,7],[1,2],[4,0],[1,-2],[1,-6],[1,-1],[1,4],[2,4],[1,0],[1,-2],[2,0],[1,-1],[0,-1],[1,-1],[0,3],[0,2],[-2,3],[1,1],[0,3],[0,1],[-5,16],[0,3],[-2,12],[0,4],[0,1],[0,5],[-1,7],[0,9],[0,2],[0,1],[0,2],[0,2],[-1,2],[1,2],[0,4],[1,3],[0,4],[0,2],[1,2],[1,0],[-1,2],[-2,0],[-1,4],[0,3],[-1,3],[-1,4],[-1,1],[-6,3],[0,2],[2,3],[4,-4],[2,2],[1,3],[1,7],[0,5],[0,5],[0,6],[0,5],[3,14],[1,5],[1,5],[1,-1],[3,-6],[1,-1],[2,2],[1,3],[1,5],[-1,7],[-1,2],[-8,1],[-8,1],[-1,-3],[-3,-9],[1,-3],[3,1],[-1,-3],[-4,0],[-2,1],[-1,3],[-1,0],[0,-4],[1,-3],[1,-1],[1,-3],[-2,1],[-1,2],[-1,4],[-2,9],[-1,7],[0,2],[0,3],[0,4],[1,2],[0,8],[1,6],[2,8],[1,9],[1,1],[0,4],[0,1],[1,0],[1,1],[3,15],[1,2],[9,2],[6,-6],[8,2],[6,2],[6,9],[2,1],[1,5],[1,3],[4,2],[1,2],[-3,0],[-2,2],[0,2],[0,1],[1,2],[0,5],[0,2],[1,6],[0,6],[0,4],[-2,4],[-1,1],[0,2],[-1,6],[-6,12],[0,1],[0,1],[0,3],[0,1],[0,2],[-1,4],[-1,2],[0,4],[1,2],[3,2],[2,-2],[1,0],[2,-3],[3,0],[2,-2],[5,-1],[4,-6],[0,-2],[2,-7],[2,-8],[3,-2],[2,-4],[7,-6],[7,-6],[9,4],[4,-1],[9,-3],[2,-3],[11,3],[3,-3],[3,-5],[10,-5],[7,-1],[7,-8],[7,-4],[8,-4],[8,-5],[9,-15],[7,-6],[4,-1],[4,-7],[4,-3],[2,-6],[9,-12],[1,0],[0,-2],[0,-1],[0,1],[-1,-4],[-10,3],[-9,3],[-2,-3],[-10,5],[-6,-1],[-3,-3],[-1,2],[0,2],[-1,6],[0,2],[-1,2],[-1,0],[-1,1],[0,2],[0,1],[-1,0],[-1,-1],[0,-2],[-1,-2],[0,2],[-1,5],[-1,2],[-2,1],[-1,-1],[-1,-3],[-4,-12],[-1,-4],[0,-3],[-1,-2],[-1,-5],[-2,-15],[0,-4],[0,-4],[-1,3],[-2,9],[-1,3],[-1,2],[-2,1],[-1,-1],[0,-2],[-1,-3],[0,-1],[-1,0],[0,1],[-2,1],[2,-7],[1,-1],[-1,-4],[1,-2],[1,0],[0,-2],[1,-2],[4,-7],[1,-1],[4,1],[0,1],[2,8],[1,3],[1,2],[6,3],[3,5],[9,2],[8,2],[9,3],[7,1],[9,2],[9,2],[-1,-10],[-3,-11],[-1,-7],[-2,-9],[-5,-24],[-7,-19],[-7,-32],[-1,0],[0,7],[0,4],[1,3],[0,3],[1,3],[-1,3],[0,2],[0,2],[1,2],[0,2],[0,2],[0,3],[0,3],[-1,2],[0,1],[1,2],[1,2],[1,-2],[1,0],[0,2],[2,7],[1,2],[1,2],[0,4],[0,2],[2,3],[0,4],[-3,3],[0,4],[-1,1],[-1,-2],[-2,1],[-1,-1],[0,-2],[0,-2],[0,-2],[0,-3],[0,-1],[0,-1],[0,-1],[-1,-5],[0,-2],[-1,-2],[0,-3],[-1,3],[0,5],[0,3],[1,3],[-1,4],[0,3],[-1,1],[-1,0],[-1,1],[0,5],[1,2],[1,1],[0,2],[-1,4],[-1,3],[-4,6],[-1,0],[-1,-3],[2,-7],[1,1],[1,-3],[0,-1],[-1,-3],[0,-7],[-3,-17],[0,-1],[0,1],[0,-2],[0,-1],[0,-1],[0,-1],[1,-3],[0,-3],[1,-4],[2,1],[1,-2],[0,-5],[0,-6],[1,-5],[-1,-3],[0,-4],[0,-2],[0,-2],[-1,-1],[0,-1],[-5,-1],[-7,-9],[-1,2],[-1,7],[-3,7],[-2,1],[-2,4],[0,5],[-1,1],[0,-3],[0,-3],[0,-3],[0,-2],[0,-3],[0,-2],[1,-4],[1,-2],[1,-3],[3,-5],[1,-4],[-1,-7],[-1,-2],[-1,-1],[-2,-7],[-3,2],[-1,-2],[1,-7],[-1,-3],[0,-8],[0,-6],[1,4],[1,3],[0,-1],[1,-4],[0,-2],[1,0],[0,4],[1,1],[1,-1],[1,-3],[1,-1],[1,1],[0,1],[1,1],[1,-1],[0,-1],[0,-3],[0,-1],[1,-1],[0,1],[7,19],[4,-2],[3,4],[3,7],[0,1],[2,0],[1,1],[1,6],[2,3],[3,14],[2,5],[0,1],[1,1],[0,2],[0,3],[0,3],[1,2],[2,6],[3,17],[1,8],[0,2],[1,1],[12,54],[2,4],[1,2],[1,1],[3,-3],[2,5],[7,-4],[2,3],[11,-3],[7,-4],[8,-7],[8,-7],[11,-17],[7,-9],[2,-5],[1,0],[0,-1],[0,-1],[1,-3],[0,-1],[5,-25],[3,-14],[1,-5],[0,-7],[-1,-7],[-1,-4],[-8,-19],[-1,0],[-1,2],[-3,10],[-2,7],[-2,12],[-1,1],[-1,-1],[-1,-4],[0,-1],[0,-1],[1,0],[-1,-5],[-1,-4],[-1,-3],[-1,-3],[0,-8],[-1,-2],[-1,-1],[-4,6],[-1,1],[-2,3],[0,1],[-1,0],[0,-2],[2,-7],[1,-1],[3,-2],[0,-2],[0,-2],[-1,-3],[0,-5],[0,-3],[-3,0],[-1,-2],[-1,-5],[5,-2],[1,1],[0,2],[1,4],[1,2],[2,-4],[4,0],[1,-2],[11,-8],[2,5],[1,0],[1,0],[0,-2],[1,0],[0,3],[1,-2],[0,-5],[1,-3],[2,-4],[1,-5],[-1,-3],[-1,-2],[-1,0],[0,2],[0,3],[0,2],[-1,0],[-1,-2],[0,-5],[0,-6],[1,-3],[0,-2],[-1,-1],[-2,-7],[0,-2],[0,-2],[0,-2],[0,-2],[1,2],[6,11],[4,1],[1,0],[1,-2],[0,-1],[-1,-1],[1,-1],[2,4],[1,0],[1,-1],[3,2],[1,-1],[1,-6],[0,-6],[0,-5],[-1,-4],[0,-3],[-1,-2],[-1,-2],[0,-3],[0,-3],[-1,-3],[1,-1],[-1,-4],[0,-1],[-1,0],[-1,-1],[-1,-4],[-1,-1],[-1,-2],[-1,-3],[0,-2],[-2,-1],[0,-1],[-4,-11],[-1,-2],[0,-1],[0,-2],[0,-1],[1,-1],[0,1],[2,2],[3,12],[1,2],[3,2],[4,13],[2,10],[1,3],[2,1],[-1,-5],[-1,-4],[0,-6],[-1,-9],[0,-3],[-1,-2],[1,2],[2,8],[2,4],[2,8],[1,2],[4,5],[2,-1],[2,-6],[9,-8],[2,-4],[1,-2],[0,-1],[1,-2],[0,-4],[1,-2],[0,-1],[1,-3],[0,-1],[1,-6],[1,-1],[-1,-3],[2,-7],[3,-10],[1,-8],[1,-5],[3,-10],[1,-8],[0,-4],[-1,-3],[-1,-1],[-11,2],[0,-4],[8,-8],[7,-15],[8,-8],[2,-5],[-1,-9],[1,7],[1,3],[5,0],[6,0],[9,6],[0,1],[0,3],[1,2],[3,0],[0,-2],[4,2],[2,-5],[7,8],[7,12],[1,5],[7,5],[7,5],[8,7],[13,7],[1,-1],[0,-3],[0,-3],[-1,-4],[0,-3],[1,-3],[0,-3],[0,-1],[1,3],[0,2],[0,3],[0,3],[0,1],[0,1],[1,7],[0,2],[1,2],[9,6],[9,-3],[11,2],[9,-2],[8,-3],[8,0],[2,-2],[1,1],[3,-4],[6,-1],[5,-9],[3,-2],[2,-6],[4,-3],[10,-17],[10,-17],[13,-39],[4,-22],[5,-40],[1,-11],[1,-10],[0,-6],[0,-6],[0,-5],[0,-6],[-2,-7],[-1,-5],[-1,-12],[0,-3],[-1,-1],[-5,5],[-1,-3],[1,0],[0,-2],[2,-5],[1,0],[1,1],[1,-3],[-1,-6],[-1,-4],[0,-6],[0,-2],[1,-2],[0,-3],[0,-3],[-1,-1],[-1,-4],[-1,-2],[-1,-4],[-1,-9],[0,-5],[0,-4],[2,-9],[-1,0],[0,-1],[1,-3],[1,-1],[0,-2],[0,-2],[0,-1],[0,-2],[0,-3],[-1,-3],[2,5],[1,2],[0,-1],[7,-17],[1,-4],[1,0],[-1,4],[-1,2],[4,-2],[1,-2],[0,-3],[0,-2],[0,-4],[0,-5],[2,5],[2,3],[2,0],[5,-5],[4,-8],[1,0],[0,3],[1,0],[0,-1],[0,-2],[1,-2],[0,-1],[1,-7],[0,-4],[0,-4],[-1,-5],[0,-6],[0,-6],[-1,-7],[1,-5],[0,-3],[0,-3],[0,-4],[0,-2],[0,-2],[1,-3],[0,-3],[0,-5],[0,-3],[1,-2],[0,-3],[0,-3],[0,-3],[0,-3],[0,-3],[-1,-2],[0,-1],[1,-3],[0,-6],[0,-4],[0,-8],[0,-4],[-1,-3],[-1,-7],[3,-9],[6,-6],[2,-4],[1,-5],[1,-6],[0,-5],[-1,-5],[-1,-10],[-1,-5],[0,-2],[-1,-3],[0,-5],[0,-2],[0,-3],[-1,-4],[0,-3],[-1,0],[0,-2],[0,-1],[0,-3],[0,-12],[0,-6],[0,-3],[-1,-2],[-1,-5],[-1,-12],[-2,-11],[-2,-1],[-5,2],[0,-6],[1,-3],[3,-1],[4,3],[1,4],[1,1],[0,1],[0,3],[3,23],[1,2],[1,5],[1,2],[-1,8],[-1,5],[1,3],[0,4],[0,2],[0,1],[3,12],[2,9],[1,3],[1,3],[0,4],[0,3],[0,3],[0,6],[0,8],[0,3],[0,4],[-1,4],[-1,6],[0,5],[0,2],[0,4],[0,2],[-1,7],[0,4],[0,3],[0,3],[-1,3],[-1,4],[2,16],[0,3],[-1,5],[0,3],[1,0],[0,1],[1,2],[0,3],[-1,3],[-1,3],[1,4],[4,8],[1,3],[1,10],[1,3],[0,1],[1,1],[1,1],[1,6],[0,1],[1,0],[1,-3],[2,1],[2,6],[-1,5],[1,3],[1,2],[2,4],[0,1],[1,1],[1,2],[-1,1],[0,1],[0,2],[0,2],[1,2],[0,-1],[1,-1],[1,-1],[0,2],[1,2],[0,4],[0,6],[1,0],[1,-5],[0,-1],[1,0]],[[8770,8109],[3,-8],[2,-2],[1,-4],[1,-3],[1,-1],[2,-13],[3,-8],[1,-6],[0,-8],[0,-6],[0,-5],[-1,-3],[-1,-1],[0,-2],[-1,-1],[0,-1],[0,-1],[0,1],[-2,5],[0,2],[0,3],[-1,4],[-9,26],[0,3],[-1,5],[0,6],[-1,3],[-3,10],[-1,1],[-1,0],[-1,1],[-1,6],[0,12],[0,3],[-1,3],[1,0],[2,-4],[4,-2],[4,-15]],[[8903,8046],[-3,-2],[-3,8],[-3,21],[-1,11],[0,8],[0,13],[0,9],[1,7],[4,8],[7,3],[8,3],[2,-4],[2,-7],[2,-12],[0,-3],[0,-1],[-1,-5],[0,-2],[0,-2],[0,-2],[0,-3],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-3],[0,-2],[-1,-3],[0,-2],[0,-4],[0,-2],[0,-2],[-2,-5],[-1,-2],[-3,0],[-1,-2],[-1,-4],[-1,-1],[-1,0],[-2,-1],[-1,-6],[-1,-1]],[[8223,8137],[-1,0],[-2,1],[-1,2],[0,1],[-1,0],[0,1],[0,4],[1,5],[0,4],[1,2],[1,3],[1,0],[1,-2],[1,-4],[1,-4],[0,-4],[0,-4],[-1,-3],[-1,-2]],[[8148,8153],[-1,-6],[-2,-7],[1,-3],[0,-2],[-2,-3],[0,-3],[0,-2],[-2,-5],[-3,-3],[-2,-5],[0,-2],[-1,-7],[-2,-12],[-2,-5],[-3,5],[-3,-1],[-2,5],[-9,4],[-1,3],[-3,9],[-2,7],[-10,11],[-3,8],[-1,5],[1,7],[1,4],[4,5],[1,-1],[3,-5],[1,-1],[2,1],[1,4],[1,6],[0,3],[0,3],[0,3],[0,2],[0,2],[0,4],[0,3],[1,15],[2,3],[9,-4],[9,-9],[8,0],[8,-4],[1,-4],[1,-5],[1,-2],[0,-5],[0,-2],[0,-3],[-1,-2],[-1,-1],[0,-8]],[[9075,8433],[-1,-1],[0,-3],[0,-1],[-2,-5],[0,-3],[1,-3],[0,-2],[1,-2],[1,-1],[1,-7],[1,-2],[10,-8],[1,3],[3,1],[1,1],[-1,5],[-1,0],[-3,-3],[1,4],[1,4],[1,2],[2,1],[0,-1],[0,-1],[0,-1],[1,-1],[2,-1],[-1,5],[-1,2],[-1,3],[1,3],[8,0],[5,-5],[8,0],[8,0],[3,-9],[1,0],[0,-9],[-1,-1],[0,-1],[-2,-7],[0,-6],[1,-5],[2,-7],[0,-3],[0,-1],[1,-1],[2,2],[1,0],[2,-2],[3,1],[1,4],[6,8],[9,-4],[9,-3],[9,-4],[2,-3],[0,-3],[1,-3],[0,-3],[-1,-2],[0,-1],[-1,-2],[2,3],[1,-1],[4,-7],[4,-2],[0,-2],[-2,-2],[0,-2],[1,-2],[1,0],[2,5],[1,2],[0,2],[0,2],[0,2],[0,2],[0,1],[1,2],[4,-1],[1,-1],[1,-2],[1,-2],[-1,-1],[-1,-4],[-2,-5],[-1,-7],[1,-2],[-1,-3],[-2,-5],[0,-3],[0,-1],[-1,-2],[0,-2],[0,-3],[0,-3],[1,-4],[0,-7],[-1,-6],[-1,-4],[-12,-10],[-11,-11],[-1,-2],[-1,-7],[0,-1],[-12,2],[-2,-4],[-10,4],[-9,4],[-10,3],[-9,21],[-1,5],[-2,2],[0,2],[0,2],[-2,5],[-4,4],[-6,1],[-6,8],[-4,12],[-8,18],[-12,12],[-3,7],[-1,2],[0,3],[0,4],[1,2],[1,6],[0,1],[0,3],[1,6],[0,7],[0,5],[-1,6],[0,2],[1,0],[1,-2],[1,1],[1,3],[-1,3],[0,2],[0,3],[1,4],[1,2],[1,0],[1,2],[-1,4],[-1,4],[0,5],[0,8],[1,7],[2,3],[1,0],[5,-10],[0,-1],[1,-5],[0,-1],[2,-5],[0,-3],[-1,-1]],[[8807,8452],[-1,0],[-1,1],[0,2],[-1,3],[0,1],[0,2],[0,1],[0,-1],[0,-4],[2,-2],[1,-3]],[[8804,8464],[0,-2],[0,3],[1,2],[1,3],[1,1],[1,2],[1,0],[-1,-2],[-4,-7]],[[8773,8407],[-11,-2],[0,5],[0,3],[0,3],[0,3],[0,4],[0,3],[0,3],[1,3],[1,3],[0,7],[2,10],[-1,7],[-1,13],[0,7],[0,1],[1,2],[0,4],[0,1],[0,3],[0,1],[0,1],[1,2],[0,4],[0,1],[0,2],[0,1],[0,1],[0,1],[1,4],[1,2],[0,2],[0,2],[0,2],[0,1],[0,1],[0,6],[0,2],[1,-1],[0,-4],[1,-5],[1,-5],[1,-6],[0,-9],[0,-1],[1,-4],[0,-3],[0,-2],[1,-1],[1,-6],[1,-2],[1,0],[1,-3],[2,-4],[1,-3],[1,-3],[0,-4],[-5,-18],[-1,-6],[0,-6],[1,-4],[1,-4],[-1,-4],[-1,-3],[-3,-8]],[[9244,8580],[-1,-1],[-2,3],[-3,7],[-1,4],[0,3],[2,4],[2,4],[3,3],[1,0],[1,-2],[0,-2],[-1,-4],[1,-3],[0,-2],[0,-2],[0,-5],[-1,-4],[-1,-3]],[[8863,8582],[13,-21],[2,-6],[4,-2],[1,-2],[2,-6],[1,-2],[-1,-3],[-1,2],[-1,-1],[0,-1],[0,-3],[0,-2],[-1,-1],[0,-1],[0,-3],[1,-2],[2,-3],[0,-1],[1,-4],[1,-2],[5,-9],[3,8],[1,-1],[2,-3],[3,-3],[1,-3],[1,-4],[0,-5],[-2,-11],[0,-1],[0,-1],[-1,-5],[0,-2],[0,-2],[0,-2],[0,-2],[1,-1],[0,-1],[6,-4],[2,2],[4,-4],[2,1],[2,3],[1,4],[-1,2],[-1,3],[-1,2],[0,2],[1,1],[-1,4],[-3,8],[1,8],[0,9],[0,10],[-1,12],[1,4],[0,4],[0,5],[0,5],[0,4],[0,3],[1,4],[1,5],[1,3],[1,1],[5,1],[10,-12],[1,2],[0,4],[-3,6],[-1,4],[-1,7],[0,1],[-1,5],[0,1],[0,1],[0,2],[0,1],[-1,2],[-1,5],[-1,3],[2,-1],[6,-13],[3,-2],[6,-15],[7,-12],[7,-24],[2,-7],[0,-2],[2,-3],[10,-8],[4,-6],[6,2],[3,-2],[-1,7],[1,3],[6,5],[2,-1],[3,-5],[0,-1],[0,-2],[1,-1],[0,1],[1,2],[0,1],[3,-2],[6,-17],[7,-7],[5,-10],[3,0],[0,-1],[1,-2],[0,-5],[2,-5],[10,-9],[3,-6],[2,-11],[1,-3],[0,-2],[-1,-1],[-3,2],[-5,-5],[-3,-7],[-5,-7],[-2,-3],[0,-3],[-1,-4],[0,-1],[1,-2],[0,-3],[-1,-3],[0,-5],[0,-1],[0,-4],[1,-1],[5,-8],[1,-4],[-1,-4],[-4,1],[-1,-4],[0,-5],[0,-6],[0,-6],[-1,-4],[0,-2],[-3,-4],[-2,-6],[-2,-10],[-1,-4],[-2,-2],[-5,0],[-4,-5],[-1,0],[0,2],[-3,4],[-2,6],[-1,2],[-6,1],[-3,-5],[-2,1],[-3,6],[-4,4],[-1,4],[-1,0],[-1,2],[-1,-1],[-1,2],[-1,2],[0,2],[0,1],[0,2],[0,1],[-1,3],[0,8],[0,2],[-1,2],[-2,-1],[-1,2],[0,4],[-1,3],[0,4],[0,7],[-1,2],[-1,2],[-2,5],[0,3],[-1,4],[0,3],[1,6],[0,4],[0,3],[0,1],[0,2],[0,3],[0,3],[3,3],[0,1],[0,3],[0,2],[3,9],[3,3],[5,10],[0,3],[1,8],[-2,15],[-2,4],[-6,-3],[0,3],[-1,1],[0,1],[-1,0],[-3,-4],[-5,2],[-1,-3],[1,0],[1,-2],[1,-2],[1,-5],[-1,-3],[-1,-1],[-9,2],[-1,-1],[0,-2],[3,-2],[2,-3],[0,-2],[-5,-1],[1,-3],[0,-1],[2,-1],[0,-2],[0,-4],[-1,-9],[0,-2],[0,-2],[0,-2],[3,-12],[-1,-11],[0,-6],[0,-5],[1,-8],[6,-20],[0,-3],[0,-4],[0,-3],[0,-3],[1,-2],[2,-6],[2,-5],[1,-9],[-1,-2],[1,-5],[1,-2],[5,-4],[5,-1],[6,-10],[6,-5],[3,2],[1,-1],[2,-6],[1,-9],[0,-3],[-1,-2],[-6,-7],[-13,-4],[-2,3],[-1,-1],[-1,-4],[-1,0],[-1,4],[0,1],[-2,0],[-1,-1],[0,-4],[-1,-7],[2,2],[0,-2],[-1,-3],[-4,-4],[-5,1],[-10,27],[0,3],[1,3],[2,2],[1,0],[2,-3],[3,-7],[1,-1],[1,3],[-1,3],[0,2],[-3,6],[0,3],[-1,4],[-10,-2],[-5,-8],[-8,-11],[-7,-1],[-13,-10],[-3,0],[-5,-9],[-8,-3],[-2,9],[0,2],[0,5],[-1,3],[0,2],[-2,0],[-1,3],[0,2],[0,6],[0,2],[-1,1],[-3,-1],[0,1],[-2,3],[0,-1],[-4,-7],[-1,-4],[1,-2],[0,-4],[-1,-5],[1,-5],[1,-3],[1,1],[1,3],[1,4],[0,-3],[-1,-1],[0,-2],[-1,-4],[-1,-6],[0,-5],[-1,-7],[0,-1],[0,-1],[-4,-19],[-3,-6],[-3,-3],[-3,1],[-6,14],[-9,8],[-9,8],[-1,3],[-2,1],[-4,13],[-1,3],[-1,6],[0,6],[-1,6],[0,3],[0,3],[0,2],[0,3],[-4,11],[-6,7],[-8,21],[-1,1],[0,7],[-1,3],[-1,6],[-3,14],[-1,3],[1,6],[0,3],[0,2],[-1,3],[-1,5],[0,3],[3,6],[2,0],[2,-11],[1,-2],[3,0],[3,4],[1,1],[0,4],[1,4],[-1,5],[0,2],[-2,2],[-2,0],[-2,-2],[-1,-3],[0,1],[-1,5],[1,2],[0,3],[0,4],[0,1],[4,0],[0,4],[-1,5],[0,2],[-2,3],[0,3],[0,3],[0,2],[0,1],[0,1],[0,2],[0,1],[0,2],[1,1],[1,0],[1,3],[-1,1],[-1,7],[-1,2],[-1,2],[0,2],[0,2],[0,3],[0,2],[0,3],[-2,3],[0,3],[2,9],[0,2],[1,3],[0,1],[0,1],[4,0],[2,-4],[5,0],[3,-7],[1,-1],[0,1],[0,1],[0,1],[0,1],[0,1],[0,2],[-1,4],[-3,6],[-1,3],[0,6],[-2,5],[-1,5],[-1,5],[0,7],[0,1],[1,3],[7,15],[6,1],[3,3],[3,11],[0,1],[-1,1],[0,1],[0,-1],[-2,-3],[-1,-4],[0,2],[0,1],[2,4],[0,1],[1,1],[1,7],[7,9],[-1,-2],[0,-1],[0,-2],[0,-3],[1,-2],[0,-2],[0,-8],[1,-3],[1,1],[0,3],[0,3],[0,2],[1,2],[0,1],[-1,2],[0,1],[1,9],[1,4],[0,2],[7,7],[8,6],[-2,-2],[-2,-2],[-1,-1],[1,0],[1,-3],[0,-1],[0,-2],[0,-2],[0,-2],[1,-5],[0,-3],[3,-12]],[[9145,8714],[-1,-1],[-5,0],[-7,0],[-5,-1],[-4,0],[-1,1],[0,3],[1,4],[3,4],[1,2],[3,4],[2,5],[2,4],[3,1],[12,8],[2,-1],[-1,-4],[-2,-3],[-1,-3],[-2,-4],[0,-5],[0,-5],[0,-6],[0,-3]],[[9351,8825],[-4,1],[-1,2],[1,4],[0,2],[1,1],[1,0],[1,-2],[1,-2],[0,-1],[0,-2],[0,-2],[0,-1]],[[8732,1768],[0,-1],[-5,-5],[-2,-6],[-2,-2],[-1,-1],[0,-2],[-1,-7],[-1,-2],[0,-1],[-2,0],[-1,-1],[-1,-3],[0,-2],[-2,1],[0,-2],[-1,-3],[0,-8],[-1,-2],[0,-2],[-1,-2],[-1,-1],[0,-2],[0,-2],[0,-3],[-1,-3],[0,-4],[0,-3],[-1,-2],[-2,-1],[-1,3],[0,1],[-1,0],[-1,-2],[-1,1],[-1,3],[0,1],[-1,2],[-1,-1],[-3,-4],[0,-1],[-1,-3],[0,-3],[0,-2],[0,-1],[0,-1],[-1,-1],[0,-1],[-2,-7],[0,-1],[-2,-9],[0,-2],[-1,-6],[0,-2],[-1,-2],[0,-1],[-1,1],[0,1],[-1,4],[0,1],[-1,0],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-2],[0,-2],[0,-2],[0,-2],[0,-1],[0,-2],[0,-3],[0,-3],[-1,-11],[-1,-6],[0,-2],[0,-1],[0,-2],[0,-4],[0,-2],[0,-2],[0,-2],[-1,-4],[0,-2],[0,-1],[-1,-1],[-1,0],[-1,1],[-1,4],[-1,2],[-1,0],[-2,-2],[0,-1],[-2,-6],[0,-1],[-1,-1],[-1,-1],[-1,2],[-1,1],[-2,-2],[0,-1],[-1,-4],[0,-1],[-1,0],[0,1],[-1,2],[0,1],[-1,0],[-1,-1],[-2,4],[-1,0],[-1,-7],[-1,-1],[-1,0],[0,1],[-1,9],[-1,4],[-1,4],[-1,1],[0,1],[-6,-9],[-1,-3],[-1,-4],[-1,-1],[-1,0],[-1,0],[0,2],[-1,4],[-1,1],[-1,3],[0,9],[0,9],[0,8],[-1,15],[-1,11],[-5,30],[0,4],[0,5],[0,4],[0,3],[0,5],[1,7],[1,4],[0,1],[0,1],[0,9],[1,3],[0,2],[1,9],[0,3],[0,6],[0,4],[0,1],[0,1],[-1,1],[0,2],[0,2],[-1,5],[0,7],[0,4],[0,5],[0,5],[0,4],[0,3],[-1,1],[-2,-6],[0,1],[0,2],[-1,5],[0,3],[0,2],[0,5],[0,3],[0,4],[0,1],[-1,1],[-1,3],[0,4],[0,5],[3,40],[1,11],[0,4],[0,3],[-1,1]],[[8649,1980],[0,-2],[1,-2],[0,-5],[1,-1],[0,2],[1,1],[0,2],[1,3],[1,5],[0,1],[0,3],[1,1],[0,1],[0,1],[0,1],[4,5],[0,1],[1,2],[0,1],[1,-1],[0,-1],[0,-1],[0,-4],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[1,-1],[0,-1],[1,0],[0,-3],[1,-5],[1,1],[0,1],[-1,4],[0,3],[0,3],[0,2],[0,2],[0,1],[0,1],[0,1],[0,1],[1,2],[0,1],[0,1],[0,1],[0,1],[0,3],[0,2],[0,1],[0,1],[0,1],[2,2],[0,1],[0,1],[0,1],[0,2],[-1,2],[0,1],[0,1],[0,2],[0,1],[0,1],[0,2],[1,3],[0,1],[0,1],[1,5],[1,0],[0,-2],[2,-4],[1,-4],[1,-3],[0,-1],[0,-1],[2,-5],[1,-1],[0,-2],[0,-5],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-3],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[1,0],[0,-1],[1,-1],[0,-1],[1,-1],[1,1],[0,1],[1,4],[1,2],[0,1],[0,1],[0,-1],[1,-1],[0,-1],[0,-2],[0,-1],[1,-1],[0,1],[2,5],[1,8],[1,-4],[0,-3],[0,-2],[0,-1],[0,-7],[0,-3],[0,-1],[0,-1],[0,-3],[0,-2],[0,-1],[1,0],[2,3],[0,-1],[1,-3],[1,-1],[1,0],[0,-2],[1,-1],[0,-2],[1,-4],[0,-2],[0,-1],[1,0],[1,0],[1,-9],[2,-41],[0,-5],[1,-31],[1,-2],[0,-1],[12,-14],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[1,-2],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[2,-6],[0,-1],[1,-1],[0,-1],[0,-1],[1,-1],[1,-1],[1,1],[1,-1],[0,-1],[1,1],[0,-1],[0,1],[1,1],[0,1],[1,2],[0,1],[1,0],[0,4],[0,1],[1,0],[0,-1],[0,-1],[0,-1],[1,1],[0,4],[0,2],[0,1],[0,1],[0,1],[0,1],[0,2],[0,1],[0,1],[1,0],[1,1],[1,3],[1,-3],[0,-1],[0,-1],[0,-1],[1,-2],[0,-1],[0,-3],[0,-1],[0,-1],[0,-1],[1,-5],[0,-1],[0,-1],[0,-1],[0,-1],[1,1],[1,-4],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[2,1],[0,1],[1,2],[0,-2],[0,-1],[0,-2],[0,-1],[1,-1],[0,-2],[0,-1],[0,-1],[1,0],[1,2],[0,1],[1,2],[0,1],[1,0],[0,1],[0,5],[0,1],[0,1],[1,0],[0,-1],[0,-2],[1,0],[0,1],[0,1],[1,-1],[0,-2],[1,0],[0,-2],[0,-2],[0,-7],[-1,-5],[1,-4],[0,-1],[0,-3],[0,-1],[0,-1],[-2,-6],[0,-4],[0,-9],[-1,0],[0,-1],[-3,-12],[-1,-1],[0,1],[-4,10],[-1,1],[-2,-1],[0,-1],[-1,-4],[-2,-3]],[[8828,3247],[-1,-3],[0,-3],[-2,-2],[0,5],[0,2],[0,-1],[0,1],[2,9],[0,3],[1,4],[0,8],[1,2],[0,1],[1,1],[0,-5],[0,-3],[0,-3],[0,-2],[-1,-2],[-1,-2],[0,-10]],[[8822,3250],[0,-2],[0,1],[-1,-1],[-1,-1],[0,4],[0,3],[1,5],[-1,12],[0,1],[0,1],[0,1],[0,2],[0,1],[1,4],[0,5],[1,2],[0,1],[0,-4],[0,-4],[0,-4],[1,-3],[0,-2],[0,-1],[0,-1],[0,-2],[-1,-2],[0,-2],[0,-3],[0,-3],[0,-4],[0,-4]],[[8868,3295],[-1,-3],[-1,0],[0,1],[0,1],[1,0],[0,2],[1,3],[0,1],[0,-2],[0,-3]],[[8838,3321],[0,2],[-1,2],[1,0],[1,0],[0,-2],[-1,-2]],[[8810,3418],[-2,-17],[-1,-3],[0,-1],[-1,-2],[1,-5],[0,-5],[0,-1],[0,-5],[-1,-3],[-1,-1],[-2,1],[0,1],[0,2],[0,2],[0,2],[0,2],[-1,0],[-2,-5],[-1,1],[0,-1],[0,-2],[0,-1],[-1,1],[-1,-3],[1,-1],[0,-5],[-1,-2],[0,1],[-1,3],[0,1],[0,2],[0,2],[0,2],[0,3],[0,3],[0,2],[1,1],[0,2],[1,2],[1,6],[0,2],[1,2],[1,1],[3,14],[2,5],[1,-5],[1,5],[0,4],[1,1],[0,-2],[1,-3],[0,-3]],[[8827,3429],[1,-1],[1,1],[1,2],[0,2],[1,-2],[0,-3],[1,-4],[0,-3],[0,-2],[0,-6],[-2,-12],[-1,-5],[4,13],[1,0],[2,1],[2,-1],[0,-5],[0,-4],[-2,-1],[0,-2],[-2,-17],[-1,0],[0,-1],[0,-7],[-1,-3],[0,-2],[0,-1],[0,-6],[1,-5],[0,-4],[-1,-6],[-3,-9],[0,-5],[-1,-3],[0,-3],[-1,-7],[-1,-5],[0,-1],[-1,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-3],[-1,0],[0,4],[-1,3],[0,2],[-2,22],[0,2],[-1,1],[0,3],[0,2],[0,7],[-1,6],[0,6],[-1,4],[-1,-3],[-1,-4],[-3,-13],[-1,-3],[-1,-2],[0,5],[1,3],[0,4],[0,9],[0,3],[0,6],[1,6],[1,5],[1,6],[0,7],[1,3],[0,1],[1,1],[0,3],[1,4],[0,6],[1,7],[-1,5],[1,4],[1,11],[0,4],[2,1],[0,-2],[2,-6],[0,-4],[1,-1],[1,-1],[1,-1]],[[9020,5128],[1,1],[1,3],[1,1],[2,-1],[1,-2],[0,-2],[1,-8],[1,-1],[0,-1],[2,-5],[1,-5],[1,0],[1,0],[0,1],[0,1],[1,3],[0,1],[1,0],[3,-3],[0,-1],[1,-2],[0,-1],[0,-1],[0,-1],[1,0],[1,1],[0,1],[1,3],[0,2],[2,2],[0,1],[0,1],[1,1],[0,1],[0,1],[0,1],[0,5],[0,2],[0,1],[0,1],[1,1],[1,1],[1,-1],[0,-1],[0,-4],[1,-4],[0,-1],[0,-1],[0,-1],[0,1],[1,1],[1,3],[1,-1],[2,-5],[0,-3],[1,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[1,-1],[1,1],[1,-3],[1,-2],[0,-3],[0,-2],[1,1],[0,-1],[1,-1],[0,-1],[0,-2],[0,-2],[0,-1],[1,-2],[0,-1],[0,-1],[0,-1],[1,-1],[1,-6],[1,-2],[0,-1],[0,-3],[0,-2],[0,-3],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[1,-2],[0,-1],[1,-3],[0,-1],[0,-1],[0,-4],[0,-1],[0,-2],[2,-9],[0,-1],[1,0],[0,-2],[1,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[-1,-2],[0,-1],[1,-6],[0,-2],[0,-3],[0,-6],[0,-2],[0,-2],[1,-6],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-2],[0,-1],[0,-2],[0,-2],[0,-2],[0,-2],[1,-1],[0,-1],[0,-1],[3,1],[0,-2],[0,-1],[0,-2],[-1,-3],[0,-2],[0,-3],[0,-2],[0,-1],[-1,-5],[0,-2],[0,-1],[0,-2],[0,-3],[0,-4],[0,-3],[0,-2],[1,-6],[0,-2],[0,-2],[0,-1],[0,-4],[0,-5],[0,-2],[0,-4],[0,-9],[0,-4],[0,-2],[-1,-1],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[-1,0],[0,-1],[0,-6],[0,-1],[0,-2],[0,-1],[0,-1],[-1,0],[0,-1],[-2,1],[-1,2],[-1,0],[0,-1],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-3],[0,-2],[0,-3],[0,-6],[0,-2],[0,-2],[1,-2],[0,-3],[0,-2],[-1,-1],[0,-3],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[0,-2],[-1,-2],[0,-1],[0,-1],[0,-1],[0,-2],[1,-2],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[-1,-5],[-1,-4],[0,-2],[0,-1],[-1,0],[0,-1],[-1,0],[-1,1],[0,1],[-1,5],[0,1],[-1,1],[-1,1],[0,1],[0,1],[0,1],[0,3],[-1,1],[0,1],[0,1],[-1,0],[-1,1],[-1,0],[0,-1],[0,-2],[0,-3],[0,-2],[0,-1],[0,-1],[-3,-9],[-1,-1],[0,-1],[-1,-2],[1,-4],[-1,-2],[0,-2],[-1,-1],[0,-1],[0,-1],[-1,0],[0,2],[-1,0],[0,-1],[-1,0],[0,-4],[-2,-5],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-4],[0,-1],[0,-2],[0,-1],[-1,-2],[-1,-1],[0,-1],[-1,-5],[0,-2],[0,-2],[0,-3],[0,-2],[0,-2],[0,-2],[0,-1],[1,-2],[0,-1],[0,-1],[2,-1],[0,-1],[0,-1],[1,-3],[0,-1],[0,-2],[0,-1],[-1,-1],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-3],[0,-4],[1,-4],[3,-14],[0,-1],[1,-2],[0,-2],[0,-1],[1,-3],[1,0],[0,1],[0,1],[0,1],[0,2],[1,1],[6,0],[2,4],[2,5],[1,-1],[1,1],[1,3],[1,0],[0,-1],[0,-2],[1,-7],[0,-1],[0,-4],[1,-4],[0,-2],[1,-1],[0,-3],[0,-2],[-1,-2],[0,-1],[0,-2],[0,-1],[0,-1],[1,-1],[0,1],[0,1],[1,0],[0,-1],[0,-2],[0,-1],[0,-2],[1,0],[0,1],[0,1],[1,2],[0,1],[0,-1],[1,-1],[0,-1],[1,-3],[0,-2],[1,-1],[0,-10],[0,-2],[0,-5],[-1,-1],[0,-2],[0,-1],[0,-3],[0,-1],[0,-1],[0,-1],[0,-1],[2,-5],[0,-2],[1,-3],[0,-6],[0,-3],[1,-1],[0,1],[2,7],[3,4],[1,-1],[0,-1],[0,-1],[0,-2],[1,-5],[0,-1],[0,-1],[1,-3],[1,-1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[0,-3],[-1,-2],[0,-1],[0,-1],[0,-1],[1,-1],[0,-1],[0,-2],[0,-2],[0,-1],[1,-1],[0,-3],[0,-1],[-1,-2],[0,-3],[0,-3],[-1,-4],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[2,-7],[0,-1],[0,-2],[0,-2],[-1,-4],[0,-2],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[0,-5],[0,-1],[0,-2],[0,-1],[0,-1],[0,-3],[0,-1],[0,-2],[0,-1],[0,-2],[-1,-4],[0,-1],[0,-1],[0,-2],[0,-1],[0,-3],[0,-1],[0,-4],[0,-2],[0,-1],[-2,-3],[-2,0],[-1,-1],[0,-1],[0,-3],[-1,-4]],[[9079,4468],[0,-1],[-4,5],[-1,2],[0,1],[0,4],[0,1],[0,3],[-2,4],[0,1],[-2,2],[-1,2],[-2,0],[-2,-8],[-1,-7],[0,-9],[0,-11],[0,-5],[-1,-5],[1,-2],[0,-3],[0,-3],[0,-2],[0,-2],[1,-4],[-1,-2],[0,-6],[-2,3],[-1,1],[-1,2],[-1,-3],[-1,-6],[-1,1],[-1,-8],[-2,1],[0,4],[0,3],[0,4],[-1,6],[-1,2],[-1,2],[-1,1],[0,2],[0,4],[0,4],[0,4],[1,4],[1,5],[0,4],[1,3],[0,6],[1,5],[-1,4],[-1,-4],[-2,2],[0,2],[-1,-5],[0,-7],[-1,2],[-1,5],[-1,4],[0,3],[0,3],[-1,-4],[-1,0],[-1,1],[-1,-3],[-3,-1],[-2,-4],[-1,1],[0,1],[-1,2],[-2,0],[-2,-5],[-4,-5],[-7,3],[1,1],[0,3],[-5,0],[1,-3],[0,-1],[1,-1],[-6,0],[-3,1],[1,1],[1,0],[1,-1],[-1,3],[0,1],[-1,1],[-1,0],[-2,-3],[-4,3],[-6,1],[-1,4],[-1,-5],[-7,-15],[-3,-5],[-2,1],[-2,4],[-1,5],[-2,1],[-1,5],[-1,-1],[-1,-5],[1,-1],[-8,-13],[-8,-19],[-9,-25],[-3,-17],[-2,-6],[-1,-2],[-1,-6],[-2,-14],[-5,-43],[-3,-20],[-2,-17],[-5,-18],[-4,-16],[-2,-5],[-1,2],[-1,-1],[-4,-10],[0,-5],[-1,-2],[0,-1],[-1,-1],[-4,-28],[-2,-9],[0,-10],[-2,-20],[-3,-44],[0,-6],[0,-10],[0,-6],[-1,-3],[-1,-2],[-1,-5],[-1,-4],[-1,-9],[-1,0],[0,3],[-1,1],[-1,-1],[0,-3],[-1,0],[-2,-2],[-1,-4],[-1,-1],[0,-2],[-1,-4],[-1,-2],[-1,-9],[-1,-14],[-2,-22],[-1,-6],[-2,3],[-1,-3],[-2,-3],[-1,-10],[-1,-12],[0,-5],[-1,-5],[-1,-1],[0,-3],[-1,-6],[-1,2],[-1,-4],[-1,-3],[-1,0],[-2,-1],[-1,-4],[-2,-6],[-1,-6],[-1,-10],[0,-5],[0,-2],[-1,2],[0,2],[-1,-3],[-1,-2],[0,-4],[0,-3],[0,-3],[0,-9],[0,-7],[-1,-3],[-1,-1],[-1,2],[0,2],[-1,-2],[1,-2],[-1,-2],[-2,-6],[0,1],[-1,-2],[-1,-4],[-1,-4],[0,-6],[-1,-10],[0,-13],[0,-4],[1,-6],[-1,-2],[0,4],[0,6],[0,10],[-1,-2],[-1,0],[-1,-3],[0,-5],[0,-6],[0,-3],[-1,-2],[0,-1],[0,-2],[0,-3],[0,-2],[-1,-2],[0,-3],[-1,-1],[0,-1],[-1,-5],[-1,-4],[0,-2],[0,-3],[0,-3],[0,-4],[0,-3],[-1,2],[-1,-3],[-1,-2],[-1,-4],[0,-4],[0,-4],[0,-3],[0,-3],[0,-6],[0,-3],[1,-3],[0,-2],[1,-1],[1,1],[0,-1],[1,-1],[-1,-2],[0,-3],[0,-5],[-1,1],[-1,6],[0,-3],[-2,0],[-1,-1],[0,-2],[0,-3],[0,-3],[0,-4],[-1,-1],[-1,-3],[-1,-1],[-1,-2],[0,-4],[-1,-4],[0,-8],[-1,-6],[0,-11],[-1,-1],[-1,-2],[0,-4],[0,-5],[-1,-2],[0,-2],[0,-3],[0,-3],[-1,-2],[-2,-2],[-1,-1],[-2,-13],[-1,-1],[0,-1],[-1,-1],[0,-4],[-1,-7],[0,-4],[0,-4],[0,-1],[0,-1],[0,-2],[0,-3],[-1,2],[0,2],[-1,2],[-1,-2],[-1,-7],[-1,-4],[-1,-4],[0,-1],[-1,-2],[-1,-6],[-2,-11],[0,-4],[0,-2],[0,-1],[-1,1],[-1,0],[-1,-1],[-1,-4],[0,-7],[-1,-3],[-1,-2],[0,-3],[-1,-7],[-1,-6],[-1,0],[-1,-3],[-2,-8],[0,1],[-1,1],[-1,-2],[0,-3],[0,-2],[0,-3],[-2,0],[0,-1],[-1,-3],[0,-2],[-2,-6],[0,-6],[-1,-5],[0,-7],[-1,-9],[-1,-2],[-1,-10],[0,-5],[-1,-3],[0,-3],[-1,-3],[0,-1],[-1,-1],[0,-2],[-1,-4],[-1,-4],[-1,-3],[-1,-3],[0,-3],[-1,-5],[0,-1],[0,-3],[-2,-5],[0,-1],[-1,0],[-2,-8],[-1,-6],[-1,2],[-2,-5],[-2,-4],[-2,-7],[0,-7],[-2,-6],[-1,-6],[-3,-13],[-1,-8],[0,-6],[-1,-3],[0,-9],[2,-24],[0,-1],[0,-3],[0,-1],[1,2],[0,1],[2,0],[8,-12],[1,-1],[0,-3],[0,-9],[1,-4],[0,-3],[0,-2],[1,-2],[4,4],[2,-1],[4,8],[3,2],[1,0],[0,-2],[1,-2],[1,2],[6,-4],[1,1],[1,5],[1,11],[0,5],[1,-3],[0,-4],[-1,-3],[0,-2],[0,-4],[1,-2],[1,0],[0,2],[1,3],[0,3],[0,1],[1,1],[0,-1],[1,-1],[0,-2],[0,-3],[0,-3],[0,-2],[1,-2],[0,-2],[0,-3],[-1,-6],[0,-5],[0,-2],[-1,-2],[0,-2],[1,-5],[0,-1],[0,-3],[0,-3],[0,-4],[0,-3],[0,-1],[-1,-1],[-1,4],[0,-2],[-1,-4],[1,-3],[0,-3],[0,-2],[0,-4],[0,-4],[1,-4],[1,-4],[-1,-5],[-1,1],[0,-1],[0,-3],[0,-3],[1,-4],[1,-2],[-1,-2],[0,-3],[-1,-3],[0,-3],[0,-3],[1,-2],[0,-2],[1,-4],[-1,-6],[0,-7],[-1,-6],[1,-4],[-2,-8],[-1,-5],[0,-5],[0,-3],[0,-3],[-1,-5],[0,-3],[0,-2],[1,-20],[0,-6],[0,-3],[0,-2],[1,-3],[1,-3],[0,-1],[1,1],[0,1],[1,2],[0,3],[0,3],[0,4],[1,1],[0,1],[1,2],[2,3],[0,-1],[1,0],[1,-4],[1,-4],[1,-3],[1,1],[0,4],[1,5],[1,23],[1,5],[0,4],[0,11],[1,1],[0,1],[0,1],[0,2],[-1,1],[-2,-2],[-1,1],[0,7],[0,3],[-3,15],[1,2],[3,13],[0,2],[1,1],[1,2],[1,5],[-1,2],[1,4],[1,2],[0,1],[1,1],[0,2],[1,2],[0,1],[0,1],[1,1],[1,0],[2,-5],[3,0],[0,2],[1,4],[0,2],[1,-2],[0,-4],[0,-2],[-1,-5],[-3,-17],[0,-5],[-1,-3],[-1,-4],[0,-5],[-1,-6],[0,-3],[0,-2],[0,-2],[-1,1],[0,1],[0,3],[0,3],[0,1],[0,1],[0,2],[0,1],[0,1],[0,1],[-1,0],[-2,-3],[0,-2],[-1,-2],[0,-2],[0,-3],[0,-2],[0,-3],[0,-2],[0,-3],[1,-2],[0,-1],[2,0],[1,-2],[3,-16],[1,-1],[0,-3],[1,0],[0,3],[1,0],[1,-3],[1,-1],[1,-1],[0,-1],[1,3],[2,3],[-1,-4],[-1,-9],[-1,-4],[-2,-5],[-1,-3],[0,-1],[0,-2],[0,-2],[-1,-1],[0,-1],[0,-1],[-1,-2],[0,-3],[0,-3],[0,-2],[0,-4],[-1,-1],[0,-4],[0,-3],[0,-2],[0,-2],[0,-3],[0,-1],[-1,-1],[0,-2],[-1,-4],[0,-3],[0,-1],[-3,1],[-1,-1],[-1,-3],[0,-3],[0,-3],[-1,-3],[0,-2],[-1,0],[0,-2],[0,-3],[0,-5],[1,-5],[2,-5],[1,0],[2,3],[7,2],[6,9],[4,16],[3,6],[0,1],[2,8],[1,5],[0,2],[1,4],[1,28],[0,5],[1,5],[0,4],[2,6],[3,10],[0,2],[1,0],[0,-2],[0,-4],[-1,-8],[0,-5],[1,-11],[0,-12],[-2,-11],[-4,-25],[-1,-5],[0,-7],[0,-4],[0,-2],[0,-1],[-2,-5],[0,-2],[-1,-3],[0,-2],[0,-6],[0,-2],[-1,-6],[0,-2],[1,0],[1,8],[4,-3],[1,0],[1,4],[0,4],[1,5],[1,6],[3,35],[1,21],[0,5],[0,4],[-1,5],[1,6],[1,9],[1,10],[0,10],[0,10],[0,12],[-3,-8],[0,1],[0,5],[1,10],[1,5],[0,2],[0,2],[0,1],[0,2],[0,2],[0,1],[0,3],[0,5],[0,3],[0,6],[-1,3],[-2,4],[2,4],[0,1],[1,0],[5,-16],[5,-7],[2,-5],[4,-3],[2,1],[7,17],[2,8],[1,2],[1,0],[0,-2],[1,-2],[-1,-5],[0,-7],[0,-3],[1,-3],[2,-6],[0,-1],[0,1],[0,2],[1,0],[-1,-2],[1,-4],[2,-13],[0,-1],[1,-1],[6,-14],[0,-3],[0,-2],[-1,-3],[0,-3],[1,-3],[0,-3],[1,-2],[0,-1],[1,0],[0,-1],[0,-2],[0,-4],[-1,-4],[0,-5],[-1,-5],[0,-5],[3,-20],[0,-1],[2,-3],[2,-9],[0,-2],[1,-1],[0,-3],[0,-1],[0,-4],[0,-2],[0,-3],[0,-3],[1,-5],[2,-4],[2,-10],[9,-22],[1,-2],[0,-2],[-1,1],[-1,4],[-1,0],[-1,-2],[0,-3],[0,-5],[0,-3],[0,-2],[1,1],[0,2],[0,1],[1,-1],[0,-2],[0,-2],[0,-3],[1,-2],[1,-2],[1,-2],[0,-5],[1,-3],[0,-2],[1,-3],[1,-4],[0,-2],[1,-1],[3,-1],[1,-1],[0,-3],[0,-3],[0,-6],[-1,-4],[0,-2],[0,-2],[0,-2],[1,2],[0,1],[1,-2],[-1,-3],[0,-3],[0,-3],[1,-2],[0,-2],[0,-2],[0,-2],[0,-1],[-2,-3],[0,-2],[-1,-8],[1,-1],[0,-2],[0,-2],[-1,-1],[0,1],[-1,4],[0,1],[-1,-1],[-1,-3],[0,-3],[-1,-3],[0,-2],[0,-2],[1,0],[1,0],[0,-3],[-1,-5],[-1,-4],[0,-1],[-1,0],[-1,1],[-3,14],[-1,2],[0,2],[-1,2],[-1,2],[0,2],[-1,7],[0,1],[-2,3],[-2,-1],[0,-4],[1,-5],[2,-6],[1,-5],[0,-1],[2,-2],[0,-2],[0,-6],[0,-1],[0,-3],[1,-5],[0,-4],[0,-1],[0,-1],[0,-2],[1,-1],[1,0],[0,-2],[0,-2],[1,-3],[0,-2],[0,-1],[0,-2],[2,-1],[0,-1],[1,-3],[0,-1],[0,-1],[2,1],[0,-2],[-1,-12],[-1,-5],[1,-2],[0,-1],[1,-3],[0,-4],[1,-3],[0,-5],[0,-4],[0,-4],[1,-3],[-1,-2],[0,-2],[0,-6],[0,-3],[0,-3],[0,-3],[-1,-3],[-1,-10],[0,-5],[-2,-9],[-1,-5],[0,-7],[1,-14],[1,-3],[1,-4],[1,-3],[0,-4],[1,-4],[1,-6],[1,-4],[0,-1],[1,-4],[0,-1],[0,-1],[2,-2],[1,1],[0,1],[1,0],[0,-1],[0,-3],[0,-2],[0,-1],[0,-1],[0,-3],[0,-3],[0,-1],[0,-3],[0,1],[-1,5],[-1,0],[0,-4],[-2,-4],[0,-3],[-1,-3],[0,-3],[-1,-3],[1,-2],[-1,-7],[-1,-5],[0,-3],[0,-5],[1,-2],[1,-2],[0,-2],[1,-6],[1,-3],[0,-4],[-1,-2],[-1,-2],[-1,-2],[0,-3],[-1,-2],[-1,-6],[-2,-8],[0,-4],[0,-10],[-1,-4],[0,-7],[-1,-4],[0,-3],[0,-4],[-4,-4],[-1,-8],[-1,-1],[0,-2],[0,-5],[0,-8],[0,-17],[0,-1],[0,-1],[0,-1],[-1,0],[-1,0],[-1,-4],[0,-3],[0,-3],[0,-5],[1,-1],[1,4],[1,-1],[0,-3],[0,-4],[-1,-3],[0,-4],[0,-3],[0,-3],[-1,-1],[-2,-1],[-1,-1],[-1,-6],[0,-7],[1,-4],[0,-2],[0,1],[0,-4],[-1,-3],[-1,-3],[0,-1],[1,-2],[0,-3],[0,-2],[0,-12],[0,-2],[0,-6],[1,-4],[0,-4],[0,-11],[-1,-2],[0,-2],[-1,-2],[0,-2],[1,-5],[0,-3],[0,-4],[-1,-3],[0,-1],[-1,0],[0,-1],[-2,-13],[0,-3],[0,-2],[0,-5],[0,-2],[0,-3],[0,-2],[0,-4],[-1,-7],[-1,-4],[0,-2],[0,-2],[0,-1],[0,-2],[0,-2],[0,-3],[0,-2],[0,-1],[0,-2],[1,-15],[-1,-2],[0,-8],[-1,-8],[0,-3],[1,-5],[1,-12],[0,-4],[0,-3],[0,-1],[0,-2],[1,-5],[0,-6],[0,-8],[0,-3],[0,-4],[-1,-5],[0,-4],[1,-5],[0,-5],[0,-5],[-1,-5],[0,-6],[1,-5],[0,-5],[0,-4],[1,-8],[1,-2],[0,1],[0,3],[1,1],[1,1],[0,1],[0,-6],[-1,-5],[-2,-8],[-1,-6],[0,-2],[-1,-2],[-1,-3],[0,-3],[0,-2],[0,-7],[-1,-4],[0,-4],[0,-2],[0,-2],[0,-3],[1,-14],[0,-3],[1,-1],[0,-1],[1,-2],[0,-5],[0,-3],[-1,-13],[0,-1],[0,-3],[0,-1],[1,-9],[0,-5],[0,-4],[1,-6],[-1,-5],[-1,-7],[0,-2],[0,-4],[0,-2],[0,-2],[0,-1],[0,-2],[-1,-8],[0,-5],[-1,-14],[0,-14],[0,-3],[-2,0],[-1,-2],[0,-2],[1,-1],[1,2],[0,-2],[0,-2],[0,-1],[-1,-1],[0,-5],[0,-2],[0,-3],[0,-2],[0,-3],[0,1],[0,2],[0,1],[0,-1],[0,-1],[0,-4],[0,-4],[0,-3],[0,-4],[0,-3],[0,-4],[0,-5],[0,-1],[-1,-1],[-1,0],[0,-1],[1,-4],[1,-1],[0,-4],[-1,-2],[-1,-2],[-1,-2],[-1,-2],[1,-1],[0,1],[1,-3],[-1,-2],[0,-2],[0,-2],[0,-1],[1,1],[0,-3],[0,4],[0,4],[1,3],[1,2],[0,1],[0,-1],[0,-1],[0,-3],[1,-2],[0,-8],[-1,-7],[0,-5],[-1,-5],[-2,-7],[0,-4],[0,-2],[0,-3],[0,-3],[0,-3],[-1,-5],[0,-4],[0,-6],[-1,-8],[0,-14],[0,-12],[0,-5],[0,-3],[0,-3],[0,-5],[0,-2],[0,-5],[0,-6],[0,-6],[0,-3],[-2,-4],[-1,-6],[0,-1],[0,-1],[0,-6],[0,-2],[-1,-2],[0,-6],[0,-2],[-1,-1],[-1,-4],[-5,-28],[0,-1],[-1,-6],[-1,-2],[0,-6],[-1,-9],[-1,-6],[-2,-15],[-1,-9],[-1,-3],[0,-1],[-2,-8],[-1,-5],[-4,-28],[0,-5],[0,-7],[-1,-6],[0,-3],[0,-3],[0,-4],[0,-3],[-1,-4],[0,-3],[-1,-3],[0,-6],[0,-1],[-1,-7],[0,-8],[-1,-5],[0,-4],[0,-4],[0,-6],[0,-5],[-1,-4],[0,-4],[0,-3]],[[8721,1339],[1,2],[0,5],[0,11],[1,10],[0,5],[-1,4],[0,4],[0,4],[0,3],[1,18],[0,5],[0,10],[0,3],[0,2],[1,1],[0,2],[1,14],[0,3],[0,3],[2,1],[0,2],[1,6],[-1,5],[0,4],[0,4],[0,6],[0,2],[0,2],[-1,5],[0,2],[0,4],[0,4],[0,3],[0,6],[0,4],[1,3],[1,4],[1,3],[0,8],[0,3],[1,2],[1,2],[1,1],[1,1],[0,1],[0,-2],[1,-1],[1,1],[0,2],[1,6],[1,2],[0,2],[1,12],[0,2],[1,3],[1,6],[0,3],[0,6],[0,5],[1,4],[0,2],[1,2],[1,12],[0,1],[0,6],[-2,13],[-1,8],[0,4],[0,6],[0,5],[-1,2],[0,1],[-1,3],[0,4],[0,4],[-1,10],[0,6],[0,8],[1,5],[1,10],[1,1],[0,7],[0,21],[1,8],[0,3],[0,-1],[-1,1],[-2,5],[-1,7],[-1,11],[0,-2],[-1,-2],[0,-1],[-1,0],[0,1],[-1,7],[-1,3]],[[9183,4380],[-2,-2],[-2,0],[0,2],[0,5],[1,3],[1,6],[1,4],[1,2],[1,2],[1,3],[0,4],[0,4],[1,-3],[1,-5],[0,-4],[0,-3],[-1,-2],[-1,-4],[-1,-7],[-1,-5]],[[9142,4427],[-1,-2],[0,-3],[0,-4],[-1,0],[-1,2],[0,4],[0,2],[0,2],[1,-1],[1,2],[0,1],[1,0],[0,-3]],[[9321,4461],[0,-4],[-1,0],[-1,0],[0,2],[0,3],[1,0],[1,-1]],[[9531,5797],[0,-1],[1,-1],[0,-2],[0,-2],[0,-1],[0,-1],[2,-2],[0,-1],[0,-2],[0,-2],[0,-4],[0,-2],[-1,-2],[0,-2],[0,-2],[0,-1],[-1,-3],[-1,-1],[0,-2],[0,-2],[0,-1],[1,-5],[0,-2],[0,-1],[0,-2],[0,-1],[0,-2],[-1,-2],[1,-1],[0,-2],[1,-8],[0,-2],[1,-1],[1,-2],[1,-1],[1,-1],[0,-1],[0,-3],[0,-1],[0,-1],[0,-2],[1,-2],[0,-1],[0,-8],[0,-1],[1,-1],[0,-1],[1,-1],[0,-1],[-1,-2],[0,-1],[-1,-1],[-1,-1],[0,-1],[0,-1],[-1,-1],[0,-3],[0,-3],[0,-1],[0,-1],[-1,1],[-1,-1],[0,-1],[0,-2],[-1,-1],[-1,-13],[0,-2],[0,-2],[0,-2],[0,-2],[0,-2],[0,-2],[0,-2],[-1,-1],[0,-2],[0,-1],[-1,1],[0,1],[0,1],[-2,2],[-3,0],[-1,-1],[-1,-2],[-2,-10],[0,-1],[0,-1],[0,-2],[0,-1],[1,-10],[0,-1],[0,-2],[0,-3],[0,-4],[0,-3],[0,-2],[0,-1],[0,-1],[1,0],[1,1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-4],[1,-1],[0,-4],[0,-1],[0,-2],[0,-2],[0,-1],[0,-1],[-2,-4],[0,-1],[0,-2],[0,-2],[0,-1],[1,-3],[0,-1],[-2,1],[0,-1],[-1,-1],[1,-2],[0,-1],[-2,-2],[0,-1],[-1,-2],[0,-1],[0,-2],[0,-2],[0,-4],[0,-2],[0,-2],[0,-1],[1,-7],[0,-2],[0,-1],[0,-1],[0,-1],[2,-3],[0,-1],[1,-1],[0,-2],[0,-3],[0,-5],[0,-3],[0,-1],[0,-1],[3,-4],[0,-1],[0,-1],[0,-1],[0,-1],[0,-2],[0,-4],[0,-2],[0,-2],[0,-13],[0,-3],[0,-2],[0,-1],[0,-1],[0,-2],[-2,-14],[-1,-1],[-1,1],[0,-1],[-1,0],[0,-2],[0,-2],[0,-2],[-1,-1],[0,-1],[0,-1],[-2,0],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-1],[1,-2],[0,-1],[0,-1],[0,-4],[1,-4],[0,-1],[0,-1],[0,-2],[-1,-2],[1,-2],[0,-1],[1,-5],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[-1,-8],[0,-2],[-3,1],[-1,-1],[0,-1],[-1,-1],[0,-1],[0,-2],[0,-2],[0,-2],[1,-2],[0,-1],[-1,-1],[0,-1],[-2,0],[0,-1],[-1,-1],[0,-2],[-1,0],[-1,2],[0,1],[0,-1],[-1,-1],[0,-3],[0,-1],[0,-1],[-2,-4],[0,-1],[0,-1],[0,-3],[0,-1],[1,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-2],[-1,-4],[0,-7],[0,-2],[0,-7],[0,-1],[0,-2],[0,-1],[0,-2],[5,-9],[1,-5],[1,-2],[1,-3],[1,-5],[0,-1],[0,-2],[0,-1],[0,-5],[0,-2],[0,-2],[0,-2],[2,-10],[0,-1],[0,-1],[1,0],[1,3],[1,1],[1,-1],[0,-2],[0,-2],[0,-6],[-1,-3],[0,-2],[0,-2],[-2,-11],[0,-1],[-1,-2],[0,-1],[0,-2],[0,-2],[0,-1],[1,-1],[0,-1],[0,-5],[1,-8],[0,-2],[0,-2],[0,-8],[0,-2],[0,-2],[0,-2],[0,-3],[0,-5],[0,-1],[0,-2],[0,-5],[0,-2],[0,-1],[0,-3],[0,-1],[0,-2],[0,-2],[0,-4],[0,-2],[-1,-1],[0,-2],[0,-2],[-1,-2],[0,-2],[-1,0],[0,1],[-1,3],[0,1],[-1,1],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[-1,0],[-1,2],[-1,0],[-2,-3],[0,-2],[-2,-9],[0,-1],[-1,0],[0,2],[-1,1],[-2,-5],[0,-1],[0,-1],[0,-2],[0,-2],[0,-2],[2,-4],[0,-4],[1,-1],[0,-1],[2,1],[0,-1],[1,-2],[1,-1],[2,-1],[1,-1],[0,-2],[0,-3],[0,-11],[0,-2],[0,-2],[0,-1],[0,-1],[0,-2],[0,-1],[-1,-4],[0,-1],[0,-2],[0,-1],[-1,-1],[-3,1],[0,-1],[-1,0],[0,-1],[0,-3],[0,-1],[0,-2],[0,-2],[1,-5],[0,-2],[0,-2],[0,-2],[0,-2],[1,-2],[0,-1],[0,-2],[2,-5],[0,-1],[0,-2],[0,-3],[0,-2],[0,-1],[-1,-5],[0,-1],[0,-1],[0,-1],[1,-1],[0,-1],[0,-1],[0,-2],[0,-2],[0,-4],[0,-2],[0,-2],[1,-11]],[[9513,5032],[0,1],[-1,2],[-1,1],[-1,-1],[0,-3],[-1,-7],[0,-4],[-1,1],[-1,0],[1,-4],[0,-3],[-1,-3],[0,-3],[-1,-3],[-1,2],[0,-1],[0,-3],[0,-5],[0,-3],[-1,-2],[0,-2],[-1,0],[0,-3],[-1,-3],[0,-3],[0,-3],[0,-2],[-1,0],[-1,-1],[1,-4],[-1,-4],[-1,-7],[-2,1],[0,3],[-1,9],[0,1],[0,-3],[0,-4],[0,-4],[1,-7],[-1,-5],[-1,1],[0,5],[-1,3],[0,-3],[0,-4],[-1,-2],[1,-5],[-1,-4],[-2,-3],[-1,-5],[-1,-1],[-1,-2],[0,-4],[0,-3],[0,-5],[-1,-1],[0,-3],[0,-2],[-1,3],[-1,1],[0,1],[-1,-2],[0,-3],[0,-3],[0,-2],[-1,-1],[0,-1],[0,-3],[1,-1],[0,-4],[-1,-1],[-1,-2],[-2,-13],[-1,-1],[-1,0],[-1,1],[0,-2],[0,-3],[0,-3],[-1,-2],[0,-6],[-1,-5],[-3,-7],[0,-3],[-1,-5],[-1,-2],[-1,0],[0,-5],[0,-5],[-1,-3],[-1,-9],[0,-8],[0,-6],[-1,-2],[-1,-3],[0,-4],[0,2],[-1,1],[0,5],[-1,2],[-1,-2],[-1,-4],[0,-5],[-1,2],[0,4],[-1,0],[-1,-4],[-1,-1],[0,-3],[0,-2],[-1,1],[-1,1],[0,6],[1,3],[-1,2],[0,-2],[0,-2],[-1,-3],[0,-5],[-1,-11],[-1,-2],[0,-2],[-1,-4],[0,-4],[-1,-1],[0,-4],[-1,-2],[-1,-1],[-1,6],[1,10],[0,2],[0,3],[0,1],[1,-1],[1,1],[-1,3],[0,2],[0,3],[1,3],[0,1],[0,3],[0,2],[0,3],[0,3],[0,3],[0,5],[-2,8],[1,1],[0,2],[1,2],[0,-3],[1,-2],[-1,7],[0,5],[1,5],[0,2],[0,1],[1,0],[1,0],[0,1],[0,3],[0,6],[1,5],[1,4],[0,5],[0,4],[1,4],[-1,0],[-1,0],[0,2],[-1,-3],[-1,1],[-1,-4],[-2,5],[-1,-2],[-1,-3],[-1,-3],[-1,-3],[0,-2],[-1,-1],[0,-3],[-1,-2],[0,1],[-1,-2],[0,-2],[0,-2],[-1,1],[-1,-1],[0,-3],[1,-1],[0,-3],[-1,0],[0,2],[-1,2],[0,1],[-1,-1],[0,2],[1,1],[0,3],[0,4],[0,2],[0,1],[1,4],[1,6],[1,6],[1,3],[-1,3],[1,2],[0,4],[1,2],[-1,5],[0,4],[0,3],[0,4],[-1,3],[0,2],[-1,3],[0,3],[1,2],[0,1],[0,4],[-1,4],[-1,-1],[-1,1],[0,-7],[-1,3],[-1,0],[1,3],[0,3],[1,3],[-1,2],[1,1],[1,0],[1,0],[0,-1],[0,-2],[1,-2],[1,3],[1,2],[-2,1],[-1,4],[1,4],[1,3],[0,4],[0,5],[0,1],[1,0],[1,-1],[0,3],[0,1],[0,3],[1,1],[0,5],[0,4],[0,2],[-1,-1],[-1,0],[0,2],[1,1],[0,2],[1,1],[0,2],[1,2],[0,2],[0,1],[1,1],[2,1],[0,4],[0,2],[1,4],[0,3],[-1,3],[0,2],[1,2],[0,3],[0,4],[0,1],[1,1],[0,4],[0,5],[0,4],[-1,1],[0,1],[2,6],[1,8],[1,4],[0,3],[0,4],[-1,2],[0,3],[0,3],[1,2],[-1,2],[0,3],[0,1],[0,2],[0,2],[0,2],[0,1],[0,3],[0,2],[0,1],[-1,1],[0,1],[0,5],[0,2],[0,-1],[-1,-1],[-1,-3],[-1,-2],[0,-2],[0,-2],[0,-1],[-1,-4],[-2,-1],[0,-3],[-1,-2],[-1,-3],[-1,-8],[-1,-6],[0,-3],[0,-4],[-1,0],[0,-3],[-1,-4],[0,-1],[0,-2],[-1,-3],[-1,1],[-1,1],[0,2],[0,-3],[0,-3],[-2,2],[0,-1],[0,2],[-1,3],[0,-5],[0,-4],[-1,-1],[-1,-2],[-1,-2],[0,-3],[-1,3],[-1,2],[0,2],[0,4],[0,4],[0,1],[1,4],[0,5],[-1,3],[1,4],[0,5],[0,3],[0,1],[-1,0],[0,-2],[0,-4],[-1,-1],[0,-2],[0,-5],[-1,3],[0,2],[-1,3],[-1,5],[-1,2],[1,1],[1,1],[0,5],[0,6],[-1,4],[-2,2],[-2,2],[-3,2],[-1,-1],[-2,0],[0,-2],[-1,-1],[-1,0],[-1,-2],[-1,-4],[-1,-5],[1,-8],[-1,-1],[-1,0],[-1,0],[-1,3],[-1,0],[-2,0],[-2,-1],[0,-2],[-1,-5],[-1,-3],[0,4],[-1,2],[-1,2],[-2,2],[-2,-1],[-1,-3],[0,-3],[-1,-8],[-1,1],[-1,-4],[-1,-5],[0,2],[-1,1],[0,-3],[-1,0],[0,1],[-1,2],[-1,3],[-1,2],[-3,3],[-3,0],[-4,3],[-2,-2],[-2,-9],[-1,-8],[0,-4],[0,-3],[0,-1],[-1,3],[-2,0],[-2,-5],[-2,-3],[-1,-1],[-1,-2],[-1,-3],[0,-6],[-1,-7],[-1,-8],[-2,-6],[-1,-3],[-2,3],[-1,0],[-1,-1],[0,-2],[1,-1],[-1,-3],[0,-4],[0,-4],[-1,-6],[0,-2],[0,-3],[0,-13],[0,-5],[0,-5],[0,-5],[0,-4],[0,-4],[0,-2],[0,-2],[0,-2],[0,-7],[1,-3],[0,-3],[-1,-2],[0,1],[0,3],[-1,1],[-1,1],[0,-1],[-1,-3],[-2,-5],[-2,-7],[-3,-17],[-5,-19],[-1,-7],[-1,-7],[0,1],[-1,-6],[-1,2],[0,-3],[1,-4],[-1,-6],[-1,-2],[0,-3],[0,-3],[0,-4],[0,-4],[0,-2],[0,-3],[0,-2],[0,-4],[0,-4],[0,-4],[0,-9],[-1,4],[0,2],[-1,1],[-1,-2],[0,-5],[-1,-2],[-1,-10],[-1,1],[-1,-2],[-1,-2],[0,-3],[-2,-1],[0,-3],[0,-5],[-1,0],[-1,-3],[0,-1],[0,-3],[0,-2],[0,-2],[0,-2],[0,-1],[-1,-1],[0,-2],[0,-1],[0,-1],[-1,3],[0,1],[-1,0],[-1,-2],[-1,-3],[-1,-5],[-1,-2],[-1,-11],[-2,-6],[-3,-8],[-2,-3],[-2,-12],[0,-3],[-1,-4],[-1,-5],[1,3],[0,3],[0,2],[0,-1],[-1,-3],[0,-3],[0,-5],[0,-1],[-1,-3],[0,-3],[-2,-6],[-2,-16],[-2,-21],[-1,-6],[-1,-13],[0,-3],[0,-4],[0,2],[0,5],[1,13],[0,8],[2,8],[-1,1],[-1,0],[-1,-3],[-1,-2],[-1,-3],[0,-7],[1,-7],[1,-6],[0,-4],[0,-5],[0,-5],[0,-5],[0,-2],[0,-4],[-1,-1],[-1,-7],[0,3],[0,3],[1,6],[-1,1],[-3,-2],[-1,-2],[-1,-4],[0,-5],[0,-5],[0,-4],[0,-7],[0,-6],[0,-7],[1,-8],[1,-9],[2,-19],[0,-2],[-1,2],[0,4],[-1,6],[0,6],[-1,4],[-1,0],[-1,-7],[0,-2],[0,-6],[0,-2],[0,-3],[0,-3],[-1,-2],[0,-3],[0,-4],[-1,1],[-1,-2],[0,-5],[1,-6],[0,-3],[1,-4],[3,-5],[1,1],[1,4],[0,4],[1,4],[-1,2],[0,3],[1,2],[0,2],[0,1],[1,1],[0,-3],[0,-4],[0,-4],[0,-3],[-1,-2],[1,-2],[0,-3],[1,-2],[-1,-2],[1,-3],[1,-2],[1,-1],[1,3],[0,4],[-1,3],[-2,4],[0,5],[0,6],[0,6],[-1,3],[1,0],[1,-2],[1,-3],[1,-3],[1,2],[1,4],[1,1],[2,-13],[2,-5],[1,1],[1,-1],[0,-2],[1,1],[0,2],[0,3],[1,2],[0,1],[1,-1],[0,-1],[1,-2],[0,-1],[0,-7],[0,-3],[1,-3],[1,-4],[0,-3],[2,-4],[1,-4],[0,-5],[0,-2],[0,-1],[1,-4],[0,-2],[-1,1],[0,-1],[-1,-3],[0,-4],[0,-4],[1,-6],[0,-3],[-1,-1],[0,-2],[0,-3],[1,-6],[-1,1],[-2,2],[-1,1],[0,-3],[-1,-2],[-1,3],[-1,2],[-1,-1],[-1,0],[-2,1],[0,-2],[0,-2],[0,-5],[0,-6],[-1,2],[-2,1],[-1,3],[0,1],[-1,3],[0,2],[-1,1],[0,4],[-2,4],[-2,-1],[-1,-4],[-1,0],[-1,-1],[0,-3],[0,-6],[1,-4],[0,-7],[-1,2],[0,-2],[-1,-4],[0,-2],[-1,1],[-2,2],[-1,-2],[-2,-7],[-1,-3],[0,1],[-1,1],[-1,4],[0,1],[-1,1],[-1,1],[-1,4],[-1,12],[-1,5],[-1,2],[-2,1],[0,3],[1,2],[0,3],[1,0],[0,3],[0,4],[-1,-4],[-1,-1],[0,-7],[-1,-2],[-1,3],[-1,4],[-1,1],[-1,3],[-2,-1],[-1,2],[-1,1],[0,-3],[0,-5],[0,-9],[0,-5],[-1,0],[-1,0],[0,-1],[0,-4],[0,-1],[1,3],[1,-2],[0,-4],[0,-3],[-1,-3],[-1,-1],[-2,1],[-1,-2],[-2,0],[-2,-4],[-1,-2],[0,-3],[0,-4],[-1,-3],[0,-3],[-1,-6],[0,-4],[0,-3],[0,-2],[0,-3],[0,-2],[0,-2],[-1,-3],[0,3],[-1,2],[-1,0],[-1,-1],[-1,-1],[0,-2],[-1,2],[0,1],[-1,5],[-2,1],[-1,1],[-1,8],[-2,5],[-1,2],[0,3],[0,6],[-1,0],[-1,3],[0,-1],[0,-1],[0,-2],[0,-3],[0,-3],[-1,-1],[-2,-4],[-2,-3],[0,-7],[0,-8],[-2,-7],[-2,-5],[-3,0],[-3,-1],[-1,-3],[-2,-3],[-2,0],[-2,4],[-1,-1],[-2,1],[-1,-6],[-2,-2],[0,2],[0,2],[1,2],[-1,3],[-1,7],[0,3],[-1,3],[0,5],[-1,5],[0,5],[0,4],[-1,5],[-1,2],[0,4],[0,3],[-1,4],[-1,5],[0,3],[0,3],[1,-1],[1,-4],[1,0],[1,-1],[0,3],[2,3],[1,1],[1,3],[1,5],[1,4],[1,-1],[2,-2],[2,1],[1,0],[1,-4],[2,-3],[1,4],[1,1],[1,-1],[1,-3],[1,3],[1,1],[2,-1],[1,2],[2,8],[2,3],[1,1],[1,0],[0,5],[-1,2],[-1,3],[-1,6],[-2,6],[0,-1],[-1,-2],[-1,-1],[-1,-2],[-1,-1],[-1,0],[0,4],[-1,3],[0,2],[-3,-1],[-1,1],[0,-1],[-1,2],[0,4],[0,1],[0,2],[1,3],[-1,5],[0,4],[-1,1],[0,3],[0,3],[0,4],[-1,7],[-1,6],[-1,5],[-2,1],[0,1],[0,1],[-1,1],[0,3],[0,3],[0,2],[-1,2],[-1,1],[0,2],[1,-1],[0,2],[-1,5],[0,2],[-1,-1],[-1,-3],[0,-5],[-1,-3],[-1,1],[-1,3],[-1,1],[-2,1],[-2,0],[-1,-3],[-2,-3],[-1,1],[0,-3],[0,-4],[1,-3],[-1,-6],[0,-1],[1,0],[0,-1],[0,-2],[0,-4],[-1,-3],[-1,0],[-1,0],[-2,-4],[-1,2],[1,2],[0,4],[0,5],[-1,-1],[-1,2],[-1,1],[-1,-1],[-2,-3],[-1,3],[2,4],[2,3],[1,0],[2,5],[0,6],[-1,-2],[-2,-2],[-2,-2],[0,2],[-2,2],[0,4],[-1,0],[0,2],[0,2],[-1,6],[-1,2],[-2,3],[-1,3],[-1,1],[-3,1],[-2,5],[-2,7],[-7,10],[-4,4],[-5,-5],[-1,4],[-1,1],[0,3],[0,-1],[-1,-2],[1,-8],[0,-3],[-5,-12],[-1,-6],[-2,3],[-1,-2],[0,-5],[0,-2],[1,-2],[1,-4],[0,-6],[-1,-1],[0,-1],[1,-4],[2,-1],[1,-3],[0,-10],[0,-4],[0,-1],[-1,5],[-1,1],[-1,1],[-1,-3],[-1,-1],[-1,-4],[-1,-2],[0,1],[0,2],[0,1],[-1,-1],[-1,-1],[0,1],[-1,2],[0,3],[0,4],[0,4],[0,3],[1,4],[-1,1],[-1,-4],[-1,-6],[0,-4],[-1,-6],[-1,-5],[0,-4],[0,-3],[0,-8],[1,-5],[1,-2],[1,1],[0,5],[1,5],[1,-1],[-1,-3],[0,-8],[1,0],[0,3],[0,4],[1,3],[1,-1],[1,-4],[0,-5],[-2,-10],[0,-5],[1,-5],[0,-6],[-1,-4],[-1,5],[-1,5],[-2,-1],[0,-8],[-1,0],[-1,5],[-1,2],[-2,-6],[-4,4],[-1,1],[0,5],[0,5],[0,4],[0,4],[-1,5],[0,6],[-1,1],[0,3],[-4,3],[-6,-6],[-2,1],[-2,-3],[-2,-8],[2,-7],[0,-5],[-1,-6],[-1,-1],[-1,-1],[-1,2],[-1,4],[0,2],[0,2],[-1,1],[-1,-1],[-1,-2],[0,-5],[0,-4],[0,-5],[-1,2],[-1,0],[-1,4],[-1,2],[-1,6],[0,2],[-1,1],[0,1],[-1,0],[-1,4],[0,1],[-1,1],[0,-1],[-1,-5],[-1,2],[-1,2],[-2,9],[-1,0],[-1,2],[-3,-2]],[[9057,550],[0,-2],[0,-1],[-1,0],[0,-4],[-1,2],[0,2],[1,0],[0,3],[0,1],[1,-1],[0,2],[0,1],[0,1],[0,1],[0,-1],[1,-2],[0,-1],[-1,0],[0,-1]],[[9052,555],[1,1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[-1,1],[0,-2],[0,1],[0,1],[0,1],[0,2],[0,2],[-1,2],[0,1],[0,1],[0,-1],[1,-1],[0,-1]],[[9059,579],[1,-2],[1,-1],[0,-1],[0,-3],[0,-2],[-1,-2],[0,-5],[-1,-5],[-1,0],[0,2],[0,4],[0,1],[0,1],[-1,1],[0,5],[1,5],[1,2]],[[9063,604],[1,-2],[0,-1],[1,-4],[-1,0],[-2,1],[0,1],[1,0],[0,2],[0,3]],[[9079,658],[0,-2],[1,-2],[0,-2],[1,1],[0,-5],[-1,-3],[0,-3],[0,-1],[-1,-1],[0,-1],[-1,-3],[0,1],[-1,0],[0,-2],[0,-2],[0,-2],[-1,-1],[0,-1],[0,-1],[-1,-1],[0,-1],[-1,1],[1,-3],[-1,-3],[0,-2],[-1,0],[0,1],[-1,3],[0,1],[-1,1],[0,4],[1,7],[0,3],[-1,2],[0,2],[2,2],[0,1],[3,10],[1,2],[0,-2],[1,3],[0,1],[1,0],[0,-2]],[[9064,796],[1,-2],[3,2],[2,5],[1,0],[-1,-10],[0,-3],[0,-1],[0,-3],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-1],[-1,4],[-1,1],[0,-1],[-1,-2],[0,-1],[0,-1],[-1,-8],[0,-3],[-1,-6],[-1,-2],[-1,-2],[-1,-1],[0,2],[-1,-2],[-1,-1],[-1,-3],[0,-2],[0,-1],[-1,-1],[0,-1],[0,-4],[0,-1],[-1,-3],[0,-2],[0,-1],[0,-2],[-3,-15],[0,-2],[-1,-8],[0,-3],[0,-7],[-1,-3],[0,-2],[0,1],[-1,-1],[-1,-3],[0,-5],[0,-5],[0,-4],[-1,-4],[0,-3],[0,-2],[-1,-2],[-2,-2],[0,-2],[-1,-5],[-1,-5],[0,-4],[0,-25],[-1,-18],[0,-1],[0,1],[0,2],[0,2],[0,3],[0,4],[0,2],[0,2],[-1,4],[-1,-2],[-1,0],[0,11],[0,6],[0,2],[0,2],[-1,2],[0,2],[0,2],[1,2],[0,1],[0,2],[1,7],[0,2],[2,6],[0,2],[0,3],[3,13],[1,7],[0,2],[1,1],[0,3],[1,4],[0,5],[0,9],[0,2],[1,4],[0,2],[0,3],[1,5],[1,2],[0,2],[0,5],[0,2],[1,7],[0,3],[1,2],[1,2],[0,2],[0,2],[0,2],[0,2],[0,2],[1,3],[0,2],[0,5],[0,4],[1,4],[0,3],[1,17],[1,7],[1,1],[1,0],[0,-1],[2,-10],[0,-1],[0,-1],[2,-7]],[[9133,1018],[-1,-3],[-1,-2],[-1,0],[-2,2],[-2,-1],[0,-2],[-3,-12],[-1,-3],[-2,-2],[-2,-4],[-2,-7],[-2,-16],[-2,-12],[-1,-10],[-1,-8],[-1,-2],[-1,-9],[-2,0],[0,1],[-1,-1],[0,-1],[-1,-3],[0,-1],[-1,6],[0,6],[-1,4],[-1,-3],[-1,-6],[0,-5],[0,-5],[0,-2],[0,-3],[0,-2],[0,-3],[-1,-5],[-1,-3],[0,-6],[0,-2],[-2,-11],[-3,-11],[-1,-12],[-2,-6],[0,-5],[-1,-15],[-2,-5],[-1,3],[-2,-9],[-1,-8],[-1,-6],[0,-5],[0,-5],[-1,-4],[-1,3],[-1,0],[0,3],[-1,4],[1,3],[0,4],[0,3],[1,5],[1,3],[0,3],[0,5],[0,4],[-1,6],[1,4],[0,-1],[0,-5],[0,-4],[1,-3],[0,-2],[1,4],[0,5],[0,5],[0,3],[-1,7],[1,2],[1,-6],[1,-2],[1,2],[1,7],[1,10],[0,4],[0,6],[-1,4],[-1,1],[-1,5],[0,8],[2,1],[1,-6],[0,-2],[1,5],[0,7],[1,5],[2,8],[2,9],[1,5],[0,2],[1,5],[0,4],[0,6],[0,6],[0,3],[0,2],[0,2],[1,2],[0,1],[1,-2],[0,-2],[1,-3],[1,0],[0,3],[1,4],[2,18],[0,5],[1,3],[3,5],[0,1],[0,4],[0,2],[0,2],[0,3],[0,10],[0,3],[-1,5],[0,3],[1,7],[0,4],[1,3],[1,4],[0,1],[0,-2],[0,-3],[0,-6],[0,-3],[1,-10],[0,-2],[0,-3],[0,-3],[0,-3],[0,-2],[1,-3],[0,-3],[1,-1],[7,6],[1,6],[1,10],[1,4],[1,2],[0,2],[0,6],[0,2],[1,3],[1,4],[1,6],[0,1],[0,1],[1,0],[0,1],[0,1],[0,5],[0,1],[1,0],[0,1],[2,5],[1,1],[1,-2],[2,-9],[0,-5],[-1,-5],[-1,-2],[0,-1],[0,-4],[1,-2],[1,-9],[0,-4],[-1,-5]],[[9182,1239],[-2,-10],[-1,-8],[-1,-4],[-1,-10],[-1,-5],[-1,-3],[-1,-6],[-1,-2],[-1,-7],[-1,-6],[0,-9],[0,-4],[0,-1],[-1,-1],[0,-3],[0,-3],[-1,-5],[-1,-2],[-1,-7],[-1,-4],[0,-2],[-1,1],[-1,0],[0,-3],[0,-4],[-2,-3],[-1,-5],[-2,-19],[-2,-9],[-1,-9],[-1,-1],[-1,7],[0,1],[0,-3],[0,-4],[0,-2],[-1,-3],[0,-1],[-3,0],[0,2],[1,2],[0,6],[0,11],[1,1],[1,5],[0,1],[0,2],[0,1],[1,5],[1,8],[0,2],[0,3],[0,1],[0,1],[0,1],[1,7],[1,7],[0,2],[2,0],[2,4],[1,10],[1,10],[1,10],[2,18],[2,5],[1,0],[1,0],[1,3],[0,2],[1,3],[0,2],[-1,6],[1,5],[1,3],[1,1],[1,4],[2,5],[2,-5],[3,5]],[[8922,1252],[0,-2],[0,-4],[0,-6],[0,-6],[0,2],[-1,0],[0,4],[0,2],[0,1],[0,3],[0,5],[1,0],[0,1]],[[9189,1291],[0,-1],[-1,2],[-1,2],[1,2],[0,1],[0,3],[0,2],[0,-2],[1,-4],[0,-2],[0,-2],[0,-1]],[[9227,1463],[0,-3],[1,2],[0,4],[1,2],[0,-5],[-3,-18],[0,-3],[0,-9],[-1,-4],[-1,-7],[-1,-2],[0,-3],[0,-2],[0,-1],[0,-2],[-1,-8],[-1,-3],[0,-3],[-1,-5],[-1,-1],[0,-5],[-1,-4],[-1,-7],[-1,-6],[-1,4],[-1,0],[0,2],[0,2],[-1,2],[0,3],[0,2],[0,3],[0,3],[1,3],[3,-1],[1,3],[1,14],[1,6],[3,16],[3,29],[2,10],[0,-1],[-1,-2],[0,-2],[0,-3]],[[9236,1508],[0,-7],[-1,1],[-2,4],[0,6],[0,6],[0,4],[1,1],[1,-1],[1,-2],[0,-6],[0,-6]],[[9250,1602],[0,-3],[-1,0],[-1,-3],[0,3],[0,5],[0,5],[0,4],[1,4],[0,4],[1,4],[0,2],[1,-4],[0,-3],[0,-5],[0,-4],[0,-6],[-1,-3]],[[9257,1684],[0,-2],[-1,2],[-1,3],[-1,4],[-1,5],[1,8],[1,1],[1,-3],[0,-3],[1,-6],[0,-3],[0,-3],[0,-3]],[[9279,1858],[-1,-3],[0,-1],[-1,0],[-1,2],[0,5],[1,4],[2,7],[0,3],[1,5],[0,3],[0,3],[0,2],[0,1],[0,6],[1,4],[1,1],[1,-1],[0,-5],[0,-6],[-1,-5],[0,-4],[-3,-10],[0,-5],[-1,-3],[1,-3]],[[9277,1904],[-1,-2],[-1,0],[0,1],[-1,3],[0,3],[0,3],[0,3],[1,0],[1,-4],[1,-1],[0,-2],[0,-2],[0,-2]],[[9292,1941],[-1,-2],[-1,2],[0,4],[-1,9],[0,4],[0,2],[0,1],[1,1],[1,0],[1,-3],[0,-1],[1,-2],[1,-4],[0,-5],[-1,-2],[-1,-2],[0,-2]],[[9301,2075],[0,-2],[1,2],[0,-5],[0,-8],[-1,-5],[0,-4],[-1,-6],[0,-4],[0,-2],[-1,-4],[0,-5],[0,-6],[0,-5],[0,-2],[1,-3],[0,-1],[0,-2],[0,-1],[0,-1],[0,-5],[0,-2],[0,-3],[-1,-2],[0,-2],[0,-1],[0,-1],[0,-3],[-1,-1],[0,-1],[0,1],[-1,-1],[0,-1],[0,-3],[0,-1],[-1,0],[0,1],[-2,5],[0,1],[0,1],[0,1],[0,3],[0,3],[0,8],[0,4],[0,4],[1,8],[0,3],[1,6],[0,3],[0,3],[1,5],[0,3],[1,8],[0,2],[0,2],[0,3],[0,2],[0,2],[0,1],[0,1],[0,1],[2,9],[0,2],[1,-2],[0,-3]],[[9290,2103],[-1,-4],[-1,1],[0,4],[-1,4],[0,3],[0,3],[1,3],[1,3],[0,1],[1,-2],[1,-7],[-1,-5],[0,-4]],[[9336,2288],[0,-1],[-1,0],[-1,-1],[0,-1],[0,-2],[0,-3],[-1,-3],[1,-2],[0,-3],[0,-3],[-1,-3],[0,-2],[0,-1],[-1,-2],[0,-2],[-1,-15],[-1,-11],[-1,-2],[0,-3],[0,-1],[-1,-2],[-1,-2],[0,-2],[-1,-9],[0,-1],[-4,-1],[-2,-3],[-1,-8],[-1,-4],[-2,-21],[0,-6],[0,-3],[-1,1],[0,5],[0,4],[0,2],[-1,2],[-1,2],[-1,0],[-1,-2],[0,3],[0,4],[0,3],[0,4],[0,2],[0,3],[1,2],[0,4],[0,3],[-1,6],[0,2],[1,4],[-1,2],[-1,7],[0,2],[0,3],[0,5],[1,2],[1,1],[2,-4],[0,3],[1,10],[1,4],[1,3],[1,1],[2,0],[0,1],[0,3],[0,1],[1,0],[1,-2],[0,-1],[1,1],[1,4],[1,5],[1,11],[0,2],[0,3],[0,5],[1,2],[0,11],[1,9],[1,4],[0,11],[0,4],[0,3],[3,14],[1,3],[1,4],[0,2],[1,-4],[1,-4],[0,-5],[0,-12],[0,-3],[0,-2],[0,-3],[0,-3],[0,-3],[1,-4],[0,-3],[0,-2],[0,-6],[-1,-3],[1,-5],[0,-3],[-1,-2],[0,-3]],[[9344,2326],[0,-6],[-1,0],[-1,2],[-1,0],[-1,-1],[0,5],[-1,2],[-1,3],[-1,4],[0,6],[0,6],[0,6],[1,4],[1,2],[0,2],[1,3],[2,9],[0,3],[1,3],[1,2],[1,-1],[1,-4],[0,-4],[0,-6],[0,-5],[0,-3],[1,-4],[-1,-4],[0,-4],[-1,-4],[0,-4],[-1,-7],[0,-5]],[[9323,2365],[0,-1],[-2,1],[-1,-1],[-1,2],[-1,3],[0,6],[-1,8],[1,9],[2,2],[2,-4],[1,-10],[0,-15]],[[8960,3224],[-1,-2],[0,1],[0,2],[1,1],[0,-1],[0,-1]],[[8967,3218],[1,-9],[2,-18],[1,-9],[1,-4],[0,-7],[-1,-5],[0,-5],[0,-5],[-1,-9],[-1,-4],[0,-16],[0,-2],[-1,-2],[-1,-1],[0,-2],[0,-2],[1,1],[1,-1],[0,-4],[1,-2],[0,-6],[0,-4],[1,-13],[0,-10],[0,-4],[-1,20],[-1,12],[0,-1],[-1,-3],[1,-5],[0,-6],[0,-1],[1,-3],[0,-13],[1,-1],[-1,-7],[1,0],[1,-1],[0,-4],[0,-1],[0,-1],[-1,-1],[0,-2],[1,-2],[1,-2],[0,-3],[0,-2],[0,-1],[-1,0],[0,1],[0,-2],[0,-1],[-1,-1],[0,-1],[0,-1],[0,-2],[1,-4],[1,-1],[0,1],[1,5],[0,-2],[0,-4],[1,-4],[0,-4],[0,-2],[0,-2],[0,-2],[0,-3],[-1,1],[0,1],[-1,-3],[0,-4],[0,-1],[1,-1],[0,-1],[0,-2],[0,-2],[0,-3],[0,-1],[0,-2],[0,-1],[0,-2],[0,-1],[1,0],[1,4],[0,1],[1,-2],[0,-6],[1,-27],[0,-10],[1,-10],[0,-5],[0,-17],[1,-4],[0,-5],[0,-5],[0,-11],[0,-11],[1,-6],[0,-4],[0,-6],[0,-6],[-1,-10],[1,-6],[0,-6],[0,-14],[-1,-20],[1,-4],[-1,-4],[0,-29],[0,-4],[-1,-8],[-1,-8],[0,-7],[-1,-6],[0,11],[0,4],[0,2],[0,3],[0,2],[0,3],[1,2],[0,1],[0,-1],[1,1],[0,3],[0,2],[-1,3],[0,1],[1,19],[0,3],[0,3],[0,1],[0,1],[-1,0],[0,-1],[0,-1],[0,-6],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-6],[-1,-11],[-1,-7],[0,-10],[0,-3],[-1,-8],[0,-6],[0,-2],[1,4],[0,-3],[0,-5],[0,-1],[0,-3],[0,-3],[0,-2],[0,-2],[0,-16],[0,-20],[-1,0],[0,6],[0,2],[-1,-1],[0,-3],[0,-3],[0,-3],[0,-2],[1,-4],[0,-2],[0,-3],[0,-5],[0,-2],[0,-2],[0,-2],[0,-3],[1,-4],[0,-4],[-1,-3],[0,-3],[0,-7],[0,-2],[1,-1],[0,-3],[0,-3],[0,-4],[0,-1],[1,-2],[0,-1],[0,1],[1,3],[0,4],[0,1],[0,-14],[2,-20],[0,-12],[0,2],[-1,-3],[0,-6],[-1,-9],[0,-5],[0,-2],[0,-2],[0,-6],[0,-3],[0,-2],[0,-5],[-1,-2],[1,-2],[0,-3],[0,-1],[1,3],[1,-1],[0,-1],[1,-1],[0,4],[0,6],[0,6],[-1,3],[0,2],[1,6],[0,5],[0,2],[1,-2],[0,-5],[0,-9],[0,-4],[2,-14],[0,-9],[1,-31],[-1,6],[-1,1],[0,-1],[-1,-4],[0,-2],[1,-2],[0,-2],[0,-2],[0,-1],[0,-1],[0,-1],[1,-1],[0,-1],[0,-1],[1,-1],[0,-2],[0,-5],[0,-2],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-1],[0,-4],[1,2],[0,6],[-1,8],[0,7],[1,-5],[0,-12],[2,-56],[2,-67],[1,-33],[2,-72],[1,-9],[3,-30],[2,-34],[1,-29],[1,-14],[1,-9],[0,-9],[1,-5],[0,-21],[0,-5],[0,-4],[1,-4],[1,-18],[0,-4],[0,-14],[0,-6],[2,-46],[3,-43],[1,-9],[0,-6],[7,-30],[0,-4],[0,-3],[0,-2],[0,-5],[0,-2],[1,-6],[0,-2],[0,-3],[-1,-3],[0,-3],[0,-10],[1,-7],[0,-4],[1,-3],[0,-4],[-1,-1],[0,1],[-1,3],[0,6],[0,5],[-1,6],[0,6],[0,8],[0,7],[-4,34],[-1,3],[0,1],[-1,7],[0,2],[0,1],[-1,2],[-1,2],[-1,8],[0,1],[0,1],[-1,2],[0,2],[0,2],[-1,6],[0,3],[0,2],[-4,25],[-1,5],[-2,4],[-8,9],[-1,1],[0,3],[2,3],[-7,4],[-1,3],[0,4],[-1,3],[0,2],[-3,1],[-1,0],[0,-3],[-1,-5],[1,-3],[1,-4],[1,-3],[-1,0],[0,-1],[0,-1],[0,-1],[1,-1],[0,1],[0,1],[8,0],[1,-2],[-1,-2],[-7,-2],[-6,-13],[-1,-3],[-2,-12],[-1,-7],[0,-8],[-1,-10],[0,-9],[0,-10],[0,-11],[0,-11],[-1,-9],[-1,-18],[-1,-17],[-2,-43],[-2,-38],[-2,-34],[-1,-19],[-2,-42],[0,-2],[-1,-1],[0,-2],[1,0],[0,-1],[0,-1],[0,-1],[0,-10],[0,-7],[-1,-2],[0,-5],[1,-31],[1,-30],[1,-15],[3,-34],[-1,0],[0,-2],[1,-1],[0,-2],[1,2],[0,-1],[1,-2],[2,-7],[1,-6],[3,-28],[0,-4],[0,-6],[1,-15],[-1,-6],[0,-11],[0,-3],[1,-12],[1,-18],[0,-5],[0,-5],[1,-5],[0,-3],[-1,0],[0,-2],[1,-2],[0,-1],[1,5],[0,-3],[0,-3],[0,-3],[-1,1],[-1,-1],[0,-6],[0,-2],[0,-3],[1,-2],[0,-1],[0,-2],[1,-8],[0,-6],[0,-2],[1,-1],[0,1],[1,2],[1,-2],[1,-1],[0,-1],[1,-3],[0,-1],[0,1],[1,2],[0,3],[0,4],[0,4],[-1,2],[-1,2],[-1,1],[-1,4],[0,1],[-1,-1],[0,1],[-1,3],[0,5],[0,5],[1,2],[0,-1],[3,-9],[0,-1],[2,-1],[1,2],[0,3],[0,5],[1,4],[1,-5],[0,-7],[1,-8],[0,-6],[0,-16],[0,-7],[0,-8],[0,-3],[0,-2],[0,-2],[0,-1],[0,-2],[0,-3],[1,-14],[0,-15],[0,-7],[0,-5],[0,-2],[1,-1],[1,-3],[-1,-5],[-1,-2],[0,-3],[0,-6],[0,-2],[0,-1],[0,-2],[-1,-1],[0,-2],[0,-3],[0,-6],[0,-4],[-1,-8],[-1,-4],[1,-17],[0,-4],[-1,-2],[0,-1],[0,-2],[0,-5],[0,-3],[-1,-2],[0,-1],[0,3],[0,6],[0,6],[0,4],[0,4],[0,2],[0,5],[0,3],[0,3],[0,2],[0,6],[0,5],[-1,23],[0,5],[-1,18],[0,5],[0,5],[0,5],[0,4],[0,2],[0,-1],[1,0],[0,2],[0,8],[0,6],[-1,1],[-2,0],[1,-9],[0,-2],[-1,0],[0,6],[-1,6],[0,2],[-1,2],[-1,2],[0,2],[-3,-2],[-2,1],[-1,2],[-1,3],[0,-2],[-2,-3],[-1,0],[-1,4],[0,7],[-1,15],[0,7],[0,3],[-1,1],[0,-1],[0,-3],[0,-1],[-1,0],[-1,-1],[-3,-9],[-2,-13],[-1,-19],[-2,-40],[-2,-15],[0,-5],[0,-2],[0,-9],[-1,-18],[0,-3],[0,-3],[0,-4],[0,-3],[0,-5],[0,-5],[-1,-4],[0,-5],[0,-2],[0,-5],[0,-3],[-2,-3],[0,-4],[0,-10],[-1,-12],[0,2],[-1,1],[0,2],[-1,14],[-2,11],[0,3],[0,16],[-1,44],[-1,9],[-1,24],[0,9],[-1,21],[0,11],[1,5],[0,2],[0,4],[0,5],[1,6],[0,15],[5,79],[0,7],[0,7],[0,7],[0,8],[0,8],[-1,7],[-1,13],[0,3],[0,3],[0,8],[0,4],[0,3],[0,3],[0,3],[0,5],[-1,8],[0,8],[0,27],[1,6],[1,12],[0,3],[0,3],[0,2],[0,3],[0,4],[0,2],[1,3],[0,1],[0,8],[1,18],[0,5],[1,4],[1,10],[1,3],[0,6],[-1,32],[0,4],[0,9],[-1,41],[0,4],[-1,15],[-5,53],[-1,19],[0,12],[0,9],[1,15],[1,1],[0,4],[1,2],[0,3],[0,2],[-1,3],[0,2],[1,3],[0,2],[0,3],[0,2],[1,5],[0,5],[0,9],[0,5],[1,5],[-1,9],[1,9],[0,8],[1,7],[0,5],[0,4],[-1,20],[1,15],[0,9],[1,9],[0,3],[0,10],[1,16],[0,14],[0,9],[0,4],[0,14],[0,14],[0,9],[0,8],[-1,9],[0,2],[1,3],[0,1],[0,2],[-1,2],[0,1],[0,3],[1,4],[0,2],[0,9],[0,4],[0,4],[0,10],[0,37],[-1,13],[1,4],[0,5],[0,5],[0,4],[-1,4],[0,9],[-1,4],[-1,23],[0,6],[0,5],[0,3],[0,3],[0,5],[0,10],[1,14],[0,17],[0,2],[0,2],[0,3],[0,3],[0,2],[1,5],[0,1],[0,1],[0,6],[0,3],[2,16],[1,7],[0,10],[1,11],[0,9],[0,6],[0,6],[-3,24],[0,5],[0,2],[0,3],[1,2],[0,2],[-1,1],[0,1],[0,2],[-1,10],[-1,11],[0,6],[0,7],[-4,22],[0,3],[0,1],[0,1],[0,4],[0,2],[0,2],[0,1],[-2,-1],[-1,6],[-1,9],[-3,12],[0,6],[2,-4],[1,-1],[0,-1],[0,1],[1,2],[0,2],[0,2],[-1,1],[0,13],[-1,7],[-2,5],[-1,7],[1,7],[0,10],[-1,19],[0,5],[0,4],[0,4],[0,5],[0,2],[1,4],[0,1],[0,1],[0,7],[-1,13],[0,10],[0,4],[0,4],[0,3],[0,4],[0,5],[0,2],[0,5],[0,2],[0,1],[3,16],[1,5],[0,4],[0,9],[0,3],[0,2],[0,3],[1,2],[0,1],[0,1],[0,6],[-1,1],[0,2],[1,7],[0,9],[0,18],[0,5],[1,8],[0,5],[0,2],[0,4],[0,3],[0,2],[0,4],[0,3],[0,1],[0,5],[1,2],[0,6],[0,2],[0,2],[0,9],[1,5],[-1,5],[0,2],[0,2],[0,4],[0,2],[-1,4],[0,6],[-1,6],[0,3],[0,3],[0,2],[0,2],[0,3],[0,3],[0,2],[-1,13],[0,3],[0,5],[0,2],[0,3],[0,2],[0,1],[0,3],[-1,6],[1,3],[2,7],[2,9],[0,1],[2,2],[2,8],[4,8],[0,-2],[0,-2],[0,-3],[1,-3],[0,-3],[-1,-3],[0,-2],[1,-4],[-1,-2],[0,-3],[0,-3],[0,-3],[1,-3],[0,-2],[1,-2],[1,0],[1,3],[2,0],[0,1],[2,10],[0,6],[1,10],[1,3],[1,7],[1,-2],[1,2],[-1,6],[-1,4],[-1,2],[-1,-2],[-1,-5],[0,-1],[0,4],[0,1],[0,7],[-1,6],[0,5],[0,5],[1,4],[1,0],[0,-1],[1,-1],[0,1],[1,2],[0,1],[1,-1],[0,-7],[1,-3],[1,1],[2,14],[0,2],[0,3],[0,2],[0,3],[0,11],[-1,5],[0,5],[-1,5],[-1,-9],[0,-14],[-1,-9],[0,-2],[-1,-3],[-1,1],[0,3],[1,4],[0,7],[1,15],[1,12],[1,10],[-1,10],[-1,10],[-2,14],[-1,11],[-2,14],[-1,5],[-1,11],[0,5],[-1,5],[-1,5],[-2,6],[0,2],[1,-1],[1,-5],[1,-1],[2,-2],[0,1],[2,6],[0,-1],[0,-2],[0,-2],[0,-3],[0,-3],[1,-4],[1,3],[-1,7],[1,1],[0,-1],[1,-2],[0,-1],[1,5],[-1,1],[0,2],[0,3],[0,5],[0,1],[1,1],[0,3],[0,2],[1,3],[0,3],[-1,2],[0,4],[0,2],[1,2],[0,1],[1,0],[0,-1],[0,-5],[1,-2],[0,-6],[0,-2],[1,-3],[0,-3],[0,-9],[1,-5]],[[9669,3273],[-1,-1],[0,1],[-1,4],[-2,9],[-1,9],[-3,13],[-1,1],[-1,1],[0,1],[-1,1],[0,2],[-1,5],[-5,23],[-2,9],[0,11],[1,0],[1,-2],[1,-5],[1,-4],[0,-3],[0,-3],[1,-1],[1,-1],[0,-1],[1,-4],[1,-1],[1,-2],[0,-7],[0,-4],[1,-2],[0,-2],[1,0],[0,-2],[1,-7],[1,-1],[0,-1],[1,-1],[0,-3],[0,-4],[1,-5],[2,-8],[0,-5],[1,-1],[1,-4],[0,-2],[0,-3]],[[9609,3485],[6,-12],[2,4],[0,-3],[1,-2],[0,-1],[0,-1],[-1,-9],[-1,-5],[1,-8],[0,-10],[0,-3],[0,-3],[1,-4],[0,-3],[1,-4],[0,-3],[1,-10],[0,-2],[1,-1],[0,-2],[0,-7],[0,-1],[1,-2],[1,-8],[0,-4],[2,-6],[1,-2],[0,-1],[1,-4],[1,-11],[0,-3],[0,-3],[0,-2],[0,-4],[0,-6],[1,-4],[0,-5],[-1,-4],[0,-4],[1,-6],[-1,2],[0,6],[-1,1],[0,1],[-1,1],[0,6],[-2,8],[0,5],[0,6],[-1,2],[-2,0],[-1,2],[-2,10],[0,2],[0,3],[0,1],[-1,3],[0,3],[-1,1],[0,2],[-1,3],[0,4],[0,2],[0,5],[-3,15],[0,2],[1,3],[0,5],[-1,5],[-1,14],[-1,4],[-1,2],[1,5],[-1,2],[-2,6],[-2,4],[-1,2],[-1,4],[2,1],[1,2],[1,3],[0,2],[1,5],[0,2],[1,2]],[[9571,4424],[0,-1],[1,1],[0,-2],[0,-3],[0,-1],[0,-12],[0,-3],[1,-4],[1,-2],[0,-7],[1,-6],[-1,-4],[0,-7],[0,-6],[0,-3],[0,-4],[0,-2],[0,-3],[-2,-10],[-2,0],[-3,1],[-1,-1],[-2,-5],[-1,-4],[0,-5],[-1,-2],[-1,3],[0,1],[-1,0],[0,2],[0,1],[-1,-1],[-1,-5],[0,-5],[-1,-4],[-1,-4],[-2,-5],[-1,-5],[-1,-2],[-1,0],[-1,-3],[-1,-11],[-4,-12],[-2,-9],[-1,-9],[-1,-6],[-1,-13],[0,4],[-1,4],[0,7],[-1,7],[0,4],[1,3],[3,3],[2,8],[1,11],[0,9],[1,6],[1,9],[1,7],[0,8],[0,6],[0,9],[1,5],[1,8],[1,9],[1,8],[0,4],[-1,0],[-1,-1],[-1,2],[0,1],[-1,-3],[-1,-3],[-1,0],[1,3],[1,3],[1,3],[5,0],[8,20],[1,5],[1,5],[2,8],[1,6],[3,7],[1,-1],[0,-10],[0,-2]],[[9845,5071],[-1,0],[-1,2],[-1,-1],[1,2],[0,3],[0,2],[-1,0],[0,-1],[-1,-6],[-1,-2],[-2,1],[0,-1],[-1,-3],[0,2],[1,2],[-1,4],[-1,3],[-1,1],[-1,-2],[1,-3],[-2,-1],[0,-2],[0,-5],[0,-3],[1,-2],[1,1],[-1,-2],[-1,-3],[0,-3],[0,-3],[-1,-3],[0,-2],[-1,-1],[-1,-2],[-1,-5],[-2,-5],[0,-3],[0,7],[0,3],[-1,2],[-3,6],[-2,8],[-1,0],[0,-2],[-1,-2],[0,-4],[0,-5],[0,-4],[0,-4],[-1,-7],[0,-2],[0,-4],[0,-3],[0,-4],[0,-4],[0,-4],[-1,-1],[-2,2],[-1,-1],[0,-2],[-1,-3],[0,-4],[0,-7],[0,-6],[-5,-17],[-1,-1],[-2,5],[-1,2],[0,4],[0,1],[-1,2],[-1,0],[0,1],[0,1],[0,2],[0,1],[-2,0],[-3,-10],[1,-4],[2,-6],[1,-2],[1,-2],[0,-3],[1,-2],[0,-4],[0,-2],[0,-3],[0,-1],[-1,-8],[-2,3],[-3,8],[-2,2],[2,-7],[1,-5],[0,-5],[0,-2],[-1,0],[-1,-1],[0,-1],[0,-3],[0,-1],[-1,-1],[0,-6],[0,-3],[-2,-1],[-5,10],[-2,-4],[1,-4],[0,-5],[1,-4],[1,-2],[0,-2],[-3,-5],[-1,1],[-1,4],[0,-7],[1,-2],[3,-2],[0,-1],[0,-1],[0,-1],[0,-2],[0,-1],[1,0],[1,2],[0,-6],[-1,-3],[-1,-2],[-1,-6],[-1,2],[-2,-1],[0,1],[-4,12],[-3,8],[1,-3],[3,-15],[1,-4],[0,-4],[0,-5],[0,-4],[0,-3],[-1,-1],[-1,1],[0,-1],[0,-4],[0,-1],[-1,-1],[0,-7],[-1,-7],[-1,-1],[0,8],[0,4],[0,4],[0,4],[-1,3],[0,1],[0,-4],[0,-5],[-1,-2],[0,-2],[0,-1],[0,-4],[0,-2],[1,-2],[0,-2],[-1,-2],[-1,-3],[-2,-1],[-1,1],[-1,4],[-1,-2],[-2,-7],[-1,0],[0,-3],[-1,-1],[0,1],[0,1],[-1,0],[1,-4],[1,-1],[0,-3],[1,-7],[-1,-2],[-3,-2],[-1,-2],[-1,-6],[0,-2],[0,4],[1,3],[0,3],[0,4],[-1,3],[0,2],[0,2],[0,2],[-1,1],[0,-1],[0,-2],[0,-2],[0,-2],[0,-2],[0,-9],[0,-2],[0,-1],[0,-1],[-1,-4],[0,-7],[0,-3],[0,-3],[-1,-3],[-1,-6],[-2,-2],[-1,-6],[-1,-1],[-1,1],[-1,0],[0,-1],[-2,-7],[-1,-1],[-4,-3],[-1,1],[0,3],[-1,1],[0,-3],[0,-4],[1,-2],[1,1],[-1,-6],[0,-3],[0,-3],[-1,-3],[-1,-2],[-2,-2],[0,2],[-1,4],[0,1],[-2,0],[0,-4],[1,1],[0,-1],[1,-8],[0,-3],[0,-1],[0,-2],[0,-2],[1,-2],[1,-3],[0,-2],[0,-2],[-1,-3],[-1,-4],[-2,-3],[-1,-5],[-1,-10],[0,-12],[-1,-25],[0,-16],[0,-5],[-1,-7],[0,-4],[-3,-7],[-1,0],[-1,3],[-1,5],[0,2],[-1,1],[-1,3],[0,3],[0,2],[0,2],[1,3],[-1,1],[-1,6],[-1,1],[-1,0],[-1,2],[-1,4],[0,11],[-2,10],[0,5],[0,7],[0,2],[4,3],[-1,3],[-1,2],[-1,0],[-1,-1],[0,-1],[-1,2],[-2,6],[0,2],[0,2],[0,3],[0,2],[0,3],[-1,4],[0,2],[0,4],[0,5],[0,4],[0,4],[-1,3],[-2,3],[-1,7],[-1,0],[-5,15],[-1,3],[-1,2],[-1,4],[3,-2],[1,2],[-2,11],[-1,2],[-1,0],[-1,2],[0,-2],[0,-1],[0,-2],[0,-2],[0,-3],[0,-1],[-1,-1],[0,1],[0,2],[-1,-1],[0,-2],[1,-2],[0,-1],[-5,2],[-7,-2],[-1,-3],[0,-1],[-1,0],[-1,1],[-1,7],[-2,2],[-1,2],[-6,-1],[-1,-3],[-4,-3],[-7,-15],[-1,3],[0,6],[-1,1],[0,-1],[0,-3],[0,-4],[0,-2],[0,-3],[0,-2],[-1,0],[-2,-9],[-2,-2],[-1,-1],[0,2],[-1,4],[-1,0],[0,-2],[0,-2],[0,-3],[0,-3],[-1,-1],[-2,11],[-1,1],[1,-8],[0,-4],[0,-4],[0,-2],[-1,2],[-1,2],[0,1],[0,-6],[0,-5],[-2,-3],[-1,-1],[-1,1],[-1,4],[-1,6],[0,5],[-1,2],[0,-2],[0,-3],[0,-2],[0,-3],[0,-2],[0,-10],[0,-3],[0,-2],[-1,-1],[-2,-5],[-1,0],[0,1],[0,-2],[1,-3],[0,-2],[0,-3],[0,-2],[-2,-5],[-1,-2],[-3,-22],[-5,-32],[-5,-29],[-2,-10],[-1,-3],[-1,-1],[-3,0],[0,3],[0,6],[0,11],[0,5],[1,9],[0,5],[0,4],[0,4],[0,3],[0,3],[1,19],[0,4],[1,4],[2,6],[0,5],[0,7],[-1,9],[0,6],[1,6],[0,5],[0,7],[-1,6],[0,3],[1,-3],[1,2],[1,6],[0,8],[0,1],[0,1],[1,3],[0,2],[0,3],[0,2],[-1,-2],[0,-3],[-1,-2],[0,1],[-1,2],[-1,1],[-5,-8],[-1,-3],[-2,-13],[-1,-2],[1,-4],[1,3],[1,11],[1,4],[3,4],[1,1],[-1,-4],[-2,-1],[-1,-3],[-3,-15],[-3,-5],[-1,-6],[-1,-6],[-2,-6],[-1,-5],[-4,-8],[-2,0],[0,10],[-1,-4],[0,-4],[0,-4],[1,-3],[-1,-7],[-3,-7],[-2,-9],[0,-2],[-1,-1],[0,1],[0,1],[-1,1],[0,1],[0,2],[0,3],[0,1],[0,2],[0,2],[1,3],[0,2],[-1,0],[-1,-3],[0,-5],[-1,-4],[0,-2],[-1,1],[-2,5],[0,-3],[0,-3],[1,-4],[0,-3],[0,-3],[1,-1],[0,2],[0,4],[1,0],[1,-1],[0,-1],[0,-2],[0,-8],[1,-3],[0,-2],[1,-4],[0,-4],[-1,-1],[0,-3],[-1,-9],[-3,-22],[-1,-2],[-2,-2],[0,-3],[-1,-5],[-1,-3],[-1,1],[0,2],[-1,3],[0,4],[0,3],[0,1],[0,1],[0,2],[0,1],[0,2],[0,2],[0,1],[0,7],[-1,12],[-1,6],[0,3],[-2,8],[-2,14],[-1,4],[-1,3],[-1,1],[-1,-2],[-1,-1],[0,-3],[0,-4],[-1,4],[0,-1],[-1,-6],[1,-7],[1,-2],[-1,-9],[-2,-3],[-2,-1],[-1,-3],[0,-3],[-1,-2],[0,-4],[0,-6],[0,-4],[1,-4],[1,-3],[-1,0],[-2,5],[-1,0],[0,1],[0,7],[0,1],[0,1],[1,-1],[0,1],[0,1],[1,3],[0,2],[0,4],[1,3],[1,7],[-2,-2],[-1,2],[-1,6],[0,2],[-1,1],[-3,1],[-1,-3],[0,1],[0,2],[0,2],[-1,3],[0,-1],[-1,-2],[0,-1],[-1,1],[0,1],[-1,0],[-2,0],[-1,-1],[0,-5],[1,-6],[0,-4],[1,-3],[0,-1],[1,0],[1,4],[0,1],[0,1],[0,1],[1,-2],[-1,-3],[0,-4],[-1,-9],[-1,-5],[0,-2],[-1,0],[-2,2],[-1,0],[-1,-2],[-2,-8],[0,-4],[-2,-5],[0,-3],[0,-6],[0,-6],[0,-6],[0,-3],[0,-8],[0,-5],[0,-3],[1,-3],[0,-5],[-1,-6],[-1,-5],[-1,-1],[-1,1],[-2,-3],[0,-2],[0,-2],[0,-1],[1,1],[-1,-3],[0,-3],[0,-2],[0,-4],[0,-1],[0,-1],[1,-6],[2,-7],[0,-4],[1,-5],[0,-5],[0,-5],[0,-10],[0,-11],[-1,-4],[0,-3],[0,-3],[-1,-1],[0,-1],[-1,1],[0,2],[1,4],[1,3],[0,2],[0,3],[-1,1],[-1,-4],[-1,3],[-1,-1],[0,-1],[-2,-3],[0,-2],[0,-2],[0,-2],[0,-3],[0,-2],[0,-3],[0,-2],[0,-2],[0,-3],[1,-2],[1,-2],[1,-4],[1,-2],[0,-5],[1,-6],[0,-7],[0,-4],[-1,-2],[-1,-4],[-1,-1],[0,1],[0,2],[0,2],[1,2],[0,3],[-1,5],[0,3],[-2,3],[0,1],[0,3],[0,1],[-2,5],[-1,-1],[0,1],[-1,0],[0,-2],[0,-2],[0,-2],[0,-1],[0,1],[-1,-1],[0,1],[0,-1],[0,-3],[0,-2],[2,-15],[0,-2],[2,-4],[0,1],[1,-6],[0,-5],[0,-4],[-1,-5],[-1,-4],[-2,-2],[-2,-5],[1,11],[1,6],[-1,3],[0,-2],[-1,-3],[-1,-14],[0,-5],[-3,1],[0,-4],[-1,0],[0,-5],[0,-2],[1,0],[0,1],[0,2],[1,1],[0,-1],[1,4],[0,-3],[0,-2],[0,-2],[0,-4],[-1,-3],[-7,-24],[-3,-16],[-2,-19],[-2,-25],[-2,-18],[-4,-49],[0,-12],[-1,-17],[0,-6],[0,-14],[1,-19],[1,-11],[2,-9],[2,-6],[2,-4],[2,-5],[3,0],[0,-5],[-1,-6],[-1,-3],[-1,-9],[0,-1],[1,1],[0,1],[4,14],[-1,11],[0,12],[0,10],[0,5],[1,7],[1,3],[0,1],[2,1],[1,0],[0,3],[1,0],[0,-1],[0,-1],[0,-1],[1,-5],[0,-2],[1,0],[1,-2],[1,-1],[0,-1],[4,-18],[2,-4],[4,-2],[0,-4],[1,-5],[1,-5],[1,-12],[-2,-8],[0,-1],[-1,-8],[1,-5],[0,-6],[-1,-3],[-1,-3],[-1,-2],[0,-7],[-1,-6],[0,-4],[-1,-8],[-3,-6],[-2,-5],[-2,-8],[-1,-9],[0,-5],[0,-18],[0,-17],[0,-10],[1,-12],[1,-11],[-1,-8],[0,-12],[-1,-6],[0,-5],[0,-16],[0,-9],[0,-15],[1,-10],[2,-1],[0,-2],[1,0],[1,-1],[1,5],[1,2],[1,2],[1,-3],[2,1],[1,2],[0,-2],[0,-3],[1,-2],[0,-3],[1,-3],[-1,-5],[0,-5],[-1,-6],[1,-7],[1,-8],[-1,-3],[0,-4],[0,-4],[0,-7],[0,-2],[1,-2],[0,-3],[-1,0],[0,-2],[0,-3],[0,-3],[1,-1],[0,-6],[1,-11],[0,-2],[0,-3],[1,-3],[0,-6],[-1,-5],[0,-11],[1,-3],[0,-5],[0,-5],[-2,-3],[0,-4],[-2,-2],[-1,-2],[-2,-6],[0,-1],[0,-1],[0,-3],[0,-3],[-1,-2],[0,-2],[0,-4],[-1,-4],[0,-6],[-1,1],[-2,2],[-1,6],[-1,2],[-1,2],[-1,9],[-1,3],[0,11],[-2,5],[-1,12],[-2,1],[0,1],[0,1],[1,2],[1,1],[1,3],[1,4],[0,4],[1,6],[0,5],[0,3],[1,2],[2,1],[2,6],[0,6],[2,8],[1,1],[1,4],[1,10],[0,4],[-2,1],[-1,4],[0,-1],[-1,-8],[-1,-13],[-1,-5],[-2,5],[-2,8],[-2,1],[-1,-4],[0,-7],[-2,-3],[-1,2],[0,2],[-1,-1],[0,-1],[-1,-4],[0,-1],[0,-2],[0,-2],[-1,-1],[0,-3],[-1,-3],[2,-9],[2,-2],[0,-5],[1,-4],[1,-5],[-1,-3],[0,-1],[-1,-1],[-1,-1],[0,-1],[0,-2],[0,-4],[-2,-1],[0,-2],[-2,-4],[-2,-6],[-3,-7],[-1,-1],[-1,-6],[0,-4],[-1,-5],[-1,-3],[0,-3],[0,-2],[1,-4],[0,-4],[-1,-7],[0,-11],[-2,-9],[0,-10],[-1,-7],[0,-8],[-1,-10],[-1,-12],[-1,-12],[-1,-7],[0,-4],[0,-14],[-1,-10],[0,-12],[0,-22],[0,-6],[1,-6],[0,-9],[0,-13],[1,-14],[1,-12],[1,-13],[0,-4],[2,-7],[1,-5],[1,-14],[0,-8],[1,-5],[2,-5],[0,-5],[0,-3],[0,-2],[-1,-2],[0,-2],[0,-3],[0,-6],[1,-4],[0,-3],[-1,-4],[-1,-3],[0,-1],[-1,0],[0,-2],[-1,-4],[0,-2],[0,-2],[0,-3],[-1,-3],[0,-2],[0,-2],[-1,-3],[-1,-4],[-1,-2],[0,-2],[-1,-3],[0,-2],[-1,-11],[-1,-4],[-1,-3],[-1,0],[0,1],[-1,2],[0,1],[-2,-2],[-3,0],[-3,-6],[-1,0],[-1,2],[0,2],[-1,2],[-1,3],[0,3],[0,5],[0,2],[-2,7],[-2,0],[-5,-7],[-4,-8],[0,-1],[0,-2],[-6,-21],[-1,-8],[-2,-7],[-2,-15],[-2,-9],[-1,-5],[0,-3],[0,-1],[-2,-2],[-1,-5],[-2,-10],[-1,-11],[-1,-8],[-1,-12],[-1,-11],[0,-5],[-2,-46],[0,-14],[1,-11],[-1,-2],[0,-2],[-1,-5],[1,-2],[1,-4],[1,-5],[0,-6],[0,5],[0,5],[-1,5],[0,4],[1,2],[1,-7],[1,-19],[-1,-3],[0,-15],[0,-5],[-1,0],[-1,5],[-1,5],[0,5],[-1,-1],[0,-1],[0,-1],[0,-2],[0,-1],[0,-3],[0,-2],[0,-2],[1,0],[1,-2],[0,-2],[0,-6],[0,-2],[0,-3],[0,-3],[0,-2],[0,-2],[1,-1],[0,2],[1,-1],[-1,-7],[0,-4],[1,-3],[0,-1],[0,1],[1,-1],[0,-6],[-1,-2],[0,-2],[-1,-2],[0,-4],[0,-3],[2,-10],[-1,7],[1,6],[1,2],[0,-3],[0,-5],[0,-5],[0,-3],[1,-2],[0,-3],[-1,-3],[1,-5],[0,-5],[0,-4],[0,-6],[-1,5],[0,-1],[-1,0],[0,-1],[-1,1],[0,1],[0,2],[0,2],[0,3],[-4,11],[-1,3],[0,5],[0,5],[1,4],[0,3],[1,3],[-2,-2],[0,-2],[-1,-7],[0,-1],[-1,1],[-1,2],[-1,2],[-1,0],[-3,-6],[0,-2],[-1,-3],[-1,-8],[-1,-4],[0,1],[-1,2],[-1,0],[-1,-2],[-1,-4],[0,-1],[-1,0],[-1,-3],[-4,-14],[-4,-20],[-4,-23],[0,-2],[-1,4],[0,6],[0,5],[-1,5],[1,5],[-1,2],[-1,10],[-1,4],[0,2],[-1,0],[0,-1],[-1,-1],[0,-2],[0,-2],[-1,-2],[-1,-3],[0,-1],[0,-4],[-1,-3],[1,-1],[1,-1],[-1,-2],[0,-4],[0,-3],[0,-7],[1,-3],[1,0],[1,0],[-1,3],[0,2],[-1,4],[1,1],[1,-1],[0,-1],[1,-1],[1,0],[0,-2],[1,-2],[0,-5],[0,-1],[-1,-2],[0,-1],[0,-1],[0,-3],[0,-2],[0,-2],[0,-3],[-1,-3],[0,2],[-1,1],[1,-8],[0,-2],[-1,-2],[-1,-1],[0,-1],[0,-5],[0,-3],[1,-2],[1,1],[0,-2],[0,-3],[1,-2],[0,-3],[0,-2],[-1,-2],[0,-5],[0,-5],[-1,-3],[-1,-2],[0,-1],[-1,4],[-1,4],[0,1],[0,-2],[0,-3],[0,-3],[0,-2],[0,-4],[0,-2],[1,-4],[1,-1],[1,-2],[-1,-3],[0,-4],[0,-3],[0,-2],[0,-1],[0,-2],[0,-3],[1,-1],[-1,-2],[-1,-3],[0,-1],[0,-2],[0,-4],[0,-1],[0,-2],[-1,-2],[0,-1],[0,-2],[1,2],[0,3],[1,2],[1,-1],[0,-2],[1,-7],[-1,-2],[-1,-6],[0,-1],[0,-2],[0,-3],[1,-1],[0,-2],[0,-3],[0,-3],[0,-1],[0,-1],[-1,-4],[0,-2],[-2,0],[0,-6],[0,-2],[-1,-2],[0,-5],[0,-2],[1,-1],[0,-2],[0,-4],[-1,-4],[0,-2],[-1,-3],[0,-1],[-1,2],[0,-2],[0,-2],[0,-6],[0,-1],[0,-3],[0,-2],[0,-1],[-1,-4],[0,-2],[0,-2],[0,-3],[0,-3],[0,-11],[-1,-4],[0,-2],[-1,-4],[0,-6],[-1,-1],[-1,-11],[-1,-5],[0,-4],[0,-2],[-2,-8],[-1,-2],[0,-7],[0,-3],[0,-2],[-1,-2],[0,-3],[-1,-6],[0,-3],[0,-2],[0,-4],[0,-1],[-1,1],[-1,-4],[-1,-6],[-1,-2],[-1,-11],[-1,2],[-1,-3],[-2,-3],[-2,-10],[-1,-11],[-1,-7],[1,-6],[-1,-7],[0,-3],[-2,1],[-1,-6],[-1,-8],[-1,-6],[0,-5],[-1,-3],[-1,-8],[-2,-9],[-1,-4],[-2,-7],[-2,-14],[-1,-8],[-1,-4],[-1,-5],[0,-5],[-1,-2],[0,-4],[-1,-4],[-1,-7],[-1,-5],[-1,-6],[0,5],[1,6],[2,19],[0,8],[-1,4],[1,7],[0,4],[-1,3],[-1,-1],[0,4],[0,4],[0,9],[0,10],[0,3],[-1,10],[-1,2],[-1,1],[-1,1],[0,1],[0,2],[-1,3],[0,5],[0,6],[0,19],[0,2],[0,3],[0,1],[-1,2],[0,2],[0,1],[0,1],[0,1],[0,2],[0,5],[0,5],[0,11],[0,3],[0,5],[0,3],[0,3],[0,5],[0,2],[0,32],[0,45],[-1,63],[-2,47],[-1,6],[0,5],[-1,8],[0,2],[-1,9],[1,1],[0,-1],[0,-2],[0,-1],[1,-3],[0,-1],[0,1],[1,-3],[0,-3],[0,-3],[0,-5],[0,3],[1,3],[1,5],[0,5],[-1,3],[-2,2],[0,1],[-1,5],[0,4],[-1,6],[-1,8],[0,9],[-2,33],[-2,33],[0,27],[-1,42],[0,17],[-1,21],[-1,51],[0,16],[0,5],[-1,17],[0,5],[0,8],[0,15],[0,5],[-1,10],[-1,19],[0,4],[0,6],[0,12],[-2,32],[-2,55],[-3,84],[0,6],[1,5],[0,5],[-1,10],[0,5],[-2,96],[0,20],[0,16],[1,22],[0,15],[1,14],[1,69],[0,10],[1,8],[0,3],[0,6],[0,2],[0,3],[0,8],[1,3],[0,1],[0,1],[1,1],[0,3],[-1,2],[0,5],[1,5],[0,2],[0,18],[1,10],[3,73],[1,4],[0,5],[1,12],[1,9],[1,14],[2,16],[1,3],[0,1],[0,-3],[-2,-17],[-1,-4],[0,-8],[1,5],[1,9],[0,4],[2,6],[0,3],[0,3],[0,2],[0,2],[1,3],[1,2],[1,3],[2,6],[1,5],[1,4],[0,4],[2,8],[0,6],[0,8],[0,8],[1,6],[0,1],[1,-5],[0,-2],[1,-1],[1,-2],[0,-5],[0,-2],[0,1],[1,8],[0,2],[1,4],[0,2],[0,4],[1,4],[2,16],[1,14],[1,15],[2,15],[0,8],[0,8],[0,22],[-1,15],[-1,10],[0,3],[-2,15],[-1,4],[-1,1],[-1,-3],[2,11],[5,20],[1,-3],[0,-7],[1,-2],[5,-9],[2,0],[2,3],[2,5],[2,10],[1,8],[0,3],[0,2],[0,6],[0,2],[0,6],[2,8],[0,5],[0,1],[0,2],[0,2],[1,0],[1,0],[1,-6],[1,-1],[4,-4],[4,2],[3,8],[1,1],[2,-5],[0,-9],[1,-4],[1,4],[-1,3],[-1,6],[0,5],[0,2],[-1,1],[-1,0],[1,3],[2,7],[0,3],[3,14],[3,13],[3,17],[4,11],[2,8],[1,3],[1,7],[2,7],[3,14],[1,3],[0,1],[0,2],[0,2],[0,1],[1,-1],[0,-2],[0,-3],[1,-2],[0,-2],[0,1],[0,3],[0,2],[-1,2],[0,2],[0,3],[0,5],[5,25],[0,5],[1,3],[1,2],[1,2],[1,1],[0,1],[0,2],[0,1],[0,1],[0,3],[1,4],[0,2],[1,7],[1,4],[0,1],[0,2],[0,4],[0,3],[1,1],[0,1],[1,1],[0,3],[1,5],[0,5],[1,5],[0,8],[0,3],[-1,3],[0,5],[0,2],[0,5],[2,9],[0,4],[0,3],[2,11],[8,38],[1,4],[0,3],[0,2],[1,1],[1,1],[1,2],[1,3],[3,23],[0,5],[0,6],[0,6],[1,4],[0,3],[1,2],[1,1],[1,1],[1,1],[1,6],[4,2],[1,3],[1,4],[1,9],[1,9],[1,2],[0,2],[1,2],[0,1],[0,3],[0,1],[1,2],[1,1],[3,17],[2,10],[2,18],[2,9],[1,12],[2,9],[1,1],[0,1],[1,1],[1,2],[2,9],[1,1],[1,4],[3,14],[1,9],[1,8],[0,6],[-1,2],[0,2],[-1,-3],[0,1],[0,6],[1,5],[0,6],[0,6],[0,5],[2,3],[4,14],[1,4],[0,5],[1,4],[4,3],[0,1],[1,1],[1,5],[1,4],[0,2],[1,4],[1,1],[1,0],[1,-3],[0,-1],[1,2],[1,2],[1,4],[1,1],[0,2],[2,8],[1,6],[1,5],[0,3],[0,2],[1,2],[1,2],[1,3],[1,-5],[2,-1],[0,3],[0,2],[0,1],[2,1],[0,2],[0,3],[1,1],[1,-3],[0,2],[2,0],[0,-1],[0,-5],[0,-7],[-1,-6],[1,-1],[0,1],[0,6],[1,5],[1,6],[0,5],[0,1],[1,3],[1,1],[0,2],[2,3],[1,2],[1,1],[1,0],[1,-3],[1,3],[1,3],[0,5],[0,4],[0,3],[-1,2],[-1,4],[-2,3],[-2,6],[-1,1],[-1,2],[2,7],[0,4],[2,3],[0,5],[0,6],[0,4],[1,3],[0,3],[3,10],[4,10],[0,1],[1,1],[0,4],[0,4],[0,5],[0,3],[2,7],[0,4],[0,7],[-1,11],[-1,4],[-2,4],[-1,1],[0,-2],[-1,-2],[0,-4],[-1,-2],[0,4],[0,3],[0,9],[1,10],[1,9],[0,6],[0,4],[1,7],[0,5],[1,4],[1,1],[1,-2],[1,2],[1,4],[0,10],[0,5],[0,10],[0,6],[0,3],[0,3],[0,3],[0,1],[0,2],[1,3],[0,6],[0,8],[0,4],[0,6],[0,3],[0,4],[0,1],[0,2],[1,4],[-1,4],[0,3],[0,3],[0,3],[0,3],[0,3],[0,2],[1,5],[0,4],[0,2],[0,3],[0,7],[0,3],[0,5],[0,3],[0,3],[0,3],[1,1],[0,3],[1,4],[0,4],[1,4],[2,1],[1,2],[2,3],[1,7],[0,6],[2,2],[0,4],[1,5],[1,0],[1,3],[3,-1],[4,-6],[1,-4],[1,-2],[1,-2],[2,0],[1,0],[2,-6],[2,1],[0,-1],[1,-10],[0,-3],[1,-1],[0,1],[0,3],[0,4],[-1,3],[-1,7],[-2,3],[-2,6],[1,5],[1,2],[3,1],[3,1],[2,-4],[4,-2],[1,0],[0,4],[-5,3],[-3,4],[-7,-1],[-3,8],[-2,5],[-2,4],[-2,1],[-2,8],[-1,10],[0,3],[-1,1],[0,3],[-1,4],[-2,0],[-2,4],[-2,0],[-2,-7],[-2,-1],[-3,-2],[-2,1],[-1,-5],[-1,-3],[-1,-6],[-1,-3],[-1,5],[-3,-1],[-1,-2],[-2,-2],[-1,1],[-1,-3],[-2,-5],[-1,-3],[-1,6],[-1,2],[0,-7],[-2,-2],[-2,-5],[0,-3],[0,-6],[0,-4],[0,-5],[-1,-1],[-1,0],[-1,-3],[1,-3],[1,1],[0,-1],[0,-3],[0,-5],[2,-3],[1,-2],[1,-4],[0,-2],[0,-3],[-1,-1],[-1,0],[-1,-1],[-1,-2],[0,-2],[0,-3],[1,-2],[-1,-3],[-1,1],[0,-1],[-1,-5],[-2,-6],[0,-3],[0,-3],[1,-8],[0,-4],[0,-2],[1,-3],[0,-3],[0,-2],[0,-2],[-1,-2],[0,-1],[0,-3],[0,-3],[2,0],[0,-4],[-1,-2],[0,-2],[0,-4],[0,-3],[-1,-2],[0,-3],[-1,-3],[0,-2],[0,-3],[0,-5],[-1,-1],[0,-2],[1,-1],[-1,-5],[0,-6],[0,-4],[-1,-3],[0,-4],[-1,-5],[0,-2],[1,-3],[1,-3],[2,-5],[1,-3],[2,-2],[2,-3],[1,-3],[0,-1],[0,-3],[0,-5],[0,-4],[-1,-2],[-1,-1],[-1,-1],[0,-1],[0,-2],[0,-1],[0,-3],[-1,3],[-1,1],[0,-1],[-1,-2],[-1,-5],[0,-3],[0,-1],[1,-2],[0,-4],[0,-2],[-2,-8],[-1,0],[-1,5],[0,9],[0,1],[-1,1],[-1,0],[0,2],[0,2],[0,3],[-1,0],[0,-2],[0,4],[0,2],[1,2],[1,1],[0,3],[0,5],[0,1],[-2,5],[-1,3],[-1,1],[-1,-1],[-2,-5],[-2,-7],[1,-5],[1,-4],[2,1],[0,-3],[1,-5],[-3,3],[-2,5],[-2,7],[-1,1]]],"transform":{"scale":[0.036003600360036005,0.004067009647564748],"translate":[-180,41.19268056200018]}};
  Datamap.prototype.rwaTopo = '__RWA__';
  Datamap.prototype.sahTopo = '__SAH__';
  Datamap.prototype.sauTopo = '__SAU__';
  Datamap.prototype.scrTopo = '__SCR__';
  Datamap.prototype.sdnTopo = '__SDN__';
  Datamap.prototype.sdsTopo = '__SDS__';
  Datamap.prototype.senTopo = '__SEN__';
  Datamap.prototype.serTopo = '__SER__';
  Datamap.prototype.sgpTopo = '__SGP__';
  Datamap.prototype.sgsTopo = '__SGS__';
  Datamap.prototype.shnTopo = '__SHN__';
  Datamap.prototype.slbTopo = '__SLB__';
  Datamap.prototype.sleTopo = '__SLE__';
  Datamap.prototype.slvTopo = '__SLV__';
  Datamap.prototype.smrTopo = '__SMR__';
  Datamap.prototype.solTopo = '__SOL__';
  Datamap.prototype.somTopo = '__SOM__';
  Datamap.prototype.spmTopo = '__SPM__';
  Datamap.prototype.srbTopo = '__SRB__';
  Datamap.prototype.stpTopo = '__STP__';
  Datamap.prototype.surTopo = '__SUR__';
  Datamap.prototype.svkTopo = '__SVK__';
  Datamap.prototype.svnTopo = '__SVN__';
  Datamap.prototype.sweTopo = '__SWE__';
  Datamap.prototype.swzTopo = '__SWZ__';
  Datamap.prototype.sxmTopo = '__SXM__';
  Datamap.prototype.sycTopo = '__SYC__';
  Datamap.prototype.syrTopo = '__SYR__';
  Datamap.prototype.tcaTopo = '__TCA__';
  Datamap.prototype.tcdTopo = '__TCD__';
  Datamap.prototype.tgoTopo = '__TGO__';
  Datamap.prototype.thaTopo = '__THA__';
  Datamap.prototype.tjkTopo = '__TJK__';
  Datamap.prototype.tkmTopo = '__TKM__';
  Datamap.prototype.tlsTopo = '__TLS__';
  Datamap.prototype.tonTopo = '__TON__';
  Datamap.prototype.ttoTopo = '__TTO__';
  Datamap.prototype.tunTopo = '__TUN__';
  Datamap.prototype.turTopo = '__TUR__';
  Datamap.prototype.tuvTopo = '__TUV__';
  Datamap.prototype.twnTopo = '__TWN__';
  Datamap.prototype.tzaTopo = '__TZA__';
  Datamap.prototype.ugaTopo = '__UGA__';
  Datamap.prototype.ukrTopo = '__UKR__';
  Datamap.prototype.umiTopo = '__UMI__';
  Datamap.prototype.uryTopo = '__URY__';
  Datamap.prototype.usaTopo = '__USA__';
  Datamap.prototype.usgTopo = '__USG__';
  Datamap.prototype.uzbTopo = '__UZB__';
  Datamap.prototype.vatTopo = '__VAT__';
  Datamap.prototype.vctTopo = '__VCT__';
  Datamap.prototype.venTopo = '__VEN__';
  Datamap.prototype.vgbTopo = '__VGB__';
  Datamap.prototype.virTopo = '__VIR__';
  Datamap.prototype.vnmTopo = '__VNM__';
  Datamap.prototype.vutTopo = '__VUT__';
  Datamap.prototype.wlfTopo = '__WLF__';
  Datamap.prototype.wsbTopo = '__WSB__';
  Datamap.prototype.wsmTopo = '__WSM__';
  Datamap.prototype.yemTopo = '__YEM__';
  Datamap.prototype.zafTopo = '__ZAF__';
  Datamap.prototype.zmbTopo = '__ZMB__';
  Datamap.prototype.zweTopo = '__ZWE__';

  /**************************************
                Utilities
  ***************************************/

  //convert lat/lng coords to X / Y coords
  Datamap.prototype.latLngToXY = function(lat, lng) {
     return this.projection([lng, lat]);
  };

  //add <g> layer to root SVG
  Datamap.prototype.addLayer = function( className, id, first ) {
    var layer;
    if ( first ) {
      layer = this.svg.insert('g', ':first-child')
    }
    else {
      layer = this.svg.append('g')
    }
    return layer.attr('id', id || '')
      .attr('class', className || '');
  };

  Datamap.prototype.updateChoropleth = function(data) {
    var svg = this.svg;
    for ( var subunit in data ) {
      if ( data.hasOwnProperty(subunit) ) {
        var color;
        var subunitData = data[subunit]
        if ( ! subunit ) {
          continue;
        }
        else if ( typeof subunitData === "string" ) {
          color = subunitData;
        }
        else if ( typeof subunitData.color === "string" ) {
          color = subunitData.color;
        }
        else {
          color = this.options.fills[ subunitData.fillKey ];
        }
        //if it's an object, overriding the previous data
        if ( subunitData === Object(subunitData) ) {
          this.options.data[subunit] = defaults(subunitData, this.options.data[subunit] || {});
          var geo = this.svg.select('.' + subunit).attr('data-info', JSON.stringify(this.options.data[subunit]));
        }
        svg
          .selectAll('.' + subunit)
          .transition()
            .style('fill', color);
      }
    }
  };

  Datamap.prototype.updatePopup = function (element, d, options) {
    var self = this;
    element.on('mousemove', null);
    element.on('mousemove', function() {
      var position = d3.mouse(self.options.element);
      d3.select(self.svg[0][0].parentNode).select('.datamaps-hoverover')
        .style('top', ( (position[1] + 30)) + "px")
        .html(function() {
          var data = JSON.parse(element.attr('data-info'));
          try {
            return options.popupTemplate(d, data);
          } catch (e) {
            return "";
          }
        })
        .style('left', ( position[0]) + "px");
    });

    d3.select(self.svg[0][0].parentNode).select('.datamaps-hoverover').style('display', 'block');
  };

  Datamap.prototype.addPlugin = function( name, pluginFn ) {
    var self = this;
    if ( typeof Datamap.prototype[name] === "undefined" ) {
      Datamap.prototype[name] = function(data, options, callback, createNewLayer) {
        var layer;
        if ( typeof createNewLayer === "undefined" ) {
          createNewLayer = false;
        }

        if ( typeof options === 'function' ) {
          callback = options;
          options = undefined;
        }

        options = defaults(options || {}, self.options[name + 'Config']);

        //add a single layer, reuse the old layer
        if ( !createNewLayer && this.options[name + 'Layer'] ) {
          layer = this.options[name + 'Layer'];
          options = options || this.options[name + 'Options'];
        }
        else {
          layer = this.addLayer(name);
          this.options[name + 'Layer'] = layer;
          this.options[name + 'Options'] = options;
        }
        pluginFn.apply(this, [layer, data, options]);
        if ( callback ) {
          callback(layer);
        }
      };
    }
  };

  // expose library
  if (typeof exports === 'object') {
    d3 = require('d3');
    topojson = require('topojson');
    module.exports = Datamap;
  }
  else if ( typeof define === "function" && define.amd ) {
    define( "datamaps", ["require", "d3", "topojson"], function(require) {
      d3 = require('d3');
      topojson = require('topojson');

      return Datamap;
    });
  }
  else {
    window.Datamap = window.Datamaps = Datamap;
  }

  if ( window.jQuery ) {
    window.jQuery.fn.datamaps = function(options, callback) {
      options = options || {};
      options.element = this[0];
      var datamap = new Datamap(options);
      if ( typeof callback === "function" ) {
        callback(datamap, options);
      }
      return this;
    };
  }
})();
