var count = (function abs(BigNumber) {
    var start = +new Date(),
        log,
        error,
        passed = 0,
        total = 0;

    if (typeof window === 'undefined') {
        log = console.log;
        error = console.error;
    } else {
        log = function (str) { document.body.innerHTML += str.replace('\n', '<br>') };
        error = function (str) { document.body.innerHTML += '<div style="color: red">' +
          str.replace('\n', '<br>') + '</div>' };
    }

    if (!BigNumber && typeof require === 'function') BigNumber = require('../bignumber');

    function assert(expected, actual) {
        total++;
        if (expected !== actual) {
           error('\n Test number: ' + total + ' failed');
           error(' Expected: ' + expected);
           error(' Actual:   ' + actual);
           //process.exit();
        }
        else {
            passed++;
            //log('\n Expected and actual: ' + actual);
        }
    }

    function T(expected, value){
        assert(String(expected), new BigNumber(String(value)).abs().toString());
    }

    log('\n Testing abs...');

    BigNumber.config({
        DECIMAL_PLACES: 20,
        ROUNDING_MODE: 4,
        ERRORS: true,
        RANGE: 1E9,
        EXPONENTIAL_AT: [-7, 21]
    });

    T(1, 1);
    T(1, -1);
    T(0.5, '0.5');
    T(0.5, '-0.5');
    T(0.1, 0.1);
    T(0.1, -0.1);
    T(1.1, 1.1);
    T(1.1, -1.1);
    T(1.5, '1.5');
    T(1.5, '-1.5');

    T(0.00001, '-1e-5');
    T(9000000000, '-9e9');
    T(123456.7891011, -123456.7891011);
    T(999.999, '-999.999');
    T(99, 99);
    T(1, new BigNumber(-1));
    T(0.001, new BigNumber(0.001));
    T(0.001, new BigNumber('-0.001'));

    T('Infinity', Infinity);
    T('Infinity', -Infinity);
    T(NaN, NaN);
    T(NaN, -NaN);
    T(0, 0);
    T(0, -0);

    var minusZero = 1 / (-1 / 0);

    function isMinusZero(n) {
        return n.toString() === '0' && n.s == -1;
    }

    T(0, 0);
    T(0, -0);
    T(0, minusZero);

    assert(true, isMinusZero(new BigNumber('-0')));
    assert(true, isMinusZero(new BigNumber(minusZero)));
    assert(false, isMinusZero(new BigNumber(-0).abs()));
    assert(false, isMinusZero(new BigNumber(minusZero).abs()));
    assert(true, !isMinusZero(new BigNumber('-0').abs()));
    assert(true, !isMinusZero(new BigNumber(minusZero).abs()));

    BigNumber.config({EXPONENTIAL_AT: 100});

    T(Number.MIN_VALUE, Number.MIN_VALUE);
    T(Number.MIN_VALUE, -Number.MIN_VALUE);
    T(Number.MAX_VALUE, Number.MAX_VALUE);
    T(Number.MAX_VALUE, -Number.MAX_VALUE);

    var two_30 = 1 << 30;

    T(two_30, two_30);
    T(two_30, -two_30);

    T(two_30 + 1, two_30 + 1);
    T(two_30 + 1, -two_30 - 1);

    T(two_30 - 1, two_30 - 1);
    T(two_30 - 1, -two_30 + 1);

    var two_31 = 2 * two_30;

    T(two_31, two_31);
    T(two_31, -two_31);

    T(two_31 + 1, two_31 + 1);
    T(two_31 + 1, -two_31 - 1);

    T(two_31 - 1, two_31 - 1);
    T(two_31 - 1, -two_31 + 1);

    BigNumber.config({ EXPONENTIAL_AT: [-7, 21] });

    T(NaN, 'NaN');
    T('0', '0');
    T('1', '-1');
    T('11.121', '11.121');
    T('0.023842', '-0.023842');
    T('1.19', '-1.19');
    T('9.622e-11', '-0.00000000009622');
    T('5.09e-10', '-0.000000000509');
    T('3838.2', '3838.2');
    T('127', '127.0');
    T('4.23073', '4.23073');
    T('2.5469', '-2.5469');
    T('29949', '-29949');
    T('277.1', '-277.10');
    T('4.97898e-15', '-0.00000000000000497898');
    T('53.456', '53.456');
    T('100564', '-100564');
    T('12431.9', '-12431.9');
    T('97633.7', '-97633.7');
    T('220', '220');
    T('188.67', '-188.67');
    T('35', '-35');
    T('2.6', '-2.6');
    T('2.2e-19', '-0.000000000000000000220');
    T('1.469', '-1.469');
    T('150.7', '-150.7');
    T('74', '-74');
    T('3.52e-9', '-0.00000000352');
    T('2221.7', '-2221.7');
    T('0.000004211', '-0.000004211');
    T('1', '-1');
    T('5.886', '-5.886');
    T('16', '16');
    T('4.4493e-9', '0.0000000044493');
    T('47.6', '47.6');
    T('1.6', '-1.60');
    T('1', '-1');
    T('1.5', '-1.5');
    T('5', '-5');
    T('1', '-1');
    T('8027', '8027');
    T('6.36e-16', '-0.000000000000000636');
    T('3.87766', '3.87766');
    T('7.4', '-7.4');
    T('4.449', '-4.449');
    T('5.2218e-19', '-0.000000000000000000522180');
    T('1.3769e-11', '-0.000000000013769');
    T('7.898e-13', '-0.0000000000007898');
    T('522.9', '-522.9');
    T('16.1', '-16.1');
    T('2.15', '2.15');
    T('4.3', '4.3');
    T('3', '-3');
    T('2.8', '-2.8');
    T('1', '-1');
    T('0.0000128696', '-0.0000128696');
    T('13.33', '-13.33');
    T('0.00000132177', '-0.00000132177');
    T('1.41516', '-1.41516');
    T('180.4', '-180.4');
    T('115079', '-115079');
    T('959', '959');
    T('714.4', '714.4');
    T('1.4544', '1.4544');
    T('53.691', '53.691');
    T('2.03832e-12', '-0.00000000000203832');
    T('1', '-1');
    T('10.8', '10.8');
    T('6189.2', '-6189.2');
    T('6.30866', '6.30866');
    T('62306', '62306');
    T('4', '-4.0');
    T('997.1', '-997.1');
    T('27.4', '-27.40');
    T('9242', '9242');
    T('31.1', '-31.1');
    T('23.4', '23.4');
    T('451818', '-451818');
    T('7', '-7');
    T('1.9', '-1.9');
    T('2', '-2');
    T('112.983', '-112.983');
    T('9.36e-8', '-0.0000000936');
    T('12.8515', '12.8515');
    T('73.1', '-73.1');
    T('18.15', '18.150');
    T('11997.8', '11997.8');
    T('23.1', '-23.1');
    T('82.022', '-82.022');
    T('3.916e-20', '-0.00000000000000000003916');
    T('3.3', '-3.3');
    T('892.1', '-892.1');
    T('24.4', '24.4');
    T('72', '72.0');
    T('0.0013346', '0.0013346');
    T('10.4', '-10.4');
    T('367.5', '367.5');
    T('7', '-7');
    T('127.195', '127.195');
    T('7.89e-13', '-0.000000000000789');
    T('63', '-63');
    T('85821.2', '-85821.2');
    T('95.6', '95.6');
    T('8.9e-14', '-0.000000000000089');
    T('112.1', '-112.1');
    T('3.68', '-3.68');
    T('9', '-9');
    T('0.0000975', '-0.0000975');
    T('393.6', '-393.6');
    T('7.4', '-7.4');
    T('69.62', '-69.62');
    T('5201.3', '5201.3');
    T('163', '163');
    T('4.30732', '4.30732');
    T('224.49', '-224.49');
    T('319.8', '-319.8');
    T('88.1', '-88.1');
    T('2.7762e-8', '0.000000027762');
    T('2.043e-7', '-0.0000002043');
    T('75459.3', '-75459.3');
    T('0.178', '0.178');
    T('0.00001633', '0.00001633');
    T('955', '955');
    T('373898', '-373898');
    T('9780.1', '9780.1');
    T('503.47', '503.47');
    T('3.44562', '-3.44562');
    T('1.6', '-1.6');
    T('1.22442', '-1.22442');
    T('1.4', '1.4');
    T('1219.1', '-1219.1');
    T('2.7', '-2.7');
    T('1057', '-1057');
    T('1938', '1938');
    T('1.1983', '1.1983');
    T('0.0012', '-0.0012');
    T('95.713', '-95.713');
    T('2', '-2');
    T('17.24', '-17.24');
    T('10.3', '-10.3');
    T('1', '-1');
    T('65.8', '-65.8');
    T('2.9', '2.9');
    T('54149', '54149');
    T('8', '-8');
    T('1', '1.0');
    T('4', '-4');
    T('6.3', '-6.3');
    T('5.25e-9', '0.00000000525');
    T('52.3', '-52.3');
    T('75290', '-75290');
    T('5.9', '-5.9');
    T('13.7', '13.7');
    T('2.3982e-9', '0.0000000023982');
    T('91.5', '-91.50');
    T('2072.39', '2072.39');
    T('385.6', '385.6');
    T('4.77', '4.77');
    T('18.72', '18.720');
    T('2817', '-2817');
    T('44535', '-44535');
    T('655', '655');
    T('2e-15', '-0.0000000000000020');
    T('0.625', '0.6250');
    T('2', '-2');
    T('5.315', '5.315');
    T('70.9', '70.90');
    T('6.4', '6.4');
    T('1824', '1824');
    T('52.595', '52.595');
    T('3662', '3662.0');
    T('3.1', '3.1');
    T('1.05032e-7', '0.000000105032');
    T('997.063', '-997.063');
    T('41746', '-41746');
    T('24.0402', '24.0402');
    T('0.009135', '0.009135');
    T('2.34e-9', '-0.00000000234');
    T('13.1', '13.1');
    T('228.8', '228.8');
    T('565.85', '565.85');
    T('4e-20', '0.000000000000000000040');
    T('1.73', '1.73');
    T('38.9', '38.9');
    T('1.02e-14', '-0.0000000000000102');
    T('302.8', '-302.8');
    T('7', '-7');
    T('1', '-1');
    T('0.00247', '0.00247');
    T('2', '-2');
    T('3.26', '-3.26');
    T('8.8', '8.8');
    T('90.6', '90.6');
    T('8.3053e-17', '-0.000000000000000083053');
    T('2.5', '-2.5');
    T('376.2', '-376.2');
    T('1.29', '1.29');
    T('1.379', '-1.379');
    T('40921.5', '-40921.5');
    T('1', '-1');
    T('12.5', '12.5');
    T('10.1', '10.1');
    T('1', '-1');
    T('226636', '226636');
    T('1', '-1');
    T('1.7', '-1.7');
    T('31.31', '31.31');
    T('79.9', '-79.9');
    T('4.027e-13', '0.0000000000004027');
    T('43.838', '43.838');
    T('6.47', '-6.47');
    T('5.292e-19', '0.0000000000000000005292');
    T('4.6', '-4.6');
    T('15918', '-15918.0');
    T('239.45', '239.45');
    T('1.02', '-1.02');
    T('14101', '-14101');
    T('7', '-7');
    T('367.34', '367.34');
    T('5', '-5');
    T('19.9', '-19.9');
    T('269.45', '-269.45');
    T('10.34', '-10.34');
    T('3.32882e-12', '-0.00000000000332882');
    T('5.9', '5.9');
    T('9', '-9.0');
    T('1.3597', '-1.3597');
    T('8', '8.0');
    T('1', '1.0');
    T('312.5', '312.5');
    T('1.554', '-1.554');
    T('210.985', '-210.985');
    T('1', '-1');
    T('1.24', '-1.24');
    T('513865', '-513865');
    T('6748', '-6748');
    T('591.51', '-591.51');
    T('2.2', '-2.2');
    T('19.5495', '19.5495');
    T('3.3', '3.3');
    T('30', '-30');
    T('94', '-94');
    T('217.55', '217.55');
    T('2', '-2');
    T('99', '99');
    T('4.067', '-4.067');
    T('702.57', '702.57');
    T('3.7', '-3.70');
    T('4', '4.0');
    T('192944', '192944');
    T('0.000022', '0.000022');
    T('47.6', '47.60');
    T('0.391', '0.3910');
    T('35', '-35');
    T('100', '-100');
    T('3.3', '-3.3');
    T('32.432', '32.432');
    T('1.07849e-18', '0.00000000000000000107849');
    T('2', '-2.0');
    T('23.27', '23.27');
    T('4.054e-15', '-0.000000000000004054');
    T('7.6', '-7.6');
    T('1305', '1305');
    T('1.501', '-1.501');
    T('3.4', '3.4');
    T('22.5', '-22.5');
    T('1.0916', '1.0916');
    T('2', '-2');
    T('58.271', '58.271');
    T('1.73e-12', '0.00000000000173');
    T('1.3458e-15', '0.0000000000000013458');
    T('309.87', '-309.87');
    T('5.318', '-5.318');
    T('1.5302e-8', '0.000000015302');
    T('596765', '596765');
    T('54.42', '-54.42');
    T('6.549e-20', '0.00000000000000000006549');
    T('29', '29');
    T('46.025', '46.025');
    T('2556.78', '-2556.78');
    T('0.00287721', '0.00287721');
    T('1.63', '-1.63');
    T('0.00041', '0.00041');
    T('698', '698');
    T('134.4', '134.4');
    T('2.1', '2.1');
    T('2.07', '-2.07');
    T('122.869', '122.869');
    T('0.00017', '-0.00017');
    T('18.6', '18.6');
    T('7', '-7');
    T('0.0180557', '0.0180557');
    T('5', '-5');
    T('6.2', '-6.2');
    T('8', '-8');
    T('450.96', '-450.96');
    T('20.2', '-20.2');
    T('176.52', '176.52');
    T('0.00017', '-0.000170');
    T('5', '-5');
    T('1', '-1');
    T('1.37856e-14', '0.0000000000000137856');
    T('76.3048', '76.3048');
    T('1803.7', '-1803.7');
    T('74', '74');
    T('1.7e-12', '0.0000000000017');
    T('48.7', '-48.7');
    T('4.48', '-4.48');
    T('1.4', '-1.4');
    T('7.69', '-7.69');
    T('23.5987', '23.5987');
    T('3074', '3074.0');
    T('8.06e-15', '-0.00000000000000806');
    T('21.3757', '-21.3757');
    T('35', '35');
    T('11.056', '11.0560');
    T('3.36e-14', '-0.0000000000000336');
    T('49139.4', '-49139.4');
    T('32.654', '-32.654');
    T('34035.4', '34035.4');
    T('15.22', '15.22');
    T('62', '62.0');
    T('8.89156', '-8.89156');
    T('14', '14');
    T('0.006', '-0.0060');
    T('1.5', '1.5');
    T('7', '-7');
    T('1.6e-11', '0.000000000016');
    T('26.6427', '26.6427');
    T('1.5e-18', '-0.0000000000000000015');
    T('1.52838e-15', '0.00000000000000152838');
    T('119.1', '119.1');
    T('0.004283', '0.004283');
    T('818', '-818');
    T('194', '194');
    T('104.788', '-104.788');
    T('3.74e-11', '0.0000000000374');
    T('6.162', '-6.162');
    T('5.19214e-18', '-0.00000000000000000519214');
    T('1.4', '-1.4');
    T('1.27', '-1.27');
    T('7.83822e-12', '-0.00000000000783822');
    T('1', '-1');
    T('4.4', '4.4');
    T('7.37382e-12', '0.00000000000737382');
    T('13.618', '13.618');
    T('1.03', '-1.03');
    T('3.7457e-13', '0.00000000000037457');
    T('5.2', '-5.2');
    T('3.5', '3.5');
    T('364', '-364');
    T('7.336', '7.336');
    T('1.1447e-16', '-0.00000000000000011447');
    T('510.63', '-510.63');
    T('5.8', '5.8');
    T('7.8', '7.8');
    T('2.96', '-2.96');
    T('15.64', '-15.64');
    T('187863', '-187863');
    T('2.73', '-2.73');
    T('2.671', '-2.671');
    T('18.179', '-18.179');
    T('855885', '855885');
    T('4.16', '4.16');
    T('5.722e-18', '0.000000000000000005722');
    T('67.62', '67.62');
    T('813.31', '813.31');
    T('40.2', '40.20');
    T('0.00002515', '0.00002515');
    T('0.0196', '0.01960');
    T('13.165', '13.165');
    T('6.743', '-6.743');
    T('1', '-1');
    T('200.56', '-200.56');
    T('1.932', '1.932');
    T('92.9', '92.90');
    T('16.74', '16.74');
    T('4.5554e-7', '-0.00000045554');
    T('2.1296e-15', '-0.0000000000000021296');
    T('2.088', '2.088');
    T('2577', '2577');
    T('45.4', '-45.4');
    T('41.3', '-41.3');
    T('3.63', '-3.63');
    T('1.09', '-1.09');
    T('1', '-1');
    T('3.7', '-3.7');
    T('204.54', '204.54');
    T('235.6', '235.6');
    T('384', '-384');
    T('0.0207', '0.02070');
    T('680', '680');
    T('1.09', '1.09');
    T('109.2', '109.2');
    T('0.00010117', '0.00010117');
    T('13.81', '13.81');
    T('192.3', '192.3');
    T('1', '-1');
    T('1.2', '1.2');
    T('4.1', '-4.1');
    T('2.5', '2.5');
    T('8.4076', '-8.4076');
    T('0.0517', '0.0517');
    T('6.3923', '-6.3923');
    T('506.179', '-506.179');
    T('375886', '375886');
    T('618858', '-618858');
    T('8.5e-11', '0.000000000085');
    T('6', '-6.0');
    T('2.4', '2.40');
    T('0.0000013', '-0.0000013');
    T('1.064', '-1.064');
    T('1', '-1');
    T('4', '-4');
    T('4.5', '-4.5');
    T('93.6206', '93.6206');
    T('3.07e-18', '0.00000000000000000307');

    BigNumber.config({EXPONENTIAL_AT: 0});

    T('5.2452468128e+1', '-5.2452468128e+1');
    T('1.41525905257189365008396e+16', '1.41525905257189365008396e+16');
    T('2.743068083928e+11', '2.743068083928e+11');
    T('1.52993064722314247378724599e+26', '-1.52993064722314247378724599e+26');
    T('3.7205576746e+10', '3.7205576746e+10');
    T('8.680996444609343472665e+17', '8.680996444609343472665e+17');
    T('1.254549e+3', '1.254549e+3');
    T('6.23417196172381875892300762819e-18', '6.23417196172381875892300762819e-18');
    T('1.31179940821919284431e+19', '1.31179940821919284431e+19');
    T('9.7697726168e+7', '9.7697726168e+7');
    T('2.663e-10', '-2.663e-10');
    T('1.26574209965030360615518e+17', '-1.26574209965030360615518e+17');
    T('1.052e+3', '1.052e+3');
    T('4.452945872502e+6', '-4.452945872502e+6');
    T('2.95732460816619226e+13', '2.95732460816619226e+13');
    T('1.1923100194288654481424e+18', '-1.1923100194288654481424e+18');
    T('8.99315449050893705e+6', '8.99315449050893705e+6');
    T('5.200726538434486963e+8', '5.200726538434486963e+8');
    T('1.182618278949368566264898065e+18', '1.182618278949368566264898065e+18');
    T('3.815873266712e-20', '-3.815873266712e-20');
    T('1.316675370382742615e+6', '-1.316675370382742615e+6');
    T('2.1032502e+6', '-2.1032502e+6');
    T('1.8e+1', '1.8e+1');
    T('1.033525906631680944018544811261e-13', '1.033525906631680944018544811261e-13');
    T('1.102361746443461856816e+14', '-1.102361746443461856816e+14');
    T('8.595358491143959e+1', '8.595358491143959e+1');
    T('3.6908859412618413e+9', '-3.6908859412618413e+9');
    T('2.25907048615912944e+5', '-2.25907048615912944e+5');
    T('1.7441871813329475518e+19', '-1.7441871813329475518e+19');
    T('3.805493087068952925e-11', '-3.805493087068952925e-11');
    T('3.58049465451e+9', '-3.58049465451e+9');
    T('8.0688614291e+10', '-8.0688614291e+10');
    T('3.337855e+4', '-3.337855e+4');
    T('2.59977855e+8', '2.59977855e+8');
    T('4.96353e+4', '-4.96353e+4');
    T('7.47233581107861762e-13', '7.47233581107861762e-13');
    T('1.73948e-2', '1.73948e-2');
    T('5.784e-15', '5.784e-15');
    T('4.448338479762497e-8', '4.448338479762497e-8');
    T('3.9008023052e+8', '3.9008023052e+8');
    T('3e+0', '3e+0');
    T('8.61435e-9', '8.61435e-9');
    T('4.37e+1', '-4.37e+1');
    T('8.4034159379836e-18', '-8.4034159379836e-18');
    T('2.002857355721079885824481e+7', '2.002857355721079885824481e+7');
    T('7.000871862e+6', '-7.000871862e+6');
    T('2.2902057767e+9', '2.2902057767e+9');
    T('5.9896443375617e+8', '5.9896443375617e+8');
    T('1.53503650707e-11', '-1.53503650707e-11');
    T('2.0508347e+6', '2.0508347e+6');
    T('4.789433e+2', '-4.789433e+2');
    T('8.28161975302168665599e+11', '8.28161975302168665599e+11');
    T('1.2518396296278445e-5', '1.2518396296278445e-5');
    T('1.44290332e+8', '-1.44290332e+8');
    T('4.6570237501625609051773e-12', '4.6570237501625609051773e-12');
    T('7.8514960198282212436e+19', '7.8514960198282212436e+19');
    T('1.6197e-20', '1.6197e-20');
    T('6.51635176e+0', '-6.51635176e+0');
    T('4.49618e+3', '-4.49618e+3');
    T('1.32052259561417e-1', '-1.32052259561417e-1');
    T('2.09089580968e-18', '2.09089580968e-18');
    T('1.4064735615678257623873854709e-1', '1.4064735615678257623873854709e-1');
    T('3.14172e+0', '-3.14172e+0');
    T('1.7458792e+1', '1.7458792e+1');
    T('9.97831655282e+11', '9.97831655282e+11');
    T('1.94594e+1', '-1.94594e+1');
    T('1.2174602334491e+5', '-1.2174602334491e+5');
    T('1.12135222651239e+6', '-1.12135222651239e+6');
    T('6.3160490484343918e-20', '6.3160490484343918e-20');
    T('1.9238315686509393329629520842e+24', '1.9238315686509393329629520842e+24');
    T('9.915274405618026e+11', '-9.915274405618026e+11');
    T('2.3564687894712721487205001557e+28', '2.3564687894712721487205001557e+28');
    T('8.127315365677288172165e+2', '8.127315365677288172165e+2');
    T('4.93e+0', '-4.93e+0');
    T('1.41530382e+0', '-1.41530382e+0');
    T('4.86451432707435321820779e+19', '-4.86451432707435321820779e+19');
    T('1.4162540859e+0', '-1.4162540859e+0');
    T('4.646e+2', '-4.646e+2');
    T('2.1172e-14', '-2.1172e-14');
    T('8.69000536011392432707132752e-11', '8.69000536011392432707132752e-11');
    T('2.52776394053478133209e+20', '2.52776394053478133209e+20');
    T('8.500211152e+9', '8.500211152e+9');
    T('1.36178922026634255436879e+23', '1.36178922026634255436879e+23');
    T('4.6398705910903109e+3', '-4.6398705910903109e+3');
    T('2.15872185740218265392874524e+18', '2.15872185740218265392874524e+18');
    T('2.4663508855569609277266393e-3', '-2.4663508855569609277266393e-3');
    T('5.247072789229625795e+11', '-5.247072789229625795e+11');
    T('1.142743622516581e-15', '-1.142743622516581e-15');
    T('3.70055552960951165e-4', '-3.70055552960951165e-4');
    T('1.01218e+3', '1.01218e+3');
    T('3.622286100282e+2', '3.622286100282e+2');
    T('9.5526239814e+3', '9.5526239814e+3');
    T('2.7619598176203983624994361644e+28', '2.7619598176203983624994361644e+28');
    T('6.8696488497688008067537526e-6', '6.8696488497688008067537526e-6');
    T('2.48936e+1', '2.48936e+1');
    T('3.27658301230616e+14', '3.27658301230616e+14');
    T('2.1887387e+0', '-2.1887387e+0');
    T('1.4779696309033248e+16', '1.4779696309033248e+16');
    T('1.471782313713309789663e+4', '1.471782313713309789663e+4');
    T('2.0674554e+2', '-2.0674554e+2');
    T('1.763392540310312024e+9', '1.763392540310312024e+9');
    T('2.66209467493293140387227569744e+26', '-2.66209467493293140387227569744e+26');
    T('1.4522423854706487171671160683e-16', '1.4522423854706487171671160683e-16');
    T('5.5534571375626084341933639e-18', '-5.5534571375626084341933639e-18');
    T('3.670610508911e-18', '-3.670610508911e-18');
    T('1.8e+1', '1.8e+1');
    T('4.21466540619392e+14', '-4.21466540619392e+14');
    T('4.57881788773078611890575215e-13', '-4.57881788773078611890575215e-13');
    T('1.14912007700989046355e+20', '1.14912007700989046355e+20');
    T('1.10572e+0', '1.10572e+0');
    T('5.45027073427600086838788178e+8', '5.45027073427600086838788178e+8');
    T('5.3607527344097728e-14', '-5.3607527344097728e-14');
    T('1.20985e+0', '1.20985e+0');
    T('2.173758396975e+4', '-2.173758396975e+4');
    T('1.443459545123362e+10', '1.443459545123362e+10');
    T('8.26154936079048787963e-19', '8.26154936079048787963e-19');
    T('1.24e+0', '-1.24e+0');
    T('6.61e+1', '6.61e+1');
    T('8.37241281e-15', '-8.37241281e-15');
    T('1.4673863119972e+5', '1.4673863119972e+5');
    T('1.052445707646628e+15', '1.052445707646628e+15');
    T('2.770216401480935105227985046e+0', '2.770216401480935105227985046e+0');
    T('1e-2', '-1e-2');
    T('2.0530189404000503380382112e+7', '-2.0530189404000503380382112e+7');
    T('7.73428930734513129e+5', '7.73428930734513129e+5');
    T('2.969e-2', '2.969e-2');
    T('3.355869237729311e-19', '3.355869237729311e-19');
    T('7.585426017526e+3', '7.585426017526e+3');
    T('1.6544419963706446557685646278e+23', '-1.6544419963706446557685646278e+23');
    T('2.92136474375552641396809118574e-18', '2.92136474375552641396809118574e-18');
    T('3.38424409165604660854e+4', '-3.38424409165604660854e+4');
    T('1.173591570196350093112e+11', '-1.173591570196350093112e+11');
    T('7.8375092064291352e+1', '-7.8375092064291352e+1');
    T('1.88191e+3', '1.88191e+3');
    T('4.6761e-2', '-4.6761e-2');
    T('5.988129995539574e+10', '5.988129995539574e+10');
    T('2.5390529009345115e+2', '2.5390529009345115e+2');
    T('2.132229656150917182e+5', '-2.132229656150917182e+5');
    T('1.0719725506854825717e-19', '-1.0719725506854825717e-19');
    T('4.3681500769125575941008112847e+28', '-4.3681500769125575941008112847e+28');
    T('1.35927075893264893848008382e-13', '-1.35927075893264893848008382e-13');
    T('1.9240692976139e-18', '-1.9240692976139e-18');
    T('4.49668506275546883445e+20', '4.49668506275546883445e+20');
    T('5.19198662387790072e+9', '5.19198662387790072e+9');
    T('1.51188431866457089e+16', '-1.51188431866457089e+16');
    T('1.4463331863500941e+12', '1.4463331863500941e+12');
    T('1e+0', '-1e+0');
    T('2.50029927958615945e+1', '-2.50029927958615945e+1');
    T('1.001415164502846757e+3', '-1.001415164502846757e+3');
    T('1.45526428e+8', '-1.45526428e+8');
    T('5.813181844e-3', '-5.813181844e-3');
    T('2.4481022856740302965057941113e+10', '2.4481022856740302965057941113e+10');
    T('5.55e+1', '-5.55e+1');
    T('3.36356932710712e+11', '-3.36356932710712e+11');
    T('5.28080163e+8', '5.28080163e+8');
    T('5.3879740593083469994135e+13', '-5.3879740593083469994135e+13');
    T('6.6759148438881472902e+19', '-6.6759148438881472902e+19');
    T('1.26e-20', '1.26e-20');
    T('1.005680289388988e+10', '-1.005680289388988e+10');
    T('1.4855958598e+0', '-1.4855958598e+0');
    T('2.94014963598446075495453768e+24', '-2.94014963598446075495453768e+24');
    T('5.219896118644e+12', '-5.219896118644e+12');
    T('6.8e+0', '-6.8e+0');
    T('5.492e-9', '-5.492e-9');
    T('1.0038e+4', '-1.0038e+4');
    T('2.781382585e+5', '2.781382585e+5');
    T('3.30150670653876784e+17', '-3.30150670653876784e+17');
    T('1.87927e+5', '-1.87927e+5');
    T('1.4774557974305197453804758396e+16', '-1.4774557974305197453804758396e+16');
    T('6.05644990832733182152086098e+18', '-6.05644990832733182152086098e+18');
    T('2.78459055955765755e-14', '-2.78459055955765755e-14');
    T('2.66385931106395122e+6', '2.66385931106395122e+6');
    T('3.3683073647556597682246e-9', '-3.3683073647556597682246e-9');
    T('7.081e+2', '7.081e+2');
    T('2.73122035866217320954404e+6', '2.73122035866217320954404e+6');
    T('1.2434001e-7', '1.2434001e-7');
    T('1.135877627944001e+14', '1.135877627944001e+14');
    T('5.59534951548380080886141393126e+21', '5.59534951548380080886141393126e+21');
    T('5.7723782191795798882571e+9', '-5.7723782191795798882571e+9');
    T('1.5162957113185485632499369443e-12', '-1.5162957113185485632499369443e-12');
    T('4.29309951955288963780116e+6', '4.29309951955288963780116e+6');
    T('3.9722643229317825409e+13', '3.9722643229317825409e+13');
    T('1.011489199242414759e-17', '1.011489199242414759e-17');
    T('1.253643670639200989056241e-19', '-1.253643670639200989056241e-19');
    T('4.4836025129185e+8', '4.4836025129185e+8');
    T('6.3777231879677253018091496e-20', '6.3777231879677253018091496e-20');
    T('4.76278478201471177044e+11', '4.76278478201471177044e+11');
    T('1.05e+2', '-1.05e+2');
    T('8.2407974521826916377252018422e+18', '8.2407974521826916377252018422e+18');
    T('2.00932156087e+4', '2.00932156087e+4');
    T('1.965992456941204354956867603e-17', '-1.965992456941204354956867603e-17');
    T('5.333218599567659131313e+2', '-5.333218599567659131313e+2');
    T('1.286162439284e+10', '-1.286162439284e+10');
    T('8.1336617205815143346477183e+16', '-8.1336617205815143346477183e+16');
    T('1.762845949430042e+13', '-1.762845949430042e+13');
    T('7.837280986421e+12', '7.837280986421e+12');
    T('2.84048190010833793e+13', '2.84048190010833793e+13');
    T('3.25755301782427035301e+20', '-3.25755301782427035301e+20');
    T('2.58959421885729898387238225e+13', '2.58959421885729898387238225e+13');
    T('1.8851093513683294449e+10', '-1.8851093513683294449e+10');
    T('1.21916240456196024666e+20', '-1.21916240456196024666e+20');
    T('5.840503333749926899855535241e-6', '5.840503333749926899855535241e-6');
    T('2.998914116e+4', '2.998914116e+4');
    T('5.97277308650934e+10', '5.97277308650934e+10');
    T('6.56e+2', '6.56e+2');
    T('1.56235984592541e+12', '-1.56235984592541e+12');
    T('3.71e+1', '3.71e+1');
    T('5.41937441824138694e+16', '-5.41937441824138694e+16');
    T('6.116633e-5', '-6.116633e-5');
    T('5.45e+2', '-5.45e+2');
    T('2.9449785444e+3', '-2.9449785444e+3');
    T('6.6706550091070638245894e+7', '-6.6706550091070638245894e+7');
    T('1.39231027e-9', '1.39231027e-9');
    T('7.45311483e+8', '7.45311483e+8');
    T('7.6856950378651228179663e+18', '7.6856950378651228179663e+18');
    T('3.094636736003620629e+8', '-3.094636736003620629e+8');
    T('5.876896131624540495694931644e+7', '-5.876896131624540495694931644e+7');
    T('1.10975974e+8', '-1.10975974e+8');
    T('1.741e+0', '1.741e+0');
    T('2.351595813466272408066e-4', '-2.351595813466272408066e-4');
    T('1.519156959043394168562e+20', '1.519156959043394168562e+20');
    T('1.620081571051799e+7', '1.620081571051799e+7');
    T('7.316815038867932520586761e+23', '7.316815038867932520586761e+23');
    T('3.094134522833396822e+0', '3.094134522833396822e+0');
    T('1.168234556e+2', '-1.168234556e+2');
    T('1.503324779432e+4', '1.503324779432e+4');
    T('5.6710777e-9', '5.6710777e-9');
    T('2.1463873346182e-6', '2.1463873346182e-6');
    T('1.2934324795526700185311026007e+28', '-1.2934324795526700185311026007e+28');
    T('1.237009087265757433674283664e+11', '1.237009087265757433674283664e+11');
    T('1.226806049797304683867e-18', '1.226806049797304683867e-18');
    T('5e+0', '-5e+0');
    T('1.091168788407093537887970016e+15', '-1.091168788407093537887970016e+15');
    T('3.87166413612272027e+12', '3.87166413612272027e+12');
    T('1.411514e+5', '1.411514e+5');
    T('1.0053454672509859631996e+22', '1.0053454672509859631996e+22');
    T('6.9265714e+0', '6.9265714e+0');
    T('1.04627709e+4', '1.04627709e+4');
    T('1.74378341199e+9', '1.74378341199e+9');
    T('8.427721739784805398864e+21', '-8.427721739784805398864e+21');
    T('3.0433401636913618083715e-20', '3.0433401636913618083715e-20');
    T('8.596751182989204e-17', '8.596751182989204e-17');
    T('2.83012114501087201358049280895e-3', '2.83012114501087201358049280895e-3');
    T('6.0621417107465763e-13', '6.0621417107465763e-13');
    T('7.927e+0', '7.927e+0');
    T('1.95309091153617e+6', '-1.95309091153617e+6');
    T('3.479245772e-4', '3.479245772e-4');
    T('9.1256366370332e-20', '-9.1256366370332e-20');
    T('6.357737394e-19', '-6.357737394e-19');
    T('4.016038725869e-1', '4.016038725869e-1');
    T('2.3600611340992838105408e-2', '-2.3600611340992838105408e-2');
    T('1.1982e+3', '1.1982e+3');
    T('1.895744317788222501065084139e+17', '1.895744317788222501065084139e+17');
    T('3.2450271098259184465439822499e+5', '3.2450271098259184465439822499e+5');
    T('1.1699868235212007000965506e+25', '-1.1699868235212007000965506e+25');
    T('7.988985662262809183538221216e+27', '-7.988985662262809183538221216e+27');
    T('1.476540158366695285164548325e+7', '-1.476540158366695285164548325e+7');
    T('8.8357361253e+1', '-8.8357361253e+1');
    T('2.6019583787920961e+15', '-2.6019583787920961e+15');
    T('2.617913486220978003463345e+24', '2.617913486220978003463345e+24');
    T('8.22380392476331112656616e+14', '-8.22380392476331112656616e+14');
    T('5.738943e+2', '-5.738943e+2');
    T('1.04315155601043625824403526143e+24', '-1.04315155601043625824403526143e+24');
    T('5.1800101324564241e-1', '-5.1800101324564241e-1');
    T('3.5101750876959537987e-8', '3.5101750876959537987e-8');
    T('2.1857385393e+3', '-2.1857385393e+3');
    T('2.29674272702302434336e+13', '2.29674272702302434336e+13');
    T('2.64606405319747e+14', '2.64606405319747e+14');
    T('2.1888980498865372455451e+1', '-2.1888980498865372455451e+1');
    T('1.51602e+0', '-1.51602e+0');
    T('5.8047548e+7', '5.8047548e+7');
    T('1.17525103769842428108679e+6', '-1.17525103769842428108679e+6');
    T('8.47642371517851e-1', '-8.47642371517851e-1');
    T('6.0574e+0', '-6.0574e+0');
    T('2.59202859815854485362744156646e-3', '2.59202859815854485362744156646e-3');
    T('1.040746238422014004691755e+15', '1.040746238422014004691755e+15');
    T('1.7064734811115159257936e+22', '-1.7064734811115159257936e+22');
    T('7.26051238227573319908663048e+26', '7.26051238227573319908663048e+26');
    T('7.4795685183599759424050861e+6', '-7.4795685183599759424050861e+6');
    T('2.9817e-16', '-2.9817e-16');
    T('2.298907884272330951e+6', '2.298907884272330951e+6');
    T('4.0531847e-8', '4.0531847e-8');
    T('2.6189e+4', '-2.6189e+4');
    T('3.911906e+3', '-3.911906e+3');
    T('9.408498865993245868145865993e+2', '-9.408498865993245868145865993e+2');
    T('4.05451047373376774e-7', '4.05451047373376774e-7');
    T('2.08836709959016517e+6', '-2.08836709959016517e+6');
    T('6.3417891663e+10', '6.3417891663e+10');
    T('8.08596745e+9', '8.08596745e+9');
    T('2.5865615419545921e+13', '2.5865615419545921e+13');
    T('1.5731674925482283378868e+22', '-1.5731674925482283378868e+22');
    T('1.19068602e+1', '-1.19068602e+1');
    T('5.3687670881355020502668e-3', '-5.3687670881355020502668e-3');
    T('1.2488884456407e+10', '-1.2488884456407e+10');
    T('2.51800212e+3', '-2.51800212e+3');
    T('3.738131519976930832896022e+24', '-3.738131519976930832896022e+24');
    T('6e+0', '6e+0');
    T('1.24131e+5', '-1.24131e+5');
    T('9.22635e+3', '-9.22635e+3');
    T('4e+0', '4e+0');
    T('1.83e+1', '1.83e+1');
    T('1.846025e+6', '-1.846025e+6');
    T('1.27e+1', '1.27e+1');
    T('2.24e+1', '2.24e+1');
    T('2.476323257183413822109348e-18', '-2.476323257183413822109348e-18');
    T('1.926752842e-7', '1.926752842e-7');
    T('8.80612762892681839383e-19', '8.80612762892681839383e-19');
    T('1.101085e+3', '-1.101085e+3');
    T('3.4906077350467600648759e+22', '3.4906077350467600648759e+22');
    T('1.04494855994965735236868e+23', '1.04494855994965735236868e+23');
    T('1.58387879923230822739579e+19', '1.58387879923230822739579e+19');
    T('4.213902971419525700930675e+19', '-4.213902971419525700930675e+19');
    T('9.13804011600009749427632034e+0', '9.13804011600009749427632034e+0');
    T('1.84491548817806624708211e+23', '-1.84491548817806624708211e+23');
    T('1.948625124086563483825890385e+22', '1.948625124086563483825890385e+22');
    T('1.3e+0', '1.3e+0');
    T('1.32939216745e+12', '1.32939216745e+12');
    T('7.078251628e+6', '7.078251628e+6');
    T('1.7313022e+2', '1.7313022e+2');
    T('3.415584872774897359156e+0', '3.415584872774897359156e+0');
    T('5.51297107980065895009041695e+23', '5.51297107980065895009041695e+23');
    T('2.5113503918614988744859e-15', '2.5113503918614988744859e-15');
    T('1.630239450859331215249576367e+27', '1.630239450859331215249576367e+27');
    T('5.4721390329589760404415744136e+18', '-5.4721390329589760404415744136e+18');
    T('2.945751278429364126367812e-17', '2.945751278429364126367812e-17');
    T('4.2782880893227686126997e+4', '4.2782880893227686126997e+4');
    T('1.9847055931e+1', '-1.9847055931e+1');
    T('2.261026e+3', '-2.261026e+3');
    T('1.52615708575e+9', '1.52615708575e+9');
    T('4.55553743697189921932e+5', '-4.55553743697189921932e+5');
    T('4.222829719336993778496867e+12', '4.222829719336993778496867e+12');
    T('4.485e+3', '4.485e+3');
    T('5.2e+0', '-5.2e+0');
    T('1.845091473820299081635836e+6', '1.845091473820299081635836e+6');
    T('5.46863948617381450255744e-14', '-5.46863948617381450255744e-14');
    T('3.0245e+4', '3.0245e+4');
    T('1.53486267119215101935302e-6', '-1.53486267119215101935302e-6');
    T('6.4843132478784299210571e+16', '6.4843132478784299210571e+16');
    T('4.386363241636966071e+13', '-4.386363241636966071e+13');
    T('7.581683508504e+6', '7.581683508504e+6');
    T('1.09730944345409824e+16', '1.09730944345409824e+16');
    T('3.594503e+6', '-3.594503e+6');
    T('4.443273220375505949638436659e+1', '4.443273220375505949638436659e+1');
    T('1.70867026016477719112e+20', '-1.70867026016477719112e+20');
    T('1.29553439888e+11', '-1.29553439888e+11');
    T('1.1130502308247230952431e-11', '1.1130502308247230952431e-11');
    T('6.058565749e+10', '-6.058565749e+10');
    T('3.87180284987679e-10', '-3.87180284987679e-10');
    T('3.49184930268913133535e+19', '3.49184930268913133535e+19');
    T('9e+0', '9e+0');
    T('1.28461567447442016927071963077e-8', '-1.28461567447442016927071963077e-8');
    T('2.72815445800161137e-19', '2.72815445800161137e-19');
    T('5.849268583211e-4', '5.849268583211e-4');
    T('3.19417089569942412006e+3', '-3.19417089569942412006e+3');
    T('1.9e+1', '-1.9e+1');
    T('3.3872886317814608310483125577e+6', '3.3872886317814608310483125577e+6');
    T('3.99977971703789643632671956e+9', '-3.99977971703789643632671956e+9');
    T('1.998549e-5', '1.998549e-5');
    T('7.18512424913e-15', '7.18512424913e-15');
    T('9.365052273317995234261e+21', '9.365052273317995234261e+21');
    T('2.569e+3', '-2.569e+3');
    T('9.460553674215355e+3', '-9.460553674215355e+3');
    T('1.22541e+2', '-1.22541e+2');
    T('2.180882957e-2', '-2.180882957e-2');
    T('3.963983308804e-5', '3.963983308804e-5');
    T('4.9059909584804e+11', '4.9059909584804e+11');
    T('3.89345544e+8', '-3.89345544e+8');
    T('3.13811755993550161609599737307e+9', '3.13811755993550161609599737307e+9');
    T('2.1684124657298e+7', '2.1684124657298e+7');
    T('4e+0', '4e+0');
    T('1.89e+1', '-1.89e+1');
    T('1.0500428125617165569673e+6', '1.0500428125617165569673e+6');
    T('3.45971690973815432646e+9', '-3.45971690973815432646e+9');
    T('4e+0', '-4e+0');
    T('1.2826728638181755448600624e+4', '-1.2826728638181755448600624e+4');
    T('5.2490288314345e+5', '5.2490288314345e+5');
    T('8.46401e+0', '8.46401e+0');
    T('2.15070506987596858e-9', '2.15070506987596858e-9');
    T('1.4569180505e+5', '-1.4569180505e+5');
    T('1.75535288191468954993283e+8', '-1.75535288191468954993283e+8');
    T('1.83e-19', '1.83e-19');
    T('3.77847393193912874449578e+6', '3.77847393193912874449578e+6');
    T('2.823610210086368e+0', '2.823610210086368e+0');
    T('3.2326e+4', '-3.2326e+4');
    T('7.21208310236919171558e+7', '-7.21208310236919171558e+7');
    T('2.537182162994085967e+11', '2.537182162994085967e+11');
    T('2.4881474405e-15', '2.4881474405e-15');
    T('6.8484737e+6', '6.8484737e+6');
    T('8.09636762896763e+1', '8.09636762896763e+1');
    T('1.387805e+1', '-1.387805e+1');
    T('1.949086825141843503e-3', '-1.949086825141843503e-3');
    T('8.22006002683570972726913386e+26', '-8.22006002683570972726913386e+26');
    T('8.82e+1', '-8.82e+1');
    T('9.8e+0', '-9.8e+0');
    T('5.73018e+5', '-5.73018e+5');
    T('2.039854296e-18', '2.039854296e-18');
    T('3.85806698884e+2', '3.85806698884e+2');
    T('7.761351239715879e-15', '-7.761351239715879e-15');
    T('2.37976961448611739e-13', '2.37976961448611739e-13');
    T('1.625694436559179391897024e-12', '-1.625694436559179391897024e-12');
    T('2.612e+1', '-2.612e+1');
    T('8.317023570754122191146041e+24', '8.317023570754122191146041e+24');
    T('8.128823e-9', '8.128823e-9');
    T('3.316888938212137e-7', '3.316888938212137e-7');
    T('4.590734e+2', '4.590734e+2');
    T('9.95284154681380079083087718e-7', '9.95284154681380079083087718e-7');
    T('1.379051e-15', '1.379051e-15');
    T('2.543347781939297185736e+21', '-2.543347781939297185736e+21');
    T('1.41496183748704601485699e-10', '-1.41496183748704601485699e-10');
    T('3.11665e+5', '-3.11665e+5');
    T('6.4377728353162694052697e+1', '6.4377728353162694052697e+1');
    T('1.36920115218557491e+17', '1.36920115218557491e+17');
    T('1.27e+1', '-1.27e+1');
    T('5.1e-4', '5.1e-4');
    T('4.124e+3', '4.124e+3');
    T('7.96e+0', '7.96e+0');
    T('1.0109019145999979839008159507e-20', '1.0109019145999979839008159507e-20');
    T('1.507784067070212e+12', '1.507784067070212e+12');
    T('5.03530585620864526983697e+10', '5.03530585620864526983697e+10');
    T('5.87771648701709094e-3', '-5.87771648701709094e-3');
    T('2.6641175511284360931e+19', '2.6641175511284360931e+19');
    T('3.5430949752e+3', '-3.5430949752e+3');
    T('1.434481e+6', '1.434481e+6');
    T('6.95e+0', '6.95e+0');
    T('2.7922814988487634078255e+17', '2.7922814988487634078255e+17');
    T('1e+0', '-1e+0');
    T('1.34094272275111823704509269719e+9', '-1.34094272275111823704509269719e+9');
    T('5.2e+0', '5.2e+0');
    T('5.961731008805248930549e+0', '5.961731008805248930549e+0');
    T('1.95863217313239788358925850999e+27', '1.95863217313239788358925850999e+27');
    T('1.115927378282807678794111117e+18', '-1.115927378282807678794111117e+18');
    T('6.6448e-6', '-6.6448e-6');
    T('1.210298078691983e-7', '1.210298078691983e-7');
    T('1.55022703113469956595e+8', '-1.55022703113469956595e+8');
    T('2.519409262126392490249e+9', '-2.519409262126392490249e+9');
    T('8.3744112435155841906e+19', '8.3744112435155841906e+19');
    T('5.56052914013431e-4', '5.56052914013431e-4');
    T('1.847716075495989e+13', '-1.847716075495989e+13');
    T('5.78580529835020695846e+19', '-5.78580529835020695846e+19');
    T('7.3177e-15', '-7.3177e-15');
    T('5.8018949e+6', '-5.8018949e+6');
    T('1.234850494854913982840923624126e+30', '1.234850494854913982840923624126e+30');
    T('3.1e+0', '3.1e+0');
    T('3.085340434810406103e+4', '3.085340434810406103e+4');
    T('1.461332e+6', '1.461332e+6');
    T('2.042933164181166e-9', '2.042933164181166e-9');
    T('1.14852656434391849784404293276e-6', '1.14852656434391849784404293276e-6');
    T('8.56930722573e-11', '8.56930722573e-11');
    T('7.753629727831898e+11', '7.753629727831898e+11');
    T('2.5807119689e+5', '-2.5807119689e+5');
    T('6.5889872564e+7', '6.5889872564e+7');
    T('6.2e+0', '6.2e+0');
    T('7.16926024589772e+14', '-7.16926024589772e+14');
    T('2.444762609546357e-12', '2.444762609546357e-12');
    T('1.58017211706879e+2', '-1.58017211706879e+2');
    T('2.74612804105217564273009e+23', '-2.74612804105217564273009e+23');
    T('8.2105e+3', '-8.2105e+3');
    T('6.2289747e+7', '-6.2289747e+7');
    T('4.47847136680063365276e+21', '-4.47847136680063365276e+21');
    T('7.599263848474204e+15', '-7.599263848474204e+15');
    T('9.534064037670226206e-11', '-9.534064037670226206e-11');
    T('5.3511395608925655035624181e+7', '-5.3511395608925655035624181e+7');
    T('2.536656469414e+8', '2.536656469414e+8');
    T('4.454301005499233196018257e+16', '-4.454301005499233196018257e+16');
    T('2.3289800995961777747097e+10', '-2.3289800995961777747097e+10');
    T('2.7363696755334e+6', '-2.7363696755334e+6');
    T('2.56e+2', '2.56e+2');
    T('7.3430201092837e+2', '7.3430201092837e+2');
    T('1.114804e+5', '1.114804e+5');
    T('3.1845809556698336607622e+4', '-3.1845809556698336607622e+4');
    T('1.7780378655260403138e+19', '-1.7780378655260403138e+19');
    T('3.608970926e-15', '3.608970926e-15');
    T('1.949e+3', '-1.949e+3');
    T('1.9021837e+4', '-1.9021837e+4');
    T('1.5e+0', '1.5e+0');
    T('3.1155266673e+10', '-3.1155266673e+10');
    T('4e+0', '-4e+0');
    T('9.09316542545977506e+14', '9.09316542545977506e+14');
    T('2.15531740334146749845e+8', '2.15531740334146749845e+8');
    T('1.5605317646e+8', '1.5605317646e+8');
    T('3.8806066633613066e+13', '-3.8806066633613066e+13');
    T('1.653298e+6', '1.653298e+6');
    T('7.920024310736e-20', '7.920024310736e-20');
    T('2.27611872e+8', '2.27611872e+8');
    T('2.76569307109179036145271e-15', '-2.76569307109179036145271e-15');
    T('1.425171314e+8', '1.425171314e+8');
    T('1.3702555167748408653e+11', '-1.3702555167748408653e+11');
    T('5.146936435e+9', '5.146936435e+9');
    T('4.183285814905222880076696e+19', '-4.183285814905222880076696e+19');
    T('2.270923702039578057376e-16', '2.270923702039578057376e-16');
    T('9.4963549e-12', '9.4963549e-12');
    T('1.453060439e-3', '1.453060439e-3');
    T('2.97303365e+2', '2.97303365e+2');
    T('1.16485757109e+2', '-1.16485757109e+2');
    T('7.7984946334626919799413338378e+5', '-7.7984946334626919799413338378e+5');
    T('1.905453e+5', '1.905453e+5');
    T('5.36989497616503e-20', '5.36989497616503e-20');
    T('4.3e+0', '4.3e+0');
    T('2.70434008699476809368089518776e+25', '-2.70434008699476809368089518776e+25');
    T('2.8813069851e+10', '2.8813069851e+10');
    T('7e+0', '7e+0');
    T('1.0577487e-18', '-1.0577487e-18');
    T('6.8e+1', '6.8e+1');
    T('1e+0', '-1e+0');
    T('8.446803887694575079e+6', '-8.446803887694575079e+6');
    T('2.3384835e-6', '-2.3384835e-6');
    T('1.072e-13', '1.072e-13');
    T('7.13295350162e-5', '7.13295350162e-5');
    T('4.59897478609e+3', '4.59897478609e+3');
    T('4.11875744698515118e+11', '4.11875744698515118e+11');
    T('3.12339620225171e+5', '3.12339620225171e+5');
    T('3.79932554e+1', '3.79932554e+1');
    T('2.457332691061964e+4', '-2.457332691061964e+4');
    T('3.944602320705902e+6', '-3.944602320705902e+6');
    T('3.164305812145e+4', '-3.164305812145e+4');
    T('7.22239735515689399e+1', '-7.22239735515689399e+1');
    T('5.261981e+3', '-5.261981e+3');
    T('2.3642968462845e+7', '2.3642968462845e+7');
    T('3.9326785e+3', '-3.9326785e+3');
    T('8.5853e-11', '-8.5853e-11');
    T('2.60532943946e+0', '2.60532943946e+0');
    T('3.64630216318427246476533e+18', '-3.64630216318427246476533e+18');
    T('3.031732127749e-3', '3.031732127749e-3');
    T('2.49298080885329502254338e-12', '-2.49298080885329502254338e-12');
    T('8.81838341457179780743504843e+2', '-8.81838341457179780743504843e+2');
    T('2.285650225267766689304972e+5', '2.285650225267766689304972e+5');
    T('4.5790517211306242e+7', '4.5790517211306242e+7');
    T('3.0033340092338313923473428e+16', '-3.0033340092338313923473428e+16');
    T('2.83879929283797623e+1', '-2.83879929283797623e+1');
    T('4.5266377717178121183759377414e-5', '4.5266377717178121183759377414e-5');
    T('5.3781e+4', '-5.3781e+4');
    T('6.722035208213298413522819127e-18', '-6.722035208213298413522819127e-18');
    T('3.02865707828281230987116e+23', '-3.02865707828281230987116e+23');
    T('5.5879983320336874473209567979e+28', '-5.5879983320336874473209567979e+28');

    log('\n ' + passed + ' of ' + total + ' tests passed in ' + (+new Date() - start) + ' ms \n');
    return [passed, total];
})(this.BigNumber);
if (typeof module !== 'undefined' && module.exports) module.exports = count;